//  
//  Copyright (C) 2009 GNOME Do
// 
//  This program is free software: you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation, either version 3 of the License, or
//  (at your option) any later version.
// 
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
// 
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
// 

using System;
using System.Collections.Generic;

using Gdk;
using Gtk;
using Cairo;
using Mono.Unix;

using Do.Interface;
using Do.Interface.CairoUtils;
using Do.Platform;
using Do.Platform.Linux;

using Docky.Core;
using Docky.Interface;
using Docky.Interface.Menus;

namespace WeatherDocklet
{
	/// <summary>
	/// Indicates what mode the docklet currently is in.
	/// </summary>
	public enum WeatherDockletStatus
	{
		Initializing,
		Normal,
		Reloading,
		ManualReload,
		Error
	}
	
	/// <summary>
	/// A docklet to display the current temp and condition as an icon and
	/// use a painter to display forecast information.
	/// </summary>
	public class WeatherDocklet : AbstractDockletItem, IRightClickable
	{
		/// <value>
		/// Indicates what mode the docklet currently is in.
		/// </value>
		WeatherDockletStatus Status { get; set; }
		
		/// <value>
		/// The weather painter that displays this docklet's weather info.
		/// </value>
		WeatherPainter weather_painter;
		
		public override string Name {
			get {
				return Catalog.GetString ("Weather");
			}
		}
		
		public override ScalingType ScalingType {
			get {
				return ScalingType.Downscaled;
			}
		}
		
		/// <summary>
		/// Creates a new weather docklet.
		/// </summary>
		public WeatherDocklet ()
		{
			Status = WeatherDockletStatus.Initializing;
			weather_painter = new WeatherPainter (this);
			DockServices.PainterService.RegisterPainter (weather_painter);
			
			WeatherController.WeatherReloading += HandleWeatherReloading;
			WeatherController.WeatherError += HandleWeatherError;
			WeatherController.WeatherUpdated += HandleWeatherUpdated;
			
			WeatherController.ResetTimer (false);
			SetText (Catalog.GetString ("Fetching data..."));
		}
		
		/// <summary>
		/// Handles when a weather source reloads data.
		/// </summary>
		void HandleWeatherReloading ()
		{
			Services.Application.RunOnMainThread (() => {
				SetText (Catalog.GetString ("Fetching data..."));
				if (Status != WeatherDockletStatus.Initializing && Status != WeatherDockletStatus.ManualReload)
					Status = WeatherDockletStatus.Reloading;
				RedrawIcon ();
			});
		}
		
		/// <summary>
		/// Handles an error with the weather source.
		/// </summary>
		/// <param name="sender">
		/// Ignored
		/// </param>
		/// <param name="e">
		/// A <see cref="WeatherErrorArgs"/> which contains the error message.
		/// </param>
		void HandleWeatherError (object sender, WeatherErrorArgs e)
		{
			Services.Application.RunOnMainThread (() => {
				SetText (e.Error);
				Status = WeatherDockletStatus.Error;
				RedrawIcon ();
			});
		}
		
		/// <summary>
		/// Handles when a weather source successfully reloads data.
		/// </summary>
		void HandleWeatherUpdated ()
		{
			Services.Application.RunOnMainThread (() => {
				IWeatherSource weather = WeatherController.Weather;
				
				string feelsLike = "";
				if (weather.SupportsFeelsLike && weather.Temp != weather.FeelsLike)
					feelsLike = " (" + weather.FeelsLike + WeatherUnits.TempUnit + ")";
				
				SetText (weather.City + "   " +
					weather.Temp + WeatherUnits.TempUnit +
					feelsLike + "   " + weather.Condition);
				Status = WeatherDockletStatus.Normal;
				RedrawIcon ();
				weather_painter.WeatherChanged ();
			});
		}
		
		/// <summary>
		/// Draws an icon onto a context using the specified offset and size.
		/// </summary>
		/// <param name="cr">
		/// A <see cref="Context"/> to draw the icon.
		/// </param>
		/// <param name="icon">
		/// A <see cref="System.String"/> containing the icon name to use.
		/// </param>
		/// <param name="xoffset">
		/// A <see cref="System.Int32"/> indicating the x offset for the icon.
		/// </param>
		/// <param name="size">
		/// A <see cref="System.Int32"/> indicating the size of the icon.
		/// </param>
		public static void RenderIconOntoContext (Context cr, string icon, int xoffset, int size)
		{
			using (Gdk.Pixbuf pbuf = IconProvider.PixbufFromIconName (icon, size))
			{
				CairoHelper.SetSourcePixbuf (cr, pbuf, xoffset, 0);
				cr.Paint ();
			}
		}
		
		protected override Surface MakeIconSurface (Cairo.Surface similar, int size)
		{
			Surface tmp_surface = similar.CreateSimilar (similar.Content, size, size);
			
			using (Context cr = new Context (tmp_surface)) {
				cr.AlphaFill ();
				
				switch (Status)
				{
				case WeatherDockletStatus.Initializing:
					RenderIconOntoContext (cr, Gtk.Stock.Refresh, 0, size);
					break;
				
				case WeatherDockletStatus.Error:
					RenderIconOntoContext (cr, "network-offline", 0, size);
					break;
				
				default:
				case WeatherDockletStatus.Reloading:
				case WeatherDockletStatus.ManualReload:
				case WeatherDockletStatus.Normal:
					if (Status == WeatherDockletStatus.ManualReload) {
						using (Gdk.Pixbuf pbuf = IconProvider.PixbufFromIconName (WeatherController.Weather.Image, size)) {
							CairoHelper.SetSourcePixbuf (cr, pbuf, 0, 0);
							cr.PaintWithAlpha (0.5);
						}
					} else {
						RenderIconOntoContext (cr, WeatherController.Weather.Image, 0, size);
					}
					
					TextRenderContext textContext = new TextRenderContext (cr, string.Empty, size);
					textContext.Alignment = Pango.Alignment.Center;
					textContext.EllipsizeMode = Pango.EllipsizeMode.None;
					
					textContext.Text = string.Format ("<b>{0}{1}</b>", WeatherController.Weather.Temp, WeatherUnits.TempUnit);
					textContext.FontSize = (int) (size / 4);
					textContext.LeftCenteredPoint = new Gdk.Point (0, size - textContext.FontSize);
					DockServices.DrawingService.TextPathAtPoint (textContext);
					cr.LineWidth = 4;
					if (Status == WeatherDockletStatus.ManualReload)
						cr.Color = new Cairo.Color (0, 0, 0, 0.3);
					else
						cr.Color = new Cairo.Color (0, 0, 0, 0.5);
					cr.Stroke ();

					DockServices.DrawingService.TextPathAtPoint (textContext);
					if (Status == WeatherDockletStatus.ManualReload)
						cr.Color = new Cairo.Color (1, 1, 1, 0.5);
					else
						cr.Color = new Cairo.Color (1, 1, 1, 0.8);
					break;
				}
				
				cr.Fill ();
			}
			
			return tmp_surface;
		}
		
		public override void Clicked (uint button, Gdk.ModifierType state, PointD position)
		{
			weather_painter.Summon ();
			base.Clicked (button, state, position);
		}
		
		public override void Scrolled (Gdk.ScrollDirection direction)
		{
			if (Preferences.Location.Length <= 1)
				return;

			Status = WeatherDockletStatus.ManualReload;
			
			if (direction == Gdk.ScrollDirection.Up)
				WeatherController.PreviousLocation ();
			else
				WeatherController.NextLocation ();
		}
		
		#region IRightClickable implementation 
		
		public event EventHandler RemoveClicked;
		
		public IEnumerable<AbstractMenuArgs> GetMenuItems ()
		{
			yield return new SeparatorMenuButtonArgs ();
			
			if (WeatherController.Weather.Condition != null)
			{
				yield return new SimpleMenuButtonArgs (WeatherController.Weather.ShowRadar,
						Catalog.GetString ("Radar Map"), WeatherController.Weather.Image);
				
				yield return new SeparatorMenuButtonArgs ();
			}
			
			bool hasForecast = false;
			
			for (int i = 0; i < WeatherController.Weather.ForecastDays; i++)
				if (WeatherController.Weather.Forecasts [i].dow != null)
				{
					hasForecast = true;
					yield return new ForecastMenuButtonArgs (i,
							Catalog.GetString (WeatherForecast.DayName (WeatherController.Weather.Forecasts [i].dow) + "'s Forecast"),
							WeatherController.Weather.Forecasts [i].image);
				}
			
			if (hasForecast)
				yield return new SeparatorMenuButtonArgs ();
			
			yield return new SimpleMenuButtonArgs (() => { Status = WeatherDockletStatus.ManualReload; WeatherController.ResetTimer(); } ,
					Catalog.GetString ("Reload Weather Data"), Gtk.Stock.Refresh);
		}
		
		#endregion 
		
	}
}
