/* gnobog_mozilla_backend.c
 *
 * Copyright (C) 2000 Frdric LESPEZ & Renaud CHAILLAT
 *
 * This program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "gnobog_mozilla_backend.h"

/*
** Definitions
*/

// Mozilla entry size max
#define MOZILLA_ENTRY_MAX_SIZE 2048

// Definitions of Mozilla line type
typedef enum {
  MZ_LINE_UNKNOWN,
  MZ_BEGIN_ENTRY,
  MZ_BEGIN_ENTRY_TITLE,
  MZ_BEGIN_ENTRY_FOLDER,
  MZ_BEGIN_ENTRY_BOOKMARK,
  MZ_BEGIN_ENTRY_SEPARATOR,
  MZ_START_FOLDER,
  MZ_STOP_FOLDER,
  MZ_DESCRIPTION_LINE
} MozillaLineType;

// Definitions of Mozilla Importer State
typedef enum {
  STATE_IS_FOLDER_END,
  STATE_GET_ENTRY,
  STATE_IS_DESCRIPTION,
  STATE_GET_DESCRIPTION,
  STATE_IS_FOLDER,
  STATE_DOWN_FOLDER,
  STATE_UP_FOLDER
} ImportState;

// Pattern for Netscape bookmarks file identification
#define MOZILLA_BOOKMARKS_FILE_ID       "<!DOCTYPE NETSCAPE-Bookmark-file-1>"

// Export Patterns
#define MOZILLA_HEADER_1                "%s<!DOCTYPE NETSCAPE-Bookmark-file-1>\n"
#define MOZILLA_HEADER_2                "%s<!-- This is an automatically generated file.\n"
#define MOZILLA_HEADER_3                "%sIt will be read and overwritten.\n"
#define MOZILLA_HEADER_4                "%sDo Not Edit! -->\n"
#define MOZILLA_HEADER_5                "%s<META HTTP-EQUIV=\"Content-Type\" CONTENT=\"text/html; charset=UTF-8\">\n"


#define MOZILLA_TITLE_1                 "%s<TITLE>%s</TITLE>\n"
#define MOZILLA_TITLE_2                 "%s<H1>%s</H1>\n"

#define MOZILLA_FOLDER_FOLDED           "%s<DT><H3 FOLDED ADD_DATE=\"%d\">%s</H3>\n"
#define MOZILLA_FOLDER_UNFOLDED         "%s<DT><H3 ADD_DATE=\"%d\">%s</H3>\n"
#define MOZILLA_FOLDER_START            "%s<DL><p>\n"
#define MOZILLA_FOLDER_STOP             "%s</DL><p>\n"
#define MOZILLA_BOOKMARK                "%s<DT><A HREF=\"%s\" ADD_DATE=\"%d\" LAST_VISIT=\"%d\" LAST_MODIFIED=\"%d\">%s</A>\n"
#define MOZILLA_BOOKMARK_ALIASED        "%s<DT><A HREF=\"%s\" ALIASID=\"%d\" ADD_DATE=\"%d\" LAST_VISIT=\"%d\" LAST_MODIFIED=\"%d\">%s</A>\n"
#define MOZILLA_BOOKMARK_ALIAS          "%s<DT><A HREF=\"%s\" ALIASOF=\"%d\" ADD_DATE=\"%d\" LAST_VISIT=\"%d\" LAST_MODIFIED=\"%d\">%s</A>\n"
#define MOZILLA_BOOKMARK                "%s<DT><A HREF=\"%s\" ADD_DATE=\"%d\" LAST_VISIT=\"%d\" LAST_MODIFIED=\"%d\">%s</A>\n"
#define MOZILLA_SEPARATOR               "%s<HR>\n"

// Import Patterns
#define MOZILLA_TITLE_PATTERN            "<H1>"
#define MOZILLA_TITLE_PREFIX             "<H1>"
#define MOZILLA_TITLE_SUFFIX             "</H1>"

#define MOZILLA_FOLDER_PATTERN           "<DT><H3"
#define MOZILLA_FOLDER_START_PATTERN     "<DL><p>"
#define MOZILLA_FOLDER_STOP_PATTERN      "</DL><p>"
#define MOZILLA_FOLDER_PREFIX            ">"
#define MOZILLA_FOLDER_SUFFIX            "</H3>"

//#define MOZILLA_BOOKMARK_PATTERN              "<DT><A"
//#define MOZILLA_BOOKMARK_LOCATION_PREFIX "<DT><A HREF=\""
/* to handle the ~#{^\ case when we have <H3> for a bookmark ! */
#define MOZILLA_BOOKMARK_PATTERN         "<A"
#define MOZILLA_BOOKMARK_LOCATION_PREFIX "HREF=\""
#define MOZILLA_BOOKMARK_NAME_PREFIX     ">"
#define MOZILLA_BOOKMARK_NAME_SUFFIX     "</A>"

#define MOZILLA_SEPARATOR_PATTERN        "<HR>"

#define MOZILLA_GENERIC_SUFFIX           "\""

#define MOZILLA_DESCRIPTION_BEGIN        "<DD>"
#define MOZILLA_DESCRIPTION_SEPARATOR    "<BR>"

#define MOZILLA_FOLDED_PATTERN           "FOLDED"
#define MOZILLA_ALIASID_PATTERN          "ALIASID=\""
#define MOZILLA_ALIASOF_PATTERN          "ALIASOF=\""
#define MOZILLA_CREATED_PATTERN          "ADD_DATE=\""
#define MOZILLA_VISITED_PATTERN          "LAST_VISIT=\""
#define MOZILLA_MODIFIED_PATTERN         "LAST_MODIFIED=\""

// Global variable
static gint nb_entry;
static gint nb_title;
static gint nb_folder;
static gint nb_separator;
static gint nb_bookmark;
static gint nb_alias;

static GHashTable*  aliasof_table;
static GHashTable*  aliasid_table;

/*
** Functions Declarations
*/

// Read a line in bookmarks file.
// Skip empty line.
static gboolean
mozilla_file_read_next_line (FILE* file, gchar* line)
{
  gchar*  status; 

  while (((status = fgets (line, MOZILLA_ENTRY_MAX_SIZE, file)) != NULL)
         && (strcmp (line, "\n") == 0));
        
  // Be sure the buffer is null terminated
  line[MOZILLA_ENTRY_MAX_SIZE-1] = '\0';
        
  // Remove '\n' at the end of the line (if needed)
  if (line[strlen (line)-1] == '\n') {
    line[strlen (line)-1] = '\0';
  }

  return (status != NULL);
}

/* Get the string value delimited by "prefix" and "suffix" in mozilla_entry string
 * and return the position of the following character.
 * mozilla_entry MUST be correctly null terminated (as must be a string!).
 */
static const gchar*
mozilla_entry_pattern_get_str_value(gchar** value, const gchar* mozilla_entry,
                                    gchar *prefix, gchar *suffix)
{
  gchar* begin;
  gchar* end;     

  /* Look for the prefix */
  begin = strstr(mozilla_entry, prefix);
  if (begin == NULL) {
    *value = g_new(gchar, 1);
    (*value)[0] = '\0';
    return mozilla_entry; /* Return the pointer to the beginning of the string */
  }
        
  /* Skip the prefix if found */
  begin += strlen(prefix);
        
  /* Look for the suffix */
  end = strstr(begin, suffix);
  if (end == NULL) {
    *value = g_new(gchar, strlen(mozilla_entry) - strlen(prefix)+1 );
    strncpy(*value, begin, strlen(mozilla_entry) - strlen(prefix));
    return begin+strlen(mozilla_entry); /* '\0' */
  }
        
  /* If prefix and suffix have been found */
  *value = g_new(gchar, (size_t)(end-begin)+1);
  if ( (end-begin)!=0 )
    strncpy(*value, begin, (size_t)(end-begin)); /* what if (end-begin)==0 ? */
  (*value)[end-begin] = '\0';
  //      return ++end;
  return end; /* return the position just after the string */
}

static const gchar*
mozilla_entry_pattern_get_int_value(gint* value, const gchar* mozilla_entry,
                                    gchar *prefix, gchar *suffix)
{
  const gchar* position;
  gchar* value_as_str;
        
  position = mozilla_entry_pattern_get_str_value(&value_as_str,
                                                 mozilla_entry,
                                                 prefix,
                                                 suffix);
  *value = atoi(value_as_str);
  g_free(value_as_str);           
  return position;
}

static gboolean
mozilla_line_is_title (const gchar* line)
{
  return (strstr (line, MOZILLA_TITLE_PATTERN) != NULL);
}

static gboolean
mozilla_line_is_folder (const gchar* line)
{
  return (strstr (line, MOZILLA_FOLDER_PATTERN) != NULL);
}

static gboolean
mozilla_line_is_bookmark (const gchar* line)
{
  return (strstr (line, MOZILLA_BOOKMARK_PATTERN) != NULL);
}

static gboolean
mozilla_line_is_separator (const gchar* line)
{
  return (strstr (line, MOZILLA_SEPARATOR_PATTERN) != NULL);
}

static gboolean
mozilla_line_is_folder_start(const gchar* line)
{
  return (strstr (line, MOZILLA_FOLDER_START_PATTERN) != NULL);
}

static gboolean
mozilla_line_is_folder_stop(const gchar* line)
{
  return (strstr (line, MOZILLA_FOLDER_STOP_PATTERN) != NULL);
}

static MozillaLineType
mozilla_line_get_type (const gchar* line)
{
  MozillaLineType type;

  //      type = MZ_DESCRIPTION_LINE;
  /* Anything is a description except the following */
  if (mozilla_line_is_title (line)) {
    type = MZ_BEGIN_ENTRY_TITLE;
  } else if (mozilla_line_is_bookmark (line)) {
    type = MZ_BEGIN_ENTRY_BOOKMARK;
  } else if (mozilla_line_is_folder (line)) { /* test AFTER bookmark in case of <H3> with url */
    type = MZ_BEGIN_ENTRY_FOLDER;
  } else if (mozilla_line_is_separator (line)) {
    type = MZ_BEGIN_ENTRY_SEPARATOR;
  } else if (mozilla_line_is_folder_start (line)) {
    type = MZ_START_FOLDER;
  } else if (mozilla_line_is_folder_stop (line)) {
    type = MZ_STOP_FOLDER;
  } else {
    type = MZ_DESCRIPTION_LINE;
  }
  return type;
}

static GnobogBookmarksNodeType
mz_node_get_type (GnobogBookmarksNode node)
{
  if (node->data != NULL) return node_get_type (node);
  else return BOOKMARK;
}

// TODO : handle damaged lines...
static GnobogBookmarksNode
mozilla_import_title (const gchar* line)
{
  GnobogBookmarksNode             node;
  gchar *name;

  (void)mozilla_entry_pattern_get_str_value (&name,
                                             line,
                                             MOZILLA_TITLE_PREFIX,
                                             MOZILLA_TITLE_SUFFIX);
  node = gnobog_bookmarks_node_import (TITLE, name, "", "", 0, 0, 0, FALSE);
  g_free(name);
  return node;
}

// TODO : handle damaged lines...
static GnobogBookmarksNode
mozilla_import_folder (const gchar* line)
{
  GnobogBookmarksNode             node;
  gchar *name;
  gchar *tempstring;
  guint created;
  gboolean folded;
  const gchar *position;

  /* look at the 'folded' state of the folder */
  if (strstr (line, MOZILLA_FOLDED_PATTERN) != NULL) {
    folded = TRUE;  
  } else {
    folded = FALSE; 
  }       
  /* get the creation date if present */
  position = mozilla_entry_pattern_get_int_value ((guint*) &created,
                                                  line,
                                                  MOZILLA_CREATED_PATTERN,
                                                  MOZILLA_GENERIC_SUFFIX);
  /* Get the name. If there was no creation date, we must increase the
     cursor position to skip the <DT><H3 prefix */
  if (position==line) {
    position = mozilla_entry_pattern_get_str_value (&tempstring,
                                                    position,
                                                    MOZILLA_FOLDER_PATTERN,
                                                    MOZILLA_FOLDER_PREFIX);
  }
  position = mozilla_entry_pattern_get_str_value (&name,
                                                  position,
                                                  MOZILLA_FOLDER_PREFIX,
                                                  MOZILLA_FOLDER_SUFFIX);
  node = gnobog_bookmarks_node_import (FOLDER, name, "", "", created, 0, 0, !folded);
  g_free(name);
  return node;
}

// TODO : handle damaged lines... see mozilla_entry_pattern_get_str
static GnobogBookmarksNode
mozilla_import_bookmark(const gchar* line)
{
  GList*  aliasof_list;
  GnobogBookmarksNode             node;
  gchar* name;
  gchar *location;
  guint created, visited, modified;
  gint aliasid, aliasof;
  const gchar *position;

  position = mozilla_entry_pattern_get_str_value (&location,
                                                  line,
                                                  MOZILLA_BOOKMARK_LOCATION_PREFIX,
                                                  MOZILLA_GENERIC_SUFFIX);
  aliasid = -1;
  aliasof = -1;
  if (strstr (position, MOZILLA_ALIASID_PATTERN) != NULL) {
    position = mozilla_entry_pattern_get_int_value (&aliasid,
                                                    position,
                                                    MOZILLA_ALIASID_PATTERN,
                                                    MOZILLA_GENERIC_SUFFIX);        
  }
  if (strstr (position, MOZILLA_ALIASOF_PATTERN) != NULL) {
    position = mozilla_entry_pattern_get_int_value (&aliasof,
                                                    position,
                                                    MOZILLA_ALIASOF_PATTERN,
                                                    MOZILLA_GENERIC_SUFFIX);        
  }
  position = mozilla_entry_pattern_get_int_value ((guint*) &created,
                                                  position,
                                                  MOZILLA_CREATED_PATTERN,
                                                  MOZILLA_GENERIC_SUFFIX);        
  position = mozilla_entry_pattern_get_int_value ((guint*) &visited,
                                                  position,
                                                  MOZILLA_VISITED_PATTERN,
                                                  MOZILLA_GENERIC_SUFFIX);        
  position = mozilla_entry_pattern_get_int_value ((guint*) &modified,
                                                  position,
                                                  MOZILLA_MODIFIED_PATTERN,
                                                  MOZILLA_GENERIC_SUFFIX);        
  position = mozilla_entry_pattern_get_str_value (&name,
                                                  position,
                                                  MOZILLA_BOOKMARK_NAME_PREFIX,
                                                  MOZILLA_BOOKMARK_NAME_SUFFIX);
  if (aliasof == -1) {
    node = gnobog_bookmarks_node_import (BOOKMARK, name, location, "", created, visited, modified, FALSE);
  } else {
    node = g_node_new (NULL);
  }
  if (aliasid != -1) {
    g_message ("AliasID : %d", aliasid);
    g_assert (g_hash_table_lookup (aliasid_table, GINT_TO_POINTER (aliasid)) == NULL);
    g_hash_table_insert (aliasid_table,
                         GINT_TO_POINTER (aliasid),
                         node);
    nb_alias++;
  }
        
  if (aliasof != -1) {
    aliasof_list = g_hash_table_lookup (aliasof_table, GINT_TO_POINTER (aliasof));
    aliasof_list = g_list_prepend (aliasof_list, node);
    g_hash_table_insert (aliasof_table, GINT_TO_POINTER (aliasof), aliasof_list);
  }
  g_free(name);
  g_free(location);
        
  return node;
}

// TODO : handle damaged lines...
static GnobogBookmarksNode
mozilla_import_separator(const gchar* line)
{
  return gnobog_bookmarks_node_import_separator ();
}

// TODO : handle damaged lines...
static GnobogBookmarksNode
mozilla_import_entry (const gchar* line)
{
  MozillaLineType type;
  GnobogBookmarksNode     node;
        
  /* Do some statistics */
  nb_entry++;
        
  type = mozilla_line_get_type (line);
        
  node = NULL;
  switch (type) {
  case MZ_BEGIN_ENTRY_TITLE:
    /* Do some statistics */
    nb_title++;
    node =  mozilla_import_title (line);
    break;
  case MZ_BEGIN_ENTRY_FOLDER:
    /* Do some statistics */
    nb_folder++;
    node =  mozilla_import_folder (line);
    break;
  case MZ_BEGIN_ENTRY_BOOKMARK:
    /* Do some statistics */
    nb_bookmark++;
    node =  mozilla_import_bookmark (line);
    break;
  case MZ_BEGIN_ENTRY_SEPARATOR:
    /* Do some statistics */
    nb_separator++;
    node =  mozilla_import_separator (line);
    break;
  default:
    /* We can't be here */
    g_assert_not_reached();
  }
        
  return node;
}

static void
mozilla_entry_complete_description (GnobogBookmarksNode node, const gchar* line)
{
  gchar*  current_description;
  gchar*  new_description;
  gchar*  tmp;
  gchar*  description_line;
                
  g_return_if_fail (node != NULL);
        
  if (node->data == NULL) {
    return;
  }
  tmp = g_strdup (line);
  description_line = tmp;
  if (strstr (tmp, MOZILLA_DESCRIPTION_BEGIN) != NULL) {
    description_line = description_line + 4;
  }
  if (strstr (description_line, MOZILLA_DESCRIPTION_SEPARATOR) != NULL) {
    description_line[strlen (description_line) - 4] = '\0';
  }
        
  // Add the new line to existing description
  current_description = node_get_description (node);
  if (strcmp (current_description, "") != 0) {
    new_description = g_strjoin ("\n", current_description, description_line, NULL);
  } else {
    new_description = g_strdup (description_line);
  }
        
  //      g_message ("%s", new_description);
        
  /* Free current description before replacing it */
  g_free (node_get_description (node));
  node_set_description (node, new_description);

  g_free (tmp);
        
  return;
}

static GNode*
mozilla_import_file (FILE* file)
{
  GnobogBookmarksNode             parent_node;
  GnobogBookmarksNode             node;
  //      GnobogBookmarksEntry    entry;
  gchar*                  line;
  MozillaLineType line_type;
  ImportState             state;
  gboolean                read_next;
  GSList*                 stacked_parent_node;
  gboolean                end_of_file;
        
  /* Allocate a buffer for reading file lines */
  line = g_new(gchar, MOZILLA_ENTRY_MAX_SIZE);
   
  aliasid_table = g_hash_table_new (g_direct_hash, g_direct_equal);
  aliasof_table = g_hash_table_new (g_direct_hash, g_direct_equal);
                
  /* No jealous ! Everybody set to NULL */
  stacked_parent_node = NULL;
  parent_node = NULL;
  node = NULL;
        
  /* More stuff initialization */
  read_next = TRUE;
  end_of_file = FALSE;
                
  /* Start with this STATE */
  state = STATE_IS_FOLDER_END;
        
  while (!end_of_file)
    {
      /* Do we read the next line ? */
      if (read_next) {
        end_of_file = !mozilla_file_read_next_line (file, line);
        //                      g_message ("Line read: %s", line);
      }
                
      /* End of file reach */
      if (end_of_file) {
        if (stacked_parent_node != NULL) {
                                // Corrupt File
          g_message ("Mozilla file corrupted : Some MZ_STOP_FOLDER are missing");
          //                              g_assert_not_reached();
        }
        /* End of File reached : Go to 'while' to exit loop */
        /* We try to display as much as possible of a corrupt file */
        continue;
      }
                
      /* Analyze current line entry type */
      line_type = mozilla_line_get_type (line);
                
      /* process the line */
      switch (state) {
      case STATE_IS_FOLDER_END:
                                /* Do not read next line. Current one must be processed */
        read_next = FALSE;
                                /* Change state to process the line */
        switch (line_type) {
        case MZ_BEGIN_ENTRY_TITLE:
        case MZ_BEGIN_ENTRY_FOLDER:
        case MZ_BEGIN_ENTRY_BOOKMARK:
        case MZ_BEGIN_ENTRY_SEPARATOR:
          state = STATE_GET_ENTRY;
          break;
        case MZ_STOP_FOLDER:
          state = STATE_UP_FOLDER;
          break;
        default:
          /* Corrupt File : we skip and try to read following lines...  */
          g_message ("Mozilla file corrupted : Expecting line type MZ_BEGIN_ENTRY or MZ_STOP_FOLDER");
          read_next = TRUE;
          //                                              g_assert_not_reached();
        }
        break;
                        
      case STATE_GET_ENTRY:
        switch (line_type) {
        case MZ_BEGIN_ENTRY_TITLE:
        case MZ_BEGIN_ENTRY_FOLDER:
        case MZ_BEGIN_ENTRY_BOOKMARK:
        case MZ_BEGIN_ENTRY_SEPARATOR:
          // Import entry
          node = mozilla_import_entry (line);                             
          if (mz_node_get_type (node) == SEPARATOR) {
            // Entry is complete. Store it.
            g_node_append (parent_node, node);
            state = STATE_IS_FOLDER_END;
          } else {
            // Entry is not complete. Fetch description.
            state = STATE_IS_DESCRIPTION;
          }
          break;
        default:
          /* impossible : state_is_folder_end has just sent us here...*/
          g_message ("Error from outer space in STATE_GET_ENTRY");
          //                                              g_assert_not_reached();
        }       
                                /* Read next line */
        read_next = TRUE;
        break;
                                
      case STATE_IS_DESCRIPTION:
        switch (line_type) {
        case MZ_DESCRIPTION_LINE:
          /* Description is not complete */
          state = STATE_GET_DESCRIPTION;
          break;
        default: /* i.e. : we recognized a known pattern ! */
          /* Description is complete (or empty) */
          /* Go process the current line */
          state = STATE_IS_FOLDER;
          break;
        }
                                /* Do not read next line. Current one has not been processed */
        read_next = FALSE;
        break;
                                
      case STATE_GET_DESCRIPTION:
        switch (line_type) {
        case MZ_DESCRIPTION_LINE:
          /* Complete description of current entry */
          mozilla_entry_complete_description (node, line);
          /* Is description complete ? */
          state = STATE_IS_DESCRIPTION;
          break;
        default:
          /* impossible : we come from state_is_description */
          g_message ("Error from plan 9 in STATE_GET_DESCRIPTION");
          //                                              g_assert_not_reached();
        }
                                /* Read next line */
        read_next = TRUE;
        break;
                        
      case STATE_IS_FOLDER:
        switch (mz_node_get_type (node)) {
        case BOOKMARK:
          /* Entry is complete. Store it. */
          /* Link node to previous one */
          g_node_append (parent_node, node);
          /* Fetch next entry */
          state = STATE_IS_FOLDER_END;
          break;
        case FOLDER:
          /* Entry is complete. Store it. */
          /* Link node to previous one */
          g_node_append (parent_node, node);
          /* Entry is a folder - Fetch its content */
          state = STATE_DOWN_FOLDER;
          break;
        case TITLE: /* How could we possibly fall here ? */
          /* Entry is a title - Fetch its content ? */
          state = STATE_DOWN_FOLDER;
          break;
        default: /* should be impossible too... */
          g_message ("Mozilla Import Bugged :-(");
          //                                              g_assert_not_reached();
        }
                                /* Do not read next line. Current one has not been processed */
                                /* (Entry we've just stored corresponds to the previous line) */
        read_next = FALSE;
        break;
                                
      case STATE_DOWN_FOLDER:
        switch (line_type) {
        case MZ_START_FOLDER:
          /* Store current parent_node while we process its children */
          stacked_parent_node = g_slist_prepend (stacked_parent_node, parent_node);
          parent_node = node;
          /* Fetch next entry */
          state = STATE_IS_FOLDER_END;
          break;
        default:
          /* We've found a folder name but no content... */
          /* 2 cases :
             - it's an URL disguised as a folder (some default
             bookmarks files have entries like this, bleah...)
             It should be handled when analyzing the line so we
             shouldn't arrive here in this case.
             - it's just a folder name with no content : bad file,
             continue nonetheless...
          */
          g_message ("Mozilla file corrupted : Expecting line type MZ_START_FOLDER");
          //                                              g_assert_not_reached();
        }
                                /* Read next line */
        read_next = TRUE;
        break;
                        
      case STATE_UP_FOLDER:
        switch (line_type) {
        case MZ_STOP_FOLDER:
          /* Retrieve parent node */
          if (stacked_parent_node == NULL) {
            /* Corrupt File : try to continue */
            g_message ("Mozilla file corrupted : Get MZ_STOP_FOLDER without MZ_START_FOLDER");
            //                                                      g_assert_not_reached();
            break;
          }
          node = parent_node;
          parent_node = stacked_parent_node->data;
          stacked_parent_node = g_slist_remove (stacked_parent_node, parent_node);
          break;
        default:
          /* Error caused by a lost neutrino */
          g_message ("Mozilla file corrupted : Expecting line type MZ_STOP_FOLDER");
          //                                              g_assert_not_reached();
        }
                                /* Read next line */
        state = STATE_IS_FOLDER_END;
        read_next = TRUE;
        break;
                        
      default:
                                /* We can't be here except in a distorted space */
        g_message ("We definitely must go and take some fresh air...");
        //                              g_assert_not_reached();
      }
    }
        
  g_free (line);
        
  return node;
}

// Import the bookmarks in file passed as argument.
// Returns TRUE if import is a success
gboolean
mozilla_import(GnobogBookmarks* bookmarks, const gchar* filename)
{
  GnobogBookmarksNode node;
  GnobogBookmarksNode bookmarks_list;
  FILE*               file;
  GnobogBookmarksNode aliasid_node;
  GList*              alias_list;
  GList*              iterator;
  /* gchar*              mime_type; */
  gchar*              line;
  gboolean            end_of_file;
  gint                i;

  g_return_val_if_fail (filename != NULL, FALSE); //FIXME
        
  if (g_file_test (filename, G_FILE_TEST_ISDIR) == TRUE) {
    g_message ("This file %s cannot be a Netscape Bookmarks File : It's a directory.", filename);
    return FALSE;
  }
  
  /* mime_type = (gchar*) gnome_mime_type_of_file (filename);
     g_message ("Mime Type of file %s : %s", filename, mime_type); */

  /* if (strcmp (mime_type, "text/html") != 0) { */
  /* if ( (strcmp (mime_type, "exported SGML document text") != 0)
       && (strcmp (mime_type, "text/html") != 0) ) {
       g_message ("This file %s cannot be a Netscape Bookmarks File : It's not an HTML file.", filename); */
    /* TODO: Warn the user and ask him */
  /*  g_message ("**** We try to import it anyway ****");
    return FALSE;
    } */

  file = fopen (filename, "r");
  if (file == NULL) {
    g_message ("Unable to open file %s : fopen error.", filename);
    return FALSE;
  }

  /* Allocate a buffer for reading file lines */
  line = g_new (gchar, MOZILLA_ENTRY_MAX_SIZE);

  /* Netscape bookmark file identification */
  end_of_file = !mozilla_file_read_next_line (file, line);
  if (end_of_file == TRUE) {
    g_message ("This file %s cannot be a Netscape Bookmarks File : It's empty.", filename);    
    return FALSE;
  }
  if (strcmp (line, MOZILLA_BOOKMARKS_FILE_ID) != 0) {
    g_message ("This file %s cannot be a Netscape Bookmarks File : Bad Signature.", filename);    
    /* TODO: Warn the user and ask him */
    g_message ("**** We try to import it anyway ****");
    //    return FALSE;
  }

  /* Seek bookmarks "<TITLE> </TITLE>" in file
   * FIXME : this stuff is very poor and badly designed...
   */
  do {
    end_of_file = !mozilla_file_read_next_line (file, line);
  } while ((strstr (line, "<TITLE>") == NULL) && !end_of_file);
        
  /* Exit if end of file */
  if (end_of_file == TRUE) {
    g_message ("Mozilla file corrupted : Is this a Mozilla File ? I don't think so :-)");
    /* TODO: Warn the user and ask him */
    g_message ("**** !  We try to import it anyway ! ****");
    //    return FALSE; /* handle error cases */
  } else {
    g_message ("Found title line : %s", line);
  }
  g_free (line);


  // Statistics initialization
  nb_entry = 0;
  nb_title = 0;
  nb_folder = 0;
  nb_bookmark = 0;
  nb_alias = 0;
  nb_separator = 0;
                
  // Go for the import process
  bookmarks_list = mozilla_import_file (file);
  
  /* If something bad occurs, return */
  /* FIXME : Improve error handling */
  if (bookmarks_list == NULL) return FALSE;

  for (i = nb_alias-1; i >= 0; i--) {
    // Get the reference node
    aliasid_node = g_hash_table_lookup (aliasid_table, GINT_TO_POINTER (i));
    // Get the list of reference node aliases
    alias_list = g_hash_table_lookup (aliasof_table, GINT_TO_POINTER (i));
    // Put the reference node at the beginning of the list
    alias_list = g_list_prepend (alias_list, aliasid_node);
    // Attach the alias_list to the reference node and its aliases
    node_set_alias_list (aliasid_node, alias_list);
    // Attach all 'aliasof' to 'aliasid' entry
    iterator = alias_list;
    while ((iterator = g_list_next (iterator)) != NULL) {
      node = iterator->data;
      node->data = aliasid_node->data;
    };      
  }
  g_hash_table_destroy (aliasid_table);
  g_hash_table_destroy (aliasof_table);
    
  g_message("Nb Folder    : %d", nb_folder);      
  g_message("Nb Separator : %d", nb_separator);   
  g_message("Nb Bookmark  : %d", nb_bookmark);    
  g_message("Nb Alias     : %d", nb_alias);       
  g_message("Nb Entry     : %d", nb_entry);       
        
  fclose (file);

  bookmarks->title = g_strdup (node_get_name (bookmarks_list));
  bookmarks->description = g_strdup (node_get_description (bookmarks_list));
  bookmarks->filename = g_strdup (filename);
  /* FIXME : Not clean :-) */
  bookmarks->node = bookmarks_list;

  return TRUE;
}

static void
mozilla_export_description (FILE* file, GnobogBookmarksNode bookmark_node)
{
  //      GList*                          iterator;
  gchar*  description;
  gchar** description_line_array; 
  gchar*  mozilla_description;
  gint    nb_description_line;
  gchar*  p;
        
  description = gnobog_bookmarks_node_get_description (bookmark_node);
  if ((description != NULL) && (strcmp(description, "") != 0)) {
        
    // Calculate the number of line in description
    nb_description_line = 1;
    p = description;
    while ((p = strstr (p, "\n")) != NULL) {
      nb_description_line++;
      p++;
    };
    //              g_message ("NB : %d", nb_description_line);
                
    // Replace '\n' by MOZILLA_DESCRIPTION_SEPARATOR + '\n'
    description_line_array = g_strsplit (description, "\n", nb_description_line);
    mozilla_description = g_strjoinv (MOZILLA_DESCRIPTION_SEPARATOR"\n", description_line_array);
    g_strfreev (description_line_array);

    //              g_message ("%s", mozilla_description);
                
    /* Put description in file */
    /* !!! Beware of % in description !!! */
    fprintf(file, MOZILLA_DESCRIPTION_BEGIN);
    fprintf(file, "%s\n", mozilla_description);
    g_free (mozilla_description);
  }
  return;
}

static void
mozilla_export_entry_recursive (FILE* file, GnobogBookmarksNode bookmarks, gchar* indent, gboolean clear)
{
  GnobogBookmarksNodeType         type;
  GnobogBookmarksNode                     child;
  gchar*                                          new_indent;
  gchar*                                   title;
  GList*                                          alias_list;
  static GList*                           aliasid_index = NULL;
  static gint                             aliasid = 0;

  if (clear == TRUE) {
    /* We are starting a new recursion, so clear static variable */
    aliasid = 0;
    g_list_free (aliasid_index);
    aliasid_index = NULL;
  }

  type = gnobog_bookmarks_node_get_type(bookmarks);
        
  new_indent = g_strconcat(indent, "    ", NULL);
        
  switch (type) {
  case TITLE:
    fprintf(file, MOZILLA_HEADER_1, indent);
    fprintf(file, MOZILLA_HEADER_2, indent);
    fprintf(file, MOZILLA_HEADER_3, indent);
    fprintf(file, MOZILLA_HEADER_4, indent);
    fprintf(file, MOZILLA_HEADER_5, indent);
    title = gnobog_bookmarks_node_get_name(bookmarks);
    if (title == NULL)
      title = "(null)";
    fprintf(file, MOZILLA_TITLE_1, indent, title);
    fprintf(file, MOZILLA_TITLE_2, indent, title);
    fprintf(file, "\n");
    break;
                
  case FOLDER:
    if (gnobog_bookmarks_node_is_folder_open(bookmarks)) {
      fprintf(file, MOZILLA_FOLDER_UNFOLDED, indent,
              gnobog_bookmarks_node_get_creation_time (bookmarks),
              gnobog_bookmarks_node_get_name(bookmarks));
    } else {
      fprintf(file, MOZILLA_FOLDER_FOLDED, indent,
              gnobog_bookmarks_node_get_creation_time (bookmarks),
              gnobog_bookmarks_node_get_name(bookmarks));
    }
    break;
                
  case SEPARATOR:
    fprintf(file, MOZILLA_SEPARATOR, indent);
    break;
                
  case BOOKMARK:
    if (gnobog_bookmarks_node_have_aliases (bookmarks)) {
      alias_list = gnobog_bookmarks_node_get_alias_list (bookmarks);
      if (g_list_index (aliasid_index, bookmarks->data) == -1) {
        fprintf(file, MOZILLA_BOOKMARK_ALIASED, indent,
                gnobog_bookmarks_node_get_location (bookmarks),
                aliasid,
                gnobog_bookmarks_node_get_creation_time (bookmarks),
                gnobog_bookmarks_node_get_visit_time (bookmarks),
                gnobog_bookmarks_node_get_modification_time (bookmarks),
                gnobog_bookmarks_node_get_name (bookmarks));
        aliasid_index = g_list_prepend (aliasid_index, bookmarks->data);
        aliasid++;
      } else {
        fprintf(file, MOZILLA_BOOKMARK_ALIAS, indent,
                gnobog_bookmarks_node_get_location (bookmarks),
                aliasid-g_list_index (aliasid_index, bookmarks->data)-1,
                gnobog_bookmarks_node_get_creation_time (bookmarks),
                gnobog_bookmarks_node_get_visit_time (bookmarks),
                gnobog_bookmarks_node_get_modification_time (bookmarks),
                gnobog_bookmarks_node_get_name (bookmarks));
      }
    } else {
      fprintf(file, MOZILLA_BOOKMARK, indent,
              gnobog_bookmarks_node_get_location (bookmarks),
              gnobog_bookmarks_node_get_creation_time (bookmarks),
              gnobog_bookmarks_node_get_visit_time (bookmarks),
              gnobog_bookmarks_node_get_modification_time (bookmarks),
              gnobog_bookmarks_node_get_name (bookmarks));
    }
    break;
                
  default:
    break;
  }
        
  mozilla_export_description(file, bookmarks);

  if ((type == TITLE) || (type == FOLDER)) {
    fprintf(file, MOZILLA_FOLDER_START, indent);
    if (!G_NODE_IS_LEAF(bookmarks)) {
      child = g_node_first_child (bookmarks);
      do {
        mozilla_export_entry_recursive (file, child, new_indent, FALSE);
      } while ((child = g_node_next_sibling(child)) != NULL);
    }
    fprintf(file, MOZILLA_FOLDER_STOP, indent);
  }                       
        
  g_free(new_indent);
  return;
}

gboolean
mozilla_export(GnobogBookmarks* bookmarks, gchar* filename)
{
  FILE* file;
  gchar* backup_filename;

  if (g_file_test (filename, G_FILE_TEST_ISDIR) == TRUE) {
    g_message ("We can save on a directory (%s)", filename);
    return FALSE;
  }
  
  if (g_file_test (filename, G_FILE_TEST_ISFILE ||G_FILE_TEST_ISLINK ) == TRUE) {
    /* FIXME : Ask user */
    backup_filename = g_strdup_printf ("%s.bak", filename);
    g_message ("Instead of overwriting existing file (%s), rename it %s", filename, backup_filename);
    /* FIXME : Do not look at return value : For now we don't care... */
    (void)rename (filename, backup_filename);
    g_free (backup_filename);
  }
  
  file = fopen (filename, "w");
  if (file == NULL) {
    g_message ("Unable to save file %s : fopen error.", filename);
    return FALSE;
  }
        
  /* The following is pretty f*cked up */
  g_free (node_get_name (bookmarks->node));
  g_free (node_get_description (bookmarks->node));
  node_set_name (bookmarks->node, g_strdup (bookmarks->title));
  node_set_description (bookmarks->node, g_strdup (bookmarks->description));
  mozilla_export_entry_recursive (file, bookmarks->node, "", TRUE);
        
  fclose(file);
  return TRUE;
}
