/* gnobog_bookmarks-private.h
 *
 * Copyright (C) 2000 Frdric LESPEZ & Renaud CHAILLAT
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#ifndef GNOBOG_BOOKMARKS_PRIVATE_H
#define GNOBOG_BOOKMARKS_PRIVATE_H

#include "gnobog_bookmarks.h"

/** GnobogBookmarksEntry
    A structure describing a bookmark entry in the list of bookmarks
*/

typedef struct _GnobogBookmarksEntry* GnobogBookmarksEntry;
struct _GnobogBookmarksEntry {
  GnobogBookmarksNodeType  type;         /* Entry Type                  */
  gchar*                   name;         /* Entry Name                  */
  gchar*                   location;     /* Entry Location              */
  gchar*                   description;  /* Entry Description           */
  GTime                    created;      /* Entry Creation Time         */
  GTime                    visited;      /* Entry Visitation Time       */
  GTime                    modified;     /* Entry Modification Time     */
  gboolean                 open;         /* Folder State                */
  GList*                   alias_list;   /* Pointer to Entry Alias List */
};

// Convenience macros to get node informations
#define node_get_type(node)               ((GnobogBookmarksEntry) (node->data))->type
#define node_get_name(node)               ((GnobogBookmarksEntry) (node->data))->name
#define node_get_location(node)           ((GnobogBookmarksEntry) (node->data))->location
#define node_get_description(node)        ((GnobogBookmarksEntry) (node->data))->description
#define node_get_creation_time(node)      ((GnobogBookmarksEntry) (node->data))->created
#define node_get_modification_time(node)  ((GnobogBookmarksEntry) (node->data))->modified
#define node_get_visit_time(node)         ((GnobogBookmarksEntry) (node->data))->visited
#define node_get_folder_state(node)       ((GnobogBookmarksEntry) (node->data))->open
#define node_get_alias_list(node)         ((GnobogBookmarksEntry) (node->data))->alias_list

// Convenience macros to set node informations
#define node_set_name(node, val)         ((GnobogBookmarksEntry) (node->data))->name = val
#define node_set_location(node, val)     ((GnobogBookmarksEntry) (node->data))->location = val
#define node_set_description(node, val)  ((GnobogBookmarksEntry) (node->data))->description = val
#define node_set_alias_list(node, val)   ((GnobogBookmarksEntry) (node->data))->alias_list = val

// Convenience macro to folder node is open or close
#define node_is_folder_open(node)  ((GnobogBookmarksEntry) (node->data))->open

// Convenience macro to test aliases existence
#define node_have_aliases(node)  (g_list_length (((GnobogBookmarksEntry) (node->data))->alias_list) > 1)

// Convenience macros to 'touch' node timestamps
#define node_touch_creation_time(node)      ((GnobogBookmarksEntry) (node->data))->created = time(NULL)
#define node_touch_modification_time(node)  ((GnobogBookmarksEntry) (node->data))->modified = time(NULL)
#define node_touch_visit_time(node)         ((GnobogBookmarksEntry) (node->data))->visited = time(NULL)

// Forward declarations of private methods
GnobogBookmarksNode gnobog_bookmarks_node_import
                                    (GnobogBookmarksNodeType type,
                                     gchar* name,
                                     gchar* location,
                                     gchar* description,
                                     guint created,
                                     guint visited,
                                     guint modified,
                                     gboolean open);
GnobogBookmarksNode gnobog_bookmarks_node_import_separator
                                    (void);
void                gnobog_bookmarks_node_update
                                    (GnobogBookmarks* bookmarks,
                                     GnobogBookmarksNode node,
                                     gchar* name,
                                     gchar* location,
                                     gchar* description);

GnobogBookmarksInsertMode gnobog_bookmarks_node_get_insert_default_mode (GnobogBookmarks* bookmarks, GnobogBookmarksNode node);



/* Node Manipulation */
void                gnobog_bookmarks_node_link
                                    (GnobogBookmarks* bookmarks,
                                     GnobogBookmarksNode anchor_node,
                                     GnobogBookmarksInsertMode mode,
                                     GnobogBookmarksNode node);
void                gnobog_bookmarks_list_link
                                    (GnobogBookmarks* bookmarks,
                                     GnobogBookmarksNode anchor_node,
                                     GnobogBookmarksInsertMode mode,
                                     GList* list);
GnobogBookmarksNode gnobog_bookmarks_node_duplicate
                                    (GnobogBookmarks* bookmarks,
                                     GnobogBookmarksNode node);
GList*              gnobog_bookmarks_list_duplicate
                                    (GnobogBookmarks* bookmarks,
                                     GList* list);
void                gnobog_bookmarks_node_unlink
                                    (GnobogBookmarks* bookmarks,
                                     GnobogBookmarksNode node);
void                gnobog_bookmarks_list_unlink
                                    (GnobogBookmarks* bookmarks,
                                     GList* list);
void                gnobog_bookmarks_node_destroy
                                    (GnobogBookmarks* bookmarks,
                                     GnobogBookmarksNode node);
void                gnobog_bookmarks_list_destroy
                                    (GnobogBookmarks* bookmarks,
                                     GList* list);

void    gnobog_bookmarks_node_remove_alias              (GnobogBookmarksNode node);

gboolean gnome_bookmarks_list_is_insertion_valid (GnobogBookmarks* bookmarks, GnobogBookmarksNode node, GnobogBookmarksInsertMode mode, GList* list);
GList* gnobog_bookmarks_list_simplify (GnobogBookmarks* bookmarks, GList* list);


/* These functions shouldn't appear in an API, even private one */
void    gnobog_bookmarks_node_link_after        (GnobogBookmarks* bookmarks, GnobogBookmarksNode anchor_node, GnobogBookmarksNode node);
void    gnobog_bookmarks_node_link_before       (GnobogBookmarks* bookmarks, GnobogBookmarksNode anchor_node, GnobogBookmarksNode node);
void    gnobog_bookmarks_node_link_into (GnobogBookmarks* bookmarks, GnobogBookmarksNode anchor_node, GnobogBookmarksNode node);

GnobogBookmarksEntry    gnobog_bookmarks_entry_new              (GnobogBookmarksNodeType type, gchar* name, gchar* location, gchar* description);
GnobogBookmarksEntry    gnobog_bookmarks_entry_import           (GnobogBookmarksNodeType type, gchar* name, gchar* location, gchar* description, guint created, guint visited, guint modified, gboolean folded);
GnobogBookmarksEntry    gnobog_bookmarks_entry_duplicate        (GnobogBookmarksEntry entry);
void                    gnobog_bookmarks_entry_destroy          (GnobogBookmarksEntry entry);

#endif
