-----------------------------------------------------------------------
--                               G P S                               --
--                                                                   --
--                      Copyright (C) 2008, AdaCore                  --
--                                                                   --
-- GPS is free  software;  you can redistribute it and/or modify  it --
-- under the terms of the GNU General Public License as published by --
-- the Free Software Foundation; either version 2 of the License, or --
-- (at your option) any later version.                               --
--                                                                   --
-- This program is  distributed in the hope that it will be  useful, --
-- but  WITHOUT ANY WARRANTY;  without even the  implied warranty of --
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU --
-- General Public License for more details. You should have received --
-- a copy of the GNU General Public License along with this program; --
-- if not,  write to the  Free Software Foundation, Inc.,  59 Temple --
-- Place - Suite 330, Boston, MA 02111-1307, USA.                    --
-----------------------------------------------------------------------

with GNAT.Templates;  use GNAT.Templates;
with Ada.Text_IO;     use Ada.Text_IO;

procedure Templates is
   Delimiter : constant Character := '%';

   function Callback (Name : String; Quoted : Boolean) return String;
   function Callback (Name : String; Quoted : Boolean) return String is
   begin
      if Name = "version" then
         return "6.0";
      elsif Name = "^" then
         return "previous";
      elsif Name = "prev" then
         return Delimiter & "^";
      elsif Name = "1" then
         return "first";
      elsif Name = "1-" then
         return "allargs";
      elsif Name = "*" then
         return "all";
      else
         raise Invalid_Substitution;
      end if;
   end Callback;

   Substrings : constant Substitution_Array :=
     ((new String'("tool"), new String'("GNAT")),
      (new String'("previous"), new String'(Delimiter & "prev")));

   procedure Assert
     (Str1, Str2 : String; Msg : String; Recurse : Boolean := False;
      Errors : Error_Handling := Keep_As_Is);

   procedure Assert
     (Str1, Str2 : String; Msg : String; Recurse : Boolean := False;
      Errors : Error_Handling := Keep_As_Is)
   is
      Str3 : constant String := Substitute
        (Str1, Substrings, Callback'Unrestricted_Access, Delimiter, Recurse,
         Errors);
   begin
      if Str2 /=  Str3 then
         Put_Line ("ERROR: " & Msg);
         Put_Line ("Expected: " & Str2);
         Put_Line ("Got     : " & Str3);
      end if;
   end Assert;

begin
   Assert ("abcd", "abcd", "no substitution");
   Assert ("abc%", "abc%", "delimiter is last");
   Assert ("abc%%", "abc%", "doubling delimiter");
   Assert ("%%abc", "%abc", "substitute at beginning of string");
   Assert ("%%%%abc", "%%abc", "substitute multiple times");

   Assert ("a%ret",  "a%ret",    "no substitution");
   Assert ("a%previ", "a%previ", "no substitution although start matches");

   Assert ("v=%version", "v=6.0", "replace version");
   Assert ("v=%{version}", "v=6.0", "replace version with curly braces");
   Assert ("v=%{version}a", "v=6.0a", "replace version followed by char");

   Assert ("t=%tool", "t=GNAT", "replace through array");
   Assert ("t=%tool%version", "t=GNAT6.0", "replace multiple times");

   Assert ("t=%prev", "t=%^", "non recursive replace");
   Assert ("t=%prev", "t=previous", "recursive replace", True);
   Assert ("t=%previous", "t=%prev", "non recursive replace and array");
   Assert ("t=%previous", "t=previous", "recursive replace and array", True);

   Assert ("t=%*ab", "t=allab", "All args");
   Assert ("t=%1ab", "t=firstab", "First arg");
   Assert ("t=%1-ab", "t=allargsab", "All args after first");

   Assert ("t=%{var:-default}", "t=default",  "Default value");

   Assert ("t=%{var:=default}", "t=%{var:=default}",  "Default value",
           Errors => Keep_As_Is);
   Assert ("t=%{var:=default}", "t=",  "Default value",
           Errors => Replace_With_Empty);

end Templates;
