with Entities;          use Entities;
with Entities.Debug;    use Entities.Debug;
with Entities.Queries;  use Entities.Queries;
with VFS;               use VFS;
with Projects;          use Projects;
with Projects.Editor;   use Projects.Editor;
with Projects.Registry; use Projects.Registry;
with Traces;            use Traces;
with ALI_Parser;        use ALI_Parser;
with CPP_Parser;        use CPP_Parser;
with Ada.Command_Line;  use Ada.Command_Line;
with GNAT.OS_Lib;       use GNAT.OS_Lib;
with Ada.Text_IO;       use Ada.Text_IO;
with Ada.Calendar;      use Ada.Calendar;
with GNAT.Directory_Operations; use GNAT.Directory_Operations;
with Test_Parse_Support; use Test_Parse_Support;
with Language_Handlers.Glide; use Language_Handlers.Glide;
with Prj;               use Prj;

--  This scripts takes a single arguments in parameter:
--     test_parse <command_file>
--  The syntax of the command file is line-oriented: empty lines are ignored,
--  as well as lines starting with exactly "--".
--  Comment lines are duplicated to the output.
--  Other lines must start with one of the following commands, and contain a
--  valid syntax for that line. Very few error handling is made in the parsing
--  of this file, and an invalid file will likely crash this program.
--  Valid commands are:
--      GENERATE_DB
--          Generate the on-disk cross-references database for languages where
--          this applies
--      CALL_GRAPH
--          Compute the call graph for all parsed files
--      PARSE_ALL [sources] [+call_graph]
--          Parse all source files in the project. If the keyword "sources" is
--          specified, the list of files to parse is computed from the list of
--          source files.
--          If the keyword "+call_graph" is present, then the call graph is
--          also computed. This is exclusive with the previous parameter.
--      PARSE source_file
--          Forces an update of the LI information for source_file
--      PAUSE delay
--          Pause for delay seconds. This is useful if you need to mesure
--          for instance the amount of memory occupied by the application.
--      DECL name:file:line:column
--          Outputs the declaration for the entity,
--          as "DECL: name:file:line:col"
--      BODY name:file:line:column
--          Outputs all the bodies for the entity
--      REFS name:file:line:column
--          Outputs all known references to the entity in the first part
--      DEP file
--          Print files that file depends on
--      DEP_ON file
--          Print files depending on file
--      PARAMS name:file:line:column
--          Print the list of parameters for that entity
--      CALLERS name:file:line:column
--          Print the list of entities that call that entity
--      CALLS name:file:line:column
--          Print the list of entities that are called by that entity
--      ENTITIES file
--          List all entities referenced in file that start with prefix
--      DUMP
--          Dump the contents of the cross-references information file
--      DUMP_ENTITY name:file:line:column
--          Dump the description of a specific entity
--      PROJECT project_name
--          Load a specific project. This replaces the currently loaded project
--          Initially, a default project is used, that only uses "." for the
--          source and object directory.
--      TOUCH file
--          Will force a reparsing of the LI information for this file

procedure Test_Parse is
   Registry     : aliased Project_Registry;
   Db           : Entities_Database;
   LI_Not_Found : File_Error_Reporter;
   ALI, CPP_LI  : LI_Handler;

   procedure Load_Project (Name : String);
   procedure Parse_Cmd_File (Name : String);
   procedure Parse_Decl_Line (Text : String);
   procedure Parse_Body_Line (Text : String);
   procedure Parse_Refs_Line (Text : String);
   procedure Parse_Callers_Line (Text : String);
   procedure Parse_Calls_Line (Text : String);
   procedure Parse_Params_Line (Text : String);
   procedure Parse_Dep_On_Line (Text : String);
   procedure Parse_Dep_Line (Text : String);
   procedure Parse_Entities (Text : String);
   procedure Parse_Dump_Entity (Text : String);
   procedure Parse_All (Text : String);
   procedure Pause (For_Delay : String);
   function Image (J : Integer) return String;
   function Image (Loc : File_Location) return String;
   function Image (Entity : Entity_Information) return String;
   function Image (S : Source_File) return String;
   procedure Parse_Entity
     (Text   : String;
      Index  : in out Natural;
      Name   : out String_Access;
      File   : out Virtual_File;
      Line   : out Natural;
      Column : out Natural;
      Has_Name : Boolean := True);

   procedure Compute_Predefined_Paths;
   --  Compute the predefined paths for the runtime

   function Create_Lang_Handler return Glide_Language_Handler;
   --  Create a language handler

   ------------------------------
   -- Compute_Predefined_Paths --
   ------------------------------

   procedure Compute_Predefined_Paths is
      Gnatls      : constant String := "gnatls";
      Gnatls_Args : Argument_List_Access :=
        Argument_String_To_List (Gnatls & " -v");
      Path        : String_Access := Locate_Exec_On_Path (Gnatls_Args (1).all);
      GNAT_Version : aliased String_Access;

   begin
      if Path /= null then
         Compute_Predefined_Paths
           (Registry,
            GNAT_Version'Access,
            Path.all,
            Gnatls_Args);

         Free (Path);
         Free (GNAT_Version);
      else
         Set_Predefined_Source_Path (Registry, "");
         Set_Predefined_Object_Path (Registry, "");
      end if;

      Free (Gnatls_Args);
   end Compute_Predefined_Paths;

   -------------------------
   -- Create_Lang_Handler --
   -------------------------

   function Create_Lang_Handler return Glide_Language_Handler is
      Handler : Glide_Language_Handler;
   begin
      Gtk_New (Handler);

      Register_Language (Handler, "ada", null);
      Prj.Register_Default_Naming_Scheme
        (Language            => Get_String ("ada"),
         Default_Spec_Suffix => Get_String (".ads"),
         Default_Body_Suffix => Get_String (".adb"));

      Register_Language (Handler, "c",   null);
      Prj.Register_Default_Naming_Scheme
        (Language            => Get_String ("c"),
         Default_Spec_Suffix => Get_String (".h"),
         Default_Body_Suffix => Get_String (".c"));

      Register_Language (Handler, "c++", null);
      Prj.Register_Default_Naming_Scheme
        (Language            => Get_String ("c++"),
         Default_Spec_Suffix => Get_String (".h"),
         Default_Body_Suffix => Get_String (".cc"));

      Set_Registry (Handler, Registry'Unrestricted_Access);
      return Handler;
   end Create_Lang_Handler;

   ------------------
   -- Load_Project --
   ------------------

   procedure Load_Project (Name : String) is
      New_Project_Loaded : Boolean;
   begin
      Load (Registry, Name, null, New_Project_Loaded);
      Recompute_View (Registry, null);
      Compute_Predefined_Paths;
      On_Project_View_Changed (CPP_LI);
   end Load_Project;

   -----------
   -- Image --
   -----------

   function Image (Entity : Entity_Information) return String is
   begin
      return Get_Name (Entity).all & ':' & Image (Get_Declaration_Of (Entity));
   end Image;

   -----------
   -- Image --
   -----------

   function Image (J : Integer) return String is
      S : constant String := Integer'Image (J);
   begin
      if J < 0 then
         return S;
      else
         return S (S'First + 1 .. S'Last);
      end if;
   end Image;

   -----------
   -- Image --
   -----------

   function Image (S : Source_File) return String is
   begin
      return Base_Name (Get_Filename (S));
   end Image;

   -----------
   -- Image --
   -----------

   function Image (Loc : File_Location) return String is
   begin
      return Base_Name (Get_Filename (Get_File (Loc)))
         & ':' & Image (Get_Line (Loc)) & ':' & Image (Get_Column (Loc));
   end Image;

   ------------------
   -- Parse_Entity --
   ------------------

   procedure Parse_Entity
     (Text   : String;
      Index  : in out Natural;
      Name   : out String_Access;
      File   : out Virtual_File;
      Line   : out Natural;
      Column : out Natural;
      Has_Name : Boolean := True)
   is
      Start : Natural := Index;
   begin
      if Has_Name then
         while Text (Index) /= ':' loop
            Index := Index + 1;
         end loop;

         Name := new String'(Text (Start .. Index - 1));

         Index := Index + 1;
         Start := Index;
      end if;

      while Text (Index) /= ':' loop
         Index := Index + 1;
      end loop;

      File := Create (Normalize_Pathname (Text (Start .. Index - 1)));

      Index := Index + 1;
      Start := Index;
      while Text (Index) /= ':' loop
         Index := Index + 1;
      end loop;

      Line := Integer'Value (Text (Start .. Index - 1));

      Index := Index + 1;
      Start := Index;
      while Index <= Text'Last and then Text (Index) /= ' ' loop
         Index := Index + 1;
      end loop;

      Column := Integer'Value (Text (Start .. Index - 1));
   end Parse_Entity;

   ---------------------
   -- Parse_Decl_Line --
   ---------------------

   procedure Parse_Decl_Line (Text : String) is
      Index  : Natural := Text'First;
      Name   : String_Access;
      File   : Virtual_File;
      Line, Column : Natural;

      Entity : Entity_Information;
      Status : Find_Decl_Or_Body_Query_Status;
   begin
      Parse_Entity (Text, Index, Name, File, Line, Column);
      Find_Declaration (Db, File, Name.all, Line, Column, Entity, Status);
      Put_Line ("DECL: " & Image (Entity) & ' ' & Status'Img);
   end Parse_Decl_Line;

   ---------------------
   -- Parse_Body_Line --
   ---------------------

   procedure Parse_Body_Line (Text : String) is
      Index  : Natural := Text'First;
      Name   : String_Access;
      File   : Virtual_File;
      Line, Column : Natural;

      Entity : Entity_Information;
      Status : Find_Decl_Or_Body_Query_Status;
      Loc    : File_Location := No_File_Location;
      First_Loc : File_Location := No_File_Location;
   begin
      Parse_Entity (Text, Index, Name, File, Line, Column);
      Find_Declaration (Db, File, Name.all, Line, Column, Entity, Status);

      loop
         Find_Next_Body (Entity, Current_Location => Loc, Location => Loc);
         exit when Loc = First_Loc;

         if First_Loc = No_File_Location then
            First_Loc := Loc;
         end if;

         Put_Line ("BODY: " & Image (Loc) & ' ' & Status'Img);
      end loop;
   end Parse_Body_Line;

   -----------------------
   -- Parse_Dump_Entity --
   -----------------------

   procedure Parse_Dump_Entity (Text : String) is
      Index  : Natural := Text'First;
      Name   : String_Access;
      File   : Virtual_File;
      Line, Column : Natural;
      Entity : Entity_Information;
      Status : Find_Decl_Or_Body_Query_Status;
   begin
      Parse_Entity (Text, Index, Name, File, Line, Column);
      Find_Declaration (Db, File, Name.all, Line, Column, Entity, Status);

      if Entity = null then
         Put_Line ("Couldn't find the entity " & Text);
      else
         Dump (Entity, Full => True, Name => "");
      end if;
   end Parse_Dump_Entity;

   ---------------------
   -- Parse_Refs_Line --
   ---------------------

   procedure Parse_Refs_Line (Text : String) is
      Index  : Natural := Text'First;
      Name   : String_Access;
      File   : Virtual_File;
      Line, Column : Natural;

      Entity : Entity_Information;
      Status : Find_Decl_Or_Body_Query_Status;
      Ref     : Entity_Reference := No_Entity_Reference;
      Iter   : Entity_Reference_Iterator;
      Renamed : Entity_Information;
   begin
      Parse_Entity (Text, Index, Name, File, Line, Column);
      Find_Declaration (Db, File, Name.all, Line, Column, Entity, Status);

      if Entity = null then
         Put_Line ("Couldn't find entity ");
      else
         Renamed := Renaming_Of (Entity);
         if Renamed /= null then
            Put_Line ("RENAMES: " & Image (Renamed));
         end if;

         Find_All_References
           (Iter,
            Entity                => Entity,
            File_Has_No_LI_Report => null,
            In_File               => null);
         while not At_End (Iter) loop
            Ref := Get (Iter);
            if Ref /= No_Entity_Reference then
               Put_Line ("REF: " & Image (Get_Location (Ref)));
            end if;

            Next (Iter);
         end loop;

         Destroy (Iter);
      end if;
   end Parse_Refs_Line;

   -----------------------
   -- Parse_Dep_On_Line --
   -----------------------

   procedure Parse_Dep_On_Line (Text : String) is
      File : constant Virtual_File := Create (Normalize_Pathname (Text));
      Iter : Dependency_Iterator;
      Dep  : Source_File;
   begin
      Find_Ancestor_Dependencies
        (Iter,
         File                  => Get_Or_Create (Db, File),
         File_Has_No_LI_Report => LI_Not_Found);

      while not At_End (Iter) loop
         Dep := Get (Iter);
         if Dep /= null then
            Put_Line ("DEP_ON (" & Base_Name (File) & "): " & Image (Dep)
                      & " Explicit=" & Is_Explicit (Iter)'Img);
         end if;

         Next (Iter);
      end loop;

      Destroy (Iter);
   end Parse_Dep_On_Line;

   --------------------
   -- Parse_Dep_Line --
   --------------------

   procedure Parse_Dep_Line (Text : String) is
      File : constant Virtual_File := Create (Normalize_Pathname (Text));
      Iter : File_Dependency_Iterator;
   begin
      Find_Dependencies
        (Iter,
         File                  => Get_Or_Create (Db, File),
         File_Has_No_LI_Report => LI_Not_Found);

      while not At_End (Iter) loop
         Put_Line ("DEP (" & Base_Name (File) & "): " & Image (Get (Iter))
                   & " Explicit=" & Is_Explicit (Iter)'Img);
         Next (Iter);
      end loop;
   end Parse_Dep_Line;

   ----------------------
   -- Parse_Calls_Line --
   ----------------------

   procedure Parse_Calls_Line (Text : String) is
      Index  : Natural := Text'First;
      Name   : String_Access;
      File   : Virtual_File;
      Line, Column : Natural;
      Entity : Entity_Information;
      Status : Find_Decl_Or_Body_Query_Status;

      Iter   : Calls_Iterator;
      Called : Entity_Information;
      Refs   : Entity_Reference_Iterator;
   begin
      Parse_Entity (Text, Index, Name, File, Line, Column);
      Find_Declaration (Db, File, Name.all, Line, Column, Entity, Status);

      if Entity = null then
         Put_Line ("CALLS <unknown entity>");
      else
         Put_Line ("CALLS " & Get_Name (Entity).all);

         Iter := Get_All_Called_Entities (Entity);
         while not At_End (Iter) loop
            Called := Get (Iter);

            Find_All_References
              (Iter     => Refs,
               Entity   => Called,
               In_Scope => Entity);

            Put ("  => " & Image (Called) & " (Refs: ");

            while not At_End (Refs) loop
               if Get (Refs) /= No_Entity_Reference then
                  Put (Image (Get_Location (Get (Refs))) & ' ');
               end if;
               Next (Refs);
            end loop;

            Destroy (Refs);

            Put_Line (")");

            Next (Iter);
         end loop;
      end if;

      Destroy (Iter);
   end Parse_Calls_Line;

   ------------------------
   -- Parse_Callers_Line --
   ------------------------

   procedure Parse_Callers_Line (Text : String) is
      Index  : Natural := Text'First;
      Name   : String_Access;
      File   : Virtual_File;
      Line, Column : Natural;
      Entity : Entity_Information;
      Status : Find_Decl_Or_Body_Query_Status;

      Iter   : Entity_Reference_Iterator;
      Ref    : Entity_Reference;
      Caller : Entity_Information;
   begin
      Parse_Entity (Text, Index, Name, File, Line, Column);
      Find_Declaration (Db, File, Name.all, Line, Column, Entity, Status);

      if Entity = null then
         Put_Line ("ENTITY NOT FOUND " & Name.all & ' ' & Status'Img);
         return;
      end if;

      Put_Line ("CALLERS " & Image (Entity));

      Find_All_References
        (Iter                  => Iter,
         Entity                => Entity,
         File_Has_No_LI_Report => null);
      while not At_End (Iter) loop
         Ref := Get (Iter);
         if Ref /= No_Entity_Reference then
            Caller := Get_Caller (Ref);
            if Caller /= null then
               Put_Line ("  REF: " & Image (Get_Location (Ref))
                         & " => " & Image (Caller));
            end if;
         end if;

         Next (Iter);
      end loop;

      Destroy (Iter);
   end Parse_Callers_Line;

   --------------------
   -- Parse_Entities --
   --------------------

   procedure Parse_Entities (Text : String) is
      VF : constant Virtual_File :=
         Create (Base_Name (Text),
                 Registry,
                 Use_Object_Path => False);
      File : constant Source_File :=
        Get_Source_Info (Get_LI_Handler (Db, VF), VF);
      Iter : Entity_Iterator;
   begin
      Find_All_Entities_In_File
        (Iter   => Iter,
         File   => File,
         Prefix => "");

      Put_Line ("ENTITIES " & Text);
      while not At_End (Iter) loop
         Put_Line ("  " & Image (Get (Iter)));
         Next (Iter);
      end loop;

      Destroy (Iter);
   end Parse_Entities;

   -----------------------
   -- Parse_Params_Line --
   -----------------------

   procedure Parse_Params_Line (Text : String) is
      Index  : Natural := Text'First;
      Name   : String_Access;
      File   : Virtual_File;
      Line, Column : Natural;
      Entity, Param : Entity_Information;
      Iter   : Subprogram_Iterator;
      Status : Find_Decl_Or_Body_Query_Status;
   begin
      Parse_Entity (Text, Index, Name, File, Line, Column);
      Find_Declaration (Db, File, Name.all, Line, Column, Entity, Status);

      Iter := Get_Subprogram_Parameters (Entity, LI_Not_Found);
      loop
         Get (Iter, Param);
         exit when Param = null;

         Put_Line ("Param (" & Get_Name (Entity).all & "): "
                   & Image (Param) & " as " & Get_Type (Iter)'Img);

         Next (Iter);
      end loop;
   end Parse_Params_Line;

   ---------------
   -- Parse_All --
   ---------------

   procedure Parse_All (Text : String) is
      Parse_From_Sources : constant Boolean := Text = "sources";
      Call_Graph         : constant Boolean := Text = "+call_graph";
      Start  : Time := Clock;
      Count  : Natural := 0;
      Files  : File_Array_Access;
      Source : Source_File;
      pragma Unreferenced (Source);
   begin
      if Parse_From_Sources then
         Files := Get_Source_Files
           (Get_Root_Project (Registry), Recursive => True);
         Start := Clock;
         if Files /= null then
            for F in Files'Range loop
               Source := Get_Source_Info
                 (Get_LI_Handler (Db, Files (F)), Files (F));
               Count  := Count + 1;
            end loop;
         end if;

         Put_Line ("Time=" & Duration'Image (Clock - Start)
                   & " (" & Count'Img & " files)");

      else
         Count := Parse_All_LI_Information
           (Handler   => ALI,
            Project   => Get_Root_Project (Registry),
            Recursive => True);
         Put_Line ("Time for Ada=" & Duration'Image (Clock - Start)
                   & " (" & Count'Img & " files)");
         Start := Clock;
         Count := Parse_All_LI_Information
           (Handler   => CPP_LI,
            Project   => Get_Root_Project (Registry),
            Recursive => True);
         Put_Line ("Time for C=" & Duration'Image (Clock - Start)
                   & " (" & Count'Img & " files)");
      end if;

      if Call_Graph then
         Start := Clock;
         Compute_All_Call_Graphs (Db);
         Put_Line ("Time for Callgraph=" & Duration'Image (Clock - Start));
      end if;
   end Parse_All;

   -----------
   -- Pause --
   -----------

   procedure Pause (For_Delay : String) is
      D : constant Duration := Duration'Value (For_Delay);
   begin
      delay D;
   end Pause;

   --------------------
   -- Parse_Cmd_File --
   --------------------

   procedure Parse_Cmd_File (Name : String) is
      File   : File_Type;
      Line   : String (1 .. 5096);
      Last   : Integer;
      VF     : Virtual_File;
      Source : Source_File;
   begin
      Open (File, In_File, Name);

      while not End_Of_File (File) loop
         Get_Line (File, Line, Last);

         if Last < 1
           or else Line (Line'First .. Line'First + 1) = "--"
         then
            Put_Line (Line (Line'First .. Last));

         elsif Last > 5
           and then Line (Line'First .. Line'First + 4) = "DECL "
         then
            Parse_Decl_Line (Line (Line'First + 5 .. Last));

         elsif Last > 5
           and then Line (Line'First .. Line'First + 4) = "BODY "
         then
            Parse_Body_Line (Line (Line'First + 5 .. Last));

         elsif Last > 5
           and then Line (Line'First .. Line'First + 4) = "REFS "
         then
            Parse_Refs_Line (Line (Line'First + 5 .. Last));

         elsif Last > 9
           and then Line (Line'First .. Line'First + 8) = "ENTITIES "
         then
            Parse_Entities (Line (Line'First + 9 .. Last));

         elsif Last > 6
           and then Line (Line'First .. Line'First + 5) = "PAUSE "
         then
            Pause (Line (Line'First + 6 .. Last));

         elsif Last >= 9
           and then Line (Line'First .. Line'First + 8) = "PARSE_ALL"
         then
            if Last >= Line'First + 10 then
               Parse_All (Line (Line'First + 10 .. Last));
            else
               Parse_All ("");
            end if;

         elsif Last > 6
           and then Line (Line'First .. Line'First + 5) = "PARSE "
         then
            VF := Create (Base_Name (Line (Line'First + 6 .. Last)),
                          Registry,
                          Use_Object_Path => False);
            Source := Get_Source_Info (Get_LI_Handler (Db, VF), VF);

         elsif Last = 11
           and then Line (Line'First .. Line'First + 10) = "GENERATE_DB"
         then
            --  ??? Should do this for all registered handlers
            declare
               LI_Iterator : LI_Handler_Iterator'Class :=
                 Generate_LI_For_Project
                   (CPP_LI, Get_Root_Project (Registry), True); -- recursive
               Finished    : Boolean;
            begin
               loop
                  Continue (LI_Iterator, Finished);
                  exit when Finished;
                  delay 0.01;
               end loop;
               Destroy (LI_Iterator);
            end;

         elsif Last = 4
           and then Line (Line'First .. Line'First + 3) = "DUMP"
         then
            Dump (Db);

         elsif Last > 11
           and then Line (Line'First .. Line'First + 11) = "DUMP_ENTITY "
         then
            Parse_Dump_Entity (Line (Line'First + 12 .. Last));

         elsif Last > 5
           and then Line (Line'First .. Line'First + 5) = "TOUCH "
         then
            VF := Create (Base_Name (Line (Line'First + 6 .. Last)),
                          Registry,
                          Use_Object_Path => False);
            Source := Get_Or_Create (Db, VF);
            Set_Time_Stamp (Source,
                            Time_Of (Year  => Year_Number'First,
                                     Month => Month_Number'First,
                                     Day   => Day_Number'First));
            Reset (Source);

            if Get_LI (Source) /= null then
               Set_Time_Stamp
                 (Get_LI (Source),
                  Time_Of (Year  => Year_Number'First,
                           Month => Month_Number'First,
                           Day   => Day_Number'First));
            end if;

         elsif Line (Line'First .. Last) = "CALL_GRAPH" then
            Compute_All_Call_Graphs (Db);

         elsif Last > 6
           and then Line (Line'First .. Line'First + 6) = "DEP_ON "
         then
            Parse_Dep_On_Line (Line (Line'First + 7 .. Last));

         elsif Last > 3
           and then Line (Line'First .. Line'First + 3) = "DEP "
         then
            Parse_Dep_Line (Line (Line'First + 4 .. Last));

         elsif Last > 7
           and then Line (Line'First .. Line'First + 6) = "PARAMS "
         then
            Parse_Params_Line (Line (Line'First + 7 .. Last));

         elsif Last > 8
           and then Line (Line'First .. Line'First + 7) = "CALLERS "
         then
            Parse_Callers_Line (Line (Line'First + 8 .. Last));

         elsif Last > 6
           and then Line (Line'First .. Line'First + 5) = "CALLS "
         then
            Parse_Calls_Line (Line (Line'First + 6 .. Last));

         elsif Last > 7
           and then Line (Line'First .. Line'First + 7) = "PROJECT "
         then
            Load_Project (Line (Line'First + 8 .. Last));

         else
            Put_Line ("Unknown command line: " & Line (Line'First .. Last));
         end if;
      end loop;

      Close (File);
   end Parse_Cmd_File;

   Handler      : Glide_Language_Handler;

begin
   Parse_Config_File (".gnatdebug");
   LI_Not_Found := new Test_LI_Not_Found_Record;

   Projects.Registry.Initialize;

   Db := Create (Registry'Unchecked_Access);

   Handler := Create_Lang_Handler;

   Load_Default_Project (Registry, Get_Current_Dir);
   Recompute_View (Registry, null);
   Compute_Predefined_Paths;

   Set_Show_Timestamp (False);

   ALI := Create_ALI_Handler (Db, Registry);
   Register_LI_Handler (Handler, "Ada", ALI);
   Set_Language_Handler
     (Handler, "Ada", Get_LI_Handler_By_Name (Handler, "Ada"));
   Register_Default_Language_Extension (Registry, "Ada", ".ads", ".adb");

   CPP_LI := Create_CPP_Handler (Db, Registry);
   Register_LI_Handler (Handler, "C", CPP_LI);
   Set_Language_Handler
     (Handler, "C", Get_LI_Handler_By_Name (Handler, "C"));
   Register_Default_Language_Extension (Registry, "C", ".c", ".h");
   Register_LI_Handler (Handler, "C++", CPP_LI);
   Set_Language_Handler
     (Handler, "C++", Get_LI_Handler_By_Name (Handler, "C++"));
   Register_Default_Language_Extension (Registry, "C++", ".cc", ".h");

   Register_Language_Handler (Db, Handler);

   declare
      Msg : constant String := Set_Executables (CPP_LI);
   begin
      if Msg /= "" then
         Put_Line ("Error: " & Msg);
         return;
      end if;
   end;

   if Argument_Count > 1 then
      Put_Line ("Usage: " & Command_Name & " <cmd_file>");
      return;
   elsif Argument_Count = 1 then
      Parse_Cmd_File (Argument (1));
   else
      Parse_Cmd_File ("test.cmd");
   end if;

   Destroy (Db);
   Traces.Finalize;

end Test_Parse;
