/* Remote target communications for VxWorks 6 targets.

   Copyright 2005
   Free Software Foundation, Inc.

   This file is part of GDB.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.  */

/* GENERAL DESCRIPTION:

   This file implements support of VxWorks 6 targets, using Wind River System's
   debugger framework. It uses two protocols:
   * WTX: we only need one WTX call from the wind registry (to get the
   connection parameters of the dfwserver).
   * DFW: used as a debug API (prefered to WTX and WDB because it supports
   RTPs). For setting breakpoints, attaching tasks and RTPs...

   (X <---Y---> Z means "X communicates with Z using the protocol Y)

   .-----.   WTX   .---------------.
   | GDB |<------->| WIND REGISTRY |
   |     |         `---------------'
   |     |   DFW   .------------.   WTX   .---------------.   WDB  .--------.
   |     |<------->| DFW SERVER |<------->| TARGET SERVER |<------>| TARGET |
   `-----'         `------------'         `---------------'        `--------'

   A connection scenario:

   (X --Y-->Z means "X sends the message Y to Z")

   .-----.              .---------------.  .------------.
   | GDB |              | WIND REGISTRY |  | DFW SERVER |
   `-----'              `---------------'  `------------'
      |    wtx_info_q query      |                 |
      |------------------------->|                 |
      |  dfw connection params   |                 |
      |<-------------------------|                 |
      |                                            |
      | -wrs-info-retrieve /Targets default:* ...  |
      |------------------------------------------->|
      |         list of known targets              |
      |<-------------------------------------------|

   et cetera...

   Here is a general description of the way we call a DFW primitive:
   1) we send the request (text format) to the DFW server;
   2) we reset the current_result_table and buffer the output in its
   string_table after a lexical analysis (stored in the dfw_lex_tree);
   3) then, we do a semantic analysis and we save the result on the
   dfw_sem_tree.
   4) finally, we can extract the information from the sem table using the
   sem_element interface.

   The result of the sem and lex analysis are stored into preallocated
   integer tables whose sizes are increased if needed. In those tables,
   can be stored:
   * lex (resp. sem) token types (e.g. DFW_LEX_COMMA);
   * index in the string buffer (for string params associated to the token
   type).

   For example, the following  DFW server output:

   ^done,i=[["defaultTarget","tgt_liege@borticado"]]

   would be stored in the lex table of current_result_table as followed:

   DFW_LEX_ITEM
   i
   DFW_LEX_EQUALS
   DFW_LEX_BEGIN_LIST
   DFW_LEX_BEGIN_LIST
   DFW_LEX_CSTRING
   defaultTarget
   DFW_LEX_COMMA
   DFW_LEX_CSTRING
   tgt_liege@borticado
   DFW_LEX_END_LIST
   DFW_LEX_END_LIST

   and the sem table would be:

   DFW_SEM_RESULT_CLASS
   done
   DFW_SEM_VARIABLE
   i
   DFW_SEM_VALUE
   DFW_SEM_BEGIN_LIST
   DFW_SEM_VALUE
   DFW_SEM_BEGIN_LIST
   DFW_SEM_VALUE
   DFW_SEM_CONST
   defaultTarget
   DFW_SEM_VALUE
   DFW_SEM_CONST
   tgt_liege@borticado
   DFW_SEM_END_LIST
   DFW_SEM_END_LIST

   (The grammar of the GDB/MI langage is defined in the GDB documentation).

   If any asynchronous output is found during the analysis of the result,
   we analyse it the same way, store the result of the analysis into
   current_async_record and call the callback which corresponds to this
   kind and async output.  */

#include "defs.h"
#include "target.h"
#include "gdb_string.h"
#include "serial.h"
#include "mi/mi-parse.h"
#include "symfile.h"
#include "objfiles.h"
#include "gdb_stat.h"
#include "gdbcmd.h"
#include "command.h"
#include "inferior.h"
#include "gdbthread.h"
#include "regcache.h"
#include "completer.h"
#include "gdb_assert.h"
#include <rpc/rpc.h>
#include <sys/time.h>
#include <stdlib.h>
#include <netdb.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <signal.h>

/* WTX connection parameters.  */

static const unsigned short wtx_registry_port = 2340;
static const unsigned int wtx_svc_base = 0x22000000;
static const int wtx_rpc_version = 4;
static const int wtx_info_q_get_prog = 204;
static struct timeval TIMEOUT = { 25, 0 };
static int load_timeout = 30;
static int task_list_timeout = 10;

/* This module's target-specific operations. */
static struct target_ops remote_dfw_ops;
static struct target_ops remote_dfwtask_ops;

/* current connection parameters.  */
static char *host_name;
static int port;
static int system_thread_id;
static const CORE_ADDR system_pid = 0;
static char *full_target_name = NULL;
static char *core_name = NULL;
static char *info_retrieval_core_branch = NULL;
static int info_retrieval_core_branch_len = 0;

/* WTX_CORE struct. Based on the corresponding WTX type */

struct wtx_core
{
  unsigned int obj_id;
  enum_t err_code;
  unsigned int prot_version;
};

/* WTX string type. Used to workaround string oddities in the WTX protocol.
   See xdr_wtx_string for details.  */

typedef char* wtx_string;

/* WTX_SVR_DESC struct. Based on the corresponding WTX type.  */

struct wtx_svr_desc
{
  wtx_string name;
  wtx_string type;
  wtx_string key;
};

/* WTX_MSG_SVR_DESC struct. Based on the corresponding WTX type.  */

struct wtx_msg_svr_desc
{
  struct wtx_core core;
  struct wtx_svr_desc desc;
};

/* Index type for tables (lex and sem) and buffer (strings): */

typedef unsigned int index_type;

/* Type of the elements in the lex (resp. sem) tables.  */

typedef int dfw_tree_element;

/* Type of description of the lex (or sem) tree element.  */

struct dfw_tree_element_desc
{
  dfw_tree_element code;
  /* Code of this tree element.  */

  char *name;
  /* Tree element name.  */

  int params;
  /* If 1, a string paramter is expected (i.e. this tree element should be
     followed, in the table, by an index in the string buffer).
     If -1, a special integer parameter is expected.
     If 0, no parameter is expected.  */
};

struct enum_desc
{
  int code;
  char *name;
};

/* Tree element codes for the lexical analysis.  */

enum dfw_lex_type
  {
    DFW_LEX_COMMA,
    DFW_LEX_EQUALS,
    DFW_LEX_BEGIN_TUPLE,
    DFW_LEX_END_TUPLE,
    DFW_LEX_BEGIN_LIST,
    DFW_LEX_END_LIST,
    DFW_LEX_CSTRING,
    DFW_LEX_ITEM,
    DFW_LEX_LAST
  };

/* Description record for the lexical tree elements.  */

static const struct dfw_tree_element_desc dfw_lex_list[] =
{
  {DFW_LEX_COMMA, "DFW_LEX_COMMA", 0},
  {DFW_LEX_EQUALS, "DFW_LEX_EQUALS", 0},
  {DFW_LEX_BEGIN_TUPLE, "DFW_LEX_BEGIN_TUPLE", 0},
  {DFW_LEX_END_TUPLE, "DFW_LEX_END_TUPLE", 0},
  {DFW_LEX_BEGIN_LIST, "DFW_LEX_BEGIN_LIST", 0},
  {DFW_LEX_END_LIST, "DFW_LEX_END_LIST", 0},
  {DFW_LEX_CSTRING, "DFW_LEX_CSTRING", 1},
  {DFW_LEX_ITEM, "DFW_LEX_ITEM", 1},
  {DFW_LEX_LAST,"DFW_LEX_UNKNOWN", 0}
};

/* Tree element codes for the semantic analysis.  */

enum dfw_sem_type
  {
    DFW_SEM_RESULT_CLASS,
    DFW_SEM_ASYNC_CLASS,
    DFW_SEM_VARIABLE,
    DFW_SEM_VALUE,
    DFW_SEM_RESULT,
    DFW_SEM_CONST,
    DFW_SEM_BEGIN_TUPLE,
    DFW_SEM_END_TUPLE,
    DFW_SEM_BEGIN_LIST,
    DFW_SEM_END_LIST,
    DFW_SEM_LAST
  };

/* Description record for the semantic tree elements.  */

static const struct dfw_tree_element_desc dfw_sem_list[] =
{
  {DFW_SEM_RESULT_CLASS, "DFW_SEM_RESULT_CLASS", -1},
  {DFW_SEM_ASYNC_CLASS, "DFW_SEM_ASYNC_CLASS", -1},
  {DFW_SEM_VARIABLE, "DFW_SEM_VARIABLE", 1},
  {DFW_SEM_VALUE, "DFW_SEM_VALUE", 0},
  {DFW_SEM_RESULT, "DFW_SEM_RESULT", 0},
  {DFW_SEM_CONST, "DFW_SEM_CONST", 1},
  {DFW_SEM_BEGIN_TUPLE, "DFW_SEM_BEGIN_TUPLE", 0},
  {DFW_SEM_END_TUPLE, "DFW_SEM_END_TUPLE", 0},
  {DFW_SEM_BEGIN_LIST, "DFW_SEM_BEGIN_LIST", 0},
  {DFW_SEM_END_LIST, "DFW_SEM_END_LIST", 0},
  {DFW_SEM_LAST,"DFW_SEM_UNKNOWN", 0}
};

/* tree element code for the result class.  */

enum result_class_type
  {
    RESULT_CLASS_DONE,
    RESULT_CLASS_RUNNING,
    RESULT_CLASS_CONNECTED,
    RESULT_CLASS_ERROR,
    RESULT_CLASS_EXIT,
    RESULT_CLASS_STOPPED,
    RESULT_CLASS_LAST
  };

/* Description record for the result classes.  */

static const struct dfw_tree_element_desc result_class_list[] =
  {
    {RESULT_CLASS_DONE, "done", -1},
    {RESULT_CLASS_RUNNING, "running", -1},
    {RESULT_CLASS_CONNECTED, "connected", -1},
    {RESULT_CLASS_ERROR, "error", -1},
    {RESULT_CLASS_EXIT, "exit", -1},
    {RESULT_CLASS_STOPPED, "stopped", -1},
    {RESULT_CLASS_LAST, "unkwown", -1}
  };

/* tree element code for the asynchronous result class.  */

enum async_class_type
  {
    /* Target State Change: */

    ASYNC_CLASS_STOPPED,
    ASYNC_CLASS_RUNNING,
    ASYNC_CLASS_CONTAINER_STOPPED,
    ASYNC_CLASS_INDETERMINATE_STATE,

    /* Target Connection Events: */

    ASYNC_CLASS_CONNECTED,
    ASYNC_CLASS_DISCONNECTED,

    /* Target Data Change: */

    ASYNC_CLASS_REGISTER_CHANGED,
    ASYNC_CLASS_MEMORY_CHANGED,

    /* Download Event Notifications: */

    ASYNC_CLASS_DOWNLOAD,
    ASYNC_CLASS_DOWNLOAD_COMPLETE,
    ASYNC_CLASS_DOWNLOAD_FAILED,
    ASYNC_CLASS_MODULES_CHANGED,

    /* TOS Event Notifications: */

    ASYNC_CLASS_CONTEXT_START,
    ASYNC_CLASS_CONTEXT_EXIT,
    ASYNC_CLASS_TOOL_DETACH,
    ASYNC_CLASS_TOOL_ATTACH,

    /* Register Definitions Changed Event Notification: */

    ASYNC_CLASS_REGDEFCHANGED,

    /* Unkwnown: */
    ASYNC_CLASS_LAST
  };

/* Description record for the asynchronous result classes.  */

static const struct dfw_tree_element_desc async_class_list[] =
  {
    {ASYNC_CLASS_STOPPED, "stopped", -1},
    {ASYNC_CLASS_RUNNING, "running", -1},
    {ASYNC_CLASS_CONTAINER_STOPPED, "container-stopped", -1},
    {ASYNC_CLASS_INDETERMINATE_STATE, "indeterminate-state", -1},
    {ASYNC_CLASS_CONNECTED, "connected", -1},
    {ASYNC_CLASS_DISCONNECTED, "disconnected", -1},
    {ASYNC_CLASS_REGISTER_CHANGED, "register-changed", -1},
    {ASYNC_CLASS_MEMORY_CHANGED, "memory-changed", -1},
    {ASYNC_CLASS_DOWNLOAD, "download", -1},
    {ASYNC_CLASS_DOWNLOAD_COMPLETE, "download-complete", -1},
    {ASYNC_CLASS_DOWNLOAD_FAILED, "download-failed", -1},
    {ASYNC_CLASS_MODULES_CHANGED, "modules-changed", -1},
    {ASYNC_CLASS_CONTEXT_START, "context-start", -1},
    {ASYNC_CLASS_CONTEXT_EXIT, "context-exit", -1},
    {ASYNC_CLASS_TOOL_DETACH, "tool-detach", -1},
    {ASYNC_CLASS_TOOL_ATTACH, "tool-attach", -1},
    {ASYNC_CLASS_REGDEFCHANGED, "regdefchanged", -1},
    {ASYNC_CLASS_LAST, "unknown", -1}
  };

/* DFW task status.  */

enum dfw_task_status
  {
    DFW_TASK_STATUS_STOPPED,
    DFW_TASK_STATUS_STOPPED_T,
    DFW_TASK_STATUS_STOPPED_P,
    DFW_TASK_STATUS_SUSPEND,
    DFW_TASK_STATUS_DELAY,  
    DFW_TASK_STATUS_PENDING,
    DFW_TASK_STATUS_READY,
    DFW_TASK_STATUS_DEAD,
    DFW_TASK_STATUS_RTP_NORMAL,
    DFW_TASK_STATUS_LAST
  };

/* Description of DFW task status.  */
static const struct enum_desc dfw_task_status_desc[] =
{
  {DFW_TASK_STATUS_STOPPED, "Stop"},
  {DFW_TASK_STATUS_STOPPED_T, "Stop+T"},
  {DFW_TASK_STATUS_STOPPED_P, "Stop+P"},
  {DFW_TASK_STATUS_SUSPEND, "Suspend"},
  {DFW_TASK_STATUS_DELAY, "Delay"},
  {DFW_TASK_STATUS_PENDING, "Pend"},
  {DFW_TASK_STATUS_READY, "Ready"},
  {DFW_TASK_STATUS_DEAD, "Dead"},
  {DFW_TASK_STATUS_RTP_NORMAL, "RTP_NORMAL"},
  {DFW_TASK_STATUS_LAST, "Unknown"}
};

/* Dynamically extensible string buffer type.  */

struct string_buffer
{
  char *beginning_of_buffer;

  /* Buffer address. The buffer is reallocated if needed.  */

  int size;

  /* Size of the buffer, in number of characters.  */

  index_type current_index;

  /* Index of the next free slot in the buffer.  */

};

/* Dynamically extensible integer table, used to store dfw sem or lex
   trees.  */

struct dfw_tree_type
{
  dfw_tree_element *beginning_of_table;

  /* Table address. The table is represented by a tree element array that
   we reallocate when needed.  */

  int size;

  /* Size of the table, in number of dfw_tree_elements.  */

  index_type table_end;

  /* Index of the next free slot in the table.  */

};

/* Result table type; used to store lexical and semantical analysis of
   an result (asynchronous or synchronous).  See an example of its use
   in the general description.  */

struct result_table
{
  /* String buffer. Used to store the strings (e.g. variable names, constant
     values) of the request.  */

  struct string_buffer string_table;

  /* Lex tree of the DFW request result.  */

  struct dfw_tree_type dfw_lex_tree;

  /* Sem tree of the DFW request result.  */

  struct dfw_tree_type dfw_sem_tree;
};

/* Semantic element extracted from the stack. It should be easier to
   manipulate when trying to get the result of a DFW request.  It contains
   a tree element code and its parameter (if any). For example, the following
   elements of the sem table:

   DFW_SEM_CONST
   defaultTarget

   would correspond to a semantic element SE, with:
   SE.type == DFW_SEM_CONST
   strcmp (SE.value.string_value, "defaultTarget") == 0
*/

struct dfw_sem_element
{
  enum dfw_sem_type type;
  union value_union
  {
    enum result_class_type result_class_value;
    enum async_class_type async_class_value;
    char *string_value;
  } value;
};

/* Full kernel/RTP task list.  */

struct vxworks_task
{
  struct vxworks_task *next;

  /* DFW id for this task's RTP.  */

  int rtp_dfw_id;

  /* Task id for this task's RTP.  */

  CORE_ADDR pid;

  /* DFW id for this task.  */

  int dfw_id;

  /* Task id for this task.  */

  CORE_ADDR tid;
};

/* DFW breakpoints list.  */

struct dfw_breakpoint
{
  struct dfw_breakpoint *next;

  /* DFW id for this breakpoint.  */

  int bp_number;

  /* Address of the breakpoint.  */

  CORE_ADDR address;
};

/* Initial size of a string buffer.  */
const static int STRING_BUFFER_INITIAL_SIZE = 1024;

/* Size of the increment of a string buffer when more storage is needed.  */

const static int STRING_BUFFER_INCREMENT = 1024;

/* Ditto for dfw trees.  */

const static int DFW_TREE_INITIAL_SIZE = 1024;
const static int DFW_TREE_INCREMENT = 1024;

/* Maximum length for the extra thread info.  */

#define DFW_EXTRA_THREAD_INFO_RESULT_BUFF_LEN 128

/* Result table for the last synchronous request output.  */

static struct result_table current_result_record;
static struct result_table current_async_record;

/* Descriptor for I/O to dfwserver.  Initialize it to NULL so that
   remote_dfw_open knows that we don't have a file open when the program
   starts.  */
static struct serial *remote_dfw_desc = NULL;

/* Maintenance stuff.  */

static int dfw_show_requests = 0;
static int dfw_show_responses = 0;
static int dfw_warn_unknown_identifiers = 0;

/* Current request token.  */
typedef unsigned char token_type;
static token_type current_token = 42;

/* Thread currently selected (for inspection through DFW routines).  */
static int current_selected_thread = -1;

/* Task currently debugged (i.e. originally attached). */
static struct vxworks_task *debugged_vxworks_task = NULL;

/* Full vxworks task list on target.  */

static struct vxworks_task *vxworks_task_list = NULL;

/* DFW breakpoint list.  */

static struct dfw_breakpoint *dfw_breakpoint_list = NULL;

/* XDR marshalling/demarshalling routines for WTX types.  */

int xdr_wtx_core (XDR *xdrs, struct wtx_core *objp);
int xdr_wtx_string (XDR *xdrs, wtx_string *objp);
int xdr_wtx_svr_desc (XDR *xdrs, struct wtx_svr_desc *objp);
bool_t xdr_wtx_msg_svr_desc(XDR *xdrs, struct wtx_msg_svr_desc *objp);

/* Initialization routines.  */

static void init_remote_dfw_ops (void);
static void init_remote_dfwtask_ops (void);
void _initialize_remote_dfw (void);

/* Target hooks.  */

static void remote_dfw_open (char *name, int from_tty);
static void remote_dfw_close (int quitting);
static void remote_dfw_files_info (struct target_ops *ignore);
static int remote_dfw_xfer_memory (CORE_ADDR mem_addr, char *buffer,
				   int mem_len, int should_write,
				   struct mem_attrib *attrib,
				   struct target_ops *target);
#if 0
static LONGEST remote_dfw_xfer_partial (struct target_ops *ops,
					enum target_object object,
					const char *annex, void *readbuf,
					const void *writebuf, ULONGEST offset,
					LONGEST len);
#endif
static void remote_dfw_find_new_threads (void);
static char * remote_dfw_extra_thread_info (struct thread_info *tp);
static void remote_dfw_fetch_registers (int regno);
static int remote_dfw_thread_alive (ptid_t ptid);
static void remote_dfw_attach (char *args, int from_tty);
static void remote_dfw_kill ();
static void remote_dfw_mourn_inferior ();
static void remote_dfw_load (char *arg, int from_tty);
static void remote_dfw_load_rtp_module (char *arg, int from_tty);
static void remote_dfw_load_kernel_module (char *arg, int from_tty);
static void remote_dfw_unload (char *arg, int from_tty);
static ptid_t remote_dfw_wait (ptid_t ptid, struct target_waitstatus *status);
static void remote_dfw_resume (ptid_t ptid, int step,
			       enum target_signal siggnal);
static void interrupt_query (void);
static void remote_dfw_interrupt (int signo);
static void remote_dfw_interrupt_twice (int signo);
static void remote_dfw_stop ();
static int remote_dfw_insert_breakpoint (CORE_ADDR, char *);
static int remote_dfw_remove_breakpoint (CORE_ADDR, char *);
static void remote_dfw_create_inferior (char *exec_file, char *args,
					char **env, int from_tty);
static void remote_dfw_create_inferior_rtp (char *exec_file, char *args,
					    char **env, int from_tty);
static void remote_dfw_create_inferior_kernel_task (char *exec_file,
						    char *args,
						    char **env,
						    int from_tty);

/* Connection handling.  */

static int readchar (int timeout);
static void write_string (const char *buf, int cnt);
static token_type write_token ();

static void wrs_info_retrieve (const char *tree_branch,
			       const int tree_branch_len,
			       const char *tree_leaf,
			       const int tree_leaf_len,
			       const char *pattern);
static int thread_select (int thread_id);
static int thread_select_current_pid ();
static void push_interrupt_event ();
static void exec_interrupt ();
static void exec_continue ();
static void exec_step_instruction ();
static char * get_dfw_object_name (long dfw_id);
static enum dfw_task_status get_dfw_task_status (long dfw_id);

static void skip_line ();
static void skip_prompt ();

/* Output record lexical parsing.  */

static int read_class (int timeout, struct string_buffer *table);
static int wait_output_record_sequence (int timeout, token_type *token);
static void read_output_record_sequence (int timeout,
					 token_type expected_token);
static void read_result_record (int timeout);
static void read_async_output (int timeout);
static void read_target_stream (int timeout);
static void read_log_stream (int timeout);

/* Test of lex tree elements, used for the sem analysis.  */

static int test_lex (struct result_table *table, index_type current_index,
		     enum dfw_lex_type lex);
static int test_const (struct result_table *table, index_type current_index);
static int test_tuple (struct result_table *table, index_type current_index);
static int test_list (struct result_table *table, index_type current_index);
static int test_result (struct result_table *table, index_type current_index);
static int test_value (struct result_table *table, index_type current_index);
static int check_lex (struct result_table *table, index_type *current_index,
		      enum dfw_lex_type lex);

/* Sem analysis of the result sequence.  */

static void read_result_list (struct result_table *table);
static void read_result (struct result_table *table,
			 index_type *current_index);
static void read_variable (struct result_table *table,
			   index_type *current_index);
static void read_value (struct result_table *table, index_type *current_index);
static void read_const (struct result_table *table, index_type *current_index);
static void read_tuple (struct result_table *table, index_type *current_index);
static void read_list (struct result_table *table, index_type *current_index);

/* String buffer support.  */

static int string_buffer_allocate (struct string_buffer *buff);
static int string_buffer_push_char (struct string_buffer *buff, char c);
static char * string_buffer_get_string_address (struct string_buffer *buff,
						index_type index);
static int string_buffer_reset (struct string_buffer * buff);
static index_type string_buffer_get_index (struct string_buffer * buff);

/* DFW tree table handling.  */

static int dfw_tree_allocate (struct dfw_tree_type *table);
static int dfw_tree_push (struct dfw_tree_type *table,
			       dfw_tree_element tree);
static int dfw_tree_reset (struct dfw_tree_type *table);
static dfw_tree_element lookup_dfw_tree_element
(const struct dfw_tree_element_desc *desc, char *name, int last);
static int lookup_enum_code (const struct enum_desc *desc, char *name,
			     int last);


/* Routines to go through the tree sem table and get sem elements from
 the indices in the sem tree table.  */

static struct dfw_sem_element next_sem_element (struct result_table *result,
						index_type *current_index);

static struct dfw_sem_element next_sem_element_with_check
(struct result_table *result, index_type *current_index,
 const char *request_desc, enum dfw_sem_type type);

static struct dfw_sem_element sem_element_process_result_header
(struct result_table *result, index_type *current_index,
 const char *request_desc);

static struct dfw_sem_element sem_element_process_const_string_value
(struct result_table *result, index_type *current_index,
 const char *request_desc, char **string_value);

static struct dfw_sem_element sem_element_process_const_integer_value
(struct result_table *result, index_type *current_index,
 const char *request_desc, int *const_value);

static struct dfw_sem_element sem_element_process_const_core_addr_value
(struct result_table *result, index_type *current_index,
 const char *request_desc, CORE_ADDR *const_value);

/* Result table handling.  */

static int result_table_allocate (struct result_table *table);

/* Debug.  */

static int print_result_record (struct result_table *result);
static void info_dfw_trees ();

/* Sync with the target.  */

static void update_symbol_files ();
static void update_vxworks_task_list ();

/* VxWorks tasks handling.  */

static struct vxworks_task * vxworks_task_build (int rtp_dfw_id, CORE_ADDR pid,
						 int dfw_id, CORE_ADDR tid);
static ptid_t vxworks_task_ptid_build (struct vxworks_task *task);
static void add_vxworks_task (struct vxworks_task *task);
static struct vxworks_task * lookup_vxworks_task (CORE_ADDR id);
static struct vxworks_task * lookup_vxworks_task_by_dfw_id (int dfw_id);
static void clear_vxworks_task_list ();
static void info_vxworks_tasks_command ();
static struct vxworks_task * duplicate_vxworks_task (struct vxworks_task *t);
static int is_being_debugged (ptid_t ptid);
static int is_being_debugged_1 (int dfw_id);
static ptid_t get_ptid_from_dfw_id (int dfw_id);

/* DFW breakpoint handling.  */

static void add_dfw_breakpoint (CORE_ADDR address, int bp_number);
static int extract_dfw_breakpoint (CORE_ADDR address);

/* Allocate and initialize a string buffer BUFF of
   STRING_BUFFER_INITIAL_SIZE bytes.  */

static int
string_buffer_allocate (struct string_buffer *buff)
{
  buff->beginning_of_buffer =
    (char *) (xmalloc (STRING_BUFFER_INITIAL_SIZE * sizeof (char)));
  if (buff->beginning_of_buffer == NULL)
    return 0;
  buff->current_index = 0;
  buff->size = STRING_BUFFER_INITIAL_SIZE;
  return 1;
}

/* Push C into the string buffer. Return the next index in BUFF.  */

static int
string_buffer_push_char (struct string_buffer *buff, char c)
{
  int char_index = buff->current_index;
  *(buff->beginning_of_buffer + buff->current_index) = c;
  buff->current_index++;
  gdb_assert (buff->current_index != 0);
  if (buff->current_index > buff->size)
    {
      buff->size = buff->size + STRING_BUFFER_INCREMENT;
      buff->beginning_of_buffer = xrealloc ((void *) buff->beginning_of_buffer,
					    buff->size * sizeof (char));
    }
  return char_index;
}

/* Push STRING into the string buffer. Return the location of STRING.  */

static int
string_buffer_push_string (struct string_buffer *buff, char *string)
{
  int char_index = buff->current_index;
  char *current = string;
  while (*current != '\0')
    {
      *(buff->beginning_of_buffer + buff->current_index) = *current;
      buff->current_index++;
      gdb_assert (buff->current_index != 0);
      if (buff->current_index > buff->size)
	{
	  buff->size = buff->size + STRING_BUFFER_INCREMENT;
	  buff->beginning_of_buffer
	    = xrealloc ((void *) buff->beginning_of_buffer,
			buff->size * sizeof (char));
	}
    }
  return char_index;
}

/* Return a pointer to the string starting at INDEX in BUFF.  */

static char *
string_buffer_get_string_address (struct string_buffer *buff, index_type index)
{
  return buff->beginning_of_buffer + index;
}

/* Reset the string buffer. All strings previously stored in the buffer are
   lost so that their memory space can be reused.  */

static int
string_buffer_reset (struct string_buffer * buff)
{
  buff->current_index = 0;
  memset (buff->beginning_of_buffer, 0, buff->size);
  return 1;
}

/* Get current index of BUFF.  */

static index_type
string_buffer_get_index (struct string_buffer * buff)
{
  return buff->current_index;
}

/* Allocate and initialize a table TABLE of DFW_TREE_INITIAL_SIZE bytes.  */

static int
dfw_tree_allocate (struct dfw_tree_type *table)
{
  table->beginning_of_table =
    (dfw_tree_element *) (xmalloc (DFW_TREE_INITIAL_SIZE
				    * sizeof (dfw_tree_element)));
  if (table->beginning_of_table == NULL)
    return 0;
  table->table_end = 0;
  table->size = DFW_TREE_INITIAL_SIZE;
  return 1;
}

/* Reset the table. All elements previously stored in the buffer are
   lost so that their memory space can be reused.  */

static int
dfw_tree_reset (struct dfw_tree_type *table)
{
  table->table_end = 0;
  memset (table->beginning_of_table, 0, table->size);
  return 1;
}

/* Push TREE into the string buffer. Return the index in the table.  */

static int
dfw_tree_push (struct dfw_tree_type *table, dfw_tree_element tree)
{
  index_type table_end = table->table_end;
  *(table->beginning_of_table + table->table_end) = tree;
  table->table_end ++;
  if (table->table_end > table->size)
    {
      table->size = table->size + DFW_TREE_INCREMENT;
      table->beginning_of_table =
	xrealloc ((void *) table->beginning_of_table,
		  table->size * sizeof (dfw_tree_element));
    }
  return table_end;
}

/* Lookup NAME in DESC, and return its code.  */

static dfw_tree_element
lookup_dfw_tree_element (const struct dfw_tree_element_desc *desc, char *name,
			 int last)
{
  dfw_tree_element result = last;
  int i;

  for (i = 0; i < last; i++)
    {
      if (strcmp (name, desc[i].name) == 0)
	{
	  result = desc[i].code;
	}
    }
  if (result == last && dfw_warn_unknown_identifiers)
    warning ("DFW: \"%s\" identifier is unknown", name);

  return result;
}

/* Lookup NAME in DESC, and return its code.  */

static int
lookup_enum_code (const struct enum_desc *desc, char *name, int last)
{
  int result = last;
  int i;

  for (i = 0; i < last; i++)
    {
      if (strcmp (name, desc[i].name) == 0)
	{
	  result = desc[i].code;
	}
    }
  if (result == last && dfw_warn_unknown_identifiers)
    warning ("DFW: \"%s\" identifier is unknown", name);

  return result;
}

/* Dump the content of RESULT on the standard output.  */

static int
print_result_record (struct result_table *result)
{
  index_type current_index;
  enum dfw_lex_type lex;
  enum dfw_sem_type sem;
  dfw_tree_element item;
  struct dfw_tree_type *lex_table = &(result->dfw_lex_tree);
  struct dfw_tree_type *sem_table = &(result->dfw_sem_tree);

  fprintf_unfiltered (gdb_stderr, "\n*** LEX ***\n\n");

  for (current_index = 0;
       current_index < lex_table->table_end;
       current_index++)
    {
      lex = lex_table->beginning_of_table[current_index];
      fprintf_unfiltered (gdb_stderr, "%s\n", dfw_lex_list [lex].name);
      switch (lex)
	{
	case DFW_LEX_COMMA:
	case DFW_LEX_EQUALS:
	case DFW_LEX_BEGIN_TUPLE:
	case DFW_LEX_END_TUPLE:
	case DFW_LEX_BEGIN_LIST:
	case DFW_LEX_END_LIST:
	case DFW_LEX_LAST:
	  /* No parameter, so nothing else to do.  */
	  break;
	case DFW_LEX_CSTRING:
	case DFW_LEX_ITEM:
	  /* String parameter.  */
	  current_index++;
	  item = lex_table->beginning_of_table[current_index];
	  fprintf_unfiltered (gdb_stderr, "%s\n",
			      string_buffer_get_string_address (&(result->string_table), item));
	  break;
	default:
	  error ("Displaying of lex %d is not implemented", lex);
	  break;
	}
    }

  fprintf_unfiltered (gdb_stderr, "\n*** SEM ***\n\n");

  for (current_index = 0;
       current_index < sem_table->table_end;
       current_index++)
    {
      sem = sem_table->beginning_of_table[current_index];
      fprintf_unfiltered (gdb_stderr, "%s\n", dfw_sem_list [sem].name);
      switch (sem)
	{
	case DFW_SEM_VALUE:
	case DFW_SEM_RESULT:
	case DFW_SEM_BEGIN_TUPLE:
	case DFW_SEM_END_TUPLE:
	case DFW_SEM_BEGIN_LIST:
	case DFW_SEM_END_LIST:
	case DFW_SEM_LAST:
	  /* No parameter, so nothing else to do.  */
	  break;

	case DFW_SEM_VARIABLE:
	case DFW_SEM_CONST:
	  /* String parameter.  */
	  current_index++;
	  item = sem_table->beginning_of_table[current_index];
	  fprintf_unfiltered
	    (gdb_stderr, "%s\n",
	     string_buffer_get_string_address (&(result->string_table), item));
	  break;

	case DFW_SEM_RESULT_CLASS:
	case DFW_SEM_ASYNC_CLASS:
	  current_index++;
	  item = sem_table->beginning_of_table[current_index];
	  fprintf_unfiltered (gdb_stderr, "%s\n",
			      async_class_list[item].name);
	  break;
	default:
	  error ("Displaying of sem %d is not implemented", sem);
	  break;

	}
    }


  fprintf_unfiltered (gdb_stderr, "\n");

  return 1;
}

/* Get the semantic element located at CURRENT_INDEX in RESULT. If
   this element has a parameter, it is also read and stored into the
   corresponding value field of the returned value.  Finally,
   CURRENT_INDEX is updated to point to the next semantic element in
   RESULT.  */

static struct dfw_sem_element
next_sem_element (struct result_table *result, index_type *current_index)
{
  enum dfw_sem_type sem;
  dfw_tree_element item;
  struct dfw_tree_type *sem_table = &(result->dfw_sem_tree);
  struct dfw_sem_element element;

  memset (&element, 0, sizeof (element));

  sem = sem_table->beginning_of_table[*current_index];
  element.type = sem;

  switch (sem)
    {
    case DFW_SEM_VALUE:
    case DFW_SEM_RESULT:
    case DFW_SEM_BEGIN_TUPLE:
    case DFW_SEM_END_TUPLE:
    case DFW_SEM_BEGIN_LIST:
    case DFW_SEM_END_LIST:
    case DFW_SEM_LAST:
      /* No parameter, so nothing else to do.  */
      break;

    case DFW_SEM_VARIABLE:
    case DFW_SEM_CONST:
      /* String parameter.  */
      (*current_index)++;
      item = sem_table->beginning_of_table[*current_index];
      element.value.string_value =
	string_buffer_get_string_address (&(result->string_table), item);
      break;

    case DFW_SEM_RESULT_CLASS:
      (*current_index)++;
      item = sem_table->beginning_of_table[*current_index];
      element.value.async_class_value = item;
      break;
    case DFW_SEM_ASYNC_CLASS:
      (*current_index)++;
      item = sem_table->beginning_of_table[*current_index];
      element.value.result_class_value = item;
      break;
    default:
      error ("sem %d is not understood", sem);
      break;
    }

  (*current_index)++;
  return element;
}

/* Same as next_sem_element, but if the semantic element pointed by
   CURRENT_INDEX is not of type TYPE, an error is raised.  */

static struct dfw_sem_element
next_sem_element_with_check  (struct result_table *result,
			      index_type *current_index,
			      const char *request_desc,
			      enum dfw_sem_type type)
{
  struct dfw_sem_element element = next_sem_element (result, current_index);

  if (element.type != type)
    error ("DFW: parse error in %s; waiting %s, got %s",
	   request_desc,
	   dfw_sem_list [type].name,
	   dfw_sem_list [element.type].name);

  return element;
}

static int
test_next_sem_element  (struct result_table *result,
			index_type current_index,
			enum dfw_sem_type type)
{
  index_type tmp_index = current_index;
  struct dfw_sem_element element = next_sem_element (result, &tmp_index);
  return element.type == type;
}

/* Read the result header semantic info of the result record stored in
   RESULT, at index CURRENT_INDEX, that is to say the result class and
   the first variable name in the output record. CURRENT_INDEX is
   updated to point to the next semantic element,

   If, at this location, the header is not found or if result class is
   not "done", an error is raised.

   Example: if RESULT, at CURRENT_INDEX, contains the following semantic
   information:

   DFW_SEM_RESULT_CLASS
   done
   DFW_SEM_VARIABLE
   i
   DFW_SEM_VALUE
   [...]

   This function would return a dfw_sem_element containing {DFW_SEM_VARIANLE,
   "i"} and current_index would then point to the DFW_SEM_VALUE item.  */

static struct dfw_sem_element
sem_element_process_result_header (struct result_table *result,
				   index_type *current_index,
				   const char *request_desc)
{
  struct dfw_sem_element element = next_sem_element (result, current_index);
  if (element.type != DFW_SEM_RESULT_CLASS
      || element.value.result_class_value == RESULT_CLASS_ERROR)
    error ("DFW: error in %s.", request_desc);

  if (*current_index < current_result_record.dfw_sem_tree.table_end)
    element = next_sem_element_with_check (result, current_index, request_desc,
					   DFW_SEM_VARIABLE);
  return element;
}

/* Read the semantic info pointed by CURRENT_INDEX into the RESULT table,
   assuming that it is a pair (value, const). exempli gratia:

   DFW_SEM_VALUE
   DFW_SEM_CONST
   "my_const_value"

   If this assumption is not correct, an error is raised. At the contrary,
   if it is true:
   * CURRENT_INDEX is updated to point to the element following this pair
   in the RESULT table;
   * string_value is set to the string value of the DFW_SEM_CONST element;
   * the function returns a dfw_sem_element containing the DFW_SEM_CONST
   and its string parameter.  */

static struct dfw_sem_element
sem_element_process_const_string_value (struct result_table *result,
					index_type *current_index,
					const char *request_desc,
					char **string_value)
{
  struct dfw_sem_element element =
    next_sem_element_with_check (result, current_index, request_desc,
				 DFW_SEM_VALUE);

  element = next_sem_element_with_check (result, current_index, request_desc,
					 DFW_SEM_CONST);

  *string_value = element.value.string_value;
  return element;
}

/* Same thing as sem_element_process_const_string_value, for an integer
   value instead of a string value. This integer value is stored into
   CONST_VALUE.  */


static struct dfw_sem_element
sem_element_process_const_integer_value (struct result_table *result,
					 index_type *current_index,
					 const char *request_desc,
					 int *const_value)
{
  char *string_value;
  struct dfw_sem_element element =
    sem_element_process_const_string_value (result, current_index,
					    request_desc,
					    &string_value);
  *const_value = atoi (string_value);
  return element;
}

/* Same thing as sem_element_process_const_integer_value, for a CORE_ADDR
   value in hexademal format (e.g. "0xDEADBEEF").  */

static struct dfw_sem_element
sem_element_process_const_core_addr_value (struct result_table *result,
					   index_type *current_index,
					   const char *request_desc,
					   CORE_ADDR *const_value)
{
  char *string_value;
  char *cptr = 0;
  struct dfw_sem_element element =
    sem_element_process_const_string_value (result, current_index,
					    request_desc,
					    &string_value);

  *const_value = (CORE_ADDR) strtoul (string_value, &cptr, 0);
  if ((cptr == string_value) || (*cptr != '\0'))
    error ("Invalid address value in %s", request_desc);
  return element;
}

/* Same thing as sem_element_process_const_core_addr, but the result
   is stored into a raw buffer.  */

static struct dfw_sem_element
sem_element_process_const_raw_value (struct result_table *result,
				     index_type *current_index,
				     const char *request_desc,
				     char *buffer,
				     int buffer_size)
{
  char *string_value;
  char *cptr = 0;
  struct dfw_sem_element element =
    sem_element_process_const_string_value (result, current_index,
					    request_desc,
					    &string_value);
  char current_byte[] = "0x00";
  int i, j;
  int size; /* size of the register in bytes.  */

  if (string_value [0] == '0' && string_value [1] == 'x')
    string_value += 2;

  size = strlen (string_value) / 2;
  gdb_assert (size <= buffer_size);

  memset (buffer, 0, sizeof (char) * buffer_size);
  for (i = 0; i < size; i++)
    {
      if (TARGET_BYTE_ORDER == BFD_ENDIAN_BIG)
	j = i;
      else
	j = size - i - 1;

      if (*string_value)
	{
	  current_byte [3] = string_value [0];
	  current_byte [4] = string_value [1];
	  buffer [j] = (char) strtoul (current_byte, &cptr, 0);
	  if ((cptr == string_value) || (*cptr != '\0'))
	    error ("Invalid address value in %s", request_desc);
	}
      else
	{
	  break;
	}
      string_value += 2;
    }
  return element;
}

/* Allocate TABLE.  */

static int
result_table_allocate (struct result_table *table)
{
  return (string_buffer_allocate (&table->string_table))
    && dfw_tree_allocate (&table->dfw_lex_tree)
    && dfw_tree_allocate (&table->dfw_sem_tree);
}

/* Reset TABLE.  */

static int
result_table_reset (struct result_table *table)
{
  return (string_buffer_reset (&table->string_table)
    && dfw_tree_reset (&table->dfw_lex_tree)
    && dfw_tree_reset (&table->dfw_sem_tree));
}

/* Parse the service key returned by the wtx_info_q_get request, and
   extract from it the connection parameters of the corresponding service.
   which are:
   * HOST_NAME_LOC: pointer to the location of the host name in KEY;
   * HOST_NAME_LEN: length of the host name;
   * PORT_LOC: pointer to the location of the TCP port in KEY;
   * PORT_LEN: length of the TCP port.  */

static void
parse_service_key (char *key, char **host_name_loc,
		   int *host_name_len, char **port_loc, int *port_len)
{
  char *begining_token, *end_token;
  char host_name_flag[] = "host;";
  int  host_name_flag_len = strlen (host_name_flag);
  char port_flag[] = "dfwserver.miport;";
  int  port_flag_len = strlen (port_flag);

  for (begining_token = key, end_token = strchr (key, ';');
       end_token != NULL;
       begining_token = end_token + 1,
	 end_token = strchr (begining_token, ';'))
    {
      if (strncmp (begining_token, host_name_flag, host_name_flag_len) == 0)
	{
	  begining_token = end_token + 1;
	  end_token = strchr (begining_token, ';');
	  *host_name_loc = begining_token;
	  if (end_token)
	    {
	      *host_name_len = strlen (begining_token) - strlen (end_token);
	    }
	  else
	    {
	      *host_name_len = strlen (begining_token);
	      break;
	    }
	}
      else if (strncmp (begining_token, port_flag, strlen (port_flag)) == 0)
	{
	  begining_token = end_token + 1;
	  end_token = strchr (begining_token, ';');
	  *port_loc = begining_token;
	  if (end_token)
	    {
	      *port_len = strlen (begining_token) - strlen (end_token);
	    }
	  else
	    {
	      *port_len = strlen (begining_token);
	      break;
	    }
	}
    }
}

/* Read a single character from the dfw end and return it.  */

static int
readchar (int timeout)
{
  int ch;

  ch = serial_readchar (remote_dfw_desc, timeout);

  if (dfw_show_responses)
    printf_unfiltered ("%c", ch);

  if (ch >= 0)
    return (ch & 0x7f);

  switch ((enum serial_rc) ch)
    {
    case SERIAL_EOF:
      target_mourn_inferior ();
      error ("DFW connection closed");
      /* no return */
    case SERIAL_ERROR:
      perror_with_name ("DFW communication error");
      /* no return */
    case SERIAL_TIMEOUT:
      error ("DFW connection timeout");
    }

  return ch;
}

/* Read an DFW output sequence from the connection output.  The syntax rules
   used are:

   output ==>
     ( out-of-band-record )* [ result-record ] "(gdb)" nl

   result-record ==>
     [ token ] "^" result-class ( "," result )* nl

   out-of-band-record ==>
     async-record | stream-record

   async-record ==>
     exec-async-output | status-async-output | notify-async-output

   exec-async-output ==>
     [ token ] "*" async-output

   status-async-output ==>
     [ token ] "+" async-output

   notify-async-output ==>
     [ token ] "=" async-output

*/

static void
read_output_record_sequence (int timeout, token_type expected_token)
{
  int output_type = '\0';
  token_type output_token = 0;

  while (1)
    {
      output_type = wait_output_record_sequence (timeout, &output_token);
  
      if (output_type == '^'
	  && (!expected_token || expected_token == output_token))
	return ;
    }
}

/* Read either a result record, or an async record. Return the caracter
   which identifies the output types.  */

static int
wait_output_record_sequence (int timeout, token_type *token)
{
  int output_type = '\0';

  *token = 0;
  while (1)
    {
      output_type = readchar (timeout);
      switch (output_type)
	{
	case '(':
	  /* Prompt */
	  skip_line ();
	  goto end_of_read;
	case '0':
	case '1':
	case '2':
	case '3':
	case '4':
	case '5':
	case '6':
	case '7':
	case '8':
	case '9':
	  *token *= 10;
	  *token += output_type - '0';
	  break;
	case '=':
	case '*':
	case '+':
	  read_async_output (timeout);
	  goto end_of_read;
	case '^':
	  read_result_record (timeout);
	  goto end_of_read;
	case '@':
	  read_target_stream (timeout);
	  goto end_of_read;
	case '&':
	  read_log_stream (timeout);
	  goto end_of_read;
	default:
	  goto end_of_read;
	}
    }

 end_of_read:
  return output_type;
}

/* Read a class (result or async) from the DFW connection and
   buffer it into TABLE. Return the last character read.  */
int
read_class (int timeout, struct string_buffer *table)
{
  int current = '\0';

  while (current != '\n' && current != ',')
    {
      current = readchar (timeout);
      if (current == '\n' || current == ',')
	{
	  string_buffer_push_char (table, '\0');
	}
      else
	{
	  string_buffer_push_char (table, (char) current);
	}
    }
  return current;
}

/* Read an async output from the DFW connection output. Syntax rule used:

   async-output ==>
     async-class ( "," result )* nl

*/

static void
read_async_output (int timeout)
{
  int current = '=';
  dfw_tree_element request_result = ASYNC_CLASS_LAST;
  index_type current_index;
  char *buf;

  dfw_tree_reset (&(current_async_record.dfw_lex_tree));

  current_index =
    string_buffer_get_index (&(current_async_record.string_table));

  /* Bufferise result class.  */
  current = read_class (timeout, &(current_async_record.string_table));
  buf =
    string_buffer_get_string_address (&(current_async_record.string_table),
				      current_index);

  request_result = lookup_dfw_tree_element (async_class_list, buf,
					    ASYNC_CLASS_LAST);
  dfw_tree_push (&(current_async_record.dfw_sem_tree),
		 (dfw_tree_element) DFW_SEM_ASYNC_CLASS);
  dfw_tree_push (&(current_async_record.dfw_sem_tree),
		 (dfw_tree_element) request_result);

  if (current != '\n')
    read_result_list (&current_async_record);
}

/* Read a result record from the DFW connection output.  Syntax rule used:

   result-record ==>
     [ token ] "^" result-class ( "," result )* nl

*/

static void
read_result_record (int timeout)
{
  int current = '^';
  dfw_tree_element request_result = RESULT_CLASS_LAST;
  index_type current_index;
  char *buf;

  result_table_reset (&current_result_record);
  current_index =
    string_buffer_get_index (&(current_result_record.string_table));

  /* Bufferise result class.  */
  current = read_class (timeout, &(current_result_record.string_table));
  buf =
    string_buffer_get_string_address (&(current_result_record.string_table),
				      current_index);

  request_result = lookup_dfw_tree_element (result_class_list, buf,
					    RESULT_CLASS_LAST);
  dfw_tree_push (&(current_result_record.dfw_sem_tree),
		 (dfw_tree_element) DFW_SEM_RESULT_CLASS);
  dfw_tree_push (&(current_result_record.dfw_sem_tree),
		 (dfw_tree_element) request_result);

  if (current != '\n')
    read_result_list (&current_result_record);
}

/* Read the list of result (in a result record) from the DFW connection
   output and buffer the result of the lexical and semantical analysis in
   TABLE.  */

static void
read_result_list (struct result_table *table)
{
  int current = '^';
  char char_to_push;
  enum dfw_lex_type lex_to_push;
  int pushing_item, end_item_reached = 0;
  index_type string_index, current_index;

  /* Buffer result items into the lex string buffer. When there is no
     ambiguity, we resolve it; when there is, we just store a
     dfw_lex_item ; we will resolve it later, during semantical analysis.  */
  string_index =
    string_buffer_get_index (&(table->string_table));

  while (current != '\n')
    {
      current = readchar (remote_timeout);
      switch (current)
	{
	case '{':
	  lex_to_push = DFW_LEX_BEGIN_TUPLE;
	  end_item_reached = 1;
	  break;
	case '}':
	  lex_to_push = DFW_LEX_END_TUPLE;
	  end_item_reached = 1;
	  break;
	case '[':
	  lex_to_push = DFW_LEX_BEGIN_LIST;
	  end_item_reached = 1;
	  break;
	case ']':
	  lex_to_push = DFW_LEX_END_LIST;
	  end_item_reached = 1;
	  break;
	case '"':
	  /* C string. We push the item on the lex table (if any) and we
	     buffer everything until we see another quote character.  */
	  if (pushing_item)
	    {
	      string_buffer_push_char (&(table->string_table),
				       (char) '\0');
	      dfw_tree_push (&(table->dfw_lex_tree),
			     (dfw_tree_element) DFW_LEX_ITEM);
	      dfw_tree_push (&(table->dfw_lex_tree),
			     (dfw_tree_element) string_index);
	      string_index =
		string_buffer_get_index (&(table->string_table));
	      pushing_item = 0;
	      end_item_reached = 0;
	    }

	  current = '\0';
	  while (current != '"')
	    {
	      current = readchar (remote_timeout);

	      if (current != '"')
		string_buffer_push_char (&(table->string_table),
					 (char) current);
	      if (current == '\\')
	      {
		current = readchar (remote_timeout);
		string_buffer_push_char (&(table->string_table),
					 (char) current);
	      }
	    }

	  string_buffer_push_char (&(table->string_table),
				   (char) '\0');
	  dfw_tree_push (&(table->dfw_lex_tree),
			 (dfw_tree_element) DFW_LEX_CSTRING);
	  dfw_tree_push (&(table->dfw_lex_tree),
			 (dfw_tree_element) string_index);
	  string_index =
	    string_buffer_get_index (&(table->string_table));

	  /* No lex left to push.  */
	  lex_to_push = DFW_LEX_LAST;
	  break;
	case ',':
	  lex_to_push = DFW_LEX_COMMA;
	  end_item_reached = 1;
	  break;
	case '=':
	  lex_to_push = DFW_LEX_EQUALS;
	  end_item_reached = 1;
	  break;
	case '\\':
	  string_buffer_push_char (&(table->string_table),
				   (char) current);
	  current = readchar (remote_timeout);
	  char_to_push = current;
	case '\n':
	  break;
	default:
	  pushing_item = 1;
	  char_to_push = current;
	  string_buffer_push_char (&(table->string_table),
				   (char) current);
	  break;
	}

      if (pushing_item && end_item_reached)
	{
	  string_buffer_push_char (&(table->string_table),
				   (char) '\0');
	  dfw_tree_push (&(table->dfw_lex_tree),
			 (dfw_tree_element) DFW_LEX_ITEM);
	  dfw_tree_push (&(table->dfw_lex_tree),
			 (dfw_tree_element) string_index);
	  string_index =
	    string_buffer_get_index (&(table->string_table));
	  pushing_item = 0;
	}
      if (end_item_reached)
	{
	  dfw_tree_push (&(table->dfw_lex_tree),
			 (dfw_tree_element) lex_to_push);
	  end_item_reached = 0;
	}
    }

  /* Now, go through the buffered lex and populate the sem table.  */

  /* result_list ==> (result ",")* */
  for (current_index = 0;
       current_index < table->dfw_lex_tree.table_end;
       )
    {
      read_result (table, &current_index);

      if (current_index < table->dfw_lex_tree.table_end)
	{
	  check_lex (table, &current_index, DFW_LEX_COMMA);
	}
    }
}

/* Return 1 if the lexical element pointed by CURRENT_INDEX in TABLE is
   of type LEX; else, return 0.  */

static int
test_lex (struct result_table *table, index_type current_index,
	  enum dfw_lex_type lex)
{
  return current_index < table->dfw_lex_tree.table_end
    && table->dfw_lex_tree.beginning_of_table[current_index] == lex;
}

/* Return 1 if the lexical element pointed by CURRENT_INDEX in TABLE
   corresponds to a DFW_SEM_CONST.  */

static int
test_const (struct result_table *table, index_type current_index)
{
  return test_lex (table, current_index, DFW_LEX_CSTRING);
}

/* Return 1 if the lexical element pointed by CURRENT_INDEX in TABLE
   corresponds to a DFW_SEM_BEGIN_TUPLE; else, return 0.  */

static int
test_tuple (struct result_table *table, index_type current_index)
{
  return test_lex (table, current_index, DFW_LEX_BEGIN_TUPLE);
}

/* Return 1 if the lexical element pointed by CURRENT_INDEX in TABLE
   corresponds to a DFW_SEM_LIST; else, return 0.  */

static int
test_list (struct result_table *table, index_type current_index)
{
  return test_lex (table, current_index, DFW_LEX_BEGIN_LIST);
}

/* Return 1 if the lexical element pointed by CURRENT_INDEX in TABLE
   corresponds to a DFW_SEM_RESULT; else, return 0.  */

static int
test_result (struct result_table *table, index_type current_index)
{
  return test_lex (table, current_index, DFW_LEX_ITEM)
    && test_lex (table, current_index + 2, DFW_LEX_EQUALS)
    && test_value (table, current_index + 3);
}

/* Return 1 if the lexical element pointed by CURRENT_INDEX in TABLE
   corresponds to a DFW_SEM_VALUE; else, return 0.  */

static int
test_value (struct result_table *table, index_type current_index)
{
  return test_const (table, current_index)
    || test_tuple (table, current_index)
    || test_list (table, current_index);
}

/* If the lexical element pointed by CURRENT_INDEX in TABLE
   is not of type LEX, raise an error; else, increment CURRENT_INDEX.  */

static int
check_lex (struct result_table *table, index_type *current_index,
	   enum dfw_lex_type lex)
{
  if (!test_lex (table, *current_index, lex))
    {
      error ("parse error in DFW output: missing %s\n",
	     dfw_lex_list [lex].name);
    }
  else
    {
      (*current_index)++;
    }

  return 0;
}

/* Read a result from the lex table. Syntax rule used:

   result ==>
     variable "=" value

*/

static void
read_result (struct result_table *table, index_type *current_index)
{
  read_variable (table, current_index);
  check_lex (table, current_index, DFW_LEX_EQUALS);
  read_value (table, current_index);
}

/* Read a variable from the lex table. Syntax rule used:

   variable ==>
     string

*/

static void
read_variable (struct result_table *table, index_type *current_index)
{
  dfw_tree_element item;

  check_lex (table, current_index, DFW_LEX_ITEM);
  dfw_tree_push (&(table->dfw_sem_tree), (dfw_tree_element) DFW_SEM_VARIABLE);

  item = table->dfw_lex_tree.beginning_of_table[*current_index];
  dfw_tree_push (&(table->dfw_sem_tree), item);
  (*current_index)++;
}

/* Read a const from the lex table. Syntax rule used:

   const ==>
     cstring

*/

static void
read_const (struct result_table *table, index_type *current_index)
{
  dfw_tree_element item;

  check_lex (table, current_index, DFW_LEX_CSTRING);
  dfw_tree_push (&(table->dfw_sem_tree), (dfw_tree_element) DFW_SEM_CONST);

  item = table->dfw_lex_tree.beginning_of_table[*current_index];
  dfw_tree_push (&(table->dfw_sem_tree), item);
  (*current_index)++;
}

/* Read a tuple from the lex table. Syntax rule used:

   tuple ==>
     "{}" | "{" result ( "," result )* "}"

*/

static void
read_tuple (struct result_table *table, index_type *current_index)
{
  check_lex (table, current_index, DFW_LEX_BEGIN_TUPLE);
  dfw_tree_push (&(table->dfw_sem_tree),
		 (dfw_tree_element) DFW_SEM_BEGIN_TUPLE);

  while (!test_lex (table, *current_index, DFW_LEX_END_TUPLE))
    {
      read_result (table, current_index);

      if (!test_lex (table, *current_index, DFW_LEX_END_TUPLE))
	{
	  check_lex (table, current_index, DFW_LEX_COMMA);
	}
    }

  check_lex (table, current_index, DFW_LEX_END_TUPLE);
  dfw_tree_push (&(table->dfw_sem_tree), (dfw_tree_element) DFW_SEM_END_TUPLE);
}

/* Read a list from the lex table. Syntax rule used:

   list ==>
     "[]"
     | "[" value ( "," value )* "]"
     | "[" result ( "," result )* "]"

*/

static void
read_list (struct result_table *table, index_type *current_index)
{
  check_lex (table, current_index, DFW_LEX_BEGIN_LIST);
  dfw_tree_push (&(table->dfw_sem_tree),
		 (dfw_tree_element) DFW_SEM_BEGIN_LIST);

  while (!test_lex (table, *current_index, DFW_LEX_END_LIST))
    {
      if (test_result (table, *current_index))
	{
	  read_result (table, current_index);
	}
      else if (test_value (table, *current_index))
	{
	  read_value (table, current_index);
	}
      else
	{
	  error ("parse error in DFW output: ambiguous list element\n");
	}

      if (!test_lex (table, *current_index, DFW_LEX_END_LIST))
	{
	  check_lex (table, current_index, DFW_LEX_COMMA);
	}
    }

  check_lex (table, current_index, DFW_LEX_END_LIST);
  dfw_tree_push (&(table->dfw_sem_tree),
		 (dfw_tree_element) DFW_SEM_END_LIST);
}

/* Read a value from the lex table. Syntax rule used:

   value ==>
    const | tuple | list

*/

static void
read_value (struct result_table *table, index_type *current_index)
{
  dfw_tree_push (&(table->dfw_sem_tree), (dfw_tree_element) DFW_SEM_VALUE);

  if (test_const (table, *current_index))
    {
      read_const (table, current_index);
    }
  else if (test_tuple (table, *current_index))
    {
      read_tuple (table, current_index);
    }
  else if (test_list (table, *current_index))
    {
      read_list (table, current_index);
    }
  else
    {
      error ("parse error in DFW output: ambiguous value.");
    }
}

/* Read a target stream from the DFW connection output. Syntax rule used:

   target-stream-output ==>
     "@" c-string

*/

static void
read_target_stream (int timeout)
{
  /* FIXME: not implemented.  */
  skip_line ();
}

/* Read a log stream from the DFW connection output. Syntax rule used:

   log-stream-output ==>
     "&" c-string

*/

static void
read_log_stream (int timeout)
{
  /* FIXME: not implemented.  */
  skip_line ();
}

/* Read until PATTERN is found. If PATTERN is not found after a max number
   of char (MAX_CHAR) is read, return 0; else, return 1.  */

static void
skip_string (char *pattern, int max_char)
{
  int current;
  int len = strlen (pattern);
  int i;
  char *p;

  for (i = 0; i < max_char; i++)
    {
      current = readchar (remote_timeout);
      if (current == pattern[0])
	{
	  for (p = pattern + 1; *p != '\0'; p++)
	    {
	      current = readchar (remote_timeout);
	      i++;
	      if (*p != current)
		break;
	    }
	  if (*p == '\0')
	    return;
	}
    }
}

static void
skip_prompt ()
{
  skip_string ("(gdb)\n", 10);
}

static void
skip_line ()
{
  skip_string ("\n", 1024);
}

#if 0
/* Perform partial transfers on OBJECT.  See target_read_partial
   and target_write_partial for details of each variant.  One, and
   only one, of readbuf or writebuf must be non-NULL.  */

static LONGEST
remote_xfer_partial (struct target_ops *ops, enum target_object object,
		     const char *annex, void *readbuf, const void *writebuf,
		     ULONGEST offset, LONGEST len)
{
  /* FIXME: Not Implemented. */
  return 0;
}
#endif

/* Read or write LEN bytes from inferior memory at MEMADDR,
   transferring to or from debugger address BUFFER.  Write to inferior if
   SHOULD_WRITE is nonzero.  Returns length of data written or read; 0
   for error.  TARGET is unused.  */

static int
remote_dfw_xfer_memory (CORE_ADDR mem_addr, char *buffer, int mem_len,
			int should_write, struct mem_attrib *attrib,
			struct target_ops *target)
{
  static const char read_command[] = "-wrs-gopher \"";
  static const char write_command[] = "-data-write-memory ";
  static int read_command_length = 0;
  static int write_command_length = 0;
  static const char read_options[] = " @b\"\n";
  static const char write_options[] = " x 1 ";
  static int read_options_length = 0;
  static int write_options_length = 0;
  char request_buffer[32];
  int offset;
  static const char read_memory_desc [] = "reading memory";
  struct dfw_sem_element element;
  index_type current_index = 0;
  CORE_ADDR result;

  if (!read_command_length)
    read_command_length = strlen (read_command);
  if (!write_command_length)
    write_command_length = strlen (write_command);
  if (!read_options_length)
    read_options_length = strlen (read_options);
  if (!write_options_length)
    write_options_length = strlen (write_options);

  thread_select_current_pid ();

  for (offset = 0; offset < mem_len; offset++)
    {
      write_token ();
      if (should_write)
	write_string (write_command, write_command_length);
      else
	write_string (read_command, read_command_length);

      sprintf (request_buffer, "0x%x", (unsigned int) (mem_addr + offset));
      write_string (request_buffer, strlen (request_buffer));

      if (should_write)
	{
	  write_string (write_options, write_options_length);
	  sprintf (request_buffer, "%d\n", *(buffer + offset));
	  write_string (request_buffer, strlen (request_buffer));
	}
      else
	{
	  write_string (read_options, read_options_length);
	}

      read_output_record_sequence (remote_timeout, current_token);
      current_index = 0;
      element = next_sem_element (&current_result_record, &current_index);
      if (element.type != DFW_SEM_RESULT_CLASS
	  || element.value.result_class_value == RESULT_CLASS_ERROR)
	{
	  return 0;
	}

      element = next_sem_element_with_check (&current_result_record,
					     &current_index,
					     read_memory_desc,
					     DFW_SEM_VARIABLE);

      if (!should_write)
	{
	  while (current_index
		 < current_result_record.dfw_sem_tree.table_end)
	    {
	      next_sem_element_with_check (&current_result_record,
					   &current_index,
					   read_memory_desc,
					   DFW_SEM_VALUE);
	      next_sem_element_with_check (&current_result_record,
					   &current_index,
					   read_memory_desc,
					   DFW_SEM_BEGIN_LIST);
	      next_sem_element_with_check (&current_result_record,
					   &current_index,
					   read_memory_desc,
					   DFW_SEM_VARIABLE);
	      sem_element_process_const_core_addr_value
		(&current_result_record,
		 &current_index,
		 read_memory_desc,
		 &result);
	      break;
	    }

	  *(buffer + offset) = (char) result;
	}
    }
  return offset;
}

static void
remote_dfw_find_new_threads (void)
{
  struct vxworks_task *current;
  ptid_t current_thread;

  if (!debugged_vxworks_task)
    return;

  update_vxworks_task_list ();
  for (current = vxworks_task_list; current != NULL; current = current->next)
    {
      if ((debugged_vxworks_task->pid == system_pid
	   && debugged_vxworks_task->tid == current->tid)
	  || (debugged_vxworks_task->pid != system_pid
	      && debugged_vxworks_task->pid == current->pid))
	{
	  current_thread = vxworks_task_ptid_build (current);
	  if (!in_thread_list (current_thread))
	    add_thread (current_thread);
	}
    }
}

static void
info_dfw_trees ()
{
  print_result_record (&current_async_record);
  print_result_record (&current_result_record);
}

/* Print the full list of task running of the VxWorks target.  */

static void
info_vxworks_tasks_command ()
{
  struct thread_info tp;
  char *extra_info;
  struct vxworks_task *current;
  ptid_t current_ptid;

 update_vxworks_task_list ();
 tp.num = 0;
  for (current = vxworks_task_list; current != NULL; current = current->next)
    {
      tp.ptid = vxworks_task_ptid_build (current);
      printf_filtered ("%d %s", tp.num, target_tid_to_str (tp.ptid));
      extra_info = target_extra_thread_info (&tp);

      if (extra_info)
	printf_filtered (" (%s)", extra_info);

      puts_filtered ("\n");
      tp.num++;
    }
}

/* Synchronize vxworks_task_list with the target system.  */

static void
update_vxworks_task_list ()
{
  static const char kernel_task_list_branch [] =
    "/ObjTypeList/KernelTask/ObjList";
  static const char rtp_list_branch [] =
    "/ObjTypeList/RTP/ObjList";
  static const char kernel_task_list_options [] = "*/{Id,TID}";
  static const char rtp_list_options [] = "*/{Id,TID,Children/*/*/{Id,TID}}";
  static int kernel_task_list_branch_len = 0;
  static int rtp_list_branch_len = 0;
  index_type current_index;
  static const char new_thread_desc [] = "finding new threads";
  struct dfw_sem_element sem_element;
  int id, rtp_id;
  CORE_ADDR tid, pid;
  struct vxworks_task* current_task;

  if (!kernel_task_list_branch_len)
    kernel_task_list_branch_len = strlen (kernel_task_list_branch);
  if (!rtp_list_branch_len)
    rtp_list_branch_len = strlen (rtp_list_branch);

  clear_vxworks_task_list ();

  wrs_info_retrieve (info_retrieval_core_branch,
		     info_retrieval_core_branch_len,
		     kernel_task_list_branch,
		     kernel_task_list_branch_len,
		     kernel_task_list_options);
  read_output_record_sequence (task_list_timeout, current_token);

  current_index = 0;
  sem_element_process_result_header (&current_result_record, &current_index,
				     new_thread_desc);

  /* The result will be a list of lists.  */
  next_sem_element_with_check (&current_result_record, &current_index,
			       new_thread_desc, DFW_SEM_VALUE);
  next_sem_element_with_check (&current_result_record, &current_index,
			       new_thread_desc, DFW_SEM_BEGIN_LIST);
  next_sem_element_with_check (&current_result_record, &current_index,
			       new_thread_desc, DFW_SEM_VALUE);
  next_sem_element_with_check (&current_result_record, &current_index,
			       new_thread_desc, DFW_SEM_BEGIN_LIST);

  /* Go through the kernel task list: */
  while (current_index < current_result_record.dfw_sem_tree.table_end)
    {
      if (test_next_sem_element (&current_result_record, current_index,
				 DFW_SEM_END_LIST))
	break;

      /* Id */

      sem_element = next_sem_element_with_check
	(&current_result_record, &current_index,
	 new_thread_desc, DFW_SEM_VARIABLE);
      sem_element_process_const_integer_value
	(&current_result_record, &current_index,
	 new_thread_desc, &id);

      /* TID */

      sem_element = next_sem_element_with_check
	(&current_result_record, &current_index,
	 new_thread_desc, DFW_SEM_VARIABLE);
      sem_element_process_const_core_addr_value
	(&current_result_record, &current_index,
	 new_thread_desc, &tid);

      current_task = vxworks_task_build (system_thread_id, system_pid,
					 id, tid);
      if (!lookup_vxworks_task (tid))
	{
	  add_vxworks_task (current_task);
	}
    }

  wrs_info_retrieve (info_retrieval_core_branch,
		     info_retrieval_core_branch_len,
		     rtp_list_branch,
		     rtp_list_branch_len,
		     rtp_list_options);
  read_output_record_sequence (task_list_timeout, current_token);

  current_index = 0;
  sem_element_process_result_header (&current_result_record, &current_index,
				     new_thread_desc);

  /* The result will be a list of lists.  */
  next_sem_element_with_check (&current_result_record, &current_index,
			       new_thread_desc, DFW_SEM_VALUE);
  next_sem_element_with_check (&current_result_record, &current_index,
			       new_thread_desc, DFW_SEM_BEGIN_LIST);
  next_sem_element_with_check (&current_result_record, &current_index,
			       new_thread_desc, DFW_SEM_VALUE);
  next_sem_element_with_check (&current_result_record, &current_index,
			       new_thread_desc, DFW_SEM_BEGIN_LIST);

  /* Go through the rtp list: */
  while (current_index < current_result_record.dfw_sem_tree.table_end)
    {
      if (test_next_sem_element (&current_result_record, current_index,
				 DFW_SEM_END_LIST))
	break;

      /* RTP Id */

      sem_element = next_sem_element_with_check
	(&current_result_record, &current_index,
	 new_thread_desc, DFW_SEM_VARIABLE);
      sem_element_process_const_integer_value
	(&current_result_record, &current_index,
	 new_thread_desc, &rtp_id);

      /* RTP TID */

      sem_element = next_sem_element_with_check
	(&current_result_record, &current_index,
	 new_thread_desc, DFW_SEM_VARIABLE);
      sem_element_process_const_core_addr_value
	(&current_result_record, &current_index,
	 new_thread_desc, &pid);

      /* Go throught this RTP's task list.  */

      next_sem_element_with_check (&current_result_record, &current_index,
				   new_thread_desc, DFW_SEM_VALUE);
      next_sem_element_with_check (&current_result_record, &current_index,
				   new_thread_desc, DFW_SEM_BEGIN_LIST);
      next_sem_element_with_check (&current_result_record, &current_index,
				   new_thread_desc, DFW_SEM_VALUE);
      next_sem_element_with_check (&current_result_record, &current_index,
				   new_thread_desc, DFW_SEM_BEGIN_LIST);
      while (current_index < current_result_record.dfw_sem_tree.table_end)
	{
	  if (test_next_sem_element (&current_result_record, current_index,
				     DFW_SEM_END_LIST))
	    {
	      next_sem_element_with_check (&current_result_record,
					   &current_index,
					   new_thread_desc,
					   DFW_SEM_END_LIST);
	      next_sem_element_with_check (&current_result_record,
					   &current_index,
					   new_thread_desc,
					   DFW_SEM_END_LIST);
	      break;
	    }

	  /* task Id */

	  sem_element = next_sem_element_with_check
	    (&current_result_record, &current_index,
	     new_thread_desc, DFW_SEM_VARIABLE);
	  sem_element_process_const_integer_value
	    (&current_result_record, &current_index,
	     new_thread_desc, &id);

	  /* task TID */

	  sem_element = next_sem_element_with_check
	    (&current_result_record, &current_index,
	     new_thread_desc, DFW_SEM_VARIABLE);
	  sem_element_process_const_core_addr_value
	    (&current_result_record, &current_index,
	     new_thread_desc, &tid);

	  current_task = vxworks_task_build (rtp_id, pid, id, tid);
	  if (!lookup_vxworks_task (tid))
	    {
	      add_vxworks_task (current_task);
	    }
	}
    }
}

static int
remote_dfw_thread_alive (ptid_t ptid)
{
  return (get_dfw_task_status (ptid_get_lwp (ptid)) != DFW_TASK_STATUS_DEAD);
}


static char *
remote_dfw_extra_thread_info (struct thread_info *tp)
{
  static char* buffer = NULL;
  char *name, *rtp_name;
  struct cleanup *old_cleanup;
  enum dfw_task_status task_status;
  struct vxworks_task *vxworks_task;

  if (!buffer)
    {
      make_cleanup (xfree, buffer);
      buffer = NULL;
    }

  if (ptid_get_lwp (tp->ptid) != 0)
    {
      name = xstrdup (get_dfw_object_name (ptid_get_lwp (tp->ptid)));
    }
  else
    {
      name = xstrdup ("??");
    }
  old_cleanup = make_cleanup (xfree, name);

  task_status = get_dfw_task_status (ptid_get_lwp (tp->ptid));
  if (ptid_get_pid (tp->ptid) == system_pid)
    {
      /* Kernel task.  */
      buffer = xstrprintf ("Kernel task, name=%s, tid=0x%lx, Status=%s",
			   name, ptid_get_tid (tp->ptid),
			   dfw_task_status_desc [task_status].name);
    }
  else
    {
      /* RTP task.  */
      vxworks_task = lookup_vxworks_task (ptid_get_tid (tp->ptid));
      if (!vxworks_task)
	rtp_name = "unknown";
      else
	rtp_name = get_dfw_object_name (vxworks_task->rtp_dfw_id);
      buffer = xstrprintf ("task in RTP %s, name=%s, tid=0x%lx, status=%s",
			   rtp_name, name, ptid_get_tid (tp->ptid),
			   dfw_task_status_desc [task_status].name);
     }
  do_cleanups (old_cleanup);
  return buffer;
}

/* Open a connection to a dfw target named NAME.  */

static void
remote_dfw_open (char *name, int from_tty)
{
  CLIENT *cl;
  char default_registry_host_name[] = "127.0.0.1";
  char *registry_host_name;
  struct sockaddr_in registry_addr;
  struct hostent *registry_hostent;
  unsigned long addr;
  int ptrace_sock;
  struct wtx_msg_svr_desc argp;
  struct wtx_msg_svr_desc clnt_res;
  static char null_string[] = "";
  static char pattern[] = "dfw";
  char *host_name_loc, *port_loc, *port_string;
  char *host_name = 0;
  int host_name_len = 0, port_len = 0;
  index_type current_index;
  dfw_tree_element item;
  struct dfw_tree_type *sem_table = &(current_result_record.dfw_sem_tree);
  struct dfw_sem_element sem_element;

  registry_host_name = getenv ("WIND_REGISTRY");
  if (!registry_host_name)
    registry_host_name = default_registry_host_name;

  addr = inet_addr (registry_host_name);
  if (addr == -1)
    {
      registry_hostent = (struct hostent *) gethostbyname (registry_host_name);
      if (registry_hostent == NULL)
        {
          error ("Invalid registry host name.");
        }
      addr = *(unsigned long *) registry_hostent->h_addr;
    }

  memset (&registry_addr, '\0', sizeof (registry_addr));

  registry_addr.sin_addr.s_addr = addr;
  registry_addr.sin_family = AF_INET;

  /* set to actual wtxreg port.  */
  registry_addr.sin_port = htons (wtx_registry_port);

  ptrace_sock = RPC_ANYSOCK;
  cl = clnttcp_create (&registry_addr, wtx_svc_base, wtx_rpc_version,
		       &ptrace_sock, 0, 0);

  if (cl == NULL)
    {
      clnt_pcreateerror (registry_host_name);
      error ("Couldn't connect to registry.");
    }
  else
    {
      memset((char *)&clnt_res, 0, sizeof (clnt_res));
      memset((char *)&argp, 0, sizeof (clnt_res));
      argp.core.prot_version = 20;
      argp.desc.name = pattern;
      argp.desc.type = null_string;
      argp.desc.key = null_string;

      if (clnt_call (cl, wtx_info_q_get_prog,
		     (xdrproc_t) xdr_wtx_msg_svr_desc,
		     (caddr_t) &argp,
		     (xdrproc_t) xdr_wtx_msg_svr_desc,
		     (caddr_t) &clnt_res,
		     TIMEOUT) != RPC_SUCCESS)
	{
	  error ("No dfwserver registered.");
	}
    }

  parse_service_key (clnt_res.desc.key, &host_name_loc,
		     &host_name_len, &port_loc, &port_len);
  clnt_destroy (cl);

  if (host_name != NULL)
    free (host_name);

  host_name = (char *) malloc (host_name_len + port_len + 2);
  memcpy (host_name, host_name_loc, host_name_len);
  *(host_name + host_name_len) = ':';
  memcpy (host_name + host_name_len + 1, port_loc, port_len);
  *(host_name + host_name_len + port_len + 1) = '\0';

  printf_unfiltered ("Connecting to DFW server %s...", host_name);

  if (remote_dfw_desc)
    serial_close (remote_dfw_desc);
  remote_dfw_desc = serial_open (host_name);

  if (!remote_dfw_desc)
    error ("Not able to connect to DFW server on %s", host_name);
  skip_prompt ();

  wrs_info_retrieve ("/Targets", 0, NULL, 0, "* -N");
  read_output_record_sequence (remote_timeout, current_token);

  if (!name)
    error ("No target name specified.");

  if (full_target_name)
    free (full_target_name);
  full_target_name = NULL;

  current_index = 0;
  sem_element = sem_element_process_result_header (&current_result_record,
						   &current_index,
						   "getting full target name");

  while (current_index < current_result_record.dfw_sem_tree.table_end)
    {
      sem_element = next_sem_element (&current_result_record, &current_index);

      if (sem_element.type == DFW_SEM_CONST)
	{
	  if (sem_element.value.string_value
	      && strncmp (sem_element.value.string_value,
			  name, strlen (name)) == 0)
	    {
	      full_target_name = xstrdup (sem_element.value.string_value);
	    }
	}
    }

  if (!full_target_name)
    error ("target %s not registered in DFW server.", name);

  printf_unfiltered ("Connecting to %s...\n", full_target_name);
  gdb_flush (gdb_stdout);

  /* Get the core name and the system thread id.  */
  {
    const char request_target_branch [] = "/Targets/";
    const char request_cores_branch [] = "/Cores";
    const char request_thread_id_branch [] = "/ThreadId";
    int request_len = strlen (request_target_branch)
      + strlen (full_target_name)
      + strlen (request_cores_branch);
    char *request = (char *) alloca (request_len + 1);

    strcpy (request, request_target_branch);
    strcat (request, full_target_name);
    strcat (request, request_cores_branch);
    wrs_info_retrieve (request, request_len, NULL, 0, "*");
    read_output_record_sequence (remote_timeout, current_token);

    current_index = 0;
    sem_element = next_sem_element (&current_result_record, &current_index);
    if (sem_element.type != DFW_SEM_RESULT_CLASS
	|| sem_element.value.result_class_value != RESULT_CLASS_DONE)
      error ("DFW: error while getting the core name.");

    /* Eat the name of the result variable, so that the next variable
       in the ouput will be the core name,  */
    sem_element = next_sem_element (&current_result_record, &current_index);
    if (sem_element.type != DFW_SEM_VARIABLE)
      error ("DFW: unexpected format in output.");

    if (core_name)
      free (core_name);
    core_name = NULL;

    for (sem_element
	   = next_sem_element (&current_result_record, &current_index);
	 current_index < current_result_record.dfw_sem_tree.table_end;
	 sem_element
	   = next_sem_element (&current_result_record, &current_index))
    {
      if (sem_element.type == DFW_SEM_VARIABLE)
	{
	  if (!core_name)
	    {
	      if (sem_element.value.string_value
		  && strcmp (sem_element.value.string_value, "Cores") != 0)
		core_name = xstrdup (sem_element.value.string_value);
	    }
	  else
	    error ("DFW: two cores are available.");
	}
    }

    info_retrieval_core_branch_len = strlen (request_target_branch)
      + strlen (full_target_name) + strlen (request_cores_branch) + 1
      + strlen (core_name);
    info_retrieval_core_branch =
      (char *) malloc (info_retrieval_core_branch_len + 1);
    memset (info_retrieval_core_branch, 0, info_retrieval_core_branch_len);

    strcpy (info_retrieval_core_branch, request_target_branch);
    strcat (info_retrieval_core_branch, full_target_name);
    strcat (info_retrieval_core_branch, request_cores_branch);
    strcat (info_retrieval_core_branch, "/");
    strcat (info_retrieval_core_branch, core_name);

    wrs_info_retrieve (info_retrieval_core_branch,
		       info_retrieval_core_branch_len,
		       request_thread_id_branch,
		       0,
		       "{Name}");
    read_output_record_sequence (remote_timeout, current_token);
  }

  system_thread_id = -1;

  current_index = 0;
  sem_element_process_result_header (&current_result_record, &current_index,
				     "getting system thread id");

  while (current_index < current_result_record.dfw_sem_tree.table_end)
    {
      sem_element = next_sem_element (&current_result_record, &current_index);

      if (sem_element.type == DFW_SEM_VARIABLE
	  && strcmp (sem_element.value.string_value, "Name") == 0)
	{

	  sem_element_process_const_integer_value (&current_result_record,
						   &current_index,
						   "getting system thread id",
						   &system_thread_id);
	}
    }

  if (system_thread_id == -1)
    error ("DFW: No system thread id returned.");

  /* Select the system thread.  */
  if (!thread_select (system_thread_id))
    error ("DFW: unable to switch to system thread.");

  push_target (&remote_dfw_ops);
  update_symbol_files ();
  printf_unfiltered ("done.\n");
}

/* Return the task status of the task whose DFW Id is DFW_ID.  */

static enum dfw_task_status
get_dfw_task_status (long dfw_id)
{
  static const char request_desc [] = "getting dfw task status";
  static const char task_status_branch [] = "/IdInfo";
  static int task_status_branch_len = 0;
  static char *options, *status;
  struct dfw_sem_element sem_element;
  index_type current_index = 0;
  enum dfw_task_status result = DFW_TASK_STATUS_LAST;
  int i;
  struct cleanup *old_cleanup = make_cleanup (xfree, options);

  if (!task_status_branch_len)
    task_status_branch_len = strlen (task_status_branch);

  options = xstrprintf ("{Status} -id %ld", dfw_id);

  wrs_info_retrieve (task_status_branch, task_status_branch_len,
		     NULL, 0, options);
  read_output_record_sequence (remote_timeout, current_token);

  sem_element = next_sem_element (&current_result_record, &current_index);
  if (sem_element.type != DFW_SEM_RESULT_CLASS)
    error ("DFW: error in %s.", request_desc);

  if (sem_element.value.result_class_value == RESULT_CLASS_ERROR)
    {
      result = DFW_TASK_STATUS_DEAD;
    }
  else
    {
      sem_element = next_sem_element_with_check
	(&current_result_record, &current_index,
	 request_desc, DFW_SEM_VARIABLE);
      sem_element = next_sem_element_with_check
	(&current_result_record, &current_index,
	 request_desc, DFW_SEM_VALUE);
      sem_element = next_sem_element_with_check
	(&current_result_record, &current_index,
	 request_desc, DFW_SEM_BEGIN_LIST);
      sem_element = next_sem_element_with_check
	(&current_result_record, &current_index,
	 request_desc, DFW_SEM_VARIABLE);
      sem_element_process_const_string_value
	(&current_result_record, &current_index,
	 request_desc, &status);
      result = lookup_enum_code (dfw_task_status_desc, status,
				 DFW_TASK_STATUS_LAST);
    }

  do_cleanups (old_cleanup);
  return result;
}

static void
remote_dfw_fetch_registers (int regno)
{
  int dfw_regno = REGISTER_DFW_REGNO (regno);
  int dfw_reggroup = REGISTER_DFW_REGGROUP (regno);
  static const char registers_branch [] = "Stack/0/RegisterInfo/Groups";
  static int registers_branch_len = 0;
  char group_branch [32];
  char options[32];
  int total_regs = gdbarch_num_regs (current_gdbarch)
    + gdbarch_num_pseudo_regs (current_gdbarch);
  int i;
  index_type current_index = 0;
  struct dfw_sem_element sem_element;
  static const char request_desc [] = "fetching registers";
  CORE_ADDR value;
  char regp[MAX_REGISTER_SIZE];

  /* Workaround for a strange DFW bug. The first request sometimes
     fails for no obvious reason.  */
  int tries = 2;

  if (!registers_branch_len)
    registers_branch_len = strlen (registers_branch);

  if (ptid_equal (inferior_ptid, null_ptid))
    return;

  thread_select (ptid_get_lwp (inferior_ptid));
  if (regno != -1)
    {
      if (dfw_reggroup < 0 || dfw_regno < 0)
	return;

      while (tries > 0)
	{
	  tries--;
	  sprintf (group_branch, "/%d/Registers", dfw_reggroup);
	  sprintf (options, "[%d-%d]/{HexVal}", dfw_regno, dfw_regno);
	  wrs_info_retrieve (registers_branch, registers_branch_len,
			     group_branch, 0, options);
	  read_output_record_sequence (remote_timeout, current_token);
	  current_index = 0;
	  sem_element = next_sem_element_with_check (&current_result_record,
						     &current_index,
						     request_desc,
						     DFW_SEM_RESULT_CLASS);
	  if (sem_element.value.result_class_value == RESULT_CLASS_ERROR)
	    continue;

	  next_sem_element_with_check (&current_result_record, &current_index,
				       request_desc, DFW_SEM_VARIABLE);
	  next_sem_element_with_check (&current_result_record, &current_index,
				       request_desc, DFW_SEM_VALUE);
	  next_sem_element_with_check (&current_result_record, &current_index,
				       request_desc, DFW_SEM_BEGIN_LIST);
	  next_sem_element_with_check (&current_result_record, &current_index,
				       request_desc, DFW_SEM_VARIABLE);
	  sem_element_process_const_raw_value (&current_result_record,
					       &current_index,
					       request_desc,
					       regp,
					       MAX_REGISTER_SIZE);
	  regcache_raw_supply (current_regcache, regno, regp);
	  break;
	}
    }
  else
    {
      for (i = 0; i < total_regs; i++)
	{
	  remote_dfw_fetch_registers (i);
	}
    }
}

/* Update the symbole files to synchronize it with the
   module list loaded on the target.  */

static void
update_symbol_files ()
{
  index_type current_index, count_index;
  struct dfw_sem_element sem_element;
  const char module_action_desc[] = "getting module info";
  struct section_addr_info *addrs;
  char *name, *junk;
  char *section_name;
  int section_index = 0;
  int current_section_index = 0;
  int number_of_sections = 0;
  CORE_ADDR start_addr = 0;
  struct stat s;

  wrs_info_retrieve ("Modules", 0, NULL, 0,
		     "*/{ObjFile,Sections/*/{Name,StartAddr}}");
  read_output_record_sequence (remote_timeout, current_token);

  current_index = 0;
  sem_element_process_result_header (&current_result_record, &current_index,
				     module_action_desc);

  /* The result will be a list of lists.  */
  next_sem_element_with_check (&current_result_record, &current_index,
			       module_action_desc, DFW_SEM_VALUE);
  next_sem_element_with_check (&current_result_record, &current_index,
			       module_action_desc, DFW_SEM_BEGIN_LIST);
  next_sem_element_with_check (&current_result_record, &current_index,
			       module_action_desc, DFW_SEM_VALUE);
  next_sem_element_with_check (&current_result_record, &current_index,
			       module_action_desc, DFW_SEM_BEGIN_LIST);

  /* Go through the module list: */
  while (current_index < current_result_record.dfw_sem_tree.table_end)
    {
      if (test_next_sem_element (&current_result_record, current_index,
				 DFW_SEM_END_LIST))
	  break;

      sem_element = next_sem_element_with_check
	(&current_result_record, &current_index,
	 module_action_desc, DFW_SEM_VARIABLE);

      /* ObjFile.  */
      sem_element_process_const_string_value
	(&current_result_record, &current_index,
	 module_action_desc, &name);

      sem_element = next_sem_element (&current_result_record,
				      &current_index);

      /* Count the sections:  */
      sem_element = next_sem_element_with_check
	(&current_result_record, &current_index,
	 module_action_desc, DFW_SEM_BEGIN_LIST);
      number_of_sections = 0;
      count_index = current_index;

      while (count_index
	     < current_result_record.dfw_sem_tree.table_end)
	{
	  if (test_next_sem_element (&current_result_record,
				     count_index, DFW_SEM_END_LIST))
	    break;

	  number_of_sections++;
	  sem_element = next_sem_element_with_check
	    (&current_result_record, &count_index,
	     module_action_desc, DFW_SEM_VARIABLE);
	  sem_element = sem_element_process_const_string_value
	    (&current_result_record, &count_index,
	     module_action_desc, &junk);
	  sem_element = next_sem_element_with_check
	    (&current_result_record, &count_index,
	     module_action_desc, DFW_SEM_VARIABLE);
	  sem_element = sem_element_process_const_string_value
	    (&current_result_record, &count_index,
	     module_action_desc, &junk);
	}

      addrs = alloc_section_addr_info (number_of_sections);

      /* Populate the section addr info structure:  */
      for (current_section_index = 0;
	   current_section_index < number_of_sections;
	   current_section_index++)
	{
	  sem_element = next_sem_element_with_check
	    (&current_result_record, &current_index,
	     module_action_desc, DFW_SEM_VARIABLE);
	  sem_element = sem_element_process_const_string_value
	    (&current_result_record, &current_index,
	     module_action_desc, &section_name);
	  sem_element = next_sem_element_with_check
	    (&current_result_record, &current_index,
	     module_action_desc, DFW_SEM_VARIABLE);
	  sem_element = sem_element_process_const_core_addr_value
	    (&current_result_record, &current_index,
	     module_action_desc, &start_addr);
	  
	  addrs->other[current_section_index].name = section_name;
	  addrs->other[current_section_index].addr = start_addr;
	}
      
      if (stat (name, &s) == 0)
	symbol_file_add (name, 0, addrs, 0, OBJF_USERLOADED);

      sem_element = next_sem_element_with_check
	(&current_result_record, &current_index,
	 module_action_desc, DFW_SEM_END_LIST);

    }
}

/* Clean up connection to a dfw target.  */

static void
remote_dfw_close (int quitting)
{
}

static void
remote_dfw_files_info (struct target_ops *ignore)
{
  printf_unfiltered ("Attached to target server %s, CPU is %s.\n",
		     full_target_name, core_name);
}

/* Attach to the task/RTP which tid is given in paramter.  */

static void
remote_dfw_attach (char *args, int from_tty)
{
  unsigned long id;
  char *cptr = 0;
  struct vxworks_task *task;

  if (!args)
    error_no_arg ("process-id to attach");

  id = strtoul (args, &cptr, 0);
  if ((cptr == args) || (*cptr != '\0'))
    error ("Invalid tid -- give a single number in decimal or 0xhex");

  init_thread_list ();
  update_vxworks_task_list ();

  task = lookup_vxworks_task (id);

  if (!task)
    error ("No task or process with id %s", args);

  if (debugged_vxworks_task)
    xfree (debugged_vxworks_task);

  debugged_vxworks_task = duplicate_vxworks_task (task);
  inferior_ptid = vxworks_task_ptid_build (debugged_vxworks_task);
  thread_select_current_pid ();
  push_target (&remote_dfwtask_ops);
  update_symbol_files ();
  exec_interrupt ();
  if (from_tty)
    printf_unfiltered ("Attaching task %s.\n",
		       hex_string ((unsigned long) id));
}

static void
remote_dfw_detach (char *args, int from_tty)
{
  struct dfw_breakpoint *current;

  for (current = dfw_breakpoint_list; current; current = current->next)
    remote_dfw_remove_breakpoint (current->address, NULL);
  thread_select_current_pid ();
  exec_continue ();
  debugged_vxworks_task = NULL;
  clear_vxworks_task_list ();
  unpush_target (&remote_dfwtask_ops);
}

static void
remote_dfw_mourn_inferior ()
{
  pop_target ();
  generic_mourn_inferior ();
}

static void
remote_dfw_kill ()
{
  const static char kill_desc[] = "killing inferior";
  const static char request[] = "-wrs-tos-object-delete -t ";
  static int request_len = 0;
  int thread_selected = thread_select_current_pid ();
  char *thread = xstrprintf ("%d\n", thread_selected);
  struct cleanup *old_cleanup = make_cleanup (xfree, thread);
  index_type current_index = 0;

  if (!request_len)
    request_len = strlen (request);

  write_token ();
  write_string (request, request_len);
  write_string (thread, strlen (thread));
  read_output_record_sequence (remote_timeout, current_token);
  sem_element_process_result_header (&current_result_record, &current_index,
				     kill_desc);
  target_mourn_inferior ();
  do_cleanups (old_cleanup);
}

static void
remote_dfw_unload (char *arg, int from_tty)
{
  static const char unload_desc [] = "unloading module";
  static const char unload_request [] = "-wrs-target-unload -module ";
  static int unload_request_len = 0;
  const char *base_name = lbasename (arg);
  char *module_name;
  index_type current_index = 0;
  struct dfw_sem_element sem_element;

  if (!unload_request_len)
    unload_request_len = strlen (unload_request);

  thread_select (system_thread_id);
  wrs_info_retrieve ("Modules", 0, NULL, 0, "*/{ObjFile}");
  read_output_record_sequence (remote_timeout, current_token);

  current_index = 0;
  sem_element_process_result_header (&current_result_record, &current_index,
				     unload_desc);

  /* The result will be a list of lists.  */
  next_sem_element_with_check (&current_result_record, &current_index,
			       unload_desc, DFW_SEM_VALUE);
  next_sem_element_with_check (&current_result_record, &current_index,
			       unload_desc, DFW_SEM_BEGIN_LIST);
  next_sem_element_with_check (&current_result_record, &current_index,
			       unload_desc, DFW_SEM_VALUE);
  next_sem_element_with_check (&current_result_record, &current_index,
			       unload_desc, DFW_SEM_BEGIN_LIST);

  /* Decode module information: */
  while (current_index
	 < current_result_record.dfw_sem_tree.table_end)
    {
      if (test_next_sem_element (&current_result_record, current_index,
				 DFW_SEM_END_LIST))
	break;

      sem_element = next_sem_element_with_check
	(&current_result_record, &current_index,
	 unload_desc, DFW_SEM_VARIABLE);

      /* ObjFile.  */
      sem_element_process_const_string_value
	(&current_result_record, &current_index,
	 unload_desc, &module_name);

      if (strcmp (lbasename (module_name), base_name) == 0)
	{
	  write_token ();
	  write_string (unload_request, unload_request_len);
	  write_string (module_name, strlen (module_name));
	  write_string ("\n", 1);
	  read_output_record_sequence (load_timeout, current_token);

	  current_index = 0;
	  sem_element_process_result_header (&current_result_record,
					     &current_index,
					     unload_desc);
	  return;
	}
    }
  error ("Module %s not found on target", arg);
}

static void
remote_dfw_load (char *arg, int from_tty)
{
  static const char *rtp_option = "rtp ";
  static int rtp_option_len = 0;

  if (!rtp_option_len)
    rtp_option_len = strlen (rtp_option);

  if (strncmp (arg, rtp_option, rtp_option_len) == 0)
    remote_dfw_load_rtp_module (arg + rtp_option_len, from_tty);
  else
    remote_dfw_load_kernel_module (arg, from_tty);
}

static void
remote_dfw_send (char *arg, int from_tty)
{
  int saved_dfw_show_responses = dfw_show_responses;
  dfw_show_responses = 1;
  write_token ();
  write_string (arg, strlen (arg));
  write_string ("\n", 1);
  read_output_record_sequence (remote_timeout, current_token);
  dfw_show_responses = saved_dfw_show_responses;
}


static void
remote_dfw_load_kernel_module (char *arg, int from_tty)
{ 
  static const char load_desc [] = "loading kernel module";
  static const char request [] = "-wrs-target-download ";
  char *full_pathname;
  static int request_len = 0;
  index_type current_index = 0;
  struct dfw_sem_element sem_element;

  if (!request_len)
    request_len = strlen (request);

  if (!source_full_path_of (arg, &full_pathname))
    {
      /* The file cannot be found on the debugger host.  It may just be
	 on the DFW server host.  */
      full_pathname = arg;
    }

  write_token ();
  write_string (request, request_len);
  write_string (full_pathname, strlen (full_pathname));
  write_string ("\n", 1);
  read_output_record_sequence (load_timeout, current_token);
  sem_element = next_sem_element_with_check (&current_result_record,
					     &current_index,
					     load_desc,
					     DFW_SEM_RESULT_CLASS);
  if (sem_element.value.result_class_value == RESULT_CLASS_ERROR)
    error ("File not found: %s", full_pathname);

  update_symbol_files ();
}

static void
remote_dfw_load_rtp_module (char *arg, int from_tty)
{
  static const char load_desc [] = "loading RTP module";
  static const char request [] = "-wrs-tos-object-create RTP rtpSpawn -- \"";
  static const char request_suffix [] = "\" 100 0x10000 0x11 0x0 0x0 0x0 0x0  -e LD_BIND_NOW 1\n";
  char *full_pathname = arg;
  static int request_len = 0;
  static int request_suffix_len = 0;
  index_type current_index = 0;
  struct dfw_sem_element sem_element;
  int dfw_id = system_thread_id;

  if (!request_len)
    request_len = strlen (request);
  if (!request_suffix_len)
    request_suffix_len = strlen (request_suffix);

  write_token ();
  write_string (request, request_len);
  write_string (full_pathname, strlen (full_pathname));
  write_string (request_suffix, request_suffix_len);
  read_output_record_sequence (load_timeout, current_token);
  sem_element = next_sem_element_with_check (&current_result_record,
					     &current_index,
					     load_desc,
					     DFW_SEM_RESULT_CLASS);
  if (sem_element.value.result_class_value == RESULT_CLASS_ERROR)
    error ("File not found: %s", full_pathname);

  sem_element = next_sem_element_with_check (&current_result_record,
					     &current_index,
					     load_desc,
					     DFW_SEM_VARIABLE);
  sem_element = sem_element_process_const_integer_value
    (&current_result_record, &current_index, load_desc, &dfw_id);

  update_vxworks_task_list ();
  if (debugged_vxworks_task)
    xfree (debugged_vxworks_task);
  debugged_vxworks_task =
    duplicate_vxworks_task (lookup_vxworks_task_by_dfw_id (dfw_id));
  if (!debugged_vxworks_task)
    error ("Could not find the RTP module. Try to attach it.");
  thread_select (dfw_id);
  update_symbol_files ();
  push_target (&remote_dfwtask_ops);
  set_initial_language ();
}

/* Send ^C to target to halt it.  Target will respond, and send us a
   packet.  */
static void (*ofunc) (int);

/* Ask the user what to do when an interrupt is received.  */

static void
interrupt_query (void)
{
  target_terminal_ours ();

  if (query ("Interrupted while waiting for the program.\n\
Give up (and stop debugging it)? "))
    {
      target_mourn_inferior ();
      throw_exception (RETURN_QUIT);
    }

  target_terminal_inferior ();
}

/* The command line interface's stop routine. This function is installed
   as a signal handler for SIGINT. The first time a user requests a
   stop, we call remote_stop to send a break or ^C. If there is no
   response from the target (it didn't stop when the user requested it),
   we ask the user if he'd like to detach from the target. */

static void
remote_dfw_interrupt (int signo)
{
  /* If this doesn't work, try more severe steps. */
  signal (signo, remote_dfw_interrupt_twice);
  target_stop ();
}

/* The user typed ^C twice.  */

static void
remote_dfw_interrupt_twice (int signo)
{
  signal (signo, ofunc);
  interrupt_query ();
  signal (signo, remote_dfw_interrupt);
}

static ptid_t
remote_dfw_wait (ptid_t ptid, struct target_waitstatus *status)
{
  char output_type = '\0';
  enum async_class_type class;
  static index_type current_index = 0;
  char *buf;
  char event_loop_desc [] = "interpreting target event";
  struct dfw_sem_element sem_element;
  int dfw_id = system_thread_id;
  int bp_id;
  char *context_type;
  int return_value;
  ptid_t event_ptid = vxworks_task_ptid_build (debugged_vxworks_task);
  token_type token;

  while (current_index < current_async_record.dfw_sem_tree.table_end)
    {
      sem_element = next_sem_element (&current_async_record, &current_index);
      if (sem_element.type == DFW_SEM_ASYNC_CLASS)
	break;
    }

  if (current_index == current_async_record.dfw_sem_tree.table_end)
    {
      result_table_reset (&current_async_record);
      current_index = 0;

      while (1)
	{
	  ofunc = signal (SIGINT, remote_dfw_interrupt);
	  output_type = wait_output_record_sequence (-1, &token);
	  
	  if (output_type == '*' || output_type == '=' || output_type == '+')
	    break;
	}
      sem_element = next_sem_element_with_check (&current_async_record,
						 &current_index,
						 event_loop_desc,
						 DFW_SEM_ASYNC_CLASS);
    }

  class = sem_element.value.async_class_value;
  switch (class)
    {
    case ASYNC_CLASS_STOPPED:
      status->kind = TARGET_WAITKIND_STOPPED;
      status->value.sig = TARGET_SIGNAL_TRAP;
      while (current_index < current_async_record.dfw_sem_tree.table_end)
	{
	  if (test_next_sem_element (&current_async_record, current_index,
				     DFW_SEM_ASYNC_CLASS))
	    break;

	  sem_element = next_sem_element_with_check (&current_async_record,
						     &current_index,
						     event_loop_desc,
						     DFW_SEM_VARIABLE);

	  if (strcmp (sem_element.value.string_value, "reason") == 0)
	    {
	      sem_element = sem_element_process_const_string_value
		(&current_async_record, &current_index, event_loop_desc, &buf);
	      if (strcmp (buf, "user-stopped") == 0)
		{
		  status->value.sig = TARGET_SIGNAL_INT;
		}
	      else
		{
		  status->value.sig = TARGET_SIGNAL_TRAP;
		}
	    }

	  else if (strcmp (sem_element.value.string_value, "thread-id") == 0)
	    {
	      sem_element = sem_element_process_const_integer_value
		(&current_async_record, &current_index, event_loop_desc,
		 &dfw_id);
	      event_ptid = get_ptid_from_dfw_id (dfw_id);
	      if (!is_being_debugged (event_ptid))
		{
		  status->kind = TARGET_WAITKIND_IGNORE;
		}
	      else
		{
		  debugged_vxworks_task =
		    duplicate_vxworks_task
		    (lookup_vxworks_task_by_dfw_id (dfw_id));
		}
	    }

	  else
	    {
	      /* Skip to next variable.  */
	      while (current_index
		     < current_async_record.dfw_sem_tree.table_end)
		{
		  if (test_next_sem_element (&current_async_record,
					     current_index,
					     DFW_SEM_VARIABLE))
		    break;
		  
		  next_sem_element (&current_async_record, &current_index);
		}
	      
	    }
	}
      break;
    case ASYNC_CLASS_CONTEXT_EXIT:
      status->kind = TARGET_WAITKIND_EXITED;
      /* DFW Thread id */
      sem_element = next_sem_element_with_check (&current_async_record,
						 &current_index,
						 event_loop_desc,
						 DFW_SEM_VARIABLE);
      if (sem_element.type == DFW_SEM_VARIABLE
	  && strcmp (sem_element.value.string_value, "thread-id") == 0)
	{
	  sem_element = sem_element_process_const_integer_value
	    (&current_async_record, &current_index, event_loop_desc,
	     &dfw_id);
	  if (!is_being_debugged_1 (dfw_id))
	    {
	      status->kind = TARGET_WAITKIND_IGNORE;
	    }
	  else
	    {
	      debugged_vxworks_task =
		duplicate_vxworks_task
		(lookup_vxworks_task_by_dfw_id (dfw_id));
	    }
	}
      /* context type.  */
      sem_element = next_sem_element_with_check (&current_async_record,
						 &current_index,
						 event_loop_desc,
						 DFW_SEM_VARIABLE);
      sem_element = sem_element_process_const_string_value
	(&current_async_record, &current_index, event_loop_desc,
	 &context_type);
      /* Return value.  */
      sem_element = next_sem_element_with_check (&current_async_record,
						 &current_index,
						 event_loop_desc,
						 DFW_SEM_VARIABLE);
      if (sem_element.type == DFW_SEM_VARIABLE
	  && strcmp (sem_element.value.string_value, "return-value") == 0)
	{
	  sem_element = sem_element_process_const_integer_value
	    (&current_async_record, &current_index, event_loop_desc,
	     &return_value);
	  status->value.integer = return_value;
	}
      break;
    case ASYNC_CLASS_CONTAINER_STOPPED:
      /* Not sure what to do in this case for now.  */
      status->kind = TARGET_WAITKIND_IGNORE;
      break;
    default:
      status->kind = TARGET_WAITKIND_IGNORE;
      break;
    }

  if (is_being_debugged (event_ptid))
    {
      inferior_ptid = event_ptid;
      thread_select (debugged_vxworks_task->dfw_id);
    }
  return inferior_ptid;
}

static void
remote_dfw_resume (ptid_t ptid, int step, enum target_signal siggnal)
{
  int pid = ptid_get_pid (inferior_ptid);
  int tid = ptid_get_pid (inferior_ptid);
  int lwp = ptid_get_lwp (inferior_ptid);
  int thread_to_select;

  if (!step)
    {
      if (pid == system_pid)
	{
	  thread_to_select = lwp;
	}
      else
	{
	  thread_to_select = lookup_vxworks_task (tid)->rtp_dfw_id;
	}
      thread_select (thread_to_select);
      exec_continue ();
    }
  else
    {
      thread_select (lwp);
      exec_step_instruction ();
    }

  /* FIXME: is there support in DFW for target signals ?  */
}

/* Look up a symbol in the target's symbol table.  NAME is the symbol
   name.  ADDRP is a CORE_ADDR * pointing to where the value of the
   symbol should be returned.  The result is 0 if successful, nonzero
   if the symbol does not exist in the target environment.  This
   function should not call error() if communication with the target
   is interrupted, since it is called from symbol reading, but should
   return nonzero, possibly doing a complain().  */

static int
remote_dfw_lookup_symbol (char *name, CORE_ADDR *address)
{
  const static char lookup_symbol_desc [] = "looking up symbol";
  const static char request_prefix [] = "-wrs-sym-list -fullname ^";
  const static char request_suffix [] =
    "$ -global -weak -kind mapvar -output address\n";
  static int request_prefix_len = 0;
  static int request_suffix_len = 0;
  struct dfw_sem_element sem_element;
  index_type current_index = 0;

  if (!request_prefix_len)
    request_prefix_len = strlen (request_prefix);
  if (!request_suffix_len)
    request_suffix_len = strlen (request_suffix);

  write_token ();
  write_string (request_prefix, request_prefix_len);
  write_string (name, strlen (name));
  write_string (request_suffix, request_suffix_len);
  read_output_record_sequence (remote_timeout, current_token);
  sem_element_process_result_header (&current_result_record, &current_index,
				     lookup_symbol_desc);
  next_sem_element_with_check (&current_result_record, &current_index,
			       lookup_symbol_desc, DFW_SEM_VALUE);
  next_sem_element_with_check (&current_result_record, &current_index,
			       lookup_symbol_desc, DFW_SEM_BEGIN_LIST);
  if (test_next_sem_element (&current_result_record, current_index,
			     DFW_SEM_END_LIST))
    return -1;

  sem_element_process_const_core_addr_value (&current_result_record,
					     &current_index,
					     lookup_symbol_desc,
					     address);
  return 0;
}

/* Tokenise string into arguments. On each call with BUF having a non-zero
   value, returns the first argument from BUF. On subsequent calls with BUFF
   passed as zero the remaining arguments are returned on at a time until
   zero is returned. BUF is tokenised by placing '\0' characters into it
   therefore it must be writable. Arguments can include quoted strings which
   may include escaped characters including '\"'. Returns a pointer to the
   next arg or 0 if no more args are present.  */

static char *
arg_tok (char *buf)
{
  char *start;
  static char *remainder = 0;

  if (buf == 0)
    buf = remainder;

  if (buf == 0)
    return 0;

  while (buf[0] == '\t' || buf[0] == ' ')
    buf++;
  start = buf;

  if (buf[0] == '\"')
    {
      /* Quoted string */
      while ((++buf)[0]) 
	{
	  if (buf[0] == '\\' && buf[1] != '\0')
	    buf++;
	  else if (buf[0] == '\"')
	    break;
	}

      if (buf[0])
	buf++;
    }
  else
    while (buf[0] && buf[0] != ' ' && buf[0] != '\t')
      buf++;
  
  if (buf[0] == '\0' || buf[1] == '\0')
    remainder = 0;	      /* Already reached end of string */
  else
    {
      buf[0] = '\0';        /* Break string with null  */
      remainder = buf + 1;  /* Mark next part to parse */
    }

  return (start == buf) ? 0 : start;
}

static void
remote_dfw_create_inferior (char *exec_file, char *args, char **env,
			    int from_tty)
{
  if (debugged_vxworks_task && debugged_vxworks_task->pid != system_pid)
    remote_dfw_create_inferior_rtp (exec_file, args, env, from_tty);
  else
    remote_dfw_create_inferior_kernel_task (exec_file, args, env, from_tty);
}

static void
remote_dfw_create_inferior_rtp (char *exec_file, char *args,
				char **env, int from_tty)
{
  thread_select_current_pid ();
  inferior_ptid = vxworks_task_ptid_build (debugged_vxworks_task);
  proceed (-1, -1, 0);
}

static void
remote_dfw_create_inferior_kernel_task (char *exec_file, char *args,
					char **env, int from_tty)
{
  static const char run_desc [] = "creating inferior";
  static const char run_task_request_suffix [] = " true true\n";
  static const char run_task_request [] =
    "-wrs-tos-object-create KernelTask taskSpawnStopped -- gdbtask 100 0x10 0x0 0x10000 ";
  static int run_task_request_len = 0;
  static int run_task_request_suffix_len = 0;
  CORE_ADDR entry_point_address;
  char *entry_point_name = arg_tok (args);
  const char *entry_point_address_as_string;
  index_type current_index = 0;
  int dfw_id;
  struct inferior_info *tp;
  struct target_waitstatus ecs;

  if (!run_task_request_len)
    run_task_request_len = strlen (run_task_request);
  if (!run_task_request_suffix_len)
    run_task_request_suffix_len = strlen (run_task_request_suffix);

  if (remote_dfw_lookup_symbol (entry_point_name, &entry_point_address) != 0)
    error ("Failed to lookup entry point address");

  entry_point_address_as_string = core_addr_to_string (entry_point_address);
  write_token ();
  write_string (run_task_request, run_task_request_len);
  write_string (entry_point_name, strlen (entry_point_name));
  write_string (run_task_request_suffix, run_task_request_suffix_len);
  read_output_record_sequence (remote_timeout, current_token);
  sem_element_process_result_header (&current_result_record, &current_index,
				     run_desc);
  sem_element_process_const_integer_value (&current_result_record,
					   &current_index, run_desc,
					   &dfw_id);
  update_vxworks_task_list ();
  if (debugged_vxworks_task)
    xfree (debugged_vxworks_task);

  debugged_vxworks_task =
    duplicate_vxworks_task (lookup_vxworks_task_by_dfw_id (dfw_id));
  thread_select_current_pid ();
  inferior_ptid = vxworks_task_ptid_build (debugged_vxworks_task);
  push_target (&remote_dfwtask_ops);

  init_wait_for_inferior ();
  ecs.kind = TARGET_WAITKIND_IGNORE;
  while (ecs.kind != TARGET_WAITKIND_STOPPED
	 || ecs.value.sig != TARGET_SIGNAL_TRAP)
    {
      remote_dfw_wait (inferior_ptid, &ecs);
    }
  proceed (-1, -1, 0);
}

static void
remote_dfw_stop ()
{
  if (thread_select_current_pid () != system_thread_id)
      exec_interrupt ();
}

/* Insert a breakpoint on target. ADDR is the target location in the target
   machine.  CONTENTS_CACHE is a pointer to memory allocated for saving the
   target contents. It is guaranteed by the caller to be long enough to save
   the number of bytes returned by BREAKPOINT_FROM_PC.  */

static int
remote_dfw_insert_breakpoint (CORE_ADDR address, char *contents_cache)
{
  static const char insert_bp_desc[] = "inserting breakpoints";
  static const char request [] = "-wrs-break-insert *";
  static int request_len = 0;
  char *address_hex_string = hex_string ((unsigned long) address);
  int bp_number;
  index_type current_index = 0;

  if (!request_len)
    request_len = strlen (request);

  thread_select_current_pid ();

  write_token ();
  write_string (request, request_len);
  write_string (address_hex_string, strlen (address_hex_string));
  write_string ("\n", 1);
  read_output_record_sequence (remote_timeout, current_token);

  sem_element_process_result_header (&current_result_record, &current_index,
				     insert_bp_desc);
  next_sem_element_with_check (&current_result_record, &current_index,
			       insert_bp_desc, DFW_SEM_VALUE);
  next_sem_element_with_check (&current_result_record, &current_index,
			       insert_bp_desc, DFW_SEM_BEGIN_TUPLE);

  /* breakpoint number.  */
  next_sem_element_with_check (&current_result_record, &current_index,
			       insert_bp_desc, DFW_SEM_VARIABLE);
  sem_element_process_const_integer_value (&current_result_record,
					   &current_index, insert_bp_desc,
					   &bp_number);
  add_dfw_breakpoint (address, bp_number);
  return 0;
}

/* Remove a breakpoint on target. ADDR is the target location in the target
   machine.  */

static int
remote_dfw_remove_breakpoint (CORE_ADDR address, char *contents_cache)
{
  static const char remove_bp_desc [] = "removing breakpoints";
  static const char request [] = "-break-delete ";
  static int request_len = 0;
  static char bp_number_string [32];
  int bp;
  index_type current_index;

  if (!request_len)
    request_len = strlen (request);

  thread_select_current_pid ();

  while ((bp = extract_dfw_breakpoint (address)) != -1)
    {
      sprintf (bp_number_string, "%d\n", bp);
      write_token ();
      write_string (request, request_len);
      write_string (bp_number_string, strlen (bp_number_string));
      read_output_record_sequence (remote_timeout, current_token);
      current_index = 0;
      sem_element_process_result_header (&current_result_record,
					 &current_index,
					 remove_bp_desc);
    }
  return 0;
}

static void
init_remote_dfwtask_ops (void)
{
  remote_dfwtask_ops = remote_dfw_ops;
  remote_dfwtask_ops.to_shortname = "dfwtask";
  remote_dfwtask_ops.to_longname = "WRS' DFW protocol (task mode)";
  remote_dfwtask_ops.to_doc = "Use WRS' DFW protocol (task mode)";
  remote_dfwtask_ops.to_has_stack = 1;
  remote_dfwtask_ops.to_has_registers = 1;
  remote_dfwtask_ops.to_has_execution = 1;
  remote_dfwtask_ops.to_stratum = process_stratum;
  remote_dfwtask_ops.to_magic = OPS_MAGIC;
  remote_dfwtask_ops.to_mourn_inferior = remote_dfw_mourn_inferior;
  remote_dfwtask_ops.to_find_new_threads = remote_dfw_find_new_threads;
  remote_dfwtask_ops.to_fetch_registers = remote_dfw_fetch_registers;
  remote_dfwtask_ops.to_thread_alive = remote_dfw_thread_alive;
  remote_dfwtask_ops.to_kill = remote_dfw_kill;
  remote_dfwtask_ops.to_wait = remote_dfw_wait;
  remote_dfwtask_ops.to_resume = remote_dfw_resume;
  remote_dfwtask_ops.to_stop = remote_dfw_stop;
}

static void
init_remote_dfw_ops (void)
{
  remote_dfw_ops.to_shortname = "dfw";
  remote_dfw_ops.to_longname = "Wind River System's DFW protocol";
  remote_dfw_ops.to_doc = "Use Wind River System's DFW protocol";
  remote_dfw_ops.to_has_memory = 1;
  remote_dfw_ops.to_has_all_memory = 1;
  remote_dfw_ops.to_has_stack = 0;
  remote_dfw_ops.to_has_registers = 0;
  remote_dfw_ops.to_has_execution = 0;
  remote_dfw_ops.to_stratum = core_stratum;
  remote_dfw_ops.to_magic = OPS_MAGIC;
  remote_dfw_ops.to_open = remote_dfw_open;
  remote_dfw_ops.to_close = remote_dfw_close;
  remote_dfw_ops.to_files_info = remote_dfw_files_info;
  remote_dfw_ops.to_extra_thread_info = remote_dfw_extra_thread_info;
  remote_dfw_ops.deprecated_xfer_memory = remote_dfw_xfer_memory;
#if 0
  /* FIXME : Implement xfer partial, instead of using the deprecated
     xfer model.  */
  remote_dfw_ops.to_xfer_partial = remote_dfw_xfer_partial;
#endif
  remote_dfw_ops.to_attach = remote_dfw_attach;
  remote_dfw_ops.to_detach = remote_dfw_detach;
  remote_dfw_ops.to_insert_breakpoint = remote_dfw_insert_breakpoint;
  remote_dfw_ops.to_remove_breakpoint = remote_dfw_remove_breakpoint;
  remote_dfw_ops.to_load = remote_dfw_load;
  remote_dfw_ops.to_create_inferior = remote_dfw_create_inferior;
}

void
_initialize_remote_dfw (void)
{
  struct cmd_list_element *c;

  init_remote_dfw_ops ();
  add_target (&remote_dfw_ops);
  init_remote_dfwtask_ops ();
  add_target (&remote_dfwtask_ops);
  result_table_allocate (&current_result_record);
  result_table_allocate (&current_async_record);

  c = add_cmd ("unload", class_files, remote_dfw_unload,
	       "Unload and close an object module that is "
	       "currently being debugged.\n", &cmdlist);
  set_cmd_completer (c, filename_completer);

  add_setshow_boolean_cmd ("dfw-show-requests", no_class, &dfw_show_requests,"\
Set whether to show the requests sent to the DFW server.", "\
Show whether to show the requests sent to the DFW server.", "\
If set, the requests sent to the DFW server are printed to stdout.", "\
Whether to show DFW requests is %s.",
			   NULL, NULL,
			   &setlist, &showlist);

  add_setshow_boolean_cmd ("dfw-show-responses", no_class,
			   &dfw_show_responses,"\
Set whether to show the responses sent by the DFW server.", "\
Show whether to show the responses sent by the DFW server.", "\
If set, the responses sent to the DFW server are printed to stdout.", "\
Whether to show DFW server responses is %s.",
			   NULL, NULL,
			   &setlist, &showlist);
  add_setshow_boolean_cmd ("dfw-warn-unknown-identifiers", no_class,
			   &dfw_warn_unknown_identifiers,"\
Set whether to warn when unknown identifiers are found in DFW responses.", "\
Show whether to warn when unknown identifiers are found in DFW responses.", "\
If set, when unknown ids are found in DFW responses, a warning is sent.", "\
Whether to warn about DFW unknown identifiers is %s.",
			   NULL, NULL,
			   &setlist, &showlist);
  add_info ("vxworks-tasks", info_vxworks_tasks_command,
	    "IDs of currently known VxWorks task on this target.");
  add_info ("dfw-sem-trees", info_dfw_trees, "current DFW trees.");


    add_setshow_uinteger_cmd ("dfw-load-timeout", no_class, &load_timeout,"\
Set timeout in seconds used when loading new objects on DFW target.\n\
The default value is 30 seconds.", "\
Show timeout in seconds used when loading new objects on DFW target.\n\
The default value is 30 seconds.", "\
Timeout in seconds used when loading new objects on DFW target.\n\
The default value is 30 seconds.", "\
DFW load timeout is %s",
			      NULL, NULL,
			      &setlist, &showlist);

    add_cmd ("dfw-send", no_class, remote_dfw_send,
	     "Send the request to the current dfw server.\n", &cmdlist);
}

int
xdr_wtx_core (XDR *xdrs, struct wtx_core *objp)
{
  return xdr_u_int (xdrs, &objp->obj_id)
    && xdr_enum (xdrs, &objp->err_code)
    && xdr_u_int (xdrs, &objp->prot_version);
}

int
xdr_wtx_string (XDR *xdrs, wtx_string *objp)
{
  /* FIXME: the XDR representation of strings in WTX start with a
     constant field, 0x00000001... I don't know why!
     And null string are reprensented by a null integer... I don't know
     why either! */
  int header = 1;
  int null = 0;

  if ( *objp && strcmp (*objp, "") == 0)
    return xdr_int (xdrs, &null);
  else
    return xdr_int (xdrs, &header)
      && xdr_string (xdrs, objp, ~0);
}

int
xdr_wtx_svr_desc (XDR *xdrs, struct wtx_svr_desc *objp)
{
  return xdr_wtx_string (xdrs, &objp->name)
    && xdr_wtx_string (xdrs, &objp->type)
    && xdr_wtx_string (xdrs, &objp->key);
}

bool_t
xdr_wtx_msg_svr_desc(XDR *xdrs, struct wtx_msg_svr_desc *objp)
{
  if (!xdr_wtx_core(xdrs, &objp->core))
    return (FALSE);
  if (!xdr_wtx_svr_desc (xdrs, &objp->desc))
    return (FALSE);
  return (TRUE);
}

/* Send a -wrs-info-retrieve request on the DFW connection stream. The
   command format is:
   "-wrs-info-retrieve -t " TREE_BRANCH TREE_LEAF "default:" PATTERN " -n"
*/

static void
wrs_info_retrieve (const char *tree_branch,
		   const int tree_branch_len,
		   const char *tree_leaf,
		   const int tree_leaf_len,
		   const char *pattern)
{
  static const char command_id[] = "-wrs-info-retrieve -t ";
  static const char url[] = " default:";
  static const char option[] = " -n";
  static int command_id_size = 0;
  static int url_size = 0;
  static int option_size = 0;
  int tree_branch_len0, tree_leaf_len0;

  if (!command_id_size)
    command_id_size = strlen (command_id);
  if (!url_size)
    url_size = strlen (url);
  if (!option_size)
    option_size = strlen (option);
  if (tree_branch)
    {
      if (tree_branch_len)
	tree_branch_len0 = tree_branch_len;
      else
	tree_branch_len0 = strlen (tree_branch);
    }
  if (tree_leaf)
    {
      if (tree_leaf_len)
	tree_leaf_len0 = tree_leaf_len;
      else
	tree_leaf_len0 = strlen (tree_leaf);
    }

  write_token ();
  write_string (command_id, command_id_size);
  if (tree_branch)
    write_string (tree_branch, tree_branch_len0);
  if (tree_leaf)
    write_string (tree_leaf, tree_leaf_len0);
  write_string (url, url_size);
  write_string (pattern, strlen (pattern));
  write_string (option, option_size);
  write_string ("\n", 1);
}

/* Select the thread whose DFW thread id is THREAD_ID.  */

static int
thread_select (int thread_id)
{
  static const char command[] = "-thread-select ";
  static int command_size = 0;
  static char thread_id_string [32];
  struct dfw_sem_element sem_element;
  index_type current_index = 0;

  if (thread_id == current_selected_thread)
    return 1;

  if (!command_size)
    command_size = strlen (command);
  sprintf (thread_id_string, "%d", thread_id);

  write_token ();
  write_string (command, command_size);
  write_string (thread_id_string, strlen (thread_id_string));
  write_string ("\n", 1);

  read_output_record_sequence (remote_timeout, current_token);

  /* Check result.  */
  sem_element = next_sem_element (&current_result_record, &current_index);
  if (sem_element.type != DFW_SEM_RESULT_CLASS
      || sem_element.value.result_class_value != RESULT_CLASS_DONE)
    return 0;
  else
    {
      current_selected_thread = thread_id;
      return 1;
    }
}

static int
thread_select_current_pid ()
{
  int thread_to_select;

  if (!debugged_vxworks_task)
    {
      /* No debugged task.  */
      thread_to_select = system_thread_id;
    }
  else if (debugged_vxworks_task->pid == system_pid)
    {
      /* Kernel task. Debug only this task.  */
      thread_to_select = debugged_vxworks_task->dfw_id;
    }
  else
    {
      /* RTP task. Debug the whole RTP.  */
      thread_to_select = debugged_vxworks_task->rtp_dfw_id;
    }

  thread_select (thread_to_select);
  return thread_to_select;
}

static void
push_interrupt_event ()
{
  index_type index;
  dfw_tree_push (&(current_async_record.dfw_sem_tree),
		 (dfw_tree_element) DFW_SEM_ASYNC_CLASS);
  dfw_tree_push (&(current_async_record.dfw_sem_tree),
		 (dfw_tree_element) ASYNC_CLASS_STOPPED);
  dfw_tree_push (&(current_async_record.dfw_sem_tree),
		 (dfw_tree_element) DFW_SEM_VARIABLE);
  index = string_buffer_push_string (&(current_async_record.string_table),
				     "reason");
  dfw_tree_push (&(current_async_record.dfw_sem_tree),
		 (dfw_tree_element) index);
  dfw_tree_push (&(current_async_record.dfw_sem_tree),
		 (dfw_tree_element) DFW_SEM_VALUE);
  dfw_tree_push (&(current_async_record.dfw_sem_tree),
		 (dfw_tree_element) DFW_SEM_CONST);
  index = string_buffer_push_string (&(current_async_record.string_table),
				     "user-stopped");
}

/* Interrupt the current selected thread.  */

static void
exec_interrupt ()
{
  static const char command[] = "-exec-interrupt\n";
  static int command_size = 0;
  struct dfw_sem_element sem_element;
  int selected_context;
  enum dfw_task_status status;
  index_type current_index = 0;

  if (!command_size)
    command_size = strlen (command);

  selected_context = thread_select_current_pid ();
  status = get_dfw_task_status (selected_context);
  if (status == DFW_TASK_STATUS_STOPPED
      || status == DFW_TASK_STATUS_STOPPED_P
      || status == DFW_TASK_STATUS_STOPPED_T)
    {
      warning ("The program has been interrupted externally");
      push_interrupt_event ();
      return;
    }

  write_token ();
  write_string (command, command_size);
  read_output_record_sequence (remote_timeout, current_token);

  /* Check result.  */
  sem_element = next_sem_element (&current_result_record, &current_index);
  if (sem_element.type != DFW_SEM_RESULT_CLASS
      || sem_element.value.result_class_value != RESULT_CLASS_DONE)
    error ("DFW: -exec-interrupt failed");
}

/* Resume the current selected thread.  */

static void
exec_continue ()
{
  static const char command[] = "-exec-continue\n";
  static int command_size = 0;
  struct dfw_sem_element sem_element;
  index_type current_index = 0;
  enum dfw_task_status status;
  int selected_context;

  if (!command_size)
    command_size = strlen (command);

  selected_context = current_selected_thread;
  status = get_dfw_task_status (selected_context);
  if (status != DFW_TASK_STATUS_STOPPED
      && status != DFW_TASK_STATUS_STOPPED_T
      && status != DFW_TASK_STATUS_STOPPED_P
      && status != DFW_TASK_STATUS_RTP_NORMAL)
    {
      /* FIXME: find a way to test that a RTP has been stopped. */
      warning ("The program has been resumed externally");
      return;
    }

  write_token ();
  write_string (command, command_size);
  read_output_record_sequence (remote_timeout, current_token);

  /* Check result.  */
  sem_element = next_sem_element (&current_result_record, &current_index);
  if (sem_element.type != DFW_SEM_RESULT_CLASS
      || sem_element.value.result_class_value != RESULT_CLASS_RUNNING)
    error ("DFW: -exec-continue failed");
}

/* Step one instruction in the current selected thread.  */

static void
exec_step_instruction ()
{
  static const char command[] = "-exec-step-instruction\n";
  static int command_size = 0;
  struct dfw_sem_element sem_element;
  index_type current_index = 0;

  if (!command_size)
    command_size = strlen (command);

  write_token ();
  write_string (command, command_size);
  read_output_record_sequence (remote_timeout, current_token);

  /* Check result.  */
  sem_element = next_sem_element (&current_result_record, &current_index);
  if (sem_element.type != DFW_SEM_RESULT_CLASS
      || sem_element.value.result_class_value != RESULT_CLASS_RUNNING)
    error ("DFW: -exec-step-instruction failed");
}

/* Return the name of the object whose dfw id is DFW_ID.  */

static char *
get_dfw_object_name (long dfw_id)
{
  static const char thread_info_branch [] = "/IdInfo";
  static int thread_info_branch_len = 0;
  index_type current_index;
  static const char thread_info_desc [] = "getting thread info";
  struct dfw_sem_element sem_element;
  char buffer [128];
  char *name;

  if (!thread_info_branch_len)
    thread_info_branch_len = strlen (thread_info_branch);
  sprintf (buffer, "{Name} -id %ld", dfw_id);

  wrs_info_retrieve (thread_info_branch, thread_info_branch_len,
		     NULL, 0, buffer);
  read_output_record_sequence (remote_timeout, current_token);

  current_index = 0;
  sem_element_process_result_header (&current_result_record,
				     &current_index,
				     thread_info_desc);

  next_sem_element_with_check (&current_result_record, &current_index,
			       thread_info_desc, DFW_SEM_VALUE);
  next_sem_element_with_check (&current_result_record, &current_index,
			       thread_info_desc, DFW_SEM_BEGIN_LIST);
  next_sem_element_with_check (&current_result_record, &current_index,
			       thread_info_desc, DFW_SEM_VARIABLE);
  sem_element_process_const_string_value (&current_result_record,
					  &current_index,
					  thread_info_desc,
					  &name);
  return name;
}

static void
write_string (const char *buf, int cnt)
{
  int i;
  if (serial_write (remote_dfw_desc, buf, cnt))
    error ("write on dfw server failed.");

  if (dfw_show_requests)
    for (i = 0; i < cnt; i++)
      printf_unfiltered ("%c", buf[i]);
}

static token_type
write_token ()
{
  static char token_string [32];

  current_token++;
  sprintf (token_string, "%d", current_token);
  write_string (token_string, strlen (token_string));
  return current_token;
}


static struct vxworks_task *
vxworks_task_build (int rtp_dfw_id, CORE_ADDR pid, int dfw_id, CORE_ADDR tid)
{
  struct vxworks_task *t;
  t = (struct vxworks_task *) xmalloc (sizeof (struct vxworks_task));
  t->rtp_dfw_id = rtp_dfw_id;
  t->pid = pid;
  t->dfw_id = dfw_id;
  t->tid = tid;
  t->next = NULL;
  return t;
}

static struct vxworks_task *
duplicate_vxworks_task (struct vxworks_task *t)
{
  if (t)
    return vxworks_task_build (t->rtp_dfw_id, t->pid, t->dfw_id, t->tid);
  else
    return NULL;
}


static void
add_vxworks_task (struct vxworks_task *task)
{
  task->next = vxworks_task_list;
  vxworks_task_list = task;
}

static struct vxworks_task *
lookup_vxworks_task (CORE_ADDR id)
{
  struct vxworks_task *current;

  for (current = vxworks_task_list; current != NULL; current = current->next)
    {
      if (current->pid == id || current->tid == id)
	{
	  return current;
	}
    }
  return NULL;
}

static void clear_vxworks_task_list ()
{
  struct vxworks_task *current;
  struct cleanup *old_cleanup;

  if (vxworks_task_list == NULL)
    return;
  else
    old_cleanup = make_cleanup (xfree, vxworks_task_list);

  for (current = vxworks_task_list->next;
       current != NULL;
       current = current->next)
    {
      make_cleanup (xfree, current);
    }
  do_cleanups (old_cleanup);
  vxworks_task_list = NULL;
}

static int
is_being_debugged (ptid_t ptid)
{
  int pid = ptid_get_pid (ptid);
  long tid = ptid_get_tid (ptid);

  if (debugged_vxworks_task->pid == system_pid)
    {
      /* We are debugging a kernel task.  */
      return (debugged_vxworks_task->tid == tid);
    }
  else
    {
      /* We are debugging a RTP.  */
      return (debugged_vxworks_task->pid == pid);
    }
}

static int
is_being_debugged_1 (int dfw_id)
{
  if (debugged_vxworks_task->pid == system_pid)
    {
      return debugged_vxworks_task->dfw_id == dfw_id;
    }
  else
    {
      return debugged_vxworks_task->rtp_dfw_id == dfw_id;
    }
}


static ptid_t
vxworks_task_ptid_build (struct vxworks_task *task)
{
  return ptid_build (task->pid, task->dfw_id, task->tid);
}

static ptid_t
get_ptid_from_dfw_id (int dfw_id)
{
  struct vxworks_task *current;

  update_vxworks_task_list ();
  current = lookup_vxworks_task_by_dfw_id (dfw_id);
  if (current)
    return vxworks_task_ptid_build (current);
  else
    return null_ptid;
}

static struct vxworks_task *
lookup_vxworks_task_by_dfw_id (int dfw_id)
{
  struct vxworks_task *current;

  for (current = vxworks_task_list; current != NULL; current = current->next)
    {
      if (current->dfw_id == dfw_id || current->rtp_dfw_id == dfw_id)
	{
	  return current;
	}
    }
  return NULL;
}

static void
add_dfw_breakpoint (CORE_ADDR address, int bp_number)
{
  struct dfw_breakpoint *bp =
    (struct dfw_breakpoint*) xmalloc (sizeof (struct dfw_breakpoint));

  bp->address = address;
  bp->bp_number = bp_number;
  bp->next = dfw_breakpoint_list;
  dfw_breakpoint_list = bp;
}

static int
extract_dfw_breakpoint (CORE_ADDR address)
{
  struct dfw_breakpoint *current;
  int result;

  if (!dfw_breakpoint_list)
    return -1;

  if (dfw_breakpoint_list->address == address)
    {
      result = dfw_breakpoint_list->bp_number;
      dfw_breakpoint_list = dfw_breakpoint_list->next;
      return result;
    }

  for (current = dfw_breakpoint_list; current->next != NULL;
       current = current->next)
    {
      if (current->next->address == address)
	{
	  result = current->next->bp_number;
	  current->next = current->next->next;
	  return result;
	}
    }

  return -1;
}
