------------------------------------------------------------------------------
--  Ada95 Interface to Oracle RDBMS                                         --
--  Copyright (C) 2000-2006 Dmitriy Anisimkov.                              --
--  License agreement and authors contact information are in file oci.ads   --
------------------------------------------------------------------------------

--  $Id: oci-lib.ads,v 1.42 2008/07/03 06:22:48 vagul Exp $

with
   System,
   Interfaces.C.Pointers,
   Interfaces.C.Strings;

package OCI.Lib is

   package C renames Interfaces.C;
   package CStr renames C.Strings;

   subtype Text_Ptr is CStr.chars_ptr;
   subtype Text is C.char_array;
   subtype OraText is Text_Ptr;
   subtype Ub1 is Interfaces.Unsigned_8;
   subtype Sb1 is Interfaces.Integer_8;
   subtype Ub2 is Interfaces.Unsigned_16;
   subtype Sb2 is Interfaces.Integer_16;
   subtype Ub4 is Interfaces.Unsigned_32;
   subtype Sb4 is Interfaces.Integer_32;
   subtype UWord is C.unsigned;
   subtype Bool is C.int;

   subtype DVoid is System.Address;

   type A_Int is access all C.int;
   type A_Sb4 is access all Sb4;
   type A_Sb2 is access all Sb2;
   type A_Ub4 is access all Ub4;
   type A_Ub2 is access all Ub2;
   type A_Ub1 is access all Ub1;

   type OCITypeCode is new Ub2;

   subtype Function_Pointer is DVoid;

   subtype OCIHandle is System.Address;
   Empty_Handle : constant OCIHandle := System.Null_Address;

   type OCIEnv is new OCIHandle;
   type OCIError is new OCIHandle;
   type OCISvcCtx is new OCIHandle;
   type OCIStmt is new OCIHandle;
   type OCIBind is new OCIHandle;
   type OCIDefine is new OCIHandle;
   type OCIDescribe is new OCIHandle;
   type OCIServer is new OCIHandle;
   type OCISession is new OCIHandle;
   type OCIComplexObject is new OCIHandle;
   type OCITrans is new OCIHandle;
   type OCISecurity is new OCIHandle;
   type OCISubscription is new OCIHandle;
   type OCIThreadId is new OCIHandle;
   type OCIThreadHandle is new OCIHandle;
   type OCIThreadMutex is new OCIHandle;

   subtype OCIType is OCIHandle;

   type OCISnapshot is new OCIHandle;
   type OCIResult is new OCIHandle;
   type OCILobLocator is new OCIHandle;
   type OCIParam is new OCIHandle;
   type OCIComplexObjectComp is new OCIHandle;
   type OCIRowid is new OCIHandle;
   type OCIDateTime is new OCIHandle;
   type OCIInterval is new OCIHandle;
   type OCIColl is new OCIHandle;

   type OCIExtProcContext is new OCIHandle;

   type OCIAQEnqOptions is new OCIHandle;
   type OCIAQDeqOptions is new OCIHandle;
   type OCIAQMsgProperties is new OCIHandle;
   type OCIAQAgent is new OCIHandle;
   type OCIAQNfyDescriptor is new OCIHandle;

   subtype SWord is C.int;
   subtype EWord is C.int;

   type OCITypeGetOpt is
     (OCI_TYPEGET_HEADER, OCI_TYPEGET_ALL);
   pragma Convention (C, OCITypeGetOpt);

   type Vector_Of_Ub1 is array (Integer range <>) of aliased Ub1;
   type Vector_Of_Ub4 is array (Integer range <>) of aliased Ub4;
   type Vector_Of_OraText is array (Integer range <>) of aliased OraText;
   type Vector_Of_OCIBind is array (Integer range <>) of aliased OCIBind;

   package Ub1_Ptr is new Interfaces.C.Pointers
      (Integer, Ub1, Vector_Of_Ub1, 0);

   subtype A_Vector_Of_Ub1 is Ub1_Ptr.Pointer;

   package OraText_Ptr is new Interfaces.C.Pointers
      (Integer, OraText, Vector_Of_OraText, CStr.Null_Ptr);

   subtype A_Vector_Of_OraText is OraText_Ptr.Pointer;

   package OCIBind_Ptr is new Interfaces.C.Pointers
      (Integer, OCIBind, Vector_Of_OCIBind, OCIBind (Empty_Handle));

   subtype A_Vector_Of_OCIBind is OCIBind_Ptr.Pointer;

   subtype Undefined is System.Address;
   Empty_Undefined : constant Undefined := System.Null_Address;

   OCI_SUCCESS           : constant := 0;
   OCI_SUCCESS_WITH_INFO : constant := 1;
   OCI_NO_DATA           : constant := 100;
   OCI_ERROR             : constant := -1;
   OCI_INVALID_HANDLE    : constant := -2;
   OCI_NEED_DATA         : constant := 99;
   OCI_STILL_EXECUTING   : constant := -3123;
   OCI_CONTINUE          : constant := -24200;

   OCI_FETCH_NEXT       : constant := 2;
   OCI_FETCH_FIRST      : constant := 16#04#;
   OCI_FETCH_LAST       : constant := 16#08#;
   OCI_FETCH_PRIOR      : constant := 16#10#;
   OCI_FETCH_ABSOLUTE   : constant := 16#20#;
   OCI_FETCH_RELATIVE   : constant := 16#40#;
   OCI_FETCH_RESERVED_1 : constant := 16#80#;

   OCI_V7_SYNTAX  : constant := 2;
   OCI_V8_SYNTAX  : constant := 3;
   OCI_NTV_SYNTAX : constant := 1;

   OCI_OTYPE_NAME : constant := 1;
   OCI_OTYPE_REF  : constant := 2;
   OCI_OTYPE_PTR  : constant := 3;

   OCI_PTYPE_UNK         : constant := 0;
   OCI_PTYPE_TABLE       : constant := 1;
   OCI_PTYPE_VIEW        : constant := 2;
   OCI_PTYPE_PROC        : constant := 3;
   OCI_PTYPE_FUNC        : constant := 4;
   OCI_PTYPE_PKG         : constant := 5;
   OCI_PTYPE_TYPE        : constant := 6;
   OCI_PTYPE_SYN         : constant := 7;
   OCI_PTYPE_SEQ         : constant := 8;
   OCI_PTYPE_COL         : constant := 9;
   OCI_PTYPE_ARG         : constant := 10;
   OCI_PTYPE_LIST        : constant := 11;
   OCI_PTYPE_TYPE_ATTR   : constant := 12;
   OCI_PTYPE_TYPE_COLL   : constant := 13;
   OCI_PTYPE_TYPE_METHOD : constant := 14;
   OCI_PTYPE_TYPE_ARG    : constant := 15;
   OCI_PTYPE_TYPE_RESULT : constant := 16;
   OCI_PTYPE_SCHEMA      : constant := 17;
   OCI_PTYPE_DATABASE    : constant := 18;

   OCI_LTYPE_UNK           : constant := 0;
   OCI_LTYPE_COLUMN        : constant := 1;
   OCI_LTYPE_ARG_PROC      : constant := 2;
   OCI_LTYPE_ARG_FUNC      : constant := 3;
   OCI_LTYPE_SUBPRG        : constant := 4;
   OCI_LTYPE_TYPE_ATTR     : constant := 5;
   OCI_LTYPE_TYPE_METHOD   : constant := 6;
   OCI_LTYPE_TYPE_ARG_PROC : constant := 7;
   OCI_LTYPE_TYPE_ARG_FUNC : constant := 8;
   OCI_LTYPE_SCH_OBJ       : constant := 9;
   OCI_LTYPE_DB_SCH        : constant := 10;

   OCI_MEMORY_CLEARED : constant := 1;

   OCI_ATTR_FNCODE                  : constant := 1;
   OCI_ATTR_OBJECT                  : constant := 2;
   OCI_ATTR_NONBLOCKING_MODE        : constant := 3;
   OCI_ATTR_SQLCODE                 : constant := 4;
   OCI_ATTR_ENV                     : constant := 5;
   OCI_ATTR_SERVER                  : constant := 6;
   OCI_ATTR_SESSION                 : constant := 7;
   OCI_ATTR_TRANS                   : constant := 8;
   OCI_ATTR_ROW_COUNT               : constant := 9;
   OCI_ATTR_SQLFNCODE               : constant := 10;
   OCI_ATTR_PREFETCH_ROWS           : constant := 11;
   OCI_ATTR_NESTED_PREFETCH_ROWS    : constant := 12;
   OCI_ATTR_PREFETCH_MEMORY         : constant := 13;
   OCI_ATTR_NESTED_PREFETCH_MEMORY  : constant := 14;
   OCI_ATTR_CHAR_COUNT              : constant := 15;
   OCI_ATTR_PDSCL                   : constant := 16;
   OCI_ATTR_PDFMT                   : constant := 17;
   OCI_ATTR_PARAM_COUNT             : constant := 18;
   OCI_ATTR_ROWID                   : constant := 19;
   OCI_ATTR_CHARSET                 : constant := 20;
   OCI_ATTR_NCHAR                   : constant := 21;
   OCI_ATTR_USERNAME                : constant := 22;
   OCI_ATTR_PASSWORD                : constant := 23;
   OCI_ATTR_STMT_TYPE               : constant := 24;
   OCI_ATTR_INTERNAL_NAME           : constant := 25;
   OCI_ATTR_EXTERNAL_NAME           : constant := 26;
   OCI_ATTR_XID                     : constant := 27;
   OCI_ATTR_TRANS_LOCK              : constant := 28;
   OCI_ATTR_TRANS_NAME              : constant := 29;
   OCI_ATTR_HEAPALLOC               : constant := 30;
   OCI_ATTR_CHARSET_ID              : constant := 31;
   OCI_ATTR_CHARSET_FORM            : constant := 32;
   OCI_ATTR_MAXDATA_SIZE            : constant := 33;
   OCI_ATTR_CACHE_OPT_SIZE          : constant := 34;
   OCI_ATTR_CACHE_MAX_SIZE          : constant := 35;
   OCI_ATTR_PINOPTION               : constant := 36;
   OCI_ATTR_ALLOC_DURATION          : constant := 37;
   OCI_ATTR_PIN_DURATION            : constant := 38;
   OCI_ATTR_FDO                     : constant := 39;
   OCI_ATTR_POSTPROCESSING_CALLBACK : constant := 40;
   OCI_ATTR_POSTPROCESSING_CONTEXT  : constant := 41;
   OCI_ATTR_ROWS_RETURNED           : constant := 42;
   OCI_ATTR_FOCBK                   : constant := 43;
   OCI_ATTR_IN_V8_MODE              : constant := 44;
   OCI_ATTR_LOBEMPTY                : constant := 45;
   OCI_ATTR_SESSLANG                : constant := 46;

   OCI_ATTR_NUM_DML_ERRORS : constant := 73;
   OCI_ATTR_DML_ROW_OFFSET : constant := 74;

   OCI_ATTR_PARSE_ERROR_OFFSET  : constant := 129;

   OCI_ATTR_DATA_SIZE : constant := 1;
   OCI_ATTR_DATA_TYPE : constant := 2;
   OCI_ATTR_DISP_SIZE : constant := 3;
   OCI_ATTR_NAME      : constant := 4;
   OCI_ATTR_PRECISION : constant := 5;
   OCI_ATTR_SCALE     : constant := 6;
   OCI_ATTR_IS_NULL   : constant := 7;
   OCI_ATTR_TYPE_NAME : constant := 8;

   OCI_ATTR_SCHEMA_NAME : constant := 9;
   OCI_ATTR_SUB_NAME    : constant := 10;
   OCI_ATTR_POSITION    : constant := 11;

   --  complex object retrieval parameter attributes
   OCI_ATTR_COMPLEXOBJECTCOMP_TYPE       : constant := 50;
   OCI_ATTR_COMPLEXOBJECTCOMP_TYPE_LEVEL : constant := 51;
   OCI_ATTR_COMPLEXOBJECT_LEVEL          : constant := 52;
   OCI_ATTR_COMPLEXOBJECT_COLL_OUTOFLINE : constant := 53;

   OCI_ATTR_NOCACHE     : constant := 87; -- Temporary LOBs
   --  Strange. I think it is wrong attribute description -^.

   OCI_ATTR_SUBSCR_NAME      : constant := 94;  -- name of subscription
   OCI_ATTR_SUBSCR_CALLBACK  : constant := 95;  -- associated callback
   OCI_ATTR_SUBSCR_CTX       : constant := 96;  -- associated callback context
   OCI_ATTR_SUBSCR_PAYLOAD   : constant := 97;  -- associated payload
   OCI_ATTR_SUBSCR_NAMESPACE : constant := 98;  -- associated namespace

   OCI_ATTR_CONSUMER_NAME : constant := 50; -- consumer name
   OCI_ATTR_DEQ_MODE      : constant := 51; -- dequeue mode
   OCI_ATTR_NAVIGATION    : constant := 52; -- navigation
   OCI_ATTR_WAIT                : constant := 53; -- wait
   OCI_ATTR_DEQ_MSGID     : constant := 54; -- dequeue message id

   OCI_ATTR_QUEUE_NAME : constant := 70;  --  queue name
   OCI_ATTR_NFY_MSGID  : constant := 71;  --  message id
   OCI_ATTR_MSG_PROP   : constant := 72;        --  message properties

   --  only named type attributes
   OCI_ATTR_TYPECODE            : constant := 216; -- object or collection
   OCI_ATTR_COLLECTION_TYPECODE : constant := 217; -- varray or nested table
   OCI_ATTR_VERSION             : constant := 218; -- user assigned version
   OCI_ATTR_IS_INCOMPLETE_TYPE  : constant := 219;
   OCI_ATTR_IS_SYSTEM_TYPE      : constant := 220; -- a system type
   OCI_ATTR_IS_PREDEFINED_TYPE  : constant := 221; -- a predefined type
   OCI_ATTR_IS_TRANSIENT_TYPE   : constant := 222; -- a transient type
   OCI_ATTR_IS_SYSTEM_GENERATED_TYPE : constant := 223;
   OCI_ATTR_HAS_NESTED_TABLE    : constant := 224;
   OCI_ATTR_HAS_LOB             : constant := 225; -- has a lob attribute
   OCI_ATTR_HAS_FILE            : constant := 226; -- has a file attribute
   OCI_ATTR_COLLECTION_ELEMENT  : constant := 227;
   OCI_ATTR_NUM_TYPE_ATTRS      : constant := 228; -- number of attribute types
   OCI_ATTR_LIST_TYPE_ATTRS     : constant := 229; -- list of type attributes
   OCI_ATTR_NUM_TYPE_METHODS    : constant := 230; -- number of type methods
   OCI_ATTR_LIST_TYPE_METHODS   : constant := 231; -- list of type methods
   OCI_ATTR_MAP_METHOD          : constant := 232; -- map method of type
   OCI_ATTR_ORDER_METHOD        : constant := 233; -- order method of type

   OCI_ATTR_CHNF_TABLENAMES     : constant := 401; -- array of table names
   OCI_ATTR_CHNF_ROWIDS         : constant := 402; -- rowids needed
   OCI_ATTR_CHNF_OPERATIONS     : constant := 403; -- notif. operation filter
   OCI_ATTR_CHNF_CHANGELAG      : constant := 404; -- transaction lag
   OCI_ATTR_CHDES_DBNAME        : constant := 405; -- source database
   OCI_ATTR_CHDES_NFYTYPE       : constant := 406; -- notification type flags
   OCI_ATTR_CHDES_XID           : constant := 407; -- XID of the transaction
   OCI_ATTR_CHDES_TABLE_CHANGES : constant := 408; -- array of table chg descr
   OCI_ATTR_CHDES_TABLE_NAME    : constant := 409; -- table name
   OCI_ATTR_CHDES_TABLE_OPFLAGS : constant := 410; -- table operation flags
   OCI_ATTR_CHDES_TABLE_ROW_CHANGES : constant := 411; -- changed rows
   OCI_ATTR_CHDES_ROW_ROWID     : constant := 412; -- rowid of changed row
   OCI_ATTR_CHDES_ROW_OPFLAGS   : constant := 413; -- row operation flags

   OCI_ATTR_CHNF_REGHANDLE      : constant := 414; -- subscription handle

   OCI_DTYPE_FIRST             : constant := 50;
   OCI_DTYPE_LOB               : constant := 50;
   OCI_DTYPE_SNAP              : constant := 51;
   OCI_DTYPE_RSET              : constant := 52;
   OCI_DTYPE_PARAM             : constant := 53;
   OCI_DTYPE_ROWID             : constant := 54;
   OCI_DTYPE_COMPLEXOBJECTCOMP : constant := 55;
   OCI_DTYPE_FILE              : constant := 56;
   OCI_DTYPE_AQENQ_OPTIONS     : constant := 57;
   OCI_DTYPE_AQDEQ_OPTIONS     : constant := 58;
   OCI_DTYPE_AQMSG_PROPERTIES  : constant := 59;
   OCI_DTYPE_AQAGENT           : constant := 60;
   OCI_DTYPE_LOCATOR           : constant := 61;
   OCI_DTYPE_DATETIME          : constant := 62;
   OCI_DTYPE_INTERVAL          : constant := 63;
   OCI_DTYPE_AQNFY_DESCRIPTOR  : constant := 64;
   OCI_DTYPE_CHDES             : constant := 77;
   OCI_DTYPE_TABLE_CHDES       : constant := 78;
   OCI_DTYPE_ROW_CHDES         : constant := 79;
   OCI_DTYPE_LAST              : constant := 79;

   OCI_BATCH_MODE             : constant := 16#01#;
   OCI_EXACT_FETCH            : constant := 16#02#;
   OCI_KEEP_FETCH_STATE       : constant := 16#04#;
   OCI_SCROLLABLE_CURSOR      : constant := 16#08#;
   OCI_DESCRIBE_ONLY          : constant := 16#10#;
   OCI_COMMIT_ON_SUCCESS      : constant := 16#20#;
   OCI_NON_BLOCKING           : constant := 16#40#;
   OCI_BATCH_ERRORS           : constant := 16#80#;
   OCI_PARSE_ONLY             : constant := 16#100#;
   OCI_EXACT_FETCH_RESERVED_1 : constant := 16#200#;

   ------------------------- Supported Namespaces ---------------------------
   OCI_SUBSCR_NAMESPACE_ANONYMOUS : constant := 0; -- Anonymous Namespace
   OCI_SUBSCR_NAMESPACE_AQ        : constant := 1; -- Advanced Queues
   OCI_SUBSCR_NAMESPACE_DBCHANGE  : constant := 2; -- DB change notification
   OCI_SUBSCR_NAMESPACE_MAX       : constant := 3; -- Max Name Space Number

   --  input data types
   SQLT_CHR      : constant := 1;   -- (ORANET TYPE) character string
   SQLT_NUM      : constant := 2;   -- (ORANET TYPE) oracle numeric
   SQLT_INT      : constant := 3;   -- (ORANET TYPE) integer
   SQLT_FLT      : constant := 4;   -- (ORANET TYPE) Floating point number
   SQLT_STR      : constant := 5;   -- zero terminated string
   SQLT_VNU      : constant := 6;   -- NUM with preceding length byte
   SQLT_PDN      : constant := 7;   -- (ORANET TYPE) Packed Decimal Numeric
   SQLT_LNG      : constant := 8;   -- long
   SQLT_VCS      : constant := 9;   -- Variable character string
   SQLT_NON      : constant := 10;  -- Null/empty PCC Descriptor entry
   SQLT_RID      : constant := 11;  -- rowid
   SQLT_DAT      : constant := 12;  -- date in oracle format
   SQLT_VBI      : constant := 15;  -- binary in VCS format
   SQLT_BFLOAT   : constant := 21;  -- Native binary float
   SQLT_BDOUBLE  : constant := 22;  -- Native binary double
   SQLT_BIN      : constant := 23;  -- binary data (DTYBIN)
   SQLT_LBI      : constant := 24;  -- long binary
   SQLT_UIN      : constant := 68;  -- unsigned integer
   SQLT_SLS      : constant := 91;  -- Display sign leading separate
   SQLT_LVC      : constant := 94;  -- Longer longs (char)
   SQLT_LVB      : constant := 95;  -- Longer long binary
   SQLT_AFC      : constant := 96;  -- Ansi fixed char
   SQLT_AVC      : constant := 97;  -- Ansi Var char
   SQLT_IBFLOAT  : constant := 100; -- binary float canonical
   SQLT_IBDOUBLE : constant := 101; -- binary double canonical
   SQLT_CUR      : constant := 102; -- cursor  type
   SQLT_RDD      : constant := 104; -- rowid descriptor
   SQLT_LAB      : constant := 105; -- label type
   SQLT_OSL      : constant := 106; -- oslabel type

   SQLT_NTY    : constant := 108; -- named object type
   SQLT_REF    : constant := 110; -- ref type
   SQLT_CLOB   : constant := 112; -- character lob
   SQLT_BLOB   : constant := 113; -- binary lob
   SQLT_BFILEE : constant := 114; -- binary file lob
   SQLT_CFILEE : constant := 115; -- character file lob
   SQLT_RSET   : constant := 116; -- result set type
   SQLT_NCO    : constant := 122; -- named collection (varray or nested table)
   SQLT_VST    : constant := 155; -- OCIString type
   SQLT_ODT    : constant := 156; -- OCIDate type

   --  datetimes and intervals
   SQLT_DATE          : constant := 184; -- ANSI Date
   SQLT_TIME          : constant := 185; -- TIME
   SQLT_TIME_TZ       : constant := 186; -- TIME WITH TIME ZONE
   SQLT_TIMESTAMP     : constant := 187; -- TIMESTAMP
   SQLT_TIMESTAMP_TZ  : constant := 188; -- TIMESTAMP WITH TIME ZONE
   SQLT_INTERVAL_YM   : constant := 189; -- INTERVAL YEAR TO MONTH
   SQLT_INTERVAL_DS   : constant := 190; -- INTERVAL DAY TO SECOND
   SQLT_TIMESTAMP_LTZ : constant := 232; -- TIMESTAMP WITH LOCAL TZ

   SQLT_PNTY : constant := 241; -- pl/sql representation of named types

   SQLT_FILE  : constant := SQLT_BFILEE;
   SQLT_CFILE : constant := SQLT_CFILEE;
   SQLT_BFILE : constant := SQLT_BFILEE;

   OCI_TYPECODE_REF             : constant := SQLT_REF;
   OCI_TYPECODE_DATE            : constant := SQLT_DAT;
   OCI_TYPECODE_SIGNED8         : constant := 27;
   OCI_TYPECODE_SIGNED16        : constant := 28;
   OCI_TYPECODE_SIGNED32        : constant := 29;
   OCI_TYPECODE_REAL            : constant := 21;
   OCI_TYPECODE_DOUBLE          : constant := 22;
   OCI_TYPECODE_FLOAT           : constant := SQLT_FLT;
   OCI_TYPECODE_NUMBER          : constant := SQLT_NUM;
   OCI_TYPECODE_DECIMAL         : constant := SQLT_PDN;
   OCI_TYPECODE_UNSIGNED8       : constant := SQLT_BIN;
   OCI_TYPECODE_UNSIGNED16      : constant := 25;
   OCI_TYPECODE_UNSIGNED32      : constant := 26;
   OCI_TYPECODE_OCTET           : constant := 245;
   OCI_TYPECODE_SMALLINT        : constant := 246;
   OCI_TYPECODE_INTEGER         : constant := SQLT_INT;
   OCI_TYPECODE_RAW             : constant := SQLT_LVB;
   OCI_TYPECODE_PTR             : constant := 32;
   OCI_TYPECODE_VARCHAR2        : constant := SQLT_VCS;
   OCI_TYPECODE_CHAR            : constant := SQLT_AFC;
   OCI_TYPECODE_VARCHAR         : constant := SQLT_CHR;
   OCI_TYPECODE_MLSLABEL        : constant := SQLT_LAB;
   OCI_TYPECODE_VARRAY          : constant := 247;
   OCI_TYPECODE_TABLE           : constant := 248;
   OCI_TYPECODE_OBJECT          : constant := SQLT_NTY;
   OCI_TYPECODE_OPAQUE          : constant := 58;
   OCI_TYPECODE_NAMEDCOLLECTION : constant := SQLT_NCO;
   OCI_TYPECODE_BLOB            : constant := SQLT_BLOB;
   OCI_TYPECODE_BFILE           : constant := SQLT_BFILE;
   OCI_TYPECODE_CLOB            : constant := SQLT_CLOB;
   OCI_TYPECODE_CFILE           : constant := SQLT_CFILE;

   --  the following are ANSI datetime datatypes added in 8.1
   OCI_TYPECODE_TIME         : constant := SQLT_TIME;
   OCI_TYPECODE_TIME_TZ      : constant := SQLT_TIME_TZ;
   OCI_TYPECODE_TIMESTAMP    : constant := SQLT_TIMESTAMP;
   OCI_TYPECODE_TIMESTAMP_TZ : constant := SQLT_TIMESTAMP_TZ;

   OCI_TYPECODE_TIMESTAMP_LTZ : constant := SQLT_TIMESTAMP_LTZ;

   OCI_TYPECODE_INTERVAL_YM : constant := SQLT_INTERVAL_YM;
   OCI_TYPECODE_INTERVAL_DS : constant := SQLT_INTERVAL_DS;
   OCI_TYPECODE_UROWID      : constant := SQLT_RDD;

   OCI_TYPECODE_OTMFIRST : constant := 228;
   OCI_TYPECODE_OTMLAST  : constant := 320;
   OCI_TYPECODE_SYSFIRST : constant := 228;
   OCI_TYPECODE_SYSLAST  : constant := 235;

   OCI_TYPECODE_PLS_INTEGER : constant := 266;

   --  NOTE : The following NCHAR related codes are just short forms for saying
   --  OCI_TYPECODE_VARCHAR2 with a charset form of SQLCS_NCHAR. These codes
   --  are intended for use in the OCIAnyData API only and nowhere else.
   OCI_TYPECODE_NCHAR     : constant := 286;
   OCI_TYPECODE_NVARCHAR2 : constant := 287;
   OCI_TYPECODE_NCLOB     : constant := 288;

   --  To indicate absence of typecode being specified
   OCI_TYPECODE_NONE : constant := 0;

   --  To indicate error has to be taken from error handle - reserved for
   --  sqlplus use
   OCI_TYPECODE_ERRHP : constant := 283;

   OCI_DEFAULT : constant := 0;

   OCI_THREADED : constant := 1;
   OCI_OBJECT   : constant := 2;
   OCI_EVENTS   : constant := 4;
   OCI_SHARED   : constant := 16#10#;

   OCI_NO_UCB   : constant := 16#40#;
   OCI_NO_MUTEX : constant := 16#80#;

   OCI_ENV_NO_UCB   : constant := 1;
   OCI_ENV_NO_MUTEX : constant := 8;

   OCI_HTYPE_FIRST                : constant := 1;
   OCI_HTYPE_ENV                  : constant := 1;
   OCI_HTYPE_ERROR                : constant := 2;
   OCI_HTYPE_SVCCTX               : constant := 3;
   OCI_HTYPE_STMT                 : constant := 4;
   OCI_HTYPE_BIND                 : constant := 5;
   OCI_HTYPE_DEFINE               : constant := 6;
   OCI_HTYPE_DESCRIBE             : constant := 7;
   OCI_HTYPE_SERVER               : constant := 8;
   OCI_HTYPE_SESSION              : constant := 9;
   OCI_HTYPE_TRANS                : constant := 10;
   OCI_HTYPE_COMPLEXOBJECT        : constant := 11;
   OCI_HTYPE_SECURITY             : constant := 12;
   OCI_HTYPE_SUBSCRIPTION         : constant := 13;
   OCI_HTYPE_DIRPATH_CTX          : constant := 14;
   OCI_HTYPE_DIRPATH_COLUMN_ARRAY : constant := 15;
   OCI_HTYPE_DIRPATH_STREAM       : constant := 16;
   OCI_HTYPE_PROC                 : constant := 17;
   OCI_HTYPE_LAST                 : constant := 17;

   OCI_ONE_PIECE   : constant := 0;
   OCI_FIRST_PIECE : constant := 1;
   OCI_NEXT_PIECE  : constant := 2;
   OCI_LAST_PIECE  : constant := 3;

   OCI_PARAM_IN  : constant := 1; -- in parameter
   OCI_PARAM_OUT : constant := 2; -- out parameter

   OCI_DATA_AT_EXEC  : constant := 2;
   OCI_DYNAMIC_FETCH : constant := 2;
   OCI_PIECEWISE     : constant := 4;

   OCI_TEMP_BLOB  : constant := 1;
   OCI_TEMP_CLOB  : constant := 2;
   OCI_TEMP_NCLOB : constant := 3;

   --  CHAR/NCHAR/VARCHAR2/NVARCHAR2/CLOB/NCLOB char set "form" information
   SQLCS_IMPLICIT : constant := 1; -- CHAR, VARCHAR2, CLOB w/o a specified set
   SQLCS_NCHAR    : constant := 2; -- NCHAR, NCHAR VARYING, NCLOB
   SQLCS_EXPLICIT : constant := 3; -- CHAR, etc, with "CHARACTER SET..." syntax
   SQLCS_FLEXIBLE : constant := 4; -- PL/SQL "flexible" parameters
   SQLCS_LIT_NULL : constant := 5; -- typecheck of NULL and empty_clob() lits

   -------------------------Credential Types-------------------------
   OCI_CRED_RDBMS   : constant := 1;  -- database username/password
   OCI_CRED_EXT     : constant := 2;  -- externally provided credentials
   OCI_CRED_PROXY   : constant := 3;  -- proxy authentication
   ------------------------------------------------------------------

   --  DB Change: Operation types  --
   OCI_OPCODE_ALLROWS : constant := 16#01#;
   OCI_OPCODE_INSERT  : constant := 16#02#;
   OCI_OPCODE_UPDATE  : constant := 16#04#;
   OCI_OPCODE_DELETE  : constant := 16#08#;
   OCI_OPCODE_ALTER   : constant := 16#10#;
   OCI_OPCODE_DROP    : constant := 16#20#;
   OCI_OPCODE_UNKNOWN : constant := 16#40#;

   OCI_NUMBER_UNSIGNED : constant := 0; -- Unsigned type - ubX
   OCI_NUMBER_SIGNED   : constant := 2; -- Signed type - sbX

   TRUE  : constant := 1;
   FALSE : constant := 0;

   subtype C_Boolean is C.int;

   Null_Indicator : constant := -1;
   Not_Null_Indicator : constant := 0;

   OCI_STMT_SELECT  : constant := 1;
   OCI_STMT_UPDATE  : constant := 2;
   OCI_STMT_DELETE  : constant := 3;
   OCI_STMT_INSERT  : constant := 4;
   OCI_STMT_CREATE  : constant := 5;
   OCI_STMT_DROP    : constant := 6;
   OCI_STMT_ALTER   : constant := 7;
   OCI_STMT_BEGIN   : constant := 8;
   OCI_STMT_DECLARE : constant := 9;

   type OCIDuration is new Ub2;

   OCI_DURATION_BEGIN   : constant OCIDuration := 10;
   OCI_DURATION_NULL    : constant OCIDuration := OCI_DURATION_BEGIN - 1;
   OCI_DURATION_DEFAULT : constant OCIDuration := OCI_DURATION_BEGIN - 2;
   OCI_DURATION_NEXT    : constant OCIDuration := OCI_DURATION_BEGIN - 3;
   OCI_DURATION_SESSION : constant OCIDuration := OCI_DURATION_BEGIN;
   OCI_DURATION_TRANS   : constant OCIDuration := OCI_DURATION_BEGIN + 1;

   OCI_UCBTYPE_ENTRY   : constant := 1; --  entry callback
   OCI_UCBTYPE_EXIT    : constant := 2; --  exit callback
   OCI_UCBTYPE_REPLACE : constant := 3; --  replacement callback

   type OCITime is record
      OCITimeHH : Ub1;
      OCITimeMI : Ub1;
      OCITimeSS : Ub1;
   end record;
   pragma Convention (C, OCITime);

   type OCIDate is record
      OCIDateYYYY : Sb2;
      OCIDateMM   : Ub1;
      OCIDateDD   : Ub1;
      OCIDateTime : OCITime;
   end record;
   pragma Convention (C, OCIDate);

   OCI_NUMBER_SIZE : constant := 22;

   type OCINumber is array (1 .. OCI_NUMBER_SIZE) of Ub1;

   pragma Convention (C, OCINumber);

   type OCIString is new DVoid;

   type A_OCILobLocator is access all OCILobLocator;

   subtype OCICLOBLOCATOR is OCILobLocator;

   subtype OCIBLOBLOCATOR is OCILobLocator;

   subtype OCIBFILELOCATOR is OCILobLocator;

   function OCIEnvCreate
     (Envh      : access OCIEnv;
      Mode      : in     Ub4;
      Ctxp      : in     DVoid     := Empty_Handle;
      Malocfp   : in     DVoid     := Empty_Handle;
      Ralocfp   : in     DVoid     := Empty_Handle;
      Mfreefp   : in     DVoid     := Empty_Handle;
      Xtramemsz : in     C.size_t  := 0;
      Usrmempp  : in     Undefined := Empty_Undefined) return SWord;

   function OCIErrorGet
     (Hndlp    : in     OCIHandle;
      Recordno : in     Ub4       := 1;
      Sqlstate : in     Undefined := Empty_Undefined;
      Errcodep : access Sb4;
      Bufp     : in     Text_Ptr;
      Bufsiz   : in     Ub4;
      Htype    : in     Ub4) return SWord;

   function OCIErrorGet
     (Hndlp    : in     OCIError;
      Recordno : in     Ub4       := 1;
      Sqlstate : in     Undefined := Empty_Undefined;
      Errcodep : access Sb4;
      Bufp     : in     Text_Ptr;
      Bufsiz   : in     Ub4;
      Htype    : in     Ub4) return SWord;

   function OCILogon
     (Envhp      : in     OCIEnv;
      Errhp      : in     OCIError;
      Svchp      : access OCISvcCtx;
      Username   : in     Text;
      Uname_Len  : in     Ub4;
      Password   : in     Text;
      Passwd_Len : in     Ub4;
      Dbname     : in     Text;
      Dbname_Len : in     Ub4) return SWord;

   function OCILogoff
     (Svchp : in OCISvcCtx; Errhp : in OCIError) return SWord;

   function OCIInitialize
     (Mode    : in Ub4;
      Ctxp    : in Undefined := Empty_Undefined;
      Malocfp : in Undefined := Empty_Undefined;
      Ralocfp : in Undefined := Empty_Undefined;
      Mfreefp : in Undefined := Empty_Undefined) return SWord;

   function OCIEnvInit
     (Envh      : access OCIEnv;
      Mode      : in     Ub4;
      Xtramemsz : in     C.size_t := 0;
      Usrmempp  : in     Undefined := Empty_Undefined) return SWord;

   function OCIHandleAlloc
     (Parenth    : in     OCIHandle;
      Hndlpp     : access OCIHandle;
      Htype      : in     Ub4;
      Xtramem_Sz : in     C.size_t  := 0;
      Usrmempp   : in     Undefined := Empty_Undefined) return SWord;

   function OCIHandleFree (Hndlp : OCIHandle; Htype : Ub4) return SWord;

   function OCIStmtPrepare
     (Stmtp    : in OCIStmt;
      Errhp    : in OCIError;
      Stmt     : in Text;
      Stmt_Len : in Ub4;
      Language : in Ub4  := OCI_NTV_SYNTAX;
      Mode     : in Ub4  := OCI_DEFAULT) return SWord;

   function OCIStmtExecute
     (Svchp    : in OCISvcCtx;
      Stmtp    : in OCIStmt;
      Errhp    : in OCIError;
      Iters    : in Ub4;
      Rowoff   : in Ub4       := 0;
      Snap_In  : in OCIHandle := Empty_Handle;
      Snap_Out : in OCIHandle := Empty_Handle;
      Mode     : in Ub4       := OCI_DEFAULT) return SWord;

   function OCIParamGet
     (Hndlp   : in     OCIHandle;
      Htype   : in     Ub4;
      Errhp   : in     OCIError;
      Parmdpp : access OCIParam;
      Pos     : in     Ub4) return SWord;

   function OCIAttrGet
     (Trgthndlp  : in OCIHandle;
      Trghndltyp : in Ub4;
      Attributep : in DVoid;
      Sizep      : in A_Ub4;
      Attrtype   : in Ub4;
      Errhp      : in OCIError) return SWord;

   function OCIAttrGet
     (Trgthndlp  : in     OCIHandle;
      Trghndltyp : in     Ub4;
      Attributep : access CStr.chars_ptr;
      Sizep      : access Ub4;
      Attrtype   : in     Ub4;
      Errhp      : in     OCIError) return SWord;

   function OCIAttrSet
     (Trgthndlp  : in OCIHandle;
      Trghndltyp : in Ub4;
      Attributep : in Text;
      Size       : in Ub4;
      Attrtype   : in Ub4;
      Errhp      : in OCIError) return SWord;

   function OCIAttrSet
     (Trgthndlp  : in OCIHandle;
      Trghndltyp : in Ub4;
      Attributep : in DVoid;
      Size       : in Ub4;
      Attrtype   : in Ub4;
      Errhp      : in OCIError) return SWord;

   function OCIStmtFetch
     (Stmtp       : in OCIStmt;
      Errhp       : in OCIError;
      Nrows       : in Ub4      := 1;
      Orientation : in Ub2      := OCI_FETCH_NEXT;
      Mode        : in Ub4      := OCI_DEFAULT) return SWord;

   function OCIBindByName
     (Stmtp       : in     OCIStmt;
      Bindpp      : access OCIBind;
      Errhp       : in     OCIError;
      Placeholder : in     Text;
      Placeh_Len  : in     Sb4;
      Valuep      : in     DVoid;
      Value_Sz    : in     Sb4;
      Dty         : in     Ub2;
      Indp        : in     A_Sb2;
      Alenp       : in     A_Ub2  := null;
      Rcodep      : in     A_Ub2  := null;
      Maxarr_Len  : in     Ub4    := 0;
      Curelep     : in     A_Ub4  := null;
      Mode        : in     Ub4    := OCI_DEFAULT) return SWord;

   function OCIBindByName
     (Stmtp       : in     OCIStmt;
      Bindpp      : access OCIBind;
      Errhp       : in     OCIError;
      Placeholder : in     Text;
      Placeh_Len  : in     Sb4;
      Valuep      : in     Text_Ptr;
      Value_Sz    : in     Sb4;
      Dty         : in     Ub2;
      Indp        : access Sb2;
      Alenp       : in     A_Ub2    := null;
      Rcodep      : in     A_Ub2    := null;
      Maxarr_Len  : in     Ub4      := 0;
      Curelep     : in     A_Ub4    := null;
      Mode        : in     Ub4      := OCI_DEFAULT) return SWord;

   function OCIBindByPos
     (Stmtp      : in     OCIStmt;
      Bindpp     : access OCIBind;
      Errhp      : in     OCIError;
      Position   : in     Ub4;
      Valuep     : in     DVoid;
      Value_Sz   : in     Sb4;
      Dty        : in     Ub2;
      Indp       : in     A_Sb2;
      Alenp      : in     A_Ub2    := null;
      Rcodep     : in     A_Ub2    := null;
      Maxarr_Len : in     Ub4      := 0;
      Curelep    : in     A_Ub4    := null;
      Mode       : in     Ub4      := OCI_DEFAULT) return SWord;

   function OCIBindByPos
     (Stmtp      : in     OCIStmt;
      Bindpp     : access OCIBind;
      Errhp      : in     OCIError;
      Position   : in     Ub4;
      Valuep     : in     Text_Ptr;
      Value_Sz   : in     Sb4;
      Dty        : in     Ub2;
      Indp       : access Sb2;
      Alenp      : in     A_Ub2    := null;
      Rcodep     : in     A_Ub2    := null;
      Maxarr_Len : in     Ub4      := 0;
      Curelep    : in     A_Ub4    := null;
      Mode       : in     Ub4      := OCI_DEFAULT) return SWord;

   function OCIDateToText
     (Err         : in     OCIError;
      Date        : in     OCIDate;
      Fmt         : in     Text;
      Fmt_Length  : in     Ub1;
      Lang_Name   : in     Text_Ptr;
      Lang_Length : in     Ub4;
      Buf_Size    : access Ub4;
      Buf         : in     Text_Ptr) return SWord;

   function OCIDateFromText
     (Err          : in     OCIError;
      Date_Str     : in     Text;
      d_Str_Length : in     Ub4;
      Fmt          : in     Text;
      Fmt_Length   : in     Ub1;
      Lang_Name    : in     Text_Ptr;
      Lang_Length  : in     Ub4;
      Date         : access OCIDate) return SWord;

   function OCIDateCompare
     (Err    : in     OCIError;
      Date1  : in     OCIDate;
      Date2  : in     OCIDate;
      Result : access SWord) return SWord;

   function OCIDefineByPos
     (Stmtp    : in     OCIStmt;
      Defnpp   : access OCIDefine;
      Errhp    : in     OCIError;
      Position : in     Ub4;
      Value    : in     DVoid;
      Value_Sz : in     Sb4;
      Dty      : in     Ub2;
      Indp     : in     A_Sb2;
      Rlenp    : in     A_Ub2 := null;
      Rcodep   : in     A_Ub2 := null;
      Mode     : in     Ub4   := OCI_DEFAULT) return SWord;

   function OCIDefineByPos
     (Stmtp    : in     OCIStmt;
      Defnpp   : access OCIDefine;
      Errhp    : in     OCIError;
      Position : in     Ub4;
      Value    : in     Text_Ptr;
      Value_Sz : in     Sb4;
      Dty      : in     Ub2;
      Indp     : access Sb2;
      Rlenp    : in     A_Ub2 := null;
      Rcodep   : in     A_Ub2 := null;
      Mode     : in     Ub4   := OCI_DEFAULT) return SWord;

   function OCINumberToText
     (Err        : in     OCIError;
      Date       : in     OCINumber;
      Fmt        : in     Text;
      Fmt_Length : in     Ub4;
      Nls_Name   : in     Undefined;
      Nls_Length : in     Ub4;
      Buf_Size   : access Ub4;
      Buf        : in     Text_Ptr) return SWord;

   function OCINumberFromText
     (Err          : in     OCIError;
      Buf          : in     Text;
      Str_Length   : in     Ub4;
      Fmt          : in     Text;
      Fmt_Length   : in     Ub4;
      NLS_params   : in     Undefined;
      nls_p_length : in     Ub4;
      number       : access OCINumber) return SWord;

   function OCIDateSysDate
     (Err : in OCIError; Date : access OCIDate) return SWord;

   function OCIStringResize
     (Env      : in     OCIEnv;
      Err      : in     OCIError;
      New_Size : in     Ub4;
      Str      : access OCIString) return SWord;

   function OCIStringAssignText
     (Env     : in     OCIEnv;
      Err     : in     OCIError;
      Rhs     : in     Text;
      Rhs_Len : in     Ub2;
      Lhs     : access OCIString) return SWord;

   function OCIStringPtr (Env : OCIEnv; Vs : OCIString) return CStr.chars_ptr;

   function OCIStringSize (Env : in OCIEnv; Vs : in OCIString) return Ub4;

   function OCIStringAllocSize
     (Env       : in     OCIEnv;
      Vs        : in     OCIString;
      Allocsize : access Ub4) return SWord;

   function OCIDescriptorAlloc
     (Parenth    : in     OCIEnv;
      Descpp     : access DVoid;
      Htype      : in     Ub4;
      Xtramem_Sz : in     C.size_t  := 0;
      Usrmempp   : in     Undefined := Empty_Undefined) return SWord;

   function OCIDescriptorFree (Descp : OCIHandle; Dtype : Ub4) return SWord;

   function OCIServerVersion
     (Hndlp    : in OCIHandle;
      Errhp    : in OCIError;
      Bufp     : in Text_Ptr;
      Bufsz    : in Ub4;
      Hndltype : in Ub1) return SWord;

   procedure OCIClientVersion
     (Major_Version   : access SWord;
      Minor_Version   : access SWord;
      Update_Num      : access SWord;
      Patch_Num       : access SWord;
      Port_Update_Num : access SWord);

   function OCITransCommit
     (Svchp : in OCISvcCtx;
      Errhp : in OCIError;
      Flags : in Ub4       := OCI_DEFAULT) return SWord;

   function OCITransRollback
     (Svchp : in OCISvcCtx;
      Errhp : in OCIError;
      Flags : in Ub4       := OCI_DEFAULT) return SWord;

   function OCINumberAbs
     (Err    : in     OCIError;
      Number : in     OCINumber;
      Result : access OCINumber) return SWord;

   function OCINumberArcCos
     (Err    : in     OCIError;
      Number : in     OCINumber;
      Result : access OCINumber) return SWord;

   function OCINumberArcSin
     (Err    : in     OCIError;
      Number : in     OCINumber;
      Result : access OCINumber) return SWord;

   function OCINumberArcTan
     (Err    : in     OCIError;
      Number : in     OCINumber;
      Result : access OCINumber) return SWord;

   function OCINumberAssign
     (Err    : in     OCIError;
      Number : in     OCINumber;
      Result : access OCINumber) return SWord;

   function OCINumberCeil
     (Err    : in     OCIError;
      Number : in     OCINumber;
      Result : access OCINumber) return SWord;

   function OCINumberCos
     (Err    : in     OCIError;
      Number : in     OCINumber;
      Result : access OCINumber) return SWord;

   function OCINumberExp
     (Err    : in     OCIError;
      Number : in     OCINumber;
      Result : access OCINumber) return SWord;

   function OCINumberFloor
     (Err    : in     OCIError;
      Number : in     OCINumber;
      Result : access OCINumber) return SWord;

   function OCINumberHypCos
     (Err    : in     OCIError;
      Number : in     OCINumber;
      Result : access OCINumber) return SWord;

   function OCINumberHypSin
     (Err    : in     OCIError;
      Number : in     OCINumber;
      Result : access OCINumber) return SWord;

   function OCINumberHypTan
     (Err    : in     OCIError;
      Number : in     OCINumber;
      Result : access OCINumber) return SWord;

   function OCINumberLn
     (Err    : in     OCIError;
      Number : in     OCINumber;
      Result : access OCINumber) return SWord;

   function OCINumberNeg
     (Err    : in     OCIError;
      Number : in     OCINumber;
      Result : access OCINumber) return SWord;

   function OCINumberSin
     (Err    : in     OCIError;
      Number : in     OCINumber;
      Result : access OCINumber) return SWord;

   function OCINumberSqrt
     (Err    : in     OCIError;
      Number : in     OCINumber;
      Result : access OCINumber) return SWord;

   function OCINumberTan
     (Err    : in     OCIError;
      Number : in     OCINumber;
      Result : access OCINumber) return SWord;

   function OCINumberArcTan2
     (Err     : in     OCIError;
      Number1 : in     OCINumber;
      Number2 : in     OCINumber;
      Result  : access OCINumber) return SWord;

   function OCINumberAdd
     (Err     : in     OCIError;
      Number1 : in     OCINumber;
      Number2 : in     OCINumber;
      Result  : access OCINumber) return SWord;

   function OCINumberCmp
     (Err     : in     OCIError;
      Number1 : in     OCINumber;
      Number2 : in     OCINumber;
      Result  : access SWord) return SWord;

   function OCINumberSign
     (Err    : in     OCIError;
      Number : in     OCINumber;
      Result : access SWord) return SWord;

   function OCINumberDiv
     (Err     : in     OCIError;
      Number1 : in     OCINumber;
      Number2 : in     OCINumber;
      Result  : access OCINumber) return SWord;

   function OCINumberLog
     (Err     : in     OCIError;
      Number1 : in     OCINumber;
      Number2 : in     OCINumber;
      Result  : access OCINumber) return SWord;

   function OCINumberMod
     (Err     : in     OCIError;
      Number1 : in     OCINumber;
      Number2 : in     OCINumber;
      Result  : access OCINumber) return SWord;

   function OCINumberMul
     (Err     : in     OCIError;
      Number1 : in     OCINumber;
      Number2 : in     OCINumber;
      Result  : access OCINumber) return SWord;

   function OCINumberPower
     (Err     : in     OCIError;
      Number1 : in     OCINumber;
      Number2 : in     OCINumber;
      Result  : access OCINumber) return SWord;

   function OCINumberSub
     (Err     : in     OCIError;
      Number1 : in     OCINumber;
      Number2 : in     OCINumber;
      Result  : access OCINumber) return SWord;

   function OCINumberToReal
     (Err        : in OCIError;
      Number     : in OCINumber;
      Rsl_Length : in UWord;
      Rsl        : in DVoid) return SWord;

   function OCINumberToInt
     (Err        : in OCIError;
      Number     : in OCINumber;
      Rsl_Length : in UWord;
      Rsl_Flag   : in UWord;
      Rsl        : in DVoid) return SWord;

   function OCINumberFromReal
     (Err         : in     OCIError;
      Rnum        : in     DVoid;
      Rnum_Length : in     UWord;
      Number      : access OCINumber) return SWord;

   function OCINumberFromInt
     (Err         : in     OCIError;
      Inum        : in     DVoid;
      Inum_Length : in     UWord;
      Inum_S_Flag : in     UWord;
      Number      : access OCINumber) return SWord;

   function OCINumberRound
     (Err      : in     OCIError;
      Number   : in     OCINumber;
      DecPlace : in     SWord;
      Result   : access OCINumber) return SWord;

   function OCINumberPrec
     (Err    : in     OCIError;
      Number : in     OCINumber;
      NDigs  : in     EWord;
      Result : access OCINumber) return SWord;

   function OCINumberTrunc
     (Err      : in     OCIError;
      Number   : in     OCINumber;
      DecPlace : in     SWord;
      Result   : access OCINumber) return SWord;

   function OCINumberIntPower
     (Err    : in     OCIError;
      Number : in     OCINumber;
      Exp    : in     SWord;
      Result : access OCINumber) return SWord;

   function OCINumberShift
     (Err    : in     OCIError;
      Number : in     OCINumber;
      nDig   : in     SWord;
      Result : access OCINumber) return SWord;

   function OCINumberIsZero
     (Err    : in     OCIError;
      Number : in     OCINumber;
      Result : access C_Boolean) return SWord;

   function OCILobAppend
     (Svchp    : in OCISvcCtx;
      Errhp    : in OCIError;
      Dst_Locp : in OCILobLocator;
      Src_Locp : in OCILobLocator) return SWord;

   function OCILobAssign
     (Envhp     : in OCIEnv;
      Errhp     : in OCIError;
      Src_Locp  : in OCILobLocator;
      Dst_Locpp : in A_OCILobLocator) return SWord;

   function OCILobCharSetForm
     (Envhp : in OCIEnv;
      Errhp : in OCIError;
      Locp  : in OCILobLocator;
      Csfrm : in A_Ub1) return SWord;

   function OCILobCharSetId
     (Envhp : in OCIEnv;
      Errhp : in OCIError;
      Locp  : in OCILobLocator;
      Csid  : in A_Ub2) return SWord;

   function OCILobCopy
     (Svchp      : in OCISvcCtx;
      Errhp      : in OCIError;
      Dst_Locp   : in OCILobLocator;
      Src_Locp   : in OCILobLocator;
      Amount     : in Ub4;
      Dst_Offset : in Ub4;
      Src_Offset : in Ub4) return SWord;

   function OCILobCreateTemporary
     (Svchp    : in OCISvcCtx;
      Errhp    : in OCIError;
      Locp     : in OCILobLocator;
      Csid     : in Ub2;
      Csfrm    : in Ub1;
      Lobtype  : in Ub1;
      Cache    : in C.int;
      Duration : in OCIDuration) return SWord;

   function OCILobClose
     (Svchp : in OCISvcCtx;
      Errhp : in OCIError;
      Locp  : in OCILobLocator) return SWord;

   function OCILobDisableBuffering
     (Svchp : in OCISvcCtx;
      Errhp : in OCIError;
      Locp  : in OCILobLocator) return SWord;

   function OCILobEnableBuffering
     (Svchp : in OCISvcCtx;
      Errhp : in OCIError;
      Locp  : in OCILobLocator) return SWord;

   function OCILobErase
     (Svchp  : in OCISvcCtx;
      Errhp  : in OCIError;
      Locp   : in OCILobLocator;
      Amount : in A_Ub4;
      Offset : in Ub4) return SWord;

   function OCILobFileClose
     (Svchp : in OCISvcCtx;
      Errhp : in OCIError;
      Filep : in OCILobLocator) return SWord;

   function OCILobFileCloseAll
     (Svchp : in OCISvcCtx;
      Errhp : in OCIError) return SWord;

   function OCILobFileExists
     (Svchp : in OCISvcCtx;
      Errhp : in OCIError;
      Filep : in OCILobLocator;
      Flag  : in C.int) return SWord;

   function OCILobFileGetName
     (Envhp     : in OCIEnv;
      Errhp     : in OCIError;
      Filep     : in OCILobLocator;
      Dir_Alias : in OraText;
      D_Length  : in A_Ub2;
      Filename  : in OraText;
      F_Length  : in A_Ub2) return SWord;

   function OCILobFileIsOpen
     (Svchp : in OCISvcCtx;
      Errhp : in OCIError;
      Filep : in OCILobLocator;
      Flag  : in A_Int) return SWord;

   function OCILobFileOpen
     (Svchp : in OCISvcCtx;
      Errhp : in OCIError;
      Filep : in OCILobLocator;
      Mode  : in Ub1) return SWord;

   function OCILobFileSetName
     (Envhp     : in OCIEnv;
      Errhp     : in OCIError;
      Filepp    : in A_OCILobLocator;
      Dir_Alias : in OraText;
      D_Length  : in Ub2;
      Filename  : in OraText;
      F_Length  : in Ub2) return SWord;

   function OCILobFlushBuffer
     (Svchp : in OCISvcCtx;
      Errhp : in OCIError;
      Locp  : in OCILobLocator;
      Flag  : in Ub4) return SWord;

   function OCILobFreeTemporary
     (Svchp : in OCISvcCtx;
      Errhp : in OCIError;
      Locp  : in OCILobLocator) return SWord;

   function OCILobGetChunkSize
     (Svchp      : in OCISvcCtx;
      Errhp      : in OCIError;
      Locp       : in OCILobLocator;
      Chunksizep : in A_Ub4) return SWord;

   function OCILobGetLength
     (Svchp : in OCISvcCtx;
      Errhp : in OCIError;
      Locp  : in OCILobLocator;
      Lenp  : in A_Ub4) return SWord;

   function OCILobIsEqual
     (Envhp    : in OCIEnv;
      X        : in OCILobLocator;
      Y        : in OCILobLocator;
      Is_Equal : in A_Int) return SWord;

   function OCILobIsOpen
     (Svchp : in OCISvcCtx;
      Errhp : in OCIError;
      Locp  : in OCILobLocator;
      Flag  : in A_Int) return SWord;

   function OCILobIsTemporary
     (Envhp        : in OCIEnv;
      Errhp        : in OCIError;
      Locp         : in OCILobLocator;
      Is_Temporary : in A_Int) return SWord;

   function OCILobLoadFromFile
     (Svchp      : in OCISvcCtx;
      Errhp      : in OCIError;
      Dst_Locp   : in OCILobLocator;
      Src_Filep  : in OCILobLocator;
      Amount     : in Ub4;
      Dst_Offset : in Ub4;
      Src_Offset : in Ub4) return SWord;

   function OCILobLocatorAssign
     (Svchp     : in OCISvcCtx;
      Errhp     : in OCIError;
      Src_Locp  : in OCILobLocator;
      Dst_Locpp : in A_OCILobLocator) return SWord;

   function OCILobLocatorIsInit
     (Envhp          : in OCIEnv;
      Errhp          : in OCIError;
      Locp           : in OCILobLocator;
      Is_Initialized : in A_Int) return SWord;

   function OCILobOpen
     (Svchp : in OCISvcCtx;
      Errhp : in OCIError;
      Locp  : in OCILobLocator;
      Mode  : in Ub1) return SWord;

   function OCILobRead
     (Svchp  : in OCISvcCtx;
      Errhp  : in OCIError;
      Locp   : in OCILobLocator;
      Amtp   : in A_Ub4;
      Offset : in Ub4;
      Bufp   : in System.Address;
      Bufl   : in Ub4;
      Ctxp   : in System.Address;
      Cbfp   : in Function_Pointer;
      Csid   : in Ub2;
      Csfrm  : in Ub1 := SQLCS_IMPLICIT) return SWord;

   function OCILobTrim
     (Svchp  : in OCISvcCtx;
      Errhp  : in OCIError;
      Locp   : in OCILobLocator;
      Newlen : in Ub4) return SWord;

   function OCILobWrite
     (Svchp  : in OCISvcCtx;
      Errhp  : in OCIError;
      Locp   : in OCILobLocator;
      Amtp   : in A_Ub4;
      Offset : in Ub4;
      Bufp   : in System.Address;
      Buflen : in Ub4;
      Piece  : in Ub1;
      Ctxp   : in System.Address;
      Cbfp   : in Function_Pointer;
      Csid   : in Ub2;
      Csfrm  : in Ub1 := SQLCS_IMPLICIT) return SWord;

   function OCILobWriteAppend
     (Svchp : in OCISvcCtx;
      Errhp : in OCIError;
      Lobp  : in OCILobLocator;
      Amtp  : in A_Ub4;
      Bufp  : in System.Address;
      Bufl  : in Ub4;
      Piece : in Ub1;
      Ctxp  : in System.Address;
      Cbfp  : in Function_Pointer;
      Csid  : in Ub2;
      Csfrm : in Ub1 := SQLCS_IMPLICIT) return SWord;

   function OCIStmtGetBindInfo
     (Stmtp    : in OCIStmt;
      Errhp    : in OCIError;
      Size     : in Ub4;
      Startloc : in Ub4;
      Found    : in A_Sb4;
      Bvnp     : in Vector_Of_OraText;
      Bvnl     : in Vector_Of_Ub1;
      Invp     : in Vector_Of_OraText;
      Inpl     : in Vector_Of_Ub1;
      Dupl     : in Vector_Of_Ub1;
      Hndl     : in Vector_Of_OCIBind) return SWord;

   function OCIStmtGetBindInfo
     (Stmtp    : in OCIStmt;
      Errhp    : in OCIError;
      Size     : in Ub4;
      Startloc : in Ub4;
      Found    : in A_Sb4;
      Bvnp     : in A_Vector_Of_OraText;
      Bvnl     : in A_Vector_Of_Ub1;
      Invp     : in A_Vector_Of_OraText;
      Inpl     : in A_Vector_Of_Ub1;
      Dupl     : in A_Vector_Of_Ub1;
      Hndl     : in A_Vector_Of_OCIBind) return SWord;

   function OCIDescribeAny
     (Svchp      : in OCISvcCtx;
      Errhp      : in OCIError;
      Objptr     : in DVoid;
      Objnm_Len  : in Ub4;
      Objptr_Typ : in Ub1;
      Info_Level : in Ub1;
      Objtyp     : in Ub1;
      Dschp      : in OCIDescribe) return SWord;

   type OCICallbackInBind is access function
     (Ictxp  : in     DVoid;
      Bindp  : in     OCIBind;
      Iter   : in     Ub4;
      Index  : in     Ub4;
      Bufpp  : access DVoid;
      Alenp  : access Ub4;
      Piecep : access Ub1;
      Indp   : access DVoid) return SWord;
   pragma Convention (C, OCICallbackInBind);

   type OCICallbackOutBind is access function
     (Octxp   : in     DVoid;
      Bindp   : in     OCIBind;
      Iter    : in     Ub4;
      Index   : in     Ub4;
      Bufp    : access DVoid;
      Alenpp  : access A_Ub4;
      Piecep  : access Ub1;
      Indpp   : access DVoid;
      Rcodepp : access A_Ub2) return SWord;
   pragma Convention (C, OCICallbackOutBind);

   function OCIBindDynamic
     (Bindp  : in OCIBind;
      Errhp  : in OCIError;
      Ictxp  : in DVoid;
      Icbfp  : in DVoid;
      Octxp  : in DVoid;
      Ocbfp  : in DVoid) return SWord;

   function OCIBindDynamic
     (Bindp  : in OCIBind;
      Errhp  : in OCIError;
      Ictxp  : in DVoid;
      Icbfp  : in OCICallbackInBind;
      Octxp  : in DVoid;
      Ocbfp  : in OCICallbackOutBind) return SWord;

   type OCICallbackDefine is access function
     (Octxp   : in     DVoid;
      Defnp   : in     OCIDefine;
      Iter    : in     Ub4;
      Bufpp   : access DVoid;
      Alenpp  : access A_Ub4;
      Piecep  : access Ub1;
      Indpp   : access DVoid;
      Rcodepp : access A_Ub2) return SWord;
   pragma Convention (C, OCICallbackDefine);

   function OCIDefineDynamic
     (Defnp  : in OCIDefine;
      Errhp  : in OCIError;
      Octxp  : in DVoid;
      Ocbfp  : in OCICallbackDefine) return SWord;

   function OCIDefineDynamic
     (Defnp  : in OCIDefine;
      Errhp  : in OCIError;
      Octxp  : in DVoid;
      Ocbfp  : in DVoid) return SWord;

   --  External Proc Callbacks  --
   function OCIExtProcGetEnv
     (octxp  : in     OCIExtProcContext;
      Envhp  : access OCIEnv;
      Svchp  : access OCISvcCtx;
      Errhp  : access OCIError) return SWord;

   function OCITerminate (Mode : Ub4 := OCI_DEFAULT) return SWord;

   type OCIUserCallback is access function
     (Ctxp       : in     DVoid;
      Hndlp      : in     OCIHandle;
      Htype      : in     Ub4;
      Fcode      : in     Ub4;
      When_Call  : in     Ub1;
      ReturnCode : in     SWord;
      Errnop     : access Sb4;
      Arglist    : in     DVoid) return SWord;
   pragma Convention (C, OCIUserCallback);

   function OCIUserCallbackRegister
     (Hndlp     : in OCIHandle;
      Htype     : in Ub4;
      Ehndlp    : in OCIError;
      Callback  : in OCIUserCallback;
      Ctxp      : in DVoid;
      Fcode     : in Ub4;
      When_Call : in Ub1;
      UcbDesc   : in DVoid) return SWord;

   function OCIUserCallbackRegister
     (Hndlp     : in OCIHandle;
      Htype     : in Ub4;
      Ehndlp    : in OCIError;
      Callback  : in DVoid;
      Ctxp      : in DVoid;
      Fcode     : in Ub4;
      When_Call : in Ub1;
      UcbDesc   : in DVoid) return SWord;

   function OCIServerAttach
     (Srvhp      : in OCIServer;
      Errhp      : in OCIError;
      Dblink     : in Text;
      Dblink_Len : in Sb4;
      Mode       : in Ub4 := OCI_DEFAULT) return SWord;

   function OCIServerDetach
     (Srvhp : in OCIServer;
      Errhp : in OCIError;
      Mode  : in Ub4 := OCI_DEFAULT) return SWord;

   function OCISessionBegin
     (Svchp : in OCISvcCtx;
      Errhp : in OCIError;
      Usrhp : in OCISession;
      Credt : in Ub4 := OCI_CRED_RDBMS;
      Mode  : in Ub4 := OCI_DEFAULT) return SWord;

   function OCISessionEnd
     (Svchp : in OCISvcCtx;
      Errhp : in OCIError;
      Usrhp : in OCISession;
      Mode  : in Ub4 := OCI_DEFAULT) return SWord;

   function OCIBreak (hndlp : in OCIHandle; ehndlp : in OCIError) return SWord;
   function OCIReset (hndlp : in OCIHandle; ehndlp : in OCIError) return SWord;

   function OCISubscriptionRegister
     (Svchp     : in     OCISvcCtx;
      Subscrhpp : access OCISubscription;
      Count     : in     Ub2;
      Errhp     : in     OCIError;
      Mode      : in     Ub4) return SWord;

   function OCISubscriptionUnRegister
     (Svchp    : in OCISvcCtx;
      Subscrhp : in OCISubscription;
      Errhp    : in OCIError;
      Mode     : in Ub4) return  SWord;

   function OCISubscriptionEnable
     (Subscrhp : in OCISubscription;
      Errhp    : in OCIError;
      Mode     : in Ub4) return SWord;

   function OCISubscriptionDisable
     (Subscrhp : in OCISubscription;
      Errhp    : in OCIError;
      Mode     : in Ub4) return SWord;

   function OCISubscriptionPost
     (Svchp      : in     OCISvcCtx;
      Subscrhpp  : access OCISubscription;
      Count      : in     Ub2;
      Errhp      : in     OCIError;
      Mode       : in     Ub4) return SWord;

   function OCIObjectNew
     (Env       : in     OCIEnv;
      Err       : in     OCIError;
      Svc       : in     OCISvcCtx;
      Typecode  : in     OCITypeCode;
      TDO       : in     OCIType;
      Table     : in     OCIHandle;
      Duration  : in     OCIDuration;
      Value     : in     Bool;
      Instance  : access OCIHandle) return SWord;

   function OCITypeByName
     (Env       : in     OCIEnv;
      Err       : in     OCIError;
      Svc       : in     OCISvcCtx;
      Schema    : in     Text;
      S_Length  : in     Ub4;
      Type_Name : in     Text;
      T_Length  : in     Ub4;
      Version   : in     Text;
      V_Length  : in     Ub4;
      Duration  : in     OCIDuration;
      Option    : in     OCITypeGetOpt;
      TDO       : access OCIType) return SWord;

   function OCIObjectFree
     (Env       : in OCIEnv;
      Err       : in OCIError;
      Instance  : in OCIHandle;
      Flags     : in Ub2) return SWord;

   function OCIObjectGetInd
     (Env         : in     OCIEnv;
      Err         : in     OCIError;
      Instance    : in     OCIHandle;
      Null_Struct : access OCIHandle) return SWord;

   function OCIDefineObject
     (Defnp    : in OCIDefine;
      Errhp    : in OCIError;
      TDO      : in OCIType;
      Value    : in DVoid;
      Value_Sz : in A_Sb4;
      Indp     : in DVoid;
      Ind_Sz   : in A_Sb4) return SWord;

   function OCIBindObject
     (Bindp    : in OCIBind;
      Errhp    : in OCIError;
      TDO      : in OCIType;
      Value    : in DVoid;
      Value_Sz : in A_Sb4;
      Indp     : in DVoid;
      Ind_Sz   : in A_Sb4) return SWord;

   function OCICollAppend
     (Env     : in     OCIEnv;
      Err     : in     OCIError;
      Elem    : in     DVoid;
      Elemind : in     DVoid := System.Null_Address;
      Coll    : in     OCIColl) return SWord;

   function OCICollSize
     (Env  : in     OCIEnv;
      Err  : in     OCIError;
      Coll : in     OCIColl;
      Size : access Sb4) return SWord;

   function OCICollGetElem
     (Env     : in     OCIEnv;
      Err     : in     OCIError;
      Coll    : in     OCIColl;
      Index   : in     Sb4;
      Exists  : access Bool;
      Elem    : access DVoid;
      Elemind : in     DVoid := System.Null_Address) return SWord;

   function OCICollGetElemArray
     (Env     : in     OCIEnv;
      Err     : in     OCIError;
      Coll    : in     OCIColl;
      Index   : in     Sb4;
      Exists  : access Bool;
      Elem    : access DVoid;
      Elemind : in     DVoid := System.Null_Address;
      Length  : access UWord) return SWord;

   function OCIStmtGetPieceInfo
     (Stmtp   : in     OCIStmt;
      Errhp   : in     OCIError;
      Hndlpp  : access DVoid;
      Typep   : access Ub4;
      In_Outp : access Ub1;
      Iterp   : access Ub4;
      Idxp    : access Ub4;
      Piecep  : access Ub1) return SWord;

   function OCIStmtSetPieceInfo
      (Hndlp  : in     DVoid;
       Typev  : in     Ub4;
       Errhp  : in     OCIError;
       Bufp   : in     DVoid;
       Alenp  : access Ub4;
       Piece  : in     Ub1;
       Indp   : in     DVoid;
       Rcodep : access Ub2) return SWord;

   procedure OCIThreadProcessInit;

   function OCIThreadInit (Hndl : in DVoid; Err : in OCIError) return SWord;

   function OCIThreadHndInit
     (Hndl : in     DVoid;
      Err  : in     OCIError;
      THnd : access OCIThreadHandle) return SWord;

   function OCIThreadCreate
     (Hndl  : in DVoid;
      Err   : in OCIError;
      Start : in Function_Pointer; -- void (*start) (dvoid *arg)
      Arg   : in DVoid;
      tid   : in OCIThreadId;
      tHnd  : in OCIThreadHandle) return SWord;

   function OCIThreadJoin
     (Hndl : in DVoid;
      Err  : in OCIError;
      THnd : in OCIThreadHandle) return SWord;

   function OCIThreadMutexInit
     (Hndl  : in     DVoid;
      Err   : in     OCIError;
      Mutex : access OCIThreadMutex) return SWord;

   function OCIThreadMutexDestroy
     (Hndl  : in     DVoid;
      Err   : in     OCIError;
      Mutex : access OCIThreadMutex) return SWord;

   function OCIThreadMutexAcquire
     (Hndl  : in DVoid;
      Err   : in OCIError;
      Mutex : in OCIThreadMutex) return SWord;

   function OCIThreadMutexRelease
     (Hndl  : in DVoid;
      Err   : in OCIError;
      Mutex : in OCIThreadMutex) return SWord;

private

   pragma Import (C, OCIEnvCreate, "OCIEnvCreate");
   pragma Import (C, OCIErrorGet, "OCIErrorGet");
   pragma Import (C, OCILogon, "OCILogon");
   pragma Import (C, OCILogoff, "OCILogoff");
   pragma Import (C, OCIInitialize, "OCIInitialize");
   pragma Import (C, OCIEnvInit, "OCIEnvInit");
   pragma Import (C, OCIHandleAlloc, "OCIHandleAlloc");
   pragma Import (C, OCIHandleFree, "OCIHandleFree");
   pragma Import (C, OCIStmtPrepare, "OCIStmtPrepare");
   pragma Import (C, OCIStmtExecute, "OCIStmtExecute");
   pragma Import (C, OCIStmtGetPieceInfo, "OCIStmtGetPieceInfo");
   pragma Import (C, OCIStmtSetPieceInfo, "OCIStmtSetPieceInfo");
   pragma Import (C, OCIParamGet, "OCIParamGet");
   pragma Import (C, OCIAttrGet, "OCIAttrGet");
   pragma Import (C, OCIAttrSet, "OCIAttrSet");
   pragma Import (C, OCIStmtFetch, "OCIStmtFetch");
   pragma Import (C, OCIBindByName, "OCIBindByName");
   pragma Import (C, OCIBindByPos, "OCIBindByPos");
   pragma Import (C, OCIDefineByPos, "OCIDefineByPos");
   pragma Import (C, OCIDateToText, "OCIDateToText");
   pragma Import (C, OCIDateFromText, "OCIDateFromText");
   pragma Import (C, OCIDateCompare, "OCIDateCompare");
   pragma Import (C, OCIDateSysDate, "OCIDateSysDate");
   pragma Import (C, OCIStringResize, "OCIStringResize");
   pragma Import (C, OCIStringAssignText, "OCIStringAssignText");
   pragma Import (C, OCIStringPtr, "OCIStringPtr");
   pragma Import (C, OCIStringSize, "OCIStringSize");
   pragma Import (C, OCIStringAllocSize, "OCIStringAllocSize");
   pragma Import (C, OCIDescriptorAlloc, "OCIDescriptorAlloc");
   pragma Import (C, OCIDescriptorFree, "OCIDescriptorFree");
   pragma Import (C, OCIServerVersion, "OCIServerVersion");
   pragma Import (C, OCIClientVersion, "OCIClientVersion");
   pragma Import (C, OCITransCommit, "OCITransCommit");
   pragma Import (C, OCITransRollback, "OCITransRollback");
   pragma Import (C, OCINumberAbs, "OCINumberAbs");
   pragma Import (C, OCINumberArcCos, "OCINumberArcCos");
   pragma Import (C, OCINumberArcSin, "OCINumberArcSin");
   pragma Import (C, OCINumberArcTan, "OCINumberArcTan");
   pragma Import (C, OCINumberAssign, "OCINumberAssign");
   pragma Import (C, OCINumberCeil, "OCINumberCeil");
   pragma Import (C, OCINumberCos, "OCINumberCos");
   pragma Import (C, OCINumberExp, "OCINumberExp");
   pragma Import (C, OCINumberFloor, "OCINumberFloor");
   pragma Import (C, OCINumberHypCos, "OCINumberHypCos");
   pragma Import (C, OCINumberHypSin, "OCINumberHypSin");
   pragma Import (C, OCINumberHypTan, "OCINumberHypTan");
   pragma Import (C, OCINumberLn, "OCINumberLn");
   pragma Import (C, OCINumberNeg, "OCINumberNeg");
   pragma Import (C, OCINumberSin, "OCINumberSin");
   pragma Import (C, OCINumberSqrt, "OCINumberSqrt");
   pragma Import (C, OCINumberTan, "OCINumberTan");
   pragma Import (C, OCINumberArcTan2, "OCINumberArcTan2");
   pragma Import (C, OCINumberAdd, "OCINumberAdd");
   pragma Import (C, OCINumberCmp, "OCINumberCmp");
   pragma Import (C, OCINumberSign, "OCINumberSign");
   pragma Import (C, OCINumberDiv, "OCINumberDiv");
   pragma Import (C, OCINumberLog, "OCINumberLog");
   pragma Import (C, OCINumberMod, "OCINumberMod");
   pragma Import (C, OCINumberMul, "OCINumberMul");
   pragma Import (C, OCINumberPower, "OCINumberPower");
   pragma Import (C, OCINumberSub, "OCINumberSub");
   pragma Import (C, OCINumberToReal, "OCINumberToReal");
   pragma Import (C, OCINumberToInt, "OCINumberToInt");
   pragma Import (C, OCINumberFromReal, "OCINumberFromReal");
   pragma Import (C, OCINumberFromInt, "OCINumberFromInt");
   pragma Import (C, OCINumberToText, "OCINumberToText");
   pragma Import (C, OCINumberFromText, "OCINumberFromText");
   pragma Import (C, OCINumberRound, "OCINumberRound");
   pragma Import (C, OCINumberTrunc, "OCINumberTrunc");
   pragma Import (C, OCINumberPrec, "OCINumberPrec");
   pragma Import (C, OCINumberIntPower, "OCINumberIntPower");
   pragma Import (C, OCINumberShift, "OCINumberShift");
   pragma Import (C, OCINumberIsZero, "OCINumberIsZero");
   pragma Import (C, OCILobAppend, "OCILobAppend");
   pragma Import (C, OCILobAssign, "OCILobAssign");
   pragma Import (C, OCILobCharSetForm, "OCILobCharSetForm");
   pragma Import (C, OCILobCharSetId, "OCILobCharSetId");
   pragma Import (C, OCILobCopy, "OCILobCopy");
   pragma Import (C, OCILobCreateTemporary, "OCILobCreateTemporary");
   pragma Import (C, OCILobClose, "OCILobClose");
   pragma Import (C, OCILobDisableBuffering, "OCILobDisableBuffering");
   pragma Import (C, OCILobEnableBuffering, "OCILobEnableBuffering");
   pragma Import (C, OCILobErase, "OCILobErase");
   pragma Import (C, OCILobFileClose, "OCILobFileClose");
   pragma Import (C, OCILobFileCloseAll, "OCILobFileCloseAll");
   pragma Import (C, OCILobFileExists, "OCILobFileExists");
   pragma Import (C, OCILobFileGetName, "OCILobFileGetName");
   pragma Import (C, OCILobFileIsOpen, "OCILobFileIsOpen");
   pragma Import (C, OCILobFileOpen, "OCILobFileOpen");
   pragma Import (C, OCILobFileSetName, "OCILobFileSetName");
   pragma Import (C, OCILobFlushBuffer, "OCILobFlushBuffer");
   pragma Import (C, OCILobFreeTemporary, "OCILobFreeTemporary");
   pragma Import (C, OCILobGetChunkSize, "OCILobGetChunkSize");
   pragma Import (C, OCILobGetLength, "OCILobGetLength");
   pragma Import (C, OCILobIsEqual, "OCILobIsEqual");
   pragma Import (C, OCILobIsOpen, "OCILobIsOpen");
   pragma Import (C, OCILobIsTemporary, "OCILobIsTemporary");
   pragma Import (C, OCILobLoadFromFile, "OCILobLoadFromFile");
   pragma Import (C, OCILobLocatorAssign, "OCILobLocatorAssign");
   pragma Import (C, OCILobLocatorIsInit, "OCILobLocatorIsInit");
   pragma Import (C, OCILobOpen, "OCILobOpen");
   pragma Import (C, OCILobRead, "OCILobRead");
   pragma Import (C, OCILobTrim, "OCILobTrim");
   pragma Import (C, OCILobWrite, "OCILobWrite");
   pragma Import (C, OCILobWriteAppend, "OCILobWriteAppend");
   pragma Import (C, OCIStmtGetBindInfo, "OCIStmtGetBindInfo");
   pragma Import (C, OCIDescribeAny, "OCIDescribeAny");
   pragma Import (C, OCIBindDynamic, "OCIBindDynamic");
   pragma Import (C, OCIDefineDynamic, "OCIDefineDynamic");
   pragma Import (C, OCIExtProcGetEnv, "ociepgoe");
   pragma Import (C, OCITerminate, "OCITerminate");
   pragma Import (C, OCIUserCallbackRegister, "OCIUserCallbackRegister");
   pragma Import (C, OCIServerAttach, "OCIServerAttach");
   pragma Import (C, OCIServerDetach, "OCIServerDetach");
   pragma Import (C, OCISessionBegin, "OCISessionBegin");
   pragma Import (C, OCISessionEnd, "OCISessionEnd");
   pragma Import (C, OCIBreak, "OCIBreak");
   pragma Import (C, OCIReset, "OCIReset");
   pragma Import (C, OCISubscriptionRegister, "OCISubscriptionRegister");
   pragma Import (C, OCISubscriptionUnRegister, "OCISubscriptionUnRegister");
   pragma Import (C, OCISubscriptionDisable, "OCISubscriptionDisable");
   pragma Import (C, OCISubscriptionEnable, "OCISubscriptionEnable");
   pragma Import (C, OCISubscriptionPost, "OCISubscriptionPost");
   pragma Import (C, OCIObjectNew, "OCIObjectNew");
   pragma Import (C, OCITypeByName, "OCITypeByName");
   pragma Import (C, OCIObjectFree, "OCIObjectFree");
   pragma Import (C, OCIObjectGetInd, "OCIObjectGetInd");
   pragma Import (C, OCIDefineObject, "OCIDefineObject");
   pragma Import (C, OCIBindObject, "OCIBindObject");
   pragma Import (C, OCICollSize, "OCICollSize");
   pragma Import (C, OCICollGetElem, "OCICollGetElem");
   pragma Import (C, OCICollGetElemArray, "OCICollGetElemArray");
   pragma Import (C, OCICollAppend, "OCICollAppend");
   pragma Import (C, OCIThreadProcessInit, "OCIThreadProcessInit");
   pragma Import (C, OCIThreadInit, "OCIThreadInit");
   pragma Import (C, OCIThreadCreate, "OCIThreadCreate");
   pragma Import (C, OCIThreadHndInit, "OCIThreadHndInit");
   pragma Import (C, OCIThreadJoin, "OCIThreadJoin");
   pragma Import (C, OCIThreadMutexInit, "OCIThreadMutexInit");
   pragma Import (C, OCIThreadMutexDestroy, "OCIThreadMutexDestroy");
   pragma Import (C, OCIThreadMutexAcquire, "OCIThreadMutexAcquire");
   pragma Import (C, OCIThreadMutexRelease, "OCIThreadMutexRelease");

end OCI.Lib;
