/* 
 *
 *   File: callbacks.c
 *
 *   Copyright (C) 2009-2012 Darran Kartaschew
 *
 *   This file is part of the gMTP package.
 *
 *   gMTP is free software; you can redistribute it and/or modify
 *   it under the terms of the BSD License as included within the
 *   file 'COPYING' located in the root directory
 *
 */

#include "config.h"

#include <glib.h>
#include <glib/gprintf.h>
#include <glib/gi18n.h>
#if GMTP_USE_GTK2
#include <gconf/gconf.h>
#include <gconf/gconf-client.h>
#else
#include <gio/gio.h>
#endif
#include <gtk/gtk.h>
#include <libmtp.h>
#include <id3tag.h>
#include <stdlib.h>

#include "main.h"
#include "callbacks.h"
#include "interface.h"
#include "mtp.h"
#include "prefs.h"
#include "dnd.h"

static gboolean formatThreadWorking = TRUE;

/**
 * on_quit1_activate - Call back for Quit toolbar and menu option.
 * @param menuitem
 * @param user_data
 */
void on_quit1_activate(GtkMenuItem *menuitem, gpointer user_data) {
    savePreferences();
#if GMTP_USE_GTK2
    gtk_exit(EXIT_SUCCESS);
#else
    exit(EXIT_SUCCESS);
#endif
} // end on_quit1_activate()

// ************************************************************************************************

/**
 * on_about1_activate - Call back for displaying the About Dialog Box
 * @param menuitem
 * @param user_data
 */
void on_about1_activate(GtkMenuItem *menuitem, gpointer user_data) {
    displayAbout();
} // end on_about1_activate()

// ************************************************************************************************

/**
 * on_deviceProperties_activate - Callback for displaying the device Properties Dialog box.
 * @param menuitem
 * @param user_data
 */
void on_deviceProperties_activate(GtkMenuItem *menuitem, gpointer user_data) {
    gchar *tmp_string;

    // We confirm our device properties, this should setup the device structure information we use below.
    deviceProperties();

    // Update the status bar with our information.
    if (DeviceMgr.storagedeviceID == MTP_DEVICE_SINGLE_STORAGE) {
        tmp_string = g_strdup_printf(_("Connected to %s - %d MB free"), DeviceMgr.devicename->str,
                (int) (DeviceMgr.devicestorage->FreeSpaceInBytes / MEGABYTE));
    } else {
        if (DeviceMgr.devicestorage->StorageDescription != NULL) {
            tmp_string = g_strdup_printf(_("Connected to %s (%s) - %d MB free"),
                    DeviceMgr.devicename->str,
                    DeviceMgr.devicestorage->StorageDescription,
                    (int) (DeviceMgr.devicestorage->FreeSpaceInBytes / MEGABYTE));
        } else {
            tmp_string = g_strdup_printf(_("Connected to %s - %d MB free"), DeviceMgr.devicename->str,
                    (int) (DeviceMgr.devicestorage->FreeSpaceInBytes / MEGABYTE));
        }
    }
    statusBarSet(tmp_string);
    g_free(tmp_string);

    // No idea how this could come about, but we should take it into account so we don't have a
    // memleak due to recreating the window multiple times.
    if (windowPropDialog != NULL) {
        gtk_widget_hide(windowPropDialog);
        gtk_widget_destroy(windowPropDialog);
    }

    // Create and show the dialog box.
    windowPropDialog = create_windowProperties();
    gtk_widget_show(GTK_WIDGET(windowPropDialog));
} // end on_deviceProperties_activate()

// ************************************************************************************************

/**
 * on_quitProp_activate - Callback used to close the Properties Dialog.
 * @param menuitem
 * @param user_data
 */
void on_quitProp_activate(GtkMenuItem *menuitem, gpointer user_data) {
    gtk_widget_hide(windowPropDialog);
    gtk_widget_destroy(windowPropDialog);
    windowPropDialog = NULL;
} // end on_quitProp_activate()

// ************************************************************************************************

/**
 * on_deviceRescan_activate - Callback to rescan the device properties and update the main
 * application window.
 * @param menuitem
 * @param user_data
 */
void on_deviceRescan_activate(GtkMenuItem *menuitem, gpointer user_data) {
    deviceRescan();
} // end on_deviceRescan_activate()

// ************************************************************************************************

/**
 * on_filesAdd_activate - Callback to initiate an Add Files operation.
 * @param menuitem
 * @param user_data
 */
void on_filesAdd_activate(GtkMenuItem *menuitem, gpointer user_data) {
    GSList* files;
    int64_t targetFol = 0;
    //uint32_t tmpFolderID = 0;
    // Set the Playlist ID to be asked if needed.
    if (Preferences.auto_add_track_to_playlist == TRUE) {
        addTrackPlaylistID = GMTP_REQUIRE_PLAYLIST;
    } else {
        addTrackPlaylistID = GMTP_NO_PLAYLIST;
    }
    // Get the files, and add them.
    files = getFileGetList2Add();

    // See if a folder is selected in the folder view, and if so add the files to that folder.
    if ((targetFol = folderListGetSelection()) != -1) {
        //tmpFolderID = currentFolderID;
        currentFolderID = (uint32_t) targetFol;
    }
    AlbumErrorIgnore = FALSE;
    if (files != NULL)
        g_slist_foreach(files, (GFunc) __filesAdd, NULL);

    // Now clear the GList;
    g_slist_foreach(files, (GFunc) g_free, NULL);
    g_slist_free(files);

    // Restore the current folder ID is we added to another folder.
    if (targetFol != -1) {
        // Disable this, so the user is taken to the folder in which the files were added to.
        //currentFolderID = tmpFolderID;
    }
    // Now do a device rescan to see the new files.
    deviceRescan();
    deviceoverwriteop = MTP_ASK;
} // end on_filesAdd_activate()

// ************************************************************************************************

/**
 * Callback to handle the Rename Device menu option.
 * @param menuitem
 * @param user_data
 */
void on_fileRenameFile_activate(GtkMenuItem *menuitem, gpointer user_data) {

    GtkTreePath *path;
    GtkTreeIter iter;
    gchar *newfilename = NULL;
    gchar *filename = NULL;
    gboolean isFolder;
    uint32_t ObjectID = 0;

    // Let's check to see if we have anything selected in our treeview?
    if (fileListGetSelection() == NULL) {

        // See if anything is selected in the folder view, if so use that as our source.
        if (folderListGetSelection() != -1) {
            on_folderRenameFolder_activate(menuitem, user_data);
        } else {
            displayInformation(_("No files/folders selected?"));
        }
        return;
    }
    GList *List = fileListGetSelection();

    // We only care about the first entry.
    // convert the referenece to a path and retrieve the iterator;
    path = gtk_tree_row_reference_get_path(List->data);
    gtk_tree_model_get_iter(GTK_TREE_MODEL(fileList), &iter, path);
    // We have our Iter now.
    // Before we download, is it a folder ?
    gtk_tree_model_get(GTK_TREE_MODEL(fileList), &iter, COL_FILENAME_ACTUAL, &filename, COL_ISFOLDER, &isFolder,
            COL_FILEID, &ObjectID, -1);

    // Make sure we are not attempting to edit the parent link folder.
    if (g_ascii_strcasecmp(filename, "..") == 0) {
        g_fprintf(stderr, _("Unable to rename parent folder\n"));
        displayInformation(_("Unable to rename this folder"));
        return;
    }
    // Get our new device name.
    newfilename = displayRenameFileDialog(filename);

    // If the user supplied something, then update the name of the device.
    if (newfilename != NULL) {
        filesRename(newfilename, ObjectID);
        g_free(newfilename);
        deviceRescan();
    }
} // end on_editDeviceName_activate()

// ************************************************************************************************

/**
 * Callback to handle the Move File menu option.
 * @param menuitem
 * @param user_data
 */
void on_fileMoveFile_activate(GtkMenuItem *menuitem, gpointer user_data) {
    GList *List = NULL;
    int64_t targetfolder = 0;

    // Let's check to see if we have anything selected in our treeview?
    if ((List = fileListGetSelection()) == NULL) {
        if (folderListGetSelection() != -1) {
            on_folderRemoveFolder_activate(menuitem, user_data);
        } else {
            displayInformation(_("No files/folders selected?"));
        }
        return;
    }

    // Prompt for the target folder location.
    targetfolder = getTargetFolderLocation();
    if ((targetfolder == -1) || (targetfolder == currentFolderID)) {
        // If the user didn't select a folder, or the target folder is the current selected folder
        // then do nothing.
        return;
    }
    fileMoveTargetFolder = targetfolder;
    fileListClearSelection();
    // List is a list of Iter's to be moved
    g_list_foreach(List, (GFunc) __fileMove, NULL);
    // We have 2 options, manually scan the file structure for that file and manually fix up...
    // or do a rescan...
    // I'll be cheap, and do a full rescan of the device.
    deviceRescan();
}


// ************************************************************************************************

/**
 * on_filesDelete_activate - Callback to initiate a Delete Files operation.
 * @param menuitem
 * @param user_data
 */
void on_filesDelete_activate(GtkMenuItem *menuitem, gpointer user_data) {
    GtkWidget *dialog;

    // Let's check to see if we have anything selected in our treeview?
    if (fileListGetSelection() == NULL) {
        if (folderListGetSelection() != -1) {
            on_folderRemoveFolder_activate(menuitem, user_data);
        } else {
            displayInformation(_("No files/folders selected?"));
        }
        return;
    }

    // Now we prompt to confirm delete?
    if (Preferences.confirm_file_delete_op == FALSE) {
        // Now download the actual file from the MTP device.
        fileListRemove(fileListGetSelection());
    } else {
        dialog = gtk_message_dialog_new(GTK_WINDOW(windowMain),
                GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
                GTK_MESSAGE_WARNING,
                GTK_BUTTONS_YES_NO,
                _("Are you sure you want to delete these files?"));
        gtk_window_set_title(GTK_WINDOW(dialog), _("Confirm Delete"));

        // Run the Dialog and get our result.
        gint result = gtk_dialog_run(GTK_DIALOG(dialog));
        if (result == GTK_RESPONSE_YES)
            fileListRemove(fileListGetSelection());

        // Destroy the dialog box.
        gtk_widget_destroy(dialog);
    }
} // on_filesDelete_activate()

// ************************************************************************************************

/**
 * on_filesDownload_activate - Callback to initiate a download files operation.
 * @param menuitem
 * @param user_data
 */
void on_filesDownload_activate(GtkMenuItem *menuitem, gpointer user_data) {
    int64_t targetfolder = 0;
    // Let's check to see if we have anything selected in our treeview?
    if (fileListGetSelection() == NULL) {
        if ((targetfolder = folderListGetSelection()) != -1) {
            folderListDownload(folderListGetSelectionName(), targetfolder);
        } else {
            displayInformation(_("No files/folders selected?"));
        }
        return;
    }

    // Download the selected files.
    fileListDownload(fileListGetSelection());
} // end on_filesDownload_activate()

// ************************************************************************************************

/**
 * on_deviceConnect_activate - Callback used to connect a device to the application.
 * @param menuitem
 * @param user_data
 */
void on_deviceConnect_activate(GtkMenuItem *menuitem, gpointer user_data) {
    gchar *tmp_string;
    GtkWidget *menuText;

    deviceConnect();
    //g_printf("Device connect/disconnect code = %d\n", result);
    // Update our label to indicate current condition.
    if (DeviceMgr.deviceConnected == TRUE) {
        // Set up our properties.
        deviceProperties();
        deviceRescan();

        // Update the toolbar to show a disconnect string.
        gtk_tool_button_set_label(GTK_TOOL_BUTTON(toolbuttonConnect), _("Disconnect"));

        // Now update the status bar;
        if (DeviceMgr.storagedeviceID == MTP_DEVICE_SINGLE_STORAGE) {
            tmp_string = g_strdup_printf(_("Connected to %s - %d MB free"), DeviceMgr.devicename->str,
                    (int) (DeviceMgr.devicestorage->FreeSpaceInBytes / MEGABYTE));
        } else {
            if (DeviceMgr.devicestorage->StorageDescription != NULL) {
                tmp_string = g_strdup_printf(_("Connected to %s (%s) - %d MB free"),
                        DeviceMgr.devicename->str,
                        DeviceMgr.devicestorage->StorageDescription,
                        (int) (DeviceMgr.devicestorage->FreeSpaceInBytes / MEGABYTE));
            } else {
                tmp_string = g_strdup_printf(_("Connected to %s - %d MB free"),
                        DeviceMgr.devicename->str,
                        (int) (DeviceMgr.devicestorage->FreeSpaceInBytes / MEGABYTE));
            }
        }
        statusBarSet(tmp_string);
        g_free(tmp_string);

        // Now update the filemenu;
        menuText = gtk_bin_get_child(GTK_BIN(fileConnect));
        gtk_label_set_text(GTK_LABEL(menuText), _("Disconnect Device"));

        // Enable the Drag'n'Drop interface for the main window and folder window.
        gmtp_drag_dest_set(scrolledwindowMain);
        gmtp_drag_dest_set(treeviewFolders);

    } else {

        // Update the toolbar to show the Connect String.
        gtk_tool_button_set_label(GTK_TOOL_BUTTON(toolbuttonConnect), _("Connect"));

        // Now update the status bar;
        statusBarSet(_("No device attached"));

        // Now update the filemenu;
        menuText = gtk_bin_get_child(GTK_BIN(fileConnect));
        gtk_label_set_text(GTK_LABEL(menuText), _("Connect Device"));

        // Now update the file list area and disable Drag'n'Drop.
        fileListClear();
        folderListClear();
        gtk_drag_dest_unset(scrolledwindowMain);
        gtk_drag_dest_unset(treeviewFolders);
        setWindowTitle(NULL);
        // Hide the find toolbar if open and force search mode to false.
        gtk_widget_hide(findToolbar);
        gtk_widget_set_sensitive(GTK_WIDGET(cfileAdd), TRUE);
        gtk_widget_set_sensitive(GTK_WIDGET(cfileNewFolder), TRUE);
        inFindMode = FALSE;
    }

    // Update the Toolbar and Menus enabling/disabling the menu items.
    SetToolbarButtonState(DeviceMgr.deviceConnected);
} // on_deviceConnect_activate()

// Here is the preferences callbacks and dialog box creation.
// ************************************************************************************************

/**
 * Callback to show the Preferences Dialog Box.
 * @param menuitem
 * @param user_data
 */
void on_preferences1_activate(GtkMenuItem *menuitem, gpointer user_data) {
    // No idea how this could come about, but we should take it into account so we don't have a memleak
    // due to recreating the window multiple times.
    if (windowPrefsDialog != NULL) {
        gtk_widget_hide(windowPrefsDialog);
        gtk_widget_destroy(windowPrefsDialog);
    }

    // Create and display the dialog
    windowPrefsDialog = create_windowPreferences();
    gtk_widget_show(GTK_WIDGET(windowPrefsDialog));

} // end on_preferences1_activate()

// ************************************************************************************************

/**
 * Callback to close the Preferences Dialog Box.
 * @param menuitem
 * @param user_data
 */
void on_quitPrefs_activate(GtkMenuItem *menuitem, gpointer user_data) {
    gtk_widget_hide(windowPrefsDialog);
    gtk_widget_destroy(windowPrefsDialog);
    windowPrefsDialog = NULL;
} // end on_quitPrefs_activate()

// ************************************************************************************************

/**
 * Callback for Auto Connect Device toggle in Preferences Dialog Box.
 * @param menuitem
 * @param user_data
 */
void on_PrefsDevice_activate(GtkMenuItem *menuitem, gpointer user_data) {
    gboolean state = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(checkbuttonDeviceConnect));

#if GMTP_USE_GTK2
    if (gconfconnect != NULL)
        gconf_client_set_bool(gconfconnect, "/apps/gMTP/autoconnectdevice", state, NULL);
#else
    if (gsettings_connect != NULL)
        g_settings_set_boolean(gsettings_connect, "autoconnectdevice", state);
    g_settings_sync();
#endif
} // end on_PrefsDevice_activate()

// ************************************************************************************************

/**
 * Callback for Confirm Delete Operations toggle in Preferences Dialog Box.
 * @param menuitem
 * @param user_data
 */
void on_PrefsConfirmDelete_activate(GtkMenuItem *menuitem, gpointer user_data) {
    gboolean state = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(checkbuttonConfirmFileOp));

#if GMTP_USE_GTK2
    if (gconfconnect != NULL)
        gconf_client_set_bool(gconfconnect, "/apps/gMTP/confirmFileDelete", state, NULL);
#else
    if (gsettings_connect != NULL)
        g_settings_set_boolean(gsettings_connect, "confirmfiledelete", state);
    g_settings_sync();
#endif
} // end on_PrefsConfirmDelete_activate()

// ************************************************************************************************

/**
 * Callback for Ask Download Path Operations toggle in Preferences Dialog Box.
 * @param menuitem
 * @param user_data
 */
void on_PrefsAskDownload_activate(GtkMenuItem *menuitem, gpointer user_data) {
    gboolean state = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(checkbuttonDownloadPath));

#if GMTP_USE_GTK2
    if (gconfconnect != NULL)
        gconf_client_set_bool(gconfconnect, "/apps/gMTP/promptDownloadPath", state, NULL);
#else
    if (gsettings_connect != NULL)
        g_settings_set_boolean(gsettings_connect, "promptdownloadpath", state);
    g_settings_sync();
#endif
} // end on_PrefsAskDownload_activate()

// ************************************************************************************************

/**
 * Callback for Ask Download Path Operations toggle in Preferences Dialog Box.
 * @param menuitem
 * @param user_data
 */
void on_PrefsAutoAddTrackPlaylist_activate(GtkMenuItem *menuitem, gpointer user_data) {
    gboolean state = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(checkbuttonAutoAddTrackPlaylist));

#if GMTP_USE_GTK2
    if (gconfconnect != NULL)
        gconf_client_set_bool(gconfconnect, "/apps/gMTP/autoAddTrackPlaylist", state, NULL);
#else
    if (gsettings_connect != NULL)
        g_settings_set_boolean(gsettings_connect, "autoaddtrackplaylist", state);
    g_settings_sync();
#endif
} // end on_PrefsAutoAddTrackPlaylist_activate()

// ************************************************************************************************

/**
 * Callback for Ignore Path in Playlist toggle in Preferences Dialog Box.
 * @param menuitem
 * @param user_data
 */
void on_PrefsIgnorePathInPlaylist_activate(GtkMenuItem *menuitem, gpointer user_data) {
    gboolean state = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(checkbuttonIgnorePathInPlaylist));

#if GMTP_USE_GTK2
    if (gconfconnect != NULL)
        gconf_client_set_bool(gconfconnect, "/apps/gMTP/ignorepathinplaylist", state, NULL);
#else
    if (gsettings_connect != NULL)
        g_settings_set_boolean(gsettings_connect, "ignorepathinplaylist", state);
    g_settings_sync();
#endif
} // end on_PrefsIgnorePathInPlaylist_activate()

// ************************************************************************************************

/**
 * Callback for Confirm Overwrite of File Operations toggle in Preferences Dialog Box.
 * @param menuitem
 * @param user_data
 */
void on_PrefsConfirmOverWriteFileOp_activate(GtkMenuItem *menuitem, gpointer user_data) {
    gboolean state = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(checkbuttonConfirmOverWriteFileOp));

#if GMTP_USE_GTK2
    if (gconfconnect != NULL)
        gconf_client_set_bool(gconfconnect, "/apps/gMTP/promptOverwriteFile", state, NULL);
#else
    if (gsettings_connect != NULL)
        g_settings_set_boolean(gsettings_connect, "promptoverwritefile", state);
    g_settings_sync();
#endif
} // end on_PrefsConfirmOverWriteFileOp_activate()

// ************************************************************************************************

/**
 * Callback for setting download path in Preferences Dialog Box.
 * @param menuitem
 * @param user_data
 */
void on_PrefsDownloadPath_activate(GtkMenuItem *menuitem, gpointer user_data) {
    // What we do here is display a find folder dialog, and save the resulting folder into the text wigdet and preferences item.
    gchar *savepath = NULL;
    GtkWidget *FileDialog;
    // First of all, lets set the download path.
    FileDialog = gtk_file_chooser_dialog_new(_("Select Path to Download to"),
            GTK_WINDOW(windowPrefsDialog), GTK_FILE_CHOOSER_ACTION_SELECT_FOLDER,
            GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
            GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT,
            NULL);
    gtk_file_chooser_set_filename(GTK_FILE_CHOOSER(FileDialog), Preferences.fileSystemDownloadPath->str);
    if (gtk_dialog_run(GTK_DIALOG(FileDialog)) == GTK_RESPONSE_ACCEPT) {
        savepath = gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(FileDialog));

        // Save our download path.
#if GMTP_USE_GTK2
        if (gconfconnect != NULL)
            gconf_client_set_string(gconfconnect, "/apps/gMTP/DownloadPath", savepath, NULL);
#else
        if (gsettings_connect != NULL)
            g_settings_set_string(gsettings_connect, "downloadpath", savepath);
        g_settings_sync();
#endif
        g_free(savepath);
    }
    gtk_widget_destroy(FileDialog);
} // on_PrefsDownloadPath_activate()

// ************************************************************************************************

/**
 * Callback for setting upload path in Preferences Dialog Box.
 * @param menuitem
 * @param user_data
 */
void on_PrefsUploadPath_activate(GtkMenuItem *menuitem, gpointer user_data) {
    // What we do here is display a find folder dialog, and save the resulting folder into the text wigdet and preferences item.
    gchar *savepath = NULL;
    GtkWidget *FileDialog;
    // First of all, lets set the upload path.
    FileDialog = gtk_file_chooser_dialog_new(_("Select Path to Upload From"),
            GTK_WINDOW(windowPrefsDialog), GTK_FILE_CHOOSER_ACTION_SELECT_FOLDER,
            GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
            GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT,
            NULL);
    gtk_file_chooser_set_filename(GTK_FILE_CHOOSER(FileDialog), Preferences.fileSystemUploadPath->str);
    if (gtk_dialog_run(GTK_DIALOG(FileDialog)) == GTK_RESPONSE_ACCEPT) {
        savepath = gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(FileDialog));

        // Save our download path.
#if GMTP_USE_GTK2
        if (gconfconnect != NULL)
            gconf_client_set_string(gconfconnect, "/apps/gMTP/UploadPath", savepath, NULL);
#else
        if (gsettings_connect != NULL)
            g_settings_set_string(gsettings_connect, "uploadpath", savepath);
        g_settings_sync();
#endif
        g_free(savepath);
    }
    gtk_widget_destroy(FileDialog);
} //on_PrefsUploadPath_activate()

// ************************************************************************************************

/**
 * Callback to handle double click on item in main window. If it's a folder, then change to it,
 * other attempt to download the file(s).
 * @param treeview
 * @param path
 * @param column
 * @param data
 */
void fileListRowActivated(GtkTreeView *treeview, GtkTreePath *path, GtkTreeViewColumn *column, gpointer data) {
    GtkTreeModel *model;
    GtkTreeModel *sortmodel;
    GtkTreeIter iter;

    gchar *filename = NULL;
    gboolean isFolder;
    uint32_t objectID;

    GtkWidget *FileDialog;
    gchar *savepath = NULL;

    // Obtain the iter, and the related objectID.
    sortmodel = gtk_tree_view_get_model(treeview);
    model = gtk_tree_model_sort_get_model(GTK_TREE_MODEL_SORT(sortmodel));
    if (gtk_tree_model_get_iter(model, &iter, gtk_tree_model_sort_convert_path_to_child_path(GTK_TREE_MODEL_SORT(sortmodel), path))) {
        gtk_tree_model_get(GTK_TREE_MODEL(fileList), &iter, COL_ISFOLDER, &isFolder, COL_FILENAME_ACTUAL, &filename, COL_FILEID, &objectID, -1);
        if (isFolder == FALSE) {
            // Now download the actual file from the MTP device.

            savepath = g_malloc0(8192);
            // Let's confirm our download path.
            if (Preferences.ask_download_path == TRUE) {
                FileDialog = gtk_file_chooser_dialog_new(_("Select Path to Download"),
                        GTK_WINDOW(windowMain), GTK_FILE_CHOOSER_ACTION_SELECT_FOLDER,
                        GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
                        GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT,
                        NULL);

                gtk_file_chooser_set_filename(GTK_FILE_CHOOSER(FileDialog), Preferences.fileSystemDownloadPath->str);
                if (gtk_dialog_run(GTK_DIALOG(FileDialog)) == GTK_RESPONSE_ACCEPT) {
                    savepath = gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(FileDialog));
                    // Save our download path.
                    Preferences.fileSystemDownloadPath = g_string_assign(Preferences.fileSystemDownloadPath, savepath);
                    // We do the deed.
                    filesDownload(filename, objectID);
                }
                gtk_widget_destroy(FileDialog);
            } else {
                // We do the deed.
                filesDownload(filename, objectID);
            }
            g_free(savepath);

        } else {
            // We have a folder so change to it?
            currentFolderID = objectID;
            on_editFindClose_activate(NULL, NULL);
        }
    }
    g_free(filename);
} // end fileListRowActivated()


// ************************************************************************************************

/**
 * Callback to handle double click on item in folder main window.
 * @param treeview
 * @param path
 * @param column
 * @param data
 */
void folderListRowActivated(GtkTreeView *treeview, GtkTreePath *path, GtkTreeViewColumn *column, gpointer data) {
    GtkTreeModel *model;
    GtkTreeModel *sortmodel;
    GtkTreeIter iter;

    uint32_t objectID;

    // Obtain the iter, and the related objectID.
    sortmodel = gtk_tree_view_get_model(treeview);
    model = gtk_tree_model_sort_get_model(GTK_TREE_MODEL_SORT(sortmodel));
    if (gtk_tree_model_get_iter(model, &iter, gtk_tree_model_sort_convert_path_to_child_path(GTK_TREE_MODEL_SORT(sortmodel), path))) {
        gtk_tree_model_get(GTK_TREE_MODEL(model), &iter, COL_FOL_ID, &objectID, -1);
        // We have a folder so change to it?
        currentFolderID = objectID;
        on_editFindClose_activate(NULL, NULL);
    }
} // end folderListRowActivated()


// ************************************************************************************************

/**
 * Callback to handle selecting NewFolder from menu or toolbar.
 * @param menuitem
 * @param user_data
 */
void on_fileNewFolder_activate(GtkMenuItem *menuitem, gpointer user_data) {
    gchar *foldername = NULL;
    if (folderListGetSelection() != -1) {
        on_folderNewFolder_activate(menuitem, user_data);
        return;
    }
    // Get the folder name by displaying a dialog.
    foldername = displayFolderNewDialog();
    if (foldername != NULL) {
        // Add in folder to MTP device.
        folderAdd(foldername);
        g_free(foldername);
        deviceRescan();
    }
} // end on_fileNewFolder_activate()


// ************************************************************************************************

/**
 * Callback to handle selecting NewFolder from menu or toolbar.
 * @param menuitem
 * @param user_data
 */
void on_folderNewFolder_activate(GtkMenuItem *menuitem, gpointer user_data) {
    gchar *foldername = NULL;
    uint32_t tmpFolderID = 0;
    // Get the folder name by displaying a dialog.
    foldername = displayFolderNewDialog();
    if (foldername != NULL) {
        // Let's see if we have anything selected in the folder view, and if not, then we add the
        // folder to the current Folder.
        if (gtk_tree_selection_count_selected_rows(folderSelection) == 0) {
            // Add in folder to MTP device.
            folderAdd(foldername);
        } else {
            // We have selected a folder in the folder view, so let's get it's ID.
            tmpFolderID = currentFolderID;
            currentFolderID = folderListGetSelection();
            folderAdd(foldername);
            currentFolderID = tmpFolderID;
        }
        g_free(foldername);
        deviceRescan();
    }
} // end on_folderNewFolder_activate()


// ************************************************************************************************

/**
 * Callback to handle selecting RemoveFolder from menu or toolbar.
 * @param menuitem
 * @param user_data
 */
void on_folderRemoveFolder_activate(GtkMenuItem *menuitem, gpointer user_data) {
    GtkWidget *dialog;
    GtkTreeModel *sortmodel;
    GtkTreeIter iter;
    GtkTreeIter childiter;

    uint32_t objectID;

    // Let's see if we have anything selected in the folder view, and if not let the user know, and return
    if (gtk_tree_selection_count_selected_rows(folderSelection) == 0) {
        // Add in folder to MTP device.
        displayInformation(_("No files/folders selected?"));
        return;
    } else {
        // We have selected a folder in the folder view, so let's get it's ID.
        sortmodel = gtk_tree_view_get_model(GTK_TREE_VIEW(treeviewFiles));
        gtk_tree_selection_get_selected(folderSelection, &sortmodel, &iter);
        gtk_tree_model_sort_convert_iter_to_child_iter(GTK_TREE_MODEL_SORT(sortmodel), &childiter, &iter);
        gtk_tree_model_get(GTK_TREE_MODEL(folderList), &childiter, COL_FOL_ID, &objectID, -1);

        // Now we prompt to confirm delete?
        if (Preferences.confirm_file_delete_op == FALSE) {
            // Now download the actual file from the MTP device.
            folderDelete(getCurrentFolderPtr(deviceFolders, objectID), 0);
        } else {
            dialog = gtk_message_dialog_new(GTK_WINDOW(windowMain),
                    GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
                    GTK_MESSAGE_WARNING,
                    GTK_BUTTONS_YES_NO,
                    _("Are you sure you want to delete this folder (and all contents)?"));
            gtk_window_set_title(GTK_WINDOW(dialog), _("Confirm Delete"));
            gint result = gtk_dialog_run(GTK_DIALOG(dialog));
            if (result == GTK_RESPONSE_YES)
                folderDelete(getCurrentFolderPtr(deviceFolders, objectID), 0);
            gtk_widget_destroy(dialog);
        }

        //folderDelete(getCurrentFolderPtr(deviceFolders, objectID), 0);
    }
    deviceRescan();
} // end on_folderRemoveFolder_activate()


// ************************************************************************************************

/**
 * Callback to handle selecting MoveFolder from context menu.
 * @param menuitem
 * @param user_data
 */
void on_folderMoveFolder_activate(GtkMenuItem *menuitem, gpointer user_data) {
    GtkTreeModel *sortmodel;
    GtkTreeIter iter;
    GtkTreeIter childiter;
    int64_t targetfolder = 0;
    uint32_t objectID;

    LIBMTP_folder_t *currentFolder = NULL;
    LIBMTP_folder_t *newFolder = NULL;
    int error;

    // Let's see if we have anything selected in the folder view, and if not let the user know, and return
    if (gtk_tree_selection_count_selected_rows(folderSelection) == 0) {
        // Add in folder to MTP device.
        displayInformation(_("No files/folders selected?"));
        return;
    } else {
        // We have selected a folder in the folder view, so let's get it's ID.
        sortmodel = gtk_tree_view_get_model(GTK_TREE_VIEW(treeviewFiles));
        gtk_tree_selection_get_selected(folderSelection, &sortmodel, &iter);
        gtk_tree_model_sort_convert_iter_to_child_iter(GTK_TREE_MODEL_SORT(sortmodel), &childiter, &iter);
        gtk_tree_model_get(GTK_TREE_MODEL(folderList), &childiter, COL_FOL_ID, &objectID, -1);

        // Prompt for the target folder location.
        targetfolder = getTargetFolderLocation();
        if ((targetfolder == -1)) {
            // If the user didn't select a folder, or the target folder is the current selected folder
            // then do nothing.
            return;
        }
        fileMoveTargetFolder = targetfolder;
        gtk_tree_selection_unselect_all(folderSelection);

        // Make sure we don't want to move the folder into itself?
        if (objectID == fileMoveTargetFolder) {
            displayError(_("Unable to move the selected folder into itself?\n"));
            g_fprintf(stderr, _("Unable to move the selected folder into itself?\n"));
            return;
        }
        // We have the target folder, so let's check to ensure that we will not create a circular
        // reference by moving a folder underneath it self.
        currentFolder = getCurrentFolderPtr(deviceFolders, objectID);
        if (currentFolder == NULL) {
            // WTF?
            g_fprintf(stderr, "File Move Error: Can't get current folder pointer\n");
            return;
        }
        // Use currentFolder as the starting point, and simply attempt to get the ptr to the new
        // folder based on this point.
        newFolder = getCurrentFolderPtr(currentFolder->child, fileMoveTargetFolder);
        if (newFolder == NULL) {
            // We are alright to proceed.
            if ((error = setNewParentFolderID(objectID, fileMoveTargetFolder)) != 0) {
                displayError(_("Unable to move the selected folder?\n"));
                g_fprintf(stderr, "File Move Error: %d\n", error);
                LIBMTP_Dump_Errorstack(DeviceMgr.device);
                LIBMTP_Clear_Errorstack(DeviceMgr.device);
            }
        } else {
            displayError(_("Unable to move the selected folder underneath itself?\n"));
            g_fprintf(stderr, _("Unable to move the selected folder underneath itself?\n"));
        }


    }
    deviceRescan();
} // end on_folderMoveFolder_activate()


// ************************************************************************************************

/**
 * Callback to handle selecting Rename Folder from menu
 * @param menuitem
 * @param user_data
 */
void on_folderRenameFolder_activate(GtkMenuItem *menuitem, gpointer user_data) {
    gchar *newfilename = NULL;
    gchar *filename = NULL;
    GtkTreeModel *sortmodel;
    GtkTreeIter iter;
    GtkTreeIter childiter;

    uint32_t objectID;

    // Let's see if we have anything selected in the folder view, and if not let the user know, and return
    if (gtk_tree_selection_count_selected_rows(folderSelection) == 0) {
        // Add in folder to MTP device.
        displayInformation(_("No files/folders selected?"));
        return;
    } else {
        // We have selected a folder in the folder view, so let's get it's ID.
        sortmodel = gtk_tree_view_get_model(GTK_TREE_VIEW(treeviewFiles));
        gtk_tree_selection_get_selected(folderSelection, &sortmodel, &iter);
        gtk_tree_model_sort_convert_iter_to_child_iter(GTK_TREE_MODEL_SORT(sortmodel), &childiter, &iter);
        gtk_tree_model_get(GTK_TREE_MODEL(folderList), &childiter, COL_FOL_ID, &objectID,
                COL_FOL_NAME_HIDDEN, &filename, -1);

        // Get our new folder name.
        newfilename = displayRenameFileDialog(filename);

        // If the user supplied something, then update the name of the device.
        if (newfilename != NULL) {
            filesRename(newfilename, objectID);
            g_free(newfilename);
            deviceRescan();
        }
    }
} // end on_folderRenameFolder_activate()


// ************************************************************************************************

/**
 * Callback handle to handle deleting a folder menu option.
 * @param menuitem
 * @param user_data
 */
void on_fileRemoveFolder_activate(GtkMenuItem *menuitem, gpointer user_data) {
    GtkWidget *dialog;

    // Let's check to see if we have anything selected in our treeview?
    if (fileListGetSelection() == NULL) {
        if (folderListGetSelection() != -1) {
            on_folderRemoveFolder_activate(menuitem, user_data);
        } else {
            displayInformation(_("No files/folders selected?"));
        }
        return;
    }

    // Now we prompt to confirm delete?
    if (Preferences.confirm_file_delete_op == FALSE) {
        // Now download the actual file from the MTP device.
        folderListRemove(fileListGetSelection());
    } else {
        dialog = gtk_message_dialog_new(GTK_WINDOW(windowMain),
                GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
                GTK_MESSAGE_WARNING,
                GTK_BUTTONS_YES_NO,
                _("Are you sure you want to delete this folder (and all contents)?"));
        gtk_window_set_title(GTK_WINDOW(dialog), _("Confirm Delete"));
        gint result = gtk_dialog_run(GTK_DIALOG(dialog));
        if (result == GTK_RESPONSE_YES)
            folderListRemove(fileListGetSelection());
        gtk_widget_destroy(dialog);
    }
} // end on_fileRemoveFolder_activate()

// ************************************************************************************************

/**
 * Callback to handle the Rename Device menu option.
 * @param menuitem
 * @param user_data
 */
void on_editDeviceName_activate(GtkMenuItem *menuitem, gpointer user_data) {
    gchar *devicename = NULL;
    gchar *tmp_string = NULL;

    // Get our new device name.
    devicename = displayChangeDeviceNameDialog(DeviceMgr.devicename->str);

    // If the user supplied something, then update the name of the device.
    if (devicename != NULL) {
        // add change to MTP device.
        setDeviceName(devicename);
        g_free(devicename);
        // Attempt to read it back as confirmation that something may of happened.
        tmp_string = LIBMTP_Get_Friendlyname(DeviceMgr.device);
        if (tmp_string == NULL) {
            DeviceMgr.devicename = g_string_new(_("N/A"));
        } else {
            DeviceMgr.devicename = g_string_new(tmp_string);
            g_free(tmp_string);
        }
        // Perform a device Rescan operation to reset all device parameters.
        deviceRescan();
    }
} // end on_editDeviceName_activate()

// ************************************************************************************************

/**
 * Callback to format the current storage device.
 * @param menuitem
 * @param user_data
 */
void on_editFormatDevice_activate(GtkMenuItem *menuitem, gpointer user_data) {
    GtkWidget *dialog;
    dialog = gtk_message_dialog_new(GTK_WINDOW(windowMain),
            GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
            GTK_MESSAGE_WARNING,
            GTK_BUTTONS_YES_NO,
            _("Are you sure you want to format this device?"));
    gtk_window_set_title(GTK_WINDOW(dialog), _("Format Device"));
    gint result = gtk_dialog_run(GTK_DIALOG(dialog));
    gtk_widget_hide(GTK_WIDGET(dialog));
    gtk_widget_destroy(dialog);

    if (result == GTK_RESPONSE_YES) {
        dialog = create_windowFormat();
        // Show progress dialog.
        gtk_widget_show_all(dialog);
        // Ensure GTK redraws the window.

        formatThreadWorking = TRUE;

        g_thread_create((GThreadFunc) on_editFormatDevice_thread, NULL, FALSE, NULL);

        while (formatThreadWorking) {
            while (gtk_events_pending())
                gtk_main_iteration();

            if (formatDialog_progressBar1 != NULL) {
                gtk_progress_bar_pulse(GTK_PROGRESS_BAR(formatDialog_progressBar1));
                g_usleep(G_USEC_PER_SEC * 0.1);
            }

        }
        // The worker thread has finished so let's continue.

        // Disconnect and reconnect the device.
        on_deviceConnect_activate(NULL, NULL);
        // Sleep for 2 secs to allow the device to settle itself
        g_usleep(G_USEC_PER_SEC * 2);
        on_deviceConnect_activate(NULL, NULL);
        // Close progress dialog.
        gtk_widget_hide(dialog);
        gtk_widget_destroy(dialog);
        formatDialog_progressBar1 = NULL;
    }
    //
} // end on_editFormatDevice_activate()

// ************************************************************************************************

/**
 * Worker thread for on_editFormatDevice_activate();
 */
void on_editFormatDevice_thread(void) {
    formatStorageDevice();
    // Add a 5 sec wait so the device has time to settle itself.
    g_usleep(G_USEC_PER_SEC * 5);
    formatThreadWorking = FALSE;
    g_thread_exit(NULL);
}

// ************************************************************************************************

/**
 * Callback to handle the displaying of the context menu.
 * @param widget
 * @param event
 * @return
 */
gboolean on_windowMainContextMenu_activate(GtkWidget *widget, GdkEvent *event) {
    GtkMenu *menu;
    GdkEventButton *event_button;
    g_return_val_if_fail(widget != NULL, FALSE);
    g_return_val_if_fail(GTK_IS_MENU(widget), FALSE);
    g_return_val_if_fail(event != NULL, FALSE);

    /* The "widget" is the menu that was supplied when
     * g_signal_connect_swapped() was called.
     */
    menu = GTK_MENU(widget);
    if (event->type == GDK_BUTTON_PRESS) {
        event_button = (GdkEventButton *) event;
        if (event_button->button == 3) {
            gtk_menu_popup(menu, NULL, NULL, NULL, NULL,
                    event_button->button, event_button->time);
            return TRUE;
        }
    }
    return FALSE;
} // end on_windowMainContextMenu_activate()


// ************************************************************************************************

/**
 * Callback to handle the displaying of the context menu.
 * @param widget
 * @param event
 * @return
 */
gboolean on_windowViewContextMenu_activate(GtkWidget *widget, GdkEvent *event) {
    GtkMenu *menu;
    GdkEventButton *event_button;
    g_return_val_if_fail(event != NULL, FALSE);

    /* The "widget" is the menu that was supplied when
     * g_signal_connect_swapped() was called.
     */
    menu = GTK_MENU(contextMenuColumn);
    if (event->type == GDK_BUTTON_PRESS) {
        event_button = (GdkEventButton *) event;
        if (event_button->button == 3) {
            gtk_menu_popup(menu, NULL, NULL, NULL, NULL,
                    event_button->button, event_button->time);
            return TRUE;
        }
    }
    return FALSE;
} // end on_windowMainContextMenu_activate()

// ************************************************************************************************

/**
 * Callback to handle the Find menu option.
 * @param menuitem
 * @param user_data
 */
void on_editFind_activate(GtkMenuItem *menuitem, gpointer user_data) {
    if (inFindMode == FALSE) {

        gtk_widget_show(findToolbar);
        gtk_widget_hide(scrolledwindowFolders);

        fileListClear();
        //folderListClear();
        inFindMode = TRUE;
        statusBarSet(_("Please enter search item."));
        setWindowTitle(_("Search"));
        gtk_tree_view_column_set_visible(column_Location, TRUE);

        //Disable some of the menu options, while in search mode.
        gtk_widget_set_sensitive(GTK_WIDGET(cfileAdd), FALSE);
        gtk_widget_set_sensitive(GTK_WIDGET(cfileNewFolder), FALSE);
        gtk_widget_set_sensitive(GTK_WIDGET(fileAdd), FALSE);
        gtk_widget_set_sensitive(GTK_WIDGET(fileNewFolder), FALSE);
        gtk_widget_set_sensitive(GTK_WIDGET(toolbuttonAddFile), FALSE);
        gtk_widget_set_sensitive(GTK_WIDGET(menu_view_folders), FALSE);
        // Get focus on text entry box.

        gtk_widget_grab_focus(GTK_WIDGET(FindToolbar_entry_FindText));
    } else {
        on_editFindClose_activate(menuitem, user_data);
    }

} // end on_editFind_activate()


// ************************************************************************************************

/**
 * Callback to handle the Find menu option.
 * @param menuitem
 * @param user_data
 */
void on_editSelectAll_activate(GtkMenuItem *menuitem, gpointer user_data) {
    fileListSelectAll();
} // end on_editSelectAll_activate()

// ************************************************************************************************

/**
 * Callback to handle the Find toolbar close option.
 * @param menuitem
 * @param user_data
 */
void on_editFindClose_activate(GtkMenuItem *menuitem, gpointer user_data) {
    gchar* tmp_string;

    gtk_widget_hide(findToolbar);
    if (Preferences.view_folders == TRUE) {
        gtk_widget_show(scrolledwindowFolders);
    }
    fileListClear();
    inFindMode = FALSE;
    gtk_tree_view_column_set_visible(column_Location, FALSE);

    // First we clear the file and folder list...
    fileListClear();
    folderListClear();
    // Refresh the file listings.
    fileListAdd();
    folderListAdd(deviceFolders, NULL);

    // Update the status bar.
    if (DeviceMgr.storagedeviceID == MTP_DEVICE_SINGLE_STORAGE) {
        tmp_string = g_strdup_printf(_("Connected to %s - %d MB free"), DeviceMgr.devicename->str,
                (int) (DeviceMgr.devicestorage->FreeSpaceInBytes / MEGABYTE));
    } else {
        if (DeviceMgr.devicestorage->StorageDescription != NULL) {
            tmp_string = g_strdup_printf(_("Connected to %s (%s) - %d MB free"), DeviceMgr.devicename->str,
                    DeviceMgr.devicestorage->StorageDescription,
                    (int) (DeviceMgr.devicestorage->FreeSpaceInBytes / MEGABYTE));
        } else {
            tmp_string = g_strdup_printf(_("Connected to %s - %d MB free"), DeviceMgr.devicename->str,
                    (int) (DeviceMgr.devicestorage->FreeSpaceInBytes / MEGABYTE));
        }
    }
    statusBarSet(tmp_string);
    g_free(tmp_string);

    // Now clear the Search GList;
    if (searchList != NULL) {
        g_slist_foreach(searchList, (GFunc) g_free_search, NULL);
        g_slist_free(searchList);
        searchList = NULL;
    }
    //Enable some of the menu options, while in search mode.
    gtk_widget_set_sensitive(GTK_WIDGET(cfileAdd), TRUE);
    gtk_widget_set_sensitive(GTK_WIDGET(cfileNewFolder), TRUE);
    gtk_widget_set_sensitive(GTK_WIDGET(fileAdd), TRUE);
    gtk_widget_set_sensitive(GTK_WIDGET(fileNewFolder), TRUE);
    gtk_widget_set_sensitive(GTK_WIDGET(toolbuttonAddFile), TRUE);
    gtk_widget_set_sensitive(GTK_WIDGET(menu_view_folders), TRUE);

} // end on_editFindClose_activate()


// ************************************************************************************************

/**
 * Callback to handle the actual searching of files/folders.
 * @param menuitem
 * @param user_data
 */
void on_editFindSearch_activate(GtkMenuItem *menuitem, gpointer user_data) {
    gchar *searchstring = NULL;
    gboolean searchfiles = FALSE;
    gboolean searchmeta = FALSE;

    statusBarSet(_("Searching..."));
    // Now clear the Search GList;
    if (searchList != NULL) {
        g_slist_foreach(searchList, (GFunc) g_free_search, NULL);
        g_slist_free(searchList);
        searchList = NULL;
    }

    // Set to upper case to perform case insensitive searches.
    searchstring = g_utf8_strup(gtk_entry_get_text(GTK_ENTRY(FindToolbar_entry_FindText)), -1);
    searchfiles = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(FindToolbar_checkbutton_FindFiles));
    searchmeta = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(FindToolbar_checkbutton_TrackInformation));

    // Let's start our search.
    searchList = filesSearch(searchstring, searchfiles, searchmeta);
    inFindMode = TRUE;
    fileListClear();
    fileListAdd();
    g_free(searchstring);
} // end on_editFindSearch_activate()

// ************************************************************************************************

/**
 * Callback to handle the Add Album Art menu option.
 * @param menuitem
 * @param user_data
 */
void on_editAddAlbumArt_activate(GtkMenuItem *menuitem, gpointer user_data) {
    // Get a filename of the album art.
    displayAddAlbumArtDialog();
} // end on_editAddAlbumArt_activate()

// ************************************************************************************************

/**
 * Callback to hanlde the Playlist menu/toolbar operations.
 * @param menuitem
 * @param user_data
 */
void on_editPlaylist_activate(GtkMenuItem *menuitem, gpointer user_data) {
    displayPlaylistDialog();
} // end on_editPlaylist_activate()

// ************************************************************************************************

/**
 * Callback to handle the select file button in the Add Album Art Dialog box.
 * @param button
 * @param user_data
 */
void on_buttonAlbumArtAdd_activate(GtkWidget *button, gpointer user_data) {
    // What we do here is display a find folder dialog, and save the resulting folder into the text wigdet and preferences item.
    //gchar *filename;
    gchar *filename = NULL;
    GtkWidget *FileDialog;
    FileDialog = gtk_file_chooser_dialog_new(_("Select Album Art File"),
            GTK_WINDOW(AlbumArtDialog), GTK_FILE_CHOOSER_ACTION_OPEN,
            GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
            GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT,
            NULL);

    gtk_file_chooser_set_local_only(GTK_FILE_CHOOSER(FileDialog), TRUE);

    // Set the default path to be the normal upload folder.
    gtk_file_chooser_set_current_folder(GTK_FILE_CHOOSER(FileDialog), Preferences.fileSystemUploadPath->str);

    if (gtk_dialog_run(GTK_DIALOG(FileDialog)) == GTK_RESPONSE_ACCEPT) {
        filename = gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(FileDialog));

        if (filename != NULL) {
            // Upload the file to the selected album.
            gint selected = gtk_combo_box_get_active(GTK_COMBO_BOX(textboxAlbumArt));
            gint count = 0;
            LIBMTP_album_t *albumlist = LIBMTP_Get_Album_List_For_Storage(DeviceMgr.device, DeviceMgr.devicestorage->id);
            LIBMTP_album_t *albuminfo = albumlist;

            while (albuminfo != NULL) {
                if (count == selected) {
                    // Found our album, so update the image on the device, then update the display.
                    albumAddArt(albuminfo->album_id, filename);
                    AlbumArtUpdateImage(albuminfo);
                    clearAlbumStruc(albumlist);
                    g_free(filename);
                    gtk_widget_destroy(FileDialog);
                    return;
                }
                // Next album_entry
                albuminfo = albuminfo->next;
                count++;
            }
            // Set a default image as we didn't find our album.
            AlbumArtUpdateImage(NULL);
            clearAlbumStruc(albumlist);

            g_free(filename);
        }
    }
    gtk_widget_destroy(FileDialog);
} // end on_buttonAlbumArtAdd_activate()

// ************************************************************************************************

/**
 * Callback to handle removal of associated album art.
 * @param button
 * @param user_data
 */
void on_buttonAlbumArtDelete_activate(GtkWidget *button, gpointer user_data) {

    // Send a blank representation.
    gint selected = gtk_combo_box_get_active(GTK_COMBO_BOX(textboxAlbumArt));
    gint count = 0;
    LIBMTP_album_t *albumlist = LIBMTP_Get_Album_List_For_Storage(DeviceMgr.device, DeviceMgr.devicestorage->id);
    LIBMTP_album_t *albuminfo = albumlist;

    while (albuminfo != NULL) {
        if (count == selected) {
            // Found our album, so update the image on the device, then update the display.
            albumDeleteArt(albuminfo->album_id);
            AlbumArtUpdateImage(NULL);
            clearAlbumStruc(albumlist);
            return;
        }
        // Next album_entry
        albuminfo = albuminfo->next;
        count++;
    }
    // Set a default image as we didn't find our album.
    AlbumArtUpdateImage(NULL);
    clearAlbumStruc(albumlist);

} // end on_buttonAlbumArtDelete_activate()

// ************************************************************************************************

/**
 * Retrieve the album art and attempt to save the file.
 * @param button
 * @param user_data
 */
void on_buttonAlbumArtDownload_activate(GtkWidget *button, gpointer user_data) {
    FILE* fd;
    gint selected = gtk_combo_box_get_active(GTK_COMBO_BOX(textboxAlbumArt));
    gint count = 0;
    GtkWidget *FileDialog = NULL;
    gchar *filename = NULL;
    LIBMTP_filesampledata_t *imagedata = NULL;
    LIBMTP_album_t *albumlist = LIBMTP_Get_Album_List_For_Storage(DeviceMgr.device, DeviceMgr.devicestorage->id);
    LIBMTP_album_t *albuminfo = albumlist;

    // Scan our albums, looking for the correct one.
    while (albuminfo != NULL) {
        if (count == selected) {
            // Found our album, let's get our data..
            imagedata = albumGetArt(albuminfo);
            if (imagedata != NULL) {

                FileDialog = gtk_file_chooser_dialog_new(_("Save Album Art File"),
                        GTK_WINDOW(AlbumArtDialog), GTK_FILE_CHOOSER_ACTION_SAVE,
                        GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
                        GTK_STOCK_SAVE, GTK_RESPONSE_ACCEPT,
                        NULL);

                gtk_file_chooser_set_local_only(GTK_FILE_CHOOSER(FileDialog), TRUE);

                // Set the default path to be the normal download folder.
                gtk_file_chooser_set_current_folder(GTK_FILE_CHOOSER(FileDialog),
                        Preferences.fileSystemDownloadPath->str);

                // Set a default name to be the album.JPG
                gtk_file_chooser_set_current_name(GTK_FILE_CHOOSER(FileDialog),
                        g_strdup_printf("%s.jpg", albuminfo->name));

                if (gtk_dialog_run(GTK_DIALOG(FileDialog)) == GTK_RESPONSE_ACCEPT) {
                    filename = gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(FileDialog));
                    if (filename != NULL) {
                        // The user has selected a file to save as, so do the deed.
                        fd = fopen(filename, "w");
                        if (fd == NULL) {
                            g_fprintf(stderr, _("Couldn't save image file %s\n"), filename);
                            displayError(_("Couldn't save image file\n"));
                        } else {
                            fwrite(imagedata->data, imagedata->size, 1, fd);
                            fclose(fd);
                        }
                        g_free(filename);
                    }
                }
                // Clean up our image data and dialog.
                LIBMTP_destroy_filesampledata_t(imagedata);
                gtk_widget_destroy(FileDialog);
            }
            clearAlbumStruc(albumlist);
            return;
        }
        // Next album_entry
        albuminfo = albuminfo->next;
        count++;
    }
    // Set a default image as we didn't find our album.
    clearAlbumStruc(albumlist);
    gtk_widget_destroy(FileDialog);
} // end on_buttonAlbumArtDownload_activate()

// ************************************************************************************************

/**
 * Update the Album Image in the Add Album Art Dialog Box.
 * @param menuitem
 * @param user_data
 */
void on_albumtextbox_activate(GtkComboBox *combobox, gpointer user_data) {
    gint selected = gtk_combo_box_get_active(combobox);
    gint count = 0;
    LIBMTP_album_t *albumlist = LIBMTP_Get_Album_List_For_Storage(DeviceMgr.device, DeviceMgr.devicestorage->id);
    LIBMTP_album_t *albuminfo = albumlist;

    while (albuminfo != NULL) {
        if (count == selected) {
            AlbumArtUpdateImage(albuminfo);
            clearAlbumStruc(albumlist);
            return;
        }
        // Text the album_entry
        albuminfo = albuminfo->next;
        count++;
    }
    // Set a default image
    AlbumArtUpdateImage(NULL);
    clearAlbumStruc(albumlist);
}

// Playlist Callbacks.
// ************************************************************************************************

/**
 * Callback to handle closing the Playlist editor dialog.
 * @param menuitem
 * @param user_data
 */
void on_quitPlaylist_activate(GtkMenuItem *menuitem, gpointer user_data) {
    // Save our current selected playlist!
    if (devicePlayLists != NULL)
        playlist_SavePlaylist(playlist_number);
    // Kill our window
    gtk_widget_hide(windowPlaylistDialog);
    gtk_widget_destroy(windowPlaylistDialog);
    windowPlaylistDialog = NULL;
    // Do a device rescan to show the new playlists in the file window
    deviceRescan();
} // end on_quitPlaylist_activate()

// ************************************************************************************************

/**
 * Callback to handle the new Playlist button in the Playlist editor dialog.
 * @param menuitem
 * @param user_data
 */
void on_Playlist_NewPlaylistButton_activate(GtkMenuItem *menuitem, gpointer user_data) {
    //g_printf("Clicked on new playlist button\n");
    gchar *playlistname = NULL;

    // Save our current selected playlist!
    if (devicePlayLists != NULL)
        playlist_SavePlaylist(playlist_number);

    playlistname = displayPlaylistNewDialog();
    if (playlistname != NULL) {
        // Add in playlist to MTP device.
        playlistAdd(playlistname);
        // Refresh our playlist information.
        devicePlayLists = getPlaylists();
        gtk_list_store_clear(GTK_LIST_STORE(playlist_PL_List));
        // Add it to our combobox

#if GMTP_USE_GTK2
        gtk_combo_box_append_text(GTK_COMBO_BOX(comboboxentry_playlist), g_strdup(playlistname));
#else
        gtk_combo_box_text_append_text(GTK_COMBO_BOX_TEXT(comboboxentry_playlist), g_strdup(playlistname));
#endif
        g_free(playlistname);

        // Set the active combobox item.
        comboboxentry_playlist_entries++;
        playlist_number = comboboxentry_playlist_entries - 1;
        gtk_combo_box_set_active(GTK_COMBO_BOX(comboboxentry_playlist), comboboxentry_playlist_entries - 1);
        SetPlaylistButtonState(TRUE);
        setPlaylistField(playlist_number);
    }
} // end on_Playlist_NewPlaylistButton_activate()

// ************************************************************************************************

/**
 * Callback to handle the Import Playlist button in the Playlist editor dialog.
 * @param menuitem
 * @param user_data
 */
void on_Playlist_ImportPlaylistButton_activate(GtkMenuItem *menuitem, gpointer user_data) {
    //g_printf("Clicked on new playlist button\n");
    gchar *playlistfilename = NULL;
    gchar *playlistname = NULL;
    GtkWidget *FileDialog;
    GtkFileFilter *OpenFormFilter, *OpenFormFilter2;

    // Save our current selected playlist!
    if (devicePlayLists != NULL)
        playlist_SavePlaylist(playlist_number);

    // Get our filename...

    FileDialog = gtk_file_chooser_dialog_new(_("Select Playlist to Import"),
            GTK_WINDOW(windowMain), GTK_FILE_CHOOSER_ACTION_OPEN,
            GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
            GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT,
            NULL);
    gtk_file_chooser_set_select_multiple(GTK_FILE_CHOOSER(FileDialog), FALSE);
    OpenFormFilter = gtk_file_filter_new();
    gtk_file_filter_add_pattern(OpenFormFilter, "*.m3u");
    gtk_file_filter_set_name(OpenFormFilter, "m3u Playlists");
    gtk_file_chooser_add_filter(GTK_FILE_CHOOSER(FileDialog), OpenFormFilter);
    OpenFormFilter2 = gtk_file_filter_new();
    gtk_file_filter_add_pattern(OpenFormFilter2, "*");
    gtk_file_filter_set_name(OpenFormFilter2, "All Files");
    gtk_file_chooser_add_filter(GTK_FILE_CHOOSER(FileDialog), OpenFormFilter2);

    if (gtk_dialog_run(GTK_DIALOG(FileDialog)) == GTK_RESPONSE_ACCEPT) {
        playlistfilename = gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(FileDialog));
    }

    gtk_widget_hide(FileDialog);
    gtk_widget_destroy(FileDialog);

    if (playlistfilename != NULL) {
        // Add in playlist to MTP device.
        playlistname = playlistImport(playlistfilename);

        // If our name is NULL, then the import failed...
        if (playlistname != NULL) {
            // Refresh our playlist information.
            devicePlayLists = getPlaylists();
            gtk_list_store_clear(GTK_LIST_STORE(playlist_PL_List));
            // Add it to our combobox

#if GMTP_USE_GTK2
            gtk_combo_box_append_text(GTK_COMBO_BOX(comboboxentry_playlist), g_strdup(playlistname));
#else
            gtk_combo_box_text_append_text(GTK_COMBO_BOX_TEXT(comboboxentry_playlist), g_strdup(playlistname));
#endif

            // Set the active combobox item.
            comboboxentry_playlist_entries++;
            playlist_number = comboboxentry_playlist_entries - 1;
            gtk_combo_box_set_active(GTK_COMBO_BOX(comboboxentry_playlist), comboboxentry_playlist_entries - 1);
            SetPlaylistButtonState(TRUE);
            setPlaylistField(playlist_number);

            // Clean up fields.
            g_free(playlistname);
        } else {
            // Let the user know the import failed.
            g_fprintf(stderr, _("The playlist failed to import correctly.\n"));
            displayError(_("The playlist failed to import correctly.\n"));
        }
        // Clean up fields.
        g_free(playlistfilename);
    }
} // end on_Playlist_ImportPlaylistButton_activate()

// ************************************************************************************************

/**
 * Callback to handle the Export Playlist button in the Playlist editor dialog.
 * @param menuitem
 * @param user_data
 */
void on_Playlist_ExportPlaylistButton_activate(GtkMenuItem *menuitem, gpointer user_data) {
    gchar *playlistfilename = NULL;
    GtkWidget *FileDialog;

    // Save our current selected playlist!
    if (devicePlayLists != NULL)
        playlist_SavePlaylist(playlist_number);

    gint PlayListID = gtk_combo_box_get_active(GTK_COMBO_BOX(comboboxentry_playlist));

    if (PlayListID != -1) {
        // We have something selected so lets do the dance.
        LIBMTP_playlist_t* tmpplaylist = devicePlayLists;
        if (PlayListID > 0) {
            while (PlayListID--)
                if (tmpplaylist->next != NULL)
                    tmpplaylist = tmpplaylist->next;
        }
        // We should be in the correct playlist LIBMTP structure.

        playlistfilename = g_strdup_printf("%s.%s", tmpplaylist->name, "m3u");

        FileDialog = gtk_file_chooser_dialog_new(_("Save as..."),
                GTK_WINDOW(windowMain), GTK_FILE_CHOOSER_ACTION_SAVE,
                GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
                GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT,
                NULL);
        gtk_file_chooser_set_current_name(GTK_FILE_CHOOSER(FileDialog), playlistfilename);

        if (gtk_dialog_run(GTK_DIALOG(FileDialog)) == GTK_RESPONSE_ACCEPT) {
            playlistfilename = gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(FileDialog));
        }

        gtk_widget_hide(FileDialog);
        gtk_widget_destroy(FileDialog);

        if (playlistfilename != NULL) {
            playlistExport(playlistfilename, tmpplaylist);
            g_free(playlistfilename);
        }
    }
} // end on_Playlist_ExportPlaylistButton_activate()

// ************************************************************************************************

/**
 * Callback to handle the Delete Playlist button in the Playlist editor dialog.
 * @param menuitem
 * @param user_data
 */
void on_Playlist_DelPlaylistButton_activate(GtkMenuItem *menuitem, gpointer user_data) {

    gint PlayListID = gtk_combo_box_get_active(GTK_COMBO_BOX(comboboxentry_playlist));

    if (PlayListID != -1) {
        // We have something selected so lets do the dance.
        LIBMTP_playlist_t* tmpplaylist = devicePlayLists;
        if (PlayListID > 0) {
            while (PlayListID--)
                if (tmpplaylist->next != NULL)
                    tmpplaylist = tmpplaylist->next;
        }
        // We should be in the correct playlist LIBMTP structure.
        playlistDelete(tmpplaylist);
        // Clear the PL list view box
        gtk_list_store_clear(GTK_LIST_STORE(playlist_PL_List));
        // Rebuild the playlist structure and combobox.
        devicePlayLists = getPlaylists();
        setPlayListComboBox();
    }
} // end on_Playlist_DelPlaylistButton_activate()

// ************************************************************************************************

/**
 * Callback to handle the Delete Track button in the Playlist editor dialog.
 * @param menuitem
 * @param user_data
 */
void on_Playlist_DelFileButton_activate(GtkMenuItem *menuitem, gpointer user_data) {
    if (playlist_PL_ListGetSelection() == NULL)
        return;
    playlist_PL_ListRemove(playlist_PL_ListGetSelection());
} // end on_Playlist_DelFileButton_activate()

// ************************************************************************************************

/**
 * Callback to handle the Add Track button in the Playlist editor dialog.
 * @param menuitem
 * @param user_data
 */
void on_Playlist_AddFileButton_activate(GtkMenuItem *menuitem, gpointer user_data) {
    //g_printf("Clicked on add file in playlist button\n");
    if (playlist_TrackList_GetSelection() == NULL)
        return;
    playlist_TrackList_Add(playlist_TrackList_GetSelection());
} // end on_Playlist_AddFileButton_activate()

// ************************************************************************************************

/**
 * Callback to handle the Move Track Up button in the Playlist editor dialog.
 * @param menuitem
 * @param user_data
 */
void on_Playlist_FileUpButton_activate(GtkMenuItem *menuitem, gpointer user_data) {
    playlist_move_files(-1);
} // end on_Playlist_FileUpButton_activate()

// ************************************************************************************************

/**
 * Callback to handle the Move Track Down button in the Playlist editor dialog.
 * @param menuitem
 * @param user_data
 */
void on_Playlist_FileDownButton_activate(GtkMenuItem *menuitem, gpointer user_data) {
    playlist_move_files(1);
} // end on_Playlist_FileDownButton_activate()

// ************************************************************************************************

/**
 * Callback to handle the change of Playlist selection in the Playlist editor dialog.
 * @param menuitem
 * @param user_data
 */
void on_Playlist_Combobox_activate(GtkComboBox *combobox, gpointer user_data) {
    // Save our current selected playlist
    playlist_SavePlaylist(playlist_number);
    // Get our new playlist ID, and display the contents of it.
    playlist_number = gtk_combo_box_get_active(GTK_COMBO_BOX(comboboxentry_playlist));
    setPlaylistField(playlist_number);
} // end on_Playlist_Combobox_activate()

// ************************************************************************************************

/**
 * Callback to handle the change of columns viewable in the main window.
 * @param menuitem
 * @param user_data
 */
void on_view_activate(GtkMenuItem *menuitem, gpointer user_data) {
#if GMTP_USE_GTK2
    gchar *gconf_path = NULL;
    gboolean state = gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(menuitem));
    // Main menu.
    if ((void *) menuitem == (void *) menu_view_filesize) gconf_path = g_strdup("/apps/gMTP/viewFileSize");
    if ((void *) menuitem == (void *) menu_view_filetype) gconf_path = g_strdup("/apps/gMTP/viewFileType");
    if ((void *) menuitem == (void *) menu_view_track_number) gconf_path = g_strdup("/apps/gMTP/viewTrackNumber");
    if ((void *) menuitem == (void *) menu_view_title) gconf_path = g_strdup("/apps/gMTP/viewTitle");
    if ((void *) menuitem == (void *) menu_view_artist) gconf_path = g_strdup("/apps/gMTP/viewArtist");
    if ((void *) menuitem == (void *) menu_view_album) gconf_path = g_strdup("/apps/gMTP/viewAlbum");
    if ((void *) menuitem == (void *) menu_view_year) gconf_path = g_strdup("/apps/gMTP/viewYear");
    if ((void *) menuitem == (void *) menu_view_genre) gconf_path = g_strdup("/apps/gMTP/viewGenre");
    if ((void *) menuitem == (void *) menu_view_duration) gconf_path = g_strdup("/apps/gMTP/viewDuration");
    if ((void *) menuitem == (void *) menu_view_folders) gconf_path = g_strdup("/apps/gMTP/viewFolders");
    // context menu
    if ((void *) menuitem == (void *) cViewSize) gconf_path = g_strdup("/apps/gMTP/viewFileSize");
    if ((void *) menuitem == (void *) cViewType) gconf_path = g_strdup("/apps/gMTP/viewFileType");
    if ((void *) menuitem == (void *) cViewTrackNumber) gconf_path = g_strdup("/apps/gMTP/viewTrackNumber");
    if ((void *) menuitem == (void *) cViewTrackName) gconf_path = g_strdup("/apps/gMTP/viewTitle");
    if ((void *) menuitem == (void *) cViewArtist) gconf_path = g_strdup("/apps/gMTP/viewArtist");
    if ((void *) menuitem == (void *) cViewAlbum) gconf_path = g_strdup("/apps/gMTP/viewAlbum");
    if ((void *) menuitem == (void *) cViewYear) gconf_path = g_strdup("/apps/gMTP/viewYear");
    if ((void *) menuitem == (void *) cViewGenre) gconf_path = g_strdup("/apps/gMTP/viewGenre");
    if ((void *) menuitem == (void *) cViewDuration) gconf_path = g_strdup("/apps/gMTP/viewDuration");

    if ((gconfconnect != NULL) && (gconf_path != NULL)) {
        gconf_client_set_bool(gconfconnect, gconf_path, state, NULL);
        g_free(gconf_path);
    }
#else
    gchar *gsetting_path = NULL;
    gboolean state = gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(menuitem));
    // main menu
    if ((void *) menuitem == (void *) menu_view_filesize) gsetting_path = g_strdup("viewfilesize");
    if ((void *) menuitem == (void *) menu_view_filetype) gsetting_path = g_strdup("viewfiletype");
    if ((void *) menuitem == (void *) menu_view_track_number) gsetting_path = g_strdup("viewtracknumber");
    if ((void *) menuitem == (void *) menu_view_title) gsetting_path = g_strdup("viewtitle");
    if ((void *) menuitem == (void *) menu_view_artist) gsetting_path = g_strdup("viewartist");
    if ((void *) menuitem == (void *) menu_view_album) gsetting_path = g_strdup("viewalbum");
    if ((void *) menuitem == (void *) menu_view_year) gsetting_path = g_strdup("viewyear");
    if ((void *) menuitem == (void *) menu_view_genre) gsetting_path = g_strdup("viewgenre");
    if ((void *) menuitem == (void *) menu_view_duration) gsetting_path = g_strdup("viewduration");
    if ((void *) menuitem == (void *) menu_view_folders) gsetting_path = g_strdup("viewfolders");

    //context menu.
    if ((void *) menuitem == (void *) cViewSize) gsetting_path = g_strdup("viewfilesize");
    if ((void *) menuitem == (void *) cViewType) gsetting_path = g_strdup("viewfiletype");
    if ((void *) menuitem == (void *) cViewTrackNumber) gsetting_path = g_strdup("viewtracknumber");
    if ((void *) menuitem == (void *) cViewTrackName) gsetting_path = g_strdup("viewtitle");
    if ((void *) menuitem == (void *) cViewArtist) gsetting_path = g_strdup("viewartist");
    if ((void *) menuitem == (void *) cViewAlbum) gsetting_path = g_strdup("viewalbum");
    if ((void *) menuitem == (void *) cViewYear) gsetting_path = g_strdup("viewyear");
    if ((void *) menuitem == (void *) cViewGenre) gsetting_path = g_strdup("viewgenre");
    if ((void *) menuitem == (void *) cViewDuration) gsetting_path = g_strdup("viewduration");

    if ((gsettings_connect != NULL) && (gsetting_path != NULL)) {
        g_settings_set_boolean(gsettings_connect, gsetting_path, state);
        g_settings_sync();
        g_free(gsetting_path);
    }
#endif
} // end on_view_activate()

// ************************************************************************************************

/**
 * Callback to handle when a user closes the Progress Dialog box, via the X button.
 * @param window
 * @param user_data
 */
void on_progressDialog_Close(GtkWidget *window, gpointer user_data) {
    // Set the global flag that the user has done it.
    progressDialog_killed = TRUE;
} // end on_progressDialog_Close()

// ************************************************************************************************

/**
 * Callback to handle when a user presses the Cancel button in the Progress Dialog box
 * @param window
 * @param user_data
 */
void on_progressDialog_Cancel(GtkWidget *button, gpointer user_data) {
    // Set the global flag that the user has done it.
    progressDialog_killed = TRUE;
    // Destroy the dialog box.
    gtk_widget_hide(progressDialog);
    gtk_widget_destroy(progressDialog);
} // end on_progressDialog_Cancel()


// ************************************************************************************************

/**
 * Callback to handle user asking to create a new playlist from the AutoAddTrack to Playlist option.
 * @param button
 * @param user_data
 */
void on_TrackPlaylist_NewPlaylistButton_activate(GtkWidget *button, gpointer user_data) {
    gchar *playlistname = NULL;
    gint combobox_entries = 0;

    playlistname = displayPlaylistNewDialog();
    if (playlistname != NULL) {
        // Add in playlist to MTP device.
        playlistAdd(playlistname);
        // Refresh our playlist information.
        devicePlayLists = getPlaylists();

        // Add it to our combobox
#if GMTP_USE_GTK2
        gtk_combo_box_append_text(GTK_COMBO_BOX(combobox_AddTrackPlaylist), g_strdup(playlistname));
#else
        gtk_combo_box_text_append_text(GTK_COMBO_BOX_TEXT(combobox_AddTrackPlaylist), g_strdup(playlistname));
#endif
        g_free(playlistname);

        // Set the active combobox item.
        combobox_entries = gtk_tree_model_iter_n_children(gtk_combo_box_get_model(GTK_COMBO_BOX(combobox_AddTrackPlaylist)), NULL);
        gtk_combo_box_set_active(GTK_COMBO_BOX(combobox_AddTrackPlaylist), combobox_entries - 1);
    }
}


// ************************************************************************************************

/**
 * Callback to handle adding file to playlist.
 * @param menuitem
 * @param user_data
 */
void on_fileAddToPlaylist_activate(GtkMenuItem *menuitem, gpointer user_data) {
    // Let's check to see if we have anything selected in our treeview?
    if (fileListGetSelection() == NULL) {
        displayInformation(_("No files/folders selected?"));
        return;
    }
    // Display the select playlist dialog;
    int32_t addTrackPlaylistID = displayAddTrackPlaylistDialog(TRUE);

    // Now add the actual files from the MTP device.
    if (addTrackPlaylistID != GMTP_NO_PLAYLIST) {
        fileListAddToPlaylist(fileListGetSelection(), addTrackPlaylistID);
    }
}


// ************************************************************************************************

/**
 * Callback to handle removing file from playlist.
 * @param menuitem
 * @param user_data
 */
void on_fileRemoveFromPlaylist_activate(GtkMenuItem *menuitem, gpointer user_data) {
    // Let's check to see if we have anything selected in our treeview?
    if (fileListGetSelection() == NULL) {
        displayInformation(_("No files/folders selected?"));
        return;
    }
    // Display the select playlist dialog;
    int32_t addTrackPlaylistID = displayAddTrackPlaylistDialog(FALSE);

    // Now remove the actual files from the MTP device.
    if (addTrackPlaylistID != GMTP_NO_PLAYLIST) {
        fileListRemoveFromPlaylist(fileListGetSelection(), addTrackPlaylistID);
    }
}


// ************************************************************************************************

/**
 * Callback to handle when a row is selected in the folder list.
 * @param treeselection
 * @param user_data
 */
void on_treeviewFolders_rowactivated(GtkTreeSelection *treeselection, gpointer user_data) {
    // Block the handler from running ...
    g_signal_handler_block((gpointer) fileSelection, fileSelectHandler);
    g_signal_handler_block((gpointer) folderSelection, folderSelectHandler);

    if ((void *) treeselection == (void *) fileSelection) {
        gtk_tree_selection_unselect_all(folderSelection);
    } else {
        gtk_tree_selection_unselect_all(fileSelection);
    }

    // Unblock the handler from running ...
    g_signal_handler_unblock((gpointer) fileSelection, fileSelectHandler);
    g_signal_handler_unblock((gpointer) folderSelection, folderSelectHandler);
}
