// -----------------------------------------------------------------------------
//
//  Gmsh C++ tutorial 15
//
//  Embedded points, lines and surfaces
//
// -----------------------------------------------------------------------------

// By default, across geometrical dimensions meshes generated by Gmsh are only
// conformal if lower dimensional entities are on the boundary of higher
// dimensional ones (i.e. if points, curves or surfaces are part of the boundary
// of volumes).

// Embedding constraints allow to force a mesh to be conformal to other lower
// dimensional entities.

#include <gmsh.h>

namespace factory = gmsh::model::geo;
namespace model = gmsh::model;

int main(int argc, char **argv)
{
  gmsh::initialize();
  gmsh::option::setNumber("General.Terminal", 1);

  gmsh::model::add("t15");

  // Copied from t1.cpp:
  double lc = 1e-2;
  factory::addPoint(0, 0, 0, lc, 1);
  factory::addPoint(.1, 0,  0, lc, 2);
  factory::addPoint(.1, .3, 0, lc, 3);
  factory::addPoint(0,  .3, 0, lc, 4);
  factory::addLine(1, 2, 1);
  factory::addLine(3, 2, 2);
  factory::addLine(3, 4, 3);
  factory::addLine(4, 1, 4);
  factory::addCurveLoop({4, 1, -2, 3}, 1);
  factory::addPlaneSurface({1}, 1);

  // We change the mesh size to generate a coarser mesh
  lc *=  4;
  factory::mesh::setSize({{0, 1}, {0, 2}, {0, 3}, {0, 4}}, lc);

  // We define a new point
  factory::addPoint(0.02, 0.02, 0., lc, 5);

  // We have to synchronize before embedding entites:
  factory::synchronize();

  // One can force this point to be included ("embedded") in the 2D mesh, using
  // the `embed()' function:
  model::mesh::embed(0, {5}, 2, 1);

  // In the same way, one use `embed()' to force a curve to be embedded in the
  // 2D mesh:
  factory::addPoint(0.02, 0.12, 0., lc, 6);
  factory::addPoint(0.04, 0.18, 0., lc, 7);
  factory::addLine(6, 7, 5);
  factory::synchronize();
  model::mesh::embed(1, {5}, 2, 1);

  // Points and curves can also be embedded in volumes
  std::vector<std::pair<int, int> > ext;
  factory::extrude({{2, 1}}, 0, 0, 0.1, ext);

  int p = factory::addPoint(0.07, 0.15, 0.025, lc);

  factory::synchronize();
  model::mesh::embed(0, {p}, 3, 1);

  factory::addPoint(0.025, 0.15, 0.025, lc, p+1);
  int l = factory::addLine(7, p+1);

  factory::synchronize();
  model::mesh::embed(1, {l}, 3, 1);

  // Finally, we can also embed a surface in a volume:
  factory::addPoint(0.02, 0.12, 0.05, lc, p+2);
  factory::addPoint(0.04, 0.12, 0.05, lc, p+3);
  factory::addPoint(0.04, 0.18, 0.05, lc, p+4);
  factory::addPoint(0.02, 0.18, 0.05, lc, p+5);

  factory::addLine(p+2, p+3, l+1);
  factory::addLine(p+3, p+4, l+2);
  factory::addLine(p+4, p+5, l+3);
  factory::addLine(p+5, p+2, l+4);

  int ll = factory::addCurveLoop({l+1, l+2, l+3, l+4});
  int s = factory::addPlaneSurface({ll});

  factory::synchronize();
  model::mesh::embed(2, {s}, 3, 1);

  model::mesh::generate(3);

  gmsh::write("t15.msh");

  gmsh::finalize();
  return 0;
}
