/* gmpc-magnatune (GMPC plugin)
 * Copyright (C) 2006-2009 Qball Cow <qball@sarine.nl>
 * Project homepage: http://gmpcwiki.sarine.nl/
 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.

 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
*/

#include <stdio.h>
#include <string.h>
#include <gtk/gtk.h>
#include <gmpc/plugin.h>
#include <gmpc/gmpc_easy_download.h>
#include <gmpc/misc.h>
#include <libmpd/libmpd-internal.h>
#include <sqlite3.h>
#include <libxml/xmlreader.h>
#include <libxml/tree.h>
#include "magnatune.h"

static void magnatune_cleanup_xml();
static sqlite3 *magnatune_sqlhandle = NULL;
GMutex *mt_db_lock = NULL;


void magnatune_db_destroy(void)
{
    if(mt_db_lock)
    {
        g_mutex_lock(mt_db_lock);
        g_mutex_unlock(mt_db_lock);
        g_mutex_free(mt_db_lock);
    }
    if(magnatune_sqlhandle)
    {
        sqlite3_close(magnatune_sqlhandle);
        magnatune_sqlhandle = NULL;
    }
}

/* run this before using the other fucntions */
void magnatune_db_init()
{
    mt_db_lock = g_mutex_new();
}

void magnatune_db_open()
{
    gchar *path = NULL;

    g_mutex_lock(mt_db_lock);	

    /**
     * if open close it
     */
    if(magnatune_sqlhandle)
    {
        sqlite3_close(magnatune_sqlhandle);
        magnatune_sqlhandle = NULL;
    }

    g_free(path);
    path = gmpc_get_user_path("magnatune.sqlite3"); 
    sqlite3_open(path, &(magnatune_sqlhandle));
    g_free(path);

    g_mutex_unlock(mt_db_lock);
}
/* FIXME */
gboolean magnatune_db_has_data()
{
    char *query = sqlite3_mprintf("SELECT * from 'sqlite_master'");
    sqlite3_stmt *stmt;
    const char *tail;
    int r;
    
    
    g_mutex_lock(mt_db_lock);
    r= sqlite3_prepare_v2(magnatune_sqlhandle, query, -1,  &stmt,  &tail);
    sqlite3_free(query);
    if(r == SQLITE_OK) {
        while((r = sqlite3_step(stmt)) == SQLITE_ROW){
            sqlite3_finalize(stmt);
            g_mutex_unlock(mt_db_lock);
            return TRUE;
        }
    }

    g_mutex_unlock(mt_db_lock);
    return FALSE;
}

MpdData * magnatune_db_get_genre_list()
{
    MpdData *list = NULL;
    xmlNodePtr **root;
    xmlNodePtr **cur;
    int i;

    g_mutex_lock(mt_db_lock);
    char *query = sqlite3_mprintf("SELECT genre from 'genres' group by genre");
    sqlite3_stmt *stmt;
    const char *tail;
    int r = sqlite3_prepare_v2(magnatune_sqlhandle, query, -1,  &stmt,  &tail);
    if(r ==SQLITE_OK) {
        while((r = sqlite3_step(stmt)) == SQLITE_ROW)
        {
            list = mpd_new_data_struct_append(list);
            list->type = MPD_DATA_TYPE_TAG;
            list->tag_type = MPD_TAG_ITEM_GENRE;
            list->tag = g_strdup(sqlite3_column_text(stmt,0));
        }
        sqlite3_finalize(stmt);
    }
    sqlite3_free(query);
    g_mutex_unlock(mt_db_lock);
    return misc_mpddata_remove_duplicate_songs(list); 
}

void magnatune_db_load_data(const char *data, const goffset length)
{
    gchar *error = NULL;
    xmlTextReaderPtr reader = NULL;
    gchar *path;

    g_mutex_lock(mt_db_lock);

    path = gmpc_get_user_path("magnatune.sqlite3"); 
    if(magnatune_sqlhandle)
    {
        sqlite3_close(magnatune_sqlhandle);
        magnatune_sqlhandle = NULL;
    }
//    if(gmpc_easy_download("http://he3.magnatune.com/info/sqlite_magnatune.db", dld))
    if(data){
        GError *error = NULL;
        gssize size= (gssize)length;
        g_file_set_contents(path, data, size, &error);    
        if(error) {
            printf("%s\n", error->message);
            g_error_free(error);
        }
        printf("%s\n", path);
  //      gmpc_easy_download_clean(dld);
    }
    /* open the db if it is closed */
    if(!magnatune_sqlhandle) {


        int retv = sqlite3_open(path, &(magnatune_sqlhandle));
        if(retv != SQLITE_OK)
        {
            /* Cleanup */
            g_mutex_unlock(mt_db_lock);
            return;
        }
    }

    sqlite3_exec(magnatune_sqlhandle, "CREATE INDEX songsAlbumname ON songs(albumname);", NULL, NULL, &error);
    if(error)printf("%i: %s",__LINE__, error);
    sqlite3_exec(magnatune_sqlhandle, "CREATE INDEX genresAlbumname ON genres(albumname);", NULL, NULL, &error);
    if(error)printf("%i: %s",__LINE__, error);
    sqlite3_exec(magnatune_sqlhandle, "CREATE INDEX albumsAlbumname ON albums(albumname);", NULL, NULL, &error);
    if(error)printf("%i: %s",__LINE__, error);
    g_mutex_unlock(mt_db_lock);

//    g_idle_add(magnatune_end_download, NULL);
    g_free(path);
}

MpdData * magnatune_db_get_artist_list(char *wanted_genre)
{
    MpdData *list = NULL;
    xmlNodePtr root;
    xmlNodePtr cur;
    /** check if there is data */
    g_mutex_lock(mt_db_lock);
    char *query = sqlite3_mprintf("SELECT albumname from 'genres' WHERE genre=%Q", wanted_genre);
    sqlite3_stmt *stmt;
    const char *tail;
    int r = sqlite3_prepare_v2(magnatune_sqlhandle, query, -1,  &stmt,  &tail);
    if(r ==SQLITE_OK) {
        while((r = sqlite3_step(stmt)) == SQLITE_ROW)
        {
            sqlite3_stmt *stmt2;
            const char *tail2;
            char *query2 = sqlite3_mprintf("SELECT artist from 'albums' WHERE albumname=%Q", sqlite3_column_text(stmt,0));
            int r2 = sqlite3_prepare_v2(magnatune_sqlhandle, query2, -1,  &stmt2,  &tail2);
            if(r2 ==SQLITE_OK) {
                while((r2 = sqlite3_step(stmt2)) == SQLITE_ROW)
                {
                    list = mpd_new_data_struct_append(list);
                    list->type = MPD_DATA_TYPE_TAG;
                    list->tag_type = MPD_TAG_ITEM_ARTIST;
                    list->tag = g_strdup(sqlite3_column_text(stmt2,0));
                }
            }
            sqlite3_finalize(stmt2);
            sqlite3_free(query2);
        }
        sqlite3_finalize(stmt);
    }
    sqlite3_free(query);
    g_mutex_unlock(mt_db_lock);
    return misc_mpddata_remove_duplicate_songs(list);
}

MpdData *magnatune_db_get_album_list(char *wanted_genre,char *wanted_artist)
{
    MpdData *list = NULL;
    xmlNodePtr **root;
    xmlNodePtr **cur;
    /** check if there is data */
    g_mutex_lock(mt_db_lock);

    char *query = sqlite3_mprintf("SELECT albumname from 'albums' WHERE artist=%Q",wanted_artist);
    sqlite3_stmt *stmt;
    const char *tail;
    int r = sqlite3_prepare_v2(magnatune_sqlhandle, query, -1,  &stmt,  &tail);
    if(r ==SQLITE_OK) {
        while((r = sqlite3_step(stmt)) == SQLITE_ROW)
        {
            sqlite3_stmt *stmt2;
            const char *tail2;
            char *query2 = sqlite3_mprintf("SELECT albumname from 'genres' WHERE albumname=%Q AND genre=%Q", sqlite3_column_text(stmt,0),wanted_genre);
            int r2 = sqlite3_prepare_v2(magnatune_sqlhandle, query2, -1,  &stmt2,  &tail2);
            if(r2 ==SQLITE_OK) {
                while((r2 = sqlite3_step(stmt2)) == SQLITE_ROW)
                {
                    list = mpd_new_data_struct_append(list);
                    list->type = MPD_DATA_TYPE_TAG;
                    list->tag_type = MPD_TAG_ITEM_ALBUM;
                    list->tag = g_strdup(sqlite3_column_text(stmt2,0));
                }
            }
            sqlite3_finalize(stmt2);
            sqlite3_free(query2);
        }
        sqlite3_finalize(stmt);
    }
    sqlite3_free(query);
    g_mutex_unlock(mt_db_lock);
    return mpd_data_get_first(list); 
}

static char *__magnatune_get_artist_name(const char *album)
{
    char *retv = NULL;
    sqlite3_stmt *stmt;
    const char *tail;
    int r = 0;
    char *query =  NULL;
    if(!album) return NULL;
    query = sqlite3_mprintf("SELECT artist from 'albums' WHERE albumname=%Q limit 1",album);
    r=sqlite3_prepare_v2(magnatune_sqlhandle, query, -1,  &stmt,  &tail);
    if(r ==SQLITE_OK) {
        if((r = sqlite3_step(stmt)) == SQLITE_ROW)
        {
            retv = g_strdup(sqlite3_column_text(stmt, 0));
        }
        sqlite3_finalize(stmt);
    }
    sqlite3_free(query);
    return retv;
}
static char *__magnatune_get_genre_name(const char *album)
{
    char *retv = NULL;
    sqlite3_stmt *stmt;
    const char *tail;
    int r = 0;
    char *query =  NULL;
    if(!album) return NULL;
    query = sqlite3_mprintf("SELECT genre from 'genres' WHERE albumname=%Q",album);
    r=sqlite3_prepare_v2(magnatune_sqlhandle, query, -1,  &stmt,  &tail);
    if(r ==SQLITE_OK) {
        while((r = sqlite3_step(stmt)) == SQLITE_ROW)
        {
            const char *temp= sqlite3_column_text(stmt, 0);
            if(retv)
            {
                gchar *t = retv;
                retv = g_strconcat(t, ", ",temp, NULL);
                g_free(t);
            }
            else retv = g_strdup(temp);
        }
        sqlite3_finalize(stmt);
    }
    sqlite3_free(query);
    return retv;
}
static MpdData *__magnatune_get_data_album(const char *album, gboolean exact)
{
    MpdData *list = NULL;
    char *query = NULL;
    sqlite3_stmt *stmt;
    const char *tail;
    int r = 0;
    if(exact)
    {
        query =  sqlite3_mprintf("SELECT songs.albumname,duration,number,desc,mp3 from 'songs' " 
        "WHERE songs.albumname=%Q",album);
    }else{
        query =  sqlite3_mprintf("SELECT songs.albumname,duration,number,desc,mp3 from 'songs' "
        "WHERE songs.albumname LIKE '%%%%%q%%%%'",album);
    }
    r=sqlite3_prepare_v2(magnatune_sqlhandle, query, -1,  &stmt,  &tail);
    if(r ==SQLITE_OK) {
        while((r = sqlite3_step(stmt)) == SQLITE_ROW)
        {
            gchar *temp = gmpc_easy_download_uri_escape(sqlite3_column_text(stmt,4));
            list = mpd_new_data_struct_append(list);
            list->type = MPD_DATA_TYPE_SONG;
            list->song = mpd_newSong();
            list->song->album = g_strdup(sqlite3_column_text(stmt,0));
            list->song->artist = __magnatune_get_artist_name(list->song->album);
            list->song->genre = __magnatune_get_genre_name(list->song->album);
            list->song->title= g_strdup(sqlite3_column_text(stmt,3));
            list->song->track = g_strdup(sqlite3_column_text(stmt,2));
            list->song->time = sqlite3_column_int(stmt,1);
            list->song->file = g_strdup_printf("http://he3.magnatune.com/all/%s",temp);
            g_free(temp);
        }
        sqlite3_finalize(stmt);
    }
    sqlite3_free(query);
    return list;
}
static gchar ** __magnatune_get_albums(const char *genre, const char *artist, gboolean exact)
{
    char **retv = NULL;
    sqlite3_stmt *stmt;
    const char *tail;
    int items = 0;
    int r = 0;
    char *query =  NULL;

    if(genre && !artist) {
        if(exact)
            query = sqlite3_mprintf("SELECT albumname FROM 'genres' WHERE genre=%Q", genre);
        else
            query = sqlite3_mprintf("SELECT albumname FROM 'genres' WHERE genre LIKE '%%%%%q%%%%'", genre);
    }
    else if (artist && !genre) {
        if(exact)
            query = sqlite3_mprintf("SELECT albumname FROM 'albums' WHERE artist=%Q", artist);
        else 
            query = sqlite3_mprintf("SELECT albumname FROM 'albums' WHERE artist LIKE '%%%%%q%%%%'", artist);
    }
    else if (artist && genre) {
        if(exact)
            query = sqlite3_mprintf("SELECT genres.albumname FROM 'albums' JOIN 'genres' on albums.albumname = genres.albumname WHERE albums.artist=%Q AND genres.genre=%Q", artist,genre);
        else
            query = sqlite3_mprintf("SELECT genres.albumname FROM 'albums' JOIN 'genres' on albums.albumname = genres.albumname WHERE albums.artist LIKE '%%%%%q%%%%' AND genres.genre LIKE '%%%%%q%%%%'", artist,genre);
    }


    r=sqlite3_prepare_v2(magnatune_sqlhandle, query, -1,  &stmt,  &tail);
    if(r ==SQLITE_OK) {
        while((r = sqlite3_step(stmt)) == SQLITE_ROW)
        {
            items++;
            retv = g_realloc(retv, (items+1)*sizeof(*retv));
            retv[items] = NULL;
            retv[items-1] = g_strdup(sqlite3_column_text(stmt, 0));
        }
        sqlite3_finalize(stmt);
    }
    sqlite3_free(query);
    return retv;
}

MpdData* magnatune_db_get_song_list(const char *wanted_genre,const char *wanted_artist, const char *wanted_album, gboolean exact)
{
    MpdData *data = NULL;
    xmlNodePtr * root;
    xmlNodePtr * cur;
    char *query;

    if(!wanted_genre && !wanted_artist && !wanted_album) return NULL;

    g_mutex_lock(mt_db_lock);
    if(wanted_album) /* album seems to be unique */
    {
        data = __magnatune_get_data_album(wanted_album, exact);
    }
    else 
    {
        char **albums = __magnatune_get_albums(wanted_genre, wanted_artist, exact);
        if(albums)
        {
            int i;
            for(i=0; albums[i];i++)
            {
                MpdData *data2 =  __magnatune_get_data_album(albums[i], exact);
                data = mpd_data_concatenate(data, data2);
            }
            g_strfreev(albums);
        }
    }

    g_mutex_unlock(mt_db_lock);
    return mpd_data_get_first(data);
}

MpdData * magnatune_db_search_title(const gchar *title)
{

    MpdData *list = NULL;
    char *query = NULL;
    sqlite3_stmt *stmt;
    const char *tail;
    int r = 0;
        query =  sqlite3_mprintf("SELECT songs.albumname,duration,number,desc,mp3 from 'songs' "
        "WHERE songs.desc LIKE '%%%%%q%%%%'",title);
    r=sqlite3_prepare_v2(magnatune_sqlhandle, query, -1,  &stmt,  &tail);
    if(r ==SQLITE_OK) {
        while((r = sqlite3_step(stmt)) == SQLITE_ROW)
        {
            gchar *temp = gmpc_easy_download_uri_escape(sqlite3_column_text(stmt,4));
            list = mpd_new_data_struct_append(list);
            list->type = MPD_DATA_TYPE_SONG;
            list->song = mpd_newSong();
            list->song->album = g_strdup(sqlite3_column_text(stmt,0));
            list->song->artist = __magnatune_get_artist_name(list->song->album);
            list->song->genre = __magnatune_get_genre_name(list->song->album);
            list->song->title= g_strdup(sqlite3_column_text(stmt,3));
            list->song->track = g_strdup(sqlite3_column_text(stmt,2));
            list->song->time = sqlite3_column_int(stmt,1);
            list->song->file = g_strdup_printf("http://he3.magnatune.com/all/%s",temp);
            g_free(temp);
        }
        sqlite3_finalize(stmt);
    }
    sqlite3_free(query);
    return list;
}



gchar *magnatune_get_artist_image(const gchar *wanted_artist)
{
    char *retv = NULL;
    sqlite3_stmt *stmt;
    char *artist =  __magnatune_process_string(wanted_artist);
    const char *tail;
    int r = 0;
    char *query =  NULL;
    query = sqlite3_mprintf("SELECT homepage from 'artists' WHERE artist LIKE '%%%%%q%%%%' limit 1",artist);
    r=sqlite3_prepare_v2(magnatune_sqlhandle, query, -1,  &stmt,  &tail);
    if(r ==SQLITE_OK) {
        if((r = sqlite3_step(stmt)) == SQLITE_ROW)
        {
            gchar *temp = gmpc_easy_download_uri_escape( sqlite3_column_text(stmt, 0));
            retv = g_strdup_printf("http://he3.magnatune.com/artists/img/%s_1.jpg", temp);
            g_free(temp);
        }
        sqlite3_finalize(stmt);
    }
    sqlite3_free(query);
    g_free(artist);
    return retv;
}

gchar *magnatune_get_album_url(const gchar *wanted_artist, const gchar *wanted_album)
{

}


gchar *magnatune_get_album_image(const gchar *wanted_artist, const gchar *wanted_album)
{
    char *retv = NULL;
    char *artist =  __magnatune_process_string(wanted_artist);
    char *album =  __magnatune_process_string(wanted_album);
    gchar *artist_enc = gmpc_easy_download_uri_escape(artist);
    gchar *album_enc = gmpc_easy_download_uri_escape(album);
    retv = g_strdup_printf("http://he3.magnatune.com/music/%s/%s/cover_600.jpg",artist_enc, album_enc);
    g_free(artist);
    g_free(album);
    g_free(artist_enc);
    g_free(album_enc);
    return retv;
}
