/*
 * Copyright 1999-2006 University of Chicago
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <globus_common.h>
#include <globus_gss_assist.h>

#include "globus_rls_client.h"
#include "globus_rls_rpc.h"

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <time.h>

/*
 * RLSLIST and friends are used to remember each globus_list_t result,
 * plus a destructor function to free each list element, returned by the API.
 * This is used by globus_rls_client_free_list() to free all memory
 * associated with a list result.  A free list is used to
 * minimize calls to malloc/free, and a mutex to synchronize access
 * to the list of lists and free list.
 */
typedef struct rlslist_ {
  globus_list_t		*list;
  globus_list_t		*last;
  void			(*destructor)();
  struct rlslist_	*nxt;
} RLSLIST;
static RLSLIST		*rlslist_freelist;
static RLSLIST		*rlslisttab;
static globus_mutex_t	rlslist_mutex;

static globus_result_t	exists_bulk2(globus_rls_handle_t *h, char *method,
				    globus_rls_obj_type_t objtype,
				    globus_list_t *keylist,
				    globus_list_t **str2bulk_list);
static globus_result_t	query2(globus_rls_handle_t *h, char *method,
			       char *key, int *offset, int reslimit,
			       globus_list_t **str2_list);
static globus_result_t	query_bulk2(globus_rls_handle_t *h, char *method,
				    globus_list_t *keylist,
				    globus_list_t **str2bulk_list);
static globus_result_t	query_wc2(globus_rls_handle_t *h, char *method,
				  char *pattern, globus_rls_pattern_t type,
				  int *offset, int reslimit,
				  globus_list_t **str2_list);
static globus_result_t	bulk_update(char *method, globus_rls_handle_t *h,
				    globus_list_t *str2_list,
				    globus_list_t  **str2bulk_list);
static globus_result_t	bulk_attr_update(int adding, globus_rls_handle_t *h,
				         globus_list_t *attr_obj_list,
					 globus_list_t **str2bulk_list);
static globus_result_t	rrpc_call(globus_rls_handle_t *h, BUFFER *bp,
			  	  char *method, int na, ...);
static globus_result_t	rrpc_getstr(globus_rls_handle_t *h, BUFFER *bp,
				    char *rbuf, int rbuflen);
static globus_result_t	rrpc_str2(globus_rls_handle_t *h, BUFFER *bp,
				  RLSLIST *rlslist, int *offset);
static globus_result_t	rrpc_str2bulk(globus_rls_handle_t *h, BUFFER *bp,
				      int getboth, RLSLIST *rlslist);
static globus_result_t	rrpc_attr_obj(globus_rls_handle_t *h, BUFFER *bp,
				RLSLIST *rlslist, int *offset, char *name,
				globus_rls_obj_type_t objtype);
static globus_result_t	rrpc_attr_obj_bulk(globus_rls_handle_t *h, BUFFER *bp,
				RLSLIST *rlslist, globus_rls_obj_type_t objtype);
static int		addstr2bulkrc(int rc, char *s1, char *s2, RLSLIST *rlslist);

static int		globus_rls_client_activate(void);
static int		globus_rls_client_deactivate(void);
static globus_result_t	checkhandle(globus_rls_handle_t *h);
static int		connect1(globus_rls_handle_t *h, char *errmsg,
				 int errlen);
static void		free_str2(void *);
static void		free_str2bulk(void *v);
static void		free_attr(void *v);
static void		free_attr_obj(void *v);
static int		rlslist_append(RLSLIST *rlslist, void *datum);
static RLSLIST		*rlslist_new(void (*destructor)());
static char		*translatepattern(char *unixpat, char *sqlpat,int len);
static globus_result_t	mkresult(int rc, char *specific);
#ifdef COMMENT
static globus_bool_t	cauthcb(void *a, globus_io_handle_t *handlep,
			        globus_result_t r, char *identity,
			        gss_ctx_id_t ch);
#endif

globus_module_descriptor_t	globus_rls_client_module = {
  "globus_rls_client",
  globus_rls_client_activate,
  globus_rls_client_deactivate,
  GLOBUS_NULL
};

static globus_mutex_t	active_list_mutex;
static globus_list_t	*active_list;
static char		*iarg(int i, char *buf);

static int
globus_rls_client_activate(void)

{
  globus_mutex_init(&active_list_mutex, GLOBUS_NULL);
  active_list = GLOBUS_NULL;
  globus_mutex_init(&rlslist_mutex, GLOBUS_NULL);
  rlslisttab = NULL;
  rlslist_freelist = NULL;
  return GLOBUS_SUCCESS;
}

static int
globus_rls_client_deactivate(void)

{
  while (!globus_list_empty(active_list))
    globus_rls_client_close(globus_list_first(active_list));
  globus_mutex_destroy(&active_list_mutex);
  return GLOBUS_SUCCESS;
}

void
globus_rls_client_certificate(char *certfile, char *keyfile)

{
  static char	certfileenv[BUFLEN];
  static char	keyfileenv[BUFLEN];

  /*
   * globus_libc_setenv() hangs if the COMMON or IO module is activated
   * first, so use putenv().  globus_libc_unsetenv() seems to work however.
   */
  if (certfile) {
    sprintf(certfileenv, "X509_USER_CERT=%s", certfile);
    putenv(certfileenv);
  }
  if (keyfile) {
    sprintf(keyfileenv, "X509_USER_KEY=%s", keyfile);
    putenv(keyfileenv);
  }
  globus_libc_unsetenv("X509_USER_PROXY");
}

void
globus_rls_client_proxy_certificate(char *proxy)

{
  static char	proxyenv[BUFLEN];

  if (!proxy) {
    globus_libc_unsetenv("X509_USER_PROXY");
    return;
  }
  sprintf(proxyenv, "X509_USER_PROXY=%s", proxy);
  putenv(proxyenv);
}

globus_result_t
globus_rls_client_connect(char *url, globus_rls_handle_t **h)

{
  int		i;
  int		haveurl = 0;
  int		rc;
  globus_list_t	*node;
  char		errmsg[MAXERRMSG];

  if (h == GLOBUS_NULL)
    return mkresult(GLOBUS_RLS_BADARG, NULL);

  *errmsg = '\0';
  if (node = globus_list_search(active_list, *h))
    return mkresult(GLOBUS_RLS_INVHANDLE, NULL);

  if ((*h = globus_libc_malloc(sizeof(globus_rls_handle_t))) == NULL)
    return mkresult(GLOBUS_RLS_NOMEMORY, NULL);

  if ((i = globus_url_parse(url, &(*h)->url)) != GLOBUS_SUCCESS) {
    snprintf(errmsg, MAXERRMSG, "globus_url_parse(%s): Error code %d",
	     url ? url : "NULL", i);
    rc = GLOBUS_RLS_BADURL;
    goto error;
  }
  haveurl++;

  if ((*h)->url.scheme &&
      strcasecmp((*h)->url.scheme, GLOBUS_RLS_URL_SCHEME_NOAUTH) != 0 &&
      strcasecmp((*h)->url.scheme, GLOBUS_RLS_URL_SCHEME) != 0) {
    snprintf(errmsg, MAXERRMSG, "scheme is %s, should be %s",
	     (*h)->url.scheme, GLOBUS_RLS_URL_SCHEME);
    rc = GLOBUS_RLS_BADURL;
    goto error;
  }

  if ((*h)->url.port == 0)
    (*h)->url.port = GLOBUS_RLS_SERVER_DEFPORT;

  if ((rc = connect1(*h, errmsg, MAXERRMSG)) != GLOBUS_RLS_SUCCESS)
    goto error;

  globus_mutex_lock(&active_list_mutex);
  i = globus_list_insert(&active_list, (void *) *h);
  globus_mutex_unlock(&active_list_mutex);
  if (i != 0) {
    rc = GLOBUS_RLS_NOMEMORY;
    goto error;
  }

  return GLOBUS_SUCCESS;

 error:
  /*
   * Some sort of error occured, so clean up and return an error.
   */
  if (haveurl)
    globus_url_destroy(&(*h)->url);
  globus_libc_free(*h);
  return mkresult(rc, errmsg);
}

globus_result_t
globus_rls_client_close(globus_rls_handle_t *h)

{
  static char		*method = "close";
  globus_result_t	r;
  globus_list_t		*node;

  globus_mutex_lock(&active_list_mutex);
  if (!(node = globus_list_search(active_list, (void *) h))) {
    globus_mutex_unlock(&active_list_mutex);
    return mkresult(GLOBUS_RLS_INVHANDLE, NULL);
  }
  if (h->flags & (FH_IOERROR|FH_CLOSED))
    r = GLOBUS_SUCCESS;
  else
    r = rrpc_call(h, NULL, method, 0);
  globus_list_remove(&active_list, node);
  globus_mutex_unlock(&active_list_mutex);
  globus_io_close(&h->handle);
  globus_url_destroy(&h->url);
  globus_libc_free(h);

  return r;
}

int
globus_rls_client_get_timeout()

{
  return rrpc_get_timeout();
}

void
globus_rls_client_set_timeout(int seconds)

{
  rrpc_set_timeout(seconds);
}

globus_result_t
globus_rls_client_admin(globus_rls_handle_t *h, globus_rls_admin_cmd_t cmd)

{
  static char		*method = "admin";
  globus_result_t	r;
  BUFFER		b;
  char			cbuf[50];

  if ((r = checkhandle(h)) != GLOBUS_SUCCESS)
    return r;

  return rrpc_call(h, &b, method, 1, iarg(cmd, cbuf));
}

globus_result_t
globus_rls_client_get_configuration(globus_rls_handle_t *h, char *option,
				    globus_list_t **conf_list)

{
  static char		*method = "get_configuration";
  globus_result_t	r;
  BUFFER		b;
  int			offset = 0;
  RLSLIST		*rlslist;

  if ((r = checkhandle(h)) != GLOBUS_SUCCESS)
    return r;

  if ((rlslist = rlslist_new(free_str2)) == NULL)
    return mkresult(GLOBUS_RLS_NOMEMORY, NULL);

  if ((r = rrpc_call(h, &b, method, 1, option)) != GLOBUS_SUCCESS)
    return r;

  if ((r = rrpc_str2(h, &b, rlslist, &offset)) == GLOBUS_SUCCESS)
    *conf_list = rlslist->list;
  else
    globus_rls_client_free_list(rlslist->list);
  return r;
}

globus_result_t
globus_rls_client_set_configuration(globus_rls_handle_t *h, char *option,
				    char *value)

{
  static char		*method = "set_configuration";
  globus_result_t	r;
  BUFFER		b;

  if ((r = checkhandle(h)) != GLOBUS_SUCCESS)
    return r;

  return rrpc_call(h, &b, method, 2, option, value);
}

globus_result_t
globus_rls_client_lrc_add(globus_rls_handle_t *h, char *lfn, char *pfn)

{
  static char		*method = "lrc_add";
  globus_result_t	r;
  BUFFER		b;

  if ((r = checkhandle(h)) != GLOBUS_SUCCESS)
    return r;

  if (!lfn || !pfn || !*lfn || !*pfn)
    return mkresult(GLOBUS_RLS_BADARG, NULL);

  return rrpc_call(h, &b, method, 2, lfn, pfn);
}

globus_result_t
globus_rls_client_lrc_add_bulk(globus_rls_handle_t *h,
	globus_list_t *str2_list, globus_list_t  **str2bulk_list)

{
  static char	*method = "lrc_add_bulk";

  return bulk_update(method, h, str2_list, str2bulk_list);
}

globus_result_t
globus_rls_client_lrc_clear(globus_rls_handle_t *h)

{
  static char		*method = "lrc_clear";
  globus_result_t	r;
  BUFFER		b;

  if ((r = checkhandle(h)) != GLOBUS_SUCCESS)
    return r;

  return rrpc_call(h, &b, method, 0);
}

globus_result_t
globus_rls_client_lrc_create(globus_rls_handle_t *h, char *lfn, char *pfn)

{
  static char		*method = "lrc_create";
  globus_result_t	r;
  BUFFER		b;

  if ((r = checkhandle(h)) != GLOBUS_SUCCESS)
    return r;

  if (!lfn || !pfn || !*lfn || !*pfn)
    return mkresult(GLOBUS_RLS_BADARG, NULL);

  return rrpc_call(h, &b, method, 2, lfn, pfn);
}

globus_result_t
globus_rls_client_lrc_create_bulk(globus_rls_handle_t *h,
	globus_list_t *str2_list, globus_list_t  **str2bulk_list)

{
  static char	*method = "lrc_create_bulk";

  return bulk_update(method, h, str2_list, str2bulk_list);
}

globus_result_t
globus_rls_client_lrc_delete(globus_rls_handle_t *h, char *lfn, char *pfn)

{
  static char		*method = "lrc_delete";
  globus_result_t	r;
  BUFFER		b;

  if ((r = checkhandle(h)) != GLOBUS_SUCCESS)
    return r;

  if (!lfn || !*lfn)
    return mkresult(GLOBUS_RLS_BADARG, NULL);

  return rrpc_call(h, &b, method, 2, lfn, pfn);
}

globus_result_t
globus_rls_client_lrc_delete_bulk(globus_rls_handle_t *h,
	globus_list_t *str2_list, globus_list_t  **str2bulk_list)

{
  static char	*method = "lrc_delete_bulk";

  return bulk_update(method, h, str2_list, str2bulk_list);
}

globus_result_t
globus_rls_client_lrc_exists(globus_rls_handle_t *h, char *key,
			     globus_rls_obj_type_t objtype)

{
  static char		*method = "lrc_exists";
  globus_result_t	r;
  BUFFER		b;
  char			obuf[BUFLEN];

  if ((r = checkhandle(h)) != GLOBUS_SUCCESS)
    return r;

  if (!key || !*key)
    return mkresult(GLOBUS_RLS_BADARG, NULL);

  return rrpc_call(h, &b, method, 2, key, iarg(objtype, obuf));
}

globus_result_t
globus_rls_client_lrc_exists_bulk(
			globus_rls_handle_t *h, 
			globus_list_t *keylist,
			globus_rls_obj_type_t objtype, 
			globus_list_t **str2bulk_list)
{
  static char		*method = "lrc_exists_bulk";
  
  return exists_bulk2(h, method, objtype, keylist, str2bulk_list);
}

globus_result_t
globus_rls_client_lrc_get_lfn(globus_rls_handle_t *h, char *pfn, int *offset,
			      int reslimit, globus_list_t **str2_list)

{
  static char	*method = "lrc_get_lfn";

  return query2(h, method, pfn, offset, reslimit, str2_list);
}

globus_result_t
globus_rls_client_lrc_get_lfn_bulk(globus_rls_handle_t *h,
	globus_list_t *pfnlist, globus_list_t **str2bulk_list)

{
  static char	*method = "lrc_get_lfn_bulk";

  return query_bulk2(h, method, pfnlist, str2bulk_list);
}

globus_result_t
globus_rls_client_lrc_get_lfn_wc(globus_rls_handle_t *h, char *pfn_pattern,
				 globus_rls_pattern_t type,  int *offset,
				 int reslimit, globus_list_t **str2_list)

{
  static char	*method = "lrc_get_lfn_wc";

  return query_wc2(h, method, pfn_pattern, type, offset, reslimit, str2_list);
}

globus_result_t
globus_rls_client_lrc_get_pfn(globus_rls_handle_t *h, char *lfn, int *offset,
			      int reslimit, globus_list_t **str2_list)

{
  static char	*method = "lrc_get_pfn";

  return query2(h, method, lfn, offset, reslimit, str2_list);
}

globus_result_t
globus_rls_client_lrc_get_pfn_bulk(globus_rls_handle_t *h,
	globus_list_t *lfnlist, globus_list_t **str2bulk_list)

{
  static char	*method = "lrc_get_pfn_bulk";

  return query_bulk2(h, method, lfnlist, str2bulk_list);
}

globus_result_t
globus_rls_client_lrc_get_pfn_wc(globus_rls_handle_t *h, char *lfn_pattern,
				 globus_rls_pattern_t type, int *offset,
				 int reslimit, globus_list_t **str2_list)

{
  static char	*method = "lrc_get_pfn_wc";

  return query_wc2(h, method, lfn_pattern, type, offset, reslimit, str2_list);
}

extern globus_result_t
globus_rls_client_lrc_mapping_exists(globus_rls_handle_t *h, char *lfn,
				     char *pfn)

{
  static char		*method = "lrc_mapping_exists";
  globus_result_t	r;
  BUFFER		b;

  if ((r = checkhandle(h)) != GLOBUS_SUCCESS)
    return r;

  if (!lfn || !*lfn || !pfn || !*pfn)
    return mkresult(GLOBUS_RLS_BADARG, NULL);

  return rrpc_call(h, &b, method, 2, lfn, pfn);
}

globus_result_t
globus_rls_client_lrc_renamelfn(globus_rls_handle_t *h, char *oldname, char *newname)

{
  static char		*method = "lrc_renamelfn";
  globus_result_t	r;
  BUFFER		b;

  if ((r = checkhandle(h)) != GLOBUS_SUCCESS)
    return r;

  if (!oldname || !newname || !*oldname || !*newname)
    return mkresult(GLOBUS_RLS_BADARG, NULL);

  return rrpc_call(h, &b, method, 2, oldname, newname);
}

globus_result_t
globus_rls_client_lrc_renamelfn_bulk(globus_rls_handle_t *h,
	globus_list_t *str2_list, globus_list_t  **str2bulk_list)

{
  static char	*method = "lrc_renamelfn_bulk";

  return bulk_update(method, h, str2_list, str2bulk_list);
}

globus_result_t
globus_rls_client_lrc_renamepfn(globus_rls_handle_t *h, char *oldname, char *newname)

{
  static char		*method = "lrc_renamepfn";
  globus_result_t	r;
  BUFFER		b;

  if ((r = checkhandle(h)) != GLOBUS_SUCCESS)
    return r;

  if (!oldname || !newname || !*oldname || !*newname)
    return mkresult(GLOBUS_RLS_BADARG, NULL);

  return rrpc_call(h, &b, method, 2, oldname, newname);
}

globus_result_t
globus_rls_client_lrc_renamepfn_bulk(globus_rls_handle_t *h,
	globus_list_t *str2_list, globus_list_t  **str2bulk_list)

{
  static char	*method = "lrc_renamepfn_bulk";

  return bulk_update(method, h, str2_list, str2bulk_list);
}

globus_result_t
globus_rls_client_lrc_rli_add(globus_rls_handle_t *h, char *rli_url,
			      int flags, char *pattern)

{
  static char		*method = "lrc_rli_add";
  globus_result_t	r;
  BUFFER		b;
  char			fbuf[100];

  if ((r = checkhandle(h)) != GLOBUS_SUCCESS)
    return r;

  if (!rli_url || !*rli_url)
    return mkresult(GLOBUS_RLS_BADARG, NULL);

  flags &= ~FRLI_RLI;		/* Internal use only, clear if user set	*/
  return rrpc_call(h, &b, method, 3, rli_url, iarg(flags, fbuf), pattern);
}

globus_result_t
globus_rls_client_lrc_rli_delete(globus_rls_handle_t *h, char *rli_url,
				 char *pattern)

{
  static char		*method = "lrc_rli_delete";
  globus_result_t	r;
  BUFFER		b;

  if ((r = checkhandle(h)) != GLOBUS_SUCCESS)
    return r;

  if (!rli_url || !*rli_url)
    return mkresult(GLOBUS_RLS_BADARG, NULL);

  return rrpc_call(h, &b, method, 2, rli_url, pattern);
}

globus_result_t
globus_rls_client_lrc_rli_get_part(globus_rls_handle_t *h, char *rli_url,
		char *pattern, globus_list_t **str2_list)


{
  static char		*method = "lrc_rli_get_part";
  globus_result_t	r;
  BUFFER		b;
  int			offset = 0;
  RLSLIST		*rlslist;

  if ((r = checkhandle(h)) != GLOBUS_SUCCESS)
    return r;

  if ((rlslist = rlslist_new(free_str2)) == NULL)
    return mkresult(GLOBUS_RLS_NOMEMORY, NULL);

  if ((r = rrpc_call(h, &b, method, 2, rli_url, pattern)) != GLOBUS_SUCCESS)
    return r;

  if ((r = rrpc_str2(h, &b, rlslist, &offset)) == GLOBUS_SUCCESS)
    *str2_list = rlslist->list;
  else
    globus_rls_client_free_list(rlslist->list);
  return r;
}

globus_result_t
globus_rls_client_lrc_rli_info(globus_rls_handle_t *h, char *rli,
			       globus_rls_rli_info_t *info)

{
  static char		*method = "lrc_rli_info";
  globus_result_t	r;
  BUFFER		b;
  char			buf[BUFLEN];

  if ((r = checkhandle(h)) != GLOBUS_SUCCESS)
    return r;
  if (!rli || !*rli || !info)
    return mkresult(GLOBUS_RLS_BADARG, NULL);

  if ((r = rrpc_call(h, &b, method, 1, rli)) != GLOBUS_SUCCESS)
    return r;

  if ((r = rrpc_getstr(h, &b, info->url, MAXURL)) != GLOBUS_SUCCESS)
    return r;
  if ((r = rrpc_getstr(h, &b, buf, BUFLEN)) != GLOBUS_SUCCESS)
    return r;
  info->updateinterval = atoi(buf);
  if ((r = rrpc_getstr(h, &b, buf, BUFLEN)) != GLOBUS_SUCCESS)
    return r;
  info->flags = atoi(buf);
  if ((r = rrpc_getstr(h, &b, buf, BUFLEN)) != GLOBUS_SUCCESS)
    return r;
  info->lastupdate = atoi(buf);

  return GLOBUS_SUCCESS;
}

globus_result_t
globus_rls_client_lrc_rli_list(globus_rls_handle_t *h,
			       globus_list_t **rliinfo_list)

{
  static char		*method = "lrc_rli_list";
  globus_result_t	r;
  BUFFER		b;
  RLSLIST		*rlslist;
  int			nomem;
  char			url[MAXURL];
  char			uibuf[100];
  char			fbuf[100];
  char			lubuf[100];
  globus_rls_rli_info_t	*rliinfo;

  if ((r = checkhandle(h)) != GLOBUS_SUCCESS)
    return r;

  if ((r = rrpc_call(h, &b, method, 0)) != GLOBUS_SUCCESS)
    return r;

  if ((rlslist = rlslist_new(globus_libc_free)) == NULL)
    nomem = 1;
  else
    nomem = 0;
  while (1) {
    if ((r = rrpc_getstr(h, &b, url, BUFLEN)) != GLOBUS_SUCCESS)
      return r;
    if (!*url)
      break;
    if ((r = rrpc_getstr(h, &b, uibuf, BUFLEN)) != GLOBUS_SUCCESS)
      return r;
    if ((r = rrpc_getstr(h, &b, fbuf, BUFLEN)) != GLOBUS_SUCCESS)
      return r;
    if ((r = rrpc_getstr(h, &b, lubuf, BUFLEN)) != GLOBUS_SUCCESS)
      return r;

    if (!(rliinfo = globus_libc_malloc(sizeof(*rliinfo)))) {
      nomem++;
      continue;
    }
    strcpy(rliinfo->url, url);
    rliinfo->updateinterval = atoi(uibuf);
    rliinfo->flags = atoi(fbuf);
    rliinfo->lastupdate = atoi(lubuf);

    if (globus_list_insert(&rlslist->list, rliinfo) != GLOBUS_SUCCESS) {
      globus_libc_free(rliinfo);
      nomem++;
      continue;
    }
  }
  if (nomem && rlslist)
    globus_rls_client_free_list(rlslist->list);
  else
    *rliinfo_list = rlslist->list;

  if (nomem)
    return mkresult(GLOBUS_RLS_NOMEMORY, NULL);
  return GLOBUS_SUCCESS;
}

globus_result_t
globus_rls_client_rli_exists(globus_rls_handle_t *h, char *key,
			     globus_rls_obj_type_t objtype)

{
  static char		*method = "rli_exists";
  globus_result_t	r;
  BUFFER		b;
  char			obuf[BUFLEN];

  if ((r = checkhandle(h)) != GLOBUS_SUCCESS)
    return r;

  if (!key || !*key)
    return mkresult(GLOBUS_RLS_BADARG, NULL);

  return rrpc_call(h, &b, method, 2, key, iarg(objtype, obuf));
}

globus_result_t
globus_rls_client_rli_exists_bulk(
			globus_rls_handle_t *h, 
			globus_list_t *keylist,
			globus_rls_obj_type_t objtype, 
			globus_list_t **str2bulk_list)
{
  static char	*method = "rli_exists_bulk";

  return exists_bulk2(h, method, objtype, keylist, str2bulk_list);
}

globus_result_t
globus_rls_client_rli_get_lrc(globus_rls_handle_t *h, char *lfn, int *offset,
			      int reslimit, globus_list_t **str2_list)

{
  static char	*method = "rli_get_lrc";

  return query2(h, method, lfn, offset, reslimit, str2_list);
}

globus_result_t
globus_rls_client_rli_get_lrc_bulk(globus_rls_handle_t *h,
	globus_list_t *lfnlist, globus_list_t **str2bulk_list)

{
  static char	*method = "rli_get_lrc_bulk";

  return query_bulk2(h, method, lfnlist, str2bulk_list);
}

globus_result_t
globus_rls_client_rli_get_lrc_wc(globus_rls_handle_t *h, char *lfn_pattern,
				 globus_rls_pattern_t type, int *offset,
				 int reslimit, globus_list_t **str2_list)

{
  static char	*method = "rli_get_lrc_wc";

  return query_wc2(h, method, lfn_pattern, type, offset, reslimit, str2_list);
}

/*
 * Deprecated.
 */
globus_result_t
globus_rls_client_rli_lrc_list(globus_rls_handle_t *h,
			       globus_list_t **senderinfo_list)

{
  return globus_rls_client_rli_sender_list(h, senderinfo_list);
}

globus_result_t
globus_rls_client_rli_sender_list(globus_rls_handle_t *h,
				  globus_list_t **senderinfo_list)

{
  static char			*method = "rli_sender_list";
  globus_result_t		r;
  BUFFER			b;
  int				nomem;
  char				url[MAXURL];
  char				lubuf[BUFLEN];
  RLSLIST			*rlslist;
  globus_rls_sender_info_t	*senderinfo;

  if ((r = checkhandle(h)) != GLOBUS_SUCCESS)
    return r;

  if ((r = rrpc_call(h, &b, method, 0)) != GLOBUS_SUCCESS)
    return r;

  if ((rlslist = rlslist_new(globus_libc_free)) == NULL)
    nomem = 1;
  else
    nomem = 0;
  while (1) {
    if ((r = rrpc_getstr(h, &b, url, BUFLEN)) != GLOBUS_SUCCESS)
      return r;
    if (!*url)
      break;
    if ((r = rrpc_getstr(h, &b, lubuf, BUFLEN)) != GLOBUS_SUCCESS)
      return r;
    if (!(senderinfo = globus_libc_malloc(sizeof(*senderinfo)))) {
      nomem++;
      continue;
    }
    strcpy(senderinfo->url, url);
    senderinfo->lastupdate = atoi(lubuf);
    if (globus_list_insert(&rlslist->list, senderinfo) != GLOBUS_SUCCESS) {
      globus_libc_free(senderinfo);
      nomem++;
      continue;
    }
  }

  if (nomem && rlslist)
    globus_rls_client_free_list(rlslist->list);
  else
    *senderinfo_list = rlslist->list;

  if (nomem)
    return mkresult(GLOBUS_RLS_NOMEMORY, NULL);
  return GLOBUS_SUCCESS;
}

extern globus_result_t
globus_rls_client_rli_mapping_exists(globus_rls_handle_t *h, char *lfn,
				     char *lrc)

{
  static char		*method = "rli_mapping_exists";
  globus_result_t	r;
  BUFFER		b;

  if ((r = checkhandle(h)) != GLOBUS_SUCCESS)
    return r;

  if (!lfn || !*lfn || !lrc || !*lrc)
    return mkresult(GLOBUS_RLS_BADARG, NULL);

  return rrpc_call(h, &b, method, 2, lfn, lrc);
}

globus_result_t
globus_rls_client_rli_rli_add(globus_rls_handle_t *h, char *rli_url,
			      char *pattern)

{
  static char		*method = "rli_rli_add";
  globus_result_t	r;
  BUFFER		b;

  if ((r = checkhandle(h)) != GLOBUS_SUCCESS)
    return r;

  if (!rli_url || !*rli_url)
    return mkresult(GLOBUS_RLS_BADARG, NULL);

  return rrpc_call(h, &b, method, 2, rli_url, pattern);
}

globus_result_t
globus_rls_client_rli_rli_delete(globus_rls_handle_t *h, char *rli_url,
				 char *pattern)

{
  static char		*method = "rli_rli_delete";
  globus_result_t	r;
  BUFFER		b;

  if ((r = checkhandle(h)) != GLOBUS_SUCCESS)
    return r;

  if (!rli_url || !*rli_url)
    return mkresult(GLOBUS_RLS_BADARG, NULL);

  return rrpc_call(h, &b, method, 2, rli_url, pattern);
}

globus_result_t
globus_rls_client_rli_rli_get_part(globus_rls_handle_t *h, char *rli_url,
		char *pattern, globus_list_t **str2_list)


{
  static char		*method = "rli_rli_get_part";
  globus_result_t	r;
  BUFFER		b;
  int			offset = 0;
  RLSLIST		*rlslist;

  if ((r = checkhandle(h)) != GLOBUS_SUCCESS)
    return r;

  if ((rlslist = rlslist_new(free_str2)) == NULL)
    return mkresult(GLOBUS_RLS_NOMEMORY, NULL);

  if ((r = rrpc_call(h, &b, method, 2, rli_url, pattern)) != GLOBUS_SUCCESS)
    return r;

  if ((r = rrpc_str2(h, &b, rlslist, &offset)) == GLOBUS_SUCCESS)
    *str2_list = rlslist->list;
  else
    globus_rls_client_free_list(rlslist->list);
  return r;
}

globus_result_t
globus_rls_client_rli_rli_list(globus_rls_handle_t *h,
			       globus_list_t **rliinfo_list)

{
  static char		*method = "rli_rli_list";
  globus_result_t	r;
  BUFFER		b;
  RLSLIST		*rlslist;
  int			nomem;
  char			url[MAXURL];
  char			uibuf[100];
  char			fbuf[100];
  char			lubuf[100];
  globus_rls_rli_info_t	*rliinfo;

  if ((r = checkhandle(h)) != GLOBUS_SUCCESS)
    return r;

  if ((r = rrpc_call(h, &b, method, 0)) != GLOBUS_SUCCESS)
    return r;

  if ((rlslist = rlslist_new(globus_libc_free)) == NULL)
    nomem = 1;
  else
    nomem = 0;
  while (1) {
    if ((r = rrpc_getstr(h, &b, url, BUFLEN)) != GLOBUS_SUCCESS)
      return r;
    if (!*url)
      break;
    if ((r = rrpc_getstr(h, &b, uibuf, BUFLEN)) != GLOBUS_SUCCESS)
      return r;
    if ((r = rrpc_getstr(h, &b, fbuf, BUFLEN)) != GLOBUS_SUCCESS)
      return r;
    if ((r = rrpc_getstr(h, &b, lubuf, BUFLEN)) != GLOBUS_SUCCESS)
      return r;

    if (!(rliinfo = globus_libc_malloc(sizeof(*rliinfo)))) {
      nomem++;
      continue;
    }
    strcpy(rliinfo->url, url);
    rliinfo->updateinterval = atoi(uibuf);
    rliinfo->flags = atoi(fbuf);
    rliinfo->lastupdate = atoi(lubuf);

    if (globus_list_insert(&rlslist->list, rliinfo) != GLOBUS_SUCCESS) {
      globus_libc_free(rliinfo);
      nomem++;
      continue;
    }
  }
  if (nomem && rlslist)
    globus_rls_client_free_list(rlslist->list);
  else
    *rliinfo_list = rlslist->list;

  if (nomem)
    return mkresult(GLOBUS_RLS_NOMEMORY, NULL);
  return GLOBUS_SUCCESS;
}

globus_result_t
globus_rls_client_stats(globus_rls_handle_t *h, globus_rls_stats_t *rlsstats)

{
  static char		*method = "stats";
  globus_result_t	r;
  BUFFER		b;
  char			buf[BUFLEN];

  if ((r = checkhandle(h)) != GLOBUS_SUCCESS)
    return r;

  if ((r = rrpc_call(h, &b, method, 0)) != GLOBUS_SUCCESS)
    return r;

  if ((r = rrpc_getstr(h, &b, buf, BUFLEN)) != GLOBUS_SUCCESS)
    return r;
  strncpy(rlsstats->version, buf, MAXVERSION);

  if ((r = rrpc_getstr(h, &b, buf, BUFLEN)) != GLOBUS_SUCCESS)
    return r;
  rlsstats->uptime = atoi(buf);

  if ((r = rrpc_getstr(h, &b, buf, BUFLEN)) != GLOBUS_SUCCESS)
    return r;
  rlsstats->flags = atoi(buf);

  if ((r = rrpc_getstr(h, &b, buf, BUFLEN)) != GLOBUS_SUCCESS)
    return r;
  rlsstats->lrc_bloomfilterui = atoi(buf);

  if ((r = rrpc_getstr(h, &b, buf, BUFLEN)) != GLOBUS_SUCCESS)
    return r;
  rlsstats->lrc_lfnlistui = atoi(buf);

  if ((r = rrpc_getstr(h, &b, buf, BUFLEN)) != GLOBUS_SUCCESS)
    return r;
  rlsstats->lrc_numlfn = atoi(buf);

  if ((r = rrpc_getstr(h, &b, buf, BUFLEN)) != GLOBUS_SUCCESS)
    return r;
  rlsstats->lrc_numpfn = atoi(buf);

  if ((r = rrpc_getstr(h, &b, buf, BUFLEN)) != GLOBUS_SUCCESS)
    return r;
  rlsstats->lrc_nummap = atoi(buf);

  if ((r = rrpc_getstr(h, &b, buf, BUFLEN)) != GLOBUS_SUCCESS)
    return r;
  rlsstats->rli_numlfn = atoi(buf);

  if ((r = rrpc_getstr(h, &b, buf, BUFLEN)) != GLOBUS_SUCCESS)
    return r;
  rlsstats->rli_numlrc = atoi(buf);

  if ((r = rrpc_getstr(h, &b, buf, BUFLEN)) != GLOBUS_SUCCESS)
    return r;
  rlsstats->rli_numsender = atoi(buf);

  if ((r = rrpc_getstr(h, &b, buf, BUFLEN)) != GLOBUS_SUCCESS)
    return r;
  rlsstats->rli_nummap = atoi(buf);

  return GLOBUS_SUCCESS;
}

globus_result_t
globus_rls_client_lrc_attr_add(globus_rls_handle_t *h, char *key,
			       globus_rls_attribute_t *attr)

{
  static char		*method = "lrc_attr_add";
  globus_result_t	r;
  BUFFER		b;
  char			obuf[100];
  char			tbuf[100];
  char			vbuf[BUFLEN];
  char			*v;

  if ((r = checkhandle(h)) != GLOBUS_SUCCESS)
    return r;

  if ((v = globus_rls_client_attr2s(attr, vbuf, BUFLEN)) == NULL)
    return mkresult(GLOBUS_RLS_INV_ATTR_TYPE, NULL);

  return rrpc_call(h, &b, method, 5, key, iarg(attr->objtype, obuf),
		   iarg(attr->type, tbuf), attr->name, v);
}

globus_result_t
globus_rls_client_lrc_attr_add_bulk(globus_rls_handle_t *h,
	globus_list_t *attr_obj_list, globus_list_t **str2bulk_list)

{
  return bulk_attr_update(1, h, attr_obj_list, str2bulk_list);
}

globus_result_t
globus_rls_client_lrc_attr_create(globus_rls_handle_t *h, char *name,
		globus_rls_obj_type_t objtype, globus_rls_attr_type_t type)

{
  static char		*method = "lrc_attr_create";
  globus_result_t	r;
  BUFFER		b;
  char			obuf[BUFLEN];
  char			tbuf[BUFLEN];

  if ((r = checkhandle(h)) != GLOBUS_SUCCESS)
    return r;

  return rrpc_call(h, &b, method, 3, name, iarg(objtype, obuf),
		   iarg(type, tbuf));
}

globus_result_t
globus_rls_client_lrc_attr_delete(globus_rls_handle_t *h, char *name,
		globus_rls_obj_type_t objtype, globus_bool_t clearvalues)

{
  static char		*method = "lrc_attr_delete";
  globus_result_t	r;
  BUFFER		b;
  char			obuf[100];
  char			buf[100];

  if ((r = checkhandle(h)) != GLOBUS_SUCCESS)
    return r;

  return rrpc_call(h, &b, method, 3, name, iarg(objtype, obuf),
		   iarg(clearvalues, buf));
}

globus_result_t
globus_rls_client_lrc_attr_get(globus_rls_handle_t *h, char *name,
	globus_rls_obj_type_t objtype, globus_list_t **attr_list)

{
  static char			*method = "lrc_attr_get";
  globus_result_t		r;
  BUFFER			b;
  char				obuf[100];
  char				nbuf[BUFLEN];
  char				tbuf[100];
  globus_rls_attribute_t	*attr;
  int				nomem;
  RLSLIST			*rlslist;

  if ((r = checkhandle(h)) != GLOBUS_SUCCESS)
    return r;

  if ((r = rrpc_call(h, &b, method, 2, name,
		     iarg(objtype, obuf))) != GLOBUS_SUCCESS)
    return r;

  if ((rlslist = rlslist_new(free_attr)) == NULL)
    nomem = 1;
  else
    nomem = 0;
  while (1) {
    if ((r = rrpc_getstr(h, &b, nbuf, BUFLEN)) != GLOBUS_SUCCESS)
      return r;
    if (!*nbuf)
      break;
    if ((r = rrpc_getstr(h, &b, tbuf, BUFLEN)) != GLOBUS_SUCCESS)
      return r;
    if (!(attr = globus_libc_calloc(1, sizeof(*attr)))) {
      nomem++;
      continue;
    }
    if ((attr->name = globus_libc_strdup(nbuf)) == NULL) {
      free_attr(attr);
      nomem++;
      continue;
    }
    attr->type = atoi(tbuf);
    attr->objtype = objtype;
    if (globus_list_insert(&rlslist->list, attr) != GLOBUS_SUCCESS) {
      free_attr(attr);
      nomem++;
      continue;
    }
  }
  if (nomem && rlslist)
    globus_rls_client_free_list(rlslist->list);
  else
    *attr_list = rlslist->list;

  if (nomem)
    return mkresult(GLOBUS_RLS_NOMEMORY, NULL);
  return GLOBUS_SUCCESS;
}

globus_result_t
globus_rls_client_lrc_attr_modify(globus_rls_handle_t *h, char *key,
				  globus_rls_attribute_t *attr)

{
  static char		*method = "lrc_attr_modify";
  globus_result_t	r;
  BUFFER		b;
  char			obuf[100];
  char			tbuf[100];
  char			vbuf[BUFLEN];

  if ((r = checkhandle(h)) != GLOBUS_SUCCESS)
    return r;

  return rrpc_call(h, &b, method, 5, key, attr->name,
		   iarg(attr->objtype, obuf), iarg(attr->type, tbuf),
		   globus_rls_client_attr2s(attr, vbuf, BUFLEN));
}

globus_result_t
globus_rls_client_lrc_attr_remove(globus_rls_handle_t *h, char *key,
				  globus_rls_attribute_t *attr)

{
  static char	*method = "lrc_attr_remove";
  globus_result_t	r;
  BUFFER		b;
  char			obuf[100];

  if ((r = checkhandle(h)) != GLOBUS_SUCCESS)
    return r;

  return rrpc_call(h, &b, method, 3, key, attr->name,iarg(attr->objtype,obuf));
}

globus_result_t
globus_rls_client_lrc_attr_remove_bulk(globus_rls_handle_t *h,
	globus_list_t *attr_obj_list, globus_list_t **str2bulk_list)

{
  return bulk_attr_update(0, h, attr_obj_list, str2bulk_list);
}

globus_result_t
globus_rls_client_lrc_attr_search(globus_rls_handle_t *h, char *name,
	globus_rls_obj_type_t objtype, globus_rls_attr_op_t op,
	globus_rls_attribute_t *operand1, globus_rls_attribute_t *operand2,
	int *offset, int reslimit, globus_list_t **attr_obj_list)

{
  static char			*method = "lrc_attr_search";
  globus_result_t		r;
  BUFFER			b;
  char				obuf[50];
  char				operbuf[50];
  char				o1buf[BUFLEN];
  char				*o1;
  char				o2buf[BUFLEN];
  char				*o2;
  char				offbuf[50];
  char				rbuf[50];
  int				loffset;
  RLSLIST			*rlslist;

  if ((r = checkhandle(h)) != GLOBUS_SUCCESS)
    return r;

  if (!offset) {
    loffset = 0;
    offset = &loffset;
  }
  iarg(reslimit, rbuf);
  iarg(objtype, obuf);
  iarg(op, operbuf);

  if (operand1) {	/* not used if op is "all"			*/
    if ((o1 = globus_rls_client_attr2s(operand1, o1buf, BUFLEN)) == NULL)
      return mkresult(GLOBUS_RLS_INV_ATTR_TYPE, NULL);
  } else
    o1 = "";
  if (operand2) {	/* operand2 only used if op is "between"	*/
    if ((o2 = globus_rls_client_attr2s(operand2, o2buf, BUFLEN)) == NULL)
      return mkresult(GLOBUS_RLS_INV_ATTR_TYPE, NULL);
  } else
    o2 = "";

  if ((rlslist = rlslist_new(free_attr_obj)) == NULL)
    return mkresult(GLOBUS_RLS_NOMEMORY, NULL);

  if (offset == &loffset) {
    while ((r = rrpc_call(h, &b, method, 7, name, obuf, operbuf, o1, o2,
			  iarg(*offset, offbuf), rbuf)) == GLOBUS_SUCCESS) {
      r = rrpc_attr_obj(h, &b, rlslist, offset, name, objtype);
      if (r != GLOBUS_SUCCESS || *offset == -1)
	break;
    }
  } else
    if ((r = rrpc_call(h, &b, method, 7, name, obuf, operbuf, o1, o2,
		       iarg(*offset, offbuf), rbuf)) == GLOBUS_SUCCESS)
      r = rrpc_attr_obj(h, &b, rlslist, offset, name, objtype);


  if (r == GLOBUS_SUCCESS)
    *attr_obj_list = rlslist->list;
  else
    globus_rls_client_free_list(rlslist->list);
  return r;
}

globus_result_t
globus_rls_client_lrc_attr_value_get(globus_rls_handle_t *h, char *key,
			char *name, globus_rls_obj_type_t objtype,
			globus_list_t **attr_list)

{
  static char			*method = "lrc_attr_value_get";
  globus_result_t		r;
  globus_result_t		saver;
  BUFFER			b;
  char				obuf[100];
  char				nbuf[BUFLEN];
  char				tbuf[BUFLEN];
  char				sval[BUFLEN];
  globus_rls_attribute_t	*attr;
  int				nomem;
  RLSLIST			*rlslist;

  if ((r = checkhandle(h)) != GLOBUS_SUCCESS)
    return r;

  if ((r = rrpc_call(h, &b, method, 3, key, name,
		     iarg(objtype, obuf))) != GLOBUS_SUCCESS)
    return r;

  saver = GLOBUS_SUCCESS;
  if ((rlslist = rlslist_new(free_attr)) == NULL)
    nomem = 1;
  else
    nomem = 0;
  while (1) {
    if ((r = rrpc_getstr(h, &b, nbuf, BUFLEN)) != GLOBUS_SUCCESS)
      return r;
    if (!*nbuf)
      break;
    if ((r = rrpc_getstr(h, &b, tbuf, BUFLEN)) != GLOBUS_SUCCESS)
      return r;
    if ((r = rrpc_getstr(h, &b, sval, BUFLEN)) != GLOBUS_SUCCESS)
      return r;

    if (!(attr = globus_libc_calloc(1, sizeof(*attr)))) {
      nomem++;
      continue;
    }
    if ((attr->name = globus_libc_strdup(nbuf)) == NULL) {
      free_attr(attr);
      nomem++;
      continue;
    }
    if ((r = globus_rls_client_s2attr(atoi(tbuf), sval,
				      attr)) != GLOBUS_RLS_SUCCESS) {
      saver = r;
      free_attr(attr);
      continue;
    }
    attr->objtype = objtype;
    if (globus_list_insert(&rlslist->list, attr) != GLOBUS_SUCCESS) {
      free_attr(attr);
      nomem++;
      continue;
    }

  }
  if (nomem && rlslist)
    globus_rls_client_free_list(rlslist->list);
  else
    *attr_list = rlslist->list;

  if (nomem)
    return mkresult(GLOBUS_RLS_NOMEMORY, NULL);
  return saver;
}

globus_result_t
globus_rls_client_lrc_attr_value_get_bulk(globus_rls_handle_t *h,
	globus_list_t *keylist, char *name, globus_rls_obj_type_t objtype,
	globus_list_t **attr_obj_list)

{
  static char		*method = "lrc_attr_value_get_bulk";
  globus_result_t	r;
  int			rc;
  BUFFER		b;
  BUFFER		obuf;
  globus_list_t		*p;
  char			buf[100];
  int			offset = 0;
  char			errmsg[MAXERRMSG];
  char			*s;
  RLSLIST		*rlslist;
  
  if ((r = checkhandle(h)) != GLOBUS_SUCCESS)
    return r;

  if (!keylist)
    return mkresult(GLOBUS_RLS_BADARG, NULL);

  if (h->flags & FH_CLOSED)
    if ((rc = connect1(h, errmsg, MAXERRMSG)) != GLOBUS_RLS_SUCCESS)
      return mkresult(rc, errmsg);

  rrpc_initbuf(&obuf);
  rc = rrpc_bufwrite(&h->handle, &obuf, method, strlen(method)+1, 0, errmsg);
  if (rc == GLOBUS_RLS_SUCCESS) {
    if (name == NULL)
      name = "";
    rc = rrpc_bufwrite(&h->handle, &obuf, name, strlen(name) + 1, 0, errmsg);
  }
  if (rc == GLOBUS_RLS_SUCCESS) {
    iarg(objtype, buf);
    rc = rrpc_bufwrite(&h->handle, &obuf, buf, strlen(buf) + 1, 0, errmsg);
  }
  if (rc != GLOBUS_RLS_SUCCESS) {
    h->flags |= FH_IOERROR;
    return mkresult(rc, errmsg);
  }
  for (p = keylist; p; p = globus_list_rest(p)) {
    s = globus_list_first(p);
    if ((rc = rrpc_bufwrite(&h->handle, &obuf, s, strlen(s) + 1,
			    0, errmsg)) != GLOBUS_RLS_SUCCESS) {
      h->flags |= FH_IOERROR;
      return mkresult(rc, errmsg);
    }
  }
  if ((rc = rrpc_bufwrite(&h->handle, &obuf, "", 1, 1,
			  errmsg)) != GLOBUS_RLS_SUCCESS) {
    h->flags |= FH_IOERROR;
    return mkresult(rc, errmsg);
  }
  if ((rc = rrpc_getresult(h, &b, errmsg)) != GLOBUS_SUCCESS)
    return mkresult(rc, errmsg);

  if ((rlslist = rlslist_new(free_attr_obj)) == NULL)
    return mkresult(GLOBUS_RLS_NOMEMORY, NULL);
  if ((r = rrpc_attr_obj_bulk(h, &b, rlslist, objtype)) == GLOBUS_SUCCESS)
    *attr_obj_list = rlslist->list;
  else
    globus_rls_client_free_list(rlslist->list);
  return r;
}

/*
 * Each globus_list_t returned by the API is remembered in rlslisttab,
 * along with a destructor function to free each list element (if
 * globus_list_t supported destructors this wouldn't be necessary).
 * This function removes list from rlslisttab, calls the destructor on
 * each element, and finally frees list.
 */
globus_result_t
globus_rls_client_free_list(globus_list_t *list)

{
  RLSLIST	*rlslist;
  RLSLIST	*p_rlslist;
  globus_list_t	*p;
  void		(*destructor)(void *);
  void		*datum;

  /*
   * Find list in rlslisttab, remember previous element so can remove.
   */
  globus_mutex_lock(&rlslist_mutex);
  p_rlslist = NULL;
  for (rlslist = rlslisttab; rlslist; rlslist = rlslist->nxt) {
    if (rlslist->list == list)
      break;
    p_rlslist = rlslist;
  }

  /*
   * rlslist points at RLSLIST element containing list, p_rlslist points
   * at previous RLSLIST element in rlslisttab, or we didn't find list
   * in rlslisttab and rlslist is NULL.
   */
  if (!rlslist) {
    globus_mutex_unlock(&rlslist_mutex);
    return mkresult(GLOBUS_RLS_BADARG, NULL);
  }

  /*
   * Remove rlslist from rlslisttab and add to free list.
   */
  if (p_rlslist)
    p_rlslist->nxt = rlslist->nxt;
  else
    rlslisttab = rlslist->nxt;
  rlslist->nxt = rlslist_freelist;
  rlslist_freelist = rlslist;

  /*
   * Remember destructor so can unlock rlslisttab asap.
   */
  destructor = rlslist->destructor;
  globus_mutex_unlock(&rlslist_mutex);
  
  /*
   * Call destructor on each list element, then free list.
   */
  if (list) {
    for (p = list; p; p = globus_list_rest(p))
      if (datum = globus_list_first(p))
	destructor(datum);
    globus_list_free(list);
  }

  return GLOBUS_SUCCESS;
}

globus_result_t
globus_rls_client_error_info(globus_result_t r, int *rc, char *buf, int buflen,
			     globus_bool_t preserve)

{
  globus_object_t	*o;
  char			*s;

  o = globus_error_get(r);
  if (rc)
    *rc = globus_error_get_type(o);
  if (buf) {
    s = globus_error_print_friendly(o);
    strncpy(buf, s, buflen);
    globus_libc_free(s);
  }
  if (preserve)
    return globus_error_put(o);
  else
    globus_object_free(o);
  return NULL;
}

/*
 * Convert globus_rls_attribute_t to string.
 */
char *
globus_rls_client_attr2s(globus_rls_attribute_t *attr, char *sval, int svallen)

{
  switch (attr->type) {
    case globus_rls_attr_type_date:
      mycftime(sval, svallen, "%Y-%m-%d %H:%M:%S", attr->val.t);
      return sval;
    case globus_rls_attr_type_flt:	
      snprintf(sval, svallen, "%f", attr->val.d);
      return sval;
    case globus_rls_attr_type_int:	
      snprintf(sval, svallen, "%d", attr->val.i);
      return sval;
    case globus_rls_attr_type_str:	
      return attr->val.s;
    default:
      return NULL;
  }
}

/*
 * Convert string attribute value to globus_rls_attribute_t.  Returns
 * GLOBUS_SUCCESS if ok, or an error if type is bad or malloc fails.
 * If type is globus_rls_attr_type_str then val.s will be malloc'ed and
 * must be freed by caller.
 */
globus_result_t
globus_rls_client_s2attr(globus_rls_attr_type_t type, char *sval,
			 globus_rls_attribute_t *attr)

{
  struct tm	t;

  switch (type) {
    case globus_rls_attr_type_date:
      if (!strptime(sval, "%Y-%m-%d %H:%M:%S", &t))
        return mkresult(GLOBUS_RLS_BADARG, sval);
      t.tm_isdst = -1;
      attr->val.t = mktime(&t);
      break;
    case globus_rls_attr_type_flt:
      attr->val.d = atof(sval);
      break;
    case globus_rls_attr_type_int:
      attr->val.i = atoi(sval);
      break;
    case globus_rls_attr_type_str:
      if ((attr->val.s = globus_libc_strdup(sval)) == NULL)
	return mkresult(GLOBUS_RLS_NOMEMORY, NULL);
      break;
    default:
      return mkresult(GLOBUS_RLS_INV_ATTR_TYPE, NULL);
  }
  attr->type = type;
  return GLOBUS_SUCCESS;
}

/*
 * For internal use only, constructs error message from base message for
 * rc and error specific message.  Exported in client API cause server
 * and this API both need it.
 */
char *
globus_rls_errmsg(int rc, char *specificmsg, char *buf, int buflen)

{
  /* Must match defines in globus_rls_client.h */
  static char	*baseerrmsg[] = {
    "No error",
    "Globus I/O error",
    "Invalid handle",
    "Bad URL",
    "No memory",
    "Buffer overflow",
    "Bad argument",
    "Permission denied",
    "Bad method name",
    "Invalid server",
    "Mapping doesn't exist",
    "LFN already exists",
    "LFN doesn't exist",
    "PFN already exists",
    "PFN doesn't exist",
    "LRC already exists",
    "LRC doesn't exist",
    "DB error",
    "RLI already exists",
    "RLI doesn't exist",
    "Mapping already exists",
    "Invalid attribute type",
    "Attribute already exists",
    "Attribute doesn't exist",
    "Invalid object type",
    "Invalid operator for attribute search",
    "Operation is unsupported",
    "IO timeout",
    "Too many connections",
    "Attribute with specified value doesn't exist",
    "Attribute still referenced by objects in database",
  };
#define GLOBUS_RLS_NUMERRORS	(sizeof(baseerrmsg) /  sizeof(char *))

  if (rc < 0 || rc >= GLOBUS_RLS_NUMERRORS)
    globus_libc_sprintf(buf, "Unknown error %d", rc);
  else if (specificmsg && *specificmsg)
    if (baseerrmsg[rc])
      snprintf(buf, buflen, "%s: %s", baseerrmsg[rc], specificmsg);
    else
      strncpy(buf, specificmsg, buflen);
  else
    strncpy(buf, baseerrmsg[rc], buflen);
  return buf;
}

/*
 * Bulk exists that returns list of strings (eg lrc_exists_bulk,
 * rli_exists_bulk).
 */
static globus_result_t
exists_bulk2(globus_rls_handle_t *h, char *method, globus_rls_obj_type_t objtype,
	    globus_list_t *keylist,
	    globus_list_t **str2bulk_list)

{
  globus_result_t	r;
  BUFFER		b;
  RLSLIST		*rlslist;
  globus_list_t		*p;
  char			errmsg[MAXERRMSG];
  int			rc;
  BUFFER		obuf;
  char			*s;
  char			objbuf[BUFLEN];

  if ((r = checkhandle(h)) != GLOBUS_SUCCESS)
    return r;

  if (!keylist)
    return mkresult(GLOBUS_RLS_BADARG, "key list is NULL");

  if (h->flags & FH_CLOSED)
    if ((rc = connect1(h, errmsg, MAXERRMSG)) != GLOBUS_RLS_SUCCESS)
      return mkresult(rc, errmsg);

  rrpc_initbuf(&obuf);
  if ((rc = rrpc_bufwrite(&h->handle, &obuf, method, strlen(method) + 1,
			  0, errmsg)) != GLOBUS_RLS_SUCCESS) {
    h->flags |= FH_IOERROR;
    return mkresult(rc, errmsg);
  }

  iarg(objtype, objbuf);
  if ((rc = rrpc_bufwrite(&h->handle, &obuf, objbuf, strlen(objbuf) + 1,
			  0, errmsg)) != GLOBUS_RLS_SUCCESS) {
    h->flags |= FH_IOERROR;
    return mkresult(rc, errmsg);
  }

  for (p = keylist; p; p = globus_list_rest(p)) {
    s = globus_list_first(p);
    if ((rc = rrpc_bufwrite(&h->handle, &obuf, s, strlen(s) + 1, 0,
			    errmsg)) != GLOBUS_RLS_SUCCESS) {
      h->flags |= FH_IOERROR;
      return mkresult(rc, errmsg);
    }
  }
  if ((rc = rrpc_bufwrite(&h->handle, &obuf, "", 1, 1,
			  errmsg)) != GLOBUS_RLS_SUCCESS) {
    h->flags |= FH_IOERROR;
    return mkresult(rc, errmsg);
  }

  if ((rlslist = rlslist_new(free_str2bulk)) == NULL)
    return mkresult(GLOBUS_RLS_NOMEMORY, NULL);
  if ((r = rrpc_str2bulk(h, &b, 0, rlslist)) == GLOBUS_SUCCESS)
    *str2bulk_list = rlslist->list;
  else
    globus_rls_client_free_list(rlslist->list);
  return r;
}

/*
 * Query that returns pairs of strings (eg lrc_get_lfn,
 * lrc_get_pfn, rli_get_lrc).
 */
static globus_result_t
query2(globus_rls_handle_t *h, char *method, char *key, int *offset,
       int reslimit, globus_list_t **str2_list)

{
  globus_result_t	r;
  BUFFER		b;
  char			obuf[50];
  char			rbuf[50];
  int			loffset;
  RLSLIST		*rlslist;

  if ((r = checkhandle(h)) != GLOBUS_SUCCESS)
    return r;

  if (!key || !*key)
    return mkresult(GLOBUS_RLS_BADARG, "key is NULL");

  if (!offset) {
    loffset = 0;
    offset = &loffset;
  }
  iarg(reslimit, rbuf);

  if ((rlslist = rlslist_new(free_str2)) == NULL)
    return mkresult(GLOBUS_RLS_NOMEMORY, NULL);

  if (offset == &loffset) {
    while ((r = rrpc_call(h, &b, method, 3, key, iarg(*offset, obuf),
			  rbuf)) == GLOBUS_SUCCESS) {
      r = rrpc_str2(h, &b, rlslist, offset);
      if (r != GLOBUS_SUCCESS || *offset == -1)
	break;
    }
  } else
    if ((r = rrpc_call(h, &b, method, 3, key, iarg(*offset, obuf),
		       rbuf)) == GLOBUS_SUCCESS)
      r = rrpc_str2(h, &b, rlslist, offset);

  if (r == GLOBUS_SUCCESS)
    *str2_list = rlslist->list;
  else
    globus_rls_client_free_list(rlslist->list);
  return r;
}

/*
 * Bulk query that returns pairs of strings (eg lrc_get_lfn,
 * lrc_get_pfn, rli_get_lrc).
 */
static globus_result_t
query_bulk2(globus_rls_handle_t *h, char *method, globus_list_t *keylist,
	    globus_list_t **str2bulk_list)

{
  globus_result_t	r;
  BUFFER		b;
  RLSLIST		*rlslist;
  globus_list_t		*p;
  char			errmsg[MAXERRMSG];
  int			rc;
  BUFFER		obuf;
  char			*s;

  if ((r = checkhandle(h)) != GLOBUS_SUCCESS)
    return r;

  if (!keylist)
    return mkresult(GLOBUS_RLS_BADARG, "key list is NULL");

  if (h->flags & FH_CLOSED)
    if ((rc = connect1(h, errmsg, MAXERRMSG)) != GLOBUS_RLS_SUCCESS)
      return mkresult(rc, errmsg);

  rrpc_initbuf(&obuf);
  if ((rc = rrpc_bufwrite(&h->handle, &obuf, method, strlen(method) + 1,
			  0, errmsg)) != GLOBUS_RLS_SUCCESS) {
    h->flags |= FH_IOERROR;
    return mkresult(rc, errmsg);
  }
  for (p = keylist; p; p = globus_list_rest(p)) {
    s = globus_list_first(p);
    if ((rc = rrpc_bufwrite(&h->handle, &obuf, s, strlen(s) + 1, 0,
			    errmsg)) != GLOBUS_RLS_SUCCESS) {
      h->flags |= FH_IOERROR;
      return mkresult(rc, errmsg);
    }
  }
  if ((rc = rrpc_bufwrite(&h->handle, &obuf, "", 1, 1,
			  errmsg)) != GLOBUS_RLS_SUCCESS) {
    h->flags |= FH_IOERROR;
    return mkresult(rc, errmsg);
  }

  if ((rlslist = rlslist_new(free_str2bulk)) == NULL)
    return mkresult(GLOBUS_RLS_NOMEMORY, NULL);
  if ((r = rrpc_str2bulk(h, &b, 0, rlslist)) == GLOBUS_SUCCESS)
    *str2bulk_list = rlslist->list;
  else
    globus_rls_client_free_list(rlslist->list);
  return r;
}

/*
 * Wildcard query that returns pairs of strings (eg lrc_get_lfn_wc,
 * lrc_get_pfn_wc, rli_get_lrc_wc).
 */
static globus_result_t
query_wc2(globus_rls_handle_t *h, char *method, char *pattern,
	  globus_rls_pattern_t type, int *offset, int reslimit,
	  globus_list_t **str2_list)

{
  globus_result_t	r;
  BUFFER		b;
  char			buf[BUFLEN*2];
  char			obuf[50];
  char			rbuf[50];
  int			otimeout;
  int			loffset;
  RLSLIST		*rlslist;

  if ((r = checkhandle(h)) != GLOBUS_SUCCESS)
    return r;

  if (!pattern || !*pattern)
    return mkresult(GLOBUS_RLS_BADARG, "pattern is NULL");

  if (type == rls_pattern_unix)
    pattern = translatepattern(pattern, buf, BUFLEN*2);

  if (!offset) {
    loffset = 0;
    offset = &loffset;
  }
  iarg(reslimit, rbuf);

  if ((rlslist = rlslist_new(free_str2)) == NULL)
    return mkresult(GLOBUS_RLS_NOMEMORY, NULL);

  if (otimeout = rrpc_get_timeout())	/* Use large timeout on wildcard*/
    rrpc_set_timeout(10 * otimeout);	/* queries			*/

  if (offset == &loffset) {
    while ((r = rrpc_call(h, &b, method, 3, pattern, iarg(*offset, obuf),
			  rbuf)) == GLOBUS_SUCCESS) {
      r = rrpc_str2(h, &b, rlslist, offset);
      if (r != GLOBUS_SUCCESS || *offset == -1)
	break;
    }
  } else
    if ((r = rrpc_call(h, &b, method, 3, pattern, iarg(*offset, obuf),
		       rbuf)) == GLOBUS_SUCCESS)
      r = rrpc_str2(h, &b, rlslist, offset);

  if (otimeout)
    rrpc_set_timeout(otimeout);

  if (r == GLOBUS_SUCCESS)
    *str2_list = rlslist->list;
  else
    globus_rls_client_free_list(rlslist->list);
  return r;
}

static globus_result_t
bulk_update(char *method, globus_rls_handle_t *h, globus_list_t *str2_list,
	    globus_list_t  **str2bulk_list)

{
  globus_result_t	r;
  BUFFER		b;
  RLSLIST		*rlslist;
  globus_list_t		*p;
  char			errmsg[MAXERRMSG];
  int			rc;
  BUFFER		obuf;
  globus_rls_string2_t	*str2;

  if ((r = checkhandle(h)) != GLOBUS_SUCCESS)
    return r;

  if (!str2_list)
    return mkresult(GLOBUS_RLS_BADARG, NULL);

  if (h->flags & FH_CLOSED)
    if ((rc = connect1(h, errmsg, MAXERRMSG)) != GLOBUS_RLS_SUCCESS)
      return mkresult(rc, errmsg);

  rrpc_initbuf(&obuf);
  if ((rc = rrpc_bufwrite(&h->handle, &obuf, method, strlen(method) + 1,
			  0, errmsg)) != GLOBUS_RLS_SUCCESS) {
    h->flags |= FH_IOERROR;
    return mkresult(rc, errmsg);
  }
  for (p = str2_list; p; p = globus_list_rest(p)) {
    str2 = (globus_rls_string2_t *) globus_list_first(p);
    if ((rc = rrpc_bufwrite(&h->handle, &obuf, str2->s1, strlen(str2->s1) + 1,
			    0, errmsg)) == GLOBUS_RLS_SUCCESS)
      rc = rrpc_bufwrite(&h->handle, &obuf, str2->s2, strlen(str2->s2) + 1,
			 0, errmsg);
    if (rc != GLOBUS_RLS_SUCCESS) {
      h->flags |= FH_IOERROR;
      return mkresult(rc, errmsg);
    }
  }
  if ((rc = rrpc_bufwrite(&h->handle, &obuf, "", 1, 1,
			  errmsg)) != GLOBUS_RLS_SUCCESS) {
    h->flags |= FH_IOERROR;
    return mkresult(rc, errmsg);
  }

  if ((rlslist = rlslist_new(free_str2bulk)) == NULL)
    return mkresult(GLOBUS_RLS_NOMEMORY, NULL);
  if ((r = rrpc_str2bulk(h, &b, 1, rlslist)) == GLOBUS_SUCCESS)
    *str2bulk_list = rlslist->list;
  else
    globus_rls_client_free_list(rlslist->list);
  return r;
}

/*
 * Send bulk attribute add/delete request.  If adding attributes then
 * the type and value are included, if removing these fields are not
 * needed.
 */
static globus_result_t
bulk_attr_update(int adding, globus_rls_handle_t *h,
		 globus_list_t *attr_obj_list, globus_list_t **str2bulk_list)

{
  globus_result_t		r;
  int				rc;
  BUFFER			obuf;
  BUFFER			b;
  globus_list_t			*p;
  char				errmsg[MAXERRMSG];
  globus_rls_attribute_object_t	*ao;
  char				buf[100];
  char				vbuf[BUFLEN];
  char				*v;
  RLSLIST			*rlslist;
  char				*method;

  if ((r = checkhandle(h)) != GLOBUS_SUCCESS)
    return r;

  if (!attr_obj_list)
    return mkresult(GLOBUS_RLS_BADARG, NULL);

  if (h->flags & FH_CLOSED)
    if ((rc = connect1(h, errmsg, MAXERRMSG)) != GLOBUS_RLS_SUCCESS)
      return mkresult(rc, errmsg);

  if (adding)
    method = "lrc_attr_add_bulk";
  else
    method = "lrc_attr_remove_bulk";

  rrpc_initbuf(&obuf);
  if ((rc = rrpc_bufwrite(&h->handle, &obuf, method, strlen(method) + 1,
			  0, errmsg)) != GLOBUS_RLS_SUCCESS) {
    h->flags |= FH_IOERROR;
    return mkresult(rc, errmsg);
  }

  if ((rlslist = rlslist_new(free_str2bulk)) == NULL)
    return mkresult(GLOBUS_RLS_NOMEMORY, NULL);

  for (p = attr_obj_list; p; p = globus_list_rest(p)) {
    ao = (globus_rls_attribute_object_t *) globus_list_first(p);
    if (adding)
      if ((v = globus_rls_client_attr2s(&ao->attr, vbuf, BUFLEN)) == NULL) {
	addstr2bulkrc(GLOBUS_RLS_NOMEMORY, ao->key, ao->attr.name, rlslist);
	continue;
      }
    rc = rrpc_bufwrite(&h->handle,&obuf,ao->key,strlen(ao->key)+1,0,errmsg);
    if (rc == GLOBUS_RLS_SUCCESS) {
      iarg(ao->attr.objtype, buf),
      rc = rrpc_bufwrite(&h->handle, &obuf, buf, strlen(buf) + 1, 0, errmsg);
    }
    if (adding && rc == GLOBUS_RLS_SUCCESS) {
      iarg(ao->attr.type, buf),
      rc = rrpc_bufwrite(&h->handle, &obuf, buf, strlen(buf) + 1, 0, errmsg);
    }
    if (rc == GLOBUS_RLS_SUCCESS)
      rc = rrpc_bufwrite(&h->handle, &obuf, ao->attr.name,
			 strlen(ao->attr.name) + 1, 0, errmsg);
    if (adding && rc == GLOBUS_RLS_SUCCESS)
      rc = rrpc_bufwrite(&h->handle, &obuf, v, strlen(v) + 1, 0, errmsg);

    if (rc != GLOBUS_RLS_SUCCESS) {
      h->flags |= FH_IOERROR;
      globus_rls_client_free_list(rlslist->list);
      return mkresult(rc, errmsg);
    }
  }
  if ((rc = rrpc_bufwrite(&h->handle, &obuf, "", 1, 1,
			  errmsg)) != GLOBUS_RLS_SUCCESS) {
    h->flags |= FH_IOERROR;
    globus_rls_client_free_list(rlslist->list);
    return mkresult(rc, errmsg);
  }

  if ((r = rrpc_str2bulk(h, &b, 1, rlslist)) == GLOBUS_SUCCESS)
    *str2bulk_list = rlslist->list;
  else
    globus_rls_client_free_list(rlslist->list);
  return r;
}

/*
 * Read list of pairs of strings returned by RPC call and return in
 * globus_list_t, where each list datum is a globus_rls_string2_t.
 */
static globus_result_t
rrpc_str2(globus_rls_handle_t *h, BUFFER *bp, RLSLIST *rlslist, int *offset)


{
  char			buf1[BUFLEN];
  char			buf2[BUFLEN];
  globus_result_t	r;
  int			nomem;
  globus_rls_string2_t	*str2;

  nomem = 0;
  while (1) {
    if ((r = rrpc_getstr(h, bp, buf1, BUFLEN)) != GLOBUS_SUCCESS)
      return r;
    if (!*buf1) {
      (*offset) = -1;
      break;
    }
    if (buf1[0] == 1 && buf1[1] == 0)
      break;
    if ((r = rrpc_getstr(h, bp, buf2, BUFLEN)) != GLOBUS_SUCCESS)
      return r;
    if (nomem)
      continue;
    if (!(str2 = globus_libc_calloc(1, sizeof(*str2)))) {
      nomem++;
      continue;
    }
    if (!(str2->s1 = globus_libc_strdup(buf1)) ||
	!(str2->s2 = globus_libc_strdup(buf2))) {
      free_str2(str2);
      nomem++;
      continue;
    }
    if (rlslist_append(rlslist, str2) != GLOBUS_RLS_SUCCESS) {
      free_str2(str2);
      nomem++;
      continue;
    }
    (*offset)++;
  }

  if (nomem)
    return mkresult(GLOBUS_RLS_NOMEMORY, NULL);
  return GLOBUS_SUCCESS;
}

/*
 * Read list of pairs of strings returned by RPC call and return in
 * globus_list_t, where each list datum is a globus_rls_string2_t.
 */
static globus_result_t
rrpc_str2bulk(globus_rls_handle_t *h, BUFFER *bp, int getboth, RLSLIST *rlslist)

{
  char				rcbuf[BUFLEN];
  char				buf1[BUFLEN];
  char				buf2[BUFLEN];
  globus_result_t		r;
  int				nomem;
  globus_rls_string2_bulk_t	*str2bulk;
  int				rc;
  char				errmsg[MAXERRMSG];

  if ((rc = rrpc_getresult(h, bp, errmsg)) != GLOBUS_SUCCESS)
    return mkresult(rc, errmsg);

  nomem = 0;
  while (1) {
    if ((r = rrpc_getstr(h, bp, rcbuf, BUFLEN)) != GLOBUS_SUCCESS)
      return r;
    if (!*rcbuf)
      break;
    rc = atoi(rcbuf);
    if ((r = rrpc_getstr(h, bp, buf1, BUFLEN)) != GLOBUS_SUCCESS)
      return r;
    *buf2 = '\0';
    if (getboth || rc == GLOBUS_RLS_SUCCESS)
      if ((r = rrpc_getstr(h, bp, buf2, BUFLEN)) != GLOBUS_SUCCESS)
        return r;
    if (nomem)
      continue;
    if (!addstr2bulkrc(rc, buf1, buf2, rlslist)) {
      nomem++;
      continue;
    }
  }

  if (nomem)
    return mkresult(GLOBUS_RLS_NOMEMORY, NULL);
  return GLOBUS_SUCCESS;
}

/*
 * Read list of lfns and attribute values, store in globus_list_t,
 * where each list datum is a globus_rls_attribute_object_t.
 */
static globus_result_t
rrpc_attr_obj(globus_rls_handle_t *h, BUFFER *bp, RLSLIST *rlslist,
	      int *offset, char *name, globus_rls_obj_type_t objtype)

{
  char				key[BUFLEN];
  char				tbuf[BUFLEN];
  char				sval[BUFLEN];
  char				rcbuf[100];
  globus_result_t		r;
  globus_result_t		saver;
  int				nomem;
  globus_rls_attribute_object_t	*attr_obj;

  saver = GLOBUS_SUCCESS;
  nomem = 0;
  while (1) {
    if ((r = rrpc_getstr(h, bp, key, BUFLEN)) != GLOBUS_SUCCESS)
      return r;
    if (!*key) {
      (*offset) = -1;
      break;
    }
    if (key[0] == 1 && key[1] == 0)
      break;
    if ((r = rrpc_getstr(h, bp, tbuf, BUFLEN)) != GLOBUS_SUCCESS)
      return r;
    if ((r = rrpc_getstr(h, bp, sval, BUFLEN)) != GLOBUS_SUCCESS)
      return r;
    if (nomem)
      continue;

    if (!(attr_obj = globus_libc_calloc(1, sizeof(*attr_obj)))) {
      nomem++;
      continue;
    }
    if ((attr_obj->attr.name = globus_libc_strdup(name)) == NULL) {
      free_attr_obj(attr_obj);
      nomem++;
      continue;
    }
    if ((r = globus_rls_client_s2attr(atoi(tbuf), sval,
				      &attr_obj->attr)) != GLOBUS_SUCCESS) {
      saver = r;
      free_attr_obj(attr_obj);
      continue;
    }
    attr_obj->attr.objtype = objtype;
    if ((attr_obj->key = globus_libc_strdup(key)) == NULL) {
      free_attr_obj(attr_obj);
      nomem++;
      continue;
    }
    if (rlslist_append(rlslist, attr_obj) != GLOBUS_RLS_SUCCESS) {
      free_attr_obj(attr_obj);
      nomem++;
      continue;
    }
    (*offset)++;
  }

  if (nomem)
    return mkresult(GLOBUS_RLS_NOMEMORY, NULL);
  return saver;
}

/*
 * Bulk read list of lfns and attribute values, store in globus_list_t,
 * where each list datum is a globus_rls_attribute_object_t.
 */
static globus_result_t
rrpc_attr_obj_bulk(globus_rls_handle_t *h, BUFFER *bp, RLSLIST *rlslist,
		   globus_rls_obj_type_t objtype)

{
  char				rcbuf[100];
  char				key[BUFLEN];
  char				name[BUFLEN];
  char				tbuf[BUFLEN];
  char				sval[BUFLEN];
  globus_result_t		r;
  globus_result_t		saver;
  int				nomem;
  globus_rls_attribute_object_t	*attr_obj;
  int				rc;

  saver = GLOBUS_SUCCESS;
  nomem = 0;
  while (1) {
    if ((r = rrpc_getstr(h, bp, rcbuf, 100)) != GLOBUS_SUCCESS)
      return r;
    if (!*rcbuf)
      break;
    rc = atoi(rcbuf);
    if ((r = rrpc_getstr(h, bp, key, BUFLEN)) != GLOBUS_SUCCESS)
      return r;
    if ((r = rrpc_getstr(h, bp, name, BUFLEN)) != GLOBUS_SUCCESS)
      return r;
    if (rc == GLOBUS_RLS_SUCCESS) {
      if ((r = rrpc_getstr(h, bp, tbuf, BUFLEN)) != GLOBUS_SUCCESS)
	return r;
      if ((r = rrpc_getstr(h, bp, sval, BUFLEN)) != GLOBUS_SUCCESS)
	return r;
    }
    if (nomem)
      continue;

    if (!(attr_obj = globus_libc_calloc(1, sizeof(*attr_obj)))) {
      nomem++;
      continue;
    }
    attr_obj->rc = rc;
    if ((attr_obj->key = globus_libc_strdup(key)) == NULL) {
      free_attr_obj(attr_obj);
      nomem++;
      continue;
    }
    if ((attr_obj->attr.name = globus_libc_strdup(name)) == NULL) {
      free_attr_obj(attr_obj);
      nomem++;
      continue;
    }
    attr_obj->attr.objtype = objtype;
    if (rc == GLOBUS_RLS_SUCCESS) {
      if ((r = globus_rls_client_s2attr(atoi(tbuf), sval,
					&attr_obj->attr)) != GLOBUS_SUCCESS) {
	saver = r;
	free_attr_obj(attr_obj);
	continue;
      }
    }
    if (rlslist_append(rlslist, attr_obj) != GLOBUS_RLS_SUCCESS) {
      free_attr_obj(attr_obj);
      nomem++;
      continue;
    }
  }

  if (nomem)
    return mkresult(GLOBUS_RLS_NOMEMORY, NULL);
  return saver;
}

/*
 * Return null terminated string from h/bp.
 */
static globus_result_t
rrpc_getstr(globus_rls_handle_t *h, BUFFER *bp, char *rbuf, int rbuflen)

{
  int	i;
  int	c;
  int	rc;
  char	errmsg[MAXERRMSG];

  i = 0;
  while (i < rbuflen && (c = NEXTC(&h->handle, bp, &rc, errmsg)) != -1) {
    rbuf[i++] = c;
    if (!c)
      return GLOBUS_SUCCESS;
  }
  if (c == -1)
    return mkresult(rc, errmsg);
  return mkresult(GLOBUS_RLS_OVERFLOW, NULL);
}

/*
 * Send RPC request.  Args are all strings, na should be number of args.
 *   NULL args are translated to empty strings.  Result is read into
 *   specified BUFFER (if bp is NULL no result is expected).
 */
static globus_result_t
rrpc_call(globus_rls_handle_t *h, BUFFER *bp, char *method, int na, ...)

{
  struct iovec		iov[10];
  globus_size_t		nb;
  int			n;
  va_list		ap;
  int			retried = 0;
  int			rc;
  char			errmsg[MAXERRMSG];

  if (h->flags & FH_CLOSED) {
    if ((rc = connect1(h, errmsg, MAXERRMSG)) != GLOBUS_RLS_SUCCESS)
      return mkresult(rc, errmsg);
    retried++;		/* No point in retry if just opened connection	*/
  }

  n = 0;
  iov[n].iov_base = method;
  iov[n++].iov_len = strlen(method) + 1;
  va_start(ap, na);
  while (n < na + 1) {
    if ((iov[n].iov_base = va_arg(ap, char *)) == NULL)
      iov[n].iov_base = "";
    iov[n].iov_len = strlen(iov[n].iov_base) + 1;
    n++;
  }
  va_end(ap);

 retry:
  if ((rc = rrpc_writev(&h->handle,iov,n,&nb,errmsg)) != GLOBUS_RLS_SUCCESS) {
    h->flags |= FH_IOERROR;
    return mkresult(rc, errmsg);
  }

  if (bp) {
    rc = rrpc_getresult(h, bp, errmsg);
    if (!retried && (rc == GLOBUS_RLS_GLOBUSERR || rc == GLOBUS_RLS_TIMEOUT ||
		     rc == GLOBUS_RLS_DBERROR)) {
      retried++;
      if (rc == GLOBUS_RLS_DBERROR)
	goto retry;
      globus_io_close(&h->handle);
      if (connect1(h, errmsg, MAXERRMSG) == GLOBUS_RLS_SUCCESS)
	goto retry;
    }
    return mkresult(rc, errmsg);
  }

  return GLOBUS_SUCCESS;
}

static int
addstr2bulkrc(int rc, char *s1, char *s2, RLSLIST *rlslist)

{
  globus_rls_string2_bulk_t	*str2bulk;

  if ((str2bulk = globus_libc_calloc(1, sizeof(*str2bulk))) == NULL)
    return 0;
  if (!(str2bulk->str2.s1 = globus_libc_strdup(s1))) {
    free_str2bulk(str2bulk);
    return 0;
  }
  if (*s2) {
    if (!(str2bulk->str2.s2 = globus_libc_strdup(s2))) {
      free_str2bulk(str2bulk);
      return 0;
    }
  }
  str2bulk->rc = rc;
  if (rlslist_append(rlslist, str2bulk) != GLOBUS_RLS_SUCCESS) {
    free_str2bulk(str2bulk);
    return 0;
  }
  return 1;
}

/*
 * checkhandle - Verifies handle is open, initializes error buffer if and
 *   clearerr is set.
 */
static globus_result_t
checkhandle(globus_rls_handle_t *h)

{
  globus_list_t		*node;
  globus_result_t	r;

  globus_mutex_lock(&active_list_mutex);
  if (!(node = globus_list_search(active_list, (void *) h)))
    r = mkresult(GLOBUS_RLS_INVHANDLE, NULL);
  else
    r = GLOBUS_SUCCESS;
  globus_mutex_unlock(&active_list_mutex);
  return r;
}

static int
connect1(globus_rls_handle_t *h, char *errmsg, int errlen)

{
  globus_result_t			r;
  globus_io_attr_t			attr;
  int					haveattr = 0;
  globus_io_secure_authorization_data_t	authdata;
  int					haveauthdata = 0;
  int					rc = GLOBUS_RLS_SUCCESS;
  BUFFER				b;

  if ((r = globus_io_tcpattr_init(&attr)) != GLOBUS_SUCCESS)
    goto error;
  haveattr++;

  if (strcasecmp(h->url.scheme, GLOBUS_RLS_URL_SCHEME_NOAUTH) != 0) {
    if ((r = globus_io_secure_authorization_data_initialize(&authdata)) !=
		  GLOBUS_SUCCESS)
      goto error;
    haveauthdata++;
#ifdef COMMENT
    if ((r = globus_io_secure_authorization_data_set_callback(&authdata,
		cauthcb, h)) != GLOBUS_SUCCESS)
      goto error;
#endif
    if ((r = globus_io_attr_set_secure_authentication_mode(
		  &attr, GLOBUS_IO_SECURE_AUTHENTICATION_MODE_GSSAPI,
		  GSS_C_NO_CREDENTIAL)) != GLOBUS_SUCCESS)
      goto error;
    if ((r = globus_io_attr_set_secure_authorization_mode(
		  &attr, GLOBUS_IO_SECURE_AUTHORIZATION_MODE_HOST,
		  &authdata)) != GLOBUS_SUCCESS)
      goto error;
  }

  if ((rc = rrpc_connect(h->url.host, h->url.port, &attr,
			 &h->handle, errmsg, errlen)) != GLOBUS_RLS_SUCCESS)
    goto error;

  if ((rc = rrpc_getresult(h, &b, errmsg)) != GLOBUS_SUCCESS) {
    globus_io_close(&h->handle);
    goto error;
  }

  globus_io_tcpattr_destroy(&attr);
  if (haveauthdata)
    globus_io_secure_authorization_data_destroy(&authdata);
  h->flags = 0;
  return GLOBUS_RLS_SUCCESS;

 error:
#ifdef COMMENT
  globus_libc_close(h->handle.fd);	/* Workaround bug in globus-io 2.3*/
#endif
  h->flags |= FH_CLOSED;
  if (haveauthdata)
    globus_io_secure_authorization_data_destroy(&authdata);
  if (haveattr)
    globus_io_tcpattr_destroy(&attr);
  if (rc != GLOBUS_RLS_SUCCESS)
    return rc;
  return rrpc_globuserr(errmsg, errlen, r);
}

static void
free_str2(void *v)

{
  globus_rls_string2_t	*str2 = (globus_rls_string2_t *) v;

  if (str2->s1)
    globus_libc_free(str2->s1);
  if (str2->s2)
    globus_libc_free(str2->s2);
  globus_libc_free(str2);
}

static void
free_str2bulk(void *v)

{
  globus_rls_string2_bulk_t	*str2bulk = (globus_rls_string2_bulk_t *) v;

  if (str2bulk->str2.s1)
    globus_libc_free(str2bulk->str2.s1);
  if (str2bulk->str2.s2)
    globus_libc_free(str2bulk->str2.s2);
  globus_libc_free(str2bulk);
}

static void
free_attr(void *v)

{
  globus_rls_attribute_t	*attr = (globus_rls_attribute_t *) v;

  if (attr->name)
    globus_libc_free(attr->name);
  if (attr->type == globus_rls_attr_type_str && attr->val.s)
    globus_libc_free(attr->val.s);
  globus_libc_free(attr);
}

static void
free_attr_obj(void *v)

{
  globus_rls_attribute_object_t	*attr_obj = (globus_rls_attribute_object_t *)v;

  if (attr_obj->attr.name)
    globus_libc_free(attr_obj->attr.name);
  if (attr_obj->attr.type == globus_rls_attr_type_str && attr_obj->attr.val.s)
    globus_libc_free(attr_obj->attr.val.s);
  if (attr_obj->key)
    globus_libc_free(attr_obj->key);
  globus_libc_free(attr_obj);
}

static int
rlslist_append(RLSLIST *rlslist, void *datum)

{
  globus_list_t	*l;

  l = NULL;
  if (globus_list_insert(&l, datum) != GLOBUS_SUCCESS)
    return GLOBUS_RLS_NOMEMORY;
  if (!rlslist->list)
    rlslist->list = l;
  else
    rlslist->last->next = l;
  rlslist->last = l;
  return GLOBUS_RLS_SUCCESS;
}

static RLSLIST *
rlslist_new(void (*destructor)())

{
  RLSLIST	*rlslist;

  globus_mutex_lock(&rlslist_mutex);
  if (rlslist_freelist) {
    rlslist = rlslist_freelist;
    rlslist_freelist = rlslist_freelist->nxt;
  } else if ((rlslist = globus_libc_malloc(sizeof(RLSLIST))) == NULL) {
    globus_mutex_unlock(&rlslist_mutex);
    return NULL;
  }
  rlslist->list = NULL;
  rlslist->destructor = destructor;
  rlslist->nxt = rlslisttab;
  rlslisttab = rlslist;
  globus_mutex_unlock(&rlslist_mutex);
  return rlslist;
}

/*
 * Translate Unix file globbing chars (*, ?) to their SQL equivalents (%, _).
 * Escapes SQL pattern matching chars.
 */
static char *
translatepattern(char *unixpat, char *sqlpat, int len)

{
  int	i;
  int	j;
  int	escaped = 0;

  i = 0;
  j = 0;
  while (unixpat[i] && i < len - 1) {
    if (unixpat[i] == '\\') {
      sqlpat[j++] = '\\';
      i++;
      escaped++;
      continue;
    }
    if (escaped)
      sqlpat[j++] = unixpat[i++];
    else if (unixpat[i] == '*') {
      sqlpat[j++] = '%';
      i++;
    } else if (unixpat[i] == '?') {
      sqlpat[j++] = '_';
      i++;
    } else {
      if (unixpat[i] == '%' || unixpat[i] == '_')
	sqlpat[j++] = '\\';
      sqlpat[j++] = unixpat[i++];
    }
    escaped = 0;
  }
  sqlpat[j] = '\0';
  return sqlpat;
}

int
globus_list_len(globus_list_t *list)

{
  int	len = 0;

  while (list) {
    list = globus_list_rest(list);
    len++;
  }
  return len;
}

/*
 * Linux doesn't have cftime() apparently, so write our own.
 */
int
mycftime(char *s, int len, char *fmt, time_t clock)

{
  struct tm	*t;
  int		size;

  globus_libc_lock();
  t = localtime(&clock);
  size = strftime(s, len, fmt, t);
  globus_libc_unlock();
  return size;
}

static char *
iarg(int i, char *buf)

{
  sprintf(buf, "%d", i);
  return buf;
}

static globus_result_t
mkresult(int rc, char *specific)

{
  globus_object_t	*o;
  char			errmsg[MAXERRMSG];

  if (rc == GLOBUS_RLS_SUCCESS)
    return GLOBUS_SUCCESS;
  o = globus_error_construct_error(GLOBUS_RLS_CLIENT_MODULE, NULL, rc,
#ifndef GLOBUS_ERROR_CONSTRUCT_ERROR_FIVE_ARGS
	NULL, NULL, 0,
#endif
        "%s", globus_rls_errmsg(rc, specific, errmsg, MAXERRMSG));
  return globus_error_put(o);
}

#ifdef COMMENT
/*** TODO: return error message to caller */
static globus_bool_t
cauthcb(void *a, globus_io_handle_t *handlep, globus_result_t r,
        char *identity, gss_ctx_id_t ch)

{
  globus_rls_handle_t	*h = (globus_rls_handle_t *) a;
  char			*cp;
  char			errmsg[MAXERRMSG];

  if (r != GLOBUS_SUCCESS) {
    rrpc_globuserr(errmsg, MAXERRMSG, r);
    return GLOBUS_FALSE;
  }
  if (!(cp = strstr(identity, "/CN="))) {
    snprintf(errmsg, MAXERRMSG, "Badly formatted identity (no CN=): %s",
	     identity);
    return GLOBUS_FALSE;
  }
  cp += 4;
  if (strncmp(cp, "rls/", 4) == 0)
    cp += 4;
  else if (strncmp(cp, "host/", 5) == 0)
    cp += 5;
  else {
    snprintf(errmsg, MAXERRMSG,
	     "Badly formatted identity (not 'rls/'): %s", identity);
    return GLOBUS_FALSE;
  }
  if (strcasecmp(cp, h->url.host) != 0) {
    snprintf(errmsg, MAXERRMSG,
	"Unexpected host %s in identity, expected %s", identity, h->url.host);
    return GLOBUS_FALSE;
  }
  return GLOBUS_TRUE;
}
#endif
