/*
 * Copyright 1999-2006 University of Chicago
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/**************************************************************************

 Transform        Perf
 Module:          transform_perf.c

 Author:          Noam Freedman

 The purpose of this module is to allow for performance information
 gathering.  The initial version will calculate packet loss, latency,
 jitter, and bandwidth.

**************************************************************************/

/* Modifications needed:

   - Need to give the option to the user to make jitter/latency correct by
     forcing a barrier on a send_rsr and then including the timestamp of
     when it was done in the next message instead of the current timestamp.

*/

/*#include <math.h>*/
#include "internal.h"
/*#include "nexus_dc.h"*/
#include <sys/time.h>
#include <nexus.h>

#define TRANSFORM_BLOCKSIZE	8
#define LARGEST_POSSIBLE_SIZEOF_ULONG 8

#define COUNT(array)  ( sizeof(array) / sizeof((array)[0]) )

static double epow[] = {
    1.01574770858668,   /* e^1/64 */
    1.0317434074991,    /* e^1/32 */
    1.06449445891786,   /* e^1/16 */
    1.13314845306683,   /* e^1/8 */
    1.28402541668774,   /* e^1/4 */
    1.64872127070013,   /* e^1/2 */
    2.71828182845905,   /* e^1 */
    7.38905609893065,   /* e^2 */
    54.5981500331442,   /* e^4 */
};

#define ESCALE  (64)
#define EMAX  (1 << COUNT(epow))
#define EEMAX   2980.95798704172827474359       /* e^8 */


/********************************/
/* Externally Visible Functions */
/********************************/

void *_nx_transform_perf_info(void);

/******************************/
/* Transform Table Functions */
/*****************************/

static int transform_perf_transform_id();
static void transform_perf_init(
			    nexus_bool_t   *modifies_data,
			    unsigned long  *header_size,
			    unsigned long  *trailer_size);
static void transform_perf_init_endpoint_state(nexus_transformattr_t *attr, 
					    nexus_transformstate_t **ep_state);
static void transform_perf_destroy_endpoint_state(
					    nexus_transformstate_t *ep_state);
static void transform_perf_init_startpoint_state(
					    nexus_transformstate_t *ep_state,
					    nexus_transformstate_t **sp_state,
					    nexus_bool_t  *copy_sp_locally,
					    nexus_bool_t  *destroy_sp_locally);
static void transform_perf_destroy_startpoint_state(
					    nexus_transformstate_t *sp_state);
static void transform_perf_copy_startpoint_state(
					nexus_transformstate_t *sp_state,
					nexus_transformstate_t **sp_state_copy);
static int transform_perf_sizeof_startpoint_state(nexus_transformstate_t *s);
static void transform_perf_put_startpoint_state(nexus_byte_t **buffer, 
						nexus_transformstate_t *state);
static void transform_perf_get_startpoint_state(nexus_byte_t **buffer, 
						int format, 
						nexus_transformstate_t **state);
static int transform_perf_transform(nexus_transformstate_t *startpoint_state,
                                        nexus_byte_t *storage_start,
                                        unsigned long storage_size,
                                        nexus_byte_t *data_start,
                                        unsigned long *data_size,
                                        nexus_bool_t must_alloc_new_buffer,
                                        nexus_byte_t *transform_info_start,
                                        nexus_byte_t **out_storage_start,
                                        unsigned long *out_storage_size,
                                        nexus_byte_t **out_data_start,
                                        unsigned long *out_data_size);

static int transform_perf_untransform(nexus_transformstate_t *endpoint_state,
					 nexus_byte_t *data_start,
					 unsigned long *data_size,
					 nexus_byte_t *transform_info_start,
					 int format,
					 nexus_byte_t **destination_start,
					 unsigned long *destination_size);

static void transform_perf_transformstate_get_info(
                                                nexus_transformstate_t *state,
                                                void *info,
                                                int flag);

int transform_perf_get_stats(nexus_endpoint_t *endpoint, int stat_type);

double fake_sqrt( double v );
double fake_fabs( double v );
double fake_exp(double v);
 
/*******************/
/* Data Structures */
/*******************/

struct perf_stats_t
{
    double decay;
    double s0;
    double s1;
    double s2;
};

typedef struct _sequence_node_t
{
    struct _sequence_node_t *next;
    
    /* number of the message last received */           
    unsigned long sequence;          
    
    struct timeval last_time;           /* Time last message was received (receiver_timestamp) */
    struct timeval last_send_time;           /* Time last message was sent (sender_timestamp) */

    unsigned long msgs_lost;

    double bandwidth;  		  /* Bandwidth to the app weighted and decayed based on message
                                             reception times */
    double bandwidth_total;        /* Ditto for bandwidth to nexus */
    double inter_msg_delay;        /* Ditto for inter-msg-delay */

    /* N.B. THE FOLLOWING IS ONLY ACCURATE IF THE TIMES ON BOTH MACHINES ARE THE SAME.
            ITS JITTER/STDDEV WILL BE CORRECT IN ANY CASE. */

    double transit_time;           /* Ditto for transit time */

    double bandwidth_stddev;
    double bandwidth_total_stddev;
    double inter_msg_delay_stddev;
    double transit_time_stddev;

    double inter_msg_delay_jitter; 
    double transit_time_jitter;

    double bandwidth_last;         /* Bandwidth of only the last packet
                                             (an "instantaneous" bandwidth) */
    double bandwidth_total_last;
    double inter_msg_delay_last;   /* Ditto for inter_msg_delay */
    double transit_time_last;      /* Ditto for transit_time */
    unsigned long size_nexus_last;
    unsigned long size_app_last;

    /* Internal information needed to generate the above statistics */
    struct perf_stats_t bw_stats;
    struct perf_stats_t bwt_stats;
    struct perf_stats_t imd_stats;
    struct perf_stats_t tt_stats;
    struct perf_stats_t ttj_stats;
    struct perf_stats_t imdj_stats;

} sequence_node_t;

typedef struct _endpoint_transform_state_t
{
    /* singly linked list of sequence nodes, one for each startpoint bound to me  */
    sequence_node_t *sequence_hdr;

} endpoint_transform_state_t; 

typedef struct _startpoint_transform_state_t
{
    /* address of my sequence node on endpoint side */
    nexus_byte_t ep_sequence_liba[LARGEST_POSSIBLE_SIZEOF_ULONG];

    /* number of the message last sent */           
    unsigned long sequence;          
} startpoint_transform_state_t;

/**************************/
/* Local Global Variables */
/**************************/

static nexus_transform_funcs_t transform_perf_funcs =
{
    transform_perf_transform_id,
    transform_perf_init,
    NULL, /* transform_perf_shutdown */
    /*NULL,*/ /* transform_perf_abort */
    NULL, /* transform_perf_transformattr_init, */
    NULL, /* transform_perf_transformattr_destroy, */
    NULL, /* transform_perf_transformattr_get_info */
    transform_perf_init_endpoint_state,
    transform_perf_destroy_endpoint_state,
    NULL, /* transform_perf_update_endpoint_state */
    transform_perf_init_startpoint_state,
    NULL, /* transform_perf_copy_startpoint_state, */
	  /* do not provide copy_sp routine when copy_locally = NEXUS_FALSE */
    transform_perf_destroy_startpoint_state,
    transform_perf_sizeof_startpoint_state,
    transform_perf_put_startpoint_state,
    transform_perf_get_startpoint_state,
    transform_perf_transform,
    transform_perf_untransform,
    transform_perf_transformstate_get_info
};

/********************************/
/* Externally Visible Functions */
/********************************/

/*
 * _nx_transform_perf_info()
 *
 * Return the function table for this module.
 */
void *_nx_transform_perf_info(void)
{
/*    nexus_printf("In _nx_transform_perf_info()\n");*/
    return ((void *) (&transform_perf_funcs));
}

/*******************/
/* Local Functions */
/*******************/

static void compute_perf_stats (struct perf_stats_t *st,
				double value,
				double weight,
				double *mean,
				double *stddev)
{
	/* Reduce the weight of everything in the past, and add
	   our new value with full weight. */

	double value_w,svar;

	/* e^( -weight/window ) */
	double decay_w =  fake_exp(st->decay * weight );  

	double old_s0 = st->s0;

	if (weight >  0)
	{
		value_w = value * weight;
		st->s0 = (st->s0 * decay_w) + weight;
		st->s1 = (st->s1 * decay_w) + value_w;
		st->s2 = (st->s2 * decay_w) + value_w * value;
	}

	if (mean != NULL)
	{
		if( st->s0 > 0 && old_s0 != 0.0)
		{
			*mean = st->s1 / st->s0;
		}
	}

	svar = (st->s2 - (st->s1*st->s1) / st->s0);

	if( stddev != NULL )
	{
		if( st->s0 <= weight)
		{
			*stddev = 0.0;
		}
		else
		{
			*stddev = fake_sqrt(fake_fabs(svar / (st->s0 - weight)));
		}
	}
}

/*
 * transform_perf_transform_id()
 *
 * Return the #define of this transform module 
 */
static int transform_perf_transform_id()
{
    return(NEXUS_TRANSFORM_PERF);
} /* transform_perf_transform_id() */

/*
 * transform_perf_init()
 *
 *  Written if the transform module needs to perform initialization code or needs
 *  anything from the command line. 
 * 
 *  modifies_data - does the transform alter user data?
 *  header_size - the size of the transform_info (key, encryption init vector, etc.)
 *  trailer_size - a *SUGGESTION* of how much padding would be useful at the end 
 *                 of the buffer to avoid allocating a new buffer, NOT GUARANTEED
 */
static void transform_perf_init(
                                nexus_bool_t   *modifies_data,
                                unsigned long  *header_size,   
                                unsigned long  *trailer_size)
{
/*    nexus_printf("In transform_perf_init()\n");*/
    NexusAssert2((modifies_data), 
	("transform_perf_init(): rcvd NULL modifies_data\n"));
    NexusAssert2((header_size), 
	("transform_perf_init(): rcvd NULL header_size\n"));
    NexusAssert2((trailer_size), 
	("transform_perf_init(): rcvd NULL trailer_size\n"));

    NexusAssert2((sizeof(unsigned long) <= LARGEST_POSSIBLE_SIZEOF_ULONG), 
("transform_perf_init(): detected sizeof(unsigned long) > LARGEST_POSSIBLE_SIZEOF_ULONG\n"));

    /* adds 1 to every byte of msg data */
    *modifies_data = NEXUS_TRUE;

    /* ep_seq_liba and sequence and timestamp */
    *header_size = 4 * LARGEST_POSSIBLE_SIZEOF_ULONG;

    /* would like to have enough extra bytes to pad to TRANSFORM_BLOCKSIZE-byte 
    boundary without allocating a new buffer */
     *trailer_size = TRANSFORM_BLOCKSIZE - 1;

} /* transform_perf_init() */

/*
 * void (*shutdown)(void);
 *
 * Written if transform module needs to do anything during normal termination,
 * e.g., notify any servers that it might have registered with.  In most 
 * modules this function will not be written.
 */

/*
 * void (*abort)(void);
 *
 * Just like shutdown() above except this gets called on abend.
 */

/*
 * transform_perf_init_endpoint_state()
 *
 * This gets called when an endpoint is being initialized.  It's single arg
 * is the transform data from the endpointattr this endpoint is being 
 * initialized with.  
 * This function must nexus_malloc enough space in a single block to
 * hold ALL the transform information for this endpoint.  It then copies
 * the attr stuff from endpointattr, if appropriate, and fills the other
 * fields.
 */
static void transform_perf_init_endpoint_state(nexus_transformattr_t *attr,
					      nexus_transformstate_t **ep_state)
{
/*nexus_printf("transform_perf_init_endpoint_state\n");   */

    /* NOTE ignoring attr for EXAMPLE */
    NexusAssert2((ep_state), 
	("transform_perf_init_endpoint_state(): rcvd NULL ep_state\n"));

    *ep_state = (endpoint_transform_state_t *) 
		    nexus_malloc(sizeof(endpoint_transform_state_t));

    ((endpoint_transform_state_t *) (*ep_state))->sequence_hdr = 
							(sequence_node_t *) NULL;
} /* transform_perf_init_endpoint_state() */

/*
 * transform_perf_destroy_endpoint_state()
 *
 * Called when destroying an endpoint.  ep_state is a pointer to the block that
 * was nexus_malloc'd in init_endpoint_state.  This function needs to 
 * nexus_free any memory in state (if complex structure with nexus_malloc'd 
 * fields) and then nexus_free(state).
 */
static void transform_perf_destroy_endpoint_state(
					    nexus_transformstate_t *ep_state)
{
    sequence_node_t *seq_node;

/*    nexus_printf("transform_perf_destroy_endpoint_state\n");*/

    NexusAssert2((ep_state), 
	("transform_perf_destroy_endpoint_state(): rcvd NULL ep_state\n"));

    while (((endpoint_transform_state_t *) (ep_state))->sequence_hdr)
    {
		seq_node = ((endpoint_transform_state_t *) (ep_state))->sequence_hdr;
		((endpoint_transform_state_t *) (ep_state))->sequence_hdr = seq_node->next;
		nexus_free(seq_node);
    } /* endwhile */

    nexus_free(ep_state);

} /* transform_perf_destroy_endpoint_state() */

/*
 * transform_perf_init_startpoint_state()
 *
 * This gets called when an startpoint is bound to an endpoint.  The single
 * arg is the transform state info from the endpoint.  This function needs
 * to nexus_malloc enough memory for a startpoint to perform transformation
 * and copy info from state as necessary.
 */
static void transform_perf_init_startpoint_state(
					    nexus_transformstate_t *ep_state,
					    nexus_transformstate_t **sp_state,
					    nexus_bool_t  *copy_sp_locally,
					    nexus_bool_t  *destroy_sp_locally)
{
    sequence_node_t *seq_node;
    unsigned long temp_ulong;

/*    nexus_printf("transform_perf_init_startpoint_state\n");*/

    NexusAssert2((ep_state), 
	("transform_perf_init_startpoint_state(): rcvd NULL ep_state\n"));
    NexusAssert2((sp_state), 
	("transform_perf_init_startpoint_state(): rcvd NULL sp_state\n"));
    NexusAssert2((copy_sp_locally), 
	("transform_perf_init_startpoint_state(): rcvd NULL copy_sp_locally\n"));
    NexusAssert2((destroy_sp_locally), 
    ("transform_perf_init_startpoint_state(): rcvd NULL destroy_sp_locally\n"));

    /* instantiate endpoint's sequence node for this startpoint state */
    seq_node = (sequence_node_t *) nexus_malloc(sizeof(sequence_node_t));
    temp_ulong = (unsigned long) seq_node;

    /* instantiate startpoint state */
    *sp_state = (startpoint_transform_state_t *) 
		nexus_malloc(sizeof(startpoint_transform_state_t));

    /* bind startpoint to endpoint */
    memcpy((void *)((startpoint_transform_state_t *) (*sp_state))->ep_sequence_liba,
	    (void *) &temp_ulong,
	    sizeof(unsigned long));

    /* initialize endpoint's sequence node and this startpoint state */
    seq_node->sequence = 0;
    seq_node->msgs_lost = 0;
    seq_node->last_time.tv_sec = 0;
    seq_node->last_time.tv_usec = 0;

    seq_node->last_send_time.tv_sec = 0;
    seq_node->last_send_time.tv_usec = 0;

    seq_node->bandwidth = 0;
    seq_node->bandwidth_total = 0;
    seq_node->inter_msg_delay = 0;
    seq_node->transit_time = 0.0;

    seq_node->inter_msg_delay_jitter = 0.0;
    seq_node->transit_time_jitter = 0.0;

    seq_node->bandwidth_stddev = 0.0;
    seq_node->bandwidth_total_stddev = 0.0;
    seq_node->inter_msg_delay_stddev = 0.0;
    seq_node->transit_time_stddev = 0.0;

    seq_node->bandwidth_last = 0.0;
    seq_node->bandwidth_total_last = 0.0;
    seq_node->inter_msg_delay_last = 0.0;
    seq_node->transit_time_last = 0.0;

    seq_node->bwt_stats.decay = -1/10.0;
    seq_node->bwt_stats.s0 = 0.0;
    seq_node->bwt_stats.s1 = 0.0;
    seq_node->bwt_stats.s2 = 0.0;

    seq_node->bw_stats.decay = -1/10.0;
    seq_node->bw_stats.s0 = 0.0;
    seq_node->bw_stats.s1 = 0.0;
    seq_node->bw_stats.s2 = 0.0;

    seq_node->imd_stats.decay = -1/10.0;
    seq_node->imd_stats.s0 = 0.0;
    seq_node->imd_stats.s1 = 0.0;
    seq_node->imd_stats.s2 = 0.0;

    seq_node->tt_stats.decay = -1/10.0;
    seq_node->tt_stats.s0 = 0.0;
    seq_node->tt_stats.s1 = 0.0;
    seq_node->tt_stats.s2 = 0.0;

    seq_node->imdj_stats.decay = -1/10.0;
    seq_node->imdj_stats.s0 = 0.0;
    seq_node->imdj_stats.s1 = 0.0;
    seq_node->imdj_stats.s2 = 0.0;

    seq_node->ttj_stats.decay = -1/10.0;
    seq_node->ttj_stats.s0 = 0.0;
    seq_node->ttj_stats.s1 = 0.0;
    seq_node->ttj_stats.s2 = 0.0;

    seq_node->size_nexus_last = 0;
    seq_node->size_app_last = 0;

    ((startpoint_transform_state_t *)(*sp_state))->sequence = 0;

    /* inserting new sequence node into ep linked list */
    seq_node->next = ((endpoint_transform_state_t *) ep_state)->sequence_hdr;
    ((endpoint_transform_state_t *) ep_state)->sequence_hdr = seq_node;

    *copy_sp_locally = NEXUS_FALSE;
    *destroy_sp_locally = NEXUS_FALSE;
} /* transform_perf_init_startpoint_state() */

/*
 * transform_perf_destroy_startpoint_state()
 *
 * This gets called when a startpoint is destroyed.  It is passed the transform
 * data that was nexus_malloc'd in init_startpoint_state.  It needs to 
 * nexus_free() this stuff.
 */
static void transform_perf_destroy_startpoint_state(
					    nexus_transformstate_t *sp_state)
{
/*    nexus_printf("transform_perf_destroy_startpoint_state\n");*/

    NexusAssert2((sp_state), 
	("transform_perf_destroy_startpoint_state(): rcvd NULL sp_state\n"));

    nexus_free(sp_state); 
} /* transform_perf_destroy_startpoint_state() */

/*
 * void *(*copy_startpoint_state) (void *state);
 *
 * This wants a copy of a startpoints transformation data.  It simply 
 * nexus_malloc's an appropriate block, copies from state, and returns
 * new block as a void *.
 */

/*
 * transform_perf_sizeof_startpoint_state()
 *
 * This wants to know the number of bytes a startpoint's transformation
 * state requires.  It returns that value as an integer.  
 * In general, a transformation state will be made up from simple data
 * fields.  The value should be calculated by calling nexus_sizeof_XXX
 * for each field.
 * For example, the transform module using compression only requires a single 
 * int.  This function in that module simply returns nexus_sizeof_int(1).
 * Transform modules requiring 8-byte keys return nexus_sizeof_byte(8).
 * Those transform modules using compression and encryption should
 * return nexus_sizeof_int(1)+nexus_sizeof_byte(8).  
 * The nexus_sizeof_XXX() macros for all fundamental data types is in
 * nexus.h.
 */
static int transform_perf_sizeof_startpoint_state(nexus_transformstate_t *s)
{
    NexusAssert2((s), 
	("transform_perf_sizeof_startpoint_state(): rcvd NULL s\n"));
    
    /* ep_sequence_liba and sequence */
    return (2 * nexus_sizeof_byte(LARGEST_POSSIBLE_SIZEOF_ULONG));
} /* transform_perf_sizeof_startpoint_state() */

/*
 * transform_perf_put_startpoint_state()
 *
 * The user wants to put a startpoint's transformation information into
 * a buffer.  Just like sizeof_state above, this is simply a putting
 * of the individual fields of the state. 
 *
 * All puts should use the fundametal nexus_dc_put_xxx() macros found
 * in nexus.h.  For example, a transform module using compression (one int)
 * and encryption (8 bytes) should call:
 *
 * nexus_dc_put_int(buffer, &(state->compression_level), 1);
 * nexus_dc_put_byte(buffer, &(state->key), 8);
 */
static void transform_perf_put_startpoint_state(nexus_byte_t **buffer, 
						nexus_transformstate_t *state)
{
/*    nexus_printf("transform_perf_put_startpoint_state\n");*/

    NexusAssert2((buffer), 
	("transform_perf_put_startpoint_state(): rcvd NULL buffer\n"));
    NexusAssert2((state), 
	("transform_perf_put_startpoint_state(): rcvd NULL state\n"));

    /* need to place ep_sequence_liba and sequence into buffer */
    nexus_dc_put_byte(buffer, 
		    ((startpoint_transform_state_t *) state)->ep_sequence_liba, 
		    LARGEST_POSSIBLE_SIZEOF_ULONG);
    nexus_dc_put_byte(buffer, 
		    &(((startpoint_transform_state_t *) state)->sequence), 
		    LARGEST_POSSIBLE_SIZEOF_ULONG);

} /* transform_perf_put_startpoint_state() */

/*
 * transform_perf_get_startpoint_state()
 *
 * This is the exact same thing as put_state, instead we are extracting
 * information from a buffer and placing it into a startpoint.  Two
 * things should be pointed out here:
 *
 * 1. use nexus_dc_get_XXXX found in nexus.h
 *	   and
 * 2. the information MUST be extracted in the same order that it was 
 *    placed in put_state().
 */
static void transform_perf_get_startpoint_state(nexus_byte_t **buffer, 
						int format, 
						nexus_transformstate_t **state)
{
/*    nexus_printf("transform_perf_get_startpoint_state\n");*/

    NexusAssert2((buffer), 
	("transform_perf_get_startpoint_state(): rcvd NULL buffer\n"));
    NexusAssert2((state), 
	("transform_perf_get_startpoint_state(): rcvd NULL state\n"));

    *state = (startpoint_transform_state_t *) 
		nexus_malloc(sizeof(startpoint_transform_state_t));

    nexus_dc_get_byte(buffer, 
	(unsigned char *) ((startpoint_transform_state_t *) (*state))->ep_sequence_liba,
			LARGEST_POSSIBLE_SIZEOF_ULONG, 
			format); 
    nexus_dc_get_byte(buffer, 
	(unsigned char *) &(((startpoint_transform_state_t *) (*state))->sequence),
			LARGEST_POSSIBLE_SIZEOF_ULONG, 
			format); 

} /* transform_perf_get_startpoint_state() */

/*
 * transform_perf_transform()
 */
static int transform_perf_transform(nexus_transformstate_t *startpoint_state,
                                        nexus_byte_t *storage_start,
                                        unsigned long storage_size,
                                        nexus_byte_t *data_start,
                                        unsigned long *data_size,
                                        nexus_bool_t must_alloc_new_buffer,
                                        nexus_byte_t *transform_info_start,
                                        nexus_byte_t **out_storage_start,
                                        unsigned long *out_storage_size,
                                        nexus_byte_t **out_data_start,
                                        unsigned long *out_data_size)
/**************************************************************************

     storage_start
     |
     |           data_start
     |                |
     v                v
     +----------------+-----------------------+----------------+
     |  unused space  |       user data       |  unused space  |
     +----------------+-----------------------+----------------+

		      |<----- data_size ----->|

     |<--------------------- storage_size -------------------->|

Transform decides whether or not it needs to allocate a new buffer
in the same way it did before, that is, it allocates a new buffer
if must_alloc_new_buffer == NEXUS_TRUE or if there's not enough space
to transform in place.  

If it doesn't need to allocate a new buffer, then simply transform in
place, and for MD5 place the MAC into transform_info_start. 

If it DOES need to allocate a new buffer, then you set all the
out_* args to draw the following picture:

     *out_storage_start
     |
     |           *out_data_start
     |                |
     v                v
     +----------------+-----------------------+
     |  unused space  |       user data       |
     +----------------+-----------------------+

		      |<-- *out_data_size --->|

     |<-------- *out_storage_size ----------->|

You must make sure that the leading "unused space" in the newly allocated
buffer is exactly the same size as it is in the original buffer, that is,
data_start - storage_start = *out_data_start - *out_storage_start.

Note that you never allocate new space for transform_info, even if you
allocate space for a new buffer.

If you allocated a new buffer, then simply tranform into that buffer.

**************************************************************************/
{
    int i;
    char *source, *dest;
    struct timeval timestamp;
    
    NexusAssert2((startpoint_state),
        ("transform_ecb_transform(): rcvd NULL startpoint_state\n"));   
    NexusAssert2((storage_start), ("transform_ecb_transform(): rcvd NULL storage_start\n"));
    NexusAssert2((data_start),
        ("transform_ecb_transform(): rcvd NULL data_start\n"));
    NexusAssert2((data_size),
        ("transform_ecb_transform(): rcvd NULL data_size\n"));
    NexusAssert2((transform_info_start),
        ("transform_ecb_transform(): rcvd NULL transform_info_start\n"));
    NexusAssert2((out_storage_start),
        ("transform_ecb_transform(): rcvd NULL out_storage_start\n"));
    NexusAssert2((out_storage_size),
        ("transform_ecb_transform(): rcvd NULL out_storage_size\n"));
    NexusAssert2((out_data_start),
        ("transform_ecb_transform(): rcvd NULL out_data_start\n"));
    NexusAssert2((out_data_size),
        ("transform_ecb_transform(): rcvd NULL out_data_size\n"));

    /* increment sequence */
    ((startpoint_transform_state_t *)(startpoint_state))->sequence += 1;
/*
nexus_printf("transform_perf_transform(): sp_sequence = %d\n", 
	((startpoint_transform_state_t *)(startpoint_state))->sequence);
*/

    /* placing ep_sequence_liba and sequence into transform_info */
    memcpy((void *) transform_info_start,
	(void *) ((startpoint_transform_state_t *) (startpoint_state))->ep_sequence_liba,
	LARGEST_POSSIBLE_SIZEOF_ULONG);

	memcpy((void *) (transform_info_start + LARGEST_POSSIBLE_SIZEOF_ULONG),
	(void *) &(((startpoint_transform_state_t *) (startpoint_state))->sequence),
	LARGEST_POSSIBLE_SIZEOF_ULONG);

    /* Placing timestamp into transform_info */

        gettimeofday(&timestamp,NULL);

	memcpy((void *) (transform_info_start +2*LARGEST_POSSIBLE_SIZEOF_ULONG),
	(void *) &timestamp.tv_sec, LARGEST_POSSIBLE_SIZEOF_ULONG);

	memcpy((void *) (transform_info_start +3*LARGEST_POSSIBLE_SIZEOF_ULONG),
	(void *) &timestamp.tv_usec, LARGEST_POSSIBLE_SIZEOF_ULONG);

    /* determine if a new buffer is needed */
    if (must_alloc_new_buffer)
    {
/* nexus_printf("transform_perf_transform(): allocating new buffer\n"); */

		/* must allocate new buffer */
		*out_storage_size = data_start - storage_start + *data_size;
		*out_storage_start = (nexus_byte_t *) nexus_malloc(*out_storage_size);
		*out_data_start = *out_storage_start + (data_start - storage_start);
		*out_data_size = *data_size;

		/* transform from old buffer to new */
		source = (char *) data_start;
		dest = (char *) *out_data_start;

                for (i = 0; i < *data_size; i++, source++, dest++)
                {
            		*dest = *source;

                }
    } /* endif */

/* nexus_printf("exit transform_perf_transform()\n"); */

    return(NEXUS_SUCCESS);

} /* transform_perf_transform() */

/*
 * transform_perf_untransform()
 */
static int transform_perf_untransform(nexus_transformstate_t *endpoint_state,
					 nexus_byte_t *data_start,
					 unsigned long *data_size,
					 nexus_byte_t *transform_info_start,
					 int format,
					 nexus_byte_t **destination_start,
					 unsigned long *destination_size)
/**************************************************************************

     data_start
	  |
	  v
	  +-----------------------+
	  |       user data       |
	  +-----------------------+

	  |<----- data_size ----->|


The untranform process works like this:

    if (*dest_start == NULL)
    {
		if (enough room to untransform in place)
		{
	   		untransform in place
		}
		else
		{
	    	*destination_start = nexus_malloc(enough to untransform)
	    	*destination_size = amount just nexus_malloc'd
	    	untransform into newly nexus_malloc'd memory
		} 
    }
    else
    {
		if (*destination_size == EXACT amount needed to untransform)
		{
	    	untransform into *destination_size
		}
		else
		{
	    	DO NOTHING ... return non-zero value
		} 
    } 
**************************************************************************/
{
    int 			i,
        			data_format;
    unsigned long 	temp_ulong,
      		  	  	sp_sequence, time_since_last_msg;
    double new_transit_time;
    struct timeval		sender_timestamp,receiver_timestamp;
    sequence_node_t *seq_node;
    char 			*source,
	 				*dest;

    double 			weight;

    gettimeofday(&receiver_timestamp,NULL);
  
    NexusAssert2((endpoint_state),
        ("transform_ecb_untransform(): rcvd NULL endpoint_state\n"));
    NexusAssert2((data_start),
        ("transform_ecb_untransform(): rcvd NULL data_start\n"));
    NexusAssert2((data_size),
        ("transform_ecb_untransform(): rcvd NULL data_size\n"));
    NexusAssert2((transform_info_start),
        ("transform_ecb_untransform(): rcvd NULL transform_info_start\n"));
    NexusAssert2((destination_start),
        ("transform_ecb_untransform(): rcvd NULL destination_start\n"));
    NexusAssert2((destination_size),
        ("transform_ecb_untransform(): rcvd NULL destination_size\n"));

    /* extracting ep_sequence_liba and sequence from transform_info */

    memcpy((void *) &temp_ulong, 
	    (void *) transform_info_start, 
	    sizeof(unsigned long));
    seq_node = (sequence_node_t *) temp_ulong;

    memcpy((void *) &sp_sequence, 
	    (void *) (transform_info_start + LARGEST_POSSIBLE_SIZEOF_ULONG), 
	    sizeof(unsigned long));

    memcpy((void *) &sender_timestamp.tv_sec,
	    (void *) (transform_info_start + 2*LARGEST_POSSIBLE_SIZEOF_ULONG), 
	    sizeof(unsigned long));

    memcpy((void *) &sender_timestamp.tv_usec,
	    (void *) (transform_info_start + 3*LARGEST_POSSIBLE_SIZEOF_ULONG), 
	    sizeof(unsigned long));

    NexusAssert2((seq_node), 
    ("transform_perf_untransform(): extracted NULL sequence node pointer from buffer\n"));


    time_since_last_msg = 0;

    if (sender_timestamp.tv_sec == receiver_timestamp.tv_sec)
    {
         new_transit_time = receiver_timestamp.tv_usec - sender_timestamp.tv_usec;
    }
    else
    {
         new_transit_time  = receiver_timestamp.tv_usec + (1000000-sender_timestamp.tv_usec) +
                    (1000000 * (receiver_timestamp.tv_sec - sender_timestamp.tv_sec - 1));
    }

    if (seq_node -> sequence > 2)
    {
	    if (receiver_timestamp.tv_sec == seq_node->last_time.tv_sec)
	    {
	         time_since_last_msg = receiver_timestamp.tv_usec - seq_node->last_time.tv_usec;
	    }
	    else
	    {
	         time_since_last_msg = receiver_timestamp.tv_usec + (1000000-seq_node->last_time.tv_usec) +
	                    (1000000 * (receiver_timestamp.tv_sec - seq_node->last_time.tv_sec - 1));
	    }

        seq_node -> bandwidth_last    = (*data_size * 1000000) / time_since_last_msg;
	seq_node -> bandwidth_total_last = ((data_start-transform_info_start+*data_size)*1000000) /time_since_last_msg;

	weight = time_since_last_msg/1000000.0;

	compute_perf_stats (&seq_node->bwt_stats,
                            seq_node->bandwidth_total_last,
                            time_since_last_msg,
                            &seq_node->bandwidth_total, &seq_node->bandwidth_total_stddev);

	compute_perf_stats (&seq_node->bw_stats,
                            seq_node->bandwidth_last,
                            time_since_last_msg, 
                            &seq_node->bandwidth, &seq_node->bandwidth_stddev);

    }
    else
    {
	weight = time_since_last_msg/1000000;
    }

	/* Make it so the jitter for the first set is 0 and not the full current value */


	if (seq_node->inter_msg_delay_last == 0)
	{
		seq_node->inter_msg_delay_last = time_since_last_msg;
	}

	if (seq_node->transit_time_last == 0)
	{
		seq_node->transit_time_last = new_transit_time;
	}

	compute_perf_stats (&seq_node->imdj_stats,
                            fake_fabs(time_since_last_msg - seq_node->inter_msg_delay_last),
                            weight,
                            &seq_node->inter_msg_delay_jitter, NULL);

	compute_perf_stats (&seq_node->ttj_stats,
                            fake_fabs(new_transit_time - seq_node->transit_time_last),
                            weight,
                            &seq_node->transit_time_jitter, NULL);

    seq_node -> transit_time_last	= new_transit_time;
    seq_node -> inter_msg_delay_last = time_since_last_msg;

	compute_perf_stats (&seq_node->imd_stats,
                            seq_node->inter_msg_delay_last,
                            weight,
                            &seq_node->inter_msg_delay, &seq_node->inter_msg_delay_stddev);

	compute_perf_stats (&seq_node->tt_stats,
                            seq_node->transit_time_last,
                            weight,
                            &seq_node->transit_time, &seq_node->transit_time_stddev);

    seq_node -> last_time	= receiver_timestamp;
    seq_node -> last_send_time	= sender_timestamp;

    seq_node -> size_nexus_last = data_start-transform_info_start+*data_size;
    seq_node -> size_app_last    = *data_size;

    /* take care of sequence and lost message information */
    if (sp_sequence == (seq_node->sequence + 1))
    {
		/* received the message number we expected */
		seq_node->sequence = sp_sequence;
    }
    else if (sp_sequence > (seq_node->sequence + 1))
    {
		/* received a message number greater than we expected -- some messages lost */
		seq_node->msgs_lost = sp_sequence - (seq_node-> sequence + 1);
		seq_node->sequence = sp_sequence;
    }
    else
    {
		/* received a message we earlier counted as lost */
		seq_node->msgs_lost -= 1;
    }

    /* see if we need to put it in a new buffer */
    if (*destination_start)
    {
	    	source = (char *) data_start;
	    	dest = (char *) *destination_start;

                for (i = 0; i < *data_size; i++, source++, dest++)
                {
	        	*dest = *source - 1;
                } 
    }

/* nexus_printf("exit transform_perf_untransform()\n"); */

    return(NEXUS_SUCCESS);

} /* transform_perf_untransform */

/*
 * retrieve the current state of startpoint or endpoint. if (flag = 0) then
 * retrieve startpoint information, otherwise, retrieve endpoint infomation.
 */

static void transform_perf_transformstate_get_info(
                                                nexus_transformstate_t *state,
                                                void *info, int flag)
{
        if(flag == 0) { /* get info. of startpoint state */
                /* *info = (startpoint_transform_state_t *)
                        malloc(1, sizeof(startpoint_transform_state_t)); */
                memcpy(info,
                        (startpoint_transform_state_t *)state,
                        sizeof(startpoint_transform_state_t));
        }  else { /* get info. of endpoint state */
                /* *info = (nexus_session_t *)
                        calloc(1, sizeof(nexus_session_t)); */
                memcpy(info,
                       ((endpoint_transform_state_t *)state)->sequence_hdr,
                        sizeof(sequence_node_t));
        }
}

/**********************************************
 * Fake math functions                        *
 **********************************************/

  /* Gives about 9-10 decimal digits of precision */
double fake_exp(double v) {
  int inv = 0;
  int iv, index, bit,tmp;
  double e;

  if(v < 0) {
        v = -v;
        inv = 1;
  }
  iv = (int)(v * ESCALE + .5);
  v -= iv/(double)ESCALE;

  e = ((v*.166666666667 + .5)*v + 1)*v + 1;     /* 1 + v + v^2/2 + v^3/6 */

/* This didn't compile with -xO3 with Sun's CC
  while(iv >= EMAX) {
        iv -= EMAX;
        e *= EEMAX;
  }
*/
  tmp = iv/EMAX;
  while(tmp--)
  {
	e *= EEMAX;
  }


  for(index = 0, bit = 1; index < COUNT(epow); index++) {
    if(iv & bit)
        e *= epow[index];
    bit += bit;
    if(iv < bit)
        break;
  }
  return inv ? 1/e : e;
}

double fake_fabs( double v)
{
    if(v < 0)
    {
	return -v;
    }
    else
    {
	return v;
    }
}

  /* Gives about 15 digits of precision */
double fake_sqrt( double v ) {
  int iv;
  int inv = 0;
  int log2 = 0;
  double s = 1, s2;

  if(v <= 0)
    return 0;

  /* Range reduction.  This would be faster with scalb(), but
   * without -lm, we don't have that either.
   */
  if(v > 4) {
    while(v > 16) {
        v *= 0.015625;
        s *= 8;
    }
    /* Now v in 1/4 .. 16 */
    if(v > 4) {
        v *= 0.25;
        s *= 2;
    }
  } else if(v < 0.25) {
    while(v < 0.0625) {
        v *= 64;
        s *= 0.125;
    }
    /* Now v in 1/16 .. 4 */
    if(v < 0.25) {
        v *= 4;
        s *= 0.5;
    }
  }
  /* Now we have v in range .25 .. 4 */
  /* This many Newton's method steps give us at least 15 digits of precision! */
  s2 = (1 + v) * .5;
  s2 = (s2 + v/s2) * .5;
  s2 = (s2 + v/s2) * .5;
  s2 = (s2 + v/s2) * .5;
  s2 = (s2 + v/s2) * .5;
  return s2*s;
}


