-- gitano.plugins
--
-- Plugin loading support for Gitano
--
-- Copyright 2014 Daniel Silverstone <daniel.silverstone@codethink.co.uk>

local util = require "gitano.util"
local log = require "gitano.log"
local i18n = require "gitano.i18n"

local luxio = require "luxio"
local sio = require "luxio.simple"

local gfind = string.gfind

local plugin_name_pattern = "^([^_]+)%.lua$"

local function find_plugins(path)
   local ret = {}
   for _, entry in ipairs(path) do
      local dirp, err = sio.opendir(entry)
      if not dirp then
         log.warning(("Unable to scan plugin directory '%s': %s")
                     :format(entry, err))
      else
         for filename, fileinfo in dirp:iterate() do
            local plugin_name = filename:match(plugin_name_pattern)
            if plugin_name then
               if not ret[plugin_name] then
                  ret[plugin_name] = entry
                  ret[#ret + 1] = plugin_name
               end
            end
         end
      end
   end
   table.sort(ret)
   return ret
end

local function load_plugins(path)
   local to_load = find_plugins(path)
   for _, plugin_name in ipairs(to_load) do
      local filepath = util.path_join(to_load[plugin_name],
                                      plugin_name .. ".lua")
      local chunk, err = loadfile(filepath)
      if not chunk then
         log.warning(("Failure loading plugin '%s' from '%s': %s")
                     :format(plugin_name, to_load[plugin_name],
                             err))
      else
         local ok, err = pcall(chunk)
         if not ok then
            log.warning(("Failure running plugin '%s' from '%s': %s")
                        :format(plugin_name, to_load[plugin_name],
                                err))
         end
      end
      i18n.add_plugin_path(util.path_join(to_load[plugin_name],
					  plugin_name))
   end
end

return {
   load_plugins = load_plugins,
}
