-- gitano.lace
--
-- Interface to Lace for rules management
--
-- Copyright 2012-2016 Daniel Silverstone <dsilvers@digital-scurf.org>

local lace = require 'lace'
local util = require 'gitano.util'
local gall = require 'gall'
local log = require 'gitano.log'

local pcre = require "rex_pcre"

local global_lookaside = {
   ["_basis"] = [[
include global:core
]]
}

local function _loader(ctx, _name)
   local global_name = _name:match("^global:(.+)$")
   local name, tree, sha = global_name or _name
   if not global_name then
      -- Project load
      if ctx.project and not ctx.project_tree then
	 ctx.project_tree = gall.tree.flatten(ctx.project.content.tree.content)
      end
      if ctx.project_tree then
	 tree = ctx.project_tree
	 sha = ctx.project.sha
      end
   else
      -- Global load
      if global_lookaside[global_name] then
	 local resolved = global_lookaside[global_name]
	 if type(resolved) == "function" then
	    resolved = resolved(ctx)
	 end
	 return "global_lookaside::" .. global_name, resolved
      end
      if not ctx.global_tree then
	 ctx.global_tree = gall.tree.flatten(ctx.global.content.tree.content)
      end
      tree = ctx.global_tree
      sha = ctx.global.sha
   end
   if not tree then
      log.ddebug("Returning empty object (nascent repository?)")
      return "empty", "# Nothing to see here\n"
   end
   local blob_name = "rules/" .. name .. ".lace"
   local real_name = sha .. "::" .. blob_name
   if not tree[blob_name] or tree[blob_name].type ~= "blob" then
      local m = ("Unable to find %s in %s (%s)"):format(blob_name, sha, _name)
      log.ddebug(m)
      return lace.error.error(m)
   end
   --log.ddebug("Lace->Loading", real_name)
   return real_name, tree[blob_name].obj.content
end

local match_types = {
   exact = function(want, have)
	      return want == have
	   end,
   prefix = function(want, have)
	       return have:sub(1, #want) == want
	    end,
   suffix = function(want, have)
	       return have:sub(-#want) == want
	    end,
   pattern = function(want, have)
		return (have:match(want) ~= nil)
	     end,
   pcre = function(want, have)
	     return (pcre.match(have, want) ~= nil)
	  end
}

-- Match aliases (auto-inverted)
match_types.is = match_types.exact
match_types.starts = match_types.prefix
match_types.ends = match_types.suffix
match_types.startswith = match_types.prefix
match_types.endswith = match_types.suffix

-- Inverted matches
do
   local inverted_matches = {}
   for k, v in pairs(match_types) do
      inverted_matches["!" .. k] = function(...) return not v(...) end
   end
   for k, v in pairs(inverted_matches) do
      match_types[k] = v
   end
end

-- Match aliases (not auto-inverted)
match_types["not"] = match_types["!is"]

local function _do_simple_match(ctx, key, matchtype, value)
   value = util.process_expansion(ctx, value)
   local kk = ctx[key] or ""
   local check = match_types[matchtype]
   if type(kk) == "function" then
      -- Realise the value first
      ctx[key] = kk(ctx)
      kk = ctx[key]
   end
   if type(kk) == "string" then
      return check(value, kk)
   else
      local ret = false
      for k in pairs(kk) do
	 ret = ret or check(value, k)
      end
      if matchtype:sub(1,1) == "!" then
	 ret = not ret
      end
      return ret
   end
end

local function _simple_match(ctx, key, matchtype, value, guard)
   if guard ~= nil then
      return lace.error.error("Unexpected additional argument", {4})
   end
   if value == nil then
      return lace.error.error("Missing matchtype or value", {1})
   end
   if match_types[matchtype] == nil then
      return lace.error.error("Unknown match type", {2})
   end
   return {
      fn = _do_simple_match,
      args = { key, matchtype, util.prep_expansion(value) }
   }
end

local simples = {
   -- Trivial strings/lists
   "operation", "owner", "ref", "group", "user", "repository",
   "repository/basename", "repository/dirname",
   -- Trees for update operations (flat lists)
   "start_tree", "target_tree",
   -- Tree diffs for update operations (flat lists)
   "treediff/targets", "treediff/added",
   "treediff/deleted", "treediff/modified", "treediff/renamed",
   "treediff/renamedto",
   -- Stuff for 'as' users
   "as_user", "as_group",
   -- Stuff for site admin commands
   "targetuser", "targetgroup", "member",
   "targetgroup/prefix", "targetgroup/suffix",
   "member/prefix", "member/suffix",
   -- Stuff from update hooks for object types
   "oldtype", "oldtaggedtype", "oldtaggedsha", "oldsigned",
   "newtype", "newtaggedtype", "newtaggedsha", "newsigned",
   -- Stuff for keyring command
   "keyringname",
}

local matchers = {}

for _, s in ipairs(simples) do
   matchers[s] = _simple_match
end

local base_compcontext = {
   _lace = {
      loader = _loader,
      controltype = matchers,
   }
}

local function cloddly_bless(ctx)
   local function indexer(tab, name)
      if name:sub(1,7) == "config/" then
	 tab[name] = _simple_match
	 log.ddebug("[lace] Auto-vivifying " .. name)
	 return _simple_match
      end
   end
   setmetatable(ctx._lace.controltype, {__index = indexer})
   return ctx
end

local function compile_ruleset(repo, adminsha, globaladminsha)
   -- repo is a gitano repository object.
   -- We trust that we can compile the repo's ruleset which involves
   -- finding the admin sha for the repo (unless given) and then the global
   -- admin sha (unless given) and using that to compile a full ruleset.
   local compcontext = cloddly_bless(util.deep_copy(base_compcontext))
   compcontext.repo = repo
   if not repo.is_nascent then
      if not adminsha then
	 compcontext.project = repo.git:get("refs/gitano/admin")
      else
	 compcontext.project = repo.git:get(adminsha)
      end
   end
   if not globaladminsha then
      compcontext.global = repo.config.commit
   else
      compcontext.global = repo.config.repo.git:get(globaladminsha)
   end

   return lace.compiler.compile(compcontext, "global:_basis")
end

local function run_ruleset(ruleset, ctx)
   return lace.engine.run(ruleset, ctx)
end

return {
   compile = compile_ruleset,
   run = run_ruleset,
}
