package storage

import (
	"net"
	"os"
	"path/filepath"
	"testing"
	"time"

	pb "gitlab.com/gitlab-org/gitaly-proto/go"
	"gitlab.com/gitlab-org/gitaly/internal/config"
	"gitlab.com/gitlab-org/gitaly/internal/testhelper"

	"google.golang.org/grpc"
	"google.golang.org/grpc/reflection"
)

var testStorage config.Storage

func TestMain(m *testing.M) {
	configureTestStorage()
	os.Exit(m.Run())
}

func configureTestStorage() {
	storagePath, err := filepath.Abs("testdata/repositories/storage1")
	if err != nil {
		panic(err)
	}

	if err := os.RemoveAll(storagePath); err != nil {
		panic(err)
	}

	if err := os.MkdirAll(storagePath, 0755); err != nil {
		panic(err)
	}

	testStorage = config.Storage{Name: "storage-will-be-deleted", Path: storagePath}

	config.Config.Storages = []config.Storage{testStorage}
}

func runStorageServer(t *testing.T) (*grpc.Server, string) {
	server := testhelper.NewTestGrpcServer(t, nil, nil)
	serverSocketPath := testhelper.GetTemporaryGitalySocketFileName()

	listener, err := net.Listen("unix", serverSocketPath)
	if err != nil {
		t.Fatal(err)
	}

	pb.RegisterStorageServiceServer(server, NewServer())
	reflection.Register(server)

	go server.Serve(listener)

	return server, serverSocketPath
}

func newStorageClient(t *testing.T, serverSocketPath string) (pb.StorageServiceClient, *grpc.ClientConn) {
	connOpts := []grpc.DialOption{
		grpc.WithInsecure(),
		grpc.WithDialer(func(addr string, timeout time.Duration) (net.Conn, error) {
			return net.DialTimeout("unix", addr, timeout)
		}),
	}
	conn, err := grpc.Dial(serverSocketPath, connOpts...)
	if err != nil {
		t.Fatal(err)
	}

	return pb.NewStorageServiceClient(conn), conn
}
