package fieldextractors

import (
	"strings"

	pb "gitlab.com/gitlab-org/gitaly-proto/go"
)

type repositoryBasedRequest interface {
	GetRepository() *pb.Repository
}

type namespaceBasedRequest interface {
	storageBasedRequest
	GetName() string
}

type storageBasedRequest interface {
	GetStorageName() string
}

func formatRepoRequest(repo *pb.Repository) map[string]interface{} {
	if repo == nil {
		// Signals that the client did not send a repo through, which
		// will be useful for logging
		return map[string]interface{}{
			"repo": nil,
		}
	}

	return map[string]interface{}{
		"repoStorage":   repo.StorageName,
		"repoPath":      repo.RelativePath,
		"topLevelGroup": getTopLevelGroupFromRepoPath(repo.RelativePath),
		"glRepository":  repo.GlRepository,
	}
}

// getTopLevelGroupFromRepoPath gives the top-level group name, given
// a repoPath. For example:
// - "gitlab-org/gitlab-ce.git" returns "gitlab-org"
// - "gitlab-org/gitter/webapp.git" returns "gitlab-org"
// - "x.git" returns ""
func getTopLevelGroupFromRepoPath(repoPath string) string {
	parts := strings.SplitN(repoPath, "/", 2)
	if len(parts) != 2 {
		return ""
	}

	return parts[0]
}

func formatStorageRequest(storageReq storageBasedRequest) map[string]interface{} {
	return map[string]interface{}{
		"StorageName": storageReq.GetStorageName(),
	}
}

func formatNamespaceRequest(namespaceReq namespaceBasedRequest) map[string]interface{} {
	return map[string]interface{}{
		"StorageName": namespaceReq.GetStorageName(),
		"Name":        namespaceReq.GetName(),
	}
}

func formatRenameNamespaceRequest(renameReq pb.RenameNamespaceRequest) map[string]interface{} {
	return map[string]interface{}{
		"StorageName": renameReq.GetStorageName(),
		"From":        renameReq.GetFrom(),
		"To":          renameReq.GetTo(),
	}
}

// FieldExtractor will extract the relevant fields from an incoming grpc request
func FieldExtractor(fullMethod string, req interface{}) map[string]interface{} {
	if req == nil {
		return nil
	}

	switch req.(type) {
	case pb.RenameNamespaceRequest:
		return formatRenameNamespaceRequest(req.(pb.RenameNamespaceRequest))
	case repositoryBasedRequest:
		return formatRepoRequest(req.(repositoryBasedRequest).GetRepository())
	case namespaceBasedRequest:
		return formatNamespaceRequest(req.(namespaceBasedRequest))
	case storageBasedRequest:
		return formatStorageRequest(req.(storageBasedRequest))
	}

	return nil

}
