# Generating the FFI library

First, you'll need to download [gir]:

```sh
git clone https://github.com/gtk-rs/gir
cd gir
cargo install --path . # so we can use gir binary directly
```

Then the `.gir` files (luckily for you, we have a repository which contains all the ones you need for sourceview!):

```sh
git clone https://github.com/gtk-rs/gir-files
```

If you look into `gir-files`, you'll see a file named `GtkSource-3.0.gir`. That's the one for sourceview.

Now let's create a new project for our `sourceview` crate:

```sh
cargo new sourceview --lib
```

Then let's create a folder inside the newly created `sourceview` folder for the `sys` part:

```sh
cd sourceview
cargo new sourceview-sys --lib
```

To indicate to [gir] what to generate, we'll need a `Gir.toml` file (inside the `sourceview-sys` folder) containing:

```toml
[options]
library = "GtkSource"
version = "3.0"
target_path = "."
min_cfg_version = "3.0"
```

* `library` stands for the library we want to generate.
* `version` stands for the version of the library to be used.
* `target_path` stands for the location where the files will be generated.
* `min_cfg_version` will be the minimum version supported by the generated bindings.

You should now have a folder looking like this:

```text
sourceview/
  |
  |---- Cargo.toml
  |---- sourceview-sys/
  |       |
  |       |---- Cargo.toml
  |       |---- Gir.toml
  |       |---- src/
  |               |
  |               |---- lib.rs
  |---- src/
          |
          |---- lib.rs
```

Let's generate the `sys` crate now:

```sh
cd sourceview-sys
# Run gir in "sys" mode (the "-m" option). We give the gir files path (the "-d" option) and output path (the "-o" option).
gir -m sys -d ../../gir-files/ -o .
```

(In case a failure happens at this point, and you can't figure out what's going on, don't hesitate to reach us so we can give you a hand!)

You should now see new files (and a new folder):

* `build.rs`
* `Cargo.toml`
* `src/lib.rs`
* `tests/`

Now let's try to build it:

```sh
cargo build
```

Surprise! It doesn't build at all and you should see a loooooot of errors. Well, that was expected. We need to add some dependencies (you can find which ones in the `.gir` files) in order to make it work. Let's update our `Gir.toml` file to make it look like this:

```toml
[options]
library = "GtkSource"
version = "3.0"
target_path = "."
min_cfg_version = "3.0"

external_libraries = [
    "Cairo",
    "Gdk",
    "GdkPixbuf",
    "Gio",
    "GLib",
    "GObject",
    "Gtk",
]
```

Now we regenerate it then rebuild it:

```sh
rm Cargo.* # we remove Cargo files
gir -m sys -d ../../gir-files/ -o .
cargo build
```

Should work just fine!

We can cleanup the command line a bit now. You can actually give the work mode ("-m" option) and the gir files repository through the `Gir.toml` file using "work_mode" and "girs_directories" options:

```toml
[options]
library = "GtkSource"
version = "3.0"
target_path = "."
min_cfg_version = "3.0"
work_mode = "sys"
girs_directories = ["../../gir-files/"]

external_libraries = [
    "Cairo",
    "Gdk",
    "GdkPixbuf",
    "Gio",
    "GLib",
    "GObject",
    "Gtk",
]
```

Now, if you want to regenerate, just run:

```sh
gir -o .
```

Now we have a working `sys` containing all functions and objects definition. Just to be sure everything was correctly generated, we can run some tests (graciously generated by [gir] as well):

```sh
cargo test
```

Normally, all tests passed. If you get an error when running those tests, it's very likely that the `sys` generation is invalid and/or incomplete.

Time to generate the high-level Rust API!

[gir]: https://github.com/gtk-rs/gir
