/*
 *  
 *  $Id: hangingprotocolcontroller.cpp $
 *  Ginkgo CADx Project
 *
 *  Copyright 2008-14 MetaEmotion S.L. All rights reserved.
 *  http://ginkgo-cadx.com
 *
 *  This file is licensed under LGPL v3 license.
 *  See License.txt for details
 *
 *
 */


#include "hangingprotocolcontroller.h"
#include <limits> 
#include <iomanip>
#include <iostream>
#include <main/controllers/configurationcontroller.h>
#include <main/controllers/controladorlog.h>
#include <main/controllers/historycontroller.h>
#include <main/controllers/dicommanager.h>
#include <api/dicom/dicomdataset.h>
#include <api/dicom/dcmdictionary.h>
#include <api/ivista.h>
#include <api/internationalization/internationalization.h>
#include <main/gui/mainwindow/ventanaprincipal.h>
#include <main/entorno.h>

#include <wx/xml/xml.h>
#include <wx/mstream.h>
#include <wx/sstream.h>
#include <wx/regex.h>
#include <wx/sizer.h>
#include <wx/display.h>

#define HP_DOCUMENT_ROOT wxT("hanging-protocols")

#define PROPERTY_NODE wxT("property")
#define PROPERTY_KEY wxT("key")
#define PROPERTY_VALUE wxT("value")

#define HP_ROOT wxT("hanging-protocol")
#define HP_ATT_NAME wxT("name")
#define HP_ATT_DESCRIPTION wxT("description")
#define HP_ATT_MODULE_SID wxT("module-sid")
#define HP_ATT_MODE_ID wxT("mode")
#define HP_ATT_ACTIVE wxT("active")

#define LOS_ROOT wxT("layouts")
#define LO_ROOT wxT("layout")
#define LO_ATT_ROWS wxT("rows")
#define LO_ATT_COLUMNS wxT("columns")
#define LO_ATT_DETACHED wxT("detached")
#define LO_ATT_MAXIMIZED wxT("maximized")
#define LO_ATT_X_POS wxT("x-pos")
#define LO_ATT_Y_POS wxT("y-pos")
#define LO_ATT_WIDTH wxT("width")
#define LO_ATT_HEIGHT wxT("height")

#define ML_ROOT wxT("modality-list")
#define DMS_ROOT wxT("def-mod-settings")
#define DMS_ATT_MODALITY wxT("modality")
#define DMS_ATT_INITIAL_WL wxT("initial-wl")
#define DMS_ATT_INTERPOLATION wxT("interpolation-mode")
#define DMS_ATT_INT_NEAREST wxT("nearest-neighbour")
#define DMS_ATT_LINEAR wxT("linear")
#define DMS_ATT_CUBIC wxT("cubic")

#define WLS_ROOT wxT("window-levels")
#define WL_ROOT wxT("window-level")
#define WL_ATT_WINDOW wxT("window")
#define WL_ATT_LEVEL wxT("level")
#define WL_ATT_LABEL wxT("label")


#define MC_ROOT wxT("matcher")
#define MC_ATT_CRITERIA wxT("criteria")

/**
Class that evaluates study and series matching, depending on matching criteria over tags
*/
GNC::GCS::Matcher::Matcher()
{
}

GNC::GCS::Matcher::Matcher(wxXmlNode* matcherNode)
{
	for (wxXmlNode* child = matcherNode->GetChildren(); child != NULL; child = child->GetNext())
	{
		if (child->GetName().CmpNoCase(MC_ATT_CRITERIA) == 0) {
			for (wxXmlNode* prop = child->GetChildren(); prop != NULL; prop = prop->GetNext()) 
			{
				if (prop->GetName().CmpNoCase(PROPERTY_NODE) == 0) {
					const wxString key = prop->GetAttribute(PROPERTY_KEY, wxEmptyString);
					const wxString value = prop->GetAttribute(PROPERTY_VALUE, wxEmptyString);
					if (!key.IsEmpty()) {
						this->matchingCriteria[std::string(key.ToUTF8())] = value.ToUTF8();
					}
				} else {
					LOG_ERROR("Matcher", "error deserializing matching criteria of Matcher");
				}
			}
		} else {
			LOG_ERROR("Matcher", "error deserializing Matcher, unknown node " << child->GetName().ToUTF8());
		}
	}	
}

GNC::GCS::Matcher::Matcher(const GNC::GCS::Matcher& other)
{
	this->matchingCriteria = other.matchingCriteria;
}

GNC::GCS::Matcher::~Matcher()
{
}

bool GNC::GCS::Matcher::MatchesStudy(long idStudy)
{	
	//tries with study tags from database...
	GIL::DICOM::DicomDataset databaseDataset;
	GNC::GCS::HistoryController::Instance()->GetDatabaseDatasetFromStudy(idStudy, databaseDataset);

	std::string tmpValue;
	for (std::map<std::string, std::string>::const_iterator it = this->matchingCriteria.begin(); it != this->matchingCriteria.end(); ++it)
	{
		if (databaseDataset.getTag((*it).first, tmpValue)) {
			wxRegEx regularExpresion(wxString::FromUTF8((*it).second.c_str()), wxRE_ICASE);
			if (!regularExpresion.Matches(wxString::FromUTF8(tmpValue.c_str()))) {
				return false;
			}
		} else {
			//this attribute isn't in database, so we have to parse a file from the study....
			GNC::GCS::IHistoryController::SeriesModelList seriesList;
			GNC::GCS::HistoryController::Instance()->GetSeriesFromStudy(idStudy, seriesList);
			if (!seriesList.empty()) {
				GNC::GCS::IHistoryController::FileModel file = GNC::GCS::HistoryController::Instance()->GetFrameOfReference(seriesList.front().pk);
				return matches(file.real_path);
			}				
		}
	}
	return true;
}
bool GNC::GCS::Matcher::MatchesSeries(long idSeries)
{
	//tries with study tags from database...
	GIL::DICOM::DicomDataset databaseDataset;
	GNC::GCS::HistoryController::Instance()->GetDatabaseDatasetFromSeries(idSeries, databaseDataset);

	std::string tmpValue;
	for (std::map<std::string, std::string>::const_iterator it = this->matchingCriteria.begin(); it != this->matchingCriteria.end(); ++it)
	{
		if (databaseDataset.getTag((*it).first, tmpValue)) {
			wxRegEx regularExpresion(wxString::FromUTF8((*it).second.c_str()), wxRE_ICASE);
			if (!regularExpresion.Matches(wxString::FromUTF8(tmpValue.c_str()))) {
				return false;
			}
		} else {
			//this attribute isn't in database, so we have to parse a file from the study....
			GNC::GCS::IHistoryController::FileModel file = GNC::GCS::HistoryController::Instance()->GetFrameOfReference(idSeries);
			return matches(file.real_path);				
		}
	}
	return true;
}

bool GNC::GCS::Matcher::matches(const std::string& pathOfFile)
{
	GIL::DICOM::DICOMManager manager;
	GIL::DICOM::DicomDataset base;
	manager.CargarFichero(pathOfFile, base);
	std::string tmpValue;
	for (std::map<std::string, std::string>::const_iterator it = this->matchingCriteria.begin(); it != this->matchingCriteria.end(); ++it)
	{
		if (base.getTag((*it).first, tmpValue)) {
			wxRegEx regularExpresion(wxString::FromUTF8((*it).second.c_str()));
			if (!regularExpresion.Matches(wxString::FromUTF8(tmpValue.c_str()))) {
				return false;
			}
		} else {
			LOG_DEBUG("Matcher", "study doesn't contains " << (*it).first << " attribute");
			return false;
		}
	}
	return true;
}

wxXmlNode* GNC::GCS::Matcher::serialize()
{
	wxXmlNode* matcherNode = new wxXmlNode(NULL, wxXML_ELEMENT_NODE, MC_ROOT);
	
	wxXmlNode* matchingCriteria = new wxXmlNode(matcherNode, wxXML_ELEMENT_NODE, MC_ATT_CRITERIA);
	for (std::map<std::string, std::string>::const_iterator it = this->matchingCriteria.begin(); it != this->matchingCriteria.end(); ++it)
	{
		wxXmlNode* propertyNode = new wxXmlNode(matchingCriteria, wxXML_ELEMENT_NODE, PROPERTY_NODE);
		propertyNode->AddAttribute(PROPERTY_KEY, wxString::FromUTF8((*it).first.c_str()));
		propertyNode->AddAttribute(PROPERTY_VALUE, wxString::FromUTF8((*it).second.c_str()));
	}

	return matcherNode;
}

std::map<std::string, std::string>& GNC::GCS::Matcher::getMatchingCriteria()
{
	return this->matchingCriteria;
}

void GNC::GCS::Matcher::setMatchingCriteria(const std::map<std::string, std::string>& criteria)
{
	this->matchingCriteria = criteria;
}


/**
Defines a specific layout, is recursive to allow splitted cells
*/
GNC::GCS::Layout::Layout():rows(1), columns(1),isDetached(false)
{
}

GNC::GCS::Layout::Layout(wxXmlNode* layoutRoot)
{
	long tmp;
	if (layoutRoot->GetAttribute(LO_ATT_ROWS, wxT("1")).ToLong(&tmp)) {
		rows = tmp;
	}
	if (layoutRoot->GetAttribute(LO_ATT_COLUMNS, wxT("1")).ToLong(&tmp)) {
		columns = tmp;
	}
	wxString foo;
	if (layoutRoot->GetAttribute(LO_ATT_DETACHED, &foo)) {
		this->isDetached = true;
		isMaximized = layoutRoot->GetAttribute(LO_ATT_MAXIMIZED, &foo);
		if (layoutRoot->GetAttribute(LO_ATT_X_POS, wxT("-1")).ToLong(&tmp)) {
			xPos = tmp;
		}
		if (layoutRoot->GetAttribute(LO_ATT_Y_POS, wxT("-1")).ToLong(&tmp)) {
			yPos = tmp;
		}
		if (layoutRoot->GetAttribute(LO_ATT_WIDTH, wxT("-1")).ToLong(&tmp)) {
			width = tmp;
		}
		if (layoutRoot->GetAttribute(LO_ATT_HEIGHT, wxT("-1")).ToLong(&tmp)) {
			height = tmp;
		}
	} else {
		this->isDetached = false;
	}

	for (wxXmlNode* child = layoutRoot->GetChildren(); child != NULL; child = child->GetNext())
	{
		if (child->GetName().CmpNoCase(LO_ROOT) == 0) {
			GNC::GCS::Ptr<Layout> childrenLayout(new Layout(child));
			cellList.push_back(childrenLayout);
		} else if (child->GetName().CmpNoCase(MC_ROOT) == 0) {
			seriesMatcher = new Matcher(child);
		} else if (child->GetName().CmpNoCase(PROPERTY_NODE) == 0) {
			const wxString key = child->GetAttribute(PROPERTY_KEY, wxEmptyString);
			const wxString value = child->GetAttribute(PROPERTY_VALUE, wxEmptyString);
			if (!key.IsEmpty()) {
				this->properties[std::string(key.ToUTF8())] = value.ToUTF8();
			}			
		} else {
			LOG_ERROR("Layout", "error deserializing layout, unknown node " << child->GetName().ToUTF8());
		}
	}	
}

GNC::GCS::Layout::Layout(const Layout& other)
{
	for (std::list<GNC::GCS::Ptr<GNC::GCS::Layout> >::const_iterator it = other.cellList.begin(); it != other.cellList.end(); it++)
	{
		GNC::GCS::Ptr<GNC::GCS::Layout> layout(new Layout(*(*it)));
		this->cellList.push_back(layout);
	}
	if (other.seriesMatcher.IsValid()) {
		GNC::GCS::Ptr<GNC::GCS::Matcher> matcher(new Matcher(*other.seriesMatcher));
		this->seriesMatcher = matcher;
	}
	this->properties = other.properties;
	this->columns = other.columns;
	this->rows = other.rows;
	this->isDetached = other.isDetached;
	this->isMaximized = other.isMaximized;
	this->xPos = other.xPos;
	this->yPos = other.yPos;
	this->width = other.width;
	this->height = other.height;
}

GNC::GCS::Layout::~Layout()
{
}
/**
Layout with a row and a column is a cell and cellList is empty
*/
bool GNC::GCS::Layout::isACell()
{
	return rows == 1 && columns == 1;
}

wxXmlNode* GNC::GCS::Layout::serialize()
{
	wxXmlNode* layoutNode = new wxXmlNode(NULL, wxXML_ELEMENT_NODE, LO_ROOT);
	layoutNode->AddAttribute(LO_ATT_ROWS, wxString::Format(wxT("%d"), this->rows));
	layoutNode->AddAttribute(LO_ATT_COLUMNS, wxString::Format(wxT("%d"), this->columns));
	for (std::list<GNC::GCS::Ptr<Layout> >::const_iterator it = this->cellList.begin(); it != this->cellList.end(); ++it)
	{
		if ((*it).IsValid()) {
			layoutNode->AddChild((*it)->serialize());
		}
	}

	if (seriesMatcher.IsValid()) {
		layoutNode->AddChild(seriesMatcher->serialize());
	}

	for (std::map<std::string, std::string>::const_iterator it = this->properties.begin(); it != this->properties.end(); ++it)
	{
		wxXmlNode* propertyNode = new wxXmlNode(layoutNode, wxXML_ELEMENT_NODE, PROPERTY_NODE);
		propertyNode->AddAttribute(PROPERTY_KEY, wxString::FromUTF8((*it).first.c_str()));
		propertyNode->AddAttribute(PROPERTY_VALUE, wxString::FromUTF8((*it).second.c_str()));
	}

	if (isDetached) {
		layoutNode->AddAttribute(LO_ATT_DETACHED, wxEmptyString);
		if (isMaximized) {
			layoutNode->AddAttribute(LO_ATT_MAXIMIZED, wxEmptyString);
		}
		layoutNode->AddAttribute(LO_ATT_X_POS, wxString::Format(wxT("%d"), this->xPos));
		layoutNode->AddAttribute(LO_ATT_Y_POS, wxString::Format(wxT("%d"), this->yPos));
		layoutNode->AddAttribute(LO_ATT_WIDTH, wxString::Format(wxT("%d"), this->width));
		layoutNode->AddAttribute(LO_ATT_HEIGHT, wxString::Format(wxT("%d"), this->height));
	}

	return layoutNode;
}

void GNC::GCS::Layout::doLayout(IHangingProtocolContract* pContract, wxSizer* parentSizer)
{
	if (isACell()) {
		//create cell...
		long pk = -1;
		if (seriesMatcher.IsValid()){
			const std::list<long>& seriesId = pContract->getOpeningSeries();
			for (std::list<long>::const_iterator it = seriesId.begin(); it != seriesId.end(); ++it) {
				if (seriesMatcher->MatchesSeries((*it))) {
					pk = (*it);
					break;
				}
			}
		}
		//
		parentSizer->Add(pContract->createSeriesSlot(this->properties, pk, seriesMatcher), 1, wxEXPAND|wxALL);
	} else {
		//parent sizer has to have vertical layout...
		wxBoxSizer* pParentBox = dynamic_cast<wxBoxSizer*>(parentSizer);
		if (pParentBox == NULL || pParentBox->GetOrientation() != wxVERTICAL) {
			wxBoxSizer* verticalSizer = new wxBoxSizer(wxVERTICAL);
			parentSizer->Add(verticalSizer, 1, wxEXPAND|wxALL);
			parentSizer = verticalSizer;
		}
		//
		std::list<GNC::GCS::Ptr<Layout> >::const_iterator itLayouts = this->cellList.begin();
		for (int i = 0; i < this->rows; i++) {
			wxBoxSizer* pRow = new wxBoxSizer(wxHORIZONTAL);
			for (int j = 0; j < this->columns; j++) {
				if (itLayouts != this->cellList.end()) {
					(*itLayouts)->doLayout(pContract, pRow);
					++itLayouts;
				} else {
					pRow->Add(pContract->createSeriesSlot(this->properties, -1, seriesMatcher), 1, wxEXPAND|wxALL);
				}
			}
			parentSizer->Add(pRow, 1, wxEXPAND);
		}
	}
}

GNC::GCS::Ptr<GNC::GCS::Layout> GNC::GCS::Layout::getLayoutFromSizerItem(wxSizerItem* pSizerItem)
{
	GNC::GCS::Ptr<GNC::GCS::Layout> pLayout(new GNC::GCS::Layout());
	//it's a cell
	if (pSizerItem->GetWindow() != NULL) {
		GNC::GCS::IHangingSeriesViewer* seriesViewerMatcher = dynamic_cast<GNC::GCS::IHangingSeriesViewer*>(pSizerItem->GetWindow());
		if (seriesViewerMatcher != NULL) {
			//build series matcher...
			GIL::DICOM::DicomDataset base;
			seriesViewerMatcher->getSeriesBase(base);
			GNC::GCS::Ptr<GNC::GCS::Matcher> pMatcher(new GNC::GCS::Matcher);
			for (GIL::DICOM::ListaTags::iterator it = base.tags.begin(); it != base.tags.end(); it++) {
				pMatcher->getMatchingCriteria()[(*it).first] = (*it).second;
			}
			pLayout->setMatcher(pMatcher);
		}
	} else if (pSizerItem->GetSizer() != NULL) {
		//count rows and columns...
		wxBoxSizer* pBoxPpal = dynamic_cast<wxBoxSizer*>(pSizerItem->GetSizer());
		if (pBoxPpal != NULL) {
			int elements = 0;
			for (wxSizerItemList::const_iterator itItems = pBoxPpal->GetChildren().begin(); itItems != pBoxPpal->GetChildren().end(); itItems++, elements++)
			{
				pLayout->cellList.push_back(getLayoutFromSizerItem((*itItems)));
			}
			if (pBoxPpal->GetOrientation() == wxVERTICAL) {
				pLayout->setRows(elements);
			} else {
				pLayout->setColumns(elements);
			}
			//if layout is 1x1 with one child point to this child
			if (pLayout->isACell() && pLayout->cellList.size() == 1) {				
				GNC::GCS::Ptr<GNC::GCS::Layout> tmp = pLayout->cellList.front();
				pLayout = tmp;
			}
		} else {
			LOG_ERROR("Layout::getLayoutFromSizer", "there is a sizer that isn't a box sizer");
		}
	} else {
		LOG_ERROR("Layout::getLayoutFromSizer", "there is a item that isn't a winwow or a sizer");
	}
	return pLayout;
}

GNC::GCS::Ptr<GNC::GCS::Layout> GNC::GCS::Layout::getLayoutFromSizer(wxSizer* pSizer)
{
	GNC::GCS::Ptr<GNC::GCS::Layout> pLayout(new GNC::GCS::Layout());
	wxBoxSizer* pBoxPpal = dynamic_cast<wxBoxSizer*>(pSizer);
	if (pBoxPpal != NULL) {
		//count rows and columns...
		wxBoxSizer* pBoxPpal = dynamic_cast<wxBoxSizer*>(pSizer);
		if (pBoxPpal != NULL) {
			int elements = 0;
			for (wxSizerItemList::const_iterator itItems = pBoxPpal->GetChildren().begin(); itItems != pBoxPpal->GetChildren().end(); itItems++, elements++)
			{
				pLayout->cellList.push_back(getLayoutFromSizerItem((*itItems)));
			}
			if (pBoxPpal->GetOrientation() == wxVERTICAL) {
				pLayout->setRows(elements);
			} else {
				pLayout->setColumns(elements);
			}
			//if layout is 1x1 with one child point to this child
			if (pLayout->isACell() && pLayout->cellList.size() == 1) {
				GNC::GCS::Ptr<GNC::GCS::Layout> tmp = pLayout->cellList.front();
				pLayout = tmp;
			}
		} else {
			LOG_ERROR("Layout::getLayoutFromSizer", "there is a sizer that isn't a box sizer");
		}
	} else {
		LOG_ERROR("Layout::getLayoutFromSizer", "there is a sizer that isn't a box sizer");
	}
	return pLayout;
}

int GNC::GCS::Layout::getRows()
{
	return this->rows;
}

void GNC::GCS::Layout::setRows(int rows)
{
	this->rows = rows;
}

int GNC::GCS::Layout::getColumns()
{
	return this->columns;
}

void GNC::GCS::Layout::setColumns(int columns)
{
	this->columns = columns;
}

std::list<GNC::GCS::Ptr<GNC::GCS::Layout> >& GNC::GCS::Layout::getCellList()
{
	return cellList;
}

void GNC::GCS::Layout::setCellList(const std::list<GNC::GCS::Ptr<GNC::GCS::Layout> >& cellList)
{
	this->cellList = cellList;
}

const GNC::GCS::Ptr<GNC::GCS::Matcher>& GNC::GCS::Layout::getMatcher()
{
	return seriesMatcher;
}

void GNC::GCS::Layout::setMatcher(const GNC::GCS::Ptr<GNC::GCS::Matcher>& matcher)
{
	this->seriesMatcher = matcher;
}


std::map<std::string, std::string>& GNC::GCS::Layout::getProperties()
{
	return this->properties;
}

void GNC::GCS::Layout::setProperties(const std::map<std::string, std::string>& properties)
{
	this->properties = properties;
}

GNC::GCS::HangingProtocol* GNC::GCS::Layout::getParent()
{
	return pParent;
}

void GNC::GCS::Layout::setParent(GNC::GCS::HangingProtocol* pH)
{
	pParent = pH;
}

void GNC::GCS::Layout::setDefaultPosition()
{
	isDetached = false;
}

void GNC::GCS::Layout::setWindowProperties(bool isMaximized, int xPos, int yPos, int width, int height)
{
	this->isDetached = true;
	this->isMaximized = isMaximized;
	this->xPos = xPos;
	this->yPos = yPos;
	this->width = width;
	this->height = height;
}

void GNC::GCS::Layout::copyWindowProperties(const GNC::GCS::Layout& other)
{
	this->isDetached = other.isDetached;
	this->isMaximized = other.isMaximized;
	this->xPos = other.xPos;
	this->yPos = other.yPos;
	this->width = other.width;
	this->height = other.height;
}

void GNC::GCS::Layout::applyWindowProperties(wxWindow* pWindow)
{
	if (isDetached) {
		int maxX = 0, maxY = 0, minX=std::numeric_limits<int>::max(), minY=std::numeric_limits<int>::max();
		int deviceCount = wxDisplay::GetCount();
		for (int i = 0; i < deviceCount; ++i) {
			wxDisplay dsply(i);
			maxX = std::max<int>(maxX, dsply.GetClientArea().x + dsply.GetCurrentMode().GetWidth());
			maxY = std::max<int>(maxY, dsply.GetClientArea().y + dsply.GetCurrentMode().GetHeight());
			minX = std::min<int>(minX, dsply.GetClientArea().x);
			minY = std::min<int>(minY, dsply.GetClientArea().y);
		}
		wxPoint position(xPos, yPos);
		if (position.x >= (minX - pWindow->GetMinSize().x) && position.x  < maxX 
			&& position.y >= (minY  - pWindow->GetMinSize().y) && position.y < maxY) {
			pWindow->SetPosition(position);
		}
		wxSize size(width, height);
		wxTopLevelWindow* pTopLevel = dynamic_cast<wxTopLevelWindow*>(pWindow);
		if (pTopLevel != NULL) {
			pTopLevel->Maximize(isMaximized);
			if (!isMaximized) {
				pWindow->SetSize(size);
			}
		} else {
			pWindow->SetSize(size);
		}
		
	}
}

void GNC::GCS::Layout::applyWindowProperties(GNC::GCS::IVista* pView)
{
	//detach view...
	wxWindow* pWindow = pView->GetWindow();
	if(pWindow != NULL) {
		wxWindow* pParentWindow = pWindow->GetParent();
		if(pParentWindow != NULL) {
			wxFrame* pdlg = dynamic_cast<wxFrame*>(pParentWindow);
			if(pdlg != NULL) {
				//it is detached
				if (!isDetached) {
					//undetach
					GNC::Entorno::Instance()->GetVentanaPrincipal()->EncajarDesencajar(pView);
				}
			}
			else {
				//it isn't detached
				if (isDetached) {
					//detach
					GNC::Entorno::Instance()->GetVentanaPrincipal()->EncajarDesencajar(pView);
					//apply properties
					applyWindowProperties(pWindow->GetParent());
				}
			}
		}
	}
}
/////////////////////////////////////////////////////////////
GNC::GCS::WindowLevelSetting::WindowLevelSetting(): window(0), level(127), label(_Std("Default"))
{
}

GNC::GCS::WindowLevelSetting::WindowLevelSetting(double window, double level, const std::string& label)
{
	this->window = window;
	this->level = level;
	this->label = label;
}

GNC::GCS::WindowLevelSetting::WindowLevelSetting(wxXmlNode* wlNode)
{
	wlNode->GetAttribute(WL_ATT_WINDOW, wxEmptyString).ToDouble(&this->window);
	wlNode->GetAttribute(WL_ATT_LEVEL, wxEmptyString).ToDouble(&this->level);
	this->label = wlNode->GetAttribute(WL_ATT_LABEL, wxEmptyString).ToUTF8();
}

GNC::GCS::WindowLevelSetting::WindowLevelSetting(const WindowLevelSetting& other)
{
	window = other.window;
	level = other.level;
	label = other.label;
}

GNC::GCS::WindowLevelSetting::~WindowLevelSetting()
{
}

wxXmlNode* GNC::GCS::WindowLevelSetting::serialize()
{
	wxXmlNode* wlNode = new wxXmlNode(NULL, wxXML_ELEMENT_NODE, WL_ROOT);
	wlNode->AddAttribute(WL_ATT_WINDOW, wxString::Format(wxT("%lf"),this->window));
	wlNode->AddAttribute(WL_ATT_LEVEL, wxString::Format(wxT("%lf"),this->level));
	wlNode->AddAttribute(WL_ATT_LABEL, wxString::FromUTF8(this->label.c_str()));	

	return wlNode;
}

double GNC::GCS::WindowLevelSetting::getWindow() const
{
	return window;
}

double GNC::GCS::WindowLevelSetting::getLevel() const
{
	return level;
}

const std::string& GNC::GCS::WindowLevelSetting::getLabel() const
{
	return label;
}

std::string GNC::GCS::WindowLevelSetting::toString() const
{
	std::ostringstream ostr;
	ostr << std::setiosflags(std::ios::fixed) << std::setprecision(2);
	ostr << "(";
	ostr << window;
	ostr << "/";
	ostr << level;
	ostr << ")";
	ostr << label;
	return ostr.str();
}

void GNC::GCS::WindowLevelSetting::setWindow(double window)
{
	this->window = window;
}

void GNC::GCS::WindowLevelSetting::setLevel(double level)
{
	this->level = level;
}

void GNC::GCS::WindowLevelSetting::setLabel(const std::string& label)
{
	this->label = label;
}
/////////////////////////////////////////////////////
GNC::GCS::DefaultModalitySettings::DefaultModalitySettings()
{
	this->interpolationMode = TIM_LINEAR;
}

GNC::GCS::DefaultModalitySettings::DefaultModalitySettings(wxXmlNode* defaultSettingsNode)
{
	this->modality = defaultSettingsNode->GetAttribute(DMS_ATT_MODALITY, wxEmptyString).ToUTF8();
	this->initialWindowLevelLabel = defaultSettingsNode->GetAttribute(DMS_ATT_INITIAL_WL, wxEmptyString).ToUTF8();
	wxString strInterpolation = defaultSettingsNode->GetAttribute(DMS_ATT_INTERPOLATION, wxEmptyString);
	if (strInterpolation.CompareTo(DMS_ATT_INT_NEAREST) == 0) {
		interpolationMode = TIM_NEAREST_NEIGHBOUR;
	} else if (strInterpolation.CompareTo(DMS_ATT_LINEAR) == 0) {
		interpolationMode = TIM_LINEAR;
	} else if (strInterpolation.CompareTo(DMS_ATT_CUBIC) == 0) {
		interpolationMode = TIM_CUBIC;
	}
	
	for (wxXmlNode* child = defaultSettingsNode->GetChildren(); child != NULL; child = child->GetNext())
	{
		if (child->GetName().CmpNoCase(WLS_ROOT) == 0) {
			//layouts...
			for (wxXmlNode* wlChild = child->GetChildren(); wlChild != NULL; wlChild = wlChild->GetNext()) {
				if (wlChild->GetName().CmpNoCase(WL_ROOT) == 0) {
					GNC::GCS::Ptr<WindowLevelSetting> pWL(new WindowLevelSetting(wlChild));
					this->addWindowLevel(pWL);
				}
			}			
		}  else {
			LOG_ERROR("HangingProtocol", "error deserializing DefaultModalitySettings, unknown node " << child->GetName().ToUTF8());
		}
	}
}

GNC::GCS::DefaultModalitySettings::DefaultModalitySettings(const DefaultModalitySettings& other)
{
	this->modality = other.modality;
	this->initialWindowLevelLabel = other.initialWindowLevelLabel;
	this->interpolationMode = other.interpolationMode;
	for (GNC::GCS::DefaultModalitySettings::TListOfWLSettings::const_iterator it = other.listOfWLSettings.begin(); it != other.listOfWLSettings.end(); ++it) {
		if ((*it).IsValid()) {
			GNC::GCS::Ptr<GNC::GCS::WindowLevelSetting> settings(new GNC::GCS::WindowLevelSetting(*(*it)));
			this->listOfWLSettings.push_back(settings);
		}
	}
}

GNC::GCS::DefaultModalitySettings::~DefaultModalitySettings()
{
}

wxXmlNode* GNC::GCS::DefaultModalitySettings::serialize()
{
	wxXmlNode* defaultSettingsNode = new wxXmlNode(NULL, wxXML_ELEMENT_NODE, DMS_ROOT);
	defaultSettingsNode->AddAttribute(DMS_ATT_MODALITY, wxString::FromUTF8(this->modality.c_str()));
	defaultSettingsNode->AddAttribute(DMS_ATT_INITIAL_WL, wxString::FromUTF8(this->initialWindowLevelLabel.c_str()));
	wxString strInterpolation;
	switch (interpolationMode) {
	case TIM_NEAREST_NEIGHBOUR:
		strInterpolation = DMS_ATT_INT_NEAREST;
		break;
	case TIM_LINEAR:
		strInterpolation = DMS_ATT_LINEAR;
		break;
	case TIM_CUBIC:
		strInterpolation = DMS_ATT_CUBIC;
		break;
	}
	defaultSettingsNode->AddAttribute(DMS_ATT_INTERPOLATION, strInterpolation );

	wxXmlNode* wlSettingsNode = new wxXmlNode(defaultSettingsNode, wxXML_ELEMENT_NODE, WLS_ROOT);
	for (GNC::GCS::DefaultModalitySettings::TListOfWLSettings::const_iterator it = listOfWLSettings.begin(); it != listOfWLSettings.end(); ++it) {
		if ((*it).IsValid()) {
			wlSettingsNode->AddChild((*it)->serialize());
		}
	} 

	return defaultSettingsNode;
}

std::string GNC::GCS::DefaultModalitySettings::getDefaultWindowLevelLabel()
{
	return _Std("Default");
}

std::string GNC::GCS::DefaultModalitySettings::getAutocalculateWindowLevelLabel()
{
	return _Std("Autocalculate");
}

const GNC::GCS::DefaultModalitySettings::TListOfWLSettings& GNC::GCS::DefaultModalitySettings::getWLSettings() const
{
	return listOfWLSettings;
}

const std::string& GNC::GCS::DefaultModalitySettings::getModality() const
{
	return modality;
}

const std::string& GNC::GCS::DefaultModalitySettings::getInitialWindowLevelLabel() const
{
	return initialWindowLevelLabel;
}

GNC::GCS::DefaultModalitySettings::TInterpolationMode GNC::GCS::DefaultModalitySettings::getInterpolationMode() const
{
	return interpolationMode;
}

void GNC::GCS::DefaultModalitySettings::addWindowLevel(const GNC::GCS::Ptr<WindowLevelSetting>& wlSetting)
{
	if (wlSetting.IsValid()) {
		listOfWLSettings.push_back(wlSetting);
	}
}

void GNC::GCS::DefaultModalitySettings::clearWindowLevels()
{
	listOfWLSettings.clear();
}

void GNC::GCS::DefaultModalitySettings::setModality(const std::string& modality)
{
	this->modality = modality;
}

void GNC::GCS::DefaultModalitySettings::setInitialWindowLevelLabel(const std::string& label)
{
	this->initialWindowLevelLabel = label;
}

void GNC::GCS::DefaultModalitySettings::setInterpolationMode(TInterpolationMode mode)
{
	this->interpolationMode = mode;
}


/////////////////////////////////////////////////////
GNC::GCS::HangingProtocol::HangingProtocol():active(true)
{
	LoadDefaultModalitySettings();
}

GNC::GCS::HangingProtocol::HangingProtocol(wxXmlNode* hangingNode)
{
	this->name = hangingNode->GetAttribute(HP_ATT_NAME, wxEmptyString).ToUTF8();
	this->description = hangingNode->GetAttribute(HP_ATT_DESCRIPTION, wxEmptyString).ToUTF8();
	this->moduleSid = hangingNode->GetAttribute(HP_ATT_MODULE_SID, wxEmptyString).ToUTF8();
	this->active = hangingNode->GetAttribute(HP_ATT_ACTIVE, wxT("true")).CmpNoCase(wxT("true")) == 0;
	long tmpLong;
	if(hangingNode->GetAttribute(HP_ATT_MODE_ID, wxEmptyString).ToLong(&tmpLong)) {
		this->openingMode = tmpLong;
	} else {
		this->openingMode = 0;
	}
	for (wxXmlNode* child = hangingNode->GetChildren(); child != NULL; child = child->GetNext())
	{
		if (child->GetName().CmpNoCase(LOS_ROOT) == 0) {
			//layouts...
			for (wxXmlNode* layoutChild = child->GetChildren(); layoutChild != NULL; layoutChild = layoutChild->GetNext()) {
				if (layoutChild->GetName().CmpNoCase(LO_ROOT) == 0) {
					GNC::GCS::Ptr<Layout> pLayout(new Layout(layoutChild));
					this->addLayout(pLayout);
				}
			}			
		} else if (child->GetName().CmpNoCase(LO_ROOT) == 0)  {
			//retrocompatib...
			GNC::GCS::Ptr<Layout> pLayout(new Layout(child));
			this->addLayout(pLayout);
		} else if (child->GetName().CmpNoCase(ML_ROOT) == 0)  {
			for (wxXmlNode* modChild = child->GetChildren(); modChild != NULL; modChild = modChild->GetNext()) {
				if (modChild->GetName().CmpNoCase(DMS_ROOT) == 0) {
					GNC::GCS::Ptr<DefaultModalitySettings> modSettings(new DefaultModalitySettings(modChild));
					this->addModalitySettings(modSettings);
				}
			}
		} else if (child->GetName().CmpNoCase(MC_ROOT) == 0) {
			this->studyMatcher = new Matcher(child);
		} else if (child->GetName().CmpNoCase(PROPERTY_NODE) == 0) {
			const wxString key = child->GetAttribute(PROPERTY_KEY, wxEmptyString);
			const wxString value = child->GetAttribute(PROPERTY_VALUE, wxEmptyString);
			if (!key.IsEmpty()) {
				this->properties[std::string(key.ToUTF8())] = value.ToUTF8();
			}			
		} else {
			LOG_ERROR("HangingProtocol", "error deserializing hanging protocol, unknown node " << child->GetName().ToUTF8());
		}
	}
	//retrocompatibility
	LoadDefaultModalitySettings();
}

GNC::GCS::HangingProtocol::HangingProtocol(const GNC::GCS::HangingProtocol& other)
{
	this->name = other.name;
	this->description = other.description;
	this->moduleSid = other.moduleSid;
	this->openingMode = other.openingMode;
	if (other.studyMatcher.IsValid()) {
		this->studyMatcher = new Matcher(*other.studyMatcher);
	}
	for (GNC::GCS::HangingProtocol::TListOfLayouts::const_iterator it = other.layouts.begin(); it != other.layouts.end(); ++it) {
		if ((*it).IsValid()) {
			GNC::GCS::Ptr<GNC::GCS::Layout> layout(new Layout(*(*it)));
			this->layouts.push_back(layout);
		}
	}
	for (GNC::GCS::HangingProtocol::TMapOfModalitySettings::const_iterator it = other.modalitySettings.begin(); it != other.modalitySettings.end(); ++it) {
		if ((*it).second.IsValid()) {
			GNC::GCS::Ptr<GNC::GCS::DefaultModalitySettings> settings(new GNC::GCS::DefaultModalitySettings(*((*it).second)));
			this->modalitySettings[(*it).first] = settings;
		}
	}
	this->properties = other.properties;
	this->active = other.active;
}

GNC::GCS::HangingProtocol::~HangingProtocol()
{
	active = true;
}

void GNC::GCS::HangingProtocol::LoadDefaultModalitySettings()
{
	//basic modality settings....
	if (modalitySettings.find("CT") == modalitySettings.end()) 
	{
		GNC::GCS::Ptr<GNC::GCS::DefaultModalitySettings> modSettings(new GNC::GCS::DefaultModalitySettings());
		modSettings->setModality("CT");
		{
			GNC::GCS::Ptr<GNC::GCS::WindowLevelSetting> wlSettings(new GNC::GCS::WindowLevelSetting(300,35,_Std("SPINAL")));
			modSettings->addWindowLevel(wlSettings);
		}
		{
			GNC::GCS::Ptr<GNC::GCS::WindowLevelSetting> wlSettings(new GNC::GCS::WindowLevelSetting(100,50,_Std("CRANIUM")));
			modSettings->addWindowLevel(wlSettings);
		}
		{
			GNC::GCS::Ptr<GNC::GCS::WindowLevelSetting> wlSettings(new GNC::GCS::WindowLevelSetting(1800,400,_Std("BONE")));
			modSettings->addWindowLevel(wlSettings);
		}
		{
			GNC::GCS::Ptr<GNC::GCS::WindowLevelSetting> wlSettings(new GNC::GCS::WindowLevelSetting(3600,1000,_Std("CRANIUM BONE")));
			modSettings->addWindowLevel(wlSettings);
		}
		{
			GNC::GCS::Ptr<GNC::GCS::WindowLevelSetting> wlSettings(new GNC::GCS::WindowLevelSetting(400,40,_Std("MEDIASTINUM")));
			modSettings->addWindowLevel(wlSettings);
		}
		{
			GNC::GCS::Ptr<GNC::GCS::WindowLevelSetting> wlSettings(new GNC::GCS::WindowLevelSetting(1500,-500,_Std("LUNG")));
			modSettings->addWindowLevel(wlSettings);
		}
		{
			GNC::GCS::Ptr<GNC::GCS::WindowLevelSetting> wlSettings(new GNC::GCS::WindowLevelSetting(2200,200,_Std("BREAST")));
			modSettings->addWindowLevel(wlSettings);
		}
		{
			GNC::GCS::Ptr<GNC::GCS::WindowLevelSetting> wlSettings(new GNC::GCS::WindowLevelSetting(2000,350,_Std("VERTEBRA")));
			modSettings->addWindowLevel(wlSettings);
		}
		addModalitySettings(modSettings);
	}
	if (modalitySettings.find("MR") == modalitySettings.end()) 
	{
		GNC::GCS::Ptr<GNC::GCS::DefaultModalitySettings> modSettings(new GNC::GCS::DefaultModalitySettings());
		modSettings->setModality("MR");
		{
			GNC::GCS::Ptr<GNC::GCS::WindowLevelSetting> wlSettings(new GNC::GCS::WindowLevelSetting(1600,678,_Std("ANGIO")));
			modSettings->addWindowLevel(wlSettings);
		}
		{
			GNC::GCS::Ptr<GNC::GCS::WindowLevelSetting> wlSettings(new GNC::GCS::WindowLevelSetting(150,500,_Std("CT CRANIUM")));
			modSettings->addWindowLevel(wlSettings);
		}
		{
			GNC::GCS::Ptr<GNC::GCS::WindowLevelSetting> wlSettings(new GNC::GCS::WindowLevelSetting(750,400,_Std("T1 CRANIUM")));
			modSettings->addWindowLevel(wlSettings);
		}
		{
			GNC::GCS::Ptr<GNC::GCS::WindowLevelSetting> wlSettings(new GNC::GCS::WindowLevelSetting(563,33,_Std("T2 CRANIUM")));
			modSettings->addWindowLevel(wlSettings);
		}
		{
			GNC::GCS::Ptr<GNC::GCS::WindowLevelSetting> wlSettings(new GNC::GCS::WindowLevelSetting(2030,1245,_Std("KNEE")));
			modSettings->addWindowLevel(wlSettings);
		}
		addModalitySettings(modSettings);
	}
	
	if (modalitySettings.find("CR") == modalitySettings.end()) 
	{
		GNC::GCS::Ptr<GNC::GCS::DefaultModalitySettings> modSettings(new GNC::GCS::DefaultModalitySettings());
		modSettings->setModality("CR");
		addModalitySettings(modSettings);
	}
	
	if (modalitySettings.find("US") == modalitySettings.end()) 
	{
		GNC::GCS::Ptr<GNC::GCS::DefaultModalitySettings> modSettings(new GNC::GCS::DefaultModalitySettings());
		modSettings->setModality("US");
		addModalitySettings(modSettings);
	}
	
	if (modalitySettings.find("MG") == modalitySettings.end()) 
	{
		GNC::GCS::Ptr<GNC::GCS::DefaultModalitySettings> modSettings(new GNC::GCS::DefaultModalitySettings());
		modSettings->setModality("MG");
		addModalitySettings(modSettings);
	}
	
	if (modalitySettings.find("NM") == modalitySettings.end()) 
	{
		GNC::GCS::Ptr<GNC::GCS::DefaultModalitySettings> modSettings(new GNC::GCS::DefaultModalitySettings());
		modSettings->setModality("NM");
		addModalitySettings(modSettings);
	}
	
	if (modalitySettings.find("RF") == modalitySettings.end()) 
	{
		GNC::GCS::Ptr<GNC::GCS::DefaultModalitySettings> modSettings(new GNC::GCS::DefaultModalitySettings());
		modSettings->setModality("RF");
		addModalitySettings(modSettings);
	}
	
	if (modalitySettings.find("SC") == modalitySettings.end()) 
	{
		GNC::GCS::Ptr<GNC::GCS::DefaultModalitySettings> modSettings(new GNC::GCS::DefaultModalitySettings());
		modSettings->setModality("SC");
		addModalitySettings(modSettings);
	}
		
	if (modalitySettings.find("XA") == modalitySettings.end()) 
	{
		GNC::GCS::Ptr<GNC::GCS::DefaultModalitySettings> modSettings(new GNC::GCS::DefaultModalitySettings());
		modSettings->setModality("XA");
		addModalitySettings(modSettings);
	}
		
	if (modalitySettings.find("OT") == modalitySettings.end()) 
	{
		GNC::GCS::Ptr<GNC::GCS::DefaultModalitySettings> modSettings(new GNC::GCS::DefaultModalitySettings());
		modSettings->setModality("OT");
		addModalitySettings(modSettings);
	}
		
	if (modalitySettings.find("PT") == modalitySettings.end()) 
	{
		GNC::GCS::Ptr<GNC::GCS::DefaultModalitySettings> modSettings(new GNC::GCS::DefaultModalitySettings());
		modSettings->setModality("PT");
		addModalitySettings(modSettings);
	}

}

wxXmlNode* GNC::GCS::HangingProtocol::serialize()
{
	wxXmlNode* hangingNode = new wxXmlNode(NULL, wxXML_ELEMENT_NODE, HP_ROOT);
	hangingNode->AddAttribute(HP_ATT_NAME, wxString::FromUTF8(this->name.c_str()));
	hangingNode->AddAttribute(HP_ATT_DESCRIPTION, wxString::FromUTF8(this->description.c_str()));
	hangingNode->AddAttribute(HP_ATT_MODULE_SID, wxString::FromUTF8(this->moduleSid.c_str()));
	hangingNode->AddAttribute(HP_ATT_MODE_ID, wxString::Format(wxT("%d"), this->openingMode));
	hangingNode->AddAttribute(HP_ATT_ACTIVE, active?wxT("true"):wxT("false"));

	wxXmlNode* layoutsNode = new wxXmlNode(hangingNode, wxXML_ELEMENT_NODE, LOS_ROOT);
	for (GNC::GCS::HangingProtocol::TListOfLayouts::const_iterator it = layouts.begin(); it != layouts.end(); ++it) {
		if ((*it).IsValid()) {
			layoutsNode->AddChild((*it)->serialize());
		}
	} 
	wxXmlNode* modalitySettingsNode = new wxXmlNode(hangingNode, wxXML_ELEMENT_NODE, ML_ROOT);
	for (GNC::GCS::HangingProtocol::TMapOfModalitySettings::const_iterator it = modalitySettings.begin(); it != modalitySettings.end(); ++it) {
		if ((*it).second.IsValid()) {
			modalitySettingsNode->AddChild((*it).second->serialize());
		}
	} 
	if (this->studyMatcher.IsValid()) {
		hangingNode->AddChild(this->studyMatcher->serialize());
	}
	
	for (std::map<std::string, std::string>::const_iterator it = this->properties.begin(); it != this->properties.end(); ++it)
	{
		wxXmlNode* propertyNode = new wxXmlNode(hangingNode, wxXML_ELEMENT_NODE, PROPERTY_NODE);
		propertyNode->AddAttribute(PROPERTY_KEY, wxString::FromUTF8((*it).first.c_str()));
		propertyNode->AddAttribute(PROPERTY_VALUE, wxString::FromUTF8((*it).second.c_str()));
	}

	return hangingNode;
}

bool GNC::GCS::HangingProtocol::matches(long idStudy)
{
	if (this->studyMatcher.IsValid()) {
		return this->studyMatcher->MatchesStudy(idStudy);
	}
	LOG_WARN("HangingProtocol", "there is a hanging protocol without matcher specified");
	return false;
}


const std::string& GNC::GCS::HangingProtocol::getName()
{
	return this->name;
}

void GNC::GCS::HangingProtocol::setName(const std::string& name)
{
	this->name = name;
}

const std::string& GNC::GCS::HangingProtocol::getDescription()
{
	return this->description;
}

void GNC::GCS::HangingProtocol::setDescription(const std::string& description)
{
	this->description = description;
}

const std::string& GNC::GCS::HangingProtocol::getModuleSid()
{
	return this->moduleSid;
}

void GNC::GCS::HangingProtocol::setModuleSid(const std::string& sid)
{
	this->moduleSid = sid;
}

int GNC::GCS::HangingProtocol::getOpeningMode()
{
	return this->openingMode;
}

void GNC::GCS::HangingProtocol::setOpeningMode(int mode)
{
	this->openingMode = mode;
}

const GNC::GCS::Ptr<GNC::GCS::Matcher>& GNC::GCS::HangingProtocol::getStudyMatcher()
{
	return this->studyMatcher;
}

void GNC::GCS::HangingProtocol::setStudyMatcher(const GNC::GCS::Ptr<GNC::GCS::Matcher>& matcher)
{
	this->studyMatcher = matcher;
}

const GNC::GCS::HangingProtocol::TListOfLayouts& GNC::GCS::HangingProtocol::getLayouts()
{
	return this->layouts;
}

void GNC::GCS::HangingProtocol::addLayout(const GNC::GCS::Ptr<GNC::GCS::Layout>& layout)
{
	if (layout.IsValid()) {
		layout->setParent(this);
		this->layouts.push_back(layout);
	}
}

void GNC::GCS::HangingProtocol::clearLayouts()
{
	this->layouts.clear();
}

const GNC::GCS::HangingProtocol::TMapOfModalitySettings& GNC::GCS::HangingProtocol::getModalitySettingsMap()
{
	return this->modalitySettings;
}

void GNC::GCS::HangingProtocol::addModalitySettings(const GNC::GCS::Ptr<DefaultModalitySettings>& settings)
{
	if (settings.IsValid()) {
		this->modalitySettings[settings->getModality()] = settings;
	}
}

void GNC::GCS::HangingProtocol::clearModalitySettings()
{
	this->modalitySettings.clear();
}

GNC::GCS::Ptr<GNC::GCS::DefaultModalitySettings> GNC::GCS::HangingProtocol::getModalitySettings(const std::string& modality)
{
	GNC::GCS::Ptr<GNC::GCS::DefaultModalitySettings> ptr;
	if (modalitySettings.find(modality) != modalitySettings.end()) {
		ptr = modalitySettings[modality];
	}
	return ptr;
}

std::map<std::string, std::string>& GNC::GCS::HangingProtocol::getProperties()
{
	return this->properties;
}

void GNC::GCS::HangingProtocol::setProperties(const std::map<std::string, std::string>& properties)
{
	this->properties = properties;
}

bool GNC::GCS::HangingProtocol::isActive()
{
	return this->active;
}

void GNC::GCS::HangingProtocol::setActive(bool active)
{
	this->active = active;
}

/**
Controller for hanging protocols
*/
GNC::GCS::HangingProtocolController* GNC::GCS::HangingProtocolController::Instance()
{
	if (m_pInstance == NULL) {
		m_pInstance = new HangingProtocolController();
	}
	return m_pInstance;
}

void GNC::GCS::HangingProtocolController::FreeInstance()
{
	if (m_pInstance != NULL) {
		delete m_pInstance;
		m_pInstance = NULL;
	}
}

GNC::GCS::HangingProtocolController::HangingProtocolController()
{
	LoadFromConfiguration();
}

GNC::GCS::HangingProtocolController::~HangingProtocolController()
{
}
GNC::GCS::HangingProtocolController* GNC::GCS::HangingProtocolController::m_pInstance = NULL;

GNC::GCS::Ptr<GNC::GCS::HangingProtocol> GNC::GCS::HangingProtocolController::GetProtocolForStudy(long idStudy)
{
	for (std::list<GNC::GCS::Ptr<HangingProtocol> >::const_iterator it = this->listOfProtocols.begin(); it != this->listOfProtocols.end(); ++it)
	{
		if ((*it)->isActive() && (*it)->matches(idStudy)) {
			return (*it);
		}
	}
	//study doesn't match we create a hanging protocol as traditional ginkgo opening mode...

	return NULL;
}

void GNC::GCS::HangingProtocolController::StoreHangingProtocolConfiguration()
{
	//serialization...
	wxXmlDocument doc;
	wxXmlNode* root = new wxXmlNode(NULL, wxXML_ELEMENT_NODE,HP_DOCUMENT_ROOT);
	doc.SetRoot(root);
	for (std::list<GNC::GCS::Ptr<HangingProtocol> >::const_iterator it = this->listOfProtocols.begin(); it != this->listOfProtocols.end(); ++it)
	{
		root->AddChild((*it)->serialize());
	}
	wxMemoryOutputStream out;
	doc.SetFileEncoding(wxT("UTF-8"));
	doc.Save(out,0);
	char* vectorWidgets = new char[out.GetLength()];
	out.CopyTo(vectorWidgets,out.GetLength());
	GNC::GCS::ConfigurationController::Instance()->writeStringUser("/GinkgoCore/HangingProtocols","HangingList", std::string(vectorWidgets, out.GetLength()));
	delete[] vectorWidgets;
}

std::list<GNC::GCS::Ptr<GNC::GCS::HangingProtocol> > GNC::GCS::HangingProtocolController::getHangingProtocols()
{	
	LoadFromConfiguration();
	std::list<GNC::GCS::Ptr<GNC::GCS::HangingProtocol> >  returningList;

	for (std::list<GNC::GCS::Ptr<HangingProtocol> >::const_iterator it = this->listOfProtocols.begin(); it != this->listOfProtocols.end(); ++it)
	{
		GNC::GCS::Ptr<HangingProtocol> pProtocol(new HangingProtocol(*(*it)));
		returningList.push_back(pProtocol);
	}

	return returningList;
}
			
void GNC::GCS::HangingProtocolController::setHangingProtocols(std::list<GNC::GCS::Ptr<GNC::GCS::HangingProtocol> > listOfProtocols)
{
	this->listOfProtocols = listOfProtocols;
	StoreHangingProtocolConfiguration();
}

void GNC::GCS::HangingProtocolController::LoadFromConfiguration()
{
	//Deserialize...
	this->listOfProtocols.clear();
	std::string serializedValue;
	GNC::GCS::ConfigurationController::Instance()->readStringUser("/GinkgoCore/HangingProtocols","HangingList", serializedValue);
	if (!serializedValue.empty()) {
		wxString wxSerializedValue = wxString::FromUTF8(serializedValue.c_str());
		wxStringInputStream istream(wxSerializedValue);
		wxXmlDocument doc;
		if (doc.Load(istream)) {
			wxXmlNode* root = doc.GetRoot();
			if (root != NULL && root->GetName().CmpNoCase(HP_DOCUMENT_ROOT) == 0) {
				for (wxXmlNode* hangingProtocolNode = root->GetChildren(); hangingProtocolNode != NULL; hangingProtocolNode = hangingProtocolNode->GetNext())
				{
					if (hangingProtocolNode->GetName().CmpNoCase(HP_ROOT) == 0) {
						GNC::GCS::Ptr<HangingProtocol> hangingProtocol(new HangingProtocol(hangingProtocolNode));
						this->listOfProtocols.push_back(hangingProtocol);
					}
				}
			}
		} else {
			LOG_ERROR("HangingProtocolController", "Error deserializing hanging protocol");
		}
	}
}




