/*
 *  
 *  $Id: vtkginkgoimageviewer.cpp 4090 2011-08-25 12:17:45Z tovar $
 *  Ginkgo CADx Project
 *
 *  Copyright 2008-10 MetaEmotion S.L. All rights reserved.
 *  http://ginkgo-cadx.com
 *
 *  This file is licensed under LGPL v3 license.
 *  See License.txt for details
 *
 *  Code based in VTK/VTKInria3D
 */
//#define _GINKGO_TRACE
#include <limits>
#include <sstream>
#include <cmath>

#include <wx/wxprec.h>

#include <api/globals.h>
#include <api/icontroladorcarga.h>
#include <api/math/geometria.h>
#include <api/math/geometria3d.h>

#include <main/controllers/controladorlog.h>

#ifdef __DEPRECATED
#undef __DEPRECATED
#endif
#include "vtkginkgoimageviewer.h"
#include "vtkopenglginkgotexture.h"
#include "interactor/ginkgointeractorstyleimage2d.h"
#include "command/ginkgoimagecommand.h"

#include <vtkRenderer.h>
#include <vtkRenderWindow.h>
#include <vtkRenderWindowInteractor.h>
#include <vtkInteractorStyle.h>

#include <vtkObjectFactory.h>
#include <vtkInformation.h>
#include <vtkAlgorithmOutput.h>
#include <vtkImageData.h>
#include <vtkImageReslice.h>
#include <vtkActor.h>
#include <vtkProperty.h>

#include <vtkImageMapToColors.h>
#include <vtkScalarsToColors.h>
#include <vtkImageMapToWindowLevelColors.h>
#include <vtkActor.h>
#include <vtkImageActor.h>
#include <vtkAlgorithm.h>
#include <vtkInformation.h>
#include <vtkDebugLeaks.h>
#include <vtkCamera.h>
#include <vtkProperty.h>
#include <vtkPlaneSource.h>
#include <vtkPlane.h>
#include <vtkPolyDataMapper.h>

#include <VTKInria3D/vtkVISUManagement/vtkLookupTableManager.h>

#include <vtkAlgorithmOutput.h>
#include <vtkDataSetAttributes.h>
#include <vtkExecutive.h>
#include <vtkInformation.h>
#include <vtkInformationVector.h>
#include <vtkStreamingDemandDrivenPipeline.h>
#include <vtkImageDataStreamer.h>
#include <vtkOpenGLTexture.h>
#include <vtkMatrix3x3.h>
#include <vtkMatrix4x4.h>
#include <vtkPointData.h>

#include <hackimageactor.h>

#define VTK_GINKGO_ZOOM_MIN 0.5f
#define VTK_GINKGO_ZOOM_MAX 200

extern int vtkrint(double a);


class vtkGinkgoTexture : public vtkOpenGLTexture {
	protected:
		vtkGinkgoTexture() {}

		~vtkGinkgoTexture() {}
	public:

		static vtkGinkgoTexture *New();

		vtkTypeMacro(vtkGinkgoTexture,vtkOpenGLTexture);

		long GetTextureId() const
		{
			return Index;
		}

		//indica si la textura tiene id (se ha intentado cargar)
		bool TextureDefined() const
		{
			return Index != 0;
		}

		operator long () const
		{
			return Index;
		}

	protected:
		long m_IdTextura;

		vtkRenderWindow *m_vtkRenderWindow;
};

vtkStandardNewMacro(vtkGinkgoTexture);

class Pipeline {
public:
	vtkSmartPointer<vtkAlgorithmOutput>             InputConnection;
	vtkSmartPointer<vtkRenderer>                    Renderer;
	vtkSmartPointer<vtkRenderWindow>                RenderWindow;
	vtkSmartPointer<vtkRenderWindowInteractor>      RenderWindowInteractor;

	vtkSmartPointer<vtkInteractorStyle>             InteractorStyle;

	vtkSmartPointer<vtkImageData>                   ImageData;
	vtkSmartPointer<vtkImageData>                   OverlayData;
	vtkSmartPointer<vtkActor>                       ImageActor;
	vtkSmartPointer<vtkActor>                       OverlayActor;
	vtkSmartPointer<vtkPlaneSource>                 PlaneSource;
	vtkSmartPointer<vtkProperty>                    PlaneProperty;
	vtkSmartPointer<vtkPlane>                       Plane;
	vtkSmartPointer<vtkGinkgoOpenGLTexture>         Textura;
	vtkSmartPointer<vtkPolyDataMapper>              MapperPlano;
	vtkSmartPointer<vtkGinkgoTexture>               TexturaOverlay;
	vtkSmartPointer<vtkPolyDataMapper>              MapperPlanoOverlay;

	vtkSmartPointer<vtkImageMapToWindowLevelColors> WindowLevel;

	vtkSmartPointer<vtkImageDataStreamer>           StreamConnector;

	vtkSmartPointer<vtkScalarsToColors>             LookupTable;
	int															LookupTableId;

	vtkSmartPointer<vtkMatrix4x4> MatrizModelo;
	vtkSmartPointer<vtkMatrix4x4> MatrizModeloInv;

	Pipeline()
	{
		ImageActor      = vtkSmartPointer<vtkActor>::New();
		OverlayActor    = vtkSmartPointer<vtkActor>::New();
		PlaneProperty   = vtkSmartPointer<vtkProperty>::New();

		MatrizModelo    = vtkSmartPointer<vtkMatrix4x4>::New();
		MatrizModeloInv = vtkSmartPointer<vtkMatrix4x4>::New();

		PlaneProperty->SetAmbient(1.0);
		PlaneProperty->SetAmbientColor(1.0,1.0,1.0);
		PlaneProperty->SetOpacity(1);

		PlaneSource = vtkSmartPointer<vtkPlaneSource>::New();
		PlaneSource->SetXResolution(1);
		PlaneSource->SetYResolution(1);

		Textura            = vtkSmartPointer<vtkGinkgoOpenGLTexture>::New();
		TexturaOverlay     = vtkSmartPointer<vtkGinkgoTexture>::New();
		MapperPlano        = vtkSmartPointer<vtkPolyDataMapper>::New();
		MapperPlanoOverlay = vtkSmartPointer<vtkPolyDataMapper>::New();
		Plane              = vtkSmartPointer<vtkPlane>::New();

		double bounds[6];
		bounds[0] = -0.5;
		bounds[1] =  0.5;
		bounds[2] = -0.5;
		bounds[3] =  0.5;
		bounds[4] = -0.5;
		bounds[5] =  0.5;

		double center[3];
		center[0] = (bounds[0] + bounds[1])/2.0;
		center[1] = (bounds[2] + bounds[3])/2.0;
		center[2] = (bounds[4] + bounds[5])/2.0;

		PlaneSource->SetOrigin(center[0],bounds[2],bounds[4]);
		PlaneSource->SetPoint1(center[0],bounds[3],bounds[4]);
		PlaneSource->SetPoint2(center[0],bounds[2],bounds[5]);

		Textura->SetPremultipliedAlpha(1.0);
		Textura->SetInterpolate(1);
		Textura->RepeatOff();
		TexturaOverlay->SetInterpolate(1);
		TexturaOverlay->RepeatOff();
		TexturaOverlay->MapColorScalarsThroughLookupTableOn();
		vtkLookupTable* tblover = vtkLookupTableManager::GetOverlayLooupTable();
		TexturaOverlay->SetLookupTable(tblover);
		TexturaOverlay->GetLookupTable()->SetRange(0, 1);
		tblover->Delete();

		MapperPlano->SetInput(vtkPolyData::SafeDownCast(PlaneSource->GetOutput()));
		MapperPlanoOverlay->SetInput(vtkPolyData::SafeDownCast(PlaneSource->GetOutput()));

		WindowLevel = vtkSmartPointer<vtkImageMapToWindowLevelColors>::New();
		StreamConnector = vtkSmartPointer<vtkImageDataStreamer>::New();
		StreamConnector->SetNumberOfStreamDivisions(1);
		vtkLookupTable* tbl = vtkLookupTableManager::GetLinearLookupTable();
		LookupTableId = vtkLookupTableManager::LUT_LINEAR;
		LookupTable = tbl;

		tbl->Delete();

		ImageActor->SetMapper(MapperPlano);
		ImageActor->SetTexture(Textura);
		ImageActor->PickableOff();
		ImageActor->GetProperty()->SetOpacity(1.0);
		OverlayActor->SetMapper(MapperPlanoOverlay);
		OverlayActor->SetTexture(TexturaOverlay);
		OverlayActor->PickableOff();

		ResetMatrices();

	}

	void ResetMatrices()
	{
		for (int j = 0; j < 4; j++)
		{
			for (int i = 0; i < 4; i++)
			{
				if (i == j) {
					MatrizModelo->SetElement(i, j, 1.0);
					MatrizModeloInv->SetElement(i, j, 1.0);
				}
				else {
					MatrizModelo->SetElement(i, j, 0.0);
					MatrizModeloInv->SetElement(i, j, 1.0);
				}
			}
		}
	}
};

class Interactuacion
{
public:
	int LeftButtonInteractionStyle;
	int RightButtonInteractionStyle;
	int MiddleButtonInteractionStyle;
	int WheelInteractionStyle;

	Interactuacion() {
		LeftButtonInteractionStyle = RightButtonInteractionStyle = MiddleButtonInteractionStyle = WheelInteractionStyle = vtkGinkgoImageViewer::ZOOM_WITH_SELECT_INTERACTION;
	}
};

class Comportamiento {
public:
	int  InteractorStyleType;
	bool ShowAnnotations;
	bool ShowDirections;

	::Interactuacion Interactuacion;

	Comportamiento()
	{
		InteractorStyleType = vtkGinkgoImageViewer::ZOOM_INTERACTION;
		ShowAnnotations     = false;
		ShowDirections      = false;
	}

};

class Propiedades {
public:
	unsigned int Orientacion;
	unsigned int Conventions;
	double       DirectionCosines[9];
	double       CameraFocalPoint[4];
	double       CameraPosition[4];
	double       CameraViewUp[4];
	double       ParallelScale;
	bool         Interaction;
	bool         LinkRender;
	bool         LinkCameraFocalAndPosition;
	bool         LinkZoom;

	bool         Modificado;



	Propiedades()
	{
		Orientacion = vtkGinkgoImageViewer::AXIAL_ID;
		Conventions = vtkGinkgoImageViewer::RADIOLOGIC;
		DirectionCosines[0] = DirectionCosines[1] = DirectionCosines[2] = DirectionCosines[3] = DirectionCosines[4] = DirectionCosines[5] = DirectionCosines[6] = DirectionCosines[7] = DirectionCosines[8] = 0.0;
		CameraFocalPoint[0] = CameraFocalPoint[1] = CameraFocalPoint[2] = CameraFocalPoint[3] = 0.0;
		CameraPosition[0] = CameraPosition[1] = CameraPosition[2] = CameraPosition[3] = 0.0;
		CameraViewUp[0] = CameraViewUp[1] = CameraViewUp[2] = CameraViewUp[3] = 0.0;
		Interaction = false;
		LinkRender = false;
		LinkCameraFocalAndPosition = true;
		LinkZoom = true;
		Modificado = true;

	}

	void SetDirection(double dir[9])
	{
		for ( int i = 0; i < 9; i++) {
			if (DirectionCosines[i] != dir[i])
			{
				DirectionCosines[i] = dir[i];
				Modificado = true;
			}
		}
	}

	void SetConventions(unsigned int c)
	{
		if (Conventions != c)
		{
			Conventions = c;
			Modificado = false;
		}
	}

	void ResetModifiedStatus()
	{
		Modificado = false;
	}


};

class Estado {
public:

	bool    Initialized;
	bool    PipelineInstalled;
	bool    FirstRender;


	bool    DefaultWindowLevelSetted;
	double  InitialWindow;
	double  InitialLevel;

	double  RollOffset;
	bool    FlipHorizontal;
	bool    FlipVertical;

	double  InitialParallelScale;

	bool    IsProcessed;

	double  Zoom;

	unsigned int CurrentNumberOfComponents;

	Estado()
	{

		Initialized               = false;
		PipelineInstalled         = false;
		FirstRender               = true;

		DefaultWindowLevelSetted  = false;
		InitialWindow             = InitialLevel = std::numeric_limits<double>::quiet_NaN();

		InitialParallelScale      = 1.0f;

		IsProcessed               = false;

		Zoom                      = 1.0f;
		CurrentNumberOfComponents = 0;
		RollOffset                = 0.0;
		FlipHorizontal            = false;
		FlipVertical              = false;
	}
};

class InternalMembers {
public:

	::Pipeline Pipeline;
	::Comportamiento Comportamiento;
	::Propiedades Propiedades;
	::Estado Estado;
};

vtkStandardNewMacro(vtkGinkgoImageViewer);


vtkGinkgoImageViewer::vtkGinkgoImageViewer()
{
	members = new InternalMembers();

	//Pipeline& p = members->Pipeline;


	vtkSmartPointer<GinkgoInteractorStyleImage2D> interactor = vtkSmartPointer<GinkgoInteractorStyleImage2D>::New();
	interactor->SetView(this);

	vtkSmartPointer<GinkgoImageCommand> cbk = vtkSmartPointer<GinkgoImageCommand>::New();
	cbk->SetView(this);
	interactor->AddObserver(vtkCommand::KeyPressEvent, cbk);
	/*
	interactor->AddObserver(vtkCommand::WindowLevelEvent, cbk);
	interactor->AddObserver(vtkCommand::StartWindowLevelEvent, cbk);
	interactor->AddObserver(vtkCommand::ResetWindowLevelEvent, cbk);
	interactor->AddObserver(vtkCommand::EndWindowLevelEvent, cbk);
	*/
	/*
	interactor->AddObserver(vtkCommand::PickEvent, cbk);
	interactor->AddObserver(vtkCommand::StartPickEvent, cbk);
	interactor->AddObserver(vtkCommand::EndPickEvent, cbk);
	*/
	//interactor->AddObserver(GinkgoImageCommand::ResetZoomEvent, cbk);
	//interactor->AddObserver(GinkgoImageCommand::ResetPositionEvent, cbk);
	//interactor->AddObserver(GinkgoImageCommand::StartZSliceMoveEvent, cbk);
	//interactor->AddObserver(GinkgoImageCommand::ZSliceMoveEvent, cbk);
	//interactor->AddObserver(GinkgoImageCommand::EndZSliceMoveEvent, cbk);
	//interactor->AddObserver(GinkgoImageCommand::StartMeasureEvent, cbk);
	//interactor->AddObserver(GinkgoImageCommand::MeasureEvent, cbk);
	//interactor->AddObserver(GinkgoImageCommand::EndAngleEvent, cbk);
	//interactor->AddObserver(GinkgoImageCommand::StartAngleEvent, cbk);
	//interactor->AddObserver(GinkgoImageCommand::AngleEvent, cbk);
	//interactor->AddObserver(GinkgoImageCommand::EndAngleEvent, cbk);
	//interactor->AddObserver(GinkgoImageCommand::FullPageEvent, cbk);
	interactor->AddObserver(GinkgoImageCommand::ZoomEvent, cbk);

	SetInteractorStyle(interactor);

}

vtkGinkgoImageViewer::~vtkGinkgoImageViewer()
{
	if (this->GetRenderer()) {
		this->GetRenderer()->RemoveAllViewProps();
	}
	delete members;
}


void vtkGinkgoImageViewer::PrepareForDelete(void)
{
	Uninitialize();
}

/**
This function is called right after setting both Renderer and RenderWindow.
It allows a class to add actors for instance without knowing when the Renderer
and RenderWindow are set. For example, vtkGinkgoImageViewer will add the corner annotations
during the call to the Initialize function.
*/
void vtkGinkgoImageViewer::Initialize(void)
{
	//std::cout << "vtkGinkgoImageViewer::Initialize()" << std::endl;
	Pipeline& p = members->Pipeline;
	Estado&   e = members->Estado;

	if (e.Initialized) {
		return;
	}

	if (p.Renderer) {
		p.Renderer->SetBackground(0.0f, 0.0f, 0.0f);
	}
	if (p.RenderWindow && p.Renderer) {
		p.RenderWindow->AddRenderer(members->Pipeline.Renderer);
	}

	if (p.RenderWindowInteractor && p.InteractorStyle) {
		p.RenderWindowInteractor->SetInteractorStyle(p.InteractorStyle);
		p.RenderWindowInteractor->SetRenderWindow(p.RenderWindow);
		e.Initialized = true;
		SetInteractionOn();
	}

	if (p.Renderer) {
		p.Renderer->GetActiveCamera()->ParallelProjectionOn();
		p.Renderer->AddActor(p.ImageActor);
		p.Renderer->AddActor(p.OverlayActor);
		p.ImageActor->VisibilityOff();
		p.OverlayActor->VisibilityOff();
	}
}

void vtkGinkgoImageViewer::Uninitialize()
{
	Pipeline& p = members->Pipeline;
	Estado&   e = members->Estado;

	if (!e.Initialized) {
		return;
	}

	if (p.Renderer) {
		p.Renderer->RemoveAllViewProps();
	}

	if (p.RenderWindow && p.Renderer) {
		p.RenderWindow->RemoveRenderer(p.Renderer);
	}

	if (p.RenderWindowInteractor) {
		p.RenderWindowInteractor->SetInteractorStyle(NULL);
		p.RenderWindowInteractor->SetRenderWindow(NULL);
	}

	e.Initialized = false;
	SetInteractionOff();
}

void vtkGinkgoImageViewer::SetInteraction()
{
	Pipeline& p = members->Pipeline;
	if ( p.RenderWindowInteractor ) {
		if ( !members->Propiedades.Interaction ) {
			p.RenderWindowInteractor->SetInteractorStyle(NULL);
		} else {
			if ( GetInteractorStyle() ) {
				p.RenderWindowInteractor->SetInteractorStyle(p.InteractorStyle);
			}
		}
	}
}

bool vtkGinkgoImageViewer::IsInstalledAndInitialized()
{
	Estado&   e = members->Estado;
	Pipeline& p = members->Pipeline;

	if (p.InputConnection != NULL || p.ImageData != NULL) {
		return e.PipelineInstalled && e.Initialized;
	} else {
		return false;
	}
}

void vtkGinkgoImageViewer::SetInteractionOff()
{
	Propiedades& p = members->Propiedades;
	p.Interaction = false;
	this->SetInteraction();
}

void vtkGinkgoImageViewer::SetInteractionOn()
{
	Propiedades& p = members->Propiedades;
	p.Interaction = true;
	this->SetInteraction();
}

bool vtkGinkgoImageViewer::GetInteraction()
{
	Propiedades& p = members->Propiedades;
	return p.Interaction;
}

/**
Detach the view, i.e. add its own children (if any) to its parent's children (if any).
*/
void vtkGinkgoImageViewer::Detach (void)
{
	vtkSmartPointer<vtkGinkgoImageViewer> parent = GetParent();
	if ( parent ) {
		parent->AddChildren( Children );
		parent->RemoveChild(this);
		Parent = NULL;
	}

	RemoveAllChildren();
}

void vtkGinkgoImageViewer::SetInput(const vtkSmartPointer<vtkImageData>& input, bool forzarSetup)
{
	//std::cout << "vtkGinkgoImageViewer::SetInput()" << std::endl;
	Pipeline& p = members->Pipeline;
	Estado&   e = members->Estado;
	p.ImageData = input;
	p.InputConnection = NULL;
	if (e.PipelineInstalled || forzarSetup) {
		SetupPipeline();
	}


}

void vtkGinkgoImageViewer::SetInputConnection(const vtkSmartPointer<vtkAlgorithmOutput>& input)
{
	//std::cout << "vtkGinkgoImageViewer::SetInputConnection()" << std::endl;
	Pipeline& p = members->Pipeline;
	Estado&   e = members->Estado;

	p.InputConnection = input;
	p.ImageData = NULL;
	if (e.PipelineInstalled) {
		SetupPipeline();
	}

}

void vtkGinkgoImageViewer::SetOverlay(const vtkSmartPointer<vtkImageData>& inputOverlay)
{
	//std::cout << "vtkGinkgoImageViewer::SetInputConnection()" << std::endl;
	Pipeline& p = members->Pipeline;
	Estado&   e = members->Estado;

	p.OverlayData = inputOverlay;
	if (e.PipelineInstalled) {
		if (GetNumberOfComponents() == 1)  { // NC = 1. Antes era 0, 2, 3 o 4
			//overlay???
			if (p.OverlayData != NULL)
			{
				p.TexturaOverlay->SetInput(p.OverlayData);
				p.TexturaOverlay->Modified();
				p.OverlayActor->Modified();
				p.OverlayActor->VisibilityOn();
			} else {
				p.OverlayActor->VisibilityOff();
			}
		} else {
			p.OverlayActor->VisibilityOff();
		}
	}
}

//====================================================================================================


void vtkGinkgoImageViewer::UpdateDisplayExtent()
{
	
	//std::cout << "UpdateDisplayExtent()" << std::endl;

	GNC::GCS::Timer t;

	Pipeline&    p  = members->Pipeline;
	Estado&      e  = members->Estado;
	Propiedades& pr = members->Propiedades;

	t.start();

	if (!e.Initialized || !e.PipelineInstalled) {
		return;
	}

	int    w_ext[6] = { 0, 0, 0, 0, 0, 0 };
	GNC::GCS::Vector3D origin;
	GNC::GCS::Vector3D spacing;
	int dimensions[3] = {0, 0, 0};
	double    dir[9] = {1.0,0.0,0.0,0.0,1.0,0.0,0.0,0.0,1.0};

	GetWholeExtent(w_ext);

	GetDimensions(dimensions);

	if (Loader.IsValid()) {
		Loader->GetDirection(dir);
	}
	if (p.InputConnection != NULL) {
		Loader->UpdateInformation();
		Loader->GetOutputOrigin(origin.v);
		Loader->GetOutputSpacing(spacing.v);
	}
	else {
		p.ImageData->UpdateInformation();
		p.ImageData->GetOrigin(origin.v);
		p.ImageData->GetSpacing(spacing.v);
	}

	#if 0
	std::cout << "Origin  = " << origin << std::endl;
	std::cout << "Spacing = " << spacing << std::endl;
	#endif

	pr.SetDirection(dir);

	p.ResetMatrices();

	{
		p.MatrizModelo->SetElement(0, 0, dir[0] * spacing.x);
		p.MatrizModelo->SetElement(1, 0, dir[3] * spacing.x);
		p.MatrizModelo->SetElement(2, 0, dir[6] * spacing.x);
		p.MatrizModelo->SetElement(3, 0, 0.0);

		p.MatrizModelo->SetElement(0, 1, dir[1] * spacing.y);
		p.MatrizModelo->SetElement(1, 1, dir[4] * spacing.y);
		p.MatrizModelo->SetElement(2, 1, dir[7] * spacing.y);
		p.MatrizModelo->SetElement(3, 1, 0.0);

		p.MatrizModelo->SetElement(0, 2, dir[2] * spacing.z);
		p.MatrizModelo->SetElement(1, 2, dir[5] * spacing.z);
		p.MatrizModelo->SetElement(2, 2, dir[8] * spacing.z);
		p.MatrizModelo->SetElement(3, 2, 0.0);


		p.MatrizModelo->SetElement(0, 3, origin.x);
		p.MatrizModelo->SetElement(1, 3, origin.y);
		p.MatrizModelo->SetElement(2, 3, origin.z);
		p.MatrizModelo->SetElement(3, 3, 1.0);

	}

	vtkMatrix4x4::Invert(p.MatrizModelo, p.MatrizModeloInv);

	#if 0
	std::cout << "Matriz Modelo:" << std::endl;
	p.MatrizModelo->Print(std::cout);
	#endif

	#if 0
	std::cout << "Matriz Modelo Invertida:" << std::endl;
	p.MatrizModeloInv->Print(std::cout);
	#endif

	double po[4] = {-0.5,                                                    -0.5,       0,    1.0};
	double p1[4] = {((double)dimensions[0]) - 0.5,                           -0.5,       0,    1.0};
	double p2[4] = {-0.5,                           ((double)dimensions[1]) - 0.5,       0,    1.0};

	double tpo[3];
	double tp1[3];
	double tp2[3];

	double* r = NULL;

	#if 0 // Si todo es correcto, ptest(0,0,0) transformado debe ser siempre el origen;
	double ptest[4]  = {0.0, 0.0, 0.0, 1.0};

	r = p.MatrizModelo->MultiplyDoublePoint(ptest);
	GNC::GCS::Vector3D tptest = r;
	std::cout << "P(0,0,0) * M  = " << tptest << std::endl;
	#endif

	r = p.MatrizModelo->MultiplyDoublePoint(po);
	tpo[0] = r[0]; tpo[1] = r[1]; tpo[2] = r[2];
	r = p.MatrizModelo->MultiplyDoublePoint(p1);
	tp1[0] = r[0]; tp1[1] = r[1]; tp1[2] = r[2];
	r = p.MatrizModelo->MultiplyDoublePoint(p2);
	tp2[0] = r[0]; tp2[1] = r[1]; tp2[2] = r[2];

	p.PlaneSource->SetOrigin(tpo);
	p.PlaneSource->SetPoint1(tp1);
	p.PlaneSource->SetPoint2(tp2);

	p.Plane->SetOrigin( p.PlaneSource->GetOrigin() );
	p.Plane->SetNormal( p.PlaneSource->GetNormal() );

	t.stop();
	//std::cout << "vtkGinkgoImageViewer::UpdateDisplayExtent(): " << t << std::endl;

}

void vtkGinkgoImageViewer::UpdateOrientation()
{

	//GNC::GCS::Timer t;
	//t.start();

	Pipeline& p = members->Pipeline;
	Estado& e = members->Estado;
	Propiedades& pr = members->Propiedades;

	GNC::GCS::Vector3D plane_center;
	GNC::GCS::Vector3D plane_normal;
	GNC::GCS::Vector3D plane_origin;
	GNC::GCS::Vector3D plane_p2;

	GNC::GCS::Vector3D camera_position;
	GNC::GCS::Vector3D camera_focal_point;
	GNC::GCS::Vector3D camera_view_up;

	p.PlaneSource->GetCenter(plane_center.v);
	p.PlaneSource->GetNormal(plane_normal.v);
	p.PlaneSource->GetOrigin(plane_origin.v);
	p.PlaneSource->GetPoint2(plane_p2.v);

	camera_focal_point = plane_center;

	bool hflip = e.FlipHorizontal;

	if (e.FlipVertical) {
		camera_view_up =  (plane_p2 - plane_origin);
		hflip = !hflip;
	}
	else {
		camera_view_up =  (plane_origin - plane_p2);
	}

	if (hflip) {
		camera_position = plane_center + plane_normal;
	}
	else {
		camera_position = plane_center - plane_normal;
	}

	vtkCamera *cam = p.Renderer ? p.Renderer->GetActiveCamera() : NULL;

	cam->SetFocalPoint(camera_focal_point.v);
	cam->SetPosition(camera_position.v);
	cam->SetViewUp(camera_view_up.v);
	
	cam->Roll(e.RollOffset);
	p.Renderer->ResetCameraClippingRange();

	pr.ResetModifiedStatus();
	//t.stop();

	//std::cout << "vtkGinkgoImageViewer::UpdateOrientation(): " << t << std::endl;
}


void vtkGinkgoImageViewer::IntersectarRayo(double wp[4], double vd[3], double pt[3])
{
	typedef GNC::GCS::Vector3D TVector;
	if (members != NULL) {
		Pipeline& p = members->Pipeline;
		TVector punto_recta = wp;
		TVector vdir_recta = vd;
		TVector centro_plano;
		TVector vnorm_plano;
		p.Plane->GetOrigin(centro_plano.v);
		p.Plane->GetNormal(vnorm_plano.v);
		TVector interseccion = TVector::InterseccionEntreRectaYPlano(punto_recta, vdir_recta, centro_plano, vnorm_plano);
		pt[0] = interseccion.x;
		pt[1] = interseccion.y;
		pt[2] = interseccion.z;

		//std::cout << "M1: x = " << pt[0] << ", y = " << pt[1] << ", z = " << pt[2] << std::endl;
	}
}

void vtkGinkgoImageViewer::Proyect2D(double wp[4], double ip[2])
{
	Pipeline& p = members->Pipeline;
	double pt[4] = {0.0, 0.0, 0.0, 0.0};
	double* r = p.MatrizModeloInv->MultiplyDoublePoint(wp);
	for (int i = 0; i < 4; i++) {
		pt[i] = r[i];
	}
	ip[0] = pt[0];
	ip[1] = pt[1];
}

void vtkGinkgoImageViewer::UnProyect2D(double ip[2], double wp[4])
{
	Pipeline& p = members->Pipeline;
	double pt[4] = {0.0, 0.0, 0.0, 1.0};
	pt[0] = ip[0];
	pt[1] = ip[1];
	double* r = p.MatrizModelo->MultiplyDoublePoint(pt);
	for (int i = 0; i < 4; i++) {
		wp[i] = r[i];
	}
}

void vtkGinkgoImageViewer::SetupPipeline()
{
	GNC::GCS::Timer t;
	Pipeline& p = members->Pipeline;
	Estado&   e = members->Estado;

	t.start();

	double oldSpacing[3] = {-1.0,-1.0,-1.0};
	double oldOrigin[3] = {0.0, 0.0, 0.0};
	int oldDimensions[3] = {-1,-1,-1};

	//se consultan las dimensiones y spacing para saber si al final hay que resetear el zoom
	bool mantenerVista = true;

	if (e.PipelineInstalled) {
		GetSpacing(oldSpacing);
		GetOrigin(oldOrigin);
		GetDimensions(oldDimensions);
	}
	else {
	    mantenerVista = false;
	}

	unsigned int nc = 0;

	if (p.ImageData != NULL) {
		p.ImageData->UpdateInformation();
		nc = p.ImageData->GetNumberOfScalarComponents();
	}
	else if (p.InputConnection != NULL) {
		p.InputConnection->GetProducer()->UpdateInformation();
		nc = GetNumberOfComponents();
	}
	else {
		//std::cout << "NULL input" << std::endl;
	}

	if (!e.PipelineInstalled) {
		if (nc == 1)  {
			p.Textura->MapColorScalarsThroughLookupTableOn();
			if (p.ImageData != NULL) {
			    p.Textura->Lock(GLOC());
				p.Textura->SetInput(p.ImageData);
				p.Textura->UnLock(GLOC());
			}
			else {
			    p.Textura->Lock(GLOC());
				p.Textura->SetInputConnection(p.InputConnection);
				p.Textura->UnLock(GLOC());
			}
			p.Textura->SetLookupTable(p.LookupTable);
			p.Textura->Modified();

			//overlay???
			if (p.OverlayData != NULL)
			{
				p.TexturaOverlay->SetInput(p.OverlayData);
				p.TexturaOverlay->Modified();
				p.OverlayActor->Modified();
				p.OverlayActor->VisibilityOn();
			} else {
				p.OverlayActor->VisibilityOff();
			}
		}//nc == 1
		else {
			p.Textura->MapColorScalarsThroughLookupTableOff();
			if (p.ImageData != NULL) {
			    p.Textura->Lock(GLOC());
				p.Textura->SetInput(p.ImageData);
				p.Textura->UnLock(GLOC());
			}
			else {
			    p.Textura->Lock(GLOC());
				p.Textura->SetInputConnection(p.InputConnection);
				p.Textura->UnLock(GLOC());
			}

			p.Textura->Modified();
			p.OverlayActor->VisibilityOff();
		}
		p.ImageActor->Modified();
		p.ImageActor->VisibilityOn();
		e.PipelineInstalled = true;
		e.CurrentNumberOfComponents = nc;
	}
	else {
		if (nc != e.CurrentNumberOfComponents) {
			if (nc == 1)  { // NC = 1. Antes era 0, 2, 3 o 4
				p.Textura->MapColorScalarsThroughLookupTableOn();
				p.StreamConnector->RemoveAllInputs();
				if (p.ImageData != NULL) {
				    p.Textura->Lock(GLOC());
					p.Textura->SetInput(p.ImageData);
					p.Textura->UnLock(GLOC());
				}
				else {
				    p.Textura->Lock(GLOC());
					p.Textura->SetInputConnection(p.InputConnection);
					p.Textura->UnLock(GLOC());
				}
				p.Textura->SetLookupTable(p.LookupTable);
				p.Textura->Modified();
				//overlay???
				if (p.OverlayData != NULL)
				{
					p.TexturaOverlay->SetInput(p.OverlayData);
					p.TexturaOverlay->Modified();
					p.OverlayActor->Modified();
					p.OverlayActor->VisibilityOn();
				} else {
					p.OverlayActor->VisibilityOff();
				}
			}
			else { // NC = 0, 2, 3 o 4. antes era 1
				p.Textura->MapColorScalarsThroughLookupTableOff();
				p.WindowLevel->RemoveAllInputs();
				p.Textura->SetLookupTable(NULL);

				if (p.ImageData != NULL) {
				    p.Textura->Lock(GLOC());
					p.Textura->SetInput(p.ImageData);
					p.Textura->UnLock(GLOC());
				}
				else {
				    p.Textura->Lock(GLOC());
					p.Textura->SetInputConnection(p.InputConnection);
					p.Textura->UnLock(GLOC());
				}
				p.Textura->Modified();

				p.OverlayActor->VisibilityOff();
			}
			e.CurrentNumberOfComponents = nc;
		} else {
			if (nc == 1) {
				p.Textura->MapColorScalarsThroughLookupTableOn();
				if (p.OverlayData != NULL)
				{
					p.TexturaOverlay->SetInput(p.OverlayData);
					p.TexturaOverlay->Modified();
					p.OverlayActor->Modified();
					p.OverlayActor->VisibilityOn();
				}
				else {
					p.OverlayActor->VisibilityOff();
				}
			} else {
				p.Textura->MapColorScalarsThroughLookupTableOff();
				p.OverlayActor->VisibilityOff();
			}
			if (p.ImageData != NULL) {
			    p.Textura->Lock(GLOC());
				p.Textura->SetInput(p.ImageData);
				p.Textura->UnLock(GLOC());
			}
			else {
			    p.Textura->Lock(GLOC());
				p.Textura->SetInputConnection(p.InputConnection);
				p.Textura->UnLock(GLOC());
			}
			p.Textura->Modified();
		}
	}

	//t.stop();
	//std::cout << "vtkGinkgoImageViewer::SetupPipeline(): " << t << std::endl;

	double curSpacing[3] = {0.0,0.0,0.0};
	double curOrigin[3] = {0.0, 0.0, 0.0};
    int curDimensions[3] = {0,0,0};

    GetSpacing(curSpacing);
    GetOrigin(curOrigin);
	GetDimensions(curDimensions);


	for (int i = 0; mantenerVista && i < 3; i++) {
	    if ( (curSpacing[i] != oldSpacing[i]) || (curOrigin[i] != oldOrigin[i]) || (curDimensions[i] != oldDimensions[i])) {
	        mantenerVista = false;
	    }
	}
	
	SetTindex(0);

	UpdateDisplayExtent();

    if (!mantenerVista)
    {

        UpdateOrientation();

        //se comprueba si han cambiado las dimensiones o el spacing
        bool mantenerZoom = true;
        {

            for(int i = 0; i < 3 && mantenerZoom; ++i) {
                mantenerZoom = mantenerZoom && oldSpacing[i] == curSpacing[i] && oldDimensions[i] == curDimensions[i];
            }
        }
        ResetZoom(mantenerZoom);
    }
}

//====================================================================================================

vtkSmartPointer<vtkAlgorithmOutput> vtkGinkgoImageViewer::GetInputConnection()
{
	Pipeline& p = members->Pipeline;
	return p.InputConnection;
}


vtkSmartPointer<vtkImageData> vtkGinkgoImageViewer::GetInput()
{
	Pipeline& p = members->Pipeline;
	if (p.InputConnection != NULL) {
		return vtkImageData::SafeDownCast(p.Textura->GetInput());
	}
	else if (p.ImageData != NULL) {
		return p.ImageData;
	}
	else {
		return NULL;
	}
}

long vtkGinkgoImageViewer::GetImageTexture()
{
	Pipeline& p = members->Pipeline;
	if (p.InputConnection != NULL) {
		return p.Textura->GetTextureId();
	}
	else if (p.ImageData != NULL) {
		return p.Textura->GetTextureId();
	}
	else {
		return -1;
	}
	return p.Textura->GetTextureId();
}

/** Set the RenderWindow */
void vtkGinkgoImageViewer::SetRenderWindow(vtkSmartPointer<vtkRenderWindow> rw)
{
	Pipeline& p = members->Pipeline;
	Estado&   e = members->Estado;

	if (p.RenderWindow == rw) {
		return;
	}

	if (e.Initialized) {
		Uninitialize();
	}

	p.RenderWindow = rw;
	p.RenderWindowInteractor = rw->GetInteractor();

	if (p.RenderWindow && p.Renderer) {
		Initialize();
	}
}


/** Set the Renderer **/
void vtkGinkgoImageViewer::SetRenderer(vtkSmartPointer<vtkRenderer> ren)
{
	Pipeline& p = members->Pipeline;
	Estado&   e = members->Estado;

	if (p.Renderer == ren) {
		return;
	}

	if (e.Initialized) {
		Uninitialize();
	}

	p.Renderer = ren;

	if (p.RenderWindow && p.Renderer) {
		Initialize();
	}
}


/** Attach an interactor to the internal RenderWindow. **/
void vtkGinkgoImageViewer::SetInteractor(vtkSmartPointer<vtkRenderWindowInteractor> rwi)
{
	Pipeline& p = members->Pipeline;
	p.RenderWindowInteractor = rwi;
}


/** Get the vtkRenderWindow associated */
vtkSmartPointer<vtkRenderWindow> vtkGinkgoImageViewer::GetRenderWindow()
{
	Pipeline& p = members->Pipeline;
	return p.RenderWindow;
}

/** Get the vtkRenderer associated */
vtkSmartPointer<vtkRenderer> vtkGinkgoImageViewer::GetRenderer()
{
	Pipeline& p = members->Pipeline;
	return p.Renderer;
}

/** Get the vtkRenderWindow associated */
vtkSmartPointer<vtkRenderWindowInteractor> vtkGinkgoImageViewer::GetRenderWindowInteractor()
{
	Pipeline& p = members->Pipeline;
	return p.RenderWindowInteractor;
}

/** Add the actor to the first renderer of the render window if exist.
Do nothing otherwise.*/
void vtkGinkgoImageViewer::AddActor(vtkSmartPointer<vtkProp> actor)
{
	Pipeline& p = members->Pipeline;
	if (p.Renderer)
	{
		p.Renderer->AddActor(actor);
	}
	else
	{
		LOG_ERROR("Viewer", "No se pudo insertar el actor en el pipeline del viewer porque no hay un renderer asociado");
	}

}

/** remove the actor from the first renderer of the render window if exist.
Do nothing otherwise.*/
void vtkGinkgoImageViewer::RemoveActor(vtkSmartPointer<vtkProp> actor)
{
	Pipeline& p = members->Pipeline;
	if (p.Renderer)
	{
		p.Renderer->RemoveActor(actor);
	}
	else
	{
		LOG_ERROR("Viewer", "No se pudo insertar el actor en el pipeline del viewer porque no hay un renderer asociado");
	}
}

void vtkGinkgoImageViewer::Reset()
{
}

void vtkGinkgoImageViewer::SetTindex(int index)
{
	Pipeline& p = members->Pipeline;
	
	if (p.Textura)
	{
		p.Textura->SetTindex(index);
		/*
		Estado& e = members->Estado;
		if (!e.Initialized || !e.PipelineInstalled) {
			return;
		}
		
		int    w_ext[6] = { 0, 0, 0, 0, 0, 0 };
		
		GetWholeExtent(w_ext);
		
		w_ext[4] = p.Textura->GetTindex();
		w_ext[5] = p.Textura->GetTindex() + 1;
		
		SetUpdateExtent(w_ext);
		 */
	}
}

int vtkGinkgoImageViewer::GetTindex()
{
	Pipeline& p = members->Pipeline;
	if (p.Textura)
	{
		return p.Textura->GetTindex();
	}
	return -1;
}

/** Call the RenderWindow's Render() method. */
void vtkGinkgoImageViewer::Render (void)
{
	return;
	//std::cout << "vtkGinkgoImageViewer::Render()" << std::endl;
	Pipeline&    p  = members->Pipeline;
	Propiedades& pr = members->Propiedades;
	Estado&      e  = members->Estado;


	if (e.FirstRender) {

		// Initialize the size if not set yet

		vtkImageData *input = this->GetInput();
		if (input)
		{
			input->UpdateInformation();
			int *w_ext = input->GetWholeExtent();
			int xs = 0, ys = 0;

			switch (pr.Orientacion)
			{
			case AXIAL_ID:
			default:
				xs = w_ext[1] - w_ext[0] + 1;
				ys = w_ext[3] - w_ext[2] + 1;
				break;

			case SAGITTAL_ID:
				xs = w_ext[1] - w_ext[0] + 1;
				ys = w_ext[5] - w_ext[4] + 1;
				break;

			case CORONAL_ID:
				xs = w_ext[3] - w_ext[2] + 1;
				ys = w_ext[5] - w_ext[4] + 1;
				break;
			}

			// if it would be smaller than 150 by 100 then limit to 150 by 100

			if (p.RenderWindow->GetSize()[0] == 0)
			{
				//p.RenderWindow->SetSize(xs < 150 ? 150 : xs, ys < 100 ? 100 : ys);
			}


			if (p.Renderer)
			{
				p.Renderer->ResetCamera();
				p.Renderer->GetActiveCamera()->SetParallelScale(xs < 150 ? 75 : (xs - 1 ) / 2.0);
			}
			e.FirstRender = 0;
		}
	}

	if (p.RenderWindow && !p.RenderWindow->GetNeverRendered()) {
		//p.RenderWindow->
	}

}

void vtkGinkgoImageViewer::SyncRender (void)
{
}

void vtkGinkgoImageViewer::Update()
{
}

void vtkGinkgoImageViewer::UpdateImage()
{

	//Pipeline& p = members->Pipeline;
	GNC::GCS::Timer t;
	t.start();
	//p.ImageActor->GetInput()->Update();
	//p.ImageActor->Print(std::cout);
	t.stop();
	std:: cout << "vtkGinkgoImageViewer::UpdateImage(): " << t << std::endl;
}

//region Propiedades / Estado
bool vtkGinkgoImageViewer::GetDimensions( int dimensions[3] )
{
	Pipeline& p  = members->Pipeline;

	int extent[6] = {0, 0, 0, 0, 0, 0};
	dimensions[0] = dimensions[1] = dimensions[2] = 0;
	bool ok = false;

	if (p.InputConnection != NULL) {
		if (p.InputConnection->GetProducer() &&  p.InputConnection->GetProducer()->GetExecutive()) {
			vtkInformationVector* iv = p.InputConnection->GetProducer()->GetExecutive()->GetOutputInformation();
			if (iv && iv->GetNumberOfInformationObjects() == 1) {
				vtkInformation* io = iv->GetInformationObject(0);
				vtkInformationIntegerVectorKey* k = vtkStreamingDemandDrivenPipeline::WHOLE_EXTENT();
				if (io->Has(k)) {
					io->Get(k, extent);
					dimensions[0] = extent[1] + 1;
					dimensions[1] = extent[3] + 1;
					dimensions[2] = extent[5] + 1;
				}
				ok = true;
			}
		}
	}
	else if (p.ImageData != NULL) {
		p.ImageData->GetDimensions(dimensions);
	}

	return ok;

}

bool vtkGinkgoImageViewer::GetSpacing( double spacing[3] )
{
	Pipeline& p  = members->Pipeline;

	spacing[0] = spacing[1] = spacing[2] = 1.0f;

	bool ok = false;

	if (p.InputConnection != NULL) {
		if (p.InputConnection->GetProducer() &&  p.InputConnection->GetProducer()->GetExecutive()) {
			vtkInformationVector* iv = p.InputConnection->GetProducer()->GetExecutive()->GetOutputInformation();
			if (iv && iv->GetNumberOfInformationObjects() == 1) {
				vtkInformation* io = iv->GetInformationObject(0);
				io->Get(vtkDataObject::SPACING(), spacing);
				ok = true;
			}
		}
	}
	else if (p.ImageData != NULL) {
		p.ImageData->GetSpacing(spacing);
	}

	for (int i = 0; i < 2; ++i) {
		if (spacing[i] <= 0.0f) {
			spacing[i] = 1.0f;
		}
	}

	return ok;
}

bool vtkGinkgoImageViewer::GetOrigin( double origin[3] )
{
	Pipeline& p  = members->Pipeline;

	origin[0] = origin[1] = origin[2] = 0.0f;

	bool ok = false;

	if (p.InputConnection != NULL) {
		if (p.InputConnection->GetProducer() &&  p.InputConnection->GetProducer()->GetExecutive()) {
			vtkInformationVector* iv = p.InputConnection->GetProducer()->GetExecutive()->GetOutputInformation();
			if (iv && iv->GetNumberOfInformationObjects() == 1) {
				vtkInformation* io = iv->GetInformationObject(0);
				io->Get(vtkDataObject::ORIGIN(), origin);
				ok = true;
			}
		}
	}
	else if (p.ImageData != NULL) {
		p.ImageData->GetOrigin(origin);
	}
	return ok;
}

bool vtkGinkgoImageViewer::GetBounds( double bounds[6]  )
{
	double b[3];

	Estado&   e  = members->Estado;
	Pipeline& p  = members->Pipeline;
	if (e.PipelineInstalled) {
		GNC::GCS::Vector3D p1, p2, pt;
		p.PlaneSource->GetOrigin(b);
		bounds[0] = b[0];
		bounds[2] = b[1];
		bounds[4] = b[2];

		p.PlaneSource->GetOrigin(pt.v);
		p.PlaneSource->GetPoint1(p1.v);
		p.PlaneSource->GetPoint2(p2.v);

		p1 = p1 - pt;
		p2 = p2 - pt;

		pt = pt + (p1 + p2);

		bounds[1] = pt.x;
		bounds[3] = pt.y;
		bounds[5] = pt.z;
		return true;
	}
	else {
		for (int i = 0; i < 6; i++) {
			bounds[i] = 0;
		}
	}
	return false;
}

bool vtkGinkgoImageViewer::GetMatrizModelo(double matriz[16])
{
	Pipeline& p  = members->Pipeline;
	int off = 0;
	for (int i = 0; i < 4; i++)
		for (int j = 0; j < 4; j++)
			matriz[off++] = p.MatrizModelo->GetElement(j, i);

	return true;

}

bool vtkGinkgoImageViewer::GetMatrizModeloInv(double matriz[16])
{
	Pipeline& p  = members->Pipeline;
	int off = 0;
	for (int i = 0; i < 4; i++)
		for (int j = 0; j < 4; j++)
			matriz[off++] = p.MatrizModeloInv->GetElement(j, i);

	return true;

}

int vtkGinkgoImageViewer::GetNumberOfComponents()
{
	Pipeline& p  = members->Pipeline;
	int nc = 0;

	if (p.InputConnection != NULL) {
		vtkInformationVector* iv = p.InputConnection->GetProducer()->GetExecutive()->GetOutputInformation();
		if (iv->GetNumberOfInformationObjects() == 1) {
			vtkInformation* io = iv->GetInformationObject(0);
			vtkInformation* scalarInfo = vtkDataObject::GetActiveFieldInformation(io, vtkImageData::FIELD_ASSOCIATION_POINTS, vtkDataSetAttributes::SCALARS);
			if (scalarInfo && scalarInfo->Has(vtkImageData::FIELD_NUMBER_OF_COMPONENTS()))
			{
				nc = scalarInfo->Get( vtkImageData::FIELD_NUMBER_OF_COMPONENTS() );
			}
		}
	} else if (p.ImageData != NULL) {
		nc = p.ImageData->GetNumberOfScalarComponents();
	}

	return nc;
}

vtkDataArray* vtkGinkgoImageViewer::GetScalars()
{
	Pipeline& p  = members->Pipeline;

	vtkImageData* pImgData = NULL;
	if (p.InputConnection != NULL) {
		if (p.InputConnection->GetProducer() &&  p.InputConnection->GetProducer()->GetExecutive()) {
			vtkInformationVector* iv = p.InputConnection->GetProducer()->GetExecutive()->GetOutputInformation();
			if (iv && iv->GetNumberOfInformationObjects() == 1) {
				vtkInformation* io = iv->GetInformationObject(0);
				pImgData = vtkImageData::SafeDownCast(io->Get(vtkDataObject::DATA_OBJECT()));
			}
		}
	} else if (p.ImageData != NULL) {
		pImgData = p.ImageData;
	}

	if (pImgData == NULL)
		return NULL;

	pImgData->Update();
	return pImgData->GetPointData()->GetScalars();
}


void* vtkGinkgoImageViewer::GetScalarPointer()
{
	Pipeline& p  = members->Pipeline;

	vtkImageData* pImgData = NULL;
	if (p.InputConnection != NULL) {
		if (p.InputConnection->GetProducer() &&  p.InputConnection->GetProducer()->GetExecutive()) {
			vtkInformationVector* iv = p.InputConnection->GetProducer()->GetExecutive()->GetOutputInformation();
			if (iv && iv->GetNumberOfInformationObjects() == 1) {
				vtkInformation* io = iv->GetInformationObject(0);
				pImgData = vtkImageData::SafeDownCast(io->Get(vtkDataObject::DATA_OBJECT()));
			}
		}
	} else if (p.ImageData != NULL) {
		pImgData = p.ImageData;
	}

	if (pImgData == NULL)
		return NULL;

	pImgData->Update();
	return pImgData->GetScalarPointer();
}

int vtkGinkgoImageViewer::GetScalarType()
{
	Pipeline& p  = members->Pipeline;

	vtkImageData* pImgData = NULL;
	if (p.InputConnection != NULL) {
		if (p.InputConnection->GetProducer() &&  p.InputConnection->GetProducer()->GetExecutive()) {
			vtkInformationVector* iv = p.InputConnection->GetProducer()->GetExecutive()->GetOutputInformation();
			if (iv && iv->GetNumberOfInformationObjects() == 1) {
				vtkInformation* io = iv->GetInformationObject(0);
				pImgData = vtkImageData::SafeDownCast(io->Get(vtkDataObject::DATA_OBJECT()));
			}
		}
	} else if (p.ImageData != NULL) {
		pImgData = p.ImageData;
	}

	if (pImgData == NULL)
		return -1;

	pImgData->Update();
	return pImgData->GetScalarType();
}

vtkSmartPointer<vtkImageData> vtkGinkgoImageViewer::GetDataObject()
{
	Pipeline& p  = members->Pipeline;

	vtkSmartPointer<vtkImageData> pImgData = NULL;
	if (p.InputConnection != NULL ) {
		if (p.InputConnection->GetProducer() &&  p.InputConnection->GetProducer()->GetExecutive()) {
			if (p.InputConnection->GetProducer()->GetExecutive()->Update())
			{
				vtkInformationVector* iv = p.InputConnection->GetProducer()->GetExecutive()->GetOutputInformation();
				if (iv && iv->GetNumberOfInformationObjects() == 1) {
					vtkInformation* io = iv->GetInformationObject(0);
					pImgData = vtkImageData::SafeDownCast(io->Get(vtkDataObject::DATA_OBJECT()));

				}
			} else {
				pImgData = NULL;
			}
		}
	} else if (p.ImageData != NULL) {
		pImgData = p.ImageData;
		pImgData->Update();
	}

	if (pImgData == NULL)
		return NULL;

	return pImgData;
}

//endregion


/**
Switch between radiological (left is right and right is left) and
neurological (left is left and right is right) conventions.
*/

void vtkGinkgoImageViewer::SetConventionsToRadiological(void)
{
	Propiedades& pr = members->Propiedades;
	pr.Conventions = vtkGinkgoImageViewer::RADIOLOGIC;
	UpdateCamera();
}

/**
Switch between radiological (left is right and right is left) and
neurological (left is left and right is right) conventions.
*/
void vtkGinkgoImageViewer::SetConventionsToNeurological(void)
{
	Propiedades& pr = members->Propiedades;
	pr.Conventions = vtkGinkgoImageViewer::NEUROLOGIC;
	UpdateCamera();
}

int vtkGinkgoImageViewer::GetOrientation()
{
	return members->Propiedades.Orientacion;
}


double vtkGinkgoImageViewer::GetCameraRotation() const
{
	const Estado& e= members->Estado;
	return M_PI * (e.RollOffset / 180.0);
}

bool vtkGinkgoImageViewer::GetCameraFlipVertical() const
{
	const Estado& e= members->Estado;
	return e.FlipVertical;
}

bool vtkGinkgoImageViewer::GetCameraFlipHorizontal() const
{
	const Estado& e= members->Estado;
	return e.FlipHorizontal;
}

void vtkGinkgoImageViewer::SetOrientation(unsigned int orientation)
{
	if (orientation != members->Propiedades.Orientacion) {
		members->Propiedades.Orientacion = orientation;
		UpdateCamera();
	}
}

/** Specify the interactor style */
void vtkGinkgoImageViewer::SetInteractorStyle(vtkSmartPointer<vtkInteractorStyle> style)
{
	Pipeline& p = members->Pipeline;
	p.InteractorStyle = style;
}

/** Get the interactor style */
vtkSmartPointer<vtkInteractorStyle> vtkGinkgoImageViewer::GetInteractorStyle()
{
	Pipeline& p = members->Pipeline;
	return p.InteractorStyle;
}

int vtkGinkgoImageViewer::GetInteractionStyle()
{
	Comportamiento& c = members->Comportamiento;
	return c.InteractorStyleType;
}

void vtkGinkgoImageViewer::SetInteractionStyle(int type)
{
	//Pipeline& p = members->Pipeline;

	switch (type) {
		case ZOOM_WITH_SELECT_INTERACTION:
			//p.InteractorStyle = NULL;
			break;
		default:
			break;
	}

}

/** Set the background color. Format is RGB, 0 <= R,G,B <=1
Example: SetBackgroundColor(0.9,0.9,0.9) for grey-white. */
void vtkGinkgoImageViewer::SetBackgroundColor(double /*r*/, double /*g*/, double /*b*/)
{
}

/** Show/Hide the annotations. Call UpdateAnnotations after this function. */
void vtkGinkgoImageViewer::SetShowAnnotations (bool show)
{
	members->Comportamiento.ShowAnnotations = show;
}

bool vtkGinkgoImageViewer::GetShowAnnotations()
{
	return members->Comportamiento.ShowAnnotations;
}

//--------------------------------------------------------------------
//region Propiedades
void vtkGinkgoImageViewer::SetInterpolationMode(int enable)
{
	Pipeline& p  = members->Pipeline;
	p.Textura->SetInterpolate(enable);
	p.TexturaOverlay->SetInterpolate(enable);
}

int vtkGinkgoImageViewer::GetInterpolationMode()
{
	Pipeline& p  = members->Pipeline;
	return p.Textura->GetInterpolate();
}

bool vtkGinkgoImageViewer::GetWholeExtent(int extent[6]) const
{

	Pipeline& p  = members->Pipeline;
	memset(extent, 0, 6 * sizeof(int));

	bool ok = false;

	if (p.InputConnection != NULL) {
		if (p.InputConnection->GetProducer() &&  p.InputConnection->GetProducer()->GetExecutive()) {
			vtkInformationVector* iv = p.InputConnection->GetProducer()->GetExecutive()->GetOutputInformation();
			if (iv && iv->GetNumberOfInformationObjects() == 1) {
				vtkInformation* io = iv->GetInformationObject(0);
				io->Get(vtkStreamingDemandDrivenPipeline::WHOLE_EXTENT(), extent);
				ok = true;
			}
		}
	}
	else if (p.ImageData != NULL)
	{
		p.ImageData->GetWholeExtent(extent);
	}

	return ok;
}

bool vtkGinkgoImageViewer::SetUpdateExtent(int extent[6])
{
	
	Pipeline& p  = members->Pipeline;
	
	bool ok = false;
	
	if (p.InputConnection != NULL) {
		if (p.InputConnection->GetProducer() &&  p.InputConnection->GetProducer()->GetExecutive()) {
			vtkInformationVector* iv = p.InputConnection->GetProducer()->GetExecutive()->GetOutputInformation();
			if (iv && iv->GetNumberOfInformationObjects() == 1) {
				vtkInformation* io = iv->GetInformationObject(0);
				io->Set(vtkStreamingDemandDrivenPipeline::WHOLE_EXTENT(), extent, 6);
				ok = true;
			}
		}
	}
	else if (p.ImageData != NULL)
	{
		p.ImageData->SetUpdateExtent(extent);
	}
	
	return ok;
}

bool vtkGinkgoImageViewer::GetSliceRange(vtkGinkgoImageViewer::Axis axis, int range[2]) const
{
	int extent[6] = {0, 0, 0, 0, 0, 0};
	bool ok = false;
	int axis_offset = (axis >> 1);
	ok = GetWholeExtent(extent);
	range[0] = extent[axis_offset];
	range[1] = extent[axis_offset + 1];
	return ok;
}

bool vtkGinkgoImageViewer::GetLinkCameraFocalAndPosition()
{
	return true;
}

void vtkGinkgoImageViewer::SetLinkCameraFocalAndPosition(bool link)
{
	Propiedades& p = members->Propiedades;
	p.LinkCameraFocalAndPosition = link;
}

bool vtkGinkgoImageViewer::GetLinkZoom()
{
	Propiedades& p = members->Propiedades;
	return p.LinkZoom;
}

void vtkGinkgoImageViewer::SetLinkZoom(bool link)
{
	Propiedades& p = members->Propiedades;
	p.LinkZoom = link;
}

void vtkGinkgoImageViewer::CopyCameraStatus(vtkGinkgoImageViewer* w)
{
	if (w == this || w == NULL) {
		return;
	}
	else {
		Estado& e        = members->Estado;
		Estado& eo       = w->members->Estado;
		e.RollOffset     = eo.RollOffset;
		e.FlipHorizontal = eo.FlipHorizontal;
		e.FlipVertical   = eo.FlipVertical;
	}
}

void vtkGinkgoImageViewer::RotateCamera( bool right)
{
 	Estado& e = members->Estado;

	right? e.RollOffset -= 90 : e.RollOffset += 90;
	
	if (e.RollOffset > 180.0) {
		e.RollOffset -= 360.0;
	}
	else if (e.RollOffset <= -180.0) {
		e.RollOffset += 360.0;
	}

	UpdateOrientation();
}

void vtkGinkgoImageViewer::Flip(bool vertical)
{
	Estado& e = members->Estado;
	if (vertical) {
		e.FlipVertical = !e.FlipVertical;
	}
	else {
		e.FlipHorizontal = !e.FlipHorizontal;
	}
	UpdateOrientation();
}

void vtkGinkgoImageViewer::SetWindowLevelFrom(vtkSmartPointer<vtkGinkgoImageViewer> /*p_view*/)
{
}

void vtkGinkgoImageViewer::SetDefaultWindowLevel(float initialWindow, float initialLevel)
{
	Estado& e = members->Estado;
	e.InitialWindow = initialWindow;
	e.InitialLevel = initialLevel;
	e.DefaultWindowLevelSetted = true;
	ResetToDefaultWindowLevel();
}

void vtkGinkgoImageViewer::SetAutoDefaultWindowLevel()
{
	Pipeline& p  = members->Pipeline;

	vtkImageData* pImgData = NULL;
	if (p.InputConnection != NULL) {
		if (p.InputConnection->GetProducer() &&  p.InputConnection->GetProducer()->GetExecutive()) {
			vtkInformationVector* iv = p.InputConnection->GetProducer()->GetExecutive()->GetOutputInformation();
			if (iv && iv->GetNumberOfInformationObjects() == 1) {
				vtkInformation* io = iv->GetInformationObject(0);
				pImgData = vtkImageData::SafeDownCast(io->Get(vtkDataObject::DATA_OBJECT()));
			}
		}
	} else if (p.ImageData != NULL) {
		pImgData = p.ImageData;
	}

	if (pImgData == NULL)
		return;

	/*
	if (this->IsLocked())
	{
		return;
	}
	*/

	if( pImgData->GetNumberOfScalarComponents()==3 || pImgData->GetNumberOfScalarComponents()==4 )
	{
		return;
	}

	double range[2];
	// TODO PETA
	pImgData->UpdateInformation();
	pImgData->UpdateData();
	pImgData->GetScalarRange(range);
	//std::cout << range[0] << ", " << range[1] << std::endl;

	double window = range[1]-range[0];
	double level = 0.5*(range[1]+range[0]);

	//no tenemos shift/scale
	/*window = (window-this->GetShift())/this->GetScale();
	level = (level-this->GetShift())/this->GetScale();*/

	Estado& e = members->Estado;
	e.InitialWindow = window;
	e.InitialLevel = level;
	e.DefaultWindowLevelSetted = true;

	//se establece
	SyncSetWindow(e.InitialWindow);
	SyncSetLevel(e.InitialLevel);
	ActualizarAnotaciones();
}

double vtkGinkgoImageViewer::GetWindow()
{
	Pipeline& p = members->Pipeline;
	return p.WindowLevel->GetWindow();
}

double vtkGinkgoImageViewer::GetLevel()
{
	Pipeline& p = members->Pipeline;
	return p.WindowLevel->GetLevel();
}

double vtkGinkgoImageViewer::GetColorWindow()
{
	Pipeline& p = members->Pipeline;
	return p.WindowLevel->GetWindow();
}

double vtkGinkgoImageViewer::GetColorLevel()
{
	Pipeline& p = members->Pipeline;
	return p.WindowLevel->GetLevel();
}

void vtkGinkgoImageViewer::SetWindow(double window)
{
	Pipeline& p = members->Pipeline;
	p.WindowLevel->SetWindow(window);

	//rango del window/level
	if (members->Pipeline.LookupTable) {
		double v_min = this->GetLevel() - 0.5 * this->GetWindow();
		double v_max = this->GetLevel() + 0.5 * this->GetWindow();

		members->Pipeline.LookupTable->SetRange(v_min,v_max);
	}
}

void vtkGinkgoImageViewer::SetLevel(double level)
{
	Pipeline& p = members->Pipeline;
	p.WindowLevel->SetLevel(level);

	//rango del window/level
	if (members->Pipeline.LookupTable) {
		double v_min = this->GetLevel() - 0.5 * this->GetWindow();
		double v_max = this->GetLevel() + 0.5 * this->GetWindow();

		members->Pipeline.LookupTable->SetRange(v_min,v_max);
	}
}

void vtkGinkgoImageViewer::SyncSetWindow (double w)
{
	SetWindow(w);
}

void vtkGinkgoImageViewer::SyncSetLevel (double w)
{
	SetLevel(w);
}

void vtkGinkgoImageViewer::ResetToDefaultWindowLevel()
{
	Estado& e = members->Estado;
	if (e.DefaultWindowLevelSetted) {
		SyncSetWindow(e.InitialWindow);
		SyncSetLevel(e.InitialLevel);
		//Modified();
		ActualizarAnotaciones();
	}
	else {
		this->SetAutoDefaultWindowLevel();
	}
}



void vtkGinkgoImageViewer::SyncSetCameraFocalAndPosition(double* /*focal*/, double* /*pos*/)
{
	;
}

void vtkGinkgoImageViewer::SetLookupTable(vtkSmartPointer<vtkScalarsToColors> lut, int idLookupTable)
{
	if (!lut) {
		return;
	}
	double v_min = this->GetLevel() - 0.5 * this->GetWindow();
	double v_max = this->GetLevel() + 0.5 * this->GetWindow();

	lut->SetRange(v_min,v_max);


	vtkLookupTable* realLut = vtkLookupTable::SafeDownCast(lut);

	if (!realLut) {
		std::cerr << "Error: Cannot cast vtkScalarsToColors to vtkLookupTable." << std::endl;
		return;
	}

	/**
		Due to the same problem as above (shift/scale), one must copy the lut
		so that it does not change values of the shared object.
	 */
	vtkLookupTable* newLut = vtkLookupTable::New();
	newLut->DeepCopy(realLut);
	newLut->SetRange(v_min, v_max);
	members->Pipeline.Textura->SetLookupTable(newLut);
	members->Pipeline.LookupTable = newLut;
	members->Pipeline.LookupTableId = (vtkLookupTableManager::LookupTableIds) idLookupTable;
	newLut->Delete();
}

vtkSmartPointer<vtkScalarsToColors> vtkGinkgoImageViewer::GetLookupTable()
{
	return members->Pipeline.LookupTable;
}

int vtkGinkgoImageViewer::GetIdLookupTable()
{
	return members->Pipeline.LookupTableId;
}

//region Control: Zoom
void vtkGinkgoImageViewer::ResetZoom(bool mantenerZoom)
{
	Pipeline& p = members->Pipeline;
	Estado& e = members->Estado;
	p.Renderer->ResetCamera();

	e.InitialParallelScale = p.Renderer->GetActiveCamera()->GetParallelScale();
	if (mantenerZoom) {
		SetZoom(e.Zoom);
	} else {
		e.Zoom = 1.0;
	}

}

/** Set the actual zoom factor of the view. */
void vtkGinkgoImageViewer::SetZoom(double factor)
{
	Pipeline& p = members->Pipeline;
	Estado& e = members->Estado;

	//se satura
#ifndef _GINKGO_DEBUG
	factor = std::max(factor,(double)VTK_GINKGO_ZOOM_MIN);
	factor = std::min(factor,(double)VTK_GINKGO_ZOOM_MAX);
#endif

	if (p.Renderer == NULL) {
		return;
	}
	

	if( this->IsLocked() )
	{
		return;
	}

	this->Lock();


	for( unsigned int i=0; i<this->Children.size(); i++)
	{

		vtkGinkgoImageViewer* view = vtkGinkgoImageViewer::SafeDownCast (this->Children[i]);

		if( view && view->GetLinkZoom() )
		{
			view->SyncSetZoom (factor);
			if( !view->GetRenderWindow()->GetNeverRendered() )
			{
				view->Render();
			}
		}
	}
	this->UnLock();


	vtkCamera *camera = p.Renderer->GetActiveCamera();

	e.Zoom = factor;

	camera->SetParallelScale(e.InitialParallelScale / e.Zoom);

	//se invoca el evento
	this->InvokeEvent (vtkGinkgoImageViewer::ViewImageZoomChangeEvent, NULL);
	/*
	if (p.RenderWindowInteractor->GetLightFollowCamera()) {
		p.Renderer->UpdateLightsGeometryToFollowCamera();
	}
	*/
}

double vtkGinkgoImageViewer::GetZoom()
{
	Estado& e = members->Estado;
	return e.Zoom;
}

void vtkGinkgoImageViewer::SyncSetZoom(double factor)
{
	SetZoom(factor);
}

int vtkGinkgoImageViewer::GetLeftButtonInteractionStyle()
{
	Comportamiento& c = members->Comportamiento;
	Interactuacion& i = c.Interactuacion;
	return i.LeftButtonInteractionStyle;
}

int vtkGinkgoImageViewer::GetRightButtonInteractionStyle()
{
	Comportamiento& c = members->Comportamiento;
	Interactuacion& i = c.Interactuacion;
	return i.RightButtonInteractionStyle;
}

int vtkGinkgoImageViewer::GetMiddleButtonInteractionStyle()
{
	Comportamiento& c = members->Comportamiento;
	Interactuacion& i = c.Interactuacion;
	return i.MiddleButtonInteractionStyle;
}

int vtkGinkgoImageViewer::GetWheelInteractionStyle()
{
	Comportamiento& c = members->Comportamiento;
	Interactuacion& i = c.Interactuacion;
	return i.WheelInteractionStyle;
}

//--------------------------------------------------------------------
//region Conversion de coordenadas

void vtkGinkgoImageViewer::CoordenadasImagenACoordenadasMundo(const double ip[2], double wp[3])
{
	double pointImagen[4] = {0.0,0.0,0.0,1.0};
	pointImagen[0] = ip[0];
	pointImagen[1] = ip[1];

	Pipeline& p = members->Pipeline;
	double pt[4] = {0.0, 0.0, 0.0, 0.0};
	double* r = p.MatrizModelo->MultiplyDoublePoint(pointImagen);
	for (int i = 0; i < 4; i++) {
		pt[i] = r[i];
	}
	//transformacion homogeneas => no homogeneas
	if (r[3]) {
		wp[0] = r[0]/r[3];
		wp[1] = r[1]/r[3];
		wp[2] = r[2]/r[3];
   }

}

void vtkGinkgoImageViewer::CoordenadasImagenACoordenadasPixel(const double ip[2], int pp[2])
{
	//Transforma a coordenadas pixel para usar con vtk (un 0.5 del desplazamiento y otro para hacer un round)
	pp[0] = floor(ip[0] + 0.5);
	pp[1] = floor(ip[1] + 0.5);
}

void vtkGinkgoImageViewer::CoordenadasImagenACoordenadasPixel(GNC::GCS::Vector& vector)
{
	vector += 0.5;
}

void vtkGinkgoImageViewer::CoordenadasImagenInvertidaACoordenadasMundo(const double ip[3], double wp[3])
{
	vtkSmartPointer<vtkImageData> input = GetInput();
	if (!input)
	{
		return;
	}

	const double* spacing = input->GetSpacing();
	const double* origin = input->GetOrigin();
	const int* dimensions = input->GetDimensions();

	wp[0] = ((double)ip[0] * spacing[0]) + origin[0];
	wp[1] = (double) -1.0f * ( (  ( dimensions[1] - ip[1] ) * spacing[1]) + origin[1] );
	wp[2] = ((double)ip[2] * spacing[2]) + origin[2];
}

void vtkGinkgoImageViewer::CoordenadasImagenInvertidaACoordenadasMundo(const double& ipx, const double& ipy, double& wpx, double& wpy)
{
	vtkSmartPointer<vtkImageData> input = GetInput();
	if (!input)
	{
		return;
	}

	const double* spacing = input->GetSpacing();
	const double* origin = input->GetOrigin();
	const int* dimensions = input->GetDimensions();

	wpx = ((double)ipx * spacing[0]) + origin[0];
	wpy = (double) -1.0f * ( (  ( dimensions[1] - ipy ) * spacing[1]) + origin[1] );
}

void vtkGinkgoImageViewer::CoordenadasMundoACoordenadasImagen(const double wp[3], double ip[3])
{
	vtkSmartPointer<vtkImageData> input = GetInput();
	if (!input)
	{
		return;
	}

	const double* spacing = input->GetSpacing();
	const double* origin = input->GetOrigin();

	ip[0] = (wp[0] - origin[0]) / spacing[0];
	ip[1] = ( ((double) -1.0f * wp[1]) - origin[1]) / spacing[1]; // Radiologic!!
	ip[2] = (wp[2] - origin[2]) / spacing[2];
}

void vtkGinkgoImageViewer::CoordenadasMundoACoordenadasImagen(const double& wpx, const double& wpy, double& ipx, double& ipy)
{
	vtkSmartPointer<vtkImageData> input = GetInput();
	if (!input)
	{
		return;
	}

	const double* spacing = input->GetSpacing();
	const double* origin = input->GetOrigin();

	ipx = (wpx - origin[0]) / spacing[0];
	ipy = (wpy - origin[1]) / spacing[1];
}

void vtkGinkgoImageViewer::CoordenadasMundoACoordenadasImagenInvertida(const double wp[3], double ip[3])
{
	vtkSmartPointer<vtkImageData> input = GetInput();
	if (!input)
	{
		return;
	}

	const double* spacing = input->GetSpacing();
	const double* origin = input->GetOrigin();
	const int* dimensions = input->GetDimensions();

	ip[0] = (wp[0] - origin[0]) / spacing[0];
	ip[1] =  (double)dimensions[1] - ( (wp[1] - origin[1]) / spacing[1] );
	ip[2] = (wp[2] - origin[2]) / spacing[2];
}

void vtkGinkgoImageViewer::CoordenadasMundoACoordenadasImagenInvertida(const double& wpx, const double& wpy, double& ipx, double& ipy)
{
	vtkSmartPointer<vtkImageData> input = GetInput();
	if (!input)
	{
		return;
	}

	const double* spacing = input->GetSpacing();
	const double* origin = input->GetOrigin();
	const int* dimensions = input->GetDimensions();

	ipx = (wpx - origin[0]) / spacing[0];
	ipy =  (double)dimensions[1] - (  ( ((double) -1.0f * wpy) - origin[1]) / spacing[1] );
}

//--------------------------------------------------------------------
//region Anotaciones
void vtkGinkgoImageViewer::ActualizarAnotaciones()
{
	/*
	SetupAnnotations();
	*/
}


//region Propiedades geometricas

void vtkGinkgoImageViewer::IntersectarRectangulo(double /*wp0*/[3], double /*wp1*/[3], double /*ip0*/[3], double /*ip1*/[3])
{
	/*
	double *imBounds = this->GetImage()->GetBounds();
	//std::cout << "bounds = " << imBounds[0] << ", " << imBounds[1] << ", " << imBounds[2] << ", " << imBounds[3] << std::endl;

	ip0[0] = std::min(std::max(imBounds[0], ip0[0]), imBounds[1]);
	ip0[1] = std::min(std::max(imBounds[2], ip0[1]), imBounds[3]);
	ip0[2] = std::min(std::max(imBounds[4], ip0[2]), imBounds[5]);

	ip1[0] = std::min(std::max(imBounds[0], ip1[0]), imBounds[1]);
	ip1[1] = std::min(std::max(imBounds[2], ip1[1]), imBounds[3]);
	ip1[2] = std::min(std::max(imBounds[4], ip1[2]), imBounds[5]);


	CoordenadasMundoACoordenadasImagen(wp0, ip0);
	CoordenadasMundoACoordenadasImagen(wp1, ip1);

	//std::cout << "ip0 = " << ip0[0] << ", " << ip0[1] << std::endl;
	//std::cout << "ip1 = " << ip1[0] << ", " << ip1[1] << std::endl;
	*/

}

/** Update the annotations. */
void vtkGinkgoImageViewer::UpdateAnnotations( void )
{
}

void vtkGinkgoImageViewer::PrintSelf(std::ostream& os, vtkIndent indent)
{
	this->Superclass::PrintSelf(os, indent);
}

unsigned int vtkGinkgoImageViewer::GetConventions()
{
	Propiedades& p = members->Propiedades;
	return p.Conventions;
}

/** Reset the camera */
void vtkGinkgoImageViewer::ResetCamera()
{
	//GNC::GCS::Timer t;

	//t.start();

	Pipeline& p = members->Pipeline;

	if (!p.Renderer) {
		return;
	}

	double zoom = GetZoom();
	vtkCamera *camera = p.Renderer->GetActiveCamera();
	double c_position[3], focal[3];
	camera->GetPosition(c_position);
	camera->GetFocalPoint(focal);

	double focal2[3], pos2[3];
	camera->GetFocalPoint(focal2);
	camera->GetPosition(pos2);

	//
	camera->SetFocalPoint(focal[0], focal[1], focal2[2]);
	camera->SetPosition(c_position[0], c_position[1], pos2[2]);

	this->SetZoom(zoom / GetZoom());

	//t.stop();
	//std::cout << "vtkGinkgoImageViewer::ResetCamera(): " << t << std::endl;
}

vtkSmartPointer<vtkGinkgoImageViewer> vtkGinkgoImageViewer::GetParent (void) const
{
	return NULL;
}

/**
Add a child to the list of children. Check if the child is already
in the list firt.
*/
void vtkGinkgoImageViewer::AddChild (vtkSmartPointer<vtkGinkgoImageViewer> p_view)
{
}

void vtkGinkgoImageViewer::AddChildren (std::vector<vtkSmartPointer<vtkGinkgoImageViewer> > p_viewlist)
{
}

/**
Remove a child form the list of children.
*/
void vtkGinkgoImageViewer::RemoveChild (vtkSmartPointer<vtkGinkgoImageViewer> view)
{
}

void vtkGinkgoImageViewer::RemoveAllChildren (void)
{
}

void vtkGinkgoImageViewer::DrawOn()
{
}

void vtkGinkgoImageViewer::DrawOff()
{
}

void vtkGinkgoImageViewer::SetIsProcessed(bool processed)
{
	Estado& e = members->Estado;
	e.IsProcessed = processed;
}

bool vtkGinkgoImageViewer::GetIsProcessed()
{
	Estado& e = members->Estado;
	return e.IsProcessed;
}

/**
Part of the function propagation mechanism, when the function Lock() is
called, the view does not transmit the function to its children (and does
not do anything in fact).
*/
void vtkGinkgoImageViewer::Lock (void)
{
}

/**
A call to UnLock() permits to transmit function calls to the view's children.
*/
void vtkGinkgoImageViewer::UnLock (void)
{
}

/**
Returns true if the view has this child in its list.
*/
bool vtkGinkgoImageViewer::HasChild (vtkSmartPointer<vtkGinkgoImageViewer>) const
{
	return false;
}

void vtkGinkgoImageViewer::SetLinkRender(bool linkRender)
{
	Propiedades& p = members->Propiedades;
	p.LinkRender = linkRender;
}

bool vtkGinkgoImageViewer::GetLinkRender()
{
	Propiedades& p = members->Propiedades;
	return p.LinkRender;
}

//------------------------------------------------------------------------------------------------------

void vtkGinkgoImageViewer::SetupAnnotations()
{
	/*
	GTRACE("vtkGinkgoImageViewer::SetupAnnotations()");
	if (!this->GetImage()) {
	return;
	}

	//	vtkViewImage2D::SetupAnnotations();

	std::ostringstream osImageSizeData;
	std::ostringstream osPixelSizeData;
	std::ostringstream osVoxelSizeData;

	std::ostringstream osCurrentWindowLevelData;

	osCurrentWindowLevelData << this->GetWindow() << " / " << this->GetLevel();

	int *dims = this->GetImage()->GetDimensions();
	double *spacing = this->GetImage()->GetSpacing();

	switch (this->OrientacionVista) {

	case ANOTACION_AXIAL :

	if (this->ShowDirections) {
	//this->SetNorthAnnotation("A");
	//this->SetSouthAnnotation("P");
	if (this->Conventions == RADIOLOGIC) {
	//this->SetEastAnnotation("L");
	//this->SetWestAnnotation("R");
	} else {
	//this->SetEastAnnotation("R");
	//this->SetWestAnnotation("L");
	}
	}

	break;


	case ANOTACION_SAGITAL :

	if (this->ShowDirections) {
	//this->SetNorthAnnotation("S");
	//this->SetSouthAnnotation("I");
	//this->SetEastAnnotation("P");
	//this->SetWestAnnotation("A");
	}

	break;

	case ANOTACION_CORONAL :

	if (this->ShowDirections) {
	//this->SetNorthAnnotation("S");
	//this->SetSouthAnnotation("I");

	if (this->Conventions == RADIOLOGIC) {
	//this->SetEastAnnotation("L");
	//this->SetWestAnnotation("R");
	} else {
	//this->SetEastAnnotation("R");
	//this->SetWestAnnotation("L");
	}

	}

	break;
	case ANOTACION_PORDEFECTO :
	break;

	case ANOTACION_DESHABILITADA:
	break;
	}

	osImageSizeData << dims[0] << " x " << dims[1];
	osPixelSizeData << spacing[0] << " x " << spacing[1];
	osVoxelSizeData << spacing[0] << " x " << spacing[1] << " x " << spacing[2];

	ImageSizeData = osImageSizeData.str();
	PixelSizeData = osPixelSizeData.str();
	VoxelSizeData = osVoxelSizeData.str();

	CurrentWindowLevelData = osCurrentWindowLevelData.str();

	if (this->GetShowAnnotations() && Annotator != NULL) {
	//this->SetUpLeftAnnotation(Annotator->GetTopLeftAnnotation(this).c_str());
	//this->SetUpRightAnnotation(Annotator->GetTopRightAnnotation(this).c_str());
	//this->SetDownLeftAnnotation(Annotator->GetBottomLeftAnnotation(this).c_str());
	//this->SetDownRightAnnotation(Annotator->GetBottomRightAnnotation(this).c_str());
	}
	*/
}

void vtkGinkgoImageViewer::UpdateCamera()
{
	/*
	GNC::GCS::Timer t;

	t.start();



	t.stop();
	std::cout << "vtkGinkgoImageViewer::UpdateCamera(): " << t << std::endl;
	*/
	/*
	GTRACE("vtkGinkgoImageViewer::UpdatePosition()");

	if (!this->GetImage()) {
	return;
	}

	double x = 0;
	double y = 0;
	double max_x = 0;
	double max_y = 0;
	double min_x = 0;
	double min_y = 0;
	double pos[3];

	this->GetCurrentPoint(pos);

	double* spacing = this->GetImage()->GetSpacing();
	double* origin = this->GetImage()->GetOrigin();
	double *imBounds = this->GetImage()->GetBounds();

	// check if pos lies inside image bounds
	if (pos[0] < imBounds[0] || pos[0] > imBounds[1] ||
	pos[1] < imBounds[2] || pos[1] > imBounds[3] ||
	pos[2] < imBounds[4] || pos[2] > imBounds[5]) {
	// we are outside image bounds
	} else {
	pos[0] = double(vtkrint((pos[0] - origin[0]) / spacing[0])) * spacing[0] + origin[0];
	pos[1] = double(vtkrint((pos[1] - origin[1]) / spacing[1])) * spacing[1] + origin[1];
	pos[2] = double(vtkrint((pos[2] - origin[2]) / spacing[2])) * spacing[2] + origin[2];

	switch (this->Orientation) {
	case SAGITTAL_ID :

	this->ImageReslice->SetResliceAxesOrigin(pos[0], 0, 0);

	x = (double) pos[1];
	y = (double) pos[2];
	max_x = this->GetWholeMaxPosition(1);
	max_y = this->GetWholeMaxPosition(2);
	min_x = this->GetWholeMinPosition(1);
	min_y = this->GetWholeMinPosition(2);
	break;

	case CORONAL_ID :

	this->ImageReslice->SetResliceAxesOrigin(0, pos[1], 0);

	if (this->Conventions == RADIOLOGIC) {
	x = (double) pos[0];
	max_x = this->GetWholeMaxPosition(0);
	min_x = this->GetWholeMinPosition(0);
	} else {
	x = (double) pos[0]*-1.0;
	max_x = this->GetWholeMaxPosition(0)*-1.0;
	min_x = this->GetWholeMinPosition(0)*-1.0;
	}
	y = (double) pos[2];
	max_y = this->GetWholeMaxPosition(2);
	min_y = this->GetWholeMinPosition(2);
	break;


	case AXIAL_ID :

	this->ImageReslice->SetResliceAxesOrigin(0, 0, pos[2]);

	if (this->Conventions == RADIOLOGIC) {
	x = (double) pos[0];
	max_x = this->GetWholeMaxPosition(0);
	min_x = this->GetWholeMinPosition(0);
	} else {
	x = (double) pos[0]*-1.0;
	max_x = this->GetWholeMaxPosition(0)*-1.0;
	min_x = this->GetWholeMinPosition(0)*-1.0;
	}
	y = (double) pos[1]*-1.0;
	max_y = this->GetWholeMaxPosition(1)*-1.0;
	min_y = this->GetWholeMinPosition(1)*-1.0;
	break;
	}

	if (this->ShowCurrentPoint) {
	this->HorizontalLineSource->SetPoint1(min_x, y, 0.001);
	this->HorizontalLineSource->SetPoint2(max_x, y, 0.001);
	this->VerticalLineSource->SetPoint1(x, min_y, 0.001);
	this->VerticalLineSource->SetPoint2(x, max_y, 0.001);
	}

	this->ImageReslice->Update(); // needed to update input Extent

	}

	std::ostringstream osCurrentSliceData;
	std::ostringstream osCurrentPointValueData;
	std::ostringstream osCurrentPointPositionData;

	int imCoor[3];
	this->GetCurrentVoxelCoordinates(imCoor);
	int dims[3];
	this->GetImage()->GetDimensions(dims);

	switch (this->Orientation) {
	case AXIAL_ID :
	osCurrentSliceData << imCoor[2] + 1 << " / " << dims[2];
	break;

	case CORONAL_ID :
	osCurrentSliceData << imCoor[1] + 1 << " / " << dims[1];
	break;

	case SAGITTAL_ID :
	osCurrentSliceData << imCoor[0] + 1 << " / " << dims[0];
	break;
	}

	const double* c_position = this->GetCurrentPoint();

	osCurrentPointValueData << this->GetCurrentPointDoubleValue();
	osCurrentPointPositionData << "x = " << c_position[0] << ", y =" << c_position[1] << ", z = " << c_position[2];

	CurrentSliceData = osCurrentSliceData.str();
	CurrentPointPositionData = osCurrentPointPositionData.str();
	CurrentPointValueData = osCurrentPointValueData.str();

	unsigned int direction = this->GetOrthogonalAxis(this->GetOrientation());
	switch (direction) {
	case X_ID:
	this->DataSetCutPlane->SetOrigin(pos[0], 0, 0);
	this->DataSetCutPlane->SetNormal(1, 0, 0);
	this->DataSetCutBox->SetBounds(this->DataSetCutPlane->GetOrigin()[0], this->DataSetCutPlane->GetOrigin()[0] + this->BoxThickness,
	this->GetWholeMinPosition(1), this->GetWholeMaxPosition(1),
	this->GetWholeMinPosition(2), this->GetWholeMaxPosition(2));

	break;
	case Y_ID:
	this->DataSetCutPlane->SetOrigin(0, pos[1], 0);
	this->DataSetCutPlane->SetNormal(0, 1, 0);
	this->DataSetCutBox->SetBounds(this->GetWholeMinPosition(0), this->GetWholeMaxPosition(0),
	this->DataSetCutPlane->GetOrigin()[1], this->DataSetCutPlane->GetOrigin()[1] + this->BoxThickness,
	this->GetWholeMinPosition(2), this->GetWholeMaxPosition(2));
	break;
	case Z_ID:
	this->DataSetCutPlane->SetOrigin(0, 0, pos[2]);
	this->DataSetCutPlane->SetNormal(0, 0, 1);
	this->DataSetCutBox->SetBounds(this->GetWholeMinPosition(0), this->GetWholeMaxPosition(0),
	this->GetWholeMinPosition(1), this->GetWholeMaxPosition(1),
	this->DataSetCutPlane->GetOrigin()[2], this->DataSetCutPlane->GetOrigin()[2] + this->BoxThickness);
	break;
	}


	if (this->DataSetList.size()) {

	this->ResetAndRestablishZoomAndCamera();

	//
	//  We need to correct for the origin of the actor. Indeed, the ImageActor
	//  has always position 0 in Z in axial view, in X in sagittal view and
	//  in Y in coronal view. The projected dataset have an origin that depends
	//  on the required slice and can be negative. In that case, the projected
	//  data are behind the image actor and thus not visible. Here, we correct
	//  this by translating the actor so that it becomes visible.
	for (unsigned int i = 0; i<this->DataSetActorList.size(); i++) {
	double Pos[3];
	this->DataSetActorList[i]->GetPosition(Pos);

	switch (direction) {
	case X_ID:
	Pos[0] = -1.0 * pos[0] + 1.0;
	break;

	case Y_ID:
	Pos[1] = -1.0 * pos[1] + 1.0;
	break;

	case Z_ID:
	Pos[2] = -1.0 * pos[2] + 1.0;
	break;
	}

	this->DataSetActorList[i]->SetPosition(Pos);
	}

	}

	if (this->GetShowAnnotations() && Annotator != NULL) {
	this->SetUpLeftAnnotation(Annotator->GetTopLeftAnnotation(this).c_str());
	this->SetUpRightAnnotation(Annotator->GetTopRightAnnotation(this).c_str());
	this->SetDownLeftAnnotation(Annotator->GetBottomLeftAnnotation(this).c_str());
	this->SetDownRightAnnotation(Annotator->GetBottomRightAnnotation(this).c_str());
	}
	*/

}

void vtkGinkgoImageViewer::ActualizarImagen()
{
	/*
	GetImage()->Modified();
	ImageReslice->Update();
	WindowLevel->Update();
	UpdateImageActor();
	*/
}
