/*
*  
*  $Id: pacscontroller.cpp 4103 2011-08-31 06:33:25Z tovar $
*  Ginkgo CADx Project
*
*  Copyright 2008-10 MetaEmotion S.L. All rights reserved.
*  http://ginkgo-cadx.com
*
*  This file is licensed under LGPL v3 license.
*  See License.txt for details
*
*
*/
#include <string>
#include <openssl/md5.h>

#include "pacscontroller.h"
#include "dcmtk/dicomimg2dcm.h"
#include "dcmtk/dicommanager.h"
#include "dcmtk/dicomservers.h"
#include "dcmtk/dicomservice.h"
#include "dcmtk/dicomnetclient.h"
#include "dcmtk/dicomfindassociation.h"
#include "dcmtk/dicommoveassociation.h"
#include "dcmtk/dicomgetassociation.h"
#include "dcmtk/dicomstoreassociation.h"
#include <dcmtk/dcmdata/dcrledrg.h>
#include <dcmtk/dcmjpeg/djdecode.h>
#include <dcmtk/dcmdata/dcdicdir.h>
#include <api/ientorno.h>
#include <api/dicom/imodelodicom.h>
#include <main/entorno.h>
#include <wx/dir.h>
#include <wx/filename.h>
#include <main/controllers/configurationcontroller.h>

#include <api/icontextoestudio.h>

#ifdef verify
#define MACRO_QUE_ESTORBA verify
#undef verify
#endif

#include <dcmtk/dcmdata/dcdeftag.h>
#include <dcmtk/dcmdata/dcdict.h>
#include <dcmtk/dcmdata/dcdicent.h>

#ifdef MACRO_QUE_ESTORBA
#define verify MACRO_QUE_ESTORBA
#endif

namespace GIL {
	namespace DICOM {
		PACSController* PACSController::m_pInstance = NULL;

		PACSController::PACSController()
		{
			m_pServiceInstance = NULL;
		}

		PACSController::~PACSController()
		{
			//std::cout << "PACSController destruyendose" << std::endl;
			DicomServerList::FreeInstance();
			//std::cout << "PACSController destruido" << std::endl;
		}


		PACSController* PACSController::Instance()
		{
			if (m_pInstance == NULL) {
				m_pInstance = new PACSController();				
			}
			return m_pInstance;
		}

		void PACSController::FreeInstance()
		{
			if (m_pInstance != NULL) {
				m_pInstance->StopService();
				delete m_pInstance;
				m_pInstance = NULL;
			}
			//se desregistran los codecs dcmtk!
			DJDecoderRegistration::cleanup();
			DcmRLEDecoderRegistration::cleanup();
			//purgar temp
			PurgarDirectorioTemporal();
		}

		void PACSController::buildDicomDir()
		{

		}

		void PACSController::Store(DcmDataset* dset)
		{

			GTRACE("PACSController::Store() ");
			// TODO: Try/catch


			OFString OFPacienteUId;
			OFString OFEstudioUId;
			OFString OFSerieUId;
			OFString OFImagenUId;

			std::string PacienteUId;
			std::string EstudioUId;
			std::string SerieUId;
			std::string ImagenUId;

			if (dset->findAndGetOFString(DCM_PatientID, OFPacienteUId).good()) {
				PacienteUId.assign(OFPacienteUId.c_str());
			}
			if (dset->findAndGetOFString(DCM_StudyInstanceUID, OFEstudioUId).good()) {
				EstudioUId.assign(OFEstudioUId.c_str());
			}
			if (dset->findAndGetOFString(DCM_SeriesInstanceUID, OFSerieUId).good()) {
				SerieUId.assign(OFSerieUId.c_str());
			}
			if (dset->findAndGetOFString(DCM_SOPInstanceUID, OFImagenUId).good()) {
				ImagenUId.assign(OFImagenUId.c_str());
			}

			DcmFileFormat f(dset);

			//std::cout << "PacienteUId = " << PacienteUId << std::endl;
			//std::cout << "EstudioUId = " << EstudioUId << std::endl;
			//std::cout << "SerieUId = " << SerieUId << std::endl;
			//std::cout << "ImagenUId = " << ImagenUId << std::endl;

			std::string rutaStd;
			GetRutaImagenTemp(PacienteUId,EstudioUId,SerieUId,ImagenUId,rutaStd);

			f.saveFile(rutaStd.c_str(), dset->getOriginalXfer());
		}

		IModeloDicom* PACSController::BuscarPaciente(
			void* connectionKey,
			const std::string& serverId,
			const std::string& idPaciente,
			const std::string& nombrePaciente,
			const std::string& /*fechaNacimiento*/,
			IModeloDicom* pModelo,
			GNC::IProxyNotificadorProgreso* pNotificador
			)
		{
			DicomServerList* listaServidores = DicomServerList::Instance();

			std::string id;

			DcmDataset query;
			DcmElement* e = NULL;

			e = newDicomElement(DCM_SpecificCharacterSet);
			e->putString("ISO_IR 192");
			query.insert(e);

			e = newDicomElement(DCM_QueryRetrieveLevel);
			e->putString("PATIENT");
			query.insert(e);

			e = newDicomElement(DCM_PatientID);
			e->putString(idPaciente.c_str());
			query.insert(e);

			e = newDicomElement(DCM_PatientName);
			e->putString(nombrePaciente.c_str());
			query.insert(e);

			e = newDicomElement(DCM_PatientBirthDate);
			query.insert(e);

			e = newDicomElement(DCM_PatientSex);
			query.insert(e);

			e = newDicomElement(DCM_ReferringPhysicianName);
			query.insert(e);

			NetClient<FindAssociation> a(GIL::DICOM::DCMTK::Network::Instance(connectionKey), "C-FIND", pNotificador);

			std::string localAET = GNC::Entorno::Instance()->GetDicomLocalAET();

			DicomServer* server = listaServidores->GetServer(serverId);
			if (server->useTLS) {
				a.SetTLS(server->GetCertificate(), server->GetPrivateKey(), server->GetverifyCredentials());
			}
			if (server->GetPACSUser() != "") {
				a.SetUserPass(server->GetPACSUser(), server->GetPACSPass());
			}
			a.QueryServer(&query, server, pModelo, localAET, CT_FindPacientes);

			return pModelo;
		}

		IModeloDicom* PACSController::BuscarEstudio(
			void* connectionKey,
			const std::string& serverId,
			const std::string& idPaciente,
			const std::string& nombrePaciente,
			const std::string& idEstudio,
			const std::string& AccNumber,
			const std::string& modality,
			const std::string& fechaDesde,
			const std::string& fechaHasta,
			const std::string& timeFrom,
			const std::string& timeTo,
			const std::string& descripcionEstudio,
			const std::string& WXUNUSED(nombreEstacion),
			IModeloDicom* pModelo,
			GNC::IProxyNotificadorProgreso* pNotificador
			)
		{
			DicomServerList* listaServidores = DicomServerList::Instance();


			std::string description;
			if (!descripcionEstudio.empty()) {
				description = descripcionEstudio;
			}

			// create date querystring
			std::string date;
			if (fechaDesde.empty() && fechaHasta.empty()) {
				date = "";
			} else {
				date = fechaDesde + "-" + fechaHasta;
			}

			if (fechaDesde == fechaHasta) {
				date = fechaDesde;
			}

			std::string time;
			if (timeFrom.empty() && timeTo.empty()) {
				time = "";
			} else {
				time = timeFrom + "-" + timeTo;
			}

			if (timeFrom == timeTo) {
				time = timeFrom;
			}


			std::string station;
			DcmDataset query;
			DcmElement* e = NULL;

			e = newDicomElement(DCM_SpecificCharacterSet);
			e->putString("ISO_IR 192");
			query.insert(e);

			e = newDicomElement(DCM_QueryRetrieveLevel);
			e->putString("STUDY");
			query.insert(e);

			e = newDicomElement(DCM_PatientName);
			e->putString(nombrePaciente.c_str());
			query.insert(e);

			e = newDicomElement(DCM_PatientID);
			e->putString(idPaciente.c_str());
			query.insert(e);

			e = newDicomElement(DCM_ModalitiesInStudy);
			if (modality != "*" && modality != "") {
				e->putString(modality.c_str());
			}
			query.insert(e);

			e = newDicomElement(DCM_PatientBirthDate);
			query.insert(e);

			e = newDicomElement(DCM_PatientSex);
			query.insert(e);

			e = newDicomElement(DCM_StudyDate);
			e->putString(date.c_str());
			query.insert(e);

			e = newDicomElement(DCM_StudyTime);
			e->putString(time.c_str());
			query.insert(e);

			e = newDicomElement(DCM_StudyID);
			query.insert(e);

			e = newDicomElement(DCM_StudyInstanceUID);
			e->putString(idEstudio.c_str());
			query.insert(e);

			e = newDicomElement(DCM_StudyDescription);
			e->putString(description.c_str());
			query.insert(e);

			e = newDicomElement(DCM_AccessionNumber);
			e->putString(AccNumber.c_str());
			query.insert(e);

			e = newDicomElement(DCM_StationName);
			e->putString(station.c_str());
			query.insert(e);

			e = newDicomElement(DCM_ReferringPhysicianName);
			query.insert(e);

			NetClient<FindAssociation> a(connectionKey, "C-FIND", pNotificador);

			std::string localAET = GNC::Entorno::Instance()->GetDicomLocalAET();

			DicomServer* server = listaServidores->GetServer(serverId);
			if (server->useTLS) {
				a.SetTLS(server->GetCertificate(), server->GetPrivateKey(), server->GetverifyCredentials());
			}
			if (server->GetPACSUser() != "") {
				a.SetUserPass(server->GetPACSUser(), server->GetPACSPass());
			}
			a.QueryServer(&query, server, pModelo, localAET, CT_FindEstudios);

			return pModelo;
		}

		IModeloDicom* PACSController::BuscarSeries(
			void* connectionKey,
			const std::string& serverId,
			const std::string& idEstudio,
			const std::string& idSerie,
			IModeloDicom* pModelo,
			GNC::IProxyNotificadorProgreso* pNotificador
			)
		{
			DicomServerList* listaServidores = DicomServerList::Instance();

			std::string estudioId;
			std::string serieId;

			if (!idEstudio.empty()) {
				estudioId.assign(idEstudio.c_str());
			}

			if (!idSerie.empty()) {
				serieId.assign(idSerie.c_str());
			}

			DcmDataset query;
			DcmElement* e = NULL;

			e = newDicomElement(DCM_SpecificCharacterSet);
			e->putString("ISO_IR 192");
			query.insert(e);

			e = newDicomElement(DCM_QueryRetrieveLevel);
			e->putString("SERIES");
			query.insert(e);

			e = newDicomElement(DCM_StudyInstanceUID);
			e->putString(estudioId.c_str());
			query.insert(e);

			e = newDicomElement(DCM_SeriesInstanceUID);
			e->putString(serieId.c_str());
			query.insert(e);

			e = newDicomElement(DCM_Modality);
			query.insert(e);

			e = newDicomElement(DCM_SeriesDescription);
			query.insert(e);

			e = newDicomElement(DCM_SeriesDate);
			query.insert(e);

			e = newDicomElement(DCM_SeriesTime);
			query.insert(e);

			e = newDicomElement(DCM_NumberOfSeriesRelatedInstances);
			query.insert(e);

			e = newDicomElement(DCM_ReferringPhysicianName);
			query.insert(e);

			NetClient<FindAssociation> a(connectionKey, "C-FIND", pNotificador);

			std::string localAET = GNC::Entorno::Instance()->GetDicomLocalAET();

			DicomServer* server = listaServidores->GetServer(serverId);
			if (server->useTLS) {
				a.SetTLS(server->GetCertificate(), server->GetPrivateKey(), server->GetverifyCredentials());
			}
			if (server->GetPACSUser() != "") {
				a.SetUserPass(server->GetPACSUser(), server->GetPACSPass());
			}
			a.QueryServer(&query, server, pModelo, localAET, CT_FindSeries);

			return pModelo;
		}

		IModeloDicom* PACSController::BuscarImagenes(
			void* connectionKey,
			const std::string& serverId,
			const std::string& idSerie,
			const std::string& idImagen,
			IModeloDicom* pModelo,
			GNC::IProxyNotificadorProgreso* pNotificador
			)
		{
			DicomServerList* listaServidores = DicomServerList::Instance();

			std::string serieId;

			if (!idSerie.empty()) {
				serieId = idSerie;
			}

			std::string imagenId;

			if (!idImagen.empty()) {
				imagenId = idImagen;
			}

			DcmDataset query;
			DcmElement* e = NULL;

			e = newDicomElement(DCM_SpecificCharacterSet);
			e->putString("ISO_IR 192");
			query.insert(e);

			e = newDicomElement(DCM_QueryRetrieveLevel);
			e->putString("IMAGE");
			query.insert(e);

			e = newDicomElement(DCM_SeriesInstanceUID);
			e->putString(serieId.c_str());
			query.insert(e);

			e = newDicomElement(DCM_SOPInstanceUID);
			query.insert(e);

			e = newDicomElement(DCM_ImageID);
			query.insert(e);

			e = newDicomElement(DCM_ImageIndex);
			query.insert(e);

			e = newDicomElement(DCM_ReferringPhysicianName);
			query.insert(e);

			NetClient<FindAssociation> a(connectionKey, "C-FIND", pNotificador);

			std::string localAET = GNC::Entorno::Instance()->GetDicomLocalAET();

			DicomServer* server = listaServidores->GetServer(serverId);
			if (server->useTLS) {
				a.SetTLS(server->GetCertificate(), server->GetPrivateKey(), server->GetverifyCredentials());
			}
			if (server->GetPACSUser() != "") {
				a.SetUserPass(server->GetPACSUser(), server->GetPACSPass());
			}
			a.QueryServer(&query, server, pModelo, localAET, CT_FindImagenes);

			return pModelo;
		}

		bool PACSController::LeerDicomDir(const std::string& pathDicomDir, IModeloDicom* pModelo)
		{
			if(pModelo == NULL || !DICOMManager::EsDicom(pathDicomDir))
			{
				return false;
			}

			DcmDicomDir            dicomdir(pathDicomDir.c_str());
			DcmDirectoryRecord *   root = &(dicomdir.getRootRecord());
			DcmDirectoryRecord *   PatientRecord = NULL;
			DcmDirectoryRecord *   StudyRecord = NULL;
			DcmDirectoryRecord *   SeriesRecord = NULL;
			DcmDirectoryRecord *   FileRecord = NULL;
			OFString            tmpString;

			wxFileName fileNameDicomDir(FROMPATH(pathDicomDir));
			std::string basePath;
			basePath = TOPATH(fileNameDicomDir.GetPath());

			if(root != NULL)
			{
				while (((PatientRecord = root->nextSub(PatientRecord)) != NULL))
				{
					OFString uidPaciente, nombrePaciente, fechaNacimiento, sexo;
					PatientRecord->findAndGetOFString(DCM_PatientID,uidPaciente);
					PatientRecord->findAndGetOFString(DCM_PatientName,nombrePaciente);
					PatientRecord->findAndGetOFString(DCM_PatientBirthDate, fechaNacimiento);
					PatientRecord->findAndGetOFString(DCM_PatientSex, sexo);

					pModelo->InsertarPaciente(uidPaciente.c_str(),nombrePaciente.c_str(),fechaNacimiento.c_str(),sexo.c_str());
					while (((StudyRecord = PatientRecord->nextSub(StudyRecord)) != NULL))
					{
						OFString uidEstudio, descripcionEstudio, fechaEstudio, horaEstudio, doctor, accesionNumber;
						StudyRecord->findAndGetOFString(DCM_StudyInstanceUID, uidEstudio);
						StudyRecord->findAndGetOFString(DCM_StudyDescription, descripcionEstudio);
						StudyRecord->findAndGetOFString(DCM_StudyDate, fechaEstudio);
						StudyRecord->findAndGetOFString(DCM_StudyTime, horaEstudio);
						StudyRecord->findAndGetOFString(DCM_ReferringPhysicianName, doctor);
						StudyRecord->findAndGetOFString(DCM_AccessionNumber, accesionNumber);

						pModelo->InsertarEstudio(uidPaciente.c_str(),uidEstudio.c_str(),accesionNumber.c_str(), descripcionEstudio.c_str(),"",fechaEstudio.c_str(),horaEstudio.c_str(),doctor.c_str());
						while (((SeriesRecord = StudyRecord->nextSub(SeriesRecord)) != NULL))
						{
							OFString uidSerie,modalidadSerie,fechaSerie,horaSerie,descripcionSerie,numeroSerie,doctorSerie;
							SeriesRecord->findAndGetOFString(DCM_SeriesInstanceUID, uidSerie);
							SeriesRecord->findAndGetOFString(DCM_Modality, modalidadSerie);
							SeriesRecord->findAndGetOFString(DCM_SeriesDate, fechaSerie);
							SeriesRecord->findAndGetOFString(DCM_SeriesTime, horaSerie);
							SeriesRecord->findAndGetOFString(DCM_SeriesDescription, descripcionSerie);
							SeriesRecord->findAndGetOFString(DCM_SeriesNumber, numeroSerie);
							SeriesRecord->findAndGetOFString(DCM_ReferringPhysicianName, doctorSerie);

							pModelo->InsertarSerie(uidEstudio.c_str(),uidSerie.c_str(),modalidadSerie.c_str(),fechaSerie.c_str(),horaSerie.c_str(),descripcionSerie.c_str(),numeroSerie.c_str(),doctorSerie.c_str());
							while(((FileRecord = SeriesRecord->nextSub(FileRecord)) != NULL))
							{
								if(FileRecord->findAndGetOFStringArray(DCM_DirectoryRecordType,tmpString).good() && tmpString == "IMAGE")
								{
									if(FileRecord->findAndGetOFStringArray(DCM_ReferencedFileID,tmpString).good())
									{
										OFString uidImagen;
										FileRecord->findAndGetOFString(DCM_ReferencedSOPInstanceUIDInFile, uidImagen);

										wxString wxStr = FROMPATH(tmpString);
										wxString separador = wxFileName::GetPathSeparator();
										wxString barra = wxT('\\');
										wxStr.Replace(barra,separador);
										std::ostringstream ostr;
										tmpString = TOPATH(wxStr);
										ostr << basePath << separador.ToUTF8();
										ostr << tmpString;

										pModelo->InsertarImagen(uidSerie.c_str(),uidImagen.c_str(),ostr.str());
									}
								}
							}
						}
					}
				}
				return true;
			} else {
				return false;
			}
		}

		bool PACSController::findAndGetTagFromFile(const std::string& ruta, unsigned short group, unsigned short element, std::string& value)
		{

			DcmFileFormat dcm;
			OFCondition   cond;
			OFString      ofval;
			bool          found = false;
			cond = dcm.loadFile(ruta.c_str());
			if (cond.good()) {
				cond = dcm.getDataset()->findAndGetOFString(DcmTagKey(group, element), ofval);
			}
			if (cond.good()) {
				value = ofval.c_str();
				found = true;
			}
			return found;
		}

		bool PACSController::EsDICOM(const std::string& path, bool accept_dicomdir, bool accept_dicomfile)
		{
			bool valido = false;
			char magic[5] = { 'X', 'X', 'X', 'X', 0 };
			std::string emptystr;
			std::fstream dcmfile;

			dcmfile.open (path.c_str(), std::ios::in | std::ios::binary);

			if (!dcmfile.eof() && dcmfile.good()) {
				dcmfile.seekp (128, std::ios::beg);
			}
			if (!dcmfile.eof() && dcmfile.good() ) {
				dcmfile.read (magic, 4);
			}
			if (!dcmfile.eof() && dcmfile.good()) {
				if ( magic[0] == 'D' && magic[1] == 'I' && magic[2] == 'C' && magic[3] == 'M' ) {
					if (!accept_dicomdir || !accept_dicomfile) {
						// Leemos el tag 0004 0x1130 => Si existe es DICOMDIR
						bool has_dcmdirtag = false;
						GIL::DICOM::TipoJerarquia base;

						if (GIL::DICOM::PACSController::Instance()->findAndGetTagFromFile(path, 0x0004, 0x1200, emptystr))
						{
							has_dcmdirtag = true;
						}

						if (has_dcmdirtag) {
							if (accept_dicomdir) {
								valido = true;
							}
						}
						else {
							if (accept_dicomfile) {
								valido = true;
							}
						}
					}
					else {
						valido = true;
					}
				}
			}

			dcmfile.close();
			return valido;
		}

		bool PACSController::ObtenerEstudio(void* connectionKey, const std::string& serverId, const std::string& uidEstudio, const std::string& AccNumber, IModeloDicom* pModelo, GNC::IProxyNotificadorProgreso* pNotificador)
		{
			GTRACE("PACSController::ObtenerEstudio( " << serverId.c_str() << ", " << uidEstudio.c_str() << ", " << uidSerie.c_str());

			if (pModelo == NULL) {
				return false;
			}
			DicomServerList* listaServidores = DicomServerList::Instance();

			DcmDataset query;
			DcmElement* e = NULL;

			e = newDicomElement(DCM_SpecificCharacterSet);
			e->putString("ISO_IR 192");
			query.insert(e);

			e = newDicomElement(DCM_PatientID);
			query.insert(e);

			e = newDicomElement(DCM_PatientName);
			query.insert(e);

			e = newDicomElement(DCM_StudyDate);
			query.insert(e);

			e = newDicomElement(DCM_StudyTime);
			query.insert(e);

			e = newDicomElement(DCM_StudyID);
			query.insert(e);

			e = newDicomElement(DCM_StudyInstanceUID);
			if (uidEstudio != "") {				
				e->putString(uidEstudio.c_str());				
			}
			query.insert(e);

			e = newDicomElement(DCM_AccessionNumber);
			if (AccNumber != ""){
				e->putString(AccNumber.c_str());				
			}
			query.insert(e);

			std::string localAET = GNC::Entorno::Instance()->GetDicomLocalAET();

			DicomServer* server = listaServidores->GetServer(serverId);
			if (server->GetRetrieveWithMove()) {
				e = newDicomElement(DCM_QueryRetrieveLevel);
				e->putString("STUDY");
				query.insert(e);

				int puerto = 11112;
				{
					GNC::GCS::ConfigurationController::Instance()->readIntGeneral("/GinkgoCore/PACS/Local", "Puerto", puerto, 11112);
				}

				if (server) {
					LOG_INFO("C-MOVE", "Obteniendo estudio del PACS " << serverId << ": " << server->AET << "@" << server->HostName << ":" << server->Port << " PDU=" << server->PDU << ", TLS=" << server->useTLS << ",  User = " << server->pacsUser << "Method=C-MOVE");
				}

				NetClient<MoveAssociation> a(connectionKey, "C-MOVE", pNotificador);

				a.SetAcceptorPort((unsigned short) puerto);
				a.SetRole(Association::RT_AcceptorRequestor);
				a.SetModelo(pModelo);
				a.SetCallbackHandler(this);

				if (server->useTLS) {
					a.SetTLS(server->GetCertificate(), server->GetPrivateKey(), server->GetverifyCredentials());
				}
				if (server->GetPACSUser() != "") {
					a.SetUserPass(server->GetPACSUser(), server->GetPACSPass());
				}
				a.QueryServer(&query, server, pModelo, localAET, CT_MoveSerie);

			}
			else {

				if (server) {
					LOG_INFO("C-GET", "Obteniendo estudio del PACS " << serverId << ": " << server->AET << "@" << server->HostName << ":" << server->Port << " PDU=" << server->PDU << ", TLS=" << server->useTLS << ",  User = " << server->pacsUser << ", Method=C-GET");
				}

				//we have to know the series uids and obtain series by series
				e = newDicomElement(DCM_QueryRetrieveLevel);
				e->putString("SERIES");
				query.insert(e);

				e = newDicomElement(DCM_StudyInstanceUID);
				e->putString(uidEstudio.c_str());
				query.insert(e);

				e = newDicomElement(DCM_SeriesInstanceUID);
				query.insert(e);

				e = newDicomElement(DCM_Modality);
				query.insert(e);

				NetClient<FindAssociation> f(connectionKey, "C-GET/FIND", pNotificador);
				f.SetMaxResults(-1);

				std::string localAET = GNC::Entorno::Instance()->GetDicomLocalAET();

				DicomServer* server = listaServidores->GetServer(serverId);
				if (server->useTLS) {
					f.SetTLS(server->GetCertificate(), server->GetPrivateKey(), server->GetverifyCredentials());
				}
				if (server->GetPACSUser() != "") {
					f.SetUserPass(server->GetPACSUser(), server->GetPACSPass());
				}

				f.QueryServer(&query, server, pModelo, localAET, CT_None);

				if (f.Stopped()){
					return false;
				}
				DcmStack* stack = f.GetResultStack();

				unsigned int numResults = 0;

				numResults = stack->card();

				for (unsigned int i = 0; i < numResults; i++) {

					if (stack->elem(i)->ident() == EVR_dataset) {
						DcmDataset* dset = dynamic_cast<DcmDataset*>(stack->elem(i));
						if (dset) {
							OFString OFSSeriesInstanceUID;
							OFString OFSeriesModality;
							if ( dset->findAndGetOFString(DCM_SeriesInstanceUID, OFSSeriesInstanceUID).good() && dset->findAndGetOFString(DCM_Modality, OFSeriesModality).good() )
							{
								std::string seriesInstanceUID(OFSSeriesInstanceUID.c_str());
								std::string seriesModality(OFSeriesModality.c_str());

								ObtenerSerie(connectionKey, serverId, seriesInstanceUID, pModelo, pNotificador, seriesModality);
							}
						}

					}
				}

				query.clear();				

			}

			return true;

		}

		bool PACSController::ObtenerSerie(void* connectionKey, const std::string& serverId, const std::string& uidSerie, IModeloDicom* pModelo, GNC::IProxyNotificadorProgreso* pNotificador, const std::string& seriesModality)
		{
			GTRACE("PACSController::ObtenerSerie( " << serverId.c_str() << ", " << uidEstudio.c_str() << ", " << uidSerie.c_str());			

			if (pModelo == NULL) {
				return false;
			}
			DicomServerList* listaServidores = DicomServerList::Instance();

			DicomServer* server = listaServidores->GetServer(serverId);
			std::string modality = seriesModality;

			DcmElement* e = NULL;
			DcmDataset query;

			unsigned int numResults = 0;

			if (!server->GetRetrieveWithMove() && modality.empty()) { // We have to find series modality
				e = newDicomElement(DCM_SpecificCharacterSet);
				e->putString("ISO_IR 192");
				query.insert(e);

				e = newDicomElement(DCM_QueryRetrieveLevel);
				e->putString("SERIES");
				query.insert(e);

				e = newDicomElement(DCM_SeriesInstanceUID);
				e->putString(uidSerie.c_str());
				query.insert(e);

				e = newDicomElement(DCM_Modality);
				query.insert(e);

				NetClient<FindAssociation> f(connectionKey, "C-GET/FIND", pNotificador);
				f.SetMaxResults(-1);

				std::string localAET = GNC::Entorno::Instance()->GetDicomLocalAET();

				DicomServer* server = listaServidores->GetServer(serverId);
				if (server->useTLS) {
					f.SetTLS(server->GetCertificate(), server->GetPrivateKey(), server->GetverifyCredentials());
				}
				if (server->GetPACSUser() != "") {
					f.SetUserPass(server->GetPACSUser(), server->GetPACSPass());
				}

				if (!f.QueryServer(&query, server, pModelo, localAET, CT_None)) 
				{
					return false;
				}

				if (f.Stopped()){
					return false;
				}
				DcmStack* stack = f.GetResultStack();

				OFString OFSeriesModality;

				numResults = stack->card();
				if (numResults >1) {
					LOG_WARN("C-GET", "Obteniendo serie del PACS " << serverId << ": " << server->AET << "@" << server->HostName << ":" << server->Port << " PDU=" << server->PDU << ", TLS=" << server->useTLS << ",  User = " << server->pacsUser << ", there have a Series with more than one modality, we are going to process only first modality");
				}

				for (unsigned int i = 0; i < numResults; i++) {

					if (stack->elem(i)->ident() == EVR_dataset) {
						DcmDataset* dset = dynamic_cast<DcmDataset*>(stack->elem(i));
						if (dset) {

							if ( dset->findAndGetOFString(DCM_Modality, OFSeriesModality).good() && OFSeriesModality.size() > 0 )
							{
								modality = OFSeriesModality.c_str();
								break;
							}
						}

					}
				}

				query.clear();
			}// end query modality


			e = newDicomElement(DCM_SpecificCharacterSet);
			e->putString("ISO_IR 192");
			query.insert(e);			

			e = newDicomElement(DCM_QueryRetrieveLevel);
			e->putString("SERIES");
			query.insert(e);

			e = newDicomElement(DCM_StudyInstanceUID);
			query.insert(e);

			e = newDicomElement(DCM_SeriesInstanceUID);
			e->putString(uidSerie.c_str());
			query.insert(e);

			e = newDicomElement(DCM_Modality);
			query.insert(e);

			e = newDicomElement(DCM_SeriesNumber);
			query.insert(e);

			std::string localAET = GNC::Entorno::Instance()->GetDicomLocalAET();

			if (server->GetRetrieveWithMove()) {

				if (server) {
					LOG_INFO("C-MOVE", "Obteniendo serie del PACS " << serverId << ": " << server->AET << "@" << server->HostName << ":" << server->Port << " PDU=" << server->PDU << ", TLS=" << server->useTLS << ",  User = " << server->pacsUser << ", Method=C-MOVE");
				}

				int puerto = 11112;
				{
					GNC::GCS::ConfigurationController::Instance()->readIntGeneral("/GinkgoCore/PACS/Local", "Puerto", puerto, 11112);
				}

				if (server) {
					LOG_INFO("C-MOVE", "Obteniendo estudio del PACS " << serverId << ": " << server->AET << "@" << server->HostName << ":" << server->Port << " PDU=" << server->PDU << ", TLS=" << server->useTLS << ",  User = " << server->pacsUser << ", Method=C-MOVE");
				}

				NetClient<MoveAssociation> a(connectionKey, "C-MOVE", pNotificador);
				a.SetAcceptorPort((unsigned short) puerto);
				a.SetRole(Association::RT_AcceptorRequestor);
				a.SetModelo(pModelo);

				a.SetCallbackHandler(this);

				if (server->useTLS) {
					a.SetTLS(server->GetCertificate(), server->GetPrivateKey(), server->GetverifyCredentials());
				}
				if (server->GetPACSUser() != "") {
					a.SetUserPass(server->GetPACSUser(), server->GetPACSPass());
				}
				a.QueryServer(&query, server, pModelo, localAET, CT_MoveSerie);

			}
			else {
				if (server) {
					LOG_INFO("C-GET", "Obteniendo serie del PACS " << serverId << ": " << server->AET << "@" << server->HostName << ":" << server->Port << " PDU=" << server->PDU << ", TLS=" << server->useTLS << ",  User = " << server->pacsUser << ", Method=C-GET");
				}

				NetClient<GetAssociation> a(connectionKey, "C-GET", pNotificador);
				a.SetMaxResults(-1);
				a.SetWellKnownNumResults(numResults);
				a.SetStorageSOPClasses(GIL::DICOM::Conformance::GetModalities().GetSupportedSOPClassUIDs(modality));
				a.SetModelo(pModelo);

				a.SetCallbackHandler(this);

				if (server->useTLS) {
					a.SetTLS(server->GetCertificate(), server->GetPrivateKey(), server->GetverifyCredentials());
				}
				if (server->GetPACSUser() != "") {
					a.SetUserPass(server->GetPACSUser(), server->GetPACSPass());
				}
				if (!a.QueryServer(&query, server, pModelo, localAET, CT_MoveSerie)) {
					return false;
				}

			}

			return true;

		}

		bool PACSController::ObtenerImagen(void* connectionKey, const std::string& serverId, const std::string& uidImagen, IModeloDicom* pModelo,GNC::IProxyNotificadorProgreso* pNotificador)
		{
			GTRACE("PACSController::ObtenerImagen( " << serverId.c_str() << ", " << uidEstudio.c_str() << ", " << uidSerie.c_str() << uidImagen.c_stR() << ")");

			if (pModelo == NULL) {
				return false;
			}
			DicomServerList* listaServidores = DicomServerList::Instance();

			DcmDataset query;
			DcmElement* e = NULL;

			e = newDicomElement(DCM_SpecificCharacterSet);
			e->putString("ISO_IR 192");
			query.insert(e);

			e = newDicomElement(DCM_QueryRetrieveLevel);
			e->putString("IMAGE");
			query.insert(e);

			e = newDicomElement(DCM_SOPInstanceUID);
			if (!uidImagen.empty()) {
				e->putString(uidImagen.c_str());
			}
			query.insert(e);

			e = newDicomElement(DCM_InstanceNumber);
			query.insert(e);

			std::string localAET = GNC::Entorno::Instance()->GetDicomLocalAET();

			DicomServer* server = listaServidores->GetServer(serverId);
			if (server->GetRetrieveWithMove()) {

				if (server) {
					LOG_INFO("C-MOVE", "Obteniendo imagen del PACS " << serverId << ": " << server->AET << "@" << server->HostName << ":" << server->Port << " PDU=" << server->PDU << ", TLS=" << server->useTLS << ",  User = " << server->pacsUser << ", Method=C-MOVE");
				}

				NetClient<MoveAssociation> a(connectionKey, "C-MOVE", pNotificador);
				a.SetModelo(pModelo);

				a.SetCallbackHandler(this);

				if (server->useTLS) {
					a.SetTLS(server->GetCertificate(), server->GetPrivateKey(), server->GetverifyCredentials());
				}
				if (server->GetPACSUser() != "") {
					a.SetUserPass(server->GetPACSUser(), server->GetPACSPass());
				}
				a.QueryServer(&query, server, pModelo, localAET, CT_MoveImagen);

			}
			else {

				if (server) {
					LOG_INFO("C-GET", "Obteniendo imagen del PACS " << serverId << ": " << server->AET << "@" << server->HostName << ":" << server->Port << " PDU=" << server->PDU << ", TLS=" << server->useTLS << ",  User = " << server->pacsUser << ", Method=C-GET");
				}

				NetClient<GetAssociation> a(connectionKey, "C-GET", pNotificador);
				a.SetModelo(pModelo);

				a.SetCallbackHandler(this);

				if (server->useTLS) {
					a.SetTLS(server->GetCertificate(), server->GetPrivateKey(), server->GetverifyCredentials());
				}
				if (server->GetPACSUser() != "") {
					a.SetUserPass(server->GetPACSUser(), server->GetPACSPass());
				}
				a.QueryServer(&query, server, pModelo, localAET, CT_MoveImagen);
			}
			return true;
		}

		bool PACSController::QueryRetrieve(
			void* connectionKey,
			const std::string& serverId,
			IModeloDicom* pModelo,
			const GIL::DICOM::TipoJerarquia& base,
			GNC::IProxyNotificadorProgreso* pNotificador)
		{
			std::string strTmp;
			if(base.getTag("0008|0052", strTmp)) {
				if (strTmp == "STUDY") {
					std::string uidEstudio, accNumber;
					base.getTag("0020|000d", uidEstudio);
					base.getTag("0008|0050", accNumber);
					if (accNumber == "" && uidEstudio == "") {
						throw GIL::DICOM::PACSException(_Std("You have to specify uid study or accession number"));
					}

					ObtenerEstudio(connectionKey, serverId, uidEstudio, accNumber, pModelo, pNotificador);
				} else if (strTmp == "SERIES") {
					std::string uidSeries;
					if (!base.getTag("0020|000e",uidSeries)) {
						throw GIL::DICOM::PACSException(_Std("You have to specify uid series"));
					}
					ObtenerSerie(connectionKey, serverId, uidSeries, pModelo, pNotificador);
				} else if (strTmp == "IMAGE")  {
					std::string uidImage;
					if (!base.getTag("0008|0018",uidImage)) {
						throw GIL::DICOM::PACSException(_Std("You have to specify uid series"));
					}
					ObtenerImagen(connectionKey, serverId, uidImage, pModelo, pNotificador);
				} else  {
					throw GIL::DICOM::PACSException(_Std("Unknown query/retrieve level"));
				}
			} else {
				throw GIL::DICOM::PACSException(_Std("Query retrieve level not specified"));
			}

			return true;
		}

		void PACSController::StartService()
		{
			StopService();

			int localPort = 0;
			std::string localAET;
			{
				GNC::GCS::ConfigurationController::Instance()->readIntGeneral("/GinkgoCore/PACS/Local", "Puerto", localPort, 11112);
				GNC::GCS::ConfigurationController::Instance()->readStringGeneral("/GinkgoCore/PACS/Local", "AET", localAET, "GINKGO_%IP4");
			}

			this->m_pServiceInstance = new GIL::DICOM::Service(_Std("PACS-Service"));
			this->m_pServiceInstance->SetAcceptorPort(localPort);
			this->m_pServiceInstance->SetLocalAET(localAET);
			this->m_pServiceInstance->Start();

		}

		void PACSController::StopService()
		{
			if (this->m_pServiceInstance != NULL) {
				this->m_pServiceInstance->Stop();
				delete this->m_pServiceInstance;
				this->m_pServiceInstance = NULL;
			}
		}


		void PACSController::GetConnection(void* connectionKey)
		{
			GIL::DICOM::DCMTK::Network::Instance(connectionKey);
		}

		void PACSController::ReleaseConnection(void* connectionKey)
		{
			GIL::DICOM::DCMTK::Network::FreeInstance(connectionKey);
		}

		void PACSController::SubirArchivos(void* /*connectionKey*/, const std::string& serverId, const std::vector<std::string> &pathsSubida, GNC::IProxyNotificadorProgreso* pNotificador, TipoTransferSyntaxEnvio transferSyntax)
		{

			GTRACE("PACSController::SubirArchivos( " << serverId.c_str() << ", [...]");
			DicomServerList* listaServidores = DicomServerList::Instance();
			std::string localAET = GNC::Entorno::Instance()->GetDicomLocalAET();
			GIL::DICOM::DicomStoreAssociation asociacion;
			DicomServer* server = listaServidores->GetServer(serverId);
			if (server) {
				LOG_INFO("PACS-STORE", "Enviando al PACS " << serverId << ": " << server->AET << "@" << server->HostName << ":" << server->Port << " PDU=" << server->PDU << ", TLS=" << server->useTLS << ",  User = " << server->pacsUser);
			}
			if (server->useTLS) {				
				asociacion.SetTLS(server->GetCertificate(), server->GetPrivateKey(), server->GetverifyCredentials());
			}
			if (server->GetPACSUser() != "") {
				asociacion.SetUserPass(server->GetPACSUser(), server->GetPACSPass());
			}
			asociacion.Store(pathsSubida,listaServidores->GetServer(serverId), localAET, pNotificador, transferSyntax);

		}

		std::string GetMD5(const std::string& cadena)
		{
			unsigned char salida[MD5_DIGEST_LENGTH];
			MD5((const unsigned char*)cadena.c_str(), cadena.size(),salida);

			std::ostringstream cadenaMD5;
			cadenaMD5.setf ( std::ios::hex, std::ios::basefield );
			for(int i = 0; i<MD5_DIGEST_LENGTH; i++)
			{
				cadenaMD5 << (int)salida[i];
			}
			return cadenaMD5.str();
		}

		bool PACSController::GetRutaSerie(const std::string& uidPaciente, const std::string& uidEstudio, const std::string& uidSerie, std::string& ruta, bool crearDirectorios,bool temporal)
		{
			bool correcto = true;
			//md5 de uidpaciente, estudio y serie
			std::stringstream concat;
			if(uidPaciente != "")
				concat << uidPaciente;
			else
				concat << _("Unknown");
			if(uidEstudio != "")
				concat << uidEstudio;
			else
				concat << _("Unknown");
			if(uidSerie != "")
				concat << uidSerie;
			else
				concat << _("Unknown");
			std::string cadena = concat.str();
			std::string resumen = GetMD5(cadena);

			wxString dirRaiz, dirSerie;
			dirRaiz = FROMPATH(GNC::Entorno::Instance()->GetGinkgoDicomDir());
			if (!wxDir::Exists(dirRaiz) && crearDirectorios) {
				correcto = correcto && wxFileName::Mkdir(dirRaiz, 0700);
			}

			if(temporal) {
				dirRaiz = dirRaiz + wxFileName::GetPathSeparator(wxPATH_NATIVE) + wxT("TEMP");
				if (!wxDir::Exists(dirRaiz)) {
					correcto = correcto && wxFileName::Mkdir(dirRaiz, 0700);
				}
			}

			dirSerie= dirRaiz + wxFileName::GetPathSeparator(wxPATH_NATIVE) + FROMPATH(resumen);
			if (!wxDir::Exists(dirSerie) && crearDirectorios) {
				correcto = correcto && wxFileName::Mkdir(dirSerie, 0700);
			}

			ruta = TOPATH(dirSerie);
			return correcto;
		}

		bool PACSController::GetRutaImagen(const std::string& uidPaciente, const std::string& uidEstudio, const std::string& uidSerie, const std::string& uidImagen, std::string& ruta, bool crearDirectorios)
		{
			bool correcto = GetRutaSerie(uidPaciente,uidEstudio,uidSerie,ruta, crearDirectorios, false);
			std::ostringstream ostr;
			ostr << ruta << (char)wxFileName::GetPathSeparator(wxPATH_NATIVE)<<GetMD5(uidImagen)<<".dcm";
			ruta = ostr.str();
			return correcto;
		}

		bool PACSController::GetRutaImagenTemp(const std::string& uidPaciente, const std::string& uidEstudio, const std::string& uidSerie, const std::string& uidImagen, std::string& ruta, bool crearDirectorios)
		{
			bool correcto = GetRutaSerie(uidPaciente,uidEstudio,uidSerie,ruta, crearDirectorios, true);
			std::ostringstream ostr;
			ostr << ruta << (char)wxFileName::GetPathSeparator(wxPATH_NATIVE)<<GetMD5(uidImagen)<<".dcm";
			ruta = ostr.str();
			return correcto;
		}

		void PACSController::DumpFileElements(const std::string& rutaFichero, IInspectCallBack* callback)
		{
			DcmFileFormat fileformat;
			if (rutaFichero.empty() || callback == NULL) {
				return;
			}
			OFCondition status = fileformat.loadFile(rutaFichero.c_str());
			if (status.good()) {

				DcmDataset* ds = fileformat.getDataset();

				DcmDataDictionary& globalDataDict = dcmDataDict.wrlock();
				DcmHashDictIterator iter(globalDataDict.normalBegin());
				DcmHashDictIterator end(globalDataDict.normalEnd());

				std::string keyStr;
				std::string descStr;
				std::string valStr;
				for (; iter != end; ++iter) {
					DcmTagKey tagkey = (*iter)->getKey();
					keyStr = tagkey.toString().c_str();
					descStr = (*iter)->getTagName();
					OFString val;
					OFCondition c = ds->findAndGetOFString(tagkey, val);
					if (c.good()) {
						valStr = val.c_str();
						callback->Inspect(keyStr, descStr, valStr);
					}
				}
				dcmDataDict.unlock();
			}
		}


		//region "Métodos de Dicomización"
		/*
		void PACSController::CrearInstanciaDeImportacion(void* ref) {
		m_ListaInstanciasDicomImg2DCM[ref] = new IDICOMImg2DCM();
	 }

	 void PACSController::LiberarInstanciaDeImportacion(void* ref) {
	 ListaInstanciasImg2DCMType::iterator it = m_ListaInstanciasDicomImg2DCM.find(ref);
	 if (it != m_ListaInstanciasDicomImg2DCM.end()) {
	 delete (*it).second;
	 m_ListaInstanciasDicomImg2DCM.erase(it);
	 }
	 }
	 */

		IDICOMImg2DCM* PACSController::CrearInstanciaDeImportacion() {
			return new DICOMImg2DCM();
		}

		void PACSController::LiberarInstanciaDeImportacion(IDICOMImg2DCM* instancia) {
			DICOMImg2DCM* i = dynamic_cast<DICOMImg2DCM*>(instancia);
			if (i != NULL) {
				delete i;
			}
		}
		//endregion

		//region "Métodos de carga/almacenamiento de DICOMS"
		IDICOMManager* PACSController::CrearInstanciaDeDICOMManager() {
			return new DICOMManager();
		}

		void PACSController::LiberarInstanciaDeDICOMManager(IDICOMManager* instancia) {
			DICOMManager* i = dynamic_cast<DICOMManager*>(instancia);
			if (i != NULL) {
				delete i;
			}
		}

		void  BorrarDirTemp(wxString dirPath)
		{
			//borra en cascada
			if(!wxRmdir(dirPath)){
				//vaciar
				wxDir dir;
				if (dir.Open(dirPath)) {
					wxString fileName;
					bool cont = dir.GetFirst(&fileName);
					while (cont) {
						fileName=dir.GetName()+ wxFileName::GetPathSeparator(wxPATH_NATIVE) +fileName;
						if(wxDir::Exists(fileName)){
							BorrarDirTemp(fileName);
						}else{
							wxRemoveFile(fileName);
						}
						cont = dir.GetNext(&fileName);
					}
				}
			}
			if(wxDir::Exists(dirPath)){
				wxRmdir(dirPath);
			}
		}

		void PACSController::PurgarDirectorioTemporal()
		{
			wxString dirTemp;
			dirTemp = FROMPATH(GNC::Entorno::Instance()->GetGinkgoDicomDir());
			if (!wxDir::Exists(dirTemp)) {
				return;
			}

			dirTemp = dirTemp + wxFileName::GetPathSeparator(wxPATH_NATIVE) + wxT("TEMP");
			if (!wxDir::Exists(dirTemp)) {
				return;
			}

			//borrar en profundidad
			BorrarDirTemp(dirTemp);
		}
		//endregion
	};
};
