/*
*
*  $Id: dialogoadquisicion.cpp 3763 2011-04-25 12:07:55Z carlos $
*  Ginkgo CADx Project
*
*  Copyright 2008-10 MetaEmotion S.L. All rights reserved.
*  http://ginkgo-cadx.com
*
*  This file is licensed under LGPL v3 license.
*  See License.txt for details
*
*
*/
//#define _GINKGO_TRACE

#include <api/globals.h>
#include <api/ilock.h>
#include <set>

#include <wx/wx.h>
#include <wx/animate.h>
#include <wx/imaglist.h>
#include <wx/dnd.h>
#include <wx/config.h>
#include <wx/ginkgostyle/ginkgostyle.h>
#include <wx/propiedades/wxpropiedades.h>

#include "dialogoadquisicion.h"

#include <resources/ginkgoresourcemanager.h>
#include <main/entorno.h>
#include <main/controllers/controladorcomandos.h>
#include <main/controllers/controladorpermisos.h>
#include <main/controllers/controladorlog.h>
#include <main/controllers/controladorhistorial.h>
#include <main/controllers/controladoreventos.h>
#include <eventos/eventosginkgo.h>
#include <api/icontextoestudio.h>

#define SIZE_ICONOS 16

#define ICONO_ESTUDIO 0
#define ICONO_SERIE 1
#define ICONO_HOMBRE 2
#define ICONO_MUJER 3
#define ICONO_OTRO 4
#define SIZE_ICONOS 16

#define COL_PACS_ID 0
#define COL_PACS_DESCRIPTION 1
#define COL_PACS_MODE 2

namespace Descargas {
	enum Columnas
	{
		Col_PatientInfo,
		Col_Description,
		Col_Progress,
		Col_Status,
	};

	enum Estado
	{
		Parada,
		Iniciada,
		Terminada,
		Erronea
	};

	class DescargasGridTable;

	class Descarga : public GNC::GCS::IObservador
	{
	public:
		Descarga(DescargasGridTable* pGridTable, const std::string& s, const std::string& pd,const std::string& des, const std::string& uid, const std::string& mod, bool seriesMode)
		{
			m_pGridTable = pGridTable;
			Server = s;
			PatientDescription = pd;
			Description = des;
			UID = uid;
			Modality = mod;
			SeriesMode = seriesMode;
			StatusId = Parada;
			m_pComando = NULL;

			GNC::GCS::ControladorEventos::Instance()->Registrar(this, GNC::GCS::Eventos::EventoProgresoComando());

			DoStop();
		}

		~Descarga()
		{
			GNC::Entorno::Instance()->GetControladorComandos()->AbortarComandosDeOwner(this);
			GNC::GCS::ControladorEventos::Instance()->DesRegistrar(this, GNC::GCS::Eventos::EventoProgresoComando());
			DoStop();
		}

		std::string Server;

		std::string PatientDescription;
		std::string Description;
		std::string Modality;
		std::string UID;
		bool SeriesMode;

		float       Progreso;
		std::string Status;
		Estado      StatusId;

		GADAPI::ComandoPACS* m_pComando;
		bool                 m_Abortar;

	private:
		void DoRun() {
			if (StatusId == Iniciada) {
				return;
			}
			if (m_pComando != NULL) {
				GNC::GCS::ControladorComandos::Instance()->AbortarComando(m_pComando, false);
				m_pComando = NULL;
			}

			GADAPI::ComandoPACSParams * pParams;
			if (SeriesMode) {
				pParams = new GADAPI::ComandoPACSParams(Server, UID, GADAPI::ComandoPACSParams::TA_Serie);
			} else {
				pParams = new GADAPI::ComandoPACSParams(Server, UID, GADAPI::ComandoPACSParams::TA_Estudio);
			}

			m_pComando = new GADAPI::ComandoPACS(pParams);
			GNC::Entorno::Instance()->GetControladorComandos()->ProcessAsync(_Std("Downloading from PACS..."), m_pComando, this);

			Status = _Std("Starting...");
			StatusId = Iniciada;

		}

		void DoStop() {
			if (StatusId == Parada || StatusId == Terminada || StatusId == Erronea) {
				return;
			}
			Progreso = 0.0f;
			m_Abortar = true;

			Status = _Std("Stopped");
			StatusId = Parada;
			if (m_pComando != NULL) {
				GNC::GCS::ControladorComandos::Instance()->AbortarComando(m_pComando, false);
				m_pComando = NULL;
				m_Abortar = true;
			}
		}

		virtual void ProcesarEvento(GNC::GCS::Eventos::IEvento *evt);

		void OnProgresoInicio();
		void OnProgresoProgreso(float progresoNormalizado, const std::string& texto);
		void OnProgresoFinalizacion();

		DescargasGridTable* m_pGridTable;

		friend class DescargasGridTable;
	};

	class ProgresoCellRenderer : public wxGridCellFloatRenderer
	{
	public:

		ProgresoCellRenderer(DescargasGridTable* table)
		{
			m_pGridTable = table;
		}

		~ProgresoCellRenderer()
		{
			;
		}

		virtual void Draw(wxGrid& grid,
			wxGridCellAttr& attr,
			wxDC& dc,
			const wxRect& cellRect,
			int row, int col,
			bool isSelected);

		virtual wxGridCellRenderer* Clone () const
		{
			return new ProgresoCellRenderer(m_pGridTable);
		}

	private:
		DescargasGridTable* m_pGridTable;
	};

	class DescargasGridTable : public wxGridTableBase
	{
	public:

		GNC::GUI::DialogoAdquisicion* pDlg;

		std::vector<Descarga*> m_Descargas;

		DescargasGridTable(GNC::GUI::DialogoAdquisicion* pParent)
		{
			pDlg = pParent;
		}

		~DescargasGridTable()
		{
			for(std::vector<Descarga*>::iterator it = m_Descargas.begin(); it!= m_Descargas.end(); ++it) {
				delete (*it);
			}
			m_Descargas.clear();
		}

		virtual int GetNumberRows()
		{
			return m_Descargas.size();
		}
		virtual int GetNumberCols()
		{
			return 4;
		}

		int FindDescarga(const std::string& uid)
		{
			int pos = -1;
			for (unsigned int i = 0; pos == -1 && i < m_Descargas.size(); i++) {
				if (m_Descargas[i]->UID == uid) {
					pos = (int)i;
				}
			}
			return pos;
		}

		virtual bool AddDescarga(const std::string& s, const std::string& pd,const std::string& sd, const std::string& su, const std::string& sm, bool initIfStopped, bool seriesMode)
		{


			if (s.empty() || su.empty()) {
				LOG_ERROR("GUI/Adquisition", "Empty mandatory values (serverId | serieInstanceUID) while adding serie to download queue");
				return false;
			}
			int pos = FindDescarga(su);
			if (pos != -1) {
				if (m_Descargas[pos]->StatusId != Descargas::Iniciada && initIfStopped) {
					GetView()->SelectRow(pos);
					IniciarDescarga(pos);

					return true;
				} else {
					return false;
				}
			}

			m_Descargas.push_back(new Descarga(this ,s, pd, sd, su, sm, seriesMode));
			wxGridTableMessage msg( this, wxGRIDTABLE_NOTIFY_ROWS_INSERTED,
				m_Descargas.size() - 1, 1);
			GetView()->Freeze();
			GetView()->ProcessTableMessage( msg );
			pos = FindDescarga(su);
			GetView()->SelectRow(pos);
			GetView()->Thaw();
			IniciarDescarga(pos);

			return true;
		}

		virtual void RemoveDescarga(Descarga* d)
		{
			RemoveDescarga(FindDescarga(d->UID));
		}

		virtual void RemoveDescarga(const std::string& uid)
		{
			RemoveDescarga(FindDescarga(uid));

		}

		virtual bool DeleteRows( size_t pos = 0, size_t numRows = 1 )
		{
			bool removed = false;
			for (unsigned int i = 0; i < numRows; i++) {
				RemoveDescarga(pos);
				removed = true;
			}
			return removed;
		}

		virtual void RemoveDescarga(int pos)
		{
			if (pos < 0) {
				return;
			}
			else {
				delete m_Descargas[pos];
				m_Descargas.erase(m_Descargas.begin() + pos);
				wxGridTableMessage msg( this, wxGRIDTABLE_NOTIFY_ROWS_DELETED, pos, 1);
				GetView()->Freeze();
				GetView()->ProcessTableMessage( msg );
				pDlg->UpdateButtons();
				GetView()->Thaw();
			}
		}

		virtual void SetProgresoDescarga(int pos, float p)
		{
			if (pos >= 0 && pos < (int)m_Descargas.size()) {
				Descarga* d = m_Descargas[pos];
				d->Progreso = p;
				wxGridTableMessage msg( this, wxGRIDTABLE_REQUEST_VIEW_GET_VALUES, pos, 1);
				GetView()->Freeze();
				GetView()->ProcessTableMessage( msg );
				GetView()->Thaw();
			}

		}

		virtual void SetEstadoDescarga(int pos, const std::string& text)
		{
			if (pos >= 0 && pos < (int)m_Descargas.size()) {
				Descarga* d = m_Descargas[pos];
				d->Status = text;
				wxGridTableMessage msg( this, wxGRIDTABLE_REQUEST_VIEW_GET_VALUES, pos, 1);
				GetView()->Freeze();
				GetView()->ProcessTableMessage( msg );
				GetView()->Thaw();
			}

		}

		virtual void IniciarDescarga(int pos)
		{
			if (pos >= 0 && pos < (int)m_Descargas.size()) {
				Descarga* d = m_Descargas[pos];
				d->DoRun();
				wxGridTableMessage msg( this, wxGRIDTABLE_REQUEST_VIEW_GET_VALUES, pos, 1);
				GetView()->Freeze();
				GetView()->ProcessTableMessage( msg );
				pDlg->UpdateButtons();
				GetView()->Thaw();
			}

		}

		virtual void DetenerDescarga(int pos)
		{
			if (pos >= 0 && pos < (int)m_Descargas.size()) {
				Descarga* d = m_Descargas[pos];
				d->DoStop();
				wxGridTableMessage msg( this, wxGRIDTABLE_REQUEST_VIEW_GET_VALUES, pos, 1);
				GetView()->Freeze();
				GetView()->ProcessTableMessage( msg );
				pDlg->UpdateButtons();
				GetView()->Thaw();
			}

		}

		virtual Descarga const * GetDescarga(int i) const
		{
			if (i < 0 || i > (int) m_Descargas.size()) {
				return NULL;
			}
			else {
				return m_Descargas[i];
			}
		}


		virtual bool IsEmptyCell( int row, int /*col*/ )
		{
			if (row >= (int)m_Descargas.size()) {
				return true;
			}
			else {
				return false;
			}
		}
		virtual wxString GetValue( int row, int col )
		{
			const Descarga& dr = *(m_Descargas[row]);

			switch ( col )
			{
			case Col_PatientInfo:
				return wxString::FromUTF8( (dr.PatientDescription).c_str() );

			case Col_Description:
				return wxString::FromUTF8( (dr.Description).c_str() );

			case Col_Progress:
				return wxString::Format(_T("%f"), 100.0f * dr.Progreso);

			case Col_Status:
				return wxString::FromUTF8( (dr.Status).c_str() );
			}

			return wxEmptyString;
		}
		virtual void SetValue( int /*row*/, int /*col*/, const wxString& /*value*/ )
		{
			//const Descarga& dr = *(m_Descargas[row]);
			;
		}

		virtual wxString GetColLabelValue( int col )
		{
			switch ( col )
			{
			case Col_PatientInfo:
				return _("Patient");

			case Col_Description:
				return _("Description");

			case Col_Progress:
				return _("Progress");

			case Col_Status:
				return _("Status");
			}

			wxFAIL_MSG(_("unknown column"));

			return wxEmptyString;
		}

		virtual wxString GetTypeName( int /*row*/, int col )
		{

			switch ( col )
			{
			case Col_PatientInfo:
				return wxGRID_VALUE_STRING;

			case Col_Description:
				return wxGRID_VALUE_STRING;

			case Col_Progress:
				return wxGRID_VALUE_FLOAT;

			case Col_Status:
				return wxGRID_VALUE_STRING;
			}

			wxFAIL_MSG(_("unknown column"));

			return wxEmptyString;
		}

		virtual bool CanGetValueAs( int /*row*/, int col, const wxString& typeName )
		{
			if ( typeName == wxGRID_VALUE_STRING )
			{
				return true;
			}
			else if ( typeName == wxGRID_VALUE_FLOAT )
			{
				if ( col == Col_Progress )
				{
					return true;
				}
			}
			return false;
		}

		virtual bool CanSetValueAs( int row, int col, const wxString& typeName )
		{
			return CanGetValueAs(row, col, typeName);
		}

		virtual double GetValueAsDouble( int row, int col )
		{
			const Descarga& dr = *(m_Descargas[row]);

			switch ( col )
			{
			case Col_Progress:
				return dr.Progreso;
			}
			return 0.0f;
		}

		virtual void SetValueAsDouble( int /*row*/, int /*col*/, double /*value*/ )
		{
			;
			/*
			BugsGridData& gd = gs_dataBugsGrid[row];

			switch ( col )
			{
			case Col_Priority:
			gd.prio = value;
			break;

			default:
			wxFAIL_MSG(_T("unexpected column"));
			}
			*/
		}

		virtual void OnChangeStatusDescarga(const Descarga& des) {
			int pos = FindDescarga(des.UID);
			wxArrayInt selectedRows = GetView()->GetSelectedRows();
			for (wxArrayInt::iterator it = selectedRows.begin(); it != selectedRows.end(); ++it) {
				if ((*it) == pos) {
					pDlg->UpdateButtons();
					break;
				}
			}
		}
	};


	void ProgresoCellRenderer::Draw(wxGrid& grid,
		wxGridCellAttr& attr,
		wxDC& dc,
		const wxRect& cellRect,
		int row, int col,
		bool isSelected)
	{
		wxGridCellRenderer::Draw(grid, attr, dc, cellRect, row, col, isSelected);

		float progreso = m_pGridTable->GetDescarga(row)->Progreso;
		if(progreso>1.0f)
		{
			progreso = 1.0f;
		}

		dc.SetPen( wxPen( wxColour(58, 143, 255), 1, wxSOLID ) );
		dc.SetBrush(*wxTRANSPARENT_BRUSH);
		wxRect cr = cellRect;
		//cr.Deflate(1);
		dc.DrawRectangle(cr.x, cr.y, cr.width, cr.height);

		cr.width = (int) ( (float)cr.width * progreso );

		dc.GradientFillLinear(cr, wxColour(224, 237, 255), wxColour(119, 169, 255), wxEAST);

		SetTextColoursAndFont(grid, attr, dc, isSelected);
		int func = dc.GetLogicalFunction();
		dc.SetLogicalFunction(wxXOR);

		wxCoord tx = 0, ty = 0;
		wxString str = wxString::Format(wxT("%.02f%%"), 100.0f * progreso);
		dc.GetTextExtent(str, &tx, &ty, NULL, NULL, NULL);

		dc.DrawText(str, cellRect.x + (cellRect.width >> 1)  - (tx >> 1), cellRect.y + (cellRect.height >> 1) - (ty >> 1) );
		dc.SetLogicalFunction(func);

	}

	void Descarga::ProcesarEvento(GNC::GCS::Eventos::IEvento *evt)
	{
		GNC::GCS::Eventos::EventoProgresoComando* pEvt = dynamic_cast<GNC::GCS::Eventos::EventoProgresoComando*> (evt);
		if (pEvt == NULL  || pEvt->GetComando() == NULL || pEvt->GetComando() != m_pComando) {
			return;
		}
		switch (pEvt->GetTipo()) {
	case GNC::GCS::Eventos::EventoProgresoComando::TEP_Iniciado:
		OnProgresoInicio();
		break;
	case GNC::GCS::Eventos::EventoProgresoComando::TEP_Progreso:
		OnProgresoProgreso(pEvt->GetProgresoNormalizado(), pEvt->GetTexto());
		break;
	case GNC::GCS::Eventos::EventoProgresoComando::TEP_Finalizado:
		OnProgresoFinalizacion();
		break;
	case GNC::GCS::Eventos::EventoProgresoComando::TEP_Unknown:
		break;
		}
		m_pGridTable->OnChangeStatusDescarga(*this);
	}

	void Descarga::OnProgresoInicio()
	{
		int did = m_pGridTable->FindDescarga(UID);
		m_pGridTable->SetEstadoDescarga(did, _Std("Started"));
	}

	void Descarga::OnProgresoProgreso(float progresoNormalizado, const std::string& texto)
	{
		int did = m_pGridTable->FindDescarga(UID);

		m_pGridTable->SetProgresoDescarga(did, progresoNormalizado);
		m_pGridTable->SetEstadoDescarga(did, texto);

	}

	void Descarga::OnProgresoFinalizacion()
	{
		int did = m_pGridTable->FindDescarga(UID);

		Progreso = 1.0f;

		if (!m_pComando->m_pPACSParams->m_pModelo->TieneImagenes()) {
			std::string errorMsg;

			if (m_pComando->m_pPACSParams->m_error.size() > 0) {
				errorMsg = _Std("Error: ") + m_pComando->m_pPACSParams->m_error;
			}
			else {
				errorMsg = _Std("Error: No images were downloaded");
			}
			Status = errorMsg;
			StatusId = Erronea;
			m_pGridTable->SetEstadoDescarga(did, errorMsg);
		}
		else {
			Status = _Std("Finished");
			StatusId = Terminada;
			m_pGridTable->SetEstadoDescarga(did, _Std("Finished"));
		}
		m_pComando = NULL;
	}
}


//---------------------------------------------------------------------------

//---------------HELPER DRAG & DROP ---------
class wxDropTargetDescargas: public wxTextDropTarget
{
public:
	wxDropTargetDescargas():wxTextDropTarget()
	{
		SetDefaultAction(wxDragCopy);
	}

	~wxDropTargetDescargas()
	{
	}

	virtual bool OnDropText(wxCoord /*x*/, wxCoord /*y*/, const wxString& text)
	{
		return text == wxT("");
	}
};

GNC::GUI::DialogoAdquisicion* GNC::GUI::DialogoAdquisicion::Instance()
{
	if (m_pInstance == NULL) {
		m_pInstance = new DialogoAdquisicion(GNC::Entorno::Instance()->GetVentanaRaiz());
	}
	return m_pInstance;
}

void GNC::GUI::DialogoAdquisicion::FreeInstance()
{
	if(m_pInstance!=NULL){
		m_pInstance->Close();
		m_pInstance = NULL;
	}
}

GNC::GUI::DialogoAdquisicion* GNC::GUI::DialogoAdquisicion::m_pInstance = NULL;


GNC::GUI::DialogoAdquisicion::DialogoAdquisicion(wxWindow* pParent) : DialogoAdquisicionBase(pParent)
{
	m_pComandoPACS = NULL;
	m_pImageList = new wxImageList(SIZE_ICONOS,SIZE_ICONOS,true);
	m_Mode = TD_SERIES;

	//animation
	m_pAnimation = new wxAnimationCtrl(m_pPanelPACS,wxID_ANY,GinkgoResourcesManager::BarraProgreso::GetIcoLoading());
	m_pAnimation->Hide();
	m_pSizerAnimation->Add(m_pAnimation, 0, wxALIGN_CENTER_VERTICAL|wxALL, 5);

	m_pBotonIniciarPararTask->Enable(false);
	m_pBotonDeleteTask->Enable(false);
	m_pBotonOpenTask->Enable(false);
	m_AutoAddSeries = false;
	m_AutoAddSeriesItem = NULL;

	m_pDescargasGridTable = new Descargas::DescargasGridTable(this);
	m_pGridDescargas->SetTable(m_pDescargasGridTable, false, wxGrid::wxGridSelectRows);

	wxGridCellAttr *attrRO = new wxGridCellAttr();
	wxGridCellAttr *attrPG = new wxGridCellAttr();

	attrRO->SetReadOnly();
	attrPG->SetReadOnly();
	attrPG->SetRenderer(new Descargas::ProgresoCellRenderer(m_pDescargasGridTable));

	for (int i = 0; i < m_pDescargasGridTable->GetNumberCols(); i++) {
		if (i != Descargas::Col_Progress) {
			m_pGridDescargas->SetColAttr(i, attrRO);
		}
		else {
			m_pGridDescargas->SetColAttr(i, attrPG);
		}
	}
	m_pGridDescargas->SetMargins(0, 0);
	//m_pGridDescargas->EnableScrolling(false, true);

	m_pGridDescargas->Connect(wxEVT_GRID_RANGE_SELECT, wxGridRangeSelectEventHandler(DialogoAdquisicion::OnGridSelect), NULL, this);
	m_pGridDescargas->Connect(wxEVT_GRID_CELL_LEFT_DCLICK, wxGridEventHandler(DialogoAdquisicion::OnGridDClick), NULL, this);
	m_pGridDescargas->Connect(wxEVT_GRID_CELL_RIGHT_CLICK, wxGridEventHandler(DialogoAdquisicion::OnGridMenu), NULL, this);
	m_pGridDescargas->Connect(wxEVT_KEY_DOWN, wxKeyEventHandler(DialogoAdquisicion::OnGridKeyDown), NULL, this);
	m_pGridDescargas->SetDropTarget(new wxDropTargetDescargas());

	m_pImageList->Add(GinkgoResourcesManager::PanelHistorial::GetIcoEstudio());
	m_pImageList->Add(GinkgoResourcesManager::PanelHistorial::GetIcoSerie());
	m_pImageList->Add(GinkgoResourcesManager::PanelHistorial::GetIcoHombre());
	m_pImageList->Add(GinkgoResourcesManager::PanelHistorial::GetIcoMujer());
	m_pImageList->Add(GinkgoResourcesManager::PanelHistorial::GetIcoOtro());
	m_pTreeListResultados->SetImageList(m_pImageList);

	m_pTreeListResultados->SetMainColumn(0);
	m_pTreeListResultados->Connect(wxEVT_COMMAND_TREE_SEL_CHANGED, wxTreeEventHandler( DialogoAdquisicion::OnTreeSelChanged ), NULL, this);
	m_pTreeListResultados->Connect(wxEVT_COMMAND_TREE_ITEM_ACTIVATED, wxTreeEventHandler( DialogoAdquisicion::OnTreeItemActivated ), NULL, this);
	m_pTreeListResultados->Connect(wxEVT_COMMAND_TREE_ITEM_MENU, wxTreeEventHandler( DialogoAdquisicion::OnTreeItemMenu ), NULL, this);
	m_pTreeListResultados->Connect(wxEVT_COMMAND_TREE_ITEM_EXPANDED, wxTreeEventHandler( DialogoAdquisicion::OnTreeItemExpanded ), NULL, this);
	m_pTreeListResultados->Connect(wxEVT_COMMAND_TREE_BEGIN_DRAG, wxTreeEventHandler( DialogoAdquisicion::OnTreeBeginDrag ), NULL, this);


	m_pPACSList->Connect(wxEVT_COMMAND_TREE_SEL_CHANGED, wxTreeEventHandler( DialogoAdquisicion::OnPACSChanged), NULL, this);

	/*
	for (int i = 0; i < 100; i++) {
	m_pDescargasGridTable->AddDescarga("server", "patientID", "patientName", "serieDescr", "serieIUID");
	m_pDescargasGridTable->SetProgresoDescarga( i, ((float) i) / 100.0f);
	}
	*/

	if(GNC::GCS::ControladorPermisos::Instance()->Get("core.pacs.limits", "patient_scope")) {
		m_HardCodedPatientSearch = true;
		bool first = true;
		std::ostringstream os;

		GNC::GCS::IEntorno::ListaModelosIntegracion& modelos = GNC::Entorno::Instance()->GetModelosIntegracion();

		for (GNC::GCS::IEntorno::ListaModelosIntegracion::iterator it = modelos.begin(); it != modelos.end(); it++) {

			GIL::IModeloIntegracion* modelo = *it;
			GIL::TListaIdentificadores& ids = modelo->Paciente.listaIdentificadores;

			for (GIL::TListaIdentificadores::iterator it = ids.begin(); it != ids.end(); it++)
			{
				if (first) {
					first = false;
				}
				else {
					os << "/";
				}
				os << (*it).valor;
			}

		}

		int tmpIndex = m_pFieldCombo->FindString(_("Id"));
		if (tmpIndex > 0) {
			m_pFieldCombo->Remove(tmpIndex,tmpIndex);
		}
		tmpIndex = m_pFieldCombo->FindString(_("Name"));
		if (tmpIndex > 0) {
			m_pFieldCombo->Remove(tmpIndex,tmpIndex);
		}

		m_HardCodedPatientId = os.str();
	}
	else {
		int tmpIndex = m_pFieldCombo->FindString(_("Name"));
		if (tmpIndex < 0) {
			m_pFieldCombo->Insert(_("Name"),0);
		}
		tmpIndex = m_pFieldCombo->FindString(_("Id"));
		if (tmpIndex < 0) {
			m_pFieldCombo->Insert(_("Id"),0);
		}
		m_HardCodedPatientSearch = false;
	}
	m_pFieldCombo->Select(0);

	//modalities
	{
		const int MaxModalityNumber=20; // Modificar la constante si se añaden o quitan códigos de modalidad
		std::string modalidades[MaxModalityNumber] = {"CR","CT","DR","DX","IO","MG","MR","NM","OT","PT","RF","RG","SC","SR","US","XA","XC","ES","ECG","HD"};
		for(int i = 0; i<MaxModalityNumber; i++) {
			wxCheckBox* pCheckBox = new wxCheckBox( m_pPanelPACS, wxID_ANY, wxString::FromUTF8(modalidades[i].c_str()), wxDefaultPosition, wxDefaultSize, 0 );
			m_pModalitySizer->Add( pCheckBox, 0, wxALL, 2 );
			m_modalitiesList.push_back(pCheckBox);
			pCheckBox->Connect( wxEVT_KEY_UP, wxKeyEventHandler( DialogoAdquisicion::OnKeyDownFormulario ), NULL, this );
		}
		m_pPanelPACS->Layout();
		m_pPanelSearch->Layout();
		m_notebook1->Layout();
	}

	m_pModeloDicom = GnkPtr<IModeloDicom>(new IModeloDicom());

	Layout();
	AutoSizeGrid();
	m_pGridDescargas->EnableDragColSize(true);

	GNC::GCS::ControladorEventos::Instance()->Registrar(this, GNC::GCS::Eventos::EventoProgresoComando());
}

GNC::GUI::DialogoAdquisicion::~DialogoAdquisicion()
{
	GNC::Entorno::Instance()->GetControladorComandos()->AbortarComandosDeOwner(this);
	{
		m_pDescargasGridTable->Clear();
		this->m_pTreeListResultados->DeleteRoot();
	}
	delete m_pImageList;
	m_pAnimation->Destroy();
}

bool GNC::GUI::DialogoAdquisicion::Show(bool show)
{
	//se recarga la lista de servidores pacs
	m_pPACSList->DeleteRoot();
	wxTreeItemId root = m_pPACSList->AddRoot(wxT("Root"));
	for (DicomServerHolder* sl = DicomServerList::Instance()->GetList(); sl != NULL; sl = sl->next) {
		wxTreeItemId idItem = m_pPACSList->AppendItem(root, wxString::FromUTF8(sl->server.ID.c_str()));
		std::ostringstream ostr;
		ostr << sl->server.AET << "@" << sl->server.HostName << ":" << sl->server.Port;
		m_pPACSList->SetItemText(idItem, COL_PACS_DESCRIPTION, wxString::FromUTF8(ostr.str().c_str()));
		//mode (series or studies)
		m_pPACSList->SetItemText(idItem, COL_PACS_MODE, (sl->server.retrieveSeries?wxT("Series"):wxT("Studies")));
	}

	if(m_pPACSList->GetChildrenCount(root) == 0){
		m_pPACSList->AppendItem(root, _("There is no PACS server configured"));
		m_pPACSList->SetColumnWidth(0,220);
		m_pPACSList->Enable(false);
		m_pPanelPACS->Enable(false);
	} else {
		m_pPACSList->SetColumnWidth(0,80);
		{
			wxConfigBase * config = wxConfigBase::Get();
			//se actualiza el fichero de configuracion
			config->SetPath(wxT("/GinkgoCore/Adquisicion"));
			wxString pacsDefecto;
			config->Read(wxT("PACSDefecto"),&pacsDefecto,wxT(""));
			wxTreeItemId idItem = m_pPACSList->FindItem(root, pacsDefecto);
			if (!idItem.IsOk()) {
				wxTreeItemIdValue cookie;
				idItem = m_pPACSList->GetFirstChild(root, cookie);
				pacsDefecto = m_pPACSList->GetItemText(idItem);
			}
			m_pPACSList->SelectItem(idItem);
			m_SelectedPACS = pacsDefecto;
			m_Mode = (m_pPACSList->GetItemText(idItem, COL_PACS_MODE) == wxT("Series"))?TD_SERIES:TD_STUDIES;
		}
		m_pPACSList->Enable(true);
		m_pPanelPACS->Enable(true);
	}

	return DialogoAdquisicionBase::Show(show);
}



//---------------------------------------------------------------------------
//region Eventos de interfaz
void GNC::GUI::DialogoAdquisicion::OnPACSChanged(wxTreeEvent & event)
{
	wxTreeItemId item = event.GetItem();
	if (item.IsOk()) {
		wxString newSelection = m_pPACSList->GetItemText(event.GetItem());
		if (m_SelectedPACS != newSelection) {
			m_SelectedPACS = newSelection;
			//se guarda la opcion seleccionada
			wxConfigBase * config = wxConfigBase::Get();
			//se actualiza el fichero de configuracion
			config->SetPath(wxT("/GinkgoCore/Adquisicion"));

			config->Write(wxT("PACSDefecto"), m_SelectedPACS);

			if(m_pComandoPACS != NULL)
			{
				GNC::Entorno::Instance()->GetControladorComandos()->AbortarComando(m_pComandoPACS, false);
			}
			m_pModeloDicom = GnkPtr<IModeloDicom>(new IModeloDicom());
			{
				//GNC::GCS::ILocker locker(m_TreeListResultadosLocker);
				this->m_pTreeListResultados->DeleteRoot();
				m_AutoAddSeries = false;
				m_Mode = (m_pPACSList->GetItemText(item, COL_PACS_MODE) == wxT("Series"))?TD_SERIES:TD_STUDIES;
			}
		}
	}
}

void GNC::GUI::DialogoAdquisicion::OnKeyDownFormulario( wxKeyEvent& event )
{
	if(event.GetKeyCode() == WXK_RETURN || event.GetKeyCode() == WXK_NUMPAD_ENTER) {
		wxCommandEvent evt;
		if (m_pTextControlField->IsEnabled()) {
			OnBusquedaClick(evt);
		}
		event.Skip(false);
	}
	else {
		if(event.GetKeyCode() == WXK_ESCAPE) {
			this->Hide();
			event.Skip(false);
		}
	}
	event.Skip(true);
}


void GNC::GUI::DialogoAdquisicion::OnFechaDesdeDateChanged( wxDateEvent& /*event*/ )
{
	m_pBetween->SetValue(true);
}



void GNC::GUI::DialogoAdquisicion::OnFechaHastaDateChanged( wxDateEvent& /*event*/ )
{
	m_pBetween->SetValue(true);
}

void GNC::GUI::DialogoAdquisicion::OnCancelClick( wxCommandEvent&  )
{
	if (m_AutoAddSeries)  {
		m_AutoAddSeries = false;
		m_AutoAddSeriesItem = NULL;
	}

	if(m_pComandoPACS != NULL)
	{
		m_pTextControlField->ShowCancelButton(false);
		ShowAnimation(false);
		GNC::Entorno::Instance()->GetControladorComandos()->AbortarComando(m_pComandoPACS, false);
		m_pComandoPACS = NULL;

		//GNC::GCS::ILocker locker(m_TreeListResultadosLocker);
		if(m_lastExpanded.IsOk()) {
			m_pTreeListResultados->Collapse(m_lastExpanded);
		} else {
			LimpiarBusquedas(false);
		}
	}
}

void GNC::GUI::DialogoAdquisicion::OnBusquedaClick( wxCommandEvent&  )
{

	if (m_AutoAddSeries)  {
		m_AutoAddSeries = false;
		m_AutoAddSeriesItem = NULL;
	}

	if(m_pComandoPACS != NULL)
	{
		m_pTextControlField->ShowCancelButton(false);
		ShowAnimation(false);
		GNC::Entorno::Instance()->GetControladorComandos()->AbortarComando(m_pComandoPACS, false);
		m_pComandoPACS = NULL;

		//GNC::GCS::ILocker locker(m_TreeListResultadosLocker);
		if(m_lastExpanded.IsOk()) {
			m_pTreeListResultados->Collapse(m_lastExpanded);
		} else {
			LimpiarBusquedas(false);
		}
	}

	std::string server = GetServerSeleccionado();
	m_pModeloDicom = GnkPtr<IModeloDicom>(new IModeloDicom());
	if(server=="")
		return;

	std::string fechaDesde("");
	std::string fechaHasta("");
	std::string timeFrom("");
	std::string timeTo("");
	if (m_pBetween->GetValue()){
		if(m_pTextControlFechaDesde->GetValue().IsValid()){
			fechaDesde =  std::string(m_pTextControlFechaDesde->GetValue().Format(wxT("%Y/%m/%d")).ToUTF8());
		}
		if(m_pTextControlFechaHasta->GetValue().IsValid()){
			fechaHasta =  std::string(m_pTextControlFechaHasta->GetValue().Format(wxT("%Y/%m/%d")).ToUTF8());
		}
	} else if (m_pToday->GetValue() || m_pTodayAM->GetValue() || m_pTodayPM->GetValue()) {
		fechaDesde = fechaHasta = wxDateTime::Now().Format(wxT("%Y/%m/%d")).ToUTF8();
		if (m_pTodayAM->GetValue()) {
			timeFrom = "000000";
			timeTo = "115959";
		} else if (m_pTodayPM->GetValue()) {
			timeFrom = "120000";
			timeTo = "235959";
		}
	} else if (m_pYesterday->GetValue()) {
		fechaHasta = fechaDesde = wxDateTime::Now().Add(wxDateSpan(0,0,0,-1)).Format(wxT("%Y/%m/%d")).ToUTF8();
	} else if (m_pLastWeek->GetValue()) {
		fechaDesde = wxDateTime::Now().Add(wxDateSpan(0,0,-1,0)).Format(wxT("%Y/%m/%d")).ToUTF8();
	} else if (m_pLastMonth->GetValue()) {
		fechaDesde = wxDateTime::Now().Add(wxDateSpan(0,-1,0,0)).Format(wxT("%Y/%m/%d")).ToUTF8();
	} else if (m_pLastThreeMonths->GetValue()) {
		fechaDesde = wxDateTime::Now().Add(wxDateSpan(0,-3,0,0)).Format(wxT("%Y/%m/%d")).ToUTF8();
	}

	if(fechaDesde == "" && fechaHasta == "" && m_pTextControlField->GetValue().size() == 0
		&& (m_HardCodedPatientSearch && m_HardCodedPatientId.size() == 0) )
	{
		int answer = wxMessageBox(_("Search without parameters could take al long time\nWould you like to continue?"),_("Search"), wxYES_NO , this);
		if (answer == wxNO) {
			return;
		}
	}

	std::string idPaciente, nombrePaciente, studyUID, accNumber;

	if (m_HardCodedPatientSearch) {
		idPaciente = m_HardCodedPatientId;
		nombrePaciente = "";
	}

	if (m_pFieldCombo->GetValue() == _("Id")) {
		idPaciente = m_pTextControlField->GetValue().ToUTF8();
	} else if (m_pFieldCombo->GetValue() == _("Name")) {
		nombrePaciente = m_pTextControlField->GetValue().ToUTF8();
	} else if (m_pFieldCombo->GetValue() == _("Acc#")) {
		accNumber = m_pTextControlField->GetValue().ToUTF8();
	} else if (m_pFieldCombo->GetValue() == _("Study UID")) {
		studyUID = m_pTextControlField->GetValue().ToUTF8();
	}

	//modalities checked


	GADAPI::ComandoPACSParams * pParams = new GADAPI::ComandoPACSParams(
		idPaciente,
		nombrePaciente,
		studyUID,
		accNumber,
		GetModalities(),
		fechaDesde,fechaHasta,
		timeFrom, timeTo,
		server, GADAPI::ComandoPACSParams::TA_Estudio, m_pModeloDicom, this); // Buscamos los estudios del paciente

	m_pComandoPACS = new GADAPI::ComandoPACS(pParams);
	GNC::Entorno::Instance()->GetControladorComandos()->ProcessAsync(_Std("Exploring PACS..."),m_pComandoPACS, this);
	m_pTextControlField->ShowCancelButton(true);
	ShowAnimation(true);
}

std::string GNC::GUI::DialogoAdquisicion::GetModalities()
{
	std::string modalities;
	bool all = true;

	for (TModalitiesVector::iterator it =m_modalitiesList.begin(); it !=m_modalitiesList.end(); it++) {
		wxCheckBox* pCheck = dynamic_cast<wxCheckBox*>(*it);
		if(pCheck != NULL)
			if(pCheck->IsChecked()){
				all = false;
				if (modalities == ""){
					modalities = pCheck->GetLabel().ToUTF8();
				}
				else{
					modalities.append("\\");
					modalities.append(pCheck->GetLabel().ToUTF8());
				}
			}
	}
	if (all) {
		modalities = "*";
	}
	return modalities;
}

void GNC::GUI::DialogoAdquisicion::OnLimpiarClick( wxCommandEvent& /*event*/ )
{
	LimpiarBusquedas();
}

void GNC::GUI::DialogoAdquisicion::OnIniciarPararTasksClick( wxCommandEvent& /*event*/ )
{
	int action = 0;

	m_pGridDescargas->BeginBatch();
	for ( int i = 0; i < m_pGridDescargas->GetNumberRows(); i++ )
	{
		if ( m_pGridDescargas->IsInSelection( i , 0 ) )
		{
			// Esto puede ser una fuente de problemas: OJO
			const Descargas::Descarga* d = m_pDescargasGridTable->GetDescarga(i);
			switch (d->StatusId) {
case Descargas::Iniciada:
	if (action == 0 ||action == 1) { // Para asegurar que solo se hace una operacion:
		m_pDescargasGridTable->DetenerDescarga(i);
		action = 1;
	}
	else {
		std::cerr << "Error en la logica de control de tareas de descarga" << std::endl;
	}
	break;
case Descargas::Parada:
	if (action == 0 || action == 2) { // Para asegurar que solo se hace una operacion:
		m_pDescargasGridTable->IniciarDescarga(i);
		action = 2;
	}
	else {
		std::cerr << "Error en la logica de control de tareas de descarga" << std::endl;
	}
	break;
case Descargas::Terminada:
	break;
case Descargas::Erronea:
	break;
default:
	std::cerr << "Error en la logica de control de tareas de descarga" << std::endl;
	break;
			}

		}
	}

	if (action == 1) { // Hemos detenido descargas iniciadas
		m_pBotonIniciarPararTask->SetLabel(_("Start"));
		m_pBotonDeleteTask->Enable(true);
	}
	else if (action == 2) { // Hemos iniciado descargas paradas
		m_pBotonIniciarPararTask->SetLabel(_("Stop"));
		m_pBotonDeleteTask->Enable(false);
	}
	m_pGridDescargas->EndBatch();
}

void GNC::GUI::DialogoAdquisicion::OnDeleteTasksClick( wxCommandEvent& /*event*/ )
{

	if ( m_pGridDescargas->IsSelection() )
	{
		int lastDeleted = -1;
		int deletedCount = 0;
		m_pGridDescargas->BeginBatch();
		for ( int i = 0; i < m_pGridDescargas->GetNumberRows(); )
		{
			if ( m_pGridDescargas->IsInSelection( i , 0 ) ) {
				m_pGridDescargas->DeleteRows( i, 1 );
				lastDeleted = i;
				++deletedCount;
			} else
				i++;
		}
		m_pGridDescargas->EndBatch();
		//cuando elimino selecciono la fila anterior
		if (lastDeleted != -1) {
			lastDeleted = lastDeleted - deletedCount;
			if ( lastDeleted < 0 && m_pGridDescargas->GetNumberRows() > 0 ) {
				m_pGridDescargas->SelectRow(0);
			} else {
				if ( lastDeleted < m_pGridDescargas->GetNumberRows() ) {
					m_pGridDescargas->SelectRow(lastDeleted);
				}
			}
		}
	}
}

void GNC::GUI::DialogoAdquisicion::OnOpenTasksClick( wxCommandEvent& /*event*/ )
{
	if ( m_pGridDescargas->IsSelection() )
	{
		m_pGridDescargas->BeginBatch();
		typedef std::list<const Descargas::Descarga*> TListaDescargas;
		TListaDescargas listaAbrir;

		for ( int i = 0; i < m_pGridDescargas->GetNumberRows();++i )
		{
			if ( m_pGridDescargas->IsInSelection( i , 0 ) ) {
				const Descargas::Descarga* d = m_pDescargasGridTable->GetDescarga(i);
				if (d->StatusId == Descargas::Terminada) {
					listaAbrir.push_back(d);
				}
			}
		}
		//se comprueba que todas las series seleccionadas existen...
		GNC::GCS::Eventos::EvtHandleDicom::ListaUIDs listaUIdsAbrir;
		TListaDescargas listaNoExisten;
		TListaDescargas listaAbrirDiagnosticos;

		for (TListaDescargas::iterator it = listaAbrir.begin(); it != listaAbrir.end();++it) {
			if ( ! GNC::GCS::ControladorHistorial::Instance()->ExisteSerie((*it)->UID) ) {
				listaNoExisten.push_back((*it));
			} else if ( (*it)->Modality == "SR" ) {
				listaAbrirDiagnosticos.push_back((*it));
			} else {
				listaUIdsAbrir.push_back((*it)->UID);
			}
		}

		//uids que no existen en el historial
		if (listaNoExisten.size() > 0) {
			DialogoNotFoundBase dlg(this);
			for ( TListaDescargas::iterator it = listaNoExisten.begin(); it != listaNoExisten.end(); ++it) {
				std::ostringstream ostr;
				if ((*it)->Description != "") {
					ostr << (*it)->Description;
				} else {
					ostr << (*it)->Modality << " (" << (*it)->UID << ")";
				}

				dlg.m_pListaSeries->Append( wxString::FromUTF8(ostr.str().c_str()) );
			}

			if (listaUIdsAbrir.size() > 0) {
				dlg.m_pPanelOk->Show(false);
			} else {
				dlg.m_pPanelOkCancel->Show(false);
			}

			int answer = dlg.ShowModal();
			if (answer == wxID_CANCEL) {
				return;
			}
		}

		//diagnosticos que se quieren abrir
		if (listaAbrirDiagnosticos.size() > 0) {
			wxMessageBox(_("Diagnostic files can't be opened, you have to open the diagnosed series"), _("Info"),
				wxOK | wxICON_INFORMATION, this);
		}

		if(listaUIdsAbrir.size() > 0) {
			GNC::GCS::ControladorEventos::Instance()->ProcesarEvento(new GNC::GCS::Eventos::EvtHandleDicom(listaUIdsAbrir, GNC::GCS::Eventos::EvtHandleDicom::OpenSeries));
		}
		m_pGridDescargas->EndBatch();
	}
}

void GNC::GUI::DialogoAdquisicion::OnClearTasksClick( wxCommandEvent& /*event*/ )
{
	m_pGridDescargas->BeginBatch();
	for ( int i = 0; i < m_pGridDescargas->GetNumberRows(); )
	{
		const Descargas::Descarga* d = m_pDescargasGridTable->GetDescarga(i);
		if ( d->StatusId == Descargas::Terminada  || d->StatusId == Descargas::Erronea ) {
			m_pGridDescargas->DeleteRows( i, 1 );
		} else
			i++;
	}
	m_pGridDescargas->EndBatch();
}

void GNC::GUI::DialogoAdquisicion::OnCloseClick( wxCommandEvent& /*event*/ )
{
	Hide();
}

void GNC::GUI::DialogoAdquisicion::OnDescargarClick( wxCommandEvent& /*event*/ )
{
	AddDescarga(true);
}

// Eventos del Treelist
void GNC::GUI::DialogoAdquisicion::OnTreeSelChanged(wxTreeEvent& event)
{

	wxTreeItemId item = event.GetItem();
	if(item.IsOk()){
		wxTreeItemId idPadre = m_pTreeListResultados->GetItemParent(item);
		m_pBDescargar->Enable(idPadre != m_pTreeListResultados->GetRootItem()); // El item es un estudio o una serie
	} else {
		m_pBDescargar->Enable(false);
	}

	event.Skip(true);

}

void GNC::GUI::DialogoAdquisicion::OnTreeItemActivated(wxTreeEvent& event)
{

	AddDescarga(false);

	event.Skip(true);

}

void GNC::GUI::DialogoAdquisicion::OnTreeItemMenu(wxTreeEvent& event)
{
	//clase menu contextual
	class PopUpMenuDescargar: public wxMenu
	{
	public:
		PopUpMenuDescargar(DialogoAdquisicion* pDlg, bool allowDownload) : wxMenu()
		{
			m_pDlg = pDlg;
			if (allowDownload) {
				wxMenuItem* descargar = new wxMenuItem( this,wxID_ANY,_("Download"));
#ifdef __WXMSW__
				descargar->SetBitmaps(GinkgoResourcesManager::Adquisicion::GetIcoDownload());
#else
				descargar->SetBitmap(GinkgoResourcesManager::Adquisicion::GetIcoDownload());
#endif
				Append(descargar);
				this->AppendSeparator();

				Connect(descargar->GetId(), wxEVT_COMMAND_MENU_SELECTED, wxCommandEventHandler( PopUpMenuDescargar::OnDescargar ), NULL, this);
			}
			wxMenuItem* props = new wxMenuItem( this, wxID_ANY, _("Properties..."));
			Append(props);
			Connect(props->GetId(), wxEVT_COMMAND_MENU_SELECTED, wxCommandEventHandler( PopUpMenuDescargar::OnProperties ),NULL, this);

		}
		~PopUpMenuDescargar()
		{
			m_pDlg = NULL;
			Disconnect(wxID_ANY, wxEVT_COMMAND_MENU_SELECTED, wxCommandEventHandler( PopUpMenuDescargar::OnDescargar ), NULL, this);
			Disconnect(wxID_ANY, wxEVT_COMMAND_MENU_SELECTED, wxCommandEventHandler( PopUpMenuDescargar::OnProperties ), NULL, this);
		}

		void OnDescargar(wxCommandEvent& )
		{
			if (m_pDlg) {
				m_pDlg->AddDescarga(true);
			}
		}

		void OnProperties(wxCommandEvent& )
		{
			if (m_pDlg) {
				m_pDlg->ShowProperties();
			}
		}

		DialogoAdquisicion* m_pDlg;

	};
	//
	wxTreeItemId item = event.GetItem();
	if(item.IsOk()){
		wxTreeItemId idPadre = m_pTreeListResultados->GetItemParent(item);
		if(idPadre != m_pTreeListResultados->GetRootItem() ){ // El item es un estudio o una serie
			wxTreeItemId idAbuelo = m_pTreeListResultados->GetItemParent(m_pTreeListResultados->GetItemParent(item));
			// Comprobamos que el item es un estudio o es una serie desplegada.
			if ( ( idAbuelo == m_pTreeListResultados->GetRootItem() ) || ! ( (m_pTreeListResultados->GetItemText(item, COLUMNA_UID)).empty() ) ) {

				PopUpMenuDescargar* pMenu = new PopUpMenuDescargar(this, true);

				m_pTreeListResultados->PopupMenu(pMenu);

				delete pMenu;
			}
		}
		else {
			PopUpMenuDescargar* pMenu = new PopUpMenuDescargar(this, false);

			m_pTreeListResultados->PopupMenu(pMenu);

			delete pMenu;
		}
	}
}

void GNC::GUI::DialogoAdquisicion::OnTreeItemExpanded(wxTreeEvent& event)
{
	{
		//GNC::GCS::ILocker locker(m_TreeListResultadosLocker);
		wxTreeItemId item = event.GetItem();

		if (m_AutoAddSeries && (m_AutoAddSeriesItem != item) )  {
			m_AutoAddSeries = false;
			m_AutoAddSeriesItem = NULL;
		}

		if(item && item.IsOk()){
			m_pTreeListResultados->SortChildren(item);
			//si el abuelo del item es root => consultar
			if(m_pTreeListResultados->IsExpanded(item) && m_pTreeListResultados->GetItemParent(m_pTreeListResultados->GetItemParent(item)) == m_pTreeListResultados->GetRootItem())
			{
				//se cancela el anterior comando y se contrae el item expandido
				if(m_pComandoPACS != NULL)
				{
					GNC::Entorno::Instance()->GetControladorComandos()->AbortarComando(m_pComandoPACS, false);
					m_pComandoPACS = NULL;
					if(m_lastExpanded.IsOk()) {
						m_pTreeListResultados->Collapse(m_lastExpanded);
					}
				}

				std::string server=GetServerSeleccionado();
				if(server=="")
					return;

				GADAPI::ComandoPACSParams * pParams = new GADAPI::ComandoPACSParams(
					"",
					"",
					std::string(m_pTreeListResultados->GetItemText(item,COLUMNA_UID).ToUTF8()),
					"",
					GetModalities(),
					"",
					"",
					"",
					"",server, GADAPI::ComandoPACSParams::TA_Serie, m_pModeloDicom, this);

				m_pComandoPACS = new GADAPI::ComandoPACS(pParams);
				GNC::Entorno::Instance()->GetControladorComandos()->ProcessAsync(_Std("Exploring PACS..."),m_pComandoPACS, this);
				m_pTextControlField->ShowCancelButton(true);
				ShowAnimation(true);
				m_lastExpanded = item;
			}
			event.Skip(false);
		} else {
			event.Skip(true);
		}
	}

}

void GNC::GUI::DialogoAdquisicion::OnTreeBeginDrag(wxTreeEvent& )
{
	wxTreeItemId seleccion = m_pTreeListResultados->GetSelection();
	if(seleccion.IsOk()){
		if(m_pTreeListResultados->GetItemParent(seleccion) == m_pTreeListResultados->GetRootItem()){
			//esto es para que de la impresion de que no podemos arrastrar la raiz
			class wxFooDataObject: public wxDataObjectSimple
			{
			public:
				wxFooDataObject():wxDataObjectSimple() {}
				~wxFooDataObject() {}
			};
			wxFooDataObject myData;
			wxDropSource dragSource(myData,this);
			dragSource.DoDragDrop(wxDrag_AllowMove);
		}else {
			//comenzamos drag & drop
			wxTextDataObject myData(wxT(""));
			wxDropSource dragSource(myData,this);
			wxDragResult result = dragSource.DoDragDrop(wxDrag_AllowMove);
			if (result == wxDragCopy) {
				AddDescarga(true);
			}
		}
	}
}

void GNC::GUI::DialogoAdquisicion::UpdateButtons()
{
	int total = 0;
	int iniciadas = 0;
	int paradas = 0;
	int terminadas = 0;

	m_pGridDescargas->BeginBatch();
	for ( int i = 0; i < m_pGridDescargas->GetNumberRows(); i++)
	{
		if ( m_pGridDescargas->IsInSelection( i , 0 ) ) {
			const Descargas::Descarga* d = m_pDescargasGridTable->GetDescarga(i);
			switch( d->StatusId ) {
case Descargas::Parada:
	paradas++;
	total++;
	break;
case Descargas::Iniciada:
	iniciadas++;
	total++;
	break;
case Descargas::Terminada:
	terminadas++;
	total++;
case Descargas::Erronea:
	terminadas++;
	total++;
	break;
			}
		}
	}
	m_pGridDescargas->EndBatch();

	if (total == 0) {
		m_pBotonIniciarPararTask->Enable(false);
		m_pBotonIniciarPararTask->SetLabel(_("Start"));
		m_pBotonOpenTask->Enable(false);
		m_pBotonDeleteTask->Enable(false);
		return;
	}

	m_pBotonDeleteTask->Enable( (total == terminadas) || (total == (terminadas + paradas)) );
	if (total == terminadas){
		m_pBotonIniciarPararTask->Enable(false);
		m_pBotonIniciarPararTask->SetLabel(_("Start"));
		m_pBotonDeleteTask->Enable(true);
	}
	else if ( total == iniciadas) {
		m_pBotonIniciarPararTask->Enable(true);
		m_pBotonIniciarPararTask->SetLabel(_("Stop"));
		m_pBotonDeleteTask->Enable(false);
	}
	else if ( total == (paradas + terminadas ) ) {
		m_pBotonIniciarPararTask->Enable(true);
		m_pBotonIniciarPararTask->SetLabel(_("Start"));
		m_pBotonDeleteTask->Enable(true);
	}
	else if (total == paradas) {
		m_pBotonIniciarPararTask->Enable(true);
		m_pBotonIniciarPararTask->SetLabel(_("Start"));
		m_pBotonDeleteTask->Enable(true);
	}
	else {
		m_pBotonIniciarPararTask->Enable(false);
		m_pBotonIniciarPararTask->SetLabel(_("Start"));
		m_pBotonDeleteTask->Enable(false);
	}

	m_pBotonOpenTask->Enable( total == terminadas);
}


void GNC::GUI::DialogoAdquisicion::OnGridSelect(wxGridRangeSelectEvent& event)
{
	UpdateButtons();

	event.Skip();
}

void GNC::GUI::DialogoAdquisicion::OnGridKeyDown(wxKeyEvent& event)
{
	switch (event.GetKeyCode())
	{
	case WXK_DELETE:
		{
			if ( m_pBotonDeleteTask->IsEnabled() ) {
				wxCommandEvent evt;
				OnDeleteTasksClick(evt);
			}
			event.Skip(false);
		}
		break;
	case WXK_NUMPAD_ENTER:
	case WXK_RETURN:
		{
			if ( m_pBotonOpenTask->IsEnabled() ) {
				wxCommandEvent evt;
				OnOpenTasksClick(evt);
			}
			event.Skip(false);
		}
		break;
	default:
		event.Skip(true);
	}
}

void GNC::GUI::DialogoAdquisicion::OnGridDClick(wxGridEvent& event)
{
	if ( m_pBotonOpenTask->IsEnabled() ) {
		wxCommandEvent evt;
		OnOpenTasksClick(evt);
	}
	event.Skip();
}

void GNC::GUI::DialogoAdquisicion::OnGridMenu(wxGridEvent& )
{
	//clase menu contextual
	class PopUpMenuListaDescargas: public wxMenu
	{
	public:
		PopUpMenuListaDescargas(DialogoAdquisicion* pDlg) : wxMenu()
		{
			m_pDlg = pDlg;

			if (m_pDlg->m_pBotonOpenTask->IsEnabled()) {
				wxMenuItem* menuItem = new wxMenuItem( this,wxID_ANY,m_pDlg->m_pBotonOpenTask->GetLabel());
				Append(menuItem);
				Connect(menuItem->GetId(), wxEVT_COMMAND_MENU_SELECTED, wxCommandEventHandler( PopUpMenuListaDescargas::OnOpen ),NULL,this);
				AppendSeparator();
			}

			if (m_pDlg->m_pBotonIniciarPararTask->IsEnabled()) {
				wxMenuItem* menuItem = new wxMenuItem( this,wxID_ANY,m_pDlg->m_pBotonIniciarPararTask->GetLabel());
				Append(menuItem);
				Connect(menuItem->GetId(), wxEVT_COMMAND_MENU_SELECTED, wxCommandEventHandler( PopUpMenuListaDescargas::OnIniciarParar ),NULL,this);
				AppendSeparator();
			}

			if (m_pDlg->m_pBotonDeleteTask->IsEnabled()) {
				wxMenuItem* menuItem = new wxMenuItem( this,wxID_ANY,m_pDlg->m_pBotonDeleteTask->GetLabel());
#ifdef __WXMSW__
				menuItem->SetBitmaps(GinkgoResourcesManager::Adquisicion::GetIcoEliminarDescarga());
#else
				menuItem->SetBitmap(GinkgoResourcesManager::Adquisicion::GetIcoEliminarDescarga());
#endif
				Append(menuItem);
				Connect(menuItem->GetId(), wxEVT_COMMAND_MENU_SELECTED, wxCommandEventHandler( PopUpMenuListaDescargas::OnDelete ),NULL,this);
			}

			{
				wxMenuItem* menuItem = new wxMenuItem( this,wxID_ANY,_("Clear Finished"));
#ifdef __WXMSW__
				menuItem->SetBitmaps(GinkgoResourcesManager::Adquisicion::GetIcoClear());
#else
				menuItem->SetBitmap(GinkgoResourcesManager::Adquisicion::GetIcoClear());
#endif
				Append(menuItem);
				Connect(menuItem->GetId(), wxEVT_COMMAND_MENU_SELECTED, wxCommandEventHandler( PopUpMenuListaDescargas::OnClear ),NULL,this);
			}
		}
		~PopUpMenuListaDescargas()
		{
			m_pDlg = NULL;
			Disconnect(wxID_ANY, wxEVT_COMMAND_MENU_SELECTED, wxCommandEventHandler( PopUpMenuListaDescargas::OnIniciarParar ),NULL,this);
			Disconnect(wxID_ANY, wxEVT_COMMAND_MENU_SELECTED, wxCommandEventHandler( PopUpMenuListaDescargas::OnDelete ),NULL,this);
			Disconnect(wxID_ANY, wxEVT_COMMAND_MENU_SELECTED, wxCommandEventHandler( PopUpMenuListaDescargas::OnOpen ),NULL,this);
		}

		void OnIniciarParar(wxCommandEvent& )
		{
			wxCommandEvent evt;
			m_pDlg->OnIniciarPararTasksClick(evt);
		}

		void OnDelete(wxCommandEvent& )
		{
			wxCommandEvent evt;
			m_pDlg->OnDeleteTasksClick(evt);
		}

		void OnOpen(wxCommandEvent& )
		{
			wxCommandEvent evt;
			m_pDlg->OnOpenTasksClick(evt);
		}

		void OnClear(wxCommandEvent& )
		{
			wxCommandEvent evt;
			m_pDlg->OnClearTasksClick(evt);
		}

		DialogoAdquisicion* m_pDlg;
	};
	//

	PopUpMenuListaDescargas* pMenu = new PopUpMenuListaDescargas(this);
	m_pGridDescargas->PopupMenu(pMenu);
	delete pMenu;
}

void GNC::GUI::DialogoAdquisicion::OnGridSize( wxSizeEvent& event )
{
	/*m_pGridDescargas->SetSize(event.GetSize());
	AutoSizeGrid();
	*/
	event.Skip();
}

//---------------------------------------------------------------------------
//region Helpers

void GNC::GUI::DialogoAdquisicion::AutoSizeGrid()
{
	wxSize s = m_pGridDescargas->GetSize();

	float prop = 1.0f / ( (float)m_pGridDescargas->GetNumberCols() );

	int cw = s.GetWidth() - 10;
	int aw = 0;
	int i = 0;
	for (; i < m_pGridDescargas->GetNumberCols() - 1; i++)
	{
		int iw = (int) (prop * (float)cw);
		aw += iw;
		m_pGridDescargas->SetColSize(i,  iw);
	}
	if ( i < m_pGridDescargas->GetNumberCols()) {
		m_pGridDescargas->SetColSize(i, cw - aw );
	}


	m_pGridDescargas->Refresh();

}

std::string GNC::GUI::DialogoAdquisicion::GetServerSeleccionado()
{
	std::string id;
	wxTreeItemId idItem = m_pPACSList->GetSelection();
	if (idItem.IsOk()) {
		id = std::string(m_pPACSList->GetItemText(idItem).ToUTF8());
	} else {
		wxMessageBox(_("Select a valid PACS Server"), _("Info"),
			wxOK | wxICON_INFORMATION, this);
	}
	return id;
}


// Obtiene el item seleccionado del TreeList de resultados, genera una descarga, la apila y la inicia.
void GNC::GUI::DialogoAdquisicion::AddDescarga(bool lanzarBusqueda)
{
	{
		//GNC::GCS::ILocker locker(m_TreeListResultadosLocker);
		wxTreeItemId seleccion = m_pTreeListResultados->GetSelection();

		if(seleccion.IsOk()){
			if(m_pTreeListResultados->GetItemParent(seleccion) == m_pTreeListResultados->GetRootItem()){
				return;
			} //raiz
			else if(m_pTreeListResultados->GetItemParent(m_pTreeListResultados->GetItemParent(seleccion)) == m_pTreeListResultados->GetRootItem()){
				//studies
				if (m_Mode == TD_STUDIES) {
					//download study node
					std::string uidPaciente = std::string(m_pTreeListResultados->GetItemText(m_pTreeListResultados->GetItemParent(seleccion), COLUMNA_UID).ToUTF8());
					std::string descPaciente = std::string(m_pTreeListResultados->GetItemText(m_pTreeListResultados->GetItemParent(seleccion)).ToUTF8());

					std::string uidStudy = std::string(m_pTreeListResultados->GetItemText(seleccion, COLUMNA_UID).ToUTF8());
					std::string descStudy = std::string( (m_pTreeListResultados->GetItemText(seleccion) + _T("(") + m_pTreeListResultados->GetItemText(seleccion, COLUMNA_MODALIDAD) + _T(")") ).ToUTF8());
					std::string modStudy =  std::string( m_pTreeListResultados->GetItemText(seleccion, COLUMNA_MODALIDAD ).ToUTF8());

					if (!modStudy.empty() && ! GNC::GCS::ControladorPermisos::Instance()->Get("core.pacs.download", modStudy)) {
						LOG_ERROR("GUI/Adquisition", _Std("Download of modality ") << modStudy << _Std(" not allowed"));
						wxMessageBox(_("You are not allowed to download this kind of modalities."),_("Modality download error"),wxICON_ERROR);
					} else {
						bool forceInit = true;
						if (m_pDescargasGridTable->FindDescarga(uidStudy) != -1) {
							forceInit = wxMessageBox(_("Series were already in the download queue.\nDo you want to repeat download?"),_("Download warning"),wxICON_INFORMATION|wxYES_NO) == wxYES;
						}

						bool downloaded = m_pDescargasGridTable->AddDescarga(GetServerSeleccionado(), descPaciente, descStudy, uidStudy,modStudy, forceInit, false);
						m_pTreeListResultados->SetItemBold(seleccion, downloaded);
					}

				} else { 
					//series mode, we have to search series in study
					if (lanzarBusqueda) {

						//estamos en un estudio
						std::string uidPaciente = std::string(m_pTreeListResultados->GetItemText(m_pTreeListResultados->GetItemParent(seleccion), COLUMNA_UID).ToUTF8());
						std::string descPaciente = std::string(m_pTreeListResultados->GetItemText(m_pTreeListResultados->GetItemParent(seleccion)).ToUTF8());

						// Necesitamos buscar las series

						// Primero comprobamos que no estan ya desplegadas, y asi ahorramos una busqueda en el PACS.
						wxTreeItemId eid = seleccion;
						std::string server = GetServerSeleccionado();
						int numSeriesDescargadas = 0;

						std::list<wxTreeItemId> s_items = m_pTreeListResultados->GetPublicChildren(eid);
						std::set<std::string> listaModalidad;
						bool forceInit = true;
						bool questionMade = false;
						for (std::list<wxTreeItemId>::iterator ititem = s_items.begin(); ititem != s_items.end(); ititem++) {
							wxTreeItemId cid = *ititem;

							std::string uidSerie = std::string(m_pTreeListResultados->GetItemText(cid, COLUMNA_UID).ToUTF8());
							std::string modalidadSerie = std::string(m_pTreeListResultados->GetItemText(cid, COLUMNA_MODALIDAD).ToUTF8());
							if (!uidSerie.empty()) { // La serie esta desplegada
								if( GNC::GCS::ControladorPermisos::Instance()->Get("core.pacs.download", modalidadSerie) ) {
									if (!questionMade && m_pDescargasGridTable->FindDescarga(uidSerie) != -1) {
										forceInit = wxMessageBox(_("Series were already in the download queue.\nDo you want to force download?"),_("Download warning"),wxICON_INFORMATION|wxYES_NO) == wxYES;
										questionMade = true;
									}

									std::string descSerie = std::string( (m_pTreeListResultados->GetItemText(cid) + _T("(") + m_pTreeListResultados->GetItemText(cid, COLUMNA_MODALIDAD) + _T(")") ).ToUTF8());
									bool downloaded = m_pDescargasGridTable->AddDescarga(server, descPaciente, descSerie, uidSerie,modalidadSerie,forceInit, true);
									m_pTreeListResultados->SetItemBold(cid, downloaded);
									numSeriesDescargadas++;
								} else {
									listaModalidad.insert( modalidadSerie );
								}
							}
						}
						if (numSeriesDescargadas == 0 && listaModalidad.size() == 0)
						{
							// Desplegamos la serie.

							wxTreeItemId item = seleccion;
							// >>------- Es un cut&paste del evento OnTreeItemExpanded, pero mejor esto que complicar la interfaz invocandolo directamente (lockers, omision de evento, reset de estado de la auto descarga de series , etc..)
							//se cancela el anterior comando y se contrae el item expandido
							if(item && item.IsOk()){
								//si el abuelo del item es root => consultar
								m_AutoAddSeries = true;
								m_AutoAddSeriesItem = item;
							}

							if(m_pComandoPACS != NULL)
							{
								GNC::Entorno::Instance()->GetControladorComandos()->AbortarComando(m_pComandoPACS, false);
								m_pComandoPACS = NULL;

								if(m_lastExpanded.IsOk()) {
									m_pTreeListResultados->Collapse(m_lastExpanded);
									m_lastExpanded.Unset();
								}
							}
							std::string server=GetServerSeleccionado();

							if(server=="") {
								return;
							}

							GADAPI::ComandoPACSParams * pParams = new GADAPI::ComandoPACSParams(
								"",
								"",
								std::string(m_pTreeListResultados->GetItemText(item,COLUMNA_UID).ToUTF8()),
								"",
								GetModalities(),
								"",
								"",
								"",
								"",server, GADAPI::ComandoPACSParams::TA_Serie, m_pModeloDicom, this);

							m_pComandoPACS = new GADAPI::ComandoPACS(pParams);
							GNC::Entorno::Instance()->GetControladorComandos()->ProcessAsync(_Std("Exploring PACS..."),m_pComandoPACS, this);
							m_pTextControlField->ShowCancelButton(true);
							ShowAnimation(true);
						}
						else if ( listaModalidad.size() > 0)
						{
							std::ostringstream ostr;
							ostr << _Std("You are not allowed to download this kind of modalities (");
							bool primero = true;
							for (std::set<std::string>::iterator it = listaModalidad.begin(); it != listaModalidad.end(); ++it) {
								if (primero) {
									ostr << (*it);
									primero = false;
								}
								else {
									ostr << ", " << (*it);
								}
							}
							ostr << ")";
							wxMessageBox(wxString::FromUTF8( ostr.str().c_str() ), _("Modality download error"), wxICON_ERROR);
						}
					}//lanzar busqueda
				}//series mode
				//fin estudio
			} else {
				//estamos en una serie
				std::string uidPaciente = std::string(m_pTreeListResultados->GetItemText(m_pTreeListResultados->GetItemParent(m_pTreeListResultados->GetItemParent(seleccion)), COLUMNA_UID).ToUTF8());
				std::string descPaciente = std::string(m_pTreeListResultados->GetItemText(m_pTreeListResultados->GetItemParent(m_pTreeListResultados->GetItemParent(seleccion))).ToUTF8());

				std::string uidSerie = std::string(m_pTreeListResultados->GetItemText(seleccion, COLUMNA_UID).ToUTF8());
				std::string descSerie = std::string( (m_pTreeListResultados->GetItemText(seleccion) + _T("(") + m_pTreeListResultados->GetItemText(seleccion, COLUMNA_MODALIDAD) + _T(")") ).ToUTF8());
				std::string modSerie =  std::string( m_pTreeListResultados->GetItemText(seleccion, COLUMNA_MODALIDAD ).ToUTF8());

				if (!modSerie.empty()) {
					if(! GNC::GCS::ControladorPermisos::Instance()->Get("core.pacs.download", modSerie)) {
						LOG_ERROR("GUI/Adquisition", _Std("Download of modality ") << modSerie << _Std(" not allowed"));
						wxMessageBox(_("You are not allowed to download this kind of modalities."),_("Modality download error"),wxICON_ERROR);
					}
					else {
						bool forceInit = true;
						if (m_pDescargasGridTable->FindDescarga(uidSerie) != -1) {
							forceInit = wxMessageBox(_("Series were already in the download queue.\nDo you want to force download?"),_("Download warning"),wxICON_INFORMATION|wxYES_NO) == wxYES;
						}

						bool downloaded = m_pDescargasGridTable->AddDescarga(GetServerSeleccionado(), descPaciente, descSerie, uidSerie,modSerie, forceInit, true);
						m_pTreeListResultados->SetItemBold(seleccion, downloaded);
					}
				}
			}
		}
		else {
			return;
		}
	}
}

// Shows a dialog with the properties of selected treelist item
void GNC::GUI::DialogoAdquisicion::ShowProperties()
{


	wxTreeItemId selection = m_pTreeListResultados->GetSelection();

	if(selection.IsOk()){
		GNC::GUI::wxPropiedades::TListaMapasPropiedades listaMapas;

		wxTreeItemId patientTreeItemId;
		wxTreeItemId studyTreeItemId;
		wxTreeItemId serieTreeItemId;

		if(m_pTreeListResultados->GetItemParent(selection) == m_pTreeListResultados->GetRootItem()){
			//Patient Level
			patientTreeItemId = selection;
		}
		else if(m_pTreeListResultados->GetItemParent(m_pTreeListResultados->GetItemParent(selection)) == m_pTreeListResultados->GetRootItem()){
			studyTreeItemId   = selection;
			patientTreeItemId = m_pTreeListResultados->GetItemParent(selection);

		}
		else {
			serieTreeItemId   = selection;
			studyTreeItemId   = m_pTreeListResultados->GetItemParent(selection);
			patientTreeItemId = m_pTreeListResultados->GetItemParent(studyTreeItemId);
		}

		if (patientTreeItemId.IsOk()) {
			GNC::GUI::wxPropiedades::TMapaPropiedades mapaPropiedades;
			mapaPropiedades[_Std("Patient's Id")]   = std::string(m_pTreeListResultados->GetItemText(patientTreeItemId, COLUMNA_UID).ToUTF8());
			listaMapas.push_back(mapaPropiedades);
		}
		if (studyTreeItemId.IsOk()) {
			GNC::GUI::wxPropiedades::TMapaPropiedades mapaPropiedades;
			mapaPropiedades[_Std("Study Instance UID")] = std::string(m_pTreeListResultados->GetItemText(studyTreeItemId, COLUMNA_UID).ToUTF8());
			mapaPropiedades[_Std("Accession number")]   = std::string(m_pTreeListResultados->GetItemText(studyTreeItemId, COLUMNA_ACCNUMBER).ToUTF8());
			mapaPropiedades[_Std("Study date")] = std::string(m_pTreeListResultados->GetItemText(studyTreeItemId, COLUMNA_FECHA).ToUTF8());
			listaMapas.push_back(mapaPropiedades);
		}
		if (serieTreeItemId.IsOk()) {
			GNC::GUI::wxPropiedades::TMapaPropiedades mapaPropiedades;
			mapaPropiedades[_Std("Series Instance UID")] = std::string(m_pTreeListResultados->GetItemText(serieTreeItemId, COLUMNA_UID).ToUTF8());
			mapaPropiedades[_Std("Modality")] = std::string(m_pTreeListResultados->GetItemText(serieTreeItemId, COLUMNA_MODALIDAD).ToUTF8());
			mapaPropiedades[_Std("Series date")] = std::string(m_pTreeListResultados->GetItemText(serieTreeItemId, COLUMNA_FECHA).ToUTF8());

			listaMapas.push_back(mapaPropiedades);
		}

		GNC::GUI::wxPropiedades* pProp = new GNC::GUI::wxPropiedades(this, _Std("Search Results"), listaMapas);
		pProp->Show();

	}

}

void GNC::GUI::DialogoAdquisicion::ShowAnimation(bool show)
{
	m_pAnimation->Show(show);
	m_pPanelPACS->Layout();
	m_pPanelPACS->Refresh(true);
	if (show) {
		m_pAnimation->Play();
	} else {
		m_pAnimation->Stop();
	}
}

void GNC::GUI::DialogoAdquisicion::LimpiarBusquedas(bool /*lock*/) {

	/*
	GNC::GCS::ILocker* pLocker = NULL;
	if (lock )
	pLocker = new GNC::GCS::ILocker(m_TreeListResultadosLocker);
	*/

	m_pTreeListResultados->DeleteRoot();
	m_pTreeListResultados->AddRoot(wxT("Top"));
	m_pModeloDicom = GnkPtr<IModeloDicom>(new IModeloDicom());

	/*
	if (pLocker != NULL) {
	delete pLocker;
	}
	*/

}

//endregion


//---------------------------------------------------------------------------
//region realizacion de la interfaz IComandoPACSNotificador
void GNC::GUI::DialogoAdquisicion::PACSCargarListado(IModeloDicom * pModelo)
{
	{
		//GNC::GCS::ILocker locker(m_TreeListResultadosLocker);
		m_pTextControlField->ShowCancelButton(false);
		ShowAnimation(false);
		if(m_pComandoPACS == NULL)
		{
			return;
		}

		if (m_pComandoPACS->m_pPACSParams->m_Ambito == GADAPI::ComandoPACSParams::TA_Serie)
		{
			wxTreeItemId pacienteTreeId;
			wxTreeItemId estudioTreeId;

			m_lastExpanded.Unset();
			//actualizar el arbol
			const IModeloEstudio* estudio = NULL;
			pModelo->BuscarEstudio(m_pComandoPACS->m_pPACSParams->m_studyUID,&estudio);

			m_pComandoPACS = NULL;

			if (estudio != NULL) {
				std::set<std::string> listaModalidad;
				{//begin freeze/thaw
					Freeze();
					Refresh(true);
					{
						bool encontrado = false;
						wxString wxUIDEstudio = wxString::FromUTF8(estudio->GetUID().c_str());

						std::list<wxTreeItemId> p_items = m_pTreeListResultados->GetPublicChildren(m_pTreeListResultados->GetRootItem());
						for (std::list<wxTreeItemId>::iterator ititem = p_items.begin(); ititem != p_items.end() && !encontrado; ititem++) {
							pacienteTreeId = *ititem;

							std::list<wxTreeItemId> e_items = m_pTreeListResultados->GetPublicChildren(pacienteTreeId);
							for (std::list<wxTreeItemId>::iterator ititem = e_items.begin(); ititem != e_items.end() && !encontrado; ititem++) {
								estudioTreeId = *ititem;

								if(m_pTreeListResultados->GetItemText(estudioTreeId,COLUMNA_UID) == wxUIDEstudio) {
									encontrado = true;
									break;
								}
							}
						}
					}

					if(estudioTreeId.IsOk() && estudio->ListaSeries().size() > 0) {
						m_pTreeListResultados->DeleteChildren(estudioTreeId);
						bool forceInit = true;
						bool questionMade = false;

						for (IModeloEstudio::ListaSeriesType::const_iterator it3 = estudio->ListaSeries().begin(); it3 != estudio->ListaSeries().end(); it3++) {
							const IModeloSerie& s = *it3;
							wxString uidSerie = wxString::FromUTF8(s.GetUID().c_str());
							wxString numeroDeImagenes = wxString::FromUTF8(s.GetNumero().c_str());
							wxString tipoSerie = wxString::FromUTF8(s.GetTipo().c_str());
							wxString descripcionSerie = wxString::FromUTF8(s.GetDescripcion().c_str());

							wxDateTime fecha;
							wxDateTime hora;
							hora.ParseFormat( wxString::FromUTF8( (s.GetHora()).c_str() ).GetData(), wxT("%H%M%S"), wxDefaultDateTime);
							fecha.ParseFormat( wxString::FromUTF8( (s.GetFecha()).c_str() ).GetData(), wxT("%Y%m%d"), wxDefaultDateTime);

							wxString fechaStr;

							if(fecha.IsValid()){
								if (hora.IsValid()) {

									fecha.SetHour(hora.GetHour());
									fecha.SetMinute(hora.GetMinute());
									fecha.SetSecond(hora.GetSecond());

									fechaStr = wxString(fecha.Format(_("%m/%d/%Y %H:%M:%S"), wxDateTime::TimeZone(wxDateTime::GMT1)));
								}
								else {
									fechaStr = wxString(fecha.Format(_("%m/%d/%Y 00:00:00"), wxDateTime::TimeZone(wxDateTime::GMT1)));
								}
							}
							else {
								fechaStr = _("00/00/0000 00:00:00");
							}


							wxTreeItemId serieTreeId = m_pTreeListResultados->AppendItem(estudioTreeId, descripcionSerie);
							m_pTreeListResultados->SetItemText(serieTreeId, COLUMNA_MODALIDAD, tipoSerie);

							m_pTreeListResultados->SetItemText(serieTreeId, COLUMNA_FECHA, fechaStr);

							m_pTreeListResultados->SetItemText(serieTreeId, COLUMNA_MEDICO, wxString::FromUTF8(s.GetDoctor().c_str()));
							m_pTreeListResultados->SetItemText(serieTreeId, COLUMNA_UID_VISIBLE, uidSerie);
							m_pTreeListResultados->SetItemText(serieTreeId, COLUMNA_UID, uidSerie);
							m_pTreeListResultados->SetItemText(serieTreeId, COLUMNA_ACCNUMBER, wxString::FromUTF8(estudio->GetAccNumber().c_str()));

							m_pTreeListResultados->SetItemImage(serieTreeId,ICONO_SERIE,wxTreeItemIcon_Normal);
							m_pTreeListResultados->SetItemImage(serieTreeId,ICONO_SERIE,wxTreeItemIcon_Selected);
							m_pTreeListResultados->SetItemImage(serieTreeId,ICONO_SERIE,wxTreeItemIcon_Expanded);
							m_pTreeListResultados->SetItemImage(serieTreeId,ICONO_SERIE,wxTreeItemIcon_SelectedExpanded);

							if (m_AutoAddSeries) {

								std::string uidPaciente = std::string(m_pTreeListResultados->GetItemText(pacienteTreeId, COLUMNA_UID).ToUTF8());
								std::string descPaciente = std::string(m_pTreeListResultados->GetItemText(pacienteTreeId).ToUTF8());

								std::string uidSerie = std::string(m_pTreeListResultados->GetItemText(serieTreeId, COLUMNA_UID).ToUTF8());
								std::string descSerie = std::string( (m_pTreeListResultados->GetItemText(serieTreeId) + _T("(") + m_pTreeListResultados->GetItemText(serieTreeId, COLUMNA_MODALIDAD) + _T(")") ).ToUTF8());

								std::string modalidadSerie = std::string(m_pTreeListResultados->GetItemText(serieTreeId, COLUMNA_MODALIDAD).ToUTF8());

								if( GNC::GCS::ControladorPermisos::Instance()->Get("core.pacs.download", modalidadSerie) ) {
									if (!questionMade && m_pDescargasGridTable->FindDescarga(uidSerie) != -1) {
										Thaw();
										forceInit = wxMessageBox(_("Series were already in the download queue.\nDo you want to force download?"),_("Download warning"),wxICON_INFORMATION|wxYES_NO) == wxYES;
										Freeze();
										questionMade = true;
									}

									bool downloaded = m_pDescargasGridTable->AddDescarga(GetServerSeleccionado(), descPaciente, descSerie, uidSerie, modalidadSerie,forceInit, true);
									m_pTreeListResultados->SetItemBold(serieTreeId, downloaded);
								} else {
									listaModalidad.insert(modalidadSerie);
								}
							}

						}
						if (m_AutoAddSeries) {
							m_AutoAddSeries = false;
							m_AutoAddSeriesItem = NULL;
						}
						m_pTreeListResultados->SortChildren(estudioTreeId);
						m_pTreeListResultados->Expand(estudioTreeId);

					} else if (estudioTreeId.IsOk()) {
						m_pTreeListResultados->Collapse(estudioTreeId);
					}
					Thaw();
				} //end freeze

				if ( listaModalidad.size() > 0) {
					std::ostringstream ostr;
					ostr << _Std("You are not allowed to download this kind of modalities (");
					bool primero = true;
					for (std::set<std::string>::iterator it = listaModalidad.begin(); it != listaModalidad.end(); ++it) {
						if (primero) {
							ostr << (*it);
							primero = false;
						} else {
							ostr << ", " << (*it);
						}
					}
					ostr << ")";
					wxMessageBox(wxString::FromUTF8( ostr.str().c_str() ), _("Modality download error"), wxICON_ERROR);
				}
			}
		}
		else
		{
			//rellenar el arbol


			wxTreeItemId pacienteTreeId;
			wxTreeItemId estudioTreeId;
			wxTreeItemId serieTreeId;

			if (m_pComandoPACS->m_pPACSParams->m_error.size() > 0) {
				m_pComandoPACS = NULL;
				return;
			}
			else if(pModelo->ListaPacientes().size() == 0){
				if (!m_pComandoPACS->EstaAbortado()) {
					LimpiarBusquedas(false);
					wxTreeItemId topTreeId = m_pTreeListResultados->GetRootItem();
					if (! topTreeId || ! topTreeId.IsOk() ) {
						topTreeId = m_pTreeListResultados->AddRoot(wxT("Top"));
					}
					m_pTreeListResultados->AppendItem(topTreeId, _("No results found"));
				}
				m_pComandoPACS = NULL;
				return;
			}

			m_pComandoPACS = NULL;

			LimpiarBusquedas(false);

			wxTreeItemId topTreeId = m_pTreeListResultados->GetRootItem();
			if (! topTreeId || ! topTreeId.IsOk() ) {
				topTreeId = m_pTreeListResultados->AddRoot(wxT("Top"));
			}

			if (!topTreeId) {
				m_pComandoPACS = NULL;
				return;
			}



			{

				wxWindowDisabler disableAll;
				Freeze();
				//Refresh(true);

				for (IModeloDicom::ListaPacientesType::const_iterator it = pModelo->ListaPacientes().begin(); it != pModelo->ListaPacientes().end(); it++) {
					const IModeloPaciente& p = *it;
					std::string pacienteNombre = p.GetNombre() + " (" + p.GetUID() + ")";
					wxString nombrePaciente = wxString::FromUTF8(pacienteNombre.c_str());
					wxString uidPaciente = wxString::FromUTF8(p.GetUID().c_str());
					//columnas

					bool notfound = true;
					wxTreeItemId tid = topTreeId;
					std::list<wxTreeItemId> p_items = m_pTreeListResultados->GetPublicChildren(topTreeId);
					for (std::list<wxTreeItemId>::iterator ititem = p_items.begin(); ititem != p_items.end(); ititem++) {
						wxTreeItemId cid = *ititem;
						if (
							m_pTreeListResultados->GetItemText(cid) == nombrePaciente &&
							m_pTreeListResultados->GetItemText(cid, COLUMNA_UID) == uidPaciente
							) {
								notfound = false;
								pacienteTreeId = cid;
						}
					}

					if (notfound) {
						pacienteTreeId = m_pTreeListResultados->AppendItem(topTreeId, nombrePaciente);
						m_pTreeListResultados->SetItemText(pacienteTreeId, COLUMNA_UID, uidPaciente);
						//
						// Estilo
						m_pTreeListResultados->SetItemBold(pacienteTreeId, true);
						m_pTreeListResultados->SetItemTextColour(pacienteTreeId, *wxBLUE);

						//iconos
						int icono = 0;
						if(p.GetSexo() == "M") {
							icono = ICONO_HOMBRE;
						} else if(p.GetSexo() == "F") {
							icono = ICONO_MUJER;
						} else {
							icono = ICONO_OTRO;
						}

						m_pTreeListResultados->SetItemImage(pacienteTreeId,icono,wxTreeItemIcon_Normal);
						m_pTreeListResultados->SetItemImage(pacienteTreeId,icono,wxTreeItemIcon_Selected);
						m_pTreeListResultados->SetItemImage(pacienteTreeId,icono,wxTreeItemIcon_Expanded);
						m_pTreeListResultados->SetItemImage(pacienteTreeId,icono,wxTreeItemIcon_SelectedExpanded);
					}

					if (!pacienteTreeId) {
						continue;
					}
					//ESTUDIOS
					for (IModeloPaciente::ListaEstudiosType::const_iterator it2 = p.ListaEstudios().begin(); it2 != p.ListaEstudios().end(); it2++) {
						const IModeloEstudio& e = *it2;
						wxDateTime fecha;
						wxDateTime hora;
						hora.ParseFormat( wxString::FromUTF8( (e.GetHora()).c_str() ).GetData(), wxT("%H%M%S"), wxDefaultDateTime);
						fecha.ParseFormat( wxString::FromUTF8( (e.GetFecha()).c_str() ).GetData(), wxT("%Y%m%d"), wxDefaultDateTime);

						wxString modalidadEstudio = wxString::FromUTF8(e.GetModalidad().c_str());
						wxString uidEstudio = wxString::FromUTF8(e.GetUID().c_str());
						wxString descripcionEstudio = wxString::FromUTF8(e.GetDescripcion().c_str());
						wxString AccNumber = wxString::FromUTF8(e.GetAccNumber().c_str());
						wxString medicoEstudio = wxString::FromUTF8(e.GetDoctor().c_str());

						wxString fechaStr;

						if(fecha.IsValid()){
							if (hora.IsValid()) {

								fecha.SetHour(hora.GetHour());
								fecha.SetMinute(hora.GetMinute());
								fecha.SetSecond(hora.GetSecond());

								fechaStr = wxString(fecha.Format(_("%m/%d/%Y %H:%M:%S"), wxDateTime::TimeZone(wxDateTime::GMT1)));
							}
							else {
								fechaStr = wxString(fecha.Format(_("%m/%d/%Y 00:00:00"), wxDateTime::TimeZone(wxDateTime::GMT1)));
							}
						}
						else {
							fechaStr = _("00/00/0000 00:00:00");
						}

						std::list<wxTreeItemId> e_items = m_pTreeListResultados->GetPublicChildren(pacienteTreeId);
						for (std::list<wxTreeItemId>::iterator ititem = e_items.begin(); ititem != e_items.end(); ititem++) {
							wxTreeItemId cid = *ititem;
							if (
								m_pTreeListResultados->GetItemText(cid) == descripcionEstudio &&
								m_pTreeListResultados->GetItemText(cid, COLUMNA_MODALIDAD) == modalidadEstudio&&
								m_pTreeListResultados->GetItemText(cid, COLUMNA_FECHA) == fechaStr &&
								m_pTreeListResultados->GetItemText(cid, COLUMNA_MEDICO) == medicoEstudio &&
								m_pTreeListResultados->GetItemText(cid, COLUMNA_UID_VISIBLE) == uidEstudio &&
								m_pTreeListResultados->GetItemText(cid, COLUMNA_UID) == uidEstudio &&
								m_pTreeListResultados->GetItemText(cid, COLUMNA_ACCNUMBER) == AccNumber
								) {
									notfound = false;
									estudioTreeId = cid;
							}
						}

						if (notfound) {
							//columnas
							estudioTreeId = m_pTreeListResultados->AppendItem(pacienteTreeId, descripcionEstudio);
							m_pTreeListResultados->SetItemText(estudioTreeId, COLUMNA_MODALIDAD, modalidadEstudio);
							m_pTreeListResultados->SetItemText(estudioTreeId, COLUMNA_FECHA, fechaStr);

							m_pTreeListResultados->SetItemText(estudioTreeId, COLUMNA_MEDICO, medicoEstudio);
							m_pTreeListResultados->SetItemText(estudioTreeId, COLUMNA_UID_VISIBLE, uidEstudio);
							m_pTreeListResultados->SetItemText(estudioTreeId, COLUMNA_UID, uidEstudio);
							m_pTreeListResultados->SetItemText(estudioTreeId, COLUMNA_ACCNUMBER, AccNumber);
							//

							//ICONOS...
							m_pTreeListResultados->SetItemImage(estudioTreeId,ICONO_ESTUDIO,wxTreeItemIcon_Normal);
							m_pTreeListResultados->SetItemImage(estudioTreeId,ICONO_ESTUDIO,wxTreeItemIcon_Selected);
							m_pTreeListResultados->SetItemImage(estudioTreeId,ICONO_ESTUDIO,wxTreeItemIcon_Expanded);
							m_pTreeListResultados->SetItemImage(estudioTreeId,ICONO_ESTUDIO,wxTreeItemIcon_SelectedExpanded);
						}

						if (!estudioTreeId) {
							continue;
						}
						
						if (m_Mode == TD_SERIES) 
						{
							for (IModeloEstudio::ListaSeriesType::const_iterator it3 = e.ListaSeries().begin(); it3 != e.ListaSeries().end(); it3++) {
								const IModeloSerie& s = *it3;
								wxString uidSerie = wxString::FromUTF8(s.GetUID().c_str());
								wxString numeroDeImagenes = wxString::FromUTF8(s.GetNumero().c_str());
								wxString tipoSerie = wxString::FromUTF8(s.GetTipo().c_str());
								wxString descripcionSerie = wxString::FromUTF8(s.GetDescripcion().c_str());
								wxString medicoSerie = wxString::FromUTF8(s.GetDoctor().c_str());
								wxDateTime fechaSerie;
								wxDateTime horaSerie;
								hora.ParseFormat( wxString::FromUTF8( (s.GetHora()).c_str() ).GetData(), wxT("%H%M%S"), wxDefaultDateTime);
								fecha.ParseFormat( wxString::FromUTF8( (s.GetFecha()).c_str() ).GetData(), wxT("%Y%m%d"), wxDefaultDateTime);

								wxString fechaSerieStr;

								if(fechaSerie.IsValid()){
									if (horaSerie.IsValid()) {
										fechaSerie.SetHour(horaSerie.GetHour());
										fechaSerie.SetMinute(horaSerie.GetMinute());
										fechaSerie.SetSecond(horaSerie.GetSecond());

										fechaSerieStr = wxString(fechaSerie.Format(_("%m/%d/%Y %H:%M:%S"), wxDateTime::TimeZone(wxDateTime::GMT1)));
									}
									else {
										fechaSerieStr = wxString(fechaSerie.Format(_("%m/%d/%Y 00:00:00"), wxDateTime::TimeZone(wxDateTime::GMT1)));
									}
								}
								else {
									fechaSerieStr = _("00/00/0000 00:00:00");
								}

								std::list<wxTreeItemId> s_items = m_pTreeListResultados->GetPublicChildren(estudioTreeId);
								for (std::list<wxTreeItemId>::iterator ititem = s_items.begin(); ititem != s_items.end(); ititem++) {
									wxTreeItemId cid = *ititem;

									if (
										m_pTreeListResultados->GetItemText(cid) == descripcionSerie &&
										m_pTreeListResultados->GetItemText(cid, COLUMNA_MODALIDAD) == tipoSerie &&
										m_pTreeListResultados->GetItemText(cid, COLUMNA_FECHA) == fechaSerieStr &&
										m_pTreeListResultados->GetItemText(cid, COLUMNA_MEDICO) == medicoSerie &&
										m_pTreeListResultados->GetItemText(cid, COLUMNA_UID_VISIBLE) == uidSerie &&
										m_pTreeListResultados->GetItemText(cid, COLUMNA_UID) == uidSerie

										) {
											notfound = false;
											serieTreeId = cid;
									}
								}
								if (notfound) {

									serieTreeId = m_pTreeListResultados->AppendItem(estudioTreeId, descripcionSerie);

									m_pTreeListResultados->SetItemText(serieTreeId, COLUMNA_MODALIDAD, tipoSerie);
									m_pTreeListResultados->SetItemText(serieTreeId, COLUMNA_FECHA, fechaSerieStr);

									m_pTreeListResultados->SetItemText(serieTreeId, COLUMNA_MEDICO, medicoSerie);
									m_pTreeListResultados->SetItemText(serieTreeId, COLUMNA_UID_VISIBLE, uidSerie);
									m_pTreeListResultados->SetItemText(serieTreeId, COLUMNA_UID, uidSerie);
									m_pTreeListResultados->SetItemText(serieTreeId, COLUMNA_ACCNUMBER, AccNumber);

									m_pTreeListResultados->SetItemImage(serieTreeId,ICONO_SERIE,wxTreeItemIcon_Normal);
									m_pTreeListResultados->SetItemImage(serieTreeId,ICONO_SERIE,wxTreeItemIcon_Selected);
									m_pTreeListResultados->SetItemImage(serieTreeId,ICONO_SERIE,wxTreeItemIcon_Expanded);
									m_pTreeListResultados->SetItemImage(serieTreeId,ICONO_SERIE,wxTreeItemIcon_SelectedExpanded);
								}
							}
							if(e.ListaSeries().size() == 0)
							{
								wxTreeItemId serieTreeId = m_pTreeListResultados->AppendItem(estudioTreeId, _("Searching..."));
							}
						}//end TD_SERIES
					}
				}
				m_pTreeListResultados->SortChildren(m_pTreeListResultados->GetRootItem());
				Thaw();
			}
		}//else rellenar el arbol

		m_pTreeListResultados->Refresh(false);
		//m_pBotonBusqueda->SetAutoLayout(true);
	}
}


void GNC::GUI::DialogoAdquisicion::ProcesarEvento(GNC::GCS::Eventos::IEvento *evt)
{
	GNC::GCS::Eventos::EventoProgresoComando* pEvt = dynamic_cast<GNC::GCS::Eventos::EventoProgresoComando*> (evt);
	if (pEvt == NULL  || pEvt->GetComando() == NULL || pEvt->GetComando() != m_pComandoPACS) {
		return;
	}
	switch (pEvt->GetTipo()) {
case GNC::GCS::Eventos::EventoProgresoComando::TEP_Iniciado:
	break;
case GNC::GCS::Eventos::EventoProgresoComando::TEP_Progreso:
	break;
case GNC::GCS::Eventos::EventoProgresoComando::TEP_Finalizado:
	m_pTextControlField->ShowCancelButton(false);
	ShowAnimation(false);
	m_pComandoPACS = NULL;
	break;
case GNC::GCS::Eventos::EventoProgresoComando::TEP_Unknown:
	break;
	}
}

//endregion
