/*
 *
 *  $Id: dicomnetclient.h 3526 2011-03-16 19:56:19Z carlos $
 *  Ginkgo CADx Project
 *
 *  Copyright 2008-10 MetaEmotion S.L. All rights reserved.
 *  http://ginkgo-cadx.com
 *
 *  This file is licensed under LGPL v3 license.
 *  See License.txt for details
 *
 *  Code adapted from Aeskulap
 *
 */
#pragma once
#include <api/icontroladorlog.h>
#include <api/iproxynotificadorprogreso.h>
#include <main/controllers/controladorlog.h>
#include "dicomnetwork.h"
#include "helpers.h"

#include <iostream>

#ifdef verify
#define MACRO_QUE_ESTORBA verify
#undef verify
#endif
#include <dcmtk/dcmdata/dcdatset.h>
#include <dcmtk/dcmdata/dcdeftag.h>
#include <api/imodelodicom.h>

namespace GNC{
	class IProxyNotificadorProgreso;
}

#ifdef MACRO_QUE_ESTORBA
#define verify MACRO_QUE_ESTORBA
#endif

typedef enum CallbackType {
	CT_FindPacientes,
	CT_FindEstudios,
	CT_FindSeries,
	CT_FindImagenes,
	CT_MoveEstudio,
	CT_MoveSerie,
	CT_MoveImagen,
	CT_None
} CallbackType;

template<class T>
class NetClient : public T {
private:
	GIL::DICOM::DCMTK::Network* Net;
	std::string                 ambitolog;

protected:
	GNC::IProxyNotificadorProgreso* m_pNotificadorProgreso;
public:
	NetClient(void* connectionKey, const std::string& ambitolog, GNC::IProxyNotificadorProgreso* pNotificadorProgreso = NULL) : T(ambitolog) {

		if (connectionKey == NULL) {
			LOG_ERROR(ambitolog, "NULL Connection key");
		}
		this->Net = GIL::DICOM::DCMTK::Network::Instance(connectionKey);
		this->m_pNotificadorProgreso = pNotificadorProgreso;
		this->ambitolog = ambitolog;
	};

	bool QueryServer(DcmDataset* query, DicomServer* server, IModeloDicom* pModelo, const std::string& local_aet, CallbackType callbackType) {
		if (server == NULL || pModelo == NULL) {
			return false;
		}

		if (!Net->Initiallized()) {
			Net->InitializeNetwork(T::GetTimeout(), T::GetRole(), T::GetAcceptorPort());
		}

		T::Create(
			server->AET,
			server->HostName,
			server->Port,
			local_aet
		);

		T::SetNotificadorProgreso(m_pNotificadorProgreso);

		LOG_DEBUG(ambitolog, "Conectando: AET = " << server->AET << ", Host = " << server->HostName << ", Puerto = " << server->Port << ", Local AET = " << local_aet << ", PDU = " << server->PDU);
		CONDITION r = T::Connect(Net, server->PDU);

		CONDITION c = ECC_Normal;

		if (r.good() == true) {
			LOG_TRACE(ambitolog, "Enviando Objeto:" << std::endl << DumpDataset(query));
			c = T::SendObject(query);
		}
		else {
			LOG_ERROR(ambitolog, "Error al conectar:" << r.text());
			T::Drop();
			T::Destroy();
			throw GIL::DICOM::PACSException(r.text());
		}

		if (!c.good()) {
			LOG_ERROR(ambitolog, "Error al enviar objeto: " << c.text());
			T::Drop();
			T::Destroy();
			throw GIL::DICOM::PACSException(c.text());
		}

		LOG_DEBUG(ambitolog, "Cerrando asociación");
		T::Drop();

		DcmStack* result = T::GetResultStack();
		if (r.good() && c.good() && result != NULL && result->card() > 0) {
			LOG_DEBUG(ambitolog, "Número de resultados: " << result->card());

			for (unsigned long i = 0; i < result->card(); ++i) {
				DcmDataset* dset = new DcmDataset(*(DcmDataset*) result->elem(i));
				switch (callbackType) {
					case CT_FindPacientes:
						OnBuscarPacientes(dset, pModelo);
						break;
					case CT_FindEstudios:
						OnBuscarEstudios(dset, pModelo);
						break;
					case CT_FindSeries:
						OnBuscarSeries(dset, pModelo);
						break;
					case CT_FindImagenes:
						OnBuscarImagenes(dset, pModelo);
						break;
					case CT_MoveEstudio:
						OnMoverEstudio(dset, pModelo);
						break;
					case CT_MoveSerie:
						OnMoverSerie(dset, pModelo);
						break;
					case CT_MoveImagen:
						OnMoverImagen(dset, pModelo);
						break;
					case CT_None:
						break;
				}
				delete dset;
			}
		}
		else {
			if (r.bad()) {
				LOG_ERROR(ambitolog, "Error en la conexión: " << r.text());
				return false;
			}
			if (c.bad()) {
				LOG_ERROR(ambitolog, "Error en la obtención: " << c.text());
				return false;
			}
		}
		return r.good();
	}

	wxCSConv GetConv(DcmDataset* dset)
	 {
		 OFString Charset;
		 wxCSConv wxConv(wxFONTENCODING_UTF8);
		 if (dset->findAndGetOFString(DCM_SpecificCharacterSet, Charset).good()) {
			 if (Charset == "ISO_IR 192" || Charset == "ISO_IR 6" || Charset == "ISO_IR 138") {
				 wxConv = wxCSConv(wxFONTENCODING_UTF8);
			 }
			 else if (Charset == "ISO_IR 100") {
				 wxConv = wxCSConv(wxFONTENCODING_ISO8859_1);
			 }
			 else if (Charset == "ISO_IR 101") {
				 wxConv = wxCSConv(wxFONTENCODING_ISO8859_2);
			 }
			 else if (Charset == "ISO_IR 109") {
				 wxConv = wxCSConv(wxFONTENCODING_ISO8859_3);
			 }
			 else if (Charset == "ISO_IR 110") {
				 wxConv = wxCSConv(wxFONTENCODING_ISO8859_4);
			 }
			 else if (Charset == "ISO_IR 148") {
				 wxConv = wxCSConv(wxFONTENCODING_ISO8859_9);
			 }
			 else if (Charset == "ISO_IR 144") {
				 wxConv = wxCSConv(wxFONTENCODING_ISO8859_5);
			 }
			 else if (Charset == "ISO_IR 127") {
				 wxConv = wxCSConv(wxFONTENCODING_ISO8859_6);
			 }
			 else if (Charset == "ISO_IR 126") {
				 wxConv = wxCSConv(wxFONTENCODING_ISO8859_7);
			 }
			 else if (Charset == "ISO_IR 138") {
				 wxConv = wxCSConv(wxFONTENCODING_ISO8859_8);
			 }
		 }
		 return wxConv;
	 }

	void OnBuscarPacientes(DcmDataset* dset, IModeloDicom* pModelo)
	{
		wxCSConv wxConv = GetConv(dset);

		OFString OFPacienteUID;
		if (dset->findAndGetOFString(DCM_PatientID, OFPacienteUID).bad()) {
			LOG_ERROR(ambitolog, "El PACS no devolvió el PatientID del paciente. Entrada ignorada.");
			return;
		}

		std::string PacienteUID;
		if (dset->findAndGetOFString(DCM_PatientID, OFPacienteUID).good()) {
			PacienteUID = wxString(OFPacienteUID.c_str(), wxConv).ToUTF8();
		}

		OFString OFPacienteNombre;
		std::string PacienteNombre;
		if (dset->findAndGetOFString(DCM_PatientName, OFPacienteNombre).good()) {
			PacienteNombre = wxString(OFPacienteNombre.c_str(), wxConv).ToUTF8();
		}

		OFString OFPacienteFechaNacimiento;
		std::string PacienteFechaNacimiento;
		if (dset->findAndGetOFString(DCM_PatientBirthDate, OFPacienteFechaNacimiento).good()) {
			PacienteFechaNacimiento = wxString(OFPacienteFechaNacimiento.c_str(), wxConv).ToUTF8();
		}

		OFString OFPacienteSexo;
		std::string PacienteSexo;
		if (dset->findAndGetOFString(DCM_PatientSex, OFPacienteSexo).good()) {
			PacienteSexo = wxString(OFPacienteSexo.c_str(), wxConv).ToUTF8();
		}

		pModelo->InsertarPaciente(PacienteUID, PacienteNombre, PacienteFechaNacimiento, PacienteSexo);
	}

	void OnBuscarSeries(DcmDataset* dset, IModeloDicom *pModelo)
	{
		wxCSConv wxConv = GetConv(dset);

		OFString OFEstudioUID;
		OFString OFSerieUID;
		if (dset->findAndGetOFString(DCM_StudyInstanceUID, OFEstudioUID).bad() || dset->findAndGetOFString(DCM_SeriesInstanceUID, OFSerieUID).bad()) {
			LOG_ERROR(ambitolog, "El PACS no devolvió el StudyInstanceUID o el SeriesInstanceUID de la serie. Entrada ignorada.");
			return;
		}

		std::string EstudioUID;
		EstudioUID = wxString(OFEstudioUID.c_str(), wxConv).ToUTF8();

		std::string SerieUID;
		SerieUID = wxString(OFSerieUID.c_str(), wxConv).ToUTF8();

		OFString OFSerieTipo;
		std::string SerieTipo;
		if (dset->findAndGetOFString(DCM_Modality, OFSerieTipo).good()) {
			SerieTipo = wxString(OFSerieTipo.c_str(), wxConv).ToUTF8();
		}

		OFString OFSerieFecha;
		std::string SerieFecha;
		if (dset->findAndGetOFString(DCM_SeriesDate, OFSerieFecha).good()) {
			SerieFecha = wxString(OFSerieFecha.c_str(), wxConv).ToUTF8();
		}

		OFString OFSerieHora;
		std::string SerieHora;
		if (dset->findAndGetOFString(DCM_SeriesTime, OFSerieHora).good()) {
			SerieHora = wxString(OFSerieHora.c_str(), wxConv).ToUTF8();
		}

		OFString OFSerieDescripcion;
		std::string SerieDescripcion;
		if (dset->findAndGetOFString(DCM_SeriesDescription, OFSerieDescripcion).good()) {
			SerieDescripcion = wxString(OFSerieDescripcion.c_str(), wxConv).ToUTF8();
		}

		OFString OFDoctor;
		std::string SerieDoctor;
		if (dset->findAndGetOFString(DCM_ReferringPhysicianName, OFDoctor).good()) {
			SerieDoctor = wxString(OFDoctor.c_str(), wxConv).ToUTF8();
		}

		OFString OFNumber;
		std::string SerieNumero;
		if (dset->findAndGetOFString(DCM_NumberOfSeriesRelatedInstances, OFNumber).good()) {
			SerieNumero = wxString(OFNumber.c_str(), wxConv).ToUTF8();
		}

		pModelo->InsertarSerie(EstudioUID, SerieUID, SerieTipo, SerieFecha, SerieHora, SerieDescripcion,SerieNumero,SerieDoctor);
	}

    void OnBuscarImagenes(DcmDataset *dset, IModeloDicom* pModelo) {
    	OFString OFSerieUID;
    	OFString OFImagenUID;
		if (dset->findAndGetOFString(DCM_SeriesInstanceUID, OFSerieUID).bad() || dset->findAndGetOFString(DCM_SOPInstanceUID, OFImagenUID).bad()) {
			LOG_ERROR(ambitolog, "El PACS no devolvió el SeriesInstanceUID o el SOPInstanceUID de la imagen. Entrada ignorada.");
			return;
		}

		std::string SerieUID;
		std::string ImagenUID;

		SerieUID.assign(OFSerieUID.c_str());
		ImagenUID.assign(OFImagenUID.c_str());

		pModelo->InsertarImagen(SerieUID, ImagenUID);

	}

    void OnBuscarEstudios(DcmDataset *dset, IModeloDicom *pModelo) {
		OFString OFEstudioUID;
		if (dset->findAndGetOFString(DCM_StudyInstanceUID, OFEstudioUID).bad()) {
			LOG_ERROR(ambitolog, "El PACS no devolvió el StudyInstanceUID del estudio. Entrada ignorada.");
			return;
		}

		wxCSConv wxConv = GetConv(dset);

		OFString OFPacienteUID;
		std::string PacienteUID;
		if (dset->findAndGetOFString(DCM_PatientID, OFPacienteUID).good()) {
			PacienteUID = wxString(OFPacienteUID.c_str(), wxConv).ToUTF8();
		}

		OFString OFPacienteNombre;
		std::string PacienteNombre;
		if (dset->findAndGetOFString(DCM_PatientName, OFPacienteNombre).good()) {
			PacienteNombre = wxString(OFPacienteNombre.c_str(), wxConv).ToUTF8();
		}

		OFString OFPacienteFechaNacimiento;
		std::string PacienteFechaNacimiento;

		if (dset->findAndGetOFString(DCM_PatientBirthDate, OFPacienteFechaNacimiento).good()) {
			PacienteFechaNacimiento = wxString(OFPacienteFechaNacimiento.c_str(), wxConv).ToUTF8();
		}

		OFString OFPacienteSexo;
		std::string PacienteSexo;
		if (dset->findAndGetOFString(DCM_PatientSex, OFPacienteSexo).good()) {
			PacienteSexo = wxString(OFPacienteSexo.c_str(), wxConv).ToUTF8();
		}

		pModelo->InsertarPaciente(PacienteUID, PacienteNombre, PacienteFechaNacimiento, PacienteSexo);

		std::string EstudioUID;
		if (dset->findAndGetOFString(DCM_StudyInstanceUID, OFEstudioUID).good()) {
			EstudioUID = wxString(OFEstudioUID.c_str(), wxConv).ToUTF8();
		}

		OFString OFEAccNumber;
		std::string AccNumber;
		if (dset->findAndGetOFString(DCM_AccessionNumber, OFEAccNumber).good()) {
			AccNumber = wxString(OFEAccNumber.c_str(), wxConv).ToUTF8();
		}

		OFString OFEstudioDescripcion;
		std::string EstudioDescripcion;
		if (dset->findAndGetOFString(DCM_StudyDescription, OFEstudioDescripcion).good()) {
			EstudioDescripcion = wxString(OFEstudioDescripcion.c_str(), wxConv).ToUTF8();
		}

		OFString OFEstudioModalidad;
		std::string EstudioModalidad;
		if (dset->findAndGetOFStringArray(DCM_ModalitiesInStudy, OFEstudioModalidad).good()) {
			EstudioModalidad = wxString(OFEstudioModalidad.c_str(), wxConv).ToUTF8();
		}

		OFString OFEstudioFecha;
		std::string EstudioFecha;
		if (dset->findAndGetOFString(DCM_StudyDate, OFEstudioFecha).good()) {
			EstudioFecha = wxString(OFEstudioFecha.c_str(), wxConv).ToUTF8();
		}

		OFString OFEstudioHora;
		std::string EstudioHora;
		if (dset->findAndGetOFString(DCM_StudyTime, OFEstudioHora).good()) {
			EstudioHora = wxString(OFEstudioHora.c_str(), wxConv).ToUTF8();
		}

		OFString OFDoctor;
		std::string EstudioDoctor;
		if (dset->findAndGetOFString(DCM_ReferringPhysicianName, OFDoctor).good()) {
			EstudioDoctor = wxString(OFDoctor.c_str(), wxConv).ToUTF8();
		}


		pModelo->InsertarEstudio(PacienteUID, EstudioUID, AccNumber, EstudioDescripcion, EstudioModalidad, EstudioFecha, EstudioHora, EstudioDoctor);

    }

	void OnMoverEstudio(DcmDataset *WXUNUSED(dset), IModeloDicom *WXUNUSED(pModelo)) {
		#ifdef _GINKGO_DEBUG
		std::cout << "mover estudio" << std::endl;
		#endif
	}

    void OnMoverSerie(DcmDataset *WXUNUSED(dset), IModeloDicom *WXUNUSED(pModelo)) {
		#ifdef _GINKGO_DEBUG
		std::cout << "mover serie" << std::endl;
		#endif
    }

	void OnMoverImagen(DcmDataset *WXUNUSED(dset), IModeloDicom *WXUNUSED(pModelo)) {
		#ifdef _GINKGO_DEBUG
		std::cout << "mover imagen" << std::endl;
		#endif
    }

    OFBool writeToFile(const char* ofname, DcmDataset *dataset) {
		/* write out as a file format */

		DcmFileFormat fileformat(dataset); // copies dataset
		OFCondition ec = fileformat.error();
		if (ec.bad()) {
			LOG_ERROR(T::ambitolog, "No se puedo escribir el fichero (" << ofname << ": " << ec.text());
			return OFFalse;
		}

		ec = fileformat.saveFile(ofname, dataset->getOriginalXfer());
		if (ec.bad()) {
			LOG_ERROR(T::ambitolog, "No se puedo escribir el fichero (" << ofname << ": " << ec.text());
			return OFFalse;
		}
    }
};
