/*
*  
*  $Id: comandoexportacion.cpp 3527 2011-03-16 21:45:37Z carlos $
*  Ginkgo CADx Project
*
*  Copyright 2008-10 MetaEmotion S.L. All rights reserved.
*  http://ginkgo-cadx.com
*
*  This file is licensed under LGPL v3 license.
*  See License.txt for details
*
*/
#ifdef _MSC_VER
#pragma warning(push)
#pragma warning(disable: 4996)
#endif

#include <sstream>

#include <wx/filename.h>
#include <wx/file.h>
#include <wx/dir.h>
#include <wx/confbase.h>

#include <api/globals.h>
#include <api/ientorno.h>
#include <api/internacionalizacion.h>
#include <api/icontextoestudio.h>

#include <main/entorno.h>
#include <main/controllers/controladoreventos.h>

#include <eventos/eventosginkgo.h>

#include "comandoexportacion.h"

#include <itkExceptionObject.h>
#include <itkImage.h>
#include <itkOrientedImage.h>
#include <itkRGBPixel.h>
#include <itkGDCMImageIO.h>
#include <itkJPEGImageIO.h>
#include <itkPNGImageIO.h>
#include <itkVectorResampleImageFilter.h>
#include <itkLinearInterpolateImageFunction.h>
#include <itkImageFileWriter.h>



namespace GADAPI {

	ComandoExportacion::ComandoExportacion(ComandoExportacionParams* pParams): GNC::GCS::IComando(pParams,"Exportacion")
	{
		m_pExportacionParams = pParams;
	}

	void ComandoExportacion::Execute()
	{
		if (!NotificarProgreso(0.0, _Std("Exporting files...")) )
			return;
		if(m_pExportacionParams->m_pDatosPersistentes->m_formatoDestino == GNC::GUI::DICOM){
			ExportarDICOM();
		}
		else{
			ExportarImagenes();
		}
	}

	void ComandoExportacion::Update()
	{
		if (m_pExportacionParams->m_hasError)
		{
			GNC::GCS::ControladorEventos::Instance()->ProcesarEvento(new GNC::GCS::Eventos::EventoMensajes(NULL,m_pExportacionParams->m_Error, GNC::GCS::Eventos::EventoMensajes::StatusMessage,true, GNC::GCS::Eventos::EventoMensajes::Error));
		} else {
			GNC::GCS::ControladorEventos::Instance()->ProcesarEvento(new GNC::GCS::Eventos::EventoMensajes(NULL, _Std("Export has been finished sucessfully"), GNC::GCS::Eventos::EventoMensajes::StatusMessage,true, GNC::GCS::Eventos::EventoMensajes::Informacion));
		}
	}

	//endregion

	void ComandoExportacion::ExportarDICOM()
	{
		GnkPtr<GNC::GUI::TipoWizardExportacion> pDatosPersistentes = m_pExportacionParams->m_pDatosPersistentes;
		bool correcto = true;

		std::vector<std::string> resultado;
		std::vector<std::string> listaPaths;


		if (pDatosPersistentes->m_ficheroActual) {
			listaPaths.push_back(pDatosPersistentes->m_pVista->GetEstudio()->GetRutaDeImagenActiva());
			if(pDatosPersistentes->m_exportarDiagnostico) {
				listaPaths.push_back(pDatosPersistentes->m_pVista->GetEstudio()->GetRutaDeDiagnosticoActivo());
			}
		}
		else
		{
			listaPaths = pDatosPersistentes->m_pVista->GetRutasImagenes();
			if(pDatosPersistentes->m_exportarDiagnostico) {
				std::vector<std::string> diagnosticos;
				diagnosticos = pDatosPersistentes->m_pVista->GetRutasDiagnosticos();
				for(std::vector<std::string>::iterator it = diagnosticos.begin(); it!= diagnosticos.end(); it++) {
					listaPaths.push_back((*it));
				}
			}
		}

		GIL::DICOM::IDICOMManager*	pDICOMManager;

		wxString wxPath = FROMPATH(pDatosPersistentes->m_pathDestino);
		wxString destino(wxT(""));

		int contador = 1;
		for(std::vector<std::string>::iterator it = listaPaths.begin(); it!= listaPaths.end(); ++it,++contador) {
			wxString cadena = wxString::Format(_("Exporting file %d of %d"), contador, listaPaths.size());
			if (!NotificarProgreso((float)contador/listaPaths.size(), std::string(cadena.ToUTF8())))
				return;

			GIL::DICOM::TipoJerarquia base;

			pDICOMManager = GNC::Entorno::Instance()->GetControladorImportacionPACS()->CrearInstanciaDeDICOMManager();
			pDICOMManager->CargarFichero((*it), base);

			pDICOMManager->ActualizarJerarquia(pDatosPersistentes->m_base);

			//anonimizar tags Ginkgo
			if(!pDatosPersistentes->m_incluirTagsGinkgo){
				pDICOMManager->AnonimizarTagsPrivados();
			}

			wxString destino = GetFichero(wxPath,wxT("dcm"));

			std::string destinoStd(TOPATH(destino));
			correcto = correcto && pDICOMManager->AlmacenarFichero(destinoStd);
			resultado.push_back(destinoStd);
			GNC::Entorno::Instance()->GetControladorImportacionPACS()->LiberarInstanciaDeDICOMManager(pDICOMManager);
		}

		if(!correcto) {
			m_pExportacionParams->m_Error = _Std("Error storing file, check the permissions over the directory.");
			m_pExportacionParams->m_hasError = true;
		}
	}


	void ComandoExportacion::ExportarImagenes()
	{
		GnkPtr<GNC::GUI::TipoWizardExportacion> pDatosPersistentes = m_pExportacionParams->m_pDatosPersistentes;
		std::vector<std::string> resultado;

		GNC::GCS::IContratoExportacionImages* pContratoImages = dynamic_cast<GNC::GCS::IContratoExportacionImages*>(pDatosPersistentes->m_pVista);

		if(pContratoImages == NULL){
			//error!!!! no deberia llegar porque en el paso uno no le deberia haber dejado seleccionar otra cosa que un dicom
			m_pExportacionParams->m_Error = _Std("Unexpected Error exporting, the view is not allowed to export to the format selected");
			m_pExportacionParams->m_hasError = true;
			return ;
		}

		std::vector<std::string> listaPaths;
		GNC::GCS::IContratoExportacionImages::ImageType::Pointer img;
		try{
			if(pDatosPersistentes->m_ficheroActual){
				wxString cadena = wxString::Format(_("Exporting file %d of %d"), 1, 1);
				if (!NotificarProgreso(0.2, std::string(cadena.ToUTF8())))
					return;

				pContratoImages->GetImageActual(img, pDatosPersistentes->m_mapasValoracion, pDatosPersistentes->m_incluirWidgets, GNC::GCS::Vector::NaN());
				wxString res=ExportarImage(img);
				if(res!=wxEmptyString){
					std::string strTmp(TOPATH(res));
					resultado.push_back(strTmp);
				}
			}
			else
			{
				int size = pDatosPersistentes->m_pVista->GetRutasImagenes().size();
				for(std::string::size_type i = 0; (int)i < size; i++){
					wxString cadena = wxString::Format(_("Exporting file %d of %d"), i+1, size);
					if (!NotificarProgreso((float)i/size, std::string(cadena.ToUTF8())))
						return;

					pContratoImages->GetImage(img, i, pDatosPersistentes->m_mapasValoracion, pDatosPersistentes->m_incluirWidgets, GNC::GCS::Vector::NaN());
					wxString res = ExportarImage(img);
					if(res != wxEmptyString){
						std::string strTmp(TOPATH(res));
						resultado.push_back(strTmp);
					}
				}
			}
		}
		catch(itk::ExceptionObject& ex) {
			std::cerr << "Error" << ex.GetDescription() << std::endl;

			m_pExportacionParams->m_Error = _Std("Failed to store the file, check permissions on the directory");
			m_pExportacionParams->m_hasError = true;
		}
	}

	wxString ComandoExportacion::ExportarImage(GNC::GCS::IContratoExportacionImages::ImageType::Pointer img)
	{
		GnkPtr<GNC::GUI::TipoWizardExportacion> pDatosPersistentes = m_pExportacionParams->m_pDatosPersistentes;
		if(img.IsNull()){
			m_pExportacionParams->m_Error = _Std("The key files will not be exported");
			m_pExportacionParams->m_hasError = true;
			return wxEmptyString;
		}

		wxString destino = wxEmptyString;

		wxString wxPath = FROMPATH(pDatosPersistentes->m_pathDestino);

		typedef itk::ImageFileWriter<GNC::GCS::IContratoExportacionImages::ImageType> TipoWriter;
		TipoWriter::Pointer writer = TipoWriter::New();
		writer->SetInput(img);


		switch(pDatosPersistentes->m_formatoDestino) {
			case GNC::GUI::BMP:
				{
					destino = GetFichero(wxPath,wxT("bmp"));
					std::string strTmp (TOPATH(destino));
					writer->SetFileName(strTmp.c_str());
				}
				break;
			case GNC::GUI::JPEG:
				{
					destino = GetFichero(wxPath,wxT("jpg"));
					std::string strTmp (TOPATH(destino));
					writer->SetFileName(strTmp.c_str());
					itk::JPEGImageIO::Pointer imageIO = itk::JPEGImageIO::New();
					writer->SetImageIO(imageIO);
					imageIO->SetUseCompression(true);
					imageIO->SetQuality(pDatosPersistentes->m_jpegCalidad);
				}
				break;
			case GNC::GUI::PNG:
				{
					destino = GetFichero(wxPath,wxT("png"));
					std::string strTmp (TOPATH(destino));
					writer->SetFileName(strTmp.c_str());
				}
				break;
			case GNC::GUI::DICOM:
				break;
		}

		writer->Update();
		return destino;
	}

	wxString ComandoExportacion::GetFichero(const wxString& dir, const wxString& extension){
		wxString destino(wxEmptyString);
		std::string nombre (wxDateTime::Now().Format(_("image_%m-%d-%Y_")).ToUTF8());
		std::string stdDir(TOPATH(dir));
		std::string stdExtension(TOPATH(extension));
		int indice=0;
		//yy-mm-dd_imagen_indice
		do {
			std::ostringstream ostr;
			ostr << stdDir << (char)wxFileName::GetPathSeparator() << nombre << indice++ << "." << stdExtension;
			destino = FROMPATH(ostr.str());
		} while(wxFile::Exists(destino));
		return destino;
	}
};

#ifdef _MSC_VER
#pragma warning(pop)
#endif

