/*
 *  
 *  $Id: iimpresion.cpp 3720 2011-04-15 13:39:05Z carlos $
 *  Ginkgo CADx Project
 *
 *  Copyright 2008-10 MetaEmotion S.L. All rights reserved.
 *  http://ginkgo-cadx.com
 *
 *  This file is licensed under LGPL v3 license.
 *  See License.txt for details
 *
 */
// Forward declarations
#include "iimpresion.h"

#include <api/iwidgetsmanager.h>
#include <api/icontexto.h>
#include <api/iwidgetsrenderer.h>

#include <vtkSmartPointer.h>
#include <vtkObject.h>
#include <vtkRenderer.h>
#include <vtkPointData.h>
#include <vtkImageData.h>
#include <vtkMatrix4x4.h>
#include <vtkImageReslice.h>
#include <vtkImageMapToRGBA.h>
#include <vtkExecutive.h>
#include <vtkInformationVector.h>
#include <vtkAlgorithmOutput.h>
#include <vtkInformation.h>
#include <itkVectorResampleImageFilter.h>

#include <itk/itkVTKImageToImageFilter.h>

namespace GNC {
	namespace GCS {

		IImpresion::IImpresion(GNC::GCS::IWidgetsManager* pWidgetsManager, const GnkPtr<GNC::GCS::IContextoEstudio>& pContextoEstudio, int index)
		{
			m_pWidgetsManager = pWidgetsManager;
			m_pEstudioReferido = GNC::GCS::IContextoEstudio::NewRef(pContextoEstudio);
			m_index = index;
		}

		vtkSmartPointer<vtkImageData> IImpresion::GetImage()
		{
			m_pEstudioReferido->SetIndiceActivo(m_index);

			m_pEstudioReferido->renderConnection->GetProducer()->UpdateInformation();

			vtkSmartPointer<vtkImageData> pImgData = NULL;
			vtkInformationVector* iv = m_pEstudioReferido->renderConnection->GetProducer()->GetExecutive()->GetOutputInformation();
			if (iv && iv->GetNumberOfInformationObjects() == 1) {
				vtkInformation* io = iv->GetInformationObject(0);
				pImgData = vtkImageData::SafeDownCast(io->Get(vtkDataObject::DATA_OBJECT()));
			}

			pImgData->Update();
			return pImgData;
		}

		void IImpresion::Imprimir(vtkImageData* imageData, bool conWidgets, const GNC::GCS::Vector& size, GNC::GCS::IContratoExportacionImages::ImageType::Pointer& out)
		{
			typedef itk::VTKImageToImageFilter<GNC::GCS::IContratoExportacionImages::ImageType> TipoFiltro;
			typedef itk::VectorResampleImageFilter<GNC::GCS::IContratoExportacionImages::ImageType, GNC::GCS::IContratoExportacionImages::ImageType> TipoResampler;

			TipoFiltro::Pointer VTK2ITKfiltro =  TipoFiltro::New();
			TipoResampler::Pointer resampler = TipoResampler::New();

			VTK2ITKfiltro->SetInput(imageData);
			VTK2ITKfiltro->GetImporter()->Update();

			resampler->SetInput(VTK2ITKfiltro->GetImporter()->GetOutput());

			GNC::GCS::Vector aspectRatio(1.0f, 1.0f);
			
			resampler->SetOutputOrigin(resampler->GetInput()->GetOrigin());
			resampler->SetOutputDirection(resampler->GetInput()->GetDirection());
			out = resampler->GetOutput();

			if (!size.EsNaN()) {
				TipoResampler::InputImageType::SizeType origsize = resampler->GetInput()->GetLargestPossibleRegion().GetSize();			
				TipoResampler::InputImageType::SpacingType origspacing = resampler->GetInput()->GetSpacing();

				TipoResampler::InputImageType::SizeType newsize;
				TipoResampler::InputImageType::SpacingType newspacing;

				GNC::GCS::Vector vOrigSize = GNC::GCS::Vector(origsize[0], origsize[1]);
				GNC::GCS::Vector vNewSize = vOrigSize.AjusteInteriorProporcional(size);

				aspectRatio = vNewSize / vOrigSize;

				newsize[0] = (int) vNewSize.x;
				newsize[1] = (int) vNewSize.y;

				newspacing[0] = origspacing[0] / aspectRatio.x;
				newspacing[1] = origspacing[1] / aspectRatio.y;

				resampler->SetSize( newsize );
				resampler->SetOutputSpacing(newspacing);
			}
			else {
				resampler->SetOutputSpacing(resampler->GetInput()->GetSpacing());
				resampler->SetSize( resampler->GetInput()->GetLargestPossibleRegion().GetSize() );
			}

			resampler->UpdateLargestPossibleRegion();
			
			if(conWidgets){
				ImprimirConWidgets(out, aspectRatio);
			}
			

		}

		void IImpresion::ImprimirConWidgets(GNC::GCS::IContratoExportacionImages::ImageType::Pointer& img, const GNC::GCS::Vector& aspectRatio)
		{
			int dim[3];
			dim[0] = img->GetLargestPossibleRegion().GetSize()[0];
			dim[1] = img->GetLargestPossibleRegion().GetSize()[1];
			dim[2] = 1;

			unsigned char* imgptr = (unsigned char*) (img->GetPixelContainer()->GetImportPointer());
			unsigned int imgsize = dim[0] * dim[1];

			GNC::GCS::Contexto3D c;
			c.ancho = dim[0];
			c.alto = dim[1];

			c.spacing[0] = img->GetSpacing()[0];
			c.spacing[1] = img->GetSpacing()[1];
			c.spacing[2] = 1.0f;

			c.origin[0] = img->GetOrigin()[0];
			c.origin[1] = img->GetOrigin()[1];
			c.origin[2] = 0.0f;

			c.OVID = m_index;
			c.pRenderer = NULL;
			c.pOffscrenViewer =  NULL;

			c.CrearOffscreen();
			
			c.factorReescalado = aspectRatio;

			CopiarRGBEnRGBA32(imgptr, c.pixelData, imgsize);
			m_pWidgetsManager->OffScreenRender(&c);
			CopiarRGBA32EnRGB(c.pixelData, imgptr, imgsize);

			c.DestruirOffscreen();
		}
	}
}
