/*
** Copyright (c) 2018-2021 Valve Corporation
** Copyright (c) 2018-2021 LunarG, Inc.
**
** Permission is hereby granted, free of charge, to any person obtaining a
** copy of this software and associated documentation files (the "Software"),
** to deal in the Software without restriction, including without limitation
** the rights to use, copy, modify, merge, publish, distribute, sublicense,
** and/or sell copies of the Software, and to permit persons to whom the
** Software is furnished to do so, subject to the following conditions:
**
** The above copyright notice and this permission notice shall be included in
** all copies or substantial portions of the Software.
**
** THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
** IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
** FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
** AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
** LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
** FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
** DEALINGS IN THE SOFTWARE.
*/

/*
** The IDs defined by this header should not be modified.  New API call
** IDs should always be appended to the existing API call ID list.
**
*/

#ifndef GFXRECON_FORMAT_API_CALL_ID_H
#define GFXRECON_FORMAT_API_CALL_ID_H

#include "util/defines.h"

#include <cstdint>

GFXRECON_BEGIN_NAMESPACE(gfxrecon)
GFXRECON_BEGIN_NAMESPACE(format)

constexpr uint32_t MakeApiCallId(uint16_t family, uint16_t api_call)
{
    return ((static_cast<uint32_t>(family) << 16) & 0xffff0000) | (static_cast<uint32_t>(api_call) & 0x0000ffff);
}

enum ApiFamilyId : uint16_t
{
    ApiFamily_None   = 0,
    ApiFamily_Vulkan = 1
};

enum ApiCallId : uint32_t
{
    // clang-format off
    ApiCall_Unknown                                                                               = 0x0000,

    // Vulkan API
    ApiCall_vkCreateInstance                                                                      = MakeApiCallId(ApiFamily_Vulkan, 0x1000),
    ApiCall_vkDestroyInstance                                                                     = MakeApiCallId(ApiFamily_Vulkan, 0x1001),
    ApiCall_vkEnumeratePhysicalDevices                                                            = MakeApiCallId(ApiFamily_Vulkan, 0x1002),
    ApiCall_vkGetPhysicalDeviceFeatures                                                           = MakeApiCallId(ApiFamily_Vulkan, 0x1003),
    ApiCall_vkGetPhysicalDeviceFormatProperties                                                   = MakeApiCallId(ApiFamily_Vulkan, 0x1004),
    ApiCall_vkGetPhysicalDeviceImageFormatProperties                                              = MakeApiCallId(ApiFamily_Vulkan, 0x1005),
    ApiCall_vkGetPhysicalDeviceProperties                                                         = MakeApiCallId(ApiFamily_Vulkan, 0x1006),
    ApiCall_vkGetPhysicalDeviceQueueFamilyProperties                                              = MakeApiCallId(ApiFamily_Vulkan, 0x1007),
    ApiCall_vkGetPhysicalDeviceMemoryProperties                                                   = MakeApiCallId(ApiFamily_Vulkan, 0x1008),
    ApiCall_vkGetInstanceProcAddr                                                                 = MakeApiCallId(ApiFamily_Vulkan, 0x1009),
    ApiCall_vkGetDeviceProcAddr                                                                   = MakeApiCallId(ApiFamily_Vulkan, 0x100a),
    ApiCall_vkCreateDevice                                                                        = MakeApiCallId(ApiFamily_Vulkan, 0x100b),
    ApiCall_vkDestroyDevice                                                                       = MakeApiCallId(ApiFamily_Vulkan, 0x100c),
    ApiCall_vkEnumerateInstanceExtensionProperties                                                = MakeApiCallId(ApiFamily_Vulkan, 0x100d),
    ApiCall_vkEnumerateDeviceExtensionProperties                                                  = MakeApiCallId(ApiFamily_Vulkan, 0x100e),
    ApiCall_vkEnumerateInstanceLayerProperties                                                    = MakeApiCallId(ApiFamily_Vulkan, 0x100f),
    ApiCall_vkEnumerateDeviceLayerProperties                                                      = MakeApiCallId(ApiFamily_Vulkan, 0x1010),
    ApiCall_vkGetDeviceQueue                                                                      = MakeApiCallId(ApiFamily_Vulkan, 0x1011),
    ApiCall_vkQueueSubmit                                                                         = MakeApiCallId(ApiFamily_Vulkan, 0x1012),
    ApiCall_vkQueueWaitIdle                                                                       = MakeApiCallId(ApiFamily_Vulkan, 0x1013),
    ApiCall_vkDeviceWaitIdle                                                                      = MakeApiCallId(ApiFamily_Vulkan, 0x1014),
    ApiCall_vkAllocateMemory                                                                      = MakeApiCallId(ApiFamily_Vulkan, 0x1015),
    ApiCall_vkFreeMemory                                                                          = MakeApiCallId(ApiFamily_Vulkan, 0x1016),
    ApiCall_vkMapMemory                                                                           = MakeApiCallId(ApiFamily_Vulkan, 0x1017),
    ApiCall_vkUnmapMemory                                                                         = MakeApiCallId(ApiFamily_Vulkan, 0x1018),
    ApiCall_vkFlushMappedMemoryRanges                                                             = MakeApiCallId(ApiFamily_Vulkan, 0x1019),
    ApiCall_vkInvalidateMappedMemoryRanges                                                        = MakeApiCallId(ApiFamily_Vulkan, 0x101a),
    ApiCall_vkGetDeviceMemoryCommitment                                                           = MakeApiCallId(ApiFamily_Vulkan, 0x101b),
    ApiCall_vkBindBufferMemory                                                                    = MakeApiCallId(ApiFamily_Vulkan, 0x101c),
    ApiCall_vkBindImageMemory                                                                     = MakeApiCallId(ApiFamily_Vulkan, 0x101d),
    ApiCall_vkGetBufferMemoryRequirements                                                         = MakeApiCallId(ApiFamily_Vulkan, 0x101e),
    ApiCall_vkGetImageMemoryRequirements                                                          = MakeApiCallId(ApiFamily_Vulkan, 0x101f),
    ApiCall_vkGetImageSparseMemoryRequirements                                                    = MakeApiCallId(ApiFamily_Vulkan, 0x1020),
    ApiCall_vkGetPhysicalDeviceSparseImageFormatProperties                                        = MakeApiCallId(ApiFamily_Vulkan, 0x1021),
    ApiCall_vkQueueBindSparse                                                                     = MakeApiCallId(ApiFamily_Vulkan, 0x1022),
    ApiCall_vkCreateFence                                                                         = MakeApiCallId(ApiFamily_Vulkan, 0x1023),
    ApiCall_vkDestroyFence                                                                        = MakeApiCallId(ApiFamily_Vulkan, 0x1024),
    ApiCall_vkResetFences                                                                         = MakeApiCallId(ApiFamily_Vulkan, 0x1025),
    ApiCall_vkGetFenceStatus                                                                      = MakeApiCallId(ApiFamily_Vulkan, 0x1026),
    ApiCall_vkWaitForFences                                                                       = MakeApiCallId(ApiFamily_Vulkan, 0x1027),
    ApiCall_vkCreateSemaphore                                                                     = MakeApiCallId(ApiFamily_Vulkan, 0x1028),
    ApiCall_vkDestroySemaphore                                                                    = MakeApiCallId(ApiFamily_Vulkan, 0x1029),
    ApiCall_vkCreateEvent                                                                         = MakeApiCallId(ApiFamily_Vulkan, 0x102a),
    ApiCall_vkDestroyEvent                                                                        = MakeApiCallId(ApiFamily_Vulkan, 0x102b),
    ApiCall_vkGetEventStatus                                                                      = MakeApiCallId(ApiFamily_Vulkan, 0x102c),
    ApiCall_vkSetEvent                                                                            = MakeApiCallId(ApiFamily_Vulkan, 0x102d),
    ApiCall_vkResetEvent                                                                          = MakeApiCallId(ApiFamily_Vulkan, 0x102e),
    ApiCall_vkCreateQueryPool                                                                     = MakeApiCallId(ApiFamily_Vulkan, 0x102f),
    ApiCall_vkDestroyQueryPool                                                                    = MakeApiCallId(ApiFamily_Vulkan, 0x1030),
    ApiCall_vkGetQueryPoolResults                                                                 = MakeApiCallId(ApiFamily_Vulkan, 0x1031),
    ApiCall_vkCreateBuffer                                                                        = MakeApiCallId(ApiFamily_Vulkan, 0x1032),
    ApiCall_vkDestroyBuffer                                                                       = MakeApiCallId(ApiFamily_Vulkan, 0x1033),
    ApiCall_vkCreateBufferView                                                                    = MakeApiCallId(ApiFamily_Vulkan, 0x1034),
    ApiCall_vkDestroyBufferView                                                                   = MakeApiCallId(ApiFamily_Vulkan, 0x1035),
    ApiCall_vkCreateImage                                                                         = MakeApiCallId(ApiFamily_Vulkan, 0x1036),
    ApiCall_vkDestroyImage                                                                        = MakeApiCallId(ApiFamily_Vulkan, 0x1037),
    ApiCall_vkGetImageSubresourceLayout                                                           = MakeApiCallId(ApiFamily_Vulkan, 0x1038),
    ApiCall_vkCreateImageView                                                                     = MakeApiCallId(ApiFamily_Vulkan, 0x1039),
    ApiCall_vkDestroyImageView                                                                    = MakeApiCallId(ApiFamily_Vulkan, 0x103a),
    ApiCall_vkCreateShaderModule                                                                  = MakeApiCallId(ApiFamily_Vulkan, 0x103b),
    ApiCall_vkDestroyShaderModule                                                                 = MakeApiCallId(ApiFamily_Vulkan, 0x103c),
    ApiCall_vkCreatePipelineCache                                                                 = MakeApiCallId(ApiFamily_Vulkan, 0x103d),
    ApiCall_vkDestroyPipelineCache                                                                = MakeApiCallId(ApiFamily_Vulkan, 0x103e),
    ApiCall_vkGetPipelineCacheData                                                                = MakeApiCallId(ApiFamily_Vulkan, 0x103f),
    ApiCall_vkMergePipelineCaches                                                                 = MakeApiCallId(ApiFamily_Vulkan, 0x1040),
    ApiCall_vkCreateGraphicsPipelines                                                             = MakeApiCallId(ApiFamily_Vulkan, 0x1041),
    ApiCall_vkCreateComputePipelines                                                              = MakeApiCallId(ApiFamily_Vulkan, 0x1042),
    ApiCall_vkDestroyPipeline                                                                     = MakeApiCallId(ApiFamily_Vulkan, 0x1043),
    ApiCall_vkCreatePipelineLayout                                                                = MakeApiCallId(ApiFamily_Vulkan, 0x1044),
    ApiCall_vkDestroyPipelineLayout                                                               = MakeApiCallId(ApiFamily_Vulkan, 0x1045),
    ApiCall_vkCreateSampler                                                                       = MakeApiCallId(ApiFamily_Vulkan, 0x1046),
    ApiCall_vkDestroySampler                                                                      = MakeApiCallId(ApiFamily_Vulkan, 0x1047),
    ApiCall_vkCreateDescriptorSetLayout                                                           = MakeApiCallId(ApiFamily_Vulkan, 0x1048),
    ApiCall_vkDestroyDescriptorSetLayout                                                          = MakeApiCallId(ApiFamily_Vulkan, 0x1049),
    ApiCall_vkCreateDescriptorPool                                                                = MakeApiCallId(ApiFamily_Vulkan, 0x104a),
    ApiCall_vkDestroyDescriptorPool                                                               = MakeApiCallId(ApiFamily_Vulkan, 0x104b),
    ApiCall_vkResetDescriptorPool                                                                 = MakeApiCallId(ApiFamily_Vulkan, 0x104c),
    ApiCall_vkAllocateDescriptorSets                                                              = MakeApiCallId(ApiFamily_Vulkan, 0x104d),
    ApiCall_vkFreeDescriptorSets                                                                  = MakeApiCallId(ApiFamily_Vulkan, 0x104e),
    ApiCall_vkUpdateDescriptorSets                                                                = MakeApiCallId(ApiFamily_Vulkan, 0x104f),
    ApiCall_vkCreateFramebuffer                                                                   = MakeApiCallId(ApiFamily_Vulkan, 0x1050),
    ApiCall_vkDestroyFramebuffer                                                                  = MakeApiCallId(ApiFamily_Vulkan, 0x1051),
    ApiCall_vkCreateRenderPass                                                                    = MakeApiCallId(ApiFamily_Vulkan, 0x1052),
    ApiCall_vkDestroyRenderPass                                                                   = MakeApiCallId(ApiFamily_Vulkan, 0x1053),
    ApiCall_vkGetRenderAreaGranularity                                                            = MakeApiCallId(ApiFamily_Vulkan, 0x1054),
    ApiCall_vkCreateCommandPool                                                                   = MakeApiCallId(ApiFamily_Vulkan, 0x1055),
    ApiCall_vkDestroyCommandPool                                                                  = MakeApiCallId(ApiFamily_Vulkan, 0x1056),
    ApiCall_vkResetCommandPool                                                                    = MakeApiCallId(ApiFamily_Vulkan, 0x1057),
    ApiCall_vkAllocateCommandBuffers                                                              = MakeApiCallId(ApiFamily_Vulkan, 0x1058),
    ApiCall_vkFreeCommandBuffers                                                                  = MakeApiCallId(ApiFamily_Vulkan, 0x1059),
    ApiCall_vkBeginCommandBuffer                                                                  = MakeApiCallId(ApiFamily_Vulkan, 0x105a),
    ApiCall_vkEndCommandBuffer                                                                    = MakeApiCallId(ApiFamily_Vulkan, 0x105b),
    ApiCall_vkResetCommandBuffer                                                                  = MakeApiCallId(ApiFamily_Vulkan, 0x105c),
    ApiCall_vkCmdBindPipeline                                                                     = MakeApiCallId(ApiFamily_Vulkan, 0x105d),
    ApiCall_vkCmdSetViewport                                                                      = MakeApiCallId(ApiFamily_Vulkan, 0x105e),
    ApiCall_vkCmdSetScissor                                                                       = MakeApiCallId(ApiFamily_Vulkan, 0x105f),
    ApiCall_vkCmdSetLineWidth                                                                     = MakeApiCallId(ApiFamily_Vulkan, 0x1060),
    ApiCall_vkCmdSetDepthBias                                                                     = MakeApiCallId(ApiFamily_Vulkan, 0x1061),
    ApiCall_vkCmdSetBlendConstants                                                                = MakeApiCallId(ApiFamily_Vulkan, 0x1062),
    ApiCall_vkCmdSetDepthBounds                                                                   = MakeApiCallId(ApiFamily_Vulkan, 0x1063),
    ApiCall_vkCmdSetStencilCompareMask                                                            = MakeApiCallId(ApiFamily_Vulkan, 0x1064),
    ApiCall_vkCmdSetStencilWriteMask                                                              = MakeApiCallId(ApiFamily_Vulkan, 0x1065),
    ApiCall_vkCmdSetStencilReference                                                              = MakeApiCallId(ApiFamily_Vulkan, 0x1066),
    ApiCall_vkCmdBindDescriptorSets                                                               = MakeApiCallId(ApiFamily_Vulkan, 0x1067),
    ApiCall_vkCmdBindIndexBuffer                                                                  = MakeApiCallId(ApiFamily_Vulkan, 0x1068),
    ApiCall_vkCmdBindVertexBuffers                                                                = MakeApiCallId(ApiFamily_Vulkan, 0x1069),
    ApiCall_vkCmdDraw                                                                             = MakeApiCallId(ApiFamily_Vulkan, 0x106a),
    ApiCall_vkCmdDrawIndexed                                                                      = MakeApiCallId(ApiFamily_Vulkan, 0x106b),
    ApiCall_vkCmdDrawIndirect                                                                     = MakeApiCallId(ApiFamily_Vulkan, 0x106c),
    ApiCall_vkCmdDrawIndexedIndirect                                                              = MakeApiCallId(ApiFamily_Vulkan, 0x106d),
    ApiCall_vkCmdDispatch                                                                         = MakeApiCallId(ApiFamily_Vulkan, 0x106e),
    ApiCall_vkCmdDispatchIndirect                                                                 = MakeApiCallId(ApiFamily_Vulkan, 0x106f),
    ApiCall_vkCmdCopyBuffer                                                                       = MakeApiCallId(ApiFamily_Vulkan, 0x1070),
    ApiCall_vkCmdCopyImage                                                                        = MakeApiCallId(ApiFamily_Vulkan, 0x1071),
    ApiCall_vkCmdBlitImage                                                                        = MakeApiCallId(ApiFamily_Vulkan, 0x1072),
    ApiCall_vkCmdCopyBufferToImage                                                                = MakeApiCallId(ApiFamily_Vulkan, 0x1073),
    ApiCall_vkCmdCopyImageToBuffer                                                                = MakeApiCallId(ApiFamily_Vulkan, 0x1074),
    ApiCall_vkCmdUpdateBuffer                                                                     = MakeApiCallId(ApiFamily_Vulkan, 0x1075),
    ApiCall_vkCmdFillBuffer                                                                       = MakeApiCallId(ApiFamily_Vulkan, 0x1076),
    ApiCall_vkCmdClearColorImage                                                                  = MakeApiCallId(ApiFamily_Vulkan, 0x1077),
    ApiCall_vkCmdClearDepthStencilImage                                                           = MakeApiCallId(ApiFamily_Vulkan, 0x1078),
    ApiCall_vkCmdClearAttachments                                                                 = MakeApiCallId(ApiFamily_Vulkan, 0x1079),
    ApiCall_vkCmdResolveImage                                                                     = MakeApiCallId(ApiFamily_Vulkan, 0x107a),
    ApiCall_vkCmdSetEvent                                                                         = MakeApiCallId(ApiFamily_Vulkan, 0x107b),
    ApiCall_vkCmdResetEvent                                                                       = MakeApiCallId(ApiFamily_Vulkan, 0x107c),
    ApiCall_vkCmdWaitEvents                                                                       = MakeApiCallId(ApiFamily_Vulkan, 0x107d),
    ApiCall_vkCmdPipelineBarrier                                                                  = MakeApiCallId(ApiFamily_Vulkan, 0x107e),
    ApiCall_vkCmdBeginQuery                                                                       = MakeApiCallId(ApiFamily_Vulkan, 0x107f),
    ApiCall_vkCmdEndQuery                                                                         = MakeApiCallId(ApiFamily_Vulkan, 0x1080),
    ApiCall_vkCmdResetQueryPool                                                                   = MakeApiCallId(ApiFamily_Vulkan, 0x1081),
    ApiCall_vkCmdWriteTimestamp                                                                   = MakeApiCallId(ApiFamily_Vulkan, 0x1082),
    ApiCall_vkCmdCopyQueryPoolResults                                                             = MakeApiCallId(ApiFamily_Vulkan, 0x1083),
    ApiCall_vkCmdPushConstants                                                                    = MakeApiCallId(ApiFamily_Vulkan, 0x1084),
    ApiCall_vkCmdBeginRenderPass                                                                  = MakeApiCallId(ApiFamily_Vulkan, 0x1085),
    ApiCall_vkCmdNextSubpass                                                                      = MakeApiCallId(ApiFamily_Vulkan, 0x1086),
    ApiCall_vkCmdEndRenderPass                                                                    = MakeApiCallId(ApiFamily_Vulkan, 0x1087),
    ApiCall_vkCmdExecuteCommands                                                                  = MakeApiCallId(ApiFamily_Vulkan, 0x1088),
    ApiCall_vkEnumerateInstanceVersion                                                            = MakeApiCallId(ApiFamily_Vulkan, 0x1089),
    ApiCall_vkBindBufferMemory2                                                                   = MakeApiCallId(ApiFamily_Vulkan, 0x108a),
    ApiCall_vkBindImageMemory2                                                                    = MakeApiCallId(ApiFamily_Vulkan, 0x108b),
    ApiCall_vkGetDeviceGroupPeerMemoryFeatures                                                    = MakeApiCallId(ApiFamily_Vulkan, 0x108c),
    ApiCall_vkCmdSetDeviceMask                                                                    = MakeApiCallId(ApiFamily_Vulkan, 0x108d),
    ApiCall_vkCmdDispatchBase                                                                     = MakeApiCallId(ApiFamily_Vulkan, 0x108e),
    ApiCall_vkEnumeratePhysicalDeviceGroups                                                       = MakeApiCallId(ApiFamily_Vulkan, 0x108f),
    ApiCall_vkGetImageMemoryRequirements2                                                         = MakeApiCallId(ApiFamily_Vulkan, 0x1090),
    ApiCall_vkGetBufferMemoryRequirements2                                                        = MakeApiCallId(ApiFamily_Vulkan, 0x1091),
    ApiCall_vkGetImageSparseMemoryRequirements2                                                   = MakeApiCallId(ApiFamily_Vulkan, 0x1092),
    ApiCall_vkGetPhysicalDeviceFeatures2                                                          = MakeApiCallId(ApiFamily_Vulkan, 0x1093),
    ApiCall_vkGetPhysicalDeviceProperties2                                                        = MakeApiCallId(ApiFamily_Vulkan, 0x1094),
    ApiCall_vkGetPhysicalDeviceFormatProperties2                                                  = MakeApiCallId(ApiFamily_Vulkan, 0x1095),
    ApiCall_vkGetPhysicalDeviceImageFormatProperties2                                             = MakeApiCallId(ApiFamily_Vulkan, 0x1096),
    ApiCall_vkGetPhysicalDeviceQueueFamilyProperties2                                             = MakeApiCallId(ApiFamily_Vulkan, 0x1097),
    ApiCall_vkGetPhysicalDeviceMemoryProperties2                                                  = MakeApiCallId(ApiFamily_Vulkan, 0x1098),
    ApiCall_vkGetPhysicalDeviceSparseImageFormatProperties2                                       = MakeApiCallId(ApiFamily_Vulkan, 0x1099),
    ApiCall_vkTrimCommandPool                                                                     = MakeApiCallId(ApiFamily_Vulkan, 0x109a),
    ApiCall_vkGetDeviceQueue2                                                                     = MakeApiCallId(ApiFamily_Vulkan, 0x109b),
    ApiCall_vkCreateSamplerYcbcrConversion                                                        = MakeApiCallId(ApiFamily_Vulkan, 0x109c),
    ApiCall_vkDestroySamplerYcbcrConversion                                                       = MakeApiCallId(ApiFamily_Vulkan, 0x109d),
    ApiCall_vkCreateDescriptorUpdateTemplate                                                      = MakeApiCallId(ApiFamily_Vulkan, 0x109e),
    ApiCall_vkDestroyDescriptorUpdateTemplate                                                     = MakeApiCallId(ApiFamily_Vulkan, 0x109f),
    ApiCall_vkUpdateDescriptorSetWithTemplate                                                     = MakeApiCallId(ApiFamily_Vulkan, 0x10a0),
    ApiCall_vkGetPhysicalDeviceExternalBufferProperties                                           = MakeApiCallId(ApiFamily_Vulkan, 0x10a1),
    ApiCall_vkGetPhysicalDeviceExternalFenceProperties                                            = MakeApiCallId(ApiFamily_Vulkan, 0x10a2),
    ApiCall_vkGetPhysicalDeviceExternalSemaphoreProperties                                        = MakeApiCallId(ApiFamily_Vulkan, 0x10a3),
    ApiCall_vkGetDescriptorSetLayoutSupport                                                       = MakeApiCallId(ApiFamily_Vulkan, 0x10a4),
    ApiCall_vkDestroySurfaceKHR                                                                   = MakeApiCallId(ApiFamily_Vulkan, 0x10a5),
    ApiCall_vkGetPhysicalDeviceSurfaceSupportKHR                                                  = MakeApiCallId(ApiFamily_Vulkan, 0x10a6),
    ApiCall_vkGetPhysicalDeviceSurfaceCapabilitiesKHR                                             = MakeApiCallId(ApiFamily_Vulkan, 0x10a7),
    ApiCall_vkGetPhysicalDeviceSurfaceFormatsKHR                                                  = MakeApiCallId(ApiFamily_Vulkan, 0x10a8),
    ApiCall_vkGetPhysicalDeviceSurfacePresentModesKHR                                             = MakeApiCallId(ApiFamily_Vulkan, 0x10a9),
    ApiCall_vkCreateSwapchainKHR                                                                  = MakeApiCallId(ApiFamily_Vulkan, 0x10aa),
    ApiCall_vkDestroySwapchainKHR                                                                 = MakeApiCallId(ApiFamily_Vulkan, 0x10ab),
    ApiCall_vkGetSwapchainImagesKHR                                                               = MakeApiCallId(ApiFamily_Vulkan, 0x10ac),
    ApiCall_vkAcquireNextImageKHR                                                                 = MakeApiCallId(ApiFamily_Vulkan, 0x10ad),
    ApiCall_vkQueuePresentKHR                                                                     = MakeApiCallId(ApiFamily_Vulkan, 0x10ae),
    ApiCall_vkGetDeviceGroupPresentCapabilitiesKHR                                                = MakeApiCallId(ApiFamily_Vulkan, 0x10af),
    ApiCall_vkGetDeviceGroupSurfacePresentModesKHR                                                = MakeApiCallId(ApiFamily_Vulkan, 0x10b0),
    ApiCall_vkGetPhysicalDevicePresentRectanglesKHR                                               = MakeApiCallId(ApiFamily_Vulkan, 0x10b1),
    ApiCall_vkAcquireNextImage2KHR                                                                = MakeApiCallId(ApiFamily_Vulkan, 0x10b2),
    ApiCall_vkGetPhysicalDeviceDisplayPropertiesKHR                                               = MakeApiCallId(ApiFamily_Vulkan, 0x10b3),
    ApiCall_vkGetPhysicalDeviceDisplayPlanePropertiesKHR                                          = MakeApiCallId(ApiFamily_Vulkan, 0x10b4),
    ApiCall_vkGetDisplayPlaneSupportedDisplaysKHR                                                 = MakeApiCallId(ApiFamily_Vulkan, 0x10b5),
    ApiCall_vkGetDisplayModePropertiesKHR                                                         = MakeApiCallId(ApiFamily_Vulkan, 0x10b6),
    ApiCall_vkCreateDisplayModeKHR                                                                = MakeApiCallId(ApiFamily_Vulkan, 0x10b7),
    ApiCall_vkGetDisplayPlaneCapabilitiesKHR                                                      = MakeApiCallId(ApiFamily_Vulkan, 0x10b8),
    ApiCall_vkCreateDisplayPlaneSurfaceKHR                                                        = MakeApiCallId(ApiFamily_Vulkan, 0x10b9),
    ApiCall_vkCreateSharedSwapchainsKHR                                                           = MakeApiCallId(ApiFamily_Vulkan, 0x10ba),
    ApiCall_vkCreateXlibSurfaceKHR                                                                = MakeApiCallId(ApiFamily_Vulkan, 0x10bb),
    ApiCall_vkGetPhysicalDeviceXlibPresentationSupportKHR                                         = MakeApiCallId(ApiFamily_Vulkan, 0x10bc),
    ApiCall_vkCreateXcbSurfaceKHR                                                                 = MakeApiCallId(ApiFamily_Vulkan, 0x10bd),
    ApiCall_vkGetPhysicalDeviceXcbPresentationSupportKHR                                          = MakeApiCallId(ApiFamily_Vulkan, 0x10be),
    ApiCall_vkCreateWaylandSurfaceKHR                                                             = MakeApiCallId(ApiFamily_Vulkan, 0x10bf),
    ApiCall_vkGetPhysicalDeviceWaylandPresentationSupportKHR                                      = MakeApiCallId(ApiFamily_Vulkan, 0x10c0),
    ApiCall_vkCreateMirSurfaceKHR                                                                 = MakeApiCallId(ApiFamily_Vulkan, 0x10c1),
    ApiCall_vkGetPhysicalDeviceMirPresentationSupportKHR                                          = MakeApiCallId(ApiFamily_Vulkan, 0x10c2),
    ApiCall_vkCreateAndroidSurfaceKHR                                                             = MakeApiCallId(ApiFamily_Vulkan, 0x10c3),
    ApiCall_vkCreateWin32SurfaceKHR                                                               = MakeApiCallId(ApiFamily_Vulkan, 0x10c4),
    ApiCall_vkGetPhysicalDeviceWin32PresentationSupportKHR                                        = MakeApiCallId(ApiFamily_Vulkan, 0x10c5),
    ApiCall_vkGetPhysicalDeviceFeatures2KHR                                                       = MakeApiCallId(ApiFamily_Vulkan, 0x10c6),
    ApiCall_vkGetPhysicalDeviceProperties2KHR                                                     = MakeApiCallId(ApiFamily_Vulkan, 0x10c7),
    ApiCall_vkGetPhysicalDeviceFormatProperties2KHR                                               = MakeApiCallId(ApiFamily_Vulkan, 0x10c8),
    ApiCall_vkGetPhysicalDeviceImageFormatProperties2KHR                                          = MakeApiCallId(ApiFamily_Vulkan, 0x10c9),
    ApiCall_vkGetPhysicalDeviceQueueFamilyProperties2KHR                                          = MakeApiCallId(ApiFamily_Vulkan, 0x10ca),
    ApiCall_vkGetPhysicalDeviceMemoryProperties2KHR                                               = MakeApiCallId(ApiFamily_Vulkan, 0x10cb),
    ApiCall_vkGetPhysicalDeviceSparseImageFormatProperties2KHR                                    = MakeApiCallId(ApiFamily_Vulkan, 0x10cc),
    ApiCall_vkGetDeviceGroupPeerMemoryFeaturesKHR                                                 = MakeApiCallId(ApiFamily_Vulkan, 0x10cd),
    ApiCall_vkCmdSetDeviceMaskKHR                                                                 = MakeApiCallId(ApiFamily_Vulkan, 0x10ce),
    ApiCall_vkCmdDispatchBaseKHR                                                                  = MakeApiCallId(ApiFamily_Vulkan, 0x10cf),
    ApiCall_vkTrimCommandPoolKHR                                                                  = MakeApiCallId(ApiFamily_Vulkan, 0x10d0),
    ApiCall_vkEnumeratePhysicalDeviceGroupsKHR                                                    = MakeApiCallId(ApiFamily_Vulkan, 0x10d1),
    ApiCall_vkGetPhysicalDeviceExternalBufferPropertiesKHR                                        = MakeApiCallId(ApiFamily_Vulkan, 0x10d2),
    ApiCall_vkGetMemoryWin32HandleKHR                                                             = MakeApiCallId(ApiFamily_Vulkan, 0x10d3),
    ApiCall_vkGetMemoryWin32HandlePropertiesKHR                                                   = MakeApiCallId(ApiFamily_Vulkan, 0x10d4),
    ApiCall_vkGetMemoryFdKHR                                                                      = MakeApiCallId(ApiFamily_Vulkan, 0x10d5),
    ApiCall_vkGetMemoryFdPropertiesKHR                                                            = MakeApiCallId(ApiFamily_Vulkan, 0x10d6),
    ApiCall_vkGetPhysicalDeviceExternalSemaphorePropertiesKHR                                     = MakeApiCallId(ApiFamily_Vulkan, 0x10d7),
    ApiCall_vkImportSemaphoreWin32HandleKHR                                                       = MakeApiCallId(ApiFamily_Vulkan, 0x10d8),
    ApiCall_vkGetSemaphoreWin32HandleKHR                                                          = MakeApiCallId(ApiFamily_Vulkan, 0x10d9),
    ApiCall_vkImportSemaphoreFdKHR                                                                = MakeApiCallId(ApiFamily_Vulkan, 0x10da),
    ApiCall_vkGetSemaphoreFdKHR                                                                   = MakeApiCallId(ApiFamily_Vulkan, 0x10db),
    ApiCall_vkCmdPushDescriptorSetKHR                                                             = MakeApiCallId(ApiFamily_Vulkan, 0x10dc),
    ApiCall_vkCmdPushDescriptorSetWithTemplateKHR                                                 = MakeApiCallId(ApiFamily_Vulkan, 0x10dd),
    ApiCall_vkCreateDescriptorUpdateTemplateKHR                                                   = MakeApiCallId(ApiFamily_Vulkan, 0x10de),
    ApiCall_vkDestroyDescriptorUpdateTemplateKHR                                                  = MakeApiCallId(ApiFamily_Vulkan, 0x10df),
    ApiCall_vkUpdateDescriptorSetWithTemplateKHR                                                  = MakeApiCallId(ApiFamily_Vulkan, 0x10e0),
    ApiCall_vkCreateRenderPass2KHR                                                                = MakeApiCallId(ApiFamily_Vulkan, 0x10e1),
    ApiCall_vkCmdBeginRenderPass2KHR                                                              = MakeApiCallId(ApiFamily_Vulkan, 0x10e2),
    ApiCall_vkCmdNextSubpass2KHR                                                                  = MakeApiCallId(ApiFamily_Vulkan, 0x10e3),
    ApiCall_vkCmdEndRenderPass2KHR                                                                = MakeApiCallId(ApiFamily_Vulkan, 0x10e4),
    ApiCall_vkGetSwapchainStatusKHR                                                               = MakeApiCallId(ApiFamily_Vulkan, 0x10e5),
    ApiCall_vkGetPhysicalDeviceExternalFencePropertiesKHR                                         = MakeApiCallId(ApiFamily_Vulkan, 0x10e6),
    ApiCall_vkImportFenceWin32HandleKHR                                                           = MakeApiCallId(ApiFamily_Vulkan, 0x10e7),
    ApiCall_vkGetFenceWin32HandleKHR                                                              = MakeApiCallId(ApiFamily_Vulkan, 0x10e8),
    ApiCall_vkImportFenceFdKHR                                                                    = MakeApiCallId(ApiFamily_Vulkan, 0x10e9),
    ApiCall_vkGetFenceFdKHR                                                                       = MakeApiCallId(ApiFamily_Vulkan, 0x10ea),
    ApiCall_vkGetPhysicalDeviceSurfaceCapabilities2KHR                                            = MakeApiCallId(ApiFamily_Vulkan, 0x10eb),
    ApiCall_vkGetPhysicalDeviceSurfaceFormats2KHR                                                 = MakeApiCallId(ApiFamily_Vulkan, 0x10ec),
    ApiCall_vkGetPhysicalDeviceDisplayProperties2KHR                                              = MakeApiCallId(ApiFamily_Vulkan, 0x10ed),
    ApiCall_vkGetPhysicalDeviceDisplayPlaneProperties2KHR                                         = MakeApiCallId(ApiFamily_Vulkan, 0x10ee),
    ApiCall_vkGetDisplayModeProperties2KHR                                                        = MakeApiCallId(ApiFamily_Vulkan, 0x10ef),
    ApiCall_vkGetDisplayPlaneCapabilities2KHR                                                     = MakeApiCallId(ApiFamily_Vulkan, 0x10f0),
    ApiCall_vkGetImageMemoryRequirements2KHR                                                      = MakeApiCallId(ApiFamily_Vulkan, 0x10f1),
    ApiCall_vkGetBufferMemoryRequirements2KHR                                                     = MakeApiCallId(ApiFamily_Vulkan, 0x10f2),
    ApiCall_vkGetImageSparseMemoryRequirements2KHR                                                = MakeApiCallId(ApiFamily_Vulkan, 0x10f3),
    ApiCall_vkCreateSamplerYcbcrConversionKHR                                                     = MakeApiCallId(ApiFamily_Vulkan, 0x10f4),
    ApiCall_vkDestroySamplerYcbcrConversionKHR                                                    = MakeApiCallId(ApiFamily_Vulkan, 0x10f5),
    ApiCall_vkBindBufferMemory2KHR                                                                = MakeApiCallId(ApiFamily_Vulkan, 0x10f6),
    ApiCall_vkBindImageMemory2KHR                                                                 = MakeApiCallId(ApiFamily_Vulkan, 0x10f7),
    ApiCall_vkGetDescriptorSetLayoutSupportKHR                                                    = MakeApiCallId(ApiFamily_Vulkan, 0x10f8),
    ApiCall_vkCmdDrawIndirectCountKHR                                                             = MakeApiCallId(ApiFamily_Vulkan, 0x10f9),
    ApiCall_vkCmdDrawIndexedIndirectCountKHR                                                      = MakeApiCallId(ApiFamily_Vulkan, 0x10fa),
    ApiCall_vkCreateDebugReportCallbackEXT                                                        = MakeApiCallId(ApiFamily_Vulkan, 0x10fb),
    ApiCall_vkDestroyDebugReportCallbackEXT                                                       = MakeApiCallId(ApiFamily_Vulkan, 0x10fc),
    ApiCall_vkDebugReportMessageEXT                                                               = MakeApiCallId(ApiFamily_Vulkan, 0x10fd),
    ApiCall_vkDebugMarkerSetObjectTagEXT                                                          = MakeApiCallId(ApiFamily_Vulkan, 0x10fe),
    ApiCall_vkDebugMarkerSetObjectNameEXT                                                         = MakeApiCallId(ApiFamily_Vulkan, 0x10ff),
    ApiCall_vkCmdDebugMarkerBeginEXT                                                              = MakeApiCallId(ApiFamily_Vulkan, 0x1100),
    ApiCall_vkCmdDebugMarkerEndEXT                                                                = MakeApiCallId(ApiFamily_Vulkan, 0x1101),
    ApiCall_vkCmdDebugMarkerInsertEXT                                                             = MakeApiCallId(ApiFamily_Vulkan, 0x1102),
    ApiCall_vkCmdDrawIndirectCountAMD                                                             = MakeApiCallId(ApiFamily_Vulkan, 0x1103),
    ApiCall_vkCmdDrawIndexedIndirectCountAMD                                                      = MakeApiCallId(ApiFamily_Vulkan, 0x1104),
    ApiCall_vkGetShaderInfoAMD                                                                    = MakeApiCallId(ApiFamily_Vulkan, 0x1105),
    ApiCall_vkGetPhysicalDeviceExternalImageFormatPropertiesNV                                    = MakeApiCallId(ApiFamily_Vulkan, 0x1106),
    ApiCall_vkGetMemoryWin32HandleNV                                                              = MakeApiCallId(ApiFamily_Vulkan, 0x1107),
    ApiCall_vkCreateViSurfaceNN                                                                   = MakeApiCallId(ApiFamily_Vulkan, 0x1108),
    ApiCall_vkCmdBeginConditionalRenderingEXT                                                     = MakeApiCallId(ApiFamily_Vulkan, 0x1109),
    ApiCall_vkCmdEndConditionalRenderingEXT                                                       = MakeApiCallId(ApiFamily_Vulkan, 0x110a),
    ApiCall_vkCmdProcessCommandsNVX                                                               = MakeApiCallId(ApiFamily_Vulkan, 0x110b),
    ApiCall_vkCmdReserveSpaceForCommandsNVX                                                       = MakeApiCallId(ApiFamily_Vulkan, 0x110c),
    ApiCall_vkCreateIndirectCommandsLayoutNVX                                                     = MakeApiCallId(ApiFamily_Vulkan, 0x110d),
    ApiCall_vkDestroyIndirectCommandsLayoutNVX                                                    = MakeApiCallId(ApiFamily_Vulkan, 0x110e),
    ApiCall_vkCreateObjectTableNVX                                                                = MakeApiCallId(ApiFamily_Vulkan, 0x110f),
    ApiCall_vkDestroyObjectTableNVX                                                               = MakeApiCallId(ApiFamily_Vulkan, 0x1110),
    ApiCall_vkRegisterObjectsNVX                                                                  = MakeApiCallId(ApiFamily_Vulkan, 0x1111),
    ApiCall_vkUnregisterObjectsNVX                                                                = MakeApiCallId(ApiFamily_Vulkan, 0x1112),
    ApiCall_vkGetPhysicalDeviceGeneratedCommandsPropertiesNVX                                     = MakeApiCallId(ApiFamily_Vulkan, 0x1113),
    ApiCall_vkCmdSetViewportWScalingNV                                                            = MakeApiCallId(ApiFamily_Vulkan, 0x1114),
    ApiCall_vkReleaseDisplayEXT                                                                   = MakeApiCallId(ApiFamily_Vulkan, 0x1115),
    ApiCall_vkAcquireXlibDisplayEXT                                                               = MakeApiCallId(ApiFamily_Vulkan, 0x1116),
    ApiCall_vkGetRandROutputDisplayEXT                                                            = MakeApiCallId(ApiFamily_Vulkan, 0x1117),
    ApiCall_vkGetPhysicalDeviceSurfaceCapabilities2EXT                                            = MakeApiCallId(ApiFamily_Vulkan, 0x1118),
    ApiCall_vkDisplayPowerControlEXT                                                              = MakeApiCallId(ApiFamily_Vulkan, 0x1119),
    ApiCall_vkRegisterDeviceEventEXT                                                              = MakeApiCallId(ApiFamily_Vulkan, 0x111a),
    ApiCall_vkRegisterDisplayEventEXT                                                             = MakeApiCallId(ApiFamily_Vulkan, 0x111b),
    ApiCall_vkGetSwapchainCounterEXT                                                              = MakeApiCallId(ApiFamily_Vulkan, 0x111c),
    ApiCall_vkGetRefreshCycleDurationGOOGLE                                                       = MakeApiCallId(ApiFamily_Vulkan, 0x111d),
    ApiCall_vkGetPastPresentationTimingGOOGLE                                                     = MakeApiCallId(ApiFamily_Vulkan, 0x111e),
    ApiCall_vkCmdSetDiscardRectangleEXT                                                           = MakeApiCallId(ApiFamily_Vulkan, 0x111f),
    ApiCall_vkSetHdrMetadataEXT                                                                   = MakeApiCallId(ApiFamily_Vulkan, 0x1120),
    ApiCall_vkCreateIOSSurfaceMVK                                                                 = MakeApiCallId(ApiFamily_Vulkan, 0x1121),
    ApiCall_vkCreateMacOSSurfaceMVK                                                               = MakeApiCallId(ApiFamily_Vulkan, 0x1122),
    ApiCall_vkSetDebugUtilsObjectNameEXT                                                          = MakeApiCallId(ApiFamily_Vulkan, 0x1123),
    ApiCall_vkSetDebugUtilsObjectTagEXT                                                           = MakeApiCallId(ApiFamily_Vulkan, 0x1124),
    ApiCall_vkQueueBeginDebugUtilsLabelEXT                                                        = MakeApiCallId(ApiFamily_Vulkan, 0x1125),
    ApiCall_vkQueueEndDebugUtilsLabelEXT                                                          = MakeApiCallId(ApiFamily_Vulkan, 0x1126),
    ApiCall_vkQueueInsertDebugUtilsLabelEXT                                                       = MakeApiCallId(ApiFamily_Vulkan, 0x1127),
    ApiCall_vkCmdBeginDebugUtilsLabelEXT                                                          = MakeApiCallId(ApiFamily_Vulkan, 0x1128),
    ApiCall_vkCmdEndDebugUtilsLabelEXT                                                            = MakeApiCallId(ApiFamily_Vulkan, 0x1129),
    ApiCall_vkCmdInsertDebugUtilsLabelEXT                                                         = MakeApiCallId(ApiFamily_Vulkan, 0x112a),
    ApiCall_vkCreateDebugUtilsMessengerEXT                                                        = MakeApiCallId(ApiFamily_Vulkan, 0x112b),
    ApiCall_vkDestroyDebugUtilsMessengerEXT                                                       = MakeApiCallId(ApiFamily_Vulkan, 0x112c),
    ApiCall_vkSubmitDebugUtilsMessageEXT                                                          = MakeApiCallId(ApiFamily_Vulkan, 0x112d),
    ApiCall_vkGetAndroidHardwareBufferPropertiesANDROID                                           = MakeApiCallId(ApiFamily_Vulkan, 0x112e),
    ApiCall_vkGetMemoryAndroidHardwareBufferANDROID                                               = MakeApiCallId(ApiFamily_Vulkan, 0x112f),
    ApiCall_vkCmdSetSampleLocationsEXT                                                            = MakeApiCallId(ApiFamily_Vulkan, 0x1130),
    ApiCall_vkGetPhysicalDeviceMultisamplePropertiesEXT                                           = MakeApiCallId(ApiFamily_Vulkan, 0x1131),
    ApiCall_vkCreateValidationCacheEXT                                                            = MakeApiCallId(ApiFamily_Vulkan, 0x1132),
    ApiCall_vkDestroyValidationCacheEXT                                                           = MakeApiCallId(ApiFamily_Vulkan, 0x1133),
    ApiCall_vkMergeValidationCachesEXT                                                            = MakeApiCallId(ApiFamily_Vulkan, 0x1134),
    ApiCall_vkGetValidationCacheDataEXT                                                           = MakeApiCallId(ApiFamily_Vulkan, 0x1135),
    ApiCall_vkGetMemoryHostPointerPropertiesEXT                                                   = MakeApiCallId(ApiFamily_Vulkan, 0x1136),
    ApiCall_vkCmdWriteBufferMarkerAMD                                                             = MakeApiCallId(ApiFamily_Vulkan, 0x1137),
    ApiCall_vkCmdBindShadingRateImageNV                                                           = MakeApiCallId(ApiFamily_Vulkan, 0x1138),
    ApiCall_vkCmdSetViewportShadingRatePaletteNV                                                  = MakeApiCallId(ApiFamily_Vulkan, 0x1139),
    ApiCall_vkCmdSetCoarseSampleOrderNV                                                           = MakeApiCallId(ApiFamily_Vulkan, 0x113a),
    ApiCall_vkGetImageDrmFormatModifierPropertiesEXT                                              = MakeApiCallId(ApiFamily_Vulkan, 0x113b),
    ApiCall_vkCreateAccelerationStructureNV                                                       = MakeApiCallId(ApiFamily_Vulkan, 0x113c),
    ApiCall_vkDestroyAccelerationStructureNV                                                      = MakeApiCallId(ApiFamily_Vulkan, 0x113d),
    ApiCall_vkGetAccelerationStructureMemoryRequirementsNV                                        = MakeApiCallId(ApiFamily_Vulkan, 0x113e),
    ApiCall_vkBindAccelerationStructureMemoryNV                                                   = MakeApiCallId(ApiFamily_Vulkan, 0x113f),
    ApiCall_vkCmdBuildAccelerationStructureNV                                                     = MakeApiCallId(ApiFamily_Vulkan, 0x1140),
    ApiCall_vkCmdCopyAccelerationStructureNV                                                      = MakeApiCallId(ApiFamily_Vulkan, 0x1141),
    ApiCall_vkCmdTraceRaysNV                                                                      = MakeApiCallId(ApiFamily_Vulkan, 0x1142),
    ApiCall_vkCreateRayTracingPipelinesNV                                                         = MakeApiCallId(ApiFamily_Vulkan, 0x1143),
    ApiCall_vkGetRayTracingShaderGroupHandlesNV                                                   = MakeApiCallId(ApiFamily_Vulkan, 0x1144),
    ApiCall_vkGetAccelerationStructureHandleNV                                                    = MakeApiCallId(ApiFamily_Vulkan, 0x1145),
    ApiCall_vkCmdWriteAccelerationStructuresPropertiesNV                                          = MakeApiCallId(ApiFamily_Vulkan, 0x1146),
    ApiCall_vkCompileDeferredNV                                                                   = MakeApiCallId(ApiFamily_Vulkan, 0x1147),
    ApiCall_vkCmdDrawMeshTasksNV                                                                  = MakeApiCallId(ApiFamily_Vulkan, 0x1148),
    ApiCall_vkCmdDrawMeshTasksIndirectNV                                                          = MakeApiCallId(ApiFamily_Vulkan, 0x1149),
    ApiCall_vkCmdDrawMeshTasksIndirectCountNV                                                     = MakeApiCallId(ApiFamily_Vulkan, 0x114a),
    ApiCall_vkCmdSetExclusiveScissorNV                                                            = MakeApiCallId(ApiFamily_Vulkan, 0x114b),
    ApiCall_vkCmdSetCheckpointNV                                                                  = MakeApiCallId(ApiFamily_Vulkan, 0x114c),
    ApiCall_vkGetQueueCheckpointDataNV                                                            = MakeApiCallId(ApiFamily_Vulkan, 0x114d),
    ApiCall_vkCreateImagePipeSurfaceFUCHSIA                                                       = MakeApiCallId(ApiFamily_Vulkan, 0x114e),
    ApiCall_vkCmdBindTransformFeedbackBuffersEXT                                                  = MakeApiCallId(ApiFamily_Vulkan, 0x114f),
    ApiCall_vkCmdBeginTransformFeedbackEXT                                                        = MakeApiCallId(ApiFamily_Vulkan, 0x1150),
    ApiCall_vkCmdEndTransformFeedbackEXT                                                          = MakeApiCallId(ApiFamily_Vulkan, 0x1151),
    ApiCall_vkCmdBeginQueryIndexedEXT                                                             = MakeApiCallId(ApiFamily_Vulkan, 0x1152),
    ApiCall_vkCmdEndQueryIndexedEXT                                                               = MakeApiCallId(ApiFamily_Vulkan, 0x1153),
    ApiCall_vkCmdDrawIndirectByteCountEXT                                                         = MakeApiCallId(ApiFamily_Vulkan, 0x1154),
    ApiCall_vkGetPhysicalDeviceCalibrateableTimeDomainsEXT                                        = MakeApiCallId(ApiFamily_Vulkan, 0x1155),
    ApiCall_vkGetCalibratedTimestampsEXT                                                          = MakeApiCallId(ApiFamily_Vulkan, 0x1156),
    ApiCall_vkGetBufferDeviceAddressEXT                                                           = MakeApiCallId(ApiFamily_Vulkan, 0x1157),
    ApiCall_vkGetPhysicalDeviceCooperativeMatrixPropertiesNV                                      = MakeApiCallId(ApiFamily_Vulkan, 0x1158),
    ApiCall_vkGetImageViewHandleNVX                                                               = MakeApiCallId(ApiFamily_Vulkan, 0x1159),
    ApiCall_vkCreateMetalSurfaceEXT                                                               = MakeApiCallId(ApiFamily_Vulkan, 0x115a),
    ApiCall_vkCreateStreamDescriptorSurfaceGGP                                                    = MakeApiCallId(ApiFamily_Vulkan, 0x115b),
    ApiCall_vkSetLocalDimmingAMD                                                                  = MakeApiCallId(ApiFamily_Vulkan, 0x115c),
    ApiCall_vkGetPhysicalDeviceSupportedFramebufferMixedSamplesCombinationsNV                     = MakeApiCallId(ApiFamily_Vulkan, 0x115d),
    ApiCall_vkGetPhysicalDeviceSurfacePresentModes2EXT                                            = MakeApiCallId(ApiFamily_Vulkan, 0x115e),
    ApiCall_vkAcquireFullScreenExclusiveModeEXT                                                   = MakeApiCallId(ApiFamily_Vulkan, 0x115f),
    ApiCall_vkReleaseFullScreenExclusiveModeEXT                                                   = MakeApiCallId(ApiFamily_Vulkan, 0x1160),
    ApiCall_vkGetDeviceGroupSurfacePresentModes2EXT                                               = MakeApiCallId(ApiFamily_Vulkan, 0x1161),
    ApiCall_vkCreateHeadlessSurfaceEXT                                                            = MakeApiCallId(ApiFamily_Vulkan, 0x1162),
    ApiCall_vkResetQueryPoolEXT                                                                   = MakeApiCallId(ApiFamily_Vulkan, 0x1163),
    ApiCall_vkGetPipelineExecutablePropertiesKHR                                                  = MakeApiCallId(ApiFamily_Vulkan, 0x1164),
    ApiCall_vkGetPipelineExecutableStatisticsKHR                                                  = MakeApiCallId(ApiFamily_Vulkan, 0x1165),
    ApiCall_vkGetPipelineExecutableInternalRepresentationsKHR                                     = MakeApiCallId(ApiFamily_Vulkan, 0x1166),
    ApiCall_vkInitializePerformanceApiINTEL                                                       = MakeApiCallId(ApiFamily_Vulkan, 0x1167),
    ApiCall_vkUninitializePerformanceApiINTEL                                                     = MakeApiCallId(ApiFamily_Vulkan, 0x1168),
    ApiCall_vkCmdSetPerformanceMarkerINTEL                                                        = MakeApiCallId(ApiFamily_Vulkan, 0x1169),
    ApiCall_vkCmdSetPerformanceStreamMarkerINTEL                                                  = MakeApiCallId(ApiFamily_Vulkan, 0x116a),
    ApiCall_vkCmdSetPerformanceOverrideINTEL                                                      = MakeApiCallId(ApiFamily_Vulkan, 0x116b),
    ApiCall_vkAcquirePerformanceConfigurationINTEL                                                = MakeApiCallId(ApiFamily_Vulkan, 0x116c),
    ApiCall_vkReleasePerformanceConfigurationINTEL                                                = MakeApiCallId(ApiFamily_Vulkan, 0x116d),
    ApiCall_vkQueueSetPerformanceConfigurationINTEL                                               = MakeApiCallId(ApiFamily_Vulkan, 0x116e),
    ApiCall_vkGetPerformanceParameterINTEL                                                        = MakeApiCallId(ApiFamily_Vulkan, 0x116f),
    ApiCall_vkCmdSetLineStippleEXT                                                                = MakeApiCallId(ApiFamily_Vulkan, 0x1170),
    ApiCall_vkGetSemaphoreCounterValueKHR                                                         = MakeApiCallId(ApiFamily_Vulkan, 0x1171),
    ApiCall_vkWaitSemaphoresKHR                                                                   = MakeApiCallId(ApiFamily_Vulkan, 0x1172),
    ApiCall_vkSignalSemaphoreKHR                                                                  = MakeApiCallId(ApiFamily_Vulkan, 0x1173),
    ApiCall_vkEnumeratePhysicalDeviceQueueFamilyPerformanceQueryCountersKHR                       = MakeApiCallId(ApiFamily_Vulkan, 0x1174),
    ApiCall_vkGetPhysicalDeviceQueueFamilyPerformanceQueryPassesKHR                               = MakeApiCallId(ApiFamily_Vulkan, 0x1175),
    ApiCall_vkAcquireProfilingLockKHR                                                             = MakeApiCallId(ApiFamily_Vulkan, 0x1176),
    ApiCall_vkReleaseProfilingLockKHR                                                             = MakeApiCallId(ApiFamily_Vulkan, 0x1177),
    ApiCall_vkGetBufferDeviceAddressKHR                                                           = MakeApiCallId(ApiFamily_Vulkan, 0x1178),
    ApiCall_vkGetBufferOpaqueCaptureAddressKHR                                                    = MakeApiCallId(ApiFamily_Vulkan, 0x1179),
    ApiCall_vkGetDeviceMemoryOpaqueCaptureAddressKHR                                              = MakeApiCallId(ApiFamily_Vulkan, 0x117a),
    ApiCall_vkGetPhysicalDeviceToolPropertiesEXT                                                  = MakeApiCallId(ApiFamily_Vulkan, 0x117b),
    ApiCall_vkCmdDrawIndirectCount                                                                = MakeApiCallId(ApiFamily_Vulkan, 0x117c),
    ApiCall_vkCmdDrawIndexedIndirectCount                                                         = MakeApiCallId(ApiFamily_Vulkan, 0x117d),
    ApiCall_vkCreateRenderPass2                                                                   = MakeApiCallId(ApiFamily_Vulkan, 0x117e),
    ApiCall_vkCmdBeginRenderPass2                                                                 = MakeApiCallId(ApiFamily_Vulkan, 0x117f),
    ApiCall_vkCmdNextSubpass2                                                                     = MakeApiCallId(ApiFamily_Vulkan, 0x1180),
    ApiCall_vkCmdEndRenderPass2                                                                   = MakeApiCallId(ApiFamily_Vulkan, 0x1181),
    ApiCall_vkResetQueryPool                                                                      = MakeApiCallId(ApiFamily_Vulkan, 0x1182),
    ApiCall_vkGetSemaphoreCounterValue                                                            = MakeApiCallId(ApiFamily_Vulkan, 0x1183),
    ApiCall_vkWaitSemaphores                                                                      = MakeApiCallId(ApiFamily_Vulkan, 0x1184),
    ApiCall_vkSignalSemaphore                                                                     = MakeApiCallId(ApiFamily_Vulkan, 0x1185),
    ApiCall_vkGetBufferDeviceAddress                                                              = MakeApiCallId(ApiFamily_Vulkan, 0x1186),
    ApiCall_vkGetBufferOpaqueCaptureAddress                                                       = MakeApiCallId(ApiFamily_Vulkan, 0x1187),
    ApiCall_vkGetDeviceMemoryOpaqueCaptureAddress                                                 = MakeApiCallId(ApiFamily_Vulkan, 0x1188),
    ApiCall_vkCreateDeferredOperationKHR                                                          = MakeApiCallId(ApiFamily_Vulkan, 0x1189),
    ApiCall_vkDestroyDeferredOperationKHR                                                         = MakeApiCallId(ApiFamily_Vulkan, 0x118a),
    ApiCall_vkGetDeferredOperationMaxConcurrencyKHR                                               = MakeApiCallId(ApiFamily_Vulkan, 0x118b),
    ApiCall_vkGetDeferredOperationResultKHR                                                       = MakeApiCallId(ApiFamily_Vulkan, 0x118c),
    ApiCall_vkDeferredOperationJoinKHR                                                            = MakeApiCallId(ApiFamily_Vulkan, 0x118d),
    ApiCall_vkCreateAccelerationStructureKHR                                                      = MakeApiCallId(ApiFamily_Vulkan, 0x118e),
    ApiCall_vkDestroyAccelerationStructureKHR                                                     = MakeApiCallId(ApiFamily_Vulkan, 0x118f),
    ApiCall_vkGetAccelerationStructureMemoryRequirementsKHR                                       = MakeApiCallId(ApiFamily_Vulkan, 0x1190),
    ApiCall_vkBindAccelerationStructureMemoryKHR                                                  = MakeApiCallId(ApiFamily_Vulkan, 0x1191),
    ApiCall_vkCmdBuildAccelerationStructuresKHR                                                   = MakeApiCallId(ApiFamily_Vulkan, 0x1192),
    ApiCall_vkCmdBuildAccelerationStructuresIndirectKHR                                           = MakeApiCallId(ApiFamily_Vulkan, 0x1193),
    ApiCall_vkBuildAccelerationStructuresKHR                                                      = MakeApiCallId(ApiFamily_Vulkan, 0x1194),
    ApiCall_vkCopyAccelerationStructureKHR                                                        = MakeApiCallId(ApiFamily_Vulkan, 0x1195),
    ApiCall_vkCopyAccelerationStructureToMemoryKHR                                                = MakeApiCallId(ApiFamily_Vulkan, 0x1196),
    ApiCall_vkCopyMemoryToAccelerationStructureKHR                                                = MakeApiCallId(ApiFamily_Vulkan, 0x1197),
    ApiCall_vkWriteAccelerationStructuresPropertiesKHR                                            = MakeApiCallId(ApiFamily_Vulkan, 0x1198),
    ApiCall_vkCmdCopyAccelerationStructureKHR                                                     = MakeApiCallId(ApiFamily_Vulkan, 0x1199),
    ApiCall_vkCmdCopyAccelerationStructureToMemoryKHR                                             = MakeApiCallId(ApiFamily_Vulkan, 0x119a),
    ApiCall_vkCmdCopyMemoryToAccelerationStructureKHR                                             = MakeApiCallId(ApiFamily_Vulkan, 0x119b),
    ApiCall_vkCmdTraceRaysKHR                                                                     = MakeApiCallId(ApiFamily_Vulkan, 0x119c),
    ApiCall_vkCreateRayTracingPipelinesKHR                                                        = MakeApiCallId(ApiFamily_Vulkan, 0x119d),
    ApiCall_vkGetRayTracingShaderGroupHandlesKHR                                                  = MakeApiCallId(ApiFamily_Vulkan, 0x119e),
    ApiCall_vkCmdWriteAccelerationStructuresPropertiesKHR                                         = MakeApiCallId(ApiFamily_Vulkan, 0x119f),
    ApiCall_vkGetAccelerationStructureDeviceAddressKHR                                            = MakeApiCallId(ApiFamily_Vulkan, 0x1200),
    ApiCall_vkGetRayTracingCaptureReplayShaderGroupHandlesKHR                                     = MakeApiCallId(ApiFamily_Vulkan, 0x1201),
    ApiCall_vkCmdTraceRaysIndirectKHR                                                             = MakeApiCallId(ApiFamily_Vulkan, 0x1202),
    ApiCall_vkGetDeviceAccelerationStructureCompatibilityKHR                                      = MakeApiCallId(ApiFamily_Vulkan, 0x1203),
    ApiCall_vkGetGeneratedCommandsMemoryRequirementsNV                                            = MakeApiCallId(ApiFamily_Vulkan, 0x1204),
    ApiCall_vkCmdPreprocessGeneratedCommandsNV                                                    = MakeApiCallId(ApiFamily_Vulkan, 0x1205),
    ApiCall_vkCmdExecuteGeneratedCommandsNV                                                       = MakeApiCallId(ApiFamily_Vulkan, 0x1206),
    ApiCall_vkCmdBindPipelineShaderGroupNV                                                        = MakeApiCallId(ApiFamily_Vulkan, 0x1207),
    ApiCall_vkCreateIndirectCommandsLayoutNV                                                      = MakeApiCallId(ApiFamily_Vulkan, 0x1208),
    ApiCall_vkDestroyIndirectCommandsLayoutNV                                                     = MakeApiCallId(ApiFamily_Vulkan, 0x1209),
    ApiCall_vkGetImageViewAddressNVX                                                              = MakeApiCallId(ApiFamily_Vulkan, 0x120a),
    ApiCall_vkCreatePrivateDataSlotEXT                                                            = MakeApiCallId(ApiFamily_Vulkan, 0x120b),
    ApiCall_vkDestroyPrivateDataSlotEXT                                                           = MakeApiCallId(ApiFamily_Vulkan, 0x120c),
    ApiCall_vkSetPrivateDataEXT                                                                   = MakeApiCallId(ApiFamily_Vulkan, 0x120d),
    ApiCall_vkGetPrivateDataEXT                                                                   = MakeApiCallId(ApiFamily_Vulkan, 0x120e),
    ApiCall_vkCmdSetCullModeEXT                                                                   = MakeApiCallId(ApiFamily_Vulkan, 0x120f),
    ApiCall_vkCmdSetFrontFaceEXT                                                                  = MakeApiCallId(ApiFamily_Vulkan, 0x1210),
    ApiCall_vkCmdSetPrimitiveTopologyEXT                                                          = MakeApiCallId(ApiFamily_Vulkan, 0x1211),
    ApiCall_vkCmdSetViewportWithCountEXT                                                          = MakeApiCallId(ApiFamily_Vulkan, 0x1212),
    ApiCall_vkCmdSetScissorWithCountEXT                                                           = MakeApiCallId(ApiFamily_Vulkan, 0x1213),
    ApiCall_vkCmdBindVertexBuffers2EXT                                                            = MakeApiCallId(ApiFamily_Vulkan, 0x1214),
    ApiCall_vkCmdSetDepthTestEnableEXT                                                            = MakeApiCallId(ApiFamily_Vulkan, 0x1215),
    ApiCall_vkCmdSetDepthWriteEnableEXT                                                           = MakeApiCallId(ApiFamily_Vulkan, 0x1216),
    ApiCall_vkCmdSetDepthCompareOpEXT                                                             = MakeApiCallId(ApiFamily_Vulkan, 0x1217),
    ApiCall_vkCmdSetDepthBoundsTestEnableEXT                                                      = MakeApiCallId(ApiFamily_Vulkan, 0x1218),
    ApiCall_vkCmdSetStencilTestEnableEXT                                                          = MakeApiCallId(ApiFamily_Vulkan, 0x1219),
    ApiCall_vkCmdSetStencilOpEXT                                                                  = MakeApiCallId(ApiFamily_Vulkan, 0x121a),
    ApiCall_vkCreateDirectFBSurfaceEXT                                                            = MakeApiCallId(ApiFamily_Vulkan, 0x121b),
    ApiCall_vkGetPhysicalDeviceDirectFBPresentationSupportEXT                                     = MakeApiCallId(ApiFamily_Vulkan, 0x121c),
    ApiCall_vkCmdCopyBuffer2KHR                                                                   = MakeApiCallId(ApiFamily_Vulkan, 0x121d),
    ApiCall_vkCmdCopyImage2KHR                                                                    = MakeApiCallId(ApiFamily_Vulkan, 0x121e),
    ApiCall_vkCmdCopyBufferToImage2KHR                                                            = MakeApiCallId(ApiFamily_Vulkan, 0x121f),
    ApiCall_vkCmdCopyImageToBuffer2KHR                                                            = MakeApiCallId(ApiFamily_Vulkan, 0x1220),
    ApiCall_vkCmdBlitImage2KHR                                                                    = MakeApiCallId(ApiFamily_Vulkan, 0x1221),
    ApiCall_vkCmdResolveImage2KHR                                                                 = MakeApiCallId(ApiFamily_Vulkan, 0x1222),
    ApiCall_vkGetAccelerationStructureBuildSizesKHR                                               = MakeApiCallId(ApiFamily_Vulkan, 0x1223),
    ApiCall_vkGetRayTracingShaderGroupStackSizeKHR                                                = MakeApiCallId(ApiFamily_Vulkan, 0x1224),
    ApiCall_vkCmdSetRayTracingPipelineStackSizeKHR                                                = MakeApiCallId(ApiFamily_Vulkan, 0x1225),
    ApiCall_vkGetPhysicalDeviceFragmentShadingRatesKHR                                            = MakeApiCallId(ApiFamily_Vulkan, 0x1226),
    ApiCall_vkCmdSetFragmentShadingRateKHR                                                        = MakeApiCallId(ApiFamily_Vulkan, 0x1227),
    ApiCall_vkCmdSetFragmentShadingRateEnumNV                                                     = MakeApiCallId(ApiFamily_Vulkan, 0x1228),
    ApiCall_vkAcquireWinrtDisplayNV                                                               = MakeApiCallId(ApiFamily_Vulkan, 0x1229),
    ApiCall_vkGetWinrtDisplayNV                                                                   = MakeApiCallId(ApiFamily_Vulkan, 0x122a),
    ApiCall_vkCmdSetEvent2KHR                                                                     = MakeApiCallId(ApiFamily_Vulkan, 0x122b),
    ApiCall_vkCmdResetEvent2KHR                                                                   = MakeApiCallId(ApiFamily_Vulkan, 0x122c),
    ApiCall_vkCmdWaitEvents2KHR                                                                   = MakeApiCallId(ApiFamily_Vulkan, 0x122d),
    ApiCall_vkCmdPipelineBarrier2KHR                                                              = MakeApiCallId(ApiFamily_Vulkan, 0x122e),
    ApiCall_vkCmdWriteTimestamp2KHR                                                               = MakeApiCallId(ApiFamily_Vulkan, 0x122f),
    ApiCall_vkQueueSubmit2KHR                                                                     = MakeApiCallId(ApiFamily_Vulkan, 0x1230),
    ApiCall_vkCmdWriteBufferMarker2AMD                                                            = MakeApiCallId(ApiFamily_Vulkan, 0x1231),
    ApiCall_vkGetQueueCheckpointData2NV                                                           = MakeApiCallId(ApiFamily_Vulkan, 0x1232),
    ApiCall_vkCmdSetVertexInputEXT                                                                = MakeApiCallId(ApiFamily_Vulkan, 0x1233),
    ApiCall_vkGetMemoryZirconHandleFUCHSIA                                                        = MakeApiCallId(ApiFamily_Vulkan, 0x1234),
    ApiCall_vkGetMemoryZirconHandlePropertiesFUCHSIA                                              = MakeApiCallId(ApiFamily_Vulkan, 0x1235),
    ApiCall_vkImportSemaphoreZirconHandleFUCHSIA                                                  = MakeApiCallId(ApiFamily_Vulkan, 0x1236),
    ApiCall_vkGetSemaphoreZirconHandleFUCHSIA                                                     = MakeApiCallId(ApiFamily_Vulkan, 0x1237),
    ApiCall_vkCmdSetPatchControlPointsEXT                                                         = MakeApiCallId(ApiFamily_Vulkan, 0x1238),
    ApiCall_vkCmdSetRasterizerDiscardEnableEXT                                                    = MakeApiCallId(ApiFamily_Vulkan, 0x1239),
    ApiCall_vkCmdSetDepthBiasEnableEXT                                                            = MakeApiCallId(ApiFamily_Vulkan, 0x123a),
    ApiCall_vkCmdSetLogicOpEXT                                                                    = MakeApiCallId(ApiFamily_Vulkan, 0x123b),
    ApiCall_vkCmdSetPrimitiveRestartEnableEXT                                                     = MakeApiCallId(ApiFamily_Vulkan, 0x123c),
    ApiCall_vkCreateScreenSurfaceQNX                                                              = MakeApiCallId(ApiFamily_Vulkan, 0x123d),
    ApiCall_vkGetPhysicalDeviceScreenPresentationSupportQNX                                       = MakeApiCallId(ApiFamily_Vulkan, 0x123e),
    ApiCall_vkCmdSetColorWriteEnableEXT                                                           = MakeApiCallId(ApiFamily_Vulkan, 0x123f),
    ApiCall_vkAcquireDrmDisplayEXT                                                                = MakeApiCallId(ApiFamily_Vulkan, 0x1240),
    ApiCall_vkGetDrmDisplayEXT                                                                    = MakeApiCallId(ApiFamily_Vulkan, 0x1241),
    ApiCall_vkCmdDrawMultiEXT                                                                     = MakeApiCallId(ApiFamily_Vulkan, 0x1242),
    ApiCall_vkCmdDrawMultiIndexedEXT                                                              = MakeApiCallId(ApiFamily_Vulkan, 0x1243),
    ApiCall_vkWaitForPresentKHR                                                                   = MakeApiCallId(ApiFamily_Vulkan, 0x1244),
    ApiCall_vkCmdBindInvocationMaskHUAWEI                                                         = MakeApiCallId(ApiFamily_Vulkan, 0x1245),
    ApiCall_vkGetMemoryRemoteAddressNV                                                            = MakeApiCallId(ApiFamily_Vulkan, 0x1246),

    ApiCall_VulkanLast
    // clang-format on
};

GFXRECON_END_NAMESPACE(gfxrecon)
GFXRECON_END_NAMESPACE(format)

#endif // GFXRECON_FORMAT_API_CALL_ID_H
