static const char* op_c_source =
"/* This file is an image processing operation for GEGL                        \n"
" *                                                                            \n"
" * GEGL is free software; you can redistribute it and/or                      \n"
" * modify it under the terms of the GNU Lesser General Public                 \n"
" * License as published by the Free Software Foundation; either               \n"
" * version 3 of the License, or (at your option) any later version.           \n"
" *                                                                            \n"
" * GEGL is distributed in the hope that it will be useful,                    \n"
" * but WITHOUT ANY WARRANTY; without even the implied warranty of             \n"
" * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU          \n"
" * Lesser General Public License for more details.                            \n"
" *                                                                            \n"
" * You should have received a copy of the GNU Lesser General Public           \n"
" * License along with GEGL; if not, see <http://www.gnu.org/licenses/>.       \n"
" *                                                                            \n"
" * Copyright 2010      Danny Robson      <danny@blubinc.net>                  \n"
" * (pfstmo)  2007      Grzegorz Krawczyk <krawczyk@mpi-sb.mpg.de>             \n"
" *           2007-2008 Ed Brambley       <E.J.Brambley@damtp.cam.ac.uk>       \n"
" *                     Lebed Dmytry                                           \n"
" *                     Radoslaw Mantiuk  <radoslaw.mantiuk@gmail.com>         \n"
" *                     Rafal Mantiuk     <mantiuk@gmail.com>                  \n"
" */                                                                           \n"
"                                                                              \n"
"#include \"config.h\"                                                         \n"
"#include <glib/gi18n-lib.h>                                                   \n"
"#include <math.h>                                                             \n"
"                                                                              \n"
"                                                                              \n"
"#ifdef GEGL_PROPERTIES                                                        \n"
"                                                                              \n"
"property_double (contrast, _(\"Contrast\"), 0.1)                              \n"
"    description (_(\"The amount of contrast compression\"))                   \n"
"    value_range (0.0, 1.0)                                                    \n"
"                                                                              \n"
"property_double (saturation, _(\"Saturation\"), 0.8)                          \n"
"    description ((\"Global color saturation factor\"))                        \n"
"    value_range (0.0, 2.0)                                                    \n"
"                                                                              \n"
"property_double (detail, _(\"Detail\"), 1.0)                                  \n"
"    description (_(\"Level of emphasis on image gradient details\"))          \n"
"    value_range (1.0, 99.0)                                                   \n"
"                                                                              \n"
"#else                                                                         \n"
"                                                                              \n"
"#define GEGL_OP_FILTER                                                        \n"
"#define GEGL_OP_NAME     mantiuk06                                            \n"
"#define GEGL_OP_C_SOURCE mantiuk06.c                                          \n"
"                                                                              \n"
"#include \"gegl-op.h\"                                                        \n"
"#include <stdio.h>                                                            \n"
"#include <stdlib.h>                                                           \n"
"                                                                              \n"
"#ifdef HAVE_OPENMP                                                            \n"
"#define _OMP(x) _Pragma(#x)                                                   \n"
"#else                                                                         \n"
"#define _OMP(x) /* OMP disabled: \"#x\" */                                    \n"
"#endif                                                                        \n"
"                                                                              \n"
"/* Common return codes for operators */                                       \n"
"#define PFSTMO_OK 1             /* Successful */                              \n"
"#define PFSTMO_ABORTED -1       /* User aborted (from callback) */            \n"
"#define PFSTMO_ERROR -2         /* Failed, encountered error */               \n"
"                                                                              \n"
"/* Return codes for the progress_callback */                                  \n"
"#define PFSTMO_CB_CONTINUE  1                                                 \n"
"#define PFSTMO_CB_ABORT    -1                                                 \n"
"                                                                              \n"
"typedef struct pyramid_s {                                                    \n"
"  guint              rows;                                                    \n"
"  guint              cols;                                                    \n"
"  gfloat            *Gx;                                                      \n"
"  gfloat            *Gy;                                                      \n"
"  struct pyramid_s  *next;                                                    \n"
"  struct pyramid_s  *prev;                                                    \n"
"} pyramid_t;                                                                  \n"
"                                                                              \n"
"                                                                              \n"
"#define PYRAMID_MIN_PIXELS 3                                                  \n"
"#define LOOKUP_W_TO_R 107                                                     \n"
"                                                                              \n"
"typedef int (*pfstmo_progress_callback)(int progress);                        \n"
"                                                                              \n"
"                                                                              \n"
"static void        mantiuk06_contrast_equalization            (pyramid_t                       *pp,\n"
"                                                               const gfloat                     contrastFactor);\n"
"static void        mantiuk06_transform_to_luminance           (pyramid_t                       *pp,\n"
"                                                               gfloat                   *const  x,\n"
"                                                               pfstmo_progress_callback         progress,\n"
"                                                               const gboolean                   bcg,\n"
"                                                               const int                        itmax,\n"
"                                                               const gfloat                     tol);\n"
"static gfloat*     mantiuk06_matrix_alloc                     (const guint                      size);\n"
"static void        mantiuk06_matrix_free                      (gfloat                          *m);\n"
"static void        mantiuk06_matrix_zero                      (const guint                      n,\n"
"                                                               gfloat     *const                m);\n"
"static void        mantiuk06_matrix_copy                      (const guint                      n,\n"
"                                                               const gfloat *const              a,\n"
"                                                               gfloat       *const              b);\n"
"static void        mantiuk06_matrix_subtract                  (const guint                      n,\n"
"                                                               const gfloat *const              a,\n"
"                                                               gfloat *const                    b);\n"
"static void        mantiuk06_matrix_multiply_const            (const guint                      n,\n"
"                                                               gfloat       *const              a,\n"
"                                                               const gfloat                     val);\n"
"static void        mantiuk06_matrix_divide                    (const guint                      n,\n"
"                                                               const gfloat *const              a,\n"
"                                                               gfloat *const                    b);\n"
"static gfloat      mantiuk06_matrix_dot_product               (const guint                      n,\n"
"                                                               const gfloat *const              a,\n"
"                                                               const gfloat *const              b);\n"
"static void        mantiuk06_calculate_and_add_divergence     (const int                        rows,\n"
"                                                               const int                        cols,\n"
"                                                               const gfloat *const              Gx,\n"
"                                                               const gfloat *const              Gy,\n"
"                                                               gfloat       *const              divG);\n"
"static void        mantiuk06_pyramid_calculate_divergence_sum (pyramid_t                       *pyramid,\n"
"                                                               gfloat                          *divG_sum);\n"
"static void        mantiuk06_calculate_scale_factor           (const int                        n,\n"
"                                                               const gfloat *const              G,\n"
"                                                               gfloat       *const              C);\n"
"static void        mantiuk06_pyramid_calculate_scale_factor   (pyramid_t                       *pyramid,\n"
"                                                               pyramid_t                       *pC);\n"
"static void        mantiuk06_scale_gradient                   (const int                        n,\n"
"                                                               gfloat       *const              G,\n"
"                                                               const gfloat *const              C);\n"
"static void        mantiuk06_pyramid_scale_gradient           (pyramid_t                       *pyramid,\n"
"                                                               pyramid_t                       *pC);\n"
"static void        mantiuk06_pyramid_free                     (pyramid_t                       *pyramid);\n"
"static pyramid_t*  mantiuk06_pyramid_allocate                 (const int                        cols,\n"
"                                                               const int                        rows);\n"
"static void        mantiuk06_calculate_gradient               (const int                        cols,\n"
"                                                               const int                        rows,\n"
"                                                               const gfloat *const              lum,\n"
"                                                               gfloat       *const              Gx,\n"
"                                                               gfloat       *const              Gy);\n"
"static void        mantiuk06_pyramid_calculate_gradient       (pyramid_t                       *pyramid,\n"
"                                                               gfloat                          *lum);\n"
"static void        mantiuk06_solveX                           (const gint                       n,\n"
"                                                               const gfloat *const              b,\n"
"                                                               gfloat       *const              x);\n"
"static void        mantiuk06_multiplyA                        (pyramid_t                       *px,\n"
"                                                               pyramid_t                       *pyramid,\n"
"                                                               const gfloat *const              x,\n"
"                                                               gfloat       *const              divG_sum);\n"
"static void        mantiuk06_linbcg                           (pyramid_t                       *pyramid,\n"
"                                                               pyramid_t                       *pC,\n"
"                                                               const gfloat *const              b,\n"
"                                                               gfloat       *const              x,\n"
"                                                               const int                        itmax,\n"
"                                                               const gfloat                     tol,\n"
"                                                               pfstmo_progress_callback         progress_cb);\n"
"static void        mantiuk06_lincg                            (pyramid_t                       *pyramid,\n"
"                                                               pyramid_t                       *pC,\n"
"                                                               const gfloat *const              b,\n"
"                                                               gfloat       *const              x,\n"
"                                                               const int                        itmax,\n"
"                                                               const gfloat                     tol,\n"
"                                                               pfstmo_progress_callback         progress_cb);\n"
"static gfloat      mantiuk06_lookup_table                     (const int                        n,\n"
"                                                               const gfloat *const              in_tab,\n"
"                                                               const gfloat *const              out_tab,\n"
"                                                               const gfloat                     val);\n"
"static void        mantiuk06_transform_to_R                   (const int                        n,\n"
"                                                               gfloat     *const                G);\n"
"static void        mantiuk06_pyramid_transform_to_R           (pyramid_t                       *pyramid);\n"
"static void        mantiuk06_transform_to_G                   (const gint                       n,\n"
"                                                               gfloat     *const                R);\n"
"static void        mantiuk06_pyramid_transform_to_G           (pyramid_t                       *pyramid);\n"
"static void        mantiuk06_pyramid_gradient_multiply        (pyramid_t                       *pyramid,\n"
"                                                               const gfloat                     val);\n"
"                                                                              \n"
"                                                                              \n"
"static const gchar *OUTPUT_FORMAT = \"RGBA float\";                           \n"
"                                                                              \n"
"static gfloat W_table[] =                                                     \n"
"{                                                                             \n"
"     0.000000,     0.010000,    0.021180,    0.031830,    0.042628,           \n"
"     0.053819,     0.065556,    0.077960,    0.091140,    0.105203,           \n"
"     0.120255,     0.136410,    0.153788,    0.172518,    0.192739,           \n"
"     0.214605,     0.238282,    0.263952,    0.291817,    0.322099,           \n"
"     0.355040,     0.390911,    0.430009,    0.472663,    0.519238,           \n"
"     0.570138,     0.625811,    0.686754,    0.753519,    0.826720,           \n"
"     0.907041,     0.995242,    1.092169,    1.198767,    1.316090,           \n"
"     1.445315,     1.587756,    1.744884,    1.918345,    2.109983,           \n"
"     2.321863,     2.556306,    2.815914,    3.103613,    3.422694,           \n"
"     3.776862,     4.170291,    4.607686,    5.094361,    5.636316,           \n"
"     6.240338,     6.914106,    7.666321,    8.506849,    9.446889,           \n"
"    10.499164,    11.678143,   13.000302,   14.484414,   16.151900,           \n"
"    18.027221,    20.138345,   22.517282,   25.200713,   28.230715,           \n"
"    31.655611,    35.530967,   39.920749,   44.898685,   50.549857,           \n"
"    56.972578,    64.280589,   72.605654,   82.100619,   92.943020,           \n"
"   105.339358,   119.530154,  135.795960,  154.464484,  175.919088,           \n"
"   200.608905,   229.060934,  261.894494,  299.838552,  343.752526,           \n"
"   394.651294,   453.735325,  522.427053,  602.414859,  695.706358,           \n"
"   804.693100,   932.229271, 1081.727632, 1257.276717, 1463.784297,           \n"
"  1707.153398,  1994.498731, 2334.413424, 2737.298517, 3215.770944,           \n"
"  3785.169959,  4464.187290, 5275.653272, 6247.520102, 7414.094945,           \n"
"  8817.590551, 10510.080619                                                   \n"
"};                                                                            \n"
"                                                                              \n"
"static gfloat R_table[] =                                                     \n"
"{                                                                             \n"
"  0.000000, 0.009434, 0.018868, 0.028302, 0.037736,                           \n"
"  0.047170, 0.056604, 0.066038, 0.075472, 0.084906,                           \n"
"  0.094340, 0.103774, 0.113208, 0.122642, 0.132075,                           \n"
"  0.141509, 0.150943, 0.160377, 0.169811, 0.179245,                           \n"
"  0.188679, 0.198113, 0.207547, 0.216981, 0.226415,                           \n"
"  0.235849, 0.245283, 0.254717, 0.264151, 0.273585,                           \n"
"  0.283019, 0.292453, 0.301887, 0.311321, 0.320755,                           \n"
"  0.330189, 0.339623, 0.349057, 0.358491, 0.367925,                           \n"
"  0.377358, 0.386792, 0.396226, 0.405660, 0.415094,                           \n"
"  0.424528, 0.433962, 0.443396, 0.452830, 0.462264,                           \n"
"  0.471698, 0.481132, 0.490566, 0.500000, 0.509434,                           \n"
"  0.518868, 0.528302, 0.537736, 0.547170, 0.556604,                           \n"
"  0.566038, 0.575472, 0.584906, 0.594340, 0.603774,                           \n"
"  0.613208, 0.622642, 0.632075, 0.641509, 0.650943,                           \n"
"  0.660377, 0.669811, 0.679245, 0.688679, 0.698113,                           \n"
"  0.707547, 0.716981, 0.726415, 0.735849, 0.745283,                           \n"
"  0.754717, 0.764151, 0.773585, 0.783019, 0.792453,                           \n"
"  0.801887, 0.811321, 0.820755, 0.830189, 0.839623,                           \n"
"  0.849057, 0.858491, 0.867925, 0.877358, 0.886792,                           \n"
"  0.896226, 0.905660, 0.915094, 0.924528, 0.933962,                           \n"
"  0.943396, 0.952830, 0.962264, 0.971698, 0.981132,                           \n"
"  0.990566, 1.000000                                                          \n"
"};                                                                            \n"
"                                                                              \n"
"                                                                              \n"
"/* upsample the matrix                                                        \n"
" * upsampled matrix is twice bigger in each direction than data[]             \n"
" * res should be a pointer to allocated memory for bigger matrix              \n"
" * cols and rows are the dimmensions of the output matrix                     \n"
" */                                                                           \n"
"static void                                                                   \n"
"mantiuk06_matrix_upsample (const gint          outCols,                       \n"
"                           const gint          outRows,                       \n"
"                           const gfloat *const in,                            \n"
"                           gfloat       *const out)                           \n"
"{                                                                             \n"
"  const int inRows = outRows/2;                                               \n"
"  const int inCols = outCols/2;                                               \n"
"  gint      x, y;                                                             \n"
"                                                                              \n"
"  /* Transpose of experimental downsampling matrix (theoretically the         \n"
"   * correct thing to do)                                                     \n"
"   */                                                                         \n"
"                                                                              \n"
"  const gfloat dx = (gfloat)inCols / ((gfloat)outCols);                       \n"
"  const gfloat dy = (gfloat)inRows / ((gfloat)outRows);                       \n"
"  const gfloat factor = 1.0f / (dx*dy); /* This gives a genuine upsampling    \n"
"                                         * matrix, not the transpose of the   \n"
"                                         * downsampling matrix                \n"
"                                         */                                   \n"
"  /* const gfloat factor = 1.0f; */     /* Theoretically, this should be the  \n"
"                                         * best.                              \n"
"                                         */                                   \n"
"                                                                              \n"
"  _OMP (omp parallel for schedule(static))                                    \n"
"  for (y = 0; y < outRows; y++)                                               \n"
"    {                                                                         \n"
"      const gfloat sy  = y * dy;                                              \n"
"      const gint   iy1 =      (  y   * inRows) / outRows;                     \n"
"      const gint   iy2 = MIN (((y+1) * inRows) / outRows, inRows-1);          \n"
"                                                                              \n"
"      for (x = 0; x < outCols; x++)                                           \n"
"        {                                                                     \n"
"          const gfloat sx  = x * dx;                                          \n"
"          const gint   ix1 =      (  x    * inCols) / outCols;                \n"
"          const gint   ix2 =  MIN (((x+1) * inCols) / outCols, inCols-1);     \n"
"                                                                              \n"
"          out[x + y*outCols] = (                                              \n"
"            ((ix1+1) - sx)*((iy1+1 - sy)) * in[ix1 + iy1*inCols] +            \n"
"            ((ix1+1) - sx)*(sy+dy - (iy1+1)) * in[ix1 + iy2*inCols] +         \n"
"            (sx+dx - (ix1+1))*((iy1+1 - sy)) * in[ix2 + iy1*inCols] +         \n"
"            (sx+dx - (ix1+1))*(sy+dx - (iy1+1)) * in[ix2 + iy2*inCols])*factor;\n"
"        }                                                                     \n"
"    }                                                                         \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"/* downsample the matrix */                                                   \n"
"static void                                                                   \n"
"mantiuk06_matrix_downsample (const gint          inCols,                      \n"
"                             const gint          inRows,                      \n"
"                             const gfloat *const data,                        \n"
"                             gfloat       *const res)                         \n"
"{                                                                             \n"
"  const int outRows = inRows / 2;                                             \n"
"  const int outCols = inCols / 2;                                             \n"
"  gint      x, y, i, j;                                                       \n"
"                                                                              \n"
"  const gfloat dx = (gfloat)inCols / ((gfloat)outCols);                       \n"
"  const gfloat dy = (gfloat)inRows / ((gfloat)outRows);                       \n"
"                                                                              \n"
"  /* New downsampling by Ed Brambley:                                         \n"
"   * Experimental downsampling that assumes pixels are square and             \n"
"   * integrates over each new pixel to find the average value of the          \n"
"   * underlying pixels.                                                       \n"
"   *                                                                          \n"
"   * Consider the original pixels laid out, and the new (larger)              \n"
"   * pixels layed out over the top of them.  Then the new value for           \n"
"   * the larger pixels is just the integral over that pixel of what           \n"
"   * shows through; i.e., the values of the pixels underneath                 \n"
"   * multiplied by how much of that pixel is showing.                         \n"
"   *                                                                          \n"
"   * (ix1, iy1) is the coordinate of the top left visible pixel.              \n"
"   * (ix2, iy2) is the coordinate of the bottom right visible pixel.          \n"
"   * (fx1, fy1) is the fraction of the top left pixel showing.                \n"
"   * (fx2, fy2) is the fraction of the bottom right pixel showing.            \n"
"   */                                                                         \n"
"                                                                              \n"
"  const gfloat normalize = 1.0f/(dx*dy);                                      \n"
"  _OMP (omp parallel for schedule(static))                                    \n"
"  for (y = 0; y < outRows; y++)                                               \n"
"    {                                                                         \n"
"      const gint   iy1 = (  y   * inRows) / outRows;                          \n"
"      const gint   iy2 = ((y+1) * inRows) / outRows;                          \n"
"      const gfloat fy1 = (iy1+1) - y * dy;                                    \n"
"      const gfloat fy2 = (y+1) * dy - iy2;                                    \n"
"                                                                              \n"
"      for (x = 0; x < outCols; x++)                                           \n"
"        {                                                                     \n"
"          const gint   ix1 = (  x   * inCols) / outCols;                      \n"
"          const gint   ix2 = ((x+1) * inCols) / outCols;                      \n"
"          const gfloat fx1 = (ix1+1) - x * dx;                                \n"
"          const gfloat fx2 = (x+1) * dx - ix2;                                \n"
"                                                                              \n"
"          gfloat pixVal = 0.0f;                                               \n"
"          gfloat factorx, factory;                                            \n"
"                                                                              \n"
"          for (i = iy1; i <= iy2 && i < inRows; i++)                          \n"
"            {                                                                 \n"
"              if (i == iy1)                                                   \n"
"                {                                                             \n"
"                  /* We're just getting the bottom edge of this pixel */      \n"
"                  factory = fy1;                                              \n"
"                }                                                             \n"
"              else if (i == iy2)                                              \n"
"                {                                                             \n"
"                  /* We're just gettting the top edge of this pixel */        \n"
"                  factory = fy2;                                              \n"
"                }                                                             \n"
"              else                                                            \n"
"                {                                                             \n"
"                  /* We've got the full height of this pixel */               \n"
"                  factory = 1.0f;                                             \n"
"                }                                                             \n"
"                                                                              \n"
"              for (j = ix1; j <= ix2 && j < inCols; j++)                      \n"
"                {                                                             \n"
"                  if (j == ix1)                                               \n"
"                    {                                                         \n"
"                      /* We've just got the right edge of this pixel */       \n"
"                      factorx = fx1;                                          \n"
"                    }                                                         \n"
"                  else if (j == ix2)                                          \n"
"                    {                                                         \n"
"                      /* We've just got the left edge of this pixel */        \n"
"                      factorx = fx2;                                          \n"
"                    }                                                         \n"
"                  else                                                        \n"
"                    {                                                         \n"
"                      /* We've got the full width of this pixel */            \n"
"                      factorx = 1.0f;                                         \n"
"                    }                                                         \n"
"                                                                              \n"
"                  pixVal += data[j + i*inCols] * factorx * factory;           \n"
"                }                                                             \n"
"            }                                                                 \n"
"                                                                              \n"
"          /* Normalize by the area of the new pixel */                        \n"
"          res[x + y * outCols] = pixVal * normalize;                          \n"
"        }                                                                     \n"
"    }                                                                         \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"/* return = a - b */                                                          \n"
"static inline void                                                            \n"
"mantiuk06_matrix_subtract (const guint         n,                             \n"
"                           const gfloat *const a,                             \n"
"                           gfloat       *const b)                             \n"
"{                                                                             \n"
"  guint i;                                                                    \n"
"                                                                              \n"
"  _OMP (omp parallel for schedule(static))                                    \n"
"  for (i = 0; i < n; i++)                                                     \n"
"    b[i] = a[i] - b[i];                                                       \n"
"}                                                                             \n"
"                                                                              \n"
"/* copy matix a to b, return = a  */                                          \n"
"static inline void                                                            \n"
"mantiuk06_matrix_copy (const guint         n,                                 \n"
"                       const gfloat *const a,                                 \n"
"                       gfloat       *const b)                                 \n"
"{                                                                             \n"
"  memcpy (b, a, sizeof (gfloat) * n);                                         \n"
"}                                                                             \n"
"                                                                              \n"
"/* multiply matrix a by scalar val */                                         \n"
"static inline void                                                            \n"
"mantiuk06_matrix_multiply_const (const guint         n,                       \n"
"                                 gfloat       *const a,                       \n"
"                                 const gfloat        val)                     \n"
"{                                                                             \n"
"  guint i;                                                                    \n"
"  _OMP (omp parallel for schedule(static))                                    \n"
"  for (i = 0; i < n; i++)                                                     \n"
"    a[i] *= val;                                                              \n"
"}                                                                             \n"
"                                                                              \n"
"/* b = a[i] / b[i] */                                                         \n"
"static inline void                                                            \n"
"mantiuk06_matrix_divide (const guint         n,                               \n"
"                         const gfloat *const a,                               \n"
"                         gfloat       *const b)                               \n"
"{                                                                             \n"
"  guint i;                                                                    \n"
"  _OMP (omp parallel for schedule(static))                                    \n"
"  for (i = 0; i < n; i++)                                                     \n"
"      b[i] = a[i] / b[i];                                                     \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"static inline gfloat *                                                        \n"
"mantiuk06_matrix_alloc (guint size)                                           \n"
"{                                                                             \n"
"  return g_new (gfloat, size);                                                \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"static inline void                                                            \n"
"mantiuk06_matrix_free (gfloat *m)                                             \n"
"{                                                                             \n"
"  g_return_if_fail (m);                                                       \n"
"  g_free (m);                                                                 \n"
"}                                                                             \n"
"                                                                              \n"
"/* multiply vector by vector (each vector should have one dimension equal to 1) */\n"
"static inline gfloat                                                          \n"
"mantiuk06_matrix_dot_product (const guint         n,                          \n"
"                              const gfloat *const a,                          \n"
"                              const gfloat *const b)                          \n"
"{                                                                             \n"
"  gfloat val = 0;                                                             \n"
"  guint j;                                                                    \n"
"                                                                              \n"
"  _OMP (omp parallel for reduction(+:val) schedule(static))                   \n"
"  for (j = 0; j < n; j++)                                                     \n"
"    val += a[j] * b[j];                                                       \n"
"                                                                              \n"
"  return val;                                                                 \n"
"}                                                                             \n"
"                                                                              \n"
"/* set zeros for matrix elements */                                           \n"
"static inline void                                                            \n"
"mantiuk06_matrix_zero (guint   n,                                             \n"
"                       gfloat *m)                                             \n"
"{                                                                             \n"
"  memset(m, 0, n * sizeof (gfloat));                                          \n"
"}                                                                             \n"
"                                                                              \n"
"/* calculate divergence of two gradient maps (Gx and Gy)                      \n"
" * divG(x,y) = Gx(x,y) - Gx(x-1,y) + Gy(x,y) - Gy(x,y-1)                      \n"
" */                                                                           \n"
"static inline void                                                            \n"
"mantiuk06_calculate_and_add_divergence (const gint          cols,             \n"
"                                        const gint          rows,             \n"
"                                        const gfloat *const Gx,               \n"
"                                        const gfloat *const Gy,               \n"
"                                        gfloat       *const divG)             \n"
"{                                                                             \n"
"  gint ky, kx;                                                                \n"
"                                                                              \n"
"  _OMP (omp parallel for schedule(static))                                    \n"
"  for (ky = 0; ky < rows; ky++)                                               \n"
"    {                                                                         \n"
"      for (kx = 0; kx<cols; kx++)                                             \n"
"        {                                                                     \n"
"          gfloat divGx, divGy;                                                \n"
"          const int idx = kx + ky*cols;                                       \n"
"                                                                              \n"
"          if (kx == 0)                                                        \n"
"            divGx = Gx[idx];                                                  \n"
"          else                                                                \n"
"            divGx = Gx[idx] - Gx[idx-1];                                      \n"
"                                                                              \n"
"          if (ky == 0)                                                        \n"
"            divGy = Gy[idx];                                                  \n"
"          else                                                                \n"
"            divGy = Gy[idx] - Gy[idx - cols];                                 \n"
"                                                                              \n"
"          divG[idx] += divGx + divGy;                                         \n"
"        }                                                                     \n"
"    }                                                                         \n"
"}                                                                             \n"
"                                                                              \n"
"/* calculate the sum of divergences for the all pyramid level. the smaller    \n"
" * divergence map is upsamled and added to the divergence map for the higher  \n"
" * level of pyramid.                                                          \n"
" * temp is a temporary matrix of size (cols, rows), assumed to already be     \n"
" * allocated                                                                  \n"
" */                                                                           \n"
"static void                                                                   \n"
"mantiuk06_pyramid_calculate_divergence_sum (pyramid_t *pyramid,               \n"
"                                            gfloat    *divG_sum)              \n"
"{                                                                             \n"
"  gfloat *temp = mantiuk06_matrix_alloc (pyramid->rows*pyramid->cols);        \n"
"                                                                              \n"
"  /* Find the coarsest pyramid, and the number of pyramid levels */           \n"
"  int levels = 1;                                                             \n"
"  while (pyramid->next != NULL)                                               \n"
"    {                                                                         \n"
"      levels++;                                                               \n"
"      pyramid = pyramid->next;                                                \n"
"    }                                                                         \n"
"                                                                              \n"
"  /* For every level, we swap temp and divG_sum.  So, if there are an odd     \n"
"   * number of levels...                                                      \n"
"   */                                                                         \n"
"  if (levels % 2)                                                             \n"
"    {                                                                         \n"
"      gfloat *const dummy = divG_sum;                                         \n"
"      divG_sum = temp;                                                        \n"
"      temp = dummy;                                                           \n"
"    }                                                                         \n"
"                                                                              \n"
"  /* Add them all together */                                                 \n"
"  while (pyramid != NULL)                                                     \n"
"    {                                                                         \n"
"      gfloat *dummy;                                                          \n"
"                                                                              \n"
"      /* Upsample or zero as needed */                                        \n"
"      if (pyramid->next != NULL)                                              \n"
"        {                                                                     \n"
"          mantiuk06_matrix_upsample (pyramid->cols,                           \n"
"                                     pyramid->rows,                           \n"
"                                     divG_sum,                                \n"
"                                     temp);                                   \n"
"        }                                                                     \n"
"      else                                                                    \n"
"        {                                                                     \n"
"          mantiuk06_matrix_zero (pyramid->rows * pyramid->cols, temp);        \n"
"        }                                                                     \n"
"                                                                              \n"
"      /* Add in the (freshly calculated) divergences */                       \n"
"      mantiuk06_calculate_and_add_divergence (pyramid->cols,                  \n"
"                                              pyramid->rows,                  \n"
"                                              pyramid->Gx,                    \n"
"                                              pyramid->Gy,                    \n"
"                                              temp);                          \n"
"                                                                              \n"
"      /* Rather than copying, just switch round the pointers: we know we get  \n"
"       * them the right way round at the end.                                 \n"
"       */                                                                     \n"
"      dummy    = divG_sum;                                                    \n"
"      divG_sum = temp;                                                        \n"
"      temp     = dummy;                                                       \n"
"                                                                              \n"
"      pyramid  = pyramid->prev;                                               \n"
"    }                                                                         \n"
"                                                                              \n"
"  mantiuk06_matrix_free (temp);                                               \n"
"}                                                                             \n"
"                                                                              \n"
"/* calculate scale factors (Cx,Cy) for gradients (Gx,Gy)                      \n"
" * C is equal to EDGE_WEIGHT for gradients smaller than GFIXATE or            \n"
" * 1.0 otherwise                                                              \n"
" */                                                                           \n"
"static inline void                                                            \n"
"mantiuk06_calculate_scale_factor (const gint          n,                      \n"
"                                  const gfloat *const G,                      \n"
"                                  gfloat       *const C)                      \n"
"{                                                                             \n"
"  const gfloat detectT = 0.001f;                                              \n"
"  const gfloat a = 0.038737;                                                  \n"
"  const gfloat b = 0.537756;                                                  \n"
"                                                                              \n"
"  gint i;                                                                     \n"
"                                                                              \n"
"  _OMP (omp parallel for schedule(static))                                    \n"
"  for (i = 0; i < n; i++)                                                     \n"
"    {                                                                         \n"
"#if 1                                                                         \n"
"      const gfloat g = MAX (detectT, fabsf (G[i]));                           \n"
"      C[i] = 1.0f / (a * powf (g, b));                                        \n"
"#else                                                                         \n"
"      const gfloat GFIXATE     = 0.10f,                                       \n"
"                  EDGE_WEIGHT = 0.01f;                                        \n"
"                                                                              \n"
"      if (fabsf (G[i]) < GFIXATE)                                             \n"
"        C[i] = 1.0f / EDGE_WEIGHT;                                            \n"
"      else                                                                    \n"
"        C[i] = 1.0f;                                                          \n"
"#endif                                                                        \n"
"    }                                                                         \n"
"}                                                                             \n"
"                                                                              \n"
"/* calculate scale factor for the whole pyramid */                            \n"
"static void                                                                   \n"
"mantiuk06_pyramid_calculate_scale_factor (pyramid_t *pyramid,                 \n"
"                                          pyramid_t *pC)                      \n"
"{                                                                             \n"
"  while (pyramid != NULL)                                                     \n"
"    {                                                                         \n"
"      const int size = pyramid->rows * pyramid->cols;                         \n"
"      mantiuk06_calculate_scale_factor (size, pyramid->Gx, pC->Gx);           \n"
"      mantiuk06_calculate_scale_factor (size, pyramid->Gy, pC->Gy);           \n"
"      pyramid = pyramid->next;                                                \n"
"      pC = pC->next;                                                          \n"
"    }                                                                         \n"
"}                                                                             \n"
"                                                                              \n"
"/* Scale gradient (Gx and Gy) by C (Cx and Cy)                                \n"
" * G = G / C                                                                  \n"
" */                                                                           \n"
"static inline void                                                            \n"
"mantiuk06_scale_gradient (const gint          n,                              \n"
"                          gfloat       *const G,                              \n"
"                          const gfloat *const C)                              \n"
"{                                                                             \n"
"  gint i;                                                                     \n"
"  _OMP (omp parallel for schedule(static))                                    \n"
"  for (i = 0; i < n; i++)                                                     \n"
"    G[i] *= C[i];                                                             \n"
"}                                                                             \n"
"                                                                              \n"
"/* scale gradients for the whole one pyramid with the use of (Cx,Cy) from the \n"
" * other pyramid                                                              \n"
" */                                                                           \n"
"static void                                                                   \n"
"mantiuk06_pyramid_scale_gradient (pyramid_t *pyramid,                         \n"
"                                  pyramid_t *pC)                              \n"
"{                                                                             \n"
"  while (pyramid != NULL)                                                     \n"
"    {                                                                         \n"
"      const int size = pyramid->rows * pyramid->cols;                         \n"
"      mantiuk06_scale_gradient (size, pyramid->Gx, pC->Gx);                   \n"
"      mantiuk06_scale_gradient (size, pyramid->Gy, pC->Gy);                   \n"
"      pyramid = pyramid->next;                                                \n"
"      pC = pC->next;                                                          \n"
"    }                                                                         \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"/* free memory allocated for the pyramid */                                   \n"
"static void                                                                   \n"
"mantiuk06_pyramid_free (pyramid_t *pyramid)                                   \n"
"{                                                                             \n"
"  while (pyramid)                                                             \n"
"    {                                                                         \n"
"      pyramid_t *const next = pyramid->next;                                  \n"
"                                                                              \n"
"      if (pyramid->Gx != NULL)                                                \n"
"        {                                                                     \n"
"          g_free (pyramid->Gx);                                               \n"
"          pyramid->Gx = NULL;                                                 \n"
"        }                                                                     \n"
"                                                                              \n"
"      if (pyramid->Gy != NULL)                                                \n"
"        {                                                                     \n"
"          g_free (pyramid->Gy);                                               \n"
"          pyramid->Gy = NULL;                                                 \n"
"        }                                                                     \n"
"                                                                              \n"
"      pyramid->prev = NULL;                                                   \n"
"      pyramid->next = NULL;                                                   \n"
"      g_free (pyramid);                                                       \n"
"      pyramid = next;                                                         \n"
"    }                                                                         \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"/* allocate memory for the pyramid */                                         \n"
"static pyramid_t*                                                             \n"
"mantiuk06_pyramid_allocate (int cols,                                         \n"
"                            int rows)                                         \n"
"{                                                                             \n"
"  pyramid_t *level = NULL;                                                    \n"
"  pyramid_t *pyramid = NULL;                                                  \n"
"  pyramid_t *prev = NULL;                                                     \n"
"                                                                              \n"
"  while (rows >= PYRAMID_MIN_PIXELS && cols >= PYRAMID_MIN_PIXELS)            \n"
"    {                                                                         \n"
"      guint size;                                                             \n"
"      level = g_new (pyramid_t, 1);                                           \n"
"      memset (level, 0, sizeof (pyramid_t));                                  \n"
"                                                                              \n"
"      level->rows = rows;                                                     \n"
"      level->cols = cols;                                                     \n"
"      size        = level->rows * level->cols;                                \n"
"      level->Gx   = mantiuk06_matrix_alloc (size);                            \n"
"      level->Gy   = mantiuk06_matrix_alloc (size);                            \n"
"                                                                              \n"
"      level->prev = prev;                                                     \n"
"      if (prev != NULL)                                                       \n"
"        prev->next = level;                                                   \n"
"      prev = level;                                                           \n"
"                                                                              \n"
"      if (pyramid == NULL)                                                    \n"
"        pyramid = level;                                                      \n"
"                                                                              \n"
"      rows /= 2;                                                              \n"
"      cols /= 2;                                                              \n"
"  }                                                                           \n"
"                                                                              \n"
"  return pyramid;                                                             \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"/* calculate gradients */                                                     \n"
"static inline void                                                            \n"
"mantiuk06_calculate_gradient (const gint          cols,                       \n"
"                              const gint          rows,                       \n"
"                              const gfloat *const lum,                        \n"
"                              gfloat       *const Gx,                         \n"
"                              gfloat       *const Gy)                         \n"
"{                                                                             \n"
"  gint ky, kx;                                                                \n"
"                                                                              \n"
"  _OMP (omp parallel for schedule(static))                                    \n"
"  for (ky = 0; ky < rows; ky++)                                               \n"
"    {                                                                         \n"
"      for (kx = 0; kx < cols; kx++)                                           \n"
"        {                                                                     \n"
"          const gint idx = kx + ky*cols;                                      \n"
"                                                                              \n"
"          if (kx == (cols - 1))                                               \n"
"            Gx[idx] = 0;                                                      \n"
"          else                                                                \n"
"            Gx[idx] = lum[idx+1] - lum[idx];                                  \n"
"                                                                              \n"
"          if (ky == (rows - 1))                                               \n"
"            Gy[idx] = 0;                                                      \n"
"          else                                                                \n"
"            Gy[idx] = lum[idx + cols] - lum[idx];                             \n"
"        }                                                                     \n"
"    }                                                                         \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"/* calculate gradients for the pyramid                                        \n"
" * lum_temp gets overwritten!                                                 \n"
" */                                                                           \n"
"static void                                                                   \n"
"mantiuk06_pyramid_calculate_gradient (pyramid_t *pyramid,                     \n"
"                                      gfloat    *lum_temp)                    \n"
"{                                                                             \n"
"  gfloat *temp = mantiuk06_matrix_alloc ((pyramid->rows / 2) *                \n"
"                                         (pyramid->cols / 2));                \n"
"  gfloat *const temp_saved = temp;                                            \n"
"                                                                              \n"
"  mantiuk06_calculate_gradient (pyramid->cols,                                \n"
"                                pyramid->rows,                                \n"
"                                lum_temp,                                     \n"
"                                pyramid->Gx,                                  \n"
"                                pyramid->Gy);                                 \n"
"                                                                              \n"
"  pyramid = pyramid->next;                                                    \n"
"                                                                              \n"
"  while (pyramid)                                                             \n"
"    {                                                                         \n"
"      gfloat *dummy;                                                          \n"
"      mantiuk06_matrix_downsample  (pyramid->prev->cols,                      \n"
"                                    pyramid->prev->rows,                      \n"
"                                    lum_temp,                                 \n"
"                                    temp);                                    \n"
"      mantiuk06_calculate_gradient (pyramid->cols,                            \n"
"                                    pyramid->rows,                            \n"
"                                    temp,                                     \n"
"                                    pyramid->Gx,                              \n"
"                                    pyramid->Gy);                             \n"
"                                                                              \n"
"      dummy    = lum_temp;                                                    \n"
"      lum_temp = temp;                                                        \n"
"      temp     = dummy;                                                       \n"
"                                                                              \n"
"      pyramid = pyramid->next;                                                \n"
"  }                                                                           \n"
"                                                                              \n"
"  mantiuk06_matrix_free (temp_saved);                                         \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"/* x = -0.25 * b */                                                           \n"
"static inline void                                                            \n"
"mantiuk06_solveX (const gint          n,                                      \n"
"                  const gfloat *const b,                                      \n"
"                  gfloat       *const x)                                      \n"
"{                                                                             \n"
"  gint i;                                                                     \n"
"                                                                              \n"
"  _OMP (omp parallel for schedule(static))                                    \n"
"  for (i = 0; i < n; i++)                                                     \n"
"    x[i] = -0.25f * b[i];                                                     \n"
"}                                                                             \n"
"                                                                              \n"
"/* divG_sum = A * x = sum (divG (x))                                          \n"
" * memory for the temporary pyramid px should be allocated                    \n"
" */                                                                           \n"
"static inline void                                                            \n"
"mantiuk06_multiplyA (pyramid_t           *px,                                 \n"
"                     pyramid_t           *pC,                                 \n"
"                     const gfloat *const  x,                                  \n"
"                     gfloat       *const  divG_sum)                           \n"
"{                                                                             \n"
"  /* use divG_sum as a temp variable */                                       \n"
"  mantiuk06_matrix_copy (pC->rows*pC->cols, x, divG_sum);                     \n"
"  mantiuk06_pyramid_calculate_gradient (px, divG_sum);                        \n"
"  /* scale gradients by Cx,Cy from main pyramid */                            \n"
"  mantiuk06_pyramid_scale_gradient (px, pC);                                  \n"
"  /* calculate the sum of divergences */                                      \n"
"  mantiuk06_pyramid_calculate_divergence_sum (px, divG_sum);                  \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"/* bi-conjugate linear equation solver                                        \n"
" * overwrites pyramid!                                                        \n"
" */                                                                           \n"
"static void                                                                   \n"
"mantiuk06_linbcg (pyramid_t           *pyramid,                               \n"
"                  pyramid_t           *pC,                                    \n"
"                  const gfloat *const  b,                                     \n"
"                  gfloat       *const  x,                                     \n"
"                  const gint           itmax,                                 \n"
"                  const gfloat         tol,                                   \n"
"                  pfstmo_progress_callback progress_cb)                       \n"
"{                                                                             \n"
"  const guint  rows = pyramid->rows,                                          \n"
"               cols = pyramid->cols,                                          \n"
"               n    = rows*cols;                                              \n"
"  const gfloat tol2 = tol*tol;                                                \n"
"                                                                              \n"
"  gfloat *const z      = mantiuk06_matrix_alloc (n),                          \n"
"         *const zz     = mantiuk06_matrix_alloc (n),                          \n"
"         *const p      = mantiuk06_matrix_alloc (n),                          \n"
"         *const pp     = mantiuk06_matrix_alloc (n),                          \n"
"         *const r      = mantiuk06_matrix_alloc (n),                          \n"
"         *const rr     = mantiuk06_matrix_alloc (n),                          \n"
"         *const x_save = mantiuk06_matrix_alloc (n);                          \n"
"                                                                              \n"
"  const gfloat bnrm2 = mantiuk06_matrix_dot_product (n, b, b);                \n"
"                                                                              \n"
"  gfloat      err2, bkden, saved_err2, ierr2, percent_sf;                     \n"
"  gint        iter  = 0, num_backwards = 0, num_backwards_ceiling = 3;        \n"
"  gboolean    reset = TRUE;                                                   \n"
"                                                                              \n"
"  mantiuk06_multiplyA (pyramid, pC, x, r); /* r = A*x = divergence (x) */     \n"
"  mantiuk06_matrix_subtract (n, b, r);     /* r = b - r               */      \n"
"  err2 = mantiuk06_matrix_dot_product (n, r, r); /* err2 = r.r */             \n"
"                                                                              \n"
"  mantiuk06_multiplyA (pyramid, pC, r, rr); /* rr = A*r */                    \n"
"                                                                              \n"
"  bkden = 0;                                                                  \n"
"  saved_err2 = err2;                                                          \n"
"  mantiuk06_matrix_copy (n, x, x_save);                                       \n"
"                                                                              \n"
"  ierr2 = err2;                                                               \n"
"  percent_sf = 100.0f /logf (tol2 * bnrm2 / ierr2);                           \n"
"                                                                              \n"
"  for (; iter < itmax; iter++)                                                \n"
"    {                                                                         \n"
"      guint i;                                                                \n"
"      gfloat bknum, ak, old_err2;                                             \n"
"                                                                              \n"
"      if (progress_cb != NULL)                                                \n"
"        progress_cb ((int) (logf (err2 / ierr2) * percent_sf));               \n"
"                                                                              \n"
"      mantiuk06_solveX (n,  r,  z); /*  z = ~A (-1) *  r = -0.25 *  r */      \n"
"      mantiuk06_solveX (n, rr, zz); /* zz = ~A (-1) * rr = -0.25 * rr */      \n"
"                                                                              \n"
"      bknum = mantiuk06_matrix_dot_product (n, z, rr);                        \n"
"                                                                              \n"
"      if (reset)                                                              \n"
"        {                                                                     \n"
"          reset = FALSE;                                                      \n"
"          mantiuk06_matrix_copy (n, z, p);                                    \n"
"          mantiuk06_matrix_copy (n, zz, pp);                                  \n"
"        }                                                                     \n"
"      else                                                                    \n"
"        {                                                                     \n"
"          const gfloat bk = bknum / bkden; /* beta = ...  */                  \n"
"                                                                              \n"
"          _OMP (omp parallel for schedule(static))                            \n"
"          for (i = 0; i < n; i++)                                             \n"
"            {                                                                 \n"
"              p[i]  =  z[i] + bk *  p[i];                                     \n"
"              pp[i] = zz[i] + bk * pp[i];                                     \n"
"            }                                                                 \n"
"        }                                                                     \n"
"                                                                              \n"
"      bkden = bknum; /* numerator becomes the dominator for the next iteration */\n"
"                                                                              \n"
"      mantiuk06_multiplyA (pyramid, pC,  p,  z); /*  z = A* p = divergence (p) */\n"
"      mantiuk06_multiplyA (pyramid, pC, pp, zz); /* zz = A*pp = divergence (pp) */\n"
"                                                                              \n"
"      ak = bknum / mantiuk06_matrix_dot_product (n, z, pp); /* alfa = ...   */\n"
"                                                                              \n"
"      _OMP (omp parallel for schedule(static))                                \n"
"      for (i = 0 ; i < n ; i++ )                                              \n"
"        {                                                                     \n"
"          r[i]  -= ak *  z[i];  /*  r =  r - alfa *  z  */                    \n"
"          rr[i] -= ak * zz[i];  /* rr = rr - alfa * zz  */                    \n"
"        }                                                                     \n"
"                                                                              \n"
"      old_err2 = err2;                                                        \n"
"      err2 = mantiuk06_matrix_dot_product (n, r, r);                          \n"
"                                                                              \n"
"      /* Have we gone unstable? */                                            \n"
"      if (err2 > old_err2)                                                    \n"
"        {                                                                     \n"
"          /* Save where we've got to if it's the best yet */                  \n"
"          if (num_backwards == 0 && old_err2 < saved_err2)                    \n"
"            {                                                                 \n"
"              saved_err2 = old_err2;                                          \n"
"              mantiuk06_matrix_copy (n, x, x_save);                           \n"
"            }                                                                 \n"
"                                                                              \n"
"          num_backwards++;                                                    \n"
"        }                                                                     \n"
"      else                                                                    \n"
"        {                                                                     \n"
"          num_backwards = 0;                                                  \n"
"        }                                                                     \n"
"                                                                              \n"
"      _OMP (omp parallel for schedule(static))                                \n"
"      for (i = 0 ; i < n ; i++ )                                              \n"
"        x[i] += ak * p[i];      /* x =  x + alfa * p */                       \n"
"                                                                              \n"
"      if (num_backwards > num_backwards_ceiling)                              \n"
"        {                                                                     \n"
"          /* Reset  */                                                        \n"
"          reset = TRUE;                                                       \n"
"          num_backwards = 0;                                                  \n"
"                                                                              \n"
"          /* Recover saved value */                                           \n"
"          mantiuk06_matrix_copy (n, x_save, x);                               \n"
"                                                                              \n"
"          /* r = Ax */                                                        \n"
"          mantiuk06_multiplyA (pyramid, pC, x, r);                            \n"
"                                                                              \n"
"          /* r = b - r */                                                     \n"
"          mantiuk06_matrix_subtract (n, b, r);                                \n"
"                                                                              \n"
"          /* err2 = r.r  */                                                   \n"
"          err2 = mantiuk06_matrix_dot_product (n, r, r);                      \n"
"          saved_err2 = err2;                                                  \n"
"                                                                              \n"
"          /* rr = A*r  */                                                     \n"
"          mantiuk06_multiplyA (pyramid, pC, r, rr);                           \n"
"        }                                                                     \n"
"                                                                              \n"
"      if (err2 / bnrm2 < tol2)                                                \n"
"        break;                                                                \n"
"    }                                                                         \n"
"                                                                              \n"
"  /* Use the best version we found */                                         \n"
"  if (err2 > saved_err2)                                                      \n"
"    {                                                                         \n"
"      err2 = saved_err2;                                                      \n"
"      mantiuk06_matrix_copy (n, x_save, x);                                   \n"
"    }                                                                         \n"
"                                                                              \n"
"  if (err2/bnrm2 > tol2)                                                      \n"
"    {                                                                         \n"
"      /* Not converged */                                                     \n"
"      if (progress_cb != NULL)                                                \n"
"        progress_cb ((int) (logf (err2 / ierr2) * percent_sf));               \n"
"      if (iter == itmax)                                                      \n"
"        g_warning (\"mantiuk06: Warning: \"                                   \n"
"                   \"Not converged (hit maximum iterations), \"               \n"
"                   \"error = %g (should be below %g).\",                      \n"
"                   sqrtf (err2 / bnrm2), tol);                                \n"
"      else                                                                    \n"
"        g_warning (\"mantiuk06: Warning: \"                                   \n"
"                   \"Not converged (going unstable), \"                       \n"
"                   \"error = %g (should be below %g).\",                      \n"
"                   sqrtf (err2 / bnrm2), tol);                                \n"
"    }                                                                         \n"
"  else if (progress_cb != NULL)                                               \n"
"    progress_cb (100);                                                        \n"
"                                                                              \n"
"  mantiuk06_matrix_free (x_save);                                             \n"
"  mantiuk06_matrix_free (p);                                                  \n"
"  mantiuk06_matrix_free (pp);                                                 \n"
"  mantiuk06_matrix_free (z);                                                  \n"
"  mantiuk06_matrix_free (zz);                                                 \n"
"  mantiuk06_matrix_free (r);                                                  \n"
"  mantiuk06_matrix_free (rr);                                                 \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"/* conjugate linear equation solver                                           \n"
" * overwrites pyramid!                                                        \n"
" */                                                                           \n"
"static void                                                                   \n"
"mantiuk06_lincg (pyramid_t           *pyramid,                                \n"
"                 pyramid_t           *pC,                                     \n"
"                 const gfloat *const  b,                                      \n"
"                 gfloat       *const  x,                                      \n"
"                 const int            itmax,                                  \n"
"                 const gfloat         tol,                                    \n"
"                 pfstmo_progress_callback progress_cb)                        \n"
"{                                                                             \n"
"  const int rows = pyramid->rows,                                             \n"
"            cols = pyramid->cols,                                             \n"
"            n    = rows*cols;                                                 \n"
"  int       iter = 0, num_backwards = 0, num_backwards_ceiling = 3;           \n"
"  const gfloat tol2 = tol*tol;                                                \n"
"                                                                              \n"
"  gfloat *const x_save = mantiuk06_matrix_alloc (n),                          \n"
"         *const r      = mantiuk06_matrix_alloc (n),                          \n"
"         *const p      = mantiuk06_matrix_alloc (n),                          \n"
"         *const Ap     = mantiuk06_matrix_alloc (n);                          \n"
"                                                                              \n"
"  /* bnrm2 = ||b|| */                                                         \n"
"  const gfloat bnrm2 = mantiuk06_matrix_dot_product (n, b, b);                \n"
"  gfloat       rdotr, irdotr, saved_rdotr, percent_sf;                        \n"
"                                                                              \n"
"  /* r = b - Ax */                                                            \n"
"  mantiuk06_multiplyA (pyramid, pC, x, r);                                    \n"
"  mantiuk06_matrix_subtract (n, b, r);                                        \n"
"  rdotr = mantiuk06_matrix_dot_product (n, r, r); /* rdotr = r.r */           \n"
"                                                                              \n"
"  /* p = r */                                                                 \n"
"  mantiuk06_matrix_copy (n, r, p);                                            \n"
"                                                                              \n"
"  /* Setup initial vector */                                                  \n"
"  saved_rdotr = rdotr;                                                        \n"
"  mantiuk06_matrix_copy (n, x, x_save);                                       \n"
"                                                                              \n"
"  irdotr = rdotr;                                                             \n"
"  percent_sf = 100.0f / logf (tol2 * bnrm2 / irdotr);                         \n"
"  for (; iter < itmax; iter++)                                                \n"
"    {                                                                         \n"
"      gint  i;                                                                \n"
"      gfloat alpha, old_rdotr;                                                \n"
"                                                                              \n"
"      if (progress_cb != NULL) {                                              \n"
"        gint ret = progress_cb ((gint) (logf (rdotr / irdotr) * percent_sf)); \n"
"        if (ret == PFSTMO_CB_ABORT && iter > 0 ) /* User requested abort */   \n"
"          break;                                                              \n"
"      }                                                                       \n"
"                                                                              \n"
"      /* Ap = A p */                                                          \n"
"      mantiuk06_multiplyA (pyramid, pC, p, Ap);                               \n"
"                                                                              \n"
"      /* alpha = r.r / (p . Ap) */                                            \n"
"      alpha = rdotr / mantiuk06_matrix_dot_product (n, p, Ap);                \n"
"                                                                              \n"
"      /* r = r - alpha Ap */                                                  \n"
"      _OMP (omp parallel for schedule(static))                                \n"
"      for (i = 0; i < n; i++)                                                 \n"
"        r[i] -= alpha * Ap[i];                                                \n"
"                                                                              \n"
"      /* rdotr = r.r */                                                       \n"
"      old_rdotr = rdotr;                                                      \n"
"      rdotr = mantiuk06_matrix_dot_product (n, r, r);                         \n"
"                                                                              \n"
"      /* Have we gone unstable? */                                            \n"
"      if (rdotr > old_rdotr)                                                  \n"
"        {                                                                     \n"
"          /* Save where we've got to */                                       \n"
"          if (num_backwards == 0 && old_rdotr < saved_rdotr)                  \n"
"            {                                                                 \n"
"              saved_rdotr = old_rdotr;                                        \n"
"              mantiuk06_matrix_copy (n, x, x_save);                           \n"
"            }                                                                 \n"
"                                                                              \n"
"          num_backwards++;                                                    \n"
"        }                                                                     \n"
"      else                                                                    \n"
"        {                                                                     \n"
"          num_backwards = 0;                                                  \n"
"        }                                                                     \n"
"                                                                              \n"
"      /* x = x + alpha p */                                                   \n"
"      _OMP (omp parallel for schedule(static))                                \n"
"      for (i = 0; i < n; i++)                                                 \n"
"        x[i] += alpha * p[i];                                                 \n"
"                                                                              \n"
"                                                                              \n"
"      /* Exit if we're done */                                                \n"
"      if (rdotr/bnrm2 < tol2)                                                 \n"
"        break;                                                                \n"
"                                                                              \n"
"      if (num_backwards > num_backwards_ceiling)                              \n"
"        {                                                                     \n"
"          /* Reset */                                                         \n"
"          num_backwards = 0;                                                  \n"
"          mantiuk06_matrix_copy (n, x_save, x);                               \n"
"                                                                              \n"
"          /* r = Ax */                                                        \n"
"          mantiuk06_multiplyA (pyramid, pC, x, r);                            \n"
"                                                                              \n"
"          /* r = b - r */                                                     \n"
"          mantiuk06_matrix_subtract (n, b, r);                                \n"
"                                                                              \n"
"          /* rdotr = r.r */                                                   \n"
"          rdotr = mantiuk06_matrix_dot_product (n, r, r);                     \n"
"          saved_rdotr = rdotr;                                                \n"
"                                                                              \n"
"          /* p = r */                                                         \n"
"          mantiuk06_matrix_copy (n, r, p);                                    \n"
"        }                                                                     \n"
"      else                                                                    \n"
"        {                                                                     \n"
"          /* p = r + beta p */                                                \n"
"          const gfloat beta = rdotr/old_rdotr;                                \n"
"                                                                              \n"
"          _OMP (omp parallel for schedule(static))                            \n"
"          for (i = 0; i < n; i++)                                             \n"
"            p[i] = r[i] + beta*p[i];                                          \n"
"        }                                                                     \n"
"    }                                                                         \n"
"                                                                              \n"
"  /* Use the best version we found */                                         \n"
"  if (rdotr > saved_rdotr)                                                    \n"
"    {                                                                         \n"
"      rdotr = saved_rdotr;                                                    \n"
"      mantiuk06_matrix_copy (n, x_save, x);                                   \n"
"    }                                                                         \n"
"                                                                              \n"
"  if (rdotr/bnrm2 > tol2)                                                     \n"
"    {                                                                         \n"
"      /* Not converged */                                                     \n"
"      if (progress_cb != NULL)                                                \n"
"        progress_cb ((int) (logf (rdotr / irdotr) * percent_sf));             \n"
"      if (iter == itmax)                                                      \n"
"        g_warning (\"mantiuk06: Warning: \"                                   \n"
"                   \"Not converged (hit maximum iterations), \"               \n"
"                   \"error = %g (should be below %g).\",                      \n"
"                   sqrtf (rdotr/bnrm2), tol);                                 \n"
"      else                                                                    \n"
"        g_warning (\"mantiuk06: Warning: \"                                   \n"
"                   \"Not converged (going unstable), \"                       \n"
"                   \"error = %g (should be below %g).\",                      \n"
"                   sqrtf (rdotr/bnrm2), tol);                                 \n"
"    }                                                                         \n"
"  else if (progress_cb != NULL)                                               \n"
"    progress_cb (100);                                                        \n"
"                                                                              \n"
"  mantiuk06_matrix_free (x_save);                                             \n"
"  mantiuk06_matrix_free (p);                                                  \n"
"  mantiuk06_matrix_free (Ap);                                                 \n"
"  mantiuk06_matrix_free (r);                                                  \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"/* in_tab and out_tab should contain inccreasing float values */              \n"
"static inline gfloat                                                          \n"
"mantiuk06_lookup_table (const gint          n,                                \n"
"                        const gfloat *const in_tab,                           \n"
"                        const gfloat *const out_tab,                          \n"
"                        const gfloat        val)                              \n"
"{                                                                             \n"
"  gint j;                                                                     \n"
"                                                                              \n"
"  if (G_UNLIKELY (val < in_tab[0]))                                           \n"
"    return out_tab[0];                                                        \n"
"                                                                              \n"
"  for (j = 1; j < n; j++)                                                     \n"
"    if (val < in_tab[j])                                                      \n"
"      {                                                                       \n"
"        const gfloat dd = (val - in_tab[j-1]) / (in_tab[j] - in_tab[j-1]);    \n"
"        return out_tab[j-1] + (out_tab[j] - out_tab[j-1]) * dd;               \n"
"      }                                                                       \n"
"                                                                              \n"
"  return out_tab[n-1];                                                        \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"/* transform gradient (Gx,Gy) to R */                                         \n"
"static inline void                                                            \n"
"mantiuk06_transform_to_R (const gint        n,                                \n"
"                          gfloat     *const G)                                \n"
"{                                                                             \n"
"  gint j;                                                                     \n"
"                                                                              \n"
"  _OMP (omp parallel for schedule(static))                                    \n"
"  for (j = 0; j < n; j++)                                                     \n"
"    {                                                                         \n"
"      /* G to W */                                                            \n"
"      const gfloat absG = fabsf (G[j]);                                       \n"
"      int sign;                                                               \n"
"      if (G[j] < 0)                                                           \n"
"        sign = -1;                                                            \n"
"      else                                                                    \n"
"        sign = 1;                                                             \n"
"      G[j] = (powf (10, absG) - 1.0f) * sign;                                 \n"
"                                                                              \n"
"      /* W to RESP */                                                         \n"
"      if (G[j] < 0)                                                           \n"
"        sign = -1;                                                            \n"
"      else                                                                    \n"
"        sign = 1;                                                             \n"
"                                                                              \n"
"      G[j] = sign * mantiuk06_lookup_table (LOOKUP_W_TO_R,                    \n"
"                                            W_table,                          \n"
"                                            R_table,                          \n"
"                                            fabsf (G[j]));                    \n"
"    }                                                                         \n"
"}                                                                             \n"
"                                                                              \n"
"/* transform gradient (Gx,Gy) to R for the whole pyramid */                   \n"
"static inline void                                                            \n"
"mantiuk06_pyramid_transform_to_R (pyramid_t *pyramid)                         \n"
"{                                                                             \n"
"  while (pyramid != NULL)                                                     \n"
"    {                                                                         \n"
"      const int size = pyramid->rows * pyramid->cols;                         \n"
"      mantiuk06_transform_to_R (size, pyramid->Gx);                           \n"
"      mantiuk06_transform_to_R (size, pyramid->Gy);                           \n"
"      pyramid = pyramid->next;                                                \n"
"    }                                                                         \n"
"}                                                                             \n"
"                                                                              \n"
"/* transform from R to G */                                                   \n"
"static inline void                                                            \n"
"mantiuk06_transform_to_G (const gint        n,                                \n"
"                          gfloat     *const R)                                \n"
"{                                                                             \n"
"  gint j;                                                                     \n"
"                                                                              \n"
"  _OMP (omp parallel for schedule(static))                                    \n"
"  for (j = 0; j < n; j++){                                                    \n"
"    /* RESP to W */                                                           \n"
"    gint sign;                                                                \n"
"    if (R[j] < 0)                                                             \n"
"      sign = -1;                                                              \n"
"    else                                                                      \n"
"      sign = 1;                                                               \n"
"    R[j] = sign * mantiuk06_lookup_table (LOOKUP_W_TO_R,                      \n"
"                                          R_table,                            \n"
"                                          W_table,                            \n"
"                                          fabsf (R[j]));                      \n"
"                                                                              \n"
"    /* W to G */                                                              \n"
"    if (R[j] < 0)                                                             \n"
"      sign = -1;                                                              \n"
"    else                                                                      \n"
"      sign = 1;                                                               \n"
"    R[j] = log10f (fabsf (R[j]) + 1.0f) * sign;                               \n"
"                                                                              \n"
"  }                                                                           \n"
"}                                                                             \n"
"                                                                              \n"
"/* transform from R to G for the pyramid */                                   \n"
"static inline void                                                            \n"
"mantiuk06_pyramid_transform_to_G (pyramid_t *pyramid)                         \n"
"{                                                                             \n"
"  while (pyramid != NULL)                                                     \n"
"    {                                                                         \n"
"      mantiuk06_transform_to_G (pyramid->rows*pyramid->cols, pyramid->Gx);    \n"
"      mantiuk06_transform_to_G (pyramid->rows*pyramid->cols, pyramid->Gy);    \n"
"      pyramid = pyramid->next;                                                \n"
"    }                                                                         \n"
"}                                                                             \n"
"                                                                              \n"
"/* multiply gradient (Gx,Gy) values by float scalar value for the             \n"
" * whole pyramid                                                              \n"
" */                                                                           \n"
"static inline void                                                            \n"
"mantiuk06_pyramid_gradient_multiply (pyramid_t    *pyramid,                   \n"
"                                     const gfloat  val)                       \n"
"{                                                                             \n"
"  while (pyramid != NULL)                                                     \n"
"    {                                                                         \n"
"      mantiuk06_matrix_multiply_const (pyramid->rows * pyramid->cols,         \n"
"                                       pyramid->Gx, val);                     \n"
"      mantiuk06_matrix_multiply_const (pyramid->rows * pyramid->cols,         \n"
"                                       pyramid->Gy, val);                     \n"
"      pyramid = pyramid->next;                                                \n"
"    }                                                                         \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"static int                                                                    \n"
"mantiuk06_sort_float (const void *const v1,                                   \n"
"                      const void *const v2)                                   \n"
"{                                                                             \n"
"  if (*((gfloat*)v1) < *((gfloat*)v2))                                        \n"
"    return -1;                                                                \n"
"                                                                              \n"
"  if (G_LIKELY (*((gfloat*)v1) > *((gfloat*)v2)))                             \n"
"    return 1;                                                                 \n"
"                                                                              \n"
"  return 0;                                                                   \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"/* transform gradients to luminance */                                        \n"
"static void                                                                   \n"
"mantiuk06_transform_to_luminance (pyramid_t                        *pp,       \n"
"                                  gfloat                    *const  x,        \n"
"                                  pfstmo_progress_callback         progress,  \n"
"                                  const gboolean                   bcg,       \n"
"                                  const gint                       itmax,     \n"
"                                  const gfloat                     tol)       \n"
"{                                                                             \n"
"  gfloat     *b;                                                              \n"
"  pyramid_t  *pC = mantiuk06_pyramid_allocate (pp->cols, pp->rows);           \n"
"  /* calculate (Cx,Cy) */                                                     \n"
"  mantiuk06_pyramid_calculate_scale_factor (pp, pC);                          \n"
"  /* scale small gradients by (Cx,Cy); */                                     \n"
"  mantiuk06_pyramid_scale_gradient (pp, pC);                                  \n"
"                                                                              \n"
"  b = mantiuk06_matrix_alloc (pp->cols * pp->rows);                           \n"
"  /* calculate the sum of divergences (equal to b) */                         \n"
"  mantiuk06_pyramid_calculate_divergence_sum (pp, b);                         \n"
"                                                                              \n"
"  /* calculate luminances from gradients */                                   \n"
"  if (bcg)                                                                    \n"
"    mantiuk06_linbcg (pp, pC, b, x, itmax, tol, progress);                    \n"
"  else                                                                        \n"
"    mantiuk06_lincg (pp, pC, b, x, itmax, tol, progress);                     \n"
"                                                                              \n"
"  mantiuk06_matrix_free (b);                                                  \n"
"  mantiuk06_pyramid_free (pC);                                                \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"struct hist_data                                                              \n"
"{                                                                             \n"
"  gfloat size;                                                                \n"
"  gfloat cdf;                                                                 \n"
"  gint index;                                                                 \n"
"};                                                                            \n"
"                                                                              \n"
"static int                                                                    \n"
"mantiuk06_hist_data_order (const void *const v1,                              \n"
"                           const void *const v2)                              \n"
"{                                                                             \n"
"  if (((struct hist_data*) v1)->size < ((struct hist_data*) v2)->size)        \n"
"    return -1;                                                                \n"
"                                                                              \n"
"  if (((struct hist_data*) v1)->size > ((struct hist_data*) v2)->size)        \n"
"    return 1;                                                                 \n"
"                                                                              \n"
"  return 0;                                                                   \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"static int                                                                    \n"
"mantiuk06_hist_data_index (const void *const v1,                              \n"
"                           const void *const v2)                              \n"
"{                                                                             \n"
"  return ((struct hist_data*) v1)->index - ((struct hist_data*) v2)->index;   \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"static void                                                                   \n"
"mantiuk06_contrast_equalization (pyramid_t   *pp,                             \n"
"                                 const gfloat  contrastFactor )               \n"
"{                                                                             \n"
"  gint              i, idx;                                                   \n"
"  struct hist_data *hist;                                                     \n"
"  gint              total_pixels = 0;                                         \n"
"                                                                              \n"
"  /* Count sizes */                                                           \n"
"  pyramid_t *l = pp;                                                          \n"
"  while (l != NULL)                                                           \n"
"    {                                                                         \n"
"      total_pixels += l->rows * l->cols;                                      \n"
"      l = l->next;                                                            \n"
"    }                                                                         \n"
"                                                                              \n"
"  /* Allocate memory */                                                       \n"
"  hist = g_new (struct hist_data, total_pixels);                              \n"
"                                                                              \n"
"  /* Build histogram info */                                                  \n"
"  l   = pp;                                                                   \n"
"  idx = 0;                                                                    \n"
"  while (l != NULL)                                                           \n"
"    {                                                                         \n"
"      const int pixels = l->rows*l->cols;                                     \n"
"      const int offset = idx;                                                 \n"
"      gint      c;                                                            \n"
"                                                                              \n"
"      _OMP (omp parallel for schedule(static))                                \n"
"      for (c = 0; c < pixels; c++)                                            \n"
"        {                                                                     \n"
"          hist[c+offset].size = sqrtf (l->Gx[c] * l->Gx[c] +                  \n"
"                                       l->Gy[c] * l->Gy[c]);                  \n"
"          hist[c+offset].index = c + offset;                                  \n"
"        }                                                                     \n"
"      idx += pixels;                                                          \n"
"      l = l->next;                                                            \n"
"    }                                                                         \n"
"                                                                              \n"
"  /* Generate histogram */                                                    \n"
"  qsort (hist, total_pixels, sizeof (struct hist_data),                       \n"
"         mantiuk06_hist_data_order);                                          \n"
"                                                                              \n"
"  /* Calculate cdf */                                                         \n"
"  {                                                                           \n"
"    const gfloat norm = 1.0f / (gfloat) total_pixels;                         \n"
"    _OMP (omp parallel for schedule(static))                                  \n"
"    for (i = 0; i < total_pixels; i++)                                        \n"
"      hist[i].cdf = ((gfloat) i) * norm;                                      \n"
"  }                                                                           \n"
"                                                                              \n"
"  /* Recalculate in terms of indexes */                                       \n"
"  qsort (hist, total_pixels,                                                  \n"
"         sizeof (struct hist_data),                                           \n"
"         mantiuk06_hist_data_index);                                          \n"
"                                                                              \n"
"  /*Remap gradient magnitudes */                                              \n"
"  l   = pp;                                                                   \n"
"  idx = 0;                                                                    \n"
"  while (l != NULL )                                                          \n"
"    {                                                                         \n"
"      gint      c;                                                            \n"
"      const int pixels = l->rows*l->cols;                                     \n"
"      const int offset = idx;                                                 \n"
"                                                                              \n"
"      _OMP (omp parallel for schedule(static))                                \n"
"      for (c = 0; c < pixels; c++)                                            \n"
"        {                                                                     \n"
"          const gfloat scale = contrastFactor      *                          \n"
"                               hist[c+offset].cdf  /                          \n"
"                               hist[c+offset].size;                           \n"
"          l->Gx[c] *= scale;                                                  \n"
"          l->Gy[c] *= scale;                                                  \n"
"        }                                                                     \n"
"      idx += pixels;                                                          \n"
"      l    = l->next;                                                         \n"
"    }                                                                         \n"
"                                                                              \n"
"  g_free (hist);                                                              \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"/* tone mapping */                                                            \n"
"static int                                                                    \n"
"mantiuk06_contmap (const int                       c,                         \n"
"                   const int                       r,                         \n"
"                   gfloat                   *const rgb,                       \n"
"                   gfloat                   *const Y,                         \n"
"                   const gfloat                    contrastFactor,            \n"
"                   const gfloat                    saturationFactor,          \n"
"                   const gboolean                  bcg,                       \n"
"                   const int                       itmax,                     \n"
"                   const gfloat                    tol,                       \n"
"                   pfstmo_progress_callback        progress)                  \n"
"{                                                                             \n"
"  const guint n = c*r;                                                        \n"
"        guint j;                                                              \n"
"                                                                              \n"
"  /* Normalize */                                                             \n"
"  gfloat Ymax = Y[0],                                                         \n"
"         clip_min;                                                            \n"
"                                                                              \n"
"  for (j = 1; j < n; j++)                                                     \n"
"      Ymax = MAX (Y[j], Ymax);                                                \n"
"                                                                              \n"
"  clip_min = 1e-7f * Ymax;                                                    \n"
"  _OMP (omp parallel for schedule(static))                                    \n"
"  for (j = 0; j < n * 4; j++)                                                 \n"
"      if (G_UNLIKELY (rgb[j] < clip_min)) rgb[j] = clip_min;                  \n"
"                                                                              \n"
"  _OMP (omp parallel for schedule(static))                                    \n"
"  for (j = 0; j < n; j++)                                                     \n"
"      if (G_UNLIKELY (  Y[j] < clip_min))   Y[j] = clip_min;                  \n"
"                                                                              \n"
"  _OMP (omp parallel for schedule(static))                                    \n"
"  for (j = 0; j < n; j++)                                                     \n"
"    {                                                                         \n"
"      rgb[j * 4 + 0] /= Y[j];                                                 \n"
"      rgb[j * 4 + 1] /= Y[j];                                                 \n"
"      rgb[j * 4 + 2] /= Y[j];                                                 \n"
"      Y[j]            = log10f (Y[j]);                                        \n"
"    }                                                                         \n"
"                                                                              \n"
"  {                                                                           \n"
"    /* create pyramid */                                                      \n"
"    pyramid_t *pp = mantiuk06_pyramid_allocate (c,r);                         \n"
"    gfloat    *tY = mantiuk06_matrix_alloc (n);                               \n"
"                                                                              \n"
"    /* copy Y to tY */                                                        \n"
"    mantiuk06_matrix_copy (n, Y, tY);                                         \n"
"    /* calculate gradients for pyramid, destroys tY */                        \n"
"    mantiuk06_pyramid_calculate_gradient (pp,tY);                             \n"
"    mantiuk06_matrix_free (tY);                                               \n"
"    /* transform gradients to R */                                            \n"
"    mantiuk06_pyramid_transform_to_R (pp);                                    \n"
"                                                                              \n"
"    /* Contrast map */                                                        \n"
"    if (contrastFactor > 0.0f)                                                \n"
"      /* Contrast mapping */                                                  \n"
"      mantiuk06_pyramid_gradient_multiply (pp, contrastFactor);               \n"
"    else                                                                      \n"
"      /* Contrast equalization */                                             \n"
"      mantiuk06_contrast_equalization (pp, -contrastFactor);                  \n"
"                                                                              \n"
"    /* transform R to gradients */                                            \n"
"    mantiuk06_pyramid_transform_to_G (pp);                                    \n"
"    /* transform gradients to luminance Y */                                  \n"
"    mantiuk06_transform_to_luminance (pp, Y, progress, bcg, itmax, tol);      \n"
"    mantiuk06_pyramid_free (pp);                                              \n"
"  }                                                                           \n"
"                                                                              \n"
"  /* Renormalize luminance */                                                 \n"
"  {                                                                           \n"
"    const gfloat CUT_MARGIN = 0.1f;                                           \n"
"    gfloat      *temp = mantiuk06_matrix_alloc (n),                           \n"
"                 trim, delta, l_min, l_max;                                   \n"
"                                                                              \n"
"    /* copy Y to temp */                                                      \n"
"    mantiuk06_matrix_copy (n, Y, temp);                                       \n"
"    /* sort temp in ascending order */                                        \n"
"    qsort (temp, n, sizeof (gfloat), mantiuk06_sort_float);                   \n"
"                                                                              \n"
"    /* const float median = (temp[(int)((n-1)/2)] + temp[(int)((n-1)/2+1)]) * 0.5f; */\n"
"    /* calculate median */                                                    \n"
"    trim  = (n - 1) * CUT_MARGIN * 0.01f;                                     \n"
"    delta = trim - floorf (trim);                                             \n"
"    l_min = temp[(int)floorf (trim)] * delta +                                \n"
"            temp[(int) ceilf (trim)] * (1.0f - delta);                        \n"
"                                                                              \n"
"    trim  = (n - 1) * (100.0f - CUT_MARGIN) * 0.01f;                          \n"
"    delta = trim - floorf (trim);                                             \n"
"    l_max = temp[(int)floorf (trim)] * delta +                                \n"
"            temp[(int) ceilf (trim)] * (1.0f - delta);                        \n"
"                                                                              \n"
"    mantiuk06_matrix_free (temp);                                             \n"
"    {                                                                         \n"
"      const gfloat disp_dyn_range = 2.3f;                                     \n"
"      _OMP (omp parallel for schedule(static))                                \n"
"      for (j = 0; j < n; j++)                                                 \n"
"          /* x scaled */                                                      \n"
"          Y[j] = ( Y[j] - l_min) /                                            \n"
"                 (l_max - l_min) *                                            \n"
"                 disp_dyn_range - disp_dyn_range;                             \n"
"    }                                                                         \n"
"                                                                              \n"
"    /* Transform to linear scale RGB */                                       \n"
"    _OMP (omp parallel for schedule(static))                                  \n"
"    for (j = 0; j < n; j++)                                                   \n"
"      {                                                                       \n"
"        Y[j] = powf (10,Y[j]);                                                \n"
"                                                                              \n"
"        rgb[j * 4 + 0] = powf (rgb[j * 4 + 0], saturationFactor) * Y[j];      \n"
"        rgb[j * 4 + 1] = powf (rgb[j * 4 + 1], saturationFactor) * Y[j];      \n"
"        rgb[j * 4 + 2] = powf (rgb[j * 4 + 2], saturationFactor) * Y[j];      \n"
"      }                                                                       \n"
"  }                                                                           \n"
"                                                                              \n"
"  return PFSTMO_OK;                                                           \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"static void                                                                   \n"
"mantiuk06_prepare (GeglOperation *operation)                                  \n"
"{                                                                             \n"
"  gegl_operation_set_format (operation, \"input\",  babl_format (OUTPUT_FORMAT));\n"
"  gegl_operation_set_format (operation, \"output\", babl_format (OUTPUT_FORMAT));\n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"static GeglRectangle                                                          \n"
"mantiuk06_get_required_for_output (GeglOperation        *operation,           \n"
"                                    const gchar         *input_pad,           \n"
"                                    const GeglRectangle *roi)                 \n"
"{                                                                             \n"
"  GeglRectangle result = *gegl_operation_source_get_bounding_box (operation,  \n"
"                                                                  \"input\"); \n"
"  return result;                                                              \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"static GeglRectangle                                                          \n"
"mantiuk06_get_cached_region (GeglOperation        *operation,                 \n"
"                              const GeglRectangle *roi)                       \n"
"{                                                                             \n"
"  return *gegl_operation_source_get_bounding_box (operation, \"input\");      \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"static gboolean                                                               \n"
"mantiuk06_process (GeglOperation       *operation,                            \n"
"                   GeglBuffer          *input,                                \n"
"                   GeglBuffer          *output,                               \n"
"                   const GeglRectangle *result,                               \n"
"                   gint                 level)                                \n"
"{                                                                             \n"
"  const GeglProperties *o      = GEGL_PROPERTIES (operation);                 \n"
"  const gint            pix_stride = 4; /* RGBA */                            \n"
"  gfloat               *lum, *pix;                                            \n"
"                                                                              \n"
"  g_return_val_if_fail (operation, FALSE);                                    \n"
"  g_return_val_if_fail (input, FALSE);                                        \n"
"  g_return_val_if_fail (output, FALSE);                                       \n"
"  g_return_val_if_fail (result, FALSE);                                       \n"
"                                                                              \n"
"  g_return_val_if_fail (babl_format_get_n_components (babl_format (OUTPUT_FORMAT)) == pix_stride, FALSE);\n"
"                                                                              \n"
"  /* Obtain the pixel data */                                                 \n"
"  lum = g_new (gfloat, result->width * result->height),                       \n"
"  gegl_buffer_get (input, result, 1.0, babl_format (\"Y float\"),             \n"
"                   lum, GEGL_AUTO_ROWSTRIDE, GEGL_ABYSS_NONE);                \n"
"                                                                              \n"
"  pix = g_new (gfloat, result->width * result->height * pix_stride);          \n"
"  gegl_buffer_get (input, result, 1.0, babl_format (OUTPUT_FORMAT),           \n"
"                   pix, GEGL_AUTO_ROWSTRIDE, GEGL_ABYSS_NONE);                \n"
"                                                                              \n"
"  mantiuk06_contmap (result->width, result->height, pix, lum,                 \n"
"                     o->contrast, o->saturation, FALSE, 200, 1e-3, NULL);     \n"
"                                                                              \n"
"  /* Cleanup and set the output */                                            \n"
"  gegl_buffer_set (output, result, 0, babl_format (OUTPUT_FORMAT), pix,       \n"
"                   GEGL_AUTO_ROWSTRIDE);                                      \n"
"  g_free (pix);                                                               \n"
"  g_free (lum);                                                               \n"
"                                                                              \n"
"  return TRUE;                                                                \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"/*                                                                            \n"
" */                                                                           \n"
"static void                                                                   \n"
"gegl_op_class_init (GeglOpClass *klass)                                       \n"
"{                                                                             \n"
"  GeglOperationClass       *operation_class;                                  \n"
"  GeglOperationFilterClass *filter_class;                                     \n"
"                                                                              \n"
"  operation_class = GEGL_OPERATION_CLASS (klass);                             \n"
"  filter_class    = GEGL_OPERATION_FILTER_CLASS (klass);                      \n"
"                                                                              \n"
"  filter_class->process = mantiuk06_process;                                  \n"
"                                                                              \n"
"  operation_class->prepare                 = mantiuk06_prepare;               \n"
"  operation_class->get_required_for_output = mantiuk06_get_required_for_output;\n"
"  operation_class->get_cached_region       = mantiuk06_get_cached_region;     \n"
"  operation_class->threaded                = FALSE;                           \n"
"                                                                              \n"
"  gegl_operation_class_set_keys (operation_class,                             \n"
"      \"name\",        \"gegl:mantiuk06\",                                    \n"
"      \"title\",       _(\"Mantiuk 2006 Tone Mapping\"),                      \n"
"      \"categories\" , \"tonemapping\",                                       \n"
"      \"reference-hash\", \"cd4aecc6eeb94085424c7adb6d43e92a\",               \n"
"      \"description\",                                                        \n"
"        _(\"Adapt an image, which may have a high dynamic range, for \"       \n"
"          \"presentation using a low dynamic range. This operator constrains \"\n"
"          \"contrasts across multiple spatial frequencies, producing \"       \n"
"          \"luminance within the range 0.0-1.0\"),                            \n"
"        NULL);                                                                \n"
"}                                                                             \n"
"                                                                              \n"
"#endif                                                                        \n"
;
