# Copyright (C) 2011-2012 - Curtis Hovey <sinzui.is at verizon.net>
# This software is licensed under the MIT license (see the file COPYING).

from gi.repository import (
    Gio,
    )

from gdpformat import FormatPlugin
from gdp import (
    config,
    )
from gdp.format import Formatter
from testing import GeditTestCase


class FormatPluginTestCase(GeditTestCase):

    def test_init(self):
        window, view, document = self.make_gedit(
            'plugins/gdp/data/snark12.txt')
        plugin = FormatPlugin()
        self.assertIs(None, plugin.controller)
        self.assertIs(None, plugin.window)

    def test_do_activate(self):
        # do_activate sets up the menus and gets the controller.
        window, view, document = self.make_gedit(
            'plugins/gdp/data/snark12.txt')
        plugin = FormatPlugin()
        plugin.window = window
        plugin.do_activate()
        self.assertIsInstance(plugin.controller, Formatter)
        self.assertEqual(
            ['active-tab-changed', 'tab-added'],
            sorted(plugin.signal_ids.keys()))

    def test_do_update_state(self):
        # do_update_state() is implemented, but it does nothing.
        window, view, document = self.make_gedit(
            'plugins/gdp/data/snark12.txt')
        plugin = FormatPlugin()
        plugin.window = window
        plugin.do_activate()
        self.assertIs(None, plugin.do_update_state())

    def test_do_deactivate(self):
        # do_deactivate removes the menus and the controller.
        window, view, document = self.make_gedit(
            'plugins/gdp/data/snark12.txt')
        plugin = FormatPlugin()
        plugin.window = window
        plugin.do_activate()
        plugin.do_deactivate()
        self.assertIs(None, plugin.controller)

    def test_on_tab_added_or_changed(self):
        window, view, document = self.make_gedit(
            'plugins/gdp/data/snark12.txt')
        plugin = FormatPlugin()
        plugin.window = window
        plugin.do_activate()
        self.assertEqual(
            ['active-tab-changed', 'tab-added'],
            sorted(plugin.signal_ids.keys()))
        location = Gio.file_new_for_path('plugins/gdp/data/snark.py')
        window.create_tab_from_location(location, None, 0, 0, False, True)
        self.assertEqual(
            ['active-tab-changed', 'saved', 'syntax-error-python',
             'tab-added'],
            sorted(plugin.signal_ids.keys()))

    def test_on_tab_added_or_changed_css(self):
        window, view, document = self.make_gedit(
            'plugins/gdp/data/snark12.txt')
        plugin = FormatPlugin()
        plugin.window = window
        plugin.do_activate()
        manager = window.get_ui_manager()
        format_css_item = '/MenuBar/ToolsMenu/ToolsOps_2/ReformatCSS'
        plugin.on_tab_added_or_changed(window, window.get_active_tab())
        self.assertFalse(manager.get_action(format_css_item).props.sensitive)
        location = Gio.file_new_for_path('plugins/gdp/data/snark.css')
        window.create_tab_from_location(location, None, 0, 0, False, True)
        self.assertTrue(manager.get_action(format_css_item).props.sensitive)

    def test_on_tab_added_or_changed_doctest(self):
        window, view, document = self.make_gedit(
            'plugins/gdp/data/snark12.txt')
        plugin = FormatPlugin()
        plugin.window = window
        plugin.do_activate()
        manager = window.get_ui_manager()
        format_css_item = '/MenuBar/ToolsMenu/ToolsOps_2/ReformatDoctest'
        plugin.on_tab_added_or_changed(window, window.get_active_tab())
        self.assertFalse(manager.get_action(format_css_item).props.sensitive)
        file_ = self.make_file('', suffix='.doctest')
        location = Gio.file_new_for_path(file_.name)
        window.create_tab_from_location(location, None, 0, 0, False, True)
        self.assertTrue(manager.get_action(format_css_item).props.sensitive)

    def test_actions(self):
        window, view, document = self.make_gedit(
            'plugins/gdp/data/snark12.txt')
        plugin = FormatPlugin()
        plugin.window = window
        plugin.do_activate()
        action = plugin.actions(plugin.controller)[12]
        self.assertEqual('ShowSyntaxErrorsOnly', action[0])
        self.assertEqual(
            config.getboolean('formatter', 'report_only_errors'), action[6])

    def test_on_document_saved_no_controller(self):
        window, view, document = self.make_gedit(
            'plugins/gdp/data/snark12.txt')
        plugin = FormatPlugin()
        plugin.window = window
        plugin.do_activate()
        plugin.controller = None
        self.assertIs(
            None, plugin.on_document_saved(document, None))
