# Copyright (C) 2009-2012 - Curtis Hovey <sinzui.is at verizon.net>
# This software is licensed under the GNU General Public License version 2
# (see the file COPYING).
"""GDP Gedit Developer Plugins."""

__all__ = [
    'config',
    'ControllerMixin',
    'set_file_line',
    'setup_file_lines_view',
    ]


from ConfigParser import SafeConfigParser
import mimetypes
import os
import re
from StringIO import StringIO

from xdg import BaseDirectory

from gi.repository import (
    GObject,
    Gio,
    Pango,
    Gtk,
    GtkSource,
    )

# Initialise the mimetypes for document type inspection.
mimetypes.init()
mimetypes.add_type('application/x-zope-configuation', '.zcml')
mimetypes.add_type('application/x-zope-page-template', '.pt')
mimetypes.add_type('text/x-python-doctest', '.doctest')
mimetypes.add_type('text/x-python', '.tac')

lang_manager = GtkSource.LanguageManager.get_default()

doctest_pattern = re.compile(
    r'^.*(doc|test|stories).*/.*\.(txt|doctest)$')


class ControllerMixin(object):
    """Provide common features to plugins"""

    def deactivate(self):
        """Clean up resources before deactivation."""
        raise NotImplementedError

    @staticmethod
    def is_editable(mime_type):
        """ Only search mime-types that Gedit can open.

        A fuzzy match of text/ or +xml is good, but some files types are
        unknown or described as application data.
        """
        editable_types = (
            'application/javascript',
            'application/sgml',
            'application/xml',
            'application/x-httpd-eruby',
            'application/x-httpd-php',
            'application/x-latex',
            'application/x-ruby',
            'application/x-sh',
            'application/x-zope-configuation',
            'application/x-zope-page-template',
            'text/x-python-doctest',
            )
        return (
            mime_type is None
            or 'text/' in mime_type
            or mime_type.endswith('+xml')
            or mime_type in editable_types)

    def is_doc_open(self, uri):
        """Return True if the window already has a document opened for uri."""
        # get_uri_for_display() can return an abs path?
        path = uri.replace('file://', '')
        for doc in self.window.get_documents():
            if doc.get_uri_for_display() in (uri, path):
                return True
        return False

    def open_doc(self, uri, jump_to=None):
        """Open document at uri if it can be, and is not already, opened."""
        if self.is_doc_open(uri):
            return
        mime_type, charset_ = mimetypes.guess_type(uri)
        if self.is_editable(mime_type):
            jump_to = jump_to or 0
            location = Gio.file_new_for_uri(uri)
            self.window.create_tab_from_location(
                location, None, jump_to, 0, False, False)
            self.window.get_active_view().scroll_to_cursor()

    def activate_open_doc(self, uri, jump_to=None):
        """Activate (or open) a document and jump to the line number."""
        self.open_doc(uri, jump_to)
        location = Gio.file_new_for_uri(uri)
        self.window.set_active_tab(
            self.window.get_tab_from_location(location))
        if jump_to is not None:
            self.window.get_active_document().goto_line(jump_to)
            self.window.get_active_view().scroll_to_cursor()

    @property
    def active_document(self):
        """The active document in the window."""
        return self.window.get_active_document()

    @property
    def text(self):
        """The text of the active Gedit.Document or None."""
        document = self.window.get_active_document()
        start_iter = document.get_start_iter()
        end_iter = document.get_end_iter()
        return document.get_text(start_iter, end_iter, True)

    def correct_language(self, document):
        """Correct the language for ambuguous mime-types."""
        if not hasattr(document, 'get_language'):
            return
        file_path = document.get_uri_for_display()
        if doctest_pattern.match(file_path):
            document.set_language(lang_manager.get_language('doctest'))
        elif file_path.endswith('.tac'):
            document.set_language(lang_manager.get_language('python'))


def set_file_line(column, cell, model, piter, cell_type):
    """Set the value as file or line information."""
    file_path = model.get_value(piter, 0)
    #icon = model.get_value(piter, 1)
    line_no = model.get_value(piter, 2)
    text = model.get_value(piter, 3)
    if text is None:
        if cell_type == 'text':
            cell.props.text = file_path
        else:
            # cell_type == 'line_no'
            cell.props.text = ''
    else:
        if cell_type == 'text':
            cell.props.text = text
        else:
            # cell_type == 'line_no'
            cell.props.text = str(line_no)


def on_file_lines_row_activated(treeview, path, view_column, plugin):
    """Open the file and jump to the line."""
    treestore = treeview.get_model()
    piter = treestore.get_iter(path)
    base_dir = treestore.get_value(piter, 4)
    path = treestore.get_value(piter, 0)
    if base_dir is None or path is None:
        # There is not enough information to open a document.
        return
    uri = 'file://%s' % os.path.abspath(os.path.join(base_dir, path))
    line_no = treestore.get_value(piter, 2) - 1
    if line_no < 0:
        line_no = 0
    plugin.activate_open_doc(uri, jump_to=line_no)


def on_file_lines_resize(treeview, allocation, column, cell):
    """Set the width of the column to update text wrapping."""
    # Wrap before the vertical scrollbar to ensure there is never a
    # horizonal scrollbar.
    parent_width = treeview.get_parent().get_allocated_width()
    new_width = parent_width - 100
    if cell.props.wrap_width == new_width:
        return
    cell.props.wrap_width = new_width
    store = treeview.get_model()
    iter = store.get_iter_first()
    while iter and store.iter_is_valid(iter):
        store.row_changed(store.get_path(iter), iter)
        iter = store.iter_next(iter)
        treeview.set_size_request(0, -1)


def setup_file_lines_view(file_lines_view, plugin, column_title):
    """Setup a TreeView to displau files and their lines."""
    treestore = Gtk.TreeStore(
        GObject.TYPE_STRING, GObject.TYPE_STRING, GObject.TYPE_INT,
        GObject.TYPE_STRING, GObject.TYPE_STRING)
    treestore.append(None, ('', None, 0, None, None))
    column = Gtk.TreeViewColumn(column_title)
    # icon.
    cell = Gtk.CellRendererPixbuf()
    cell.set_property('stock-size', Gtk.IconSize.MENU)
    cell.props.yalign = 0
    column.pack_start(cell, False)
    column.add_attribute(cell, 'icon-name', 1)
    # line number.
    cell = Gtk.CellRendererText()
    cell.props.yalign = 0
    cell.props.xalign = 1
    cell.props.alignment = Pango.Alignment.RIGHT
    cell.props.family = 'Monospace'
    column.pack_start(cell, False)
    column.set_cell_data_func(cell, set_file_line, 'line_no')
    # line text.
    cell = Gtk.CellRendererText()
    cell.props.wrap_mode = Pango.WrapMode.WORD
    cell.props.wrap_width = 310
    column.pack_start(cell, False)
    column.set_cell_data_func(cell, set_file_line, 'text')
    file_lines_view.set_model(treestore)
    file_lines_view.set_level_indentation(-18)
    file_lines_view.append_column(column)
    file_lines_view.set_search_column(0)
    file_lines_view.connect(
        'row-activated', on_file_lines_row_activated, plugin)
    file_lines_view.connect_after(
        'size-allocate', on_file_lines_resize, column, cell)


CONFIG_VERSION = 1

default_conf = """\
[gdp]
version = %s

[completer]
show_accel = <Control>space
suggest_completions = False

[formatter]
report_only_errors = False

[finder]
paths = <Current File>
    <Working Directory>
matches =
files = <Any Text File>
substitutions =
""" % CONFIG_VERSION


class Config(SafeConfigParser):
    """A config parser that knows it's defaults and file locations."""

    def __init__(self):
        SafeConfigParser.__init__(self)
        self.do_update_state = False
        self.readfp(StringIO(default_conf), 'default_conf')
        self.dir_path = os.path.join(BaseDirectory.xdg_config_home, 'gdp')
        self.file_path = os.path.join(self.dir_path, 'gdp.conf')
        self._loaded_file_path = None

    def load(self):
        """Read the optional conf."""
        self._loaded_file_path = self.read(self.file_path)

    def dump(self):
        """Write the user changes to the optional conf."""
        if not os.path.isdir(self.dir_path):
            os.makedirs(self.dir_path)
        with open(self.file_path, 'wb') as config_file:
            self.write(config_file)

    def getlist(self, section, key):
        return self.get(section, key).split('\n')

    def setlist(self, section, key, value_list):
        value = '\n'.join(value_list)
        self.set(section, key, value)


config = Config()
config.load()
