/* -*- mode: C++; c-basic-offset: 2; indent-tabs-mode: nil -*- */
/*
 *  Main authors:
 *     Christian Schulte <schulte@gecode.org>
 *
 *  Copyright:
 *     Christian Schulte, 2010
 *
 *  Last modified:
 *     $Date: 2013-02-25 21:43:24 +0100 (Mon, 25 Feb 2013) $ by $Author: schulte $
 *     $Revision: 13406 $
 *
 *  This file is part of Gecode, the generic constraint
 *  development environment:
 *     http://www.gecode.org
 *
 *  Permission is hereby granted, free of charge, to any person obtaining
 *  a copy of this software and associated documentation files (the
 *  "Software"), to deal in the Software without restriction, including
 *  without limitation the rights to use, copy, modify, merge, publish,
 *  distribute, sublicense, and/or sell copies of the Software, and to
 *  permit persons to whom the Software is furnished to do so, subject to
 *  the following conditions:
 *
 *  The above copyright notice and this permission notice shall be
 *  included in all copies or substantial portions of the Software.
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 *  EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 *  MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 *  NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
 *  LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
 *  OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
 *  WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 */

#include <gecode/driver.hh>

#include <gecode/int.hh>
#include <gecode/minimodel.hh>

#include <algorithm>

using namespace Gecode;

// Instance data
namespace {

  // Instances
  extern const int* bpp[];
  // Instance names
  extern const char* name[];

  /// A wrapper class for instance data
  class Spec {
  protected:
    /// Raw instance data
    const int* data;
    /// Lower and upper bound
    int l, u;
  public:
    /// Whether a valid specification has been found
    bool valid(void) const {
      return data != NULL;
    }
    /// Return maximal capacity of a bin
    int capacity(void) const {
      return data[0];
    }
    /// Return number of items
    int items(void) const {
      return data[1];
    }
    /// Return size of item \a i
    int size(int i) const {
      return data[i+2];
    }
  protected:
    /// Find instance by name \a s
    static const int* find(const char* s) {
      for (int i=0; name[i] != NULL; i++)
        if (!strcmp(s,name[i]))
          return bpp[i];
      return NULL;
    }
    /// Compute lower bound
    int clower(void) const {
      /*
       * The lower bound is due to: S. Martello, P. Toth. Lower bounds
       * and reduction procedures for the bin packing problem.
       * Discrete and applied mathematics, 28(1):59-70, 1990.
       */
      const int c = capacity(), n = items();
      int l = 0;

      // Items in N1 are from 0 ... n1 - 1
      int n1 = 0;
      // Items in N2 are from n1 ... n12 - 1, we count elements in N1 and N2
      int n12 = 0;
      // Items in N3 are from n12 ... n3 - 1 
      int n3 = 0;
      // Free space in N2
      int f2 = 0;
      // Total size of items in N3
      int s3 = 0;

      // Initialize n12 and f2
      for (; (n12 < n) && (size(n12) > c/2); n12++)
        f2 += c - size(n12);

      // Initialize n3 and s3
      for (n3 = n12; n3 < n; n3++)
        s3 += size(n3);
        
      // Compute lower bounds
      for (int k=0; k<=c/2; k++) {
        // Make N1 larger by adding elements and N2 smaller
        for (; (n1 < n) && (size(n1) > c-k); n1++)
          f2 -= c - size(n1);
        assert(n1 <= n12);
        // Make N3 smaller by removing elements
        for (; (size(n3-1) < k) && (n3 > n12); n3--)
          s3 -= size(n3-1);
        // Overspill
        int o = (s3 > f2) ? ((s3 - f2 + c - 1) / c) : 0;
        l = std::max(l, n12 + o);
      }
      return l;
    }
    /// Compute upper bound
    int cupper(void) const {
      // Use a naive greedy algorithm
      const int c = capacity(), n = items();

      int* f = new int[n];
      for (int i=0; i<n; i++)
        f[i] = c;
      
      int u=0;
      for (int i=0; i<n; i++) {
        // Skip bins with insufficient free space
        int j=0;
        while (f[j] < size(i))
          j++;
        f[j] -= size(i);
        u = std::max(u,j);
      }
      delete [] f;
      return u+1;
    }
  public:
    /// Initialize
    Spec(const char* s) : data(find(s)), l(0), u(0) {
      if (valid()) {
        l = clower(); u = cupper();
      }
    }
    /// Return sum of all item sizes
    int total(void) const {
      int t=0;
      for (int i=0; i<items(); i++)
        t += size(i);
      return t;
    }
    /// Return lower bound
    int lower(void) const {
      return l;
    }
    /// Return upper bound
    int upper(void) const {
      return u;
    }
  };

}

/** \brief Custom brancher implementing CDBF
 *
 * This class implements complete decreasing best fit branching (CDBF)
 * from: Ian Gent and Toby Walsh. From approximate to optimal solutions:
 * Constructing pruning and propagation rules. IJCAI 1997.
 *
 * Additional domination rules are taken from: Paul Shaw. A Constraint 
 * for Bin Packing. CP 2004
 *
 * \relates BinPacking
 */
class CDBF : public Brancher {
protected:
  /// Views for the loads
  ViewArray<Int::IntView> load;
  /// Views for the bins
  ViewArray<Int::IntView> bin;
  /// Array of sizes (shared)
  IntSharedArray size;
  /// Next view to branch on
  mutable int item;
  /// %Choice
  class Choice : public Gecode::Choice {
  public:
    /// Item
    int item;
    /// Bins with same slack
    int* same;
    /// Number of bins with same slack
    int n_same;
    /** Initialize choice for brancher \a b, alternatives \a a, 
     *  item \a i and same bins \a s.
     */
    Choice(const Brancher& b, unsigned int a, int i, int* s, int n_s)
      : Gecode::Choice(b,a), item(i), 
        same(heap.alloc<int>(n_s)), n_same(n_s) {
      for (int k=n_same; k--; )
        same[k] = s[k];
    }
    /// Report size occupied
    virtual size_t size(void) const {
      return sizeof(Choice) + sizeof(int) * n_same;
    }
    /// Archive into \a e
    virtual void archive(Archive& e) const {
      Gecode::Choice::archive(e);
      e << alternatives() << item << n_same;
      for (int i=n_same; i--;) 
        e << same[i];
    }
    /// Destructor
    virtual ~Choice(void) {
      heap.free<int>(same,n_same);
    }
  };
 
public:
  /// Construct brancher
  CDBF(Home home, ViewArray<Int::IntView>& l, ViewArray<Int::IntView>& b,
       IntSharedArray& s) 
    : Brancher(home), load(l), bin(b), size(s), item(0) {
    home.notice(*this,AP_DISPOSE);
  }
  /// Brancher post function
  static BrancherHandle post(Home home, ViewArray<Int::IntView>& l, 
                             ViewArray<Int::IntView>& b,
                             IntSharedArray& s) {
    return *new (home) CDBF(home, l, b, s);
  }
  /// Copy constructor
  CDBF(Space& home, bool share, CDBF& cdbf) 
    : Brancher(home, share, cdbf), item(cdbf.item) {
    load.update(home, share, cdbf.load);
    bin.update(home, share, cdbf.bin);
    size.update(home, share, cdbf.size);
  }
  /// Copy brancher
  virtual Actor* copy(Space& home, bool share) {
    return new (home) CDBF(home, share, *this);
  }
  /// Delete brancher and return its size
  virtual size_t dispose(Space& home) {
    home.ignore(*this,AP_DISPOSE);
    size.~IntSharedArray();
    return sizeof(*this);
  }
  /// Check status of brancher, return true if alternatives left
  virtual bool status(const Space&) const {
    for (int i = item; i < bin.size(); i++)
      if (!bin[i].assigned()) {
        item = i; return true;
      }
    return false;
  }
  /// Return choice
  virtual Gecode::Choice* choice(Space& home) {
    assert(!bin[item].assigned());

    int n = bin.size(), m = load.size();

    Region region(home);

    // Free space in bins
    int* free = region.alloc<int>(m);

    for (int j=m; j--; )
      free[j] = load[j].max();
    for (int i=n; i--; )
      if (bin[i].assigned())
        free[bin[i].val()] -= size[i];

    // Equivalent bins with same free space
    int* same = region.alloc<int>(m+1);
    unsigned int n_same = 0;
    unsigned int n_possible = 0;
    
    // Initialize such that failure is guaranteed (pack into bin -1)
    same[n_same++] = -1;

    // Find a best-fit bin for item
    int slack = INT_MAX;
    for (Int::ViewValues<Int::IntView> j(bin[item]); j(); ++j) 
      if (size[item] <= free[j.val()]) {
        // Item still can fit into the bin
        n_possible++;
        if (free[j.val()] - size[item] < slack) {
          // A new, better fit
          slack = free[j.val()] - size[item];
          same[0] = j.val(); n_same = 1;
        } else if (free[j.val()] - size[item] == slack) {
          // An equivalent bin, remember it
          same[n_same++] = j.val();
        }
      }
    /* 
     * Domination rules: 
     *  - if the item fits the bin exactly, just assign
     *  - if all possible bins are equivalent, just assign
     *
     * Also catches failure: if no possible bin was found, commit
     * the item into bin -1.
     */
    if ((slack == 0) || (n_same == n_possible) || (slack == INT_MAX))
      return new Choice(*this, 1, item, same, 1);
    else
      return new Choice(*this, 2, item, same, n_same);
  }
  /// Return choice
  virtual const Gecode::Choice* choice(const Space& home, Archive& e) {
    int alt, item, n_same;
    e >> alt >> item >> n_same;
    Region re(home);
    int* same = re.alloc<int>(n_same);
    for (int i=n_same; i--;) e >> same[i];
    return new Choice(*this, alt, item, same, n_same);
  }
  /// Perform commit for choice \a _c and alternative \a a
  virtual ExecStatus commit(Space& home, const Gecode::Choice& _c, 
                            unsigned int a) {
    const Choice& c = static_cast<const Choice&>(_c);
    // This catches also the case that the choice has a single aternative only
    if (a == 0) {
      GECODE_ME_CHECK(bin[c.item].eq(home, c.same[0]));
    } else {
      Iter::Values::Array same(c.same, c.n_same);

      GECODE_ME_CHECK(bin[c.item].minus_v(home, same));

      for (int i = c.item+1; (i<bin.size()) && 
                             (size[i] == size[c.item]); i++) {
        same.reset();
        GECODE_ME_CHECK(bin[i].minus_v(home, same));
      }
    }
    return ES_OK;
  }
};

/// Post branching (assumes that \a s is sorted)
BrancherHandle cdbf(Home home, const IntVarArgs& l, const IntVarArgs& b,
                    const IntArgs& s) {
  if (b.size() != s.size())
    throw Int::ArgumentSizeMismatch("cdbf");      
  ViewArray<Int::IntView> load(home, l);
  ViewArray<Int::IntView> bin(home, b);
  IntSharedArray size(s);
  return CDBF::post(home, load, bin, size);
}



/**
 * \brief %Example: Bin packing
 *
 * \ingroup Example
 *
 */
class BinPacking : public MinimizeScript {
protected:
  /// Specification
  const Spec spec;
  /// Load for each bin
  IntVarArray load;
  /// Bin for each item
  IntVarArray bin;
  /// Number of bins
  IntVar bins;
public:
  /// Model variants
  enum {
    MODEL_NAIVE, ///< Use naive model
    MODEL_PACKING ///< Use bin packing constraint
  };
  /// Branching to use for model
  enum {
    BRANCH_NAIVE, ///< Use naive branching
    BRANCH_CDBF, ///< Use CDBF
  };
  /// Actual model
  BinPacking(const InstanceOptions& opt) 
    : spec(opt.instance()),
      load(*this, spec.upper(), 0, spec.capacity()),
      bin(*this, spec.items(), 0, spec.upper()-1),
      bins(*this, spec.lower(), spec.upper()) {
    // Number of items
    int n = bin.size();
    // Number of bins
    int m = load.size();

    // Size of all items
    int s = 0;
    for (int i=0; i<n; i++)
      s += spec.size(i);

    // Array of sizes
    IntArgs sizes(n);
    for (int i=0; i<n; i++)
      sizes[i] = spec.size(i);
      
    switch (opt.model()) {
    case MODEL_NAIVE:
      {
        // All loads must add up to all item sizes
        linear(*this, load, IRT_EQ, s);

        // Load must be equal to packed items
        BoolVarArgs _x(*this, n*m, 0, 1);
        Matrix<BoolVarArgs> x(_x, n, m);
      
        for (int i=0; i<n; i++)
          channel(*this, x.col(i), bin[i]);

        for (int j=0; j<m; j++)
          linear(*this, sizes, x.row(j), IRT_EQ, load[j]);
      }
      break;
    case MODEL_PACKING:
      binpacking(*this, load, bin, sizes);
      break;
    }

    // Break symmetries
    for (int i=1; i<n; i++)
      if (spec.size(i-1) == spec.size(i))
        rel(*this, bin[i-1] <= bin[i]);

    // Pack items that require a bin for sure! (wlog)
    {
      int i = 0;
      // These items all need a bin due to their own size
      for (; (i < n) && (i < m) && (spec.size(i) * 2 > spec.capacity()); i++)
        rel(*this, bin[i] == i);
      // Check if the next item cannot fit to position i-1
      if ((i < n) && (i < m) && (i > 0) && 
          (spec.size(i-1) + spec.size(i) > spec.capacity()))
        rel(*this, bin[i] == i);
    }

    // All excess bins must be empty
    for (int j=spec.lower()+1; j <= spec.upper(); j++)
      rel(*this, (bins < j) == (load[j-1] == 0));

    branch(*this, bins, INT_VAL_MIN());
    switch (opt.branching()) {
    case BRANCH_NAIVE:
      branch(*this, bin, INT_VAR_NONE(), INT_VAL_MIN());
      break;
    case BRANCH_CDBF:
      cdbf(*this, load, bin, sizes);
      break;
    }
  }
  /// Return cost
  virtual IntVar cost(void) const {
    return bins;
  }
  /// Constructor for cloning \a s
  BinPacking(bool share, BinPacking& s) 
    : MinimizeScript(share,s), spec(s.spec) {
    load.update(*this, share, s.load);
    bin.update(*this, share, s.bin);
    bins.update(*this, share, s.bins);
  }
  /// Copy during cloning
  virtual Space*
  copy(bool share) {
    return new BinPacking(share,*this);
  }
  /// Print solution
  virtual void
  print(std::ostream& os) const {
    int n = bin.size();
    int m = load.size();
    os << "Bins used: " << bins << " (from " << m << " bins)." << std::endl;
    for (int j=0; j<m; j++) {
      bool fst = true;
      os << "\t[" << j << "]={";
      for (int i=0; i<n; i++)
        if (bin[i].assigned() && (bin[i].val() == j)) {
          if (fst) {
            fst = false;
          } else {
            os << ",";
          }
          os << i;
        }
      os << "} #" << load[j] << std::endl;
    }
    if (!bin.assigned()) {
      os << std::endl 
         << "Unpacked items:" << std::endl;
      for (int i=0;i<n; i++)
        if (!bin[i].assigned())
          os << "\t[" << i << "] = " << bin[i] << std::endl;
    }
  }
};

/** \brief Main-function
 *  \relates BinPacking
 */
int
main(int argc, char* argv[]) {
  InstanceOptions opt("BinPacking");
  opt.model(BinPacking::MODEL_PACKING);
  opt.model(BinPacking::MODEL_NAIVE, "naive", 
            "use naive model (decomposition)");
  opt.model(BinPacking::MODEL_PACKING, "packing", 
            "use bin packing constraint");
  opt.branching(BinPacking::BRANCH_CDBF);
  opt.branching(BinPacking::BRANCH_NAIVE, "naive");
  opt.branching(BinPacking::BRANCH_CDBF, "cdbf");
  opt.instance(name[0]);
  opt.solutions(0);
  opt.parse(argc,argv);
  if (!Spec(opt.instance()).valid()) {
    std::cerr << "Error: unkown instance" << std::endl;
    return 1;
  }
  MinimizeScript::run<BinPacking,BAB,InstanceOptions>(opt);
  return 0;
}

namespace {

  /*
   * Instances taken from:
   * A. Scholl, R. Klein, and C. Jrgens: BISON: a fast hybrid procedure
   * for exactly solving the one-dimensional bin packing problem.
   * Computers & Operations Research 24 (1997) 627-645. 
   *
   * The item size have been sorted for simplicty.
   *
   */

  /*
   * Data set 1
   *
   */
  const int n1c1w1_a[] = {
    100, // Capacity
    50, // Number of items
    // Size of items (sorted)
    99,99,96,96,92,92,91,88,87,86,85,76,74,72,69,67,67,62,61,56,52,
    51,49,46,44,42,40,40,33,33,30,30,29,28,28,27,25,24,23,22,21,20,
    17,14,13,11,10,7,7,3
  };
  const int n1c1w1_b[] = {
    100, // Capacity
    50, // Number of items
    // Size of items (sorted)
    100,99,97,97,97,93,93,92,92,88,83,83,79,76,76,75,72,71,70,69,
    67,66,63,62,62,61,61,51,50,44,44,43,43,40,39,37,37,30,23,20,19,
    18,17,15,14,13,13,12,8,8
  };
  const int n1c1w1_c[] = {
    100, // Capacity
    50, // Number of items
    // Size of items (sorted)
    92,89,87,84,82,82,81,75,73,71,67,67,63,59,57,56,52,49,48,47,46,
    41,39,38,36,35,34,34,30,29,26,21,20,19,18,15,15,13,11,10,10,10,
    9,8,8,7,6,6,6,3
  };
  const int n1c1w1_d[] = {
    100, // Capacity
    50, // Number of items
    // Size of items (sorted)
    100,99,98,97,95,94,92,92,91,82,80,77,76,75,73,73,73,71,68,65,
    65,63,63,63,60,59,53,45,44,40,31,25,24,24,24,23,22,21,21,15,14,
    14,10,10,7,7,6,3,2,2
  };
  const int n1c1w1_e[] = {
    100, // Capacity
    50, // Number of items
    // Size of items (sorted)
    91,88,88,87,87,86,86,85,85,84,83,80,79,78,77,70,70,68,67,66,59,
    52,49,48,47,47,44,42,38,37,37,34,34,33,31,29,27,24,21,17,16,16,
    15,14,8,6,5,4,2,2
  };
  const int n1c1w1_f[] = {
    100, // Capacity
    50, // Number of items
    // Size of items (sorted)
    99,98,98,93,92,89,89,84,84,83,78,77,75,73,72,71,70,69,69,68,60,
    60,57,56,54,50,49,49,45,37,36,35,30,30,27,26,26,25,24,21,20,19,
    15,14,13,11,11,8,2,2
  };
  const int n1c1w1_g[] = {
    100, // Capacity
    50, // Number of items
    // Size of items (sorted)
    100,99,98,98,98,91,90,87,84,84,78,77,72,71,70,69,69,64,63,58,
    58,46,45,45,43,43,42,41,37,37,37,35,34,31,30,29,24,23,22,21,20,
    17,12,11,10,9,7,6,5,4
  };
  const int n1c1w1_h[] = {
    100, // Capacity
    50, // Number of items
    // Size of items (sorted)
    97,93,93,92,92,91,90,88,86,85,85,85,82,81,80,79,75,73,71,70,70,
    67,66,64,62,62,61,54,48,48,47,46,44,41,40,39,34,29,24,24,21,18,
    16,16,14,13,11,10,5,1
  };
  const int n1c1w1_i[] = {
    100, // Capacity
    50, // Number of items
    // Size of items (sorted)
    95,92,87,87,85,84,83,79,77,77,75,73,69,68,65,63,63,62,61,58,57,
    52,50,44,43,40,40,38,38,38,35,33,33,32,31,29,27,24,24,22,19,19,
    18,16,14,11,6,4,3,2
  };
  const int n1c1w1_j[] = {
    100, // Capacity
    50, // Number of items
    // Size of items (sorted)
    99,99,95,94,94,93,91,90,86,81,81,80,79,77,74,69,69,63,55,54,54,
    53,52,50,44,40,39,38,37,36,36,36,36,34,31,31,26,25,23,22,18,17,
    15,14,13,12,10,7,2,1
  };
  const int n1c1w1_k[] = {
    100, // Capacity
    50, // Number of items
    // Size of items (sorted)
    96,91,91,89,87,85,84,83,82,79,78,77,77,75,75,70,68,66,64,62,62,
    56,53,51,44,41,40,38,38,36,34,31,30,29,28,27,26,23,17,16,15,14,
    14,12,11,10,8,8,4,2
  };
  const int n1c1w1_l[] = {
    100, // Capacity
    50, // Number of items
    // Size of items (sorted)
    99,99,98,96,95,93,92,92,89,87,85,85,82,80,72,71,68,68,64,64,63,
    61,59,59,57,57,57,55,55,52,52,51,49,48,47,47,40,39,38,37,29,28,
    28,22,22,19,17,16,9,4
  };
  const int n1c1w1_m[] = {
    100, // Capacity
    50, // Number of items
    // Size of items (sorted)
    100,100,99,97,94,93,91,90,89,88,87,87,86,86,79,77,72,71,70,69,
    68,68,65,64,61,60,59,51,50,50,43,42,39,37,29,27,25,24,21,19,17,
    16,13,13,8,6,6,3,2,1
  };
  const int n1c1w1_n[] = {
    100, // Capacity
    50, // Number of items
    // Size of items (sorted)
    99,98,95,95,95,94,94,91,88,87,86,85,76,74,73,71,68,60,55,54,51,
    45,42,40,39,39,36,34,33,32,32,31,31,30,29,26,26,23,21,21,21,19,
    18,18,16,15,5,5,4,1
  };
  const int n1c1w1_o[] = {
    100, // Capacity
    50, // Number of items
    // Size of items (sorted)
    100,99,98,97,97,94,92,91,91,90,88,87,85,81,81,80,79,72,70,67,
    67,66,64,63,61,59,58,56,55,51,50,50,50,49,46,41,39,39,38,30,30,
    24,22,21,20,19,14,8,7,5
  };
  const int n1c1w1_p[] = {
    100, // Capacity
    50, // Number of items
    // Size of items (sorted)
    96,94,91,90,82,81,80,77,76,75,74,72,70,68,65,63,63,63,60,60,59,
    58,57,55,51,47,46,36,36,34,32,32,30,30,28,28,27,26,24,24,19,19,
    17,17,11,9,9,7,4,4
  };
  const int n1c1w1_q[] = {
    100, // Capacity
    50, // Number of items
    // Size of items (sorted)
    97,92,90,85,83,83,82,81,77,76,74,73,71,67,67,67,67,63,63,62,59,
    58,58,56,56,55,53,50,47,42,41,41,41,39,37,35,32,31,30,26,25,22,
    20,17,16,15,13,13,10,5
  };
  const int n1c1w1_r[] = {
    100, // Capacity
    50, // Number of items
    // Size of items (sorted)
    95,94,93,92,87,81,81,79,78,76,75,72,72,71,70,65,62,61,60,55,54,
    54,51,49,46,45,38,38,37,36,36,36,32,31,28,27,26,25,24,24,21,20,
    20,17,14,10,9,7,7,3
  };
  const int n1c1w1_s[] = {
    100, // Capacity
    50, // Number of items
    // Size of items (sorted)
    100,99,99,97,96,95,87,87,87,86,84,82,80,80,80,76,75,74,71,68,
    67,63,62,60,52,52,52,48,44,44,43,43,37,34,33,31,29,28,25,21,20,
    17,16,13,11,9,6,5,4,3
  };
  const int n1c1w1_t[] = {
    100, // Capacity
    50, // Number of items
    // Size of items (sorted)
    100,97,92,91,89,88,83,82,82,82,78,77,77,77,73,72,68,67,66,65,
    64,62,60,60,57,53,50,48,46,42,40,40,38,37,37,31,30,29,28,21,20,
    20,20,20,18,18,15,15,11,1
  };
  const int n1c1w2_a[] = {
    100, // Capacity
    50, // Number of items
    // Size of items (sorted)
    96,93,86,86,85,83,80,80,80,79,77,68,67,64,64,63,60,57,55,54,54,
    54,54,52,52,52,51,44,43,41,41,39,39,39,38,36,36,35,34,34,31,31,
    29,29,28,24,23,22,22,20
  };
  const int n1c1w2_b[] = {
    100, // Capacity
    50, // Number of items
    // Size of items (sorted)
    99,96,95,95,91,91,91,90,89,86,85,85,84,79,76,69,68,68,65,64,63,
    58,58,54,53,52,50,49,48,48,45,45,43,42,36,35,33,31,31,30,30,30,
    29,27,27,26,22,22,22,21
  };
  const int n1c1w2_c[] = {
    100, // Capacity
    50, // Number of items
    // Size of items (sorted)
    100,99,98,97,94,93,91,89,89,89,85,85,84,83,81,81,78,73,73,73,
    73,70,69,68,64,64,63,59,54,49,48,45,45,43,42,41,39,37,37,36,32,
    30,26,26,25,24,24,23,21,21
  };
  const int n1c1w2_d[] = {
    100, // Capacity
    50, // Number of items
    // Size of items (sorted)
    97,97,90,89,89,89,85,83,82,81,77,76,76,75,71,71,68,68,66,63,63,
    63,62,61,61,59,58,54,53,50,50,50,46,43,40,36,36,33,32,31,31,31,
    28,27,27,26,26,24,23,22
  };
  const int n1c1w2_e[] = {
    100, // Capacity
    50, // Number of items
    // Size of items (sorted)
    99,96,94,94,90,90,90,90,87,86,85,85,84,84,84,84,84,83,81,81,79,
    71,71,70,65,65,65,63,62,59,51,51,50,49,49,49,47,45,44,43,41,35,
    35,33,31,27,23,23,22,22
  };
  const int n1c1w2_f[] = {
    100, // Capacity
    50, // Number of items
    // Size of items (sorted)
    99,94,94,89,88,86,86,85,84,84,83,79,77,76,74,73,71,71,66,65,63,
    62,60,54,53,50,49,48,48,48,48,43,41,40,40,39,38,35,34,32,31,29,
    28,25,23,23,22,21,20,20
  };
  const int n1c1w2_g[] = {
    100, // Capacity
    50, // Number of items
    // Size of items (sorted)
    100,99,94,91,90,88,86,85,85,83,82,80,79,77,73,71,71,71,67,65,
    65,58,57,57,55,53,52,51,45,40,39,39,38,38,38,37,36,36,35,35,32,
    29,28,27,27,27,24,23,21,20
  };
  const int n1c1w2_h[] = {
    100, // Capacity
    50, // Number of items
    // Size of items (sorted)
    100,100,96,95,95,92,92,92,91,90,90,89,89,86,84,83,81,78,76,73,
    73,73,71,71,67,66,61,60,59,57,54,54,44,42,42,38,36,33,31,31,28,
    28,27,27,27,27,26,25,21,20
  };
  const int n1c1w2_i[] = {
    100, // Capacity
    50, // Number of items
    // Size of items (sorted)
    100,100,98,97,96,94,93,93,85,85,84,83,83,83,82,79,76,76,76,75,
    74,73,73,72,68,66,60,60,56,55,53,52,49,47,46,45,42,41,38,37,37,
    37,36,32,31,31,31,28,24,21
  };
  const int n1c1w2_j[] = {
    100, // Capacity
    50, // Number of items
    // Size of items (sorted)
    100,99,98,95,93,90,87,85,84,84,83,83,81,81,80,79,75,75,71,70,
    68,67,63,63,62,62,61,58,56,51,51,50,49,48,48,42,40,39,37,37,36,
    34,32,30,29,28,28,27,26,26
  };
  const int n1c1w2_k[] = {
    100, // Capacity
    50, // Number of items
    // Size of items (sorted)
    100,99,98,97,97,96,95,94,92,89,89,87,85,77,76,73,71,69,68,68,
    67,66,66,65,64,64,63,62,58,58,52,50,49,48,47,46,44,43,43,35,35,
    32,29,26,26,25,25,23,20,20
  };
  const int n1c1w2_l[] = {
    100, // Capacity
    50, // Number of items
    // Size of items (sorted)
    98,95,94,93,92,91,89,88,87,87,84,82,82,74,73,73,72,69,65,64,63,
    63,62,62,60,59,57,54,54,52,48,47,46,44,43,41,35,33,30,30,30,29,
    29,28,28,27,27,26,24,23
  };
  const int n1c1w2_m[] = {
    100, // Capacity
    50, // Number of items
    // Size of items (sorted)
    99,95,90,89,89,85,82,80,80,79,79,79,77,74,70,70,66,65,65,64,57,
    56,56,55,55,55,53,52,50,49,48,47,45,42,40,37,36,36,36,32,31,31,
    31,31,30,28,28,25,22,20
  };
  const int n1c1w2_n[] = {
    100, // Capacity
    50, // Number of items
    // Size of items (sorted)
    98,96,95,85,84,84,83,82,81,80,78,76,76,74,72,72,71,71,69,66,65,
    64,64,62,61,60,56,53,52,52,49,48,47,45,43,43,42,40,40,40,39,37,
    32,30,28,26,21,21,21,20
  };
  const int n1c1w2_o[] = {
    100, // Capacity
    50, // Number of items
    // Size of items (sorted)
    100,100,100,96,95,93,86,82,82,80,79,75,73,71,71,70,69,69,68,63,
    60,59,58,56,53,52,50,45,44,44,43,42,37,37,36,36,35,31,30,30,29,
    28,28,27,27,22,21,21,20,20
  };
  const int n1c1w2_p[] = {
    100, // Capacity
    50, // Number of items
    // Size of items (sorted)
    100,96,95,95,95,93,92,87,87,83,83,82,79,78,77,76,76,76,72,71,
    69,69,68,64,63,60,57,55,54,54,51,50,46,42,41,40,40,38,38,37,31,
    30,30,29,28,27,26,26,22,20
  };
  const int n1c1w2_q[] = {
    100, // Capacity
    50, // Number of items
    // Size of items (sorted)
    97,96,96,93,93,93,91,88,86,86,85,85,85,82,81,78,75,74,71,71,69,
    67,67,65,65,65,64,61,61,60,58,58,56,54,53,49,45,44,43,40,38,38,
    38,34,33,31,30,26,23,23
  };
  const int n1c1w2_r[] = {
    100, // Capacity
    50, // Number of items
    // Size of items (sorted)
    98,97,97,97,94,91,89,85,84,82,81,80,79,79,75,73,70,69,69,69,68,
    68,68,66,61,55,54,52,52,51,51,49,49,48,47,47,47,45,44,37,37,36,
    35,34,34,30,29,29,27,24
  };
  const int n1c1w2_s[] = {
    100, // Capacity
    50, // Number of items
    // Size of items (sorted)
    99,99,98,96,95,93,92,91,91,91,88,86,84,84,84,80,80,79,78,77,76,
    76,73,72,71,71,69,68,67,64,64,61,59,58,54,52,49,49,41,40,38,31,
    31,29,28,27,27,27,22,20
  };
  const int n1c1w2_t[] = {
    100, // Capacity
    50, // Number of items
    // Size of items (sorted)
    100,100,100,97,96,92,91,91,89,86,85,84,83,83,82,81,79,79,77,74,
    74,73,73,70,68,67,67,65,63,62,62,55,55,52,50,47,45,44,44,44,44,
    43,41,39,37,32,30,26,24,23
  };
  const int n1c1w4_a[] = {
    100, // Capacity
    50, // Number of items
    // Size of items (sorted)
    99,95,93,92,91,89,89,88,88,85,84,84,84,80,80,79,77,76,72,69,65,
    64,64,63,63,60,56,56,53,53,52,51,50,50,49,49,47,44,41,41,40,40,
    40,35,35,34,32,31,31,30
  };
  const int n1c1w4_b[] = {
    100, // Capacity
    50, // Number of items
    // Size of items (sorted)
    100,100,98,97,97,94,92,92,91,85,84,84,83,82,82,80,78,78,78,78,
    75,74,73,72,71,70,70,68,66,65,65,54,50,50,50,49,49,49,47,44,44,
    42,42,41,41,41,40,36,36,30
  };
  const int n1c1w4_c[] = {
    100, // Capacity
    50, // Number of items
    // Size of items (sorted)
    94,92,89,88,88,87,86,84,82,82,81,79,77,77,77,76,73,72,70,69,68,
    68,65,63,63,61,59,58,57,55,54,52,52,52,51,48,46,43,40,38,37,37,
    36,35,35,35,34,34,34,33
  };
  const int n1c1w4_d[] = {
    100, // Capacity
    50, // Number of items
    // Size of items (sorted)
    100,97,95,95,95,95,94,93,93,91,90,89,87,83,82,79,79,78,77,77,
    74,71,69,68,68,65,65,64,61,58,55,55,54,53,53,51,51,49,46,44,42,
    41,39,38,37,37,37,35,33,31
  };
  const int n1c1w4_e[] = {
    100, // Capacity
    50, // Number of items
    // Size of items (sorted)
    100,99,94,92,92,92,89,88,85,83,83,80,79,79,79,79,77,74,74,73,
    71,70,69,68,65,62,62,62,61,61,58,56,56,55,55,55,48,47,46,46,44,
    43,43,43,40,40,36,35,32,30
  };
  const int n1c1w4_f[] = {
    100, // Capacity
    50, // Number of items
    // Size of items (sorted)
    98,98,93,93,92,91,89,86,85,84,80,80,79,78,76,70,68,67,66,62,60,
    59,59,58,58,53,52,52,50,50,49,48,48,48,47,45,43,41,41,40,40,40,
    35,33,32,31,31,30,30,30
  };
  const int n1c1w4_g[] = {
    100, // Capacity
    50, // Number of items
    // Size of items (sorted)
    100,100,100,99,97,95,95,95,93,93,91,90,87,87,86,85,85,84,84,84,
    82,80,77,76,72,70,67,66,65,64,59,56,55,52,48,46,45,44,41,38,37,
    35,35,34,34,33,33,32,32,31
  };
  const int n1c1w4_h[] = {
    100, // Capacity
    50, // Number of items
    // Size of items (sorted)
    100,100,99,98,98,97,96,92,91,91,91,87,86,85,83,83,81,79,78,78,
    75,75,75,74,73,73,70,66,66,65,64,64,63,62,61,60,59,56,55,54,46,
    45,44,41,37,35,34,32,31,30
  };
  const int n1c1w4_i[] = {
    100, // Capacity
    50, // Number of items
    // Size of items (sorted)
    95,92,91,91,90,88,87,87,86,86,85,81,79,76,76,76,72,72,69,65,63,
    63,63,63,61,61,59,59,58,56,54,54,52,51,50,47,47,45,45,45,43,40,
    40,36,35,35,34,32,32,31
  };
  const int n1c1w4_j[] = {
    100, // Capacity
    50, // Number of items
    // Size of items (sorted)
    99,98,93,93,92,90,88,87,87,83,83,81,78,77,77,77,76,75,73,73,71,
    68,66,64,63,63,63,62,60,59,58,54,53,52,52,51,49,47,47,42,42,41,
    40,40,40,39,35,32,32,31
  };
  const int n1c1w4_k[] = {
    100, // Capacity
    50, // Number of items
    // Size of items (sorted)
    100,98,95,94,94,94,93,92,87,85,85,84,83,82,81,78,78,75,73,72,
    71,71,70,70,68,67,67,66,65,64,60,59,58,57,56,56,56,55,55,54,51,
    49,46,45,43,43,43,37,36,35
  };
  const int n1c1w4_l[] = {
    100, // Capacity
    50, // Number of items
    // Size of items (sorted)
    100,99,98,98,97,96,95,91,91,90,88,88,87,86,81,80,79,76,75,67,
    66,65,65,64,60,59,59,58,57,57,55,53,53,50,49,49,49,46,44,43,42,
    38,37,37,36,35,34,34,31,30
  };
  const int n1c1w4_m[] = {
    100, // Capacity
    50, // Number of items
    // Size of items (sorted)
    100,99,99,94,93,92,91,89,88,88,87,80,79,77,75,74,73,71,71,71,
    69,66,64,64,64,63,63,63,62,60,60,59,59,59,55,55,55,53,51,49,49,
    48,46,46,45,42,42,34,33,31
  };
  const int n1c1w4_n[] = {
    100, // Capacity
    50, // Number of items
    // Size of items (sorted)
    99,97,97,96,96,95,94,93,92,90,86,85,85,84,82,82,82,80,79,75,73,
    72,72,71,70,69,69,68,68,66,65,63,61,60,57,55,53,49,48,47,44,41,
    41,39,36,34,32,31,31,31
  };
  const int n1c1w4_o[] = {
    100, // Capacity
    50, // Number of items
    // Size of items (sorted)
    100,90,89,89,89,87,84,81,80,77,77,77,74,71,71,71,67,66,65,63,
    62,61,60,59,59,57,56,56,54,54,51,51,49,48,48,47,47,46,40,39,37,
    36,36,35,34,34,33,32,31,30
  };
  const int n1c1w4_p[] = {
    100, // Capacity
    50, // Number of items
    // Size of items (sorted)
    99,98,95,95,93,93,90,88,87,87,85,83,82,80,79,79,79,77,74,74,73,
    73,72,71,70,66,63,61,61,61,60,60,59,57,55,54,51,48,45,43,42,39,
    39,37,37,36,36,35,32,32
  };
  const int n1c1w4_q[] = {
    100, // Capacity
    50, // Number of items
    // Size of items (sorted)
    95,94,92,91,91,91,90,89,89,84,84,82,79,74,74,74,70,69,68,67,63,
    62,59,59,57,56,56,55,53,52,51,50,50,49,48,48,47,45,43,42,41,41,
    41,40,38,35,35,32,31,30
  };
  const int n1c1w4_r[] = {
    100, // Capacity
    50, // Number of items
    // Size of items (sorted)
    100,99,98,97,95,94,93,93,93,92,92,92,92,85,85,83,81,79,77,76,
    75,73,71,70,70,69,66,63,60,60,59,59,58,58,57,49,48,47,45,42,41,
    41,40,38,38,36,36,35,34,30
  };
  const int n1c1w4_s[] = {
    100, // Capacity
    50, // Number of items
    // Size of items (sorted)
    99,99,98,97,97,94,94,93,91,90,87,87,86,85,85,81,80,78,78,77,76,
    72,66,66,64,59,58,57,57,53,52,50,50,50,48,48,47,46,43,40,39,37,
    37,36,36,35,33,32,30,30
  };
  const int n1c1w4_t[] = {
    100, // Capacity
    50, // Number of items
    // Size of items (sorted)
    98,96,94,87,86,85,83,81,80,79,77,77,76,75,72,70,69,69,69,68,68,
    68,68,67,67,66,65,65,63,62,60,60,60,59,58,56,53,53,52,52,50,50,
    49,45,45,44,39,36,32,30
  };
  const int n1c2w1_a[] = {
    120, // Capacity
    50, // Number of items
    // Size of items (sorted)
    100,97,96,92,89,88,88,87,83,75,75,72,71,70,69,66,63,62,62,61,
    60,58,50,47,46,40,40,37,36,32,31,30,28,27,27,26,24,18,16,14,13,
    12,10,10,10,8,7,5,4,2
  };
  const int n1c2w1_b[] = {
    120, // Capacity
    50, // Number of items
    // Size of items (sorted)
    99,96,96,96,95,95,94,90,90,88,87,84,82,78,77,77,77,75,75,70,70,
    69,68,56,54,53,53,50,50,49,48,47,45,38,36,35,34,28,25,21,19,18,
    16,13,13,7,7,6,3,3
  };
  const int n1c2w1_c[] = {
    120, // Capacity
    50, // Number of items
    // Size of items (sorted)
    100,97,96,92,89,86,83,83,82,79,77,76,73,73,70,69,69,61,60,60,
    60,58,56,56,53,51,49,48,48,48,47,46,42,41,36,35,34,32,32,32,31,
    22,17,12,12,6,6,5,3,2
  };
  const int n1c2w1_d[] = {
    120, // Capacity
    50, // Number of items
    // Size of items (sorted)
    98,96,96,87,87,87,86,85,83,83,82,81,77,74,67,65,64,64,63,60,57,
    57,56,55,50,49,46,43,43,42,37,33,31,31,27,27,26,25,23,23,19,18,
    15,13,10,9,6,3,2,1
  };
  const int n1c2w1_e[] = {
    120, // Capacity
    50, // Number of items
    // Size of items (sorted)
    94,92,89,89,87,82,82,81,80,80,78,71,70,67,66,63,58,52,50,48,46,
    36,34,33,31,30,27,26,21,21,20,19,18,18,17,12,11,11,11,11,10,10,
    7,7,7,6,5,5,4,3
  };
  const int n1c2w1_f[] = {
    120, // Capacity
    50, // Number of items
    // Size of items (sorted)
    99,95,95,94,91,90,89,84,82,81,78,78,77,73,72,69,62,60,59,58,56,
    56,52,52,51,48,48,47,47,45,43,42,38,32,32,31,28,28,28,26,23,21,
    20,18,14,12,8,3,2,1
  };
  const int n1c2w1_g[] = {
    120, // Capacity
    50, // Number of items
    // Size of items (sorted)
    100,100,99,96,96,95,94,90,88,84,81,79,76,70,67,65,60,60,57,57,
    56,52,47,45,44,42,39,37,36,36,35,31,31,28,27,27,25,19,18,17,14,
    14,12,9,9,9,9,3,2,1
  };
  const int n1c2w1_h[] = {
    120, // Capacity
    50, // Number of items
    // Size of items (sorted)
    99,97,94,94,90,90,87,83,82,81,79,77,76,76,75,74,72,67,66,65,63,
    59,59,55,51,50,50,49,47,41,41,39,38,38,37,37,35,34,33,33,21,20,
    18,15,14,9,8,3,1,1
  };
  const int n1c2w1_i[] = {
    120, // Capacity
    50, // Number of items
    // Size of items (sorted)
    100,100,89,89,89,89,88,87,81,78,78,77,76,75,74,73,70,70,69,66,
    66,64,64,64,63,61,60,58,54,52,51,50,49,48,48,48,46,45,45,43,40,
    39,35,34,33,24,9,4,4,1
  };
  const int n1c2w1_j[] = {
    120, // Capacity
    50, // Number of items
    // Size of items (sorted)
    99,98,96,96,95,92,91,89,88,87,86,84,82,82,79,79,78,77,75,72,69,
    66,64,63,61,60,56,55,54,54,49,49,48,44,44,44,41,41,39,27,23,22,
    22,21,15,13,7,5,3,1
  };
  const int n1c2w1_k[] = {
    120, // Capacity
    50, // Number of items
    // Size of items (sorted)
    97,96,96,94,94,91,88,87,85,81,81,77,74,74,74,71,69,68,68,66,65,
    63,60,59,57,57,46,46,45,45,44,43,41,37,35,35,32,30,28,27,25,23,
    23,19,18,16,14,14,10,8
  };
  const int n1c2w1_l[] = {
    120, // Capacity
    50, // Number of items
    // Size of items (sorted)
    98,98,98,97,97,93,92,91,90,89,89,82,82,77,76,75,74,74,73,63,62,
    62,61,60,56,51,49,49,47,47,45,44,43,42,39,37,33,33,32,28,25,21,
    20,19,11,11,6,3,2,1
  };
  const int n1c2w1_m[] = {
    120, // Capacity
    50, // Number of items
    // Size of items (sorted)
    100,99,98,98,95,93,92,89,80,80,78,77,77,73,72,71,71,71,70,70,
    67,66,66,65,64,60,59,53,50,48,48,47,47,45,39,38,37,33,33,28,27,
    19,15,14,14,12,9,9,9,1
  };
  const int n1c2w1_n[] = {
    120, // Capacity
    50, // Number of items
    // Size of items (sorted)
    93,87,85,85,82,79,76,75,70,70,69,69,68,67,66,64,62,61,59,58,58,
    57,56,56,55,53,53,49,45,45,43,42,40,30,30,24,24,22,22,21,20,18,
    18,14,13,11,9,9,6,3
  };
  const int n1c2w1_o[] = {
    120, // Capacity
    50, // Number of items
    // Size of items (sorted)
    99,86,83,83,78,76,68,59,58,58,54,53,53,51,51,48,47,45,43,40,37,
    32,32,32,32,31,31,28,24,22,20,19,19,19,19,15,14,13,12,12,11,10,
    10,10,10,6,5,4,2,1
  };
  const int n1c2w1_p[] = {
    120, // Capacity
    50, // Number of items
    // Size of items (sorted)
    97,96,94,94,93,80,79,78,77,77,76,76,72,72,71,70,67,67,63,60,59,
    55,54,52,51,49,48,47,46,43,34,32,28,27,27,26,25,23,22,20,17,14,
    13,12,12,10,5,4,3,2
  };
  const int n1c2w1_q[] = {
    120, // Capacity
    50, // Number of items
    // Size of items (sorted)
    98,96,95,91,91,90,88,87,83,83,77,74,73,72,72,70,70,67,66,66,63,
    60,59,58,58,57,56,55,54,45,45,41,31,31,29,26,24,21,18,16,16,15,
    14,14,9,9,8,8,6,2
  };
  const int n1c2w1_r[] = {
    120, // Capacity
    50, // Number of items
    // Size of items (sorted)
    100,99,98,96,95,95,92,91,87,85,85,84,78,78,77,76,74,69,68,67,
    65,64,62,55,52,45,43,41,40,38,33,29,27,27,26,24,24,24,23,22,22,
    21,14,13,12,10,8,2,1,1
  };
  const int n1c2w1_s[] = {
    120, // Capacity
    50, // Number of items
    // Size of items (sorted)
    97,93,92,90,87,83,82,82,80,80,78,78,72,71,68,67,63,62,60,59,56,
    56,55,54,54,51,50,48,46,45,42,41,35,32,32,28,26,25,25,25,24,22,
    21,21,14,12,10,9,9,7
  };
  const int n1c2w1_t[] = {
    120, // Capacity
    50, // Number of items
    // Size of items (sorted)
    100,93,93,89,89,87,81,81,79,78,77,70,68,67,66,66,65,64,62,61,
    60,57,53,53,52,52,52,48,44,44,43,43,42,41,39,39,37,35,34,30,30,
    29,26,25,16,16,10,10,7,6
  };
  const int n1c2w2_a[] = {
    120, // Capacity
    50, // Number of items
    // Size of items (sorted)
    100,97,97,95,93,87,87,86,82,82,78,76,76,75,74,71,68,66,65,63,
    59,59,58,58,57,52,51,46,46,46,43,42,42,41,41,41,38,37,36,36,32,
    32,31,30,27,25,22,22,22,21
  };
  const int n1c2w2_b[] = {
    120, // Capacity
    50, // Number of items
    // Size of items (sorted)
    100,98,98,97,95,94,90,90,89,86,85,83,81,79,79,74,72,72,71,68,
    67,65,64,64,62,59,58,56,55,55,54,51,51,50,47,46,45,44,43,40,36,
    34,33,31,29,28,27,27,26,21
  };
  const int n1c2w2_c[] = {
    120, // Capacity
    50, // Number of items
    // Size of items (sorted)
    100,98,97,95,93,91,90,87,85,83,83,81,81,79,76,74,74,73,73,71,
    71,70,67,67,66,62,62,60,57,54,54,53,52,51,51,50,49,48,48,45,44,
    44,40,36,34,32,31,27,26,20
  };
  const int n1c2w2_d[] = {
    120, // Capacity
    50, // Number of items
    // Size of items (sorted)
    99,98,98,97,96,90,88,86,82,82,80,79,76,76,76,74,69,67,66,64,62,
    59,55,52,51,51,50,49,44,43,41,41,41,41,41,37,35,33,32,32,31,31,
    31,30,29,23,23,22,20,20
  };
  const int n1c2w2_e[] = {
    120, // Capacity
    50, // Number of items
    // Size of items (sorted)
    100,99,99,99,99,98,98,94,93,92,92,89,89,89,84,83,80,80,78,77,
    75,74,74,70,70,68,68,66,63,62,60,59,58,58,58,55,54,53,52,49,42,
    41,36,35,35,31,26,23,22,20
  };
  const int n1c2w2_f[] = {
    120, // Capacity
    50, // Number of items
    // Size of items (sorted)
    100,100,99,99,98,91,90,84,83,81,78,78,75,73,72,72,71,70,68,66,
    62,59,58,58,57,54,53,53,51,51,51,51,48,45,45,42,42,39,37,37,35,
    32,31,31,26,26,25,21,21,20
  };
  const int n1c2w2_g[] = {
    120, // Capacity
    50, // Number of items
    // Size of items (sorted)
    100,97,94,93,93,91,89,89,86,85,85,82,81,80,80,80,80,79,77,75,
    74,72,67,67,63,62,59,58,58,57,54,54,53,51,48,47,46,44,44,41,41,
    39,36,35,33,32,32,29,28,24
  };
  const int n1c2w2_h[] = {
    120, // Capacity
    50, // Number of items
    // Size of items (sorted)
    99,98,93,93,91,88,85,82,80,78,76,70,68,67,66,65,61,61,57,56,56,
    53,52,52,52,51,48,47,46,44,43,43,43,41,41,41,37,37,36,36,35,33,
    33,32,31,27,26,22,22,21
  };
  const int n1c2w2_i[] = {
    120, // Capacity
    50, // Number of items
    // Size of items (sorted)
    96,92,92,91,91,90,89,88,83,83,81,79,77,76,76,71,70,68,68,66,63,
    63,63,62,60,60,58,57,53,53,52,52,49,47,45,44,41,38,37,34,33,32,
    31,29,27,26,25,23,21,21
  };
  const int n1c2w2_j[] = {
    120, // Capacity
    50, // Number of items
    // Size of items (sorted)
    100,98,96,95,95,93,91,89,89,88,88,81,80,78,73,72,69,67,64,61,
    60,54,52,52,51,50,50,49,49,47,46,44,43,42,41,40,40,39,36,33,33,
    28,26,26,25,23,22,22,22,20
  };
  const int n1c2w2_k[] = {
    120, // Capacity
    50, // Number of items
    // Size of items (sorted)
    97,97,95,91,91,89,85,85,82,82,81,75,74,73,70,70,70,69,68,67,67,
    67,65,63,63,63,62,61,60,60,55,48,46,45,45,45,45,44,43,43,42,41,
    39,37,36,30,28,22,22,22
  };
  const int n1c2w2_l[] = {
    120, // Capacity
    50, // Number of items
    // Size of items (sorted)
    96,95,93,92,90,87,87,86,86,86,85,84,83,82,78,78,78,78,77,76,76,
    72,72,71,70,68,65,65,62,59,58,51,42,42,40,38,38,36,34,34,33,32,
    30,29,29,27,26,25,24,23
  };
  const int n1c2w2_m[] = {
    120, // Capacity
    50, // Number of items
    // Size of items (sorted)
    100,99,99,99,97,95,95,94,93,92,92,88,86,86,86,84,79,78,78,77,
    76,69,68,65,61,60,58,57,57,55,54,54,53,53,52,52,51,48,47,43,43,
    40,39,38,36,34,33,28,27,25
  };
  const int n1c2w2_n[] = {
    120, // Capacity
    50, // Number of items
    // Size of items (sorted)
    99,97,95,94,88,87,85,83,82,78,75,72,71,71,70,69,67,67,65,64,63,
    62,59,59,58,58,58,58,58,54,53,53,52,49,49,48,45,45,44,43,43,42,
    40,38,36,34,30,30,24,20
  };
  const int n1c2w2_o[] = {
    120, // Capacity
    50, // Number of items
    // Size of items (sorted)
    100,99,98,96,94,90,89,88,88,86,84,81,81,80,79,79,78,76,72,72,
    72,68,68,65,63,63,63,62,62,57,57,55,48,48,47,45,44,44,41,39,36,
    33,31,30,28,26,25,24,22,20
  };
  const int n1c2w2_p[] = {
    120, // Capacity
    50, // Number of items
    // Size of items (sorted)
    94,93,91,90,90,88,87,82,77,75,72,71,70,70,69,69,66,65,63,59,57,
    56,53,51,48,48,48,47,44,44,43,42,41,40,39,38,37,36,36,32,31,31,
    29,29,27,23,23,21,20,20
  };
  const int n1c2w2_q[] = {
    120, // Capacity
    50, // Number of items
    // Size of items (sorted)
    96,96,91,90,89,86,86,84,83,83,82,82,82,82,79,75,73,72,71,69,68,
    67,67,66,65,63,62,61,59,59,59,59,58,56,56,55,54,53,50,45,41,39,
    35,33,29,25,24,21,20,20
  };
  const int n1c2w2_r[] = {
    120, // Capacity
    50, // Number of items
    // Size of items (sorted)
    99,98,96,91,88,88,86,86,82,82,81,78,77,77,76,76,72,72,70,68,67,
    64,61,60,59,56,55,49,48,47,47,46,44,43,43,42,40,40,39,38,35,34,
    30,30,29,27,26,21,20,20
  };
  const int n1c2w2_s[] = {
    120, // Capacity
    50, // Number of items
    // Size of items (sorted)
    100,94,94,92,91,87,87,85,82,78,76,75,72,72,72,69,61,61,61,61,
    61,56,55,54,53,51,51,50,47,44,44,44,44,42,42,39,38,36,34,33,33,
    32,31,30,29,28,26,25,23,23
  };
  const int n1c2w2_t[] = {
    120, // Capacity
    50, // Number of items
    // Size of items (sorted)
    100,96,96,91,84,83,83,83,81,81,80,80,77,77,72,70,70,68,68,67,
    65,64,63,62,60,59,58,51,51,50,49,47,47,47,46,45,43,43,41,38,37,
    36,35,31,31,29,28,27,26,20
  };
  const int n1c2w4_a[] = {
    120, // Capacity
    50, // Number of items
    // Size of items (sorted)
    100,99,97,97,96,96,95,92,92,90,90,88,87,87,85,84,83,82,81,79,
    74,68,68,63,59,58,56,55,55,51,50,49,49,49,47,44,44,42,39,37,37,
    34,34,34,33,33,31,30,30,30
  };
  const int n1c2w4_b[] = {
    120, // Capacity
    50, // Number of items
    // Size of items (sorted)
    99,96,94,93,93,91,87,87,87,84,84,83,83,83,83,83,82,81,81,78,77,
    77,77,76,67,65,61,61,59,58,53,53,50,49,48,47,47,46,46,44,43,42,
    41,41,38,35,34,32,32,31
  };
  const int n1c2w4_c[] = {
    120, // Capacity
    50, // Number of items
    // Size of items (sorted)
    100,100,99,96,96,93,91,90,90,87,84,83,80,80,80,75,74,72,72,71,
    71,70,69,66,65,63,60,58,57,56,54,54,53,53,53,51,51,49,46,43,40,
    39,38,37,37,34,33,33,31,31
  };
  const int n1c2w4_d[] = {
    120, // Capacity
    50, // Number of items
    // Size of items (sorted)
    97,97,96,94,93,91,89,89,86,83,79,78,77,77,77,75,75,74,71,68,68,
    67,65,63,61,61,58,57,56,54,48,46,44,43,41,41,40,38,36,36,35,35,
    35,35,35,34,33,33,33,31
  };
  const int n1c2w4_e[] = {
    120, // Capacity
    50, // Number of items
    // Size of items (sorted)
    100,99,99,97,97,96,96,96,93,93,91,84,83,81,79,78,77,74,71,67,
    66,63,62,61,61,61,59,59,59,58,57,56,54,54,53,53,51,50,49,48,45,
    45,45,40,40,39,39,34,32,30
  };
  const int n1c2w4_f[] = {
    120, // Capacity
    50, // Number of items
    // Size of items (sorted)
    99,98,98,97,96,93,88,86,86,85,85,81,80,80,77,76,74,73,73,72,69,
    69,67,66,66,65,64,63,63,62,60,59,59,59,54,54,51,49,49,46,43,43,
    38,38,38,38,36,36,35,33
  };
  const int n1c2w4_g[] = {
    120, // Capacity
    50, // Number of items
    // Size of items (sorted)
    100,99,99,97,95,93,91,91,90,90,88,88,87,86,82,80,79,75,70,69,
    68,66,66,64,62,62,61,60,60,57,56,55,53,51,47,46,44,42,38,37,36,
    36,36,36,35,35,32,32,31,31
  };
  const int n1c2w4_h[] = {
    120, // Capacity
    50, // Number of items
    // Size of items (sorted)
    99,98,97,95,94,93,93,93,92,91,91,89,86,85,81,77,74,70,69,68,67,
    66,66,65,63,62,61,60,59,58,57,57,56,56,52,50,49,48,47,43,43,43,
    40,39,37,36,36,35,30,30
  };
  const int n1c2w4_i[] = {
    120, // Capacity
    50, // Number of items
    // Size of items (sorted)
    97,92,91,88,87,86,85,85,84,84,84,83,80,80,79,78,76,76,76,76,75,
    75,75,74,74,74,72,71,71,70,67,63,59,59,57,55,55,54,50,49,44,42,
    40,38,37,35,31,31,30,30
  };
  const int n1c2w4_j[] = {
    120, // Capacity
    50, // Number of items
    // Size of items (sorted)
    100,97,96,90,86,84,83,82,79,78,76,74,72,70,70,70,68,68,67,67,
    66,66,66,65,64,64,63,63,62,59,57,57,57,55,54,54,51,49,48,47,43,
    41,40,40,37,37,34,33,32,32
  };
  const int n1c2w4_k[] = {
    120, // Capacity
    50, // Number of items
    // Size of items (sorted)
    100,100,100,99,98,93,91,89,88,87,84,82,80,80,78,78,77,77,77,76,
    75,75,73,71,71,70,65,61,61,60,59,58,58,55,53,52,51,49,49,44,43,
    42,40,40,40,39,38,38,32,32
  };
  const int n1c2w4_l[] = {
    120, // Capacity
    50, // Number of items
    // Size of items (sorted)
    99,99,98,98,94,93,92,90,90,89,89,88,84,81,79,78,77,77,76,75,74,
    72,72,70,69,66,64,63,60,57,57,56,54,52,47,45,43,43,43,41,40,39,
    39,38,37,37,36,35,34,30
  };
  const int n1c2w4_m[] = {
    120, // Capacity
    50, // Number of items
    // Size of items (sorted)
    99,99,99,97,95,94,92,91,90,90,90,90,88,83,79,78,78,76,76,70,68,
    67,66,63,62,62,61,60,58,58,58,58,56,56,55,54,53,51,50,48,48,47,
    42,37,37,37,36,32,31,30
  };
  const int n1c2w4_n[] = {
    120, // Capacity
    50, // Number of items
    // Size of items (sorted)
    98,96,93,92,91,91,91,90,90,90,89,89,88,88,84,82,77,76,76,75,74,
    73,72,69,69,66,65,59,59,58,57,56,54,53,52,52,51,51,49,48,47,47,
    46,42,41,40,39,36,35,33
  };
  const int n1c2w4_o[] = {
    120, // Capacity
    50, // Number of items
    // Size of items (sorted)
    100,97,94,93,91,91,86,84,83,78,78,78,77,77,77,77,75,74,74,73,
    71,69,68,64,64,62,62,61,57,54,54,53,50,49,49,48,47,47,47,46,45,
    45,44,44,42,40,39,35,35,35
  };
  const int n1c2w4_p[] = {
    120, // Capacity
    50, // Number of items
    // Size of items (sorted)
    98,98,95,95,93,91,91,89,89,87,83,83,82,78,77,76,75,74,72,67,62,
    61,59,57,55,55,54,52,50,49,49,48,47,47,45,45,44,44,43,43,42,40,
    39,39,38,37,36,33,33,31
  };
  const int n1c2w4_q[] = {
    120, // Capacity
    50, // Number of items
    // Size of items (sorted)
    100,98,98,98,91,90,90,88,87,87,87,86,86,83,82,81,80,80,76,73,
    72,71,71,70,69,68,68,67,67,66,65,64,60,54,53,52,52,47,46,46,46,
    41,40,37,37,36,36,35,34,33
  };
  const int n1c2w4_r[] = {
    120, // Capacity
    50, // Number of items
    // Size of items (sorted)
    100,99,99,98,95,95,95,94,90,87,87,86,85,85,83,82,80,79,79,76,
    73,73,72,71,70,69,69,68,68,66,65,63,63,62,58,57,56,55,54,53,52,
    49,47,46,46,43,42,35,34,31
  };
  const int n1c2w4_s[] = {
    120, // Capacity
    50, // Number of items
    // Size of items (sorted)
    98,98,93,93,93,92,92,92,92,90,89,86,86,85,85,84,83,83,83,81,81,
    78,77,77,75,74,71,70,70,68,66,66,65,65,63,62,61,61,59,57,50,50,
    49,49,47,44,40,32,31,30
  };
  const int n1c2w4_t[] = {
    120, // Capacity
    50, // Number of items
    // Size of items (sorted)
    97,95,91,89,88,87,86,83,82,82,81,73,73,69,69,68,68,68,65,62,61,
    60,60,60,58,58,58,56,55,54,54,52,51,51,51,49,49,47,45,44,43,42,
    42,41,41,40,36,33,30,30
  };
  const int n1c3w1_a[] = {
    150, // Capacity
    50, // Number of items
    // Size of items (sorted)
    100,100,96,94,90,88,87,85,83,81,80,80,77,74,65,62,62,62,61,59,
    59,57,54,51,45,45,40,38,37,37,37,36,29,29,27,26,22,22,21,17,14,
    14,8,7,6,5,5,3,3,1
  };
  const int n1c3w1_b[] = {
    150, // Capacity
    50, // Number of items
    // Size of items (sorted)
    95,88,88,86,85,84,84,82,81,79,72,71,69,69,69,68,68,65,61,61,61,
    61,60,58,57,57,53,44,43,36,29,29,27,23,23,22,21,17,14,14,14,13,
    12,11,11,6,5,3,3,2
  };
  const int n1c3w1_c[] = {
    150, // Capacity
    50, // Number of items
    // Size of items (sorted)
    100,99,95,94,87,85,85,83,81,81,80,80,77,76,75,74,73,73,72,66,
    63,60,52,50,47,45,44,43,39,39,38,38,35,34,33,32,25,25,23,20,17,
    15,15,14,12,11,10,10,8,8
  };
  const int n1c3w1_d[] = {
    150, // Capacity
    50, // Number of items
    // Size of items (sorted)
    99,96,95,95,92,91,90,86,86,86,85,80,77,77,76,76,71,70,70,69,68,
    64,64,61,60,60,56,55,53,52,50,48,44,41,40,38,38,37,35,21,19,14,
    12,9,6,6,6,4,3,2
  };
  const int n1c3w1_e[] = {
    150, // Capacity
    50, // Number of items
    // Size of items (sorted)
    99,97,97,96,95,89,88,83,81,81,79,77,76,75,74,61,55,51,50,50,48,
    48,47,46,45,42,42,38,35,34,32,32,31,26,25,21,14,13,11,10,9,9,
    9,8,8,7,5,5,5,1
  };
  const int n1c3w1_f[] = {
    150, // Capacity
    50, // Number of items
    // Size of items (sorted)
    100,98,97,96,95,93,92,88,88,86,84,83,80,80,78,77,76,76,76,74,
    73,70,69,68,65,64,63,62,62,61,60,60,53,51,51,42,41,28,26,23,22,
    21,16,13,9,9,7,5,2,2
  };
  const int n1c3w1_g[] = {
    150, // Capacity
    50, // Number of items
    // Size of items (sorted)
    97,92,91,91,88,86,85,84,79,76,75,67,66,65,62,61,61,58,54,54,50,
    47,46,45,44,44,42,37,37,30,27,27,26,23,23,21,20,20,19,13,12,11,
    10,9,9,6,5,5,5,1
  };
  const int n1c3w1_h[] = {
    150, // Capacity
    50, // Number of items
    // Size of items (sorted)
    99,91,89,89,89,88,86,85,83,82,80,80,80,80,78,76,73,69,67,66,65,
    65,64,64,60,60,57,56,56,52,51,45,43,42,42,38,37,32,32,32,29,28,
    26,25,18,15,10,6,6,4
  };
  const int n1c3w1_i[] = {
    150, // Capacity
    50, // Number of items
    // Size of items (sorted)
    100,98,97,95,87,87,87,84,80,77,76,73,71,66,66,62,61,60,60,60,
    57,56,53,52,51,49,46,44,44,43,43,38,33,31,30,29,29,28,24,22,18,
    17,16,16,16,15,12,8,3,2
  };
  const int n1c3w1_j[] = {
    150, // Capacity
    50, // Number of items
    // Size of items (sorted)
    99,98,92,91,90,88,87,86,82,80,77,74,73,72,72,71,69,69,63,61,55,
    54,53,50,48,48,48,37,37,37,34,33,32,29,26,22,19,17,15,14,10,9,
    7,3,3,2,2,2,1,1
  };
  const int n1c3w1_k[] = {
    150, // Capacity
    50, // Number of items
    // Size of items (sorted)
    100,96,95,94,94,92,92,90,86,84,77,73,66,66,59,56,56,56,55,54,
    53,53,53,52,49,48,47,45,45,45,41,41,41,37,36,24,22,21,20,18,16,
    15,14,14,13,12,10,8,4,1
  };
  const int n1c3w1_l[] = {
    150, // Capacity
    50, // Number of items
    // Size of items (sorted)
    99,99,93,93,90,90,87,87,81,81,80,78,77,76,68,64,63,62,60,60,59,
    58,53,52,52,47,45,44,44,42,39,39,36,35,29,29,28,26,25,18,9,7,
    7,7,7,6,5,5,5,1
  };
  const int n1c3w1_m[] = {
    150, // Capacity
    50, // Number of items
    // Size of items (sorted)
    100,100,99,94,90,88,88,86,86,84,84,80,77,73,70,69,69,66,66,61,
    58,58,57,57,52,51,47,44,43,42,36,34,28,27,26,25,21,18,18,17,13,
    12,12,12,11,9,8,7,4,4
  };
  const int n1c3w1_n[] = {
    150, // Capacity
    50, // Number of items
    // Size of items (sorted)
    98,97,91,90,90,90,88,87,87,85,83,81,79,78,78,76,74,74,73,72,68,
    66,64,63,61,57,56,56,56,55,55,48,48,46,44,44,39,37,35,35,34,32,
    31,29,27,26,19,18,17,11
  };
  const int n1c3w1_o[] = {
    150, // Capacity
    50, // Number of items
    // Size of items (sorted)
    96,96,96,94,94,87,86,84,84,83,82,82,80,77,75,57,57,56,55,54,52,
    51,48,48,48,46,46,45,42,34,34,34,32,32,30,23,16,16,16,15,15,14,
    12,10,6,6,3,1,1,1
  };
  const int n1c3w1_p[] = {
    150, // Capacity
    50, // Number of items
    // Size of items (sorted)
    99,99,98,98,96,93,93,92,91,89,85,82,80,79,78,73,73,71,70,69,69,
    61,61,55,54,52,47,47,46,43,43,42,41,38,36,35,34,28,27,25,24,21,
    17,13,10,9,6,5,5,2
  };
  const int n1c3w1_q[] = {
    150, // Capacity
    50, // Number of items
    // Size of items (sorted)
    100,100,100,100,98,96,95,93,90,89,86,86,85,85,84,81,79,78,74,
    70,69,68,66,62,62,61,58,56,55,54,53,51,48,44,42,40,36,35,33,32,
    31,24,23,23,18,13,12,4,4,2
  };
  const int n1c3w1_r[] = {
    150, // Capacity
    50, // Number of items
    // Size of items (sorted)
    100,99,97,97,97,95,94,91,88,87,87,86,86,86,82,77,77,75,74,73,
    72,71,70,65,63,62,60,59,56,56,51,50,50,49,49,47,47,46,36,29,23,
    23,21,20,18,16,13,11,9,3
  };
  const int n1c3w1_s[] = {
    150, // Capacity
    50, // Number of items
    // Size of items (sorted)
    95,90,88,87,86,83,79,78,76,75,71,70,70,68,64,63,63,61,59,58,57,
    57,53,52,52,49,44,40,36,36,32,29,25,23,23,22,22,20,19,19,19,17,
    16,11,11,7,6,5,3,2
  };
  const int n1c3w1_t[] = {
    150, // Capacity
    50, // Number of items
    // Size of items (sorted)
    98,98,97,96,93,93,92,89,83,82,76,76,76,74,70,69,67,66,66,65,62,
    60,58,56,56,55,55,54,53,51,49,47,42,35,31,31,26,22,22,22,18,17,
    17,17,16,9,8,5,4,4
  };
  const int n1c3w2_a[] = {
    150, // Capacity
    50, // Number of items
    // Size of items (sorted)
    100,96,94,93,91,91,91,88,84,83,80,78,78,76,75,74,72,72,70,65,
    61,60,56,52,51,51,48,46,45,38,38,37,37,37,36,35,35,32,32,31,30,
    29,29,28,27,27,23,23,22,21
  };
  const int n1c3w2_b[] = {
    150, // Capacity
    50, // Number of items
    // Size of items (sorted)
    98,96,95,94,92,89,88,88,87,87,86,85,83,80,80,77,76,76,73,72,71,
    69,69,69,57,57,53,50,45,45,44,44,43,42,37,36,36,35,35,34,33,31,
    30,27,24,24,23,21,20,20
  };
  const int n1c3w2_c[] = {
    150, // Capacity
    50, // Number of items
    // Size of items (sorted)
    98,98,96,95,94,93,92,91,89,88,88,88,86,83,83,82,80,79,78,76,76,
    75,73,67,63,63,62,55,54,53,52,51,51,51,47,45,45,42,42,40,37,37,
    36,36,29,29,25,24,20,20
  };
  const int n1c3w2_d[] = {
    150, // Capacity
    50, // Number of items
    // Size of items (sorted)
    100,99,98,96,94,92,90,89,89,89,87,86,81,80,78,77,74,74,72,72,
    63,62,60,60,55,55,54,53,50,50,46,46,45,42,42,41,38,35,34,33,33,
    32,28,28,27,26,23,21,21,20
  };
  const int n1c3w2_e[] = {
    150, // Capacity
    50, // Number of items
    // Size of items (sorted)
    100,100,99,96,95,94,92,92,90,89,89,84,82,80,80,79,74,74,72,71,
    69,67,67,64,62,60,60,59,58,55,51,48,47,46,45,43,42,41,41,40,38,
    34,33,32,27,26,24,24,23,20
  };
  const int n1c3w2_f[] = {
    150, // Capacity
    50, // Number of items
    // Size of items (sorted)
    100,99,99,98,97,96,93,91,89,86,85,82,78,76,75,74,73,71,68,68,
    66,65,65,64,63,63,63,63,63,62,60,59,56,55,55,53,51,50,48,45,43,
    43,42,42,39,39,35,31,27,26
  };
  const int n1c3w2_g[] = {
    150, // Capacity
    50, // Number of items
    // Size of items (sorted)
    98,98,98,96,93,93,92,91,90,90,87,87,86,85,83,82,81,78,78,75,75,
    74,74,72,72,71,70,69,68,66,61,60,60,59,57,53,51,42,40,40,35,34,
    34,31,30,30,24,22,21,20
  };
  const int n1c3w2_h[] = {
    150, // Capacity
    50, // Number of items
    // Size of items (sorted)
    99,98,98,97,97,95,94,93,91,91,88,87,82,80,80,79,79,79,75,74,73,
    72,71,69,68,66,63,63,61,60,58,58,55,54,53,53,52,50,46,45,44,42,
    40,38,37,35,29,24,24,20
  };
  const int n1c3w2_i[] = {
    150, // Capacity
    50, // Number of items
    // Size of items (sorted)
    96,95,91,89,87,86,85,81,78,78,68,67,66,66,65,62,61,60,60,59,58,
    56,54,51,50,50,49,49,49,48,47,46,46,46,45,45,44,41,41,41,40,36,
    35,34,33,32,31,27,26,26
  };
  const int n1c3w2_j[] = {
    150, // Capacity
    50, // Number of items
    // Size of items (sorted)
    99,96,95,95,94,93,93,92,91,91,90,89,87,86,86,84,81,80,73,68,66,
    64,62,61,61,59,59,56,55,54,49,48,48,47,46,45,45,43,42,41,41,40,
    39,37,36,34,32,26,24,20
  };
  const int n1c3w2_k[] = {
    150, // Capacity
    50, // Number of items
    // Size of items (sorted)
    95,94,93,93,91,89,89,89,88,85,82,82,78,78,77,76,73,73,73,70,70,
    70,70,69,68,66,63,62,59,55,55,53,51,49,42,42,41,41,40,38,35,32,
    31,30,30,28,28,24,23,23
  };
  const int n1c3w2_l[] = {
    150, // Capacity
    50, // Number of items
    // Size of items (sorted)
    99,99,98,98,97,95,92,92,87,85,84,83,80,78,77,75,73,73,69,68,66,
    63,63,63,59,57,56,56,53,53,51,50,50,48,48,46,46,44,43,42,39,37,
    34,32,29,25,24,22,22,21
  };
  const int n1c3w2_m[] = {
    150, // Capacity
    50, // Number of items
    // Size of items (sorted)
    100,99,96,94,92,91,91,89,85,84,81,81,79,79,78,77,76,75,74,73,
    67,65,64,63,63,59,57,57,54,52,51,49,49,47,46,46,44,44,43,43,40,
    38,34,33,32,31,30,29,25,22
  };
  const int n1c3w2_n[] = {
    150, // Capacity
    50, // Number of items
    // Size of items (sorted)
    98,95,95,91,91,89,89,88,88,87,86,84,83,82,80,79,78,75,74,74,73,
    72,72,70,70,68,68,67,65,59,58,58,57,55,54,53,51,42,41,39,37,36,
    35,34,32,25,25,21,21,20
  };
  const int n1c3w2_o[] = {
    150, // Capacity
    50, // Number of items
    // Size of items (sorted)
    99,99,96,93,88,83,82,80,79,79,77,77,75,75,73,73,72,71,71,71,71,
    69,69,67,62,62,61,58,58,56,54,53,52,49,46,45,45,41,40,39,35,35,
    34,33,31,27,27,26,22,21
  };
  const int n1c3w2_p[] = {
    150, // Capacity
    50, // Number of items
    // Size of items (sorted)
    95,94,88,88,88,86,85,84,83,79,73,72,72,72,71,70,64,63,61,58,55,
    53,53,52,51,51,51,48,48,46,45,40,39,38,36,36,35,33,32,28,25,24,
    24,23,23,23,22,22,20,20
  };
  const int n1c3w2_q[] = {
    150, // Capacity
    50, // Number of items
    // Size of items (sorted)
    96,91,87,86,84,83,83,83,81,80,79,74,72,70,70,67,62,61,60,59,58,
    56,55,55,54,52,51,51,51,50,49,48,44,43,43,42,40,39,38,34,34,34,
    33,32,31,31,29,29,22,21
  };
  const int n1c3w2_r[] = {
    150, // Capacity
    50, // Number of items
    // Size of items (sorted)
    100,98,91,87,82,78,77,77,77,75,75,74,72,72,72,70,70,66,66,65,
    63,63,62,59,57,56,55,53,52,51,49,48,47,46,46,44,44,42,36,35,34,
    34,31,30,29,26,23,22,21,20
  };
  const int n1c3w2_s[] = {
    150, // Capacity
    50, // Number of items
    // Size of items (sorted)
    100,99,97,96,96,95,94,91,90,88,85,83,83,81,79,79,78,77,77,74,
    72,70,69,66,64,63,63,61,58,56,52,51,45,42,36,36,36,35,34,33,32,
    32,31,30,28,25,24,21,21,20
  };
  const int n1c3w2_t[] = {
    150, // Capacity
    50, // Number of items
    // Size of items (sorted)
    100,99,96,95,93,91,91,88,87,87,85,85,85,84,83,83,78,77,76,75,
    74,70,67,65,63,63,62,60,60,58,56,55,55,54,52,50,49,49,45,42,29,
    29,27,27,26,25,24,23,22,20
  };
  const int n1c3w4_a[] = {
    150, // Capacity
    50, // Number of items
    // Size of items (sorted)
    97,95,92,91,90,90,86,85,85,82,82,81,80,79,78,76,71,70,69,67,63,
    63,63,62,58,58,56,55,54,53,52,51,51,48,47,46,44,44,42,42,41,40,
    39,39,37,35,34,32,31,31
  };
  const int n1c3w4_b[] = {
    150, // Capacity
    50, // Number of items
    // Size of items (sorted)
    100,98,97,97,92,92,92,91,88,84,83,82,77,77,76,75,74,73,72,70,
    70,67,66,65,63,62,62,62,62,58,57,57,54,53,52,52,50,46,45,43,42,
    41,41,41,40,37,37,36,33,33
  };
  const int n1c3w4_c[] = {
    150, // Capacity
    50, // Number of items
    // Size of items (sorted)
    99,99,95,94,92,91,90,87,86,84,83,82,82,81,81,81,80,80,78,78,78,
    77,77,74,72,71,69,68,66,66,64,63,62,62,61,60,57,55,52,52,46,46,
    45,45,42,39,39,38,35,32
  };
  const int n1c3w4_d[] = {
    150, // Capacity
    50, // Number of items
    // Size of items (sorted)
    100,96,93,90,88,88,86,85,84,84,83,83,80,80,79,77,77,74,70,68,
    67,64,61,61,58,58,58,56,54,54,53,51,49,48,47,45,45,44,43,41,41,
    40,40,37,36,34,34,33,33,31
  };
  const int n1c3w4_e[] = {
    150, // Capacity
    50, // Number of items
    // Size of items (sorted)
    98,97,96,95,95,94,93,93,93,93,91,90,87,87,80,80,80,77,72,71,68,
    68,67,64,63,62,60,60,60,57,57,56,54,53,53,52,49,47,45,43,41,41,
    39,38,38,37,37,36,35,31
  };
  const int n1c3w4_f[] = {
    150, // Capacity
    50, // Number of items
    // Size of items (sorted)
    95,92,92,89,88,87,85,84,83,82,82,81,81,81,76,76,73,72,69,68,68,
    67,65,65,63,63,61,61,57,56,54,54,54,52,50,50,49,47,46,40,40,39,
    39,39,37,37,34,33,32,30
  };
  const int n1c3w4_g[] = {
    150, // Capacity
    50, // Number of items
    // Size of items (sorted)
    99,99,97,97,96,92,90,88,87,87,87,86,86,85,85,83,81,79,78,77,77,
    74,73,73,73,72,68,65,62,58,56,55,55,55,52,52,51,50,49,46,42,40,
    39,38,37,36,36,33,31,31
  };
  const int n1c3w4_h[] = {
    150, // Capacity
    50, // Number of items
    // Size of items (sorted)
    100,100,99,97,95,94,92,90,88,87,86,85,83,80,79,78,78,78,75,75,
    74,73,71,70,69,67,65,64,59,58,57,57,55,54,54,52,51,50,49,48,46,
    46,45,43,43,42,39,38,33,32
  };
  const int n1c3w4_i[] = {
    150, // Capacity
    50, // Number of items
    // Size of items (sorted)
    99,98,95,89,88,88,87,87,87,87,86,84,84,83,78,77,74,74,73,73,73,
    72,72,70,68,67,64,64,64,63,63,60,59,58,56,54,51,50,49,49,39,37,
    37,36,36,36,34,34,31,30
  };
  const int n1c3w4_j[] = {
    150, // Capacity
    50, // Number of items
    // Size of items (sorted)
    100,93,91,91,89,89,88,86,85,84,83,83,82,80,79,78,77,76,76,73,
    72,68,68,63,63,61,60,60,58,57,57,56,54,53,52,50,48,47,47,45,41,
    41,36,35,34,34,33,31,31,30
  };
  const int n1c3w4_k[] = {
    150, // Capacity
    50, // Number of items
    // Size of items (sorted)
    100,97,96,94,94,93,90,89,89,86,85,84,83,83,83,82,80,78,75,74,
    72,72,71,70,69,69,66,64,64,63,62,60,59,59,58,57,57,57,57,56,50,
    50,47,44,43,41,37,36,35,33
  };
  const int n1c3w4_l[] = {
    150, // Capacity
    50, // Number of items
    // Size of items (sorted)
    100,100,93,91,88,86,86,84,83,75,75,75,75,75,73,72,70,69,67,66,
    66,65,61,58,56,55,55,54,52,51,51,51,50,47,45,44,42,42,41,40,39,
    36,35,35,33,33,33,32,31,30
  };
  const int n1c3w4_m[] = {
    150, // Capacity
    50, // Number of items
    // Size of items (sorted)
    99,98,97,95,90,87,87,85,85,83,80,80,76,71,71,70,69,68,67,66,65,
    63,63,62,62,60,60,60,58,56,55,53,50,49,45,42,42,41,38,36,36,34,
    34,33,32,32,31,31,31,30
  };
  const int n1c3w4_n[] = {
    150, // Capacity
    50, // Number of items
    // Size of items (sorted)
    100,92,91,90,89,85,84,81,80,80,78,78,77,77,76,75,74,73,69,69,
    68,68,67,67,65,64,63,63,61,60,56,54,54,51,49,45,43,42,39,39,39,
    38,36,35,34,34,33,32,31,30
  };
  const int n1c3w4_o[] = {
    150, // Capacity
    50, // Number of items
    // Size of items (sorted)
    100,100,96,96,94,94,93,85,83,82,82,81,80,79,76,76,76,72,72,72,
    71,70,70,70,68,67,66,64,64,58,58,57,49,49,46,42,39,39,39,38,37,
    37,36,35,33,32,32,30,30,30
  };
  const int n1c3w4_p[] = {
    150, // Capacity
    50, // Number of items
    // Size of items (sorted)
    100,98,98,96,95,95,94,94,94,91,90,90,89,86,85,85,85,84,78,78,
    77,76,75,73,72,72,70,70,69,69,68,68,66,60,59,55,50,50,48,48,47,
    47,44,43,42,40,39,39,37,35
  };
  const int n1c3w4_q[] = {
    150, // Capacity
    50, // Number of items
    // Size of items (sorted)
    100,99,98,97,97,95,92,92,91,90,89,88,87,84,84,83,82,80,80,78,
    77,77,76,76,75,72,70,68,67,64,63,61,61,60,58,57,57,56,55,49,49,
    48,40,40,37,35,32,31,31,30
  };
  const int n1c3w4_r[] = {
    150, // Capacity
    50, // Number of items
    // Size of items (sorted)
    98,94,94,93,92,92,92,91,85,84,84,81,81,79,79,78,76,73,72,71,68,
    68,67,67,65,63,61,60,60,59,59,58,57,56,55,48,47,46,45,43,40,40,
    39,38,37,35,34,32,31,31
  };
  const int n1c3w4_s[] = {
    150, // Capacity
    50, // Number of items
    // Size of items (sorted)
    99,98,97,95,95,93,93,92,89,80,80,79,79,77,76,75,74,74,73,71,71,
    70,68,66,64,63,61,60,57,57,55,54,53,50,50,49,48,47,46,46,42,42,
    39,38,38,37,37,34,32,31
  };
  const int n1c3w4_t[] = {
    150, // Capacity
    50, // Number of items
    // Size of items (sorted)
    100,98,98,97,97,97,96,94,93,90,89,88,88,85,84,84,83,83,81,80,
    78,76,75,73,73,71,71,70,69,66,65,64,64,63,60,60,57,56,54,54,53,
    53,48,43,42,38,34,32,31,30
  };
  const int n2c1w1_a[] = {
    100, // Capacity
    100, // Number of items
    // Size of items (sorted)
    99,97,95,95,94,92,91,89,86,86,85,84,80,80,80,80,80,79,76,76,75,
    74,73,71,71,69,65,64,64,64,63,63,62,60,59,58,57,54,53,52,51,50,
    48,48,48,46,44,43,43,43,43,42,41,40,40,39,38,38,38,38,37,37,37,
    37,36,35,34,33,32,30,29,28,26,26,26,24,23,22,21,21,19,18,17,16,
    16,15,14,13,12,12,11,9,9,8,8,7,6,6,5,1
  };
  const int n2c1w1_b[] = {
    100, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,99,99,98,98,96,96,93,89,84,84,83,83,82,81,80,79,79,79,79,
    78,77,76,75,74,71,71,70,69,69,68,67,67,66,62,56,55,54,53,51,50,
    50,50,49,48,48,47,45,45,45,42,42,42,41,41,40,40,39,38,37,36,36,
    34,34,33,32,32,31,29,28,28,28,26,24,24,22,22,22,21,18,18,17,17,
    15,14,14,12,12,11,10,10,9,8,7,7,5,3,3,2,2
  };
  const int n2c1w1_c[] = {
    100, // Capacity
    100, // Number of items
    // Size of items (sorted)
    98,97,94,92,91,91,90,89,86,85,84,83,82,81,78,76,75,73,73,72,72,
    71,70,70,69,69,66,64,60,60,59,58,57,56,55,54,53,52,52,51,50,49,
    49,48,47,47,45,43,43,43,42,42,42,42,40,39,39,36,35,34,34,34,33,
    32,30,30,30,29,29,28,25,23,22,22,22,22,22,20,20,19,19,18,16,16,
    16,15,15,15,13,12,12,10,9,8,6,5,4,4,2,2
  };
  const int n2c1w1_d[] = {
    100, // Capacity
    100, // Number of items
    // Size of items (sorted)
    99,98,96,93,93,92,90,89,89,89,88,88,87,86,84,84,81,80,80,80,80,
    78,78,77,75,73,72,70,69,68,65,65,64,63,63,63,62,61,60,58,58,58,
    57,56,54,52,51,49,49,46,45,45,44,44,42,42,41,41,38,38,37,36,36,
    34,34,31,30,30,28,27,26,25,24,24,24,23,22,21,21,18,17,17,16,14,
    13,12,12,11,10,10,9,8,6,5,5,4,4,3,2,1
  };
  const int n2c1w1_e[] = {
    100, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,99,99,98,96,95,95,95,93,93,92,92,92,91,90,89,89,89,87,87,
    87,85,84,81,81,80,79,77,74,74,74,73,73,72,71,70,70,66,66,65,65,
    65,64,63,63,63,63,63,61,57,56,54,52,52,51,49,48,46,44,44,44,42,
    40,40,40,38,38,35,34,31,31,31,30,27,27,25,25,24,21,21,21,18,17,
    17,16,16,16,15,15,11,11,9,9,9,8,5,5,5,3,1
  };
  const int n2c1w1_f[] = {
    100, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,100,99,97,96,96,95,95,95,94,93,93,92,92,91,89,85,84,78,76,
    76,76,76,75,73,73,70,70,69,67,67,66,63,62,60,60,60,58,56,55,53,
    53,52,51,50,50,50,49,49,48,47,47,46,45,45,42,41,41,39,37,36,36,
    35,34,34,30,30,29,29,28,28,26,26,23,22,22,22,22,21,21,21,19,18,
    17,17,15,14,14,11,10,8,7,7,6,5,2,2,1,1,1
  };
  const int n2c1w1_g[] = {
    100, // Capacity
    100, // Number of items
    // Size of items (sorted)
    99,96,93,93,93,92,92,91,90,89,88,88,88,87,87,86,84,84,82,81,80,
    80,80,79,79,79,79,76,75,75,75,75,75,74,74,73,71,68,64,62,61,61,
    61,60,58,58,58,58,57,57,57,55,54,53,52,51,51,51,50,50,47,45,44,
    41,40,39,39,39,38,36,36,35,35,34,33,32,31,30,30,29,29,29,28,24,
    22,21,19,19,18,10,9,8,8,7,6,5,5,4,3,2
  };
  const int n2c1w1_h[] = {
    100, // Capacity
    100, // Number of items
    // Size of items (sorted)
    98,98,98,98,94,94,94,93,92,91,89,89,87,86,85,84,80,80,78,76,76,
    75,73,73,72,71,71,71,70,69,67,65,64,64,62,62,62,62,59,56,55,55,
    54,53,53,53,52,52,50,49,49,49,49,49,45,44,43,43,43,43,43,39,38,
    38,38,37,37,36,36,34,34,33,29,29,29,28,27,27,27,25,22,22,19,17,
    17,17,16,15,14,14,14,13,13,13,10,8,6,6,5,3
  };
  const int n2c1w1_i[] = {
    100, // Capacity
    100, // Number of items
    // Size of items (sorted)
    99,98,97,96,95,95,94,94,94,90,88,86,86,86,86,85,85,85,85,85,83,
    83,82,81,81,80,80,79,79,78,77,77,76,76,76,75,75,74,74,74,72,71,
    69,67,67,66,66,65,65,63,61,61,59,59,57,57,56,56,55,54,53,49,48,
    46,45,41,39,39,38,38,37,37,36,36,35,32,30,30,30,28,28,28,27,26,
    26,25,24,23,22,22,17,17,13,11,10,10,6,3,2,1
  };
  const int n2c1w1_j[] = {
    100, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,100,99,98,95,94,93,93,93,92,92,91,91,91,88,88,87,86,85,83,
    81,81,81,80,80,80,79,77,77,77,76,75,73,71,71,71,70,69,68,67,66,
    65,63,60,60,59,59,59,59,56,54,54,54,54,53,53,52,51,51,49,46,44,
    44,43,42,42,41,41,41,39,35,34,34,32,32,31,30,29,28,27,22,22,21,
    21,20,17,14,12,12,11,11,10,10,8,8,6,6,5,5,4
  };
  const int n2c1w1_k[] = {
    100, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,99,98,97,97,97,97,97,92,91,91,91,88,86,86,85,84,84,83,81,
    80,79,79,79,78,77,77,75,75,75,74,74,71,71,70,69,64,64,63,63,62,
    62,61,61,56,56,56,56,55,53,53,52,52,51,49,48,46,44,44,43,43,42,
    42,40,38,37,36,35,34,32,32,31,30,29,29,28,28,28,27,26,24,24,22,
    20,20,18,17,16,16,14,13,13,12,11,10,8,6,4,2,1
  };
  const int n2c1w1_l[] = {
    100, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,100,98,97,96,96,95,95,95,94,94,94,93,92,90,87,87,84,83,83,
    83,81,80,77,77,77,77,75,74,74,73,72,71,71,71,70,70,70,69,69,67,
    63,63,63,63,62,58,55,55,55,54,53,53,51,49,49,49,47,45,42,41,39,
    38,35,34,29,28,28,28,28,27,27,26,26,25,25,25,24,24,23,21,19,17,
    15,15,15,14,12,11,7,7,7,6,5,5,5,2,2,1,1
  };
  const int n2c1w1_m[] = {
    100, // Capacity
    100, // Number of items
    // Size of items (sorted)
    97,96,95,94,90,88,88,87,86,85,84,84,82,81,81,80,80,80,79,79,78,
    74,73,69,69,68,68,67,67,65,64,63,63,60,60,58,57,56,55,53,53,51,
    51,51,47,47,46,46,45,41,41,39,38,37,37,37,37,35,34,33,33,33,33,
    32,31,31,31,30,30,28,22,22,20,20,20,20,19,19,17,17,17,16,16,15,
    13,13,12,12,10,10,9,8,8,8,5,5,5,4,4,1
  };
  const int n2c1w1_n[] = {
    100, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,98,97,95,90,90,89,89,87,87,85,83,82,82,81,81,81,80,79,78,
    77,76,74,73,72,70,70,68,67,64,63,63,60,60,58,58,57,57,55,54,54,
    53,52,52,52,51,50,50,50,48,45,45,45,44,44,43,41,38,37,34,34,34,
    33,32,32,31,30,30,30,30,26,25,24,23,20,19,19,19,18,17,16,15,13,
    12,12,11,11,11,11,10,9,8,8,8,7,4,3,3,2,1
  };
  const int n2c1w1_o[] = {
    100, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,100,98,97,95,94,92,92,92,91,90,89,89,88,88,88,87,85,84,83,
    81,79,79,77,77,76,72,70,70,69,69,68,64,63,62,62,61,61,60,59,59,
    58,57,55,52,52,51,47,47,46,43,43,42,37,36,35,35,35,35,34,32,32,
    31,31,29,29,28,28,25,23,22,22,21,19,17,16,15,14,12,11,11,11,11,
    11,11,10,8,8,7,6,5,5,4,4,3,3,2,2,1,1
  };
  const int n2c1w1_p[] = {
    100, // Capacity
    100, // Number of items
    // Size of items (sorted)
    99,99,96,96,95,93,92,92,91,91,90,90,88,88,87,86,83,83,83,83,81,
    81,80,80,78,78,76,76,74,73,72,72,70,69,69,68,67,66,58,57,56,55,
    55,55,54,54,54,54,53,51,51,51,48,48,47,47,47,46,46,46,45,44,43,
    43,43,42,41,40,40,35,34,31,29,26,24,24,23,23,22,22,22,21,20,18,
    17,17,15,14,12,12,11,9,9,8,6,4,3,3,1,1
  };
  const int n2c1w1_q[] = {
    100, // Capacity
    100, // Number of items
    // Size of items (sorted)
    99,98,97,97,96,94,94,94,93,90,84,82,81,78,76,76,75,75,73,70,70,
    69,69,66,66,65,65,65,63,61,60,59,59,59,58,58,56,55,54,54,53,53,
    50,50,50,48,48,47,46,45,45,45,45,41,41,40,39,39,36,36,35,35,34,
    33,33,31,30,29,28,27,26,26,24,24,19,19,19,18,18,18,18,16,14,14,
    13,12,11,11,10,10,10,7,7,6,6,6,4,3,1,1
  };
  const int n2c1w1_r[] = {
    100, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,100,99,97,97,96,96,95,94,94,94,94,92,92,91,90,88,87,85,84,
    84,83,82,81,80,78,75,74,72,72,71,70,69,69,68,65,64,64,62,61,61,
    60,59,58,58,58,57,57,55,54,54,54,53,53,50,49,48,47,47,46,46,45,
    45,44,43,42,40,36,36,35,34,34,33,32,31,30,30,26,26,25,24,23,23,
    22,22,21,20,19,18,18,17,17,17,15,9,8,7,6,3,3
  };
  const int n2c1w1_s[] = {
    100, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,99,96,96,95,94,94,93,91,89,89,88,81,80,75,74,73,72,69,69,
    69,68,64,63,63,62,61,58,57,57,57,57,56,56,54,54,54,51,49,49,49,
    48,48,48,48,48,48,47,47,47,44,43,43,41,40,40,39,38,38,36,35,33,
    31,30,30,30,30,29,29,28,25,25,23,23,20,19,18,16,15,14,14,14,12,
    12,11,10,9,9,8,8,8,7,7,7,5,4,4,3,2,2
  };
  const int n2c1w1_t[] = {
    100, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,100,100,98,97,96,95,94,92,91,91,90,90,90,88,87,87,85,84,83,
    81,78,76,74,71,71,70,68,68,66,66,65,64,63,63,62,62,61,59,59,59,
    59,59,57,57,56,54,53,52,51,50,50,49,46,45,43,41,41,40,40,40,39,
    36,35,34,33,33,32,32,32,30,30,29,29,29,28,27,27,27,23,21,21,20,
    20,19,19,17,15,15,15,11,9,6,5,5,5,4,3,2,1
  };
  const int n2c1w2_a[] = {
    100, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,100,100,99,99,98,96,95,95,94,93,93,92,90,90,89,86,86,85,85,
    84,83,82,82,82,81,80,79,77,77,77,76,75,75,75,74,73,71,71,69,68,
    67,67,67,65,63,63,60,57,56,56,55,55,54,54,54,53,53,51,51,47,46,
    46,45,45,45,44,44,44,44,43,41,40,40,39,39,39,39,38,36,36,34,33,
    33,32,32,31,30,29,28,26,25,24,24,23,22,22,22,21,20
  };
  const int n2c1w2_b[] = {
    100, // Capacity
    100, // Number of items
    // Size of items (sorted)
    99,96,96,94,94,93,93,90,90,88,88,88,87,87,86,85,84,84,84,83,83,
    83,82,81,81,80,80,77,75,75,75,74,73,69,69,67,67,66,66,65,65,64,
    64,63,63,63,59,58,56,55,54,54,53,53,52,50,50,50,48,48,47,47,45,
    43,42,42,42,41,41,41,40,39,38,38,34,34,32,32,32,31,31,30,30,29,
    27,26,26,26,26,25,25,25,24,23,22,22,22,21,21,20
  };
  const int n2c1w2_c[] = {
    100, // Capacity
    100, // Number of items
    // Size of items (sorted)
    98,96,95,95,94,94,92,91,89,88,86,85,84,84,83,83,82,82,81,80,80,
    79,77,77,77,75,75,75,75,75,72,71,70,69,68,68,66,66,66,66,64,64,
    64,64,63,62,62,61,59,58,58,58,57,56,56,56,56,55,55,54,54,53,51,
    51,51,50,50,49,49,49,48,48,48,45,45,44,43,41,40,40,36,34,33,32,
    32,32,29,27,27,27,27,25,25,25,24,23,23,21,21,20
  };
  const int n2c1w2_d[] = {
    100, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,99,98,97,96,95,94,94,94,93,93,93,92,92,92,91,90,90,89,88,
    88,87,86,85,85,85,84,83,83,83,79,78,78,78,77,77,77,76,74,74,73,
    72,72,71,71,70,70,69,68,67,65,64,64,63,61,61,60,59,59,58,57,57,
    56,55,55,55,54,54,54,54,52,52,51,51,49,46,46,46,45,44,43,41,40,
    39,38,37,35,35,32,32,32,30,30,30,29,28,27,23,22,20
  };
  const int n2c1w2_e[] = {
    100, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,100,100,99,99,99,99,98,97,96,95,94,94,91,90,90,90,89,89,89,
    88,88,87,87,86,85,85,85,84,82,81,80,80,79,79,77,76,74,73,71,70,
    69,68,68,67,67,66,65,65,65,62,62,62,59,59,59,57,57,55,55,54,51,
    50,49,47,47,46,45,45,43,42,41,41,41,39,38,37,35,35,34,34,34,33,
    32,31,30,29,29,27,26,26,25,24,24,24,21,21,21,20,20
  };
  const int n2c1w2_f[] = {
    100, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,99,99,98,98,98,96,96,96,96,95,95,94,94,93,91,90,90,89,89,
    89,88,88,86,85,83,83,83,83,81,81,79,79,78,78,78,77,76,75,75,72,
    71,68,68,67,66,61,60,60,59,59,58,58,58,57,56,52,52,52,52,50,47,
    47,47,44,43,43,43,41,41,41,40,39,38,36,36,32,32,32,31,29,29,29,
    28,28,28,28,27,27,27,26,25,24,24,24,24,23,23,21,21
  };
  const int n2c1w2_g[] = {
    100, // Capacity
    100, // Number of items
    // Size of items (sorted)
    99,99,99,99,97,97,95,94,92,92,92,91,91,90,90,90,89,88,87,87,86,
    85,84,83,83,83,81,80,79,78,78,77,76,76,74,73,73,72,72,72,71,70,
    70,70,68,68,67,67,65,65,65,64,64,64,64,63,63,63,63,61,60,59,58,
    57,57,56,55,54,53,51,50,49,48,48,48,47,47,45,41,39,39,38,38,37,
    36,35,29,28,27,26,26,24,22,22,22,22,22,21,20,20
  };
  const int n2c1w2_h[] = {
    100, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,99,95,95,94,94,93,93,93,92,91,88,87,86,86,86,86,85,85,85,
    84,84,84,83,82,81,79,78,77,76,76,76,76,75,75,73,72,71,71,69,69,
    69,69,67,67,65,65,64,64,64,64,63,63,62,61,61,60,59,59,59,57,57,
    56,56,55,55,54,53,51,49,47,45,45,43,43,43,42,42,42,38,37,36,36,
    33,31,29,28,28,28,28,27,27,27,26,26,25,24,22,22,20
  };
  const int n2c1w2_i[] = {
    100, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,99,98,97,97,96,95,95,93,93,93,93,91,91,90,89,89,89,89,89,
    89,88,88,87,86,84,84,81,80,79,78,78,76,75,74,72,72,71,71,70,69,
    69,66,66,63,63,62,62,61,60,59,59,57,57,55,55,55,54,54,54,53,53,
    52,52,51,50,50,50,49,49,48,47,47,41,40,40,39,38,36,35,34,33,33,
    32,31,31,31,31,30,30,28,27,24,23,23,22,21,20,20,20
  };
  const int n2c1w2_j[] = {
    100, // Capacity
    100, // Number of items
    // Size of items (sorted)
    99,97,96,95,95,95,94,94,94,93,92,90,90,89,89,89,89,89,89,88,88,
    86,86,85,85,85,84,84,83,82,82,80,79,78,78,78,77,77,77,76,75,75,
    69,67,66,66,66,65,65,65,64,64,62,62,58,58,58,58,58,55,54,53,53,
    51,50,50,50,49,49,46,45,42,42,42,41,40,39,39,37,37,37,37,35,33,
    33,32,31,30,29,28,26,25,21,21,21,21,21,20,20,20
  };
  const int n2c1w2_k[] = {
    100, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,99,98,97,95,95,93,92,91,91,91,91,90,89,89,88,88,86,85,85,
    83,81,81,81,80,80,79,78,77,77,77,76,76,76,75,75,74,74,73,73,71,
    71,70,70,69,69,69,67,67,67,67,66,65,63,63,63,63,62,62,62,61,57,
    55,53,53,51,51,51,50,50,49,49,48,48,48,47,47,46,43,41,41,40,36,
    36,36,36,35,35,33,32,32,31,31,29,28,28,25,25,23,21
  };
  const int n2c1w2_l[] = {
    100, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,97,96,96,94,94,94,93,93,93,91,91,90,90,88,83,83,82,82,81,
    81,80,78,78,78,76,75,75,74,72,72,71,70,70,70,70,70,67,65,64,64,
    64,63,62,62,61,60,60,58,58,57,55,55,54,53,52,52,51,50,49,48,47,
    47,47,46,45,45,45,44,43,42,42,41,41,40,39,38,38,36,36,35,35,35,
    33,32,31,30,30,29,27,26,25,24,24,23,23,22,22,22,20
  };
  const int n2c1w2_m[] = {
    100, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,100,99,98,97,97,97,96,95,95,95,95,94,92,92,91,91,90,90,89,
    89,89,87,86,85,83,82,82,80,80,79,78,76,75,74,72,72,71,71,71,70,
    66,65,63,63,63,63,62,61,60,60,60,60,59,57,55,55,55,53,52,51,46,
    46,46,45,45,42,41,41,41,40,40,39,39,39,39,38,38,37,36,36,35,35,
    35,35,34,34,31,30,29,29,28,27,27,27,27,26,26,22,22
  };
  const int n2c1w2_n[] = {
    100, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,100,99,99,99,98,96,95,95,94,94,94,93,93,92,92,92,91,91,89,
    86,86,85,85,83,82,81,81,80,78,77,77,75,74,74,73,70,70,69,69,68,
    68,67,66,65,64,63,63,62,60,59,59,58,56,56,56,55,54,51,50,50,49,
    48,47,47,46,46,46,44,44,43,42,39,39,38,38,37,37,34,34,32,32,31,
    30,30,29,29,28,28,27,27,27,25,24,24,24,23,21,20,20
  };
  const int n2c1w2_o[] = {
    100, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,98,98,98,98,97,96,95,95,94,93,92,90,90,89,88,88,88,87,87,
    86,85,84,83,83,83,82,82,80,80,79,79,78,78,76,74,74,74,74,71,69,
    68,68,67,67,66,64,64,64,64,62,62,61,60,60,55,55,53,53,50,49,49,
    47,45,44,44,43,43,42,42,42,41,41,39,36,35,35,33,33,32,31,31,31,
    31,30,30,29,28,25,25,23,23,22,22,21,21,21,20,20,20
  };
  const int n2c1w2_p[] = {
    100, // Capacity
    100, // Number of items
    // Size of items (sorted)
    99,98,97,96,96,95,94,93,93,92,92,90,90,89,89,88,88,88,88,86,86,
    85,83,82,82,80,80,80,79,79,77,77,77,76,76,76,74,73,73,71,71,70,
    69,69,69,68,68,67,66,66,65,63,60,59,57,57,57,57,56,53,53,52,51,
    51,51,51,50,47,46,45,44,44,44,43,42,42,39,39,38,38,38,37,36,36,
    36,32,31,30,28,28,27,27,27,26,26,24,24,22,22,20
  };
  const int n2c1w2_q[] = {
    100, // Capacity
    100, // Number of items
    // Size of items (sorted)
    97,97,97,96,96,95,94,94,94,90,89,86,85,84,83,79,78,78,78,77,77,
    77,76,76,75,75,74,74,72,72,71,71,70,69,69,67,67,66,66,66,66,65,
    65,64,63,63,62,62,61,60,59,59,57,56,56,55,53,53,52,52,51,51,51,
    50,50,49,49,49,49,48,48,47,47,45,43,40,39,37,37,35,34,33,33,32,
    32,31,30,29,28,28,28,27,27,27,25,24,24,23,23,22
  };
  const int n2c1w2_r[] = {
    100, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,99,98,98,98,98,97,97,96,96,96,94,94,93,92,90,88,87,87,86,
    86,85,85,85,85,85,84,84,83,83,83,83,80,79,79,78,77,77,76,75,75,
    74,71,70,69,67,65,64,62,62,62,62,61,61,60,58,57,56,55,55,55,54,
    54,53,52,51,49,49,47,46,45,44,44,43,43,41,41,40,39,37,34,32,32,
    31,29,28,28,27,26,26,25,25,24,24,23,23,22,22,21,20
  };
  const int n2c1w2_s[] = {
    100, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,98,98,97,96,94,94,93,93,91,90,90,90,89,89,87,87,86,86,86,
    84,84,82,82,81,81,80,79,77,77,77,76,76,75,75,73,72,72,71,70,70,
    70,70,67,64,62,62,59,59,59,58,58,58,55,55,54,54,53,53,53,51,51,
    50,50,50,49,49,48,47,46,46,45,45,44,41,41,39,39,37,37,37,37,35,
    34,34,34,33,33,33,32,31,29,27,25,25,24,23,22,20,20
  };
  const int n2c1w2_t[] = {
    100, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,99,99,99,98,97,95,94,94,94,93,93,92,92,91,90,90,90,90,89,
    89,87,86,85,83,82,80,80,79,79,78,78,78,77,75,72,71,70,70,67,65,
    64,63,62,62,62,61,60,60,59,58,58,58,57,57,56,56,56,55,55,54,52,
    51,49,49,48,47,46,46,46,46,46,44,44,43,42,42,39,37,36,36,35,34,
    34,33,33,33,32,30,30,30,27,26,25,24,24,24,21,21,20
  };
  const int n2c1w4_a[] = {
    100, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,99,97,96,96,96,94,94,94,93,93,93,92,91,90,90,90,89,89,88,
    88,83,83,82,82,81,80,80,80,79,79,79,79,78,78,78,76,74,74,73,73,
    71,70,69,69,68,67,67,66,65,64,63,63,63,62,59,58,58,57,56,56,56,
    56,53,53,53,52,51,51,50,49,48,48,48,47,46,46,45,43,42,41,41,39,
    39,39,38,38,38,38,38,37,37,37,36,36,33,32,32,31,31
  };
  const int n2c1w4_b[] = {
    100, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,100,99,99,99,97,96,95,95,93,93,93,91,89,89,89,88,87,87,86,
    85,85,84,83,81,80,80,79,79,78,78,78,77,75,75,73,73,73,72,71,71,
    70,70,69,66,65,65,63,60,60,59,59,58,58,57,57,55,55,55,55,54,54,
    53,53,52,51,50,50,49,49,49,48,45,45,45,45,44,44,43,43,41,41,40,
    40,40,36,36,35,34,34,33,33,33,33,33,32,32,32,32,30
  };
  const int n2c1w4_c[] = {
    100, // Capacity
    100, // Number of items
    // Size of items (sorted)
    99,97,97,96,96,94,93,93,92,92,91,90,90,90,88,87,87,86,86,86,85,
    85,85,85,84,84,83,83,82,82,81,81,81,79,79,78,77,76,76,76,76,76,
    74,74,73,71,71,70,70,69,69,67,67,66,65,65,65,63,62,62,61,60,60,
    60,59,59,58,57,56,56,55,55,54,53,52,51,50,50,48,48,43,40,38,38,
    38,37,35,35,35,35,34,33,33,32,32,31,31,31,31,30
  };
  const int n2c1w4_d[] = {
    100, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,100,99,98,98,97,97,96,95,95,94,94,94,93,92,89,89,88,88,88,
    88,87,86,85,84,84,82,81,81,80,79,78,77,77,76,76,76,76,74,74,74,
    73,72,72,72,71,71,71,69,69,68,68,68,68,67,67,66,66,65,65,64,64,
    62,61,58,57,57,57,56,55,54,54,54,53,53,52,52,52,52,51,51,50,49,
    49,48,47,46,45,45,40,40,39,37,37,35,34,34,33,33,30
  };
  const int n2c1w4_e[] = {
    100, // Capacity
    100, // Number of items
    // Size of items (sorted)
    99,99,98,97,97,96,96,95,95,95,94,94,94,94,91,91,89,88,87,86,86,
    85,84,83,82,82,82,81,81,79,78,78,76,76,76,76,73,72,71,71,70,70,
    70,69,69,69,69,69,68,68,67,66,65,64,61,61,61,61,60,60,59,59,58,
    57,57,55,54,54,48,45,45,44,44,43,42,42,42,42,41,41,39,38,37,37,
    36,36,35,35,35,35,34,34,34,33,33,32,31,31,31,30
  };
  const int n2c1w4_f[] = {
    100, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,100,99,97,97,95,95,95,94,93,92,91,90,89,89,88,87,87,86,84,
    83,82,80,80,80,80,80,80,79,79,79,79,78,76,76,76,76,73,73,72,71,
    71,70,69,69,69,69,68,67,66,66,66,64,64,64,62,62,62,62,61,60,60,
    59,58,58,58,58,57,57,56,56,56,56,56,53,52,50,49,48,47,44,44,43,
    42,40,39,37,37,36,36,36,35,35,34,33,33,33,32,30,30
  };
  const int n2c1w4_g[] = {
    100, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,100,98,98,96,95,95,95,94,94,93,93,88,87,85,84,80,80,80,79,
    78,78,78,77,77,77,76,76,73,71,71,70,70,70,70,69,69,68,67,67,66,
    66,66,66,66,66,66,64,63,63,63,61,61,61,61,60,59,59,59,58,57,57,
    57,56,55,54,54,53,51,51,49,49,49,48,47,45,44,44,42,41,41,41,40,
    39,39,39,38,38,37,37,37,36,35,34,34,33,32,32,32,31
  };
  const int n2c1w4_h[] = {
    100, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,100,99,99,98,98,97,96,96,94,94,94,94,93,91,90,89,87,87,87,
    86,84,84,84,83,82,80,79,75,75,75,74,74,73,73,73,72,71,70,69,69,
    69,68,68,68,67,65,65,63,63,61,61,61,61,60,60,60,60,60,59,59,58,
    57,57,56,56,55,54,54,54,51,50,50,49,49,49,49,48,48,48,46,46,44,
    42,42,41,40,40,38,37,35,35,34,34,33,33,33,33,32,31
  };
  const int n2c1w4_i[] = {
    100, // Capacity
    100, // Number of items
    // Size of items (sorted)
    98,97,97,96,96,95,95,95,95,92,92,92,91,91,91,91,90,88,87,86,85,
    83,82,81,80,79,77,76,76,75,75,75,74,74,72,72,72,71,71,71,70,70,
    70,69,69,68,67,65,65,64,63,63,62,62,62,61,61,60,59,59,59,59,58,
    58,56,56,55,55,52,51,50,48,48,47,47,47,46,45,44,44,42,42,42,41,
    40,39,38,36,36,36,35,35,35,35,34,32,32,32,30,30
  };
  const int n2c1w4_j[] = {
    100, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,99,99,98,97,97,97,96,96,96,95,93,91,90,87,87,86,86,84,83,
    82,81,81,81,80,79,79,77,77,76,76,75,74,72,72,72,71,70,70,70,69,
    69,68,68,67,67,67,66,66,66,65,65,65,64,64,62,60,59,57,57,57,57,
    55,55,55,55,53,53,52,52,52,50,50,50,49,49,48,47,47,45,45,45,44,
    43,42,39,39,39,38,38,38,37,35,35,34,32,32,31,30,30
  };
  const int n2c1w4_k[] = {
    100, // Capacity
    100, // Number of items
    // Size of items (sorted)
    99,98,98,97,97,97,95,94,94,94,93,93,91,91,90,89,89,88,88,87,86,
    83,83,82,82,81,81,80,80,79,79,78,76,74,73,73,72,71,71,70,70,70,
    68,68,67,66,66,65,64,64,61,61,60,59,59,57,56,56,56,56,56,55,54,
    53,51,51,51,51,50,50,50,49,47,47,47,46,46,45,45,43,43,42,41,40,
    40,39,39,38,38,37,35,34,34,34,33,33,32,30,30,30
  };
  const int n2c1w4_l[] = {
    100, // Capacity
    100, // Number of items
    // Size of items (sorted)
    99,99,96,96,95,95,94,94,93,91,91,88,88,87,87,87,87,84,84,83,83,
    82,82,82,81,81,81,80,78,77,77,76,76,76,74,74,74,74,74,73,73,73,
    73,73,72,72,71,71,70,70,69,68,67,64,64,63,62,60,60,59,59,59,58,
    58,57,57,57,55,55,53,52,51,50,49,48,46,46,45,43,43,42,42,42,42,
    42,40,40,40,38,37,36,36,34,34,33,33,33,31,30,30
  };
  const int n2c1w4_m[] = {
    100, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,100,99,99,99,99,98,98,97,96,96,96,96,95,95,95,95,91,90,89,
    88,87,86,84,83,83,82,80,79,77,77,76,76,74,74,74,73,72,72,71,71,
    70,69,68,67,67,66,66,65,63,60,60,59,59,58,57,57,56,56,54,53,53,
    53,53,52,51,50,50,50,50,49,47,47,46,46,45,44,43,42,42,42,41,41,
    39,38,38,38,37,37,36,36,36,35,35,35,33,32,32,32,31
  };
  const int n2c1w4_n[] = {
    100, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,100,99,99,98,98,97,97,96,96,96,95,94,94,92,91,91,90,90,90,
    88,87,85,85,84,83,83,81,80,79,79,78,76,76,76,75,74,74,74,73,71,
    70,67,67,67,66,66,66,64,64,64,64,63,63,61,59,59,58,58,58,56,56,
    56,54,53,53,52,51,50,50,49,48,48,48,48,46,45,44,41,40,40,40,39,
    39,37,37,36,36,36,35,35,34,33,33,33,33,32,31,31,30
  };
  const int n2c1w4_o[] = {
    100, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,100,100,100,99,99,98,98,98,97,97,97,96,95,95,94,94,94,94,
    93,93,93,92,92,92,91,91,90,87,86,86,85,85,84,83,83,80,79,78,78,
    77,76,74,72,72,72,71,71,71,71,70,70,69,68,67,66,65,64,63,63,62,
    62,62,60,59,59,58,58,57,57,56,55,55,54,53,52,52,51,51,51,49,46,
    42,41,41,41,40,40,39,39,39,38,36,36,34,34,33,31,30,30
  };
  const int n2c1w4_p[] = {
    100, // Capacity
    100, // Number of items
    // Size of items (sorted)
    99,99,98,96,93,93,92,91,91,91,90,89,89,88,85,85,83,82,82,81,80,
    79,78,78,74,74,70,69,69,66,65,65,64,64,64,64,63,63,62,62,62,62,
    61,61,61,61,61,59,59,59,58,58,57,57,56,55,55,54,53,53,52,52,51,
    49,48,48,47,47,47,47,45,45,45,44,44,43,43,43,42,42,42,42,41,41,
    41,40,40,39,37,37,36,36,35,34,34,34,32,32,30,30
  };
  const int n2c1w4_q[] = {
    100, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,100,98,98,97,97,94,93,93,92,92,92,91,91,91,90,89,89,89,88,
    87,86,85,83,83,83,82,81,80,80,80,79,79,78,77,77,77,77,77,75,75,
    74,74,74,72,70,69,69,69,66,66,66,66,65,64,64,63,62,61,61,60,60,
    60,58,57,57,56,56,54,52,50,49,49,48,47,46,44,43,42,42,40,40,40,
    40,39,39,39,39,38,38,38,38,36,36,35,35,35,34,33,32
  };
  const int n2c1w4_r[] = {
    100, // Capacity
    100, // Number of items
    // Size of items (sorted)
    99,98,98,97,96,96,96,95,95,94,94,93,93,92,92,91,90,89,87,86,85,
    84,82,82,80,79,79,78,78,77,76,75,75,75,75,74,74,74,73,70,69,67,
    67,66,64,64,63,62,62,62,61,61,60,60,59,59,58,58,57,57,56,55,54,
    54,54,51,50,49,49,49,48,48,48,47,47,44,43,43,42,41,41,41,40,40,
    40,40,39,39,38,36,36,36,35,35,33,32,32,32,31,31
  };
  const int n2c1w4_s[] = {
    100, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,100,100,100,99,99,99,99,98,97,97,97,96,96,96,95,94,94,93,
    92,91,91,91,90,89,89,88,88,85,85,82,82,80,80,79,78,77,76,75,75,
    75,75,74,73,72,71,71,70,69,69,69,67,67,66,66,66,66,65,64,64,64,
    64,62,62,61,59,59,59,58,56,56,56,55,55,54,52,50,50,49,49,48,48,
    48,47,46,44,44,43,43,40,40,39,38,35,35,33,33,31,30,30
  };
  const int n2c1w4_t[] = {
    100, // Capacity
    100, // Number of items
    // Size of items (sorted)
    98,97,97,97,96,96,95,92,91,90,89,89,88,88,87,87,87,86,86,86,85,
    85,83,83,83,82,81,80,79,78,78,78,78,75,71,70,70,70,70,69,68,67,
    65,65,64,64,63,61,61,61,61,60,60,60,60,59,57,57,54,54,54,54,53,
    53,53,52,51,50,50,50,49,46,46,46,46,46,45,44,44,44,42,42,41,40,
    40,39,39,38,38,38,37,36,35,35,34,34,34,34,32,32
  };
  const int n2c2w1_a[] = {
    120, // Capacity
    100, // Number of items
    // Size of items (sorted)
    99,98,98,98,97,96,94,92,91,90,90,89,86,84,82,81,81,80,80,79,79,
    79,77,75,73,72,71,71,71,70,67,65,65,62,61,59,56,55,55,55,55,54,
    54,53,52,51,50,48,48,48,47,47,46,45,44,43,43,43,43,42,42,40,39,
    38,38,36,34,30,30,29,27,26,26,24,22,21,21,20,19,18,18,18,15,14,
    13,11,9,8,7,7,6,6,6,4,4,3,3,2,1,1
  };
  const int n2c2w1_b[] = {
    120, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,100,100,99,99,98,97,96,95,95,91,91,91,90,90,88,88,88,88,87,
    87,85,85,82,82,81,79,78,78,78,78,78,78,77,77,77,75,74,72,71,69,
    69,68,67,64,64,62,62,60,58,57,55,55,54,51,51,51,48,48,47,46,45,
    44,42,38,38,36,34,34,31,30,30,30,28,28,28,26,26,25,25,23,23,22,
    21,20,19,18,18,17,16,13,9,8,5,4,4,4,4,3,1
  };
  const int n2c2w1_c[] = {
    120, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,100,97,97,96,95,94,91,90,89,88,84,84,84,83,82,81,80,80,80,
    78,73,72,72,72,69,69,66,65,65,65,65,65,64,63,63,62,60,58,58,57,
    54,54,53,52,51,50,49,49,48,47,46,44,42,40,40,40,39,38,37,37,35,
    35,33,32,31,30,30,29,28,27,27,23,21,20,20,20,19,19,19,18,17,16,
    16,15,14,13,12,12,12,11,10,8,7,5,5,4,3,3,1
  };
  const int n2c2w1_d[] = {
    120, // Capacity
    100, // Number of items
    // Size of items (sorted)
    99,97,97,96,94,94,93,93,89,89,89,88,87,85,85,84,84,82,82,78,77,
    76,75,73,73,71,71,67,66,63,63,62,62,61,61,59,59,57,57,57,57,55,
    53,53,52,51,51,50,49,49,48,48,48,47,46,46,46,44,44,41,38,37,37,
    37,37,35,35,34,34,32,32,31,31,30,29,28,27,27,26,26,26,25,25,24,
    21,19,18,15,13,13,12,12,12,10,10,5,4,3,2,1
  };
  const int n2c2w1_e[] = {
    120, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,100,99,96,94,93,92,92,92,90,90,89,89,89,87,84,82,82,82,81,
    80,77,77,77,77,75,73,72,71,69,68,68,64,64,62,61,58,54,53,53,53,
    52,52,51,51,49,49,48,48,46,45,45,44,43,42,41,40,37,37,36,35,35,
    34,34,33,33,33,31,29,27,24,24,23,22,21,20,18,17,17,16,15,14,14,
    14,13,13,13,11,11,9,8,7,7,6,4,3,1,1,1,1
  };
  const int n2c2w1_f[] = {
    120, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,100,100,100,99,99,97,97,97,97,95,92,91,89,88,88,88,88,88,
    86,85,85,83,82,81,81,80,80,80,79,78,76,75,75,71,70,70,70,69,69,
    68,67,67,65,63,63,62,62,62,56,54,54,54,53,52,52,51,49,49,47,42,
    42,42,41,40,40,38,38,35,34,34,33,31,31,31,31,30,30,29,27,27,26,
    23,22,22,21,19,19,17,16,15,15,12,11,10,9,9,8,4,1
  };
  const int n2c2w1_g[] = {
    120, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,100,100,99,99,98,98,96,95,94,93,91,90,90,89,89,88,86,83,83,
    82,81,81,80,80,80,79,79,79,76,75,74,73,73,70,70,65,63,60,59,59,
    58,57,55,54,54,52,52,51,51,51,50,47,47,46,45,45,45,43,42,42,41,
    36,35,35,35,34,33,33,29,29,29,29,29,28,24,22,22,22,22,22,20,20,
    20,19,18,17,17,16,15,12,11,11,9,8,6,3,1,1,1
  };
  const int n2c2w1_h[] = {
    120, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,99,99,98,98,97,96,94,94,93,93,92,92,90,88,88,87,87,86,86,
    86,85,85,78,78,77,77,77,74,71,71,68,68,67,66,65,65,62,62,60,59,
    59,55,55,54,53,52,52,51,51,50,49,49,48,47,46,46,46,45,45,45,42,
    42,41,41,40,38,36,36,34,33,32,32,32,31,29,27,23,22,22,21,21,20,
    18,16,15,11,10,10,9,9,8,6,6,5,5,4,3,1,1
  };
  const int n2c2w1_i[] = {
    120, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,100,99,98,97,96,96,96,93,93,92,91,88,87,86,85,84,82,82,79,
    79,79,77,77,76,72,71,71,70,68,67,66,66,65,64,64,63,63,62,62,62,
    62,61,60,59,59,58,57,56,55,55,54,51,51,50,50,48,47,47,46,46,46,
    45,44,41,41,38,37,35,33,32,31,29,29,29,28,28,27,26,25,25,22,19,
    19,18,18,13,11,10,10,9,6,5,5,4,3,3,2,1,1
  };
  const int n2c2w1_j[] = {
    120, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,100,99,98,97,96,95,93,87,87,86,85,85,85,84,83,82,82,81,80,
    80,79,79,77,75,75,75,72,72,70,69,69,66,66,66,63,62,62,61,61,60,
    57,57,57,55,53,52,52,48,48,47,46,43,43,42,41,41,40,40,38,37,37,
    37,36,34,32,31,31,31,30,29,29,28,28,26,26,26,25,24,22,19,16,16,
    15,15,14,14,13,9,9,8,7,6,6,5,4,4,4,3,1
  };
  const int n2c2w1_k[] = {
    120, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,100,97,96,95,95,93,93,92,90,90,90,89,88,88,87,85,84,82,78,
    78,78,78,77,74,74,70,69,68,67,67,66,66,65,61,60,60,59,57,56,55,
    55,54,54,52,52,51,51,50,50,49,48,48,48,47,44,43,41,41,40,39,37,
    37,32,32,31,30,30,29,28,27,26,25,24,24,24,23,23,22,21,19,18,18,
    17,16,15,14,12,10,10,8,6,5,4,3,3,2,2,2,1
  };
  const int n2c2w1_l[] = {
    120, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,100,100,99,99,99,98,98,96,96,95,95,95,94,94,93,92,90,90,88,
    87,85,85,85,82,81,81,80,80,80,76,76,76,75,73,73,73,73,72,71,71,
    68,68,64,64,64,61,60,59,58,57,57,56,51,51,50,49,47,45,45,45,44,
    42,40,38,38,36,36,36,35,34,33,30,30,29,29,28,28,27,23,22,20,20,
    19,17,16,16,11,11,9,8,8,7,7,5,5,3,2,2,1
  };
  const int n2c2w1_m[] = {
    120, // Capacity
    100, // Number of items
    // Size of items (sorted)
    98,97,95,93,93,92,92,92,91,90,89,89,89,88,86,84,84,84,83,83,82,
    82,81,81,79,78,77,75,73,72,72,71,71,70,69,68,65,65,64,64,62,61,
    60,57,55,55,53,51,51,50,50,50,48,46,45,42,42,41,41,41,41,41,40,
    39,39,37,36,35,34,33,33,33,30,30,29,27,25,23,23,23,23,19,19,16,
    16,14,14,14,14,12,12,10,8,8,7,7,6,5,3,3
  };
  const int n2c2w1_n[] = {
    120, // Capacity
    100, // Number of items
    // Size of items (sorted)
    99,99,96,96,95,93,92,89,89,88,87,85,81,80,80,78,77,77,76,75,74,
    72,71,71,70,70,69,69,67,67,67,65,65,65,65,64,62,62,59,59,59,58,
    58,56,56,56,56,55,55,54,52,50,50,49,49,48,47,45,43,43,43,41,40,
    39,38,38,37,36,36,36,35,35,35,30,30,29,26,26,26,26,24,24,23,23,
    17,17,17,15,13,13,12,11,11,11,6,5,4,4,3,1
  };
  const int n2c2w1_o[] = {
    120, // Capacity
    100, // Number of items
    // Size of items (sorted)
    98,97,97,97,97,94,93,93,93,92,91,91,90,89,89,88,87,87,87,85,84,
    84,83,83,82,81,81,81,81,78,76,76,75,75,74,73,70,69,68,68,68,66,
    65,64,64,63,59,58,57,56,56,52,51,51,50,49,48,48,47,47,46,46,45,
    45,44,44,43,43,42,40,40,40,37,33,31,30,29,28,26,25,25,24,19,19,
    19,19,17,16,16,15,15,14,13,12,12,7,4,2,1,1
  };
  const int n2c2w1_p[] = {
    120, // Capacity
    100, // Number of items
    // Size of items (sorted)
    99,99,99,99,99,96,96,96,95,94,93,93,91,91,91,89,87,87,86,86,85,
    85,84,83,82,82,81,81,76,75,75,74,72,68,68,66,65,64,64,64,63,61,
    61,60,60,59,58,56,56,56,55,55,54,54,52,51,51,46,44,43,41,40,39,
    39,39,39,38,37,37,36,36,35,33,29,28,27,26,23,23,21,17,17,14,13,
    11,11,10,10,10,9,9,9,8,6,6,4,4,3,3,2
  };
  const int n2c2w1_q[] = {
    120, // Capacity
    100, // Number of items
    // Size of items (sorted)
    98,98,98,98,96,93,92,91,90,89,87,87,86,86,85,84,83,83,81,78,78,
    78,78,78,78,77,72,72,71,70,70,70,69,68,67,65,65,64,64,64,63,63,
    62,62,62,62,61,61,60,60,59,59,58,57,57,56,56,56,55,54,51,50,49,
    49,47,46,46,39,39,38,38,34,33,32,30,30,29,28,27,26,24,23,23,22,
    22,22,20,18,18,15,12,9,6,6,5,3,3,2,2,2
  };
  const int n2c2w1_r[] = {
    120, // Capacity
    100, // Number of items
    // Size of items (sorted)
    98,97,94,94,93,91,90,89,89,89,88,86,86,84,83,80,79,78,77,75,75,
    72,71,70,69,67,66,65,64,64,62,61,60,60,60,59,57,56,56,56,56,56,
    55,55,55,54,51,50,50,49,49,49,48,47,47,46,44,43,42,40,40,37,37,
    36,36,36,36,34,33,33,32,32,30,30,28,28,25,25,24,24,24,22,22,21,
    20,19,17,16,13,12,10,9,6,5,5,4,3,3,2,1
  };
  const int n2c2w1_s[] = {
    120, // Capacity
    100, // Number of items
    // Size of items (sorted)
    99,98,97,96,95,94,93,93,91,90,89,88,87,87,86,86,85,84,83,82,79,
    79,78,77,77,77,77,73,73,72,71,71,70,68,67,63,63,62,61,61,61,61,
    60,59,57,56,52,51,49,48,47,47,47,46,45,44,44,44,44,43,43,42,42,
    39,39,39,34,33,33,32,31,31,28,28,27,25,25,24,24,24,24,22,21,20,
    18,17,17,16,14,14,13,10,10,9,9,7,7,7,7,6
  };
  const int n2c2w1_t[] = {
    120, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,99,99,98,98,95,94,94,91,90,89,87,84,80,80,77,75,74,73,73,
    72,72,72,69,69,65,64,63,62,62,59,59,59,59,59,59,57,56,53,53,51,
    51,51,50,50,50,49,49,48,47,47,47,47,44,44,43,43,40,39,38,37,36,
    34,34,32,30,29,29,27,23,23,23,21,18,18,18,18,17,16,16,16,15,15,
    14,12,12,11,10,10,9,8,8,7,7,5,4,4,4,2,1
  };
  const int n2c2w2_a[] = {
    120, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,100,98,95,94,94,93,93,93,92,90,90,90,89,88,87,87,86,86,84,
    84,83,82,82,81,80,79,79,79,77,77,76,75,75,75,75,74,73,71,69,69,
    68,65,63,60,59,59,58,57,57,56,56,56,56,55,55,54,54,54,54,50,50,
    49,48,48,48,45,45,44,44,43,43,39,38,38,37,37,37,37,36,36,33,33,
    31,29,28,27,27,26,26,26,26,25,25,25,23,23,23,22,22
  };
  const int n2c2w2_b[] = {
    120, // Capacity
    100, // Number of items
    // Size of items (sorted)
    99,99,98,97,96,94,93,93,93,92,91,91,91,91,90,89,88,87,85,85,85,
    82,82,81,80,80,79,78,76,76,75,75,74,74,72,71,71,70,70,69,69,66,
    65,65,65,64,64,63,63,60,60,60,59,59,58,57,56,56,55,54,53,53,53,
    52,52,51,51,50,49,49,49,48,48,47,47,47,47,46,45,45,43,43,41,41,
    40,37,37,36,36,36,31,31,30,29,28,23,22,21,21,20
  };
  const int n2c2w2_c[] = {
    120, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,99,98,98,98,98,98,97,96,94,93,92,90,89,89,88,87,84,83,82,
    81,81,80,80,78,78,78,78,75,75,75,75,74,71,71,71,70,70,69,69,69,
    68,68,66,65,64,64,64,64,63,61,58,57,56,56,55,55,55,54,54,54,54,
    51,50,50,49,48,46,45,45,44,44,43,41,41,40,40,40,39,37,37,36,36,
    35,35,35,35,33,32,31,31,30,29,29,27,27,25,24,21,20
  };
  const int n2c2w2_d[] = {
    120, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,100,96,96,95,95,94,93,92,92,90,89,89,88,88,87,87,87,86,86,
    85,85,85,85,85,84,83,82,77,77,77,76,74,74,72,72,72,71,70,69,67,
    67,66,62,62,60,59,59,59,57,57,56,56,56,55,53,52,52,51,49,48,47,
    46,43,43,43,43,43,41,41,40,40,39,38,37,36,36,36,36,35,34,34,33,
    33,33,33,31,31,29,28,27,27,24,24,23,22,21,20,20,20
  };
  const int n2c2w2_e[] = {
    120, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,99,99,98,97,97,97,95,95,93,92,92,90,90,89,88,88,87,87,85,
    84,84,84,82,80,80,80,79,79,79,78,78,77,77,72,71,71,68,68,66,66,
    66,64,62,61,60,60,59,58,58,57,57,56,55,55,55,54,53,50,50,49,47,
    47,45,45,45,45,45,43,43,43,43,42,42,42,42,42,40,40,39,37,36,36,
    36,33,33,33,30,28,27,27,26,24,23,23,22,22,22,22,21
  };
  const int n2c2w2_f[] = {
    120, // Capacity
    100, // Number of items
    // Size of items (sorted)
    99,96,95,94,92,92,92,92,91,90,89,88,87,86,85,83,83,83,83,82,80,
    80,80,78,77,76,76,75,75,74,74,73,72,71,71,71,68,68,68,66,64,62,
    59,58,58,55,55,54,54,53,53,53,52,52,51,50,50,47,46,45,43,42,41,
    41,40,40,39,39,38,38,37,37,36,35,35,35,35,33,33,33,32,32,32,30,
    28,27,27,26,25,25,25,24,24,23,23,22,22,21,21,20
  };
  const int n2c2w2_g[] = {
    120, // Capacity
    100, // Number of items
    // Size of items (sorted)
    98,98,97,97,96,96,96,95,95,95,95,93,92,92,90,90,90,89,88,88,88,
    85,84,84,82,81,81,80,79,79,77,77,74,73,73,72,71,70,70,70,68,67,
    66,65,65,64,63,63,63,60,58,58,58,57,56,56,56,56,56,55,52,51,51,
    50,49,49,48,48,46,45,45,44,43,43,42,41,41,38,36,36,35,34,34,33,
    32,31,31,30,30,30,29,28,27,26,26,26,23,22,21,20
  };
  const int n2c2w2_h[] = {
    120, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,99,99,98,98,98,96,96,95,94,94,94,93,92,91,90,90,89,88,87,
    84,83,82,79,78,78,78,77,76,74,74,74,73,73,72,71,70,69,69,67,64,
    64,63,63,63,62,61,61,60,60,59,58,57,56,55,54,54,54,54,53,53,51,
    51,50,50,50,49,48,48,48,47,45,44,44,44,43,42,42,41,41,40,38,38,
    38,38,37,35,30,29,28,27,27,26,26,25,25,24,22,22,21
  };
  const int n2c2w2_i[] = {
    120, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,99,99,96,96,92,92,91,91,91,89,87,87,86,86,86,85,84,83,82,
    81,79,79,78,77,76,76,75,75,74,74,73,71,69,69,69,68,68,66,64,63,
    63,63,62,62,61,61,58,57,56,56,54,53,53,52,52,52,50,50,50,49,49,
    48,48,47,45,44,43,42,41,41,40,39,38,37,36,36,35,34,34,32,32,32,
    31,26,25,24,24,24,24,24,23,23,22,22,21,20,20,20,20
  };
  const int n2c2w2_j[] = {
    120, // Capacity
    100, // Number of items
    // Size of items (sorted)
    99,98,98,97,97,96,95,93,93,93,93,93,92,91,91,91,89,87,86,83,83,
    82,81,80,80,80,76,76,76,75,75,75,75,75,73,71,71,70,70,70,69,67,
    66,65,64,63,62,62,61,61,61,61,60,60,59,58,58,58,57,56,55,55,55,
    54,53,52,52,52,52,51,51,50,49,47,46,46,45,45,44,44,43,43,39,39,
    38,37,37,34,33,32,29,28,28,26,25,24,22,22,21,20
  };
  const int n2c2w2_k[] = {
    120, // Capacity
    100, // Number of items
    // Size of items (sorted)
    98,98,98,97,96,95,94,94,92,90,88,88,86,86,86,85,85,83,83,81,80,
    79,78,78,77,77,76,76,75,74,72,71,71,70,70,67,66,65,65,62,61,61,
    60,59,59,59,58,58,57,57,57,56,55,53,53,53,52,52,50,50,49,49,49,
    47,47,47,46,46,44,44,42,42,41,41,40,39,39,39,38,38,36,34,33,33,
    32,29,29,26,26,26,26,25,25,25,25,24,22,21,21,20
  };
  const int n2c2w2_l[] = {
    120, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,100,98,98,98,98,97,97,96,93,91,91,91,91,89,88,87,86,86,85,
    83,83,83,82,82,80,79,78,78,76,75,75,75,74,72,72,72,72,71,69,68,
    66,66,66,62,61,60,59,58,58,57,56,55,54,53,51,50,50,50,50,49,48,
    48,47,47,47,47,46,46,45,45,42,41,40,40,39,39,38,38,37,36,36,36,
    36,33,32,30,30,30,27,25,24,24,24,23,23,22,21,21,20
  };
  const int n2c2w2_m[] = {
    120, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,99,98,98,98,98,97,96,95,95,93,92,92,91,90,90,89,88,88,87,
    85,85,85,85,84,84,83,83,83,82,81,80,79,79,79,78,77,74,74,73,72,
    71,64,61,60,60,59,58,57,57,57,54,54,54,52,51,50,50,49,49,49,48,
    48,47,47,47,46,45,45,44,43,41,41,40,39,36,36,35,34,34,34,32,31,
    30,29,29,28,28,28,27,26,26,25,25,24,23,23,22,22,20
  };
  const int n2c2w2_n[] = {
    120, // Capacity
    100, // Number of items
    // Size of items (sorted)
    99,98,98,97,97,97,97,97,96,95,95,92,92,92,92,91,91,90,90,89,88,
    87,85,85,83,82,82,82,82,81,79,77,76,76,75,75,74,74,71,71,70,69,
    68,66,66,64,63,62,61,61,60,59,56,53,52,51,50,50,48,47,46,43,42,
    41,41,40,40,40,39,39,38,36,34,34,33,33,33,32,32,32,31,31,30,30,
    30,29,29,29,27,27,25,24,23,22,22,21,21,21,20,20
  };
  const int n2c2w2_o[] = {
    120, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,100,98,98,97,97,97,95,93,93,89,89,88,87,86,84,83,82,81,80,
    79,79,79,77,75,73,73,72,72,71,71,71,69,68,68,67,67,66,65,65,64,
    63,60,59,59,58,58,57,57,56,56,55,55,55,55,54,54,54,53,51,51,50,
    50,50,48,47,47,47,47,46,46,45,44,43,41,41,40,40,39,37,36,32,32,
    31,29,28,27,27,27,27,26,25,25,25,25,24,24,22,21,20
  };
  const int n2c2w2_p[] = {
    120, // Capacity
    100, // Number of items
    // Size of items (sorted)
    99,97,97,96,96,95,95,93,93,92,92,91,91,89,89,88,87,86,86,85,84,
    84,83,82,79,78,78,76,72,71,71,71,70,68,68,68,67,66,65,64,62,62,
    62,61,61,59,59,57,57,55,55,54,53,52,52,51,49,48,47,47,47,46,46,
    45,45,44,43,43,42,42,40,39,39,39,39,39,38,37,36,36,35,34,33,32,
    31,30,29,28,28,27,25,25,25,24,23,22,22,21,20,20
  };
  const int n2c2w2_q[] = {
    120, // Capacity
    100, // Number of items
    // Size of items (sorted)
    98,97,97,97,97,96,96,96,96,95,93,93,92,91,90,90,88,88,87,87,87,
    86,86,86,85,83,83,80,80,80,77,76,76,76,75,75,75,70,69,69,68,67,
    66,65,65,65,64,61,60,59,59,58,58,58,55,55,54,54,54,54,54,53,53,
    52,52,52,50,50,46,46,46,45,45,44,44,41,41,40,39,39,37,33,32,31,
    30,30,29,29,29,28,26,24,24,23,22,22,21,21,20,20
  };
  const int n2c2w2_r[] = {
    120, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,99,99,98,97,97,96,95,95,94,93,93,91,91,91,90,89,88,86,86,
    85,82,82,82,81,81,80,79,79,78,78,76,74,73,69,68,67,67,66,66,66,
    66,64,63,62,62,60,60,59,58,56,54,53,52,51,50,50,49,48,47,46,46,
    44,44,43,43,43,43,43,42,42,41,41,40,39,36,35,34,33,33,33,32,32,
    32,31,30,30,30,29,29,27,26,25,24,24,23,22,22,20,20
  };
  const int n2c2w2_s[] = {
    120, // Capacity
    100, // Number of items
    // Size of items (sorted)
    99,99,98,97,96,95,94,94,94,93,93,92,92,92,92,90,90,90,89,88,88,
    87,87,85,85,84,81,79,76,75,74,74,74,72,72,72,72,72,71,70,70,69,
    68,68,68,67,67,65,65,64,64,63,63,63,61,61,61,60,60,59,58,57,57,
    56,56,55,54,53,52,51,49,49,49,49,47,47,46,44,41,40,38,37,37,37,
    35,34,34,33,32,32,31,30,29,27,25,24,23,22,22,20
  };
  const int n2c2w2_t[] = {
    120, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,100,100,99,99,99,97,97,96,93,91,90,87,86,86,86,85,85,85,84,
    84,83,83,82,81,81,79,77,75,75,74,74,73,72,72,72,71,70,70,70,70,
    69,69,69,68,68,67,67,66,65,64,59,59,59,59,57,57,57,56,56,55,54,
    54,52,49,49,48,45,44,44,43,42,42,42,42,41,40,40,39,39,39,38,38,
    36,35,35,35,33,33,32,30,30,29,28,27,27,26,25,25,22
  };
  const int n2c2w4_a[] = {
    120, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,99,99,98,93,93,93,93,93,93,92,92,92,91,91,90,90,89,86,86,
    85,84,84,83,82,82,80,79,77,77,76,76,76,74,74,73,71,71,71,70,69,
    68,68,68,68,67,67,66,64,64,63,62,62,60,60,60,58,56,56,55,55,51,
    50,49,49,46,45,45,45,44,43,43,42,41,41,40,40,40,40,38,38,37,36,
    36,36,36,36,35,34,34,33,32,32,31,31,30,30,30,30,30
  };
  const int n2c2w4_b[] = {
    120, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,99,99,99,98,96,96,96,96,95,94,93,92,92,90,90,90,89,88,86,
    84,84,84,80,80,79,79,79,78,75,75,75,75,74,74,74,72,72,71,71,70,
    70,70,69,69,69,68,67,67,67,67,66,66,65,63,61,60,60,58,57,57,57,
    56,56,55,55,54,53,52,51,50,50,47,47,46,45,43,43,43,42,41,41,40,
    40,39,39,39,38,37,37,37,37,34,34,33,33,32,32,32,30
  };
  const int n2c2w4_c[] = {
    120, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,100,100,100,99,97,96,95,94,94,94,93,90,90,89,89,89,89,88,
    88,87,87,87,86,85,84,84,84,83,83,83,82,80,80,79,78,78,76,75,75,
    74,70,70,69,69,69,69,68,68,68,68,67,66,65,65,64,64,64,63,63,62,
    62,61,61,60,60,59,58,58,57,57,55,54,53,53,51,51,49,49,49,48,47,
    47,46,46,42,41,38,37,35,34,33,32,32,32,31,31,30,30,30
  };
  const int n2c2w4_d[] = {
    120, // Capacity
    100, // Number of items
    // Size of items (sorted)
    99,99,99,98,98,98,97,97,97,96,96,95,94,94,92,91,90,88,88,87,86,
    86,86,86,84,84,83,82,82,82,81,81,81,81,80,79,78,77,77,76,75,75,
    75,75,74,74,73,72,72,69,67,66,63,63,63,61,60,60,59,59,58,58,56,
    56,55,55,54,52,50,49,48,48,48,47,47,47,46,46,44,42,40,40,39,38,
    37,37,36,36,36,35,34,33,33,32,31,31,31,30,30,30
  };
  const int n2c2w4_e[] = {
    120, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,100,99,99,98,98,98,98,98,97,97,96,95,95,95,93,93,91,89,89,
    88,88,87,87,87,86,84,84,84,84,83,83,83,83,81,79,77,76,74,73,71,
    70,69,69,68,68,68,66,66,64,64,64,64,63,61,61,60,60,60,60,59,58,
    58,56,56,56,54,54,51,51,50,50,48,48,47,46,45,45,43,43,43,42,42,
    41,40,37,36,36,36,36,34,33,33,33,33,32,31,31,30,30
  };
  const int n2c2w4_f[] = {
    120, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,99,99,98,97,97,96,96,95,95,94,92,92,90,90,89,87,87,86,85,
    85,85,84,84,84,83,82,81,81,80,80,79,79,79,78,78,76,75,74,73,72,
    72,70,70,68,67,65,65,64,64,63,63,63,62,62,61,59,58,58,57,57,56,
    55,54,54,54,53,52,51,50,47,47,43,42,42,42,42,41,41,40,40,39,38,
    38,38,37,36,35,35,35,35,34,34,33,33,33,32,32,31,31
  };
  const int n2c2w4_g[] = {
    120, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,100,100,99,99,98,96,96,96,95,95,92,91,91,91,91,91,88,87,87,
    87,87,85,85,84,84,82,81,81,80,79,78,77,75,74,74,74,74,72,71,70,
    70,70,70,70,69,69,68,68,67,66,66,65,65,64,63,63,62,61,61,60,58,
    58,56,55,54,54,54,53,53,53,53,52,51,47,47,45,45,44,44,43,43,42,
    41,41,39,38,37,36,36,36,35,35,34,34,33,33,32,32,30
  };
  const int n2c2w4_h[] = {
    120, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,100,99,99,98,97,97,97,96,96,96,96,95,94,93,89,88,87,86,85,
    85,85,85,84,84,84,83,83,82,81,81,81,80,80,79,78,78,77,77,77,76,
    75,72,72,70,69,69,69,69,66,66,65,64,64,63,63,62,59,59,58,58,57,
    57,57,55,54,52,52,51,51,51,48,47,47,47,46,46,45,45,45,44,43,43,
    42,42,42,42,39,37,37,37,35,34,33,32,32,31,31,30,30
  };
  const int n2c2w4_i[] = {
    120, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,99,99,98,97,94,94,94,94,93,93,92,91,91,91,90,90,89,88,87,
    87,87,85,84,83,83,82,82,82,82,79,78,78,77,74,74,74,74,72,72,71,
    71,70,68,67,67,66,66,64,63,63,62,61,61,60,60,59,59,58,56,53,52,
    52,52,52,52,52,52,51,51,50,49,49,48,47,46,46,45,45,45,43,41,40,
    40,39,38,38,38,37,37,35,35,33,33,32,31,30,30,30,30
  };
  const int n2c2w4_j[] = {
    120, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,100,100,99,98,98,98,98,97,97,96,95,95,93,92,91,90,90,90,89,
    88,88,86,86,85,85,83,82,81,81,80,76,76,76,74,74,73,73,73,71,71,
    71,70,70,69,68,68,67,67,67,66,66,66,65,64,64,64,62,61,59,58,58,
    55,55,55,54,52,51,50,50,49,49,49,49,48,47,47,47,44,44,43,43,40,
    40,38,38,38,37,37,37,36,36,36,36,35,33,32,32,31,30
  };
  const int n2c2w4_k[] = {
    120, // Capacity
    100, // Number of items
    // Size of items (sorted)
    99,97,97,97,96,95,94,94,93,93,93,91,90,89,88,86,84,83,83,83,82,
    82,81,81,81,80,78,78,78,77,75,75,74,73,73,73,73,71,71,71,70,69,
    69,68,68,67,66,65,64,64,63,63,63,63,62,62,61,60,59,58,57,57,57,
    57,56,55,54,54,53,52,52,52,52,50,50,49,49,49,48,48,46,45,45,44,
    44,42,39,39,37,34,34,34,34,33,33,32,31,31,30,30
  };
  const int n2c2w4_l[] = {
    120, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,99,99,97,97,97,96,93,91,89,89,88,88,88,85,84,82,82,80,80,
    78,78,78,78,78,77,77,76,76,75,75,75,74,74,74,72,71,70,69,69,69,
    67,67,67,66,65,65,65,64,63,63,61,61,60,60,60,60,59,58,58,57,57,
    57,56,56,54,53,53,52,52,51,51,47,47,46,45,45,45,44,44,43,43,43,
    43,42,37,37,37,35,34,34,33,33,33,33,32,32,31,30,30
  };
  const int n2c2w4_m[] = {
    120, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,99,98,97,96,96,95,94,94,94,93,93,92,92,91,91,91,90,90,90,
    89,86,86,85,84,84,83,82,82,77,77,77,77,77,76,75,75,74,73,72,71,
    71,70,70,70,70,69,69,68,67,67,66,65,64,64,63,61,60,58,58,58,57,
    57,57,54,54,54,53,52,52,52,51,51,51,48,46,46,46,45,44,44,44,43,
    43,43,41,39,38,38,36,36,35,35,34,32,31,31,31,30,30
  };
  const int n2c2w4_n[] = {
    120, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,99,99,98,97,95,95,94,94,94,93,92,92,91,91,91,90,89,87,87,
    86,86,85,84,81,81,81,81,80,79,79,79,79,78,77,75,75,75,74,74,73,
    73,73,71,71,70,70,69,67,67,66,64,64,63,63,63,62,61,61,61,61,60,
    59,59,59,59,58,58,56,56,54,54,53,53,53,52,52,51,49,45,44,44,43,
    43,39,37,37,37,37,37,37,36,36,35,33,32,32,31,31,30
  };
  const int n2c2w4_o[] = {
    120, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,99,97,97,97,94,94,93,93,93,92,92,92,91,91,90,90,90,88,88,
    88,88,87,87,87,86,86,86,86,85,85,84,84,83,83,81,81,80,79,79,79,
    79,77,74,74,73,72,72,70,70,67,67,66,66,66,65,64,64,64,63,62,61,
    59,58,54,53,53,52,51,47,47,45,44,43,43,42,41,41,41,39,39,39,39,
    37,37,36,35,35,34,34,33,33,33,32,31,31,30,30,30,30
  };
  const int n2c2w4_p[] = {
    120, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,99,99,99,98,97,97,96,96,95,94,94,93,91,89,89,89,87,87,86,
    85,84,84,84,83,83,83,83,79,79,76,76,75,74,73,73,72,71,71,70,70,
    70,70,68,67,67,66,64,64,63,62,62,62,62,62,59,58,58,56,56,56,54,
    54,54,53,53,53,51,51,50,49,49,48,48,48,47,46,46,45,44,43,43,43,
    42,41,41,41,41,40,39,38,38,38,38,37,36,35,32,31,30
  };
  const int n2c2w4_q[] = {
    120, // Capacity
    100, // Number of items
    // Size of items (sorted)
    99,98,98,98,96,95,94,91,90,90,90,89,88,86,85,85,84,83,83,83,83,
    82,80,80,79,79,78,78,77,77,77,77,77,76,76,76,76,76,76,76,76,73,
    73,72,71,71,70,70,68,67,67,67,66,65,64,63,62,62,62,61,59,57,56,
    56,56,56,55,54,54,54,54,53,52,52,51,51,50,48,47,47,47,45,45,44,
    44,42,41,41,38,37,36,34,34,34,32,32,32,31,30,30
  };
  const int n2c2w4_r[] = {
    120, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,99,99,98,97,97,97,96,94,94,93,93,93,91,89,89,89,89,89,88,
    87,87,86,86,85,85,84,83,80,79,78,77,77,77,73,73,71,70,70,69,69,
    68,67,65,63,62,62,62,62,61,60,60,59,59,59,58,58,58,57,57,56,56,
    55,54,53,52,51,49,48,47,46,45,45,45,44,43,42,42,42,42,41,40,39,
    39,38,37,35,35,35,35,34,33,33,32,32,31,30,30,30,30
  };
  const int n2c2w4_s[] = {
    120, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,100,97,96,96,95,94,94,94,90,90,90,87,86,86,86,83,83,83,83,
    83,82,82,82,80,79,79,78,77,77,77,76,76,75,71,71,71,70,70,68,68,
    67,67,66,66,65,63,63,63,62,61,61,60,60,59,59,59,58,56,55,53,53,
    53,52,51,49,49,47,45,45,45,45,45,44,42,42,42,41,41,41,41,41,39,
    39,38,38,38,37,33,33,33,33,32,32,32,31,31,31,31,30
  };
  const int n2c2w4_t[] = {
    120, // Capacity
    100, // Number of items
    // Size of items (sorted)
    99,99,98,98,97,97,97,96,93,92,91,91,90,89,88,88,87,86,86,85,85,
    84,84,83,83,81,80,80,78,76,75,75,74,72,72,71,69,69,68,68,68,68,
    67,66,66,65,62,61,61,60,60,60,59,58,58,57,57,57,56,56,54,54,53,
    53,53,52,52,51,50,50,50,49,48,48,46,46,46,46,45,45,43,42,42,41,
    41,41,38,37,36,36,35,34,34,34,33,33,33,32,30,30
  };
  const int n2c3w1_a[] = {
    150, // Capacity
    100, // Number of items
    // Size of items (sorted)
    99,99,97,97,96,96,96,94,93,93,92,90,90,90,89,88,88,87,83,82,81,
    81,81,80,79,78,77,77,76,76,75,74,74,74,71,69,69,68,67,67,66,62,
    59,58,57,56,55,54,54,53,53,52,52,49,49,48,47,46,45,44,43,43,42,
    42,39,38,37,35,35,34,32,32,31,31,30,29,24,24,21,21,21,20,18,16,
    13,12,11,9,7,7,7,6,5,5,4,4,2,2,1,1
  };
  const int n2c3w1_b[] = {
    150, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,99,96,94,93,92,92,91,91,91,89,88,86,86,86,85,84,84,84,81,
    81,80,79,79,78,77,77,77,77,73,71,69,67,66,65,65,64,64,64,62,60,
    57,57,56,56,56,56,53,52,51,51,50,50,48,47,46,45,44,43,42,41,41,
    40,40,39,39,38,37,36,36,36,34,33,31,31,29,29,26,25,22,22,22,20,
    17,11,11,10,9,7,7,7,7,6,5,3,2,2,1,1,1
  };
  const int n2c3w1_c[] = {
    150, // Capacity
    100, // Number of items
    // Size of items (sorted)
    98,97,97,97,96,95,95,95,95,93,92,88,87,86,86,85,81,81,80,78,78,
    78,77,77,76,75,74,72,71,70,70,69,69,67,67,67,65,65,65,64,64,63,
    62,58,58,56,56,56,55,52,51,50,50,50,49,49,47,45,43,43,43,42,41,
    40,40,40,39,38,36,35,33,33,32,30,29,28,28,25,25,22,22,20,20,18,
    17,16,15,11,11,10,8,5,5,5,4,4,2,2,2,1
  };
  const int n2c3w1_d[] = {
    150, // Capacity
    100, // Number of items
    // Size of items (sorted)
    99,99,97,97,96,96,94,92,92,92,92,91,90,90,89,89,88,85,84,84,84,
    80,80,78,78,77,77,77,76,75,75,75,74,73,73,72,71,71,70,68,66,65,
    64,62,61,60,57,56,56,55,55,54,54,52,50,50,48,48,47,47,45,45,45,
    44,42,40,40,39,38,38,38,36,34,32,30,29,29,29,28,28,28,26,25,25,
    24,21,18,17,14,13,12,12,10,10,9,9,8,5,4,1
  };
  const int n2c3w1_e[] = {
    150, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,99,99,98,98,96,93,91,89,89,88,86,86,85,85,85,84,84,82,82,
    81,80,79,78,77,76,75,75,73,72,71,70,69,68,68,66,66,64,63,63,62,
    62,58,57,55,54,52,51,50,50,49,48,48,46,46,44,43,41,41,38,37,34,
    33,31,31,31,31,29,29,28,28,27,27,27,26,26,26,25,22,22,21,20,20,
    19,18,18,16,15,15,15,14,14,13,9,8,8,8,2,2,2
  };
  const int n2c3w1_f[] = {
    150, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,100,100,98,98,97,97,96,94,92,90,87,86,84,84,83,83,81,81,81,
    81,80,77,77,77,75,74,74,74,73,70,69,69,68,67,66,66,65,65,64,63,
    62,62,61,60,59,57,57,57,57,56,56,54,52,50,50,47,45,43,43,43,40,
    38,37,37,36,36,35,35,33,33,32,31,31,29,27,27,24,23,19,18,16,14,
    13,13,12,12,11,10,9,8,8,8,4,4,4,3,2,2,1
  };
  const int n2c3w1_g[] = {
    150, // Capacity
    100, // Number of items
    // Size of items (sorted)
    99,98,96,94,93,92,91,91,88,88,87,87,87,86,85,84,83,82,81,79,79,
    77,75,73,73,73,72,71,69,68,67,66,65,65,64,64,62,62,61,60,60,57,
    55,55,54,50,50,50,49,48,48,47,45,44,44,44,42,42,39,38,35,35,34,
    34,34,33,33,32,31,31,29,29,28,26,25,23,21,21,20,19,18,18,16,16,
    15,14,13,13,11,11,11,10,8,6,6,5,5,4,3,2
  };
  const int n2c3w1_h[] = {
    150, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,99,98,98,98,94,93,91,91,89,87,87,87,86,86,86,85,85,84,83,
    83,81,81,80,78,77,77,76,76,75,75,73,73,70,69,69,65,63,63,63,62,
    62,62,60,59,58,57,57,55,54,53,52,51,51,50,49,49,48,47,47,44,44,
    42,38,37,37,32,32,32,30,30,29,28,27,27,25,25,25,23,23,23,22,22,
    21,20,19,17,15,14,13,13,10,9,8,6,5,4,3,2,1
  };
  const int n2c3w1_i[] = {
    150, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,99,97,96,94,94,92,92,92,91,91,89,87,86,86,86,85,85,83,83,
    80,80,78,76,75,73,72,68,66,65,64,63,63,62,62,61,60,58,58,56,56,
    56,54,54,53,53,52,51,51,50,49,49,49,48,47,47,46,45,43,43,42,42,
    42,40,37,37,36,36,34,34,33,33,31,29,25,24,24,23,21,21,20,17,16,
    15,13,13,12,11,11,11,10,9,9,8,8,7,7,5,3,1
  };
  const int n2c3w1_j[] = {
    150, // Capacity
    100, // Number of items
    // Size of items (sorted)
    99,99,98,97,97,95,95,92,91,90,90,89,88,87,86,86,86,85,83,83,83,
    82,80,78,78,77,76,76,75,75,74,72,70,69,67,62,61,61,59,59,59,58,
    58,56,56,55,52,52,52,51,51,49,47,47,46,44,43,42,42,39,37,37,36,
    31,31,31,28,27,25,25,25,23,21,19,18,17,16,16,16,16,15,14,14,14,
    14,13,13,10,10,9,7,7,6,6,5,4,2,2,1,1
  };
  const int n2c3w1_k[] = {
    150, // Capacity
    100, // Number of items
    // Size of items (sorted)
    98,98,96,95,95,94,94,93,93,92,92,92,90,89,89,88,87,87,87,87,85,
    85,83,83,82,81,80,80,79,76,75,75,74,73,71,70,68,68,66,66,63,63,
    63,59,59,58,58,58,58,56,55,54,53,51,49,49,47,46,46,45,44,44,43,
    42,40,37,37,37,36,33,33,33,30,30,29,26,26,26,26,25,24,23,22,21,
    21,20,18,17,17,16,15,10,7,6,5,4,3,2,1,1
  };
  const int n2c3w1_l[] = {
    150, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,99,99,97,97,96,95,95,95,93,93,90,89,89,86,85,82,81,79,79,
    78,77,77,76,76,76,74,74,74,73,71,71,70,70,69,67,66,66,65,65,61,
    61,61,60,59,59,58,57,54,52,48,48,47,47,46,46,46,46,44,44,42,42,
    41,41,39,39,39,39,36,35,34,31,31,26,26,26,24,22,21,21,19,18,17,
    17,16,16,15,15,14,14,13,12,10,7,7,7,3,3,2,2
  };
  const int n2c3w1_m[] = {
    150, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,100,98,97,95,94,92,89,87,87,83,81,81,81,80,80,78,77,75,74,
    74,71,69,68,67,66,66,65,64,64,64,64,64,64,64,63,58,56,55,54,52,
    50,49,49,46,46,45,44,43,41,40,40,37,35,35,35,34,34,33,32,32,32,
    31,30,29,27,27,26,25,25,24,24,23,22,21,21,19,19,19,18,18,18,17,
    17,15,14,14,14,11,11,8,6,6,5,4,3,2,2,1,1
  };
  const int n2c3w1_n[] = {
    150, // Capacity
    100, // Number of items
    // Size of items (sorted)
    98,98,96,94,94,91,89,88,88,87,87,87,86,85,85,84,84,82,81,81,80,
    80,79,79,78,76,75,72,72,70,69,69,68,67,66,65,64,63,58,57,54,54,
    53,53,53,53,50,49,47,44,44,43,43,42,42,40,38,38,37,36,34,33,33,
    30,30,30,29,26,25,25,23,23,20,20,19,19,16,16,15,15,15,15,13,12,
    12,11,10,10,9,9,7,6,6,4,4,3,2,2,1,1
  };
  const int n2c3w1_o[] = {
    150, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,98,96,96,94,93,93,92,91,91,90,89,89,86,86,85,84,83,82,82,
    79,79,79,79,77,75,75,75,74,74,74,74,71,71,70,68,68,67,66,63,63,
    62,62,60,59,59,58,55,54,54,52,49,48,47,47,46,45,44,43,43,42,40,
    39,39,37,37,36,35,34,33,28,26,26,25,25,23,22,21,20,19,19,19,18,
    17,17,16,12,12,12,10,10,9,9,8,7,7,7,6,3,2
  };
  const int n2c3w1_p[] = {
    150, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,97,96,94,94,93,92,92,91,90,90,87,86,86,86,84,84,82,81,80,
    77,76,76,76,75,74,74,73,73,72,72,71,71,70,70,70,69,68,68,67,66,
    66,65,64,63,62,62,60,59,59,59,59,57,52,52,50,49,48,47,46,44,42,
    41,38,36,36,34,33,30,28,27,25,25,24,22,20,20,17,16,16,15,15,15,
    13,13,12,11,11,10,10,10,10,9,8,8,6,5,5,4,3
  };
  const int n2c3w1_q[] = {
    150, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,99,97,94,93,91,89,88,86,85,85,84,83,81,81,80,79,78,77,76,
    75,75,74,71,71,70,69,68,68,68,68,66,64,63,63,62,62,62,61,59,58,
    56,55,55,54,54,54,54,52,52,47,46,46,46,45,44,41,41,39,39,39,38,
    38,37,36,36,35,35,34,34,34,33,31,30,29,29,29,29,28,28,27,27,27,
    26,26,26,23,23,22,20,20,20,17,14,8,8,6,3,1,1
  };
  const int n2c3w1_r[] = {
    150, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,98,95,95,94,92,92,92,90,88,88,87,87,87,86,86,83,83,82,82,
    81,80,77,76,75,75,75,74,73,70,70,68,66,66,66,65,64,64,60,59,58,
    56,55,52,52,52,52,52,51,49,49,48,46,44,42,42,41,41,41,40,40,39,
    38,36,36,35,34,34,34,31,31,30,27,27,27,24,24,22,21,20,15,15,15,
    14,14,12,12,11,10,9,7,6,6,5,4,4,3,3,2,1
  };
  const int n2c3w1_s[] = {
    150, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,99,99,98,97,96,95,95,94,91,91,89,88,88,86,83,82,79,78,78,
    76,75,75,74,72,71,70,70,69,69,69,68,66,65,64,64,63,63,62,62,61,
    60,58,58,57,56,56,55,55,54,52,52,49,49,49,48,48,47,46,46,45,45,
    41,40,40,39,37,36,36,36,35,35,35,35,33,32,31,31,31,28,28,25,24,
    24,21,20,19,19,19,18,16,16,16,16,13,13,11,8,6,5
  };
  const int n2c3w1_t[] = {
    150, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,99,98,96,95,95,95,91,90,90,90,89,88,85,85,83,81,80,80,80,
    79,79,78,77,77,77,76,76,75,74,74,73,73,71,68,67,66,65,64,63,62,
    58,56,56,55,53,51,51,51,50,49,46,44,44,43,43,42,42,42,40,39,38,
    37,37,37,36,36,36,34,34,34,33,32,31,30,30,29,27,26,26,25,22,19,
    18,17,16,16,15,14,12,12,10,9,7,6,5,4,4,3,1
  };
  const int n2c3w2_a[] = {
    150, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,99,98,96,96,96,96,96,96,94,93,93,92,92,92,91,91,91,90,87,
    84,83,83,79,78,78,77,77,76,76,75,75,75,73,73,73,72,72,72,72,72,
    71,71,70,70,66,66,65,64,63,59,58,57,56,56,55,55,54,53,53,52,51,
    49,47,46,46,45,44,43,43,42,41,41,39,39,38,37,35,35,34,34,33,33,
    32,32,32,32,31,30,30,29,28,24,23,22,22,22,22,21,20
  };
  const int n2c3w2_b[] = {
    150, // Capacity
    100, // Number of items
    // Size of items (sorted)
    99,97,96,96,96,95,95,95,95,94,94,93,92,92,92,91,91,91,90,89,89,
    89,88,88,88,87,86,86,85,85,84,83,82,81,81,77,77,76,76,75,73,73,
    73,72,72,72,72,70,69,67,66,65,65,64,62,61,60,58,57,56,55,53,52,
    52,52,48,48,46,45,43,42,39,39,38,38,38,38,37,36,35,34,34,32,31,
    30,30,28,27,27,27,25,24,24,24,23,23,22,22,22,21
  };
  const int n2c3w2_c[] = {
    150, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,99,99,98,97,97,97,96,96,95,95,95,94,93,93,93,92,91,89,88,
    87,86,84,84,83,83,82,81,81,81,78,78,75,74,73,72,72,71,70,68,67,
    66,65,64,63,63,62,60,60,59,59,58,57,56,56,55,54,51,49,49,48,47,
    47,46,45,45,45,45,44,44,44,44,43,41,41,40,39,39,39,37,37,37,35,
    35,34,32,31,31,30,28,26,25,24,24,23,23,22,21,20,20
  };
  const int n2c3w2_d[] = {
    150, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,100,100,99,99,98,97,96,95,95,95,94,94,91,91,90,90,88,86,84,
    83,83,79,78,77,74,74,72,72,70,69,69,69,69,68,68,68,67,67,67,66,
    66,65,64,63,63,63,63,63,62,62,61,60,60,59,59,59,59,57,55,55,55,
    53,53,52,52,51,50,49,48,47,47,45,44,44,43,43,42,42,41,41,38,37,
    36,36,36,36,34,34,29,29,28,27,25,24,23,23,22,22,20
  };
  const int n2c3w2_e[] = {
    150, // Capacity
    100, // Number of items
    // Size of items (sorted)
    99,98,98,98,93,93,92,90,90,89,89,87,85,85,84,81,81,81,80,77,76,
    75,75,74,74,73,71,70,70,69,68,67,67,67,66,66,65,65,64,63,62,62,
    61,61,59,58,57,57,57,56,55,54,54,54,52,52,52,52,52,51,51,50,50,
    50,49,47,47,47,47,47,45,45,44,43,42,42,39,39,39,39,39,39,38,37,
    37,37,34,33,33,32,32,31,31,31,29,28,28,27,25,22
  };
  const int n2c3w2_f[] = {
    150, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,99,99,98,98,97,97,96,95,94,92,92,92,90,86,86,85,85,83,83,
    74,74,73,73,73,72,71,71,71,70,70,70,70,69,69,67,67,66,66,66,66,
    65,65,63,63,62,61,57,56,56,56,55,54,54,53,53,53,51,49,47,47,47,
    46,46,45,44,44,44,42,41,40,40,37,37,35,35,35,35,33,32,32,32,32,
    31,31,30,28,28,27,27,27,26,24,23,22,21,21,21,21,20
  };
  const int n2c3w2_g[] = {
    150, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,99,99,99,97,97,96,96,95,94,94,93,93,92,91,91,90,89,88,88,
    87,87,86,85,84,83,83,83,82,82,78,75,75,73,73,72,72,70,69,69,67,
    67,65,65,63,61,61,60,59,58,58,58,58,57,57,57,55,54,54,54,52,52,
    52,51,48,47,47,47,46,45,45,45,44,42,41,40,37,35,34,31,30,29,27,
    26,26,26,25,25,25,24,24,24,24,23,23,23,23,23,22,20
  };
  const int n2c3w2_h[] = {
    150, // Capacity
    100, // Number of items
    // Size of items (sorted)
    99,98,98,98,96,92,92,91,89,87,86,86,85,85,82,81,81,80,80,77,77,
    76,76,75,74,74,74,73,71,71,69,69,68,68,66,66,65,64,63,63,63,62,
    61,59,59,57,56,55,54,54,53,53,53,51,50,50,49,49,49,48,48,47,47,
    46,44,44,44,43,42,41,36,36,36,36,36,35,33,33,32,32,32,32,30,30,
    30,30,29,28,28,28,25,25,25,24,24,22,22,22,20,20
  };
  const int n2c3w2_i[] = {
    150, // Capacity
    100, // Number of items
    // Size of items (sorted)
    99,99,99,99,98,97,97,97,96,95,95,95,93,93,93,92,92,91,91,91,90,
    90,89,88,87,87,86,84,83,82,81,80,79,79,79,78,78,77,77,76,74,73,
    72,71,70,69,69,68,66,66,65,65,65,64,63,63,63,63,62,61,60,60,59,
    57,57,54,54,52,49,48,48,47,47,47,47,46,46,45,44,43,43,37,37,36,
    36,34,33,32,30,30,30,27,25,22,22,22,21,21,20,20
  };
  const int n2c3w2_j[] = {
    150, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,100,99,99,99,98,97,97,96,96,96,95,94,94,94,93,93,93,91,90,
    89,87,87,86,85,84,83,83,82,81,80,80,80,79,79,78,78,78,78,77,76,
    75,74,72,72,72,71,70,70,69,67,66,66,63,62,60,60,57,56,56,56,56,
    53,52,52,50,50,48,48,45,44,44,44,44,43,40,38,38,38,37,37,37,36,
    36,35,33,32,30,30,28,28,27,27,26,26,25,24,23,22,22
  };
  const int n2c3w2_k[] = {
    150, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,99,99,99,98,98,97,95,95,95,94,94,93,93,93,90,89,87,87,87,
    87,86,85,85,84,84,83,83,82,81,81,80,79,79,78,74,74,73,72,71,71,
    70,70,69,68,67,67,67,66,64,62,62,61,61,59,59,58,56,55,54,52,52,
    52,52,51,50,50,48,48,48,47,47,42,41,39,38,36,34,34,34,34,33,33,
    32,32,32,31,31,30,29,29,27,27,26,26,25,24,23,20,20
  };
  const int n2c3w2_l[] = {
    150, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,100,98,98,96,95,95,93,93,93,92,92,91,91,91,90,90,89,87,87,
    85,85,84,84,82,82,81,80,78,78,75,74,72,72,71,70,69,68,67,66,65,
    65,65,65,64,63,63,63,61,61,61,61,61,61,60,60,59,58,57,57,57,56,
    54,54,53,53,53,52,49,48,47,47,47,45,43,43,42,40,40,40,40,38,36,
    36,34,32,32,29,28,27,27,27,25,23,23,23,22,22,22,21
  };
  const int n2c3w2_m[] = {
    150, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,100,100,98,98,98,97,96,95,95,94,92,92,91,91,91,90,90,89,89,
    89,89,87,87,85,84,84,83,82,81,78,78,78,77,77,77,76,75,74,72,72,
    71,69,69,68,67,67,67,66,65,62,62,62,61,60,60,60,60,60,59,58,58,
    57,55,55,54,52,52,48,46,46,45,45,44,44,43,43,43,42,42,41,41,40,
    40,37,35,33,33,33,32,31,30,29,29,29,25,25,24,23,21
  };
  const int n2c3w2_n[] = {
    150, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,100,98,96,94,94,93,92,92,92,91,91,90,89,89,87,87,85,85,81,
    81,81,80,79,79,78,78,78,78,78,77,77,76,76,76,76,75,75,75,74,73,
    72,72,69,68,67,66,66,65,64,63,62,61,58,56,56,55,55,54,54,51,49,
    49,49,48,47,47,46,44,44,44,43,43,40,39,38,38,38,38,37,37,36,35,
    35,34,32,32,32,31,30,27,27,25,25,24,23,23,22,21,20
  };
  const int n2c3w2_o[] = {
    150, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,99,99,99,98,97,96,95,95,95,94,93,93,93,92,92,91,88,88,88,
    88,87,86,86,85,85,85,85,84,82,82,81,81,81,78,78,77,77,76,76,75,
    72,72,72,71,71,70,68,68,67,66,64,64,63,63,63,63,61,60,60,57,56,
    56,55,55,55,53,53,52,52,51,51,50,49,48,48,47,45,45,43,42,40,39,
    38,38,37,37,37,37,36,34,34,33,33,33,32,31,26,25,21
  };
  const int n2c3w2_p[] = {
    150, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,100,100,100,99,99,98,98,97,96,96,94,94,94,92,91,90,88,87,
    86,85,84,83,82,82,82,81,80,79,75,74,73,72,72,72,72,71,69,68,68,
    67,65,65,65,65,65,64,62,60,60,59,59,58,57,57,57,56,55,54,54,53,
    52,52,49,49,47,45,45,45,43,42,41,41,40,39,39,36,35,34,34,34,33,
    31,31,31,30,30,30,29,28,27,26,26,24,23,22,21,20,20,20
  };
  const int n2c3w2_q[] = {
    150, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,97,95,95,94,94,93,92,92,92,91,89,88,88,88,87,86,86,85,85,
    83,83,82,81,80,75,75,75,74,74,73,73,72,72,69,69,69,69,69,69,68,
    68,68,68,66,65,64,63,63,63,63,61,59,59,58,58,57,56,53,52,50,50,
    49,48,48,46,46,45,44,43,43,42,42,42,42,42,42,41,41,39,38,38,38,
    37,37,35,34,32,31,30,29,28,28,27,25,24,24,22,21,21
  };
  const int n2c3w2_r[] = {
    150, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,98,98,97,97,96,96,96,96,92,91,91,87,86,84,83,82,82,81,81,
    81,81,80,79,79,79,78,78,78,76,76,76,76,76,75,73,73,71,71,70,69,
    69,66,66,65,63,62,61,60,58,57,57,57,55,52,51,49,46,46,46,46,46,
    46,45,45,45,44,43,43,43,42,42,42,41,40,40,37,37,37,35,35,34,34,
    33,32,32,27,27,26,26,25,24,23,22,22,22,21,20,20,20
  };
  const int n2c3w2_s[] = {
    150, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,100,99,99,99,99,98,97,97,97,96,96,95,95,95,94,92,91,91,90,
    90,89,87,84,83,83,83,82,82,82,82,81,80,80,79,79,79,78,78,77,77,
    77,75,74,73,69,68,65,64,64,63,62,62,62,62,62,61,61,60,58,57,56,
    55,51,49,48,47,46,45,45,44,43,42,41,39,38,38,37,36,36,36,35,34,
    34,34,33,33,32,32,31,31,29,28,26,26,25,25,20,20,20
  };
  const int n2c3w2_t[] = {
    150, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,100,99,97,95,95,94,93,93,92,91,90,89,89,88,88,86,86,85,84,
    84,82,82,82,81,81,80,80,79,79,77,77,76,74,74,74,73,72,71,70,69,
    69,69,67,67,66,66,65,64,64,63,63,62,61,61,61,61,60,59,59,59,58,
    57,57,57,57,56,55,54,54,54,51,50,50,50,49,48,47,46,46,45,44,42,
    41,40,40,40,39,38,35,34,29,27,26,25,25,23,23,22,20
  };
  const int n2c3w4_a[] = {
    150, // Capacity
    100, // Number of items
    // Size of items (sorted)
    99,99,98,98,97,97,96,96,96,96,95,94,93,92,91,89,87,87,87,86,85,
    84,84,83,83,83,82,81,80,79,79,79,77,77,76,74,74,74,73,72,72,71,
    71,69,69,69,66,65,64,64,64,63,62,61,60,59,57,57,57,56,56,55,54,
    53,52,52,51,51,49,47,47,46,46,46,46,46,46,44,43,43,43,41,40,40,
    39,39,38,36,36,35,34,34,33,32,32,31,31,30,30,30
  };
  const int n2c3w4_b[] = {
    150, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,99,99,98,98,97,95,95,95,94,94,94,94,93,93,92,91,90,90,90,
    90,89,89,88,86,85,85,84,83,83,82,81,81,80,79,79,77,76,76,73,72,
    71,71,71,69,69,68,67,67,63,61,61,61,60,60,59,58,57,57,57,57,56,
    56,56,56,56,55,53,53,53,51,51,49,48,48,47,47,47,47,46,46,45,45,
    44,44,43,43,42,42,39,38,38,37,36,35,33,32,31,30,30
  };
  const int n2c3w4_c[] = {
    150, // Capacity
    100, // Number of items
    // Size of items (sorted)
    99,99,98,97,96,93,92,92,91,91,91,90,90,90,89,88,88,87,85,85,84,
    84,84,82,80,80,80,80,78,77,76,75,74,73,72,70,70,69,68,68,67,66,
    65,65,65,65,64,62,59,59,59,58,58,57,57,56,56,56,55,55,54,51,51,
    50,49,48,46,46,46,46,46,46,45,44,44,41,41,41,41,40,40,39,39,38,
    37,36,36,36,35,35,35,35,34,34,34,34,32,32,31,30
  };
  const int n2c3w4_d[] = {
    150, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,100,99,99,99,99,98,98,98,97,97,97,94,94,93,93,92,90,89,88,
    87,86,85,83,83,82,81,80,79,78,77,76,75,73,73,73,73,72,72,71,71,
    71,70,68,67,66,65,64,64,64,64,63,62,62,62,61,57,56,55,55,54,53,
    53,53,53,52,52,52,51,51,49,49,48,48,45,45,45,45,44,44,43,42,41,
    41,40,40,38,35,34,34,34,34,33,33,32,32,32,30,30,30
  };
  const int n2c3w4_e[] = {
    150, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,100,99,99,98,98,98,96,96,95,94,94,93,93,92,92,91,91,90,89,
    88,88,88,88,88,87,86,86,85,85,85,85,84,84,84,83,83,83,81,80,80,
    80,79,77,77,75,75,74,72,72,69,68,68,66,65,65,64,64,63,61,61,60,
    60,58,58,58,58,57,57,56,56,55,54,49,49,47,47,47,46,45,44,43,42,
    42,41,40,40,36,34,34,33,33,32,32,32,32,32,31,30,30
  };
  const int n2c3w4_f[] = {
    150, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,100,99,98,97,96,94,93,92,91,90,89,89,87,87,85,85,85,84,84,
    84,83,83,83,83,83,81,81,80,80,79,79,79,78,78,77,76,75,74,74,74,
    73,73,71,71,71,71,70,69,69,68,68,68,66,66,65,64,63,63,63,62,61,
    59,58,58,57,56,56,56,56,55,52,50,49,47,46,46,45,45,43,43,43,42,
    42,41,41,38,37,37,36,36,35,35,34,34,34,33,31,31,30
  };
  const int n2c3w4_g[] = {
    150, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,100,99,98,97,97,95,94,94,94,93,93,91,90,90,89,88,88,86,85,
    85,84,84,84,82,82,82,81,81,81,80,75,75,75,75,74,74,74,73,72,71,
    70,69,69,69,68,67,65,64,64,63,63,63,63,61,61,59,58,58,58,56,56,
    55,54,53,53,53,51,50,49,48,48,46,46,44,44,44,43,43,43,43,42,42,
    42,41,41,40,40,39,39,39,39,38,36,35,35,35,33,32,32
  };
  const int n2c3w4_h[] = {
    150, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,97,97,97,95,95,95,94,94,94,94,93,93,93,92,92,90,89,86,85,
    83,82,82,81,79,78,77,76,75,74,74,74,74,74,73,73,72,71,71,71,70,
    69,68,66,66,65,64,64,64,63,63,62,62,62,61,61,61,59,59,59,58,58,
    57,57,55,54,52,50,49,48,47,46,46,45,45,44,44,44,42,42,41,41,40,
    39,39,39,37,37,36,36,36,35,35,35,32,32,32,31,30,30
  };
  const int n2c3w4_i[] = {
    150, // Capacity
    100, // Number of items
    // Size of items (sorted)
    99,99,99,99,98,97,97,92,92,91,91,90,89,89,88,88,88,86,85,84,83,
    83,81,80,80,80,80,80,79,79,78,77,77,77,77,76,76,75,74,72,72,72,
    71,70,69,69,69,67,67,66,66,66,66,65,64,61,60,59,59,59,58,57,56,
    56,54,53,52,51,51,51,50,50,50,50,49,48,48,47,47,47,45,43,43,43,
    42,41,41,38,37,37,36,35,33,32,32,32,31,31,30,30
  };
  const int n2c3w4_j[] = {
    150, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,100,100,99,99,99,99,98,98,96,96,95,95,93,92,92,91,91,90,88,
    85,84,84,82,81,80,80,76,75,74,73,73,72,71,71,70,69,69,68,67,65,
    65,65,64,64,64,64,63,62,61,61,61,60,57,57,56,56,54,52,52,51,51,
    51,50,48,48,48,47,46,46,46,45,45,45,44,44,44,43,43,43,42,42,41,
    41,41,41,39,39,38,37,36,36,36,34,34,33,33,32,32,31
  };
  const int n2c3w4_k[] = {
    150, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,100,99,98,96,96,95,94,94,94,93,93,93,93,91,91,91,90,90,89,
    89,87,87,87,87,85,84,84,84,83,82,81,81,81,80,79,79,78,78,77,77,
    77,75,75,74,74,74,74,69,68,68,67,67,65,65,64,63,61,59,59,58,58,
    58,58,57,56,55,55,55,54,54,53,53,52,51,50,50,50,49,49,48,48,48,
    48,47,47,43,43,42,40,40,39,37,37,35,34,34,33,31,30
  };
  const int n2c3w4_l[] = {
    150, // Capacity
    100, // Number of items
    // Size of items (sorted)
    99,97,96,95,94,93,92,92,92,91,90,88,88,88,86,86,86,86,85,85,85,
    85,85,83,83,83,82,81,81,80,79,78,76,76,75,75,74,74,74,74,74,73,
    73,72,71,70,70,70,69,68,67,66,65,65,64,64,63,61,61,60,59,58,58,
    58,57,57,57,56,56,56,55,54,54,53,53,53,53,50,48,48,48,46,46,46,
    46,45,43,43,42,41,40,39,37,35,35,34,34,31,31,30
  };
  const int n2c3w4_m[] = {
    150, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,100,100,99,98,98,95,92,91,91,89,89,89,89,88,88,87,86,86,85,
    85,84,84,83,82,82,81,81,81,80,79,79,79,78,78,78,77,76,75,75,74,
    74,73,72,72,70,69,68,68,67,66,65,64,63,62,62,62,60,59,58,56,56,
    55,53,53,53,51,51,50,50,46,44,44,44,44,43,42,42,41,41,40,39,39,
    38,37,37,36,36,36,36,35,35,35,34,33,33,33,32,32,30
  };
  const int n2c3w4_n[] = {
    150, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,99,99,97,96,95,95,94,94,94,93,87,86,85,85,85,85,85,85,85,
    84,84,83,83,82,81,81,80,80,80,80,80,80,79,79,78,77,77,76,76,75,
    75,75,74,72,70,69,68,68,67,67,65,64,64,64,63,62,60,59,59,59,58,
    58,58,57,57,56,56,54,54,52,51,51,48,48,48,47,47,47,46,45,44,44,
    42,41,41,39,38,38,37,36,36,36,35,34,33,33,33,32,31
  };
  const int n2c3w4_o[] = {
    150, // Capacity
    100, // Number of items
    // Size of items (sorted)
    98,98,98,97,97,96,96,96,96,94,94,93,93,93,92,92,92,91,91,90,90,
    89,88,87,87,87,85,85,83,78,77,77,77,77,76,75,74,73,71,71,70,70,
    70,70,70,69,68,68,65,65,64,63,63,61,61,61,61,60,60,59,59,59,59,
    58,58,57,54,54,52,52,52,51,49,49,49,48,47,47,47,45,45,45,43,42,
    42,41,41,40,40,40,40,39,38,37,36,35,34,32,31,30
  };
  const int n2c3w4_p[] = {
    150, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,99,99,98,96,96,96,95,94,92,91,90,90,89,89,88,88,88,88,86,
    86,85,85,85,84,83,83,83,83,82,82,81,80,80,79,79,77,77,77,75,75,
    74,72,71,70,70,70,69,69,69,68,68,67,65,64,64,62,62,61,59,59,57,
    57,54,54,54,54,53,53,52,50,50,49,48,48,48,46,43,42,42,42,39,39,
    38,38,37,37,37,36,36,35,34,34,34,34,33,32,32,30,30
  };
  const int n2c3w4_q[] = {
    150, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,99,98,98,98,97,97,97,96,96,96,95,95,95,94,93,93,93,92,91,
    91,88,88,87,87,86,85,85,84,82,81,79,79,79,78,78,77,77,76,76,75,
    73,73,73,73,72,72,72,71,70,69,68,67,66,65,65,64,63,62,61,61,60,
    60,59,59,57,56,55,54,54,53,53,52,51,50,50,50,49,49,48,48,47,47,
    47,46,45,45,45,44,38,35,35,35,34,34,34,33,33,31,31
  };
  const int n2c3w4_r[] = {
    150, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,98,98,98,98,98,97,97,96,95,95,93,92,90,89,87,86,86,84,84,
    84,84,80,80,80,79,79,78,77,74,73,73,72,72,72,71,71,71,70,69,69,
    69,68,67,66,65,64,64,63,63,62,60,57,57,57,55,55,55,54,53,53,52,
    52,52,51,51,50,49,47,46,46,45,44,44,44,43,43,43,42,41,41,41,41,
    40,40,39,39,39,39,38,38,37,36,35,35,34,32,31,30,30
  };
  const int n2c3w4_s[] = {
    150, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,99,98,97,97,96,95,94,94,93,92,91,90,90,88,88,88,87,84,81,
    80,80,79,79,76,76,75,75,75,73,73,71,71,71,70,70,70,69,69,67,67,
    66,65,64,64,62,61,60,60,59,59,59,59,58,56,55,54,54,53,53,53,51,
    51,50,49,48,48,48,47,47,47,46,46,45,45,45,45,45,44,44,44,42,42,
    41,41,40,39,38,37,34,34,34,33,33,32,32,31,31,31,30
  };
  const int n2c3w4_t[] = {
    150, // Capacity
    100, // Number of items
    // Size of items (sorted)
    100,100,99,99,97,97,95,95,95,94,94,93,93,93,92,91,91,91,91,91,
    89,89,86,86,85,85,84,82,81,81,79,79,78,76,75,74,74,74,74,73,73,
    71,70,70,69,69,67,67,67,66,66,66,66,65,65,64,64,63,63,62,61,61,
    61,60,60,58,57,54,54,53,53,53,52,52,51,50,48,48,47,46,46,46,45,
    44,42,40,39,39,39,37,36,35,34,33,33,33,32,32,30,30
  };
  const int n3c1w1_a[] = {
    100, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,99,99,97,97,97,94,93,92,92,91,89,89,88,88,88,88,87,87,86,
    86,86,86,86,85,84,83,83,82,81,81,81,81,80,80,79,79,79,78,78,77,
    77,77,76,76,76,75,74,74,73,73,73,73,72,72,72,72,72,71,71,69,69,
    68,67,67,66,66,66,66,64,64,64,64,63,63,62,61,61,61,60,60,59,59,
    57,56,56,56,55,55,55,54,54,53,53,52,52,52,51,50,50,50,49,49,49,
    49,47,47,46,46,46,46,46,46,45,45,45,45,44,44,42,41,40,40,40,39,
    39,38,38,38,38,38,38,37,37,36,36,36,36,34,34,34,34,34,34,31,31,
    31,30,30,30,30,30,29,29,27,27,27,26,24,24,23,22,22,22,22,22,20,
    18,17,17,17,16,16,15,15,14,14,14,13,13,12,11,11,11,10,10,8,8,
    8,6,6,5,5,4,4,3,3,3,1,1
  };
  const int n3c1w1_b[] = {
    100, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,99,99,99,98,98,98,95,93,93,92,92,92,92,91,
    90,90,89,89,89,89,88,88,88,88,87,86,86,86,86,86,85,85,85,84,84,
    84,83,83,81,81,80,79,77,77,77,75,75,75,75,74,74,74,74,73,73,73,
    72,71,71,71,71,70,70,70,70,70,69,68,68,68,68,68,67,67,67,66,65,
    65,65,64,64,63,63,63,62,61,61,60,60,59,59,59,58,58,57,57,57,56,
    53,53,53,52,52,52,52,51,50,49,49,48,48,48,47,46,45,44,44,44,44,
    42,42,41,40,40,40,39,39,39,38,38,38,37,37,36,36,36,36,34,34,33,
    33,33,33,33,33,32,32,32,32,31,30,29,28,27,27,26,26,26,25,24,23,
    21,21,20,20,17,16,16,15,14,14,14,13,13,13,13,13,12,12,11,11,10,
    9,9,7,7,7,7,6,5,5,4,4,3,3
  };
  const int n3c1w1_c[] = {
    100, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,100,99,99,99,97,96,96,95,95,94,92,92,91,91,91,91,90,90,
    90,89,89,88,88,87,86,86,85,85,85,83,82,82,82,81,81,80,80,80,79,
    79,79,76,75,75,74,74,73,72,72,72,71,71,70,68,67,67,67,67,66,66,
    65,65,64,64,64,63,63,63,62,62,62,61,61,60,60,59,59,59,59,58,58,
    57,57,56,56,56,56,55,55,54,52,51,51,50,50,49,48,48,47,47,47,47,
    46,46,43,43,42,42,42,41,41,40,40,40,39,37,37,36,36,34,34,34,34,
    33,33,33,32,31,30,30,29,29,28,28,27,27,26,26,26,26,25,25,24,24,
    23,23,23,23,22,22,21,21,21,20,20,20,20,19,19,18,17,17,16,16,15,
    14,14,14,14,14,13,13,12,12,11,11,11,11,10,9,9,8,8,8,8,7,7,7,6,
    6,6,5,4,4,4,2,2,1
  };
  const int n3c1w1_d[] = {
    100, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,99,99,99,98,97,97,97,96,96,95,95,95,94,94,93,93,93,93,93,
    92,92,91,90,89,89,89,88,87,87,87,87,87,87,87,86,85,84,84,83,82,
    80,80,80,80,79,79,78,78,77,76,76,74,74,74,74,73,73,71,70,69,69,
    68,68,68,68,68,68,67,67,66,66,66,65,64,63,63,62,62,62,61,61,61,
    60,60,60,60,59,59,58,57,57,57,57,55,55,54,54,53,53,53,51,51,51,
    50,49,49,48,48,48,48,47,46,46,46,45,45,45,43,43,43,42,42,42,42,
    42,41,41,40,39,38,37,37,37,37,37,36,36,35,35,35,35,34,34,34,32,
    31,31,30,29,29,28,28,26,26,26,25,24,24,24,23,22,21,21,21,20,20,
    20,19,19,19,19,19,19,17,14,13,12,12,11,10,10,10,9,9,8,8,8,8,7,
    6,6,5,5,5,4,3,2,2,2
  };
  const int n3c1w1_e[] = {
    100, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,100,100,98,98,97,97,96,96,95,95,95,95,94,93,93,93,91,
    91,91,91,91,91,90,90,87,87,86,85,85,85,84,84,82,81,81,81,79,78,
    78,76,76,75,75,75,75,74,74,74,72,72,72,72,71,70,69,69,69,69,67,
    67,67,67,66,66,66,65,64,64,64,64,63,62,61,61,60,60,59,58,57,56,
    55,55,55,54,53,53,53,52,52,50,50,49,47,47,46,46,45,44,44,43,43,
    42,42,41,41,41,40,40,39,39,39,39,38,38,38,37,36,35,35,34,34,33,
    33,32,32,32,32,32,32,31,31,31,30,30,30,30,30,29,28,28,27,27,26,
    25,24,24,24,23,23,23,23,22,22,22,21,21,21,20,19,19,19,18,18,17,
    17,16,16,15,15,14,14,13,12,12,11,10,10,9,8,8,8,8,7,7,7,7,6,6,
    5,4,3,3,3,3,2,2,1,1
  };
  const int n3c1w1_f[] = {
    100, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,99,99,99,98,98,98,97,97,97,97,96,96,95,94,94,94,94,94,
    94,93,93,93,93,93,92,91,90,90,90,90,89,87,86,86,86,85,85,85,85,
    85,84,83,83,83,82,82,81,81,80,80,78,77,76,76,76,75,75,74,74,74,
    74,74,73,72,71,71,70,70,70,69,69,68,68,68,67,67,67,67,66,66,65,
    64,63,63,62,61,61,61,60,60,60,60,60,60,59,59,58,58,58,57,57,56,
    56,54,54,53,53,50,50,49,49,49,48,48,48,46,46,46,45,44,42,41,40,
    40,37,37,37,36,36,34,33,32,32,31,30,29,28,28,27,27,27,26,25,25,
    25,24,24,23,23,23,23,23,23,23,22,22,21,21,20,20,20,19,18,17,16,
    16,15,15,14,14,14,13,12,12,12,11,10,10,10,10,9,8,8,8,8,7,7,7,
    7,6,5,5,5,5,4,3,2,1
  };
  const int n3c1w1_g[] = {
    100, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,99,99,98,98,97,95,95,94,94,93,93,93,93,92,91,91,91,91,90,
    90,90,89,89,89,88,88,87,87,86,86,86,86,86,85,85,84,84,84,83,82,
    81,81,80,80,79,79,79,78,77,77,76,76,75,75,74,74,74,74,73,73,73,
    73,73,72,72,72,71,70,70,69,69,68,68,68,67,67,66,62,62,62,62,62,
    62,61,60,60,60,60,60,59,58,57,57,57,57,56,56,54,54,53,53,52,52,
    52,52,52,51,50,50,50,49,49,49,48,47,46,46,46,45,44,43,43,42,42,
    40,40,40,39,39,38,36,36,36,35,35,34,33,33,32,32,32,31,30,30,29,
    29,29,28,27,27,26,26,26,25,25,25,24,24,24,24,23,23,23,22,22,22,
    22,21,20,20,19,16,15,15,14,14,14,13,11,11,10,10,10,9,9,7,6,6,
    5,5,5,4,4,3,2,1,1,1,1
  };
  const int n3c1w1_h[] = {
    100, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,99,99,97,97,97,97,97,97,96,96,96,96,95,95,95,95,94,93,
    93,93,92,92,91,90,89,89,88,88,88,87,87,87,86,86,85,85,84,84,83,
    83,82,81,80,80,80,79,79,79,78,77,77,77,77,76,75,75,74,74,73,72,
    71,71,71,71,71,71,71,69,69,69,68,65,65,63,63,62,62,62,62,61,61,
    60,60,59,58,58,58,56,56,56,54,53,53,52,51,51,51,50,49,49,48,48,
    48,47,46,46,46,46,46,46,43,43,42,41,40,39,39,38,37,37,36,36,36,
    35,34,34,33,33,32,32,32,32,32,32,32,30,30,29,29,28,27,27,27,27,
    26,26,26,26,25,25,24,24,23,22,21,21,21,21,20,19,19,18,17,17,17,
    16,16,16,15,15,15,14,14,13,12,11,11,10,9,9,7,6,6,6,6,6,4,4,4,
    4,4,3,2,1,1,1,1,1
  };
  const int n3c1w1_i[] = {
    100, // Capacity
    200, // Number of items
    // Size of items (sorted)
    99,97,97,96,96,95,93,92,92,92,92,92,92,92,91,91,90,89,88,87,87,
    87,86,85,85,84,84,84,83,83,83,83,83,83,82,81,80,79,78,78,78,78,
    77,77,76,76,76,75,75,75,74,73,72,71,71,70,70,69,69,68,68,67,66,
    66,65,65,63,63,63,63,62,61,61,61,59,58,58,58,58,58,58,58,58,57,
    56,56,56,54,53,52,52,52,51,50,50,50,50,50,49,49,48,48,48,48,48,
    47,47,46,45,45,44,43,43,43,43,43,43,42,41,41,40,40,38,38,37,37,
    37,37,36,36,36,35,35,34,33,32,32,31,31,29,29,29,28,27,27,27,26,
    26,25,24,24,23,22,22,22,21,21,21,20,20,19,18,18,18,18,17,16,16,
    16,16,15,15,14,14,14,13,13,12,12,11,11,11,11,8,8,7,6,5,3,3,2,
    2,2,2,2,2,1,1,1,1
  };
  const int n3c1w1_j[] = {
    100, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,99,98,97,97,97,97,97,96,96,95,95,93,93,93,92,92,91,91,
    89,88,88,88,88,88,86,86,85,85,85,84,83,83,83,82,81,80,79,79,78,
    78,77,77,75,74,74,74,73,73,72,72,72,71,71,71,70,70,70,70,69,69,
    67,67,66,66,65,65,65,64,64,64,63,63,63,62,62,62,61,60,60,59,59,
    59,59,59,58,58,57,57,57,56,56,55,55,55,55,54,54,52,52,52,51,51,
    51,50,50,50,49,49,49,49,48,47,47,47,45,44,44,44,43,43,43,43,43,
    41,41,41,40,40,39,39,39,39,38,37,37,37,36,36,36,35,35,34,33,33,
    31,31,30,29,28,28,28,27,27,25,25,24,23,23,23,22,22,21,21,21,19,
    19,19,17,17,17,17,16,16,15,14,14,14,14,13,13,12,11,10,10,10,9,
    9,9,8,7,6,6,4,4,3,3,3,2
  };
  const int n3c1w1_k[] = {
    100, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,99,99,99,98,98,98,98,97,95,95,95,95,94,94,92,92,92,92,91,
    90,88,88,88,88,87,87,87,86,85,84,84,83,83,83,82,82,82,82,81,81,
    81,81,80,80,80,79,78,77,75,75,74,74,74,73,73,72,72,71,71,70,70,
    70,69,68,68,68,68,67,67,66,66,65,64,63,62,61,60,60,58,58,57,57,
    56,56,55,55,55,55,55,55,54,53,53,53,52,51,50,49,49,49,48,48,48,
    48,47,47,47,46,45,43,43,42,42,42,42,41,41,41,41,40,40,39,39,38,
    38,38,38,36,35,35,34,33,32,32,30,28,28,28,28,28,26,26,25,25,24,
    24,23,23,23,22,22,22,22,21,21,21,21,20,20,20,19,19,19,18,17,17,
    16,15,15,14,14,13,13,12,12,11,11,11,10,9,9,9,8,7,6,6,5,5,4,4,
    4,3,3,3,2,2,2,2,1
  };
  const int n3c1w1_l[] = {
    100, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,99,99,99,99,97,96,96,94,94,94,93,93,93,93,92,92,92,89,
    88,87,87,85,84,84,84,84,83,83,83,83,82,80,80,79,79,78,76,75,75,
    75,74,73,73,73,73,73,72,72,72,71,71,70,70,70,70,70,69,69,69,68,
    67,67,66,66,64,63,63,63,62,62,61,61,59,59,59,59,58,58,57,56,56,
    55,55,54,53,52,52,51,51,50,50,50,50,50,50,48,48,48,48,47,47,47,
    46,46,46,46,45,44,43,41,41,39,39,38,37,37,37,36,36,35,35,35,34,
    34,33,33,33,32,32,31,31,31,31,30,30,30,29,29,28,28,25,25,25,25,
    24,24,24,23,23,23,23,22,21,20,20,20,20,19,18,18,18,16,16,16,15,
    14,14,14,14,13,12,11,11,11,11,11,10,10,9,9,9,8,8,8,7,7,7,6,4,
    4,3,3,2,2,2,1,1,1
  };
  const int n3c1w1_m[] = {
    100, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,99,99,98,98,97,97,97,97,97,96,96,96,96,95,95,94,92,92,92,
    92,91,91,91,90,90,90,89,87,87,86,85,85,83,83,83,82,82,80,78,78,
    78,77,77,77,77,76,76,75,75,74,74,74,74,72,71,71,71,70,70,69,69,
    69,68,67,67,67,67,66,66,66,66,65,65,65,65,64,63,61,61,60,60,60,
    59,59,58,58,58,57,55,54,54,54,54,54,54,54,54,52,52,52,52,51,51,
    51,51,49,47,47,46,46,45,44,44,44,44,44,43,42,42,42,41,41,41,41,
    40,39,38,37,37,35,35,35,33,32,31,30,30,29,29,29,28,28,27,27,26,
    26,25,25,25,24,23,23,23,23,23,21,21,20,19,19,19,18,18,18,17,17,
    17,17,16,16,16,15,15,15,15,15,14,14,13,12,12,11,11,10,10,10,10,
    10,9,7,6,6,5,5,4,3,2,1,1
  };
  const int n3c1w1_n[] = {
    100, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,99,99,99,98,98,97,96,95,95,93,93,93,91,90,90,88,88,87,
    84,82,82,81,81,81,81,81,81,80,80,79,79,78,78,77,77,77,77,76,75,
    75,74,73,73,72,71,71,71,70,70,70,69,67,66,66,66,66,66,65,65,65,
    64,64,63,59,59,59,59,58,58,56,56,54,54,53,53,53,51,51,51,51,50,
    49,48,48,48,48,47,47,47,47,46,46,46,46,46,46,46,46,46,46,45,44,
    44,44,43,41,41,40,40,40,39,39,39,38,36,36,35,34,34,34,33,33,33,
    32,32,32,32,31,31,31,30,30,29,28,28,27,27,27,26,25,25,24,24,23,
    23,22,22,22,22,21,21,21,20,19,19,18,16,16,16,15,15,15,15,15,15,
    14,13,13,13,12,12,12,12,11,10,10,10,9,9,9,8,8,8,8,7,7,7,7,7,5,
    5,4,3,3,3,2,2,2
  };
  const int n3c1w1_o[] = {
    100, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,99,98,98,98,97,96,96,95,95,95,94,92,91,91,90,90,89,89,89,
    87,87,86,86,86,86,86,84,84,83,83,83,82,82,82,82,81,79,79,78,77,
    77,76,76,76,76,76,76,76,76,76,76,75,74,73,72,72,71,69,69,67,66,
    66,66,65,65,64,64,63,63,63,63,62,60,60,60,59,59,57,56,56,55,54,
    54,54,54,54,53,52,52,52,51,51,51,50,48,48,47,47,46,45,45,45,45,
    45,42,42,41,41,41,40,40,39,39,38,38,37,37,37,36,35,35,35,34,34,
    34,34,31,30,30,30,29,29,29,29,29,29,28,28,28,28,28,26,26,26,25,
    25,25,24,24,24,23,22,22,22,22,21,21,21,21,21,20,19,19,19,18,18,
    18,18,18,17,17,16,16,16,16,15,14,14,14,13,13,12,12,11,10,10,9,
    8,8,8,7,7,6,6,5,4,4,3,2
  };
  const int n3c1w1_p[] = {
    100, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,99,98,98,98,97,97,97,97,96,96,95,92,92,92,
    92,91,91,91,91,90,89,89,87,87,87,86,86,86,86,86,85,85,85,84,84,
    84,83,83,83,82,82,82,81,81,81,79,78,77,77,76,75,75,75,75,75,72,
    72,72,72,72,72,72,71,71,71,71,70,70,70,69,68,65,64,64,64,63,63,
    62,62,61,60,60,59,59,59,59,59,58,58,57,57,57,57,56,56,55,53,53,
    52,52,51,51,50,48,48,48,47,46,46,46,44,44,43,43,42,42,41,41,38,
    38,37,37,37,37,36,35,35,34,33,33,33,32,32,31,30,30,30,29,29,28,
    28,28,28,27,26,25,25,25,24,24,23,23,23,22,22,22,21,21,21,21,21,
    20,19,18,18,17,16,16,16,16,16,16,15,15,14,14,13,13,13,13,12,12,
    11,9,9,8,8,7,7,6,4,2,2,2,2
  };
  const int n3c1w1_q[] = {
    100, // Capacity
    200, // Number of items
    // Size of items (sorted)
    99,98,97,95,95,93,93,93,93,93,92,92,92,92,92,92,91,91,90,90,90,
    90,89,88,87,85,85,85,85,85,84,84,83,82,82,81,81,80,79,79,79,79,
    78,78,77,77,77,76,76,76,76,75,74,74,73,72,72,71,71,70,70,70,70,
    69,69,67,67,66,66,65,65,65,64,63,61,60,60,59,58,54,53,53,52,52,
    51,51,50,50,50,49,48,48,48,48,47,46,46,46,46,45,45,43,42,42,42,
    42,41,41,41,40,40,39,38,38,37,36,36,36,35,35,35,35,34,34,34,33,
    32,32,32,31,31,31,31,30,30,29,28,27,27,27,26,25,25,25,24,23,23,
    23,23,23,23,22,22,21,21,21,20,20,20,20,20,19,19,18,17,17,17,17,
    17,16,16,16,15,14,14,14,14,13,12,11,11,11,11,11,8,7,7,7,5,5,5,
    4,3,2,2,2,2,2,1,1
  };
  const int n3c1w1_r[] = {
    100, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,99,99,98,98,98,97,97,96,96,95,95,94,94,94,92,92,91,90,
    90,89,89,87,86,86,85,84,84,84,83,82,82,81,80,80,79,79,79,78,78,
    78,77,77,77,77,77,77,76,76,75,75,75,74,74,73,73,72,72,71,67,67,
    67,67,66,65,65,65,64,64,63,62,61,61,60,60,59,59,59,58,58,58,58,
    58,58,57,57,56,56,56,55,54,54,53,52,52,50,50,50,49,47,46,45,45,
    45,44,43,43,41,41,41,40,40,40,40,39,39,38,38,38,38,38,37,36,35,
    35,35,34,33,33,32,30,30,30,30,28,28,27,27,27,26,26,26,25,25,25,
    24,24,24,24,23,22,21,21,20,20,19,19,19,19,19,18,16,16,16,16,15,
    15,14,14,14,14,14,12,11,11,11,10,10,10,9,8,8,8,7,7,6,6,6,6,6,
    5,5,3,2,2,1,1,1,1
  };
  const int n3c1w1_s[] = {
    100, // Capacity
    200, // Number of items
    // Size of items (sorted)
    99,99,98,97,97,97,97,96,96,96,95,95,93,93,92,92,90,89,88,88,88,
    88,87,87,86,86,86,86,86,86,85,84,83,83,83,82,82,82,81,81,81,80,
    80,80,80,78,77,76,76,74,73,72,71,71,71,70,70,70,70,69,69,69,69,
    67,66,66,65,65,64,63,63,63,62,62,62,61,61,61,61,59,58,58,56,56,
    54,52,52,51,51,51,50,50,50,50,50,49,49,48,48,47,47,45,45,44,44,
    44,44,44,43,42,42,42,42,42,41,39,38,38,38,37,36,36,36,36,35,35,
    35,34,33,33,32,31,31,31,31,31,31,30,30,29,29,28,28,28,27,27,27,
    26,25,25,25,24,24,23,23,23,22,21,21,21,20,20,20,19,19,17,17,17,
    17,16,15,15,15,14,14,14,14,13,11,11,10,10,10,9,9,8,8,8,8,7,7,
    6,6,4,3,3,2,1,1,1
  };
  const int n3c1w1_t[] = {
    100, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,100,99,99,98,97,96,96,96,96,95,94,94,93,92,92,92,91,91,
    91,90,90,89,88,87,87,87,87,87,86,86,86,85,84,83,83,83,83,82,82,
    81,81,81,81,80,80,79,79,79,78,78,78,78,78,76,76,76,76,76,76,75,
    74,74,74,73,73,72,71,69,69,69,67,66,65,64,63,63,63,62,61,61,60,
    59,57,57,56,56,56,55,55,54,54,54,54,54,53,53,52,52,51,50,48,48,
    48,48,47,46,46,45,45,45,43,42,40,40,40,39,39,39,39,38,38,37,37,
    37,36,35,34,32,31,31,30,30,29,28,27,27,26,25,24,24,24,24,24,22,
    22,21,21,21,21,20,19,19,18,18,18,18,18,17,16,16,16,15,15,14,14,
    13,13,12,12,12,12,11,11,11,11,10,9,9,8,7,6,6,6,6,6,6,5,5,5,4,
    4,3,3,3,3,2,1,1
  };
  const int n3c1w2_a[] = {
    100, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,99,99,99,98,98,98,98,98,97,97,96,96,96,95,94,94,93,93,
    91,91,91,90,90,90,89,89,88,88,88,88,87,87,86,85,85,84,83,83,83,
    83,82,81,79,79,79,79,78,78,77,77,77,76,76,76,76,75,75,74,73,73,
    73,72,72,72,71,71,71,70,70,69,69,69,69,69,68,68,68,67,67,67,67,
    65,65,65,65,65,64,63,63,63,63,61,61,61,61,61,60,60,60,59,59,59,
    58,58,58,57,56,56,55,55,55,55,54,54,54,53,53,51,51,50,50,50,50,
    49,49,48,48,48,48,47,46,46,45,44,43,43,42,42,41,40,40,40,40,40,
    39,38,38,38,38,37,36,36,35,35,34,34,34,33,33,33,33,33,33,32,32,
    32,32,32,32,32,31,31,30,28,27,26,26,25,25,24,24,23,23,22,22,22,
    21,21,21,20,20,20,20,20,20,20,20,20
  };
  const int n3c1w2_b[] = {
    100, // Capacity
    200, // Number of items
    // Size of items (sorted)
    99,99,99,97,96,95,94,93,93,93,93,93,91,91,91,90,89,89,89,89,88,
    88,87,87,85,85,84,84,84,84,82,81,81,81,80,80,79,78,78,77,77,76,
    76,76,76,75,75,74,74,74,74,74,74,73,73,73,72,72,72,72,72,71,71,
    70,69,69,69,69,68,68,68,67,67,67,67,67,67,67,66,66,66,65,65,65,
    64,64,64,63,63,62,61,61,60,59,59,58,58,58,58,58,58,58,57,57,57,
    57,56,56,55,55,54,54,54,54,54,53,53,53,53,53,52,52,52,51,51,50,
    49,48,48,48,47,47,46,46,46,45,45,44,43,43,42,41,40,40,38,38,38,
    38,38,37,36,36,36,36,36,36,36,36,35,35,35,34,34,33,33,33,33,32,
    32,32,32,31,31,31,30,30,29,29,28,28,27,27,27,26,26,25,25,23,22,
    21,21,21,21,21,21,21,20,20,20,20
  };
  const int n3c1w2_c[] = {
    100, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,100,99,99,98,98,98,96,96,96,95,95,94,94,94,93,93,92,92,
    92,91,91,90,90,90,89,89,89,89,88,88,87,87,86,86,85,85,85,85,84,
    84,83,82,82,82,82,81,81,81,81,81,80,80,79,79,78,78,78,78,77,76,
    76,76,75,74,74,74,73,72,72,71,71,71,70,70,70,70,69,68,68,68,66,
    66,66,65,65,65,65,63,62,61,61,60,60,60,60,58,58,58,58,57,57,57,
    57,56,56,55,54,54,53,52,52,52,52,52,52,52,52,52,51,51,50,50,49,
    48,47,47,47,47,46,45,45,45,45,45,44,43,43,42,42,42,41,41,41,41,
    40,40,39,39,39,38,37,37,37,36,36,36,35,35,35,34,34,33,33,33,32,
    32,32,32,31,31,31,30,30,28,28,28,28,28,27,27,27,26,26,26,24,24,
    23,23,23,23,22,22,22,21,21,20,20,20
  };
  const int n3c1w2_d[] = {
    100, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,100,99,98,98,98,97,97,97,97,96,96,96,96,95,95,95,94,94,
    94,94,93,93,92,92,92,91,91,91,91,90,90,89,87,87,86,86,85,84,84,
    83,83,82,81,81,81,80,80,79,79,79,79,79,79,78,78,78,78,77,77,77,
    77,77,76,76,76,76,75,75,75,74,74,73,73,73,73,73,72,72,72,71,71,
    71,70,70,70,69,69,69,69,69,68,67,67,67,66,65,65,65,65,64,63,63,
    63,63,62,62,62,61,61,61,60,59,59,59,59,59,58,57,57,57,57,57,56,
    56,55,54,54,53,53,53,53,53,52,52,52,51,50,48,48,47,47,47,47,46,
    46,44,44,44,43,43,42,41,41,41,41,40,40,39,38,37,36,36,36,36,35,
    34,34,33,33,32,31,31,31,30,30,29,29,28,28,28,27,27,27,27,26,25,
    25,24,24,23,23,22,22,22,22,21,21,20
  };
  const int n3c1w2_e[] = {
    100, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,99,99,98,98,97,97,97,96,96,96,95,95,95,95,94,94,94,93,
    93,92,91,91,90,89,89,89,89,88,88,87,87,87,87,86,86,86,85,85,85,
    84,84,83,83,82,82,82,81,81,81,81,80,80,79,79,79,78,77,77,77,76,
    76,76,76,74,73,73,73,73,73,73,73,73,72,72,72,72,71,71,70,70,70,
    70,70,68,68,68,68,67,66,66,66,66,66,65,64,63,63,63,62,61,61,61,
    61,61,60,60,59,59,59,58,58,57,57,57,56,56,56,55,54,54,53,53,53,
    52,52,51,50,50,49,49,49,48,47,47,47,46,45,45,44,44,43,43,43,43,
    43,42,42,42,42,41,41,41,41,40,40,39,39,38,37,36,36,35,35,34,34,
    34,33,33,33,32,30,30,30,29,29,28,28,28,28,28,27,27,27,26,25,25,
    24,24,23,23,23,22,22,22,21,21,20,20
  };
  const int n3c1w2_f[] = {
    100, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,99,98,98,98,98,97,97,97,96,96,96,95,94,94,93,93,92,91,91,
    90,90,90,90,89,88,88,88,87,87,86,86,85,85,84,84,83,82,81,81,80,
    79,79,79,78,78,78,78,78,78,78,78,77,77,77,77,76,76,75,75,74,74,
    74,73,73,73,72,71,71,70,70,69,69,69,68,68,67,65,65,65,65,65,65,
    64,64,63,63,62,62,62,62,62,61,61,61,61,60,59,59,58,58,58,57,57,
    56,56,56,56,54,54,54,52,52,52,52,52,50,50,50,49,49,47,47,47,46,
    46,46,45,45,45,45,45,44,44,44,43,43,43,43,42,42,42,42,41,41,40,
    39,39,38,38,37,37,37,37,37,37,36,36,35,35,35,35,35,34,34,34,33,
    33,33,33,32,32,32,31,31,31,30,30,30,28,28,27,26,23,22,22,22,22,
    22,21,21,21,21,20,20,20,20,20,20,20
  };
  const int n3c1w2_g[] = {
    100, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,100,100,99,99,99,98,98,98,97,96,96,96,96,95,95,95,95,
    94,94,94,94,94,93,93,93,92,92,92,92,92,92,92,92,91,91,90,89,88,
    88,88,88,87,87,87,87,87,86,85,85,85,85,85,84,83,83,83,83,82,81,
    81,80,80,80,80,80,79,79,78,78,78,77,77,77,77,76,75,75,74,74,73,
    72,72,71,69,69,69,69,69,68,68,67,67,66,64,63,62,62,62,62,61,61,
    61,61,60,59,58,58,58,57,57,57,57,56,56,55,54,54,54,53,52,51,51,
    51,50,50,50,50,50,49,47,47,46,44,43,43,42,42,42,42,42,42,42,42,
    41,41,41,40,40,39,39,38,38,37,37,37,36,36,36,36,36,35,35,35,34,
    33,33,33,32,32,32,31,30,30,30,30,30,29,29,28,28,28,27,27,26,26,
    25,25,24,24,23,23,22,22,22,22,22,21,20
  };
  const int n3c1w2_h[] = {
    100, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,99,99,99,99,99,98,97,97,96,96,96,96,95,95,94,94,94,94,
    93,93,93,91,91,91,91,91,90,90,90,90,90,90,90,89,89,89,89,89,88,
    88,88,87,86,86,86,85,85,85,84,84,84,84,83,83,83,81,81,80,80,80,
    80,80,79,79,78,78,77,77,76,76,75,75,75,74,73,73,72,71,71,70,70,
    70,70,69,68,68,67,67,67,65,65,65,64,64,62,62,62,62,61,61,60,60,
    59,59,58,58,58,57,57,57,57,56,56,55,55,55,54,54,52,51,50,50,49,
    48,48,48,48,47,47,46,45,45,43,43,43,42,42,41,41,41,40,40,40,40,
    39,39,38,38,38,37,37,36,35,35,35,35,34,34,34,34,33,33,32,32,32,
    31,31,30,30,30,30,28,28,28,27,27,27,26,26,26,26,25,25,25,25,25,
    25,24,24,24,24,24,23,22,20,20,20,20
  };
  const int n3c1w2_i[] = {
    100, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,100,100,98,97,97,97,96,95,95,95,94,93,93,92,92,92,92,
    91,91,91,90,90,90,88,88,88,87,87,87,87,86,86,85,85,84,84,84,83,
    83,83,83,83,82,82,82,82,82,82,81,81,80,80,79,79,79,78,78,77,77,
    76,75,74,74,72,72,72,71,71,71,69,69,69,68,68,68,68,68,68,67,67,
    66,65,65,65,64,64,64,64,63,63,63,62,62,62,62,61,61,60,60,59,59,
    59,59,59,58,58,57,57,57,56,56,56,55,55,54,53,53,52,52,51,51,51,
    51,50,49,49,49,48,46,46,45,45,45,45,44,44,44,43,42,42,42,42,41,
    41,41,41,40,40,40,39,39,38,38,38,38,37,37,36,35,34,34,34,33,33,
    32,31,31,31,30,30,30,29,29,29,29,27,27,27,26,25,25,25,24,24,24,
    23,23,23,23,23,22,22,21,20,20,20,20,20
  };
  const int n3c1w2_j[] = {
    100, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,100,100,99,99,98,98,98,97,97,97,96,96,96,95,95,94,94,
    93,93,93,93,93,93,92,92,91,89,88,88,88,88,88,87,87,87,87,87,87,
    86,85,85,85,84,83,83,82,82,82,81,80,80,80,80,80,79,79,79,78,77,
    77,76,76,76,76,76,75,75,75,75,74,73,73,73,72,71,71,71,71,70,69,
    69,68,68,68,68,67,65,65,65,62,62,60,60,60,60,60,59,59,59,59,59,
    58,58,58,58,58,57,56,55,55,54,54,53,53,53,53,52,50,50,49,49,49,
    48,48,48,47,47,46,46,46,45,45,45,43,43,43,42,42,42,41,41,41,41,
    40,40,40,40,39,39,37,37,37,37,37,36,36,36,35,34,33,33,32,32,32,
    30,30,30,30,29,29,29,29,29,28,27,27,26,26,25,25,25,25,24,24,24,
    24,24,23,23,23,22,22,21,21,21,20,20,20
  };
  const int n3c1w2_k[] = {
    100, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,99,99,98,98,98,98,97,96,96,95,95,95,95,94,93,93,93,93,
    92,92,91,91,90,90,89,89,89,89,89,88,87,87,85,85,84,84,84,84,84,
    83,83,83,82,82,82,78,78,77,77,77,77,77,76,76,76,75,74,73,73,72,
    72,71,70,70,70,69,69,68,67,67,66,66,66,65,64,64,64,63,63,63,63,
    63,62,61,60,60,60,59,59,59,59,57,57,56,56,55,55,54,53,53,53,53,
    52,52,52,51,51,50,50,49,49,49,48,47,47,47,47,47,46,46,46,45,44,
    44,43,43,43,43,43,43,42,42,42,41,41,40,40,40,40,40,39,39,39,38,
    38,38,38,37,37,37,36,36,36,36,34,33,33,32,32,32,32,32,31,31,31,
    30,30,30,30,30,29,29,28,28,28,28,28,28,27,27,27,26,26,26,26,25,
    25,24,24,23,22,21,21,21,20,20,20,20
  };
  const int n3c1w2_l[] = {
    100, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,99,99,99,98,98,98,98,97,97,97,97,97,96,96,95,95,94,94,
    94,94,93,92,92,92,92,92,92,92,91,91,90,90,90,90,89,89,89,88,88,
    88,87,87,86,86,86,86,85,85,85,84,84,84,83,83,82,81,80,80,79,79,
    78,77,77,77,76,76,76,76,75,75,74,74,74,74,73,73,72,72,71,71,71,
    71,70,70,70,69,69,68,68,68,67,67,67,66,66,66,66,65,64,64,63,63,
    63,62,61,60,60,60,60,59,59,59,59,58,58,58,57,57,56,55,55,54,54,
    54,52,52,52,51,51,51,51,50,49,49,48,48,47,47,47,47,47,46,46,45,
    45,45,44,44,44,43,43,43,42,42,41,41,40,39,39,39,39,37,37,37,37,
    36,36,36,35,35,34,33,33,33,33,33,32,31,31,30,27,27,26,25,24,24,
    24,24,23,23,23,23,23,22,21,21,20,20
  };
  const int n3c1w2_m[] = {
    100, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,100,99,98,98,98,97,97,97,96,96,94,93,93,92,92,92,91,90,
    90,90,90,89,89,89,89,88,87,87,86,86,86,86,85,85,85,85,85,84,84,
    84,84,83,82,82,82,82,82,81,81,81,81,80,80,79,79,79,79,77,76,76,
    75,75,74,74,74,73,72,72,72,72,72,72,72,72,72,71,71,70,70,69,68,
    68,68,68,67,67,67,67,65,65,65,64,64,63,62,62,62,62,62,61,60,59,
    59,58,58,58,58,58,58,57,57,57,57,57,57,56,56,55,55,55,55,54,54,
    54,53,53,53,52,52,52,51,51,50,49,49,49,48,48,47,47,47,47,47,46,
    44,44,44,44,44,43,42,42,41,41,41,40,39,38,38,37,36,36,36,36,36,
    35,35,34,33,33,32,32,31,31,31,30,30,30,29,29,28,27,27,27,26,26,
    26,25,24,23,23,23,22,22,22,21,21,20
  };
  const int n3c1w2_n[] = {
    100, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,100,100,99,99,99,99,98,98,98,96,96,95,95,94,94,94,93,
    93,93,93,93,92,91,91,91,91,90,90,90,89,89,89,89,89,88,87,87,87,
    86,86,86,85,85,84,84,82,82,81,81,80,80,80,80,79,78,77,77,77,77,
    77,76,76,75,75,75,73,73,73,72,71,71,70,70,70,70,69,69,68,68,68,
    68,68,67,67,67,67,66,66,66,65,65,65,64,63,63,63,62,62,62,61,60,
    60,59,59,59,58,58,58,58,58,57,57,55,55,55,55,55,55,54,54,54,54,
    53,52,52,52,52,52,51,51,50,50,50,50,50,49,49,49,49,49,48,48,48,
    48,46,45,45,45,44,44,44,43,43,42,42,41,41,41,39,39,39,39,38,37,
    37,37,37,36,36,36,36,35,34,34,34,34,34,34,33,33,33,32,31,31,30,
    30,29,28,27,26,25,25,24,24,22,21,21,20
  };
  const int n3c1w2_o[] = {
    100, // Capacity
    200, // Number of items
    // Size of items (sorted)
    99,99,99,99,98,98,98,98,97,97,96,96,96,95,95,95,94,94,94,92,91,
    91,90,90,90,90,89,89,88,88,87,87,87,87,86,86,86,85,84,84,84,84,
    83,83,82,82,82,81,81,81,81,81,80,79,79,79,79,78,78,78,77,77,76,
    76,74,74,74,73,73,73,73,73,72,71,71,70,70,69,69,68,68,68,67,66,
    65,65,64,64,63,63,62,61,61,61,61,61,61,61,60,60,59,58,57,57,57,
    57,57,56,56,56,56,56,55,55,54,54,54,53,53,53,52,52,52,51,51,50,
    50,49,49,48,48,48,48,46,45,45,45,44,44,44,44,43,43,42,42,41,41,
    41,40,39,39,39,39,38,38,37,37,35,35,34,34,33,33,32,32,32,32,30,
    30,30,29,29,28,28,28,28,28,27,27,26,26,25,25,25,24,24,24,24,24,
    24,24,23,22,22,22,21,21,21,21,20
  };
  const int n3c1w2_p[] = {
    100, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,99,99,98,97,97,97,96,96,95,95,95,95,94,94,94,93,93,92,
    92,92,92,91,90,90,90,90,89,89,88,88,88,88,87,87,85,84,83,83,83,
    82,82,82,82,81,81,81,81,79,79,79,78,78,78,78,77,77,77,77,76,76,
    75,73,73,72,71,70,70,70,70,70,70,69,69,69,67,67,66,66,66,66,65,
    65,65,65,63,63,63,63,62,62,61,61,61,61,61,60,60,59,59,59,58,58,
    56,55,55,55,54,53,52,52,52,51,50,49,49,49,49,48,48,48,48,48,47,
    47,47,46,46,46,45,45,45,45,45,45,45,44,44,44,43,43,43,43,43,42,
    42,41,41,41,41,41,40,40,39,38,38,37,37,36,36,36,35,34,33,33,33,
    32,32,32,31,31,30,30,30,29,29,27,27,27,26,26,26,25,24,23,23,22,
    22,22,22,22,21,21,21,21,21,20,20,20
  };
  const int n3c1w2_q[] = {
    100, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,99,99,98,97,97,97,96,96,94,93,93,92,92,92,
    91,91,91,90,90,90,88,88,88,88,88,88,87,86,86,85,85,85,85,85,84,
    84,84,84,83,83,82,82,81,81,81,80,80,80,79,79,78,78,78,77,77,77,
    77,77,76,75,75,75,75,74,74,74,74,74,74,74,73,73,73,72,72,71,71,
    70,70,70,69,68,68,68,67,67,67,67,67,67,67,67,66,66,66,65,64,64,
    64,64,63,63,62,62,62,61,61,60,60,60,59,59,59,59,56,56,56,54,53,
    52,52,51,51,51,50,50,50,50,49,49,49,49,48,48,47,46,46,46,46,46,
    45,45,43,43,43,42,41,41,39,39,39,39,38,37,37,37,36,36,36,35,34,
    34,34,34,32,32,31,29,29,28,28,28,27,27,26,26,26,25,25,24,24,23,
    23,22,22,21,21,21,21,21,20,20,20,20,20
  };
  const int n3c1w2_r[] = {
    100, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,100,100,100,99,99,99,99,99,98,98,97,97,97,
    95,95,95,95,95,94,94,93,93,92,92,92,91,90,90,89,89,89,89,89,88,
    88,88,88,88,88,85,85,85,85,84,84,83,83,82,82,82,82,81,81,80,80,
    78,78,76,75,75,74,73,72,72,70,70,69,69,67,67,66,66,65,65,65,64,
    64,63,62,62,61,61,60,60,60,60,60,57,57,57,56,56,56,56,55,55,54,
    54,54,54,53,52,52,51,51,51,50,50,50,50,49,49,49,48,48,48,48,48,
    48,48,48,46,46,45,45,44,44,43,43,43,42,41,41,40,40,40,40,40,39,
    39,39,39,39,39,38,38,37,36,36,35,35,34,34,34,33,33,33,33,32,32,
    31,31,31,31,31,30,30,30,29,29,29,28,28,28,28,26,25,25,25,24,24,
    24,23,23,23,23,22,22,22,21,20,20,20,20,20
  };
  const int n3c1w2_s[] = {
    100, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,98,98,98,98,97,97,97,97,97,96,96,96,95,95,95,94,94,92,91,
    90,90,89,89,89,88,88,88,88,87,87,86,86,86,85,85,85,84,84,84,83,
    83,82,82,80,80,80,79,78,78,78,78,78,77,77,77,76,75,75,74,74,74,
    73,73,72,72,72,72,71,71,71,70,70,68,68,68,67,67,66,66,66,66,65,
    65,65,64,64,64,64,63,63,63,63,63,63,63,63,61,61,60,59,59,59,59,
    58,58,58,57,57,57,57,55,54,54,53,53,53,53,53,52,52,51,51,51,50,
    50,50,50,50,50,49,49,49,48,48,48,48,47,47,47,46,46,45,45,44,43,
    42,41,41,41,40,40,40,39,39,39,38,38,38,38,38,38,37,37,36,36,36,
    35,34,34,34,34,33,33,32,31,31,31,30,29,27,27,25,25,24,24,24,23,
    23,23,23,23,23,21,21,21,20,20,20,20
  };
  const int n3c1w2_t[] = {
    100, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,99,99,99,98,98,98,98,98,97,96,96,96,95,95,95,94,93,93,92,
    92,91,91,90,90,90,89,88,88,87,87,87,87,86,86,85,85,85,85,84,84,
    84,84,84,83,83,83,83,82,81,80,80,80,79,78,78,78,78,77,76,76,75,
    74,74,74,73,72,72,72,71,71,71,71,71,68,68,67,67,67,67,66,66,65,
    65,65,65,63,63,63,63,63,63,63,63,62,62,62,61,61,61,60,60,60,60,
    59,59,59,59,58,58,58,57,57,56,56,56,56,55,55,54,54,54,53,53,53,
    52,52,52,52,51,51,51,51,51,50,50,50,49,49,48,48,48,48,47,47,46,
    46,46,46,45,44,44,43,42,42,42,42,42,42,42,41,40,39,38,37,37,36,
    36,36,35,35,34,33,33,33,33,33,32,32,31,30,29,28,28,28,27,27,26,
    25,25,24,23,23,23,23,22,21,21,20,20
  };
  const int n3c1w4_a[] = {
    100, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,100,100,99,99,99,99,98,98,98,98,98,97,97,96,96,95,95,
    95,95,94,94,93,93,92,91,91,91,91,91,90,90,90,89,89,89,89,89,88,
    88,88,88,88,87,87,87,87,86,86,86,85,85,85,84,84,83,83,83,82,82,
    82,82,81,81,81,81,80,80,79,79,79,79,79,78,77,77,77,77,75,74,74,
    73,73,73,72,72,71,71,70,70,70,69,69,69,69,68,68,67,67,67,67,67,
    67,65,65,65,65,64,64,64,63,63,63,62,62,62,62,60,60,60,59,59,59,
    58,57,57,56,56,56,56,55,55,54,54,54,54,54,54,52,52,52,52,52,51,
    51,51,50,50,49,49,48,48,48,47,47,47,46,46,45,45,44,44,44,43,43,
    43,43,42,42,41,41,41,40,40,39,39,39,39,39,38,38,37,37,36,36,36,
    36,35,35,35,35,33,32,32,32,32,30,30,30
  };
  const int n3c1w4_b[] = {
    100, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,99,99,98,98,97,97,97,96,96,96,95,95,95,93,93,93,93,93,
    92,92,92,92,91,91,91,90,90,89,89,88,87,87,87,87,86,86,85,85,85,
    85,84,84,84,84,83,83,83,83,83,83,82,80,80,80,79,79,79,78,78,78,
    78,78,78,77,76,76,76,75,75,75,75,75,73,73,73,72,72,72,71,71,70,
    70,70,70,70,70,69,69,68,68,68,68,68,67,67,66,66,66,66,65,65,65,
    64,64,64,63,62,61,61,61,60,60,60,59,59,58,58,58,58,58,58,57,57,
    57,57,57,56,55,55,55,55,54,54,54,54,54,53,53,53,52,52,52,52,51,
    51,50,49,49,49,49,48,48,47,46,46,46,45,44,44,42,42,42,42,41,41,
    41,40,40,40,40,39,39,39,39,38,38,38,37,37,37,37,37,36,36,36,36,
    35,35,34,34,33,33,32,32,31,31,30,30
  };
  const int n3c1w4_c[] = {
    100, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,99,99,98,98,97,97,96,96,96,96,96,96,96,95,95,94,94,92,
    92,92,92,92,92,92,91,91,91,90,89,89,89,89,89,87,86,85,85,84,84,
    84,84,83,83,83,83,83,81,81,80,80,80,80,79,79,79,79,78,78,78,78,
    77,77,77,77,77,77,76,76,76,76,76,75,75,75,75,74,74,74,74,73,72,
    72,72,70,70,70,70,70,69,69,69,68,68,67,67,66,65,65,65,65,64,64,
    64,64,64,63,62,62,61,60,60,60,60,60,60,60,59,59,59,58,58,58,58,
    57,57,55,55,55,53,53,53,52,52,52,52,51,51,49,49,49,49,49,49,49,
    48,48,48,48,48,46,46,45,45,45,45,44,44,44,44,43,43,43,43,43,43,
    42,42,42,41,40,40,40,40,40,39,38,38,38,38,37,37,35,34,34,34,34,
    33,33,33,32,32,32,31,30,30,30,30,30
  };
  const int n3c1w4_d[] = {
    100, // Capacity
    200, // Number of items
    // Size of items (sorted)
    99,99,98,98,98,98,97,97,96,96,95,94,94,94,94,93,93,93,92,92,92,
    92,92,92,92,92,91,91,91,91,90,90,89,89,88,88,87,87,87,87,87,87,
    86,86,85,85,85,84,84,83,83,83,83,83,83,82,82,82,82,82,81,81,81,
    81,80,79,78,78,77,77,77,76,76,75,75,75,74,74,74,74,73,73,73,73,
    73,73,72,72,71,70,70,70,70,70,69,69,69,68,68,68,67,67,66,66,66,
    66,66,65,64,63,63,63,63,62,62,62,61,60,60,60,60,59,59,59,59,58,
    57,56,56,56,55,55,55,55,55,53,53,53,52,52,52,51,51,51,50,50,49,
    49,49,49,48,48,48,48,47,47,46,46,46,46,46,44,43,43,43,42,42,41,
    41,41,41,40,40,40,39,39,39,39,38,38,38,38,38,37,36,36,35,35,34,
    34,34,33,33,33,32,32,32,31,31,30
  };
  const int n3c1w4_e[] = {
    100, // Capacity
    200, // Number of items
    // Size of items (sorted)
    99,99,99,98,97,97,97,97,96,96,95,95,95,95,94,94,94,93,93,93,93,
    93,92,92,91,90,89,88,87,86,86,86,86,85,85,85,85,84,84,84,83,83,
    82,82,82,82,82,82,81,81,81,81,81,80,80,80,79,78,78,77,76,76,75,
    74,74,74,74,73,73,73,73,73,73,72,72,72,71,71,71,70,70,70,69,69,
    69,69,69,69,68,68,67,67,67,67,67,66,66,66,65,64,64,64,63,63,62,
    62,61,61,61,61,60,60,59,59,59,59,59,57,56,55,54,53,53,53,53,52,
    52,52,51,51,51,50,50,50,50,50,49,48,48,48,48,48,47,47,47,46,46,
    46,45,45,45,45,45,44,44,44,43,43,43,43,43,43,42,42,42,42,41,41,
    40,40,40,40,39,39,39,38,37,36,36,36,36,35,35,35,35,34,34,32,32,
    32,32,31,31,31,30,30,30,30,30,30
  };
  const int n3c1w4_f[] = {
    100, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,100,99,99,98,98,98,97,97,96,96,96,96,96,95,94,94,94,93,
    93,93,91,91,91,90,90,90,90,90,89,89,89,89,89,88,88,88,88,87,87,
    87,87,86,86,86,86,85,84,83,83,83,83,82,82,82,82,81,81,81,81,81,
    80,80,79,79,77,76,76,76,76,76,75,74,74,74,73,73,72,72,72,71,70,
    69,68,68,68,68,68,67,67,67,66,66,66,65,64,64,64,63,63,62,62,62,
    61,60,60,59,59,59,58,58,58,58,57,56,56,55,55,55,54,54,54,53,53,
    53,52,52,51,51,50,50,50,50,50,50,49,49,49,49,48,48,47,47,46,45,
    45,45,45,45,44,44,43,43,42,42,42,42,41,41,40,40,40,40,40,40,38,
    38,38,38,38,37,37,37,37,36,36,36,35,35,35,35,34,34,34,33,33,33,
    33,32,32,32,32,31,31,31,31,31,30,30
  };
  const int n3c1w4_g[] = {
    100, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,99,98,97,97,96,96,96,95,95,94,94,94,94,93,93,92,92,91,91,
    89,89,89,89,88,88,88,88,88,87,87,87,87,86,86,86,86,86,85,85,85,
    84,84,83,83,83,82,82,82,82,82,81,80,80,80,80,80,80,80,79,79,79,
    79,78,78,78,78,77,77,77,76,76,75,75,75,75,75,74,74,74,74,73,73,
    73,73,73,72,72,72,72,72,71,71,71,71,70,70,70,70,70,69,68,68,67,
    67,67,66,66,66,65,65,64,62,62,62,61,61,60,60,59,59,59,59,59,59,
    59,58,58,58,57,57,57,56,55,55,55,54,54,54,54,53,52,52,51,51,50,
    50,50,48,48,48,48,47,47,46,46,45,45,43,43,43,41,41,41,40,40,39,
    39,39,39,38,38,38,38,37,37,37,37,37,36,36,36,35,35,34,34,33,33,
    32,32,32,32,32,31,31,31,30,30,30,30
  };
  const int n3c1w4_h[] = {
    100, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,99,99,99,98,98,98,98,97,97,97,97,97,96,96,95,94,94,93,
    93,93,91,91,91,90,90,89,89,89,89,88,88,88,87,87,86,86,86,86,85,
    85,85,84,84,84,83,83,81,81,81,81,81,80,80,80,80,79,78,78,78,77,
    77,76,76,76,76,76,75,75,74,74,73,73,73,72,72,72,72,72,71,71,70,
    70,70,69,69,69,68,68,66,66,66,66,66,65,65,65,64,64,63,63,63,63,
    62,62,62,62,61,61,61,60,60,59,59,59,58,58,57,57,57,56,55,54,54,
    54,54,52,52,51,51,51,50,50,50,50,50,49,49,49,48,48,48,48,48,47,
    47,47,47,46,46,46,45,45,45,44,44,44,43,43,42,41,41,40,39,39,38,
    38,37,37,37,37,37,37,37,36,36,35,34,34,34,34,34,34,33,33,33,33,
    33,32,32,31,31,31,31,31,31,30,30,30
  };
  const int n3c1w4_i[] = {
    100, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,99,99,99,99,98,98,98,97,97,97,96,96,96,95,
    95,95,94,94,94,94,94,93,93,93,92,91,90,89,89,89,89,89,88,88,87,
    87,87,86,86,86,85,84,84,83,82,82,81,81,81,81,80,80,80,79,78,78,
    77,77,76,76,76,75,75,74,74,74,74,74,73,73,73,73,73,72,72,72,72,
    71,71,70,70,70,68,68,67,67,66,65,65,64,64,63,63,63,63,63,62,61,
    61,60,60,59,59,59,58,57,57,56,56,56,55,55,55,55,54,53,52,52,52,
    52,52,52,52,52,52,49,49,49,49,49,49,48,47,47,47,47,46,46,46,45,
    45,44,43,43,43,43,42,42,42,41,41,41,41,41,40,40,40,40,40,39,39,
    38,38,38,37,37,37,36,36,36,36,35,35,35,35,34,34,34,34,34,34,34,
    33,33,33,33,32,32,32,32,31,31,31,30,30
  };
  const int n3c1w4_j[] = {
    100, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,99,99,98,98,98,97,97,97,96,96,96,96,96,95,94,94,93,93,
    93,92,92,92,92,92,91,91,91,90,90,89,89,89,89,88,88,87,87,86,86,
    85,85,85,85,84,84,84,84,83,83,82,82,82,82,82,82,82,81,80,79,79,
    79,78,78,78,77,76,76,75,75,75,74,73,73,73,72,72,72,72,71,71,70,
    70,69,69,69,69,69,68,67,66,66,66,66,66,66,65,65,65,65,64,64,64,
    63,63,62,62,61,61,60,60,60,59,59,59,59,58,58,58,58,58,58,58,57,
    56,56,56,56,53,53,53,52,52,52,52,51,51,51,50,50,50,49,48,48,48,
    48,47,47,47,46,46,46,46,44,44,44,44,43,43,42,42,42,41,40,40,40,
    40,40,39,39,38,38,38,38,38,37,37,37,36,35,34,34,34,34,34,34,34,
    33,33,32,32,32,32,31,31,31,30,30,30
  };
  const int n3c1w4_k[] = {
    100, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,100,99,99,99,99,99,99,98,98,97,97,97,95,95,95,95,95,94,
    94,94,94,94,93,93,93,93,92,92,92,91,90,89,89,89,89,89,88,88,88,
    87,87,87,87,87,86,86,85,84,83,83,83,83,82,82,81,79,79,79,79,78,
    78,77,76,76,76,75,75,75,74,73,73,72,72,72,72,71,70,70,70,70,70,
    70,69,69,69,69,68,68,68,66,66,66,66,66,66,66,66,65,65,65,64,64,
    63,63,63,63,62,62,62,61,61,61,61,61,59,59,59,59,59,59,58,58,58,
    57,57,57,57,57,56,56,56,55,55,55,55,54,54,52,52,51,51,51,50,50,
    50,50,49,48,47,47,47,46,46,46,46,45,45,44,44,44,43,42,42,41,41,
    41,41,41,40,40,39,38,38,38,38,38,38,37,36,36,36,35,34,33,32,32,
    32,31,31,31,31,30,30,30,30,30,30,30
  };
  const int n3c1w4_l[] = {
    100, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,100,100,99,99,99,98,98,98,98,98,97,96,96,96,96,96,95,
    95,95,95,94,94,94,93,93,92,92,92,92,91,90,90,89,88,88,88,88,87,
    87,86,86,86,85,83,83,83,82,82,82,81,81,80,80,80,80,80,80,80,80,
    79,79,78,78,77,77,76,75,75,75,75,75,75,74,74,74,73,73,72,72,72,
    71,71,71,71,71,69,69,68,68,67,67,66,66,66,66,66,65,65,65,65,65,
    64,64,63,62,62,62,62,62,62,62,62,61,61,60,60,60,59,59,59,59,58,
    58,58,57,57,57,57,57,56,56,56,56,56,56,55,55,54,54,53,52,51,50,
    50,49,49,49,49,48,48,48,47,46,45,44,44,44,44,44,43,43,43,43,42,
    42,41,41,40,40,40,39,39,39,39,38,38,37,37,37,37,37,37,36,36,35,
    35,34,34,34,34,33,32,32,31,31,31,30,30
  };
  const int n3c1w4_m[] = {
    100, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,100,99,99,99,98,98,97,97,97,97,97,96,96,96,95,95,94,94,
    94,93,92,92,92,91,91,90,90,90,90,89,88,88,88,88,87,87,86,86,86,
    86,86,84,84,84,83,83,83,83,82,82,82,82,82,81,81,80,80,80,79,79,
    79,79,79,78,78,78,78,78,77,77,77,76,76,76,76,75,74,74,73,73,73,
    72,71,71,71,70,70,70,69,69,69,69,68,68,67,67,67,67,66,66,66,66,
    65,65,65,64,64,64,64,64,64,63,62,62,62,61,61,60,60,59,59,59,59,
    59,58,57,56,55,55,55,55,55,55,54,54,54,54,53,53,53,53,52,52,52,
    52,51,50,49,48,48,48,48,48,47,47,45,45,45,45,44,44,44,43,43,42,
    41,41,40,40,39,39,39,38,38,38,37,37,37,36,35,34,34,33,33,33,33,
    33,32,32,31,31,31,31,31,30,30,30,30
  };
  const int n3c1w4_n[] = {
    100, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,99,99,98,98,98,98,98,98,97,97,97,96,95,94,93,93,93,93,92,
    92,92,92,92,91,91,91,90,87,87,87,85,85,85,84,84,84,83,83,82,82,
    82,82,81,81,81,81,80,80,80,80,79,79,78,78,78,78,76,76,76,75,75,
    74,73,72,72,72,72,72,71,71,71,71,70,70,70,69,69,69,68,68,68,68,
    68,68,68,68,67,67,67,65,64,63,63,63,63,63,63,63,62,62,62,61,60,
    60,60,60,60,60,59,59,59,59,58,58,58,57,57,56,56,56,56,55,55,55,
    55,54,54,54,54,54,54,53,53,53,53,53,52,52,52,52,51,51,51,51,51,
    51,50,49,49,49,49,47,47,46,46,46,45,45,45,45,44,44,43,43,43,42,
    42,41,40,40,39,39,39,39,38,38,37,37,37,37,37,37,35,34,34,33,32,
    32,32,32,31,31,31,31,31,30,30,30,30
  };
  const int n3c1w4_o[] = {
    100, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,99,99,99,97,97,97,96,95,95,95,95,94,94,93,93,92,92,91,
    91,89,89,88,88,87,86,86,86,86,85,85,84,84,83,83,82,82,82,82,81,
    81,81,81,81,81,80,80,80,79,79,79,79,78,77,77,77,77,77,77,77,77,
    76,76,75,75,75,74,74,73,73,73,73,72,72,72,72,71,71,71,71,70,70,
    70,70,70,70,69,69,69,69,69,67,66,66,65,65,65,64,63,62,62,62,62,
    61,61,61,61,60,60,60,58,58,58,58,58,58,58,58,58,57,55,55,54,53,
    53,53,53,53,52,52,52,52,52,51,51,51,50,50,50,49,49,48,48,47,47,
    46,46,45,45,45,45,44,44,43,42,42,42,42,41,41,41,41,40,40,37,37,
    37,36,36,36,36,35,35,35,35,35,35,35,34,34,34,34,34,34,33,33,33,
    33,33,32,32,32,32,32,32,32,31,31,30
  };
  const int n3c1w4_p[] = {
    100, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,100,100,99,99,99,99,99,98,98,97,96,96,95,
    95,94,94,94,93,92,92,92,92,92,92,91,90,89,89,89,89,88,88,88,88,
    87,87,87,86,86,85,84,83,82,82,82,81,81,81,81,79,79,79,78,78,78,
    77,77,77,77,77,76,76,76,76,75,75,75,75,74,74,74,74,74,73,73,73,
    71,71,71,71,71,71,71,69,69,68,67,66,66,66,65,64,64,64,63,63,63,
    63,63,63,62,62,62,62,61,60,60,60,60,59,59,59,59,59,58,58,58,57,
    56,56,56,56,56,54,53,53,53,52,52,52,51,51,51,51,51,50,49,49,49,
    48,47,47,47,47,46,46,46,45,45,44,44,43,43,42,42,42,41,41,41,41,
    41,40,40,40,39,39,39,38,37,36,36,36,36,35,35,35,35,34,34,34,34,
    33,33,33,33,33,32,32,32,32,31,31,30,30,30
  };
  const int n3c1w4_q[] = {
    100, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,100,100,99,99,99,99,98,98,98,97,97,96,96,96,96,96,95,
    95,95,95,94,93,93,93,92,92,92,92,92,92,91,91,90,90,90,89,87,87,
    87,86,86,86,86,86,86,85,85,85,85,84,83,83,83,82,81,81,81,80,80,
    80,79,79,79,79,79,79,79,79,78,78,77,77,76,76,76,75,75,75,74,73,
    72,72,72,72,71,70,70,70,70,69,69,69,68,68,68,68,68,68,67,67,66,
    66,65,65,65,65,64,64,64,62,62,62,62,61,60,60,59,58,58,58,58,57,
    57,57,57,57,56,56,55,54,54,54,54,53,53,53,53,52,52,51,51,50,50,
    50,49,49,48,48,48,48,47,47,46,45,45,45,44,44,43,43,43,42,42,42,
    42,41,41,40,40,40,40,39,39,39,38,38,37,37,36,36,36,35,35,34,34,
    33,33,33,33,32,32,32,32,31,30,30,30,30
  };
  const int n3c1w4_r[] = {
    100, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,100,99,98,97,97,97,96,96,96,96,96,96,96,96,95,95,93,93,
    93,93,92,92,92,91,91,91,91,90,90,90,90,89,88,88,87,87,87,86,85,
    85,84,84,83,83,82,82,82,81,81,81,80,80,80,80,80,79,79,78,78,77,
    77,77,76,75,74,74,73,73,73,73,72,72,71,71,70,70,69,69,69,69,68,
    68,68,68,68,67,67,67,67,67,66,66,65,65,65,64,63,63,63,62,60,60,
    60,60,59,59,59,59,59,58,58,58,58,58,57,57,57,57,57,57,57,57,56,
    56,56,55,55,55,55,54,54,54,54,53,53,52,51,51,51,51,51,50,50,50,
    49,48,47,46,46,46,46,45,45,44,44,44,43,43,43,42,42,42,41,41,41,
    41,41,40,40,40,40,39,39,38,38,38,38,37,37,37,37,36,36,35,35,35,
    35,34,33,33,33,32,32,31,31,31,30,30
  };
  const int n3c1w4_s[] = {
    100, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,99,99,99,98,98,98,98,98,98,97,96,96,96,95,94,93,92,92,
    92,92,91,91,91,91,91,90,90,90,90,89,89,89,89,89,88,88,87,86,86,
    86,84,82,82,82,80,80,80,80,80,79,79,79,78,77,77,77,77,77,76,76,
    76,76,75,75,74,74,74,73,73,72,72,72,72,72,71,71,71,71,70,70,70,
    70,70,69,69,68,68,67,67,67,67,67,67,66,65,65,65,65,65,64,63,63,
    63,62,62,62,61,61,61,60,60,60,60,60,60,60,60,59,59,58,58,58,58,
    57,57,57,55,55,55,55,55,55,54,53,53,53,53,52,52,51,51,50,49,49,
    49,49,48,48,48,47,47,46,45,45,45,45,44,43,43,43,42,42,42,42,42,
    42,41,40,40,40,39,39,38,38,37,37,37,37,35,35,35,33,33,33,33,32,
    32,32,31,31,31,31,31,30,30,30,30,30
  };
  const int n3c1w4_t[] = {
    100, // Capacity
    200, // Number of items
    // Size of items (sorted)
    98,98,98,98,97,97,97,96,96,95,95,95,95,95,94,94,93,93,93,92,92,
    91,91,91,91,91,90,90,90,90,90,89,89,88,88,88,88,88,87,86,86,86,
    86,86,85,85,84,84,83,82,82,81,80,80,80,80,80,80,79,79,79,79,79,
    78,78,78,77,77,77,77,76,76,76,76,75,75,74,74,74,74,73,72,72,71,
    71,71,71,71,71,70,69,69,69,69,69,69,68,68,68,67,67,67,67,67,66,
    66,65,65,65,65,65,64,63,62,61,61,61,60,60,59,58,58,57,57,57,56,
    56,56,56,55,55,54,54,54,53,53,53,52,52,52,51,51,51,50,49,49,48,
    48,48,47,47,46,45,45,45,45,44,44,44,43,43,43,43,43,43,43,42,42,
    42,41,41,40,40,40,39,39,38,38,36,35,34,34,34,33,33,33,33,33,32,
    32,32,31,31,31,31,30,30,30,30,30
  };
  const int n3c2w1_a[] = {
    120, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,100,99,99,99,99,98,98,97,97,95,95,95,95,94,94,94,93,92,
    92,91,91,91,91,91,90,90,90,90,89,89,89,88,87,87,87,87,87,86,86,
    86,85,83,83,82,82,81,81,80,80,79,79,78,78,78,77,77,76,76,76,75,
    74,74,74,74,73,72,72,72,72,71,70,70,69,69,67,67,67,65,64,64,63,
    62,61,60,60,60,60,59,59,59,58,58,57,57,57,56,56,55,54,53,53,51,
    51,50,49,48,47,47,46,46,46,46,45,45,45,44,44,43,43,42,42,41,41,
    40,40,40,40,40,39,38,38,38,38,38,36,36,35,32,32,30,30,30,30,29,
    29,28,25,24,24,24,24,23,23,23,23,23,22,22,21,20,19,19,19,19,17,
    17,16,16,16,16,16,16,15,15,13,13,13,12,10,10,9,9,8,8,7,7,5,4,
    4,4,4,4,4,3,2,2,2,1
  };
  const int n3c2w1_b[] = {
    120, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,99,98,97,96,96,96,95,95,94,93,93,93,92,90,
    90,90,89,89,89,88,87,87,87,86,83,82,81,81,80,80,80,79,79,79,78,
    77,77,77,77,76,76,76,75,73,72,72,72,72,71,70,68,68,68,68,67,66,
    66,66,66,66,65,65,65,63,63,63,62,61,60,60,60,60,58,58,57,57,56,
    56,56,56,55,55,55,55,55,53,52,51,51,50,50,50,50,49,49,48,48,48,
    48,47,47,46,46,45,45,45,45,43,43,42,41,40,40,40,40,40,39,39,39,
    39,39,38,38,37,36,35,35,34,34,34,33,33,31,30,30,30,27,27,25,25,
    24,24,23,23,23,23,22,22,21,21,20,19,19,19,19,19,18,18,17,17,17,
    16,16,15,15,15,14,14,14,13,13,12,12,12,12,12,10,9,9,9,9,9,9,9,
    8,7,5,5,4,4,3,2,1,1,1
  };
  const int n3c2w1_c[] = {
    120, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,98,97,97,96,96,96,96,93,93,92,90,90,89,89,89,89,89,88,
    88,87,86,86,86,85,85,85,85,83,82,81,81,81,80,80,79,79,78,77,77,
    76,76,76,75,75,75,74,74,73,73,72,72,72,72,72,71,70,70,70,70,70,
    69,69,68,68,67,66,66,65,65,63,63,63,62,62,62,62,60,60,59,59,58,
    58,58,57,57,57,55,55,54,54,53,53,53,52,52,51,51,51,50,50,49,48,
    48,47,47,47,46,44,43,43,43,42,42,41,40,40,40,40,39,39,39,39,39,
    38,37,36,36,36,35,35,34,34,34,34,33,33,33,33,32,32,32,32,31,30,
    29,29,29,29,28,27,26,25,24,23,23,22,22,20,20,20,19,19,19,18,18,
    17,17,17,16,16,15,15,15,13,13,13,13,13,12,12,10,10,9,9,9,8,8,
    7,7,7,5,4,4,3,3,1,1,1
  };
  const int n3c2w1_d[] = {
    120, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,100,99,99,98,98,98,97,96,95,95,95,94,94,93,93,93,93,92,
    92,92,91,90,90,89,89,88,87,86,86,85,85,84,84,84,83,83,83,83,81,
    79,78,78,77,77,76,76,75,75,75,75,75,74,74,74,74,74,73,73,73,72,
    71,71,70,69,69,68,68,66,65,65,65,65,65,64,64,63,61,61,61,61,60,
    60,60,60,60,59,59,58,58,57,57,56,55,54,53,53,52,51,51,51,50,49,
    48,47,46,46,45,44,44,43,41,41,39,39,38,38,38,37,37,37,36,36,35,
    35,35,34,34,34,34,34,33,32,32,32,31,29,28,28,28,27,27,26,25,25,
    23,23,23,23,23,22,22,22,22,21,20,18,18,17,17,17,16,16,15,15,14,
    13,13,12,12,12,11,11,11,11,11,10,8,8,8,8,8,6,6,6,6,6,5,5,4,4,
    3,3,2,2,1,1,1,1
  };
  const int n3c2w1_e[] = {
    120, // Capacity
    200, // Number of items
    // Size of items (sorted)
    99,99,99,99,98,98,98,97,96,95,95,95,95,95,94,94,93,93,93,91,91,
    91,90,90,90,90,90,90,89,89,88,87,87,86,86,85,85,85,85,84,84,83,
    82,82,80,80,79,79,79,78,78,78,78,77,77,77,76,76,76,75,75,75,72,
    72,71,71,70,70,69,67,67,67,67,66,65,65,64,64,64,63,63,63,62,62,
    61,61,59,59,58,58,58,57,57,57,57,56,55,55,55,54,53,52,51,51,50,
    50,49,48,47,46,45,44,44,43,43,42,40,40,38,37,37,36,36,35,35,35,
    35,33,33,32,32,32,31,31,31,31,31,31,30,29,29,29,28,27,27,26,26,
    25,24,24,24,22,22,21,20,19,19,19,18,17,16,16,16,15,15,15,15,15,
    14,14,14,13,13,12,12,12,12,11,11,10,9,9,8,7,6,6,6,6,5,5,5,4,4,
    4,3,3,3,3,3,2
  };
  const int n3c2w1_f[] = {
    120, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,99,98,98,98,98,97,96,95,95,95,94,93,93,93,
    92,92,91,90,90,90,89,89,89,88,88,88,87,87,87,86,84,83,83,83,83,
    83,82,82,80,80,79,79,79,78,75,75,75,75,74,74,73,72,72,72,72,70,
    69,69,69,69,68,67,67,67,66,66,64,64,64,63,63,63,62,62,62,61,61,
    61,61,61,61,61,60,59,59,59,59,59,59,57,57,57,56,55,55,54,54,54,
    53,53,53,52,51,51,50,50,50,49,49,48,47,47,46,45,45,45,42,42,42,
    40,39,37,36,36,35,35,34,34,34,34,34,32,32,32,30,30,29,28,27,27,
    27,25,25,25,24,24,24,24,24,23,22,22,22,22,21,20,19,19,18,17,17,
    16,15,15,15,14,12,12,12,11,11,11,10,10,10,10,9,9,9,9,8,8,8,7,
    6,6,5,5,4,2,2,2,1,1,1
  };
  const int n3c2w1_g[] = {
    120, // Capacity
    200, // Number of items
    // Size of items (sorted)
    99,99,98,98,97,97,96,96,95,94,94,92,92,92,90,90,89,89,89,88,88,
    88,87,86,86,86,85,85,85,85,85,84,84,83,82,82,81,81,81,80,80,80,
    79,79,79,78,78,75,75,75,74,74,74,74,73,73,72,72,71,70,69,69,68,
    67,67,67,67,67,67,67,66,65,65,64,63,63,63,63,63,62,62,61,60,60,
    60,59,59,58,58,58,58,57,57,57,56,55,55,55,54,53,52,52,52,52,52,
    51,51,50,50,49,49,49,49,49,47,46,46,46,46,44,44,43,43,42,42,42,
    41,41,41,40,39,39,37,36,36,36,35,35,35,34,34,33,33,33,32,31,31,
    31,30,30,29,29,29,29,28,28,28,27,26,26,25,24,23,23,23,23,23,22,
    22,22,22,22,20,20,19,19,19,17,15,15,14,12,11,10,9,8,7,7,5,5,5,
    4,4,4,3,3,1,1,1,1
  };
  const int n3c2w1_h[] = {
    120, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,100,100,99,99,98,98,97,97,96,96,95,94,94,94,93,93,93,
    92,92,90,90,90,89,89,87,87,86,85,85,85,85,85,85,84,84,83,82,82,
    82,81,81,80,79,79,77,77,77,77,75,74,74,73,72,72,71,71,71,70,70,
    70,69,69,68,67,67,66,66,66,64,63,62,62,62,62,62,62,60,59,59,59,
    59,59,58,58,57,57,57,56,56,56,55,55,54,54,53,53,52,52,52,52,51,
    51,50,50,50,50,50,49,48,48,48,48,47,47,46,46,44,44,43,43,43,42,
    42,41,41,41,40,40,38,38,37,36,36,35,35,33,32,32,31,31,31,30,30,
    28,28,28,27,25,25,24,24,24,24,24,21,20,20,19,19,18,18,17,17,17,
    17,17,16,16,16,15,14,14,14,14,13,13,12,12,12,11,11,9,9,9,8,6,
    6,6,5,4,4,3,3,2,1,1,1,1
  };
  const int n3c2w1_i[] = {
    120, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,99,99,99,99,98,97,97,97,97,97,97,97,96,96,95,95,95,95,95,
    94,93,93,93,92,92,92,91,91,90,90,88,88,88,88,87,86,85,84,84,84,
    84,83,83,81,79,79,79,78,78,77,76,76,75,74,74,73,73,73,72,72,72,
    71,71,71,70,70,70,69,69,68,68,67,67,66,65,64,64,63,63,60,60,60,
    59,58,58,58,58,57,56,56,55,55,54,53,53,52,52,51,51,51,50,50,50,
    49,49,48,48,48,47,47,47,45,45,43,43,42,42,41,41,41,40,40,40,39,
    38,38,37,37,36,36,35,35,35,35,35,34,33,33,32,32,31,30,29,29,27,
    26,25,25,24,24,24,23,23,23,23,21,20,20,20,20,20,19,18,17,17,16,
    16,16,14,14,13,13,13,13,13,12,12,11,11,10,10,9,9,8,8,8,8,7,6,
    6,6,5,4,4,3,3,2,2,1
  };
  const int n3c2w1_j[] = {
    120, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,100,100,99,99,99,98,98,97,95,95,95,94,93,92,92,92,92,
    91,91,88,87,87,86,86,85,84,84,84,83,83,82,82,82,81,81,81,80,80,
    79,78,78,77,76,76,76,75,74,74,74,73,72,70,69,68,68,67,67,67,67,
    67,67,66,66,66,65,65,65,65,65,65,64,64,64,63,63,63,62,61,60,59,
    59,59,58,58,58,57,57,57,56,56,56,56,55,55,54,54,54,53,53,52,52,
    51,50,50,50,49,49,49,48,47,47,46,46,45,45,45,44,44,44,43,43,43,
    41,41,41,39,38,37,36,36,36,36,36,36,35,35,35,34,33,33,32,31,31,
    30,30,29,29,29,29,29,28,28,26,26,26,26,26,25,25,25,24,23,23,21,
    20,20,20,20,20,19,19,19,18,18,17,16,15,15,15,13,12,11,10,9,9,
    9,8,7,7,7,5,4,3,3,2,2,1,1
  };
  const int n3c2w1_k[] = {
    120, // Capacity
    200, // Number of items
    // Size of items (sorted)
    99,99,99,99,98,98,96,95,95,92,92,92,91,91,91,91,89,89,89,88,88,
    87,85,85,84,84,84,83,83,83,83,83,82,81,80,80,79,79,77,77,76,74,
    73,73,73,73,73,70,69,68,66,66,66,66,65,65,65,64,63,63,62,62,61,
    61,59,59,59,58,58,57,57,56,56,55,55,54,54,54,53,52,52,51,50,50,
    50,50,49,49,48,48,48,48,48,47,47,46,46,46,45,45,45,44,44,44,43,
    43,43,42,42,42,41,41,40,40,40,39,38,38,36,36,35,35,35,34,33,33,
    33,33,33,33,32,32,32,31,30,30,30,28,28,27,27,27,26,25,24,23,23,
    22,22,22,21,20,20,18,18,17,17,17,16,15,15,14,14,14,13,13,13,12,
    12,12,12,12,11,11,11,11,10,9,8,7,7,7,7,7,7,7,7,7,6,6,5,5,5,5,
    5,4,4,3,2,1
  };
  const int n3c2w1_l[] = {
    120, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,99,99,99,99,99,97,96,96,96,95,95,95,94,94,94,94,93,93,
    93,93,93,92,92,92,92,91,91,88,88,88,87,87,86,85,85,85,83,83,82,
    82,82,81,81,80,80,79,79,78,78,77,77,77,77,76,74,74,74,73,71,70,
    69,68,67,67,67,67,66,66,66,66,66,65,65,65,65,64,64,64,64,64,64,
    63,63,62,61,61,60,60,60,59,58,57,56,56,56,56,55,55,55,54,54,54,
    53,53,52,52,52,51,50,49,48,48,47,47,45,45,44,44,44,44,43,43,43,
    43,42,41,41,40,40,40,40,40,40,40,38,37,37,37,35,35,33,33,33,31,
    31,30,30,28,27,25,25,25,24,24,24,23,22,22,20,20,19,19,19,18,18,
    18,18,17,16,15,14,14,13,13,12,11,11,11,10,10,10,8,8,7,7,7,6,5,
    5,5,5,5,3,2,2,2,1,1
  };
  const int n3c2w1_m[] = {
    120, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,99,99,98,97,97,96,96,95,95,93,92,92,91,88,88,88,87,86,
    86,86,85,85,83,83,83,82,82,82,82,81,81,81,81,81,81,80,80,79,78,
    78,78,77,77,77,75,75,74,73,73,72,72,72,72,72,72,71,71,71,70,70,
    69,69,69,68,67,66,66,65,65,64,64,64,63,63,63,63,62,61,61,61,61,
    60,60,60,59,59,58,57,56,55,55,54,54,54,53,53,53,53,53,52,52,52,
    50,48,48,46,46,46,46,45,44,44,43,43,43,43,43,42,42,42,42,40,40,
    40,39,38,36,36,36,36,36,36,32,32,32,31,31,30,30,28,28,27,27,27,
    26,26,25,25,25,24,24,23,22,22,22,21,21,21,20,20,20,20,20,19,19,
    19,18,18,18,18,16,16,15,13,13,12,11,11,10,10,9,9,8,8,8,7,7,6,
    5,5,4,3,3,2,2,2,2,2
  };
  const int n3c2w1_n[] = {
    120, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,100,98,98,97,97,97,96,96,95,94,94,94,94,93,93,93,92,91,
    91,91,91,89,89,89,89,88,88,88,87,86,86,86,85,84,84,84,83,83,82,
    81,81,80,80,80,80,79,79,79,79,78,77,77,77,76,76,75,75,75,75,75,
    74,74,73,72,72,72,71,71,70,70,69,69,69,68,67,67,66,66,64,64,64,
    63,62,62,62,61,60,60,60,60,60,59,58,58,57,56,56,54,54,53,53,52,
    52,52,52,51,49,49,49,49,49,47,47,47,46,46,46,45,45,44,44,42,41,
    41,41,40,40,39,38,38,37,36,36,36,33,32,31,31,30,30,30,30,29,28,
    27,26,26,23,22,21,21,21,21,21,20,20,20,20,19,18,18,18,16,16,15,
    13,13,12,12,11,10,10,10,10,9,9,9,8,8,7,7,7,6,6,5,5,4,4,3,3,3,
    3,2,2,2,1,1,1
  };
  const int n3c2w1_o[] = {
    120, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,99,98,98,96,94,93,92,92,92,91,91,90,90,89,89,89,88,88,
    87,87,87,86,86,84,84,84,83,81,79,79,79,78,77,77,77,77,77,75,75,
    75,74,74,74,73,73,73,73,72,72,71,71,70,70,69,68,68,67,67,66,66,
    65,65,64,64,64,63,63,63,63,63,63,62,62,61,61,61,61,60,60,60,60,
    59,59,58,58,58,58,58,57,57,57,56,55,55,55,54,54,53,53,53,52,51,
    51,50,48,48,47,47,46,46,44,43,42,41,41,41,41,40,40,40,39,39,39,
    39,38,37,36,36,36,35,35,35,34,33,32,32,32,31,31,31,30,29,28,28,
    27,27,27,27,27,24,23,23,21,20,20,19,19,19,18,18,18,17,17,16,16,
    15,14,13,13,13,13,12,12,11,11,9,9,8,8,8,8,7,7,7,6,4,4,3,3,3,3,
    2,2,2,1,1,1,1
  };
  const int n3c2w1_p[] = {
    120, // Capacity
    200, // Number of items
    // Size of items (sorted)
    99,99,97,97,97,97,97,96,96,96,96,96,96,94,94,94,93,92,92,89,89,
    89,88,88,87,87,86,85,85,85,84,84,84,83,83,83,83,83,83,82,81,81,
    81,80,80,80,79,79,79,78,78,77,76,76,75,74,73,72,71,71,71,71,69,
    69,68,68,68,68,67,67,66,66,66,65,65,65,65,65,64,64,64,63,63,60,
    60,58,58,58,58,57,57,57,56,56,56,55,54,54,53,53,53,53,52,52,50,
    50,49,49,47,46,45,45,45,44,44,43,42,42,41,41,41,41,40,40,40,40,
    40,40,39,39,38,38,38,37,37,37,37,36,36,35,34,34,34,34,34,33,33,
    32,32,31,31,31,30,30,29,28,27,27,27,26,25,25,24,23,22,22,21,21,
    21,21,20,19,19,19,18,17,17,17,16,15,13,13,13,10,10,9,9,9,9,9,
    9,8,7,6,6,5,4,3,2,1
  };
  const int n3c2w1_q[] = {
    120, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,98,97,97,97,96,96,96,96,96,95,94,93,93,93,92,92,92,91,90,
    90,90,90,90,89,89,88,88,87,87,86,85,84,84,82,82,81,81,80,79,79,
    77,75,75,75,75,73,73,72,72,71,71,71,71,71,70,70,69,69,69,69,68,
    68,67,67,66,66,65,65,65,64,62,62,62,60,59,59,59,59,58,58,58,57,
    57,56,55,55,55,54,54,53,53,53,53,52,52,51,50,50,48,47,47,46,46,
    46,45,44,44,43,43,42,41,41,41,41,40,40,39,39,39,37,37,36,36,36,
    35,33,32,32,32,32,32,31,31,31,31,30,30,30,29,29,28,27,26,26,26,
    25,25,25,25,24,24,24,22,22,21,20,20,19,18,18,18,17,15,15,15,15,
    14,14,13,12,12,12,11,10,10,10,10,10,9,8,8,8,8,8,8,7,7,6,6,5,5,
    5,5,5,4,4,4,2,2
  };
  const int n3c2w1_r[] = {
    120, // Capacity
    200, // Number of items
    // Size of items (sorted)
    99,99,99,99,99,98,98,97,96,95,95,93,92,91,91,90,90,90,89,89,89,
    86,84,84,84,83,82,82,80,80,79,79,78,78,77,77,77,76,76,76,76,74,
    74,74,72,72,71,71,71,71,70,70,70,69,69,69,68,67,66,66,65,65,64,
    64,64,64,63,63,62,62,62,61,61,60,60,60,59,59,58,58,58,57,56,56,
    55,54,53,53,52,52,52,52,52,51,51,51,50,50,50,49,49,47,47,46,46,
    45,44,44,44,44,43,43,42,42,42,42,41,41,41,41,40,40,40,40,40,39,
    39,39,39,37,36,35,35,34,34,33,33,33,32,32,32,32,31,30,30,29,29,
    28,27,27,26,26,26,26,25,25,25,24,24,24,23,23,23,22,21,21,21,19,
    18,18,18,17,17,16,16,15,14,14,14,13,12,11,11,10,9,7,7,7,7,7,7,
    6,5,4,4,3,2,2,1,1
  };
  const int n3c2w1_s[] = {
    120, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,99,98,98,97,97,96,95,95,94,94,94,94,94,93,
    93,93,93,92,92,92,91,90,89,89,89,89,88,88,88,88,87,87,87,86,86,
    85,84,84,84,83,83,82,81,81,80,79,79,78,78,77,77,77,76,76,76,75,
    75,74,73,73,73,70,70,69,68,66,66,66,65,65,65,63,63,62,62,62,60,
    59,59,59,59,57,57,57,57,57,57,57,55,55,53,53,53,53,53,52,52,52,
    51,51,50,49,49,49,48,47,47,46,45,45,45,44,44,44,42,42,42,41,40,
    40,40,39,39,39,39,36,36,36,35,34,34,34,33,33,31,31,30,30,30,29,
    29,29,27,27,27,26,26,26,25,25,25,25,24,23,23,22,22,21,20,20,20,
    20,19,17,17,17,16,16,16,16,15,15,14,13,12,12,12,12,12,12,12,11,
    11,11,9,9,9,9,9,8,8,6,6,6,6
  };
  const int n3c2w1_t[] = {
    120, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,100,99,99,98,97,97,96,96,96,95,94,94,92,92,91,91,90,90,
    89,89,89,88,88,88,87,87,87,87,85,85,85,84,84,84,84,84,83,82,82,
    82,82,80,79,79,79,78,78,78,77,76,76,75,71,71,69,69,69,68,68,68,
    68,67,67,66,66,66,66,65,65,65,64,63,63,61,58,58,58,57,57,56,55,
    55,55,54,54,54,53,53,52,51,50,50,49,49,49,48,47,46,46,46,45,44,
    44,44,44,44,44,44,43,43,43,42,42,42,41,41,40,40,39,39,39,39,38,
    38,38,37,35,35,35,33,32,32,31,31,30,30,29,29,28,28,27,27,26,26,
    25,25,24,24,23,23,22,22,22,22,22,21,21,20,20,20,19,19,18,16,16,
    15,15,14,14,14,13,13,13,12,12,12,12,12,11,11,10,10,10,9,8,8,7,
    7,6,6,3,3,2,2,1,1,1,1
  };
  const int n3c2w2_a[] = {
    120, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,99,99,99,99,98,98,98,98,97,97,96,96,96,95,95,95,94,94,
    94,94,93,92,92,91,91,90,90,89,88,88,88,87,87,87,86,86,86,85,85,
    84,84,83,83,83,82,82,81,81,81,81,80,80,78,78,78,78,78,77,77,76,
    76,76,76,75,75,75,75,74,74,74,73,73,72,71,70,70,69,69,68,68,68,
    68,67,67,67,67,66,66,66,66,65,65,65,65,65,64,64,63,63,62,61,61,
    61,60,59,58,58,58,57,57,57,57,56,55,55,55,55,54,54,54,53,52,51,
    51,51,50,50,50,49,49,49,48,48,47,47,47,47,47,46,46,46,45,44,44,
    44,43,42,42,42,42,41,41,41,40,40,39,38,38,37,37,35,35,35,34,34,
    34,34,33,32,32,32,31,31,31,31,30,30,29,29,28,28,27,27,27,27,26,
    26,25,25,25,23,22,22,21,21,20,20,20
  };
  const int n3c2w2_b[] = {
    120, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,99,99,99,98,98,98,97,97,97,97,96,94,94,93,
    93,91,91,91,91,91,90,90,90,89,88,88,87,87,87,86,86,85,85,85,84,
    84,83,82,82,82,81,81,80,79,79,79,79,79,79,79,78,77,77,77,77,77,
    76,75,75,73,73,72,72,72,72,72,70,70,70,69,69,68,68,68,67,67,67,
    67,66,66,65,65,65,64,64,64,64,63,63,63,62,62,61,61,61,61,61,61,
    60,60,60,59,58,57,57,57,56,56,55,55,54,53,53,53,52,52,51,51,50,
    50,49,48,47,47,46,45,45,45,45,44,43,43,43,42,42,42,42,42,40,39,
    38,37,37,36,36,36,36,35,34,34,33,33,33,33,32,32,32,32,31,30,30,
    30,30,30,29,29,29,29,29,28,28,27,27,27,27,26,26,26,25,25,25,25,
    24,24,24,23,22,22,22,22,21,20,20,20,20
  };
  const int n3c2w2_c[] = {
    120, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,100,100,98,98,97,97,97,97,96,95,95,94,94,93,93,93,92,
    92,92,92,91,90,90,90,90,89,89,89,89,89,88,88,88,87,87,86,86,86,
    85,85,84,84,83,83,83,82,81,81,80,80,79,79,78,78,78,78,78,78,77,
    76,76,76,76,75,75,75,75,74,73,73,72,71,69,69,69,68,68,68,68,67,
    66,66,66,66,65,65,65,64,64,64,63,63,63,62,62,62,61,61,60,59,58,
    58,57,56,55,55,55,54,54,52,51,51,51,50,50,50,49,49,49,49,48,48,
    48,48,47,47,47,47,47,46,46,46,46,45,45,44,44,44,43,43,43,42,42,
    41,41,41,41,40,40,40,40,40,40,39,39,38,38,38,38,38,37,37,36,36,
    36,35,35,34,34,33,33,33,33,33,32,30,29,27,27,27,26,26,25,25,25,
    25,25,25,24,22,22,21,21,21,21,21,20,20
  };
  const int n3c2w2_d[] = {
    120, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,100,98,97,96,96,96,96,96,95,95,95,94,94,94,93,93,93,93,
    93,92,92,92,92,91,91,91,90,90,89,89,89,88,88,88,87,86,85,85,85,
    84,84,84,84,84,83,83,83,83,83,83,82,82,82,81,81,81,80,79,78,78,
    78,77,77,76,76,75,75,75,75,75,75,74,74,73,72,72,72,70,70,70,70,
    69,68,68,68,68,68,67,66,66,65,65,65,64,64,63,61,61,60,60,60,60,
    59,59,59,58,58,57,57,57,56,55,55,55,54,54,53,52,52,52,51,51,51,
    51,50,50,50,50,49,49,49,49,47,47,47,47,45,45,45,43,43,42,41,41,
    41,41,40,40,40,40,39,39,38,38,38,38,38,37,37,37,37,37,36,36,36,
    36,36,35,35,34,34,34,34,33,33,33,33,32,32,31,30,29,29,28,28,27,
    26,25,24,24,24,23,23,22,22,21,20,20
  };
  const int n3c2w2_e[] = {
    120, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,99,99,99,99,98,98,98,98,98,97,97,97,97,96,
    96,96,96,96,95,95,95,94,94,94,93,92,92,92,92,91,91,91,91,90,90,
    90,90,89,89,89,89,88,88,87,87,87,87,87,87,86,86,86,85,85,84,83,
    83,82,82,81,81,81,80,80,80,79,79,79,78,78,77,77,76,76,75,75,74,
    74,74,74,73,72,69,69,69,67,67,66,66,66,66,65,65,64,64,63,63,62,
    62,62,62,62,62,61,60,59,58,58,58,57,57,56,55,55,55,55,54,53,53,
    53,53,53,53,53,53,52,52,52,52,51,50,49,49,49,49,49,48,48,47,47,
    47,46,46,46,46,45,45,44,44,43,42,41,40,40,40,40,40,40,39,38,38,
    38,38,37,37,36,36,34,34,34,32,32,32,31,30,30,29,28,27,26,26,26,
    25,25,25,25,25,24,24,23,23,22,21,20,20
  };
  const int n3c2w2_f[] = {
    120, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,99,99,98,98,98,97,97,97,96,96,95,95,95,95,
    94,94,94,94,92,92,92,92,92,92,91,91,91,90,90,90,90,90,90,89,88,
    87,86,86,86,86,85,84,84,84,84,84,84,84,83,82,82,82,82,82,81,80,
    80,80,80,79,78,78,77,77,76,76,76,75,75,75,75,74,74,74,73,73,72,
    72,71,70,70,69,68,67,67,67,67,66,64,63,63,63,62,62,61,60,59,59,
    59,59,57,57,57,56,54,54,54,54,53,53,53,53,53,51,51,51,51,50,50,
    49,48,48,48,48,48,47,47,46,46,45,45,44,44,44,43,43,43,43,42,42,
    41,40,39,38,38,38,38,38,38,38,38,37,37,36,35,35,35,35,34,34,33,
    32,32,31,31,30,30,30,30,30,30,29,29,29,28,28,28,27,27,27,27,26,
    26,26,24,23,23,22,22,22,21,21,21,20,20
  };
  const int n3c2w2_g[] = {
    120, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,99,98,98,98,98,98,97,96,96,95,95,92,92,92,
    92,92,92,91,91,91,91,90,90,89,89,89,89,89,88,88,88,87,87,85,84,
    84,83,83,83,82,82,82,81,81,81,81,80,79,79,79,79,78,78,77,77,77,
    77,76,76,76,76,75,75,75,74,74,74,74,73,73,70,69,69,68,67,66,66,
    66,64,64,64,64,63,63,63,63,63,62,62,61,61,61,61,60,60,59,59,57,
    57,57,57,57,57,56,55,54,54,53,53,53,53,52,52,52,51,50,50,50,50,
    49,48,48,48,47,46,46,46,45,45,45,45,44,44,43,42,41,41,40,40,39,
    39,39,39,38,38,38,37,37,37,37,36,36,36,36,35,35,35,35,34,34,33,
    33,33,31,31,30,30,30,29,29,29,29,29,27,27,27,26,25,25,24,24,24,
    24,23,23,23,22,21,21,21,21,21,21,21,20
  };
  const int n3c2w2_h[] = {
    120, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,99,98,98,98,97,97,97,97,97,96,96,96,96,96,95,95,95,95,95,
    95,94,94,94,93,93,93,93,92,92,92,91,91,91,90,90,89,89,89,88,88,
    88,87,86,86,85,85,85,85,84,84,83,83,83,82,82,82,81,81,80,80,80,
    80,79,79,79,79,78,78,78,77,77,77,76,76,75,75,75,74,74,74,73,72,
    72,72,72,72,71,71,71,71,69,69,69,69,68,68,68,66,66,66,65,65,64,
    64,64,63,63,62,61,61,61,61,61,61,60,60,59,59,59,59,58,58,57,56,
    56,56,56,55,55,55,54,54,53,52,52,51,51,51,51,51,50,50,49,48,45,
    45,44,44,44,43,43,42,42,42,42,41,39,38,38,38,37,37,37,37,36,36,
    35,35,34,34,33,33,33,32,32,31,30,30,30,30,29,28,28,28,28,27,27,
    26,26,25,25,25,25,24,24,23,22,22,20
  };
  const int n3c2w2_i[] = {
    120, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,99,99,99,98,98,97,97,97,96,96,95,95,95,93,93,92,92,92,
    92,91,91,91,90,89,89,89,89,88,88,88,88,87,87,87,87,87,86,86,86,
    86,86,85,85,85,84,84,84,84,84,83,83,82,81,80,80,79,78,77,77,76,
    76,76,75,74,74,74,73,73,73,72,72,71,70,69,68,66,66,66,66,65,65,
    65,65,64,64,63,63,62,61,61,61,60,59,59,59,59,58,58,58,57,57,57,
    56,55,55,55,55,55,54,54,54,53,52,52,52,52,52,51,51,50,50,50,50,
    49,49,49,49,48,47,47,46,46,45,45,45,44,43,43,42,42,42,41,41,41,
    40,39,38,38,37,37,36,36,36,35,34,34,33,33,33,33,32,32,31,31,31,
    30,30,29,29,29,29,28,28,28,28,28,27,27,27,26,25,25,25,25,24,24,
    24,24,23,23,22,22,21,21,21,21,20,20
  };
  const int n3c2w2_j[] = {
    120, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,100,99,97,97,96,96,96,96,95,94,94,94,94,93,92,91,91,91,
    90,90,90,90,90,90,89,89,89,89,88,88,87,87,87,87,86,86,85,84,84,
    83,83,83,83,83,82,82,82,82,82,81,81,81,80,80,79,78,78,78,76,76,
    76,75,75,75,75,74,74,74,74,73,73,73,72,72,71,71,71,70,69,69,68,
    68,68,67,67,66,66,66,65,65,65,64,64,63,63,63,62,62,61,60,60,60,
    60,58,58,58,58,58,58,57,57,57,57,57,55,54,54,53,52,52,52,52,52,
    52,51,51,51,50,50,49,49,48,47,47,47,46,46,46,46,45,45,44,43,43,
    43,43,42,42,42,42,42,41,41,41,40,40,40,39,39,39,38,38,38,38,37,
    37,37,36,36,36,36,35,35,34,34,33,31,30,30,29,29,28,28,28,28,25,
    25,24,24,22,22,21,21,21,20,20,20,20
  };
  const int n3c2w2_k[] = {
    120, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,99,99,99,99,98,96,96,96,95,95,95,94,94,94,94,93,93,93,93,
    93,92,92,91,91,91,90,90,89,89,89,89,89,88,87,87,87,86,85,85,85,
    84,84,84,83,83,82,82,81,81,81,80,80,79,79,79,79,78,77,77,76,76,
    75,75,75,74,74,74,73,73,73,72,72,72,72,72,71,71,71,71,71,71,70,
    69,69,68,67,67,67,67,67,67,66,66,65,65,64,64,64,64,63,63,63,62,
    62,61,61,61,61,60,59,59,58,57,57,57,57,56,56,56,55,54,54,54,54,
    53,52,51,51,50,49,49,49,48,47,47,47,47,46,46,46,45,45,45,45,45,
    44,43,42,42,42,41,41,41,41,40,40,39,38,38,37,36,36,36,36,35,35,
    34,33,33,33,33,32,32,32,31,31,31,31,30,30,28,28,28,28,27,27,26,
    26,26,25,23,22,22,21,21,21,21,20,20
  };
  const int n3c2w2_l[] = {
    120, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,99,99,99,98,97,97,97,97,96,96,95,95,95,94,94,94,94,94,
    94,93,93,92,92,92,92,92,91,91,90,89,89,88,88,87,87,86,86,85,85,
    85,84,84,84,84,81,81,80,80,80,80,79,78,78,77,77,77,77,77,76,76,
    75,75,74,73,73,73,72,72,71,71,70,69,69,69,69,69,68,68,68,67,67,
    67,66,66,66,66,66,66,65,65,65,64,64,63,63,63,63,62,62,61,61,61,
    60,60,59,58,58,57,57,57,56,56,56,55,55,55,55,54,54,53,53,52,51,
    51,51,51,51,51,50,49,49,49,48,48,47,47,46,45,45,44,44,44,44,43,
    43,43,42,42,40,40,40,40,39,39,38,38,37,37,36,36,36,34,34,34,33,
    32,32,31,31,30,30,29,28,28,28,28,28,27,27,27,27,27,26,26,25,25,
    25,24,24,23,22,22,21,21,21,20,20,20
  };
  const int n3c2w2_m[] = {
    120, // Capacity
    200, // Number of items
    // Size of items (sorted)
    99,99,99,98,98,98,97,97,97,97,97,96,96,95,95,95,95,95,94,94,94,
    93,92,92,92,91,90,90,90,89,89,89,89,89,88,87,87,86,86,85,85,85,
    85,84,84,84,84,84,83,83,83,83,82,82,82,81,81,81,80,80,80,78,77,
    77,76,76,75,75,74,74,73,72,71,71,70,70,70,70,70,69,68,68,68,68,
    67,67,66,66,66,66,66,65,65,64,64,63,62,62,62,61,61,61,61,60,60,
    59,59,59,59,58,58,58,57,57,57,57,57,56,56,55,55,54,54,53,53,53,
    52,52,52,51,51,50,50,50,50,50,49,49,48,48,47,47,47,47,47,46,45,
    45,44,43,43,43,43,42,42,40,39,39,39,39,39,38,38,37,37,37,36,36,
    36,35,35,34,33,33,33,33,32,32,32,32,31,31,30,29,27,27,26,24,24,
    24,22,22,22,22,22,22,22,21,21,20
  };
  const int n3c2w2_n[] = {
    120, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,100,99,99,98,98,98,97,97,97,97,96,96,96,96,95,95,95,95,
    95,94,94,94,94,92,92,92,90,90,90,89,88,88,87,87,87,86,86,84,83,
    83,82,81,81,81,81,81,80,80,79,79,78,78,78,77,77,77,77,77,77,76,
    76,76,75,75,75,74,74,73,73,73,72,72,72,71,71,71,70,70,69,68,68,
    67,67,66,66,65,64,63,63,63,63,63,62,62,62,62,61,61,60,60,59,59,
    59,58,58,58,58,57,57,57,57,57,55,55,55,54,54,54,53,53,53,52,52,
    50,50,49,48,48,48,47,47,46,46,46,46,44,44,44,43,43,43,42,42,42,
    41,41,41,41,41,41,41,40,40,38,38,37,37,37,37,36,36,36,36,36,35,
    35,35,34,34,34,33,33,33,32,32,31,30,30,29,29,28,28,28,27,27,27,
    26,26,26,26,26,25,25,23,23,22,22,20
  };
  const int n3c2w2_o[] = {
    120, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,99,99,98,98,97,97,96,96,96,96,95,94,93,93,92,91,90,89,
    89,88,88,88,88,88,88,88,87,87,87,87,87,87,87,87,86,86,85,85,85,
    84,83,83,82,82,82,81,81,81,80,80,79,78,78,78,77,77,76,76,76,76,
    75,75,74,74,74,74,74,74,72,72,72,72,71,71,70,70,70,70,70,69,68,
    67,67,67,67,66,66,66,66,66,65,65,64,64,63,62,61,61,61,61,60,60,
    60,60,58,58,57,57,57,57,56,56,55,55,55,55,54,54,53,53,53,52,52,
    52,52,52,51,51,51,51,49,49,49,49,48,47,47,47,46,45,44,44,44,44,
    44,43,42,42,42,41,41,40,40,39,39,39,39,38,38,36,36,36,36,35,35,
    35,34,34,34,34,34,34,33,33,33,33,31,30,29,29,28,26,25,25,25,24,
    24,24,24,23,22,22,21,21,21,20,20,20
  };
  const int n3c2w2_p[] = {
    120, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,100,100,99,99,97,97,97,97,97,97,96,96,95,95,94,94,93,
    93,92,91,90,90,90,90,90,89,89,89,89,89,89,88,88,87,87,86,86,85,
    85,85,84,84,84,84,84,83,83,83,82,81,81,81,81,81,80,79,79,78,78,
    78,77,76,76,75,75,75,74,74,74,74,73,73,71,71,70,70,70,70,70,68,
    67,67,67,67,65,65,65,65,65,64,64,63,62,62,62,62,61,60,59,59,59,
    58,58,58,57,56,56,55,55,54,54,53,53,53,53,52,52,52,52,51,51,51,
    51,51,51,51,51,50,50,50,50,49,49,49,48,48,48,47,47,46,46,46,46,
    45,45,44,44,43,43,43,42,42,39,39,39,39,38,38,37,37,37,37,36,35,
    34,33,33,33,33,33,32,32,32,32,31,31,30,30,30,29,29,29,27,27,27,
    26,25,25,23,23,22,22,22,21,20,20,20,20
  };
  const int n3c2w2_q[] = {
    120, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,100,99,99,99,99,98,96,96,96,95,94,94,94,93,93,93,92,92,
    92,91,91,90,88,88,88,88,88,87,86,85,85,85,84,84,84,83,83,83,82,
    82,82,82,81,81,81,81,81,79,79,78,77,77,76,76,76,75,75,74,73,73,
    72,72,71,70,70,70,70,69,69,69,69,68,68,67,67,66,66,65,65,65,65,
    64,64,64,64,64,64,63,63,63,63,63,63,63,62,62,62,62,61,60,59,59,
    59,59,59,59,59,58,58,58,58,57,57,57,56,55,55,55,54,53,53,53,53,
    53,52,52,51,51,50,50,50,50,49,49,49,48,48,47,47,47,45,44,44,44,
    42,41,41,41,41,41,40,40,40,40,39,38,38,38,37,37,37,37,37,36,36,
    36,35,34,32,32,32,31,31,31,30,30,29,29,29,29,28,26,26,26,25,24,
    24,24,23,23,22,21,20,20,20,20,20,20
  };
  const int n3c2w2_r[] = {
    120, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,99,99,99,98,98,98,97,97,97,97,97,96,96,96,95,95,95,93,93,
    92,92,91,91,91,91,90,90,89,89,89,88,88,87,87,87,87,86,86,86,85,
    85,85,85,84,84,84,84,84,83,83,83,82,82,82,81,81,81,81,80,80,80,
    79,79,79,78,78,77,76,76,74,74,74,74,73,73,72,72,72,72,72,72,71,
    71,71,70,69,68,68,68,67,66,66,66,65,65,65,64,63,62,62,62,61,61,
    61,61,59,58,58,58,57,57,57,56,56,56,56,56,56,55,55,55,55,55,55,
    54,53,53,50,48,48,46,46,46,46,46,45,45,45,45,45,45,43,43,43,42,
    42,42,42,41,41,39,38,38,38,37,37,37,36,36,35,35,35,35,34,34,33,
    33,32,32,32,32,31,30,30,30,29,29,29,29,27,25,25,25,25,25,25,25,
    24,24,23,23,22,22,22,21,21,21,20,20
  };
  const int n3c2w2_s[] = {
    120, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,100,100,98,98,97,97,97,96,96,96,96,95,95,95,94,94,94,
    94,93,93,93,93,92,92,92,91,91,91,91,91,91,90,90,89,89,86,86,86,
    85,85,85,85,84,83,82,82,82,81,80,80,79,79,79,78,78,78,78,77,77,
    77,77,75,75,75,74,74,74,74,74,74,73,73,73,72,72,72,71,71,71,70,
    68,68,68,67,67,67,67,67,66,66,66,66,65,64,64,64,63,63,62,62,62,
    62,61,61,60,59,58,57,57,56,56,55,55,55,54,53,53,53,53,52,52,52,
    51,50,50,49,48,47,47,47,47,46,46,45,45,45,45,45,44,44,44,42,41,
    40,40,40,39,39,39,38,38,38,36,36,36,36,36,36,35,35,35,35,34,34,
    34,34,33,33,33,32,32,31,31,30,30,30,29,28,28,27,27,27,26,25,24,
    24,23,23,23,23,22,22,22,22,21,21,21,20
  };
  const int n3c2w2_t[] = {
    120, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,99,98,97,97,97,97,96,96,96,95,95,95,94,94,94,94,93,93,
    92,92,92,91,91,91,91,91,90,89,88,87,87,86,85,85,84,84,83,83,83,
    82,82,81,81,80,80,80,80,80,80,79,79,79,79,79,79,78,77,77,76,76,
    76,76,75,75,74,74,73,71,71,71,70,70,69,69,69,69,68,68,68,68,67,
    67,67,67,67,67,67,67,66,65,64,63,63,63,62,61,61,61,61,61,61,60,
    60,60,59,59,58,58,57,57,56,56,55,55,55,55,55,55,54,54,53,53,52,
    51,51,50,49,49,48,48,47,46,46,46,46,45,45,44,43,43,43,43,43,42,
    42,41,41,41,40,40,39,39,39,38,38,38,37,37,37,37,37,36,35,35,35,
    35,35,34,34,33,33,32,32,31,31,31,31,31,31,31,31,30,30,30,29,28,
    28,25,25,25,24,24,24,22,22,22,21,20
  };
  const int n3c2w4_a[] = {
    120, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,99,99,98,98,97,97,97,96,96,96,95,94,94,93,
    93,92,92,92,91,91,91,90,90,89,89,88,88,87,87,86,86,85,85,85,83,
    83,83,83,82,82,81,80,80,80,80,79,79,79,78,78,78,77,77,77,77,77,
    77,76,76,75,74,74,74,73,73,73,72,72,72,71,71,70,70,70,70,69,69,
    69,69,69,68,68,68,67,67,67,66,66,66,66,65,64,64,64,64,64,64,64,
    63,63,61,61,61,61,60,60,59,59,58,58,58,57,57,57,57,57,56,56,56,
    55,55,55,55,54,54,53,53,53,53,53,52,51,51,51,50,50,49,49,49,48,
    48,48,47,47,47,46,46,45,44,44,44,44,43,43,43,42,41,40,40,39,38,
    38,38,38,38,38,38,38,37,37,37,36,36,36,36,35,35,35,34,33,33,33,
    32,32,32,32,31,31,31,30,30,30,30,30,30
  };
  const int n3c2w4_b[] = {
    120, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,100,100,98,98,98,98,98,98,97,97,97,97,96,96,95,95,95,
    94,94,93,93,92,92,90,90,90,90,89,89,89,87,87,87,87,86,85,84,84,
    84,84,83,83,83,82,82,82,81,81,81,81,81,80,79,79,78,78,78,77,77,
    77,77,77,76,76,75,75,73,72,72,72,72,71,70,70,69,69,69,68,68,68,
    68,66,66,65,64,64,64,64,63,63,63,63,62,62,62,62,61,61,61,60,60,
    59,59,59,59,59,58,58,58,57,57,57,57,56,56,56,55,55,55,54,54,54,
    54,53,53,53,52,52,52,52,52,51,51,51,51,51,51,50,50,50,50,48,48,
    48,48,48,48,48,46,46,46,45,45,44,43,42,42,42,42,41,40,39,39,39,
    39,39,39,38,38,37,37,37,36,36,35,35,35,35,34,34,34,34,34,33,33,
    33,33,33,32,32,32,31,31,31,31,30,30,30
  };
  const int n3c2w4_c[] = {
    120, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,100,100,99,98,98,97,97,97,97,97,97,97,97,96,96,96,96,
    96,95,95,95,95,93,92,90,90,90,90,90,90,90,89,89,89,89,89,89,88,
    88,88,88,88,88,87,87,86,86,84,83,83,82,82,82,82,81,81,81,81,80,
    80,80,79,79,79,79,78,78,78,78,78,78,77,77,77,77,77,77,76,76,75,
    74,73,73,73,73,73,73,73,73,72,72,72,72,71,71,71,70,70,69,69,69,
    69,68,68,68,68,68,68,67,67,66,66,66,66,66,65,65,65,65,64,63,63,
    62,61,60,60,60,59,59,58,58,58,57,57,56,56,55,55,55,55,55,55,54,
    54,54,54,53,53,53,53,53,52,52,52,51,51,50,50,50,49,49,48,48,47,
    47,47,46,46,45,45,45,44,44,44,41,40,40,40,40,39,38,37,37,37,36,
    36,36,36,35,35,34,34,33,32,32,31,31,30
  };
  const int n3c2w4_d[] = {
    120, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,99,99,98,98,98,98,98,98,97,97,97,96,96,96,96,95,95,95,
    94,94,93,92,92,92,92,91,90,90,89,89,89,89,89,88,88,88,87,87,86,
    85,85,85,84,83,82,81,81,81,81,81,80,79,78,78,77,77,77,75,75,75,
    74,74,74,74,74,73,73,73,73,72,72,72,72,72,71,71,70,70,70,69,69,
    68,68,68,67,67,67,67,66,66,66,66,66,66,65,65,63,63,63,63,62,62,
    62,61,60,60,60,60,60,60,59,59,59,58,58,57,57,56,56,56,56,56,55,
    55,55,54,54,54,53,53,53,52,52,52,51,51,50,50,50,50,49,49,49,48,
    48,48,46,46,46,46,46,45,45,45,45,44,44,44,43,42,42,42,41,40,40,
    40,39,39,39,39,39,38,38,37,37,37,37,36,36,36,35,35,35,34,34,34,
    34,33,33,32,32,31,31,31,30,30,30,30
  };
  const int n3c2w4_e[] = {
    120, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,99,99,99,98,98,98,98,97,97,96,95,95,94,94,94,94,93,93,93,
    93,90,90,90,89,89,89,88,87,87,86,86,86,86,85,84,83,83,83,82,81,
    81,81,80,80,80,80,79,79,79,78,78,77,77,77,77,77,77,76,76,76,76,
    75,75,75,75,73,73,73,72,72,72,71,69,69,68,68,68,67,67,67,66,66,
    66,66,66,66,66,66,65,65,64,63,63,62,62,62,62,61,61,61,60,60,60,
    60,59,59,59,58,58,58,58,57,57,57,57,57,56,56,56,55,54,54,54,53,
    53,52,51,51,50,50,50,50,49,49,49,49,49,49,48,48,48,48,48,47,47,
    47,46,45,44,44,44,44,44,44,43,43,43,43,42,42,42,42,42,41,40,39,
    38,38,37,37,37,37,37,37,37,37,36,36,36,36,36,35,35,35,35,34,34,
    34,33,33,33,33,33,32,32,32,31,30,30
  };
  const int n3c2w4_f[] = {
    120, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,100,99,99,99,99,98,98,97,97,97,96,96,95,95,95,95,94,94,
    94,93,92,90,90,90,90,89,88,88,88,87,87,86,86,86,85,85,85,84,84,
    83,83,82,82,81,81,81,80,80,79,79,79,78,78,78,78,77,77,77,76,76,
    76,76,75,75,75,74,73,73,72,72,72,72,71,71,71,71,71,71,71,70,70,
    69,68,68,68,67,67,67,67,66,66,66,66,66,65,64,64,64,64,64,64,63,
    63,63,62,62,61,61,61,61,60,60,60,60,60,59,58,58,58,57,57,57,57,
    56,55,54,54,54,54,54,53,52,52,51,51,51,50,50,50,50,49,48,48,47,
    47,46,46,45,45,44,43,43,42,42,41,41,41,41,41,41,40,40,40,40,40,
    40,39,39,39,39,38,38,37,37,37,36,36,36,36,36,36,35,35,35,35,33,
    33,33,33,33,32,32,31,31,31,30,30,30
  };
  const int n3c2w4_g[] = {
    120, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,100,99,99,99,99,99,99,98,98,98,98,97,97,96,96,96,95,95,
    95,94,94,94,94,94,93,93,92,91,91,91,91,91,91,90,90,89,88,88,88,
    87,87,87,86,86,85,85,85,84,84,83,83,83,83,83,82,82,82,82,82,81,
    81,81,81,80,80,80,80,79,78,78,77,77,77,76,76,76,76,76,76,75,75,
    74,74,73,73,73,73,72,72,70,70,69,69,68,68,68,68,68,68,68,67,67,
    67,67,67,66,66,65,65,64,63,63,63,62,61,61,61,61,60,60,60,60,59,
    58,58,58,58,57,56,56,53,53,53,53,53,53,52,52,52,52,51,51,50,50,
    49,49,49,48,48,48,48,48,47,46,45,45,44,44,43,43,43,43,42,42,42,
    42,41,41,41,41,40,40,39,39,38,37,37,36,36,36,36,36,35,35,35,35,
    35,35,34,33,33,33,32,32,32,31,30,30
  };
  const int n3c2w4_h[] = {
    120, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,100,99,99,98,98,98,97,97,97,97,95,95,94,94,94,94,93,93,
    93,93,92,92,92,91,91,91,90,89,88,88,88,87,86,85,85,85,85,85,84,
    83,83,82,82,81,81,80,79,78,78,78,78,77,77,76,76,76,75,75,75,74,
    74,74,73,73,73,73,72,72,70,70,70,70,69,69,69,69,69,68,68,68,68,
    67,67,67,67,67,67,66,66,66,66,66,66,65,65,65,64,63,63,63,62,62,
    61,61,60,60,60,60,59,59,59,58,57,57,57,56,56,55,55,54,53,53,53,
    53,53,52,52,52,51,51,51,51,50,50,50,49,49,49,49,48,48,48,48,47,
    47,46,46,46,45,45,44,44,44,44,43,43,43,43,43,42,42,42,41,41,40,
    40,40,39,39,39,39,39,39,39,38,38,37,36,36,36,36,35,35,35,34,33,
    33,33,33,33,32,32,32,32,32,32,30,30
  };
  const int n3c2w4_i[] = {
    120, // Capacity
    200, // Number of items
    // Size of items (sorted)
    99,98,98,98,98,98,96,96,95,95,95,94,93,92,92,92,91,91,91,90,89,
    89,89,88,88,88,88,88,87,86,85,85,84,84,83,83,83,82,82,81,81,81,
    80,80,80,80,79,79,78,78,78,78,77,77,77,77,77,76,76,75,75,75,74,
    74,74,74,74,73,72,72,71,71,71,71,70,69,69,69,69,68,68,68,67,67,
    67,67,67,67,66,66,66,66,65,65,65,65,64,64,64,63,63,63,63,63,63,
    62,62,61,61,61,61,61,61,60,60,60,60,59,59,58,58,58,58,57,56,55,
    55,54,54,53,53,53,52,52,52,52,52,52,52,52,52,51,51,51,50,50,50,
    50,50,50,49,49,49,48,48,48,48,47,47,47,46,46,45,45,44,44,43,43,
    43,43,43,42,42,41,41,40,39,39,38,38,37,37,37,36,36,35,35,35,34,
    34,33,33,33,32,32,31,31,30,30,30
  };
  const int n3c2w4_j[] = {
    120, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,99,99,98,97,97,96,96,96,95,95,94,94,93,93,91,91,91,91,
    90,90,90,90,88,88,88,88,87,87,86,86,86,86,86,85,85,85,85,84,84,
    83,83,83,82,82,82,82,82,82,82,81,81,80,80,80,80,79,79,78,78,77,
    77,76,76,75,75,75,74,73,73,73,73,72,72,72,72,71,71,70,70,70,69,
    69,69,69,69,68,68,68,67,67,67,66,66,65,65,65,65,65,65,65,65,65,
    64,64,64,64,64,64,64,63,63,62,62,62,62,60,60,60,59,59,58,58,58,
    58,58,57,56,56,56,56,56,55,55,54,54,53,53,53,53,52,52,52,52,52,
    52,52,51,51,51,50,50,49,49,49,47,46,46,46,46,45,45,44,44,44,44,
    44,44,43,43,42,41,41,41,38,38,38,37,35,35,35,35,34,33,33,33,33,
    33,33,33,32,32,31,31,31,30,30,30,30
  };
  const int n3c2w4_k[] = {
    120, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,100,100,99,99,99,98,98,98,98,98,97,97,97,96,96,95,94,
    94,94,94,94,93,93,92,91,91,90,90,90,90,89,89,88,88,88,88,88,87,
    87,87,86,85,85,85,85,85,85,85,83,83,82,82,82,82,81,81,81,80,80,
    80,79,78,77,77,77,76,76,76,75,75,74,74,74,74,73,73,73,72,72,71,
    71,71,71,69,69,69,68,68,67,67,66,66,66,65,65,64,64,64,64,64,64,
    64,63,62,62,61,61,61,61,60,60,60,60,60,60,59,58,58,57,57,57,57,
    56,56,55,55,54,54,53,53,53,53,53,52,52,52,52,52,52,50,49,48,48,
    48,48,48,47,47,47,47,47,47,47,47,46,46,45,44,44,44,44,42,42,42,
    42,42,41,41,41,40,40,39,38,38,37,37,37,37,37,37,36,35,35,35,35,
    35,34,34,33,33,32,32,31,31,31,30,30,30
  };
  const int n3c2w4_l[] = {
    120, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,99,99,99,99,99,98,97,97,97,97,95,95,95,94,94,94,93,93,93,
    92,92,92,92,91,91,91,91,90,90,90,89,89,88,88,88,88,87,87,87,87,
    86,85,85,85,84,84,84,83,83,83,82,82,81,81,80,80,80,80,80,79,79,
    78,78,78,78,78,77,77,77,77,77,76,76,76,76,75,75,74,74,74,73,73,
    72,72,71,71,71,70,70,70,69,68,68,68,68,67,66,66,65,65,65,65,65,
    64,63,62,62,61,61,61,61,61,60,60,60,58,58,58,58,57,56,56,56,56,
    56,56,55,55,55,55,55,54,53,52,52,52,51,51,51,51,49,49,47,47,46,
    45,45,45,45,45,45,44,44,44,44,43,42,41,41,41,40,40,39,39,39,39,
    38,38,38,37,37,36,36,36,36,36,36,36,35,35,35,35,34,34,34,34,34,
    33,33,33,33,33,32,32,32,31,31,30,30
  };
  const int n3c2w4_m[] = {
    120, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,100,99,99,99,99,99,99,99,98,98,98,98,97,97,97,97,97,96,
    96,96,95,95,95,95,95,95,94,93,92,92,92,92,92,91,91,90,90,90,89,
    88,88,86,86,86,85,85,85,84,83,82,82,82,82,81,81,81,80,80,80,80,
    80,79,79,79,79,78,78,78,78,77,76,76,75,74,73,73,73,72,72,72,71,
    71,70,70,69,69,69,68,68,68,68,68,67,67,67,66,66,65,64,64,64,64,
    64,63,63,63,63,62,62,62,62,62,62,61,61,61,61,60,59,59,58,58,57,
    57,55,54,54,53,53,53,53,53,53,53,53,53,53,52,52,51,51,50,50,50,
    49,48,46,46,45,45,45,45,44,43,42,41,41,41,40,40,40,40,39,39,38,
    38,38,38,38,37,37,37,36,36,35,35,35,35,35,34,34,34,34,33,33,33,
    32,32,32,32,32,32,32,31,30,30,30,30
  };
  const int n3c2w4_n[] = {
    120, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,100,99,99,99,99,98,98,98,98,97,97,97,96,96,
    95,95,95,94,93,93,92,92,92,91,90,90,89,88,88,88,88,88,88,87,87,
    87,87,86,85,85,85,85,85,84,84,82,82,82,81,81,81,80,80,80,80,80,
    80,80,78,78,78,78,78,77,77,77,75,75,75,74,74,73,72,71,71,71,70,
    70,70,70,69,69,69,69,68,68,67,67,65,65,65,64,64,64,64,64,63,63,
    63,62,62,61,61,60,60,59,59,59,58,58,57,57,56,56,56,56,56,55,55,
    55,55,54,54,54,53,53,53,53,52,52,51,51,51,50,50,50,50,49,49,49,
    48,47,47,47,46,46,46,46,45,45,45,44,44,44,44,44,44,44,43,43,41,
    41,40,40,39,39,39,38,38,37,37,36,36,36,36,36,36,35,35,34,33,33,
    33,32,32,32,32,32,32,31,31,30,30,30,30
  };
  const int n3c2w4_o[] = {
    120, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,99,99,99,97,97,97,96,96,96,95,95,95,94,93,
    93,93,93,93,93,92,92,92,90,90,90,90,90,90,89,89,89,88,88,88,88,
    87,87,86,86,85,84,83,83,83,82,82,82,82,81,81,80,80,80,80,79,79,
    78,78,78,77,77,77,77,77,76,75,75,74,74,73,72,71,70,69,69,68,67,
    67,67,67,67,66,66,66,65,65,65,65,64,64,64,63,63,61,61,61,61,60,
    60,59,59,59,59,58,57,57,57,57,56,56,55,55,55,55,54,54,54,54,53,
    53,53,52,52,52,51,51,51,51,51,50,50,50,50,50,49,49,49,48,48,47,
    47,47,47,47,45,45,44,44,44,43,43,42,42,42,41,41,41,41,40,40,40,
    39,39,39,38,38,37,37,37,36,36,36,36,35,34,34,34,34,34,33,33,33,
    33,32,32,31,31,31,31,31,31,30,30,30,30
  };
  const int n3c2w4_p[] = {
    120, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,100,99,99,99,99,99,99,98,98,98,97,97,96,96,94,94,93,93,
    93,93,92,92,91,91,91,90,90,90,90,90,89,89,89,89,89,88,88,88,87,
    87,87,86,86,86,86,85,84,84,83,83,83,83,83,82,82,82,82,81,81,81,
    81,81,80,80,79,79,79,79,79,78,78,78,78,78,77,77,76,76,75,75,75,
    74,74,74,74,72,72,72,71,71,71,70,70,70,70,69,68,67,67,67,67,67,
    66,66,66,66,65,65,64,63,63,62,61,60,60,60,60,59,59,59,59,58,58,
    58,58,57,56,56,56,55,55,55,54,54,53,53,52,52,52,52,52,51,51,51,
    51,50,49,49,49,48,47,46,46,46,45,44,44,43,42,42,41,40,40,40,40,
    40,39,39,39,39,38,38,38,38,37,37,37,37,37,37,36,36,35,35,35,35,
    34,33,33,33,32,31,31,30,30,30,30,30
  };
  const int n3c2w4_q[] = {
    120, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,100,100,98,98,98,98,98,97,97,97,97,97,97,97,96,96,96,
    96,95,94,93,93,93,93,92,92,92,92,91,90,90,89,89,89,88,87,86,86,
    86,86,85,85,85,84,84,84,83,83,82,82,81,81,81,80,80,80,79,79,79,
    79,78,78,78,78,77,77,77,77,76,76,76,75,75,75,74,74,74,74,73,72,
    72,72,72,72,72,71,70,70,70,69,69,69,68,68,68,67,66,66,65,65,65,
    64,64,64,64,64,63,63,63,63,62,62,61,60,60,59,59,59,58,58,57,57,
    57,56,56,55,55,55,55,55,54,54,54,54,53,53,53,52,51,51,51,50,50,
    50,49,48,48,48,47,47,47,47,46,46,46,46,45,44,44,44,43,43,43,42,
    42,42,41,41,41,40,40,40,39,39,39,39,38,38,38,37,36,36,36,36,35,
    35,34,34,33,32,32,32,32,32,32,31,31,30
  };
  const int n3c2w4_r[] = {
    120, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,100,100,99,99,99,99,98,98,98,98,97,97,96,96,96,95,95,
    94,94,94,93,93,93,93,92,92,91,91,91,90,90,89,89,88,88,88,88,88,
    87,87,87,87,86,86,85,85,84,84,84,84,83,82,82,81,81,81,81,81,80,
    80,79,79,79,78,78,78,78,78,78,77,77,77,77,77,76,75,75,74,74,73,
    73,72,72,72,72,71,71,70,70,70,70,70,69,68,68,68,68,68,68,67,67,
    66,66,65,65,65,65,65,65,64,64,63,62,62,61,60,60,60,60,59,59,58,
    58,58,57,56,56,56,55,55,55,54,54,54,54,54,54,53,53,53,53,53,53,
    52,52,52,51,50,50,49,49,49,48,48,47,47,47,46,46,46,46,45,45,44,
    44,43,43,43,42,42,42,42,42,42,41,40,39,38,38,38,38,38,38,37,37,
    37,36,36,35,34,34,33,32,32,32,31,30,30
  };
  const int n3c2w4_s[] = {
    120, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,99,99,99,98,98,97,96,96,96,96,95,95,95,94,94,94,93,93,93,
    93,93,93,93,93,92,92,92,91,91,90,90,89,89,89,88,88,88,88,88,87,
    87,86,86,86,86,86,86,86,85,84,84,83,83,83,81,81,81,81,80,80,79,
    79,79,79,78,78,78,78,77,77,77,77,76,76,76,75,75,74,74,73,73,72,
    72,71,71,70,70,70,70,69,69,69,68,68,68,68,68,67,67,67,66,66,66,
    66,65,65,65,64,63,63,62,61,61,59,58,58,57,57,57,56,56,56,55,55,
    55,54,52,51,51,50,50,50,50,50,50,49,49,49,49,49,49,48,48,48,47,
    47,47,46,46,46,46,46,45,45,44,43,43,43,42,42,42,41,41,41,41,40,
    40,40,40,40,39,39,38,37,37,37,37,37,37,36,36,36,36,36,35,35,35,
    34,34,33,32,32,32,31,31,30,30,30,30
  };
  const int n3c2w4_t[] = {
    120, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,99,99,99,98,98,98,97,97,97,96,96,96,96,96,95,95,95,95,
    94,94,94,92,92,92,91,91,91,91,90,90,90,90,90,89,89,88,88,87,87,
    87,87,86,86,86,86,86,85,85,85,84,83,82,82,81,81,81,81,81,81,81,
    80,80,80,80,78,78,78,78,78,77,77,77,76,75,75,75,75,73,73,73,72,
    71,71,71,71,70,70,69,69,69,68,67,67,67,66,66,66,65,65,65,64,63,
    63,63,62,62,62,62,61,61,61,61,61,60,60,60,59,59,59,59,58,58,57,
    56,56,56,56,56,55,55,54,54,53,53,53,52,52,52,51,51,50,50,50,49,
    49,48,48,48,48,46,46,46,46,45,45,44,44,44,43,43,43,43,43,43,42,
    41,41,41,41,40,39,39,38,37,36,36,36,36,35,35,35,34,34,34,34,33,
    33,32,32,32,32,31,31,30,30,30,30,30
  };
  const int n3c3w1_a[] = {
    150, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,100,99,99,99,98,98,98,97,96,96,96,95,95,95,94,93,92,91,
    91,91,90,90,90,89,87,87,86,86,86,84,84,83,83,82,82,82,80,80,80,
    79,78,77,77,77,77,77,75,74,73,73,73,73,72,71,71,71,70,69,68,68,
    68,68,67,65,65,65,65,65,65,64,63,63,62,62,62,61,60,59,58,58,57,
    57,54,54,53,53,52,52,52,52,51,51,50,50,49,49,49,48,48,47,46,45,
    44,44,44,43,42,42,41,40,39,39,39,39,39,38,37,37,37,37,37,37,37,
    37,36,36,35,35,35,35,34,34,33,33,32,32,31,31,29,29,29,28,27,26,
    26,25,25,24,23,21,21,21,20,20,18,18,17,17,17,16,16,16,16,15,15,
    14,13,13,13,13,13,13,13,12,11,9,8,8,7,6,6,6,5,5,5,5,4,4,4,4,4,
    3,3,2,2,2,1,1
  };
  const int n3c3w1_b[] = {
    150, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,99,99,98,98,98,98,98,98,98,96,95,91,91,90,90,90,90,90,89,
    88,88,87,87,87,85,85,85,84,84,83,83,82,81,81,81,81,80,80,80,80,
    80,79,79,79,79,78,77,77,76,75,74,74,73,73,73,73,73,72,71,71,71,
    70,70,70,69,69,69,69,69,68,68,68,67,67,66,65,65,64,64,64,63,63,
    63,62,61,61,61,61,61,59,59,59,58,58,58,58,57,56,56,56,55,55,55,
    55,54,54,53,53,52,52,51,51,50,50,50,50,49,49,48,48,48,46,46,46,
    46,43,42,42,42,40,39,39,39,39,39,38,36,36,36,35,35,34,34,33,32,
    31,31,29,27,26,26,26,25,25,24,24,24,23,22,22,21,21,20,20,19,19,
    18,18,17,17,17,17,17,15,15,14,14,14,13,13,12,12,12,12,12,10,10,
    10,10,10,10,10,9,8,5,4,4,4,1
  };
  const int n3c3w1_c[] = {
    150, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,100,100,99,99,98,98,97,96,96,95,95,94,94,94,93,91,90,
    90,89,89,89,89,88,88,88,88,88,88,87,85,85,84,84,84,83,83,82,82,
    81,80,80,78,78,78,78,78,78,78,77,77,77,76,76,76,75,75,74,74,74,
    74,74,73,73,72,70,67,67,67,66,66,66,66,66,65,65,65,63,63,63,62,
    62,61,61,61,61,61,60,60,59,58,57,56,54,54,54,53,52,52,51,50,50,
    49,48,48,48,47,47,47,47,46,46,46,45,45,45,42,42,39,39,39,38,38,
    37,37,37,36,36,35,34,34,34,33,33,31,31,31,31,31,29,28,28,27,27,
    26,26,26,26,26,26,25,25,25,24,23,22,22,22,21,21,21,21,20,20,19,
    16,16,16,15,15,15,14,14,13,13,12,12,12,11,10,10,10,9,9,9,8,7,
    7,6,6,6,5,5,5,3,3,3,2,1
  };
  const int n3c3w1_d[] = {
    150, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,100,100,99,99,99,98,97,97,96,96,96,95,95,95,94,94,93,
    92,92,92,91,91,90,89,87,87,86,86,86,86,86,85,84,84,83,83,81,80,
    80,79,78,78,77,76,76,76,73,72,72,71,70,70,67,67,67,66,66,65,63,
    63,62,62,61,60,60,59,58,57,56,56,56,55,55,55,55,54,54,54,53,53,
    53,52,52,51,51,50,50,50,49,48,48,47,46,46,44,44,44,44,44,43,41,
    41,40,40,40,39,39,39,39,36,36,36,36,36,35,35,35,35,33,33,33,32,
    32,32,32,31,30,30,29,29,29,29,28,28,26,26,26,25,25,25,25,25,24,
    23,23,22,22,22,22,21,21,21,21,21,20,20,20,20,20,19,18,17,17,17,
    17,15,15,15,14,13,13,12,12,12,12,11,10,10,9,9,9,8,8,8,7,7,6,6,
    5,4,4,4,3,3,3,2,1,1
  };
  const int n3c3w1_e[] = {
    150, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,100,99,99,99,98,98,98,98,97,97,97,97,95,95,94,94,93,93,
    92,92,91,91,90,90,90,90,89,89,89,89,88,88,87,86,85,84,84,84,84,
    83,83,82,82,82,82,81,80,79,78,78,77,76,76,75,74,74,74,73,72,71,
    71,70,70,70,70,70,70,69,69,68,68,68,67,66,65,64,64,63,63,62,62,
    61,60,59,57,57,57,56,55,55,55,55,54,54,53,53,52,52,52,52,50,48,
    48,48,47,47,46,46,45,45,44,44,43,43,43,42,42,42,42,41,41,40,40,
    39,39,36,35,34,33,32,32,31,30,29,29,28,28,27,27,24,24,24,24,23,
    23,23,23,23,23,21,21,20,20,19,19,18,17,17,17,16,16,15,15,15,15,
    14,14,13,13,13,12,12,12,12,11,11,11,10,10,9,9,8,8,8,8,7,7,7,6,
    5,4,4,3,3,1,1,1,1
  };
  const int n3c3w1_f[] = {
    150, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,100,99,99,98,98,98,98,96,96,95,95,93,92,92,92,91,89,89,
    88,88,88,87,87,87,87,86,86,86,85,85,84,83,83,82,80,80,80,79,79,
    78,78,77,76,76,75,75,74,74,73,73,73,72,71,70,70,70,69,69,69,69,
    68,68,66,66,66,66,65,64,64,64,64,64,64,63,63,63,62,62,61,60,60,
    59,58,58,58,58,58,58,57,57,55,55,55,53,52,52,52,51,51,50,50,50,
    49,49,49,49,49,48,48,46,46,45,45,45,44,43,42,42,42,41,41,40,40,
    40,39,39,39,37,37,37,36,36,36,36,35,35,35,33,33,33,33,32,32,31,
    31,31,31,30,29,29,29,29,28,27,27,27,26,26,24,22,22,22,21,21,20,
    19,18,17,17,16,16,15,14,14,13,12,11,11,11,11,10,9,8,7,7,7,7,7,
    6,6,5,4,4,4,3,3,2,1
  };
  const int n3c3w1_g[] = {
    150, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,97,97,97,96,96,96,96,95,95,95,95,95,94,94,92,92,91,91,
    90,89,87,86,86,86,86,85,84,84,84,84,83,83,81,81,81,80,78,77,77,
    76,75,75,74,74,73,73,73,72,71,71,71,70,70,69,68,66,65,65,64,64,
    64,64,63,63,63,62,61,61,61,60,60,60,60,59,58,58,58,58,58,58,57,
    57,55,55,55,54,54,53,52,52,51,51,51,51,51,51,50,49,49,49,48,47,
    46,46,45,45,44,44,44,43,43,43,41,41,40,40,40,39,37,36,36,35,35,
    35,35,34,34,34,33,32,31,31,30,30,30,29,29,28,28,27,27,27,27,25,
    25,24,23,22,22,21,21,21,21,21,21,21,20,19,18,17,17,16,16,15,15,
    14,14,13,13,13,13,13,12,11,10,9,9,8,8,6,6,5,5,5,5,4,4,4,3,3,3,
    2,2,2,1,1,1,1
  };
  const int n3c3w1_h[] = {
    150, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,99,99,98,98,97,96,96,96,96,96,96,95,94,94,94,93,92,91,
    91,90,89,89,89,88,87,86,86,86,86,85,85,85,84,84,84,84,84,84,83,
    82,82,81,80,78,78,77,77,77,77,77,76,76,75,75,74,74,74,74,70,70,
    70,69,69,69,68,68,68,68,67,66,66,66,65,65,65,64,64,64,64,64,63,
    63,62,62,60,58,57,56,56,56,56,56,56,55,55,55,55,55,53,53,51,51,
    51,50,50,49,47,47,47,44,43,43,43,42,42,40,40,38,38,38,37,37,37,
    36,36,35,34,34,34,33,33,33,33,32,32,30,30,29,28,28,27,27,26,26,
    26,25,25,25,25,25,24,24,23,23,22,22,21,21,21,19,19,19,18,17,17,
    16,16,15,14,14,14,13,13,13,13,12,11,11,10,10,9,9,9,8,8,8,7,7,
    7,6,4,4,4,4,3,2,1,1
  };
  const int n3c3w1_i[] = {
    150, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,99,99,99,98,97,96,94,93,93,93,92,92,91,90,
    89,89,88,88,88,88,88,88,88,86,86,86,86,86,85,85,84,84,84,83,83,
    83,83,83,83,82,82,81,79,79,76,76,76,76,75,75,75,75,75,75,74,74,
    73,72,71,71,71,68,68,67,67,67,66,66,66,65,65,64,64,63,63,63,62,
    62,62,61,60,60,60,58,58,57,57,56,56,55,55,55,54,54,54,54,53,51,
    50,50,49,48,48,47,47,47,46,46,45,45,44,43,43,41,40,40,39,39,39,
    37,37,37,36,34,33,32,31,31,31,31,30,30,29,29,29,29,29,28,27,24,
    24,23,23,23,23,23,22,22,21,21,20,19,19,18,18,17,17,17,17,16,16,
    16,15,15,15,15,15,14,14,14,13,12,12,12,12,11,11,11,10,8,8,7,6,
    6,5,5,5,5,5,4,4,4,3,2,1
  };
  const int n3c3w1_j[] = {
    150, // Capacity
    200, // Number of items
    // Size of items (sorted)
    99,99,99,98,98,98,96,95,95,94,94,94,93,93,92,92,92,91,91,90,88,
    86,86,85,85,84,84,84,83,82,82,82,81,81,81,80,80,79,79,79,78,78,
    78,77,77,77,76,74,74,73,73,72,71,71,71,71,70,70,68,68,68,67,66,
    66,66,66,66,65,64,63,63,63,62,61,60,60,59,58,58,58,57,57,57,57,
    56,55,54,53,53,51,51,51,51,50,50,50,49,47,47,47,46,46,45,45,45,
    45,45,44,43,43,42,42,41,41,40,40,39,39,37,37,36,36,35,35,34,34,
    34,34,34,33,32,32,32,31,31,29,28,27,27,26,26,26,25,25,25,25,25,
    25,25,25,22,22,22,21,21,21,21,21,21,19,19,19,18,17,17,17,17,17,
    17,16,16,15,14,14,14,13,13,12,11,10,10,10,10,9,8,7,6,5,4,4,4,
    4,3,3,3,3,3,3,2,2
  };
  const int n3c3w1_k[] = {
    150, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,99,99,99,99,98,98,98,97,96,95,94,93,93,93,92,91,91,91,91,
    91,90,90,88,88,88,87,87,87,86,86,85,85,84,84,84,83,83,82,81,81,
    81,81,77,77,76,76,75,74,74,74,73,73,72,72,71,71,70,69,69,69,69,
    68,68,66,66,65,64,63,63,63,62,61,61,59,59,59,58,58,57,57,57,57,
    55,55,53,53,52,52,49,49,49,48,48,47,47,46,46,46,46,45,45,44,43,
    43,43,41,40,40,40,39,39,38,38,38,37,37,35,35,35,34,34,33,33,32,
    31,31,29,29,28,28,27,26,25,25,24,24,24,23,23,23,23,23,23,22,22,
    22,21,20,19,19,19,18,18,18,18,18,17,15,15,14,13,13,13,12,11,10,
    9,9,8,8,8,8,8,8,7,7,6,6,6,5,5,5,5,4,4,4,4,4,4,4,4,4,3,3,2,1,1,
    1,1
  };
  const int n3c3w1_l[] = {
    150, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,100,99,97,97,96,95,95,95,94,92,91,91,91,91,90,90,89,89,
    89,88,88,87,87,87,86,86,86,85,85,85,85,85,84,84,83,83,81,81,81,
    80,80,80,79,79,79,78,78,77,77,77,77,76,75,74,74,74,72,72,71,71,
    70,69,68,68,67,65,64,64,63,63,63,62,62,62,62,61,61,60,60,60,60,
    60,60,59,59,59,59,58,58,57,56,55,55,55,55,54,53,53,52,52,52,51,
    51,51,51,50,50,49,49,48,45,45,43,42,42,41,40,40,39,39,38,38,37,
    36,36,35,35,34,34,34,33,33,32,31,31,31,31,30,29,29,29,29,29,28,
    28,28,27,26,26,25,25,24,24,24,22,22,21,20,19,19,19,19,18,18,18,
    15,15,15,14,14,13,13,12,12,11,10,10,9,9,8,8,8,7,7,7,6,6,6,5,5,
    5,4,3,3,2,1,1,1
  };
  const int n3c3w1_m[] = {
    150, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,99,99,99,98,97,97,96,96,95,94,93,93,93,92,92,92,92,92,92,
    91,91,91,91,90,90,89,89,89,89,86,86,86,85,85,84,83,83,83,82,82,
    82,81,81,80,80,80,79,78,77,77,77,77,76,76,76,76,75,75,73,72,72,
    71,70,70,70,70,68,68,68,68,68,67,65,65,64,64,62,62,61,60,60,59,
    59,59,59,59,58,58,57,57,56,56,56,56,55,54,53,53,53,53,52,52,52,
    51,51,51,50,50,50,50,49,49,49,49,49,49,48,48,48,47,46,46,46,45,
    44,43,42,42,42,41,39,37,37,36,36,35,35,35,34,34,33,33,32,32,31,
    31,31,30,29,29,29,29,28,28,27,26,25,25,25,25,24,23,23,23,23,23,
    22,22,22,21,18,18,18,17,16,16,16,15,14,14,13,13,12,11,11,11,11,
    9,8,8,5,4,4,3,2,2,2,1,1
  };
  const int n3c3w1_n[] = {
    150, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,99,99,98,98,97,97,96,95,95,95,95,94,94,93,92,92,92,92,91,
    90,88,87,87,87,87,87,87,87,86,86,85,85,84,84,84,82,82,82,82,81,
    81,81,81,80,80,80,80,79,79,78,78,77,76,75,75,75,75,73,72,72,71,
    71,71,70,70,70,69,69,68,67,66,66,66,65,64,63,62,62,62,61,61,61,
    60,59,59,57,57,56,56,55,55,53,53,52,51,51,51,51,50,50,49,49,49,
    49,48,47,47,47,47,47,47,47,46,46,46,45,45,45,45,45,43,43,43,43,
    42,41,40,38,38,38,38,36,36,36,35,35,34,34,33,33,32,32,31,30,30,
    28,28,28,27,27,27,26,26,25,25,22,21,20,19,19,18,17,17,17,17,16,
    14,14,14,13,13,13,12,12,11,11,11,10,10,9,8,7,6,6,4,4,4,4,4,4,
    3,3,3,3,3,1,1,1,1
  };
  const int n3c3w1_o[] = {
    150, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,99,98,98,97,97,96,96,96,95,95,94,92,92,91,91,91,91,91,
    91,90,90,90,89,89,88,88,87,87,86,85,82,81,81,81,81,80,80,80,80,
    79,79,78,78,78,78,77,77,77,77,76,75,74,74,74,74,74,73,73,73,73,
    73,71,70,70,70,69,69,69,69,68,68,67,66,64,64,64,63,61,59,58,58,
    57,57,55,54,54,52,52,52,52,52,51,50,50,48,48,47,47,47,46,45,45,
    45,44,43,43,43,42,41,40,40,39,39,38,38,38,38,36,36,34,34,34,33,
    33,32,32,32,32,31,31,31,30,30,30,28,28,26,26,26,26,26,26,25,25,
    25,25,24,24,23,23,23,20,20,20,20,20,18,17,16,16,16,16,15,15,14,
    13,13,12,12,12,11,11,11,10,10,10,9,9,8,8,6,5,5,4,4,4,4,4,3,3,
    3,2,2,2,1,1,1,1
  };
  const int n3c3w1_p[] = {
    150, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,99,99,98,98,97,97,96,96,96,95,95,94,94,94,
    94,93,92,91,91,90,90,90,90,90,90,89,89,88,87,85,85,85,83,83,83,
    82,82,82,81,81,81,80,80,79,79,79,78,78,77,77,77,76,76,76,75,75,
    75,73,73,72,72,72,71,71,70,70,70,69,68,67,67,67,67,67,66,66,65,
    65,64,64,64,63,62,62,61,61,61,61,60,60,60,58,58,58,56,55,54,54,
    53,53,53,53,51,51,49,49,49,48,48,48,47,46,46,45,44,44,42,42,42,
    42,42,41,41,41,41,41,40,40,39,38,38,37,36,36,34,34,34,34,33,32,
    32,32,31,31,31,29,29,28,27,26,26,25,25,24,23,22,21,21,21,21,20,
    19,19,18,17,17,16,16,15,15,14,13,13,13,12,11,11,11,10,10,9,9,
    8,8,8,7,7,6,5,5,4,3,3,2,1
  };
  const int n3c3w1_q[] = {
    150, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,98,98,97,97,97,97,97,96,96,96,96,94,94,94,93,93,92,91,91,
    90,90,90,89,89,89,88,87,87,86,86,85,85,83,83,83,83,82,82,82,81,
    80,79,79,78,78,78,78,77,77,77,77,77,77,76,75,74,74,73,72,72,72,
    71,70,70,69,69,69,67,67,66,66,66,66,66,66,66,66,64,63,62,62,62,
    61,61,61,60,60,60,59,59,59,58,58,57,56,56,56,55,54,54,54,54,54,
    54,54,53,53,53,53,53,51,51,51,50,50,50,50,49,49,48,47,46,46,45,
    45,45,44,44,44,43,43,42,41,41,40,40,40,39,39,39,38,38,37,37,37,
    36,36,36,36,36,34,34,34,34,33,30,29,29,28,28,27,27,27,25,25,25,
    25,24,24,23,22,22,22,22,19,18,18,16,16,15,14,13,13,13,11,11,10,
    10,8,7,5,5,5,4,4,2,1,1,1
  };
  const int n3c3w1_r[] = {
    150, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,99,99,99,99,99,98,97,97,97,96,96,96,94,94,94,94,93,92,
    91,91,91,90,90,90,89,88,88,87,87,86,86,86,86,86,85,84,82,81,81,
    78,78,78,77,77,77,76,76,74,74,74,73,72,72,71,70,69,69,69,68,68,
    68,68,68,67,66,66,66,65,64,64,64,64,63,61,60,60,59,58,57,57,55,
    55,55,54,54,52,52,52,51,51,50,49,48,48,47,47,47,46,46,46,46,43,
    43,43,43,43,42,42,42,41,41,41,40,40,40,40,40,39,39,39,39,38,38,
    38,37,37,37,37,36,36,35,34,33,33,32,31,31,31,31,30,29,29,29,28,
    28,28,25,25,23,23,22,22,22,20,20,20,19,19,19,17,17,16,16,16,15,
    14,13,13,12,12,11,10,10,9,9,9,9,8,8,8,8,8,7,7,6,6,6,6,5,5,5,4,
    4,3,2,2,1,1
  };
  const int n3c3w1_s[] = {
    150, // Capacity
    200, // Number of items
    // Size of items (sorted)
    99,99,97,96,96,95,95,95,95,94,94,94,93,93,93,93,93,92,92,91,91,
    90,90,90,89,89,89,87,86,86,86,86,85,84,84,84,84,83,83,83,78,78,
    75,75,75,75,74,74,71,71,70,70,70,70,69,69,69,69,69,69,68,67,67,
    67,67,67,65,65,65,64,64,63,62,62,62,61,61,60,59,59,59,59,58,57,
    57,57,57,56,56,56,55,55,54,54,54,54,54,54,54,53,53,51,50,49,49,
    49,49,49,48,47,47,47,44,43,42,41,40,40,40,40,39,39,38,38,38,38,
    38,37,37,36,36,35,35,33,33,33,33,32,32,32,31,31,30,30,30,30,29,
    29,28,28,28,28,27,27,27,27,26,26,25,25,25,24,24,24,24,23,23,22,
    20,17,17,17,17,16,16,16,14,13,12,12,11,11,10,9,9,8,7,7,6,6,6,
    5,4,4,2,2,2,2,1,1
  };
  const int n3c3w1_t[] = {
    150, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,99,98,98,98,98,98,98,97,97,97,96,95,94,94,94,94,94,92,91,
    91,91,90,89,88,88,88,87,87,86,86,86,86,85,85,85,84,84,83,83,83,
    82,82,80,80,80,80,80,79,79,78,77,77,76,75,74,74,73,73,72,71,71,
    70,69,69,69,68,68,67,67,67,67,66,66,66,65,63,63,63,62,61,61,61,
    61,61,60,59,59,58,57,57,56,56,56,56,55,55,53,53,52,52,50,50,49,
    49,47,47,47,46,46,46,46,45,44,44,43,42,42,42,41,41,41,41,40,40,
    40,39,39,37,37,37,37,37,36,36,35,35,35,35,34,33,33,33,32,32,31,
    31,30,30,29,27,25,25,23,23,22,22,22,21,21,20,20,19,19,19,19,19,
    18,18,18,17,17,16,16,14,14,14,13,12,12,11,10,10,9,9,8,7,7,6,5,
    5,5,4,4,4,2,2,2,1,1
  };
  const int n3c3w2_a[] = {
    150, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,100,100,99,99,99,98,98,98,97,97,97,97,97,97,96,96,96,
    95,94,94,93,93,93,93,93,92,92,91,91,90,89,89,88,88,88,87,87,87,
    86,86,86,85,85,85,84,84,84,83,82,81,81,80,80,79,79,79,79,79,78,
    76,76,76,76,75,75,75,75,75,75,74,73,73,73,73,72,72,72,72,72,71,
    71,70,70,70,70,69,68,68,68,67,67,65,65,65,64,64,64,64,63,63,63,
    63,62,62,62,62,61,60,60,59,59,59,58,58,58,58,56,56,56,56,56,56,
    56,56,55,53,52,52,51,51,50,50,50,49,49,49,48,48,47,47,46,46,45,
    45,44,44,44,43,43,43,42,42,42,41,41,40,40,39,37,37,37,37,36,36,
    35,35,35,34,34,31,30,29,29,29,29,29,28,28,28,28,27,27,26,26,25,
    25,25,24,24,23,22,21,21,21,21,21,20,20
  };
  const int n3c3w2_b[] = {
    150, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,100,100,99,99,99,99,98,98,97,97,95,95,95,94,93,92,92,
    91,91,90,90,89,89,89,89,89,89,88,87,87,86,86,86,86,85,84,83,83,
    82,82,82,81,81,81,81,81,80,80,80,79,79,79,78,77,77,76,76,75,74,
    74,73,73,73,73,73,72,72,70,70,70,70,70,69,68,68,68,68,68,67,66,
    66,66,66,66,66,65,65,65,65,65,64,64,64,64,63,63,62,62,61,59,59,
    59,59,58,58,56,56,56,56,56,55,55,55,55,54,54,54,54,54,54,53,53,
    53,53,53,52,51,51,51,50,49,49,49,49,48,48,48,47,47,47,46,46,46,
    46,46,45,45,44,44,44,42,42,42,41,39,38,38,38,37,37,36,36,36,36,
    35,34,34,33,33,32,32,32,31,31,31,30,30,29,29,29,29,28,28,27,26,
    25,23,23,23,22,22,22,22,22,21,21,21,21
  };
  const int n3c3w2_c[] = {
    150, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,100,99,98,98,97,96,96,96,96,96,96,95,95,94,94,94,94,93,
    93,93,93,93,93,92,92,92,90,89,89,89,89,87,87,86,86,86,86,85,85,
    84,84,84,84,83,83,83,83,83,81,81,81,80,80,79,79,79,79,78,78,77,
    77,77,76,76,76,74,74,74,74,73,73,73,73,73,72,70,70,69,69,69,69,
    68,67,66,66,66,66,65,65,65,64,64,63,62,62,61,61,60,60,60,58,58,
    57,57,57,57,56,56,55,55,55,55,55,54,54,54,54,54,53,53,53,53,52,
    51,51,51,50,50,50,50,50,49,49,48,48,46,46,45,44,44,44,43,43,43,
    40,40,40,40,40,39,39,38,38,37,37,37,37,37,36,35,35,34,34,33,33,
    33,33,32,32,32,32,31,31,30,29,29,29,29,29,28,28,27,27,27,27,26,
    26,26,25,24,23,22,22,22,21,21,21,20
  };
  const int n3c3w2_d[] = {
    150, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,99,99,98,98,98,96,95,95,94,94,94,93,93,92,92,89,89,89,89,
    88,88,88,88,87,87,87,87,86,86,86,85,84,84,83,83,83,83,83,82,81,
    80,80,80,79,79,79,78,78,77,77,77,77,77,77,75,74,74,74,73,73,72,
    72,71,71,71,71,71,71,70,69,68,68,67,66,66,66,65,65,65,65,65,64,
    64,64,64,62,62,62,62,61,61,61,60,60,60,59,59,59,59,58,58,58,58,
    57,57,57,57,56,56,56,55,54,54,54,54,54,53,53,53,53,52,51,50,50,
    50,49,48,48,48,48,48,48,47,47,45,45,45,44,44,43,43,43,43,43,42,
    42,41,41,41,40,40,40,40,40,39,39,38,38,38,37,37,36,36,36,35,35,
    34,34,33,33,32,32,31,31,31,30,29,29,28,27,26,25,25,25,24,24,24,
    24,24,23,22,22,22,21,21,21,20,20,20
  };
  const int n3c3w2_e[] = {
    150, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,99,97,97,96,96,96,95,95,95,95,94,94,93,93,93,93,92,92,91,
    90,90,90,90,90,90,90,90,89,89,88,88,88,87,86,86,86,84,84,84,84,
    83,83,81,81,80,80,80,78,78,78,77,77,77,76,75,75,75,74,73,73,73,
    72,71,71,71,70,70,70,69,69,69,68,67,67,67,66,66,65,64,64,63,63,
    63,62,62,62,62,62,62,61,61,61,60,60,60,59,59,59,58,58,58,58,57,
    57,57,56,55,55,55,55,53,53,53,52,51,51,51,51,50,50,50,49,49,49,
    49,48,47,46,46,45,45,45,44,44,44,44,43,43,43,43,43,42,41,41,41,
    40,40,40,40,40,39,39,39,39,39,38,37,37,36,36,35,34,34,34,34,33,
    33,32,32,32,31,31,31,31,30,30,30,29,28,27,27,26,25,25,25,24,24,
    24,23,23,23,22,22,22,22,21,21,21,20
  };
  const int n3c3w2_f[] = {
    150, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,100,100,99,99,98,98,97,97,97,96,95,95,95,95,95,94,94,
    94,94,93,93,93,93,92,90,89,89,89,89,88,88,88,87,87,87,86,85,85,
    85,84,84,84,83,83,82,82,82,82,82,81,81,80,80,80,79,79,79,79,78,
    78,78,76,75,75,74,74,74,73,72,72,72,72,72,72,71,70,70,70,69,68,
    68,68,66,65,65,64,64,64,62,61,61,60,59,59,58,58,57,57,57,56,56,
    55,55,55,55,54,54,54,53,53,52,52,52,52,51,51,51,50,50,50,50,50,
    49,49,48,48,47,47,46,46,46,46,45,45,44,44,44,44,44,44,44,43,43,
    43,43,43,43,43,42,42,42,41,41,41,41,40,40,39,39,38,38,38,37,37,
    36,36,35,35,35,35,34,34,34,33,31,31,31,30,30,30,30,30,29,28,27,
    26,26,25,25,24,24,22,22,21,20,20,20,20
  };
  const int n3c3w2_g[] = {
    150, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,100,99,99,98,98,98,97,97,96,96,95,94,93,93,
    93,92,91,90,90,90,89,89,88,88,88,88,88,87,87,87,87,86,86,85,85,
    85,84,84,84,84,84,83,83,83,82,81,81,80,80,79,78,77,77,77,77,76,
    76,75,75,75,75,74,74,74,73,73,73,73,72,71,70,70,70,70,69,68,68,
    68,68,68,67,67,67,67,66,66,65,65,65,64,63,63,63,63,63,63,62,62,
    62,60,60,59,59,59,58,57,56,55,55,54,53,53,52,51,50,50,50,50,49,
    48,48,48,48,48,47,47,47,47,46,46,45,44,44,43,43,43,43,43,43,42,
    42,41,41,39,39,38,38,37,37,37,36,36,36,35,34,34,34,34,33,33,32,
    31,31,31,31,30,30,30,30,30,29,28,27,27,26,26,26,25,25,25,25,25,
    25,24,24,24,23,23,22,21,21,21,20,20,20
  };
  const int n3c3w2_h[] = {
    150, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,99,99,99,99,99,98,98,97,97,97,96,94,94,94,
    94,94,94,94,93,93,91,91,91,90,89,89,89,88,88,87,87,87,86,86,86,
    86,86,86,86,85,85,85,85,84,84,83,83,82,82,81,81,81,80,80,79,79,
    78,78,77,77,76,75,75,75,74,74,74,74,74,73,73,72,71,71,70,69,68,
    68,67,67,66,66,66,66,65,65,65,65,65,64,63,63,63,63,63,61,61,61,
    60,60,60,60,59,59,58,58,58,57,57,56,56,56,55,54,54,53,53,52,52,
    52,51,50,50,48,48,47,46,46,44,44,44,44,44,43,43,43,43,42,41,41,
    41,40,40,40,40,39,39,39,39,38,38,38,38,38,38,38,37,37,36,36,36,
    35,35,34,34,33,32,32,32,32,31,31,30,30,30,29,28,27,27,26,26,26,
    26,25,25,25,24,23,22,22,22,21,21,20,20
  };
  const int n3c3w2_i[] = {
    150, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,99,99,99,99,99,99,98,98,98,96,96,96,95,95,95,95,95,95,95,
    95,94,94,92,92,92,92,92,92,92,92,92,91,89,89,87,87,86,86,86,85,
    85,85,84,84,84,83,83,83,82,82,81,81,81,81,79,79,79,79,77,76,75,
    75,74,74,73,72,70,69,69,69,69,69,69,69,69,68,67,67,64,64,64,64,
    64,64,63,63,63,63,63,62,62,62,62,61,59,58,58,57,57,56,55,55,54,
    54,52,52,52,52,52,51,51,50,50,50,48,47,46,46,45,45,45,45,45,45,
    45,44,44,44,44,43,42,42,41,41,41,41,41,41,40,40,39,39,38,38,38,
    37,37,36,36,36,35,35,35,35,35,35,35,35,34,34,34,34,33,33,32,31,
    31,31,31,31,30,30,30,29,29,28,28,28,28,28,27,26,26,26,26,25,24,
    24,23,23,23,22,22,22,22,21,21,20,20
  };
  const int n3c3w2_j[] = {
    150, // Capacity
    200, // Number of items
    // Size of items (sorted)
    99,99,99,99,99,99,98,98,98,97,97,97,97,96,96,96,95,95,95,95,95,
    95,94,94,94,93,93,92,92,92,92,92,91,91,90,90,87,87,87,87,87,86,
    86,85,84,84,84,83,83,83,83,82,82,82,82,82,82,81,80,80,79,78,78,
    77,76,76,75,75,74,74,73,73,72,72,72,71,71,71,70,70,69,69,69,68,
    68,68,68,68,67,67,66,66,66,65,65,65,64,64,64,64,63,63,61,60,59,
    59,59,59,58,58,57,57,57,57,56,56,55,55,54,54,54,54,54,53,52,52,
    52,52,50,50,49,49,49,48,48,48,48,48,48,47,47,47,47,46,45,44,44,
    43,43,43,43,43,42,41,41,40,40,40,40,40,39,38,37,36,36,35,34,34,
    33,33,32,32,31,30,30,29,28,28,28,28,28,27,26,26,25,24,23,23,23,
    23,23,22,22,22,21,21,21,21,21,20
  };
  const int n3c3w2_k[] = {
    150, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,99,99,98,98,98,98,97,97,96,96,96,95,95,94,
    94,93,93,93,92,91,91,91,91,91,90,89,89,89,89,89,88,88,88,88,88,
    87,87,86,86,86,86,85,85,85,84,84,84,83,83,83,82,82,82,82,82,81,
    81,80,80,80,80,79,79,79,79,79,79,78,75,75,75,74,74,73,73,73,73,
    73,71,71,70,70,68,68,67,67,67,67,67,66,65,65,65,65,64,64,63,62,
    62,62,62,61,61,60,59,58,58,57,56,56,55,54,54,53,52,52,52,52,52,
    51,51,51,51,51,51,51,48,48,47,47,46,46,46,46,46,45,45,44,43,43,
    43,43,43,42,42,41,39,39,39,38,36,34,34,33,33,33,33,33,32,32,31,
    31,31,30,30,30,29,29,29,29,28,28,28,28,28,27,27,26,26,26,26,26,
    25,25,25,25,24,24,22,22,21,21,21,21,20
  };
  const int n3c3w2_l[] = {
    150, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,99,99,99,99,99,98,98,98,98,97,97,97,97,97,96,96,96,95,
    95,94,94,94,93,93,92,91,91,90,90,89,89,89,89,89,88,87,85,85,85,
    85,85,84,83,83,83,82,82,81,81,80,80,80,80,79,79,79,79,78,78,76,
    75,75,74,74,74,74,74,73,73,73,72,71,70,70,69,69,69,69,68,67,67,
    67,67,66,66,66,65,64,64,64,63,63,63,63,62,62,61,61,60,60,60,60,
    60,60,58,58,57,56,56,56,56,56,56,55,55,55,54,54,53,51,51,51,51,
    51,50,50,50,49,48,48,47,46,46,46,45,45,45,45,45,44,44,43,42,41,
    41,41,40,40,40,39,39,39,39,38,38,37,37,37,37,36,35,35,35,34,34,
    34,33,33,32,30,30,30,30,30,29,29,28,28,28,27,26,26,26,25,25,25,
    25,24,24,24,24,23,23,23,23,23,22,21
  };
  const int n3c3w2_m[] = {
    150, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,100,99,99,99,99,98,98,97,97,97,96,96,96,96,96,96,95,95,
    94,93,93,93,93,92,92,92,91,91,91,91,91,91,91,90,89,89,89,88,86,
    86,86,85,85,85,85,84,84,83,83,82,82,82,82,80,80,80,80,80,79,79,
    79,78,77,77,77,74,74,73,73,73,73,73,73,72,71,71,70,70,69,69,69,
    69,69,68,68,68,67,66,65,65,65,64,64,64,63,62,61,61,61,61,61,60,
    60,60,59,58,57,57,57,57,56,56,56,56,56,55,55,55,54,54,54,54,54,
    53,53,52,52,52,51,50,50,50,50,49,49,49,48,47,47,46,46,45,45,45,
    44,44,44,44,44,43,42,42,41,38,38,38,38,38,37,37,37,35,35,35,35,
    35,33,32,32,32,32,31,31,31,31,30,30,29,29,29,29,28,27,26,26,25,
    25,25,25,25,25,24,24,23,23,21,20,20
  };
  const int n3c3w2_n[] = {
    150, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,100,99,98,98,97,97,97,96,94,94,93,93,92,91,90,90,89,89,
    89,89,89,88,88,88,87,87,87,87,86,86,86,86,85,85,83,83,83,82,82,
    82,82,81,80,80,80,80,78,77,77,76,76,74,73,73,73,73,72,72,72,71,
    71,71,70,70,70,69,69,69,68,68,68,68,67,67,66,66,66,65,65,65,65,
    64,64,64,64,63,62,60,59,58,58,58,57,57,57,57,57,57,56,55,55,53,
    52,52,52,51,50,50,49,48,48,48,48,48,48,48,47,46,46,46,46,45,45,
    45,45,44,44,44,44,43,43,43,42,42,42,42,41,40,40,39,39,39,39,38,
    38,38,38,38,38,36,36,35,34,34,33,33,33,33,33,33,32,32,32,32,32,
    31,31,31,31,31,30,30,30,30,29,28,27,27,27,26,26,25,25,25,24,24,
    23,23,23,22,22,21,21,20,20,20,20,20
  };
  const int n3c3w2_o[] = {
    150, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,100,100,99,98,98,97,97,97,97,97,97,96,96,95,94,93,93,
    92,91,91,91,90,90,90,90,89,89,89,89,88,88,88,88,87,87,86,86,86,
    85,85,85,85,85,84,84,84,84,83,82,82,82,82,82,81,81,81,81,80,79,
    79,79,79,78,78,78,78,77,76,76,75,75,74,74,73,71,71,70,70,70,70,
    69,69,68,68,68,67,67,67,66,65,65,65,65,63,63,62,61,61,61,61,59,
    59,59,59,59,58,58,58,57,57,57,56,56,56,55,55,55,54,54,54,54,53,
    53,53,53,53,52,52,51,51,50,50,50,49,48,47,46,45,45,44,43,42,42,
    42,41,41,41,41,40,40,39,39,38,37,36,36,35,34,34,34,34,34,34,33,
    33,32,31,31,30,30,29,29,29,29,29,28,28,27,26,25,25,25,24,24,24,
    23,23,22,22,22,21,21,21,20,20,20,20,20
  };
  const int n3c3w2_p[] = {
    150, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,99,99,99,99,99,98,98,98,98,96,96,96,96,95,95,94,93,93,92,
    92,92,92,91,91,91,91,90,90,90,89,89,87,87,87,86,85,84,84,84,83,
    82,82,82,81,81,80,80,79,79,79,78,78,78,76,76,76,76,75,75,75,73,
    73,73,72,72,71,71,71,71,70,70,70,69,69,68,68,68,68,67,67,67,67,
    67,67,67,66,66,66,65,65,64,64,64,63,63,63,62,62,62,62,61,61,60,
    59,59,59,58,57,57,56,55,55,55,55,55,53,52,52,51,51,51,51,51,50,
    50,50,50,49,49,49,48,47,47,46,46,45,44,44,44,44,43,43,41,41,41,
    40,40,38,38,37,37,37,37,36,36,36,36,36,35,34,34,34,34,33,33,33,
    32,32,32,31,31,31,30,30,29,27,27,27,27,26,26,25,25,25,25,25,24,
    24,24,23,23,23,22,22,22,20,20,20,20
  };
  const int n3c3w2_q[] = {
    150, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,99,99,99,98,98,98,98,98,97,97,96,96,95,94,94,94,93,93,93,
    92,92,91,91,91,91,90,90,89,88,88,88,87,87,87,86,86,86,85,85,84,
    84,83,82,80,80,80,79,79,79,79,78,78,77,77,77,76,74,74,73,73,73,
    72,71,71,71,70,70,70,70,68,68,68,67,67,67,67,66,66,65,64,64,63,
    63,61,61,60,60,60,60,59,59,58,58,58,58,57,57,57,56,56,55,54,51,
    51,50,49,48,48,48,47,45,45,45,44,44,44,44,43,43,43,43,43,43,42,
    42,42,42,41,41,40,39,39,39,39,38,38,38,38,38,38,38,38,37,37,37,
    36,36,35,35,35,35,34,34,34,34,34,33,33,33,33,32,32,31,31,31,30,
    30,29,28,28,28,27,25,25,24,24,24,24,24,23,23,23,23,23,22,22,22,
    22,22,21,21,21,21,21,21,21,20,20,20
  };
  const int n3c3w2_r[] = {
    150, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,99,99,99,97,96,96,96,95,95,95,95,95,94,94,94,94,93,93,
    93,92,92,91,90,89,89,89,88,88,87,87,87,87,86,85,85,84,84,83,83,
    83,82,82,81,81,81,80,80,80,80,80,79,78,78,77,77,76,76,75,74,74,
    73,73,73,72,71,71,71,70,70,70,69,68,68,68,67,67,67,66,65,65,65,
    64,64,63,62,62,62,61,61,61,60,60,60,59,58,58,58,58,58,58,57,57,
    57,57,56,56,55,54,53,53,53,53,52,52,52,51,51,50,50,50,49,49,49,
    48,46,46,46,46,46,46,44,43,43,43,42,42,42,41,41,40,40,40,39,39,
    39,38,38,38,37,37,37,36,36,36,36,35,35,35,35,33,33,33,33,33,32,
    32,32,32,32,31,31,30,30,29,29,29,29,29,29,29,29,28,28,28,28,27,
    26,26,26,25,24,24,24,23,22,21,21,21
  };
  const int n3c3w2_s[] = {
    150, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,98,98,98,98,97,97,97,96,96,95,95,95,94,94,94,93,92,91,91,
    91,90,89,89,88,88,87,87,87,87,87,86,86,86,86,85,85,85,84,84,84,
    83,83,82,81,80,80,80,80,80,79,78,78,78,78,77,77,77,77,77,77,77,
    76,76,76,74,74,74,74,74,73,73,73,72,71,71,71,69,69,69,69,69,68,
    68,67,67,67,66,66,66,65,65,65,65,64,64,64,62,62,62,62,62,61,61,
    61,61,59,59,59,57,57,57,56,55,55,54,52,52,52,51,51,50,50,50,50,
    49,49,48,48,47,46,46,45,45,45,44,44,44,43,42,41,41,41,40,39,39,
    38,37,37,37,37,37,36,36,35,35,35,34,34,34,33,33,33,32,31,31,31,
    31,30,30,30,29,29,29,28,28,28,28,27,27,27,27,26,26,25,25,24,24,
    24,23,23,23,22,22,22,22,21,21,20,20
  };
  const int n3c3w2_t[] = {
    150, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,99,99,99,99,99,98,97,97,96,95,95,95,94,94,94,93,92,92,
    92,91,91,90,90,90,88,88,87,85,85,84,84,84,84,84,84,84,84,84,83,
    83,82,82,82,82,82,82,81,81,80,80,79,79,78,78,78,78,78,78,77,77,
    77,76,76,75,74,74,74,74,73,73,72,71,70,69,69,69,67,67,66,65,64,
    64,62,62,62,61,61,61,60,60,60,60,59,59,58,57,57,56,56,56,56,56,
    56,55,55,55,55,54,53,53,53,53,52,52,51,51,49,49,49,49,49,49,49,
    48,47,47,47,46,46,45,44,44,44,44,43,43,42,42,42,42,41,39,39,38,
    37,37,37,36,36,36,36,35,35,33,33,33,33,33,32,32,32,31,31,31,31,
    30,30,30,30,30,30,29,29,29,29,28,28,28,28,26,25,25,25,24,24,24,
    23,23,23,23,23,22,22,21,21,21,21,20
  };
  const int n3c3w4_a[] = {
    150, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,100,99,99,99,99,98,98,98,98,98,98,97,97,96,
    96,96,96,96,95,95,95,94,94,93,93,93,92,92,92,91,90,90,89,89,89,
    89,89,89,89,89,89,88,88,87,86,86,86,85,85,85,85,84,84,83,83,82,
    82,82,81,80,80,80,80,79,79,78,78,78,78,77,76,76,76,75,74,73,73,
    73,73,73,72,72,72,71,68,68,68,68,68,67,66,66,65,65,65,65,65,65,
    64,64,63,63,62,62,62,62,60,59,59,59,58,58,58,56,56,56,55,55,55,
    54,54,54,54,53,53,53,52,52,52,51,51,51,51,51,50,50,50,50,50,49,
    49,49,49,48,48,48,48,47,46,46,45,45,45,45,44,43,43,43,43,42,42,
    41,41,41,40,40,40,39,39,39,39,39,38,38,38,37,37,37,36,35,35,34,
    34,34,34,33,33,33,33,32,32,31,30,30,30
  };
  const int n3c3w4_b[] = {
    150, // Capacity
    200, // Number of items
    // Size of items (sorted)
    99,99,98,98,97,97,97,96,96,96,96,95,95,95,94,94,93,93,92,92,91,
    91,91,91,91,90,89,89,89,88,88,87,87,87,86,86,86,86,86,86,86,84,
    84,83,82,82,82,82,81,81,81,81,80,80,80,79,79,79,79,78,78,78,78,
    77,77,77,77,77,76,76,75,75,75,75,74,74,74,73,72,72,72,72,72,72,
    72,71,71,70,70,70,69,69,69,69,69,68,68,68,68,67,67,67,67,67,67,
    65,65,64,63,63,62,62,62,62,62,61,61,61,60,60,59,58,57,57,56,55,
    55,55,55,53,53,52,52,52,52,51,51,51,51,50,50,50,49,49,49,48,48,
    48,48,47,47,46,45,45,45,44,44,44,44,44,43,43,43,43,42,42,42,42,
    42,42,41,40,40,39,38,38,38,37,37,36,36,36,36,36,35,35,35,34,34,
    33,33,33,32,32,32,31,31,31,31,30
  };
  const int n3c3w4_c[] = {
    150, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,99,98,98,98,97,97,97,97,97,97,96,96,96,96,96,95,95,95,95,
    95,95,94,94,94,94,94,94,93,93,92,92,92,92,91,91,90,89,89,89,89,
    88,88,88,88,87,87,87,87,86,85,84,84,83,83,83,83,82,82,82,82,81,
    80,79,79,79,79,77,77,77,76,76,74,74,74,73,73,73,73,72,72,72,71,
    71,71,71,71,71,71,70,69,69,69,69,68,68,67,67,66,65,65,64,63,63,
    63,63,62,62,62,62,60,60,59,59,59,59,59,58,58,58,58,58,58,57,57,
    56,56,56,56,55,55,54,53,53,53,52,52,52,52,51,51,50,50,50,49,49,
    48,48,48,48,47,47,46,46,46,46,46,45,45,44,43,43,43,43,42,41,41,
    39,38,38,38,38,38,37,37,37,37,37,37,36,36,36,36,35,35,35,35,34,
    34,34,34,34,33,33,33,32,32,31,31,30
  };
  const int n3c3w4_d[] = {
    150, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,100,99,98,98,98,97,96,96,96,96,95,95,95,94,
    94,94,94,94,93,92,92,92,92,91,91,91,90,90,90,90,88,87,87,86,86,
    86,86,85,85,85,83,83,82,82,82,82,81,81,81,80,80,79,79,79,79,79,
    78,78,78,78,78,78,77,76,75,75,75,75,75,75,74,74,73,73,73,73,72,
    72,72,71,70,70,69,68,68,68,67,66,65,65,65,65,64,64,63,63,63,63,
    63,62,61,61,60,60,60,59,59,59,59,58,58,56,56,56,56,56,56,55,55,
    55,55,55,54,54,54,53,53,53,52,52,52,51,51,51,51,50,50,50,49,48,
    48,48,48,48,48,48,48,47,47,47,47,47,46,46,46,45,45,45,45,44,43,
    43,43,42,42,42,41,40,38,37,37,37,37,36,36,36,36,35,34,34,34,33,
    33,33,33,33,32,32,32,32,32,32,30,30,30
  };
  const int n3c3w4_e[] = {
    150, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,99,99,98,98,97,96,96,95,94,94,93,93,93,93,93,92,92,91,
    90,90,90,90,89,89,89,88,88,88,88,87,87,87,87,86,86,85,85,85,84,
    84,83,83,83,82,81,81,80,80,80,79,79,78,78,78,77,77,77,77,76,76,
    75,75,75,75,74,74,74,74,73,73,73,72,71,71,71,71,70,70,69,68,68,
    68,68,68,68,68,67,67,67,66,66,66,65,64,64,64,64,63,63,63,63,62,
    62,61,61,61,60,60,58,58,58,58,58,57,57,56,56,56,56,56,56,55,55,
    55,54,54,54,53,53,52,52,52,52,51,51,51,50,50,50,49,49,49,48,48,
    47,47,47,47,46,46,46,46,46,45,44,44,44,44,44,43,43,42,42,42,42,
    41,41,41,39,39,39,39,39,39,38,38,37,37,37,37,36,35,35,34,34,34,
    34,34,33,33,33,33,32,32,31,30,30,30
  };
  const int n3c3w4_f[] = {
    150, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,99,99,99,98,98,98,98,98,97,97,97,97,96,96,95,94,94,93,
    93,93,92,92,92,91,90,90,87,87,87,86,86,86,86,85,85,84,83,83,83,
    82,82,81,81,80,80,80,80,80,80,80,79,79,79,79,79,79,78,78,78,76,
    75,75,74,73,73,72,71,71,71,71,71,70,69,69,69,68,68,67,67,67,66,
    66,66,66,66,66,66,66,65,65,65,63,63,63,63,62,62,62,62,61,61,60,
    60,60,60,60,60,58,58,58,58,58,58,57,56,56,56,56,55,55,54,54,54,
    53,53,53,52,52,51,51,51,49,49,49,48,48,48,48,48,48,47,46,46,46,
    46,45,45,44,44,44,43,43,42,42,42,42,41,41,41,40,40,40,40,39,39,
    39,39,39,39,39,38,38,38,38,37,36,36,36,36,36,36,35,35,35,35,34,
    34,33,33,32,31,31,31,31,30,30,30,30
  };
  const int n3c3w4_g[] = {
    150, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,99,99,98,98,98,98,98,98,98,97,97,97,97,97,
    96,95,94,94,94,93,93,92,92,92,91,91,91,91,91,90,90,90,89,89,89,
    89,89,88,88,88,88,88,87,87,87,87,86,86,86,86,85,85,84,84,84,84,
    84,84,83,83,83,83,82,82,81,81,81,80,80,80,80,79,78,77,77,77,76,
    76,76,76,76,76,76,75,75,75,75,74,74,74,74,74,72,72,71,71,71,70,
    70,69,68,68,68,68,68,67,67,66,66,65,65,65,64,63,63,62,62,61,61,
    61,60,60,60,60,60,60,59,59,59,58,58,58,58,57,57,56,56,55,55,55,
    55,54,54,54,54,54,54,52,52,51,50,50,49,49,49,48,47,47,47,47,46,
    46,46,45,44,44,43,43,42,42,40,40,39,38,38,38,38,37,37,36,36,35,
    35,35,35,35,35,34,34,32,31,31,31,31,30
  };
  const int n3c3w4_h[] = {
    150, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,99,99,99,97,97,96,95,95,94,94,94,94,93,92,92,92,92,92,92,
    92,91,91,91,91,90,90,89,89,89,89,88,87,87,86,86,86,85,85,85,84,
    84,84,83,83,83,82,82,82,82,81,81,81,81,79,79,77,77,76,76,76,76,
    75,75,74,74,74,74,73,72,71,71,70,70,68,68,67,67,67,66,66,66,65,
    65,64,63,63,63,62,62,62,62,62,61,61,61,61,60,60,60,60,60,60,60,
    58,58,57,57,57,56,56,56,56,56,55,55,55,55,54,54,53,53,53,53,53,
    52,52,52,52,52,51,51,51,51,51,51,50,50,50,50,49,49,49,48,48,48,
    48,48,47,47,47,47,46,46,45,45,45,44,44,44,43,43,43,42,42,42,41,
    40,40,39,39,39,39,38,38,37,37,37,37,37,36,36,35,35,35,35,35,34,
    34,34,34,33,33,33,32,31,31,30,30,30
  };
  const int n3c3w4_i[] = {
    150, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,100,99,99,97,97,97,96,96,96,96,96,95,95,95,95,94,94,93,
    93,93,93,92,92,92,92,92,91,91,91,90,90,90,90,89,89,89,89,89,88,
    88,88,88,88,88,87,87,86,86,85,85,85,85,85,84,84,84,83,83,83,82,
    81,81,81,80,79,79,79,79,79,79,78,78,78,78,78,77,77,76,76,75,75,
    75,75,74,74,74,73,72,72,72,72,71,71,71,70,70,70,70,69,69,69,69,
    69,68,67,67,67,67,66,66,66,65,65,65,64,63,63,63,63,62,62,62,61,
    61,61,61,60,60,59,59,58,58,58,58,56,56,55,55,55,53,53,52,52,52,
    52,51,51,50,49,48,48,48,48,47,46,46,46,46,45,45,45,44,44,43,43,
    42,42,41,41,40,40,40,40,39,39,38,38,38,38,37,37,37,36,36,36,35,
    35,35,34,34,33,32,32,32,32,31,31,30
  };
  const int n3c3w4_j[] = {
    150, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,99,98,97,97,97,96,96,96,95,95,95,95,94,94,94,94,94,94,
    93,93,93,93,93,93,92,91,91,91,90,90,90,89,89,89,87,87,86,86,85,
    85,85,85,85,84,84,83,83,83,83,82,82,82,82,81,81,81,81,81,81,81,
    80,80,78,78,78,78,77,77,77,76,76,75,75,75,75,74,74,74,74,73,73,
    73,71,71,71,71,70,70,69,69,68,68,67,67,67,66,66,66,65,64,63,63,
    63,62,61,61,61,61,61,61,60,60,60,60,58,58,58,58,57,57,57,57,56,
    56,56,56,56,56,55,54,53,53,53,53,52,52,52,52,51,51,50,50,49,49,
    49,48,48,48,48,48,48,47,47,46,46,46,46,46,44,44,44,43,43,43,42,
    42,42,41,41,39,39,39,38,37,37,37,36,36,36,34,32,32,32,32,32,31,
    31,31,31,31,31,31,31,31,31,30,30,30
  };
  const int n3c3w4_k[] = {
    150, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,100,99,99,99,99,98,98,98,98,97,97,97,96,96,96,96,96,95,
    95,95,94,94,94,92,92,92,92,92,92,91,91,90,90,90,90,90,90,89,89,
    88,88,88,87,87,86,86,85,85,85,84,84,84,84,83,82,82,81,81,79,79,
    78,77,77,77,77,77,76,76,75,75,74,74,74,73,73,73,73,73,73,72,71,
    70,70,70,70,70,69,69,69,69,68,68,67,67,67,66,66,65,65,64,64,63,
    63,63,62,62,62,62,62,60,60,60,60,59,59,59,58,58,58,58,58,58,57,
    57,57,56,56,56,56,55,55,55,54,54,54,53,53,53,53,53,53,52,51,50,
    49,49,49,49,49,48,48,48,47,47,47,47,47,47,46,45,45,45,44,44,43,
    43,43,42,42,41,41,41,41,40,39,39,39,38,38,38,37,37,37,36,36,36,
    35,35,35,34,33,33,33,33,32,31,31,30
  };
  const int n3c3w4_l[] = {
    150, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,99,99,99,98,98,98,97,97,97,97,96,96,96,96,96,95,95,95,
    95,94,94,93,93,92,92,91,91,91,90,90,90,90,89,89,89,88,88,88,87,
    86,86,86,86,85,85,85,84,84,84,84,83,83,83,83,83,82,82,82,82,82,
    81,81,81,81,80,80,80,80,79,79,78,78,77,77,77,76,75,75,74,74,74,
    73,73,73,72,72,71,71,71,71,70,70,69,68,67,65,65,64,64,64,63,63,
    63,62,62,62,62,60,60,60,60,59,59,59,58,58,58,58,57,56,56,56,56,
    55,55,54,54,54,53,53,53,53,53,53,52,52,52,52,52,50,50,50,50,50,
    50,49,49,48,48,48,47,47,46,45,45,45,44,44,44,44,44,43,43,43,43,
    43,42,42,42,42,41,41,40,40,40,39,39,38,37,36,36,36,36,35,35,34,
    34,33,33,32,32,32,31,31,31,30,30,30
  };
  const int n3c3w4_m[] = {
    150, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,100,99,99,98,98,98,98,97,96,95,94,94,94,94,93,93,93,93,
    93,92,92,92,91,90,90,90,90,90,90,89,89,88,88,87,87,86,86,86,86,
    86,85,85,85,85,84,84,83,83,83,82,82,82,82,82,81,81,80,80,79,79,
    79,79,79,79,78,78,78,77,77,76,76,76,76,75,75,75,74,74,74,74,74,
    73,73,73,73,72,72,71,69,69,69,69,68,68,68,67,67,66,65,65,65,63,
    63,63,62,61,61,61,61,60,60,59,59,59,59,58,58,58,58,58,56,56,56,
    55,55,54,54,54,53,53,53,53,53,52,52,52,52,51,51,51,51,51,50,50,
    49,49,49,48,48,47,46,46,46,46,45,45,45,44,44,44,42,42,42,41,41,
    39,39,38,38,38,38,38,37,37,37,37,37,37,37,36,36,36,36,35,35,35,
    34,34,34,33,32,31,30,30,30,30,30,30
  };
  const int n3c3w4_n[] = {
    150, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,99,99,98,98,97,97,97,97,96,95,95,93,93,93,
    93,92,91,91,90,90,89,89,89,88,88,88,87,87,87,86,86,86,86,86,85,
    85,85,84,84,84,84,84,84,83,83,83,82,82,82,81,81,81,80,80,79,79,
    79,78,78,78,78,78,77,77,76,75,75,75,75,75,75,74,74,74,74,74,72,
    71,71,71,71,71,71,70,69,69,69,68,67,66,65,65,65,64,64,63,63,62,
    62,62,61,60,59,59,59,59,58,58,58,57,57,57,57,56,56,56,56,55,54,
    54,53,52,52,51,50,49,49,49,49,48,48,48,48,48,47,47,47,46,46,46,
    46,46,45,45,45,45,44,44,44,44,44,44,43,43,43,42,42,42,41,41,41,
    41,40,40,40,40,40,40,39,39,38,38,37,37,36,36,35,34,34,34,34,34,
    33,33,33,33,33,33,32,32,32,32,31,30,30
  };
  const int n3c3w4_o[] = {
    150, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,99,98,98,98,98,97,97,97,96,96,96,96,96,96,
    95,94,94,93,92,92,92,91,91,91,91,90,90,90,89,89,89,89,89,87,87,
    87,86,86,86,86,86,85,85,85,83,83,82,82,81,81,81,80,80,79,79,78,
    78,78,78,77,77,77,77,76,76,76,75,75,75,75,73,73,73,72,72,71,71,
    70,70,70,69,69,68,68,67,67,67,67,66,65,64,64,64,64,63,63,63,63,
    62,62,61,61,61,61,60,60,60,60,59,59,59,59,59,58,58,58,58,57,57,
    57,57,56,56,55,55,55,55,54,54,53,53,53,51,51,51,50,50,50,50,50,
    49,49,48,47,47,47,47,47,46,45,45,44,44,43,42,42,41,41,41,40,40,
    40,40,39,39,37,37,37,37,37,36,36,36,35,35,35,35,35,34,34,33,33,
    33,33,32,31,31,31,31,31,31,31,30,30,30
  };
  const int n3c3w4_p[] = {
    150, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,100,99,99,97,97,97,96,95,95,95,94,94,94,93,93,93,92,92,
    92,92,92,92,91,91,91,91,90,90,89,88,88,86,85,85,83,83,83,82,82,
    81,81,80,80,80,79,79,79,77,77,77,77,77,77,77,77,77,76,76,76,75,
    75,74,74,74,74,74,74,73,73,72,72,72,71,71,70,70,70,68,68,68,67,
    67,67,67,67,66,66,66,66,66,65,65,65,65,64,64,64,64,63,63,62,62,
    62,62,62,62,61,61,61,60,60,60,60,60,59,59,58,58,58,58,57,57,57,
    56,56,56,55,54,54,54,54,54,53,53,53,53,52,52,51,51,50,50,50,50,
    50,49,49,49,48,48,48,47,47,46,46,46,45,45,45,44,44,44,43,43,42,
    41,41,40,39,38,38,38,38,37,37,37,36,36,35,35,35,34,34,34,34,33,
    33,33,33,33,32,32,31,30,30,30,30,30
  };
  const int n3c3w4_q[] = {
    150, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,99,99,99,99,98,98,98,98,98,96,96,96,95,95,95,95,95,94,
    94,94,92,92,92,91,91,91,90,89,89,88,88,86,86,85,85,85,84,83,83,
    82,82,81,81,81,81,80,80,79,79,79,79,79,79,79,78,78,78,78,78,77,
    77,77,77,77,77,77,76,75,75,75,74,73,73,73,73,72,72,72,71,71,71,
    70,70,70,68,68,67,67,66,66,66,66,66,66,65,65,65,65,65,64,63,63,
    63,63,63,62,62,62,62,62,62,61,61,61,61,61,60,60,59,59,57,56,56,
    56,56,56,55,55,55,54,53,53,52,52,52,51,50,50,50,50,50,49,49,48,
    48,48,47,47,46,46,46,46,45,44,44,44,44,44,43,43,43,42,42,41,41,
    41,41,41,41,41,40,40,40,40,39,38,38,38,38,38,38,37,37,36,36,35,
    35,34,34,33,33,33,33,33,32,32,32,30
  };
  const int n3c3w4_r[] = {
    150, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,99,99,98,98,98,98,98,98,97,97,97,96,95,95,
    94,93,92,92,92,92,91,91,91,91,91,90,90,90,90,90,89,89,88,88,88,
    87,86,85,85,85,85,84,83,83,83,81,80,80,80,79,79,79,79,78,78,78,
    78,78,78,77,77,77,77,76,76,76,76,76,75,75,75,74,73,73,73,73,73,
    73,72,72,71,71,70,69,69,68,67,67,67,67,66,66,65,65,65,64,62,62,
    61,61,61,61,61,61,60,59,59,59,59,59,58,58,58,58,57,57,57,57,57,
    57,56,56,56,55,55,55,54,54,54,54,54,54,53,53,53,52,51,50,50,50,
    49,49,49,48,48,47,47,46,46,45,45,45,44,44,44,43,42,42,42,41,41,
    41,40,40,39,39,39,38,38,37,37,36,36,35,34,33,33,33,33,33,33,32,
    32,32,32,32,31,31,31,31,31,30,30,30,30
  };
  const int n3c3w4_s[] = {
    150, // Capacity
    200, // Number of items
    // Size of items (sorted)
    98,98,98,97,97,97,96,96,96,94,94,94,93,93,93,93,92,90,90,89,88,
    87,87,87,86,86,86,86,86,85,85,85,84,84,83,83,82,82,81,81,80,80,
    80,80,78,78,78,77,77,77,77,77,77,76,76,75,75,75,74,74,74,73,73,
    73,72,72,72,71,71,71,71,71,71,71,71,71,70,69,69,69,68,68,68,68,
    67,67,66,66,66,66,66,66,65,64,64,64,64,63,63,63,63,62,62,62,62,
    61,61,61,60,60,60,59,58,58,58,57,57,56,56,55,55,55,54,54,54,53,
    53,53,53,53,53,52,52,52,52,51,51,50,50,50,50,50,50,49,49,48,48,
    47,47,47,47,47,46,46,45,45,44,43,43,43,42,42,41,41,41,41,40,40,
    39,39,39,38,38,38,37,37,37,37,36,36,36,35,34,33,33,33,33,33,32,
    32,32,32,32,31,31,31,31,30,30,30
  };
  const int n3c3w4_t[] = {
    150, // Capacity
    200, // Number of items
    // Size of items (sorted)
    100,100,99,99,99,98,98,98,98,98,97,97,96,96,96,96,94,93,93,92,
    92,90,90,89,89,89,88,88,88,88,88,88,87,87,87,87,86,86,85,85,84,
    83,82,82,81,81,80,80,80,80,80,80,79,79,79,78,78,77,77,76,76,76,
    75,75,75,75,75,74,74,74,74,73,72,72,72,71,71,71,71,71,70,70,69,
    69,69,69,68,67,66,66,66,65,65,65,64,62,61,61,61,61,61,61,60,60,
    60,59,59,59,59,58,58,58,57,57,56,56,56,56,54,54,54,54,53,53,53,
    53,53,53,52,52,52,51,51,51,50,49,49,49,48,48,47,47,47,47,46,46,
    46,46,45,45,45,44,43,43,43,43,42,42,41,41,41,41,41,40,40,40,40,
    40,39,39,38,38,37,37,37,37,37,36,36,36,36,35,35,35,35,35,35,34,
    34,34,34,34,34,33,33,32,31,31,30,30
  };
  const int n4c1w1_a[] = {
    100, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,99,99,99,99,98,98,98,98,97,97,97,97,97,97,97,96,96,96,96,
    96,96,96,95,95,95,95,95,94,94,94,94,93,93,93,92,92,92,91,91,91,
    91,90,90,90,89,89,89,89,89,88,88,88,88,87,87,87,87,87,87,86,86,
    86,86,86,86,85,85,85,84,84,83,83,83,83,83,83,82,82,82,82,81,81,
    81,81,80,80,80,80,80,79,79,79,78,78,78,78,78,78,77,77,77,76,76,
    76,76,76,75,75,75,75,75,75,74,74,74,74,73,73,73,73,73,73,73,72,
    72,72,72,71,71,71,71,71,70,70,70,70,70,70,70,70,70,69,69,69,69,
    68,68,67,67,67,67,67,66,66,66,65,65,65,64,64,64,64,63,63,63,63,
    63,63,62,62,62,62,62,62,62,61,61,61,60,60,60,60,60,60,59,59,59,
    58,58,58,58,58,58,57,57,57,57,57,56,56,56,56,56,55,55,54,54,54,
    54,54,54,54,53,53,53,53,53,52,52,52,51,51,51,51,50,50,50,50,50,
    49,49,49,48,48,48,48,48,48,47,47,47,46,46,46,46,46,46,45,45,45,
    45,44,44,44,44,43,43,43,43,43,43,43,42,42,42,42,42,42,42,42,42,
    42,41,41,41,41,41,40,40,40,40,39,39,39,39,38,38,38,38,38,38,37,
    37,37,37,37,37,37,37,36,36,36,36,36,36,35,35,35,35,35,35,34,34,
    34,34,33,33,33,33,33,32,32,32,32,32,31,31,31,31,31,31,31,31,31,
    30,30,30,30,29,29,29,29,29,28,28,28,28,28,28,28,27,27,27,27,27,
    27,27,27,26,26,26,26,26,26,26,25,25,25,25,24,24,24,24,24,24,24,
    23,23,23,23,23,22,22,22,22,22,22,21,21,21,21,20,20,20,20,20,20,
    19,19,19,19,19,19,19,19,18,18,18,18,18,17,17,17,17,17,17,17,16,
    16,15,15,15,15,15,15,15,15,14,14,14,14,13,13,13,13,13,13,13,13,
    13,12,12,12,11,11,11,11,11,11,11,11,11,10,10,10,10,10,10,10,10,
    9,9,9,9,9,8,8,8,7,7,7,7,7,7,6,6,5,5,5,4,4,4,4,4,4,3,3,3,2,2,2,
    2,2,1,1,1,1,1,1
  };
  const int n4c1w1_b[] = {
    100, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,100,99,99,99,99,99,99,99,98,98,98,98,98,98,
    98,97,97,97,97,97,97,96,96,96,95,94,94,93,93,93,93,93,93,93,92,
    92,92,92,92,92,91,91,91,91,91,91,91,90,90,90,90,90,90,90,90,90,
    90,90,89,89,89,88,88,88,87,87,86,86,86,86,85,85,85,85,85,84,84,
    84,84,84,84,83,83,83,82,82,82,82,82,81,81,80,80,80,80,80,80,79,
    79,79,79,78,78,78,78,77,77,77,77,77,77,77,77,77,76,76,76,76,76,
    75,75,75,75,75,75,74,74,74,73,73,73,73,72,72,72,72,72,72,72,71,
    71,71,70,70,70,70,70,69,69,69,68,68,68,68,67,67,67,67,67,66,66,
    66,66,66,65,65,65,65,65,65,65,64,64,64,64,64,64,63,63,63,63,63,
    63,63,62,62,62,62,62,62,62,62,62,62,61,61,61,61,61,61,60,60,60,
    60,60,60,60,60,60,59,59,59,59,59,59,59,59,58,58,57,57,57,56,56,
    56,56,56,55,55,55,55,55,54,54,54,54,54,53,53,52,52,52,52,51,51,
    51,51,50,49,49,49,49,49,49,49,49,49,48,48,48,48,48,48,47,47,47,
    47,47,47,47,46,46,46,46,46,45,45,45,44,44,44,44,44,44,44,44,44,
    43,43,43,43,43,43,43,42,42,42,41,41,41,41,41,41,41,41,40,40,40,
    40,40,40,39,39,39,39,39,38,38,38,38,37,37,37,37,37,37,37,36,36,
    36,36,36,36,36,36,35,35,35,35,35,35,35,35,34,34,33,33,33,32,32,
    32,32,32,31,31,31,30,30,30,30,30,30,30,30,30,29,29,28,28,28,28,
    27,27,26,26,26,26,26,26,26,26,26,26,26,25,25,25,25,25,24,24,24,
    24,23,23,23,22,22,22,22,22,22,21,21,21,21,20,20,20,20,20,19,19,
    19,19,19,19,18,18,18,18,18,17,17,17,16,16,16,16,16,16,15,15,15,
    15,15,15,15,14,14,14,14,13,13,12,12,12,12,12,12,12,11,11,11,11,
    11,11,11,10,10,9,9,9,9,8,8,8,8,7,7,7,7,7,6,5,5,5,4,4,4,4,3,3,
    3,3,3,3,3,3,2,2,2,1,1,1
  };
  const int n4c1w1_c[] = {
    100, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,99,99,99,98,98,98,98,98,98,98,98,97,97,97,97,97,97,
    97,97,97,97,97,96,96,96,96,96,95,95,95,95,95,95,94,93,93,93,92,
    92,92,92,92,92,92,92,91,91,91,90,90,89,89,89,88,88,87,87,87,87,
    87,87,87,86,86,86,85,85,84,84,84,83,83,83,83,83,82,82,82,82,82,
    82,82,81,81,81,81,81,80,80,80,80,80,79,79,79,79,79,79,78,78,77,
    77,77,77,77,77,76,75,75,75,74,74,74,74,73,73,73,73,73,73,73,72,
    72,71,71,71,71,71,71,71,70,70,70,70,70,69,68,68,68,68,68,67,67,
    67,66,66,66,66,65,65,65,65,65,65,65,65,64,64,64,64,64,64,64,64,
    64,64,64,63,63,63,63,63,62,62,61,61,61,60,60,60,60,59,59,59,59,
    58,58,58,58,57,57,57,57,56,56,56,56,56,56,55,55,55,55,55,55,55,
    55,55,55,54,54,53,53,53,53,52,52,52,52,51,51,51,51,51,51,50,50,
    50,50,50,50,50,49,49,49,49,49,49,49,48,48,47,47,46,46,46,45,45,
    45,45,44,44,44,44,43,43,43,43,43,42,42,42,42,42,42,42,42,42,42,
    41,41,41,41,40,40,40,40,39,39,39,39,39,38,38,38,38,38,38,38,38,
    37,37,37,37,37,37,37,37,37,36,36,36,36,36,35,35,35,35,35,35,35,
    34,34,34,34,34,33,33,33,33,33,33,33,33,32,32,32,32,31,31,31,31,
    31,31,31,31,30,30,30,30,30,29,29,29,29,28,28,28,28,27,27,26,26,
    26,26,25,25,25,25,25,24,24,24,24,24,24,23,23,23,23,23,23,22,22,
    22,22,22,21,21,21,21,20,20,20,20,20,19,19,19,19,19,19,19,19,19,
    19,18,18,18,18,17,17,17,17,17,17,17,17,17,17,16,16,16,16,16,16,
    15,15,15,15,15,15,15,15,15,14,14,14,14,13,13,13,13,13,12,12,12,
    12,11,11,11,11,10,10,10,10,10,9,9,9,9,9,9,9,9,9,9,8,8,8,8,8,7,
    7,7,7,7,7,7,7,6,6,6,6,5,5,5,5,5,5,4,4,4,4,4,4,4,3,3,3,3,3,2,2,
    2,2,1
  };
  const int n4c1w1_d[] = {
    100, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,100,99,99,99,99,99,98,98,97,97,97,97,97,97,
    97,96,96,96,96,96,95,95,95,95,95,95,94,94,94,94,94,93,93,93,93,
    93,93,93,92,92,92,92,92,92,91,91,91,91,90,90,90,90,89,89,89,89,
    89,89,89,89,88,88,88,88,88,88,88,88,88,88,87,87,87,87,86,86,86,
    86,85,85,85,85,84,84,84,84,84,84,84,83,83,83,83,83,83,83,82,81,
    81,81,81,81,81,81,80,80,80,79,79,79,79,78,78,78,78,77,77,77,77,
    76,76,76,76,76,75,74,74,74,74,74,73,73,72,72,72,72,71,71,70,70,
    70,70,69,69,69,69,69,68,68,68,68,68,68,68,68,67,67,67,67,67,66,
    66,65,65,65,64,64,63,63,63,63,63,63,63,63,63,63,62,62,61,61,61,
    60,60,60,60,59,59,59,58,58,58,57,57,56,56,56,56,56,56,56,55,55,
    55,55,54,54,54,54,54,53,53,53,53,52,52,52,51,51,51,51,51,51,51,
    51,51,51,50,50,50,50,50,50,50,50,50,49,49,49,49,48,48,47,46,46,
    46,46,46,46,46,45,45,45,44,44,44,44,43,43,43,43,43,43,42,42,42,
    42,42,42,42,42,42,42,42,41,41,41,41,41,40,40,40,40,39,39,39,39,
    39,39,38,38,38,38,37,37,37,37,37,37,37,36,36,35,35,35,35,34,34,
    33,33,33,33,33,33,33,33,33,32,32,32,32,32,32,32,32,32,32,32,31,
    31,31,31,31,30,30,30,30,30,30,29,29,29,29,28,28,28,27,27,27,27,
    26,26,26,26,26,26,25,25,25,25,25,25,24,24,24,24,24,23,23,23,23,
    22,22,22,22,22,22,21,21,21,21,21,21,21,21,20,20,20,20,20,19,19,
    19,19,19,19,19,19,18,18,17,17,17,17,17,16,16,16,16,16,16,15,15,
    15,15,14,14,14,14,14,14,13,13,13,13,13,13,13,12,12,12,12,12,12,
    12,11,11,11,11,11,11,10,10,10,10,10,9,9,9,9,9,8,8,7,7,7,7,7,7,
    7,7,6,6,6,6,5,5,5,5,5,4,4,4,4,4,4,4,4,3,3,3,3,2,2,2,2,2,2,2,1,
    1,1,1,1,1
  };
  const int n4c1w1_e[] = {
    100, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,99,99,99,98,98,98,98,98,98,97,97,97,97,97,97,97,96,
    96,96,96,96,96,96,96,95,95,95,95,95,95,94,94,94,94,93,93,93,93,
    93,92,92,92,92,90,90,90,90,90,90,90,89,89,89,89,89,89,88,88,88,
    88,88,88,88,88,87,87,86,86,86,86,86,85,85,85,85,84,84,84,83,83,
    83,83,82,82,82,82,82,82,82,81,81,81,81,81,81,80,80,80,79,79,79,
    79,78,78,78,78,78,78,78,78,78,78,78,78,77,77,77,77,77,77,76,76,
    76,76,76,76,75,75,75,74,74,74,74,74,74,73,73,73,73,73,73,72,72,
    72,72,72,72,72,71,71,71,71,71,71,70,70,70,70,70,70,70,70,69,69,
    69,68,68,68,68,68,68,68,67,67,67,67,67,67,66,66,66,66,66,66,65,
    65,65,64,64,64,63,63,63,63,63,63,63,63,63,63,63,62,62,62,62,61,
    60,60,60,60,60,60,59,59,59,58,58,58,58,58,57,57,57,57,57,57,56,
    56,56,56,56,55,55,55,55,54,54,54,54,54,54,54,54,53,53,53,53,53,
    53,52,52,52,52,52,52,52,51,51,51,51,51,51,51,51,51,50,50,50,50,
    50,50,50,49,49,49,49,48,48,48,48,48,48,47,47,47,47,47,46,46,46,
    46,46,45,45,45,45,44,44,44,43,43,43,43,43,42,42,42,41,41,41,40,
    40,40,40,39,39,39,39,39,38,38,38,38,38,38,37,37,36,36,36,36,35,
    35,34,34,34,34,33,33,33,33,32,32,32,32,32,32,31,31,31,31,31,31,
    30,30,30,30,30,30,30,29,29,29,29,28,28,28,28,28,28,27,27,27,26,
    26,25,25,25,24,24,24,24,24,24,23,23,23,23,23,23,23,22,22,22,22,
    21,21,21,21,21,20,20,20,20,19,19,19,19,18,18,18,18,17,17,17,17,
    17,17,16,16,16,16,16,16,16,16,16,16,15,15,15,14,14,14,14,14,13,
    13,13,13,12,12,12,12,12,12,12,11,11,11,11,10,10,10,10,9,9,9,9,
    8,8,8,7,7,7,7,7,7,6,6,6,6,6,6,6,5,5,5,5,5,5,4,4,4,3,3,2,2,2,2,
    2,1,1,1,1,1,1
  };
  const int n4c1w1_f[] = {
    100, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,99,99,98,98,98,98,98,97,97,97,97,97,97,96,
    96,96,96,95,95,95,95,95,94,94,93,93,93,93,93,93,92,92,92,92,92,
    92,91,91,91,91,91,91,90,90,90,90,90,90,90,90,89,89,89,89,89,89,
    88,88,88,87,87,87,87,87,87,87,86,86,86,86,86,86,86,86,85,85,84,
    84,84,84,84,84,84,84,83,83,83,83,83,83,82,82,81,81,81,81,81,81,
    80,79,79,79,79,79,79,78,78,78,78,77,77,77,77,77,77,76,76,76,76,
    76,75,75,75,75,75,74,74,74,74,73,73,73,73,72,72,71,71,71,71,71,
    71,71,71,71,71,70,70,70,70,70,70,70,69,69,69,68,68,68,68,68,67,
    67,66,66,66,66,66,66,66,66,65,65,65,65,65,65,65,65,64,64,64,64,
    64,63,63,63,63,63,63,63,62,62,62,61,61,61,61,61,61,61,60,60,60,
    60,60,60,59,59,59,59,59,59,59,59,58,58,58,58,57,57,57,57,57,57,
    57,57,56,56,56,56,56,55,55,55,55,55,53,53,53,53,52,52,52,51,51,
    51,51,51,51,50,50,50,50,50,49,49,49,49,49,49,49,49,49,48,48,48,
    47,47,47,47,47,47,47,47,46,46,46,46,46,45,45,45,45,45,45,44,44,
    44,43,43,43,43,42,42,42,42,42,42,41,41,41,41,41,41,40,40,40,40,
    40,40,39,39,39,39,39,38,38,38,38,38,37,37,37,37,37,37,37,37,37,
    37,36,36,36,36,36,36,36,36,36,35,34,34,33,33,33,33,32,32,32,32,
    32,32,32,32,32,32,32,32,31,31,31,31,31,31,31,30,30,30,30,29,29,
    29,29,28,28,28,28,28,28,28,27,27,27,27,27,26,26,26,26,26,26,26,
    25,25,25,25,24,24,24,24,24,24,24,24,24,23,23,23,23,23,23,22,22,
    22,21,21,21,21,20,20,20,20,20,20,19,19,19,19,18,18,17,17,17,17,
    17,17,17,17,16,15,15,15,14,14,13,13,13,12,12,12,12,11,11,11,11,
    11,10,10,10,10,10,9,9,8,8,8,7,7,7,7,7,6,6,6,6,5,5,5,5,4,4,4,3,
    3,3,2,2,2,2,2,2,1,1,1,1
  };
  const int n4c1w1_g[] = {
    100, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,99,99,99,99,98,98,98,97,97,97,97,97,97,96,96,96,96,96,96,
    96,95,95,95,95,95,94,94,94,94,94,94,94,93,93,93,92,92,92,91,91,
    91,90,90,90,90,90,90,89,89,89,89,89,89,89,89,88,88,88,88,88,88,
    88,88,88,88,88,87,87,87,87,87,86,86,86,86,86,85,85,85,85,85,85,
    85,85,85,84,84,84,84,83,83,83,82,82,82,81,81,81,81,80,80,80,80,
    80,80,80,80,80,79,79,79,79,79,79,79,79,79,78,78,78,78,78,78,78,
    78,77,77,77,77,76,76,76,75,75,75,75,74,74,74,74,74,74,73,73,73,
    73,72,72,72,72,71,70,70,70,70,69,69,69,69,68,68,68,68,68,68,68,
    67,67,67,67,67,67,67,67,67,67,66,66,66,66,66,66,66,65,65,64,64,
    64,64,63,62,62,62,62,61,61,61,60,60,60,60,60,60,59,59,59,59,58,
    58,58,58,58,58,58,58,58,57,57,57,57,57,57,56,56,56,56,55,55,55,
    54,54,54,54,53,53,53,53,53,52,52,52,52,51,51,51,51,51,50,50,50,
    50,49,49,49,49,49,49,48,48,48,48,48,48,48,48,47,47,47,47,46,46,
    46,46,46,45,45,45,45,44,44,44,44,44,44,44,44,44,44,43,43,43,43,
    43,43,43,42,42,42,42,42,41,41,41,40,40,40,39,39,39,39,39,39,38,
    38,38,38,38,38,38,38,37,37,37,37,36,36,36,36,36,35,35,35,34,34,
    34,33,33,33,33,33,33,32,31,31,31,31,31,30,30,30,30,30,30,30,29,
    29,28,28,28,28,28,28,28,27,27,27,27,27,27,26,26,26,26,26,26,26,
    26,26,26,26,26,25,25,24,24,24,23,23,21,21,21,21,21,21,20,20,20,
    20,20,19,19,19,19,19,18,18,18,18,18,18,18,17,17,17,17,17,17,17,
    17,17,17,16,16,16,16,16,16,15,15,15,15,15,14,14,14,14,14,13,13,
    13,12,12,12,12,12,12,12,12,11,11,11,11,11,10,10,9,9,9,9,9,9,9,
    9,8,8,8,7,7,7,7,6,6,6,6,6,6,6,5,5,5,5,4,4,4,4,4,3,3,2,2,2,2,2,
    2,1,1,1,1,1
  };
  const int n4c1w1_h[] = {
    100, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,99,99,99,99,98,98,98,97,97,97,97,96,96,96,96,96,96,96,
    95,95,95,94,94,94,93,93,92,92,92,92,92,92,92,92,92,91,91,91,91,
    91,90,90,90,90,90,90,89,89,89,89,89,89,88,88,88,88,88,88,88,88,
    88,88,87,87,86,86,86,86,85,85,85,84,84,84,84,83,83,83,83,83,82,
    82,82,82,82,82,82,81,81,81,80,80,80,80,79,79,79,79,79,79,78,78,
    78,77,77,77,76,76,76,76,76,76,76,75,75,75,75,75,75,75,74,74,74,
    74,74,74,74,74,74,74,73,73,73,73,72,72,72,72,72,72,72,72,71,71,
    70,70,69,69,69,69,69,69,69,68,68,68,68,67,67,67,67,67,67,66,66,
    66,66,66,66,66,66,66,66,65,65,63,63,63,63,63,63,63,63,63,62,62,
    62,62,62,62,62,62,61,61,61,60,60,60,60,60,60,60,59,59,59,59,59,
    59,59,58,58,58,58,58,58,57,57,57,56,56,56,56,55,55,55,54,54,53,
    53,53,53,52,52,52,52,52,52,52,52,52,51,51,51,51,51,51,50,50,50,
    50,50,50,49,48,48,48,48,48,48,47,47,47,47,47,47,46,46,46,46,46,
    46,45,45,44,44,43,43,43,42,42,42,42,42,41,41,41,41,40,40,40,40,
    40,40,39,39,39,39,39,38,38,38,38,38,37,37,37,37,36,36,36,36,36,
    36,36,36,36,35,35,35,34,34,34,34,34,33,33,33,33,32,32,32,32,32,
    32,32,32,32,32,32,31,31,31,31,30,30,30,30,30,30,29,29,29,29,29,
    29,29,28,28,28,28,28,28,27,27,27,26,26,26,26,26,26,26,26,25,25,
    25,25,24,24,23,23,23,23,23,22,22,22,22,21,21,21,21,21,21,21,21,
    20,20,20,20,20,20,20,20,19,19,19,19,19,18,18,18,18,17,17,17,17,
    17,16,16,16,16,16,15,15,14,14,14,14,14,14,14,14,14,14,14,13,13,
    12,12,12,12,12,12,12,11,11,11,11,10,10,10,10,10,10,9,9,9,8,8,
    8,8,8,7,7,7,7,7,7,6,6,6,6,6,6,6,6,6,5,5,5,5,5,5,4,4,4,3,3,3,3,
    2,2,2,1,1,1,1
  };
  const int n4c1w1_i[] = {
    100, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,100,99,99,99,99,99,99,99,99,98,98,98,98,98,
    98,98,97,97,97,97,97,96,96,95,95,95,95,94,94,93,93,93,93,92,92,
    92,92,92,91,91,91,91,91,90,90,90,90,90,90,90,90,89,89,89,88,88,
    88,88,88,88,88,87,87,87,87,87,87,86,86,86,86,86,86,86,85,85,85,
    85,85,84,84,84,84,84,83,83,82,82,82,82,82,82,82,81,81,81,81,81,
    81,80,80,80,80,80,79,78,78,78,78,77,77,77,77,77,76,76,76,76,76,
    75,75,75,75,75,75,75,74,74,74,74,74,74,73,73,73,73,73,73,73,72,
    72,72,72,70,70,70,69,69,69,69,69,68,68,68,68,68,68,67,67,66,66,
    66,65,65,65,65,65,64,64,64,63,63,63,63,63,63,63,63,63,62,62,62,
    62,62,61,61,60,60,59,59,59,59,59,59,58,58,58,58,58,58,58,58,58,
    58,57,57,56,56,56,56,55,55,55,55,55,55,54,54,54,54,54,53,53,53,
    53,53,53,53,53,52,52,52,52,51,51,51,51,51,50,50,50,50,50,50,50,
    50,50,50,50,50,49,49,49,49,49,48,48,48,48,48,48,48,48,48,47,47,
    47,47,47,47,46,46,46,46,45,45,45,45,44,44,44,44,44,43,43,43,43,
    42,42,42,42,41,41,41,41,41,41,41,41,41,41,41,41,41,40,40,40,40,
    40,40,39,39,39,39,39,39,39,39,39,38,38,38,38,38,38,37,37,37,37,
    37,37,37,37,36,36,36,35,35,35,35,34,34,34,34,34,34,34,34,33,33,
    33,32,32,32,32,32,32,32,32,32,32,32,32,31,31,31,31,31,31,31,29,
    29,29,29,28,28,28,28,28,28,28,27,27,27,27,26,26,25,25,25,25,24,
    24,23,23,23,23,23,23,23,22,22,21,21,20,20,20,20,20,19,19,19,19,
    18,18,18,18,18,18,17,17,17,17,16,16,15,15,15,14,14,14,14,14,14,
    14,14,14,13,13,13,13,13,12,12,12,11,11,11,11,11,10,10,10,9,9,
    9,9,9,9,8,7,7,7,7,7,6,6,6,6,6,6,5,5,5,5,4,4,4,4,3,3,2,2,2,2,2,
    2,2,2,1,1,1,1,1,1
  };
  const int n4c1w1_j[] = {
    100, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,99,99,99,99,99,99,99,99,98,98,97,97,97,97,
    97,97,97,96,96,96,96,96,96,96,96,96,96,95,95,95,95,95,95,95,94,
    93,93,93,93,93,93,93,93,93,92,92,92,92,92,92,92,92,92,92,92,91,
    91,91,91,90,90,90,90,90,90,90,89,88,88,88,88,88,87,87,87,87,87,
    87,86,86,86,86,85,85,85,85,84,84,84,84,84,83,83,83,83,83,83,83,
    82,82,82,82,82,81,81,81,81,81,81,80,80,80,80,80,80,79,79,79,78,
    78,78,78,78,78,78,78,78,77,77,77,77,77,77,77,76,76,76,76,76,76,
    75,75,75,75,75,75,74,74,74,74,73,73,73,73,72,72,72,72,71,71,71,
    71,71,71,70,70,70,70,70,69,69,69,69,69,69,68,68,67,67,67,67,67,
    66,66,66,66,66,65,65,65,65,65,65,65,64,64,64,64,64,64,64,64,64,
    64,63,63,63,63,63,63,63,63,63,63,63,62,62,62,62,61,61,61,60,60,
    60,60,60,60,59,59,59,59,59,59,58,58,58,58,58,58,58,58,58,58,58,
    57,57,57,57,57,57,56,56,56,55,55,55,55,55,55,55,54,54,54,54,54,
    53,53,53,52,52,52,52,52,52,52,52,52,51,51,51,51,50,49,49,48,48,
    48,48,48,47,47,46,46,46,45,45,45,45,45,45,45,45,44,44,44,44,44,
    43,43,43,43,43,42,42,42,41,41,40,39,39,39,39,39,39,38,38,38,37,
    37,37,36,36,36,36,36,36,36,35,35,34,34,34,33,33,33,33,33,33,33,
    33,33,32,32,31,31,31,31,31,31,31,30,30,30,30,30,30,29,29,29,29,
    28,28,28,27,27,27,27,27,27,26,26,26,25,25,25,25,24,24,24,24,24,
    24,24,24,23,23,23,23,23,23,23,22,22,22,22,22,22,22,21,21,20,20,
    20,20,20,19,19,19,19,18,18,18,18,18,18,18,17,16,16,16,16,16,15,
    15,14,14,14,14,14,14,13,13,13,13,13,13,13,12,11,10,10,10,9,8,
    8,8,8,8,8,8,7,7,7,6,6,6,6,5,5,5,5,5,5,5,5,5,5,4,4,3,3,3,3,3,3,
    3,3,3,3,2,2,2,1,1
  };
  const int n4c1w1_k[] = {
    100, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,99,99,99,99,98,98,98,97,97,97,97,97,97,96,96,
    96,95,95,94,94,94,94,94,93,93,93,93,93,93,92,92,92,92,91,91,91,
    90,90,90,90,90,90,89,89,89,89,89,88,88,87,87,87,86,86,86,86,86,
    85,85,85,85,85,85,85,84,84,84,84,83,83,83,83,83,83,82,82,81,81,
    81,81,81,80,80,80,80,80,80,80,80,79,79,79,79,78,78,78,78,78,78,
    78,78,77,77,77,77,76,76,76,76,75,75,75,75,74,74,74,74,74,74,74,
    74,73,73,73,73,73,73,72,72,72,72,72,72,71,71,71,71,71,70,70,70,
    70,69,69,69,69,69,69,69,68,68,68,68,68,68,68,67,67,67,67,67,66,
    66,66,66,66,66,65,65,65,64,64,64,64,64,64,63,63,63,63,62,62,62,
    61,61,61,61,61,61,60,60,60,60,60,60,60,60,59,59,58,58,58,58,58,
    58,58,58,58,58,57,57,57,56,56,56,55,55,55,55,55,55,54,54,54,54,
    54,53,53,53,53,53,53,53,52,52,52,52,52,51,51,51,50,50,50,50,50,
    50,49,49,49,49,49,49,49,49,49,49,49,49,48,48,47,47,46,46,46,46,
    46,46,46,46,46,46,46,45,45,45,44,44,44,43,43,43,43,43,42,42,42,
    42,42,42,41,41,41,40,40,40,40,40,40,40,39,39,39,39,39,39,38,38,
    37,37,37,37,37,37,36,36,36,36,36,35,35,35,35,35,35,35,35,35,34,
    34,34,33,33,33,33,33,32,32,32,32,32,31,31,31,30,30,30,30,30,30,
    30,29,29,29,29,29,29,28,28,28,28,28,28,28,28,28,27,27,27,27,27,
    26,26,26,26,26,25,25,25,24,24,23,23,23,22,22,22,22,22,22,22,22,
    22,22,21,21,21,21,20,20,20,19,19,19,19,19,18,18,18,17,17,17,17,
    17,17,17,17,17,16,16,16,16,16,15,15,15,15,14,14,14,14,13,13,13,
    12,12,12,12,12,11,11,10,10,10,10,10,10,10,8,8,8,8,8,8,8,7,7,7,
    6,6,6,6,6,6,5,5,5,5,5,5,5,5,4,4,4,4,4,3,3,3,3,3,3,2,2,2,2,2,1,
    1,1,1,1,1,1
  };
  const int n4c1w1_l[] = {
    100, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,99,99,99,99,99,99,99,98,97,97,97,96,96,96,
    96,96,96,96,96,95,95,95,95,94,94,94,94,94,94,93,93,93,93,92,91,
    91,91,91,91,90,90,89,89,89,89,88,88,88,88,87,87,87,87,87,87,87,
    86,86,86,86,86,85,85,85,85,85,85,85,85,85,85,85,85,84,84,84,84,
    84,83,83,82,82,82,82,82,81,81,81,81,81,80,80,80,79,79,79,79,79,
    79,79,78,78,78,78,78,78,77,77,77,77,77,77,76,76,76,76,76,76,76,
    75,75,75,75,75,74,74,74,74,74,74,74,74,74,73,73,73,73,73,72,72,
    72,72,72,71,71,71,71,71,71,70,70,70,69,69,69,69,69,69,68,68,68,
    68,68,68,68,68,67,67,67,67,66,66,66,66,66,66,66,65,65,65,65,65,
    64,64,64,64,64,63,63,63,62,62,62,62,62,62,62,62,61,61,61,61,60,
    60,60,59,59,59,58,58,58,58,58,58,58,57,57,57,57,57,57,56,56,56,
    56,56,56,56,55,55,55,55,54,54,54,54,53,53,53,53,52,52,52,52,52,
    52,51,51,51,51,51,51,50,50,49,49,49,49,49,48,48,48,48,48,47,47,
    47,47,47,46,46,46,45,45,44,44,44,44,44,44,43,43,43,43,42,42,42,
    42,42,42,42,42,41,41,41,41,41,40,40,40,39,39,39,38,38,38,38,38,
    38,37,37,37,37,36,36,36,36,36,36,36,36,35,35,35,35,35,35,34,34,
    34,34,34,34,34,34,34,33,33,33,32,31,31,31,31,31,31,30,30,30,30,
    30,29,29,29,29,29,29,29,28,28,28,27,27,27,27,26,26,26,26,26,26,
    25,25,25,25,25,25,24,24,24,24,24,24,23,23,23,23,23,23,22,22,22,
    22,21,21,21,21,21,21,21,21,19,18,18,18,18,18,18,18,17,17,17,17,
    17,17,17,17,17,16,16,16,16,15,15,15,15,15,15,15,15,15,14,14,14,
    13,13,13,13,12,12,12,12,12,11,11,10,10,10,10,10,10,10,9,9,9,9,
    9,8,8,7,7,7,7,7,7,6,6,6,6,5,5,5,5,5,4,4,4,4,4,4,3,3,3,3,3,3,3,
    2,2,2,2,1,1,1,1
  };
  const int n4c1w1_m[] = {
    100, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,100,100,100,99,99,99,99,99,99,99,98,98,97,
    97,97,96,96,95,95,95,94,94,94,94,94,94,94,93,93,93,93,93,93,93,
    92,92,92,92,91,91,91,90,90,90,90,90,90,89,89,89,89,89,88,88,88,
    88,88,88,87,87,87,87,87,86,86,86,86,86,86,85,84,84,84,83,83,83,
    83,83,83,83,82,82,82,82,82,82,82,81,81,81,81,81,81,81,80,80,79,
    79,79,79,79,78,78,78,78,78,78,78,77,77,77,76,76,76,76,75,75,75,
    74,74,74,73,73,73,73,73,73,73,72,72,72,72,72,72,72,72,71,71,70,
    70,70,70,70,70,69,69,69,69,68,68,68,67,67,67,67,67,66,66,66,66,
    66,64,64,64,64,63,63,63,63,63,63,63,62,62,62,62,61,61,60,60,60,
    60,60,60,60,60,60,60,59,59,58,58,58,58,58,58,57,57,57,57,56,56,
    56,56,56,56,54,53,53,53,53,52,52,52,52,52,52,52,52,52,51,51,51,
    50,50,50,49,49,49,49,49,49,49,49,49,49,48,48,47,47,46,46,46,46,
    46,46,46,45,45,45,45,45,45,45,44,44,44,44,43,43,42,42,42,42,42,
    42,42,42,41,41,41,41,41,41,41,41,41,40,40,40,39,39,39,39,38,38,
    38,38,38,38,37,37,37,37,37,37,37,36,36,36,36,36,36,36,36,36,36,
    35,35,35,34,34,34,34,34,33,33,33,33,33,33,32,32,32,32,32,32,32,
    32,31,31,31,30,30,30,30,30,30,30,29,29,29,29,29,28,28,28,28,28,
    28,28,28,27,27,27,26,26,26,26,26,26,26,26,26,25,25,25,25,25,25,
    25,25,24,24,24,24,24,23,23,23,23,23,23,23,22,22,22,22,21,21,21,
    20,20,20,20,19,19,19,19,18,18,18,18,18,18,17,17,17,17,17,17,17,
    17,16,16,16,16,16,15,15,15,15,15,15,15,14,14,14,14,14,14,13,13,
    13,12,12,12,12,12,12,11,11,11,11,11,11,11,11,11,10,10,10,9,9,
    9,9,8,8,8,8,7,7,7,7,7,7,6,6,6,6,5,5,5,5,5,4,4,4,4,4,4,4,4,3,3,
    3,3,3,2,2,2,2,1,1,1
  };
  const int n4c1w1_n[] = {
    100, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,100,100,99,99,99,98,98,98,98,98,98,97,97,
    97,97,97,97,97,96,96,96,96,95,95,95,95,94,94,94,94,94,94,94,94,
    94,93,93,93,93,92,92,92,92,91,91,91,90,90,90,89,89,89,89,89,89,
    89,88,88,87,87,87,87,87,86,86,86,86,86,85,85,84,84,84,84,84,83,
    83,83,83,83,83,83,83,83,82,82,82,82,82,81,81,81,81,81,80,80,80,
    80,79,79,79,79,79,78,78,78,78,77,77,76,76,76,76,76,76,75,75,75,
    75,75,75,75,75,75,75,74,74,73,73,73,73,73,73,72,72,72,72,72,71,
    71,71,71,70,70,70,70,69,69,69,68,68,68,68,68,68,68,68,68,67,67,
    67,67,66,66,66,66,66,66,66,66,66,65,64,64,64,64,64,64,64,64,63,
    63,63,63,63,63,63,62,62,62,62,62,62,62,62,61,61,61,61,61,61,61,
    60,59,59,59,59,58,58,58,58,57,57,57,57,57,56,55,55,55,55,55,55,
    54,54,54,54,54,54,54,53,53,53,53,53,53,53,52,52,52,51,51,51,51,
    51,51,51,50,50,50,50,50,50,49,49,49,49,49,49,49,48,48,48,47,47,
    46,46,45,45,45,45,45,45,45,45,44,44,44,44,44,44,44,43,43,42,42,
    42,42,42,41,41,41,41,41,41,40,40,40,40,40,39,39,39,39,38,38,38,
    37,37,37,37,36,36,36,36,35,35,35,35,35,34,34,34,34,34,34,34,34,
    34,33,33,33,33,33,33,33,32,32,32,31,31,31,31,30,30,30,30,29,29,
    29,29,28,28,28,28,28,28,28,27,27,27,26,26,26,26,25,25,25,25,24,
    24,24,24,23,23,23,23,23,23,22,22,22,22,22,21,21,21,21,21,20,20,
    20,19,19,19,19,19,19,19,18,18,18,18,18,18,18,17,17,17,17,17,16,
    15,15,15,15,15,15,15,14,14,14,14,14,13,13,13,13,13,13,13,12,12,
    12,12,12,12,12,11,11,11,10,10,10,10,10,10,10,10,9,9,9,9,9,8,8,
    8,7,7,7,7,7,7,7,6,6,5,5,5,5,5,4,4,4,4,4,3,3,3,3,3,2,2,2,2,2,2,
    2,2,1,1,1,1,1,1
  };
  const int n4c1w1_o[] = {
    100, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,99,99,99,99,99,99,98,98,98,98,98,98,98,98,97,97,97,
    97,97,97,96,96,96,96,95,95,95,95,94,94,94,94,93,93,93,93,93,92,
    92,92,92,91,91,91,90,90,90,90,90,90,90,90,90,90,90,89,89,89,88,
    88,88,88,87,87,87,87,87,87,87,87,87,86,86,86,86,86,86,86,85,85,
    85,85,85,85,84,84,84,83,83,83,83,83,82,82,82,82,82,82,82,81,81,
    81,81,81,81,81,81,81,81,80,80,80,80,79,79,79,79,79,79,79,78,78,
    78,78,78,78,77,77,77,77,77,76,76,76,76,76,76,76,75,75,75,74,74,
    74,74,73,73,73,73,73,72,72,72,72,72,71,71,71,71,69,69,69,69,69,
    69,68,68,67,67,67,67,67,66,66,66,66,65,65,65,65,65,64,64,63,62,
    62,62,62,61,61,61,61,60,60,60,60,60,60,60,60,59,59,59,59,59,59,
    59,59,58,58,58,58,57,57,57,57,57,57,57,57,56,55,55,55,55,54,53,
    53,53,53,53,53,53,53,53,52,52,52,52,52,52,52,51,51,51,51,50,50,
    50,49,49,49,49,49,49,49,49,49,48,48,48,48,48,48,47,47,47,47,47,
    47,47,46,46,46,46,45,45,45,45,44,44,44,44,44,44,44,44,43,43,43,
    43,43,43,42,42,42,42,42,42,41,41,41,41,40,40,40,40,39,39,38,38,
    37,37,37,37,37,37,37,36,36,36,36,35,35,35,35,35,35,34,34,34,34,
    34,34,33,33,33,33,33,32,32,32,31,31,31,31,31,31,30,29,29,29,29,
    29,28,28,28,28,28,28,27,27,26,26,26,26,26,26,25,25,25,25,25,24,
    24,24,24,24,23,23,23,23,22,22,22,21,21,21,21,21,21,20,20,20,20,
    20,19,19,19,18,18,18,18,17,17,16,16,16,16,16,16,16,15,15,15,15,
    15,15,15,15,15,15,15,15,14,14,14,14,14,14,13,13,13,13,13,13,12,
    12,12,12,12,12,11,11,11,11,10,10,9,9,9,9,8,8,8,8,8,8,7,7,7,7,
    7,7,7,6,6,6,6,6,6,6,5,5,4,4,4,4,4,4,3,3,3,3,3,3,3,2,2,2,1,1,1,
    1,1,1,1
  };
  const int n4c1w1_p[] = {
    100, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,100,99,99,99,99,98,98,97,97,97,97,97,97,97,
    96,96,96,96,96,95,95,95,95,95,95,94,94,94,94,94,94,93,93,93,93,
    93,92,92,92,91,91,91,91,91,91,90,90,90,90,90,90,90,90,89,89,89,
    89,89,89,89,89,88,88,88,88,88,88,88,87,87,87,87,86,86,86,86,86,
    85,85,85,85,85,84,84,84,84,84,83,83,83,83,83,83,82,82,82,82,81,
    81,81,81,81,81,81,80,80,80,80,80,80,79,78,78,78,78,78,77,77,77,
    77,77,77,77,76,76,76,76,76,76,76,76,75,75,75,75,75,75,74,74,74,
    74,73,73,73,73,72,72,72,72,72,72,72,71,71,71,71,71,71,70,70,70,
    70,70,70,70,69,69,69,69,69,68,68,68,68,68,67,66,66,66,65,65,65,
    65,65,65,65,64,64,63,63,63,63,63,62,62,62,62,62,62,61,61,61,61,
    61,60,60,60,60,60,60,60,59,59,59,59,59,59,59,59,59,59,59,58,58,
    58,58,58,57,57,57,57,57,56,56,56,56,56,56,56,56,56,56,55,55,55,
    55,54,54,54,54,54,52,52,52,52,52,51,51,51,51,50,50,50,50,49,49,
    49,49,49,49,49,48,48,48,47,47,47,47,47,46,46,46,46,46,46,45,45,
    45,45,44,44,44,44,43,43,43,43,42,42,41,41,41,41,41,40,40,40,39,
    39,39,39,38,38,38,38,37,37,37,37,37,36,36,36,35,35,34,34,34,33,
    33,33,32,32,32,32,32,32,32,31,30,30,30,30,30,30,30,30,30,29,29,
    29,29,29,29,29,29,28,28,28,28,28,28,27,27,27,27,27,26,26,26,26,
    26,25,25,25,25,24,24,24,24,24,23,23,23,23,23,22,22,22,22,21,21,
    21,21,21,20,20,20,20,20,20,20,20,19,19,19,19,19,18,18,17,17,16,
    16,16,16,16,15,15,15,15,15,15,15,14,14,14,14,14,13,13,13,12,12,
    12,12,12,12,11,11,11,11,11,11,10,10,10,10,9,9,9,9,9,9,9,8,8,8,
    8,8,8,8,7,7,7,6,6,6,6,6,6,6,6,6,6,5,5,4,4,4,3,3,3,3,2,2,2,2,1,
    1,1,1,1,1,1
  };
  const int n4c1w1_q[] = {
    100, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,99,99,99,99,98,98,98,98,97,97,97,97,96,96,96,96,96,
    96,96,96,96,95,95,95,94,94,94,94,94,94,94,93,93,93,93,92,92,92,
    91,91,91,90,90,90,89,89,89,89,89,89,89,88,88,88,88,87,87,87,87,
    87,86,86,86,86,86,86,86,86,86,85,85,85,85,84,84,84,84,84,84,84,
    83,83,83,83,83,83,82,82,82,82,82,81,81,81,81,81,81,81,81,80,80,
    80,79,79,79,79,78,78,78,78,78,77,77,77,77,77,77,77,77,77,77,77,
    76,76,76,75,75,75,75,75,75,74,74,74,73,73,73,73,73,73,73,72,72,
    72,72,72,72,71,71,71,71,71,70,70,70,70,70,69,69,69,69,69,68,68,
    68,68,68,68,68,68,67,67,67,67,67,67,67,67,66,66,66,66,66,66,66,
    66,66,65,65,65,65,65,65,64,64,64,64,64,64,63,63,63,63,63,63,62,
    62,62,62,62,62,62,61,61,61,61,61,61,61,61,61,60,60,60,60,60,60,
    59,59,59,59,59,58,57,57,57,57,56,56,56,56,56,56,56,56,56,55,55,
    55,54,54,54,54,53,53,53,53,53,53,53,53,52,52,51,51,51,51,51,51,
    51,51,50,50,50,50,50,50,49,49,49,49,49,48,48,48,48,47,47,47,47,
    46,46,45,45,45,44,44,43,43,43,42,42,42,41,41,41,41,41,41,41,40,
    40,39,39,39,39,39,39,39,38,38,37,37,37,36,36,36,36,36,36,36,36,
    36,35,35,35,35,34,34,34,34,34,34,34,33,33,32,32,32,32,32,32,32,
    32,31,31,30,30,30,30,29,29,28,28,28,28,28,28,28,28,27,27,27,27,
    27,26,26,26,26,25,25,25,25,25,25,25,24,24,24,24,24,23,23,23,22,
    21,21,21,21,20,20,20,20,20,20,19,19,19,19,18,18,18,18,18,18,17,
    17,17,16,16,16,16,16,15,15,15,15,15,14,14,14,14,13,13,13,13,13,
    13,13,13,13,12,12,12,12,11,11,11,10,10,10,9,9,8,8,7,7,7,6,6,6,
    6,6,5,5,5,5,5,5,4,4,4,4,4,4,4,4,4,4,4,3,3,3,3,3,3,2,2,2,2,1,1,
    1,1,1,1,1
  };
  const int n4c1w1_r[] = {
    100, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,99,99,98,98,98,98,98,98,97,97,97,96,96,96,
    96,96,95,95,95,95,95,95,95,94,94,94,94,94,93,93,93,92,92,92,92,
    92,92,92,91,91,91,90,90,90,90,90,89,89,89,89,89,89,88,88,88,88,
    88,88,87,87,87,86,86,86,86,85,85,84,84,84,84,84,84,84,84,83,83,
    83,83,83,83,82,82,81,81,81,81,80,80,80,80,80,80,80,79,79,79,78,
    78,78,78,78,78,77,77,76,76,76,76,76,75,75,75,75,75,75,75,74,74,
    74,74,74,73,73,73,73,73,73,72,71,71,71,71,71,71,70,70,70,70,70,
    70,69,69,69,69,69,68,68,68,68,68,67,67,67,67,67,67,67,67,67,66,
    66,65,65,65,65,65,64,64,64,64,63,63,63,63,62,62,62,62,62,62,61,
    61,61,61,61,61,61,60,60,60,60,60,60,59,59,59,59,59,59,59,59,59,
    58,58,58,58,58,57,57,57,56,56,56,56,56,56,56,56,55,55,55,55,55,
    54,54,54,54,53,53,53,53,53,52,52,52,51,51,51,51,50,50,50,49,49,
    49,48,48,48,48,48,47,47,47,47,46,46,46,46,46,46,45,45,45,45,45,
    45,45,45,45,44,44,44,44,44,43,43,43,43,43,43,43,43,42,42,42,42,
    42,42,42,42,41,41,40,40,40,40,40,40,39,39,39,39,39,39,38,38,38,
    38,38,38,38,37,37,37,37,37,37,37,36,36,35,35,35,35,35,35,34,34,
    34,34,34,34,34,33,33,33,33,33,33,33,33,33,33,32,32,32,32,32,31,
    31,31,31,31,30,30,30,29,29,29,29,28,28,28,28,28,28,28,28,27,27,
    27,27,26,26,26,26,25,25,25,25,25,25,25,25,25,24,24,24,23,22,21,
    21,21,21,21,21,21,21,20,20,20,20,20,20,20,20,19,19,19,19,19,19,
    18,18,18,18,18,18,17,17,17,17,17,17,16,16,16,16,15,15,15,15,15,
    15,14,14,14,14,14,14,14,14,13,13,12,12,12,12,12,11,11,11,11,10,
    10,9,9,9,9,9,8,8,8,8,8,8,7,7,7,7,7,7,6,6,6,6,6,6,6,5,5,5,5,5,
    4,4,4,4,4,3,3,3,2,1
  };
  const int n4c1w1_s[] = {
    100, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,99,99,99,99,99,99,99,99,99,99,98,98,98,98,98,98,98,98,98,
    97,96,96,96,96,96,96,96,95,95,95,95,95,95,95,94,94,93,92,92,92,
    92,91,91,91,91,91,91,90,90,90,90,90,89,89,88,88,88,88,88,88,88,
    88,87,87,87,87,87,86,86,86,86,85,85,85,85,85,85,85,85,85,84,84,
    84,84,83,83,83,83,83,82,82,82,82,82,82,82,82,82,81,81,81,81,81,
    81,81,80,80,80,80,80,79,79,79,79,79,78,78,78,78,78,78,78,78,78,
    78,77,77,77,77,77,77,77,77,76,76,76,76,75,75,75,75,74,74,74,74,
    73,73,73,73,73,73,72,71,71,71,70,70,70,69,69,69,69,69,69,68,68,
    68,68,68,68,68,68,67,67,66,66,66,66,66,66,66,66,66,66,66,65,65,
    65,65,65,65,65,64,64,64,64,64,63,63,63,63,63,62,62,62,62,62,62,
    61,61,61,61,61,61,61,60,60,60,60,60,60,60,59,59,59,59,59,59,59,
    58,58,57,57,57,57,55,54,54,54,54,53,53,53,53,52,52,52,51,51,50,
    50,50,50,49,49,48,48,48,48,47,47,47,46,46,46,46,46,46,45,45,44,
    44,44,44,44,44,44,44,44,44,43,43,43,43,43,43,42,42,42,42,41,41,
    41,41,41,41,40,40,40,40,39,39,39,39,39,39,39,39,39,39,38,38,38,
    38,38,38,37,37,37,36,36,36,36,36,35,35,35,35,35,35,35,35,34,34,
    34,34,34,33,33,33,32,32,32,32,32,31,31,31,31,31,30,30,30,29,29,
    29,29,29,29,28,28,27,27,27,27,27,27,27,27,26,26,26,26,26,25,25,
    25,24,24,24,24,23,23,23,23,23,23,23,22,22,22,22,22,21,21,21,21,
    21,21,20,20,20,20,20,20,19,19,19,19,19,19,19,18,18,18,17,17,17,
    17,17,17,17,17,17,17,17,16,16,16,16,16,16,16,15,15,15,14,14,14,
    14,14,14,13,13,13,13,13,13,12,11,11,11,11,10,10,10,10,9,9,9,9,
    8,8,8,8,8,7,7,7,6,6,6,6,6,6,5,5,4,4,4,3,3,3,3,3,3,3,3,3,2,2,2,
    2,2,2,1,1,1,1
  };
  const int n4c1w1_t[] = {
    100, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,99,99,99,99,99,99,99,99,99,99,98,98,98,98,
    98,97,97,97,97,97,97,97,97,96,96,96,95,95,95,94,94,94,93,93,93,
    93,93,92,92,92,92,91,91,91,91,90,90,90,90,90,90,90,89,89,88,88,
    88,88,88,87,87,87,87,87,86,86,86,86,86,86,86,86,86,86,86,85,85,
    84,84,84,84,84,84,83,83,83,82,82,82,82,82,82,81,81,81,81,81,81,
    81,81,80,80,80,80,80,80,79,79,79,79,79,79,79,78,78,78,77,76,76,
    76,76,76,75,75,75,75,75,74,74,74,74,73,73,73,73,73,72,72,72,72,
    71,71,71,71,71,71,71,70,70,70,70,70,70,69,69,69,69,69,69,69,68,
    68,68,68,68,67,67,67,67,67,66,65,65,65,65,65,65,64,64,63,63,63,
    62,62,62,61,61,61,61,60,60,60,60,60,59,59,59,59,59,58,58,58,58,
    58,58,58,58,57,57,57,57,57,57,56,56,56,56,55,55,55,54,54,54,54,
    54,54,54,54,54,53,53,53,53,52,52,52,52,52,51,51,51,51,51,51,51,
    50,50,50,50,50,50,50,50,50,50,49,49,49,49,49,49,48,48,48,48,47,
    47,47,46,46,46,46,46,46,46,46,46,46,45,45,45,45,45,44,44,44,43,
    43,43,43,43,43,42,42,42,42,42,41,40,40,40,40,40,40,39,39,39,38,
    38,38,38,38,38,38,38,37,37,37,37,37,36,35,35,35,35,34,34,34,34,
    34,34,33,33,33,33,32,31,31,31,30,30,30,30,29,29,29,29,29,29,28,
    28,28,28,27,27,27,27,27,27,27,27,26,26,26,26,26,26,26,25,25,25,
    25,25,24,24,24,24,23,23,23,23,23,23,22,22,21,21,21,21,21,20,20,
    20,20,20,20,19,19,18,18,18,18,17,17,17,17,16,16,16,15,15,15,14,
    14,14,14,13,13,12,12,12,12,12,12,12,12,12,12,12,11,11,11,11,11,
    11,10,10,10,10,9,9,9,9,9,9,9,9,9,9,9,9,8,8,8,8,7,7,7,6,6,6,6,
    5,5,5,5,5,5,5,4,4,4,3,3,3,3,3,3,2,2,2,2,2,2,1,1,1,1,1,1,1,1,1,
    1,1
  };
  const int n4c1w2_a[] = {
    100, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,99,99,99,99,98,98,98,98,98,97,97,97,97,97,97,
    97,97,97,97,97,96,96,96,96,96,96,96,96,95,95,95,95,95,95,94,94,
    94,94,94,94,94,93,93,93,93,92,92,92,92,92,91,91,91,91,91,91,91,
    90,90,90,90,90,90,90,90,89,89,89,89,89,89,89,89,89,88,88,88,88,
    88,88,88,88,88,88,88,88,87,87,87,87,87,86,86,86,86,86,86,86,86,
    86,85,85,85,85,85,85,85,84,84,84,84,84,84,84,83,83,83,83,83,83,
    82,82,82,82,82,81,81,81,80,80,80,80,80,80,80,80,80,80,79,79,79,
    79,78,78,78,78,78,78,78,77,77,77,77,77,77,77,77,76,76,76,76,75,
    74,74,73,73,73,72,72,72,72,72,72,72,71,71,71,71,71,71,71,71,71,
    71,71,70,70,70,70,70,70,70,70,70,70,69,69,69,69,68,68,68,68,68,
    68,67,67,67,67,67,66,66,66,66,66,65,65,65,65,64,64,64,63,63,63,
    63,63,62,62,62,62,62,62,62,62,62,61,61,61,61,61,61,61,61,61,61,
    60,60,60,60,60,59,59,58,57,57,57,57,57,57,57,57,56,56,56,56,56,
    55,55,55,55,55,55,55,54,54,54,54,54,54,53,53,53,53,53,52,52,52,
    52,52,51,51,51,51,51,51,50,50,50,50,50,49,49,49,49,49,49,48,48,
    48,48,48,48,48,48,48,48,48,47,47,47,47,47,47,47,47,47,47,47,46,
    46,46,46,46,46,45,45,45,45,45,44,44,44,44,44,43,43,43,43,42,42,
    42,42,42,42,41,41,41,41,40,40,40,40,40,40,40,39,39,39,38,38,38,
    38,38,38,38,38,38,38,38,38,37,37,37,37,37,37,37,37,37,37,37,37,
    36,36,36,36,36,36,36,35,35,35,35,35,35,35,34,34,33,33,33,33,33,
    33,32,32,32,32,32,32,32,31,31,31,31,31,31,31,31,31,31,30,29,29,
    29,29,29,29,29,29,28,28,28,28,28,27,27,27,27,27,27,27,27,27,26,
    26,26,26,26,26,26,25,25,25,24,24,24,24,24,24,23,23,23,22,22,22,
    22,22,22,21,21,21,21,21,21,21,20,20,20,20,20,20,20,20,20
  };
  const int n4c1w2_b[] = {
    100, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,100,100,100,100,100,100,99,99,99,98,98,98,
    98,97,97,97,97,97,97,97,97,96,96,96,96,96,96,96,95,95,95,95,94,
    94,94,94,93,93,93,93,93,92,92,92,92,92,92,91,91,91,91,91,91,91,
    90,90,90,90,90,90,90,90,90,90,90,89,89,89,89,88,88,88,88,88,87,
    87,87,87,87,87,87,87,86,86,86,86,85,85,85,85,85,85,84,84,84,84,
    83,83,83,83,82,82,82,82,82,82,82,81,81,81,80,80,80,80,80,80,80,
    80,80,80,80,79,79,79,79,79,79,79,79,78,78,78,78,78,77,77,77,77,
    77,77,77,77,77,76,76,76,76,76,76,75,75,75,75,75,74,74,74,74,74,
    74,74,74,74,74,74,74,73,73,73,73,73,72,72,72,72,72,72,72,72,72,
    72,72,72,71,71,71,71,71,71,71,70,70,70,70,70,69,69,69,69,69,69,
    68,68,68,68,68,68,67,67,67,67,67,67,67,67,67,67,66,66,66,66,66,
    65,65,65,65,65,65,64,64,64,64,63,63,63,63,63,62,62,62,62,62,62,
    62,61,61,61,61,61,61,61,60,60,60,60,60,60,59,59,59,59,59,59,59,
    59,59,58,58,58,58,58,58,58,57,57,57,57,57,57,57,56,56,56,56,56,
    56,56,56,55,55,55,55,55,55,54,54,54,54,54,54,54,54,53,53,53,53,
    53,52,52,52,52,52,51,51,51,51,51,51,51,51,50,50,50,50,50,50,49,
    49,48,48,48,48,48,48,47,47,47,47,47,47,47,47,47,46,46,46,46,46,
    46,46,46,45,45,45,44,44,44,44,43,43,43,43,43,43,43,43,42,42,42,
    42,42,41,41,41,41,41,40,40,40,40,40,40,39,39,39,39,39,39,39,39,
    39,38,38,37,37,37,37,36,36,36,36,36,35,35,35,35,35,35,35,35,34,
    34,34,34,33,33,33,33,33,33,33,32,32,32,32,32,31,31,31,31,31,31,
    30,30,30,30,30,30,30,30,30,29,29,29,28,28,28,28,28,28,28,28,28,
    28,28,27,27,27,27,27,26,26,26,25,25,25,25,25,25,25,25,25,25,24,
    24,24,24,24,24,23,23,23,23,23,23,22,22,22,21,20,20,20,20,20,20
  };
  const int n4c1w2_c[] = {
    100, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,99,99,99,99,99,99,99,98,98,98,98,98,98,98,
    97,97,97,96,96,96,96,95,95,95,95,94,94,93,93,93,93,93,93,93,93,
    93,92,92,92,92,92,91,91,91,91,91,91,91,90,90,90,90,90,89,89,89,
    89,89,89,89,89,88,88,88,87,87,86,86,86,86,86,86,86,86,86,86,85,
    85,85,85,85,85,85,85,85,84,84,83,83,83,83,83,82,82,82,82,82,82,
    82,81,81,80,80,80,80,80,80,80,80,80,80,79,79,79,79,79,79,79,79,
    79,79,78,78,78,78,78,78,78,78,78,78,78,78,78,77,77,77,77,77,77,
    77,76,76,76,76,76,76,76,76,76,76,76,75,75,75,75,75,75,74,74,74,
    74,74,74,74,74,74,73,73,73,73,73,72,72,72,71,71,71,71,71,70,70,
    70,70,70,70,69,68,68,67,67,67,67,67,67,66,66,66,66,66,66,66,66,
    66,66,66,66,65,65,65,65,65,65,65,64,64,64,64,64,64,63,63,63,63,
    62,62,62,62,62,61,61,61,60,60,60,60,60,60,60,59,59,59,59,59,59,
    59,59,58,58,58,58,58,58,58,57,57,57,57,57,57,57,57,56,56,56,56,
    56,56,55,55,55,55,54,54,54,54,54,54,54,54,53,53,53,53,53,53,52,
    52,52,52,52,52,52,52,52,52,52,52,52,52,51,51,51,51,50,50,50,50,
    50,50,49,49,49,49,49,49,49,49,49,49,48,48,47,47,47,47,47,47,47,
    46,46,46,46,46,46,46,45,45,45,45,44,44,44,44,44,44,43,43,43,43,
    42,42,42,42,42,41,41,41,41,41,41,41,41,40,40,40,40,40,40,40,40,
    40,40,39,39,39,39,39,39,38,38,38,38,37,37,37,37,37,37,37,37,37,
    36,36,36,36,36,36,36,36,36,36,35,35,35,35,35,35,35,34,34,34,34,
    34,34,34,33,33,33,33,33,32,32,32,31,31,31,31,31,31,31,31,31,31,
    31,30,30,30,30,30,30,30,30,30,30,29,29,29,29,29,29,28,28,28,28,
    28,28,27,27,27,27,27,27,27,27,26,26,26,26,25,25,25,24,24,24,24,
    24,24,23,23,23,23,23,23,22,22,22,21,21,21,21,20,20,20,20
  };
  const int n4c1w2_d[] = {
    100, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,99,99,99,99,99,99,99,99,99,99,98,98,98,98,
    98,97,97,97,97,97,97,96,96,96,96,96,96,95,95,95,95,95,95,94,94,
    94,94,94,94,94,94,93,93,93,92,92,92,92,92,92,92,91,91,91,91,91,
    91,91,90,90,90,90,90,89,89,89,89,89,88,88,88,88,87,87,87,87,87,
    86,86,86,86,86,86,86,86,85,85,85,85,85,85,84,84,84,84,84,84,84,
    84,84,84,83,83,83,83,83,83,83,83,82,82,82,82,82,82,82,82,82,81,
    81,81,81,81,80,80,80,80,80,79,79,79,79,79,79,79,79,79,79,78,78,
    78,77,77,77,77,77,77,77,77,76,76,76,76,76,76,76,75,75,75,75,75,
    75,75,75,75,74,74,74,74,74,73,73,73,73,73,72,72,72,72,72,72,71,
    71,70,70,70,70,69,69,69,69,69,69,69,69,69,69,69,68,68,68,68,68,
    67,67,67,67,66,66,66,66,66,65,65,65,65,65,65,65,65,65,65,64,64,
    64,64,64,64,64,63,63,63,63,63,63,63,63,62,62,62,62,62,62,62,61,
    61,61,61,61,61,61,61,61,61,60,60,60,60,60,60,59,59,59,59,59,59,
    59,59,59,58,58,58,58,58,58,58,58,57,57,57,57,57,57,57,57,56,56,
    56,56,55,55,55,55,55,55,54,54,54,54,54,54,54,54,53,53,53,53,53,
    52,52,52,52,51,51,51,51,50,50,49,49,49,49,49,49,49,49,49,48,48,
    48,48,47,47,47,47,47,47,47,47,47,47,46,46,46,46,45,45,45,45,45,
    45,44,44,43,43,43,43,43,43,43,43,42,42,41,41,41,41,41,40,40,40,
    40,40,40,39,39,39,39,38,38,38,37,37,37,37,37,37,37,36,36,36,36,
    36,36,36,36,36,36,36,36,35,35,35,34,34,34,34,34,33,33,32,32,32,
    32,32,32,32,31,31,31,30,30,30,30,29,29,29,29,29,29,29,29,29,28,
    28,28,28,28,28,28,28,28,28,27,27,27,27,27,27,26,26,26,26,26,26,
    26,26,26,25,25,25,25,25,24,24,24,24,24,23,23,23,22,22,22,22,22,
    22,22,22,22,21,21,21,21,21,20,20,20,20,20,20,20,20,20,20
  };
  const int n4c1w2_e[] = {
    100, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,99,99,99,99,99,99,98,98,98,98,98,98,98,98,
    98,98,97,97,97,97,97,97,97,97,97,96,96,96,96,96,96,96,96,95,95,
    95,95,95,94,94,94,94,93,93,93,93,93,92,92,92,92,92,92,91,91,91,
    91,91,91,90,90,90,90,90,90,90,90,90,89,89,89,88,88,88,88,87,87,
    87,87,87,87,86,86,86,86,85,85,85,85,85,85,84,84,84,83,83,83,83,
    82,82,82,82,82,82,82,81,81,81,81,81,81,81,81,81,81,81,81,81,81,
    81,81,81,80,80,80,80,79,79,79,78,78,78,78,77,77,77,77,76,76,76,
    76,75,75,75,75,75,74,74,74,74,74,74,74,74,73,73,73,73,73,73,72,
    72,72,72,72,71,71,71,71,70,70,70,70,70,69,69,69,69,69,69,68,68,
    68,68,68,68,68,68,68,68,68,67,67,67,67,67,66,66,66,66,65,65,65,
    65,65,65,64,64,64,64,64,64,64,64,64,64,64,64,64,64,63,63,63,63,
    63,62,62,62,62,61,61,61,61,61,61,61,61,61,60,60,59,59,59,59,59,
    58,58,58,58,58,58,57,57,57,57,57,56,56,56,56,55,55,55,55,55,55,
    55,54,54,54,53,53,53,53,53,53,53,53,52,52,52,52,52,52,52,52,52,
    52,51,51,51,51,51,51,50,50,50,50,50,50,50,49,49,48,48,48,48,48,
    48,47,47,47,47,47,47,47,46,46,46,46,46,46,46,45,45,45,45,45,45,
    45,45,45,45,44,43,43,43,43,43,43,43,43,42,42,42,42,42,42,42,41,
    41,41,41,41,41,41,41,40,40,40,40,40,40,40,40,40,39,39,39,39,39,
    39,39,39,38,38,38,37,37,37,37,37,37,37,37,37,36,36,36,36,36,36,
    35,35,35,35,35,35,35,35,35,34,33,33,33,33,33,33,33,33,33,33,32,
    32,32,32,31,31,31,31,31,31,31,31,31,31,30,30,30,30,30,30,30,29,
    29,28,28,28,28,27,27,27,27,27,27,27,27,26,26,26,26,26,25,25,25,
    25,25,25,25,25,25,24,24,24,24,24,24,23,23,23,23,23,23,23,23,22,
    22,22,22,22,22,22,21,21,21,21,21,20,20,20,20,20,20,20,20
  };
  const int n4c1w2_f[] = {
    100, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,100,100,99,99,99,99,99,99,99,98,98,98,98,
    98,98,97,97,97,97,97,97,97,96,96,96,96,96,96,96,95,95,95,95,94,
    94,94,94,93,93,93,93,93,93,93,93,93,92,92,92,92,92,92,92,92,92,
    91,91,91,91,91,91,90,90,90,90,89,89,89,89,89,89,89,88,88,88,88,
    88,88,88,87,87,87,87,87,86,86,86,86,86,86,86,86,85,85,85,85,85,
    85,84,84,84,84,84,84,84,84,83,83,83,82,82,82,82,82,82,81,81,80,
    80,80,80,79,79,79,79,79,79,79,79,79,79,78,78,78,78,77,77,76,76,
    76,76,76,76,76,76,76,76,75,75,75,75,75,75,75,74,74,74,74,74,74,
    74,73,73,73,73,73,72,72,72,72,72,72,72,72,71,71,71,71,71,71,71,
    70,70,70,70,70,70,70,69,69,69,69,69,69,69,69,68,68,68,68,68,67,
    67,67,67,67,66,66,66,66,66,66,66,65,65,65,65,65,65,65,64,64,64,
    64,64,64,64,64,64,64,63,63,63,63,63,63,62,62,62,62,62,61,61,61,
    61,61,61,60,60,60,60,60,60,60,59,59,59,59,59,59,59,59,59,59,58,
    58,58,57,57,57,57,56,56,56,56,56,56,55,55,55,55,55,55,55,55,55,
    55,55,55,54,54,54,54,53,53,53,53,53,53,52,52,52,52,51,51,51,51,
    51,51,51,51,51,51,50,50,50,50,50,49,49,49,48,48,48,48,48,48,47,
    47,47,47,47,46,46,46,46,46,45,45,45,45,44,44,44,44,43,43,43,43,
    43,43,43,43,43,43,42,42,42,42,42,42,42,42,42,41,41,41,41,41,41,
    41,40,40,40,40,40,40,40,40,39,39,39,39,39,38,38,38,38,38,38,38,
    38,37,37,37,37,37,37,37,37,37,36,36,36,35,35,35,35,35,35,34,34,
    33,33,33,33,33,33,33,33,33,32,32,32,32,32,31,31,31,31,31,31,31,
    31,31,31,31,31,31,30,30,30,30,30,30,30,29,29,29,28,28,28,28,28,
    28,27,27,27,26,26,26,26,26,25,25,25,25,24,24,24,24,24,23,23,23,
    23,23,22,22,22,22,21,21,21,20,20,20,20,20,20,20,20,20,20,20
  };
  const int n4c1w2_g[] = {
    100, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,99,99,99,99,99,99,99,99,98,98,98,98,98,98,97,
    97,97,97,97,97,96,96,96,96,96,95,95,95,95,95,95,95,95,95,94,94,
    94,94,94,94,94,93,93,93,93,93,93,92,92,92,92,92,92,91,91,91,90,
    90,90,90,90,90,89,89,89,89,89,89,89,88,88,88,88,88,87,87,87,86,
    86,86,86,85,85,85,85,85,85,84,84,84,83,83,82,82,82,82,82,82,82,
    82,81,81,81,81,81,81,81,81,80,80,80,80,80,80,80,80,80,80,80,79,
    79,79,79,79,78,78,78,78,78,78,77,77,76,76,76,76,76,76,76,75,75,
    75,75,75,75,75,75,74,74,74,74,74,74,74,73,73,73,73,73,73,72,72,
    72,72,72,72,72,72,71,71,71,71,70,70,70,70,70,70,70,69,69,69,68,
    68,68,68,67,67,67,67,66,66,66,66,66,66,66,66,66,65,65,65,65,65,
    65,65,64,64,64,64,64,63,63,63,63,62,62,62,62,62,62,61,61,61,61,
    61,61,60,60,60,59,59,59,59,59,59,59,59,58,58,58,58,58,58,58,57,
    57,57,56,56,56,56,56,55,55,55,55,55,55,55,55,55,55,55,54,54,54,
    54,54,54,53,53,53,53,53,53,52,52,52,52,52,52,51,51,51,51,50,50,
    50,50,50,50,49,49,49,49,49,49,49,49,49,48,48,48,48,48,48,48,48,
    48,47,47,46,46,46,46,45,45,45,45,45,45,45,45,45,45,44,44,44,44,
    44,44,44,43,43,43,43,43,43,42,42,42,42,42,42,42,42,41,41,41,41,
    41,41,41,41,41,40,40,40,40,40,40,40,40,39,39,39,39,38,38,38,38,
    38,38,37,37,37,37,37,37,37,37,36,36,36,36,36,36,36,35,35,35,35,
    35,35,35,35,35,35,35,34,34,34,34,34,34,34,34,34,34,34,33,33,33,
    33,33,33,33,33,33,33,32,32,32,31,31,31,31,31,31,30,30,30,30,30,
    30,30,29,29,29,29,29,29,29,29,28,28,28,28,28,28,27,27,27,26,26,
    26,26,26,26,26,25,25,25,25,25,25,24,24,24,24,24,23,23,23,22,22,
    22,22,22,21,21,21,21,21,21,21,21,21,20,20,20,20,20,20,20
  };
  const int n4c1w2_h[] = {
    100, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,99,99,99,98,98,98,97,97,97,97,97,97,96,96,
    96,96,96,96,95,95,95,95,95,95,95,95,95,95,94,94,94,94,94,94,94,
    94,94,93,93,93,93,92,92,92,92,92,92,91,91,91,91,91,90,90,90,90,
    90,89,89,89,89,89,88,88,88,88,88,87,87,87,87,86,86,86,86,85,85,
    85,85,85,85,84,84,84,84,84,84,83,83,83,83,83,83,83,83,82,82,82,
    82,82,82,82,82,81,81,81,80,80,80,80,80,80,80,79,79,79,79,78,78,
    78,77,77,77,77,77,77,77,77,77,76,76,76,76,76,76,76,75,75,75,75,
    75,75,75,75,74,74,74,74,74,73,73,73,73,73,72,72,72,72,72,72,71,
    71,71,71,71,70,70,70,70,70,70,70,70,69,69,69,69,69,69,69,69,68,
    68,68,68,68,68,68,68,68,68,67,67,67,67,67,67,67,67,66,66,66,66,
    66,66,66,65,65,65,65,65,65,64,64,64,64,64,64,64,64,63,63,63,63,
    63,63,62,62,62,62,62,62,61,61,61,61,61,60,60,60,60,60,60,59,59,
    59,59,59,59,59,59,59,58,58,58,58,58,58,58,58,58,58,58,57,57,56,
    56,56,56,56,56,56,56,56,55,55,55,55,55,55,54,54,53,53,53,53,53,
    53,52,52,52,52,52,52,51,51,51,51,51,51,51,50,50,50,50,49,49,49,
    49,49,49,49,49,49,49,49,49,48,48,48,48,48,48,48,48,48,47,47,47,
    46,46,46,46,46,46,46,46,45,45,45,45,45,45,45,44,44,44,44,44,44,
    44,44,43,43,43,43,42,42,42,42,42,41,41,41,41,41,41,41,41,41,41,
    41,40,40,39,39,39,39,39,39,39,38,38,38,38,38,38,37,37,37,37,37,
    37,37,36,36,36,36,35,35,35,35,35,34,34,34,34,33,33,33,33,33,33,
    33,33,33,33,33,32,32,32,32,32,32,31,31,31,31,31,30,30,30,30,30,
    30,30,30,30,30,29,29,29,29,29,29,28,28,28,27,27,27,27,27,27,26,
    26,26,26,26,26,26,26,26,25,25,25,25,25,24,24,24,23,23,23,23,23,
    22,22,22,22,22,22,22,21,21,21,20,20,20,20,20,20,20,20,20
  };
  const int n4c1w2_i[] = {
    100, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,99,99,99,98,98,98,98,97,97,97,97,96,96,96,
    96,96,96,96,96,96,95,95,95,95,95,94,94,94,94,94,94,94,93,93,93,
    93,93,92,92,92,92,92,92,92,91,91,91,91,91,90,90,90,90,89,89,89,
    89,89,89,89,89,89,89,89,89,89,89,88,88,87,87,87,87,87,86,86,86,
    86,86,86,86,86,86,85,85,85,85,85,84,84,84,84,84,83,83,83,83,82,
    82,82,82,82,82,82,82,81,81,81,81,81,81,80,80,80,80,80,79,79,79,
    79,79,79,79,79,79,79,78,78,78,78,77,77,77,77,77,76,76,76,76,76,
    75,75,75,75,75,75,75,75,75,75,75,74,74,74,74,73,73,73,73,73,73,
    73,72,72,72,72,72,72,72,71,71,71,71,71,71,71,71,70,70,70,70,69,
    69,69,69,69,69,69,68,68,68,68,67,67,67,66,66,66,66,66,66,65,65,
    64,64,64,64,64,64,64,64,63,63,63,63,63,63,63,63,63,62,62,62,62,
    61,61,61,61,61,61,60,60,60,60,59,59,59,59,59,59,59,58,58,58,58,
    57,57,57,57,57,56,56,56,56,56,56,56,56,56,55,55,55,55,55,55,54,
    54,54,54,54,53,53,53,53,53,53,53,53,52,52,52,52,52,52,52,51,51,
    50,50,50,50,49,49,49,49,49,49,48,48,48,48,48,47,47,46,46,46,46,
    46,46,46,45,45,45,45,45,45,45,45,45,44,44,44,43,43,43,43,43,43,
    43,43,43,43,42,42,42,42,41,41,41,41,40,39,39,39,39,39,39,39,39,
    39,38,38,38,38,37,37,37,37,37,37,37,37,37,37,36,36,36,36,35,35,
    35,35,35,35,35,35,35,35,34,34,34,34,34,34,34,34,34,33,33,33,33,
    33,33,33,33,33,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,31,
    31,31,31,31,31,30,30,30,30,30,29,29,29,28,28,28,28,28,28,27,27,
    27,27,27,27,26,26,26,26,26,26,26,25,25,25,25,25,25,25,25,25,25,
    25,25,25,24,24,24,24,24,24,24,24,24,24,23,23,23,23,23,23,23,23,
    22,22,22,22,22,22,22,21,21,21,21,21,20,20,20,20,20,20,20
  };
  const int n4c1w2_j[] = {
    100, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,99,99,99,99,99,99,99,98,98,98,98,98,97,97,97,
    97,97,97,96,96,96,96,96,96,96,96,96,95,95,95,95,95,95,95,95,95,
    95,94,94,94,94,94,94,93,93,93,93,93,93,93,92,92,92,92,92,92,91,
    91,91,91,91,91,91,90,90,90,90,90,90,89,88,88,88,88,88,88,88,87,
    87,87,87,87,87,87,86,86,86,86,86,85,85,85,85,84,84,84,84,84,83,
    83,83,83,83,82,82,82,82,82,82,82,82,82,82,82,82,81,81,81,81,81,
    80,80,80,80,80,79,79,79,79,79,79,79,79,79,79,79,78,78,78,78,77,
    77,77,77,77,77,77,76,76,76,76,76,76,75,75,75,75,74,74,74,74,74,
    73,73,73,73,73,73,73,73,72,72,72,71,71,71,71,71,71,71,71,70,70,
    70,70,69,69,69,69,68,68,68,68,68,68,68,67,67,67,67,67,66,66,66,
    66,66,65,65,65,65,65,65,65,65,65,64,64,64,64,64,64,64,64,64,64,
    64,63,63,63,62,62,62,62,62,62,62,61,61,61,60,60,60,60,60,59,59,
    59,59,59,59,58,58,58,58,57,57,57,57,57,56,56,56,56,56,56,56,55,
    54,54,54,54,54,54,54,54,54,54,53,53,53,53,53,53,53,52,52,52,52,
    52,51,51,51,51,51,51,51,50,50,50,50,50,50,49,49,49,49,48,48,48,
    47,47,47,46,46,46,46,46,46,46,46,45,45,45,45,45,45,44,43,43,43,
    43,43,43,43,42,42,42,42,42,41,41,41,41,41,41,41,41,41,41,41,40,
    40,40,40,40,40,40,40,40,39,39,39,39,39,39,39,39,39,39,39,38,38,
    38,38,38,38,38,37,37,37,37,37,36,36,36,36,36,36,35,35,35,35,35,
    34,34,34,34,34,34,33,33,33,33,33,32,32,32,32,32,32,32,31,31,31,
    31,31,31,31,31,31,31,31,31,31,31,30,30,30,30,30,30,30,30,30,29,
    29,29,29,29,29,28,28,28,28,28,28,28,28,28,28,27,27,27,27,26,26,
    26,26,26,25,25,25,25,25,25,25,25,25,24,24,24,24,24,24,23,23,22,
    22,22,22,22,22,22,22,21,21,21,21,21,20,20,20,20,20,20,20
  };
  const int n4c1w2_k[] = {
    100, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,100,100,100,99,99,99,99,99,99,99,99,97,97,
    97,97,97,97,97,96,96,96,96,96,96,96,96,95,95,95,95,95,95,95,94,
    93,93,93,93,93,92,92,92,92,92,92,91,91,91,91,90,90,90,90,90,90,
    89,89,89,89,89,89,89,89,88,88,88,88,88,88,88,88,88,88,87,87,87,
    87,87,87,86,86,86,86,86,86,86,85,85,84,84,84,84,84,84,84,84,83,
    83,83,83,82,82,82,82,82,82,82,82,81,81,81,81,81,81,80,80,80,80,
    80,80,79,79,79,79,79,78,78,78,78,78,78,77,77,77,77,77,77,77,77,
    76,76,76,76,75,75,75,75,75,75,75,75,74,74,74,74,74,74,73,73,73,
    73,73,73,73,73,73,72,72,72,71,71,71,71,71,71,71,71,70,70,70,70,
    70,70,69,69,69,69,69,69,69,69,68,68,68,68,68,67,67,67,67,67,67,
    67,67,67,66,66,66,66,66,66,65,65,65,65,65,65,65,64,64,64,64,64,
    63,63,63,63,63,63,62,62,61,61,61,61,61,61,61,60,60,60,60,60,60,
    59,59,59,59,59,59,59,59,58,58,58,58,58,58,58,57,57,57,57,57,57,
    56,56,56,56,55,55,55,55,55,55,55,55,55,54,54,54,54,53,53,53,52,
    52,52,52,52,52,51,51,51,51,51,51,51,51,51,51,50,50,50,49,49,48,
    48,48,48,47,47,47,47,47,47,46,46,46,45,45,45,45,45,45,45,45,45,
    44,44,44,44,44,44,44,43,43,43,43,43,43,43,43,43,42,42,42,42,41,
    41,41,41,41,40,40,40,39,39,39,39,39,39,38,38,38,38,38,38,38,38,
    37,37,37,36,36,36,36,36,36,36,36,35,35,35,35,35,35,35,35,35,35,
    34,34,34,34,34,33,33,33,33,33,33,33,33,33,33,33,33,33,32,32,32,
    32,32,32,32,32,31,31,31,31,31,30,30,30,30,30,30,30,30,30,30,30,
    29,29,29,29,29,29,29,29,29,29,29,28,28,28,28,27,27,27,26,26,25,
    25,25,25,25,25,25,25,25,25,24,24,24,24,24,24,24,24,24,24,24,23,
    23,23,23,22,22,22,22,22,22,22,22,22,21,21,21,21,20,20,20,20
  };
  const int n4c1w2_l[] = {
    100, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,100,100,100,100,100,99,99,99,99,99,99,98,
    98,98,98,98,98,97,97,97,97,97,97,97,96,96,96,96,96,96,95,95,95,
    95,95,95,95,95,94,94,94,93,93,93,92,92,92,91,91,91,91,91,91,90,
    90,90,90,89,89,89,89,89,89,89,89,88,88,88,88,88,88,88,88,88,88,
    87,87,87,86,86,86,86,86,86,86,85,85,85,85,85,85,85,84,84,84,84,
    84,84,84,84,84,83,83,83,83,83,83,83,83,83,83,82,82,82,82,82,82,
    81,81,81,81,81,81,81,80,80,80,79,79,78,78,78,78,78,78,78,78,78,
    77,77,77,77,77,77,77,77,77,76,76,76,75,75,74,74,74,74,74,74,73,
    73,73,73,73,73,72,72,72,72,71,71,71,71,71,71,71,70,70,70,70,69,
    69,69,69,69,69,69,69,68,68,68,68,68,68,68,68,67,67,67,66,66,66,
    66,65,65,65,65,65,65,65,65,64,64,64,64,64,63,63,63,63,63,63,62,
    62,62,62,62,62,62,62,62,61,61,61,61,61,61,61,60,60,60,60,60,60,
    60,60,60,59,59,59,59,58,58,58,58,58,58,58,58,57,57,57,57,57,57,
    57,57,57,57,56,56,56,56,56,56,56,56,56,56,56,56,55,55,55,54,54,
    54,53,53,53,53,53,53,52,52,52,52,52,52,52,52,51,51,51,51,51,51,
    50,50,50,50,50,50,49,49,49,49,49,49,48,48,48,48,48,47,47,47,47,
    47,47,46,46,46,46,46,46,45,45,45,45,45,45,45,44,44,44,44,44,44,
    43,43,43,43,43,43,43,43,43,42,42,42,42,42,42,42,42,41,41,40,40,
    40,39,39,39,39,39,39,39,39,39,38,38,38,38,38,38,38,38,38,37,37,
    37,36,36,36,36,36,35,35,35,35,35,35,34,34,34,34,34,33,33,33,33,
    33,33,33,33,32,32,31,31,31,31,30,30,30,30,30,29,29,29,29,29,29,
    29,28,28,28,28,28,27,27,27,27,27,27,27,27,26,26,26,26,26,26,25,
    25,25,25,25,25,25,24,24,24,24,24,24,23,23,23,23,23,23,23,22,22,
    22,22,22,22,22,22,22,21,21,21,21,21,21,21,21,20,20,20,20,20,20
  };
  const int n4c1w2_m[] = {
    100, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,100,100,100,99,99,99,99,99,99,99,99,99,98,
    98,98,98,98,98,98,98,97,97,97,97,97,97,97,97,97,97,97,96,96,96,
    96,95,95,95,95,95,95,95,95,94,94,94,94,94,94,94,93,93,92,92,92,
    92,92,91,91,91,91,91,91,91,90,90,90,90,89,89,89,89,88,88,88,88,
    88,87,87,87,87,86,86,86,86,85,85,85,85,85,84,84,84,83,83,83,83,
    83,83,83,82,82,82,82,82,82,81,81,81,81,81,81,81,81,81,80,80,80,
    80,80,79,79,79,79,79,79,79,79,79,78,78,78,78,78,78,78,78,78,78,
    78,77,77,77,77,77,77,77,77,76,76,76,76,76,76,76,75,75,75,75,74,
    74,74,74,74,73,73,73,73,73,72,72,72,72,72,72,72,72,72,72,71,71,
    71,71,71,70,70,70,70,70,69,69,69,69,69,69,69,68,68,68,68,68,68,
    68,68,68,68,68,68,67,67,67,67,67,66,66,66,66,66,66,66,66,65,65,
    65,65,65,65,65,64,64,64,64,64,64,63,63,63,63,63,63,63,63,63,63,
    62,62,62,62,62,62,62,61,61,61,61,61,61,61,61,61,60,60,60,60,60,
    59,59,59,59,59,59,59,59,58,58,58,58,57,57,57,57,56,56,56,56,56,
    56,55,55,55,54,54,53,53,53,53,53,53,53,53,53,53,52,52,52,52,52,
    51,51,51,51,50,50,50,50,50,50,49,49,49,49,48,48,48,48,48,48,47,
    47,47,47,47,47,47,46,46,46,46,46,46,46,46,46,46,46,46,46,45,45,
    45,45,45,45,44,44,44,44,44,44,44,44,43,43,43,43,43,43,43,42,42,
    42,42,42,42,42,42,41,41,41,40,40,40,40,40,39,39,39,39,38,38,38,
    37,37,37,37,37,36,36,36,36,36,36,35,35,35,35,35,35,35,34,34,34,
    33,33,33,33,33,33,32,32,32,32,32,32,31,31,31,31,31,31,31,30,30,
    30,30,30,30,30,30,30,30,30,30,29,29,29,29,28,28,28,28,28,28,28,
    28,28,27,27,27,27,27,27,26,26,25,25,25,25,24,24,24,24,24,24,24,
    23,23,23,22,22,22,22,22,22,22,21,21,21,21,21,21,20,20,20,20
  };
  const int n4c1w2_n[] = {
    100, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,100,100,100,100,100,99,99,99,99,99,98,98,
    98,98,98,98,98,97,97,97,97,97,97,96,96,96,96,96,96,96,95,95,95,
    95,95,95,94,94,94,94,94,94,94,94,94,93,93,93,93,93,93,93,92,92,
    92,92,92,92,92,92,91,91,91,91,91,91,91,90,90,90,90,90,90,90,90,
    89,89,89,89,89,89,89,89,89,88,88,88,88,88,88,88,87,87,87,87,87,
    87,86,86,86,86,85,85,85,85,85,85,85,85,85,84,84,84,84,84,83,83,
    83,83,82,81,81,81,81,81,80,80,80,80,79,79,79,79,79,79,79,78,78,
    78,78,78,77,77,77,77,76,76,76,76,76,76,75,75,75,75,74,74,73,73,
    73,73,73,72,72,72,72,72,72,71,71,71,71,71,70,70,70,70,70,70,69,
    69,69,69,69,68,68,68,68,68,68,68,67,67,67,66,66,66,66,66,66,66,
    66,65,65,64,64,63,63,63,63,63,63,63,63,63,63,63,62,62,62,62,61,
    61,61,61,61,61,61,60,60,60,60,59,59,59,59,59,58,58,58,58,58,57,
    57,57,57,57,57,56,56,56,56,56,56,56,56,55,55,55,55,55,55,55,55,
    54,54,54,54,54,54,53,53,53,53,53,53,53,53,53,52,52,52,52,52,52,
    52,52,52,52,51,51,51,51,51,51,51,50,50,50,50,50,50,50,50,49,49,
    49,49,49,49,49,49,48,48,48,48,47,47,46,46,46,45,45,45,45,44,44,
    44,44,44,44,44,44,44,44,44,43,43,43,42,42,42,42,42,42,42,41,41,
    41,41,41,41,41,41,40,40,40,40,40,40,40,40,39,39,39,39,39,39,38,
    38,38,38,38,38,38,38,38,38,37,37,37,37,37,37,37,36,36,36,36,36,
    35,35,35,34,34,34,34,34,34,34,34,34,33,33,33,33,33,33,33,33,32,
    32,32,32,32,32,32,32,31,31,31,31,31,31,31,31,30,30,30,30,30,30,
    30,30,30,30,29,29,29,29,29,29,29,28,28,28,28,27,27,27,26,26,26,
    26,26,26,26,25,25,25,25,25,24,24,24,24,24,23,23,23,23,23,22,22,
    22,22,21,21,21,21,21,21,21,21,21,20,20,20,20,20,20,20,20,20,20
  };
  const int n4c1w2_o[] = {
    100, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,100,100,99,99,99,99,99,99,98,98,98,98,98,
    98,97,97,97,97,97,97,96,96,96,96,96,96,96,95,95,95,95,95,95,95,
    95,94,94,94,94,93,93,93,93,93,92,92,91,91,91,91,91,91,91,90,90,
    90,89,89,89,89,89,89,89,88,88,88,88,88,88,88,88,88,88,87,87,87,
    87,87,86,86,86,86,86,86,86,86,86,86,86,86,85,85,85,85,85,85,84,
    84,84,84,84,84,84,84,84,84,84,84,83,83,83,83,83,83,83,82,82,82,
    82,81,81,81,81,81,81,81,81,81,81,80,80,80,80,80,80,79,79,79,78,
    78,78,78,78,78,77,77,77,77,77,77,76,76,76,76,76,76,76,75,75,75,
    75,75,75,74,74,74,74,74,74,74,74,74,73,73,73,73,73,73,72,72,72,
    71,71,71,71,71,71,71,71,71,69,69,68,68,68,68,68,68,68,68,68,67,
    67,66,66,66,65,65,65,65,65,64,64,64,64,64,64,64,64,63,63,63,63,
    63,63,62,62,62,62,62,62,62,61,61,61,61,61,61,61,61,60,60,60,60,
    60,60,60,60,60,59,59,59,59,58,58,58,58,58,58,57,57,57,57,57,57,
    56,56,56,56,56,56,56,56,55,55,55,55,55,55,55,55,55,54,54,54,54,
    53,53,53,53,53,53,53,52,52,52,52,52,52,52,52,51,51,51,51,51,50,
    50,50,50,50,50,49,49,49,49,49,48,48,48,48,48,47,47,47,47,47,47,
    47,47,47,47,47,47,47,46,46,46,46,46,46,45,45,45,45,45,45,45,44,
    44,44,44,44,44,44,43,43,43,42,42,42,42,42,42,42,41,40,40,40,40,
    40,40,39,39,39,39,39,39,38,38,38,38,38,37,37,37,37,37,36,36,36,
    36,36,36,35,35,35,35,34,34,34,34,33,33,33,32,32,32,32,32,32,32,
    32,31,31,31,31,31,31,31,31,31,31,30,30,30,30,30,30,30,30,29,29,
    29,29,29,29,29,29,29,29,29,29,28,28,28,28,28,28,27,27,27,27,27,
    27,27,26,26,26,26,26,25,25,25,25,25,25,25,25,24,24,24,24,24,23,
    23,23,23,22,22,22,22,21,21,21,21,21,21,21,21,20,20,20,20,20
  };
  const int n4c1w2_p[] = {
    100, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,99,99,99,99,98,98,98,98,98,98,98,98,98,98,97,
    97,97,97,96,96,96,96,95,95,95,95,95,95,94,94,94,94,94,94,94,94,
    94,94,94,94,93,93,93,93,93,93,93,93,93,92,92,92,92,92,92,92,91,
    91,91,91,90,90,90,90,89,89,89,89,89,89,88,88,88,87,87,87,87,86,
    86,86,86,85,85,84,84,84,84,84,84,84,84,84,84,83,83,83,83,83,83,
    83,83,82,82,82,82,82,82,81,81,81,81,81,81,80,80,80,80,80,80,80,
    80,79,79,79,79,79,78,78,78,77,77,77,77,77,77,77,76,76,76,76,76,
    75,75,75,75,74,74,74,74,74,73,73,73,73,73,73,73,73,73,73,72,72,
    72,72,72,72,72,72,72,71,71,71,71,71,71,71,70,70,70,70,70,70,70,
    70,70,69,69,69,69,69,69,69,69,69,69,69,68,68,68,68,68,67,67,67,
    67,67,67,66,66,66,66,65,65,65,65,65,65,65,64,64,64,64,64,63,63,
    63,63,63,63,62,62,62,62,62,62,62,61,61,61,61,61,61,61,61,61,60,
    60,60,60,60,60,59,59,59,59,59,58,58,58,58,58,58,57,57,57,57,57,
    57,57,57,57,56,56,56,56,56,56,55,55,54,54,54,54,54,54,54,54,54,
    54,54,54,53,53,53,53,53,53,52,52,52,52,51,51,51,51,51,51,50,50,
    50,50,49,49,49,48,48,48,48,48,48,48,48,48,48,47,47,47,47,47,46,
    46,46,46,46,46,46,46,45,45,45,45,45,45,45,45,44,44,44,43,43,43,
    43,43,43,43,42,42,42,42,42,42,42,41,41,41,41,41,41,41,40,40,40,
    40,40,40,39,39,39,39,39,39,39,39,39,38,38,38,38,38,38,37,37,37,
    37,37,37,37,37,37,37,36,36,36,36,36,36,36,35,35,34,34,34,34,34,
    34,33,33,33,33,32,32,32,32,32,31,31,31,31,31,31,31,30,30,30,30,
    30,30,30,30,30,30,29,29,29,29,29,29,28,28,28,28,28,27,27,27,27,
    27,27,26,26,26,26,26,25,25,25,25,25,25,25,25,24,24,24,23,23,23,
    23,23,23,22,22,21,21,21,21,21,21,20,20,20,20,20,20,20,20
  };
  const int n4c1w2_q[] = {
    100, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,100,100,100,100,99,99,99,99,99,98,98,98,98,
    97,97,97,97,97,97,96,96,96,96,96,96,96,96,95,95,95,94,94,94,94,
    94,93,93,93,93,93,93,93,93,92,92,92,92,92,92,92,92,92,91,91,91,
    91,91,91,91,91,90,90,90,90,90,89,89,89,89,89,89,89,89,89,89,88,
    88,88,88,87,87,87,87,87,86,86,86,86,86,86,85,85,85,85,84,84,84,
    84,84,84,84,84,83,83,82,82,82,82,82,82,82,81,81,81,81,81,81,81,
    80,80,80,80,80,80,80,80,79,79,79,79,79,79,79,78,78,78,78,77,77,
    77,77,77,77,77,77,76,76,76,76,76,76,76,76,76,76,75,75,75,75,74,
    74,74,74,74,74,74,74,74,74,74,74,73,73,73,73,73,72,72,72,72,72,
    71,71,71,71,71,71,71,71,71,71,71,70,70,70,70,70,70,69,69,69,69,
    69,68,68,68,68,68,67,67,67,66,66,66,66,66,66,66,66,65,65,65,65,
    65,65,65,65,64,64,64,63,63,63,62,62,62,62,61,61,61,61,61,61,61,
    61,61,60,60,60,60,59,59,59,59,59,59,59,58,58,57,57,57,57,57,57,
    57,57,57,57,56,56,56,56,56,56,55,55,55,55,55,55,55,55,55,55,54,
    54,54,54,54,54,54,54,54,53,53,53,53,53,52,52,52,51,51,51,51,50,
    50,50,50,50,50,50,50,50,50,49,49,49,48,48,48,48,48,48,48,48,47,
    47,47,46,46,46,46,46,46,46,46,45,45,45,45,45,45,45,44,44,44,44,
    44,44,44,44,44,43,43,42,42,42,42,42,41,41,41,41,41,41,40,40,40,
    40,40,40,40,40,39,39,39,39,39,39,39,39,39,38,38,38,37,37,37,37,
    37,37,36,36,36,36,36,36,36,35,35,35,35,34,34,34,34,34,34,34,34,
    34,34,33,33,33,33,33,32,32,32,31,31,31,31,31,31,31,30,30,30,30,
    30,30,29,29,29,29,29,28,28,28,28,28,28,28,27,27,27,27,27,26,26,
    26,26,26,26,25,25,25,25,24,24,24,24,24,24,24,23,23,23,23,23,23,
    23,22,22,22,22,22,22,22,21,21,21,21,20,20,20,20,20,20,20,20
  };
  const int n4c1w2_r[] = {
    100, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,100,100,100,99,99,99,99,99,99,99,99,99,99,
    99,99,99,98,98,98,98,98,97,97,97,96,96,96,96,96,96,96,96,96,96,
    96,95,95,95,94,94,93,93,93,93,93,93,93,92,92,92,92,92,92,92,91,
    91,91,90,90,90,89,89,89,89,89,89,89,89,89,89,89,88,88,88,88,88,
    88,88,88,88,87,87,87,87,87,87,87,87,86,86,86,86,86,86,86,86,85,
    85,85,84,84,84,84,84,83,83,83,83,83,83,83,83,82,82,82,82,82,82,
    81,81,81,80,80,80,80,80,79,79,79,79,78,78,78,78,78,78,78,78,78,
    78,78,77,77,77,77,77,77,77,76,76,76,76,75,75,75,75,75,75,75,75,
    75,74,74,74,74,74,74,74,74,73,73,73,73,73,73,72,72,72,72,71,71,
    71,71,71,70,70,70,70,70,70,70,70,69,69,69,69,69,69,69,68,68,68,
    68,68,67,67,67,67,67,67,67,67,67,67,66,66,66,66,65,65,65,65,65,
    65,65,64,64,64,64,64,64,64,64,64,63,63,63,63,62,62,62,62,62,61,
    61,61,61,60,60,60,60,59,59,59,59,59,58,58,58,58,58,58,58,58,58,
    58,58,57,57,57,57,56,56,56,56,56,56,56,55,55,55,55,55,54,54,54,
    54,53,53,52,52,52,52,52,52,52,52,51,51,51,51,51,51,50,50,50,50,
    49,49,49,49,49,49,49,49,49,49,49,49,48,48,48,48,48,48,48,47,47,
    46,46,46,46,46,46,46,46,46,46,46,45,45,44,44,44,44,44,44,43,43,
    43,43,43,43,43,43,43,42,42,42,42,42,42,42,42,42,42,41,41,41,41,
    40,40,40,40,40,40,40,40,39,39,39,39,38,38,38,38,38,38,37,37,37,
    37,37,37,36,36,36,36,36,35,35,35,35,35,34,34,34,34,34,34,34,33,
    33,33,33,33,33,33,33,32,31,31,31,31,30,30,30,30,30,30,30,29,29,
    29,29,29,29,29,29,28,28,28,28,27,27,27,27,27,26,26,26,26,26,26,
    25,25,25,25,25,25,25,24,24,24,24,24,24,23,22,22,22,22,22,22,22,
    22,22,21,21,21,21,21,21,21,21,21,21,21,21,20,20,20,20,20,20
  };
  const int n4c1w2_s[] = {
    100, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,100,99,99,99,99,99,99,99,98,98,98,98,98,98,
    98,98,98,98,97,97,97,97,97,97,97,97,96,96,96,95,95,95,95,95,94,
    94,94,94,94,93,93,93,93,93,93,93,93,92,92,92,92,91,91,91,91,91,
    91,91,91,91,90,90,90,89,89,89,89,89,89,89,89,89,89,89,89,89,89,
    88,88,88,88,88,87,87,87,87,87,87,87,86,86,86,85,85,85,85,85,85,
    85,85,85,84,84,84,84,84,84,83,83,83,83,83,83,83,83,83,83,82,82,
    82,82,82,82,82,81,81,80,80,79,79,79,79,79,79,78,78,78,77,77,77,
    77,76,76,76,76,76,75,75,74,74,73,73,73,73,73,73,73,73,73,72,72,
    72,72,72,72,72,71,71,71,71,70,70,70,69,69,69,69,69,69,69,69,69,
    68,68,68,68,68,68,68,68,68,68,68,67,67,67,67,66,66,66,66,66,65,
    65,65,65,65,65,65,65,64,64,64,64,64,63,63,63,63,63,63,63,63,63,
    63,63,62,62,62,62,62,62,62,61,61,61,61,61,60,60,59,59,59,59,59,
    59,59,59,58,58,58,58,58,58,58,57,57,57,57,57,57,57,57,57,57,56,
    56,56,56,56,56,56,55,55,55,55,55,55,54,54,54,54,54,54,54,53,53,
    53,53,52,52,52,52,52,51,51,51,51,51,50,50,50,49,49,49,49,48,47,
    47,47,47,47,46,46,46,46,45,45,45,45,45,45,45,44,44,44,44,44,44,
    44,44,43,43,43,43,43,43,43,43,42,42,42,42,42,42,42,42,42,41,41,
    41,41,41,40,40,40,40,40,40,40,40,39,39,39,39,39,39,39,39,39,39,
    39,38,38,38,38,38,38,38,38,38,38,38,38,37,37,37,37,37,37,36,36,
    36,36,36,36,36,36,35,35,35,35,35,35,35,34,34,34,34,33,33,33,33,
    33,32,32,32,32,32,31,31,31,31,31,31,31,30,30,30,30,30,30,30,30,
    29,29,29,29,29,29,29,28,28,28,28,28,28,28,28,27,27,27,27,27,27,
    26,26,26,26,26,26,26,26,26,25,25,25,25,25,25,25,25,24,24,24,24,
    23,23,23,23,23,23,23,22,22,22,22,21,21,21,21,21,20,20,20
  };
  const int n4c1w2_t[] = {
    100, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,100,99,99,99,99,99,99,99,99,99,99,98,98,98,
    98,98,98,98,97,97,96,96,96,96,96,96,96,96,96,96,95,95,95,95,95,
    95,95,94,94,94,94,94,93,93,93,93,93,93,93,92,92,92,92,92,92,92,
    91,91,91,91,91,91,90,90,90,90,90,90,89,89,89,89,89,89,89,89,89,
    89,88,88,87,87,87,87,87,86,86,86,86,85,85,85,84,84,84,84,84,83,
    83,83,83,83,83,83,82,82,82,81,80,80,80,80,80,80,80,80,80,80,79,
    79,79,79,79,78,78,78,78,78,78,78,77,77,77,76,76,76,76,76,76,76,
    76,75,75,75,75,75,75,75,74,74,74,74,74,74,74,74,74,74,74,74,73,
    73,73,73,73,73,73,72,72,72,72,72,72,72,72,72,71,71,71,71,71,71,
    71,70,70,70,70,70,70,69,69,69,69,69,69,69,68,68,68,67,67,67,67,
    67,67,67,67,66,66,66,65,65,65,65,65,65,65,65,65,64,64,64,64,64,
    64,63,63,63,63,62,62,62,62,62,62,62,61,61,61,61,61,61,61,61,61,
    60,60,60,60,60,60,60,60,59,59,59,59,59,59,59,58,58,58,58,57,57,
    57,57,57,56,56,55,55,55,55,55,55,55,55,55,54,54,54,54,54,54,54,
    54,53,53,53,53,53,53,53,53,52,52,52,52,51,51,51,51,51,51,51,51,
    51,50,50,50,50,50,50,50,49,49,49,49,49,49,49,48,48,48,48,47,47,
    47,47,47,47,47,46,46,46,46,46,46,46,45,45,45,45,45,45,45,45,45,
    45,45,45,44,44,44,44,44,44,44,44,43,43,43,43,43,43,43,43,42,42,
    42,42,42,42,41,41,41,41,41,41,40,40,40,40,40,39,39,39,39,39,38,
    38,38,38,38,38,37,37,36,36,36,36,36,36,36,36,36,36,35,35,35,34,
    34,34,33,33,33,32,32,32,32,32,32,32,32,32,32,31,31,31,31,31,31,
    30,30,30,30,29,29,29,29,29,28,28,28,28,27,27,27,26,26,26,26,26,
    25,25,25,25,25,25,24,24,24,24,23,23,23,23,22,22,22,22,22,21,21,
    21,21,21,21,21,21,21,21,20,20,20,20,20,20,20,20,20,20,20
  };
  const int n4c1w4_a[] = {
    100, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,100,99,99,99,99,99,99,98,98,98,98,98,97,97,
    97,97,97,97,97,96,96,96,96,96,96,96,96,95,95,95,95,95,95,95,95,
    95,95,95,95,95,95,94,94,94,94,94,94,93,93,93,93,93,93,92,92,92,
    92,92,92,91,91,91,91,91,91,91,91,90,90,90,90,90,90,89,89,89,89,
    89,89,89,89,88,88,88,88,88,88,88,88,88,88,88,87,87,87,87,87,87,
    87,87,87,87,86,86,86,86,86,86,86,86,85,85,85,85,85,85,85,85,85,
    84,84,84,84,84,83,83,83,83,83,83,83,83,83,83,83,82,82,82,82,81,
    81,81,81,81,81,81,81,80,80,80,80,80,79,79,79,79,79,79,79,78,78,
    78,78,78,78,77,77,77,77,77,76,76,76,76,76,76,76,75,75,75,75,74,
    74,74,74,74,74,74,74,74,73,73,73,73,73,73,73,73,73,73,73,73,73,
    73,73,73,73,72,72,72,72,72,72,71,71,71,71,70,70,70,70,70,69,69,
    69,69,69,69,69,69,69,69,68,68,68,67,67,67,67,67,67,67,67,66,66,
    66,66,66,66,65,65,65,65,65,65,65,65,64,64,64,64,64,64,63,63,63,
    63,63,63,63,63,62,62,62,62,62,62,62,61,61,61,61,60,60,60,60,60,
    60,60,60,60,60,59,59,59,59,59,59,59,59,58,58,58,58,58,58,58,58,
    58,57,57,57,56,56,56,56,56,56,56,56,56,56,55,55,55,55,55,54,54,
    54,54,54,54,54,53,53,53,53,53,53,53,53,53,52,52,52,52,52,52,51,
    51,50,50,50,50,50,50,50,50,50,50,49,49,49,49,48,48,48,48,48,48,
    48,48,48,47,47,47,47,47,47,47,47,47,47,47,47,47,47,47,47,47,46,
    46,46,46,46,46,46,45,45,45,45,45,44,44,44,44,44,44,44,44,43,43,
    43,43,42,42,42,42,42,42,42,42,42,41,41,41,41,41,40,40,40,40,40,
    40,39,39,39,39,39,38,38,38,38,38,38,37,37,37,37,37,36,36,36,36,
    36,36,36,36,36,35,35,35,35,35,35,35,35,35,34,34,34,33,33,33,33,
    33,33,33,32,32,32,32,32,32,31,31,31,31,31,31,30,30,30,30
  };
  const int n4c1w4_b[] = {
    100, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,100,100,100,100,99,99,99,99,99,99,99,99,98,
    98,98,98,98,98,98,98,98,98,98,97,97,97,97,97,97,96,96,96,96,96,
    96,96,96,96,95,95,95,95,95,95,94,94,93,93,93,93,93,93,92,92,92,
    92,92,92,92,91,91,91,91,91,91,91,91,90,90,90,90,90,90,89,89,89,
    89,89,89,89,89,89,89,88,88,88,88,88,87,87,87,87,87,87,87,86,86,
    86,86,85,85,85,85,85,84,84,83,83,83,83,83,83,82,82,82,82,81,81,
    81,81,81,81,81,81,81,81,81,80,80,80,80,79,79,79,79,79,79,79,78,
    78,78,78,78,78,78,77,77,77,77,77,77,76,76,76,76,76,76,76,75,75,
    75,75,75,75,75,75,75,74,74,74,74,74,73,73,73,73,73,73,73,72,72,
    72,72,72,72,71,70,70,70,69,69,69,69,69,69,69,69,69,68,68,68,68,
    68,68,68,68,68,68,67,67,67,67,67,66,66,66,66,66,65,65,65,65,65,
    65,65,64,64,64,64,64,64,63,63,63,63,63,63,63,62,62,62,62,62,62,
    62,62,62,61,61,61,61,61,61,60,60,60,60,60,60,59,59,59,59,59,59,
    58,58,58,58,58,58,58,58,58,58,58,58,58,58,57,57,57,57,57,57,57,
    57,56,56,56,56,56,56,56,56,56,55,55,55,55,55,55,55,54,54,53,53,
    53,53,53,53,53,53,53,53,53,53,53,52,52,52,52,52,52,52,52,51,51,
    51,51,51,51,51,51,51,51,51,51,51,50,50,50,50,50,50,50,49,49,49,
    49,49,49,49,49,49,48,48,48,48,48,48,48,48,48,47,47,47,47,47,47,
    47,47,47,47,47,46,46,46,46,46,46,46,45,45,45,45,45,45,45,44,44,
    44,44,44,44,44,43,43,43,43,43,43,43,43,43,43,43,42,42,42,42,42,
    42,42,42,42,41,41,41,41,41,41,41,40,40,39,39,39,39,39,39,38,38,
    38,38,37,37,37,37,37,37,37,37,37,36,36,36,36,36,35,35,35,35,35,
    35,35,35,35,35,35,35,35,34,34,34,34,34,34,34,34,34,33,33,33,33,
    33,33,33,33,33,33,32,32,32,32,32,31,31,31,31,30,30,30,30,30
  };
  const int n4c1w4_c[] = {
    100, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,99,99,99,99,99,98,98,98,98,98,98,98,98,98,98,
    97,97,97,97,96,96,96,96,96,96,96,95,95,95,95,95,95,95,95,95,94,
    94,94,94,94,94,94,94,93,93,93,93,93,93,93,93,92,92,92,92,92,92,
    92,92,92,91,91,91,91,91,91,90,90,90,90,90,90,90,90,90,90,89,89,
    89,89,89,89,89,89,89,89,89,89,88,88,88,88,88,87,87,87,87,87,87,
    87,87,86,86,86,86,86,85,85,85,84,84,83,83,83,83,83,82,82,82,82,
    82,82,81,81,81,81,80,80,80,80,80,80,80,80,80,80,80,79,79,79,79,
    78,78,78,78,78,78,77,77,77,77,77,76,76,76,76,76,76,76,76,76,76,
    76,76,75,75,75,74,74,74,74,74,74,74,74,74,73,73,73,73,73,73,73,
    73,73,72,72,72,71,71,71,71,71,71,71,71,71,70,70,70,70,70,70,70,
    69,69,69,69,69,69,69,69,69,69,68,68,68,68,68,68,67,67,67,67,67,
    67,67,67,67,67,67,67,67,67,67,67,66,66,66,66,66,66,66,66,66,66,
    65,65,65,65,65,65,65,64,64,64,64,64,64,64,64,64,64,63,63,63,63,
    63,63,62,62,62,62,62,62,62,62,62,62,61,61,61,61,61,61,61,60,60,
    60,60,60,60,60,60,60,60,59,59,59,59,59,59,59,59,58,58,58,58,58,
    58,58,58,58,57,57,57,56,56,56,55,55,55,55,55,55,55,54,54,54,54,
    54,54,53,53,53,53,52,52,52,52,51,51,51,51,51,51,51,51,51,51,50,
    50,50,50,50,50,50,50,50,50,50,49,49,49,49,49,49,49,49,48,48,48,
    48,48,48,47,47,47,47,47,47,46,46,46,46,46,45,45,45,45,45,44,44,
    44,44,44,44,44,44,44,43,43,43,43,43,43,43,43,43,42,42,42,42,42,
    41,41,41,41,41,41,41,41,41,40,40,40,39,39,39,39,39,39,38,38,38,
    38,38,38,38,38,38,37,37,37,37,37,37,37,36,36,36,36,36,36,36,36,
    35,35,35,35,34,34,34,34,34,34,33,33,33,33,33,33,33,33,33,33,32,
    32,32,32,32,32,31,31,31,31,31,31,31,30,30,30,30,30,30,30
  };
  const int n4c1w4_d[] = {
    100, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,100,100,100,100,100,100,100,99,99,99,99,99,
    99,98,98,98,98,98,98,97,97,97,97,97,97,97,97,97,97,96,96,96,96,
    95,95,95,95,95,95,95,94,94,94,93,93,93,93,93,93,93,93,92,92,92,
    92,92,92,91,91,91,90,90,90,89,89,89,89,89,89,89,89,89,89,88,88,
    88,88,88,87,87,87,87,86,86,86,86,86,86,86,86,86,86,86,85,85,85,
    85,85,85,85,85,85,85,85,84,84,84,84,83,83,83,83,83,83,83,82,82,
    82,82,82,82,81,80,80,80,80,80,80,80,80,80,79,79,79,79,79,78,78,
    78,78,78,78,77,77,77,77,77,77,77,77,76,76,76,76,76,76,76,76,75,
    75,75,75,75,75,75,75,75,74,74,74,74,74,73,73,73,73,73,73,73,73,
    73,73,72,72,72,72,71,71,71,71,71,71,71,71,71,70,70,70,70,69,69,
    69,69,68,68,68,68,68,68,67,67,67,67,67,66,66,66,66,66,65,65,65,
    65,65,65,64,64,64,64,64,64,64,64,64,64,63,63,63,63,63,63,63,62,
    62,62,62,62,62,62,62,62,62,62,62,62,62,62,62,62,62,61,61,61,61,
    61,61,61,61,61,61,60,60,60,60,60,60,60,59,59,59,59,59,59,59,59,
    58,58,58,58,58,58,58,58,58,57,57,57,57,57,57,57,56,56,56,56,56,
    56,56,55,55,55,55,55,55,55,54,54,54,54,54,54,54,54,53,53,53,53,
    53,53,53,53,53,53,53,52,52,52,52,52,52,52,52,52,51,51,51,51,51,
    51,51,51,50,50,50,50,50,49,49,49,49,49,49,49,49,49,48,48,48,48,
    47,47,47,47,47,47,47,47,46,46,46,46,46,46,46,46,46,45,45,45,45,
    45,45,44,44,44,44,44,44,44,44,43,43,43,43,43,43,42,42,42,42,42,
    42,42,41,41,41,41,41,41,41,41,40,40,40,40,40,40,39,39,39,39,39,
    38,38,38,38,38,38,38,38,38,38,37,37,37,37,37,37,36,36,36,36,36,
    36,36,36,36,36,36,36,35,35,35,35,35,35,35,35,34,34,34,34,34,34,
    34,33,33,33,33,33,33,33,32,31,31,31,31,31,30,30,30,30,30,30,30
  };
  const int n4c1w4_e[] = {
    100, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,100,100,100,100,100,99,99,99,99,99,98,98,
    98,98,98,98,98,98,98,98,98,98,97,97,97,97,97,97,97,97,97,96,96,
    96,96,96,96,96,96,96,96,96,95,95,95,95,94,94,94,94,94,93,93,93,
    93,93,93,92,92,92,92,92,92,92,91,91,91,91,91,90,90,90,90,90,90,
    90,90,89,89,89,89,89,89,89,89,89,89,88,88,88,88,88,88,87,87,87,
    87,87,87,86,86,86,86,85,85,85,85,85,85,85,85,85,85,84,84,84,84,
    84,84,84,83,83,83,83,83,83,83,82,82,82,82,82,82,82,81,81,81,81,
    81,81,80,80,80,80,80,80,80,80,80,80,80,80,80,80,79,79,79,79,79,
    79,78,78,78,78,78,78,78,77,77,77,77,77,76,76,76,76,76,76,76,76,
    76,76,76,76,76,75,75,75,75,75,75,75,74,74,74,74,74,74,74,74,74,
    74,74,74,73,73,73,73,73,73,73,73,72,72,72,72,72,72,71,71,71,71,
    71,71,70,70,70,70,70,70,70,70,70,69,69,69,69,69,69,69,69,68,68,
    68,68,68,68,67,67,67,67,67,67,67,67,67,67,67,67,67,67,66,66,66,
    66,66,66,66,66,65,65,65,65,64,64,64,64,63,63,63,63,63,63,63,63,
    63,63,62,62,62,62,62,61,61,61,61,61,60,60,60,60,60,60,59,59,59,
    59,59,59,59,59,59,59,59,59,59,58,58,58,58,57,57,57,57,57,57,57,
    57,57,56,56,56,56,56,56,55,55,55,55,55,55,54,54,54,54,53,53,53,
    53,53,53,53,53,52,52,52,52,51,51,51,51,51,51,50,50,49,49,49,49,
    49,49,48,48,48,48,48,48,47,47,47,47,47,47,47,46,46,46,46,46,46,
    46,45,45,45,45,45,44,44,44,43,43,43,43,43,43,43,43,43,43,42,42,
    42,42,42,42,42,42,42,42,41,41,41,41,41,41,40,40,39,39,39,39,39,
    39,39,38,38,38,38,38,38,38,38,37,37,36,36,36,36,36,36,35,35,35,
    35,35,35,35,35,34,34,34,34,33,33,33,33,33,33,32,32,32,32,32,32,
    32,32,32,32,32,32,31,31,31,31,31,31,31,31,31,30,30,30,30,30,30
  };
  const int n4c1w4_f[] = {
    100, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,99,99,99,99,99,99,99,99,98,98,98,98,98,97,97,97,97,
    97,97,96,96,96,96,96,96,96,94,94,94,94,94,94,93,93,93,93,93,92,
    92,92,91,91,91,91,91,90,90,90,90,90,89,89,89,89,89,89,89,88,88,
    88,88,88,87,87,87,87,87,87,86,86,86,86,85,85,85,84,84,84,84,84,
    84,84,83,83,83,83,83,83,83,83,83,83,82,82,82,82,82,82,81,81,81,
    81,81,81,81,81,80,80,80,80,80,80,80,80,80,79,79,79,79,79,79,79,
    78,78,78,78,78,78,78,78,78,77,77,77,77,77,77,77,76,76,76,76,76,
    76,76,76,76,75,75,75,75,75,75,75,75,75,75,74,74,74,74,74,74,74,
    73,73,73,73,73,73,73,73,73,73,73,73,73,73,73,73,73,73,72,72,72,
    72,72,72,72,71,71,71,70,70,70,70,69,69,69,69,69,69,69,69,69,69,
    69,69,68,68,68,68,68,68,68,68,68,68,68,67,67,66,66,66,66,65,65,
    65,65,65,65,65,65,64,64,64,64,64,64,64,64,64,64,64,64,64,63,63,
    63,63,63,63,63,63,62,62,62,62,62,61,61,61,61,61,60,60,60,60,60,
    60,60,60,60,60,60,60,60,60,60,59,59,59,59,59,59,59,59,59,59,58,
    58,58,58,58,58,58,58,58,58,57,57,57,57,57,57,57,57,57,57,57,57,
    56,56,56,56,56,56,56,56,55,55,55,55,55,55,55,54,54,54,54,54,54,
    54,54,54,54,53,53,53,53,53,53,53,52,52,52,52,52,52,52,52,52,51,
    51,51,51,51,51,51,51,51,50,50,50,50,50,50,50,50,50,49,49,49,48,
    48,48,48,48,47,47,47,47,47,47,46,46,46,46,46,46,46,46,46,46,45,
    45,45,45,44,44,44,44,44,44,44,44,44,44,44,43,43,43,42,42,42,41,
    41,41,41,41,40,40,40,40,40,40,40,40,40,40,39,39,39,39,39,39,39,
    39,39,38,38,38,38,38,38,37,37,37,37,37,37,37,37,37,36,36,36,36,
    36,35,35,35,35,35,35,35,35,34,34,34,34,34,33,33,33,33,32,32,32,
    32,32,32,32,32,31,31,31,31,31,31,30,30,30,30,30,30,30
  };
  const int n4c1w4_g[] = {
    100, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,100,99,99,99,99,99,99,99,99,99,98,98,98,98,
    98,98,98,98,97,97,97,97,97,97,97,96,96,96,96,96,96,95,95,95,95,
    95,95,95,95,95,94,94,94,94,94,93,93,93,93,93,93,93,93,93,93,92,
    92,92,92,92,92,91,91,91,91,91,91,91,90,90,90,90,90,90,89,89,89,
    89,89,89,89,89,89,89,89,88,88,88,88,87,87,87,87,87,87,87,86,86,
    86,86,86,86,86,86,86,86,85,85,85,85,85,85,84,84,83,83,83,83,83,
    82,82,82,82,82,82,82,82,82,82,82,82,82,81,81,81,81,81,81,81,81,
    81,80,80,80,80,80,80,79,79,79,79,79,79,79,79,79,78,78,78,78,78,
    78,78,78,78,78,77,77,77,77,77,77,77,77,77,77,77,76,76,76,75,75,
    75,75,75,75,75,75,75,74,74,74,74,74,74,74,73,73,73,73,73,73,73,
    73,73,73,73,72,72,72,72,72,72,72,72,71,71,71,71,71,70,70,70,70,
    70,70,69,69,69,69,69,68,68,68,68,68,68,68,68,68,68,68,68,67,67,
    67,66,66,66,66,66,66,66,65,65,65,65,65,65,65,64,64,64,64,64,63,
    63,63,63,63,63,62,62,62,62,62,62,61,61,61,60,60,60,60,60,60,60,
    60,60,60,59,59,59,59,59,59,59,59,59,58,58,58,58,58,58,57,57,57,
    56,56,56,56,55,55,55,55,55,55,55,54,54,54,54,54,54,54,54,54,54,
    53,53,53,53,53,52,52,52,52,52,52,51,51,51,51,51,51,51,51,51,50,
    50,50,50,50,50,49,49,49,49,49,49,49,48,48,48,48,48,48,48,48,47,
    47,47,47,47,47,47,47,46,46,46,46,46,46,45,45,45,45,44,44,44,44,
    44,44,44,43,43,43,43,43,43,43,42,42,42,42,42,42,42,42,42,42,41,
    41,41,41,41,41,41,41,41,41,40,40,40,40,40,39,39,39,39,39,39,39,
    39,38,38,38,38,37,37,37,37,37,37,36,36,36,36,35,35,35,35,35,35,
    35,35,35,34,34,34,34,33,33,33,33,33,33,32,32,32,32,32,32,32,32,
    32,32,31,31,31,31,31,31,31,30,30,30,30,30,30,30,30,30,30
  };
  const int n4c1w4_h[] = {
    100, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,100,100,100,99,99,99,99,99,99,99,99,99,99,
    99,99,99,98,98,98,98,98,98,98,97,97,97,97,97,97,97,97,97,97,97,
    96,96,96,96,96,95,95,95,95,95,95,95,95,95,95,95,95,94,94,94,94,
    94,94,93,93,93,93,93,92,92,92,92,92,92,91,91,91,91,91,91,91,91,
    91,91,91,91,91,91,90,90,90,90,90,89,89,89,89,89,89,89,89,88,88,
    88,88,88,87,87,87,87,87,87,87,86,86,86,86,86,86,86,86,85,85,85,
    85,84,84,84,84,84,84,84,83,83,83,83,82,82,82,82,82,82,82,82,82,
    82,82,81,81,81,81,81,81,81,81,80,80,80,80,80,80,80,80,80,80,79,
    79,79,79,79,79,79,78,78,78,78,78,78,78,78,77,77,77,76,76,76,76,
    76,76,76,76,76,76,76,76,76,75,75,75,75,74,74,74,74,74,74,73,73,
    73,72,72,72,71,71,71,71,71,71,71,70,70,70,70,70,70,70,70,70,69,
    69,69,69,69,68,68,68,68,68,68,68,67,67,67,67,67,67,66,66,66,66,
    66,66,66,66,66,65,65,65,65,65,65,65,65,65,65,64,64,64,64,64,63,
    63,63,63,63,63,62,62,62,62,62,61,61,61,61,61,61,61,61,61,60,60,
    60,60,60,59,59,59,59,58,58,58,58,58,58,58,58,58,57,57,57,57,57,
    57,56,56,56,56,56,56,56,56,56,56,55,55,55,55,55,55,55,55,55,55,
    54,54,54,54,54,53,53,52,52,52,52,52,51,51,51,51,50,50,49,49,49,
    49,49,48,48,48,48,48,48,48,47,47,47,47,46,46,46,46,46,46,45,45,
    45,45,45,45,45,45,45,44,44,44,44,44,44,44,44,44,44,43,43,43,43,
    43,43,43,43,42,42,42,42,42,42,42,42,42,41,41,41,41,41,41,40,40,
    40,39,39,39,38,38,38,38,38,38,38,38,38,38,38,37,37,37,37,37,37,
    37,37,36,36,36,36,36,36,36,35,35,35,35,35,35,35,35,34,34,34,34,
    34,34,34,34,34,34,33,33,33,33,33,33,33,32,32,32,32,32,32,32,32,
    32,31,31,31,31,31,31,31,31,31,31,31,31,30,30,30,30,30,30,30
  };
  const int n4c1w4_i[] = {
    100, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,100,100,100,100,100,100,99,99,99,99,98,98,
    98,98,98,98,98,98,97,97,97,97,97,97,97,97,97,97,97,96,96,96,96,
    96,96,95,95,95,95,95,95,95,95,95,95,94,94,94,94,94,94,94,94,94,
    93,93,93,93,93,93,93,92,92,92,92,92,92,92,92,91,91,91,91,91,91,
    91,91,91,91,91,91,91,90,90,90,90,89,89,89,89,89,89,89,89,88,88,
    88,88,88,88,87,87,87,87,87,87,87,87,86,86,86,86,86,86,85,85,85,
    85,85,85,84,84,84,84,84,84,84,84,83,83,83,83,82,82,82,82,82,82,
    81,81,81,81,81,80,80,80,80,80,80,80,80,80,80,80,80,79,79,79,79,
    78,78,78,78,78,78,77,77,77,76,76,76,76,76,76,76,76,75,75,75,75,
    75,75,75,75,75,75,75,74,74,74,74,74,74,74,73,73,73,72,72,72,72,
    72,71,71,71,71,71,71,71,71,71,71,70,70,70,70,70,70,69,69,69,69,
    69,69,68,68,68,68,68,68,68,67,67,67,67,67,67,67,66,66,66,66,66,
    66,66,66,66,65,65,65,65,64,64,64,64,64,64,64,63,63,63,63,62,62,
    62,62,62,62,61,61,61,61,61,61,61,61,61,61,61,60,60,60,60,60,60,
    59,59,59,59,59,59,59,59,58,58,58,58,58,58,58,57,57,57,57,57,57,
    57,56,56,56,56,56,55,55,55,55,55,55,55,54,54,54,54,53,53,53,53,
    53,52,52,52,52,52,52,52,52,52,52,51,51,51,51,51,50,50,50,50,50,
    50,49,49,49,49,49,49,49,49,49,48,48,48,48,48,47,47,47,47,47,46,
    46,45,45,45,45,45,45,45,44,44,44,44,44,44,44,44,44,44,44,44,44,
    43,43,43,43,43,43,43,43,42,42,42,42,42,42,41,41,41,41,41,40,40,
    40,40,40,40,40,40,40,40,40,40,39,39,39,39,39,39,39,39,39,39,38,
    38,38,37,37,37,37,37,37,37,36,36,36,36,36,36,36,36,36,36,36,36,
    35,35,35,35,35,35,35,34,34,34,34,34,34,34,34,34,34,33,33,33,33,
    33,33,32,32,32,32,31,31,31,31,31,31,31,31,31,31,30,30,30,30,30
  };
  const int n4c1w4_j[] = {
    100, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,100,100,100,99,99,99,99,99,99,99,99,99,98,
    98,98,98,98,98,98,98,97,97,97,97,96,96,96,96,96,96,96,96,96,96,
    96,95,95,95,95,95,95,95,94,94,94,94,94,94,94,93,93,93,93,93,93,
    93,93,92,92,92,92,92,92,92,92,92,92,92,92,91,91,91,91,91,91,90,
    90,90,90,90,90,90,90,90,90,89,89,89,89,89,89,89,88,88,88,88,88,
    87,87,87,87,87,87,87,87,87,87,86,86,86,86,86,86,85,85,85,85,85,
    85,85,85,85,85,85,84,84,84,84,84,83,83,83,83,83,83,83,83,83,83,
    82,82,82,82,82,82,82,81,81,81,80,80,80,80,80,80,80,80,80,80,80,
    80,79,79,79,79,79,79,79,79,79,78,78,78,78,77,77,77,77,77,77,76,
    76,76,76,76,76,76,76,76,75,75,75,75,75,75,74,74,74,74,74,74,73,
    73,73,73,73,73,73,73,72,72,72,71,71,71,71,71,71,71,70,70,70,70,
    70,70,70,70,70,70,70,69,69,69,69,69,69,69,69,68,68,68,68,68,67,
    67,67,66,66,65,65,65,65,65,65,65,65,64,64,64,64,64,63,63,63,63,
    63,63,63,62,62,62,62,62,62,62,62,62,62,62,61,61,61,61,61,61,61,
    61,61,60,60,60,60,60,60,59,59,59,59,59,59,59,59,59,59,59,59,59,
    59,58,58,58,58,57,57,57,57,57,57,57,56,56,56,56,56,56,56,56,55,
    55,55,55,54,54,54,54,54,54,54,53,53,53,53,53,52,52,52,52,52,52,
    52,51,51,51,51,51,51,51,51,51,50,50,50,50,50,50,49,49,49,48,48,
    48,48,48,48,48,48,48,48,48,47,47,47,47,46,46,46,46,46,46,46,45,
    45,45,45,45,45,45,44,44,44,44,44,44,44,44,43,43,43,43,43,43,42,
    42,42,42,42,42,42,41,41,41,41,40,40,40,40,40,40,39,39,39,39,39,
    39,39,38,38,38,38,38,38,38,37,37,37,37,37,37,36,36,36,36,36,36,
    35,35,35,35,35,35,34,34,34,34,34,34,34,34,34,34,34,34,34,34,33,
    33,33,33,32,32,32,32,32,32,31,31,31,31,31,31,30,30,30,30,30
  };
  const int n4c1w4_k[] = {
    100, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,100,100,99,99,99,99,99,99,99,99,99,99,98,
    98,98,98,98,98,98,97,97,97,97,97,97,97,97,97,96,96,96,96,96,96,
    96,96,96,96,95,95,95,95,95,95,95,95,95,94,94,94,93,93,93,93,93,
    93,92,92,92,92,92,92,92,91,91,91,91,91,91,91,90,90,90,90,90,89,
    89,89,89,89,89,89,89,89,89,88,88,88,88,88,88,88,88,88,88,88,88,
    88,88,87,87,87,87,87,87,86,86,86,86,86,86,86,86,86,86,85,85,85,
    85,85,84,84,84,84,84,84,84,84,84,84,84,84,84,83,83,83,83,83,83,
    83,82,82,82,81,81,81,80,80,80,80,80,79,79,79,79,79,78,78,78,78,
    78,78,77,77,77,77,77,77,77,77,77,76,76,76,75,75,75,75,75,75,74,
    74,74,74,74,74,74,74,74,73,73,73,73,73,73,73,73,73,73,73,72,72,
    72,72,72,72,72,71,71,71,71,71,71,71,71,71,71,70,70,70,70,70,70,
    70,70,70,69,69,69,69,69,69,69,69,69,69,69,69,68,68,68,67,67,67,
    67,67,67,66,66,66,66,66,66,66,66,65,65,65,65,65,64,64,64,64,64,
    64,64,64,64,64,63,63,63,63,63,63,63,63,63,63,62,62,62,62,62,62,
    61,61,61,61,60,60,60,60,59,59,59,59,59,59,59,59,59,59,59,59,59,
    58,58,58,58,58,58,58,57,57,57,57,57,56,56,56,56,55,55,55,55,55,
    55,55,55,55,55,55,54,54,54,54,54,54,54,54,53,53,53,53,53,53,52,
    52,52,52,52,51,51,51,51,51,50,50,50,50,50,50,50,50,50,49,49,49,
    49,49,49,48,48,48,48,48,48,48,48,47,47,47,46,46,46,46,46,46,46,
    46,46,45,45,45,45,45,44,44,44,44,44,44,44,44,44,44,43,43,43,43,
    43,43,43,43,43,42,42,42,42,42,42,41,41,41,41,41,40,40,40,40,40,
    40,39,39,39,39,39,39,38,38,38,38,38,37,37,37,36,36,36,36,36,36,
    35,35,35,35,35,35,35,35,34,34,34,34,34,34,34,33,33,33,33,33,32,
    32,32,32,32,32,32,32,31,31,31,31,31,31,30,30,30,30,30,30,30
  };
  const int n4c1w4_l[] = {
    100, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,100,100,100,100,100,99,99,99,99,99,99,99,
    98,98,98,98,98,98,98,98,98,98,98,98,98,98,97,97,97,96,96,96,96,
    96,96,96,95,95,95,95,95,95,95,95,95,95,95,94,94,94,94,94,94,94,
    94,94,94,94,93,93,93,93,93,92,92,92,91,91,91,91,91,91,91,90,90,
    90,90,89,89,89,89,89,89,88,88,88,88,87,87,87,87,87,87,87,86,86,
    86,86,86,86,85,85,85,85,85,85,85,85,85,84,84,84,83,83,83,83,83,
    83,83,83,83,83,82,82,82,82,82,81,81,81,81,80,80,80,80,80,80,80,
    80,80,80,79,79,79,78,78,78,78,78,77,77,77,77,77,77,77,76,76,76,
    76,76,76,76,76,76,76,76,76,76,76,75,75,75,75,75,74,74,74,74,74,
    73,73,73,73,73,73,72,72,72,72,72,71,71,71,71,71,71,71,71,71,71,
    71,71,70,70,70,70,70,70,70,69,69,69,69,69,69,68,68,68,68,68,68,
    67,67,67,67,66,66,66,66,66,66,66,66,66,65,65,65,65,65,65,65,65,
    64,64,64,64,64,64,64,64,63,63,63,63,62,62,62,62,62,62,62,62,62,
    61,61,61,61,61,61,61,61,60,60,60,60,60,60,60,60,60,60,60,60,60,
    60,59,59,59,59,59,59,58,58,58,58,57,57,57,57,57,57,57,57,56,56,
    56,56,56,56,56,55,55,55,55,54,54,54,54,53,53,53,53,53,52,52,52,
    51,51,51,51,51,51,51,51,50,50,50,50,50,49,49,49,49,49,48,48,48,
    48,48,48,48,48,48,48,48,47,47,47,47,47,47,47,47,47,46,46,46,46,
    46,46,46,46,46,46,45,45,45,45,44,44,44,44,44,44,44,44,44,43,43,
    43,43,43,43,43,43,42,42,42,42,42,42,42,42,41,41,41,41,41,41,41,
    41,41,40,40,40,40,40,39,39,39,39,39,39,39,38,38,38,38,38,38,38,
    38,38,37,37,37,37,37,36,36,36,36,36,36,36,36,36,36,35,35,35,35,
    35,35,35,35,35,34,34,33,33,33,33,33,33,33,33,32,32,32,32,32,32,
    32,32,32,31,31,31,31,31,31,31,31,31,31,30,30,30,30,30,30,30,30
  };
  const int n4c1w4_m[] = {
    100, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,100,100,99,99,99,99,99,98,98,98,98,98,98,
    98,98,98,97,97,97,97,96,96,96,96,95,95,95,95,95,95,95,95,94,94,
    94,94,94,94,94,94,94,94,94,94,94,94,93,93,93,93,93,93,92,92,92,
    92,92,92,91,91,91,91,91,91,91,90,90,90,90,90,90,90,90,90,90,90,
    90,89,89,89,89,89,88,88,88,88,88,88,88,87,87,87,87,87,87,87,87,
    87,87,87,86,86,86,86,86,86,86,86,85,85,85,85,85,85,85,85,85,85,
    84,83,83,83,83,83,83,82,82,82,82,82,82,82,81,81,81,81,81,80,80,
    80,80,80,80,80,80,80,80,79,79,79,79,79,79,78,78,78,78,78,78,77,
    77,77,77,77,77,76,76,76,75,75,75,75,75,74,74,74,74,74,74,73,73,
    73,73,73,73,73,73,73,72,72,72,72,72,72,72,72,72,72,71,71,71,71,
    71,71,71,71,71,70,70,70,70,70,70,70,70,69,69,69,69,69,68,68,68,
    68,68,68,68,68,68,68,68,67,67,67,67,67,67,67,67,67,66,66,66,66,
    66,66,66,65,65,65,65,65,65,65,64,64,64,64,64,63,63,63,63,63,63,
    62,62,62,62,62,62,62,61,61,61,61,61,61,60,60,60,60,60,60,60,59,
    59,59,59,59,59,59,59,58,58,58,58,57,57,57,57,57,57,57,56,56,56,
    56,56,56,56,56,55,55,55,55,55,55,55,55,55,54,54,54,54,54,54,54,
    54,54,54,54,54,53,53,53,53,53,52,52,52,52,52,52,51,51,51,51,51,
    50,50,50,50,50,50,50,50,50,50,49,49,49,49,49,48,48,48,48,47,47,
    47,47,47,47,47,47,47,46,46,46,46,45,45,45,44,44,44,44,44,44,44,
    44,44,44,44,43,43,43,43,43,43,43,42,42,42,42,41,41,41,41,41,41,
    41,41,40,40,40,40,40,40,40,40,39,39,39,39,39,39,39,39,38,38,38,
    37,37,37,37,37,37,37,36,36,36,36,36,36,36,36,35,35,35,35,34,34,
    34,34,34,34,34,34,33,33,33,33,33,33,33,32,32,32,32,32,32,32,32,
    32,32,32,32,31,31,31,31,31,31,31,31,30,30,30,30,30,30,30,30
  };
  const int n4c1w4_n[] = {
    100, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,99,99,99,99,99,99,98,98,98,98,97,97,97,96,
    96,96,96,96,95,95,95,95,95,95,95,95,95,95,95,95,95,94,94,94,94,
    94,94,94,94,93,93,93,93,93,93,92,92,92,92,92,92,91,91,91,91,91,
    91,91,91,90,90,90,90,89,89,89,89,89,89,89,89,89,88,88,88,88,88,
    88,88,87,87,87,87,87,87,87,87,87,87,86,86,86,86,86,86,86,85,85,
    85,85,85,85,85,84,84,84,84,84,84,84,84,83,83,83,83,83,83,83,82,
    82,82,82,82,82,82,82,82,81,81,81,81,81,81,81,81,80,80,80,80,80,
    80,80,80,79,79,79,79,79,79,78,78,78,78,78,78,78,78,77,77,77,77,
    77,77,77,77,76,76,75,75,75,75,75,75,75,75,75,74,74,74,74,74,74,
    74,74,74,74,74,74,74,74,73,73,73,73,73,73,73,73,73,73,72,72,72,
    72,72,72,71,71,71,71,71,70,70,70,70,70,70,70,70,70,69,69,69,69,
    69,69,69,69,69,68,68,68,68,68,67,67,67,67,66,66,66,66,66,66,66,
    66,66,65,65,65,65,65,65,65,64,64,64,64,64,64,64,63,63,63,63,62,
    62,62,62,62,62,62,61,61,61,61,61,61,61,61,60,60,60,60,60,60,60,
    60,60,59,59,59,59,58,58,58,58,58,58,58,58,58,58,58,57,57,57,57,
    57,57,56,56,56,56,56,56,55,55,55,55,55,55,54,54,54,54,54,54,54,
    54,54,54,54,53,53,53,53,53,53,53,53,52,52,52,52,52,52,52,52,51,
    51,51,51,51,51,50,50,50,50,50,50,49,49,49,49,49,49,48,48,48,48,
    48,48,48,48,47,47,47,47,47,47,46,46,46,46,46,46,46,45,45,45,45,
    45,45,45,45,45,45,44,44,44,44,43,43,43,43,43,42,42,42,42,42,42,
    41,41,41,41,41,41,41,41,41,41,40,40,40,40,40,40,40,40,40,40,39,
    39,39,39,39,38,38,38,37,37,37,36,36,36,36,36,35,35,35,35,35,35,
    35,35,35,35,35,34,34,34,34,34,34,33,33,33,33,33,33,32,32,32,32,
    32,32,31,31,31,31,31,31,31,31,30,30,30,30,30,30,30,30,30
  };
  const int n4c1w4_o[] = {
    100, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,100,100,100,99,99,99,98,98,98,98,98,98,98,
    98,98,97,97,97,97,97,97,97,96,96,96,96,96,96,96,95,95,95,95,94,
    94,94,94,93,93,93,93,93,93,93,93,92,92,92,92,92,92,92,92,91,91,
    91,91,91,91,90,90,90,90,90,90,90,90,90,90,90,90,90,89,89,89,89,
    89,89,89,89,88,88,88,88,88,88,87,87,87,87,86,85,85,85,85,84,84,
    84,84,84,84,84,84,84,84,84,83,83,83,83,83,83,83,83,83,83,83,83,
    82,82,82,82,81,81,81,81,81,81,80,80,80,79,79,79,79,79,79,79,79,
    79,78,78,78,78,78,78,78,78,78,78,78,78,77,77,77,77,77,77,77,76,
    76,76,76,76,76,75,75,75,75,74,74,74,74,74,73,73,73,73,73,73,73,
    72,72,72,72,72,72,72,72,71,71,71,71,71,71,71,70,70,70,70,70,70,
    69,69,69,69,69,69,68,68,68,68,68,68,68,67,66,66,66,66,66,66,66,
    66,66,66,66,65,65,65,64,64,64,64,64,64,64,64,64,64,64,63,63,63,
    63,63,63,63,63,62,62,62,62,62,62,62,62,61,61,60,60,60,60,60,59,
    59,59,59,59,59,59,59,59,59,59,58,58,58,58,58,58,58,57,57,57,57,
    57,57,57,56,56,56,56,56,56,56,55,55,55,55,55,55,55,55,55,54,54,
    54,54,54,54,54,54,54,54,53,53,53,53,53,53,53,52,52,52,52,52,52,
    52,51,51,51,51,51,51,51,51,50,50,50,50,50,50,50,50,49,49,49,49,
    49,49,48,48,48,48,48,48,47,47,47,47,47,47,47,47,46,46,46,46,46,
    46,46,45,45,45,45,45,45,45,45,45,45,45,44,44,44,44,44,43,43,43,
    43,43,43,43,43,43,43,43,43,43,42,42,42,42,42,41,41,41,41,41,41,
    41,41,41,41,41,41,41,40,40,40,40,40,39,39,39,39,39,39,39,39,39,
    38,38,38,38,38,38,38,37,37,37,37,37,37,37,36,36,36,36,36,36,36,
    36,36,35,35,35,35,35,35,34,34,34,34,34,34,34,34,33,33,33,33,33,
    33,33,33,32,32,32,32,32,32,32,31,31,31,31,31,31,30,30,30,30
  };
  const int n4c1w4_p[] = {
    100, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,100,99,99,99,99,98,98,98,98,98,98,98,98,98,
    97,97,97,97,97,97,97,96,96,96,96,95,95,95,95,95,95,95,94,94,94,
    94,94,93,93,93,93,93,93,92,92,92,92,92,92,92,92,91,91,91,91,91,
    91,91,91,91,91,90,90,90,90,90,90,90,90,89,89,89,89,89,89,88,88,
    88,88,88,88,88,88,88,88,88,88,88,87,87,87,87,87,87,87,87,87,87,
    87,86,86,86,86,86,86,86,86,86,86,86,85,85,85,85,84,84,84,84,84,
    84,84,84,83,83,83,83,83,83,83,83,83,83,83,83,83,82,82,82,82,82,
    82,82,82,82,81,81,81,81,81,81,81,81,80,80,80,80,80,80,79,79,79,
    79,79,79,79,79,78,78,78,78,78,78,78,78,78,78,77,77,77,77,77,76,
    76,76,76,76,76,76,76,75,75,75,75,75,75,75,74,74,74,74,74,74,74,
    74,74,74,73,73,72,72,72,72,71,71,71,71,70,70,70,70,70,70,70,70,
    70,70,70,69,69,69,68,68,68,68,68,68,68,67,67,67,67,67,67,67,67,
    66,66,66,66,65,65,64,64,64,64,64,64,64,64,64,64,63,63,63,63,63,
    63,63,63,63,62,62,62,62,62,62,62,62,61,61,61,61,61,61,60,60,60,
    60,60,59,59,59,59,59,59,59,59,58,58,58,58,58,58,58,58,57,57,57,
    57,57,57,56,56,56,56,56,55,55,55,55,55,55,55,55,55,55,55,55,55,
    55,54,54,54,54,53,53,53,53,53,52,52,52,52,52,52,52,52,52,52,51,
    51,51,51,51,51,50,50,50,50,50,49,49,49,49,49,48,48,48,48,48,47,
    47,47,47,47,47,47,47,46,46,46,46,46,46,46,46,45,45,45,44,44,44,
    44,43,43,43,43,43,43,43,43,43,42,42,42,42,42,41,41,41,41,41,41,
    41,40,40,40,40,40,40,40,40,40,40,39,39,39,39,39,39,39,39,39,39,
    39,38,38,38,38,38,38,37,37,37,37,37,37,36,36,36,36,36,35,35,35,
    35,35,35,35,35,35,34,34,34,34,34,34,34,34,33,33,33,33,33,33,32,
    32,32,31,31,31,31,31,31,31,30,30,30,30,30,30,30,30,30,30
  };
  const int n4c1w4_q[] = {
    100, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,100,100,100,100,100,99,99,99,99,99,99,98,
    98,98,98,98,98,97,97,97,97,96,96,96,96,96,96,95,95,95,95,94,94,
    94,94,94,94,94,93,93,93,93,93,93,93,92,92,92,92,92,92,91,91,91,
    91,91,91,91,90,90,90,90,90,90,90,90,90,90,90,90,90,89,89,89,88,
    88,88,88,88,87,87,87,87,87,87,87,87,86,86,86,86,85,85,85,85,85,
    84,84,84,84,84,84,84,84,84,84,84,83,83,83,83,83,83,83,82,82,82,
    82,82,82,82,82,82,81,81,81,81,81,81,81,81,81,80,80,80,80,80,80,
    80,80,80,79,79,79,79,79,79,78,78,78,78,78,78,78,78,78,78,78,78,
    77,77,77,76,76,76,76,76,76,75,75,75,74,74,74,74,74,74,74,74,74,
    73,73,73,72,72,72,72,72,72,72,72,72,72,71,71,71,71,71,71,71,71,
    71,71,70,70,70,70,70,70,69,69,69,69,69,69,68,68,68,68,68,67,67,
    67,67,67,67,67,66,66,66,66,66,66,66,65,65,65,65,64,64,64,64,64,
    64,64,64,64,64,64,64,64,63,63,63,63,63,63,63,62,62,62,62,62,62,
    61,61,61,60,60,60,60,60,60,60,60,60,60,59,59,59,59,59,59,59,59,
    59,58,58,58,58,58,57,57,57,57,57,57,57,57,57,57,56,56,56,56,56,
    56,56,56,56,56,55,55,55,55,55,55,55,55,54,54,54,54,54,54,54,53,
    53,53,53,53,53,53,53,52,52,52,52,52,52,52,52,52,51,51,51,51,51,
    51,50,50,50,50,49,49,49,49,49,49,49,49,48,48,48,48,48,47,47,47,
    47,47,47,47,47,46,46,46,46,46,46,46,46,46,46,45,45,45,45,45,45,
    44,44,44,44,44,44,44,44,44,44,43,43,43,43,43,43,42,42,42,42,42,
    42,41,41,41,41,41,41,41,41,41,41,41,41,40,40,40,40,40,40,40,40,
    39,39,39,39,39,39,39,39,39,38,38,38,38,38,38,38,38,37,37,37,37,
    37,37,37,36,36,36,36,36,35,35,35,35,34,34,34,34,34,34,34,33,33,
    33,33,33,33,33,33,33,32,32,32,32,31,31,31,31,31,31,31,31,30,30
  };
  const int n4c1w4_r[] = {
    100, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,100,100,100,99,99,99,99,99,99,99,99,98,98,
    98,98,98,98,97,97,97,97,97,97,97,97,96,96,96,96,96,96,96,96,96,
    96,96,96,96,96,95,95,95,94,94,94,94,94,94,94,94,94,94,93,93,93,
    93,93,93,93,92,92,92,92,92,92,92,92,92,92,92,92,92,91,91,91,91,
    91,90,90,90,90,90,90,90,89,89,89,89,89,89,89,89,89,88,88,88,88,
    88,88,88,88,88,88,88,88,87,87,87,87,87,87,87,87,87,86,86,86,86,
    86,85,85,85,85,85,84,84,84,84,84,83,83,83,83,83,83,83,82,82,82,
    82,82,82,81,81,81,81,81,81,81,81,80,80,80,80,80,80,80,80,80,80,
    80,79,79,79,79,79,79,78,78,78,78,78,77,77,77,77,77,76,76,76,76,
    76,76,76,76,76,75,75,75,75,75,74,74,74,74,74,74,73,73,73,73,73,
    73,73,73,73,72,72,72,72,72,71,71,71,71,70,70,70,70,70,69,69,69,
    69,69,69,69,69,69,69,68,68,68,67,67,67,67,66,66,66,66,66,66,66,
    65,65,65,65,65,65,65,65,65,65,64,64,64,64,64,64,64,64,63,63,63,
    63,63,63,63,62,62,61,61,61,61,61,61,61,61,60,60,60,60,60,60,59,
    59,59,59,59,59,59,59,59,59,58,58,58,58,58,57,57,57,57,57,57,57,
    57,57,57,56,56,56,56,56,56,55,55,55,55,55,55,55,55,55,54,54,54,
    54,54,53,53,53,53,53,53,53,53,53,53,53,53,53,52,52,52,52,52,52,
    52,52,52,52,52,51,51,51,51,51,51,51,51,51,50,50,50,49,49,49,49,
    49,49,49,49,48,48,48,48,47,47,47,47,47,47,47,47,47,47,47,46,46,
    46,46,45,45,45,45,45,45,45,45,45,45,45,44,44,44,44,44,44,43,43,
    43,43,43,43,43,43,43,43,43,42,42,42,42,42,42,41,41,41,41,41,40,
    40,40,40,40,40,40,39,39,39,39,39,38,38,37,37,37,37,37,37,37,37,
    36,36,36,36,36,35,35,35,35,34,34,34,34,34,34,34,33,33,33,33,33,
    33,33,33,33,32,32,32,32,32,32,31,31,31,31,31,31,30,30,30,30
  };
  const int n4c1w4_s[] = {
    100, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,100,99,99,99,99,99,99,99,99,99,98,98,98,97,
    97,97,97,97,97,96,96,96,96,96,96,95,95,95,95,95,94,94,94,94,94,
    94,93,93,93,93,93,93,93,93,93,92,92,92,92,92,92,92,91,91,91,91,
    91,91,91,90,90,90,90,90,90,90,90,90,89,89,89,89,89,88,88,88,88,
    88,88,88,88,88,88,87,87,87,87,87,86,86,86,86,86,86,86,86,85,85,
    85,85,85,85,85,85,85,85,84,84,84,84,84,84,84,84,84,84,83,83,83,
    83,83,82,82,82,82,81,81,81,81,81,81,81,80,80,80,80,80,80,80,80,
    80,79,79,79,79,79,79,79,78,78,78,78,78,78,78,78,78,77,77,77,77,
    77,76,76,76,76,76,76,76,76,76,75,75,75,75,75,75,75,75,75,75,74,
    74,74,74,74,74,74,73,73,73,73,73,73,73,73,73,73,73,72,72,72,72,
    72,72,72,72,71,71,71,70,70,70,70,69,69,69,69,69,69,69,69,69,69,
    68,68,68,68,67,67,67,67,67,67,67,66,66,66,66,66,66,65,65,65,65,
    64,64,64,64,64,64,63,63,63,63,63,63,63,63,63,62,62,62,62,61,61,
    61,61,61,61,61,61,60,60,60,60,60,60,59,59,59,59,59,59,59,59,59,
    59,58,58,58,58,57,57,57,57,57,57,57,57,57,57,56,56,56,55,55,55,
    55,55,55,54,54,54,54,54,54,54,54,53,53,53,53,53,53,53,53,52,52,
    52,52,52,52,52,52,51,51,51,51,50,50,50,50,50,50,50,50,49,49,49,
    49,49,49,49,49,49,49,48,48,48,48,47,47,47,47,47,47,47,47,46,46,
    46,46,46,46,46,46,46,45,45,45,45,44,44,44,44,44,44,44,43,43,43,
    43,43,42,42,42,42,42,41,41,41,41,41,41,41,41,41,40,40,40,40,40,
    40,40,39,39,39,39,39,39,39,39,39,39,39,39,39,39,38,38,38,38,38,
    38,38,38,38,37,37,37,37,37,37,37,37,36,36,36,36,36,36,35,35,35,
    35,35,35,35,35,35,35,35,35,35,34,34,34,34,34,34,34,34,33,33,33,
    33,33,33,32,32,32,32,32,31,31,31,31,30,30,30,30,30,30,30
  };
  const int n4c1w4_t[] = {
    100, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,100,100,100,99,99,99,99,99,99,99,99,99,98,
    98,98,98,98,97,97,97,97,97,97,96,96,96,96,95,95,95,95,95,95,95,
    95,95,95,95,95,95,94,94,94,94,94,94,93,93,93,93,93,92,92,92,92,
    92,92,91,91,91,91,91,91,90,90,90,89,89,89,89,89,89,89,89,89,89,
    88,88,87,87,87,87,86,86,86,86,86,86,86,86,85,85,85,85,85,85,85,
    85,84,84,84,84,84,84,84,84,84,83,83,83,83,82,82,82,82,82,82,82,
    82,82,82,81,81,81,80,80,80,80,80,80,80,79,79,79,79,79,79,78,78,
    78,78,78,78,78,78,78,78,78,77,77,77,76,76,76,76,76,76,76,76,75,
    75,75,75,75,75,74,74,74,73,73,73,73,73,73,72,72,72,72,72,72,72,
    72,72,72,72,72,72,72,72,71,71,71,71,70,70,70,70,70,70,70,70,70,
    70,70,70,69,69,69,69,69,69,69,69,69,69,68,68,68,68,68,68,68,68,
    68,68,68,67,67,67,67,67,67,67,67,66,66,66,65,65,65,65,65,65,65,
    65,65,65,65,64,64,64,64,63,63,63,63,63,63,62,62,62,62,62,61,61,
    61,61,61,61,60,60,60,60,60,59,59,59,59,59,59,58,58,58,58,57,57,
    57,57,57,57,56,56,56,56,56,56,55,55,55,55,55,55,54,54,54,54,54,
    54,53,53,53,53,53,53,53,53,52,52,52,52,52,52,52,51,51,51,51,50,
    50,50,50,50,50,50,50,50,49,49,49,49,49,49,48,48,48,48,48,48,47,
    47,47,47,46,46,46,46,46,45,45,45,45,45,45,45,45,44,44,44,44,44,
    44,44,44,44,44,44,44,43,43,43,43,43,43,42,42,42,42,42,42,42,42,
    42,42,42,42,42,41,41,41,41,41,41,41,40,40,40,40,40,40,40,40,40,
    39,39,39,39,39,39,38,38,38,38,38,38,38,38,38,37,37,37,37,37,37,
    36,36,36,36,36,36,36,36,36,35,35,35,35,35,35,35,35,35,35,35,35,
    35,35,34,34,34,34,34,34,34,34,34,33,33,33,33,33,33,32,32,32,32,
    32,32,32,32,31,31,31,31,31,30,30,30,30,30,30,30,30,30,30,30
  };
  const int n4c2w1_a[] = {
    120, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,99,99,99,99,99,98,98,98,98,98,98,98,97,96,96,
    96,95,95,95,95,94,94,94,94,94,94,94,93,93,93,93,93,93,93,93,93,
    92,92,92,92,92,92,91,91,91,91,91,91,90,90,90,90,90,90,90,90,89,
    89,88,88,88,88,88,88,87,87,87,87,86,86,86,85,85,85,85,85,84,84,
    84,84,83,83,83,83,83,82,82,82,82,82,81,81,81,81,81,81,80,80,80,
    80,80,79,79,79,79,78,78,78,78,78,78,78,77,77,76,76,76,76,75,75,
    75,75,75,75,74,74,74,73,73,72,72,72,72,72,72,71,71,71,71,71,71,
    70,70,69,69,69,68,68,68,68,68,68,68,68,67,66,66,66,66,66,66,65,
    65,65,65,65,64,64,64,64,64,64,64,63,63,63,63,63,62,62,61,61,61,
    61,61,61,60,60,60,59,59,59,59,59,59,58,58,58,58,58,58,58,57,57,
    57,57,57,57,57,57,56,56,56,56,55,55,55,55,55,55,55,55,55,54,54,
    54,54,54,53,53,53,53,53,53,53,53,53,52,52,52,52,51,51,50,50,50,
    50,50,50,50,49,49,49,49,49,49,49,48,48,48,48,48,47,47,47,46,46,
    46,46,45,45,45,45,45,45,45,45,45,45,45,44,44,44,44,43,43,43,43,
    43,43,42,42,42,42,41,41,41,41,41,41,41,40,40,40,39,38,38,38,38,
    37,37,37,37,36,36,36,36,36,35,35,35,35,35,35,34,34,34,34,34,33,
    33,33,33,33,33,33,33,32,32,32,32,32,32,32,31,30,30,30,30,29,29,
    29,29,29,29,29,28,28,28,28,28,28,28,28,27,27,27,26,26,26,26,26,
    25,25,25,25,25,24,24,24,24,24,24,24,24,24,24,23,22,22,22,22,22,
    21,21,21,21,21,21,20,20,20,20,20,19,19,19,19,19,19,18,18,18,17,
    17,17,17,17,16,16,16,15,15,15,15,15,14,14,14,14,14,14,13,13,13,
    13,13,13,12,12,12,12,12,12,12,12,12,11,11,11,10,10,10,10,10,10,
    10,9,9,9,9,9,8,8,8,8,8,8,7,6,6,6,6,6,6,6,5,5,5,4,4,4,4,4,3,3,
    3,3,3,3,2,2,2,1,1,1
  };
  const int n4c2w1_b[] = {
    120, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,99,99,99,99,99,98,98,98,98,98,97,97,97,97,96,96,96,
    96,95,95,95,95,95,95,95,94,94,94,94,93,93,93,93,93,93,92,92,92,
    92,91,91,90,90,90,90,90,90,90,89,89,89,89,88,88,88,88,87,87,87,
    86,86,86,86,86,86,86,86,86,85,85,85,85,85,84,84,84,84,84,84,83,
    83,83,83,83,83,83,82,82,82,82,82,82,81,81,81,81,80,80,79,79,79,
    79,79,79,79,79,78,78,78,78,78,78,78,78,77,77,77,77,77,77,77,77,
    76,76,76,76,76,75,75,75,75,75,75,75,74,74,74,74,73,73,73,73,72,
    72,72,72,72,71,71,71,71,71,71,70,70,69,69,69,69,69,69,69,69,68,
    68,68,68,68,68,67,67,67,67,66,66,65,65,65,65,65,65,65,64,64,64,
    63,63,62,62,62,62,62,62,61,61,61,61,61,61,61,61,61,61,61,60,60,
    60,60,60,60,60,59,59,59,59,59,59,58,58,58,58,58,58,58,57,57,57,
    57,56,56,56,56,56,56,56,56,55,55,55,55,54,54,54,54,53,53,53,53,
    53,53,53,52,52,52,52,52,51,51,51,51,51,51,51,50,50,50,49,49,48,
    47,47,47,47,47,47,47,47,47,47,46,46,45,45,44,44,44,44,44,43,42,
    42,42,42,42,42,41,41,41,40,40,40,40,40,40,40,39,39,39,39,38,38,
    38,38,38,38,37,37,36,36,36,36,36,35,35,34,34,34,34,33,33,33,33,
    33,33,33,32,32,31,31,31,30,30,29,29,29,29,29,29,28,28,28,28,28,
    28,28,27,27,26,26,26,26,26,26,25,25,25,25,24,24,24,24,24,24,24,
    24,24,24,23,23,23,23,23,22,22,22,22,22,21,21,21,21,21,20,20,20,
    20,20,19,19,18,18,18,18,18,17,17,17,17,17,16,16,16,15,14,14,14,
    14,14,14,14,13,13,13,13,13,13,13,13,12,12,12,11,11,11,11,11,10,
    10,10,10,10,10,10,9,9,9,9,9,9,9,9,9,9,8,8,8,8,8,7,7,7,7,7,6,6,
    6,5,5,5,5,5,4,4,4,4,4,4,4,4,3,3,3,3,3,3,3,2,2,2,2,2,2,2,1,1,1,
    1
  };
  const int n4c2w1_c[] = {
    120, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,99,99,99,99,98,98,98,98,98,98,98,98,98,98,98,97,97,
    97,97,96,96,96,96,96,96,96,95,95,95,95,95,95,95,95,95,94,93,93,
    93,93,92,92,92,92,92,92,92,91,91,91,91,91,90,90,90,90,90,90,90,
    90,90,89,89,88,88,88,88,88,88,87,87,87,86,86,86,86,86,85,85,84,
    84,84,83,83,83,83,83,83,82,82,82,82,82,81,81,81,80,80,80,80,80,
    80,80,80,80,79,79,79,79,79,79,79,78,77,77,76,76,76,75,75,75,74,
    74,74,74,73,73,73,73,73,73,73,73,73,73,73,73,73,72,72,72,72,72,
    72,71,71,71,71,71,71,70,70,70,69,69,69,69,69,68,68,67,67,67,67,
    67,67,67,67,66,66,66,65,65,65,65,65,64,64,64,64,64,63,63,63,63,
    63,62,62,62,62,62,62,62,62,62,61,61,60,60,60,60,60,59,59,58,58,
    58,58,58,57,57,57,56,56,56,56,56,56,55,55,55,55,55,54,54,54,54,
    53,53,53,53,53,52,52,52,52,52,52,51,51,51,51,51,50,50,50,50,50,
    49,49,49,49,49,49,49,49,48,48,48,48,47,47,47,47,47,46,46,46,45,
    45,45,45,45,45,44,44,44,44,44,44,44,44,44,43,43,43,43,43,42,42,
    42,42,42,41,41,41,41,41,41,40,40,40,39,39,39,39,39,39,39,38,38,
    38,37,37,37,37,37,37,37,36,36,36,36,36,36,35,35,35,35,35,35,35,
    35,35,34,34,33,33,33,33,33,32,32,32,32,32,31,31,31,31,31,31,31,
    30,30,30,30,30,30,29,29,29,29,29,29,29,29,28,28,28,28,28,28,28,
    27,27,27,26,26,26,26,26,25,25,25,24,24,24,24,24,24,24,23,23,23,
    23,23,22,22,22,22,22,21,21,21,21,21,20,20,20,20,20,19,19,19,19,
    19,18,18,18,18,17,17,17,16,16,16,16,16,16,15,15,15,14,14,14,14,
    14,14,14,14,13,13,13,13,13,12,12,12,12,12,12,11,11,10,9,9,9,9,
    9,9,8,8,8,8,8,7,7,7,6,6,6,6,6,5,5,5,5,4,4,4,4,3,3,3,3,2,2,2,2,
    2,2,1,1,1,1,1
  };
  const int n4c2w1_d[] = {
    120, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,99,99,99,99,99,98,98,98,98,97,97,97,97,97,96,
    96,96,96,96,96,95,95,95,95,94,94,94,94,94,94,93,93,93,93,93,93,
    92,92,92,92,92,91,91,91,91,91,91,91,90,90,90,90,89,89,88,88,88,
    87,87,87,86,85,85,85,85,85,85,85,84,84,84,83,83,83,83,82,82,82,
    82,82,82,81,81,81,81,80,80,79,79,79,78,78,78,78,78,77,77,77,77,
    77,77,77,77,76,76,76,76,76,76,75,75,75,74,74,74,74,73,73,73,73,
    73,73,73,72,72,72,72,72,71,71,70,70,70,70,70,70,69,68,68,68,68,
    67,67,67,66,66,65,65,65,65,65,64,64,64,64,64,64,64,64,64,64,64,
    63,63,63,63,62,62,62,62,61,61,61,60,59,59,59,58,58,58,58,58,58,
    57,57,57,57,57,56,56,56,54,54,54,54,54,54,53,53,53,53,53,53,53,
    52,52,51,51,51,51,51,51,51,50,50,50,50,49,49,49,48,48,48,48,48,
    47,47,47,47,47,47,46,46,46,46,46,46,46,46,46,46,46,46,45,45,45,
    45,45,45,45,45,44,44,44,43,43,43,43,43,42,42,42,42,41,41,41,41,
    41,41,41,40,40,40,40,40,40,40,40,39,39,39,39,38,38,38,38,38,38,
    38,38,38,38,37,37,37,37,36,36,36,36,36,36,35,35,34,34,34,34,33,
    33,33,33,33,33,33,32,32,32,32,32,32,31,31,31,31,31,31,31,30,30,
    30,30,30,30,30,30,29,29,29,29,28,28,28,28,28,28,28,28,28,28,27,
    27,27,26,26,26,26,26,26,26,26,26,26,26,25,25,25,25,25,25,25,25,
    24,24,24,23,23,23,23,23,23,23,23,23,23,23,23,22,22,22,21,21,21,
    21,21,21,21,20,20,20,20,20,20,20,20,19,19,18,18,18,18,17,17,17,
    17,17,16,16,16,16,16,16,16,16,15,15,15,15,14,14,13,13,13,13,12,
    12,12,12,12,12,12,12,12,11,11,11,11,11,10,10,10,10,9,9,9,9,8,
    8,8,8,8,8,8,8,8,8,7,7,7,7,7,7,7,6,6,6,5,5,5,4,4,4,4,3,3,3,3,3,
    2,2,2,2,2,1,1,1
  };
  const int n4c2w1_e[] = {
    120, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,99,99,99,99,98,98,98,98,97,97,97,97,97,97,97,
    96,96,96,96,96,96,96,96,95,95,95,95,95,94,94,94,94,94,93,93,93,
    93,93,93,93,93,92,92,92,92,92,92,91,91,90,90,90,90,90,90,90,90,
    90,89,89,89,88,88,88,88,88,88,88,87,87,87,87,86,86,86,85,85,84,
    84,84,83,83,83,82,82,82,82,81,81,81,81,81,81,81,81,81,80,80,80,
    80,80,80,79,79,79,79,78,78,78,78,77,77,77,77,77,76,76,76,76,76,
    76,76,75,75,75,75,75,75,75,74,74,74,73,73,73,73,73,73,73,73,72,
    72,72,72,72,72,71,71,71,71,71,70,70,70,70,70,69,69,69,69,69,69,
    69,68,68,68,68,68,68,68,68,67,67,67,66,66,66,66,65,65,65,64,64,
    64,63,63,62,62,62,62,62,62,62,61,61,61,60,60,60,60,60,59,59,59,
    59,59,59,59,58,58,58,58,57,57,57,57,56,56,56,56,56,56,56,56,56,
    55,55,55,55,55,55,55,55,55,54,54,54,54,54,54,54,54,54,53,53,53,
    53,52,52,52,52,51,51,51,51,51,51,51,50,50,50,50,50,49,49,49,49,
    49,49,48,48,48,48,47,47,47,47,47,46,46,45,45,45,44,44,44,44,44,
    43,43,43,43,43,43,43,42,42,42,42,42,41,41,41,41,41,41,41,40,40,
    40,39,39,39,38,38,38,37,36,36,36,36,36,36,36,35,35,35,35,35,35,
    35,35,34,34,34,34,34,34,34,33,33,33,33,33,32,32,32,32,32,32,31,
    31,31,31,31,30,30,30,30,30,30,30,30,29,29,29,29,29,29,29,29,29,
    28,27,27,27,27,27,27,27,27,26,25,25,25,24,24,23,23,23,23,23,22,
    22,22,21,21,21,21,21,20,20,20,20,19,19,19,19,19,19,18,18,18,18,
    18,18,17,17,17,16,16,16,16,16,16,16,16,16,16,15,15,14,14,14,14,
    14,13,13,13,13,13,13,12,12,12,12,12,12,12,12,11,11,11,11,11,10,
    10,10,10,10,9,9,9,8,8,8,8,8,8,7,7,7,7,7,6,6,6,6,5,5,5,4,4,4,4,
    3,3,3,3,3,3,2,2,2,2,1
  };
  const int n4c2w1_f[] = {
    120, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,99,99,99,99,99,98,98,98,98,98,98,98,98,97,97,96,96,96,96,
    95,95,94,94,94,94,94,94,93,93,93,93,93,93,93,93,92,92,92,92,92,
    91,91,91,91,90,90,90,90,90,90,89,89,89,89,89,89,89,88,88,88,87,
    87,87,87,87,86,86,86,86,86,86,86,86,86,86,85,85,85,85,84,84,84,
    84,83,83,83,83,83,83,83,83,82,82,81,81,81,81,81,80,80,80,80,80,
    79,79,79,79,79,79,78,77,77,77,76,76,76,76,76,76,75,75,74,74,73,
    73,73,73,73,72,72,72,71,71,71,70,70,70,70,70,70,70,70,69,69,69,
    69,68,68,68,67,67,67,67,67,66,65,65,65,64,64,64,64,64,64,63,63,
    63,63,63,63,63,63,62,62,62,62,62,62,62,61,61,61,61,61,61,61,60,
    60,60,60,60,60,60,60,59,59,57,57,57,57,57,56,56,56,56,56,56,55,
    55,55,55,55,55,55,54,54,54,54,54,54,54,53,53,53,53,52,52,52,52,
    52,52,52,52,51,51,51,51,51,50,50,50,50,50,50,50,50,49,49,49,49,
    49,49,49,49,48,48,47,47,47,47,47,46,46,46,46,46,46,46,46,45,45,
    45,44,44,44,44,44,43,43,43,43,42,42,42,42,41,41,41,40,40,40,40,
    40,39,39,39,39,38,38,38,38,38,37,37,37,37,36,36,36,36,36,35,35,
    35,35,35,34,34,34,34,34,34,34,34,34,33,33,33,33,32,32,32,32,32,
    31,31,31,31,31,31,31,30,30,30,29,29,29,29,29,29,28,28,28,27,27,
    27,27,27,27,26,26,26,26,26,26,25,25,25,25,24,24,24,24,24,24,23,
    23,23,23,23,22,22,22,22,21,21,21,21,21,21,20,20,20,20,19,19,19,
    19,18,18,18,17,17,17,17,16,16,16,16,16,15,15,15,14,14,14,14,14,
    13,13,13,13,13,13,13,12,12,12,12,11,11,11,10,10,10,10,10,10,10,
    10,9,9,9,8,8,8,8,8,7,7,7,7,7,7,7,7,7,7,7,7,6,6,6,6,6,6,5,5,5,
    5,5,5,5,4,4,4,4,4,3,3,3,3,3,2,2,2,2,2,2,2,2,2,2,2,1,1,1,1,1
  };
  const int n4c2w1_g[] = {
    120, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,100,100,100,100,100,100,100,99,99,99,99,99,
    99,99,99,99,98,98,98,98,97,97,97,97,96,96,96,96,96,96,96,96,96,
    96,96,95,95,95,94,94,94,94,94,94,94,93,93,93,93,92,92,92,92,92,
    92,91,91,91,91,91,90,90,90,89,89,89,89,89,89,88,88,88,88,88,88,
    87,87,86,86,86,86,86,85,85,85,84,84,84,84,84,83,83,83,83,83,83,
    82,82,82,82,82,82,81,81,81,81,81,80,80,80,79,79,79,79,79,78,78,
    78,78,78,77,77,77,77,77,77,77,77,76,76,76,76,75,75,74,74,74,74,
    74,73,73,73,73,73,73,72,72,72,72,71,71,71,71,71,70,70,70,70,70,
    70,70,69,69,69,69,69,68,68,68,67,67,67,66,66,65,64,64,64,63,63,
    63,63,63,62,62,62,62,61,60,60,60,60,59,59,59,59,59,58,58,58,58,
    58,57,57,57,57,57,56,56,55,55,55,55,55,54,54,54,53,53,53,53,53,
    52,52,52,52,52,51,51,51,51,51,50,50,50,50,49,49,49,49,49,49,48,
    48,48,48,47,47,47,47,47,47,47,47,47,46,46,46,46,46,46,46,45,45,
    45,45,45,44,44,44,44,44,44,43,43,43,43,42,41,41,41,41,40,40,40,
    40,40,39,39,39,38,38,38,38,38,38,38,38,38,37,37,37,37,37,36,36,
    36,36,36,36,36,36,36,35,35,35,35,35,34,34,34,34,34,33,33,33,33,
    33,33,33,33,32,32,32,32,32,31,31,31,30,30,30,30,30,30,29,29,29,
    29,29,29,29,29,29,29,29,28,27,27,27,27,27,27,26,26,26,26,26,26,
    26,26,26,25,25,25,25,24,24,24,24,24,24,24,23,22,22,22,22,22,21,
    21,21,20,20,20,19,19,19,19,19,19,18,18,18,17,17,17,17,17,17,17,
    17,17,17,16,16,16,16,16,15,15,15,14,14,14,14,14,13,13,13,13,13,
    13,12,12,12,12,12,11,11,11,11,10,10,10,10,10,10,10,10,9,9,9,9,
    9,9,9,9,8,8,8,8,8,8,8,7,7,7,7,6,6,6,5,5,5,4,4,4,4,3,3,3,2,2,2,
    2,2,2,2,1,1,1,1,1,1
  };
  const int n4c2w1_h[] = {
    120, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,99,99,99,99,99,98,98,98,98,98,97,97,97,97,97,
    96,96,96,96,96,96,96,96,96,96,96,95,95,94,94,94,94,94,93,93,93,
    93,93,93,92,92,92,91,91,91,91,90,90,90,89,89,89,89,89,88,88,88,
    88,87,87,87,87,86,86,86,86,85,85,85,85,85,85,84,84,84,84,84,84,
    84,84,83,83,83,82,82,82,82,81,81,81,81,81,81,81,80,80,80,80,80,
    80,80,79,79,79,79,79,79,79,79,79,78,78,78,78,78,78,78,78,78,78,
    77,77,77,77,77,77,77,77,76,76,76,76,76,74,74,74,74,74,73,73,73,
    73,73,73,72,72,72,71,71,71,71,70,70,70,70,70,70,70,69,69,69,69,
    69,69,68,68,68,68,68,67,67,67,67,67,66,66,66,65,65,65,65,64,64,
    64,64,63,63,63,63,63,63,63,63,63,63,62,62,62,62,62,62,61,61,61,
    61,61,61,60,60,60,60,60,60,60,60,59,58,58,58,58,57,57,56,56,56,
    56,55,55,55,54,54,54,54,54,54,54,53,53,53,53,53,53,53,52,52,52,
    52,52,52,52,52,52,51,51,51,51,50,50,50,50,50,49,49,48,48,48,47,
    47,46,46,46,46,46,46,46,45,45,44,43,43,43,43,42,42,42,42,42,42,
    41,41,41,41,40,40,40,40,39,39,39,39,39,39,38,38,38,38,38,38,38,
    38,37,37,37,37,37,37,37,36,36,36,36,35,35,35,35,35,35,35,34,34,
    34,34,34,34,33,33,33,33,32,32,32,32,32,32,32,32,31,31,31,31,30,
    30,30,30,30,30,29,29,29,29,29,28,28,28,28,28,27,27,27,27,26,26,
    26,26,26,26,26,26,26,25,25,25,24,24,24,24,24,23,23,23,23,23,23,
    23,22,22,22,22,21,21,21,20,20,20,19,19,19,19,19,19,18,18,18,18,
    18,18,18,17,17,17,17,17,17,16,16,16,16,16,15,15,15,15,14,14,14,
    13,13,12,12,12,12,12,12,12,11,11,11,11,11,11,10,10,10,9,9,9,9,
    9,8,8,8,8,7,7,7,6,6,6,6,6,6,6,6,6,5,5,5,5,5,5,4,4,3,3,3,3,2,2,
    2,2,2,1,1,1,1,1
  };
  const int n4c2w1_i[] = {
    120, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,99,99,99,99,99,99,99,99,99,99,99,98,98,98,
    98,98,98,98,97,97,97,96,96,96,96,96,96,96,96,95,95,95,95,94,94,
    94,94,94,93,92,92,92,92,91,91,91,91,91,91,90,90,90,90,90,89,89,
    89,89,89,88,88,88,88,88,87,87,87,86,86,86,86,85,85,85,85,84,84,
    84,84,84,84,83,83,83,83,83,83,82,82,82,82,82,82,82,82,81,81,81,
    81,81,80,80,80,80,79,79,79,79,78,78,78,77,77,77,76,76,75,75,74,
    74,74,74,74,73,73,73,72,72,72,72,72,72,72,72,71,71,71,71,71,70,
    70,70,70,70,70,70,70,69,69,69,69,68,68,67,67,67,67,67,67,67,66,
    66,66,65,65,65,65,64,64,64,64,64,64,64,64,64,64,63,63,63,63,63,
    63,63,62,62,62,62,62,61,61,61,61,61,60,60,60,59,59,58,58,58,58,
    58,58,57,57,57,57,56,56,56,56,55,55,55,55,55,55,54,54,54,54,53,
    53,53,52,52,52,52,52,51,51,51,51,51,50,50,50,50,50,50,50,50,50,
    49,49,49,48,48,48,47,47,47,47,47,47,46,46,46,46,46,45,45,45,45,
    44,44,44,43,43,43,43,43,43,43,42,42,42,42,42,42,42,42,41,41,41,
    41,41,41,40,40,40,40,40,40,40,39,39,39,39,39,38,38,38,38,37,37,
    37,37,37,36,36,36,36,36,36,36,36,35,35,34,34,34,34,34,34,34,34,
    33,33,33,33,33,32,32,31,31,31,31,31,31,30,29,29,29,28,28,28,28,
    28,28,28,27,27,27,27,26,26,26,26,26,26,26,26,25,25,25,25,25,25,
    24,24,24,24,24,24,24,24,24,23,23,23,22,22,22,21,21,21,21,21,21,
    20,20,20,20,20,20,19,19,19,19,18,18,18,18,18,18,18,18,18,18,17,
    17,17,17,17,16,16,16,16,16,15,15,15,15,15,14,14,14,14,14,13,13,
    13,13,13,12,12,12,12,11,11,11,11,11,11,10,10,10,10,10,9,9,9,8,
    7,7,7,7,7,7,7,7,7,6,6,6,5,5,5,5,5,5,5,5,5,4,4,3,3,3,3,3,2,2,2,
    2,2,2,2,2,2,1,1
  };
  const int n4c2w1_j[] = {
    120, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,99,99,98,98,98,98,97,97,97,97,97,97,96,96,96,
    96,95,95,94,94,94,94,94,94,94,94,93,93,93,93,93,93,93,93,92,92,
    92,92,91,91,91,90,90,89,89,89,89,89,89,89,89,88,88,88,87,87,87,
    87,86,86,86,86,85,85,85,85,85,84,84,83,83,83,82,82,82,82,82,82,
    81,81,81,81,81,81,81,81,80,80,80,80,80,80,80,79,79,79,79,78,78,
    78,78,78,78,78,78,77,77,76,76,76,76,76,76,75,75,75,75,75,75,75,
    75,74,74,74,74,73,73,73,73,73,73,73,72,72,72,72,72,72,72,72,72,
    71,71,70,70,70,69,69,69,69,69,68,68,67,67,67,67,66,66,66,66,66,
    66,66,65,65,65,65,65,65,64,64,64,64,63,63,62,62,61,61,61,60,60,
    60,59,59,59,59,59,58,58,58,58,58,58,58,58,58,57,57,57,57,57,57,
    56,56,55,55,55,55,55,55,54,54,54,53,53,53,52,52,52,52,52,51,51,
    51,51,51,51,51,51,50,50,50,49,49,49,49,49,49,48,48,48,48,48,48,
    47,47,47,47,47,47,46,45,45,45,45,45,44,44,44,44,44,44,43,43,43,
    42,42,42,42,42,42,41,41,41,41,41,41,41,41,40,40,40,40,40,40,39,
    39,39,39,39,39,39,39,38,38,37,37,37,37,37,37,36,36,36,36,36,36,
    36,36,36,36,35,35,35,35,34,34,33,33,33,33,33,33,32,32,32,32,32,
    31,30,30,30,30,30,30,30,30,30,29,29,29,29,29,29,29,29,28,28,28,
    28,27,27,27,27,26,26,26,25,25,25,25,25,24,24,24,24,24,23,23,23,
    22,22,22,22,21,21,21,21,21,20,20,20,20,20,20,20,20,19,19,19,19,
    18,18,18,18,18,17,17,17,17,17,17,16,16,16,16,15,15,15,15,14,14,
    14,14,13,13,13,13,13,13,12,12,12,12,12,12,11,11,11,11,10,10,10,
    10,9,9,9,9,9,9,9,9,9,9,8,8,8,8,8,8,8,8,8,8,8,8,7,7,7,7,7,6,6,
    6,6,6,5,5,5,5,5,4,4,4,3,3,3,3,2,2,2,2,2,2,2,2,2,2,2,2,1,1,1,1
  };
  const int n4c2w1_k[] = {
    120, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,100,100,99,99,98,98,98,97,97,97,97,97,96,
    96,96,96,95,95,95,95,95,95,95,95,94,94,94,94,94,93,93,93,93,93,
    92,92,92,92,92,91,91,91,91,91,90,90,90,89,89,88,88,88,88,88,88,
    88,88,88,87,87,86,86,86,86,86,85,85,85,85,85,85,85,85,85,84,84,
    84,84,83,83,83,83,83,83,82,82,82,82,82,81,81,81,81,80,80,80,80,
    80,79,79,79,79,79,79,78,78,78,78,78,77,77,77,77,77,77,76,76,76,
    76,76,76,75,75,75,75,75,74,74,74,74,74,74,73,73,73,73,72,72,71,
    71,71,71,70,70,70,70,69,69,69,69,68,68,68,67,67,66,66,66,66,66,
    66,66,66,65,65,65,64,64,64,64,64,64,64,64,64,64,64,63,63,63,62,
    62,62,62,62,61,61,61,61,61,60,60,60,60,60,59,59,59,59,59,58,58,
    57,57,57,57,56,56,56,56,56,56,56,56,55,55,55,55,55,55,55,55,54,
    54,54,54,53,53,53,53,53,53,53,53,52,52,52,52,52,52,52,51,51,51,
    50,50,50,50,50,50,50,49,49,49,49,48,48,48,48,48,48,48,48,47,47,
    46,46,46,46,46,45,45,45,45,45,45,45,44,44,44,44,44,44,44,44,44,
    44,43,43,43,42,42,42,42,41,41,41,40,40,40,40,39,39,39,39,39,39,
    39,39,38,37,37,37,37,37,36,36,36,36,36,35,35,35,35,34,34,34,34,
    33,33,33,32,32,32,31,31,31,31,31,31,30,30,30,30,30,30,30,30,30,
    29,29,29,28,28,28,28,28,28,28,27,27,27,27,27,27,27,27,26,26,26,
    26,26,26,26,26,25,25,25,25,25,25,24,24,24,24,24,23,23,22,22,22,
    22,22,22,22,21,21,21,21,20,20,20,20,20,20,20,19,19,19,19,19,19,
    19,18,18,18,18,18,17,17,16,16,16,16,16,15,15,15,14,14,13,13,12,
    12,12,12,12,11,11,11,11,11,11,11,11,11,11,10,10,10,10,10,10,10,
    10,9,9,9,8,8,8,8,7,6,6,6,6,6,6,6,6,5,5,5,5,4,4,4,4,3,3,3,3,3,
    3,3,2,2,2,2,1,1,1,1,1
  };
  const int n4c2w1_l[] = {
    120, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,99,99,99,99,99,99,99,98,98,98,97,97,96,96,95,95,95,
    95,95,95,95,95,94,94,94,94,93,93,93,93,93,93,93,92,92,92,92,92,
    92,92,91,91,90,90,90,89,89,89,89,88,88,88,87,87,87,87,87,87,87,
    86,86,86,86,86,86,85,85,85,85,85,85,85,85,85,84,84,84,84,84,84,
    84,84,84,83,83,83,83,83,83,83,82,82,82,81,81,81,81,80,80,80,80,
    79,79,79,79,78,78,78,78,78,77,77,77,77,76,76,76,76,75,75,75,75,
    74,74,74,73,73,73,73,73,72,72,71,71,71,71,71,71,70,70,70,70,70,
    70,70,70,69,69,69,69,69,69,69,69,69,69,68,68,68,68,68,68,67,67,
    67,66,66,66,65,65,64,64,64,64,64,63,63,63,62,62,62,62,62,62,62,
    62,62,61,61,61,61,61,60,60,60,60,60,59,59,59,59,59,59,58,58,58,
    58,58,58,57,57,57,57,57,56,56,56,56,56,56,56,56,55,55,55,55,55,
    55,55,55,54,54,54,54,54,54,54,53,53,53,52,52,52,52,52,51,51,50,
    50,50,50,49,49,49,49,49,49,49,48,48,48,48,48,48,47,47,47,47,47,
    46,46,46,46,46,46,46,45,45,45,44,44,44,43,43,42,42,42,42,41,41,
    41,41,41,40,40,40,40,40,40,40,40,39,39,39,39,39,39,38,38,38,38,
    38,38,37,37,37,37,37,36,36,36,36,35,35,35,35,34,34,34,34,33,33,
    33,32,32,32,32,32,32,32,31,31,31,31,31,31,31,30,30,30,30,30,30,
    30,29,29,29,29,29,29,29,29,28,28,28,27,27,27,26,26,26,26,26,25,
    25,25,25,24,24,24,24,24,24,24,23,23,23,23,22,22,22,22,22,22,21,
    21,21,21,21,21,21,21,20,20,20,20,20,20,20,19,19,19,19,18,18,18,
    18,18,18,17,17,17,17,17,16,16,16,16,16,15,14,13,13,13,13,12,12,
    12,12,12,11,11,10,10,10,10,9,9,9,9,9,9,8,8,8,8,7,7,7,7,6,6,5,
    5,5,4,4,4,4,4,4,4,4,4,4,3,3,3,3,3,3,3,2,2,2,2,2,2,1,1,1,1,1,1,
    1,1,1
  };
  const int n4c2w1_m[] = {
    120, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,100,100,99,99,99,99,99,99,98,98,98,97,97,
    97,97,96,96,96,96,96,96,96,95,95,95,95,94,94,94,94,93,93,93,93,
    93,93,93,93,93,93,93,92,92,91,91,91,91,90,90,90,90,89,89,89,89,
    89,89,89,89,89,88,88,88,88,87,87,87,87,86,86,86,86,86,86,86,86,
    86,85,85,85,85,85,85,84,84,84,83,83,83,83,82,82,82,82,82,82,81,
    81,81,81,80,80,80,80,80,80,79,79,79,78,78,78,78,77,77,77,77,77,
    77,77,76,76,76,76,76,75,75,75,75,75,75,75,74,74,74,74,74,73,73,
    73,72,72,72,72,72,72,72,71,71,71,71,71,71,70,70,70,70,69,69,68,
    68,68,68,68,68,68,68,67,67,67,67,67,67,66,66,66,66,66,66,66,66,
    65,65,65,65,65,64,64,64,64,63,63,63,63,62,62,62,61,61,61,60,60,
    60,60,60,60,60,60,60,60,60,60,59,59,59,59,59,58,58,58,57,57,57,
    57,57,57,57,56,56,55,55,55,55,55,54,54,54,54,54,54,54,53,53,53,
    53,53,53,52,52,52,52,52,51,51,51,51,51,51,51,51,50,50,50,50,50,
    49,49,49,49,49,49,49,49,48,48,48,48,48,48,47,47,47,46,46,46,45,
    45,45,45,44,44,44,44,44,44,44,43,43,43,42,42,42,41,41,41,41,40,
    40,39,39,39,39,38,38,38,38,38,38,37,37,37,37,36,36,36,36,36,36,
    35,35,34,34,34,34,34,33,33,33,33,33,32,32,32,32,32,32,32,31,31,
    31,30,30,30,30,30,30,29,29,29,28,28,28,28,28,28,28,28,27,27,27,
    27,26,26,26,26,26,26,26,26,26,26,25,25,25,25,25,24,24,24,23,23,
    23,23,22,22,22,21,21,21,21,20,20,20,20,20,20,20,20,20,19,19,19,
    19,18,18,18,18,18,17,17,17,17,17,17,17,16,16,16,15,15,15,15,15,
    14,14,14,14,14,14,14,13,13,13,13,13,13,12,12,12,12,11,11,11,11,
    10,10,10,10,10,9,9,9,9,9,9,8,8,8,8,8,8,8,7,7,7,7,7,6,6,6,5,5,
    5,5,5,5,5,4,3,3,2,2,1,1,1
  };
  const int n4c2w1_n[] = {
    120, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,99,99,99,99,99,98,98,98,98,98,97,97,96,96,96,
    96,95,95,95,94,94,94,94,94,93,93,93,93,93,93,92,92,92,91,91,91,
    91,91,91,91,90,90,90,89,89,88,88,88,88,88,88,88,88,87,87,87,87,
    87,87,87,87,87,86,86,86,86,86,86,86,85,85,84,84,84,84,83,83,83,
    83,83,82,82,82,82,82,81,81,81,81,80,80,80,80,80,80,79,79,79,79,
    78,78,78,78,78,78,78,77,77,76,76,75,75,75,75,75,75,75,75,75,74,
    74,74,74,74,74,74,74,73,73,73,73,73,72,72,72,72,72,71,70,70,69,
    69,69,69,69,69,69,68,68,68,68,67,67,67,67,67,67,66,66,66,66,66,
    66,65,65,65,65,65,64,64,64,64,64,64,64,64,64,63,63,63,63,63,63,
    63,62,62,62,62,61,61,61,61,61,61,61,60,60,60,60,59,59,59,59,59,
    59,58,58,58,58,58,57,57,57,57,56,56,56,56,56,56,56,55,55,55,54,
    54,54,54,53,53,52,52,52,52,52,52,52,51,51,51,51,51,50,50,50,50,
    50,50,50,49,49,49,49,48,48,48,48,48,48,48,48,48,47,47,47,47,47,
    47,47,47,47,46,46,46,46,46,46,45,45,45,45,44,44,44,44,44,44,43,
    43,43,43,42,42,42,41,41,41,41,41,40,40,40,40,40,40,39,39,39,39,
    39,39,39,38,38,38,38,38,38,37,37,37,37,37,36,36,36,35,35,35,35,
    34,34,34,33,33,33,32,32,32,32,32,31,31,31,31,31,31,30,30,30,30,
    30,30,30,29,29,29,29,29,28,28,27,27,27,27,27,27,26,26,25,25,25,
    25,25,25,25,25,24,24,24,24,24,24,24,24,24,23,23,22,22,22,22,21,
    21,21,21,21,20,20,20,20,20,19,19,19,19,18,18,18,18,18,17,17,17,
    17,17,17,16,16,16,16,16,16,15,15,15,15,15,14,14,14,14,14,13,13,
    13,13,13,13,12,12,12,12,11,11,11,11,11,11,11,10,10,10,10,10,10,
    9,9,9,9,9,9,9,8,8,8,8,7,7,7,7,6,6,6,5,5,5,5,5,4,4,4,4,3,3,3,3,
    2,2,2,2,2,1,1,1,1
  };
  const int n4c2w1_o[] = {
    120, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,100,99,99,99,99,98,98,98,98,98,97,97,97,97,
    96,96,96,96,96,96,96,96,95,95,95,95,94,94,93,93,93,93,93,93,93,
    92,92,92,92,92,92,91,91,91,91,90,90,90,90,90,89,89,89,89,89,88,
    88,88,88,87,87,87,87,86,86,85,85,85,85,84,84,84,84,83,83,83,82,
    82,82,82,81,81,81,81,81,81,81,81,81,81,81,81,81,80,80,80,79,79,
    79,79,79,79,79,79,79,78,78,78,78,77,77,77,77,77,77,76,76,76,76,
    76,76,76,75,75,74,74,74,74,74,74,74,74,73,73,73,73,72,72,72,72,
    72,72,72,72,71,71,71,71,71,71,71,71,71,71,70,70,70,70,70,70,70,
    69,69,69,69,69,68,67,67,66,66,65,65,65,65,65,65,65,64,64,63,63,
    63,63,63,63,63,63,63,63,63,62,62,62,61,61,61,61,61,61,60,60,60,
    60,59,59,59,59,59,59,58,58,58,58,58,57,57,57,56,56,56,56,56,56,
    56,56,55,55,55,55,55,54,54,54,54,54,53,53,53,53,53,53,53,52,51,
    51,50,50,50,50,49,49,49,48,48,47,47,47,47,47,47,47,47,47,47,47,
    47,46,46,46,46,46,45,45,45,45,44,44,44,44,44,43,43,43,43,42,42,
    42,42,42,42,42,42,41,41,41,40,40,39,39,39,39,39,38,38,38,38,38,
    37,37,37,37,37,36,36,36,36,36,36,35,35,35,35,35,35,34,34,34,34,
    34,34,33,33,33,33,33,32,32,32,32,31,31,31,31,30,30,30,30,30,29,
    29,29,29,29,29,29,29,29,28,28,28,28,28,27,27,27,27,27,27,26,26,
    26,26,26,25,25,25,25,25,25,25,25,24,24,24,24,24,24,24,23,22,22,
    22,22,21,21,21,21,21,21,20,19,19,19,19,19,18,18,18,18,18,17,17,
    17,17,17,17,16,16,16,16,15,15,15,15,14,14,14,14,14,13,13,13,13,
    13,12,12,12,12,12,12,12,11,11,11,11,11,10,10,10,10,9,9,9,9,8,
    8,8,7,7,6,6,6,6,5,5,5,5,4,4,4,4,4,4,4,4,4,4,4,3,3,3,3,3,2,2,2,
    1,1,1,1,1,1,1,1
  };
  const int n4c2w1_p[] = {
    120, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,100,100,99,99,99,99,99,99,98,98,98,97,97,
    97,96,96,96,96,96,96,96,95,95,95,95,95,94,94,93,93,93,92,92,92,
    92,92,92,92,91,91,90,90,90,90,90,90,90,89,89,89,89,89,88,88,88,
    87,87,87,87,87,86,85,85,85,85,85,85,85,85,84,84,84,84,84,84,84,
    84,83,83,83,83,83,82,82,82,82,82,82,81,81,81,81,81,81,81,80,80,
    80,79,79,78,78,78,78,78,78,78,77,77,77,77,77,76,76,76,76,76,76,
    76,75,75,75,74,74,74,74,74,74,74,74,73,73,72,72,72,71,71,71,70,
    70,70,70,70,70,70,70,69,69,69,69,69,69,68,68,68,68,68,68,68,68,
    68,68,67,67,67,67,66,66,66,66,66,66,66,65,65,65,65,65,64,64,64,
    64,64,64,64,63,63,63,63,63,62,62,62,62,61,61,61,61,60,60,60,60,
    59,59,59,59,59,58,58,58,57,57,57,57,56,56,55,55,55,55,55,55,54,
    54,54,54,54,53,53,53,53,53,53,53,53,53,53,53,53,53,53,52,52,52,
    51,51,51,51,51,51,51,50,50,50,50,50,50,49,49,49,49,49,48,48,48,
    48,48,48,48,47,47,47,47,47,47,47,47,46,46,46,46,46,45,45,45,45,
    44,44,44,44,44,44,44,43,43,43,43,42,42,42,42,42,41,41,41,41,41,
    40,40,40,39,39,38,38,38,38,38,38,37,37,37,37,36,36,36,35,35,35,
    35,35,35,35,34,34,34,34,34,33,33,33,32,32,32,32,31,31,31,31,31,
    30,30,30,30,29,29,29,29,29,29,28,28,28,27,27,26,26,26,26,26,26,
    26,26,26,26,25,25,24,24,24,24,24,24,23,23,23,23,23,23,23,23,22,
    22,22,21,21,21,20,20,20,19,19,19,19,19,19,18,18,17,17,16,16,16,
    16,16,16,15,15,15,15,15,15,14,14,14,14,14,14,14,14,13,13,13,13,
    13,13,13,13,13,13,12,12,12,12,11,11,11,11,11,11,11,11,10,9,9,
    9,9,8,8,8,8,8,8,7,7,7,7,6,6,6,6,6,5,5,5,5,5,4,4,3,3,3,3,3,3,2,
    2,2,2,2,2,2,2,1,1,1
  };
  const int n4c2w1_q[] = {
    120, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,100,100,99,99,99,98,98,98,98,98,98,98,98,
    97,97,97,97,97,97,97,97,97,96,96,96,96,96,96,96,96,95,95,95,95,
    95,94,94,94,94,94,94,94,93,93,93,92,91,91,91,91,90,90,89,89,89,
    89,89,89,89,89,88,88,88,88,88,88,87,87,87,87,86,86,86,86,85,85,
    85,85,85,85,85,84,84,84,84,84,84,84,84,84,83,83,83,83,82,82,81,
    81,81,80,80,80,79,79,79,78,78,77,77,77,77,77,76,76,76,75,75,75,
    75,75,75,74,74,74,74,73,73,73,73,73,73,73,73,72,72,72,72,72,72,
    72,72,72,72,71,71,71,71,71,71,71,70,70,69,69,69,69,69,68,68,68,
    67,67,67,66,66,66,66,66,65,65,65,65,65,65,64,64,64,64,63,63,63,
    63,62,62,62,61,61,61,61,61,61,61,61,61,60,60,60,60,60,60,59,59,
    59,59,59,59,59,58,58,58,58,58,57,56,56,56,56,55,55,55,55,55,55,
    55,54,53,53,53,53,53,52,52,52,52,52,52,52,52,52,52,51,51,51,51,
    51,51,51,50,50,49,49,49,48,48,48,48,48,48,48,48,47,47,47,46,46,
    46,46,46,45,45,45,45,44,44,44,44,44,44,44,44,44,43,43,43,43,42,
    42,42,41,41,41,41,41,40,40,40,40,40,39,39,39,39,39,39,39,39,38,
    38,38,37,37,37,37,36,36,36,36,35,35,35,34,34,34,34,34,34,34,33,
    33,33,33,33,33,33,32,32,32,32,31,31,31,31,30,30,30,30,29,29,29,
    29,29,29,28,28,28,28,28,28,27,27,27,27,27,26,26,25,25,25,25,24,
    24,24,23,23,23,23,23,23,23,23,22,22,22,22,22,22,21,21,21,21,20,
    20,20,20,20,19,19,19,19,19,19,19,19,19,19,18,18,18,18,17,17,17,
    17,17,17,17,16,16,16,15,15,15,14,14,14,13,12,12,12,12,11,11,11,
    10,10,10,10,10,10,10,9,9,9,9,9,9,9,9,9,8,8,8,8,8,7,7,7,7,7,7,
    7,7,7,7,6,6,5,5,5,5,5,5,4,4,4,4,4,3,3,3,3,3,2,2,2,2,2,1,1,1,1,
    1,1,1,1
  };
  const int n4c2w1_r[] = {
    120, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,100,100,99,99,99,99,99,99,99,99,98,98,98,
    98,97,97,97,97,97,97,97,96,96,96,96,96,96,95,95,95,95,95,94,93,
    93,93,93,93,93,93,93,92,92,92,92,92,92,92,92,91,91,91,91,91,90,
    90,89,89,89,89,89,89,89,88,88,87,87,87,87,87,87,86,86,86,86,86,
    86,86,86,86,86,86,85,85,85,83,83,83,83,83,82,82,82,82,82,82,81,
    80,80,80,80,79,79,79,78,78,78,78,78,78,77,77,77,77,77,76,76,76,
    76,76,76,76,76,75,75,75,75,75,75,74,74,74,73,73,73,73,72,72,71,
    71,71,71,71,70,70,70,70,70,69,69,69,69,69,69,69,68,68,67,66,66,
    65,65,65,65,65,65,64,64,64,64,64,63,63,63,63,63,63,62,62,62,62,
    62,61,61,61,61,61,61,61,61,61,61,61,60,60,60,60,60,60,60,59,59,
    59,59,59,59,58,58,58,58,58,57,57,57,57,56,56,56,56,56,56,55,55,
    55,55,55,54,54,54,53,53,53,53,53,53,52,52,52,52,52,52,51,51,51,
    51,51,50,50,49,49,49,49,49,48,48,48,48,48,48,48,48,47,47,47,46,
    46,45,45,45,45,45,45,45,45,45,45,45,45,44,43,43,43,43,43,43,43,
    42,42,42,42,42,42,42,42,42,41,41,41,41,40,40,40,40,40,40,39,39,
    39,39,39,39,39,38,38,38,37,37,37,37,37,37,37,36,36,36,36,36,36,
    35,35,35,35,34,34,34,34,34,34,34,33,33,33,33,32,32,32,31,31,31,
    31,31,30,30,30,29,29,29,29,29,28,28,28,28,28,27,27,27,27,27,27,
    27,26,26,26,26,26,25,25,25,25,25,24,24,24,24,24,24,24,24,23,23,
    22,22,22,22,21,21,21,20,20,20,19,19,19,19,19,19,18,18,18,18,17,
    17,17,16,16,16,16,16,16,16,15,15,15,15,14,13,13,13,13,12,12,12,
    12,12,12,12,12,12,11,11,11,10,10,10,10,10,10,10,9,9,8,8,8,7,7,
    7,7,7,7,7,7,7,6,6,6,5,5,5,5,5,5,4,4,4,4,4,4,4,3,3,3,3,3,3,3,2,
    1,1,1,1,1,1,1,1
  };
  const int n4c2w1_s[] = {
    120, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,99,99,99,98,98,98,98,97,97,97,97,96,96,96,96,96,96,
    95,95,95,94,94,94,94,94,94,94,93,93,93,93,93,93,93,93,93,92,91,
    91,91,91,91,90,90,90,90,90,90,89,89,89,89,89,88,88,88,88,88,88,
    88,88,87,87,87,86,86,86,85,85,85,85,85,85,85,85,85,84,84,84,84,
    83,83,83,83,83,82,82,82,82,82,82,82,81,81,81,81,81,80,80,80,80,
    80,80,80,79,79,79,79,78,77,77,77,77,77,76,76,76,75,74,74,74,74,
    73,73,73,73,73,73,72,72,72,72,72,72,71,71,71,70,70,70,69,69,69,
    68,68,68,68,68,68,68,68,68,67,66,66,66,66,66,66,65,65,65,65,65,
    65,65,65,65,65,65,65,64,64,63,63,63,63,63,63,63,63,63,62,62,62,
    62,62,62,62,62,62,61,61,61,61,61,61,61,61,61,60,60,60,60,59,59,
    59,59,58,58,58,57,57,57,57,56,56,56,56,56,56,55,55,54,54,54,54,
    53,53,53,53,53,53,52,52,52,52,51,51,51,51,51,50,50,50,50,49,49,
    49,49,48,48,48,47,47,47,47,46,46,46,46,46,46,46,46,45,45,45,45,
    45,45,45,44,44,44,44,44,43,43,43,43,43,43,43,43,43,42,42,42,42,
    42,42,42,42,41,41,41,41,41,41,40,40,40,40,40,40,40,39,39,39,39,
    39,39,38,38,37,37,37,36,36,36,36,36,36,35,35,35,35,35,35,35,34,
    34,34,34,34,33,33,33,33,33,33,32,32,32,32,32,32,32,31,31,31,31,
    31,31,30,30,30,30,30,29,29,29,29,28,28,28,27,27,27,27,26,26,26,
    26,26,26,26,25,25,24,24,24,24,24,24,23,23,23,22,22,22,22,21,21,
    21,21,21,20,20,19,19,19,19,19,19,19,19,19,18,18,18,18,17,17,17,
    17,17,17,17,17,16,16,16,16,15,15,14,14,14,14,13,12,12,12,12,12,
    12,11,11,11,11,11,11,11,11,10,10,10,9,9,9,9,9,9,9,9,9,8,8,8,8,
    8,8,8,8,8,7,7,7,7,6,6,6,6,6,6,5,5,5,5,5,5,5,5,4,4,4,3,3,3,2,2,
    2,1,1,1
  };
  const int n4c2w1_t[] = {
    120, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,100,99,99,99,99,99,99,99,98,98,98,98,98,98,
    97,97,97,97,97,97,96,96,96,96,96,96,96,96,95,95,95,94,94,94,94,
    94,94,94,94,93,93,93,93,92,92,92,92,92,92,91,91,91,91,90,90,90,
    90,90,89,89,89,89,89,89,89,89,88,88,88,88,87,87,87,87,86,86,85,
    85,85,84,84,84,84,84,84,84,84,83,83,83,83,82,82,82,82,82,81,81,
    81,81,81,81,81,81,80,80,80,80,79,79,79,79,79,78,78,78,78,77,77,
    77,77,76,76,76,75,75,75,75,75,75,75,74,74,74,74,74,73,73,73,72,
    72,72,71,71,71,70,70,70,70,69,69,69,69,69,69,68,68,68,67,67,67,
    67,67,67,67,67,67,66,66,66,66,66,66,66,65,65,65,65,64,64,64,64,
    64,63,63,63,62,62,62,62,61,61,61,61,61,61,61,60,60,60,59,59,59,
    59,59,59,58,58,58,58,58,58,57,57,57,57,57,56,56,56,55,55,55,54,
    54,54,53,53,53,53,53,53,52,52,52,52,51,51,51,51,51,50,50,50,50,
    50,49,49,49,49,49,49,49,49,48,48,48,48,48,47,47,47,46,46,46,46,
    46,46,45,45,45,45,44,44,44,44,44,44,44,44,43,43,43,43,43,43,42,
    42,42,42,42,42,42,41,41,41,41,41,41,40,40,40,39,39,39,39,38,37,
    37,37,37,37,37,37,37,36,36,36,36,36,35,35,34,34,34,34,33,33,33,
    33,33,33,33,32,32,32,31,31,31,31,31,31,31,30,30,29,29,29,29,29,
    29,27,27,26,26,25,25,25,25,25,25,25,25,24,24,24,24,24,24,24,24,
    24,24,23,23,23,23,22,22,22,22,21,21,21,21,21,21,21,21,21,21,20,
    20,20,20,19,19,19,19,19,19,18,18,18,18,18,18,18,18,18,17,17,17,
    17,17,17,16,16,16,16,15,14,14,14,14,14,14,14,14,13,13,13,13,12,
    12,12,12,12,12,12,12,12,11,11,10,10,10,10,9,9,9,9,8,8,8,8,8,8,
    7,7,7,7,7,7,7,6,6,6,6,6,6,6,5,5,5,5,5,4,4,4,4,3,3,3,3,3,3,2,2,
    2,2,2,2,2,1
  };
  const int n4c2w2_a[] = {
    120, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,99,99,99,99,99,99,99,99,99,99,98,97,97,97,97,
    97,97,97,96,96,96,96,96,96,96,96,96,96,95,95,95,95,95,95,95,95,
    95,94,94,94,94,94,94,94,94,94,93,93,93,93,92,92,92,92,92,92,92,
    92,92,91,91,91,91,91,91,90,90,90,90,90,89,89,89,89,89,89,89,89,
    89,88,88,88,88,88,88,87,87,87,87,87,86,86,86,86,86,86,86,85,85,
    85,85,85,85,85,84,84,84,84,84,84,84,83,83,82,82,82,82,82,81,81,
    81,81,81,80,80,80,80,80,80,80,79,79,79,79,79,79,78,78,78,78,78,
    78,78,77,77,77,77,76,76,76,76,75,75,75,75,75,75,75,75,75,75,74,
    73,73,73,73,73,73,73,73,73,73,73,72,72,72,72,72,72,71,71,71,71,
    71,71,70,69,69,69,69,69,69,69,69,68,68,68,68,68,67,67,67,67,67,
    67,67,67,67,67,66,66,66,66,65,65,65,65,65,65,65,65,64,64,64,63,
    63,63,62,62,62,62,62,62,62,62,62,62,61,61,61,61,61,60,60,60,60,
    60,59,59,59,59,59,59,59,59,58,58,58,58,58,58,57,57,57,57,57,57,
    57,57,57,56,56,56,56,56,56,55,54,54,54,54,54,53,53,53,53,53,52,
    52,52,52,52,52,52,52,52,51,51,50,50,50,50,50,50,50,50,50,49,49,
    49,49,49,49,48,48,48,48,48,48,48,48,47,47,47,46,46,46,46,46,46,
    46,45,45,45,45,45,45,45,45,45,45,44,44,44,44,44,44,43,43,43,43,
    43,43,42,42,42,42,41,41,41,41,41,41,41,41,40,40,40,40,40,39,39,
    39,39,39,39,39,38,38,38,38,38,37,37,37,36,36,36,35,35,35,35,35,
    35,35,34,34,34,34,34,33,33,33,33,33,33,33,33,33,32,32,32,32,32,
    32,32,32,31,31,31,31,31,31,30,30,30,30,30,30,30,29,29,29,29,29,
    29,29,29,28,28,28,28,28,28,27,27,27,27,27,27,27,26,26,26,26,26,
    26,26,26,25,25,25,25,24,24,24,24,24,24,24,24,24,23,23,23,23,23,
    23,23,22,22,22,22,22,21,21,21,21,21,21,21,21,20,20,20,20
  };
  const int n4c2w2_b[] = {
    120, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,100,99,99,99,99,99,99,98,98,98,98,98,98,98,
    97,97,97,97,97,97,97,97,97,97,97,97,97,97,96,96,96,95,95,95,95,
    95,94,94,94,94,94,94,94,94,93,93,93,93,93,93,93,93,93,93,93,92,
    92,92,92,92,91,91,91,91,91,91,90,90,90,90,89,89,89,89,89,89,89,
    89,88,88,88,88,88,87,86,86,86,86,86,86,85,85,85,84,84,84,84,84,
    84,83,83,83,83,83,83,82,82,82,82,82,82,82,82,81,81,81,81,81,81,
    81,81,80,80,79,79,79,79,79,79,79,79,78,78,78,78,78,77,77,77,77,
    77,77,77,76,76,76,76,76,76,76,76,76,75,75,75,74,74,74,74,74,74,
    74,74,74,73,73,73,73,72,72,72,72,72,72,72,71,70,70,70,70,70,69,
    69,69,69,69,69,69,69,69,69,69,68,68,68,68,68,68,67,67,67,67,67,
    67,67,66,66,66,66,66,66,65,65,65,65,65,65,65,64,64,64,64,63,63,
    63,63,63,63,63,63,62,62,62,62,62,61,61,61,61,61,61,60,60,60,60,
    60,59,59,59,59,59,59,59,58,58,57,57,57,56,56,56,56,56,56,56,55,
    55,55,55,55,55,55,55,54,54,54,54,54,53,53,53,53,53,53,53,53,52,
    52,52,52,52,52,52,51,51,51,51,51,51,51,50,50,50,50,50,50,50,50,
    50,50,49,49,49,49,49,49,48,48,48,48,48,48,48,48,47,47,47,47,47,
    47,47,47,47,47,46,46,46,46,45,45,45,44,44,44,43,43,42,42,42,42,
    42,41,41,41,41,41,41,41,41,41,41,40,40,40,39,39,39,39,39,38,38,
    38,38,37,37,37,37,37,37,37,37,37,37,36,36,36,36,36,36,35,35,35,
    35,35,35,35,34,34,34,34,34,33,33,33,33,33,33,33,33,33,32,32,32,
    32,31,31,31,31,31,31,31,30,30,30,30,30,30,29,29,29,28,28,28,28,
    28,28,28,28,28,27,27,27,27,27,27,27,26,26,26,26,26,26,25,25,25,
    25,25,25,25,25,25,25,25,24,24,24,24,24,24,24,24,23,23,23,23,23,
    23,23,23,22,22,22,22,22,22,21,21,21,21,20,20,20,20,20,20
  };
  const int n4c2w2_c[] = {
    120, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,99,99,99,99,99,98,98,98,98,98,98,98,98,98,97,
    97,97,97,97,97,96,96,96,95,95,95,95,95,95,95,95,94,94,94,94,94,
    94,93,93,93,93,93,93,93,92,92,92,92,92,92,92,91,91,91,91,91,91,
    90,90,90,90,90,90,90,90,89,89,89,89,89,89,89,89,89,88,88,88,88,
    88,88,88,87,87,87,86,86,86,86,86,85,85,85,85,85,84,84,84,84,84,
    84,83,83,83,83,83,83,83,82,82,82,82,81,81,81,81,81,81,81,80,80,
    80,80,78,78,78,78,78,78,78,78,78,78,77,77,77,76,76,76,76,76,76,
    76,75,75,75,75,74,74,74,74,74,74,74,73,73,73,73,73,73,73,73,72,
    72,72,72,72,72,72,72,72,71,71,71,71,71,71,71,70,70,70,69,69,69,
    69,69,68,68,67,67,67,67,67,66,66,66,66,66,66,66,65,65,65,65,65,
    65,65,65,65,65,65,65,65,64,64,64,64,64,64,64,64,64,63,63,63,63,
    62,62,62,62,62,62,61,61,61,61,61,61,61,60,60,60,60,60,60,59,59,
    59,59,59,59,59,59,59,58,58,58,58,58,58,58,58,58,57,57,57,57,57,
    56,56,56,56,56,56,56,56,55,55,55,54,54,53,53,53,53,53,53,53,52,
    52,52,52,52,51,51,51,50,50,50,50,49,49,49,49,49,49,49,49,48,48,
    48,48,48,48,48,47,47,47,47,47,47,47,46,46,46,45,45,45,45,45,45,
    45,45,44,44,44,44,44,44,43,43,43,43,43,43,43,43,43,42,42,42,42,
    42,42,42,42,41,41,41,41,41,41,41,41,41,40,40,40,40,40,39,39,39,
    39,39,38,38,38,38,38,37,37,37,37,37,36,36,36,35,35,35,35,35,35,
    35,34,34,34,34,34,34,34,34,34,33,33,33,33,33,32,32,32,32,32,32,
    32,32,32,31,31,31,31,31,31,31,31,30,30,30,30,30,30,29,29,29,29,
    29,29,29,29,28,28,28,28,28,27,27,27,27,27,27,27,26,26,26,26,26,
    26,25,25,25,25,25,24,24,24,24,24,24,24,23,23,23,23,23,23,23,23,
    23,22,22,22,22,21,21,21,21,21,21,21,21,21,20,20,20,20,20
  };
  const int n4c2w2_d[] = {
    120, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,99,99,99,99,99,99,99,98,98,98,98,98,98,98,97,97,97,97,97,
    97,97,96,96,96,96,96,96,96,96,96,95,95,95,95,95,95,95,95,94,94,
    94,94,94,93,93,93,93,93,92,92,92,92,91,91,91,91,91,90,90,90,90,
    90,90,90,90,90,90,90,90,90,89,89,89,89,89,89,89,89,88,88,88,88,
    88,88,87,87,87,87,86,86,86,86,86,86,86,86,86,86,85,84,84,84,84,
    84,84,84,83,83,83,83,83,83,83,83,83,83,83,82,82,82,82,82,82,82,
    82,81,81,81,81,81,81,81,80,80,80,80,80,79,79,79,79,79,79,78,78,
    78,78,78,78,78,78,77,77,77,77,77,76,76,76,76,76,76,76,76,75,75,
    75,75,75,75,74,74,74,74,74,74,74,74,73,73,73,73,73,73,73,73,72,
    72,72,72,72,71,71,71,71,71,71,71,71,70,70,70,70,70,70,70,69,69,
    69,68,68,68,68,68,68,67,67,67,67,67,66,66,65,65,65,65,65,64,64,
    64,63,63,63,63,62,62,62,62,62,62,61,61,61,61,61,61,60,60,60,60,
    60,60,60,59,59,59,59,59,59,59,58,58,58,58,58,57,57,57,57,57,57,
    57,56,56,56,55,55,55,55,55,54,54,54,54,54,54,54,54,54,54,53,53,
    53,53,53,52,52,52,52,52,52,51,51,51,51,51,51,50,50,50,50,50,50,
    50,49,49,49,49,49,49,49,49,48,48,48,48,48,47,47,47,47,47,47,47,
    46,45,45,45,45,45,45,45,45,44,44,44,43,43,43,43,43,43,42,42,42,
    42,42,42,42,42,42,41,41,41,41,41,40,40,40,40,40,39,39,39,39,39,
    39,39,39,39,39,38,38,38,38,38,38,38,38,37,37,37,37,37,37,37,36,
    36,36,36,36,36,36,36,36,35,35,35,35,35,35,35,35,35,35,34,34,34,
    34,34,33,33,33,33,33,33,33,33,33,32,32,32,32,32,32,32,31,31,31,
    31,31,31,31,30,30,30,30,29,29,28,28,28,28,28,28,28,27,27,27,27,
    26,26,26,26,26,25,25,25,25,25,24,24,24,24,24,24,24,23,22,22,22,
    22,22,22,21,21,21,21,21,21,21,21,21,20,20,20,20,20,20
  };
  const int n4c2w2_e[] = {
    120, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,100,100,99,99,98,98,98,98,98,98,98,97,97,
    97,96,96,96,96,96,96,96,96,96,96,95,95,95,95,95,95,95,95,95,94,
    94,94,94,94,93,93,93,93,93,93,93,93,93,92,92,92,92,92,91,91,91,
    91,91,91,91,91,91,91,90,90,90,90,89,89,88,88,88,88,88,88,87,87,
    87,87,87,86,86,86,86,85,85,85,84,84,84,84,84,84,83,83,83,83,83,
    83,83,83,82,82,82,82,82,81,81,81,81,81,80,80,80,80,80,80,79,79,
    79,79,79,79,78,78,78,77,77,77,77,77,77,77,77,76,76,76,76,76,76,
    76,76,75,75,75,75,75,75,75,75,74,74,74,74,73,73,73,73,73,73,73,
    73,72,72,72,72,72,72,72,72,72,71,71,71,71,71,71,71,70,70,70,70,
    70,70,70,69,69,69,68,68,68,68,68,68,67,67,67,67,67,67,67,67,67,
    66,66,66,66,66,66,66,66,66,66,65,65,65,65,65,65,65,64,64,64,64,
    64,63,63,63,63,63,62,62,62,62,62,62,62,62,62,61,61,61,61,61,61,
    61,61,61,61,61,61,60,60,60,60,59,59,59,59,59,59,59,58,58,58,58,
    58,57,57,57,57,56,56,56,56,56,56,55,55,55,55,55,55,55,55,54,54,
    54,54,54,54,54,54,53,53,53,53,53,53,53,53,53,53,52,52,52,52,52,
    52,52,51,51,51,51,51,51,51,51,50,50,50,50,50,50,50,50,49,49,49,
    49,49,49,49,49,49,48,48,48,47,47,47,47,47,47,47,46,46,46,46,46,
    46,45,45,45,45,45,44,44,44,44,44,44,44,43,43,42,42,42,42,41,41,
    40,40,40,40,40,39,39,39,39,39,39,39,38,38,38,38,38,37,37,36,36,
    36,36,36,36,35,35,35,35,35,35,35,35,35,35,35,35,34,34,34,34,34,
    34,34,34,34,34,34,33,33,33,33,33,33,32,32,32,32,32,32,31,31,31,
    31,30,30,30,30,30,30,29,29,28,28,27,27,27,27,27,27,27,26,26,26,
    26,26,25,25,25,25,25,25,25,25,25,25,24,24,24,24,24,24,24,23,23,
    23,23,23,23,23,23,23,22,22,22,22,21,21,21,21,20,20,20,20,20
  };
  const int n4c2w2_f[] = {
    120, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,100,100,100,100,99,99,99,99,99,99,99,99,99,
    99,99,99,98,98,98,98,98,98,97,97,97,97,97,96,95,95,95,95,95,94,
    94,94,94,94,94,94,93,93,93,93,93,93,93,93,93,93,92,92,92,92,92,
    91,91,91,91,91,91,91,91,91,91,90,90,90,90,90,90,90,90,90,90,89,
    89,89,89,89,89,89,88,88,88,88,87,87,87,87,87,86,86,86,86,86,86,
    86,86,86,86,85,85,85,85,85,85,84,84,84,84,84,83,83,83,83,83,83,
    83,83,83,83,83,82,82,82,82,82,82,82,81,81,81,81,81,81,80,80,80,
    79,79,79,79,79,79,79,78,78,78,78,78,78,78,77,77,77,77,77,77,77,
    76,76,76,76,76,76,75,75,75,75,75,75,75,75,75,74,74,74,74,74,74,
    74,74,74,74,73,73,73,73,72,72,72,72,72,72,72,72,72,72,71,71,71,
    71,70,70,70,70,70,70,70,69,69,69,69,69,69,68,68,68,68,68,68,68,
    68,68,68,67,67,67,67,67,66,66,66,66,66,65,65,65,65,65,64,64,64,
    64,63,63,63,63,63,63,62,62,62,62,62,62,62,62,62,61,61,61,61,61,
    61,60,59,58,58,58,58,57,57,57,57,57,56,56,56,56,56,56,56,56,56,
    56,55,55,55,54,54,54,54,53,53,53,53,52,52,52,52,52,51,51,51,51,
    51,51,50,50,50,50,50,50,50,50,50,50,49,49,48,48,48,48,48,48,48,
    47,47,46,46,46,46,46,46,46,46,45,45,45,45,45,45,44,44,43,43,43,
    43,42,42,42,42,42,42,42,41,41,41,41,41,41,40,40,40,39,39,38,38,
    38,38,37,37,37,37,37,37,37,37,37,37,37,37,37,36,36,36,36,36,36,
    36,36,36,35,35,35,35,35,35,34,34,34,34,34,34,34,34,34,33,33,33,
    33,33,33,32,32,32,32,32,32,32,32,31,31,31,31,31,31,30,30,30,30,
    30,29,29,29,29,29,28,28,28,28,28,28,28,28,28,27,27,27,27,27,26,
    26,26,26,25,25,25,25,25,25,25,25,24,24,24,23,23,23,23,23,23,23,
    23,23,22,22,22,22,22,22,22,22,22,21,21,21,21,21,20,20,20,20
  };
  const int n4c2w2_g[] = {
    120, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,100,100,100,100,100,99,99,99,99,98,98,98,
    98,98,98,97,97,97,97,97,97,96,96,96,96,96,95,95,95,95,95,95,95,
    95,95,95,95,94,94,94,94,94,94,94,94,93,93,93,93,93,93,93,93,92,
    92,92,91,91,91,91,90,90,90,90,90,90,90,89,89,89,89,89,89,88,88,
    88,88,88,88,88,87,87,87,87,86,86,86,86,85,85,85,85,85,85,85,84,
    84,84,84,84,84,83,83,83,83,83,82,82,82,81,81,81,81,80,80,80,80,
    79,79,79,79,79,79,78,78,78,78,78,78,78,78,78,77,77,77,77,77,76,
    76,76,76,75,75,75,75,75,75,75,75,74,74,74,74,74,74,73,73,73,72,
    72,72,72,72,72,71,71,71,71,71,71,70,70,70,70,70,69,69,69,69,69,
    69,69,69,69,69,68,68,68,68,68,68,68,68,68,68,68,68,68,67,67,67,
    67,66,66,66,65,65,65,65,65,65,65,65,65,64,64,64,64,64,64,63,63,
    63,63,63,63,63,63,63,63,63,62,62,62,62,62,61,61,61,60,60,60,60,
    60,60,59,59,59,59,59,59,59,59,59,59,58,58,58,57,57,57,57,57,57,
    57,57,56,56,56,56,56,56,56,55,55,55,55,55,55,55,54,54,54,54,54,
    54,54,54,54,54,53,53,53,53,53,52,52,52,52,52,52,51,51,51,51,51,
    51,51,51,50,50,50,50,50,50,50,50,50,49,49,49,48,48,48,48,47,47,
    47,47,47,47,47,47,47,46,46,46,46,45,45,45,45,45,45,44,44,44,43,
    43,43,43,43,42,42,42,42,42,42,41,41,41,41,41,41,40,40,40,40,40,
    39,39,39,39,39,38,38,37,37,37,37,37,36,36,36,36,35,35,35,35,35,
    35,35,35,35,34,34,34,34,34,33,33,33,33,33,33,33,32,32,32,32,32,
    31,31,31,31,31,31,31,30,30,30,30,30,30,30,30,30,29,29,29,29,28,
    28,28,28,28,27,27,27,27,27,27,27,27,27,26,26,26,26,26,26,25,25,
    25,24,24,24,24,23,23,23,23,23,23,22,22,22,22,22,22,22,22,21,21,
    21,21,21,21,21,21,21,21,21,21,21,21,21,21,20,20,20,20,20,20,20
  };
  const int n4c2w2_h[] = {
    120, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,99,99,99,99,99,99,99,98,98,98,98,98,98,97,97,97,97,96,96,
    96,96,96,96,96,96,96,96,95,95,95,95,95,95,95,94,94,94,94,93,93,
    93,93,93,93,93,92,92,92,92,92,92,92,92,92,91,91,91,91,91,91,90,
    90,90,90,90,89,89,89,89,89,89,89,88,88,88,88,88,88,88,88,87,87,
    86,86,86,86,86,86,86,86,86,85,85,85,85,85,85,85,85,85,85,85,84,
    84,84,84,84,83,83,83,83,83,83,82,82,82,82,82,82,82,81,81,81,81,
    81,80,80,80,80,80,80,80,80,79,79,79,79,79,79,79,79,78,78,78,78,
    77,77,77,77,77,77,77,77,77,77,77,76,76,76,76,76,76,76,75,75,75,
    75,75,74,74,74,73,73,73,73,72,72,72,72,71,71,71,71,71,71,71,71,
    70,70,70,70,70,69,69,69,69,69,68,68,68,68,68,68,68,67,67,67,67,
    67,66,66,66,66,66,66,65,65,65,65,65,64,64,64,64,64,64,63,63,62,
    62,62,62,62,61,61,61,60,60,60,60,60,60,60,60,60,60,60,59,59,59,
    59,59,59,59,59,59,59,59,58,58,58,58,58,58,58,57,57,57,57,57,57,
    56,56,56,56,56,56,56,55,55,55,55,55,54,54,54,54,54,53,53,53,53,
    53,53,53,52,52,52,52,52,51,51,51,51,51,51,51,51,49,49,49,49,49,
    48,48,48,48,47,47,47,47,47,47,47,47,47,47,47,47,47,46,46,46,46,
    46,46,46,45,45,45,45,45,45,45,45,44,44,43,43,43,43,43,43,43,43,
    42,42,42,42,41,41,41,41,41,41,41,41,40,40,40,38,38,38,38,38,38,
    38,37,37,37,37,37,37,37,36,36,36,36,36,36,36,36,36,35,35,35,35,
    35,34,34,34,34,34,34,34,34,33,33,33,33,33,33,33,32,32,32,32,32,
    32,31,31,31,31,31,31,30,30,30,30,30,30,29,29,29,29,29,28,28,28,
    27,27,27,27,27,27,27,26,26,26,26,26,25,25,25,25,25,25,25,25,25,
    25,25,24,24,24,23,23,23,23,23,23,23,23,22,22,22,22,22,22,22,22,
    21,21,21,21,21,21,21,21,21,21,21,21,20,20,20,20,20,20
  };
  const int n4c2w2_i[] = {
    120, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,99,99,99,99,99,99,98,98,98,98,97,97,97,97,97,
    97,97,96,96,96,96,96,95,95,95,95,95,95,95,95,95,95,95,94,94,94,
    94,94,94,94,94,94,94,94,93,93,93,93,93,93,93,93,93,92,92,92,92,
    92,92,91,91,91,91,91,91,90,90,90,90,90,90,90,90,90,90,89,89,89,
    88,88,88,88,87,87,87,87,87,87,87,87,87,87,87,86,86,86,85,85,85,
    85,85,85,85,84,84,84,84,83,83,83,83,83,82,82,82,82,82,82,82,82,
    82,81,81,81,81,81,81,81,80,80,80,80,80,79,79,79,79,79,79,79,78,
    78,78,77,77,77,77,77,77,76,76,76,76,76,76,76,76,76,76,75,75,75,
    75,75,75,75,74,74,74,74,74,73,73,73,73,73,72,72,72,72,72,72,72,
    72,72,72,72,72,71,71,71,71,71,70,70,70,70,70,70,70,70,70,70,69,
    69,68,68,68,68,68,68,68,67,67,67,67,67,66,66,66,66,65,65,65,65,
    65,65,65,65,65,65,65,64,64,64,64,64,63,63,63,62,62,62,62,62,61,
    61,61,61,61,60,60,60,60,60,59,59,59,59,59,59,59,58,58,58,58,58,
    58,58,58,58,57,57,57,57,57,57,57,57,56,56,55,55,55,54,54,54,53,
    53,53,53,53,53,53,52,51,51,50,50,50,50,49,49,49,49,49,49,49,49,
    48,48,48,48,48,48,48,48,48,48,48,47,47,47,47,47,47,47,46,46,46,
    46,46,46,45,45,45,45,45,45,44,44,44,44,44,43,43,43,43,43,43,43,
    43,43,43,43,43,43,42,42,42,42,42,42,42,41,41,41,41,41,41,41,40,
    40,40,40,40,40,39,39,39,39,39,38,38,38,38,37,37,37,37,37,37,36,
    35,35,35,35,35,35,35,35,35,34,34,34,34,34,34,34,34,33,33,32,32,
    32,32,32,32,31,31,31,31,31,31,31,30,30,30,30,30,30,29,29,29,29,
    29,29,29,29,29,29,28,28,28,28,28,28,27,27,27,27,27,26,26,26,26,
    25,25,25,25,25,25,25,25,25,24,24,24,24,24,24,23,23,23,23,23,23,
    22,22,22,22,22,22,22,22,21,21,21,20,20,20,20,20,20,20,20
  };
  const int n4c2w2_j[] = {
    120, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,99,99,99,99,98,98,98,98,98,98,98,98,98,98,98,97,97,
    97,97,97,97,97,97,96,96,96,96,95,95,95,95,95,95,94,94,94,94,94,
    94,94,93,93,93,93,93,93,93,93,93,93,92,92,92,92,92,92,92,91,91,
    91,91,91,91,91,91,91,90,90,90,90,89,89,89,89,88,88,88,88,87,87,
    87,87,87,87,87,87,86,86,86,86,86,85,85,85,85,85,85,85,84,84,84,
    84,83,83,83,83,83,83,83,82,82,82,82,82,82,82,82,82,81,81,81,81,
    81,81,81,81,80,80,80,79,79,79,79,79,79,78,78,78,78,78,78,78,77,
    77,77,77,76,76,76,75,75,75,75,75,75,75,74,74,74,74,73,73,73,72,
    72,72,72,72,71,71,71,71,71,71,70,70,70,70,70,70,69,69,69,69,69,
    69,69,69,68,68,68,68,68,68,68,68,67,67,67,67,67,67,67,67,67,67,
    66,66,66,66,66,66,66,66,65,65,65,65,65,65,65,65,65,65,65,65,64,
    64,64,64,64,64,64,63,63,63,63,62,62,61,61,61,61,61,61,61,61,61,
    61,61,59,59,59,59,59,59,59,58,58,58,58,58,58,58,58,58,57,57,57,
    57,57,57,56,56,56,56,56,56,56,55,55,55,55,55,55,55,54,54,54,54,
    54,54,54,53,53,53,53,53,53,53,53,53,53,53,53,52,52,52,52,52,52,
    52,51,51,51,51,51,51,51,51,51,51,51,51,50,50,50,50,50,49,49,49,
    49,49,49,48,48,48,47,47,47,47,47,46,45,45,45,45,45,45,44,44,43,
    43,43,43,43,42,42,42,42,42,42,42,42,41,41,41,41,41,41,41,41,40,
    40,40,40,40,39,39,39,39,39,39,38,38,38,38,38,38,38,38,38,38,38,
    37,37,37,36,35,35,35,35,35,35,35,35,35,35,35,35,34,34,34,34,34,
    34,34,33,33,33,33,33,32,32,32,32,32,32,32,32,32,31,31,31,31,30,
    30,30,30,29,29,29,28,28,28,28,28,28,28,28,27,27,27,27,27,27,27,
    26,26,26,25,25,25,24,24,24,24,24,24,24,23,23,23,23,23,23,23,23,
    23,23,22,22,22,22,22,22,21,21,21,20,20,20,20,20,20,20
  };
  const int n4c2w2_k[] = {
    120, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,100,100,100,99,99,99,99,99,99,99,98,98,98,
    98,98,98,98,98,97,97,97,97,97,97,96,96,96,96,96,96,96,96,96,96,
    95,95,95,95,95,94,94,94,94,94,94,94,94,93,93,93,93,92,92,92,92,
    92,92,92,91,91,91,91,91,91,91,91,91,90,89,89,89,89,89,89,88,88,
    88,88,88,88,87,87,87,87,87,87,86,86,86,86,86,86,85,85,85,85,85,
    84,84,84,84,83,83,83,83,83,83,83,83,82,82,82,82,82,82,82,81,81,
    81,81,81,80,80,80,80,80,80,80,79,79,79,79,79,78,78,78,78,77,77,
    77,77,77,77,77,77,77,77,76,76,76,75,75,75,75,75,75,75,74,74,74,
    74,74,74,74,74,73,73,73,73,72,72,72,72,72,71,71,71,71,71,71,71,
    71,71,70,70,70,70,70,70,69,69,69,69,69,68,68,68,68,68,68,67,67,
    67,67,67,66,66,66,66,66,66,66,65,65,65,65,65,65,65,65,65,65,65,
    65,64,64,64,64,64,64,64,63,63,63,63,62,62,62,62,62,62,62,62,61,
    61,61,61,60,60,60,60,60,60,60,59,59,59,59,59,58,58,58,58,58,57,
    56,56,56,56,55,55,55,55,55,55,55,55,54,54,54,54,54,54,54,54,54,
    54,53,53,53,53,53,53,53,53,52,52,52,52,52,52,52,51,51,51,51,51,
    51,51,51,51,51,50,50,50,50,50,49,49,49,48,48,48,48,48,47,47,47,
    47,46,46,46,46,46,45,44,44,44,44,44,44,44,44,44,44,44,44,44,43,
    43,43,43,43,43,42,42,42,42,42,41,41,41,41,41,41,41,40,40,40,40,
    39,39,39,39,39,39,39,38,38,38,38,38,38,38,38,38,37,37,37,37,37,
    37,37,37,37,37,36,36,36,36,35,35,35,35,35,35,35,35,34,34,34,34,
    34,34,34,34,33,33,33,33,33,32,32,32,32,32,32,32,31,31,31,31,31,
    31,31,31,31,30,30,30,30,30,30,30,29,29,29,29,29,29,29,29,28,28,
    28,28,28,28,28,28,27,27,27,27,27,27,26,26,26,25,25,25,25,24,24,
    23,23,23,23,23,23,23,23,22,22,22,22,21,21,21,21,20,20,20,20
  };
  const int n4c2w2_l[] = {
    120, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,99,99,99,99,99,99,99,99,98,98,98,98,98,98,
    98,98,98,98,98,97,97,97,97,97,97,97,96,96,96,96,96,96,96,96,96,
    95,95,95,95,95,94,94,94,94,93,93,93,93,93,92,92,92,92,92,91,91,
    91,91,91,90,90,90,90,90,89,89,89,89,89,89,89,88,88,88,88,88,88,
    88,87,87,87,87,87,86,86,86,86,86,86,86,86,86,86,85,85,85,85,85,
    85,85,85,85,85,84,84,84,84,84,84,84,84,84,84,84,84,83,83,83,83,
    83,82,82,82,82,81,81,81,81,81,80,79,79,79,79,79,79,79,79,79,78,
    78,78,78,78,77,77,77,77,77,77,77,77,76,76,76,76,76,76,76,76,76,
    75,75,75,75,75,75,74,74,74,74,74,74,74,74,74,73,73,73,73,73,73,
    73,73,73,73,72,72,72,72,72,72,72,71,71,71,71,71,71,71,71,70,70,
    69,69,69,69,69,68,68,68,67,67,67,67,67,66,66,66,66,66,65,65,65,
    65,65,65,65,64,64,64,64,64,64,64,63,63,63,63,63,63,63,62,62,62,
    61,61,61,61,61,60,60,60,60,60,60,59,59,59,59,59,59,59,59,59,58,
    58,58,58,58,57,57,57,57,57,57,56,56,56,55,55,55,55,55,54,54,54,
    54,54,53,53,53,53,53,53,53,53,53,53,53,52,52,52,52,51,51,50,50,
    50,50,50,50,50,50,50,49,49,49,48,48,48,48,48,48,47,47,47,47,47,
    47,47,47,46,46,45,45,45,45,45,45,45,45,44,44,44,44,44,44,44,44,
    43,43,43,43,43,43,42,42,42,42,42,41,41,40,40,40,40,40,39,39,39,
    39,39,39,39,39,39,39,38,38,38,38,37,37,37,37,37,37,37,36,36,36,
    36,36,35,35,35,35,35,35,34,34,34,34,34,33,33,33,33,33,33,33,32,
    32,32,32,32,32,32,32,32,32,31,31,31,31,31,31,31,30,30,30,30,30,
    30,29,29,29,29,29,29,29,29,29,29,28,28,28,28,28,28,28,27,27,27,
    27,27,27,26,26,26,26,26,26,25,25,25,25,25,24,24,24,24,24,24,24,
    24,24,23,23,23,23,22,22,22,22,22,21,21,21,21,21,20,20,20
  };
  const int n4c2w2_m[] = {
    120, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,100,100,99,99,99,99,99,99,99,99,99,98,98,
    98,98,98,97,97,97,97,97,97,97,96,96,96,96,96,95,95,95,94,94,94,
    94,93,93,93,93,93,93,93,92,92,92,92,92,92,92,92,91,91,91,91,91,
    91,91,91,91,90,90,90,90,90,90,89,88,88,88,88,87,87,87,87,87,87,
    87,87,86,86,86,86,86,86,86,86,86,85,85,85,85,84,84,84,84,84,83,
    83,83,83,83,83,82,82,82,82,82,82,82,82,82,82,82,81,81,81,81,81,
    81,81,81,80,80,80,80,79,79,79,79,78,78,78,78,78,78,78,77,77,77,
    77,77,77,77,77,77,77,77,77,76,76,76,76,76,76,76,75,75,75,75,75,
    75,75,75,75,75,75,75,74,74,74,74,74,74,74,73,73,73,73,73,72,72,
    72,72,71,71,71,71,71,71,71,71,70,70,70,70,70,69,69,69,69,69,69,
    69,69,69,69,68,68,68,68,67,67,67,67,67,66,65,65,65,64,64,63,63,
    63,63,63,63,63,62,62,62,62,62,62,62,62,62,61,61,61,61,60,60,60,
    60,60,60,60,59,59,59,59,59,58,58,57,57,57,57,57,57,57,57,57,56,
    56,56,56,56,56,55,55,55,55,55,54,54,54,54,54,54,54,54,54,53,53,
    53,53,53,52,52,52,52,52,52,52,52,52,52,52,51,51,51,51,51,51,50,
    50,50,50,50,49,49,49,49,49,49,49,49,49,49,48,48,48,48,48,48,48,
    48,47,47,47,47,47,47,47,46,46,46,46,46,46,46,46,46,45,45,45,45,
    45,44,44,44,44,44,44,44,44,43,43,43,43,43,43,43,42,42,42,42,41,
    41,41,41,41,41,41,41,40,40,40,40,40,39,39,39,39,39,38,38,37,37,
    37,37,37,37,37,36,36,36,35,35,35,35,35,35,35,34,34,34,34,34,34,
    34,34,33,33,33,32,32,32,32,32,32,31,31,31,31,31,31,31,30,30,30,
    30,30,30,29,29,28,28,28,28,28,28,27,27,27,26,26,25,25,25,25,25,
    25,25,25,24,24,24,24,24,23,23,23,23,22,22,22,22,22,22,22,21,21,
    21,21,21,21,21,21,21,21,21,21,21,21,21,20,20,20,20,20,20,20
  };
  const int n4c2w2_n[] = {
    120, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,100,99,99,99,99,99,99,98,98,98,98,98,98,98,
    98,97,97,97,97,97,97,96,96,96,96,95,95,95,95,95,95,95,95,95,94,
    94,94,94,94,94,93,93,93,93,93,93,93,93,93,93,92,92,91,91,91,91,
    90,90,90,90,90,90,89,89,89,89,89,89,89,89,89,89,89,89,88,88,88,
    88,88,88,88,87,87,87,87,87,87,87,86,86,86,86,86,86,84,84,84,84,
    84,84,84,83,83,83,83,83,83,83,82,82,82,82,82,82,82,82,81,81,81,
    80,80,80,80,80,80,80,80,80,80,80,79,79,79,79,79,79,78,78,78,78,
    78,78,78,78,77,77,77,77,77,77,77,77,77,76,76,76,76,76,76,75,75,
    75,75,75,75,75,74,74,74,74,74,74,73,73,72,72,72,72,71,71,71,71,
    71,70,70,70,70,70,70,70,70,69,69,68,68,68,68,68,68,67,67,67,67,
    67,67,67,66,66,66,66,66,66,66,65,64,64,64,64,64,64,64,64,64,64,
    64,63,63,63,63,63,63,62,62,62,62,62,62,62,61,61,61,61,61,61,61,
    61,61,60,60,60,60,60,59,59,59,59,59,59,58,58,58,58,58,57,57,57,
    57,57,57,57,56,56,56,56,56,56,56,56,56,56,56,56,56,55,55,55,55,
    55,55,55,54,54,54,54,53,52,52,52,52,52,52,52,52,51,51,51,51,51,
    51,51,51,51,51,50,50,50,50,50,49,49,49,49,49,49,49,48,48,48,48,
    48,48,48,48,48,47,47,47,47,46,46,46,46,46,46,46,46,46,46,46,45,
    45,45,45,44,44,44,44,44,44,44,43,43,43,43,42,42,42,42,42,41,41,
    41,41,41,41,41,40,40,40,40,40,39,39,39,39,39,39,39,38,38,38,37,
    37,37,37,37,37,37,37,36,36,36,36,36,36,36,36,36,36,36,36,35,35,
    35,35,35,35,35,35,35,35,35,35,35,34,34,34,34,34,34,34,33,33,33,
    33,33,33,33,33,33,32,32,32,32,32,31,31,31,31,31,31,30,30,30,30,
    30,29,29,29,29,29,29,29,28,28,28,27,27,27,27,27,27,26,26,26,25,
    25,24,24,24,23,23,22,22,22,22,21,21,21,21,20,20,20,20,20
  };
  const int n4c2w2_o[] = {
    120, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,100,100,100,99,99,99,99,99,99,99,98,98,98,
    98,98,98,97,97,97,97,97,97,96,96,96,96,96,95,95,95,95,95,95,94,
    94,94,94,94,94,94,94,93,93,93,93,93,92,92,92,92,92,91,91,91,91,
    90,90,90,90,90,89,89,89,89,89,89,88,88,88,88,87,87,87,86,86,86,
    86,86,86,86,86,85,85,85,85,85,85,84,84,84,84,84,83,83,83,83,83,
    83,83,83,83,82,82,82,82,82,82,81,81,81,81,81,81,81,81,80,80,80,
    80,80,79,79,79,79,79,78,78,78,78,78,78,78,77,77,77,77,77,77,76,
    76,76,76,76,75,75,75,75,75,75,75,75,75,74,74,74,74,73,73,73,73,
    73,73,73,72,72,72,72,72,72,72,72,72,71,71,70,70,70,70,70,70,70,
    70,70,70,69,69,69,69,69,69,68,68,68,68,67,67,67,67,67,67,66,66,
    66,66,66,66,66,66,66,65,65,65,65,65,65,65,65,65,65,65,64,64,64,
    64,64,63,63,63,63,62,62,62,62,62,62,61,61,61,61,60,60,60,60,60,
    60,59,59,59,59,59,59,58,58,58,58,58,58,58,58,58,58,57,57,57,57,
    57,56,56,56,56,56,56,55,55,55,55,54,54,54,54,54,54,53,53,53,53,
    52,52,52,52,52,51,51,51,51,51,51,51,51,50,50,50,50,49,49,49,49,
    49,48,48,48,48,48,48,48,48,47,47,47,47,47,46,46,45,45,45,44,44,
    44,44,44,44,43,43,43,43,43,43,43,43,42,42,42,42,42,42,42,42,42,
    41,41,41,41,41,41,41,41,41,40,40,40,40,40,40,39,39,39,38,38,38,
    38,38,38,38,38,38,37,37,37,37,37,37,37,37,37,37,36,36,36,36,36,
    35,35,35,35,35,35,34,34,34,34,34,34,34,34,34,34,34,33,33,33,33,
    33,33,33,33,32,32,32,32,32,32,31,31,31,31,31,31,31,31,31,31,30,
    30,30,30,30,30,30,30,30,29,29,29,29,28,28,28,28,28,28,27,27,27,
    27,27,27,26,26,26,26,26,25,25,25,25,25,25,25,25,25,25,25,24,24,
    23,23,23,23,23,23,22,22,22,22,22,21,21,21,21,21,21,21,21,20
  };
  const int n4c2w2_p[] = {
    120, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,100,99,99,99,99,99,98,98,98,98,98,98,98,98,
    98,98,97,97,97,97,96,96,96,96,96,96,96,96,95,95,95,95,95,95,95,
    94,94,94,94,94,94,94,94,93,93,93,93,93,93,93,92,92,92,92,92,92,
    92,92,92,92,92,92,92,92,92,91,91,91,91,90,90,90,90,90,89,89,89,
    89,89,89,89,88,88,88,88,88,88,88,88,88,88,88,87,87,87,87,87,86,
    86,86,86,86,86,85,85,85,85,85,85,85,85,84,84,84,83,83,83,83,83,
    83,83,83,82,82,82,82,82,82,81,81,81,81,81,81,80,80,80,80,80,80,
    79,79,79,79,79,79,79,79,79,78,78,78,78,78,78,77,77,76,76,76,76,
    75,75,75,75,74,74,74,74,74,73,73,73,73,73,73,72,72,72,72,72,72,
    72,72,71,71,71,71,71,70,70,70,70,70,70,69,69,69,69,69,69,69,69,
    69,68,68,68,68,68,68,67,67,67,67,67,67,67,67,67,67,66,66,66,66,
    66,66,65,65,65,65,65,65,64,64,64,64,64,64,64,64,63,63,63,63,63,
    62,62,62,62,61,61,61,61,61,61,60,60,60,60,60,60,59,59,59,59,59,
    59,59,59,59,58,58,58,58,58,58,58,57,57,57,57,56,56,56,56,56,56,
    55,55,55,55,54,54,54,54,54,54,53,53,53,53,53,53,53,53,53,53,53,
    52,52,51,51,51,51,50,50,50,50,50,50,50,50,50,50,50,50,50,50,50,
    49,49,48,48,48,48,48,48,48,47,47,46,46,46,45,45,45,45,45,44,44,
    44,43,43,43,43,43,42,42,42,42,42,41,41,40,40,40,40,40,40,40,40,
    39,39,39,39,39,39,39,39,39,39,38,38,38,38,37,37,37,37,37,36,36,
    36,36,36,35,35,35,35,35,35,35,35,35,35,35,35,35,34,34,34,34,34,
    34,33,33,33,33,32,32,32,32,31,31,31,31,31,31,30,30,30,30,30,30,
    29,29,29,29,28,28,28,28,27,27,27,27,27,27,27,27,26,26,26,25,25,
    25,25,25,24,24,24,24,24,24,24,24,23,23,23,23,23,23,23,23,22,22,
    22,22,22,22,22,21,21,21,21,21,20,20,20,20,20,20,20,20,20
  };
  const int n4c2w2_q[] = {
    120, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,100,99,99,99,99,99,99,99,99,98,98,98,98,98,
    98,97,97,97,97,97,97,97,97,97,97,97,97,96,96,96,96,96,95,95,95,
    95,95,94,94,94,94,94,94,94,94,93,93,93,93,92,92,92,92,92,92,92,
    91,91,91,91,91,91,91,91,90,90,90,90,90,90,90,90,90,89,89,89,89,
    89,89,89,89,88,88,87,87,87,87,86,86,86,86,86,85,85,85,85,85,84,
    84,84,84,84,84,83,83,83,82,82,82,82,82,81,81,81,81,81,81,81,80,
    80,80,79,79,79,79,79,79,79,79,79,79,79,79,78,78,78,78,78,78,78,
    78,78,77,77,77,77,77,77,77,77,76,76,76,76,76,76,75,75,75,75,74,
    74,74,74,74,74,74,74,73,73,73,73,73,73,73,72,72,72,72,71,71,71,
    70,70,70,70,70,70,70,69,69,69,69,68,68,68,67,67,67,67,67,67,66,
    66,66,66,66,66,65,65,65,65,65,64,64,64,64,64,64,64,63,63,63,63,
    63,62,62,62,62,62,62,61,61,61,61,61,61,61,60,60,60,60,60,60,60,
    59,59,59,59,58,58,58,58,58,58,58,57,57,57,57,57,57,57,57,56,56,
    56,56,56,56,56,55,55,55,55,55,55,54,54,54,54,53,53,53,53,53,53,
    53,52,52,52,52,52,52,52,51,51,51,51,51,51,51,51,50,50,50,50,50,
    50,50,50,49,49,49,49,49,49,49,48,48,48,48,47,47,47,47,47,47,46,
    46,46,46,46,46,46,45,45,45,45,45,45,45,45,44,44,44,44,44,44,44,
    44,43,43,43,43,43,43,43,43,43,42,42,42,42,41,41,41,41,41,41,41,
    41,40,40,40,40,40,40,40,40,39,39,39,39,38,38,38,38,37,37,37,37,
    37,36,36,36,36,35,35,35,34,34,34,34,34,34,34,34,33,33,33,33,33,
    33,33,33,33,33,33,33,32,32,32,32,32,32,31,31,31,30,30,30,29,29,
    29,29,29,29,28,28,28,28,28,28,27,27,27,27,27,27,27,27,27,26,26,
    26,26,26,26,25,25,25,25,25,25,24,24,24,23,23,23,23,23,23,23,23,
    23,23,22,22,22,22,22,22,22,21,21,21,21,21,21,20,20,20,20
  };
  const int n4c2w2_r[] = {
    120, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,99,99,99,99,98,98,98,98,98,98,98,98,97,97,97,
    97,97,97,97,97,97,97,96,96,95,95,95,95,95,95,95,94,94,94,94,94,
    94,94,94,94,94,93,93,92,92,92,92,92,91,91,91,90,90,90,90,90,90,
    89,89,89,89,89,89,89,89,89,89,89,88,88,87,87,86,86,86,86,86,86,
    86,86,86,86,85,85,85,84,84,84,84,84,84,84,84,84,84,84,84,84,83,
    83,83,83,83,83,83,83,82,82,82,82,82,82,82,81,81,81,81,81,81,81,
    81,81,81,81,81,81,80,80,80,80,80,80,79,79,79,79,79,79,79,78,78,
    78,78,78,78,77,77,77,77,77,76,76,76,76,76,76,76,76,75,75,75,74,
    74,74,74,74,74,73,73,73,73,73,72,72,72,72,72,72,72,71,71,71,71,
    71,71,70,70,70,70,70,69,69,69,69,69,68,68,68,68,67,67,66,66,66,
    66,66,66,66,66,66,65,65,65,65,65,65,65,65,64,64,64,64,64,64,64,
    64,64,64,64,63,63,63,63,63,63,63,62,62,62,62,62,62,62,62,62,61,
    61,61,60,60,60,60,60,59,59,59,59,58,58,58,58,58,58,58,58,57,57,
    57,56,56,56,56,56,56,56,55,55,55,55,55,55,55,55,55,54,54,54,54,
    54,54,54,54,53,53,53,53,53,53,53,52,52,52,52,52,51,51,51,51,51,
    51,51,51,51,51,50,50,49,49,49,49,48,48,48,48,48,48,48,47,47,47,
    47,47,47,47,46,46,46,46,46,46,46,46,45,44,44,44,44,44,44,44,43,
    43,43,43,43,42,42,41,41,41,41,41,40,40,40,40,40,40,40,39,39,39,
    39,39,39,39,38,38,38,38,38,38,37,37,37,37,37,37,37,37,36,36,36,
    36,36,36,36,36,36,36,36,36,35,35,34,34,34,34,34,33,33,33,33,32,
    32,32,32,31,31,31,31,31,31,31,30,30,30,29,29,29,29,29,29,29,29,
    29,28,28,28,28,28,27,27,27,27,27,27,26,26,26,26,25,25,25,25,25,
    25,25,25,24,24,24,24,24,24,24,24,24,24,23,23,23,23,23,22,22,22,
    22,22,21,21,21,21,21,21,21,21,21,21,21,20,20,20,20,20,20
  };
  const int n4c2w2_s[] = {
    120, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,100,99,99,99,99,99,98,98,98,98,97,97,97,97,
    97,97,97,97,97,97,97,96,96,96,96,96,96,96,96,96,96,95,95,95,94,
    94,94,94,94,94,94,94,94,93,93,93,93,93,93,93,93,93,92,92,92,92,
    91,91,91,91,91,91,90,90,90,90,90,90,89,89,89,89,89,89,89,89,89,
    89,89,89,88,88,88,88,88,87,87,87,87,87,87,87,86,86,86,85,85,85,
    85,85,84,84,84,84,83,83,83,83,83,82,82,81,81,81,81,81,81,80,80,
    80,80,80,80,80,79,79,79,79,78,78,78,78,78,78,78,78,78,77,77,77,
    77,77,77,77,77,77,77,77,77,77,76,76,76,76,76,76,76,76,76,75,75,
    75,75,75,74,74,74,74,74,74,74,74,74,74,74,73,73,73,72,72,72,72,
    72,72,72,72,72,72,72,72,71,71,71,71,71,71,71,71,70,70,70,70,70,
    70,70,70,69,69,69,69,69,68,68,68,68,68,68,68,68,68,68,67,67,66,
    66,66,66,66,66,66,66,66,65,65,65,65,65,65,65,64,64,64,64,63,63,
    63,63,63,63,63,63,62,62,62,62,62,61,61,61,61,61,61,61,60,60,60,
    60,60,60,60,60,59,59,59,59,59,59,59,58,58,58,57,57,57,57,57,57,
    57,56,56,56,56,56,56,56,56,55,55,55,55,55,55,55,54,54,54,53,53,
    52,52,52,52,52,51,51,51,51,51,50,50,50,50,50,50,50,49,49,49,49,
    49,48,48,48,48,47,47,47,47,47,47,46,46,46,45,45,45,45,45,45,45,
    45,45,44,44,44,44,44,43,43,43,43,43,43,43,43,42,42,41,41,41,41,
    41,41,41,41,41,40,40,39,39,39,39,38,38,38,38,38,38,37,37,37,37,
    37,37,37,36,36,36,36,36,36,36,36,36,36,36,35,35,35,35,35,35,34,
    34,34,34,34,34,33,33,33,33,32,32,32,32,32,32,31,31,31,31,31,30,
    30,30,29,29,29,29,28,28,28,27,27,27,27,26,26,26,26,26,26,26,26,
    25,25,24,24,24,24,24,24,24,24,24,23,23,23,23,23,23,23,23,23,23,
    23,22,22,22,22,22,21,21,21,21,21,21,20,20,20,20,20,20,20
  };
  const int n4c2w2_t[] = {
    120, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,99,99,99,99,99,99,99,99,99,99,99,98,98,98,
    98,98,98,98,98,98,98,98,97,97,97,97,97,97,97,97,96,96,96,96,96,
    96,95,95,95,95,95,95,95,95,94,94,94,94,94,94,93,93,93,93,92,92,
    92,92,92,92,91,91,91,91,90,90,90,90,90,89,89,89,89,89,89,89,88,
    88,88,87,87,87,87,87,87,86,86,86,86,86,86,86,86,86,86,85,85,85,
    85,85,85,84,84,84,84,83,83,83,83,83,83,82,82,82,82,82,82,82,82,
    82,82,82,82,81,81,81,81,81,81,81,81,81,80,80,80,80,80,80,80,80,
    80,80,79,79,79,78,78,78,78,77,77,77,77,77,76,76,76,76,75,75,75,
    75,75,75,75,75,75,74,74,74,74,74,74,74,74,74,74,73,73,73,72,72,
    72,72,72,72,71,70,70,70,70,69,69,69,69,68,68,68,68,68,68,68,67,
    67,67,66,66,66,66,66,66,66,65,65,65,65,65,65,65,65,65,64,64,64,
    64,64,64,63,63,62,62,62,62,62,61,61,61,61,60,60,60,60,60,60,60,
    59,59,59,59,59,59,58,58,58,58,57,57,57,56,56,56,56,56,56,56,55,
    55,55,55,54,54,54,54,54,54,54,53,53,53,53,53,53,53,52,52,52,52,
    52,52,52,52,51,51,51,51,51,51,51,50,50,50,50,49,49,48,48,48,48,
    48,48,48,48,48,48,47,47,47,47,47,47,46,46,46,46,46,46,46,46,45,
    45,45,45,45,45,45,44,44,44,44,44,44,43,43,43,42,42,42,42,42,41,
    41,41,41,41,40,40,40,40,40,39,39,39,38,38,38,38,38,37,37,37,37,
    37,37,37,36,36,36,36,36,36,35,35,35,35,35,34,34,34,34,34,34,34,
    34,34,34,33,33,33,33,33,33,33,33,33,33,32,32,32,32,32,32,32,32,
    32,32,32,31,31,31,31,31,31,30,30,30,30,30,30,30,29,29,29,29,29,
    29,29,28,28,28,28,28,28,28,28,27,27,27,27,27,27,27,27,27,27,26,
    26,26,26,26,26,26,26,25,25,25,25,25,25,25,24,24,23,23,23,23,23,
    23,23,23,22,22,22,22,22,22,22,21,21,21,21,21,21,20,20,20
  };
  const int n4c2w4_a[] = {
    120, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,100,100,99,99,99,99,99,99,99,99,99,98,98,
    98,98,98,98,98,98,97,97,97,97,97,97,97,97,97,97,97,96,96,96,96,
    96,96,96,96,96,96,96,95,95,95,95,95,95,95,95,95,95,95,95,95,94,
    94,94,94,94,93,93,92,92,92,92,92,92,92,92,91,91,91,91,91,91,90,
    90,90,90,90,90,90,90,89,89,89,89,89,89,89,89,88,88,88,88,88,88,
    88,87,87,87,87,87,87,87,87,87,87,87,87,86,86,85,85,85,85,85,85,
    84,84,84,84,84,83,83,83,83,83,83,82,82,82,81,81,81,81,81,81,81,
    81,80,80,79,79,79,79,79,79,78,78,78,78,78,78,78,78,78,78,78,78,
    77,77,77,77,76,76,76,76,76,76,76,76,75,75,75,75,75,75,75,75,75,
    75,74,74,74,74,74,74,74,74,74,74,74,74,73,73,73,73,73,73,73,72,
    72,72,71,71,71,71,71,71,71,71,71,70,70,70,70,70,69,69,69,69,68,
    68,68,68,68,68,68,68,68,68,68,68,68,67,67,67,67,67,66,66,66,66,
    66,66,66,66,66,65,65,65,65,65,65,65,65,64,64,63,63,63,63,63,63,
    63,63,63,62,62,62,62,62,61,61,61,61,61,60,60,60,60,60,59,59,59,
    59,59,59,59,59,58,58,58,58,58,58,57,57,57,57,57,57,57,56,56,56,
    56,56,55,55,55,55,55,54,54,54,54,54,54,54,54,53,53,53,53,52,52,
    52,52,52,52,52,52,51,51,51,51,51,51,50,50,50,50,50,50,50,50,50,
    50,49,49,49,49,49,49,49,49,49,48,48,48,48,48,47,47,47,47,47,47,
    47,47,47,46,46,46,46,45,45,45,45,45,45,45,44,44,44,44,44,44,43,
    43,43,43,43,43,43,43,43,42,42,41,41,41,41,40,40,40,40,40,40,40,
    40,40,40,40,40,40,39,39,39,39,39,39,38,38,38,38,38,38,38,38,38,
    37,37,37,37,37,37,37,36,36,36,36,36,36,35,35,35,35,35,35,35,35,
    35,35,35,35,35,35,34,34,34,34,34,34,34,34,34,33,33,33,33,33,33,
    33,33,33,33,33,32,32,32,32,32,32,31,31,31,31,30,30,30,30,30
  };
  const int n4c2w4_b[] = {
    120, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,99,99,99,99,98,98,98,98,98,98,97,97,97,97,97,
    97,96,96,95,95,95,95,95,95,95,95,94,94,94,94,94,94,94,94,94,94,
    94,93,93,93,93,93,93,93,93,92,92,92,92,92,92,92,92,92,92,91,91,
    91,91,91,91,90,90,90,90,90,89,89,89,89,89,89,89,88,88,88,88,88,
    88,88,87,87,87,87,86,86,86,86,85,85,85,84,84,84,84,83,83,83,83,
    82,82,82,82,82,82,82,82,82,82,82,82,81,81,81,81,81,81,80,80,80,
    80,80,80,80,79,79,79,79,79,79,79,79,79,79,78,78,78,78,78,78,78,
    77,77,77,77,77,77,77,77,77,76,76,76,76,76,76,76,76,76,76,75,75,
    75,75,75,75,74,74,74,74,74,74,74,74,74,74,73,73,73,73,73,73,73,
    72,72,72,72,72,72,72,72,72,72,72,71,71,71,71,71,71,70,70,70,70,
    70,70,70,69,69,69,69,69,69,69,69,69,69,68,68,68,68,68,68,68,68,
    67,67,67,67,67,67,66,66,66,65,65,65,65,64,64,63,63,63,63,63,63,
    63,62,62,62,62,62,61,61,61,61,61,61,61,61,60,60,60,60,60,60,60,
    59,59,59,59,59,59,59,58,58,58,58,58,58,58,58,58,58,58,57,57,57,
    57,57,57,57,56,56,56,56,56,56,56,55,55,55,55,55,55,55,54,54,54,
    54,54,54,54,54,53,53,53,53,53,53,52,52,52,52,52,52,52,51,51,51,
    51,51,51,51,51,51,50,50,50,50,50,50,50,50,50,49,49,49,49,49,49,
    49,49,49,49,49,49,48,48,48,48,48,48,48,47,47,47,47,47,47,46,46,
    46,46,46,46,46,45,45,45,45,45,45,45,45,44,44,44,44,44,44,43,43,
    43,43,43,43,43,42,42,42,42,42,42,42,42,42,41,41,41,41,41,41,41,
    41,40,40,40,40,40,40,40,40,39,39,38,38,38,38,38,38,38,37,37,37,
    37,37,37,36,36,36,36,36,36,35,35,35,35,35,35,35,35,34,34,34,34,
    34,34,34,33,33,33,33,32,32,32,32,32,32,32,32,32,32,32,32,32,32,
    31,31,31,31,31,31,31,31,30,30,30,30,30,30,30,30,30,30,30
  };
  const int n4c2w4_c[] = {
    120, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,100,99,99,99,98,98,97,97,97,97,97,96,96,95,
    95,95,95,95,95,95,95,95,95,94,94,94,94,94,93,93,93,93,93,92,92,
    92,92,92,92,92,92,92,92,91,91,91,91,90,90,90,90,89,89,89,89,89,
    89,89,88,88,88,88,88,88,88,88,88,88,88,88,87,87,87,86,86,86,86,
    86,86,86,86,86,86,86,86,86,85,85,85,85,84,84,84,84,84,84,84,83,
    83,83,83,83,83,83,83,83,83,82,82,82,82,82,82,81,81,81,81,81,81,
    80,80,80,80,80,80,80,80,79,79,79,79,79,79,79,79,79,78,78,78,78,
    78,77,77,77,77,77,77,77,77,77,77,77,77,77,77,77,77,77,77,76,76,
    76,76,76,76,76,76,76,76,76,75,75,75,75,75,75,75,75,75,75,75,75,
    75,75,74,74,74,74,74,74,74,73,73,73,73,73,73,73,73,73,73,72,72,
    72,72,72,72,72,71,71,71,70,70,70,70,70,70,70,70,70,69,69,69,69,
    69,69,69,69,69,69,69,68,68,68,68,68,68,68,68,67,67,67,67,66,66,
    66,66,66,65,65,65,65,64,64,64,64,64,64,63,63,63,63,63,63,63,63,
    62,62,62,62,62,62,62,62,62,62,62,61,61,61,61,61,60,60,60,60,60,
    59,59,59,59,59,58,58,58,58,58,58,57,57,57,57,57,57,57,57,57,56,
    56,56,56,56,56,55,55,55,55,55,55,55,55,55,55,55,54,54,54,54,54,
    54,53,53,53,53,53,52,52,52,52,52,52,52,52,52,52,52,52,52,51,51,
    51,50,50,50,50,50,50,50,49,49,49,49,49,49,49,49,48,48,48,48,48,
    48,48,48,48,48,48,47,47,47,47,46,46,46,46,46,46,45,45,45,45,45,
    45,45,45,45,45,44,44,44,44,44,44,44,43,43,43,43,43,43,42,42,42,
    42,41,41,41,41,41,41,41,40,40,40,40,40,40,39,39,39,39,39,39,39,
    38,38,38,38,38,38,38,38,37,37,37,37,37,36,36,36,36,36,35,35,35,
    34,34,34,34,34,34,34,34,33,33,33,33,32,32,32,32,32,32,32,31,31,
    31,31,31,31,31,31,31,31,31,31,31,31,30,30,30,30,30,30,30
  };
  const int n4c2w4_d[] = {
    120, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,99,99,99,99,99,99,98,98,98,98,98,98,98,98,98,98,97,97,
    97,97,97,97,96,96,96,96,96,96,96,95,95,94,94,94,94,94,94,94,93,
    93,93,93,92,92,92,92,92,92,91,91,91,91,91,91,91,91,90,90,90,90,
    90,90,90,89,89,89,89,89,89,89,89,89,89,89,89,89,88,88,88,88,87,
    87,87,87,87,87,87,87,87,87,87,86,86,86,86,86,86,86,86,86,86,85,
    85,85,85,85,85,84,84,84,84,84,83,83,83,83,83,82,82,82,82,82,82,
    82,82,82,82,82,82,82,82,81,81,81,81,81,81,81,81,81,81,80,80,80,
    80,80,80,80,80,80,79,79,79,79,79,79,79,79,78,78,78,78,78,78,78,
    77,77,77,77,77,77,77,77,77,77,77,77,77,76,76,76,76,75,75,75,75,
    75,74,74,74,74,74,74,73,73,73,73,73,73,73,73,73,72,72,72,72,72,
    72,72,72,72,71,71,71,71,71,71,71,71,70,70,70,70,70,70,70,69,69,
    69,69,68,68,68,68,68,67,67,67,67,67,67,67,66,66,66,66,66,66,65,
    65,65,65,65,65,65,65,65,65,65,64,64,64,64,64,64,64,64,63,63,63,
    63,63,63,63,62,62,62,62,62,62,62,61,61,61,61,61,61,60,60,60,60,
    60,60,60,60,60,60,59,59,59,59,59,59,58,58,58,58,58,57,57,57,57,
    57,57,57,57,57,57,56,56,56,56,56,56,55,55,55,55,55,55,54,54,54,
    54,54,54,54,54,53,53,53,53,53,53,52,52,52,52,52,52,52,51,51,51,
    51,51,51,51,50,50,49,49,49,49,49,49,49,48,48,48,48,48,48,48,47,
    47,47,47,47,47,47,47,46,46,46,46,46,46,45,45,45,45,44,44,44,44,
    44,44,44,44,44,43,43,43,43,43,43,43,42,42,42,42,42,42,42,42,42,
    41,41,41,41,41,41,41,41,41,41,41,41,40,40,40,40,39,39,39,39,39,
    39,39,39,38,38,38,38,38,37,37,37,37,37,37,36,36,36,36,36,36,36,
    35,35,35,35,34,34,34,34,34,34,34,34,33,33,33,33,33,33,33,33,32,
    32,32,32,32,31,31,31,31,31,31,31,31,31,31,31,30,30,30
  };
  const int n4c2w4_e[] = {
    120, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,99,99,99,99,99,99,99,99,99,99,98,98,98,98,98,98,98,
    98,98,98,98,98,98,98,98,98,97,97,97,97,97,97,97,97,97,97,96,96,
    96,96,96,96,96,95,95,95,95,95,94,94,94,93,93,93,93,93,93,92,92,
    92,92,92,92,92,91,91,91,91,91,91,90,90,90,90,89,89,89,89,89,89,
    89,89,88,88,88,88,88,87,87,87,87,87,87,87,87,86,86,86,86,86,85,
    85,85,85,85,85,84,84,84,84,84,84,84,84,84,84,83,83,83,83,83,83,
    83,83,82,82,82,82,82,82,81,81,81,81,81,81,81,81,80,80,80,80,80,
    80,80,79,79,79,79,79,79,78,78,78,78,78,78,78,78,77,77,77,77,77,
    77,77,76,76,76,76,76,76,76,76,76,76,75,75,75,75,75,75,74,74,74,
    74,74,73,73,73,72,72,72,72,72,72,72,72,71,71,71,71,70,69,69,69,
    69,69,69,69,68,68,68,68,67,67,67,67,67,67,67,67,67,66,66,66,66,
    66,66,66,66,66,66,66,66,65,65,65,65,65,65,65,65,64,64,64,64,63,
    63,63,63,63,62,62,62,62,62,62,62,62,62,61,61,61,61,61,60,60,60,
    60,60,60,60,60,60,60,59,59,59,59,59,59,58,58,58,58,58,57,57,57,
    57,57,57,57,57,57,56,56,56,56,56,55,55,55,55,55,55,55,55,55,55,
    55,55,55,54,54,54,54,54,54,54,54,54,54,54,53,53,53,53,53,53,53,
    53,53,52,52,52,52,52,52,52,52,52,52,51,51,51,51,51,51,51,51,50,
    50,50,50,50,50,50,50,49,49,49,49,49,49,48,48,47,47,47,47,46,46,
    46,46,46,46,46,46,46,46,46,45,45,45,45,45,45,45,45,45,44,44,44,
    44,44,43,43,43,43,43,42,42,42,42,42,42,42,41,41,41,41,40,40,40,
    40,40,40,40,40,40,40,39,39,39,39,39,39,38,38,38,38,38,38,38,38,
    38,37,37,37,37,37,37,37,37,37,37,36,36,36,36,36,36,36,36,36,36,
    35,35,35,34,34,34,34,34,34,34,34,34,34,34,33,33,33,32,32,32,32,
    32,32,32,31,31,31,31,31,31,31,31,31,31,30,30,30,30,30
  };
  const int n4c2w4_f[] = {
    120, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,100,99,99,99,99,99,99,99,99,99,99,98,98,98,
    98,98,98,98,98,98,97,97,97,97,97,97,97,97,96,96,96,96,96,96,95,
    95,95,95,95,95,95,95,94,94,94,94,94,94,94,94,93,93,93,93,93,93,
    93,92,92,92,92,92,92,92,91,91,91,91,91,90,90,90,90,90,90,90,89,
    89,89,89,89,88,88,88,88,88,88,88,88,88,88,88,87,87,87,87,87,87,
    86,86,86,86,86,85,85,85,85,85,85,85,85,85,84,84,84,84,84,84,83,
    83,83,83,83,83,83,82,82,82,82,82,82,82,82,81,81,81,81,81,81,81,
    81,81,81,81,81,81,80,80,80,80,80,80,80,80,80,80,80,80,80,80,80,
    80,80,79,79,79,79,79,79,79,79,78,78,78,78,78,78,77,77,77,76,76,
    76,76,75,75,75,75,75,75,74,74,74,74,74,73,73,73,73,73,73,73,72,
    72,72,72,72,72,72,72,72,71,71,71,71,71,71,71,71,71,70,70,70,70,
    70,70,69,69,69,69,69,69,69,69,69,68,68,68,68,68,68,67,67,67,67,
    67,67,67,67,66,66,66,66,66,66,66,66,65,65,65,65,65,65,64,64,64,
    64,64,63,63,63,63,63,63,63,63,62,62,62,62,62,62,61,61,61,61,61,
    61,61,61,61,61,61,60,60,60,59,59,59,59,59,59,59,59,59,58,58,58,
    58,58,58,58,58,57,57,57,57,57,56,56,56,56,56,56,56,56,56,55,55,
    55,55,55,55,55,55,55,55,54,54,54,54,54,54,54,53,53,53,53,53,53,
    53,52,52,52,52,52,52,52,52,52,52,52,51,51,51,51,51,50,50,50,50,
    50,50,50,50,50,50,49,49,48,48,48,48,48,48,48,47,47,47,46,46,46,
    46,46,46,46,46,46,46,46,45,45,45,45,45,45,45,44,44,44,44,44,43,
    43,43,43,43,43,43,43,42,42,42,41,41,41,41,41,41,41,40,40,40,40,
    40,40,40,40,40,40,40,40,39,39,39,39,39,39,39,39,39,38,38,38,38,
    38,37,37,37,37,36,36,36,36,35,35,35,35,35,34,34,34,33,33,33,33,
    33,33,33,32,32,32,32,31,31,31,31,31,31,31,30,30,30,30,30
  };
  const int n4c2w4_g[] = {
    120, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,100,100,100,100,100,100,100,99,99,99,99,99,
    99,99,99,99,98,98,98,97,97,97,97,97,97,97,97,97,97,96,96,96,96,
    96,96,96,96,96,95,95,95,95,95,95,94,94,94,94,94,94,94,93,93,93,
    93,92,92,92,92,92,91,91,91,91,91,91,91,91,91,91,91,91,90,90,90,
    90,90,90,90,90,90,90,89,89,89,89,89,89,89,89,89,89,89,89,88,88,
    88,88,88,87,87,87,87,87,86,86,86,86,85,85,85,85,85,85,85,85,85,
    84,84,84,84,84,84,84,84,84,83,83,83,83,83,83,83,83,83,82,82,82,
    82,82,82,81,81,81,81,81,81,81,81,80,80,80,80,79,79,79,79,79,79,
    79,78,78,78,78,78,78,78,77,77,77,77,77,77,77,77,76,76,76,76,76,
    76,75,75,75,75,74,74,74,74,74,74,74,73,73,73,73,72,72,72,72,72,
    72,71,71,71,71,71,71,71,71,70,70,70,70,70,70,70,69,69,69,69,69,
    69,69,69,68,68,68,68,68,68,67,67,67,67,67,67,66,66,66,66,66,66,
    65,65,65,65,65,65,65,64,64,64,64,64,64,64,64,63,63,63,63,63,62,
    62,62,62,62,62,62,61,61,61,61,61,61,61,60,60,60,60,60,60,60,60,
    59,59,59,59,59,59,59,58,58,58,58,57,57,57,57,56,56,56,56,56,55,
    55,55,55,55,55,55,55,55,54,54,54,54,54,54,54,53,53,53,53,53,52,
    52,52,52,52,52,52,51,51,51,51,51,51,51,50,50,50,50,50,50,50,49,
    49,49,49,49,48,48,48,48,48,48,48,47,47,47,47,47,47,46,46,46,46,
    45,45,45,45,44,44,44,44,44,43,43,43,43,43,43,42,42,42,42,42,42,
    42,42,42,42,42,41,41,41,41,41,41,41,41,41,40,40,40,40,40,40,40,
    39,39,39,39,39,38,38,38,38,38,38,38,38,37,37,37,37,37,37,37,37,
    37,36,36,36,36,36,36,36,36,35,35,35,35,35,35,35,35,35,34,34,34,
    34,34,34,34,34,34,34,34,34,34,34,33,33,33,33,33,33,33,33,33,33,
    33,33,32,32,32,32,32,32,32,31,31,31,31,31,30,30,30,30,30,30,30
  };
  const int n4c2w4_h[] = {
    120, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,100,99,99,99,98,98,98,98,98,98,98,97,97,97,
    97,97,97,97,97,97,97,97,97,97,97,96,96,96,96,96,95,95,95,95,95,
    94,94,94,94,94,94,94,93,93,93,92,92,92,92,92,91,91,91,91,91,90,
    90,90,90,90,90,90,90,89,89,89,89,89,89,89,89,89,88,88,88,88,88,
    88,88,88,88,87,87,86,86,86,86,86,86,86,86,85,85,85,85,85,85,85,
    85,85,84,84,84,84,84,84,84,84,83,83,83,83,83,83,82,82,82,81,81,
    81,81,81,81,81,81,80,80,80,80,80,80,79,79,79,79,79,79,79,79,78,
    78,78,78,78,78,78,78,78,78,77,77,77,77,77,77,77,77,76,76,76,76,
    76,76,76,75,75,75,75,75,75,75,75,75,75,75,75,75,74,74,74,74,74,
    74,74,73,73,73,73,73,73,73,73,73,72,72,72,72,72,72,71,71,71,71,
    71,71,71,71,71,71,71,71,70,70,70,70,70,70,70,69,69,69,69,69,69,
    69,69,69,68,68,68,68,68,68,68,68,68,67,67,67,67,67,67,67,66,66,
    66,66,66,66,66,66,65,65,65,65,65,65,65,65,65,64,64,64,64,64,64,
    64,64,63,63,63,63,62,62,62,62,62,62,62,62,61,61,61,61,61,61,61,
    60,60,60,60,60,60,59,59,59,58,58,58,58,58,58,58,58,58,57,57,57,
    57,57,57,57,57,57,56,56,56,56,56,56,56,55,55,55,55,55,55,55,54,
    54,54,54,53,53,53,53,53,52,52,52,52,52,52,52,52,51,51,51,51,51,
    51,51,51,51,51,50,50,50,50,50,50,50,50,50,50,49,49,49,49,49,49,
    49,49,49,49,48,48,48,48,48,48,47,47,47,47,47,47,47,47,47,47,46,
    46,46,46,46,45,45,45,45,45,44,44,44,44,44,44,44,44,43,43,43,43,
    42,42,42,42,41,41,41,41,41,41,41,41,41,41,40,40,40,40,40,40,39,
    39,39,39,39,39,38,38,38,38,38,38,38,37,37,37,36,36,36,36,36,36,
    35,35,35,35,35,35,34,34,34,34,34,33,33,33,33,33,33,32,32,32,32,
    32,32,32,32,32,32,32,31,31,31,31,31,31,31,30,30,30,30,30
  };
  const int n4c2w4_i[] = {
    120, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,100,100,100,100,100,100,99,99,99,99,99,99,
    98,98,98,98,98,97,97,97,97,97,97,97,97,97,97,97,97,96,96,96,96,
    96,96,95,95,95,95,95,95,95,95,95,94,94,94,93,93,93,93,93,93,93,
    93,93,93,92,92,92,92,92,92,92,92,91,91,91,91,91,91,90,90,90,90,
    89,89,89,89,89,89,89,89,89,88,88,88,88,87,87,87,87,87,86,86,86,
    86,86,86,86,86,86,86,85,85,85,85,85,85,84,84,84,84,84,84,83,83,
    83,83,83,83,83,83,82,82,82,82,82,81,81,81,81,81,80,80,80,80,80,
    80,80,80,80,80,80,79,79,79,79,79,79,79,78,78,78,78,78,77,77,77,
    77,77,77,76,76,76,76,76,75,75,75,75,74,74,74,74,74,74,74,74,74,
    74,73,73,73,73,73,72,72,72,72,71,71,71,71,71,71,71,71,70,70,70,
    70,70,70,70,70,69,69,69,68,68,68,68,68,68,68,68,67,67,67,67,67,
    67,67,66,66,66,66,66,66,66,65,65,65,65,65,65,65,64,64,64,64,64,
    64,63,63,63,63,63,63,63,62,62,62,62,62,62,62,62,62,62,62,62,61,
    61,61,61,61,61,61,61,61,61,60,60,60,60,60,60,60,60,60,60,59,59,
    59,59,59,59,58,58,58,58,58,58,58,58,58,58,58,58,58,57,57,57,57,
    57,57,57,56,56,56,55,55,55,55,55,55,55,55,55,54,54,54,54,54,54,
    54,53,53,53,53,53,53,53,53,53,52,52,52,52,52,52,52,52,51,51,51,
    51,50,50,50,50,49,49,49,49,49,49,49,49,48,48,48,48,47,47,47,47,
    47,47,47,47,47,46,46,46,45,45,45,45,45,45,45,44,44,44,44,44,44,
    44,43,43,43,43,43,43,43,43,42,42,42,42,42,42,41,41,41,41,41,41,
    41,41,41,41,41,40,40,40,40,40,40,39,39,39,39,39,39,39,39,38,38,
    38,38,38,38,38,38,37,37,37,37,37,37,37,37,37,37,37,36,36,36,36,
    35,35,35,35,35,35,34,34,34,34,33,33,33,33,33,33,33,33,32,32,32,
    32,32,32,32,32,32,31,31,31,31,31,31,31,30,30,30,30,30,30,30,30
  };
  const int n4c2w4_j[] = {
    120, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,99,99,99,99,99,99,99,99,98,98,98,98,98,98,
    97,97,97,97,97,97,97,97,97,97,97,97,96,96,96,96,96,96,95,95,95,
    95,95,94,94,94,94,94,94,94,94,94,94,94,94,94,93,93,93,93,93,93,
    93,93,93,93,92,92,92,92,92,92,92,91,91,91,91,91,91,91,91,91,90,
    90,90,90,90,90,90,90,90,90,90,90,90,89,89,89,89,89,89,89,89,89,
    88,88,88,88,88,88,88,88,88,87,87,87,86,86,86,86,86,85,85,85,84,
    84,83,83,83,83,83,83,83,83,82,82,82,82,82,81,81,81,81,81,81,81,
    80,80,80,80,80,80,80,80,80,80,80,80,79,79,79,79,79,79,79,79,79,
    79,79,78,78,78,78,78,78,78,78,77,77,77,77,77,77,77,77,76,76,76,
    76,76,76,75,75,75,75,75,75,75,74,74,74,74,73,73,72,72,72,72,72,
    72,71,71,71,71,71,71,70,70,70,70,70,70,69,69,69,69,69,69,69,69,
    69,69,69,68,68,68,68,68,68,68,68,67,67,67,67,67,67,67,66,66,66,
    66,66,66,66,66,66,66,65,65,65,65,65,65,65,64,64,64,64,64,64,64,
    64,64,63,63,63,63,63,63,63,63,62,62,62,62,62,62,62,62,62,61,61,
    61,61,61,61,61,61,60,60,60,60,59,59,59,59,59,58,58,58,58,57,57,
    57,57,57,57,57,57,57,56,56,56,56,56,56,56,56,56,56,56,56,56,55,
    55,55,55,55,55,55,54,54,54,54,54,54,54,53,53,53,53,53,53,53,53,
    53,53,52,52,52,52,52,51,51,51,51,51,51,50,50,50,50,50,50,50,49,
    49,49,49,49,49,49,48,48,48,48,48,47,47,47,47,47,47,47,46,46,46,
    46,46,46,46,46,45,45,45,44,44,44,44,44,44,44,44,44,44,43,43,43,
    43,43,42,42,42,42,42,42,42,42,41,41,41,41,41,41,41,41,41,41,40,
    40,40,40,40,40,40,40,40,40,39,39,39,39,39,39,39,38,38,38,38,38,
    38,37,37,37,37,36,36,36,36,35,35,35,35,35,34,34,34,34,34,34,34,
    33,33,33,32,32,32,32,32,32,31,31,31,31,30,30,30,30,30,30
  };
  const int n4c2w4_k[] = {
    120, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,99,99,99,99,99,99,99,99,99,99,99,98,98,98,
    98,97,97,97,97,97,97,97,96,96,96,96,96,96,95,95,95,95,95,95,95,
    95,95,95,94,94,94,94,94,94,94,94,94,93,93,93,93,92,92,92,92,92,
    92,92,92,91,91,91,90,90,90,90,90,90,90,90,90,89,89,89,89,89,89,
    89,88,88,88,88,88,88,88,88,88,87,87,87,87,87,87,87,87,87,86,86,
    86,86,86,85,85,85,85,85,85,85,84,84,84,84,84,84,84,83,83,83,83,
    83,83,82,82,82,82,82,82,82,82,82,82,82,81,81,81,81,81,81,81,81,
    80,80,80,80,80,79,79,79,79,79,79,79,79,78,78,78,78,78,78,78,78,
    78,78,77,77,77,77,77,77,76,76,76,76,76,75,75,75,75,75,75,74,74,
    74,74,74,74,74,74,74,73,73,73,73,73,73,73,72,72,72,72,72,72,72,
    72,72,72,71,71,71,70,70,70,70,70,70,69,69,69,69,69,69,68,68,68,
    68,68,68,68,67,67,67,67,67,67,67,66,66,66,66,66,66,66,65,65,65,
    65,65,65,65,65,64,64,64,64,64,64,63,63,63,63,62,62,62,62,62,61,
    61,61,60,60,60,60,60,60,59,59,59,59,59,59,58,58,58,58,58,58,58,
    58,57,57,57,57,57,57,57,57,57,57,56,56,56,56,56,56,55,55,55,55,
    55,55,55,54,54,54,54,54,54,54,54,54,54,54,54,53,53,53,53,53,53,
    53,53,52,52,52,52,52,52,51,51,51,51,51,51,50,50,50,50,50,50,50,
    50,50,49,49,49,49,49,49,49,49,49,49,48,48,48,48,48,48,47,47,47,
    47,46,46,46,46,46,46,46,45,45,45,45,45,45,45,44,44,44,44,43,43,
    43,42,42,42,42,42,42,42,42,41,41,41,41,41,40,40,40,40,40,40,40,
    40,40,40,40,40,40,40,39,39,39,39,39,39,39,39,38,38,38,38,38,38,
    38,38,38,38,38,38,37,36,36,36,36,36,36,36,36,36,36,36,36,36,36,
    35,35,35,35,35,35,35,34,34,34,34,34,34,34,33,33,33,33,33,33,32,
    32,32,32,32,32,32,32,31,31,30,30,30,30,30,30,30,30,30,30
  };
  const int n4c2w4_l[] = {
    120, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,100,100,100,100,100,99,99,99,99,99,99,99,
    99,99,99,98,98,98,98,98,98,98,98,98,98,98,98,98,98,97,97,97,97,
    97,97,97,97,97,97,96,96,96,96,96,96,96,95,95,95,95,95,94,94,94,
    94,94,94,94,94,94,93,93,93,93,93,93,93,93,93,92,92,92,92,92,92,
    92,91,91,91,91,91,91,91,91,90,90,90,90,90,90,90,89,89,89,88,88,
    88,88,88,88,88,88,87,87,87,87,87,87,87,87,87,86,86,86,86,86,85,
    85,85,85,85,84,84,84,84,83,83,83,83,83,83,83,82,82,82,81,81,81,
    81,81,81,81,80,80,80,80,80,80,79,79,79,79,79,79,79,79,78,78,78,
    78,78,78,78,78,77,77,77,77,77,76,76,76,76,75,75,75,75,75,75,75,
    74,74,74,74,74,74,74,74,74,74,73,73,73,73,72,72,72,72,72,72,72,
    72,72,71,71,71,71,71,71,71,71,71,70,70,70,70,69,69,69,69,69,69,
    69,69,68,68,68,68,68,68,68,68,68,68,68,68,68,68,67,67,67,67,67,
    67,67,67,67,66,66,66,66,66,66,66,66,66,66,65,65,65,65,65,65,64,
    64,64,64,64,64,63,63,63,63,63,62,62,62,62,61,61,61,61,60,60,60,
    60,60,59,59,59,59,59,59,59,59,59,59,59,59,58,58,58,58,58,58,58,
    58,58,57,57,56,56,56,56,56,55,55,55,55,55,55,55,55,55,54,54,54,
    54,54,54,54,54,53,53,53,53,53,53,53,53,53,53,52,52,52,52,52,52,
    51,51,51,51,51,50,50,49,49,49,49,49,49,49,49,48,48,48,48,48,48,
    47,47,47,47,47,47,47,47,47,46,46,46,46,46,46,46,46,46,45,45,45,
    45,44,44,44,44,44,44,44,43,43,43,43,42,42,42,42,42,42,42,41,41,
    41,41,41,41,41,41,41,40,40,40,40,40,40,40,40,40,39,39,39,39,39,
    39,39,39,39,38,38,38,38,38,38,38,37,37,37,37,37,37,37,37,37,37,
    36,36,36,36,36,36,36,35,35,35,35,34,34,34,34,34,34,33,33,33,33,
    33,33,33,33,33,33,33,33,32,31,31,31,31,31,30,30,30,30,30,30,30
  };
  const int n4c2w4_m[] = {
    120, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,100,100,100,100,99,99,99,99,99,99,99,99,98,
    98,98,98,98,97,97,97,97,97,97,97,97,96,96,96,96,96,95,95,95,95,
    95,94,94,94,94,94,94,94,94,93,93,93,93,93,93,93,93,92,92,92,92,
    91,91,91,91,91,90,90,90,90,90,90,90,90,89,89,89,89,89,89,89,89,
    89,89,89,89,88,88,88,88,87,87,87,87,87,87,87,87,86,86,86,86,86,
    86,86,86,85,85,85,85,85,85,85,85,84,84,84,84,84,84,84,84,84,84,
    84,84,83,83,83,83,83,83,82,82,82,81,81,81,81,81,81,81,80,80,80,
    80,80,80,79,79,79,79,79,79,79,79,79,79,78,78,78,78,78,78,78,78,
    78,78,78,78,77,77,77,77,76,76,76,76,76,76,75,75,75,75,75,75,75,
    75,75,74,74,74,74,74,73,73,73,73,73,73,73,72,72,72,72,72,72,72,
    71,71,71,71,70,70,70,70,70,70,70,70,69,69,69,69,69,69,69,69,69,
    68,68,68,68,68,68,68,67,67,67,67,67,67,67,67,67,67,66,66,66,66,
    65,65,65,64,64,64,64,64,63,63,63,63,63,63,62,62,62,62,62,62,62,
    62,61,61,61,61,60,60,60,60,60,60,60,60,59,59,59,59,59,59,59,58,
    58,58,58,58,57,57,57,57,57,57,57,57,57,56,56,56,56,56,56,56,55,
    55,55,55,55,55,55,54,54,54,54,54,54,54,53,53,53,53,53,53,53,53,
    53,53,53,53,53,52,52,52,52,52,52,52,51,51,51,51,51,51,50,50,50,
    50,50,49,49,49,49,49,49,48,48,48,48,48,47,47,47,47,47,46,46,46,
    46,46,46,45,45,45,45,45,45,44,44,44,44,44,44,44,44,44,44,44,43,
    43,43,43,43,43,43,43,43,43,42,42,42,42,42,42,41,41,41,41,40,40,
    40,39,39,39,39,39,39,39,39,38,38,38,38,38,38,38,38,37,37,37,37,
    37,37,37,37,37,37,36,36,36,36,36,36,36,36,36,36,36,35,35,35,35,
    35,34,34,34,34,34,34,34,34,33,33,33,33,33,33,33,33,33,33,33,32,
    32,32,32,32,32,32,32,32,31,31,31,31,30,30,30,30,30,30,30,30
  };
  const int n4c2w4_n[] = {
    120, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,100,100,99,99,99,99,99,98,98,97,97,97,97,
    97,97,97,96,96,96,96,96,96,96,96,96,96,95,95,95,95,95,95,95,95,
    95,95,94,94,94,94,94,94,94,94,94,94,93,93,93,93,93,93,93,93,93,
    92,92,92,92,92,92,92,92,92,92,92,92,92,92,92,91,91,91,91,91,91,
    91,90,90,90,90,90,89,89,89,89,89,89,89,89,89,89,88,88,88,87,87,
    87,87,86,86,86,86,85,85,85,85,84,84,84,84,84,84,84,84,84,84,84,
    84,84,84,84,83,83,83,83,83,82,82,82,82,82,81,81,81,81,81,81,81,
    81,81,81,81,81,80,80,80,80,80,80,80,80,79,79,79,79,79,79,79,79,
    79,79,78,78,78,78,78,78,78,78,78,77,77,77,77,76,76,76,76,76,76,
    76,75,75,75,75,75,75,74,74,74,74,74,74,74,74,73,73,73,73,73,73,
    72,72,72,72,72,72,72,72,72,72,71,71,71,71,70,70,70,69,69,69,69,
    69,69,69,69,69,69,69,69,68,68,68,68,68,68,67,67,67,67,67,67,67,
    67,67,67,67,66,66,66,66,66,66,65,65,65,65,64,64,64,64,64,64,64,
    64,63,63,63,63,63,63,63,63,62,62,62,62,62,62,62,61,61,61,61,61,
    61,60,60,60,60,60,60,60,60,60,60,60,60,60,59,59,59,59,58,58,58,
    58,58,58,57,57,57,57,57,57,57,57,57,57,56,56,56,55,55,55,55,55,
    55,55,55,55,55,55,54,54,54,54,54,53,53,53,53,53,53,53,52,52,52,
    52,52,52,51,51,51,51,51,50,50,50,50,50,50,50,50,50,49,49,49,49,
    49,49,49,49,49,48,48,48,47,47,47,47,47,47,47,46,46,46,46,46,46,
    46,46,45,45,45,45,45,45,45,45,45,45,45,44,44,44,44,44,44,44,44,
    44,43,43,43,43,42,42,42,41,41,41,41,41,41,41,41,41,41,41,40,40,
    40,40,40,40,39,39,39,39,39,39,39,39,39,39,38,38,38,38,38,37,37,
    37,37,37,36,36,36,36,36,36,36,35,35,34,34,34,34,34,33,33,33,33,
    33,33,32,32,32,32,32,32,32,31,31,31,31,31,31,31,31,31,30,30
  };
  const int n4c2w4_o[] = {
    120, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,99,99,99,99,99,98,98,98,98,98,98,98,98,98,
    98,97,97,97,97,97,96,96,96,96,95,95,95,95,95,95,95,95,94,94,94,
    94,94,94,94,94,93,93,93,93,93,92,92,92,92,92,92,92,92,92,92,92,
    92,91,91,91,91,91,91,91,91,90,90,90,90,90,90,90,90,89,89,89,89,
    89,88,88,88,88,88,88,88,88,88,88,88,87,87,87,87,87,87,86,86,86,
    86,86,86,86,86,86,86,86,86,86,85,85,85,85,85,85,85,84,84,84,84,
    84,84,84,84,84,83,83,83,83,83,83,83,83,83,83,82,82,82,82,82,82,
    82,81,81,81,81,81,81,81,80,80,80,80,80,80,79,79,79,79,79,79,79,
    78,78,78,78,78,78,78,78,78,77,77,77,77,76,76,76,76,76,76,76,75,
    75,75,75,75,75,75,74,74,74,74,73,73,73,73,73,72,72,72,72,72,72,
    72,72,72,71,71,71,71,71,71,71,71,71,71,71,71,70,70,70,70,70,70,
    70,70,69,69,69,69,69,69,68,68,68,67,67,67,67,66,66,66,66,66,66,
    66,66,65,65,65,65,64,64,64,63,63,63,62,62,62,62,62,62,62,61,61,
    61,61,61,61,61,60,60,60,60,59,59,59,59,58,58,58,58,58,58,58,58,
    58,58,58,57,57,57,57,57,57,57,57,57,57,56,56,56,56,56,56,56,56,
    56,55,55,55,55,55,55,55,54,54,54,54,54,54,54,54,54,54,54,54,53,
    53,53,53,53,53,53,52,52,52,52,52,52,51,51,51,51,51,51,50,50,50,
    50,50,50,50,50,49,49,49,49,49,48,48,47,47,47,47,47,47,47,47,47,
    47,46,46,46,46,46,46,45,45,45,45,45,45,44,44,44,44,44,44,44,44,
    43,43,43,43,43,43,43,43,43,42,42,42,42,42,42,42,41,41,41,41,41,
    41,41,41,41,41,40,40,40,40,39,39,39,39,39,39,39,39,39,39,38,38,
    38,38,38,38,38,38,38,37,37,37,37,37,37,37,37,37,36,36,36,36,35,
    35,35,35,35,34,34,34,34,34,34,34,34,34,34,33,33,33,33,33,32,32,
    32,32,32,32,32,32,31,31,31,31,31,31,31,31,30,30,30,30,30
  };
  const int n4c2w4_p[] = {
    120, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,100,100,100,99,99,99,99,99,99,99,99,98,98,
    98,98,98,98,97,97,97,97,97,97,97,97,97,97,97,96,96,96,96,96,95,
    95,95,95,95,95,95,95,95,95,94,94,94,94,94,94,94,94,94,94,94,93,
    93,93,93,93,93,93,92,92,92,92,91,91,91,91,91,91,91,91,91,91,91,
    90,90,90,90,90,90,89,89,89,89,89,89,89,89,89,88,88,88,88,88,88,
    88,88,88,87,87,87,87,87,87,87,87,87,87,86,86,86,86,86,85,85,85,
    85,85,85,85,85,84,84,84,84,84,84,84,83,83,83,83,83,83,82,82,82,
    82,82,82,82,82,82,82,81,81,81,81,81,81,81,81,81,81,81,80,80,80,
    80,80,79,79,79,79,79,79,79,78,78,78,78,78,78,78,77,77,77,77,77,
    76,76,75,75,75,75,75,75,75,74,74,74,74,74,74,74,74,74,73,73,73,
    73,73,73,73,72,72,72,72,72,72,72,72,72,72,72,71,71,71,70,70,70,
    70,70,70,70,70,70,70,69,69,69,69,68,68,68,68,67,67,66,66,66,66,
    66,66,66,66,66,66,66,66,66,65,65,65,64,64,64,64,64,63,63,63,63,
    63,63,63,63,63,63,62,62,62,62,62,62,62,62,62,61,61,61,61,61,61,
    60,60,60,60,60,60,60,59,59,59,59,59,59,59,59,58,58,58,58,58,58,
    57,57,57,57,56,56,56,56,56,56,56,55,55,55,55,55,55,55,55,55,55,
    54,54,54,54,54,54,54,54,54,54,54,54,53,53,53,53,52,52,52,52,52,
    51,51,51,51,51,51,51,50,50,50,50,50,50,50,50,50,50,50,50,49,49,
    49,49,49,49,48,48,48,48,48,48,48,48,48,47,47,47,47,47,47,46,46,
    46,46,46,46,46,46,46,46,46,45,45,45,44,44,44,44,44,43,43,43,43,
    43,43,42,42,42,42,42,42,41,41,41,41,41,41,41,41,40,40,40,40,39,
    39,39,39,39,39,39,38,38,38,38,38,38,37,37,37,37,37,37,36,36,36,
    36,36,35,35,34,34,34,34,34,34,34,34,34,33,33,33,33,33,33,33,33,
    33,32,32,32,32,32,32,32,32,32,32,32,32,31,31,31,31,30,30,30
  };
  const int n4c2w4_q[] = {
    120, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,99,99,99,99,99,98,98,98,98,98,98,97,97,97,97,97,96,
    96,96,96,96,96,96,96,96,96,96,96,96,95,95,95,95,95,95,95,94,94,
    94,94,94,94,93,93,93,93,93,93,93,93,93,93,92,92,92,92,92,92,91,
    91,91,91,91,90,90,90,90,90,90,89,89,89,89,89,89,89,88,88,88,88,
    88,88,87,87,87,86,86,86,86,86,86,86,85,85,85,85,85,84,84,84,84,
    84,84,84,84,84,84,83,83,83,83,83,83,83,83,83,83,83,83,83,83,83,
    83,83,82,82,82,82,82,82,82,82,82,82,82,82,82,82,81,81,81,81,81,
    81,81,81,81,81,80,80,80,80,80,80,80,80,80,80,80,79,79,79,79,79,
    79,79,79,79,78,78,78,78,77,77,77,77,77,76,76,76,76,75,75,75,75,
    75,75,75,75,75,74,73,73,73,73,73,73,73,73,73,72,72,72,72,71,71,
    71,71,71,71,71,70,70,70,70,70,70,69,69,69,69,69,69,68,68,68,68,
    67,67,67,67,67,67,67,67,67,66,66,66,66,65,65,65,65,65,65,65,64,
    64,64,64,64,64,63,63,63,63,63,63,63,63,63,63,63,63,63,63,62,62,
    62,62,62,62,62,62,62,61,61,61,61,61,60,60,60,60,60,60,60,60,60,
    60,60,60,59,59,59,59,59,59,59,59,59,59,59,58,58,58,58,58,57,57,
    57,57,57,57,57,56,56,56,56,56,55,55,55,55,55,55,54,54,54,54,54,
    53,53,53,53,53,53,53,53,53,53,52,52,52,52,51,51,51,51,51,51,51,
    51,50,50,50,50,50,50,50,50,50,50,49,49,49,49,49,49,49,48,48,48,
    47,47,47,47,46,46,46,46,46,46,45,45,45,45,45,44,44,43,43,43,43,
    43,43,43,43,42,42,42,42,42,41,41,41,41,41,41,41,41,40,40,40,40,
    40,40,40,40,40,40,40,39,39,39,39,39,39,39,38,38,38,38,38,38,37,
    37,37,37,37,37,36,36,36,36,36,36,36,35,35,35,35,35,35,34,34,34,
    34,34,33,33,33,33,32,32,32,32,32,32,32,32,32,32,32,31,31,31,31,
    31,31,31,31,31,31,30,30,30,30,30,30,30,30,30,30,30,30
  };
  const int n4c2w4_r[] = {
    120, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,99,99,99,99,98,98,98,98,98,98,98,98,98,97,97,
    97,97,97,97,97,97,97,97,97,96,96,96,96,96,96,96,96,96,95,95,95,
    95,95,95,95,95,95,95,95,95,94,94,94,94,93,93,93,93,93,93,93,92,
    92,92,92,92,91,91,91,91,91,91,91,91,91,90,90,90,90,90,90,90,89,
    89,89,89,88,88,88,88,88,87,87,87,87,87,86,86,86,86,86,86,86,85,
    85,85,85,84,84,84,84,84,84,84,83,83,83,83,83,83,83,83,83,83,83,
    83,83,82,82,82,82,82,82,81,81,81,81,81,81,81,81,81,81,80,80,80,
    80,80,80,79,79,79,79,79,79,79,78,78,78,78,78,77,77,77,77,77,77,
    77,77,77,76,76,76,76,76,76,76,76,75,75,75,75,75,75,75,74,74,74,
    74,74,74,74,74,73,73,73,73,73,72,72,72,72,72,71,71,71,71,71,71,
    71,71,71,71,70,70,70,70,70,70,70,70,70,69,69,69,69,69,69,69,69,
    69,68,68,68,68,67,67,67,67,67,67,67,67,66,66,66,66,66,66,66,66,
    66,65,65,65,65,65,65,65,65,65,65,65,65,65,64,64,64,64,64,64,64,
    64,64,63,63,63,63,63,63,63,63,63,63,62,62,62,62,62,62,62,62,61,
    61,61,61,61,61,60,60,60,60,60,59,59,59,59,58,58,58,58,58,58,57,
    57,57,57,57,57,57,57,56,56,56,56,56,56,56,55,55,55,55,54,54,54,
    54,54,54,54,54,54,54,54,54,53,53,53,53,52,52,52,52,52,52,52,52,
    51,51,51,51,51,51,50,50,50,50,50,50,49,49,49,49,49,49,49,48,48,
    47,47,47,47,46,46,46,46,46,46,46,45,45,45,45,45,45,45,45,44,44,
    44,44,44,44,44,44,43,43,43,43,43,43,43,43,43,43,43,42,42,42,42,
    42,42,41,41,41,41,41,40,40,40,39,39,39,39,39,39,39,39,39,39,38,
    38,38,38,38,38,38,38,37,37,37,37,37,37,36,36,36,36,36,36,36,36,
    36,36,36,35,35,35,35,35,35,35,35,34,34,34,34,34,34,34,34,33,33,
    33,33,33,33,33,32,32,32,32,32,32,32,32,32,31,31,31,30,30
  };
  const int n4c2w4_s[] = {
    120, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,100,100,100,100,99,99,99,99,99,98,98,98,98,
    98,98,97,97,97,97,96,96,96,95,95,95,95,95,95,95,95,95,95,95,94,
    94,94,94,94,94,94,94,93,93,93,93,93,93,93,93,93,93,93,93,93,93,
    92,92,92,92,91,91,91,91,90,90,90,90,90,90,90,90,89,89,89,89,89,
    89,88,88,88,87,87,87,87,87,87,86,86,86,86,86,86,86,86,86,86,85,
    85,85,85,85,85,85,85,85,85,85,85,84,84,84,84,84,83,83,83,83,83,
    83,83,83,83,83,82,82,82,82,82,82,82,82,81,81,81,81,80,80,80,80,
    79,79,79,79,79,79,79,79,78,78,78,78,78,78,78,78,78,78,77,77,77,
    77,76,76,76,76,76,76,76,76,76,76,75,75,75,75,75,75,75,75,75,75,
    74,74,74,74,74,74,74,73,73,73,73,73,73,73,72,72,72,72,72,72,71,
    71,71,71,71,71,71,71,71,71,71,71,71,70,70,70,70,70,70,70,70,70,
    69,69,69,69,69,69,68,68,68,68,68,68,68,68,67,67,67,66,66,66,66,
    65,65,65,65,65,65,64,64,64,64,64,64,63,63,63,63,63,63,63,63,63,
    63,62,62,62,62,62,62,62,61,61,61,61,61,61,61,61,60,60,60,60,60,
    60,59,59,59,59,59,59,59,59,58,58,58,58,58,58,57,57,57,57,57,57,
    57,57,56,56,56,56,56,56,56,55,55,55,55,55,55,55,54,54,54,54,54,
    53,53,53,53,53,53,53,53,53,53,52,52,52,52,52,51,51,51,50,50,50,
    50,50,50,50,50,50,50,50,50,50,50,50,49,49,49,49,49,49,49,49,48,
    48,48,48,48,48,48,48,47,47,47,47,47,47,46,46,46,46,46,45,45,45,
    45,45,44,44,44,44,44,44,43,43,43,43,43,43,43,43,42,42,42,42,42,
    42,42,42,42,41,41,41,41,41,41,41,41,40,40,40,40,40,40,40,40,40,
    40,40,39,39,39,39,39,38,37,37,37,37,37,37,36,36,36,36,36,36,36,
    36,35,35,35,35,35,35,34,34,34,34,34,34,34,33,33,33,33,33,32,32,
    32,32,32,32,32,32,32,32,31,31,31,31,30,30,30,30,30,30,30,30
  };
  const int n4c2w4_t[] = {
    120, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,100,100,99,99,99,99,99,98,98,98,98,97,97,
    96,96,96,96,96,96,96,96,95,95,95,95,95,95,95,94,94,94,94,94,94,
    94,94,94,93,93,93,93,93,92,92,92,92,92,92,91,91,91,91,91,91,91,
    91,91,91,91,91,91,91,91,91,90,90,90,90,90,90,89,89,89,89,89,89,
    88,88,88,88,88,88,88,88,88,88,87,87,87,87,87,87,87,87,86,86,86,
    85,85,85,85,85,85,84,84,84,83,83,83,83,83,83,83,82,82,82,82,82,
    82,82,82,82,82,82,81,81,81,81,81,81,81,81,80,80,80,80,79,79,79,
    79,79,79,79,78,78,78,78,78,78,78,78,77,77,77,77,77,77,77,77,77,
    77,77,76,76,76,76,76,75,75,75,75,75,75,74,74,74,74,74,74,74,74,
    73,73,73,73,73,73,73,73,73,72,72,72,72,72,72,72,71,71,71,71,71,
    71,71,71,71,70,70,70,70,70,70,70,70,70,69,69,69,69,69,69,69,69,
    68,68,68,68,68,68,67,67,67,67,67,67,67,67,67,66,66,66,65,65,65,
    65,65,65,64,64,64,64,64,64,64,64,64,63,63,63,63,63,63,63,63,63,
    63,63,62,62,62,62,62,62,62,61,61,61,60,60,60,60,59,59,59,59,59,
    59,59,59,59,58,58,58,58,58,58,58,57,57,57,57,57,57,56,56,56,56,
    56,56,55,55,55,55,55,55,55,55,55,54,54,54,54,54,53,53,53,53,53,
    53,53,53,52,52,52,52,52,52,52,51,51,51,51,51,51,51,50,50,50,50,
    50,49,49,49,49,49,49,48,48,48,48,48,48,48,48,47,47,47,47,46,46,
    46,46,46,46,46,46,46,46,45,45,45,45,45,45,45,45,45,44,44,44,44,
    44,43,43,43,43,43,43,43,42,42,42,42,42,42,42,41,41,41,41,41,41,
    40,40,40,40,40,40,40,40,39,39,39,39,39,39,38,38,38,38,38,38,38,
    37,37,37,37,37,37,37,37,37,37,37,36,36,36,36,36,36,36,36,36,36,
    35,35,35,35,34,34,34,34,34,34,34,33,33,33,33,33,32,32,32,32,32,
    31,31,31,31,31,31,31,31,31,31,31,30,30,30,30,30,30,30,30,30
  };
  const int n4c3w1_a[] = {
    150, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,99,99,99,99,99,99,99,99,98,98,97,97,97,96,
    96,96,96,96,96,96,96,95,95,95,95,94,94,94,93,93,93,93,93,92,92,
    92,92,92,91,91,91,91,91,90,90,89,89,89,89,89,89,88,88,88,88,86,
    86,85,85,85,84,84,84,84,83,83,83,83,83,83,83,82,82,81,81,81,81,
    81,81,81,81,81,81,80,80,80,80,79,79,79,79,79,79,79,78,78,78,78,
    78,78,78,77,77,77,77,77,77,76,75,75,74,74,74,74,74,74,74,73,73,
    73,72,72,72,72,72,72,72,72,72,71,70,70,69,69,68,68,68,68,68,67,
    66,66,66,65,65,65,65,65,65,65,65,65,64,64,64,64,64,64,64,64,63,
    63,63,62,62,62,62,62,62,61,61,61,60,60,60,60,60,60,60,59,59,59,
    59,59,59,59,59,59,59,59,58,58,58,57,57,57,57,56,56,56,56,56,56,
    56,55,55,55,54,54,54,54,54,54,54,54,54,53,53,53,52,52,52,52,51,
    51,51,51,50,50,50,49,49,49,48,48,48,48,48,48,48,48,48,48,48,48,
    47,47,47,47,47,46,46,46,46,45,45,45,45,45,45,45,45,44,44,44,44,
    44,43,43,43,43,43,43,43,43,42,42,42,42,42,42,41,41,41,41,41,41,
    41,41,40,40,40,40,39,39,39,39,39,38,38,38,37,37,37,37,37,37,36,
    36,36,36,36,35,35,35,35,34,34,34,34,34,34,34,34,33,33,33,33,33,
    32,32,32,32,32,32,32,31,31,31,31,30,30,30,30,30,30,30,30,30,30,
    29,29,29,28,28,28,28,28,28,27,27,27,27,26,26,26,25,25,25,25,25,
    25,25,24,24,24,24,24,24,23,23,23,23,23,22,22,22,22,22,22,21,21,
    21,21,21,21,21,21,21,20,20,20,20,20,19,19,19,19,19,19,19,19,18,
    18,18,18,18,18,18,18,18,17,17,16,16,16,15,15,15,15,15,14,14,14,
    14,14,14,14,13,13,13,13,12,12,12,11,11,11,11,11,10,10,10,10,10,
    9,9,9,9,8,8,8,7,7,7,7,7,7,7,6,6,6,6,6,6,6,6,5,5,4,4,4,3,3,3,3,
    3,2,2,2,2,1,1,1,1
  };
  const int n4c3w1_b[] = {
    150, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,100,100,100,100,100,100,100,100,99,99,99,
    99,99,99,99,98,98,98,97,97,97,97,96,96,96,95,95,95,95,95,95,94,
    93,93,93,92,92,92,92,92,91,91,91,91,91,91,90,89,89,88,87,87,87,
    87,87,86,86,86,86,86,85,85,85,85,84,84,84,84,84,84,83,83,83,82,
    82,82,82,82,81,81,81,81,81,80,80,80,80,80,80,80,80,80,79,79,79,
    79,78,78,78,77,77,77,76,76,76,75,75,75,75,75,75,74,74,73,73,73,
    73,72,72,72,72,72,71,71,70,69,69,69,69,69,68,68,68,68,68,68,68,
    68,68,67,67,67,66,65,65,65,65,65,65,64,64,64,64,64,63,63,63,63,
    62,62,61,61,61,61,61,60,60,60,60,60,60,60,60,60,59,59,59,59,59,
    59,59,58,58,58,58,58,58,58,58,57,57,57,57,56,56,56,56,55,55,55,
    55,55,55,55,54,54,54,54,54,54,53,53,53,53,53,52,52,52,52,52,52,
    52,52,52,52,51,51,51,51,51,51,51,51,50,50,50,50,50,50,50,49,49,
    49,49,49,49,49,48,48,48,48,48,48,47,47,47,47,47,47,46,46,46,45,
    45,45,45,45,45,45,45,45,44,44,44,44,44,44,44,43,43,43,43,43,43,
    42,42,42,42,42,41,41,41,40,40,40,40,39,39,39,39,39,39,38,38,38,
    38,37,37,37,36,36,36,36,36,36,35,35,35,35,35,34,34,34,34,34,33,
    33,33,33,33,33,32,32,32,32,31,31,31,31,31,31,30,30,30,30,30,30,
    30,30,29,29,29,29,29,29,29,29,29,28,28,28,28,28,28,28,27,27,27,
    26,26,26,26,26,25,25,25,25,24,24,24,24,24,23,23,23,23,23,23,23,
    22,22,22,22,22,22,22,22,22,22,22,21,21,21,21,20,20,20,20,19,19,
    18,18,18,18,18,18,17,17,17,17,17,17,16,16,16,16,16,15,15,15,15,
    15,15,14,14,14,14,14,14,13,13,13,13,13,13,13,13,13,12,12,12,11,
    10,10,9,9,9,9,9,9,9,8,7,7,7,6,6,6,6,5,5,5,5,5,5,4,4,4,3,3,3,3,
    3,3,3,3,3,2,2,2,1,1,1,1,1
  };
  const int n4c3w1_c[] = {
    150, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,99,99,99,99,99,99,99,98,98,98,98,98,97,97,96,96,96,96,
    96,96,96,95,95,95,94,94,94,94,94,93,93,93,93,93,93,92,92,92,92,
    92,92,91,91,91,91,90,90,90,90,89,89,89,89,89,89,89,88,88,88,88,
    88,88,88,87,87,87,87,86,86,86,86,86,86,85,84,84,83,83,83,83,83,
    82,82,81,81,81,80,80,79,79,78,78,78,78,78,78,78,77,77,77,77,77,
    77,77,76,76,76,75,75,75,75,75,75,75,75,74,74,74,74,74,74,74,73,
    73,73,73,72,72,72,72,71,71,71,71,71,71,71,70,70,70,70,70,69,69,
    69,69,69,68,68,68,68,68,68,67,67,66,66,66,66,66,66,66,66,65,65,
    65,65,64,64,64,64,64,63,63,63,63,63,63,62,62,62,62,62,62,62,61,
    61,61,61,60,60,60,59,59,59,59,59,59,59,58,58,58,58,58,58,57,57,
    57,57,57,57,56,56,56,56,56,56,56,55,55,55,55,55,54,53,53,53,53,
    53,53,53,53,53,53,52,52,52,52,51,51,51,51,51,50,50,50,50,49,49,
    49,49,49,48,48,48,48,48,48,48,48,47,47,47,47,46,46,46,45,45,45,
    45,45,45,45,45,45,44,44,44,44,44,44,44,43,43,43,42,42,42,42,42,
    42,42,41,40,40,40,39,39,39,39,39,38,38,38,38,38,37,37,37,37,37,
    37,37,37,37,37,36,36,36,36,36,36,36,36,36,35,35,34,34,34,34,34,
    33,33,33,33,33,32,32,32,32,32,32,32,32,31,31,31,31,31,30,30,30,
    30,29,29,29,29,29,28,27,27,27,27,27,27,27,26,25,25,25,25,25,25,
    25,24,24,24,24,24,24,23,23,23,22,22,22,22,22,22,21,21,21,21,21,
    20,20,19,19,19,19,19,19,19,19,19,19,19,19,19,18,18,18,17,17,17,
    16,16,15,15,15,15,15,15,15,15,14,14,14,14,14,14,14,13,13,13,13,
    13,13,13,12,12,12,12,12,12,12,11,11,11,11,11,10,10,10,10,9,9,
    8,8,8,8,7,7,7,6,6,6,6,5,5,5,5,5,5,5,4,4,4,4,4,3,3,3,3,3,3,3,2,
    2,2,2,2,2,1,1,1
  };
  const int n4c3w1_d[] = {
    150, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,99,99,99,99,99,99,99,98,98,98,98,97,97,97,96,96,96,
    96,96,96,95,95,94,94,93,93,93,93,93,93,93,92,92,92,92,92,91,91,
    91,91,91,91,90,90,90,90,90,90,89,88,87,87,86,86,86,86,86,85,85,
    85,85,85,85,84,84,84,84,83,83,83,83,83,82,82,82,81,81,80,80,80,
    79,79,79,78,78,78,77,77,77,77,77,77,77,76,76,76,76,75,75,74,74,
    73,73,73,73,73,72,72,72,72,72,72,72,72,71,71,71,71,70,70,70,70,
    70,69,69,69,69,69,69,69,69,68,68,68,67,67,67,67,67,66,66,66,66,
    66,66,66,65,65,65,65,65,65,65,65,65,64,64,64,64,63,63,63,63,63,
    62,62,62,61,61,60,60,60,60,60,59,59,58,58,58,58,58,57,57,57,57,
    57,57,57,57,56,56,56,56,56,55,55,55,55,55,55,55,55,54,54,54,54,
    54,54,54,54,54,53,53,53,52,52,52,52,51,51,50,50,50,50,49,49,49,
    49,48,48,48,48,48,48,48,48,48,47,47,47,46,46,46,46,46,45,45,45,
    45,45,45,45,45,44,44,44,44,44,44,43,43,43,43,43,43,42,42,42,42,
    41,41,41,41,41,40,40,40,40,39,39,39,39,39,38,38,38,38,38,38,38,
    37,37,37,37,37,37,36,36,36,36,35,35,35,35,35,35,34,34,34,34,34,
    34,33,33,32,32,32,32,31,31,31,30,30,30,30,30,30,30,30,30,30,30,
    30,30,29,29,29,29,29,29,28,28,28,28,28,28,28,27,27,27,27,27,27,
    27,26,26,26,26,26,26,26,25,25,25,25,25,25,25,25,25,24,24,24,24,
    24,23,23,23,23,23,23,22,22,21,21,21,21,21,21,20,20,20,20,20,20,
    20,19,19,19,19,18,18,17,17,17,17,17,17,17,17,16,16,16,15,15,15,
    15,14,14,14,14,14,14,14,13,13,13,13,12,12,12,12,12,12,11,11,11,
    11,11,11,11,11,11,11,10,10,10,10,10,10,9,9,9,9,9,9,8,8,8,8,8,
    8,7,7,7,7,7,6,6,6,6,5,5,5,5,5,5,4,4,4,4,4,4,4,3,3,2,2,2,2,2,2,
    2,2,2,1,1
  };
  const int n4c3w1_e[] = {
    150, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,99,99,98,98,98,98,97,97,97,97,96,96,96,96,96,96,96,
    96,95,95,95,95,95,95,95,94,94,93,93,93,93,92,92,92,91,91,91,90,
    90,90,90,89,89,89,89,88,88,88,88,88,88,87,87,87,87,87,86,86,86,
    86,86,86,86,86,86,86,86,86,85,85,85,85,85,85,84,84,84,84,84,84,
    84,84,84,83,83,83,83,83,82,82,82,82,82,82,81,81,81,81,80,80,80,
    80,80,80,79,79,79,79,79,79,79,78,78,77,77,77,77,77,77,76,76,76,
    75,75,75,75,75,75,74,74,74,74,73,73,73,73,73,73,73,73,73,73,72,
    72,72,72,71,71,71,70,70,69,69,69,69,69,69,68,68,68,68,68,68,68,
    67,67,67,67,66,66,66,66,66,65,65,65,65,65,65,65,65,65,64,64,64,
    64,63,63,63,63,62,62,62,62,62,62,61,60,60,60,60,60,60,59,59,59,
    59,59,58,58,58,57,57,57,57,57,56,56,56,56,55,55,55,55,55,55,55,
    54,54,54,54,54,53,53,52,52,51,51,51,51,50,50,50,50,50,50,50,49,
    49,49,49,48,48,48,48,48,48,47,47,46,46,46,46,46,45,45,45,44,44,
    44,44,43,43,43,43,42,42,42,42,42,41,41,41,41,41,41,41,41,41,40,
    40,40,40,40,39,39,39,39,38,38,38,37,37,37,37,37,37,36,36,36,35,
    35,35,34,34,34,34,34,34,34,33,33,33,33,33,33,33,33,32,32,32,32,
    31,31,31,31,31,31,30,30,30,30,30,30,30,29,29,29,29,28,28,28,27,
    27,27,27,26,26,26,26,26,26,26,25,25,25,24,24,23,23,23,23,23,23,
    23,23,22,22,22,21,21,21,21,21,21,20,20,20,19,19,19,19,19,19,19,
    19,19,18,18,18,18,17,17,17,16,16,16,16,16,16,15,15,15,15,15,14,
    14,14,14,14,13,13,13,13,13,12,12,12,12,12,12,12,12,12,11,11,11,
    11,11,11,11,11,10,10,10,9,9,9,9,9,8,8,8,8,8,8,7,7,7,7,7,7,6,6,
    6,6,6,6,5,5,5,5,5,4,4,4,4,4,4,4,4,4,3,3,3,3,2,2,2,2,2,1,1,1,1,
    1,1
  };
  const int n4c3w1_f[] = {
    150, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,99,99,99,98,98,97,97,97,97,96,96,96,96,95,
    95,95,95,94,94,94,94,94,94,94,93,93,92,92,92,92,92,91,91,91,91,
    91,91,90,90,90,90,89,89,89,89,89,89,88,88,88,88,88,88,87,87,87,
    87,86,86,86,86,86,86,86,85,85,85,85,84,84,84,84,84,83,83,83,83,
    83,83,83,83,83,83,83,83,82,82,82,82,81,81,81,80,80,80,80,79,79,
    79,78,78,78,78,78,78,77,77,77,77,77,77,76,76,76,76,76,75,75,75,
    75,74,74,74,73,73,73,73,73,73,73,73,73,72,72,71,71,71,71,71,71,
    71,70,70,70,70,69,69,69,68,68,68,67,67,67,67,67,67,67,67,67,66,
    66,66,66,66,66,66,66,65,64,64,64,64,64,64,63,63,62,62,61,61,61,
    60,60,59,59,59,59,59,59,58,58,58,58,57,57,57,57,56,56,56,56,56,
    56,55,55,55,54,54,54,54,54,54,54,54,53,53,53,52,52,52,52,51,51,
    51,51,51,51,50,50,50,50,50,50,49,49,49,49,48,48,48,48,47,47,47,
    47,47,46,46,46,46,46,46,45,45,45,45,45,45,44,44,44,44,43,43,43,
    43,42,42,42,42,42,42,42,42,42,41,41,40,40,40,40,40,39,39,39,39,
    38,38,38,37,37,37,37,37,37,36,36,36,36,36,36,36,36,36,36,35,35,
    35,35,34,34,34,34,34,34,33,33,33,33,33,32,32,32,32,32,31,31,31,
    31,30,30,30,30,30,30,30,30,30,30,29,29,29,29,29,28,28,28,28,28,
    27,27,27,26,26,26,26,26,25,25,25,25,25,25,25,25,25,24,24,24,24,
    24,24,24,24,24,24,24,24,24,23,23,23,23,23,22,22,22,22,22,22,22,
    22,21,21,21,21,21,21,20,20,20,20,20,20,20,20,19,19,19,19,18,18,
    18,18,18,18,18,18,17,17,17,17,17,16,16,15,14,14,14,14,14,14,14,
    13,13,13,13,12,11,11,9,9,9,9,9,9,9,9,8,8,8,8,8,7,7,7,7,7,6,6,
    6,5,5,5,5,5,5,5,5,5,4,4,4,4,3,3,3,3,3,3,3,2,2,2,2,2,2,2,2,2,1,
    1,1,1
  };
  const int n4c3w1_g[] = {
    150, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,99,99,99,98,98,98,98,98,97,97,97,97,96,96,
    96,96,96,96,95,95,95,95,95,95,94,94,94,94,94,94,94,94,93,93,93,
    93,93,93,93,93,93,92,92,92,92,92,92,91,91,91,91,91,91,90,90,90,
    89,89,89,88,87,87,87,87,87,86,86,86,86,86,85,85,85,84,84,84,84,
    83,83,83,83,83,83,83,82,82,82,82,81,81,81,81,81,81,81,80,80,80,
    80,80,80,80,80,79,79,79,79,79,79,78,78,78,78,78,78,77,77,77,77,
    77,76,76,76,75,75,75,75,75,75,75,74,74,73,73,73,72,72,72,72,72,
    71,71,71,71,71,71,71,71,70,70,70,69,69,69,69,68,68,68,68,68,68,
    67,67,67,67,67,66,66,65,65,65,65,65,65,65,64,64,64,64,64,64,63,
    63,63,63,63,63,62,62,61,61,61,61,61,61,61,60,60,60,60,59,59,59,
    58,58,57,57,57,57,56,56,56,56,56,56,55,55,55,55,55,55,55,55,55,
    55,54,54,53,53,53,53,53,53,53,53,53,53,52,52,52,52,52,52,51,51,
    50,50,49,49,49,49,49,49,49,49,48,48,48,48,48,48,48,48,47,47,47,
    47,47,47,47,47,47,47,47,46,46,46,46,45,45,45,45,45,44,44,44,44,
    44,44,43,43,43,42,42,42,42,41,41,41,41,41,41,40,39,39,39,39,38,
    38,38,38,38,38,38,38,38,37,37,37,37,37,37,36,36,36,36,36,36,35,
    34,34,33,33,33,33,33,33,32,32,32,32,31,30,30,29,29,29,29,29,28,
    28,28,28,27,27,27,27,27,26,26,26,26,26,26,26,26,26,26,25,25,25,
    25,25,25,25,24,24,24,24,24,23,23,23,23,23,23,23,22,22,22,21,21,
    21,21,21,21,21,21,21,20,20,20,20,20,20,19,19,19,18,18,18,18,18,
    18,17,17,17,16,16,16,16,15,15,15,15,14,14,14,14,13,13,13,13,12,
    12,12,12,12,11,11,11,11,10,10,9,9,9,9,9,8,8,8,8,8,7,7,7,7,7,7,
    6,6,6,6,5,5,5,5,5,5,5,4,4,4,4,4,4,4,4,4,3,3,3,3,2,2,2,2,1,1,1,
    1,1,1,1
  };
  const int n4c3w1_h[] = {
    150, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,99,98,98,97,97,97,97,97,97,97,97,97,97,96,
    96,96,96,95,95,95,95,95,94,94,94,94,94,94,94,94,93,93,93,93,92,
    92,92,92,92,92,91,91,91,91,90,90,90,90,90,90,90,90,89,89,89,89,
    89,88,88,88,88,88,88,87,87,87,87,87,87,87,87,87,87,87,86,86,86,
    86,86,86,86,86,86,85,85,85,85,85,85,84,84,84,83,83,83,82,82,82,
    82,82,81,81,81,81,81,81,80,80,79,79,79,79,79,79,79,79,79,78,78,
    78,78,78,77,77,77,76,76,76,76,75,75,75,75,74,74,74,74,74,73,73,
    73,73,73,72,72,72,71,70,70,70,70,70,70,70,69,69,69,69,69,68,68,
    68,68,68,68,68,68,67,67,67,67,67,67,67,66,66,66,66,66,66,66,65,
    65,65,65,65,65,64,64,63,63,63,63,63,63,62,62,62,62,62,61,61,61,
    61,60,60,60,60,60,60,59,59,59,59,59,59,58,58,58,58,57,57,57,57,
    56,56,55,55,55,55,54,54,54,54,54,54,53,53,53,53,53,53,53,53,52,
    52,52,52,51,51,50,50,50,50,50,49,49,49,49,48,47,47,47,47,47,47,
    47,47,47,47,46,46,46,46,46,45,45,44,44,43,43,42,42,42,41,41,41,
    41,41,40,40,40,40,40,40,40,40,40,40,40,39,39,39,39,39,38,38,38,
    38,37,37,37,37,36,36,36,35,35,35,35,35,35,34,34,34,33,33,33,33,
    33,33,32,32,32,32,32,32,32,32,32,32,32,31,31,30,30,30,30,30,29,
    29,28,28,28,27,27,27,27,27,26,26,26,26,26,26,25,25,25,25,25,24,
    24,24,23,23,23,23,23,23,22,22,22,22,22,22,21,21,21,21,21,21,20,
    20,20,20,19,19,19,19,18,18,18,18,18,17,17,16,16,16,16,16,16,16,
    16,15,15,15,15,15,15,15,15,15,14,14,14,14,14,13,13,13,13,12,12,
    12,12,12,12,11,11,11,11,11,11,10,10,9,9,9,9,9,8,8,8,8,8,8,7,7,
    7,7,7,7,7,7,7,6,6,6,6,6,6,5,5,5,5,5,4,4,4,3,3,3,3,3,3,3,2,2,2,
    2,2,1,1,1
  };
  const int n4c3w1_i[] = {
    150, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,99,99,99,99,99,99,99,99,98,97,97,96,96,96,96,
    96,96,95,95,94,94,94,94,93,93,93,92,92,92,92,92,91,91,90,90,90,
    90,90,90,89,89,89,89,89,88,88,88,88,88,87,87,87,87,86,86,86,86,
    86,86,85,85,85,85,85,85,85,84,84,84,83,83,83,82,82,82,82,81,81,
    81,81,81,81,81,81,80,80,80,80,80,80,80,79,79,79,79,79,79,78,78,
    78,78,78,78,78,78,77,77,77,77,77,77,77,77,77,76,76,75,75,75,75,
    74,74,74,74,74,73,73,73,73,73,72,72,72,72,72,72,72,71,71,71,71,
    71,71,70,70,70,70,70,70,69,69,69,68,68,68,68,67,67,67,67,67,67,
    67,66,66,66,66,66,66,66,66,66,65,65,65,65,65,65,64,64,64,64,64,
    64,63,63,63,63,63,62,62,62,62,62,61,61,61,61,61,60,60,60,60,60,
    60,60,59,59,58,58,58,58,58,58,57,57,57,56,56,56,56,56,55,55,55,
    55,55,55,54,54,54,54,53,53,53,53,53,53,52,52,52,52,51,51,50,50,
    50,50,49,48,48,48,48,48,48,47,47,47,47,47,47,47,47,46,46,46,46,
    46,46,46,45,45,44,44,44,44,43,43,43,42,42,42,41,41,41,41,41,41,
    41,40,40,40,40,40,40,39,39,38,38,38,38,38,38,37,37,37,37,37,37,
    37,37,37,37,36,36,35,35,35,35,35,35,35,34,34,33,33,33,33,33,32,
    32,32,32,31,31,31,31,31,31,31,30,30,30,30,30,30,30,29,29,29,29,
    29,29,29,29,29,28,28,27,27,27,27,27,27,26,26,26,26,26,26,26,26,
    26,25,25,25,25,25,25,24,24,24,24,24,24,24,24,23,23,23,23,22,22,
    22,22,21,21,21,21,21,21,21,20,20,20,20,20,20,20,20,19,19,19,19,
    19,18,18,18,18,18,17,17,16,16,16,16,16,16,16,15,15,15,15,14,14,
    14,14,14,13,13,13,13,12,12,12,12,12,12,12,12,11,11,11,11,11,11,
    10,10,10,10,10,9,8,8,8,8,8,8,8,7,6,6,6,5,5,5,5,5,5,4,4,4,4,4,
    4,3,3,3,2,2,2,1,1,1,1,1
  };
  const int n4c3w1_j[] = {
    150, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,100,100,99,99,99,99,98,98,98,98,98,98,97,
    97,96,96,95,95,95,95,95,95,95,95,94,93,93,93,92,92,92,92,92,92,
    92,91,91,91,91,91,91,90,89,89,89,89,88,88,88,88,87,87,87,87,87,
    87,87,87,87,86,86,86,86,86,86,85,85,85,85,85,84,84,84,84,84,83,
    83,82,82,82,82,82,82,81,81,81,81,81,81,81,81,80,80,80,80,80,80,
    80,79,79,79,79,79,78,78,78,78,78,77,77,77,77,77,77,76,76,76,76,
    76,76,75,75,75,75,75,75,74,73,73,73,73,73,73,72,72,72,72,72,72,
    71,71,71,71,71,71,71,70,70,69,69,69,68,68,68,68,68,68,68,68,67,
    67,67,67,67,66,66,66,66,66,66,66,66,65,65,65,65,65,65,65,64,64,
    63,63,62,62,62,62,62,62,61,61,61,61,61,61,60,60,59,59,59,59,59,
    59,59,59,58,58,58,58,58,58,58,58,58,57,57,56,56,56,56,56,55,55,
    55,55,55,55,55,55,54,54,53,53,53,53,53,53,52,52,52,52,52,52,52,
    51,51,51,51,51,51,50,50,50,50,50,49,49,49,49,49,49,48,48,48,48,
    48,47,47,47,47,47,47,47,46,46,46,46,46,46,45,45,45,45,45,44,44,
    44,44,44,44,43,43,42,42,42,42,42,42,42,41,41,41,41,40,40,40,40,
    40,40,40,40,39,39,39,39,39,39,38,38,38,38,37,37,37,37,37,36,36,
    36,36,36,36,35,35,35,35,35,35,34,34,34,34,34,33,33,33,33,33,32,
    32,32,31,30,30,30,30,30,30,29,29,29,28,28,28,28,27,27,26,26,25,
    25,25,25,24,24,24,24,23,23,23,23,23,23,23,22,22,22,22,22,22,21,
    21,21,20,20,20,20,20,19,19,19,19,19,18,18,18,17,17,17,17,17,17,
    17,17,16,16,16,16,16,16,15,15,14,14,14,14,14,14,14,13,13,13,13,
    13,12,12,12,11,11,10,10,10,10,10,10,9,9,9,9,9,9,8,8,8,8,8,8,8,
    8,7,7,7,7,7,7,7,6,6,5,5,5,4,4,4,4,4,3,3,3,3,3,2,2,2,2,2,2,2,2,
    2,2,2,1,1,1
  };
  const int n4c3w1_k[] = {
    150, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,100,100,100,99,99,99,99,99,99,99,99,99,98,
    98,97,97,97,97,97,97,96,96,96,96,96,96,96,95,95,95,94,94,94,94,
    94,94,93,93,92,92,91,91,91,91,91,90,90,90,90,89,89,89,89,89,89,
    88,88,88,87,87,86,86,85,85,85,85,84,84,84,84,84,83,83,83,83,83,
    82,82,82,82,82,82,81,81,81,81,81,81,80,80,80,80,80,79,79,79,79,
    79,78,78,78,78,78,78,78,77,77,77,77,77,77,76,76,76,76,76,76,76,
    75,75,75,75,74,74,74,74,74,74,73,73,73,72,72,72,72,72,72,72,71,
    71,70,70,70,70,70,70,69,69,69,69,68,68,68,68,68,67,67,67,67,67,
    67,67,66,66,66,66,66,66,66,65,65,65,64,64,64,64,63,63,63,63,63,
    63,63,63,62,62,62,62,60,59,59,59,59,59,59,59,59,58,58,58,58,56,
    56,56,56,55,55,55,54,53,53,53,53,52,52,52,52,52,52,51,51,51,51,
    51,51,51,50,50,50,49,49,49,48,48,48,48,48,48,48,47,47,47,47,47,
    47,47,46,46,46,46,46,45,45,45,45,45,45,45,44,44,44,44,43,43,43,
    43,43,42,42,42,42,42,42,41,41,41,41,41,41,41,41,40,40,40,40,40,
    40,40,40,39,39,39,39,39,38,38,37,37,37,37,36,36,36,36,36,36,36,
    35,35,35,35,35,35,35,35,35,34,34,34,34,33,33,33,33,33,33,32,32,
    32,32,31,31,31,31,31,30,30,30,29,29,29,29,29,29,29,28,28,28,28,
    28,27,27,27,26,26,26,26,26,26,25,25,25,25,25,24,24,24,24,23,23,
    23,23,23,23,23,23,22,22,22,22,21,21,21,21,21,20,20,20,20,20,20,
    20,19,19,19,19,19,18,18,18,18,18,18,18,18,18,18,18,17,17,17,17,
    17,17,16,16,15,15,14,14,14,14,14,14,14,14,13,13,13,13,13,13,12,
    12,12,12,11,11,11,10,10,10,10,9,9,9,9,9,9,9,8,8,8,8,7,7,7,7,7,
    7,7,7,6,6,6,6,6,5,5,5,4,4,4,4,3,3,3,3,3,3,3,3,2,2,2,2,2,2,2,2,
    1,1,1,1,1
  };
  const int n4c3w1_l[] = {
    150, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,99,99,99,98,98,98,98,98,98,97,97,97,97,97,
    97,97,97,97,96,96,95,95,94,94,94,94,93,93,93,93,93,93,92,92,92,
    92,92,92,91,91,91,91,91,90,89,89,88,88,88,88,88,87,87,87,87,86,
    85,85,85,85,84,84,84,83,83,83,83,82,81,81,81,81,81,81,81,80,80,
    79,79,79,79,79,79,79,79,78,78,78,78,78,78,77,77,77,77,77,77,77,
    76,76,76,76,76,75,75,75,74,74,74,74,74,74,74,74,73,73,73,73,72,
    72,72,72,72,72,71,71,71,71,70,70,70,70,70,69,69,69,69,68,68,68,
    68,67,67,67,67,67,66,66,66,66,66,66,65,65,65,65,65,64,64,64,64,
    64,64,64,63,63,63,63,63,63,63,62,62,61,61,61,60,60,60,60,59,59,
    59,59,59,58,58,58,58,57,57,57,57,57,57,57,57,56,56,56,56,56,55,
    55,55,54,54,54,53,53,53,52,52,52,52,52,52,52,51,51,51,50,50,50,
    50,50,50,50,49,49,49,49,49,48,48,48,48,48,48,48,48,48,47,47,47,
    47,47,47,46,46,46,46,46,46,45,45,45,45,45,45,45,44,44,44,44,44,
    44,44,43,43,43,43,43,43,43,43,43,43,42,42,42,42,42,42,42,41,41,
    41,41,40,40,40,40,39,39,39,39,39,38,38,38,38,38,38,37,37,37,36,
    36,36,36,36,35,35,35,35,35,35,35,35,34,34,34,33,32,32,32,32,32,
    32,32,32,32,32,32,32,31,31,31,31,31,31,30,30,30,30,30,29,29,29,
    29,28,28,28,28,28,28,28,28,27,27,27,27,26,26,26,26,26,26,26,26,
    26,26,25,25,25,25,25,25,25,24,24,24,23,23,23,23,23,23,23,23,23,
    22,22,22,22,22,21,21,21,21,21,21,21,21,20,20,20,19,19,18,18,18,
    17,17,17,17,16,16,16,15,15,14,14,14,14,14,14,13,13,13,13,13,13,
    13,12,12,11,11,11,11,11,11,11,11,11,10,10,10,10,10,10,9,9,9,8,
    8,8,8,8,8,7,7,7,7,7,7,7,6,6,6,5,5,5,5,4,4,4,4,4,4,4,4,4,3,3,3,
    3,2,2,2,2,1,1,1
  };
  const int n4c3w1_m[] = {
    150, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,99,99,99,98,98,98,98,98,98,98,97,97,97,96,96,
    96,96,96,95,95,95,95,95,94,94,93,93,93,93,92,92,92,92,91,90,90,
    89,89,89,89,89,89,88,88,87,87,87,87,87,87,87,87,87,86,86,86,85,
    85,85,85,85,85,84,84,84,84,84,84,84,84,84,83,83,83,82,82,82,82,
    82,81,81,81,81,81,81,81,80,80,80,80,80,80,79,79,79,78,78,77,77,
    77,76,76,76,76,76,76,76,75,75,75,75,75,75,75,75,74,74,74,74,74,
    74,74,74,74,73,73,73,73,73,73,73,73,73,72,72,72,72,71,71,71,71,
    71,71,71,71,71,71,70,70,70,70,70,70,70,70,69,69,69,69,69,69,69,
    68,68,68,68,68,68,67,67,67,67,67,67,67,66,66,66,66,66,66,65,65,
    65,64,64,64,64,64,63,62,62,62,62,61,61,60,60,60,60,60,60,59,59,
    59,59,59,58,58,58,58,58,57,57,56,56,56,55,55,55,55,54,54,54,54,
    54,54,54,54,54,54,53,53,53,53,53,52,51,51,51,51,51,50,50,50,50,
    50,50,49,49,49,49,49,49,49,48,48,48,47,47,47,47,47,46,46,45,45,
    45,45,45,45,45,45,44,44,44,44,43,43,43,43,43,42,42,42,42,42,42,
    42,41,41,41,41,41,41,41,41,41,40,40,40,40,39,39,39,38,38,38,38,
    37,37,37,37,37,37,37,36,36,36,36,35,35,35,35,35,35,35,34,34,34,
    34,34,33,33,33,33,32,32,32,32,32,32,32,32,32,31,31,31,31,31,31,
    29,29,29,29,29,29,29,28,28,27,27,27,27,27,27,26,26,26,26,25,25,
    25,25,25,24,24,24,24,24,23,23,23,22,22,22,21,21,21,21,20,20,20,
    20,20,18,18,18,18,18,18,18,17,17,17,17,17,17,17,17,17,16,16,16,
    16,15,15,15,15,15,15,15,15,15,14,14,14,14,14,14,14,13,13,13,13,
    13,13,13,12,12,12,12,12,12,11,11,11,11,11,11,11,10,10,10,10,10,
    10,10,10,10,9,8,8,8,8,8,7,7,7,7,6,6,6,6,5,5,5,4,4,4,4,4,3,3,3,
    3,2,2,2,2,2,2,1,1,1,1
  };
  const int n4c3w1_n[] = {
    150, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,99,99,99,99,98,98,98,98,98,98,98,98,98,97,97,
    97,97,97,97,97,97,96,96,96,96,96,95,95,95,95,95,95,95,94,94,94,
    94,94,94,94,94,93,93,93,93,93,93,93,93,93,92,92,92,92,92,92,92,
    91,91,91,90,90,90,89,89,89,89,89,89,89,89,89,89,89,88,88,88,88,
    87,86,86,86,86,85,85,84,84,84,84,84,84,83,83,83,83,83,83,83,82,
    82,81,81,81,81,81,81,81,81,80,80,80,80,80,80,80,79,79,79,79,79,
    79,79,78,78,78,78,77,77,77,77,77,77,77,77,76,76,76,76,75,75,75,
    75,74,74,74,74,73,73,73,73,72,72,72,71,71,71,71,71,71,71,71,71,
    71,71,70,70,70,69,69,69,69,69,69,69,68,68,67,67,67,67,67,67,67,
    67,67,66,66,66,65,65,65,65,64,64,64,64,64,64,64,64,64,63,63,63,
    63,63,63,63,62,62,61,61,61,60,60,60,60,59,59,59,59,59,59,59,58,
    58,58,58,58,58,58,58,58,57,57,57,57,57,57,57,56,56,56,55,55,54,
    54,54,54,54,54,53,53,53,52,52,52,52,52,51,51,51,51,51,51,51,51,
    51,51,50,50,50,50,50,49,49,49,48,48,48,47,46,46,46,46,45,45,45,
    45,44,44,44,44,44,43,43,43,43,43,43,42,41,41,41,41,41,41,41,40,
    40,40,40,39,39,39,39,38,38,38,38,38,38,37,37,37,37,37,37,35,35,
    35,34,34,34,33,33,33,33,32,32,32,32,32,32,32,32,31,31,31,31,31,
    30,30,30,30,30,30,30,30,29,29,29,29,29,29,28,28,28,27,27,27,26,
    26,26,26,26,26,26,26,26,25,25,25,25,25,25,24,24,24,23,23,23,23,
    23,23,22,22,22,21,21,21,20,20,19,19,19,19,19,19,18,18,18,18,18,
    18,18,17,17,17,17,17,16,15,15,15,15,14,14,14,14,14,14,13,13,13,
    13,13,12,12,11,11,11,10,10,10,10,10,9,9,9,9,9,8,8,8,8,8,7,7,7,
    7,7,7,7,6,6,6,5,5,5,5,5,5,4,4,4,4,3,3,3,3,3,3,3,3,3,2,2,2,2,2,
    2,2,1,1,1
  };
  const int n4c3w1_o[] = {
    150, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,100,100,100,100,100,99,99,99,99,99,98,98,
    98,97,97,97,97,97,97,96,96,95,95,95,95,95,95,95,95,94,94,94,94,
    94,94,93,92,92,92,92,91,91,91,91,91,91,91,91,91,90,90,90,90,90,
    90,90,89,89,89,89,89,89,89,88,88,88,88,87,87,87,87,87,87,86,86,
    86,86,85,84,84,84,84,83,83,83,82,82,82,82,82,82,82,82,82,82,82,
    81,81,81,81,81,81,81,81,80,80,80,80,80,80,79,79,78,78,77,77,77,
    77,77,76,76,76,75,75,75,75,75,74,74,74,74,74,73,73,72,72,72,72,
    71,71,70,70,70,70,70,70,69,69,69,69,68,68,68,68,67,67,67,67,66,
    66,66,66,66,66,65,65,65,65,64,64,64,64,64,64,64,64,64,64,63,63,
    63,63,63,62,62,62,62,62,62,61,61,61,60,60,60,60,60,60,59,59,59,
    58,58,58,58,58,57,57,57,57,57,57,56,56,56,56,56,56,56,56,56,55,
    55,55,55,55,54,54,54,54,54,53,53,53,53,53,53,52,52,52,52,52,52,
    52,52,51,51,51,49,49,49,49,49,49,48,48,48,48,48,47,47,47,47,47,
    46,46,46,46,46,46,46,46,45,45,45,45,44,44,44,44,43,43,42,42,42,
    42,42,42,42,42,42,41,41,41,41,41,41,41,40,40,40,39,39,38,38,38,
    38,38,38,38,38,37,37,36,36,36,35,35,35,34,34,34,33,33,33,33,33,
    32,32,32,32,32,32,31,31,31,31,31,31,31,31,31,30,30,30,30,30,29,
    29,28,28,28,28,27,27,27,27,27,27,26,26,26,26,25,25,25,25,25,25,
    25,25,25,24,24,24,24,24,23,23,23,23,23,23,23,23,22,22,22,22,22,
    22,22,21,21,21,21,20,20,20,20,20,19,19,18,18,18,18,18,18,17,17,
    17,17,17,17,16,16,16,16,16,15,15,15,14,14,14,13,13,13,13,13,13,
    13,12,12,12,12,12,11,10,10,10,10,10,10,10,9,9,9,9,9,9,8,8,8,8,
    8,7,7,7,7,7,7,7,6,6,6,5,5,5,5,5,5,5,4,4,4,4,4,4,3,3,3,2,2,2,2,
    2,2,2,1,1,1,1,1
  };
  const int n4c3w1_p[] = {
    150, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,99,99,99,98,98,98,98,97,97,97,97,97,97,97,97,96,96,
    96,96,96,96,95,95,95,95,94,94,94,94,94,94,94,94,94,93,93,93,93,
    93,93,93,93,92,91,91,91,91,90,90,89,89,89,89,89,89,88,88,87,86,
    86,86,86,86,85,85,85,85,85,85,85,84,84,84,84,84,84,84,83,83,82,
    82,82,82,82,81,80,80,79,79,79,79,79,79,79,79,79,78,78,78,78,78,
    78,77,77,77,77,77,76,76,76,76,76,76,75,75,75,75,74,74,74,74,74,
    74,74,74,74,74,74,74,74,74,73,73,73,73,72,72,72,72,72,72,72,72,
    72,72,72,71,71,71,71,71,71,70,70,70,70,70,70,69,69,69,69,69,69,
    69,68,68,68,68,68,68,67,67,67,66,66,66,66,65,65,65,65,65,65,65,
    64,64,64,64,63,63,63,63,63,63,62,62,62,61,61,61,61,61,60,60,59,
    59,59,59,59,59,59,58,58,58,58,58,57,57,56,56,56,56,54,54,54,54,
    54,54,53,53,53,53,53,53,52,52,52,52,52,52,52,51,51,51,51,51,50,
    50,50,50,49,49,48,48,48,48,48,48,48,47,47,47,46,46,46,46,46,46,
    46,46,45,45,45,45,45,44,44,44,44,44,44,44,44,44,44,43,43,43,43,
    43,43,42,42,41,41,41,41,41,41,41,40,40,40,40,39,39,38,38,38,38,
    37,37,37,37,36,36,36,36,36,35,35,35,35,35,35,34,34,34,34,34,33,
    33,33,33,32,32,32,32,32,31,31,31,30,29,29,29,29,29,29,28,28,28,
    28,28,27,27,27,27,26,26,26,26,26,26,26,26,25,25,25,25,24,24,24,
    24,24,23,23,23,23,23,23,22,22,22,22,22,22,21,21,21,20,20,20,20,
    20,19,19,18,18,18,17,17,17,17,17,16,16,16,16,16,16,16,16,16,15,
    14,14,14,14,14,14,14,13,13,13,13,13,12,12,12,12,12,12,12,11,11,
    11,11,11,11,11,10,10,10,10,10,10,10,10,9,9,9,9,9,8,8,8,8,8,7,
    7,6,6,6,6,5,5,5,5,5,4,4,4,4,4,4,4,4,3,3,3,3,2,2,2,2,2,2,2,2,1,
    1,1,1,1,1
  };
  const int n4c3w1_q[] = {
    150, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,99,98,98,98,98,97,97,97,97,97,96,96,96,96,
    96,96,96,96,96,95,95,95,95,95,95,94,94,94,94,94,94,93,93,93,93,
    93,92,92,92,92,92,92,92,91,91,90,90,90,90,90,89,89,89,89,89,89,
    89,88,87,87,87,87,87,86,86,86,86,86,86,86,85,85,85,85,85,85,84,
    84,84,84,84,84,84,83,83,83,83,83,83,83,83,83,82,82,81,81,81,81,
    81,80,80,80,80,79,79,79,79,78,78,78,78,78,78,77,77,77,77,77,76,
    76,76,76,76,76,76,76,76,75,75,74,74,74,74,73,73,73,72,72,72,72,
    72,72,71,71,71,71,71,71,71,70,70,70,70,70,70,69,69,69,69,69,68,
    68,68,68,68,67,67,67,67,67,67,67,67,67,66,66,66,66,66,66,66,66,
    66,66,66,66,65,65,65,65,65,64,64,64,64,64,64,64,64,64,64,63,63,
    62,62,62,62,62,62,61,61,61,61,60,60,60,60,60,59,59,59,58,58,58,
    58,58,58,58,58,57,57,57,56,56,56,56,56,56,56,56,56,55,55,55,54,
    54,54,54,53,53,53,53,53,53,53,53,52,52,51,51,51,51,51,51,51,50,
    50,50,50,49,49,49,49,48,48,48,48,48,47,47,46,46,46,46,45,45,44,
    44,44,44,43,43,43,43,43,42,42,42,42,42,42,42,42,41,41,41,41,41,
    41,41,40,40,40,40,40,40,40,40,40,40,40,39,39,39,39,39,39,39,39,
    39,38,38,38,38,38,37,37,37,37,36,36,36,36,36,36,36,36,36,35,35,
    35,34,34,34,34,33,33,33,32,32,32,31,31,31,31,31,30,30,29,29,29,
    28,28,28,28,28,28,27,27,27,26,26,26,26,26,26,26,26,25,25,25,25,
    25,25,25,24,23,23,23,23,23,22,22,21,21,20,20,20,20,20,20,19,18,
    18,18,18,17,17,17,17,16,16,16,15,15,15,15,15,15,15,15,15,14,14,
    14,14,14,14,13,13,13,13,13,13,12,12,12,12,12,12,12,12,11,11,11,
    10,10,10,10,10,10,10,9,9,9,9,9,8,8,8,8,8,8,7,7,7,7,6,6,5,5,4,
    4,4,3,2,2,2,2,2,2,1,1,1,1
  };
  const int n4c3w1_r[] = {
    150, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,100,100,99,99,99,99,99,98,98,98,98,98,97,
    97,96,96,96,96,95,95,95,95,95,95,95,95,95,95,94,94,94,94,93,93,
    93,93,92,92,92,92,92,92,91,91,91,91,90,90,90,90,90,90,89,89,89,
    89,88,88,88,88,87,87,87,87,87,87,86,86,85,85,84,84,83,83,83,83,
    83,83,82,82,82,82,81,81,81,81,80,80,80,80,80,80,80,80,79,79,79,
    79,79,79,79,79,79,79,78,78,78,78,77,77,77,76,76,76,76,75,75,75,
    75,75,75,74,74,73,73,73,72,72,72,72,72,72,72,72,71,71,71,71,71,
    71,71,71,70,70,70,70,70,70,70,69,69,69,68,68,68,68,67,67,67,67,
    67,67,67,67,67,66,66,66,66,65,65,65,65,65,64,64,64,64,63,63,63,
    63,62,62,62,62,62,61,61,61,61,61,61,61,61,61,61,61,61,60,60,60,
    60,60,59,59,59,58,58,58,58,58,58,58,58,57,57,57,57,56,56,55,55,
    55,55,55,54,54,54,54,53,53,53,53,53,52,52,52,52,52,52,51,51,51,
    51,51,51,51,51,50,49,48,48,48,48,48,48,47,47,47,46,46,46,46,45,
    45,45,45,45,45,44,44,43,43,43,42,42,42,42,42,41,41,41,40,40,40,
    40,40,40,40,40,40,40,39,39,39,39,38,38,38,38,38,38,38,38,37,37,
    37,37,36,36,36,36,36,34,34,34,34,33,33,33,33,33,32,32,32,32,32,
    32,31,31,31,31,31,31,31,31,30,30,30,30,29,29,29,29,29,29,29,29,
    29,28,28,28,28,28,28,28,28,28,28,28,28,27,27,27,27,26,26,26,26,
    26,25,25,25,25,25,24,24,24,24,24,24,23,23,23,23,23,22,22,22,22,
    22,22,21,21,21,20,20,19,19,19,19,19,19,19,19,18,18,18,18,17,17,
    17,17,17,17,16,16,16,16,15,15,14,14,14,14,13,13,13,13,13,13,13,
    12,12,12,12,11,11,11,11,11,11,11,11,11,11,10,10,10,9,9,9,9,9,
    9,8,8,8,8,7,7,7,7,6,6,6,6,6,6,6,6,5,5,5,5,5,4,4,4,4,4,4,4,4,4,
    3,3,3,2,2,2,1,1,1
  };
  const int n4c3w1_s[] = {
    150, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,99,99,99,99,99,98,98,98,98,98,98,97,97,97,97,96,96,96,
    96,96,96,95,95,95,95,95,95,95,95,94,94,94,94,94,94,93,93,93,93,
    93,92,92,92,92,91,91,91,91,91,90,90,90,90,90,90,89,89,89,89,89,
    89,89,88,88,87,87,87,86,86,86,86,86,86,85,85,85,85,85,85,85,84,
    84,84,84,83,83,83,82,82,82,82,81,81,80,80,80,80,80,80,79,79,78,
    78,78,78,78,78,78,78,78,78,77,77,77,77,77,76,76,76,76,76,76,75,
    75,75,74,74,74,74,74,74,73,73,73,73,72,72,71,71,71,71,70,70,70,
    70,70,70,70,69,69,69,68,68,68,68,68,67,67,67,66,66,66,66,66,66,
    66,66,66,66,65,65,65,64,64,64,63,63,63,63,62,62,62,62,62,61,61,
    61,61,61,61,60,60,60,60,59,59,59,59,58,58,58,58,58,58,58,58,57,
    57,57,57,57,57,57,57,56,56,55,55,55,55,55,55,54,54,54,54,54,54,
    54,54,53,53,53,53,52,52,52,52,52,52,52,52,52,51,51,51,51,51,50,
    50,50,50,50,50,50,50,49,49,49,49,49,49,49,49,49,48,48,48,48,48,
    47,47,47,47,47,46,46,46,46,46,45,45,45,45,45,45,45,45,44,44,43,
    43,43,43,42,42,42,41,40,40,39,39,39,39,39,38,38,38,38,37,37,37,
    37,36,36,36,36,36,35,35,35,34,34,34,34,34,33,33,33,33,33,33,33,
    32,32,32,32,32,32,32,32,31,31,31,30,30,30,30,30,29,29,29,29,29,
    29,29,29,29,29,28,28,27,27,27,27,27,26,26,26,26,26,26,25,25,25,
    25,25,25,25,25,24,24,24,24,24,24,24,24,23,23,23,23,23,22,22,22,
    22,22,22,21,21,21,21,21,21,20,20,20,20,20,19,19,19,18,18,18,18,
    18,18,17,17,17,16,15,15,15,15,14,14,14,14,13,13,13,13,13,13,12,
    12,12,12,12,12,11,11,11,11,11,11,11,11,10,10,10,10,10,10,10,10,
    9,9,9,9,9,8,8,8,7,7,7,7,6,5,5,5,5,4,4,4,4,4,4,4,3,3,3,3,3,3,2,
    2,2,2,2,1,1,1,1
  };
  const int n4c3w1_t[] = {
    150, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,99,99,98,98,98,97,97,97,97,96,96,96,96,96,96,95,95,
    95,95,95,95,94,94,94,94,94,94,93,93,93,92,92,92,92,92,91,91,91,
    91,91,90,90,90,90,90,90,89,89,89,89,89,89,88,88,88,88,88,88,88,
    88,88,88,87,87,86,86,86,86,85,85,85,85,85,85,84,84,84,84,83,83,
    82,82,82,82,82,82,81,81,81,81,80,80,80,80,79,79,79,79,79,79,79,
    79,79,79,79,78,78,78,78,78,78,77,77,76,76,76,76,76,76,76,76,76,
    75,75,75,75,75,74,74,74,74,74,74,74,74,74,74,73,73,73,73,73,73,
    73,72,72,72,72,72,72,72,71,71,70,70,70,70,70,70,70,70,70,70,70,
    70,70,69,69,69,69,69,69,68,68,68,68,68,68,67,67,67,67,67,66,66,
    66,66,65,65,65,65,65,65,65,64,63,63,63,62,62,62,62,61,61,61,61,
    60,60,60,60,59,59,59,59,59,59,58,58,58,58,58,57,57,57,57,57,56,
    56,56,56,56,55,55,55,55,55,54,54,54,54,54,53,53,53,53,53,53,53,
    53,52,51,51,51,50,50,50,50,50,50,50,50,50,49,49,49,49,49,49,49,
    48,48,48,48,47,47,47,46,46,46,46,46,45,45,45,44,44,44,44,44,43,
    43,43,43,43,43,43,43,42,42,42,41,41,41,41,41,40,40,40,40,40,40,
    40,40,40,39,39,39,38,38,38,37,37,37,37,37,37,37,37,37,37,37,36,
    36,36,36,36,35,35,35,34,34,34,34,33,33,33,33,32,32,32,32,31,31,
    31,31,31,31,31,31,31,31,30,30,30,29,29,29,29,29,28,28,28,28,28,
    27,27,27,27,27,26,26,26,26,26,26,25,25,25,24,24,24,24,24,24,23,
    23,23,23,23,22,22,22,22,22,22,21,21,21,21,20,20,20,20,19,19,19,
    18,18,18,18,17,17,17,17,17,16,16,16,16,16,16,15,15,15,14,14,14,
    14,14,13,13,13,13,13,13,12,12,12,12,12,12,12,12,11,11,11,11,11,
    11,11,10,9,9,9,9,9,9,9,9,8,8,8,8,7,7,7,7,7,7,6,6,6,6,5,4,4,3,
    3,3,3,3,3,3,3,2,2,2
  };
  const int n4c3w2_a[] = {
    150, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,99,99,99,99,99,99,99,98,98,98,98,97,97,97,97,
    97,97,96,96,96,96,96,96,96,96,96,96,96,95,95,95,95,95,95,95,95,
    95,93,93,93,93,93,93,93,93,92,92,92,92,91,91,91,91,91,91,91,91,
    90,90,90,90,90,90,90,90,89,89,89,89,89,89,89,88,88,88,88,88,88,
    88,88,88,87,87,87,87,87,87,87,87,87,87,86,86,86,86,86,86,85,85,
    85,85,85,85,85,84,84,84,84,84,84,83,83,83,83,83,83,82,82,82,82,
    81,81,81,81,81,81,81,81,81,81,81,81,80,80,79,79,79,78,78,78,78,
    78,77,77,77,77,76,76,76,76,76,76,76,75,75,75,75,75,75,75,74,74,
    74,74,74,74,74,74,74,74,74,73,73,73,73,73,73,73,72,72,72,72,72,
    71,71,71,71,71,71,71,70,70,70,70,70,70,70,70,69,69,69,69,68,68,
    68,68,68,68,67,67,67,67,67,67,67,67,67,67,66,66,66,66,65,65,65,
    64,64,64,64,64,64,64,64,64,64,64,64,63,63,63,63,63,63,63,63,63,
    62,62,62,62,62,61,61,61,61,61,61,61,60,60,60,60,60,59,59,59,59,
    59,59,58,58,58,58,58,57,57,57,57,57,56,56,56,56,56,56,56,56,56,
    55,54,54,54,54,54,53,53,53,53,52,52,52,52,52,52,52,52,51,51,51,
    51,51,51,50,50,50,50,50,50,49,49,49,49,49,49,49,49,48,47,47,47,
    47,47,47,47,47,47,47,46,46,46,46,46,45,45,45,45,45,45,44,44,44,
    44,44,43,42,42,42,42,42,42,42,42,41,41,41,41,41,41,40,40,40,40,
    40,40,40,39,39,39,39,39,39,38,38,38,38,38,38,38,38,37,37,37,37,
    37,37,37,37,37,37,36,36,36,36,36,35,35,35,35,35,34,34,34,34,34,
    34,34,33,33,33,33,33,33,33,32,32,32,32,32,31,31,31,31,31,31,30,
    30,30,30,29,29,29,29,29,28,28,28,28,28,27,27,27,27,27,26,26,26,
    25,25,25,25,25,25,24,24,24,24,24,24,24,24,23,23,23,23,23,23,23,
    23,22,22,22,22,22,22,22,21,21,21,21,21,21,21,21,21,20,20
  };
  const int n4c3w2_b[] = {
    150, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,100,100,99,99,99,99,98,98,98,98,98,97,97,
    97,97,97,97,96,96,96,96,96,95,95,95,95,95,95,94,94,94,94,94,94,
    94,94,93,93,93,93,93,92,92,92,92,92,92,92,91,91,91,91,91,91,91,
    91,90,90,89,89,89,89,89,89,89,89,89,89,89,88,88,88,88,88,87,87,
    87,86,86,86,86,86,86,86,85,85,85,85,85,85,84,84,84,84,83,83,83,
    83,83,83,83,83,83,83,83,82,82,82,82,82,81,81,81,81,81,81,80,80,
    80,80,80,80,80,80,80,80,80,79,79,79,79,79,79,79,78,78,78,78,78,
    78,78,78,77,77,77,77,77,77,77,76,76,76,76,76,76,76,76,76,76,75,
    75,75,74,74,74,74,74,73,73,73,73,73,73,73,73,73,72,72,72,72,72,
    72,72,72,71,71,71,71,71,71,71,71,71,70,70,70,70,70,69,69,69,69,
    69,68,68,68,68,68,68,68,68,68,68,68,67,67,67,67,67,67,67,67,66,
    66,66,66,66,66,65,65,65,65,65,64,64,64,63,63,63,63,63,63,63,62,
    62,62,62,62,62,62,61,61,61,61,61,61,60,60,60,60,59,59,59,58,58,
    58,58,58,57,57,57,56,56,56,56,55,55,55,55,55,55,55,54,54,54,54,
    54,54,54,54,54,54,53,53,53,53,52,52,52,52,52,52,52,51,51,51,51,
    50,50,50,50,50,49,49,49,49,49,49,49,49,48,48,48,48,48,47,47,47,
    47,47,47,47,47,47,47,46,46,46,45,45,45,45,45,45,44,44,44,44,44,
    43,43,43,43,43,43,42,42,42,42,41,41,41,41,41,41,40,40,40,40,40,
    40,40,40,40,39,39,39,39,39,39,38,38,38,38,38,38,37,37,37,37,37,
    37,36,36,36,36,36,36,36,36,36,36,36,35,35,35,35,35,35,35,35,35,
    34,34,34,34,33,33,33,33,33,33,32,32,32,32,32,32,32,32,32,32,31,
    31,31,31,31,31,30,30,30,30,30,30,30,30,29,29,29,29,29,29,28,28,
    28,28,28,28,28,28,26,26,26,26,26,26,26,25,25,25,24,24,24,24,23,
    23,23,23,22,22,22,22,22,22,21,21,21,21,21,20,20,20,20,20,20
  };
  const int n4c3w2_c[] = {
    150, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,99,99,99,99,99,99,99,99,99,99,98,98,98,97,97,
    97,97,97,97,97,96,96,96,96,96,95,95,95,94,94,94,94,94,93,93,93,
    93,93,93,93,92,92,92,92,92,92,92,92,91,91,91,91,91,91,91,91,91,
    90,90,90,90,90,89,89,89,89,89,88,88,88,88,88,88,88,88,88,87,87,
    87,87,87,87,87,87,87,86,86,86,86,86,86,86,85,85,85,85,84,84,84,
    83,83,83,83,83,83,83,82,82,82,82,82,82,82,81,81,81,81,80,80,80,
    80,79,79,79,79,79,79,79,78,78,78,78,78,77,77,77,77,77,77,77,77,
    77,76,76,76,76,76,76,75,75,75,75,75,75,74,74,74,74,74,74,73,73,
    73,73,72,72,72,72,72,71,71,71,71,71,71,71,71,70,70,70,70,70,70,
    70,70,70,70,70,69,69,69,69,69,69,69,69,69,68,68,68,68,68,68,68,
    68,68,68,68,68,67,67,67,67,66,66,66,65,65,64,64,64,64,64,64,63,
    63,63,63,63,63,63,63,62,62,62,62,62,62,62,61,61,61,60,60,60,60,
    60,60,60,60,60,60,59,59,59,59,59,59,59,58,58,58,58,58,58,58,58,
    58,58,57,57,57,57,57,57,56,56,56,56,56,56,56,56,56,56,55,55,55,
    55,54,54,54,54,54,54,54,54,54,54,53,53,53,53,53,52,52,52,52,52,
    52,52,51,51,51,51,51,51,50,50,50,50,50,49,49,49,49,49,48,48,48,
    47,47,47,47,47,47,47,46,46,46,46,46,46,46,45,45,45,45,45,45,44,
    44,44,44,44,44,44,43,42,42,42,42,42,41,41,41,41,40,40,40,40,40,
    39,39,39,39,39,39,39,39,39,38,38,38,38,37,37,37,36,36,36,36,36,
    36,35,35,35,35,34,34,34,34,34,34,34,34,34,33,33,33,33,33,33,32,
    32,32,32,32,32,32,31,31,31,31,31,31,31,31,30,30,29,29,29,29,29,
    29,28,28,28,28,28,28,27,27,27,27,27,27,26,26,26,26,26,26,26,26,
    26,26,26,25,25,25,25,25,24,24,24,24,24,24,24,23,23,23,23,23,22,
    22,22,22,22,22,22,21,21,21,21,21,21,21,21,20,20,20,20,20
  };
  const int n4c3w2_d[] = {
    150, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,99,99,99,99,99,99,99,98,98,98,98,98,97,97,97,
    97,97,97,97,96,96,96,95,95,95,95,95,95,95,94,94,94,94,94,94,93,
    93,93,93,93,93,93,93,93,92,92,92,92,91,91,91,91,91,90,90,90,90,
    90,90,90,89,89,89,89,89,89,89,88,88,88,88,88,88,88,87,87,87,87,
    87,87,87,87,86,86,86,86,86,85,85,85,85,84,84,84,84,84,83,83,83,
    83,83,82,82,81,81,80,80,80,80,80,80,80,80,80,79,79,79,79,79,79,
    79,79,78,78,78,78,78,78,78,78,78,78,78,78,77,77,77,77,77,77,77,
    77,77,77,76,76,76,76,76,76,75,75,75,75,74,74,74,74,74,74,74,73,
    73,73,73,73,73,72,72,72,71,71,71,71,71,71,71,70,70,70,70,70,69,
    69,69,69,69,69,69,69,69,68,68,68,67,67,67,67,67,66,66,66,66,66,
    65,65,65,65,65,65,65,65,64,64,64,64,64,64,63,63,63,63,63,63,63,
    63,63,63,62,62,62,62,62,62,61,61,61,61,61,61,61,60,60,60,60,60,
    60,60,60,60,59,59,59,59,59,59,59,59,59,59,59,59,59,58,58,58,58,
    58,58,57,57,57,57,57,56,56,56,56,56,56,55,55,54,54,54,54,54,54,
    54,54,54,54,54,53,53,53,53,53,53,53,53,52,52,52,52,52,52,52,52,
    52,52,51,51,51,51,51,50,50,50,50,50,50,49,49,49,49,49,49,48,48,
    48,47,47,47,47,47,47,47,46,46,46,46,46,46,45,45,45,45,45,45,45,
    45,44,43,43,43,43,43,43,42,42,42,42,41,41,41,40,40,40,40,40,40,
    40,40,40,39,39,39,39,39,39,39,39,39,39,38,38,38,38,38,38,38,37,
    37,37,37,37,37,37,36,36,36,36,35,35,35,35,35,35,34,34,34,34,34,
    34,34,34,34,34,34,34,34,33,33,33,32,32,32,32,32,32,31,31,30,30,
    30,30,30,30,30,29,29,29,29,29,29,29,29,29,28,28,28,28,28,27,27,
    27,27,27,27,27,27,26,26,26,26,25,25,25,24,24,24,23,22,22,22,22,
    22,22,22,21,21,21,21,21,21,21,21,21,20,20,20,20,20,20,20
  };
  const int n4c3w2_e[] = {
    150, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,100,100,100,99,99,99,99,99,98,98,98,98,98,
    98,97,97,97,97,97,97,97,96,96,96,96,96,96,96,96,96,96,95,95,95,
    95,94,94,94,94,94,94,93,93,93,93,93,92,92,92,92,91,91,91,91,91,
    91,90,90,90,90,90,90,90,90,90,90,90,89,89,89,89,89,89,88,88,88,
    88,87,87,87,87,87,86,86,85,85,85,85,85,85,85,84,84,84,84,84,84,
    83,83,83,83,83,83,83,83,83,82,82,82,82,82,82,82,82,82,82,82,82,
    82,81,81,81,81,80,80,80,80,80,79,79,79,79,79,79,79,79,78,78,78,
    78,78,77,77,77,77,77,77,76,76,76,76,76,75,75,75,75,74,74,74,74,
    74,74,74,74,74,74,73,73,73,73,73,73,72,72,72,72,72,72,71,71,71,
    71,71,71,70,70,70,69,69,69,69,69,69,69,69,69,69,69,69,68,68,68,
    68,68,68,68,68,68,67,67,66,66,66,66,66,65,65,64,64,64,64,64,63,
    63,63,63,62,62,62,62,61,61,61,61,61,60,60,60,60,60,59,59,59,59,
    59,59,58,58,58,58,58,58,58,57,57,57,57,57,57,56,56,56,56,56,56,
    56,55,55,55,55,54,54,54,54,54,54,54,53,53,53,53,53,53,53,52,52,
    52,52,51,51,50,50,50,49,49,49,49,49,49,49,49,48,48,48,48,48,48,
    48,48,47,47,47,47,47,46,46,46,46,46,46,46,46,45,45,45,45,45,45,
    45,45,45,44,44,44,44,44,44,43,43,43,43,43,43,43,43,43,43,43,42,
    42,42,42,42,42,41,41,41,41,41,40,40,40,40,40,40,39,39,39,39,39,
    38,38,38,38,38,38,38,37,37,37,37,37,37,37,37,37,36,36,35,35,35,
    35,35,35,35,34,34,34,34,33,33,33,33,32,32,32,32,32,32,32,32,32,
    32,32,32,32,32,32,31,31,31,31,31,31,31,31,30,30,30,30,30,30,30,
    30,30,30,29,29,29,29,29,29,29,29,28,28,28,28,28,28,28,28,28,28,
    27,27,27,27,27,26,26,26,26,26,25,25,24,24,24,24,24,23,23,23,23,
    23,23,23,23,22,22,22,22,22,22,22,22,22,21,21,20,20,20,20,20
  };
  const int n4c3w2_f[] = {
    150, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,100,100,100,100,100,100,99,99,99,99,99,99,
    99,98,98,98,98,98,97,97,97,97,97,97,97,97,97,96,96,96,96,95,95,
    95,95,95,95,95,95,95,95,95,95,94,94,94,94,94,94,94,94,94,94,93,
    93,93,93,93,92,92,92,91,91,91,91,91,91,91,90,90,90,90,90,90,90,
    90,90,89,89,89,89,89,89,89,88,88,88,88,88,87,87,87,87,87,87,87,
    86,86,86,86,85,85,85,85,85,85,85,84,84,84,84,84,84,84,84,84,83,
    83,83,83,83,83,83,83,83,83,82,82,82,82,82,82,82,82,81,81,81,81,
    81,81,80,80,80,80,80,79,79,79,79,79,79,79,79,79,78,78,78,78,78,
    78,78,77,77,77,76,76,76,76,76,76,75,75,75,75,75,75,74,74,74,74,
    74,73,73,73,73,73,73,73,73,73,73,72,72,72,72,72,72,72,72,72,72,
    71,71,71,71,71,70,70,70,69,69,69,69,68,68,68,68,68,68,68,68,67,
    67,67,67,67,67,67,67,66,66,66,66,66,65,65,65,65,64,64,64,64,64,
    63,63,63,63,63,63,63,62,62,62,62,62,61,61,61,61,61,61,61,60,60,
    60,60,60,59,59,59,59,59,59,59,59,58,58,58,58,58,58,57,57,57,57,
    57,57,57,56,56,56,56,56,56,55,55,55,55,55,55,54,54,54,54,54,54,
    54,54,53,53,52,52,52,52,52,52,52,51,51,51,51,51,50,50,49,49,49,
    49,49,49,48,48,48,48,48,48,48,48,48,48,47,47,47,47,47,47,47,46,
    46,46,46,46,46,46,45,45,45,45,45,45,45,45,44,44,44,44,44,44,43,
    43,43,43,43,43,43,42,42,42,42,42,42,42,41,41,41,41,41,41,41,40,
    40,40,39,39,39,38,38,38,38,38,38,38,37,37,37,37,37,37,37,36,35,
    35,35,35,35,34,34,34,34,34,34,34,33,33,33,33,33,32,32,32,32,32,
    31,31,31,31,31,30,30,30,30,30,30,30,29,29,29,29,29,29,28,28,28,
    28,28,28,28,27,27,27,27,27,27,27,26,26,26,26,26,26,25,25,24,24,
    24,24,24,24,23,22,22,22,22,22,22,22,22,21,21,21,21,20,20,20,20
  };
  const int n4c3w2_g[] = {
    150, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,99,99,99,99,99,99,99,99,98,98,98,98,98,98,
    97,97,97,97,97,96,96,96,96,96,96,96,95,95,95,95,95,95,94,94,94,
    94,94,94,93,93,93,93,93,93,93,92,92,92,92,92,92,92,91,91,91,91,
    91,91,91,91,90,90,89,89,88,88,88,88,88,88,87,87,87,87,86,86,86,
    86,86,85,85,85,85,85,85,85,85,85,85,85,85,85,85,85,84,84,84,84,
    84,83,83,83,83,82,82,82,82,81,81,81,81,81,80,80,80,80,80,79,79,
    79,79,79,79,79,79,78,78,78,78,78,77,77,77,77,77,77,77,76,76,76,
    76,76,76,76,75,75,75,75,75,75,75,75,75,74,74,74,74,74,74,74,74,
    74,74,74,73,73,73,73,73,73,73,72,72,72,72,72,72,71,71,71,71,70,
    70,70,70,70,70,70,70,69,69,69,69,69,68,68,68,68,68,68,67,67,67,
    67,67,67,67,66,66,66,66,66,66,65,65,65,65,65,65,64,64,64,63,63,
    63,63,62,62,62,62,61,61,61,61,61,61,61,61,61,60,60,60,59,59,59,
    59,59,58,58,58,58,58,57,57,57,57,57,57,57,57,57,56,56,56,56,56,
    56,56,56,56,56,56,56,55,55,55,54,54,54,54,54,54,54,53,53,53,53,
    53,53,53,53,53,53,52,52,52,52,52,52,51,51,51,51,51,51,51,51,50,
    50,50,50,50,49,49,49,49,49,49,49,49,49,49,49,48,48,48,48,48,48,
    48,47,47,47,47,47,47,46,46,46,46,45,45,45,45,45,45,45,45,44,44,
    44,44,44,43,43,43,43,42,42,42,42,41,41,41,40,40,40,40,39,39,39,
    39,39,39,38,38,38,38,38,38,38,38,37,37,37,37,37,37,37,36,36,36,
    36,36,36,36,36,35,35,35,35,35,35,34,34,34,34,34,33,33,33,33,33,
    33,33,33,33,33,33,32,32,32,32,32,32,32,32,31,31,31,31,31,31,31,
    31,30,30,30,30,30,30,29,29,29,29,29,28,28,28,28,27,27,27,27,27,
    27,27,27,27,27,27,26,26,26,26,26,25,24,24,24,24,24,24,24,23,23,
    23,23,23,22,22,22,22,22,22,22,21,21,21,21,21,21,21,20,20
  };
  const int n4c3w2_h[] = {
    150, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,100,100,100,99,99,99,99,98,98,98,98,98,98,
    97,97,97,96,96,96,96,96,96,95,95,95,94,94,94,94,94,94,94,93,93,
    93,93,93,93,92,92,92,92,91,91,91,91,91,91,91,91,91,90,90,90,90,
    89,89,89,89,89,89,89,88,88,88,88,88,88,88,88,87,87,87,87,87,86,
    86,86,86,86,86,85,85,85,85,85,85,84,84,84,84,84,84,84,83,83,83,
    83,83,83,83,83,82,82,82,82,82,82,82,82,81,81,81,81,81,81,81,80,
    80,80,80,80,80,80,80,80,79,79,79,79,79,79,79,79,78,78,78,78,78,
    77,77,77,77,77,77,76,76,76,75,75,75,75,75,74,74,74,74,74,74,74,
    74,73,73,73,73,73,73,73,72,72,72,72,72,72,71,71,71,71,71,71,71,
    71,71,70,70,70,70,70,70,70,69,69,69,68,68,68,68,68,68,68,67,67,
    67,67,67,67,67,66,66,66,66,66,66,66,66,65,65,65,65,65,65,65,64,
    64,64,64,64,64,64,64,64,64,63,63,63,63,62,62,62,62,61,61,61,61,
    60,60,60,60,60,60,60,60,60,60,59,59,59,59,59,58,58,58,58,58,58,
    58,58,58,58,57,57,57,57,57,57,56,56,56,56,56,56,55,55,55,55,55,
    54,54,54,53,53,52,52,52,52,52,52,52,52,51,51,51,51,51,50,50,50,
    50,50,49,49,49,49,49,49,49,48,48,48,48,48,48,48,48,47,47,47,47,
    47,47,46,46,46,46,46,46,45,45,45,45,45,45,45,45,45,45,45,44,44,
    44,44,43,43,43,43,43,43,42,41,41,41,41,41,41,41,41,40,40,40,40,
    40,40,40,40,40,40,40,39,39,39,38,38,38,37,37,37,37,37,37,37,36,
    36,36,36,35,35,35,35,35,35,35,34,34,34,34,34,34,34,33,33,33,33,
    33,33,32,32,31,31,31,31,31,31,31,30,30,30,30,30,30,30,29,29,29,
    29,29,29,29,29,29,29,29,28,28,28,28,28,28,28,28,28,28,27,27,27,
    27,27,26,26,26,26,26,26,25,25,25,25,25,25,25,25,25,24,24,24,23,
    23,23,23,23,23,23,22,22,22,22,22,22,21,21,21,21,21,20,20,20
  };
  const int n4c3w2_i[] = {
    150, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,100,100,99,99,99,99,99,99,99,99,98,98,98,
    98,98,98,98,97,97,97,97,97,96,96,96,96,95,95,95,95,95,94,94,94,
    94,94,93,93,93,93,92,92,92,92,92,92,92,92,91,91,91,91,91,91,90,
    90,90,90,90,89,89,89,89,89,89,88,88,88,87,87,87,87,87,87,87,86,
    86,86,86,85,85,85,85,84,84,84,84,83,83,83,83,83,83,83,83,83,82,
    82,82,82,81,81,81,81,81,81,81,80,80,80,80,80,80,80,80,80,79,79,
    79,79,79,79,79,79,79,78,78,78,77,77,77,77,77,76,76,76,76,76,75,
    75,75,75,75,75,75,74,74,74,74,74,73,73,73,73,73,73,73,72,72,72,
    72,72,71,71,71,71,71,71,71,70,70,70,70,69,69,69,69,69,69,68,68,
    68,68,68,68,68,68,68,67,67,67,67,67,67,66,66,66,66,66,66,65,65,
    65,65,65,65,65,64,64,64,64,64,64,64,64,63,63,63,63,63,63,63,62,
    62,62,62,62,62,61,61,61,61,61,61,60,60,60,60,60,60,60,60,60,60,
    59,59,59,59,59,58,58,58,58,58,58,57,57,57,57,57,56,56,56,56,56,
    56,56,56,56,55,55,55,55,55,54,54,54,54,54,54,54,53,53,53,53,53,
    52,52,52,52,52,52,52,51,51,51,51,51,51,50,50,50,50,50,50,49,49,
    49,49,49,48,48,48,48,48,48,48,47,47,47,47,46,46,46,46,46,46,46,
    45,45,45,45,45,45,45,45,44,44,44,44,44,44,44,44,44,44,44,44,43,
    43,43,43,43,42,42,42,42,42,42,42,42,42,42,42,41,41,41,40,40,40,
    39,38,38,38,38,38,38,38,37,37,37,37,37,37,37,37,37,36,36,36,36,
    36,36,35,35,35,34,34,34,34,34,34,33,33,33,33,33,32,32,32,32,32,
    32,31,31,31,31,31,31,31,31,30,30,30,30,30,30,30,30,30,30,30,29,
    29,29,29,29,29,29,28,28,28,28,28,28,27,27,27,27,27,27,27,27,26,
    26,26,26,26,26,26,26,25,25,25,25,25,25,25,24,24,24,24,24,24,24,
    24,24,24,23,23,23,23,22,22,21,21,21,21,21,21,21,21,20,20,20
  };
  const int n4c3w2_j[] = {
    150, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,100,100,100,99,99,99,99,99,99,99,99,98,98,
    98,98,98,98,98,98,98,98,98,97,97,97,97,97,96,96,96,96,96,95,95,
    95,95,95,94,94,94,94,93,93,93,93,93,92,92,92,92,91,91,91,91,91,
    91,91,90,90,90,90,90,90,90,90,90,90,89,89,89,89,88,88,88,88,88,
    88,88,88,88,87,87,87,87,86,86,86,86,86,86,86,86,85,85,84,84,84,
    84,84,84,83,83,83,83,83,83,83,82,82,82,82,82,82,82,82,82,82,81,
    81,81,81,80,80,80,80,80,80,79,79,78,78,78,78,78,78,78,78,78,78,
    78,77,77,77,77,77,77,77,77,77,77,77,76,76,76,76,76,76,75,75,75,
    75,75,74,74,74,74,74,74,74,74,74,74,73,73,73,73,73,73,72,72,72,
    72,71,71,71,71,70,70,70,70,70,70,70,69,69,69,69,69,68,68,68,67,
    67,67,67,67,67,67,67,66,66,66,66,66,66,66,65,65,65,65,65,64,64,
    63,63,63,63,62,62,62,62,62,62,62,62,62,61,61,61,61,60,60,60,60,
    60,60,60,60,60,60,60,59,59,59,59,59,58,58,58,58,58,58,58,58,58,
    57,57,57,56,56,56,56,56,56,56,56,55,55,55,55,54,54,54,54,54,53,
    53,53,52,52,52,52,52,52,52,52,52,52,52,52,51,51,51,51,51,50,50,
    50,50,50,50,50,50,49,49,49,49,49,49,49,49,49,49,49,48,48,48,48,
    48,48,48,48,48,48,48,48,47,47,47,47,47,46,46,46,46,46,46,46,45,
    45,45,45,45,44,44,44,44,44,44,44,44,44,43,43,43,43,43,43,43,42,
    42,42,42,42,42,42,41,41,40,40,40,40,40,40,40,39,39,39,39,39,39,
    38,38,38,38,38,38,37,37,37,37,37,36,36,36,36,36,35,35,35,35,35,
    35,34,34,33,33,33,33,33,33,33,33,33,32,32,32,32,32,32,32,32,31,
    31,31,31,30,30,30,30,30,30,29,29,29,29,29,29,29,28,28,28,27,27,
    27,27,27,27,27,26,26,26,26,26,26,26,25,25,25,25,24,24,24,24,23,
    23,23,23,23,23,23,23,22,22,22,22,22,22,22,21,21,21,21,21,20
  };
  const int n4c3w2_k[] = {
    150, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,100,100,99,99,99,99,99,99,99,98,98,98,98,
    98,98,98,98,97,97,97,97,97,96,96,96,96,96,96,96,96,96,95,95,95,
    95,95,95,95,95,95,94,94,94,94,94,94,93,93,93,93,93,93,93,93,93,
    92,92,92,92,92,92,92,92,92,91,91,91,91,91,91,91,91,91,90,90,90,
    90,90,89,89,89,89,89,88,88,88,88,88,88,88,88,88,87,87,87,87,87,
    87,86,86,85,85,85,85,84,84,84,84,84,84,84,83,83,83,83,83,82,82,
    82,82,82,82,82,81,81,81,81,80,80,80,79,79,79,79,79,78,78,78,78,
    78,78,78,78,78,77,77,77,77,77,77,77,77,76,76,76,76,76,76,76,76,
    75,75,75,75,75,75,75,75,74,74,74,73,73,73,73,73,73,73,72,72,72,
    72,72,72,72,72,72,71,71,71,71,71,70,70,70,70,70,69,69,69,69,69,
    68,68,68,68,68,68,67,67,67,67,67,66,66,66,66,66,65,65,65,65,65,
    65,65,65,65,65,65,64,64,64,64,64,64,64,63,63,63,63,63,63,63,63,
    63,63,63,62,62,62,61,61,61,61,61,61,61,61,60,60,60,59,59,58,58,
    58,58,58,57,57,57,57,57,57,57,57,56,56,56,56,56,56,55,55,55,55,
    54,54,54,54,54,53,53,53,53,53,53,52,52,52,52,52,52,52,52,51,51,
    51,51,51,50,50,50,50,50,49,49,49,49,48,48,48,48,48,47,47,46,46,
    46,46,46,46,46,46,46,45,45,45,45,45,45,45,44,44,44,43,43,43,43,
    43,43,43,42,42,42,42,41,41,41,41,41,41,41,41,41,41,40,40,40,40,
    40,40,40,40,40,40,40,39,39,39,39,39,39,39,38,38,38,38,38,37,37,
    37,37,37,37,37,36,35,35,35,35,35,35,35,35,35,34,34,34,34,34,34,
    33,33,33,33,33,32,32,32,32,32,32,32,32,31,31,31,30,30,30,30,29,
    29,29,29,29,29,29,28,28,28,28,28,27,27,27,27,27,27,26,26,26,26,
    25,25,25,25,25,25,25,25,24,24,24,24,24,24,24,24,24,24,24,24,24,
    23,23,23,22,22,22,22,21,21,21,21,21,20,20,20,20,20,20,20,20
  };
  const int n4c3w2_l[] = {
    150, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,100,100,99,99,99,99,99,99,98,98,98,98,98,
    98,98,98,97,97,97,97,97,97,97,97,96,96,96,95,95,94,94,94,94,94,
    94,94,93,93,93,93,93,93,93,93,92,92,92,92,92,92,92,92,91,91,91,
    91,91,91,91,91,90,90,90,90,90,90,90,89,89,89,89,89,89,88,88,88,
    88,88,88,88,88,87,87,87,87,86,86,86,86,86,86,86,86,86,85,85,85,
    85,85,85,85,84,84,84,84,84,84,84,84,83,83,83,83,83,83,83,83,83,
    82,82,81,81,81,81,81,81,81,80,80,80,80,80,79,79,79,79,79,79,79,
    79,79,79,78,78,78,78,78,78,78,77,77,76,76,76,76,75,75,75,75,75,
    75,75,74,74,74,74,74,74,73,73,73,73,73,72,72,72,72,72,72,72,71,
    71,71,71,71,71,70,70,70,70,70,70,70,70,69,69,69,69,69,69,69,69,
    68,68,68,68,68,68,67,67,67,66,66,66,66,66,66,65,65,65,65,65,64,
    64,64,64,64,64,63,63,63,63,63,63,62,62,62,62,62,62,62,61,61,61,
    61,61,60,60,60,60,60,60,59,59,59,59,59,59,59,59,58,58,58,58,58,
    57,57,57,57,57,56,56,56,56,56,56,56,56,56,56,55,55,55,55,55,55,
    55,54,54,53,53,53,53,52,52,52,51,51,51,50,50,50,50,50,49,49,49,
    49,48,48,48,48,48,48,48,48,47,47,47,47,47,47,46,46,46,45,45,45,
    45,45,45,45,45,44,44,44,44,44,44,44,44,43,43,43,42,42,42,42,42,
    42,42,42,41,41,41,40,40,40,40,40,40,40,39,39,39,39,39,39,39,39,
    38,38,38,38,38,38,38,38,38,37,37,37,37,37,37,37,37,37,36,36,36,
    36,35,35,35,35,35,35,35,35,34,34,34,34,33,33,33,33,33,33,33,33,
    33,33,33,33,32,32,32,32,32,32,32,32,31,31,30,30,30,29,29,29,28,
    28,28,28,28,28,28,27,27,27,26,26,26,26,26,25,25,25,25,25,25,25,
    25,25,25,25,25,24,24,24,24,24,24,24,24,24,24,23,23,23,23,23,23,
    23,23,23,23,22,21,21,21,21,21,21,21,21,21,20,20,20,20,20,20
  };
  const int n4c3w2_m[] = {
    150, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,100,100,99,99,99,99,99,99,99,99,99,98,98,
    98,98,98,98,97,97,97,97,97,97,97,97,96,96,96,96,95,95,95,94,94,
    94,94,93,93,93,93,93,93,93,93,92,92,92,91,91,91,91,91,91,91,91,
    91,91,91,90,90,90,90,90,89,89,89,88,88,88,88,88,88,87,87,87,87,
    87,87,87,86,86,86,85,85,85,85,85,85,84,84,84,84,84,84,84,83,83,
    83,83,83,83,82,82,82,82,82,82,81,81,81,81,81,80,80,80,80,80,80,
    79,79,79,79,79,79,79,78,78,78,78,78,78,77,77,77,77,77,77,77,77,
    77,76,76,76,76,76,76,76,76,75,75,75,75,75,75,75,74,74,74,74,73,
    73,73,73,73,72,72,72,72,72,72,71,71,71,71,71,71,71,70,70,70,70,
    70,70,70,69,69,69,69,69,68,68,68,68,67,67,67,67,67,66,66,66,66,
    66,66,65,65,65,65,65,64,64,64,64,64,63,63,63,63,63,62,62,62,62,
    62,62,62,62,62,62,61,61,61,61,61,61,61,60,60,60,60,60,60,59,59,
    59,59,59,59,59,58,58,58,58,57,57,57,57,57,57,56,56,56,56,56,56,
    56,56,56,55,55,55,55,55,55,55,55,55,54,54,54,54,54,54,54,53,53,
    53,53,53,53,53,53,53,52,52,52,52,51,51,50,50,50,50,50,50,50,50,
    50,50,49,49,49,49,48,48,48,48,48,48,48,48,48,47,46,46,46,46,46,
    45,45,45,45,45,44,44,44,44,44,43,43,43,43,42,42,42,42,42,41,41,
    41,41,41,41,40,40,40,40,40,40,40,39,39,39,39,39,39,39,39,39,39,
    39,38,38,37,37,37,37,37,37,36,36,36,36,36,36,36,36,36,35,35,35,
    35,34,34,34,33,33,33,33,33,33,33,33,33,33,32,32,32,32,32,32,31,
    31,31,31,31,31,30,30,30,30,30,29,29,29,29,29,29,29,28,28,28,28,
    28,27,27,27,27,26,26,26,26,26,26,25,25,25,25,25,25,25,25,25,24,
    24,24,24,24,24,24,24,23,23,23,23,23,23,23,23,22,22,22,22,22,21,
    21,21,21,21,21,21,21,21,21,21,20,20,20,20,20,20,20,20,20,20
  };
  const int n4c3w2_n[] = {
    150, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,99,99,99,99,98,98,98,98,98,98,97,97,97,97,
    97,97,97,97,96,96,96,96,96,96,96,95,95,95,95,94,94,94,94,94,94,
    94,94,94,94,93,93,93,92,92,92,92,91,91,91,91,91,91,91,91,91,91,
    90,90,90,90,90,90,89,89,89,88,88,88,88,87,87,87,87,87,87,87,86,
    86,86,86,86,85,85,85,84,84,84,84,84,83,83,83,83,83,83,83,83,83,
    83,82,82,82,82,82,82,81,81,81,81,81,81,81,81,80,80,80,80,79,79,
    79,79,79,79,79,79,78,78,78,78,78,78,78,78,78,77,77,77,76,76,76,
    76,76,76,76,76,75,75,75,75,75,74,74,74,74,73,73,73,73,73,73,73,
    73,73,72,72,72,72,72,72,72,72,72,72,71,71,71,71,71,71,71,70,70,
    70,70,70,70,70,69,69,69,68,68,68,68,68,67,67,67,67,66,66,66,65,
    65,65,65,65,65,64,64,64,64,64,64,63,63,63,63,63,63,63,63,62,62,
    62,62,62,62,62,61,61,61,61,60,60,60,60,60,59,59,59,59,59,59,59,
    59,59,59,59,59,59,58,58,58,58,58,57,57,57,57,57,57,56,56,56,56,
    56,56,56,56,55,55,55,55,55,54,54,54,54,54,54,54,54,54,54,54,53,
    53,53,53,53,53,53,52,52,51,51,51,51,51,51,51,51,50,50,50,50,50,
    49,49,49,49,49,49,49,48,48,48,48,48,47,47,47,47,47,47,47,46,46,
    46,46,46,45,45,45,45,45,44,44,44,44,44,44,43,43,43,43,43,43,42,
    42,42,42,42,42,42,41,41,41,40,40,40,40,39,39,39,39,39,39,39,39,
    38,38,38,38,38,38,38,38,37,37,37,37,37,37,37,36,36,36,35,35,35,
    35,35,34,34,34,34,33,33,33,33,33,33,33,33,33,33,33,33,33,32,32,
    32,32,32,32,32,32,32,31,31,31,31,31,31,31,31,31,31,31,30,30,30,
    30,30,29,29,29,29,29,28,28,27,27,27,27,26,26,26,26,26,25,25,25,
    25,25,25,24,24,24,24,24,24,24,23,23,23,23,23,23,22,22,22,22,22,
    22,21,21,21,21,21,21,21,21,21,21,20,20,20,20,20,20,20,20
  };
  const int n4c3w2_o[] = {
    150, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,100,100,100,100,100,100,100,100,99,99,99,
    99,99,98,98,98,97,97,97,97,96,96,96,96,96,96,96,96,96,95,95,95,
    95,95,95,95,95,95,95,94,94,94,94,94,94,93,93,93,93,93,93,93,93,
    92,92,92,92,92,92,91,91,91,91,91,90,90,90,90,90,90,89,89,89,89,
    89,89,89,88,87,87,87,87,87,86,86,86,86,86,86,86,86,85,85,85,85,
    85,85,85,85,85,85,84,84,84,84,84,84,83,83,83,82,82,82,82,82,82,
    81,81,81,81,81,81,81,81,81,81,80,80,80,79,79,79,79,79,78,78,78,
    78,78,78,78,77,77,77,77,77,77,77,77,77,77,76,76,76,76,76,76,76,
    75,75,75,74,74,74,74,74,74,74,74,74,73,73,73,73,73,73,72,72,72,
    72,72,71,71,71,71,71,70,70,70,70,70,70,70,70,70,70,70,70,70,70,
    69,69,69,69,69,69,69,69,69,69,69,69,69,68,68,68,68,68,68,68,68,
    68,68,68,67,67,67,67,67,67,67,66,66,66,66,65,65,65,65,65,65,64,
    64,64,63,63,63,63,63,63,63,62,62,62,62,62,62,62,61,61,61,61,61,
    61,61,61,60,60,60,60,59,59,59,59,59,58,58,58,58,58,57,57,57,57,
    57,57,57,57,57,56,56,56,56,56,56,55,55,55,55,54,54,54,54,54,54,
    54,54,53,53,53,53,53,52,52,52,52,52,52,52,52,52,51,51,51,51,51,
    51,51,51,51,51,51,51,51,51,51,50,50,50,50,50,49,49,49,49,49,49,
    49,49,48,48,48,48,48,48,48,47,47,47,47,47,47,47,47,47,45,45,45,
    44,44,44,44,44,44,44,43,43,43,43,43,42,42,42,42,42,42,41,41,41,
    41,41,41,41,40,40,40,40,40,40,40,39,39,39,39,39,39,39,38,38,38,
    38,37,37,37,37,37,37,37,36,36,36,35,35,35,35,35,34,34,34,34,34,
    33,33,32,32,32,32,32,32,32,31,31,31,31,31,30,30,30,30,30,30,29,
    29,29,28,28,28,28,28,27,27,27,26,26,26,26,26,25,24,24,24,23,23,
    22,22,22,22,22,22,21,21,21,21,21,21,21,21,20,20,20,20,20,20,20,
    20
  };
  const int n4c3w2_p[] = {
    150, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,100,100,100,100,100,100,100,99,99,99,99,99,
    99,99,98,98,98,98,98,98,97,97,97,97,97,97,96,96,96,96,96,95,95,
    95,95,95,95,94,94,94,94,94,93,93,93,93,93,93,92,92,92,92,92,91,
    91,91,91,91,91,90,90,90,90,90,90,90,90,90,89,89,89,89,89,88,88,
    88,88,88,88,87,87,87,87,87,87,87,87,86,86,86,86,85,85,85,85,85,
    85,85,85,85,85,84,84,84,84,84,84,84,84,84,84,83,83,83,83,83,83,
    83,83,83,82,82,82,81,81,81,80,80,80,80,80,80,80,79,79,79,79,78,
    78,78,78,78,78,78,78,77,77,77,77,77,77,77,76,76,76,75,75,74,74,
    74,74,74,74,74,74,74,73,73,73,73,73,73,72,72,72,72,71,71,71,71,
    71,71,70,70,70,70,70,70,70,69,69,68,68,68,68,68,68,67,67,67,67,
    67,67,67,66,66,66,66,65,65,65,65,65,64,64,64,64,64,64,64,63,63,
    63,63,63,63,63,63,62,62,62,62,62,62,62,62,61,61,61,60,60,60,60,
    60,60,60,60,60,59,59,59,59,59,59,59,59,59,58,58,58,58,58,57,57,
    57,57,57,56,56,56,56,56,56,56,55,55,55,55,55,54,54,54,54,54,53,
    53,53,53,53,53,53,52,52,52,52,52,51,51,51,51,51,50,50,50,50,50,
    49,49,49,49,49,49,48,48,48,48,48,47,47,47,47,47,47,47,47,47,47,
    46,46,46,46,46,46,45,45,45,45,45,44,44,44,44,44,44,43,43,43,43,
    43,43,43,42,42,42,42,42,42,42,42,42,42,42,42,41,41,41,41,41,41,
    41,41,40,40,39,39,39,39,39,39,39,39,39,39,38,38,38,37,37,37,37,
    37,37,37,37,37,37,37,37,36,36,36,36,35,34,34,34,34,34,34,34,34,
    34,33,33,33,33,33,33,33,32,32,32,32,32,31,31,31,30,30,30,29,29,
    29,29,29,29,29,29,29,28,28,28,28,28,28,27,27,27,27,27,27,27,27,
    26,26,25,25,25,25,25,25,25,25,24,24,24,24,24,24,24,24,23,23,23,
    23,22,22,22,22,22,22,22,22,22,22,21,21,21,21,21,21,20,20,20,20
  };
  const int n4c3w2_q[] = {
    150, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,100,100,100,99,99,99,98,98,98,98,98,98,98,
    98,98,98,97,97,97,97,97,97,97,97,96,96,96,96,96,95,95,95,95,94,
    94,94,94,94,94,94,94,94,93,93,93,93,93,93,93,93,93,93,93,92,92,
    92,92,92,92,92,92,92,91,91,91,90,90,90,90,90,90,89,89,89,89,89,
    89,88,88,88,88,88,88,88,88,88,88,87,87,87,87,87,86,86,86,86,86,
    86,86,86,86,85,85,85,85,85,85,85,84,84,84,84,84,84,84,84,84,84,
    83,83,83,82,82,82,82,81,81,81,81,81,81,81,80,80,80,80,79,79,79,
    79,79,79,79,78,78,78,78,77,77,77,77,76,76,76,76,76,75,75,75,75,
    74,74,73,73,73,73,73,73,72,72,72,72,72,72,72,72,71,71,71,71,71,
    71,71,70,70,70,70,70,70,70,69,69,69,69,69,68,68,68,68,68,67,67,
    67,67,66,66,65,65,65,65,65,64,64,64,64,64,64,64,64,64,64,64,63,
    63,63,63,63,63,63,62,62,62,62,62,62,62,62,61,61,61,61,60,60,60,
    60,60,60,59,59,59,58,58,58,57,57,57,57,56,56,56,56,56,56,55,55,
    55,55,54,54,54,54,54,54,54,54,54,54,54,53,53,53,53,53,53,52,52,
    52,52,52,52,51,51,51,51,51,51,51,51,50,50,49,49,49,49,49,49,49,
    48,48,48,48,48,48,48,48,48,48,47,47,46,46,46,46,46,46,46,45,45,
    45,45,45,45,45,45,44,44,44,44,44,44,43,43,43,43,42,42,42,42,42,
    41,41,41,41,40,40,40,40,39,39,39,38,38,38,38,38,37,37,37,36,36,
    36,36,36,35,35,35,34,34,34,34,34,34,34,34,34,33,33,33,33,33,33,
    33,33,32,32,32,32,32,32,32,31,31,31,31,31,31,31,30,30,30,30,30,
    30,30,30,29,29,29,29,29,29,29,29,28,28,28,28,28,28,28,28,27,27,
    27,27,27,26,26,26,26,26,26,26,26,25,25,25,25,25,25,25,25,25,25,
    25,24,24,24,24,24,24,24,24,24,24,23,23,23,23,22,22,22,22,22,22,
    22,22,22,22,21,21,21,21,21,21,21,20,20,20,20,20,20,20,20,20
  };
  const int n4c3w2_r[] = {
    150, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,100,99,99,99,98,98,98,98,98,97,97,97,97,96,
    96,96,95,95,95,95,95,95,95,95,94,94,94,94,94,93,93,92,92,92,92,
    92,92,91,91,91,91,91,91,90,90,90,90,90,90,90,89,89,89,89,89,89,
    89,89,88,88,88,87,87,87,87,87,86,86,86,86,86,86,86,86,86,86,85,
    85,85,85,85,85,85,84,84,84,84,84,84,84,83,83,83,83,83,83,83,83,
    83,83,83,82,82,82,82,82,82,82,82,82,82,82,82,81,81,81,81,80,80,
    80,80,80,80,79,79,78,78,78,77,77,77,77,77,77,76,76,76,76,76,75,
    75,75,74,74,73,73,73,73,73,73,73,72,72,72,72,72,72,72,72,72,72,
    72,71,71,71,71,71,70,70,70,70,70,69,69,68,68,68,68,67,67,67,67,
    67,67,67,67,66,66,66,66,66,66,66,66,66,66,66,65,65,65,65,65,64,
    64,64,64,64,64,64,64,64,64,63,63,63,63,62,62,61,61,61,61,61,61,
    61,61,61,61,61,60,60,60,60,60,60,60,60,59,59,59,59,59,59,59,59,
    59,59,58,58,58,58,57,57,57,57,57,57,57,56,56,56,55,55,55,55,55,
    55,55,54,54,54,54,54,54,54,53,53,53,53,53,53,52,52,52,52,52,52,
    52,52,52,51,51,51,51,51,51,51,50,50,50,49,49,49,49,49,48,48,48,
    48,48,48,47,47,47,47,47,47,46,46,46,46,46,46,46,45,45,45,45,44,
    44,44,44,44,43,43,43,43,43,43,42,42,42,42,42,42,42,41,41,41,41,
    41,40,40,40,40,40,40,39,39,39,39,39,39,38,38,38,38,37,37,37,37,
    37,36,36,35,35,35,35,35,34,34,34,34,34,34,34,34,33,33,33,33,33,
    33,33,33,33,32,32,32,32,32,32,32,31,31,31,31,31,31,31,30,30,30,
    30,30,30,30,30,30,29,29,29,29,29,29,29,29,28,28,28,28,28,28,28,
    28,28,27,27,27,27,27,27,27,27,27,26,26,26,25,25,25,25,25,24,24,
    24,24,24,24,24,24,23,23,23,23,23,23,23,23,23,23,23,22,22,22,22,
    22,21,21,21,21,21,21,21,21,20,20,20,20,20,20,20,20,20,20
  };
  const int n4c3w2_s[] = {
    150, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,100,99,99,99,99,99,99,99,99,98,98,98,98,97,
    97,97,97,97,97,97,97,97,96,96,96,96,96,96,95,95,95,95,95,95,94,
    94,94,94,94,94,94,94,93,93,93,93,93,93,93,92,92,92,92,92,91,91,
    91,91,90,90,90,89,89,89,89,89,89,89,89,89,88,88,88,88,88,87,87,
    87,87,87,87,87,86,86,86,86,86,86,86,85,85,85,85,85,85,85,84,83,
    83,83,83,83,83,82,82,82,82,82,81,81,81,81,81,81,81,81,80,80,80,
    80,80,80,80,80,80,80,80,79,79,79,79,79,79,79,78,78,78,78,78,78,
    78,78,77,77,76,76,76,76,75,75,75,75,74,74,74,74,73,73,73,73,73,
    73,72,72,72,72,72,71,71,71,70,70,70,69,69,69,69,68,68,68,68,68,
    67,67,67,67,67,67,67,66,66,66,66,66,66,66,66,66,66,65,65,65,65,
    65,65,65,64,64,64,64,64,64,64,63,63,63,63,63,63,63,63,63,63,62,
    62,62,62,62,62,61,61,61,60,60,60,60,60,60,60,59,59,59,59,59,59,
    58,58,58,57,57,57,57,57,57,56,56,56,56,55,55,55,55,55,55,54,54,
    54,53,53,53,53,53,53,53,53,53,52,52,52,52,52,52,52,52,51,51,51,
    51,51,51,51,50,50,50,50,50,50,50,50,49,49,49,49,49,49,49,48,48,
    48,48,48,48,48,47,47,47,46,46,46,45,45,45,45,45,45,44,44,44,43,
    43,43,43,43,43,43,42,42,42,42,42,42,41,41,41,41,41,41,41,41,40,
    40,40,40,40,39,39,39,39,39,39,38,38,38,38,38,38,38,38,37,37,37,
    37,37,37,37,36,36,36,36,36,36,36,36,36,36,36,35,35,35,35,35,35,
    35,35,34,34,34,34,34,34,33,33,33,32,32,32,32,32,32,31,31,31,31,
    31,31,31,31,31,31,30,30,30,30,30,29,29,29,29,29,28,28,28,28,28,
    28,27,27,27,27,27,27,27,26,26,26,26,26,26,26,25,25,25,25,24,24,
    24,24,24,24,24,24,24,24,24,23,23,23,23,23,23,23,23,23,23,22,22,
    22,22,22,22,22,21,21,21,21,21,21,20,20,20,20,20,20,20,20
  };
  const int n4c3w2_t[] = {
    150, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,99,99,99,99,99,99,98,98,98,97,97,97,97,97,
    97,97,97,96,96,96,96,96,95,95,95,95,95,95,95,94,94,94,93,93,93,
    93,93,93,93,92,92,92,92,91,91,91,91,91,90,89,89,89,89,89,89,88,
    88,88,88,87,87,87,87,87,86,86,86,86,85,85,85,85,85,85,85,85,84,
    84,84,84,84,84,84,84,83,83,83,83,83,83,82,82,82,82,82,81,81,81,
    81,81,81,80,80,80,80,80,79,79,79,79,78,78,78,78,78,78,78,77,77,
    77,77,77,77,77,77,77,77,77,77,77,77,76,76,76,76,76,75,75,75,75,
    75,75,75,75,75,75,74,74,73,73,73,73,73,73,72,72,72,72,71,71,71,
    71,71,71,71,70,70,70,70,70,70,70,70,70,69,69,69,69,68,68,68,68,
    67,67,67,67,67,67,67,67,66,66,66,65,65,65,65,64,64,64,64,64,64,
    64,63,63,63,63,62,62,62,61,61,61,61,61,61,61,61,60,60,59,59,59,
    59,59,59,59,59,59,59,58,58,58,58,58,58,58,58,57,57,57,57,57,57,
    57,56,56,56,56,56,56,56,56,56,56,56,55,55,55,55,55,54,54,54,54,
    54,54,54,54,53,53,53,53,53,53,52,52,52,52,52,52,51,51,51,51,51,
    51,51,50,50,50,50,50,50,50,50,49,49,49,49,49,49,49,49,49,49,49,
    48,48,48,48,48,48,48,48,48,48,48,47,47,47,47,47,47,47,47,46,46,
    46,46,46,46,46,45,45,45,45,44,44,44,44,44,43,43,43,43,43,43,43,
    43,42,42,42,42,42,42,41,41,41,41,41,40,40,40,40,40,40,40,40,39,
    39,39,39,39,39,39,38,38,38,38,37,37,37,37,37,37,37,37,37,37,36,
    36,36,36,36,36,36,35,35,35,35,34,34,34,34,33,33,33,33,33,33,32,
    32,32,31,31,31,31,31,31,31,31,30,29,29,29,29,28,28,28,28,28,28,
    28,28,28,28,27,27,27,27,27,27,27,27,27,26,26,26,26,26,26,26,26,
    25,25,25,25,25,24,24,24,24,24,24,24,24,24,23,23,23,22,22,22,22,
    22,22,21,21,21,21,21,21,21,21,20,20,20,20,20,20,20,20,20
  };
  const int n4c3w4_a[] = {
    150, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,100,100,99,99,99,99,99,98,98,98,98,98,98,
    98,98,98,98,97,97,97,97,97,97,97,96,96,96,96,96,96,96,96,95,95,
    95,95,95,94,94,94,94,94,94,93,93,93,93,93,93,93,92,92,92,92,92,
    92,92,92,91,91,91,91,91,91,91,91,91,90,90,90,90,90,90,89,89,89,
    89,88,88,88,88,88,88,87,87,87,87,87,87,87,87,87,86,86,86,86,86,
    86,85,85,85,85,85,85,85,85,85,85,85,84,84,84,84,84,84,84,84,83,
    83,83,83,83,83,83,82,82,82,81,81,81,81,81,81,80,80,80,80,80,80,
    80,80,79,79,79,79,78,78,78,78,78,78,78,77,77,77,77,77,77,77,76,
    76,76,76,76,76,76,76,76,76,75,75,75,75,75,75,75,75,74,74,73,73,
    73,73,73,73,73,73,73,72,72,72,72,72,72,72,72,72,72,72,72,72,71,
    71,71,71,71,70,70,70,70,70,70,70,70,70,70,70,69,69,69,69,69,69,
    68,68,68,67,67,67,67,67,66,66,66,66,66,65,65,65,65,65,65,65,65,
    65,64,64,64,64,64,64,64,63,63,63,63,63,63,63,62,62,62,62,62,62,
    62,61,61,61,61,61,60,60,60,60,60,59,59,59,59,58,58,58,58,58,58,
    58,58,58,57,57,57,57,57,57,57,57,57,56,56,56,56,56,56,56,56,56,
    55,55,55,55,55,55,55,55,55,54,54,54,54,53,53,53,53,53,53,53,53,
    53,53,53,53,52,52,52,52,52,52,52,52,52,52,52,52,51,51,51,51,51,
    51,50,50,50,49,49,49,49,48,48,48,48,48,48,48,48,48,48,48,48,48,
    47,47,47,46,46,46,46,46,46,45,45,45,45,45,45,44,44,44,44,44,44,
    43,43,43,43,43,43,43,43,43,43,43,43,42,42,41,41,41,41,40,40,40,
    40,40,40,40,40,40,40,40,40,39,39,39,39,39,38,38,38,38,38,38,38,
    38,38,38,37,37,37,37,37,37,36,36,36,36,36,36,36,36,35,35,35,35,
    35,35,34,34,33,33,33,33,33,33,33,32,32,32,32,32,32,32,32,31,31,
    31,31,31,31,31,31,31,31,31,31,30,30,30,30,30,30,30,30,30,30
  };
  const int n4c3w4_b[] = {
    150, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,99,99,99,99,99,98,98,98,98,98,98,98,98,98,98,
    98,97,97,97,97,97,97,97,97,97,97,96,96,95,95,95,95,95,94,94,94,
    94,94,94,94,94,93,93,93,93,93,93,93,92,92,92,92,92,92,91,91,91,
    91,91,91,91,91,91,91,90,90,90,90,90,90,89,89,89,89,89,89,89,89,
    89,88,88,88,88,88,88,88,87,87,87,87,87,87,86,86,86,86,86,85,85,
    85,85,85,85,85,85,85,84,84,84,84,84,84,84,84,83,83,83,82,82,82,
    82,82,81,81,81,81,81,81,81,80,80,80,80,80,80,80,79,79,79,79,79,
    79,78,78,78,78,78,78,77,77,77,77,77,77,77,77,76,76,76,76,76,75,
    75,75,75,75,75,75,74,74,74,74,74,74,74,74,74,74,74,73,73,73,73,
    73,73,73,73,73,72,72,72,72,72,72,71,71,71,71,71,71,71,70,70,70,
    70,70,70,70,69,69,69,69,69,68,68,68,67,67,67,67,67,67,67,67,67,
    67,67,66,66,66,66,66,65,65,65,65,65,64,64,64,64,64,63,63,63,63,
    63,63,63,62,62,62,62,62,62,62,62,61,61,61,61,60,60,60,60,60,60,
    60,60,60,60,60,59,59,59,59,58,58,58,58,58,58,58,57,57,57,57,57,
    56,56,56,56,56,56,55,55,55,55,55,55,55,55,55,55,54,54,54,54,54,
    54,53,53,53,53,53,53,53,52,52,52,52,52,51,51,51,51,51,51,51,51,
    51,51,51,51,50,50,50,50,50,49,49,49,49,49,49,49,48,48,48,48,48,
    48,47,47,47,47,47,47,47,47,47,46,46,46,46,46,46,46,46,46,45,45,
    45,45,45,45,45,45,44,44,44,44,44,43,43,43,43,43,43,43,43,43,43,
    43,43,43,43,42,42,42,42,42,42,42,42,42,42,42,42,42,41,41,41,41,
    41,41,41,41,41,40,40,40,40,40,40,40,40,40,39,39,39,39,39,38,38,
    38,38,38,37,37,37,37,37,37,36,36,36,36,36,36,36,35,35,35,35,35,
    35,35,35,35,34,34,34,34,34,34,34,34,34,34,34,33,33,33,33,33,33,
    32,32,32,32,32,31,31,31,31,31,31,31,31,30,30,30,30,30,30
  };
  const int n4c3w4_c[] = {
    150, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,100,100,100,100,100,100,99,99,99,99,99,99,
    99,99,99,99,99,99,98,98,98,98,97,97,97,97,97,97,96,96,96,96,96,
    96,96,96,96,95,95,95,95,95,95,95,95,94,94,94,94,94,94,93,93,93,
    93,93,93,93,93,92,92,92,92,92,92,91,91,91,91,91,91,91,91,90,90,
    90,90,90,89,89,89,89,89,89,88,88,88,88,88,88,87,87,86,86,86,86,
    86,86,85,85,85,85,85,85,85,85,84,84,84,83,83,83,83,83,83,83,83,
    83,83,82,82,82,82,82,81,81,81,81,81,81,81,81,80,80,80,80,80,80,
    80,80,80,80,80,80,80,80,79,79,79,79,79,79,79,79,79,79,79,78,78,
    78,77,77,77,77,77,77,77,76,76,76,76,76,76,76,76,76,75,75,75,75,
    74,74,74,74,74,74,74,74,74,73,73,73,73,73,73,73,72,72,72,72,72,
    72,72,72,71,71,71,71,71,70,70,70,70,70,70,70,70,69,69,69,69,68,
    68,68,68,68,68,68,67,67,67,67,66,66,66,66,66,66,66,66,65,65,65,
    65,65,65,65,65,64,64,64,64,64,64,64,64,64,64,63,63,63,63,63,63,
    62,62,62,62,62,62,62,62,61,61,61,61,61,60,59,59,59,59,58,58,58,
    58,58,58,58,58,58,57,57,57,57,57,57,57,56,56,56,56,56,56,56,56,
    56,56,56,55,55,55,55,55,55,55,54,54,54,54,54,53,53,53,53,53,52,
    52,52,52,52,52,52,52,51,51,51,51,51,51,51,51,50,50,50,50,50,50,
    50,50,50,50,50,49,49,49,49,49,49,49,48,48,48,48,48,47,47,47,47,
    47,47,47,47,47,46,46,45,45,44,44,44,44,44,44,44,44,44,44,44,44,
    43,43,43,43,43,43,43,43,43,43,42,42,42,42,42,41,41,41,41,41,41,
    41,41,41,40,40,40,40,39,39,39,39,39,38,38,38,38,38,38,38,38,38,
    38,38,38,37,37,37,37,37,37,37,37,37,36,36,36,36,36,36,36,36,36,
    36,35,35,35,35,35,35,35,35,34,34,34,34,34,33,33,33,33,33,33,33,
    33,33,33,32,32,32,32,32,32,32,32,32,31,31,31,30,30,30,30,30,30
  };
  const int n4c3w4_d[] = {
    150, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,99,99,99,99,99,98,98,98,98,98,98,97,97,97,97,97,97,97,97,
    96,96,96,96,96,96,96,95,95,95,95,95,94,94,94,94,94,94,94,94,94,
    93,93,93,93,92,92,92,92,92,92,92,92,92,92,91,91,91,91,91,91,91,
    90,90,90,90,89,89,89,89,89,88,88,88,88,88,88,88,88,87,87,87,87,
    87,87,86,86,86,86,86,86,85,85,85,84,84,84,84,84,84,84,84,84,84,
    84,84,83,83,83,83,83,83,83,82,82,82,82,82,82,82,82,82,82,81,81,
    81,81,81,81,81,81,81,81,81,80,80,80,80,80,80,80,80,80,79,79,79,
    79,78,78,78,78,78,77,77,77,77,77,77,76,76,76,76,76,76,76,76,76,
    76,75,75,75,75,75,75,75,75,75,75,74,74,74,74,74,74,74,74,74,74,
    74,74,73,73,73,73,72,72,72,72,71,71,71,71,70,70,70,70,70,70,70,
    69,69,69,69,69,69,69,69,69,69,69,69,68,68,68,68,68,68,68,68,68,
    68,68,68,67,67,67,67,67,66,66,66,66,66,66,66,66,66,66,66,65,65,
    65,65,65,65,64,64,64,64,64,64,64,64,63,63,63,63,63,63,63,62,62,
    62,62,62,62,62,61,61,61,61,61,61,61,61,60,60,60,60,60,60,59,59,
    59,59,59,58,58,58,58,58,58,58,58,58,58,58,58,58,57,57,57,57,57,
    56,56,56,56,55,55,55,55,55,54,54,54,54,54,54,54,53,53,53,53,53,
    53,53,52,52,52,52,52,52,52,52,51,51,51,51,51,51,50,50,50,50,50,
    50,50,50,49,49,49,49,49,48,48,48,48,48,48,47,47,47,47,47,47,47,
    46,46,46,46,45,45,45,45,45,45,45,45,45,44,44,44,44,44,44,44,44,
    44,44,43,43,43,43,43,43,43,43,42,42,42,41,41,41,41,41,41,41,40,
    40,40,40,40,40,40,40,39,39,39,39,39,39,38,38,38,38,38,38,38,37,
    37,37,37,37,37,36,36,36,36,36,36,36,36,35,35,35,35,35,35,35,35,
    35,35,34,34,34,34,34,34,34,34,34,33,33,33,33,33,33,32,32,32,32,
    32,32,32,31,31,31,31,31,31,30,30,30,30,30,30,30,30,30
  };
  const int n4c3w4_e[] = {
    150, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,100,100,100,100,99,99,99,99,99,99,99,98,98,
    98,98,98,98,98,97,97,97,97,97,97,97,97,96,96,96,96,96,95,95,95,
    95,95,95,94,94,94,94,94,94,94,94,94,94,94,93,93,93,92,92,92,92,
    92,92,92,92,92,92,92,92,92,91,91,91,91,91,91,91,91,91,90,90,90,
    90,90,89,89,88,88,88,88,88,88,87,87,87,87,87,87,87,86,86,86,86,
    86,85,85,85,85,85,85,85,84,84,83,83,83,83,83,83,83,83,82,82,82,
    82,82,82,82,82,82,82,81,81,81,81,81,81,80,80,80,80,80,80,80,80,
    80,80,80,79,79,79,79,79,79,78,78,78,78,78,78,78,77,77,76,76,76,
    76,76,76,76,76,76,75,75,75,75,75,75,75,75,75,75,75,74,74,74,74,
    74,74,74,74,74,74,73,73,73,73,73,73,73,73,73,72,72,72,72,72,72,
    72,71,71,71,71,71,71,71,70,70,70,70,70,70,70,70,70,70,69,69,69,
    68,68,68,67,67,67,67,67,67,67,67,67,66,66,66,66,65,65,65,65,65,
    65,65,64,64,64,64,64,64,64,63,63,63,63,63,63,63,63,63,62,62,62,
    62,62,62,61,61,61,61,61,61,61,61,61,61,60,60,60,60,60,60,59,59,
    59,59,59,59,58,58,58,58,58,58,58,58,57,57,57,57,57,57,57,57,57,
    56,56,56,56,56,56,55,55,55,55,55,55,54,54,54,54,54,54,54,54,54,
    54,54,53,53,53,53,53,52,52,52,52,52,51,51,51,51,51,51,51,51,51,
    50,50,50,50,50,50,50,50,50,50,49,49,49,49,49,48,48,48,48,48,48,
    48,48,47,47,47,47,47,47,47,46,46,46,46,46,46,46,46,46,46,46,45,
    45,45,45,45,45,45,45,45,45,45,44,44,44,44,44,44,44,44,43,43,43,
    43,42,42,42,42,42,41,41,41,41,40,40,40,40,40,39,39,39,39,39,39,
    39,39,39,39,39,38,38,38,38,38,38,38,37,37,37,37,37,37,37,37,36,
    36,36,36,35,35,35,35,35,35,35,35,35,35,34,34,34,34,34,34,34,34,
    33,33,33,33,33,32,32,32,32,32,32,31,31,31,31,31,30,30,30,30
  };
  const int n4c3w4_f[] = {
    150, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,99,99,99,99,98,98,98,98,98,98,98,97,97,97,97,97,97,97,
    97,96,96,96,96,96,96,96,95,95,95,95,95,95,95,95,95,95,94,94,94,
    94,94,94,94,94,94,94,94,94,94,94,94,94,94,93,93,93,93,92,92,92,
    92,92,92,92,92,92,92,92,91,91,91,91,91,91,91,91,90,90,90,90,90,
    89,89,89,89,89,89,89,89,89,89,89,89,88,88,88,88,88,88,87,87,87,
    87,87,87,87,86,86,86,86,86,86,86,85,85,85,85,85,85,85,84,84,84,
    84,84,84,84,84,83,83,83,83,83,83,83,83,82,82,82,82,82,82,82,82,
    82,82,82,82,82,82,82,81,81,81,81,81,81,81,81,80,80,80,80,80,79,
    79,79,79,79,79,79,79,78,78,78,78,78,78,78,78,78,77,77,77,77,77,
    77,76,76,76,76,76,75,75,75,75,75,75,74,74,74,74,74,74,74,74,74,
    73,73,73,73,73,73,73,73,73,72,72,72,72,72,72,72,72,71,71,71,71,
    71,71,70,70,70,69,69,69,69,69,69,69,69,68,68,68,68,68,68,68,68,
    67,67,67,66,66,66,66,66,65,65,65,65,65,64,64,63,63,63,63,63,63,
    63,63,62,62,62,62,62,61,61,61,61,61,61,61,61,61,60,60,60,60,60,
    60,60,59,59,59,59,59,59,58,58,58,58,57,57,57,57,57,57,56,56,56,
    56,56,56,56,56,56,55,55,55,55,55,55,55,55,55,54,54,54,54,54,54,
    53,53,53,53,53,52,52,52,52,52,52,50,50,50,50,50,50,50,50,50,50,
    50,49,49,49,49,49,49,49,49,49,49,48,48,48,48,48,48,47,47,47,47,
    47,47,46,46,46,45,45,45,45,45,44,44,44,44,44,44,44,44,43,43,43,
    43,43,43,42,42,42,42,42,42,42,41,41,41,41,41,41,41,41,41,41,40,
    40,40,40,40,40,40,38,38,38,38,38,38,38,38,37,37,37,37,37,37,37,
    37,36,36,36,36,36,36,36,35,35,35,35,35,35,35,35,35,35,35,35,34,
    34,34,34,34,34,34,33,33,33,33,33,33,33,32,32,32,32,32,32,32,32,
    31,31,31,31,31,31,31,30,30,30,30,30,30,30,30,30,30,30
  };
  const int n4c3w4_g[] = {
    150, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,100,100,100,100,100,100,99,99,99,99,99,98,
    98,98,98,98,97,97,97,97,97,97,97,96,96,96,96,96,96,96,95,95,95,
    95,95,94,94,94,94,94,93,93,93,93,93,93,93,93,93,93,93,93,92,92,
    92,92,92,92,92,92,92,92,92,92,91,91,91,91,91,91,90,90,90,90,89,
    89,89,89,89,89,89,88,88,88,88,88,88,87,87,87,87,87,86,86,86,86,
    86,86,86,86,86,86,86,85,85,85,85,85,85,85,84,84,84,84,84,84,84,
    84,84,83,83,83,83,83,83,83,82,82,82,82,82,82,82,82,81,81,81,81,
    81,81,81,81,81,81,80,80,80,80,80,80,80,79,79,79,79,79,79,79,79,
    79,79,78,78,77,77,77,77,77,77,76,76,76,75,75,75,75,75,75,75,75,
    75,74,74,74,74,74,74,74,74,74,74,74,74,73,73,73,73,73,72,72,72,
    72,72,72,72,72,72,72,71,71,71,71,71,71,71,70,70,70,70,70,70,70,
    69,69,69,69,69,69,68,68,68,68,68,68,68,68,68,68,68,68,68,68,68,
    67,67,67,67,67,67,67,67,67,66,66,66,66,66,66,66,66,66,66,66,66,
    66,66,65,65,65,65,65,65,65,64,64,64,63,63,63,63,63,63,63,62,62,
    62,62,62,61,61,61,61,61,60,60,60,60,60,60,60,60,60,60,60,60,59,
    59,59,59,59,59,59,59,59,59,58,58,58,58,58,58,58,57,57,57,57,57,
    57,56,56,56,56,56,56,56,56,55,55,55,55,55,55,55,55,54,54,54,54,
    54,54,54,54,53,53,53,53,53,52,52,52,52,52,51,51,51,51,51,51,50,
    50,50,49,49,49,49,49,49,49,48,48,48,48,47,47,47,47,47,47,46,46,
    46,46,46,46,46,46,46,45,45,45,45,45,45,44,44,44,44,43,43,43,43,
    43,42,42,42,42,42,42,42,41,41,41,41,41,40,40,40,40,39,39,39,39,
    39,39,39,39,38,38,38,38,38,38,38,38,38,37,37,37,37,36,36,36,36,
    36,36,36,36,36,35,35,35,35,34,34,34,34,33,33,33,33,33,33,33,33,
    32,32,32,32,32,32,32,31,31,31,31,31,31,31,31,31,30,30,30,30,30
  };
  const int n4c3w4_h[] = {
    150, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,100,100,99,99,99,99,99,99,99,99,98,98,98,
    98,98,98,98,98,97,97,97,97,96,96,96,96,96,96,96,96,96,96,96,96,
    95,95,95,95,95,95,94,94,94,94,94,94,94,94,93,93,93,93,93,93,93,
    93,92,92,92,92,92,91,91,91,91,91,91,91,91,91,91,90,90,90,90,89,
    89,88,88,88,88,88,88,88,88,88,87,87,87,87,87,87,87,87,87,87,87,
    86,86,86,86,86,86,85,85,85,85,85,85,85,84,84,84,84,83,83,83,83,
    83,83,83,83,83,83,83,83,82,82,82,82,82,82,82,82,82,82,81,81,81,
    81,81,81,81,81,80,80,80,80,80,80,80,80,79,79,79,79,79,79,79,79,
    79,79,79,78,78,78,78,78,78,78,77,77,77,76,76,76,76,76,76,76,75,
    75,75,75,75,74,74,74,74,74,73,73,73,73,73,73,73,72,72,72,72,72,
    72,71,71,71,71,71,70,70,70,70,70,70,70,70,70,69,69,69,69,69,69,
    69,68,68,68,68,68,68,68,68,68,68,67,67,67,67,67,67,66,66,66,66,
    66,66,66,65,65,65,65,65,65,65,65,64,64,64,64,64,64,63,63,63,63,
    63,63,63,62,62,62,62,62,61,61,61,61,61,61,60,60,60,60,60,60,60,
    60,59,59,59,59,59,59,59,58,58,58,58,58,58,58,58,58,58,57,57,57,
    57,57,57,57,57,57,56,56,56,56,56,56,55,55,55,55,55,55,55,55,54,
    54,54,54,54,54,54,53,53,53,53,53,53,53,52,52,52,52,52,52,52,52,
    52,51,51,51,51,51,51,51,51,50,50,50,49,49,49,49,49,49,49,49,49,
    49,49,48,48,48,48,47,47,46,46,46,46,46,45,45,45,45,45,45,45,44,
    44,44,44,44,44,43,43,43,43,43,43,42,42,42,42,42,42,42,41,41,41,
    41,41,41,40,40,40,40,40,40,39,39,39,39,39,39,39,38,38,38,38,38,
    38,38,37,37,37,37,37,37,37,36,36,36,36,36,36,36,36,35,35,35,35,
    35,35,34,34,34,34,34,34,33,33,33,33,33,32,32,32,32,32,32,31,31,
    31,31,31,31,31,31,31,31,31,31,30,30,30,30,30,30,30,30,30,30
  };
  const int n4c3w4_i[] = {
    150, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,100,100,99,99,99,99,99,99,99,99,99,99,99,
    99,99,99,98,98,98,98,98,98,97,97,97,97,97,97,97,97,97,96,96,96,
    96,96,96,96,96,96,95,95,95,95,95,95,95,95,95,94,94,94,94,94,94,
    94,94,94,94,94,94,93,93,93,93,93,93,92,92,92,92,92,92,92,92,92,
    91,91,91,91,91,91,91,91,90,90,90,90,90,89,89,89,89,89,89,89,88,
    88,88,88,88,88,88,87,87,87,87,87,87,87,87,86,86,86,86,86,85,85,
    85,85,85,85,85,84,84,84,84,84,84,84,84,84,84,83,83,83,83,83,83,
    83,83,83,83,83,83,83,82,82,82,82,82,82,82,82,81,81,81,81,80,80,
    80,80,80,80,80,80,80,80,79,79,79,79,79,79,79,78,78,78,78,78,77,
    77,77,77,76,76,76,76,76,75,75,75,75,74,74,74,74,74,73,73,73,73,
    73,73,73,73,72,72,72,72,72,72,72,71,71,71,71,71,71,71,71,70,70,
    70,70,70,70,70,70,70,70,69,69,69,69,69,69,69,68,68,68,68,68,68,
    67,67,67,67,67,67,67,67,66,66,66,66,66,66,66,65,65,65,64,64,64,
    64,64,63,63,63,63,63,63,63,63,62,62,62,62,62,62,61,61,61,61,61,
    61,61,61,61,61,60,60,60,60,60,60,60,60,59,59,59,58,58,58,58,58,
    57,57,57,57,56,56,56,56,56,55,55,55,55,55,55,55,55,55,54,54,54,
    54,54,54,54,54,54,54,54,54,53,53,53,53,53,53,53,52,52,52,52,52,
    52,52,52,52,52,52,52,52,51,51,51,51,50,50,50,50,50,49,49,49,49,
    49,49,49,49,48,48,48,48,48,48,48,48,47,47,47,47,47,47,47,47,46,
    46,46,46,45,45,45,45,45,45,44,44,44,43,43,43,43,43,42,42,42,42,
    42,41,41,41,41,41,41,40,40,39,39,39,39,39,39,39,39,39,39,38,38,
    38,38,38,38,37,37,37,37,37,37,37,36,36,36,36,36,36,36,36,36,35,
    35,35,34,34,34,34,34,34,34,33,33,33,33,33,33,33,33,33,32,32,32,
    32,32,32,32,32,31,31,31,31,31,31,31,31,30,30,30,30,30,30,30
  };
  const int n4c3w4_j[] = {
    150, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,100,99,99,99,98,98,98,98,98,98,98,98,97,97,
    97,97,97,97,97,97,97,96,96,96,96,95,95,95,95,95,94,94,94,94,94,
    93,93,93,93,92,92,92,92,92,92,92,92,91,91,91,90,90,90,90,90,90,
    90,90,90,89,89,89,89,89,89,89,89,89,89,89,89,88,88,88,88,88,87,
    87,87,87,87,87,86,86,86,86,86,86,85,85,85,85,85,84,84,84,84,84,
    84,83,83,83,83,82,82,82,82,82,81,81,81,81,81,80,80,80,80,80,80,
    80,79,79,79,79,79,79,78,78,78,78,78,78,78,77,77,77,77,77,77,77,
    77,77,76,76,76,76,76,76,76,75,75,75,75,75,75,75,74,74,74,74,74,
    74,74,74,74,73,73,73,73,73,73,73,72,72,72,72,72,72,72,72,72,71,
    71,71,71,71,71,71,71,71,71,71,70,70,70,70,70,70,69,69,69,69,69,
    69,69,69,69,69,69,68,68,68,68,67,67,67,67,67,67,66,66,66,66,66,
    66,66,66,66,66,65,65,65,65,65,65,65,65,64,64,64,64,64,63,63,63,
    63,63,63,63,63,63,63,63,62,62,62,62,62,62,62,62,62,61,61,61,61,
    60,60,60,60,60,59,59,59,59,59,59,59,59,58,58,58,58,58,57,57,57,
    57,57,57,57,57,56,56,56,56,56,56,56,56,56,55,55,55,54,54,54,54,
    54,53,53,53,53,53,53,53,53,52,52,52,52,52,52,52,52,52,52,51,51,
    51,51,51,51,51,51,51,50,50,50,50,50,50,50,50,50,50,50,49,49,49,
    49,49,49,49,49,49,49,49,49,48,48,48,48,48,48,48,48,48,47,47,47,
    47,47,47,47,47,46,46,46,46,46,45,45,45,45,45,45,45,44,44,44,44,
    44,44,44,44,44,43,43,43,43,43,42,42,42,42,42,41,41,41,40,40,40,
    40,40,40,40,40,40,40,40,40,39,39,39,39,39,39,39,39,39,38,38,38,
    38,38,38,38,38,38,38,37,37,37,37,37,37,37,37,37,37,36,36,36,36,
    35,35,35,35,35,35,35,35,35,35,34,34,34,34,34,33,33,33,33,33,33,
    32,32,32,32,32,32,32,31,31,31,31,30,30,30,30,30,30,30,30
  };
  const int n4c3w4_k[] = {
    150, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,100,100,100,99,99,99,99,99,99,99,99,99,99,
    98,97,97,97,97,97,97,97,97,96,96,96,96,96,96,96,96,96,96,96,95,
    95,95,95,95,95,94,94,94,94,94,94,94,94,93,93,93,93,92,92,92,92,
    92,92,92,92,92,91,90,90,90,89,89,88,88,88,88,88,88,88,88,88,88,
    88,87,87,87,86,86,86,86,86,86,86,85,85,85,85,85,85,85,85,85,85,
    84,84,84,84,84,84,83,83,83,83,83,82,82,82,81,81,81,81,81,81,80,
    79,79,79,79,79,79,79,79,79,79,79,79,78,78,78,78,78,78,78,78,78,
    77,77,77,77,77,77,77,77,77,76,76,76,76,76,76,76,76,75,75,75,75,
    75,75,75,75,75,74,74,74,74,74,74,74,74,73,73,73,73,73,73,73,72,
    72,72,72,72,72,72,71,71,71,71,71,71,71,71,71,71,71,71,71,71,71,
    71,71,71,71,71,70,70,70,70,70,70,70,70,69,69,69,69,68,68,68,68,
    67,67,67,67,67,67,67,66,66,66,66,65,65,65,65,65,65,65,65,65,65,
    65,65,65,64,64,64,64,63,63,63,63,62,62,62,62,62,61,61,61,61,61,
    61,61,60,60,60,60,60,60,59,59,59,58,58,58,58,58,58,57,57,57,57,
    57,57,57,57,56,56,56,55,55,55,55,55,55,55,55,55,54,54,54,54,54,
    54,54,54,54,53,53,53,53,53,53,52,52,52,52,52,52,52,52,52,51,51,
    51,51,51,51,50,50,50,50,50,50,50,49,49,49,49,49,49,49,49,49,49,
    49,49,49,49,49,48,48,48,48,48,48,48,48,48,48,48,48,48,47,47,47,
    47,47,47,47,47,47,46,46,46,46,46,45,45,45,45,45,45,45,45,45,44,
    44,44,44,44,44,43,43,43,43,43,43,43,43,42,42,42,42,42,42,42,41,
    41,41,41,41,41,41,41,41,41,40,40,40,40,40,40,39,39,39,39,39,39,
    39,39,39,39,38,38,38,38,38,37,37,37,37,37,36,36,36,36,36,36,36,
    36,36,36,35,35,35,35,35,35,35,35,35,34,34,33,33,33,33,33,33,33,
    32,32,32,32,32,32,31,31,31,31,31,31,31,31,30,30,30,30,30,30
  };
  const int n4c3w4_l[] = {
    150, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,100,100,99,99,99,99,99,99,99,98,98,98,98,
    97,97,97,97,97,97,97,97,97,97,97,96,96,96,96,96,96,96,96,96,95,
    95,95,95,95,94,94,94,94,94,94,94,94,93,93,93,93,93,92,92,92,92,
    92,92,92,92,92,91,91,91,91,91,91,91,91,91,90,90,90,90,90,90,89,
    89,89,89,89,89,89,89,89,88,88,88,88,88,88,88,88,87,87,87,87,87,
    87,87,87,87,86,86,86,86,86,86,86,86,86,86,86,86,85,85,85,85,85,
    84,84,84,84,84,84,84,84,84,84,83,83,83,83,83,81,81,81,81,81,81,
    81,80,80,80,80,80,80,80,79,79,79,79,79,79,79,79,78,78,78,78,78,
    77,77,76,76,76,76,76,76,76,76,76,75,75,75,75,75,75,75,75,74,74,
    74,74,74,74,73,73,73,73,73,73,73,73,73,73,73,72,72,72,72,71,71,
    71,71,71,70,70,70,70,70,70,70,69,69,69,69,69,69,68,68,68,68,68,
    68,68,68,67,67,67,67,67,66,66,66,66,66,66,66,65,65,65,65,65,65,
    65,65,65,65,65,64,64,64,64,64,64,64,64,64,64,63,63,63,63,63,63,
    62,62,62,62,62,62,62,62,62,62,62,62,61,61,61,61,61,61,60,60,60,
    60,60,60,60,60,60,60,60,60,60,59,59,58,58,58,58,58,58,57,57,57,
    57,56,56,56,56,56,55,55,55,55,54,54,54,54,54,54,54,54,54,53,53,
    53,53,52,52,52,52,52,52,51,51,51,51,50,50,50,50,50,50,50,49,49,
    49,49,49,49,49,49,49,48,48,48,48,48,47,47,47,47,47,47,47,46,46,
    46,46,46,46,46,45,45,45,45,45,45,45,45,44,44,44,44,44,44,44,44,
    44,43,43,43,43,43,43,43,43,43,43,43,43,43,42,42,42,42,42,41,41,
    41,41,41,41,41,40,40,40,40,40,40,39,39,39,39,39,38,38,38,38,38,
    38,38,37,37,37,37,37,37,37,37,37,37,36,36,36,36,36,36,35,35,35,
    35,34,34,34,34,34,34,33,33,33,33,33,33,33,33,33,33,33,32,32,32,
    32,32,32,32,31,31,31,31,31,31,31,30,30,30,30,30,30,30,30,30
  };
  const int n4c3w4_m[] = {
    150, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,99,99,99,99,99,99,99,99,99,99,99,99,98,98,98,
    98,98,98,98,98,98,97,97,97,97,97,97,97,96,96,96,96,96,95,95,95,
    94,94,94,94,94,94,94,93,93,93,93,93,93,93,92,92,92,92,92,91,91,
    91,91,91,91,91,90,90,90,90,90,90,90,90,90,89,89,89,89,89,89,89,
    88,88,88,88,87,87,87,87,87,87,87,86,86,86,86,86,86,86,86,85,85,
    85,85,85,85,84,84,84,84,84,84,84,84,83,83,83,82,82,82,82,81,81,
    81,80,80,80,80,80,80,80,80,80,79,79,79,79,79,79,79,78,78,78,78,
    78,78,78,78,78,78,77,77,77,77,77,77,77,77,77,77,76,76,76,76,76,
    76,76,76,76,76,76,75,75,75,75,74,74,74,74,74,74,73,73,73,73,73,
    73,73,72,72,72,72,72,72,72,72,71,71,71,71,71,71,71,71,71,70,70,
    70,70,70,70,70,70,69,69,69,69,69,68,68,68,68,68,68,68,67,67,67,
    67,67,67,67,67,66,66,66,66,66,66,66,66,66,66,66,65,65,65,65,65,
    65,65,65,64,63,63,63,63,63,62,62,62,62,62,62,62,62,61,61,61,61,
    61,60,60,60,60,60,60,60,60,60,60,59,59,58,58,58,58,58,58,57,57,
    57,57,57,57,57,57,56,56,56,55,55,55,55,55,55,55,55,55,55,54,54,
    54,54,54,54,53,53,53,53,53,53,53,53,53,53,53,52,52,52,52,52,52,
    52,52,52,52,52,52,52,51,51,51,51,51,51,51,50,50,50,50,50,50,49,
    49,49,49,49,49,49,49,49,49,48,48,48,48,48,48,48,48,48,47,47,47,
    47,47,47,46,46,46,46,46,46,46,45,45,45,45,45,45,45,45,44,44,44,
    44,44,44,43,43,43,43,43,43,43,43,42,42,42,42,42,42,42,42,42,42,
    41,41,41,41,41,41,41,41,41,41,40,40,40,40,40,40,40,40,39,39,39,
    39,38,38,38,38,38,38,37,37,37,37,37,37,36,36,36,36,36,36,36,36,
    35,35,35,35,35,35,35,35,34,34,34,34,34,34,34,33,33,33,33,33,33,
    32,32,32,32,32,32,31,31,31,31,31,31,30,30,30,30,30,30,30
  };
  const int n4c3w4_n[] = {
    150, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,100,100,100,100,100,100,99,99,99,99,99,99,
    99,99,99,99,98,98,98,98,97,97,97,97,97,97,97,97,96,96,96,96,96,
    96,96,96,96,96,96,96,96,96,96,96,95,95,95,94,94,94,94,94,94,94,
    94,94,94,93,93,93,93,93,92,92,92,92,92,92,92,92,92,92,92,92,91,
    91,91,91,91,90,90,90,90,89,89,89,89,89,89,89,88,88,88,88,88,88,
    88,88,88,88,88,87,87,87,87,87,87,86,86,86,86,86,86,86,86,85,85,
    85,85,85,85,84,84,84,84,84,84,84,84,84,83,83,83,83,83,83,83,82,
    82,82,82,82,82,81,81,81,81,81,81,81,81,81,81,81,81,81,80,80,80,
    80,80,80,80,80,80,79,79,79,79,79,79,79,79,78,78,78,78,78,78,78,
    77,77,77,77,77,77,76,76,76,76,76,76,76,76,75,75,75,75,75,75,75,
    75,74,74,74,74,74,74,74,73,73,73,73,73,73,73,73,72,72,72,72,72,
    72,71,71,71,70,70,70,70,70,70,70,70,70,70,69,69,69,69,69,69,68,
    68,68,68,68,68,68,68,68,67,67,67,67,67,67,67,66,66,66,66,66,66,
    65,65,65,65,65,65,65,65,65,64,64,64,64,64,64,64,63,63,63,63,63,
    63,63,63,62,62,62,62,62,62,61,61,61,61,61,61,61,60,60,60,60,60,
    60,60,60,60,59,59,59,58,58,58,58,57,57,57,57,57,57,56,56,56,55,
    55,55,55,55,54,54,54,54,54,54,54,54,53,53,53,53,53,53,52,52,51,
    51,51,51,51,50,50,50,50,50,49,49,49,49,49,49,49,48,48,48,48,48,
    48,48,47,47,47,47,47,47,47,47,47,46,46,46,46,46,46,46,46,45,45,
    45,45,45,44,44,44,44,44,44,44,44,44,43,43,43,43,43,42,42,42,42,
    42,42,42,42,42,42,42,41,41,41,41,41,40,40,40,40,40,40,40,39,39,
    39,39,39,39,39,38,38,38,38,38,38,38,38,37,36,36,36,36,36,36,36,
    36,36,36,35,35,35,35,35,35,35,35,35,34,34,34,34,33,33,33,33,33,
    33,33,33,32,32,32,32,32,32,32,31,31,31,31,31,30,30,30,30,30,30
  };
  const int n4c3w4_o[] = {
    150, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,100,99,99,99,99,99,99,99,99,98,98,98,98,98,
    98,98,98,98,98,97,97,97,97,97,97,97,97,97,97,96,96,95,95,95,95,
    95,95,95,95,94,94,94,94,94,94,94,94,93,93,93,93,93,93,93,93,93,
    93,92,92,92,91,91,91,91,91,91,91,90,90,90,90,90,90,90,90,90,89,
    89,89,89,89,89,89,89,88,88,88,88,88,88,88,88,88,88,87,87,87,87,
    87,87,87,87,87,86,86,86,86,86,86,86,85,85,85,85,85,85,85,85,85,
    84,84,84,84,84,84,84,84,84,83,83,83,83,83,83,83,83,83,83,83,83,
    82,82,82,82,82,82,82,82,81,81,81,81,81,81,81,81,80,80,80,80,80,
    79,79,79,79,79,79,79,79,79,79,79,78,78,78,78,78,78,78,77,77,77,
    77,77,77,77,76,76,76,76,76,76,75,75,75,75,75,75,75,75,75,75,75,
    74,74,74,74,74,74,74,74,73,73,73,73,73,73,72,72,72,72,72,72,72,
    71,71,71,71,71,71,71,71,71,71,71,70,70,70,70,70,70,70,70,70,70,
    69,69,69,69,69,68,68,68,68,68,68,68,68,67,67,67,67,67,67,67,66,
    66,66,66,66,66,66,66,66,65,65,65,65,65,64,64,64,64,64,64,64,64,
    64,64,64,64,64,64,63,63,63,63,63,63,63,62,62,62,62,62,61,61,61,
    60,60,60,60,59,59,59,59,59,58,58,58,58,58,58,57,57,57,57,57,57,
    57,57,57,56,56,56,56,56,56,56,56,56,56,55,55,55,55,55,55,55,55,
    55,54,54,54,54,54,54,54,53,53,53,53,53,53,53,52,52,52,52,52,52,
    51,51,51,50,50,50,50,49,49,49,49,49,49,48,48,48,48,48,48,48,48,
    48,47,47,47,47,46,46,46,46,45,44,44,44,44,44,44,44,43,43,43,43,
    43,43,43,42,42,42,42,42,41,41,40,40,40,40,40,39,39,39,39,38,38,
    38,38,38,38,38,37,37,37,37,37,37,37,37,37,37,36,36,36,36,36,35,
    35,35,35,35,35,35,35,35,35,34,34,34,34,34,34,34,34,34,34,33,33,
    33,32,32,32,32,32,32,32,32,31,31,31,31,31,31,30,30,30,30
  };
  const int n4c3w4_p[] = {
    150, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,99,99,99,99,99,99,99,99,98,98,98,98,98,98,
    97,97,97,97,97,97,97,97,97,97,97,97,96,96,96,96,96,96,95,95,95,
    95,95,95,94,94,93,93,93,93,93,92,92,92,92,92,92,92,92,92,92,92,
    92,91,91,91,91,91,91,91,91,91,91,90,90,90,90,90,90,90,90,90,90,
    90,90,90,89,89,89,89,89,89,89,89,89,89,88,88,88,88,88,88,88,87,
    87,87,87,87,87,87,87,86,86,86,86,86,86,86,85,85,85,84,84,84,84,
    84,84,83,83,82,82,82,82,81,81,81,81,81,81,81,81,80,80,80,80,80,
    80,80,80,79,79,79,79,79,79,79,79,79,78,78,78,78,78,78,78,77,77,
    77,77,77,77,77,77,76,76,76,76,76,76,76,76,75,75,75,75,75,75,75,
    74,74,74,74,74,74,74,74,73,73,73,73,73,73,73,73,73,72,72,72,72,
    72,71,71,71,71,71,71,71,70,70,70,70,70,69,69,69,69,69,69,69,68,
    68,68,68,68,68,68,67,67,67,67,66,66,66,66,66,66,66,66,65,65,65,
    65,65,65,65,64,64,64,64,64,64,64,64,64,63,63,63,63,63,63,63,62,
    62,62,61,61,61,61,61,61,61,61,60,60,60,60,60,60,60,60,60,59,59,
    59,59,59,59,59,58,58,58,58,58,57,57,57,57,57,57,56,56,56,56,56,
    56,56,56,55,55,55,55,55,55,54,54,54,54,54,54,53,53,53,53,53,53,
    53,53,53,53,53,52,52,52,52,52,52,51,51,51,51,51,51,51,50,50,50,
    50,50,49,49,49,49,49,48,48,48,48,48,48,48,47,47,47,47,47,46,46,
    46,46,46,46,46,46,46,46,46,46,45,45,45,45,45,45,45,45,45,45,44,
    44,44,44,44,44,44,43,43,43,43,43,43,42,42,42,42,42,42,42,41,41,
    41,41,41,41,41,41,41,40,40,40,39,39,39,39,39,39,39,38,38,38,38,
    38,38,37,37,37,37,37,37,37,37,37,37,37,37,36,36,36,36,36,36,35,
    35,35,35,35,35,35,34,34,34,34,34,34,34,33,33,33,33,33,33,33,33,
    32,32,32,32,32,31,31,31,31,31,31,30,30,30,30,30,30,30,30
  };
  const int n4c3w4_q[] = {
    150, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,99,99,99,99,99,99,99,99,99,99,98,98,98,98,98,98,98,
    98,98,98,98,98,98,98,97,97,97,97,97,97,96,96,96,96,96,96,96,95,
    95,95,94,94,94,94,94,94,94,93,93,93,93,93,93,93,93,93,92,92,92,
    92,92,91,91,91,91,91,91,91,91,91,91,91,91,90,90,90,90,90,90,90,
    90,89,89,89,89,89,89,89,89,89,88,88,88,88,88,88,88,88,88,88,87,
    87,87,87,87,86,86,86,86,86,86,86,86,86,85,85,85,85,85,84,84,84,
    84,84,84,84,84,83,83,83,83,83,83,83,83,82,82,82,82,81,81,81,81,
    81,81,80,80,80,80,80,80,80,80,80,79,79,79,79,79,78,78,78,78,78,
    77,77,77,77,77,77,77,76,76,76,76,76,76,76,76,76,75,75,75,75,75,
    75,75,75,74,74,74,74,73,73,73,73,72,72,72,72,72,72,72,72,72,72,
    72,72,71,71,71,71,71,71,71,71,70,70,70,70,70,70,70,70,70,69,69,
    69,69,69,68,68,68,68,68,68,68,68,67,67,67,67,67,67,67,66,66,66,
    66,66,66,66,66,66,66,65,65,65,65,65,65,65,65,65,64,64,64,64,64,
    63,63,63,63,63,63,63,62,62,62,62,62,62,62,62,62,62,61,61,61,61,
    61,61,61,61,60,60,60,60,60,60,60,60,60,60,60,59,59,59,59,59,59,
    58,58,58,58,58,58,57,57,57,56,56,56,56,56,56,55,55,55,55,55,55,
    55,54,54,54,54,53,53,53,53,53,53,53,53,52,52,52,52,52,52,52,52,
    51,51,51,51,51,51,51,51,51,50,50,50,50,50,50,50,49,49,49,49,49,
    49,49,49,49,49,48,48,48,48,48,47,47,47,47,47,47,47,46,46,46,46,
    46,46,45,45,45,45,45,45,45,44,44,43,43,43,43,43,43,43,43,42,42,
    42,42,42,42,42,41,41,41,41,41,41,41,41,41,41,41,41,40,40,40,40,
    40,40,39,39,39,39,39,39,39,39,38,38,38,38,38,38,37,37,37,37,37,
    36,36,36,36,36,35,35,35,35,35,35,34,34,34,34,34,34,33,33,33,33,
    33,33,32,32,32,32,31,31,31,31,31,30,30,30,30,30,30,30
  };
  const int n4c3w4_r[] = {
    150, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,100,99,99,99,99,99,99,98,98,98,98,98,98,98,
    98,97,97,97,97,97,97,97,97,97,97,96,96,96,96,96,96,96,96,95,95,
    95,95,95,95,94,94,94,94,94,94,94,93,93,93,92,92,92,92,92,92,92,
    92,92,91,91,91,91,91,91,90,90,90,90,90,90,90,90,89,89,89,89,89,
    89,89,89,88,88,88,88,88,88,87,87,87,87,87,86,86,86,86,86,86,85,
    85,85,85,85,84,84,84,84,84,84,84,84,83,83,83,83,83,83,83,83,82,
    82,82,82,82,82,81,81,81,81,81,80,80,80,80,80,80,80,80,80,79,79,
    79,79,78,78,78,78,78,78,78,78,78,78,78,78,78,77,77,77,77,77,77,
    77,77,77,77,77,76,76,76,76,76,75,75,75,75,75,75,74,74,74,74,74,
    74,73,73,73,73,73,73,73,73,73,73,72,72,72,72,72,72,72,71,71,71,
    71,70,70,70,70,70,70,69,69,69,69,69,68,68,68,68,68,67,67,67,67,
    67,67,66,66,66,66,66,65,65,65,65,65,64,64,64,64,63,63,63,63,63,
    63,63,63,63,63,62,62,62,62,62,62,62,62,61,60,60,60,60,60,60,60,
    59,59,59,59,59,59,58,58,58,58,58,58,58,57,57,57,57,57,57,57,57,
    56,56,56,56,56,56,56,56,56,55,55,55,55,55,55,54,54,54,54,54,53,
    53,53,53,53,53,53,53,52,52,52,52,52,52,52,51,51,51,51,51,51,50,
    50,50,50,49,49,49,49,49,49,48,48,48,48,48,48,48,48,48,48,48,47,
    47,47,47,47,47,47,46,46,46,46,46,46,45,45,45,45,45,44,44,44,44,
    44,44,43,43,43,43,43,43,43,43,43,43,43,42,42,42,42,42,42,42,42,
    41,41,41,41,41,41,40,40,40,40,40,40,40,40,40,40,40,40,40,39,39,
    39,39,38,38,38,38,38,38,38,38,38,38,38,38,37,37,37,37,37,37,37,
    37,37,37,37,36,36,36,36,36,36,36,36,36,35,35,35,35,35,35,35,35,
    34,34,34,34,34,34,34,34,34,34,34,34,33,33,33,33,33,33,32,32,32,
    32,32,32,32,31,31,31,31,31,31,31,31,31,30,30,30,30,30,30
  };
  const int n4c3w4_s[] = {
    150, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,100,100,99,99,99,99,99,99,99,98,98,98,98,
    98,98,97,97,97,97,96,96,96,96,96,96,96,95,95,94,94,94,94,94,94,
    94,94,94,94,93,93,93,93,93,93,93,93,93,93,93,93,93,92,92,92,92,
    92,92,92,91,91,91,91,91,91,91,90,90,90,90,90,90,89,89,89,89,89,
    88,88,88,88,88,88,88,88,87,87,87,87,87,87,87,86,86,86,86,86,86,
    86,86,86,85,85,85,85,85,85,84,84,84,83,83,83,83,83,83,82,82,82,
    82,82,82,82,82,82,81,81,81,81,81,80,80,80,80,80,80,80,80,80,79,
    79,79,79,79,79,79,78,78,78,78,77,77,77,77,77,76,76,76,76,76,76,
    76,76,75,75,75,75,75,75,75,75,75,74,74,74,74,74,74,74,74,73,73,
    73,73,73,73,73,73,73,73,73,73,73,72,72,72,72,72,71,71,71,71,71,
    71,71,71,70,70,70,70,70,69,69,69,69,69,69,69,69,69,68,68,68,68,
    68,68,68,68,68,68,67,67,67,67,66,66,66,66,66,66,66,66,65,65,65,
    65,65,65,65,65,64,64,64,64,64,64,64,63,63,63,63,63,63,63,62,62,
    62,62,62,62,62,61,61,61,61,61,61,60,60,60,60,60,60,59,59,59,59,
    59,59,59,58,58,58,58,58,58,57,57,57,57,57,56,56,56,56,56,56,56,
    56,55,55,55,55,55,55,55,55,55,55,55,54,54,54,54,54,54,54,54,54,
    53,53,53,52,52,52,52,52,52,51,51,51,51,51,51,51,51,51,51,50,50,
    50,50,50,50,49,49,49,49,49,49,49,49,49,49,48,48,48,48,48,48,47,
    47,46,46,46,46,45,45,45,45,45,45,45,45,45,45,45,45,45,45,45,44,
    44,44,44,44,44,44,43,43,43,43,43,43,43,43,43,42,42,42,42,42,41,
    41,41,41,41,41,41,40,40,40,40,40,40,40,40,39,39,39,39,38,38,38,
    38,37,37,37,37,37,37,37,36,36,36,36,36,36,36,36,35,35,35,35,35,
    35,35,35,34,34,34,34,34,34,34,34,34,33,33,33,33,33,32,32,32,32,
    32,32,32,32,32,31,31,31,31,31,31,31,31,31,31,31,31,31,30,30
  };
  const int n4c3w4_t[] = {
    150, // Capacity
    500, // Number of items
    // Size of items (sorted)
    100,100,100,100,100,99,99,99,99,99,99,99,99,99,98,98,98,98,98,
    98,97,97,97,97,97,97,97,96,96,96,96,96,96,96,96,95,95,95,95,95,
    95,95,95,94,94,94,94,94,93,93,93,93,93,93,93,92,92,92,92,92,92,
    91,91,91,91,90,90,90,90,90,90,90,90,90,90,90,90,90,90,90,90,90,
    89,89,88,88,88,88,88,88,88,88,88,88,88,88,87,87,87,87,87,86,86,
    86,86,86,86,86,86,85,85,85,85,85,85,85,85,84,84,84,83,83,82,82,
    82,82,82,82,82,82,82,82,82,82,82,81,81,81,81,81,81,81,80,80,80,
    80,80,79,79,79,79,78,78,78,77,77,77,77,77,77,77,77,76,76,76,76,
    75,75,75,75,75,75,75,74,74,74,74,74,74,74,74,74,74,74,73,73,73,
    73,73,73,73,73,73,72,72,72,72,72,72,72,72,71,71,71,71,71,71,71,
    70,70,70,70,70,70,69,69,69,69,69,69,69,69,68,68,68,68,68,68,68,
    68,67,67,67,67,67,67,67,67,67,67,66,66,66,66,66,66,66,65,65,65,
    65,65,65,64,64,64,64,64,64,64,63,63,63,63,63,63,63,63,62,62,62,
    62,62,61,61,61,60,60,60,60,60,60,60,59,59,59,59,59,59,59,59,58,
    58,58,58,57,57,57,57,56,56,56,56,56,56,56,56,56,55,55,55,55,55,
    55,55,55,55,54,54,54,54,54,54,54,53,53,53,53,53,52,52,52,52,52,
    52,52,52,51,51,51,51,51,51,51,51,50,50,50,50,49,49,49,49,49,49,
    49,48,48,48,48,48,48,48,48,47,47,47,47,47,46,46,46,46,46,46,46,
    46,45,45,45,45,45,44,44,44,44,44,44,44,44,44,44,44,44,43,43,43,
    43,43,43,42,42,42,42,42,42,42,42,41,41,41,41,41,41,40,40,40,40,
    40,39,39,39,39,39,38,38,38,38,38,38,38,38,38,38,38,37,37,37,37,
    37,37,37,37,37,37,36,36,36,36,36,36,35,35,35,35,35,35,35,35,35,
    35,35,34,34,34,34,34,34,34,34,33,33,33,33,33,33,33,32,32,32,32,
    32,32,32,32,32,32,32,31,31,31,30,30,30,30,30,30,30,30,30
  };

  /*
   * Data set 2
   *
   */
  const int n1w1b1r0[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    395,394,394,391,390,389,388,384,383,382,380,379,376,371,368,365,
    360,360,354,350,346,346,344,342,340,335,335,333,330,330,328,327,
    317,316,311,310,310,306,300,300,297,296,295,294,294,286,285,278,
    275,275
  };
  const int n1w1b1r1[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    392,392,391,390,390,388,386,382,381,380,380,380,375,375,375,374,
    373,372,370,364,360,360,359,355,346,345,343,341,332,320,317,317,
    314,313,311,308,307,305,303,296,294,290,283,282,280,274,273,272,
    269,267
  };
  const int n1w1b1r2[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    396,393,392,389,389,385,383,383,381,380,380,380,379,378,376,369,
    367,363,361,361,358,358,357,357,355,353,346,343,341,337,336,335,
    334,333,329,323,321,312,311,302,295,295,293,292,291,288,280,279,
    274,271
  };
  const int n1w1b1r3[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    390,389,388,384,382,381,377,377,377,375,375,373,364,363,363,362,
    357,357,353,347,344,341,337,336,336,335,334,333,333,332,332,326,
    323,319,314,311,309,307,306,301,301,297,295,293,292,292,290,284,
    280,278
  };
  const int n1w1b1r4[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    396,394,388,381,380,378,377,377,372,363,359,358,358,358,353,352,
    352,350,350,349,346,340,337,333,332,328,326,323,319,317,313,312,
    309,298,297,295,295,294,286,285,285,282,281,280,278,278,276,275,
    274,271
  };
  const int n1w1b1r5[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    394,392,391,386,383,382,380,370,369,368,368,365,356,356,355,354,
    348,342,339,338,337,335,333,333,332,326,326,326,324,321,321,318,
    317,312,305,304,303,302,299,291,287,281,281,279,278,278,274,274,
    267,266
  };
  const int n1w1b1r6[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    396,394,394,392,387,387,384,367,366,365,364,363,362,361,358,356,
    351,350,346,340,339,337,335,333,332,332,328,327,324,323,323,322,
    320,317,314,312,310,308,307,306,306,304,303,299,295,292,288,283,
    282,277
  };
  const int n1w1b1r7[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    396,395,394,391,389,388,382,381,380,379,376,371,366,366,365,364,
    359,356,353,348,346,345,343,336,335,335,327,325,320,320,320,308,
    306,302,299,297,295,294,290,286,285,283,281,280,277,275,272,270,
    269,269
  };
  const int n1w1b1r8[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    396,394,391,390,390,389,386,382,380,379,378,377,377,369,368,361,
    359,358,357,356,353,350,348,345,341,340,333,332,328,327,322,319,
    315,306,305,305,304,304,300,300,294,293,291,285,280,279,274,271,
    269,266
  };
  const int n1w1b1r9[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    394,393,391,385,384,377,373,371,370,366,365,364,359,359,359,358,
    357,356,352,348,346,346,324,324,323,323,323,321,320,317,316,315,
    310,300,296,295,295,291,289,288,287,285,283,282,281,280,280,280,
    274,269
  };
  const int n1w1b2r0[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    494,489,481,470,468,467,443,442,440,437,434,418,404,401,400,393,
    374,371,363,362,361,355,353,351,349,347,337,333,328,322,321,315,
    283,260,257,255,255,246,237,231,224,212,211,205,191,186,184,182,
    174,173
  };
  const int n1w1b2r1[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    483,476,471,455,443,441,434,434,426,426,421,417,408,397,395,394,
    389,380,380,378,375,373,357,340,325,319,318,310,304,292,291,277,
    275,271,265,265,263,244,240,224,218,214,202,202,198,195,189,184,
    181,169
  };
  const int n1w1b2r2[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    492,489,483,482,481,455,452,448,443,439,438,423,419,410,405,389,
    386,381,374,367,366,361,357,348,322,316,300,293,292,285,283,279,
    279,276,271,264,254,249,241,231,226,223,220,201,193,192,189,182,
    178,170
  };
  const int n1w1b2r3[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    490,489,485,473,456,444,436,428,424,420,409,407,395,384,382,376,
    372,370,360,358,340,338,338,335,326,319,305,302,293,291,287,271,
    262,256,249,248,245,231,203,198,196,194,194,194,182,182,171,169,
    169,168
  };
  const int n1w1b2r4[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    492,491,485,480,467,463,458,455,451,446,437,422,421,416,409,406,
    404,387,385,379,354,343,336,332,323,316,309,301,290,288,284,281,
    275,255,253,244,243,229,227,223,223,215,214,211,208,203,203,185,
    176,167
  };
  const int n1w1b2r5[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    489,488,473,468,459,450,443,434,429,417,415,404,393,379,376,376,
    375,372,363,362,360,359,348,348,343,341,338,334,334,332,324,301,
    291,289,288,270,268,255,255,242,228,228,227,218,203,196,195,181,
    179,173
  };
  const int n1w1b2r6[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    478,469,466,465,444,439,436,434,433,429,428,418,398,395,387,387,
    386,385,376,374,360,355,349,345,341,340,330,324,320,299,279,278,
    264,260,257,249,247,241,237,219,215,205,199,196,193,191,187,185,
    182,175
  };
  const int n1w1b2r7[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    495,492,489,488,487,487,486,475,473,469,469,463,455,454,452,432,
    430,404,401,396,396,377,368,352,344,341,321,311,309,288,285,282,
    275,274,266,256,252,245,244,238,227,226,213,207,203,203,197,196,
    170,168
  };
  const int n1w1b2r8[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    491,473,468,467,449,447,444,422,420,410,408,402,392,385,378,377,
    358,358,356,342,334,329,327,322,319,314,306,303,296,279,264,263,
    263,263,252,250,244,235,230,228,217,217,210,206,190,185,182,175,
    172,168
  };
  const int n1w1b2r9[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    489,489,486,484,478,475,463,460,460,452,447,447,436,432,432,429,
    427,426,420,419,382,369,367,356,341,336,329,324,311,304,302,283,
    283,274,271,271,267,262,261,258,243,236,225,223,218,203,202,200,
    186,186
  };
  const int n1w1b3r0[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    627,600,598,588,551,543,536,518,509,503,487,484,472,468,463,461,
    424,417,405,401,397,369,369,356,340,339,324,304,272,269,250,225,
    217,183,168,162,156,155,147,132,125,117,115,114,114,95,77,71,
    69,48
  };
  const int n1w1b3r1[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    626,618,617,606,588,561,558,530,526,523,518,500,496,486,483,476,
    472,463,459,452,424,374,346,345,319,318,303,296,278,276,257,238,
    236,216,211,193,181,171,164,161,159,157,128,115,114,108,108,82,
    38,35
  };
  const int n1w1b3r2[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    624,617,601,599,583,553,513,484,478,468,466,465,462,421,410,403,
    370,368,358,353,347,325,321,318,281,262,253,237,215,201,194,184,
    183,173,159,158,148,140,133,123,116,87,84,81,78,77,74,57,51,46
  };
  const int n1w1b3r3[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    623,596,581,568,568,563,544,517,481,478,467,444,428,408,398,387,
    382,378,364,363,357,356,353,343,341,330,304,300,260,252,252,252,
    239,221,217,195,178,163,156,153,147,144,143,143,138,137,127,78,
    68,59
  };
  const int n1w1b3r4[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    627,626,604,580,565,546,540,524,517,509,506,489,485,481,476,472,
    446,441,426,411,410,407,404,390,385,379,374,368,364,354,351,345,
    316,303,300,287,282,232,203,197,166,153,137,136,124,120,111,99,
    96,88
  };
  const int n1w1b3r5[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    627,611,609,607,559,554,550,525,517,508,484,481,476,475,457,438,
    427,425,414,407,401,391,369,352,334,330,314,295,235,234,232,208,
    195,175,168,154,145,113,107,103,100,97,90,82,77,70,55,52,43,39
  };
  const int n1w1b3r6[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    614,600,591,569,557,536,518,515,514,507,504,498,476,460,436,425,
    418,411,408,380,344,322,313,313,299,274,273,243,231,218,210,204,
    198,176,171,167,134,121,119,112,99,94,83,74,61,56,56,53,52,38
  };
  const int n1w1b3r7[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    603,599,578,556,539,532,531,524,522,522,520,520,514,514,495,492,
    478,471,458,457,457,445,439,434,433,413,374,364,338,333,320,300,
    284,278,205,199,197,194,190,179,161,157,154,130,122,118,97,85,
    69,37
  };
  const int n1w1b3r8[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    611,561,544,528,521,472,470,462,458,439,434,432,426,424,412,375,
    373,365,363,359,350,348,344,344,341,313,310,309,301,294,290,279,
    260,245,221,219,211,206,203,199,198,145,124,112,110,82,78,69,
    66,39
  };
  const int n1w1b3r9[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    607,597,582,581,571,552,550,543,532,499,491,482,477,458,453,449,
    419,417,412,403,394,392,385,363,343,339,299,299,290,286,283,269,
    256,250,237,229,192,162,146,115,105,104,103,90,87,73,72,70,55,
    38
  };
  const int n1w2b1r0[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    239,236,235,234,232,232,230,230,230,230,228,226,225,223,220,218,
    217,217,216,215,214,213,213,210,210,209,209,206,206,205,205,198,
    197,196,196,196,196,192,189,186,184,180,176,174,172,167,164,164,
    164,163
  };
  const int n1w2b1r1[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    240,239,238,235,234,234,233,232,232,232,230,228,226,226,226,224,
    220,215,215,214,214,210,209,209,207,206,205,201,198,197,195,194,
    191,191,185,183,181,181,181,178,177,176,176,174,171,171,171,170,
    168,168
  };
  const int n1w2b1r2[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    239,237,237,235,234,232,231,231,231,228,224,224,221,220,218,217,
    216,214,212,210,208,208,202,199,198,198,197,193,193,191,189,189,
    185,184,184,183,181,179,177,176,176,175,174,173,172,171,171,164,
    162,162
  };
  const int n1w2b1r3[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    239,238,237,237,235,234,233,232,231,231,230,228,224,224,222,222,
    221,220,218,216,214,214,210,206,205,204,202,202,200,199,198,198,
    197,197,197,192,191,186,185,184,184,181,180,173,173,173,167,166,
    165,164
  };
  const int n1w2b1r4[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    240,239,239,237,237,233,233,232,231,228,228,227,227,226,225,225,
    225,225,221,220,220,214,214,214,210,209,206,206,205,202,202,200,
    198,198,198,198,197,192,190,185,184,177,176,175,171,170,167,166,
    163,162
  };
  const int n1w2b1r5[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    240,237,235,234,233,232,231,227,224,224,223,217,215,213,213,212,
    210,206,205,205,204,204,203,202,201,201,200,199,193,190,189,186,
    185,183,181,180,178,173,171,169,169,169,168,166,166,166,165,165,
    164,163
  };
  const int n1w2b1r6[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    240,238,237,237,236,234,231,225,225,224,221,220,220,218,217,215,
    214,212,209,209,202,201,200,200,199,197,197,197,197,196,195,193,
    189,189,187,187,185,182,180,180,179,178,177,175,170,169,169,168,
    167,163
  };
  const int n1w2b1r7[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    240,239,238,238,237,236,234,232,228,226,225,222,218,215,213,211,
    210,210,206,204,203,203,203,202,201,200,199,197,196,196,195,188,
    188,188,187,186,185,184,182,181,180,178,177,175,169,167,166,164,
    164,163
  };
  const int n1w2b1r8[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    240,240,240,239,238,238,237,231,229,228,228,221,219,218,216,213,
    209,209,206,202,202,202,201,201,199,197,197,196,190,189,189,186,
    184,184,181,178,178,176,176,174,174,174,168,168,167,164,164,164,
    163,163
  };
  const int n1w2b1r9[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    240,240,239,239,238,237,236,234,233,231,228,228,223,223,222,219,
    218,218,215,213,212,211,209,204,198,197,196,195,188,186,185,185,
    184,182,182,182,181,179,178,178,178,177,176,173,170,165,165,162,
    162,162
  };
  const int n1w2b2r0[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    299,295,295,287,278,277,271,269,264,258,253,241,241,232,230,228,
    226,221,213,212,211,210,203,202,200,198,197,194,172,172,170,167,
    163,158,156,149,149,145,140,139,137,135,127,126,120,114,113,111,
    109,102
  };
  const int n1w2b2r1[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    297,288,285,281,279,275,274,269,268,268,267,266,262,250,244,243,
    241,241,238,230,229,226,220,219,218,203,202,201,201,201,189,188,
    188,188,180,180,179,176,162,158,156,150,146,120,116,112,111,109,
    104,102
  };
  const int n1w2b2r2[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    297,296,288,279,271,249,241,239,234,232,231,227,226,220,214,212,
    212,209,205,200,199,194,193,191,187,186,184,183,175,172,167,154,
    151,150,146,143,141,138,137,129,127,122,121,115,113,110,110,107,
    104,103
  };
  const int n1w2b2r3[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    297,297,294,280,277,270,270,269,260,255,255,254,252,250,241,237,
    223,222,221,217,216,211,209,209,206,204,193,192,192,191,187,182,
    173,172,166,165,161,160,149,148,146,139,135,131,130,125,118,116,
    111,102
  };
  const int n1w2b2r4[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    300,283,280,259,259,258,257,254,250,248,246,244,242,239,237,236,
    225,222,212,206,205,205,203,201,193,190,188,185,185,185,182,179,
    178,174,174,161,157,153,150,141,141,133,124,123,122,121,117,110,
    106,103
  };
  const int n1w2b2r5[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    299,295,295,290,286,283,282,276,268,259,254,251,245,242,242,240,
    236,234,231,223,217,214,208,205,200,183,181,179,172,171,169,165,
    159,153,152,150,149,147,144,142,135,135,134,126,125,124,114,113,
    106,105
  };
  const int n1w2b2r6[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    295,295,292,288,280,279,274,266,255,253,252,249,246,242,225,223,
    217,212,210,209,203,200,190,188,173,172,171,165,164,163,158,157,
    153,147,146,144,143,143,141,141,139,138,134,121,120,114,108,105,
    104,103
  };
  const int n1w2b2r7[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    295,285,276,275,270,268,266,265,257,254,246,242,242,241,241,236,
    231,231,229,224,223,216,215,209,207,200,195,194,178,177,177,159,
    150,149,146,143,143,141,139,139,136,131,130,125,116,115,113,113,
    103,102
  };
  const int n1w2b2r8[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    298,298,298,297,293,293,291,285,283,278,277,272,270,264,258,250,
    246,236,232,231,230,229,225,219,216,216,215,211,208,193,192,190,
    181,175,173,172,170,149,149,141,135,132,130,120,119,115,113,109,
    107,105
  };
  const int n1w2b2r9[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    299,295,293,292,282,278,273,271,270,267,263,260,259,256,255,254,
    245,238,229,228,228,228,228,226,206,205,204,198,196,195,191,163,
    160,153,151,149,148,145,144,143,137,137,132,132,127,124,120,114,
    109,105
  };
  const int n1w2b3r0[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    367,358,357,344,340,335,329,326,320,316,307,307,300,289,274,270,
    244,225,225,216,212,208,200,193,190,186,186,167,166,163,157,156,
    152,142,138,134,134,131,107,79,79,79,77,73,41,40,37,34,28,23
  };
  const int n1w2b3r1[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    376,355,355,350,336,327,314,308,308,300,299,297,296,277,275,264,
    263,251,247,247,246,245,225,217,198,191,186,184,183,181,173,161,
    157,153,137,133,121,109,108,107,93,80,80,76,76,74,69,67,44,26
  };
  const int n1w2b3r2[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    370,366,354,352,348,342,341,335,334,329,326,323,320,316,312,310,
    302,270,264,247,231,217,217,202,183,181,180,150,141,136,135,135,
    131,131,126,120,119,111,78,70,62,60,56,55,52,46,40,38,34,30
  };
  const int n1w2b3r3[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    350,348,338,335,334,328,322,306,306,305,296,288,287,286,284,279,
    266,264,247,231,228,227,219,205,204,202,195,192,158,155,149,138,
    135,134,131,129,128,121,118,118,113,103,103,98,96,83,82,82,77,
    30
  };
  const int n1w2b3r4[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    374,372,342,328,313,313,293,290,283,282,280,244,243,234,233,227,
    226,223,218,200,190,179,179,178,174,169,168,162,159,158,153,153,
    152,129,126,121,119,114,111,93,85,82,67,67,54,49,46,36,25,25
  };
  const int n1w2b3r5[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    379,363,361,343,328,314,312,302,299,289,289,288,285,274,267,266,
    263,257,255,234,220,212,208,194,186,186,184,164,163,160,160,125,
    118,110,99,97,90,89,87,85,85,83,80,74,72,61,50,41,39,32
  };
  const int n1w2b3r6[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    375,360,360,355,342,331,325,321,305,299,296,294,292,288,262,257,
    241,235,234,231,231,229,229,215,210,210,209,207,190,182,174,172,
    163,163,161,159,141,135,125,106,102,89,87,72,58,46,34,34,29,27
  };
  const int n1w2b3r7[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    375,365,363,356,351,349,338,324,314,304,290,286,273,267,253,241,
    240,238,223,220,219,213,211,208,193,182,167,139,133,132,132,131,
    128,124,103,94,86,78,75,74,73,66,60,56,49,49,46,44,35,30
  };
  const int n1w2b3r8[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    370,364,361,326,323,323,319,310,303,300,289,284,278,267,257,244,
    244,240,236,232,228,225,224,222,221,204,184,183,182,181,180,180,
    179,177,173,170,143,140,136,131,125,121,93,87,80,67,64,59,37,
    23
  };
  const int n1w2b3r9[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    361,360,352,350,343,324,311,300,298,290,277,277,275,274,269,267,
    259,255,245,238,210,210,208,204,193,193,167,162,156,149,147,146,
    141,134,132,125,123,112,105,81,76,72,71,62,58,56,41,36,33,24
  };
  const int n1w3b1r0[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    167,167,164,160,158,158,158,158,157,152,152,150,150,149,149,148,
    146,144,144,144,142,142,141,137,137,136,135,134,133,133,133,133,
    131,129,129,127,125,125,124,124,124,123,123,123,122,122,121,121,
    119,118
  };
  const int n1w3b1r1[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    167,165,165,164,163,163,162,161,160,159,158,158,157,156,155,153,
    153,151,151,151,150,148,148,147,147,147,147,147,146,146,146,143,
    143,141,140,140,138,137,135,135,134,133,129,128,127,126,125,124,
    123,115
  };
  const int n1w3b1r2[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    168,167,166,165,165,162,162,161,160,157,155,155,153,151,149,148,
    148,144,144,144,143,141,141,141,140,139,137,136,134,134,133,133,
    132,131,131,131,128,127,127,125,125,123,122,121,119,118,116,116,
    115,114
  };
  const int n1w3b1r3[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    165,165,164,162,161,161,159,157,156,156,155,155,155,154,154,153,
    151,150,149,148,148,146,146,146,145,144,138,138,137,137,136,135,
    134,133,132,131,131,130,124,123,121,120,120,119,119,117,117,117,
    116,114
  };
  const int n1w3b1r4[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    168,166,166,166,165,164,163,161,160,160,158,157,156,152,152,151,
    148,148,147,146,144,144,143,141,139,139,139,135,134,133,133,133,
    132,131,129,129,128,127,125,123,120,119,118,118,117,117,116,116,
    116,115
  };
  const int n1w3b1r5[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    166,165,164,163,163,163,162,162,159,156,156,156,155,155,152,151,
    151,150,149,149,148,147,146,145,143,143,143,137,137,135,135,134,
    134,133,133,132,131,130,128,128,126,125,123,123,120,119,117,117,
    117,115
  };
  const int n1w3b1r6[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    168,168,167,167,163,163,162,161,160,158,158,158,157,156,156,156,
    156,155,154,154,153,152,151,151,149,149,148,145,143,142,142,142,
    140,139,138,136,134,132,131,128,126,124,121,120,120,120,116,115,
    114,114
  };
  const int n1w3b1r7[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    168,167,166,165,164,163,162,161,161,159,159,158,156,154,153,152,
    152,152,151,151,150,148,146,145,145,139,138,137,136,136,135,135,
    134,133,132,130,127,126,126,125,125,124,122,120,120,119,118,117,
    117,116
  };
  const int n1w3b1r8[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    168,166,164,162,161,161,160,159,157,155,155,155,155,154,153,152,
    151,148,148,146,144,144,144,143,142,141,140,137,136,135,132,131,
    131,130,130,128,124,123,123,122,122,121,121,120,119,118,117,116,
    115,114
  };
  const int n1w3b1r9[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    168,167,165,164,164,163,162,160,158,154,153,152,150,150,149,148,
    147,147,146,144,144,143,142,142,141,141,140,139,136,135,135,134,
    133,133,131,129,129,128,128,127,121,121,120,120,120,119,118,117,
    116,115
  };
  const int n1w3b2r0[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    210,202,202,198,195,194,190,190,189,186,181,179,179,178,173,169,
    168,166,165,165,158,148,146,143,140,137,137,135,133,129,126,121,
    119,117,115,114,113,113,111,109,108,106,104,103,93,91,81,81,74,
    74
  };
  const int n1w3b2r1[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    204,203,203,202,201,194,192,189,186,186,182,182,181,180,179,179,
    176,174,172,171,163,161,155,154,154,151,147,146,144,140,134,132,
    132,132,126,117,117,108,106,105,101,92,92,90,89,88,86,85,78,77
  };
  const int n1w3b2r2[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    208,203,203,201,193,193,191,190,189,172,169,168,166,165,165,162,
    161,161,159,156,156,153,152,150,147,145,145,142,141,138,138,138,
    128,121,119,118,113,110,109,107,106,101,101,97,91,84,83,74,74,
    73
  };
  const int n1w3b2r3[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    204,202,199,199,195,192,191,190,187,181,172,169,169,166,163,163,
    163,160,157,153,152,150,143,142,140,139,132,127,125,124,123,121,
    119,116,113,108,108,107,98,95,95,94,90,90,88,86,82,81,80,78
  };
  const int n1w3b2r4[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    207,192,192,190,187,187,186,181,179,177,175,170,167,163,162,148,
    148,148,147,147,133,132,131,130,130,129,127,125,122,119,118,114,
    114,109,109,106,106,105,104,102,101,96,96,94,90,90,90,89,85,78
  };
  const int n1w3b2r5[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    205,201,200,200,189,187,180,177,173,170,169,167,166,162,160,151,
    151,146,145,144,143,143,142,142,141,139,137,137,131,130,125,122,
    120,120,119,116,107,104,95,92,91,90,88,85,84,83,83,79,76,73
  };
  const int n1w3b2r6[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    208,207,206,203,202,199,197,196,192,189,189,176,175,175,175,174,
    171,170,167,164,164,158,156,156,154,153,152,150,148,143,141,134,
    132,130,125,119,117,106,103,92,89,88,84,81,76,75,73,73,72,72
  };
  const int n1w3b2r7[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    210,207,205,204,203,202,201,192,191,190,187,185,184,183,181,178,
    177,175,172,172,171,170,169,162,156,143,143,142,136,135,135,135,
    129,124,122,119,116,112,97,95,92,89,87,81,80,78,75,74,73,72
  };
  const int n1w3b2r8[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    210,201,195,193,192,190,189,180,178,177,175,174,173,172,170,170,
    167,166,166,165,164,163,162,159,159,158,156,148,147,145,143,136,
    129,121,119,117,116,111,111,108,101,96,90,82,80,80,76,74,72,72
  };
  const int n1w3b2r9[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    208,205,204,204,202,196,190,190,188,185,182,181,175,169,166,164,
    163,162,158,158,156,155,154,152,150,149,145,142,139,139,129,128,
    123,119,113,102,102,95,93,92,90,89,86,84,81,80,80,75,75,73
  };
  const int n1w3b3r0[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    265,257,251,250,246,242,221,218,217,217,207,203,180,176,172,167,
    162,162,160,156,145,141,140,135,132,132,129,126,121,116,113,112,
    109,108,105,102,100,92,87,82,76,61,51,46,45,37,36,32,18,17
  };
  const int n1w3b3r1[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    251,249,247,241,235,227,222,215,207,207,203,199,198,196,195,185,
    179,179,175,174,171,168,163,159,159,155,150,149,148,148,130,124,
    119,112,109,105,100,95,89,72,68,64,58,57,55,51,45,27,26,21
  };
  const int n1w3b3r2[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    266,265,257,245,240,238,236,228,220,205,202,194,188,184,179,169,
    164,163,159,156,154,153,145,143,135,134,130,127,115,109,100,88,
    79,68,60,59,58,57,56,53,51,47,45,45,43,41,41,32,32,19
  };
  const int n1w3b3r3[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    254,248,246,238,237,223,221,219,219,217,215,208,208,208,202,198,
    194,189,184,180,177,176,166,166,165,163,152,146,142,138,125,123,
    115,114,113,110,96,94,88,88,86,78,67,56,43,35,34,32,25,16
  };
  const int n1w3b3r4[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    261,259,259,257,249,244,236,231,229,228,206,204,195,182,180,175,
    172,170,169,165,161,160,156,155,153,148,147,147,146,131,115,113,
    110,109,102,93,89,89,85,82,78,77,68,66,59,49,40,37,26,23
  };
  const int n1w3b3r5[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    259,252,249,240,235,216,199,194,189,177,175,172,170,170,167,167,
    165,164,154,152,147,145,144,140,132,123,120,116,116,112,111,111,
    108,95,79,75,75,71,66,64,55,52,50,49,49,47,35,22,19,19
  };
  const int n1w3b3r6[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    261,260,257,251,250,231,229,224,222,214,210,202,195,191,191,190,
    189,175,165,160,159,157,156,146,139,137,133,132,132,126,123,119,
    119,105,97,89,79,76,76,74,68,59,42,39,33,27,23,22,19,17
  };
  const int n1w3b3r7[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    266,265,259,258,258,242,240,235,229,227,218,213,211,206,204,199,
    197,190,180,173,169,168,162,153,153,151,149,147,141,138,136,136,
    130,122,120,118,94,90,88,87,75,65,61,45,43,27,27,25,22,22
  };
  const int n1w3b3r8[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    254,250,247,244,243,235,235,226,225,225,216,204,189,188,184,166,
    159,139,135,133,130,126,121,119,118,114,108,104,102,94,93,89,
    88,88,75,75,65,57,54,47,47,45,44,39,33,33,28,23,20,16
  };
  const int n1w3b3r9[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    265,262,259,251,251,249,244,243,234,233,227,224,200,200,195,189,
    182,175,173,167,160,159,141,126,125,124,123,123,121,114,112,111,
    103,100,95,72,70,65,55,49,49,44,36,28,25,25,24,20,19,16
  };
  const int n1w4b1r0[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    131,131,131,131,130,130,128,128,127,125,125,125,121,119,119,119,
    118,117,116,113,111,110,109,109,108,108,106,106,105,104,104,103,
    103,102,101,101,100,99,98,96,95,93,92,91,91,90,90,90,90,90
  };
  const int n1w4b1r1[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    132,131,131,130,130,129,128,128,127,127,127,126,124,122,122,122,
    121,120,120,119,118,116,116,116,116,116,114,113,111,110,108,107,
    104,104,101,101,99,97,95,95,95,94,93,92,92,92,92,91,91,91
  };
  const int n1w4b1r2[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    132,132,132,131,130,129,128,126,124,123,123,123,122,121,120,119,
    119,118,118,118,118,115,113,113,110,109,108,108,107,104,103,102,
    102,100,100,99,98,98,96,95,95,95,94,94,94,93,92,92,91,90
  };
  const int n1w4b1r3[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    132,132,131,130,130,127,124,124,123,122,122,121,121,120,119,119,
    118,118,117,117,113,112,111,110,110,110,109,109,109,106,105,103,
    103,103,101,101,98,98,98,97,97,97,97,96,95,94,94,92,91,91
  };
  const int n1w4b1r4[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    130,129,129,128,128,126,126,125,124,124,124,122,121,121,121,120,
    120,119,119,116,114,114,114,114,112,112,111,110,109,107,107,103,
    102,101,101,101,101,101,100,100,99,97,97,96,95,94,93,92,92,90
  };
  const int n1w4b1r5[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    132,132,132,131,129,127,127,125,125,123,122,121,120,118,116,116,
    115,115,115,113,112,111,110,108,107,106,105,105,105,104,103,102,
    102,101,99,99,99,98,97,96,96,95,94,93,93,93,92,92,91,90
  };
  const int n1w4b1r6[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    131,131,131,128,127,126,126,124,123,122,122,120,119,118,118,117,
    117,116,115,115,114,114,113,112,111,110,110,109,107,107,107,106,
    104,104,103,103,101,99,97,94,94,93,92,92,92,90,90,90,90,90
  };
  const int n1w4b1r7[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    132,130,130,130,130,130,128,128,127,126,126,124,124,122,121,120,
    118,117,115,113,112,112,112,111,111,111,111,110,109,109,108,108,
    105,105,105,101,100,99,99,98,96,95,94,94,94,93,92,92,92,90
  };
  const int n1w4b1r8[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    131,131,128,127,127,126,124,123,123,122,120,119,119,115,113,113,
    112,112,112,111,110,109,109,108,105,105,103,102,102,102,102,101,
    99,99,99,97,97,97,96,96,96,94,94,94,94,93,92,92,91,90
  };
  const int n1w4b1r9[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    132,130,130,128,125,124,123,121,121,121,120,119,117,116,116,115,
    113,112,111,111,111,110,110,109,109,107,107,106,106,105,104,102,
    102,101,101,100,99,98,97,96,96,95,95,94,92,92,92,91,91,90
  };
  const int n1w4b2r0[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    165,164,161,158,157,155,154,153,153,149,144,144,140,138,138,138,
    137,134,133,133,131,128,124,120,119,117,117,115,112,111,107,107,
    104,97,90,85,83,80,79,78,76,76,70,68,66,65,65,59,57,57
  };
  const int n1w4b2r1[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    163,156,155,154,152,151,150,149,146,137,136,128,126,125,122,122,
    121,121,117,114,113,106,103,99,98,96,93,83,80,80,79,78,78,76,
    74,71,70,69,68,68,68,67,67,67,64,59,59,59,59,58
  };
  const int n1w4b2r2[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    165,163,161,157,152,150,146,144,141,137,136,135,135,134,133,130,
    122,120,118,117,116,112,111,108,105,104,100,97,96,95,94,91,89,
    89,86,85,82,81,80,79,77,70,70,68,65,61,60,60,57,57
  };
  const int n1w4b2r3[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    165,164,164,159,155,155,155,150,146,141,138,138,137,135,131,130,
    130,127,126,125,122,122,121,120,119,119,118,114,113,112,111,108,
    104,104,100,97,96,89,83,79,76,75,75,73,70,67,65,64,62,60
  };
  const int n1w4b2r4[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    163,162,162,161,159,155,148,148,145,141,140,139,137,135,133,130,
    130,123,122,122,120,117,117,115,113,113,111,111,111,109,105,105,
    98,98,97,94,91,87,82,80,77,76,73,72,69,65,64,64,63,60
  };
  const int n1w4b2r5[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    165,165,164,163,162,156,155,154,153,152,152,149,148,143,140,137,
    135,134,129,128,128,126,124,120,119,119,118,118,116,115,108,106,
    105,101,98,97,97,96,94,89,85,82,79,77,76,75,67,65,64,58
  };
  const int n1w4b2r6[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    164,164,161,154,154,153,152,146,144,134,132,132,130,130,130,127,
    125,124,123,123,120,119,116,115,114,111,110,109,108,105,105,103,
    101,98,90,87,85,83,83,82,80,79,76,75,75,74,67,67,65,60
  };
  const int n1w4b2r7[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    162,159,157,150,148,145,136,136,135,133,133,132,128,126,126,125,
    121,120,120,116,114,113,110,106,105,103,100,100,97,96,92,92,88,
    83,78,78,75,75,75,75,73,65,65,65,64,64,58,57,57,57
  };
  const int n1w4b2r8[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    165,165,164,157,156,155,155,154,150,150,150,149,147,145,142,142,
    139,137,137,136,134,131,127,126,124,122,121,116,115,112,111,109,
    108,107,101,98,97,94,91,91,89,86,86,84,81,71,69,64,61,59
  };
  const int n1w4b2r9[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    163,158,156,154,153,153,148,142,131,130,128,126,125,119,117,117,
    117,116,114,111,110,109,106,105,104,101,100,100,99,98,97,96,95,
    93,89,86,86,81,80,78,78,78,75,72,72,71,65,65,59,58
  };
  const int n1w4b3r0[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    209,199,199,196,192,191,190,175,175,172,166,160,158,151,149,148,
    140,135,134,126,121,113,113,103,94,94,93,87,84,82,77,69,67,64,
    60,60,60,54,52,45,37,35,32,23,22,21,19,18,14,13
  };
  const int n1w4b3r1[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    209,204,184,183,179,170,169,167,167,166,163,163,160,157,152,150,
    148,142,139,133,132,132,127,125,125,123,116,111,104,95,92,89,
    86,79,76,74,70,65,62,60,45,43,37,30,29,29,25,22,15,13
  };
  const int n1w4b3r2[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    209,207,206,206,204,190,189,188,188,186,186,181,180,180,178,178,
    177,175,171,157,156,153,138,136,135,134,133,128,123,98,98,97,
    87,83,79,77,77,71,70,65,62,62,58,53,43,39,37,37,34,14
  };
  const int n1w4b3r3[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    204,195,192,192,190,188,184,178,176,170,157,155,148,146,138,135,
    132,128,124,124,115,114,113,107,95,94,92,91,84,83,82,80,79,77,
    76,76,75,69,68,64,60,59,58,52,50,38,33,22,19,15
  };
  const int n1w4b3r4[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    209,209,206,195,195,193,191,188,186,181,178,173,170,163,162,150,
    133,131,129,127,126,125,124,117,113,109,101,98,93,89,86,85,77,
    75,74,70,60,60,55,54,42,40,36,28,23,23,20,19,16,13
  };
  const int n1w4b3r5[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    206,203,201,197,196,184,177,176,174,174,173,168,164,162,161,160,
    159,153,152,152,146,146,146,138,136,131,129,125,123,111,107,105,
    103,93,79,79,79,73,70,61,59,55,52,44,37,33,32,31,26,18
  };
  const int n1w4b3r6[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    204,203,201,199,188,187,185,178,176,173,170,166,163,157,154,153,
    145,143,131,131,126,124,124,121,118,114,107,103,95,91,86,85,81,
    78,68,67,67,61,60,59,49,47,38,35,26,21,21,20,17,14
  };
  const int n1w4b3r7[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    208,204,203,202,202,197,185,182,177,173,166,164,157,157,150,146,
    137,127,126,125,124,120,113,112,109,93,92,88,88,84,82,79,78,72,
    71,55,44,43,42,40,36,35,33,32,28,25,25,24,17,14
  };
  const int n1w4b3r8[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    208,204,200,196,192,190,189,186,186,177,174,169,157,147,144,140,
    132,129,129,128,127,126,124,117,115,113,108,106,105,105,104,104,
    102,101,94,89,85,85,79,71,68,65,57,42,40,36,16,16,15,13
  };
  const int n1w4b3r9[] = {
    1000, // Capacity
    50, // Number of items
    // Size of items (sorted)
    207,206,205,193,187,173,170,168,167,166,165,162,160,156,150,145,
    145,143,139,138,135,132,128,125,124,117,114,114,112,111,108,103,
    100,93,88,83,79,69,65,65,58,57,46,45,42,42,36,32,25,25
  };
  const int n2w1b1r0[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    393,390,390,389,386,382,381,381,381,380,379,379,377,375,372,370,
    368,368,367,366,366,365,365,363,361,359,359,357,357,356,355,355,
    355,353,352,352,347,347,346,344,344,341,337,336,334,334,333,333,
    333,332,332,329,328,326,326,324,324,319,319,318,316,312,312,311,
    310,309,307,306,305,305,301,300,299,298,298,296,296,294,292,290,
    289,289,286,284,284,283,281,280,278,278,277,277,273,273,272,271,
    269,268,268,267
  };
  const int n2w1b1r1[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    393,393,391,390,390,388,386,386,385,385,385,384,379,378,377,376,
    375,374,373,372,368,367,367,366,366,365,364,364,362,362,361,358,
    356,355,355,353,352,352,350,348,348,346,345,342,342,341,340,337,
    337,336,335,332,332,332,331,328,327,326,324,322,322,320,320,319,
    318,316,315,312,311,307,307,305,305,305,304,304,303,299,298,297,
    296,296,295,291,291,291,288,287,283,282,282,282,280,278,277,276,
    275,272,266,266
  };
  const int n2w1b1r2[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    396,394,393,393,393,392,392,387,387,385,384,384,382,382,381,378,
    377,375,371,367,367,366,366,362,359,359,356,356,351,347,346,346,
    346,346,345,341,341,341,340,339,339,336,334,334,332,330,326,325,
    325,322,320,320,320,319,319,317,317,316,316,315,315,315,314,314,
    312,312,310,310,306,306,306,303,300,299,298,298,295,295,295,292,
    292,291,290,289,284,284,282,281,279,278,276,275,275,274,273,273,
    271,270,270,268
  };
  const int n2w1b1r3[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    396,395,393,389,387,387,386,384,384,384,383,383,382,381,381,379,
    377,376,376,376,375,371,371,370,367,364,363,360,359,359,358,357,
    356,355,355,355,352,349,348,347,346,346,344,344,343,343,342,341,
    338,336,335,335,332,332,328,325,325,324,321,321,318,318,312,312,
    311,310,307,307,306,306,304,302,301,301,300,299,299,298,298,296,
    295,294,293,293,292,289,289,288,284,283,282,280,280,279,277,277,
    277,275,266,266
  };
  const int n2w1b1r4[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    394,390,390,389,388,384,383,381,380,380,380,378,377,377,377,376,
    375,370,369,367,367,366,366,365,364,360,359,358,358,357,354,353,
    353,353,352,351,349,347,346,346,345,345,343,343,340,339,338,334,
    333,333,326,326,324,321,321,319,319,317,315,314,314,313,311,310,
    308,307,306,305,303,302,302,301,301,300,299,299,296,295,292,292,
    290,289,287,283,281,281,278,277,277,275,274,274,273,273,273,272,
    272,267,267,266
  };
  const int n2w1b1r5[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    395,394,394,393,391,390,389,386,386,384,383,377,376,371,369,368,
    367,367,366,365,362,362,361,360,359,359,359,355,353,350,350,349,
    349,349,345,343,342,342,340,340,339,338,336,335,332,329,328,327,
    327,327,323,321,320,316,315,312,312,311,311,310,310,309,308,306,
    305,303,303,302,302,297,297,296,295,294,294,292,292,292,288,287,
    287,287,284,282,282,282,282,282,281,278,278,277,273,272,272,270,
    270,269,268,268
  };
  const int n2w1b1r6[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    396,396,394,394,393,389,388,387,387,387,386,386,385,383,383,381,
    379,379,378,378,376,376,375,374,371,371,365,364,363,363,363,363,
    361,358,357,355,354,353,350,349,349,348,346,346,346,345,344,343,
    342,342,341,341,339,336,334,331,331,331,329,328,328,327,326,324,
    321,318,316,316,314,311,310,307,305,303,299,297,297,290,290,287,
    286,284,284,282,282,281,278,277,277,277,276,275,275,273,272,271,
    271,267,267,266
  };
  const int n2w1b1r7[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    394,387,387,387,386,385,383,383,379,379,379,379,378,377,377,376,
    375,375,374,374,373,372,367,366,364,364,360,357,356,355,355,353,
    352,352,352,349,348,347,344,344,343,342,341,338,335,334,331,331,
    331,330,328,327,326,325,325,325,325,325,325,324,324,323,323,322,
    321,318,315,315,310,309,307,305,305,305,303,303,303,297,293,291,
    291,291,291,290,289,289,287,282,282,281,280,280,277,276,275,274,
    273,273,271,268
  };
  const int n2w1b1r8[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    396,395,394,394,393,389,387,387,387,385,385,384,383,380,379,378,
    375,374,373,373,373,372,370,367,365,364,361,358,358,354,353,351,
    348,347,347,347,344,344,343,343,342,342,342,341,341,340,340,338,
    336,334,334,332,330,329,329,326,326,325,324,323,322,321,321,321,
    319,317,316,312,311,310,310,310,309,306,306,305,301,300,300,298,
    298,298,295,293,292,289,287,286,286,285,281,281,280,280,276,275,
    274,274,274,271
  };
  const int n2w1b1r9[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    395,394,393,393,390,388,387,387,386,385,384,382,381,380,377,376,
    375,373,370,369,367,367,367,363,362,361,360,358,358,357,356,356,
    354,354,354,354,351,350,349,349,348,348,346,345,345,337,335,335,
    334,333,332,329,329,328,328,325,325,322,322,321,321,320,320,317,
    316,312,309,308,308,307,306,305,305,303,303,303,303,301,301,300,
    297,294,294,287,285,284,282,281,281,280,278,277,276,275,274,273,
    273,269,268,267
  };
  const int n2w1b2r0[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    494,493,490,488,477,474,470,465,462,449,449,448,447,447,444,442,
    436,436,432,428,428,423,421,418,417,416,410,409,408,405,402,401,
    401,400,399,395,395,394,388,387,387,380,378,378,372,372,364,364,
    360,356,354,347,346,346,332,331,331,326,317,317,315,314,313,312,
    308,305,303,301,299,295,294,292,291,288,288,283,282,279,278,275,
    272,270,268,268,255,255,242,240,237,236,234,215,211,208,206,206,
    203,196,191,167
  };
  const int n2w1b2r1[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    495,495,494,494,486,485,484,479,469,465,462,456,450,447,447,444,
    441,437,436,423,419,414,410,410,405,404,400,396,395,389,388,387,
    385,380,374,373,373,370,369,369,368,366,364,352,351,342,342,337,
    335,333,331,326,325,319,317,313,303,294,293,293,292,292,285,284,
    281,257,257,253,250,247,245,243,241,240,238,237,234,233,233,232,
    229,228,224,223,222,205,202,198,196,192,190,189,183,182,182,181,
    178,175,172,170
  };
  const int n2w1b2r2[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    493,489,486,476,470,468,460,457,455,451,450,449,447,447,445,445,
    443,442,440,437,432,430,425,424,424,418,415,412,408,408,408,407,
    404,404,402,400,394,389,389,388,386,384,380,379,373,373,373,367,
    364,362,362,359,346,343,343,342,332,330,326,320,312,302,298,293,
    284,283,281,278,276,273,273,272,271,266,259,255,255,245,243,242,
    240,239,239,233,230,214,209,209,207,205,200,199,195,194,185,184,
    181,179,177,175
  };
  const int n2w1b2r3[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    491,489,485,485,483,479,477,476,476,475,473,472,471,464,462,461,
    459,456,454,453,449,446,443,439,438,437,417,415,415,410,408,404,
    400,399,396,391,388,385,381,380,373,372,370,369,364,362,359,356,
    355,354,353,352,348,345,343,333,330,329,326,323,320,310,307,307,
    290,288,285,285,282,279,276,273,264,263,263,260,254,251,250,248,
    246,233,232,231,218,214,205,201,198,196,195,195,195,192,185,184,
    183,180,170,170
  };
  const int n2w1b2r4[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    493,489,488,486,482,480,470,467,449,444,443,432,430,425,423,415,
    414,411,410,407,404,401,398,398,392,389,384,378,377,376,374,374,
    373,370,369,368,366,366,361,354,346,342,341,338,332,328,328,327,
    318,317,315,311,311,310,305,302,302,299,298,294,290,285,282,277,
    274,272,269,268,260,257,256,254,253,252,252,251,241,236,234,231,
    224,223,222,221,220,219,216,216,213,205,193,190,182,180,179,177,
    176,172,169,167
  };
  const int n2w1b2r5[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    495,493,487,485,484,479,478,478,477,475,470,469,467,466,465,463,
    461,458,457,456,455,454,453,452,450,446,436,429,425,422,414,409,
    409,405,402,397,397,397,391,387,387,375,370,369,364,355,354,351,
    338,337,335,331,329,319,309,307,299,294,293,293,292,291,290,290,
    289,288,285,282,272,272,269,265,247,245,242,242,240,234,233,229,
    229,229,226,221,217,217,212,209,206,201,201,194,194,191,186,183,
    182,179,179,175
  };
  const int n2w1b2r6[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    495,487,487,485,484,484,481,477,471,467,466,466,463,462,458,449,
    448,445,443,431,422,420,419,418,415,414,406,405,403,400,399,398,
    396,392,392,386,385,377,376,375,374,373,372,371,370,370,370,369,
    365,365,360,360,355,350,346,346,331,327,321,310,308,305,304,303,
    299,293,291,290,286,276,271,270,266,264,261,261,260,260,256,254,
    252,251,250,248,242,241,212,211,209,206,205,201,195,195,192,191,
    191,189,174,167
  };
  const int n2w1b2r7[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    494,485,482,475,475,460,458,458,454,454,445,445,442,436,435,431,
    424,424,422,413,412,411,409,408,405,403,400,398,392,392,380,380,
    379,378,375,370,370,366,360,353,348,343,343,343,342,340,338,334,
    333,329,328,326,314,312,309,297,297,294,293,290,287,285,280,275,
    274,274,272,267,263,263,258,253,252,248,243,236,235,235,233,230,
    229,229,228,227,226,225,211,209,204,200,196,190,189,188,186,178,
    177,172,170,169
  };
  const int n2w1b2r8[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    494,493,491,485,480,478,473,472,462,459,458,457,452,452,446,443,
    439,438,437,437,436,429,425,422,421,416,415,415,410,408,407,406,
    399,394,391,391,388,386,385,383,373,373,372,361,361,357,353,346,
    344,342,340,327,325,325,320,319,313,308,307,305,303,298,294,290,
    287,283,283,280,280,278,277,275,273,273,267,267,265,262,258,253,
    248,243,243,242,240,232,232,228,223,211,209,207,198,197,192,192,
    191,176,172,171
  };
  const int n2w1b2r9[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    494,491,483,473,472,465,464,461,461,460,457,453,445,444,443,442,
    442,438,435,424,421,421,412,409,406,405,402,395,395,391,391,389,
    389,380,378,375,374,371,369,366,361,360,360,357,353,349,348,346,
    343,341,338,336,335,334,330,326,316,310,308,307,302,298,288,287,
    283,281,272,263,262,259,255,248,247,243,234,230,229,229,228,226,
    223,222,221,218,214,205,203,196,195,192,189,187,183,182,180,176,
    175,175,173,173
  };
  const int n2w1b3r0[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    617,617,610,608,606,604,600,597,588,585,584,578,568,564,555,552,
    533,531,531,521,506,500,494,486,485,476,475,474,471,468,462,450,
    446,445,440,419,418,409,407,401,398,394,393,387,372,370,367,361,
    360,351,345,339,319,316,313,304,299,297,294,279,275,275,258,257,
    252,251,247,246,246,223,220,215,213,213,212,207,206,200,191,181,
    174,166,163,160,156,149,144,144,133,131,131,114,84,77,75,60,57,
    54,44,35
  };
  const int n2w1b3r1[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    618,608,597,594,578,573,572,568,567,567,564,550,545,542,540,539,
    536,535,525,511,510,505,504,496,485,478,475,473,457,451,445,441,
    436,436,430,429,416,411,406,401,385,380,350,347,341,337,321,311,
    308,304,303,297,290,288,285,285,279,275,268,260,249,248,244,234,
    230,222,215,195,185,185,182,179,179,175,166,164,153,146,137,129,
    116,113,112,106,99,98,97,91,90,89,83,68,64,64,62,56,55,49,47,
    45
  };
  const int n2w1b3r2[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    618,617,614,614,610,609,601,589,588,586,586,583,575,568,563,560,
    552,548,547,535,527,520,519,514,511,511,509,509,505,502,491,481,
    474,471,459,446,443,425,416,413,403,398,397,396,396,392,387,386,
    382,367,359,352,332,331,322,321,311,306,289,281,264,256,255,244,
    243,241,219,215,214,206,204,199,196,194,192,187,183,183,183,179,
    177,176,175,173,173,169,160,154,126,94,87,86,81,72,65,63,54,47,
    41,36
  };
  const int n2w1b3r3[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    618,611,604,602,594,588,583,583,582,582,573,554,538,536,534,521,
    505,500,499,494,493,492,477,475,470,448,445,442,432,430,429,429,
    420,412,408,408,404,401,393,389,388,374,369,363,362,359,354,340,
    327,326,325,318,317,308,304,291,286,275,268,267,264,263,249,212,
    207,200,200,200,197,192,182,182,178,177,177,172,168,164,159,153,
    150,138,134,132,127,116,109,92,87,83,77,75,67,60,59,51,47,45,
    37,36
  };
  const int n2w1b3r4[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    623,610,595,582,582,581,574,568,565,564,563,555,553,545,539,537,
    534,534,523,516,513,509,506,504,502,489,474,471,468,468,465,463,
    461,460,457,437,437,429,419,411,399,396,391,384,384,375,358,356,
    344,342,322,308,306,305,303,294,294,288,284,266,264,252,251,237,
    235,234,232,222,206,193,190,189,189,187,184,183,171,171,154,148,
    138,135,134,134,124,123,122,120,116,93,87,65,54,52,52,51,48,41,
    41,36
  };
  const int n2w1b3r5[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    621,620,617,607,602,591,589,586,585,581,579,569,561,558,555,554,
    546,544,539,539,526,503,502,498,489,471,456,451,450,443,438,436,
    434,425,424,424,420,420,418,408,405,404,377,371,361,359,346,340,
    331,321,320,313,310,308,299,286,281,274,270,269,264,262,262,254,
    250,215,214,208,205,200,193,183,177,171,163,162,158,156,154,146,
    146,136,124,118,115,109,105,101,101,94,92,88,86,79,76,74,73,73,
    67,66
  };
  const int n2w1b3r6[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    625,622,620,609,604,601,597,582,582,574,572,570,544,542,537,537,
    535,530,523,507,485,483,480,456,447,447,444,439,429,426,425,414,
    412,406,406,401,397,394,378,367,364,360,341,327,324,321,314,307,
    297,291,289,272,270,267,263,236,231,230,227,227,226,225,219,215,
    215,212,211,205,178,176,170,149,145,139,138,138,135,129,122,115,
    114,108,108,105,87,86,85,83,81,69,68,67,58,56,55,51,45,41,40,
    37
  };
  const int n2w1b3r7[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    626,617,608,606,606,602,586,579,573,567,551,548,514,514,510,492,
    492,491,471,469,465,443,441,440,436,431,430,427,422,410,393,392,
    392,379,377,376,360,343,341,339,330,323,322,321,314,313,307,304,
    299,298,296,294,291,278,277,276,273,269,239,228,226,222,216,214,
    211,192,191,181,176,166,166,164,161,155,148,135,133,131,130,125,
    120,117,106,101,101,100,98,98,94,92,91,76,66,61,56,55,52,47,47,
    35
  };
  const int n2w1b3r8[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    626,611,609,604,598,592,586,584,578,576,574,568,557,553,549,541,
    541,533,533,529,527,525,524,517,514,511,507,504,499,496,492,488,
    477,476,471,459,456,442,436,425,421,419,401,388,386,362,358,354,
    352,345,322,322,317,298,293,280,262,261,258,249,247,241,238,233,
    219,209,205,204,203,190,186,177,174,174,164,163,154,153,153,133,
    133,126,122,121,120,119,119,113,110,101,97,90,70,68,66,59,52,
    45,39,37
  };
  const int n2w1b3r9[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    624,606,606,598,598,577,563,557,536,520,514,495,494,487,487,487,
    485,477,471,467,449,447,437,436,421,413,413,412,400,393,392,391,
    382,377,366,356,350,345,343,340,331,331,330,328,320,320,296,294,
    292,286,277,273,271,260,254,250,245,227,226,221,219,215,203,197,
    196,166,165,157,156,153,151,147,144,144,133,127,127,126,125,125,
    123,122,121,119,117,104,96,84,77,76,73,65,57,55,51,48,42,38,37,
    35
  };
  const int n2w2b1r0[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    240,239,238,235,232,231,231,231,231,230,229,228,228,228,227,226,
    222,219,218,217,217,217,217,217,216,216,214,214,213,212,212,211,
    210,209,208,208,208,206,206,206,206,205,205,204,204,203,200,199,
    199,199,198,198,197,197,196,195,193,193,193,193,191,191,188,188,
    188,187,186,186,183,183,182,181,179,178,177,177,177,177,176,176,
    176,175,175,175,172,172,171,170,170,169,168,168,167,167,166,166,
    164,163,163,162
  };
  const int n2w2b1r1[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    239,237,237,235,234,234,234,233,232,232,231,229,229,227,226,226,
    225,224,224,223,222,222,222,220,220,219,215,212,212,207,206,205,
    205,205,204,204,203,203,202,201,201,201,201,200,200,199,198,198,
    197,195,195,195,194,193,192,191,191,191,190,189,189,189,188,187,
    187,186,186,185,185,183,183,182,182,182,181,180,180,180,180,179,
    178,177,177,174,173,173,173,173,170,170,169,168,168,167,167,166,
    163,163,162,162
  };
  const int n2w2b1r2[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    240,240,238,237,237,235,235,234,234,233,233,233,233,232,232,231,
    230,230,229,229,228,228,228,227,225,225,222,222,222,222,220,219,
    218,216,214,213,213,213,213,212,211,211,210,210,210,208,207,207,
    207,205,204,204,203,202,202,200,200,199,199,197,197,197,196,195,
    195,194,192,191,188,187,186,185,183,182,181,180,180,177,177,176,
    174,174,174,174,173,172,171,168,166,166,165,163,163,162,162,162,
    162,162,162,162
  };
  const int n2w2b1r3[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    239,238,237,237,236,236,236,235,235,234,234,232,232,231,230,230,
    230,230,229,228,228,227,227,226,226,223,221,220,220,219,217,217,
    216,213,212,212,211,211,208,207,207,207,204,204,204,203,203,203,
    200,200,198,198,197,197,195,195,195,194,193,193,193,192,187,186,
    186,185,185,185,183,183,183,183,183,182,182,182,182,180,180,180,
    179,179,177,176,174,174,173,172,170,170,169,169,168,166,166,165,
    165,164,163,162
  };
  const int n2w2b1r4[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    240,240,240,239,238,236,236,235,234,233,231,230,229,229,228,228,
    227,227,224,224,224,223,222,221,219,219,219,219,217,217,216,216,
    215,214,214,214,214,212,212,211,210,209,209,209,208,208,207,207,
    207,206,206,206,205,205,205,205,204,202,202,198,197,197,195,195,
    195,194,193,192,189,185,185,185,182,181,180,179,178,175,175,175,
    175,172,171,170,169,168,168,168,167,167,167,167,167,166,166,165,
    164,164,163,162
  };
  const int n2w2b1r5[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    239,238,237,237,236,236,235,235,234,234,234,234,233,233,233,232,
    232,231,230,230,229,228,228,228,227,226,225,225,223,223,222,221,
    221,221,218,216,216,216,215,213,213,212,212,211,211,209,207,207,
    207,206,206,206,206,206,204,203,201,201,200,199,199,198,198,197,
    197,195,195,192,192,192,191,190,189,188,185,185,184,184,183,183,
    182,180,179,178,177,177,172,171,171,170,168,168,166,166,166,166,
    163,163,162,162
  };
  const int n2w2b1r6[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    238,236,236,236,235,235,234,233,233,232,231,231,231,231,230,230,
    230,229,229,228,228,227,227,227,225,224,224,224,224,223,221,221,
    218,216,215,215,215,214,214,213,213,213,211,210,208,207,207,206,
    205,204,203,200,200,199,198,197,195,195,195,193,192,191,191,190,
    190,189,188,188,185,185,184,183,183,183,182,181,181,181,180,179,
    179,177,176,174,172,172,172,171,170,170,169,168,168,168,166,163,
    163,163,163,162
  };
  const int n2w2b1r7[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    240,240,239,237,235,235,235,235,235,232,231,230,230,229,228,228,
    227,226,225,223,222,220,219,219,219,218,217,217,216,216,216,216,
    216,215,215,215,214,214,214,213,212,211,211,210,210,209,208,208,
    208,207,206,203,202,202,201,200,198,196,196,194,194,193,189,189,
    188,188,187,186,185,184,184,182,182,182,180,178,178,177,176,176,
    173,172,171,171,171,171,171,170,170,170,169,168,168,167,166,165,
    165,165,163,162
  };
  const int n2w2b1r8[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    240,240,240,239,239,239,239,238,238,238,237,236,233,232,231,230,
    230,230,228,223,222,219,219,218,218,218,217,217,216,214,214,213,
    212,212,211,211,210,210,209,208,208,208,207,207,206,206,206,204,
    203,203,203,203,203,202,201,201,200,200,200,200,199,199,199,198,
    196,196,196,194,194,191,189,188,188,188,188,187,185,185,185,183,
    182,182,181,179,179,178,177,176,176,175,175,172,172,168,167,166,
    163,163,163,163
  };
  const int n2w2b1r9[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    236,234,233,232,232,231,230,230,230,229,228,226,226,225,225,222,
    222,221,220,220,219,219,217,217,217,215,215,214,214,213,212,211,
    211,209,208,208,208,208,207,207,206,206,206,205,205,204,204,201,
    201,201,201,201,200,200,198,197,197,196,195,195,194,194,194,194,
    194,193,192,192,189,188,188,188,187,187,183,182,181,180,179,177,
    175,175,174,172,171,171,171,169,169,169,169,169,167,167,165,164,
    163,163,163,162
  };
  const int n2w2b2r0[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    299,298,295,293,293,291,290,289,288,288,282,282,281,281,280,280,
    279,279,278,275,274,271,271,270,267,267,263,260,258,256,256,256,
    249,247,247,246,245,239,239,239,236,236,232,230,222,218,215,214,
    213,213,213,210,206,204,202,202,201,191,190,189,189,187,187,181,
    181,179,170,169,168,166,166,161,158,151,149,148,146,145,142,139,
    137,135,132,130,128,127,123,123,121,120,118,109,107,107,105,105,
    104,104,102,102
  };
  const int n2w2b2r1[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    296,295,295,294,291,290,288,288,287,286,283,282,280,279,279,278,
    277,275,273,269,266,262,261,254,251,250,248,248,246,246,245,244,
    244,239,238,234,233,233,232,231,229,229,216,214,211,211,210,198,
    196,195,195,194,192,192,191,191,190,188,187,187,185,184,180,177,
    172,172,172,171,167,167,166,165,160,160,158,155,148,146,145,143,
    140,140,131,131,128,126,123,122,121,121,117,117,113,111,108,107,
    106,106,103,103
  };
  const int n2w2b2r2[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    300,299,295,293,292,289,286,285,285,285,284,284,281,278,275,273,
    271,270,269,265,263,263,262,261,260,257,257,255,251,247,238,237,
    236,235,233,233,232,232,231,223,221,218,214,211,209,208,207,207,
    205,204,203,201,198,195,193,192,190,187,182,175,175,175,175,174,
    174,172,169,168,167,166,159,157,156,152,151,150,148,148,146,145,
    144,143,142,141,139,136,136,133,132,126,125,122,121,119,118,116,
    110,106,105,102
  };
  const int n2w2b2r3[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    300,300,298,295,292,290,289,287,287,286,286,286,284,283,278,273,
    271,269,269,269,268,268,267,262,258,256,256,255,255,255,254,252,
    251,249,248,246,245,244,242,238,237,237,236,227,227,226,224,224,
    223,222,214,212,208,206,206,205,202,202,202,200,200,199,197,195,
    195,192,192,189,185,179,178,178,171,171,167,165,162,161,158,152,
    149,146,143,143,139,136,136,131,127,126,126,124,121,118,114,113,
    106,105,102,102
  };
  const int n2w2b2r4[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    300,298,297,294,292,290,287,287,286,283,282,281,280,280,275,273,
    270,269,269,268,267,266,265,265,265,264,262,262,262,261,255,254,
    253,252,252,250,246,245,238,238,237,236,236,232,231,231,230,229,
    228,228,228,227,224,223,220,217,216,216,215,214,213,211,203,203,
    201,199,198,198,197,197,195,187,185,181,178,171,170,165,165,162,
    160,158,150,147,139,135,131,131,129,128,127,126,118,117,115,107,
    107,107,106,105
  };
  const int n2w2b2r5[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    297,296,293,292,290,290,286,281,279,278,276,274,273,271,267,265,
    261,260,260,259,259,259,258,255,246,245,243,242,242,239,236,236,
    234,234,226,224,221,221,219,219,219,211,210,209,208,208,204,203,
    203,202,202,202,201,200,199,198,196,191,188,188,177,176,173,172,
    172,172,171,171,162,162,160,157,153,150,148,148,145,141,139,137,
    137,134,134,132,130,128,126,125,119,117,116,115,114,114,109,108,
    106,105,104,102
  };
  const int n2w2b2r6[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    300,299,298,295,293,292,291,289,285,280,279,279,277,275,271,269,
    265,263,260,259,259,256,251,248,248,247,246,245,243,242,240,239,
    239,239,233,233,232,232,230,229,225,221,220,219,219,217,216,215,
    214,213,212,206,206,195,195,193,189,189,189,188,187,186,181,177,
    174,171,170,169,168,168,166,166,165,165,150,149,148,148,148,147,
    146,144,142,141,140,139,139,137,134,131,130,128,126,126,120,117,
    113,106,104,103
  };
  const int n2w2b2r7[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    300,297,296,290,289,288,286,285,282,281,278,275,275,272,267,265,
    262,259,255,252,251,249,244,243,239,237,237,236,236,232,231,230,
    230,229,224,223,222,222,220,219,218,215,214,213,206,204,204,201,
    196,195,193,191,187,187,184,184,181,180,172,171,164,163,162,161,
    161,160,155,155,149,149,145,142,142,141,141,140,139,137,136,135,
    132,131,127,127,123,121,119,119,119,117,116,116,115,113,108,108,
    106,105,103,103
  };
  const int n2w2b2r8[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    299,299,299,297,294,288,285,279,277,277,276,275,274,273,272,271,
    271,269,266,262,260,260,257,255,254,254,253,252,252,245,244,243,
    241,240,235,235,233,230,229,228,228,226,226,225,224,223,223,219,
    219,218,214,211,206,199,198,197,196,191,186,183,183,183,180,179,
    179,177,176,174,174,173,172,163,159,158,153,147,146,146,146,145,
    145,141,139,131,131,128,125,123,123,123,122,120,119,117,114,114,
    114,106,104,104
  };
  const int n2w2b2r9[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    298,296,291,289,287,287,281,279,279,277,276,275,274,273,272,271,
    267,265,262,258,257,255,254,253,251,250,244,243,242,235,233,232,
    232,230,229,224,221,220,220,218,216,214,211,207,206,202,201,200,
    199,199,192,190,190,188,187,187,185,184,183,182,182,180,180,179,
    174,173,171,168,167,166,163,161,161,160,158,157,148,148,147,147,
    143,140,134,133,132,131,127,124,120,119,117,116,114,113,111,109,
    108,106,106,103
  };
  const int n2w2b3r0[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    379,379,367,366,363,358,358,355,352,345,343,337,335,329,329,325,
    324,320,317,317,311,303,296,294,292,288,280,277,268,268,267,264,
    261,259,256,255,254,247,247,244,236,235,234,231,230,228,224,217,
    216,212,208,207,207,204,191,190,189,186,182,180,173,173,164,159,
    157,154,152,150,141,138,136,130,119,116,105,103,100,98,88,87,
    86,86,85,65,63,63,60,57,57,57,53,52,50,29,25,24,24,23,22,22
  };
  const int n2w2b3r1[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    373,368,368,367,365,360,352,335,335,332,324,321,321,320,316,304,
    304,303,299,298,294,292,288,286,284,273,273,273,266,266,263,262,
    262,259,258,256,255,249,245,237,230,227,221,220,216,208,206,206,
    202,189,188,185,184,180,179,178,176,173,167,158,154,148,148,147,
    145,139,135,132,130,124,122,122,116,114,111,111,111,104,98,89,
    84,79,72,70,63,61,60,59,55,54,50,44,44,41,39,32,31,30,26,25
  };
  const int n2w2b3r2[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    375,373,369,367,366,363,362,360,360,359,356,346,345,342,339,334,
    334,333,332,331,328,328,327,326,322,320,311,305,291,291,289,288,
    277,275,270,262,250,231,228,228,225,218,217,216,213,210,207,205,
    204,201,201,200,193,187,173,171,170,166,165,162,161,160,155,155,
    154,152,150,148,145,143,135,134,134,132,130,124,123,123,108,105,
    104,99,97,93,91,86,85,79,75,61,57,56,51,49,41,40,40,30,30,22
  };
  const int n2w2b3r3[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    378,377,360,355,354,342,331,331,330,327,323,323,320,320,313,311,
    301,296,295,293,292,286,283,277,276,271,265,264,253,252,233,233,
    232,232,229,224,221,217,217,212,211,211,207,205,205,203,198,198,
    197,194,192,191,190,186,178,165,164,163,156,155,152,148,148,147,
    143,142,134,133,132,130,124,115,113,107,103,91,85,80,79,78,77,
    68,62,60,60,59,56,55,52,43,42,39,34,33,32,32,32,31,27,26
  };
  const int n2w2b3r4[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    380,380,379,376,372,366,363,356,351,351,350,348,348,347,347,339,
    338,337,332,331,331,329,328,322,322,312,307,305,295,290,287,279,
    278,269,269,268,267,263,263,255,250,249,249,244,240,240,236,235,
    229,223,223,217,189,183,182,169,157,154,153,148,146,144,142,129,
    128,122,121,117,109,105,102,101,100,96,96,87,87,85,82,81,80,79,
    78,77,73,72,70,66,65,65,63,54,52,39,38,35,34,32,31,23
  };
  const int n2w2b3r5[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    376,374,373,360,358,351,348,345,344,343,332,328,327,327,323,317,
    317,315,313,308,307,305,297,297,291,289,285,284,277,276,263,262,
    261,261,258,258,256,251,244,242,241,235,235,235,235,234,230,227,
    226,225,222,218,218,208,203,202,184,178,177,176,169,165,161,159,
    154,142,137,134,133,132,127,125,123,123,121,116,111,109,109,103,
    102,93,81,79,75,71,71,57,57,50,46,45,38,37,28,27,27,22,22,22
  };
  const int n2w2b3r6[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    378,377,374,373,369,369,366,353,351,338,337,337,337,334,330,330,
    323,322,320,319,317,313,306,305,298,297,295,287,283,276,276,268,
    267,267,265,262,257,257,248,247,240,237,236,233,231,217,201,195,
    193,187,184,171,170,166,163,161,159,158,158,157,141,139,138,137,
    126,122,119,116,115,112,106,104,102,101,100,98,98,91,86,84,82,
    82,78,73,62,61,60,60,58,58,55,52,48,48,41,40,38,36,31,26
  };
  const int n2w2b3r7[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    372,372,371,371,367,366,365,365,365,364,363,360,352,350,350,350,
    348,345,333,331,317,315,310,310,308,306,305,304,304,299,295,292,
    286,279,277,263,262,262,258,248,241,235,235,231,229,222,208,207,
    204,203,202,200,196,195,195,195,192,191,186,184,170,168,165,163,
    162,157,150,139,135,127,126,125,124,124,123,120,117,117,116,109,
    106,95,82,81,79,76,68,59,58,56,54,53,51,51,40,37,32,25,23,22
  };
  const int n2w2b3r8[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    371,365,363,354,352,351,346,345,345,339,338,338,334,332,329,327,
    322,321,319,314,305,302,299,296,294,288,285,284,282,281,277,276,
    269,268,262,257,252,250,250,248,245,243,236,234,232,230,229,224,
    220,214,211,209,206,198,195,192,188,177,171,163,158,157,157,147,
    142,140,124,118,111,111,111,111,102,93,88,87,86,82,82,80,78,78,
    76,75,72,69,65,63,54,51,50,49,43,41,39,36,29,29,27,25
  };
  const int n2w2b3r9[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    378,377,374,373,367,365,363,357,353,348,338,336,331,322,313,308,
    307,306,304,299,299,298,291,291,283,283,281,279,277,272,270,270,
    269,263,260,257,251,247,246,243,239,238,237,228,227,208,202,197,
    191,186,186,180,177,176,174,171,170,170,164,151,149,146,146,146,
    145,143,140,139,137,116,116,115,114,113,110,102,100,99,91,87,
    85,82,81,81,80,73,72,69,55,53,49,47,46,44,43,39,36,34,28,23
  };
  const int n2w3b1r0[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    168,168,168,167,167,167,166,166,165,165,165,165,164,164,164,164,
    164,163,163,163,162,161,160,159,159,159,157,157,155,154,154,154,
    154,153,153,153,151,150,149,149,149,148,148,147,147,147,147,146,
    145,145,145,144,143,143,142,142,142,141,139,138,137,136,135,135,
    133,133,133,133,132,131,130,130,129,129,129,128,128,128,127,127,
    126,125,125,124,124,122,122,121,121,121,120,120,119,119,119,118,
    118,118,115,115
  };
  const int n2w3b1r1[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    168,168,167,166,165,165,165,165,164,164,163,163,163,163,163,163,
    163,162,162,162,162,162,162,161,161,159,157,157,157,157,156,156,
    155,155,153,153,153,152,151,151,150,150,149,149,149,147,147,147,
    147,146,145,144,144,143,142,142,142,141,139,138,134,133,133,133,
    132,132,131,130,129,128,128,128,128,127,127,127,127,127,125,125,
    124,123,123,123,121,119,119,119,118,117,117,117,117,117,117,116,
    116,115,115,114
  };
  const int n2w3b1r2[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    168,168,167,167,167,167,167,166,166,165,165,165,164,163,163,162,
    160,160,160,159,159,159,158,158,158,158,158,158,157,157,156,156,
    155,155,154,154,154,154,154,154,154,153,153,152,151,150,150,149,
    148,148,148,147,145,144,144,143,142,142,141,140,139,138,138,138,
    137,136,136,136,136,136,135,135,135,134,132,131,131,129,126,126,
    126,126,125,124,124,123,122,122,121,120,120,119,119,118,117,117,
    116,116,114,114
  };
  const int n2w3b1r3[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    166,166,166,166,165,164,164,164,163,163,162,162,162,161,160,159,
    159,159,158,158,157,156,156,152,151,150,149,149,149,147,147,146,
    145,145,144,144,144,142,142,141,141,141,141,140,140,140,139,138,
    138,137,137,137,137,135,135,134,133,133,133,133,132,132,132,131,
    131,131,130,130,130,130,130,130,129,129,129,128,128,126,126,125,
    125,124,123,123,121,120,120,120,119,119,119,118,117,117,117,117,
    115,115,115,114
  };
  const int n2w3b1r4[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    168,168,167,166,166,166,165,165,164,164,164,163,163,163,162,162,
    161,160,160,159,158,158,158,157,156,156,156,155,155,152,152,152,
    151,151,149,148,148,148,148,147,147,145,145,145,144,143,143,143,
    143,143,143,140,140,139,138,138,137,137,136,136,136,135,134,133,
    132,132,132,132,131,131,131,130,130,130,130,130,129,127,126,124,
    124,124,122,122,122,122,121,121,121,121,120,120,119,118,117,117,
    116,116,115,114
  };
  const int n2w3b1r5[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    167,167,166,166,165,165,165,165,165,164,164,164,162,161,160,160,
    160,160,159,158,158,157,157,157,155,154,153,153,152,152,152,151,
    151,151,150,150,150,149,148,147,145,145,144,144,143,143,143,143,
    140,140,140,140,140,139,139,137,137,137,136,135,134,134,133,133,
    132,132,131,129,129,128,127,127,127,126,125,125,123,123,123,123,
    122,122,122,120,120,119,119,119,118,117,117,117,116,116,115,115,
    115,115,115,115
  };
  const int n2w3b1r6[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    167,167,166,166,164,164,164,163,162,162,162,162,162,161,161,160,
    159,159,158,158,158,158,157,157,154,154,154,153,153,153,153,152,
    152,151,151,151,151,151,151,151,150,150,149,148,148,147,147,146,
    145,144,143,143,143,143,143,143,142,141,141,139,139,137,136,136,
    135,135,135,133,133,132,132,131,130,128,128,128,127,127,126,125,
    125,124,124,123,123,122,121,121,121,120,120,120,120,119,119,118,
    118,117,116,115
  };
  const int n2w3b1r7[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    168,168,167,167,167,166,166,165,165,164,164,164,163,163,163,163,
    163,160,159,159,159,158,158,158,158,158,158,156,156,155,155,154,
    154,153,152,150,149,148,147,145,145,144,144,144,143,143,142,138,
    138,138,138,137,137,136,134,134,133,133,132,132,131,131,130,130,
    130,129,129,128,128,125,125,124,123,123,123,123,122,122,122,122,
    121,121,121,120,120,120,119,119,118,118,118,117,115,115,115,115,
    114,114,114,114
  };
  const int n2w3b1r8[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    168,168,167,167,167,166,166,165,165,164,164,164,163,163,162,162,
    161,161,160,159,158,158,157,156,156,155,155,155,154,154,154,154,
    153,153,152,152,151,150,149,148,148,147,147,146,145,144,144,144,
    143,143,143,138,136,135,135,134,133,132,132,131,129,129,129,129,
    128,127,126,126,126,126,126,125,125,124,124,124,123,123,122,121,
    121,120,120,120,119,119,119,118,117,117,117,116,116,115,115,115,
    115,114,114,114
  };
  const int n2w3b1r9[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    168,168,166,165,165,165,165,165,165,165,165,164,163,163,162,162,
    162,162,161,160,160,159,159,159,157,157,157,156,156,156,155,154,
    154,153,153,153,150,150,150,150,148,147,146,146,146,145,145,144,
    143,143,143,143,142,141,141,141,140,140,139,138,137,136,135,135,
    135,135,135,133,133,132,131,131,130,130,130,130,129,128,128,128,
    127,127,125,124,124,124,124,123,121,121,120,120,120,119,119,118,
    117,117,115,114
  };
  const int n2w3b2r0[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    209,207,205,204,202,199,199,199,196,194,194,194,193,190,188,186,
    184,183,182,182,179,178,178,178,176,176,176,173,173,172,169,167,
    167,167,164,163,163,162,160,160,156,156,156,154,152,150,146,145,
    145,145,142,141,139,139,136,136,135,134,133,133,129,127,127,127,
    126,123,122,120,119,117,113,113,112,112,108,106,104,97,96,95,
    95,95,94,94,90,90,90,87,87,85,84,83,82,80,79,77,77,75,74,73
  };
  const int n2w3b2r1[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    210,209,209,208,207,206,205,203,201,200,197,192,192,192,191,191,
    190,189,187,185,184,183,182,182,181,177,175,170,168,166,166,165,
    162,162,159,156,154,152,151,151,151,150,149,148,147,145,145,145,
    144,143,142,137,137,136,136,133,133,131,128,127,125,124,115,114,
    113,112,112,108,107,106,105,105,104,104,102,101,99,97,96,95,95,
    95,89,89,89,88,87,86,85,84,84,83,81,80,77,77,77,76,72,72
  };
  const int n2w3b2r2[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    210,210,208,207,203,201,200,199,199,197,196,195,193,192,192,190,
    189,188,188,187,187,186,185,185,182,182,181,180,180,179,177,171,
    170,169,168,166,166,165,165,164,164,161,159,153,151,150,150,149,
    147,147,145,144,142,142,141,139,138,136,136,133,133,130,129,129,
    125,122,122,121,120,119,119,118,118,115,114,110,108,108,107,105,
    105,105,102,102,92,92,87,85,83,80,79,78,77,77,76,76,74,72,72,
    72
  };
  const int n2w3b2r3[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    210,208,206,200,199,198,198,197,195,195,194,193,190,186,186,186,
    182,181,181,180,178,175,175,173,173,172,170,169,168,168,167,166,
    165,164,164,163,159,159,156,152,149,149,148,145,143,143,143,142,
    141,141,141,140,139,139,138,136,135,135,132,131,130,128,126,126,
    125,125,123,123,123,122,120,120,115,115,114,111,108,108,108,103,
    100,99,98,98,96,96,92,91,90,87,86,85,85,84,83,82,80,76,75,74
  };
  const int n2w3b2r4[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    207,202,199,199,198,197,194,192,191,188,186,185,185,184,184,182,
    181,181,180,178,176,174,173,173,171,168,168,168,167,166,164,164,
    163,163,162,159,158,157,155,154,154,153,153,153,151,150,150,148,
    148,143,143,142,142,141,138,138,137,137,134,133,131,131,126,125,
    125,123,121,120,119,118,118,113,111,110,109,108,107,107,106,103,
    99,98,98,95,95,92,91,91,89,88,88,88,87,84,81,77,77,74,74,72
  };
  const int n2w3b2r5[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    209,208,206,206,204,202,200,200,200,195,194,193,193,192,191,189,
    188,188,187,186,185,185,184,184,178,177,176,169,167,164,164,162,
    160,152,152,151,151,149,148,148,147,142,139,137,136,135,135,134,
    132,131,128,127,126,119,119,119,113,113,111,110,109,109,108,107,
    107,107,106,106,105,105,104,104,104,103,102,102,101,101,98,97,
    97,97,97,96,95,95,95,94,89,86,85,83,82,82,79,78,75,74,73,72
  };
  const int n2w3b2r6[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    210,206,205,204,203,202,202,202,200,199,198,192,189,186,185,183,
    183,183,182,181,176,176,175,175,174,170,170,170,170,168,162,161,
    159,156,152,149,149,148,146,146,146,145,144,144,144,141,141,141,
    141,139,138,135,135,135,135,134,134,133,127,127,126,126,125,124,
    119,119,119,116,115,115,108,107,103,98,97,96,94,94,93,91,90,89,
    89,89,89,87,86,86,84,83,82,82,82,81,80,78,77,74,73,72
  };
  const int n2w3b2r7[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    210,209,209,206,206,204,203,202,202,199,199,197,196,195,195,194,
    193,192,191,191,190,190,186,185,185,184,180,171,171,170,168,167,
    166,166,165,163,163,162,161,161,160,160,159,158,158,157,156,156,
    153,151,150,150,148,147,147,145,141,140,137,136,136,132,129,128,
    128,127,127,122,121,118,111,110,109,106,106,102,102,98,98,95,
    95,95,95,93,90,90,90,89,83,82,81,79,78,78,76,75,74,73,73,72
  };
  const int n2w3b2r8[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    210,209,207,202,199,196,196,195,194,193,190,188,187,187,185,185,
    184,184,182,179,178,178,178,176,171,169,169,168,168,167,167,165,
    164,159,158,158,154,152,151,150,148,147,142,142,142,140,140,139,
    138,137,136,136,134,125,125,123,123,121,121,120,120,118,118,117,
    117,116,114,114,112,111,111,108,108,107,106,104,102,102,102,97,
    97,96,94,94,94,92,88,84,84,83,81,81,80,80,78,76,76,76,74,73
  };
  const int n2w3b2r9[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    207,205,204,203,203,200,199,198,196,196,196,195,195,195,192,190,
    189,188,188,187,187,185,180,179,176,175,172,171,170,170,169,168,
    168,165,164,164,163,163,161,160,158,155,154,153,152,150,150,149,
    149,148,148,143,139,137,136,136,134,134,132,132,131,129,127,127,
    127,125,120,120,117,117,116,116,113,112,109,107,105,103,99,99,
    97,95,95,95,95,95,93,91,86,84,82,81,80,79,77,77,77,76,74,72
  };
  const int n2w3b3r0[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    265,263,256,254,253,251,250,249,247,247,246,243,239,238,238,233,
    225,225,224,223,219,216,211,210,208,207,206,204,204,202,202,201,
    192,191,188,171,166,166,160,157,156,155,154,153,153,149,146,146,
    145,144,139,138,130,127,125,124,123,117,115,112,112,104,101,101,
    100,99,99,97,89,87,85,85,81,80,78,75,74,70,70,70,69,67,67,60,
    57,53,52,48,46,46,45,39,33,33,29,29,24,22,21,18
  };
  const int n2w3b3r1[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    260,256,255,253,249,248,245,243,238,234,233,232,229,229,218,213,
    206,205,196,194,187,187,184,181,178,177,176,175,170,170,162,162,
    160,159,156,151,149,141,136,135,135,134,134,133,129,124,123,119,
    116,116,114,113,112,110,105,102,101,99,98,95,95,93,93,83,82,81,
    78,77,73,73,72,70,70,69,68,67,65,64,62,58,54,53,53,50,48,47,43,
    43,43,42,42,41,36,33,24,21,20,19,19,18
  };
  const int n2w3b3r2[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    261,259,256,256,250,249,244,237,235,233,230,228,225,224,223,222,
    219,218,215,213,209,206,205,204,200,197,195,188,188,186,183,180,
    180,176,176,172,165,164,161,161,154,148,146,143,139,138,137,135,
    134,134,128,126,126,122,121,120,117,114,112,109,108,107,106,104,
    99,99,97,97,92,91,90,88,87,86,84,83,83,82,78,74,71,66,64,61,57,
    54,51,47,45,44,42,33,32,28,27,26,26,19,16,16
  };
  const int n2w3b3r3[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    265,264,263,261,254,248,247,246,245,241,233,229,228,227,224,223,
    220,219,218,216,215,212,209,205,198,194,186,180,180,180,177,169,
    166,165,161,160,159,158,157,156,155,154,152,152,151,148,139,137,
    135,127,125,125,120,112,111,111,109,109,107,106,101,101,98,97,
    95,95,95,92,91,90,89,86,84,83,82,80,78,77,77,75,75,74,69,68,68,
    63,58,52,52,52,47,40,33,31,28,27,23,19,17,16
  };
  const int n2w3b3r4[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    266,265,263,262,257,256,250,249,248,244,243,240,240,239,239,238,
    238,237,237,236,235,233,227,227,227,222,220,215,211,210,208,202,
    200,199,193,188,188,186,185,172,171,169,166,163,161,158,148,147,
    143,142,136,130,124,123,123,122,120,119,117,116,110,107,106,98,
    98,96,91,90,85,84,81,79,78,77,77,74,71,69,69,68,67,66,65,64,64,
    61,49,44,44,42,41,40,38,30,26,25,22,21,20,17
  };
  const int n2w3b3r5[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    265,262,262,262,260,255,253,252,248,245,242,239,237,236,225,225,
    222,221,219,218,216,214,213,211,211,209,203,201,201,199,198,197,
    191,187,187,187,182,181,174,173,172,172,170,157,152,150,150,149,
    147,147,145,145,144,143,143,136,135,134,130,129,128,125,115,108,
    107,104,100,98,96,84,82,82,77,75,74,73,73,64,63,61,60,55,51,51,
    46,46,45,37,36,35,33,32,32,27,24,23,22,22,21,16
  };
  const int n2w3b3r6[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    265,259,258,256,253,253,250,250,247,246,241,240,232,229,228,227,
    226,225,225,224,216,215,213,211,209,203,202,202,199,196,196,193,
    185,184,181,181,181,180,177,171,169,167,164,161,155,153,151,150,
    148,143,141,132,130,128,127,126,125,123,119,119,113,112,103,102,
    101,99,97,96,95,91,90,90,86,86,85,79,79,78,77,71,71,64,60,60,
    59,54,49,42,38,38,32,30,28,28,26,24,20,16,16,16
  };
  const int n2w3b3r7[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    260,252,248,243,243,238,237,236,236,227,223,217,216,207,207,207,
    204,203,200,198,197,195,188,177,172,170,169,168,168,165,162,159,
    157,153,150,150,149,148,145,144,143,142,138,137,126,126,126,124,
    123,122,121,121,116,114,113,112,110,109,108,106,105,101,101,99,
    80,78,78,73,72,71,69,69,66,65,64,63,63,58,58,57,57,52,48,48,48,
    46,46,45,43,42,39,37,36,33,22,19,18,17,16,16
  };
  const int n2w3b3r8[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    264,264,263,261,260,259,258,258,257,256,250,249,245,243,242,239,
    239,237,235,233,231,230,226,216,209,206,201,200,195,188,186,185,
    185,183,179,176,171,169,167,166,165,164,158,154,148,148,143,141,
    133,133,130,128,127,121,121,118,118,116,114,113,112,110,101,101,
    96,94,92,91,87,87,86,85,83,83,81,81,72,63,63,61,57,54,51,50,50,
    50,47,45,42,39,37,33,31,29,27,19,19,18,18,16
  };
  const int n2w3b3r9[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    263,261,258,258,252,252,249,248,248,247,244,242,239,233,229,226,
    224,214,210,203,202,202,196,195,195,193,192,187,171,171,169,168,
    168,162,158,156,156,155,155,155,154,149,149,146,144,140,135,135,
    133,131,125,124,122,119,118,114,114,111,107,105,102,96,93,91,
    90,90,87,85,85,84,82,80,79,78,77,76,76,68,66,66,62,60,58,54,54,
    52,49,46,42,39,37,32,30,26,26,25,22,20,18,18
  };
  const int n2w4b1r0[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    132,132,132,132,132,130,130,130,130,130,129,129,128,128,128,128,
    128,127,126,126,125,125,125,125,124,123,123,123,122,122,122,122,
    121,121,121,121,120,120,119,118,118,117,116,115,115,115,114,114,
    114,114,113,113,113,113,112,112,112,111,111,110,110,109,109,108,
    108,107,107,107,107,106,105,103,103,103,102,102,101,101,99,98,
    98,98,98,96,96,96,95,95,95,94,94,93,93,92,91,91,91,91,90,90
  };
  const int n2w4b1r1[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    132,132,132,132,131,131,131,130,130,130,129,129,128,126,126,126,
    125,124,123,122,122,121,121,120,120,120,120,120,119,119,118,118,
    117,117,117,117,116,116,115,115,115,114,114,113,113,112,112,112,
    112,112,112,110,110,110,110,109,109,108,108,108,107,107,107,105,
    105,105,105,105,104,103,102,101,101,101,100,100,100,99,99,98,
    98,98,97,97,97,96,96,96,94,94,93,93,93,92,92,92,91,90,90,90
  };
  const int n2w4b1r2[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    132,131,130,130,130,130,129,129,129,129,128,127,127,127,127,127,
    126,125,125,125,124,124,123,122,122,120,120,120,120,120,120,120,
    120,119,119,119,118,118,118,118,118,117,117,116,116,115,115,115,
    114,114,113,113,112,112,112,112,112,111,111,111,110,110,109,108,
    108,108,108,108,106,106,106,106,105,104,104,104,104,104,103,103,
    103,102,102,101,101,100,99,99,98,98,97,95,94,94,93,93,93,92,91,
    90
  };
  const int n2w4b1r3[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    132,132,132,132,132,131,131,130,130,129,129,128,128,128,128,128,
    128,127,127,127,126,126,126,126,125,125,124,123,122,122,122,122,
    121,121,120,120,120,119,119,119,118,117,117,116,115,115,114,113,
    113,112,112,111,111,111,110,109,109,108,107,107,107,105,105,105,
    105,105,104,103,103,103,102,102,102,102,101,100,100,99,99,99,
    98,98,98,98,97,97,97,96,96,95,95,95,93,92,92,92,91,91,91,90
  };
  const int n2w4b1r4[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    132,132,132,132,131,131,131,130,130,130,129,129,128,128,128,127,
    127,127,127,126,125,125,124,124,124,123,123,121,121,121,120,120,
    119,119,118,118,118,117,117,117,117,116,116,116,115,115,114,114,
    114,114,114,113,113,113,113,112,112,112,111,107,106,105,105,105,
    105,105,104,103,103,102,102,102,102,101,100,100,99,99,99,97,97,
    96,96,96,96,95,95,94,94,93,93,92,92,92,92,92,91,91,90,90
  };
  const int n2w4b1r5[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    132,132,132,131,130,130,130,130,129,129,129,128,127,127,127,127,
    126,126,126,125,125,124,124,124,123,123,123,123,122,121,121,121,
    121,120,120,120,120,119,119,119,118,118,118,118,117,117,116,115,
    115,114,113,113,113,111,110,110,109,109,109,109,108,108,107,106,
    106,106,106,105,104,104,103,103,102,100,99,99,98,98,98,98,96,
    96,96,96,95,95,94,94,93,93,93,91,91,90,90,90,90,90,90,90
  };
  const int n2w4b1r6[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    131,130,130,129,129,128,128,127,127,127,126,126,125,123,122,122,
    122,121,121,121,120,120,120,120,119,119,118,117,117,116,116,116,
    115,115,115,114,114,114,113,113,113,113,113,112,111,111,111,110,
    110,109,109,109,108,108,108,108,108,108,107,107,106,105,104,104,
    104,104,103,103,103,102,102,102,102,101,101,101,100,100,99,99,
    99,99,98,98,98,97,97,97,96,94,94,93,93,93,92,92,92,91,91,90
  };
  const int n2w4b1r7[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    132,132,132,131,130,130,129,129,129,128,128,128,127,127,127,126,
    125,125,124,124,123,123,123,122,122,122,122,121,121,121,120,120,
    120,118,118,118,117,117,116,116,116,116,116,115,115,115,114,113,
    112,112,110,110,110,109,108,108,108,107,107,107,106,106,106,105,
    105,104,104,104,103,103,102,102,101,101,101,99,99,98,98,97,97,
    97,97,96,95,95,94,94,93,93,93,92,92,92,92,91,90,90,90,90
  };
  const int n2w4b1r8[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    132,132,131,131,130,129,129,129,128,127,127,126,126,125,125,124,
    124,124,123,122,122,121,120,120,119,119,119,118,118,118,117,117,
    117,117,117,116,115,115,114,114,113,113,113,111,110,110,110,109,
    108,108,108,107,107,107,107,107,106,105,105,104,103,103,103,102,
    102,102,101,101,101,100,100,100,100,99,98,98,98,98,97,97,97,96,
    96,96,96,95,95,95,94,93,93,93,93,93,92,92,92,91,90,90
  };
  const int n2w4b1r9[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    130,130,128,127,127,127,127,126,126,126,126,126,125,125,125,124,
    124,124,123,122,122,122,122,121,121,120,120,119,119,118,118,117,
    117,117,117,116,116,115,115,115,114,114,114,114,113,112,112,110,
    110,109,108,108,108,106,106,106,105,105,105,105,105,104,104,103,
    103,103,102,102,101,101,101,100,100,100,99,99,98,98,98,98,97,
    95,95,95,95,94,93,93,93,92,92,91,91,91,91,91,91,90,90,90
  };
  const int n2w4b2r0[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    163,162,161,159,159,156,155,153,152,150,150,150,149,148,141,140,
    139,138,137,137,137,136,134,134,134,133,132,130,130,128,127,126,
    126,125,124,123,121,121,120,119,119,116,116,115,115,115,115,114,
    111,108,107,106,105,104,102,102,100,100,99,98,97,96,96,90,90,
    89,89,89,87,86,83,82,81,78,76,74,74,74,72,70,69,68,68,66,65,65,
    64,64,63,62,62,62,62,61,60,60,59,58,58,58
  };
  const int n2w4b2r1[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    165,165,164,160,159,157,155,154,154,153,150,150,150,147,146,144,
    143,140,139,138,138,137,135,134,131,131,131,130,129,128,127,125,
    123,121,118,116,116,115,115,114,113,113,113,111,111,109,108,107,
    103,103,102,102,101,100,97,96,95,95,94,94,94,93,92,91,90,89,86,
    86,86,86,85,85,85,84,84,83,82,82,80,79,78,76,74,74,71,70,68,67,
    67,67,66,65,65,62,61,61,61,61,60,59
  };
  const int n2w4b2r2[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    165,165,162,159,156,155,155,154,152,151,150,150,149,149,148,147,
    146,145,145,144,143,143,142,141,141,138,134,134,133,132,131,128,
    127,126,125,124,123,122,121,121,121,120,119,114,114,112,112,110,
    109,108,107,107,107,106,102,102,99,99,98,97,97,95,95,95,94,94,
    93,93,92,91,90,88,87,87,86,83,82,80,80,79,78,77,76,76,70,69,68,
    68,68,66,65,62,61,60,60,59,58,58,58,57
  };
  const int n2w4b2r3[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    162,161,159,159,157,157,156,155,154,152,152,148,147,147,142,142,
    140,138,137,132,131,130,129,126,124,124,123,123,123,122,121,120,
    120,119,119,116,116,115,114,113,113,112,110,109,108,107,107,105,
    104,104,102,100,99,98,96,94,94,94,93,93,93,92,91,90,90,88,87,
    85,83,82,82,78,78,78,77,76,76,75,75,74,73,73,71,70,69,69,68,68,
    67,66,65,64,64,63,61,61,60,59,58,57
  };
  const int n2w4b2r4[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    165,165,164,164,161,161,156,155,155,154,154,154,154,151,151,150,
    149,149,148,146,144,142,142,141,139,139,138,136,136,135,134,133,
    132,132,131,131,131,131,130,130,129,129,124,124,123,120,118,118,
    118,117,116,116,116,116,114,114,107,106,105,105,104,102,101,101,
    98,97,96,96,94,91,91,91,88,86,86,86,84,79,79,78,78,77,76,74,71,
    71,70,69,67,65,65,64,60,60,59,59,59,59,59,59
  };
  const int n2w4b2r5[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    163,161,159,159,157,156,156,156,155,154,153,152,151,150,148,147,
    147,146,146,145,145,144,141,139,139,138,138,138,136,136,135,135,
    131,130,128,126,125,124,123,123,122,122,122,120,118,118,117,116,
    112,111,110,109,107,106,106,106,106,106,104,104,103,102,102,102,
    101,101,99,99,98,98,97,95,95,93,90,90,87,84,84,83,80,80,79,75,
    75,74,74,74,72,69,69,66,66,65,63,62,61,61,59,59
  };
  const int n2w4b2r6[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    164,164,163,159,158,154,153,152,152,152,152,150,150,147,147,145,
    145,145,144,143,143,142,141,140,140,140,139,139,138,137,136,135,
    131,128,125,124,122,120,119,118,118,118,117,114,114,114,112,111,
    111,110,110,109,109,107,107,107,107,107,106,102,101,101,100,99,
    98,97,96,96,96,95,94,93,92,91,89,87,86,86,84,83,80,79,78,78,74,
    73,73,73,68,68,68,67,66,66,65,65,64,61,60,59
  };
  const int n2w4b2r7[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    163,163,163,161,159,158,158,157,156,156,156,155,154,154,153,153,
    153,153,153,152,149,144,139,135,135,135,131,127,126,125,124,123,
    121,121,120,120,119,118,118,117,116,115,114,112,112,111,111,110,
    109,108,107,107,106,106,105,105,105,103,102,100,98,97,96,95,95,
    93,92,88,87,86,85,82,82,82,81,80,79,79,79,76,75,73,70,68,68,68,
    65,64,64,63,62,62,61,61,60,59,58,58,58,57
  };
  const int n2w4b2r8[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    164,161,161,161,159,159,159,159,158,158,157,157,157,156,155,154,
    151,150,150,149,149,148,148,148,148,147,147,146,146,145,143,139,
    139,138,137,136,136,136,134,133,131,131,128,128,127,127,127,126,
    121,120,120,119,118,118,118,114,112,112,112,111,110,110,107,106,
    104,104,103,102,101,99,97,94,94,94,91,91,89,87,83,82,82,80,79,
    79,77,76,72,72,72,70,69,69,68,67,67,64,62,61,58,57
  };
  const int n2w4b2r9[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    163,162,157,157,156,155,151,150,149,149,149,146,145,145,144,143,
    142,141,140,140,139,139,138,137,130,130,128,128,128,127,127,127,
    126,126,125,125,125,125,123,123,122,122,119,118,118,118,117,115,
    115,114,114,111,106,106,105,104,104,103,102,102,102,100,99,99,
    93,93,92,92,91,90,88,85,81,79,79,79,79,78,74,73,73,72,68,68,67,
    67,66,65,65,65,64,64,63,63,62,61,60,60,59,58
  };
  const int n2w4b3r0[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    209,206,205,201,197,191,191,190,187,187,186,184,183,182,182,182,
    178,176,174,172,171,171,171,169,166,164,162,161,161,156,155,155,
    152,149,147,144,142,136,132,131,125,124,122,121,117,117,115,113,
    113,110,104,103,101,101,100,96,96,95,95,92,87,83,77,77,76,72,
    70,70,70,68,68,66,65,62,59,56,55,54,51,49,47,44,43,43,42,41,41,
    40,39,37,34,34,31,31,30,26,26,20,14,13
  };
  const int n2w4b3r1[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    208,208,208,203,202,201,199,195,195,195,192,191,190,181,175,172,
    172,171,166,163,162,159,158,158,156,155,154,148,147,145,143,139,
    135,133,131,131,131,131,130,129,128,126,125,123,123,122,122,121,
    120,118,117,117,116,110,106,103,103,99,97,94,92,88,86,86,83,81,
    79,78,77,77,77,76,71,71,69,62,61,59,58,57,57,57,57,54,46,46,43,
    42,38,37,35,33,31,23,21,17,14,14,14,13
  };
  const int n2w4b3r2[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    206,205,200,200,199,199,197,197,194,193,193,193,191,188,185,185,
    184,182,178,175,172,170,167,165,161,161,161,159,159,159,158,155,
    154,153,153,153,149,146,143,141,141,139,137,135,130,128,126,125,
    122,120,120,119,118,115,113,109,109,109,108,107,104,104,103,103,
    101,99,97,94,90,90,90,87,86,86,82,79,77,74,67,63,54,48,48,46,
    45,44,37,35,35,34,34,27,25,23,23,23,19,17,16,14
  };
  const int n2w4b3r3[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    201,201,200,199,198,197,196,195,195,194,190,188,187,184,182,181,
    181,180,179,177,172,171,169,165,165,163,158,154,154,153,153,148,
    148,144,142,138,137,131,129,125,123,122,118,117,117,116,115,113,
    109,105,105,104,103,101,100,96,89,87,86,84,84,82,78,78,77,76,
    72,71,71,69,69,69,67,66,64,64,63,62,58,56,53,52,50,49,45,45,40,
    39,37,37,33,28,25,24,22,22,16,15,15,13
  };
  const int n2w4b3r4[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    204,204,202,202,200,200,197,194,194,191,189,187,181,180,180,179,
    179,177,176,175,174,173,169,169,168,167,161,158,151,145,143,139,
    136,136,135,135,134,133,131,130,130,128,124,124,123,122,120,116,
    113,112,111,110,109,109,106,105,104,103,102,101,99,99,97,96,81,
    81,78,78,77,75,73,72,68,67,64,64,62,62,55,54,51,47,45,45,35,34,
    34,32,32,31,30,28,26,25,23,22,20,17,15,13
  };
  const int n2w4b3r5[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    209,207,205,204,204,202,201,200,200,197,194,193,188,187,185,180,
    176,168,166,161,159,159,156,154,154,148,145,145,143,138,135,132,
    128,125,124,122,121,118,116,114,112,112,108,106,105,105,104,101,
    97,95,94,93,87,85,85,72,72,71,70,69,68,64,63,63,62,61,61,58,55,
    54,53,52,52,51,50,48,48,47,45,43,40,37,34,33,27,27,27,24,24,23,
    22,22,20,20,18,17,16,15,14,13
  };
  const int n2w4b3r6[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    209,207,206,201,201,200,199,198,194,191,190,188,186,185,182,181,
    179,178,178,174,172,170,170,170,160,159,155,154,144,143,142,136,
    135,134,132,130,128,126,126,122,118,117,116,113,112,106,106,105,
    103,103,101,96,95,90,90,89,82,81,81,80,78,77,76,74,72,71,71,70,
    68,66,64,62,62,61,60,58,57,57,57,57,54,48,46,44,42,36,33,30,29,
    25,24,23,23,22,22,21,17,14,13,13
  };
  const int n2w4b3r7[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    209,209,207,205,199,193,193,189,188,186,181,180,178,175,174,170,
    169,169,168,166,164,161,157,156,155,155,153,153,152,152,148,147,
    145,145,144,144,141,133,133,133,126,125,123,119,118,117,116,110,
    109,108,106,103,100,99,98,96,95,94,92,90,87,86,84,79,77,74,72,
    72,71,71,62,61,59,56,55,55,54,53,48,47,44,42,42,41,39,38,37,36,
    32,29,29,27,27,25,24,24,22,21,14,14
  };
  const int n2w4b3r8[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    209,207,205,205,203,202,202,201,199,195,193,192,192,191,187,184,
    183,182,178,177,175,171,164,162,155,154,153,152,150,148,146,144,
    144,142,136,135,134,134,132,127,127,125,124,123,122,120,119,114,
    107,104,96,96,94,94,93,89,87,86,86,84,83,82,81,81,78,77,77,76,
    75,70,67,67,64,57,56,51,47,46,42,41,41,41,41,41,40,40,40,39,38,
    35,32,31,27,25,23,23,23,17,17,14
  };
  const int n2w4b3r9[] = {
    1000, // Capacity
    100, // Number of items
    // Size of items (sorted)
    206,206,206,206,205,205,204,200,198,196,193,192,189,188,188,187,
    184,178,178,176,176,172,172,171,169,168,168,167,162,158,156,153,
    152,151,151,151,145,141,139,139,137,136,129,127,124,122,118,115,
    115,115,111,111,110,109,109,103,102,102,99,98,98,97,94,91,91,
    90,86,85,83,81,79,78,78,74,74,73,73,71,67,64,59,58,57,51,50,50,
    50,49,46,44,43,39,33,30,27,26,23,21,20,19
  };
  const int n3w1b1r0[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    395,395,395,395,395,394,394,394,393,393,393,393,393,393,392,390,
    389,388,388,388,387,386,386,385,384,383,383,382,380,380,379,379,
    378,378,377,375,375,374,374,373,372,372,372,371,370,368,368,367,
    367,366,366,365,365,363,362,361,360,360,360,359,357,357,356,355,
    355,350,350,349,348,348,348,347,347,347,347,347,346,346,346,346,
    345,345,344,344,344,343,343,343,343,342,341,341,340,338,337,336,
    336,335,335,335,334,333,333,332,331,330,329,329,328,328,327,327,
    326,326,325,324,323,323,322,322,321,321,320,320,320,320,316,316,
    316,315,315,315,313,312,312,311,309,309,308,306,305,305,305,305,
    303,302,302,302,300,300,299,298,298,298,297,297,296,296,295,295,
    293,293,291,291,290,290,290,290,287,286,286,286,286,282,281,281,
    281,280,280,279,275,275,274,274,274,274,273,272,272,271,271,270,
    270,269,269,269,268,267,266,266
  };
  const int n3w1b1r1[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    394,393,393,392,391,391,390,389,389,389,387,387,387,387,387,387,
    385,384,383,382,382,382,381,380,380,380,379,378,378,378,378,377,
    376,376,374,373,373,372,371,371,371,371,370,370,370,369,369,369,
    368,368,367,367,365,365,364,364,364,363,363,362,362,360,360,360,
    359,359,358,357,356,356,355,354,354,353,353,352,351,349,349,348,
    347,346,346,343,343,342,342,342,341,341,340,340,339,339,338,338,
    338,337,336,336,335,333,333,332,332,331,329,328,326,326,326,325,
    325,325,323,323,323,322,322,321,320,319,319,318,318,315,315,314,
    314,313,313,311,310,310,309,309,309,309,308,308,307,306,306,306,
    305,305,302,301,299,299,299,299,298,297,296,296,296,296,295,294,
    294,294,292,292,291,290,290,289,288,286,285,285,285,284,283,282,
    282,282,280,280,280,279,278,277,277,277,277,275,275,275,274,273,
    273,272,272,271,270,270,269,268
  };
  const int n3w1b1r2[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    396,395,395,395,394,394,392,392,391,391,390,389,389,388,387,387,
    385,385,385,385,384,384,383,383,383,382,381,380,379,378,378,378,
    377,374,374,374,373,373,372,371,370,370,370,364,364,363,363,363,
    362,362,360,359,359,357,357,356,356,356,355,354,354,354,353,353,
    353,353,352,352,351,348,347,346,346,346,346,345,344,344,343,343,
    342,342,341,340,339,339,338,338,338,338,338,337,336,336,336,336,
    335,334,334,334,333,333,332,331,329,328,328,328,327,327,327,327,
    326,324,323,322,321,320,319,319,316,315,313,313,312,312,311,310,
    310,309,308,308,308,307,305,305,304,304,304,304,303,302,301,300,
    299,299,298,298,297,297,296,295,295,293,292,292,292,291,291,290,
    289,288,288,288,287,284,284,284,283,282,282,281,280,279,279,279,
    278,278,278,278,277,277,275,275,275,275,274,273,273,271,271,270,
    269,269,269,269,268,267,266,266
  };
  const int n3w1b1r3[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    396,395,394,393,393,392,391,390,389,388,387,387,386,386,386,385,
    385,382,381,380,379,379,378,378,378,378,377,377,377,377,376,376,
    374,373,373,370,369,368,368,368,368,367,367,367,367,367,366,366,
    366,366,365,364,363,362,361,361,361,361,359,359,358,357,357,356,
    356,355,353,352,350,349,348,348,348,348,348,347,347,347,346,345,
    345,345,344,344,343,343,342,342,342,341,340,339,336,336,336,336,
    335,335,335,334,334,333,331,330,328,328,328,327,327,327,325,324,
    324,323,322,322,322,321,321,320,320,320,320,320,318,317,317,315,
    315,315,315,314,314,313,313,312,311,309,309,309,309,308,307,307,
    306,305,305,304,304,303,302,302,301,301,301,301,300,299,299,298,
    298,297,296,296,294,293,293,292,291,290,290,289,289,288,288,288,
    286,286,284,284,284,283,283,282,281,280,279,275,275,274,273,272,
    271,270,269,269,269,268,267,267
  };
  const int n3w1b1r4[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    396,396,396,396,395,394,394,393,393,393,392,392,392,391,391,391,
    389,388,388,388,387,387,385,385,384,384,384,383,383,383,382,382,
    382,382,381,380,380,379,378,378,377,375,375,375,374,371,370,370,
    369,368,368,365,365,364,363,362,361,361,360,359,357,356,355,354,
    353,353,353,352,352,352,351,351,351,350,350,349,348,347,347,346,
    345,345,345,344,343,342,341,340,340,339,338,338,338,337,336,335,
    335,335,334,334,332,331,331,331,330,330,329,327,327,326,326,325,
    325,325,325,324,323,323,322,322,321,319,318,316,316,315,314,313,
    313,312,311,311,310,310,310,310,309,309,306,304,304,303,303,302,
    302,301,301,300,299,299,297,297,297,293,293,293,291,291,290,290,
    290,288,287,286,286,285,284,284,283,283,283,283,282,282,282,280,
    279,278,278,278,278,278,277,276,276,275,275,274,273,273,271,271,
    271,269,269,268,268,267,266,266
  };
  const int n3w1b1r5[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    396,396,396,395,394,392,391,390,389,386,386,386,385,383,383,382,
    381,380,379,379,378,377,377,375,375,375,375,374,374,373,373,373,
    372,372,371,370,370,369,369,368,367,367,367,367,367,367,365,365,
    364,362,362,362,361,361,360,359,357,357,357,357,356,356,354,354,
    353,353,351,350,349,349,349,348,348,348,347,346,346,344,342,342,
    342,340,338,338,338,337,337,337,336,336,336,335,335,335,335,335,
    334,334,334,333,333,333,332,330,328,328,328,328,327,327,327,327,
    326,325,325,324,323,323,322,322,321,321,318,318,318,317,317,317,
    316,316,316,315,315,315,315,313,313,313,312,311,311,310,310,310,
    309,307,307,306,306,306,306,305,304,302,302,301,299,299,297,297,
    297,296,293,290,290,289,289,288,288,287,287,286,285,285,283,283,
    283,283,282,281,280,279,277,276,275,274,274,274,274,273,272,270,
    270,270,268,268,267,267,267,266
  };
  const int n3w1b1r6[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    396,395,394,394,394,394,394,394,393,393,393,392,392,392,391,389,
    389,388,387,387,386,385,384,384,383,382,382,380,380,380,379,379,
    379,377,377,377,377,376,376,376,374,374,371,370,370,369,369,368,
    368,368,367,367,366,362,362,361,361,360,360,359,359,359,359,358,
    357,357,356,356,356,355,355,355,355,353,352,352,351,351,351,350,
    350,349,349,349,348,347,346,345,345,345,344,344,343,343,343,342,
    342,342,341,338,337,337,336,336,336,335,334,333,333,332,331,330,
    330,328,327,326,326,326,325,325,324,323,323,321,321,320,319,319,
    318,318,317,316,314,314,313,313,312,311,311,310,310,308,307,307,
    304,303,302,301,300,296,296,294,293,293,293,292,292,291,291,290,
    289,289,289,288,288,287,286,285,285,284,283,283,283,282,282,280,
    280,280,280,279,279,279,278,278,276,275,274,273,273,272,271,270,
    270,269,268,267,267,267,266,266
  };
  const int n3w1b1r7[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    396,395,395,394,394,392,392,392,389,388,387,386,385,385,384,384,
    383,383,383,382,382,381,379,378,378,378,375,375,375,375,370,370,
    370,370,368,366,365,363,363,361,361,360,360,359,359,359,359,356,
    356,354,354,353,353,352,352,351,350,349,348,348,348,345,345,344,
    343,343,343,343,342,342,341,340,339,339,339,338,338,336,336,335,
    334,333,331,330,330,330,329,327,327,326,325,325,325,324,323,322,
    322,322,322,321,321,321,321,320,320,319,319,318,318,318,317,317,
    317,317,317,316,316,314,313,313,313,311,310,310,308,308,307,306,
    305,305,305,304,304,304,303,302,302,301,301,301,299,299,297,295,
    295,295,294,294,293,292,290,290,289,289,289,289,288,287,287,284,
    283,283,283,283,281,281,280,280,280,280,280,279,279,279,279,278,
    278,278,278,276,276,276,275,275,275,275,274,273,273,271,271,271,
    271,270,270,270,269,269,267,266
  };
  const int n3w1b1r8[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    396,395,394,392,391,391,390,390,390,389,388,388,388,387,387,387,
    387,386,386,386,384,384,382,381,381,381,381,381,380,379,378,378,
    377,376,376,375,375,374,373,371,370,369,369,367,367,367,366,366,
    366,364,364,364,364,362,362,361,360,359,358,357,357,355,355,354,
    354,354,353,352,351,350,349,349,348,348,347,347,347,346,346,346,
    344,341,341,341,341,340,340,340,339,338,338,336,336,335,335,334,
    334,334,334,333,332,332,329,329,327,326,326,325,324,324,324,324,
    324,323,323,323,322,321,321,320,320,320,319,317,316,315,313,313,
    313,312,312,311,311,311,310,310,308,308,308,307,306,306,306,305,
    305,305,304,300,300,300,299,299,297,296,295,294,294,294,293,293,
    292,292,291,290,290,290,289,288,286,285,285,284,284,283,283,282,
    281,281,280,280,279,279,277,277,277,276,275,275,275,274,274,274,
    274,271,271,270,269,269,268,267
  };
  const int n3w1b1r9[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    396,394,394,394,394,394,393,391,391,390,390,389,389,388,387,386,
    386,386,385,384,384,384,384,383,383,382,380,379,378,378,377,376,
    376,376,375,375,374,374,373,371,371,370,370,369,369,369,367,366,
    365,363,363,363,362,361,360,359,359,357,357,356,354,354,351,351,
    351,350,350,350,349,349,349,348,347,346,346,345,345,344,343,343,
    342,342,340,340,339,337,337,337,337,336,336,335,334,334,333,333,
    333,333,333,332,332,332,331,330,330,330,329,329,329,328,328,327,
    325,324,324,323,322,322,322,322,320,319,319,318,315,314,314,313,
    313,313,313,312,312,310,309,308,308,307,306,306,305,304,304,304,
    301,299,299,299,298,298,298,297,297,297,296,294,294,294,294,294,
    293,292,291,291,290,290,289,289,288,286,286,285,284,280,280,279,
    278,277,277,276,275,275,275,274,273,272,272,271,271,270,270,270,
    269,269,268,267,266,266,266,266
  };
  const int n3w1b2r0[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    495,494,493,490,489,488,487,486,485,485,483,481,479,477,475,474,
    473,471,471,470,469,464,463,459,455,452,445,445,445,444,444,442,
    439,438,436,435,435,435,435,433,429,429,428,428,422,422,421,418,
    417,417,417,411,410,407,405,404,401,400,398,398,398,397,395,393,
    391,389,389,385,384,378,377,376,375,375,375,373,373,369,368,362,
    362,359,358,354,353,352,352,351,349,346,344,342,341,337,337,336,
    335,335,334,334,334,333,330,330,330,330,328,326,325,324,324,320,
    318,317,317,316,316,316,315,312,308,306,304,302,299,296,295,292,
    292,290,284,282,278,276,276,271,270,270,270,269,268,263,261,259,
    258,257,254,252,252,250,247,246,244,244,243,243,242,242,233,232,
    231,230,228,224,223,223,220,220,213,213,212,209,209,206,204,201,
    200,199,197,195,195,194,194,193,192,189,188,188,186,184,182,179,
    179,175,173,173,172,171,169,168
  };
  const int n3w1b2r1[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    495,493,493,487,486,486,483,483,481,478,477,476,474,473,472,472,
    472,471,470,469,467,464,464,462,461,458,456,454,451,450,449,448,
    444,443,441,440,437,433,432,432,430,429,428,425,421,419,418,417,
    417,411,411,409,409,408,405,405,403,401,400,399,397,393,390,388,
    387,387,387,385,384,383,382,381,379,378,376,375,374,374,371,370,
    367,364,358,355,355,353,353,350,349,346,346,345,342,341,339,338,
    336,335,334,334,331,331,330,326,326,325,324,321,320,319,316,316,
    315,313,313,311,311,311,311,309,308,307,307,306,303,302,302,302,
    298,298,297,297,295,294,291,288,284,283,283,282,281,281,280,277,
    277,276,273,272,270,265,264,264,264,263,259,253,253,251,250,247,
    247,245,240,237,237,236,232,232,231,231,227,222,221,213,213,210,
    203,203,202,201,201,196,195,193,193,191,189,188,188,185,182,181,
    179,179,177,176,175,172,169,169
  };
  const int n3w1b2r2[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    491,488,487,479,479,474,473,470,469,469,468,468,465,463,462,462,
    459,457,457,453,451,449,448,446,444,442,440,438,433,433,432,430,
    427,426,426,423,421,417,415,413,413,411,410,410,410,409,408,408,
    407,406,404,403,402,401,400,399,397,391,391,389,388,387,387,387,
    386,384,382,377,377,375,373,373,373,372,372,369,366,365,364,363,
    363,363,359,357,356,351,350,350,350,348,347,346,338,335,333,331,
    330,330,328,328,326,325,323,322,322,320,317,316,311,307,306,306,
    305,301,300,297,296,296,292,289,289,288,285,276,275,274,273,272,
    268,266,265,264,262,257,257,256,255,255,255,255,252,249,248,245,
    243,243,241,237,236,236,235,232,231,228,228,226,226,225,224,223,
    223,223,221,218,216,208,206,206,205,204,203,202,202,202,196,194,
    193,193,193,190,190,189,189,188,187,186,183,182,181,179,179,178,
    172,171,171,171,169,169,168,167
  };
  const int n3w1b2r3[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    494,492,491,488,487,483,480,479,479,478,476,476,476,474,472,469,
    466,466,460,459,459,456,453,452,446,446,446,442,442,442,437,434,
    430,429,425,422,422,421,417,416,412,411,405,405,402,400,399,399,
    394,387,387,387,387,386,385,379,378,376,376,373,372,372,371,371,
    371,371,370,369,367,365,361,361,360,359,356,356,355,353,352,352,
    351,348,348,347,346,346,346,346,345,343,343,342,341,341,340,338,
    337,337,331,330,330,329,326,322,321,317,316,315,311,309,308,307,
    305,304,303,299,299,298,295,294,294,292,288,284,280,279,279,279,
    278,277,276,274,274,271,268,267,267,266,265,262,262,260,259,258,
    252,248,247,246,245,242,240,238,232,231,231,229,229,228,226,225,
    224,224,222,220,216,216,215,214,212,209,205,201,200,200,199,198,
    197,196,194,194,191,190,190,186,186,185,184,183,181,181,179,179,
    177,177,177,175,174,169,168,168
  };
  const int n3w1b2r4[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    492,489,488,484,484,483,482,481,480,478,477,476,474,474,473,472,
    469,469,468,468,466,462,460,458,458,455,453,451,450,449,449,448,
    446,445,442,442,440,439,437,435,435,435,435,432,432,430,428,425,
    423,421,421,420,417,416,411,408,406,406,406,404,403,403,403,402,
    402,399,399,398,397,394,393,392,391,391,390,389,385,384,382,376,
    368,367,367,366,365,362,361,360,358,356,354,352,351,348,348,348,
    345,343,340,336,334,334,334,333,328,328,327,326,325,321,320,317,
    315,315,315,314,313,311,308,308,308,305,302,302,301,300,295,295,
    293,293,293,292,292,291,286,284,284,281,281,273,273,272,271,267,
    267,267,266,265,265,264,263,262,261,258,258,255,253,242,241,240,
    240,239,238,236,235,234,233,231,228,224,224,223,221,219,217,214,
    212,210,205,202,201,199,197,197,197,194,189,187,187,186,185,184,
    183,179,178,175,173,172,171,168
  };
  const int n3w1b2r5[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    495,492,487,483,483,481,481,479,476,471,470,465,458,457,454,453,
    452,452,452,450,450,448,444,440,439,439,437,437,435,434,432,430,
    429,429,428,428,427,425,424,424,422,419,419,417,414,412,411,408,
    406,406,405,403,403,397,396,395,392,390,390,389,389,386,384,383,
    382,382,380,380,379,378,378,377,374,371,364,361,361,358,355,351,
    350,350,350,349,348,348,346,343,340,339,333,333,331,331,329,328,
    327,323,322,320,319,317,314,313,313,311,311,311,309,309,306,297,
    295,295,293,292,292,287,283,282,282,281,280,280,280,277,276,275,
    273,272,272,272,269,266,265,264,261,260,259,259,258,256,256,255,
    254,251,247,247,245,240,239,239,239,238,236,235,232,230,228,227,
    227,227,223,222,222,220,220,220,215,214,210,208,206,205,201,201,
    200,199,198,193,192,192,191,189,189,187,185,184,182,181,181,179,
    179,173,173,173,171,169,167,167
  };
  const int n3w1b2r6[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    495,494,491,490,490,490,489,488,486,485,480,479,479,472,469,467,
    467,465,462,461,461,461,460,457,453,451,451,449,447,444,444,443,
    442,442,437,436,435,435,435,432,432,431,430,430,429,429,429,425,
    423,422,421,419,418,415,411,407,404,402,401,400,395,394,394,391,
    385,384,383,379,377,376,374,373,372,370,369,368,364,363,361,361,
    361,359,358,358,357,357,353,351,350,346,344,344,342,342,342,341,
    339,339,336,333,332,331,330,330,326,325,323,317,313,308,306,305,
    300,297,296,293,292,290,287,287,286,282,281,277,277,273,273,272,
    272,271,267,265,261,259,258,254,254,254,253,253,249,248,248,247,
    247,246,246,246,244,243,243,242,241,241,240,240,240,239,236,235,
    234,234,233,233,230,229,228,226,221,221,220,217,215,215,210,208,
    206,204,203,202,200,198,197,197,191,191,184,181,181,180,179,175,
    174,173,173,172,171,171,169,168
  };
  const int n3w1b2r7[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    495,493,492,487,487,485,482,480,480,479,475,475,473,473,469,469,
    465,464,460,459,457,456,455,454,453,451,450,449,445,443,441,439,
    438,435,433,431,427,423,423,421,421,420,420,417,415,414,414,411,
    411,408,406,404,401,399,395,395,394,392,391,390,390,386,384,384,
    380,378,377,377,374,373,370,369,369,369,368,367,366,363,360,359,
    354,353,350,349,348,347,346,346,344,342,341,337,336,334,332,332,
    332,329,328,327,323,321,321,317,317,316,315,313,310,310,306,305,
    305,303,303,301,301,300,297,296,293,292,291,291,290,289,286,286,
    286,284,283,282,282,282,282,282,282,280,279,276,275,272,272,270,
    270,270,260,256,256,255,254,253,245,244,240,236,235,234,234,234,
    233,230,228,227,226,226,225,222,222,221,217,217,214,211,208,207,
    207,206,204,203,203,202,202,202,200,199,198,197,192,189,187,186,
    183,178,177,177,174,170,170,168
  };
  const int n3w1b2r8[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    495,490,489,487,487,486,486,485,483,482,481,477,477,477,475,469,
    467,465,465,461,461,457,454,453,452,449,447,445,443,442,441,439,
    435,433,433,433,432,432,432,429,428,428,425,424,421,419,418,418,
    414,410,409,409,409,408,407,406,406,404,403,400,398,398,397,396,
    394,394,392,392,390,388,388,383,382,381,369,369,368,365,364,362,
    360,360,359,357,355,351,350,350,344,341,340,338,337,332,331,328,
    327,327,325,324,316,315,313,311,310,309,308,308,307,301,299,298,
    297,296,295,295,288,283,280,279,279,278,278,278,277,277,276,276,
    274,274,273,270,269,268,267,266,264,264,264,263,263,261,260,258,
    257,257,255,251,251,249,248,242,242,241,241,241,241,238,234,231,
    230,229,229,227,227,227,224,222,219,218,218,215,213,212,207,207,
    205,204,203,203,195,192,191,188,188,187,187,187,184,181,180,180,
    180,180,179,176,175,172,171,171
  };
  const int n3w1b2r9[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    495,494,493,493,493,492,489,482,482,478,478,475,473,473,472,471,
    469,463,461,461,459,455,454,452,448,444,444,442,440,439,439,436,
    434,433,432,431,429,425,423,423,422,422,420,420,417,416,412,411,
    411,410,410,409,408,403,401,401,400,399,397,394,394,393,392,392,
    390,389,387,386,385,384,384,382,380,380,376,375,374,372,372,370,
    370,368,366,357,353,353,353,350,349,346,345,345,345,345,342,342,
    338,332,331,325,324,324,322,321,317,314,314,312,312,311,310,308,
    307,307,307,306,301,299,299,296,295,294,293,290,288,287,287,286,
    285,283,283,280,279,278,275,274,272,271,271,270,269,268,266,266,
    265,264,263,257,256,248,247,242,240,236,233,233,233,229,227,222,
    219,219,217,217,212,212,209,208,207,206,205,205,205,205,205,203,
    203,201,199,198,198,197,192,192,192,191,189,188,184,184,183,182,
    182,179,179,178,176,175,168,167
  };
  const int n3w1b3r0[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    626,624,624,624,622,620,615,613,608,607,601,596,595,595,595,591,
    591,586,583,582,582,579,579,573,572,569,567,566,557,556,554,554,
    553,550,550,546,545,545,543,540,539,535,535,532,527,526,520,515,
    513,509,506,504,502,500,497,492,491,490,489,485,484,484,478,474,
    456,452,450,448,441,441,440,436,428,427,424,422,422,420,419,414,
    413,410,410,408,406,405,396,388,386,378,369,366,365,364,345,345,
    341,337,335,330,324,323,320,316,312,303,302,296,293,291,288,286,
    284,282,282,282,282,279,272,271,265,258,256,254,250,249,248,240,
    234,232,231,226,225,225,221,217,216,212,208,206,204,201,200,200,
    200,199,194,194,189,189,185,184,181,180,177,176,171,163,160,160,
    157,155,149,141,137,132,130,127,126,125,125,122,121,120,118,114,
    114,112,111,103,94,93,88,86,80,77,77,77,73,69,62,57,55,55,55,
    51,49,47,44,39
  };
  const int n3w1b3r1[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    623,623,619,615,614,614,613,611,603,599,599,597,586,569,568,567,
    564,563,562,561,559,553,544,544,542,539,537,537,532,528,527,517,
    517,509,506,494,494,489,489,487,486,485,484,483,474,473,472,471,
    471,463,462,460,458,456,451,450,447,447,446,435,431,430,422,417,
    415,412,410,407,406,405,399,399,393,392,392,386,385,381,381,380,
    379,378,376,367,362,362,361,360,356,354,348,346,342,341,340,339,
    338,336,328,328,324,318,318,315,313,312,311,308,300,298,296,296,
    295,290,285,282,282,282,279,278,278,269,260,259,258,255,254,254,
    244,227,226,225,225,223,218,217,216,214,207,206,206,205,204,203,
    203,202,200,195,193,190,188,186,183,183,181,181,180,179,179,172,
    171,170,167,166,165,160,158,155,149,148,148,139,138,136,132,130,
    130,129,128,127,125,120,119,118,118,115,109,107,104,101,95,91,
    90,76,60,55,53,45,39,37
  };
  const int n3w1b3r2[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    624,624,619,617,617,616,614,613,609,607,590,584,580,580,578,577,
    576,576,574,570,568,566,565,561,554,552,552,549,544,543,534,534,
    531,530,516,515,511,507,507,501,501,501,499,497,496,496,490,488,
    487,486,485,482,473,470,466,462,461,458,458,453,452,451,450,447,
    443,443,442,435,435,431,430,425,415,412,410,408,406,404,402,401,
    396,395,389,388,388,387,387,387,386,384,379,379,379,376,375,373,
    370,367,367,363,359,359,357,341,335,333,332,326,312,312,310,306,
    300,299,299,293,283,278,277,275,272,271,270,261,260,258,257,257,
    256,256,253,249,236,231,215,211,209,209,206,206,196,194,189,188,
    186,186,184,181,172,170,169,167,159,155,152,150,150,149,148,147,
    146,140,140,138,134,130,129,128,121,119,119,116,113,107,103,102,
    94,93,90,89,87,87,85,85,78,76,74,73,72,72,67,65,64,64,63,60,46,
    46,39,35
  };
  const int n3w1b3r3[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    625,619,619,618,614,613,612,611,609,605,602,598,598,590,589,587,
    586,585,579,578,576,566,566,564,563,563,561,558,549,542,542,541,
    536,535,529,522,515,512,501,501,500,498,496,495,494,492,492,487,
    485,481,479,466,466,466,465,464,462,454,453,450,448,442,441,440,
    440,439,437,436,436,432,432,422,422,421,417,412,408,408,393,384,
    377,377,376,375,373,373,372,371,371,369,365,359,358,353,353,342,
    334,327,324,324,321,320,314,312,311,309,308,296,296,293,291,288,
    285,278,270,269,265,262,262,261,260,259,256,254,251,248,244,237,
    235,235,234,229,229,227,225,223,222,222,216,212,208,207,206,205,
    192,191,181,181,180,179,175,175,164,162,162,159,158,157,156,151,
    148,148,146,143,139,139,134,129,129,128,119,116,109,105,95,93,
    87,83,83,83,80,78,78,77,76,74,72,65,64,63,62,56,55,55,53,39,38,
    37,36,36
  };
  const int n3w1b3r4[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    627,626,618,615,614,613,609,604,603,603,600,599,595,594,591,585,
    580,576,571,567,565,562,559,559,555,554,553,551,548,546,543,542,
    539,537,536,533,533,533,530,527,525,521,520,519,519,519,519,518,
    518,516,509,508,499,498,494,492,489,489,482,475,462,460,450,448,
    443,441,440,439,438,438,436,435,433,429,427,426,424,421,420,410,
    409,403,403,393,391,381,378,378,374,372,366,364,364,354,352,349,
    349,347,346,341,339,339,336,332,331,331,325,321,320,320,318,318,
    315,310,302,299,298,297,296,295,293,282,281,267,261,252,252,248,
    246,244,233,232,228,221,217,216,214,213,210,209,208,207,202,200,
    200,196,193,192,190,190,188,183,183,179,179,175,171,165,152,151,
    142,135,134,133,132,127,126,124,121,120,116,116,109,108,107,104,
    104,101,95,92,91,89,86,84,83,81,72,68,67,64,60,58,52,49,47,43,
    38,38,37,37
  };
  const int n3w1b3r5[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    627,621,621,613,610,604,604,594,592,582,575,575,575,574,572,571,
    571,570,564,564,563,560,557,556,556,548,547,540,532,523,523,519,
    518,517,517,514,514,510,505,503,501,494,492,487,480,479,477,477,
    473,473,472,467,464,464,459,455,454,452,451,449,449,447,445,440,
    438,430,429,427,424,420,420,417,415,411,409,408,407,404,401,390,
    385,378,369,361,361,359,356,352,347,343,343,341,338,337,335,334,
    322,321,317,316,308,307,305,301,301,289,289,284,283,277,277,271,
    270,269,269,267,267,267,259,256,253,249,247,245,242,242,237,233,
    233,229,227,224,219,219,217,215,215,209,208,208,202,199,199,198,
    194,193,179,176,172,165,160,159,158,148,145,139,139,139,138,137,
    137,133,122,120,120,115,114,112,110,109,109,108,102,101,99,92,
    86,86,85,80,80,77,76,74,73,70,70,67,64,63,60,58,54,54,46,41,37,
    36,35,35
  };
  const int n3w1b3r6[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    626,622,621,619,614,612,609,608,608,605,600,595,575,572,571,571,
    567,564,563,554,552,551,549,548,544,542,542,538,538,535,533,529,
    527,524,524,515,510,510,509,504,502,501,496,490,488,481,480,478,
    475,470,469,468,458,454,451,446,446,442,438,436,432,430,422,414,
    413,412,411,408,397,389,386,386,385,383,382,373,372,372,371,369,
    366,364,362,361,360,360,356,354,351,348,343,338,334,331,326,325,
    323,322,320,320,320,320,317,317,316,308,308,305,301,300,299,298,
    297,295,295,289,287,285,285,282,281,279,279,266,259,257,257,254,
    250,250,249,248,244,243,237,236,225,223,222,219,216,215,210,209,
    199,199,196,189,186,185,184,183,182,182,181,176,169,169,168,168,
    167,158,156,155,141,141,136,135,132,131,131,131,125,121,118,116,
    116,115,107,96,95,93,93,88,84,84,78,78,75,72,65,62,62,60,53,51,
    43,43,36,35
  };
  const int n3w1b3r7[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    627,626,619,616,611,611,611,610,609,608,607,592,592,582,582,579,
    575,571,571,566,565,561,558,549,543,542,542,537,530,527,520,514,
    513,512,511,505,495,495,493,493,482,481,480,479,473,466,466,460,
    460,459,458,458,455,453,445,441,433,431,425,424,418,415,409,409,
    407,407,401,400,399,397,393,393,385,380,379,372,369,360,353,351,
    347,338,337,330,316,315,309,309,301,300,299,298,297,296,292,287,
    287,284,283,274,272,270,269,269,266,264,263,261,258,249,247,238,
    235,235,234,234,234,233,218,217,211,210,206,204,202,196,193,188,
    188,187,187,180,180,178,177,174,173,168,167,165,162,159,158,157,
    157,151,150,148,146,143,143,143,139,137,136,132,125,123,121,120,
    114,114,114,106,105,104,101,101,101,99,96,95,93,92,92,89,88,87,
    87,87,85,84,83,82,79,78,69,65,64,62,62,58,55,53,43,42,39,38,37,
    35
  };
  const int n3w1b3r8[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    619,616,616,613,613,612,607,607,604,601,590,585,579,578,569,566,
    561,561,559,557,551,551,550,546,546,543,535,534,528,524,520,519,
    507,505,505,504,503,502,502,501,500,494,492,486,484,481,476,473,
    473,470,470,468,467,465,456,455,450,445,442,442,442,437,435,433,
    432,432,431,426,421,420,417,407,407,403,398,396,393,390,385,380,
    380,379,375,373,371,368,367,357,355,351,346,346,345,342,339,339,
    338,334,332,332,331,326,325,317,316,310,307,302,300,300,298,296,
    295,293,292,288,286,285,279,271,271,270,267,265,260,259,256,252,
    245,241,240,231,230,223,222,222,220,216,215,213,210,205,202,197,
    197,194,189,185,184,181,180,174,173,170,162,161,159,158,150,139,
    135,134,133,131,127,126,126,123,121,121,119,117,112,108,101,98,
    98,91,89,87,87,86,83,82,78,78,67,56,55,55,54,54,52,45,43,41,41,
    40,39,35
  };
  const int n3w1b3r9[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    627,623,620,617,616,611,598,594,594,590,589,584,581,579,575,569,
    568,566,563,562,562,554,554,554,553,552,548,548,544,535,534,532,
    531,530,528,523,518,516,516,512,508,500,496,496,496,494,494,494,
    492,491,485,483,481,479,477,476,475,467,461,459,455,454,448,448,
    444,440,439,439,438,437,436,434,431,430,423,422,417,415,409,408,
    408,404,400,398,398,398,396,396,394,387,385,384,379,378,378,374,
    373,372,368,367,360,359,353,348,348,342,337,331,331,329,329,324,
    319,316,315,315,314,312,310,308,308,308,306,297,294,288,284,284,
    283,277,268,266,266,264,258,253,252,248,242,236,235,231,229,229,
    227,226,224,220,216,214,210,202,201,198,193,192,185,185,184,177,
    175,173,173,168,166,163,149,148,148,145,145,138,137,135,134,133,
    130,118,116,108,103,102,102,101,96,95,90,83,82,80,80,71,68,64,
    62,61,60,54,53,52
  };
  const int n3w2b1r0[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    240,240,240,240,239,238,238,238,237,236,236,235,234,234,234,234,
    234,232,232,232,232,231,231,231,231,230,230,229,229,229,228,227,
    226,226,226,225,225,224,224,224,224,223,223,222,222,222,221,221,
    221,221,220,220,220,220,220,219,219,219,219,219,218,218,218,217,
    216,216,215,215,215,215,215,215,215,214,214,214,213,213,212,212,
    211,211,211,210,210,210,210,209,207,207,207,207,206,205,204,204,
    204,203,202,202,201,200,200,200,199,199,199,198,198,198,197,197,
    197,196,196,195,195,194,194,193,192,192,192,191,191,191,191,191,
    190,190,190,189,188,188,188,188,188,186,186,185,184,184,184,183,
    183,183,183,182,182,182,181,180,180,180,179,179,178,178,177,177,
    176,176,176,176,175,175,174,173,173,172,172,171,171,171,170,170,
    170,169,169,168,168,168,167,166,166,165,165,164,164,163,163,163,
    163,163,163,163,162,162,162,162
  };
  const int n3w2b1r1[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    240,239,239,239,238,237,237,236,235,235,234,234,234,233,233,233,
    233,232,232,232,232,231,230,229,229,228,228,228,227,227,227,225,
    225,225,225,224,224,224,223,223,223,221,221,221,221,221,220,220,
    220,220,220,219,219,219,218,218,218,218,217,217,217,217,216,216,
    215,215,215,214,213,213,213,213,213,212,212,212,211,211,210,209,
    209,209,208,208,208,208,208,207,207,206,206,206,206,204,204,204,
    204,204,204,204,204,203,202,202,202,201,201,201,200,200,199,199,
    199,199,199,198,197,197,197,197,197,197,196,196,196,196,195,194,
    194,193,193,193,193,192,190,190,189,189,189,187,187,186,186,186,
    186,185,184,184,184,183,182,182,182,181,181,181,179,178,177,177,
    177,176,176,176,176,176,175,175,175,173,173,173,172,172,172,172,
    172,172,171,171,171,171,170,170,170,169,169,169,167,167,167,165,
    164,164,164,164,164,163,163,162
  };
  const int n3w2b1r2[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    240,240,240,239,238,238,238,238,237,237,236,236,236,235,235,234,
    233,232,232,231,230,230,230,230,229,229,228,228,228,227,226,226,
    225,225,224,224,224,224,224,223,223,223,222,222,221,221,221,221,
    220,220,219,219,217,217,216,216,216,215,215,215,214,214,214,213,
    213,213,212,211,211,210,209,209,209,209,208,208,208,208,207,207,
    207,206,206,205,205,205,205,204,204,204,203,203,203,203,203,203,
    203,202,202,202,202,201,201,201,200,200,199,199,198,197,197,196,
    196,195,195,194,194,194,194,194,193,193,193,193,193,192,191,191,
    191,189,189,188,188,188,188,187,187,187,187,186,186,186,186,185,
    184,183,183,183,183,183,182,182,182,181,181,181,180,178,178,177,
    177,177,176,176,175,175,175,175,173,173,172,172,172,172,172,172,
    171,170,169,169,169,169,169,168,167,167,167,165,165,165,165,165,
    165,165,164,163,163,163,162,162
  };
  const int n3w2b1r3[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    240,240,240,240,239,238,238,238,237,237,237,237,236,234,233,232,
    232,232,231,231,230,229,228,228,228,228,228,228,227,226,226,225,
    225,225,224,224,223,223,223,222,222,222,222,221,221,221,220,220,
    219,219,218,218,218,218,217,217,217,217,216,216,215,215,215,212,
    212,212,212,212,211,211,211,210,210,210,209,209,209,209,208,208,
    208,208,207,207,207,206,206,206,206,205,205,204,204,203,203,203,
    202,202,202,202,202,201,201,200,199,199,199,199,198,198,198,198,
    197,197,197,196,196,196,194,193,193,193,193,192,192,192,192,191,
    191,191,190,190,189,189,189,188,188,188,187,186,186,186,185,185,
    185,185,184,184,183,183,182,182,182,182,182,181,181,180,179,179,
    179,179,178,177,177,176,175,175,175,175,174,173,173,172,172,172,
    170,170,170,169,168,168,168,168,167,167,166,166,166,165,164,164,
    164,164,163,163,163,163,163,163
  };
  const int n3w2b1r4[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    239,238,237,237,237,237,237,237,236,235,235,235,234,233,233,232,
    232,231,231,231,230,230,230,229,229,228,228,227,227,227,226,226,
    226,226,225,225,224,224,224,223,223,223,222,221,221,221,221,219,
    219,219,218,217,217,217,216,216,216,216,214,214,214,214,214,213,
    212,211,211,210,210,210,209,209,208,208,206,206,206,205,204,203,
    203,203,202,201,201,201,201,200,200,199,199,198,198,198,197,197,
    197,197,196,196,196,196,195,195,194,194,193,193,192,191,191,191,
    190,190,189,189,189,189,189,189,189,189,188,188,188,188,188,187,
    187,187,186,186,185,185,184,183,183,183,183,183,182,181,181,181,
    180,180,179,179,179,179,178,177,177,177,176,175,175,174,174,174,
    173,173,173,173,172,172,172,172,171,171,171,171,170,170,169,169,
    169,168,168,167,167,167,167,167,166,166,166,165,165,165,164,164,
    163,163,163,162,162,162,162,162
  };
  const int n3w2b1r5[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    240,239,239,238,238,238,238,238,238,237,237,236,236,236,236,234,
    234,234,233,233,233,233,233,232,230,230,230,229,229,229,229,228,
    228,227,227,227,225,225,224,224,223,223,223,222,222,222,222,221,
    221,221,220,220,219,219,219,217,217,217,217,217,217,217,216,215,
    214,214,214,213,213,213,213,213,213,213,212,212,212,211,211,211,
    211,210,208,208,207,207,207,206,206,205,205,202,202,202,202,202,
    201,200,199,199,199,199,198,198,198,198,197,197,196,196,196,195,
    195,194,194,194,194,194,193,193,193,192,192,191,191,191,190,189,
    189,188,188,188,188,187,185,184,183,183,183,182,182,182,181,181,
    181,180,180,179,179,179,177,177,177,177,176,175,175,175,175,175,
    174,173,172,172,172,172,171,171,171,171,170,170,169,169,169,169,
    169,169,169,168,168,168,168,167,167,167,166,166,165,165,164,164,
    164,164,163,163,162,162,162,162
  };
  const int n3w2b1r6[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    240,240,240,240,239,239,238,238,238,237,237,237,237,234,234,234,
    233,233,233,232,231,231,231,231,230,230,230,230,230,229,229,229,
    229,229,228,228,228,228,228,228,228,227,227,227,226,226,225,225,
    225,225,224,223,223,222,221,221,220,220,219,219,218,217,217,217,
    216,216,216,216,215,215,215,214,214,213,213,212,212,212,211,211,
    211,210,210,209,209,209,208,208,208,208,207,207,207,206,205,205,
    205,205,204,203,203,202,202,202,201,200,200,199,199,198,198,198,
    198,197,197,196,196,196,194,194,194,194,193,192,192,191,191,190,
    190,189,189,189,189,188,187,186,185,184,184,184,183,182,182,182,
    182,182,181,181,181,180,178,178,177,177,176,176,176,175,175,175,
    175,175,175,175,174,174,174,173,173,173,172,172,171,171,171,171,
    171,170,170,170,169,169,169,169,169,168,168,168,166,166,165,165,
    165,164,164,164,163,163,163,162
  };
  const int n3w2b1r7[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    240,240,240,239,239,239,238,237,237,237,237,236,235,234,234,234,
    233,233,233,233,233,232,231,231,230,230,230,229,229,226,226,226,
    226,226,225,224,224,223,223,222,221,221,221,221,221,220,219,219,
    218,218,218,218,218,217,217,217,217,217,217,217,217,216,216,215,
    215,215,213,213,213,212,212,212,211,211,209,208,207,207,207,206,
    206,206,206,205,205,205,205,205,205,203,203,203,203,202,202,202,
    202,201,201,201,199,199,199,198,197,197,197,195,194,194,194,194,
    193,193,193,193,192,192,192,191,190,190,190,190,190,190,189,189,
    189,188,188,188,188,188,188,187,187,187,187,186,186,186,186,186,
    186,185,185,185,183,183,183,182,182,182,181,180,180,180,179,179,
    179,179,179,178,178,178,178,178,178,178,177,176,176,176,175,175,
    172,172,172,171,171,171,170,170,170,170,169,169,167,167,167,165,
    165,165,165,165,164,163,163,163
  };
  const int n3w2b1r8[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    240,240,240,239,239,239,238,238,238,238,238,237,236,236,236,236,
    235,234,234,234,234,233,233,233,232,232,232,231,231,231,231,230,
    230,230,229,229,229,227,226,226,226,225,225,225,223,223,223,223,
    223,221,221,221,219,219,219,217,217,216,216,216,215,215,214,214,
    214,213,213,213,211,210,210,209,209,209,208,208,208,208,208,207,
    207,207,207,207,207,206,205,205,205,204,204,204,203,203,203,202,
    201,201,201,200,200,200,199,199,198,198,198,197,197,197,196,196,
    195,194,194,194,193,192,192,191,191,191,190,189,188,187,186,186,
    185,185,185,185,185,185,184,183,183,183,182,182,182,181,180,180,
    180,180,179,179,179,179,178,178,177,177,177,176,176,176,176,175,
    175,174,174,174,173,173,173,172,171,171,171,171,171,170,170,169,
    169,168,168,168,168,168,168,167,166,166,166,166,166,165,165,165,
    165,164,164,164,163,163,162,162
  };
  const int n3w2b1r9[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    240,240,240,239,239,238,238,238,238,238,238,238,237,237,237,237,
    236,236,235,235,234,234,232,232,232,232,232,230,230,230,230,230,
    229,229,229,229,229,229,228,228,228,225,225,225,225,225,224,224,
    224,224,223,223,222,221,221,220,220,220,220,219,219,219,219,218,
    217,217,216,215,215,213,213,213,212,212,211,211,211,211,210,210,
    210,210,209,209,209,208,207,207,207,205,203,203,202,202,202,201,
    200,199,199,199,198,198,198,198,197,197,197,196,196,195,195,195,
    194,193,192,192,192,191,190,190,190,190,189,189,189,189,188,188,
    188,187,187,187,186,186,185,184,184,184,183,183,182,182,181,181,
    181,181,181,180,179,179,178,178,177,177,177,177,176,176,176,176,
    175,175,175,175,174,174,174,174,173,173,173,173,173,172,172,171,
    171,171,171,170,170,169,169,169,168,168,168,167,167,167,167,167,
    166,166,166,164,164,163,162,162
  };
  const int n3w2b2r0[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    300,300,299,299,298,297,295,295,294,294,293,289,288,287,285,284,
    284,282,281,279,277,276,276,275,274,274,272,272,270,269,267,264,
    263,263,261,260,260,260,258,255,255,255,255,254,253,250,247,247,
    247,246,245,245,244,243,241,241,241,241,239,238,238,238,238,238,
    238,237,235,234,233,232,231,231,229,229,229,228,228,226,225,225,
    223,221,220,219,217,216,216,216,213,210,208,208,207,205,202,201,
    201,201,201,199,199,198,196,195,195,194,194,193,191,189,189,188,
    188,187,186,184,184,182,182,181,179,178,177,175,174,173,172,171,
    171,171,169,169,168,168,167,167,166,165,164,163,162,158,158,157,
    157,156,153,153,151,151,148,147,147,146,146,145,145,144,144,144,
    143,141,139,138,137,136,134,134,129,126,125,125,123,122,122,121,
    121,121,120,120,118,118,116,114,113,112,111,110,108,108,107,107,
    106,106,103,103,103,103,102,102
  };
  const int n3w2b2r1[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    300,299,298,298,297,297,294,291,290,289,288,288,286,285,283,282,
    280,279,277,276,275,274,274,272,272,271,271,269,269,268,268,267,
    267,267,265,265,264,263,262,262,259,259,256,253,253,251,249,249,
    248,246,246,245,244,242,241,238,237,237,236,235,233,233,232,229,
    229,228,228,228,228,227,227,226,225,224,223,223,221,220,220,219,
    218,218,218,217,214,212,209,207,205,204,203,202,202,201,200,199,
    198,196,195,193,193,192,190,190,189,187,187,187,186,186,185,185,
    185,184,183,182,182,182,181,181,181,181,180,178,177,177,175,175,
    174,174,174,173,173,172,170,170,168,168,167,166,164,162,161,160,
    160,159,156,155,151,150,150,149,149,148,148,148,145,143,140,138,
    136,134,133,133,132,131,131,130,129,129,128,126,125,124,124,121,
    120,120,118,116,115,115,114,114,113,112,111,111,110,110,110,109,
    108,107,107,107,105,104,103,102
  };
  const int n3w2b2r2[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    299,299,298,298,296,295,295,292,291,289,289,289,288,287,287,285,
    285,285,282,281,280,280,278,277,277,276,275,272,271,271,269,269,
    268,265,264,261,260,260,260,260,259,258,257,255,254,251,251,250,
    250,247,247,240,239,238,237,237,236,236,236,236,235,234,234,231,
    231,230,227,227,227,226,225,225,225,223,223,218,217,217,216,216,
    215,215,214,213,212,212,210,207,207,206,204,202,202,201,200,198,
    195,194,193,191,191,188,188,186,185,185,183,183,181,179,179,177,
    176,175,174,174,173,170,169,169,166,166,165,163,161,161,160,159,
    158,158,156,156,156,153,153,153,150,149,147,146,146,145,145,141,
    140,139,138,137,137,136,136,135,134,134,134,132,132,131,130,130,
    130,129,128,128,128,127,126,125,124,124,122,121,121,121,119,119,
    117,117,116,116,114,114,114,113,112,112,111,111,110,110,108,107,
    106,105,105,104,104,104,103,102
  };
  const int n3w2b2r3[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    300,297,295,293,288,288,287,286,286,286,284,282,281,281,280,280,
    278,276,273,272,271,270,269,269,267,265,265,264,263,261,260,255,
    254,254,253,252,251,251,250,248,247,244,238,238,238,237,237,237,
    235,235,235,231,231,230,230,230,230,230,229,228,228,227,225,225,
    224,223,223,223,220,220,220,219,217,216,216,216,214,214,213,213,
    213,207,207,206,205,204,204,203,202,201,201,200,200,199,199,199,
    197,197,196,196,195,195,195,195,194,194,193,190,189,188,188,187,
    186,185,182,182,180,173,172,171,170,169,168,168,167,166,163,162,
    162,161,160,160,158,158,157,156,156,154,153,151,151,150,149,148,
    147,145,143,143,143,142,141,139,139,138,138,137,136,136,136,132,
    131,131,131,130,129,128,127,127,126,126,125,124,122,120,120,119,
    118,116,116,115,115,115,114,113,113,112,112,112,111,111,111,110,
    110,109,108,107,106,105,105,102
  };
  const int n3w2b2r4[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    300,297,294,293,293,293,292,292,290,289,289,288,287,287,286,286,
    285,284,284,283,280,280,280,279,278,278,277,277,276,275,275,274,
    274,273,272,268,268,267,265,265,265,264,264,262,262,261,261,261,
    261,259,256,254,254,251,250,249,249,248,247,245,245,243,240,239,
    239,238,237,235,235,231,230,229,229,228,221,220,217,215,215,214,
    213,212,211,210,210,210,209,209,209,208,208,206,206,205,205,203,
    202,202,201,201,200,200,199,198,196,193,192,192,192,190,188,188,
    186,186,186,185,183,181,181,180,179,179,176,175,174,174,173,173,
    171,170,168,167,167,166,164,163,163,161,161,160,155,154,152,150,
    150,148,147,147,146,146,145,145,145,145,144,144,143,143,142,139,
    139,139,139,138,137,135,134,132,127,126,126,126,126,125,125,125,
    125,124,124,124,123,123,122,122,122,120,119,118,118,117,114,114,
    113,112,111,111,110,107,106,104
  };
  const int n3w2b2r5[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    297,296,296,296,293,292,292,290,290,289,289,287,284,282,282,279,
    278,277,277,275,273,273,268,267,267,266,265,264,264,264,261,260,
    260,259,259,259,257,257,256,253,252,252,252,251,251,251,250,249,
    245,243,243,243,243,242,242,236,236,236,231,231,231,229,229,229,
    227,225,223,223,223,222,222,218,217,217,217,216,215,214,212,211,
    210,210,210,210,208,208,207,207,206,204,203,202,199,198,196,196,
    195,195,194,191,190,190,190,190,190,187,186,185,184,184,183,183,
    183,182,181,181,179,179,179,175,175,175,175,174,174,173,173,173,
    172,171,171,169,169,168,168,167,167,166,166,165,163,163,163,162,
    160,159,159,159,155,154,153,153,153,151,151,150,149,143,142,141,
    141,141,140,138,136,135,132,132,130,130,129,128,128,127,126,125,
    125,125,125,122,122,121,121,119,119,118,113,112,112,112,112,111,
    110,110,110,109,109,107,103,102
  };
  const int n3w2b2r6[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    300,298,298,298,298,295,295,293,293,292,290,289,288,288,288,287,
    286,286,285,285,284,284,283,283,280,279,279,277,275,273,271,270,
    269,268,266,266,265,261,260,260,258,254,253,252,252,252,250,250,
    249,249,248,244,244,241,240,238,238,238,235,234,232,231,231,230,
    230,227,226,226,225,225,225,224,224,223,223,222,222,222,222,221,
    221,220,220,220,220,220,219,219,217,216,215,213,213,212,210,210,
    210,206,205,205,204,203,203,203,203,196,193,192,191,188,188,187,
    186,185,183,183,182,181,178,176,175,174,173,172,172,171,171,171,
    170,167,166,164,164,163,163,161,161,159,157,155,154,153,152,152,
    152,151,148,147,146,146,144,144,143,142,141,141,139,139,136,136,
    136,135,135,133,132,132,132,127,127,126,123,123,122,121,120,120,
    120,118,117,115,114,113,113,112,112,111,111,111,111,110,109,108,
    108,107,107,105,104,104,104,102
  };
  const int n3w2b2r7[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    300,300,297,296,295,295,295,294,292,291,287,286,285,284,283,283,
    282,282,282,280,280,278,276,275,275,268,268,267,264,263,262,261,
    261,260,259,259,259,258,258,257,253,253,253,251,249,249,249,249,
    248,246,246,245,245,245,242,241,241,240,238,237,234,233,233,229,
    226,224,224,223,223,223,222,222,221,220,220,218,218,217,217,217,
    216,216,216,216,215,214,214,213,213,212,211,210,209,207,207,205,
    202,202,201,200,199,198,197,195,195,195,194,194,194,193,191,191,
    191,187,186,185,184,178,175,175,175,175,175,174,173,172,171,168,
    168,168,166,165,165,164,162,161,161,160,160,157,156,155,155,155,
    152,151,150,149,147,144,144,143,142,142,141,141,141,140,139,139,
    139,139,139,138,137,136,135,135,134,134,133,132,132,131,131,131,
    131,131,130,129,129,126,125,124,122,122,122,120,120,118,117,115,
    113,108,107,104,103,103,102,102
  };
  const int n3w2b2r8[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    300,298,298,297,295,294,293,292,292,290,290,289,289,289,288,288,
    288,288,287,287,286,286,286,285,284,283,282,282,282,281,278,277,
    276,275,275,274,273,272,272,272,272,271,270,269,268,267,267,266,
    266,265,263,263,263,262,260,259,259,258,256,255,254,254,253,251,
    249,249,248,247,246,245,245,241,241,238,234,233,233,231,230,228,
    227,227,227,225,224,223,223,221,219,219,219,218,217,216,214,214,
    214,214,210,209,208,207,204,204,204,203,202,200,199,198,197,194,
    194,192,192,192,191,190,190,190,189,188,187,186,185,183,182,181,
    181,181,179,178,173,173,171,171,171,169,168,167,167,165,165,165,
    163,160,159,158,158,157,157,154,153,153,151,151,151,151,149,148,
    146,145,144,142,141,141,141,139,139,139,136,135,134,134,134,131,
    130,127,125,123,123,121,120,119,119,119,118,118,116,116,115,115,
    112,111,110,107,107,106,105,105
  };
  const int n3w2b2r9[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    299,299,298,297,294,291,291,291,289,288,288,288,287,286,286,285,
    284,284,282,281,281,280,280,279,279,278,277,276,275,275,273,273,
    270,268,267,263,261,261,259,259,258,257,256,254,253,251,251,250,
    250,249,248,243,240,239,239,238,238,238,237,237,236,235,234,233,
    233,233,232,231,229,228,226,226,225,222,221,221,219,219,219,219,
    217,216,216,215,214,214,214,214,214,212,211,211,208,204,204,202,
    202,202,200,199,198,197,197,196,196,196,195,195,194,193,192,190,
    184,184,180,179,178,177,176,176,175,174,173,171,170,169,168,167,
    167,167,167,166,166,166,166,165,164,164,163,161,161,159,159,159,
    155,154,151,151,149,149,149,147,147,144,143,139,137,137,135,134,
    134,134,133,133,133,132,132,130,129,127,127,124,122,120,120,118,
    117,115,114,114,114,113,113,113,112,111,111,111,108,108,108,106,
    106,105,105,103,103,103,103,102
  };
  const int n3w2b3r0[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    378,374,373,372,371,371,371,370,362,362,361,358,358,357,356,354,
    353,351,351,350,348,346,346,344,341,340,339,338,336,336,334,332,
    330,330,328,324,324,321,320,319,318,317,317,316,316,309,309,309,
    308,308,307,307,306,304,303,302,301,300,300,299,290,290,289,287,
    282,279,272,270,269,267,266,263,262,261,258,257,255,254,253,253,
    250,249,246,242,242,242,242,238,238,238,237,235,232,230,230,228,
    225,221,221,219,217,213,210,210,209,206,205,203,203,200,199,198,
    198,197,195,190,190,187,180,178,177,177,176,167,166,166,165,159,
    159,157,155,154,154,153,151,151,151,150,147,141,139,139,138,136,
    129,128,128,127,126,125,123,115,110,105,104,101,100,99,96,96,
    93,92,92,91,89,89,88,87,86,79,77,76,73,70,68,65,57,54,54,53,49,
    48,46,46,42,38,38,37,37,37,34,33,30,30,30,27,25,22,22,22
  };
  const int n3w2b3r1[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    377,375,373,369,368,362,362,361,360,360,358,357,357,356,355,354,
    348,343,340,339,338,336,332,329,328,327,324,321,321,320,320,320,
    318,314,311,310,309,305,303,302,302,301,299,297,297,295,292,291,
    290,289,289,288,287,286,280,279,277,275,274,265,264,257,257,256,
    255,247,247,246,246,243,242,240,240,237,236,232,230,230,229,227,
    226,223,221,219,217,213,213,212,209,208,208,207,202,201,200,199,
    198,197,193,191,189,188,188,187,184,182,182,181,181,180,180,180,
    180,177,176,170,169,169,169,164,164,163,163,156,156,156,153,148,
    147,145,141,139,134,134,134,132,128,125,124,123,123,122,121,120,
    116,116,116,115,115,113,109,104,104,104,103,102,89,88,86,85,84,
    84,84,82,80,77,76,75,74,74,74,73,68,67,66,65,62,62,59,51,49,49,
    49,48,48,46,46,44,43,43,42,39,38,33,30,29,27,26,26,24
  };
  const int n3w2b3r2[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    378,378,377,377,375,374,371,367,367,365,365,361,356,353,349,345,
    342,339,337,334,334,330,330,330,329,328,325,325,324,322,317,316,
    316,315,313,312,310,307,305,303,300,293,290,284,283,283,281,281,
    280,280,278,275,272,270,270,263,260,258,255,253,251,251,251,249,
    248,248,246,245,243,242,242,239,239,237,235,234,234,233,232,230,
    230,228,227,225,225,224,220,218,217,217,215,210,204,202,201,200,
    197,196,195,194,191,180,173,173,172,172,172,170,168,166,163,163,
    163,162,161,160,157,155,154,151,148,147,144,144,143,142,142,142,
    141,141,141,137,133,132,132,131,131,127,124,122,120,120,117,116,
    115,113,112,111,109,108,107,104,103,100,99,98,97,96,94,91,90,
    89,89,88,88,87,82,82,80,77,76,75,75,71,67,65,65,63,61,60,58,55,
    53,52,51,48,47,47,43,43,37,34,34,31,27,27,26,25,24,23
  };
  const int n3w2b3r3[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    378,375,370,368,364,364,364,361,360,360,350,349,349,347,345,340,
    340,339,339,339,335,332,330,321,321,321,317,316,313,312,311,310,
    307,304,303,298,295,294,292,292,279,277,277,274,271,267,267,267,
    265,263,262,261,259,256,255,254,253,251,251,250,248,247,246,245,
    245,243,242,242,241,239,238,238,236,236,235,234,232,231,230,229,
    225,223,223,222,221,220,216,216,216,216,215,213,213,212,210,209,
    203,200,198,197,197,192,191,190,187,187,186,185,185,178,178,175,
    174,174,172,170,169,165,165,157,156,154,154,154,154,148,148,147,
    145,144,142,142,139,136,136,135,134,133,129,129,128,128,127,127,
    125,124,124,124,123,122,118,113,112,111,108,108,107,106,101,98,
    96,96,94,94,91,89,88,86,82,79,76,72,71,70,67,65,65,63,63,62,61,
    60,58,57,55,47,47,47,45,36,35,31,28,28,28,28,28,25,24,23
  };
  const int n3w2b3r4[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    380,379,378,377,377,373,373,370,369,368,367,365,364,364,361,355,
    354,352,351,348,342,340,339,338,337,336,333,329,326,326,325,325,
    325,322,321,320,319,319,318,317,317,316,316,311,305,304,301,301,
    299,295,293,292,292,288,287,285,285,282,281,281,280,280,279,279,
    279,278,272,272,270,267,264,263,255,254,254,251,249,249,245,243,
    243,242,241,240,236,233,229,228,228,225,225,222,222,217,216,216,
    215,210,210,206,206,205,204,202,202,199,199,198,198,197,196,188,
    188,187,185,179,178,177,176,176,175,175,175,174,173,173,171,166,
    165,162,161,161,160,159,158,158,158,158,155,154,153,152,149,149,
    144,140,139,138,135,131,129,127,127,125,119,118,118,116,116,114,
    106,102,98,92,91,91,89,89,86,85,84,83,82,79,77,75,75,71,70,67,
    65,59,58,57,56,55,52,41,40,40,36,33,31,30,30,28,27,23,22,22
  };
  const int n3w2b3r5[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    380,378,378,373,370,370,370,369,368,368,367,366,360,357,354,353,
    351,350,348,347,340,340,339,338,337,335,333,328,328,327,324,323,
    321,320,316,315,311,311,308,307,300,300,297,297,297,295,294,292,
    285,280,280,277,277,275,275,272,266,265,264,264,263,262,261,259,
    257,255,255,249,249,245,244,244,243,243,242,241,241,240,238,238,
    237,234,228,227,226,226,225,224,224,221,220,218,217,217,217,214,
    211,209,206,203,203,202,202,201,201,200,197,196,189,188,188,187,
    186,186,186,185,179,178,177,172,167,165,165,163,161,159,158,158,
    157,156,155,155,152,149,146,144,140,139,138,130,128,127,125,122,
    120,117,117,115,113,109,105,103,103,99,99,96,94,93,92,92,91,90,
    88,82,81,80,76,74,73,67,66,66,66,59,58,57,56,56,55,53,52,51,50,
    49,48,44,43,40,39,38,35,34,33,29,29,27,26,24,24,22
  };
  const int n3w2b3r6[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    379,378,372,372,372,370,370,368,368,365,364,364,363,358,357,356,
    355,353,348,344,343,343,341,340,339,339,336,332,331,331,325,323,
    323,323,321,320,319,318,316,315,313,312,306,304,302,301,301,298,
    297,296,292,292,290,288,286,286,285,283,277,272,270,267,266,266,
    261,261,258,256,254,253,252,252,252,251,250,249,248,242,242,236,
    236,235,233,230,230,226,225,223,220,219,215,213,208,206,203,202,
    201,200,199,196,193,192,191,187,184,183,183,181,175,174,173,173,
    172,172,172,172,171,167,167,167,166,165,165,163,163,161,157,156,
    156,154,151,143,136,134,131,129,125,125,124,120,120,118,117,116,
    115,113,113,112,112,112,108,105,104,103,102,99,97,97,96,95,88,
    87,86,85,83,76,73,71,69,69,68,68,68,66,63,61,61,55,54,53,52,52,
    52,47,47,44,43,42,41,41,39,36,34,33,31,31,31,27,23,22
  };
  const int n3w2b3r7[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    380,378,377,377,376,375,372,370,366,364,364,362,357,357,357,356,
    354,354,352,350,350,346,346,343,342,341,341,340,338,334,332,332,
    332,330,329,328,326,326,322,321,320,319,318,318,317,314,313,305,
    304,303,302,300,293,292,292,291,288,287,287,286,285,284,280,277,
    276,275,275,262,261,259,259,258,257,253,249,249,248,242,237,236,
    232,230,230,229,229,224,223,220,217,217,217,216,215,214,209,207,
    206,205,203,203,202,200,200,200,196,196,194,192,189,188,186,186,
    182,182,182,181,181,177,175,174,172,168,164,160,160,160,159,157,
    156,156,154,152,151,148,146,145,138,136,135,134,134,132,131,129,
    127,125,124,123,119,115,112,107,106,105,105,104,102,99,98,98,
    96,93,93,89,87,86,84,82,79,79,78,77,77,70,70,69,69,67,65,60,59,
    59,59,56,53,50,49,49,47,43,43,42,38,37,32,32,31,30,28,24
  };
  const int n3w2b3r8[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    378,378,375,374,373,366,363,362,359,358,353,352,350,348,348,347,
    345,343,339,339,330,329,323,323,322,321,320,318,317,315,314,313,
    311,308,306,301,298,297,292,292,292,291,283,283,282,281,281,269,
    266,266,266,265,265,262,258,256,256,252,247,246,244,242,241,241,
    241,239,239,237,235,235,231,231,229,228,224,223,223,221,220,218,
    212,210,210,207,207,206,205,205,202,200,193,193,193,190,189,189,
    188,188,187,187,186,184,182,180,178,178,177,175,173,172,172,171,
    169,167,167,162,161,159,159,159,158,157,156,155,154,153,152,151,
    149,149,149,146,146,145,144,144,142,137,137,135,134,133,132,132,
    128,124,124,123,120,116,116,115,115,110,107,107,103,101,98,96,
    91,91,86,84,83,83,82,79,75,74,74,72,72,65,62,61,59,59,54,52,50,
    47,46,45,43,43,41,39,39,39,37,35,34,33,31,30,29,28,26,22
  };
  const int n3w2b3r9[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    378,376,373,372,372,372,372,370,367,367,362,358,355,355,354,350,
    346,344,340,340,339,336,335,334,334,334,334,333,329,328,321,318,
    317,317,316,316,311,308,306,303,302,300,299,299,298,297,294,293,
    292,285,278,278,277,276,275,274,270,268,267,263,261,259,255,253,
    252,251,251,251,246,244,242,241,240,239,238,238,237,235,234,233,
    232,232,230,225,224,222,216,215,213,210,204,197,193,185,176,176,
    174,173,172,172,171,168,165,160,160,158,156,156,154,153,152,151,
    151,151,150,148,146,145,144,143,143,140,140,138,138,135,134,133,
    128,127,126,122,122,120,119,119,115,115,113,111,110,110,107,106,
    106,105,105,103,103,102,102,102,101,99,99,98,94,93,93,93,92,91,
    90,89,89,88,87,85,82,81,81,79,78,78,75,75,72,72,71,69,66,62,59,
    58,57,56,52,52,48,45,41,41,37,33,31,30,29,26,24,23
  };
  const int n3w3b1r0[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    168,168,167,167,166,166,166,166,165,164,163,163,163,163,163,163,
    162,162,162,162,162,161,160,160,160,160,160,159,159,159,159,159,
    159,159,159,159,158,158,157,157,157,157,157,157,156,156,156,156,
    156,155,155,155,155,154,154,154,154,153,153,152,152,152,152,152,
    152,151,150,150,148,148,148,148,148,148,147,147,147,147,146,146,
    146,145,144,144,143,143,143,143,143,142,142,141,141,141,140,140,
    140,139,139,139,139,139,139,139,138,138,137,137,137,136,136,136,
    136,135,135,135,134,134,134,133,133,133,133,132,132,132,132,132,
    131,131,131,130,130,130,130,130,130,130,129,129,129,129,128,128,
    128,127,127,127,126,126,126,126,125,125,125,125,124,124,124,124,
    124,124,123,123,123,122,122,122,122,122,121,120,120,119,119,119,
    119,119,118,118,118,118,117,117,117,116,116,116,116,115,115,115,
    115,115,115,115,115,114,114,114
  };
  const int n3w3b1r1[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    168,168,168,168,168,167,167,167,167,166,166,165,165,165,165,164,
    164,164,163,163,163,163,162,162,161,161,161,161,160,160,160,160,
    160,158,158,158,158,157,157,157,157,157,156,156,156,156,156,155,
    155,154,154,153,153,152,152,152,152,151,151,150,150,150,150,149,
    149,148,147,147,147,147,146,146,146,146,146,146,145,145,145,145,
    144,143,143,143,143,143,142,142,141,141,140,140,140,140,139,139,
    139,138,138,138,137,137,137,137,136,136,136,136,136,136,135,135,
    135,134,134,134,134,134,133,133,133,133,132,132,132,132,132,132,
    132,132,132,131,131,131,131,131,131,130,130,130,129,129,129,128,
    128,128,128,128,127,127,127,126,126,126,126,125,124,123,123,123,
    123,122,122,122,122,122,122,122,121,121,121,121,120,120,119,119,
    119,119,119,118,118,117,117,117,117,117,117,116,116,116,116,116,
    116,116,115,115,114,114,114,114
  };
  const int n3w3b1r2[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    168,168,168,168,168,167,167,167,167,166,166,165,165,165,165,165,
    165,164,164,164,163,163,162,161,161,160,160,160,160,159,159,159,
    159,159,158,158,158,158,158,158,158,157,157,157,157,157,157,156,
    156,155,155,155,155,155,154,154,154,154,153,153,153,153,153,153,
    152,152,151,151,151,151,150,150,150,150,150,149,149,149,149,148,
    148,148,148,148,147,147,147,147,147,147,146,146,146,146,145,145,
    145,144,144,143,143,143,143,143,142,142,142,142,141,140,140,139,
    139,139,139,138,138,138,138,138,138,137,136,136,135,135,135,135,
    135,134,134,133,133,133,132,131,130,130,129,129,129,128,128,127,
    126,126,126,126,126,125,125,125,125,125,125,124,123,123,123,123,
    123,122,122,122,122,122,122,121,121,121,121,120,120,120,120,120,
    120,119,119,119,119,118,117,117,117,117,117,117,116,116,116,115,
    115,115,115,115,114,114,114,114
  };
  const int n3w3b1r3[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    168,168,168,168,168,168,168,167,167,167,165,165,164,164,164,164,
    164,163,163,163,163,162,162,162,162,161,161,161,161,160,160,159,
    159,158,158,157,157,156,156,156,156,155,155,155,155,155,154,154,
    154,153,153,152,152,151,151,151,151,151,151,151,151,150,150,150,
    149,149,149,148,148,148,148,148,147,147,147,146,146,145,145,145,
    144,144,144,144,143,143,143,143,142,142,142,142,142,142,141,141,
    141,141,141,141,141,140,140,140,140,140,140,139,139,139,138,138,
    138,137,137,137,137,137,136,136,136,136,135,135,135,135,135,134,
    134,134,134,133,133,133,133,133,133,133,132,132,132,131,130,130,
    130,130,130,130,130,130,129,128,128,127,127,126,126,125,125,125,
    125,125,125,125,124,124,124,124,124,123,123,123,123,122,122,122,
    121,121,120,120,120,118,118,117,117,117,117,116,115,115,115,115,
    115,115,115,114,114,114,114,114
  };
  const int n3w3b1r4[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    168,167,167,167,166,166,165,165,165,164,163,163,163,163,162,162,
    162,162,162,161,161,161,161,161,160,160,160,160,160,160,160,159,
    158,158,158,158,157,157,157,157,157,156,156,155,155,155,155,155,
    155,154,154,154,154,154,153,153,153,153,153,153,152,152,152,152,
    152,151,151,151,151,150,150,150,150,150,149,149,148,147,147,147,
    146,146,146,145,145,145,145,144,143,143,143,142,142,142,142,142,
    142,142,142,142,141,141,141,140,139,139,139,139,139,139,138,137,
    137,137,137,137,136,136,136,136,136,135,135,134,133,133,133,133,
    132,132,132,132,131,131,131,130,130,130,130,130,130,129,129,128,
    128,128,128,127,127,127,127,126,126,126,126,126,125,125,125,125,
    125,124,124,124,124,124,123,123,123,123,123,123,122,122,122,121,
    121,121,121,120,119,119,119,119,118,118,117,117,116,116,116,116,
    116,115,115,115,114,114,114,114
  };
  const int n3w3b1r5[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    168,168,168,167,167,167,167,167,166,166,166,166,165,164,164,164,
    164,162,162,161,161,161,160,160,159,159,159,159,159,159,159,158,
    158,158,158,158,157,157,157,157,156,156,156,156,155,155,155,155,
    155,155,155,155,154,154,154,154,154,154,153,153,152,152,152,151,
    150,150,149,149,149,149,149,148,148,147,147,147,147,146,146,146,
    145,145,145,144,144,144,144,143,143,143,143,143,142,142,141,141,
    141,141,140,140,140,139,139,138,138,138,138,138,138,138,138,137,
    137,137,136,136,136,135,135,135,135,135,135,134,134,133,133,133,
    133,133,132,132,132,132,131,131,131,131,131,130,130,130,130,130,
    129,129,129,128,128,128,128,128,128,127,127,127,127,127,126,126,
    126,125,125,125,124,124,124,124,123,122,122,121,121,121,121,120,
    120,119,119,119,117,117,117,117,117,116,116,116,116,116,116,116,
    116,115,115,115,115,115,114,114
  };
  const int n3w3b1r6[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    168,168,168,168,168,167,167,167,166,166,166,166,166,165,165,165,
    165,165,164,164,163,163,162,162,162,162,162,162,162,161,161,161,
    160,160,160,160,160,160,160,160,160,160,159,159,159,159,159,159,
    159,159,159,157,157,156,156,155,155,155,155,155,154,154,153,153,
    152,152,152,151,151,151,149,149,148,148,148,148,148,147,147,147,
    145,144,144,143,143,142,142,141,141,140,140,139,139,139,139,139,
    139,138,138,138,138,138,137,137,137,137,137,137,136,136,136,135,
    135,135,135,134,134,134,134,133,133,132,132,132,132,132,131,131,
    130,130,130,130,130,129,129,128,128,128,128,127,127,126,126,126,
    126,126,126,125,125,125,125,125,124,124,124,124,123,123,123,123,
    123,122,122,122,122,122,122,121,121,121,121,121,121,121,119,119,
    119,119,119,119,119,118,118,118,118,118,118,117,117,117,116,116,
    116,116,116,115,115,115,114,114
  };
  const int n3w3b1r7[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    168,168,168,168,168,168,168,167,167,167,167,166,166,165,165,165,
    164,164,163,163,163,162,162,162,162,161,161,161,161,161,161,161,
    160,160,160,160,160,160,158,158,158,158,158,158,157,157,157,157,
    157,156,156,156,154,154,154,154,153,153,153,152,152,151,151,151,
    151,150,150,150,149,149,149,149,149,149,149,148,148,148,148,148,
    147,147,147,147,147,147,147,146,146,146,146,146,145,145,145,145,
    144,144,144,144,144,144,144,144,143,143,143,142,141,141,141,140,
    140,140,140,139,139,138,138,138,138,138,138,138,138,137,137,137,
    137,137,137,136,136,136,135,135,134,134,133,133,132,132,131,131,
    131,131,131,130,130,129,129,129,128,128,127,127,127,127,126,126,
    126,126,126,125,124,124,124,123,123,123,122,122,122,121,121,120,
    120,120,120,120,119,119,119,119,118,118,117,117,117,116,116,116,
    116,116,116,116,115,115,115,115
  };
  const int n3w3b1r8[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    168,168,167,167,166,166,165,165,165,165,165,165,165,164,163,163,
    163,163,163,162,162,161,161,160,160,160,160,160,160,159,159,159,
    158,158,157,157,156,156,156,156,155,155,155,155,155,155,154,154,
    154,153,153,153,152,152,152,152,152,152,151,151,151,150,150,150,
    149,149,149,149,148,148,148,148,148,148,147,147,147,147,147,147,
    146,146,146,146,145,144,143,142,142,142,142,142,142,142,141,141,
    141,140,140,140,140,140,139,139,139,139,139,138,138,138,138,138,
    138,137,136,136,136,136,135,134,134,134,134,133,133,133,133,133,
    132,132,132,132,132,131,131,131,131,130,130,130,130,130,130,130,
    130,130,130,129,129,129,129,128,128,127,127,127,127,127,127,127,
    126,126,126,126,125,125,125,124,124,124,123,123,123,122,122,122,
    121,121,121,120,120,120,120,119,119,118,118,118,118,117,117,116,
    116,116,116,115,115,115,114,114
  };
  const int n3w3b1r9[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    168,168,167,167,167,167,166,166,166,165,165,165,165,165,164,164,
    164,164,163,163,163,162,162,162,162,162,161,161,160,160,160,160,
    160,159,159,159,159,158,158,158,157,157,157,157,156,156,155,155,
    155,155,155,155,155,155,155,155,154,154,153,153,153,153,152,152,
    151,151,150,150,150,150,150,150,149,149,148,148,148,148,148,148,
    148,148,148,147,147,147,146,146,146,146,146,145,145,145,145,144,
    144,143,143,142,142,142,141,141,140,140,140,140,140,140,139,139,
    138,138,138,138,137,137,136,136,136,136,136,136,136,135,135,135,
    134,134,134,133,133,132,131,131,131,130,130,130,130,130,129,129,
    129,129,128,128,128,128,128,128,127,127,127,127,127,126,126,126,
    126,126,126,125,125,125,125,125,125,123,123,123,123,123,122,122,
    122,122,122,122,121,121,121,119,118,118,117,117,117,117,117,117,
    117,115,115,115,114,114,114,114
  };
  const int n3w3b2r0[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    210,209,208,207,207,207,207,206,205,205,204,203,202,201,200,199,
    198,198,198,197,197,197,197,197,197,195,195,193,193,193,192,192,
    190,189,189,188,187,187,186,185,185,185,183,181,179,179,178,177,
    177,176,175,175,175,174,174,174,172,171,170,169,169,168,168,168,
    167,166,166,166,166,166,164,164,163,162,162,162,161,160,159,159,
    158,157,156,156,155,155,154,153,153,152,151,151,150,150,149,148,
    147,147,147,146,145,145,145,144,144,142,142,142,142,141,140,139,
    138,138,138,135,133,131,131,131,129,129,128,126,125,124,123,122,
    121,121,120,118,118,117,117,115,115,115,114,114,113,111,111,111,
    110,110,109,106,106,105,105,104,102,99,99,98,98,96,96,95,94,93,
    93,93,93,91,89,89,88,88,88,87,86,86,85,85,84,84,83,83,83,83,82,
    81,80,79,79,79,78,78,76,76,76,76,76,76,75,74,74,72
  };
  const int n3w3b2r1[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    210,210,210,209,207,206,205,205,204,204,203,202,202,202,201,200,
    198,198,198,198,198,197,196,193,193,192,192,191,191,190,190,189,
    188,188,187,186,186,184,184,184,183,183,183,183,182,182,181,181,
    180,180,179,178,177,177,177,175,175,175,173,173,172,171,171,169,
    168,167,167,167,166,166,165,165,163,162,161,160,159,157,157,157,
    155,154,154,154,151,150,149,148,148,147,146,144,144,142,140,140,
    139,138,138,137,137,137,136,136,135,135,135,133,132,131,131,130,
    129,127,126,126,125,124,124,124,123,123,123,122,122,120,120,120,
    120,120,120,118,117,117,116,116,114,113,113,113,112,111,108,107,
    107,106,105,105,105,103,103,102,101,101,101,100,100,100,99,99,
    98,98,98,95,94,94,94,93,91,89,88,87,87,87,85,85,85,85,85,84,82,
    80,79,79,78,78,78,77,76,75,75,75,74,74,74,74,73,73,73,72
  };
  const int n3w3b2r2[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    210,210,210,210,208,208,207,207,206,205,205,205,203,202,202,201,
    200,200,200,200,199,199,199,199,198,198,198,197,197,197,195,193,
    193,192,192,191,190,188,187,185,184,183,182,179,179,178,177,176,
    176,174,173,173,173,173,173,172,172,171,169,169,169,169,168,168,
    167,166,166,165,164,164,164,163,163,162,162,162,162,162,161,160,
    158,158,157,157,156,155,153,151,150,150,147,147,145,144,141,140,
    138,137,137,136,135,135,134,128,127,126,125,125,125,125,124,124,
    122,122,122,121,119,118,118,118,117,117,116,116,116,115,115,114,
    113,111,110,110,110,110,109,109,109,109,109,108,108,108,108,107,
    107,106,106,105,105,104,103,101,101,101,99,98,97,96,95,95,94,
    94,94,94,94,94,93,93,92,92,91,91,91,87,86,86,85,83,83,83,82,82,
    81,80,80,79,79,79,79,77,77,77,76,76,76,75,74,73,73,72
  };
  const int n3w3b2r3[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    210,209,208,208,208,207,207,207,206,205,205,204,204,204,204,203,
    202,202,202,201,201,201,201,200,200,199,198,197,196,194,194,192,
    191,191,188,188,188,188,188,187,187,186,186,182,181,181,181,180,
    179,177,176,176,173,172,172,172,171,168,168,167,167,166,166,166,
    165,165,164,163,163,163,159,159,158,158,158,158,157,156,156,154,
    152,152,151,150,150,149,149,149,148,147,147,147,146,146,145,142,
    142,141,140,140,140,140,139,139,138,138,137,136,135,135,134,134,
    133,133,132,131,131,129,127,127,127,127,126,123,122,119,119,119,
    119,119,119,118,118,117,116,115,115,115,115,115,114,114,114,113,
    112,111,111,110,110,109,106,106,105,105,105,103,103,103,101,101,
    101,100,95,94,94,92,91,90,90,89,89,89,89,88,87,87,86,85,85,85,
    85,84,83,83,82,82,80,79,79,77,76,75,75,75,74,74,74,74,74,72
  };
  const int n3w3b2r4[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    210,210,210,208,207,207,207,206,206,206,205,205,205,205,204,204,
    203,203,202,201,201,200,200,198,198,198,197,196,196,194,192,192,
    192,190,190,189,189,188,187,187,187,186,186,186,185,185,184,184,
    183,182,182,181,181,180,179,179,179,178,177,177,177,176,175,175,
    174,173,173,172,170,169,169,168,167,167,167,166,166,165,164,164,
    162,159,158,158,157,157,156,155,154,152,151,150,150,150,149,148,
    148,147,147,146,146,146,146,146,146,145,145,143,143,142,140,140,
    138,138,136,136,135,134,133,133,133,132,132,131,131,130,129,129,
    129,127,127,127,124,124,122,122,121,121,119,119,118,117,116,115,
    114,114,114,113,113,112,112,112,111,109,108,106,102,102,101,101,
    100,100,99,99,97,97,96,95,95,94,93,93,93,92,92,91,91,90,89,89,
    89,88,86,86,86,85,84,84,84,82,82,82,81,81,77,76,75,74,74,72
  };
  const int n3w3b2r5[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    207,206,206,206,206,204,202,202,201,201,200,199,199,197,195,195,
    194,194,193,191,190,189,189,189,189,188,188,187,187,185,184,184,
    182,181,181,180,179,178,178,176,176,175,175,174,173,173,173,172,
    171,171,168,168,166,166,165,164,164,163,163,163,163,163,161,161,
    161,160,159,158,158,158,157,157,157,157,156,154,154,153,152,152,
    151,150,150,150,150,150,149,147,147,147,147,147,146,145,144,144,
    144,144,143,143,141,141,140,140,140,139,139,138,138,138,138,138,
    137,137,136,135,135,135,135,135,134,134,133,133,133,133,129,129,
    129,127,126,126,125,124,123,123,123,121,120,120,119,119,118,118,
    117,116,116,114,113,111,110,109,109,106,106,104,104,104,103,102,
    102,101,100,100,99,99,99,99,98,98,97,97,97,95,94,94,93,92,92,
    91,89,88,88,88,88,87,86,86,85,84,83,81,81,81,80,78,76,76,74,73
  };
  const int n3w3b2r6[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    210,210,209,209,207,207,206,205,205,204,204,204,204,204,202,200,
    199,198,198,197,196,196,196,196,195,195,195,194,193,192,191,190,
    189,189,188,188,187,185,185,184,184,184,183,182,182,181,181,180,
    179,179,179,179,176,176,175,174,174,171,171,171,171,170,170,169,
    168,167,167,165,163,163,162,160,160,159,158,158,155,154,153,153,
    152,151,151,150,150,150,149,148,148,148,148,148,146,145,145,145,
    145,145,144,143,142,141,141,141,141,140,140,140,139,138,138,136,
    136,136,135,135,135,134,134,134,128,127,127,126,126,125,124,124,
    124,124,123,121,121,120,120,119,118,118,117,116,116,114,114,114,
    112,112,112,109,108,106,106,104,104,102,101,100,100,100,99,99,
    99,98,96,96,93,93,93,93,93,93,92,92,91,91,89,89,87,87,87,87,86,
    86,84,84,82,81,79,78,78,78,78,77,77,76,76,74,74,73,73,72
  };
  const int n3w3b2r7[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    209,208,208,208,207,207,207,206,206,204,204,204,204,203,203,203,
    203,201,200,199,199,198,196,196,196,195,195,195,194,193,191,189,
    188,188,186,186,185,184,184,183,183,183,181,181,180,180,177,177,
    176,176,175,174,173,172,172,171,170,170,170,169,167,166,166,163,
    163,162,161,160,159,159,159,159,158,157,157,157,157,157,156,155,
    155,154,154,152,152,150,150,147,144,143,143,143,141,140,138,138,
    138,136,135,134,133,133,130,130,129,129,129,128,127,126,126,125,
    124,122,122,121,120,120,120,120,118,117,116,116,116,115,115,115,
    113,112,112,112,111,111,110,110,110,109,109,108,108,106,106,105,
    104,104,103,103,103,101,99,99,98,97,96,95,95,95,94,93,93,93,93,
    92,92,92,91,90,90,89,88,88,87,87,87,86,86,84,84,84,84,84,83,82,
    80,80,79,78,78,76,76,76,75,75,75,74,74,73,72,72
  };
  const int n3w3b2r8[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    209,209,209,207,206,206,205,205,204,204,202,202,202,202,202,201,
    200,199,198,196,196,195,194,192,192,191,190,189,188,188,186,185,
    184,184,183,183,182,182,181,180,179,178,177,177,177,177,177,176,
    176,175,174,174,174,174,173,173,172,172,170,169,168,167,166,165,
    164,162,162,161,161,160,160,160,160,159,158,157,157,157,156,156,
    155,155,155,154,154,154,153,152,151,151,150,149,146,146,146,145,
    144,143,143,142,142,140,140,138,133,132,131,131,130,130,126,125,
    125,124,123,122,122,120,120,119,118,118,115,115,113,113,111,111,
    111,111,111,111,111,109,109,109,108,108,107,107,105,105,105,105,
    105,102,101,101,101,101,100,99,99,98,97,97,97,97,96,95,95,93,
    92,91,91,91,90,90,89,89,89,88,84,84,83,83,83,82,82,82,82,80,80,
    80,80,78,78,78,78,78,77,75,75,75,74,74,73,73,73,72
  };
  const int n3w3b2r9[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    209,208,207,207,207,207,206,204,203,202,201,201,201,199,199,199,
    197,196,196,195,194,194,193,192,192,192,191,191,191,189,189,187,
    187,186,186,185,184,183,182,182,182,182,181,179,178,177,177,177,
    176,176,175,174,174,174,174,172,170,170,169,169,168,168,167,167,
    167,166,166,165,165,164,164,164,163,163,163,162,162,162,161,161,
    161,160,159,158,157,156,156,156,156,155,154,153,152,150,149,149,
    148,146,146,146,146,145,144,144,143,143,142,142,142,141,141,139,
    139,137,136,136,135,135,135,133,133,132,132,132,131,129,127,127,
    125,125,124,124,123,122,122,122,121,120,118,118,118,115,114,114,
    113,111,110,109,106,106,104,102,102,102,102,101,101,100,99,98,
    97,96,96,95,95,95,95,94,94,93,92,92,90,90,88,88,88,87,85,83,83,
    82,82,82,81,79,79,77,77,77,76,75,75,75,74,74,74,72,72,72
  };
  const int n3w3b3r0[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    263,260,260,259,258,256,254,253,252,251,249,248,246,243,243,241,
    239,239,238,237,235,235,232,232,227,227,225,225,223,221,220,219,
    217,216,216,215,214,211,211,211,208,208,208,208,207,206,206,205,
    203,202,197,197,195,195,194,192,192,191,190,188,188,185,182,181,
    181,181,180,180,179,177,176,174,172,170,169,165,165,164,163,161,
    159,159,158,157,154,152,149,148,148,146,144,143,142,137,137,133,
    132,130,130,124,123,123,121,121,119,119,112,111,110,109,108,108,
    105,105,104,103,102,101,99,98,98,97,96,95,95,94,93,88,87,83,81,
    80,79,78,78,77,77,76,75,75,74,73,72,72,71,67,66,65,64,63,58,58,
    57,54,54,54,53,53,53,52,52,52,50,50,49,49,49,48,47,47,46,45,45,
    45,43,42,39,37,37,37,36,36,36,35,34,34,31,30,29,28,28,24,24,20,
    20,20,19,19,17,17
  };
  const int n3w3b3r1[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    265,264,262,261,260,259,259,258,258,255,254,250,250,249,248,245,
    244,244,242,241,238,235,234,227,227,225,224,224,224,223,222,222,
    219,218,217,216,215,212,212,210,206,206,205,203,201,201,199,198,
    197,196,196,196,195,194,193,193,191,191,190,190,188,187,184,183,
    181,179,178,176,173,172,172,172,169,169,167,163,162,160,157,156,
    155,154,152,151,149,149,149,145,144,144,143,142,142,142,141,139,
    135,134,133,133,131,130,130,127,126,120,119,119,115,113,113,112,
    105,105,104,101,100,99,98,96,96,95,94,94,91,89,88,86,86,86,84,
    83,76,75,74,73,72,72,72,69,68,66,65,65,63,63,62,62,58,57,56,56,
    56,55,54,53,52,52,52,51,51,51,51,49,47,47,46,46,45,44,43,42,41,
    40,39,38,38,38,38,38,37,37,36,35,34,34,30,29,27,27,24,23,23,23,
    20,20,20,20,16,16
  };
  const int n3w3b3r2[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    266,264,263,262,261,258,258,254,253,252,251,250,250,250,247,246,
    245,243,242,241,239,236,235,234,232,231,230,228,226,225,225,225,
    223,221,220,217,216,215,214,214,211,210,209,208,207,206,205,202,
    202,202,201,200,200,199,199,198,197,197,196,196,194,190,188,188,
    187,184,183,183,182,182,181,180,179,179,179,176,176,176,175,174,
    174,173,172,171,170,170,169,169,168,166,165,162,162,162,160,160,
    159,158,156,155,154,154,153,152,152,151,151,149,149,148,147,147,
    143,143,142,142,141,135,134,131,130,126,124,124,123,121,120,120,
    117,115,114,111,109,109,107,106,105,104,103,103,103,97,94,94,
    92,88,83,83,81,78,77,76,76,74,74,73,71,70,65,64,63,62,62,61,60,
    59,56,54,54,51,51,51,50,48,45,43,42,42,42,40,40,39,37,32,31,30,
    29,29,28,27,25,25,24,22,22,21,21,19,18,17
  };
  const int n3w3b3r3[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    265,265,262,262,262,260,259,259,256,251,251,251,249,248,246,245,
    244,241,239,238,238,238,238,237,237,232,226,224,222,220,219,218,
    217,217,216,214,212,211,209,208,208,208,207,206,205,204,204,203,
    203,201,198,197,197,197,191,191,189,188,188,187,187,182,180,180,
    180,179,179,177,175,175,175,173,173,173,173,173,168,167,166,166,
    166,165,163,162,159,158,158,158,157,155,153,153,151,151,151,150,
    150,149,149,148,144,143,142,138,135,135,135,134,134,133,132,130,
    129,127,126,126,123,121,121,120,118,118,116,116,115,113,113,112,
    111,110,109,108,108,107,106,105,104,100,99,99,98,98,97,97,92,
    91,90,90,88,88,84,84,84,80,76,74,73,71,69,69,68,68,67,67,66,65,
    64,63,63,62,59,59,58,58,57,57,56,55,53,52,52,49,47,46,44,44,40,
    36,32,31,29,29,28,27,24,23,21,20,18,16
  };
  const int n3w3b3r4[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    264,263,262,261,260,260,259,255,255,255,253,252,250,248,243,242,
    241,241,241,236,235,234,233,232,231,230,230,226,226,225,225,224,
    224,221,220,218,216,210,208,206,205,203,203,203,200,196,196,196,
    195,192,192,190,189,189,188,188,187,186,184,184,183,182,180,179,
    179,175,175,173,173,172,171,170,169,169,166,165,163,162,162,162,
    160,160,160,159,159,158,158,157,157,156,153,151,149,149,149,148,
    148,147,147,146,146,146,144,143,142,141,141,139,139,139,138,138,
    138,137,133,132,132,132,126,125,123,121,121,119,119,119,118,118,
    118,116,115,113,109,108,106,105,104,102,100,99,99,97,97,97,97,
    93,93,91,88,85,84,84,83,83,82,81,80,80,79,77,75,73,73,69,69,68,
    66,66,64,63,62,61,57,55,54,53,52,50,49,47,46,45,43,42,37,36,35,
    35,34,34,31,28,28,26,24,24,24,22,18,17
  };
  const int n3w3b3r5[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    266,265,265,261,258,258,256,256,252,250,250,250,249,248,247,246,
    246,245,241,241,238,235,234,228,228,227,227,227,225,225,224,222,
    221,221,217,216,215,214,214,213,209,206,204,204,204,201,201,196,
    195,195,195,194,194,193,192,191,191,191,191,191,191,190,187,187,
    185,183,183,180,178,177,176,175,172,171,170,170,168,167,167,166,
    165,164,164,161,157,156,154,153,153,148,147,146,145,143,143,141,
    141,139,139,138,138,135,134,131,128,128,128,127,127,127,126,125,
    123,123,119,118,115,115,113,113,111,108,107,106,104,99,99,97,
    94,92,91,88,88,87,87,86,86,85,84,84,81,81,79,79,78,78,77,75,74,
    70,69,69,68,66,65,64,64,62,61,61,60,59,54,54,53,52,49,46,46,45,
    44,44,43,41,39,37,35,35,34,34,33,33,33,32,31,29,29,29,28,28,28,
    28,27,25,25,24,23,22,21,21
  };
  const int n3w3b3r6[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    266,264,264,264,264,263,262,262,258,258,256,255,254,252,252,250,
    250,249,248,248,247,245,243,241,237,236,234,233,229,229,229,229,
    229,227,227,227,226,226,225,223,223,220,220,219,219,219,216,212,
    209,208,207,206,204,203,202,197,197,196,193,191,190,190,188,187,
    185,183,182,182,178,177,174,173,171,170,170,169,169,166,165,162,
    161,161,161,159,156,155,153,150,150,148,148,147,147,147,146,144,
    143,143,142,139,138,138,137,137,137,133,133,132,132,128,128,126,
    124,122,121,121,120,117,116,115,115,115,115,114,111,111,107,107,
    106,105,103,100,100,100,98,98,96,96,93,91,91,90,89,87,83,79,79,
    79,78,77,75,69,69,67,67,67,67,64,61,61,58,56,55,54,53,52,51,51,
    51,50,49,48,46,46,46,46,45,44,43,42,41,37,36,36,36,36,35,34,33,
    31,30,29,28,26,25,23,23,21,18,17
  };
  const int n3w3b3r7[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    266,263,263,261,259,259,258,258,255,255,254,252,248,248,247,246,
    245,243,241,236,236,234,234,233,230,230,229,229,228,227,225,224,
    223,221,220,220,218,217,216,216,215,215,214,213,213,212,211,210,
    210,209,209,209,207,206,205,202,202,201,201,201,200,199,195,194,
    191,190,189,188,186,179,178,178,178,178,177,176,174,173,171,168,
    168,166,166,166,164,162,161,161,160,158,156,155,153,153,152,150,
    150,149,149,149,146,144,141,140,138,138,138,137,135,134,132,130,
    128,125,119,119,118,117,112,111,111,110,109,107,106,105,102,102,
    99,99,98,97,96,95,93,92,91,90,89,88,85,84,84,84,83,83,83,82,79,
    78,77,75,74,74,73,73,62,62,61,58,56,55,55,54,54,52,50,49,47,43,
    42,42,42,41,40,39,38,34,34,33,32,29,29,28,27,26,26,25,24,24,23,
    23,21,21,20,17,17,17,16,16
  };
  const int n3w3b3r8[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    266,264,260,260,259,258,257,255,251,251,246,244,244,244,243,242,
    242,240,238,238,237,236,235,232,232,231,231,229,228,228,227,227,
    227,227,223,222,220,218,217,214,212,212,211,210,210,209,207,207,
    203,202,202,201,200,196,196,194,194,192,191,189,188,188,187,181,
    179,179,178,178,177,176,175,174,173,173,172,171,170,169,168,168,
    168,167,167,159,159,158,157,157,156,156,156,152,152,151,151,150,
    148,148,147,146,146,144,143,142,142,141,141,139,139,137,135,134,
    134,133,133,128,127,126,123,123,123,119,119,118,117,117,115,113,
    113,112,111,110,110,108,108,107,106,106,103,102,100,99,98,97,
    97,97,96,91,90,88,88,88,88,82,81,81,78,76,75,75,75,74,74,73,72,
    70,69,68,68,65,64,62,62,60,57,55,54,53,52,52,51,45,43,41,41,38,
    38,37,33,33,30,30,28,28,27,27,26,25,18,17
  };
  const int n3w3b3r9[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    264,263,262,261,259,257,256,256,255,255,253,253,253,251,250,249,
    248,247,246,246,245,244,244,241,240,240,237,235,234,233,229,229,
    229,227,226,225,222,222,222,221,221,218,217,217,216,216,215,215,
    214,213,211,211,211,208,208,208,208,207,206,204,204,199,193,193,
    192,191,191,190,189,189,188,187,185,184,183,181,180,176,175,175,
    175,171,170,169,169,165,164,161,160,159,159,158,158,158,154,154,
    152,151,149,148,146,145,143,142,141,140,137,136,135,131,130,130,
    128,127,126,125,125,124,120,120,119,118,115,114,108,107,107,104,
    103,101,101,97,97,97,96,95,94,94,93,92,92,91,90,89,89,88,85,84,
    84,83,83,78,76,75,74,74,72,70,70,69,68,67,66,65,64,64,60,56,56,
    56,56,52,51,51,50,48,44,41,41,40,37,36,36,35,35,31,31,30,28,28,
    27,26,25,22,21,18,17,17,16,16
  };
  const int n3w4b1r0[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    132,132,132,131,131,131,130,130,129,129,129,129,129,129,128,128,
    128,128,128,127,127,127,126,126,126,126,126,125,125,125,125,125,
    125,125,124,124,123,123,123,123,123,123,123,123,122,122,122,121,
    121,121,121,121,121,121,120,120,120,120,120,119,119,119,119,119,
    119,119,119,119,119,118,118,118,117,117,117,117,117,117,116,116,
    116,116,115,115,115,114,114,114,114,114,113,113,113,113,113,113,
    112,112,112,112,112,111,111,111,111,111,111,110,110,110,110,110,
    110,109,109,109,109,109,109,109,109,108,108,107,107,106,106,106,
    105,105,105,105,104,104,104,104,104,104,104,104,103,103,102,102,
    102,101,101,101,101,101,100,100,100,99,99,99,98,98,98,98,98,97,
    97,97,97,96,96,96,96,96,96,96,95,95,95,95,95,95,94,94,94,94,93,
    93,93,93,93,92,92,92,92,91,91,90,90,90,90,90,90,90
  };
  const int n3w4b1r1[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    132,132,132,132,132,132,132,132,132,131,131,131,131,131,130,130,
    130,129,129,129,129,128,128,128,128,128,128,127,127,127,127,126,
    126,126,126,126,125,125,125,124,124,124,123,123,123,123,122,122,
    122,122,121,121,121,120,120,120,120,120,120,120,119,119,119,119,
    119,119,118,117,117,117,117,117,117,116,116,116,116,116,116,116,
    116,116,116,116,116,116,115,115,114,114,114,114,114,113,113,113,
    113,113,112,112,111,111,111,111,111,111,110,110,110,110,110,110,
    109,109,109,109,109,108,108,108,108,108,107,107,107,106,106,106,
    106,105,105,105,105,104,104,104,104,104,103,103,102,102,102,102,
    102,102,102,102,101,100,100,100,99,99,99,98,98,98,98,97,97,96,
    96,96,96,96,96,96,95,95,95,95,95,94,94,94,94,94,94,94,93,93,92,
    92,92,92,92,91,91,91,91,91,91,91,91,90,90,90,90,90
  };
  const int n3w4b1r2[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    132,132,132,132,132,132,131,131,131,131,131,130,130,130,130,130,
    129,129,129,129,129,129,128,128,128,128,128,128,127,127,127,126,
    126,126,125,125,124,124,124,124,124,124,123,123,123,123,122,122,
    122,122,122,121,121,121,121,121,121,121,121,121,121,120,120,120,
    120,120,120,120,119,119,119,118,118,118,118,118,118,118,118,118,
    117,117,117,117,116,116,116,116,116,116,115,115,114,114,114,114,
    114,114,114,114,113,113,113,113,113,112,112,112,112,112,112,112,
    111,111,111,111,111,110,110,110,110,109,109,108,108,108,107,107,
    107,106,106,106,106,106,106,105,105,105,105,105,105,105,104,104,
    104,104,104,104,104,103,103,103,103,103,102,102,101,101,100,100,
    100,100,100,99,98,98,97,97,97,96,96,96,96,96,96,95,95,95,95,95,
    94,94,93,93,93,92,92,92,92,92,92,91,91,90,90,90,90,90,90,90
  };
  const int n3w4b1r3[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    131,131,131,130,130,130,130,130,130,130,130,129,129,129,128,128,
    128,128,128,128,128,128,126,126,126,126,126,126,125,125,125,125,
    125,124,124,124,124,124,124,124,123,123,123,123,123,122,122,122,
    121,121,121,121,121,120,120,120,120,119,119,119,119,119,118,118,
    118,118,117,117,117,117,117,116,116,116,116,116,116,116,116,115,
    115,115,115,114,114,114,114,114,114,114,114,114,113,113,112,112,
    112,112,112,112,111,111,111,110,110,110,110,110,110,110,110,109,
    109,109,109,108,108,108,107,107,107,107,107,107,107,107,106,106,
    106,106,106,106,106,106,105,105,105,104,104,104,104,104,103,103,
    103,103,103,103,103,102,102,101,101,101,101,100,99,99,99,99,99,
    99,99,99,98,98,98,98,98,98,97,97,97,97,97,97,97,96,96,96,96,96,
    95,95,94,94,94,94,93,93,93,93,93,92,92,92,92,91,91,91
  };
  const int n3w4b1r4[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    132,132,132,132,132,131,131,131,131,131,130,130,130,130,129,129,
    129,129,129,128,127,126,126,126,125,125,125,125,124,124,124,124,
    124,124,123,123,123,123,123,123,123,123,122,122,122,122,122,121,
    121,121,121,121,121,120,120,120,119,119,119,119,119,119,119,119,
    118,118,118,118,118,118,118,118,117,117,116,116,116,115,115,115,
    114,114,114,114,114,114,114,113,113,113,113,112,112,112,112,112,
    112,111,111,111,111,111,111,110,110,110,109,109,109,109,109,109,
    108,108,108,107,107,107,107,107,107,106,106,106,106,106,106,105,
    105,105,105,105,105,104,104,104,104,104,103,103,103,103,103,103,
    103,103,103,102,102,102,102,101,101,101,101,101,101,100,100,100,
    100,100,100,99,98,98,97,97,97,96,96,96,96,96,95,95,95,95,95,95,
    95,95,94,94,93,93,93,93,93,92,92,92,92,91,91,91,91,90,90,90
  };
  const int n3w4b1r5[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    132,132,132,132,132,132,132,131,131,130,130,130,130,130,130,129,
    129,129,129,128,128,128,128,128,128,127,127,127,127,126,126,126,
    126,126,126,125,124,124,124,124,124,123,123,123,122,122,121,121,
    121,121,120,120,120,120,120,120,119,119,119,118,118,118,118,118,
    118,117,117,117,116,116,116,116,116,115,115,115,115,115,115,115,
    114,114,114,114,114,113,113,113,113,113,113,113,113,112,112,112,
    111,111,111,111,111,110,110,109,109,109,109,109,108,108,108,108,
    108,108,108,107,107,107,107,107,107,107,107,106,106,106,106,105,
    104,104,104,104,104,104,104,103,103,103,103,102,102,102,102,102,
    102,101,101,101,101,101,101,100,100,100,100,100,100,100,100,100,
    99,99,99,99,99,98,98,98,98,97,97,97,96,96,95,95,95,94,94,94,94,
    94,93,93,93,93,93,92,92,92,92,91,91,91,91,90,90,90,90,90
  };
  const int n3w4b1r6[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    132,132,132,132,132,132,131,131,131,131,131,131,131,130,130,130,
    130,129,129,129,129,129,129,128,128,128,128,128,128,127,127,127,
    127,126,126,126,126,126,125,125,125,125,125,125,125,124,124,123,
    123,123,123,123,122,122,122,121,121,121,121,121,121,121,120,120,
    120,120,119,119,118,118,118,117,117,117,117,117,116,116,116,116,
    116,116,116,115,115,115,115,114,114,114,114,113,113,113,113,113,
    113,112,112,112,112,112,111,111,111,111,111,111,111,111,111,111,
    111,111,110,109,109,109,109,109,109,108,108,108,108,107,107,107,
    107,107,107,107,107,106,106,106,106,106,106,105,105,105,105,105,
    105,105,104,104,104,104,104,103,103,103,103,103,103,102,102,101,
    100,100,99,99,99,99,99,98,98,98,98,97,97,97,97,97,96,96,96,96,
    96,96,95,95,95,95,94,94,94,92,92,92,91,91,91,91,90,90,90,90
  };
  const int n3w4b1r7[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    132,132,132,132,132,131,131,131,131,131,131,131,131,130,130,130,
    130,130,129,129,129,129,129,129,129,129,128,128,128,127,127,127,
    127,127,126,126,126,126,125,125,125,124,123,123,123,123,123,123,
    123,122,122,122,121,120,120,120,120,120,120,120,120,120,119,119,
    119,119,118,118,118,118,118,117,117,117,117,117,116,116,116,116,
    115,115,115,115,115,114,114,114,114,113,113,113,113,113,113,112,
    112,112,111,111,111,110,110,110,109,109,109,109,109,108,108,107,
    107,107,107,106,106,106,105,105,105,105,105,104,104,104,104,104,
    104,104,104,104,103,103,103,103,102,102,102,102,102,101,101,101,
    100,100,100,100,100,100,100,100,100,99,99,99,99,99,99,99,98,98,
    98,98,97,97,97,97,96,96,96,96,96,96,95,95,95,95,95,94,94,94,94,
    93,93,93,93,93,93,92,92,92,92,92,91,91,90,90,90,90
  };
  const int n3w4b1r8[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    132,132,132,132,131,131,131,131,131,131,131,131,131,131,130,130,
    130,130,130,130,129,129,129,129,129,129,129,129,128,128,128,127,
    127,127,127,126,126,126,126,126,126,126,125,125,124,124,124,124,
    124,123,123,123,123,123,123,123,123,122,122,122,122,122,122,121,
    121,121,121,121,121,121,120,120,120,120,120,120,119,119,119,119,
    119,118,118,118,118,117,117,117,117,116,116,116,115,115,115,115,
    114,114,114,113,113,113,113,112,112,112,111,111,111,111,110,110,
    110,110,110,110,109,109,109,109,109,109,108,108,108,108,107,107,
    107,107,107,106,106,106,106,105,105,105,105,105,105,104,104,104,
    104,103,102,102,102,102,102,102,101,101,101,101,100,100,99,99,
    99,98,98,98,98,98,97,97,97,97,96,96,96,95,95,94,94,94,94,94,94,
    94,94,93,93,92,92,92,91,91,91,91,91,91,90,90,90,90,90,90
  };
  const int n3w4b1r9[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    132,132,132,132,132,132,132,131,131,131,130,130,130,130,130,130,
    129,129,129,129,128,128,127,127,127,127,127,127,127,126,126,126,
    125,125,125,124,124,124,124,124,124,123,123,123,123,122,122,122,
    120,120,120,119,119,119,118,118,118,118,117,117,117,117,117,116,
    116,116,116,116,116,115,115,115,115,115,115,114,114,114,114,114,
    114,113,113,113,113,113,113,113,112,112,112,112,112,112,112,111,
    111,111,111,110,110,110,110,110,110,110,109,109,109,109,108,108,
    108,108,107,107,107,107,107,106,106,106,106,106,106,106,106,105,
    105,105,105,105,105,105,105,105,105,105,104,104,104,103,103,103,
    103,103,102,102,102,102,102,102,101,101,101,101,101,101,100,100,
    100,99,99,99,98,98,98,98,97,97,97,97,96,96,96,96,95,95,95,95,
    95,94,94,94,94,93,93,93,93,93,92,92,92,92,91,90,90,90,90,90
  };
  const int n3w4b2r0[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    165,165,165,165,164,164,164,163,163,163,162,162,161,160,160,159,
    159,157,157,157,156,156,156,156,155,155,154,154,154,154,152,152,
    152,151,151,150,150,149,148,147,147,147,147,146,146,146,146,146,
    144,144,144,143,143,142,142,142,141,140,139,138,136,135,135,135,
    134,134,134,134,133,133,133,133,133,132,132,131,129,128,127,126,
    125,123,122,120,119,119,119,119,117,116,116,116,116,116,116,114,
    114,113,113,113,112,110,110,109,108,108,108,107,105,105,104,102,
    100,100,100,100,100,100,99,99,99,98,97,97,96,96,96,96,95,94,93,
    92,90,90,89,89,88,88,88,88,88,88,87,87,86,86,85,85,85,85,84,83,
    83,83,83,82,81,80,80,80,79,79,79,78,78,77,77,76,76,74,74,72,72,
    71,71,70,70,70,70,69,68,68,68,68,67,67,67,67,64,63,62,62,61,61,
    61,61,61,60,58,58
  };
  const int n3w4b2r1[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    165,164,164,163,163,161,161,160,160,159,159,159,158,158,156,156,
    155,154,153,153,152,152,152,152,152,151,151,150,150,150,149,149,
    149,148,148,147,147,146,146,145,145,143,143,143,142,142,141,140,
    140,139,139,138,138,138,137,137,137,136,135,134,134,133,133,132,
    131,130,129,128,127,127,127,127,127,126,126,126,125,123,122,122,
    120,120,120,120,120,120,119,119,116,116,116,116,115,114,113,112,
    112,112,110,110,109,108,108,107,106,106,105,104,104,103,103,103,
    102,101,101,101,101,100,100,100,99,99,98,98,98,97,94,90,89,89,
    89,88,88,87,87,85,84,84,83,83,83,82,82,82,82,82,81,81,80,79,79,
    79,77,76,76,76,74,74,73,73,73,72,72,72,71,70,70,68,68,67,67,67,
    66,66,66,65,65,65,63,63,63,62,62,62,61,61,61,61,60,60,60,58,58,
    58,58,58,57,57,57,57
  };
  const int n3w4b2r2[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    165,165,163,163,163,162,161,160,160,160,158,157,157,156,156,156,
    155,155,154,153,151,151,150,148,148,147,146,146,146,145,144,144,
    144,143,143,142,141,140,140,139,139,139,138,138,138,137,136,136,
    136,135,135,135,134,134,133,133,133,133,132,129,129,128,125,124,
    123,122,122,122,122,121,121,120,119,119,118,118,118,116,116,115,
    115,115,114,114,114,114,113,113,112,112,112,111,111,111,110,110,
    110,110,109,108,108,105,104,104,104,103,103,103,102,102,102,101,
    100,100,98,98,97,96,95,94,94,94,91,90,89,89,89,88,88,87,85,85,
    85,84,83,83,82,82,82,82,82,82,81,81,81,81,80,79,79,79,78,78,78,
    77,76,75,74,74,74,74,73,73,73,72,72,72,72,71,70,70,70,70,69,69,
    67,66,65,65,64,64,64,63,62,62,62,61,61,61,61,61,59,59,59,59,58,
    58,57,57,57,57
  };
  const int n3w4b2r3[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    165,164,163,162,162,161,160,160,160,159,159,159,158,157,157,157,
    157,156,155,155,154,154,153,153,153,152,151,150,148,147,145,145,
    144,142,142,141,141,141,139,139,139,138,138,137,136,135,134,133,
    132,132,131,131,131,130,130,129,129,127,127,125,125,124,124,124,
    124,123,123,122,122,122,121,121,121,120,119,119,119,119,118,118,
    117,117,116,116,116,115,115,114,114,113,113,113,112,111,111,111,
    109,109,107,107,107,106,106,105,105,104,104,104,104,102,102,100,
    100,99,99,99,98,98,98,97,97,97,96,96,95,94,93,93,92,92,92,92,
    91,91,91,91,91,89,89,89,88,88,88,86,86,86,86,86,85,84,84,84,83,
    82,82,80,80,80,79,79,79,79,78,77,76,76,76,75,74,74,74,73,72,70,
    70,70,69,68,68,67,67,67,66,64,64,63,63,62,61,61,60,59,58,58,58,
    57,57,57,57,57
  };
  const int n3w4b2r4[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    165,165,165,164,164,163,162,162,161,161,160,160,159,158,156,156,
    155,155,154,154,154,153,152,151,151,151,150,149,149,147,147,147,
    146,145,144,144,142,142,141,141,141,141,138,138,138,138,138,138,
    136,136,135,135,135,135,134,134,134,134,133,133,133,132,132,132,
    131,130,130,129,128,128,126,126,126,126,125,124,123,123,122,121,
    121,121,120,119,118,117,116,116,114,114,112,112,111,111,111,111,
    110,109,108,108,108,106,106,106,105,105,103,103,103,103,102,102,
    102,102,101,101,101,101,101,101,99,99,99,98,97,97,95,95,95,94,
    93,92,92,91,91,90,90,88,88,88,86,86,86,85,84,84,84,83,83,83,82,
    81,81,80,80,80,79,78,77,76,76,75,74,73,73,73,72,71,71,70,69,69,
    69,69,69,67,67,67,67,66,66,65,63,62,62,62,60,60,60,60,60,60,59,
    58,58,58,58,58,57,57
  };
  const int n3w4b2r5[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    165,164,164,164,164,164,163,162,161,161,160,159,158,158,158,158,
    157,157,156,156,156,156,155,155,153,153,152,152,152,151,151,151,
    150,149,148,148,148,147,147,147,146,145,145,144,144,143,142,142,
    142,142,142,140,139,139,139,138,137,136,135,135,133,133,133,132,
    132,132,132,132,131,131,130,128,128,127,127,127,127,126,125,125,
    123,123,123,122,122,122,121,121,121,121,119,119,118,117,117,117,
    117,116,116,115,115,114,114,113,113,111,111,111,111,110,110,109,
    109,109,108,108,108,108,106,106,105,104,103,103,102,102,101,98,
    98,98,98,98,97,97,97,96,95,95,94,93,92,92,91,91,90,90,89,87,87,
    87,86,85,85,85,84,84,83,83,82,82,81,81,80,79,78,78,78,78,77,77,
    77,77,76,76,76,76,75,75,73,72,71,71,70,69,67,67,66,66,66,64,64,
    63,62,61,61,61,59,59,58,57
  };
  const int n3w4b2r6[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    165,165,164,162,162,162,162,161,161,161,160,159,155,154,153,153,
    152,152,151,150,150,149,149,149,148,148,146,146,145,144,143,143,
    143,142,142,142,142,141,141,141,141,141,139,138,138,138,138,138,
    138,137,137,136,135,135,135,134,132,132,131,129,129,129,128,128,
    128,128,127,127,127,125,125,125,125,125,124,123,122,121,120,120,
    119,119,117,115,115,115,114,114,113,113,112,111,111,111,110,110,
    109,109,109,109,108,108,108,107,107,106,106,106,106,105,105,105,
    105,104,104,102,101,101,101,100,97,96,96,96,95,95,95,95,94,94,
    94,93,93,92,92,91,91,90,90,88,88,87,87,86,86,85,85,85,85,85,84,
    84,82,81,81,80,79,79,78,78,78,77,77,77,75,74,73,73,72,71,71,71,
    70,70,69,69,68,68,68,68,68,67,67,65,65,64,64,64,63,63,63,62,62,
    59,59,59,59,58,57,57
  };
  const int n3w4b2r7[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    165,163,163,162,162,161,159,159,159,158,157,157,157,157,155,154,
    154,154,154,153,153,152,152,152,151,151,151,151,151,151,150,148,
    147,147,146,146,144,143,143,143,140,140,139,139,138,138,138,137,
    136,136,135,135,135,134,133,132,132,131,130,130,130,129,129,128,
    128,127,127,127,124,124,124,123,123,119,118,118,116,116,116,115,
    115,114,114,112,110,110,110,110,109,109,109,107,107,106,106,106,
    105,105,105,104,103,103,103,102,101,101,101,101,101,100,100,99,
    99,99,98,98,98,98,97,97,97,96,95,95,93,93,93,92,92,92,91,90,90,
    90,90,89,89,88,88,87,86,86,86,86,85,85,84,83,83,82,81,81,81,81,
    80,79,79,79,78,77,77,76,76,75,75,75,75,74,73,73,73,72,72,72,72,
    70,70,69,68,68,67,67,67,66,66,65,65,65,64,62,61,61,60,59,59,58,
    58,58,57,57
  };
  const int n3w4b2r8[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    164,163,162,162,160,159,159,159,158,157,157,157,156,156,156,155,
    154,154,153,153,152,152,152,152,151,151,151,150,150,150,150,148,
    148,147,147,147,147,146,145,145,145,145,144,144,143,142,142,142,
    142,139,139,139,139,138,137,137,137,136,136,135,133,132,132,130,
    130,130,129,129,127,127,126,126,125,125,125,123,123,122,122,122,
    121,121,120,120,120,119,119,118,118,118,116,116,116,115,115,115,
    114,113,111,111,111,111,111,110,109,108,107,107,107,107,106,105,
    105,105,104,103,101,101,100,100,99,98,97,95,95,94,93,93,92,92,
    92,92,90,90,89,89,89,88,88,87,87,87,86,86,86,85,84,84,84,84,83,
    82,81,80,80,79,79,78,78,77,77,77,77,76,75,75,74,74,73,73,73,73,
    71,71,71,71,70,70,70,69,67,66,66,66,66,66,65,64,64,63,63,62,61,
    60,59,59,58,58,57,57
  };
  const int n3w4b2r9[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    163,162,161,161,159,157,157,154,154,153,153,152,152,151,149,149,
    149,149,148,148,147,146,145,144,144,144,143,143,142,142,141,141,
    141,140,139,139,139,138,137,137,137,136,136,136,135,133,132,132,
    131,131,131,130,130,130,129,129,128,128,128,128,128,125,125,124,
    124,124,123,122,122,121,121,121,120,120,120,120,118,118,118,117,
    117,116,116,115,115,113,113,112,111,111,110,110,109,108,107,106,
    106,106,104,104,104,103,103,103,103,103,103,102,102,99,98,97,
    97,97,96,96,95,94,94,93,92,92,91,91,91,91,90,90,90,88,87,87,87,
    86,86,86,86,86,85,85,84,84,84,84,83,83,82,81,81,81,80,80,79,79,
    79,78,78,78,77,76,76,76,75,75,74,74,74,72,72,71,71,71,71,70,70,
    70,69,68,68,68,67,67,67,66,65,63,63,62,61,60,60,60,60,59,59,58,
    58,58,57,57
  };
  const int n3w4b3r0[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    209,208,207,205,205,204,203,201,200,200,199,199,198,198,198,196,
    196,196,196,195,194,193,192,192,192,189,188,187,186,185,185,183,
    182,182,181,181,181,180,179,178,178,177,175,174,174,173,171,170,
    170,170,169,168,166,165,165,164,163,163,162,161,161,161,161,157,
    156,156,154,154,154,151,150,149,148,147,146,146,146,145,144,143,
    141,141,138,138,137,136,136,135,132,130,130,129,128,128,128,127,
    126,126,126,126,122,121,118,118,116,116,114,112,112,111,111,111,
    110,110,110,109,108,108,107,106,105,104,102,101,101,99,94,94,
    94,93,92,92,90,90,90,90,89,88,87,87,86,84,84,82,82,82,81,80,79,
    77,74,74,72,71,70,69,69,68,68,67,66,61,60,57,57,56,56,56,55,49,
    48,48,47,47,46,44,44,39,38,38,38,35,34,33,31,31,30,29,28,26,24,
    24,21,20,20,17,16,16,15,13
  };
  const int n3w4b3r1[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    208,208,207,206,204,202,198,197,197,197,197,196,196,196,195,194,
    192,191,190,189,189,189,186,185,183,181,181,180,179,178,177,177,
    175,172,169,169,165,165,164,163,163,161,161,160,160,159,157,155,
    155,154,153,152,151,151,150,147,147,146,146,145,145,144,144,143,
    142,142,141,141,140,139,136,135,135,132,132,131,130,130,129,128,
    128,128,128,126,123,123,122,121,121,121,119,118,117,117,114,114,
    111,110,110,109,108,108,107,106,106,103,103,98,98,97,97,94,94,
    93,92,90,90,89,89,88,88,88,86,86,84,83,83,83,81,79,77,76,76,76,
    76,73,72,71,71,69,69,68,67,66,66,66,66,66,64,63,63,62,62,61,59,
    57,53,52,52,48,48,46,46,46,45,43,43,42,41,41,38,35,34,33,33,32,
    31,30,29,29,28,28,25,24,23,20,19,19,18,18,18,18,17,16,16,14,14,
    14,13,13
  };
  const int n3w4b3r2[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    206,206,206,206,203,200,200,198,197,196,196,196,194,193,193,192,
    192,192,192,192,191,191,191,190,189,188,188,187,187,186,184,180,
    180,177,177,176,175,175,172,172,171,171,170,170,169,168,168,164,
    162,160,159,159,158,156,154,153,152,149,149,149,148,145,145,145,
    144,144,141,141,140,140,138,138,137,137,136,135,135,135,134,133,
    131,131,130,129,129,129,128,128,127,124,124,124,122,121,120,119,
    115,115,114,113,113,113,113,111,111,111,108,107,107,106,104,104,
    104,103,103,103,102,101,101,100,95,93,92,92,91,91,89,89,88,88,
    87,84,84,84,79,78,78,77,74,72,71,70,69,69,67,66,66,64,63,63,62,
    62,59,57,55,54,54,54,54,52,52,51,50,49,49,49,47,45,45,45,43,43,
    42,41,40,38,38,38,38,37,37,33,31,31,31,29,26,26,25,25,23,22,22,
    21,21,18,18,17,17,13
  };
  const int n3w4b3r3[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    208,206,205,205,204,203,203,202,201,201,201,200,200,199,199,198,
    198,197,196,196,196,195,195,194,193,191,191,189,189,189,188,187,
    187,186,185,183,183,183,183,182,182,181,179,179,179,179,179,177,
    177,176,176,174,173,172,171,170,170,167,166,164,163,163,162,162,
    161,158,155,155,153,151,149,149,148,146,146,144,142,142,142,141,
    141,141,137,136,136,134,134,134,134,134,131,129,129,128,127,125,
    125,124,123,123,123,123,122,120,119,119,118,118,115,115,114,113,
    113,111,106,106,105,104,103,102,101,101,101,100,97,96,96,96,95,
    94,92,92,91,91,91,89,89,89,88,86,86,85,81,79,79,73,72,71,70,70,
    69,68,67,66,65,63,62,60,60,60,59,58,58,58,56,55,53,53,53,49,46,
    43,43,41,40,40,39,39,39,35,34,30,30,30,30,29,28,28,25,24,24,21,
    20,19,18,18,16,15,14,13
  };
  const int n3w4b3r4[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    208,206,205,205,205,204,202,201,201,199,199,198,198,195,194,194,
    193,192,192,191,191,191,187,187,186,186,184,183,182,182,182,182,
    180,180,180,177,175,173,173,172,172,171,171,170,170,169,169,165,
    164,164,163,163,161,157,156,156,155,155,153,152,151,151,151,150,
    148,145,145,145,144,144,144,144,143,142,142,138,136,136,136,134,
    133,132,130,130,129,129,129,127,127,126,123,122,120,119,118,117,
    116,115,112,112,111,111,108,108,108,107,107,107,107,106,106,103,
    102,101,101,101,99,97,94,93,92,92,91,89,87,85,84,83,82,82,82,
    81,81,81,78,78,78,78,76,76,74,71,69,68,68,66,66,63,62,61,59,59,
    58,58,55,55,54,54,53,52,50,48,48,48,47,46,44,44,44,43,43,41,40,
    38,35,35,35,33,32,31,30,29,29,28,27,26,24,24,23,23,22,22,18,18,
    18,17,17,15,14,14
  };
  const int n3w4b3r5[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    209,208,208,207,207,206,206,205,204,203,202,201,200,200,200,199,
    197,197,197,196,195,195,193,192,190,190,188,188,186,186,186,185,
    184,184,184,184,183,181,177,177,173,172,172,170,169,167,166,164,
    163,159,156,156,156,155,154,154,153,153,152,152,152,152,151,146,
    145,145,145,143,143,142,141,138,138,138,137,137,136,135,134,133,
    132,132,131,130,130,129,127,127,126,126,124,124,124,122,120,120,
    119,117,116,110,108,107,106,103,102,98,97,97,95,94,93,93,93,92,
    92,89,88,88,85,85,85,84,80,79,78,77,76,76,75,74,74,74,74,73,72,
    71,71,69,68,67,66,65,65,65,65,65,64,63,63,60,59,55,53,52,52,52,
    51,49,47,47,47,46,45,44,44,44,43,42,42,40,40,40,38,37,36,35,35,
    35,34,33,31,28,27,27,26,24,24,24,24,21,19,18,17,16,15,14,13,13,
    13,13
  };
  const int n3w4b3r6[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    209,208,207,205,205,205,203,199,198,198,197,197,194,192,191,189,
    189,187,186,184,183,183,183,181,180,179,179,177,176,174,174,174,
    173,173,172,168,168,168,166,166,165,165,165,165,164,161,160,160,
    159,159,158,158,157,157,154,153,153,152,151,150,150,148,146,146,
    145,145,144,143,143,141,139,138,138,138,138,137,136,136,135,133,
    133,131,130,129,127,124,124,123,121,119,118,117,116,115,115,115,
    115,114,113,112,111,111,111,110,110,107,106,105,105,105,104,103,
    102,102,102,101,100,100,99,99,99,98,97,96,96,95,92,91,87,86,86,
    85,85,84,84,84,82,81,80,78,78,76,74,74,72,71,71,70,70,67,67,64,
    64,63,62,60,59,58,58,56,55,55,54,53,53,52,52,51,50,49,49,46,46,
    44,44,44,43,43,41,36,35,34,34,34,32,32,29,29,28,28,27,27,21,19,
    17,14,13,13,13,13
  };
  const int n3w4b3r7[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    207,203,202,199,197,196,196,195,195,194,193,192,190,189,189,189,
    188,186,185,184,182,181,179,179,178,178,177,176,176,174,173,172,
    171,171,170,169,168,167,166,164,163,161,161,161,161,154,154,154,
    154,152,150,150,149,149,149,144,143,142,141,141,139,139,139,138,
    137,137,137,136,136,135,135,134,134,133,133,132,130,128,128,127,
    126,125,124,122,121,120,119,117,116,115,115,114,113,112,112,112,
    109,109,109,109,107,106,105,104,102,102,102,101,98,98,98,96,95,
    95,94,94,91,86,86,85,83,82,82,80,75,73,71,70,70,69,69,68,67,67,
    66,65,65,63,62,59,59,58,57,57,54,53,52,51,51,50,50,50,48,46,45,
    44,43,43,43,42,42,41,41,40,39,38,35,35,35,34,33,33,32,32,31,28,
    27,26,24,24,24,24,22,22,20,19,19,18,17,17,17,17,17,16,16,15,15,
    13,13,13
  };
  const int n3w4b3r8[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    209,208,208,207,205,205,205,204,204,202,202,201,201,195,194,194,
    193,193,193,192,192,191,190,190,190,189,187,185,184,183,182,181,
    179,178,176,175,174,174,174,173,172,170,170,167,167,166,166,164,
    161,159,159,158,158,157,155,153,153,152,152,151,151,148,148,147,
    147,143,142,142,141,140,140,139,139,138,137,136,136,134,133,133,
    132,132,131,131,130,129,129,127,125,125,124,123,122,122,122,120,
    119,118,117,115,114,114,111,109,109,108,108,107,107,106,105,105,
    104,102,101,98,96,92,92,91,91,91,88,87,87,87,86,82,81,81,80,80,
    75,75,75,75,73,72,72,70,70,69,69,69,68,66,66,66,65,64,62,61,61,
    61,59,58,56,55,54,52,51,50,49,49,49,47,47,46,44,44,43,42,42,42,
    40,40,40,36,36,34,33,32,32,31,31,28,28,27,26,21,21,20,19,19,17,
    17,16,15,15,14
  };
  const int n3w4b3r9[] = {
    1000, // Capacity
    200, // Number of items
    // Size of items (sorted)
    209,208,207,206,205,204,204,204,204,202,201,198,198,198,197,197,
    196,195,189,189,189,189,187,187,186,186,186,186,185,183,182,181,
    181,177,176,176,176,175,173,172,171,168,167,166,164,164,163,162,
    161,159,159,159,159,157,157,156,155,155,153,153,152,152,152,150,
    149,148,147,147,146,142,141,140,137,134,132,131,131,129,128,128,
    127,125,125,124,124,122,119,119,118,118,117,113,111,111,111,111,
    111,109,109,109,108,108,107,106,106,105,105,105,104,103,102,102,
    100,99,99,98,96,96,94,91,90,90,89,87,87,86,83,81,80,79,79,78,
    78,74,72,72,72,71,71,70,70,70,69,67,63,62,60,58,57,57,57,55,55,
    54,53,53,53,51,51,51,49,48,45,45,45,45,44,43,43,40,37,37,36,36,
    36,35,34,34,33,30,30,30,29,29,27,26,26,24,24,23,22,22,22,22,21,
    20,18,18,16,14
  };
  const int n4w1b1r0[] = {
    1000, // Capacity
    500, // Number of items
    // Size of items (sorted)
    396,396,396,396,395,395,394,394,394,393,393,393,392,392,392,391,
    391,391,391,391,391,391,391,390,390,390,390,390,390,390,389,389,
    388,388,388,388,388,388,388,387,387,387,386,386,385,384,384,384,
    383,382,382,382,382,381,381,381,381,381,380,380,380,379,379,379,
    379,378,378,378,378,378,378,378,377,377,377,376,376,376,376,376,
    376,375,374,374,374,374,374,373,373,372,371,371,370,370,370,370,
    369,369,369,368,368,368,368,368,367,367,367,367,367,367,366,366,
    366,365,364,364,364,364,364,363,363,363,363,362,362,362,362,361,
    360,360,359,359,359,358,358,358,357,357,357,357,357,356,356,356,
    356,356,355,355,355,354,354,354,354,354,354,354,353,353,353,353,
    353,353,353,352,352,352,352,352,352,352,351,351,351,349,349,348,
    348,348,347,347,347,347,347,347,346,346,346,345,345,345,345,345,
    344,344,343,343,343,343,343,343,343,342,342,342,342,341,341,341,
    341,340,340,339,339,338,338,338,338,338,337,337,337,337,336,336,
    336,335,335,334,334,334,333,333,333,333,332,332,331,330,330,330,
    329,328,328,328,328,327,327,327,327,326,326,326,326,326,325,325,
    325,325,324,324,324,323,323,323,322,322,322,322,322,321,321,320,
    320,319,319,319,318,318,318,318,318,318,318,318,317,317,317,317,
    317,317,317,317,317,317,316,315,314,314,314,314,314,313,313,313,
    312,312,312,312,311,311,311,310,310,310,310,310,309,309,309,308,
    308,308,308,306,306,306,306,305,305,305,305,305,304,304,304,303,
    303,302,302,301,301,301,301,300,300,300,299,299,298,298,298,298,
    298,298,298,297,297,297,297,296,296,296,296,296,295,295,295,295,
    294,294,294,294,294,293,293,293,293,293,292,292,292,292,292,291,
    291,291,290,290,290,290,289,289,288,288,288,288,288,288,287,287,
    287,287,286,286,286,285,284,284,284,284,284,283,283,283,283,283,
    282,282,282,282,282,282,281,281,281,281,280,280,280,280,279,279,
    279,278,278,278,278,278,277,277,277,277,276,276,276,276,276,276,
    276,276,275,275,275,275,275,275,275,274,274,274,273,273,273,272,
    272,272,272,272,271,271,271,271,271,271,271,270,270,270,270,269,
    269,269,269,269,268,268,268,267,267,267,267,267,266,266,266,266,
    266,266,266,266
  };
  const int n4w1b1r1[] = {
    1000, // Capacity
    500, // Number of items
    // Size of items (sorted)
    396,396,396,396,396,396,395,395,394,393,393,393,393,392,392,391,
    391,391,390,389,389,389,389,389,388,387,387,387,387,387,386,386,
    385,385,385,385,385,384,384,384,384,384,383,383,383,383,383,382,
    382,382,381,381,380,380,380,380,380,380,379,379,378,378,377,377,
    376,376,376,375,375,375,374,374,373,373,373,373,373,373,373,373,
    372,372,372,372,371,371,371,371,371,370,370,370,370,369,368,368,
    368,368,368,367,367,367,367,367,367,366,366,366,365,364,363,363,
    363,361,360,360,360,359,359,359,359,358,358,358,358,358,357,357,
    357,356,356,356,356,355,355,355,355,355,354,354,354,354,353,353,
    353,352,352,352,351,351,351,350,350,349,349,349,349,349,349,349,
    349,348,348,348,347,347,347,347,347,347,347,346,346,346,346,345,
    345,345,345,344,344,344,344,343,343,343,343,343,343,343,342,342,
    342,340,340,340,340,340,339,339,339,339,339,338,338,338,337,337,
    337,336,336,336,336,335,335,335,334,334,334,333,333,333,333,333,
    332,332,332,332,332,332,332,332,332,332,331,330,330,329,329,328,
    328,328,328,328,328,328,328,327,327,327,327,327,326,326,326,326,
    325,325,325,325,324,324,324,324,324,323,323,323,323,322,322,321,
    321,321,321,321,321,320,320,320,320,320,319,319,319,318,318,317,
    317,317,317,316,316,315,315,315,315,315,315,315,314,314,314,314,
    314,313,313,313,313,313,313,312,312,312,311,311,311,311,310,310,
    310,309,309,308,308,308,308,307,307,307,306,306,306,305,305,305,
    305,304,304,304,303,303,303,303,303,303,303,302,302,302,301,301,
    301,300,300,300,300,300,299,299,299,299,299,298,298,298,298,298,
    298,297,297,296,296,296,295,295,295,295,295,294,293,293,293,293,
    293,293,292,292,292,292,291,291,290,290,290,289,289,288,288,288,
    288,288,288,287,287,287,287,287,287,286,286,286,285,285,285,285,
    285,284,284,284,284,284,284,284,284,283,282,282,282,282,282,281,
    281,281,281,281,281,281,281,281,280,280,279,279,279,279,279,278,
    278,277,277,277,276,276,276,275,275,274,274,274,274,274,274,273,
    272,272,272,272,272,272,272,271,271,271,271,270,270,270,270,270,
    270,269,269,269,269,269,269,269,268,268,268,267,267,267,267,267,
    266,266,266,266
  };
  const int n4w1b1r2[] = {
    1000, // Capacity
    500, // Number of items
    // Size of items (sorted)
    396,396,395,394,394,394,394,394,394,394,394,394,394,393,393,393,
    393,393,392,392,392,392,391,391,391,391,391,389,389,389,388,388,
    387,387,387,387,386,386,386,386,386,385,385,385,385,384,384,383,
    383,383,383,383,383,382,382,381,381,381,381,380,380,380,380,379,
    379,378,378,377,377,377,377,376,376,376,376,376,375,375,375,375,
    375,374,374,374,373,373,373,372,372,372,372,372,371,370,370,370,
    370,369,369,369,368,368,368,368,368,368,368,367,367,367,367,366,
    366,366,366,366,366,365,365,365,365,365,365,365,364,364,364,364,
    364,364,364,364,364,363,363,363,363,363,362,362,362,362,361,361,
    360,360,360,360,360,360,360,359,359,359,358,358,357,357,357,356,
    356,355,355,355,355,354,354,354,354,354,353,353,353,352,352,352,
    352,351,351,351,351,351,350,349,349,348,347,347,347,347,347,345,
    345,344,344,343,343,343,343,343,343,343,342,342,342,342,342,342,
    342,342,342,342,341,341,340,340,340,340,340,339,339,339,339,338,
    337,337,337,337,336,336,336,336,335,335,335,335,334,334,334,334,
    334,333,333,333,333,332,331,331,331,330,330,329,329,329,329,329,
    329,329,328,328,328,328,327,327,327,327,327,327,326,326,326,325,
    325,325,324,323,323,323,322,322,321,321,321,321,321,321,320,319,
    319,318,318,318,317,317,316,316,316,316,316,315,315,314,314,314,
    314,314,314,313,313,313,313,311,311,311,311,311,311,310,310,309,
    309,308,308,308,307,307,307,307,306,306,306,306,306,306,305,305,
    305,304,304,304,304,304,304,304,303,303,302,302,301,301,300,300,
    300,299,299,299,298,298,298,297,297,297,296,296,296,296,296,296,
    296,296,295,295,295,295,295,294,294,293,293,293,293,293,292,291,
    291,291,291,291,290,290,289,289,289,289,289,289,288,288,288,288,
    288,288,287,287,287,287,287,286,286,286,286,286,285,285,285,285,
    285,285,285,284,284,284,283,283,283,283,282,282,282,282,282,281,
    281,281,280,280,280,280,280,279,279,279,279,278,278,278,278,277,
    277,277,276,275,275,275,275,275,275,275,275,274,274,273,273,273,
    273,273,272,272,272,272,272,271,271,271,271,271,271,270,270,270,
    270,270,270,269,269,269,268,268,268,267,267,267,267,267,267,267,
    266,266,266,266
  };
  const int n4w1b1r3[] = {
    1000, // Capacity
    500, // Number of items
    // Size of items (sorted)
    396,396,396,396,395,395,395,394,394,393,393,393,392,392,392,392,
    392,391,391,390,390,390,390,389,389,389,388,388,388,387,387,387,
    387,387,386,386,386,386,386,385,385,385,385,384,384,383,383,383,
    383,383,382,382,382,382,381,381,381,381,381,380,380,379,379,379,
    379,379,378,378,378,378,378,378,377,377,377,377,377,377,376,376,
    376,375,375,375,375,375,375,375,375,375,375,375,374,374,374,374,
    373,373,373,373,373,373,373,372,371,371,371,371,371,370,370,370,
    370,370,369,369,368,368,368,368,367,367,367,367,367,366,366,365,
    365,365,364,364,363,363,363,363,363,363,363,363,362,362,362,362,
    362,361,361,361,361,360,360,360,359,359,359,359,359,358,358,358,
    358,358,357,357,357,356,356,355,355,355,354,354,354,354,354,354,
    353,353,353,353,353,352,351,351,351,351,351,350,350,350,350,350,
    349,348,348,347,347,347,347,346,345,345,345,344,344,344,343,343,
    341,341,341,340,340,340,340,340,340,340,339,339,339,339,338,338,
    338,337,337,337,337,337,337,336,336,336,335,335,335,335,334,334,
    334,334,334,333,333,333,333,333,333,333,332,332,332,331,330,330,
    330,330,329,328,328,327,327,327,327,326,326,326,326,325,325,325,
    324,324,324,324,324,324,323,323,323,323,323,323,323,321,321,321,
    321,320,320,320,320,320,320,319,318,318,317,317,317,317,317,316,
    316,316,316,315,315,315,315,315,315,314,314,314,314,314,313,313,
    312,312,311,311,311,311,311,311,310,310,310,310,310,310,309,309,
    309,309,308,308,308,308,308,307,307,306,306,305,305,304,304,303,
    302,302,302,302,301,301,301,301,301,300,300,300,300,299,299,298,
    298,297,297,297,297,297,296,295,295,295,294,294,294,294,293,293,
    293,293,293,293,293,292,292,292,292,291,291,290,290,290,290,290,
    289,289,289,289,289,289,288,288,288,288,288,287,286,286,286,285,
    285,285,285,285,284,284,284,283,283,283,283,283,283,282,282,282,
    282,281,281,281,281,281,281,280,280,280,280,280,279,279,278,278,
    278,278,278,278,277,277,277,276,276,276,276,275,275,275,275,275,
    275,275,274,274,274,274,274,273,273,273,273,272,272,272,272,272,
    271,271,271,270,269,269,268,268,268,268,268,267,267,267,267,267,
    267,267,267,266
  };
  const int n4w1b1r4[] = {
    1000, // Capacity
    500, // Number of items
    // Size of items (sorted)
    396,396,395,395,394,394,393,393,392,392,392,392,392,392,392,392,
    391,391,391,391,390,390,390,390,390,389,389,389,389,388,387,387,
    387,386,386,386,386,386,385,385,384,383,382,382,382,382,382,382,
    381,381,381,381,381,380,380,380,379,379,378,378,377,377,377,377,
    376,376,376,376,376,376,375,375,375,375,375,374,374,373,373,373,
    373,373,373,373,372,372,372,371,371,371,371,371,371,371,370,369,
    369,369,369,369,368,368,368,368,367,367,367,367,367,367,366,366,
    366,366,365,365,365,365,365,365,365,365,363,363,362,361,361,360,
    360,360,360,359,359,359,358,358,358,357,357,357,357,356,355,355,
    355,355,354,354,354,354,354,353,353,353,352,352,351,351,351,350,
    350,350,349,349,349,349,349,349,349,348,348,348,348,348,348,348,
    348,348,348,347,347,347,346,346,346,346,345,345,344,344,344,344,
    344,344,343,343,343,343,343,343,343,342,341,341,341,341,341,341,
    340,340,339,339,339,339,339,339,339,338,338,338,338,338,338,338,
    338,337,337,337,336,336,336,336,336,335,335,335,335,335,334,334,
    334,334,334,333,333,333,333,333,332,332,332,332,332,331,331,331,
    331,331,330,330,330,329,329,329,328,327,327,327,327,327,326,326,
    326,325,325,325,325,325,325,325,324,324,324,323,322,322,322,322,
    321,321,321,321,320,320,320,320,320,320,320,319,319,319,319,318,
    318,317,317,317,317,316,316,316,316,316,315,314,314,313,313,313,
    312,312,312,312,312,312,312,311,311,311,311,311,310,310,310,310,
    310,309,309,309,309,308,308,308,308,308,308,307,307,306,306,305,
    305,305,305,304,304,304,303,303,302,302,302,301,301,301,301,301,
    301,300,300,299,299,298,297,297,297,296,296,296,296,296,296,295,
    295,295,295,295,295,295,294,294,294,294,294,294,294,293,293,293,
    293,292,292,292,292,292,292,292,291,291,291,290,290,290,290,290,
    289,289,289,289,288,288,288,288,288,287,287,287,287,286,286,286,
    285,285,285,285,284,284,284,284,283,283,283,283,282,282,281,281,
    280,280,280,280,280,279,279,279,279,279,279,279,278,278,277,277,
    277,276,276,275,275,275,274,274,274,274,273,273,273,273,272,272,
    272,269,269,268,268,268,268,268,268,268,267,267,267,267,267,267,
    267,266,266,266
  };
  const int n4w1b1r5[] = {
    1000, // Capacity
    500, // Number of items
    // Size of items (sorted)
    396,396,396,396,395,395,394,394,394,394,393,393,393,392,392,392,
    391,391,391,390,389,389,389,389,389,389,389,388,388,388,387,387,
    387,386,386,386,386,386,386,386,385,385,385,384,384,384,383,382,
    382,381,380,380,379,379,379,379,379,379,378,378,377,377,377,377,
    377,377,377,376,376,376,376,375,375,374,374,374,374,374,374,373,
    373,373,372,372,372,372,372,372,371,371,371,371,370,370,370,369,
    369,369,368,368,368,367,367,367,367,366,366,365,365,365,364,364,
    364,364,364,364,363,363,363,362,362,362,362,361,361,361,360,360,
    360,359,359,359,359,359,359,358,357,357,357,357,357,355,354,354,
    354,353,353,353,353,353,353,353,352,351,351,351,351,351,350,350,
    350,350,350,349,349,349,348,348,348,348,348,348,348,347,347,347,
    347,346,346,346,345,345,344,344,344,344,344,344,343,343,343,343,
    343,342,342,342,341,341,341,341,341,340,339,339,339,339,339,338,
    338,338,338,337,337,337,337,336,336,335,335,335,335,335,335,335,
    334,334,334,334,333,333,333,332,332,332,331,331,331,331,330,330,
    328,328,328,328,328,328,327,327,327,327,327,327,326,326,326,326,
    325,325,325,325,325,324,324,323,323,323,323,323,323,323,323,323,
    322,322,322,321,321,321,321,320,320,320,319,319,319,319,318,318,
    318,318,318,317,317,317,317,317,317,316,316,316,316,315,315,315,
    314,314,314,314,314,314,313,313,313,313,313,312,312,312,312,311,
    311,311,310,310,309,309,308,308,308,307,306,306,306,306,306,306,
    305,305,305,305,304,304,304,303,303,303,302,302,302,301,301,300,
    300,300,300,300,300,299,299,299,298,297,297,297,297,297,296,296,
    296,296,296,296,295,295,294,294,294,293,293,292,292,291,291,291,
    291,291,291,290,290,290,290,289,289,288,288,288,288,288,288,288,
    287,287,287,287,287,287,287,286,286,286,286,286,285,285,285,284,
    284,284,284,284,283,283,283,283,282,282,281,281,281,281,280,280,
    280,280,280,279,279,279,279,278,278,278,278,278,278,278,278,277,
    277,277,276,276,276,276,276,275,275,275,275,274,274,274,274,274,
    274,273,273,273,273,273,273,273,272,272,272,271,271,271,270,270,
    270,270,269,269,269,269,269,269,269,268,268,268,268,268,267,267,
    267,266,266,266
  };
  const int n4w1b1r6[] = {
    1000, // Capacity
    500, // Number of items
    // Size of items (sorted)
    396,396,396,396,396,395,395,395,394,394,394,394,394,394,393,393,
    393,393,393,392,392,392,392,392,392,392,391,391,391,391,391,391,
    391,390,390,390,390,389,388,388,388,387,387,387,387,387,387,387,
    387,386,385,385,385,385,385,385,384,384,384,384,384,384,383,383,
    383,383,382,382,382,382,382,382,382,382,381,381,381,381,381,380,
    379,379,379,378,378,378,377,377,377,377,377,377,376,376,376,375,
    375,374,374,374,373,373,373,372,372,372,372,371,371,371,371,370,
    370,370,370,370,370,369,369,369,368,368,368,368,367,367,367,367,
    367,367,366,366,366,366,365,365,365,365,364,364,364,363,363,363,
    362,362,362,362,362,362,362,361,361,360,360,360,360,359,358,358,
    357,357,357,357,356,356,356,356,356,356,356,355,355,355,355,354,
    354,354,354,354,353,353,353,353,352,352,352,352,351,351,351,350,
    349,349,349,349,349,348,348,348,347,347,347,347,347,346,346,346,
    345,345,344,344,344,343,343,343,343,343,342,342,342,342,342,342,
    341,341,341,340,340,340,340,340,339,339,338,338,338,338,337,336,
    336,336,336,336,336,335,335,335,335,334,334,334,333,333,333,333,
    332,332,332,332,331,331,331,330,330,330,330,330,330,328,328,328,
    328,327,327,327,326,326,326,326,325,325,325,324,324,324,324,324,
    323,323,323,323,323,323,322,322,321,321,321,321,321,320,320,319,
    319,319,319,319,319,318,318,317,317,317,317,316,316,316,316,316,
    316,315,315,315,315,314,314,314,314,313,313,313,313,313,312,312,
    312,312,311,310,309,309,309,309,309,308,308,308,308,307,307,307,
    307,306,306,306,305,305,305,305,304,304,304,304,303,303,303,302,
    302,302,302,302,301,301,301,301,299,299,299,298,296,296,296,296,
    295,295,295,294,294,294,294,294,294,294,293,293,293,293,293,292,
    292,292,291,291,291,291,291,291,290,289,289,288,288,287,287,287,
    287,286,286,286,285,285,284,284,284,284,284,283,283,283,282,282,
    282,281,281,280,280,280,279,279,278,278,278,278,278,277,277,277,
    276,276,276,276,276,276,276,276,276,276,275,275,275,275,275,275,
    275,275,274,274,274,273,273,272,272,272,272,272,272,272,271,271,
    271,271,271,271,271,270,270,270,270,269,269,269,268,268,267,267,
    267,266,266,266
  };
  const int n4w1b1r7[] = {
    1000, // Capacity
    500, // Number of items
    // Size of items (sorted)
    396,396,395,395,394,394,394,393,392,392,392,392,392,391,391,391,
    391,390,390,390,390,390,390,389,389,388,388,388,387,387,387,387,
    386,386,385,385,385,385,384,384,384,384,384,384,383,383,383,383,
    383,382,382,382,381,381,381,381,381,380,379,379,379,379,379,379,
    379,378,378,378,378,378,377,377,377,377,376,376,375,375,374,374,
    374,374,374,373,373,372,372,372,371,371,371,370,370,370,370,369,
    369,369,369,369,368,368,368,367,367,367,366,366,365,365,365,364,
    364,364,364,363,363,362,362,361,361,360,360,360,360,360,360,360,
    360,360,359,359,358,358,358,358,357,357,357,357,356,356,356,355,
    355,355,354,353,353,353,352,352,352,352,352,352,352,352,352,351,
    351,351,350,350,350,349,349,349,349,349,348,348,348,347,347,347,
    347,346,346,346,345,345,345,344,344,344,344,344,343,343,343,342,
    342,342,342,342,342,342,342,341,341,341,341,340,340,340,340,339,
    339,338,338,338,337,337,337,337,337,337,336,336,336,336,336,336,
    336,336,335,335,335,335,334,334,333,333,333,332,332,332,332,332,
    332,332,331,331,331,331,331,330,330,330,330,330,330,330,330,330,
    330,329,329,329,329,329,328,328,328,327,327,326,326,326,326,325,
    324,324,324,323,323,322,322,322,321,321,321,321,320,320,320,320,
    319,319,318,318,318,318,318,318,317,317,317,317,316,316,316,316,
    316,315,315,315,314,314,314,314,313,313,313,313,313,313,311,311,
    311,310,310,310,310,310,309,307,307,306,306,306,306,306,306,306,
    305,305,305,305,304,304,304,304,303,303,303,303,303,303,303,303,
    302,302,302,301,301,301,301,301,301,301,301,301,300,300,299,299,
    299,299,298,298,297,297,297,296,296,296,295,295,295,294,294,293,
    293,293,293,293,292,292,292,292,292,292,291,291,291,291,291,291,
    291,291,291,291,290,289,289,288,288,288,287,287,287,286,286,286,
    285,285,284,284,284,284,284,284,283,283,283,283,283,283,282,282,
    282,282,282,281,281,281,281,281,281,280,280,280,280,280,280,280,
    280,280,279,279,279,279,279,278,277,277,276,276,275,275,275,275,
    275,275,275,274,274,274,273,273,273,271,271,271,271,271,271,271,
    270,270,270,270,270,269,269,269,269,268,268,268,267,267,267,267,
    267,267,267,267
  };
  const int n4w1b1r8[] = {
    1000, // Capacity
    500, // Number of items
    // Size of items (sorted)
    396,396,396,395,395,394,394,393,393,393,393,393,392,392,392,392,
    392,391,391,390,390,390,390,389,389,389,389,389,389,389,388,388,
    388,387,387,387,387,387,386,386,385,385,385,384,384,384,383,383,
    383,383,383,383,382,382,382,382,382,381,381,381,380,380,379,379,
    379,379,379,378,378,378,378,377,377,377,377,376,376,376,375,375,
    375,375,375,375,374,374,374,373,373,373,372,372,372,371,371,371,
    370,370,370,370,369,368,368,368,367,367,367,367,366,366,366,365,
    365,365,365,365,365,365,364,364,364,363,363,363,363,362,362,362,
    362,361,361,361,361,361,361,361,360,360,360,360,359,359,359,359,
    358,358,358,357,357,357,357,357,356,355,355,355,355,355,355,354,
    354,354,354,354,353,353,353,353,352,352,352,351,351,351,351,350,
    350,349,347,347,347,347,346,346,345,344,344,343,343,343,343,343,
    343,343,342,342,342,342,342,341,341,341,340,340,340,340,339,339,
    339,338,337,337,337,337,337,337,337,336,336,336,335,335,335,335,
    335,334,334,334,333,333,333,332,332,332,331,330,330,329,329,329,
    328,328,328,328,327,327,327,327,326,326,326,325,325,325,324,324,
    324,324,323,323,323,323,323,323,321,321,321,321,321,321,320,320,
    319,319,319,318,318,318,318,317,317,316,316,316,316,315,315,315,
    315,315,314,314,314,314,313,313,313,313,313,313,312,312,312,311,
    311,311,311,311,310,310,310,309,309,309,309,308,308,308,308,307,
    307,307,307,306,306,306,306,306,306,305,304,304,304,304,304,303,
    303,303,303,303,303,302,302,301,301,300,300,300,300,300,299,299,
    299,299,299,299,298,298,298,298,298,297,297,297,296,296,296,296,
    296,296,296,295,295,295,295,294,294,294,294,294,293,293,293,293,
    293,292,292,291,291,291,291,291,291,290,290,290,290,290,290,290,
    289,289,289,289,289,288,288,288,287,287,287,286,286,286,285,285,
    284,284,284,284,283,283,283,283,283,283,283,282,282,282,282,281,
    281,281,281,280,280,280,280,279,279,279,279,278,278,278,278,278,
    278,277,277,277,277,277,277,277,277,277,276,276,276,276,275,275,
    275,275,275,274,274,274,274,273,272,272,272,272,272,272,271,271,
    270,270,270,270,270,270,270,270,270,268,268,268,267,267,267,267,
    266,266,266,266
  };
  const int n4w1b1r9[] = {
    1000, // Capacity
    500, // Number of items
    // Size of items (sorted)
    396,396,396,396,395,395,395,395,395,395,395,394,394,394,393,393,
    393,392,392,392,392,392,392,390,390,389,389,389,389,389,388,388,
    388,388,388,387,387,387,387,387,387,386,386,385,385,385,385,384,
    384,384,384,384,384,384,384,383,383,383,383,383,382,382,382,382,
    382,381,381,381,381,380,380,380,380,380,380,379,379,379,379,378,
    378,378,377,377,377,377,376,376,376,376,376,376,376,375,375,375,
    374,374,374,374,374,373,373,373,372,372,372,372,371,371,371,371,
    371,371,371,371,371,371,370,370,369,369,369,369,368,368,368,367,
    367,367,367,367,367,366,365,365,365,365,364,364,364,364,363,363,
    363,363,362,362,361,361,360,360,360,360,360,360,359,359,359,359,
    358,358,358,358,358,358,357,357,357,357,356,356,356,355,355,355,
    355,354,353,353,353,353,353,353,353,353,352,352,352,352,352,351,
    350,350,350,350,350,350,350,349,349,349,349,349,348,348,347,347,
    346,346,346,346,346,345,345,344,344,344,343,343,343,342,342,342,
    342,342,342,342,341,341,341,341,341,340,340,340,340,340,340,339,
    339,339,339,339,339,338,338,338,338,337,337,337,337,337,336,336,
    335,334,334,334,333,333,333,333,333,332,332,331,331,331,331,331,
    331,330,329,329,328,328,327,327,327,327,326,326,326,325,325,325,
    325,325,325,325,324,324,324,323,323,323,323,322,322,322,322,322,
    321,320,320,320,320,319,318,318,318,318,318,317,317,316,316,316,
    316,316,315,315,315,315,315,315,315,315,315,315,314,314,314,314,
    313,313,313,313,312,312,312,312,312,311,311,310,310,310,309,309,
    308,308,307,307,307,307,307,307,306,306,306,306,304,304,304,303,
    303,303,302,302,302,302,301,300,300,300,300,300,300,299,299,298,
    297,297,297,297,295,295,295,295,295,295,295,295,294,294,294,294,
    293,293,293,292,292,292,291,291,291,291,291,291,291,290,290,290,
    290,290,289,289,289,289,288,287,287,287,287,286,285,285,284,284,
    284,284,284,283,283,283,282,282,282,281,281,281,281,280,280,279,
    279,279,279,278,277,277,276,276,276,276,276,276,275,275,275,274,
    274,274,274,273,273,273,272,272,272,272,272,272,272,272,271,271,
    270,270,270,269,269,269,269,268,268,268,268,267,267,267,267,266,
    266,266,266,266
  };
  const int n4w1b2r0[] = {
    1000, // Capacity
    500, // Number of items
    // Size of items (sorted)
    495,492,491,489,489,489,488,488,486,485,485,484,483,482,481,481,
    479,479,478,478,477,476,475,475,475,475,473,473,472,472,469,468,
    468,468,468,467,467,466,466,466,466,465,465,464,463,462,461,459,
    459,459,457,457,456,456,456,456,456,454,453,452,452,452,451,449,
    448,448,447,446,446,446,446,445,444,444,444,444,443,443,443,443,
    442,442,442,439,438,437,436,435,435,434,434,433,433,431,431,431,
    430,430,430,430,429,427,427,426,426,425,425,425,424,424,424,423,
    422,422,422,422,421,421,418,417,417,416,416,416,416,415,414,413,
    412,412,411,411,411,410,408,407,406,405,403,403,403,402,400,399,
    399,399,398,398,397,397,397,395,395,395,393,392,392,391,390,390,
    387,385,384,383,383,382,381,381,381,380,380,379,379,378,378,377,
    376,376,375,375,374,373,372,371,371,371,370,370,370,369,368,367,
    366,366,366,365,365,365,364,364,364,362,362,362,360,356,355,354,
    354,353,353,351,351,350,349,348,346,346,344,344,343,341,341,340,
    339,338,336,333,333,333,332,332,329,329,327,327,327,326,325,325,
    325,325,323,323,323,322,322,321,321,321,321,321,321,320,320,320,
    319,318,318,317,317,316,316,316,315,314,312,312,312,312,311,311,
    311,311,309,308,306,306,305,305,305,305,304,304,304,304,303,303,
    303,303,303,299,299,299,298,298,297,297,296,296,295,294,293,292,
    292,290,290,289,288,288,288,287,285,285,285,284,283,282,279,277,
    277,277,277,276,275,275,274,273,272,272,270,268,267,266,266,266,
    266,265,264,264,264,264,264,264,263,263,263,263,262,261,261,261,
    259,258,257,257,256,255,255,255,254,253,253,253,251,251,251,250,
    250,250,249,247,246,245,244,244,242,241,240,238,237,237,236,235,
    233,233,233,232,232,231,231,230,230,229,228,227,227,226,226,225,
    225,225,225,224,223,222,221,221,220,219,216,216,216,215,214,214,
    214,213,213,212,212,211,211,209,208,207,207,207,206,206,205,205,
    205,204,204,203,203,202,201,201,201,201,201,200,199,198,198,197,
    197,195,193,193,192,191,190,190,190,188,188,187,187,187,187,186,
    186,185,185,184,184,183,182,182,182,182,182,180,180,180,180,180,
    180,179,177,177,177,176,175,175,175,175,174,172,171,171,170,169,
    168,168,168,167
  };
  const int n4w1b2r1[] = {
    1000, // Capacity
    500, // Number of items
    // Size of items (sorted)
    494,494,493,492,490,489,487,487,486,485,485,485,485,483,483,482,
    482,481,481,480,478,477,476,476,475,475,475,474,474,474,474,473,
    473,472,471,471,471,471,470,470,470,467,467,467,467,466,466,466,
    466,464,464,464,463,463,460,460,459,459,459,458,458,458,456,455,
    455,455,454,452,452,452,451,450,449,447,446,446,446,446,445,445,
    444,444,443,442,442,441,441,441,440,438,438,437,437,436,436,435,
    435,434,433,432,432,432,431,431,430,427,427,427,426,426,425,425,
    423,423,423,422,422,422,421,421,420,420,419,418,417,417,417,416,
    416,416,413,413,413,412,412,411,410,410,409,409,407,407,407,407,
    405,404,404,402,402,400,399,398,396,396,395,394,394,394,393,393,
    393,391,390,389,389,389,388,388,388,387,386,385,385,384,384,383,
    383,382,382,382,380,380,380,380,379,379,378,378,378,378,377,377,
    375,375,374,373,373,373,372,371,370,370,369,369,368,368,367,366,
    366,366,365,364,364,364,364,364,361,361,361,360,359,359,359,358,
    357,357,355,355,354,354,354,353,352,352,351,351,350,349,349,349,
    349,348,347,347,346,345,345,345,345,344,343,343,343,343,342,342,
    341,341,341,341,340,338,338,337,336,336,336,335,335,335,334,334,
    332,331,330,330,330,329,329,329,329,328,328,328,327,327,325,325,
    325,325,323,323,322,322,321,320,319,318,318,317,316,315,315,315,
    314,313,313,313,312,311,310,309,307,307,306,306,306,306,304,304,
    303,303,302,302,300,300,300,299,298,298,297,297,296,295,295,294,
    293,293,292,291,291,291,290,288,286,285,285,284,284,283,282,282,
    282,279,278,277,276,276,276,275,274,273,273,272,272,271,270,270,
    270,269,269,266,266,265,262,262,261,261,260,260,256,255,253,253,
    251,251,250,249,249,246,246,242,241,241,241,240,240,239,239,237,
    236,235,235,235,234,233,233,233,232,232,232,230,229,228,227,226,
    225,224,223,223,222,222,220,220,220,219,219,217,217,216,215,215,
    215,214,213,212,212,211,210,210,209,208,208,208,208,207,207,206,
    206,205,205,205,204,203,203,201,200,199,199,198,198,198,198,197,
    196,196,195,195,194,194,190,190,190,190,189,186,186,184,183,183,
    181,180,179,179,177,177,176,175,174,174,174,174,173,172,171,171,
    170,168,167,167
  };
  const int n4w1b2r2[] = {
    1000, // Capacity
    500, // Number of items
    // Size of items (sorted)
    495,494,494,493,492,491,491,490,490,489,489,488,488,487,487,487,
    485,485,485,484,484,483,483,482,481,479,479,479,478,478,478,476,
    476,475,474,474,474,474,472,470,469,468,468,467,466,466,466,466,
    465,465,465,464,464,463,462,462,461,461,460,459,459,456,455,452,
    452,452,451,450,449,449,449,449,449,448,448,446,442,442,441,441,
    441,440,440,440,439,439,438,437,437,437,435,435,434,433,432,431,
    431,431,431,431,430,429,429,427,427,427,426,426,425,423,422,420,
    420,419,418,415,414,414,414,413,413,413,413,410,409,409,408,408,
    407,406,406,406,405,404,404,404,403,402,402,401,400,400,399,398,
    393,393,392,391,391,389,389,387,387,385,385,384,383,382,382,381,
    381,381,379,379,378,375,373,372,371,370,370,370,368,367,367,366,
    365,364,363,363,362,361,361,360,360,360,359,358,357,357,357,356,
    356,355,354,353,350,350,348,347,347,347,346,346,345,345,344,343,
    343,343,342,342,341,341,341,341,341,341,341,340,340,337,337,335,
    335,335,335,333,332,332,332,331,330,329,329,328,327,327,326,325,
    325,325,324,324,322,322,322,321,321,319,317,316,316,316,316,316,
    315,315,313,313,313,313,312,311,310,309,308,307,307,307,305,304,
    304,304,302,302,301,301,301,301,300,300,299,299,299,298,297,296,
    296,296,296,296,294,294,292,292,290,290,289,288,288,287,287,287,
    287,286,286,285,285,284,283,282,282,281,281,281,280,280,280,278,
    278,278,278,276,276,275,274,273,273,272,271,271,271,269,269,266,
    265,265,264,264,263,263,262,262,262,261,261,258,258,257,256,256,
    255,254,254,254,254,253,253,253,251,251,250,250,250,250,250,249,
    249,248,248,248,248,248,247,247,247,246,246,246,246,243,241,240,
    240,238,238,238,238,237,237,237,237,236,236,235,235,234,232,230,
    229,229,229,228,228,228,228,228,227,227,226,226,225,224,224,224,
    223,222,222,222,221,220,220,220,219,219,216,213,213,213,212,212,
    212,212,210,210,209,209,208,208,208,207,207,207,207,206,206,206,
    206,204,204,203,203,202,202,202,202,201,201,199,199,198,197,196,
    196,195,195,195,194,193,193,192,190,190,189,188,187,186,186,186,
    185,185,184,184,184,184,183,182,180,178,175,173,171,170,170,169,
    168,167,167,167
  };
  const int n4w1b2r3[] = {
    1000, // Capacity
    500, // Number of items
    // Size of items (sorted)
    495,493,493,490,490,489,489,489,488,488,487,486,486,486,485,485,
    485,485,485,484,484,483,482,481,480,480,478,477,475,475,475,474,
    474,474,473,472,471,470,470,470,470,469,468,467,467,467,466,465,
    465,464,464,464,464,463,462,459,458,458,458,457,457,456,456,455,
    454,454,454,454,452,451,451,449,449,449,448,446,444,444,443,442,
    439,438,438,438,438,438,437,436,436,435,434,433,432,432,432,431,
    431,430,429,428,427,426,426,425,425,425,424,424,423,423,422,421,
    419,419,419,418,418,417,416,416,414,413,413,413,411,411,411,410,
    409,409,409,407,404,404,403,402,401,401,400,400,398,398,397,397,
    396,396,396,396,395,395,394,393,393,392,389,388,388,386,386,385,
    385,385,384,384,384,383,383,383,381,381,380,380,379,378,378,377,
    376,375,374,374,374,372,372,372,370,370,369,369,368,368,368,367,
    367,366,366,366,365,364,362,362,362,361,361,359,359,359,357,356,
    356,355,354,354,354,353,353,351,350,350,350,350,348,348,348,347,
    347,346,345,345,344,344,344,343,343,342,342,341,340,340,340,340,
    340,339,338,337,336,335,333,333,332,332,330,330,326,323,323,323,
    323,322,321,321,320,319,319,317,316,316,315,315,314,314,312,312,
    311,311,311,311,311,311,311,311,309,308,307,307,307,306,305,304,
    304,304,303,302,300,300,299,298,297,297,296,295,295,295,294,293,
    293,293,293,292,291,290,290,289,288,288,287,286,286,286,285,283,
    282,282,282,281,280,280,280,280,279,278,278,278,278,277,276,275,
    275,275,274,274,273,273,272,272,271,271,271,271,270,269,268,267,
    267,266,265,265,265,263,262,261,261,260,259,259,258,258,257,257,
    256,256,256,254,254,253,253,253,252,251,250,247,247,246,244,244,
    244,243,243,242,242,241,240,240,239,239,239,238,237,237,237,237,
    237,236,235,234,234,234,233,232,232,232,231,231,230,230,229,229,
    227,227,225,225,225,224,223,222,221,220,220,220,218,218,217,216,
    216,216,214,213,213,213,212,211,211,210,209,208,208,207,207,206,
    206,206,206,205,205,203,202,201,201,200,200,200,200,198,197,197,
    196,196,195,195,194,193,191,191,189,188,187,186,185,184,183,182,
    181,181,181,179,178,178,177,177,176,176,176,175,175,174,173,171,
    170,169,168,167
  };
  const int n4w1b2r4[] = {
    1000, // Capacity
    500, // Number of items
    // Size of items (sorted)
    495,492,492,491,491,490,490,490,489,488,487,486,486,486,485,484,
    481,480,480,480,479,479,478,476,475,475,473,473,471,471,471,470,
    470,468,468,468,467,467,465,464,463,463,462,461,460,459,459,458,
    458,458,456,452,452,451,450,450,448,447,447,447,447,446,446,446,
    445,445,443,443,442,442,441,441,441,440,439,438,438,438,438,437,
    436,436,435,435,434,434,432,432,432,432,430,430,429,429,429,428,
    428,427,426,425,424,423,423,423,422,421,419,419,418,418,417,417,
    416,414,413,413,413,413,412,411,410,409,409,408,406,406,405,404,
    404,404,403,402,400,398,398,398,397,397,397,395,394,393,393,392,
    392,392,390,389,389,389,389,385,385,385,385,385,384,383,383,383,
    381,381,379,379,377,377,376,375,375,375,375,374,373,372,371,371,
    370,369,369,369,369,369,366,366,366,365,364,364,364,363,363,362,
    362,361,361,361,360,359,357,356,356,356,356,356,355,353,353,353,
    352,352,351,351,349,349,348,348,347,347,347,346,346,346,345,344,
    343,343,342,340,340,340,339,338,337,337,336,335,333,333,333,332,
    332,330,330,330,329,329,329,327,326,326,324,324,322,322,321,321,
    321,320,320,319,319,319,318,318,318,318,318,317,317,316,314,313,
    312,312,310,310,310,309,308,308,308,306,306,306,306,305,305,304,
    302,301,301,300,299,298,298,296,295,295,293,293,293,293,293,292,
    292,292,291,291,290,290,289,288,288,288,286,285,285,285,285,284,
    284,284,283,281,281,280,280,280,278,278,277,277,276,276,276,275,
    274,274,273,271,271,270,270,270,269,268,268,268,267,266,266,265,
    264,263,262,262,262,262,261,261,260,260,260,260,259,258,258,256,
    256,255,254,253,252,251,251,249,248,247,246,246,246,246,246,245,
    245,245,245,244,244,244,244,243,243,243,242,242,240,240,239,239,
    239,238,238,236,235,235,235,234,234,234,233,233,233,232,231,229,
    228,228,228,227,226,226,225,222,222,219,219,218,218,217,216,216,
    215,215,215,213,212,212,212,211,211,210,210,209,209,208,208,207,
    207,206,206,205,204,203,202,201,200,200,200,200,198,197,197,196,
    195,193,192,191,191,190,189,189,189,189,189,188,188,187,186,185,
    185,181,181,180,180,177,176,176,174,174,172,172,171,170,169,169,
    169,168,167,167
  };
  const int n4w1b2r5[] = {
    1000, // Capacity
    500, // Number of items
    // Size of items (sorted)
    495,493,491,491,491,490,490,490,488,488,486,486,486,484,484,484,
    484,483,482,482,482,478,477,476,476,473,473,470,470,469,468,468,
    467,467,467,467,466,466,466,465,465,464,463,460,459,459,459,457,
    457,456,455,455,455,453,453,452,451,450,449,449,449,448,448,448,
    448,448,447,446,446,444,444,443,442,440,440,439,439,436,434,433,
    432,431,431,430,427,427,426,426,426,426,425,424,424,424,423,423,
    419,419,418,417,416,415,415,415,414,413,411,411,410,409,409,407,
    407,407,406,406,405,404,404,403,403,402,401,400,399,399,399,398,
    397,397,397,396,396,395,394,394,394,394,393,393,392,392,391,390,
    390,389,388,387,387,386,385,384,383,381,381,381,381,380,379,378,
    378,377,376,374,373,373,373,373,372,371,370,370,370,369,369,369,
    369,369,368,368,366,365,364,364,364,364,362,362,362,361,360,360,
    360,359,358,358,357,356,356,356,355,355,355,353,353,352,352,351,
    351,350,350,350,349,348,348,348,346,346,346,346,346,343,343,343,
    341,340,340,339,337,337,336,336,336,334,331,331,331,331,330,328,
    327,325,324,323,323,321,318,318,318,315,315,315,313,313,313,312,
    311,309,309,309,309,308,308,307,307,306,306,305,304,304,302,302,
    301,300,299,298,297,297,297,296,296,296,296,295,294,294,293,293,
    291,290,289,289,289,288,287,285,283,283,282,280,280,280,279,279,
    279,278,278,277,277,277,277,276,275,275,275,275,274,274,273,272,
    272,272,271,270,270,270,269,269,269,268,268,267,266,266,264,264,
    264,264,264,264,263,261,260,260,260,259,259,258,258,257,256,256,
    254,254,253,252,252,251,250,249,249,249,249,248,248,246,245,245,
    244,243,243,243,243,240,240,240,239,238,238,238,238,237,237,236,
    235,235,234,232,231,231,231,230,229,228,228,227,226,226,223,223,
    222,222,221,221,220,220,219,218,217,216,216,214,214,214,214,212,
    212,212,212,211,210,210,210,209,207,206,205,203,202,202,201,201,
    200,199,199,198,198,197,196,195,195,194,193,193,192,192,192,191,
    191,190,190,190,189,189,188,188,187,186,186,186,185,185,185,184,
    183,182,182,181,180,180,180,179,179,179,179,178,178,178,177,177,
    176,176,176,175,174,174,173,173,171,171,171,170,170,170,168,168,
    167,167,167,167
  };
  const int n4w1b2r6[] = {
    1000, // Capacity
    500, // Number of items
    // Size of items (sorted)
    495,494,493,493,492,492,491,490,490,490,490,489,487,487,487,486,
    486,486,485,485,484,484,484,483,479,478,478,476,475,474,473,473,
    472,471,471,469,467,466,464,462,462,462,462,462,461,461,461,460,
    459,459,458,457,457,456,456,455,454,454,453,453,453,453,453,452,
    451,451,450,449,449,449,449,449,448,447,446,446,445,445,444,443,
    441,441,441,440,438,438,438,437,437,436,435,435,435,434,434,434,
    434,433,433,432,432,431,431,431,430,430,429,428,428,428,428,428,
    428,428,427,427,426,425,425,424,424,423,423,423,423,421,420,420,
    419,418,418,417,417,417,417,417,417,417,416,415,415,414,414,414,
    411,411,410,410,409,408,408,408,407,406,405,405,404,402,402,402,
    402,401,401,401,401,401,400,400,398,397,396,396,395,395,394,393,
    393,393,392,391,390,389,388,388,387,387,387,385,385,384,384,383,
    382,382,381,380,380,379,379,378,378,377,377,377,375,374,374,373,
    373,373,373,371,371,371,370,370,370,370,369,369,366,364,363,360,
    360,359,359,358,357,357,357,355,355,355,355,353,352,352,351,349,
    349,349,348,347,347,345,344,344,344,342,341,341,341,340,339,338,
    337,337,335,335,334,334,334,334,333,333,333,332,332,332,331,331,
    329,329,328,327,327,325,324,324,323,323,322,322,322,320,319,319,
    319,319,318,317,315,315,314,314,313,313,313,312,311,310,310,309,
    308,307,306,305,305,304,303,300,296,296,295,294,293,292,291,290,
    290,289,288,285,285,284,283,283,282,282,279,279,278,278,276,275,
    275,275,275,273,271,271,270,270,270,270,269,269,268,268,267,267,
    266,265,265,263,263,263,262,262,262,261,259,259,258,258,258,256,
    256,256,255,254,254,253,253,253,251,251,250,249,247,245,244,243,
    241,238,238,238,237,236,236,235,235,234,232,231,231,231,229,229,
    229,228,227,227,227,226,225,224,224,224,224,222,222,222,221,219,
    218,218,218,218,217,215,214,214,213,212,211,211,210,210,210,208,
    208,207,206,206,205,205,205,204,204,203,203,203,201,201,200,200,
    200,198,196,196,196,196,196,195,195,194,194,192,191,190,189,189,
    188,188,186,186,185,184,184,184,184,183,183,182,181,180,180,179,
    179,176,175,175,174,173,173,172,172,172,172,171,170,170,169,169,
    168,168,168,168
  };
  const int n4w1b2r7[] = {
    1000, // Capacity
    500, // Number of items
    // Size of items (sorted)
    495,495,495,495,495,494,494,493,493,492,492,491,490,490,490,489,
    489,489,488,488,486,486,485,485,484,483,482,482,480,479,479,478,
    477,476,474,472,472,471,471,471,471,471,470,469,468,468,467,466,
    466,464,463,462,462,462,462,461,460,460,460,460,459,459,459,457,
    457,456,455,455,454,454,454,453,453,452,452,451,451,451,450,449,
    448,448,447,447,446,446,446,445,444,444,443,442,440,440,440,440,
    440,440,438,438,436,436,434,433,431,431,430,430,428,427,426,425,
    418,417,416,416,415,415,414,414,414,413,412,412,411,411,411,411,
    411,410,409,408,408,407,406,406,405,405,405,405,404,404,404,404,
    403,403,403,402,402,401,401,401,400,399,398,397,397,397,396,396,
    395,395,395,395,394,393,391,391,386,385,385,385,384,383,382,381,
    380,380,380,379,378,378,377,376,375,375,374,374,373,373,373,372,
    372,371,371,370,370,369,368,367,367,367,365,364,364,364,364,362,
    360,360,359,359,359,358,358,358,357,357,356,355,354,354,354,354,
    354,352,352,351,351,351,350,350,350,349,347,347,346,345,345,342,
    342,341,341,341,341,339,339,339,338,337,337,337,337,337,336,335,
    335,334,333,333,332,332,328,326,326,326,326,324,323,323,321,321,
    320,319,318,317,316,316,316,315,315,315,314,313,313,313,311,311,
    311,311,311,311,310,310,310,309,309,309,309,308,308,308,307,307,
    306,306,304,303,303,302,301,300,299,299,298,298,298,297,297,297,
    297,295,294,294,293,293,292,292,292,291,291,290,290,290,289,287,
    287,286,283,283,282,281,281,280,279,279,278,278,276,276,275,274,
    274,274,271,269,269,268,268,268,266,265,263,261,261,257,257,257,
    256,255,255,253,253,252,251,251,250,249,249,248,247,246,245,245,
    244,244,242,242,241,239,238,237,236,235,235,234,234,233,233,232,
    231,230,230,230,229,228,227,226,225,225,224,223,222,221,221,220,
    218,218,217,215,214,214,214,214,214,214,213,213,211,210,209,208,
    208,207,207,207,207,206,206,203,203,203,202,202,200,198,198,197,
    197,196,196,196,195,195,195,194,193,193,192,192,192,191,191,190,
    189,187,187,187,187,186,186,186,186,185,185,184,184,184,183,183,
    182,182,182,180,180,179,178,178,177,175,175,174,171,171,168,168,
    168,168,168,167
  };
  const int n4w1b2r8[] = {
    1000, // Capacity
    500, // Number of items
    // Size of items (sorted)
    495,495,495,495,493,492,491,491,490,490,490,489,489,488,488,488,
    487,487,487,487,487,485,485,484,482,482,481,481,480,480,480,479,
    479,478,478,478,478,478,477,477,477,476,475,475,474,474,474,473,
    472,471,470,470,468,467,466,466,465,465,465,465,464,464,464,463,
    462,462,462,461,461,457,457,457,456,456,455,455,454,453,448,448,
    448,448,447,447,447,446,443,442,441,437,436,436,436,436,435,435,
    434,434,433,432,432,432,432,431,431,431,430,429,429,429,428,427,
    426,426,425,425,425,425,425,424,424,422,421,420,420,418,418,416,
    415,415,415,414,414,413,413,413,410,409,409,409,408,407,406,405,
    404,404,404,403,403,401,401,400,399,398,397,396,396,396,395,395,
    394,393,393,392,392,392,391,391,390,388,388,387,387,387,386,386,
    385,385,384,383,383,382,380,380,380,380,380,378,376,376,375,374,
    374,374,373,373,371,369,369,367,367,366,366,366,366,365,364,364,
    363,363,363,363,362,362,359,359,358,357,356,356,355,355,355,354,
    354,353,353,352,351,350,350,348,348,347,347,346,346,345,344,343,
    342,342,341,341,339,338,338,338,337,337,337,336,336,334,333,332,
    332,331,329,329,328,328,326,323,323,322,322,322,321,321,320,318,
    317,316,315,315,314,314,313,312,312,310,310,309,308,308,307,306,
    306,305,305,304,304,303,302,301,301,300,299,298,298,296,295,295,
    292,292,291,291,291,290,290,288,288,288,285,285,285,284,284,282,
    282,281,281,281,281,278,278,276,275,275,274,274,273,273,272,272,
    271,270,270,268,267,267,267,264,263,263,263,263,261,261,260,259,
    258,258,258,256,255,255,255,255,254,252,252,250,249,248,248,248,
    248,247,246,246,246,245,245,245,245,244,244,244,244,244,244,242,
    242,240,240,240,239,239,238,237,237,236,236,234,234,232,232,232,
    231,230,229,228,228,227,227,226,225,225,225,223,223,222,222,222,
    220,220,220,218,218,215,215,214,214,213,213,213,212,211,211,210,
    209,208,208,207,207,207,206,204,204,204,204,202,202,200,200,199,
    197,197,196,196,196,195,194,194,193,193,191,189,188,187,185,185,
    185,184,183,183,183,183,183,182,182,182,179,179,179,179,178,178,
    178,178,177,177,176,176,176,176,175,175,174,174,172,171,170,169,
    169,167,167,167
  };
  const int n4w1b2r9[] = {
    1000, // Capacity
    500, // Number of items
    // Size of items (sorted)
    494,494,494,494,493,492,492,491,491,490,490,490,490,489,489,487,
    486,486,486,485,485,484,484,483,482,481,480,479,477,477,476,476,
    474,474,474,473,473,473,473,473,472,470,470,468,468,468,467,467,
    467,466,465,462,462,462,461,460,460,460,460,459,459,458,457,457,
    457,456,456,455,452,452,452,452,451,450,449,449,448,448,446,446,
    446,445,443,443,443,443,441,441,441,440,440,440,439,438,436,436,
    435,434,434,433,433,432,431,431,430,429,428,427,427,426,426,424,
    424,422,422,422,421,421,421,419,418,418,418,417,417,416,415,415,
    414,414,413,413,413,412,412,412,411,411,410,408,408,407,407,406,
    406,405,405,404,403,403,403,401,401,400,400,400,400,398,396,396,
    396,395,395,393,393,393,393,392,391,391,390,390,390,390,390,389,
    388,387,385,384,384,384,384,383,383,382,382,380,380,379,378,378,
    377,376,376,376,376,375,373,373,371,371,371,371,370,369,369,369,
    369,368,367,367,365,365,364,364,364,364,363,363,363,363,363,362,
    362,362,361,361,359,359,359,358,358,357,357,355,354,353,353,353,
    353,351,351,351,351,351,350,349,348,348,347,346,345,345,344,344,
    343,342,342,341,341,340,339,338,337,336,336,336,336,336,335,334,
    333,333,333,333,332,332,331,330,329,328,328,327,326,326,325,323,
    321,321,320,319,318,318,317,317,317,317,316,315,315,313,313,312,
    312,311,310,310,309,309,309,308,308,308,307,307,305,304,303,302,
    301,301,299,298,297,297,294,293,290,289,289,289,288,287,287,286,
    286,285,284,284,283,282,281,279,278,278,278,278,277,277,276,276,
    271,271,270,269,269,266,265,265,265,264,264,263,263,263,263,262,
    258,257,257,257,254,253,253,252,251,250,250,249,247,247,246,243,
    243,242,242,241,239,238,238,236,236,235,235,234,234,233,232,229,
    228,228,228,224,223,223,221,220,219,218,217,216,216,215,215,214,
    214,212,212,212,210,210,209,208,208,208,206,206,205,204,204,203,
    203,202,202,202,201,201,201,200,200,199,199,197,197,197,196,196,
    196,195,195,194,194,194,193,193,193,192,192,190,190,190,190,189,
    188,188,187,187,186,185,185,183,182,182,181,181,181,180,180,180,
    179,178,178,177,177,176,175,175,175,174,174,174,173,171,170,170,
    169,169,169,167
  };
  const int n4w1b3r0[] = {
    1000, // Capacity
    500, // Number of items
    // Size of items (sorted)
    626,622,621,619,619,619,617,617,617,615,613,611,610,610,608,607,
    607,607,607,606,605,602,602,600,599,599,599,597,595,593,590,590,
    589,589,589,588,588,586,585,584,583,583,583,582,581,581,580,578,
    578,578,576,576,576,574,573,573,572,571,570,569,569,567,563,562,
    562,560,559,558,556,555,553,551,548,546,545,542,541,537,536,534,
    533,531,530,529,528,528,526,525,524,523,523,523,522,521,521,517,
    512,509,509,505,501,498,497,496,496,494,493,493,492,490,490,489,
    485,482,482,481,481,479,478,477,477,475,473,472,467,465,465,465,
    464,463,462,462,461,460,459,459,458,456,456,456,455,453,453,449,
    449,448,448,448,446,446,445,444,443,442,442,441,439,438,438,436,
    436,435,435,435,434,433,431,431,428,428,427,426,424,421,420,419,
    419,418,418,417,416,413,413,412,409,406,404,403,403,402,402,402,
    401,398,396,395,393,389,387,386,384,384,384,382,381,380,379,376,
    376,375,373,370,369,367,366,365,364,364,363,363,362,360,359,357,
    356,355,354,354,351,350,349,348,347,347,347,346,342,341,339,338,
    338,337,336,334,333,330,330,330,329,329,329,328,327,327,327,325,
    322,322,319,318,318,317,313,308,307,307,306,305,303,302,302,301,
    301,301,298,297,297,296,295,294,293,289,286,286,285,285,284,284,
    284,281,280,278,274,273,273,272,271,270,270,269,269,268,267,267,
    266,264,264,261,259,257,257,255,254,253,253,252,250,249,249,249,
    248,248,247,243,243,243,242,242,242,242,241,239,237,236,236,233,
    231,229,229,228,227,227,227,226,225,224,223,222,222,219,218,218,
    215,215,215,213,213,211,210,208,207,206,204,202,201,199,197,197,
    196,194,193,193,192,190,189,189,184,184,183,182,181,181,181,181,
    175,173,172,171,169,169,163,161,158,158,157,157,155,155,154,153,
    153,151,150,149,148,147,147,144,144,144,143,143,141,141,139,137,
    137,137,136,136,134,131,130,130,130,130,126,126,121,120,117,117,
    116,115,114,110,108,107,106,105,105,102,101,99,96,95,91,91,91,
    89,87,85,84,82,82,81,80,80,77,77,74,72,72,71,71,70,70,69,68,68,
    68,67,66,66,63,61,59,58,55,54,54,54,53,52,52,52,51,50,49,48,47,
    46,42,41,39,38,37,36,35,35
  };
  const int n4w1b3r1[] = {
    1000, // Capacity
    500, // Number of items
    // Size of items (sorted)
    627,626,625,625,624,623,619,619,618,617,616,616,614,614,613,612,
    611,608,608,607,607,607,603,602,602,602,602,599,599,599,596,593,
    593,593,592,591,591,590,589,589,588,586,586,585,584,584,583,582,
    581,581,580,577,575,572,571,569,567,566,565,564,563,562,562,562,
    561,561,561,561,559,558,557,557,556,553,550,550,549,549,547,546,
    545,544,542,540,539,539,538,536,535,535,535,531,531,529,529,527,
    526,526,523,520,520,519,517,516,513,512,512,512,512,511,511,510,
    508,507,506,506,505,505,504,503,503,499,499,499,497,496,494,493,
    490,489,489,487,487,487,482,480,480,480,478,476,475,472,469,468,
    467,466,466,466,464,464,462,460,460,459,458,457,457,454,453,453,
    452,451,451,449,448,446,445,443,443,442,442,440,440,439,439,438,
    437,436,434,432,431,431,429,428,425,425,423,423,423,422,422,420,
    419,419,418,417,416,415,415,413,413,411,410,408,408,406,397,397,
    393,392,388,385,384,381,381,380,380,379,379,377,377,376,375,375,
    374,373,373,373,370,369,368,367,366,365,364,363,363,363,362,360,
    359,355,353,351,348,347,346,346,344,342,341,340,340,338,337,336,
    336,335,334,333,332,331,330,330,329,329,328,328,328,326,325,324,
    322,322,321,319,319,318,318,318,316,314,313,312,311,308,307,304,
    303,301,300,298,294,292,292,292,291,289,286,285,285,283,279,278,
    275,270,270,270,269,269,268,267,265,264,263,262,259,255,254,252,
    251,247,245,243,243,241,241,239,239,235,232,232,231,229,229,228,
    228,225,224,218,217,217,215,213,212,211,211,210,210,208,207,203,
    202,201,201,201,200,200,198,198,198,196,195,194,194,193,192,191,
    191,191,191,191,191,189,189,188,187,185,185,182,181,180,180,179,
    178,176,176,175,175,174,170,169,167,167,166,164,164,164,163,163,
    161,159,159,157,157,156,156,156,148,148,148,146,145,145,144,143,
    142,139,137,136,133,131,130,129,128,127,126,124,124,122,121,120,
    117,116,116,115,115,113,112,110,109,107,104,103,101,101,100,99,
    99,98,98,97,97,97,97,96,94,94,94,92,91,91,91,91,90,88,87,85,85,
    84,83,82,82,81,80,79,77,76,74,73,71,67,67,63,61,60,60,56,54,51,
    50,48,46,45,43,42,40,40,39,36
  };
  const int n4w1b3r2[] = {
    1000, // Capacity
    500, // Number of items
    // Size of items (sorted)
    627,621,618,617,616,615,615,614,611,611,610,609,609,609,609,608,
    608,608,605,605,604,603,602,601,598,598,598,597,596,596,596,596,
    596,595,594,593,592,591,588,587,586,585,584,584,583,582,580,579,
    579,578,578,576,574,574,573,571,571,570,570,570,570,569,567,566,
    565,565,564,564,563,561,561,561,559,559,559,556,556,555,551,550,
    548,547,546,546,543,543,540,538,538,536,532,532,531,531,529,529,
    528,528,527,525,524,523,523,522,521,520,519,517,516,512,512,510,
    510,510,509,509,506,506,505,503,503,502,501,501,500,500,500,499,
    499,497,497,496,495,495,495,494,491,490,489,488,487,486,486,486,
    483,482,481,481,479,478,477,477,477,476,475,474,473,471,471,469,
    467,467,463,461,456,453,452,451,451,451,449,448,447,447,444,443,
    441,440,440,438,438,432,431,430,429,428,427,426,425,425,423,422,
    422,421,421,420,420,418,418,414,413,413,412,412,411,409,409,408,
    405,404,401,398,398,395,394,390,390,389,389,388,388,387,387,386,
    385,384,383,381,380,380,378,377,376,376,374,373,370,369,369,365,
    362,361,361,360,358,356,353,353,352,351,350,348,346,346,345,343,
    342,341,341,338,337,337,335,334,333,331,331,329,326,324,323,322,
    321,321,318,317,314,314,314,312,312,312,311,308,306,304,303,301,
    301,299,299,299,298,297,295,294,293,293,290,287,286,280,280,278,
    278,276,274,274,274,274,272,269,269,269,268,262,260,259,258,257,
    257,256,255,255,254,252,251,245,241,240,240,239,237,237,236,235,
    233,231,231,230,227,226,226,223,222,222,222,220,219,218,216,208,
    208,207,206,206,206,206,206,206,204,203,202,202,200,200,197,196,
    193,192,191,189,188,186,186,185,185,183,181,181,180,179,178,177,
    176,176,174,174,174,174,172,171,168,167,167,166,166,163,161,159,
    159,159,157,157,156,156,152,151,149,148,146,146,145,143,142,140,
    139,136,136,135,134,134,130,128,128,127,126,126,125,124,123,121,
    120,118,114,113,113,112,111,111,110,109,109,108,108,108,107,106,
    105,105,103,103,103,101,101,98,97,96,93,90,90,89,85,84,81,80,
    76,75,75,75,75,74,74,70,68,66,64,63,62,62,61,60,57,55,55,55,52,
    51,51,47,42,41,40,40,39,38,38,37,37,36
  };
  const int n4w1b3r3[] = {
    1000, // Capacity
    500, // Number of items
    // Size of items (sorted)
    625,625,624,623,622,622,621,619,619,618,614,613,612,611,611,609,
    607,606,605,604,600,599,596,596,595,594,592,591,588,586,583,581,
    579,577,577,576,573,573,573,573,572,571,570,569,567,566,566,566,
    566,565,563,562,560,559,559,559,559,558,558,556,553,552,552,548,
    548,547,546,545,545,542,542,542,542,541,540,539,539,535,532,530,
    529,529,528,527,527,525,524,524,524,520,517,517,514,514,511,510,
    509,509,509,509,508,507,507,505,504,504,504,502,499,499,496,494,
    493,491,490,489,489,489,488,485,485,483,483,481,480,479,479,476,
    475,475,474,473,467,466,466,466,465,464,461,461,461,461,461,460,
    460,459,459,457,456,454,454,454,452,450,449,448,448,447,443,442,
    442,441,439,439,439,439,438,437,433,433,433,433,433,433,432,432,
    432,431,431,429,428,428,426,425,425,423,423,422,420,420,420,420,
    417,414,411,410,410,409,409,408,407,407,405,400,399,398,397,397,
    395,394,394,394,389,389,387,384,384,381,380,379,379,379,378,377,
    377,376,374,373,373,372,372,369,368,368,368,368,367,366,365,363,
    363,361,358,355,350,348,347,344,344,343,339,339,337,336,335,334,
    333,333,332,332,331,330,328,327,327,326,326,326,325,325,321,321,
    320,320,320,317,311,311,311,310,309,309,306,304,302,302,300,299,
    298,297,295,295,294,293,293,292,291,291,291,289,289,289,288,288,
    285,284,284,284,282,282,279,279,278,277,276,276,275,274,270,270,
    269,269,269,268,268,260,260,259,259,259,258,256,254,253,250,249,
    248,246,246,245,243,243,243,242,239,239,238,235,232,231,231,225,
    224,220,219,219,215,214,212,212,211,210,209,207,206,205,205,204,
    202,202,202,201,200,200,199,198,198,197,196,192,190,190,187,187,
    182,180,180,178,177,177,175,175,173,172,168,166,165,161,160,159,
    157,155,152,152,150,150,145,145,144,139,139,139,139,138,138,137,
    133,132,131,131,130,130,129,129,127,123,123,122,121,121,120,120,
    118,118,118,118,118,115,113,113,111,111,109,109,107,107,103,102,
    102,102,99,98,95,95,94,93,90,89,87,87,86,85,81,81,80,79,78,78,
    76,75,74,72,69,69,66,64,63,59,58,57,56,56,56,55,54,54,54,53,53,
    51,51,50,49,49,47,47,44,40,40,36
  };
  const int n4w1b3r4[] = {
    1000, // Capacity
    500, // Number of items
    // Size of items (sorted)
    626,626,625,623,623,622,621,619,619,617,616,615,614,613,613,610,
    607,605,604,601,600,598,596,595,592,591,590,589,589,588,587,586,
    584,583,581,581,577,574,572,571,568,565,565,563,563,563,558,557,
    557,556,555,554,553,553,553,546,545,545,543,543,543,542,541,540,
    538,537,537,535,533,532,531,530,529,527,526,525,520,520,519,518,
    517,515,514,513,511,509,508,506,505,501,497,497,496,493,491,486,
    485,485,481,477,475,473,471,468,468,467,467,467,464,463,461,460,
    457,457,457,456,450,450,448,447,447,445,445,443,443,441,439,438,
    438,437,434,434,431,430,427,425,424,424,423,422,422,421,420,419,
    419,418,415,412,412,412,410,410,408,407,407,406,405,403,403,399,
    398,397,397,396,395,394,394,393,390,388,387,386,386,385,381,378,
    378,377,377,376,375,372,370,369,368,367,366,366,366,366,366,364,
    363,362,362,362,361,360,359,358,357,356,356,352,351,350,350,350,
    349,348,347,347,343,343,343,342,342,340,340,338,338,337,337,337,
    336,334,333,331,330,329,328,326,323,323,322,321,319,318,318,317,
    316,316,316,316,314,313,310,310,308,308,308,307,305,305,305,304,
    304,304,304,304,303,303,303,302,300,299,298,298,297,297,297,293,
    290,290,289,288,287,286,286,281,280,279,278,277,276,274,273,272,
    271,269,269,269,268,266,266,266,264,263,263,263,260,259,259,258,
    258,254,252,248,247,245,245,244,242,242,241,240,239,235,235,232,
    232,231,230,229,228,227,227,225,225,220,220,219,217,216,213,213,
    212,211,208,208,208,208,203,200,200,199,199,198,198,197,197,197,
    195,195,194,194,192,190,190,188,187,187,186,185,183,183,182,182,
    182,180,180,178,177,176,176,175,174,172,172,171,170,167,166,166,
    161,160,160,158,158,156,156,156,156,153,153,152,150,148,147,147,
    147,141,140,139,139,138,138,138,135,134,131,131,130,128,126,126,
    125,125,125,124,123,123,123,120,119,119,118,117,116,115,114,113,
    113,112,111,110,107,106,105,105,104,103,103,101,100,100,98,98,
    98,98,98,96,94,93,91,89,88,85,84,82,81,78,78,77,75,75,74,72,71,
    70,68,67,66,64,64,64,64,59,58,58,57,56,54,54,52,51,50,49,46,45,
    45,43,43,43,42,39,38,38,37,36
  };
  const int n4w1b3r5[] = {
    1000, // Capacity
    500, // Number of items
    // Size of items (sorted)
    627,626,625,624,624,621,619,618,618,617,616,609,608,608,608,606,
    606,605,604,604,604,602,601,600,598,595,594,592,591,590,589,589,
    586,586,584,583,583,581,581,580,579,577,576,575,575,574,574,572,
    570,570,569,567,567,564,563,563,563,560,558,554,553,552,550,550,
    549,548,548,548,546,545,543,543,542,542,540,539,537,536,536,534,
    533,530,526,523,522,521,520,520,519,519,517,517,516,516,511,510,
    510,506,503,503,502,502,499,498,497,497,496,495,491,491,491,490,
    489,489,486,482,481,481,481,478,477,477,477,476,475,475,474,472,
    471,471,469,467,467,467,466,463,462,462,461,461,458,457,454,453,
    452,450,449,449,449,446,446,445,443,441,441,437,435,434,434,432,
    432,430,429,426,425,425,424,421,421,418,418,417,415,411,411,411,
    408,407,406,405,404,404,403,403,403,402,400,399,396,395,395,395,
    392,391,391,391,390,390,388,388,387,385,384,381,381,381,380,380,
    380,380,377,377,375,374,373,372,371,371,369,368,366,366,366,365,
    364,364,359,355,351,351,350,348,347,347,346,344,342,340,339,338,
    337,336,335,332,331,331,331,329,329,327,327,326,325,324,324,324,
    320,320,320,319,318,318,317,316,315,314,314,314,314,312,306,304,
    303,301,300,300,299,297,297,296,292,291,288,288,288,284,283,282,
    277,275,272,272,271,270,268,263,261,261,261,261,260,256,256,256,
    254,254,250,249,249,246,246,243,242,239,237,231,231,230,230,230,
    229,225,224,223,223,222,222,216,216,215,214,214,213,212,211,210,
    209,209,208,206,203,201,199,199,199,198,196,196,195,195,192,192,
    190,188,185,183,183,181,181,180,179,178,176,175,173,170,170,170,
    168,167,167,161,159,156,156,156,156,155,154,154,153,152,151,150,
    149,148,144,143,142,141,140,140,139,138,137,136,136,130,129,129,
    128,124,122,121,121,121,115,115,114,114,112,112,111,111,108,108,
    108,107,107,106,106,106,106,106,102,101,101,99,98,98,98,98,97,
    97,95,94,90,89,89,88,86,86,86,85,84,81,81,80,80,79,79,79,77,77,
    76,75,75,74,74,74,74,73,72,68,67,66,65,65,64,63,62,62,61,61,60,
    60,60,59,58,58,55,55,54,53,53,50,48,46,45,45,45,44,43,43,40,39,
    38,37,37,37
  };
  const int n4w1b3r6[] = {
    1000, // Capacity
    500, // Number of items
    // Size of items (sorted)
    626,626,625,625,622,621,621,621,620,620,620,619,618,616,616,616,
    616,615,615,611,610,610,608,606,603,602,601,599,598,597,597,595,
    594,594,592,591,589,586,586,584,581,578,578,578,577,575,574,573,
    570,570,568,564,562,561,560,558,556,555,554,553,552,551,549,547,
    547,546,546,543,542,541,540,539,539,538,536,535,533,532,530,529,
    529,528,527,526,523,522,521,520,517,516,515,515,512,512,512,512,
    511,511,510,509,509,506,505,503,503,503,502,502,501,501,501,501,
    499,498,496,495,493,492,492,491,489,489,488,488,488,487,487,484,
    480,480,478,477,476,476,474,474,474,474,472,471,468,468,465,464,
    464,463,463,462,461,459,459,458,454,451,449,449,449,447,447,446,
    446,443,443,441,440,439,439,436,434,432,432,432,431,430,428,426,
    425,423,423,422,420,418,418,417,416,415,412,409,409,403,402,401,
    400,399,399,398,394,394,392,392,392,391,388,386,384,384,384,382,
    382,381,380,379,379,378,377,377,374,374,373,373,372,371,370,370,
    370,369,368,368,367,367,367,366,366,366,363,363,363,363,362,361,
    361,360,360,358,357,357,356,355,355,350,350,349,348,347,345,345,
    342,341,340,339,337,336,336,335,334,333,331,331,329,329,327,324,
    323,323,316,316,313,312,311,309,309,307,304,302,301,297,296,295,
    294,293,293,292,292,290,289,288,286,286,283,281,279,278,278,276,
    272,272,272,270,269,268,267,265,265,263,262,260,259,258,258,254,
    252,252,252,248,248,246,246,245,244,244,241,241,240,239,237,236,
    231,230,229,228,224,223,220,218,218,218,217,216,215,215,214,214,
    212,211,211,211,209,209,206,206,204,203,200,198,194,193,193,193,
    193,192,191,189,189,189,188,188,187,187,187,187,186,183,182,181,
    180,179,179,178,178,177,174,173,170,170,169,167,166,164,164,164,
    161,160,159,158,158,157,157,157,157,156,155,153,152,151,151,150,
    148,147,144,142,140,137,136,134,134,133,130,130,129,129,128,127,
    127,127,124,124,124,124,123,121,118,115,115,115,112,112,110,105,
    104,103,101,100,100,99,98,94,94,94,93,93,93,86,85,84,83,82,81,
    81,81,79,78,78,77,75,73,71,65,64,64,63,63,62,60,59,57,56,56,54,
    53,53,53,49,48,45,45,42,42,41,39,36
  };
  const int n4w1b3r7[] = {
    1000, // Capacity
    500, // Number of items
    // Size of items (sorted)
    626,625,624,621,621,620,618,618,617,616,615,615,615,614,614,609,
    605,603,602,602,601,600,599,597,597,597,592,592,589,588,587,583,
    583,582,582,579,579,578,578,572,571,568,567,567,566,564,564,564,
    563,563,563,562,562,562,560,560,560,559,555,555,555,554,554,554,
    551,550,549,548,547,546,545,545,542,542,541,538,537,536,535,535,
    535,534,532,532,531,531,530,528,527,522,515,514,514,510,510,509,
    509,508,507,507,507,505,504,504,502,501,501,499,496,494,491,491,
    490,490,486,485,485,485,485,482,482,480,480,477,477,475,473,472,
    472,472,470,470,466,465,463,462,461,460,456,456,454,453,451,451,
    449,447,445,444,444,440,440,437,436,435,435,435,435,433,433,428,
    428,426,426,425,424,423,417,415,415,414,411,411,411,409,408,403,
    403,401,399,399,398,397,396,396,395,393,390,390,389,385,385,384,
    383,383,382,382,379,379,378,376,374,374,373,373,368,366,365,363,
    362,362,362,360,359,357,357,356,355,353,352,352,351,351,350,349,
    348,347,346,346,345,344,343,342,342,341,341,340,340,340,340,340,
    340,339,338,337,337,336,335,332,331,328,325,324,324,323,321,321,
    319,318,318,314,313,312,310,310,310,309,309,308,306,306,306,305,
    301,296,295,295,293,293,292,292,292,290,290,290,289,287,286,283,
    282,281,281,278,277,275,273,272,270,269,268,268,263,262,260,260,
    257,256,256,256,255,255,248,247,246,244,243,242,239,238,235,235,
    233,231,229,229,228,227,227,227,226,226,225,224,220,213,212,212,
    210,209,208,208,206,205,204,204,202,201,199,198,197,196,195,194,
    194,194,191,191,188,188,183,182,181,181,181,181,181,177,176,175,
    175,173,173,172,171,171,170,170,170,169,167,166,166,165,164,163,
    163,161,161,161,161,159,157,157,155,155,154,152,152,152,152,150,
    150,149,148,147,146,145,144,141,140,140,139,137,137,136,136,136,
    134,131,130,130,130,126,125,124,123,119,119,118,117,117,115,113,
    113,112,112,112,112,111,111,109,108,104,99,96,96,94,93,91,91,
    91,91,90,90,89,88,88,81,77,74,74,72,70,69,67,67,66,65,65,64,63,
    59,58,57,56,56,56,55,53,53,51,50,48,47,47,46,46,44,44,43,43,40,
    40,39,38,38,37,37,36,36,35
  };
  const int n4w1b3r8[] = {
    1000, // Capacity
    500, // Number of items
    // Size of items (sorted)
    626,625,624,622,620,620,620,619,613,611,610,609,608,606,606,604,
    601,601,601,600,598,598,597,591,587,586,586,586,584,584,584,584,
    583,583,582,582,581,581,581,579,579,579,578,578,578,576,573,570,
    569,567,567,565,564,562,559,559,558,557,555,553,553,550,550,547,
    545,544,543,542,541,541,540,540,539,539,537,536,535,533,532,531,
    529,528,527,527,525,524,524,523,521,520,520,518,518,518,517,517,
    516,516,515,514,514,512,507,506,505,505,504,503,502,502,502,501,
    500,499,499,497,497,496,495,495,495,494,493,491,491,487,485,484,
    483,482,480,479,478,475,475,475,472,471,471,469,468,467,466,465,
    465,463,463,462,462,462,462,461,461,461,460,458,457,457,456,454,
    454,452,451,447,443,443,442,439,439,439,438,437,435,434,433,431,
    431,428,428,428,427,427,425,425,423,421,420,419,417,416,415,412,
    411,411,406,405,404,401,401,400,397,397,396,395,394,394,394,393,
    393,390,390,388,388,386,385,383,381,378,378,377,377,376,375,375,
    373,372,370,369,369,367,366,365,365,364,364,363,360,359,359,358,
    354,353,353,353,352,350,349,348,345,345,345,344,342,342,341,340,
    335,333,333,332,331,331,329,328,327,326,326,325,325,322,322,321,
    321,321,320,318,317,317,317,317,317,317,316,315,314,313,313,312,
    310,308,307,307,306,306,306,302,298,296,296,295,295,295,293,293,
    291,289,288,287,287,286,285,285,282,281,280,275,274,274,270,269,
    269,268,268,266,265,265,263,263,263,263,262,261,258,257,257,257,
    255,253,252,250,250,246,243,243,240,240,237,237,236,234,234,233,
    231,230,228,227,226,226,225,225,223,221,220,220,218,217,217,216,
    214,212,212,211,206,206,203,203,202,202,201,201,201,201,200,194,
    194,194,192,191,190,186,186,183,183,174,171,167,167,167,166,163,
    163,162,159,158,157,156,156,151,150,148,145,145,143,142,141,137,
    136,132,132,131,131,129,129,128,126,126,125,125,122,121,120,119,
    114,113,112,111,109,109,109,109,106,105,105,102,102,100,95,95,
    91,91,88,88,87,84,84,82,81,80,78,76,75,75,73,73,73,72,69,69,68,
    67,65,65,64,64,62,61,59,57,57,53,51,51,49,49,49,49,48,47,46,45,
    44,43,42,42,41,39,39,38,37,35
  };
  const int n4w1b3r9[] = {
    1000, // Capacity
    500, // Number of items
    // Size of items (sorted)
    627,627,625,625,621,614,612,608,608,608,607,607,606,605,603,602,
    601,601,601,599,599,598,598,597,592,591,590,589,589,586,586,583,
    582,581,581,580,579,578,577,577,576,573,573,572,569,567,566,564,
    563,563,563,563,562,561,560,557,556,555,555,552,549,548,545,545,
    541,541,541,537,536,535,535,533,533,531,527,526,526,523,522,522,
    521,520,518,518,516,515,515,515,513,513,510,508,508,508,507,505,
    505,504,502,500,500,499,498,495,494,491,490,489,486,484,484,480,
    479,478,477,475,474,473,472,468,464,463,462,462,461,460,459,458,
    458,458,456,456,451,451,451,451,450,448,447,446,444,442,442,442,
    440,439,439,438,438,437,437,437,436,435,433,429,429,428,425,424,
    424,423,423,421,421,417,415,413,411,411,409,408,407,404,404,403,
    403,402,402,401,397,397,396,395,394,393,393,390,390,388,387,385,
    384,384,382,382,382,379,377,377,377,375,375,374,374,374,374,372,
    364,364,364,363,363,362,361,361,360,359,358,358,358,357,356,355,
    354,349,349,348,347,346,345,344,344,341,341,341,340,338,336,334,
    334,333,333,332,331,331,329,328,323,321,320,318,317,316,315,315,
    315,311,311,310,307,307,306,305,302,301,299,298,298,297,296,296,
    295,293,292,290,287,285,285,284,283,283,282,280,280,280,279,279,
    278,277,272,272,271,270,269,269,267,266,263,262,260,260,254,254,
    252,250,250,250,249,247,245,244,243,243,242,242,240,239,239,239,
    239,238,234,231,230,230,229,228,228,225,225,225,224,224,223,222,
    220,219,217,214,213,213,211,211,206,205,205,203,203,202,202,201,
    200,198,198,197,196,195,194,192,192,190,190,190,190,190,189,186,
    186,186,184,183,182,182,181,179,178,178,178,177,176,175,175,175,
    167,166,165,162,160,160,160,159,159,158,157,156,155,153,153,152,
    150,150,149,149,147,147,147,144,144,143,143,141,139,133,132,130,
    127,127,126,126,125,125,123,122,121,120,119,117,117,115,115,112,
    111,110,110,108,108,106,106,106,106,104,102,101,100,99,99,98,
    98,96,93,93,93,92,88,86,84,83,82,82,80,79,79,78,78,76,75,73,73,
    71,71,70,70,68,66,61,61,60,58,56,56,56,55,54,51,47,47,47,47,46,
    45,44,44,44,43,40,40,39,37,37
  };
  const int n4w2b1r0[] = {
    1000, // Capacity
    500, // Number of items
    // Size of items (sorted)
    240,240,240,240,240,240,240,239,239,239,239,239,239,238,237,237,
    237,237,237,237,237,237,237,237,237,236,236,236,236,236,236,236,
    236,235,235,235,235,235,234,234,234,234,234,234,234,233,233,233,
    233,232,232,232,232,231,231,231,231,231,231,231,230,230,230,230,
    230,230,229,229,229,229,229,229,228,228,228,228,228,228,228,227,
    227,227,227,227,227,226,226,226,226,226,226,226,226,226,225,225,
    225,225,225,225,225,225,225,224,224,224,224,224,224,223,223,223,
    223,223,223,223,223,223,222,221,221,221,221,220,220,220,220,220,
    220,219,219,219,219,219,219,218,218,218,218,218,218,218,218,218,
    217,217,217,217,217,217,217,217,217,217,216,216,216,216,216,216,
    215,215,215,215,215,215,215,214,214,214,214,214,214,214,214,213,
    213,213,212,212,212,212,212,212,212,211,211,211,211,211,211,211,
    210,210,210,210,210,210,210,210,209,209,209,209,209,208,208,208,
    208,208,208,208,208,207,207,207,207,207,207,207,207,206,206,206,
    206,206,206,206,205,205,205,205,205,205,205,205,205,204,204,204,
    204,203,203,203,203,203,203,203,202,201,201,201,201,201,201,200,
    200,200,200,200,200,200,200,200,200,199,199,199,199,199,198,198,
    198,198,198,197,197,197,197,197,197,197,197,196,196,196,195,195,
    195,195,195,195,195,195,195,195,195,195,195,194,194,194,193,193,
    193,193,193,192,192,192,192,192,192,192,192,192,192,191,191,191,
    191,191,191,191,191,191,191,190,190,190,190,190,190,190,190,189,
    189,189,189,189,189,189,189,188,188,188,188,188,188,187,187,187,
    187,187,186,186,186,186,186,186,185,185,185,185,184,184,184,183,
    183,183,182,182,182,182,182,182,181,181,181,181,181,181,181,181,
    181,180,180,180,180,180,180,180,179,179,179,179,179,178,178,178,
    178,178,178,177,177,176,176,176,176,176,176,176,175,175,175,175,
    175,175,174,174,174,174,174,174,174,174,173,173,173,172,172,172,
    172,172,172,172,172,171,171,170,170,170,170,170,170,170,170,169,
    169,169,169,169,169,169,169,168,168,168,168,168,168,168,168,168,
    167,167,167,167,167,166,166,166,166,166,166,166,166,165,165,165,
    165,165,165,165,165,164,164,164,163,163,163,163,162,162,162,162,
    162,162,162,162
  };
  const int n4w2b1r1[] = {
    1000, // Capacity
    500, // Number of items
    // Size of items (sorted)
    240,240,240,240,240,240,239,239,239,239,239,239,239,239,239,238,
    238,238,238,238,237,237,237,237,237,236,236,236,236,236,236,236,
    236,235,235,235,235,235,235,234,234,234,234,233,233,233,233,233,
    232,232,232,232,231,231,231,231,231,231,230,230,230,230,230,230,
    230,230,229,229,229,229,228,228,228,228,228,228,228,227,227,227,
    227,227,227,227,227,226,226,226,226,225,225,225,225,225,225,225,
    225,225,225,225,224,224,224,224,224,223,223,223,223,223,223,223,
    223,222,222,222,222,221,221,221,221,220,220,220,220,220,219,219,
    219,219,219,219,219,218,218,218,218,218,218,218,217,217,217,216,
    216,216,216,215,215,215,215,214,214,214,214,214,214,214,214,214,
    214,213,213,213,213,213,213,213,213,213,212,212,212,212,212,212,
    211,211,211,211,211,211,211,210,210,210,209,209,209,209,209,209,
    209,209,208,208,208,208,208,208,208,208,208,207,207,207,207,206,
    206,206,206,206,206,206,206,205,205,205,205,205,205,205,204,204,
    204,204,204,204,204,204,204,204,203,203,203,203,203,202,202,202,
    202,202,202,201,201,201,201,201,201,200,200,200,200,200,200,200,
    200,200,200,199,199,199,199,199,199,198,198,198,198,198,198,198,
    197,197,197,197,197,197,197,197,197,196,196,196,196,196,196,196,
    195,195,195,195,195,195,195,195,195,194,194,194,194,194,194,193,
    193,193,193,193,192,192,192,192,192,192,192,191,191,191,191,191,
    191,191,191,191,190,190,190,190,190,190,190,190,190,190,189,189,
    189,189,189,189,189,189,188,188,188,188,188,187,187,187,187,187,
    187,186,186,186,186,186,185,185,185,185,185,184,184,184,184,184,
    184,184,183,183,183,183,183,182,182,182,182,182,182,181,181,181,
    181,181,181,181,181,181,180,180,180,180,180,180,179,179,179,179,
    179,178,178,178,178,178,178,178,178,178,177,177,177,177,176,176,
    176,176,176,176,175,175,175,175,175,175,175,175,174,174,174,174,
    174,174,174,173,173,173,173,173,172,172,172,172,172,172,171,171,
    171,171,171,171,170,170,170,169,169,169,169,169,169,168,168,168,
    168,168,168,167,167,167,167,167,166,166,166,166,166,166,166,165,
    165,165,165,165,164,164,164,163,163,163,163,163,163,162,162,162,
    162,162,162,162
  };
  const int n4w2b1r2[] = {
    1000, // Capacity
    500, // Number of items
    // Size of items (sorted)
    240,240,240,240,240,240,239,239,239,239,239,239,239,239,239,238,
    238,238,238,238,238,237,237,237,237,237,237,236,236,236,236,236,
    236,236,236,236,235,235,234,234,234,234,234,234,234,234,233,233,
    233,233,232,232,232,232,232,232,232,231,231,231,231,231,231,231,
    230,230,230,230,230,230,229,229,229,229,228,228,228,228,228,228,
    228,227,227,227,226,226,226,226,225,225,225,225,225,225,225,225,
    225,225,224,224,224,224,223,223,223,223,223,223,223,222,222,222,
    222,222,222,222,221,221,221,220,220,220,220,219,219,219,219,219,
    219,219,219,218,218,218,218,218,218,217,217,217,217,217,217,216,
    216,216,216,215,215,215,215,215,215,215,214,214,214,214,214,214,
    214,214,214,214,213,213,213,213,212,212,212,212,212,211,211,211,
    211,210,210,210,210,210,210,210,210,210,210,209,209,209,209,209,
    209,209,209,209,208,208,208,208,208,208,207,207,207,207,207,207,
    207,207,206,206,206,206,206,205,205,205,205,204,204,204,204,204,
    204,204,204,204,204,204,204,204,204,203,203,203,203,203,203,203,
    203,203,203,202,202,202,202,201,201,201,201,201,201,201,201,200,
    200,200,199,199,199,199,198,198,198,198,198,198,198,198,198,198,
    198,198,197,197,197,197,197,197,197,196,196,196,196,196,196,196,
    196,196,196,195,195,195,195,194,194,194,194,194,194,194,194,193,
    193,192,192,192,191,191,191,191,191,191,191,191,190,190,190,190,
    190,189,189,189,189,189,189,189,189,188,188,188,188,187,187,187,
    187,187,187,187,187,187,187,187,186,186,186,186,186,185,185,185,
    185,185,185,185,185,184,184,184,184,184,184,183,183,183,183,183,
    182,182,182,182,182,182,182,182,182,182,182,182,181,181,181,181,
    181,181,180,180,180,180,180,179,179,179,179,179,178,178,178,178,
    178,177,177,177,177,176,176,176,176,175,175,175,174,174,174,174,
    174,174,174,174,174,174,173,173,173,173,173,173,173,173,173,172,
    172,172,172,172,171,171,171,171,171,171,171,171,171,171,171,170,
    170,170,170,170,170,170,169,169,169,169,169,169,169,169,169,169,
    168,168,168,168,168,167,167,167,167,167,166,166,166,166,165,165,
    165,164,164,164,164,164,164,164,164,163,163,163,163,162,162,162,
    162,162,162,162
  };
  const int n4w2b1r3[] = {
    1000, // Capacity
    500, // Number of items
    // Size of items (sorted)
    240,240,240,240,240,239,239,239,239,239,239,239,239,239,239,238,
    238,237,237,237,237,237,237,236,236,236,236,236,236,235,235,235,
    235,235,235,235,234,234,234,234,233,233,233,233,233,233,233,232,
    232,232,232,232,232,231,231,231,231,231,231,230,230,230,230,230,
    230,229,229,229,229,229,229,229,228,228,228,228,228,228,227,227,
    227,226,226,226,226,226,225,225,225,225,224,224,224,223,223,223,
    223,223,223,223,223,223,222,222,222,222,222,222,222,222,221,221,
    221,221,221,221,221,221,221,220,220,220,220,220,220,220,220,219,
    219,219,219,219,219,219,218,218,218,218,218,218,218,217,217,217,
    217,217,217,217,217,217,217,217,216,216,216,216,216,216,215,215,
    215,215,215,215,214,214,214,214,214,214,214,214,214,213,213,213,
    212,212,212,212,211,211,211,211,211,210,210,210,210,210,210,210,
    210,209,209,209,209,209,208,208,208,208,208,208,208,208,208,207,
    207,207,207,207,207,206,206,206,205,205,205,205,205,204,204,204,
    204,203,203,203,203,203,203,203,203,203,202,202,202,202,202,201,
    201,201,201,201,200,200,200,200,200,200,200,199,199,199,199,199,
    199,198,198,198,198,198,198,198,198,198,198,197,197,197,197,197,
    197,196,196,195,195,195,195,194,194,194,194,194,194,194,193,193,
    193,193,193,193,193,193,193,193,192,192,192,192,191,191,191,190,
    190,190,190,190,190,190,190,189,189,189,189,189,189,189,188,188,
    188,187,187,187,187,187,186,186,186,186,186,186,186,185,185,185,
    185,185,185,185,184,184,184,184,184,184,184,184,184,184,184,183,
    183,183,183,183,183,183,182,182,182,182,182,181,181,181,180,180,
    180,180,180,180,180,180,180,179,179,179,179,179,179,178,178,178,
    178,178,178,178,178,177,177,177,177,177,177,177,177,176,176,176,
    176,176,176,175,175,175,175,175,175,175,175,174,174,174,174,174,
    173,173,173,173,173,173,173,172,172,172,172,172,172,172,172,172,
    172,172,172,172,172,171,171,171,171,171,171,171,170,170,169,169,
    169,168,168,168,168,168,167,167,167,167,167,167,167,167,167,167,
    166,166,166,166,166,166,166,166,165,165,165,165,165,165,165,165,
    165,164,164,164,164,164,164,163,163,163,163,163,163,163,163,162,
    162,162,162,162
  };
  const int n4w2b1r4[] = {
    1000, // Capacity
    500, // Number of items
    // Size of items (sorted)
    240,240,240,240,240,239,239,239,239,238,238,237,237,237,237,237,
    236,236,236,236,236,236,236,236,236,236,236,235,235,235,235,235,
    235,234,234,234,234,234,234,233,233,233,233,233,233,232,232,232,
    232,231,231,231,231,231,231,231,230,230,230,230,230,230,230,230,
    230,230,230,229,229,229,229,228,228,227,227,227,227,227,227,227,
    227,226,226,226,226,225,225,225,225,224,224,224,224,224,224,224,
    223,223,223,223,222,222,222,221,221,221,221,221,221,221,220,220,
    220,220,220,219,219,219,219,219,219,218,218,218,218,218,218,218,
    218,218,217,217,217,217,217,217,216,216,216,216,216,216,216,215,
    215,215,215,215,215,214,214,214,214,214,213,213,213,213,213,213,
    213,213,213,213,213,213,212,212,212,212,212,212,212,212,212,211,
    211,211,211,211,210,210,210,210,210,209,209,209,209,209,209,208,
    208,208,208,208,208,208,208,207,207,207,206,206,206,206,206,206,
    206,206,206,206,206,205,205,205,205,205,205,205,204,204,204,204,
    204,204,204,203,203,203,203,203,203,203,203,202,202,202,202,201,
    201,201,201,201,201,200,200,200,200,200,200,200,200,200,200,200,
    199,199,199,199,198,198,198,198,198,198,198,198,198,198,197,197,
    197,197,197,197,197,196,196,196,196,196,196,196,196,196,195,195,
    195,195,195,195,195,195,195,195,195,195,194,194,194,193,193,193,
    192,192,192,192,192,192,192,192,192,192,191,191,191,191,191,191,
    191,191,191,190,190,190,190,190,190,189,189,189,189,188,188,188,
    188,188,188,188,188,188,187,187,187,187,187,187,186,186,186,186,
    186,186,185,185,185,185,185,184,184,183,183,183,183,183,182,182,
    182,182,182,182,182,182,182,182,182,181,181,181,181,181,181,181,
    181,181,180,180,180,180,180,179,179,179,179,179,178,178,178,178,
    177,177,177,177,176,176,176,176,176,176,176,176,176,175,175,175,
    175,175,174,174,174,174,174,173,173,173,173,173,172,172,172,172,
    172,171,171,171,171,171,171,171,171,171,170,170,170,170,170,170,
    170,170,169,169,169,169,169,168,168,168,167,167,167,167,167,167,
    167,167,167,167,167,167,167,167,167,167,167,166,166,166,166,166,
    165,165,165,165,165,164,164,164,164,163,163,163,163,162,162,162,
    162,162,162,162
  };
  const int n4w2b1r5[] = {
    1000, // Capacity
    500, // Number of items
    // Size of items (sorted)
    240,240,240,240,240,240,240,240,240,239,239,239,239,239,239,238,
    238,238,238,238,238,238,237,237,237,237,237,237,237,237,237,237,
    237,236,236,236,236,236,236,236,236,236,236,236,236,236,236,235,
    235,235,235,235,235,234,234,234,234,233,233,233,233,233,233,233,
    232,232,232,232,232,232,231,231,231,231,231,231,231,231,231,231,
    231,231,230,230,230,230,230,230,229,229,229,229,229,229,229,229,
    228,228,228,228,228,228,228,228,228,227,227,227,227,227,227,227,
    227,227,227,227,227,226,226,226,226,225,225,225,225,225,225,225,
    225,224,224,224,224,224,224,223,223,223,223,223,223,223,223,222,
    222,222,222,222,222,222,222,221,221,221,221,220,220,220,220,220,
    219,219,219,219,219,219,219,219,218,218,218,218,218,218,218,218,
    218,217,217,217,217,217,217,217,217,217,217,216,216,216,216,216,
    216,215,215,215,215,215,215,215,214,214,214,214,214,214,214,214,
    213,213,213,213,213,212,212,212,212,212,211,211,211,211,211,210,
    210,210,210,210,210,209,209,209,209,208,208,208,208,208,208,208,
    208,208,207,207,207,207,207,206,206,206,206,205,205,204,204,203,
    203,203,202,202,202,201,201,201,201,201,200,200,200,200,200,199,
    199,199,199,199,198,198,198,198,198,198,198,197,197,197,197,197,
    197,197,196,196,196,196,196,196,196,195,195,195,195,195,195,195,
    194,194,194,194,194,194,194,194,194,193,193,193,193,193,192,192,
    192,192,192,192,191,191,191,191,191,191,190,190,190,190,190,189,
    189,189,189,189,189,189,189,189,188,188,188,187,187,187,187,186,
    186,186,186,185,185,185,185,185,185,185,185,185,185,185,185,185,
    185,184,184,184,184,184,184,184,184,184,184,183,183,183,183,183,
    182,182,181,181,181,181,181,181,181,181,180,180,180,180,179,179,
    179,179,179,179,179,179,179,179,178,178,178,178,177,177,177,177,
    177,177,177,177,176,176,176,176,175,175,175,175,175,175,174,174,
    174,174,174,173,173,173,173,173,173,172,172,172,172,172,171,171,
    171,171,170,170,170,169,169,168,168,168,168,168,168,168,168,168,
    168,168,167,167,167,167,167,167,167,166,166,166,166,165,165,165,
    165,165,165,164,164,164,164,164,164,164,163,163,163,163,162,162,
    162,162,162,162
  };
  const int n4w2b1r6[] = {
    1000, // Capacity
    500, // Number of items
    // Size of items (sorted)
    240,240,240,240,240,240,239,239,239,239,239,239,239,239,238,238,
    238,238,238,238,237,237,237,237,237,237,236,236,236,236,236,236,
    236,236,235,235,235,235,235,234,234,234,234,234,234,234,234,234,
    234,233,233,233,233,233,233,233,233,232,232,232,232,231,231,231,
    231,230,230,230,230,230,230,230,230,230,230,229,229,229,229,229,
    229,229,228,228,228,228,228,227,227,227,227,227,227,227,226,226,
    226,226,226,226,225,225,225,225,224,224,224,224,224,223,223,223,
    223,223,223,223,223,223,223,223,222,222,222,222,222,222,222,222,
    221,221,221,221,220,220,220,220,220,220,219,219,219,219,219,219,
    219,219,218,218,218,218,218,218,217,217,217,216,216,216,216,216,
    216,216,216,216,216,216,215,215,215,214,214,214,214,214,214,214,
    214,213,213,213,213,213,213,213,213,213,213,212,212,211,211,211,
    211,210,210,210,210,210,210,210,210,210,210,210,209,209,209,208,
    208,208,208,208,208,208,208,208,207,207,207,207,207,207,207,207,
    207,207,206,206,206,206,206,206,206,206,206,206,206,205,205,205,
    205,204,204,204,204,203,203,203,203,203,203,203,202,202,202,202,
    202,201,201,201,201,201,201,201,200,200,200,200,200,200,200,200,
    200,200,200,199,199,198,198,198,198,198,197,197,197,197,197,196,
    196,196,196,196,195,195,195,194,194,194,194,194,194,193,193,193,
    193,193,192,192,192,191,191,191,191,191,191,191,191,191,191,191,
    191,190,190,190,190,190,190,189,189,189,189,188,188,188,188,188,
    188,188,188,188,188,188,188,188,188,188,187,187,187,187,187,187,
    187,186,186,186,186,186,186,186,185,185,185,185,185,184,184,184,
    184,184,184,184,183,183,183,183,183,183,182,182,182,182,182,182,
    181,181,180,180,180,180,179,179,179,179,179,179,179,178,178,178,
    178,178,178,178,177,176,176,176,175,175,175,175,175,175,175,175,
    175,174,174,174,174,174,173,173,173,173,173,172,172,172,172,171,
    171,171,171,171,171,171,170,170,170,170,170,170,169,169,169,169,
    169,169,169,169,169,169,168,168,168,168,168,168,168,168,168,168,
    168,167,167,167,167,167,167,167,166,166,166,166,166,166,166,165,
    165,165,165,165,164,164,164,164,163,163,163,163,163,163,163,162,
    162,162,162,162
  };
  const int n4w2b1r7[] = {
    1000, // Capacity
    500, // Number of items
    // Size of items (sorted)
    240,240,240,240,240,240,240,240,240,240,240,240,239,239,239,239,
    239,239,238,238,238,238,238,238,237,237,237,237,237,237,237,237,
    237,236,236,236,236,236,236,236,236,236,235,235,235,235,235,235,
    235,235,234,234,234,234,233,233,233,233,233,232,232,232,232,232,
    231,231,231,231,230,230,230,230,230,230,229,229,229,228,228,228,
    228,227,227,227,227,227,227,227,227,227,227,226,226,226,225,225,
    225,225,224,224,224,224,224,224,223,223,223,223,223,223,223,222,
    222,222,222,222,222,221,221,220,220,220,220,220,220,220,219,219,
    219,219,218,218,218,218,218,218,217,217,217,217,217,217,217,216,
    216,216,216,216,216,216,216,215,215,214,214,214,214,214,214,214,
    213,213,213,213,212,212,212,212,211,211,211,211,210,210,210,210,
    209,209,209,209,209,209,208,208,208,208,207,207,207,207,207,207,
    207,207,207,207,207,206,206,206,206,206,206,205,205,205,205,205,
    205,205,204,204,204,203,203,203,203,203,203,203,203,203,202,202,
    202,202,202,202,202,202,202,202,202,202,201,201,200,200,200,200,
    200,200,199,199,199,198,198,198,198,198,198,198,198,198,197,197,
    197,197,197,197,196,196,196,196,196,195,195,195,195,195,195,195,
    195,195,195,195,194,194,194,194,194,194,194,194,194,194,194,193,
    193,193,193,193,193,193,192,192,192,192,192,191,191,191,191,191,
    191,191,191,191,190,190,190,190,190,190,189,189,189,189,188,188,
    188,188,188,188,188,188,188,188,188,188,187,187,187,187,187,187,
    186,186,186,186,186,186,186,186,185,185,185,185,185,185,185,185,
    185,185,185,184,184,184,184,184,183,183,183,183,183,183,183,183,
    183,183,183,182,182,182,182,181,181,181,181,181,181,181,181,181,
    180,180,180,180,180,180,180,180,180,180,179,179,179,179,179,178,
    178,178,178,178,177,177,177,177,177,176,176,176,176,176,176,176,
    175,175,175,175,175,174,174,174,173,173,173,173,173,173,173,173,
    173,172,172,172,172,172,172,172,172,171,171,171,171,171,171,170,
    170,170,170,170,170,170,170,169,169,169,169,169,168,168,168,168,
    168,167,167,167,167,167,166,166,166,166,166,166,165,165,165,165,
    165,165,165,164,164,164,164,164,164,164,163,163,163,163,163,162,
    162,162,162,162
  };
  const int n4w2b1r8[] = {
    1000, // Capacity
    500, // Number of items
    // Size of items (sorted)
    240,240,240,240,240,240,239,239,239,239,239,239,239,239,238,238,
    238,238,238,237,237,237,237,237,237,237,237,236,236,236,236,236,
    236,236,235,235,235,235,235,235,235,234,234,233,233,233,233,232,
    232,232,232,232,232,232,231,231,231,230,230,230,230,230,230,230,
    230,230,229,229,229,229,229,228,228,227,227,227,227,227,227,227,
    227,227,226,226,226,226,226,225,225,225,225,225,224,224,224,224,
    223,223,223,223,222,222,222,222,222,222,222,221,221,221,221,221,
    221,221,221,221,221,221,221,220,220,220,220,220,220,220,220,219,
    219,219,219,219,219,219,219,219,219,218,218,218,218,218,218,218,
    218,218,217,217,217,216,216,216,215,215,215,215,215,215,214,214,
    214,214,214,214,214,213,213,213,213,213,213,213,213,213,212,212,
    212,212,212,211,211,211,211,211,211,211,211,211,210,210,210,210,
    210,210,210,209,209,208,208,208,208,208,208,207,207,207,207,207,
    206,206,206,206,206,206,206,206,205,205,205,204,204,204,204,204,
    204,204,203,203,203,203,203,203,203,203,203,203,202,202,202,202,
    202,202,202,202,202,202,202,202,201,201,201,201,201,201,201,201,
    201,201,200,200,200,200,200,200,199,199,198,198,198,198,198,198,
    197,197,196,196,196,196,196,195,195,195,195,195,195,194,194,194,
    194,194,193,193,193,193,193,193,193,193,192,192,192,192,192,192,
    191,191,191,191,190,190,190,190,190,190,190,190,190,190,190,189,
    189,189,189,189,189,189,188,188,188,188,188,188,188,188,188,187,
    187,187,187,187,187,187,187,187,186,186,186,186,185,185,185,185,
    185,185,185,185,185,185,185,184,184,184,184,184,184,183,183,183,
    183,183,183,183,182,182,182,182,182,182,182,182,182,182,182,182,
    181,181,181,181,181,181,181,181,181,180,180,180,180,180,179,179,
    179,179,179,179,179,178,178,178,178,178,178,178,178,178,178,177,
    177,177,177,177,177,177,176,176,176,176,176,176,175,175,175,175,
    175,174,174,174,174,174,173,173,173,172,172,172,172,171,171,171,
    171,171,170,170,170,170,169,169,169,169,168,168,168,168,168,168,
    167,167,166,166,166,166,166,166,166,166,166,165,165,165,165,165,
    165,165,164,164,164,164,164,164,164,164,163,163,163,163,162,162,
    162,162,162,162
  };
  const int n4w2b1r9[] = {
    1000, // Capacity
    500, // Number of items
    // Size of items (sorted)
    240,240,240,240,240,240,240,239,239,239,239,239,239,239,239,238,
    238,238,238,237,237,237,237,237,237,237,237,236,236,236,236,235,
    235,235,235,234,234,234,234,234,234,234,234,233,233,233,233,233,
    232,232,232,232,232,232,232,232,232,231,231,231,231,231,230,230,
    230,230,230,230,230,229,229,229,229,229,229,228,228,228,228,228,
    228,227,227,227,227,226,226,226,226,226,226,226,225,225,225,224,
    224,224,224,224,224,224,224,224,223,223,223,223,223,223,223,222,
    222,222,222,221,221,221,221,221,221,221,221,221,220,220,220,220,
    220,220,220,220,219,219,219,219,219,219,219,219,218,218,218,218,
    218,217,217,217,217,216,216,216,216,216,216,216,216,216,216,215,
    215,215,215,215,215,215,215,215,215,215,215,214,214,214,214,214,
    213,213,213,213,213,213,212,212,212,212,212,212,211,211,211,211,
    211,210,210,210,210,210,210,210,210,210,210,210,209,209,209,209,
    209,209,209,209,209,209,209,208,208,208,208,208,207,207,207,207,
    207,206,206,206,206,206,206,206,205,205,205,205,205,205,205,205,
    204,204,204,204,203,203,203,203,202,202,202,202,201,201,201,201,
    201,201,201,201,200,200,200,200,200,200,200,199,199,199,199,199,
    199,198,198,198,198,197,197,197,197,197,197,197,196,196,196,196,
    196,196,196,195,195,195,194,194,194,194,194,193,193,193,193,193,
    192,192,192,192,192,192,192,191,191,191,191,190,190,190,190,190,
    190,189,189,189,189,189,188,188,188,188,187,187,187,186,186,186,
    186,186,186,186,186,185,185,185,185,185,185,185,185,184,184,184,
    184,184,184,183,183,183,183,183,183,182,182,182,182,182,181,181,
    181,181,180,180,180,180,180,179,179,179,179,179,179,179,178,178,
    178,178,178,178,178,177,177,177,177,177,176,176,176,176,176,175,
    175,175,175,175,175,175,175,174,174,174,173,173,173,173,173,173,
    172,172,172,172,172,172,172,171,171,171,171,171,170,170,170,170,
    170,170,169,169,169,169,169,169,169,168,168,168,168,168,168,168,
    167,167,167,167,167,167,167,167,167,166,166,166,166,166,166,166,
    166,166,166,165,165,165,165,165,165,165,165,165,165,164,164,164,
    164,164,164,164,163,163,163,163,163,163,163,163,163,163,162,162,
    162,162,162,162
  };
  const int n4w2b2r0[] = {
    1000, // Capacity
    500, // Number of items
    // Size of items (sorted)
    300,299,299,299,298,298,297,297,296,295,295,295,295,295,295,294,
    294,293,293,292,292,292,292,291,291,290,290,290,289,289,289,288,
    288,288,288,287,287,287,287,285,285,285,284,283,283,283,283,283,
    283,282,282,282,281,281,279,278,277,277,276,276,276,275,275,275,
    275,275,275,275,275,275,274,274,274,273,273,272,272,272,271,271,
    271,271,271,271,270,270,269,269,269,269,268,267,267,266,265,265,
    265,264,264,264,264,264,263,263,263,262,262,261,261,260,260,260,
    260,259,259,258,257,257,256,255,255,255,254,253,252,252,252,252,
    251,251,251,250,249,248,248,248,247,247,246,245,245,245,244,244,
    244,244,243,243,243,243,242,242,242,241,241,241,240,240,239,239,
    239,238,237,237,237,236,235,235,235,234,234,234,234,233,233,232,
    232,231,231,231,230,230,229,229,229,229,228,228,228,227,226,225,
    224,224,224,223,223,223,222,222,222,222,222,221,221,220,219,217,
    217,217,217,217,216,215,215,214,214,213,212,212,212,211,210,209,
    209,208,207,207,207,207,207,207,206,206,206,206,204,204,204,204,
    203,203,199,199,199,199,199,198,198,197,197,197,197,197,197,196,
    196,196,195,195,194,194,194,193,193,193,193,192,192,190,190,189,
    189,189,188,188,187,186,186,186,186,186,185,184,184,184,184,182,
    182,182,182,182,181,181,181,180,179,179,179,178,178,177,177,177,
    177,176,176,176,175,175,175,173,173,172,172,172,171,171,171,170,
    170,170,169,169,169,168,168,168,167,166,166,166,166,166,165,165,
    164,164,163,162,162,161,161,160,160,160,160,159,159,159,158,158,
    158,157,156,156,153,153,153,153,152,152,152,152,151,151,151,151,
    150,150,149,149,149,149,149,149,149,149,148,147,147,146,145,145,
    145,143,143,142,142,142,142,142,141,141,141,141,141,140,140,139,
    139,138,137,137,136,134,134,134,134,133,132,132,132,132,132,132,
    131,131,131,130,130,130,129,128,128,127,127,126,126,125,125,125,
    125,124,124,124,123,123,122,122,122,122,121,121,121,120,119,119,
    118,118,118,118,117,117,117,117,117,116,116,116,116,115,115,114,
    114,113,113,113,113,112,112,112,112,111,110,110,110,110,110,109,
    109,109,108,108,108,107,106,106,106,105,105,104,104,104,103,103,
    103,103,103,102
  };
  const int n4w2b2r1[] = {
    1000, // Capacity
    500, // Number of items
    // Size of items (sorted)
    300,299,299,299,297,297,297,297,297,296,296,296,295,295,294,294,
    294,293,293,293,292,291,290,290,290,289,288,288,288,288,288,288,
    287,287,287,287,286,286,286,286,286,285,285,285,285,285,284,284,
    283,283,283,282,282,281,280,279,279,279,278,278,278,277,277,276,
    276,276,275,274,274,274,274,273,272,272,271,271,271,271,270,270,
    270,270,270,270,269,269,269,268,267,267,266,265,265,264,264,264,
    264,264,264,263,263,263,262,262,262,261,261,261,261,260,260,259,
    258,256,256,255,255,254,254,254,253,253,253,253,253,252,251,250,
    250,250,250,250,249,248,245,244,243,243,243,242,241,241,241,241,
    241,240,240,240,240,240,239,239,239,238,238,237,237,236,236,236,
    235,235,234,233,232,231,230,230,230,229,229,228,228,228,227,227,
    227,227,226,226,225,225,225,225,224,224,223,223,223,222,221,221,
    219,219,219,219,219,218,217,217,217,217,216,216,215,214,214,213,
    213,213,213,213,212,212,212,212,211,211,211,211,210,210,210,210,
    209,209,208,207,207,207,206,205,205,205,205,204,204,203,203,202,
    202,201,201,201,200,199,199,199,198,197,196,196,194,194,194,193,
    193,193,192,192,192,192,192,191,191,191,190,190,189,189,189,188,
    188,187,187,187,187,187,186,186,185,185,184,184,184,183,182,182,
    182,182,182,180,180,180,180,179,179,178,177,177,176,176,175,175,
    175,174,174,173,173,173,173,173,172,171,171,171,170,170,170,170,
    170,170,169,169,168,167,167,167,167,166,166,165,165,165,165,164,
    164,163,163,162,162,162,162,162,161,161,161,160,159,159,159,158,
    158,157,157,157,156,156,156,155,155,155,154,154,153,153,152,151,
    151,150,150,150,150,150,150,150,149,149,149,148,148,148,148,147,
    147,147,147,147,146,146,145,144,144,143,143,143,142,142,142,142,
    140,140,139,139,139,139,139,138,138,138,137,136,136,136,136,136,
    136,136,135,135,135,135,134,134,134,133,133,133,132,132,132,132,
    130,129,129,128,128,128,128,127,127,127,127,126,126,126,125,124,
    124,124,124,119,118,118,117,117,116,116,116,115,115,115,115,114,
    114,114,113,113,113,113,113,113,112,111,111,111,110,110,110,110,
    110,109,109,108,108,108,108,107,106,106,105,105,105,104,104,104,
    103,103,102,102
  };
  const int n4w2b2r2[] = {
    1000, // Capacity
    500, // Number of items
    // Size of items (sorted)
    300,300,300,300,298,298,298,295,295,295,294,294,293,292,292,292,
    292,292,291,291,290,290,290,290,290,290,290,288,288,288,288,287,
    287,287,287,286,286,286,286,286,285,285,285,285,285,285,285,284,
    284,284,284,283,283,283,283,282,281,281,281,281,281,281,280,280,
    280,280,280,280,279,279,279,279,279,278,277,276,276,276,275,275,
    274,274,274,274,274,273,273,273,272,271,271,271,271,270,270,270,
    270,270,269,269,269,268,268,268,267,267,267,267,266,266,266,264,
    263,263,263,263,262,262,261,261,261,260,259,259,257,257,257,257,
    257,257,257,256,255,254,254,254,253,253,252,251,251,250,250,249,
    249,248,247,247,247,246,246,245,244,243,243,242,240,240,240,240,
    239,239,239,238,238,237,236,236,236,235,235,234,234,234,234,233,
    232,232,232,232,232,231,231,231,230,230,230,229,227,227,227,227,
    226,225,225,224,224,223,223,222,221,220,220,220,220,220,220,219,
    219,219,218,217,217,217,217,217,216,216,215,214,214,214,214,213,
    212,212,212,212,212,212,211,211,210,210,210,210,210,210,209,208,
    208,207,207,206,206,205,205,204,204,204,204,204,203,203,203,203,
    203,202,202,202,202,201,201,200,200,199,199,199,198,198,198,197,
    197,195,195,195,195,195,194,194,193,193,193,192,192,192,191,191,
    191,190,190,190,189,189,188,188,188,188,187,187,186,186,185,185,
    185,185,185,184,184,184,183,183,183,182,182,182,181,180,180,180,
    180,179,179,179,178,178,178,177,175,175,174,174,174,173,172,172,
    172,170,170,170,169,168,167,166,166,166,166,165,165,164,164,164,
    164,164,163,163,163,162,162,162,161,161,161,161,161,160,160,160,
    159,159,157,157,157,155,154,154,153,153,153,152,152,152,152,151,
    151,151,151,149,149,148,146,146,146,145,144,144,144,144,143,142,
    142,142,142,141,140,140,139,138,138,138,138,137,137,136,136,136,
    136,135,135,135,134,134,134,133,132,132,132,132,132,131,131,130,
    130,130,130,129,127,126,125,124,124,123,123,123,122,122,122,122,
    121,121,121,121,121,121,117,117,117,116,116,116,115,115,115,114,
    114,114,114,113,113,112,112,112,112,111,111,110,110,109,108,108,
    107,106,106,106,105,105,105,105,105,105,105,104,104,104,103,103,
    102,102,102,102
  };
  const int n4w2b2r3[] = {
    1000, // Capacity
    500, // Number of items
    // Size of items (sorted)
    300,299,299,299,298,298,298,298,298,298,297,297,296,296,295,295,
    295,295,295,295,295,294,294,293,293,292,292,292,292,291,291,290,
    289,288,288,288,287,287,287,287,286,285,285,285,284,284,282,282,
    281,280,280,279,279,278,278,277,277,277,277,277,276,276,276,275,
    274,274,274,274,274,274,274,273,273,272,272,271,271,271,271,271,
    270,270,270,270,269,269,269,268,267,267,266,266,266,263,263,262,
    262,262,261,260,260,260,260,260,259,258,258,258,258,257,257,257,
    257,257,256,256,256,255,255,254,254,254,254,254,254,254,253,253,
    253,252,252,252,251,250,250,249,249,249,248,247,247,247,247,246,
    246,246,245,245,245,245,244,244,243,243,242,242,241,241,241,241,
    241,240,239,239,238,238,238,238,237,236,236,236,236,236,235,235,
    234,234,234,234,233,233,232,231,231,231,231,230,229,229,229,228,
    228,227,227,227,226,225,225,225,225,225,223,223,222,221,220,220,
    220,220,220,220,220,219,218,218,218,218,217,217,217,216,216,215,
    215,214,214,214,213,213,211,211,210,210,210,210,209,209,208,207,
    207,207,207,205,204,204,204,204,203,203,202,201,201,200,200,200,
    199,199,198,198,198,197,197,196,196,196,196,196,195,195,195,195,
    194,193,193,193,193,193,193,193,193,193,193,191,191,191,191,190,
    190,188,188,188,187,186,186,186,185,185,185,185,184,184,184,183,
    183,183,182,182,181,180,180,179,179,179,179,179,178,178,178,178,
    177,176,176,175,175,175,174,174,173,173,173,173,171,170,169,168,
    166,166,165,165,164,164,164,163,163,162,161,161,161,161,160,159,
    158,158,157,157,157,157,156,156,156,155,155,154,153,153,153,153,
    152,152,152,151,151,151,150,150,150,150,149,149,149,148,148,148,
    148,148,147,147,147,146,146,145,145,144,144,144,144,142,142,142,
    142,141,141,141,141,140,140,139,139,139,139,137,137,136,136,135,
    135,135,135,135,135,135,135,134,134,134,132,132,132,132,130,130,
    129,128,127,127,127,126,126,126,126,125,125,125,125,124,124,122,
    122,122,121,121,120,120,120,120,120,119,119,119,118,118,117,116,
    116,115,114,114,113,113,112,111,111,111,111,110,110,109,109,109,
    109,109,109,108,108,108,107,107,107,106,106,105,105,105,105,105,
    104,103,102,102
  };
  const int n4w2b2r4[] = {
    1000, // Capacity
    500, // Number of items
    // Size of items (sorted)
    300,300,299,299,299,298,298,297,296,296,296,296,295,295,293,293,
    293,292,292,292,292,291,291,291,290,290,289,289,289,289,289,288,
    288,287,287,287,287,286,286,286,285,285,285,284,284,283,283,282,
    281,281,280,280,279,279,279,278,278,277,277,277,276,276,276,275,
    274,274,274,274,273,273,273,272,272,271,270,270,269,269,269,269,
    267,267,266,266,265,265,265,264,264,263,263,262,262,262,262,261,
    261,261,260,259,259,259,258,257,255,255,254,254,254,253,253,253,
    252,252,252,251,251,251,249,248,248,248,247,247,246,245,244,244,
    244,244,243,243,243,242,241,239,239,239,238,237,236,236,236,236,
    235,235,233,233,233,233,232,232,232,232,232,230,230,230,230,229,
    229,229,229,229,228,228,228,226,226,226,226,226,226,225,225,224,
    224,224,224,224,224,223,222,222,221,221,221,221,221,221,221,220,
    220,220,220,219,218,218,218,217,217,217,217,216,216,216,215,214,
    214,213,213,213,213,213,213,213,212,211,211,210,210,210,210,210,
    209,209,209,208,208,208,207,207,207,207,206,205,205,205,205,205,
    204,204,204,204,204,204,203,203,203,202,202,202,201,200,200,199,
    199,199,198,198,198,197,197,197,197,196,195,194,193,193,192,192,
    192,191,191,190,190,190,190,190,189,189,188,187,187,187,187,187,
    186,185,184,183,183,182,180,180,179,179,179,178,178,177,177,176,
    176,175,175,175,175,174,174,173,173,173,172,172,171,170,170,170,
    170,169,168,168,168,168,168,167,167,166,166,165,165,165,165,165,
    164,164,164,163,162,162,161,161,161,161,160,160,160,160,160,159,
    157,157,157,157,156,156,156,156,155,155,155,155,154,154,154,153,
    152,151,150,150,149,149,148,148,148,148,147,147,146,146,146,145,
    145,144,144,143,142,142,142,141,141,140,140,139,139,137,137,137,
    137,137,136,136,135,135,135,134,133,133,132,132,132,132,130,130,
    129,129,129,129,128,128,128,128,127,127,125,125,125,125,125,124,
    124,124,123,123,122,122,122,120,120,120,120,120,120,119,119,119,
    118,118,117,117,117,117,117,116,116,115,115,114,114,114,114,114,
    113,113,113,113,113,112,112,112,111,111,110,110,110,109,109,109,
    108,108,108,108,108,107,106,106,106,105,105,105,105,104,104,102,
    102,102,102,102
  };
  const int n4w2b2r5[] = {
    1000, // Capacity
    500, // Number of items
    // Size of items (sorted)
    300,300,300,300,299,298,298,297,296,296,295,295,294,294,293,293,
    291,290,289,289,288,287,287,287,286,286,286,285,284,284,284,284,
    283,283,282,281,281,280,280,280,280,279,279,279,278,278,278,278,
    278,278,276,276,276,276,276,276,276,275,275,275,275,274,274,273,
    272,272,272,271,271,270,270,269,269,269,269,268,268,266,266,266,
    265,265,265,265,265,264,263,263,263,263,263,263,262,262,262,262,
    261,261,261,261,261,260,260,260,259,259,259,258,258,258,258,257,
    257,256,255,255,254,253,253,253,252,252,251,251,251,251,250,250,
    250,249,249,249,248,248,248,247,247,247,247,247,246,246,246,246,
    246,246,245,245,245,245,244,244,244,244,244,244,243,243,243,243,
    243,243,242,242,242,242,240,239,238,237,237,237,237,237,237,237,
    236,236,235,234,234,233,233,232,232,232,231,231,231,231,231,230,
    229,229,229,229,229,228,228,227,227,227,227,227,226,226,224,224,
    223,222,222,222,222,222,221,221,221,220,220,219,219,219,219,219,
    218,218,217,217,217,217,216,216,216,216,216,216,215,215,215,215,
    214,214,214,214,213,212,212,211,210,210,209,209,208,208,208,208,
    208,207,207,207,207,206,206,206,206,205,205,204,204,203,203,202,
    202,202,202,202,201,201,201,200,199,198,198,197,195,192,192,192,
    191,190,190,190,190,189,189,189,189,188,188,187,187,185,185,185,
    185,184,184,183,183,182,182,182,181,181,181,181,180,180,180,180,
    179,179,177,177,176,176,175,175,175,174,174,174,174,174,174,174,
    172,172,172,172,171,169,168,167,167,166,166,166,165,164,164,164,
    164,163,163,163,163,162,162,162,162,161,161,160,159,159,159,158,
    157,155,155,154,154,153,153,153,153,153,152,152,151,151,150,149,
    149,149,148,147,147,147,147,147,146,146,145,145,144,144,144,143,
    142,142,142,141,141,140,140,140,139,139,139,138,138,137,137,137,
    137,136,136,136,136,135,135,134,134,134,134,134,133,133,133,133,
    132,132,130,130,129,128,128,127,127,127,126,126,126,126,126,126,
    124,124,123,123,122,122,122,121,121,121,119,119,119,118,117,117,
    117,116,116,116,114,114,114,114,113,113,112,110,110,110,110,110,
    110,109,109,108,108,108,107,107,106,106,105,104,104,104,104,103,
    103,102,102,102
  };
  const int n4w2b2r6[] = {
    1000, // Capacity
    500, // Number of items
    // Size of items (sorted)
    300,300,300,299,298,298,298,297,297,297,296,295,295,295,295,295,
    294,294,294,294,294,293,293,293,293,292,292,292,291,291,291,291,
    289,289,289,289,288,288,288,288,288,288,287,286,285,285,284,284,
    284,284,284,283,283,283,282,282,282,282,281,281,281,280,279,279,
    279,278,278,278,277,276,275,275,275,275,274,274,273,272,272,272,
    272,271,271,271,270,269,269,269,268,268,268,268,267,267,267,267,
    266,266,265,265,265,264,264,263,263,263,262,262,262,262,260,259,
    259,259,259,259,258,257,256,256,256,256,256,255,253,253,252,252,
    251,251,251,250,250,250,249,249,248,248,248,247,247,247,247,247,
    246,246,246,246,246,246,245,244,243,243,242,242,242,241,241,241,
    241,241,241,241,240,240,240,239,239,239,239,239,238,237,237,237,
    236,235,235,234,233,233,233,232,232,232,231,231,229,229,228,228,
    228,227,227,227,227,227,226,226,226,225,225,225,225,223,223,223,
    223,223,223,222,222,222,221,221,221,220,220,220,220,220,219,219,
    218,218,218,217,217,216,216,216,216,215,215,214,213,212,211,211,
    211,211,211,210,210,209,209,207,206,206,205,204,204,203,203,203,
    203,202,201,201,201,201,201,200,199,199,199,198,197,196,196,196,
    195,194,194,194,193,193,192,192,192,191,191,190,190,189,189,188,
    188,188,188,188,188,188,188,187,186,186,186,185,185,185,185,184,
    184,184,183,183,183,182,182,182,182,182,182,181,181,181,181,180,
    180,180,179,179,179,178,177,177,176,176,176,176,176,175,175,175,
    175,174,174,172,171,171,171,171,171,171,171,168,168,168,168,167,
    167,167,167,166,166,165,164,164,164,163,163,162,162,162,162,162,
    161,161,160,160,159,159,158,157,157,157,157,157,156,156,154,153,
    152,151,151,150,150,150,149,148,148,147,146,146,146,145,145,145,
    145,145,144,144,143,143,143,140,140,139,139,138,138,136,136,135,
    134,133,133,133,133,133,132,132,132,131,131,131,131,131,131,131,
    130,130,129,128,127,127,127,127,127,127,126,126,124,124,123,123,
    123,122,121,121,120,119,119,119,118,118,118,118,118,117,117,117,
    117,116,116,116,115,114,113,113,113,113,112,112,111,111,110,110,
    109,108,108,108,107,107,107,106,106,106,106,105,105,105,105,105,
    105,103,103,102
  };
  const int n4w2b2r7[] = {
    1000, // Capacity
    500, // Number of items
    // Size of items (sorted)
    300,300,300,299,299,298,298,298,297,297,297,297,296,295,295,295,
    294,294,294,293,293,293,293,292,291,291,291,291,291,291,291,290,
    290,289,289,288,288,287,287,287,286,286,286,285,285,285,284,283,
    283,283,283,282,282,282,280,280,279,279,279,279,279,278,277,277,
    276,276,275,275,275,275,274,273,273,273,273,273,273,271,271,271,
    271,271,271,270,270,270,270,270,269,269,269,268,267,267,266,265,
    265,264,264,264,263,262,262,262,261,261,260,260,259,259,259,258,
    258,257,256,255,254,254,254,253,253,252,252,252,251,251,251,250,
    250,250,250,249,249,249,249,248,248,248,248,247,247,247,247,246,
    246,246,245,244,244,244,243,243,243,243,242,241,241,241,241,240,
    238,238,237,237,236,235,235,233,233,232,232,232,232,232,232,232,
    231,230,229,229,229,228,228,228,227,227,227,227,226,226,226,226,
    225,225,224,224,222,222,221,221,220,220,219,217,217,217,217,216,
    216,216,215,215,215,214,214,214,214,214,214,213,213,212,212,212,
    212,212,212,211,211,211,210,210,210,210,210,210,209,209,208,208,
    207,206,206,205,205,205,204,204,204,204,203,203,202,202,202,202,
    202,202,202,202,201,201,201,201,201,199,198,198,198,198,196,196,
    196,195,193,193,193,193,193,193,192,192,192,192,192,191,190,190,
    189,189,189,188,188,188,187,187,186,186,186,186,184,184,183,183,
    182,181,181,180,179,179,178,178,177,177,176,175,175,175,175,174,
    174,174,172,172,171,171,171,171,170,170,170,168,167,167,167,166,
    166,166,166,166,166,165,165,165,165,165,164,164,164,162,161,161,
    159,159,159,158,158,158,158,158,158,157,156,156,155,155,155,154,
    154,154,153,152,151,151,151,151,150,149,148,147,147,146,146,146,
    146,146,145,145,144,143,142,141,141,140,140,140,140,139,139,138,
    137,137,137,137,137,137,137,136,136,135,135,135,134,134,134,134,
    133,133,132,131,131,131,130,130,130,130,129,129,126,126,126,126,
    126,125,125,125,125,124,124,124,123,123,122,121,121,121,121,120,
    120,119,119,119,118,118,118,117,117,117,116,116,115,114,114,113,
    112,112,112,112,111,111,111,110,109,109,109,109,109,108,108,108,
    107,106,106,106,105,105,105,105,105,104,104,104,103,103,102,102,
    102,102,102,102
  };
  const int n4w2b2r8[] = {
    1000, // Capacity
    500, // Number of items
    // Size of items (sorted)
    300,299,298,296,296,295,295,295,295,293,292,292,292,291,291,290,
    290,288,288,288,288,288,288,287,287,286,286,286,285,285,284,284,
    284,283,282,281,281,280,280,280,279,279,279,278,278,278,278,278,
    277,277,276,274,274,274,273,273,273,272,271,271,270,269,269,268,
    267,267,267,267,266,266,265,265,265,265,264,264,264,263,263,262,
    262,261,261,261,260,259,259,259,258,258,257,257,257,257,256,256,
    255,254,254,254,254,254,254,254,253,253,252,251,251,251,251,251,
    250,250,249,249,249,248,248,248,247,247,246,246,246,245,245,244,
    244,244,244,241,241,241,240,240,240,239,239,239,239,239,239,238,
    238,238,238,238,237,236,236,236,236,235,235,235,235,235,233,233,
    232,232,232,230,230,230,229,229,228,227,227,226,226,226,225,224,
    223,223,223,223,222,222,221,221,221,220,220,220,220,220,219,219,
    219,219,218,218,218,217,216,216,216,216,215,215,214,213,213,213,
    212,212,212,211,211,211,211,210,210,209,209,209,209,209,208,208,
    208,208,208,207,207,207,206,206,205,205,204,204,203,202,202,201,
    201,201,201,201,200,199,199,198,196,196,196,195,195,195,195,194,
    194,193,193,193,192,192,191,191,191,190,190,189,188,188,188,188,
    187,186,185,185,185,184,184,184,183,183,183,182,182,182,181,181,
    181,180,180,180,179,178,178,178,178,177,177,177,177,177,177,176,
    176,176,176,176,175,175,175,174,174,173,173,173,172,172,171,171,
    171,169,169,169,168,168,168,168,168,168,167,167,167,166,166,165,
    165,165,165,164,164,164,164,164,163,163,162,162,161,161,161,160,
    160,159,159,159,159,159,159,158,157,157,156,156,156,156,156,155,
    155,155,154,153,153,153,153,152,152,152,152,151,151,151,150,149,
    149,149,149,149,148,148,148,147,147,146,146,146,145,145,145,145,
    145,145,144,144,143,143,143,142,141,141,141,140,140,140,140,139,
    139,139,138,137,137,137,136,135,135,135,135,134,134,134,134,132,
    132,131,131,131,130,128,128,127,127,127,127,126,126,126,125,125,
    124,124,123,122,122,121,121,119,118,118,118,117,117,116,116,116,
    116,115,115,114,113,113,113,113,112,111,111,111,111,111,110,109,
    109,109,108,108,108,108,107,106,106,106,106,106,105,105,104,104,
    104,103,102,102
  };
  const int n4w2b2r9[] = {
    1000, // Capacity
    500, // Number of items
    // Size of items (sorted)
    300,300,299,299,298,298,298,295,295,295,294,294,294,294,293,293,
    293,292,292,292,292,292,290,290,290,288,288,288,287,287,287,287,
    287,286,286,286,285,285,285,284,284,283,283,283,283,283,282,282,
    282,282,281,281,280,280,279,279,279,278,278,277,277,277,276,275,
    275,275,274,274,274,274,273,273,272,272,271,271,271,271,271,270,
    270,270,270,270,269,269,269,269,268,268,268,268,268,268,267,266,
    266,266,266,266,265,265,264,264,264,263,262,262,261,261,261,261,
    260,260,259,259,259,259,258,258,257,256,256,255,255,254,253,253,
    253,252,252,251,251,251,251,250,250,250,250,250,249,249,248,248,
    247,247,247,246,246,246,245,244,244,244,242,241,241,241,241,240,
    239,239,239,238,238,238,238,237,236,236,236,236,236,236,236,235,
    235,235,235,235,234,234,234,234,233,233,233,231,231,231,230,229,
    229,229,228,228,228,227,227,226,226,225,225,224,224,224,223,223,
    222,222,222,221,221,221,220,220,220,220,219,219,219,219,219,218,
    218,217,216,216,216,215,215,215,214,213,213,212,211,211,211,211,
    211,210,210,210,209,208,207,207,206,205,205,205,204,203,203,201,
    201,201,200,200,199,199,199,199,198,197,197,197,197,196,196,196,
    195,194,194,193,193,193,193,192,192,190,189,189,188,188,188,188,
    188,188,187,187,187,185,185,184,183,182,182,182,182,182,182,181,
    181,181,180,180,179,179,179,179,179,178,178,178,176,175,175,175,
    174,173,173,173,173,173,172,172,172,172,172,170,169,169,169,169,
    169,168,168,167,167,166,166,166,166,165,164,164,164,163,162,162,
    159,159,159,157,157,157,157,156,156,156,156,156,156,156,155,154,
    153,152,152,152,152,152,152,152,151,151,150,150,150,149,149,148,
    148,145,145,145,144,144,144,143,143,142,142,142,142,142,142,141,
    141,141,140,140,140,139,139,138,138,137,137,137,137,136,136,135,
    134,134,133,133,133,133,133,132,132,130,130,130,130,129,129,128,
    128,128,128,127,127,127,126,126,125,125,125,125,125,125,124,124,
    123,123,123,122,122,122,121,120,120,120,120,120,120,119,119,119,
    118,117,117,117,116,116,116,116,115,115,115,114,113,113,112,112,
    112,112,110,110,109,109,109,108,108,108,108,107,107,107,105,105,
    105,104,103,103
  };
  const int n4w2b3r0[] = {
    1000, // Capacity
    500, // Number of items
    // Size of items (sorted)
    380,380,380,379,379,379,378,377,377,377,376,376,374,373,373,372,
    370,370,370,370,370,369,369,368,367,366,365,365,365,365,364,363,
    362,361,361,360,360,359,359,358,358,357,357,357,357,356,355,353,
    352,351,350,350,349,348,348,348,348,348,347,345,345,345,341,341,
    339,338,337,337,337,337,336,334,334,332,331,329,329,327,327,325,
    323,323,322,321,320,320,320,319,319,317,314,313,312,312,310,308,
    308,307,306,306,306,306,304,304,304,303,303,303,302,302,300,299,
    295,294,294,294,293,293,293,290,290,287,286,286,286,285,285,283,
    282,281,281,280,279,278,278,277,277,277,274,273,273,272,272,271,
    270,270,269,268,267,266,266,264,264,262,261,261,261,261,261,260,
    260,260,260,258,258,257,257,257,256,256,254,254,254,253,253,252,
    252,252,252,251,251,249,249,248,247,247,246,246,245,245,242,242,
    240,240,240,239,239,237,237,236,236,235,234,234,234,234,233,233,
    233,232,230,230,229,228,227,226,225,225,225,225,224,224,222,221,
    220,219,219,218,217,217,216,216,214,214,214,213,212,212,210,210,
    210,209,209,208,206,206,206,204,203,203,202,202,201,199,199,198,
    198,197,196,195,195,195,195,194,194,194,192,191,191,189,188,188,
    185,185,185,182,182,181,180,180,179,179,179,179,178,178,175,174,
    173,172,172,172,171,171,168,168,168,167,166,166,165,165,165,165,
    164,164,163,163,162,160,159,159,159,158,158,157,154,153,153,151,
    151,149,148,148,147,147,146,146,146,145,144,144,143,141,141,141,
    141,140,140,139,139,139,139,138,138,136,136,136,136,136,135,134,
    134,133,132,131,131,129,127,127,127,126,125,124,124,120,120,119,
    117,117,116,116,115,115,115,114,113,111,111,110,109,109,108,108,
    108,107,106,106,106,105,105,101,99,99,98,96,96,96,95,94,92,91,
    91,90,89,88,88,88,87,86,85,83,83,83,82,82,81,78,77,77,77,75,74,
    73,73,73,73,73,73,72,70,69,65,63,62,62,60,60,59,57,57,57,57,57,
    56,56,54,54,54,53,52,51,50,48,48,47,47,46,46,45,45,44,44,44,44,
    44,43,43,43,42,41,40,40,39,39,39,38,38,38,37,34,33,33,33,32,32,
    31,30,30,29,28,28,28,28,28,25,23,22,22,22
  };
  const int n4w2b3r1[] = {
    1000, // Capacity
    500, // Number of items
    // Size of items (sorted)
    380,379,379,379,378,376,376,376,374,373,373,370,369,368,366,366,
    365,364,362,362,362,361,361,360,359,359,359,358,356,356,355,355,
    355,355,352,352,352,351,351,351,349,349,348,348,348,346,345,344,
    344,344,343,343,343,341,341,340,340,339,338,336,335,335,335,334,
    334,333,333,332,332,331,330,330,330,329,328,327,327,327,327,327,
    326,326,325,324,322,322,321,320,320,319,319,318,315,313,313,313,
    313,313,313,309,307,306,306,303,301,300,299,298,297,296,296,295,
    294,294,294,294,293,293,292,292,292,292,292,291,291,291,290,290,
    289,289,288,288,288,288,286,285,283,282,281,280,278,277,276,275,
    274,273,271,271,270,270,269,269,269,268,268,267,267,266,265,265,
    265,261,260,260,259,259,258,258,258,257,257,257,257,256,254,253,
    252,251,251,251,249,249,249,249,247,247,246,246,246,245,244,243,
    243,242,242,241,241,241,239,239,238,237,236,236,235,235,235,234,
    234,234,232,232,231,230,228,228,228,227,227,226,225,224,223,222,
    222,221,221,221,220,220,217,216,216,216,216,216,215,214,213,213,
    213,210,210,210,210,210,210,209,208,208,207,207,206,205,205,203,
    203,201,200,200,200,199,199,199,198,196,192,189,189,188,188,187,
    186,186,185,184,181,180,180,180,179,179,178,174,174,173,173,172,
    171,170,170,169,168,167,167,166,166,166,164,163,163,163,162,162,
    161,161,160,160,159,159,159,157,156,155,153,153,152,151,150,150,
    150,149,148,148,148,148,146,145,145,144,144,143,142,141,140,138,
    138,138,137,137,136,135,134,133,132,132,132,131,130,130,129,129,
    129,129,129,128,127,127,127,127,127,126,123,123,122,122,122,121,
    121,121,120,120,120,118,118,115,114,114,114,113,113,112,112,112,
    111,111,110,110,109,109,108,107,107,106,106,105,103,102,102,98,
    98,97,97,97,96,91,90,90,89,89,88,87,86,84,84,83,83,81,80,80,80,
    80,79,79,78,78,77,77,77,76,76,76,75,71,71,71,70,69,68,67,65,65,
    65,64,64,63,62,62,62,58,56,55,54,53,52,50,50,50,49,49,48,48,48,
    47,46,46,45,44,43,42,42,41,39,39,39,39,38,38,37,35,35,34,34,33,
    33,32,32,32,31,29,26,26,26,24,24,23,23,22,22,22
  };
  const int n4w2b3r2[] = {
    1000, // Capacity
    500, // Number of items
    // Size of items (sorted)
    380,380,380,379,379,378,377,377,376,376,374,373,372,371,370,368,
    368,368,367,367,367,367,366,365,363,362,361,361,360,360,359,359,
    359,358,358,357,357,356,355,354,354,354,353,353,353,351,351,350,
    348,346,344,343,343,342,341,341,341,341,340,339,339,338,338,338,
    337,335,334,332,331,331,329,329,325,325,324,320,319,318,318,318,
    318,318,316,316,315,312,312,311,308,308,307,306,306,305,304,304,
    304,304,303,302,301,300,300,299,299,298,298,297,297,296,295,294,
    294,292,292,291,291,291,291,291,290,289,289,287,287,286,286,286,
    286,284,284,283,282,282,281,280,279,279,278,278,277,274,272,271,
    271,269,267,267,267,266,265,265,265,265,264,264,262,262,262,261,
    261,260,260,260,259,259,259,258,257,257,257,256,256,255,255,255,
    255,254,254,251,251,250,248,248,248,243,240,240,240,239,239,237,
    235,235,233,233,231,231,230,229,229,228,228,227,225,225,223,223,
    222,221,219,218,218,218,217,217,215,215,213,213,212,211,211,210,
    210,208,207,207,206,206,206,205,205,203,201,200,200,200,199,199,
    198,198,197,197,197,196,196,196,195,195,194,194,193,191,191,191,
    189,188,188,187,187,186,186,186,185,185,185,185,184,183,181,181,
    180,180,179,177,177,176,176,175,175,174,172,172,172,171,171,171,
    171,170,170,169,168,167,167,166,164,163,162,161,159,158,157,157,
    157,155,154,153,152,152,152,151,151,150,150,148,148,147,147,146,
    146,144,144,144,144,143,143,143,142,142,141,141,140,140,139,138,
    137,137,137,136,135,135,135,135,134,133,132,130,130,130,129,129,
    129,127,125,124,124,124,124,123,123,122,122,122,120,120,119,117,
    117,116,115,115,114,112,110,109,109,108,107,105,105,105,105,104,
    103,103,103,102,102,101,101,100,100,100,99,99,98,98,98,97,96,
    96,93,93,93,92,92,92,90,88,88,87,86,85,85,84,84,83,82,80,80,79,
    76,75,75,74,74,73,73,72,71,71,70,70,69,68,68,66,65,65,63,63,62,
    62,62,62,62,60,60,58,58,57,57,56,56,55,53,52,52,51,51,50,49,48,
    47,47,46,46,44,44,44,42,41,41,41,41,40,39,37,36,36,36,36,36,36,
    35,35,33,32,31,30,29,29,28,27,26,26,24,23,23
  };
  const int n4w2b3r3[] = {
    1000, // Capacity
    500, // Number of items
    // Size of items (sorted)
    380,380,378,376,375,375,374,372,371,370,370,370,369,369,368,368,
    365,365,365,364,363,362,361,360,359,359,357,354,354,353,353,352,
    350,349,349,349,349,349,348,347,347,346,345,345,342,341,340,340,
    339,338,337,337,337,335,334,334,334,333,333,332,331,331,329,329,
    329,328,328,327,326,325,325,324,324,323,322,320,320,320,320,319,
    318,317,314,314,314,313,313,312,309,306,306,305,303,303,303,302,
    302,301,301,301,299,299,297,296,296,295,295,294,293,293,293,292,
    292,292,292,291,291,291,289,289,288,288,288,287,286,286,286,286,
    285,284,284,284,283,283,283,282,280,279,278,278,277,277,276,276,
    275,274,271,271,270,270,269,269,269,268,268,268,267,267,267,266,
    265,265,265,263,263,262,262,260,259,258,258,258,258,257,256,256,
    255,255,254,254,254,252,252,252,251,250,250,249,249,247,246,246,
    244,244,242,242,241,241,241,241,241,240,238,237,236,236,232,231,
    230,229,229,229,228,228,228,226,225,224,223,222,221,221,220,219,
    219,219,218,217,215,214,213,212,211,210,210,210,209,209,209,208,
    207,207,207,207,206,206,205,205,204,202,202,202,200,199,199,198,
    196,195,192,192,191,191,191,190,190,189,188,186,186,184,184,184,
    183,183,183,182,182,182,182,180,180,180,179,179,179,178,178,178,
    177,176,176,176,175,175,174,174,174,174,171,170,170,169,167,167,
    166,163,161,160,159,157,156,156,156,156,155,154,154,153,152,151,
    151,151,150,150,150,148,148,146,146,146,145,145,144,144,144,144,
    144,142,142,141,140,138,138,137,136,133,132,132,131,131,131,131,
    130,129,128,126,125,123,123,123,121,121,120,120,120,120,120,120,
    118,117,116,116,114,114,112,112,112,112,108,108,107,107,106,104,
    104,104,103,103,100,98,98,95,94,94,94,93,93,93,92,92,89,89,89,
    88,87,86,86,83,83,81,80,80,79,79,77,77,76,76,76,76,76,75,75,75,
    74,74,74,74,74,73,73,71,71,71,71,70,69,68,68,68,67,67,67,65,62,
    62,62,61,60,60,59,58,58,57,57,56,55,55,55,55,53,53,53,51,50,50,
    50,50,48,48,47,46,46,45,44,43,43,40,38,36,35,33,33,32,32,32,31,
    29,28,27,25,25,25,24,24,24,24,22,22,22
  };
  const int n4w2b3r4[] = {
    1000, // Capacity
    500, // Number of items
    // Size of items (sorted)
    380,380,379,378,378,378,377,376,374,374,372,372,372,371,370,370,
    369,368,368,368,367,366,366,365,362,361,361,360,359,359,358,356,
    356,355,355,355,355,353,353,352,351,351,350,350,349,349,348,348,
    348,348,347,347,346,345,344,344,343,343,343,342,341,341,339,339,
    339,339,336,335,334,331,329,329,329,329,328,328,328,325,325,325,
    325,322,322,321,321,320,320,320,319,318,318,318,317,316,316,315,
    315,315,314,314,313,313,312,312,312,311,310,309,308,307,307,307,
    306,304,301,300,300,299,299,298,298,297,296,295,295,295,295,295,
    295,293,293,293,292,291,289,288,285,284,280,278,277,276,275,274,
    274,273,273,273,273,272,272,269,269,268,268,267,267,264,264,264,
    264,262,260,260,260,258,258,257,257,256,255,254,253,253,253,252,
    252,251,251,250,249,249,248,246,245,244,243,243,243,242,242,241,
    241,241,241,239,238,238,237,237,237,234,234,231,230,229,228,228,
    227,227,226,226,226,226,225,225,224,224,224,224,221,221,219,219,
    219,219,218,218,215,215,214,214,212,212,210,209,208,208,207,205,
    204,203,201,200,198,198,198,198,197,197,197,196,196,195,194,193,
    192,191,188,187,187,186,185,185,185,185,184,184,183,183,183,181,
    181,181,180,180,180,179,179,178,177,177,176,175,173,173,173,173,
    171,171,170,168,168,168,168,162,161,159,158,158,158,157,157,156,
    155,154,154,154,153,152,152,151,151,148,148,148,147,146,144,144,
    144,143,142,140,138,138,138,137,137,136,136,136,135,134,133,133,
    133,132,132,132,131,129,129,128,128,127,126,124,123,123,122,122,
    120,120,120,120,120,118,118,118,117,117,117,117,116,115,115,115,
    114,114,113,110,110,109,108,107,106,106,106,104,103,102,102,101,
    100,97,97,96,96,95,95,91,90,90,89,89,88,88,87,86,86,85,85,84,
    84,84,84,83,83,83,81,81,81,80,79,78,77,77,77,76,73,73,71,71,70,
    70,70,69,68,68,67,66,65,65,62,61,61,61,59,59,59,59,57,57,56,54,
    54,54,54,53,53,53,52,51,50,50,50,49,48,48,48,48,47,45,44,42,41,
    41,41,41,38,38,38,37,34,33,32,31,31,31,31,31,30,30,29,28,28,28,
    27,26,26,26,26,26,25,24,23,23,22,22
  };
  const int n4w2b3r5[] = {
    1000, // Capacity
    500, // Number of items
    // Size of items (sorted)
    380,380,380,380,378,378,378,378,377,377,375,374,374,373,372,372,
    371,370,369,368,367,365,363,363,362,362,361,360,359,359,358,358,
    357,357,357,357,356,355,354,353,352,352,351,351,351,349,349,349,
    348,347,347,347,346,344,344,343,340,339,339,337,336,335,335,335,
    335,335,332,331,331,331,330,330,329,329,327,326,326,325,325,323,
    322,321,321,321,320,317,317,316,315,314,312,312,311,311,310,310,
    309,307,306,306,306,303,303,302,301,300,299,298,298,297,297,294,
    294,294,293,292,292,292,291,291,290,290,289,289,288,288,287,285,
    284,284,283,282,281,281,280,279,278,276,275,274,274,274,273,272,
    272,271,271,271,271,270,270,269,269,269,268,267,266,266,265,265,
    264,264,264,264,264,263,260,260,259,259,256,256,256,256,256,255,
    255,255,254,253,253,251,251,250,250,250,249,248,248,248,247,246,
    246,245,245,245,243,242,242,241,240,239,237,236,236,236,235,234,
    233,232,230,230,229,228,228,228,228,228,226,225,223,222,220,220,
    219,218,216,215,213,212,212,211,210,209,209,209,208,208,205,205,
    204,203,202,202,202,202,202,200,199,198,198,198,198,197,196,196,
    195,194,194,193,193,192,192,192,191,189,189,188,186,186,186,185,
    183,183,183,183,181,180,180,180,179,178,177,176,176,176,175,175,
    174,172,171,169,169,168,168,167,167,165,165,165,164,164,164,163,
    161,160,160,158,158,158,157,157,157,156,156,156,155,155,155,154,
    154,151,151,150,149,149,148,148,147,146,145,144,144,143,141,141,
    139,138,137,137,136,135,135,135,132,132,132,130,130,130,129,129,
    128,128,128,127,126,126,126,126,126,126,125,123,122,122,121,120,
    120,119,119,119,117,116,115,115,115,114,114,113,112,111,111,110,
    109,108,108,107,106,105,105,104,104,104,102,101,101,100,99,98,
    98,98,95,95,95,94,93,93,92,91,91,90,90,89,89,88,86,83,82,82,81,
    80,79,77,77,75,75,73,72,72,72,72,70,69,69,67,66,65,65,65,65,64,
    64,64,64,64,64,62,59,58,58,57,55,55,53,52,51,48,48,48,48,47,46,
    46,46,46,46,46,45,44,43,43,39,39,39,37,37,36,34,32,32,31,31,31,
    29,28,27,27,26,26,25,24,24,23,23,23,23,22,22,22
  };
  const int n4w2b3r6[] = {
    1000, // Capacity
    500, // Number of items
    // Size of items (sorted)
    378,378,377,377,377,374,374,373,372,372,371,371,370,369,368,366,
    366,365,364,364,363,363,362,361,358,357,357,357,356,356,355,355,
    351,351,349,348,345,345,344,344,340,339,338,338,337,336,335,335,
    334,332,332,331,330,329,329,329,327,327,326,325,324,323,323,321,
    321,321,320,318,318,318,317,316,315,315,315,314,314,313,312,312,
    311,311,310,308,306,306,305,304,304,303,303,301,301,299,298,298,
    296,295,295,294,292,291,289,288,287,286,286,285,285,284,284,283,
    282,282,282,282,282,282,280,279,279,279,278,278,278,277,277,276,
    276,274,274,273,272,272,271,271,271,271,269,267,267,265,264,264,
    264,263,263,263,262,262,261,261,259,258,257,255,255,254,252,251,
    251,250,250,250,249,248,247,247,246,245,245,243,243,242,241,240,
    240,240,238,237,236,236,235,235,234,233,231,231,230,230,229,228,
    227,227,227,226,225,225,224,223,223,222,222,222,222,221,220,219,
    219,218,218,217,216,215,215,215,214,212,212,211,211,210,209,209,
    209,208,206,206,206,204,203,202,202,202,201,200,200,200,200,200,
    198,198,198,197,196,195,194,194,192,191,190,189,189,188,188,188,
    187,186,186,186,185,185,185,185,184,183,182,182,182,181,181,180,
    179,179,179,177,177,177,177,176,174,174,174,174,173,173,173,172,
    172,170,168,168,167,165,165,164,164,163,163,163,162,160,160,159,
    159,158,157,156,156,156,155,155,155,155,154,154,153,153,152,152,
    151,150,149,149,148,148,147,147,147,147,146,146,144,144,143,143,
    143,141,140,139,139,139,138,138,138,136,136,135,135,135,133,133,
    132,132,132,131,130,130,129,128,126,126,124,124,124,123,123,120,
    120,119,119,118,118,118,117,116,115,115,113,112,111,111,111,110,
    110,110,110,109,108,108,108,108,107,107,105,105,105,104,103,103,
    103,102,101,101,100,100,97,97,96,96,95,95,95,95,95,94,90,88,88,
    87,86,86,86,85,85,85,84,83,81,81,81,79,79,76,76,76,74,74,73,72,
    72,72,72,71,70,68,67,66,65,65,63,61,59,58,58,58,57,56,55,55,55,
    54,54,52,51,50,50,49,47,47,46,46,43,42,42,42,41,41,41,41,39,39,
    39,36,33,33,31,31,29,29,28,27,27,27,26,25,25,23,23,22
  };
  const int n4w2b3r7[] = {
    1000, // Capacity
    500, // Number of items
    // Size of items (sorted)
    380,380,380,379,379,379,379,378,378,378,377,376,376,376,374,372,
    372,372,370,370,369,368,368,367,366,366,366,366,365,365,365,364,
    364,363,361,361,361,360,358,358,358,357,356,356,356,356,355,354,
    353,351,351,350,350,349,349,349,348,343,342,342,340,340,339,337,
    337,336,336,336,334,334,333,332,331,330,330,330,328,328,327,326,
    325,324,324,322,322,322,321,321,320,320,320,320,319,319,318,318,
    316,315,313,312,311,310,310,310,309,308,308,308,308,307,305,305,
    305,305,305,304,303,303,302,301,300,297,297,297,296,294,294,291,
    291,290,290,290,289,289,288,288,287,287,284,284,283,283,282,282,
    280,280,280,279,279,279,278,277,277,277,277,277,276,275,275,272,
    270,269,268,268,268,267,267,267,266,266,265,263,261,258,258,257,
    257,256,253,252,252,250,250,249,249,248,247,246,246,245,245,244,
    244,242,242,241,241,241,241,239,239,237,235,234,233,233,228,228,
    226,226,226,225,224,224,223,223,222,221,221,221,220,219,218,218,
    218,217,217,216,215,214,213,213,213,212,210,209,208,208,207,207,
    206,205,203,202,201,201,201,200,198,196,193,193,193,192,191,191,
    190,189,188,187,187,185,184,183,183,182,181,181,181,181,180,179,
    178,178,178,175,175,175,174,174,174,174,173,173,173,172,172,172,
    170,170,169,169,167,167,166,166,166,166,165,164,164,164,163,162,
    162,162,161,161,160,159,157,157,157,156,156,154,153,151,151,149,
    149,149,148,147,147,147,147,146,143,143,141,140,139,138,138,138,
    136,136,134,131,131,129,128,128,128,127,125,124,124,123,122,122,
    121,121,120,120,119,117,115,114,113,113,113,112,112,112,110,110,
    108,108,108,107,106,105,104,104,104,103,101,100,100,100,100,99,
    98,98,95,95,94,94,94,94,93,93,92,92,92,92,92,92,91,90,89,89,87,
    87,85,84,84,83,82,81,79,78,78,78,77,76,75,75,74,72,71,71,71,70,
    69,68,67,66,66,66,66,65,64,63,63,63,62,61,61,61,60,59,59,58,57,
    57,56,54,53,52,52,52,52,51,51,50,50,48,48,46,46,45,44,44,43,43,
    39,39,39,38,38,37,36,35,35,34,34,33,33,32,32,31,31,30,30,30,27,
    27,27,26,25,25,25,24,24,23,23,22
  };
  const int n4w2b3r8[] = {
    1000, // Capacity
    500, // Number of items
    // Size of items (sorted)
    380,379,378,378,376,375,374,373,372,372,371,370,370,366,366,364,
    363,363,362,361,361,361,361,361,360,360,359,357,356,356,356,355,
    353,352,352,350,350,349,347,346,346,346,345,345,344,343,342,342,
    340,340,339,339,339,339,338,337,335,335,335,333,333,331,331,331,
    330,330,329,328,328,327,327,325,324,324,324,324,323,321,321,321,
    320,320,318,316,315,315,314,314,313,311,308,308,308,307,307,306,
    305,305,304,304,302,302,300,300,299,298,298,297,296,295,292,291,
    289,289,289,288,288,287,287,287,286,286,286,285,285,284,284,283,
    283,281,281,280,280,279,278,278,278,277,276,275,274,274,273,272,
    272,272,271,270,269,268,266,265,265,263,260,259,258,258,258,258,
    257,257,257,256,255,255,253,253,253,252,251,250,250,249,248,248,
    246,245,245,244,243,243,242,241,241,238,238,238,237,236,234,234,
    233,232,232,231,230,230,228,228,228,228,227,226,225,225,225,222,
    222,222,221,221,220,219,217,216,216,216,215,214,213,213,213,212,
    212,211,208,208,208,207,206,206,204,203,202,202,201,201,196,195,
    195,195,195,194,194,193,192,191,191,189,189,189,188,187,186,186,
    185,184,184,184,183,183,182,182,182,182,181,181,180,180,179,178,
    177,176,175,175,175,174,173,171,171,170,170,170,170,169,168,168,
    168,167,167,166,166,166,164,164,164,162,162,162,162,161,161,161,
    160,158,157,156,155,154,153,152,152,151,150,150,150,149,148,148,
    148,147,147,147,145,145,145,142,141,139,139,139,139,138,138,138,
    136,135,134,133,133,132,132,132,131,130,129,129,127,127,125,125,
    125,124,123,121,121,121,120,119,119,119,118,118,118,117,117,117,
    117,116,115,115,114,112,112,111,111,111,109,109,109,108,108,107,
    107,105,104,102,102,100,99,99,99,99,96,95,94,94,93,89,88,87,86,
    85,85,85,85,84,84,83,83,82,82,82,82,81,81,81,80,79,78,78,78,77,
    76,76,74,74,73,72,72,71,71,71,69,67,65,64,64,64,64,63,62,61,61,
    60,59,57,55,55,53,53,52,51,51,51,50,50,49,48,48,48,47,46,46,45,
    45,45,43,42,42,42,42,40,40,40,40,40,39,38,38,34,34,34,34,33,33,
    32,32,30,30,30,29,27,27,23,23,22,22,22
  };
  const int n4w2b3r9[] = {
    1000, // Capacity
    500, // Number of items
    // Size of items (sorted)
    379,378,378,378,375,375,373,373,373,372,372,372,371,371,370,369,
    369,369,369,368,368,366,365,365,365,364,364,363,363,362,361,361,
    361,358,358,356,354,354,354,354,353,353,351,350,349,349,349,349,
    349,346,346,346,346,346,346,346,345,345,342,342,342,341,340,337,
    337,337,337,336,336,335,333,331,328,327,327,327,326,325,325,323,
    321,321,321,320,319,318,318,317,317,316,316,315,315,314,314,313,
    312,312,312,310,309,309,307,306,305,305,304,303,301,300,300,299,
    299,298,298,297,297,296,296,296,295,295,295,295,294,294,293,292,
    292,292,291,291,291,289,289,288,285,284,284,284,282,281,281,280,
    279,279,279,278,278,274,274,273,272,272,272,271,271,270,269,269,
    269,268,267,267,266,265,264,264,263,262,260,260,258,258,257,257,
    256,256,256,255,254,254,253,253,252,252,252,252,251,250,248,247,
    247,246,246,246,242,242,242,241,240,240,240,239,236,236,236,234,
    234,233,232,231,231,230,225,224,223,223,222,220,219,219,218,217,
    217,215,215,215,215,214,214,214,211,211,210,210,210,210,209,207,
    205,204,204,203,202,201,200,200,199,199,199,198,198,197,195,195,
    195,194,192,191,190,190,189,188,188,187,186,186,184,183,182,182,
    182,181,181,181,180,180,180,178,178,178,177,177,176,175,174,174,
    174,174,174,173,173,172,171,171,169,169,169,169,167,167,165,165,
    164,164,164,163,163,162,162,162,159,157,157,155,155,154,153,153,
    152,151,151,151,150,148,147,147,147,145,144,142,142,142,141,140,
    138,136,136,135,135,135,134,133,133,133,132,131,131,130,129,128,
    128,125,125,125,124,123,123,121,120,120,119,118,118,117,117,116,
    116,115,113,113,113,113,113,112,112,112,110,110,109,108,108,107,
    107,107,107,107,106,105,104,104,101,101,100,100,100,100,99,98,
    97,96,96,96,96,95,95,94,94,94,93,93,92,91,91,88,88,87,86,86,84,
    83,82,82,81,79,78,78,78,77,74,74,74,73,73,72,71,71,71,71,71,71,
    68,68,67,67,67,65,63,63,61,60,59,58,56,56,55,54,54,53,52,51,50,
    49,49,48,48,48,47,47,46,46,45,41,40,39,38,38,38,37,35,35,35,34,
    34,33,33,31,29,29,28,28,28,27,24,24,23,22,22,22
  };
  const int n4w3b1r0[] = {
    1000, // Capacity
    500, // Number of items
    // Size of items (sorted)
    168,168,168,168,168,168,168,168,168,167,167,167,167,167,167,167,
    167,167,167,167,167,166,166,166,166,166,165,165,165,165,165,165,
    165,165,165,165,165,165,164,164,164,164,164,164,164,164,164,164,
    164,164,164,164,164,164,163,163,163,163,163,163,163,163,162,162,
    162,162,162,162,162,162,162,162,162,162,162,161,161,161,161,161,
    161,161,161,161,161,161,161,161,161,160,160,160,160,160,160,160,
    160,160,160,160,160,159,159,159,159,159,159,158,157,157,157,157,
    157,157,157,157,157,156,156,156,156,156,156,156,156,156,156,156,
    156,155,155,155,155,155,155,155,155,155,154,154,154,154,154,154,
    154,153,153,153,153,153,153,152,152,152,152,152,152,152,151,151,
    151,151,151,151,151,151,151,151,151,150,150,150,150,150,150,150,
    150,149,149,149,149,148,148,148,148,148,147,147,147,147,147,147,
    146,146,146,146,146,146,146,146,145,145,145,145,145,145,145,145,
    145,145,145,145,145,145,145,145,144,144,144,144,144,144,144,144,
    144,144,143,143,143,143,143,143,143,143,143,143,142,142,142,142,
    142,142,142,142,142,142,141,141,141,141,141,141,141,140,140,140,
    140,140,140,140,140,140,140,140,139,139,139,139,139,139,139,138,
    138,138,138,138,137,137,137,137,137,137,137,137,137,137,137,137,
    137,137,136,136,136,136,136,136,136,136,136,135,135,135,135,135,
    135,135,135,135,135,134,134,134,134,134,134,134,134,134,134,134,
    133,133,133,132,132,132,132,132,132,132,132,132,132,132,132,132,
    132,131,131,131,131,131,131,131,131,131,131,131,131,131,131,131,
    131,131,130,130,130,130,130,130,130,129,129,129,129,129,129,129,
    129,128,128,128,128,128,128,128,127,127,127,127,127,127,126,126,
    126,126,126,126,126,125,125,125,125,125,125,125,125,125,125,125,
    125,124,124,124,124,124,124,124,124,123,123,123,123,123,123,123,
    122,122,122,122,122,122,122,122,121,121,121,121,121,121,121,121,
    121,121,120,120,120,120,120,120,120,119,119,119,119,119,119,119,
    118,118,118,118,118,118,118,118,118,118,118,118,118,118,117,117,
    117,117,117,117,117,116,116,116,116,116,116,116,116,115,115,115,
    115,115,115,115,115,115,115,114,114,114,114,114,114,114,114,114,
    114,114,114,114
  };
  const int n4w3b1r1[] = {
    1000, // Capacity
    500, // Number of items
    // Size of items (sorted)
    168,168,168,168,168,168,168,168,168,167,167,167,167,167,167,167,
    167,166,166,166,166,166,166,166,166,166,165,165,165,165,165,165,
    165,165,165,165,165,164,164,164,164,164,164,164,164,164,164,163,
    163,163,163,163,163,163,163,163,162,162,162,162,162,162,162,162,
    162,162,162,161,161,161,161,161,161,161,160,160,160,160,160,160,
    160,160,160,160,160,160,160,160,160,159,159,159,158,158,158,158,
    158,158,157,157,157,157,157,157,157,157,157,157,157,157,157,156,
    156,156,156,156,156,156,156,156,156,155,155,155,155,155,155,155,
    155,155,155,155,155,154,154,154,154,154,154,154,153,153,153,153,
    153,152,152,152,152,152,152,152,152,152,152,152,152,152,151,151,
    151,151,151,151,151,151,151,151,150,150,150,150,150,150,150,150,
    150,150,150,150,150,150,150,150,150,150,149,149,149,149,149,149,
    149,149,149,148,148,148,148,148,148,148,147,147,147,147,147,147,
    147,147,146,146,146,146,146,145,145,145,145,145,145,145,145,145,
    145,144,144,144,144,144,144,144,144,144,144,144,144,143,143,143,
    143,143,143,143,143,143,142,142,142,142,142,142,142,142,141,141,
    141,141,141,141,141,140,140,140,140,140,140,139,139,139,139,139,
    139,139,139,139,139,139,139,139,139,139,139,139,138,138,138,138,
    138,138,138,138,138,137,137,137,137,137,137,137,137,137,137,137,
    137,137,137,137,136,136,136,136,136,135,135,135,135,135,135,135,
    135,134,134,134,134,134,134,133,133,133,133,133,133,133,133,133,
    133,132,132,132,132,132,132,132,132,132,131,131,131,131,131,131,
    131,131,131,131,131,131,130,130,130,130,130,130,130,130,130,129,
    129,129,129,129,129,129,129,129,129,129,129,128,128,128,128,128,
    128,128,128,128,128,127,127,127,127,127,126,126,126,126,126,125,
    125,125,125,125,125,125,125,125,125,125,124,124,124,124,124,124,
    124,124,124,123,123,123,123,123,123,123,123,123,123,122,122,122,
    122,121,121,121,121,121,121,120,120,120,120,120,120,119,119,119,
    119,119,119,119,119,119,118,118,118,118,118,118,118,118,118,118,
    118,118,118,117,117,117,117,117,117,116,116,116,116,116,116,116,
    116,116,115,115,115,115,115,114,114,114,114,114,114,114,114,114,
    114,114,114,114
  };
  const int n4w3b1r2[] = {
    1000, // Capacity
    500, // Number of items
    // Size of items (sorted)
    168,168,168,168,168,167,167,167,167,167,167,167,167,167,167,167,
    167,167,167,167,167,166,166,166,166,166,166,166,166,166,166,166,
    165,165,165,165,165,165,165,165,165,165,164,164,164,164,164,164,
    163,163,163,163,163,163,162,162,162,162,162,162,162,162,162,162,
    162,161,161,161,161,161,161,161,161,161,161,161,161,161,161,161,
    160,160,160,160,160,160,160,160,160,160,160,160,160,160,159,159,
    159,159,159,159,159,159,159,159,159,159,159,159,159,158,158,158,
    158,157,157,157,157,157,157,156,156,156,156,156,156,156,156,156,
    156,155,155,155,155,155,155,155,155,155,155,155,154,154,154,154,
    154,154,153,153,153,153,153,153,153,153,152,152,152,152,152,152,
    152,152,151,151,151,151,151,151,151,151,150,150,150,150,150,150,
    149,149,149,149,149,149,149,149,149,149,149,149,148,148,148,148,
    148,148,148,148,148,148,148,148,147,147,147,147,147,147,147,147,
    147,146,146,146,146,146,146,146,146,146,146,146,146,146,146,145,
    145,145,145,145,145,145,145,145,144,144,144,144,143,143,143,143,
    143,143,143,142,142,142,142,142,142,142,141,141,141,141,141,141,
    141,141,141,141,141,141,141,141,141,140,140,140,140,140,139,139,
    139,139,139,139,139,139,138,138,138,138,138,138,138,138,138,137,
    137,137,137,137,137,137,137,137,136,136,136,136,136,136,136,136,
    136,136,136,135,135,135,135,135,135,135,135,135,135,135,134,134,
    134,134,134,134,134,134,134,134,134,134,134,134,134,133,133,133,
    133,133,133,133,133,133,132,132,132,132,132,132,132,131,131,131,
    131,131,131,131,130,130,130,130,130,130,130,130,129,129,129,129,
    129,129,129,129,129,129,129,128,128,128,128,128,128,127,127,127,
    127,127,126,126,126,126,126,126,126,126,126,126,125,125,125,125,
    125,125,124,124,124,124,124,124,124,124,124,124,124,124,123,123,
    123,123,123,123,122,122,122,122,122,122,122,121,121,121,121,121,
    121,121,121,121,121,121,121,120,120,120,120,120,120,120,120,120,
    119,119,119,119,119,119,119,119,119,118,118,118,118,118,118,118,
    118,118,118,118,118,118,118,118,118,117,117,117,117,117,117,117,
    117,116,116,116,116,116,116,116,116,115,115,115,115,114,114,114,
    114,114,114,114
  };
  const int n4w3b1r3[] = {
    1000, // Capacity
    500, // Number of items
    // Size of items (sorted)
    168,168,168,168,168,168,168,168,168,168,168,168,167,167,167,167,
    167,167,167,166,166,166,166,166,166,166,165,165,165,165,165,165,
    165,164,164,163,163,163,163,163,163,163,163,163,162,162,162,162,
    161,161,161,161,161,161,161,161,161,161,161,161,161,160,160,160,
    160,160,160,160,160,160,160,159,159,159,159,158,158,158,158,158,
    158,158,158,158,158,158,158,157,157,157,157,157,157,157,157,157,
    157,157,157,156,156,156,156,156,156,156,156,156,155,155,155,155,
    155,155,154,154,154,154,154,154,154,153,153,153,153,152,152,152,
    152,152,152,152,152,152,152,152,151,151,151,151,151,151,151,151,
    151,151,151,151,151,151,150,150,150,150,150,150,150,150,150,150,
    149,149,149,149,149,149,149,149,149,148,148,148,148,147,147,147,
    147,147,147,147,147,146,146,146,146,146,146,146,146,146,146,146,
    146,146,146,146,146,146,146,146,145,145,145,145,145,145,145,145,
    145,145,144,144,144,144,144,144,144,143,143,143,143,143,143,143,
    143,142,142,142,142,142,142,142,142,142,142,142,142,141,141,141,
    141,141,141,141,141,141,140,140,140,140,140,140,140,140,140,140,
    140,139,139,139,139,139,139,139,138,138,138,138,138,138,138,137,
    137,137,137,137,137,137,137,136,136,136,136,136,136,136,136,136,
    136,135,135,135,135,135,135,135,135,135,134,134,134,134,134,134,
    134,134,134,134,134,134,134,134,134,134,133,133,133,133,133,133,
    133,133,133,133,133,132,132,132,132,132,132,132,132,132,132,131,
    131,131,131,131,131,131,131,131,131,130,130,130,130,130,130,130,
    130,129,129,129,129,129,129,129,129,129,129,129,128,128,128,128,
    128,128,128,127,127,127,127,127,127,127,127,126,126,126,126,126,
    126,126,126,126,125,125,125,125,125,125,125,125,125,124,124,124,
    124,124,124,123,123,123,123,123,123,123,122,122,122,122,122,122,
    122,122,122,122,122,122,122,122,122,122,122,122,122,121,121,121,
    121,121,121,121,120,120,120,120,120,120,120,120,120,120,120,120,
    119,119,119,119,119,119,119,119,119,118,118,118,118,118,118,118,
    118,118,118,118,118,117,117,117,117,117,116,116,116,116,116,116,
    115,115,115,115,115,115,115,114,114,114,114,114,114,114,114,114,
    114,114,114,114
  };
  const int n4w3b1r4[] = {
    1000, // Capacity
    500, // Number of items
    // Size of items (sorted)
    168,168,168,168,168,168,168,168,168,168,168,167,167,167,167,167,
    167,167,167,167,166,166,166,166,166,166,166,165,165,165,165,165,
    165,165,164,164,164,164,164,164,164,164,164,164,164,164,163,163,
    163,163,163,163,162,162,162,162,162,162,162,162,162,162,162,162,
    162,161,161,161,161,161,161,161,161,161,161,161,160,160,160,160,
    160,160,160,159,159,159,159,159,159,159,158,158,158,158,158,158,
    157,157,157,157,157,157,157,157,157,157,157,156,156,156,156,156,
    156,155,155,155,155,155,155,155,155,155,155,154,154,154,154,154,
    154,154,154,153,153,153,153,153,153,153,153,153,152,152,152,152,
    152,152,152,151,151,151,151,151,150,150,150,150,150,150,150,150,
    150,149,149,149,149,149,149,149,149,148,148,148,148,148,148,148,
    148,148,147,147,147,147,147,147,147,147,146,146,146,146,146,146,
    146,146,145,145,145,145,145,145,145,145,145,145,145,145,145,144,
    144,144,144,144,144,144,144,144,144,143,143,143,143,143,143,143,
    143,143,143,143,143,143,143,143,143,142,142,142,142,142,142,142,
    142,142,142,142,141,141,141,141,141,141,141,141,140,140,140,140,
    140,140,140,140,140,140,140,139,139,139,139,139,139,139,139,139,
    138,138,138,138,138,138,138,138,138,138,138,138,137,137,137,137,
    137,137,137,137,137,137,136,136,136,136,136,136,136,136,136,135,
    135,135,135,135,135,135,135,135,135,135,135,135,134,134,134,134,
    134,134,133,133,133,133,133,133,133,133,132,132,132,132,132,132,
    132,132,132,132,132,132,132,131,131,131,131,131,131,131,130,130,
    130,130,130,130,130,129,129,129,129,129,129,129,128,128,128,128,
    128,128,128,128,128,128,127,127,127,127,127,127,127,127,127,126,
    126,126,126,126,126,126,126,126,126,126,125,125,125,125,125,125,
    125,125,124,124,124,124,124,124,124,124,124,124,123,123,123,123,
    123,123,123,123,123,123,122,122,122,122,122,122,121,121,121,121,
    121,121,121,120,120,120,120,120,120,120,120,120,120,119,119,119,
    119,119,119,119,119,118,118,118,118,118,118,118,118,118,117,117,
    117,117,117,117,117,117,117,117,117,116,116,116,116,116,116,116,
    116,116,116,116,116,116,115,115,115,115,115,115,115,115,115,114,
    114,114,114,114
  };
  const int n4w3b1r5[] = {
    1000, // Capacity
    500, // Number of items
    // Size of items (sorted)
    168,168,168,168,168,168,168,168,167,167,167,167,167,167,167,167,
    167,167,167,166,166,166,166,166,166,166,166,166,166,165,165,165,
    165,165,165,165,165,165,165,165,165,165,165,164,164,164,164,164,
    164,164,164,164,163,163,163,163,163,163,163,163,163,163,163,162,
    162,162,162,162,162,162,162,161,161,161,161,161,161,161,161,160,
    160,160,160,160,160,160,160,160,160,159,159,159,159,159,159,159,
    159,159,159,159,159,158,158,158,158,158,158,158,158,158,157,157,
    157,157,157,157,157,157,157,157,157,157,156,156,156,156,156,155,
    155,155,155,155,155,155,155,155,155,154,154,154,154,154,154,153,
    153,153,153,153,153,153,153,153,152,152,152,152,152,152,152,152,
    151,151,151,151,151,151,151,151,151,151,151,151,151,150,150,150,
    150,150,149,149,149,149,148,148,148,148,147,147,147,147,147,147,
    147,147,147,146,146,146,146,146,146,146,146,146,146,145,145,145,
    145,145,145,145,145,145,144,144,144,144,144,144,144,144,144,144,
    144,144,144,144,143,143,143,143,143,143,143,142,142,142,142,142,
    142,142,142,142,141,141,141,141,141,141,141,141,141,141,140,140,
    140,140,140,140,140,139,139,139,139,139,139,139,139,139,139,139,
    138,138,138,138,138,138,137,137,137,137,137,137,136,136,136,136,
    136,136,136,136,136,136,136,135,135,135,135,135,135,135,135,135,
    135,135,135,135,135,134,134,134,134,134,134,134,133,133,133,133,
    133,133,133,133,133,133,133,133,133,132,132,132,132,132,132,132,
    131,131,131,131,131,131,131,131,131,131,130,130,130,130,130,130,
    129,129,129,129,129,129,129,129,129,129,129,129,129,128,128,128,
    128,128,128,128,128,128,127,127,127,127,127,127,126,126,126,126,
    126,126,126,126,126,126,126,126,125,125,125,125,125,125,125,125,
    125,125,125,124,124,124,124,124,124,123,123,123,123,123,123,123,
    123,123,123,123,122,122,122,122,122,122,122,122,122,121,121,121,
    121,121,121,121,121,121,121,121,121,121,121,120,120,120,120,120,
    120,120,120,120,120,119,119,119,119,119,119,119,119,118,118,118,
    118,118,118,118,118,118,117,117,117,117,117,117,117,117,117,117,
    116,116,116,116,115,115,115,115,114,114,114,114,114,114,114,114,
    114,114,114,114
  };
  const int n4w3b1r6[] = {
    1000, // Capacity
    500, // Number of items
    // Size of items (sorted)
    168,168,168,168,168,168,168,168,167,167,167,167,167,167,167,167,
    167,167,166,166,166,166,166,165,165,165,165,165,165,165,165,165,
    164,164,164,164,164,164,164,164,164,164,164,164,164,164,164,163,
    163,163,163,163,163,163,163,162,162,162,162,162,161,161,161,161,
    161,161,161,161,161,161,161,161,161,160,160,160,160,160,159,159,
    159,158,158,158,158,158,158,158,158,157,157,157,157,157,157,157,
    157,156,156,156,156,156,156,156,155,155,155,155,155,155,155,155,
    155,155,155,155,155,155,154,154,154,154,153,153,153,153,153,153,
    153,153,153,152,152,152,152,152,152,152,152,152,152,152,152,152,
    152,152,152,151,151,151,151,151,151,151,151,150,150,150,150,150,
    150,150,150,150,149,149,149,149,149,149,149,149,149,148,148,148,
    148,148,148,148,148,148,148,147,147,147,147,147,147,147,147,147,
    146,146,146,146,146,146,146,146,146,146,146,145,145,145,145,145,
    145,145,145,145,144,144,144,144,144,144,144,144,144,143,143,143,
    143,143,143,143,143,143,143,143,142,142,142,142,142,142,142,142,
    142,142,141,141,141,141,140,140,140,140,140,140,140,140,139,139,
    139,139,139,139,139,138,138,138,138,138,138,137,137,137,137,137,
    137,137,137,137,136,136,136,136,136,136,135,135,135,135,135,135,
    135,135,135,135,134,134,134,134,134,134,134,134,134,134,134,133,
    133,133,133,133,133,133,133,133,132,132,132,132,132,132,131,131,
    131,131,131,131,131,131,131,131,131,131,130,130,130,130,130,130,
    130,129,129,129,129,129,129,129,129,129,129,129,128,128,128,128,
    128,128,128,128,128,128,128,128,128,128,127,127,127,127,127,127,
    127,127,127,127,127,126,126,126,126,126,126,126,126,126,126,126,
    126,126,126,126,125,125,125,125,125,125,125,125,125,125,125,125,
    124,124,124,124,124,124,124,124,123,123,123,123,123,123,123,123,
    123,123,123,123,123,123,123,122,122,122,122,122,122,122,122,122,
    122,121,121,121,121,121,121,120,120,120,120,120,120,120,119,119,
    119,119,119,119,119,119,118,118,118,118,118,118,117,117,117,117,
    117,117,117,117,117,117,117,116,116,116,116,116,116,116,116,116,
    116,115,115,115,115,115,115,115,115,115,114,114,114,114,114,114,
    114,114,114,114
  };
  const int n4w3b1r7[] = {
    1000, // Capacity
    500, // Number of items
    // Size of items (sorted)
    168,168,168,168,168,168,168,168,168,168,168,167,167,167,167,167,
    167,167,167,166,166,166,166,166,166,166,166,166,166,166,166,166,
    166,165,165,165,165,165,165,165,165,165,164,164,164,164,164,164,
    164,163,163,163,163,163,163,163,163,163,163,163,163,162,162,162,
    162,162,162,162,162,161,161,161,161,161,161,161,161,161,161,161,
    161,160,160,160,160,160,160,160,159,159,159,159,159,159,159,159,
    158,158,158,158,158,158,158,157,157,157,157,157,156,156,156,156,
    156,156,156,155,155,155,155,155,155,154,154,154,154,154,154,154,
    154,154,154,153,153,153,153,153,153,153,153,153,153,153,153,153,
    152,152,152,152,152,152,152,152,151,151,151,151,151,151,151,151,
    151,151,151,150,150,150,150,150,150,150,150,150,149,149,149,149,
    149,149,149,149,149,149,148,148,148,148,148,148,148,148,148,148,
    148,148,147,147,147,147,147,147,147,146,146,146,146,146,146,146,
    146,146,145,145,145,145,145,145,145,145,144,144,144,144,144,144,
    144,143,143,143,143,143,143,143,143,143,143,143,143,142,142,142,
    142,142,142,142,141,141,141,141,141,141,141,140,140,140,140,140,
    140,140,140,140,139,139,139,139,139,139,139,138,138,138,138,138,
    138,137,137,137,137,137,137,137,136,136,136,136,136,135,135,135,
    135,134,134,134,134,134,134,134,134,134,133,133,133,133,133,133,
    133,133,133,133,133,133,133,132,132,132,132,132,132,132,131,131,
    131,131,131,131,130,130,130,130,130,130,130,130,130,129,129,129,
    129,129,129,128,128,128,128,128,128,128,128,128,127,127,127,127,
    127,127,127,127,127,127,127,127,127,126,126,126,126,126,126,125,
    125,125,125,125,125,125,125,125,125,124,124,124,124,124,124,124,
    124,124,123,123,123,123,123,123,123,122,122,122,122,122,122,122,
    122,122,122,121,121,121,121,121,121,121,121,121,121,121,121,120,
    120,120,120,120,120,120,120,120,119,119,119,119,119,119,119,119,
    119,119,119,118,118,118,118,118,118,118,118,118,118,118,118,118,
    118,118,117,117,117,117,117,117,117,117,117,116,116,116,116,116,
    116,116,116,116,116,116,116,116,116,116,115,115,115,115,115,115,
    115,115,115,115,115,115,114,114,114,114,114,114,114,114,114,114,
    114,114,114,114
  };
  const int n4w3b1r8[] = {
    1000, // Capacity
    500, // Number of items
    // Size of items (sorted)
    168,168,168,168,168,168,167,167,167,167,167,167,167,167,167,167,
    167,166,166,166,166,166,166,166,166,166,166,166,166,166,166,166,
    165,165,165,165,165,165,165,165,165,164,164,164,164,164,164,164,
    164,164,163,163,163,163,163,163,163,163,163,163,162,162,162,162,
    162,162,162,161,161,161,161,160,159,159,159,159,159,159,159,159,
    159,159,158,158,158,158,158,158,158,158,157,157,157,157,157,156,
    156,156,156,156,156,156,155,155,155,155,155,155,155,155,155,154,
    154,154,154,154,154,154,154,154,154,154,154,153,153,153,153,153,
    153,153,152,152,152,152,152,152,152,152,152,151,151,151,151,151,
    151,151,151,151,150,150,150,150,150,150,150,150,150,150,149,149,
    149,149,149,149,149,149,149,149,148,148,148,148,148,148,148,148,
    148,148,148,148,148,148,147,147,147,147,147,147,147,147,146,146,
    146,146,146,146,146,146,146,146,146,146,145,145,145,145,145,145,
    145,145,145,144,144,144,144,144,144,144,143,143,143,143,143,143,
    143,143,142,142,142,142,142,142,142,142,142,142,142,141,141,141,
    141,141,141,141,141,141,140,140,140,140,140,140,140,140,140,140,
    140,139,139,139,139,139,139,138,138,138,138,138,138,138,138,138,
    138,138,138,137,137,137,137,137,137,137,137,137,137,137,136,136,
    136,136,136,136,136,136,136,135,135,135,135,135,135,135,135,135,
    135,135,135,135,135,134,134,134,134,133,133,133,133,133,133,133,
    133,133,132,132,132,132,132,132,132,132,132,132,132,131,131,131,
    131,130,130,130,130,130,130,130,130,130,129,129,129,129,129,129,
    129,129,129,129,129,128,128,128,128,128,128,128,128,127,127,127,
    127,127,127,127,127,127,127,127,127,126,126,126,126,126,126,126,
    126,126,125,125,125,125,125,125,125,124,124,124,124,124,124,124,
    123,123,123,123,123,123,123,123,122,122,122,122,122,122,122,122,
    122,122,121,121,121,121,121,121,121,121,120,120,120,120,120,120,
    120,119,119,119,119,119,119,119,119,119,119,119,119,118,118,118,
    118,118,118,118,118,118,118,118,117,117,117,117,117,117,117,117,
    117,117,117,117,117,116,116,116,116,116,116,116,116,116,116,116,
    116,116,116,116,116,115,115,115,115,115,115,115,115,114,114,114,
    114,114,114,114
  };
  const int n4w3b1r9[] = {
    1000, // Capacity
    500, // Number of items
    // Size of items (sorted)
    168,168,168,168,168,168,168,168,168,167,167,167,167,167,167,167,
    167,167,167,166,166,166,166,166,166,166,166,165,165,165,165,165,
    165,165,165,165,165,165,165,165,165,164,164,164,164,164,164,164,
    164,163,163,163,163,163,163,162,162,162,162,162,162,162,162,162,
    162,162,161,161,161,161,161,161,161,161,161,161,161,161,161,160,
    160,160,160,160,160,160,160,160,160,160,159,159,159,159,159,159,
    159,159,158,158,158,158,158,158,158,158,158,158,158,158,158,157,
    157,157,157,157,157,157,157,157,157,157,157,157,157,157,157,157,
    157,157,156,156,156,156,156,156,156,155,155,155,155,155,155,155,
    155,154,154,154,154,154,153,153,153,152,152,152,152,152,152,152,
    152,152,152,152,152,151,151,151,151,151,151,151,151,151,151,151,
    150,150,150,150,150,150,150,150,150,150,150,150,149,149,149,149,
    149,149,149,149,148,148,148,148,148,148,148,147,147,147,147,147,
    147,147,147,146,146,146,146,146,146,146,146,146,146,146,146,146,
    145,145,145,145,145,145,145,145,145,145,145,145,144,144,144,144,
    144,144,144,144,144,144,144,144,143,143,143,143,143,143,143,142,
    142,142,142,142,142,142,142,142,141,141,141,141,141,140,140,140,
    140,140,140,140,140,140,139,139,139,139,139,139,139,138,138,138,
    138,138,138,138,137,137,137,137,137,137,137,137,136,136,136,136,
    136,136,136,136,136,136,135,135,135,135,135,135,135,135,134,134,
    134,134,134,134,134,133,133,133,133,133,133,133,133,133,132,132,
    132,132,132,132,132,132,132,132,132,132,131,131,131,131,131,131,
    131,131,131,131,130,130,130,130,130,130,129,129,129,129,129,129,
    129,129,129,129,129,128,128,128,128,128,128,128,128,128,127,127,
    127,127,127,127,127,126,126,126,126,126,126,126,126,126,125,125,
    125,125,125,125,125,125,125,125,125,124,124,124,124,124,124,124,
    124,124,123,123,123,123,123,122,122,122,122,122,122,121,121,121,
    121,121,121,121,121,121,120,120,120,120,120,120,120,120,120,119,
    119,119,119,119,119,119,119,119,119,119,118,118,118,118,118,118,
    118,118,118,118,117,117,117,117,117,117,117,117,116,116,116,116,
    116,116,116,115,115,115,115,115,115,115,115,114,114,114,114,114,
    114,114,114,114
  };
  const int n4w3b2r0[] = {
    1000, // Capacity
    500, // Number of items
    // Size of items (sorted)
    210,210,210,209,209,209,209,208,208,208,208,207,207,206,206,206,
    206,205,205,205,205,205,205,204,204,202,201,201,201,201,200,200,
    200,200,200,200,199,199,199,199,199,199,198,198,197,197,197,197,
    197,197,197,197,197,197,196,196,196,196,196,195,195,195,195,195,
    195,195,194,194,194,193,192,192,191,191,191,190,190,190,190,189,
    189,189,189,188,188,187,187,187,186,186,186,185,185,185,185,185,
    185,184,184,183,183,183,183,183,183,182,182,182,182,181,181,181,
    180,180,180,179,179,179,179,179,178,178,178,178,177,176,176,176,
    176,175,175,175,174,174,174,174,173,173,172,172,172,172,171,171,
    171,171,170,170,170,169,169,169,168,168,168,168,168,168,168,168,
    167,166,166,165,165,164,164,164,164,164,163,163,163,162,162,162,
    161,161,161,161,161,161,160,160,159,159,159,159,159,159,158,158,
    158,158,157,157,156,156,156,156,155,155,155,155,154,154,154,154,
    154,154,154,153,153,153,153,152,152,152,151,151,151,151,150,150,
    150,150,149,149,148,148,148,148,148,148,148,148,148,148,148,147,
    147,147,146,145,145,144,144,144,144,144,144,143,143,143,143,142,
    142,142,142,142,141,141,141,141,141,140,140,140,139,139,139,139,
    138,138,137,137,136,136,136,136,135,134,134,134,134,134,133,133,
    132,131,131,131,130,130,130,130,130,129,129,128,128,127,127,126,
    126,126,126,126,126,126,125,125,125,123,123,123,123,123,122,122,
    122,121,121,121,121,119,119,119,119,119,119,118,117,116,116,116,
    116,116,115,115,115,114,114,114,114,113,113,113,113,113,113,113,
    113,112,111,111,111,111,111,110,110,110,109,109,109,108,108,108,
    107,107,107,106,106,106,105,105,105,104,104,104,104,103,103,102,
    101,101,101,101,101,101,99,99,99,99,99,98,98,98,98,98,98,97,97,
    97,96,96,96,95,95,95,95,95,94,94,94,94,94,94,93,93,93,93,92,92,
    92,91,91,91,91,90,90,89,89,89,88,88,88,88,88,87,87,87,86,86,86,
    86,85,85,85,84,84,84,83,83,82,82,81,81,81,81,81,80,80,80,80,80,
    80,79,79,79,78,78,78,78,78,78,78,78,77,76,76,76,75,75,75,74,74,
    74,73,73,73,73,73,73,73,73,72,72,72,72
  };
  const int n4w3b2r1[] = {
    1000, // Capacity
    500, // Number of items
    // Size of items (sorted)
    210,209,208,208,208,207,207,206,206,205,205,205,204,204,204,203,
    203,202,202,202,201,201,200,200,200,199,199,199,198,198,198,197,
    197,197,196,196,196,196,195,195,195,195,194,193,193,193,193,192,
    192,192,192,192,192,191,191,191,191,191,191,190,190,189,189,188,
    188,188,187,187,187,187,187,187,186,186,186,186,186,186,185,185,
    184,184,184,183,182,182,182,182,182,182,182,181,181,181,181,180,
    180,179,179,179,179,178,178,178,178,178,177,177,177,177,176,176,
    176,176,175,175,174,174,174,174,174,174,173,173,173,173,172,171,
    171,171,171,171,170,170,170,170,170,169,169,169,169,169,168,168,
    168,168,168,168,168,167,167,166,166,166,165,165,165,164,164,164,
    163,163,163,163,162,162,161,161,161,160,159,159,159,159,158,158,
    158,158,158,157,157,156,156,156,156,156,156,156,156,155,155,155,
    155,155,154,154,154,154,153,153,153,153,153,152,152,152,152,152,
    151,151,151,150,150,150,150,148,148,147,147,147,147,147,147,147,
    147,146,146,146,145,145,145,145,145,145,144,144,144,144,143,143,
    143,143,143,142,142,142,142,142,142,142,142,141,141,141,140,140,
    139,139,139,137,137,137,137,137,137,136,136,136,136,136,136,135,
    135,135,135,135,135,134,134,134,134,133,133,133,133,133,132,132,
    131,131,131,131,130,130,129,129,129,129,129,128,128,128,128,127,
    127,127,127,127,127,126,126,125,125,125,125,125,125,124,124,124,
    123,123,122,122,121,121,121,121,120,120,120,120,120,119,119,119,
    119,118,117,117,117,117,117,117,116,116,115,115,114,114,114,114,
    114,113,113,113,113,113,112,112,112,112,112,111,111,110,110,110,
    110,109,109,108,108,108,106,106,106,106,105,105,105,105,104,104,
    104,104,103,103,103,103,103,103,103,102,102,102,100,100,100,100,
    100,99,99,99,98,98,98,98,97,97,97,96,96,96,96,95,95,95,94,94,
    94,94,94,94,94,93,93,93,92,92,92,92,92,92,92,91,91,91,90,90,90,
    90,89,89,89,89,89,88,88,88,87,87,87,87,86,86,86,86,86,86,85,85,
    84,84,84,83,83,83,82,82,81,81,80,80,80,79,79,79,78,78,78,77,77,
    77,77,77,76,76,75,75,75,75,74,74,74,73,73,73,72,72
  };
  const int n4w3b2r2[] = {
    1000, // Capacity
    500, // Number of items
    // Size of items (sorted)
    210,210,210,209,209,208,208,208,208,208,207,207,206,206,205,204,
    203,203,203,202,202,202,202,202,202,202,201,200,200,200,200,199,
    199,199,198,198,198,198,197,197,197,197,197,197,197,196,196,196,
    196,196,196,196,195,195,195,195,195,195,195,195,194,192,192,192,
    192,191,191,190,190,190,190,190,190,189,189,189,189,189,188,188,
    188,187,187,186,186,186,185,185,185,185,185,185,185,185,185,184,
    183,183,183,183,182,182,182,181,181,181,181,180,180,180,179,179,
    179,179,179,179,178,178,177,177,176,176,176,175,175,175,175,174,
    174,174,174,173,173,172,172,172,172,172,172,172,171,171,171,171,
    171,170,170,170,170,170,169,169,169,169,169,168,168,168,168,167,
    167,167,167,167,166,166,166,166,165,165,165,165,164,164,164,163,
    163,163,163,162,162,162,162,162,161,161,161,161,160,160,160,160,
    159,159,159,158,158,158,157,156,155,155,155,154,154,154,154,154,
    153,153,153,153,153,153,152,152,151,151,150,150,150,150,150,149,
    149,149,149,148,148,148,148,148,147,146,146,145,144,144,144,144,
    143,143,142,142,142,141,141,141,140,140,140,140,140,140,139,139,
    139,139,138,138,138,137,137,136,136,136,135,135,135,135,135,135,
    135,135,134,134,134,133,133,133,133,133,133,133,132,132,132,132,
    132,132,131,131,131,131,130,130,129,128,128,128,127,127,127,127,
    127,126,126,126,125,125,125,124,124,124,124,123,123,123,123,122,
    122,121,121,121,121,120,119,118,118,118,117,117,117,116,116,116,
    116,116,115,115,115,115,114,114,113,113,113,112,112,112,112,111,
    111,111,111,111,111,110,110,110,110,109,109,108,108,107,107,107,
    107,106,105,105,105,105,105,105,105,104,104,104,104,104,103,103,
    102,102,101,101,100,100,100,100,100,98,98,98,98,98,98,98,98,97,
    97,97,97,97,97,96,96,96,96,95,95,95,95,94,94,94,94,93,93,92,92,
    91,91,91,91,91,90,90,89,89,89,89,89,88,88,87,87,86,86,86,85,84,
    84,84,84,84,83,83,83,83,83,83,83,83,82,81,81,81,81,81,81,81,81,
    80,80,79,79,79,79,79,79,78,78,78,78,78,78,77,76,76,76,75,75,75,
    74,74,74,74,74,74,73,73,73,73,73,73,73,72
  };
  const int n4w3b2r3[] = {
    1000, // Capacity
    500, // Number of items
    // Size of items (sorted)
    210,210,209,209,209,209,209,209,208,208,208,207,206,206,206,206,
    206,206,205,205,205,205,204,204,204,204,204,204,203,203,203,203,
    202,202,202,202,202,201,201,201,201,201,200,200,200,200,199,199,
    199,199,199,199,199,198,198,197,197,197,197,196,196,196,196,195,
    195,195,195,194,192,192,192,192,191,191,190,190,189,189,189,188,
    188,188,188,188,188,187,186,186,185,185,185,185,184,183,183,183,
    183,183,183,183,183,183,182,182,181,181,180,180,180,179,179,179,
    179,179,179,179,178,178,178,177,177,177,176,176,176,176,176,175,
    175,175,174,174,173,173,173,173,173,173,173,172,172,172,172,171,
    171,171,170,170,170,168,168,168,168,168,168,167,167,166,166,166,
    166,165,165,165,163,163,163,162,162,162,161,161,161,160,160,160,
    160,160,159,159,159,159,159,159,159,158,158,158,157,157,157,156,
    156,156,156,155,155,155,154,154,154,154,154,154,153,153,153,152,
    151,151,151,151,151,150,150,150,149,149,149,149,149,148,148,147,
    147,147,146,146,146,146,145,145,145,145,145,144,144,144,144,143,
    143,143,142,141,141,141,141,141,141,141,140,140,139,139,139,139,
    138,138,138,137,137,137,136,136,136,136,136,135,134,133,132,132,
    132,132,132,132,131,131,131,130,130,130,130,130,130,130,129,129,
    129,129,129,129,129,129,128,128,128,128,128,127,127,126,126,125,
    125,125,125,125,124,124,124,124,124,123,123,122,122,121,121,120,
    120,120,119,119,119,118,118,118,118,118,117,117,117,117,117,117,
    116,115,115,115,115,114,114,114,113,113,113,113,112,112,112,112,
    111,111,111,111,110,110,110,110,110,110,109,109,109,109,108,108,
    108,108,108,107,107,107,106,106,106,106,106,106,106,105,104,104,
    103,103,103,102,102,102,102,101,101,101,101,100,100,100,100,99,
    99,99,99,98,98,98,98,97,96,95,95,95,95,95,95,94,94,94,94,93,93,
    92,92,92,91,91,91,91,91,91,91,91,91,91,90,90,89,89,89,89,89,88,
    88,88,88,88,88,88,88,88,87,87,87,86,85,85,85,85,85,84,84,84,83,
    83,83,82,82,82,82,81,81,80,80,80,79,79,79,79,78,77,77,77,76,76,
    76,76,76,76,75,75,74,74,74,74,73,73,73,72,72,72
  };
  const int n4w3b2r4[] = {
    1000, // Capacity
    500, // Number of items
    // Size of items (sorted)
    210,210,210,210,209,209,209,209,208,208,207,207,207,207,207,207,
    206,206,206,206,206,206,206,206,206,205,205,204,204,203,203,203,
    203,202,202,202,201,200,200,200,200,200,200,199,199,199,198,198,
    198,198,198,198,197,197,197,197,197,197,197,196,196,196,195,195,
    194,194,194,194,194,193,192,192,192,192,192,191,191,190,190,189,
    189,188,188,187,187,187,187,187,187,186,186,186,186,185,185,185,
    185,185,184,184,184,184,184,183,183,183,183,183,183,183,183,182,
    182,182,182,181,181,181,181,180,180,180,179,179,179,179,179,178,
    178,178,178,178,178,178,177,177,176,176,175,175,175,175,175,174,
    174,173,173,173,173,173,173,172,172,172,172,172,172,171,171,171,
    171,171,170,170,169,169,169,169,169,169,169,169,169,168,168,167,
    167,166,166,166,166,165,165,165,165,165,164,164,164,164,164,164,
    164,164,164,164,163,163,163,162,162,162,161,161,161,161,160,160,
    160,160,160,160,159,159,158,158,158,157,157,156,156,156,155,155,
    154,153,153,152,152,152,152,152,151,151,151,151,151,151,151,151,
    150,150,150,150,150,149,149,149,148,147,147,147,147,147,147,146,
    145,145,145,145,144,144,143,142,141,141,141,140,140,140,140,139,
    139,139,139,139,138,138,137,136,134,134,134,134,134,132,132,132,
    132,132,132,132,131,131,131,131,131,131,131,131,130,130,130,129,
    129,129,129,129,128,128,128,128,127,127,127,127,127,126,126,126,
    125,125,125,124,124,124,123,123,123,122,122,122,122,122,122,121,
    121,121,121,120,120,119,119,119,119,118,118,118,117,117,117,117,
    117,116,116,116,114,114,114,114,114,114,113,113,113,112,112,112,
    112,112,112,112,111,111,111,111,110,110,110,109,109,109,109,109,
    107,107,107,107,107,107,107,106,106,106,105,105,105,105,105,103,
    102,102,102,102,102,101,100,99,99,99,98,98,97,97,97,97,96,96,
    96,96,96,96,96,96,95,95,95,94,94,94,93,93,93,93,93,93,93,93,92,
    92,92,92,92,91,91,91,91,90,90,90,88,88,87,87,86,86,86,85,85,85,
    84,84,84,84,83,83,83,83,83,83,83,82,82,82,82,81,81,80,80,80,80,
    79,79,78,78,78,76,76,76,76,75,75,75,74,74,73,73,72,72,72
  };
  const int n4w3b2r5[] = {
    1000, // Capacity
    500, // Number of items
    // Size of items (sorted)
    210,210,210,210,210,210,210,209,209,209,209,208,208,208,208,207,
    207,207,207,207,207,207,206,206,206,206,205,205,204,204,203,203,
    203,203,203,202,201,201,201,201,201,200,200,200,199,199,199,199,
    199,198,198,198,197,197,197,197,196,196,196,195,195,195,195,195,
    195,195,195,194,194,194,193,193,193,193,193,192,192,191,190,190,
    190,189,189,189,189,189,189,189,188,186,186,186,186,186,185,184,
    183,183,183,183,183,182,182,182,182,182,182,182,182,182,181,181,
    181,181,180,180,180,180,180,180,179,179,179,178,178,177,177,177,
    177,177,177,177,176,176,175,175,175,175,175,174,174,174,174,174,
    174,173,173,173,173,172,172,172,172,172,172,172,172,171,170,170,
    170,169,169,169,168,168,168,168,168,167,167,167,167,167,166,166,
    165,165,165,165,164,164,164,164,164,164,164,163,162,161,161,161,
    161,161,160,160,160,160,159,159,158,158,157,157,156,156,156,155,
    155,155,155,154,153,153,153,152,152,151,151,151,151,151,150,150,
    150,149,149,149,149,149,149,148,148,148,148,148,147,147,147,146,
    146,146,145,145,145,143,143,143,142,142,141,141,141,140,140,140,
    140,140,140,139,139,139,138,138,138,138,138,137,137,137,136,136,
    136,135,135,135,134,134,134,133,133,133,132,132,132,131,131,129,
    129,128,128,128,128,127,127,127,126,126,126,125,125,125,125,125,
    125,124,124,124,124,124,123,123,123,123,123,122,122,122,121,121,
    120,120,120,120,119,119,118,118,118,118,118,117,117,117,116,116,
    116,115,115,115,114,114,114,114,113,112,112,112,112,112,112,112,
    111,111,111,111,111,110,110,110,110,110,109,109,109,109,109,108,
    108,108,108,108,108,108,107,107,107,107,106,106,106,106,106,106,
    104,104,104,103,103,103,102,102,102,102,102,101,100,100,100,99,
    99,99,99,99,99,98,98,97,97,97,97,97,97,97,97,96,96,95,95,95,95,
    94,94,94,94,94,93,93,93,93,92,92,92,91,91,91,91,91,91,91,90,89,
    89,88,88,87,87,87,87,87,86,86,85,85,85,84,83,83,83,83,83,82,82,
    82,82,81,80,80,80,80,80,79,79,79,79,78,78,78,78,78,77,77,76,76,
    75,75,75,75,75,75,74,74,74,73,73,73,73,73,72,72
  };
  const int n4w3b2r6[] = {
    1000, // Capacity
    500, // Number of items
    // Size of items (sorted)
    210,210,210,209,209,209,209,208,208,207,207,206,206,206,205,205,
    204,204,204,204,202,202,202,202,202,201,201,200,200,200,200,200,
    199,199,199,198,198,197,197,197,197,197,197,197,196,194,194,193,
    193,193,193,193,192,192,192,192,191,191,191,190,190,190,190,190,
    190,190,189,188,188,188,188,188,187,187,187,187,187,187,186,186,
    186,186,185,185,185,184,184,183,183,183,183,183,182,182,182,181,
    181,181,180,180,180,180,179,179,179,179,178,178,178,177,177,177,
    176,176,176,175,175,175,175,174,174,174,174,173,173,173,173,173,
    171,171,171,170,170,169,169,169,169,169,168,167,167,167,167,167,
    167,167,166,166,166,166,166,166,166,166,166,165,165,165,165,164,
    164,164,164,163,163,162,162,162,161,161,161,161,161,161,161,161,
    160,160,160,160,159,159,159,158,158,157,156,156,156,156,156,156,
    155,155,155,154,154,154,154,154,153,153,153,153,153,153,153,153,
    152,152,152,152,152,152,152,152,151,151,150,150,149,149,149,148,
    148,148,147,147,146,146,146,146,146,145,145,145,145,145,145,145,
    144,144,144,144,144,143,143,143,143,142,142,141,141,141,141,141,
    141,140,140,140,140,140,140,139,139,139,139,139,139,139,138,138,
    138,138,138,138,138,138,138,137,137,137,136,136,135,135,135,135,
    134,134,134,134,133,133,133,133,132,132,132,132,132,132,132,131,
    131,130,130,129,129,129,128,127,127,126,126,124,124,124,123,123,
    123,122,122,122,121,121,121,120,120,120,119,119,119,119,119,118,
    118,118,117,117,117,117,116,116,116,115,115,114,114,114,114,114,
    114,114,114,114,113,113,113,112,112,111,111,111,111,111,110,110,
    110,110,109,109,109,108,108,108,107,106,106,106,105,105,105,103,
    103,102,100,100,100,99,99,99,98,98,98,97,97,96,96,96,96,95,95,
    95,95,95,95,95,95,95,95,95,94,94,94,93,93,93,93,92,92,92,92,92,
    92,92,92,91,91,91,91,91,90,90,90,90,90,90,90,89,89,89,88,88,87,
    87,87,87,87,86,86,86,86,86,86,85,85,85,85,85,85,84,84,84,83,83,
    83,82,82,82,82,82,80,80,80,79,79,79,78,78,78,78,77,77,77,76,76,
    75,75,75,75,74,74,74,74,74,74,74,74,73
  };
  const int n4w3b2r7[] = {
    1000, // Capacity
    500, // Number of items
    // Size of items (sorted)
    210,210,210,209,209,209,209,208,208,208,207,207,206,206,206,206,
    206,205,205,205,205,205,205,205,205,204,204,204,204,203,203,202,
    202,202,202,202,202,201,201,201,201,201,200,199,199,199,198,198,
    198,198,198,197,197,197,196,196,196,196,196,195,195,195,195,194,
    194,193,193,193,193,193,193,192,191,191,191,191,190,190,190,189,
    189,189,189,189,189,188,188,188,188,187,187,187,187,187,187,186,
    186,186,186,185,185,185,184,184,184,184,184,184,183,183,182,182,
    182,182,182,181,181,180,180,180,180,179,179,179,179,177,177,177,
    177,177,177,177,176,176,176,175,175,174,173,173,173,173,173,172,
    171,171,171,171,171,171,171,171,171,170,169,169,169,169,169,168,
    167,167,167,167,166,166,166,166,166,166,165,165,164,164,163,163,
    163,163,162,162,162,161,161,161,161,161,161,160,160,158,158,157,
    157,157,157,157,157,156,156,156,155,155,155,155,155,154,154,153,
    152,152,152,152,151,151,150,149,149,148,148,147,146,146,146,145,
    145,145,144,144,144,143,143,143,143,142,141,141,141,141,141,140,
    140,140,140,139,139,139,138,138,138,137,137,137,137,137,137,136,
    136,135,135,134,134,133,133,132,131,131,131,131,130,130,130,130,
    130,129,129,129,128,128,127,127,127,127,126,125,125,125,124,124,
    124,123,123,123,122,122,122,121,121,121,121,120,120,120,120,120,
    119,119,119,119,118,118,118,118,117,117,117,117,116,116,116,116,
    116,115,115,115,114,114,114,114,114,113,113,113,113,113,112,112,
    111,111,111,111,111,111,110,110,110,110,110,109,109,109,108,108,
    108,107,107,107,107,107,107,107,106,106,106,106,106,106,105,105,
    105,105,105,105,105,104,104,103,103,103,103,103,102,102,101,101,
    101,101,100,100,100,100,100,100,99,99,99,99,99,98,98,98,98,97,
    96,96,96,96,95,95,95,94,94,94,94,94,94,93,93,93,93,93,93,93,92,
    92,92,91,91,91,91,90,88,88,88,88,87,87,86,86,86,85,85,85,85,84,
    84,84,84,83,83,83,83,83,82,82,82,82,82,82,81,81,81,80,79,79,78,
    78,78,77,77,77,76,76,76,76,76,76,76,75,75,75,75,75,75,75,74,74,
    74,74,74,73,73,73,73,72,72,72,72,72,72,72
  };
  const int n4w3b2r8[] = {
    1000, // Capacity
    500, // Number of items
    // Size of items (sorted)
    210,210,210,210,209,209,208,208,208,208,208,207,207,207,207,206,
    206,205,205,205,205,205,205,204,204,204,204,203,203,203,202,202,
    201,201,201,201,201,200,200,200,200,199,199,199,199,199,199,199,
    198,198,198,198,198,197,197,197,197,197,197,196,196,196,196,196,
    195,195,195,194,194,194,193,193,192,192,192,192,192,191,191,191,
    190,190,189,189,189,189,188,188,188,187,187,187,187,186,186,186,
    186,185,185,185,185,184,184,184,184,184,184,183,183,182,182,181,
    181,181,181,180,180,180,180,179,179,179,178,178,178,178,178,177,
    176,176,175,175,175,174,173,173,173,172,172,171,171,170,170,170,
    170,169,169,169,169,169,168,168,167,167,167,167,167,167,166,166,
    166,166,166,165,164,164,164,163,163,163,162,162,161,161,160,160,
    160,160,160,160,159,159,159,158,158,158,158,158,158,157,157,156,
    156,155,155,155,155,154,153,153,153,153,152,152,152,152,152,152,
    152,151,151,151,151,150,150,150,150,150,149,149,149,149,149,149,
    148,148,148,148,147,147,147,146,146,145,144,144,144,144,144,144,
    144,144,144,144,143,143,143,143,142,142,141,141,141,141,141,141,
    140,140,140,139,139,139,139,139,139,139,139,138,138,137,137,137,
    137,137,137,136,136,136,136,135,135,135,135,135,134,134,134,134,
    134,133,133,132,132,131,131,131,131,130,130,130,129,128,128,128,
    127,126,126,126,126,126,126,125,125,125,125,125,124,124,123,123,
    123,123,123,123,123,123,122,122,122,122,121,121,121,121,120,120,
    120,120,120,120,120,120,119,119,119,119,119,118,118,118,117,116,
    116,116,116,116,115,115,114,114,114,114,113,113,113,113,113,112,
    112,112,112,111,111,111,110,110,109,109,109,109,108,107,107,107,
    107,106,106,106,106,105,104,104,104,104,104,103,103,103,103,103,
    103,102,102,102,102,102,101,101,101,100,100,100,99,99,99,98,98,
    98,98,97,97,96,96,96,96,96,96,96,94,94,94,94,93,93,92,92,92,91,
    91,91,91,91,90,90,89,89,89,89,88,88,87,87,86,86,86,86,86,86,85,
    85,85,85,85,84,84,83,83,83,82,82,81,80,79,79,79,78,78,78,78,78,
    78,77,77,76,76,76,75,75,74,74,74,74,74,74,73,72,72,72,72,72
  };
  const int n4w3b2r9[] = {
    1000, // Capacity
    500, // Number of items
    // Size of items (sorted)
    210,209,209,209,209,208,208,208,208,208,207,206,206,206,205,205,
    205,204,204,204,203,203,203,203,202,202,202,202,202,202,201,201,
    200,200,200,199,199,198,198,198,198,197,196,196,195,195,195,194,
    194,194,194,194,193,193,193,193,193,193,193,192,191,191,191,190,
    190,190,189,189,189,189,189,189,189,189,188,188,188,188,187,187,
    187,187,187,187,187,187,186,186,186,185,185,185,185,185,184,184,
    184,183,183,183,183,181,181,180,180,180,179,179,178,178,178,177,
    177,177,176,176,175,175,175,175,175,175,174,174,174,174,174,174,
    174,173,173,173,172,172,172,171,171,171,171,171,171,171,170,170,
    170,169,169,169,169,169,169,169,168,168,168,167,167,167,167,166,
    166,166,166,165,165,165,165,163,163,162,161,161,161,160,159,159,
    158,158,158,158,158,158,157,157,157,157,157,157,156,156,156,156,
    154,154,154,154,153,153,153,153,153,152,152,152,152,151,150,150,
    150,150,150,149,149,149,149,149,149,148,148,148,148,147,147,147,
    147,147,147,147,147,146,146,146,145,145,145,145,145,145,145,144,
    144,144,144,144,144,143,143,142,142,142,142,142,141,140,139,139,
    139,139,139,138,138,138,137,137,136,136,136,135,135,135,135,134,
    134,133,133,132,132,132,132,131,131,131,131,131,130,129,128,128,
    128,128,128,127,127,127,127,127,125,125,124,124,124,123,123,122,
    122,122,122,122,122,121,121,121,121,121,120,120,120,120,119,119,
    118,118,118,118,117,117,116,116,116,116,115,115,115,114,114,113,
    113,113,113,113,113,112,112,112,112,111,111,111,110,110,109,109,
    109,109,108,108,108,108,108,107,107,107,107,107,106,106,106,106,
    106,105,105,104,104,104,104,104,103,103,103,102,102,102,102,101,
    101,100,100,100,100,99,99,99,99,98,98,98,98,98,97,97,97,96,96,
    96,96,96,95,95,95,95,94,94,94,93,93,93,93,92,92,92,92,92,91,91,
    90,90,90,90,89,89,89,89,88,88,87,87,87,86,86,86,86,86,86,85,85,
    84,84,84,84,83,83,83,83,83,83,82,82,82,82,82,81,81,80,80,80,80,
    80,79,79,79,79,78,78,78,78,78,78,77,77,77,77,76,76,76,75,75,75,
    75,74,74,74,74,74,73,73,73,72,72,72,72
  };
  const int n4w3b3r0[] = {
    1000, // Capacity
    500, // Number of items
    // Size of items (sorted)
    266,266,266,266,265,263,263,261,261,261,260,260,260,260,259,259,
    259,258,257,257,257,257,256,256,256,255,255,254,253,253,253,253,
    253,252,252,251,250,249,249,249,249,247,247,246,246,245,245,244,
    244,244,243,242,242,240,240,240,239,239,239,239,238,237,237,237,
    236,236,236,235,235,234,234,234,234,234,233,233,233,232,232,232,
    230,230,229,229,227,227,227,227,226,226,226,226,224,224,224,224,
    223,223,223,223,223,222,222,221,221,220,219,219,219,218,218,218,
    217,217,217,216,216,216,215,214,214,214,213,213,211,210,210,209,
    209,209,208,208,207,206,206,206,205,205,203,203,203,203,202,202,
    201,201,200,199,199,199,197,197,197,196,195,195,193,192,192,192,
    191,191,191,190,190,189,188,187,185,185,185,184,184,183,183,182,
    182,182,182,182,181,181,181,181,181,180,180,180,180,180,180,179,
    179,178,177,177,176,176,176,174,173,173,172,172,171,171,170,170,
    170,169,169,169,168,168,168,167,165,164,164,164,162,162,162,162,
    162,161,160,158,157,156,156,155,155,154,153,152,152,150,150,150,
    149,149,149,146,146,146,146,145,145,144,144,144,143,142,142,142,
    141,139,138,138,138,138,137,135,134,134,134,133,132,132,132,131,
    131,131,131,131,131,130,128,128,127,127,125,125,125,122,122,122,
    122,122,122,121,121,120,120,120,120,120,120,119,119,119,118,118,
    118,117,117,116,116,116,115,114,114,114,113,112,111,111,111,110,
    110,109,108,108,107,105,105,104,101,101,101,101,100,100,100,100,
    100,100,99,97,97,97,96,95,95,93,91,91,91,90,90,90,89,89,89,88,
    87,87,86,86,85,85,84,81,81,80,79,79,77,77,77,76,76,76,75,75,74,
    74,73,73,72,72,72,71,71,70,70,69,69,69,68,68,68,68,68,67,67,66,
    66,66,66,66,66,66,66,65,65,64,64,64,63,62,62,61,59,59,58,57,57,
    57,57,56,56,55,55,54,54,53,53,53,53,53,52,52,51,51,51,51,51,50,
    49,49,49,49,49,47,47,47,46,46,45,42,41,41,40,39,37,37,37,37,36,
    36,36,34,34,34,33,33,33,33,32,32,31,30,29,29,27,27,26,26,25,25,
    25,23,23,22,22,22,21,21,21,20,20,19,19,19,18,17,16,16
  };
  const int n4w3b3r1[] = {
    1000, // Capacity
    500, // Number of items
    // Size of items (sorted)
    265,265,264,264,264,262,262,261,259,259,258,256,255,255,254,254,
    254,253,252,251,250,250,250,250,250,248,248,247,247,247,246,246,
    246,245,244,243,243,243,242,242,242,242,242,242,242,240,240,240,
    240,237,237,236,236,236,235,234,233,233,232,232,232,231,230,230,
    230,230,229,229,228,227,227,226,226,225,225,225,223,222,222,222,
    222,222,221,221,220,220,220,220,220,219,219,219,219,219,219,218,
    218,218,217,217,215,215,215,215,215,215,214,213,213,213,212,212,
    211,211,209,209,208,207,206,206,205,205,204,204,204,204,204,204,
    204,203,202,201,200,200,199,199,199,199,198,196,196,195,194,193,
    193,192,192,191,191,191,189,189,189,189,189,189,188,188,187,186,
    186,185,185,184,184,183,183,182,182,181,181,181,180,179,178,178,
    178,178,178,177,177,177,176,175,175,175,173,173,173,172,171,171,
    171,171,170,170,168,168,167,166,166,166,166,164,164,164,163,163,
    162,162,162,161,161,160,159,159,159,158,157,157,156,155,155,155,
    153,152,152,152,151,151,151,151,149,149,149,149,148,148,148,147,
    147,147,146,146,146,145,145,145,144,143,143,142,141,141,141,141,
    141,140,140,140,139,139,138,138,138,136,135,135,135,135,135,133,
    133,132,132,132,132,131,131,131,131,130,130,129,129,129,128,128,
    128,128,128,127,127,127,125,125,125,123,123,122,121,120,120,117,
    117,116,115,114,114,110,110,109,109,109,108,108,106,105,105,105,
    104,104,104,103,101,101,101,101,101,100,100,99,99,99,99,98,97,
    97,96,96,94,94,94,93,93,93,92,92,91,91,91,91,91,91,90,90,89,89,
    88,87,87,87,87,87,87,86,85,84,84,83,82,81,81,81,80,80,79,79,78,
    78,76,75,74,74,74,73,73,73,72,72,71,70,70,70,70,69,69,68,68,67,
    67,66,65,64,64,64,62,62,61,61,60,59,58,58,57,56,55,55,54,53,53,
    53,53,51,51,51,51,51,51,50,50,50,49,49,49,48,48,48,47,47,47,46,
    45,45,44,43,43,42,42,42,42,42,40,39,39,38,37,37,37,36,35,34,33,
    32,32,32,31,31,31,30,28,28,28,27,27,26,26,26,25,25,24,24,22,21,
    21,21,21,20,20,18,18,18,18,17,17,17,17,16,16,16
  };
  const int n4w3b3r2[] = {
    1000, // Capacity
    500, // Number of items
    // Size of items (sorted)
    266,266,265,265,265,263,263,262,262,262,262,262,261,260,260,259,
    258,258,257,257,257,257,255,254,254,253,252,252,252,252,250,249,
    249,248,248,247,246,246,245,245,244,244,243,243,243,242,242,241,
    241,240,240,240,240,240,240,239,239,239,239,239,238,238,237,237,
    236,236,235,234,234,233,232,231,230,229,228,228,227,227,227,226,
    226,226,225,225,225,225,225,224,223,223,223,223,223,223,222,222,
    222,221,221,220,218,217,217,215,215,215,215,214,214,214,213,213,
    213,212,212,212,211,210,210,210,208,208,207,207,207,206,205,205,
    204,204,203,203,203,203,201,201,201,200,200,200,200,200,199,198,
    198,197,197,196,195,195,195,194,194,194,194,194,193,193,193,193,
    191,191,190,190,190,190,190,189,189,189,188,187,187,186,185,185,
    185,185,184,183,182,181,181,180,180,180,179,179,178,177,177,177,
    176,176,175,174,174,174,174,173,172,172,171,170,170,170,170,169,
    168,168,167,166,165,163,163,162,162,161,161,161,161,160,159,159,
    158,158,158,158,157,157,156,155,154,154,153,153,153,153,153,150,
    150,149,149,148,148,146,146,145,145,144,143,143,142,142,141,141,
    141,140,140,139,139,138,138,137,137,137,137,136,136,136,136,136,
    135,135,135,134,134,133,132,131,131,131,131,130,130,128,128,127,
    127,127,127,127,125,124,124,124,124,122,122,122,121,121,121,121,
    121,121,121,121,120,118,118,118,117,117,117,116,116,115,114,113,
    113,111,111,108,108,107,106,106,104,104,103,103,102,102,102,101,
    101,100,100,100,100,99,98,98,97,94,94,93,93,92,92,92,90,90,88,
    88,88,87,86,86,85,85,84,84,84,83,82,81,81,80,79,79,79,79,78,78,
    78,76,76,76,75,73,72,72,71,71,71,70,69,69,68,67,67,67,66,65,64,
    64,63,63,62,62,62,58,58,57,57,57,57,56,55,55,54,54,53,53,52,52,
    50,50,50,50,50,49,48,48,48,47,47,47,47,46,46,46,45,45,45,45,44,
    43,42,41,41,40,40,39,38,38,38,37,37,37,36,36,36,35,35,34,34,34,
    33,32,31,31,31,31,31,30,30,30,30,29,29,29,29,29,29,28,27,27,27,
    27,26,26,25,24,23,23,22,20,20,19,18,18,17,17,17,16,16,16
  };
  const int n4w3b3r3[] = {
    1000, // Capacity
    500, // Number of items
    // Size of items (sorted)
    266,265,265,265,265,263,263,262,261,261,260,259,259,257,257,257,
    255,255,255,255,255,254,254,253,252,252,251,251,251,251,248,247,
    247,246,246,246,246,246,245,244,243,242,242,242,242,241,240,239,
    239,239,237,237,237,237,237,237,237,236,236,235,235,235,235,235,
    234,234,232,232,232,232,230,230,230,230,229,229,229,229,228,228,
    227,227,227,226,225,224,224,224,223,223,223,223,223,223,222,220,
    220,219,219,219,218,218,218,218,217,216,216,216,215,215,214,213,
    213,212,211,211,210,210,209,209,209,208,205,205,204,204,203,203,
    201,201,201,200,199,198,198,198,197,197,197,196,196,195,195,193,
    193,192,192,191,191,191,191,191,190,190,187,187,187,187,186,186,
    185,185,185,184,184,183,183,182,182,182,182,181,181,180,180,180,
    179,178,178,177,176,176,174,174,174,173,173,172,172,172,171,171,
    171,170,170,169,168,166,166,166,166,166,165,165,165,165,165,164,
    163,163,162,162,161,161,160,160,159,159,159,158,157,157,157,156,
    156,156,155,155,155,155,155,154,154,153,153,152,150,150,149,148,
    148,147,146,146,146,144,143,143,143,143,143,142,141,141,141,141,
    140,140,140,139,136,136,135,134,132,131,131,131,130,130,130,130,
    129,129,129,129,128,127,126,125,123,122,122,121,121,121,120,120,
    119,119,119,118,118,117,117,116,115,114,114,113,113,113,112,112,
    111,111,111,110,110,110,110,109,109,109,108,108,107,107,107,106,
    105,105,105,105,104,101,100,100,100,100,99,99,99,98,97,95,95,
    95,94,93,92,92,92,92,91,91,90,90,89,88,88,87,87,87,87,87,86,86,
    86,85,85,83,83,83,83,82,82,82,80,80,79,79,78,78,78,78,77,77,77,
    76,76,76,75,75,75,74,74,73,72,72,71,71,71,71,70,70,69,69,68,67,
    65,65,65,64,63,62,62,62,61,61,61,60,59,59,59,59,58,58,58,58,57,
    56,56,55,55,54,53,53,53,52,52,52,51,51,50,50,50,50,49,46,46,46,
    45,45,45,43,43,43,41,40,40,38,37,37,37,37,36,35,33,33,32,32,32,
    32,32,32,32,32,31,31,31,30,30,29,28,27,26,26,26,26,24,24,23,22,
    22,21,21,21,21,20,20,20,19,19,19,19,18,17,17,16
  };
  const int n4w3b3r4[] = {
    1000, // Capacity
    500, // Number of items
    // Size of items (sorted)
    266,266,266,266,266,263,262,262,262,262,261,261,261,261,261,260,
    260,260,260,259,258,258,258,257,257,257,257,256,256,255,255,254,
    254,253,253,252,252,251,251,251,251,250,250,249,249,249,248,248,
    247,247,247,246,245,245,243,243,242,241,240,240,239,238,238,238,
    237,237,237,236,236,235,235,235,234,234,233,233,233,233,233,232,
    232,231,231,230,230,228,228,228,228,227,226,226,226,225,225,224,
    224,223,223,221,221,221,220,220,220,220,218,218,217,217,216,215,
    215,215,215,214,214,214,213,213,213,213,211,211,211,211,210,210,
    210,209,209,207,206,205,204,203,203,203,202,201,201,201,200,200,
    200,199,198,197,195,195,195,195,194,194,193,193,192,192,191,191,
    190,189,189,189,188,188,186,186,186,186,185,184,183,182,182,181,
    180,179,178,177,177,176,175,175,175,175,174,174,174,173,173,172,
    172,171,171,171,171,169,169,167,167,166,165,165,165,165,164,164,
    163,162,162,161,161,161,160,160,159,159,158,158,157,156,156,156,
    156,156,156,155,154,154,154,154,153,152,152,151,151,151,151,151,
    150,150,150,150,149,149,149,147,147,147,146,145,145,144,144,143,
    142,142,142,141,141,141,140,137,136,136,134,134,134,133,132,132,
    132,130,130,129,129,129,128,128,127,127,127,126,125,125,124,123,
    123,123,123,122,122,121,120,120,119,119,118,118,118,118,115,115,
    114,114,114,113,112,112,111,111,110,110,110,110,109,109,108,108,
    108,107,105,104,104,104,103,103,102,102,102,102,102,102,101,101,
    101,101,100,99,99,99,98,98,98,97,96,95,95,95,94,94,93,92,92,91,
    91,91,91,91,90,90,89,89,88,87,87,87,86,86,85,84,84,83,82,82,81,
    81,81,81,80,80,79,78,78,78,78,77,77,76,76,75,74,74,74,73,71,71,
    71,71,71,70,70,69,68,68,67,66,66,65,65,64,64,64,63,63,61,61,61,
    61,60,59,58,58,58,57,57,56,54,54,54,53,52,52,52,51,51,50,50,49,
    48,48,48,47,47,47,46,46,44,44,44,43,42,42,41,40,38,38,38,38,37,
    36,36,36,36,35,35,35,34,32,31,31,28,27,27,27,27,26,26,25,25,25,
    25,24,24,23,23,23,23,22,22,21,21,20,19,19,19,19,19,17
  };
  const int n4w3b3r5[] = {
    1000, // Capacity
    500, // Number of items
    // Size of items (sorted)
    266,266,266,266,266,265,264,263,263,262,262,262,262,262,262,262,
    261,261,261,261,260,260,260,259,259,258,256,256,256,255,255,253,
    252,252,252,252,251,251,250,248,248,247,247,247,247,246,246,246,
    245,245,245,244,244,243,242,242,241,241,241,240,240,240,239,239,
    238,238,238,236,236,235,235,235,234,234,233,233,233,232,232,231,
    229,229,229,228,228,227,227,227,226,226,226,225,225,223,221,221,
    221,221,221,220,220,220,219,218,218,218,216,215,215,215,214,214,
    213,213,212,212,211,211,211,210,210,209,209,209,209,209,207,207,
    206,205,205,205,205,204,204,204,203,202,202,201,199,199,198,198,
    198,198,198,197,196,196,195,195,195,194,194,193,193,193,193,192,
    192,191,191,191,191,190,190,189,189,188,188,188,188,187,187,186,
    186,186,185,185,183,183,182,182,182,181,181,180,180,180,178,178,
    178,177,176,176,176,176,175,175,175,174,174,174,173,173,172,171,
    171,171,171,170,169,168,168,168,167,167,165,165,165,164,163,161,
    161,161,160,159,159,158,158,157,156,155,155,155,154,154,154,153,
    153,152,151,151,149,149,148,147,146,144,143,143,143,142,142,142,
    141,139,139,139,139,138,137,137,136,136,136,135,135,134,134,133,
    133,132,132,132,131,131,130,129,128,128,127,127,127,126,125,125,
    125,125,124,124,123,122,122,122,122,122,122,121,121,121,120,118,
    118,117,117,116,116,116,116,114,114,113,113,113,112,112,112,112,
    111,111,111,111,110,109,109,109,108,108,107,107,105,105,105,105,
    105,104,104,103,103,103,102,102,102,101,100,100,100,100,100,99,
    99,98,98,98,97,95,95,94,94,94,93,91,91,90,90,90,90,89,88,88,88,
    88,87,86,86,85,85,84,84,84,83,83,83,80,80,80,78,78,76,76,75,75,
    74,74,73,73,72,71,71,70,69,69,69,68,68,68,67,67,66,65,63,63,61,
    61,60,59,59,59,59,59,58,58,58,58,57,56,56,54,52,52,52,51,49,49,
    49,47,46,46,46,45,45,45,45,45,44,44,44,43,43,43,42,41,41,41,40,
    39,39,36,35,33,33,33,33,32,32,32,32,31,31,30,29,28,28,28,28,27,
    26,26,25,25,25,25,24,24,22,22,21,20,20,20,20,20,19,18,18,17,16,
    16
  };
  const int n4w3b3r6[] = {
    1000, // Capacity
    500, // Number of items
    // Size of items (sorted)
    266,265,265,265,264,263,262,260,260,260,259,259,258,258,258,257,
    257,256,256,255,253,253,252,252,252,252,252,251,251,250,249,249,
    248,247,246,246,246,246,245,244,244,244,243,243,242,241,240,237,
    237,237,237,236,236,235,233,233,232,232,230,229,228,228,228,228,
    228,228,227,226,226,225,225,225,225,224,224,224,224,224,224,223,
    222,222,222,221,221,219,219,219,219,219,218,218,218,216,215,215,
    215,215,215,214,214,214,214,214,213,213,212,212,212,212,209,209,
    209,208,208,208,208,207,207,207,207,206,205,205,205,205,204,204,
    203,203,202,202,201,200,199,199,199,198,197,197,197,196,195,195,
    194,194,193,193,192,192,191,191,190,190,189,189,189,189,188,188,
    187,186,186,186,185,185,185,184,183,183,183,183,182,182,182,181,
    181,180,180,179,179,178,178,178,177,176,176,175,175,173,173,172,
    171,171,170,170,169,169,169,168,168,168,167,165,165,165,164,164,
    164,163,163,163,162,161,161,161,160,160,159,159,159,158,157,156,
    155,155,155,155,155,155,155,154,154,154,154,154,153,153,153,153,
    152,152,152,151,151,151,150,150,150,150,150,150,149,149,148,147,
    146,146,145,144,144,143,143,143,143,143,141,141,141,141,140,140,
    140,139,139,139,139,139,138,136,136,135,135,134,134,132,131,129,
    129,129,129,129,129,128,127,127,126,126,126,125,125,125,125,125,
    124,124,123,122,122,121,121,121,120,120,120,120,119,119,118,117,
    116,116,116,116,115,115,115,115,114,112,112,111,111,110,108,107,
    106,105,105,104,104,104,102,102,101,101,101,101,100,100,100,99,
    99,98,97,97,97,97,95,95,94,94,93,93,92,92,92,92,92,91,91,90,89,
    89,89,88,88,88,88,87,86,86,85,84,83,82,81,81,80,79,78,77,77,77,
    77,77,77,76,75,74,74,73,73,73,73,72,72,72,72,72,72,72,72,72,71,
    69,69,68,67,67,67,66,66,65,65,65,65,64,63,63,61,61,60,58,56,56,
    55,54,53,52,52,51,50,50,50,49,48,47,47,47,46,46,45,44,43,43,42,
    42,41,40,40,40,39,39,35,35,34,33,33,32,32,32,32,31,31,29,29,28,
    28,28,27,27,26,26,26,25,25,25,24,23,22,19,19,19,19,18,17,17,16,
    16
  };
  const int n4w3b3r7[] = {
    1000, // Capacity
    500, // Number of items
    // Size of items (sorted)
    265,265,265,265,263,263,263,262,262,261,261,260,260,258,258,258,
    258,258,257,257,257,257,257,256,256,255,255,254,254,254,253,253,
    253,253,253,252,252,251,251,250,250,250,249,248,248,248,248,247,
    247,247,246,246,246,246,245,243,243,242,241,241,241,240,240,240,
    240,238,238,238,238,238,238,238,238,238,237,236,235,235,234,234,
    234,232,232,230,230,229,228,227,227,227,226,226,226,226,226,226,
    225,224,223,223,223,223,223,223,222,222,222,221,221,221,220,220,
    219,219,218,217,217,217,217,217,216,216,215,215,215,214,212,212,
    212,212,211,211,210,210,209,208,208,207,205,205,204,204,204,203,
    203,203,202,202,201,201,201,200,200,200,199,198,197,197,196,195,
    195,194,194,194,194,194,194,193,193,192,190,190,190,190,190,189,
    189,189,189,189,188,188,188,187,187,186,186,185,185,185,185,184,
    184,183,183,182,181,181,180,180,179,179,177,176,176,176,175,174,
    174,173,167,167,166,166,165,165,165,165,164,164,164,163,161,160,
    160,159,159,159,156,156,155,155,154,154,154,153,152,152,152,150,
    150,150,149,147,146,145,144,144,144,144,143,143,142,142,142,141,
    140,139,139,138,138,138,138,137,136,135,135,135,134,134,134,133,
    132,132,132,132,131,131,130,130,130,130,129,128,128,128,128,128,
    128,127,127,127,127,127,125,124,124,124,124,123,123,123,122,121,
    121,121,121,120,120,119,119,118,118,117,117,116,116,115,115,114,
    114,114,113,112,112,112,112,111,111,111,111,110,109,108,108,108,
    107,107,107,106,105,105,104,102,102,101,101,101,99,98,98,97,97,
    97,97,96,95,94,94,93,91,91,91,91,90,90,90,89,88,88,88,88,88,87,
    86,86,85,85,85,85,84,84,84,82,82,82,81,81,81,81,80,80,79,79,78,
    78,78,74,74,74,74,72,71,70,70,69,68,68,67,65,65,65,65,63,61,61,
    61,61,60,60,59,58,58,58,58,58,57,56,56,56,55,55,54,54,54,54,53,
    53,51,51,48,48,47,47,46,46,45,44,44,43,42,42,42,41,41,41,40,39,
    38,37,36,35,34,33,32,32,32,32,31,31,30,28,28,27,27,27,27,26,26,
    24,24,23,22,21,20,20,20,19,19,19,18,18,18,18,17,17,16,16,16,16
  };
  const int n4w3b3r8[] = {
    1000, // Capacity
    500, // Number of items
    // Size of items (sorted)
    266,266,265,264,264,264,263,263,261,261,261,260,259,259,259,259,
    258,257,256,255,254,254,252,252,252,251,251,251,250,250,248,246,
    246,245,244,243,243,243,242,241,241,241,241,241,240,240,240,240,
    238,238,238,237,236,236,235,235,235,235,234,234,234,234,234,233,
    233,232,232,232,232,231,231,230,230,230,230,229,228,227,226,226,
    226,226,226,225,225,225,224,223,223,223,223,223,222,221,220,220,
    218,218,217,216,215,214,214,213,213,213,213,212,212,212,212,212,
    211,211,210,209,209,209,209,209,209,208,208,208,207,206,206,206,
    204,204,203,203,203,202,202,202,201,201,201,200,200,199,199,199,
    199,199,199,198,198,197,197,196,196,196,195,195,193,192,192,192,
    191,191,189,189,188,188,188,188,187,186,185,185,184,183,183,182,
    181,181,181,181,180,179,179,178,178,178,178,177,177,176,174,174,
    174,174,174,173,173,173,172,172,169,169,168,168,168,167,167,166,
    165,164,163,163,163,162,162,162,161,161,161,1