/* -*- mode: C++; c-basic-offset: 2; indent-tabs-mode: nil -*- */
/*
 *  Main authors:
 *     Christian Schulte <schulte@gecode.org>
 *     Mikael Lagerkvist <lagerkvist@gecode.org>
 *
 *  Copyright:
 *     Christian Schulte, 2008
 *     Mikael Lagerkvist, 2008
 *
 *  Last modified:
 *     $Date: 2009-03-05 12:06:40 +0100 (Thu, 05 Mar 2009) $ by $Author: zayenz $
 *     $Revision: 8370 $
 *
 *  This file is part of Gecode, the generic constraint
 *  development environment:
 *     http://www.gecode.org
 *
 *  Permission is hereby granted, free of charge, to any person obtaining
 *  a copy of this software and associated documentation files (the
 *  "Software"), to deal in the Software without restriction, including
 *  without limitation the rights to use, copy, modify, merge, publish,
 *  distribute, sublicense, and/or sell copies of the Software, and to
 *  permit persons to whom the Software is furnished to do so, subject to
 *  the following conditions:
 *
 *  The above copyright notice and this permission notice shall be
 *  included in all copies or substantial portions of the Software.
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 *  EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 *  MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 *  NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
 *  LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
 *  OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
 *  WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 */

#include <gecode/kernel.hh>

namespace Gecode {

  const VarBranchOptions VarBranchOptions::def;

  const ValBranchOptions ValBranchOptions::def;

  const TieBreakVarBranchOptions TieBreakVarBranchOptions::def;


  /*
   * Function branching
   */

  /// Branching for calling a function
  class GECODE_KERNEL_EXPORT FunctionBranch : public Branching {
  protected:
    /// Minimal branching description storing no information
    class GECODE_KERNEL_EXPORT Description : public BranchingDesc {
    public:
      /// Initialize description for branching \a b, number of alternatives \a a.
      Description(const Branching& b, unsigned int a) : BranchingDesc(b,a) {}
      /// Report size occupied
      virtual size_t size(void) const { return sizeof(Description); }
    };
    /// Function to call
    void (*f)(Space&);
    /// Call function just once
    bool done;
    /// Construct branching
    FunctionBranch(Space& home, void (*f0)(Space&))
      : Branching(home), f(f0), done(false) {}
    /// Copy constructor
    FunctionBranch(Space& home, bool share, FunctionBranch& b)
      : Branching(home,share,b), f(b.f), done(b.done) {}
  public:
    /// Check status of branching, return true if alternatives left
    virtual bool status(const Space&) const {
      return !done;
    }
    /// Return branching description
    virtual const BranchingDesc* description(Space&) {
      assert(!done);
      return new Description(*this,1);
    }
    /// Perform commit
    virtual ExecStatus 
    commit(Space& home, const BranchingDesc&, unsigned int) {
      done = true;
      f(home);
      return home.failed() ? ES_FAILED : ES_OK;
    }
    /// Copy branching
    virtual Actor* copy(Space& home, bool share) {
      return new (home) FunctionBranch(home,share,*this);
    }
    /// Post branching
    static void post(Space& home, void (*f)(Space&)) {
      (void) new (home) FunctionBranch(home,f);
    }
  };


  void
  branch(Space& home, void (*f)(Space& home)) {
    if (home.failed())
      return;
    FunctionBranch::post(home,f);
  }

}

// STATISTICS: kernel-branch
