/* -*- mode: C++; c-basic-offset: 2; indent-tabs-mode: nil -*- */
/*
 *  Main authors:
 *     Christian Schulte <schulte@gecode.org>
 *
 *  Copyright:
 *     Christian Schulte, 2007
 *
 *  Last modified:
 *     $Date: 2009-01-20 23:44:27 +0100 (Tue, 20 Jan 2009) $ by $Author: schulte $
 *     $Revision: 8082 $
 *
 *  This file is part of Gecode, the generic constraint
 *  development environment:
 *     http://www.gecode.org
 *
 *  Permission is hereby granted, free of charge, to any person obtaining
 *  a copy of this software and associated documentation files (the
 *  "Software"), to deal in the Software without restriction, including
 *  without limitation the rights to use, copy, modify, merge, publish,
 *  distribute, sublicense, and/or sell copies of the Software, and to
 *  permit persons to whom the Software is furnished to do so, subject to
 *  the following conditions:
 *
 *  The above copyright notice and this permission notice shall be
 *  included in all copies or substantial portions of the Software.
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 *  EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 *  MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 *  NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
 *  LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
 *  OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
 *  WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 */

namespace Gecode { namespace Iter { namespace Values {

  /**
   * \brief Value iterator for pointwise minus of a value iterator
   *
   * This iterator in effect changes the order of how values
   * are iterated: the first values of the input iterator defines
   * the last value of the Minus iterator. Upon initialization
   * all values of the input iterator are stored in an array
   * which later allows iteration in inverse direction.
   *
   * \ingroup FuncIterValues
   */

  template <class I>
  class Minus {
  private:
    /// Check that \a I is a value iterator;
    IsValueIter<I> _checkI;
  private:
    /// %Values stored
    SharedArray<int> v;
    /// Current value position
    int c;
  public:
    /// \name Constructors and initialization
    //@{
    /// Default constructor
    Minus(void);
    /// Initialize with values from \a i
    Minus(I& i);
    /// Initialize with values from \a i
    void init(I& i);
    //@}

    /// \name Iteration control
    //@{
    /// Test whether iterator is still at a value or done
    bool operator ()(void) const;
    /// Move iterator to next value (if possible)
    void operator ++(void);
    //@}

    /// \name Value access
    //@{
    /// Return current value
    int val(void) const;
    //@}
  };


  template <class I>
  forceinline
  Minus<I>::Minus(void) {}

  template <class I>
  inline void
  Minus<I>::init(I& i) {
    Support::DynamicArray<int,Heap> d(heap);
    int n=0;
    while (i()) {
      d[n] = -i.val();
      ++n; ++i;
    }
    v.init(n);
    for (int j=n; j--; )
      v[j]=d[j];
    c = n-1;
  }

  template <class I>
  forceinline
  Minus<I>::Minus(I& i) {
    init(i);
  }

  template <class I>
  forceinline void
  Minus<I>::operator ++(void) {
    c--;
  }
  template <class I>
  forceinline bool
  Minus<I>::operator ()(void) const {
    return c >= 0;
  }

  template <class I>
  forceinline int
  Minus<I>::val(void) const {
    return v[c];
  }

}}}

// STATISTICS: iter-any
