/* -*- mode: C++; c-basic-offset: 2; indent-tabs-mode: nil -*- */
/*
 *  Main authors:
 *     Patrick Pekczynski <pekczynski@ps.uni-sb.de>
 *
 *  Copyright:
 *     Patrick Pekczynski, 2004
 *
 *  Last modified: $Date: 2009-05-08 09:36:37 +0200 (Fri, 08 May 2009) $ by $Author: tack $
 *  $Revision: 9020 $
 *
 *  This file is part of Gecode, the generic constrain
 *  development environment:
 *     http://www.gecode.org
 *
 *
 *  Permission is hereby granted, free of charge, to any person obtaining
 *  a copy of this software and associated documentation files (the
 *  "Software"), to deal in the Software without restriction, including
 *  without limitation the rights to use, copy, modify, merge, publish,
 *  distribute, sublicense, and/or sell copies of the Software, and to
 *  permit persons to whom the Software is furnished to do so, subject to
 *  the following conditions:
 *
 *  The above copyright notice and this permission notice shall be
 *  included in all copies or substantial portions of the Software.
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 *  EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 *  MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 *  NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
 *  LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
 *  OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
 *  WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

namespace Gecode { namespace Int { namespace GCC {

  /**
   * \brief Tuple containing the lower and upper cardinality bounds
   *
   */
  class OccurBndsView {
  private:
    int _min;
    int _max;
    int c;
    int count;
  public:
    OccurBndsView(void);
    int min(void) const;
    int max(void) const;
    int card(void) const;
    int counter(void) const;

    void min(int);
    void max(int);
    void card(int c);
    void counter(int c);

    void init(Space& home, int min, int max, int c);
    ModEvent lq(Space& home, int n);
    ModEvent gq(Space& home, int n);
    ModEvent eq(Space& home, int n);
    bool assigned(void) const;
    bool range(void) const;
    ModEvent inc(void);

    void cancel(Space&, Propagator& , PropCond ) {}
    void subscribe(Space&, Propagator& , PropCond, bool=true) {}

    void cancel(Space&, Advisor&) {}
    void subscribe(Space&, Advisor&) {}

    void update(Space&, bool, OccurBndsView&);
  };

  forceinline
  OccurBndsView::OccurBndsView(void) {}

  forceinline int
  OccurBndsView::min(void) const {
    return _min;
  }

  forceinline int
  OccurBndsView::max(void) const {
    return _max;
  }

  forceinline int
  OccurBndsView::card(void) const {
    return c;
  }

  forceinline int
  OccurBndsView::counter(void) const {
    return count;
  }

  forceinline void
  OccurBndsView::min(int m) {
    _min = m;
  }

  forceinline void
  OccurBndsView::max(int m) {
    _max = m;
  }

  forceinline void
  OccurBndsView::card(int ca) {
    c = ca;
  }

  forceinline void
  OccurBndsView::counter(int count0) {
    count = count0;
  }

  forceinline void
  OccurBndsView::init(Space&, int min, int max, int val) {
    _min = min; _max=max;
    c = val;
    count = 0;
  }

  forceinline ModEvent
  OccurBndsView::inc(void) {
    count++;
    if (count > _max) {
      return ME_GEN_FAILED;
    } else {
      return ME_GEN_NONE;
    }
  }

  forceinline bool
  OccurBndsView::assigned(void) const {
    return _min==_max;
  }

  forceinline bool
  OccurBndsView::range(void) const {
    return true;
  }


  forceinline ModEvent
  OccurBndsView::lq(Space&, int i){
    // the maximum can be made consistent
    if (_min > i) {
      return ME_GEN_FAILED;
    } else {
      return ME_GEN_NONE;
    }
  }

  forceinline ModEvent
  OccurBndsView::gq(Space&, int i){
    // this bound is fix
    if (_max < i) {
      return ME_GEN_FAILED;
    }
    return ME_GEN_NONE;
  }

  forceinline ModEvent
  OccurBndsView::eq(Space&, int i){
    if (_min > i || _max < i) {
      return ME_GEN_FAILED;
    } else {
      return ME_GEN_NONE;
    }
  }

  forceinline void
  OccurBndsView::update(Space&, bool, OccurBndsView& oc) {
    _min = oc._min;
    _max = oc._max;
    c = oc.c;
    count = oc.count;
  }


  /**
   * \brief Return the index of v in the array
   *
   *  Complexity is \f$O(log(|k|))\f$
   */
  template <class T>
  forceinline int
  lookupValue(T& a, int v){
    int idx = -1;

    int l  = 0;
    int r  = a.size() - 1;

    if (r == 0) {
      if (a[0].card() == v) {
        return 0;
      } else {
        return -1;
      }
    }

    while ( l < r ) {
      if ( a[l].card() == v) {
        idx = l;
        break;
      }
      if ( a[r].card() == v) {
        idx = r;
        break;
      }
      int p  = (l + r) / 2;
      if ( v == a[p].card()) {
        idx = p;
        break;
      } else {
        if ( v < a[p].card()) {
          r = p;
        } else {
          l = p;
        }
      }
      if (l == r - 1) {
        break;
      }
    }

    return idx;
  }


  /**
   * \brief Card integer view
   *
   */
  class CardView : public DerivedViewBase<IntView> {
  protected:
    /// Card
    int c;
    /// Counter
    int count;
    using DerivedViewBase<IntView>::view;
  public:
    CardView(void);
    /// Initialize with integer view \a x and value \a c
    CardView(const IntView& x, int c);
    /// Initialize with integer view \a x and value \a c
    void init(const IntView& x, int c);
    void init(Space& home, int mi, int ma , int c);

    /// Return value
    int card(void) const;
    void card(int ca);

    /// Increment counter
    ModEvent inc(void);
    /// Set the counter to the number of times value \a c occurs
    void counter(int);
    /// Return the number of times value \a c occurs
    int counter(void);

    /// \name Value access
    //@{
    /// Assignment operator
    void operator =(const IntView& x);
    /// Assignment operator
    void operator =(const Gecode::Int::GCC::CardView& x);
    /// Return minimum of domain
    int min(void) const;
    /// Return maximum of domain
    int max(void) const;
    /// Return median of domain (greatest element not greater than the median)
    int med(void) const;
    /// Return assigned value (only if assigned)
    int val(void) const;
    /// Return used IntView
    IntView intview(void);
    /// Return size (cardinality) of domain
    unsigned int size(void) const;
    /// Return width of domain (distance between maximum and minimum)
    unsigned int width(void) const;
    /// Return regret of domain minimum (distance to next larger value)
    unsigned int regret_min(void) const;
    /// Return regret of domain maximum (distance to next smaller value)
    unsigned int regret_max(void) const;
    ///@}

    /// \name Domain tests
    ///@{
    /// Test whether domain is a range
    bool range(void) const;
    /// Test whether view is assigned
    bool assigned(void) const;

    /// Test whether \a n is contained in domain
    bool in(int n) const;
    /// Test whether \a n is contained in domain
    bool in(double n) const;
    ///@}

    /// \name Domain update by value
    ///@{
    /// Restrict domain values to be less or equal than \a n
    ModEvent lq(Space& home, int n);
    /// Restrict domain values to be less or equal than \a n
    ModEvent lq(Space& home, double n);
    /// Restrict domain values to be less than \a n
    ModEvent le(Space& home, int n);
    /// Restrict domain values to be less than \a n
    ModEvent le(Space& home, double n);
    /// Restrict domain values to be greater or equal than \a n
    ModEvent gq(Space& home, int n);
    /// Restrict domain values to be greater or equal than \a n
    ModEvent gq(Space& home, double n);
    /// Restrict domain values to be greater than \a n
    ModEvent gr(Space& home, int n);
    /// Restrict domain values to be greater than \a n
    ModEvent gr(Space& home, double n);
    /// Restrict domain values to be different from \a n
    ModEvent nq(Space& home, int n);
    /// Restrict domain values to be different from \a n
    ModEvent nq(Space& home, double n);
    /// Restrict domain values to be equal to \a n
    ModEvent eq(Space& home, int n);
    /// Restrict domain values to be equal to \a n
    ModEvent eq(Space& home, double n);
    ///@}

    /**
     * \name Domain update by iterator
     *
     * Views can be both updated by range and value iterators.
     * Value iterators do not need to be strict in that the same value
     * is allowed to occur more than once in the iterated sequence.
     *
     * The argument \a depends must be true, if the iterator
     * passed as argument depends on the view on which the operation
     * is invoked. In this case, the view is only updated after the
     * iterator has been consumed. Otherwise, the domain might be updated
     * concurrently while following the iterator.
     *
     */
    //@{
    /// Replace domain by ranges described by \a i
    template <class I>
    ModEvent narrow_r(Space& home, I& i, bool depends=true);
    /// Intersect domain with ranges described by \a i
    template <class I>
    ModEvent inter_r(Space& home, I& i, bool depends=true);
    /// Remove from domain the ranges described by \a i
    template <class I>
    ModEvent minus_r(Space& home, I& i, bool depends=true);
    /// Replace domain by values described by \a i
    template <class I>
    ModEvent narrow_v(Space& home, I& i, bool depends=true);
    /// Intersect domain with values described by \a i
    template <class I>
    ModEvent inter_v(Space& home, I& i, bool depends=true);
    /// Remove from domain the values described by \a i
    template <class I>
    ModEvent minus_v(Space& home, I& i, bool depends=true);
    //@}

    /// \name View-dependent propagator support
    ///@{
    /// Schedule propagator \a p with modification event \a me
    static void schedule(Space& home, Propagator& p, ModEvent me);
    /// Return modification event for view type in \a med
    static ModEvent me(const ModEventDelta& med);
    /// Translate modification event \a me to modification event delta for view
    static ModEventDelta med(ModEvent me);
    ///@}

    /// \name Dependencies
    ///@{
    /// Subscribe propagator \a p with propagation condition \a pc to view
    void subscribe(Space& home, Propagator& p, PropCond pc, bool process=true);
    /// Cancel subscription of propagator \a p with propagation condition \a pc to view
    void cancel(Space& home, Propagator& p, PropCond pc);
    /// Subscribe advisor \a a to variable
    void subscribe(Space& home, Advisor& a);
    /// Cancel subscription of advisor \a a
    void cancel(Space& home, Advisor& a);

    ///@}

    /// \name Cloning
    ///@{
    /// Update this view to be a clone of view \a x
    void update(Space& home, bool share, CardView& x);
    ///@}

    /// \name View comparison
    ///@{
    /// Test whether this view is the same as \a x
    bool operator ==(const CardView& x) const;
    /// Test whether this view is not the same as \a x
    bool operator !=(const CardView& x) const;
    /// Test whether this view is smaller than \a x (arbitrary order)
    bool operator < (const CardView& x) const;
    /// Test whether this view is larger than \a x (arbitrary order)
    bool operator > (const CardView& x) const;
    ///@}
  };

  /*
   * Constructors and initialization
   *
   */
  forceinline
  CardView::CardView(void) {}

  forceinline
  CardView::CardView(const IntView& x, int d)
    : DerivedViewBase<IntView>(x), c(d), count(0) {}

  forceinline void
  CardView::init(const IntView& x, int d) {
    view  = x;
    c     = d;
    count = 0;
  }


  forceinline void
  CardView::init(Space& home, int mi, int ma, int d) {
    IntVar ivar(home, mi, ma);
    IntView iview(ivar);
    view  = iview;
    c     = d;
    count = 0;
  }

  forceinline void
  CardView::card(int ca) {
    c = ca;
  }

  forceinline int
  CardView::card(void) const {
    return c;
  }

  forceinline ModEvent
  CardView::inc(void) {
    count++;
    if (count > this->max()) {
      return ME_GEN_FAILED;
    } else {
      return ME_GEN_NONE;
    }
  }

  forceinline void
  CardView::counter(int c) {
    count = c;
  }

  forceinline int
  CardView::counter(void) {
    return count;
  }

  /*
   * Value access
   *
   */

  forceinline void
  CardView::operator =(const IntView& x) {
    view  = x;
    c     = 0;
    count = 0;
  }

  forceinline void
  CardView::operator =(const CardView& x) {
    view  = x.view;
    c     = x.c;
    count = x.count;
  }


  forceinline int
  CardView::min(void) const {
    return view.min();
  }
  forceinline int
  CardView::max(void) const {
    return view.max();
  }
  forceinline int
  CardView::med(void) const {
    return view.med();
  }

  forceinline int
  CardView::val(void) const {
    return view.val();
  }

  forceinline IntView
  CardView::intview(void){
    return view;
  }


  forceinline unsigned int
  CardView::width(void) const {
    return view.width();
  }
  forceinline unsigned int
  CardView::size(void) const {
    return view.size();
  }
  forceinline unsigned int
  CardView::regret_min(void) const {
    return view.regret_min();
  }
  forceinline unsigned int
  CardView::regret_max(void) const {
    return view.regret_max();
  }

  /*
   * Domain tests
   *
   */
  forceinline bool
  CardView::range(void) const {
    return view.range();
  }
  forceinline bool
  CardView::assigned(void) const {
    return view.assigned();
  }

  forceinline bool
  CardView::in(int n) const {
    return view.in(n);
  }
  forceinline bool
  CardView::in(double n) const {
    return view.in(n);
  }


  /*
   * Domain update by value
   *
   */
  forceinline ModEvent
  CardView::lq(Space& home, int n) {
    return view.lq(home,n);
  }
  forceinline ModEvent
  CardView::lq(Space& home, double n) {
    return view.lq(home,n);
  }
  forceinline ModEvent
  CardView::le(Space& home, int n) {
    return view.le(home,n);
  }
  forceinline ModEvent
  CardView::le(Space& home, double n) {
    return view.le(home,n);
  }
  forceinline ModEvent
  CardView::gq(Space& home, int n) {
    return view.gq(home,n);
  }
  forceinline ModEvent
  CardView::gq(Space& home, double n) {
    return view.gq(home,n);
  }
  forceinline ModEvent
  CardView::gr(Space& home, int n) {
    return view.gr(home,n);
  }
  forceinline ModEvent
  CardView::gr(Space& home, double n) {
    return view.gr(home,n);
  }
  forceinline ModEvent
  CardView::nq(Space& home, int n) {
    return view.nq(home,n);
  }
  forceinline ModEvent
  CardView::nq(Space& home, double n) {
    return view.nq(home,n);
  }
  forceinline ModEvent
  CardView::eq(Space& home, int n) {
    return view.eq(home,n);
  }
  forceinline ModEvent
  CardView::eq(Space& home, double n) {
    return view.eq(home,n);
  }


  /*
   * Domain update by iterator
   *
   */
  template <class I>
  ModEvent
  CardView::narrow_r(Space& home, I& i, bool depends) {
    return view.narrow_r(home,i,depends);
  }
  template <class I>
  ModEvent
  CardView::inter_r(Space& home, I& i, bool depends) {
    return view.inter_r(home,i,depends);
  }
  template <class I>
  ModEvent
  CardView::minus_r(Space& home, I& i, bool depends) {
    return view.minus_r(home,i,depends);
  }
  template <class I>
  ModEvent
  CardView::narrow_v(Space& home, I& i, bool depends) {
    return view.narrow_v(home,i,depends);
  }
  template <class I>
  ModEvent
  CardView::inter_v(Space& home, I& i, bool depends) {
    return view.inter_v(home,i,depends);
  }
  template <class I>
  ModEvent
  CardView::minus_v(Space& home, I& i, bool depends) {
    return view.minus_v(home,i,depends);
  }



  /*
   * Propagator modification events
   *
   */
  forceinline void
  CardView::schedule(Space& home, Propagator& p, ModEvent me) {
    return IntView::schedule(home,p,me);
  }
  forceinline ModEvent
  CardView::me(const ModEventDelta& med) {
    return IntView::me(med);
  }
  forceinline ModEventDelta
  CardView::med(ModEvent me) {
    return IntView::med(me);
  }


  /*
   * Dependencies
   *
   */
  forceinline void
  CardView::subscribe(Space& home, Propagator& p, PropCond pc, bool process) {
    view.subscribe(home, p, pc, process);
  }
  forceinline void
  CardView::cancel(Space& home, Propagator& p, PropCond pc) {
    view.cancel(home,p, pc);
  }
  forceinline void
  CardView::subscribe(Space& home, Advisor& a) {
    view.subscribe(home, a);
  }
  forceinline void
  CardView::cancel(Space& home, Advisor& a) {
    view.cancel(home, a);
  }


  /*
   * Cloning
   *
   */
  forceinline void
  CardView::update(Space& home, bool share, CardView& x) {
    c     = x.c;
    count = x.count;
    view.update(home,share,x.view);
  }

}


  /**
   * \brief %Range iterator for indexed problem variables
   */
  template <>
  class ViewRanges<GCC::CardView>
    : public Gecode::Int::ViewRanges<IntView> {
  public:
    /// \name Constructors and initialization
    ///@{
    /// Default constructor
    ViewRanges(void);
    /// Initialize with ranges for view \a x
    ViewRanges(const GCC::CardView& x);
    /// Initialize with ranges for view \a x
    void init(const GCC::CardView& x);
    ///@}
  };

  forceinline
  ViewRanges<GCC::CardView>::ViewRanges(void) :
    Gecode::Int::ViewRanges<IntView>()  {}

  forceinline
  ViewRanges<GCC::CardView>::ViewRanges (const GCC::CardView& x)
    : Gecode::Int::ViewRanges<IntView>(x.base())  {}

  forceinline void
  ViewRanges<GCC::CardView>::init(const GCC::CardView& x) {
    Gecode::Int::ViewRanges<IntView> xi(x.base());
  }

}}



// STATISTICS: int-prop
