/*
 *  Main authors:
 *     Christian Schulte <schulte@gecode.org>
 *     Guido Tack <tack@gecode.org>
 *
 *  Copyright:
 *     Christian Schulte, 2002
 *     Guido Tack, 2004
 *
 *  Last modified:
 *     $Date: 2005-11-29 13:36:03 +0100 (Tue, 29 Nov 2005) $ by $Author: schulte $
 *     $Revision: 2660 $
 *
 *  This file is part of Gecode, the generic constraint
 *  development environment:
 *     http://www.gecode.org
 *
 *  See the file "LICENSE" for information on usage and
 *  redistribution of this file, and for a
 *     DISCLAIMER OF ALL WARRANTIES.
 *
 */

#ifndef __GECODE_INT_ARITHMETIC_HH__
#define __GECODE_INT_ARITHMETIC_HH__

#include "int.hh"

#include "int/rel.hh"
#include "int/linear.hh"

/**
 * \namespace Gecode::Int::Arithmetic
 * \brief Numerical (arithmetic) propagators
 */

namespace Gecode { namespace Int { namespace Arithmetic {

  /**
   * \brief Bounds-consistent absolute value propagator
   *
   * Requires \code #include "int/arithmetic.hh" \endcode
   * \ingroup FuncIntProp
   */
  template <class View>
  class Abs : public BinaryPropagator<View,PC_INT_BND> {
  protected:
    using BinaryPropagator<View,PC_INT_BND>::x0;
    using BinaryPropagator<View,PC_INT_BND>::x1;

    /// Constructor for cloning \a p
    Abs(Space* home, bool share, Abs& p);
    /// Constructor for posting
    Abs(Space* home, View x0, View x1);
  public:
    /// Copy propagator during cloning
    virtual Actor* copy(Space* home, bool share);
    /// Perform propagation
    virtual ExecStatus  propagate(Space* home);
    /// Post propagator \f$ |x_0|=x_1\f$ 
    static  ExecStatus  post(Space* home, View x0, View x1);
  };

  /**
   * \brief Bounds-consistent ternary maximum propagator
   *
   * Requires \code #include "int/arithmetic.hh" \endcode
   * \ingroup FuncIntProp
   */
  template <class View>
  class Max : public TernaryPropagator<View,PC_INT_BND> {
  protected:
    using TernaryPropagator<View,PC_INT_BND>::x0;
    using TernaryPropagator<View,PC_INT_BND>::x1;
    using TernaryPropagator<View,PC_INT_BND>::x2;

    /// Constructor for cloning \a p
    Max(Space* home, bool share, Max& p);
    /// Constructor for posting
    Max(Space* home, View x0, View x1, View x2);
  public:
    /// Copy propagator during cloning
    virtual Actor* copy(Space* home, bool share);
    /// Perform propagation
    virtual ExecStatus propagate(Space* home);
    /// Post propagator \f$ \max\{x_0,x_1\}=x_2\f$ 
    static  ExecStatus post(Space* home, View x0, View x1, View x2);
  };

  /**
   * \brief Bounds-consistent n-ary maximum propagator
   *
   * Requires \code #include "int/arithmetic.hh" \endcode
   * \ingroup FuncIntProp
   */
  template <class View>
  class NaryMax : public NaryOnePropagator<View,PC_INT_BND> {
  protected:
    using NaryOnePropagator<View,PC_INT_BND>::x;
    using NaryOnePropagator<View,PC_INT_BND>::y;

    /// Constructor for cloning \a p
    NaryMax(Space* home, bool share, NaryMax& p);
    /// Constructor for posting
    NaryMax(Space* home, ViewArray<View>& x, View y);
  public:
    /// Copy propagator during cloning
    virtual Actor* copy(Space* home, bool share);
    /// Perform propagation
    virtual ExecStatus propagate(Space* home);
    /// Post propagator \f$ \max x=y\f$ 
    static  ExecStatus post(Space* home, ViewArray<View>& x, View y);
  };




  /**
   * \brief Bounds-consistent positive square propagator
   *
   * This propagator provides multiplication for positive views only.
   */
  template <class VA, class VB>
  class SquarePlus : public Propagator {
  protected:
    VA x0; VB x1;
  public:
    /// Constructor for posting
    SquarePlus(Space* home, VA x0, VB x1);
    /// Post propagator \f$x_0\cdot x_0=x_1\f$ 
    static ExecStatus post(Space* home, VA x0, VB x1);
    /// Constructor for cloning \a p
    SquarePlus(Space* home, bool share, SquarePlus<VA,VB>& p);
    /// Copy propagator during cloning
    virtual Actor* copy(Space* home, bool share);
    /// Perform propagation
    virtual ExecStatus propagate(Space* home);
    /// Cost function (defined as PC_TERNARY_HI)
    virtual PropCost cost(void) const;
    /// Destructor
    virtual ~SquarePlus(void);
  };

  /**
   * \brief Bounds-consistent square propagator
   *
   * Requires \code #include "int/arithmetic.hh" \endcode
   * \ingroup FuncIntProp
   */
  template <class View>
  class Square : public BinaryPropagator<View,PC_INT_BND> {
  protected:
    using BinaryPropagator<View,PC_INT_BND>::x0;
    using BinaryPropagator<View,PC_INT_BND>::x1;

    /// Constructor for cloning \a p
    Square(Space* home, bool share, Square<View>& p);
    /// Constructor for posting
    Square(Space* home, View x0, View x1);
  public:
    /// Copy propagator during cloning
    virtual Actor* copy(Space* home, bool share);
    /// Perform propagation
    virtual ExecStatus propagate(Space* home);
    /// Cost function (defined as PC_BINARY_HI)
    virtual PropCost cost(void) const;
    /// Post propagator \f$x_0\cdot x_0=x_1\f$ 
    static  ExecStatus post(Space* home, View x0, View x1);
  };

  /**
   * \brief Bounds-consistent positive multiplication propagator
   *
   * This propagator provides multiplication for positive views only.
   */
  template <class VA, class VB, class VC>
  class MultPlus : public Propagator {
  protected:
    VA x0; VB x1; VC x2;
  public:
    /// Constructor for posting
    MultPlus(Space* home, VA x0, VB x1, VC x2);
    /// Post propagator \f$x_0\cdot x_1=x_2\f$ 
    static ExecStatus post(Space* home, VA x0, VB x1, VC x2);
    /// Constructor for cloning \a p
    MultPlus(Space* home, bool share, MultPlus<VA,VB,VC>& p);
    /// Copy propagator during cloning
    virtual Actor* copy(Space* home, bool share);
    /// Perform propagation
    virtual ExecStatus propagate(Space* home);
    /// Cost function (defined as PC_TERNARY_HI)
    virtual PropCost cost(void) const;
    /// Destructor
    virtual ~MultPlus(void);
  };

  /**
   * \brief Bounds-consistent multiplication propagator
   *
   * Requires \code #include "int/arithmetic.hh" \endcode
   *
   * \todo Currently this propagator only works for \a View
   * being IntView. This will change when integer views are
   * available in fully generic form.
   *
   * \ingroup FuncIntProp
   */
  template <class View>
  class Mult : public TernaryPropagator<View,PC_INT_BND> {
  protected:
    using TernaryPropagator<View,PC_INT_BND>::x0;
    using TernaryPropagator<View,PC_INT_BND>::x1;
    using TernaryPropagator<View,PC_INT_BND>::x2;

    /// Constructor for cloning \a p
    Mult(Space* home, bool share, Mult<View>& p);
  public:
    /// Constructor for posting
    Mult(Space* home, View x0, View x1, View x2);
    /// Post propagator \f$x_0\cdot x_1=x_2\f$ 
    static  ExecStatus post(Space* home, View x0, View x1, View x2);
    /// Copy propagator during cloning
    virtual Actor* copy(Space* home, bool share);
    /// Perform propagation
    virtual ExecStatus propagate(Space* home);
    /// Cost function (defined as PC_TERNARY_HI)
    virtual PropCost cost(void) const;
  };

}}}

#include "int/arithmetic/abs.icc"
#include "int/arithmetic/minmax.icc"
#include "int/arithmetic/mult.icc"

#endif

// STATISTICS: int-prop

