// Written in the D programming language.

/++
    Functions which operate on unicode characters.

    For functions which operate on ASCII characters and ignore unicode
    characters, see $(LINK2 std_ascii.html, std.ascii).

    References:
        $(WEB www.digitalmars.com/d/ascii-table.html, ASCII Table),
        $(WEB en.wikipedia.org/wiki/Unicode, Wikipedia),
        $(WEB www.unicode.org, The Unicode Consortium)

    Trademarks:
        Unicode(tm) is a trademark of Unicode, Inc.

    Macros:
        WIKI=Phobos/StdUni

    Copyright: Copyright 2000 -
    License:   $(WEB www.boost.org/LICENSE_1_0.txt, Boost License 1.0).
    Authors:   $(WEB digitalmars.com, Walter Bright), Jonathan M Davis, and Kenji Hara
    Source:    $(PHOBOSSRC std/_uni.d)
  +/
module std.uni;

static import std.ascii;

enum dchar lineSep = '\u2028'; /// UTF line separator
enum dchar paraSep = '\u2029'; /// UTF paragraph separator

/++
    Whether or not $(D c) is a unicode whitespace character.
    (general unicode category: Part of C0(tab, vertical tab, form feed,
    carriage return, and linefeed characters), Zs, Zl, Zp, and NEL(U+0085))
  +/
bool isWhite(dchar c) @safe pure nothrow
{
    return std.ascii.isWhite(c) ||
           c == lineSep || c == paraSep ||
           c == '\u0085' || c == '\u00A0' || c == '\u1680' || c == '\u180E' ||
           (c >= '\u2000' && c <= '\u200A') ||
           c == '\u202F' || c == '\u205F' || c == '\u3000';
}


/++
   $(RED Scheduled for deprecation in January 2012. Please use
   $(D isLower) instead.)

    Return whether $(D c) is a unicode lowercase character.
  +/
alias isLower isUniLower;

/++
    Return whether $(D c) is a unicode lowercase character.
  +/
bool isLower(dchar c) @safe pure nothrow
{
    if(std.ascii.isASCII(c))
        return std.ascii.isLower(c);

    return isAlpha(c) && c == toLower(c);
}


/++
   $(RED Scheduled for deprecation in January 2012. Please use
   $(D isUpper) instead.)

    Return whether $(D c) is a unicode uppercase character.
  +/
alias isUpper isUniUpper;

/++
    Return whether $(D c) is a unicode uppercase character.
  +/
bool isUpper(dchar c) @safe pure nothrow
{
    if(std.ascii.isASCII(c))
        return std.ascii.isUpper(c);

    return isAlpha(c) && c == toUpper(c);
}


/++
   $(RED Scheduled for deprecation in January 2012. Please use
   $(D toLower) instead.)

    If $(D c) is a unicode uppercase character, then its lowercase equivalent
    is returned. Otherwise $(D c) is returned.
  +/
alias toLower toUniLower;

/++
    If $(D c) is a unicode uppercase character, then its lowercase equivalent
    is returned. Otherwise $(D c) is returned.
  +/
dchar toLower(dchar c) @safe pure nothrow
{
    if(std.ascii.isUpper(c))
        c += 32;
    else if(c >= 0x00C0)
    {
        if((c >= 0x00C0 && c <= 0x00D6) ||
           (c >= 0x00D8 && c<=0x00DE))
        {
            c += 32;
        }
        else if((c >= 0x0100 && c < 0x0138) ||
                (c > 0x0149 && c < 0x0178))
        {
            if(c == 0x0130)
                c = 0x0069;
            else if((c & 1) == 0)
                ++c;
        }
        else if(c == 0x0178)
            c = 0x00FF;
        else if((c >= 0x0139 && c < 0x0149) ||
                (c > 0x0178 && c < 0x017F))
        {
            if(c & 1)
                ++c;
        }
        else if(c >= 0x0200 && c <= 0x0217)
        {
            if((c & 1) == 0)
                ++c;
        }
        else if((c >= 0x0401 && c <= 0x040C) ||
                (c>= 0x040E && c <= 0x040F))
        {
            c += 80;
        }
        else if(c >= 0x0410 && c <= 0x042F)
            c += 32;
        else if(c >= 0x0460 && c <= 0x047F)
        {
            if((c & 1) == 0)
                ++c;
        }
        else if(c >= 0x0531 && c <= 0x0556)
            c += 48;
        else if(c >= 0x10A0 && c <= 0x10C5)
            c += 48;
        else if(c >= 0xFF21 && c <= 0xFF3A)
            c += 32;
    }

    return c;
}


/++
   $(RED Scheduled for deprecation in January 2012. Please use
   $(D toUpper) instead.)

    If $(D c) is a unicode lowercase character, then its uppercase equivalent
    is returned. Otherwise $(D c) is returned.
  +/
alias toUpper toUniUpper;

/++
    If $(D c) is a unicode lowercase character, then its uppercase equivalent
    is returned. Otherwise $(D c) is returned.
  +/
dchar toUpper(dchar c) @safe pure nothrow
{
    if(std.ascii.isLower(c))
        c -= 32;
    else if(c >= 0x00E0)
    {
        if((c >= 0x00E0 && c <= 0x00F6) ||
           (c >= 0x00F8 && c <= 0x00FE))
        {
            c -= 32;
        }
        else if(c == 0x00FF)
            c = 0x0178;
        else if((c >= 0x0100 && c < 0x0138) ||
                (c > 0x0149 && c < 0x0178))
        {
            if(c == 0x0131)
                c = 0x0049;
            else if(c & 1)
                --c;
        }
        else if((c >= 0x0139 && c < 0x0149) ||
                (c > 0x0178 && c < 0x017F))
        {
            if((c & 1) == 0)
                --c;
        }
        else if(c == 0x017F)
            c = 0x0053;
        else if(c >= 0x0200 && c <= 0x0217)
        {
            if(c & 1)
                --c;
        }
        else if(c >= 0x0430 && c<= 0x044F)
            c -= 32;
        else if((c >= 0x0451 && c <= 0x045C) ||
                (c >=0x045E && c<= 0x045F))
        {
            c -= 80;
        }
        else if(c >= 0x0460 && c <= 0x047F)
        {
            if(c & 1)
                --c;
        }
        else if(c >= 0x0561 && c < 0x0587)
            c -= 48;
        else if(c >= 0xFF41 && c <= 0xFF5A)
            c -= 32;
    }

    return c;
}


/++
   $(RED Scheduled for deprecation in January 2012. Please use
   $(D isAlpha) instead.)

    Returns whether $(D c) is a unicode alpha character (general unicode
    category: Lu, Ll, Lt, Lm, and Lo).

    Standards: Unicode 5.0.0.
  +/
alias isAlpha isUniAlpha;

/++
    Returns whether $(D c) is a unicode alpha character (general unicode
    category: Lu, Ll, Lt, Lm, and Lo).

    Standards: Unicode 5.0.0.
  +/
bool isAlpha(dchar c) @safe pure nothrow
{
    static immutable dchar table[][2] =
    [
    [ 'A', 'Z' ],
    [ 'a', 'z' ],
    [ 0x00AA, 0x00AA ],
    [ 0x00B5, 0x00B5 ],
    [ 0x00BA, 0x00BA ],
    [ 0x00C0, 0x00D6 ],
    [ 0x00D8, 0x00F6 ],
    [ 0x00F8, 0x02C1 ],
    [ 0x02C6, 0x02D1 ],
    [ 0x02E0, 0x02E4 ],
    [ 0x02EE, 0x02EE ],
    [ 0x037A, 0x037D ],
    [ 0x0386, 0x0386 ],
    [ 0x0388, 0x038A ],
    [ 0x038C, 0x038C ],
    [ 0x038E, 0x03A1 ],
    [ 0x03A3, 0x03CE ],
    [ 0x03D0, 0x03F5 ],
    [ 0x03F7, 0x0481 ],
    [ 0x048A, 0x0513 ],
    [ 0x0531, 0x0556 ],
    [ 0x0559, 0x0559 ],
    [ 0x0561, 0x0587 ],
    [ 0x05D0, 0x05EA ],
    [ 0x05F0, 0x05F2 ],
    [ 0x0621, 0x063A ],
    [ 0x0640, 0x064A ],
    [ 0x066E, 0x066F ],
    [ 0x0671, 0x06D3 ],
    [ 0x06D5, 0x06D5 ],
    [ 0x06E5, 0x06E6 ],
    [ 0x06EE, 0x06EF ],
    [ 0x06FA, 0x06FC ],
    [ 0x06FF, 0x06FF ],
    [ 0x0710, 0x0710 ],
    [ 0x0712, 0x072F ],
    [ 0x074D, 0x076D ],
    [ 0x0780, 0x07A5 ],
    [ 0x07B1, 0x07B1 ],
    [ 0x07CA, 0x07EA ],
    [ 0x07F4, 0x07F5 ],
    [ 0x07FA, 0x07FA ],
    [ 0x0904, 0x0939 ],
    [ 0x093D, 0x093D ],
    [ 0x0950, 0x0950 ],
    [ 0x0958, 0x0961 ],
    [ 0x097B, 0x097F ],
    [ 0x0985, 0x098C ],
    [ 0x098F, 0x0990 ],
    [ 0x0993, 0x09A8 ],
    [ 0x09AA, 0x09B0 ],
    [ 0x09B2, 0x09B2 ],
    [ 0x09B6, 0x09B9 ],
    [ 0x09BD, 0x09BD ],
    [ 0x09CE, 0x09CE ],
    [ 0x09DC, 0x09DD ],
    [ 0x09DF, 0x09E1 ],
    [ 0x09F0, 0x09F1 ],
    [ 0x0A05, 0x0A0A ],
    [ 0x0A0F, 0x0A10 ],
    [ 0x0A13, 0x0A28 ],
    [ 0x0A2A, 0x0A30 ],
    [ 0x0A32, 0x0A33 ],
    [ 0x0A35, 0x0A36 ],
    [ 0x0A38, 0x0A39 ],
    [ 0x0A59, 0x0A5C ],
    [ 0x0A5E, 0x0A5E ],
    [ 0x0A72, 0x0A74 ],
    [ 0x0A85, 0x0A8D ],
    [ 0x0A8F, 0x0A91 ],
    [ 0x0A93, 0x0AA8 ],
    [ 0x0AAA, 0x0AB0 ],
    [ 0x0AB2, 0x0AB3 ],
    [ 0x0AB5, 0x0AB9 ],
    [ 0x0ABD, 0x0ABD ],
    [ 0x0AD0, 0x0AD0 ],
    [ 0x0AE0, 0x0AE1 ],
    [ 0x0B05, 0x0B0C ],
    [ 0x0B0F, 0x0B10 ],
    [ 0x0B13, 0x0B28 ],
    [ 0x0B2A, 0x0B30 ],
    [ 0x0B32, 0x0B33 ],
    [ 0x0B35, 0x0B39 ],
    [ 0x0B3D, 0x0B3D ],
    [ 0x0B5C, 0x0B5D ],
    [ 0x0B5F, 0x0B61 ],
    [ 0x0B71, 0x0B71 ],
    [ 0x0B83, 0x0B83 ],
    [ 0x0B85, 0x0B8A ],
    [ 0x0B8E, 0x0B90 ],
    [ 0x0B92, 0x0B95 ],
    [ 0x0B99, 0x0B9A ],
    [ 0x0B9C, 0x0B9C ],
    [ 0x0B9E, 0x0B9F ],
    [ 0x0BA3, 0x0BA4 ],
    [ 0x0BA8, 0x0BAA ],
    [ 0x0BAE, 0x0BB9 ],
    [ 0x0C05, 0x0C0C ],
    [ 0x0C0E, 0x0C10 ],
    [ 0x0C12, 0x0C28 ],
    [ 0x0C2A, 0x0C33 ],
    [ 0x0C35, 0x0C39 ],
    [ 0x0C60, 0x0C61 ],
    [ 0x0C85, 0x0C8C ],
    [ 0x0C8E, 0x0C90 ],
    [ 0x0C92, 0x0CA8 ],
    [ 0x0CAA, 0x0CB3 ],
    [ 0x0CB5, 0x0CB9 ],
    [ 0x0CBD, 0x0CBD ],
    [ 0x0CDE, 0x0CDE ],
    [ 0x0CE0, 0x0CE1 ],
    [ 0x0D05, 0x0D0C ],
    [ 0x0D0E, 0x0D10 ],
    [ 0x0D12, 0x0D28 ],
    [ 0x0D2A, 0x0D39 ],
    [ 0x0D60, 0x0D61 ],
    [ 0x0D85, 0x0D96 ],
    [ 0x0D9A, 0x0DB1 ],
    [ 0x0DB3, 0x0DBB ],
    [ 0x0DBD, 0x0DBD ],
    [ 0x0DC0, 0x0DC6 ],
    [ 0x0E01, 0x0E30 ],
    [ 0x0E32, 0x0E33 ],
    [ 0x0E40, 0x0E46 ],
    [ 0x0E81, 0x0E82 ],
    [ 0x0E84, 0x0E84 ],
    [ 0x0E87, 0x0E88 ],
    [ 0x0E8A, 0x0E8A ],
    [ 0x0E8D, 0x0E8D ],
    [ 0x0E94, 0x0E97 ],
    [ 0x0E99, 0x0E9F ],
    [ 0x0EA1, 0x0EA3 ],
    [ 0x0EA5, 0x0EA5 ],
    [ 0x0EA7, 0x0EA7 ],
    [ 0x0EAA, 0x0EAB ],
    [ 0x0EAD, 0x0EB0 ],
    [ 0x0EB2, 0x0EB3 ],
    [ 0x0EBD, 0x0EBD ],
    [ 0x0EC0, 0x0EC4 ],
    [ 0x0EC6, 0x0EC6 ],
    [ 0x0EDC, 0x0EDD ],
    [ 0x0F00, 0x0F00 ],
    [ 0x0F40, 0x0F47 ],
    [ 0x0F49, 0x0F6A ],
    [ 0x0F88, 0x0F8B ],
    [ 0x1000, 0x1021 ],
    [ 0x1023, 0x1027 ],
    [ 0x1029, 0x102A ],
    [ 0x1050, 0x1055 ],
    [ 0x10A0, 0x10C5 ],
    [ 0x10D0, 0x10FA ],
    [ 0x10FC, 0x10FC ],
    [ 0x1100, 0x1159 ],
    [ 0x115F, 0x11A2 ],
    [ 0x11A8, 0x11F9 ],
    [ 0x1200, 0x1248 ],
    [ 0x124A, 0x124D ],
    [ 0x1250, 0x1256 ],
    [ 0x1258, 0x1258 ],
    [ 0x125A, 0x125D ],
    [ 0x1260, 0x1288 ],
    [ 0x128A, 0x128D ],
    [ 0x1290, 0x12B0 ],
    [ 0x12B2, 0x12B5 ],
    [ 0x12B8, 0x12BE ],
    [ 0x12C0, 0x12C0 ],
    [ 0x12C2, 0x12C5 ],
    [ 0x12C8, 0x12D6 ],
    [ 0x12D8, 0x1310 ],
    [ 0x1312, 0x1315 ],
    [ 0x1318, 0x135A ],
    [ 0x1380, 0x138F ],
    [ 0x13A0, 0x13F4 ],
    [ 0x1401, 0x166C ],
    [ 0x166F, 0x1676 ],
    [ 0x1681, 0x169A ],
    [ 0x16A0, 0x16EA ],
    [ 0x1700, 0x170C ],
    [ 0x170E, 0x1711 ],
    [ 0x1720, 0x1731 ],
    [ 0x1740, 0x1751 ],
    [ 0x1760, 0x176C ],
    [ 0x176E, 0x1770 ],
    [ 0x1780, 0x17B3 ],
    [ 0x17D7, 0x17D7 ],
    [ 0x17DC, 0x17DC ],
    [ 0x1820, 0x1877 ],
    [ 0x1880, 0x18A8 ],
    [ 0x1900, 0x191C ],
    [ 0x1950, 0x196D ],
    [ 0x1970, 0x1974 ],
    [ 0x1980, 0x19A9 ],
    [ 0x19C1, 0x19C7 ],
    [ 0x1A00, 0x1A16 ],
    [ 0x1B05, 0x1B33 ],
    [ 0x1B45, 0x1B4B ],
    [ 0x1D00, 0x1DBF ],
    [ 0x1E00, 0x1E9B ],
    [ 0x1EA0, 0x1EF9 ],
    [ 0x1F00, 0x1F15 ],
    [ 0x1F18, 0x1F1D ],
    [ 0x1F20, 0x1F45 ],
    [ 0x1F48, 0x1F4D ],
    [ 0x1F50, 0x1F57 ],
    [ 0x1F59, 0x1F59 ],
    [ 0x1F5B, 0x1F5B ],
    [ 0x1F5D, 0x1F5D ],
    [ 0x1F5F, 0x1F7D ],
    [ 0x1F80, 0x1FB4 ],
    [ 0x1FB6, 0x1FBC ],
    [ 0x1FBE, 0x1FBE ],
    [ 0x1FC2, 0x1FC4 ],
    [ 0x1FC6, 0x1FCC ],
    [ 0x1FD0, 0x1FD3 ],
    [ 0x1FD6, 0x1FDB ],
    [ 0x1FE0, 0x1FEC ],
    [ 0x1FF2, 0x1FF4 ],
    [ 0x1FF6, 0x1FFC ],
    [ 0x2071, 0x2071 ],
    [ 0x207F, 0x207F ],
    [ 0x2090, 0x2094 ],
    [ 0x2102, 0x2102 ],
    [ 0x2107, 0x2107 ],
    [ 0x210A, 0x2113 ],
    [ 0x2115, 0x2115 ],
    [ 0x2119, 0x211D ],
    [ 0x2124, 0x2124 ],
    [ 0x2126, 0x2126 ],
    [ 0x2128, 0x2128 ],
    [ 0x212A, 0x212D ],
    [ 0x212F, 0x2139 ],
    [ 0x213C, 0x213F ],
    [ 0x2145, 0x2149 ],
    [ 0x214E, 0x214E ],
    [ 0x2183, 0x2184 ],
    [ 0x2C00, 0x2C2E ],
    [ 0x2C30, 0x2C5E ],
    [ 0x2C60, 0x2C6C ],
    [ 0x2C74, 0x2C77 ],
    [ 0x2C80, 0x2CE4 ],
    [ 0x2D00, 0x2D25 ],
    [ 0x2D30, 0x2D65 ],
    [ 0x2D6F, 0x2D6F ],
    [ 0x2D80, 0x2D96 ],
    [ 0x2DA0, 0x2DA6 ],
    [ 0x2DA8, 0x2DAE ],
    [ 0x2DB0, 0x2DB6 ],
    [ 0x2DB8, 0x2DBE ],
    [ 0x2DC0, 0x2DC6 ],
    [ 0x2DC8, 0x2DCE ],
    [ 0x2DD0, 0x2DD6 ],
    [ 0x2DD8, 0x2DDE ],
    [ 0x3005, 0x3006 ],
    [ 0x3031, 0x3035 ],
    [ 0x303B, 0x303C ],
    [ 0x3041, 0x3096 ],
    [ 0x309D, 0x309F ],
    [ 0x30A1, 0x30FA ],
    [ 0x30FC, 0x30FF ],
    [ 0x3105, 0x312C ],
    [ 0x3131, 0x318E ],
    [ 0x31A0, 0x31B7 ],
    [ 0x31F0, 0x31FF ],
    [ 0x3400, 0x4DB5 ],
    [ 0x4E00, 0x9FBB ],
    [ 0xA000, 0xA48C ],
    [ 0xA717, 0xA71A ],
    [ 0xA800, 0xA801 ],
    [ 0xA803, 0xA805 ],
    [ 0xA807, 0xA80A ],
    [ 0xA80C, 0xA822 ],
    [ 0xA840, 0xA873 ],
    [ 0xAC00, 0xD7A3 ],
    [ 0xF900, 0xFA2D ],
    [ 0xFA30, 0xFA6A ],
    [ 0xFA70, 0xFAD9 ],
    [ 0xFB00, 0xFB06 ],
    [ 0xFB13, 0xFB17 ],
    [ 0xFB1D, 0xFB1D ],
    [ 0xFB1F, 0xFB28 ],
    [ 0xFB2A, 0xFB36 ],
    [ 0xFB38, 0xFB3C ],
    [ 0xFB3E, 0xFB3E ],
    [ 0xFB40, 0xFB41 ],
    [ 0xFB43, 0xFB44 ],
    [ 0xFB46, 0xFBB1 ],
    [ 0xFBD3, 0xFD3D ],
    [ 0xFD50, 0xFD8F ],
    [ 0xFD92, 0xFDC7 ],
    [ 0xFDF0, 0xFDFB ],
    [ 0xFE70, 0xFE74 ],
    [ 0xFE76, 0xFEFC ],
    [ 0xFF21, 0xFF3A ],
    [ 0xFF41, 0xFF5A ],
    [ 0xFF66, 0xFFBE ],
    [ 0xFFC2, 0xFFC7 ],
    [ 0xFFCA, 0xFFCF ],
    [ 0xFFD2, 0xFFD7 ],
    [ 0xFFDA, 0xFFDC ],
    [ 0x10000, 0x1000B ],
    [ 0x1000D, 0x10026 ],
    [ 0x10028, 0x1003A ],
    [ 0x1003C, 0x1003D ],
    [ 0x1003F, 0x1004D ],
    [ 0x10050, 0x1005D ],
    [ 0x10080, 0x100FA ],
    [ 0x10300, 0x1031E ],
    [ 0x10330, 0x10340 ],
    [ 0x10342, 0x10349 ],
    [ 0x10380, 0x1039D ],
    [ 0x103A0, 0x103C3 ],
    [ 0x103C8, 0x103CF ],
    [ 0x10400, 0x1049D ],
    [ 0x10800, 0x10805 ],
    [ 0x10808, 0x10808 ],
    [ 0x1080A, 0x10835 ],
    [ 0x10837, 0x10838 ],
    [ 0x1083C, 0x1083C ],
    [ 0x1083F, 0x1083F ],
    [ 0x10900, 0x10915 ],
    [ 0x10A00, 0x10A00 ],
    [ 0x10A10, 0x10A13 ],
    [ 0x10A15, 0x10A17 ],
    [ 0x10A19, 0x10A33 ],
    [ 0x12000, 0x1236E ],
    [ 0x1D400, 0x1D454 ],
    [ 0x1D456, 0x1D49C ],
    [ 0x1D49E, 0x1D49F ],
    [ 0x1D4A2, 0x1D4A2 ],
    [ 0x1D4A5, 0x1D4A6 ],
    [ 0x1D4A9, 0x1D4AC ],
    [ 0x1D4AE, 0x1D4B9 ],
    [ 0x1D4BB, 0x1D4BB ],
    [ 0x1D4BD, 0x1D4C3 ],
    [ 0x1D4C5, 0x1D505 ],
    [ 0x1D507, 0x1D50A ],
    [ 0x1D50D, 0x1D514 ],
    [ 0x1D516, 0x1D51C ],
    [ 0x1D51E, 0x1D539 ],
    [ 0x1D53B, 0x1D53E ],
    [ 0x1D540, 0x1D544 ],
    [ 0x1D546, 0x1D546 ],
    [ 0x1D54A, 0x1D550 ],
    [ 0x1D552, 0x1D6A5 ],
    [ 0x1D6A8, 0x1D6C0 ],
    [ 0x1D6C2, 0x1D6DA ],
    [ 0x1D6DC, 0x1D6FA ],
    [ 0x1D6FC, 0x1D714 ],
    [ 0x1D716, 0x1D734 ],
    [ 0x1D736, 0x1D74E ],
    [ 0x1D750, 0x1D76E ],
    [ 0x1D770, 0x1D788 ],
    [ 0x1D78A, 0x1D7A8 ],
    [ 0x1D7AA, 0x1D7C2 ],
    [ 0x1D7C4, 0x1D7CB ],
    [ 0x20000, 0x2A6D6 ],
    [ 0x2F800, 0x2FA1D ],
    ];

    // optimization
    if(c < 0xAA)
    {
        if(c < 'A')
            return false;
        if(c <= 'Z')
            return true;
        if(c < 'a')
            return false;
        if(c <= 'z')
            return true;
        return false;
    }

    return binarySearch!table(c);
}

unittest
{
    for(dchar c = 0; c < 0x80; ++c)
    {
        if(c >= 'A' && c <= 'Z')
            assert(isAlpha(c));
        else if(c >= 'a' && c <= 'z')
            assert(isAlpha(c));
        else
            assert(!isAlpha(c));
    }
}


/+
    (general unicode category: Mn, Me, Mc)
    Standards: Unicode 6.0.0.
  +/

bool isMark(dchar c) @safe pure nothrow
{
    static immutable dchar tableMn[][2] =
    [
    [ 0x0300, 0x036F ],
    [ 0x0483, 0x0487 ],
    [ 0x0591, 0x05BD ],
    [ 0x05BF, 0x05BF ],
    [ 0x05C1, 0x05C2 ],
    [ 0x05C4, 0x05C5 ],
    [ 0x05C7, 0x05C7 ],
    [ 0x0610, 0x061A ],
    [ 0x064B, 0x065F ],
    [ 0x0670, 0x0670 ],
    [ 0x06D6, 0x06DC ],
    [ 0x06DF, 0x06E4 ],
    [ 0x06E7, 0x06E8 ],
    [ 0x06EA, 0x06ED ],
    [ 0x0711, 0x0711 ],
    [ 0x0730, 0x074A ],
    [ 0x07A6, 0x07B0 ],
    [ 0x07EB, 0x07F3 ],
    [ 0x0816, 0x0819 ],
    [ 0x081B, 0x0823 ],
    [ 0x0825, 0x0827 ],
    [ 0x0829, 0x082D ],
    [ 0x0859, 0x085B ],
    [ 0x0900, 0x0902 ],
    [ 0x093A, 0x093A ],
    [ 0x093C, 0x093C ],
    [ 0x0941, 0x0948 ],
    [ 0x094D, 0x094D ],
    [ 0x0951, 0x0957 ],
    [ 0x0962, 0x0963 ],
    [ 0x0981, 0x0981 ],
    [ 0x09BC, 0x09BC ],
    [ 0x09C1, 0x09C4 ],
    [ 0x09CD, 0x09CD ],
    [ 0x09E2, 0x09E3 ],
    [ 0x0A01, 0x0A02 ],
    [ 0x0A3C, 0x0A3C ],
    [ 0x0A41, 0x0A42 ],
    [ 0x0A47, 0x0A48 ],
    [ 0x0A4B, 0x0A4D ],
    [ 0x0A51, 0x0A51 ],
    [ 0x0A70, 0x0A71 ],
    [ 0x0A75, 0x0A75 ],
    [ 0x0A81, 0x0A82 ],
    [ 0x0ABC, 0x0ABC ],
    [ 0x0AC1, 0x0AC5 ],
    [ 0x0AC7, 0x0AC8 ],
    [ 0x0ACD, 0x0ACD ],
    [ 0x0AE2, 0x0AE3 ],
    [ 0x0B01, 0x0B01 ],
    [ 0x0B3C, 0x0B3C ],
    [ 0x0B3F, 0x0B3F ],
    [ 0x0B41, 0x0B44 ],
    [ 0x0B4D, 0x0B4D ],
    [ 0x0B56, 0x0B56 ],
    [ 0x0B62, 0x0B63 ],
    [ 0x0B82, 0x0B82 ],
    [ 0x0BC0, 0x0BC0 ],
    [ 0x0BCD, 0x0BCD ],
    [ 0x0C3E, 0x0C40 ],
    [ 0x0C46, 0x0C48 ],
    [ 0x0C4A, 0x0C4D ],
    [ 0x0C55, 0x0C56 ],
    [ 0x0C62, 0x0C63 ],
    [ 0x0CBC, 0x0CBC ],
    [ 0x0CBF, 0x0CBF ],
    [ 0x0CC6, 0x0CC6 ],
    [ 0x0CCC, 0x0CCD ],
    [ 0x0CE2, 0x0CE3 ],
    [ 0x0D41, 0x0D44 ],
    [ 0x0D4D, 0x0D4D ],
    [ 0x0D62, 0x0D63 ],
    [ 0x0DCA, 0x0DCA ],
    [ 0x0DD2, 0x0DD4 ],
    [ 0x0DD6, 0x0DD6 ],
    [ 0x0E31, 0x0E31 ],
    [ 0x0E34, 0x0E3A ],
    [ 0x0E47, 0x0E4E ],
    [ 0x0EB1, 0x0EB1 ],
    [ 0x0EB4, 0x0EB9 ],
    [ 0x0EBB, 0x0EBC ],
    [ 0x0EC8, 0x0ECD ],
    [ 0x0F18, 0x0F19 ],
    [ 0x0F35, 0x0F35 ],
    [ 0x0F37, 0x0F37 ],
    [ 0x0F39, 0x0F39 ],
    [ 0x0F71, 0x0F7E ],
    [ 0x0F80, 0x0F84 ],
    [ 0x0F86, 0x0F87 ],
    [ 0x0F8D, 0x0F97 ],
    [ 0x0F99, 0x0FBC ],
    [ 0x0FC6, 0x0FC6 ],
    [ 0x102D, 0x1030 ],
    [ 0x1032, 0x1037 ],
    [ 0x1039, 0x103A ],
    [ 0x103D, 0x103E ],
    [ 0x1058, 0x1059 ],
    [ 0x105E, 0x1060 ],
    [ 0x1071, 0x1074 ],
    [ 0x1082, 0x1082 ],
    [ 0x1085, 0x1086 ],
    [ 0x108D, 0x108D ],
    [ 0x109D, 0x109D ],
    [ 0x135D, 0x135F ],
    [ 0x1712, 0x1714 ],
    [ 0x1732, 0x1734 ],
    [ 0x1752, 0x1753 ],
    [ 0x1772, 0x1773 ],
    [ 0x17B7, 0x17BD ],
    [ 0x17C6, 0x17C6 ],
    [ 0x17C9, 0x17D3 ],
    [ 0x17DD, 0x17DD ],
    [ 0x180B, 0x180D ],
    [ 0x18A9, 0x18A9 ],
    [ 0x1920, 0x1922 ],
    [ 0x1927, 0x1928 ],
    [ 0x1932, 0x1932 ],
    [ 0x1939, 0x193B ],
    [ 0x1A17, 0x1A18 ],
    [ 0x1A56, 0x1A56 ],
    [ 0x1A58, 0x1A5E ],
    [ 0x1A60, 0x1A60 ],
    [ 0x1A62, 0x1A62 ],
    [ 0x1A65, 0x1A6C ],
    [ 0x1A73, 0x1A7C ],
    [ 0x1A7F, 0x1A7F ],
    [ 0x1B00, 0x1B03 ],
    [ 0x1B34, 0x1B34 ],
    [ 0x1B36, 0x1B3A ],
    [ 0x1B3C, 0x1B3C ],
    [ 0x1B42, 0x1B42 ],
    [ 0x1B6B, 0x1B73 ],
    [ 0x1B80, 0x1B81 ],
    [ 0x1BA2, 0x1BA5 ],
    [ 0x1BA8, 0x1BA9 ],
    [ 0x1BE6, 0x1BE6 ],
    [ 0x1BE8, 0x1BE9 ],
    [ 0x1BED, 0x1BED ],
    [ 0x1BEF, 0x1BF1 ],
    [ 0x1C2C, 0x1C33 ],
    [ 0x1C36, 0x1C37 ],
    [ 0x1CD0, 0x1CD2 ],
    [ 0x1CD4, 0x1CE0 ],
    [ 0x1CE2, 0x1CE8 ],
    [ 0x1CED, 0x1CED ],
    [ 0x1DC0, 0x1DE6 ],
    [ 0x1DFC, 0x1DFF ],
    [ 0x20D0, 0x20DC ],
    [ 0x20E1, 0x20E1 ],
    [ 0x20E5, 0x20F0 ],
    [ 0x2CEF, 0x2CF1 ],
    [ 0x2D7F, 0x2D7F ],
    [ 0x2DE0, 0x2DFF ],
    [ 0x302A, 0x302F ],
    [ 0x3099, 0x309A ],
    [ 0xA66F, 0xA66F ],
    [ 0xA67C, 0xA67D ],
    [ 0xA6F0, 0xA6F1 ],
    [ 0xA802, 0xA802 ],
    [ 0xA806, 0xA806 ],
    [ 0xA80B, 0xA80B ],
    [ 0xA825, 0xA826 ],
    [ 0xA8C4, 0xA8C4 ],
    [ 0xA8E0, 0xA8F1 ],
    [ 0xA926, 0xA92D ],
    [ 0xA947, 0xA951 ],
    [ 0xA980, 0xA982 ],
    [ 0xA9B3, 0xA9B3 ],
    [ 0xA9B6, 0xA9B9 ],
    [ 0xA9BC, 0xA9BC ],
    [ 0xAA29, 0xAA2E ],
    [ 0xAA31, 0xAA32 ],
    [ 0xAA35, 0xAA36 ],
    [ 0xAA43, 0xAA43 ],
    [ 0xAA4C, 0xAA4C ],
    [ 0xAAB0, 0xAAB0 ],
    [ 0xAAB2, 0xAAB4 ],
    [ 0xAAB7, 0xAAB8 ],
    [ 0xAABE, 0xAABF ],
    [ 0xAAC1, 0xAAC1 ],
    [ 0xABE5, 0xABE5 ],
    [ 0xABE8, 0xABE8 ],
    [ 0xABED, 0xABED ],
    [ 0xFB1E, 0xFB1E ],
    [ 0xFE00, 0xFE0F ],
    [ 0xFE20, 0xFE26 ],
    [ 0x101FD, 0x101FD ],
    [ 0x10A01, 0x10A03 ],
    [ 0x10A05, 0x10A06 ],
    [ 0x10A0C, 0x10A0F ],
    [ 0x10A38, 0x10A3A ],
    [ 0x10A3F, 0x10A3F ],
    [ 0x11001, 0x11001 ],
    [ 0x11038, 0x11046 ],
    [ 0x11080, 0x11081 ],
    [ 0x110B3, 0x110B6 ],
    [ 0x110B9, 0x110BA ],
    [ 0x1D167, 0x1D169 ],
    [ 0x1D17B, 0x1D182 ],
    [ 0x1D185, 0x1D18B ],
    [ 0x1D1AA, 0x1D1AD ],
    [ 0x1D242, 0x1D244 ],
    [ 0xE0100, 0xE01EF ],
    ];

    static immutable dchar tableMe[][2] =
    [
    [ 0x0488, 0x0489 ],
    [ 0x20DD, 0x20E0 ],
    [ 0x20E2, 0x20E4 ],
    [ 0xA670, 0xA672 ],
    ];

    static immutable dchar tableMc[][2] =
    [
    [ 0x0903, 0x0903 ],
    [ 0x093B, 0x093B ],
    [ 0x093E, 0x0940 ],
    [ 0x0949, 0x094C ],
    [ 0x094E, 0x094F ],
    [ 0x0982, 0x0983 ],
    [ 0x09BE, 0x09C0 ],
    [ 0x09C7, 0x09C8 ],
    [ 0x09CB, 0x09CC ],
    [ 0x09D7, 0x09D7 ],
    [ 0x0A03, 0x0A03 ],
    [ 0x0A3E, 0x0A40 ],
    [ 0x0A83, 0x0A83 ],
    [ 0x0ABE, 0x0AC0 ],
    [ 0x0AC9, 0x0AC9 ],
    [ 0x0ACB, 0x0ACC ],
    [ 0x0B02, 0x0B03 ],
    [ 0x0B3E, 0x0B3E ],
    [ 0x0B40, 0x0B40 ],
    [ 0x0B47, 0x0B48 ],
    [ 0x0B4B, 0x0B4C ],
    [ 0x0B57, 0x0B57 ],
    [ 0x0BBE, 0x0BBF ],
    [ 0x0BC1, 0x0BC2 ],
    [ 0x0BC6, 0x0BC8 ],
    [ 0x0BCA, 0x0BCC ],
    [ 0x0BD7, 0x0BD7 ],
    [ 0x0C01, 0x0C03 ],
    [ 0x0C41, 0x0C44 ],
    [ 0x0C82, 0x0C83 ],
    [ 0x0CBE, 0x0CBE ],
    [ 0x0CC0, 0x0CC4 ],
    [ 0x0CC7, 0x0CC8 ],
    [ 0x0CCA, 0x0CCB ],
    [ 0x0CD5, 0x0CD6 ],
    [ 0x0D02, 0x0D03 ],
    [ 0x0D3E, 0x0D40 ],
    [ 0x0D46, 0x0D48 ],
    [ 0x0D4A, 0x0D4C ],
    [ 0x0D57, 0x0D57 ],
    [ 0x0D82, 0x0D83 ],
    [ 0x0DCF, 0x0DD1 ],
    [ 0x0DD8, 0x0DDF ],
    [ 0x0DF2, 0x0DF3 ],
    [ 0x0F3E, 0x0F3F ],
    [ 0x0F7F, 0x0F7F ],
    [ 0x102B, 0x102C ],
    [ 0x1031, 0x1031 ],
    [ 0x1038, 0x1038 ],
    [ 0x103B, 0x103C ],
    [ 0x1056, 0x1057 ],
    [ 0x1062, 0x1064 ],
    [ 0x1067, 0x106D ],
    [ 0x1083, 0x1084 ],
    [ 0x1087, 0x108C ],
    [ 0x108F, 0x108F ],
    [ 0x109A, 0x109C ],
    [ 0x17B6, 0x17B6 ],
    [ 0x17BE, 0x17C5 ],
    [ 0x17C7, 0x17C8 ],
    [ 0x1923, 0x1926 ],
    [ 0x1929, 0x192B ],
    [ 0x1930, 0x1931 ],
    [ 0x1933, 0x1938 ],
    [ 0x19B0, 0x19C0 ],
    [ 0x19C8, 0x19C9 ],
    [ 0x1A19, 0x1A1B ],
    [ 0x1A55, 0x1A55 ],
    [ 0x1A57, 0x1A57 ],
    [ 0x1A61, 0x1A61 ],
    [ 0x1A63, 0x1A64 ],
    [ 0x1A6D, 0x1A72 ],
    [ 0x1B04, 0x1B04 ],
    [ 0x1B35, 0x1B35 ],
    [ 0x1B3B, 0x1B3B ],
    [ 0x1B3D, 0x1B41 ],
    [ 0x1B43, 0x1B44 ],
    [ 0x1B82, 0x1B82 ],
    [ 0x1BA1, 0x1BA1 ],
    [ 0x1BA6, 0x1BA7 ],
    [ 0x1BAA, 0x1BAA ],
    [ 0x1BE7, 0x1BE7 ],
    [ 0x1BEA, 0x1BEC ],
    [ 0x1BEE, 0x1BEE ],
    [ 0x1BF2, 0x1BF3 ],
    [ 0x1C24, 0x1C2B ],
    [ 0x1C34, 0x1C35 ],
    [ 0x1CE1, 0x1CE1 ],
    [ 0x1CF2, 0x1CF2 ],
    [ 0xA823, 0xA824 ],
    [ 0xA827, 0xA827 ],
    [ 0xA880, 0xA881 ],
    [ 0xA8B4, 0xA8C3 ],
    [ 0xA952, 0xA953 ],
    [ 0xA983, 0xA983 ],
    [ 0xA9B4, 0xA9B5 ],
    [ 0xA9BA, 0xA9BB ],
    [ 0xA9BD, 0xA9C0 ],
    [ 0xAA2F, 0xAA30 ],
    [ 0xAA33, 0xAA34 ],
    [ 0xAA4D, 0xAA4D ],
    [ 0xAA7B, 0xAA7B ],
    [ 0xABE3, 0xABE4 ],
    [ 0xABE6, 0xABE7 ],
    [ 0xABE9, 0xABEA ],
    [ 0xABEC, 0xABEC ],
    [ 0x11000, 0x11000 ],
    [ 0x11002, 0x11002 ],
    [ 0x11082, 0x11082 ],
    [ 0x110B0, 0x110B2 ],
    [ 0x110B7, 0x110B8 ],
    [ 0x1D165, 0x1D166 ],
    [ 0x1D16D, 0x1D172 ],
    ];

    return binarySearch!tableMn(c) || binarySearch!tableMe(c) || binarySearch!tableMc(c);
}

unittest
{
    assert(isMark('\u0300'));
    assert(isMark('\u0488'));
    assert(isMark('\u0903'));
}


/+
    (general unicode category: Nd, Nl, No)
    Standards: Unicode 6.0.0.
  +/

bool isNumber(dchar c) @safe pure nothrow
{
    static immutable dchar tableNd[][2] =
    [
    [ 0x0030, 0x0039 ],
    [ 0x0660, 0x0669 ],
    [ 0x06F0, 0x06F9 ],
    [ 0x07C0, 0x07C9 ],
    [ 0x0966, 0x096F ],
    [ 0x09E6, 0x09EF ],
    [ 0x0A66, 0x0A6F ],
    [ 0x0AE6, 0x0AEF ],
    [ 0x0B66, 0x0B6F ],
    [ 0x0BE6, 0x0BEF ],
    [ 0x0C66, 0x0C6F ],
    [ 0x0CE6, 0x0CEF ],
    [ 0x0D66, 0x0D6F ],
    [ 0x0E50, 0x0E59 ],
    [ 0x0ED0, 0x0ED9 ],
    [ 0x0F20, 0x0F29 ],
    [ 0x1040, 0x1049 ],
    [ 0x1090, 0x1099 ],
    [ 0x17E0, 0x17E9 ],
    [ 0x1810, 0x1819 ],
    [ 0x1946, 0x194F ],
    [ 0x19D0, 0x19D9 ],
    [ 0x1A80, 0x1A89 ],
    [ 0x1A90, 0x1A99 ],
    [ 0x1B50, 0x1B59 ],
    [ 0x1BB0, 0x1BB9 ],
    [ 0x1C40, 0x1C49 ],
    [ 0x1C50, 0x1C59 ],
    [ 0xA620, 0xA629 ],
    [ 0xA8D0, 0xA8D9 ],
    [ 0xA900, 0xA909 ],
    [ 0xA9D0, 0xA9D9 ],
    [ 0xAA50, 0xAA59 ],
    [ 0xABF0, 0xABF9 ],
    [ 0xFF10, 0xFF19 ],
    [ 0x104A0, 0x104A9 ],
    [ 0x11066, 0x1106F ],
    [ 0x1D7CE, 0x1D7FF ],
    ];

    static immutable dchar tableNl[][2] =
    [
    [ 0x16EE, 0x16F0 ],
    [ 0x2160, 0x2182 ],
    [ 0x2185, 0x2188 ],
    [ 0x3007, 0x3007 ],
    [ 0x3021, 0x3029 ],
    [ 0x3038, 0x303A ],
    [ 0xA6E6, 0xA6EF ],
    [ 0x10140, 0x10174 ],
    [ 0x10341, 0x10341 ],
    [ 0x1034A, 0x1034A ],
    [ 0x103D1, 0x103D5 ],
    [ 0x12400, 0x12462 ],
    ];

    static immutable dchar tableNo[][2] =
    [
    [ 0x00B2, 0x00B3 ],
    [ 0x00B9, 0x00B9 ],
    [ 0x00BC, 0x00BE ],
    [ 0x09F4, 0x09F9 ],
    [ 0x0B72, 0x0B77 ],
    [ 0x0BF0, 0x0BF2 ],
    [ 0x0C78, 0x0C7E ],
    [ 0x0D70, 0x0D75 ],
    [ 0x0F2A, 0x0F33 ],
    [ 0x1369, 0x137C ],
    [ 0x17F0, 0x17F9 ],
    [ 0x19DA, 0x19DA ],
    [ 0x2070, 0x2070 ],
    [ 0x2074, 0x2079 ],
    [ 0x2080, 0x2089 ],
    [ 0x2150, 0x215F ],
    [ 0x2189, 0x2189 ],
    [ 0x2460, 0x249B ],
    [ 0x24EA, 0x24FF ],
    [ 0x2776, 0x2793 ],
    [ 0x2CFD, 0x2CFD ],
    [ 0x3192, 0x3195 ],
    [ 0x3220, 0x3229 ],
    [ 0x3251, 0x325F ],
    [ 0x3280, 0x3289 ],
    [ 0x32B1, 0x32BF ],
    [ 0xA830, 0xA835 ],
    [ 0x10107, 0x10133 ],
    [ 0x10175, 0x10178 ],
    [ 0x1018A, 0x1018A ],
    [ 0x10320, 0x10323 ],
    [ 0x10858, 0x1085F ],
    [ 0x10916, 0x1091B ],
    [ 0x10A40, 0x10A47 ],
    [ 0x10A7D, 0x10A7E ],
    [ 0x10B58, 0x10B5F ],
    [ 0x10B78, 0x10B7F ],
    [ 0x10E60, 0x10E7E ],
    [ 0x11052, 0x11065 ],
    [ 0x1D360, 0x1D371 ],
    [ 0x1F100, 0x1F10A ],
    ];

    return binarySearch!tableNd(c)
        || binarySearch!tableNl(c)
        || binarySearch!tableNo(c);
}

unittest
{
    for (dchar c = '0'; c < '9'; ++c)
    {
        assert(isNumber(c));
    }
}


/+
    (general unicode category: Pd, Ps, Pe, Pc, Po, Pi, Pf)
    Standards: Unicode 6.0.0.
  +/

bool isPunctuation(dchar c) @safe pure nothrow
{
    static immutable dchar tablePd[][2] =
    [
    [ 0x002D, 0x002D ],
    [ 0x058A, 0x058A ],
    [ 0x05BE, 0x05BE ],
    [ 0x1400, 0x1400 ],
    [ 0x1806, 0x1806 ],
    [ 0x2010, 0x2015 ],
    [ 0x2E17, 0x2E17 ],
    [ 0x2E1A, 0x2E1A ],
    [ 0x301C, 0x301C ],
    [ 0x3030, 0x3030 ],
    [ 0x30A0, 0x30A0 ],
    [ 0xFE31, 0xFE32 ],
    [ 0xFE58, 0xFE58 ],
    [ 0xFE63, 0xFE63 ],
    [ 0xFF0D, 0xFF0D ],
    ];

    static immutable dchar tablePs[][2] =
    [
    [ 0x0028, 0x0028 ],
    [ 0x005B, 0x005B ],
    [ 0x007B, 0x007B ],
    [ 0x0F3A, 0x0F3A ],
    [ 0x0F3C, 0x0F3C ],
    [ 0x169B, 0x169B ],
    [ 0x201A, 0x201A ],
    [ 0x201E, 0x201E ],
    [ 0x2045, 0x2045 ],
    [ 0x207D, 0x207D ],
    [ 0x208D, 0x208D ],
    [ 0x2329, 0x2329 ],
    [ 0x2768, 0x2768 ],
    [ 0x276A, 0x276A ],
    [ 0x276C, 0x276C ],
    [ 0x276E, 0x276E ],
    [ 0x2770, 0x2770 ],
    [ 0x2772, 0x2772 ],
    [ 0x2774, 0x2774 ],
    [ 0x27C5, 0x27C5 ],
    [ 0x27E6, 0x27E6 ],
    [ 0x27E8, 0x27E8 ],
    [ 0x27EA, 0x27EA ],
    [ 0x27EC, 0x27EC ],
    [ 0x27EE, 0x27EE ],
    [ 0x2983, 0x2983 ],
    [ 0x2985, 0x2985 ],
    [ 0x2987, 0x2987 ],
    [ 0x2989, 0x2989 ],
    [ 0x298B, 0x298B ],
    [ 0x298D, 0x298D ],
    [ 0x298F, 0x298F ],
    [ 0x2991, 0x2991 ],
    [ 0x2993, 0x2993 ],
    [ 0x2995, 0x2995 ],
    [ 0x2997, 0x2997 ],
    [ 0x29D8, 0x29D8 ],
    [ 0x29DA, 0x29DA ],
    [ 0x29FC, 0x29FC ],
    [ 0x2E22, 0x2E22 ],
    [ 0x2E24, 0x2E24 ],
    [ 0x2E26, 0x2E26 ],
    [ 0x2E28, 0x2E28 ],
    [ 0x3008, 0x3008 ],
    [ 0x300A, 0x300A ],
    [ 0x300C, 0x300C ],
    [ 0x300E, 0x300E ],
    [ 0x3010, 0x3010 ],
    [ 0x3014, 0x3014 ],
    [ 0x3016, 0x3016 ],
    [ 0x3018, 0x3018 ],
    [ 0x301A, 0x301A ],
    [ 0x301D, 0x301D ],
    [ 0xFD3E, 0xFD3E ],
    [ 0xFE17, 0xFE17 ],
    [ 0xFE35, 0xFE35 ],
    [ 0xFE37, 0xFE37 ],
    [ 0xFE39, 0xFE39 ],
    [ 0xFE3B, 0xFE3B ],
    [ 0xFE3D, 0xFE3D ],
    [ 0xFE3F, 0xFE3F ],
    [ 0xFE41, 0xFE41 ],
    [ 0xFE43, 0xFE43 ],
    [ 0xFE47, 0xFE47 ],
    [ 0xFE59, 0xFE59 ],
    [ 0xFE5B, 0xFE5B ],
    [ 0xFE5D, 0xFE5D ],
    [ 0xFF08, 0xFF08 ],
    [ 0xFF3B, 0xFF3B ],
    [ 0xFF5B, 0xFF5B ],
    [ 0xFF5F, 0xFF5F ],
    [ 0xFF62, 0xFF62 ],
    ];

    static immutable dchar tablePe[][2] =
    [
    [ 0x0029, 0x0029 ],
    [ 0x005D, 0x005D ],
    [ 0x007D, 0x007D ],
    [ 0x0F3B, 0x0F3B ],
    [ 0x0F3D, 0x0F3D ],
    [ 0x169C, 0x169C ],
    [ 0x2046, 0x2046 ],
    [ 0x207E, 0x207E ],
    [ 0x208E, 0x208E ],
    [ 0x232A, 0x232A ],
    [ 0x2769, 0x2769 ],
    [ 0x276B, 0x276B ],
    [ 0x276D, 0x276D ],
    [ 0x276F, 0x276F ],
    [ 0x2771, 0x2771 ],
    [ 0x2773, 0x2773 ],
    [ 0x2775, 0x2775 ],
    [ 0x27C6, 0x27C6 ],
    [ 0x27E7, 0x27E7 ],
    [ 0x27E9, 0x27E9 ],
    [ 0x27EB, 0x27EB ],
    [ 0x27ED, 0x27ED ],
    [ 0x27EF, 0x27EF ],
    [ 0x2984, 0x2984 ],
    [ 0x2986, 0x2986 ],
    [ 0x2988, 0x2988 ],
    [ 0x298A, 0x298A ],
    [ 0x298C, 0x298C ],
    [ 0x298E, 0x298E ],
    [ 0x2990, 0x2990 ],
    [ 0x2992, 0x2992 ],
    [ 0x2994, 0x2994 ],
    [ 0x2996, 0x2996 ],
    [ 0x2998, 0x2998 ],
    [ 0x29D9, 0x29D9 ],
    [ 0x29DB, 0x29DB ],
    [ 0x29FD, 0x29FD ],
    [ 0x2E23, 0x2E23 ],
    [ 0x2E25, 0x2E25 ],
    [ 0x2E27, 0x2E27 ],
    [ 0x2E29, 0x2E29 ],
    [ 0x3009, 0x3009 ],
    [ 0x300B, 0x300B ],
    [ 0x300D, 0x300D ],
    [ 0x300F, 0x300F ],
    [ 0x3011, 0x3011 ],
    [ 0x3015, 0x3015 ],
    [ 0x3017, 0x3017 ],
    [ 0x3019, 0x3019 ],
    [ 0x301B, 0x301B ],
    [ 0x301E, 0x301F ],
    [ 0xFD3F, 0xFD3F ],
    [ 0xFE18, 0xFE18 ],
    [ 0xFE36, 0xFE36 ],
    [ 0xFE38, 0xFE38 ],
    [ 0xFE3A, 0xFE3A ],
    [ 0xFE3C, 0xFE3C ],
    [ 0xFE3E, 0xFE3E ],
    [ 0xFE40, 0xFE40 ],
    [ 0xFE42, 0xFE42 ],
    [ 0xFE44, 0xFE44 ],
    [ 0xFE48, 0xFE48 ],
    [ 0xFE5A, 0xFE5A ],
    [ 0xFE5C, 0xFE5C ],
    [ 0xFE5E, 0xFE5E ],
    [ 0xFF09, 0xFF09 ],
    [ 0xFF3D, 0xFF3D ],
    [ 0xFF5D, 0xFF5D ],
    [ 0xFF60, 0xFF60 ],
    [ 0xFF63, 0xFF63 ],
    ];

    static immutable dchar tablePc[][2] =
    [
    [ 0x005F, 0x005F ],
    [ 0x203F, 0x2040 ],
    [ 0x2054, 0x2054 ],
    [ 0xFE33, 0xFE34 ],
    [ 0xFE4D, 0xFE4F ],
    [ 0xFF3F, 0xFF3F ],
    ];

    static immutable dchar tablePo[][2] =
    [
    [ 0x0021, 0x0023 ],
    [ 0x0025, 0x0027 ],
    [ 0x002A, 0x002A ],
    [ 0x002C, 0x002C ],
    [ 0x002E, 0x002F ],
    [ 0x003A, 0x003B ],
    [ 0x003F, 0x0040 ],
    [ 0x005C, 0x005C ],
    [ 0x00A1, 0x00A1 ],
    [ 0x00B7, 0x00B7 ],
    [ 0x00BF, 0x00BF ],
    [ 0x037E, 0x037E ],
    [ 0x0387, 0x0387 ],
    [ 0x055A, 0x055F ],
    [ 0x0589, 0x0589 ],
    [ 0x05C0, 0x05C0 ],
    [ 0x05C3, 0x05C3 ],
    [ 0x05C6, 0x05C6 ],
    [ 0x05F3, 0x05F4 ],
    [ 0x0609, 0x060A ],
    [ 0x060C, 0x060D ],
    [ 0x061B, 0x061B ],
    [ 0x061E, 0x061F ],
    [ 0x066A, 0x066D ],
    [ 0x06D4, 0x06D4 ],
    [ 0x0700, 0x070D ],
    [ 0x07F7, 0x07F9 ],
    [ 0x0830, 0x083E ],
    [ 0x085E, 0x085E ],
    [ 0x0964, 0x0965 ],
    [ 0x0970, 0x0970 ],
    [ 0x0DF4, 0x0DF4 ],
    [ 0x0E4F, 0x0E4F ],
    [ 0x0E5A, 0x0E5B ],
    [ 0x0F04, 0x0F12 ],
    [ 0x0F85, 0x0F85 ],
    [ 0x0FD0, 0x0FD4 ],
    [ 0x0FD9, 0x0FDA ],
    [ 0x104A, 0x104F ],
    [ 0x10FB, 0x10FB ],
    [ 0x1361, 0x1368 ],
    [ 0x166D, 0x166E ],
    [ 0x16EB, 0x16ED ],
    [ 0x1735, 0x1736 ],
    [ 0x17D4, 0x17D6 ],
    [ 0x17D8, 0x17DA ],
    [ 0x1800, 0x1805 ],
    [ 0x1807, 0x180A ],
    [ 0x1944, 0x1945 ],
    [ 0x1A1E, 0x1A1F ],
    [ 0x1AA0, 0x1AA6 ],
    [ 0x1AA8, 0x1AAD ],
    [ 0x1B5A, 0x1B60 ],
    [ 0x1BFC, 0x1BFF ],
    [ 0x1C3B, 0x1C3F ],
    [ 0x1C7E, 0x1C7F ],
    [ 0x1CD3, 0x1CD3 ],
    [ 0x2016, 0x2017 ],
    [ 0x2020, 0x2027 ],
    [ 0x2030, 0x2038 ],
    [ 0x203B, 0x203E ],
    [ 0x2041, 0x2043 ],
    [ 0x2047, 0x2051 ],
    [ 0x2053, 0x2053 ],
    [ 0x2055, 0x205E ],
    [ 0x2CF9, 0x2CFC ],
    [ 0x2CFE, 0x2CFF ],
    [ 0x2D70, 0x2D70 ],
    [ 0x2E00, 0x2E01 ],
    [ 0x2E06, 0x2E08 ],
    [ 0x2E0B, 0x2E0B ],
    [ 0x2E0E, 0x2E16 ],
    [ 0x2E18, 0x2E19 ],
    [ 0x2E1B, 0x2E1B ],
    [ 0x2E1E, 0x2E1F ],
    [ 0x2E2A, 0x2E2E ],
    [ 0x2E30, 0x2E31 ],
    [ 0x3001, 0x3003 ],
    [ 0x303D, 0x303D ],
    [ 0x30FB, 0x30FB ],
    [ 0xA4FE, 0xA4FF ],
    [ 0xA60D, 0xA60F ],
    [ 0xA673, 0xA673 ],
    [ 0xA67E, 0xA67E ],
    [ 0xA6F2, 0xA6F7 ],
    [ 0xA874, 0xA877 ],
    [ 0xA8CE, 0xA8CF ],
    [ 0xA8F8, 0xA8FA ],
    [ 0xA92E, 0xA92F ],
    [ 0xA95F, 0xA95F ],
    [ 0xA9C1, 0xA9CD ],
    [ 0xA9DE, 0xA9DF ],
    [ 0xAA5C, 0xAA5F ],
    [ 0xAADE, 0xAADF ],
    [ 0xABEB, 0xABEB ],
    [ 0xFE10, 0xFE16 ],
    [ 0xFE19, 0xFE19 ],
    [ 0xFE30, 0xFE30 ],
    [ 0xFE45, 0xFE46 ],
    [ 0xFE49, 0xFE4C ],
    [ 0xFE50, 0xFE52 ],
    [ 0xFE54, 0xFE57 ],
    [ 0xFE5F, 0xFE61 ],
    [ 0xFE68, 0xFE68 ],
    [ 0xFE6A, 0xFE6B ],
    [ 0xFF01, 0xFF03 ],
    [ 0xFF05, 0xFF07 ],
    [ 0xFF0A, 0xFF0A ],
    [ 0xFF0C, 0xFF0C ],
    [ 0xFF0E, 0xFF0F ],
    [ 0xFF1A, 0xFF1B ],
    [ 0xFF1F, 0xFF20 ],
    [ 0xFF3C, 0xFF3C ],
    [ 0xFF61, 0xFF61 ],
    [ 0xFF64, 0xFF65 ],
    [ 0x10100, 0x10101 ],
    [ 0x1039F, 0x1039F ],
    [ 0x103D0, 0x103D0 ],
    [ 0x10857, 0x10857 ],
    [ 0x1091F, 0x1091F ],
    [ 0x1093F, 0x1093F ],
    [ 0x10A50, 0x10A58 ],
    [ 0x10A7F, 0x10A7F ],
    [ 0x10B39, 0x10B3F ],
    [ 0x11047, 0x1104D ],
    [ 0x110BB, 0x110BC ],
    [ 0x110BE, 0x110C1 ],
    [ 0x12470, 0x12473 ],
    ];

    static immutable dchar tablePi[][2] =
    [
    [ 0x00AB, 0x00AB ],
    [ 0x2018, 0x2018 ],
    [ 0x201B, 0x201C ],
    [ 0x201F, 0x201F ],
    [ 0x2039, 0x2039 ],
    [ 0x2E02, 0x2E02 ],
    [ 0x2E04, 0x2E04 ],
    [ 0x2E09, 0x2E09 ],
    [ 0x2E0C, 0x2E0C ],
    [ 0x2E1C, 0x2E1C ],
    [ 0x2E20, 0x2E20 ],
    ];

    static immutable dchar tablePf[][2] =
    [
    [ 0x00BB, 0x00BB ],
    [ 0x2019, 0x2019 ],
    [ 0x201D, 0x201D ],
    [ 0x203A, 0x203A ],
    [ 0x2E03, 0x2E03 ],
    [ 0x2E05, 0x2E05 ],
    [ 0x2E0A, 0x2E0A ],
    [ 0x2E0D, 0x2E0D ],
    [ 0x2E1D, 0x2E1D ],
    [ 0x2E21, 0x2E21 ],
    ];

    return binarySearch!tablePd(c)
        || binarySearch!tablePs(c)
        || binarySearch!tablePe(c)
        || binarySearch!tablePc(c)
        || binarySearch!tablePo(c)
        || binarySearch!tablePi(c)
        || binarySearch!tablePf(c);
}

unittest
{
    assert(isPunctuation('\u0021'));
    assert(isPunctuation('\u0028'));
    assert(isPunctuation('\u0029'));
    assert(isPunctuation('\u002D'));
    assert(isPunctuation('\u005F'));
    assert(isPunctuation('\u00AB'));
    assert(isPunctuation('\u00BB'));
}


/+
    (general unicode category: Sm, Sc, Sk, So)
    Standards: Unicode 6.0.0.
  +/

bool isSymbol(dchar c) @safe pure nothrow
{
    static immutable dchar tableSm[][2] =
    [
    [ 0x002B, 0x002B ],
    [ 0x003C, 0x003E ],
    [ 0x007C, 0x007C ],
    [ 0x007E, 0x007E ],
    [ 0x00AC, 0x00AC ],
    [ 0x00B1, 0x00B1 ],
    [ 0x00D7, 0x00D7 ],
    [ 0x00F7, 0x00F7 ],
    [ 0x03F6, 0x03F6 ],
    [ 0x0606, 0x0608 ],
    [ 0x2044, 0x2044 ],
    [ 0x2052, 0x2052 ],
    [ 0x207A, 0x207C ],
    [ 0x208A, 0x208C ],
    [ 0x2118, 0x2118 ],
    [ 0x2140, 0x2144 ],
    [ 0x214B, 0x214B ],
    [ 0x2190, 0x2194 ],
    [ 0x219A, 0x219B ],
    [ 0x21A0, 0x21A0 ],
    [ 0x21A3, 0x21A3 ],
    [ 0x21A6, 0x21A6 ],
    [ 0x21AE, 0x21AE ],
    [ 0x21CE, 0x21CF ],
    [ 0x21D2, 0x21D2 ],
    [ 0x21D4, 0x21D4 ],
    [ 0x21F4, 0x22FF ],
    [ 0x2308, 0x230B ],
    [ 0x2320, 0x2321 ],
    [ 0x237C, 0x237C ],
    [ 0x239B, 0x23B3 ],
    [ 0x23DC, 0x23E1 ],
    [ 0x25B7, 0x25B7 ],
    [ 0x25C1, 0x25C1 ],
    [ 0x25F8, 0x25FF ],
    [ 0x266F, 0x266F ],
    [ 0x27C0, 0x27C4 ],
    [ 0x27C7, 0x27CA ],
    [ 0x27CC, 0x27CC ],
    [ 0x27CE, 0x27E5 ],
    [ 0x27F0, 0x27FF ],
    [ 0x2900, 0x2982 ],
    [ 0x2999, 0x29D7 ],
    [ 0x29DC, 0x29FB ],
    [ 0x29FE, 0x2AFF ],
    [ 0x2B30, 0x2B44 ],
    [ 0x2B47, 0x2B4C ],
    [ 0xFB29, 0xFB29 ],
    [ 0xFE62, 0xFE62 ],
    [ 0xFE64, 0xFE66 ],
    [ 0xFF0B, 0xFF0B ],
    [ 0xFF1C, 0xFF1E ],
    [ 0xFF5C, 0xFF5C ],
    [ 0xFF5E, 0xFF5E ],
    [ 0xFFE2, 0xFFE2 ],
    [ 0xFFE9, 0xFFEC ],
    [ 0x1D6C1, 0x1D6C1 ],
    [ 0x1D6DB, 0x1D6DB ],
    [ 0x1D6FB, 0x1D6FB ],
    [ 0x1D715, 0x1D715 ],
    [ 0x1D735, 0x1D735 ],
    [ 0x1D74F, 0x1D74F ],
    [ 0x1D76F, 0x1D76F ],
    [ 0x1D789, 0x1D789 ],
    [ 0x1D7A9, 0x1D7A9 ],
    [ 0x1D7C3, 0x1D7C3 ],
    ];

    static immutable dchar tableSc[][2] =
    [
    [ 0x0024, 0x0024 ],
    [ 0x00A2, 0x00A5 ],
    [ 0x060B, 0x060B ],
    [ 0x09F2, 0x09F3 ],
    [ 0x09FB, 0x09FB ],
    [ 0x0AF1, 0x0AF1 ],
    [ 0x0BF9, 0x0BF9 ],
    [ 0x0E3F, 0x0E3F ],
    [ 0x17DB, 0x17DB ],
    [ 0x20A0, 0x20B9 ],
    [ 0xA838, 0xA838 ],
    [ 0xFDFC, 0xFDFC ],
    [ 0xFE69, 0xFE69 ],
    [ 0xFF04, 0xFF04 ],
    [ 0xFFE0, 0xFFE1 ],
    [ 0xFFE5, 0xFFE6 ],
    ];

    static immutable dchar tableSk[][2] =
    [
    [ 0x005E, 0x005E ],
    [ 0x0060, 0x0060 ],
    [ 0x00A8, 0x00A8 ],
    [ 0x00AF, 0x00AF ],
    [ 0x00B4, 0x00B4 ],
    [ 0x00B8, 0x00B8 ],
    [ 0x02C2, 0x02C5 ],
    [ 0x02D2, 0x02DF ],
    [ 0x02E5, 0x02EB ],
    [ 0x02ED, 0x02ED ],
    [ 0x02EF, 0x02FF ],
    [ 0x0375, 0x0375 ],
    [ 0x0384, 0x0385 ],
    [ 0x1FBD, 0x1FBD ],
    [ 0x1FBF, 0x1FC1 ],
    [ 0x1FCD, 0x1FCF ],
    [ 0x1FDD, 0x1FDF ],
    [ 0x1FED, 0x1FEF ],
    [ 0x1FFD, 0x1FFE ],
    [ 0x309B, 0x309C ],
    [ 0xA700, 0xA716 ],
    [ 0xA720, 0xA721 ],
    [ 0xA789, 0xA78A ],
    [ 0xFBB2, 0xFBC1 ],
    [ 0xFF3E, 0xFF3E ],
    [ 0xFF40, 0xFF40 ],
    [ 0xFFE3, 0xFFE3 ],
    ];

    static immutable dchar tableSo[][2] =
    [
    [ 0x00A6, 0x00A7 ],
    [ 0x00A9, 0x00A9 ],
    [ 0x00AE, 0x00AE ],
    [ 0x00B0, 0x00B0 ],
    [ 0x00B6, 0x00B6 ],
    [ 0x0482, 0x0482 ],
    [ 0x060E, 0x060F ],
    [ 0x06DE, 0x06DE ],
    [ 0x06E9, 0x06E9 ],
    [ 0x06FD, 0x06FE ],
    [ 0x07F6, 0x07F6 ],
    [ 0x09FA, 0x09FA ],
    [ 0x0B70, 0x0B70 ],
    [ 0x0BF3, 0x0BF8 ],
    [ 0x0BFA, 0x0BFA ],
    [ 0x0C7F, 0x0C7F ],
    [ 0x0D79, 0x0D79 ],
    [ 0x0F01, 0x0F03 ],
    [ 0x0F13, 0x0F17 ],
    [ 0x0F1A, 0x0F1F ],
    [ 0x0F34, 0x0F34 ],
    [ 0x0F36, 0x0F36 ],
    [ 0x0F38, 0x0F38 ],
    [ 0x0FBE, 0x0FC5 ],
    [ 0x0FC7, 0x0FCC ],
    [ 0x0FCE, 0x0FCF ],
    [ 0x0FD5, 0x0FD8 ],
    [ 0x109E, 0x109F ],
    [ 0x1360, 0x1360 ],
    [ 0x1390, 0x1399 ],
    [ 0x1940, 0x1940 ],
    [ 0x19DE, 0x19FF ],
    [ 0x1B61, 0x1B6A ],
    [ 0x1B74, 0x1B7C ],
    [ 0x2100, 0x2101 ],
    [ 0x2103, 0x2106 ],
    [ 0x2108, 0x2109 ],
    [ 0x2114, 0x2114 ],
    [ 0x2116, 0x2117 ],
    [ 0x211E, 0x2123 ],
    [ 0x2125, 0x2125 ],
    [ 0x2127, 0x2127 ],
    [ 0x2129, 0x2129 ],
    [ 0x212E, 0x212E ],
    [ 0x213A, 0x213B ],
    [ 0x214A, 0x214A ],
    [ 0x214C, 0x214D ],
    [ 0x214F, 0x214F ],
    [ 0x2195, 0x2199 ],
    [ 0x219C, 0x219F ],
    [ 0x21A1, 0x21A2 ],
    [ 0x21A4, 0x21A5 ],
    [ 0x21A7, 0x21AD ],
    [ 0x21AF, 0x21CD ],
    [ 0x21D0, 0x21D1 ],
    [ 0x21D3, 0x21D3 ],
    [ 0x21D5, 0x21F3 ],
    [ 0x2300, 0x2307 ],
    [ 0x230C, 0x231F ],
    [ 0x2322, 0x2328 ],
    [ 0x232B, 0x237B ],
    [ 0x237D, 0x239A ],
    [ 0x23B4, 0x23DB ],
    [ 0x23E2, 0x23F3 ],
    [ 0x2400, 0x2426 ],
    [ 0x2440, 0x244A ],
    [ 0x249C, 0x24E9 ],
    [ 0x2500, 0x25B6 ],
    [ 0x25B8, 0x25C0 ],
    [ 0x25C2, 0x25F7 ],
    [ 0x2600, 0x266E ],
    [ 0x2670, 0x26FF ],
    [ 0x2701, 0x2767 ],
    [ 0x2794, 0x27BF ],
    [ 0x2800, 0x28FF ],
    [ 0x2B00, 0x2B2F ],
    [ 0x2B45, 0x2B46 ],
    [ 0x2B50, 0x2B59 ],
    [ 0x2CE5, 0x2CEA ],
    [ 0x2E80, 0x2E99 ],
    [ 0x2E9B, 0x2EF3 ],
    [ 0x2F00, 0x2FD5 ],
    [ 0x2FF0, 0x2FFB ],
    [ 0x3004, 0x3004 ],
    [ 0x3012, 0x3013 ],
    [ 0x3020, 0x3020 ],
    [ 0x3036, 0x3037 ],
    [ 0x303E, 0x303F ],
    [ 0x3190, 0x3191 ],
    [ 0x3196, 0x319F ],
    [ 0x31C0, 0x31E3 ],
    [ 0x3200, 0x321E ],
    [ 0x322A, 0x3250 ],
    [ 0x3260, 0x327F ],
    [ 0x328A, 0x32B0 ],
    [ 0x32C0, 0x32FE ],
    [ 0x3300, 0x33FF ],
    [ 0x4DC0, 0x4DFF ],
    [ 0xA490, 0xA4C6 ],
    [ 0xA828, 0xA82B ],
    [ 0xA836, 0xA837 ],
    [ 0xA839, 0xA839 ],
    [ 0xAA77, 0xAA79 ],
    [ 0xFDFD, 0xFDFD ],
    [ 0xFFE4, 0xFFE4 ],
    [ 0xFFE8, 0xFFE8 ],
    [ 0xFFED, 0xFFEE ],
    [ 0xFFFC, 0xFFFD ],
    [ 0x10102, 0x10102 ],
    [ 0x10137, 0x1013F ],
    [ 0x10179, 0x10189 ],
    [ 0x10190, 0x1019B ],
    [ 0x101D0, 0x101FC ],
    [ 0x1D000, 0x1D0F5 ],
    [ 0x1D100, 0x1D126 ],
    [ 0x1D129, 0x1D164 ],
    [ 0x1D16A, 0x1D16C ],
    [ 0x1D183, 0x1D184 ],
    [ 0x1D18C, 0x1D1A9 ],
    [ 0x1D1AE, 0x1D1DD ],
    [ 0x1D200, 0x1D241 ],
    [ 0x1D245, 0x1D245 ],
    [ 0x1D300, 0x1D356 ],
    [ 0x1F000, 0x1F02B ],
    [ 0x1F030, 0x1F093 ],
    [ 0x1F0A0, 0x1F0AE ],
    [ 0x1F0B1, 0x1F0BE ],
    [ 0x1F0C1, 0x1F0CF ],
    [ 0x1F0D1, 0x1F0DF ],
    [ 0x1F110, 0x1F12E ],
    [ 0x1F130, 0x1F169 ],
    [ 0x1F170, 0x1F19A ],
    [ 0x1F1E6, 0x1F202 ],
    [ 0x1F210, 0x1F23A ],
    [ 0x1F240, 0x1F248 ],
    [ 0x1F250, 0x1F251 ],
    [ 0x1F300, 0x1F320 ],
    [ 0x1F330, 0x1F335 ],
    [ 0x1F337, 0x1F37C ],
    [ 0x1F380, 0x1F393 ],
    [ 0x1F3A0, 0x1F3C4 ],
    [ 0x1F3C6, 0x1F3CA ],
    [ 0x1F3E0, 0x1F3F0 ],
    [ 0x1F400, 0x1F43E ],
    [ 0x1F440, 0x1F440 ],
    [ 0x1F442, 0x1F4F7 ],
    [ 0x1F4F9, 0x1F4FC ],
    [ 0x1F500, 0x1F53D ],
    [ 0x1F550, 0x1F567 ],
    [ 0x1F5FB, 0x1F5FF ],
    [ 0x1F601, 0x1F610 ],
    [ 0x1F612, 0x1F614 ],
    [ 0x1F616, 0x1F616 ],
    [ 0x1F618, 0x1F618 ],
    [ 0x1F61A, 0x1F61A ],
    [ 0x1F61C, 0x1F61E ],
    [ 0x1F620, 0x1F625 ],
    [ 0x1F628, 0x1F62B ],
    [ 0x1F62D, 0x1F62D ],
    [ 0x1F630, 0x1F633 ],
    [ 0x1F635, 0x1F640 ],
    [ 0x1F645, 0x1F64F ],
    [ 0x1F680, 0x1F6C5 ],
    [ 0x1F700, 0x1F773 ],
    ];

    return binarySearch!tableSm(c)
        || binarySearch!tableSc(c)
        || binarySearch!tableSk(c)
        || binarySearch!tableSo(c);
}

unittest
{
    assert(isSymbol('\u0024'));
    assert(isSymbol('\u002B'));
    assert(isSymbol('\u005E'));
    assert(isSymbol('\u00A6'));
}


/+
    (general unicode category: Zs)
    Standards: Unicode 6.0.0.
  +/

bool isSpace(dchar c) @safe pure nothrow
{
    return (c == 0x0020 ||
        c == 0x00A0 || c == 0x1680 || c == 0x180E ||
        (0x2000 <= c && c <= 0x200A) ||
        c == 0x202F || c == 0x205F || c == 0x3000);
}

unittest
{
    assert(isSpace('\u0020'));
}


/+
    (general unicode category: L, M, N, P, S, Zs)
    Standards: Unicode 6.0.0.
  +/

bool isGraphical(dchar c) @safe pure nothrow
{
    return isAlpha(c) || isNumber(c) || isSpace(c)
        || isMark(c) || isPunctuation(c) || isSymbol(c);
}

unittest
{
}


/+
    (general unicode category: Cc)
    Standards: Unicode 6.0.0.
  +/

bool isControl(dchar c) @safe pure nothrow
{
    return (c <= 0x1F || (0x80 <= c && c <= 0x9F));
}

unittest
{
    assert(isControl('\u0000'));
}


/+
    (general unicode category: Cf)
    Standards: Unicode 6.0.0.
  +/

bool isFormat(dchar c) @safe pure nothrow
{
    static immutable dchar tableCf[][2] =
    [
    [ 0x00AD, 0x00AD ],
    [ 0x0600, 0x0603 ],
    [ 0x06DD, 0x06DD ],
    [ 0x070F, 0x070F ],
    [ 0x17B4, 0x17B5 ],
    [ 0x200B, 0x200F ],
    [ 0x202A, 0x202E ],
    [ 0x2060, 0x2064 ],
    [ 0x206A, 0x206F ],
    [ 0xFEFF, 0xFEFF ],
    [ 0xFFF9, 0xFFFB ],
    [ 0x110BD, 0x110BD ],
    [ 0x1D173, 0x1D17A ],
    [ 0xE0001, 0xE0001 ],
    [ 0xE0020, 0xE007F ],
    ];

    return binarySearch!tableCf(c);
}

unittest
{
    assert(isFormat('\u00AD'));
}


/+
    (general unicode category: Co)
    Standards: Unicode 6.0.0.
  +/

bool isPrivateUse(dchar c) @safe pure nothrow
{
    return (0x00_E000 <= c && c <= 0x00_F8FF)
        || (0x0F_0000 <= c && c <= 0x0F_FFFD)
        || (0x10_0000 <= c && c <= 0x10_FFFD);
}


unittest
{
}


/+
    (general unicode category: Cs)
    Standards: Unicode 6.0.0.
  +/

bool isSurrogate(dchar c) @safe pure nothrow
{
    return (0xD800 <= c && c <= 0xDFFF);
}

// ditto
bool isSurrogateHi(dchar c) @safe pure nothrow
{
    return (0xD800 <= c && c <= 0xDBFF);
}

// ditto
bool isSurrogateLo(dchar c) @safe pure nothrow
{
    return (0xDC00 <= c && c <= 0xDFFF);
}

unittest
{
}


/+
    (general unicode category: Cn)
    Standards: Unicode 6.0.0.
  +/

bool isNonCharacter(dchar c) @safe pure nothrow
{
    static immutable dchar table[][2] =
    [
    [ 0x0378, 0x0379 ],
    [ 0x037F, 0x0383 ],
    [ 0x038B, 0x038B ],
    [ 0x038D, 0x038D ],
    [ 0x03A2, 0x03A2 ],
    [ 0x0528, 0x0530 ],
    [ 0x0557, 0x0558 ],
    [ 0x0560, 0x0560 ],
    [ 0x0588, 0x0588 ],
    [ 0x058B, 0x0590 ],
    [ 0x05C8, 0x05CF ],
    [ 0x05EB, 0x05EF ],
    [ 0x05F5, 0x05FF ],
    [ 0x0604, 0x0605 ],
    [ 0x061C, 0x061D ],
    [ 0x070E, 0x070E ],
    [ 0x074B, 0x074C ],
    [ 0x07B2, 0x07BF ],
    [ 0x07FB, 0x07FF ],
    [ 0x082E, 0x082F ],
    [ 0x083F, 0x083F ],
    [ 0x085C, 0x085D ],
    [ 0x085F, 0x08FF ],
    [ 0x0978, 0x0978 ],
    [ 0x0980, 0x0980 ],
    [ 0x0984, 0x0984 ],
    [ 0x098D, 0x098E ],
    [ 0x0991, 0x0992 ],
    [ 0x09A9, 0x09A9 ],
    [ 0x09B1, 0x09B1 ],
    [ 0x09B3, 0x09B5 ],
    [ 0x09BA, 0x09BB ],
    [ 0x09C5, 0x09C6 ],
    [ 0x09C9, 0x09CA ],
    [ 0x09CF, 0x09D6 ],
    [ 0x09D8, 0x09DB ],
    [ 0x09DE, 0x09DE ],
    [ 0x09E4, 0x09E5 ],
    [ 0x09FC, 0x0A00 ],
    [ 0x0A04, 0x0A04 ],
    [ 0x0A0B, 0x0A0E ],
    [ 0x0A11, 0x0A12 ],
    [ 0x0A29, 0x0A29 ],
    [ 0x0A31, 0x0A31 ],
    [ 0x0A34, 0x0A34 ],
    [ 0x0A37, 0x0A37 ],
    [ 0x0A3A, 0x0A3B ],
    [ 0x0A3D, 0x0A3D ],
    [ 0x0A43, 0x0A46 ],
    [ 0x0A49, 0x0A4A ],
    [ 0x0A4E, 0x0A50 ],
    [ 0x0A52, 0x0A58 ],
    [ 0x0A5D, 0x0A5D ],
    [ 0x0A5F, 0x0A65 ],
    [ 0x0A76, 0x0A80 ],
    [ 0x0A84, 0x0A84 ],
    [ 0x0A8E, 0x0A8E ],
    [ 0x0A92, 0x0A92 ],
    [ 0x0AA9, 0x0AA9 ],
    [ 0x0AB1, 0x0AB1 ],
    [ 0x0AB4, 0x0AB4 ],
    [ 0x0ABA, 0x0ABB ],
    [ 0x0AC6, 0x0AC6 ],
    [ 0x0ACA, 0x0ACA ],
    [ 0x0ACE, 0x0ACF ],
    [ 0x0AD1, 0x0ADF ],
    [ 0x0AE4, 0x0AE5 ],
    [ 0x0AF0, 0x0AF0 ],
    [ 0x0AF2, 0x0B00 ],
    [ 0x0B04, 0x0B04 ],
    [ 0x0B0D, 0x0B0E ],
    [ 0x0B11, 0x0B12 ],
    [ 0x0B29, 0x0B29 ],
    [ 0x0B31, 0x0B31 ],
    [ 0x0B34, 0x0B34 ],
    [ 0x0B3A, 0x0B3B ],
    [ 0x0B45, 0x0B46 ],
    [ 0x0B49, 0x0B4A ],
    [ 0x0B4E, 0x0B55 ],
    [ 0x0B58, 0x0B5B ],
    [ 0x0B5E, 0x0B5E ],
    [ 0x0B64, 0x0B65 ],
    [ 0x0B78, 0x0B81 ],
    [ 0x0B84, 0x0B84 ],
    [ 0x0B8B, 0x0B8D ],
    [ 0x0B91, 0x0B91 ],
    [ 0x0B96, 0x0B98 ],
    [ 0x0B9B, 0x0B9B ],
    [ 0x0B9D, 0x0B9D ],
    [ 0x0BA0, 0x0BA2 ],
    [ 0x0BA5, 0x0BA7 ],
    [ 0x0BAB, 0x0BAD ],
    [ 0x0BBA, 0x0BBD ],
    [ 0x0BC3, 0x0BC5 ],
    [ 0x0BC9, 0x0BC9 ],
    [ 0x0BCE, 0x0BCF ],
    [ 0x0BD1, 0x0BD6 ],
    [ 0x0BD8, 0x0BE5 ],
    [ 0x0BFB, 0x0C00 ],
    [ 0x0C04, 0x0C04 ],
    [ 0x0C0D, 0x0C0D ],
    [ 0x0C11, 0x0C11 ],
    [ 0x0C29, 0x0C29 ],
    [ 0x0C34, 0x0C34 ],
    [ 0x0C3A, 0x0C3C ],
    [ 0x0C45, 0x0C45 ],
    [ 0x0C49, 0x0C49 ],
    [ 0x0C4E, 0x0C54 ],
    [ 0x0C57, 0x0C57 ],
    [ 0x0C5A, 0x0C5F ],
    [ 0x0C64, 0x0C65 ],
    [ 0x0C70, 0x0C77 ],
    [ 0x0C80, 0x0C81 ],
    [ 0x0C84, 0x0C84 ],
    [ 0x0C8D, 0x0C8D ],
    [ 0x0C91, 0x0C91 ],
    [ 0x0CA9, 0x0CA9 ],
    [ 0x0CB4, 0x0CB4 ],
    [ 0x0CBA, 0x0CBB ],
    [ 0x0CC5, 0x0CC5 ],
    [ 0x0CC9, 0x0CC9 ],
    [ 0x0CCE, 0x0CD4 ],
    [ 0x0CD7, 0x0CDD ],
    [ 0x0CDF, 0x0CDF ],
    [ 0x0CE4, 0x0CE5 ],
    [ 0x0CF0, 0x0CF0 ],
    [ 0x0CF3, 0x0D01 ],
    [ 0x0D04, 0x0D04 ],
    [ 0x0D0D, 0x0D0D ],
    [ 0x0D11, 0x0D11 ],
    [ 0x0D3B, 0x0D3C ],
    [ 0x0D45, 0x0D45 ],
    [ 0x0D49, 0x0D49 ],
    [ 0x0D4F, 0x0D56 ],
    [ 0x0D58, 0x0D5F ],
    [ 0x0D64, 0x0D65 ],
    [ 0x0D76, 0x0D78 ],
    [ 0x0D80, 0x0D81 ],
    [ 0x0D84, 0x0D84 ],
    [ 0x0D97, 0x0D99 ],
    [ 0x0DB2, 0x0DB2 ],
    [ 0x0DBC, 0x0DBC ],
    [ 0x0DBE, 0x0DBF ],
    [ 0x0DC7, 0x0DC9 ],
    [ 0x0DCB, 0x0DCE ],
    [ 0x0DD5, 0x0DD5 ],
    [ 0x0DD7, 0x0DD7 ],
    [ 0x0DE0, 0x0DF1 ],
    [ 0x0DF5, 0x0E00 ],
    [ 0x0E3B, 0x0E3E ],
    [ 0x0E5C, 0x0E80 ],
    [ 0x0E83, 0x0E83 ],
    [ 0x0E85, 0x0E86 ],
    [ 0x0E89, 0x0E89 ],
    [ 0x0E8B, 0x0E8C ],
    [ 0x0E8E, 0x0E93 ],
    [ 0x0E98, 0x0E98 ],
    [ 0x0EA0, 0x0EA0 ],
    [ 0x0EA4, 0x0EA4 ],
    [ 0x0EA6, 0x0EA6 ],
    [ 0x0EA8, 0x0EA9 ],
    [ 0x0EAC, 0x0EAC ],
    [ 0x0EBA, 0x0EBA ],
    [ 0x0EBE, 0x0EBF ],
    [ 0x0EC5, 0x0EC5 ],
    [ 0x0EC7, 0x0EC7 ],
    [ 0x0ECE, 0x0ECF ],
    [ 0x0EDA, 0x0EDB ],
    [ 0x0EDE, 0x0EFF ],
    [ 0x0F48, 0x0F48 ],
    [ 0x0F6D, 0x0F70 ],
    [ 0x0F98, 0x0F98 ],
    [ 0x0FBD, 0x0FBD ],
    [ 0x0FCD, 0x0FCD ],
    [ 0x0FDB, 0x0FFF ],
    [ 0x10C6, 0x10CF ],
    [ 0x10FD, 0x10FF ],
    [ 0x1249, 0x1249 ],
    [ 0x124E, 0x124F ],
    [ 0x1257, 0x1257 ],
    [ 0x1259, 0x1259 ],
    [ 0x125E, 0x125F ],
    [ 0x1289, 0x1289 ],
    [ 0x128E, 0x128F ],
    [ 0x12B1, 0x12B1 ],
    [ 0x12B6, 0x12B7 ],
    [ 0x12BF, 0x12BF ],
    [ 0x12C1, 0x12C1 ],
    [ 0x12C6, 0x12C7 ],
    [ 0x12D7, 0x12D7 ],
    [ 0x1311, 0x1311 ],
    [ 0x1316, 0x1317 ],
    [ 0x135B, 0x135C ],
    [ 0x137D, 0x137F ],
    [ 0x139A, 0x139F ],
    [ 0x13F5, 0x13FF ],
    [ 0x169D, 0x169F ],
    [ 0x16F1, 0x16FF ],
    [ 0x170D, 0x170D ],
    [ 0x1715, 0x171F ],
    [ 0x1737, 0x173F ],
    [ 0x1754, 0x175F ],
    [ 0x176D, 0x176D ],
    [ 0x1771, 0x1771 ],
    [ 0x1774, 0x177F ],
    [ 0x17DE, 0x17DF ],
    [ 0x17EA, 0x17EF ],
    [ 0x17FA, 0x17FF ],
    [ 0x180F, 0x180F ],
    [ 0x181A, 0x181F ],
    [ 0x1878, 0x187F ],
    [ 0x18AB, 0x18AF ],
    [ 0x18F6, 0x18FF ],
    [ 0x191D, 0x191F ],
    [ 0x192C, 0x192F ],
    [ 0x193C, 0x193F ],
    [ 0x1941, 0x1943 ],
    [ 0x196E, 0x196F ],
    [ 0x1975, 0x197F ],
    [ 0x19AC, 0x19AF ],
    [ 0x19CA, 0x19CF ],
    [ 0x19DB, 0x19DD ],
    [ 0x1A1C, 0x1A1D ],
    [ 0x1A5F, 0x1A5F ],
    [ 0x1A7D, 0x1A7E ],
    [ 0x1A8A, 0x1A8F ],
    [ 0x1A9A, 0x1A9F ],
    [ 0x1AAE, 0x1AFF ],
    [ 0x1B4C, 0x1B4F ],
    [ 0x1B7D, 0x1B7F ],
    [ 0x1BAB, 0x1BAD ],
    [ 0x1BBA, 0x1BBF ],
    [ 0x1BF4, 0x1BFB ],
    [ 0x1C38, 0x1C3A ],
    [ 0x1C4A, 0x1C4C ],
    [ 0x1C80, 0x1CCF ],
    [ 0x1CF3, 0x1CFF ],
    [ 0x1DE7, 0x1DFB ],
    [ 0x1F16, 0x1F17 ],
    [ 0x1F1E, 0x1F1F ],
    [ 0x1F46, 0x1F47 ],
    [ 0x1F4E, 0x1F4F ],
    [ 0x1F58, 0x1F58 ],
    [ 0x1F5A, 0x1F5A ],
    [ 0x1F5C, 0x1F5C ],
    [ 0x1F5E, 0x1F5E ],
    [ 0x1F7E, 0x1F7F ],
    [ 0x1FB5, 0x1FB5 ],
    [ 0x1FC5, 0x1FC5 ],
    [ 0x1FD4, 0x1FD5 ],
    [ 0x1FDC, 0x1FDC ],
    [ 0x1FF0, 0x1FF1 ],
    [ 0x1FF5, 0x1FF5 ],
    [ 0x1FFF, 0x1FFF ],
    [ 0x2065, 0x2069 ],
    [ 0x2072, 0x2073 ],
    [ 0x208F, 0x208F ],
    [ 0x209D, 0x209F ],
    [ 0x20BA, 0x20CF ],
    [ 0x20F1, 0x20FF ],
    [ 0x218A, 0x218F ],
    [ 0x23F4, 0x23FF ],
    [ 0x2427, 0x243F ],
    [ 0x244B, 0x245F ],
    [ 0x2700, 0x2700 ],
    [ 0x27CB, 0x27CB ],
    [ 0x27CD, 0x27CD ],
    [ 0x2B4D, 0x2B4F ],
    [ 0x2B5A, 0x2BFF ],
    [ 0x2C2F, 0x2C2F ],
    [ 0x2C5F, 0x2C5F ],
    [ 0x2CF2, 0x2CF8 ],
    [ 0x2D26, 0x2D2F ],
    [ 0x2D66, 0x2D6E ],
    [ 0x2D71, 0x2D7E ],
    [ 0x2D97, 0x2D9F ],
    [ 0x2DA7, 0x2DA7 ],
    [ 0x2DAF, 0x2DAF ],
    [ 0x2DB7, 0x2DB7 ],
    [ 0x2DBF, 0x2DBF ],
    [ 0x2DC7, 0x2DC7 ],
    [ 0x2DCF, 0x2DCF ],
    [ 0x2DD7, 0x2DD7 ],
    [ 0x2DDF, 0x2DDF ],
    [ 0x2E32, 0x2E7F ],
    [ 0x2E9A, 0x2E9A ],
    [ 0x2EF4, 0x2EFF ],
    [ 0x2FD6, 0x2FEF ],
    [ 0x2FFC, 0x2FFF ],
    [ 0x3040, 0x3040 ],
    [ 0x3097, 0x3098 ],
    [ 0x3100, 0x3104 ],
    [ 0x312E, 0x3130 ],
    [ 0x318F, 0x318F ],
    [ 0x31BB, 0x31BF ],
    [ 0x31E4, 0x31EF ],
    [ 0x321F, 0x321F ],
    [ 0x32FF, 0x32FF ],
    [ 0x4DB6, 0x4DBF ],
    [ 0x9FCC, 0x9FFF ],
    [ 0xA48D, 0xA48F ],
    [ 0xA4C7, 0xA4CF ],
    [ 0xA62C, 0xA63F ],
    [ 0xA674, 0xA67B ],
    [ 0xA698, 0xA69F ],
    [ 0xA6F8, 0xA6FF ],
    [ 0xA78F, 0xA78F ],
    [ 0xA792, 0xA79F ],
    [ 0xA7AA, 0xA7F9 ],
    [ 0xA82C, 0xA82F ],
    [ 0xA83A, 0xA83F ],
    [ 0xA878, 0xA87F ],
    [ 0xA8C5, 0xA8CD ],
    [ 0xA8DA, 0xA8DF ],
    [ 0xA8FC, 0xA8FF ],
    [ 0xA954, 0xA95E ],
    [ 0xA97D, 0xA97F ],
    [ 0xA9CE, 0xA9CE ],
    [ 0xA9DA, 0xA9DD ],
    [ 0xA9E0, 0xA9FF ],
    [ 0xAA37, 0xAA3F ],
    [ 0xAA4E, 0xAA4F ],
    [ 0xAA5A, 0xAA5B ],
    [ 0xAA7C, 0xAA7F ],
    [ 0xAAC3, 0xAADA ],
    [ 0xAAE0, 0xAB00 ],
    [ 0xAB07, 0xAB08 ],
    [ 0xAB0F, 0xAB10 ],
    [ 0xAB17, 0xAB1F ],
    [ 0xAB27, 0xAB27 ],
    [ 0xAB2F, 0xABBF ],
    [ 0xABEE, 0xABEF ],
    [ 0xABFA, 0xABFF ],
    [ 0xD7A4, 0xD7AF ],
    [ 0xD7C7, 0xD7CA ],
    [ 0xD7FC, 0xD7FF ],
    [ 0xFA2E, 0xFA2F ],
    [ 0xFA6E, 0xFA6F ],
    [ 0xFADA, 0xFAFF ],
    [ 0xFB07, 0xFB12 ],
    [ 0xFB18, 0xFB1C ],
    [ 0xFB37, 0xFB37 ],
    [ 0xFB3D, 0xFB3D ],
    [ 0xFB3F, 0xFB3F ],
    [ 0xFB42, 0xFB42 ],
    [ 0xFB45, 0xFB45 ],
    [ 0xFBC2, 0xFBD2 ],
    [ 0xFD40, 0xFD4F ],
    [ 0xFD90, 0xFD91 ],
    [ 0xFDC8, 0xFDEF ],
    [ 0xFDFE, 0xFDFF ],
    [ 0xFE1A, 0xFE1F ],
    [ 0xFE27, 0xFE2F ],
    [ 0xFE53, 0xFE53 ],
    [ 0xFE67, 0xFE67 ],
    [ 0xFE6C, 0xFE6F ],
    [ 0xFE75, 0xFE75 ],
    [ 0xFEFD, 0xFEFE ],
    [ 0xFF00, 0xFF00 ],
    [ 0xFFBF, 0xFFC1 ],
    [ 0xFFC8, 0xFFC9 ],
    [ 0xFFD0, 0xFFD1 ],
    [ 0xFFD8, 0xFFD9 ],
    [ 0xFFDD, 0xFFDF ],
    [ 0xFFE7, 0xFFE7 ],
    [ 0xFFEF, 0xFFF8 ],
    [ 0xFFFE, 0xFFFF ],
    [ 0x1000C, 0x1000C ],
    [ 0x10027, 0x10027 ],
    [ 0x1003B, 0x1003B ],
    [ 0x1003E, 0x1003E ],
    [ 0x1004E, 0x1004F ],
    [ 0x1005E, 0x1007F ],
    [ 0x100FB, 0x100FF ],
    [ 0x10103, 0x10106 ],
    [ 0x10134, 0x10136 ],
    [ 0x1018B, 0x1018F ],
    [ 0x1019C, 0x101CF ],
    [ 0x101FE, 0x1027F ],
    [ 0x1029D, 0x1029F ],
    [ 0x102D1, 0x102FF ],
    [ 0x1031F, 0x1031F ],
    [ 0x10324, 0x1032F ],
    [ 0x1034B, 0x1037F ],
    [ 0x1039E, 0x1039E ],
    [ 0x103C4, 0x103C7 ],
    [ 0x103D6, 0x103FF ],
    [ 0x1049E, 0x1049F ],
    [ 0x104AA, 0x107FF ],
    [ 0x10806, 0x10807 ],
    [ 0x10809, 0x10809 ],
    [ 0x10836, 0x10836 ],
    [ 0x10839, 0x1083B ],
    [ 0x1083D, 0x1083E ],
    [ 0x10856, 0x10856 ],
    [ 0x10860, 0x108FF ],
    [ 0x1091C, 0x1091E ],
    [ 0x1093A, 0x1093E ],
    [ 0x10940, 0x109FF ],
    [ 0x10A04, 0x10A04 ],
    [ 0x10A07, 0x10A0B ],
    [ 0x10A14, 0x10A14 ],
    [ 0x10A18, 0x10A18 ],
    [ 0x10A34, 0x10A37 ],
    [ 0x10A3B, 0x10A3E ],
    [ 0x10A48, 0x10A4F ],
    [ 0x10A59, 0x10A5F ],
    [ 0x10A80, 0x10AFF ],
    [ 0x10B36, 0x10B38 ],
    [ 0x10B56, 0x10B57 ],
    [ 0x10B73, 0x10B77 ],
    [ 0x10B80, 0x10BFF ],
    [ 0x10C49, 0x10E5F ],
    [ 0x10E7F, 0x10FFF ],
    [ 0x1104E, 0x11051 ],
    [ 0x11070, 0x1107F ],
    [ 0x110C2, 0x11FFF ],
    [ 0x1236F, 0x123FF ],
    [ 0x12463, 0x1246F ],
    [ 0x12474, 0x12FFF ],
    [ 0x1342F, 0x167FF ],
    [ 0x16A39, 0x1AFFF ],
    [ 0x1B002, 0x1CFFF ],
    [ 0x1D0F6, 0x1D0FF ],
    [ 0x1D127, 0x1D128 ],
    [ 0x1D1DE, 0x1D1FF ],
    [ 0x1D246, 0x1D2FF ],
    [ 0x1D357, 0x1D35F ],
    [ 0x1D372, 0x1D3FF ],
    [ 0x1D455, 0x1D455 ],
    [ 0x1D49D, 0x1D49D ],
    [ 0x1D4A0, 0x1D4A1 ],
    [ 0x1D4A3, 0x1D4A4 ],
    [ 0x1D4A7, 0x1D4A8 ],
    [ 0x1D4AD, 0x1D4AD ],
    [ 0x1D4BA, 0x1D4BA ],
    [ 0x1D4BC, 0x1D4BC ],
    [ 0x1D4C4, 0x1D4C4 ],
    [ 0x1D506, 0x1D506 ],
    [ 0x1D50B, 0x1D50C ],
    [ 0x1D515, 0x1D515 ],
    [ 0x1D51D, 0x1D51D ],
    [ 0x1D53A, 0x1D53A ],
    [ 0x1D53F, 0x1D53F ],
    [ 0x1D545, 0x1D545 ],
    [ 0x1D547, 0x1D549 ],
    [ 0x1D551, 0x1D551 ],
    [ 0x1D6A6, 0x1D6A7 ],
    [ 0x1D7CC, 0x1D7CD ],
    [ 0x1D800, 0x1EFFF ],
    [ 0x1F02C, 0x1F02F ],
    [ 0x1F094, 0x1F09F ],
    [ 0x1F0AF, 0x1F0B0 ],
    [ 0x1F0BF, 0x1F0C0 ],
    [ 0x1F0D0, 0x1F0D0 ],
    [ 0x1F0E0, 0x1F0FF ],
    [ 0x1F10B, 0x1F10F ],
    [ 0x1F12F, 0x1F12F ],
    [ 0x1F16A, 0x1F16F ],
    [ 0x1F19B, 0x1F1E5 ],
    [ 0x1F203, 0x1F20F ],
    [ 0x1F23B, 0x1F23F ],
    [ 0x1F249, 0x1F24F ],
    [ 0x1F252, 0x1F2FF ],
    [ 0x1F321, 0x1F32F ],
    [ 0x1F336, 0x1F336 ],
    [ 0x1F37D, 0x1F37F ],
    [ 0x1F394, 0x1F39F ],
    [ 0x1F3C5, 0x1F3C5 ],
    [ 0x1F3CB, 0x1F3DF ],
    [ 0x1F3F1, 0x1F3FF ],
    [ 0x1F43F, 0x1F43F ],
    [ 0x1F441, 0x1F441 ],
    [ 0x1F4F8, 0x1F4F8 ],
    [ 0x1F4FD, 0x1F4FF ],
    [ 0x1F53E, 0x1F54F ],
    [ 0x1F568, 0x1F5FA ],
    [ 0x1F600, 0x1F600 ],
    [ 0x1F611, 0x1F611 ],
    [ 0x1F615, 0x1F615 ],
    [ 0x1F617, 0x1F617 ],
    [ 0x1F619, 0x1F619 ],
    [ 0x1F61B, 0x1F61B ],
    [ 0x1F61F, 0x1F61F ],
    [ 0x1F626, 0x1F627 ],
    [ 0x1F62C, 0x1F62C ],
    [ 0x1F62E, 0x1F62F ],
    [ 0x1F634, 0x1F634 ],
    [ 0x1F641, 0x1F644 ],
    [ 0x1F650, 0x1F67F ],
    [ 0x1F6C6, 0x1F6FF ],
    [ 0x1F774, 0x1FFFF ],
    [ 0x2A6D7, 0x2A6FF ],
    [ 0x2B735, 0x2B73F ],
    [ 0x2B81E, 0x2F7FF ],
    [ 0x2FA1E, 0xE0000 ],
    [ 0xE0002, 0xE001F ],
    [ 0xE0080, 0xE00FF ],
    [ 0xE01F0, 0xEFFFF ],
    [ 0xFFFFE, 0xFFFFF ],
    [ 0x10FFFE, 0x10FFFF ],
    ];

    return binarySearch!table(c);
}

unittest
{
}


//==============================================================================
// Private Section.
//==============================================================================
private:

bool binarySearch(alias table)(dchar c) @safe pure nothrow
{
    static @property bool checkTableEntry(alias table)()
    {
        foreach(i, entry; table)
        {
            assert(table[i][0] <= table[i][1]);
            if(i < table.length - 1)
                assert(table[i][1] < table[i + 1][0]);
        }
        return true;
    }
    static assert(checkTableEntry!table());

    return binarySearch2(c, table);
}

bool binarySearch2(dchar c, immutable dchar[2][] table) @safe pure nothrow
{
    // Binary search
    size_t mid;
    size_t low;
    size_t high;

    low = 0;
    high = table.length - 1;
    while(cast(int)low <= cast(int)high)
    {
        mid = (low + high) >> 1;

        if(c < table[mid][0])
            high = mid - 1;
        else if(c > table[mid][1])
            low = mid + 1;
        else
            goto Lis;
    }

Lisnot:
    debug
    {
        for(size_t i = 0; i < table.length; ++i)
            assert(c < table[i][0] || c > table[i][1]);
    }

    return false;

Lis:
    debug
    {
        for(size_t i = 0; i < table.length; ++i)
        {
            if(c >= table[i][0] && c <= table[i][1])
                return true;
        }

        assert(0);      // should have been in table
    }
    else
        return true;
};
