/******************************************************************************
 *
 * Project:  CPL
 * Purpose:  Implementation of a thread-safe queue
 * Author:   Even Rouault <even dot rouault at spatialys.com>
 *
 ******************************************************************************
 * Copyright (c) 2022, Even Rouault <even dot rouault at spatialys.com>
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 ****************************************************************************/

#ifndef CPL_THREADSAFE_QUEUE_INCLUDED
#define CPL_THREADSAFE_QUEUE_INCLUDED

#include <condition_variable>
#include <mutex>
#include <queue>

namespace cpl
{
template <class T> class ThreadSafeQueue
{
private:
    mutable std::mutex      m_mutex{};
    std::condition_variable m_cv{};
    std::queue<T>           m_queue{};

public:
    ThreadSafeQueue() = default;

    void clear()
    {
        std::lock_guard<std::mutex> lock(m_mutex);
        while( !m_queue.empty() )
            m_queue.pop();
    }

    bool empty() const
    {
        std::lock_guard<std::mutex> lock(m_mutex);
        return m_queue.empty();
    }

    size_t size() const
    {
        std::lock_guard<std::mutex> lock(m_mutex);
        return m_queue.size();
    }

    void push(const T& value)
    {
        std::lock_guard<std::mutex> lock(m_mutex);
        m_queue.push(value);
        m_cv.notify_one();
    }

    void push(T&& value)
    {
        std::lock_guard<std::mutex> lock(m_mutex);
        m_queue.push(std::move(value));
        m_cv.notify_one();
    }

    T get_and_pop_front()
    {
        std::unique_lock<std::mutex> lock(m_mutex);
        while( m_queue.empty() )
        {
            m_cv.wait(lock);
        }
        T val = m_queue.front();
        m_queue.pop();
        return val;
    }
};

} // namespace cpl

#endif // CPL_THREADSAFE_QUEUE_INCLUDED
