/*
** Copyright (C) 10 Feb 1999 Jonas Munsin <jmunsin@iki.fi>
**  
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
** 
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
** 
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software 
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <gtk/gtk.h>
#include <sys/time.h>
#include <unistd.h>

#include "generic_output_handler.h"
#include "locks.h"
#include "linebuffer.h"
#include "readers.h"
#include "progressbars_gui.h"
#include "preferences.h"
#include "vector_commands.h"
#include "contractions.h"
#include "progressbars.h"
#include "globals.h"

static GSList *cdda2wav_cmds;
static cmd_v *cdrecord_audio_track_nofix, *cdrecord_fixate;

static int track_now, total_tracks, cdda2wav_burn;

/* TODO: these (and the functions using them) should be moved to a new file
 * when the great restructuring is done */
static two_pipe_handlers mkisofs_pipe_handlers;
static pipe_handler mkisofs_stdout_handler;
static pipe_handler mkisofs_stderr_handler;

/* functions for make_image_only */

static void close_mkisofs_f(void) {
	if (misc_prefs.own_progresswin) {
		gtk_widget_destroy(progress_win);
		progress_win = NULL;
	} else {
		gtk_widget_set_sensitive(progress_vbox, FALSE);
	}

	if (ask_win) {
		gtk_widget_destroy(ask_win);
		ask_win = 0;
	}

	destroy_buffers();

	not_running();
}

static int wait_mkisofs_output(gpointer data) {
	static int stdout_ok = TRUE, stderr_ok = TRUE;
	fd_set rset;
	struct timeval tv;
	int result, max;
	int *fd = data;

	g_assert((stdout_ok == TRUE) || (stderr_ok == TRUE));

	tv.tv_sec = 0;
	tv.tv_usec = 0;

	FD_ZERO(&rset);
	max = -1;

	if (stdout_ok) {
		FD_SET(fd[0], &rset);
		max = fd[0];
	}
	if (stderr_ok) {
		FD_SET(fd[1], &rset);
		if (-1 == max)
			max = fd[1];
		else
			max = fd[0]>fd[1] ? fd[0] : fd[1];
	}

	result = select(max + 1, &rset, NULL, NULL, &tv);

	if (result < 0)
		g_error("wait_mkisofs_output: select returned negative!");
	else if (result == 0)
		return TRUE;

	if (stdout_ok && FD_ISSET(fd[0], &rset))
		stdout_ok = ni_read_mkisofs_stdout(fd[0]);
	if (stderr_ok && FD_ISSET(fd[1], &rset))
		stderr_ok = ni_read_mkisofs_stderr(fd[1]);

	if (!(stdout_ok || stderr_ok)) {
		wait_pids();
		stdout_ok = stderr_ok = TRUE;
		close_mkisofs_f();
		return FALSE;
	} else
		return TRUE;
}

static void mkisofs_done(void) {
	wait_pids();
	close_mkisofs_f();
}

static void initialize_mkisofs_handlers(int *fds) {
	mkisofs_stdout_handler.fd = fds[0];
	mkisofs_stdout_handler.callback = ni_read_mkisofs_stdout;
	mkisofs_stdout_handler.status = TRUE;
	mkisofs_pipe_handlers.ph1 = &mkisofs_stdout_handler;

	mkisofs_stderr_handler.fd = fds[1];
	mkisofs_stderr_handler.callback = ni_read_mkisofs_stderr;
	mkisofs_stderr_handler.status = TRUE;
	mkisofs_pipe_handlers.ph2 = &mkisofs_stderr_handler;

	mkisofs_pipe_handlers.end_callback = mkisofs_done;
}

void open_mkisofs_progress_bar(int *output) {
	cdda2wav_burn = FALSE;
	open_progress_win(_("Making image, please wait."));

	init_buffers(BUF_MKISOFS_STDERR | BUF_MKISOFS_STDOUT);

/*	initialize_mkisofs_handlers(output);
	gtk_timeout_add(TIMEOUT_GTK, handle_output, &mkisofs_pipe_handlers); */

	gtk_timeout_add(TIMEOUT_GTK, wait_mkisofs_output, output);
}

/* functions for burn_from_image only */

static void close_cdrecord_f(void) {
	if (misc_prefs.own_progresswin) {
		gtk_widget_destroy(progress_win);
		progress_win = NULL;
	} else {
		gtk_widget_set_sensitive(progress_vbox, FALSE);
	}
	if (ask_win) {
		gtk_widget_destroy(ask_win);
		ask_win = 0;
	}

	destroy_buffers();

	not_running();
}

static int wait_cdrecord_output(gpointer data) {
	static int stdout_ok = TRUE, stderr_ok = TRUE;
	fd_set rset;
	struct timeval tv;
	int result, max;
	int *fd = data;

	g_assert((stdout_ok == TRUE) || (stderr_ok == TRUE));

	tv.tv_sec = 0;
	tv.tv_usec = 0;

	FD_ZERO(&rset);
	max = -1;

	if (stdout_ok) {
		FD_SET(fd[0], &rset);
		max = fd[0];
	}
	if (stderr_ok) {
		FD_SET(fd[1], &rset);
		if (-1 == max)
			max = fd[1];
		else
			max = fd[0]>fd[1] ? fd[0] : fd[1];
	}

	result = select(max + 1, &rset, NULL, NULL, &tv);

	if (result < 0)
		g_error("wait_cdrecord_output: select returned negative!");
	else if (result == 0)
		return TRUE;

	if (stdout_ok && FD_ISSET(fd[0], &rset))
		stdout_ok = ni_read_cdrecord_stdout(fd[0]);
	if (stderr_ok && FD_ISSET(fd[1], &rset))
		stderr_ok = ni_read_cdrecord_stderr(fd[1], FALSE);

	if (!(stdout_ok || stderr_ok)) {
		wait_pids();
		stdout_ok = stderr_ok = TRUE;
		close_cdrecord_f();
		return FALSE;
	} else
		return TRUE;
}

void open_cdrecord_progress_bar(int *output) {
	countdown_counter = 9;
	cdda2wav_burn = FALSE;
	open_progress_win(_("Waiting for burn to start\n"));

	init_buffers(BUF_CDRECORD_STDERR | BUF_CDRECORD_STDOUT); 
	gtk_timeout_add(TIMEOUT_GTK, wait_cdrecord_output, output);
}

/* functions for make_image_and_burn */

static void close_mkisofs_cdrecord(void) {
	if (misc_prefs.own_progresswin) {
		gtk_widget_destroy(progress_win);
		progress_win = NULL;
	} else {
		gtk_widget_set_sensitive(progress_vbox, FALSE);
	}
	if (ask_win) {
		gtk_widget_destroy(ask_win);
		ask_win = 0;
	}

	destroy_buffers();

	not_running();
}

static cmd_v *get_next_cdda2wav(void) {
	GSList *e;
	cmd_v *cmd;

	if (g_slist_length(cdda2wav_cmds) == 0) {
		g_slist_free(cdda2wav_cmds);
		cdda2wav_cmds = NULL;
		return NULL;
	}

	e = g_slist_nth(cdda2wav_cmds, 0);
	cmd = (cmd_v *)(e->data);
	cdda2wav_cmds = g_slist_remove_link(cdda2wav_cmds, e);
	g_slist_free(e);

	return cmd;
}


static int wait_mkisofs_cdrecord_output(gpointer data) {
	static int iso_stderr_ok = TRUE, cdr_stdout_ok = TRUE, cdr_stderr_ok = TRUE;
	fd_set rset;
	struct timeval tv;
	int result;
	int max;
	int *fd = data;

	g_assert((iso_stderr_ok == TRUE) || (cdr_stdout_ok == TRUE) || (cdr_stderr_ok == TRUE));

	tv.tv_sec = 0;
	tv.tv_usec = 0;

	FD_ZERO(&rset);
	max = -1;

/* [0] is mkisofs's STDERR,
 * [1] is cdrecord's STDOUT and
 * [2] is cdrecord's STDERR
 */
	if (iso_stderr_ok) {
		FD_SET(fd[0], &rset);
		max = fd[0];
	}
	if (cdr_stdout_ok) {
		FD_SET(fd[1], &rset);
		if (-1 == max)
			max = fd[1];
		else
			max = fd[0]>fd[1] ? fd[0] : fd[1];
	}
	if (cdr_stderr_ok) {
		FD_SET(fd[2], &rset);
		if (-1 == max)
			max = fd[2];
		else
			max = max>fd[2] ? max : fd[2];
	}

	result = select(max + 1, &rset, NULL, NULL, &tv);
	if (result < 0)
		g_warning("%s::%i: wait_mkisofs_cdrecord_output: select returned negative!", __FILE__, __LINE__);
	else if (result == 0) {
/* FIXME: bailed_out must be tested so we know when we've been told to quit by the
 * user, the current redirection hack doesn't seem to recognize an abrupt kill() */
 		if (bailed_out) {
			iso_stderr_ok = cdr_stdout_ok = cdr_stderr_ok = TRUE;
			close_mkisofs_cdrecord();
			return FALSE;
		} else {
			return TRUE;
		}
	}

	if (cdda2wav_burn) {
		if (iso_stderr_ok && FD_ISSET(fd[0], &rset))
			iso_stderr_ok = ni_read_cdda2wav_stderr(fd[0]);
	} else {
		if (iso_stderr_ok && FD_ISSET(fd[0], &rset))
			iso_stderr_ok = ni_read_mkisofs_stderr(fd[0]);
	}
	if (cdr_stdout_ok && FD_ISSET(fd[1], &rset))
		cdr_stdout_ok = ni_read_cdrecord_stdout(fd[1]);
	if (cdr_stderr_ok && FD_ISSET(fd[2], &rset))
		cdr_stderr_ok = ni_read_cdrecord_stderr(fd[2], cdda2wav_burn);

	if (cdda2wav_burn && ((countdown_counter == 0) || ((countdown_counter % 2) == 0)))
		set_track_burning_now_total(track_now, total_tracks);

/*  - 13 Aug: copied most of this function for verify routines and started wondering;
 *            shouldn't the &&'s be || instead? changing them to || leaves the progress
 *            win open - maybe bad interaction with the waitpids or something ?
 */
	if (!(cdr_stdout_ok && cdr_stderr_ok && iso_stderr_ok)) {
		int *output = NULL;
		wait_pids();
		countdown_counter = 9;
		iso_stderr_ok = cdr_stdout_ok = cdr_stderr_ok = TRUE;

		if (cdda2wav_burn) {
			cmd_v *cd;

			cd = get_next_cdda2wav();
			destroy_buffers();
			init_buffers(BUF_MKISOFS_STDERR | BUF_MKISOFS_STDOUT
					| BUF_CDRECORD_STDERR | BUF_CDRECORD_STDOUT); 

			if (cd == NULL) {
				if (NULL != cdrecord_fixate) {
					print_cmd(cdrecord_fixate);
					if (!(output = popen_re_unbuffered(cdrecord_fixate))) {
						g_warning("%s::%i: error executing command", __FILE__, __LINE__);
						destroy_cmd(cdrecord_fixate);
						destroy_cmd(cdrecord_audio_track_nofix);
						return FALSE;
					}
					destroy_cmd(cdrecord_fixate);
					destroy_cmd(cdrecord_audio_track_nofix);
					set_copy_info("preparing to fixate", "preparing to fixate");
					gtk_timeout_add(TIMEOUT_GTK, wait_cdrecord_output, output);
				} else
					destroy_cmd(cdrecord_audio_track_nofix);
				return FALSE;
			} else {
				print_cmd(cd);
				print_cmd(cdrecord_audio_track_nofix);
				if (!(output = popen_r_err_in(cd, cdrecord_audio_track_nofix))) {
					g_warning("%s::%i: error executing command", __FILE__, __LINE__);
					destroy_cmd(cd);
					return FALSE;
				}
				destroy_cmd(cd);
				set_track_burning_now_total(++track_now, total_tracks);
			}
			gtk_timeout_add(TIMEOUT_GTK, wait_mkisofs_cdrecord_output, output);
		} else {
			close_mkisofs_cdrecord();
		}
		return FALSE;
	} else
		return TRUE;
}

void open_mkisofs_cdrecord_progress_bar(int *output) {
	countdown_counter = 9;
	cdda2wav_burn = FALSE;
	bailed_out = FALSE;
	open_progress_win(_("Waiting for burn to start\n"));

	init_buffers(BUF_MKISOFS_STDERR | BUF_MKISOFS_STDOUT
			| BUF_CDRECORD_STDERR | BUF_CDRECORD_STDOUT); 
	gtk_timeout_add(TIMEOUT_GTK, wait_mkisofs_cdrecord_output, output);
}

void open_cdda2wav_cdrecord_progress_bar(GSList *cdda2wav, cmd_v *cdr, cmd_v *cdr_close) {
	int *output;
	cmd_v *cd;

	cdda2wav_burn = TRUE;

	track_now = 0;
	total_tracks = g_slist_length(cdda2wav);

	cdrecord_audio_track_nofix = cdr;
	cdrecord_fixate = cdr_close;

	cdda2wav_cmds = cdda2wav;

	if (NULL == (cd = get_next_cdda2wav())) {
		g_warning("%s::%i: no cdda2wav commands", __FILE__, __LINE__);
		return;
	}

	print_cmd(cd);
	print_cmd(cdr);

	if (!(output = popen_r_err_in(cd, cdr))) {
		g_warning("%s::%i: error executing command", __FILE__, __LINE__);
		return;
	}

	destroy_cmd(cd);

	countdown_counter = 9;
	bailed_out = FALSE;
	open_progress_win(_("Waiting for burn to start\n"));

	init_buffers(BUF_MKISOFS_STDERR | BUF_MKISOFS_STDOUT
			| BUF_CDRECORD_STDERR | BUF_CDRECORD_STDOUT); 

	set_track_burning_now_total(++track_now, total_tracks);

	gtk_timeout_add(TIMEOUT_GTK, wait_mkisofs_cdrecord_output, output);
}

