/*
 * Copyright 2022 Nico Sonack <nsonack@herrhotzenplotz.de>
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above
 * copyright notice, this list of conditions and the following
 * disclaimer in the documentation and/or other materials provided
 * with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <config.h>

#include <gcli/cmd/ci.h>
#include <gcli/cmd/cmd.h>
#include <gcli/cmd/cmdconfig.h>
#include <gcli/cmd/colour.h>
#include <gcli/cmd/comment.h>
#include <gcli/cmd/editor.h>
#include <gcli/cmd/gitconfig.h>
#include <gcli/cmd/pipelines.h>
#include <gcli/cmd/pulls.h>
#include <gcli/cmd/table.h>

#include <gcli/comments.h>
#include <gcli/forges.h>
#include <gcli/gitlab/pipelines.h>
#include <gcli/pulls.h>

#ifdef HAVE_GETOPT_H
#include <getopt.h>
#endif

#include <assert.h>
#include <stdlib.h>

static void
usage(void)
{
	fprintf(stderr, "usage: gcli pulls create [-o owner -r repo] [-f from]\n");
	fprintf(stderr, "                         [-t to] [-d] [-l label] pull-request-title\n");
	fprintf(stderr, "       gcli pulls [-o owner -r repo] [-a] [-A author] [-n number]\n");
	fprintf(stderr, "                  [-L label] [-s]\n");
	fprintf(stderr, "       gcli pulls [-o owner -r repo] -i pull-id actions...\n");
	fprintf(stderr, "OPTIONS:\n");
	fprintf(stderr, "  -o owner        The repository owner\n");
	fprintf(stderr, "  -r repo         The repository name\n");
	fprintf(stderr, "  -a              Fetch everything including closed and merged PRs\n");
	fprintf(stderr, "  -A author       Filter pull requests by the given author\n");
	fprintf(stderr, "  -L label        Filter pull requests by the given label\n");
	fprintf(stderr, "  -d              Mark newly created PR as a draft\n");
	fprintf(stderr, "  -f owner:branch Specify the owner and branch of the fork that is the head of a PR.\n");
	fprintf(stderr, "  -l label        Add the given label when creating the PR\n");
	fprintf(stderr, "  -n number       Number of PRs to fetch (-1 = everything)\n");
	fprintf(stderr, "  -i id           ID of PR to perform actions on\n");
	fprintf(stderr, "  -s              Print (sort) in reverse order\n");
	fprintf(stderr, "  -t branch       Specify target branch of the PR\n");
	fprintf(stderr, "  -y              Do not ask for confirmation.\n");
	fprintf(stderr, "ACTIONS:\n");
	fprintf(stderr, "  all             Display status, commits, op and checks of the PR\n");
	fprintf(stderr, "  op              Display original post\n");
	fprintf(stderr, "  status          Display PR metadata\n");
	fprintf(stderr, "  comments        Display comments\n");
	fprintf(stderr, "  notes           Alias for notes\n");
	fprintf(stderr, "  commits         Display commits of the PR\n");
	fprintf(stderr, "  ci              Display CI/Pipeline status information about the PR\n");
	fprintf(stderr, "  merge [-s] [-D] Merge the PR (-s = squash commits, -d = inhibit deleting source branch)\n");
	fprintf(stderr, "  milestone <id>  Assign this PR to a milestone\n");
	fprintf(stderr, "  milestone -d    Clear associated milestones from the PR\n");
	fprintf(stderr, "  close           Close the PR\n");
	fprintf(stderr, "  reopen          Reopen a closed PR\n");
	fprintf(stderr, "  labels ...      Add or remove labels:\n");
	fprintf(stderr, "                     add <name>\n");
	fprintf(stderr, "                     remove <name>\n");
	fprintf(stderr, "  diff            Display changes as diff\n");

	fprintf(stderr, "\n");
	version();
	copyright();
}

void
gcli_print_pulls(enum gcli_output_flags const flags,
                 gcli_pull_list const *const list, int const max)
{
	int n;
	gcli_tbl table;
	gcli_tblcoldef cols[] = {
		{ .name = "NUMBER",  .type = GCLI_TBLCOLTYPE_INT,    .flags = GCLI_TBLCOL_JUSTIFYR },
		{ .name = "STATE",   .type = GCLI_TBLCOLTYPE_STRING, .flags = GCLI_TBLCOL_STATECOLOURED },
		{ .name = "MERGED",  .type = GCLI_TBLCOLTYPE_BOOL,   .flags = 0 },
		{ .name = "CREATOR", .type = GCLI_TBLCOLTYPE_STRING, .flags = GCLI_TBLCOL_BOLD },
		{ .name = "NOTES",   .type = GCLI_TBLCOLTYPE_INT,    .flags = GCLI_TBLCOL_JUSTIFYR },
		{ .name = "TITLE",   .type = GCLI_TBLCOLTYPE_STRING, .flags = 0 },
	};

	if (list->pulls_size == 0) {
		puts("No Pull Requests");
		return;
	}

	/* Determine number of items to print */
	if (max < 0 || (size_t)(max) > list->pulls_size)
		n = list->pulls_size;
	else
		n = max;

	/* Fill the table */
	table = gcli_tbl_begin(cols, ARRAY_SIZE(cols));
	if (!table)
		errx(1, "error: cannot init table");

	if (flags & OUTPUT_SORTED) {
		for (int i = 0; i < n; ++i) {
			gcli_tbl_add_row(table,
			                 list->pulls[n-i-1].number,
			                 list->pulls[n-i-1].state,
			                 list->pulls[n-i-1].merged,
			                 list->pulls[n-i-1].author,
			                 list->pulls[n-i-1].comments,
			                 list->pulls[n-i-1].title);
		}
	} else {
		for (int i = 0; i < n; ++i) {
			gcli_tbl_add_row(table,
			                 list->pulls[i].number,
			                 list->pulls[i].state,
			                 list->pulls[i].merged,
			                 list->pulls[i].author,
			                 list->pulls[i].comments,
			                 list->pulls[i].title);
		}
	}

	gcli_tbl_end(table);
}

int
gcli_print_pull_diff(FILE *stream, char const *owner, char const *reponame,
                     int pr_number)
{
	return gcli_pull_get_diff(g_clictx, stream, owner, reponame, pr_number);
}

void
gcli_pull_print(gcli_pull const *const it)
{
	gcli_dict dict;
	gcli_forge_descriptor const *const forge = gcli_forge(g_clictx);
    int const quirks = forge->pull_summary_quirks;

	dict = gcli_dict_begin();

	gcli_dict_add(dict,        "NUMBER", 0, 0, "%d", it->number);
	gcli_dict_add_string(dict, "TITLE", 0, 0, it->title);
	gcli_dict_add_string(dict, "HEAD", 0, 0, it->head_label);
	gcli_dict_add_string(dict, "BASE", 0, 0, it->base_label);
	gcli_dict_add_string(dict, "CREATED", 0, 0, it->created_at);
	gcli_dict_add_string(dict, "AUTHOR", GCLI_TBLCOL_BOLD, 0, it->author);
	gcli_dict_add_string(dict, "STATE", GCLI_TBLCOL_STATECOLOURED, 0, it->state);
	gcli_dict_add(dict,        "COMMENTS", 0, 0, "%d", it->comments);

	if (it->milestone)
		gcli_dict_add_string(dict, "MILESTONE", 0, 0, it->milestone);

	if ((quirks & GCLI_PRS_QUIRK_ADDDEL) == 0)
		/* FIXME: move printing colours into the dictionary printer? */
		gcli_dict_add(dict, "ADD:DEL", 0, 0, "%s%d%s:%s%d%s",
		              gcli_setcolour(GCLI_COLOR_GREEN),
		              it->additions,
		              gcli_resetcolour(),
		              gcli_setcolour(GCLI_COLOR_RED),
		              it->deletions,
		              gcli_resetcolour());

	if ((quirks & GCLI_PRS_QUIRK_COMMITS) == 0)
		gcli_dict_add(dict, "COMMITS", 0, 0, "%d", it->commits);

	if ((quirks & GCLI_PRS_QUIRK_CHANGES) == 0)
		gcli_dict_add(dict, "CHANGED", 0, 0, "%d", it->changed_files);

	if ((quirks & GCLI_PRS_QUIRK_MERGED) == 0)
		gcli_dict_add_string(dict, "MERGED", 0, 0, sn_bool_yesno(it->merged));

	gcli_dict_add_string(dict, "MERGEABLE", 0, 0, sn_bool_yesno(it->mergeable));
	if ((quirks & GCLI_PRS_QUIRK_DRAFT) == 0)
		gcli_dict_add_string(dict, "DRAFT", 0, 0, sn_bool_yesno(it->draft));

	if ((quirks & GCLI_PRS_QUIRK_COVERAGE) == 0 && it->coverage)
		gcli_dict_add_string(dict, "COVERAGE", 0, 0, it->coverage);

	if (it->labels_size) {
		gcli_dict_add_sv_list(dict, "LABELS", it->labels, it->labels_size);
	} else {
		gcli_dict_add_string(dict, "LABELS", 0, 0, "none");
	}

	gcli_dict_end(dict);
}

void
gcli_pull_print_op(gcli_pull const *const pull)
{
	if (pull->body)
		pretty_print(pull->body, 4, 80, stdout);
}

static void
gcli_print_checks_list(gcli_pull_checks_list const *const list)
{
	switch (list->forge_type) {
	case GCLI_FORGE_GITHUB:
		github_print_checks((github_check_list const *)(list));
		break;
	case GCLI_FORGE_GITLAB:
		gitlab_print_pipelines((gitlab_pipeline_list const*)(list));
		break;
	default:
		assert(0 && "unreachable");
	}
}

int
gcli_pull_checks(char const *owner, char const *repo, int pr_number)
{
	gcli_pull_checks_list list = {0};
	gcli_forge_type t = gcli_config_get_forge_type(g_clictx);

	list.forge_type = t;

	switch (t) {
	case GCLI_FORGE_GITHUB:
	case GCLI_FORGE_GITLAB: {
		int rc = gcli_pull_get_checks(g_clictx, owner, repo, pr_number, &list);
		if (rc < 0)
			return rc;

		gcli_print_checks_list(&list);
		gcli_pull_checks_free(&list);

		return 0;
	} break;
	default:
		puts("No checks.");
		return 0;               /* no CI support / not implemented */
	}
}

/**
 * Get a copy of the first line of the passed string.
 */
static char *
cut_newline(char const *const _it)
{
	char *it = strdup(_it);
	char *foo = it;
	while (*foo) {
		if (*foo == '\n') {
			*foo = 0;
			break;
		}
		foo += 1;
	}

	return it;
}

void
gcli_print_commits(gcli_commit_list const *const list)
{
	gcli_tbl table;
	gcli_tblcoldef cols[] = {
		{ .name = "SHA",     .type = GCLI_TBLCOLTYPE_STRING, .flags = GCLI_TBLCOL_COLOUREXPL },
		{ .name = "AUTHOR",  .type = GCLI_TBLCOLTYPE_STRING, .flags = GCLI_TBLCOL_BOLD },
		{ .name = "EMAIL",   .type = GCLI_TBLCOLTYPE_STRING, .flags = 0 },
		{ .name = "DATE",    .type = GCLI_TBLCOLTYPE_STRING, .flags = 0 },
		{ .name = "MESSAGE", .type = GCLI_TBLCOLTYPE_STRING, .flags = 0 },
	};

	if (list->commits_size == 0) {
		puts("No commits");
		return;
	}

	table = gcli_tbl_begin(cols, ARRAY_SIZE(cols));
	if (!table)
		errx(1, "error: could not initialize table");

	for (size_t i = 0; i < list->commits_size; ++i) {
		char *message = cut_newline(list->commits[i].message);
		gcli_tbl_add_row(table, GCLI_COLOR_YELLOW, list->commits[i].sha,
		                 list->commits[i].author,
		                 list->commits[i].email,
		                 list->commits[i].date,
		                 message);
		free(message);          /* message is copied by the function above */
	}

	gcli_tbl_end(table);
}

int
gcli_pull_commits(char const *owner, char const *repo,
                  int const pr_number)
{
	gcli_commit_list commits = {0};
	int rc = 0;

	rc = gcli_pull_get_commits(g_clictx, owner, repo, pr_number, &commits);
	if (rc < 0)
		return rc;

	gcli_print_commits(&commits);
	gcli_commits_free(&commits);

	return rc;
}

static void
pull_init_user_file(gcli_ctx *ctx, FILE *stream, void *_opts)
{
	gcli_submit_pull_options *opts = _opts;

	(void) ctx;
	fprintf(
		stream,
		"! PR TITLE : "SV_FMT"\n"
		"! Enter PR comments above.\n"
		"! All lines starting with '!' will be discarded.\n",
		SV_ARGS(opts->title));
}

static sn_sv
gcli_pull_get_user_message(gcli_submit_pull_options *opts)
{
	return gcli_editor_get_user_message(g_clictx, pull_init_user_file, opts);
}

static int
create_pull(gcli_submit_pull_options opts, int always_yes)
{
	opts.body = gcli_pull_get_user_message(&opts);

	fprintf(stdout,
	        "The following PR will be created:\n"
	        "\n"
	        "TITLE   : "SV_FMT"\n"
	        "BASE    : "SV_FMT"\n"
	        "HEAD    : "SV_FMT"\n"
	        "IN      : %s/%s\n"
	        "MESSAGE :\n"SV_FMT"\n",
	        SV_ARGS(opts.title),SV_ARGS(opts.to),
	        SV_ARGS(opts.from),
	        opts.owner, opts.repo,
	        SV_ARGS(opts.body));

	fputc('\n', stdout);

	if (!always_yes)
		if (!sn_yesno("Do you want to continue?"))
			errx(1, "PR aborted.");

	return gcli_pull_submit(g_clictx, opts);
}

static sn_sv
pr_try_derive_head(void)
{
	char const *account;
	sn_sv branch  = {0};

	if ((account = gcli_config_get_account_name(g_clictx)) == NULL) {
		errx(1,
		     "error: Cannot derive PR head. Please specify --from or set the\n"
		     "       account in the users gcli config file.\n"
		     "note:  %s",
		     gcli_get_error(g_clictx));
	}

	if (!(branch = gcli_gitconfig_get_current_branch()).length)
		errx(1,
		     "error: Cannot derive PR head. Please specify --from or, if you\n"
		     "       are in »detached HEAD« state, checkout the branch you \n"
		     "       want to pull request.\n");

	return sn_sv_fmt("%s:"SV_FMT, account, SV_ARGS(branch));
}

static int
subcommand_pull_create(int argc, char *argv[])
{
	/* we'll use getopt_long here to parse the arguments */
	int ch;
	gcli_submit_pull_options opts   = {0};
	int always_yes = 0;

	const struct option options[] = {
		{ .name = "from",
		  .has_arg = required_argument,
		  .flag = NULL,
		  .val = 'f' },
		{ .name = "to",
		  .has_arg = required_argument,
		  .flag = NULL,
		  .val = 't' },
		{ .name = "owner",
		  .has_arg = required_argument,
		  .flag = NULL,
		  .val = 'o' },
		{ .name = "repo",
		  .has_arg = required_argument,
		  .flag = NULL,
		  .val = 'r' },
		{ .name = "draft",
		  .has_arg = no_argument,
		  .flag = &opts.draft,
		  .val = 1   },
		{ .name = "label",
		  .has_arg = required_argument,
		  .flag = NULL,
		  .val = 'l' },
		{0},
	};

	while ((ch = getopt_long(argc, argv, "yf:t:do:r:l:", options, NULL)) != -1) {
		switch (ch) {
		case 'f':
			opts.from  = SV(optarg);
			break;
		case 't':
			opts.to    = SV(optarg);
			break;
		case 'd':
			opts.draft = 1;
			break;
		case 'o':
			opts.owner = optarg;
			break;
		case 'r':
			opts.repo = optarg;
			break;
		case 'l': /* add a label */
			opts.labels = realloc(
				opts.labels, sizeof(*opts.labels) * (opts.labels_size + 1));
			opts.labels[opts.labels_size++] = optarg;
			break;
		case 'y':
			always_yes = 1;
			break;
		default:
			usage();
			return EXIT_FAILURE;
		}
	}

	argc -= optind;
	argv += optind;

	if (!opts.from.length)
		opts.from = pr_try_derive_head();

	if (!opts.to.length) {
		if (!(opts.to = gcli_config_get_base(g_clictx)).length)
			errx(1,
			     "error: PR base is missing. Please either specify "
			     "--to branch-name or set pr.base in .gcli.");
	}

	check_owner_and_repo(&opts.owner, &opts.repo);

	if (argc != 1) {
		fprintf(stderr, "error: Missing title to PR\n");
		usage();
		return EXIT_FAILURE;
	}

	opts.title = SV(argv[0]);

	if (create_pull(opts, always_yes) < 0)
		errx(1, "error: failed to submit pull request: %s",
		     gcli_get_error(g_clictx));

	free(opts.labels);

	return EXIT_SUCCESS;
}

/* Forward declaration */
static int handle_pull_actions(int argc, char *argv[],
                               char const *owner,
                               char const *repo,
                               int pr);

int
subcommand_pulls(int argc, char *argv[])
{
	char *endptr = NULL;
	char const *owner = NULL;
	char const *repo = NULL;
	gcli_pull_list pulls = {0};
	int ch = 0;
	int pr = -1;
	int n = 30;                 /* how many prs to fetch at least */
	gcli_pull_fetch_details details = {0};
	enum gcli_output_flags flags = 0;

	/* detect whether we wanna create a PR */
	if (argc > 1 && (strcmp(argv[1], "create") == 0)) {
		shift(&argc, &argv);
		return subcommand_pull_create(argc, argv);
	}

	const struct option options[] = {
		{ .name    = "all",
		  .has_arg = no_argument,
		  .flag    = NULL,
		  .val     = 'a' },
		{ .name    = "author",
		  .has_arg = no_argument,
		  .flag    = NULL,
		  .val     = 'A' },
		{ .name    = "label",
		  .has_arg = no_argument,
		  .flag    = NULL,
		  .val     = 'L' },
		{ .name    = "sorted",
		  .has_arg = no_argument,
		  .flag    = NULL,
		  .val     = 's' },
		{ .name    = "count",
		  .has_arg = required_argument,
		  .flag    = NULL,
		  .val     = 'n' },
		{ .name    = "repo",
		  .has_arg = required_argument,
		  .flag    = NULL,
		  .val     = 'r' },
		{ .name    = "owner",
		  .has_arg = required_argument,
		  .flag    = NULL,
		  .val     = 'o' },
		{ .name    = "id",
		  .has_arg = required_argument,
		  .flag    = NULL,
		  .val     = 'i' },
		{0},
	};

	/* Parse commandline options */
	while ((ch = getopt_long(argc, argv, "+n:o:r:i:asA:L:", options, NULL)) != -1) {
		switch (ch) {
		case 'o':
			owner = optarg;
			break;
		case 'r':
			repo = optarg;
			break;
		case 'i': {
			pr = strtoul(optarg, &endptr, 10);
			if (endptr != (optarg + strlen(optarg)))
				err(1, "error: cannot parse pr number »%s«", optarg);

			if (pr <= 0)
				errx(1, "error: pr number is out of range");
		} break;
		case 'n': {
			n = strtoul(optarg, &endptr, 10);
			if (endptr != (optarg + strlen(optarg)))
				err(1, "error: cannot parse pr count »%s«", optarg);

			if (n < -1)
				errx(1, "error: pr count is out of range");

			if (n == 0)
				errx(1, "error: pr count must not be zero");
		} break;
		case 'a': {
			details.all = true;
		} break;
		case 'A': {
			details.author = optarg;
		} break;
		case 'L': {
			details.label = optarg;
		} break;
		case 's': {
			flags |= OUTPUT_SORTED;
		} break;
		case '?':
		default:
			usage();
			return EXIT_FAILURE;
		}
	}

	argc -= optind;
	argv += optind;

	check_owner_and_repo(&owner, &repo);

	/* In case no explicit PR number was specified, list all
	 * open PRs and exit */
	if (pr < 0) {
		if (gcli_get_pulls(g_clictx, owner, repo, &details, n, &pulls) < 0)
			errx(1, "error: could not fetch pull requests: %s",
			     gcli_get_error(g_clictx));

		gcli_print_pulls(flags, &pulls, n);
		gcli_pulls_free(&pulls);

		return EXIT_SUCCESS;
	}

	/* If a PR number was given, require -a to be unset */
	if (details.all || details.author) {
		fprintf(stderr, "error: -a and -A cannot be combined with operations on a PR\n");
		usage();
		return EXIT_FAILURE;
	}

	/* Hand off to actions handling */
	return handle_pull_actions(argc, argv, owner, repo, pr);
}

/** Helper routine for fetching a PR if required */
static void
ensure_pull(char const *owner, char const *repo, int pr,
            int *const fetched_pull, gcli_pull *const pull)
{
	if (*fetched_pull)
		return;

	if (gcli_get_pull(g_clictx, owner, repo, pr, pull) < 0)
		errx(1, "error: failed to fetch pull request data: %s",
		     gcli_get_error(g_clictx));

	*fetched_pull = 1;
}

/** Handling routine for Pull Request related actions specified on the
 * command line. Make sure that the usage at the top is consistent
 * with the actions implemented here. */
static int
handle_pull_actions(int argc, char *argv[],
                    char const *owner, char const *repo,
                    int pr)
{
	/* For ease of handling and not making redundant calls to the API
	 * we'll fetch the summary only if a command requires it. Then
	 * we'll proceed to actually handling it. */
	int fetched_pull = 0;
	gcli_pull pull = {0};

	/* Check if the user missed out on supplying actions */
	if (argc == 0) {
		fprintf(stderr, "error: no actions supplied\n");
		usage();
		exit(EXIT_FAILURE);
	}

	/* Iterate over the argument list until the end */
	while (argc > 0) {

		/* Grab the next action from the argument list */
		const char *action = shift(&argc, &argv);

		/* Check if it is a valid one. When we find any of
		 *
		 *      all, op or status
		 *
		 * we must ensure that the summary has been fetched. */
		if (strcmp(action, "all") == 0) {

			/* First make sure we have the data ready */
			ensure_pull(owner, repo, pr, &fetched_pull, &pull);

			/* Print meta */
			gcli_pull_print(&pull);

			/* OP */
			puts("\nORIGINAL POST");
			gcli_pull_print_op(&pull);

			/* Commits */
			puts("\nCOMMITS");
			if (gcli_pull_commits(owner, repo, pr) < 0)
				errx(1, "error: failed to fetch pull request checks: %s",
				     gcli_get_error(g_clictx));

			/* Checks */
			puts("\nCHECKS");
			if (gcli_pull_checks(owner, repo, pr) < 0)
				errx(1, "error: failed to fetch pull request checks: %s",
				     gcli_get_error(g_clictx));

		} else if (strcmp(action, "op") == 0) {

			/* Ensure we have fetched the data */
			ensure_pull(owner, repo, pr, &fetched_pull, &pull);

			/* Print it */
			gcli_pull_print_op(&pull);

		} else if (strcmp(action, "status") == 0) {

			/* Ensure we have the data */
			ensure_pull(owner, repo, pr, &fetched_pull, &pull);

			/* Print meta information */
			gcli_pull_print(&pull);

		} else if (strcmp(action, "commits") == 0) {

			/* Does not require the summary */
			gcli_pull_commits(owner, repo, pr);

		} else if (strcmp(action, "diff") == 0) {
			if (gcli_print_pull_diff(stdout, owner, repo, pr) < 0)
				errx(1, "error: failed to fetch diff: %s",
				     gcli_get_error(g_clictx));

		} else if (strcmp(action, "comments") == 0 ||
		           strcmp(action, "notes") == 0) {
			if (gcli_pull_comments(owner, repo, pr) < 0)
				errx(1, "error: failed to fetch pull comments: %s",
				     gcli_get_error(g_clictx));

		} else if (strcmp(action, "ci") == 0) {
			if (gcli_pull_checks(owner, repo, pr) < 0)
				errx(1, "error: failed to fetch pull request checks: %s",
				     gcli_get_error(g_clictx));

		} else if (strcmp(action, "merge") == 0) {
			enum gcli_merge_flags flags = GCLI_PULL_MERGE_DELETEHEAD;

			/* Default behaviour */
			if (gcli_config_pr_inhibit_delete_source_branch(g_clictx))
			    flags = 0;

			if (argc > 0) {
				/* Check whether the user intends a squash-merge
				 * and/or wants to delete the source branch of the
				 * PR */
				if (strcmp(argv[0], "-s") == 0 ||
				    strcmp(argv[0], "--squash") == 0) {
					--argc; ++argv;
					flags |= GCLI_PULL_MERGE_SQUASH;
				} else if (strcmp(argv[0], "-D") == 0 ||
				           strcmp(argv[0], "--inhibit-delete") == 0) {
					--argc; ++argv;
					flags &= ~GCLI_PULL_MERGE_DELETEHEAD;
				}
			}

			if (gcli_pull_merge(g_clictx, owner, repo, pr, flags) < 0)
				errx(1, "error: failed to merge pull request: %s",
				     gcli_get_error(g_clictx));

		} else if (strcmp(action, "close") == 0) {
			if (gcli_pull_close(g_clictx, owner, repo, pr) < 0)
				errx(1, "error: failed to close pull request: %s",
				     gcli_get_error(g_clictx));

		} else if (strcmp(action, "reopen") == 0) {
			if (gcli_pull_reopen(g_clictx, owner, repo, pr) < 0)
				errx(1, "error: failed to reopen pull request: %s",
				     gcli_get_error(g_clictx));

		} else if (strcmp("labels", action) == 0) {
			const char **add_labels = NULL;
			size_t add_labels_size = 0;
			const char **remove_labels = NULL;
			size_t remove_labels_size = 0;
			int rc = 0;

			if (argc == 0) {
				fprintf(stderr, "error: expected label action\n");
				usage();
				return EXIT_FAILURE;
			}

			parse_labels_options(&argc, &argv,
			                     &add_labels,    &add_labels_size,
			                     &remove_labels, &remove_labels_size);

			/* actually go about deleting and adding the labels */
			if (add_labels_size) {
				rc = gcli_pull_add_labels(
					g_clictx, owner, repo, pr, add_labels,
					add_labels_size);
				if (rc < 0)
					errx(1, "error: failed to add labels: %s",
					     gcli_get_error(g_clictx));
			}
			if (remove_labels_size) {
				rc = gcli_pull_remove_labels(
					g_clictx, owner, repo, pr, remove_labels,
					remove_labels_size);

				if (rc < 0)
					errx(1, "error: failed to remove labels: %s",
					     gcli_get_error(g_clictx));
			}

			free(add_labels);
			free(remove_labels);

		} else if (strcmp("milestone", action) == 0) {
			char const *arg = shift(&argc, &argv);

			if (strcmp(arg, "-d") == 0) {
				if (gcli_pull_clear_milestone(g_clictx, owner, repo, pr) < 0)
					errx(1, "error: failed to clear milestone: %s",
					     gcli_get_error(g_clictx));

			} else {
				int milestone_id = 0;
				char *endptr;
				int rc = 0;

				milestone_id = strtoul(arg, &endptr, 10);
				if (endptr != arg + strlen(arg)) {
					fprintf(stderr, "error: cannot parse milestone id »%s«\n", arg);
					return EXIT_FAILURE;
				}

				rc = gcli_pull_set_milestone(g_clictx, owner, repo, pr,
				                             milestone_id);
				if (rc < 0)
					errx(1, "error: failed to set milestone: %s",
					     gcli_get_error(g_clictx));
			}
		} else {
			/* At this point we found an unknown action / stray
			 * options on the command line. Error out in this case. */

			fprintf(stderr, "error: unknown action %s\n", action);
			usage();
			return EXIT_FAILURE;

		}

		if (argc)
			putchar('\n');

	} /* Next action */

	/* Free the pull request data only when we actually fetched it */
	if (fetched_pull)
		gcli_pull_free(&pull);

	return EXIT_SUCCESS;
}
