// 2007-02-04  Edward Smith-Rowland <3dw4rd@verizon.net>
//
// Copyright (C) 2007-2016 Free Software Foundation, Inc.
//
// This file is part of the GNU ISO C++ Library.  This library is free
// software; you can redistribute it and/or modify it under the
// terms of the GNU General Public License as published by the
// Free Software Foundation; either version 3, or (at your option)
// any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License along
// with this library; see the file COPYING3.  If not see
// <http://www.gnu.org/licenses/>.

//  hyperg


//  Compare against values generated by the GNU Scientific Library.
//  The GSL can be found on the web: http://www.gnu.org/software/gsl/

#include <tr1/cmath>
#if defined(__TEST_DEBUG)
#include <iostream>
#define VERIFY(A) \
if (!(A)) \
  { \
    std::cout << "line " << __LINE__ \
      << "  max_abs_frac = " << max_abs_frac \
      << std::endl; \
  }
#else
#include <testsuite_hooks.h>
#endif
#include "../testcase.h"


// Test data for a=0.0000000000000000, b=0.0000000000000000, c=2.0000000000000000.
testcase_hyperg<double> data001[] = {
  { 1.0000000000000000, 0.0000000000000000, 0.0000000000000000,
          2.0000000000000000, -0.90000000000000002 },
  { 1.0000000000000000, 0.0000000000000000, 0.0000000000000000,
          2.0000000000000000, -0.80000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 0.0000000000000000,
          2.0000000000000000, -0.69999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 0.0000000000000000,
          2.0000000000000000, -0.59999999999999998 },
  { 1.0000000000000000, 0.0000000000000000, 0.0000000000000000,
          2.0000000000000000, -0.50000000000000000 },
  { 1.0000000000000000, 0.0000000000000000, 0.0000000000000000,
          2.0000000000000000, -0.40000000000000002 },
  { 1.0000000000000000, 0.0000000000000000, 0.0000000000000000,
          2.0000000000000000, -0.30000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 0.0000000000000000,
          2.0000000000000000, -0.19999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 0.0000000000000000,
          2.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 0.0000000000000000, 0.0000000000000000,
          2.0000000000000000, 0.0000000000000000 },
  { 1.0000000000000000, 0.0000000000000000, 0.0000000000000000,
          2.0000000000000000, 0.10000000000000009 },
  { 1.0000000000000000, 0.0000000000000000, 0.0000000000000000,
          2.0000000000000000, 0.19999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 0.0000000000000000,
          2.0000000000000000, 0.30000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 0.0000000000000000,
          2.0000000000000000, 0.39999999999999991 },
  { 1.0000000000000000, 0.0000000000000000, 0.0000000000000000,
          2.0000000000000000, 0.50000000000000000 },
  { 1.0000000000000000, 0.0000000000000000, 0.0000000000000000,
          2.0000000000000000, 0.60000000000000009 },
  { 1.0000000000000000, 0.0000000000000000, 0.0000000000000000,
          2.0000000000000000, 0.69999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 0.0000000000000000,
          2.0000000000000000, 0.80000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 0.0000000000000000,
          2.0000000000000000, 0.89999999999999991 },
};

// Test function for a=0.0000000000000000, b=0.0000000000000000, c=2.0000000000000000.
template <typename Tp>
void test001()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data001)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data001[i].a), Tp(data001[i].b),
                   Tp(data001[i].c), Tp(data001[i].x));
      const Tp f0 = data001[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=0.0000000000000000, b=0.0000000000000000, c=4.0000000000000000.
testcase_hyperg<double> data002[] = {
  { 1.0000000000000000, 0.0000000000000000, 0.0000000000000000,
          4.0000000000000000, -0.90000000000000002 },
  { 1.0000000000000000, 0.0000000000000000, 0.0000000000000000,
          4.0000000000000000, -0.80000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 0.0000000000000000,
          4.0000000000000000, -0.69999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 0.0000000000000000,
          4.0000000000000000, -0.59999999999999998 },
  { 1.0000000000000000, 0.0000000000000000, 0.0000000000000000,
          4.0000000000000000, -0.50000000000000000 },
  { 1.0000000000000000, 0.0000000000000000, 0.0000000000000000,
          4.0000000000000000, -0.40000000000000002 },
  { 1.0000000000000000, 0.0000000000000000, 0.0000000000000000,
          4.0000000000000000, -0.30000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 0.0000000000000000,
          4.0000000000000000, -0.19999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 0.0000000000000000,
          4.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 0.0000000000000000, 0.0000000000000000,
          4.0000000000000000, 0.0000000000000000 },
  { 1.0000000000000000, 0.0000000000000000, 0.0000000000000000,
          4.0000000000000000, 0.10000000000000009 },
  { 1.0000000000000000, 0.0000000000000000, 0.0000000000000000,
          4.0000000000000000, 0.19999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 0.0000000000000000,
          4.0000000000000000, 0.30000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 0.0000000000000000,
          4.0000000000000000, 0.39999999999999991 },
  { 1.0000000000000000, 0.0000000000000000, 0.0000000000000000,
          4.0000000000000000, 0.50000000000000000 },
  { 1.0000000000000000, 0.0000000000000000, 0.0000000000000000,
          4.0000000000000000, 0.60000000000000009 },
  { 1.0000000000000000, 0.0000000000000000, 0.0000000000000000,
          4.0000000000000000, 0.69999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 0.0000000000000000,
          4.0000000000000000, 0.80000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 0.0000000000000000,
          4.0000000000000000, 0.89999999999999991 },
};

// Test function for a=0.0000000000000000, b=0.0000000000000000, c=4.0000000000000000.
template <typename Tp>
void test002()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data002)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data002[i].a), Tp(data002[i].b),
                   Tp(data002[i].c), Tp(data002[i].x));
      const Tp f0 = data002[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=0.0000000000000000, b=0.0000000000000000, c=6.0000000000000000.
testcase_hyperg<double> data003[] = {
  { 1.0000000000000000, 0.0000000000000000, 0.0000000000000000,
          6.0000000000000000, -0.90000000000000002 },
  { 1.0000000000000000, 0.0000000000000000, 0.0000000000000000,
          6.0000000000000000, -0.80000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 0.0000000000000000,
          6.0000000000000000, -0.69999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 0.0000000000000000,
          6.0000000000000000, -0.59999999999999998 },
  { 1.0000000000000000, 0.0000000000000000, 0.0000000000000000,
          6.0000000000000000, -0.50000000000000000 },
  { 1.0000000000000000, 0.0000000000000000, 0.0000000000000000,
          6.0000000000000000, -0.40000000000000002 },
  { 1.0000000000000000, 0.0000000000000000, 0.0000000000000000,
          6.0000000000000000, -0.30000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 0.0000000000000000,
          6.0000000000000000, -0.19999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 0.0000000000000000,
          6.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 0.0000000000000000, 0.0000000000000000,
          6.0000000000000000, 0.0000000000000000 },
  { 1.0000000000000000, 0.0000000000000000, 0.0000000000000000,
          6.0000000000000000, 0.10000000000000009 },
  { 1.0000000000000000, 0.0000000000000000, 0.0000000000000000,
          6.0000000000000000, 0.19999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 0.0000000000000000,
          6.0000000000000000, 0.30000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 0.0000000000000000,
          6.0000000000000000, 0.39999999999999991 },
  { 1.0000000000000000, 0.0000000000000000, 0.0000000000000000,
          6.0000000000000000, 0.50000000000000000 },
  { 1.0000000000000000, 0.0000000000000000, 0.0000000000000000,
          6.0000000000000000, 0.60000000000000009 },
  { 1.0000000000000000, 0.0000000000000000, 0.0000000000000000,
          6.0000000000000000, 0.69999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 0.0000000000000000,
          6.0000000000000000, 0.80000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 0.0000000000000000,
          6.0000000000000000, 0.89999999999999991 },
};

// Test function for a=0.0000000000000000, b=0.0000000000000000, c=6.0000000000000000.
template <typename Tp>
void test003()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data003)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data003[i].a), Tp(data003[i].b),
                   Tp(data003[i].c), Tp(data003[i].x));
      const Tp f0 = data003[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=0.0000000000000000, b=0.0000000000000000, c=8.0000000000000000.
testcase_hyperg<double> data004[] = {
  { 1.0000000000000000, 0.0000000000000000, 0.0000000000000000,
          8.0000000000000000, -0.90000000000000002 },
  { 1.0000000000000000, 0.0000000000000000, 0.0000000000000000,
          8.0000000000000000, -0.80000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 0.0000000000000000,
          8.0000000000000000, -0.69999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 0.0000000000000000,
          8.0000000000000000, -0.59999999999999998 },
  { 1.0000000000000000, 0.0000000000000000, 0.0000000000000000,
          8.0000000000000000, -0.50000000000000000 },
  { 1.0000000000000000, 0.0000000000000000, 0.0000000000000000,
          8.0000000000000000, -0.40000000000000002 },
  { 1.0000000000000000, 0.0000000000000000, 0.0000000000000000,
          8.0000000000000000, -0.30000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 0.0000000000000000,
          8.0000000000000000, -0.19999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 0.0000000000000000,
          8.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 0.0000000000000000, 0.0000000000000000,
          8.0000000000000000, 0.0000000000000000 },
  { 1.0000000000000000, 0.0000000000000000, 0.0000000000000000,
          8.0000000000000000, 0.10000000000000009 },
  { 1.0000000000000000, 0.0000000000000000, 0.0000000000000000,
          8.0000000000000000, 0.19999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 0.0000000000000000,
          8.0000000000000000, 0.30000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 0.0000000000000000,
          8.0000000000000000, 0.39999999999999991 },
  { 1.0000000000000000, 0.0000000000000000, 0.0000000000000000,
          8.0000000000000000, 0.50000000000000000 },
  { 1.0000000000000000, 0.0000000000000000, 0.0000000000000000,
          8.0000000000000000, 0.60000000000000009 },
  { 1.0000000000000000, 0.0000000000000000, 0.0000000000000000,
          8.0000000000000000, 0.69999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 0.0000000000000000,
          8.0000000000000000, 0.80000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 0.0000000000000000,
          8.0000000000000000, 0.89999999999999991 },
};

// Test function for a=0.0000000000000000, b=0.0000000000000000, c=8.0000000000000000.
template <typename Tp>
void test004()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data004)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data004[i].a), Tp(data004[i].b),
                   Tp(data004[i].c), Tp(data004[i].x));
      const Tp f0 = data004[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=0.0000000000000000, b=0.0000000000000000, c=10.000000000000000.
testcase_hyperg<double> data005[] = {
  { 1.0000000000000000, 0.0000000000000000, 0.0000000000000000,
          10.000000000000000, -0.90000000000000002 },
  { 1.0000000000000000, 0.0000000000000000, 0.0000000000000000,
          10.000000000000000, -0.80000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 0.0000000000000000,
          10.000000000000000, -0.69999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 0.0000000000000000,
          10.000000000000000, -0.59999999999999998 },
  { 1.0000000000000000, 0.0000000000000000, 0.0000000000000000,
          10.000000000000000, -0.50000000000000000 },
  { 1.0000000000000000, 0.0000000000000000, 0.0000000000000000,
          10.000000000000000, -0.40000000000000002 },
  { 1.0000000000000000, 0.0000000000000000, 0.0000000000000000,
          10.000000000000000, -0.30000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 0.0000000000000000,
          10.000000000000000, -0.19999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 0.0000000000000000,
          10.000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 0.0000000000000000, 0.0000000000000000,
          10.000000000000000, 0.0000000000000000 },
  { 1.0000000000000000, 0.0000000000000000, 0.0000000000000000,
          10.000000000000000, 0.10000000000000009 },
  { 1.0000000000000000, 0.0000000000000000, 0.0000000000000000,
          10.000000000000000, 0.19999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 0.0000000000000000,
          10.000000000000000, 0.30000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 0.0000000000000000,
          10.000000000000000, 0.39999999999999991 },
  { 1.0000000000000000, 0.0000000000000000, 0.0000000000000000,
          10.000000000000000, 0.50000000000000000 },
  { 1.0000000000000000, 0.0000000000000000, 0.0000000000000000,
          10.000000000000000, 0.60000000000000009 },
  { 1.0000000000000000, 0.0000000000000000, 0.0000000000000000,
          10.000000000000000, 0.69999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 0.0000000000000000,
          10.000000000000000, 0.80000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 0.0000000000000000,
          10.000000000000000, 0.89999999999999991 },
};

// Test function for a=0.0000000000000000, b=0.0000000000000000, c=10.000000000000000.
template <typename Tp>
void test005()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data005)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data005[i].a), Tp(data005[i].b),
                   Tp(data005[i].c), Tp(data005[i].x));
      const Tp f0 = data005[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=0.0000000000000000, b=0.50000000000000000, c=2.0000000000000000.
testcase_hyperg<double> data006[] = {
  { 1.0000000000000000, 0.0000000000000000, 0.50000000000000000,
          2.0000000000000000, -0.90000000000000002 },
  { 1.0000000000000000, 0.0000000000000000, 0.50000000000000000,
          2.0000000000000000, -0.80000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 0.50000000000000000,
          2.0000000000000000, -0.69999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 0.50000000000000000,
          2.0000000000000000, -0.59999999999999998 },
  { 1.0000000000000000, 0.0000000000000000, 0.50000000000000000,
          2.0000000000000000, -0.50000000000000000 },
  { 1.0000000000000000, 0.0000000000000000, 0.50000000000000000,
          2.0000000000000000, -0.40000000000000002 },
  { 1.0000000000000000, 0.0000000000000000, 0.50000000000000000,
          2.0000000000000000, -0.30000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 0.50000000000000000,
          2.0000000000000000, -0.19999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 0.50000000000000000,
          2.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 0.0000000000000000, 0.50000000000000000,
          2.0000000000000000, 0.0000000000000000 },
  { 1.0000000000000000, 0.0000000000000000, 0.50000000000000000,
          2.0000000000000000, 0.10000000000000009 },
  { 1.0000000000000000, 0.0000000000000000, 0.50000000000000000,
          2.0000000000000000, 0.19999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 0.50000000000000000,
          2.0000000000000000, 0.30000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 0.50000000000000000,
          2.0000000000000000, 0.39999999999999991 },
  { 1.0000000000000000, 0.0000000000000000, 0.50000000000000000,
          2.0000000000000000, 0.50000000000000000 },
  { 1.0000000000000000, 0.0000000000000000, 0.50000000000000000,
          2.0000000000000000, 0.60000000000000009 },
  { 1.0000000000000000, 0.0000000000000000, 0.50000000000000000,
          2.0000000000000000, 0.69999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 0.50000000000000000,
          2.0000000000000000, 0.80000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 0.50000000000000000,
          2.0000000000000000, 0.89999999999999991 },
};

// Test function for a=0.0000000000000000, b=0.50000000000000000, c=2.0000000000000000.
template <typename Tp>
void test006()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data006)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data006[i].a), Tp(data006[i].b),
                   Tp(data006[i].c), Tp(data006[i].x));
      const Tp f0 = data006[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=0.0000000000000000, b=0.50000000000000000, c=4.0000000000000000.
testcase_hyperg<double> data007[] = {
  { 1.0000000000000000, 0.0000000000000000, 0.50000000000000000,
          4.0000000000000000, -0.90000000000000002 },
  { 1.0000000000000000, 0.0000000000000000, 0.50000000000000000,
          4.0000000000000000, -0.80000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 0.50000000000000000,
          4.0000000000000000, -0.69999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 0.50000000000000000,
          4.0000000000000000, -0.59999999999999998 },
  { 1.0000000000000000, 0.0000000000000000, 0.50000000000000000,
          4.0000000000000000, -0.50000000000000000 },
  { 1.0000000000000000, 0.0000000000000000, 0.50000000000000000,
          4.0000000000000000, -0.40000000000000002 },
  { 1.0000000000000000, 0.0000000000000000, 0.50000000000000000,
          4.0000000000000000, -0.30000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 0.50000000000000000,
          4.0000000000000000, -0.19999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 0.50000000000000000,
          4.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 0.0000000000000000, 0.50000000000000000,
          4.0000000000000000, 0.0000000000000000 },
  { 1.0000000000000000, 0.0000000000000000, 0.50000000000000000,
          4.0000000000000000, 0.10000000000000009 },
  { 1.0000000000000000, 0.0000000000000000, 0.50000000000000000,
          4.0000000000000000, 0.19999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 0.50000000000000000,
          4.0000000000000000, 0.30000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 0.50000000000000000,
          4.0000000000000000, 0.39999999999999991 },
  { 1.0000000000000000, 0.0000000000000000, 0.50000000000000000,
          4.0000000000000000, 0.50000000000000000 },
  { 1.0000000000000000, 0.0000000000000000, 0.50000000000000000,
          4.0000000000000000, 0.60000000000000009 },
  { 1.0000000000000000, 0.0000000000000000, 0.50000000000000000,
          4.0000000000000000, 0.69999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 0.50000000000000000,
          4.0000000000000000, 0.80000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 0.50000000000000000,
          4.0000000000000000, 0.89999999999999991 },
};

// Test function for a=0.0000000000000000, b=0.50000000000000000, c=4.0000000000000000.
template <typename Tp>
void test007()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data007)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data007[i].a), Tp(data007[i].b),
                   Tp(data007[i].c), Tp(data007[i].x));
      const Tp f0 = data007[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=0.0000000000000000, b=0.50000000000000000, c=6.0000000000000000.
testcase_hyperg<double> data008[] = {
  { 1.0000000000000000, 0.0000000000000000, 0.50000000000000000,
          6.0000000000000000, -0.90000000000000002 },
  { 1.0000000000000000, 0.0000000000000000, 0.50000000000000000,
          6.0000000000000000, -0.80000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 0.50000000000000000,
          6.0000000000000000, -0.69999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 0.50000000000000000,
          6.0000000000000000, -0.59999999999999998 },
  { 1.0000000000000000, 0.0000000000000000, 0.50000000000000000,
          6.0000000000000000, -0.50000000000000000 },
  { 1.0000000000000000, 0.0000000000000000, 0.50000000000000000,
          6.0000000000000000, -0.40000000000000002 },
  { 1.0000000000000000, 0.0000000000000000, 0.50000000000000000,
          6.0000000000000000, -0.30000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 0.50000000000000000,
          6.0000000000000000, -0.19999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 0.50000000000000000,
          6.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 0.0000000000000000, 0.50000000000000000,
          6.0000000000000000, 0.0000000000000000 },
  { 1.0000000000000000, 0.0000000000000000, 0.50000000000000000,
          6.0000000000000000, 0.10000000000000009 },
  { 1.0000000000000000, 0.0000000000000000, 0.50000000000000000,
          6.0000000000000000, 0.19999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 0.50000000000000000,
          6.0000000000000000, 0.30000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 0.50000000000000000,
          6.0000000000000000, 0.39999999999999991 },
  { 1.0000000000000000, 0.0000000000000000, 0.50000000000000000,
          6.0000000000000000, 0.50000000000000000 },
  { 1.0000000000000000, 0.0000000000000000, 0.50000000000000000,
          6.0000000000000000, 0.60000000000000009 },
  { 1.0000000000000000, 0.0000000000000000, 0.50000000000000000,
          6.0000000000000000, 0.69999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 0.50000000000000000,
          6.0000000000000000, 0.80000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 0.50000000000000000,
          6.0000000000000000, 0.89999999999999991 },
};

// Test function for a=0.0000000000000000, b=0.50000000000000000, c=6.0000000000000000.
template <typename Tp>
void test008()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data008)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data008[i].a), Tp(data008[i].b),
                   Tp(data008[i].c), Tp(data008[i].x));
      const Tp f0 = data008[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=0.0000000000000000, b=0.50000000000000000, c=8.0000000000000000.
testcase_hyperg<double> data009[] = {
  { 1.0000000000000000, 0.0000000000000000, 0.50000000000000000,
          8.0000000000000000, -0.90000000000000002 },
  { 1.0000000000000000, 0.0000000000000000, 0.50000000000000000,
          8.0000000000000000, -0.80000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 0.50000000000000000,
          8.0000000000000000, -0.69999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 0.50000000000000000,
          8.0000000000000000, -0.59999999999999998 },
  { 1.0000000000000000, 0.0000000000000000, 0.50000000000000000,
          8.0000000000000000, -0.50000000000000000 },
  { 1.0000000000000000, 0.0000000000000000, 0.50000000000000000,
          8.0000000000000000, -0.40000000000000002 },
  { 1.0000000000000000, 0.0000000000000000, 0.50000000000000000,
          8.0000000000000000, -0.30000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 0.50000000000000000,
          8.0000000000000000, -0.19999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 0.50000000000000000,
          8.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 0.0000000000000000, 0.50000000000000000,
          8.0000000000000000, 0.0000000000000000 },
  { 1.0000000000000000, 0.0000000000000000, 0.50000000000000000,
          8.0000000000000000, 0.10000000000000009 },
  { 1.0000000000000000, 0.0000000000000000, 0.50000000000000000,
          8.0000000000000000, 0.19999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 0.50000000000000000,
          8.0000000000000000, 0.30000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 0.50000000000000000,
          8.0000000000000000, 0.39999999999999991 },
  { 1.0000000000000000, 0.0000000000000000, 0.50000000000000000,
          8.0000000000000000, 0.50000000000000000 },
  { 1.0000000000000000, 0.0000000000000000, 0.50000000000000000,
          8.0000000000000000, 0.60000000000000009 },
  { 1.0000000000000000, 0.0000000000000000, 0.50000000000000000,
          8.0000000000000000, 0.69999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 0.50000000000000000,
          8.0000000000000000, 0.80000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 0.50000000000000000,
          8.0000000000000000, 0.89999999999999991 },
};

// Test function for a=0.0000000000000000, b=0.50000000000000000, c=8.0000000000000000.
template <typename Tp>
void test009()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data009)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data009[i].a), Tp(data009[i].b),
                   Tp(data009[i].c), Tp(data009[i].x));
      const Tp f0 = data009[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=0.0000000000000000, b=0.50000000000000000, c=10.000000000000000.
testcase_hyperg<double> data010[] = {
  { 1.0000000000000000, 0.0000000000000000, 0.50000000000000000,
          10.000000000000000, -0.90000000000000002 },
  { 1.0000000000000000, 0.0000000000000000, 0.50000000000000000,
          10.000000000000000, -0.80000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 0.50000000000000000,
          10.000000000000000, -0.69999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 0.50000000000000000,
          10.000000000000000, -0.59999999999999998 },
  { 1.0000000000000000, 0.0000000000000000, 0.50000000000000000,
          10.000000000000000, -0.50000000000000000 },
  { 1.0000000000000000, 0.0000000000000000, 0.50000000000000000,
          10.000000000000000, -0.40000000000000002 },
  { 1.0000000000000000, 0.0000000000000000, 0.50000000000000000,
          10.000000000000000, -0.30000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 0.50000000000000000,
          10.000000000000000, -0.19999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 0.50000000000000000,
          10.000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 0.0000000000000000, 0.50000000000000000,
          10.000000000000000, 0.0000000000000000 },
  { 1.0000000000000000, 0.0000000000000000, 0.50000000000000000,
          10.000000000000000, 0.10000000000000009 },
  { 1.0000000000000000, 0.0000000000000000, 0.50000000000000000,
          10.000000000000000, 0.19999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 0.50000000000000000,
          10.000000000000000, 0.30000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 0.50000000000000000,
          10.000000000000000, 0.39999999999999991 },
  { 1.0000000000000000, 0.0000000000000000, 0.50000000000000000,
          10.000000000000000, 0.50000000000000000 },
  { 1.0000000000000000, 0.0000000000000000, 0.50000000000000000,
          10.000000000000000, 0.60000000000000009 },
  { 1.0000000000000000, 0.0000000000000000, 0.50000000000000000,
          10.000000000000000, 0.69999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 0.50000000000000000,
          10.000000000000000, 0.80000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 0.50000000000000000,
          10.000000000000000, 0.89999999999999991 },
};

// Test function for a=0.0000000000000000, b=0.50000000000000000, c=10.000000000000000.
template <typename Tp>
void test010()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data010)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data010[i].a), Tp(data010[i].b),
                   Tp(data010[i].c), Tp(data010[i].x));
      const Tp f0 = data010[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=0.0000000000000000, b=1.0000000000000000, c=2.0000000000000000.
testcase_hyperg<double> data011[] = {
  { 1.0000000000000000, 0.0000000000000000, 1.0000000000000000,
          2.0000000000000000, -0.90000000000000002 },
  { 1.0000000000000000, 0.0000000000000000, 1.0000000000000000,
          2.0000000000000000, -0.80000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 1.0000000000000000,
          2.0000000000000000, -0.69999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 1.0000000000000000,
          2.0000000000000000, -0.59999999999999998 },
  { 1.0000000000000000, 0.0000000000000000, 1.0000000000000000,
          2.0000000000000000, -0.50000000000000000 },
  { 1.0000000000000000, 0.0000000000000000, 1.0000000000000000,
          2.0000000000000000, -0.40000000000000002 },
  { 1.0000000000000000, 0.0000000000000000, 1.0000000000000000,
          2.0000000000000000, -0.30000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 1.0000000000000000,
          2.0000000000000000, -0.19999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 1.0000000000000000,
          2.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 0.0000000000000000, 1.0000000000000000,
          2.0000000000000000, 0.0000000000000000 },
  { 1.0000000000000000, 0.0000000000000000, 1.0000000000000000,
          2.0000000000000000, 0.10000000000000009 },
  { 1.0000000000000000, 0.0000000000000000, 1.0000000000000000,
          2.0000000000000000, 0.19999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 1.0000000000000000,
          2.0000000000000000, 0.30000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 1.0000000000000000,
          2.0000000000000000, 0.39999999999999991 },
  { 1.0000000000000000, 0.0000000000000000, 1.0000000000000000,
          2.0000000000000000, 0.50000000000000000 },
  { 1.0000000000000000, 0.0000000000000000, 1.0000000000000000,
          2.0000000000000000, 0.60000000000000009 },
  { 1.0000000000000000, 0.0000000000000000, 1.0000000000000000,
          2.0000000000000000, 0.69999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 1.0000000000000000,
          2.0000000000000000, 0.80000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 1.0000000000000000,
          2.0000000000000000, 0.89999999999999991 },
};

// Test function for a=0.0000000000000000, b=1.0000000000000000, c=2.0000000000000000.
template <typename Tp>
void test011()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data011)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data011[i].a), Tp(data011[i].b),
                   Tp(data011[i].c), Tp(data011[i].x));
      const Tp f0 = data011[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=0.0000000000000000, b=1.0000000000000000, c=4.0000000000000000.
testcase_hyperg<double> data012[] = {
  { 1.0000000000000000, 0.0000000000000000, 1.0000000000000000,
          4.0000000000000000, -0.90000000000000002 },
  { 1.0000000000000000, 0.0000000000000000, 1.0000000000000000,
          4.0000000000000000, -0.80000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 1.0000000000000000,
          4.0000000000000000, -0.69999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 1.0000000000000000,
          4.0000000000000000, -0.59999999999999998 },
  { 1.0000000000000000, 0.0000000000000000, 1.0000000000000000,
          4.0000000000000000, -0.50000000000000000 },
  { 1.0000000000000000, 0.0000000000000000, 1.0000000000000000,
          4.0000000000000000, -0.40000000000000002 },
  { 1.0000000000000000, 0.0000000000000000, 1.0000000000000000,
          4.0000000000000000, -0.30000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 1.0000000000000000,
          4.0000000000000000, -0.19999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 1.0000000000000000,
          4.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 0.0000000000000000, 1.0000000000000000,
          4.0000000000000000, 0.0000000000000000 },
  { 1.0000000000000000, 0.0000000000000000, 1.0000000000000000,
          4.0000000000000000, 0.10000000000000009 },
  { 1.0000000000000000, 0.0000000000000000, 1.0000000000000000,
          4.0000000000000000, 0.19999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 1.0000000000000000,
          4.0000000000000000, 0.30000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 1.0000000000000000,
          4.0000000000000000, 0.39999999999999991 },
  { 1.0000000000000000, 0.0000000000000000, 1.0000000000000000,
          4.0000000000000000, 0.50000000000000000 },
  { 1.0000000000000000, 0.0000000000000000, 1.0000000000000000,
          4.0000000000000000, 0.60000000000000009 },
  { 1.0000000000000000, 0.0000000000000000, 1.0000000000000000,
          4.0000000000000000, 0.69999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 1.0000000000000000,
          4.0000000000000000, 0.80000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 1.0000000000000000,
          4.0000000000000000, 0.89999999999999991 },
};

// Test function for a=0.0000000000000000, b=1.0000000000000000, c=4.0000000000000000.
template <typename Tp>
void test012()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data012)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data012[i].a), Tp(data012[i].b),
                   Tp(data012[i].c), Tp(data012[i].x));
      const Tp f0 = data012[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=0.0000000000000000, b=1.0000000000000000, c=6.0000000000000000.
testcase_hyperg<double> data013[] = {
  { 1.0000000000000000, 0.0000000000000000, 1.0000000000000000,
          6.0000000000000000, -0.90000000000000002 },
  { 1.0000000000000000, 0.0000000000000000, 1.0000000000000000,
          6.0000000000000000, -0.80000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 1.0000000000000000,
          6.0000000000000000, -0.69999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 1.0000000000000000,
          6.0000000000000000, -0.59999999999999998 },
  { 1.0000000000000000, 0.0000000000000000, 1.0000000000000000,
          6.0000000000000000, -0.50000000000000000 },
  { 1.0000000000000000, 0.0000000000000000, 1.0000000000000000,
          6.0000000000000000, -0.40000000000000002 },
  { 1.0000000000000000, 0.0000000000000000, 1.0000000000000000,
          6.0000000000000000, -0.30000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 1.0000000000000000,
          6.0000000000000000, -0.19999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 1.0000000000000000,
          6.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 0.0000000000000000, 1.0000000000000000,
          6.0000000000000000, 0.0000000000000000 },
  { 1.0000000000000000, 0.0000000000000000, 1.0000000000000000,
          6.0000000000000000, 0.10000000000000009 },
  { 1.0000000000000000, 0.0000000000000000, 1.0000000000000000,
          6.0000000000000000, 0.19999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 1.0000000000000000,
          6.0000000000000000, 0.30000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 1.0000000000000000,
          6.0000000000000000, 0.39999999999999991 },
  { 1.0000000000000000, 0.0000000000000000, 1.0000000000000000,
          6.0000000000000000, 0.50000000000000000 },
  { 1.0000000000000000, 0.0000000000000000, 1.0000000000000000,
          6.0000000000000000, 0.60000000000000009 },
  { 1.0000000000000000, 0.0000000000000000, 1.0000000000000000,
          6.0000000000000000, 0.69999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 1.0000000000000000,
          6.0000000000000000, 0.80000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 1.0000000000000000,
          6.0000000000000000, 0.89999999999999991 },
};

// Test function for a=0.0000000000000000, b=1.0000000000000000, c=6.0000000000000000.
template <typename Tp>
void test013()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data013)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data013[i].a), Tp(data013[i].b),
                   Tp(data013[i].c), Tp(data013[i].x));
      const Tp f0 = data013[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=0.0000000000000000, b=1.0000000000000000, c=8.0000000000000000.
testcase_hyperg<double> data014[] = {
  { 1.0000000000000000, 0.0000000000000000, 1.0000000000000000,
          8.0000000000000000, -0.90000000000000002 },
  { 1.0000000000000000, 0.0000000000000000, 1.0000000000000000,
          8.0000000000000000, -0.80000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 1.0000000000000000,
          8.0000000000000000, -0.69999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 1.0000000000000000,
          8.0000000000000000, -0.59999999999999998 },
  { 1.0000000000000000, 0.0000000000000000, 1.0000000000000000,
          8.0000000000000000, -0.50000000000000000 },
  { 1.0000000000000000, 0.0000000000000000, 1.0000000000000000,
          8.0000000000000000, -0.40000000000000002 },
  { 1.0000000000000000, 0.0000000000000000, 1.0000000000000000,
          8.0000000000000000, -0.30000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 1.0000000000000000,
          8.0000000000000000, -0.19999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 1.0000000000000000,
          8.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 0.0000000000000000, 1.0000000000000000,
          8.0000000000000000, 0.0000000000000000 },
  { 1.0000000000000000, 0.0000000000000000, 1.0000000000000000,
          8.0000000000000000, 0.10000000000000009 },
  { 1.0000000000000000, 0.0000000000000000, 1.0000000000000000,
          8.0000000000000000, 0.19999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 1.0000000000000000,
          8.0000000000000000, 0.30000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 1.0000000000000000,
          8.0000000000000000, 0.39999999999999991 },
  { 1.0000000000000000, 0.0000000000000000, 1.0000000000000000,
          8.0000000000000000, 0.50000000000000000 },
  { 1.0000000000000000, 0.0000000000000000, 1.0000000000000000,
          8.0000000000000000, 0.60000000000000009 },
  { 1.0000000000000000, 0.0000000000000000, 1.0000000000000000,
          8.0000000000000000, 0.69999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 1.0000000000000000,
          8.0000000000000000, 0.80000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 1.0000000000000000,
          8.0000000000000000, 0.89999999999999991 },
};

// Test function for a=0.0000000000000000, b=1.0000000000000000, c=8.0000000000000000.
template <typename Tp>
void test014()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data014)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data014[i].a), Tp(data014[i].b),
                   Tp(data014[i].c), Tp(data014[i].x));
      const Tp f0 = data014[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=0.0000000000000000, b=1.0000000000000000, c=10.000000000000000.
testcase_hyperg<double> data015[] = {
  { 1.0000000000000000, 0.0000000000000000, 1.0000000000000000,
          10.000000000000000, -0.90000000000000002 },
  { 1.0000000000000000, 0.0000000000000000, 1.0000000000000000,
          10.000000000000000, -0.80000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 1.0000000000000000,
          10.000000000000000, -0.69999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 1.0000000000000000,
          10.000000000000000, -0.59999999999999998 },
  { 1.0000000000000000, 0.0000000000000000, 1.0000000000000000,
          10.000000000000000, -0.50000000000000000 },
  { 1.0000000000000000, 0.0000000000000000, 1.0000000000000000,
          10.000000000000000, -0.40000000000000002 },
  { 1.0000000000000000, 0.0000000000000000, 1.0000000000000000,
          10.000000000000000, -0.30000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 1.0000000000000000,
          10.000000000000000, -0.19999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 1.0000000000000000,
          10.000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 0.0000000000000000, 1.0000000000000000,
          10.000000000000000, 0.0000000000000000 },
  { 1.0000000000000000, 0.0000000000000000, 1.0000000000000000,
          10.000000000000000, 0.10000000000000009 },
  { 1.0000000000000000, 0.0000000000000000, 1.0000000000000000,
          10.000000000000000, 0.19999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 1.0000000000000000,
          10.000000000000000, 0.30000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 1.0000000000000000,
          10.000000000000000, 0.39999999999999991 },
  { 1.0000000000000000, 0.0000000000000000, 1.0000000000000000,
          10.000000000000000, 0.50000000000000000 },
  { 1.0000000000000000, 0.0000000000000000, 1.0000000000000000,
          10.000000000000000, 0.60000000000000009 },
  { 1.0000000000000000, 0.0000000000000000, 1.0000000000000000,
          10.000000000000000, 0.69999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 1.0000000000000000,
          10.000000000000000, 0.80000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 1.0000000000000000,
          10.000000000000000, 0.89999999999999991 },
};

// Test function for a=0.0000000000000000, b=1.0000000000000000, c=10.000000000000000.
template <typename Tp>
void test015()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data015)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data015[i].a), Tp(data015[i].b),
                   Tp(data015[i].c), Tp(data015[i].x));
      const Tp f0 = data015[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=0.0000000000000000, b=2.0000000000000000, c=2.0000000000000000.
testcase_hyperg<double> data016[] = {
  { 1.0000000000000000, 0.0000000000000000, 2.0000000000000000,
          2.0000000000000000, -0.90000000000000002 },
  { 1.0000000000000000, 0.0000000000000000, 2.0000000000000000,
          2.0000000000000000, -0.80000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 2.0000000000000000,
          2.0000000000000000, -0.69999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 2.0000000000000000,
          2.0000000000000000, -0.59999999999999998 },
  { 1.0000000000000000, 0.0000000000000000, 2.0000000000000000,
          2.0000000000000000, -0.50000000000000000 },
  { 1.0000000000000000, 0.0000000000000000, 2.0000000000000000,
          2.0000000000000000, -0.40000000000000002 },
  { 1.0000000000000000, 0.0000000000000000, 2.0000000000000000,
          2.0000000000000000, -0.30000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 2.0000000000000000,
          2.0000000000000000, -0.19999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 2.0000000000000000,
          2.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 0.0000000000000000, 2.0000000000000000,
          2.0000000000000000, 0.0000000000000000 },
  { 1.0000000000000000, 0.0000000000000000, 2.0000000000000000,
          2.0000000000000000, 0.10000000000000009 },
  { 1.0000000000000000, 0.0000000000000000, 2.0000000000000000,
          2.0000000000000000, 0.19999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 2.0000000000000000,
          2.0000000000000000, 0.30000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 2.0000000000000000,
          2.0000000000000000, 0.39999999999999991 },
  { 1.0000000000000000, 0.0000000000000000, 2.0000000000000000,
          2.0000000000000000, 0.50000000000000000 },
  { 1.0000000000000000, 0.0000000000000000, 2.0000000000000000,
          2.0000000000000000, 0.60000000000000009 },
  { 1.0000000000000000, 0.0000000000000000, 2.0000000000000000,
          2.0000000000000000, 0.69999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 2.0000000000000000,
          2.0000000000000000, 0.80000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 2.0000000000000000,
          2.0000000000000000, 0.89999999999999991 },
};

// Test function for a=0.0000000000000000, b=2.0000000000000000, c=2.0000000000000000.
template <typename Tp>
void test016()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data016)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data016[i].a), Tp(data016[i].b),
                   Tp(data016[i].c), Tp(data016[i].x));
      const Tp f0 = data016[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=0.0000000000000000, b=2.0000000000000000, c=4.0000000000000000.
testcase_hyperg<double> data017[] = {
  { 1.0000000000000000, 0.0000000000000000, 2.0000000000000000,
          4.0000000000000000, -0.90000000000000002 },
  { 1.0000000000000000, 0.0000000000000000, 2.0000000000000000,
          4.0000000000000000, -0.80000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 2.0000000000000000,
          4.0000000000000000, -0.69999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 2.0000000000000000,
          4.0000000000000000, -0.59999999999999998 },
  { 1.0000000000000000, 0.0000000000000000, 2.0000000000000000,
          4.0000000000000000, -0.50000000000000000 },
  { 1.0000000000000000, 0.0000000000000000, 2.0000000000000000,
          4.0000000000000000, -0.40000000000000002 },
  { 1.0000000000000000, 0.0000000000000000, 2.0000000000000000,
          4.0000000000000000, -0.30000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 2.0000000000000000,
          4.0000000000000000, -0.19999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 2.0000000000000000,
          4.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 0.0000000000000000, 2.0000000000000000,
          4.0000000000000000, 0.0000000000000000 },
  { 1.0000000000000000, 0.0000000000000000, 2.0000000000000000,
          4.0000000000000000, 0.10000000000000009 },
  { 1.0000000000000000, 0.0000000000000000, 2.0000000000000000,
          4.0000000000000000, 0.19999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 2.0000000000000000,
          4.0000000000000000, 0.30000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 2.0000000000000000,
          4.0000000000000000, 0.39999999999999991 },
  { 1.0000000000000000, 0.0000000000000000, 2.0000000000000000,
          4.0000000000000000, 0.50000000000000000 },
  { 1.0000000000000000, 0.0000000000000000, 2.0000000000000000,
          4.0000000000000000, 0.60000000000000009 },
  { 1.0000000000000000, 0.0000000000000000, 2.0000000000000000,
          4.0000000000000000, 0.69999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 2.0000000000000000,
          4.0000000000000000, 0.80000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 2.0000000000000000,
          4.0000000000000000, 0.89999999999999991 },
};

// Test function for a=0.0000000000000000, b=2.0000000000000000, c=4.0000000000000000.
template <typename Tp>
void test017()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data017)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data017[i].a), Tp(data017[i].b),
                   Tp(data017[i].c), Tp(data017[i].x));
      const Tp f0 = data017[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=0.0000000000000000, b=2.0000000000000000, c=6.0000000000000000.
testcase_hyperg<double> data018[] = {
  { 1.0000000000000000, 0.0000000000000000, 2.0000000000000000,
          6.0000000000000000, -0.90000000000000002 },
  { 1.0000000000000000, 0.0000000000000000, 2.0000000000000000,
          6.0000000000000000, -0.80000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 2.0000000000000000,
          6.0000000000000000, -0.69999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 2.0000000000000000,
          6.0000000000000000, -0.59999999999999998 },
  { 1.0000000000000000, 0.0000000000000000, 2.0000000000000000,
          6.0000000000000000, -0.50000000000000000 },
  { 1.0000000000000000, 0.0000000000000000, 2.0000000000000000,
          6.0000000000000000, -0.40000000000000002 },
  { 1.0000000000000000, 0.0000000000000000, 2.0000000000000000,
          6.0000000000000000, -0.30000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 2.0000000000000000,
          6.0000000000000000, -0.19999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 2.0000000000000000,
          6.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 0.0000000000000000, 2.0000000000000000,
          6.0000000000000000, 0.0000000000000000 },
  { 1.0000000000000000, 0.0000000000000000, 2.0000000000000000,
          6.0000000000000000, 0.10000000000000009 },
  { 1.0000000000000000, 0.0000000000000000, 2.0000000000000000,
          6.0000000000000000, 0.19999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 2.0000000000000000,
          6.0000000000000000, 0.30000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 2.0000000000000000,
          6.0000000000000000, 0.39999999999999991 },
  { 1.0000000000000000, 0.0000000000000000, 2.0000000000000000,
          6.0000000000000000, 0.50000000000000000 },
  { 1.0000000000000000, 0.0000000000000000, 2.0000000000000000,
          6.0000000000000000, 0.60000000000000009 },
  { 1.0000000000000000, 0.0000000000000000, 2.0000000000000000,
          6.0000000000000000, 0.69999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 2.0000000000000000,
          6.0000000000000000, 0.80000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 2.0000000000000000,
          6.0000000000000000, 0.89999999999999991 },
};

// Test function for a=0.0000000000000000, b=2.0000000000000000, c=6.0000000000000000.
template <typename Tp>
void test018()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data018)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data018[i].a), Tp(data018[i].b),
                   Tp(data018[i].c), Tp(data018[i].x));
      const Tp f0 = data018[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=0.0000000000000000, b=2.0000000000000000, c=8.0000000000000000.
testcase_hyperg<double> data019[] = {
  { 1.0000000000000000, 0.0000000000000000, 2.0000000000000000,
          8.0000000000000000, -0.90000000000000002 },
  { 1.0000000000000000, 0.0000000000000000, 2.0000000000000000,
          8.0000000000000000, -0.80000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 2.0000000000000000,
          8.0000000000000000, -0.69999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 2.0000000000000000,
          8.0000000000000000, -0.59999999999999998 },
  { 1.0000000000000000, 0.0000000000000000, 2.0000000000000000,
          8.0000000000000000, -0.50000000000000000 },
  { 1.0000000000000000, 0.0000000000000000, 2.0000000000000000,
          8.0000000000000000, -0.40000000000000002 },
  { 1.0000000000000000, 0.0000000000000000, 2.0000000000000000,
          8.0000000000000000, -0.30000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 2.0000000000000000,
          8.0000000000000000, -0.19999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 2.0000000000000000,
          8.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 0.0000000000000000, 2.0000000000000000,
          8.0000000000000000, 0.0000000000000000 },
  { 1.0000000000000000, 0.0000000000000000, 2.0000000000000000,
          8.0000000000000000, 0.10000000000000009 },
  { 1.0000000000000000, 0.0000000000000000, 2.0000000000000000,
          8.0000000000000000, 0.19999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 2.0000000000000000,
          8.0000000000000000, 0.30000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 2.0000000000000000,
          8.0000000000000000, 0.39999999999999991 },
  { 1.0000000000000000, 0.0000000000000000, 2.0000000000000000,
          8.0000000000000000, 0.50000000000000000 },
  { 1.0000000000000000, 0.0000000000000000, 2.0000000000000000,
          8.0000000000000000, 0.60000000000000009 },
  { 1.0000000000000000, 0.0000000000000000, 2.0000000000000000,
          8.0000000000000000, 0.69999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 2.0000000000000000,
          8.0000000000000000, 0.80000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 2.0000000000000000,
          8.0000000000000000, 0.89999999999999991 },
};

// Test function for a=0.0000000000000000, b=2.0000000000000000, c=8.0000000000000000.
template <typename Tp>
void test019()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data019)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data019[i].a), Tp(data019[i].b),
                   Tp(data019[i].c), Tp(data019[i].x));
      const Tp f0 = data019[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=0.0000000000000000, b=2.0000000000000000, c=10.000000000000000.
testcase_hyperg<double> data020[] = {
  { 1.0000000000000000, 0.0000000000000000, 2.0000000000000000,
          10.000000000000000, -0.90000000000000002 },
  { 1.0000000000000000, 0.0000000000000000, 2.0000000000000000,
          10.000000000000000, -0.80000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 2.0000000000000000,
          10.000000000000000, -0.69999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 2.0000000000000000,
          10.000000000000000, -0.59999999999999998 },
  { 1.0000000000000000, 0.0000000000000000, 2.0000000000000000,
          10.000000000000000, -0.50000000000000000 },
  { 1.0000000000000000, 0.0000000000000000, 2.0000000000000000,
          10.000000000000000, -0.40000000000000002 },
  { 1.0000000000000000, 0.0000000000000000, 2.0000000000000000,
          10.000000000000000, -0.30000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 2.0000000000000000,
          10.000000000000000, -0.19999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 2.0000000000000000,
          10.000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 0.0000000000000000, 2.0000000000000000,
          10.000000000000000, 0.0000000000000000 },
  { 1.0000000000000000, 0.0000000000000000, 2.0000000000000000,
          10.000000000000000, 0.10000000000000009 },
  { 1.0000000000000000, 0.0000000000000000, 2.0000000000000000,
          10.000000000000000, 0.19999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 2.0000000000000000,
          10.000000000000000, 0.30000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 2.0000000000000000,
          10.000000000000000, 0.39999999999999991 },
  { 1.0000000000000000, 0.0000000000000000, 2.0000000000000000,
          10.000000000000000, 0.50000000000000000 },
  { 1.0000000000000000, 0.0000000000000000, 2.0000000000000000,
          10.000000000000000, 0.60000000000000009 },
  { 1.0000000000000000, 0.0000000000000000, 2.0000000000000000,
          10.000000000000000, 0.69999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 2.0000000000000000,
          10.000000000000000, 0.80000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 2.0000000000000000,
          10.000000000000000, 0.89999999999999991 },
};

// Test function for a=0.0000000000000000, b=2.0000000000000000, c=10.000000000000000.
template <typename Tp>
void test020()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data020)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data020[i].a), Tp(data020[i].b),
                   Tp(data020[i].c), Tp(data020[i].x));
      const Tp f0 = data020[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=0.0000000000000000, b=5.0000000000000000, c=2.0000000000000000.
testcase_hyperg<double> data021[] = {
  { 1.0000000000000000, 0.0000000000000000, 5.0000000000000000,
          2.0000000000000000, -0.90000000000000002 },
  { 1.0000000000000000, 0.0000000000000000, 5.0000000000000000,
          2.0000000000000000, -0.80000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 5.0000000000000000,
          2.0000000000000000, -0.69999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 5.0000000000000000,
          2.0000000000000000, -0.59999999999999998 },
  { 1.0000000000000000, 0.0000000000000000, 5.0000000000000000,
          2.0000000000000000, -0.50000000000000000 },
  { 1.0000000000000000, 0.0000000000000000, 5.0000000000000000,
          2.0000000000000000, -0.40000000000000002 },
  { 1.0000000000000000, 0.0000000000000000, 5.0000000000000000,
          2.0000000000000000, -0.30000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 5.0000000000000000,
          2.0000000000000000, -0.19999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 5.0000000000000000,
          2.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 0.0000000000000000, 5.0000000000000000,
          2.0000000000000000, 0.0000000000000000 },
  { 1.0000000000000000, 0.0000000000000000, 5.0000000000000000,
          2.0000000000000000, 0.10000000000000009 },
  { 1.0000000000000000, 0.0000000000000000, 5.0000000000000000,
          2.0000000000000000, 0.19999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 5.0000000000000000,
          2.0000000000000000, 0.30000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 5.0000000000000000,
          2.0000000000000000, 0.39999999999999991 },
  { 1.0000000000000000, 0.0000000000000000, 5.0000000000000000,
          2.0000000000000000, 0.50000000000000000 },
  { 1.0000000000000000, 0.0000000000000000, 5.0000000000000000,
          2.0000000000000000, 0.60000000000000009 },
  { 1.0000000000000000, 0.0000000000000000, 5.0000000000000000,
          2.0000000000000000, 0.69999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 5.0000000000000000,
          2.0000000000000000, 0.80000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 5.0000000000000000,
          2.0000000000000000, 0.89999999999999991 },
};

// Test function for a=0.0000000000000000, b=5.0000000000000000, c=2.0000000000000000.
template <typename Tp>
void test021()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data021)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data021[i].a), Tp(data021[i].b),
                   Tp(data021[i].c), Tp(data021[i].x));
      const Tp f0 = data021[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=0.0000000000000000, b=5.0000000000000000, c=4.0000000000000000.
testcase_hyperg<double> data022[] = {
  { 1.0000000000000000, 0.0000000000000000, 5.0000000000000000,
          4.0000000000000000, -0.90000000000000002 },
  { 1.0000000000000000, 0.0000000000000000, 5.0000000000000000,
          4.0000000000000000, -0.80000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 5.0000000000000000,
          4.0000000000000000, -0.69999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 5.0000000000000000,
          4.0000000000000000, -0.59999999999999998 },
  { 1.0000000000000000, 0.0000000000000000, 5.0000000000000000,
          4.0000000000000000, -0.50000000000000000 },
  { 1.0000000000000000, 0.0000000000000000, 5.0000000000000000,
          4.0000000000000000, -0.40000000000000002 },
  { 1.0000000000000000, 0.0000000000000000, 5.0000000000000000,
          4.0000000000000000, -0.30000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 5.0000000000000000,
          4.0000000000000000, -0.19999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 5.0000000000000000,
          4.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 0.0000000000000000, 5.0000000000000000,
          4.0000000000000000, 0.0000000000000000 },
  { 1.0000000000000000, 0.0000000000000000, 5.0000000000000000,
          4.0000000000000000, 0.10000000000000009 },
  { 1.0000000000000000, 0.0000000000000000, 5.0000000000000000,
          4.0000000000000000, 0.19999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 5.0000000000000000,
          4.0000000000000000, 0.30000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 5.0000000000000000,
          4.0000000000000000, 0.39999999999999991 },
  { 1.0000000000000000, 0.0000000000000000, 5.0000000000000000,
          4.0000000000000000, 0.50000000000000000 },
  { 1.0000000000000000, 0.0000000000000000, 5.0000000000000000,
          4.0000000000000000, 0.60000000000000009 },
  { 1.0000000000000000, 0.0000000000000000, 5.0000000000000000,
          4.0000000000000000, 0.69999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 5.0000000000000000,
          4.0000000000000000, 0.80000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 5.0000000000000000,
          4.0000000000000000, 0.89999999999999991 },
};

// Test function for a=0.0000000000000000, b=5.0000000000000000, c=4.0000000000000000.
template <typename Tp>
void test022()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data022)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data022[i].a), Tp(data022[i].b),
                   Tp(data022[i].c), Tp(data022[i].x));
      const Tp f0 = data022[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=0.0000000000000000, b=5.0000000000000000, c=6.0000000000000000.
testcase_hyperg<double> data023[] = {
  { 1.0000000000000000, 0.0000000000000000, 5.0000000000000000,
          6.0000000000000000, -0.90000000000000002 },
  { 1.0000000000000000, 0.0000000000000000, 5.0000000000000000,
          6.0000000000000000, -0.80000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 5.0000000000000000,
          6.0000000000000000, -0.69999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 5.0000000000000000,
          6.0000000000000000, -0.59999999999999998 },
  { 1.0000000000000000, 0.0000000000000000, 5.0000000000000000,
          6.0000000000000000, -0.50000000000000000 },
  { 1.0000000000000000, 0.0000000000000000, 5.0000000000000000,
          6.0000000000000000, -0.40000000000000002 },
  { 1.0000000000000000, 0.0000000000000000, 5.0000000000000000,
          6.0000000000000000, -0.30000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 5.0000000000000000,
          6.0000000000000000, -0.19999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 5.0000000000000000,
          6.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 0.0000000000000000, 5.0000000000000000,
          6.0000000000000000, 0.0000000000000000 },
  { 1.0000000000000000, 0.0000000000000000, 5.0000000000000000,
          6.0000000000000000, 0.10000000000000009 },
  { 1.0000000000000000, 0.0000000000000000, 5.0000000000000000,
          6.0000000000000000, 0.19999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 5.0000000000000000,
          6.0000000000000000, 0.30000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 5.0000000000000000,
          6.0000000000000000, 0.39999999999999991 },
  { 1.0000000000000000, 0.0000000000000000, 5.0000000000000000,
          6.0000000000000000, 0.50000000000000000 },
  { 1.0000000000000000, 0.0000000000000000, 5.0000000000000000,
          6.0000000000000000, 0.60000000000000009 },
  { 1.0000000000000000, 0.0000000000000000, 5.0000000000000000,
          6.0000000000000000, 0.69999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 5.0000000000000000,
          6.0000000000000000, 0.80000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 5.0000000000000000,
          6.0000000000000000, 0.89999999999999991 },
};

// Test function for a=0.0000000000000000, b=5.0000000000000000, c=6.0000000000000000.
template <typename Tp>
void test023()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data023)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data023[i].a), Tp(data023[i].b),
                   Tp(data023[i].c), Tp(data023[i].x));
      const Tp f0 = data023[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=0.0000000000000000, b=5.0000000000000000, c=8.0000000000000000.
testcase_hyperg<double> data024[] = {
  { 1.0000000000000000, 0.0000000000000000, 5.0000000000000000,
          8.0000000000000000, -0.90000000000000002 },
  { 1.0000000000000000, 0.0000000000000000, 5.0000000000000000,
          8.0000000000000000, -0.80000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 5.0000000000000000,
          8.0000000000000000, -0.69999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 5.0000000000000000,
          8.0000000000000000, -0.59999999999999998 },
  { 1.0000000000000000, 0.0000000000000000, 5.0000000000000000,
          8.0000000000000000, -0.50000000000000000 },
  { 1.0000000000000000, 0.0000000000000000, 5.0000000000000000,
          8.0000000000000000, -0.40000000000000002 },
  { 1.0000000000000000, 0.0000000000000000, 5.0000000000000000,
          8.0000000000000000, -0.30000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 5.0000000000000000,
          8.0000000000000000, -0.19999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 5.0000000000000000,
          8.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 0.0000000000000000, 5.0000000000000000,
          8.0000000000000000, 0.0000000000000000 },
  { 1.0000000000000000, 0.0000000000000000, 5.0000000000000000,
          8.0000000000000000, 0.10000000000000009 },
  { 1.0000000000000000, 0.0000000000000000, 5.0000000000000000,
          8.0000000000000000, 0.19999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 5.0000000000000000,
          8.0000000000000000, 0.30000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 5.0000000000000000,
          8.0000000000000000, 0.39999999999999991 },
  { 1.0000000000000000, 0.0000000000000000, 5.0000000000000000,
          8.0000000000000000, 0.50000000000000000 },
  { 1.0000000000000000, 0.0000000000000000, 5.0000000000000000,
          8.0000000000000000, 0.60000000000000009 },
  { 1.0000000000000000, 0.0000000000000000, 5.0000000000000000,
          8.0000000000000000, 0.69999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 5.0000000000000000,
          8.0000000000000000, 0.80000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 5.0000000000000000,
          8.0000000000000000, 0.89999999999999991 },
};

// Test function for a=0.0000000000000000, b=5.0000000000000000, c=8.0000000000000000.
template <typename Tp>
void test024()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data024)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data024[i].a), Tp(data024[i].b),
                   Tp(data024[i].c), Tp(data024[i].x));
      const Tp f0 = data024[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=0.0000000000000000, b=5.0000000000000000, c=10.000000000000000.
testcase_hyperg<double> data025[] = {
  { 1.0000000000000000, 0.0000000000000000, 5.0000000000000000,
          10.000000000000000, -0.90000000000000002 },
  { 1.0000000000000000, 0.0000000000000000, 5.0000000000000000,
          10.000000000000000, -0.80000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 5.0000000000000000,
          10.000000000000000, -0.69999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 5.0000000000000000,
          10.000000000000000, -0.59999999999999998 },
  { 1.0000000000000000, 0.0000000000000000, 5.0000000000000000,
          10.000000000000000, -0.50000000000000000 },
  { 1.0000000000000000, 0.0000000000000000, 5.0000000000000000,
          10.000000000000000, -0.40000000000000002 },
  { 1.0000000000000000, 0.0000000000000000, 5.0000000000000000,
          10.000000000000000, -0.30000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 5.0000000000000000,
          10.000000000000000, -0.19999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 5.0000000000000000,
          10.000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 0.0000000000000000, 5.0000000000000000,
          10.000000000000000, 0.0000000000000000 },
  { 1.0000000000000000, 0.0000000000000000, 5.0000000000000000,
          10.000000000000000, 0.10000000000000009 },
  { 1.0000000000000000, 0.0000000000000000, 5.0000000000000000,
          10.000000000000000, 0.19999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 5.0000000000000000,
          10.000000000000000, 0.30000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 5.0000000000000000,
          10.000000000000000, 0.39999999999999991 },
  { 1.0000000000000000, 0.0000000000000000, 5.0000000000000000,
          10.000000000000000, 0.50000000000000000 },
  { 1.0000000000000000, 0.0000000000000000, 5.0000000000000000,
          10.000000000000000, 0.60000000000000009 },
  { 1.0000000000000000, 0.0000000000000000, 5.0000000000000000,
          10.000000000000000, 0.69999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 5.0000000000000000,
          10.000000000000000, 0.80000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 5.0000000000000000,
          10.000000000000000, 0.89999999999999991 },
};

// Test function for a=0.0000000000000000, b=5.0000000000000000, c=10.000000000000000.
template <typename Tp>
void test025()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data025)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data025[i].a), Tp(data025[i].b),
                   Tp(data025[i].c), Tp(data025[i].x));
      const Tp f0 = data025[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=0.0000000000000000, b=10.000000000000000, c=2.0000000000000000.
testcase_hyperg<double> data026[] = {
  { 1.0000000000000000, 0.0000000000000000, 10.000000000000000,
          2.0000000000000000, -0.90000000000000002 },
  { 1.0000000000000000, 0.0000000000000000, 10.000000000000000,
          2.0000000000000000, -0.80000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 10.000000000000000,
          2.0000000000000000, -0.69999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 10.000000000000000,
          2.0000000000000000, -0.59999999999999998 },
  { 1.0000000000000000, 0.0000000000000000, 10.000000000000000,
          2.0000000000000000, -0.50000000000000000 },
  { 1.0000000000000000, 0.0000000000000000, 10.000000000000000,
          2.0000000000000000, -0.40000000000000002 },
  { 1.0000000000000000, 0.0000000000000000, 10.000000000000000,
          2.0000000000000000, -0.30000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 10.000000000000000,
          2.0000000000000000, -0.19999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 10.000000000000000,
          2.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 0.0000000000000000, 10.000000000000000,
          2.0000000000000000, 0.0000000000000000 },
  { 1.0000000000000000, 0.0000000000000000, 10.000000000000000,
          2.0000000000000000, 0.10000000000000009 },
  { 1.0000000000000000, 0.0000000000000000, 10.000000000000000,
          2.0000000000000000, 0.19999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 10.000000000000000,
          2.0000000000000000, 0.30000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 10.000000000000000,
          2.0000000000000000, 0.39999999999999991 },
  { 1.0000000000000000, 0.0000000000000000, 10.000000000000000,
          2.0000000000000000, 0.50000000000000000 },
  { 1.0000000000000000, 0.0000000000000000, 10.000000000000000,
          2.0000000000000000, 0.60000000000000009 },
  { 1.0000000000000000, 0.0000000000000000, 10.000000000000000,
          2.0000000000000000, 0.69999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 10.000000000000000,
          2.0000000000000000, 0.80000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 10.000000000000000,
          2.0000000000000000, 0.89999999999999991 },
};

// Test function for a=0.0000000000000000, b=10.000000000000000, c=2.0000000000000000.
template <typename Tp>
void test026()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data026)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data026[i].a), Tp(data026[i].b),
                   Tp(data026[i].c), Tp(data026[i].x));
      const Tp f0 = data026[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=0.0000000000000000, b=10.000000000000000, c=4.0000000000000000.
testcase_hyperg<double> data027[] = {
  { 1.0000000000000000, 0.0000000000000000, 10.000000000000000,
          4.0000000000000000, -0.90000000000000002 },
  { 1.0000000000000000, 0.0000000000000000, 10.000000000000000,
          4.0000000000000000, -0.80000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 10.000000000000000,
          4.0000000000000000, -0.69999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 10.000000000000000,
          4.0000000000000000, -0.59999999999999998 },
  { 1.0000000000000000, 0.0000000000000000, 10.000000000000000,
          4.0000000000000000, -0.50000000000000000 },
  { 1.0000000000000000, 0.0000000000000000, 10.000000000000000,
          4.0000000000000000, -0.40000000000000002 },
  { 1.0000000000000000, 0.0000000000000000, 10.000000000000000,
          4.0000000000000000, -0.30000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 10.000000000000000,
          4.0000000000000000, -0.19999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 10.000000000000000,
          4.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 0.0000000000000000, 10.000000000000000,
          4.0000000000000000, 0.0000000000000000 },
  { 1.0000000000000000, 0.0000000000000000, 10.000000000000000,
          4.0000000000000000, 0.10000000000000009 },
  { 1.0000000000000000, 0.0000000000000000, 10.000000000000000,
          4.0000000000000000, 0.19999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 10.000000000000000,
          4.0000000000000000, 0.30000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 10.000000000000000,
          4.0000000000000000, 0.39999999999999991 },
  { 1.0000000000000000, 0.0000000000000000, 10.000000000000000,
          4.0000000000000000, 0.50000000000000000 },
  { 1.0000000000000000, 0.0000000000000000, 10.000000000000000,
          4.0000000000000000, 0.60000000000000009 },
  { 1.0000000000000000, 0.0000000000000000, 10.000000000000000,
          4.0000000000000000, 0.69999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 10.000000000000000,
          4.0000000000000000, 0.80000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 10.000000000000000,
          4.0000000000000000, 0.89999999999999991 },
};

// Test function for a=0.0000000000000000, b=10.000000000000000, c=4.0000000000000000.
template <typename Tp>
void test027()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data027)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data027[i].a), Tp(data027[i].b),
                   Tp(data027[i].c), Tp(data027[i].x));
      const Tp f0 = data027[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=0.0000000000000000, b=10.000000000000000, c=6.0000000000000000.
testcase_hyperg<double> data028[] = {
  { 1.0000000000000000, 0.0000000000000000, 10.000000000000000,
          6.0000000000000000, -0.90000000000000002 },
  { 1.0000000000000000, 0.0000000000000000, 10.000000000000000,
          6.0000000000000000, -0.80000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 10.000000000000000,
          6.0000000000000000, -0.69999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 10.000000000000000,
          6.0000000000000000, -0.59999999999999998 },
  { 1.0000000000000000, 0.0000000000000000, 10.000000000000000,
          6.0000000000000000, -0.50000000000000000 },
  { 1.0000000000000000, 0.0000000000000000, 10.000000000000000,
          6.0000000000000000, -0.40000000000000002 },
  { 1.0000000000000000, 0.0000000000000000, 10.000000000000000,
          6.0000000000000000, -0.30000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 10.000000000000000,
          6.0000000000000000, -0.19999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 10.000000000000000,
          6.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 0.0000000000000000, 10.000000000000000,
          6.0000000000000000, 0.0000000000000000 },
  { 1.0000000000000000, 0.0000000000000000, 10.000000000000000,
          6.0000000000000000, 0.10000000000000009 },
  { 1.0000000000000000, 0.0000000000000000, 10.000000000000000,
          6.0000000000000000, 0.19999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 10.000000000000000,
          6.0000000000000000, 0.30000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 10.000000000000000,
          6.0000000000000000, 0.39999999999999991 },
  { 1.0000000000000000, 0.0000000000000000, 10.000000000000000,
          6.0000000000000000, 0.50000000000000000 },
  { 1.0000000000000000, 0.0000000000000000, 10.000000000000000,
          6.0000000000000000, 0.60000000000000009 },
  { 1.0000000000000000, 0.0000000000000000, 10.000000000000000,
          6.0000000000000000, 0.69999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 10.000000000000000,
          6.0000000000000000, 0.80000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 10.000000000000000,
          6.0000000000000000, 0.89999999999999991 },
};

// Test function for a=0.0000000000000000, b=10.000000000000000, c=6.0000000000000000.
template <typename Tp>
void test028()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data028)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data028[i].a), Tp(data028[i].b),
                   Tp(data028[i].c), Tp(data028[i].x));
      const Tp f0 = data028[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=0.0000000000000000, b=10.000000000000000, c=8.0000000000000000.
testcase_hyperg<double> data029[] = {
  { 1.0000000000000000, 0.0000000000000000, 10.000000000000000,
          8.0000000000000000, -0.90000000000000002 },
  { 1.0000000000000000, 0.0000000000000000, 10.000000000000000,
          8.0000000000000000, -0.80000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 10.000000000000000,
          8.0000000000000000, -0.69999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 10.000000000000000,
          8.0000000000000000, -0.59999999999999998 },
  { 1.0000000000000000, 0.0000000000000000, 10.000000000000000,
          8.0000000000000000, -0.50000000000000000 },
  { 1.0000000000000000, 0.0000000000000000, 10.000000000000000,
          8.0000000000000000, -0.40000000000000002 },
  { 1.0000000000000000, 0.0000000000000000, 10.000000000000000,
          8.0000000000000000, -0.30000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 10.000000000000000,
          8.0000000000000000, -0.19999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 10.000000000000000,
          8.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 0.0000000000000000, 10.000000000000000,
          8.0000000000000000, 0.0000000000000000 },
  { 1.0000000000000000, 0.0000000000000000, 10.000000000000000,
          8.0000000000000000, 0.10000000000000009 },
  { 1.0000000000000000, 0.0000000000000000, 10.000000000000000,
          8.0000000000000000, 0.19999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 10.000000000000000,
          8.0000000000000000, 0.30000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 10.000000000000000,
          8.0000000000000000, 0.39999999999999991 },
  { 1.0000000000000000, 0.0000000000000000, 10.000000000000000,
          8.0000000000000000, 0.50000000000000000 },
  { 1.0000000000000000, 0.0000000000000000, 10.000000000000000,
          8.0000000000000000, 0.60000000000000009 },
  { 1.0000000000000000, 0.0000000000000000, 10.000000000000000,
          8.0000000000000000, 0.69999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 10.000000000000000,
          8.0000000000000000, 0.80000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 10.000000000000000,
          8.0000000000000000, 0.89999999999999991 },
};

// Test function for a=0.0000000000000000, b=10.000000000000000, c=8.0000000000000000.
template <typename Tp>
void test029()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data029)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data029[i].a), Tp(data029[i].b),
                   Tp(data029[i].c), Tp(data029[i].x));
      const Tp f0 = data029[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=0.0000000000000000, b=10.000000000000000, c=10.000000000000000.
testcase_hyperg<double> data030[] = {
  { 1.0000000000000000, 0.0000000000000000, 10.000000000000000,
          10.000000000000000, -0.90000000000000002 },
  { 1.0000000000000000, 0.0000000000000000, 10.000000000000000,
          10.000000000000000, -0.80000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 10.000000000000000,
          10.000000000000000, -0.69999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 10.000000000000000,
          10.000000000000000, -0.59999999999999998 },
  { 1.0000000000000000, 0.0000000000000000, 10.000000000000000,
          10.000000000000000, -0.50000000000000000 },
  { 1.0000000000000000, 0.0000000000000000, 10.000000000000000,
          10.000000000000000, -0.40000000000000002 },
  { 1.0000000000000000, 0.0000000000000000, 10.000000000000000,
          10.000000000000000, -0.30000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 10.000000000000000,
          10.000000000000000, -0.19999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 10.000000000000000,
          10.000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 0.0000000000000000, 10.000000000000000,
          10.000000000000000, 0.0000000000000000 },
  { 1.0000000000000000, 0.0000000000000000, 10.000000000000000,
          10.000000000000000, 0.10000000000000009 },
  { 1.0000000000000000, 0.0000000000000000, 10.000000000000000,
          10.000000000000000, 0.19999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 10.000000000000000,
          10.000000000000000, 0.30000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 10.000000000000000,
          10.000000000000000, 0.39999999999999991 },
  { 1.0000000000000000, 0.0000000000000000, 10.000000000000000,
          10.000000000000000, 0.50000000000000000 },
  { 1.0000000000000000, 0.0000000000000000, 10.000000000000000,
          10.000000000000000, 0.60000000000000009 },
  { 1.0000000000000000, 0.0000000000000000, 10.000000000000000,
          10.000000000000000, 0.69999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 10.000000000000000,
          10.000000000000000, 0.80000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 10.000000000000000,
          10.000000000000000, 0.89999999999999991 },
};

// Test function for a=0.0000000000000000, b=10.000000000000000, c=10.000000000000000.
template <typename Tp>
void test030()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data030)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data030[i].a), Tp(data030[i].b),
                   Tp(data030[i].c), Tp(data030[i].x));
      const Tp f0 = data030[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=0.0000000000000000, b=20.000000000000000, c=2.0000000000000000.
testcase_hyperg<double> data031[] = {
  { 1.0000000000000000, 0.0000000000000000, 20.000000000000000,
          2.0000000000000000, -0.90000000000000002 },
  { 1.0000000000000000, 0.0000000000000000, 20.000000000000000,
          2.0000000000000000, -0.80000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 20.000000000000000,
          2.0000000000000000, -0.69999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 20.000000000000000,
          2.0000000000000000, -0.59999999999999998 },
  { 1.0000000000000000, 0.0000000000000000, 20.000000000000000,
          2.0000000000000000, -0.50000000000000000 },
  { 1.0000000000000000, 0.0000000000000000, 20.000000000000000,
          2.0000000000000000, -0.40000000000000002 },
  { 1.0000000000000000, 0.0000000000000000, 20.000000000000000,
          2.0000000000000000, -0.30000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 20.000000000000000,
          2.0000000000000000, -0.19999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 20.000000000000000,
          2.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 0.0000000000000000, 20.000000000000000,
          2.0000000000000000, 0.0000000000000000 },
  { 1.0000000000000000, 0.0000000000000000, 20.000000000000000,
          2.0000000000000000, 0.10000000000000009 },
  { 1.0000000000000000, 0.0000000000000000, 20.000000000000000,
          2.0000000000000000, 0.19999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 20.000000000000000,
          2.0000000000000000, 0.30000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 20.000000000000000,
          2.0000000000000000, 0.39999999999999991 },
  { 1.0000000000000000, 0.0000000000000000, 20.000000000000000,
          2.0000000000000000, 0.50000000000000000 },
  { 1.0000000000000000, 0.0000000000000000, 20.000000000000000,
          2.0000000000000000, 0.60000000000000009 },
  { 1.0000000000000000, 0.0000000000000000, 20.000000000000000,
          2.0000000000000000, 0.69999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 20.000000000000000,
          2.0000000000000000, 0.80000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 20.000000000000000,
          2.0000000000000000, 0.89999999999999991 },
};

// Test function for a=0.0000000000000000, b=20.000000000000000, c=2.0000000000000000.
template <typename Tp>
void test031()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data031)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data031[i].a), Tp(data031[i].b),
                   Tp(data031[i].c), Tp(data031[i].x));
      const Tp f0 = data031[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=0.0000000000000000, b=20.000000000000000, c=4.0000000000000000.
testcase_hyperg<double> data032[] = {
  { 1.0000000000000000, 0.0000000000000000, 20.000000000000000,
          4.0000000000000000, -0.90000000000000002 },
  { 1.0000000000000000, 0.0000000000000000, 20.000000000000000,
          4.0000000000000000, -0.80000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 20.000000000000000,
          4.0000000000000000, -0.69999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 20.000000000000000,
          4.0000000000000000, -0.59999999999999998 },
  { 1.0000000000000000, 0.0000000000000000, 20.000000000000000,
          4.0000000000000000, -0.50000000000000000 },
  { 1.0000000000000000, 0.0000000000000000, 20.000000000000000,
          4.0000000000000000, -0.40000000000000002 },
  { 1.0000000000000000, 0.0000000000000000, 20.000000000000000,
          4.0000000000000000, -0.30000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 20.000000000000000,
          4.0000000000000000, -0.19999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 20.000000000000000,
          4.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 0.0000000000000000, 20.000000000000000,
          4.0000000000000000, 0.0000000000000000 },
  { 1.0000000000000000, 0.0000000000000000, 20.000000000000000,
          4.0000000000000000, 0.10000000000000009 },
  { 1.0000000000000000, 0.0000000000000000, 20.000000000000000,
          4.0000000000000000, 0.19999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 20.000000000000000,
          4.0000000000000000, 0.30000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 20.000000000000000,
          4.0000000000000000, 0.39999999999999991 },
  { 1.0000000000000000, 0.0000000000000000, 20.000000000000000,
          4.0000000000000000, 0.50000000000000000 },
  { 1.0000000000000000, 0.0000000000000000, 20.000000000000000,
          4.0000000000000000, 0.60000000000000009 },
  { 1.0000000000000000, 0.0000000000000000, 20.000000000000000,
          4.0000000000000000, 0.69999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 20.000000000000000,
          4.0000000000000000, 0.80000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 20.000000000000000,
          4.0000000000000000, 0.89999999999999991 },
};

// Test function for a=0.0000000000000000, b=20.000000000000000, c=4.0000000000000000.
template <typename Tp>
void test032()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data032)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data032[i].a), Tp(data032[i].b),
                   Tp(data032[i].c), Tp(data032[i].x));
      const Tp f0 = data032[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=0.0000000000000000, b=20.000000000000000, c=6.0000000000000000.
testcase_hyperg<double> data033[] = {
  { 1.0000000000000000, 0.0000000000000000, 20.000000000000000,
          6.0000000000000000, -0.90000000000000002 },
  { 1.0000000000000000, 0.0000000000000000, 20.000000000000000,
          6.0000000000000000, -0.80000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 20.000000000000000,
          6.0000000000000000, -0.69999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 20.000000000000000,
          6.0000000000000000, -0.59999999999999998 },
  { 1.0000000000000000, 0.0000000000000000, 20.000000000000000,
          6.0000000000000000, -0.50000000000000000 },
  { 1.0000000000000000, 0.0000000000000000, 20.000000000000000,
          6.0000000000000000, -0.40000000000000002 },
  { 1.0000000000000000, 0.0000000000000000, 20.000000000000000,
          6.0000000000000000, -0.30000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 20.000000000000000,
          6.0000000000000000, -0.19999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 20.000000000000000,
          6.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 0.0000000000000000, 20.000000000000000,
          6.0000000000000000, 0.0000000000000000 },
  { 1.0000000000000000, 0.0000000000000000, 20.000000000000000,
          6.0000000000000000, 0.10000000000000009 },
  { 1.0000000000000000, 0.0000000000000000, 20.000000000000000,
          6.0000000000000000, 0.19999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 20.000000000000000,
          6.0000000000000000, 0.30000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 20.000000000000000,
          6.0000000000000000, 0.39999999999999991 },
  { 1.0000000000000000, 0.0000000000000000, 20.000000000000000,
          6.0000000000000000, 0.50000000000000000 },
  { 1.0000000000000000, 0.0000000000000000, 20.000000000000000,
          6.0000000000000000, 0.60000000000000009 },
  { 1.0000000000000000, 0.0000000000000000, 20.000000000000000,
          6.0000000000000000, 0.69999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 20.000000000000000,
          6.0000000000000000, 0.80000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 20.000000000000000,
          6.0000000000000000, 0.89999999999999991 },
};

// Test function for a=0.0000000000000000, b=20.000000000000000, c=6.0000000000000000.
template <typename Tp>
void test033()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data033)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data033[i].a), Tp(data033[i].b),
                   Tp(data033[i].c), Tp(data033[i].x));
      const Tp f0 = data033[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=0.0000000000000000, b=20.000000000000000, c=8.0000000000000000.
testcase_hyperg<double> data034[] = {
  { 1.0000000000000000, 0.0000000000000000, 20.000000000000000,
          8.0000000000000000, -0.90000000000000002 },
  { 1.0000000000000000, 0.0000000000000000, 20.000000000000000,
          8.0000000000000000, -0.80000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 20.000000000000000,
          8.0000000000000000, -0.69999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 20.000000000000000,
          8.0000000000000000, -0.59999999999999998 },
  { 1.0000000000000000, 0.0000000000000000, 20.000000000000000,
          8.0000000000000000, -0.50000000000000000 },
  { 1.0000000000000000, 0.0000000000000000, 20.000000000000000,
          8.0000000000000000, -0.40000000000000002 },
  { 1.0000000000000000, 0.0000000000000000, 20.000000000000000,
          8.0000000000000000, -0.30000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 20.000000000000000,
          8.0000000000000000, -0.19999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 20.000000000000000,
          8.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 0.0000000000000000, 20.000000000000000,
          8.0000000000000000, 0.0000000000000000 },
  { 1.0000000000000000, 0.0000000000000000, 20.000000000000000,
          8.0000000000000000, 0.10000000000000009 },
  { 1.0000000000000000, 0.0000000000000000, 20.000000000000000,
          8.0000000000000000, 0.19999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 20.000000000000000,
          8.0000000000000000, 0.30000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 20.000000000000000,
          8.0000000000000000, 0.39999999999999991 },
  { 1.0000000000000000, 0.0000000000000000, 20.000000000000000,
          8.0000000000000000, 0.50000000000000000 },
  { 1.0000000000000000, 0.0000000000000000, 20.000000000000000,
          8.0000000000000000, 0.60000000000000009 },
  { 1.0000000000000000, 0.0000000000000000, 20.000000000000000,
          8.0000000000000000, 0.69999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 20.000000000000000,
          8.0000000000000000, 0.80000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 20.000000000000000,
          8.0000000000000000, 0.89999999999999991 },
};

// Test function for a=0.0000000000000000, b=20.000000000000000, c=8.0000000000000000.
template <typename Tp>
void test034()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data034)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data034[i].a), Tp(data034[i].b),
                   Tp(data034[i].c), Tp(data034[i].x));
      const Tp f0 = data034[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=0.0000000000000000, b=20.000000000000000, c=10.000000000000000.
testcase_hyperg<double> data035[] = {
  { 1.0000000000000000, 0.0000000000000000, 20.000000000000000,
          10.000000000000000, -0.90000000000000002 },
  { 1.0000000000000000, 0.0000000000000000, 20.000000000000000,
          10.000000000000000, -0.80000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 20.000000000000000,
          10.000000000000000, -0.69999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 20.000000000000000,
          10.000000000000000, -0.59999999999999998 },
  { 1.0000000000000000, 0.0000000000000000, 20.000000000000000,
          10.000000000000000, -0.50000000000000000 },
  { 1.0000000000000000, 0.0000000000000000, 20.000000000000000,
          10.000000000000000, -0.40000000000000002 },
  { 1.0000000000000000, 0.0000000000000000, 20.000000000000000,
          10.000000000000000, -0.30000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 20.000000000000000,
          10.000000000000000, -0.19999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 20.000000000000000,
          10.000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 0.0000000000000000, 20.000000000000000,
          10.000000000000000, 0.0000000000000000 },
  { 1.0000000000000000, 0.0000000000000000, 20.000000000000000,
          10.000000000000000, 0.10000000000000009 },
  { 1.0000000000000000, 0.0000000000000000, 20.000000000000000,
          10.000000000000000, 0.19999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 20.000000000000000,
          10.000000000000000, 0.30000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 20.000000000000000,
          10.000000000000000, 0.39999999999999991 },
  { 1.0000000000000000, 0.0000000000000000, 20.000000000000000,
          10.000000000000000, 0.50000000000000000 },
  { 1.0000000000000000, 0.0000000000000000, 20.000000000000000,
          10.000000000000000, 0.60000000000000009 },
  { 1.0000000000000000, 0.0000000000000000, 20.000000000000000,
          10.000000000000000, 0.69999999999999996 },
  { 1.0000000000000000, 0.0000000000000000, 20.000000000000000,
          10.000000000000000, 0.80000000000000004 },
  { 1.0000000000000000, 0.0000000000000000, 20.000000000000000,
          10.000000000000000, 0.89999999999999991 },
};

// Test function for a=0.0000000000000000, b=20.000000000000000, c=10.000000000000000.
template <typename Tp>
void test035()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data035)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data035[i].a), Tp(data035[i].b),
                   Tp(data035[i].c), Tp(data035[i].x));
      const Tp f0 = data035[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=0.50000000000000000, b=0.0000000000000000, c=2.0000000000000000.
testcase_hyperg<double> data036[] = {
  { 1.0000000000000000, 0.50000000000000000, 0.0000000000000000,
          2.0000000000000000, -0.90000000000000002 },
  { 1.0000000000000000, 0.50000000000000000, 0.0000000000000000,
          2.0000000000000000, -0.80000000000000004 },
  { 1.0000000000000000, 0.50000000000000000, 0.0000000000000000,
          2.0000000000000000, -0.69999999999999996 },
  { 1.0000000000000000, 0.50000000000000000, 0.0000000000000000,
          2.0000000000000000, -0.59999999999999998 },
  { 1.0000000000000000, 0.50000000000000000, 0.0000000000000000,
          2.0000000000000000, -0.50000000000000000 },
  { 1.0000000000000000, 0.50000000000000000, 0.0000000000000000,
          2.0000000000000000, -0.40000000000000002 },
  { 1.0000000000000000, 0.50000000000000000, 0.0000000000000000,
          2.0000000000000000, -0.30000000000000004 },
  { 1.0000000000000000, 0.50000000000000000, 0.0000000000000000,
          2.0000000000000000, -0.19999999999999996 },
  { 1.0000000000000000, 0.50000000000000000, 0.0000000000000000,
          2.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 0.50000000000000000, 0.0000000000000000,
          2.0000000000000000, 0.0000000000000000 },
  { 1.0000000000000000, 0.50000000000000000, 0.0000000000000000,
          2.0000000000000000, 0.10000000000000009 },
  { 1.0000000000000000, 0.50000000000000000, 0.0000000000000000,
          2.0000000000000000, 0.19999999999999996 },
  { 1.0000000000000000, 0.50000000000000000, 0.0000000000000000,
          2.0000000000000000, 0.30000000000000004 },
  { 1.0000000000000000, 0.50000000000000000, 0.0000000000000000,
          2.0000000000000000, 0.39999999999999991 },
  { 1.0000000000000000, 0.50000000000000000, 0.0000000000000000,
          2.0000000000000000, 0.50000000000000000 },
  { 1.0000000000000000, 0.50000000000000000, 0.0000000000000000,
          2.0000000000000000, 0.60000000000000009 },
  { 1.0000000000000000, 0.50000000000000000, 0.0000000000000000,
          2.0000000000000000, 0.69999999999999996 },
  { 1.0000000000000000, 0.50000000000000000, 0.0000000000000000,
          2.0000000000000000, 0.80000000000000004 },
  { 1.0000000000000000, 0.50000000000000000, 0.0000000000000000,
          2.0000000000000000, 0.89999999999999991 },
};

// Test function for a=0.50000000000000000, b=0.0000000000000000, c=2.0000000000000000.
template <typename Tp>
void test036()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data036)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data036[i].a), Tp(data036[i].b),
                   Tp(data036[i].c), Tp(data036[i].x));
      const Tp f0 = data036[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=0.50000000000000000, b=0.0000000000000000, c=4.0000000000000000.
testcase_hyperg<double> data037[] = {
  { 1.0000000000000000, 0.50000000000000000, 0.0000000000000000,
          4.0000000000000000, -0.90000000000000002 },
  { 1.0000000000000000, 0.50000000000000000, 0.0000000000000000,
          4.0000000000000000, -0.80000000000000004 },
  { 1.0000000000000000, 0.50000000000000000, 0.0000000000000000,
          4.0000000000000000, -0.69999999999999996 },
  { 1.0000000000000000, 0.50000000000000000, 0.0000000000000000,
          4.0000000000000000, -0.59999999999999998 },
  { 1.0000000000000000, 0.50000000000000000, 0.0000000000000000,
          4.0000000000000000, -0.50000000000000000 },
  { 1.0000000000000000, 0.50000000000000000, 0.0000000000000000,
          4.0000000000000000, -0.40000000000000002 },
  { 1.0000000000000000, 0.50000000000000000, 0.0000000000000000,
          4.0000000000000000, -0.30000000000000004 },
  { 1.0000000000000000, 0.50000000000000000, 0.0000000000000000,
          4.0000000000000000, -0.19999999999999996 },
  { 1.0000000000000000, 0.50000000000000000, 0.0000000000000000,
          4.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 0.50000000000000000, 0.0000000000000000,
          4.0000000000000000, 0.0000000000000000 },
  { 1.0000000000000000, 0.50000000000000000, 0.0000000000000000,
          4.0000000000000000, 0.10000000000000009 },
  { 1.0000000000000000, 0.50000000000000000, 0.0000000000000000,
          4.0000000000000000, 0.19999999999999996 },
  { 1.0000000000000000, 0.50000000000000000, 0.0000000000000000,
          4.0000000000000000, 0.30000000000000004 },
  { 1.0000000000000000, 0.50000000000000000, 0.0000000000000000,
          4.0000000000000000, 0.39999999999999991 },
  { 1.0000000000000000, 0.50000000000000000, 0.0000000000000000,
          4.0000000000000000, 0.50000000000000000 },
  { 1.0000000000000000, 0.50000000000000000, 0.0000000000000000,
          4.0000000000000000, 0.60000000000000009 },
  { 1.0000000000000000, 0.50000000000000000, 0.0000000000000000,
          4.0000000000000000, 0.69999999999999996 },
  { 1.0000000000000000, 0.50000000000000000, 0.0000000000000000,
          4.0000000000000000, 0.80000000000000004 },
  { 1.0000000000000000, 0.50000000000000000, 0.0000000000000000,
          4.0000000000000000, 0.89999999999999991 },
};

// Test function for a=0.50000000000000000, b=0.0000000000000000, c=4.0000000000000000.
template <typename Tp>
void test037()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data037)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data037[i].a), Tp(data037[i].b),
                   Tp(data037[i].c), Tp(data037[i].x));
      const Tp f0 = data037[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=0.50000000000000000, b=0.0000000000000000, c=6.0000000000000000.
testcase_hyperg<double> data038[] = {
  { 1.0000000000000000, 0.50000000000000000, 0.0000000000000000,
          6.0000000000000000, -0.90000000000000002 },
  { 1.0000000000000000, 0.50000000000000000, 0.0000000000000000,
          6.0000000000000000, -0.80000000000000004 },
  { 1.0000000000000000, 0.50000000000000000, 0.0000000000000000,
          6.0000000000000000, -0.69999999999999996 },
  { 1.0000000000000000, 0.50000000000000000, 0.0000000000000000,
          6.0000000000000000, -0.59999999999999998 },
  { 1.0000000000000000, 0.50000000000000000, 0.0000000000000000,
          6.0000000000000000, -0.50000000000000000 },
  { 1.0000000000000000, 0.50000000000000000, 0.0000000000000000,
          6.0000000000000000, -0.40000000000000002 },
  { 1.0000000000000000, 0.50000000000000000, 0.0000000000000000,
          6.0000000000000000, -0.30000000000000004 },
  { 1.0000000000000000, 0.50000000000000000, 0.0000000000000000,
          6.0000000000000000, -0.19999999999999996 },
  { 1.0000000000000000, 0.50000000000000000, 0.0000000000000000,
          6.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 0.50000000000000000, 0.0000000000000000,
          6.0000000000000000, 0.0000000000000000 },
  { 1.0000000000000000, 0.50000000000000000, 0.0000000000000000,
          6.0000000000000000, 0.10000000000000009 },
  { 1.0000000000000000, 0.50000000000000000, 0.0000000000000000,
          6.0000000000000000, 0.19999999999999996 },
  { 1.0000000000000000, 0.50000000000000000, 0.0000000000000000,
          6.0000000000000000, 0.30000000000000004 },
  { 1.0000000000000000, 0.50000000000000000, 0.0000000000000000,
          6.0000000000000000, 0.39999999999999991 },
  { 1.0000000000000000, 0.50000000000000000, 0.0000000000000000,
          6.0000000000000000, 0.50000000000000000 },
  { 1.0000000000000000, 0.50000000000000000, 0.0000000000000000,
          6.0000000000000000, 0.60000000000000009 },
  { 1.0000000000000000, 0.50000000000000000, 0.0000000000000000,
          6.0000000000000000, 0.69999999999999996 },
  { 1.0000000000000000, 0.50000000000000000, 0.0000000000000000,
          6.0000000000000000, 0.80000000000000004 },
  { 1.0000000000000000, 0.50000000000000000, 0.0000000000000000,
          6.0000000000000000, 0.89999999999999991 },
};

// Test function for a=0.50000000000000000, b=0.0000000000000000, c=6.0000000000000000.
template <typename Tp>
void test038()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data038)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data038[i].a), Tp(data038[i].b),
                   Tp(data038[i].c), Tp(data038[i].x));
      const Tp f0 = data038[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=0.50000000000000000, b=0.0000000000000000, c=8.0000000000000000.
testcase_hyperg<double> data039[] = {
  { 1.0000000000000000, 0.50000000000000000, 0.0000000000000000,
          8.0000000000000000, -0.90000000000000002 },
  { 1.0000000000000000, 0.50000000000000000, 0.0000000000000000,
          8.0000000000000000, -0.80000000000000004 },
  { 1.0000000000000000, 0.50000000000000000, 0.0000000000000000,
          8.0000000000000000, -0.69999999999999996 },
  { 1.0000000000000000, 0.50000000000000000, 0.0000000000000000,
          8.0000000000000000, -0.59999999999999998 },
  { 1.0000000000000000, 0.50000000000000000, 0.0000000000000000,
          8.0000000000000000, -0.50000000000000000 },
  { 1.0000000000000000, 0.50000000000000000, 0.0000000000000000,
          8.0000000000000000, -0.40000000000000002 },
  { 1.0000000000000000, 0.50000000000000000, 0.0000000000000000,
          8.0000000000000000, -0.30000000000000004 },
  { 1.0000000000000000, 0.50000000000000000, 0.0000000000000000,
          8.0000000000000000, -0.19999999999999996 },
  { 1.0000000000000000, 0.50000000000000000, 0.0000000000000000,
          8.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 0.50000000000000000, 0.0000000000000000,
          8.0000000000000000, 0.0000000000000000 },
  { 1.0000000000000000, 0.50000000000000000, 0.0000000000000000,
          8.0000000000000000, 0.10000000000000009 },
  { 1.0000000000000000, 0.50000000000000000, 0.0000000000000000,
          8.0000000000000000, 0.19999999999999996 },
  { 1.0000000000000000, 0.50000000000000000, 0.0000000000000000,
          8.0000000000000000, 0.30000000000000004 },
  { 1.0000000000000000, 0.50000000000000000, 0.0000000000000000,
          8.0000000000000000, 0.39999999999999991 },
  { 1.0000000000000000, 0.50000000000000000, 0.0000000000000000,
          8.0000000000000000, 0.50000000000000000 },
  { 1.0000000000000000, 0.50000000000000000, 0.0000000000000000,
          8.0000000000000000, 0.60000000000000009 },
  { 1.0000000000000000, 0.50000000000000000, 0.0000000000000000,
          8.0000000000000000, 0.69999999999999996 },
  { 1.0000000000000000, 0.50000000000000000, 0.0000000000000000,
          8.0000000000000000, 0.80000000000000004 },
  { 1.0000000000000000, 0.50000000000000000, 0.0000000000000000,
          8.0000000000000000, 0.89999999999999991 },
};

// Test function for a=0.50000000000000000, b=0.0000000000000000, c=8.0000000000000000.
template <typename Tp>
void test039()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data039)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data039[i].a), Tp(data039[i].b),
                   Tp(data039[i].c), Tp(data039[i].x));
      const Tp f0 = data039[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=0.50000000000000000, b=0.0000000000000000, c=10.000000000000000.
testcase_hyperg<double> data040[] = {
  { 1.0000000000000000, 0.50000000000000000, 0.0000000000000000,
          10.000000000000000, -0.90000000000000002 },
  { 1.0000000000000000, 0.50000000000000000, 0.0000000000000000,
          10.000000000000000, -0.80000000000000004 },
  { 1.0000000000000000, 0.50000000000000000, 0.0000000000000000,
          10.000000000000000, -0.69999999999999996 },
  { 1.0000000000000000, 0.50000000000000000, 0.0000000000000000,
          10.000000000000000, -0.59999999999999998 },
  { 1.0000000000000000, 0.50000000000000000, 0.0000000000000000,
          10.000000000000000, -0.50000000000000000 },
  { 1.0000000000000000, 0.50000000000000000, 0.0000000000000000,
          10.000000000000000, -0.40000000000000002 },
  { 1.0000000000000000, 0.50000000000000000, 0.0000000000000000,
          10.000000000000000, -0.30000000000000004 },
  { 1.0000000000000000, 0.50000000000000000, 0.0000000000000000,
          10.000000000000000, -0.19999999999999996 },
  { 1.0000000000000000, 0.50000000000000000, 0.0000000000000000,
          10.000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 0.50000000000000000, 0.0000000000000000,
          10.000000000000000, 0.0000000000000000 },
  { 1.0000000000000000, 0.50000000000000000, 0.0000000000000000,
          10.000000000000000, 0.10000000000000009 },
  { 1.0000000000000000, 0.50000000000000000, 0.0000000000000000,
          10.000000000000000, 0.19999999999999996 },
  { 1.0000000000000000, 0.50000000000000000, 0.0000000000000000,
          10.000000000000000, 0.30000000000000004 },
  { 1.0000000000000000, 0.50000000000000000, 0.0000000000000000,
          10.000000000000000, 0.39999999999999991 },
  { 1.0000000000000000, 0.50000000000000000, 0.0000000000000000,
          10.000000000000000, 0.50000000000000000 },
  { 1.0000000000000000, 0.50000000000000000, 0.0000000000000000,
          10.000000000000000, 0.60000000000000009 },
  { 1.0000000000000000, 0.50000000000000000, 0.0000000000000000,
          10.000000000000000, 0.69999999999999996 },
  { 1.0000000000000000, 0.50000000000000000, 0.0000000000000000,
          10.000000000000000, 0.80000000000000004 },
  { 1.0000000000000000, 0.50000000000000000, 0.0000000000000000,
          10.000000000000000, 0.89999999999999991 },
};

// Test function for a=0.50000000000000000, b=0.0000000000000000, c=10.000000000000000.
template <typename Tp>
void test040()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data040)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data040[i].a), Tp(data040[i].b),
                   Tp(data040[i].c), Tp(data040[i].x));
      const Tp f0 = data040[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=0.50000000000000000, b=0.50000000000000000, c=2.0000000000000000.
testcase_hyperg<double> data041[] = {
  { 0.91383715388743758, 0.50000000000000000, 0.50000000000000000,
          2.0000000000000000, -0.90000000000000002 },
  { 0.92151232618202283, 0.50000000000000000, 0.50000000000000000,
          2.0000000000000000, -0.80000000000000004 },
  { 0.92955086110354823, 0.50000000000000000, 0.50000000000000000,
          2.0000000000000000, -0.69999999999999996 },
  { 0.93798900119104844, 0.50000000000000000, 0.50000000000000000,
          2.0000000000000000, -0.59999999999999998 },
  { 0.94686887307107304, 0.50000000000000000, 0.50000000000000000,
          2.0000000000000000, -0.50000000000000000 },
  { 0.95623987262143295, 0.50000000000000000, 0.50000000000000000,
          2.0000000000000000, -0.40000000000000002 },
  { 0.96616049387450154, 0.50000000000000000, 0.50000000000000000,
          2.0000000000000000, -0.30000000000000004 },
  { 0.97670078782187519, 0.50000000000000000, 0.50000000000000000,
          2.0000000000000000, -0.19999999999999996 },
  { 0.98794573712298384, 0.50000000000000000, 0.50000000000000000,
          2.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 0.50000000000000000, 0.50000000000000000,
          2.0000000000000000, 0.0000000000000000 },
  { 1.0129947682256604, 0.50000000000000000, 0.50000000000000000,
          2.0000000000000000, 0.10000000000000009 },
  { 1.0270980168168973, 0.50000000000000000, 0.50000000000000000,
          2.0000000000000000, 0.19999999999999996 },
  { 1.0425304520063581, 0.50000000000000000, 0.50000000000000000,
          2.0000000000000000, 0.30000000000000004 },
  { 1.0595915916161471, 0.50000000000000000, 0.50000000000000000,
          2.0000000000000000, 0.39999999999999991 },
  { 1.0787052023767585, 0.50000000000000000, 0.50000000000000000,
          2.0000000000000000, 0.50000000000000000 },
  { 1.1005053642285867, 0.50000000000000000, 0.50000000000000000,
          2.0000000000000000, 0.60000000000000009 },
  { 1.1260196351148746, 0.50000000000000000, 0.50000000000000000,
          2.0000000000000000, 0.69999999999999996 },
  { 1.1571341977338991, 0.50000000000000000, 0.50000000000000000,
          2.0000000000000000, 0.80000000000000004 },
  { 1.1982111053717450, 0.50000000000000000, 0.50000000000000000,
          2.0000000000000000, 0.89999999999999991 },
};

// Test function for a=0.50000000000000000, b=0.50000000000000000, c=2.0000000000000000.
template <typename Tp>
void test041()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data041)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data041[i].a), Tp(data041[i].b),
                   Tp(data041[i].c), Tp(data041[i].x));
      const Tp f0 = data041[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=0.50000000000000000, b=0.50000000000000000, c=4.0000000000000000.
testcase_hyperg<double> data042[] = {
  { 0.95255425675562733, 0.50000000000000000, 0.50000000000000000,
          4.0000000000000000, -0.90000000000000002 },
  { 0.95712841850078245, 0.50000000000000000, 0.50000000000000000,
          4.0000000000000000, -0.80000000000000004 },
  { 0.96184734120034532, 0.50000000000000000, 0.50000000000000000,
          4.0000000000000000, -0.69999999999999996 },
  { 0.96672141255196242, 0.50000000000000000, 0.50000000000000000,
          4.0000000000000000, -0.59999999999999998 },
  { 0.97176228710138646, 0.50000000000000000, 0.50000000000000000,
          4.0000000000000000, -0.50000000000000000 },
  { 0.97698311668286286, 0.50000000000000000, 0.50000000000000000,
          4.0000000000000000, -0.40000000000000002 },
  { 0.98239883902556069, 0.50000000000000000, 0.50000000000000000,
          4.0000000000000000, -0.30000000000000004 },
  { 0.98802654401961032, 0.50000000000000000, 0.50000000000000000,
          4.0000000000000000, -0.19999999999999996 },
  { 0.99388594556732701, 0.50000000000000000, 0.50000000000000000,
          4.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 0.50000000000000000, 0.50000000000000000,
          4.0000000000000000, 0.0000000000000000 },
  { 1.0063957328951061, 0.50000000000000000, 0.50000000000000000,
          4.0000000000000000, 0.10000000000000009 },
  { 1.0131053706824598, 0.50000000000000000, 0.50000000000000000,
          4.0000000000000000, 0.19999999999999996 },
  { 1.0201679332118803, 0.50000000000000000, 0.50000000000000000,
          4.0000000000000000, 0.30000000000000004 },
  { 1.0276315524377497, 0.50000000000000000, 0.50000000000000000,
          4.0000000000000000, 0.39999999999999991 },
  { 1.0355569942816882, 0.50000000000000000, 0.50000000000000000,
          4.0000000000000000, 0.50000000000000000 },
  { 1.0440233080381554, 0.50000000000000000, 0.50000000000000000,
          4.0000000000000000, 0.60000000000000009 },
  { 1.0531375808028993, 0.50000000000000000, 0.50000000000000000,
          4.0000000000000000, 0.69999999999999996 },
  { 1.0630536689840200, 0.50000000000000000, 0.50000000000000000,
          4.0000000000000000, 0.80000000000000004 },
  { 1.0740149570414563, 0.50000000000000000, 0.50000000000000000,
          4.0000000000000000, 0.89999999999999991 },
};

// Test function for a=0.50000000000000000, b=0.50000000000000000, c=4.0000000000000000.
template <typename Tp>
void test042()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data042)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data042[i].a), Tp(data042[i].b),
                   Tp(data042[i].c), Tp(data042[i].x));
      const Tp f0 = data042[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=0.50000000000000000, b=0.50000000000000000, c=6.0000000000000000.
testcase_hyperg<double> data043[] = {
  { 0.96694084713323891, 0.50000000000000000, 0.50000000000000000,
          6.0000000000000000, -0.90000000000000002 },
  { 0.97024454918852587, 0.50000000000000000, 0.50000000000000000,
          6.0000000000000000, -0.80000000000000004 },
  { 0.97362815600391461, 0.50000000000000000, 0.50000000000000000,
          6.0000000000000000, -0.69999999999999996 },
  { 0.97709622064205115, 0.50000000000000000, 0.50000000000000000,
          6.0000000000000000, -0.59999999999999998 },
  { 0.98065374770570624, 0.50000000000000000, 0.50000000000000000,
          6.0000000000000000, -0.50000000000000000 },
  { 0.98430626119885489, 0.50000000000000000, 0.50000000000000000,
          6.0000000000000000, -0.40000000000000002 },
  { 0.98805988669621037, 0.50000000000000000, 0.50000000000000000,
          6.0000000000000000, -0.30000000000000004 },
  { 0.99192145185739655, 0.50000000000000000, 0.50000000000000000,
          6.0000000000000000, -0.19999999999999996 },
  { 0.99589861079880937, 0.50000000000000000, 0.50000000000000000,
          6.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 0.50000000000000000, 0.50000000000000000,
          6.0000000000000000, 0.0000000000000000 },
  { 1.0042354366729904, 0.50000000000000000, 0.50000000000000000,
          6.0000000000000000, 0.10000000000000009 },
  { 1.0086161755545404, 0.50000000000000000, 0.50000000000000000,
          6.0000000000000000, 0.19999999999999996 },
  { 1.0131552481403503, 0.50000000000000000, 0.50000000000000000,
          6.0000000000000000, 0.30000000000000004 },
  { 1.0178679218284707, 0.50000000000000000, 0.50000000000000000,
          6.0000000000000000, 0.39999999999999991 },
  { 1.0227723400312978, 0.50000000000000000, 0.50000000000000000,
          6.0000000000000000, 0.50000000000000000 },
  { 1.0278904483717863, 0.50000000000000000, 0.50000000000000000,
          6.0000000000000000, 0.60000000000000009 },
  { 1.0332494012993472, 0.50000000000000000, 0.50000000000000000,
          6.0000000000000000, 0.69999999999999996 },
  { 1.0388838453357794, 0.50000000000000000, 0.50000000000000000,
          6.0000000000000000, 0.80000000000000004 },
  { 1.0448400142331342, 0.50000000000000000, 0.50000000000000000,
          6.0000000000000000, 0.89999999999999991 },
};

// Test function for a=0.50000000000000000, b=0.50000000000000000, c=6.0000000000000000.
template <typename Tp>
void test043()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data043)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data043[i].a), Tp(data043[i].b),
                   Tp(data043[i].c), Tp(data043[i].x));
      const Tp f0 = data043[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=0.50000000000000000, b=0.50000000000000000, c=8.0000000000000000.
testcase_hyperg<double> data044[] = {
  { 0.97456073259047449, 0.50000000000000000, 0.50000000000000000,
          8.0000000000000000, -0.90000000000000002 },
  { 0.97715689327833344, 0.50000000000000000, 0.50000000000000000,
          8.0000000000000000, -0.80000000000000004 },
  { 0.97980416868943099, 0.50000000000000000, 0.50000000000000000,
          8.0000000000000000, -0.69999999999999996 },
  { 0.98250498942832509, 0.50000000000000000, 0.50000000000000000,
          8.0000000000000000, -0.59999999999999998 },
  { 0.98526199049760832, 0.50000000000000000, 0.50000000000000000,
          8.0000000000000000, -0.50000000000000000 },
  { 0.98807803762902813, 0.50000000000000000, 0.50000000000000000,
          8.0000000000000000, -0.40000000000000002 },
  { 0.99095625840920332, 0.50000000000000000, 0.50000000000000000,
          8.0000000000000000, -0.30000000000000004 },
  { 0.99390007937387959, 0.50000000000000000, 0.50000000000000000,
          8.0000000000000000, -0.19999999999999996 },
  { 0.99691327061866730, 0.50000000000000000, 0.50000000000000000,
          8.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 0.50000000000000000, 0.50000000000000000,
          8.0000000000000000, 0.0000000000000000 },
  { 1.0031648997547440, 0.50000000000000000, 0.50000000000000000,
          8.0000000000000000, 0.10000000000000009 },
  { 1.0064131494767281, 0.50000000000000000, 0.50000000000000000,
          8.0000000000000000, 0.19999999999999996 },
  { 1.0097505810668461, 0.50000000000000000, 0.50000000000000000,
          8.0000000000000000, 0.30000000000000004 },
  { 1.0131838138968663, 0.50000000000000000, 0.50000000000000000,
          8.0000000000000000, 0.39999999999999991 },
  { 1.0167204326938339, 0.50000000000000000, 0.50000000000000000,
          8.0000000000000000, 0.50000000000000000 },
  { 1.0203692279382193, 0.50000000000000000, 0.50000000000000000,
          8.0000000000000000, 0.60000000000000009 },
  { 1.0241405318057402, 0.50000000000000000, 0.50000000000000000,
          8.0000000000000000, 0.69999999999999996 },
  { 1.0280467087844301, 0.50000000000000000, 0.50000000000000000,
          8.0000000000000000, 0.80000000000000004 },
  { 1.0321029179180026, 0.50000000000000000, 0.50000000000000000,
          8.0000000000000000, 0.89999999999999991 },
};

// Test function for a=0.50000000000000000, b=0.50000000000000000, c=8.0000000000000000.
template <typename Tp>
void test044()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data044)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data044[i].a), Tp(data044[i].b),
                   Tp(data044[i].c), Tp(data044[i].x));
      const Tp f0 = data044[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=0.50000000000000000, b=0.50000000000000000, c=10.000000000000000.
testcase_hyperg<double> data045[] = {
  { 0.97930223035212161, 0.50000000000000000, 0.50000000000000000,
          10.000000000000000, -0.90000000000000002 },
  { 0.98144406855076416, 0.50000000000000000, 0.50000000000000000,
          10.000000000000000, -0.80000000000000004 },
  { 0.98362155940297302, 0.50000000000000000, 0.50000000000000000,
          10.000000000000000, -0.69999999999999996 },
  { 0.98583616201745805, 0.50000000000000000, 0.50000000000000000,
          10.000000000000000, -0.59999999999999998 },
  { 0.98808944235385077, 0.50000000000000000, 0.50000000000000000,
          10.000000000000000, -0.50000000000000000 },
  { 0.99038308530635433, 0.50000000000000000, 0.50000000000000000,
          10.000000000000000, -0.40000000000000002 },
  { 0.99271890872975732, 0.50000000000000000, 0.50000000000000000,
          10.000000000000000, -0.30000000000000004 },
  { 0.99509887982916734, 0.50000000000000000, 0.50000000000000000,
          10.000000000000000, -0.19999999999999996 },
  { 0.99752513445413604, 0.50000000000000000, 0.50000000000000000,
          10.000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 0.50000000000000000, 0.50000000000000000,
          10.000000000000000, 0.0000000000000000 },
  { 1.0025260228440118, 0.50000000000000000, 0.50000000000000000,
          10.000000000000000, 0.10000000000000009 },
  { 1.0051060015613384, 0.50000000000000000, 0.50000000000000000,
          10.000000000000000, 0.19999999999999996 },
  { 1.0077430276253163, 0.50000000000000000, 0.50000000000000000,
          10.000000000000000, 0.30000000000000004 },
  { 1.0104405359789990, 0.50000000000000000, 0.50000000000000000,
          10.000000000000000, 0.39999999999999991 },
  { 1.0132023689128868, 0.50000000000000000, 0.50000000000000000,
          10.000000000000000, 0.50000000000000000 },
  { 1.0160328583559475, 0.50000000000000000, 0.50000000000000000,
          10.000000000000000, 0.60000000000000009 },
  { 1.0189369344885053, 0.50000000000000000, 0.50000000000000000,
          10.000000000000000, 0.69999999999999996 },
  { 1.0219202735809589, 0.50000000000000000, 0.50000000000000000,
          10.000000000000000, 0.80000000000000004 },
  { 1.0249895076611382, 0.50000000000000000, 0.50000000000000000,
          10.000000000000000, 0.89999999999999991 },
};

// Test function for a=0.50000000000000000, b=0.50000000000000000, c=10.000000000000000.
template <typename Tp>
void test045()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data045)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data045[i].a), Tp(data045[i].b),
                   Tp(data045[i].c), Tp(data045[i].x));
      const Tp f0 = data045[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=0.50000000000000000, b=1.0000000000000000, c=2.0000000000000000.
testcase_hyperg<double> data046[] = {
  { 0.84089972268671609, 0.50000000000000000, 1.0000000000000000,
          2.0000000000000000, -0.90000000000000002 },
  { 0.85410196624968437, 0.50000000000000000, 1.0000000000000000,
          2.0000000000000000, -0.80000000000000004 },
  { 0.86811566011579933, 0.50000000000000000, 1.0000000000000000,
          2.0000000000000000, -0.69999999999999996 },
  { 0.88303688022450544, 0.50000000000000000, 1.0000000000000000,
          2.0000000000000000, -0.59999999999999998 },
  { 0.89897948556635565, 0.50000000000000000, 1.0000000000000000,
          2.0000000000000000, -0.50000000000000000 },
  { 0.91607978309961580, 0.50000000000000000, 1.0000000000000000,
          2.0000000000000000, -0.40000000000000002 },
  { 0.93450283399425305, 0.50000000000000000, 1.0000000000000000,
          2.0000000000000000, -0.30000000000000004 },
  { 0.95445115010332193, 0.50000000000000000, 1.0000000000000000,
          2.0000000000000000, -0.19999999999999996 },
  { 0.97617696340303095, 0.50000000000000000, 1.0000000000000000,
          2.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 0.50000000000000000, 1.0000000000000000,
          2.0000000000000000, 0.0000000000000000 },
  { 1.0263340389897240, 0.50000000000000000, 1.0000000000000000,
          2.0000000000000000, 0.10000000000000009 },
  { 1.0557280900008410, 0.50000000000000000, 1.0000000000000000,
          2.0000000000000000, 0.19999999999999996 },
  { 1.0889331564394962, 0.50000000000000000, 1.0000000000000000,
          2.0000000000000000, 0.30000000000000004 },
  { 1.1270166537925830, 0.50000000000000000, 1.0000000000000000,
          2.0000000000000000, 0.39999999999999991 },
  { 1.1715728752538095, 0.50000000000000000, 1.0000000000000000,
          2.0000000000000000, 0.50000000000000000 },
  { 1.2251482265544145, 0.50000000000000000, 1.0000000000000000,
          2.0000000000000000, 0.60000000000000009 },
  { 1.2922212642709541, 0.50000000000000000, 1.0000000000000000,
          2.0000000000000000, 0.69999999999999996 },
  { 1.3819660112501042, 0.50000000000000000, 1.0000000000000000,
          2.0000000000000000, 0.80000000000000004 },
  { 1.5194938532959119, 0.50000000000000000, 1.0000000000000000,
          2.0000000000000000, 0.89999999999999991 },
};

// Test function for a=0.50000000000000000, b=1.0000000000000000, c=2.0000000000000000.
template <typename Tp>
void test046()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data046)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data046[i].a), Tp(data046[i].b),
                   Tp(data046[i].c), Tp(data046[i].x));
      const Tp f0 = data046[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=0.50000000000000000, b=1.0000000000000000, c=4.0000000000000000.
testcase_hyperg<double> data047[] = {
  { 0.90992197313391499, 0.50000000000000000, 1.0000000000000000,
          4.0000000000000000, -0.90000000000000002 },
  { 0.91822592662244507, 0.50000000000000000, 1.0000000000000000,
          4.0000000000000000, -0.80000000000000004 },
  { 0.92687104566419531, 0.50000000000000000, 1.0000000000000000,
          4.0000000000000000, -0.69999999999999996 },
  { 0.93588628166548815, 0.50000000000000000, 1.0000000000000000,
          4.0000000000000000, -0.59999999999999998 },
  { 0.94530459215552909, 0.50000000000000000, 1.0000000000000000,
          4.0000000000000000, -0.50000000000000000 },
  { 0.95516374875247467, 0.50000000000000000, 1.0000000000000000,
          4.0000000000000000, -0.40000000000000002 },
  { 0.96550736800511827, 0.50000000000000000, 1.0000000000000000,
          4.0000000000000000, -0.30000000000000004 },
  { 0.97638624595136270, 0.50000000000000000, 1.0000000000000000,
          4.0000000000000000, -0.19999999999999996 },
  { 0.98786011482678993, 0.50000000000000000, 1.0000000000000000,
          4.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 0.50000000000000000, 1.0000000000000000,
          4.0000000000000000, 0.0000000000000000 },
  { 1.0128914530682316, 0.50000000000000000, 1.0000000000000000,
          4.0000000000000000, 0.10000000000000009 },
  { 1.0266391040215350, 0.50000000000000000, 1.0000000000000000,
          4.0000000000000000, 0.19999999999999996 },
  { 1.0413732738729464, 0.50000000000000000, 1.0000000000000000,
          4.0000000000000000, 0.30000000000000004 },
  { 1.0572599536532992, 0.50000000000000000, 1.0000000000000000,
          4.0000000000000000, 0.39999999999999991 },
  { 1.0745166004060953, 0.50000000000000000, 1.0000000000000000,
          4.0000000000000000, 0.50000000000000000 },
  { 1.0934387388831386, 0.50000000000000000, 1.0000000000000000,
          4.0000000000000000, 0.60000000000000009 },
  { 1.1144486980714641, 0.50000000000000000, 1.0000000000000000,
          4.0000000000000000, 0.69999999999999996 },
  { 1.1381966011250106, 0.50000000000000000, 1.0000000000000000,
          4.0000000000000000, 0.80000000000000004 },
  { 1.1658171625342397, 0.50000000000000000, 1.0000000000000000,
          4.0000000000000000, 0.89999999999999991 },
};

// Test function for a=0.50000000000000000, b=1.0000000000000000, c=4.0000000000000000.
template <typename Tp>
void test047()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data047)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data047[i].a), Tp(data047[i].b),
                   Tp(data047[i].c), Tp(data047[i].x));
      const Tp f0 = data047[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=0.50000000000000000, b=1.0000000000000000, c=6.0000000000000000.
testcase_hyperg<double> data048[] = {
  { 0.93641908369732885, 0.50000000000000000, 1.0000000000000000,
          6.0000000000000000, -0.90000000000000002 },
  { 0.94256349654111315, 0.50000000000000000, 1.0000000000000000,
          6.0000000000000000, -0.80000000000000004 },
  { 0.94890138508461319, 0.50000000000000000, 1.0000000000000000,
          6.0000000000000000, -0.69999999999999996 },
  { 0.95544578858430007, 0.50000000000000000, 1.0000000000000000,
          6.0000000000000000, -0.59999999999999998 },
  { 0.96221121193620718, 0.50000000000000000, 1.0000000000000000,
          6.0000000000000000, -0.50000000000000000 },
  { 0.96921386948293542, 0.50000000000000000, 1.0000000000000000,
          6.0000000000000000, -0.40000000000000002 },
  { 0.97647198488394704, 0.50000000000000000, 1.0000000000000000,
          6.0000000000000000, -0.30000000000000004 },
  { 0.98400616412578656, 0.50000000000000000, 1.0000000000000000,
          6.0000000000000000, -0.19999999999999996 },
  { 0.99183986544963032, 0.50000000000000000, 1.0000000000000000,
          6.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 0.50000000000000000, 1.0000000000000000,
          6.0000000000000000, 0.0000000000000000 },
  { 1.0085177124149158, 0.50000000000000000, 1.0000000000000000,
          6.0000000000000000, 0.10000000000000009 },
  { 1.0174294150407122, 0.50000000000000000, 1.0000000000000000,
          6.0000000000000000, 0.19999999999999996 },
  { 1.0267781897388850, 0.50000000000000000, 1.0000000000000000,
          6.0000000000000000, 0.30000000000000004 },
  { 1.0366157405967285, 0.50000000000000000, 1.0000000000000000,
          6.0000000000000000, 0.39999999999999991 },
  { 1.0470052068648839, 0.50000000000000000, 1.0000000000000000,
          6.0000000000000000, 0.50000000000000000 },
  { 1.0580253905513313, 0.50000000000000000, 1.0000000000000000,
          6.0000000000000000, 0.60000000000000009 },
  { 1.0697774741209765, 0.50000000000000000, 1.0000000000000000,
          6.0000000000000000, 0.69999999999999996 },
  { 1.0823965556448414, 0.50000000000000000, 1.0000000000000000,
          6.0000000000000000, 0.80000000000000004 },
  { 1.0960739512057103, 0.50000000000000000, 1.0000000000000000,
          6.0000000000000000, 0.89999999999999991 },
};

// Test function for a=0.50000000000000000, b=1.0000000000000000, c=6.0000000000000000.
template <typename Tp>
void test048()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data048)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data048[i].a), Tp(data048[i].b),
                   Tp(data048[i].c), Tp(data048[i].x));
      const Tp f0 = data048[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=0.50000000000000000, b=1.0000000000000000, c=8.0000000000000000.
testcase_hyperg<double> data049[] = {
  { 0.95069883346936235, 0.50000000000000000, 1.0000000000000000,
          8.0000000000000000, -0.90000000000000002 },
  { 0.95559618047704165, 0.50000000000000000, 1.0000000000000000,
          8.0000000000000000, -0.80000000000000004 },
  { 0.96061938755931653, 0.50000000000000000, 1.0000000000000000,
          8.0000000000000000, -0.69999999999999996 },
  { 0.96577553912851344, 0.50000000000000000, 1.0000000000000000,
          8.0000000000000000, -0.59999999999999998 },
  { 0.97107239473807716, 0.50000000000000000, 1.0000000000000000,
          8.0000000000000000, -0.50000000000000000 },
  { 0.97651848528588503, 0.50000000000000000, 1.0000000000000000,
          8.0000000000000000, -0.40000000000000002 },
  { 0.98212322830227139, 0.50000000000000000, 1.0000000000000000,
          8.0000000000000000, -0.30000000000000004 },
  { 0.98789706736195781, 0.50000000000000000, 1.0000000000000000,
          8.0000000000000000, -0.19999999999999996 },
  { 0.99385164237825074, 0.50000000000000000, 1.0000000000000000,
          8.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 0.50000000000000000, 1.0000000000000000,
          8.0000000000000000, 0.0000000000000000 },
  { 1.0063568569383123, 0.50000000000000000, 1.0000000000000000,
          8.0000000000000000, 0.10000000000000009 },
  { 1.0129389344715818, 0.50000000000000000, 1.0000000000000000,
          8.0000000000000000, 0.19999999999999996 },
  { 1.0197653907773940, 0.50000000000000000, 1.0000000000000000,
          8.0000000000000000, 0.30000000000000004 },
  { 1.0268583912277143, 0.50000000000000000, 1.0000000000000000,
          8.0000000000000000, 0.39999999999999991 },
  { 1.0342438793937092, 0.50000000000000000, 1.0000000000000000,
          8.0000000000000000, 0.50000000000000000 },
  { 1.0419526514766855, 0.50000000000000000, 1.0000000000000000,
          8.0000000000000000, 0.60000000000000009 },
  { 1.0500219124976327, 0.50000000000000000, 1.0000000000000000,
          8.0000000000000000, 0.69999999999999996 },
  { 1.0584976491907043, 0.50000000000000000, 1.0000000000000000,
          8.0000000000000000, 0.80000000000000004 },
  { 1.0674385240268101, 0.50000000000000000, 1.0000000000000000,
          8.0000000000000000, 0.89999999999999991 },
};

// Test function for a=0.50000000000000000, b=1.0000000000000000, c=8.0000000000000000.
template <typename Tp>
void test049()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data049)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data049[i].a), Tp(data049[i].b),
                   Tp(data049[i].c), Tp(data049[i].x));
      const Tp f0 = data049[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=0.50000000000000000, b=1.0000000000000000, c=10.000000000000000.
testcase_hyperg<double> data050[] = {
  { 0.95968319138913893, 0.50000000000000000, 1.0000000000000000,
          10.000000000000000, -0.90000000000000002 },
  { 0.96376169072755768, 0.50000000000000000, 1.0000000000000000,
          10.000000000000000, -0.80000000000000004 },
  { 0.96792900082729316, 0.50000000000000000, 1.0000000000000000,
          10.000000000000000, -0.69999999999999996 },
  { 0.97218942798115737, 0.50000000000000000, 1.0000000000000000,
          10.000000000000000, -0.59999999999999998 },
  { 0.97654763592586857, 0.50000000000000000, 1.0000000000000000,
          10.000000000000000, -0.50000000000000000 },
  { 0.98100869054353856, 0.50000000000000000, 1.0000000000000000,
          10.000000000000000, -0.40000000000000002 },
  { 0.98557811238699278, 0.50000000000000000, 1.0000000000000000,
          10.000000000000000, -0.30000000000000004 },
  { 0.99026193885795544, 0.50000000000000000, 1.0000000000000000,
          10.000000000000000, -0.19999999999999996 },
  { 0.99506679842072221, 0.50000000000000000, 1.0000000000000000,
          10.000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 0.50000000000000000, 1.0000000000000000,
          10.000000000000000, 0.0000000000000000 },
  { 1.0050696417919618, 0.50000000000000000, 1.0000000000000000,
          10.000000000000000, 0.10000000000000009 },
  { 1.0102847452747090, 0.50000000000000000, 1.0000000000000000,
          10.000000000000000, 0.19999999999999996 },
  { 1.0156554225057022, 0.50000000000000000, 1.0000000000000000,
          10.000000000000000, 0.30000000000000004 },
  { 1.0211930882963096, 0.50000000000000000, 1.0000000000000000,
          10.000000000000000, 0.39999999999999991 },
  { 1.0269107343740711, 0.50000000000000000, 1.0000000000000000,
          10.000000000000000, 0.50000000000000000 },
  { 1.0328232917216298, 0.50000000000000000, 1.0000000000000000,
          10.000000000000000, 0.60000000000000009 },
  { 1.0389481230247195, 0.50000000000000000, 1.0000000000000000,
          10.000000000000000, 0.69999999999999996 },
  { 1.0453057164134614, 0.50000000000000000, 1.0000000000000000,
          10.000000000000000, 0.80000000000000004 },
  { 1.0519207114461246, 0.50000000000000000, 1.0000000000000000,
          10.000000000000000, 0.89999999999999991 },
};

// Test function for a=0.50000000000000000, b=1.0000000000000000, c=10.000000000000000.
template <typename Tp>
void test050()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data050)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data050[i].a), Tp(data050[i].b),
                   Tp(data050[i].c), Tp(data050[i].x));
      const Tp f0 = data050[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=0.50000000000000000, b=2.0000000000000000, c=2.0000000000000000.
testcase_hyperg<double> data051[] = {
  { 0.72547625011001171, 0.50000000000000000, 2.0000000000000000,
          2.0000000000000000, -0.90000000000000002 },
  { 0.74535599249992990, 0.50000000000000000, 2.0000000000000000,
          2.0000000000000000, -0.80000000000000004 },
  { 0.76696498884737041, 0.50000000000000000, 2.0000000000000000,
          2.0000000000000000, -0.69999999999999996 },
  { 0.79056941504209477, 0.50000000000000000, 2.0000000000000000,
          2.0000000000000000, -0.59999999999999998 },
  { 0.81649658092772603, 0.50000000000000000, 2.0000000000000000,
          2.0000000000000000, -0.50000000000000000 },
  { 0.84515425472851657, 0.50000000000000000, 2.0000000000000000,
          2.0000000000000000, -0.40000000000000002 },
  { 0.87705801930702920, 0.50000000000000000, 2.0000000000000000,
          2.0000000000000000, -0.30000000000000004 },
  { 0.91287092917527690, 0.50000000000000000, 2.0000000000000000,
          2.0000000000000000, -0.19999999999999996 },
  { 0.95346258924559224, 0.50000000000000000, 2.0000000000000000,
          2.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 0.50000000000000000, 2.0000000000000000,
          2.0000000000000000, 0.0000000000000000 },
  { 1.0540925533894598, 0.50000000000000000, 2.0000000000000000,
          2.0000000000000000, 0.10000000000000009 },
  { 1.1180339887498949, 0.50000000000000000, 2.0000000000000000,
          2.0000000000000000, 0.19999999999999996 },
  { 1.1952286093343938, 0.50000000000000000, 2.0000000000000000,
          2.0000000000000000, 0.30000000000000004 },
  { 1.2909944487358056, 0.50000000000000000, 2.0000000000000000,
          2.0000000000000000, 0.39999999999999991 },
  { 1.4142135623730949, 0.50000000000000000, 2.0000000000000000,
          2.0000000000000000, 0.50000000000000000 },
  { 1.5811388300841900, 0.50000000000000000, 2.0000000000000000,
          2.0000000000000000, 0.60000000000000009 },
  { 1.8257418583505536, 0.50000000000000000, 2.0000000000000000,
          2.0000000000000000, 0.69999999999999996 },
  { 2.2360679774997898, 0.50000000000000000, 2.0000000000000000,
          2.0000000000000000, 0.80000000000000004 },
  { 3.1622776601683782, 0.50000000000000000, 2.0000000000000000,
          2.0000000000000000, 0.89999999999999991 },
};

// Test function for a=0.50000000000000000, b=2.0000000000000000, c=2.0000000000000000.
template <typename Tp>
void test051()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data051)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data051[i].a), Tp(data051[i].b),
                   Tp(data051[i].c), Tp(data051[i].x));
      const Tp f0 = data051[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=0.50000000000000000, b=2.0000000000000000, c=4.0000000000000000.
testcase_hyperg<double> data052[] = {
  { 0.83664260086443798, 0.50000000000000000, 2.0000000000000000,
          4.0000000000000000, -0.90000000000000002 },
  { 0.85046584300227146, 0.50000000000000000, 2.0000000000000000,
          4.0000000000000000, -0.80000000000000004 },
  { 0.86509574979651593, 0.50000000000000000, 2.0000000000000000,
          4.0000000000000000, -0.69999999999999996 },
  { 0.88062082573041867, 0.50000000000000000, 2.0000000000000000,
          4.0000000000000000, -0.59999999999999998 },
  { 0.89714464248521586, 0.50000000000000000, 2.0000000000000000,
          4.0000000000000000, -0.50000000000000000 },
  { 0.91478946588967558, 0.50000000000000000, 2.0000000000000000,
          4.0000000000000000, -0.40000000000000002 },
  { 0.93370105322348573, 0.50000000000000000, 2.0000000000000000,
          4.0000000000000000, -0.30000000000000004 },
  { 0.95405511057700887, 0.50000000000000000, 2.0000000000000000,
          4.0000000000000000, -0.19999999999999996 },
  { 0.97606616007978142, 0.50000000000000000, 2.0000000000000000,
          4.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 0.50000000000000000, 2.0000000000000000,
          4.0000000000000000, 0.0000000000000000 },
  { 1.0261916902334731, 0.50000000000000000, 2.0000000000000000,
          4.0000000000000000, 0.10000000000000009 },
  { 1.0550723519434702, 0.50000000000000000, 2.0000000000000000,
          4.0000000000000000, 0.19999999999999996 },
  { 1.0872106588188091, 0.50000000000000000, 2.0000000000000000,
          4.0000000000000000, 0.30000000000000004 },
  { 1.1233801699379020, 0.50000000000000000, 2.0000000000000000,
          4.0000000000000000, 0.39999999999999991 },
  { 1.1646752981725688, 0.50000000000000000, 2.0000000000000000,
          4.0000000000000000, 0.50000000000000000 },
  { 1.2127272514219511, 0.50000000000000000, 2.0000000000000000,
          4.0000000000000000, 0.60000000000000009 },
  { 1.2701518651068637, 0.50000000000000000, 2.0000000000000000,
          4.0000000000000000, 0.69999999999999996 },
  { 1.3416407864998725, 0.50000000000000000, 2.0000000000000000,
          4.0000000000000000, 0.80000000000000004 },
  { 1.4374795179111102, 0.50000000000000000, 2.0000000000000000,
          4.0000000000000000, 0.89999999999999991 },
};

// Test function for a=0.50000000000000000, b=2.0000000000000000, c=4.0000000000000000.
template <typename Tp>
void test052()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data052)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data052[i].a), Tp(data052[i].b),
                   Tp(data052[i].c), Tp(data052[i].x));
      const Tp f0 = data052[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=0.50000000000000000, b=2.0000000000000000, c=6.0000000000000000.
testcase_hyperg<double> data053[] = {
  { 0.88195381730235833, 0.50000000000000000, 2.0000000000000000,
          6.0000000000000000, -0.90000000000000002 },
  { 0.89265078469555093, 0.50000000000000000, 2.0000000000000000,
          6.0000000000000000, -0.80000000000000004 },
  { 0.90382937908303707, 0.50000000000000000, 2.0000000000000000,
          6.0000000000000000, -0.69999999999999996 },
  { 0.91553161389880577, 0.50000000000000000, 2.0000000000000000,
          6.0000000000000000, -0.59999999999999998 },
  { 0.92780530349281576, 0.50000000000000000, 2.0000000000000000,
          6.0000000000000000, -0.50000000000000000 },
  { 0.94070521140346042, 0.50000000000000000, 2.0000000000000000,
          6.0000000000000000, -0.40000000000000002 },
  { 0.95429450630523349, 0.50000000000000000, 2.0000000000000000,
          6.0000000000000000, -0.30000000000000004 },
  { 0.96864663325785849, 0.50000000000000000, 2.0000000000000000,
          6.0000000000000000, -0.19999999999999996 },
  { 0.98384775588541795, 0.50000000000000000, 2.0000000000000000,
          6.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 0.50000000000000000, 2.0000000000000000,
          6.0000000000000000, 0.0000000000000000 },
  { 1.0172258496884334, 0.50000000000000000, 2.0000000000000000,
          6.0000000000000000, 0.10000000000000009 },
  { 1.0356742479163459, 0.50000000000000000, 2.0000000000000000,
          6.0000000000000000, 0.19999999999999996 },
  { 1.0555293036908924, 0.50000000000000000, 2.0000000000000000,
          6.0000000000000000, 0.30000000000000004 },
  { 1.0770231491562379, 0.50000000000000000, 2.0000000000000000,
          6.0000000000000000, 0.39999999999999991 },
  { 1.1004557416484888, 0.50000000000000000, 2.0000000000000000,
          6.0000000000000000, 0.50000000000000000 },
  { 1.1262270515731978, 0.50000000000000000, 2.0000000000000000,
          6.0000000000000000, 0.60000000000000009 },
  { 1.1548932919125086, 0.50000000000000000, 2.0000000000000000,
          6.0000000000000000, 0.69999999999999996 },
  { 1.1872757758134724, 0.50000000000000000, 2.0000000000000000,
          6.0000000000000000, 0.80000000000000004 },
  { 1.2247091713458949, 0.50000000000000000, 2.0000000000000000,
          6.0000000000000000, 0.89999999999999991 },
};

// Test function for a=0.50000000000000000, b=2.0000000000000000, c=6.0000000000000000.
template <typename Tp>
void test053()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data053)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data053[i].a), Tp(data053[i].b),
                   Tp(data053[i].c), Tp(data053[i].x));
      const Tp f0 = data053[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=0.50000000000000000, b=2.0000000000000000, c=8.0000000000000000.
testcase_hyperg<double> data054[] = {
  { 0.90716919697107301, 0.50000000000000000, 2.0000000000000000,
          8.0000000000000000, -0.90000000000000002 },
  { 0.91592299407142519, 0.50000000000000000, 2.0000000000000000,
          8.0000000000000000, -0.80000000000000004 },
  { 0.92500027075874236, 0.50000000000000000, 2.0000000000000000,
          8.0000000000000000, -0.69999999999999996 },
  { 0.93442464185467167, 0.50000000000000000, 2.0000000000000000,
          8.0000000000000000, -0.59999999999999998 },
  { 0.94422248683737009, 0.50000000000000000, 2.0000000000000000,
          8.0000000000000000, -0.50000000000000000 },
  { 0.95442341810133347, 0.50000000000000000, 2.0000000000000000,
          8.0000000000000000, -0.40000000000000002 },
  { 0.96506085725516400, 0.50000000000000000, 2.0000000000000000,
          8.0000000000000000, -0.30000000000000004 },
  { 0.97617275213704069, 0.50000000000000000, 2.0000000000000000,
          8.0000000000000000, -0.19999999999999996 },
  { 0.98780247986309799, 0.50000000000000000, 2.0000000000000000,
          8.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 0.50000000000000000, 2.0000000000000000,
          8.0000000000000000, 0.0000000000000000 },
  { 1.0128233505813447, 0.50000000000000000, 2.0000000000000000,
          8.0000000000000000, 0.10000000000000009 },
  { 1.0263406246541855, 0.50000000000000000, 2.0000000000000000,
          8.0000000000000000, 0.19999999999999996 },
  { 1.0406326381700366, 0.50000000000000000, 2.0000000000000000,
          8.0000000000000000, 0.30000000000000004 },
  { 1.0557966239802845, 0.50000000000000000, 2.0000000000000000,
          8.0000000000000000, 0.39999999999999991 },
  { 1.0719515075786321, 0.50000000000000000, 2.0000000000000000,
          8.0000000000000000, 0.50000000000000000 },
  { 1.0892457392422055, 0.50000000000000000, 2.0000000000000000,
          8.0000000000000000, 0.60000000000000009 },
  { 1.1078695188000958, 0.50000000000000000, 2.0000000000000000,
          8.0000000000000000, 0.69999999999999996 },
  { 1.1280752258974340, 0.50000000000000000, 2.0000000000000000,
          8.0000000000000000, 0.80000000000000004 },
  { 1.1502152002706476, 0.50000000000000000, 2.0000000000000000,
          8.0000000000000000, 0.89999999999999991 },
};

// Test function for a=0.50000000000000000, b=2.0000000000000000, c=8.0000000000000000.
template <typename Tp>
void test054()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data054)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data054[i].a), Tp(data054[i].b),
                   Tp(data054[i].c), Tp(data054[i].x));
      const Tp f0 = data054[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=0.50000000000000000, b=2.0000000000000000, c=10.000000000000000.
testcase_hyperg<double> data055[] = {
  { 0.92336416053263093, 0.50000000000000000, 2.0000000000000000,
          10.000000000000000, -0.90000000000000002 },
  { 0.93078397248364519, 0.50000000000000000, 2.0000000000000000,
          10.000000000000000, -0.80000000000000004 },
  { 0.93843714333600226, 0.50000000000000000, 2.0000000000000000,
          10.000000000000000, -0.69999999999999996 },
  { 0.94633837784068076, 0.50000000000000000, 2.0000000000000000,
          10.000000000000000, -0.59999999999999998 },
  { 0.95450388104967909, 0.50000000000000000, 2.0000000000000000,
          10.000000000000000, -0.50000000000000000 },
  { 0.96295158125742764, 0.50000000000000000, 2.0000000000000000,
          10.000000000000000, -0.40000000000000002 },
  { 0.97170139827854296, 0.50000000000000000, 2.0000000000000000,
          10.000000000000000, -0.30000000000000004 },
  { 0.98077556918512687, 0.50000000000000000, 2.0000000000000000,
          10.000000000000000, -0.19999999999999996 },
  { 0.99019904777750845, 0.50000000000000000, 2.0000000000000000,
          10.000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 0.50000000000000000, 2.0000000000000000,
          10.000000000000000, 0.0000000000000000 },
  { 1.0102104261941198, 0.50000000000000000, 2.0000000000000000,
          10.000000000000000, 0.10000000000000009 },
  { 1.0208669540935695, 0.50000000000000000, 2.0000000000000000,
          10.000000000000000, 0.19999999999999996 },
  { 1.0320118665407505, 0.50000000000000000, 2.0000000000000000,
          10.000000000000000, 0.30000000000000004 },
  { 1.0436944599504387, 0.50000000000000000, 2.0000000000000000,
          10.000000000000000, 0.39999999999999991 },
  { 1.0559728828278145, 0.50000000000000000, 2.0000000000000000,
          10.000000000000000, 0.50000000000000000 },
  { 1.0689166967761712, 0.50000000000000000, 2.0000000000000000,
          10.000000000000000, 0.60000000000000009 },
  { 1.0826105758119842, 0.50000000000000000, 2.0000000000000000,
          10.000000000000000, 0.69999999999999996 },
  { 1.0971599106346146, 0.50000000000000000, 2.0000000000000000,
          10.000000000000000, 0.80000000000000004 },
  { 1.1126998828023964, 0.50000000000000000, 2.0000000000000000,
          10.000000000000000, 0.89999999999999991 },
};

// Test function for a=0.50000000000000000, b=2.0000000000000000, c=10.000000000000000.
template <typename Tp>
void test055()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data055)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data055[i].a), Tp(data055[i].b),
                   Tp(data055[i].c), Tp(data055[i].x));
      const Tp f0 = data055[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=0.50000000000000000, b=5.0000000000000000, c=2.0000000000000000.
testcase_hyperg<double> data056[] = {
  { 0.52275983209457511, 0.50000000000000000, 5.0000000000000000,
          2.0000000000000000, -0.90000000000000002 },
  { 0.54700336898142965, 0.50000000000000000, 5.0000000000000000,
          2.0000000000000000, -0.80000000000000004 },
  { 0.57468955512601971, 0.50000000000000000, 5.0000000000000000,
          2.0000000000000000, -0.69999999999999996 },
  { 0.60665490543315015, 0.50000000000000000, 5.0000000000000000,
          2.0000000000000000, -0.59999999999999998 },
  { 0.64403057859056123, 0.50000000000000000, 5.0000000000000000,
          2.0000000000000000, -0.50000000000000000 },
  { 0.68838183648623719, 0.50000000000000000, 5.0000000000000000,
          2.0000000000000000, -0.40000000000000002 },
  { 0.74193265039311085, 0.50000000000000000, 5.0000000000000000,
          2.0000000000000000, -0.30000000000000004 },
  { 0.80794095908995300, 0.50000000000000000, 5.0000000000000000,
          2.0000000000000000, -0.19999999999999996 },
  { 0.89135275749639320, 0.50000000000000000, 5.0000000000000000,
          2.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 0.50000000000000000, 5.0000000000000000,
          2.0000000000000000, 0.0000000000000000 },
  { 1.1469266219310688, 0.50000000000000000, 5.0000000000000000,
          2.0000000000000000, 0.10000000000000009 },
  { 1.3552340708357489, 0.50000000000000000, 5.0000000000000000,
          2.0000000000000000, 0.19999999999999996 },
  { 1.6690840478838305, 0.50000000000000000, 5.0000000000000000,
          2.0000000000000000, 0.30000000000000004 },
  { 2.1815415453174483, 0.50000000000000000, 5.0000000000000000,
          2.0000000000000000, 0.39999999999999991 },
  { 3.1156892546032235, 0.50000000000000000, 5.0000000000000000,
          2.0000000000000000, 0.50000000000000000 },
  { 5.1109077417760416, 0.50000000000000000, 5.0000000000000000,
          2.0000000000000000, 0.60000000000000009 },
  { 10.560352936466296, 0.50000000000000000, 5.0000000000000000,
          2.0000000000000000, 0.69999999999999996 },
  { 33.541019662496815, 0.50000000000000000, 5.0000000000000000,
          2.0000000000000000, 0.80000000000000004 },
  { 300.66343065819501, 0.50000000000000000, 5.0000000000000000,
          2.0000000000000000, 0.89999999999999991 },
};

// Test function for a=0.50000000000000000, b=5.0000000000000000, c=2.0000000000000000.
template <typename Tp>
void test056()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data056)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data056[i].a), Tp(data056[i].b),
                   Tp(data056[i].c), Tp(data056[i].x));
      const Tp f0 = data056[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=0.50000000000000000, b=5.0000000000000000, c=4.0000000000000000.
testcase_hyperg<double> data057[] = {
  { 0.68252041951139264, 0.50000000000000000, 5.0000000000000000,
          4.0000000000000000, -0.90000000000000002 },
  { 0.70394732624993395, 0.50000000000000000, 5.0000000000000000,
          4.0000000000000000, -0.80000000000000004 },
  { 0.72748884971552041, 0.50000000000000000, 5.0000000000000000,
          4.0000000000000000, -0.69999999999999996 },
  { 0.75351147371199689, 0.50000000000000000, 5.0000000000000000,
          4.0000000000000000, -0.59999999999999998 },
  { 0.78247589005573748, 0.50000000000000000, 5.0000000000000000,
          4.0000000000000000, -0.50000000000000000 },
  { 0.81497017420249818, 0.50000000000000000, 5.0000000000000000,
          4.0000000000000000, -0.40000000000000002 },
  { 0.85175826875009564, 0.50000000000000000, 5.0000000000000000,
          4.0000000000000000, -0.30000000000000004 },
  { 0.89385278481745867, 0.50000000000000000, 5.0000000000000000,
          4.0000000000000000, -0.19999999999999996 },
  { 0.94262778709507411, 0.50000000000000000, 5.0000000000000000,
          4.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 0.50000000000000000, 5.0000000000000000,
          4.0000000000000000, 0.0000000000000000 },
  { 1.0687327277420910, 0.50000000000000000, 5.0000000000000000,
          4.0000000000000000, 0.10000000000000009 },
  { 1.1529725508983291, 0.50000000000000000, 5.0000000000000000,
          4.0000000000000000, 0.19999999999999996 },
  { 1.2592587134058799, 0.50000000000000000, 5.0000000000000000,
          4.0000000000000000, 0.30000000000000004 },
  { 1.3985773194637892, 0.50000000000000000, 5.0000000000000000,
          4.0000000000000000, 0.39999999999999991 },
  { 1.5909902576697317, 0.50000000000000000, 5.0000000000000000,
          4.0000000000000000, 0.50000000000000000 },
  { 1.8776023607249752, 0.50000000000000000, 5.0000000000000000,
          4.0000000000000000, 0.60000000000000009 },
  { 2.3582499003694646, 0.50000000000000000, 5.0000000000000000,
          4.0000000000000000, 0.69999999999999996 },
  { 3.3541019662496838, 0.50000000000000000, 5.0000000000000000,
          4.0000000000000000, 0.80000000000000004 },
  { 6.7198400278577859, 0.50000000000000000, 5.0000000000000000,
          4.0000000000000000, 0.89999999999999991 },
};

// Test function for a=0.50000000000000000, b=5.0000000000000000, c=4.0000000000000000.
template <typename Tp>
void test057()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data057)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data057[i].a), Tp(data057[i].b),
                   Tp(data057[i].c), Tp(data057[i].x));
      const Tp f0 = data057[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=0.50000000000000000, b=5.0000000000000000, c=6.0000000000000000.
testcase_hyperg<double> data058[] = {
  { 0.75755211927082589, 0.50000000000000000, 5.0000000000000000,
          6.0000000000000000, -0.90000000000000002 },
  { 0.77603550233010998, 0.50000000000000000, 5.0000000000000000,
          6.0000000000000000, -0.80000000000000004 },
  { 0.79596241913438492, 0.50000000000000000, 5.0000000000000000,
          6.0000000000000000, -0.69999999999999996 },
  { 0.81753360792105201, 0.50000000000000000, 5.0000000000000000,
          6.0000000000000000, -0.59999999999999998 },
  { 0.84099165409805521, 0.50000000000000000, 5.0000000000000000,
          6.0000000000000000, -0.50000000000000000 },
  { 0.86663303852180873, 0.50000000000000000, 5.0000000000000000,
          6.0000000000000000, -0.40000000000000002 },
  { 0.89482475828629970, 0.50000000000000000, 5.0000000000000000,
          6.0000000000000000, -0.30000000000000004 },
  { 0.92602774279590350, 0.50000000000000000, 5.0000000000000000,
          6.0000000000000000, -0.19999999999999996 },
  { 0.96083064727087386, 0.50000000000000000, 5.0000000000000000,
          6.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 0.50000000000000000, 5.0000000000000000,
          6.0000000000000000, 0.0000000000000000 },
  { 1.0445570841313008, 0.50000000000000000, 5.0000000000000000,
          6.0000000000000000, 0.10000000000000009 },
  { 1.0959004638926031, 0.50000000000000000, 5.0000000000000000,
          6.0000000000000000, 0.19999999999999996 },
  { 1.1560106261370562, 0.50000000000000000, 5.0000000000000000,
          6.0000000000000000, 0.30000000000000004 },
  { 1.2278121770678145, 0.50000000000000000, 5.0000000000000000,
          6.0000000000000000, 0.39999999999999991 },
  { 1.3158640214709998, 0.50000000000000000, 5.0000000000000000,
          6.0000000000000000, 0.50000000000000000 },
  { 1.4278095344155000, 0.50000000000000000, 5.0000000000000000,
          6.0000000000000000, 0.60000000000000009 },
  { 1.5778700502946612, 0.50000000000000000, 5.0000000000000000,
          6.0000000000000000, 0.69999999999999996 },
  { 1.7972173289196469, 0.50000000000000000, 5.0000000000000000,
          6.0000000000000000, 0.80000000000000004 },
  { 2.1789970569269732, 0.50000000000000000, 5.0000000000000000,
          6.0000000000000000, 0.89999999999999991 },
};

// Test function for a=0.50000000000000000, b=5.0000000000000000, c=6.0000000000000000.
template <typename Tp>
void test058()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data058)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data058[i].a), Tp(data058[i].b),
                   Tp(data058[i].c), Tp(data058[i].x));
      const Tp f0 = data058[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=0.50000000000000000, b=5.0000000000000000, c=8.0000000000000000.
testcase_hyperg<double> data059[] = {
  { 0.80270093579329471, 0.50000000000000000, 5.0000000000000000,
          8.0000000000000000, -0.90000000000000002 },
  { 0.81884974572462788, 0.50000000000000000, 5.0000000000000000,
          8.0000000000000000, -0.80000000000000004 },
  { 0.83605266330015271, 0.50000000000000000, 5.0000000000000000,
          8.0000000000000000, -0.69999999999999996 },
  { 0.85443340762795972, 0.50000000000000000, 5.0000000000000000,
          8.0000000000000000, -0.59999999999999998 },
  { 0.87413762182790655, 0.50000000000000000, 5.0000000000000000,
          8.0000000000000000, -0.50000000000000000 },
  { 0.89533826626907331, 0.50000000000000000, 5.0000000000000000,
          8.0000000000000000, -0.40000000000000002 },
  { 0.91824276674115268, 0.50000000000000000, 5.0000000000000000,
          8.0000000000000000, -0.30000000000000004 },
  { 0.94310265050720576, 0.50000000000000000, 5.0000000000000000,
          8.0000000000000000, -0.19999999999999996 },
  { 0.97022678857609712, 0.50000000000000000, 5.0000000000000000,
          8.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 0.50000000000000000, 5.0000000000000000,
          8.0000000000000000, 0.0000000000000000 },
  { 1.0329098673199812, 0.50000000000000000, 5.0000000000000000,
          8.0000000000000000, 0.10000000000000009 },
  { 1.0695865684573389, 0.50000000000000000, 5.0000000000000000,
          8.0000000000000000, 0.19999999999999996 },
  { 1.1108642103944570, 0.50000000000000000, 5.0000000000000000,
          8.0000000000000000, 0.30000000000000004 },
  { 1.1578795055970506, 0.50000000000000000, 5.0000000000000000,
          8.0000000000000000, 0.39999999999999991 },
  { 1.2122394794169442, 0.50000000000000000, 5.0000000000000000,
          8.0000000000000000, 0.50000000000000000 },
  { 1.2763274721556934, 0.50000000000000000, 5.0000000000000000,
          8.0000000000000000, 0.60000000000000009 },
  { 1.3539179650251021, 0.50000000000000000, 5.0000000000000000,
          8.0000000000000000, 0.69999999999999996 },
  { 1.4515986118197148, 0.50000000000000000, 5.0000000000000000,
          8.0000000000000000, 0.80000000000000004 },
  { 1.5829284571614219, 0.50000000000000000, 5.0000000000000000,
          8.0000000000000000, 0.89999999999999991 },
};

// Test function for a=0.50000000000000000, b=5.0000000000000000, c=8.0000000000000000.
template <typename Tp>
void test059()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data059)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data059[i].a), Tp(data059[i].b),
                   Tp(data059[i].c), Tp(data059[i].x));
      const Tp f0 = data059[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=0.50000000000000000, b=5.0000000000000000, c=10.000000000000000.
testcase_hyperg<double> data060[] = {
  { 0.83322694172301959, 0.50000000000000000, 5.0000000000000000,
          10.000000000000000, -0.90000000000000002 },
  { 0.84753931604765664, 0.50000000000000000, 5.0000000000000000,
          10.000000000000000, -0.80000000000000004 },
  { 0.86265784532195022, 0.50000000000000000, 5.0000000000000000,
          10.000000000000000, -0.69999999999999996 },
  { 0.87866479300707079, 0.50000000000000000, 5.0000000000000000,
          10.000000000000000, -0.59999999999999998 },
  { 0.89565516540263501, 0.50000000000000000, 5.0000000000000000,
          10.000000000000000, -0.50000000000000000 },
  { 0.91373946207610557, 0.50000000000000000, 5.0000000000000000,
          10.000000000000000, -0.40000000000000002 },
  { 0.93304721345881891, 0.50000000000000000, 5.0000000000000000,
          10.000000000000000, -0.30000000000000004 },
  { 0.95373159512905148, 0.50000000000000000, 5.0000000000000000,
          10.000000000000000, -0.19999999999999996 },
  { 0.97597554238828121, 0.50000000000000000, 5.0000000000000000,
          10.000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 0.50000000000000000, 5.0000000000000000,
          10.000000000000000, 0.0000000000000000 },
  { 1.0260752851887982, 0.50000000000000000, 5.0000000000000000,
          10.000000000000000, 0.10000000000000009 },
  { 1.0545371197996178, 0.50000000000000000, 5.0000000000000000,
          10.000000000000000, 0.19999999999999996 },
  { 1.0858099017045830, 0.50000000000000000, 5.0000000000000000,
          10.000000000000000, 0.30000000000000004 },
  { 1.1204416568688709, 0.50000000000000000, 5.0000000000000000,
          10.000000000000000, 0.39999999999999991 },
  { 1.1591587835964847, 0.50000000000000000, 5.0000000000000000,
          10.000000000000000, 0.50000000000000000 },
  { 1.2029564720303347, 0.50000000000000000, 5.0000000000000000,
          10.000000000000000, 0.60000000000000009 },
  { 1.2532588722007874, 0.50000000000000000, 5.0000000000000000,
          10.000000000000000, 0.69999999999999996 },
  { 1.3122319926925459, 0.50000000000000000, 5.0000000000000000,
          10.000000000000000, 0.80000000000000004 },
  { 1.3834948587364100, 0.50000000000000000, 5.0000000000000000,
          10.000000000000000, 0.89999999999999991 },
};

// Test function for a=0.50000000000000000, b=5.0000000000000000, c=10.000000000000000.
template <typename Tp>
void test060()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data060)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data060[i].a), Tp(data060[i].b),
                   Tp(data060[i].c), Tp(data060[i].x));
      const Tp f0 = data060[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=0.50000000000000000, b=10.000000000000000, c=2.0000000000000000.
testcase_hyperg<double> data061[] = {
  { 0.37727530159464628, 0.50000000000000000, 10.000000000000000,
          2.0000000000000000, -0.90000000000000002 },
  { 0.39816010922169010, 0.50000000000000000, 10.000000000000000,
          2.0000000000000000, -0.80000000000000004 },
  { 0.42283703041362453, 0.50000000000000000, 10.000000000000000,
          2.0000000000000000, -0.69999999999999996 },
  { 0.45255640448730505, 0.50000000000000000, 10.000000000000000,
          2.0000000000000000, -0.59999999999999998 },
  { 0.48919507154431141, 0.50000000000000000, 10.000000000000000,
          2.0000000000000000, -0.50000000000000000 },
  { 0.53569358917731880, 0.50000000000000000, 10.000000000000000,
          2.0000000000000000, -0.40000000000000002 },
  { 0.59689778897029566, 0.50000000000000000, 10.000000000000000,
          2.0000000000000000, -0.30000000000000004 },
  { 0.68128587569875731, 0.50000000000000000, 10.000000000000000,
          2.0000000000000000, -0.19999999999999996 },
  { 0.80478739308790359, 0.50000000000000000, 10.000000000000000,
          2.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 0.50000000000000000, 10.000000000000000,
          2.0000000000000000, 0.0000000000000000 },
  { 1.3408664196153621, 0.50000000000000000, 10.000000000000000,
          2.0000000000000000, 0.10000000000000009 },
  { 2.0175364359923860, 0.50000000000000000, 10.000000000000000,
          2.0000000000000000, 0.19999999999999996 },
  { 3.6011214553736646, 0.50000000000000000, 10.000000000000000,
          2.0000000000000000, 0.30000000000000004 },
  { 8.1799429939495312, 0.50000000000000000, 10.000000000000000,
          2.0000000000000000, 0.39999999999999991 },
  { 25.644834637536000, 0.50000000000000000, 10.000000000000000,
          2.0000000000000000, 0.50000000000000000 },
  { 123.13738891597615, 0.50000000000000000, 10.000000000000000,
          2.0000000000000000, 0.60000000000000009 },
  { 1088.7122410321333, 0.50000000000000000, 10.000000000000000,
          2.0000000000000000, 0.69999999999999996 },
  { 27358.291704709951, 0.50000000000000000, 10.000000000000000,
          2.0000000000000000, 0.80000000000000004 },
  { 8174369.0266731177, 0.50000000000000000, 10.000000000000000,
          2.0000000000000000, 0.89999999999999991 },
};

// Test function for a=0.50000000000000000, b=10.000000000000000, c=2.0000000000000000.
template <typename Tp>
void test061()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data061)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data061[i].a), Tp(data061[i].b),
                   Tp(data061[i].c), Tp(data061[i].x));
      const Tp f0 = data061[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=0.50000000000000000, b=10.000000000000000, c=4.0000000000000000.
testcase_hyperg<double> data062[] = {
  { 0.53905528308450834, 0.50000000000000000, 10.000000000000000,
          4.0000000000000000, -0.90000000000000002 },
  { 0.56235533974376162, 0.50000000000000000, 10.000000000000000,
          4.0000000000000000, -0.80000000000000004 },
  { 0.58887657983263575, 0.50000000000000000, 10.000000000000000,
          4.0000000000000000, -0.69999999999999996 },
  { 0.61941227047262915, 0.50000000000000000, 10.000000000000000,
          4.0000000000000000, -0.59999999999999998 },
  { 0.65504896640793853, 0.50000000000000000, 10.000000000000000,
          4.0000000000000000, -0.50000000000000000 },
  { 0.69731666644529999, 0.50000000000000000, 10.000000000000000,
          4.0000000000000000, -0.40000000000000002 },
  { 0.74844073299399128, 0.50000000000000000, 10.000000000000000,
          4.0000000000000000, -0.30000000000000004 },
  { 0.81178446800105752, 0.50000000000000000, 10.000000000000000,
          4.0000000000000000, -0.19999999999999996 },
  { 0.89266981277598023, 0.50000000000000000, 10.000000000000000,
          4.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 0.50000000000000000, 10.000000000000000,
          4.0000000000000000, 0.0000000000000000 },
  { 1.1497248473106778, 0.50000000000000000, 10.000000000000000,
          4.0000000000000000, 0.10000000000000009 },
  { 1.3729717112654571, 0.50000000000000000, 10.000000000000000,
          4.0000000000000000, 0.19999999999999996 },
  { 1.7374982340374392, 0.50000000000000000, 10.000000000000000,
          4.0000000000000000, 0.30000000000000004 },
  { 2.4134479340960580, 0.50000000000000000, 10.000000000000000,
          4.0000000000000000, 0.39999999999999991 },
  { 3.9191255240471192, 0.50000000000000000, 10.000000000000000,
          4.0000000000000000, 0.50000000000000000 },
  { 8.3316373077761270, 0.50000000000000000, 10.000000000000000,
          4.0000000000000000, 0.60000000000000009 },
  { 28.323020339843335, 0.50000000000000000, 10.000000000000000,
          4.0000000000000000, 0.69999999999999996 },
  { 225.84286572747891, 0.50000000000000000, 10.000000000000000,
          4.0000000000000000, 0.80000000000000004 },
  { 12757.127591286655, 0.50000000000000000, 10.000000000000000,
          4.0000000000000000, 0.89999999999999991 },
};

// Test function for a=0.50000000000000000, b=10.000000000000000, c=4.0000000000000000.
template <typename Tp>
void test062()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data062)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data062[i].a), Tp(data062[i].b),
                   Tp(data062[i].c), Tp(data062[i].x));
      const Tp f0 = data062[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=0.50000000000000000, b=10.000000000000000, c=6.0000000000000000.
testcase_hyperg<double> data063[] = {
  { 0.62672622092226071, 0.50000000000000000, 10.000000000000000,
          6.0000000000000000, -0.90000000000000002 },
  { 0.64931010269769829, 0.50000000000000000, 10.000000000000000,
          6.0000000000000000, -0.80000000000000004 },
  { 0.67448067519076316, 0.50000000000000000, 10.000000000000000,
          6.0000000000000000, -0.69999999999999996 },
  { 0.70276306239803676, 0.50000000000000000, 10.000000000000000,
          6.0000000000000000, -0.59999999999999998 },
  { 0.73484179773087555, 0.50000000000000000, 10.000000000000000,
          6.0000000000000000, -0.50000000000000000 },
  { 0.77162761412743897, 0.50000000000000000, 10.000000000000000,
          6.0000000000000000, -0.40000000000000002 },
  { 0.81436116844816531, 0.50000000000000000, 10.000000000000000,
          6.0000000000000000, -0.30000000000000004 },
  { 0.86477994787944557, 0.50000000000000000, 10.000000000000000,
          6.0000000000000000, -0.19999999999999996 },
  { 0.92539820516603888, 0.50000000000000000, 10.000000000000000,
          6.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 0.50000000000000000, 10.000000000000000,
          6.0000000000000000, 0.0000000000000000 },
  { 1.0945599448210315, 0.50000000000000000, 10.000000000000000,
          6.0000000000000000, 0.10000000000000009 },
  { 1.2190897395597264, 0.50000000000000000, 10.000000000000000,
          6.0000000000000000, 0.19999999999999996 },
  { 1.3916844336856475, 0.50000000000000000, 10.000000000000000,
          6.0000000000000000, 0.30000000000000004 },
  { 1.6484497630432013, 0.50000000000000000, 10.000000000000000,
          6.0000000000000000, 0.39999999999999991 },
  { 2.0717772717131155, 0.50000000000000000, 10.000000000000000,
          6.0000000000000000, 0.50000000000000000 },
  { 2.8893613630810924, 0.50000000000000000, 10.000000000000000,
          6.0000000000000000, 0.60000000000000009 },
  { 4.9459404075413529, 0.50000000000000000, 10.000000000000000,
          6.0000000000000000, 0.69999999999999996 },
  { 13.487394149998716, 0.50000000000000000, 10.000000000000000,
          6.0000000000000000, 0.80000000000000004 },
  { 136.57616044013972, 0.50000000000000000, 10.000000000000000,
          6.0000000000000000, 0.89999999999999991 },
};

// Test function for a=0.50000000000000000, b=10.000000000000000, c=6.0000000000000000.
template <typename Tp>
void test063()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data063)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data063[i].a), Tp(data063[i].b),
                   Tp(data063[i].c), Tp(data063[i].x));
      const Tp f0 = data063[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=0.50000000000000000, b=10.000000000000000, c=8.0000000000000000.
testcase_hyperg<double> data064[] = {
  { 0.68421604440344341, 0.50000000000000000, 10.000000000000000,
          8.0000000000000000, -0.90000000000000002 },
  { 0.70548098055548891, 0.50000000000000000, 10.000000000000000,
          8.0000000000000000, -0.80000000000000004 },
  { 0.72884342311710348, 0.50000000000000000, 10.000000000000000,
          8.0000000000000000, -0.69999999999999996 },
  { 0.75466953437856243, 0.50000000000000000, 10.000000000000000,
          8.0000000000000000, -0.59999999999999998 },
  { 0.78342090924662600, 0.50000000000000000, 10.000000000000000,
          8.0000000000000000, -0.50000000000000000 },
  { 0.81568884278645049, 0.50000000000000000, 10.000000000000000,
          8.0000000000000000, -0.40000000000000002 },
  { 0.85224480241465206, 0.50000000000000000, 10.000000000000000,
          8.0000000000000000, -0.30000000000000004 },
  { 0.89411692571131696, 0.50000000000000000, 10.000000000000000,
          8.0000000000000000, -0.19999999999999996 },
  { 0.94270986892954756, 0.50000000000000000, 10.000000000000000,
          8.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 0.50000000000000000, 10.000000000000000,
          8.0000000000000000, 0.0000000000000000 },
  { 1.0688682849120232, 0.50000000000000000, 10.000000000000000,
          8.0000000000000000, 0.10000000000000009 },
  { 1.1537004376097553, 0.50000000000000000, 10.000000000000000,
          8.0000000000000000, 0.19999999999999996 },
  { 1.2615455028370031, 0.50000000000000000, 10.000000000000000,
          8.0000000000000000, 0.30000000000000004 },
  { 1.4045541456153436, 0.50000000000000000, 10.000000000000000,
          8.0000000000000000, 0.39999999999999991 },
  { 1.6057216489444517, 0.50000000000000000, 10.000000000000000,
          8.0000000000000000, 0.50000000000000000 },
  { 1.9146603020550739, 0.50000000000000000, 10.000000000000000,
          8.0000000000000000, 0.60000000000000009 },
  { 2.4617931307620298, 0.50000000000000000, 10.000000000000000,
          8.0000000000000000, 0.69999999999999996 },
  { 3.7267799624996498, 0.50000000000000000, 10.000000000000000,
          8.0000000000000000, 0.80000000000000004 },
  { 9.3880118036248401, 0.50000000000000000, 10.000000000000000,
          8.0000000000000000, 0.89999999999999991 },
};

// Test function for a=0.50000000000000000, b=10.000000000000000, c=8.0000000000000000.
template <typename Tp>
void test064()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data064)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data064[i].a), Tp(data064[i].b),
                   Tp(data064[i].c), Tp(data064[i].x));
      const Tp f0 = data064[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=0.50000000000000000, b=10.000000000000000, c=10.000000000000000.
testcase_hyperg<double> data065[] = {
  { 0.72547625011001171, 0.50000000000000000, 10.000000000000000,
          10.000000000000000, -0.90000000000000002 },
  { 0.74535599249992990, 0.50000000000000000, 10.000000000000000,
          10.000000000000000, -0.80000000000000004 },
  { 0.76696498884737041, 0.50000000000000000, 10.000000000000000,
          10.000000000000000, -0.69999999999999996 },
  { 0.79056941504209477, 0.50000000000000000, 10.000000000000000,
          10.000000000000000, -0.59999999999999998 },
  { 0.81649658092772603, 0.50000000000000000, 10.000000000000000,
          10.000000000000000, -0.50000000000000000 },
  { 0.84515425472851657, 0.50000000000000000, 10.000000000000000,
          10.000000000000000, -0.40000000000000002 },
  { 0.87705801930702920, 0.50000000000000000, 10.000000000000000,
          10.000000000000000, -0.30000000000000004 },
  { 0.91287092917527690, 0.50000000000000000, 10.000000000000000,
          10.000000000000000, -0.19999999999999996 },
  { 0.95346258924559224, 0.50000000000000000, 10.000000000000000,
          10.000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 0.50000000000000000, 10.000000000000000,
          10.000000000000000, 0.0000000000000000 },
  { 1.0540925533894598, 0.50000000000000000, 10.000000000000000,
          10.000000000000000, 0.10000000000000009 },
  { 1.1180339887498949, 0.50000000000000000, 10.000000000000000,
          10.000000000000000, 0.19999999999999996 },
  { 1.1952286093343938, 0.50000000000000000, 10.000000000000000,
          10.000000000000000, 0.30000000000000004 },
  { 1.2909944487358056, 0.50000000000000000, 10.000000000000000,
          10.000000000000000, 0.39999999999999991 },
  { 1.4142135623730949, 0.50000000000000000, 10.000000000000000,
          10.000000000000000, 0.50000000000000000 },
  { 1.5811388300841900, 0.50000000000000000, 10.000000000000000,
          10.000000000000000, 0.60000000000000009 },
  { 1.8257418583505536, 0.50000000000000000, 10.000000000000000,
          10.000000000000000, 0.69999999999999996 },
  { 2.2360679774997898, 0.50000000000000000, 10.000000000000000,
          10.000000000000000, 0.80000000000000004 },
  { 3.1622776601683782, 0.50000000000000000, 10.000000000000000,
          10.000000000000000, 0.89999999999999991 },
};

// Test function for a=0.50000000000000000, b=10.000000000000000, c=10.000000000000000.
template <typename Tp>
void test065()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data065)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data065[i].a), Tp(data065[i].b),
                   Tp(data065[i].c), Tp(data065[i].x));
      const Tp f0 = data065[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=0.50000000000000000, b=20.000000000000000, c=2.0000000000000000.
testcase_hyperg<double> data066[] = {
  { 0.26690449940521566, 0.50000000000000000, 20.000000000000000,
          2.0000000000000000, -0.90000000000000002 },
  { 0.28252302866181805, 0.50000000000000000, 20.000000000000000,
          2.0000000000000000, -0.80000000000000004 },
  { 0.30123616141153819, 0.50000000000000000, 20.000000000000000,
          2.0000000000000000, -0.69999999999999996 },
  { 0.32421384687602628, 0.50000000000000000, 20.000000000000000,
          2.0000000000000000, -0.59999999999999998 },
  { 0.35334630811776752, 0.50000000000000000, 20.000000000000000,
          2.0000000000000000, -0.50000000000000000 },
  { 0.39191793127467034, 0.50000000000000000, 20.000000000000000,
          2.0000000000000000, -0.40000000000000002 },
  { 0.44620488618129206, 0.50000000000000000, 20.000000000000000,
          2.0000000000000000, -0.30000000000000004 },
  { 0.52980896919265685, 0.50000000000000000, 20.000000000000000,
          2.0000000000000000, -0.19999999999999996 },
  { 0.67754711477562357, 0.50000000000000000, 20.000000000000000,
          2.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 0.50000000000000000, 20.000000000000000,
          2.0000000000000000, 0.0000000000000000 },
  { 1.9567557771780317, 0.50000000000000000, 20.000000000000000,
          2.0000000000000000, 0.10000000000000009 },
  { 6.1816042148333086, 0.50000000000000000, 20.000000000000000,
          2.0000000000000000, 0.19999999999999996 },
  { 35.653088618561227, 0.50000000000000000, 20.000000000000000,
          2.0000000000000000, 0.30000000000000004 },
  { 377.51482843179906, 0.50000000000000000, 20.000000000000000,
          2.0000000000000000, 0.39999999999999991 },
  { 7645.8816551195359, 0.50000000000000000, 20.000000000000000,
          2.0000000000000000, 0.50000000000000000 },
  { 354791.74537980522, 0.50000000000000000, 20.000000000000000,
          2.0000000000000000, 0.60000000000000009 },
  { 57009889.966638684, 0.50000000000000000, 20.000000000000000,
          2.0000000000000000, 0.69999999999999996 },
  { 83771357024.863937, 0.50000000000000000, 20.000000000000000,
          2.0000000000000000, 0.80000000000000004 },
  { 25866972896376408., 0.50000000000000000, 20.000000000000000,
          2.0000000000000000, 0.89999999999999991 },
};

// Test function for a=0.50000000000000000, b=20.000000000000000, c=2.0000000000000000.
template <typename Tp>
void test066()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data066)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data066[i].a), Tp(data066[i].b),
                   Tp(data066[i].c), Tp(data066[i].x));
      const Tp f0 = data066[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=0.50000000000000000, b=20.000000000000000, c=4.0000000000000000.
testcase_hyperg<double> data067[] = {
  { 0.40342659436153405, 0.50000000000000000, 20.000000000000000,
          4.0000000000000000, -0.90000000000000002 },
  { 0.42420571192034318, 0.50000000000000000, 20.000000000000000,
          4.0000000000000000, -0.80000000000000004 },
  { 0.44852768286073008, 0.50000000000000000, 20.000000000000000,
          4.0000000000000000, -0.69999999999999996 },
  { 0.47751245808592863, 0.50000000000000000, 20.000000000000000,
          4.0000000000000000, -0.59999999999999998 },
  { 0.51283632632707754, 0.50000000000000000, 20.000000000000000,
          4.0000000000000000, -0.50000000000000000 },
  { 0.55713468814894307, 0.50000000000000000, 20.000000000000000,
          4.0000000000000000, -0.40000000000000002 },
  { 0.61481320817757312, 0.50000000000000000, 20.000000000000000,
          4.0000000000000000, -0.30000000000000004 },
  { 0.69383483410097202, 0.50000000000000000, 20.000000000000000,
          4.0000000000000000, -0.19999999999999996 },
  { 0.81012002526006033, 0.50000000000000000, 20.000000000000000,
          4.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 0.50000000000000000, 20.000000000000000,
          4.0000000000000000, 0.0000000000000000 },
  { 1.3622225506603911, 0.50000000000000000, 20.000000000000000,
          4.0000000000000000, 0.10000000000000009 },
  { 2.2349513086109001, 0.50000000000000000, 20.000000000000000,
          4.0000000000000000, 0.19999999999999996 },
  { 5.1864917536761723, 0.50000000000000000, 20.000000000000000,
          4.0000000000000000, 0.30000000000000004 },
  { 21.020560423779411, 0.50000000000000000, 20.000000000000000,
          4.0000000000000000, 0.39999999999999991 },
  { 175.19649997100612, 0.50000000000000000, 20.000000000000000,
          4.0000000000000000, 0.50000000000000000 },
  { 3467.1587803688708, 0.50000000000000000, 20.000000000000000,
          4.0000000000000000, 0.60000000000000009 },
  { 225003.88683445856, 0.50000000000000000, 20.000000000000000,
          4.0000000000000000, 0.69999999999999996 },
  { 110837674.65652709, 0.50000000000000000, 20.000000000000000,
          4.0000000000000000, 0.80000000000000004 },
  { 6688966964170.6807, 0.50000000000000000, 20.000000000000000,
          4.0000000000000000, 0.89999999999999991 },
};

// Test function for a=0.50000000000000000, b=20.000000000000000, c=4.0000000000000000.
template <typename Tp>
void test067()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data067)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data067[i].a), Tp(data067[i].b),
                   Tp(data067[i].c), Tp(data067[i].x));
      const Tp f0 = data067[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=0.50000000000000000, b=20.000000000000000, c=6.0000000000000000.
testcase_hyperg<double> data068[] = {
  { 0.48716309885816761, 0.50000000000000000, 20.000000000000000,
          6.0000000000000000, -0.90000000000000002 },
  { 0.50965859152542281, 0.50000000000000000, 20.000000000000000,
          6.0000000000000000, -0.80000000000000004 },
  { 0.53554809210658971, 0.50000000000000000, 20.000000000000000,
          6.0000000000000000, -0.69999999999999996 },
  { 0.56576689207507136, 0.50000000000000000, 20.000000000000000,
          6.0000000000000000, -0.59999999999999998 },
  { 0.60164849637133688, 0.50000000000000000, 20.000000000000000,
          6.0000000000000000, -0.50000000000000000 },
  { 0.64516711595404375, 0.50000000000000000, 20.000000000000000,
          6.0000000000000000, -0.40000000000000002 },
  { 0.69938278735493542, 0.50000000000000000, 20.000000000000000,
          6.0000000000000000, -0.30000000000000004 },
  { 0.76931621518401860, 0.50000000000000000, 20.000000000000000,
          6.0000000000000000, -0.19999999999999996 },
  { 0.86381808725530695, 0.50000000000000000, 20.000000000000000,
          6.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 0.50000000000000000, 20.000000000000000,
          6.0000000000000000, 0.0000000000000000 },
  { 1.2152051956815531, 0.50000000000000000, 20.000000000000000,
          6.0000000000000000, 0.10000000000000009 },
  { 1.6052546785425543, 0.50000000000000000, 20.000000000000000,
          6.0000000000000000, 0.19999999999999996 },
  { 2.4765586046012635, 0.50000000000000000, 20.000000000000000,
          6.0000000000000000, 0.30000000000000004 },
  { 5.1564492216997486, 0.50000000000000000, 20.000000000000000,
          6.0000000000000000, 0.39999999999999991 },
  { 18.446158392136365, 0.50000000000000000, 20.000000000000000,
          6.0000000000000000, 0.50000000000000000 },
  { 150.44577670123971, 0.50000000000000000, 20.000000000000000,
          6.0000000000000000, 0.60000000000000009 },
  { 3862.6317400115768, 0.50000000000000000, 20.000000000000000,
          6.0000000000000000, 0.69999999999999996 },
  { 632428.34833625401, 0.50000000000000000, 20.000000000000000,
          6.0000000000000000, 0.80000000000000004 },
  { 7426927663.3808765, 0.50000000000000000, 20.000000000000000,
          6.0000000000000000, 0.89999999999999991 },
};

// Test function for a=0.50000000000000000, b=20.000000000000000, c=6.0000000000000000.
template <typename Tp>
void test068()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data068)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data068[i].a), Tp(data068[i].b),
                   Tp(data068[i].c), Tp(data068[i].x));
      const Tp f0 = data068[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=0.50000000000000000, b=20.000000000000000, c=8.0000000000000000.
testcase_hyperg<double> data069[] = {
  { 0.54703266209548362, 0.50000000000000000, 20.000000000000000,
          8.0000000000000000, -0.90000000000000002 },
  { 0.56997321774144971, 0.50000000000000000, 20.000000000000000,
          8.0000000000000000, -0.80000000000000004 },
  { 0.59603026159654970, 0.50000000000000000, 20.000000000000000,
          8.0000000000000000, -0.69999999999999996 },
  { 0.62596978851120511, 0.50000000000000000, 20.000000000000000,
          8.0000000000000000, -0.59999999999999998 },
  { 0.66084565876898926, 0.50000000000000000, 20.000000000000000,
          8.0000000000000000, -0.50000000000000000 },
  { 0.70215256667232839, 0.50000000000000000, 20.000000000000000,
          8.0000000000000000, -0.40000000000000002 },
  { 0.75208916592008568, 0.50000000000000000, 20.000000000000000,
          8.0000000000000000, -0.30000000000000004 },
  { 0.81403631111658648, 0.50000000000000000, 20.000000000000000,
          8.0000000000000000, -0.19999999999999996 },
  { 0.89348608489854608, 0.50000000000000000, 20.000000000000000,
          8.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 0.50000000000000000, 20.000000000000000,
          8.0000000000000000, 0.0000000000000000 },
  { 1.1517793185139173, 0.50000000000000000, 20.000000000000000,
          8.0000000000000000, 0.10000000000000009 },
  { 1.3878110313656598, 0.50000000000000000, 20.000000000000000,
          8.0000000000000000, 0.19999999999999996 },
  { 1.8061071794572381, 0.50000000000000000, 20.000000000000000,
          8.0000000000000000, 0.30000000000000004 },
  { 2.7148594517859586, 0.50000000000000000, 20.000000000000000,
          8.0000000000000000, 0.39999999999999991 },
  { 5.4529435709049361, 0.50000000000000000, 20.000000000000000,
          8.0000000000000000, 0.50000000000000000 },
  { 19.487310275377109, 0.50000000000000000, 20.000000000000000,
          8.0000000000000000, 0.60000000000000009 },
  { 191.69079165937470, 0.50000000000000000, 20.000000000000000,
          8.0000000000000000, 0.69999999999999996 },
  { 10218.543981792311, 0.50000000000000000, 20.000000000000000,
          8.0000000000000000, 0.80000000000000004 },
  { 23160836.646583911, 0.50000000000000000, 20.000000000000000,
          8.0000000000000000, 0.89999999999999991 },
};

// Test function for a=0.50000000000000000, b=20.000000000000000, c=8.0000000000000000.
template <typename Tp>
void test069()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data069)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data069[i].a), Tp(data069[i].b),
                   Tp(data069[i].c), Tp(data069[i].x));
      const Tp f0 = data069[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=0.50000000000000000, b=20.000000000000000, c=10.000000000000000.
testcase_hyperg<double> data070[] = {
  { 0.59292067298616002, 0.50000000000000000, 20.000000000000000,
          10.000000000000000, -0.90000000000000002 },
  { 0.61572496720679915, 0.50000000000000000, 20.000000000000000,
          10.000000000000000, -0.80000000000000004 },
  { 0.64135339122875623, 0.50000000000000000, 20.000000000000000,
          10.000000000000000, -0.69999999999999996 },
  { 0.67043457419280483, 0.50000000000000000, 20.000000000000000,
          10.000000000000000, -0.59999999999999998 },
  { 0.70380956268170980, 0.50000000000000000, 20.000000000000000,
          10.000000000000000, -0.50000000000000000 },
  { 0.74263251901495231, 0.50000000000000000, 20.000000000000000,
          10.000000000000000, -0.40000000000000002 },
  { 0.78853555445528278, 0.50000000000000000, 20.000000000000000,
          10.000000000000000, -0.30000000000000004 },
  { 0.84391122775673766, 0.50000000000000000, 20.000000000000000,
          10.000000000000000, -0.19999999999999996 },
  { 0.91242401018807406, 0.50000000000000000, 20.000000000000000,
          10.000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 0.50000000000000000, 20.000000000000000,
          10.000000000000000, 0.0000000000000000 },
  { 1.1169059681274873, 0.50000000000000000, 20.000000000000000,
          10.000000000000000, 0.10000000000000009 },
  { 1.2825928301302667, 0.50000000000000000, 20.000000000000000,
          10.000000000000000, 0.19999999999999996 },
  { 1.5385937789924939, 0.50000000000000000, 20.000000000000000,
          10.000000000000000, 0.30000000000000004 },
  { 1.9895771187893898, 0.50000000000000000, 20.000000000000000,
          10.000000000000000, 0.39999999999999991 },
  { 2.9707335806970168, 0.50000000000000000, 20.000000000000000,
          10.000000000000000, 0.50000000000000000 },
  { 6.0299506157180467, 0.50000000000000000, 20.000000000000000,
          10.000000000000000, 0.60000000000000009 },
  { 24.259090336955577, 0.50000000000000000, 20.000000000000000,
          10.000000000000000, 0.69999999999999996 },
  { 406.27267173257223, 0.50000000000000000, 20.000000000000000,
          10.000000000000000, 0.80000000000000004 },
  { 174330.03997220192, 0.50000000000000000, 20.000000000000000,
          10.000000000000000, 0.89999999999999991 },
};

// Test function for a=0.50000000000000000, b=20.000000000000000, c=10.000000000000000.
template <typename Tp>
void test070()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data070)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data070[i].a), Tp(data070[i].b),
                   Tp(data070[i].c), Tp(data070[i].x));
      const Tp f0 = data070[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=1.0000000000000000, b=0.0000000000000000, c=2.0000000000000000.
testcase_hyperg<double> data071[] = {
  { 1.0000000000000000, 1.0000000000000000, 0.0000000000000000,
          2.0000000000000000, -0.90000000000000002 },
  { 1.0000000000000000, 1.0000000000000000, 0.0000000000000000,
          2.0000000000000000, -0.80000000000000004 },
  { 1.0000000000000000, 1.0000000000000000, 0.0000000000000000,
          2.0000000000000000, -0.69999999999999996 },
  { 1.0000000000000000, 1.0000000000000000, 0.0000000000000000,
          2.0000000000000000, -0.59999999999999998 },
  { 1.0000000000000000, 1.0000000000000000, 0.0000000000000000,
          2.0000000000000000, -0.50000000000000000 },
  { 1.0000000000000000, 1.0000000000000000, 0.0000000000000000,
          2.0000000000000000, -0.40000000000000002 },
  { 1.0000000000000000, 1.0000000000000000, 0.0000000000000000,
          2.0000000000000000, -0.30000000000000004 },
  { 1.0000000000000000, 1.0000000000000000, 0.0000000000000000,
          2.0000000000000000, -0.19999999999999996 },
  { 1.0000000000000000, 1.0000000000000000, 0.0000000000000000,
          2.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 1.0000000000000000, 0.0000000000000000,
          2.0000000000000000, 0.0000000000000000 },
  { 1.0000000000000000, 1.0000000000000000, 0.0000000000000000,
          2.0000000000000000, 0.10000000000000009 },
  { 1.0000000000000000, 1.0000000000000000, 0.0000000000000000,
          2.0000000000000000, 0.19999999999999996 },
  { 1.0000000000000000, 1.0000000000000000, 0.0000000000000000,
          2.0000000000000000, 0.30000000000000004 },
  { 1.0000000000000000, 1.0000000000000000, 0.0000000000000000,
          2.0000000000000000, 0.39999999999999991 },
  { 1.0000000000000000, 1.0000000000000000, 0.0000000000000000,
          2.0000000000000000, 0.50000000000000000 },
  { 1.0000000000000000, 1.0000000000000000, 0.0000000000000000,
          2.0000000000000000, 0.60000000000000009 },
  { 1.0000000000000000, 1.0000000000000000, 0.0000000000000000,
          2.0000000000000000, 0.69999999999999996 },
  { 1.0000000000000000, 1.0000000000000000, 0.0000000000000000,
          2.0000000000000000, 0.80000000000000004 },
  { 1.0000000000000000, 1.0000000000000000, 0.0000000000000000,
          2.0000000000000000, 0.89999999999999991 },
};

// Test function for a=1.0000000000000000, b=0.0000000000000000, c=2.0000000000000000.
template <typename Tp>
void test071()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data071)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data071[i].a), Tp(data071[i].b),
                   Tp(data071[i].c), Tp(data071[i].x));
      const Tp f0 = data071[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=1.0000000000000000, b=0.0000000000000000, c=4.0000000000000000.
testcase_hyperg<double> data072[] = {
  { 1.0000000000000000, 1.0000000000000000, 0.0000000000000000,
          4.0000000000000000, -0.90000000000000002 },
  { 1.0000000000000000, 1.0000000000000000, 0.0000000000000000,
          4.0000000000000000, -0.80000000000000004 },
  { 1.0000000000000000, 1.0000000000000000, 0.0000000000000000,
          4.0000000000000000, -0.69999999999999996 },
  { 1.0000000000000000, 1.0000000000000000, 0.0000000000000000,
          4.0000000000000000, -0.59999999999999998 },
  { 1.0000000000000000, 1.0000000000000000, 0.0000000000000000,
          4.0000000000000000, -0.50000000000000000 },
  { 1.0000000000000000, 1.0000000000000000, 0.0000000000000000,
          4.0000000000000000, -0.40000000000000002 },
  { 1.0000000000000000, 1.0000000000000000, 0.0000000000000000,
          4.0000000000000000, -0.30000000000000004 },
  { 1.0000000000000000, 1.0000000000000000, 0.0000000000000000,
          4.0000000000000000, -0.19999999999999996 },
  { 1.0000000000000000, 1.0000000000000000, 0.0000000000000000,
          4.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 1.0000000000000000, 0.0000000000000000,
          4.0000000000000000, 0.0000000000000000 },
  { 1.0000000000000000, 1.0000000000000000, 0.0000000000000000,
          4.0000000000000000, 0.10000000000000009 },
  { 1.0000000000000000, 1.0000000000000000, 0.0000000000000000,
          4.0000000000000000, 0.19999999999999996 },
  { 1.0000000000000000, 1.0000000000000000, 0.0000000000000000,
          4.0000000000000000, 0.30000000000000004 },
  { 1.0000000000000000, 1.0000000000000000, 0.0000000000000000,
          4.0000000000000000, 0.39999999999999991 },
  { 1.0000000000000000, 1.0000000000000000, 0.0000000000000000,
          4.0000000000000000, 0.50000000000000000 },
  { 1.0000000000000000, 1.0000000000000000, 0.0000000000000000,
          4.0000000000000000, 0.60000000000000009 },
  { 1.0000000000000000, 1.0000000000000000, 0.0000000000000000,
          4.0000000000000000, 0.69999999999999996 },
  { 1.0000000000000000, 1.0000000000000000, 0.0000000000000000,
          4.0000000000000000, 0.80000000000000004 },
  { 1.0000000000000000, 1.0000000000000000, 0.0000000000000000,
          4.0000000000000000, 0.89999999999999991 },
};

// Test function for a=1.0000000000000000, b=0.0000000000000000, c=4.0000000000000000.
template <typename Tp>
void test072()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data072)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data072[i].a), Tp(data072[i].b),
                   Tp(data072[i].c), Tp(data072[i].x));
      const Tp f0 = data072[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=1.0000000000000000, b=0.0000000000000000, c=6.0000000000000000.
testcase_hyperg<double> data073[] = {
  { 1.0000000000000000, 1.0000000000000000, 0.0000000000000000,
          6.0000000000000000, -0.90000000000000002 },
  { 1.0000000000000000, 1.0000000000000000, 0.0000000000000000,
          6.0000000000000000, -0.80000000000000004 },
  { 1.0000000000000000, 1.0000000000000000, 0.0000000000000000,
          6.0000000000000000, -0.69999999999999996 },
  { 1.0000000000000000, 1.0000000000000000, 0.0000000000000000,
          6.0000000000000000, -0.59999999999999998 },
  { 1.0000000000000000, 1.0000000000000000, 0.0000000000000000,
          6.0000000000000000, -0.50000000000000000 },
  { 1.0000000000000000, 1.0000000000000000, 0.0000000000000000,
          6.0000000000000000, -0.40000000000000002 },
  { 1.0000000000000000, 1.0000000000000000, 0.0000000000000000,
          6.0000000000000000, -0.30000000000000004 },
  { 1.0000000000000000, 1.0000000000000000, 0.0000000000000000,
          6.0000000000000000, -0.19999999999999996 },
  { 1.0000000000000000, 1.0000000000000000, 0.0000000000000000,
          6.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 1.0000000000000000, 0.0000000000000000,
          6.0000000000000000, 0.0000000000000000 },
  { 1.0000000000000000, 1.0000000000000000, 0.0000000000000000,
          6.0000000000000000, 0.10000000000000009 },
  { 1.0000000000000000, 1.0000000000000000, 0.0000000000000000,
          6.0000000000000000, 0.19999999999999996 },
  { 1.0000000000000000, 1.0000000000000000, 0.0000000000000000,
          6.0000000000000000, 0.30000000000000004 },
  { 1.0000000000000000, 1.0000000000000000, 0.0000000000000000,
          6.0000000000000000, 0.39999999999999991 },
  { 1.0000000000000000, 1.0000000000000000, 0.0000000000000000,
          6.0000000000000000, 0.50000000000000000 },
  { 1.0000000000000000, 1.0000000000000000, 0.0000000000000000,
          6.0000000000000000, 0.60000000000000009 },
  { 1.0000000000000000, 1.0000000000000000, 0.0000000000000000,
          6.0000000000000000, 0.69999999999999996 },
  { 1.0000000000000000, 1.0000000000000000, 0.0000000000000000,
          6.0000000000000000, 0.80000000000000004 },
  { 1.0000000000000000, 1.0000000000000000, 0.0000000000000000,
          6.0000000000000000, 0.89999999999999991 },
};

// Test function for a=1.0000000000000000, b=0.0000000000000000, c=6.0000000000000000.
template <typename Tp>
void test073()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data073)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data073[i].a), Tp(data073[i].b),
                   Tp(data073[i].c), Tp(data073[i].x));
      const Tp f0 = data073[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=1.0000000000000000, b=0.0000000000000000, c=8.0000000000000000.
testcase_hyperg<double> data074[] = {
  { 1.0000000000000000, 1.0000000000000000, 0.0000000000000000,
          8.0000000000000000, -0.90000000000000002 },
  { 1.0000000000000000, 1.0000000000000000, 0.0000000000000000,
          8.0000000000000000, -0.80000000000000004 },
  { 1.0000000000000000, 1.0000000000000000, 0.0000000000000000,
          8.0000000000000000, -0.69999999999999996 },
  { 1.0000000000000000, 1.0000000000000000, 0.0000000000000000,
          8.0000000000000000, -0.59999999999999998 },
  { 1.0000000000000000, 1.0000000000000000, 0.0000000000000000,
          8.0000000000000000, -0.50000000000000000 },
  { 1.0000000000000000, 1.0000000000000000, 0.0000000000000000,
          8.0000000000000000, -0.40000000000000002 },
  { 1.0000000000000000, 1.0000000000000000, 0.0000000000000000,
          8.0000000000000000, -0.30000000000000004 },
  { 1.0000000000000000, 1.0000000000000000, 0.0000000000000000,
          8.0000000000000000, -0.19999999999999996 },
  { 1.0000000000000000, 1.0000000000000000, 0.0000000000000000,
          8.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 1.0000000000000000, 0.0000000000000000,
          8.0000000000000000, 0.0000000000000000 },
  { 1.0000000000000000, 1.0000000000000000, 0.0000000000000000,
          8.0000000000000000, 0.10000000000000009 },
  { 1.0000000000000000, 1.0000000000000000, 0.0000000000000000,
          8.0000000000000000, 0.19999999999999996 },
  { 1.0000000000000000, 1.0000000000000000, 0.0000000000000000,
          8.0000000000000000, 0.30000000000000004 },
  { 1.0000000000000000, 1.0000000000000000, 0.0000000000000000,
          8.0000000000000000, 0.39999999999999991 },
  { 1.0000000000000000, 1.0000000000000000, 0.0000000000000000,
          8.0000000000000000, 0.50000000000000000 },
  { 1.0000000000000000, 1.0000000000000000, 0.0000000000000000,
          8.0000000000000000, 0.60000000000000009 },
  { 1.0000000000000000, 1.0000000000000000, 0.0000000000000000,
          8.0000000000000000, 0.69999999999999996 },
  { 1.0000000000000000, 1.0000000000000000, 0.0000000000000000,
          8.0000000000000000, 0.80000000000000004 },
  { 1.0000000000000000, 1.0000000000000000, 0.0000000000000000,
          8.0000000000000000, 0.89999999999999991 },
};

// Test function for a=1.0000000000000000, b=0.0000000000000000, c=8.0000000000000000.
template <typename Tp>
void test074()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data074)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data074[i].a), Tp(data074[i].b),
                   Tp(data074[i].c), Tp(data074[i].x));
      const Tp f0 = data074[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=1.0000000000000000, b=0.0000000000000000, c=10.000000000000000.
testcase_hyperg<double> data075[] = {
  { 1.0000000000000000, 1.0000000000000000, 0.0000000000000000,
          10.000000000000000, -0.90000000000000002 },
  { 1.0000000000000000, 1.0000000000000000, 0.0000000000000000,
          10.000000000000000, -0.80000000000000004 },
  { 1.0000000000000000, 1.0000000000000000, 0.0000000000000000,
          10.000000000000000, -0.69999999999999996 },
  { 1.0000000000000000, 1.0000000000000000, 0.0000000000000000,
          10.000000000000000, -0.59999999999999998 },
  { 1.0000000000000000, 1.0000000000000000, 0.0000000000000000,
          10.000000000000000, -0.50000000000000000 },
  { 1.0000000000000000, 1.0000000000000000, 0.0000000000000000,
          10.000000000000000, -0.40000000000000002 },
  { 1.0000000000000000, 1.0000000000000000, 0.0000000000000000,
          10.000000000000000, -0.30000000000000004 },
  { 1.0000000000000000, 1.0000000000000000, 0.0000000000000000,
          10.000000000000000, -0.19999999999999996 },
  { 1.0000000000000000, 1.0000000000000000, 0.0000000000000000,
          10.000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 1.0000000000000000, 0.0000000000000000,
          10.000000000000000, 0.0000000000000000 },
  { 1.0000000000000000, 1.0000000000000000, 0.0000000000000000,
          10.000000000000000, 0.10000000000000009 },
  { 1.0000000000000000, 1.0000000000000000, 0.0000000000000000,
          10.000000000000000, 0.19999999999999996 },
  { 1.0000000000000000, 1.0000000000000000, 0.0000000000000000,
          10.000000000000000, 0.30000000000000004 },
  { 1.0000000000000000, 1.0000000000000000, 0.0000000000000000,
          10.000000000000000, 0.39999999999999991 },
  { 1.0000000000000000, 1.0000000000000000, 0.0000000000000000,
          10.000000000000000, 0.50000000000000000 },
  { 1.0000000000000000, 1.0000000000000000, 0.0000000000000000,
          10.000000000000000, 0.60000000000000009 },
  { 1.0000000000000000, 1.0000000000000000, 0.0000000000000000,
          10.000000000000000, 0.69999999999999996 },
  { 1.0000000000000000, 1.0000000000000000, 0.0000000000000000,
          10.000000000000000, 0.80000000000000004 },
  { 1.0000000000000000, 1.0000000000000000, 0.0000000000000000,
          10.000000000000000, 0.89999999999999991 },
};

// Test function for a=1.0000000000000000, b=0.0000000000000000, c=10.000000000000000.
template <typename Tp>
void test075()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data075)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data075[i].a), Tp(data075[i].b),
                   Tp(data075[i].c), Tp(data075[i].x));
      const Tp f0 = data075[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=1.0000000000000000, b=0.50000000000000000, c=2.0000000000000000.
testcase_hyperg<double> data076[] = {
  { 0.84089972268671609, 1.0000000000000000, 0.50000000000000000,
          2.0000000000000000, -0.90000000000000002 },
  { 0.85410196624968437, 1.0000000000000000, 0.50000000000000000,
          2.0000000000000000, -0.80000000000000004 },
  { 0.86811566011579933, 1.0000000000000000, 0.50000000000000000,
          2.0000000000000000, -0.69999999999999996 },
  { 0.88303688022450544, 1.0000000000000000, 0.50000000000000000,
          2.0000000000000000, -0.59999999999999998 },
  { 0.89897948556635565, 1.0000000000000000, 0.50000000000000000,
          2.0000000000000000, -0.50000000000000000 },
  { 0.91607978309961580, 1.0000000000000000, 0.50000000000000000,
          2.0000000000000000, -0.40000000000000002 },
  { 0.93450283399425305, 1.0000000000000000, 0.50000000000000000,
          2.0000000000000000, -0.30000000000000004 },
  { 0.95445115010332193, 1.0000000000000000, 0.50000000000000000,
          2.0000000000000000, -0.19999999999999996 },
  { 0.97617696340303095, 1.0000000000000000, 0.50000000000000000,
          2.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 1.0000000000000000, 0.50000000000000000,
          2.0000000000000000, 0.0000000000000000 },
  { 1.0263340389897240, 1.0000000000000000, 0.50000000000000000,
          2.0000000000000000, 0.10000000000000009 },
  { 1.0557280900008410, 1.0000000000000000, 0.50000000000000000,
          2.0000000000000000, 0.19999999999999996 },
  { 1.0889331564394962, 1.0000000000000000, 0.50000000000000000,
          2.0000000000000000, 0.30000000000000004 },
  { 1.1270166537925830, 1.0000000000000000, 0.50000000000000000,
          2.0000000000000000, 0.39999999999999991 },
  { 1.1715728752538095, 1.0000000000000000, 0.50000000000000000,
          2.0000000000000000, 0.50000000000000000 },
  { 1.2251482265544145, 1.0000000000000000, 0.50000000000000000,
          2.0000000000000000, 0.60000000000000009 },
  { 1.2922212642709541, 1.0000000000000000, 0.50000000000000000,
          2.0000000000000000, 0.69999999999999996 },
  { 1.3819660112501042, 1.0000000000000000, 0.50000000000000000,
          2.0000000000000000, 0.80000000000000004 },
  { 1.5194938532959119, 1.0000000000000000, 0.50000000000000000,
          2.0000000000000000, 0.89999999999999991 },
};

// Test function for a=1.0000000000000000, b=0.50000000000000000, c=2.0000000000000000.
template <typename Tp>
void test076()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data076)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data076[i].a), Tp(data076[i].b),
                   Tp(data076[i].c), Tp(data076[i].x));
      const Tp f0 = data076[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=1.0000000000000000, b=0.50000000000000000, c=4.0000000000000000.
testcase_hyperg<double> data077[] = {
  { 0.90992197313391499, 1.0000000000000000, 0.50000000000000000,
          4.0000000000000000, -0.90000000000000002 },
  { 0.91822592662244507, 1.0000000000000000, 0.50000000000000000,
          4.0000000000000000, -0.80000000000000004 },
  { 0.92687104566419531, 1.0000000000000000, 0.50000000000000000,
          4.0000000000000000, -0.69999999999999996 },
  { 0.93588628166548815, 1.0000000000000000, 0.50000000000000000,
          4.0000000000000000, -0.59999999999999998 },
  { 0.94530459215552909, 1.0000000000000000, 0.50000000000000000,
          4.0000000000000000, -0.50000000000000000 },
  { 0.95516374875247467, 1.0000000000000000, 0.50000000000000000,
          4.0000000000000000, -0.40000000000000002 },
  { 0.96550736800511827, 1.0000000000000000, 0.50000000000000000,
          4.0000000000000000, -0.30000000000000004 },
  { 0.97638624595136270, 1.0000000000000000, 0.50000000000000000,
          4.0000000000000000, -0.19999999999999996 },
  { 0.98786011482678993, 1.0000000000000000, 0.50000000000000000,
          4.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 1.0000000000000000, 0.50000000000000000,
          4.0000000000000000, 0.0000000000000000 },
  { 1.0128914530682316, 1.0000000000000000, 0.50000000000000000,
          4.0000000000000000, 0.10000000000000009 },
  { 1.0266391040215350, 1.0000000000000000, 0.50000000000000000,
          4.0000000000000000, 0.19999999999999996 },
  { 1.0413732738729464, 1.0000000000000000, 0.50000000000000000,
          4.0000000000000000, 0.30000000000000004 },
  { 1.0572599536532992, 1.0000000000000000, 0.50000000000000000,
          4.0000000000000000, 0.39999999999999991 },
  { 1.0745166004060953, 1.0000000000000000, 0.50000000000000000,
          4.0000000000000000, 0.50000000000000000 },
  { 1.0934387388831386, 1.0000000000000000, 0.50000000000000000,
          4.0000000000000000, 0.60000000000000009 },
  { 1.1144486980714641, 1.0000000000000000, 0.50000000000000000,
          4.0000000000000000, 0.69999999999999996 },
  { 1.1381966011250106, 1.0000000000000000, 0.50000000000000000,
          4.0000000000000000, 0.80000000000000004 },
  { 1.1658171625342397, 1.0000000000000000, 0.50000000000000000,
          4.0000000000000000, 0.89999999999999991 },
};

// Test function for a=1.0000000000000000, b=0.50000000000000000, c=4.0000000000000000.
template <typename Tp>
void test077()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data077)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data077[i].a), Tp(data077[i].b),
                   Tp(data077[i].c), Tp(data077[i].x));
      const Tp f0 = data077[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=1.0000000000000000, b=0.50000000000000000, c=6.0000000000000000.
testcase_hyperg<double> data078[] = {
  { 0.93641908369732885, 1.0000000000000000, 0.50000000000000000,
          6.0000000000000000, -0.90000000000000002 },
  { 0.94256349654111315, 1.0000000000000000, 0.50000000000000000,
          6.0000000000000000, -0.80000000000000004 },
  { 0.94890138508461319, 1.0000000000000000, 0.50000000000000000,
          6.0000000000000000, -0.69999999999999996 },
  { 0.95544578858430007, 1.0000000000000000, 0.50000000000000000,
          6.0000000000000000, -0.59999999999999998 },
  { 0.96221121193620718, 1.0000000000000000, 0.50000000000000000,
          6.0000000000000000, -0.50000000000000000 },
  { 0.96921386948293542, 1.0000000000000000, 0.50000000000000000,
          6.0000000000000000, -0.40000000000000002 },
  { 0.97647198488394704, 1.0000000000000000, 0.50000000000000000,
          6.0000000000000000, -0.30000000000000004 },
  { 0.98400616412578656, 1.0000000000000000, 0.50000000000000000,
          6.0000000000000000, -0.19999999999999996 },
  { 0.99183986544963032, 1.0000000000000000, 0.50000000000000000,
          6.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 1.0000000000000000, 0.50000000000000000,
          6.0000000000000000, 0.0000000000000000 },
  { 1.0085177124149158, 1.0000000000000000, 0.50000000000000000,
          6.0000000000000000, 0.10000000000000009 },
  { 1.0174294150407122, 1.0000000000000000, 0.50000000000000000,
          6.0000000000000000, 0.19999999999999996 },
  { 1.0267781897388850, 1.0000000000000000, 0.50000000000000000,
          6.0000000000000000, 0.30000000000000004 },
  { 1.0366157405967285, 1.0000000000000000, 0.50000000000000000,
          6.0000000000000000, 0.39999999999999991 },
  { 1.0470052068648839, 1.0000000000000000, 0.50000000000000000,
          6.0000000000000000, 0.50000000000000000 },
  { 1.0580253905513313, 1.0000000000000000, 0.50000000000000000,
          6.0000000000000000, 0.60000000000000009 },
  { 1.0697774741209765, 1.0000000000000000, 0.50000000000000000,
          6.0000000000000000, 0.69999999999999996 },
  { 1.0823965556448414, 1.0000000000000000, 0.50000000000000000,
          6.0000000000000000, 0.80000000000000004 },
  { 1.0960739512057103, 1.0000000000000000, 0.50000000000000000,
          6.0000000000000000, 0.89999999999999991 },
};

// Test function for a=1.0000000000000000, b=0.50000000000000000, c=6.0000000000000000.
template <typename Tp>
void test078()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data078)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data078[i].a), Tp(data078[i].b),
                   Tp(data078[i].c), Tp(data078[i].x));
      const Tp f0 = data078[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=1.0000000000000000, b=0.50000000000000000, c=8.0000000000000000.
testcase_hyperg<double> data079[] = {
  { 0.95069883346936235, 1.0000000000000000, 0.50000000000000000,
          8.0000000000000000, -0.90000000000000002 },
  { 0.95559618047704165, 1.0000000000000000, 0.50000000000000000,
          8.0000000000000000, -0.80000000000000004 },
  { 0.96061938755931653, 1.0000000000000000, 0.50000000000000000,
          8.0000000000000000, -0.69999999999999996 },
  { 0.96577553912851344, 1.0000000000000000, 0.50000000000000000,
          8.0000000000000000, -0.59999999999999998 },
  { 0.97107239473807716, 1.0000000000000000, 0.50000000000000000,
          8.0000000000000000, -0.50000000000000000 },
  { 0.97651848528588503, 1.0000000000000000, 0.50000000000000000,
          8.0000000000000000, -0.40000000000000002 },
  { 0.98212322830227139, 1.0000000000000000, 0.50000000000000000,
          8.0000000000000000, -0.30000000000000004 },
  { 0.98789706736195781, 1.0000000000000000, 0.50000000000000000,
          8.0000000000000000, -0.19999999999999996 },
  { 0.99385164237825074, 1.0000000000000000, 0.50000000000000000,
          8.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 1.0000000000000000, 0.50000000000000000,
          8.0000000000000000, 0.0000000000000000 },
  { 1.0063568569383123, 1.0000000000000000, 0.50000000000000000,
          8.0000000000000000, 0.10000000000000009 },
  { 1.0129389344715818, 1.0000000000000000, 0.50000000000000000,
          8.0000000000000000, 0.19999999999999996 },
  { 1.0197653907773940, 1.0000000000000000, 0.50000000000000000,
          8.0000000000000000, 0.30000000000000004 },
  { 1.0268583912277143, 1.0000000000000000, 0.50000000000000000,
          8.0000000000000000, 0.39999999999999991 },
  { 1.0342438793937092, 1.0000000000000000, 0.50000000000000000,
          8.0000000000000000, 0.50000000000000000 },
  { 1.0419526514766855, 1.0000000000000000, 0.50000000000000000,
          8.0000000000000000, 0.60000000000000009 },
  { 1.0500219124976327, 1.0000000000000000, 0.50000000000000000,
          8.0000000000000000, 0.69999999999999996 },
  { 1.0584976491907043, 1.0000000000000000, 0.50000000000000000,
          8.0000000000000000, 0.80000000000000004 },
  { 1.0674385240268101, 1.0000000000000000, 0.50000000000000000,
          8.0000000000000000, 0.89999999999999991 },
};

// Test function for a=1.0000000000000000, b=0.50000000000000000, c=8.0000000000000000.
template <typename Tp>
void test079()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data079)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data079[i].a), Tp(data079[i].b),
                   Tp(data079[i].c), Tp(data079[i].x));
      const Tp f0 = data079[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=1.0000000000000000, b=0.50000000000000000, c=10.000000000000000.
testcase_hyperg<double> data080[] = {
  { 0.95968319138913893, 1.0000000000000000, 0.50000000000000000,
          10.000000000000000, -0.90000000000000002 },
  { 0.96376169072755768, 1.0000000000000000, 0.50000000000000000,
          10.000000000000000, -0.80000000000000004 },
  { 0.96792900082729316, 1.0000000000000000, 0.50000000000000000,
          10.000000000000000, -0.69999999999999996 },
  { 0.97218942798115737, 1.0000000000000000, 0.50000000000000000,
          10.000000000000000, -0.59999999999999998 },
  { 0.97654763592586857, 1.0000000000000000, 0.50000000000000000,
          10.000000000000000, -0.50000000000000000 },
  { 0.98100869054353856, 1.0000000000000000, 0.50000000000000000,
          10.000000000000000, -0.40000000000000002 },
  { 0.98557811238699278, 1.0000000000000000, 0.50000000000000000,
          10.000000000000000, -0.30000000000000004 },
  { 0.99026193885795544, 1.0000000000000000, 0.50000000000000000,
          10.000000000000000, -0.19999999999999996 },
  { 0.99506679842072221, 1.0000000000000000, 0.50000000000000000,
          10.000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 1.0000000000000000, 0.50000000000000000,
          10.000000000000000, 0.0000000000000000 },
  { 1.0050696417919618, 1.0000000000000000, 0.50000000000000000,
          10.000000000000000, 0.10000000000000009 },
  { 1.0102847452747090, 1.0000000000000000, 0.50000000000000000,
          10.000000000000000, 0.19999999999999996 },
  { 1.0156554225057022, 1.0000000000000000, 0.50000000000000000,
          10.000000000000000, 0.30000000000000004 },
  { 1.0211930882963096, 1.0000000000000000, 0.50000000000000000,
          10.000000000000000, 0.39999999999999991 },
  { 1.0269107343740711, 1.0000000000000000, 0.50000000000000000,
          10.000000000000000, 0.50000000000000000 },
  { 1.0328232917216298, 1.0000000000000000, 0.50000000000000000,
          10.000000000000000, 0.60000000000000009 },
  { 1.0389481230247195, 1.0000000000000000, 0.50000000000000000,
          10.000000000000000, 0.69999999999999996 },
  { 1.0453057164134614, 1.0000000000000000, 0.50000000000000000,
          10.000000000000000, 0.80000000000000004 },
  { 1.0519207114461246, 1.0000000000000000, 0.50000000000000000,
          10.000000000000000, 0.89999999999999991 },
};

// Test function for a=1.0000000000000000, b=0.50000000000000000, c=10.000000000000000.
template <typename Tp>
void test080()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data080)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data080[i].a), Tp(data080[i].b),
                   Tp(data080[i].c), Tp(data080[i].x));
      const Tp f0 = data080[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=1.0000000000000000, b=1.0000000000000000, c=2.0000000000000000.
testcase_hyperg<double> data081[] = {
  { 0.71317098463599404, 1.0000000000000000, 1.0000000000000000,
          2.0000000000000000, -0.90000000000000002 },
  { 0.73473333112764871, 1.0000000000000000, 1.0000000000000000,
          2.0000000000000000, -0.80000000000000004 },
  { 0.75804035866024377, 1.0000000000000000, 1.0000000000000000,
          2.0000000000000000, -0.69999999999999996 },
  { 0.78333938207622600, 1.0000000000000000, 1.0000000000000000,
          2.0000000000000000, -0.59999999999999998 },
  { 0.81093021621632877, 1.0000000000000000, 1.0000000000000000,
          2.0000000000000000, -0.50000000000000000 },
  { 0.84118059155303226, 1.0000000000000000, 1.0000000000000000,
          2.0000000000000000, -0.40000000000000002 },
  { 0.87454754822497005, 1.0000000000000000, 1.0000000000000000,
          2.0000000000000000, -0.30000000000000004 },
  { 0.91160778396977304, 1.0000000000000000, 1.0000000000000000,
          2.0000000000000000, -0.19999999999999996 },
  { 0.95310179804324857, 1.0000000000000000, 1.0000000000000000,
          2.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 1.0000000000000000, 1.0000000000000000,
          2.0000000000000000, 0.0000000000000000 },
  { 1.0536051565782629, 1.0000000000000000, 1.0000000000000000,
          2.0000000000000000, 0.10000000000000009 },
  { 1.1157177565710485, 1.0000000000000000, 1.0000000000000000,
          2.0000000000000000, 0.19999999999999996 },
  { 1.1889164797957747, 1.0000000000000000, 1.0000000000000000,
          2.0000000000000000, 0.30000000000000004 },
  { 1.2770640594149765, 1.0000000000000000, 1.0000000000000000,
          2.0000000000000000, 0.39999999999999991 },
  { 1.3862943611198899, 1.0000000000000000, 1.0000000000000000,
          2.0000000000000000, 0.50000000000000000 },
  { 1.5271512197902593, 1.0000000000000000, 1.0000000000000000,
          2.0000000000000000, 0.60000000000000009 },
  { 1.7199611490370503, 1.0000000000000000, 1.0000000000000000,
          2.0000000000000000, 0.69999999999999996 },
  { 2.0117973905426232, 1.0000000000000000, 1.0000000000000000,
          2.0000000000000000, 0.80000000000000004 },
  { 2.5584278811044912, 1.0000000000000000, 1.0000000000000000,
          2.0000000000000000, 0.89999999999999991 },
};

// Test function for a=1.0000000000000000, b=1.0000000000000000, c=2.0000000000000000.
template <typename Tp>
void test081()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data081)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data081[i].a), Tp(data081[i].b),
                   Tp(data081[i].c), Tp(data081[i].x));
      const Tp f0 = data081[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=1.0000000000000000, b=1.0000000000000000, c=4.0000000000000000.
testcase_hyperg<double> data082[] = {
  { 0.83165649828125576, 1.0000000000000000, 1.0000000000000000,
          4.0000000000000000, -0.90000000000000002 },
  { 0.84626246650116577, 1.0000000000000000, 1.0000000000000000,
          4.0000000000000000, -0.80000000000000004 },
  { 0.86165287670267410, 1.0000000000000000, 1.0000000000000000,
          4.0000000000000000, -0.69999999999999996 },
  { 0.87790681762615241, 1.0000000000000000, 1.0000000000000000,
          4.0000000000000000, -0.59999999999999998 },
  { 0.89511583784087689, 1.0000000000000000, 1.0000000000000000,
          4.0000000000000000, -0.50000000000000000 },
  { 0.91338673957393823, 1.0000000000000000, 1.0000000000000000,
          4.0000000000000000, -0.40000000000000002 },
  { 0.93284521667331954, 1.0000000000000000, 1.0000000000000000,
          4.0000000000000000, -0.30000000000000004 },
  { 0.95364066873549813, 1.0000000000000000, 1.0000000000000000,
          4.0000000000000000, -0.19999999999999996 },
  { 0.97595268969924187, 1.0000000000000000, 1.0000000000000000,
          4.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 1.0000000000000000, 1.0000000000000000,
          4.0000000000000000, 0.0000000000000000 },
  { 1.0260530485179122, 1.0000000000000000, 1.0000000000000000,
          4.0000000000000000, 0.10000000000000009 },
  { 1.0544523154103413, 1.0000000000000000, 1.0000000000000000,
          4.0000000000000000, 0.19999999999999996 },
  { 1.0856358366643180, 1.0000000000000000, 1.0000000000000000,
          4.0000000000000000, 0.30000000000000004 },
  { 1.1201824010510930, 1.0000000000000000, 1.0000000000000000,
          4.0000000000000000, 0.39999999999999991 },
  { 1.1588830833596719, 1.0000000000000000, 1.0000000000000000,
          4.0000000000000000, 0.50000000000000000 },
  { 1.2028682930536780, 1.0000000000000000, 1.0000000000000000,
          4.0000000000000000, 0.60000000000000009 },
  { 1.2538561433469468, 1.0000000000000000, 1.0000000000000000,
          4.0000000000000000, 0.69999999999999996 },
  { 1.3147120107267418, 1.0000000000000000, 1.0000000000000000,
          4.0000000000000000, 0.80000000000000004 },
  { 1.3910528844853491, 1.0000000000000000, 1.0000000000000000,
          4.0000000000000000, 0.89999999999999991 },
};

// Test function for a=1.0000000000000000, b=1.0000000000000000, c=4.0000000000000000.
template <typename Tp>
void test082()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data082)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data082[i].a), Tp(data082[i].b),
                   Tp(data082[i].c), Tp(data082[i].x));
      const Tp f0 = data082[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=1.0000000000000000, b=1.0000000000000000, c=6.0000000000000000.
testcase_hyperg<double> data083[] = {
  { 0.87917686994924527, 1.0000000000000000, 1.0000000000000000,
          6.0000000000000000, -0.90000000000000002 },
  { 0.89033956110358414, 1.0000000000000000, 1.0000000000000000,
          6.0000000000000000, -0.80000000000000004 },
  { 0.90196195126098366, 1.0000000000000000, 1.0000000000000000,
          6.0000000000000000, -0.69999999999999996 },
  { 0.91408080149514692, 1.0000000000000000, 1.0000000000000000,
          6.0000000000000000, -0.59999999999999998 },
  { 0.92673756761314952, 1.0000000000000000, 1.0000000000000000,
          6.0000000000000000, -0.50000000000000000 },
  { 0.93997926630123407, 1.0000000000000000, 1.0000000000000000,
          6.0000000000000000, -0.40000000000000002 },
  { 0.95385955885019291, 1.0000000000000000, 1.0000000000000000,
          6.0000000000000000, -0.30000000000000004 },
  { 0.96844012412988900, 1.0000000000000000, 1.0000000000000000,
          6.0000000000000000, -0.19999999999999996 },
  { 0.98379242268046208, 1.0000000000000000, 1.0000000000000000,
          6.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 1.0000000000000000, 1.0000000000000000,
          6.0000000000000000, 0.0000000000000000 },
  { 1.0171615499181177, 1.0000000000000000, 1.0000000000000000,
          6.0000000000000000, 0.10000000000000009 },
  { 1.0353950776091037, 1.0000000000000000, 1.0000000000000000,
          6.0000000000000000, 0.19999999999999996 },
  { 1.0548437030651112, 1.0000000000000000, 1.0000000000000000,
          6.0000000000000000, 0.30000000000000004 },
  { 1.0756840039415978, 1.0000000000000000, 1.0000000000000000,
          6.0000000000000000, 0.39999999999999991 },
  { 1.0981384722661196, 1.0000000000000000, 1.0000000000000000,
          6.0000000000000000, 0.50000000000000000 },
  { 1.1224950318916129, 1.0000000000000000, 1.0000000000000000,
          6.0000000000000000, 0.60000000000000009 },
  { 1.1491396357184527, 1.0000000000000000, 1.0000000000000000,
          6.0000000000000000, 0.69999999999999996 },
  { 1.1786158344507012, 1.0000000000000000, 1.0000000000000000,
          6.0000000000000000, 0.80000000000000004 },
  { 1.2117500593515478, 1.0000000000000000, 1.0000000000000000,
          6.0000000000000000, 0.89999999999999991 },
};

// Test function for a=1.0000000000000000, b=1.0000000000000000, c=6.0000000000000000.
template <typename Tp>
void test083()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data083)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data083[i].a), Tp(data083[i].b),
                   Tp(data083[i].c), Tp(data083[i].x));
      const Tp f0 = data083[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=1.0000000000000000, b=1.0000000000000000, c=8.0000000000000000.
testcase_hyperg<double> data084[] = {
  { 0.90538259348578409, 1.0000000000000000, 1.0000000000000000,
          8.0000000000000000, -0.90000000000000002 },
  { 0.91444830598832050, 1.0000000000000000, 1.0000000000000000,
          8.0000000000000000, -0.80000000000000004 },
  { 0.92381915945974002, 1.0000000000000000, 1.0000000000000000,
          8.0000000000000000, -0.69999999999999996 },
  { 0.93351553488501815, 1.0000000000000000, 1.0000000000000000,
          8.0000000000000000, -0.59999999999999998 },
  { 0.94356001859234884, 1.0000000000000000, 1.0000000000000000,
          8.0000000000000000, -0.50000000000000000 },
  { 0.95397775039949584, 1.0000000000000000, 1.0000000000000000,
          8.0000000000000000, -0.40000000000000002 },
  { 0.96479684710618019, 1.0000000000000000, 1.0000000000000000,
          8.0000000000000000, -0.30000000000000004 },
  { 0.97604892281308531, 1.0000000000000000, 1.0000000000000000,
          8.0000000000000000, -0.19999999999999996 },
  { 0.98776973540356938, 1.0000000000000000, 1.0000000000000000,
          8.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 1.0000000000000000, 1.0000000000000000,
          8.0000000000000000, 0.0000000000000000 },
  { 1.0127864273812119, 1.0000000000000000, 1.0000000000000000,
          8.0000000000000000, 0.10000000000000009 },
  { 1.0261830717772533, 1.0000000000000000, 1.0000000000000000,
          8.0000000000000000, 0.19999999999999996 },
  { 1.0402531144740719, 1.0000000000000000, 1.0000000000000000,
          8.0000000000000000, 0.30000000000000004 },
  { 1.0550712790827002, 1.0000000000000000, 1.0000000000000000,
          8.0000000000000000, 0.39999999999999991 },
  { 1.0707271945059007, 1.0000000000000000, 1.0000000000000000,
          8.0000000000000000, 0.50000000000000000 },
  { 1.0873302420658923, 1.0000000000000000, 1.0000000000000000,
          8.0000000000000000, 0.60000000000000009 },
  { 1.1050168587085545, 1.0000000000000000, 1.0000000000000000,
          8.0000000000000000, 0.69999999999999996 },
  { 1.1239622188477687, 1.0000000000000000, 1.0000000000000000,
          8.0000000000000000, 0.80000000000000004 },
  { 1.1444006183097781, 1.0000000000000000, 1.0000000000000000,
          8.0000000000000000, 0.89999999999999991 },
};

// Test function for a=1.0000000000000000, b=1.0000000000000000, c=8.0000000000000000.
template <typename Tp>
void test084()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data084)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data084[i].a), Tp(data084[i].b),
                   Tp(data084[i].c), Tp(data084[i].x));
      const Tp f0 = data084[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=1.0000000000000000, b=1.0000000000000000, c=10.000000000000000.
testcase_hyperg<double> data085[] = {
  { 0.92211295632330403, 1.0000000000000000, 1.0000000000000000,
          10.000000000000000, -0.90000000000000002 },
  { 0.92975727737040659, 1.0000000000000000, 1.0000000000000000,
          10.000000000000000, -0.80000000000000004 },
  { 0.93761992348333467, 1.0000000000000000, 1.0000000000000000,
          10.000000000000000, -0.69999999999999996 },
  { 0.94571346180587801, 1.0000000000000000, 1.0000000000000000,
          10.000000000000000, -0.59999999999999998 },
  { 0.95405164371146867, 1.0000000000000000, 1.0000000000000000,
          10.000000000000000, -0.50000000000000000 },
  { 0.96264956879205954, 1.0000000000000000, 1.0000000000000000,
          10.000000000000000, -0.40000000000000002 },
  { 0.97152388013493096, 1.0000000000000000, 1.0000000000000000,
          10.000000000000000, -0.30000000000000004 },
  { 0.98069299877709348, 1.0000000000000000, 1.0000000000000000,
          10.000000000000000, -0.19999999999999996 },
  { 0.99017740778385854, 1.0000000000000000, 1.0000000000000000,
          10.000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 1.0000000000000000, 1.0000000000000000,
          10.000000000000000, 0.0000000000000000 },
  { 1.0101865087004833, 1.0000000000000000, 1.0000000000000000,
          10.000000000000000, 0.10000000000000009 },
  { 1.0207660479892111, 1.0000000000000000, 1.0000000000000000,
          10.000000000000000, 0.19999999999999996 },
  { 1.0317718013185031, 1.0000000000000000, 1.0000000000000000,
          10.000000000000000, 0.30000000000000004 },
  { 1.0432419144892398, 1.0000000000000000, 1.0000000000000000,
          10.000000000000000, 0.39999999999999991 },
  { 1.0552206786504446, 1.0000000000000000, 1.0000000000000000,
          10.000000000000000, 0.50000000000000000 },
  { 1.0677601383233675, 1.0000000000000000, 1.0000000000000000,
          10.000000000000000, 0.60000000000000009 },
  { 1.0809223485579968, 1.0000000000000000, 1.0000000000000000,
          10.000000000000000, 0.69999999999999996 },
  { 1.0947826783002668, 1.0000000000000000, 1.0000000000000000,
          10.000000000000000, 0.80000000000000004 },
  { 1.1094349304493603, 1.0000000000000000, 1.0000000000000000,
          10.000000000000000, 0.89999999999999991 },
};

// Test function for a=1.0000000000000000, b=1.0000000000000000, c=10.000000000000000.
template <typename Tp>
void test085()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data085)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data085[i].a), Tp(data085[i].b),
                   Tp(data085[i].c), Tp(data085[i].x));
      const Tp f0 = data085[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=1.0000000000000000, b=2.0000000000000000, c=2.0000000000000000.
testcase_hyperg<double> data086[] = {
  { 0.52631578947368429, 1.0000000000000000, 2.0000000000000000,
          2.0000000000000000, -0.90000000000000002 },
  { 0.55555555555555558, 1.0000000000000000, 2.0000000000000000,
          2.0000000000000000, -0.80000000000000004 },
  { 0.58823529411764708, 1.0000000000000000, 2.0000000000000000,
          2.0000000000000000, -0.69999999999999996 },
  { 0.62500000000000000, 1.0000000000000000, 2.0000000000000000,
          2.0000000000000000, -0.59999999999999998 },
  { 0.66666666666666663, 1.0000000000000000, 2.0000000000000000,
          2.0000000000000000, -0.50000000000000000 },
  { 0.71428571428571430, 1.0000000000000000, 2.0000000000000000,
          2.0000000000000000, -0.40000000000000002 },
  { 0.76923076923076927, 1.0000000000000000, 2.0000000000000000,
          2.0000000000000000, -0.30000000000000004 },
  { 0.83333333333333337, 1.0000000000000000, 2.0000000000000000,
          2.0000000000000000, -0.19999999999999996 },
  { 0.90909090909090906, 1.0000000000000000, 2.0000000000000000,
          2.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 1.0000000000000000, 2.0000000000000000,
          2.0000000000000000, 0.0000000000000000 },
  { 1.1111111111111112, 1.0000000000000000, 2.0000000000000000,
          2.0000000000000000, 0.10000000000000009 },
  { 1.2500000000000000, 1.0000000000000000, 2.0000000000000000,
          2.0000000000000000, 0.19999999999999996 },
  { 1.4285714285714286, 1.0000000000000000, 2.0000000000000000,
          2.0000000000000000, 0.30000000000000004 },
  { 1.6666666666666663, 1.0000000000000000, 2.0000000000000000,
          2.0000000000000000, 0.39999999999999991 },
  { 2.0000000000000000, 1.0000000000000000, 2.0000000000000000,
          2.0000000000000000, 0.50000000000000000 },
  { 2.5000000000000004, 1.0000000000000000, 2.0000000000000000,
          2.0000000000000000, 0.60000000000000009 },
  { 3.3333333333333330, 1.0000000000000000, 2.0000000000000000,
          2.0000000000000000, 0.69999999999999996 },
  { 5.0000000000000009, 1.0000000000000000, 2.0000000000000000,
          2.0000000000000000, 0.80000000000000004 },
  { 9.9999999999999929, 1.0000000000000000, 2.0000000000000000,
          2.0000000000000000, 0.89999999999999991 },
};

// Test function for a=1.0000000000000000, b=2.0000000000000000, c=2.0000000000000000.
template <typename Tp>
void test086()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data086)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data086[i].a), Tp(data086[i].b),
                   Tp(data086[i].c), Tp(data086[i].x));
      const Tp f0 = data086[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=1.0000000000000000, b=2.0000000000000000, c=4.0000000000000000.
testcase_hyperg<double> data087[] = {
  { 0.70351947549341520, 1.0000000000000000, 2.0000000000000000,
          4.0000000000000000, -0.90000000000000002 },
  { 0.72637503722092711, 1.0000000000000000, 2.0000000000000000,
          4.0000000000000000, -0.80000000000000004 },
  { 0.75099661564391251, 1.0000000000000000, 2.0000000000000000,
          4.0000000000000000, -0.69999999999999996 },
  { 0.77761647796730871, 1.0000000000000000, 2.0000000000000000,
          4.0000000000000000, -0.59999999999999998 },
  { 0.80651221621216451, 1.0000000000000000, 2.0000000000000000,
          4.0000000000000000, -0.50000000000000000 },
  { 0.83801894346580275, 1.0000000000000000, 2.0000000000000000,
          4.0000000000000000, -0.40000000000000002 },
  { 0.87254582050258467, 1.0000000000000000, 2.0000000000000000,
          4.0000000000000000, -0.30000000000000004 },
  { 0.91059888544083678, 1.0000000000000000, 2.0000000000000000,
          4.0000000000000000, -0.19999999999999996 },
  { 0.95281329145592386, 1.0000000000000000, 2.0000000000000000,
          4.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 1.0000000000000000, 2.0000000000000000,
          4.0000000000000000, 0.0000000000000000 },
  { 1.0532154477379738, 1.0000000000000000, 2.0000000000000000,
          4.0000000000000000, 0.10000000000000009 },
  { 1.1138692114741471, 1.0000000000000000, 2.0000000000000000,
          4.0000000000000000, 0.19999999999999996 },
  { 1.1838976095305187, 1.0000000000000000, 2.0000000000000000,
          4.0000000000000000, 0.30000000000000004 },
  { 1.2660586631630237, 1.0000000000000000, 2.0000000000000000,
          4.0000000000000000, 0.39999999999999991 },
  { 1.3644676665613118, 1.0000000000000000, 2.0000000000000000,
          4.0000000000000000, 0.50000000000000000 },
  { 1.4856585347316102, 1.0000000000000000, 2.0000000000000000,
          4.0000000000000000, 0.60000000000000009 },
  { 1.6409590443536872, 1.0000000000000000, 2.0000000000000000,
          4.0000000000000000, 0.69999999999999996 },
  { 1.8528798927325769, 1.0000000000000000, 2.0000000000000000,
          4.0000000000000000, 0.80000000000000004 },
  { 2.1789423102929644, 1.0000000000000000, 2.0000000000000000,
          4.0000000000000000, 0.89999999999999991 },
};

// Test function for a=1.0000000000000000, b=2.0000000000000000, c=4.0000000000000000.
template <typename Tp>
void test087()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data087)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data087[i].a), Tp(data087[i].b),
                   Tp(data087[i].c), Tp(data087[i].x));
      const Tp f0 = data087[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=1.0000000000000000, b=2.0000000000000000, c=6.0000000000000000.
testcase_hyperg<double> data088[] = {
  { 0.78068027379106253, 1.0000000000000000, 2.0000000000000000,
          6.0000000000000000, -0.90000000000000002 },
  { 0.79924541976981278, 1.0000000000000000, 2.0000000000000000,
          6.0000000000000000, -0.80000000000000004 },
  { 0.81891305585650942, 1.0000000000000000, 2.0000000000000000,
          6.0000000000000000, -0.69999999999999996 },
  { 0.83979799626213270, 1.0000000000000000, 2.0000000000000000,
          6.0000000000000000, -0.59999999999999998 },
  { 0.86203315303160166, 1.0000000000000000, 2.0000000000000000,
          6.0000000000000000, -0.50000000000000000 },
  { 0.88577352485361693, 1.0000000000000000, 2.0000000000000000,
          6.0000000000000000, -0.40000000000000002 },
  { 0.91120135738402208, 1.0000000000000000, 2.0000000000000000,
          6.0000000000000000, -0.30000000000000004 },
  { 0.93853291956703588, 1.0000000000000000, 2.0000000000000000,
          6.0000000000000000, -0.19999999999999996 },
  { 0.96802755388922956, 1.0000000000000000, 2.0000000000000000,
          6.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 1.0000000000000000, 2.0000000000000000,
          6.0000000000000000, 0.0000000000000000 },
  { 1.0348375559194773, 1.0000000000000000, 2.0000000000000000,
          6.0000000000000000, 0.10000000000000009 },
  { 1.0730246119544820, 1.0000000000000000, 2.0000000000000000,
          6.0000000000000000, 0.19999999999999996 },
  { 1.1151788396279341, 1.0000000000000000, 2.0000000000000000,
          6.0000000000000000, 0.30000000000000004 },
  { 1.1621066403893472, 1.0000000000000000, 2.0000000000000000,
          6.0000000000000000, 0.39999999999999991 },
  { 1.2148922218710421, 1.0000000000000000, 2.0000000000000000,
          6.0000000000000000, 0.50000000000000000 },
  { 1.2750496810838674, 1.0000000000000000, 2.0000000000000000,
          6.0000000000000000, 0.60000000000000009 },
  { 1.3448048570872917, 1.0000000000000000, 2.0000000000000000,
          6.0000000000000000, 0.69999999999999996 },
  { 1.4276833109859521, 1.0000000000000000, 2.0000000000000000,
          6.0000000000000000, 0.80000000000000004 },
  { 1.5299976259379788, 1.0000000000000000, 2.0000000000000000,
          6.0000000000000000, 0.89999999999999991 },
};

// Test function for a=1.0000000000000000, b=2.0000000000000000, c=6.0000000000000000.
template <typename Tp>
void test088()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data088)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data088[i].a), Tp(data088[i].b),
                   Tp(data088[i].c), Tp(data088[i].x));
      const Tp f0 = data088[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=1.0000000000000000, b=2.0000000000000000, c=8.0000000000000000.
testcase_hyperg<double> data089[] = {
  { 0.82510759951857582, 1.0000000000000000, 2.0000000000000000,
          8.0000000000000000, -0.90000000000000002 },
  { 0.84072786892782092, 1.0000000000000000, 2.0000000000000000,
          8.0000000000000000, -0.80000000000000004 },
  { 0.85710884896562356, 1.0000000000000000, 2.0000000000000000,
          8.0000000000000000, -0.69999999999999996 },
  { 0.87431674418118333, 1.0000000000000000, 2.0000000000000000,
          8.0000000000000000, -0.59999999999999998 },
  { 0.89242659229727017, 1.0000000000000000, 2.0000000000000000,
          8.0000000000000000, -0.50000000000000000 },
  { 0.91152392685930339, 1.0000000000000000, 2.0000000000000000,
          8.0000000000000000, -0.40000000000000002 },
  { 0.93170685950993570, 1.0000000000000000, 2.0000000000000000,
          8.0000000000000000, -0.30000000000000004 },
  { 0.95308871926790661, 1.0000000000000000, 2.0000000000000000,
          8.0000000000000000, -0.19999999999999996 },
  { 0.97580144325325802, 1.0000000000000000, 2.0000000000000000,
          8.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 1.0000000000000000, 2.0000000000000000,
          8.0000000000000000, 0.0000000000000000 },
  { 1.0258682619030324, 1.0000000000000000, 2.0000000000000000,
          8.0000000000000000, 0.10000000000000009 },
  { 1.0536269616706000, 1.0000000000000000, 2.0000000000000000,
          8.0000000000000000, 0.19999999999999996 },
  { 1.0835447330793833, 1.0000000000000000, 2.0000000000000000,
          8.0000000000000000, 0.30000000000000004 },
  { 1.1159538758396654, 1.0000000000000000, 2.0000000000000000,
          8.0000000000000000, 0.39999999999999991 },
  { 1.1512736659291880, 1.0000000000000000, 2.0000000000000000,
          8.0000000000000000, 0.50000000000000000 },
  { 1.1900463690116090, 1.0000000000000000, 2.0000000000000000,
          8.0000000000000000, 0.60000000000000009 },
  { 1.2329961591622411, 1.0000000000000000, 2.0000000000000000,
          8.0000000000000000, 0.69999999999999996 },
  { 1.2811334345669059, 1.0000000000000000, 2.0000000000000000,
          8.0000000000000000, 0.80000000000000004 },
  { 1.3359629014132051, 1.0000000000000000, 2.0000000000000000,
          8.0000000000000000, 0.89999999999999991 },
};

// Test function for a=1.0000000000000000, b=2.0000000000000000, c=8.0000000000000000.
template <typename Tp>
void test089()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data089)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data089[i].a), Tp(data089[i].b),
                   Tp(data089[i].c), Tp(data089[i].x));
      const Tp f0 = data089[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=1.0000000000000000, b=2.0000000000000000, c=10.000000000000000.
testcase_hyperg<double> data090[] = {
  { 0.85426123653345876, 1.0000000000000000, 2.0000000000000000,
          10.000000000000000, -0.90000000000000002 },
  { 0.86774543390930370, 1.0000000000000000, 2.0000000000000000,
          10.000000000000000, -0.80000000000000004 },
  { 0.88178859537254239, 1.0000000000000000, 2.0000000000000000,
          10.000000000000000, -0.69999999999999996 },
  { 0.89643269097060951, 1.0000000000000000, 2.0000000000000000,
          10.000000000000000, -0.59999999999999998 },
  { 0.91172456687216763, 1.0000000000000000, 2.0000000000000000,
          10.000000000000000, -0.50000000000000000 },
  { 0.92771674975966134, 1.0000000000000000, 2.0000000000000000,
          10.000000000000000, -0.40000000000000002 },
  { 0.94446842993888647, 1.0000000000000000, 2.0000000000000000,
          10.000000000000000, -0.30000000000000004 },
  { 0.96204667481937678, 1.0000000000000000, 2.0000000000000000,
          10.000000000000000, -0.19999999999999996 },
  { 0.98052794339012128, 1.0000000000000000, 2.0000000000000000,
          10.000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 1.0000000000000000, 2.0000000000000000,
          10.000000000000000, 0.0000000000000000 },
  { 1.0205643671068179, 1.0000000000000000, 2.0000000000000000,
          10.000000000000000, 0.10000000000000009 },
  { 1.0423395201078882, 1.0000000000000000, 2.0000000000000000,
          10.000000000000000, 0.19999999999999996 },
  { 1.0654651277885334, 1.0000000000000000, 2.0000000000000000,
          10.000000000000000, 0.30000000000000004 },
  { 1.0901078068101382, 1.0000000000000000, 2.0000000000000000,
          10.000000000000000, 0.39999999999999991 },
  { 1.1164691415928940, 1.0000000000000000, 2.0000000000000000,
          10.000000000000000, 0.50000000000000000 },
  { 1.1447972335326551, 1.0000000000000000, 2.0000000000000000,
          10.000000000000000, 0.60000000000000009 },
  { 1.1754040384534161, 1.0000000000000000, 2.0000000000000000,
          10.000000000000000, 0.69999999999999996 },
  { 1.2086928679893112, 1.0000000000000000, 2.0000000000000000,
          10.000000000000000, 0.80000000000000004 },
  { 1.2452055640510711, 1.0000000000000000, 2.0000000000000000,
          10.000000000000000, 0.89999999999999991 },
};

// Test function for a=1.0000000000000000, b=2.0000000000000000, c=10.000000000000000.
template <typename Tp>
void test090()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data090)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data090[i].a), Tp(data090[i].b),
                   Tp(data090[i].c), Tp(data090[i].x));
      const Tp f0 = data090[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=1.0000000000000000, b=5.0000000000000000, c=2.0000000000000000.
testcase_hyperg<double> data091[] = {
  { 0.25646288779245091, 1.0000000000000000, 5.0000000000000000,
          2.0000000000000000, -0.90000000000000002 },
  { 0.28273129096174382, 1.0000000000000000, 5.0000000000000000,
          2.0000000000000000, -0.80000000000000004 },
  { 0.31438201170962976, 1.0000000000000000, 5.0000000000000000,
          2.0000000000000000, -0.69999999999999996 },
  { 0.35308837890625017, 1.0000000000000000, 5.0000000000000000,
          2.0000000000000000, -0.59999999999999998 },
  { 0.40123456790123463, 1.0000000000000000, 5.0000000000000000,
          2.0000000000000000, -0.50000000000000000 },
  { 0.46230737192836319, 1.0000000000000000, 5.0000000000000000,
          2.0000000000000000, -0.40000000000000002 },
  { 0.54156016946185359, 1.0000000000000000, 5.0000000000000000,
          2.0000000000000000, -0.30000000000000004 },
  { 0.64718364197530875, 1.0000000000000000, 5.0000000000000000,
          2.0000000000000000, -0.19999999999999996 },
  { 0.79246636158732342, 1.0000000000000000, 5.0000000000000000,
          2.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 1.0000000000000000, 5.0000000000000000,
          2.0000000000000000, 0.0000000000000000 },
  { 1.3103947568968148, 1.0000000000000000, 5.0000000000000000,
          2.0000000000000000, 0.10000000000000009 },
  { 1.8017578125000004, 1.0000000000000000, 5.0000000000000000,
          2.0000000000000000, 0.19999999999999996 },
  { 2.6374427321949185, 1.0000000000000000, 5.0000000000000000,
          2.0000000000000000, 0.30000000000000004 },
  { 4.1975308641975282, 1.0000000000000000, 5.0000000000000000,
          2.0000000000000000, 0.39999999999999991 },
  { 7.4999999999999964, 1.0000000000000000, 5.0000000000000000,
          2.0000000000000000, 0.50000000000000000 },
  { 15.859375000000012, 1.0000000000000000, 5.0000000000000000,
          2.0000000000000000, 0.60000000000000009 },
  { 43.734567901234513, 1.0000000000000000, 5.0000000000000000,
          2.0000000000000000, 0.69999999999999996 },
  { 194.99999999999994, 1.0000000000000000, 5.0000000000000000,
          2.0000000000000000, 0.80000000000000004 },
  { 2777.4999999999832, 1.0000000000000000, 5.0000000000000000,
          2.0000000000000000, 0.89999999999999991 },
};

// Test function for a=1.0000000000000000, b=5.0000000000000000, c=2.0000000000000000.
template <typename Tp>
void test091()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data091)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data091[i].a), Tp(data091[i].b),
                   Tp(data091[i].c), Tp(data091[i].x));
      const Tp f0 = data091[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=1.0000000000000000, b=5.0000000000000000, c=4.0000000000000000.
testcase_hyperg<double> data092[] = {
  { 0.46398891966759004, 1.0000000000000000, 5.0000000000000000,
          4.0000000000000000, -0.90000000000000002 },
  { 0.49382716049382713, 1.0000000000000000, 5.0000000000000000,
          4.0000000000000000, -0.80000000000000004 },
  { 0.52768166089965396, 1.0000000000000000, 5.0000000000000000,
          4.0000000000000000, -0.69999999999999996 },
  { 0.56640624999999978, 1.0000000000000000, 5.0000000000000000,
          4.0000000000000000, -0.59999999999999998 },
  { 0.61111111111111116, 1.0000000000000000, 5.0000000000000000,
          4.0000000000000000, -0.50000000000000000 },
  { 0.66326530612244894, 1.0000000000000000, 5.0000000000000000,
          4.0000000000000000, -0.40000000000000002 },
  { 0.72485207100591698, 1.0000000000000000, 5.0000000000000000,
          4.0000000000000000, -0.30000000000000004 },
  { 0.79861111111111094, 1.0000000000000000, 5.0000000000000000,
          4.0000000000000000, -0.19999999999999996 },
  { 0.88842975206611552, 1.0000000000000000, 5.0000000000000000,
          4.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 1.0000000000000000, 5.0000000000000000,
          4.0000000000000000, 0.0000000000000000 },
  { 1.1419753086419753, 1.0000000000000000, 5.0000000000000000,
          4.0000000000000000, 0.10000000000000009 },
  { 1.3281249999999998, 1.0000000000000000, 5.0000000000000000,
          4.0000000000000000, 0.19999999999999996 },
  { 1.5816326530612239, 1.0000000000000000, 5.0000000000000000,
          4.0000000000000000, 0.30000000000000004 },
  { 1.9444444444444444, 1.0000000000000000, 5.0000000000000000,
          4.0000000000000000, 0.39999999999999991 },
  { 2.5000000000000000, 1.0000000000000000, 5.0000000000000000,
          4.0000000000000000, 0.50000000000000000 },
  { 3.4374999999999996, 1.0000000000000000, 5.0000000000000000,
          4.0000000000000000, 0.60000000000000009 },
  { 5.2777777777777715, 1.0000000000000000, 5.0000000000000000,
          4.0000000000000000, 0.69999999999999996 },
  { 9.9999999999999947, 1.0000000000000000, 5.0000000000000000,
          4.0000000000000000, 0.80000000000000004 },
  { 32.499999999999837, 1.0000000000000000, 5.0000000000000000,
          4.0000000000000000, 0.89999999999999991 },
};

// Test function for a=1.0000000000000000, b=5.0000000000000000, c=4.0000000000000000.
template <typename Tp>
void test092()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data092)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data092[i].a), Tp(data092[i].b),
                   Tp(data092[i].c), Tp(data092[i].x));
      const Tp f0 = data092[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=1.0000000000000000, b=5.0000000000000000, c=6.0000000000000000.
testcase_hyperg<double> data093[] = {
  { 0.57476744883397501, 1.0000000000000000, 5.0000000000000000,
          6.0000000000000000, -0.90000000000000002 },
  { 0.60302731682513933, 1.0000000000000000, 5.0000000000000000,
          6.0000000000000000, -0.80000000000000004 },
  { 0.63425708719096396, 1.0000000000000000, 5.0000000000000000,
          6.0000000000000000, -0.69999999999999996 },
  { 0.66895764182970419, 1.0000000000000000, 5.0000000000000000,
          6.0000000000000000, -0.59999999999999998 },
  { 0.70775063063963428, 1.0000000000000000, 5.0000000000000000,
          6.0000000000000000, -0.50000000000000000 },
  { 0.75141762103495924, 1.0000000000000000, 5.0000000000000000,
          6.0000000000000000, -0.40000000000000002 },
  { 0.80095569442603298, 1.0000000000000000, 5.0000000000000000,
          6.0000000000000000, -0.30000000000000004 },
  { 0.85765823887436754, 1.0000000000000000, 5.0000000000000000,
          6.0000000000000000, -0.19999999999999996 },
  { 0.92323549576335540, 1.0000000000000000, 5.0000000000000000,
          6.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 1.0000000000000000, 5.0000000000000000,
          6.0000000000000000, 0.0000000000000000 },
  { 1.0911622464839472, 1.0000000000000000, 5.0000000000000000,
          6.0000000000000000, 0.10000000000000009 },
  { 1.2013226178607666, 1.0000000000000000, 5.0000000000000000,
          6.0000000000000000, 0.19999999999999996 },
  { 1.3373332072682687, 1.0000000000000000, 5.0000000000000000,
          6.0000000000000000, 0.30000000000000004 },
  { 1.5099074378209716, 1.0000000000000000, 5.0000000000000000,
          6.0000000000000000, 0.39999999999999991 },
  { 1.7368822229245819, 1.0000000000000000, 5.0000000000000000,
          6.0000000000000000, 0.50000000000000000 },
  { 2.0505871832661429, 1.0000000000000000, 5.0000000000000000,
          6.0000000000000000, 0.60000000000000009 },
  { 2.5172389775867967, 1.0000000000000000, 5.0000000000000000,
          6.0000000000000000, 0.69999999999999996 },
  { 3.3015631983556144, 1.0000000000000000, 5.0000000000000000,
          6.0000000000000000, 0.80000000000000004 },
  { 5.0005935155044519, 1.0000000000000000, 5.0000000000000000,
          6.0000000000000000, 0.89999999999999991 },
};

// Test function for a=1.0000000000000000, b=5.0000000000000000, c=6.0000000000000000.
template <typename Tp>
void test093()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data093)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data093[i].a), Tp(data093[i].b),
                   Tp(data093[i].c), Tp(data093[i].x));
      const Tp f0 = data093[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=1.0000000000000000, b=5.0000000000000000, c=8.0000000000000000.
testcase_hyperg<double> data094[] = {
  { 0.64582752605387961, 1.0000000000000000, 5.0000000000000000,
          8.0000000000000000, -0.90000000000000002 },
  { 0.67184161997264169, 1.0000000000000000, 5.0000000000000000,
          8.0000000000000000, -0.80000000000000004 },
  { 0.70012779922368040, 1.0000000000000000, 5.0000000000000000,
          8.0000000000000000, -0.69999999999999996 },
  { 0.73100784656910256, 1.0000000000000000, 5.0000000000000000,
          8.0000000000000000, -0.59999999999999998 },
  { 0.76486919089091077, 1.0000000000000000, 5.0000000000000000,
          8.0000000000000000, -0.50000000000000000 },
  { 0.80218301124334579, 1.0000000000000000, 5.0000000000000000,
          8.0000000000000000, -0.40000000000000002 },
  { 0.84352883533234446, 1.0000000000000000, 5.0000000000000000,
          8.0000000000000000, -0.30000000000000004 },
  { 0.88962858902212572, 1.0000000000000000, 5.0000000000000000,
          8.0000000000000000, -0.19999999999999996 },
  { 0.94139473468584123, 1.0000000000000000, 5.0000000000000000,
          8.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 1.0000000000000000, 5.0000000000000000,
          8.0000000000000000, 0.0000000000000000 },
  { 1.0669812691939897, 1.0000000000000000, 5.0000000000000000,
          8.0000000000000000, 0.10000000000000009 },
  { 1.1443996012177726, 1.0000000000000000, 5.0000000000000000,
          8.0000000000000000, 0.19999999999999996 },
  { 1.2350966976721314, 1.0000000000000000, 5.0000000000000000,
          8.0000000000000000, 0.30000000000000004 },
  { 1.3431264370409088, 1.0000000000000000, 5.0000000000000000,
          8.0000000000000000, 0.39999999999999991 },
  { 1.4745266814162399, 1.0000000000000000, 5.0000000000000000,
          8.0000000000000000, 0.50000000000000000 },
  { 1.6388137104840066, 1.0000000000000000, 5.0000000000000000,
          8.0000000000000000, 0.60000000000000009 },
  { 1.8522074849776518, 1.0000000000000000, 5.0000000000000000,
          8.0000000000000000, 0.69999999999999996 },
  { 2.1458016978417458, 1.0000000000000000, 5.0000000000000000,
          8.0000000000000000, 0.80000000000000004 },
  { 2.5927464669826339, 1.0000000000000000, 5.0000000000000000,
          8.0000000000000000, 0.89999999999999991 },
};

// Test function for a=1.0000000000000000, b=5.0000000000000000, c=8.0000000000000000.
template <typename Tp>
void test094()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data094)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data094[i].a), Tp(data094[i].b),
                   Tp(data094[i].c), Tp(data094[i].x));
      const Tp f0 = data094[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=1.0000000000000000, b=5.0000000000000000, c=10.000000000000000.
testcase_hyperg<double> data095[] = {
  { 0.69583236336670629, 1.0000000000000000, 5.0000000000000000,
          10.000000000000000, -0.90000000000000002 },
  { 0.71968920666899694, 1.0000000000000000, 5.0000000000000000,
          10.000000000000000, -0.80000000000000004 },
  { 0.74533885416044232, 1.0000000000000000, 5.0000000000000000,
          10.000000000000000, -0.69999999999999996 },
  { 0.77300145361503014, 1.0000000000000000, 5.0000000000000000,
          10.000000000000000, -0.59999999999999998 },
  { 0.80293630810919514, 1.0000000000000000, 5.0000000000000000,
          10.000000000000000, -0.50000000000000000 },
  { 0.83545132638592035, 1.0000000000000000, 5.0000000000000000,
          10.000000000000000, -0.40000000000000002 },
  { 0.87091544744412508, 1.0000000000000000, 5.0000000000000000,
          10.000000000000000, -0.30000000000000004 },
  { 0.90977522877919847, 1.0000000000000000, 5.0000000000000000,
          10.000000000000000, -0.19999999999999996 },
  { 0.95257738192069130, 1.0000000000000000, 5.0000000000000000,
          10.000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 1.0000000000000000, 5.0000000000000000,
          10.000000000000000, 0.0000000000000000 },
  { 1.0528968282789379, 1.0000000000000000, 5.0000000000000000,
          10.000000000000000, 0.10000000000000009 },
  { 1.1123617169062123, 1.0000000000000000, 5.0000000000000000,
          10.000000000000000, 0.19999999999999996 },
  { 1.1798254572896132, 1.0000000000000000, 5.0000000000000000,
          10.000000000000000, 0.30000000000000004 },
  { 1.2572069000522696, 1.0000000000000000, 5.0000000000000000,
          10.000000000000000, 0.39999999999999991 },
  { 1.3471600884974377, 1.0000000000000000, 5.0000000000000000,
          10.000000000000000, 0.50000000000000000 },
  { 1.4535032279573519, 1.0000000000000000, 5.0000000000000000,
          10.000000000000000, 0.60000000000000009 },
  { 1.5820245752814948, 1.0000000000000000, 5.0000000000000000,
          10.000000000000000, 0.69999999999999996 },
  { 1.7421756366906538, 1.0000000000000000, 5.0000000000000000,
          10.000000000000000, 0.80000000000000004 },
  { 1.9513145531098233, 1.0000000000000000, 5.0000000000000000,
          10.000000000000000, 0.89999999999999991 },
};

// Test function for a=1.0000000000000000, b=5.0000000000000000, c=10.000000000000000.
template <typename Tp>
void test095()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data095)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data095[i].a), Tp(data095[i].b),
                   Tp(data095[i].c), Tp(data095[i].x));
      const Tp f0 = data095[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=1.0000000000000000, b=10.000000000000000, c=2.0000000000000000.
testcase_hyperg<double> data096[] = {
  { 0.12307420104127871, 1.0000000000000000, 10.000000000000000,
          2.0000000000000000, -0.90000000000000002 },
  { 0.13818870041457423, 1.0000000000000000, 10.000000000000000,
          2.0000000000000000, -0.80000000000000004 },
  { 0.15739165631811691, 1.0000000000000000, 10.000000000000000,
          2.0000000000000000, -0.69999999999999996 },
  { 0.18249038606882068, 1.0000000000000000, 10.000000000000000,
          2.0000000000000000, -0.59999999999999998 },
  { 0.21644171225027786, 1.0000000000000000, 10.000000000000000,
          2.0000000000000000, -0.50000000000000000 },
  { 0.26433326159804160, 1.0000000000000000, 10.000000000000000,
          2.0000000000000000, -0.40000000000000002 },
  { 0.33544459430654533, 1.0000000000000000, 10.000000000000000,
          2.0000000000000000, -0.30000000000000004 },
  { 0.44788516696232517, 1.0000000000000000, 10.000000000000000,
          2.0000000000000000, -0.19999999999999996 },
  { 0.63989153514168362, 1.0000000000000000, 10.000000000000000,
          2.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 1.0000000000000000, 10.000000000000000,
          2.0000000000000000, 0.0000000000000000 },
  { 1.7568608796813312, 1.0000000000000000, 10.000000000000000,
          2.0000000000000000, 0.10000000000000009 },
  { 3.5836558871799027, 1.0000000000000000, 10.000000000000000,
          2.0000000000000000, 0.19999999999999996 },
  { 8.8077526749963226, 1.0000000000000000, 10.000000000000000,
          2.0000000000000000, 0.30000000000000004 },
  { 27.285841702089190, 1.0000000000000000, 10.000000000000000,
          2.0000000000000000, 0.39999999999999991 },
  { 113.55555555555557, 1.0000000000000000, 10.000000000000000,
          2.0000000000000000, 0.50000000000000000 },
  { 706.24023437500091, 1.0000000000000000, 10.000000000000000,
          2.0000000000000000, 0.60000000000000009 },
  { 8064.1687976651992, 1.0000000000000000, 10.000000000000000,
          2.0000000000000000, 0.69999999999999996 },
  { 271267.22222222196, 1.0000000000000000, 10.000000000000000,
          2.0000000000000000, 0.80000000000000004 },
  { 123456789.99999890, 1.0000000000000000, 10.000000000000000,
          2.0000000000000000, 0.89999999999999991 },
};

// Test function for a=1.0000000000000000, b=10.000000000000000, c=2.0000000000000000.
template <typename Tp>
void test096()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data096)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data096[i].a), Tp(data096[i].b),
                   Tp(data096[i].c), Tp(data096[i].x));
      const Tp f0 = data096[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=1.0000000000000000, b=10.000000000000000, c=4.0000000000000000.
testcase_hyperg<double> data097[] = {
  { 0.28363728383055758, 1.0000000000000000, 10.000000000000000,
          4.0000000000000000, -0.90000000000000002 },
  { 0.30933003169808387, 1.0000000000000000, 10.000000000000000,
          4.0000000000000000, -0.80000000000000004 },
  { 0.33998437757128797, 1.0000000000000000, 10.000000000000000,
          4.0000000000000000, -0.69999999999999996 },
  { 0.37713553224291119, 1.0000000000000000, 10.000000000000000,
          4.0000000000000000, -0.59999999999999998 },
  { 0.42299736538419658, 1.0000000000000000, 10.000000000000000,
          4.0000000000000000, -0.50000000000000000 },
  { 0.48086597727600106, 1.0000000000000000, 10.000000000000000,
          4.0000000000000000, -0.40000000000000002 },
  { 0.55583495759293045, 1.0000000000000000, 10.000000000000000,
          4.0000000000000000, -0.30000000000000004 },
  { 0.65612850114039667, 1.0000000000000000, 10.000000000000000,
          4.0000000000000000, -0.19999999999999996 },
  { 0.79573668772968120, 1.0000000000000000, 10.000000000000000,
          4.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 1.0000000000000000, 10.000000000000000,
          4.0000000000000000, 0.0000000000000000 },
  { 1.3184712058058303, 1.0000000000000000, 10.000000000000000,
          4.0000000000000000, 0.10000000000000009 },
  { 1.8576958065941214, 1.0000000000000000, 10.000000000000000,
          4.0000000000000000, 0.19999999999999996 },
  { 2.8759509651764228, 1.0000000000000000, 10.000000000000000,
          4.0000000000000000, 0.30000000000000004 },
  { 5.1046225531822182, 1.0000000000000000, 10.000000000000000,
          4.0000000000000000, 0.39999999999999991 },
  { 11.095238095238095, 1.0000000000000000, 10.000000000000000,
          4.0000000000000000, 0.50000000000000000 },
  { 32.797154017857174, 1.0000000000000000, 10.000000000000000,
          4.0000000000000000, 0.60000000000000009 },
  { 158.01935680536477, 1.0000000000000000, 10.000000000000000,
          4.0000000000000000, 0.69999999999999996 },
  { 1815.9523809523814, 1.0000000000000000, 10.000000000000000,
          4.0000000000000000, 0.80000000000000004 },
  { 163302.14285714156, 1.0000000000000000, 10.000000000000000,
          4.0000000000000000, 0.89999999999999991 },
};

// Test function for a=1.0000000000000000, b=10.000000000000000, c=4.0000000000000000.
template <typename Tp>
void test097()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data097)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data097[i].a), Tp(data097[i].b),
                   Tp(data097[i].c), Tp(data097[i].x));
      const Tp f0 = data097[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=1.0000000000000000, b=10.000000000000000, c=6.0000000000000000.
testcase_hyperg<double> data098[] = {
  { 0.39006633302741794, 1.0000000000000000, 10.000000000000000,
          6.0000000000000000, -0.90000000000000002 },
  { 0.41898885698103294, 1.0000000000000000, 10.000000000000000,
          6.0000000000000000, -0.80000000000000004 },
  { 0.45245557983812590, 1.0000000000000000, 10.000000000000000,
          6.0000000000000000, -0.69999999999999996 },
  { 0.49160548618861633, 1.0000000000000000, 10.000000000000000,
          6.0000000000000000, -0.59999999999999998 },
  { 0.53798419230517980, 1.0000000000000000, 10.000000000000000,
          6.0000000000000000, -0.50000000000000000 },
  { 0.59373881442067344, 1.0000000000000000, 10.000000000000000,
          6.0000000000000000, -0.40000000000000002 },
  { 0.66193391357076115, 1.0000000000000000, 10.000000000000000,
          6.0000000000000000, -0.30000000000000004 },
  { 0.74708402736952129, 1.0000000000000000, 10.000000000000000,
          6.0000000000000000, -0.19999999999999996 },
  { 0.85609281019430605, 1.0000000000000000, 10.000000000000000,
          6.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 1.0000000000000000, 10.000000000000000,
          6.0000000000000000, 0.0000000000000000 },
  { 1.1974451135148187, 1.0000000000000000, 10.000000000000000,
          6.0000000000000000, 0.10000000000000009 },
  { 1.4820886036706347, 1.0000000000000000, 10.000000000000000,
          6.0000000000000000, 0.19999999999999996 },
  { 1.9201183180477521, 1.0000000000000000, 10.000000000000000,
          6.0000000000000000, 0.30000000000000004 },
  { 2.6569338297733336, 1.0000000000000000, 10.000000000000000,
          6.0000000000000000, 0.39999999999999991 },
  { 4.0634920634920650, 1.0000000000000000, 10.000000000000000,
          6.0000000000000000, 0.50000000000000000 },
  { 7.3102678571428568, 1.0000000000000000, 10.000000000000000,
          6.0000000000000000, 0.60000000000000009 },
  { 17.512574302697733, 1.0000000000000000, 10.000000000000000,
          6.0000000000000000, 0.69999999999999996 },
  { 74.206349206349131, 1.0000000000000000, 10.000000000000000,
          6.0000000000000000, 0.80000000000000004 },
  { 1342.8571428571363, 1.0000000000000000, 10.000000000000000,
          6.0000000000000000, 0.89999999999999991 },
};

// Test function for a=1.0000000000000000, b=10.000000000000000, c=6.0000000000000000.
template <typename Tp>
void test098()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data098)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data098[i].a), Tp(data098[i].b),
                   Tp(data098[i].c), Tp(data098[i].x));
      const Tp f0 = data098[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=1.0000000000000000, b=10.000000000000000, c=8.0000000000000000.
testcase_hyperg<double> data099[] = {
  { 0.46726928123633193, 1.0000000000000000, 10.000000000000000,
          8.0000000000000000, -0.90000000000000002 },
  { 0.49687547629934464, 1.0000000000000000, 10.000000000000000,
          8.0000000000000000, -0.80000000000000004 },
  { 0.53045208856322235, 1.0000000000000000, 10.000000000000000,
          8.0000000000000000, -0.69999999999999996 },
  { 0.56884765624999978, 1.0000000000000000, 10.000000000000000,
          8.0000000000000000, -0.59999999999999998 },
  { 0.61316872427983526, 1.0000000000000000, 10.000000000000000,
          8.0000000000000000, -0.50000000000000000 },
  { 0.66488500161969544, 1.0000000000000000, 10.000000000000000,
          8.0000000000000000, -0.40000000000000002 },
  { 0.72598998634501577, 1.0000000000000000, 10.000000000000000,
          8.0000000000000000, -0.30000000000000004 },
  { 0.79925411522633782, 1.0000000000000000, 10.000000000000000,
          8.0000000000000000, -0.19999999999999996 },
  { 0.88863845062192193, 1.0000000000000000, 10.000000000000000,
          8.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 1.0000000000000000, 10.000000000000000,
          8.0000000000000000, 0.0000000000000000 },
  { 1.1423563481176653, 1.0000000000000000, 10.000000000000000,
          8.0000000000000000, 0.10000000000000009 },
  { 1.3302951388888888, 1.0000000000000000, 10.000000000000000,
          8.0000000000000000, 0.19999999999999996 },
  { 1.5889212827988335, 1.0000000000000000, 10.000000000000000,
          8.0000000000000000, 0.30000000000000004 },
  { 1.9650205761316870, 1.0000000000000000, 10.000000000000000,
          8.0000000000000000, 0.39999999999999991 },
  { 2.5555555555555549, 1.0000000000000000, 10.000000000000000,
          8.0000000000000000, 0.50000000000000000 },
  { 3.5937500000000013, 1.0000000000000000, 10.000000000000000,
          8.0000000000000000, 0.60000000000000009 },
  { 5.7818930041152203, 1.0000000000000000, 10.000000000000000,
          8.0000000000000000, 0.69999999999999996 },
  { 12.222222222222220, 1.0000000000000000, 10.000000000000000,
          8.0000000000000000, 0.80000000000000004 },
  { 54.999999999999780, 1.0000000000000000, 10.000000000000000,
          8.0000000000000000, 0.89999999999999991 },
};

// Test function for a=1.0000000000000000, b=10.000000000000000, c=8.0000000000000000.
template <typename Tp>
void test099()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data099)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data099[i].a), Tp(data099[i].b),
                   Tp(data099[i].c), Tp(data099[i].x));
      const Tp f0 = data099[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=1.0000000000000000, b=10.000000000000000, c=10.000000000000000.
testcase_hyperg<double> data100[] = {
  { 0.52631578947368429, 1.0000000000000000, 10.000000000000000,
          10.000000000000000, -0.90000000000000002 },
  { 0.55555555555555558, 1.0000000000000000, 10.000000000000000,
          10.000000000000000, -0.80000000000000004 },
  { 0.58823529411764708, 1.0000000000000000, 10.000000000000000,
          10.000000000000000, -0.69999999999999996 },
  { 0.62500000000000000, 1.0000000000000000, 10.000000000000000,
          10.000000000000000, -0.59999999999999998 },
  { 0.66666666666666663, 1.0000000000000000, 10.000000000000000,
          10.000000000000000, -0.50000000000000000 },
  { 0.71428571428571430, 1.0000000000000000, 10.000000000000000,
          10.000000000000000, -0.40000000000000002 },
  { 0.76923076923076927, 1.0000000000000000, 10.000000000000000,
          10.000000000000000, -0.30000000000000004 },
  { 0.83333333333333337, 1.0000000000000000, 10.000000000000000,
          10.000000000000000, -0.19999999999999996 },
  { 0.90909090909090906, 1.0000000000000000, 10.000000000000000,
          10.000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 1.0000000000000000, 10.000000000000000,
          10.000000000000000, 0.0000000000000000 },
  { 1.1111111111111112, 1.0000000000000000, 10.000000000000000,
          10.000000000000000, 0.10000000000000009 },
  { 1.2500000000000000, 1.0000000000000000, 10.000000000000000,
          10.000000000000000, 0.19999999999999996 },
  { 1.4285714285714286, 1.0000000000000000, 10.000000000000000,
          10.000000000000000, 0.30000000000000004 },
  { 1.6666666666666663, 1.0000000000000000, 10.000000000000000,
          10.000000000000000, 0.39999999999999991 },
  { 2.0000000000000000, 1.0000000000000000, 10.000000000000000,
          10.000000000000000, 0.50000000000000000 },
  { 2.5000000000000004, 1.0000000000000000, 10.000000000000000,
          10.000000000000000, 0.60000000000000009 },
  { 3.3333333333333330, 1.0000000000000000, 10.000000000000000,
          10.000000000000000, 0.69999999999999996 },
  { 5.0000000000000009, 1.0000000000000000, 10.000000000000000,
          10.000000000000000, 0.80000000000000004 },
  { 9.9999999999999929, 1.0000000000000000, 10.000000000000000,
          10.000000000000000, 0.89999999999999991 },
};

// Test function for a=1.0000000000000000, b=10.000000000000000, c=10.000000000000000.
template <typename Tp>
void test100()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data100)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data100[i].a), Tp(data100[i].b),
                   Tp(data100[i].c), Tp(data100[i].x));
      const Tp f0 = data100[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=1.0000000000000000, b=20.000000000000000, c=2.0000000000000000.
testcase_hyperg<double> data101[] = {
  { 0.058479236576646373, 1.0000000000000000, 20.000000000000000,
          2.0000000000000000, -0.90000000000000002 },
  { 0.065788544763137669, 1.0000000000000000, 20.000000000000000,
          2.0000000000000000, -0.80000000000000004 },
  { 0.075184824937824662, 1.0000000000000000, 20.000000000000000,
          2.0000000000000000, -0.69999999999999996 },
  { 0.087707688693157260, 1.0000000000000000, 20.000000000000000,
          2.0000000000000000, -0.59999999999999998 },
  { 0.10521567442213345, 1.0000000000000000, 20.000000000000000,
          2.0000000000000000, -0.50000000000000000 },
  { 0.13135877960541525, 1.0000000000000000, 20.000000000000000,
          2.0000000000000000, -0.40000000000000002 },
  { 0.17423854066297104, 1.0000000000000000, 20.000000000000000,
          2.0000000000000000, -0.30000000000000004 },
  { 0.25492082527223525, 1.0000000000000000, 20.000000000000000,
          2.0000000000000000, -0.19999999999999996 },
  { 0.44025895219654843, 1.0000000000000000, 20.000000000000000,
          2.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 1.0000000000000000, 20.000000000000000,
          2.0000000000000000, 0.0000000000000000 },
  { 3.3698615820910360, 1.0000000000000000, 20.000000000000000,
          2.0000000000000000, 0.10000000000000009 },
  { 17.997089220808483, 1.0000000000000000, 20.000000000000000,
          2.0000000000000000, 0.19999999999999996 },
  { 153.73298291118951, 1.0000000000000000, 20.000000000000000,
          2.0000000000000000, 0.30000000000000004 },
  { 2159.1667587825627, 1.0000000000000000, 20.000000000000000,
          2.0000000000000000, 0.39999999999999991 },
  { 55188.105263157879, 1.0000000000000000, 20.000000000000000,
          2.0000000000000000, 0.50000000000000000 },
  { 3191209.3921857267, 1.0000000000000000, 20.000000000000000,
          2.0000000000000000, 0.60000000000000009 },
  { 646910975.29152656, 1.0000000000000000, 20.000000000000000,
          2.0000000000000000, 0.69999999999999996 },
  { 1254834626850.2659, 1.0000000000000000, 20.000000000000000,
          2.0000000000000000, 0.80000000000000004 },
  { 5.8479532163741414e+17, 1.0000000000000000, 20.000000000000000,
          2.0000000000000000, 0.89999999999999991 },
};

// Test function for a=1.0000000000000000, b=20.000000000000000, c=2.0000000000000000.
template <typename Tp>
void test101()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data101)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data101[i].a), Tp(data101[i].b),
                   Tp(data101[i].c), Tp(data101[i].x));
      const Tp f0 = data101[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=1.0000000000000000, b=20.000000000000000, c=4.0000000000000000.
testcase_hyperg<double> data102[] = {
  { 0.15519511120894947, 1.0000000000000000, 20.000000000000000,
          4.0000000000000000, -0.90000000000000002 },
  { 0.17197165701692899, 1.0000000000000000, 20.000000000000000,
          4.0000000000000000, -0.80000000000000004 },
  { 0.19276847315207363, 1.0000000000000000, 20.000000000000000,
          4.0000000000000000, -0.69999999999999996 },
  { 0.21920107206179093, 1.0000000000000000, 20.000000000000000,
          4.0000000000000000, -0.59999999999999998 },
  { 0.25386158960390581, 1.0000000000000000, 20.000000000000000,
          4.0000000000000000, -0.50000000000000000 },
  { 0.30115970686600657, 1.0000000000000000, 20.000000000000000,
          4.0000000000000000, -0.40000000000000002 },
  { 0.36916408142057117, 1.0000000000000000, 20.000000000000000,
          4.0000000000000000, -0.30000000000000004 },
  { 0.47406175901569558, 1.0000000000000000, 20.000000000000000,
          4.0000000000000000, -0.19999999999999996 },
  { 0.65237908266239919, 1.0000000000000000, 20.000000000000000,
          4.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 1.0000000000000000, 20.000000000000000,
          4.0000000000000000, 0.0000000000000000 },
  { 1.8227213362622299, 1.0000000000000000, 20.000000000000000,
          4.0000000000000000, 0.10000000000000009 },
  { 4.3716358339791332, 1.0000000000000000, 20.000000000000000,
          4.0000000000000000, 0.19999999999999996 },
  { 15.670841312959222, 1.0000000000000000, 20.000000000000000,
          4.0000000000000000, 0.30000000000000004 },
  { 94.742651122760179, 1.0000000000000000, 20.000000000000000,
          4.0000000000000000, 0.39999999999999991 },
  { 1081.7275541795671, 1.0000000000000000, 20.000000000000000,
          4.0000000000000000, 0.50000000000000000 },
  { 27809.787731465960, 1.0000000000000000, 20.000000000000000,
          4.0000000000000000, 0.60000000000000009 },
  { 2329811.1715181042, 1.0000000000000000, 20.000000000000000,
          4.0000000000000000, 0.69999999999999996 },
  { 1537787532.6780224, 1.0000000000000000, 20.000000000000000,
          4.0000000000000000, 0.80000000000000004 },
  { 141562653506999.88, 1.0000000000000000, 20.000000000000000,
          4.0000000000000000, 0.89999999999999991 },
};

// Test function for a=1.0000000000000000, b=20.000000000000000, c=4.0000000000000000.
template <typename Tp>
void test102()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data102)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data102[i].a), Tp(data102[i].b),
                   Tp(data102[i].c), Tp(data102[i].x));
      const Tp f0 = data102[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=1.0000000000000000, b=20.000000000000000, c=6.0000000000000000.
testcase_hyperg<double> data103[] = {
  { 0.23253645591196570, 1.0000000000000000, 20.000000000000000,
          6.0000000000000000, -0.90000000000000002 },
  { 0.25484220947068353, 1.0000000000000000, 20.000000000000000,
          6.0000000000000000, -0.80000000000000004 },
  { 0.28181987881113829, 1.0000000000000000, 20.000000000000000,
          6.0000000000000000, -0.69999999999999996 },
  { 0.31508211677735765, 1.0000000000000000, 20.000000000000000,
          6.0000000000000000, -0.59999999999999998 },
  { 0.35706285886959599, 1.0000000000000000, 20.000000000000000,
          6.0000000000000000, -0.50000000000000000 },
  { 0.41160053409238195, 1.0000000000000000, 20.000000000000000,
          6.0000000000000000, -0.40000000000000002 },
  { 0.48508083111181949, 1.0000000000000000, 20.000000000000000,
          6.0000000000000000, -0.30000000000000004 },
  { 0.58885194371375260, 1.0000000000000000, 20.000000000000000,
          6.0000000000000000, -0.19999999999999996 },
  { 0.74482241684585748, 1.0000000000000000, 20.000000000000000,
          6.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 1.0000000000000000, 20.000000000000000,
          6.0000000000000000, 0.0000000000000000 },
  { 1.4700356864367146, 1.0000000000000000, 20.000000000000000,
          6.0000000000000000, 0.10000000000000009 },
  { 2.4955144453055143, 1.0000000000000000, 20.000000000000000,
          6.0000000000000000, 0.19999999999999996 },
  { 5.3506594845833471, 1.0000000000000000, 20.000000000000000,
          6.0000000000000000, 0.30000000000000004 },
  { 16.618413752184221, 1.0000000000000000, 20.000000000000000,
          6.0000000000000000, 0.39999999999999991 },
  { 89.310629514963878, 1.0000000000000000, 20.000000000000000,
          6.0000000000000000, 0.50000000000000000 },
  { 1029.3439900542960, 1.0000000000000000, 20.000000000000000,
          6.0000000000000000, 0.60000000000000009 },
  { 35659.847863372350, 1.0000000000000000, 20.000000000000000,
          6.0000000000000000, 0.69999999999999996 },
  { 8009309.6233230168, 1.0000000000000000, 20.000000000000000,
          6.0000000000000000, 0.80000000000000004 },
  { 145640590027.39731, 1.0000000000000000, 20.000000000000000,
          6.0000000000000000, 0.89999999999999991 },
};

// Test function for a=1.0000000000000000, b=20.000000000000000, c=6.0000000000000000.
template <typename Tp>
void test103()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data103)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data103[i].a), Tp(data103[i].b),
                   Tp(data103[i].c), Tp(data103[i].x));
      const Tp f0 = data103[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=1.0000000000000000, b=20.000000000000000, c=8.0000000000000000.
testcase_hyperg<double> data104[] = {
  { 0.29614148314592498, 1.0000000000000000, 20.000000000000000,
          8.0000000000000000, -0.90000000000000002 },
  { 0.32176277356430810, 1.0000000000000000, 20.000000000000000,
          8.0000000000000000, -0.80000000000000004 },
  { 0.35217870475550522, 1.0000000000000000, 20.000000000000000,
          8.0000000000000000, -0.69999999999999996 },
  { 0.38885270445515091, 1.0000000000000000, 20.000000000000000,
          8.0000000000000000, -0.59999999999999998 },
  { 0.43389978380608424, 1.0000000000000000, 20.000000000000000,
          8.0000000000000000, -0.50000000000000000 },
  { 0.49048612522269436, 1.0000000000000000, 20.000000000000000,
          8.0000000000000000, -0.40000000000000002 },
  { 0.56355539635634611, 1.0000000000000000, 20.000000000000000,
          8.0000000000000000, -0.30000000000000004 },
  { 0.66123153239117682, 1.0000000000000000, 20.000000000000000,
          8.0000000000000000, -0.19999999999999996 },
  { 0.79773363961895427, 1.0000000000000000, 20.000000000000000,
          8.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 1.0000000000000000, 20.000000000000000,
          8.0000000000000000, 0.0000000000000000 },
  { 1.3245132157016595, 1.0000000000000000, 20.000000000000000,
          8.0000000000000000, 0.10000000000000009 },
  { 1.9065148749742076, 1.0000000000000000, 20.000000000000000,
          8.0000000000000000, 0.19999999999999996 },
  { 3.1328798652457452, 1.0000000000000000, 20.000000000000000,
          8.0000000000000000, 0.30000000000000004 },
  { 6.4172532944033476, 1.0000000000000000, 20.000000000000000,
          8.0000000000000000, 0.39999999999999991 },
  { 19.071683734222436, 1.0000000000000000, 20.000000000000000,
          8.0000000000000000, 0.50000000000000000 },
  { 104.41989641582512, 1.0000000000000000, 20.000000000000000,
          8.0000000000000000, 0.60000000000000009 },
  { 1510.5743992324240, 1.0000000000000000, 20.000000000000000,
          8.0000000000000000, 0.69999999999999996 },
  { 115518.14360562043, 1.0000000000000000, 20.000000000000000,
          8.0000000000000000, 0.80000000000000004 },
  { 414930455.29173034, 1.0000000000000000, 20.000000000000000,
          8.0000000000000000, 0.89999999999999991 },
};

// Test function for a=1.0000000000000000, b=20.000000000000000, c=8.0000000000000000.
template <typename Tp>
void test104()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data104)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data104[i].a), Tp(data104[i].b),
                   Tp(data104[i].c), Tp(data104[i].x));
      const Tp f0 = data104[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=1.0000000000000000, b=20.000000000000000, c=10.000000000000000.
testcase_hyperg<double> data105[] = {
  { 0.34954259539177696, 1.0000000000000000, 20.000000000000000,
          10.000000000000000, -0.90000000000000002 },
  { 0.37714038609235123, 1.0000000000000000, 20.000000000000000,
          10.000000000000000, -0.80000000000000004 },
  { 0.40942091659748725, 1.0000000000000000, 20.000000000000000,
          10.000000000000000, -0.69999999999999996 },
  { 0.44767109606846428, 1.0000000000000000, 20.000000000000000,
          10.000000000000000, -0.59999999999999998 },
  { 0.49368984777532254, 1.0000000000000000, 20.000000000000000,
          10.000000000000000, -0.50000000000000000 },
  { 0.55006638216982318, 1.0000000000000000, 20.000000000000000,
          10.000000000000000, -0.40000000000000002 },
  { 0.62065830207408912, 1.0000000000000000, 20.000000000000000,
          10.000000000000000, -0.30000000000000004 },
  { 0.71145554513583786, 1.0000000000000000, 20.000000000000000,
          10.000000000000000, -0.19999999999999996 },
  { 0.83223839666914623, 1.0000000000000000, 20.000000000000000,
          10.000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 1.0000000000000000, 20.000000000000000,
          10.000000000000000, 0.0000000000000000 },
  { 1.2466748028187731, 1.0000000000000000, 20.000000000000000,
          10.000000000000000, 0.10000000000000009 },
  { 1.6386752725021749, 1.0000000000000000, 20.000000000000000,
          10.000000000000000, 0.19999999999999996 },
  { 2.3340068725479681, 1.0000000000000000, 20.000000000000000,
          10.000000000000000, 0.30000000000000004 },
  { 3.7848108613132054, 1.0000000000000000, 20.000000000000000,
          10.000000000000000, 0.39999999999999991 },
  { 7.6754638550304133, 1.0000000000000000, 20.000000000000000,
          10.000000000000000, 0.50000000000000000 },
  { 23.344217312927277, 1.0000000000000000, 20.000000000000000,
          10.000000000000000, 0.60000000000000009 },
  { 149.83491198246921, 1.0000000000000000, 20.000000000000000,
          10.000000000000000, 0.69999999999999996 },
  { 3936.9253501916060, 1.0000000000000000, 20.000000000000000,
          10.000000000000000, 0.80000000000000004 },
  { 2794143.5036480185, 1.0000000000000000, 20.000000000000000,
          10.000000000000000, 0.89999999999999991 },
};

// Test function for a=1.0000000000000000, b=20.000000000000000, c=10.000000000000000.
template <typename Tp>
void test105()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data105)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data105[i].a), Tp(data105[i].b),
                   Tp(data105[i].c), Tp(data105[i].x));
      const Tp f0 = data105[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=2.0000000000000000, b=0.0000000000000000, c=2.0000000000000000.
testcase_hyperg<double> data106[] = {
  { 1.0000000000000000, 2.0000000000000000, 0.0000000000000000,
          2.0000000000000000, -0.90000000000000002 },
  { 1.0000000000000000, 2.0000000000000000, 0.0000000000000000,
          2.0000000000000000, -0.80000000000000004 },
  { 1.0000000000000000, 2.0000000000000000, 0.0000000000000000,
          2.0000000000000000, -0.69999999999999996 },
  { 1.0000000000000000, 2.0000000000000000, 0.0000000000000000,
          2.0000000000000000, -0.59999999999999998 },
  { 1.0000000000000000, 2.0000000000000000, 0.0000000000000000,
          2.0000000000000000, -0.50000000000000000 },
  { 1.0000000000000000, 2.0000000000000000, 0.0000000000000000,
          2.0000000000000000, -0.40000000000000002 },
  { 1.0000000000000000, 2.0000000000000000, 0.0000000000000000,
          2.0000000000000000, -0.30000000000000004 },
  { 1.0000000000000000, 2.0000000000000000, 0.0000000000000000,
          2.0000000000000000, -0.19999999999999996 },
  { 1.0000000000000000, 2.0000000000000000, 0.0000000000000000,
          2.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 2.0000000000000000, 0.0000000000000000,
          2.0000000000000000, 0.0000000000000000 },
  { 1.0000000000000000, 2.0000000000000000, 0.0000000000000000,
          2.0000000000000000, 0.10000000000000009 },
  { 1.0000000000000000, 2.0000000000000000, 0.0000000000000000,
          2.0000000000000000, 0.19999999999999996 },
  { 1.0000000000000000, 2.0000000000000000, 0.0000000000000000,
          2.0000000000000000, 0.30000000000000004 },
  { 1.0000000000000000, 2.0000000000000000, 0.0000000000000000,
          2.0000000000000000, 0.39999999999999991 },
  { 1.0000000000000000, 2.0000000000000000, 0.0000000000000000,
          2.0000000000000000, 0.50000000000000000 },
  { 1.0000000000000000, 2.0000000000000000, 0.0000000000000000,
          2.0000000000000000, 0.60000000000000009 },
  { 1.0000000000000000, 2.0000000000000000, 0.0000000000000000,
          2.0000000000000000, 0.69999999999999996 },
  { 1.0000000000000000, 2.0000000000000000, 0.0000000000000000,
          2.0000000000000000, 0.80000000000000004 },
  { 1.0000000000000000, 2.0000000000000000, 0.0000000000000000,
          2.0000000000000000, 0.89999999999999991 },
};

// Test function for a=2.0000000000000000, b=0.0000000000000000, c=2.0000000000000000.
template <typename Tp>
void test106()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data106)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data106[i].a), Tp(data106[i].b),
                   Tp(data106[i].c), Tp(data106[i].x));
      const Tp f0 = data106[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=2.0000000000000000, b=0.0000000000000000, c=4.0000000000000000.
testcase_hyperg<double> data107[] = {
  { 1.0000000000000000, 2.0000000000000000, 0.0000000000000000,
          4.0000000000000000, -0.90000000000000002 },
  { 1.0000000000000000, 2.0000000000000000, 0.0000000000000000,
          4.0000000000000000, -0.80000000000000004 },
  { 1.0000000000000000, 2.0000000000000000, 0.0000000000000000,
          4.0000000000000000, -0.69999999999999996 },
  { 1.0000000000000000, 2.0000000000000000, 0.0000000000000000,
          4.0000000000000000, -0.59999999999999998 },
  { 1.0000000000000000, 2.0000000000000000, 0.0000000000000000,
          4.0000000000000000, -0.50000000000000000 },
  { 1.0000000000000000, 2.0000000000000000, 0.0000000000000000,
          4.0000000000000000, -0.40000000000000002 },
  { 1.0000000000000000, 2.0000000000000000, 0.0000000000000000,
          4.0000000000000000, -0.30000000000000004 },
  { 1.0000000000000000, 2.0000000000000000, 0.0000000000000000,
          4.0000000000000000, -0.19999999999999996 },
  { 1.0000000000000000, 2.0000000000000000, 0.0000000000000000,
          4.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 2.0000000000000000, 0.0000000000000000,
          4.0000000000000000, 0.0000000000000000 },
  { 1.0000000000000000, 2.0000000000000000, 0.0000000000000000,
          4.0000000000000000, 0.10000000000000009 },
  { 1.0000000000000000, 2.0000000000000000, 0.0000000000000000,
          4.0000000000000000, 0.19999999999999996 },
  { 1.0000000000000000, 2.0000000000000000, 0.0000000000000000,
          4.0000000000000000, 0.30000000000000004 },
  { 1.0000000000000000, 2.0000000000000000, 0.0000000000000000,
          4.0000000000000000, 0.39999999999999991 },
  { 1.0000000000000000, 2.0000000000000000, 0.0000000000000000,
          4.0000000000000000, 0.50000000000000000 },
  { 1.0000000000000000, 2.0000000000000000, 0.0000000000000000,
          4.0000000000000000, 0.60000000000000009 },
  { 1.0000000000000000, 2.0000000000000000, 0.0000000000000000,
          4.0000000000000000, 0.69999999999999996 },
  { 1.0000000000000000, 2.0000000000000000, 0.0000000000000000,
          4.0000000000000000, 0.80000000000000004 },
  { 1.0000000000000000, 2.0000000000000000, 0.0000000000000000,
          4.0000000000000000, 0.89999999999999991 },
};

// Test function for a=2.0000000000000000, b=0.0000000000000000, c=4.0000000000000000.
template <typename Tp>
void test107()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data107)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data107[i].a), Tp(data107[i].b),
                   Tp(data107[i].c), Tp(data107[i].x));
      const Tp f0 = data107[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=2.0000000000000000, b=0.0000000000000000, c=6.0000000000000000.
testcase_hyperg<double> data108[] = {
  { 1.0000000000000000, 2.0000000000000000, 0.0000000000000000,
          6.0000000000000000, -0.90000000000000002 },
  { 1.0000000000000000, 2.0000000000000000, 0.0000000000000000,
          6.0000000000000000, -0.80000000000000004 },
  { 1.0000000000000000, 2.0000000000000000, 0.0000000000000000,
          6.0000000000000000, -0.69999999999999996 },
  { 1.0000000000000000, 2.0000000000000000, 0.0000000000000000,
          6.0000000000000000, -0.59999999999999998 },
  { 1.0000000000000000, 2.0000000000000000, 0.0000000000000000,
          6.0000000000000000, -0.50000000000000000 },
  { 1.0000000000000000, 2.0000000000000000, 0.0000000000000000,
          6.0000000000000000, -0.40000000000000002 },
  { 1.0000000000000000, 2.0000000000000000, 0.0000000000000000,
          6.0000000000000000, -0.30000000000000004 },
  { 1.0000000000000000, 2.0000000000000000, 0.0000000000000000,
          6.0000000000000000, -0.19999999999999996 },
  { 1.0000000000000000, 2.0000000000000000, 0.0000000000000000,
          6.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 2.0000000000000000, 0.0000000000000000,
          6.0000000000000000, 0.0000000000000000 },
  { 1.0000000000000000, 2.0000000000000000, 0.0000000000000000,
          6.0000000000000000, 0.10000000000000009 },
  { 1.0000000000000000, 2.0000000000000000, 0.0000000000000000,
          6.0000000000000000, 0.19999999999999996 },
  { 1.0000000000000000, 2.0000000000000000, 0.0000000000000000,
          6.0000000000000000, 0.30000000000000004 },
  { 1.0000000000000000, 2.0000000000000000, 0.0000000000000000,
          6.0000000000000000, 0.39999999999999991 },
  { 1.0000000000000000, 2.0000000000000000, 0.0000000000000000,
          6.0000000000000000, 0.50000000000000000 },
  { 1.0000000000000000, 2.0000000000000000, 0.0000000000000000,
          6.0000000000000000, 0.60000000000000009 },
  { 1.0000000000000000, 2.0000000000000000, 0.0000000000000000,
          6.0000000000000000, 0.69999999999999996 },
  { 1.0000000000000000, 2.0000000000000000, 0.0000000000000000,
          6.0000000000000000, 0.80000000000000004 },
  { 1.0000000000000000, 2.0000000000000000, 0.0000000000000000,
          6.0000000000000000, 0.89999999999999991 },
};

// Test function for a=2.0000000000000000, b=0.0000000000000000, c=6.0000000000000000.
template <typename Tp>
void test108()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data108)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data108[i].a), Tp(data108[i].b),
                   Tp(data108[i].c), Tp(data108[i].x));
      const Tp f0 = data108[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=2.0000000000000000, b=0.0000000000000000, c=8.0000000000000000.
testcase_hyperg<double> data109[] = {
  { 1.0000000000000000, 2.0000000000000000, 0.0000000000000000,
          8.0000000000000000, -0.90000000000000002 },
  { 1.0000000000000000, 2.0000000000000000, 0.0000000000000000,
          8.0000000000000000, -0.80000000000000004 },
  { 1.0000000000000000, 2.0000000000000000, 0.0000000000000000,
          8.0000000000000000, -0.69999999999999996 },
  { 1.0000000000000000, 2.0000000000000000, 0.0000000000000000,
          8.0000000000000000, -0.59999999999999998 },
  { 1.0000000000000000, 2.0000000000000000, 0.0000000000000000,
          8.0000000000000000, -0.50000000000000000 },
  { 1.0000000000000000, 2.0000000000000000, 0.0000000000000000,
          8.0000000000000000, -0.40000000000000002 },
  { 1.0000000000000000, 2.0000000000000000, 0.0000000000000000,
          8.0000000000000000, -0.30000000000000004 },
  { 1.0000000000000000, 2.0000000000000000, 0.0000000000000000,
          8.0000000000000000, -0.19999999999999996 },
  { 1.0000000000000000, 2.0000000000000000, 0.0000000000000000,
          8.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 2.0000000000000000, 0.0000000000000000,
          8.0000000000000000, 0.0000000000000000 },
  { 1.0000000000000000, 2.0000000000000000, 0.0000000000000000,
          8.0000000000000000, 0.10000000000000009 },
  { 1.0000000000000000, 2.0000000000000000, 0.0000000000000000,
          8.0000000000000000, 0.19999999999999996 },
  { 1.0000000000000000, 2.0000000000000000, 0.0000000000000000,
          8.0000000000000000, 0.30000000000000004 },
  { 1.0000000000000000, 2.0000000000000000, 0.0000000000000000,
          8.0000000000000000, 0.39999999999999991 },
  { 1.0000000000000000, 2.0000000000000000, 0.0000000000000000,
          8.0000000000000000, 0.50000000000000000 },
  { 1.0000000000000000, 2.0000000000000000, 0.0000000000000000,
          8.0000000000000000, 0.60000000000000009 },
  { 1.0000000000000000, 2.0000000000000000, 0.0000000000000000,
          8.0000000000000000, 0.69999999999999996 },
  { 1.0000000000000000, 2.0000000000000000, 0.0000000000000000,
          8.0000000000000000, 0.80000000000000004 },
  { 1.0000000000000000, 2.0000000000000000, 0.0000000000000000,
          8.0000000000000000, 0.89999999999999991 },
};

// Test function for a=2.0000000000000000, b=0.0000000000000000, c=8.0000000000000000.
template <typename Tp>
void test109()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data109)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data109[i].a), Tp(data109[i].b),
                   Tp(data109[i].c), Tp(data109[i].x));
      const Tp f0 = data109[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=2.0000000000000000, b=0.0000000000000000, c=10.000000000000000.
testcase_hyperg<double> data110[] = {
  { 1.0000000000000000, 2.0000000000000000, 0.0000000000000000,
          10.000000000000000, -0.90000000000000002 },
  { 1.0000000000000000, 2.0000000000000000, 0.0000000000000000,
          10.000000000000000, -0.80000000000000004 },
  { 1.0000000000000000, 2.0000000000000000, 0.0000000000000000,
          10.000000000000000, -0.69999999999999996 },
  { 1.0000000000000000, 2.0000000000000000, 0.0000000000000000,
          10.000000000000000, -0.59999999999999998 },
  { 1.0000000000000000, 2.0000000000000000, 0.0000000000000000,
          10.000000000000000, -0.50000000000000000 },
  { 1.0000000000000000, 2.0000000000000000, 0.0000000000000000,
          10.000000000000000, -0.40000000000000002 },
  { 1.0000000000000000, 2.0000000000000000, 0.0000000000000000,
          10.000000000000000, -0.30000000000000004 },
  { 1.0000000000000000, 2.0000000000000000, 0.0000000000000000,
          10.000000000000000, -0.19999999999999996 },
  { 1.0000000000000000, 2.0000000000000000, 0.0000000000000000,
          10.000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 2.0000000000000000, 0.0000000000000000,
          10.000000000000000, 0.0000000000000000 },
  { 1.0000000000000000, 2.0000000000000000, 0.0000000000000000,
          10.000000000000000, 0.10000000000000009 },
  { 1.0000000000000000, 2.0000000000000000, 0.0000000000000000,
          10.000000000000000, 0.19999999999999996 },
  { 1.0000000000000000, 2.0000000000000000, 0.0000000000000000,
          10.000000000000000, 0.30000000000000004 },
  { 1.0000000000000000, 2.0000000000000000, 0.0000000000000000,
          10.000000000000000, 0.39999999999999991 },
  { 1.0000000000000000, 2.0000000000000000, 0.0000000000000000,
          10.000000000000000, 0.50000000000000000 },
  { 1.0000000000000000, 2.0000000000000000, 0.0000000000000000,
          10.000000000000000, 0.60000000000000009 },
  { 1.0000000000000000, 2.0000000000000000, 0.0000000000000000,
          10.000000000000000, 0.69999999999999996 },
  { 1.0000000000000000, 2.0000000000000000, 0.0000000000000000,
          10.000000000000000, 0.80000000000000004 },
  { 1.0000000000000000, 2.0000000000000000, 0.0000000000000000,
          10.000000000000000, 0.89999999999999991 },
};

// Test function for a=2.0000000000000000, b=0.0000000000000000, c=10.000000000000000.
template <typename Tp>
void test110()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data110)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data110[i].a), Tp(data110[i].b),
                   Tp(data110[i].c), Tp(data110[i].x));
      const Tp f0 = data110[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=2.0000000000000000, b=0.50000000000000000, c=2.0000000000000000.
testcase_hyperg<double> data111[] = {
  { 0.72547625011001171, 2.0000000000000000, 0.50000000000000000,
          2.0000000000000000, -0.90000000000000002 },
  { 0.74535599249992990, 2.0000000000000000, 0.50000000000000000,
          2.0000000000000000, -0.80000000000000004 },
  { 0.76696498884737041, 2.0000000000000000, 0.50000000000000000,
          2.0000000000000000, -0.69999999999999996 },
  { 0.79056941504209477, 2.0000000000000000, 0.50000000000000000,
          2.0000000000000000, -0.59999999999999998 },
  { 0.81649658092772603, 2.0000000000000000, 0.50000000000000000,
          2.0000000000000000, -0.50000000000000000 },
  { 0.84515425472851657, 2.0000000000000000, 0.50000000000000000,
          2.0000000000000000, -0.40000000000000002 },
  { 0.87705801930702920, 2.0000000000000000, 0.50000000000000000,
          2.0000000000000000, -0.30000000000000004 },
  { 0.91287092917527690, 2.0000000000000000, 0.50000000000000000,
          2.0000000000000000, -0.19999999999999996 },
  { 0.95346258924559224, 2.0000000000000000, 0.50000000000000000,
          2.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 2.0000000000000000, 0.50000000000000000,
          2.0000000000000000, 0.0000000000000000 },
  { 1.0540925533894598, 2.0000000000000000, 0.50000000000000000,
          2.0000000000000000, 0.10000000000000009 },
  { 1.1180339887498949, 2.0000000000000000, 0.50000000000000000,
          2.0000000000000000, 0.19999999999999996 },
  { 1.1952286093343938, 2.0000000000000000, 0.50000000000000000,
          2.0000000000000000, 0.30000000000000004 },
  { 1.2909944487358056, 2.0000000000000000, 0.50000000000000000,
          2.0000000000000000, 0.39999999999999991 },
  { 1.4142135623730949, 2.0000000000000000, 0.50000000000000000,
          2.0000000000000000, 0.50000000000000000 },
  { 1.5811388300841900, 2.0000000000000000, 0.50000000000000000,
          2.0000000000000000, 0.60000000000000009 },
  { 1.8257418583505536, 2.0000000000000000, 0.50000000000000000,
          2.0000000000000000, 0.69999999999999996 },
  { 2.2360679774997898, 2.0000000000000000, 0.50000000000000000,
          2.0000000000000000, 0.80000000000000004 },
  { 3.1622776601683782, 2.0000000000000000, 0.50000000000000000,
          2.0000000000000000, 0.89999999999999991 },
};

// Test function for a=2.0000000000000000, b=0.50000000000000000, c=2.0000000000000000.
template <typename Tp>
void test111()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data111)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data111[i].a), Tp(data111[i].b),
                   Tp(data111[i].c), Tp(data111[i].x));
      const Tp f0 = data111[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=2.0000000000000000, b=0.50000000000000000, c=4.0000000000000000.
testcase_hyperg<double> data112[] = {
  { 0.83664260086443798, 2.0000000000000000, 0.50000000000000000,
          4.0000000000000000, -0.90000000000000002 },
  { 0.85046584300227146, 2.0000000000000000, 0.50000000000000000,
          4.0000000000000000, -0.80000000000000004 },
  { 0.86509574979651593, 2.0000000000000000, 0.50000000000000000,
          4.0000000000000000, -0.69999999999999996 },
  { 0.88062082573041867, 2.0000000000000000, 0.50000000000000000,
          4.0000000000000000, -0.59999999999999998 },
  { 0.89714464248521586, 2.0000000000000000, 0.50000000000000000,
          4.0000000000000000, -0.50000000000000000 },
  { 0.91478946588967558, 2.0000000000000000, 0.50000000000000000,
          4.0000000000000000, -0.40000000000000002 },
  { 0.93370105322348573, 2.0000000000000000, 0.50000000000000000,
          4.0000000000000000, -0.30000000000000004 },
  { 0.95405511057700887, 2.0000000000000000, 0.50000000000000000,
          4.0000000000000000, -0.19999999999999996 },
  { 0.97606616007978142, 2.0000000000000000, 0.50000000000000000,
          4.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 2.0000000000000000, 0.50000000000000000,
          4.0000000000000000, 0.0000000000000000 },
  { 1.0261916902334731, 2.0000000000000000, 0.50000000000000000,
          4.0000000000000000, 0.10000000000000009 },
  { 1.0550723519434702, 2.0000000000000000, 0.50000000000000000,
          4.0000000000000000, 0.19999999999999996 },
  { 1.0872106588188091, 2.0000000000000000, 0.50000000000000000,
          4.0000000000000000, 0.30000000000000004 },
  { 1.1233801699379020, 2.0000000000000000, 0.50000000000000000,
          4.0000000000000000, 0.39999999999999991 },
  { 1.1646752981725688, 2.0000000000000000, 0.50000000000000000,
          4.0000000000000000, 0.50000000000000000 },
  { 1.2127272514219511, 2.0000000000000000, 0.50000000000000000,
          4.0000000000000000, 0.60000000000000009 },
  { 1.2701518651068637, 2.0000000000000000, 0.50000000000000000,
          4.0000000000000000, 0.69999999999999996 },
  { 1.3416407864998725, 2.0000000000000000, 0.50000000000000000,
          4.0000000000000000, 0.80000000000000004 },
  { 1.4374795179111102, 2.0000000000000000, 0.50000000000000000,
          4.0000000000000000, 0.89999999999999991 },
};

// Test function for a=2.0000000000000000, b=0.50000000000000000, c=4.0000000000000000.
template <typename Tp>
void test112()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data112)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data112[i].a), Tp(data112[i].b),
                   Tp(data112[i].c), Tp(data112[i].x));
      const Tp f0 = data112[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=2.0000000000000000, b=0.50000000000000000, c=6.0000000000000000.
testcase_hyperg<double> data113[] = {
  { 0.88195381730235833, 2.0000000000000000, 0.50000000000000000,
          6.0000000000000000, -0.90000000000000002 },
  { 0.89265078469555093, 2.0000000000000000, 0.50000000000000000,
          6.0000000000000000, -0.80000000000000004 },
  { 0.90382937908303707, 2.0000000000000000, 0.50000000000000000,
          6.0000000000000000, -0.69999999999999996 },
  { 0.91553161389880577, 2.0000000000000000, 0.50000000000000000,
          6.0000000000000000, -0.59999999999999998 },
  { 0.92780530349281576, 2.0000000000000000, 0.50000000000000000,
          6.0000000000000000, -0.50000000000000000 },
  { 0.94070521140346042, 2.0000000000000000, 0.50000000000000000,
          6.0000000000000000, -0.40000000000000002 },
  { 0.95429450630523349, 2.0000000000000000, 0.50000000000000000,
          6.0000000000000000, -0.30000000000000004 },
  { 0.96864663325785849, 2.0000000000000000, 0.50000000000000000,
          6.0000000000000000, -0.19999999999999996 },
  { 0.98384775588541795, 2.0000000000000000, 0.50000000000000000,
          6.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 2.0000000000000000, 0.50000000000000000,
          6.0000000000000000, 0.0000000000000000 },
  { 1.0172258496884334, 2.0000000000000000, 0.50000000000000000,
          6.0000000000000000, 0.10000000000000009 },
  { 1.0356742479163459, 2.0000000000000000, 0.50000000000000000,
          6.0000000000000000, 0.19999999999999996 },
  { 1.0555293036908924, 2.0000000000000000, 0.50000000000000000,
          6.0000000000000000, 0.30000000000000004 },
  { 1.0770231491562379, 2.0000000000000000, 0.50000000000000000,
          6.0000000000000000, 0.39999999999999991 },
  { 1.1004557416484888, 2.0000000000000000, 0.50000000000000000,
          6.0000000000000000, 0.50000000000000000 },
  { 1.1262270515731978, 2.0000000000000000, 0.50000000000000000,
          6.0000000000000000, 0.60000000000000009 },
  { 1.1548932919125086, 2.0000000000000000, 0.50000000000000000,
          6.0000000000000000, 0.69999999999999996 },
  { 1.1872757758134724, 2.0000000000000000, 0.50000000000000000,
          6.0000000000000000, 0.80000000000000004 },
  { 1.2247091713458949, 2.0000000000000000, 0.50000000000000000,
          6.0000000000000000, 0.89999999999999991 },
};

// Test function for a=2.0000000000000000, b=0.50000000000000000, c=6.0000000000000000.
template <typename Tp>
void test113()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data113)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data113[i].a), Tp(data113[i].b),
                   Tp(data113[i].c), Tp(data113[i].x));
      const Tp f0 = data113[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=2.0000000000000000, b=0.50000000000000000, c=8.0000000000000000.
testcase_hyperg<double> data114[] = {
  { 0.90716919697107301, 2.0000000000000000, 0.50000000000000000,
          8.0000000000000000, -0.90000000000000002 },
  { 0.91592299407142519, 2.0000000000000000, 0.50000000000000000,
          8.0000000000000000, -0.80000000000000004 },
  { 0.92500027075874236, 2.0000000000000000, 0.50000000000000000,
          8.0000000000000000, -0.69999999999999996 },
  { 0.93442464185467167, 2.0000000000000000, 0.50000000000000000,
          8.0000000000000000, -0.59999999999999998 },
  { 0.94422248683737009, 2.0000000000000000, 0.50000000000000000,
          8.0000000000000000, -0.50000000000000000 },
  { 0.95442341810133347, 2.0000000000000000, 0.50000000000000000,
          8.0000000000000000, -0.40000000000000002 },
  { 0.96506085725516400, 2.0000000000000000, 0.50000000000000000,
          8.0000000000000000, -0.30000000000000004 },
  { 0.97617275213704069, 2.0000000000000000, 0.50000000000000000,
          8.0000000000000000, -0.19999999999999996 },
  { 0.98780247986309799, 2.0000000000000000, 0.50000000000000000,
          8.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 2.0000000000000000, 0.50000000000000000,
          8.0000000000000000, 0.0000000000000000 },
  { 1.0128233505813447, 2.0000000000000000, 0.50000000000000000,
          8.0000000000000000, 0.10000000000000009 },
  { 1.0263406246541855, 2.0000000000000000, 0.50000000000000000,
          8.0000000000000000, 0.19999999999999996 },
  { 1.0406326381700366, 2.0000000000000000, 0.50000000000000000,
          8.0000000000000000, 0.30000000000000004 },
  { 1.0557966239802845, 2.0000000000000000, 0.50000000000000000,
          8.0000000000000000, 0.39999999999999991 },
  { 1.0719515075786321, 2.0000000000000000, 0.50000000000000000,
          8.0000000000000000, 0.50000000000000000 },
  { 1.0892457392422055, 2.0000000000000000, 0.50000000000000000,
          8.0000000000000000, 0.60000000000000009 },
  { 1.1078695188000958, 2.0000000000000000, 0.50000000000000000,
          8.0000000000000000, 0.69999999999999996 },
  { 1.1280752258974340, 2.0000000000000000, 0.50000000000000000,
          8.0000000000000000, 0.80000000000000004 },
  { 1.1502152002706476, 2.0000000000000000, 0.50000000000000000,
          8.0000000000000000, 0.89999999999999991 },
};

// Test function for a=2.0000000000000000, b=0.50000000000000000, c=8.0000000000000000.
template <typename Tp>
void test114()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data114)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data114[i].a), Tp(data114[i].b),
                   Tp(data114[i].c), Tp(data114[i].x));
      const Tp f0 = data114[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=2.0000000000000000, b=0.50000000000000000, c=10.000000000000000.
testcase_hyperg<double> data115[] = {
  { 0.92336416053263093, 2.0000000000000000, 0.50000000000000000,
          10.000000000000000, -0.90000000000000002 },
  { 0.93078397248364519, 2.0000000000000000, 0.50000000000000000,
          10.000000000000000, -0.80000000000000004 },
  { 0.93843714333600226, 2.0000000000000000, 0.50000000000000000,
          10.000000000000000, -0.69999999999999996 },
  { 0.94633837784068076, 2.0000000000000000, 0.50000000000000000,
          10.000000000000000, -0.59999999999999998 },
  { 0.95450388104967909, 2.0000000000000000, 0.50000000000000000,
          10.000000000000000, -0.50000000000000000 },
  { 0.96295158125742764, 2.0000000000000000, 0.50000000000000000,
          10.000000000000000, -0.40000000000000002 },
  { 0.97170139827854296, 2.0000000000000000, 0.50000000000000000,
          10.000000000000000, -0.30000000000000004 },
  { 0.98077556918512687, 2.0000000000000000, 0.50000000000000000,
          10.000000000000000, -0.19999999999999996 },
  { 0.99019904777750845, 2.0000000000000000, 0.50000000000000000,
          10.000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 2.0000000000000000, 0.50000000000000000,
          10.000000000000000, 0.0000000000000000 },
  { 1.0102104261941198, 2.0000000000000000, 0.50000000000000000,
          10.000000000000000, 0.10000000000000009 },
  { 1.0208669540935695, 2.0000000000000000, 0.50000000000000000,
          10.000000000000000, 0.19999999999999996 },
  { 1.0320118665407505, 2.0000000000000000, 0.50000000000000000,
          10.000000000000000, 0.30000000000000004 },
  { 1.0436944599504387, 2.0000000000000000, 0.50000000000000000,
          10.000000000000000, 0.39999999999999991 },
  { 1.0559728828278145, 2.0000000000000000, 0.50000000000000000,
          10.000000000000000, 0.50000000000000000 },
  { 1.0689166967761712, 2.0000000000000000, 0.50000000000000000,
          10.000000000000000, 0.60000000000000009 },
  { 1.0826105758119842, 2.0000000000000000, 0.50000000000000000,
          10.000000000000000, 0.69999999999999996 },
  { 1.0971599106346146, 2.0000000000000000, 0.50000000000000000,
          10.000000000000000, 0.80000000000000004 },
  { 1.1126998828023964, 2.0000000000000000, 0.50000000000000000,
          10.000000000000000, 0.89999999999999991 },
};

// Test function for a=2.0000000000000000, b=0.50000000000000000, c=10.000000000000000.
template <typename Tp>
void test115()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data115)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data115[i].a), Tp(data115[i].b),
                   Tp(data115[i].c), Tp(data115[i].x));
      const Tp f0 = data115[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=2.0000000000000000, b=1.0000000000000000, c=2.0000000000000000.
testcase_hyperg<double> data116[] = {
  { 0.52631578947368429, 2.0000000000000000, 1.0000000000000000,
          2.0000000000000000, -0.90000000000000002 },
  { 0.55555555555555558, 2.0000000000000000, 1.0000000000000000,
          2.0000000000000000, -0.80000000000000004 },
  { 0.58823529411764708, 2.0000000000000000, 1.0000000000000000,
          2.0000000000000000, -0.69999999999999996 },
  { 0.62500000000000000, 2.0000000000000000, 1.0000000000000000,
          2.0000000000000000, -0.59999999999999998 },
  { 0.66666666666666663, 2.0000000000000000, 1.0000000000000000,
          2.0000000000000000, -0.50000000000000000 },
  { 0.71428571428571430, 2.0000000000000000, 1.0000000000000000,
          2.0000000000000000, -0.40000000000000002 },
  { 0.76923076923076927, 2.0000000000000000, 1.0000000000000000,
          2.0000000000000000, -0.30000000000000004 },
  { 0.83333333333333337, 2.0000000000000000, 1.0000000000000000,
          2.0000000000000000, -0.19999999999999996 },
  { 0.90909090909090906, 2.0000000000000000, 1.0000000000000000,
          2.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 2.0000000000000000, 1.0000000000000000,
          2.0000000000000000, 0.0000000000000000 },
  { 1.1111111111111112, 2.0000000000000000, 1.0000000000000000,
          2.0000000000000000, 0.10000000000000009 },
  { 1.2500000000000000, 2.0000000000000000, 1.0000000000000000,
          2.0000000000000000, 0.19999999999999996 },
  { 1.4285714285714286, 2.0000000000000000, 1.0000000000000000,
          2.0000000000000000, 0.30000000000000004 },
  { 1.6666666666666663, 2.0000000000000000, 1.0000000000000000,
          2.0000000000000000, 0.39999999999999991 },
  { 2.0000000000000000, 2.0000000000000000, 1.0000000000000000,
          2.0000000000000000, 0.50000000000000000 },
  { 2.5000000000000004, 2.0000000000000000, 1.0000000000000000,
          2.0000000000000000, 0.60000000000000009 },
  { 3.3333333333333330, 2.0000000000000000, 1.0000000000000000,
          2.0000000000000000, 0.69999999999999996 },
  { 5.0000000000000009, 2.0000000000000000, 1.0000000000000000,
          2.0000000000000000, 0.80000000000000004 },
  { 9.9999999999999929, 2.0000000000000000, 1.0000000000000000,
          2.0000000000000000, 0.89999999999999991 },
};

// Test function for a=2.0000000000000000, b=1.0000000000000000, c=2.0000000000000000.
template <typename Tp>
void test116()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data116)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data116[i].a), Tp(data116[i].b),
                   Tp(data116[i].c), Tp(data116[i].x));
      const Tp f0 = data116[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=2.0000000000000000, b=1.0000000000000000, c=4.0000000000000000.
testcase_hyperg<double> data117[] = {
  { 0.70351947549341520, 2.0000000000000000, 1.0000000000000000,
          4.0000000000000000, -0.90000000000000002 },
  { 0.72637503722092711, 2.0000000000000000, 1.0000000000000000,
          4.0000000000000000, -0.80000000000000004 },
  { 0.75099661564391251, 2.0000000000000000, 1.0000000000000000,
          4.0000000000000000, -0.69999999999999996 },
  { 0.77761647796730871, 2.0000000000000000, 1.0000000000000000,
          4.0000000000000000, -0.59999999999999998 },
  { 0.80651221621216451, 2.0000000000000000, 1.0000000000000000,
          4.0000000000000000, -0.50000000000000000 },
  { 0.83801894346580275, 2.0000000000000000, 1.0000000000000000,
          4.0000000000000000, -0.40000000000000002 },
  { 0.87254582050258467, 2.0000000000000000, 1.0000000000000000,
          4.0000000000000000, -0.30000000000000004 },
  { 0.91059888544083678, 2.0000000000000000, 1.0000000000000000,
          4.0000000000000000, -0.19999999999999996 },
  { 0.95281329145592386, 2.0000000000000000, 1.0000000000000000,
          4.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 2.0000000000000000, 1.0000000000000000,
          4.0000000000000000, 0.0000000000000000 },
  { 1.0532154477379738, 2.0000000000000000, 1.0000000000000000,
          4.0000000000000000, 0.10000000000000009 },
  { 1.1138692114741471, 2.0000000000000000, 1.0000000000000000,
          4.0000000000000000, 0.19999999999999996 },
  { 1.1838976095305187, 2.0000000000000000, 1.0000000000000000,
          4.0000000000000000, 0.30000000000000004 },
  { 1.2660586631630237, 2.0000000000000000, 1.0000000000000000,
          4.0000000000000000, 0.39999999999999991 },
  { 1.3644676665613118, 2.0000000000000000, 1.0000000000000000,
          4.0000000000000000, 0.50000000000000000 },
  { 1.4856585347316102, 2.0000000000000000, 1.0000000000000000,
          4.0000000000000000, 0.60000000000000009 },
  { 1.6409590443536872, 2.0000000000000000, 1.0000000000000000,
          4.0000000000000000, 0.69999999999999996 },
  { 1.8528798927325769, 2.0000000000000000, 1.0000000000000000,
          4.0000000000000000, 0.80000000000000004 },
  { 2.1789423102929644, 2.0000000000000000, 1.0000000000000000,
          4.0000000000000000, 0.89999999999999991 },
};

// Test function for a=2.0000000000000000, b=1.0000000000000000, c=4.0000000000000000.
template <typename Tp>
void test117()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data117)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data117[i].a), Tp(data117[i].b),
                   Tp(data117[i].c), Tp(data117[i].x));
      const Tp f0 = data117[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=2.0000000000000000, b=1.0000000000000000, c=6.0000000000000000.
testcase_hyperg<double> data118[] = {
  { 0.78068027379106253, 2.0000000000000000, 1.0000000000000000,
          6.0000000000000000, -0.90000000000000002 },
  { 0.79924541976981278, 2.0000000000000000, 1.0000000000000000,
          6.0000000000000000, -0.80000000000000004 },
  { 0.81891305585650942, 2.0000000000000000, 1.0000000000000000,
          6.0000000000000000, -0.69999999999999996 },
  { 0.83979799626213270, 2.0000000000000000, 1.0000000000000000,
          6.0000000000000000, -0.59999999999999998 },
  { 0.86203315303160166, 2.0000000000000000, 1.0000000000000000,
          6.0000000000000000, -0.50000000000000000 },
  { 0.88577352485361693, 2.0000000000000000, 1.0000000000000000,
          6.0000000000000000, -0.40000000000000002 },
  { 0.91120135738402208, 2.0000000000000000, 1.0000000000000000,
          6.0000000000000000, -0.30000000000000004 },
  { 0.93853291956703588, 2.0000000000000000, 1.0000000000000000,
          6.0000000000000000, -0.19999999999999996 },
  { 0.96802755388922956, 2.0000000000000000, 1.0000000000000000,
          6.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 2.0000000000000000, 1.0000000000000000,
          6.0000000000000000, 0.0000000000000000 },
  { 1.0348375559194773, 2.0000000000000000, 1.0000000000000000,
          6.0000000000000000, 0.10000000000000009 },
  { 1.0730246119544820, 2.0000000000000000, 1.0000000000000000,
          6.0000000000000000, 0.19999999999999996 },
  { 1.1151788396279341, 2.0000000000000000, 1.0000000000000000,
          6.0000000000000000, 0.30000000000000004 },
  { 1.1621066403893472, 2.0000000000000000, 1.0000000000000000,
          6.0000000000000000, 0.39999999999999991 },
  { 1.2148922218710421, 2.0000000000000000, 1.0000000000000000,
          6.0000000000000000, 0.50000000000000000 },
  { 1.2750496810838674, 2.0000000000000000, 1.0000000000000000,
          6.0000000000000000, 0.60000000000000009 },
  { 1.3448048570872917, 2.0000000000000000, 1.0000000000000000,
          6.0000000000000000, 0.69999999999999996 },
  { 1.4276833109859521, 2.0000000000000000, 1.0000000000000000,
          6.0000000000000000, 0.80000000000000004 },
  { 1.5299976259379788, 2.0000000000000000, 1.0000000000000000,
          6.0000000000000000, 0.89999999999999991 },
};

// Test function for a=2.0000000000000000, b=1.0000000000000000, c=6.0000000000000000.
template <typename Tp>
void test118()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data118)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data118[i].a), Tp(data118[i].b),
                   Tp(data118[i].c), Tp(data118[i].x));
      const Tp f0 = data118[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=2.0000000000000000, b=1.0000000000000000, c=8.0000000000000000.
testcase_hyperg<double> data119[] = {
  { 0.82510759951857582, 2.0000000000000000, 1.0000000000000000,
          8.0000000000000000, -0.90000000000000002 },
  { 0.84072786892782092, 2.0000000000000000, 1.0000000000000000,
          8.0000000000000000, -0.80000000000000004 },
  { 0.85710884896562356, 2.0000000000000000, 1.0000000000000000,
          8.0000000000000000, -0.69999999999999996 },
  { 0.87431674418118333, 2.0000000000000000, 1.0000000000000000,
          8.0000000000000000, -0.59999999999999998 },
  { 0.89242659229727017, 2.0000000000000000, 1.0000000000000000,
          8.0000000000000000, -0.50000000000000000 },
  { 0.91152392685930339, 2.0000000000000000, 1.0000000000000000,
          8.0000000000000000, -0.40000000000000002 },
  { 0.93170685950993570, 2.0000000000000000, 1.0000000000000000,
          8.0000000000000000, -0.30000000000000004 },
  { 0.95308871926790661, 2.0000000000000000, 1.0000000000000000,
          8.0000000000000000, -0.19999999999999996 },
  { 0.97580144325325802, 2.0000000000000000, 1.0000000000000000,
          8.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 2.0000000000000000, 1.0000000000000000,
          8.0000000000000000, 0.0000000000000000 },
  { 1.0258682619030324, 2.0000000000000000, 1.0000000000000000,
          8.0000000000000000, 0.10000000000000009 },
  { 1.0536269616706000, 2.0000000000000000, 1.0000000000000000,
          8.0000000000000000, 0.19999999999999996 },
  { 1.0835447330793833, 2.0000000000000000, 1.0000000000000000,
          8.0000000000000000, 0.30000000000000004 },
  { 1.1159538758396654, 2.0000000000000000, 1.0000000000000000,
          8.0000000000000000, 0.39999999999999991 },
  { 1.1512736659291880, 2.0000000000000000, 1.0000000000000000,
          8.0000000000000000, 0.50000000000000000 },
  { 1.1900463690116090, 2.0000000000000000, 1.0000000000000000,
          8.0000000000000000, 0.60000000000000009 },
  { 1.2329961591622411, 2.0000000000000000, 1.0000000000000000,
          8.0000000000000000, 0.69999999999999996 },
  { 1.2811334345669059, 2.0000000000000000, 1.0000000000000000,
          8.0000000000000000, 0.80000000000000004 },
  { 1.3359629014132051, 2.0000000000000000, 1.0000000000000000,
          8.0000000000000000, 0.89999999999999991 },
};

// Test function for a=2.0000000000000000, b=1.0000000000000000, c=8.0000000000000000.
template <typename Tp>
void test119()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data119)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data119[i].a), Tp(data119[i].b),
                   Tp(data119[i].c), Tp(data119[i].x));
      const Tp f0 = data119[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=2.0000000000000000, b=1.0000000000000000, c=10.000000000000000.
testcase_hyperg<double> data120[] = {
  { 0.85426123653345876, 2.0000000000000000, 1.0000000000000000,
          10.000000000000000, -0.90000000000000002 },
  { 0.86774543390930370, 2.0000000000000000, 1.0000000000000000,
          10.000000000000000, -0.80000000000000004 },
  { 0.88178859537254239, 2.0000000000000000, 1.0000000000000000,
          10.000000000000000, -0.69999999999999996 },
  { 0.89643269097060951, 2.0000000000000000, 1.0000000000000000,
          10.000000000000000, -0.59999999999999998 },
  { 0.91172456687216763, 2.0000000000000000, 1.0000000000000000,
          10.000000000000000, -0.50000000000000000 },
  { 0.92771674975966134, 2.0000000000000000, 1.0000000000000000,
          10.000000000000000, -0.40000000000000002 },
  { 0.94446842993888647, 2.0000000000000000, 1.0000000000000000,
          10.000000000000000, -0.30000000000000004 },
  { 0.96204667481937678, 2.0000000000000000, 1.0000000000000000,
          10.000000000000000, -0.19999999999999996 },
  { 0.98052794339012128, 2.0000000000000000, 1.0000000000000000,
          10.000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 2.0000000000000000, 1.0000000000000000,
          10.000000000000000, 0.0000000000000000 },
  { 1.0205643671068179, 2.0000000000000000, 1.0000000000000000,
          10.000000000000000, 0.10000000000000009 },
  { 1.0423395201078882, 2.0000000000000000, 1.0000000000000000,
          10.000000000000000, 0.19999999999999996 },
  { 1.0654651277885334, 2.0000000000000000, 1.0000000000000000,
          10.000000000000000, 0.30000000000000004 },
  { 1.0901078068101382, 2.0000000000000000, 1.0000000000000000,
          10.000000000000000, 0.39999999999999991 },
  { 1.1164691415928940, 2.0000000000000000, 1.0000000000000000,
          10.000000000000000, 0.50000000000000000 },
  { 1.1447972335326551, 2.0000000000000000, 1.0000000000000000,
          10.000000000000000, 0.60000000000000009 },
  { 1.1754040384534161, 2.0000000000000000, 1.0000000000000000,
          10.000000000000000, 0.69999999999999996 },
  { 1.2086928679893112, 2.0000000000000000, 1.0000000000000000,
          10.000000000000000, 0.80000000000000004 },
  { 1.2452055640510711, 2.0000000000000000, 1.0000000000000000,
          10.000000000000000, 0.89999999999999991 },
};

// Test function for a=2.0000000000000000, b=1.0000000000000000, c=10.000000000000000.
template <typename Tp>
void test120()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data120)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data120[i].a), Tp(data120[i].b),
                   Tp(data120[i].c), Tp(data120[i].x));
      const Tp f0 = data120[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=2.0000000000000000, b=2.0000000000000000, c=2.0000000000000000.
testcase_hyperg<double> data121[] = {
  { 0.27700831024930750, 2.0000000000000000, 2.0000000000000000,
          2.0000000000000000, -0.90000000000000002 },
  { 0.30864197530864196, 2.0000000000000000, 2.0000000000000000,
          2.0000000000000000, -0.80000000000000004 },
  { 0.34602076124567477, 2.0000000000000000, 2.0000000000000000,
          2.0000000000000000, -0.69999999999999996 },
  { 0.39062499999999994, 2.0000000000000000, 2.0000000000000000,
          2.0000000000000000, -0.59999999999999998 },
  { 0.44444444444444442, 2.0000000000000000, 2.0000000000000000,
          2.0000000000000000, -0.50000000000000000 },
  { 0.51020408163265307, 2.0000000000000000, 2.0000000000000000,
          2.0000000000000000, -0.40000000000000002 },
  { 0.59171597633136097, 2.0000000000000000, 2.0000000000000000,
          2.0000000000000000, -0.30000000000000004 },
  { 0.69444444444444453, 2.0000000000000000, 2.0000000000000000,
          2.0000000000000000, -0.19999999999999996 },
  { 0.82644628099173545, 2.0000000000000000, 2.0000000000000000,
          2.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 2.0000000000000000, 2.0000000000000000,
          2.0000000000000000, 0.0000000000000000 },
  { 1.2345679012345681, 2.0000000000000000, 2.0000000000000000,
          2.0000000000000000, 0.10000000000000009 },
  { 1.5624999999999998, 2.0000000000000000, 2.0000000000000000,
          2.0000000000000000, 0.19999999999999996 },
  { 2.0408163265306127, 2.0000000000000000, 2.0000000000000000,
          2.0000000000000000, 0.30000000000000004 },
  { 2.7777777777777768, 2.0000000000000000, 2.0000000000000000,
          2.0000000000000000, 0.39999999999999991 },
  { 4.0000000000000000, 2.0000000000000000, 2.0000000000000000,
          2.0000000000000000, 0.50000000000000000 },
  { 6.2500000000000036, 2.0000000000000000, 2.0000000000000000,
          2.0000000000000000, 0.60000000000000009 },
  { 11.111111111111109, 2.0000000000000000, 2.0000000000000000,
          2.0000000000000000, 0.69999999999999996 },
  { 25.000000000000007, 2.0000000000000000, 2.0000000000000000,
          2.0000000000000000, 0.80000000000000004 },
  { 99.999999999999872, 2.0000000000000000, 2.0000000000000000,
          2.0000000000000000, 0.89999999999999991 },
};

// Test function for a=2.0000000000000000, b=2.0000000000000000, c=2.0000000000000000.
template <typename Tp>
void test121()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data121)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data121[i].a), Tp(data121[i].b),
                   Tp(data121[i].c), Tp(data121[i].x));
      const Tp f0 = data121[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=2.0000000000000000, b=2.0000000000000000, c=4.0000000000000000.
testcase_hyperg<double> data122[] = {
  { 0.50515448477320868, 2.0000000000000000, 2.0000000000000000,
          4.0000000000000000, -0.90000000000000002 },
  { 0.53674994210078020, 2.0000000000000000, 2.0000000000000000,
          4.0000000000000000, -0.80000000000000004 },
  { 0.57194655162437447, 2.0000000000000000, 2.0000000000000000,
          4.0000000000000000, -0.69999999999999996 },
  { 0.61137322330312327, 2.0000000000000000, 2.0000000000000000,
          4.0000000000000000, -0.59999999999999998 },
  { 0.65581297297972574, 2.0000000000000000, 2.0000000000000000,
          4.0000000000000000, -0.50000000000000000 },
  { 0.70625323977290932, 2.0000000000000000, 2.0000000000000000,
          4.0000000000000000, -0.40000000000000002 },
  { 0.76395739449542643, 2.0000000000000000, 2.0000000000000000,
          4.0000000000000000, -0.30000000000000004 },
  { 0.83056871002513311, 2.0000000000000000, 2.0000000000000000,
          4.0000000000000000, -0.19999999999999996 },
  { 0.90826553449323655, 2.0000000000000000, 2.0000000000000000,
          4.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 2.0000000000000000, 2.0000000000000000,
          4.0000000000000000, 0.0000000000000000 },
  { 1.1098784992198341, 2.0000000000000000, 2.0000000000000000,
          4.0000000000000000, 0.10000000000000009 },
  { 1.2437942741831700, 2.0000000000000000, 2.0000000000000000,
          4.0000000000000000, 0.19999999999999996 },
  { 1.4105343768544543, 2.0000000000000000, 2.0000000000000000,
          4.0000000000000000, 0.30000000000000004 },
  { 1.6238435648986016, 2.0000000000000000, 2.0000000000000000,
          4.0000000000000000, 0.39999999999999991 },
  { 1.9065970003160624, 2.0000000000000000, 2.0000000000000000,
          4.0000000000000000, 0.50000000000000000 },
  { 2.3001951284393627, 2.0000000000000000, 2.0000000000000000,
          4.0000000000000000, 0.60000000000000009 },
  { 2.8891774744673464, 2.0000000000000000, 2.0000000000000000,
          4.0000000000000000, 0.69999999999999996 },
  { 3.8827206436045336, 2.0000000000000000, 2.0000000000000000,
          4.0000000000000000, 0.80000000000000004 },
  { 6.0316345867773542, 2.0000000000000000, 2.0000000000000000,
          4.0000000000000000, 0.89999999999999991 },
};

// Test function for a=2.0000000000000000, b=2.0000000000000000, c=4.0000000000000000.
template <typename Tp>
void test122()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data122)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data122[i].a), Tp(data122[i].b),
                   Tp(data122[i].c), Tp(data122[i].x));
      const Tp f0 = data122[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=2.0000000000000000, b=2.0000000000000000, c=6.0000000000000000.
testcase_hyperg<double> data123[] = {
  { 0.61824560969673303, 2.0000000000000000, 2.0000000000000000,
          6.0000000000000000, -0.90000000000000002 },
  { 0.64645665839161048, 2.0000000000000000, 2.0000000000000000,
          6.0000000000000000, -0.80000000000000004 },
  { 0.67712272792612083, 2.0000000000000000, 2.0000000000000000,
          6.0000000000000000, -0.69999999999999996 },
  { 0.71058076074636822, 2.0000000000000000, 2.0000000000000000,
          6.0000000000000000, -0.59999999999999998 },
  { 0.74723387423852861, 2.0000000000000000, 2.0000000000000000,
          6.0000000000000000, -0.50000000000000000 },
  { 0.78756892188863126, 2.0000000000000000, 2.0000000000000000,
          6.0000000000000000, -0.40000000000000002 },
  { 0.83218012557592713, 2.0000000000000000, 2.0000000000000000,
          6.0000000000000000, -0.30000000000000004 },
  { 0.88180144818204143, 2.0000000000000000, 2.0000000000000000,
          6.0000000000000000, -0.19999999999999996 },
  { 0.93735184459468934, 2.0000000000000000, 2.0000000000000000,
          6.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 2.0000000000000000, 2.0000000000000000,
          6.0000000000000000, 0.0000000000000000 },
  { 1.0712594799044883, 2.0000000000000000, 2.0000000000000000,
          6.0000000000000000, 0.10000000000000009 },
  { 1.1531330932162096, 2.0000000000000000, 2.0000000000000000,
          6.0000000000000000, 0.19999999999999996 },
  { 1.2483404191094898, 2.0000000000000000, 2.0000000000000000,
          6.0000000000000000, 0.30000000000000004 },
  { 1.3606934909972501, 2.0000000000000000, 2.0000000000000000,
          6.0000000000000000, 0.39999999999999991 },
  { 1.4957544469027071, 2.0000000000000000, 2.0000000000000000,
          6.0000000000000000, 0.50000000000000000 },
  { 1.6620777107871287, 2.0000000000000000, 2.0000000000000000,
          6.0000000000000000, 0.60000000000000009 },
  { 1.8738132387064506, 2.0000000000000000, 2.0000000000000000,
          6.0000000000000000, 0.69999999999999996 },
  { 2.1570670242247409, 2.0000000000000000, 2.0000000000000000,
          6.0000000000000000, 0.80000000000000004 },
  { 2.5700735959225494, 2.0000000000000000, 2.0000000000000000,
          6.0000000000000000, 0.89999999999999991 },
};

// Test function for a=2.0000000000000000, b=2.0000000000000000, c=6.0000000000000000.
template <typename Tp>
void test123()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data123)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data123[i].a), Tp(data123[i].b),
                   Tp(data123[i].c), Tp(data123[i].x));
      const Tp f0 = data123[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=2.0000000000000000, b=2.0000000000000000, c=8.0000000000000000.
testcase_hyperg<double> data124[] = {
  { 0.68776713859043437, 2.0000000000000000, 2.0000000000000000,
          8.0000000000000000, -0.90000000000000002 },
  { 0.71280582849489826, 2.0000000000000000, 2.0000000000000000,
          8.0000000000000000, -0.80000000000000004 },
  { 0.73962983054724929, 2.0000000000000000, 2.0000000000000000,
          8.0000000000000000, -0.69999999999999996 },
  { 0.76844343025262085, 2.0000000000000000, 2.0000000000000000,
          8.0000000000000000, -0.59999999999999998 },
  { 0.79948476671182911, 2.0000000000000000, 2.0000000000000000,
          8.0000000000000000, -0.50000000000000000 },
  { 0.83303347721461229, 2.0000000000000000, 2.0000000000000000,
          8.0000000000000000, -0.40000000000000002 },
  { 0.86942060391338782, 2.0000000000000000, 2.0000000000000000,
          8.0000000000000000, -0.30000000000000004 },
  { 0.90904161711581655, 2.0000000000000000, 2.0000000000000000,
          8.0000000000000000, -0.19999999999999996 },
  { 0.95237381468647742, 2.0000000000000000, 2.0000000000000000,
          8.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 2.0000000000000000, 2.0000000000000000,
          8.0000000000000000, 0.0000000000000000 },
  { 1.0526413941912305, 2.0000000000000000, 2.0000000000000000,
          8.0000000000000000, 0.10000000000000009 },
  { 1.1112045278881502, 2.0000000000000000, 2.0000000000000000,
          8.0000000000000000, 0.19999999999999996 },
  { 1.1768500306393046, 2.0000000000000000, 2.0000000000000000,
          8.0000000000000000, 0.30000000000000004 },
  { 1.2510971588297888, 2.0000000000000000, 2.0000000000000000,
          8.0000000000000000, 0.39999999999999991 },
  { 1.3359896747789315, 2.0000000000000000, 2.0000000000000000,
          8.0000000000000000, 0.50000000000000000 },
  { 1.4343740183432725, 2.0000000000000000, 2.0000000000000000,
          8.0000000000000000, 0.60000000000000009 },
  { 1.5504011881337365, 2.0000000000000000, 2.0000000000000000,
          8.0000000000000000, 0.69999999999999996 },
  { 1.6905307012604318, 2.0000000000000000, 2.0000000000000000,
          8.0000000000000000, 0.80000000000000004 },
  { 1.8658920279264424, 2.0000000000000000, 2.0000000000000000,
          8.0000000000000000, 0.89999999999999991 },
};

// Test function for a=2.0000000000000000, b=2.0000000000000000, c=8.0000000000000000.
template <typename Tp>
void test124()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data124)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data124[i].a), Tp(data124[i].b),
                   Tp(data124[i].c), Tp(data124[i].x));
      const Tp f0 = data124[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=2.0000000000000000, b=2.0000000000000000, c=10.000000000000000.
testcase_hyperg<double> data125[] = {
  { 0.73530262886958830, 2.0000000000000000, 2.0000000000000000,
          10.000000000000000, -0.90000000000000002 },
  { 0.75768898977673638, 2.0000000000000000, 2.0000000000000000,
          10.000000000000000, -0.80000000000000004 },
  { 0.78143483544640069, 2.0000000000000000, 2.0000000000000000,
          10.000000000000000, -0.69999999999999996 },
  { 0.80667428603297209, 2.0000000000000000, 2.0000000000000000,
          10.000000000000000, -0.59999999999999998 },
  { 0.83356078772438325, 2.0000000000000000, 2.0000000000000000,
          10.000000000000000, -0.50000000000000000 },
  { 0.86227093001346145, 2.0000000000000000, 2.0000000000000000,
          10.000000000000000, -0.40000000000000002 },
  { 0.89300925500556994, 2.0000000000000000, 2.0000000000000000,
          10.000000000000000, -0.30000000000000004 },
  { 0.92601438873425990, 2.0000000000000000, 2.0000000000000000,
          10.000000000000000, -0.19999999999999996 },
  { 0.96156696230910810, 2.0000000000000000, 2.0000000000000000,
          10.000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 2.0000000000000000, 2.0000000000000000,
          10.000000000000000, 0.0000000000000000 },
  { 1.0417127776179342, 2.0000000000000000, 2.0000000000000000,
          10.000000000000000, 0.10000000000000009 },
  { 1.0871896789480930, 2.0000000000000000, 2.0000000000000000,
          10.000000000000000, 0.19999999999999996 },
  { 1.1370264514689949, 2.0000000000000000, 2.0000000000000000,
          10.000000000000000, 0.30000000000000004 },
  { 1.1919677804049154, 2.0000000000000000, 2.0000000000000000,
          10.000000000000000, 0.39999999999999991 },
  { 1.2529628761065934, 2.0000000000000000, 2.0000000000000000,
          10.000000000000000, 0.50000000000000000 },
  { 1.3212511796458866, 2.0000000000000000, 2.0000000000000000,
          10.000000000000000, 0.60000000000000009 },
  { 1.3985017309668506, 2.0000000000000000, 2.0000000000000000,
          10.000000000000000, 0.69999999999999996 },
  { 1.4870567523847895, 2.0000000000000000, 2.0000000000000000,
          10.000000000000000, 0.80000000000000004 },
  { 1.5904049523738040, 2.0000000000000000, 2.0000000000000000,
          10.000000000000000, 0.89999999999999991 },
};

// Test function for a=2.0000000000000000, b=2.0000000000000000, c=10.000000000000000.
template <typename Tp>
void test125()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data125)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data125[i].a), Tp(data125[i].b),
                   Tp(data125[i].c), Tp(data125[i].x));
      const Tp f0 = data125[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=2.0000000000000000, b=5.0000000000000000, c=2.0000000000000000.
testcase_hyperg<double> data126[] = {
  { 0.040386107340619266, 2.0000000000000000, 5.0000000000000000,
          2.0000000000000000, -0.90000000000000002 },
  { 0.052922149401344633, 2.0000000000000000, 5.0000000000000000,
          2.0000000000000000, -0.80000000000000004 },
  { 0.070429627772374270, 2.0000000000000000, 5.0000000000000000,
          2.0000000000000000, -0.69999999999999996 },
  { 0.095367431640624972, 2.0000000000000000, 5.0000000000000000,
          2.0000000000000000, -0.59999999999999998 },
  { 0.13168724279835387, 2.0000000000000000, 5.0000000000000000,
          2.0000000000000000, -0.50000000000000000 },
  { 0.18593443208187066, 2.0000000000000000, 5.0000000000000000,
          2.0000000000000000, -0.40000000000000002 },
  { 0.26932907434290437, 2.0000000000000000, 5.0000000000000000,
          2.0000000000000000, -0.30000000000000004 },
  { 0.40187757201646096, 2.0000000000000000, 5.0000000000000000,
          2.0000000000000000, -0.19999999999999996 },
  { 0.62092132305915493, 2.0000000000000000, 5.0000000000000000,
          2.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 2.0000000000000000, 5.0000000000000000,
          2.0000000000000000, 0.0000000000000000 },
  { 1.6935087808430296, 2.0000000000000000, 5.0000000000000000,
          2.0000000000000000, 0.10000000000000009 },
  { 3.0517578124999991, 2.0000000000000000, 5.0000000000000000,
          2.0000000000000000, 0.19999999999999996 },
  { 5.9499018266198629, 2.0000000000000000, 5.0000000000000000,
          2.0000000000000000, 0.30000000000000004 },
  { 12.860082304526737, 2.0000000000000000, 5.0000000000000000,
          2.0000000000000000, 0.39999999999999991 },
  { 32.000000000000000, 2.0000000000000000, 5.0000000000000000,
          2.0000000000000000, 0.50000000000000000 },
  { 97.656250000000114, 2.0000000000000000, 5.0000000000000000,
          2.0000000000000000, 0.60000000000000009 },
  { 411.52263374485580, 2.0000000000000000, 5.0000000000000000,
          2.0000000000000000, 0.69999999999999996 },
  { 3124.9999999999991, 2.0000000000000000, 5.0000000000000000,
          2.0000000000000000, 0.80000000000000004 },
  { 99999.999999999665, 2.0000000000000000, 5.0000000000000000,
          2.0000000000000000, 0.89999999999999991 },
};

// Test function for a=2.0000000000000000, b=5.0000000000000000, c=2.0000000000000000.
template <typename Tp>
void test126()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data126)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data126[i].a), Tp(data126[i].b),
                   Tp(data126[i].c), Tp(data126[i].x));
      const Tp f0 = data126[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=2.0000000000000000, b=5.0000000000000000, c=4.0000000000000000.
testcase_hyperg<double> data127[] = {
  { 0.21140107887447140, 2.0000000000000000, 5.0000000000000000,
          4.0000000000000000, -0.90000000000000002 },
  { 0.24005486968449935, 2.0000000000000000, 5.0000000000000000,
          4.0000000000000000, -0.80000000000000004 },
  { 0.27478119275391821, 2.0000000000000000, 5.0000000000000000,
          4.0000000000000000, -0.69999999999999996 },
  { 0.31738281249999994, 2.0000000000000000, 5.0000000000000000,
          4.0000000000000000, -0.59999999999999998 },
  { 0.37037037037037035, 2.0000000000000000, 5.0000000000000000,
          4.0000000000000000, -0.50000000000000000 },
  { 0.43731778425655982, 2.0000000000000000, 5.0000000000000000,
          4.0000000000000000, -0.40000000000000002 },
  { 0.52344105598543444, 2.0000000000000000, 5.0000000000000000,
          4.0000000000000000, -0.30000000000000004 },
  { 0.63657407407407429, 2.0000000000000000, 5.0000000000000000,
          4.0000000000000000, -0.19999999999999996 },
  { 0.78888054094665638, 2.0000000000000000, 5.0000000000000000,
          4.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 2.0000000000000000, 5.0000000000000000,
          4.0000000000000000, 0.0000000000000000 },
  { 1.3031550068587108, 2.0000000000000000, 5.0000000000000000,
          4.0000000000000000, 0.10000000000000009 },
  { 1.7578125000000002, 2.0000000000000000, 5.0000000000000000,
          4.0000000000000000, 0.19999999999999996 },
  { 2.4781341107871717, 2.0000000000000000, 5.0000000000000000,
          4.0000000000000000, 0.30000000000000004 },
  { 3.7037037037037037, 2.0000000000000000, 5.0000000000000000,
          4.0000000000000000, 0.39999999999999991 },
  { 5.9999999999999982, 2.0000000000000000, 5.0000000000000000,
          4.0000000000000000, 0.50000000000000000 },
  { 10.937500000000005, 2.0000000000000000, 5.0000000000000000,
          4.0000000000000000, 0.60000000000000009 },
  { 24.074074074074076, 2.0000000000000000, 5.0000000000000000,
          4.0000000000000000, 0.69999999999999996 },
  { 74.999999999999957, 2.0000000000000000, 5.0000000000000000,
          4.0000000000000000, 0.80000000000000004 },
  { 549.99999999999670, 2.0000000000000000, 5.0000000000000000,
          4.0000000000000000, 0.89999999999999991 },
};

// Test function for a=2.0000000000000000, b=5.0000000000000000, c=4.0000000000000000.
template <typename Tp>
void test127()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data127)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data127[i].a), Tp(data127[i].b),
                   Tp(data127[i].c), Tp(data127[i].x));
      const Tp f0 = data127[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=2.0000000000000000, b=5.0000000000000000, c=6.0000000000000000.
testcase_hyperg<double> data128[] = {
  { 0.33250915203252107, 2.0000000000000000, 5.0000000000000000,
          6.0000000000000000, -0.90000000000000002 },
  { 0.36566851047721960, 2.0000000000000000, 5.0000000000000000,
          6.0000000000000000, -0.80000000000000004 },
  { 0.40414812182437942, 2.0000000000000000, 5.0000000000000000,
          6.0000000000000000, -0.69999999999999996 },
  { 0.44916943268118470, 2.0000000000000000, 5.0000000000000000,
          6.0000000000000000, -0.59999999999999998 },
  { 0.50233081077479569, 2.0000000000000000, 5.0000000000000000,
          6.0000000000000000, -0.50000000000000000 },
  { 0.56575808728873322, 2.0000000000000000, 5.0000000000000000,
          6.0000000000000000, -0.40000000000000002 },
  { 0.64233106844971422, 2.0000000000000000, 5.0000000000000000,
          6.0000000000000000, -0.30000000000000004 },
  { 0.73603371116919514, 2.0000000000000000, 5.0000000000000000,
          6.0000000000000000, -0.19999999999999996 },
  { 0.85251256240112439, 2.0000000000000000, 5.0000000000000000,
          6.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 2.0000000000000000, 5.0000000000000000,
          6.0000000000000000, 0.0000000000000000 },
  { 1.1909065696197674, 2.0000000000000000, 5.0000000000000000,
          6.0000000000000000, 0.10000000000000009 },
  { 1.4447095285569311, 2.0000000000000000, 5.0000000000000000,
          6.0000000000000000, 0.19999999999999996 },
  { 1.7935243137840653, 2.0000000000000000, 5.0000000000000000,
          6.0000000000000000, 0.30000000000000004 },
  { 2.2937035820494454, 2.0000000000000000, 5.0000000000000000,
          6.0000000000000000, 0.39999999999999991 },
  { 3.0524711083016687, 2.0000000000000000, 5.0000000000000000,
          6.0000000000000000, 0.50000000000000000 },
  { 4.2976512669354259, 2.0000000000000000, 5.0000000000000000,
          6.0000000000000000, 0.60000000000000009 },
  { 6.5977107563194677, 2.0000000000000000, 5.0000000000000000,
          6.0000000000000000, 0.69999999999999996 },
  { 11.793747206577530, 2.0000000000000000, 5.0000000000000000,
          6.0000000000000000, 0.80000000000000004 },
  { 29.997625937982058, 2.0000000000000000, 5.0000000000000000,
          6.0000000000000000, 0.89999999999999991 },
};

// Test function for a=2.0000000000000000, b=5.0000000000000000, c=6.0000000000000000.
template <typename Tp>
void test128()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data128)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data128[i].a), Tp(data128[i].b),
                   Tp(data128[i].c), Tp(data128[i].x));
      const Tp f0 = data128[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=2.0000000000000000, b=5.0000000000000000, c=8.0000000000000000.
testcase_hyperg<double> data129[] = {
  { 0.42108197362250305, 2.0000000000000000, 5.0000000000000000,
          8.0000000000000000, -0.90000000000000002 },
  { 0.45503172013983051, 2.0000000000000000, 5.0000000000000000,
          8.0000000000000000, -0.80000000000000004 },
  { 0.49345609813624303, 2.0000000000000000, 5.0000000000000000,
          8.0000000000000000, -0.69999999999999996 },
  { 0.53720880551221295, 2.0000000000000000, 5.0000000000000000,
          8.0000000000000000, -0.59999999999999998 },
  { 0.58736431524847466, 2.0000000000000000, 5.0000000000000000,
          8.0000000000000000, -0.50000000000000000 },
  { 0.64529222467897962, 2.0000000000000000, 5.0000000000000000,
          8.0000000000000000, -0.40000000000000002 },
  { 0.71276337354393904, 2.0000000000000000, 5.0000000000000000,
          8.0000000000000000, -0.30000000000000004 },
  { 0.79210466220795306, 2.0000000000000000, 5.0000000000000000,
          8.0000000000000000, -0.19999999999999996 },
  { 0.88643063455510596, 2.0000000000000000, 5.0000000000000000,
          8.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 2.0000000000000000, 5.0000000000000000,
          8.0000000000000000, 0.0000000000000000 },
  { 1.1387832139040652, 2.0000000000000000, 5.0000000000000000,
          8.0000000000000000, 0.10000000000000009 },
  { 1.3114025920844752, 2.0000000000000000, 5.0000000000000000,
          8.0000000000000000, 0.19999999999999996 },
  { 1.5307655016768162, 2.0000000000000000, 5.0000000000000000,
          8.0000000000000000, 0.30000000000000004 },
  { 1.8170727950333345, 2.0000000000000000, 5.0000000000000000,
          8.0000000000000000, 0.39999999999999991 },
  { 2.2037865486700836, 2.0000000000000000, 5.0000000000000000,
          8.0000000000000000, 0.50000000000000000 },
  { 2.7506766056439380, 2.0000000000000000, 5.0000000000000000,
          8.0000000000000000, 0.60000000000000009 },
  { 3.5764534935716972, 2.0000000000000000, 5.0000000000000000,
          8.0000000000000000, 0.69999999999999996 },
  { 4.9587762302155403, 2.0000000000000000, 5.0000000000000000,
          8.0000000000000000, 0.80000000000000004 },
  { 7.7740847924166800, 2.0000000000000000, 5.0000000000000000,
          8.0000000000000000, 0.89999999999999991 },
};

// Test function for a=2.0000000000000000, b=5.0000000000000000, c=8.0000000000000000.
template <typename Tp>
void test129()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data129)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data129[i].a), Tp(data129[i].b),
                   Tp(data129[i].c), Tp(data129[i].x));
      const Tp f0 = data129[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=2.0000000000000000, b=5.0000000000000000, c=10.000000000000000.
testcase_hyperg<double> data130[] = {
  { 0.48860241312958425, 2.0000000000000000, 5.0000000000000000,
          10.000000000000000, -0.90000000000000002 },
  { 0.52193382517068487, 2.0000000000000000, 5.0000000000000000,
          10.000000000000000, -0.80000000000000004 },
  { 0.55902375003954219, 2.0000000000000000, 5.0000000000000000,
          10.000000000000000, -0.69999999999999996 },
  { 0.60049055150230346, 2.0000000000000000, 5.0000000000000000,
          10.000000000000000, -0.59999999999999998 },
  { 0.64709127927203480, 2.0000000000000000, 5.0000000000000000,
          10.000000000000000, -0.50000000000000000 },
  { 0.69976233335368998, 2.0000000000000000, 5.0000000000000000,
          10.000000000000000, -0.40000000000000002 },
  { 0.75967529501081055, 2.0000000000000000, 5.0000000000000000,
          10.000000000000000, -0.30000000000000004 },
  { 0.82831498895254407, 2.0000000000000000, 5.0000000000000000,
          10.000000000000000, -0.19999999999999996 },
  { 0.90759090169653933, 2.0000000000000000, 5.0000000000000000,
          10.000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 2.0000000000000000, 5.0000000000000000,
          10.000000000000000, 0.0000000000000000 },
  { 1.1088712278667465, 2.0000000000000000, 5.0000000000000000,
          10.000000000000000, 0.10000000000000009 },
  { 1.2387445478440853, 2.0000000000000000, 5.0000000000000000,
          10.000000000000000, 0.19999999999999996 },
  { 1.3959812720437546, 2.0000000000000000, 5.0000000000000000,
          10.000000000000000, 0.30000000000000004 },
  { 1.5897930661091164, 2.0000000000000000, 5.0000000000000000,
          10.000000000000000, 0.39999999999999991 },
  { 1.8340789380307454, 2.0000000000000000, 5.0000000000000000,
          10.000000000000000, 0.50000000000000000 },
  { 2.1509548085970764, 2.0000000000000000, 5.0000000000000000,
          10.000000000000000, 0.60000000000000009 },
  { 2.5782406951207504, 2.0000000000000000, 5.0000000000000000,
          10.000000000000000, 0.69999999999999996 },
  { 3.1877847194242737, 2.0000000000000000, 5.0000000000000000,
          10.000000000000000, 0.80000000000000004 },
  { 4.1421596631676900, 2.0000000000000000, 5.0000000000000000,
          10.000000000000000, 0.89999999999999991 },
};

// Test function for a=2.0000000000000000, b=5.0000000000000000, c=10.000000000000000.
template <typename Tp>
void test130()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data130)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data130[i].a), Tp(data130[i].b),
                   Tp(data130[i].c), Tp(data130[i].x));
      const Tp f0 = data130[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=2.0000000000000000, b=10.000000000000000, c=2.0000000000000000.
testcase_hyperg<double> data131[] = {
  { 0.0016310376661280216, 2.0000000000000000, 10.000000000000000,
          2.0000000000000000, -0.90000000000000002 },
  { 0.0028007538972582421, 2.0000000000000000, 10.000000000000000,
          2.0000000000000000, -0.80000000000000004 },
  { 0.0049603324681551939, 2.0000000000000000, 10.000000000000000,
          2.0000000000000000, -0.69999999999999996 },
  { 0.0090949470177292789, 2.0000000000000000, 10.000000000000000,
          2.0000000000000000, -0.59999999999999998 },
  { 0.017341529915832606, 2.0000000000000000, 10.000000000000000,
          2.0000000000000000, -0.50000000000000000 },
  { 0.034571613033607777, 2.0000000000000000, 10.000000000000000,
          2.0000000000000000, -0.40000000000000002 },
  { 0.072538150286405714, 2.0000000000000000, 10.000000000000000,
          2.0000000000000000, -0.30000000000000004 },
  { 0.16150558288984579, 2.0000000000000000, 10.000000000000000,
          2.0000000000000000, -0.19999999999999996 },
  { 0.38554328942953148, 2.0000000000000000, 10.000000000000000,
          2.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 2.0000000000000000, 10.000000000000000,
          2.0000000000000000, 0.0000000000000000 },
  { 2.8679719907924444, 2.0000000000000000, 10.000000000000000,
          2.0000000000000000, 0.10000000000000009 },
  { 9.3132257461547816, 2.0000000000000000, 10.000000000000000,
          2.0000000000000000, 0.19999999999999996 },
  { 35.401331746414378, 2.0000000000000000, 10.000000000000000,
          2.0000000000000000, 0.30000000000000004 },
  { 165.38171687920172, 2.0000000000000000, 10.000000000000000,
          2.0000000000000000, 0.39999999999999991 },
  { 1024.0000000000000, 2.0000000000000000, 10.000000000000000,
          2.0000000000000000, 0.50000000000000000 },
  { 9536.7431640625200, 2.0000000000000000, 10.000000000000000,
          2.0000000000000000, 0.60000000000000009 },
  { 169350.87808430271, 2.0000000000000000, 10.000000000000000,
          2.0000000000000000, 0.69999999999999996 },
  { 9765624.9999999944, 2.0000000000000000, 10.000000000000000,
          2.0000000000000000, 0.80000000000000004 },
  { 9999999999.9999332, 2.0000000000000000, 10.000000000000000,
          2.0000000000000000, 0.89999999999999991 },
};

// Test function for a=2.0000000000000000, b=10.000000000000000, c=2.0000000000000000.
template <typename Tp>
void test131()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data131)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data131[i].a), Tp(data131[i].b),
                   Tp(data131[i].c), Tp(data131[i].x));
      const Tp f0 = data131[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=2.0000000000000000, b=10.000000000000000, c=4.0000000000000000.
testcase_hyperg<double> data132[] = {
  { 0.071191280690193509, 2.0000000000000000, 10.000000000000000,
          4.0000000000000000, -0.90000000000000002 },
  { 0.085646504654238079, 2.0000000000000000, 10.000000000000000,
          4.0000000000000000, -0.80000000000000004 },
  { 0.10478215656371073, 2.0000000000000000, 10.000000000000000,
          4.0000000000000000, -0.69999999999999996 },
  { 0.13074816337653575, 2.0000000000000000, 10.000000000000000,
          4.0000000000000000, -0.59999999999999998 },
  { 0.16701141666848116, 2.0000000000000000, 10.000000000000000,
          4.0000000000000000, -0.50000000000000000 },
  { 0.21939323375313971, 2.0000000000000000, 10.000000000000000,
          4.0000000000000000, -0.40000000000000002 },
  { 0.29813515331786627, 2.0000000000000000, 10.000000000000000,
          4.0000000000000000, -0.30000000000000004 },
  { 0.42225974638874386, 2.0000000000000000, 10.000000000000000,
          4.0000000000000000, -0.19999999999999996 },
  { 0.62942145962174867, 2.0000000000000000, 10.000000000000000,
          4.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 2.0000000000000000, 10.000000000000000,
          4.0000000000000000, 0.0000000000000000 },
  { 1.7218685262373197, 2.0000000000000000, 10.000000000000000,
          4.0000000000000000, 0.10000000000000009 },
  { 3.2855760483514689, 2.0000000000000000, 10.000000000000000,
          4.0000000000000000, 0.19999999999999996 },
  { 7.1616652508907093, 2.0000000000000000, 10.000000000000000,
          4.0000000000000000, 0.30000000000000004 },
  { 18.612326808485907, 2.0000000000000000, 10.000000000000000,
          4.0000000000000000, 0.39999999999999991 },
  { 61.476190476190474, 2.0000000000000000, 10.000000000000000,
          4.0000000000000000, 0.50000000000000000 },
  { 286.27580915178623, 2.0000000000000000, 10.000000000000000,
          4.0000000000000000, 0.60000000000000009 },
  { 2274.9441142102296, 2.0000000000000000, 10.000000000000000,
          4.0000000000000000, 0.69999999999999996 },
  { 47229.761904761865, 2.0000000000000000, 10.000000000000000,
          4.0000000000000000, 0.80000000000000004 },
  { 9961460.7142856438, 2.0000000000000000, 10.000000000000000,
          4.0000000000000000, 0.89999999999999991 },
};

// Test function for a=2.0000000000000000, b=10.000000000000000, c=4.0000000000000000.
template <typename Tp>
void test132()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data132)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data132[i].a), Tp(data132[i].b),
                   Tp(data132[i].c), Tp(data132[i].x));
      const Tp f0 = data132[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=2.0000000000000000, b=10.000000000000000, c=6.0000000000000000.
testcase_hyperg<double> data133[] = {
  { 0.14747230019381052, 2.0000000000000000, 10.000000000000000,
          6.0000000000000000, -0.90000000000000002 },
  { 0.17073600100690603, 2.0000000000000000, 10.000000000000000,
          6.0000000000000000, -0.80000000000000004 },
  { 0.19982795745135354, 2.0000000000000000, 10.000000000000000,
          6.0000000000000000, -0.69999999999999996 },
  { 0.23681776864188067, 2.0000000000000000, 10.000000000000000,
          6.0000000000000000, -0.59999999999999998 },
  { 0.28475624360398022, 2.0000000000000000, 10.000000000000000,
          6.0000000000000000, -0.50000000000000000 },
  { 0.34827500743063144, 2.0000000000000000, 10.000000000000000,
          6.0000000000000000, -0.40000000000000002 },
  { 0.43464829159684681, 2.0000000000000000, 10.000000000000000,
          6.0000000000000000, -0.30000000000000004 },
  { 0.55576053438064787, 2.0000000000000000, 10.000000000000000,
          6.0000000000000000, -0.19999999999999996 },
  { 0.73195020913445485, 2.0000000000000000, 10.000000000000000,
          6.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 2.0000000000000000, 10.000000000000000,
          6.0000000000000000, 0.0000000000000000 },
  { 1.4310223867822929, 2.0000000000000000, 10.000000000000000,
          6.0000000000000000, 0.10000000000000009 },
  { 2.1742563399057540, 2.0000000000000000, 10.000000000000000,
          6.0000000000000000, 0.19999999999999996 },
  { 3.5769231236256043, 2.0000000000000000, 10.000000000000000,
          6.0000000000000000, 0.30000000000000004 },
  { 6.5620441134844363, 2.0000000000000000, 10.000000000000000,
          6.0000000000000000, 0.39999999999999991 },
  { 14.063492063492063, 2.0000000000000000, 10.000000000000000,
          6.0000000000000000, 0.50000000000000000 },
  { 38.085937500000036, 2.0000000000000000, 10.000000000000000,
          6.0000000000000000, 0.60000000000000009 },
  { 150.92973632068282, 2.0000000000000000, 10.000000000000000,
          6.0000000000000000, 0.69999999999999996 },
  { 1212.3015873015852, 2.0000000000000000, 10.000000000000000,
          6.0000000000000000, 0.80000000000000004 },
  { 55107.142857142389, 2.0000000000000000, 10.000000000000000,
          6.0000000000000000, 0.89999999999999991 },
};

// Test function for a=2.0000000000000000, b=10.000000000000000, c=6.0000000000000000.
template <typename Tp>
void test133()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data133)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data133[i].a), Tp(data133[i].b),
                   Tp(data133[i].c), Tp(data133[i].x));
      const Tp f0 = data133[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=2.0000000000000000, b=10.000000000000000, c=8.0000000000000000.
testcase_hyperg<double> data134[] = {
  { 0.21658059714090588, 2.0000000000000000, 10.000000000000000,
          8.0000000000000000, -0.90000000000000002 },
  { 0.24513539602702844, 2.0000000000000000, 10.000000000000000,
          8.0000000000000000, -0.80000000000000004 },
  { 0.27967018274845046, 2.0000000000000000, 10.000000000000000,
          8.0000000000000000, -0.69999999999999996 },
  { 0.32196044921874994, 2.0000000000000000, 10.000000000000000,
          8.0000000000000000, -0.59999999999999998 },
  { 0.37448559670781900, 2.0000000000000000, 10.000000000000000,
          8.0000000000000000, -0.50000000000000000 },
  { 0.44078856032208796, 2.0000000000000000, 10.000000000000000,
          8.0000000000000000, -0.40000000000000002 },
  { 0.52606701446027815, 2.0000000000000000, 10.000000000000000,
          8.0000000000000000, -0.30000000000000004 },
  { 0.63818158436214001, 2.0000000000000000, 10.000000000000000,
          8.0000000000000000, -0.19999999999999996 },
  { 0.78944971882612769, 2.0000000000000000, 10.000000000000000,
          8.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 2.0000000000000000, 10.000000000000000,
          8.0000000000000000, 0.0000000000000000 },
  { 1.3044251384443430, 2.0000000000000000, 10.000000000000000,
          8.0000000000000000, 0.10000000000000009 },
  { 1.7659505208333335, 2.0000000000000000, 10.000000000000000,
          8.0000000000000000, 0.19999999999999996 },
  { 2.5093710953769270, 2.0000000000000000, 10.000000000000000,
          8.0000000000000000, 0.30000000000000004 },
  { 3.8065843621399158, 2.0000000000000000, 10.000000000000000,
          8.0000000000000000, 0.39999999999999991 },
  { 6.3333333333333313, 2.0000000000000000, 10.000000000000000,
          8.0000000000000000, 0.50000000000000000 },
  { 12.109375000000004, 2.0000000000000000, 10.000000000000000,
          8.0000000000000000, 0.60000000000000009 },
  { 29.115226337448540, 2.0000000000000000, 10.000000000000000,
          8.0000000000000000, 0.69999999999999996 },
  { 108.33333333333330, 2.0000000000000000, 10.000000000000000,
          8.0000000000000000, 0.80000000000000004 },
  { 1224.9999999999923, 2.0000000000000000, 10.000000000000000,
          8.0000000000000000, 0.89999999999999991 },
};

// Test function for a=2.0000000000000000, b=10.000000000000000, c=8.0000000000000000.
template <typename Tp>
void test134()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data134)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data134[i].a), Tp(data134[i].b),
                   Tp(data134[i].c), Tp(data134[i].x));
      const Tp f0 = data134[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=2.0000000000000000, b=10.000000000000000, c=10.000000000000000.
testcase_hyperg<double> data135[] = {
  { 0.27700831024930750, 2.0000000000000000, 10.000000000000000,
          10.000000000000000, -0.90000000000000002 },
  { 0.30864197530864196, 2.0000000000000000, 10.000000000000000,
          10.000000000000000, -0.80000000000000004 },
  { 0.34602076124567477, 2.0000000000000000, 10.000000000000000,
          10.000000000000000, -0.69999999999999996 },
  { 0.39062499999999994, 2.0000000000000000, 10.000000000000000,
          10.000000000000000, -0.59999999999999998 },
  { 0.44444444444444442, 2.0000000000000000, 10.000000000000000,
          10.000000000000000, -0.50000000000000000 },
  { 0.51020408163265307, 2.0000000000000000, 10.000000000000000,
          10.000000000000000, -0.40000000000000002 },
  { 0.59171597633136097, 2.0000000000000000, 10.000000000000000,
          10.000000000000000, -0.30000000000000004 },
  { 0.69444444444444453, 2.0000000000000000, 10.000000000000000,
          10.000000000000000, -0.19999999999999996 },
  { 0.82644628099173545, 2.0000000000000000, 10.000000000000000,
          10.000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 2.0000000000000000, 10.000000000000000,
          10.000000000000000, 0.0000000000000000 },
  { 1.2345679012345681, 2.0000000000000000, 10.000000000000000,
          10.000000000000000, 0.10000000000000009 },
  { 1.5624999999999998, 2.0000000000000000, 10.000000000000000,
          10.000000000000000, 0.19999999999999996 },
  { 2.0408163265306127, 2.0000000000000000, 10.000000000000000,
          10.000000000000000, 0.30000000000000004 },
  { 2.7777777777777768, 2.0000000000000000, 10.000000000000000,
          10.000000000000000, 0.39999999999999991 },
  { 4.0000000000000000, 2.0000000000000000, 10.000000000000000,
          10.000000000000000, 0.50000000000000000 },
  { 6.2500000000000036, 2.0000000000000000, 10.000000000000000,
          10.000000000000000, 0.60000000000000009 },
  { 11.111111111111109, 2.0000000000000000, 10.000000000000000,
          10.000000000000000, 0.69999999999999996 },
  { 25.000000000000007, 2.0000000000000000, 10.000000000000000,
          10.000000000000000, 0.80000000000000004 },
  { 99.999999999999872, 2.0000000000000000, 10.000000000000000,
          10.000000000000000, 0.89999999999999991 },
};

// Test function for a=2.0000000000000000, b=10.000000000000000, c=10.000000000000000.
template <typename Tp>
void test135()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data135)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data135[i].a), Tp(data135[i].b),
                   Tp(data135[i].c), Tp(data135[i].x));
      const Tp f0 = data135[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=2.0000000000000000, b=20.000000000000000, c=2.0000000000000000.
testcase_hyperg<double> data136[] = {
  { 2.6602838683283435e-06, 2.0000000000000000, 20.000000000000000,
          2.0000000000000000, -0.90000000000000002 },
  { 7.8442223930072316e-06, 2.0000000000000000, 20.000000000000000,
          2.0000000000000000, -0.80000000000000004 },
  { 2.4604898194634598e-05, 2.0000000000000000, 20.000000000000000,
          2.0000000000000000, -0.69999999999999996 },
  { 8.2718061255302686e-05, 2.0000000000000000, 20.000000000000000,
          2.0000000000000000, -0.59999999999999998 },
  { 0.00030072865982171723, 2.0000000000000000, 20.000000000000000,
          2.0000000000000000, -0.50000000000000000 },
  { 0.0011951964277455193, 2.0000000000000000, 20.000000000000000,
          2.0000000000000000, -0.40000000000000002 },
  { 0.0052617832469731814, 2.0000000000000000, 20.000000000000000,
          2.0000000000000000, -0.30000000000000004 },
  { 0.026084053304588847, 2.0000000000000000, 20.000000000000000,
          2.0000000000000000, -0.19999999999999996 },
  { 0.14864362802414346, 2.0000000000000000, 20.000000000000000,
          2.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 2.0000000000000000, 20.000000000000000,
          2.0000000000000000, 0.0000000000000000 },
  { 8.2252633399699757, 2.0000000000000000, 20.000000000000000,
          2.0000000000000000, 0.10000000000000009 },
  { 86.736173798840269, 2.0000000000000000, 20.000000000000000,
          2.0000000000000000, 0.19999999999999996 },
  { 1253.2542894196865, 2.0000000000000000, 20.000000000000000,
          2.0000000000000000, 0.30000000000000004 },
  { 27351.112277912434, 2.0000000000000000, 20.000000000000000,
          2.0000000000000000, 0.39999999999999991 },
  { 1048576.0000000000, 2.0000000000000000, 20.000000000000000,
          2.0000000000000000, 0.50000000000000000 },
  { 90949470.177293226, 2.0000000000000000, 20.000000000000000,
          2.0000000000000000, 0.60000000000000009 },
  { 28679719907.924358, 2.0000000000000000, 20.000000000000000,
          2.0000000000000000, 0.69999999999999996 },
  { 95367431640624.906, 2.0000000000000000, 20.000000000000000,
          2.0000000000000000, 0.80000000000000004 },
  { 9.9999999999998657e+19, 2.0000000000000000, 20.000000000000000,
          2.0000000000000000, 0.89999999999999991 },
};

// Test function for a=2.0000000000000000, b=20.000000000000000, c=2.0000000000000000.
template <typename Tp>
void test136()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data136)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data136[i].a), Tp(data136[i].b),
                   Tp(data136[i].c), Tp(data136[i].x));
      const Tp f0 = data136[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(5.0000000000000039e-13));
}

// Test data for a=2.0000000000000000, b=20.000000000000000, c=4.0000000000000000.
testcase_hyperg<double> data137[] = {
  { 0.018828092583720951, 2.0000000000000000, 20.000000000000000,
          4.0000000000000000, -0.90000000000000002 },
  { 0.023381944060455316, 2.0000000000000000, 20.000000000000000,
          4.0000000000000000, -0.80000000000000004 },
  { 0.029789623984280793, 2.0000000000000000, 20.000000000000000,
          4.0000000000000000, -0.69999999999999996 },
  { 0.039191021482500497, 2.0000000000000000, 20.000000000000000,
          4.0000000000000000, -0.59999999999999998 },
  { 0.053727813036721514, 2.0000000000000000, 20.000000000000000,
          4.0000000000000000, -0.50000000000000000 },
  { 0.077762010061669024, 2.0000000000000000, 20.000000000000000,
          4.0000000000000000, -0.40000000000000002 },
  { 0.12110505620123302, 2.0000000000000000, 20.000000000000000,
          4.0000000000000000, -0.30000000000000004 },
  { 0.20870149809080590, 2.0000000000000000, 20.000000000000000,
          4.0000000000000000, -0.19999999999999996 },
  { 0.41429234328785769, 2.0000000000000000, 20.000000000000000,
          4.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 2.0000000000000000, 20.000000000000000,
          4.0000000000000000, 0.0000000000000000 },
  { 3.1308087404153113, 2.0000000000000000, 20.000000000000000,
          4.0000000000000000, 0.10000000000000009 },
  { 13.586180626453050, 2.0000000000000000, 20.000000000000000,
          4.0000000000000000, 0.19999999999999996 },
  { 87.117304082784415, 2.0000000000000000, 20.000000000000000,
          4.0000000000000000, 0.30000000000000004 },
  { 889.26474381242826, 2.0000000000000000, 20.000000000000000,
          4.0000000000000000, 0.39999999999999991 },
  { 16231.913312693494, 2.0000000000000000, 20.000000000000000,
          4.0000000000000000, 0.50000000000000000 },
  { 653537.51168945129, 2.0000000000000000, 20.000000000000000,
          4.0000000000000000, 0.60000000000000009 },
  { 87756230.793848589, 2.0000000000000000, 20.000000000000000,
          4.0000000000000000, 0.69999999999999996 },
  { 101493977171.74945, 2.0000000000000000, 20.000000000000000,
          4.0000000000000000, 0.80000000000000004 },
  { 21375960679556916., 2.0000000000000000, 20.000000000000000,
          4.0000000000000000, 0.89999999999999991 },
};

// Test function for a=2.0000000000000000, b=20.000000000000000, c=4.0000000000000000.
template <typename Tp>
void test137()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data137)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data137[i].a), Tp(data137[i].b),
                   Tp(data137[i].c), Tp(data137[i].x));
      const Tp f0 = data137[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(1.0000000000000008e-12));
}

// Test data for a=2.0000000000000000, b=20.000000000000000, c=6.0000000000000000.
testcase_hyperg<double> data138[] = {
  { 0.049200410661854252, 2.0000000000000000, 20.000000000000000,
          6.0000000000000000, -0.90000000000000002 },
  { 0.059460876757152226, 2.0000000000000000, 20.000000000000000,
          6.0000000000000000, -0.80000000000000004 },
  { 0.073244762686653350, 2.0000000000000000, 20.000000000000000,
          6.0000000000000000, -0.69999999999999996 },
  { 0.092334626017932922, 2.0000000000000000, 20.000000000000000,
          6.0000000000000000, -0.59999999999999998 },
  { 0.11976760350696837, 2.0000000000000000, 20.000000000000000,
          6.0000000000000000, -0.50000000000000000 },
  { 0.16102414609169383, 2.0000000000000000, 20.000000000000000,
          6.0000000000000000, -0.40000000000000002 },
  { 0.22670456785796222, 2.0000000000000000, 20.000000000000000,
          6.0000000000000000, -0.30000000000000004 },
  { 0.33912903252727361, 2.0000000000000000, 20.000000000000000,
          6.0000000000000000, -0.19999999999999996 },
  { 0.55049794600858060, 2.0000000000000000, 20.000000000000000,
          6.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 2.0000000000000000, 20.000000000000000,
          6.0000000000000000, 0.0000000000000000 },
  { 2.1254722872032232, 2.0000000000000000, 20.000000000000000,
          6.0000000000000000, 0.10000000000000009 },
  { 5.6261213886736172, 2.0000000000000000, 20.000000000000000,
          6.0000000000000000, 0.19999999999999996 },
  { 20.137315891130996, 2.0000000000000000, 20.000000000000000,
          6.0000000000000000, 0.30000000000000004 },
  { 108.04381584643853, 2.0000000000000000, 20.000000000000000,
          6.0000000000000000, 0.39999999999999991 },
  { 992.41692466460245, 2.0000000000000000, 20.000000000000000,
          6.0000000000000000, 0.50000000000000000 },
  { 19055.363816004465, 2.0000000000000000, 20.000000000000000,
          6.0000000000000000, 0.60000000000000009 },
  { 1105471.9504312086, 2.0000000000000000, 20.000000000000000,
          6.0000000000000000, 0.69999999999999996 },
  { 448521363.90608919, 2.0000000000000000, 20.000000000000000,
          6.0000000000000000, 0.80000000000000004 },
  { 19078917293639.004, 2.0000000000000000, 20.000000000000000,
          6.0000000000000000, 0.89999999999999991 },
};

// Test function for a=2.0000000000000000, b=20.000000000000000, c=6.0000000000000000.
template <typename Tp>
void test138()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data138)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data138[i].a), Tp(data138[i].b),
                   Tp(data138[i].c), Tp(data138[i].x));
      const Tp f0 = data138[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(5.0000000000000039e-13));
}

// Test data for a=2.0000000000000000, b=20.000000000000000, c=8.0000000000000000.
testcase_hyperg<double> data139[] = {
  { 0.083753547015334745, 2.0000000000000000, 20.000000000000000,
          8.0000000000000000, -0.90000000000000002 },
  { 0.099238444687035701, 2.0000000000000000, 20.000000000000000,
          8.0000000000000000, -0.80000000000000004 },
  { 0.11938294012867758, 2.0000000000000000, 20.000000000000000,
          8.0000000000000000, -0.69999999999999996 },
  { 0.14622683905023326, 2.0000000000000000, 20.000000000000000,
          8.0000000000000000, -0.59999999999999998 },
  { 0.18303556733713025, 2.0000000000000000, 20.000000000000000,
          8.0000000000000000, -0.50000000000000000 },
  { 0.23527764069382409, 2.0000000000000000, 20.000000000000000,
          8.0000000000000000, -0.40000000000000002 },
  { 0.31261681740827085, 2.0000000000000000, 20.000000000000000,
          8.0000000000000000, -0.30000000000000004 },
  { 0.43327581880538862, 2.0000000000000000, 20.000000000000000,
          8.0000000000000000, -0.19999999999999996 },
  { 0.63445840637296658, 2.0000000000000000, 20.000000000000000,
          8.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 2.0000000000000000, 20.000000000000000,
          8.0000000000000000, 0.0000000000000000 },
  { 1.7438842395813297, 2.0000000000000000, 20.000000000000000,
          8.0000000000000000, 0.10000000000000009 },
  { 3.5070840938209269, 2.0000000000000000, 20.000000000000000,
          8.0000000000000000, 0.19999999999999996 },
  { 8.6573372006089713, 2.0000000000000000, 20.000000000000000,
          8.0000000000000000, 0.30000000000000004 },
  { 28.779342118408906, 2.0000000000000000, 20.000000000000000,
          8.0000000000000000, 0.39999999999999991 },
  { 147.50178613955714, 2.0000000000000000, 20.000000000000000,
          8.0000000000000000, 0.50000000000000000 },
  { 1427.1686016136398, 2.0000000000000000, 20.000000000000000,
          8.0000000000000000, 0.60000000000000009 },
  { 36780.643714655642, 2.0000000000000000, 20.000000000000000,
          8.0000000000000000, 0.69999999999999996 },
  { 5313869.6058585485, 2.0000000000000000, 20.000000000000000,
          8.0000000000000000, 0.80000000000000004 },
  { 46057280607.381966, 2.0000000000000000, 20.000000000000000,
          8.0000000000000000, 0.89999999999999991 },
};

// Test function for a=2.0000000000000000, b=20.000000000000000, c=8.0000000000000000.
template <typename Tp>
void test139()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data139)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data139[i].a), Tp(data139[i].b),
                   Tp(data139[i].c), Tp(data139[i].x));
      const Tp f0 = data139[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=2.0000000000000000, b=20.000000000000000, c=10.000000000000000.
testcase_hyperg<double> data140[] = {
  { 0.11920045035073683, 2.0000000000000000, 20.000000000000000,
          10.000000000000000, -0.90000000000000002 },
  { 0.13907946814302774, 2.0000000000000000, 20.000000000000000,
          10.000000000000000, -0.80000000000000004 },
  { 0.16431439792559688, 2.0000000000000000, 20.000000000000000,
          10.000000000000000, -0.69999999999999996 },
  { 0.19698796016987008, 2.0000000000000000, 20.000000000000000,
          10.000000000000000, -0.59999999999999998 },
  { 0.24028510928790570, 2.0000000000000000, 20.000000000000000,
          10.000000000000000, -0.50000000000000000 },
  { 0.29926031296483119, 2.0000000000000000, 20.000000000000000,
          10.000000000000000, -0.40000000000000002 },
  { 0.38229327814229169, 2.0000000000000000, 20.000000000000000,
          10.000000000000000, -0.30000000000000004 },
  { 0.50402047283093110, 2.0000000000000000, 20.000000000000000,
          10.000000000000000, -0.19999999999999996 },
  { 0.69167261179586503, 2.0000000000000000, 20.000000000000000,
          10.000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 2.0000000000000000, 20.000000000000000,
          10.000000000000000, 0.0000000000000000 },
  { 1.5503152253394308, 2.0000000000000000, 20.000000000000000,
          10.000000000000000, 0.10000000000000009 },
  { 2.6469548193635797, 2.0000000000000000, 20.000000000000000,
          10.000000000000000, 0.19999999999999996 },
  { 5.1882631330566813, 2.0000000000000000, 20.000000000000000,
          10.000000000000000, 0.30000000000000004 },
  { 12.476792759124516, 2.0000000000000000, 20.000000000000000,
          10.000000000000000, 0.39999999999999991 },
  { 41.026391565091259, 2.0000000000000000, 20.000000000000000,
          10.000000000000000, 0.50000000000000000 },
  { 220.92584715988204, 2.0000000000000000, 20.000000000000000,
          10.000000000000000, 0.60000000000000009 },
  { 2677.0834450236207, 2.0000000000000000, 20.000000000000000,
          10.000000000000000, 0.69999999999999996 },
  { 141774.31260689779, 2.0000000000000000, 20.000000000000000,
          10.000000000000000, 0.80000000000000004 },
  { 254267148.83196995, 2.0000000000000000, 20.000000000000000,
          10.000000000000000, 0.89999999999999991 },
};

// Test function for a=2.0000000000000000, b=20.000000000000000, c=10.000000000000000.
template <typename Tp>
void test140()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data140)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data140[i].a), Tp(data140[i].b),
                   Tp(data140[i].c), Tp(data140[i].x));
      const Tp f0 = data140[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=5.0000000000000000, b=0.0000000000000000, c=2.0000000000000000.
testcase_hyperg<double> data141[] = {
  { 1.0000000000000000, 5.0000000000000000, 0.0000000000000000,
          2.0000000000000000, -0.90000000000000002 },
  { 1.0000000000000000, 5.0000000000000000, 0.0000000000000000,
          2.0000000000000000, -0.80000000000000004 },
  { 1.0000000000000000, 5.0000000000000000, 0.0000000000000000,
          2.0000000000000000, -0.69999999999999996 },
  { 1.0000000000000000, 5.0000000000000000, 0.0000000000000000,
          2.0000000000000000, -0.59999999999999998 },
  { 1.0000000000000000, 5.0000000000000000, 0.0000000000000000,
          2.0000000000000000, -0.50000000000000000 },
  { 1.0000000000000000, 5.0000000000000000, 0.0000000000000000,
          2.0000000000000000, -0.40000000000000002 },
  { 1.0000000000000000, 5.0000000000000000, 0.0000000000000000,
          2.0000000000000000, -0.30000000000000004 },
  { 1.0000000000000000, 5.0000000000000000, 0.0000000000000000,
          2.0000000000000000, -0.19999999999999996 },
  { 1.0000000000000000, 5.0000000000000000, 0.0000000000000000,
          2.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 5.0000000000000000, 0.0000000000000000,
          2.0000000000000000, 0.0000000000000000 },
  { 1.0000000000000000, 5.0000000000000000, 0.0000000000000000,
          2.0000000000000000, 0.10000000000000009 },
  { 1.0000000000000000, 5.0000000000000000, 0.0000000000000000,
          2.0000000000000000, 0.19999999999999996 },
  { 1.0000000000000000, 5.0000000000000000, 0.0000000000000000,
          2.0000000000000000, 0.30000000000000004 },
  { 1.0000000000000000, 5.0000000000000000, 0.0000000000000000,
          2.0000000000000000, 0.39999999999999991 },
  { 1.0000000000000000, 5.0000000000000000, 0.0000000000000000,
          2.0000000000000000, 0.50000000000000000 },
  { 1.0000000000000000, 5.0000000000000000, 0.0000000000000000,
          2.0000000000000000, 0.60000000000000009 },
  { 1.0000000000000000, 5.0000000000000000, 0.0000000000000000,
          2.0000000000000000, 0.69999999999999996 },
  { 1.0000000000000000, 5.0000000000000000, 0.0000000000000000,
          2.0000000000000000, 0.80000000000000004 },
  { 1.0000000000000000, 5.0000000000000000, 0.0000000000000000,
          2.0000000000000000, 0.89999999999999991 },
};

// Test function for a=5.0000000000000000, b=0.0000000000000000, c=2.0000000000000000.
template <typename Tp>
void test141()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data141)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data141[i].a), Tp(data141[i].b),
                   Tp(data141[i].c), Tp(data141[i].x));
      const Tp f0 = data141[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=5.0000000000000000, b=0.0000000000000000, c=4.0000000000000000.
testcase_hyperg<double> data142[] = {
  { 1.0000000000000000, 5.0000000000000000, 0.0000000000000000,
          4.0000000000000000, -0.90000000000000002 },
  { 1.0000000000000000, 5.0000000000000000, 0.0000000000000000,
          4.0000000000000000, -0.80000000000000004 },
  { 1.0000000000000000, 5.0000000000000000, 0.0000000000000000,
          4.0000000000000000, -0.69999999999999996 },
  { 1.0000000000000000, 5.0000000000000000, 0.0000000000000000,
          4.0000000000000000, -0.59999999999999998 },
  { 1.0000000000000000, 5.0000000000000000, 0.0000000000000000,
          4.0000000000000000, -0.50000000000000000 },
  { 1.0000000000000000, 5.0000000000000000, 0.0000000000000000,
          4.0000000000000000, -0.40000000000000002 },
  { 1.0000000000000000, 5.0000000000000000, 0.0000000000000000,
          4.0000000000000000, -0.30000000000000004 },
  { 1.0000000000000000, 5.0000000000000000, 0.0000000000000000,
          4.0000000000000000, -0.19999999999999996 },
  { 1.0000000000000000, 5.0000000000000000, 0.0000000000000000,
          4.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 5.0000000000000000, 0.0000000000000000,
          4.0000000000000000, 0.0000000000000000 },
  { 1.0000000000000000, 5.0000000000000000, 0.0000000000000000,
          4.0000000000000000, 0.10000000000000009 },
  { 1.0000000000000000, 5.0000000000000000, 0.0000000000000000,
          4.0000000000000000, 0.19999999999999996 },
  { 1.0000000000000000, 5.0000000000000000, 0.0000000000000000,
          4.0000000000000000, 0.30000000000000004 },
  { 1.0000000000000000, 5.0000000000000000, 0.0000000000000000,
          4.0000000000000000, 0.39999999999999991 },
  { 1.0000000000000000, 5.0000000000000000, 0.0000000000000000,
          4.0000000000000000, 0.50000000000000000 },
  { 1.0000000000000000, 5.0000000000000000, 0.0000000000000000,
          4.0000000000000000, 0.60000000000000009 },
  { 1.0000000000000000, 5.0000000000000000, 0.0000000000000000,
          4.0000000000000000, 0.69999999999999996 },
  { 1.0000000000000000, 5.0000000000000000, 0.0000000000000000,
          4.0000000000000000, 0.80000000000000004 },
  { 1.0000000000000000, 5.0000000000000000, 0.0000000000000000,
          4.0000000000000000, 0.89999999999999991 },
};

// Test function for a=5.0000000000000000, b=0.0000000000000000, c=4.0000000000000000.
template <typename Tp>
void test142()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data142)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data142[i].a), Tp(data142[i].b),
                   Tp(data142[i].c), Tp(data142[i].x));
      const Tp f0 = data142[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=5.0000000000000000, b=0.0000000000000000, c=6.0000000000000000.
testcase_hyperg<double> data143[] = {
  { 1.0000000000000000, 5.0000000000000000, 0.0000000000000000,
          6.0000000000000000, -0.90000000000000002 },
  { 1.0000000000000000, 5.0000000000000000, 0.0000000000000000,
          6.0000000000000000, -0.80000000000000004 },
  { 1.0000000000000000, 5.0000000000000000, 0.0000000000000000,
          6.0000000000000000, -0.69999999999999996 },
  { 1.0000000000000000, 5.0000000000000000, 0.0000000000000000,
          6.0000000000000000, -0.59999999999999998 },
  { 1.0000000000000000, 5.0000000000000000, 0.0000000000000000,
          6.0000000000000000, -0.50000000000000000 },
  { 1.0000000000000000, 5.0000000000000000, 0.0000000000000000,
          6.0000000000000000, -0.40000000000000002 },
  { 1.0000000000000000, 5.0000000000000000, 0.0000000000000000,
          6.0000000000000000, -0.30000000000000004 },
  { 1.0000000000000000, 5.0000000000000000, 0.0000000000000000,
          6.0000000000000000, -0.19999999999999996 },
  { 1.0000000000000000, 5.0000000000000000, 0.0000000000000000,
          6.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 5.0000000000000000, 0.0000000000000000,
          6.0000000000000000, 0.0000000000000000 },
  { 1.0000000000000000, 5.0000000000000000, 0.0000000000000000,
          6.0000000000000000, 0.10000000000000009 },
  { 1.0000000000000000, 5.0000000000000000, 0.0000000000000000,
          6.0000000000000000, 0.19999999999999996 },
  { 1.0000000000000000, 5.0000000000000000, 0.0000000000000000,
          6.0000000000000000, 0.30000000000000004 },
  { 1.0000000000000000, 5.0000000000000000, 0.0000000000000000,
          6.0000000000000000, 0.39999999999999991 },
  { 1.0000000000000000, 5.0000000000000000, 0.0000000000000000,
          6.0000000000000000, 0.50000000000000000 },
  { 1.0000000000000000, 5.0000000000000000, 0.0000000000000000,
          6.0000000000000000, 0.60000000000000009 },
  { 1.0000000000000000, 5.0000000000000000, 0.0000000000000000,
          6.0000000000000000, 0.69999999999999996 },
  { 1.0000000000000000, 5.0000000000000000, 0.0000000000000000,
          6.0000000000000000, 0.80000000000000004 },
  { 1.0000000000000000, 5.0000000000000000, 0.0000000000000000,
          6.0000000000000000, 0.89999999999999991 },
};

// Test function for a=5.0000000000000000, b=0.0000000000000000, c=6.0000000000000000.
template <typename Tp>
void test143()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data143)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data143[i].a), Tp(data143[i].b),
                   Tp(data143[i].c), Tp(data143[i].x));
      const Tp f0 = data143[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=5.0000000000000000, b=0.0000000000000000, c=8.0000000000000000.
testcase_hyperg<double> data144[] = {
  { 1.0000000000000000, 5.0000000000000000, 0.0000000000000000,
          8.0000000000000000, -0.90000000000000002 },
  { 1.0000000000000000, 5.0000000000000000, 0.0000000000000000,
          8.0000000000000000, -0.80000000000000004 },
  { 1.0000000000000000, 5.0000000000000000, 0.0000000000000000,
          8.0000000000000000, -0.69999999999999996 },
  { 1.0000000000000000, 5.0000000000000000, 0.0000000000000000,
          8.0000000000000000, -0.59999999999999998 },
  { 1.0000000000000000, 5.0000000000000000, 0.0000000000000000,
          8.0000000000000000, -0.50000000000000000 },
  { 1.0000000000000000, 5.0000000000000000, 0.0000000000000000,
          8.0000000000000000, -0.40000000000000002 },
  { 1.0000000000000000, 5.0000000000000000, 0.0000000000000000,
          8.0000000000000000, -0.30000000000000004 },
  { 1.0000000000000000, 5.0000000000000000, 0.0000000000000000,
          8.0000000000000000, -0.19999999999999996 },
  { 1.0000000000000000, 5.0000000000000000, 0.0000000000000000,
          8.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 5.0000000000000000, 0.0000000000000000,
          8.0000000000000000, 0.0000000000000000 },
  { 1.0000000000000000, 5.0000000000000000, 0.0000000000000000,
          8.0000000000000000, 0.10000000000000009 },
  { 1.0000000000000000, 5.0000000000000000, 0.0000000000000000,
          8.0000000000000000, 0.19999999999999996 },
  { 1.0000000000000000, 5.0000000000000000, 0.0000000000000000,
          8.0000000000000000, 0.30000000000000004 },
  { 1.0000000000000000, 5.0000000000000000, 0.0000000000000000,
          8.0000000000000000, 0.39999999999999991 },
  { 1.0000000000000000, 5.0000000000000000, 0.0000000000000000,
          8.0000000000000000, 0.50000000000000000 },
  { 1.0000000000000000, 5.0000000000000000, 0.0000000000000000,
          8.0000000000000000, 0.60000000000000009 },
  { 1.0000000000000000, 5.0000000000000000, 0.0000000000000000,
          8.0000000000000000, 0.69999999999999996 },
  { 1.0000000000000000, 5.0000000000000000, 0.0000000000000000,
          8.0000000000000000, 0.80000000000000004 },
  { 1.0000000000000000, 5.0000000000000000, 0.0000000000000000,
          8.0000000000000000, 0.89999999999999991 },
};

// Test function for a=5.0000000000000000, b=0.0000000000000000, c=8.0000000000000000.
template <typename Tp>
void test144()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data144)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data144[i].a), Tp(data144[i].b),
                   Tp(data144[i].c), Tp(data144[i].x));
      const Tp f0 = data144[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=5.0000000000000000, b=0.0000000000000000, c=10.000000000000000.
testcase_hyperg<double> data145[] = {
  { 1.0000000000000000, 5.0000000000000000, 0.0000000000000000,
          10.000000000000000, -0.90000000000000002 },
  { 1.0000000000000000, 5.0000000000000000, 0.0000000000000000,
          10.000000000000000, -0.80000000000000004 },
  { 1.0000000000000000, 5.0000000000000000, 0.0000000000000000,
          10.000000000000000, -0.69999999999999996 },
  { 1.0000000000000000, 5.0000000000000000, 0.0000000000000000,
          10.000000000000000, -0.59999999999999998 },
  { 1.0000000000000000, 5.0000000000000000, 0.0000000000000000,
          10.000000000000000, -0.50000000000000000 },
  { 1.0000000000000000, 5.0000000000000000, 0.0000000000000000,
          10.000000000000000, -0.40000000000000002 },
  { 1.0000000000000000, 5.0000000000000000, 0.0000000000000000,
          10.000000000000000, -0.30000000000000004 },
  { 1.0000000000000000, 5.0000000000000000, 0.0000000000000000,
          10.000000000000000, -0.19999999999999996 },
  { 1.0000000000000000, 5.0000000000000000, 0.0000000000000000,
          10.000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 5.0000000000000000, 0.0000000000000000,
          10.000000000000000, 0.0000000000000000 },
  { 1.0000000000000000, 5.0000000000000000, 0.0000000000000000,
          10.000000000000000, 0.10000000000000009 },
  { 1.0000000000000000, 5.0000000000000000, 0.0000000000000000,
          10.000000000000000, 0.19999999999999996 },
  { 1.0000000000000000, 5.0000000000000000, 0.0000000000000000,
          10.000000000000000, 0.30000000000000004 },
  { 1.0000000000000000, 5.0000000000000000, 0.0000000000000000,
          10.000000000000000, 0.39999999999999991 },
  { 1.0000000000000000, 5.0000000000000000, 0.0000000000000000,
          10.000000000000000, 0.50000000000000000 },
  { 1.0000000000000000, 5.0000000000000000, 0.0000000000000000,
          10.000000000000000, 0.60000000000000009 },
  { 1.0000000000000000, 5.0000000000000000, 0.0000000000000000,
          10.000000000000000, 0.69999999999999996 },
  { 1.0000000000000000, 5.0000000000000000, 0.0000000000000000,
          10.000000000000000, 0.80000000000000004 },
  { 1.0000000000000000, 5.0000000000000000, 0.0000000000000000,
          10.000000000000000, 0.89999999999999991 },
};

// Test function for a=5.0000000000000000, b=0.0000000000000000, c=10.000000000000000.
template <typename Tp>
void test145()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data145)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data145[i].a), Tp(data145[i].b),
                   Tp(data145[i].c), Tp(data145[i].x));
      const Tp f0 = data145[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=5.0000000000000000, b=0.50000000000000000, c=2.0000000000000000.
testcase_hyperg<double> data146[] = {
  { 0.52275983209457511, 5.0000000000000000, 0.50000000000000000,
          2.0000000000000000, -0.90000000000000002 },
  { 0.54700336898142965, 5.0000000000000000, 0.50000000000000000,
          2.0000000000000000, -0.80000000000000004 },
  { 0.57468955512601971, 5.0000000000000000, 0.50000000000000000,
          2.0000000000000000, -0.69999999999999996 },
  { 0.60665490543315015, 5.0000000000000000, 0.50000000000000000,
          2.0000000000000000, -0.59999999999999998 },
  { 0.64403057859056123, 5.0000000000000000, 0.50000000000000000,
          2.0000000000000000, -0.50000000000000000 },
  { 0.68838183648623719, 5.0000000000000000, 0.50000000000000000,
          2.0000000000000000, -0.40000000000000002 },
  { 0.74193265039311085, 5.0000000000000000, 0.50000000000000000,
          2.0000000000000000, -0.30000000000000004 },
  { 0.80794095908995300, 5.0000000000000000, 0.50000000000000000,
          2.0000000000000000, -0.19999999999999996 },
  { 0.89135275749639320, 5.0000000000000000, 0.50000000000000000,
          2.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 5.0000000000000000, 0.50000000000000000,
          2.0000000000000000, 0.0000000000000000 },
  { 1.1469266219310688, 5.0000000000000000, 0.50000000000000000,
          2.0000000000000000, 0.10000000000000009 },
  { 1.3552340708357489, 5.0000000000000000, 0.50000000000000000,
          2.0000000000000000, 0.19999999999999996 },
  { 1.6690840478838305, 5.0000000000000000, 0.50000000000000000,
          2.0000000000000000, 0.30000000000000004 },
  { 2.1815415453174483, 5.0000000000000000, 0.50000000000000000,
          2.0000000000000000, 0.39999999999999991 },
  { 3.1156892546032235, 5.0000000000000000, 0.50000000000000000,
          2.0000000000000000, 0.50000000000000000 },
  { 5.1109077417760416, 5.0000000000000000, 0.50000000000000000,
          2.0000000000000000, 0.60000000000000009 },
  { 10.560352936466296, 5.0000000000000000, 0.50000000000000000,
          2.0000000000000000, 0.69999999999999996 },
  { 33.541019662496815, 5.0000000000000000, 0.50000000000000000,
          2.0000000000000000, 0.80000000000000004 },
  { 300.66343065819501, 5.0000000000000000, 0.50000000000000000,
          2.0000000000000000, 0.89999999999999991 },
};

// Test function for a=5.0000000000000000, b=0.50000000000000000, c=2.0000000000000000.
template <typename Tp>
void test146()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data146)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data146[i].a), Tp(data146[i].b),
                   Tp(data146[i].c), Tp(data146[i].x));
      const Tp f0 = data146[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=5.0000000000000000, b=0.50000000000000000, c=4.0000000000000000.
testcase_hyperg<double> data147[] = {
  { 0.68252041951139264, 5.0000000000000000, 0.50000000000000000,
          4.0000000000000000, -0.90000000000000002 },
  { 0.70394732624993395, 5.0000000000000000, 0.50000000000000000,
          4.0000000000000000, -0.80000000000000004 },
  { 0.72748884971552041, 5.0000000000000000, 0.50000000000000000,
          4.0000000000000000, -0.69999999999999996 },
  { 0.75351147371199689, 5.0000000000000000, 0.50000000000000000,
          4.0000000000000000, -0.59999999999999998 },
  { 0.78247589005573748, 5.0000000000000000, 0.50000000000000000,
          4.0000000000000000, -0.50000000000000000 },
  { 0.81497017420249818, 5.0000000000000000, 0.50000000000000000,
          4.0000000000000000, -0.40000000000000002 },
  { 0.85175826875009564, 5.0000000000000000, 0.50000000000000000,
          4.0000000000000000, -0.30000000000000004 },
  { 0.89385278481745867, 5.0000000000000000, 0.50000000000000000,
          4.0000000000000000, -0.19999999999999996 },
  { 0.94262778709507411, 5.0000000000000000, 0.50000000000000000,
          4.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 5.0000000000000000, 0.50000000000000000,
          4.0000000000000000, 0.0000000000000000 },
  { 1.0687327277420910, 5.0000000000000000, 0.50000000000000000,
          4.0000000000000000, 0.10000000000000009 },
  { 1.1529725508983291, 5.0000000000000000, 0.50000000000000000,
          4.0000000000000000, 0.19999999999999996 },
  { 1.2592587134058799, 5.0000000000000000, 0.50000000000000000,
          4.0000000000000000, 0.30000000000000004 },
  { 1.3985773194637892, 5.0000000000000000, 0.50000000000000000,
          4.0000000000000000, 0.39999999999999991 },
  { 1.5909902576697317, 5.0000000000000000, 0.50000000000000000,
          4.0000000000000000, 0.50000000000000000 },
  { 1.8776023607249752, 5.0000000000000000, 0.50000000000000000,
          4.0000000000000000, 0.60000000000000009 },
  { 2.3582499003694646, 5.0000000000000000, 0.50000000000000000,
          4.0000000000000000, 0.69999999999999996 },
  { 3.3541019662496838, 5.0000000000000000, 0.50000000000000000,
          4.0000000000000000, 0.80000000000000004 },
  { 6.7198400278577859, 5.0000000000000000, 0.50000000000000000,
          4.0000000000000000, 0.89999999999999991 },
};

// Test function for a=5.0000000000000000, b=0.50000000000000000, c=4.0000000000000000.
template <typename Tp>
void test147()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data147)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data147[i].a), Tp(data147[i].b),
                   Tp(data147[i].c), Tp(data147[i].x));
      const Tp f0 = data147[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=5.0000000000000000, b=0.50000000000000000, c=6.0000000000000000.
testcase_hyperg<double> data148[] = {
  { 0.75755211927082589, 5.0000000000000000, 0.50000000000000000,
          6.0000000000000000, -0.90000000000000002 },
  { 0.77603550233010998, 5.0000000000000000, 0.50000000000000000,
          6.0000000000000000, -0.80000000000000004 },
  { 0.79596241913438492, 5.0000000000000000, 0.50000000000000000,
          6.0000000000000000, -0.69999999999999996 },
  { 0.81753360792105201, 5.0000000000000000, 0.50000000000000000,
          6.0000000000000000, -0.59999999999999998 },
  { 0.84099165409805521, 5.0000000000000000, 0.50000000000000000,
          6.0000000000000000, -0.50000000000000000 },
  { 0.86663303852180873, 5.0000000000000000, 0.50000000000000000,
          6.0000000000000000, -0.40000000000000002 },
  { 0.89482475828629970, 5.0000000000000000, 0.50000000000000000,
          6.0000000000000000, -0.30000000000000004 },
  { 0.92602774279590350, 5.0000000000000000, 0.50000000000000000,
          6.0000000000000000, -0.19999999999999996 },
  { 0.96083064727087386, 5.0000000000000000, 0.50000000000000000,
          6.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 5.0000000000000000, 0.50000000000000000,
          6.0000000000000000, 0.0000000000000000 },
  { 1.0445570841313008, 5.0000000000000000, 0.50000000000000000,
          6.0000000000000000, 0.10000000000000009 },
  { 1.0959004638926031, 5.0000000000000000, 0.50000000000000000,
          6.0000000000000000, 0.19999999999999996 },
  { 1.1560106261370562, 5.0000000000000000, 0.50000000000000000,
          6.0000000000000000, 0.30000000000000004 },
  { 1.2278121770678145, 5.0000000000000000, 0.50000000000000000,
          6.0000000000000000, 0.39999999999999991 },
  { 1.3158640214709998, 5.0000000000000000, 0.50000000000000000,
          6.0000000000000000, 0.50000000000000000 },
  { 1.4278095344155000, 5.0000000000000000, 0.50000000000000000,
          6.0000000000000000, 0.60000000000000009 },
  { 1.5778700502946612, 5.0000000000000000, 0.50000000000000000,
          6.0000000000000000, 0.69999999999999996 },
  { 1.7972173289196469, 5.0000000000000000, 0.50000000000000000,
          6.0000000000000000, 0.80000000000000004 },
  { 2.1789970569269732, 5.0000000000000000, 0.50000000000000000,
          6.0000000000000000, 0.89999999999999991 },
};

// Test function for a=5.0000000000000000, b=0.50000000000000000, c=6.0000000000000000.
template <typename Tp>
void test148()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data148)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data148[i].a), Tp(data148[i].b),
                   Tp(data148[i].c), Tp(data148[i].x));
      const Tp f0 = data148[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=5.0000000000000000, b=0.50000000000000000, c=8.0000000000000000.
testcase_hyperg<double> data149[] = {
  { 0.80270093579329471, 5.0000000000000000, 0.50000000000000000,
          8.0000000000000000, -0.90000000000000002 },
  { 0.81884974572462788, 5.0000000000000000, 0.50000000000000000,
          8.0000000000000000, -0.80000000000000004 },
  { 0.83605266330015271, 5.0000000000000000, 0.50000000000000000,
          8.0000000000000000, -0.69999999999999996 },
  { 0.85443340762795972, 5.0000000000000000, 0.50000000000000000,
          8.0000000000000000, -0.59999999999999998 },
  { 0.87413762182790655, 5.0000000000000000, 0.50000000000000000,
          8.0000000000000000, -0.50000000000000000 },
  { 0.89533826626907331, 5.0000000000000000, 0.50000000000000000,
          8.0000000000000000, -0.40000000000000002 },
  { 0.91824276674115268, 5.0000000000000000, 0.50000000000000000,
          8.0000000000000000, -0.30000000000000004 },
  { 0.94310265050720576, 5.0000000000000000, 0.50000000000000000,
          8.0000000000000000, -0.19999999999999996 },
  { 0.97022678857609712, 5.0000000000000000, 0.50000000000000000,
          8.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 5.0000000000000000, 0.50000000000000000,
          8.0000000000000000, 0.0000000000000000 },
  { 1.0329098673199812, 5.0000000000000000, 0.50000000000000000,
          8.0000000000000000, 0.10000000000000009 },
  { 1.0695865684573389, 5.0000000000000000, 0.50000000000000000,
          8.0000000000000000, 0.19999999999999996 },
  { 1.1108642103944570, 5.0000000000000000, 0.50000000000000000,
          8.0000000000000000, 0.30000000000000004 },
  { 1.1578795055970506, 5.0000000000000000, 0.50000000000000000,
          8.0000000000000000, 0.39999999999999991 },
  { 1.2122394794169442, 5.0000000000000000, 0.50000000000000000,
          8.0000000000000000, 0.50000000000000000 },
  { 1.2763274721556934, 5.0000000000000000, 0.50000000000000000,
          8.0000000000000000, 0.60000000000000009 },
  { 1.3539179650251021, 5.0000000000000000, 0.50000000000000000,
          8.0000000000000000, 0.69999999999999996 },
  { 1.4515986118197148, 5.0000000000000000, 0.50000000000000000,
          8.0000000000000000, 0.80000000000000004 },
  { 1.5829284571614219, 5.0000000000000000, 0.50000000000000000,
          8.0000000000000000, 0.89999999999999991 },
};

// Test function for a=5.0000000000000000, b=0.50000000000000000, c=8.0000000000000000.
template <typename Tp>
void test149()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data149)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data149[i].a), Tp(data149[i].b),
                   Tp(data149[i].c), Tp(data149[i].x));
      const Tp f0 = data149[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=5.0000000000000000, b=0.50000000000000000, c=10.000000000000000.
testcase_hyperg<double> data150[] = {
  { 0.83322694172301959, 5.0000000000000000, 0.50000000000000000,
          10.000000000000000, -0.90000000000000002 },
  { 0.84753931604765664, 5.0000000000000000, 0.50000000000000000,
          10.000000000000000, -0.80000000000000004 },
  { 0.86265784532195022, 5.0000000000000000, 0.50000000000000000,
          10.000000000000000, -0.69999999999999996 },
  { 0.87866479300707079, 5.0000000000000000, 0.50000000000000000,
          10.000000000000000, -0.59999999999999998 },
  { 0.89565516540263501, 5.0000000000000000, 0.50000000000000000,
          10.000000000000000, -0.50000000000000000 },
  { 0.91373946207610557, 5.0000000000000000, 0.50000000000000000,
          10.000000000000000, -0.40000000000000002 },
  { 0.93304721345881891, 5.0000000000000000, 0.50000000000000000,
          10.000000000000000, -0.30000000000000004 },
  { 0.95373159512905148, 5.0000000000000000, 0.50000000000000000,
          10.000000000000000, -0.19999999999999996 },
  { 0.97597554238828121, 5.0000000000000000, 0.50000000000000000,
          10.000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 5.0000000000000000, 0.50000000000000000,
          10.000000000000000, 0.0000000000000000 },
  { 1.0260752851887982, 5.0000000000000000, 0.50000000000000000,
          10.000000000000000, 0.10000000000000009 },
  { 1.0545371197996178, 5.0000000000000000, 0.50000000000000000,
          10.000000000000000, 0.19999999999999996 },
  { 1.0858099017045830, 5.0000000000000000, 0.50000000000000000,
          10.000000000000000, 0.30000000000000004 },
  { 1.1204416568688709, 5.0000000000000000, 0.50000000000000000,
          10.000000000000000, 0.39999999999999991 },
  { 1.1591587835964847, 5.0000000000000000, 0.50000000000000000,
          10.000000000000000, 0.50000000000000000 },
  { 1.2029564720303347, 5.0000000000000000, 0.50000000000000000,
          10.000000000000000, 0.60000000000000009 },
  { 1.2532588722007874, 5.0000000000000000, 0.50000000000000000,
          10.000000000000000, 0.69999999999999996 },
  { 1.3122319926925459, 5.0000000000000000, 0.50000000000000000,
          10.000000000000000, 0.80000000000000004 },
  { 1.3834948587364100, 5.0000000000000000, 0.50000000000000000,
          10.000000000000000, 0.89999999999999991 },
};

// Test function for a=5.0000000000000000, b=0.50000000000000000, c=10.000000000000000.
template <typename Tp>
void test150()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data150)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data150[i].a), Tp(data150[i].b),
                   Tp(data150[i].c), Tp(data150[i].x));
      const Tp f0 = data150[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=5.0000000000000000, b=1.0000000000000000, c=2.0000000000000000.
testcase_hyperg<double> data151[] = {
  { 0.25646288779245091, 5.0000000000000000, 1.0000000000000000,
          2.0000000000000000, -0.90000000000000002 },
  { 0.28273129096174382, 5.0000000000000000, 1.0000000000000000,
          2.0000000000000000, -0.80000000000000004 },
  { 0.31438201170962976, 5.0000000000000000, 1.0000000000000000,
          2.0000000000000000, -0.69999999999999996 },
  { 0.35308837890625017, 5.0000000000000000, 1.0000000000000000,
          2.0000000000000000, -0.59999999999999998 },
  { 0.40123456790123463, 5.0000000000000000, 1.0000000000000000,
          2.0000000000000000, -0.50000000000000000 },
  { 0.46230737192836319, 5.0000000000000000, 1.0000000000000000,
          2.0000000000000000, -0.40000000000000002 },
  { 0.54156016946185359, 5.0000000000000000, 1.0000000000000000,
          2.0000000000000000, -0.30000000000000004 },
  { 0.64718364197530875, 5.0000000000000000, 1.0000000000000000,
          2.0000000000000000, -0.19999999999999996 },
  { 0.79246636158732342, 5.0000000000000000, 1.0000000000000000,
          2.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 5.0000000000000000, 1.0000000000000000,
          2.0000000000000000, 0.0000000000000000 },
  { 1.3103947568968148, 5.0000000000000000, 1.0000000000000000,
          2.0000000000000000, 0.10000000000000009 },
  { 1.8017578125000004, 5.0000000000000000, 1.0000000000000000,
          2.0000000000000000, 0.19999999999999996 },
  { 2.6374427321949185, 5.0000000000000000, 1.0000000000000000,
          2.0000000000000000, 0.30000000000000004 },
  { 4.1975308641975282, 5.0000000000000000, 1.0000000000000000,
          2.0000000000000000, 0.39999999999999991 },
  { 7.4999999999999964, 5.0000000000000000, 1.0000000000000000,
          2.0000000000000000, 0.50000000000000000 },
  { 15.859375000000012, 5.0000000000000000, 1.0000000000000000,
          2.0000000000000000, 0.60000000000000009 },
  { 43.734567901234513, 5.0000000000000000, 1.0000000000000000,
          2.0000000000000000, 0.69999999999999996 },
  { 194.99999999999994, 5.0000000000000000, 1.0000000000000000,
          2.0000000000000000, 0.80000000000000004 },
  { 2777.4999999999832, 5.0000000000000000, 1.0000000000000000,
          2.0000000000000000, 0.89999999999999991 },
};

// Test function for a=5.0000000000000000, b=1.0000000000000000, c=2.0000000000000000.
template <typename Tp>
void test151()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data151)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data151[i].a), Tp(data151[i].b),
                   Tp(data151[i].c), Tp(data151[i].x));
      const Tp f0 = data151[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=5.0000000000000000, b=1.0000000000000000, c=4.0000000000000000.
testcase_hyperg<double> data152[] = {
  { 0.46398891966759004, 5.0000000000000000, 1.0000000000000000,
          4.0000000000000000, -0.90000000000000002 },
  { 0.49382716049382713, 5.0000000000000000, 1.0000000000000000,
          4.0000000000000000, -0.80000000000000004 },
  { 0.52768166089965396, 5.0000000000000000, 1.0000000000000000,
          4.0000000000000000, -0.69999999999999996 },
  { 0.56640624999999978, 5.0000000000000000, 1.0000000000000000,
          4.0000000000000000, -0.59999999999999998 },
  { 0.61111111111111116, 5.0000000000000000, 1.0000000000000000,
          4.0000000000000000, -0.50000000000000000 },
  { 0.66326530612244894, 5.0000000000000000, 1.0000000000000000,
          4.0000000000000000, -0.40000000000000002 },
  { 0.72485207100591698, 5.0000000000000000, 1.0000000000000000,
          4.0000000000000000, -0.30000000000000004 },
  { 0.79861111111111094, 5.0000000000000000, 1.0000000000000000,
          4.0000000000000000, -0.19999999999999996 },
  { 0.88842975206611552, 5.0000000000000000, 1.0000000000000000,
          4.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 5.0000000000000000, 1.0000000000000000,
          4.0000000000000000, 0.0000000000000000 },
  { 1.1419753086419753, 5.0000000000000000, 1.0000000000000000,
          4.0000000000000000, 0.10000000000000009 },
  { 1.3281249999999998, 5.0000000000000000, 1.0000000000000000,
          4.0000000000000000, 0.19999999999999996 },
  { 1.5816326530612239, 5.0000000000000000, 1.0000000000000000,
          4.0000000000000000, 0.30000000000000004 },
  { 1.9444444444444444, 5.0000000000000000, 1.0000000000000000,
          4.0000000000000000, 0.39999999999999991 },
  { 2.5000000000000000, 5.0000000000000000, 1.0000000000000000,
          4.0000000000000000, 0.50000000000000000 },
  { 3.4374999999999996, 5.0000000000000000, 1.0000000000000000,
          4.0000000000000000, 0.60000000000000009 },
  { 5.2777777777777715, 5.0000000000000000, 1.0000000000000000,
          4.0000000000000000, 0.69999999999999996 },
  { 9.9999999999999947, 5.0000000000000000, 1.0000000000000000,
          4.0000000000000000, 0.80000000000000004 },
  { 32.499999999999837, 5.0000000000000000, 1.0000000000000000,
          4.0000000000000000, 0.89999999999999991 },
};

// Test function for a=5.0000000000000000, b=1.0000000000000000, c=4.0000000000000000.
template <typename Tp>
void test152()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data152)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data152[i].a), Tp(data152[i].b),
                   Tp(data152[i].c), Tp(data152[i].x));
      const Tp f0 = data152[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=5.0000000000000000, b=1.0000000000000000, c=6.0000000000000000.
testcase_hyperg<double> data153[] = {
  { 0.57476744883397501, 5.0000000000000000, 1.0000000000000000,
          6.0000000000000000, -0.90000000000000002 },
  { 0.60302731682513933, 5.0000000000000000, 1.0000000000000000,
          6.0000000000000000, -0.80000000000000004 },
  { 0.63425708719096396, 5.0000000000000000, 1.0000000000000000,
          6.0000000000000000, -0.69999999999999996 },
  { 0.66895764182970419, 5.0000000000000000, 1.0000000000000000,
          6.0000000000000000, -0.59999999999999998 },
  { 0.70775063063963428, 5.0000000000000000, 1.0000000000000000,
          6.0000000000000000, -0.50000000000000000 },
  { 0.75141762103495924, 5.0000000000000000, 1.0000000000000000,
          6.0000000000000000, -0.40000000000000002 },
  { 0.80095569442603298, 5.0000000000000000, 1.0000000000000000,
          6.0000000000000000, -0.30000000000000004 },
  { 0.85765823887436754, 5.0000000000000000, 1.0000000000000000,
          6.0000000000000000, -0.19999999999999996 },
  { 0.92323549576335540, 5.0000000000000000, 1.0000000000000000,
          6.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 5.0000000000000000, 1.0000000000000000,
          6.0000000000000000, 0.0000000000000000 },
  { 1.0911622464839472, 5.0000000000000000, 1.0000000000000000,
          6.0000000000000000, 0.10000000000000009 },
  { 1.2013226178607666, 5.0000000000000000, 1.0000000000000000,
          6.0000000000000000, 0.19999999999999996 },
  { 1.3373332072682687, 5.0000000000000000, 1.0000000000000000,
          6.0000000000000000, 0.30000000000000004 },
  { 1.5099074378209716, 5.0000000000000000, 1.0000000000000000,
          6.0000000000000000, 0.39999999999999991 },
  { 1.7368822229245819, 5.0000000000000000, 1.0000000000000000,
          6.0000000000000000, 0.50000000000000000 },
  { 2.0505871832661429, 5.0000000000000000, 1.0000000000000000,
          6.0000000000000000, 0.60000000000000009 },
  { 2.5172389775867967, 5.0000000000000000, 1.0000000000000000,
          6.0000000000000000, 0.69999999999999996 },
  { 3.3015631983556144, 5.0000000000000000, 1.0000000000000000,
          6.0000000000000000, 0.80000000000000004 },
  { 5.0005935155044519, 5.0000000000000000, 1.0000000000000000,
          6.0000000000000000, 0.89999999999999991 },
};

// Test function for a=5.0000000000000000, b=1.0000000000000000, c=6.0000000000000000.
template <typename Tp>
void test153()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data153)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data153[i].a), Tp(data153[i].b),
                   Tp(data153[i].c), Tp(data153[i].x));
      const Tp f0 = data153[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=5.0000000000000000, b=1.0000000000000000, c=8.0000000000000000.
testcase_hyperg<double> data154[] = {
  { 0.64582752605387961, 5.0000000000000000, 1.0000000000000000,
          8.0000000000000000, -0.90000000000000002 },
  { 0.67184161997264169, 5.0000000000000000, 1.0000000000000000,
          8.0000000000000000, -0.80000000000000004 },
  { 0.70012779922368040, 5.0000000000000000, 1.0000000000000000,
          8.0000000000000000, -0.69999999999999996 },
  { 0.73100784656910256, 5.0000000000000000, 1.0000000000000000,
          8.0000000000000000, -0.59999999999999998 },
  { 0.76486919089091077, 5.0000000000000000, 1.0000000000000000,
          8.0000000000000000, -0.50000000000000000 },
  { 0.80218301124334579, 5.0000000000000000, 1.0000000000000000,
          8.0000000000000000, -0.40000000000000002 },
  { 0.84352883533234446, 5.0000000000000000, 1.0000000000000000,
          8.0000000000000000, -0.30000000000000004 },
  { 0.88962858902212572, 5.0000000000000000, 1.0000000000000000,
          8.0000000000000000, -0.19999999999999996 },
  { 0.94139473468584123, 5.0000000000000000, 1.0000000000000000,
          8.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 5.0000000000000000, 1.0000000000000000,
          8.0000000000000000, 0.0000000000000000 },
  { 1.0669812691939897, 5.0000000000000000, 1.0000000000000000,
          8.0000000000000000, 0.10000000000000009 },
  { 1.1443996012177726, 5.0000000000000000, 1.0000000000000000,
          8.0000000000000000, 0.19999999999999996 },
  { 1.2350966976721314, 5.0000000000000000, 1.0000000000000000,
          8.0000000000000000, 0.30000000000000004 },
  { 1.3431264370409088, 5.0000000000000000, 1.0000000000000000,
          8.0000000000000000, 0.39999999999999991 },
  { 1.4745266814162399, 5.0000000000000000, 1.0000000000000000,
          8.0000000000000000, 0.50000000000000000 },
  { 1.6388137104840066, 5.0000000000000000, 1.0000000000000000,
          8.0000000000000000, 0.60000000000000009 },
  { 1.8522074849776518, 5.0000000000000000, 1.0000000000000000,
          8.0000000000000000, 0.69999999999999996 },
  { 2.1458016978417458, 5.0000000000000000, 1.0000000000000000,
          8.0000000000000000, 0.80000000000000004 },
  { 2.5927464669826339, 5.0000000000000000, 1.0000000000000000,
          8.0000000000000000, 0.89999999999999991 },
};

// Test function for a=5.0000000000000000, b=1.0000000000000000, c=8.0000000000000000.
template <typename Tp>
void test154()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data154)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data154[i].a), Tp(data154[i].b),
                   Tp(data154[i].c), Tp(data154[i].x));
      const Tp f0 = data154[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=5.0000000000000000, b=1.0000000000000000, c=10.000000000000000.
testcase_hyperg<double> data155[] = {
  { 0.69583236336670629, 5.0000000000000000, 1.0000000000000000,
          10.000000000000000, -0.90000000000000002 },
  { 0.71968920666899694, 5.0000000000000000, 1.0000000000000000,
          10.000000000000000, -0.80000000000000004 },
  { 0.74533885416044232, 5.0000000000000000, 1.0000000000000000,
          10.000000000000000, -0.69999999999999996 },
  { 0.77300145361503014, 5.0000000000000000, 1.0000000000000000,
          10.000000000000000, -0.59999999999999998 },
  { 0.80293630810919514, 5.0000000000000000, 1.0000000000000000,
          10.000000000000000, -0.50000000000000000 },
  { 0.83545132638592035, 5.0000000000000000, 1.0000000000000000,
          10.000000000000000, -0.40000000000000002 },
  { 0.87091544744412508, 5.0000000000000000, 1.0000000000000000,
          10.000000000000000, -0.30000000000000004 },
  { 0.90977522877919847, 5.0000000000000000, 1.0000000000000000,
          10.000000000000000, -0.19999999999999996 },
  { 0.95257738192069130, 5.0000000000000000, 1.0000000000000000,
          10.000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 5.0000000000000000, 1.0000000000000000,
          10.000000000000000, 0.0000000000000000 },
  { 1.0528968282789379, 5.0000000000000000, 1.0000000000000000,
          10.000000000000000, 0.10000000000000009 },
  { 1.1123617169062123, 5.0000000000000000, 1.0000000000000000,
          10.000000000000000, 0.19999999999999996 },
  { 1.1798254572896132, 5.0000000000000000, 1.0000000000000000,
          10.000000000000000, 0.30000000000000004 },
  { 1.2572069000522696, 5.0000000000000000, 1.0000000000000000,
          10.000000000000000, 0.39999999999999991 },
  { 1.3471600884974377, 5.0000000000000000, 1.0000000000000000,
          10.000000000000000, 0.50000000000000000 },
  { 1.4535032279573519, 5.0000000000000000, 1.0000000000000000,
          10.000000000000000, 0.60000000000000009 },
  { 1.5820245752814948, 5.0000000000000000, 1.0000000000000000,
          10.000000000000000, 0.69999999999999996 },
  { 1.7421756366906538, 5.0000000000000000, 1.0000000000000000,
          10.000000000000000, 0.80000000000000004 },
  { 1.9513145531098233, 5.0000000000000000, 1.0000000000000000,
          10.000000000000000, 0.89999999999999991 },
};

// Test function for a=5.0000000000000000, b=1.0000000000000000, c=10.000000000000000.
template <typename Tp>
void test155()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data155)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data155[i].a), Tp(data155[i].b),
                   Tp(data155[i].c), Tp(data155[i].x));
      const Tp f0 = data155[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=5.0000000000000000, b=2.0000000000000000, c=2.0000000000000000.
testcase_hyperg<double> data156[] = {
  { 0.040386107340619266, 5.0000000000000000, 2.0000000000000000,
          2.0000000000000000, -0.90000000000000002 },
  { 0.052922149401344633, 5.0000000000000000, 2.0000000000000000,
          2.0000000000000000, -0.80000000000000004 },
  { 0.070429627772374270, 5.0000000000000000, 2.0000000000000000,
          2.0000000000000000, -0.69999999999999996 },
  { 0.095367431640624972, 5.0000000000000000, 2.0000000000000000,
          2.0000000000000000, -0.59999999999999998 },
  { 0.13168724279835387, 5.0000000000000000, 2.0000000000000000,
          2.0000000000000000, -0.50000000000000000 },
  { 0.18593443208187066, 5.0000000000000000, 2.0000000000000000,
          2.0000000000000000, -0.40000000000000002 },
  { 0.26932907434290437, 5.0000000000000000, 2.0000000000000000,
          2.0000000000000000, -0.30000000000000004 },
  { 0.40187757201646096, 5.0000000000000000, 2.0000000000000000,
          2.0000000000000000, -0.19999999999999996 },
  { 0.62092132305915493, 5.0000000000000000, 2.0000000000000000,
          2.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 5.0000000000000000, 2.0000000000000000,
          2.0000000000000000, 0.0000000000000000 },
  { 1.6935087808430296, 5.0000000000000000, 2.0000000000000000,
          2.0000000000000000, 0.10000000000000009 },
  { 3.0517578124999991, 5.0000000000000000, 2.0000000000000000,
          2.0000000000000000, 0.19999999999999996 },
  { 5.9499018266198629, 5.0000000000000000, 2.0000000000000000,
          2.0000000000000000, 0.30000000000000004 },
  { 12.860082304526737, 5.0000000000000000, 2.0000000000000000,
          2.0000000000000000, 0.39999999999999991 },
  { 32.000000000000000, 5.0000000000000000, 2.0000000000000000,
          2.0000000000000000, 0.50000000000000000 },
  { 97.656250000000114, 5.0000000000000000, 2.0000000000000000,
          2.0000000000000000, 0.60000000000000009 },
  { 411.52263374485580, 5.0000000000000000, 2.0000000000000000,
          2.0000000000000000, 0.69999999999999996 },
  { 3124.9999999999991, 5.0000000000000000, 2.0000000000000000,
          2.0000000000000000, 0.80000000000000004 },
  { 99999.999999999665, 5.0000000000000000, 2.0000000000000000,
          2.0000000000000000, 0.89999999999999991 },
};

// Test function for a=5.0000000000000000, b=2.0000000000000000, c=2.0000000000000000.
template <typename Tp>
void test156()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data156)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data156[i].a), Tp(data156[i].b),
                   Tp(data156[i].c), Tp(data156[i].x));
      const Tp f0 = data156[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=5.0000000000000000, b=2.0000000000000000, c=4.0000000000000000.
testcase_hyperg<double> data157[] = {
  { 0.21140107887447140, 5.0000000000000000, 2.0000000000000000,
          4.0000000000000000, -0.90000000000000002 },
  { 0.24005486968449935, 5.0000000000000000, 2.0000000000000000,
          4.0000000000000000, -0.80000000000000004 },
  { 0.27478119275391821, 5.0000000000000000, 2.0000000000000000,
          4.0000000000000000, -0.69999999999999996 },
  { 0.31738281249999994, 5.0000000000000000, 2.0000000000000000,
          4.0000000000000000, -0.59999999999999998 },
  { 0.37037037037037035, 5.0000000000000000, 2.0000000000000000,
          4.0000000000000000, -0.50000000000000000 },
  { 0.43731778425655982, 5.0000000000000000, 2.0000000000000000,
          4.0000000000000000, -0.40000000000000002 },
  { 0.52344105598543444, 5.0000000000000000, 2.0000000000000000,
          4.0000000000000000, -0.30000000000000004 },
  { 0.63657407407407429, 5.0000000000000000, 2.0000000000000000,
          4.0000000000000000, -0.19999999999999996 },
  { 0.78888054094665638, 5.0000000000000000, 2.0000000000000000,
          4.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 5.0000000000000000, 2.0000000000000000,
          4.0000000000000000, 0.0000000000000000 },
  { 1.3031550068587108, 5.0000000000000000, 2.0000000000000000,
          4.0000000000000000, 0.10000000000000009 },
  { 1.7578125000000002, 5.0000000000000000, 2.0000000000000000,
          4.0000000000000000, 0.19999999999999996 },
  { 2.4781341107871717, 5.0000000000000000, 2.0000000000000000,
          4.0000000000000000, 0.30000000000000004 },
  { 3.7037037037037037, 5.0000000000000000, 2.0000000000000000,
          4.0000000000000000, 0.39999999999999991 },
  { 5.9999999999999982, 5.0000000000000000, 2.0000000000000000,
          4.0000000000000000, 0.50000000000000000 },
  { 10.937500000000005, 5.0000000000000000, 2.0000000000000000,
          4.0000000000000000, 0.60000000000000009 },
  { 24.074074074074076, 5.0000000000000000, 2.0000000000000000,
          4.0000000000000000, 0.69999999999999996 },
  { 74.999999999999957, 5.0000000000000000, 2.0000000000000000,
          4.0000000000000000, 0.80000000000000004 },
  { 549.99999999999670, 5.0000000000000000, 2.0000000000000000,
          4.0000000000000000, 0.89999999999999991 },
};

// Test function for a=5.0000000000000000, b=2.0000000000000000, c=4.0000000000000000.
template <typename Tp>
void test157()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data157)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data157[i].a), Tp(data157[i].b),
                   Tp(data157[i].c), Tp(data157[i].x));
      const Tp f0 = data157[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=5.0000000000000000, b=2.0000000000000000, c=6.0000000000000000.
testcase_hyperg<double> data158[] = {
  { 0.33250915203252107, 5.0000000000000000, 2.0000000000000000,
          6.0000000000000000, -0.90000000000000002 },
  { 0.36566851047721960, 5.0000000000000000, 2.0000000000000000,
          6.0000000000000000, -0.80000000000000004 },
  { 0.40414812182437942, 5.0000000000000000, 2.0000000000000000,
          6.0000000000000000, -0.69999999999999996 },
  { 0.44916943268118470, 5.0000000000000000, 2.0000000000000000,
          6.0000000000000000, -0.59999999999999998 },
  { 0.50233081077479569, 5.0000000000000000, 2.0000000000000000,
          6.0000000000000000, -0.50000000000000000 },
  { 0.56575808728873322, 5.0000000000000000, 2.0000000000000000,
          6.0000000000000000, -0.40000000000000002 },
  { 0.64233106844971422, 5.0000000000000000, 2.0000000000000000,
          6.0000000000000000, -0.30000000000000004 },
  { 0.73603371116919514, 5.0000000000000000, 2.0000000000000000,
          6.0000000000000000, -0.19999999999999996 },
  { 0.85251256240112439, 5.0000000000000000, 2.0000000000000000,
          6.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 5.0000000000000000, 2.0000000000000000,
          6.0000000000000000, 0.0000000000000000 },
  { 1.1909065696197674, 5.0000000000000000, 2.0000000000000000,
          6.0000000000000000, 0.10000000000000009 },
  { 1.4447095285569311, 5.0000000000000000, 2.0000000000000000,
          6.0000000000000000, 0.19999999999999996 },
  { 1.7935243137840653, 5.0000000000000000, 2.0000000000000000,
          6.0000000000000000, 0.30000000000000004 },
  { 2.2937035820494454, 5.0000000000000000, 2.0000000000000000,
          6.0000000000000000, 0.39999999999999991 },
  { 3.0524711083016687, 5.0000000000000000, 2.0000000000000000,
          6.0000000000000000, 0.50000000000000000 },
  { 4.2976512669354259, 5.0000000000000000, 2.0000000000000000,
          6.0000000000000000, 0.60000000000000009 },
  { 6.5977107563194677, 5.0000000000000000, 2.0000000000000000,
          6.0000000000000000, 0.69999999999999996 },
  { 11.793747206577530, 5.0000000000000000, 2.0000000000000000,
          6.0000000000000000, 0.80000000000000004 },
  { 29.997625937982058, 5.0000000000000000, 2.0000000000000000,
          6.0000000000000000, 0.89999999999999991 },
};

// Test function for a=5.0000000000000000, b=2.0000000000000000, c=6.0000000000000000.
template <typename Tp>
void test158()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data158)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data158[i].a), Tp(data158[i].b),
                   Tp(data158[i].c), Tp(data158[i].x));
      const Tp f0 = data158[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=5.0000000000000000, b=2.0000000000000000, c=8.0000000000000000.
testcase_hyperg<double> data159[] = {
  { 0.42108197362250305, 5.0000000000000000, 2.0000000000000000,
          8.0000000000000000, -0.90000000000000002 },
  { 0.45503172013983051, 5.0000000000000000, 2.0000000000000000,
          8.0000000000000000, -0.80000000000000004 },
  { 0.49345609813624303, 5.0000000000000000, 2.0000000000000000,
          8.0000000000000000, -0.69999999999999996 },
  { 0.53720880551221295, 5.0000000000000000, 2.0000000000000000,
          8.0000000000000000, -0.59999999999999998 },
  { 0.58736431524847466, 5.0000000000000000, 2.0000000000000000,
          8.0000000000000000, -0.50000000000000000 },
  { 0.64529222467897962, 5.0000000000000000, 2.0000000000000000,
          8.0000000000000000, -0.40000000000000002 },
  { 0.71276337354393904, 5.0000000000000000, 2.0000000000000000,
          8.0000000000000000, -0.30000000000000004 },
  { 0.79210466220795306, 5.0000000000000000, 2.0000000000000000,
          8.0000000000000000, -0.19999999999999996 },
  { 0.88643063455510596, 5.0000000000000000, 2.0000000000000000,
          8.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 5.0000000000000000, 2.0000000000000000,
          8.0000000000000000, 0.0000000000000000 },
  { 1.1387832139040652, 5.0000000000000000, 2.0000000000000000,
          8.0000000000000000, 0.10000000000000009 },
  { 1.3114025920844752, 5.0000000000000000, 2.0000000000000000,
          8.0000000000000000, 0.19999999999999996 },
  { 1.5307655016768162, 5.0000000000000000, 2.0000000000000000,
          8.0000000000000000, 0.30000000000000004 },
  { 1.8170727950333345, 5.0000000000000000, 2.0000000000000000,
          8.0000000000000000, 0.39999999999999991 },
  { 2.2037865486700836, 5.0000000000000000, 2.0000000000000000,
          8.0000000000000000, 0.50000000000000000 },
  { 2.7506766056439380, 5.0000000000000000, 2.0000000000000000,
          8.0000000000000000, 0.60000000000000009 },
  { 3.5764534935716972, 5.0000000000000000, 2.0000000000000000,
          8.0000000000000000, 0.69999999999999996 },
  { 4.9587762302155403, 5.0000000000000000, 2.0000000000000000,
          8.0000000000000000, 0.80000000000000004 },
  { 7.7740847924166800, 5.0000000000000000, 2.0000000000000000,
          8.0000000000000000, 0.89999999999999991 },
};

// Test function for a=5.0000000000000000, b=2.0000000000000000, c=8.0000000000000000.
template <typename Tp>
void test159()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data159)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data159[i].a), Tp(data159[i].b),
                   Tp(data159[i].c), Tp(data159[i].x));
      const Tp f0 = data159[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=5.0000000000000000, b=2.0000000000000000, c=10.000000000000000.
testcase_hyperg<double> data160[] = {
  { 0.48860241312958425, 5.0000000000000000, 2.0000000000000000,
          10.000000000000000, -0.90000000000000002 },
  { 0.52193382517068487, 5.0000000000000000, 2.0000000000000000,
          10.000000000000000, -0.80000000000000004 },
  { 0.55902375003954219, 5.0000000000000000, 2.0000000000000000,
          10.000000000000000, -0.69999999999999996 },
  { 0.60049055150230346, 5.0000000000000000, 2.0000000000000000,
          10.000000000000000, -0.59999999999999998 },
  { 0.64709127927203480, 5.0000000000000000, 2.0000000000000000,
          10.000000000000000, -0.50000000000000000 },
  { 0.69976233335368998, 5.0000000000000000, 2.0000000000000000,
          10.000000000000000, -0.40000000000000002 },
  { 0.75967529501081055, 5.0000000000000000, 2.0000000000000000,
          10.000000000000000, -0.30000000000000004 },
  { 0.82831498895254407, 5.0000000000000000, 2.0000000000000000,
          10.000000000000000, -0.19999999999999996 },
  { 0.90759090169653933, 5.0000000000000000, 2.0000000000000000,
          10.000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 5.0000000000000000, 2.0000000000000000,
          10.000000000000000, 0.0000000000000000 },
  { 1.1088712278667465, 5.0000000000000000, 2.0000000000000000,
          10.000000000000000, 0.10000000000000009 },
  { 1.2387445478440853, 5.0000000000000000, 2.0000000000000000,
          10.000000000000000, 0.19999999999999996 },
  { 1.3959812720437546, 5.0000000000000000, 2.0000000000000000,
          10.000000000000000, 0.30000000000000004 },
  { 1.5897930661091164, 5.0000000000000000, 2.0000000000000000,
          10.000000000000000, 0.39999999999999991 },
  { 1.8340789380307454, 5.0000000000000000, 2.0000000000000000,
          10.000000000000000, 0.50000000000000000 },
  { 2.1509548085970764, 5.0000000000000000, 2.0000000000000000,
          10.000000000000000, 0.60000000000000009 },
  { 2.5782406951207504, 5.0000000000000000, 2.0000000000000000,
          10.000000000000000, 0.69999999999999996 },
  { 3.1877847194242737, 5.0000000000000000, 2.0000000000000000,
          10.000000000000000, 0.80000000000000004 },
  { 4.1421596631676900, 5.0000000000000000, 2.0000000000000000,
          10.000000000000000, 0.89999999999999991 },
};

// Test function for a=5.0000000000000000, b=2.0000000000000000, c=10.000000000000000.
template <typename Tp>
void test160()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data160)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data160[i].a), Tp(data160[i].b),
                   Tp(data160[i].c), Tp(data160[i].x));
      const Tp f0 = data160[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=5.0000000000000000, b=5.0000000000000000, c=2.0000000000000000.
testcase_hyperg<double> data161[] = {
  { -0.0047236848832209691, 5.0000000000000000, 5.0000000000000000,
          2.0000000000000000, -0.90000000000000002 },
  { -0.0073321496427103212, 5.0000000000000000, 5.0000000000000000,
          2.0000000000000000, -0.80000000000000004 },
  { -0.010977302557845672, 5.0000000000000000, 5.0000000000000000,
          2.0000000000000000, -0.69999999999999996 },
  { -0.015692785382270907, 5.0000000000000000, 5.0000000000000000,
          2.0000000000000000, -0.59999999999999998 },
  { -0.020728547477518677, 5.0000000000000000, 5.0000000000000000,
          2.0000000000000000, -0.50000000000000000 },
  { -0.022767481479412769, 5.0000000000000000, 5.0000000000000000,
          2.0000000000000000, -0.40000000000000002 },
  { -0.010634636868114139, 5.0000000000000000, 5.0000000000000000,
          2.0000000000000000, -0.30000000000000004 },
  { 0.050699832580781923, 5.0000000000000000, 5.0000000000000000,
          2.0000000000000000, -0.19999999999999996 },
  { 0.27045765367659280, 5.0000000000000000, 5.0000000000000000,
          2.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 5.0000000000000000, 5.0000000000000000,
          2.0000000000000000, 0.0000000000000000 },
  { 3.4387055868901171, 5.0000000000000000, 5.0000000000000000,
          2.0000000000000000, 0.10000000000000009 },
  { 12.052059173583981, 5.0000000000000000, 5.0000000000000000,
          2.0000000000000000, 0.19999999999999996 },
  { 45.565319600798020, 5.0000000000000000, 5.0000000000000000,
          2.0000000000000000, 0.30000000000000004 },
  { 196.23532998018572, 5.0000000000000000, 5.0000000000000000,
          2.0000000000000000, 0.39999999999999991 },
  { 1032.0000000000002, 5.0000000000000000, 5.0000000000000000,
          2.0000000000000000, 0.50000000000000000 },
  { 7376.0986328125073, 5.0000000000000000, 5.0000000000000000,
          2.0000000000000000, 0.60000000000000009 },
  { 86964.639536655843, 5.0000000000000000, 5.0000000000000000,
          2.0000000000000000, 0.69999999999999996 },
  { 2596875.0000000009, 5.0000000000000000, 5.0000000000000000,
          2.0000000000000000, 0.80000000000000004 },
  { 766224999.99999273, 5.0000000000000000, 5.0000000000000000,
          2.0000000000000000, 0.89999999999999991 },
};

// Test function for a=5.0000000000000000, b=5.0000000000000000, c=2.0000000000000000.
template <typename Tp>
void test161()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data161)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data161[i].a), Tp(data161[i].b),
                   Tp(data161[i].c), Tp(data161[i].x));
      const Tp f0 = data161[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000015e-12));
}

// Test data for a=5.0000000000000000, b=5.0000000000000000, c=4.0000000000000000.
testcase_hyperg<double> data162[] = {
  { 0.016473280625778897, 5.0000000000000000, 5.0000000000000000,
          4.0000000000000000, -0.90000000000000002 },
  { 0.023520955289486407, 5.0000000000000000, 5.0000000000000000,
          4.0000000000000000, -0.80000000000000004 },
  { 0.034179084066004943, 5.0000000000000000, 5.0000000000000000,
          4.0000000000000000, -0.69999999999999996 },
  { 0.050663948059082052, 5.0000000000000000, 5.0000000000000000,
          4.0000000000000000, -0.59999999999999998 },
  { 0.076817558299039843, 5.0000000000000000, 5.0000000000000000,
          4.0000000000000000, -0.50000000000000000 },
  { 0.11952927776691676, 5.0000000000000000, 5.0000000000000000,
          4.0000000000000000, -0.40000000000000002 },
  { 0.19163799520552813, 5.0000000000000000, 5.0000000000000000,
          4.0000000000000000, -0.30000000000000004 },
  { 0.31815307784636504, 5.0000000000000000, 5.0000000000000000,
          4.0000000000000000, -0.19999999999999996 },
  { 0.55036208180243285, 5.0000000000000000, 5.0000000000000000,
          4.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 5.0000000000000000, 5.0000000000000000,
          4.0000000000000000, 0.0000000000000000 },
  { 1.9287183337378946, 5.0000000000000000, 5.0000000000000000,
          4.0000000000000000, 0.10000000000000009 },
  { 4.0054321289062473, 5.0000000000000000, 5.0000000000000000,
          4.0000000000000000, 0.19999999999999996 },
  { 9.1373492337376394, 5.0000000000000000, 5.0000000000000000,
          4.0000000000000000, 0.30000000000000004 },
  { 23.576817558299005, 5.0000000000000000, 5.0000000000000000,
          4.0000000000000000, 0.39999999999999991 },
  { 71.999999999999972, 5.0000000000000000, 5.0000000000000000,
          4.0000000000000000, 0.50000000000000000 },
  { 280.76171875000023, 5.0000000000000000, 5.0000000000000000,
          4.0000000000000000, 0.60000000000000009 },
  { 1611.7969821673514, 5.0000000000000000, 5.0000000000000000,
          4.0000000000000000, 0.69999999999999996 },
  { 18749.999999999996, 5.0000000000000000, 5.0000000000000000,
          4.0000000000000000, 0.80000000000000004 },
  { 1224999.9999999879, 5.0000000000000000, 5.0000000000000000,
          4.0000000000000000, 0.89999999999999991 },
};

// Test function for a=5.0000000000000000, b=5.0000000000000000, c=4.0000000000000000.
template <typename Tp>
void test162()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data162)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data162[i].a), Tp(data162[i].b),
                   Tp(data162[i].c), Tp(data162[i].x));
      const Tp f0 = data162[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(5.0000000000000039e-13));
}

// Test data for a=5.0000000000000000, b=5.0000000000000000, c=6.0000000000000000.
testcase_hyperg<double> data163[] = {
  { 0.067462409738203513, 5.0000000000000000, 5.0000000000000000,
          6.0000000000000000, -0.90000000000000002 },
  { 0.084813629887172517, 5.0000000000000000, 5.0000000000000000,
          6.0000000000000000, -0.80000000000000004 },
  { 0.10799223563666410, 5.0000000000000000, 5.0000000000000000,
          6.0000000000000000, -0.69999999999999996 },
  { 0.13947766136095369, 5.0000000000000000, 5.0000000000000000,
          6.0000000000000000, -0.59999999999999998 },
  { 0.18305927261494304, 5.0000000000000000, 5.0000000000000000,
          6.0000000000000000, -0.50000000000000000 },
  { 0.24468431546783445, 5.0000000000000000, 5.0000000000000000,
          6.0000000000000000, -0.40000000000000002 },
  { 0.33397274564972956, 5.0000000000000000, 5.0000000000000000,
          6.0000000000000000, -0.30000000000000004 },
  { 0.46703323887436765, 5.0000000000000000, 5.0000000000000000,
          6.0000000000000000, -0.19999999999999996 },
  { 0.67194346197695642, 5.0000000000000000, 5.0000000000000000,
          6.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 5.0000000000000000, 5.0000000000000000,
          6.0000000000000000, 0.0000000000000000 },
  { 1.5503148146900136, 5.0000000000000000, 5.0000000000000000,
          6.0000000000000000, 0.10000000000000009 },
  { 2.5278200136940998, 5.0000000000000000, 5.0000000000000000,
          6.0000000000000000, 0.19999999999999996 },
  { 4.3933515329658954, 5.0000000000000000, 5.0000000000000000,
          6.0000000000000000, 0.30000000000000004 },
  { 8.3000308946110888, 5.0000000000000000, 5.0000000000000000,
          6.0000000000000000, 0.39999999999999991 },
  { 17.570215556257921, 5.0000000000000000, 5.0000000000000000,
          6.0000000000000000, 0.50000000000000000 },
  { 43.847462183266167, 5.0000000000000000, 5.0000000000000000,
          6.0000000000000000, 0.60000000000000009 },
  { 141.86909082943853, 5.0000000000000000, 5.0000000000000000,
          6.0000000000000000, 0.69999999999999996 },
  { 736.63489653168926, 5.0000000000000000, 5.0000000000000000,
          6.0000000000000000, 0.80000000000000004 },
  { 12117.500593515439, 5.0000000000000000, 5.0000000000000000,
          6.0000000000000000, 0.89999999999999991 },
};

// Test function for a=5.0000000000000000, b=5.0000000000000000, c=6.0000000000000000.
template <typename Tp>
void test163()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data163)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data163[i].a), Tp(data163[i].b),
                   Tp(data163[i].c), Tp(data163[i].x));
      const Tp f0 = data163[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=5.0000000000000000, b=5.0000000000000000, c=8.0000000000000000.
testcase_hyperg<double> data164[] = {
  { 0.12409443806004232, 5.0000000000000000, 5.0000000000000000,
          8.0000000000000000, -0.90000000000000002 },
  { 0.14886910375100412, 5.0000000000000000, 5.0000000000000000,
          8.0000000000000000, -0.80000000000000004 },
  { 0.18023328876836334, 5.0000000000000000, 5.0000000000000000,
          8.0000000000000000, -0.69999999999999996 },
  { 0.22044046981094714, 5.0000000000000000, 5.0000000000000000,
          8.0000000000000000, -0.59999999999999998 },
  { 0.27271160690708801, 5.0000000000000000, 5.0000000000000000,
          8.0000000000000000, -0.50000000000000000 },
  { 0.34174821195025840, 5.0000000000000000, 5.0000000000000000,
          8.0000000000000000, -0.40000000000000002 },
  { 0.43457788826160237, 5.0000000000000000, 5.0000000000000000,
          8.0000000000000000, -0.30000000000000004 },
  { 0.56199385898404552, 5.0000000000000000, 5.0000000000000000,
          8.0000000000000000, -0.19999999999999996 },
  { 0.74109892753745221, 5.0000000000000000, 5.0000000000000000,
          8.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 5.0000000000000000, 5.0000000000000000,
          8.0000000000000000, 0.0000000000000000 },
  { 1.3869229400096228, 5.0000000000000000, 5.0000000000000000,
          8.0000000000000000, 0.10000000000000009 },
  { 1.9890168748121255, 5.0000000000000000, 5.0000000000000000,
          8.0000000000000000, 0.19999999999999996 },
  { 2.9741205609307424, 5.0000000000000000, 5.0000000000000000,
          8.0000000000000000, 0.30000000000000004 },
  { 4.6924751038237300, 5.0000000000000000, 5.0000000000000000,
          8.0000000000000000, 0.39999999999999991 },
  { 7.9555939380658254, 5.0000000000000000, 5.0000000000000000,
          8.0000000000000000, 0.50000000000000000 },
  { 14.933102063314404, 5.0000000000000000, 5.0000000000000000,
          8.0000000000000000, 0.60000000000000009 },
  { 32.780461638447491, 5.0000000000000000, 5.0000000000000000,
          8.0000000000000000, 0.69999999999999996 },
  { 94.848124287773530, 5.0000000000000000, 5.0000000000000000,
          8.0000000000000000, 0.80000000000000004 },
  { 523.16034401517425, 5.0000000000000000, 5.0000000000000000,
          8.0000000000000000, 0.89999999999999991 },
};

// Test function for a=5.0000000000000000, b=5.0000000000000000, c=8.0000000000000000.
template <typename Tp>
void test164()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data164)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data164[i].a), Tp(data164[i].b),
                   Tp(data164[i].c), Tp(data164[i].x));
      const Tp f0 = data164[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=5.0000000000000000, b=5.0000000000000000, c=10.000000000000000.
testcase_hyperg<double> data165[] = {
  { 0.17885405888526873, 5.0000000000000000, 5.0000000000000000,
          10.000000000000000, -0.90000000000000002 },
  { 0.20861302518993380, 5.0000000000000000, 5.0000000000000000,
          10.000000000000000, -0.80000000000000004 },
  { 0.24504033307244924, 5.0000000000000000, 5.0000000000000000,
          10.000000000000000, -0.69999999999999996 },
  { 0.29007236051133478, 5.0000000000000000, 5.0000000000000000,
          10.000000000000000, -0.59999999999999998 },
  { 0.34635542859732726, 5.0000000000000000, 5.0000000000000000,
          10.000000000000000, -0.50000000000000000 },
  { 0.41756858504598376, 5.0000000000000000, 5.0000000000000000,
          10.000000000000000, -0.40000000000000002 },
  { 0.50892615622124382, 5.0000000000000000, 5.0000000000000000,
          10.000000000000000, -0.30000000000000004 },
  { 0.62798173270509761, 5.0000000000000000, 5.0000000000000000,
          10.000000000000000, -0.19999999999999996 },
  { 0.78595487360378424, 5.0000000000000000, 5.0000000000000000,
          10.000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 5.0000000000000000, 5.0000000000000000,
          10.000000000000000, 0.0000000000000000 },
  { 1.2972517637384813, 5.0000000000000000, 5.0000000000000000,
          10.000000000000000, 0.10000000000000009 },
  { 1.7224028197396388, 5.0000000000000000, 5.0000000000000000,
          10.000000000000000, 0.19999999999999996 },
  { 2.3527690438263305, 5.0000000000000000, 5.0000000000000000,
          10.000000000000000, 0.30000000000000004 },
  { 3.3305218060101116, 5.0000000000000000, 5.0000000000000000,
          10.000000000000000, 0.39999999999999991 },
  { 4.9383884076775466, 5.0000000000000000, 5.0000000000000000,
          10.000000000000000, 0.50000000000000000 },
  { 7.8007604680775229, 5.0000000000000000, 5.0000000000000000,
          10.000000000000000, 0.60000000000000009 },
  { 13.518663719271885, 5.0000000000000000, 5.0000000000000000,
          10.000000000000000, 0.69999999999999996 },
  { 27.285345906502567, 5.0000000000000000, 5.0000000000000000,
          10.000000000000000, 0.80000000000000004 },
  { 75.572415101501988, 5.0000000000000000, 5.0000000000000000,
          10.000000000000000, 0.89999999999999991 },
};

// Test function for a=5.0000000000000000, b=5.0000000000000000, c=10.000000000000000.
template <typename Tp>
void test165()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data165)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data165[i].a), Tp(data165[i].b),
                   Tp(data165[i].c), Tp(data165[i].x));
      const Tp f0 = data165[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=5.0000000000000000, b=10.000000000000000, c=2.0000000000000000.
testcase_hyperg<double> data166[] = {
  { 0.00063586451658060152, 5.0000000000000000, 10.000000000000000,
          2.0000000000000000, -0.90000000000000002 },
  { 0.0010334743461762443, 5.0000000000000000, 10.000000000000000,
          2.0000000000000000, -0.80000000000000004 },
  { 0.0015326246054669515, 5.0000000000000000, 10.000000000000000,
          2.0000000000000000, -0.69999999999999996 },
  { 0.0019007018181583387, 5.0000000000000000, 10.000000000000000,
          2.0000000000000000, -0.59999999999999998 },
  { 0.0012845577715431577, 5.0000000000000000, 10.000000000000000,
          2.0000000000000000, -0.50000000000000000 },
  { -0.0027213806178058826, 5.0000000000000000, 10.000000000000000,
          2.0000000000000000, -0.40000000000000002 },
  { -0.015121744574954068, 5.0000000000000000, 10.000000000000000,
          2.0000000000000000, -0.30000000000000004 },
  { -0.036637840562974443, 5.0000000000000000, 10.000000000000000,
          2.0000000000000000, -0.19999999999999996 },
  { 0.019117849062621491, 5.0000000000000000, 10.000000000000000,
          2.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 5.0000000000000000, 10.000000000000000,
          2.0000000000000000, 0.0000000000000000 },
  { 9.8116901852350615, 5.0000000000000000, 10.000000000000000,
          2.0000000000000000, 0.10000000000000009 },
  { 84.255589172244044, 5.0000000000000000, 10.000000000000000,
          2.0000000000000000, 0.19999999999999996 },
  { 773.87517619421294, 5.0000000000000000, 10.000000000000000,
          2.0000000000000000, 0.30000000000000004 },
  { 8556.9725363053585, 5.0000000000000000, 10.000000000000000,
          2.0000000000000000, 0.39999999999999991 },
  { 129023.99999999996, 5.0000000000000000, 10.000000000000000,
          2.0000000000000000, 0.50000000000000000 },
  { 3174543.3807373112, 5.0000000000000000, 10.000000000000000,
          2.0000000000000000, 0.60000000000000009 },
  { 175133896.95814410, 5.0000000000000000, 10.000000000000000,
          2.0000000000000000, 0.69999999999999996 },
  { 43564453125.000061, 5.0000000000000000, 10.000000000000000,
          2.0000000000000000, 0.80000000000000004 },
  { 446859999999993.50, 5.0000000000000000, 10.000000000000000,
          2.0000000000000000, 0.89999999999999991 },
};

// Test function for a=5.0000000000000000, b=10.000000000000000, c=2.0000000000000000.
template <typename Tp>
void test166()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data166)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data166[i].a), Tp(data166[i].b),
                   Tp(data166[i].c), Tp(data166[i].x));
      const Tp f0 = data166[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(1.0000000000000006e-11));
}

// Test data for a=5.0000000000000000, b=10.000000000000000, c=4.0000000000000000.
testcase_hyperg<double> data167[] = {
  { -0.00030045430691819899, 5.0000000000000000, 10.000000000000000,
          4.0000000000000000, -0.90000000000000002 },
  { -0.00031119487747328581, 5.0000000000000000, 10.000000000000000,
          4.0000000000000000, -0.80000000000000004 },
  { -0.00014589213141649274, 5.0000000000000000, 10.000000000000000,
          4.0000000000000000, -0.69999999999999996 },
  { 0.00056843418860809121, 5.0000000000000000, 10.000000000000000,
          4.0000000000000000, -0.59999999999999998 },
  { 0.0028902549859721725, 5.0000000000000000, 10.000000000000000,
          4.0000000000000000, -0.50000000000000000 },
  { 0.0098776037238877470, 5.0000000000000000, 10.000000000000000,
          4.0000000000000000, -0.40000000000000002 },
  { 0.030689217428863914, 5.0000000000000000, 10.000000000000000,
          4.0000000000000000, -0.30000000000000004 },
  { 0.094211590019076599, 5.0000000000000000, 10.000000000000000,
          4.0000000000000000, -0.19999999999999996 },
  { 0.29791981455918376, 5.0000000000000000, 10.000000000000000,
          4.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 5.0000000000000000, 10.000000000000000,
          4.0000000000000000, 0.0000000000000000 },
  { 3.6646308771236793, 5.0000000000000000, 10.000000000000000,
          4.0000000000000000, 0.10000000000000009 },
  { 15.133991837501521, 5.0000000000000000, 10.000000000000000,
          4.0000000000000000, 0.19999999999999996 },
  { 73.331330046144089, 5.0000000000000000, 10.000000000000000,
          4.0000000000000000, 0.30000000000000004 },
  { 441.01791167787133, 5.0000000000000000, 10.000000000000000,
          4.0000000000000000, 0.39999999999999991 },
  { 3583.9999999999991, 5.0000000000000000, 10.000000000000000,
          4.0000000000000000, 0.50000000000000000 },
  { 45299.530029296984, 5.0000000000000000, 10.000000000000000,
          4.0000000000000000, 0.60000000000000009 },
  { 1157231.0002427341, 5.0000000000000000, 10.000000000000000,
          4.0000000000000000, 0.69999999999999996 },
  { 107421875.00000016, 5.0000000000000000, 10.000000000000000,
          4.0000000000000000, 0.80000000000000004 },
  { 234999999999.99734, 5.0000000000000000, 10.000000000000000,
          4.0000000000000000, 0.89999999999999991 },
};

// Test function for a=5.0000000000000000, b=10.000000000000000, c=4.0000000000000000.
template <typename Tp>
void test167()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data167)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data167[i].a), Tp(data167[i].b),
                   Tp(data167[i].c), Tp(data167[i].x));
      const Tp f0 = data167[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000014e-11));
}

// Test data for a=5.0000000000000000, b=10.000000000000000, c=6.0000000000000000.
testcase_hyperg<double> data168[] = {
  { 0.0058530497315411210, 5.0000000000000000, 10.000000000000000,
          6.0000000000000000, -0.90000000000000002 },
  { 0.0088526869356855692, 5.0000000000000000, 10.000000000000000,
          6.0000000000000000, -0.80000000000000004 },
  { 0.013770987983443108, 5.0000000000000000, 10.000000000000000,
          6.0000000000000000, -0.69999999999999996 },
  { 0.022108932690960800, 5.0000000000000000, 10.000000000000000,
          6.0000000000000000, -0.59999999999999998 },
  { 0.036786236450921578, 5.0000000000000000, 10.000000000000000,
          6.0000000000000000, -0.50000000000000000 },
  { 0.063750669040426422, 5.0000000000000000, 10.000000000000000,
          6.0000000000000000, -0.40000000000000002 },
  { 0.11577228680714464, 5.0000000000000000, 10.000000000000000,
          6.0000000000000000, -0.30000000000000004 },
  { 0.22197573416125735, 5.0000000000000000, 10.000000000000000,
          6.0000000000000000, -0.19999999999999996 },
  { 0.45361312968415268, 5.0000000000000000, 10.000000000000000,
          6.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 5.0000000000000000, 10.000000000000000,
          6.0000000000000000, 0.0000000000000000 },
  { 2.4162889363082747, 5.0000000000000000, 10.000000000000000,
          6.0000000000000000, 0.10000000000000009 },
  { 6.5381564791240399, 5.0000000000000000, 10.000000000000000,
          6.0000000000000000, 0.19999999999999996 },
  { 20.415771011498428, 5.0000000000000000, 10.000000000000000,
          6.0000000000000000, 0.30000000000000004 },
  { 76.870682056629221, 5.0000000000000000, 10.000000000000000,
          6.0000000000000000, 0.39999999999999991 },
  { 373.58730158730162, 5.0000000000000000, 10.000000000000000,
          6.0000000000000000, 0.50000000000000000 },
  { 2626.2555803571477, 5.0000000000000000, 10.000000000000000,
          6.0000000000000000, 0.60000000000000009 },
  { 33060.960671081048, 5.0000000000000000, 10.000000000000000,
          6.0000000000000000, 0.69999999999999996 },
  { 1203521.8253968258, 5.0000000000000000, 10.000000000000000,
          6.0000000000000000, 0.80000000000000004 },
  { 584564285.71427989, 5.0000000000000000, 10.000000000000000,
          6.0000000000000000, 0.89999999999999991 },
};

// Test function for a=5.0000000000000000, b=10.000000000000000, c=6.0000000000000000.
template <typename Tp>
void test168()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data168)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data168[i].a), Tp(data168[i].b),
                   Tp(data168[i].c), Tp(data168[i].x));
      const Tp f0 = data168[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000015e-12));
}

// Test data for a=5.0000000000000000, b=10.000000000000000, c=8.0000000000000000.
testcase_hyperg<double> data169[] = {
  { 0.020248990107069400, 5.0000000000000000, 10.000000000000000,
          8.0000000000000000, -0.90000000000000002 },
  { 0.027876687750502421, 5.0000000000000000, 10.000000000000000,
          8.0000000000000000, -0.80000000000000004 },
  { 0.039154648888447781, 5.0000000000000000, 10.000000000000000,
          8.0000000000000000, -0.69999999999999996 },
  { 0.056251883506774923, 5.0000000000000000, 10.000000000000000,
          8.0000000000000000, -0.59999999999999998 },
  { 0.082914189910074432, 5.0000000000000000, 10.000000000000000,
          8.0000000000000000, -0.50000000000000000 },
  { 0.12585357817786477, 5.0000000000000000, 10.000000000000000,
          8.0000000000000000, -0.40000000000000002 },
  { 0.19761423206224929, 5.0000000000000000, 10.000000000000000,
          8.0000000000000000, -0.30000000000000004 },
  { 0.32280443863359243, 5.0000000000000000, 10.000000000000000,
          8.0000000000000000, -0.19999999999999996 },
  { 0.55250024062839465, 5.0000000000000000, 10.000000000000000,
          8.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 5.0000000000000000, 10.000000000000000,
          8.0000000000000000, 0.0000000000000000 },
  { 1.9374297986599267, 5.0000000000000000, 10.000000000000000,
          8.0000000000000000, 0.10000000000000009 },
  { 4.0849049886067696, 5.0000000000000000, 10.000000000000000,
          8.0000000000000000, 0.19999999999999996 },
  { 9.5926988633258983, 5.0000000000000000, 10.000000000000000,
          8.0000000000000000, 0.30000000000000004 },
  { 25.958314281359531, 5.0000000000000000, 10.000000000000000,
          8.0000000000000000, 0.39999999999999991 },
  { 85.333333333333300, 5.0000000000000000, 10.000000000000000,
          8.0000000000000000, 0.50000000000000000 },
  { 372.31445312500028, 5.0000000000000000, 10.000000000000000,
          8.0000000000000000, 0.60000000000000009 },
  { 2545.3436976070675, 5.0000000000000000, 10.000000000000000,
          8.0000000000000000, 0.69999999999999996 },
  { 39583.333333333343, 5.0000000000000000, 10.000000000000000,
          8.0000000000000000, 0.80000000000000004 },
  { 4599999.9999999627, 5.0000000000000000, 10.000000000000000,
          8.0000000000000000, 0.89999999999999991 },
};

// Test function for a=5.0000000000000000, b=10.000000000000000, c=8.0000000000000000.
template <typename Tp>
void test169()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data169)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data169[i].a), Tp(data169[i].b),
                   Tp(data169[i].c), Tp(data169[i].x));
      const Tp f0 = data169[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(5.0000000000000039e-13));
}

// Test data for a=5.0000000000000000, b=10.000000000000000, c=10.000000000000000.
testcase_hyperg<double> data170[] = {
  { 0.040386107340619266, 5.0000000000000000, 10.000000000000000,
          10.000000000000000, -0.90000000000000002 },
  { 0.052922149401344633, 5.0000000000000000, 10.000000000000000,
          10.000000000000000, -0.80000000000000004 },
  { 0.070429627772374270, 5.0000000000000000, 10.000000000000000,
          10.000000000000000, -0.69999999999999996 },
  { 0.095367431640624972, 5.0000000000000000, 10.000000000000000,
          10.000000000000000, -0.59999999999999998 },
  { 0.13168724279835387, 5.0000000000000000, 10.000000000000000,
          10.000000000000000, -0.50000000000000000 },
  { 0.18593443208187066, 5.0000000000000000, 10.000000000000000,
          10.000000000000000, -0.40000000000000002 },
  { 0.26932907434290437, 5.0000000000000000, 10.000000000000000,
          10.000000000000000, -0.30000000000000004 },
  { 0.40187757201646096, 5.0000000000000000, 10.000000000000000,
          10.000000000000000, -0.19999999999999996 },
  { 0.62092132305915493, 5.0000000000000000, 10.000000000000000,
          10.000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 5.0000000000000000, 10.000000000000000,
          10.000000000000000, 0.0000000000000000 },
  { 1.6935087808430296, 5.0000000000000000, 10.000000000000000,
          10.000000000000000, 0.10000000000000009 },
  { 3.0517578124999991, 5.0000000000000000, 10.000000000000000,
          10.000000000000000, 0.19999999999999996 },
  { 5.9499018266198629, 5.0000000000000000, 10.000000000000000,
          10.000000000000000, 0.30000000000000004 },
  { 12.860082304526737, 5.0000000000000000, 10.000000000000000,
          10.000000000000000, 0.39999999999999991 },
  { 32.000000000000000, 5.0000000000000000, 10.000000000000000,
          10.000000000000000, 0.50000000000000000 },
  { 97.656250000000114, 5.0000000000000000, 10.000000000000000,
          10.000000000000000, 0.60000000000000009 },
  { 411.52263374485580, 5.0000000000000000, 10.000000000000000,
          10.000000000000000, 0.69999999999999996 },
  { 3124.9999999999991, 5.0000000000000000, 10.000000000000000,
          10.000000000000000, 0.80000000000000004 },
  { 99999.999999999665, 5.0000000000000000, 10.000000000000000,
          10.000000000000000, 0.89999999999999991 },
};

// Test function for a=5.0000000000000000, b=10.000000000000000, c=10.000000000000000.
template <typename Tp>
void test170()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data170)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data170[i].a), Tp(data170[i].b),
                   Tp(data170[i].c), Tp(data170[i].x));
      const Tp f0 = data170[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=5.0000000000000000, b=20.000000000000000, c=2.0000000000000000.
testcase_hyperg<double> data171[] = {
  { -1.8650300348791041e-05, 5.0000000000000000, 20.000000000000000,
          2.0000000000000000, -0.90000000000000002 },
  { -3.6488008415183135e-05, 5.0000000000000000, 20.000000000000000,
          2.0000000000000000, -0.80000000000000004 },
  { -6.4614776410999025e-05, 5.0000000000000000, 20.000000000000000,
          2.0000000000000000, -0.69999999999999996 },
  { -8.4495207102575916e-05, 5.0000000000000000, 20.000000000000000,
          2.0000000000000000, -0.59999999999999998 },
  { 2.2276197023819217e-05, 5.0000000000000000, 20.000000000000000,
          2.0000000000000000, -0.50000000000000000 },
  { 0.00070736115111467578, 5.0000000000000000, 20.000000000000000,
          2.0000000000000000, -0.40000000000000002 },
  { 0.0027829732057272809, 5.0000000000000000, 20.000000000000000,
          2.0000000000000000, -0.30000000000000004 },
  { 0.0013283545664373570, 5.0000000000000000, 20.000000000000000,
          2.0000000000000000, -0.19999999999999996 },
  { -0.041767631015048733, 5.0000000000000000, 20.000000000000000,
          2.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 5.0000000000000000, 20.000000000000000,
          2.0000000000000000, 0.0000000000000000 },
  { 61.311496556100003, 5.0000000000000000, 20.000000000000000,
          2.0000000000000000, 0.10000000000000009 },
  { 2397.4420539085681, 5.0000000000000000, 20.000000000000000,
          2.0000000000000000, 0.19999999999999996 },
  { 103687.60998586559, 5.0000000000000000, 20.000000000000000,
          2.0000000000000000, 0.30000000000000004 },
  { 6247196.6451068865, 5.0000000000000000, 20.000000000000000,
          2.0000000000000000, 0.39999999999999991 },
  { 656408576.00000000, 5.0000000000000000, 20.000000000000000,
          2.0000000000000000, 0.50000000000000000 },
  { 165334768098.54715, 5.0000000000000000, 20.000000000000000,
          2.0000000000000000, 0.60000000000000009 },
  { 175097125520816.81, 5.0000000000000000, 20.000000000000000,
          2.0000000000000000, 0.69999999999999996 },
  { 2.6818275451660257e+18, 5.0000000000000000, 20.000000000000000,
          2.0000000000000000, 0.80000000000000004 },
  { 2.9794599999999321e+25, 5.0000000000000000, 20.000000000000000,
          2.0000000000000000, 0.89999999999999991 },
};

// Test function for a=5.0000000000000000, b=20.000000000000000, c=2.0000000000000000.
template <typename Tp>
void test171()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data171)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data171[i].a), Tp(data171[i].b),
                   Tp(data171[i].c), Tp(data171[i].x));
      const Tp f0 = data171[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(5.0000000000000034e-10));
}

// Test data for a=5.0000000000000000, b=20.000000000000000, c=4.0000000000000000.
testcase_hyperg<double> data172[] = {
  { -3.6403884515183385e-06, 5.0000000000000000, 20.000000000000000,
          4.0000000000000000, -0.90000000000000002 },
  { -9.5873829247725586e-06, 5.0000000000000000, 20.000000000000000,
          4.0000000000000000, -0.80000000000000004 },
  { -2.6052245147354694e-05, 5.0000000000000000, 20.000000000000000,
          4.0000000000000000, -0.69999999999999996 },
  { -7.2378303598294010e-05, 5.0000000000000000, 20.000000000000000,
          4.0000000000000000, -0.59999999999999998 },
  { -0.00020048577321454082, 5.0000000000000000, 20.000000000000000,
          4.0000000000000000, -0.50000000000000000 },
  { -0.00051222704046236022, 5.0000000000000000, 20.000000000000000,
          4.0000000000000000, -0.40000000000000002 },
  { -0.00080950511491911315, 5.0000000000000000, 20.000000000000000,
          4.0000000000000000, -0.30000000000000004 },
  { 0.0043473422174314449, 5.0000000000000000, 20.000000000000000,
          4.0000000000000000, -0.19999999999999996 },
  { 0.081078342558623825, 5.0000000000000000, 20.000000000000000,
          4.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 5.0000000000000000, 20.000000000000000,
          4.0000000000000000, 0.0000000000000000 },
  { 12.794854084397739, 5.0000000000000000, 20.000000000000000,
          4.0000000000000000, 0.10000000000000009 },
  { 195.15639104739046, 5.0000000000000000, 20.000000000000000,
          4.0000000000000000, 0.19999999999999996 },
  { 3938.7991953190131, 5.0000000000000000, 20.000000000000000,
          4.0000000000000000, 0.30000000000000004 },
  { 118521.48653762060, 5.0000000000000000, 20.000000000000000,
          4.0000000000000000, 0.39999999999999991 },
  { 6291455.9999999972, 5.0000000000000000, 20.000000000000000,
          4.0000000000000000, 0.50000000000000000 },
  { 773070496.50699198, 5.0000000000000000, 20.000000000000000,
          4.0000000000000000, 0.60000000000000009 },
  { 363276452167.04102, 5.0000000000000000, 20.000000000000000,
          4.0000000000000000, 0.69999999999999996 },
  { 2002716064453133.0, 5.0000000000000000, 20.000000000000000,
          4.0000000000000000, 0.80000000000000004 },
  { 4.5999999999999109e+21, 5.0000000000000000, 20.000000000000000,
          4.0000000000000000, 0.89999999999999991 },
};

// Test function for a=5.0000000000000000, b=20.000000000000000, c=4.0000000000000000.
template <typename Tp>
void test172()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data172)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data172[i].a), Tp(data172[i].b),
                   Tp(data172[i].c), Tp(data172[i].x));
      const Tp f0 = data172[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000013e-09));
}

// Test data for a=5.0000000000000000, b=20.000000000000000, c=6.0000000000000000.
testcase_hyperg<double> data173[] = {
  { 0.00014313323624069244, 5.0000000000000000, 20.000000000000000,
          6.0000000000000000, -0.90000000000000002 },
  { 0.00025426183473140697, 5.0000000000000000, 20.000000000000000,
          6.0000000000000000, -0.80000000000000004 },
  { 0.00048255612836426809, 5.0000000000000000, 20.000000000000000,
          6.0000000000000000, -0.69999999999999996 },
  { 0.00099096904674788092, 5.0000000000000000, 20.000000000000000,
          6.0000000000000000, -0.59999999999999998 },
  { 0.0022347805521915607, 5.0000000000000000, 20.000000000000000,
          6.0000000000000000, -0.50000000000000000 },
  { 0.0056271390060292376, 5.0000000000000000, 20.000000000000000,
          6.0000000000000000, -0.40000000000000002 },
  { 0.016109059519227226, 5.0000000000000000, 20.000000000000000,
          6.0000000000000000, -0.30000000000000004 },
  { 0.053453465775608999, 5.0000000000000000, 20.000000000000000,
          6.0000000000000000, -0.19999999999999996 },
  { 0.20995202901839258, 5.0000000000000000, 20.000000000000000,
          6.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 5.0000000000000000, 20.000000000000000,
          6.0000000000000000, 0.0000000000000000 },
  { 5.9534372167648799, 5.0000000000000000, 20.000000000000000,
          6.0000000000000000, 0.10000000000000009 },
  { 46.157632071205875, 5.0000000000000000, 20.000000000000000,
          6.0000000000000000, 0.19999999999999996 },
  { 494.32074431164915, 5.0000000000000000, 20.000000000000000,
          6.0000000000000000, 0.30000000000000004 },
  { 7989.5277611775946, 5.0000000000000000, 20.000000000000000,
          6.0000000000000000, 0.39999999999999991 },
  { 224179.55830753347, 5.0000000000000000, 20.000000000000000,
          6.0000000000000000, 0.50000000000000000 },
  { 13848144.485282511, 5.0000000000000000, 20.000000000000000,
          6.0000000000000000, 0.60000000000000009 },
  { 2948587692.8891716, 5.0000000000000000, 20.000000000000000,
          6.0000000000000000, 0.69999999999999996 },
  { 5940513286161.6602, 5.0000000000000000, 20.000000000000000,
          6.0000000000000000, 0.80000000000000004 },
  { 2.8531757655945201e+18, 5.0000000000000000, 20.000000000000000,
          6.0000000000000000, 0.89999999999999991 },
};

// Test function for a=5.0000000000000000, b=20.000000000000000, c=6.0000000000000000.
template <typename Tp>
void test173()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data173)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data173[i].a), Tp(data173[i].b),
                   Tp(data173[i].c), Tp(data173[i].x));
      const Tp f0 = data173[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(1.0000000000000006e-10));
}

// Test data for a=5.0000000000000000, b=20.000000000000000, c=8.0000000000000000.
testcase_hyperg<double> data174[] = {
  { 0.0012492049968742865, 5.0000000000000000, 20.000000000000000,
          8.0000000000000000, -0.90000000000000002 },
  { 0.0019931241968014451, 5.0000000000000000, 20.000000000000000,
          8.0000000000000000, -0.80000000000000004 },
  { 0.0033203386861411057, 5.0000000000000000, 20.000000000000000,
          8.0000000000000000, -0.69999999999999996 },
  { 0.0058191894509855282, 5.0000000000000000, 20.000000000000000,
          8.0000000000000000, -0.59999999999999998 },
  { 0.010830090368313866, 5.0000000000000000, 20.000000000000000,
          8.0000000000000000, -0.50000000000000000 },
  { 0.021653062305193163, 5.0000000000000000, 20.000000000000000,
          8.0000000000000000, -0.40000000000000002 },
  { 0.047180821280919084, 5.0000000000000000, 20.000000000000000,
          8.0000000000000000, -0.30000000000000004 },
  { 0.11405637279736180, 5.0000000000000000, 20.000000000000000,
          8.0000000000000000, -0.19999999999999996 },
  { 0.31275468794720990, 5.0000000000000000, 20.000000000000000,
          8.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 5.0000000000000000, 20.000000000000000,
          8.0000000000000000, 0.0000000000000000 },
  { 3.8598904658643969, 5.0000000000000000, 20.000000000000000,
          8.0000000000000000, 0.10000000000000009 },
  { 18.806301417906667, 5.0000000000000000, 20.000000000000000,
          8.0000000000000000, 0.19999999999999996 },
  { 122.77054465017432, 5.0000000000000000, 20.000000000000000,
          8.0000000000000000, 0.30000000000000004 },
  { 1168.4762146808946, 5.0000000000000000, 20.000000000000000,
          8.0000000000000000, 0.39999999999999991 },
  { 18437.511788521082, 5.0000000000000000, 20.000000000000000,
          8.0000000000000000, 0.50000000000000000 },
  { 597441.79669264762, 5.0000000000000000, 20.000000000000000,
          8.0000000000000000, 0.60000000000000009 },
  { 59390411.369227782, 5.0000000000000000, 20.000000000000000,
          8.0000000000000000, 0.69999999999999996 },
  { 44681668993.361603, 5.0000000000000000, 20.000000000000000,
          8.0000000000000000, 0.80000000000000004 },
  { 4559673269683164.0, 5.0000000000000000, 20.000000000000000,
          8.0000000000000000, 0.89999999999999991 },
};

// Test function for a=5.0000000000000000, b=20.000000000000000, c=8.0000000000000000.
template <typename Tp>
void test174()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data174)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data174[i].a), Tp(data174[i].b),
                   Tp(data174[i].c), Tp(data174[i].x));
      const Tp f0 = data174[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(1.0000000000000006e-11));
}

// Test data for a=5.0000000000000000, b=20.000000000000000, c=10.000000000000000.
testcase_hyperg<double> data175[] = {
  { 0.0038867957051371450, 5.0000000000000000, 20.000000000000000,
          10.000000000000000, -0.90000000000000002 },
  { 0.0058484892597364443, 5.0000000000000000, 20.000000000000000,
          10.000000000000000, -0.80000000000000004 },
  { 0.0090987656053757009, 5.0000000000000000, 20.000000000000000,
          10.000000000000000, -0.69999999999999996 },
  { 0.014714392537270733, 5.0000000000000000, 20.000000000000000,
          10.000000000000000, -0.59999999999999998 },
  { 0.024900404542056769, 5.0000000000000000, 20.000000000000000,
          10.000000000000000, -0.50000000000000000 },
  { 0.044460184663785055, 5.0000000000000000, 20.000000000000000,
          10.000000000000000, -0.40000000000000002 },
  { 0.084638849196356836, 5.0000000000000000, 20.000000000000000,
          10.000000000000000, -0.30000000000000004 },
  { 0.17409058241290998, 5.0000000000000000, 20.000000000000000,
          10.000000000000000, -0.19999999999999996 },
  { 0.39357055823580755, 5.0000000000000000, 20.000000000000000,
          10.000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 5.0000000000000000, 20.000000000000000,
          10.000000000000000, 0.0000000000000000 },
  { 2.9410794636226596, 5.0000000000000000, 20.000000000000000,
          10.000000000000000, 0.10000000000000009 },
  { 10.417226071414344, 5.0000000000000000, 20.000000000000000,
          10.000000000000000, 0.19999999999999996 },
  { 46.930585873140835, 5.0000000000000000, 20.000000000000000,
          10.000000000000000, 0.30000000000000004 },
  { 290.76717121814852, 5.0000000000000000, 20.000000000000000,
          10.000000000000000, 0.39999999999999991 },
  { 2788.1641083374830, 5.0000000000000000, 20.000000000000000,
          10.000000000000000, 0.50000000000000000 },
  { 50228.117718560752, 5.0000000000000000, 20.000000000000000,
          10.000000000000000, 0.60000000000000009 },
  { 2433042.3476752634, 5.0000000000000000, 20.000000000000000,
          10.000000000000000, 0.69999999999999996 },
  { 705345246.77141762, 5.0000000000000000, 20.000000000000000,
          10.000000000000000, 0.80000000000000004 },
  { 15652478868616.762, 5.0000000000000000, 20.000000000000000,
          10.000000000000000, 0.89999999999999991 },
};

// Test function for a=5.0000000000000000, b=20.000000000000000, c=10.000000000000000.
template <typename Tp>
void test175()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data175)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data175[i].a), Tp(data175[i].b),
                   Tp(data175[i].c), Tp(data175[i].x));
      const Tp f0 = data175[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(1.0000000000000008e-12));
}

// Test data for a=10.000000000000000, b=0.0000000000000000, c=2.0000000000000000.
testcase_hyperg<double> data176[] = {
  { 1.0000000000000000, 10.000000000000000, 0.0000000000000000,
          2.0000000000000000, -0.90000000000000002 },
  { 1.0000000000000000, 10.000000000000000, 0.0000000000000000,
          2.0000000000000000, -0.80000000000000004 },
  { 1.0000000000000000, 10.000000000000000, 0.0000000000000000,
          2.0000000000000000, -0.69999999999999996 },
  { 1.0000000000000000, 10.000000000000000, 0.0000000000000000,
          2.0000000000000000, -0.59999999999999998 },
  { 1.0000000000000000, 10.000000000000000, 0.0000000000000000,
          2.0000000000000000, -0.50000000000000000 },
  { 1.0000000000000000, 10.000000000000000, 0.0000000000000000,
          2.0000000000000000, -0.40000000000000002 },
  { 1.0000000000000000, 10.000000000000000, 0.0000000000000000,
          2.0000000000000000, -0.30000000000000004 },
  { 1.0000000000000000, 10.000000000000000, 0.0000000000000000,
          2.0000000000000000, -0.19999999999999996 },
  { 1.0000000000000000, 10.000000000000000, 0.0000000000000000,
          2.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 10.000000000000000, 0.0000000000000000,
          2.0000000000000000, 0.0000000000000000 },
  { 1.0000000000000000, 10.000000000000000, 0.0000000000000000,
          2.0000000000000000, 0.10000000000000009 },
  { 1.0000000000000000, 10.000000000000000, 0.0000000000000000,
          2.0000000000000000, 0.19999999999999996 },
  { 1.0000000000000000, 10.000000000000000, 0.0000000000000000,
          2.0000000000000000, 0.30000000000000004 },
  { 1.0000000000000000, 10.000000000000000, 0.0000000000000000,
          2.0000000000000000, 0.39999999999999991 },
  { 1.0000000000000000, 10.000000000000000, 0.0000000000000000,
          2.0000000000000000, 0.50000000000000000 },
  { 1.0000000000000000, 10.000000000000000, 0.0000000000000000,
          2.0000000000000000, 0.60000000000000009 },
  { 1.0000000000000000, 10.000000000000000, 0.0000000000000000,
          2.0000000000000000, 0.69999999999999996 },
  { 1.0000000000000000, 10.000000000000000, 0.0000000000000000,
          2.0000000000000000, 0.80000000000000004 },
  { 1.0000000000000000, 10.000000000000000, 0.0000000000000000,
          2.0000000000000000, 0.89999999999999991 },
};

// Test function for a=10.000000000000000, b=0.0000000000000000, c=2.0000000000000000.
template <typename Tp>
void test176()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data176)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data176[i].a), Tp(data176[i].b),
                   Tp(data176[i].c), Tp(data176[i].x));
      const Tp f0 = data176[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=10.000000000000000, b=0.0000000000000000, c=4.0000000000000000.
testcase_hyperg<double> data177[] = {
  { 1.0000000000000000, 10.000000000000000, 0.0000000000000000,
          4.0000000000000000, -0.90000000000000002 },
  { 1.0000000000000000, 10.000000000000000, 0.0000000000000000,
          4.0000000000000000, -0.80000000000000004 },
  { 1.0000000000000000, 10.000000000000000, 0.0000000000000000,
          4.0000000000000000, -0.69999999999999996 },
  { 1.0000000000000000, 10.000000000000000, 0.0000000000000000,
          4.0000000000000000, -0.59999999999999998 },
  { 1.0000000000000000, 10.000000000000000, 0.0000000000000000,
          4.0000000000000000, -0.50000000000000000 },
  { 1.0000000000000000, 10.000000000000000, 0.0000000000000000,
          4.0000000000000000, -0.40000000000000002 },
  { 1.0000000000000000, 10.000000000000000, 0.0000000000000000,
          4.0000000000000000, -0.30000000000000004 },
  { 1.0000000000000000, 10.000000000000000, 0.0000000000000000,
          4.0000000000000000, -0.19999999999999996 },
  { 1.0000000000000000, 10.000000000000000, 0.0000000000000000,
          4.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 10.000000000000000, 0.0000000000000000,
          4.0000000000000000, 0.0000000000000000 },
  { 1.0000000000000000, 10.000000000000000, 0.0000000000000000,
          4.0000000000000000, 0.10000000000000009 },
  { 1.0000000000000000, 10.000000000000000, 0.0000000000000000,
          4.0000000000000000, 0.19999999999999996 },
  { 1.0000000000000000, 10.000000000000000, 0.0000000000000000,
          4.0000000000000000, 0.30000000000000004 },
  { 1.0000000000000000, 10.000000000000000, 0.0000000000000000,
          4.0000000000000000, 0.39999999999999991 },
  { 1.0000000000000000, 10.000000000000000, 0.0000000000000000,
          4.0000000000000000, 0.50000000000000000 },
  { 1.0000000000000000, 10.000000000000000, 0.0000000000000000,
          4.0000000000000000, 0.60000000000000009 },
  { 1.0000000000000000, 10.000000000000000, 0.0000000000000000,
          4.0000000000000000, 0.69999999999999996 },
  { 1.0000000000000000, 10.000000000000000, 0.0000000000000000,
          4.0000000000000000, 0.80000000000000004 },
  { 1.0000000000000000, 10.000000000000000, 0.0000000000000000,
          4.0000000000000000, 0.89999999999999991 },
};

// Test function for a=10.000000000000000, b=0.0000000000000000, c=4.0000000000000000.
template <typename Tp>
void test177()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data177)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data177[i].a), Tp(data177[i].b),
                   Tp(data177[i].c), Tp(data177[i].x));
      const Tp f0 = data177[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=10.000000000000000, b=0.0000000000000000, c=6.0000000000000000.
testcase_hyperg<double> data178[] = {
  { 1.0000000000000000, 10.000000000000000, 0.0000000000000000,
          6.0000000000000000, -0.90000000000000002 },
  { 1.0000000000000000, 10.000000000000000, 0.0000000000000000,
          6.0000000000000000, -0.80000000000000004 },
  { 1.0000000000000000, 10.000000000000000, 0.0000000000000000,
          6.0000000000000000, -0.69999999999999996 },
  { 1.0000000000000000, 10.000000000000000, 0.0000000000000000,
          6.0000000000000000, -0.59999999999999998 },
  { 1.0000000000000000, 10.000000000000000, 0.0000000000000000,
          6.0000000000000000, -0.50000000000000000 },
  { 1.0000000000000000, 10.000000000000000, 0.0000000000000000,
          6.0000000000000000, -0.40000000000000002 },
  { 1.0000000000000000, 10.000000000000000, 0.0000000000000000,
          6.0000000000000000, -0.30000000000000004 },
  { 1.0000000000000000, 10.000000000000000, 0.0000000000000000,
          6.0000000000000000, -0.19999999999999996 },
  { 1.0000000000000000, 10.000000000000000, 0.0000000000000000,
          6.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 10.000000000000000, 0.0000000000000000,
          6.0000000000000000, 0.0000000000000000 },
  { 1.0000000000000000, 10.000000000000000, 0.0000000000000000,
          6.0000000000000000, 0.10000000000000009 },
  { 1.0000000000000000, 10.000000000000000, 0.0000000000000000,
          6.0000000000000000, 0.19999999999999996 },
  { 1.0000000000000000, 10.000000000000000, 0.0000000000000000,
          6.0000000000000000, 0.30000000000000004 },
  { 1.0000000000000000, 10.000000000000000, 0.0000000000000000,
          6.0000000000000000, 0.39999999999999991 },
  { 1.0000000000000000, 10.000000000000000, 0.0000000000000000,
          6.0000000000000000, 0.50000000000000000 },
  { 1.0000000000000000, 10.000000000000000, 0.0000000000000000,
          6.0000000000000000, 0.60000000000000009 },
  { 1.0000000000000000, 10.000000000000000, 0.0000000000000000,
          6.0000000000000000, 0.69999999999999996 },
  { 1.0000000000000000, 10.000000000000000, 0.0000000000000000,
          6.0000000000000000, 0.80000000000000004 },
  { 1.0000000000000000, 10.000000000000000, 0.0000000000000000,
          6.0000000000000000, 0.89999999999999991 },
};

// Test function for a=10.000000000000000, b=0.0000000000000000, c=6.0000000000000000.
template <typename Tp>
void test178()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data178)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data178[i].a), Tp(data178[i].b),
                   Tp(data178[i].c), Tp(data178[i].x));
      const Tp f0 = data178[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=10.000000000000000, b=0.0000000000000000, c=8.0000000000000000.
testcase_hyperg<double> data179[] = {
  { 1.0000000000000000, 10.000000000000000, 0.0000000000000000,
          8.0000000000000000, -0.90000000000000002 },
  { 1.0000000000000000, 10.000000000000000, 0.0000000000000000,
          8.0000000000000000, -0.80000000000000004 },
  { 1.0000000000000000, 10.000000000000000, 0.0000000000000000,
          8.0000000000000000, -0.69999999999999996 },
  { 1.0000000000000000, 10.000000000000000, 0.0000000000000000,
          8.0000000000000000, -0.59999999999999998 },
  { 1.0000000000000000, 10.000000000000000, 0.0000000000000000,
          8.0000000000000000, -0.50000000000000000 },
  { 1.0000000000000000, 10.000000000000000, 0.0000000000000000,
          8.0000000000000000, -0.40000000000000002 },
  { 1.0000000000000000, 10.000000000000000, 0.0000000000000000,
          8.0000000000000000, -0.30000000000000004 },
  { 1.0000000000000000, 10.000000000000000, 0.0000000000000000,
          8.0000000000000000, -0.19999999999999996 },
  { 1.0000000000000000, 10.000000000000000, 0.0000000000000000,
          8.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 10.000000000000000, 0.0000000000000000,
          8.0000000000000000, 0.0000000000000000 },
  { 1.0000000000000000, 10.000000000000000, 0.0000000000000000,
          8.0000000000000000, 0.10000000000000009 },
  { 1.0000000000000000, 10.000000000000000, 0.0000000000000000,
          8.0000000000000000, 0.19999999999999996 },
  { 1.0000000000000000, 10.000000000000000, 0.0000000000000000,
          8.0000000000000000, 0.30000000000000004 },
  { 1.0000000000000000, 10.000000000000000, 0.0000000000000000,
          8.0000000000000000, 0.39999999999999991 },
  { 1.0000000000000000, 10.000000000000000, 0.0000000000000000,
          8.0000000000000000, 0.50000000000000000 },
  { 1.0000000000000000, 10.000000000000000, 0.0000000000000000,
          8.0000000000000000, 0.60000000000000009 },
  { 1.0000000000000000, 10.000000000000000, 0.0000000000000000,
          8.0000000000000000, 0.69999999999999996 },
  { 1.0000000000000000, 10.000000000000000, 0.0000000000000000,
          8.0000000000000000, 0.80000000000000004 },
  { 1.0000000000000000, 10.000000000000000, 0.0000000000000000,
          8.0000000000000000, 0.89999999999999991 },
};

// Test function for a=10.000000000000000, b=0.0000000000000000, c=8.0000000000000000.
template <typename Tp>
void test179()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data179)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data179[i].a), Tp(data179[i].b),
                   Tp(data179[i].c), Tp(data179[i].x));
      const Tp f0 = data179[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=10.000000000000000, b=0.0000000000000000, c=10.000000000000000.
testcase_hyperg<double> data180[] = {
  { 1.0000000000000000, 10.000000000000000, 0.0000000000000000,
          10.000000000000000, -0.90000000000000002 },
  { 1.0000000000000000, 10.000000000000000, 0.0000000000000000,
          10.000000000000000, -0.80000000000000004 },
  { 1.0000000000000000, 10.000000000000000, 0.0000000000000000,
          10.000000000000000, -0.69999999999999996 },
  { 1.0000000000000000, 10.000000000000000, 0.0000000000000000,
          10.000000000000000, -0.59999999999999998 },
  { 1.0000000000000000, 10.000000000000000, 0.0000000000000000,
          10.000000000000000, -0.50000000000000000 },
  { 1.0000000000000000, 10.000000000000000, 0.0000000000000000,
          10.000000000000000, -0.40000000000000002 },
  { 1.0000000000000000, 10.000000000000000, 0.0000000000000000,
          10.000000000000000, -0.30000000000000004 },
  { 1.0000000000000000, 10.000000000000000, 0.0000000000000000,
          10.000000000000000, -0.19999999999999996 },
  { 1.0000000000000000, 10.000000000000000, 0.0000000000000000,
          10.000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 10.000000000000000, 0.0000000000000000,
          10.000000000000000, 0.0000000000000000 },
  { 1.0000000000000000, 10.000000000000000, 0.0000000000000000,
          10.000000000000000, 0.10000000000000009 },
  { 1.0000000000000000, 10.000000000000000, 0.0000000000000000,
          10.000000000000000, 0.19999999999999996 },
  { 1.0000000000000000, 10.000000000000000, 0.0000000000000000,
          10.000000000000000, 0.30000000000000004 },
  { 1.0000000000000000, 10.000000000000000, 0.0000000000000000,
          10.000000000000000, 0.39999999999999991 },
  { 1.0000000000000000, 10.000000000000000, 0.0000000000000000,
          10.000000000000000, 0.50000000000000000 },
  { 1.0000000000000000, 10.000000000000000, 0.0000000000000000,
          10.000000000000000, 0.60000000000000009 },
  { 1.0000000000000000, 10.000000000000000, 0.0000000000000000,
          10.000000000000000, 0.69999999999999996 },
  { 1.0000000000000000, 10.000000000000000, 0.0000000000000000,
          10.000000000000000, 0.80000000000000004 },
  { 1.0000000000000000, 10.000000000000000, 0.0000000000000000,
          10.000000000000000, 0.89999999999999991 },
};

// Test function for a=10.000000000000000, b=0.0000000000000000, c=10.000000000000000.
template <typename Tp>
void test180()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data180)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data180[i].a), Tp(data180[i].b),
                   Tp(data180[i].c), Tp(data180[i].x));
      const Tp f0 = data180[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=10.000000000000000, b=0.50000000000000000, c=2.0000000000000000.
testcase_hyperg<double> data181[] = {
  { 0.37727530159464628, 10.000000000000000, 0.50000000000000000,
          2.0000000000000000, -0.90000000000000002 },
  { 0.39816010922169010, 10.000000000000000, 0.50000000000000000,
          2.0000000000000000, -0.80000000000000004 },
  { 0.42283703041362453, 10.000000000000000, 0.50000000000000000,
          2.0000000000000000, -0.69999999999999996 },
  { 0.45255640448730505, 10.000000000000000, 0.50000000000000000,
          2.0000000000000000, -0.59999999999999998 },
  { 0.48919507154431141, 10.000000000000000, 0.50000000000000000,
          2.0000000000000000, -0.50000000000000000 },
  { 0.53569358917731880, 10.000000000000000, 0.50000000000000000,
          2.0000000000000000, -0.40000000000000002 },
  { 0.59689778897029566, 10.000000000000000, 0.50000000000000000,
          2.0000000000000000, -0.30000000000000004 },
  { 0.68128587569875731, 10.000000000000000, 0.50000000000000000,
          2.0000000000000000, -0.19999999999999996 },
  { 0.80478739308790359, 10.000000000000000, 0.50000000000000000,
          2.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 10.000000000000000, 0.50000000000000000,
          2.0000000000000000, 0.0000000000000000 },
  { 1.3408664196153621, 10.000000000000000, 0.50000000000000000,
          2.0000000000000000, 0.10000000000000009 },
  { 2.0175364359923860, 10.000000000000000, 0.50000000000000000,
          2.0000000000000000, 0.19999999999999996 },
  { 3.6011214553736646, 10.000000000000000, 0.50000000000000000,
          2.0000000000000000, 0.30000000000000004 },
  { 8.1799429939495312, 10.000000000000000, 0.50000000000000000,
          2.0000000000000000, 0.39999999999999991 },
  { 25.644834637536000, 10.000000000000000, 0.50000000000000000,
          2.0000000000000000, 0.50000000000000000 },
  { 123.13738891597615, 10.000000000000000, 0.50000000000000000,
          2.0000000000000000, 0.60000000000000009 },
  { 1088.7122410321333, 10.000000000000000, 0.50000000000000000,
          2.0000000000000000, 0.69999999999999996 },
  { 27358.291704709951, 10.000000000000000, 0.50000000000000000,
          2.0000000000000000, 0.80000000000000004 },
  { 8174369.0266731177, 10.000000000000000, 0.50000000000000000,
          2.0000000000000000, 0.89999999999999991 },
};

// Test function for a=10.000000000000000, b=0.50000000000000000, c=2.0000000000000000.
template <typename Tp>
void test181()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data181)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data181[i].a), Tp(data181[i].b),
                   Tp(data181[i].c), Tp(data181[i].x));
      const Tp f0 = data181[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=10.000000000000000, b=0.50000000000000000, c=4.0000000000000000.
testcase_hyperg<double> data182[] = {
  { 0.53905528308450834, 10.000000000000000, 0.50000000000000000,
          4.0000000000000000, -0.90000000000000002 },
  { 0.56235533974376162, 10.000000000000000, 0.50000000000000000,
          4.0000000000000000, -0.80000000000000004 },
  { 0.58887657983263575, 10.000000000000000, 0.50000000000000000,
          4.0000000000000000, -0.69999999999999996 },
  { 0.61941227047262915, 10.000000000000000, 0.50000000000000000,
          4.0000000000000000, -0.59999999999999998 },
  { 0.65504896640793853, 10.000000000000000, 0.50000000000000000,
          4.0000000000000000, -0.50000000000000000 },
  { 0.69731666644529999, 10.000000000000000, 0.50000000000000000,
          4.0000000000000000, -0.40000000000000002 },
  { 0.74844073299399128, 10.000000000000000, 0.50000000000000000,
          4.0000000000000000, -0.30000000000000004 },
  { 0.81178446800105752, 10.000000000000000, 0.50000000000000000,
          4.0000000000000000, -0.19999999999999996 },
  { 0.89266981277598023, 10.000000000000000, 0.50000000000000000,
          4.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 10.000000000000000, 0.50000000000000000,
          4.0000000000000000, 0.0000000000000000 },
  { 1.1497248473106778, 10.000000000000000, 0.50000000000000000,
          4.0000000000000000, 0.10000000000000009 },
  { 1.3729717112654571, 10.000000000000000, 0.50000000000000000,
          4.0000000000000000, 0.19999999999999996 },
  { 1.7374982340374392, 10.000000000000000, 0.50000000000000000,
          4.0000000000000000, 0.30000000000000004 },
  { 2.4134479340960580, 10.000000000000000, 0.50000000000000000,
          4.0000000000000000, 0.39999999999999991 },
  { 3.9191255240471192, 10.000000000000000, 0.50000000000000000,
          4.0000000000000000, 0.50000000000000000 },
  { 8.3316373077761270, 10.000000000000000, 0.50000000000000000,
          4.0000000000000000, 0.60000000000000009 },
  { 28.323020339843335, 10.000000000000000, 0.50000000000000000,
          4.0000000000000000, 0.69999999999999996 },
  { 225.84286572747891, 10.000000000000000, 0.50000000000000000,
          4.0000000000000000, 0.80000000000000004 },
  { 12757.127591286655, 10.000000000000000, 0.50000000000000000,
          4.0000000000000000, 0.89999999999999991 },
};

// Test function for a=10.000000000000000, b=0.50000000000000000, c=4.0000000000000000.
template <typename Tp>
void test182()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data182)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data182[i].a), Tp(data182[i].b),
                   Tp(data182[i].c), Tp(data182[i].x));
      const Tp f0 = data182[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=10.000000000000000, b=0.50000000000000000, c=6.0000000000000000.
testcase_hyperg<double> data183[] = {
  { 0.62672622092226071, 10.000000000000000, 0.50000000000000000,
          6.0000000000000000, -0.90000000000000002 },
  { 0.64931010269769829, 10.000000000000000, 0.50000000000000000,
          6.0000000000000000, -0.80000000000000004 },
  { 0.67448067519076316, 10.000000000000000, 0.50000000000000000,
          6.0000000000000000, -0.69999999999999996 },
  { 0.70276306239803676, 10.000000000000000, 0.50000000000000000,
          6.0000000000000000, -0.59999999999999998 },
  { 0.73484179773087555, 10.000000000000000, 0.50000000000000000,
          6.0000000000000000, -0.50000000000000000 },
  { 0.77162761412743897, 10.000000000000000, 0.50000000000000000,
          6.0000000000000000, -0.40000000000000002 },
  { 0.81436116844816531, 10.000000000000000, 0.50000000000000000,
          6.0000000000000000, -0.30000000000000004 },
  { 0.86477994787944557, 10.000000000000000, 0.50000000000000000,
          6.0000000000000000, -0.19999999999999996 },
  { 0.92539820516603888, 10.000000000000000, 0.50000000000000000,
          6.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 10.000000000000000, 0.50000000000000000,
          6.0000000000000000, 0.0000000000000000 },
  { 1.0945599448210315, 10.000000000000000, 0.50000000000000000,
          6.0000000000000000, 0.10000000000000009 },
  { 1.2190897395597264, 10.000000000000000, 0.50000000000000000,
          6.0000000000000000, 0.19999999999999996 },
  { 1.3916844336856475, 10.000000000000000, 0.50000000000000000,
          6.0000000000000000, 0.30000000000000004 },
  { 1.6484497630432013, 10.000000000000000, 0.50000000000000000,
          6.0000000000000000, 0.39999999999999991 },
  { 2.0717772717131155, 10.000000000000000, 0.50000000000000000,
          6.0000000000000000, 0.50000000000000000 },
  { 2.8893613630810924, 10.000000000000000, 0.50000000000000000,
          6.0000000000000000, 0.60000000000000009 },
  { 4.9459404075413529, 10.000000000000000, 0.50000000000000000,
          6.0000000000000000, 0.69999999999999996 },
  { 13.487394149998716, 10.000000000000000, 0.50000000000000000,
          6.0000000000000000, 0.80000000000000004 },
  { 136.57616044013972, 10.000000000000000, 0.50000000000000000,
          6.0000000000000000, 0.89999999999999991 },
};

// Test function for a=10.000000000000000, b=0.50000000000000000, c=6.0000000000000000.
template <typename Tp>
void test183()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data183)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data183[i].a), Tp(data183[i].b),
                   Tp(data183[i].c), Tp(data183[i].x));
      const Tp f0 = data183[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=10.000000000000000, b=0.50000000000000000, c=8.0000000000000000.
testcase_hyperg<double> data184[] = {
  { 0.68421604440344341, 10.000000000000000, 0.50000000000000000,
          8.0000000000000000, -0.90000000000000002 },
  { 0.70548098055548891, 10.000000000000000, 0.50000000000000000,
          8.0000000000000000, -0.80000000000000004 },
  { 0.72884342311710348, 10.000000000000000, 0.50000000000000000,
          8.0000000000000000, -0.69999999999999996 },
  { 0.75466953437856243, 10.000000000000000, 0.50000000000000000,
          8.0000000000000000, -0.59999999999999998 },
  { 0.78342090924662600, 10.000000000000000, 0.50000000000000000,
          8.0000000000000000, -0.50000000000000000 },
  { 0.81568884278645049, 10.000000000000000, 0.50000000000000000,
          8.0000000000000000, -0.40000000000000002 },
  { 0.85224480241465206, 10.000000000000000, 0.50000000000000000,
          8.0000000000000000, -0.30000000000000004 },
  { 0.89411692571131696, 10.000000000000000, 0.50000000000000000,
          8.0000000000000000, -0.19999999999999996 },
  { 0.94270986892954756, 10.000000000000000, 0.50000000000000000,
          8.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 10.000000000000000, 0.50000000000000000,
          8.0000000000000000, 0.0000000000000000 },
  { 1.0688682849120232, 10.000000000000000, 0.50000000000000000,
          8.0000000000000000, 0.10000000000000009 },
  { 1.1537004376097553, 10.000000000000000, 0.50000000000000000,
          8.0000000000000000, 0.19999999999999996 },
  { 1.2615455028370031, 10.000000000000000, 0.50000000000000000,
          8.0000000000000000, 0.30000000000000004 },
  { 1.4045541456153436, 10.000000000000000, 0.50000000000000000,
          8.0000000000000000, 0.39999999999999991 },
  { 1.6057216489444517, 10.000000000000000, 0.50000000000000000,
          8.0000000000000000, 0.50000000000000000 },
  { 1.9146603020550739, 10.000000000000000, 0.50000000000000000,
          8.0000000000000000, 0.60000000000000009 },
  { 2.4617931307620298, 10.000000000000000, 0.50000000000000000,
          8.0000000000000000, 0.69999999999999996 },
  { 3.7267799624996498, 10.000000000000000, 0.50000000000000000,
          8.0000000000000000, 0.80000000000000004 },
  { 9.3880118036248401, 10.000000000000000, 0.50000000000000000,
          8.0000000000000000, 0.89999999999999991 },
};

// Test function for a=10.000000000000000, b=0.50000000000000000, c=8.0000000000000000.
template <typename Tp>
void test184()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data184)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data184[i].a), Tp(data184[i].b),
                   Tp(data184[i].c), Tp(data184[i].x));
      const Tp f0 = data184[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=10.000000000000000, b=0.50000000000000000, c=10.000000000000000.
testcase_hyperg<double> data185[] = {
  { 0.72547625011001171, 10.000000000000000, 0.50000000000000000,
          10.000000000000000, -0.90000000000000002 },
  { 0.74535599249992990, 10.000000000000000, 0.50000000000000000,
          10.000000000000000, -0.80000000000000004 },
  { 0.76696498884737041, 10.000000000000000, 0.50000000000000000,
          10.000000000000000, -0.69999999999999996 },
  { 0.79056941504209477, 10.000000000000000, 0.50000000000000000,
          10.000000000000000, -0.59999999999999998 },
  { 0.81649658092772603, 10.000000000000000, 0.50000000000000000,
          10.000000000000000, -0.50000000000000000 },
  { 0.84515425472851657, 10.000000000000000, 0.50000000000000000,
          10.000000000000000, -0.40000000000000002 },
  { 0.87705801930702920, 10.000000000000000, 0.50000000000000000,
          10.000000000000000, -0.30000000000000004 },
  { 0.91287092917527690, 10.000000000000000, 0.50000000000000000,
          10.000000000000000, -0.19999999999999996 },
  { 0.95346258924559224, 10.000000000000000, 0.50000000000000000,
          10.000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 10.000000000000000, 0.50000000000000000,
          10.000000000000000, 0.0000000000000000 },
  { 1.0540925533894598, 10.000000000000000, 0.50000000000000000,
          10.000000000000000, 0.10000000000000009 },
  { 1.1180339887498949, 10.000000000000000, 0.50000000000000000,
          10.000000000000000, 0.19999999999999996 },
  { 1.1952286093343938, 10.000000000000000, 0.50000000000000000,
          10.000000000000000, 0.30000000000000004 },
  { 1.2909944487358056, 10.000000000000000, 0.50000000000000000,
          10.000000000000000, 0.39999999999999991 },
  { 1.4142135623730949, 10.000000000000000, 0.50000000000000000,
          10.000000000000000, 0.50000000000000000 },
  { 1.5811388300841900, 10.000000000000000, 0.50000000000000000,
          10.000000000000000, 0.60000000000000009 },
  { 1.8257418583505536, 10.000000000000000, 0.50000000000000000,
          10.000000000000000, 0.69999999999999996 },
  { 2.2360679774997898, 10.000000000000000, 0.50000000000000000,
          10.000000000000000, 0.80000000000000004 },
  { 3.1622776601683782, 10.000000000000000, 0.50000000000000000,
          10.000000000000000, 0.89999999999999991 },
};

// Test function for a=10.000000000000000, b=0.50000000000000000, c=10.000000000000000.
template <typename Tp>
void test185()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data185)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data185[i].a), Tp(data185[i].b),
                   Tp(data185[i].c), Tp(data185[i].x));
      const Tp f0 = data185[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=10.000000000000000, b=1.0000000000000000, c=2.0000000000000000.
testcase_hyperg<double> data186[] = {
  { 0.12307420104127871, 10.000000000000000, 1.0000000000000000,
          2.0000000000000000, -0.90000000000000002 },
  { 0.13818870041457423, 10.000000000000000, 1.0000000000000000,
          2.0000000000000000, -0.80000000000000004 },
  { 0.15739165631811691, 10.000000000000000, 1.0000000000000000,
          2.0000000000000000, -0.69999999999999996 },
  { 0.18249038606882068, 10.000000000000000, 1.0000000000000000,
          2.0000000000000000, -0.59999999999999998 },
  { 0.21644171225027786, 10.000000000000000, 1.0000000000000000,
          2.0000000000000000, -0.50000000000000000 },
  { 0.26433326159804160, 10.000000000000000, 1.0000000000000000,
          2.0000000000000000, -0.40000000000000002 },
  { 0.33544459430654533, 10.000000000000000, 1.0000000000000000,
          2.0000000000000000, -0.30000000000000004 },
  { 0.44788516696232517, 10.000000000000000, 1.0000000000000000,
          2.0000000000000000, -0.19999999999999996 },
  { 0.63989153514168362, 10.000000000000000, 1.0000000000000000,
          2.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 10.000000000000000, 1.0000000000000000,
          2.0000000000000000, 0.0000000000000000 },
  { 1.7568608796813312, 10.000000000000000, 1.0000000000000000,
          2.0000000000000000, 0.10000000000000009 },
  { 3.5836558871799027, 10.000000000000000, 1.0000000000000000,
          2.0000000000000000, 0.19999999999999996 },
  { 8.8077526749963226, 10.000000000000000, 1.0000000000000000,
          2.0000000000000000, 0.30000000000000004 },
  { 27.285841702089190, 10.000000000000000, 1.0000000000000000,
          2.0000000000000000, 0.39999999999999991 },
  { 113.55555555555557, 10.000000000000000, 1.0000000000000000,
          2.0000000000000000, 0.50000000000000000 },
  { 706.24023437500091, 10.000000000000000, 1.0000000000000000,
          2.0000000000000000, 0.60000000000000009 },
  { 8064.1687976651992, 10.000000000000000, 1.0000000000000000,
          2.0000000000000000, 0.69999999999999996 },
  { 271267.22222222196, 10.000000000000000, 1.0000000000000000,
          2.0000000000000000, 0.80000000000000004 },
  { 123456789.99999890, 10.000000000000000, 1.0000000000000000,
          2.0000000000000000, 0.89999999999999991 },
};

// Test function for a=10.000000000000000, b=1.0000000000000000, c=2.0000000000000000.
template <typename Tp>
void test186()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data186)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data186[i].a), Tp(data186[i].b),
                   Tp(data186[i].c), Tp(data186[i].x));
      const Tp f0 = data186[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=10.000000000000000, b=1.0000000000000000, c=4.0000000000000000.
testcase_hyperg<double> data187[] = {
  { 0.28363728383055758, 10.000000000000000, 1.0000000000000000,
          4.0000000000000000, -0.90000000000000002 },
  { 0.30933003169808387, 10.000000000000000, 1.0000000000000000,
          4.0000000000000000, -0.80000000000000004 },
  { 0.33998437757128797, 10.000000000000000, 1.0000000000000000,
          4.0000000000000000, -0.69999999999999996 },
  { 0.37713553224291119, 10.000000000000000, 1.0000000000000000,
          4.0000000000000000, -0.59999999999999998 },
  { 0.42299736538419658, 10.000000000000000, 1.0000000000000000,
          4.0000000000000000, -0.50000000000000000 },
  { 0.48086597727600106, 10.000000000000000, 1.0000000000000000,
          4.0000000000000000, -0.40000000000000002 },
  { 0.55583495759293045, 10.000000000000000, 1.0000000000000000,
          4.0000000000000000, -0.30000000000000004 },
  { 0.65612850114039667, 10.000000000000000, 1.0000000000000000,
          4.0000000000000000, -0.19999999999999996 },
  { 0.79573668772968120, 10.000000000000000, 1.0000000000000000,
          4.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 10.000000000000000, 1.0000000000000000,
          4.0000000000000000, 0.0000000000000000 },
  { 1.3184712058058303, 10.000000000000000, 1.0000000000000000,
          4.0000000000000000, 0.10000000000000009 },
  { 1.8576958065941214, 10.000000000000000, 1.0000000000000000,
          4.0000000000000000, 0.19999999999999996 },
  { 2.8759509651764228, 10.000000000000000, 1.0000000000000000,
          4.0000000000000000, 0.30000000000000004 },
  { 5.1046225531822182, 10.000000000000000, 1.0000000000000000,
          4.0000000000000000, 0.39999999999999991 },
  { 11.095238095238095, 10.000000000000000, 1.0000000000000000,
          4.0000000000000000, 0.50000000000000000 },
  { 32.797154017857174, 10.000000000000000, 1.0000000000000000,
          4.0000000000000000, 0.60000000000000009 },
  { 158.01935680536477, 10.000000000000000, 1.0000000000000000,
          4.0000000000000000, 0.69999999999999996 },
  { 1815.9523809523814, 10.000000000000000, 1.0000000000000000,
          4.0000000000000000, 0.80000000000000004 },
  { 163302.14285714156, 10.000000000000000, 1.0000000000000000,
          4.0000000000000000, 0.89999999999999991 },
};

// Test function for a=10.000000000000000, b=1.0000000000000000, c=4.0000000000000000.
template <typename Tp>
void test187()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data187)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data187[i].a), Tp(data187[i].b),
                   Tp(data187[i].c), Tp(data187[i].x));
      const Tp f0 = data187[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=10.000000000000000, b=1.0000000000000000, c=6.0000000000000000.
testcase_hyperg<double> data188[] = {
  { 0.39006633302741794, 10.000000000000000, 1.0000000000000000,
          6.0000000000000000, -0.90000000000000002 },
  { 0.41898885698103294, 10.000000000000000, 1.0000000000000000,
          6.0000000000000000, -0.80000000000000004 },
  { 0.45245557983812590, 10.000000000000000, 1.0000000000000000,
          6.0000000000000000, -0.69999999999999996 },
  { 0.49160548618861633, 10.000000000000000, 1.0000000000000000,
          6.0000000000000000, -0.59999999999999998 },
  { 0.53798419230517980, 10.000000000000000, 1.0000000000000000,
          6.0000000000000000, -0.50000000000000000 },
  { 0.59373881442067344, 10.000000000000000, 1.0000000000000000,
          6.0000000000000000, -0.40000000000000002 },
  { 0.66193391357076115, 10.000000000000000, 1.0000000000000000,
          6.0000000000000000, -0.30000000000000004 },
  { 0.74708402736952129, 10.000000000000000, 1.0000000000000000,
          6.0000000000000000, -0.19999999999999996 },
  { 0.85609281019430605, 10.000000000000000, 1.0000000000000000,
          6.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 10.000000000000000, 1.0000000000000000,
          6.0000000000000000, 0.0000000000000000 },
  { 1.1974451135148187, 10.000000000000000, 1.0000000000000000,
          6.0000000000000000, 0.10000000000000009 },
  { 1.4820886036706347, 10.000000000000000, 1.0000000000000000,
          6.0000000000000000, 0.19999999999999996 },
  { 1.9201183180477521, 10.000000000000000, 1.0000000000000000,
          6.0000000000000000, 0.30000000000000004 },
  { 2.6569338297733336, 10.000000000000000, 1.0000000000000000,
          6.0000000000000000, 0.39999999999999991 },
  { 4.0634920634920650, 10.000000000000000, 1.0000000000000000,
          6.0000000000000000, 0.50000000000000000 },
  { 7.3102678571428568, 10.000000000000000, 1.0000000000000000,
          6.0000000000000000, 0.60000000000000009 },
  { 17.512574302697733, 10.000000000000000, 1.0000000000000000,
          6.0000000000000000, 0.69999999999999996 },
  { 74.206349206349131, 10.000000000000000, 1.0000000000000000,
          6.0000000000000000, 0.80000000000000004 },
  { 1342.8571428571363, 10.000000000000000, 1.0000000000000000,
          6.0000000000000000, 0.89999999999999991 },
};

// Test function for a=10.000000000000000, b=1.0000000000000000, c=6.0000000000000000.
template <typename Tp>
void test188()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data188)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data188[i].a), Tp(data188[i].b),
                   Tp(data188[i].c), Tp(data188[i].x));
      const Tp f0 = data188[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=10.000000000000000, b=1.0000000000000000, c=8.0000000000000000.
testcase_hyperg<double> data189[] = {
  { 0.46726928123633193, 10.000000000000000, 1.0000000000000000,
          8.0000000000000000, -0.90000000000000002 },
  { 0.49687547629934464, 10.000000000000000, 1.0000000000000000,
          8.0000000000000000, -0.80000000000000004 },
  { 0.53045208856322235, 10.000000000000000, 1.0000000000000000,
          8.0000000000000000, -0.69999999999999996 },
  { 0.56884765624999978, 10.000000000000000, 1.0000000000000000,
          8.0000000000000000, -0.59999999999999998 },
  { 0.61316872427983526, 10.000000000000000, 1.0000000000000000,
          8.0000000000000000, -0.50000000000000000 },
  { 0.66488500161969544, 10.000000000000000, 1.0000000000000000,
          8.0000000000000000, -0.40000000000000002 },
  { 0.72598998634501577, 10.000000000000000, 1.0000000000000000,
          8.0000000000000000, -0.30000000000000004 },
  { 0.79925411522633782, 10.000000000000000, 1.0000000000000000,
          8.0000000000000000, -0.19999999999999996 },
  { 0.88863845062192193, 10.000000000000000, 1.0000000000000000,
          8.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 10.000000000000000, 1.0000000000000000,
          8.0000000000000000, 0.0000000000000000 },
  { 1.1423563481176653, 10.000000000000000, 1.0000000000000000,
          8.0000000000000000, 0.10000000000000009 },
  { 1.3302951388888888, 10.000000000000000, 1.0000000000000000,
          8.0000000000000000, 0.19999999999999996 },
  { 1.5889212827988335, 10.000000000000000, 1.0000000000000000,
          8.0000000000000000, 0.30000000000000004 },
  { 1.9650205761316870, 10.000000000000000, 1.0000000000000000,
          8.0000000000000000, 0.39999999999999991 },
  { 2.5555555555555549, 10.000000000000000, 1.0000000000000000,
          8.0000000000000000, 0.50000000000000000 },
  { 3.5937500000000013, 10.000000000000000, 1.0000000000000000,
          8.0000000000000000, 0.60000000000000009 },
  { 5.7818930041152203, 10.000000000000000, 1.0000000000000000,
          8.0000000000000000, 0.69999999999999996 },
  { 12.222222222222220, 10.000000000000000, 1.0000000000000000,
          8.0000000000000000, 0.80000000000000004 },
  { 54.999999999999780, 10.000000000000000, 1.0000000000000000,
          8.0000000000000000, 0.89999999999999991 },
};

// Test function for a=10.000000000000000, b=1.0000000000000000, c=8.0000000000000000.
template <typename Tp>
void test189()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data189)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data189[i].a), Tp(data189[i].b),
                   Tp(data189[i].c), Tp(data189[i].x));
      const Tp f0 = data189[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=10.000000000000000, b=1.0000000000000000, c=10.000000000000000.
testcase_hyperg<double> data190[] = {
  { 0.52631578947368429, 10.000000000000000, 1.0000000000000000,
          10.000000000000000, -0.90000000000000002 },
  { 0.55555555555555558, 10.000000000000000, 1.0000000000000000,
          10.000000000000000, -0.80000000000000004 },
  { 0.58823529411764708, 10.000000000000000, 1.0000000000000000,
          10.000000000000000, -0.69999999999999996 },
  { 0.62500000000000000, 10.000000000000000, 1.0000000000000000,
          10.000000000000000, -0.59999999999999998 },
  { 0.66666666666666663, 10.000000000000000, 1.0000000000000000,
          10.000000000000000, -0.50000000000000000 },
  { 0.71428571428571430, 10.000000000000000, 1.0000000000000000,
          10.000000000000000, -0.40000000000000002 },
  { 0.76923076923076927, 10.000000000000000, 1.0000000000000000,
          10.000000000000000, -0.30000000000000004 },
  { 0.83333333333333337, 10.000000000000000, 1.0000000000000000,
          10.000000000000000, -0.19999999999999996 },
  { 0.90909090909090906, 10.000000000000000, 1.0000000000000000,
          10.000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 10.000000000000000, 1.0000000000000000,
          10.000000000000000, 0.0000000000000000 },
  { 1.1111111111111112, 10.000000000000000, 1.0000000000000000,
          10.000000000000000, 0.10000000000000009 },
  { 1.2500000000000000, 10.000000000000000, 1.0000000000000000,
          10.000000000000000, 0.19999999999999996 },
  { 1.4285714285714286, 10.000000000000000, 1.0000000000000000,
          10.000000000000000, 0.30000000000000004 },
  { 1.6666666666666663, 10.000000000000000, 1.0000000000000000,
          10.000000000000000, 0.39999999999999991 },
  { 2.0000000000000000, 10.000000000000000, 1.0000000000000000,
          10.000000000000000, 0.50000000000000000 },
  { 2.5000000000000004, 10.000000000000000, 1.0000000000000000,
          10.000000000000000, 0.60000000000000009 },
  { 3.3333333333333330, 10.000000000000000, 1.0000000000000000,
          10.000000000000000, 0.69999999999999996 },
  { 5.0000000000000009, 10.000000000000000, 1.0000000000000000,
          10.000000000000000, 0.80000000000000004 },
  { 9.9999999999999929, 10.000000000000000, 1.0000000000000000,
          10.000000000000000, 0.89999999999999991 },
};

// Test function for a=10.000000000000000, b=1.0000000000000000, c=10.000000000000000.
template <typename Tp>
void test190()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data190)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data190[i].a), Tp(data190[i].b),
                   Tp(data190[i].c), Tp(data190[i].x));
      const Tp f0 = data190[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=10.000000000000000, b=2.0000000000000000, c=2.0000000000000000.
testcase_hyperg<double> data191[] = {
  { 0.0016310376661280216, 10.000000000000000, 2.0000000000000000,
          2.0000000000000000, -0.90000000000000002 },
  { 0.0028007538972582421, 10.000000000000000, 2.0000000000000000,
          2.0000000000000000, -0.80000000000000004 },
  { 0.0049603324681551939, 10.000000000000000, 2.0000000000000000,
          2.0000000000000000, -0.69999999999999996 },
  { 0.0090949470177292789, 10.000000000000000, 2.0000000000000000,
          2.0000000000000000, -0.59999999999999998 },
  { 0.017341529915832606, 10.000000000000000, 2.0000000000000000,
          2.0000000000000000, -0.50000000000000000 },
  { 0.034571613033607777, 10.000000000000000, 2.0000000000000000,
          2.0000000000000000, -0.40000000000000002 },
  { 0.072538150286405714, 10.000000000000000, 2.0000000000000000,
          2.0000000000000000, -0.30000000000000004 },
  { 0.16150558288984579, 10.000000000000000, 2.0000000000000000,
          2.0000000000000000, -0.19999999999999996 },
  { 0.38554328942953148, 10.000000000000000, 2.0000000000000000,
          2.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 10.000000000000000, 2.0000000000000000,
          2.0000000000000000, 0.0000000000000000 },
  { 2.8679719907924444, 10.000000000000000, 2.0000000000000000,
          2.0000000000000000, 0.10000000000000009 },
  { 9.3132257461547816, 10.000000000000000, 2.0000000000000000,
          2.0000000000000000, 0.19999999999999996 },
  { 35.401331746414378, 10.000000000000000, 2.0000000000000000,
          2.0000000000000000, 0.30000000000000004 },
  { 165.38171687920172, 10.000000000000000, 2.0000000000000000,
          2.0000000000000000, 0.39999999999999991 },
  { 1024.0000000000000, 10.000000000000000, 2.0000000000000000,
          2.0000000000000000, 0.50000000000000000 },
  { 9536.7431640625200, 10.000000000000000, 2.0000000000000000,
          2.0000000000000000, 0.60000000000000009 },
  { 169350.87808430271, 10.000000000000000, 2.0000000000000000,
          2.0000000000000000, 0.69999999999999996 },
  { 9765624.9999999944, 10.000000000000000, 2.0000000000000000,
          2.0000000000000000, 0.80000000000000004 },
  { 9999999999.9999332, 10.000000000000000, 2.0000000000000000,
          2.0000000000000000, 0.89999999999999991 },
};

// Test function for a=10.000000000000000, b=2.0000000000000000, c=2.0000000000000000.
template <typename Tp>
void test191()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data191)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data191[i].a), Tp(data191[i].b),
                   Tp(data191[i].c), Tp(data191[i].x));
      const Tp f0 = data191[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=10.000000000000000, b=2.0000000000000000, c=4.0000000000000000.
testcase_hyperg<double> data192[] = {
  { 0.071191280690193509, 10.000000000000000, 2.0000000000000000,
          4.0000000000000000, -0.90000000000000002 },
  { 0.085646504654238079, 10.000000000000000, 2.0000000000000000,
          4.0000000000000000, -0.80000000000000004 },
  { 0.10478215656371073, 10.000000000000000, 2.0000000000000000,
          4.0000000000000000, -0.69999999999999996 },
  { 0.13074816337653575, 10.000000000000000, 2.0000000000000000,
          4.0000000000000000, -0.59999999999999998 },
  { 0.16701141666848116, 10.000000000000000, 2.0000000000000000,
          4.0000000000000000, -0.50000000000000000 },
  { 0.21939323375313971, 10.000000000000000, 2.0000000000000000,
          4.0000000000000000, -0.40000000000000002 },
  { 0.29813515331786627, 10.000000000000000, 2.0000000000000000,
          4.0000000000000000, -0.30000000000000004 },
  { 0.42225974638874386, 10.000000000000000, 2.0000000000000000,
          4.0000000000000000, -0.19999999999999996 },
  { 0.62942145962174867, 10.000000000000000, 2.0000000000000000,
          4.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 10.000000000000000, 2.0000000000000000,
          4.0000000000000000, 0.0000000000000000 },
  { 1.7218685262373197, 10.000000000000000, 2.0000000000000000,
          4.0000000000000000, 0.10000000000000009 },
  { 3.2855760483514689, 10.000000000000000, 2.0000000000000000,
          4.0000000000000000, 0.19999999999999996 },
  { 7.1616652508907093, 10.000000000000000, 2.0000000000000000,
          4.0000000000000000, 0.30000000000000004 },
  { 18.612326808485907, 10.000000000000000, 2.0000000000000000,
          4.0000000000000000, 0.39999999999999991 },
  { 61.476190476190474, 10.000000000000000, 2.0000000000000000,
          4.0000000000000000, 0.50000000000000000 },
  { 286.27580915178623, 10.000000000000000, 2.0000000000000000,
          4.0000000000000000, 0.60000000000000009 },
  { 2274.9441142102296, 10.000000000000000, 2.0000000000000000,
          4.0000000000000000, 0.69999999999999996 },
  { 47229.761904761865, 10.000000000000000, 2.0000000000000000,
          4.0000000000000000, 0.80000000000000004 },
  { 9961460.7142856438, 10.000000000000000, 2.0000000000000000,
          4.0000000000000000, 0.89999999999999991 },
};

// Test function for a=10.000000000000000, b=2.0000000000000000, c=4.0000000000000000.
template <typename Tp>
void test192()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data192)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data192[i].a), Tp(data192[i].b),
                   Tp(data192[i].c), Tp(data192[i].x));
      const Tp f0 = data192[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=10.000000000000000, b=2.0000000000000000, c=6.0000000000000000.
testcase_hyperg<double> data193[] = {
  { 0.14747230019381052, 10.000000000000000, 2.0000000000000000,
          6.0000000000000000, -0.90000000000000002 },
  { 0.17073600100690603, 10.000000000000000, 2.0000000000000000,
          6.0000000000000000, -0.80000000000000004 },
  { 0.19982795745135354, 10.000000000000000, 2.0000000000000000,
          6.0000000000000000, -0.69999999999999996 },
  { 0.23681776864188067, 10.000000000000000, 2.0000000000000000,
          6.0000000000000000, -0.59999999999999998 },
  { 0.28475624360398022, 10.000000000000000, 2.0000000000000000,
          6.0000000000000000, -0.50000000000000000 },
  { 0.34827500743063144, 10.000000000000000, 2.0000000000000000,
          6.0000000000000000, -0.40000000000000002 },
  { 0.43464829159684681, 10.000000000000000, 2.0000000000000000,
          6.0000000000000000, -0.30000000000000004 },
  { 0.55576053438064787, 10.000000000000000, 2.0000000000000000,
          6.0000000000000000, -0.19999999999999996 },
  { 0.73195020913445485, 10.000000000000000, 2.0000000000000000,
          6.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 10.000000000000000, 2.0000000000000000,
          6.0000000000000000, 0.0000000000000000 },
  { 1.4310223867822929, 10.000000000000000, 2.0000000000000000,
          6.0000000000000000, 0.10000000000000009 },
  { 2.1742563399057540, 10.000000000000000, 2.0000000000000000,
          6.0000000000000000, 0.19999999999999996 },
  { 3.5769231236256043, 10.000000000000000, 2.0000000000000000,
          6.0000000000000000, 0.30000000000000004 },
  { 6.5620441134844363, 10.000000000000000, 2.0000000000000000,
          6.0000000000000000, 0.39999999999999991 },
  { 14.063492063492063, 10.000000000000000, 2.0000000000000000,
          6.0000000000000000, 0.50000000000000000 },
  { 38.085937500000036, 10.000000000000000, 2.0000000000000000,
          6.0000000000000000, 0.60000000000000009 },
  { 150.92973632068282, 10.000000000000000, 2.0000000000000000,
          6.0000000000000000, 0.69999999999999996 },
  { 1212.3015873015852, 10.000000000000000, 2.0000000000000000,
          6.0000000000000000, 0.80000000000000004 },
  { 55107.142857142389, 10.000000000000000, 2.0000000000000000,
          6.0000000000000000, 0.89999999999999991 },
};

// Test function for a=10.000000000000000, b=2.0000000000000000, c=6.0000000000000000.
template <typename Tp>
void test193()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data193)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data193[i].a), Tp(data193[i].b),
                   Tp(data193[i].c), Tp(data193[i].x));
      const Tp f0 = data193[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=10.000000000000000, b=2.0000000000000000, c=8.0000000000000000.
testcase_hyperg<double> data194[] = {
  { 0.21658059714090588, 10.000000000000000, 2.0000000000000000,
          8.0000000000000000, -0.90000000000000002 },
  { 0.24513539602702844, 10.000000000000000, 2.0000000000000000,
          8.0000000000000000, -0.80000000000000004 },
  { 0.27967018274845046, 10.000000000000000, 2.0000000000000000,
          8.0000000000000000, -0.69999999999999996 },
  { 0.32196044921874994, 10.000000000000000, 2.0000000000000000,
          8.0000000000000000, -0.59999999999999998 },
  { 0.37448559670781900, 10.000000000000000, 2.0000000000000000,
          8.0000000000000000, -0.50000000000000000 },
  { 0.44078856032208796, 10.000000000000000, 2.0000000000000000,
          8.0000000000000000, -0.40000000000000002 },
  { 0.52606701446027815, 10.000000000000000, 2.0000000000000000,
          8.0000000000000000, -0.30000000000000004 },
  { 0.63818158436214001, 10.000000000000000, 2.0000000000000000,
          8.0000000000000000, -0.19999999999999996 },
  { 0.78944971882612769, 10.000000000000000, 2.0000000000000000,
          8.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 10.000000000000000, 2.0000000000000000,
          8.0000000000000000, 0.0000000000000000 },
  { 1.3044251384443430, 10.000000000000000, 2.0000000000000000,
          8.0000000000000000, 0.10000000000000009 },
  { 1.7659505208333335, 10.000000000000000, 2.0000000000000000,
          8.0000000000000000, 0.19999999999999996 },
  { 2.5093710953769270, 10.000000000000000, 2.0000000000000000,
          8.0000000000000000, 0.30000000000000004 },
  { 3.8065843621399158, 10.000000000000000, 2.0000000000000000,
          8.0000000000000000, 0.39999999999999991 },
  { 6.3333333333333313, 10.000000000000000, 2.0000000000000000,
          8.0000000000000000, 0.50000000000000000 },
  { 12.109375000000004, 10.000000000000000, 2.0000000000000000,
          8.0000000000000000, 0.60000000000000009 },
  { 29.115226337448540, 10.000000000000000, 2.0000000000000000,
          8.0000000000000000, 0.69999999999999996 },
  { 108.33333333333330, 10.000000000000000, 2.0000000000000000,
          8.0000000000000000, 0.80000000000000004 },
  { 1224.9999999999923, 10.000000000000000, 2.0000000000000000,
          8.0000000000000000, 0.89999999999999991 },
};

// Test function for a=10.000000000000000, b=2.0000000000000000, c=8.0000000000000000.
template <typename Tp>
void test194()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data194)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data194[i].a), Tp(data194[i].b),
                   Tp(data194[i].c), Tp(data194[i].x));
      const Tp f0 = data194[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=10.000000000000000, b=2.0000000000000000, c=10.000000000000000.
testcase_hyperg<double> data195[] = {
  { 0.27700831024930750, 10.000000000000000, 2.0000000000000000,
          10.000000000000000, -0.90000000000000002 },
  { 0.30864197530864196, 10.000000000000000, 2.0000000000000000,
          10.000000000000000, -0.80000000000000004 },
  { 0.34602076124567477, 10.000000000000000, 2.0000000000000000,
          10.000000000000000, -0.69999999999999996 },
  { 0.39062499999999994, 10.000000000000000, 2.0000000000000000,
          10.000000000000000, -0.59999999999999998 },
  { 0.44444444444444442, 10.000000000000000, 2.0000000000000000,
          10.000000000000000, -0.50000000000000000 },
  { 0.51020408163265307, 10.000000000000000, 2.0000000000000000,
          10.000000000000000, -0.40000000000000002 },
  { 0.59171597633136097, 10.000000000000000, 2.0000000000000000,
          10.000000000000000, -0.30000000000000004 },
  { 0.69444444444444453, 10.000000000000000, 2.0000000000000000,
          10.000000000000000, -0.19999999999999996 },
  { 0.82644628099173545, 10.000000000000000, 2.0000000000000000,
          10.000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 10.000000000000000, 2.0000000000000000,
          10.000000000000000, 0.0000000000000000 },
  { 1.2345679012345681, 10.000000000000000, 2.0000000000000000,
          10.000000000000000, 0.10000000000000009 },
  { 1.5624999999999998, 10.000000000000000, 2.0000000000000000,
          10.000000000000000, 0.19999999999999996 },
  { 2.0408163265306127, 10.000000000000000, 2.0000000000000000,
          10.000000000000000, 0.30000000000000004 },
  { 2.7777777777777768, 10.000000000000000, 2.0000000000000000,
          10.000000000000000, 0.39999999999999991 },
  { 4.0000000000000000, 10.000000000000000, 2.0000000000000000,
          10.000000000000000, 0.50000000000000000 },
  { 6.2500000000000036, 10.000000000000000, 2.0000000000000000,
          10.000000000000000, 0.60000000000000009 },
  { 11.111111111111109, 10.000000000000000, 2.0000000000000000,
          10.000000000000000, 0.69999999999999996 },
  { 25.000000000000007, 10.000000000000000, 2.0000000000000000,
          10.000000000000000, 0.80000000000000004 },
  { 99.999999999999872, 10.000000000000000, 2.0000000000000000,
          10.000000000000000, 0.89999999999999991 },
};

// Test function for a=10.000000000000000, b=2.0000000000000000, c=10.000000000000000.
template <typename Tp>
void test195()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data195)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data195[i].a), Tp(data195[i].b),
                   Tp(data195[i].c), Tp(data195[i].x));
      const Tp f0 = data195[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=10.000000000000000, b=5.0000000000000000, c=2.0000000000000000.
testcase_hyperg<double> data196[] = {
  { 0.00063586451658060152, 10.000000000000000, 5.0000000000000000,
          2.0000000000000000, -0.90000000000000002 },
  { 0.0010334743461762443, 10.000000000000000, 5.0000000000000000,
          2.0000000000000000, -0.80000000000000004 },
  { 0.0015326246054669515, 10.000000000000000, 5.0000000000000000,
          2.0000000000000000, -0.69999999999999996 },
  { 0.0019007018181583387, 10.000000000000000, 5.0000000000000000,
          2.0000000000000000, -0.59999999999999998 },
  { 0.0012845577715431577, 10.000000000000000, 5.0000000000000000,
          2.0000000000000000, -0.50000000000000000 },
  { -0.0027213806178058826, 10.000000000000000, 5.0000000000000000,
          2.0000000000000000, -0.40000000000000002 },
  { -0.015121744574954068, 10.000000000000000, 5.0000000000000000,
          2.0000000000000000, -0.30000000000000004 },
  { -0.036637840562974443, 10.000000000000000, 5.0000000000000000,
          2.0000000000000000, -0.19999999999999996 },
  { 0.019117849062621491, 10.000000000000000, 5.0000000000000000,
          2.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 10.000000000000000, 5.0000000000000000,
          2.0000000000000000, 0.0000000000000000 },
  { 9.8116901852350615, 10.000000000000000, 5.0000000000000000,
          2.0000000000000000, 0.10000000000000009 },
  { 84.255589172244044, 10.000000000000000, 5.0000000000000000,
          2.0000000000000000, 0.19999999999999996 },
  { 773.87517619421294, 10.000000000000000, 5.0000000000000000,
          2.0000000000000000, 0.30000000000000004 },
  { 8556.9725363053585, 10.000000000000000, 5.0000000000000000,
          2.0000000000000000, 0.39999999999999991 },
  { 129023.99999999996, 10.000000000000000, 5.0000000000000000,
          2.0000000000000000, 0.50000000000000000 },
  { 3174543.3807373112, 10.000000000000000, 5.0000000000000000,
          2.0000000000000000, 0.60000000000000009 },
  { 175133896.95814410, 10.000000000000000, 5.0000000000000000,
          2.0000000000000000, 0.69999999999999996 },
  { 43564453125.000061, 10.000000000000000, 5.0000000000000000,
          2.0000000000000000, 0.80000000000000004 },
  { 446859999999993.50, 10.000000000000000, 5.0000000000000000,
          2.0000000000000000, 0.89999999999999991 },
};

// Test function for a=10.000000000000000, b=5.0000000000000000, c=2.0000000000000000.
template <typename Tp>
void test196()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data196)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data196[i].a), Tp(data196[i].b),
                   Tp(data196[i].c), Tp(data196[i].x));
      const Tp f0 = data196[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(1.0000000000000006e-11));
}

// Test data for a=10.000000000000000, b=5.0000000000000000, c=4.0000000000000000.
testcase_hyperg<double> data197[] = {
  { -0.00030045430691819899, 10.000000000000000, 5.0000000000000000,
          4.0000000000000000, -0.90000000000000002 },
  { -0.00031119487747328581, 10.000000000000000, 5.0000000000000000,
          4.0000000000000000, -0.80000000000000004 },
  { -0.00014589213141649274, 10.000000000000000, 5.0000000000000000,
          4.0000000000000000, -0.69999999999999996 },
  { 0.00056843418860809121, 10.000000000000000, 5.0000000000000000,
          4.0000000000000000, -0.59999999999999998 },
  { 0.0028902549859721725, 10.000000000000000, 5.0000000000000000,
          4.0000000000000000, -0.50000000000000000 },
  { 0.0098776037238877470, 10.000000000000000, 5.0000000000000000,
          4.0000000000000000, -0.40000000000000002 },
  { 0.030689217428863914, 10.000000000000000, 5.0000000000000000,
          4.0000000000000000, -0.30000000000000004 },
  { 0.094211590019076599, 10.000000000000000, 5.0000000000000000,
          4.0000000000000000, -0.19999999999999996 },
  { 0.29791981455918376, 10.000000000000000, 5.0000000000000000,
          4.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 10.000000000000000, 5.0000000000000000,
          4.0000000000000000, 0.0000000000000000 },
  { 3.6646308771236793, 10.000000000000000, 5.0000000000000000,
          4.0000000000000000, 0.10000000000000009 },
  { 15.133991837501521, 10.000000000000000, 5.0000000000000000,
          4.0000000000000000, 0.19999999999999996 },
  { 73.331330046144089, 10.000000000000000, 5.0000000000000000,
          4.0000000000000000, 0.30000000000000004 },
  { 441.01791167787133, 10.000000000000000, 5.0000000000000000,
          4.0000000000000000, 0.39999999999999991 },
  { 3583.9999999999991, 10.000000000000000, 5.0000000000000000,
          4.0000000000000000, 0.50000000000000000 },
  { 45299.530029296984, 10.000000000000000, 5.0000000000000000,
          4.0000000000000000, 0.60000000000000009 },
  { 1157231.0002427341, 10.000000000000000, 5.0000000000000000,
          4.0000000000000000, 0.69999999999999996 },
  { 107421875.00000016, 10.000000000000000, 5.0000000000000000,
          4.0000000000000000, 0.80000000000000004 },
  { 234999999999.99734, 10.000000000000000, 5.0000000000000000,
          4.0000000000000000, 0.89999999999999991 },
};

// Test function for a=10.000000000000000, b=5.0000000000000000, c=4.0000000000000000.
template <typename Tp>
void test197()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data197)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data197[i].a), Tp(data197[i].b),
                   Tp(data197[i].c), Tp(data197[i].x));
      const Tp f0 = data197[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000014e-11));
}

// Test data for a=10.000000000000000, b=5.0000000000000000, c=6.0000000000000000.
testcase_hyperg<double> data198[] = {
  { 0.0058530497315411210, 10.000000000000000, 5.0000000000000000,
          6.0000000000000000, -0.90000000000000002 },
  { 0.0088526869356855692, 10.000000000000000, 5.0000000000000000,
          6.0000000000000000, -0.80000000000000004 },
  { 0.013770987983443108, 10.000000000000000, 5.0000000000000000,
          6.0000000000000000, -0.69999999999999996 },
  { 0.022108932690960800, 10.000000000000000, 5.0000000000000000,
          6.0000000000000000, -0.59999999999999998 },
  { 0.036786236450921578, 10.000000000000000, 5.0000000000000000,
          6.0000000000000000, -0.50000000000000000 },
  { 0.063750669040426422, 10.000000000000000, 5.0000000000000000,
          6.0000000000000000, -0.40000000000000002 },
  { 0.11577228680714464, 10.000000000000000, 5.0000000000000000,
          6.0000000000000000, -0.30000000000000004 },
  { 0.22197573416125735, 10.000000000000000, 5.0000000000000000,
          6.0000000000000000, -0.19999999999999996 },
  { 0.45361312968415268, 10.000000000000000, 5.0000000000000000,
          6.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 10.000000000000000, 5.0000000000000000,
          6.0000000000000000, 0.0000000000000000 },
  { 2.4162889363082747, 10.000000000000000, 5.0000000000000000,
          6.0000000000000000, 0.10000000000000009 },
  { 6.5381564791240399, 10.000000000000000, 5.0000000000000000,
          6.0000000000000000, 0.19999999999999996 },
  { 20.415771011498428, 10.000000000000000, 5.0000000000000000,
          6.0000000000000000, 0.30000000000000004 },
  { 76.870682056629221, 10.000000000000000, 5.0000000000000000,
          6.0000000000000000, 0.39999999999999991 },
  { 373.58730158730162, 10.000000000000000, 5.0000000000000000,
          6.0000000000000000, 0.50000000000000000 },
  { 2626.2555803571477, 10.000000000000000, 5.0000000000000000,
          6.0000000000000000, 0.60000000000000009 },
  { 33060.960671081048, 10.000000000000000, 5.0000000000000000,
          6.0000000000000000, 0.69999999999999996 },
  { 1203521.8253968258, 10.000000000000000, 5.0000000000000000,
          6.0000000000000000, 0.80000000000000004 },
  { 584564285.71427989, 10.000000000000000, 5.0000000000000000,
          6.0000000000000000, 0.89999999999999991 },
};

// Test function for a=10.000000000000000, b=5.0000000000000000, c=6.0000000000000000.
template <typename Tp>
void test198()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data198)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data198[i].a), Tp(data198[i].b),
                   Tp(data198[i].c), Tp(data198[i].x));
      const Tp f0 = data198[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000015e-12));
}

// Test data for a=10.000000000000000, b=5.0000000000000000, c=8.0000000000000000.
testcase_hyperg<double> data199[] = {
  { 0.020248990107069400, 10.000000000000000, 5.0000000000000000,
          8.0000000000000000, -0.90000000000000002 },
  { 0.027876687750502421, 10.000000000000000, 5.0000000000000000,
          8.0000000000000000, -0.80000000000000004 },
  { 0.039154648888447781, 10.000000000000000, 5.0000000000000000,
          8.0000000000000000, -0.69999999999999996 },
  { 0.056251883506774923, 10.000000000000000, 5.0000000000000000,
          8.0000000000000000, -0.59999999999999998 },
  { 0.082914189910074432, 10.000000000000000, 5.0000000000000000,
          8.0000000000000000, -0.50000000000000000 },
  { 0.12585357817786477, 10.000000000000000, 5.0000000000000000,
          8.0000000000000000, -0.40000000000000002 },
  { 0.19761423206224929, 10.000000000000000, 5.0000000000000000,
          8.0000000000000000, -0.30000000000000004 },
  { 0.32280443863359243, 10.000000000000000, 5.0000000000000000,
          8.0000000000000000, -0.19999999999999996 },
  { 0.55250024062839465, 10.000000000000000, 5.0000000000000000,
          8.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 10.000000000000000, 5.0000000000000000,
          8.0000000000000000, 0.0000000000000000 },
  { 1.9374297986599267, 10.000000000000000, 5.0000000000000000,
          8.0000000000000000, 0.10000000000000009 },
  { 4.0849049886067696, 10.000000000000000, 5.0000000000000000,
          8.0000000000000000, 0.19999999999999996 },
  { 9.5926988633258983, 10.000000000000000, 5.0000000000000000,
          8.0000000000000000, 0.30000000000000004 },
  { 25.958314281359531, 10.000000000000000, 5.0000000000000000,
          8.0000000000000000, 0.39999999999999991 },
  { 85.333333333333300, 10.000000000000000, 5.0000000000000000,
          8.0000000000000000, 0.50000000000000000 },
  { 372.31445312500028, 10.000000000000000, 5.0000000000000000,
          8.0000000000000000, 0.60000000000000009 },
  { 2545.3436976070675, 10.000000000000000, 5.0000000000000000,
          8.0000000000000000, 0.69999999999999996 },
  { 39583.333333333343, 10.000000000000000, 5.0000000000000000,
          8.0000000000000000, 0.80000000000000004 },
  { 4599999.9999999627, 10.000000000000000, 5.0000000000000000,
          8.0000000000000000, 0.89999999999999991 },
};

// Test function for a=10.000000000000000, b=5.0000000000000000, c=8.0000000000000000.
template <typename Tp>
void test199()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data199)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data199[i].a), Tp(data199[i].b),
                   Tp(data199[i].c), Tp(data199[i].x));
      const Tp f0 = data199[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(5.0000000000000039e-13));
}

// Test data for a=10.000000000000000, b=5.0000000000000000, c=10.000000000000000.
testcase_hyperg<double> data200[] = {
  { 0.040386107340619266, 10.000000000000000, 5.0000000000000000,
          10.000000000000000, -0.90000000000000002 },
  { 0.052922149401344633, 10.000000000000000, 5.0000000000000000,
          10.000000000000000, -0.80000000000000004 },
  { 0.070429627772374270, 10.000000000000000, 5.0000000000000000,
          10.000000000000000, -0.69999999999999996 },
  { 0.095367431640624972, 10.000000000000000, 5.0000000000000000,
          10.000000000000000, -0.59999999999999998 },
  { 0.13168724279835387, 10.000000000000000, 5.0000000000000000,
          10.000000000000000, -0.50000000000000000 },
  { 0.18593443208187066, 10.000000000000000, 5.0000000000000000,
          10.000000000000000, -0.40000000000000002 },
  { 0.26932907434290437, 10.000000000000000, 5.0000000000000000,
          10.000000000000000, -0.30000000000000004 },
  { 0.40187757201646096, 10.000000000000000, 5.0000000000000000,
          10.000000000000000, -0.19999999999999996 },
  { 0.62092132305915493, 10.000000000000000, 5.0000000000000000,
          10.000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 10.000000000000000, 5.0000000000000000,
          10.000000000000000, 0.0000000000000000 },
  { 1.6935087808430296, 10.000000000000000, 5.0000000000000000,
          10.000000000000000, 0.10000000000000009 },
  { 3.0517578124999991, 10.000000000000000, 5.0000000000000000,
          10.000000000000000, 0.19999999999999996 },
  { 5.9499018266198629, 10.000000000000000, 5.0000000000000000,
          10.000000000000000, 0.30000000000000004 },
  { 12.860082304526737, 10.000000000000000, 5.0000000000000000,
          10.000000000000000, 0.39999999999999991 },
  { 32.000000000000000, 10.000000000000000, 5.0000000000000000,
          10.000000000000000, 0.50000000000000000 },
  { 97.656250000000114, 10.000000000000000, 5.0000000000000000,
          10.000000000000000, 0.60000000000000009 },
  { 411.52263374485580, 10.000000000000000, 5.0000000000000000,
          10.000000000000000, 0.69999999999999996 },
  { 3124.9999999999991, 10.000000000000000, 5.0000000000000000,
          10.000000000000000, 0.80000000000000004 },
  { 99999.999999999665, 10.000000000000000, 5.0000000000000000,
          10.000000000000000, 0.89999999999999991 },
};

// Test function for a=10.000000000000000, b=5.0000000000000000, c=10.000000000000000.
template <typename Tp>
void test200()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data200)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data200[i].a), Tp(data200[i].b),
                   Tp(data200[i].c), Tp(data200[i].x));
      const Tp f0 = data200[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=10.000000000000000, b=10.000000000000000, c=2.0000000000000000.
testcase_hyperg<double> data201[] = {
  { 2.3388730079478623e-05, 10.000000000000000, 10.000000000000000,
          2.0000000000000000, -0.90000000000000002 },
  { -2.3204970759807341e-05, 10.000000000000000, 10.000000000000000,
          2.0000000000000000, -0.80000000000000004 },
  { -0.00016219730505520291, 10.000000000000000, 10.000000000000000,
          2.0000000000000000, -0.69999999999999996 },
  { -0.00044366962360925706, 10.000000000000000, 10.000000000000000,
          2.0000000000000000, -0.59999999999999998 },
  { -0.00071863577205454770, 10.000000000000000, 10.000000000000000,
          2.0000000000000000, -0.50000000000000000 },
  { 4.4378596544453810e-05, 10.000000000000000, 10.000000000000000,
          2.0000000000000000, -0.40000000000000002 },
  { 0.0044446568070623570, 10.000000000000000, 10.000000000000000,
          2.0000000000000000, -0.30000000000000004 },
  { 0.0071045155183571103, 10.000000000000000, 10.000000000000000,
          2.0000000000000000, -0.19999999999999996 },
  { -0.049961558159890146, 10.000000000000000, 10.000000000000000,
          2.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 10.000000000000000, 10.000000000000000,
          2.0000000000000000, 0.0000000000000000 },
  { 51.305449964107403, 10.000000000000000, 10.000000000000000,
          2.0000000000000000, 0.10000000000000009 },
  { 1435.9545414461309, 10.000000000000000, 10.000000000000000,
          2.0000000000000000, 0.19999999999999996 },
  { 39657.913058984115, 10.000000000000000, 10.000000000000000,
          2.0000000000000000, 0.30000000000000004 },
  { 1346016.4468570501, 10.000000000000000, 10.000000000000000,
          2.0000000000000000, 0.39999999999999991 },
  { 68086556.444444403, 10.000000000000000, 10.000000000000000,
          2.0000000000000000, 0.50000000000000000 },
  { 6646235808.7301531, 10.000000000000000, 10.000000000000000,
          2.0000000000000000, 0.60000000000000009 },
  { 1954852335479.9702, 10.000000000000000, 10.000000000000000,
          2.0000000000000000, 0.69999999999999996 },
  { 4573796225043418.0, 10.000000000000000, 10.000000000000000,
          2.0000000000000000, 0.80000000000000004 },
  { 1.8280190368899683e+21, 10.000000000000000, 10.000000000000000,
          2.0000000000000000, 0.89999999999999991 },
};

// Test function for a=10.000000000000000, b=10.000000000000000, c=2.0000000000000000.
template <typename Tp>
void test201()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data201)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data201[i].a), Tp(data201[i].b),
                   Tp(data201[i].c), Tp(data201[i].x));
      const Tp f0 = data201[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(1.0000000000000006e-10));
}

// Test data for a=10.000000000000000, b=10.000000000000000, c=4.0000000000000000.
testcase_hyperg<double> data202[] = {
  { 1.3504013648882651e-05, 10.000000000000000, 10.000000000000000,
          4.0000000000000000, -0.90000000000000002 },
  { 3.1753432098404372e-05, 10.000000000000000, 10.000000000000000,
          4.0000000000000000, -0.80000000000000004 },
  { 6.2032098207659688e-05, 10.000000000000000, 10.000000000000000,
          4.0000000000000000, -0.69999999999999996 },
  { 8.8747213942768282e-05, 10.000000000000000, 10.000000000000000,
          4.0000000000000000, -0.59999999999999998 },
  { 1.0478094697243911e-05, 10.000000000000000, 10.000000000000000,
          4.0000000000000000, -0.50000000000000000 },
  { -0.00055998751006011325, 10.000000000000000, 10.000000000000000,
          4.0000000000000000, -0.40000000000000002 },
  { -0.0024718654966577563, 10.000000000000000, 10.000000000000000,
          4.0000000000000000, -0.30000000000000004 },
  { -0.0027000264053619817, 10.000000000000000, 10.000000000000000,
          4.0000000000000000, -0.19999999999999996 },
  { 0.066515394406810674, 10.000000000000000, 10.000000000000000,
          4.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 10.000000000000000, 10.000000000000000,
          4.0000000000000000, 0.0000000000000000 },
  { 11.579200866389527, 10.000000000000000, 10.000000000000000,
          4.0000000000000000, 0.10000000000000009 },
  { 137.50750548795256, 10.000000000000000, 10.000000000000000,
          4.0000000000000000, 0.19999999999999996 },
  { 1901.3196072993419, 10.000000000000000, 10.000000000000000,
          4.0000000000000000, 0.30000000000000004 },
  { 34210.659507137796, 10.000000000000000, 10.000000000000000,
          4.0000000000000000, 0.39999999999999991 },
  { 920588.19047619053, 10.000000000000000, 10.000000000000000,
          4.0000000000000000, 0.50000000000000000 },
  { 45876220.933028772, 10.000000000000000, 10.000000000000000,
          4.0000000000000000, 0.60000000000000009 },
  { 6234608574.0963297, 10.000000000000000, 10.000000000000000,
          4.0000000000000000, 0.69999999999999996 },
  { 5445391090029.7783, 10.000000000000000, 10.000000000000000,
          4.0000000000000000, 0.80000000000000004 },
  { 4.6508713107142163e+17, 10.000000000000000, 10.000000000000000,
          4.0000000000000000, 0.89999999999999991 },
};

// Test function for a=10.000000000000000, b=10.000000000000000, c=4.0000000000000000.
template <typename Tp>
void test202()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data202)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data202[i].a), Tp(data202[i].b),
                   Tp(data202[i].c), Tp(data202[i].x));
      const Tp f0 = data202[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000013e-09));
}

// Test data for a=10.000000000000000, b=10.000000000000000, c=6.0000000000000000.
testcase_hyperg<double> data203[] = {
  { -2.6846726901567720e-05, 10.000000000000000, 10.000000000000000,
          6.0000000000000000, -0.90000000000000002 },
  { -4.7817237144207266e-05, 10.000000000000000, 10.000000000000000,
          6.0000000000000000, -0.80000000000000004 },
  { -7.2908121941975601e-05, 10.000000000000000, 10.000000000000000,
          6.0000000000000000, -0.69999999999999996 },
  { -6.0427853197480476e-05, 10.000000000000000, 10.000000000000000,
          6.0000000000000000, -0.59999999999999998 },
  { 0.00020559720946645182, 10.000000000000000, 10.000000000000000,
          6.0000000000000000, -0.50000000000000000 },
  { 0.0017056910683365828, 10.000000000000000, 10.000000000000000,
          6.0000000000000000, -0.40000000000000002 },
  { 0.0088037230970526795, 10.000000000000000, 10.000000000000000,
          6.0000000000000000, -0.30000000000000004 },
  { 0.041510819735141674, 10.000000000000000, 10.000000000000000,
          6.0000000000000000, -0.19999999999999996 },
  { 0.19754880805677258, 10.000000000000000, 10.000000000000000,
          6.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 10.000000000000000, 10.000000000000000,
          6.0000000000000000, 0.0000000000000000 },
  { 5.6130947302779246, 10.000000000000000, 10.000000000000000,
          6.0000000000000000, 0.10000000000000009 },
  { 36.475357196722442, 10.000000000000000, 10.000000000000000,
          6.0000000000000000, 0.19999999999999996 },
  { 289.29483001400672, 10.000000000000000, 10.000000000000000,
          6.0000000000000000, 0.30000000000000004 },
  { 3010.8676549536503, 10.000000000000000, 10.000000000000000,
          6.0000000000000000, 0.39999999999999991 },
  { 45844.317460317419, 10.000000000000000, 10.000000000000000,
          6.0000000000000000, 0.50000000000000000 },
  { 1221852.6431492427, 10.000000000000000, 10.000000000000000,
          6.0000000000000000, 0.60000000000000009 },
  { 79585968.928968787, 10.000000000000000, 10.000000000000000,
          6.0000000000000000, 0.69999999999999996 },
  { 26733475942.460335, 10.000000000000000, 10.000000000000000,
          6.0000000000000000, 0.80000000000000004 },
  { 500206428571421.19, 10.000000000000000, 10.000000000000000,
          6.0000000000000000, 0.89999999999999991 },
};

// Test function for a=10.000000000000000, b=10.000000000000000, c=6.0000000000000000.
template <typename Tp>
void test203()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data203)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data203[i].a), Tp(data203[i].b),
                   Tp(data203[i].c), Tp(data203[i].x));
      const Tp f0 = data203[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000017e-10));
}

// Test data for a=10.000000000000000, b=10.000000000000000, c=8.0000000000000000.
testcase_hyperg<double> data204[] = {
  { 0.00025866179054245944, 10.000000000000000, 10.000000000000000,
          8.0000000000000000, -0.90000000000000002 },
  { 0.00053402577739214288, 10.000000000000000, 10.000000000000000,
          8.0000000000000000, -0.80000000000000004 },
  { 0.0011390075227240345, 10.000000000000000, 10.000000000000000,
          8.0000000000000000, -0.69999999999999996 },
  { 0.0025224267119483192, 10.000000000000000, 10.000000000000000,
          8.0000000000000000, -0.59999999999999998 },
  { 0.0058340332124251467, 10.000000000000000, 10.000000000000000,
          8.0000000000000000, -0.50000000000000000 },
  { 0.014189256143045285, 10.000000000000000, 10.000000000000000,
          8.0000000000000000, -0.40000000000000002 },
  { 0.036590990011337567, 10.000000000000000, 10.000000000000000,
          8.0000000000000000, -0.30000000000000004 },
  { 0.10106560781146992, 10.000000000000000, 10.000000000000000,
          8.0000000000000000, -0.19999999999999996 },
  { 0.30278778538531409, 10.000000000000000, 10.000000000000000,
          8.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 10.000000000000000, 10.000000000000000,
          8.0000000000000000, 0.0000000000000000 },
  { 3.7187249990350599, 10.000000000000000, 10.000000000000000,
          8.0000000000000000, 0.10000000000000009 },
  { 16.023275545901704, 10.000000000000000, 10.000000000000000,
          8.0000000000000000, 0.19999999999999996 },
  { 83.265377219882822, 10.000000000000000, 10.000000000000000,
          8.0000000000000000, 0.30000000000000004 },
  { 553.31413918843987, 10.000000000000000, 10.000000000000000,
          8.0000000000000000, 0.39999999999999991 },
  { 5148.4444444444416, 10.000000000000000, 10.000000000000000,
          8.0000000000000000, 0.50000000000000000 },
  { 78082.084655761908, 10.000000000000000, 10.000000000000000,
          8.0000000000000000, 0.60000000000000009 },
  { 2565874.8781353114, 10.000000000000000, 10.000000000000000,
          8.0000000000000000, 0.69999999999999996 },
  { 346137152.77777809, 10.000000000000000, 10.000000000000000,
          8.0000000000000000, 0.80000000000000004 },
  { 1472499999999.9834, 10.000000000000000, 10.000000000000000,
          8.0000000000000000, 0.89999999999999991 },
};

// Test function for a=10.000000000000000, b=10.000000000000000, c=8.0000000000000000.
template <typename Tp>
void test204()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data204)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data204[i].a), Tp(data204[i].b),
                   Tp(data204[i].c), Tp(data204[i].x));
      const Tp f0 = data204[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(1.0000000000000006e-10));
}

// Test data for a=10.000000000000000, b=10.000000000000000, c=10.000000000000000.
testcase_hyperg<double> data205[] = {
  { 0.0016310376661280216, 10.000000000000000, 10.000000000000000,
          10.000000000000000, -0.90000000000000002 },
  { 0.0028007538972582421, 10.000000000000000, 10.000000000000000,
          10.000000000000000, -0.80000000000000004 },
  { 0.0049603324681551939, 10.000000000000000, 10.000000000000000,
          10.000000000000000, -0.69999999999999996 },
  { 0.0090949470177292789, 10.000000000000000, 10.000000000000000,
          10.000000000000000, -0.59999999999999998 },
  { 0.017341529915832606, 10.000000000000000, 10.000000000000000,
          10.000000000000000, -0.50000000000000000 },
  { 0.034571613033607777, 10.000000000000000, 10.000000000000000,
          10.000000000000000, -0.40000000000000002 },
  { 0.072538150286405714, 10.000000000000000, 10.000000000000000,
          10.000000000000000, -0.30000000000000004 },
  { 0.16150558288984579, 10.000000000000000, 10.000000000000000,
          10.000000000000000, -0.19999999999999996 },
  { 0.38554328942953148, 10.000000000000000, 10.000000000000000,
          10.000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 10.000000000000000, 10.000000000000000,
          10.000000000000000, 0.0000000000000000 },
  { 2.8679719907924444, 10.000000000000000, 10.000000000000000,
          10.000000000000000, 0.10000000000000009 },
  { 9.3132257461547816, 10.000000000000000, 10.000000000000000,
          10.000000000000000, 0.19999999999999996 },
  { 35.401331746414378, 10.000000000000000, 10.000000000000000,
          10.000000000000000, 0.30000000000000004 },
  { 165.38171687920172, 10.000000000000000, 10.000000000000000,
          10.000000000000000, 0.39999999999999991 },
  { 1024.0000000000000, 10.000000000000000, 10.000000000000000,
          10.000000000000000, 0.50000000000000000 },
  { 9536.7431640625200, 10.000000000000000, 10.000000000000000,
          10.000000000000000, 0.60000000000000009 },
  { 169350.87808430271, 10.000000000000000, 10.000000000000000,
          10.000000000000000, 0.69999999999999996 },
  { 9765624.9999999944, 10.000000000000000, 10.000000000000000,
          10.000000000000000, 0.80000000000000004 },
  { 9999999999.9999332, 10.000000000000000, 10.000000000000000,
          10.000000000000000, 0.89999999999999991 },
};

// Test function for a=10.000000000000000, b=10.000000000000000, c=10.000000000000000.
template <typename Tp>
void test205()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data205)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data205[i].a), Tp(data205[i].b),
                   Tp(data205[i].c), Tp(data205[i].x));
      const Tp f0 = data205[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=10.000000000000000, b=20.000000000000000, c=2.0000000000000000.
testcase_hyperg<double> data206[] = {
  { -2.1776535308707967e-07, 10.000000000000000, 20.000000000000000,
          2.0000000000000000, -0.90000000000000002 },
  { -2.9128833151427998e-06, 10.000000000000000, 20.000000000000000,
          2.0000000000000000, -0.80000000000000004 },
  { -9.4755553429035093e-06, 10.000000000000000, 20.000000000000000,
          2.0000000000000000, -0.69999999999999996 },
  { -1.2844297353813116e-05, 10.000000000000000, 20.000000000000000,
          2.0000000000000000, -0.59999999999999998 },
  { 3.6576965483568809e-05, 10.000000000000000, 20.000000000000000,
          2.0000000000000000, -0.50000000000000000 },
  { 0.00020847453890692649, 10.000000000000000, 20.000000000000000,
          2.0000000000000000, -0.40000000000000002 },
  { -0.00022868510398174632, 10.000000000000000, 20.000000000000000,
          2.0000000000000000, -0.30000000000000004 },
  { -0.0021855513841942732, 10.000000000000000, 20.000000000000000,
          2.0000000000000000, -0.19999999999999996 },
  { 0.014662111759334634, 10.000000000000000, 20.000000000000000,
          2.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 10.000000000000000, 20.000000000000000,
          2.0000000000000000, 0.0000000000000000 },
  { 746.44776348798098, 10.000000000000000, 20.000000000000000,
          2.0000000000000000, 0.10000000000000009 },
  { 136080.48445225612, 10.000000000000000, 20.000000000000000,
          2.0000000000000000, 0.19999999999999996 },
  { 23094279.597826406, 10.000000000000000, 20.000000000000000,
          2.0000000000000000, 0.30000000000000004 },
  { 5315913395.5545301, 10.000000000000000, 20.000000000000000,
          2.0000000000000000, 0.39999999999999991 },
  { 2261935718399.9990, 10.000000000000000, 20.000000000000000,
          2.0000000000000000, 0.50000000000000000 },
  { 2669150854828235.0, 10.000000000000000, 20.000000000000000,
          2.0000000000000000, 0.60000000000000009 },
  { 1.7499363099365994e+19, 10.000000000000000, 20.000000000000000,
          2.0000000000000000, 0.69999999999999996 },
  { 2.8881518494606140e+24, 10.000000000000000, 20.000000000000000,
          2.0000000000000000, 0.80000000000000004 },
  { 1.4165513933661626e+33, 10.000000000000000, 20.000000000000000,
          2.0000000000000000, 0.89999999999999991 },
};

// Test function for a=10.000000000000000, b=20.000000000000000, c=2.0000000000000000.
template <typename Tp>
void test206()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data206)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data206[i].a), Tp(data206[i].b),
                   Tp(data206[i].c), Tp(data206[i].x));
      const Tp f0 = data206[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(1.0000000000000005e-08));
}

// Test data for a=10.000000000000000, b=20.000000000000000, c=4.0000000000000000.
testcase_hyperg<double> data207[] = {
  { 1.7149006973860441e-07, 10.000000000000000, 20.000000000000000,
          4.0000000000000000, -0.90000000000000002 },
  { 3.2399324904088936e-07, 10.000000000000000, 20.000000000000000,
          4.0000000000000000, -0.80000000000000004 },
  { 1.6015317712089860e-07, 10.000000000000000, 20.000000000000000,
          4.0000000000000000, -0.69999999999999996 },
  { -2.0500917204199595e-06, 10.000000000000000, 20.000000000000000,
          4.0000000000000000, -0.59999999999999998 },
  { -1.0175546788599472e-05, 10.000000000000000, 20.000000000000000,
          4.0000000000000000, -0.50000000000000000 },
  { -1.1720101988202453e-05, 10.000000000000000, 20.000000000000000,
          4.0000000000000000, -0.40000000000000002 },
  { 0.00014199637113974185, 10.000000000000000, 20.000000000000000,
          4.0000000000000000, -0.30000000000000004 },
  { 0.00021263363640642297, 10.000000000000000, 20.000000000000000,
          4.0000000000000000, -0.19999999999999996 },
  { -0.0072649256698441751, 10.000000000000000, 20.000000000000000,
          4.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 10.000000000000000, 20.000000000000000,
          4.0000000000000000, 0.0000000000000000 },
  { 90.430293772869618, 10.000000000000000, 20.000000000000000,
          4.0000000000000000, 0.10000000000000009 },
  { 6248.1455940292308, 10.000000000000000, 20.000000000000000,
          4.0000000000000000, 0.19999999999999996 },
  { 501143.39852548984, 10.000000000000000, 20.000000000000000,
          4.0000000000000000, 0.30000000000000004 },
  { 58852027.356439680, 10.000000000000000, 20.000000000000000,
          4.0000000000000000, 0.39999999999999991 },
  { 12942923093.333330, 10.000000000000000, 20.000000000000000,
          4.0000000000000000, 0.50000000000000000 },
  { 7618073993853.6592, 10.000000000000000, 20.000000000000000,
          4.0000000000000000, 0.60000000000000009 },
  { 22630251562549288., 10.000000000000000, 20.000000000000000,
          4.0000000000000000, 0.69999999999999996 },
  { 1.3708372433980356e+21, 10.000000000000000, 20.000000000000000,
          4.0000000000000000, 0.80000000000000004 },
  { 1.4154113619999653e+29, 10.000000000000000, 20.000000000000000,
          4.0000000000000000, 0.89999999999999991 },
};

// Test function for a=10.000000000000000, b=20.000000000000000, c=4.0000000000000000.
template <typename Tp>
void test207()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data207)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data207[i].a), Tp(data207[i].b),
                   Tp(data207[i].c), Tp(data207[i].x));
      const Tp f0 = data207[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(5.0000000000000024e-08));
}

// Test data for a=10.000000000000000, b=20.000000000000000, c=6.0000000000000000.
testcase_hyperg<double> data208[] = {
  { -1.6667473370780257e-08, 10.000000000000000, 20.000000000000000,
          6.0000000000000000, -0.90000000000000002 },
  { 8.6214844067774863e-08, 10.000000000000000, 20.000000000000000,
          6.0000000000000000, -0.80000000000000004 },
  { 5.7778331238835108e-07, 10.000000000000000, 20.000000000000000,
          6.0000000000000000, -0.69999999999999996 },
  { 2.1911400500362969e-06, 10.000000000000000, 20.000000000000000,
          6.0000000000000000, -0.59999999999999998 },
  { 4.7440049217100417e-06, 10.000000000000000, 20.000000000000000,
          6.0000000000000000, -0.50000000000000000 },
  { -1.0564233314924258e-05, 10.000000000000000, 20.000000000000000,
          6.0000000000000000, -0.40000000000000002 },
  { -0.00017990026051856349, 10.000000000000000, 20.000000000000000,
          6.0000000000000000, -0.30000000000000004 },
  { -0.00027618146288744351, 10.000000000000000, 20.000000000000000,
          6.0000000000000000, -0.19999999999999996 },
  { 0.030606019577723392, 10.000000000000000, 20.000000000000000,
          6.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 10.000000000000000, 20.000000000000000,
          6.0000000000000000, 0.0000000000000000 },
  { 27.832854169493341, 10.000000000000000, 20.000000000000000,
          6.0000000000000000, 0.10000000000000009 },
  { 874.00624088575228, 10.000000000000000, 20.000000000000000,
          6.0000000000000000, 0.19999999999999996 },
  { 36049.199340831554, 10.000000000000000, 20.000000000000000,
          6.0000000000000000, 0.30000000000000004 },
  { 2270967.7298624986, 10.000000000000000, 20.000000000000000,
          6.0000000000000000, 0.39999999999999991 },
  { 266979100.44444439, 10.000000000000000, 20.000000000000000,
          6.0000000000000000, 0.50000000000000000 },
  { 80311224337.493027, 10.000000000000000, 20.000000000000000,
          6.0000000000000000, 0.60000000000000009 },
  { 110111693103799.72, 10.000000000000000, 20.000000000000000,
          6.0000000000000000, 0.69999999999999996 },
  { 2.4838871426052618e+18, 10.000000000000000, 20.000000000000000,
          6.0000000000000000, 0.80000000000000004 },
  { 5.4626349999998603e+25, 10.000000000000000, 20.000000000000000,
          6.0000000000000000, 0.89999999999999991 },
};

// Test function for a=10.000000000000000, b=20.000000000000000, c=6.0000000000000000.
template <typename Tp>
void test208()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data208)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data208[i].a), Tp(data208[i].b),
                   Tp(data208[i].c), Tp(data208[i].x));
      const Tp f0 = data208[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(5.0000000000000019e-07));
}

// Test data for a=10.000000000000000, b=20.000000000000000, c=8.0000000000000000.
testcase_hyperg<double> data209[] = {
  { -1.5843795893321480e-07, 10.000000000000000, 20.000000000000000,
          8.0000000000000000, -0.90000000000000002 },
  { -5.4877275994033766e-07, 10.000000000000000, 20.000000000000000,
          8.0000000000000000, -0.80000000000000004 },
  { -1.7169507967745992e-06, 10.000000000000000, 20.000000000000000,
          8.0000000000000000, -0.69999999999999996 },
  { -4.5236439748752000e-06, 10.000000000000000, 20.000000000000000,
          8.0000000000000000, -0.59999999999999998 },
  { -5.5690492560325806e-06, 10.000000000000000, 20.000000000000000,
          8.0000000000000000, -0.50000000000000000 },
  { 5.6914115606934911e-05, 10.000000000000000, 20.000000000000000,
          8.0000000000000000, -0.40000000000000002 },
  { 0.00082507252097519922, 10.000000000000000, 20.000000000000000,
          8.0000000000000000, -0.30000000000000004 },
  { 0.0085739249288229857, 10.000000000000000, 20.000000000000000,
          8.0000000000000000, -0.19999999999999996 },
  { 0.088244357683754757, 10.000000000000000, 20.000000000000000,
          8.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 10.000000000000000, 20.000000000000000,
          8.0000000000000000, 0.0000000000000000 },
  { 13.387208440156897, 10.000000000000000, 20.000000000000000,
          8.0000000000000000, 0.10000000000000009 },
  { 226.77895441155110, 10.000000000000000, 20.000000000000000,
          8.0000000000000000, 0.19999999999999996 },
  { 5281.5716482686785, 10.000000000000000, 20.000000000000000,
          8.0000000000000000, 0.30000000000000004 },
  { 189431.77762850464, 10.000000000000000, 20.000000000000000,
          8.0000000000000000, 0.39999999999999991 },
  { 12408149.333333332, 10.000000000000000, 20.000000000000000,
          8.0000000000000000, 0.50000000000000000 },
  { 1966782292.5839682, 10.000000000000000, 20.000000000000000,
          8.0000000000000000, 0.60000000000000009 },
  { 1274123112205.7495, 10.000000000000000, 20.000000000000000,
          8.0000000000000000, 0.69999999999999996 },
  { 10903676350911508., 10.000000000000000, 20.000000000000000,
          8.0000000000000000, 0.80000000000000004 },
  { 5.1849999999998819e+22, 10.000000000000000, 20.000000000000000,
          8.0000000000000000, 0.89999999999999991 },
};

// Test function for a=10.000000000000000, b=20.000000000000000, c=8.0000000000000000.
template <typename Tp>
void test209()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data209)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data209[i].a), Tp(data209[i].b),
                   Tp(data209[i].c), Tp(data209[i].x));
      const Tp f0 = data209[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000012e-08));
}

// Test data for a=10.000000000000000, b=20.000000000000000, c=10.000000000000000.
testcase_hyperg<double> data210[] = {
  { 2.6602838683283435e-06, 10.000000000000000, 20.000000000000000,
          10.000000000000000, -0.90000000000000002 },
  { 7.8442223930072316e-06, 10.000000000000000, 20.000000000000000,
          10.000000000000000, -0.80000000000000004 },
  { 2.4604898194634598e-05, 10.000000000000000, 20.000000000000000,
          10.000000000000000, -0.69999999999999996 },
  { 8.2718061255302686e-05, 10.000000000000000, 20.000000000000000,
          10.000000000000000, -0.59999999999999998 },
  { 0.00030072865982171723, 10.000000000000000, 20.000000000000000,
          10.000000000000000, -0.50000000000000000 },
  { 0.0011951964277455193, 10.000000000000000, 20.000000000000000,
          10.000000000000000, -0.40000000000000002 },
  { 0.0052617832469731814, 10.000000000000000, 20.000000000000000,
          10.000000000000000, -0.30000000000000004 },
  { 0.026084053304588847, 10.000000000000000, 20.000000000000000,
          10.000000000000000, -0.19999999999999996 },
  { 0.14864362802414346, 10.000000000000000, 20.000000000000000,
          10.000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 10.000000000000000, 20.000000000000000,
          10.000000000000000, 0.0000000000000000 },
  { 8.2252633399699757, 10.000000000000000, 20.000000000000000,
          10.000000000000000, 0.10000000000000009 },
  { 86.736173798840269, 10.000000000000000, 20.000000000000000,
          10.000000000000000, 0.19999999999999996 },
  { 1253.2542894196865, 10.000000000000000, 20.000000000000000,
          10.000000000000000, 0.30000000000000004 },
  { 27351.112277912434, 10.000000000000000, 20.000000000000000,
          10.000000000000000, 0.39999999999999991 },
  { 1048576.0000000000, 10.000000000000000, 20.000000000000000,
          10.000000000000000, 0.50000000000000000 },
  { 90949470.177293226, 10.000000000000000, 20.000000000000000,
          10.000000000000000, 0.60000000000000009 },
  { 28679719907.924358, 10.000000000000000, 20.000000000000000,
          10.000000000000000, 0.69999999999999996 },
  { 95367431640624.906, 10.000000000000000, 20.000000000000000,
          10.000000000000000, 0.80000000000000004 },
  { 9.9999999999998657e+19, 10.000000000000000, 20.000000000000000,
          10.000000000000000, 0.89999999999999991 },
};

// Test function for a=10.000000000000000, b=20.000000000000000, c=10.000000000000000.
template <typename Tp>
void test210()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data210)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data210[i].a), Tp(data210[i].b),
                   Tp(data210[i].c), Tp(data210[i].x));
      const Tp f0 = data210[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(5.0000000000000039e-13));
}

// Test data for a=20.000000000000000, b=0.0000000000000000, c=2.0000000000000000.
testcase_hyperg<double> data211[] = {
  { 1.0000000000000000, 20.000000000000000, 0.0000000000000000,
          2.0000000000000000, -0.90000000000000002 },
  { 1.0000000000000000, 20.000000000000000, 0.0000000000000000,
          2.0000000000000000, -0.80000000000000004 },
  { 1.0000000000000000, 20.000000000000000, 0.0000000000000000,
          2.0000000000000000, -0.69999999999999996 },
  { 1.0000000000000000, 20.000000000000000, 0.0000000000000000,
          2.0000000000000000, -0.59999999999999998 },
  { 1.0000000000000000, 20.000000000000000, 0.0000000000000000,
          2.0000000000000000, -0.50000000000000000 },
  { 1.0000000000000000, 20.000000000000000, 0.0000000000000000,
          2.0000000000000000, -0.40000000000000002 },
  { 1.0000000000000000, 20.000000000000000, 0.0000000000000000,
          2.0000000000000000, -0.30000000000000004 },
  { 1.0000000000000000, 20.000000000000000, 0.0000000000000000,
          2.0000000000000000, -0.19999999999999996 },
  { 1.0000000000000000, 20.000000000000000, 0.0000000000000000,
          2.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 20.000000000000000, 0.0000000000000000,
          2.0000000000000000, 0.0000000000000000 },
  { 1.0000000000000000, 20.000000000000000, 0.0000000000000000,
          2.0000000000000000, 0.10000000000000009 },
  { 1.0000000000000000, 20.000000000000000, 0.0000000000000000,
          2.0000000000000000, 0.19999999999999996 },
  { 1.0000000000000000, 20.000000000000000, 0.0000000000000000,
          2.0000000000000000, 0.30000000000000004 },
  { 1.0000000000000000, 20.000000000000000, 0.0000000000000000,
          2.0000000000000000, 0.39999999999999991 },
  { 1.0000000000000000, 20.000000000000000, 0.0000000000000000,
          2.0000000000000000, 0.50000000000000000 },
  { 1.0000000000000000, 20.000000000000000, 0.0000000000000000,
          2.0000000000000000, 0.60000000000000009 },
  { 1.0000000000000000, 20.000000000000000, 0.0000000000000000,
          2.0000000000000000, 0.69999999999999996 },
  { 1.0000000000000000, 20.000000000000000, 0.0000000000000000,
          2.0000000000000000, 0.80000000000000004 },
  { 1.0000000000000000, 20.000000000000000, 0.0000000000000000,
          2.0000000000000000, 0.89999999999999991 },
};

// Test function for a=20.000000000000000, b=0.0000000000000000, c=2.0000000000000000.
template <typename Tp>
void test211()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data211)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data211[i].a), Tp(data211[i].b),
                   Tp(data211[i].c), Tp(data211[i].x));
      const Tp f0 = data211[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=20.000000000000000, b=0.0000000000000000, c=4.0000000000000000.
testcase_hyperg<double> data212[] = {
  { 1.0000000000000000, 20.000000000000000, 0.0000000000000000,
          4.0000000000000000, -0.90000000000000002 },
  { 1.0000000000000000, 20.000000000000000, 0.0000000000000000,
          4.0000000000000000, -0.80000000000000004 },
  { 1.0000000000000000, 20.000000000000000, 0.0000000000000000,
          4.0000000000000000, -0.69999999999999996 },
  { 1.0000000000000000, 20.000000000000000, 0.0000000000000000,
          4.0000000000000000, -0.59999999999999998 },
  { 1.0000000000000000, 20.000000000000000, 0.0000000000000000,
          4.0000000000000000, -0.50000000000000000 },
  { 1.0000000000000000, 20.000000000000000, 0.0000000000000000,
          4.0000000000000000, -0.40000000000000002 },
  { 1.0000000000000000, 20.000000000000000, 0.0000000000000000,
          4.0000000000000000, -0.30000000000000004 },
  { 1.0000000000000000, 20.000000000000000, 0.0000000000000000,
          4.0000000000000000, -0.19999999999999996 },
  { 1.0000000000000000, 20.000000000000000, 0.0000000000000000,
          4.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 20.000000000000000, 0.0000000000000000,
          4.0000000000000000, 0.0000000000000000 },
  { 1.0000000000000000, 20.000000000000000, 0.0000000000000000,
          4.0000000000000000, 0.10000000000000009 },
  { 1.0000000000000000, 20.000000000000000, 0.0000000000000000,
          4.0000000000000000, 0.19999999999999996 },
  { 1.0000000000000000, 20.000000000000000, 0.0000000000000000,
          4.0000000000000000, 0.30000000000000004 },
  { 1.0000000000000000, 20.000000000000000, 0.0000000000000000,
          4.0000000000000000, 0.39999999999999991 },
  { 1.0000000000000000, 20.000000000000000, 0.0000000000000000,
          4.0000000000000000, 0.50000000000000000 },
  { 1.0000000000000000, 20.000000000000000, 0.0000000000000000,
          4.0000000000000000, 0.60000000000000009 },
  { 1.0000000000000000, 20.000000000000000, 0.0000000000000000,
          4.0000000000000000, 0.69999999999999996 },
  { 1.0000000000000000, 20.000000000000000, 0.0000000000000000,
          4.0000000000000000, 0.80000000000000004 },
  { 1.0000000000000000, 20.000000000000000, 0.0000000000000000,
          4.0000000000000000, 0.89999999999999991 },
};

// Test function for a=20.000000000000000, b=0.0000000000000000, c=4.0000000000000000.
template <typename Tp>
void test212()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data212)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data212[i].a), Tp(data212[i].b),
                   Tp(data212[i].c), Tp(data212[i].x));
      const Tp f0 = data212[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=20.000000000000000, b=0.0000000000000000, c=6.0000000000000000.
testcase_hyperg<double> data213[] = {
  { 1.0000000000000000, 20.000000000000000, 0.0000000000000000,
          6.0000000000000000, -0.90000000000000002 },
  { 1.0000000000000000, 20.000000000000000, 0.0000000000000000,
          6.0000000000000000, -0.80000000000000004 },
  { 1.0000000000000000, 20.000000000000000, 0.0000000000000000,
          6.0000000000000000, -0.69999999999999996 },
  { 1.0000000000000000, 20.000000000000000, 0.0000000000000000,
          6.0000000000000000, -0.59999999999999998 },
  { 1.0000000000000000, 20.000000000000000, 0.0000000000000000,
          6.0000000000000000, -0.50000000000000000 },
  { 1.0000000000000000, 20.000000000000000, 0.0000000000000000,
          6.0000000000000000, -0.40000000000000002 },
  { 1.0000000000000000, 20.000000000000000, 0.0000000000000000,
          6.0000000000000000, -0.30000000000000004 },
  { 1.0000000000000000, 20.000000000000000, 0.0000000000000000,
          6.0000000000000000, -0.19999999999999996 },
  { 1.0000000000000000, 20.000000000000000, 0.0000000000000000,
          6.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 20.000000000000000, 0.0000000000000000,
          6.0000000000000000, 0.0000000000000000 },
  { 1.0000000000000000, 20.000000000000000, 0.0000000000000000,
          6.0000000000000000, 0.10000000000000009 },
  { 1.0000000000000000, 20.000000000000000, 0.0000000000000000,
          6.0000000000000000, 0.19999999999999996 },
  { 1.0000000000000000, 20.000000000000000, 0.0000000000000000,
          6.0000000000000000, 0.30000000000000004 },
  { 1.0000000000000000, 20.000000000000000, 0.0000000000000000,
          6.0000000000000000, 0.39999999999999991 },
  { 1.0000000000000000, 20.000000000000000, 0.0000000000000000,
          6.0000000000000000, 0.50000000000000000 },
  { 1.0000000000000000, 20.000000000000000, 0.0000000000000000,
          6.0000000000000000, 0.60000000000000009 },
  { 1.0000000000000000, 20.000000000000000, 0.0000000000000000,
          6.0000000000000000, 0.69999999999999996 },
  { 1.0000000000000000, 20.000000000000000, 0.0000000000000000,
          6.0000000000000000, 0.80000000000000004 },
  { 1.0000000000000000, 20.000000000000000, 0.0000000000000000,
          6.0000000000000000, 0.89999999999999991 },
};

// Test function for a=20.000000000000000, b=0.0000000000000000, c=6.0000000000000000.
template <typename Tp>
void test213()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data213)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data213[i].a), Tp(data213[i].b),
                   Tp(data213[i].c), Tp(data213[i].x));
      const Tp f0 = data213[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=20.000000000000000, b=0.0000000000000000, c=8.0000000000000000.
testcase_hyperg<double> data214[] = {
  { 1.0000000000000000, 20.000000000000000, 0.0000000000000000,
          8.0000000000000000, -0.90000000000000002 },
  { 1.0000000000000000, 20.000000000000000, 0.0000000000000000,
          8.0000000000000000, -0.80000000000000004 },
  { 1.0000000000000000, 20.000000000000000, 0.0000000000000000,
          8.0000000000000000, -0.69999999999999996 },
  { 1.0000000000000000, 20.000000000000000, 0.0000000000000000,
          8.0000000000000000, -0.59999999999999998 },
  { 1.0000000000000000, 20.000000000000000, 0.0000000000000000,
          8.0000000000000000, -0.50000000000000000 },
  { 1.0000000000000000, 20.000000000000000, 0.0000000000000000,
          8.0000000000000000, -0.40000000000000002 },
  { 1.0000000000000000, 20.000000000000000, 0.0000000000000000,
          8.0000000000000000, -0.30000000000000004 },
  { 1.0000000000000000, 20.000000000000000, 0.0000000000000000,
          8.0000000000000000, -0.19999999999999996 },
  { 1.0000000000000000, 20.000000000000000, 0.0000000000000000,
          8.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 20.000000000000000, 0.0000000000000000,
          8.0000000000000000, 0.0000000000000000 },
  { 1.0000000000000000, 20.000000000000000, 0.0000000000000000,
          8.0000000000000000, 0.10000000000000009 },
  { 1.0000000000000000, 20.000000000000000, 0.0000000000000000,
          8.0000000000000000, 0.19999999999999996 },
  { 1.0000000000000000, 20.000000000000000, 0.0000000000000000,
          8.0000000000000000, 0.30000000000000004 },
  { 1.0000000000000000, 20.000000000000000, 0.0000000000000000,
          8.0000000000000000, 0.39999999999999991 },
  { 1.0000000000000000, 20.000000000000000, 0.0000000000000000,
          8.0000000000000000, 0.50000000000000000 },
  { 1.0000000000000000, 20.000000000000000, 0.0000000000000000,
          8.0000000000000000, 0.60000000000000009 },
  { 1.0000000000000000, 20.000000000000000, 0.0000000000000000,
          8.0000000000000000, 0.69999999999999996 },
  { 1.0000000000000000, 20.000000000000000, 0.0000000000000000,
          8.0000000000000000, 0.80000000000000004 },
  { 1.0000000000000000, 20.000000000000000, 0.0000000000000000,
          8.0000000000000000, 0.89999999999999991 },
};

// Test function for a=20.000000000000000, b=0.0000000000000000, c=8.0000000000000000.
template <typename Tp>
void test214()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data214)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data214[i].a), Tp(data214[i].b),
                   Tp(data214[i].c), Tp(data214[i].x));
      const Tp f0 = data214[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=20.000000000000000, b=0.0000000000000000, c=10.000000000000000.
testcase_hyperg<double> data215[] = {
  { 1.0000000000000000, 20.000000000000000, 0.0000000000000000,
          10.000000000000000, -0.90000000000000002 },
  { 1.0000000000000000, 20.000000000000000, 0.0000000000000000,
          10.000000000000000, -0.80000000000000004 },
  { 1.0000000000000000, 20.000000000000000, 0.0000000000000000,
          10.000000000000000, -0.69999999999999996 },
  { 1.0000000000000000, 20.000000000000000, 0.0000000000000000,
          10.000000000000000, -0.59999999999999998 },
  { 1.0000000000000000, 20.000000000000000, 0.0000000000000000,
          10.000000000000000, -0.50000000000000000 },
  { 1.0000000000000000, 20.000000000000000, 0.0000000000000000,
          10.000000000000000, -0.40000000000000002 },
  { 1.0000000000000000, 20.000000000000000, 0.0000000000000000,
          10.000000000000000, -0.30000000000000004 },
  { 1.0000000000000000, 20.000000000000000, 0.0000000000000000,
          10.000000000000000, -0.19999999999999996 },
  { 1.0000000000000000, 20.000000000000000, 0.0000000000000000,
          10.000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 20.000000000000000, 0.0000000000000000,
          10.000000000000000, 0.0000000000000000 },
  { 1.0000000000000000, 20.000000000000000, 0.0000000000000000,
          10.000000000000000, 0.10000000000000009 },
  { 1.0000000000000000, 20.000000000000000, 0.0000000000000000,
          10.000000000000000, 0.19999999999999996 },
  { 1.0000000000000000, 20.000000000000000, 0.0000000000000000,
          10.000000000000000, 0.30000000000000004 },
  { 1.0000000000000000, 20.000000000000000, 0.0000000000000000,
          10.000000000000000, 0.39999999999999991 },
  { 1.0000000000000000, 20.000000000000000, 0.0000000000000000,
          10.000000000000000, 0.50000000000000000 },
  { 1.0000000000000000, 20.000000000000000, 0.0000000000000000,
          10.000000000000000, 0.60000000000000009 },
  { 1.0000000000000000, 20.000000000000000, 0.0000000000000000,
          10.000000000000000, 0.69999999999999996 },
  { 1.0000000000000000, 20.000000000000000, 0.0000000000000000,
          10.000000000000000, 0.80000000000000004 },
  { 1.0000000000000000, 20.000000000000000, 0.0000000000000000,
          10.000000000000000, 0.89999999999999991 },
};

// Test function for a=20.000000000000000, b=0.0000000000000000, c=10.000000000000000.
template <typename Tp>
void test215()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data215)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data215[i].a), Tp(data215[i].b),
                   Tp(data215[i].c), Tp(data215[i].x));
      const Tp f0 = data215[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=20.000000000000000, b=0.50000000000000000, c=2.0000000000000000.
testcase_hyperg<double> data216[] = {
  { 0.26690449940521566, 20.000000000000000, 0.50000000000000000,
          2.0000000000000000, -0.90000000000000002 },
  { 0.28252302866181805, 20.000000000000000, 0.50000000000000000,
          2.0000000000000000, -0.80000000000000004 },
  { 0.30123616141153819, 20.000000000000000, 0.50000000000000000,
          2.0000000000000000, -0.69999999999999996 },
  { 0.32421384687602628, 20.000000000000000, 0.50000000000000000,
          2.0000000000000000, -0.59999999999999998 },
  { 0.35334630811776752, 20.000000000000000, 0.50000000000000000,
          2.0000000000000000, -0.50000000000000000 },
  { 0.39191793127467034, 20.000000000000000, 0.50000000000000000,
          2.0000000000000000, -0.40000000000000002 },
  { 0.44620488618129206, 20.000000000000000, 0.50000000000000000,
          2.0000000000000000, -0.30000000000000004 },
  { 0.52980896919265685, 20.000000000000000, 0.50000000000000000,
          2.0000000000000000, -0.19999999999999996 },
  { 0.67754711477562357, 20.000000000000000, 0.50000000000000000,
          2.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 20.000000000000000, 0.50000000000000000,
          2.0000000000000000, 0.0000000000000000 },
  { 1.9567557771780317, 20.000000000000000, 0.50000000000000000,
          2.0000000000000000, 0.10000000000000009 },
  { 6.1816042148333086, 20.000000000000000, 0.50000000000000000,
          2.0000000000000000, 0.19999999999999996 },
  { 35.653088618561227, 20.000000000000000, 0.50000000000000000,
          2.0000000000000000, 0.30000000000000004 },
  { 377.51482843179906, 20.000000000000000, 0.50000000000000000,
          2.0000000000000000, 0.39999999999999991 },
  { 7645.8816551195359, 20.000000000000000, 0.50000000000000000,
          2.0000000000000000, 0.50000000000000000 },
  { 354791.74537980522, 20.000000000000000, 0.50000000000000000,
          2.0000000000000000, 0.60000000000000009 },
  { 57009889.966638684, 20.000000000000000, 0.50000000000000000,
          2.0000000000000000, 0.69999999999999996 },
  { 83771357024.863937, 20.000000000000000, 0.50000000000000000,
          2.0000000000000000, 0.80000000000000004 },
  { 25866972896376408., 20.000000000000000, 0.50000000000000000,
          2.0000000000000000, 0.89999999999999991 },
};

// Test function for a=20.000000000000000, b=0.50000000000000000, c=2.0000000000000000.
template <typename Tp>
void test216()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data216)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data216[i].a), Tp(data216[i].b),
                   Tp(data216[i].c), Tp(data216[i].x));
      const Tp f0 = data216[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=20.000000000000000, b=0.50000000000000000, c=4.0000000000000000.
testcase_hyperg<double> data217[] = {
  { 0.40342659436153405, 20.000000000000000, 0.50000000000000000,
          4.0000000000000000, -0.90000000000000002 },
  { 0.42420571192034318, 20.000000000000000, 0.50000000000000000,
          4.0000000000000000, -0.80000000000000004 },
  { 0.44852768286073008, 20.000000000000000, 0.50000000000000000,
          4.0000000000000000, -0.69999999999999996 },
  { 0.47751245808592863, 20.000000000000000, 0.50000000000000000,
          4.0000000000000000, -0.59999999999999998 },
  { 0.51283632632707754, 20.000000000000000, 0.50000000000000000,
          4.0000000000000000, -0.50000000000000000 },
  { 0.55713468814894307, 20.000000000000000, 0.50000000000000000,
          4.0000000000000000, -0.40000000000000002 },
  { 0.61481320817757312, 20.000000000000000, 0.50000000000000000,
          4.0000000000000000, -0.30000000000000004 },
  { 0.69383483410097202, 20.000000000000000, 0.50000000000000000,
          4.0000000000000000, -0.19999999999999996 },
  { 0.81012002526006033, 20.000000000000000, 0.50000000000000000,
          4.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 20.000000000000000, 0.50000000000000000,
          4.0000000000000000, 0.0000000000000000 },
  { 1.3622225506603911, 20.000000000000000, 0.50000000000000000,
          4.0000000000000000, 0.10000000000000009 },
  { 2.2349513086109001, 20.000000000000000, 0.50000000000000000,
          4.0000000000000000, 0.19999999999999996 },
  { 5.1864917536761723, 20.000000000000000, 0.50000000000000000,
          4.0000000000000000, 0.30000000000000004 },
  { 21.020560423779411, 20.000000000000000, 0.50000000000000000,
          4.0000000000000000, 0.39999999999999991 },
  { 175.19649997100612, 20.000000000000000, 0.50000000000000000,
          4.0000000000000000, 0.50000000000000000 },
  { 3467.1587803688708, 20.000000000000000, 0.50000000000000000,
          4.0000000000000000, 0.60000000000000009 },
  { 225003.88683445856, 20.000000000000000, 0.50000000000000000,
          4.0000000000000000, 0.69999999999999996 },
  { 110837674.65652709, 20.000000000000000, 0.50000000000000000,
          4.0000000000000000, 0.80000000000000004 },
  { 6688966964170.6807, 20.000000000000000, 0.50000000000000000,
          4.0000000000000000, 0.89999999999999991 },
};

// Test function for a=20.000000000000000, b=0.50000000000000000, c=4.0000000000000000.
template <typename Tp>
void test217()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data217)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data217[i].a), Tp(data217[i].b),
                   Tp(data217[i].c), Tp(data217[i].x));
      const Tp f0 = data217[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=20.000000000000000, b=0.50000000000000000, c=6.0000000000000000.
testcase_hyperg<double> data218[] = {
  { 0.48716309885816761, 20.000000000000000, 0.50000000000000000,
          6.0000000000000000, -0.90000000000000002 },
  { 0.50965859152542281, 20.000000000000000, 0.50000000000000000,
          6.0000000000000000, -0.80000000000000004 },
  { 0.53554809210658971, 20.000000000000000, 0.50000000000000000,
          6.0000000000000000, -0.69999999999999996 },
  { 0.56576689207507136, 20.000000000000000, 0.50000000000000000,
          6.0000000000000000, -0.59999999999999998 },
  { 0.60164849637133688, 20.000000000000000, 0.50000000000000000,
          6.0000000000000000, -0.50000000000000000 },
  { 0.64516711595404375, 20.000000000000000, 0.50000000000000000,
          6.0000000000000000, -0.40000000000000002 },
  { 0.69938278735493542, 20.000000000000000, 0.50000000000000000,
          6.0000000000000000, -0.30000000000000004 },
  { 0.76931621518401860, 20.000000000000000, 0.50000000000000000,
          6.0000000000000000, -0.19999999999999996 },
  { 0.86381808725530695, 20.000000000000000, 0.50000000000000000,
          6.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 20.000000000000000, 0.50000000000000000,
          6.0000000000000000, 0.0000000000000000 },
  { 1.2152051956815531, 20.000000000000000, 0.50000000000000000,
          6.0000000000000000, 0.10000000000000009 },
  { 1.6052546785425543, 20.000000000000000, 0.50000000000000000,
          6.0000000000000000, 0.19999999999999996 },
  { 2.4765586046012635, 20.000000000000000, 0.50000000000000000,
          6.0000000000000000, 0.30000000000000004 },
  { 5.1564492216997486, 20.000000000000000, 0.50000000000000000,
          6.0000000000000000, 0.39999999999999991 },
  { 18.446158392136365, 20.000000000000000, 0.50000000000000000,
          6.0000000000000000, 0.50000000000000000 },
  { 150.44577670123971, 20.000000000000000, 0.50000000000000000,
          6.0000000000000000, 0.60000000000000009 },
  { 3862.6317400115768, 20.000000000000000, 0.50000000000000000,
          6.0000000000000000, 0.69999999999999996 },
  { 632428.34833625401, 20.000000000000000, 0.50000000000000000,
          6.0000000000000000, 0.80000000000000004 },
  { 7426927663.3808765, 20.000000000000000, 0.50000000000000000,
          6.0000000000000000, 0.89999999999999991 },
};

// Test function for a=20.000000000000000, b=0.50000000000000000, c=6.0000000000000000.
template <typename Tp>
void test218()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data218)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data218[i].a), Tp(data218[i].b),
                   Tp(data218[i].c), Tp(data218[i].x));
      const Tp f0 = data218[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=20.000000000000000, b=0.50000000000000000, c=8.0000000000000000.
testcase_hyperg<double> data219[] = {
  { 0.54703266209548362, 20.000000000000000, 0.50000000000000000,
          8.0000000000000000, -0.90000000000000002 },
  { 0.56997321774144971, 20.000000000000000, 0.50000000000000000,
          8.0000000000000000, -0.80000000000000004 },
  { 0.59603026159654970, 20.000000000000000, 0.50000000000000000,
          8.0000000000000000, -0.69999999999999996 },
  { 0.62596978851120511, 20.000000000000000, 0.50000000000000000,
          8.0000000000000000, -0.59999999999999998 },
  { 0.66084565876898926, 20.000000000000000, 0.50000000000000000,
          8.0000000000000000, -0.50000000000000000 },
  { 0.70215256667232839, 20.000000000000000, 0.50000000000000000,
          8.0000000000000000, -0.40000000000000002 },
  { 0.75208916592008568, 20.000000000000000, 0.50000000000000000,
          8.0000000000000000, -0.30000000000000004 },
  { 0.81403631111658648, 20.000000000000000, 0.50000000000000000,
          8.0000000000000000, -0.19999999999999996 },
  { 0.89348608489854608, 20.000000000000000, 0.50000000000000000,
          8.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 20.000000000000000, 0.50000000000000000,
          8.0000000000000000, 0.0000000000000000 },
  { 1.1517793185139173, 20.000000000000000, 0.50000000000000000,
          8.0000000000000000, 0.10000000000000009 },
  { 1.3878110313656598, 20.000000000000000, 0.50000000000000000,
          8.0000000000000000, 0.19999999999999996 },
  { 1.8061071794572381, 20.000000000000000, 0.50000000000000000,
          8.0000000000000000, 0.30000000000000004 },
  { 2.7148594517859586, 20.000000000000000, 0.50000000000000000,
          8.0000000000000000, 0.39999999999999991 },
  { 5.4529435709049361, 20.000000000000000, 0.50000000000000000,
          8.0000000000000000, 0.50000000000000000 },
  { 19.487310275377109, 20.000000000000000, 0.50000000000000000,
          8.0000000000000000, 0.60000000000000009 },
  { 191.69079165937470, 20.000000000000000, 0.50000000000000000,
          8.0000000000000000, 0.69999999999999996 },
  { 10218.543981792311, 20.000000000000000, 0.50000000000000000,
          8.0000000000000000, 0.80000000000000004 },
  { 23160836.646583911, 20.000000000000000, 0.50000000000000000,
          8.0000000000000000, 0.89999999999999991 },
};

// Test function for a=20.000000000000000, b=0.50000000000000000, c=8.0000000000000000.
template <typename Tp>
void test219()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data219)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data219[i].a), Tp(data219[i].b),
                   Tp(data219[i].c), Tp(data219[i].x));
      const Tp f0 = data219[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=20.000000000000000, b=0.50000000000000000, c=10.000000000000000.
testcase_hyperg<double> data220[] = {
  { 0.59292067298616002, 20.000000000000000, 0.50000000000000000,
          10.000000000000000, -0.90000000000000002 },
  { 0.61572496720679915, 20.000000000000000, 0.50000000000000000,
          10.000000000000000, -0.80000000000000004 },
  { 0.64135339122875623, 20.000000000000000, 0.50000000000000000,
          10.000000000000000, -0.69999999999999996 },
  { 0.67043457419280483, 20.000000000000000, 0.50000000000000000,
          10.000000000000000, -0.59999999999999998 },
  { 0.70380956268170980, 20.000000000000000, 0.50000000000000000,
          10.000000000000000, -0.50000000000000000 },
  { 0.74263251901495231, 20.000000000000000, 0.50000000000000000,
          10.000000000000000, -0.40000000000000002 },
  { 0.78853555445528278, 20.000000000000000, 0.50000000000000000,
          10.000000000000000, -0.30000000000000004 },
  { 0.84391122775673766, 20.000000000000000, 0.50000000000000000,
          10.000000000000000, -0.19999999999999996 },
  { 0.91242401018807406, 20.000000000000000, 0.50000000000000000,
          10.000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 20.000000000000000, 0.50000000000000000,
          10.000000000000000, 0.0000000000000000 },
  { 1.1169059681274873, 20.000000000000000, 0.50000000000000000,
          10.000000000000000, 0.10000000000000009 },
  { 1.2825928301302667, 20.000000000000000, 0.50000000000000000,
          10.000000000000000, 0.19999999999999996 },
  { 1.5385937789924939, 20.000000000000000, 0.50000000000000000,
          10.000000000000000, 0.30000000000000004 },
  { 1.9895771187893898, 20.000000000000000, 0.50000000000000000,
          10.000000000000000, 0.39999999999999991 },
  { 2.9707335806970168, 20.000000000000000, 0.50000000000000000,
          10.000000000000000, 0.50000000000000000 },
  { 6.0299506157180467, 20.000000000000000, 0.50000000000000000,
          10.000000000000000, 0.60000000000000009 },
  { 24.259090336955577, 20.000000000000000, 0.50000000000000000,
          10.000000000000000, 0.69999999999999996 },
  { 406.27267173257223, 20.000000000000000, 0.50000000000000000,
          10.000000000000000, 0.80000000000000004 },
  { 174330.03997220192, 20.000000000000000, 0.50000000000000000,
          10.000000000000000, 0.89999999999999991 },
};

// Test function for a=20.000000000000000, b=0.50000000000000000, c=10.000000000000000.
template <typename Tp>
void test220()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data220)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data220[i].a), Tp(data220[i].b),
                   Tp(data220[i].c), Tp(data220[i].x));
      const Tp f0 = data220[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=20.000000000000000, b=1.0000000000000000, c=2.0000000000000000.
testcase_hyperg<double> data221[] = {
  { 0.058479236576646373, 20.000000000000000, 1.0000000000000000,
          2.0000000000000000, -0.90000000000000002 },
  { 0.065788544763137669, 20.000000000000000, 1.0000000000000000,
          2.0000000000000000, -0.80000000000000004 },
  { 0.075184824937824662, 20.000000000000000, 1.0000000000000000,
          2.0000000000000000, -0.69999999999999996 },
  { 0.087707688693157260, 20.000000000000000, 1.0000000000000000,
          2.0000000000000000, -0.59999999999999998 },
  { 0.10521567442213345, 20.000000000000000, 1.0000000000000000,
          2.0000000000000000, -0.50000000000000000 },
  { 0.13135877960541525, 20.000000000000000, 1.0000000000000000,
          2.0000000000000000, -0.40000000000000002 },
  { 0.17423854066297104, 20.000000000000000, 1.0000000000000000,
          2.0000000000000000, -0.30000000000000004 },
  { 0.25492082527223525, 20.000000000000000, 1.0000000000000000,
          2.0000000000000000, -0.19999999999999996 },
  { 0.44025895219654843, 20.000000000000000, 1.0000000000000000,
          2.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 20.000000000000000, 1.0000000000000000,
          2.0000000000000000, 0.0000000000000000 },
  { 3.3698615820910360, 20.000000000000000, 1.0000000000000000,
          2.0000000000000000, 0.10000000000000009 },
  { 17.997089220808483, 20.000000000000000, 1.0000000000000000,
          2.0000000000000000, 0.19999999999999996 },
  { 153.73298291118951, 20.000000000000000, 1.0000000000000000,
          2.0000000000000000, 0.30000000000000004 },
  { 2159.1667587825627, 20.000000000000000, 1.0000000000000000,
          2.0000000000000000, 0.39999999999999991 },
  { 55188.105263157879, 20.000000000000000, 1.0000000000000000,
          2.0000000000000000, 0.50000000000000000 },
  { 3191209.3921857267, 20.000000000000000, 1.0000000000000000,
          2.0000000000000000, 0.60000000000000009 },
  { 646910975.29152656, 20.000000000000000, 1.0000000000000000,
          2.0000000000000000, 0.69999999999999996 },
  { 1254834626850.2659, 20.000000000000000, 1.0000000000000000,
          2.0000000000000000, 0.80000000000000004 },
  { 5.8479532163741414e+17, 20.000000000000000, 1.0000000000000000,
          2.0000000000000000, 0.89999999999999991 },
};

// Test function for a=20.000000000000000, b=1.0000000000000000, c=2.0000000000000000.
template <typename Tp>
void test221()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data221)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data221[i].a), Tp(data221[i].b),
                   Tp(data221[i].c), Tp(data221[i].x));
      const Tp f0 = data221[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=20.000000000000000, b=1.0000000000000000, c=4.0000000000000000.
testcase_hyperg<double> data222[] = {
  { 0.15519511120894947, 20.000000000000000, 1.0000000000000000,
          4.0000000000000000, -0.90000000000000002 },
  { 0.17197165701692899, 20.000000000000000, 1.0000000000000000,
          4.0000000000000000, -0.80000000000000004 },
  { 0.19276847315207363, 20.000000000000000, 1.0000000000000000,
          4.0000000000000000, -0.69999999999999996 },
  { 0.21920107206179093, 20.000000000000000, 1.0000000000000000,
          4.0000000000000000, -0.59999999999999998 },
  { 0.25386158960390581, 20.000000000000000, 1.0000000000000000,
          4.0000000000000000, -0.50000000000000000 },
  { 0.30115970686600657, 20.000000000000000, 1.0000000000000000,
          4.0000000000000000, -0.40000000000000002 },
  { 0.36916408142057117, 20.000000000000000, 1.0000000000000000,
          4.0000000000000000, -0.30000000000000004 },
  { 0.47406175901569558, 20.000000000000000, 1.0000000000000000,
          4.0000000000000000, -0.19999999999999996 },
  { 0.65237908266239919, 20.000000000000000, 1.0000000000000000,
          4.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 20.000000000000000, 1.0000000000000000,
          4.0000000000000000, 0.0000000000000000 },
  { 1.8227213362622299, 20.000000000000000, 1.0000000000000000,
          4.0000000000000000, 0.10000000000000009 },
  { 4.3716358339791332, 20.000000000000000, 1.0000000000000000,
          4.0000000000000000, 0.19999999999999996 },
  { 15.670841312959222, 20.000000000000000, 1.0000000000000000,
          4.0000000000000000, 0.30000000000000004 },
  { 94.742651122760179, 20.000000000000000, 1.0000000000000000,
          4.0000000000000000, 0.39999999999999991 },
  { 1081.7275541795671, 20.000000000000000, 1.0000000000000000,
          4.0000000000000000, 0.50000000000000000 },
  { 27809.787731465960, 20.000000000000000, 1.0000000000000000,
          4.0000000000000000, 0.60000000000000009 },
  { 2329811.1715181042, 20.000000000000000, 1.0000000000000000,
          4.0000000000000000, 0.69999999999999996 },
  { 1537787532.6780224, 20.000000000000000, 1.0000000000000000,
          4.0000000000000000, 0.80000000000000004 },
  { 141562653506999.88, 20.000000000000000, 1.0000000000000000,
          4.0000000000000000, 0.89999999999999991 },
};

// Test function for a=20.000000000000000, b=1.0000000000000000, c=4.0000000000000000.
template <typename Tp>
void test222()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data222)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data222[i].a), Tp(data222[i].b),
                   Tp(data222[i].c), Tp(data222[i].x));
      const Tp f0 = data222[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=20.000000000000000, b=1.0000000000000000, c=6.0000000000000000.
testcase_hyperg<double> data223[] = {
  { 0.23253645591196570, 20.000000000000000, 1.0000000000000000,
          6.0000000000000000, -0.90000000000000002 },
  { 0.25484220947068353, 20.000000000000000, 1.0000000000000000,
          6.0000000000000000, -0.80000000000000004 },
  { 0.28181987881113829, 20.000000000000000, 1.0000000000000000,
          6.0000000000000000, -0.69999999999999996 },
  { 0.31508211677735765, 20.000000000000000, 1.0000000000000000,
          6.0000000000000000, -0.59999999999999998 },
  { 0.35706285886959599, 20.000000000000000, 1.0000000000000000,
          6.0000000000000000, -0.50000000000000000 },
  { 0.41160053409238195, 20.000000000000000, 1.0000000000000000,
          6.0000000000000000, -0.40000000000000002 },
  { 0.48508083111181949, 20.000000000000000, 1.0000000000000000,
          6.0000000000000000, -0.30000000000000004 },
  { 0.58885194371375260, 20.000000000000000, 1.0000000000000000,
          6.0000000000000000, -0.19999999999999996 },
  { 0.74482241684585748, 20.000000000000000, 1.0000000000000000,
          6.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 20.000000000000000, 1.0000000000000000,
          6.0000000000000000, 0.0000000000000000 },
  { 1.4700356864367146, 20.000000000000000, 1.0000000000000000,
          6.0000000000000000, 0.10000000000000009 },
  { 2.4955144453055143, 20.000000000000000, 1.0000000000000000,
          6.0000000000000000, 0.19999999999999996 },
  { 5.3506594845833471, 20.000000000000000, 1.0000000000000000,
          6.0000000000000000, 0.30000000000000004 },
  { 16.618413752184221, 20.000000000000000, 1.0000000000000000,
          6.0000000000000000, 0.39999999999999991 },
  { 89.310629514963878, 20.000000000000000, 1.0000000000000000,
          6.0000000000000000, 0.50000000000000000 },
  { 1029.3439900542960, 20.000000000000000, 1.0000000000000000,
          6.0000000000000000, 0.60000000000000009 },
  { 35659.847863372350, 20.000000000000000, 1.0000000000000000,
          6.0000000000000000, 0.69999999999999996 },
  { 8009309.6233230168, 20.000000000000000, 1.0000000000000000,
          6.0000000000000000, 0.80000000000000004 },
  { 145640590027.39731, 20.000000000000000, 1.0000000000000000,
          6.0000000000000000, 0.89999999999999991 },
};

// Test function for a=20.000000000000000, b=1.0000000000000000, c=6.0000000000000000.
template <typename Tp>
void test223()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data223)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data223[i].a), Tp(data223[i].b),
                   Tp(data223[i].c), Tp(data223[i].x));
      const Tp f0 = data223[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=20.000000000000000, b=1.0000000000000000, c=8.0000000000000000.
testcase_hyperg<double> data224[] = {
  { 0.29614148314592498, 20.000000000000000, 1.0000000000000000,
          8.0000000000000000, -0.90000000000000002 },
  { 0.32176277356430810, 20.000000000000000, 1.0000000000000000,
          8.0000000000000000, -0.80000000000000004 },
  { 0.35217870475550522, 20.000000000000000, 1.0000000000000000,
          8.0000000000000000, -0.69999999999999996 },
  { 0.38885270445515091, 20.000000000000000, 1.0000000000000000,
          8.0000000000000000, -0.59999999999999998 },
  { 0.43389978380608424, 20.000000000000000, 1.0000000000000000,
          8.0000000000000000, -0.50000000000000000 },
  { 0.49048612522269436, 20.000000000000000, 1.0000000000000000,
          8.0000000000000000, -0.40000000000000002 },
  { 0.56355539635634611, 20.000000000000000, 1.0000000000000000,
          8.0000000000000000, -0.30000000000000004 },
  { 0.66123153239117682, 20.000000000000000, 1.0000000000000000,
          8.0000000000000000, -0.19999999999999996 },
  { 0.79773363961895427, 20.000000000000000, 1.0000000000000000,
          8.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 20.000000000000000, 1.0000000000000000,
          8.0000000000000000, 0.0000000000000000 },
  { 1.3245132157016595, 20.000000000000000, 1.0000000000000000,
          8.0000000000000000, 0.10000000000000009 },
  { 1.9065148749742076, 20.000000000000000, 1.0000000000000000,
          8.0000000000000000, 0.19999999999999996 },
  { 3.1328798652457452, 20.000000000000000, 1.0000000000000000,
          8.0000000000000000, 0.30000000000000004 },
  { 6.4172532944033476, 20.000000000000000, 1.0000000000000000,
          8.0000000000000000, 0.39999999999999991 },
  { 19.071683734222436, 20.000000000000000, 1.0000000000000000,
          8.0000000000000000, 0.50000000000000000 },
  { 104.41989641582512, 20.000000000000000, 1.0000000000000000,
          8.0000000000000000, 0.60000000000000009 },
  { 1510.5743992324240, 20.000000000000000, 1.0000000000000000,
          8.0000000000000000, 0.69999999999999996 },
  { 115518.14360562043, 20.000000000000000, 1.0000000000000000,
          8.0000000000000000, 0.80000000000000004 },
  { 414930455.29173034, 20.000000000000000, 1.0000000000000000,
          8.0000000000000000, 0.89999999999999991 },
};

// Test function for a=20.000000000000000, b=1.0000000000000000, c=8.0000000000000000.
template <typename Tp>
void test224()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data224)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data224[i].a), Tp(data224[i].b),
                   Tp(data224[i].c), Tp(data224[i].x));
      const Tp f0 = data224[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=20.000000000000000, b=1.0000000000000000, c=10.000000000000000.
testcase_hyperg<double> data225[] = {
  { 0.34954259539177696, 20.000000000000000, 1.0000000000000000,
          10.000000000000000, -0.90000000000000002 },
  { 0.37714038609235123, 20.000000000000000, 1.0000000000000000,
          10.000000000000000, -0.80000000000000004 },
  { 0.40942091659748725, 20.000000000000000, 1.0000000000000000,
          10.000000000000000, -0.69999999999999996 },
  { 0.44767109606846428, 20.000000000000000, 1.0000000000000000,
          10.000000000000000, -0.59999999999999998 },
  { 0.49368984777532254, 20.000000000000000, 1.0000000000000000,
          10.000000000000000, -0.50000000000000000 },
  { 0.55006638216982318, 20.000000000000000, 1.0000000000000000,
          10.000000000000000, -0.40000000000000002 },
  { 0.62065830207408912, 20.000000000000000, 1.0000000000000000,
          10.000000000000000, -0.30000000000000004 },
  { 0.71145554513583786, 20.000000000000000, 1.0000000000000000,
          10.000000000000000, -0.19999999999999996 },
  { 0.83223839666914623, 20.000000000000000, 1.0000000000000000,
          10.000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 20.000000000000000, 1.0000000000000000,
          10.000000000000000, 0.0000000000000000 },
  { 1.2466748028187731, 20.000000000000000, 1.0000000000000000,
          10.000000000000000, 0.10000000000000009 },
  { 1.6386752725021749, 20.000000000000000, 1.0000000000000000,
          10.000000000000000, 0.19999999999999996 },
  { 2.3340068725479681, 20.000000000000000, 1.0000000000000000,
          10.000000000000000, 0.30000000000000004 },
  { 3.7848108613132054, 20.000000000000000, 1.0000000000000000,
          10.000000000000000, 0.39999999999999991 },
  { 7.6754638550304133, 20.000000000000000, 1.0000000000000000,
          10.000000000000000, 0.50000000000000000 },
  { 23.344217312927277, 20.000000000000000, 1.0000000000000000,
          10.000000000000000, 0.60000000000000009 },
  { 149.83491198246921, 20.000000000000000, 1.0000000000000000,
          10.000000000000000, 0.69999999999999996 },
  { 3936.9253501916060, 20.000000000000000, 1.0000000000000000,
          10.000000000000000, 0.80000000000000004 },
  { 2794143.5036480185, 20.000000000000000, 1.0000000000000000,
          10.000000000000000, 0.89999999999999991 },
};

// Test function for a=20.000000000000000, b=1.0000000000000000, c=10.000000000000000.
template <typename Tp>
void test225()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data225)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data225[i].a), Tp(data225[i].b),
                   Tp(data225[i].c), Tp(data225[i].x));
      const Tp f0 = data225[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=20.000000000000000, b=2.0000000000000000, c=2.0000000000000000.
testcase_hyperg<double> data226[] = {
  { 2.6602838683283435e-06, 20.000000000000000, 2.0000000000000000,
          2.0000000000000000, -0.90000000000000002 },
  { 7.8442223930072316e-06, 20.000000000000000, 2.0000000000000000,
          2.0000000000000000, -0.80000000000000004 },
  { 2.4604898194634598e-05, 20.000000000000000, 2.0000000000000000,
          2.0000000000000000, -0.69999999999999996 },
  { 8.2718061255302686e-05, 20.000000000000000, 2.0000000000000000,
          2.0000000000000000, -0.59999999999999998 },
  { 0.00030072865982171723, 20.000000000000000, 2.0000000000000000,
          2.0000000000000000, -0.50000000000000000 },
  { 0.0011951964277455193, 20.000000000000000, 2.0000000000000000,
          2.0000000000000000, -0.40000000000000002 },
  { 0.0052617832469731814, 20.000000000000000, 2.0000000000000000,
          2.0000000000000000, -0.30000000000000004 },
  { 0.026084053304588847, 20.000000000000000, 2.0000000000000000,
          2.0000000000000000, -0.19999999999999996 },
  { 0.14864362802414346, 20.000000000000000, 2.0000000000000000,
          2.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 20.000000000000000, 2.0000000000000000,
          2.0000000000000000, 0.0000000000000000 },
  { 8.2252633399699757, 20.000000000000000, 2.0000000000000000,
          2.0000000000000000, 0.10000000000000009 },
  { 86.736173798840269, 20.000000000000000, 2.0000000000000000,
          2.0000000000000000, 0.19999999999999996 },
  { 1253.2542894196865, 20.000000000000000, 2.0000000000000000,
          2.0000000000000000, 0.30000000000000004 },
  { 27351.112277912434, 20.000000000000000, 2.0000000000000000,
          2.0000000000000000, 0.39999999999999991 },
  { 1048576.0000000000, 20.000000000000000, 2.0000000000000000,
          2.0000000000000000, 0.50000000000000000 },
  { 90949470.177293226, 20.000000000000000, 2.0000000000000000,
          2.0000000000000000, 0.60000000000000009 },
  { 28679719907.924358, 20.000000000000000, 2.0000000000000000,
          2.0000000000000000, 0.69999999999999996 },
  { 95367431640624.906, 20.000000000000000, 2.0000000000000000,
          2.0000000000000000, 0.80000000000000004 },
  { 9.9999999999998657e+19, 20.000000000000000, 2.0000000000000000,
          2.0000000000000000, 0.89999999999999991 },
};

// Test function for a=20.000000000000000, b=2.0000000000000000, c=2.0000000000000000.
template <typename Tp>
void test226()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data226)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data226[i].a), Tp(data226[i].b),
                   Tp(data226[i].c), Tp(data226[i].x));
      const Tp f0 = data226[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(5.0000000000000039e-13));
}

// Test data for a=20.000000000000000, b=2.0000000000000000, c=4.0000000000000000.
testcase_hyperg<double> data227[] = {
  { 0.018828092583720951, 20.000000000000000, 2.0000000000000000,
          4.0000000000000000, -0.90000000000000002 },
  { 0.023381944060455316, 20.000000000000000, 2.0000000000000000,
          4.0000000000000000, -0.80000000000000004 },
  { 0.029789623984280793, 20.000000000000000, 2.0000000000000000,
          4.0000000000000000, -0.69999999999999996 },
  { 0.039191021482500497, 20.000000000000000, 2.0000000000000000,
          4.0000000000000000, -0.59999999999999998 },
  { 0.053727813036721514, 20.000000000000000, 2.0000000000000000,
          4.0000000000000000, -0.50000000000000000 },
  { 0.077762010061669024, 20.000000000000000, 2.0000000000000000,
          4.0000000000000000, -0.40000000000000002 },
  { 0.12110505620123302, 20.000000000000000, 2.0000000000000000,
          4.0000000000000000, -0.30000000000000004 },
  { 0.20870149809080590, 20.000000000000000, 2.0000000000000000,
          4.0000000000000000, -0.19999999999999996 },
  { 0.41429234328785769, 20.000000000000000, 2.0000000000000000,
          4.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 20.000000000000000, 2.0000000000000000,
          4.0000000000000000, 0.0000000000000000 },
  { 3.1308087404153113, 20.000000000000000, 2.0000000000000000,
          4.0000000000000000, 0.10000000000000009 },
  { 13.586180626453050, 20.000000000000000, 2.0000000000000000,
          4.0000000000000000, 0.19999999999999996 },
  { 87.117304082784415, 20.000000000000000, 2.0000000000000000,
          4.0000000000000000, 0.30000000000000004 },
  { 889.26474381242826, 20.000000000000000, 2.0000000000000000,
          4.0000000000000000, 0.39999999999999991 },
  { 16231.913312693494, 20.000000000000000, 2.0000000000000000,
          4.0000000000000000, 0.50000000000000000 },
  { 653537.51168945129, 20.000000000000000, 2.0000000000000000,
          4.0000000000000000, 0.60000000000000009 },
  { 87756230.793848589, 20.000000000000000, 2.0000000000000000,
          4.0000000000000000, 0.69999999999999996 },
  { 101493977171.74945, 20.000000000000000, 2.0000000000000000,
          4.0000000000000000, 0.80000000000000004 },
  { 21375960679556916., 20.000000000000000, 2.0000000000000000,
          4.0000000000000000, 0.89999999999999991 },
};

// Test function for a=20.000000000000000, b=2.0000000000000000, c=4.0000000000000000.
template <typename Tp>
void test227()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data227)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data227[i].a), Tp(data227[i].b),
                   Tp(data227[i].c), Tp(data227[i].x));
      const Tp f0 = data227[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(1.0000000000000008e-12));
}

// Test data for a=20.000000000000000, b=2.0000000000000000, c=6.0000000000000000.
testcase_hyperg<double> data228[] = {
  { 0.049200410661854252, 20.000000000000000, 2.0000000000000000,
          6.0000000000000000, -0.90000000000000002 },
  { 0.059460876757152226, 20.000000000000000, 2.0000000000000000,
          6.0000000000000000, -0.80000000000000004 },
  { 0.073244762686653350, 20.000000000000000, 2.0000000000000000,
          6.0000000000000000, -0.69999999999999996 },
  { 0.092334626017932922, 20.000000000000000, 2.0000000000000000,
          6.0000000000000000, -0.59999999999999998 },
  { 0.11976760350696837, 20.000000000000000, 2.0000000000000000,
          6.0000000000000000, -0.50000000000000000 },
  { 0.16102414609169383, 20.000000000000000, 2.0000000000000000,
          6.0000000000000000, -0.40000000000000002 },
  { 0.22670456785796222, 20.000000000000000, 2.0000000000000000,
          6.0000000000000000, -0.30000000000000004 },
  { 0.33912903252727361, 20.000000000000000, 2.0000000000000000,
          6.0000000000000000, -0.19999999999999996 },
  { 0.55049794600858060, 20.000000000000000, 2.0000000000000000,
          6.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 20.000000000000000, 2.0000000000000000,
          6.0000000000000000, 0.0000000000000000 },
  { 2.1254722872032232, 20.000000000000000, 2.0000000000000000,
          6.0000000000000000, 0.10000000000000009 },
  { 5.6261213886736172, 20.000000000000000, 2.0000000000000000,
          6.0000000000000000, 0.19999999999999996 },
  { 20.137315891130996, 20.000000000000000, 2.0000000000000000,
          6.0000000000000000, 0.30000000000000004 },
  { 108.04381584643853, 20.000000000000000, 2.0000000000000000,
          6.0000000000000000, 0.39999999999999991 },
  { 992.41692466460245, 20.000000000000000, 2.0000000000000000,
          6.0000000000000000, 0.50000000000000000 },
  { 19055.363816004465, 20.000000000000000, 2.0000000000000000,
          6.0000000000000000, 0.60000000000000009 },
  { 1105471.9504312086, 20.000000000000000, 2.0000000000000000,
          6.0000000000000000, 0.69999999999999996 },
  { 448521363.90608919, 20.000000000000000, 2.0000000000000000,
          6.0000000000000000, 0.80000000000000004 },
  { 19078917293639.004, 20.000000000000000, 2.0000000000000000,
          6.0000000000000000, 0.89999999999999991 },
};

// Test function for a=20.000000000000000, b=2.0000000000000000, c=6.0000000000000000.
template <typename Tp>
void test228()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data228)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data228[i].a), Tp(data228[i].b),
                   Tp(data228[i].c), Tp(data228[i].x));
      const Tp f0 = data228[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(5.0000000000000039e-13));
}

// Test data for a=20.000000000000000, b=2.0000000000000000, c=8.0000000000000000.
testcase_hyperg<double> data229[] = {
  { 0.083753547015334745, 20.000000000000000, 2.0000000000000000,
          8.0000000000000000, -0.90000000000000002 },
  { 0.099238444687035701, 20.000000000000000, 2.0000000000000000,
          8.0000000000000000, -0.80000000000000004 },
  { 0.11938294012867758, 20.000000000000000, 2.0000000000000000,
          8.0000000000000000, -0.69999999999999996 },
  { 0.14622683905023326, 20.000000000000000, 2.0000000000000000,
          8.0000000000000000, -0.59999999999999998 },
  { 0.18303556733713025, 20.000000000000000, 2.0000000000000000,
          8.0000000000000000, -0.50000000000000000 },
  { 0.23527764069382409, 20.000000000000000, 2.0000000000000000,
          8.0000000000000000, -0.40000000000000002 },
  { 0.31261681740827085, 20.000000000000000, 2.0000000000000000,
          8.0000000000000000, -0.30000000000000004 },
  { 0.43327581880538862, 20.000000000000000, 2.0000000000000000,
          8.0000000000000000, -0.19999999999999996 },
  { 0.63445840637296658, 20.000000000000000, 2.0000000000000000,
          8.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 20.000000000000000, 2.0000000000000000,
          8.0000000000000000, 0.0000000000000000 },
  { 1.7438842395813297, 20.000000000000000, 2.0000000000000000,
          8.0000000000000000, 0.10000000000000009 },
  { 3.5070840938209269, 20.000000000000000, 2.0000000000000000,
          8.0000000000000000, 0.19999999999999996 },
  { 8.6573372006089713, 20.000000000000000, 2.0000000000000000,
          8.0000000000000000, 0.30000000000000004 },
  { 28.779342118408906, 20.000000000000000, 2.0000000000000000,
          8.0000000000000000, 0.39999999999999991 },
  { 147.50178613955714, 20.000000000000000, 2.0000000000000000,
          8.0000000000000000, 0.50000000000000000 },
  { 1427.1686016136398, 20.000000000000000, 2.0000000000000000,
          8.0000000000000000, 0.60000000000000009 },
  { 36780.643714655642, 20.000000000000000, 2.0000000000000000,
          8.0000000000000000, 0.69999999999999996 },
  { 5313869.6058585485, 20.000000000000000, 2.0000000000000000,
          8.0000000000000000, 0.80000000000000004 },
  { 46057280607.381966, 20.000000000000000, 2.0000000000000000,
          8.0000000000000000, 0.89999999999999991 },
};

// Test function for a=20.000000000000000, b=2.0000000000000000, c=8.0000000000000000.
template <typename Tp>
void test229()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data229)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data229[i].a), Tp(data229[i].b),
                   Tp(data229[i].c), Tp(data229[i].x));
      const Tp f0 = data229[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=20.000000000000000, b=2.0000000000000000, c=10.000000000000000.
testcase_hyperg<double> data230[] = {
  { 0.11920045035073683, 20.000000000000000, 2.0000000000000000,
          10.000000000000000, -0.90000000000000002 },
  { 0.13907946814302774, 20.000000000000000, 2.0000000000000000,
          10.000000000000000, -0.80000000000000004 },
  { 0.16431439792559688, 20.000000000000000, 2.0000000000000000,
          10.000000000000000, -0.69999999999999996 },
  { 0.19698796016987008, 20.000000000000000, 2.0000000000000000,
          10.000000000000000, -0.59999999999999998 },
  { 0.24028510928790570, 20.000000000000000, 2.0000000000000000,
          10.000000000000000, -0.50000000000000000 },
  { 0.29926031296483119, 20.000000000000000, 2.0000000000000000,
          10.000000000000000, -0.40000000000000002 },
  { 0.38229327814229169, 20.000000000000000, 2.0000000000000000,
          10.000000000000000, -0.30000000000000004 },
  { 0.50402047283093110, 20.000000000000000, 2.0000000000000000,
          10.000000000000000, -0.19999999999999996 },
  { 0.69167261179586503, 20.000000000000000, 2.0000000000000000,
          10.000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 20.000000000000000, 2.0000000000000000,
          10.000000000000000, 0.0000000000000000 },
  { 1.5503152253394308, 20.000000000000000, 2.0000000000000000,
          10.000000000000000, 0.10000000000000009 },
  { 2.6469548193635797, 20.000000000000000, 2.0000000000000000,
          10.000000000000000, 0.19999999999999996 },
  { 5.1882631330566813, 20.000000000000000, 2.0000000000000000,
          10.000000000000000, 0.30000000000000004 },
  { 12.476792759124516, 20.000000000000000, 2.0000000000000000,
          10.000000000000000, 0.39999999999999991 },
  { 41.026391565091259, 20.000000000000000, 2.0000000000000000,
          10.000000000000000, 0.50000000000000000 },
  { 220.92584715988204, 20.000000000000000, 2.0000000000000000,
          10.000000000000000, 0.60000000000000009 },
  { 2677.0834450236207, 20.000000000000000, 2.0000000000000000,
          10.000000000000000, 0.69999999999999996 },
  { 141774.31260689779, 20.000000000000000, 2.0000000000000000,
          10.000000000000000, 0.80000000000000004 },
  { 254267148.83196995, 20.000000000000000, 2.0000000000000000,
          10.000000000000000, 0.89999999999999991 },
};

// Test function for a=20.000000000000000, b=2.0000000000000000, c=10.000000000000000.
template <typename Tp>
void test230()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data230)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data230[i].a), Tp(data230[i].b),
                   Tp(data230[i].c), Tp(data230[i].x));
      const Tp f0 = data230[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
}

// Test data for a=20.000000000000000, b=5.0000000000000000, c=2.0000000000000000.
testcase_hyperg<double> data231[] = {
  { -1.8650300348791041e-05, 20.000000000000000, 5.0000000000000000,
          2.0000000000000000, -0.90000000000000002 },
  { -3.6488008415183135e-05, 20.000000000000000, 5.0000000000000000,
          2.0000000000000000, -0.80000000000000004 },
  { -6.4614776410999025e-05, 20.000000000000000, 5.0000000000000000,
          2.0000000000000000, -0.69999999999999996 },
  { -8.4495207102575916e-05, 20.000000000000000, 5.0000000000000000,
          2.0000000000000000, -0.59999999999999998 },
  { 2.2276197023819217e-05, 20.000000000000000, 5.0000000000000000,
          2.0000000000000000, -0.50000000000000000 },
  { 0.00070736115111467578, 20.000000000000000, 5.0000000000000000,
          2.0000000000000000, -0.40000000000000002 },
  { 0.0027829732057272809, 20.000000000000000, 5.0000000000000000,
          2.0000000000000000, -0.30000000000000004 },
  { 0.0013283545664373570, 20.000000000000000, 5.0000000000000000,
          2.0000000000000000, -0.19999999999999996 },
  { -0.041767631015048733, 20.000000000000000, 5.0000000000000000,
          2.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 20.000000000000000, 5.0000000000000000,
          2.0000000000000000, 0.0000000000000000 },
  { 61.311496556100003, 20.000000000000000, 5.0000000000000000,
          2.0000000000000000, 0.10000000000000009 },
  { 2397.4420539085681, 20.000000000000000, 5.0000000000000000,
          2.0000000000000000, 0.19999999999999996 },
  { 103687.60998586559, 20.000000000000000, 5.0000000000000000,
          2.0000000000000000, 0.30000000000000004 },
  { 6247196.6451068865, 20.000000000000000, 5.0000000000000000,
          2.0000000000000000, 0.39999999999999991 },
  { 656408576.00000000, 20.000000000000000, 5.0000000000000000,
          2.0000000000000000, 0.50000000000000000 },
  { 165334768098.54715, 20.000000000000000, 5.0000000000000000,
          2.0000000000000000, 0.60000000000000009 },
  { 175097125520816.81, 20.000000000000000, 5.0000000000000000,
          2.0000000000000000, 0.69999999999999996 },
  { 2.6818275451660257e+18, 20.000000000000000, 5.0000000000000000,
          2.0000000000000000, 0.80000000000000004 },
  { 2.9794599999999321e+25, 20.000000000000000, 5.0000000000000000,
          2.0000000000000000, 0.89999999999999991 },
};

// Test function for a=20.000000000000000, b=5.0000000000000000, c=2.0000000000000000.
template <typename Tp>
void test231()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data231)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data231[i].a), Tp(data231[i].b),
                   Tp(data231[i].c), Tp(data231[i].x));
      const Tp f0 = data231[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(5.0000000000000034e-10));
}

// Test data for a=20.000000000000000, b=5.0000000000000000, c=4.0000000000000000.
testcase_hyperg<double> data232[] = {
  { -3.6403884515183385e-06, 20.000000000000000, 5.0000000000000000,
          4.0000000000000000, -0.90000000000000002 },
  { -9.5873829247725586e-06, 20.000000000000000, 5.0000000000000000,
          4.0000000000000000, -0.80000000000000004 },
  { -2.6052245147354694e-05, 20.000000000000000, 5.0000000000000000,
          4.0000000000000000, -0.69999999999999996 },
  { -7.2378303598294010e-05, 20.000000000000000, 5.0000000000000000,
          4.0000000000000000, -0.59999999999999998 },
  { -0.00020048577321454082, 20.000000000000000, 5.0000000000000000,
          4.0000000000000000, -0.50000000000000000 },
  { -0.00051222704046236022, 20.000000000000000, 5.0000000000000000,
          4.0000000000000000, -0.40000000000000002 },
  { -0.00080950511491911315, 20.000000000000000, 5.0000000000000000,
          4.0000000000000000, -0.30000000000000004 },
  { 0.0043473422174314449, 20.000000000000000, 5.0000000000000000,
          4.0000000000000000, -0.19999999999999996 },
  { 0.081078342558623825, 20.000000000000000, 5.0000000000000000,
          4.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 20.000000000000000, 5.0000000000000000,
          4.0000000000000000, 0.0000000000000000 },
  { 12.794854084397739, 20.000000000000000, 5.0000000000000000,
          4.0000000000000000, 0.10000000000000009 },
  { 195.15639104739046, 20.000000000000000, 5.0000000000000000,
          4.0000000000000000, 0.19999999999999996 },
  { 3938.7991953190131, 20.000000000000000, 5.0000000000000000,
          4.0000000000000000, 0.30000000000000004 },
  { 118521.48653762060, 20.000000000000000, 5.0000000000000000,
          4.0000000000000000, 0.39999999999999991 },
  { 6291455.9999999972, 20.000000000000000, 5.0000000000000000,
          4.0000000000000000, 0.50000000000000000 },
  { 773070496.50699198, 20.000000000000000, 5.0000000000000000,
          4.0000000000000000, 0.60000000000000009 },
  { 363276452167.04102, 20.000000000000000, 5.0000000000000000,
          4.0000000000000000, 0.69999999999999996 },
  { 2002716064453133.0, 20.000000000000000, 5.0000000000000000,
          4.0000000000000000, 0.80000000000000004 },
  { 4.5999999999999109e+21, 20.000000000000000, 5.0000000000000000,
          4.0000000000000000, 0.89999999999999991 },
};

// Test function for a=20.000000000000000, b=5.0000000000000000, c=4.0000000000000000.
template <typename Tp>
void test232()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data232)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data232[i].a), Tp(data232[i].b),
                   Tp(data232[i].c), Tp(data232[i].x));
      const Tp f0 = data232[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000013e-09));
}

// Test data for a=20.000000000000000, b=5.0000000000000000, c=6.0000000000000000.
testcase_hyperg<double> data233[] = {
  { 0.00014313323624069244, 20.000000000000000, 5.0000000000000000,
          6.0000000000000000, -0.90000000000000002 },
  { 0.00025426183473140697, 20.000000000000000, 5.0000000000000000,
          6.0000000000000000, -0.80000000000000004 },
  { 0.00048255612836426809, 20.000000000000000, 5.0000000000000000,
          6.0000000000000000, -0.69999999999999996 },
  { 0.00099096904674788092, 20.000000000000000, 5.0000000000000000,
          6.0000000000000000, -0.59999999999999998 },
  { 0.0022347805521915607, 20.000000000000000, 5.0000000000000000,
          6.0000000000000000, -0.50000000000000000 },
  { 0.0056271390060292376, 20.000000000000000, 5.0000000000000000,
          6.0000000000000000, -0.40000000000000002 },
  { 0.016109059519227226, 20.000000000000000, 5.0000000000000000,
          6.0000000000000000, -0.30000000000000004 },
  { 0.053453465775608999, 20.000000000000000, 5.0000000000000000,
          6.0000000000000000, -0.19999999999999996 },
  { 0.20995202901839258, 20.000000000000000, 5.0000000000000000,
          6.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 20.000000000000000, 5.0000000000000000,
          6.0000000000000000, 0.0000000000000000 },
  { 5.9534372167648799, 20.000000000000000, 5.0000000000000000,
          6.0000000000000000, 0.10000000000000009 },
  { 46.157632071205875, 20.000000000000000, 5.0000000000000000,
          6.0000000000000000, 0.19999999999999996 },
  { 494.32074431164915, 20.000000000000000, 5.0000000000000000,
          6.0000000000000000, 0.30000000000000004 },
  { 7989.5277611775946, 20.000000000000000, 5.0000000000000000,
          6.0000000000000000, 0.39999999999999991 },
  { 224179.55830753347, 20.000000000000000, 5.0000000000000000,
          6.0000000000000000, 0.50000000000000000 },
  { 13848144.485282511, 20.000000000000000, 5.0000000000000000,
          6.0000000000000000, 0.60000000000000009 },
  { 2948587692.8891716, 20.000000000000000, 5.0000000000000000,
          6.0000000000000000, 0.69999999999999996 },
  { 5940513286161.6602, 20.000000000000000, 5.0000000000000000,
          6.0000000000000000, 0.80000000000000004 },
  { 2.8531757655945201e+18, 20.000000000000000, 5.0000000000000000,
          6.0000000000000000, 0.89999999999999991 },
};

// Test function for a=20.000000000000000, b=5.0000000000000000, c=6.0000000000000000.
template <typename Tp>
void test233()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data233)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data233[i].a), Tp(data233[i].b),
                   Tp(data233[i].c), Tp(data233[i].x));
      const Tp f0 = data233[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(1.0000000000000006e-10));
}

// Test data for a=20.000000000000000, b=5.0000000000000000, c=8.0000000000000000.
testcase_hyperg<double> data234[] = {
  { 0.0012492049968742865, 20.000000000000000, 5.0000000000000000,
          8.0000000000000000, -0.90000000000000002 },
  { 0.0019931241968014451, 20.000000000000000, 5.0000000000000000,
          8.0000000000000000, -0.80000000000000004 },
  { 0.0033203386861411057, 20.000000000000000, 5.0000000000000000,
          8.0000000000000000, -0.69999999999999996 },
  { 0.0058191894509855282, 20.000000000000000, 5.0000000000000000,
          8.0000000000000000, -0.59999999999999998 },
  { 0.010830090368313866, 20.000000000000000, 5.0000000000000000,
          8.0000000000000000, -0.50000000000000000 },
  { 0.021653062305193163, 20.000000000000000, 5.0000000000000000,
          8.0000000000000000, -0.40000000000000002 },
  { 0.047180821280919084, 20.000000000000000, 5.0000000000000000,
          8.0000000000000000, -0.30000000000000004 },
  { 0.11405637279736180, 20.000000000000000, 5.0000000000000000,
          8.0000000000000000, -0.19999999999999996 },
  { 0.31275468794720990, 20.000000000000000, 5.0000000000000000,
          8.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 20.000000000000000, 5.0000000000000000,
          8.0000000000000000, 0.0000000000000000 },
  { 3.8598904658643969, 20.000000000000000, 5.0000000000000000,
          8.0000000000000000, 0.10000000000000009 },
  { 18.806301417906667, 20.000000000000000, 5.0000000000000000,
          8.0000000000000000, 0.19999999999999996 },
  { 122.77054465017432, 20.000000000000000, 5.0000000000000000,
          8.0000000000000000, 0.30000000000000004 },
  { 1168.4762146808946, 20.000000000000000, 5.0000000000000000,
          8.0000000000000000, 0.39999999999999991 },
  { 18437.511788521082, 20.000000000000000, 5.0000000000000000,
          8.0000000000000000, 0.50000000000000000 },
  { 597441.79669264762, 20.000000000000000, 5.0000000000000000,
          8.0000000000000000, 0.60000000000000009 },
  { 59390411.369227782, 20.000000000000000, 5.0000000000000000,
          8.0000000000000000, 0.69999999999999996 },
  { 44681668993.361603, 20.000000000000000, 5.0000000000000000,
          8.0000000000000000, 0.80000000000000004 },
  { 4559673269683164.0, 20.000000000000000, 5.0000000000000000,
          8.0000000000000000, 0.89999999999999991 },
};

// Test function for a=20.000000000000000, b=5.0000000000000000, c=8.0000000000000000.
template <typename Tp>
void test234()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data234)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data234[i].a), Tp(data234[i].b),
                   Tp(data234[i].c), Tp(data234[i].x));
      const Tp f0 = data234[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(1.0000000000000006e-11));
}

// Test data for a=20.000000000000000, b=5.0000000000000000, c=10.000000000000000.
testcase_hyperg<double> data235[] = {
  { 0.0038867957051371450, 20.000000000000000, 5.0000000000000000,
          10.000000000000000, -0.90000000000000002 },
  { 0.0058484892597364443, 20.000000000000000, 5.0000000000000000,
          10.000000000000000, -0.80000000000000004 },
  { 0.0090987656053757009, 20.000000000000000, 5.0000000000000000,
          10.000000000000000, -0.69999999999999996 },
  { 0.014714392537270733, 20.000000000000000, 5.0000000000000000,
          10.000000000000000, -0.59999999999999998 },
  { 0.024900404542056769, 20.000000000000000, 5.0000000000000000,
          10.000000000000000, -0.50000000000000000 },
  { 0.044460184663785055, 20.000000000000000, 5.0000000000000000,
          10.000000000000000, -0.40000000000000002 },
  { 0.084638849196356836, 20.000000000000000, 5.0000000000000000,
          10.000000000000000, -0.30000000000000004 },
  { 0.17409058241290998, 20.000000000000000, 5.0000000000000000,
          10.000000000000000, -0.19999999999999996 },
  { 0.39357055823580755, 20.000000000000000, 5.0000000000000000,
          10.000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 20.000000000000000, 5.0000000000000000,
          10.000000000000000, 0.0000000000000000 },
  { 2.9410794636226596, 20.000000000000000, 5.0000000000000000,
          10.000000000000000, 0.10000000000000009 },
  { 10.417226071414344, 20.000000000000000, 5.0000000000000000,
          10.000000000000000, 0.19999999999999996 },
  { 46.930585873140835, 20.000000000000000, 5.0000000000000000,
          10.000000000000000, 0.30000000000000004 },
  { 290.76717121814852, 20.000000000000000, 5.0000000000000000,
          10.000000000000000, 0.39999999999999991 },
  { 2788.1641083374830, 20.000000000000000, 5.0000000000000000,
          10.000000000000000, 0.50000000000000000 },
  { 50228.117718560752, 20.000000000000000, 5.0000000000000000,
          10.000000000000000, 0.60000000000000009 },
  { 2433042.3476752634, 20.000000000000000, 5.0000000000000000,
          10.000000000000000, 0.69999999999999996 },
  { 705345246.77141762, 20.000000000000000, 5.0000000000000000,
          10.000000000000000, 0.80000000000000004 },
  { 15652478868616.762, 20.000000000000000, 5.0000000000000000,
          10.000000000000000, 0.89999999999999991 },
};

// Test function for a=20.000000000000000, b=5.0000000000000000, c=10.000000000000000.
template <typename Tp>
void test235()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data235)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data235[i].a), Tp(data235[i].b),
                   Tp(data235[i].c), Tp(data235[i].x));
      const Tp f0 = data235[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(1.0000000000000008e-12));
}

// Test data for a=20.000000000000000, b=10.000000000000000, c=2.0000000000000000.
testcase_hyperg<double> data236[] = {
  { -2.1776535308707967e-07, 20.000000000000000, 10.000000000000000,
          2.0000000000000000, -0.90000000000000002 },
  { -2.9128833151427998e-06, 20.000000000000000, 10.000000000000000,
          2.0000000000000000, -0.80000000000000004 },
  { -9.4755553429035093e-06, 20.000000000000000, 10.000000000000000,
          2.0000000000000000, -0.69999999999999996 },
  { -1.2844297353813116e-05, 20.000000000000000, 10.000000000000000,
          2.0000000000000000, -0.59999999999999998 },
  { 3.6576965483568809e-05, 20.000000000000000, 10.000000000000000,
          2.0000000000000000, -0.50000000000000000 },
  { 0.00020847453890692649, 20.000000000000000, 10.000000000000000,
          2.0000000000000000, -0.40000000000000002 },
  { -0.00022868510398174632, 20.000000000000000, 10.000000000000000,
          2.0000000000000000, -0.30000000000000004 },
  { -0.0021855513841942732, 20.000000000000000, 10.000000000000000,
          2.0000000000000000, -0.19999999999999996 },
  { 0.014662111759334634, 20.000000000000000, 10.000000000000000,
          2.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 20.000000000000000, 10.000000000000000,
          2.0000000000000000, 0.0000000000000000 },
  { 746.44776348798098, 20.000000000000000, 10.000000000000000,
          2.0000000000000000, 0.10000000000000009 },
  { 136080.48445225612, 20.000000000000000, 10.000000000000000,
          2.0000000000000000, 0.19999999999999996 },
  { 23094279.597826406, 20.000000000000000, 10.000000000000000,
          2.0000000000000000, 0.30000000000000004 },
  { 5315913395.5545301, 20.000000000000000, 10.000000000000000,
          2.0000000000000000, 0.39999999999999991 },
  { 2261935718399.9990, 20.000000000000000, 10.000000000000000,
          2.0000000000000000, 0.50000000000000000 },
  { 2669150854828235.0, 20.000000000000000, 10.000000000000000,
          2.0000000000000000, 0.60000000000000009 },
  { 1.7499363099365994e+19, 20.000000000000000, 10.000000000000000,
          2.0000000000000000, 0.69999999999999996 },
  { 2.8881518494606140e+24, 20.000000000000000, 10.000000000000000,
          2.0000000000000000, 0.80000000000000004 },
  { 1.4165513933661626e+33, 20.000000000000000, 10.000000000000000,
          2.0000000000000000, 0.89999999999999991 },
};

// Test function for a=20.000000000000000, b=10.000000000000000, c=2.0000000000000000.
template <typename Tp>
void test236()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data236)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data236[i].a), Tp(data236[i].b),
                   Tp(data236[i].c), Tp(data236[i].x));
      const Tp f0 = data236[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(1.0000000000000005e-08));
}

// Test data for a=20.000000000000000, b=10.000000000000000, c=4.0000000000000000.
testcase_hyperg<double> data237[] = {
  { 1.7149006973860441e-07, 20.000000000000000, 10.000000000000000,
          4.0000000000000000, -0.90000000000000002 },
  { 3.2399324904088936e-07, 20.000000000000000, 10.000000000000000,
          4.0000000000000000, -0.80000000000000004 },
  { 1.6015317712089860e-07, 20.000000000000000, 10.000000000000000,
          4.0000000000000000, -0.69999999999999996 },
  { -2.0500917204199595e-06, 20.000000000000000, 10.000000000000000,
          4.0000000000000000, -0.59999999999999998 },
  { -1.0175546788599472e-05, 20.000000000000000, 10.000000000000000,
          4.0000000000000000, -0.50000000000000000 },
  { -1.1720101988202453e-05, 20.000000000000000, 10.000000000000000,
          4.0000000000000000, -0.40000000000000002 },
  { 0.00014199637113974185, 20.000000000000000, 10.000000000000000,
          4.0000000000000000, -0.30000000000000004 },
  { 0.00021263363640642297, 20.000000000000000, 10.000000000000000,
          4.0000000000000000, -0.19999999999999996 },
  { -0.0072649256698441751, 20.000000000000000, 10.000000000000000,
          4.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 20.000000000000000, 10.000000000000000,
          4.0000000000000000, 0.0000000000000000 },
  { 90.430293772869618, 20.000000000000000, 10.000000000000000,
          4.0000000000000000, 0.10000000000000009 },
  { 6248.1455940292308, 20.000000000000000, 10.000000000000000,
          4.0000000000000000, 0.19999999999999996 },
  { 501143.39852548984, 20.000000000000000, 10.000000000000000,
          4.0000000000000000, 0.30000000000000004 },
  { 58852027.356439680, 20.000000000000000, 10.000000000000000,
          4.0000000000000000, 0.39999999999999991 },
  { 12942923093.333330, 20.000000000000000, 10.000000000000000,
          4.0000000000000000, 0.50000000000000000 },
  { 7618073993853.6592, 20.000000000000000, 10.000000000000000,
          4.0000000000000000, 0.60000000000000009 },
  { 22630251562549288., 20.000000000000000, 10.000000000000000,
          4.0000000000000000, 0.69999999999999996 },
  { 1.3708372433980356e+21, 20.000000000000000, 10.000000000000000,
          4.0000000000000000, 0.80000000000000004 },
  { 1.4154113619999653e+29, 20.000000000000000, 10.000000000000000,
          4.0000000000000000, 0.89999999999999991 },
};

// Test function for a=20.000000000000000, b=10.000000000000000, c=4.0000000000000000.
template <typename Tp>
void test237()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data237)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data237[i].a), Tp(data237[i].b),
                   Tp(data237[i].c), Tp(data237[i].x));
      const Tp f0 = data237[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(5.0000000000000024e-08));
}

// Test data for a=20.000000000000000, b=10.000000000000000, c=6.0000000000000000.
testcase_hyperg<double> data238[] = {
  { -1.6667473370780257e-08, 20.000000000000000, 10.000000000000000,
          6.0000000000000000, -0.90000000000000002 },
  { 8.6214844067774863e-08, 20.000000000000000, 10.000000000000000,
          6.0000000000000000, -0.80000000000000004 },
  { 5.7778331238835108e-07, 20.000000000000000, 10.000000000000000,
          6.0000000000000000, -0.69999999999999996 },
  { 2.1911400500362969e-06, 20.000000000000000, 10.000000000000000,
          6.0000000000000000, -0.59999999999999998 },
  { 4.7440049217100417e-06, 20.000000000000000, 10.000000000000000,
          6.0000000000000000, -0.50000000000000000 },
  { -1.0564233314924258e-05, 20.000000000000000, 10.000000000000000,
          6.0000000000000000, -0.40000000000000002 },
  { -0.00017990026051856349, 20.000000000000000, 10.000000000000000,
          6.0000000000000000, -0.30000000000000004 },
  { -0.00027618146288744351, 20.000000000000000, 10.000000000000000,
          6.0000000000000000, -0.19999999999999996 },
  { 0.030606019577723392, 20.000000000000000, 10.000000000000000,
          6.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 20.000000000000000, 10.000000000000000,
          6.0000000000000000, 0.0000000000000000 },
  { 27.832854169493341, 20.000000000000000, 10.000000000000000,
          6.0000000000000000, 0.10000000000000009 },
  { 874.00624088575228, 20.000000000000000, 10.000000000000000,
          6.0000000000000000, 0.19999999999999996 },
  { 36049.199340831554, 20.000000000000000, 10.000000000000000,
          6.0000000000000000, 0.30000000000000004 },
  { 2270967.7298624986, 20.000000000000000, 10.000000000000000,
          6.0000000000000000, 0.39999999999999991 },
  { 266979100.44444439, 20.000000000000000, 10.000000000000000,
          6.0000000000000000, 0.50000000000000000 },
  { 80311224337.493027, 20.000000000000000, 10.000000000000000,
          6.0000000000000000, 0.60000000000000009 },
  { 110111693103799.72, 20.000000000000000, 10.000000000000000,
          6.0000000000000000, 0.69999999999999996 },
  { 2.4838871426052618e+18, 20.000000000000000, 10.000000000000000,
          6.0000000000000000, 0.80000000000000004 },
  { 5.4626349999998603e+25, 20.000000000000000, 10.000000000000000,
          6.0000000000000000, 0.89999999999999991 },
};

// Test function for a=20.000000000000000, b=10.000000000000000, c=6.0000000000000000.
template <typename Tp>
void test238()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data238)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data238[i].a), Tp(data238[i].b),
                   Tp(data238[i].c), Tp(data238[i].x));
      const Tp f0 = data238[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(5.0000000000000019e-07));
}

// Test data for a=20.000000000000000, b=10.000000000000000, c=8.0000000000000000.
testcase_hyperg<double> data239[] = {
  { -1.5843795893321480e-07, 20.000000000000000, 10.000000000000000,
          8.0000000000000000, -0.90000000000000002 },
  { -5.4877275994033766e-07, 20.000000000000000, 10.000000000000000,
          8.0000000000000000, -0.80000000000000004 },
  { -1.7169507967745992e-06, 20.000000000000000, 10.000000000000000,
          8.0000000000000000, -0.69999999999999996 },
  { -4.5236439748752000e-06, 20.000000000000000, 10.000000000000000,
          8.0000000000000000, -0.59999999999999998 },
  { -5.5690492560325806e-06, 20.000000000000000, 10.000000000000000,
          8.0000000000000000, -0.50000000000000000 },
  { 5.6914115606934911e-05, 20.000000000000000, 10.000000000000000,
          8.0000000000000000, -0.40000000000000002 },
  { 0.00082507252097519922, 20.000000000000000, 10.000000000000000,
          8.0000000000000000, -0.30000000000000004 },
  { 0.0085739249288229857, 20.000000000000000, 10.000000000000000,
          8.0000000000000000, -0.19999999999999996 },
  { 0.088244357683754757, 20.000000000000000, 10.000000000000000,
          8.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 20.000000000000000, 10.000000000000000,
          8.0000000000000000, 0.0000000000000000 },
  { 13.387208440156897, 20.000000000000000, 10.000000000000000,
          8.0000000000000000, 0.10000000000000009 },
  { 226.77895441155110, 20.000000000000000, 10.000000000000000,
          8.0000000000000000, 0.19999999999999996 },
  { 5281.5716482686785, 20.000000000000000, 10.000000000000000,
          8.0000000000000000, 0.30000000000000004 },
  { 189431.77762850464, 20.000000000000000, 10.000000000000000,
          8.0000000000000000, 0.39999999999999991 },
  { 12408149.333333332, 20.000000000000000, 10.000000000000000,
          8.0000000000000000, 0.50000000000000000 },
  { 1966782292.5839682, 20.000000000000000, 10.000000000000000,
          8.0000000000000000, 0.60000000000000009 },
  { 1274123112205.7495, 20.000000000000000, 10.000000000000000,
          8.0000000000000000, 0.69999999999999996 },
  { 10903676350911508., 20.000000000000000, 10.000000000000000,
          8.0000000000000000, 0.80000000000000004 },
  { 5.1849999999998819e+22, 20.000000000000000, 10.000000000000000,
          8.0000000000000000, 0.89999999999999991 },
};

// Test function for a=20.000000000000000, b=10.000000000000000, c=8.0000000000000000.
template <typename Tp>
void test239()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data239)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data239[i].a), Tp(data239[i].b),
                   Tp(data239[i].c), Tp(data239[i].x));
      const Tp f0 = data239[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000012e-08));
}

// Test data for a=20.000000000000000, b=10.000000000000000, c=10.000000000000000.
testcase_hyperg<double> data240[] = {
  { 2.6602838683283435e-06, 20.000000000000000, 10.000000000000000,
          10.000000000000000, -0.90000000000000002 },
  { 7.8442223930072316e-06, 20.000000000000000, 10.000000000000000,
          10.000000000000000, -0.80000000000000004 },
  { 2.4604898194634598e-05, 20.000000000000000, 10.000000000000000,
          10.000000000000000, -0.69999999999999996 },
  { 8.2718061255302686e-05, 20.000000000000000, 10.000000000000000,
          10.000000000000000, -0.59999999999999998 },
  { 0.00030072865982171723, 20.000000000000000, 10.000000000000000,
          10.000000000000000, -0.50000000000000000 },
  { 0.0011951964277455193, 20.000000000000000, 10.000000000000000,
          10.000000000000000, -0.40000000000000002 },
  { 0.0052617832469731814, 20.000000000000000, 10.000000000000000,
          10.000000000000000, -0.30000000000000004 },
  { 0.026084053304588847, 20.000000000000000, 10.000000000000000,
          10.000000000000000, -0.19999999999999996 },
  { 0.14864362802414346, 20.000000000000000, 10.000000000000000,
          10.000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 20.000000000000000, 10.000000000000000,
          10.000000000000000, 0.0000000000000000 },
  { 8.2252633399699757, 20.000000000000000, 10.000000000000000,
          10.000000000000000, 0.10000000000000009 },
  { 86.736173798840269, 20.000000000000000, 10.000000000000000,
          10.000000000000000, 0.19999999999999996 },
  { 1253.2542894196865, 20.000000000000000, 10.000000000000000,
          10.000000000000000, 0.30000000000000004 },
  { 27351.112277912434, 20.000000000000000, 10.000000000000000,
          10.000000000000000, 0.39999999999999991 },
  { 1048576.0000000000, 20.000000000000000, 10.000000000000000,
          10.000000000000000, 0.50000000000000000 },
  { 90949470.177293226, 20.000000000000000, 10.000000000000000,
          10.000000000000000, 0.60000000000000009 },
  { 28679719907.924358, 20.000000000000000, 10.000000000000000,
          10.000000000000000, 0.69999999999999996 },
  { 95367431640624.906, 20.000000000000000, 10.000000000000000,
          10.000000000000000, 0.80000000000000004 },
  { 9.9999999999998657e+19, 20.000000000000000, 10.000000000000000,
          10.000000000000000, 0.89999999999999991 },
};

// Test function for a=20.000000000000000, b=10.000000000000000, c=10.000000000000000.
template <typename Tp>
void test240()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data240)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data240[i].a), Tp(data240[i].b),
                   Tp(data240[i].c), Tp(data240[i].x));
      const Tp f0 = data240[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(5.0000000000000039e-13));
}

// Test data for a=20.000000000000000, b=20.000000000000000, c=2.0000000000000000.
testcase_hyperg<double> data241[] = {
  { 7.4612991101768883e-09, 20.000000000000000, 20.000000000000000,
          2.0000000000000000, -0.90000000000000002 },
  { 1.1006588946889981e-07, 20.000000000000000, 20.000000000000000,
          2.0000000000000000, -0.80000000000000004 },
  { 2.0126933743389316e-07, 20.000000000000000, 20.000000000000000,
          2.0000000000000000, -0.69999999999999996 },
  { -1.0013775379801016e-06, 20.000000000000000, 20.000000000000000,
          2.0000000000000000, -0.59999999999999998 },
  { -3.0371956856472516e-06, 20.000000000000000, 20.000000000000000,
          2.0000000000000000, -0.50000000000000000 },
  { 2.2012669924527286e-05, 20.000000000000000, 20.000000000000000,
          2.0000000000000000, -0.40000000000000002 },
  { -6.2415598025417670e-05, 20.000000000000000, 20.000000000000000,
          2.0000000000000000, -0.30000000000000004 },
  { 0.00033551320394378602, 20.000000000000000, 20.000000000000000,
          2.0000000000000000, -0.19999999999999996 },
  { -0.0062342152641436353, 20.000000000000000, 20.000000000000000,
          2.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 20.000000000000000, 20.000000000000000,
          2.0000000000000000, 0.0000000000000000 },
  { 34830.688900741610, 20.000000000000000, 20.000000000000000,
          2.0000000000000000, 0.10000000000000009 },
  { 67626221.263030857, 20.000000000000000, 20.000000000000000,
          2.0000000000000000, 0.19999999999999996 },
  { 102764604848.69762, 20.000000000000000, 20.000000000000000,
          2.0000000000000000, 0.30000000000000004 },
  { 220278355222373.38, 20.000000000000000, 20.000000000000000,
          2.0000000000000000, 0.39999999999999991 },
  { 1.0422324699794536e+18, 20.000000000000000, 20.000000000000000,
          2.0000000000000000, 0.50000000000000000 },
  { 1.9128731788368004e+22, 20.000000000000000, 20.000000000000000,
          2.0000000000000000, 0.60000000000000009 },
  { 3.5234592919485287e+27, 20.000000000000000, 20.000000000000000,
          2.0000000000000000, 0.69999999999999996 },
  { 5.0867023209025249e+34, 20.000000000000000, 20.000000000000000,
          2.0000000000000000, 0.80000000000000004 },
  { 3.7461088506658564e+46, 20.000000000000000, 20.000000000000000,
          2.0000000000000000, 0.89999999999999991 },
};

// Test function for a=20.000000000000000, b=20.000000000000000, c=2.0000000000000000.
template <typename Tp>
void test241()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data241)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data241[i].a), Tp(data241[i].b),
                   Tp(data241[i].c), Tp(data241[i].x));
      const Tp f0 = data241[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(1.0000000000000005e-07));
}

// Test data for a=20.000000000000000, b=20.000000000000000, c=4.0000000000000000.
testcase_hyperg<double> data242[] = {
  { -1.5895900796973045e-09, 20.000000000000000, 20.000000000000000,
          4.0000000000000000, -0.90000000000000002 },
  { -2.4403576837293198e-09, 20.000000000000000, 20.000000000000000,
          4.0000000000000000, -0.80000000000000004 },
  { 1.1622915290732117e-08, 20.000000000000000, 20.000000000000000,
          4.0000000000000000, -0.69999999999999996 },
  { 6.3899796307731726e-08, 20.000000000000000, 20.000000000000000,
          4.0000000000000000, -0.59999999999999998 },
  { -1.3503608352807462e-07, 20.000000000000000, 20.000000000000000,
          4.0000000000000000, -0.50000000000000000 },
  { -1.2198533623899163e-06, 20.000000000000000, 20.000000000000000,
          4.0000000000000000, -0.40000000000000002 },
  { 9.9086618119129001e-06, 20.000000000000000, 20.000000000000000,
          4.0000000000000000, -0.30000000000000004 },
  { -7.6797020080162010e-05, 20.000000000000000, 20.000000000000000,
          4.0000000000000000, -0.19999999999999996 },
  { 0.0013196405087170875, 20.000000000000000, 20.000000000000000,
          4.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 20.000000000000000, 20.000000000000000,
          4.0000000000000000, 0.0000000000000000 },
  { 2274.2044768143564, 20.000000000000000, 20.000000000000000,
          4.0000000000000000, 0.10000000000000009 },
  { 1611640.1560475440, 20.000000000000000, 20.000000000000000,
          4.0000000000000000, 0.19999999999999996 },
  { 1147063984.7359734, 20.000000000000000, 20.000000000000000,
          4.0000000000000000, 0.30000000000000004 },
  { 1253162497163.8311, 20.000000000000000, 20.000000000000000,
          4.0000000000000000, 0.39999999999999991 },
  { 3071321673390476.0, 20.000000000000000, 20.000000000000000,
          4.0000000000000000, 0.50000000000000000 },
  { 2.8221123559124324e+19, 20.000000000000000, 20.000000000000000,
          4.0000000000000000, 0.60000000000000009 },
  { 2.3658463807419519e+24, 20.000000000000000, 20.000000000000000,
          4.0000000000000000, 0.69999999999999996 },
  { 1.2596553731345468e+31, 20.000000000000000, 20.000000000000000,
          4.0000000000000000, 0.80000000000000004 },
  { 1.9627175792062075e+42, 20.000000000000000, 20.000000000000000,
          4.0000000000000000, 0.89999999999999991 },
};

// Test function for a=20.000000000000000, b=20.000000000000000, c=4.0000000000000000.
template <typename Tp>
void test242()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data242)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data242[i].a), Tp(data242[i].b),
                   Tp(data242[i].c), Tp(data242[i].x));
      const Tp f0 = data242[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000006e-06));
}

// Test data for a=20.000000000000000, b=20.000000000000000, c=6.0000000000000000.
testcase_hyperg<double> data243[] = {
  { 8.0159808156941562e-11, 20.000000000000000, 20.000000000000000,
          6.0000000000000000, -0.90000000000000002 },
  { -6.4422687845093557e-10, 20.000000000000000, 20.000000000000000,
          6.0000000000000000, -0.80000000000000004 },
  { -3.7526134186308981e-09, 20.000000000000000, 20.000000000000000,
          6.0000000000000000, -0.69999999999999996 },
  { -1.7692034167897114e-09, 20.000000000000000, 20.000000000000000,
          6.0000000000000000, -0.59999999999999998 },
  { 7.9304558772837909e-08, 20.000000000000000, 20.000000000000000,
          6.0000000000000000, -0.50000000000000000 },
  { 5.9348070318594204e-08, 20.000000000000000, 20.000000000000000,
          6.0000000000000000, -0.40000000000000002 },
  { -3.5827694517425210e-06, 20.000000000000000, 20.000000000000000,
          6.0000000000000000, -0.30000000000000004 },
  { 4.4951490418284159e-05, 20.000000000000000, 20.000000000000000,
          6.0000000000000000, -0.19999999999999996 },
  { -0.0013716249406310486, 20.000000000000000, 20.000000000000000,
          6.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 20.000000000000000, 20.000000000000000,
          6.0000000000000000, 0.0000000000000000 },
  { 415.32493304415505, 20.000000000000000, 20.000000000000000,
          6.0000000000000000, 0.10000000000000009 },
  { 121300.42991518594, 20.000000000000000, 20.000000000000000,
          6.0000000000000000, 0.19999999999999996 },
  { 42725673.833462097, 20.000000000000000, 20.000000000000000,
          6.0000000000000000, 0.30000000000000004 },
  { 24588915328.261719, 20.000000000000000, 20.000000000000000,
          6.0000000000000000, 0.39999999999999991 },
  { 31929082412503.652, 20.000000000000000, 20.000000000000000,
          6.0000000000000000, 0.50000000000000000 },
  { 1.4934954443280477e+17, 20.000000000000000, 20.000000000000000,
          6.0000000000000000, 0.60000000000000009 },
  { 5.7726220597696125e+21, 20.000000000000000, 20.000000000000000,
          6.0000000000000000, 0.69999999999999996 },
  { 1.1454387824049374e+28, 20.000000000000000, 20.000000000000000,
          6.0000000000000000, 0.80000000000000004 },
  { 3.8088637321581534e+38, 20.000000000000000, 20.000000000000000,
          6.0000000000000000, 0.89999999999999991 },
};

// Test function for a=20.000000000000000, b=20.000000000000000, c=6.0000000000000000.
template <typename Tp>
void test243()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data243)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data243[i].a), Tp(data243[i].b),
                   Tp(data243[i].c), Tp(data243[i].x));
      const Tp f0 = data243[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000005e-05));
}

// Test data for a=20.000000000000000, b=20.000000000000000, c=8.0000000000000000.
testcase_hyperg<double> data244[] = {
  { 1.0699072529874453e-10, 20.000000000000000, 20.000000000000000,
          8.0000000000000000, -0.90000000000000002 },
  { 5.4297753417228627e-10, 20.000000000000000, 20.000000000000000,
          8.0000000000000000, -0.80000000000000004 },
  { 9.7625471266824426e-10, 20.000000000000000, 20.000000000000000,
          8.0000000000000000, -0.69999999999999996 },
  { -6.7257762867770348e-09, 20.000000000000000, 20.000000000000000,
          8.0000000000000000, -0.59999999999999998 },
  { -5.4634571496175302e-08, 20.000000000000000, 20.000000000000000,
          8.0000000000000000, -0.50000000000000000 },
  { 1.4595644213893387e-07, 20.000000000000000, 20.000000000000000,
          8.0000000000000000, -0.40000000000000002 },
  { 3.3515966497049909e-06, 20.000000000000000, 20.000000000000000,
          8.0000000000000000, -0.30000000000000004 },
  { -6.5848086985914887e-05, 20.000000000000000, 20.000000000000000,
          8.0000000000000000, -0.19999999999999996 },
  { 0.0034800171306214813, 20.000000000000000, 20.000000000000000,
          8.0000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 20.000000000000000, 20.000000000000000,
          8.0000000000000000, 0.0000000000000000 },
  { 130.93865856750304, 20.000000000000000, 20.000000000000000,
          8.0000000000000000, 0.10000000000000009 },
  { 17850.203502975532, 20.000000000000000, 20.000000000000000,
          8.0000000000000000, 0.19999999999999996 },
  { 3307058.5655149994, 20.000000000000000, 20.000000000000000,
          8.0000000000000000, 0.30000000000000004 },
  { 1041065396.2302787, 20.000000000000000, 20.000000000000000,
          8.0000000000000000, 0.39999999999999991 },
  { 735221357488.41736, 20.000000000000000, 20.000000000000000,
          8.0000000000000000, 0.50000000000000000 },
  { 1785176805049585.2, 20.000000000000000, 20.000000000000000,
          8.0000000000000000, 0.60000000000000009 },
  { 3.2302829930269192e+19, 20.000000000000000, 20.000000000000000,
          8.0000000000000000, 0.69999999999999996 },
  { 2.4184909805178299e+25, 20.000000000000000, 20.000000000000000,
          8.0000000000000000, 0.80000000000000004 },
  { 1.7340021007794567e+35, 20.000000000000000, 20.000000000000000,
          8.0000000000000000, 0.89999999999999991 },
};

// Test function for a=20.000000000000000, b=20.000000000000000, c=8.0000000000000000.
template <typename Tp>
void test244()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data244)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data244[i].a), Tp(data244[i].b),
                   Tp(data244[i].c), Tp(data244[i].x));
      const Tp f0 = data244[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000005e-05));
}

// Test data for a=20.000000000000000, b=20.000000000000000, c=10.000000000000000.
testcase_hyperg<double> data245[] = {
  { -1.7945363894141429e-10, 20.000000000000000, 20.000000000000000,
          10.000000000000000, -0.90000000000000002 },
  { -4.4440666675421800e-10, 20.000000000000000, 20.000000000000000,
          10.000000000000000, -0.80000000000000004 },
  { 6.6171630913717945e-10, 20.000000000000000, 20.000000000000000,
          10.000000000000000, -0.69999999999999996 },
  { 1.5453889642199650e-08, 20.000000000000000, 20.000000000000000,
          10.000000000000000, -0.59999999999999998 },
  { 7.5754083860094422e-08, 20.000000000000000, 20.000000000000000,
          10.000000000000000, -0.50000000000000000 },
  { -4.1113628639873710e-07, 20.000000000000000, 20.000000000000000,
          10.000000000000000, -0.40000000000000002 },
  { -9.5300704265214247e-06, 20.000000000000000, 20.000000000000000,
          10.000000000000000, -0.30000000000000004 },
  { 0.00016081533175785109, 20.000000000000000, 20.000000000000000,
          10.000000000000000, -0.19999999999999996 },
  { 0.017684650940379486, 20.000000000000000, 20.000000000000000,
          10.000000000000000, -0.099999999999999978 },
  { 1.0000000000000000, 20.000000000000000, 20.000000000000000,
          10.000000000000000, 0.0000000000000000 },
  { 57.562247312454403, 20.000000000000000, 20.000000000000000,
          10.000000000000000, 0.10000000000000009 },
  { 4124.4159820362511, 20.000000000000000, 20.000000000000000,
          10.000000000000000, 0.19999999999999996 },
  { 428774.21436196787, 20.000000000000000, 20.000000000000000,
          10.000000000000000, 0.30000000000000004 },
  { 76996819.900892526, 20.000000000000000, 20.000000000000000,
          10.000000000000000, 0.39999999999999991 },
  { 30473174828.943691, 20.000000000000000, 20.000000000000000,
          10.000000000000000, 0.50000000000000000 },
  { 39291970835753.094, 20.000000000000000, 20.000000000000000,
          10.000000000000000, 0.60000000000000009 },
  { 3.3890331048069018e+17, 20.000000000000000, 20.000000000000000,
          10.000000000000000, 0.69999999999999996 },
  { 9.7157373454594049e+22, 20.000000000000000, 20.000000000000000,
          10.000000000000000, 0.80000000000000004 },
  { 1.5205808288860858e+32, 20.000000000000000, 20.000000000000000,
          10.000000000000000, 0.89999999999999991 },
};

// Test function for a=20.000000000000000, b=20.000000000000000, c=10.000000000000000.
template <typename Tp>
void test245()
{
  const Tp eps = std::numeric_limits<Tp>::epsilon();
  Tp max_abs_diff = -Tp(1);
  Tp max_abs_frac = -Tp(1);
  unsigned int num_datum = sizeof(data245)
                         / sizeof(testcase_hyperg<double>);
  for (unsigned int i = 0; i < num_datum; ++i)
    {
      const Tp f = std::tr1::hyperg(Tp(data245[i].a), Tp(data245[i].b),
                   Tp(data245[i].c), Tp(data245[i].x));
      const Tp f0 = data245[i].f0;
      const Tp diff = f - f0;
      if (std::abs(diff) > max_abs_diff)
        max_abs_diff = std::abs(diff);
      if (std::abs(f0) > Tp(10) * eps && std::abs(f) > Tp(10) * eps)
        {
          const Tp frac = diff / f0;
          if (std::abs(frac) > max_abs_frac)
            max_abs_frac = std::abs(frac);
        }
    }
  VERIFY(max_abs_frac < Tp(2.5000000000000005e-05));
}

int main(int, char**)
{
  test001<double>();
  test002<double>();
  test003<double>();
  test004<double>();
  test005<double>();
  test006<double>();
  test007<double>();
  test008<double>();
  test009<double>();
  test010<double>();
  test011<double>();
  test012<double>();
  test013<double>();
  test014<double>();
  test015<double>();
  test016<double>();
  test017<double>();
  test018<double>();
  test019<double>();
  test020<double>();
  test021<double>();
  test022<double>();
  test023<double>();
  test024<double>();
  test025<double>();
  test026<double>();
  test027<double>();
  test028<double>();
  test029<double>();
  test030<double>();
  test031<double>();
  test032<double>();
  test033<double>();
  test034<double>();
  test035<double>();
  test036<double>();
  test037<double>();
  test038<double>();
  test039<double>();
  test040<double>();
  test041<double>();
  test042<double>();
  test043<double>();
  test044<double>();
  test045<double>();
  test046<double>();
  test047<double>();
  test048<double>();
  test049<double>();
  test050<double>();
  test051<double>();
  test052<double>();
  test053<double>();
  test054<double>();
  test055<double>();
  test056<double>();
  test057<double>();
  test058<double>();
  test059<double>();
  test060<double>();
  test061<double>();
  test062<double>();
  test063<double>();
  test064<double>();
  test065<double>();
  test066<double>();
  test067<double>();
  test068<double>();
  test069<double>();
  test070<double>();
  test071<double>();
  test072<double>();
  test073<double>();
  test074<double>();
  test075<double>();
  test076<double>();
  test077<double>();
  test078<double>();
  test079<double>();
  test080<double>();
  test081<double>();
  test082<double>();
  test083<double>();
  test084<double>();
  test085<double>();
  test086<double>();
  test087<double>();
  test088<double>();
  test089<double>();
  test090<double>();
  test091<double>();
  test092<double>();
  test093<double>();
  test094<double>();
  test095<double>();
  test096<double>();
  test097<double>();
  test098<double>();
  test099<double>();
  test100<double>();
  test101<double>();
  test102<double>();
  test103<double>();
  test104<double>();
  test105<double>();
  test106<double>();
  test107<double>();
  test108<double>();
  test109<double>();
  test110<double>();
  test111<double>();
  test112<double>();
  test113<double>();
  test114<double>();
  test115<double>();
  test116<double>();
  test117<double>();
  test118<double>();
  test119<double>();
  test120<double>();
  test121<double>();
  test122<double>();
  test123<double>();
  test124<double>();
  test125<double>();
  test126<double>();
  test127<double>();
  test128<double>();
  test129<double>();
  test130<double>();
  test131<double>();
  test132<double>();
  test133<double>();
  test134<double>();
  test135<double>();
  test136<double>();
  test137<double>();
  test138<double>();
  test139<double>();
  test140<double>();
  test141<double>();
  test142<double>();
  test143<double>();
  test144<double>();
  test145<double>();
  test146<double>();
  test147<double>();
  test148<double>();
  test149<double>();
  test150<double>();
  test151<double>();
  test152<double>();
  test153<double>();
  test154<double>();
  test155<double>();
  test156<double>();
  test157<double>();
  test158<double>();
  test159<double>();
  test160<double>();
  test161<double>();
  test162<double>();
  test163<double>();
  test164<double>();
  test165<double>();
  test166<double>();
  test167<double>();
  test168<double>();
  test169<double>();
  test170<double>();
  test171<double>();
  test172<double>();
  test173<double>();
  test174<double>();
  test175<double>();
  test176<double>();
  test177<double>();
  test178<double>();
  test179<double>();
  test180<double>();
  test181<double>();
  test182<double>();
  test183<double>();
  test184<double>();
  test185<double>();
  test186<double>();
  test187<double>();
  test188<double>();
  test189<double>();
  test190<double>();
  test191<double>();
  test192<double>();
  test193<double>();
  test194<double>();
  test195<double>();
  test196<double>();
  test197<double>();
  test198<double>();
  test199<double>();
  test200<double>();
  test201<double>();
  test202<double>();
  test203<double>();
  test204<double>();
  test205<double>();
  test206<double>();
  test207<double>();
  test208<double>();
  test209<double>();
  test210<double>();
  test211<double>();
  test212<double>();
  test213<double>();
  test214<double>();
  test215<double>();
  test216<double>();
  test217<double>();
  test218<double>();
  test219<double>();
  test220<double>();
  test221<double>();
  test222<double>();
  test223<double>();
  test224<double>();
  test225<double>();
  test226<double>();
  test227<double>();
  test228<double>();
  test229<double>();
  test230<double>();
  test231<double>();
  test232<double>();
  test233<double>();
  test234<double>();
  test235<double>();
  test236<double>();
  test237<double>();
  test238<double>();
  test239<double>();
  test240<double>();
  test241<double>();
  test242<double>();
  test243<double>();
  test244<double>();
  test245<double>();
  return 0;
}
