// Copyright 2009 The Go Authors. All rights reserved.
// Use of this source code is governed by a BSD-style
// license that can be found in the LICENSE file.

package tls

import (
	"bytes"
	"crypto/rsa"
	"crypto/x509"
	"encoding/hex"
	"encoding/pem"
	"flag"
	"fmt"
	"io"
	"log"
	"math/big"
	"net"
	"os"
	"strconv"
	"strings"
	"sync"
	"testing"
	"time"
)

type zeroSource struct{}

func (zeroSource) Read(b []byte) (n int, err error) {
	for i := range b {
		b[i] = 0
	}

	return len(b), nil
}

var testConfig *Config

func init() {
	testConfig = new(Config)
	testConfig.Time = func() time.Time { return time.Unix(0, 0) }
	testConfig.Rand = zeroSource{}
	testConfig.Certificates = make([]Certificate, 2)
	testConfig.Certificates[0].Certificate = [][]byte{testCertificate}
	testConfig.Certificates[0].PrivateKey = testPrivateKey
	testConfig.Certificates[1].Certificate = [][]byte{testSNICertificate}
	testConfig.Certificates[1].PrivateKey = testPrivateKey
	testConfig.BuildNameToCertificate()
	testConfig.CipherSuites = []uint16{TLS_RSA_WITH_RC4_128_SHA}
	testConfig.InsecureSkipVerify = true
}

func testClientHelloFailure(t *testing.T, m handshakeMessage, expected error) {
	// Create in-memory network connection,
	// send message to server.  Should return
	// expected error.
	c, s := net.Pipe()
	go func() {
		cli := Client(c, testConfig)
		if ch, ok := m.(*clientHelloMsg); ok {
			cli.vers = ch.vers
		}
		cli.writeRecord(recordTypeHandshake, m.marshal())
		c.Close()
	}()
	err := Server(s, testConfig).Handshake()
	s.Close()
	if e, ok := err.(*net.OpError); !ok || e.Err != expected {
		t.Errorf("Got error: %s; expected: %s", err, expected)
	}
}

func TestSimpleError(t *testing.T) {
	testClientHelloFailure(t, &serverHelloDoneMsg{}, alertUnexpectedMessage)
}

var badProtocolVersions = []uint16{0x0000, 0x0005, 0x0100, 0x0105, 0x0200, 0x0205}

func TestRejectBadProtocolVersion(t *testing.T) {
	for _, v := range badProtocolVersions {
		testClientHelloFailure(t, &clientHelloMsg{vers: v}, alertProtocolVersion)
	}
}

func TestNoSuiteOverlap(t *testing.T) {
	clientHello := &clientHelloMsg{
		vers:               0x0301,
		cipherSuites:       []uint16{0xff00},
		compressionMethods: []uint8{0},
	}
	testClientHelloFailure(t, clientHello, alertHandshakeFailure)
}

func TestNoCompressionOverlap(t *testing.T) {
	clientHello := &clientHelloMsg{
		vers:               0x0301,
		cipherSuites:       []uint16{TLS_RSA_WITH_RC4_128_SHA},
		compressionMethods: []uint8{0xff},
	}
	testClientHelloFailure(t, clientHello, alertHandshakeFailure)
}

func TestAlertForwarding(t *testing.T) {
	c, s := net.Pipe()
	go func() {
		Client(c, testConfig).sendAlert(alertUnknownCA)
		c.Close()
	}()

	err := Server(s, testConfig).Handshake()
	s.Close()
	if e, ok := err.(*net.OpError); !ok || e.Err != error(alertUnknownCA) {
		t.Errorf("Got error: %s; expected: %s", err, alertUnknownCA)
	}
}

func TestClose(t *testing.T) {
	c, s := net.Pipe()
	go c.Close()

	err := Server(s, testConfig).Handshake()
	s.Close()
	if err != io.EOF {
		t.Errorf("Got error: %s; expected: %s", err, io.EOF)
	}
}

func testServerScript(t *testing.T, name string, serverScript [][]byte, config *Config, peers []*x509.Certificate) {
	c, s := net.Pipe()
	srv := Server(s, config)
	pchan := make(chan []*x509.Certificate, 1)
	go func() {
		srv.Write([]byte("hello, world\n"))
		srv.Close()
		s.Close()
		st := srv.ConnectionState()
		pchan <- st.PeerCertificates
	}()

	for i, b := range serverScript {
		if i%2 == 0 {
			c.Write(b)
			continue
		}
		bb := make([]byte, len(b))
		n, err := io.ReadFull(c, bb)
		if err != nil {
			t.Fatalf("%s #%d: %s\nRead %d, wanted %d, got %x, wanted %x\n", name, i, err, n, len(bb), bb[:n], b)
		}
		if !bytes.Equal(b, bb) {
			t.Fatalf("%s #%d: mismatch on read: got:%x want:%x", name, i, bb, b)
		}
	}
	c.Close()

	if peers != nil {
		gotpeers := <-pchan
		if len(peers) == len(gotpeers) {
			for i := range peers {
				if !peers[i].Equal(gotpeers[i]) {
					t.Fatalf("%s: mismatch on peer cert %d", name, i)
				}
			}
		} else {
			t.Fatalf("%s: mismatch on peer list length: %d (wanted) != %d (got)", name, len(peers), len(gotpeers))
		}
	}
}

func TestHandshakeServerRC4(t *testing.T) {
	testServerScript(t, "RC4", rc4ServerScript, testConfig, nil)
}

func TestHandshakeServer3DES(t *testing.T) {
	des3Config := new(Config)
	*des3Config = *testConfig
	des3Config.CipherSuites = []uint16{TLS_RSA_WITH_3DES_EDE_CBC_SHA}
	testServerScript(t, "3DES", des3ServerScript, des3Config, nil)
}

func TestHandshakeServerAES(t *testing.T) {
	aesConfig := new(Config)
	*aesConfig = *testConfig
	aesConfig.CipherSuites = []uint16{TLS_RSA_WITH_AES_128_CBC_SHA}
	testServerScript(t, "AES", aesServerScript, aesConfig, nil)
}

func TestHandshakeServerSSLv3(t *testing.T) {
	testServerScript(t, "SSLv3", sslv3ServerScript, testConfig, nil)
}

// TestHandshakeServerSNI involves a client sending an SNI extension of
// "snitest.com", which happens to match the CN of testSNICertificate. The test
// verifies that the server correctly selects that certificate.
func TestHandshakeServerSNI(t *testing.T) {
	testServerScript(t, "SNI", selectCertificateBySNIScript, testConfig, nil)
}

func TestResumption(t *testing.T) {
	testServerScript(t, "IssueTicket", issueSessionTicketTest, testConfig, nil)
	testServerScript(t, "Resume", serverResumeTest, testConfig, nil)
}

type clientauthTest struct {
	name       string
	clientauth ClientAuthType
	peers      []*x509.Certificate
	script     [][]byte
}

func TestClientAuth(t *testing.T) {
	for _, cat := range clientauthTests {
		t.Log("running", cat.name)
		cfg := new(Config)
		*cfg = *testConfig
		cfg.ClientAuth = cat.clientauth
		testServerScript(t, cat.name, cat.script, cfg, cat.peers)
	}
}

// recordingConn is a net.Conn that records the traffic that passes through it.
// WriteTo can be used to produce Go code that contains the recorded traffic.
type recordingConn struct {
	net.Conn
	lock             sync.Mutex
	flows            [][]byte
	currentlyReading bool
}

func (r *recordingConn) Read(b []byte) (n int, err error) {
	if n, err = r.Conn.Read(b); n == 0 {
		return
	}
	b = b[:n]

	r.lock.Lock()
	defer r.lock.Unlock()

	if l := len(r.flows); l == 0 || !r.currentlyReading {
		buf := make([]byte, len(b))
		copy(buf, b)
		r.flows = append(r.flows, buf)
	} else {
		r.flows[l-1] = append(r.flows[l-1], b[:n]...)
	}
	r.currentlyReading = true
	return
}

func (r *recordingConn) Write(b []byte) (n int, err error) {
	if n, err = r.Conn.Write(b); n == 0 {
		return
	}
	b = b[:n]

	r.lock.Lock()
	defer r.lock.Unlock()

	if l := len(r.flows); l == 0 || r.currentlyReading {
		buf := make([]byte, len(b))
		copy(buf, b)
		r.flows = append(r.flows, buf)
	} else {
		r.flows[l-1] = append(r.flows[l-1], b[:n]...)
	}
	r.currentlyReading = false
	return
}

// WriteTo writes Go source code to w that contains the recorded traffic.
func (r *recordingConn) WriteTo(w io.Writer) {
	fmt.Fprintf(w, "var changeMe = [][]byte {\n")
	for _, buf := range r.flows {
		fmt.Fprintf(w, "\t{")
		for i, b := range buf {
			if i%8 == 0 {
				fmt.Fprintf(w, "\n\t\t")
			}
			fmt.Fprintf(w, "0x%02x, ", b)
		}
		fmt.Fprintf(w, "\n\t},\n")
	}
	fmt.Fprintf(w, "}\n")
}

var serve = flag.Bool("serve", false, "run a TLS server on :10443")
var testCipherSuites = flag.String("ciphersuites",
	"0x"+strconv.FormatInt(int64(TLS_RSA_WITH_RC4_128_SHA), 16),
	"cipher suites to accept in serving mode")
var testClientAuth = flag.Int("clientauth", 0, "value for tls.Config.ClientAuth")

func GetTestConfig() *Config {
	var config = *testConfig
	suites := strings.Split(*testCipherSuites, ",")
	config.CipherSuites = make([]uint16, len(suites))
	for i := range suites {
		suite, err := strconv.ParseUint(suites[i], 0, 64)
		if err != nil {
			panic(err)
		}
		config.CipherSuites[i] = uint16(suite)
	}

	config.ClientAuth = ClientAuthType(*testClientAuth)
	return &config
}

func TestRunServer(t *testing.T) {
	if !*serve {
		return
	}

	config := GetTestConfig()

	const addr = ":10443"
	l, err := net.Listen("tcp", addr)
	if err != nil {
		t.Fatal(err)
	}
	log.Printf("Now listening for connections on %s", addr)

	for {
		tcpConn, err := l.Accept()
		if err != nil {
			log.Printf("error accepting connection: %s", err)
			break
		}

		record := &recordingConn{
			Conn: tcpConn,
		}

		conn := Server(record, config)
		if err := conn.Handshake(); err != nil {
			log.Printf("error from TLS handshake: %s", err)
			break
		}

		_, err = conn.Write([]byte("hello, world\n"))
		if err != nil {
			log.Printf("error from Write: %s", err)
			continue
		}

		conn.Close()

		record.WriteTo(os.Stdout)
	}
}

func bigFromString(s string) *big.Int {
	ret := new(big.Int)
	ret.SetString(s, 10)
	return ret
}

func fromHex(s string) []byte {
	b, _ := hex.DecodeString(s)
	return b
}

var testCertificate = fromHex("308202b030820219a00302010202090085b0bba48a7fb8ca300d06092a864886f70d01010505003045310b3009060355040613024155311330110603550408130a536f6d652d53746174653121301f060355040a1318496e7465726e6574205769646769747320507479204c7464301e170d3130303432343039303933385a170d3131303432343039303933385a3045310b3009060355040613024155311330110603550408130a536f6d652d53746174653121301f060355040a1318496e7465726e6574205769646769747320507479204c746430819f300d06092a864886f70d010101050003818d0030818902818100bb79d6f517b5e5bf4610d0dc69bee62b07435ad0032d8a7a4385b71452e7a5654c2c78b8238cb5b482e5de1f953b7e62a52ca533d6fe125c7a56fcf506bffa587b263fb5cd04d3d0c921964ac7f4549f5abfef427100fe1899077f7e887d7df10439c4a22edb51c97ce3c04c3b326601cfafb11db8719a1ddbdb896baeda2d790203010001a381a73081a4301d0603551d0e04160414b1ade2855acfcb28db69ce2369ded3268e18883930750603551d23046e306c8014b1ade2855acfcb28db69ce2369ded3268e188839a149a4473045310b3009060355040613024155311330110603550408130a536f6d652d53746174653121301f060355040a1318496e7465726e6574205769646769747320507479204c746482090085b0bba48a7fb8ca300c0603551d13040530030101ff300d06092a864886f70d010105050003818100086c4524c76bb159ab0c52ccf2b014d7879d7a6475b55a9566e4c52b8eae12661feb4f38b36e60d392fdf74108b52513b1187a24fb301dbaed98b917ece7d73159db95d31d78ea50565cd5825a2d5a5f33c4b6d8c97590968c0f5298b5cd981f89205ff2a01ca31b9694dda9fd57e970e8266d71999b266e3850296c90a7bdd9")

var testSNICertificate = fromHex("308201f23082015da003020102020100300b06092a864886f70d01010530283110300e060355040a130741636d6520436f311430120603550403130b736e69746573742e636f6d301e170d3132303431313137343033355a170d3133303431313137343533355a30283110300e060355040a130741636d6520436f311430120603550403130b736e69746573742e636f6d30819d300b06092a864886f70d01010103818d0030818902818100bb79d6f517b5e5bf4610d0dc69bee62b07435ad0032d8a7a4385b71452e7a5654c2c78b8238cb5b482e5de1f953b7e62a52ca533d6fe125c7a56fcf506bffa587b263fb5cd04d3d0c921964ac7f4549f5abfef427100fe1899077f7e887d7df10439c4a22edb51c97ce3c04c3b326601cfafb11db8719a1ddbdb896baeda2d790203010001a3323030300e0603551d0f0101ff0404030200a0300d0603551d0e0406040401020304300f0603551d2304083006800401020304300b06092a864886f70d0101050381810089c6455f1c1f5ef8eb1ab174ee2439059f5c4259bb1a8d86cdb1d056f56a717da40e95ab90f59e8deaf627c157995094db0802266eb34fc6842dea8a4b68d9c1389103ab84fb9e1f85d9b5d23ff2312c8670fbb540148245a4ebafe264d90c8a4cf4f85b0fac12ac2fc4a3154bad52462868af96c62c6525d652b6e31845bdcc")

var testPrivateKey = &rsa.PrivateKey{
	PublicKey: rsa.PublicKey{
		N: bigFromString("131650079503776001033793877885499001334664249354723305978524647182322416328664556247316495448366990052837680518067798333412266673813370895702118944398081598789828837447552603077848001020611640547221687072142537202428102790818451901395596882588063427854225330436740647715202971973145151161964464812406232198521"),
		E: 65537,
	},
	D: bigFromString("29354450337804273969007277378287027274721892607543397931919078829901848876371746653677097639302788129485893852488285045793268732234230875671682624082413996177431586734171663258657462237320300610850244186316880055243099640544518318093544057213190320837094958164973959123058337475052510833916491060913053867729"),
	Primes: []*big.Int{
		bigFromString("11969277782311800166562047708379380720136961987713178380670422671426759650127150688426177829077494755200794297055316163155755835813760102405344560929062149"),
		bigFromString("10998999429884441391899182616418192492905073053684657075974935218461686523870125521822756579792315215543092255516093840728890783887287417039645833477273829"),
	},
}

func loadPEMCert(in string) *x509.Certificate {
	block, _ := pem.Decode([]byte(in))
	if block.Type == "CERTIFICATE" && len(block.Headers) == 0 {
		cert, err := x509.ParseCertificate(block.Bytes)
		if err == nil {
			return cert
		}
		panic("error parsing cert")
	}
	panic("error parsing PEM")
}

// Script of interaction with gnutls implementation.
// The values for this test are obtained by building and running in server mode:
//   % go test -test.run "TestRunServer" -serve
// The recorded bytes are written to stdout.
var rc4ServerScript = [][]byte{
	{
		0x16, 0x03, 0x01, 0x00, 0x54, 0x01, 0x00, 0x00,
		0x50, 0x03, 0x01, 0x50, 0x77, 0x3d, 0xbd, 0x32,
		0x13, 0xd7, 0xea, 0x33, 0x65, 0x02, 0xb8, 0x70,
		0xb7, 0x84, 0xc4, 0x05, 0x1f, 0xa4, 0x24, 0xc4,
		0x91, 0x69, 0x04, 0x32, 0x96, 0xfe, 0x5b, 0x49,
		0x71, 0x60, 0x9a, 0x00, 0x00, 0x28, 0x00, 0x39,
		0x00, 0x38, 0x00, 0x35, 0x00, 0x16, 0x00, 0x13,
		0x00, 0x0a, 0x00, 0x33, 0x00, 0x32, 0x00, 0x2f,
		0x00, 0x05, 0x00, 0x04, 0x00, 0x15, 0x00, 0x12,
		0x00, 0x09, 0x00, 0x14, 0x00, 0x11, 0x00, 0x08,
		0x00, 0x06, 0x00, 0x03, 0x00, 0xff, 0x02, 0x01,
		0x00,
	},
	{
		0x16, 0x03, 0x01, 0x00, 0x2a, 0x02, 0x00, 0x00,
		0x26, 0x03, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x16,
		0x03, 0x01, 0x02, 0xbe, 0x0b, 0x00, 0x02, 0xba,
		0x00, 0x02, 0xb7, 0x00, 0x02, 0xb4, 0x30, 0x82,
		0x02, 0xb0, 0x30, 0x82, 0x02, 0x19, 0xa0, 0x03,
		0x02, 0x01, 0x02, 0x02, 0x09, 0x00, 0x85, 0xb0,
		0xbb, 0xa4, 0x8a, 0x7f, 0xb8, 0xca, 0x30, 0x0d,
		0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d,
		0x01, 0x01, 0x05, 0x05, 0x00, 0x30, 0x45, 0x31,
		0x0b, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06,
		0x13, 0x02, 0x41, 0x55, 0x31, 0x13, 0x30, 0x11,
		0x06, 0x03, 0x55, 0x04, 0x08, 0x13, 0x0a, 0x53,
		0x6f, 0x6d, 0x65, 0x2d, 0x53, 0x74, 0x61, 0x74,
		0x65, 0x31, 0x21, 0x30, 0x1f, 0x06, 0x03, 0x55,
		0x04, 0x0a, 0x13, 0x18, 0x49, 0x6e, 0x74, 0x65,
		0x72, 0x6e, 0x65, 0x74, 0x20, 0x57, 0x69, 0x64,
		0x67, 0x69, 0x74, 0x73, 0x20, 0x50, 0x74, 0x79,
		0x20, 0x4c, 0x74, 0x64, 0x30, 0x1e, 0x17, 0x0d,
		0x31, 0x30, 0x30, 0x34, 0x32, 0x34, 0x30, 0x39,
		0x30, 0x39, 0x33, 0x38, 0x5a, 0x17, 0x0d, 0x31,
		0x31, 0x30, 0x34, 0x32, 0x34, 0x30, 0x39, 0x30,
		0x39, 0x33, 0x38, 0x5a, 0x30, 0x45, 0x31, 0x0b,
		0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13,
		0x02, 0x41, 0x55, 0x31, 0x13, 0x30, 0x11, 0x06,
		0x03, 0x55, 0x04, 0x08, 0x13, 0x0a, 0x53, 0x6f,
		0x6d, 0x65, 0x2d, 0x53, 0x74, 0x61, 0x74, 0x65,
		0x31, 0x21, 0x30, 0x1f, 0x06, 0x03, 0x55, 0x04,
		0x0a, 0x13, 0x18, 0x49, 0x6e, 0x74, 0x65, 0x72,
		0x6e, 0x65, 0x74, 0x20, 0x57, 0x69, 0x64, 0x67,
		0x69, 0x74, 0x73, 0x20, 0x50, 0x74, 0x79, 0x20,
		0x4c, 0x74, 0x64, 0x30, 0x81, 0x9f, 0x30, 0x0d,
		0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d,
		0x01, 0x01, 0x01, 0x05, 0x00, 0x03, 0x81, 0x8d,
		0x00, 0x30, 0x81, 0x89, 0x02, 0x81, 0x81, 0x00,
		0xbb, 0x79, 0xd6, 0xf5, 0x17, 0xb5, 0xe5, 0xbf,
		0x46, 0x10, 0xd0, 0xdc, 0x69, 0xbe, 0xe6, 0x2b,
		0x07, 0x43, 0x5a, 0xd0, 0x03, 0x2d, 0x8a, 0x7a,
		0x43, 0x85, 0xb7, 0x14, 0x52, 0xe7, 0xa5, 0x65,
		0x4c, 0x2c, 0x78, 0xb8, 0x23, 0x8c, 0xb5, 0xb4,
		0x82, 0xe5, 0xde, 0x1f, 0x95, 0x3b, 0x7e, 0x62,
		0xa5, 0x2c, 0xa5, 0x33, 0xd6, 0xfe, 0x12, 0x5c,
		0x7a, 0x56, 0xfc, 0xf5, 0x06, 0xbf, 0xfa, 0x58,
		0x7b, 0x26, 0x3f, 0xb5, 0xcd, 0x04, 0xd3, 0xd0,
		0xc9, 0x21, 0x96, 0x4a, 0xc7, 0xf4, 0x54, 0x9f,
		0x5a, 0xbf, 0xef, 0x42, 0x71, 0x00, 0xfe, 0x18,
		0x99, 0x07, 0x7f, 0x7e, 0x88, 0x7d, 0x7d, 0xf1,
		0x04, 0x39, 0xc4, 0xa2, 0x2e, 0xdb, 0x51, 0xc9,
		0x7c, 0xe3, 0xc0, 0x4c, 0x3b, 0x32, 0x66, 0x01,
		0xcf, 0xaf, 0xb1, 0x1d, 0xb8, 0x71, 0x9a, 0x1d,
		0xdb, 0xdb, 0x89, 0x6b, 0xae, 0xda, 0x2d, 0x79,
		0x02, 0x03, 0x01, 0x00, 0x01, 0xa3, 0x81, 0xa7,
		0x30, 0x81, 0xa4, 0x30, 0x1d, 0x06, 0x03, 0x55,
		0x1d, 0x0e, 0x04, 0x16, 0x04, 0x14, 0xb1, 0xad,
		0xe2, 0x85, 0x5a, 0xcf, 0xcb, 0x28, 0xdb, 0x69,
		0xce, 0x23, 0x69, 0xde, 0xd3, 0x26, 0x8e, 0x18,
		0x88, 0x39, 0x30, 0x75, 0x06, 0x03, 0x55, 0x1d,
		0x23, 0x04, 0x6e, 0x30, 0x6c, 0x80, 0x14, 0xb1,
		0xad, 0xe2, 0x85, 0x5a, 0xcf, 0xcb, 0x28, 0xdb,
		0x69, 0xce, 0x23, 0x69, 0xde, 0xd3, 0x26, 0x8e,
		0x18, 0x88, 0x39, 0xa1, 0x49, 0xa4, 0x47, 0x30,
		0x45, 0x31, 0x0b, 0x30, 0x09, 0x06, 0x03, 0x55,
		0x04, 0x06, 0x13, 0x02, 0x41, 0x55, 0x31, 0x13,
		0x30, 0x11, 0x06, 0x03, 0x55, 0x04, 0x08, 0x13,
		0x0a, 0x53, 0x6f, 0x6d, 0x65, 0x2d, 0x53, 0x74,
		0x61, 0x74, 0x65, 0x31, 0x21, 0x30, 0x1f, 0x06,
		0x03, 0x55, 0x04, 0x0a, 0x13, 0x18, 0x49, 0x6e,
		0x74, 0x65, 0x72, 0x6e, 0x65, 0x74, 0x20, 0x57,
		0x69, 0x64, 0x67, 0x69, 0x74, 0x73, 0x20, 0x50,
		0x74, 0x79, 0x20, 0x4c, 0x74, 0x64, 0x82, 0x09,
		0x00, 0x85, 0xb0, 0xbb, 0xa4, 0x8a, 0x7f, 0xb8,
		0xca, 0x30, 0x0c, 0x06, 0x03, 0x55, 0x1d, 0x13,
		0x04, 0x05, 0x30, 0x03, 0x01, 0x01, 0xff, 0x30,
		0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7,
		0x0d, 0x01, 0x01, 0x05, 0x05, 0x00, 0x03, 0x81,
		0x81, 0x00, 0x08, 0x6c, 0x45, 0x24, 0xc7, 0x6b,
		0xb1, 0x59, 0xab, 0x0c, 0x52, 0xcc, 0xf2, 0xb0,
		0x14, 0xd7, 0x87, 0x9d, 0x7a, 0x64, 0x75, 0xb5,
		0x5a, 0x95, 0x66, 0xe4, 0xc5, 0x2b, 0x8e, 0xae,
		0x12, 0x66, 0x1f, 0xeb, 0x4f, 0x38, 0xb3, 0x6e,
		0x60, 0xd3, 0x92, 0xfd, 0xf7, 0x41, 0x08, 0xb5,
		0x25, 0x13, 0xb1, 0x18, 0x7a, 0x24, 0xfb, 0x30,
		0x1d, 0xba, 0xed, 0x98, 0xb9, 0x17, 0xec, 0xe7,
		0xd7, 0x31, 0x59, 0xdb, 0x95, 0xd3, 0x1d, 0x78,
		0xea, 0x50, 0x56, 0x5c, 0xd5, 0x82, 0x5a, 0x2d,
		0x5a, 0x5f, 0x33, 0xc4, 0xb6, 0xd8, 0xc9, 0x75,
		0x90, 0x96, 0x8c, 0x0f, 0x52, 0x98, 0xb5, 0xcd,
		0x98, 0x1f, 0x89, 0x20, 0x5f, 0xf2, 0xa0, 0x1c,
		0xa3, 0x1b, 0x96, 0x94, 0xdd, 0xa9, 0xfd, 0x57,
		0xe9, 0x70, 0xe8, 0x26, 0x6d, 0x71, 0x99, 0x9b,
		0x26, 0x6e, 0x38, 0x50, 0x29, 0x6c, 0x90, 0xa7,
		0xbd, 0xd9, 0x16, 0x03, 0x01, 0x00, 0x04, 0x0e,
		0x00, 0x00, 0x00,
	},
	{
		0x16, 0x03, 0x01, 0x00, 0x86, 0x10, 0x00, 0x00,
		0x82, 0x00, 0x80, 0x2d, 0x09, 0x7c, 0x7f, 0xfc,
		0x84, 0xce, 0xb3, 0x30, 0x9b, 0xf9, 0xb7, 0xc8,
		0xc3, 0xff, 0xee, 0x6f, 0x20, 0x8a, 0xf4, 0xfb,
		0x86, 0x55, 0x1f, 0x6a, 0xb4, 0x81, 0x50, 0x3a,
		0x46, 0x1b, 0xd3, 0xca, 0x4b, 0x11, 0xff, 0xef,
		0x02, 0xbc, 0x18, 0xb8, 0x4a, 0x7d, 0x43, 0x23,
		0x96, 0x92, 0x27, 0x7c, 0xca, 0xcf, 0xe6, 0x91,
		0xe8, 0x14, 0x97, 0x68, 0xb4, 0xe5, 0xc0, 0xc9,
		0x23, 0xdd, 0x54, 0x07, 0xa6, 0x2e, 0x8c, 0x98,
		0xfc, 0xc6, 0x8c, 0x04, 0x6b, 0x1b, 0x5f, 0xd5,
		0x3d, 0x8b, 0x6c, 0x55, 0x4f, 0x7a, 0xe6, 0x6c,
		0x74, 0x2c, 0x1e, 0x34, 0xdb, 0xfb, 0x00, 0xb1,
		0x4e, 0x10, 0x21, 0x16, 0xe0, 0x3e, 0xc5, 0x64,
		0x84, 0x28, 0x2b, 0x2b, 0x29, 0x47, 0x51, 0x34,
		0x76, 0x15, 0x20, 0x71, 0x0b, 0x30, 0xa1, 0x85,
		0xd5, 0x15, 0x18, 0x14, 0x64, 0x4b, 0x40, 0x7c,
		0x4f, 0xb3, 0x7b, 0x14, 0x03, 0x01, 0x00, 0x01,
		0x01, 0x16, 0x03, 0x01, 0x00, 0x24, 0xab, 0xee,
		0xf5, 0x97, 0x5f, 0xc6, 0x78, 0xf3, 0xc6, 0x83,
		0x5b, 0x55, 0x4f, 0xcb, 0x45, 0x3f, 0xfa, 0xf7,
		0x05, 0x02, 0xc2, 0x63, 0x87, 0x18, 0xb5, 0x9a,
		0x62, 0xe2, 0x3f, 0x88, 0x5a, 0x60, 0x61, 0x72,
		0xfa, 0x9c,
	},
	{
		0x14, 0x03, 0x01, 0x00, 0x01, 0x01, 0x16, 0x03,
		0x01, 0x00, 0x24, 0x72, 0xa4, 0xe4, 0xaa, 0xd2,
		0xc4, 0x39, 0x7e, 0x2a, 0xc1, 0x6f, 0x34, 0x42,
		0x28, 0xcb, 0x9d, 0x7a, 0x09, 0xca, 0x96, 0xad,
		0x0e, 0x11, 0x51, 0x8a, 0x06, 0xb0, 0xe9, 0xca,
		0xeb, 0xce, 0xe2, 0xd5, 0x2e, 0xc1, 0x8d, 0x17,
		0x03, 0x01, 0x00, 0x21, 0x2e, 0x61, 0x86, 0x17,
		0xdb, 0xa6, 0x30, 0xe2, 0x62, 0x06, 0x2a, 0x8b,
		0x75, 0x2c, 0x2d, 0xcf, 0xf5, 0x01, 0x11, 0x52,
		0x81, 0x38, 0xcf, 0xd5, 0xf7, 0xdc, 0x52, 0x31,
		0x1f, 0x97, 0x43, 0xc2, 0x71, 0x15, 0x03, 0x01,
		0x00, 0x16, 0xe0, 0x21, 0xfe, 0x36, 0x2e, 0x68,
		0x2c, 0xf1, 0xbe, 0x04, 0xec, 0xd4, 0xc6, 0xdd,
		0xac, 0x6f, 0x4c, 0x85, 0x32, 0x3f, 0x87, 0x1b,
	},
}

var des3ServerScript = [][]byte{
	{
		0x16, 0x03, 0x00, 0x00, 0xc5, 0x01, 0x00, 0x00,
		0xc1, 0x03, 0x03, 0x50, 0xae, 0x5d, 0x38, 0xec,
		0xaa, 0x2f, 0x41, 0xf9, 0xd2, 0x7b, 0xa1, 0xfd,
		0x0f, 0xff, 0x4e, 0x54, 0x0e, 0x15, 0x57, 0xaf,
		0x2c, 0x91, 0xb5, 0x35, 0x5b, 0x2e, 0xb0, 0xec,
		0x20, 0xe5, 0xd2, 0x00, 0x00, 0x50, 0xc0, 0x09,
		0xc0, 0x23, 0xc0, 0x2b, 0xc0, 0x0a, 0xc0, 0x24,
		0xc0, 0x2c, 0xc0, 0x08, 0xc0, 0x13, 0xc0, 0x27,
		0xc0, 0x2f, 0xc0, 0x14, 0xc0, 0x30, 0xc0, 0x12,
		0x00, 0x33, 0x00, 0x67, 0x00, 0x45, 0x00, 0x9e,
		0x00, 0x39, 0x00, 0x6b, 0x00, 0x88, 0x00, 0x16,
		0x00, 0x32, 0x00, 0x40, 0x00, 0x44, 0x00, 0xa2,
		0x00, 0x38, 0x00, 0x6a, 0x00, 0x87, 0x00, 0x13,
		0x00, 0x66, 0x00, 0x2f, 0x00, 0x3c, 0x00, 0x41,
		0x00, 0x9c, 0x00, 0x35, 0x00, 0x3d, 0x00, 0x84,
		0x00, 0x0a, 0x00, 0x05, 0x00, 0x04, 0x01, 0x00,
		0x00, 0x48, 0x00, 0x05, 0x00, 0x05, 0x01, 0x00,
		0x00, 0x00, 0x00, 0xff, 0x01, 0x00, 0x01, 0x00,
		0x00, 0x23, 0x00, 0x00, 0x00, 0x0a, 0x00, 0x0c,
		0x00, 0x0a, 0x00, 0x13, 0x00, 0x15, 0x00, 0x17,
		0x00, 0x18, 0x00, 0x19, 0x00, 0x0b, 0x00, 0x02,
		0x01, 0x00, 0x00, 0x0d, 0x00, 0x1c, 0x00, 0x1a,
		0x04, 0x01, 0x04, 0x02, 0x04, 0x03, 0x05, 0x01,
		0x05, 0x03, 0x06, 0x01, 0x06, 0x03, 0x03, 0x01,
		0x03, 0x02, 0x03, 0x03, 0x02, 0x01, 0x02, 0x02,
		0x02, 0x03,
	},
	{
		0x16, 0x03, 0x01, 0x00, 0x30, 0x02, 0x00, 0x00,
		0x2c, 0x03, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x0a, 0x00, 0x00,
		0x04, 0x00, 0x23, 0x00, 0x00, 0x16, 0x03, 0x01,
		0x02, 0xbe, 0x0b, 0x00, 0x02, 0xba, 0x00, 0x02,
		0xb7, 0x00, 0x02, 0xb4, 0x30, 0x82, 0x02, 0xb0,
		0x30, 0x82, 0x02, 0x19, 0xa0, 0x03, 0x02, 0x01,
		0x02, 0x02, 0x09, 0x00, 0x85, 0xb0, 0xbb, 0xa4,
		0x8a, 0x7f, 0xb8, 0xca, 0x30, 0x0d, 0x06, 0x09,
		0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01,
		0x05, 0x05, 0x00, 0x30, 0x45, 0x31, 0x0b, 0x30,
		0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02,
		0x41, 0x55, 0x31, 0x13, 0x30, 0x11, 0x06, 0x03,
		0x55, 0x04, 0x08, 0x13, 0x0a, 0x53, 0x6f, 0x6d,
		0x65, 0x2d, 0x53, 0x74, 0x61, 0x74, 0x65, 0x31,
		0x21, 0x30, 0x1f, 0x06, 0x03, 0x55, 0x04, 0x0a,
		0x13, 0x18, 0x49, 0x6e, 0x74, 0x65, 0x72, 0x6e,
		0x65, 0x74, 0x20, 0x57, 0x69, 0x64, 0x67, 0x69,
		0x74, 0x73, 0x20, 0x50, 0x74, 0x79, 0x20, 0x4c,
		0x74, 0x64, 0x30, 0x1e, 0x17, 0x0d, 0x31, 0x30,
		0x30, 0x34, 0x32, 0x34, 0x30, 0x39, 0x30, 0x39,
		0x33, 0x38, 0x5a, 0x17, 0x0d, 0x31, 0x31, 0x30,
		0x34, 0x32, 0x34, 0x30, 0x39, 0x30, 0x39, 0x33,
		0x38, 0x5a, 0x30, 0x45, 0x31, 0x0b, 0x30, 0x09,
		0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x41,
		0x55, 0x31, 0x13, 0x30, 0x11, 0x06, 0x03, 0x55,
		0x04, 0x08, 0x13, 0x0a, 0x53, 0x6f, 0x6d, 0x65,
		0x2d, 0x53, 0x74, 0x61, 0x74, 0x65, 0x31, 0x21,
		0x30, 0x1f, 0x06, 0x03, 0x55, 0x04, 0x0a, 0x13,
		0x18, 0x49, 0x6e, 0x74, 0x65, 0x72, 0x6e, 0x65,
		0x74, 0x20, 0x57, 0x69, 0x64, 0x67, 0x69, 0x74,
		0x73, 0x20, 0x50, 0x74, 0x79, 0x20, 0x4c, 0x74,
		0x64, 0x30, 0x81, 0x9f, 0x30, 0x0d, 0x06, 0x09,
		0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01,
		0x01, 0x05, 0x00, 0x03, 0x81, 0x8d, 0x00, 0x30,
		0x81, 0x89, 0x02, 0x81, 0x81, 0x00, 0xbb, 0x79,
		0xd6, 0xf5, 0x17, 0xb5, 0xe5, 0xbf, 0x46, 0x10,
		0xd0, 0xdc, 0x69, 0xbe, 0xe6, 0x2b, 0x07, 0x43,
		0x5a, 0xd0, 0x03, 0x2d, 0x8a, 0x7a, 0x43, 0x85,
		0xb7, 0x14, 0x52, 0xe7, 0xa5, 0x65, 0x4c, 0x2c,
		0x78, 0xb8, 0x23, 0x8c, 0xb5, 0xb4, 0x82, 0xe5,
		0xde, 0x1f, 0x95, 0x3b, 0x7e, 0x62, 0xa5, 0x2c,
		0xa5, 0x33, 0xd6, 0xfe, 0x12, 0x5c, 0x7a, 0x56,
		0xfc, 0xf5, 0x06, 0xbf, 0xfa, 0x58, 0x7b, 0x26,
		0x3f, 0xb5, 0xcd, 0x04, 0xd3, 0xd0, 0xc9, 0x21,
		0x96, 0x4a, 0xc7, 0xf4, 0x54, 0x9f, 0x5a, 0xbf,
		0xef, 0x42, 0x71, 0x00, 0xfe, 0x18, 0x99, 0x07,
		0x7f, 0x7e, 0x88, 0x7d, 0x7d, 0xf1, 0x04, 0x39,
		0xc4, 0xa2, 0x2e, 0xdb, 0x51, 0xc9, 0x7c, 0xe3,
		0xc0, 0x4c, 0x3b, 0x32, 0x66, 0x01, 0xcf, 0xaf,
		0xb1, 0x1d, 0xb8, 0x71, 0x9a, 0x1d, 0xdb, 0xdb,
		0x89, 0x6b, 0xae, 0xda, 0x2d, 0x79, 0x02, 0x03,
		0x01, 0x00, 0x01, 0xa3, 0x81, 0xa7, 0x30, 0x81,
		0xa4, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e,
		0x04, 0x16, 0x04, 0x14, 0xb1, 0xad, 0xe2, 0x85,
		0x5a, 0xcf, 0xcb, 0x28, 0xdb, 0x69, 0xce, 0x23,
		0x69, 0xde, 0xd3, 0x26, 0x8e, 0x18, 0x88, 0x39,
		0x30, 0x75, 0x06, 0x03, 0x55, 0x1d, 0x23, 0x04,
		0x6e, 0x30, 0x6c, 0x80, 0x14, 0xb1, 0xad, 0xe2,
		0x85, 0x5a, 0xcf, 0xcb, 0x28, 0xdb, 0x69, 0xce,
		0x23, 0x69, 0xde, 0xd3, 0x26, 0x8e, 0x18, 0x88,
		0x39, 0xa1, 0x49, 0xa4, 0x47, 0x30, 0x45, 0x31,
		0x0b, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06,
		0x13, 0x02, 0x41, 0x55, 0x31, 0x13, 0x30, 0x11,
		0x06, 0x03, 0x55, 0x04, 0x08, 0x13, 0x0a, 0x53,
		0x6f, 0x6d, 0x65, 0x2d, 0x53, 0x74, 0x61, 0x74,
		0x65, 0x31, 0x21, 0x30, 0x1f, 0x06, 0x03, 0x55,
		0x04, 0x0a, 0x13, 0x18, 0x49, 0x6e, 0x74, 0x65,
		0x72, 0x6e, 0x65, 0x74, 0x20, 0x57, 0x69, 0x64,
		0x67, 0x69, 0x74, 0x73, 0x20, 0x50, 0x74, 0x79,
		0x20, 0x4c, 0x74, 0x64, 0x82, 0x09, 0x00, 0x85,
		0xb0, 0xbb, 0xa4, 0x8a, 0x7f, 0xb8, 0xca, 0x30,
		0x0c, 0x06, 0x03, 0x55, 0x1d, 0x13, 0x04, 0x05,
		0x30, 0x03, 0x01, 0x01, 0xff, 0x30, 0x0d, 0x06,
		0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01,
		0x01, 0x05, 0x05, 0x00, 0x03, 0x81, 0x81, 0x00,
		0x08, 0x6c, 0x45, 0x24, 0xc7, 0x6b, 0xb1, 0x59,
		0xab, 0x0c, 0x52, 0xcc, 0xf2, 0xb0, 0x14, 0xd7,
		0x87, 0x9d, 0x7a, 0x64, 0x75, 0xb5, 0x5a, 0x95,
		0x66, 0xe4, 0xc5, 0x2b, 0x8e, 0xae, 0x12, 0x66,
		0x1f, 0xeb, 0x4f, 0x38, 0xb3, 0x6e, 0x60, 0xd3,
		0x92, 0xfd, 0xf7, 0x41, 0x08, 0xb5, 0x25, 0x13,
		0xb1, 0x18, 0x7a, 0x24, 0xfb, 0x30, 0x1d, 0xba,
		0xed, 0x98, 0xb9, 0x17, 0xec, 0xe7, 0xd7, 0x31,
		0x59, 0xdb, 0x95, 0xd3, 0x1d, 0x78, 0xea, 0x50,
		0x56, 0x5c, 0xd5, 0x82, 0x5a, 0x2d, 0x5a, 0x5f,
		0x33, 0xc4, 0xb6, 0xd8, 0xc9, 0x75, 0x90, 0x96,
		0x8c, 0x0f, 0x52, 0x98, 0xb5, 0xcd, 0x98, 0x1f,
		0x89, 0x20, 0x5f, 0xf2, 0xa0, 0x1c, 0xa3, 0x1b,
		0x96, 0x94, 0xdd, 0xa9, 0xfd, 0x57, 0xe9, 0x70,
		0xe8, 0x26, 0x6d, 0x71, 0x99, 0x9b, 0x26, 0x6e,
		0x38, 0x50, 0x29, 0x6c, 0x90, 0xa7, 0xbd, 0xd9,
		0x16, 0x03, 0x01, 0x00, 0x04, 0x0e, 0x00, 0x00,
		0x00,
	},
	{
		0x16, 0x03, 0x01, 0x00, 0x86, 0x10, 0x00, 0x00,
		0x82, 0x00, 0x80, 0x51, 0x04, 0xf1, 0x7a, 0xbf,
		0xe8, 0xa5, 0x86, 0x09, 0xa7, 0xf3, 0xcc, 0x93,
		0x00, 0x10, 0x5b, 0xb8, 0xc1, 0x51, 0x0d, 0x5b,
		0xcd, 0xed, 0x26, 0x01, 0x69, 0x73, 0xf4, 0x05,
		0x8a, 0x6a, 0xc3, 0xb1, 0x9e, 0x84, 0x4e, 0x39,
		0xcf, 0x5e, 0x55, 0xa9, 0x70, 0x19, 0x96, 0x91,
		0xcd, 0x2c, 0x78, 0x3c, 0xa2, 0x6d, 0xb0, 0x49,
		0x86, 0xf6, 0xd1, 0x3a, 0xde, 0x00, 0x4b, 0xa6,
		0x25, 0xbf, 0x85, 0x39, 0xce, 0xb1, 0xcf, 0xbc,
		0x16, 0xc7, 0x66, 0xac, 0xf8, 0xd2, 0x3b, 0xd1,
		0xcc, 0x16, 0xac, 0x63, 0x3c, 0xbe, 0xd9, 0xb6,
		0x6a, 0xe4, 0x13, 0x8a, 0xf4, 0x56, 0x2f, 0x92,
		0x54, 0xd8, 0xf0, 0x84, 0x01, 0x32, 0x1a, 0xa9,
		0x2d, 0xaf, 0x82, 0x0e, 0x00, 0xfa, 0x07, 0x88,
		0xd9, 0x87, 0xe7, 0xdc, 0x9e, 0xe9, 0x72, 0x49,
		0xb8, 0xfa, 0x8c, 0x7b, 0x07, 0x0b, 0x03, 0x7c,
		0x10, 0x8c, 0x8a, 0x14, 0x03, 0x01, 0x00, 0x01,
		0x01, 0x16, 0x03, 0x01, 0x00, 0xa8, 0x61, 0xa4,
		0xf4, 0x5f, 0x8a, 0x1f, 0x5c, 0x92, 0x3f, 0x8c,
		0xdb, 0xd6, 0x10, 0xcd, 0x9e, 0xe7, 0xf0, 0xc4,
		0x3c, 0xb6, 0x1c, 0x9a, 0x56, 0x73, 0x7f, 0xa6,
		0x14, 0x24, 0xcb, 0x96, 0x1f, 0xe0, 0xaf, 0xcd,
		0x3c, 0x66, 0x43, 0xb7, 0x37, 0x65, 0x34, 0x47,
		0xf8, 0x43, 0xf1, 0xcc, 0x15, 0xb8, 0xdc, 0x35,
		0xe0, 0xa4, 0x2d, 0x78, 0x94, 0xe0, 0x02, 0xf3,
		0x76, 0x46, 0xf7, 0x9b, 0x8d, 0x0d, 0x5d, 0x0b,
		0xd3, 0xdd, 0x9a, 0x9e, 0x62, 0x2e, 0xc5, 0x98,
		0x75, 0x63, 0x0c, 0xbf, 0x8e, 0x49, 0x33, 0x23,
		0x7c, 0x00, 0xcf, 0xfb, 0xcf, 0xba, 0x0f, 0x41,
		0x39, 0x89, 0xb9, 0xcc, 0x59, 0xd0, 0x2b, 0xb6,
		0xec, 0x04, 0xe2, 0xc0, 0x52, 0xc7, 0xcf, 0x71,
		0x47, 0xff, 0x70, 0x7e, 0xa9, 0xbd, 0x1c, 0xdd,
		0x17, 0xa5, 0x6c, 0xb7, 0x10, 0x4f, 0x42, 0x18,
		0x37, 0x69, 0xa9, 0xd2, 0xb3, 0x18, 0x84, 0x92,
		0xa7, 0x47, 0x21, 0xf6, 0x95, 0x63, 0x29, 0xd6,
		0xa5, 0xb6, 0xda, 0x65, 0x67, 0x69, 0xc4, 0x26,
		0xac, 0x8b, 0x08, 0x58, 0xdd, 0x3c, 0x31, 0x20,
		0xd5, 0x0c, 0x88, 0x72, 0x18, 0x16, 0x88, 0x1e,
		0x4a, 0x0f, 0xe1, 0xcf, 0x95, 0x24,
	},
	{
		0x16, 0x03, 0x01, 0x00, 0x72, 0x04, 0x00, 0x00,
		0x6e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x68, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x65,
		0xe8, 0x4b, 0xde, 0xef, 0xba, 0x3e, 0x18, 0x1c,
		0x1e, 0x5e, 0xbc, 0x87, 0xf1, 0x87, 0x8d, 0x72,
		0xe3, 0xbe, 0x0f, 0xdf, 0xfd, 0xd0, 0xb2, 0x89,
		0xf8, 0x05, 0x9a, 0x52, 0x47, 0x77, 0x9e, 0xe8,
		0xb1, 0x1d, 0x18, 0xed, 0x6a, 0x4b, 0x63, 0x1d,
		0xf1, 0x62, 0xd2, 0x65, 0x21, 0x26, 0x73, 0xd4,
		0x35, 0x5b, 0x95, 0x89, 0x12, 0x59, 0x23, 0x8c,
		0xc3, 0xfc, 0xf9, 0x4d, 0x21, 0x79, 0xa0, 0xbd,
		0xff, 0x33, 0xa2, 0x3d, 0x0b, 0x6f, 0x89, 0xc9,
		0x23, 0xe4, 0xe7, 0x9f, 0x1d, 0x98, 0xf6, 0xed,
		0x02, 0x8d, 0xac, 0x1a, 0xf9, 0xcb, 0xa5, 0x14,
		0x03, 0x01, 0x00, 0x01, 0x01, 0x16, 0x03, 0x01,
		0x00, 0x28, 0x91, 0x56, 0x80, 0xe2, 0x6d, 0x51,
		0x88, 0x03, 0xf8, 0x49, 0xe6, 0x6a, 0x5a, 0xfb,
		0x2f, 0x0b, 0xb5, 0xa1, 0x0d, 0x63, 0x83, 0xae,
		0xb9, 0xbc, 0x05, 0xf0, 0x81, 0x00, 0x61, 0x83,
		0x38, 0xda, 0x14, 0xf6, 0xea, 0xd8, 0x78, 0x65,
		0xc7, 0x26, 0x17, 0x03, 0x01, 0x00, 0x18, 0x81,
		0x30, 0x8b, 0x22, 0x5a, 0xd3, 0x7f, 0xc8, 0xf2,
		0x8a, 0x6b, 0xa3, 0xba, 0x4d, 0xe7, 0x6e, 0xd2,
		0xfd, 0xbf, 0xf2, 0xc5, 0x28, 0xa0, 0x62, 0x17,
		0x03, 0x01, 0x00, 0x28, 0x17, 0x83, 0x3c, 0x78,
		0x18, 0xfa, 0x8d, 0x58, 0x5c, 0xaa, 0x05, 0x7d,
		0x67, 0x96, 0x11, 0x60, 0x11, 0xc0, 0x1e, 0x0d,
		0x6a, 0x6e, 0x5f, 0x1d, 0x98, 0x4b, 0xff, 0x82,
		0xee, 0x21, 0x06, 0x29, 0xd3, 0x8b, 0x80, 0x78,
		0x39, 0x05, 0x34, 0x9b, 0x15, 0x03, 0x01, 0x00,
		0x18, 0xa9, 0x38, 0x18, 0x4f, 0x9d, 0x84, 0x75,
		0x88, 0x53, 0xd6, 0x85, 0xc2, 0x15, 0x4b, 0xe3,
		0xe3, 0x35, 0x9a, 0x74, 0xc9, 0x3e, 0x13, 0xc1,
		0x8c,
	},
}

var aesServerScript = [][]byte{
	{
		0x16, 0x03, 0x00, 0x00, 0xc5, 0x01, 0x00, 0x00,
		0xc1, 0x03, 0x03, 0x50, 0xae, 0x5c, 0xe9, 0x5e,
		0x31, 0x93, 0x82, 0xa5, 0x6f, 0x51, 0x82, 0xc8,
		0x55, 0x4f, 0x1f, 0x2e, 0x90, 0x98, 0x81, 0x13,
		0x27, 0x80, 0x68, 0xb4, 0x2d, 0xba, 0x3a, 0x76,
		0xd8, 0xd7, 0x2c, 0x00, 0x00, 0x50, 0xc0, 0x09,
		0xc0, 0x23, 0xc0, 0x2b, 0xc0, 0x0a, 0xc0, 0x24,
		0xc0, 0x2c, 0xc0, 0x08, 0xc0, 0x13, 0xc0, 0x27,
		0xc0, 0x2f, 0xc0, 0x14, 0xc0, 0x30, 0xc0, 0x12,
		0x00, 0x33, 0x00, 0x67, 0x00, 0x45, 0x00, 0x9e,
		0x00, 0x39, 0x00, 0x6b, 0x00, 0x88, 0x00, 0x16,
		0x00, 0x32, 0x00, 0x40, 0x00, 0x44, 0x00, 0xa2,
		0x00, 0x38, 0x00, 0x6a, 0x00, 0x87, 0x00, 0x13,
		0x00, 0x66, 0x00, 0x2f, 0x00, 0x3c, 0x00, 0x41,
		0x00, 0x9c, 0x00, 0x35, 0x00, 0x3d, 0x00, 0x84,
		0x00, 0x0a, 0x00, 0x05, 0x00, 0x04, 0x01, 0x00,
		0x00, 0x48, 0x00, 0x05, 0x00, 0x05, 0x01, 0x00,
		0x00, 0x00, 0x00, 0xff, 0x01, 0x00, 0x01, 0x00,
		0x00, 0x23, 0x00, 0x00, 0x00, 0x0a, 0x00, 0x0c,
		0x00, 0x0a, 0x00, 0x13, 0x00, 0x15, 0x00, 0x17,
		0x00, 0x18, 0x00, 0x19, 0x00, 0x0b, 0x00, 0x02,
		0x01, 0x00, 0x00, 0x0d, 0x00, 0x1c, 0x00, 0x1a,
		0x04, 0x01, 0x04, 0x02, 0x04, 0x03, 0x05, 0x01,
		0x05, 0x03, 0x06, 0x01, 0x06, 0x03, 0x03, 0x01,
		0x03, 0x02, 0x03, 0x03, 0x02, 0x01, 0x02, 0x02,
		0x02, 0x03,
	},
	{
		0x16, 0x03, 0x01, 0x00, 0x30, 0x02, 0x00, 0x00,
		0x2c, 0x03, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x2f, 0x00, 0x00,
		0x04, 0x00, 0x23, 0x00, 0x00, 0x16, 0x03, 0x01,
		0x02, 0xbe, 0x0b, 0x00, 0x02, 0xba, 0x00, 0x02,
		0xb7, 0x00, 0x02, 0xb4, 0x30, 0x82, 0x02, 0xb0,
		0x30, 0x82, 0x02, 0x19, 0xa0, 0x03, 0x02, 0x01,
		0x02, 0x02, 0x09, 0x00, 0x85, 0xb0, 0xbb, 0xa4,
		0x8a, 0x7f, 0xb8, 0xca, 0x30, 0x0d, 0x06, 0x09,
		0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01,
		0x05, 0x05, 0x00, 0x30, 0x45, 0x31, 0x0b, 0x30,
		0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02,
		0x41, 0x55, 0x31, 0x13, 0x30, 0x11, 0x06, 0x03,
		0x55, 0x04, 0x08, 0x13, 0x0a, 0x53, 0x6f, 0x6d,
		0x65, 0x2d, 0x53, 0x74, 0x61, 0x74, 0x65, 0x31,
		0x21, 0x30, 0x1f, 0x06, 0x03, 0x55, 0x04, 0x0a,
		0x13, 0x18, 0x49, 0x6e, 0x74, 0x65, 0x72, 0x6e,
		0x65, 0x74, 0x20, 0x57, 0x69, 0x64, 0x67, 0x69,
		0x74, 0x73, 0x20, 0x50, 0x74, 0x79, 0x20, 0x4c,
		0x74, 0x64, 0x30, 0x1e, 0x17, 0x0d, 0x31, 0x30,
		0x30, 0x34, 0x32, 0x34, 0x30, 0x39, 0x30, 0x39,
		0x33, 0x38, 0x5a, 0x17, 0x0d, 0x31, 0x31, 0x30,
		0x34, 0x32, 0x34, 0x30, 0x39, 0x30, 0x39, 0x33,
		0x38, 0x5a, 0x30, 0x45, 0x31, 0x0b, 0x30, 0x09,
		0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x41,
		0x55, 0x31, 0x13, 0x30, 0x11, 0x06, 0x03, 0x55,
		0x04, 0x08, 0x13, 0x0a, 0x53, 0x6f, 0x6d, 0x65,
		0x2d, 0x53, 0x74, 0x61, 0x74, 0x65, 0x31, 0x21,
		0x30, 0x1f, 0x06, 0x03, 0x55, 0x04, 0x0a, 0x13,
		0x18, 0x49, 0x6e, 0x74, 0x65, 0x72, 0x6e, 0x65,
		0x74, 0x20, 0x57, 0x69, 0x64, 0x67, 0x69, 0x74,
		0x73, 0x20, 0x50, 0x74, 0x79, 0x20, 0x4c, 0x74,
		0x64, 0x30, 0x81, 0x9f, 0x30, 0x0d, 0x06, 0x09,
		0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01,
		0x01, 0x05, 0x00, 0x03, 0x81, 0x8d, 0x00, 0x30,
		0x81, 0x89, 0x02, 0x81, 0x81, 0x00, 0xbb, 0x79,
		0xd6, 0xf5, 0x17, 0xb5, 0xe5, 0xbf, 0x46, 0x10,
		0xd0, 0xdc, 0x69, 0xbe, 0xe6, 0x2b, 0x07, 0x43,
		0x5a, 0xd0, 0x03, 0x2d, 0x8a, 0x7a, 0x43, 0x85,
		0xb7, 0x14, 0x52, 0xe7, 0xa5, 0x65, 0x4c, 0x2c,
		0x78, 0xb8, 0x23, 0x8c, 0xb5, 0xb4, 0x82, 0xe5,
		0xde, 0x1f, 0x95, 0x3b, 0x7e, 0x62, 0xa5, 0x2c,
		0xa5, 0x33, 0xd6, 0xfe, 0x12, 0x5c, 0x7a, 0x56,
		0xfc, 0xf5, 0x06, 0xbf, 0xfa, 0x58, 0x7b, 0x26,
		0x3f, 0xb5, 0xcd, 0x04, 0xd3, 0xd0, 0xc9, 0x21,
		0x96, 0x4a, 0xc7, 0xf4, 0x54, 0x9f, 0x5a, 0xbf,
		0xef, 0x42, 0x71, 0x00, 0xfe, 0x18, 0x99, 0x07,
		0x7f, 0x7e, 0x88, 0x7d, 0x7d, 0xf1, 0x04, 0x39,
		0xc4, 0xa2, 0x2e, 0xdb, 0x51, 0xc9, 0x7c, 0xe3,
		0xc0, 0x4c, 0x3b, 0x32, 0x66, 0x01, 0xcf, 0xaf,
		0xb1, 0x1d, 0xb8, 0x71, 0x9a, 0x1d, 0xdb, 0xdb,
		0x89, 0x6b, 0xae, 0xda, 0x2d, 0x79, 0x02, 0x03,
		0x01, 0x00, 0x01, 0xa3, 0x81, 0xa7, 0x30, 0x81,
		0xa4, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e,
		0x04, 0x16, 0x04, 0x14, 0xb1, 0xad, 0xe2, 0x85,
		0x5a, 0xcf, 0xcb, 0x28, 0xdb, 0x69, 0xce, 0x23,
		0x69, 0xde, 0xd3, 0x26, 0x8e, 0x18, 0x88, 0x39,
		0x30, 0x75, 0x06, 0x03, 0x55, 0x1d, 0x23, 0x04,
		0x6e, 0x30, 0x6c, 0x80, 0x14, 0xb1, 0xad, 0xe2,
		0x85, 0x5a, 0xcf, 0xcb, 0x28, 0xdb, 0x69, 0xce,
		0x23, 0x69, 0xde, 0xd3, 0x26, 0x8e, 0x18, 0x88,
		0x39, 0xa1, 0x49, 0xa4, 0x47, 0x30, 0x45, 0x31,
		0x0b, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06,
		0x13, 0x02, 0x41, 0x55, 0x31, 0x13, 0x30, 0x11,
		0x06, 0x03, 0x55, 0x04, 0x08, 0x13, 0x0a, 0x53,
		0x6f, 0x6d, 0x65, 0x2d, 0x53, 0x74, 0x61, 0x74,
		0x65, 0x31, 0x21, 0x30, 0x1f, 0x06, 0x03, 0x55,
		0x04, 0x0a, 0x13, 0x18, 0x49, 0x6e, 0x74, 0x65,
		0x72, 0x6e, 0x65, 0x74, 0x20, 0x57, 0x69, 0x64,
		0x67, 0x69, 0x74, 0x73, 0x20, 0x50, 0x74, 0x79,
		0x20, 0x4c, 0x74, 0x64, 0x82, 0x09, 0x00, 0x85,
		0xb0, 0xbb, 0xa4, 0x8a, 0x7f, 0xb8, 0xca, 0x30,
		0x0c, 0x06, 0x03, 0x55, 0x1d, 0x13, 0x04, 0x05,
		0x30, 0x03, 0x01, 0x01, 0xff, 0x30, 0x0d, 0x06,
		0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01,
		0x01, 0x05, 0x05, 0x00, 0x03, 0x81, 0x81, 0x00,
		0x08, 0x6c, 0x45, 0x24, 0xc7, 0x6b, 0xb1, 0x59,
		0xab, 0x0c, 0x52, 0xcc, 0xf2, 0xb0, 0x14, 0xd7,
		0x87, 0x9d, 0x7a, 0x64, 0x75, 0xb5, 0x5a, 0x95,
		0x66, 0xe4, 0xc5, 0x2b, 0x8e, 0xae, 0x12, 0x66,
		0x1f, 0xeb, 0x4f, 0x38, 0xb3, 0x6e, 0x60, 0xd3,
		0x92, 0xfd, 0xf7, 0x41, 0x08, 0xb5, 0x25, 0x13,
		0xb1, 0x18, 0x7a, 0x24, 0xfb, 0x30, 0x1d, 0xba,
		0xed, 0x98, 0xb9, 0x17, 0xec, 0xe7, 0xd7, 0x31,
		0x59, 0xdb, 0x95, 0xd3, 0x1d, 0x78, 0xea, 0x50,
		0x56, 0x5c, 0xd5, 0x82, 0x5a, 0x2d, 0x5a, 0x5f,
		0x33, 0xc4, 0xb6, 0xd8, 0xc9, 0x75, 0x90, 0x96,
		0x8c, 0x0f, 0x52, 0x98, 0xb5, 0xcd, 0x98, 0x1f,
		0x89, 0x20, 0x5f, 0xf2, 0xa0, 0x1c, 0xa3, 0x1b,
		0x96, 0x94, 0xdd, 0xa9, 0xfd, 0x57, 0xe9, 0x70,
		0xe8, 0x26, 0x6d, 0x71, 0x99, 0x9b, 0x26, 0x6e,
		0x38, 0x50, 0x29, 0x6c, 0x90, 0xa7, 0xbd, 0xd9,
		0x16, 0x03, 0x01, 0x00, 0x04, 0x0e, 0x00, 0x00,
		0x00,
	},
	{
		0x16, 0x03, 0x01, 0x00, 0x86, 0x10, 0x00, 0x00,
		0x82, 0x00, 0x80, 0x51, 0x2e, 0xec, 0x0d, 0x86,
		0xf3, 0x9f, 0xf2, 0x77, 0x04, 0x27, 0x2b, 0x0e,
		0x9c, 0xab, 0x35, 0x84, 0x65, 0xff, 0x36, 0xef,
		0xc0, 0x08, 0xc9, 0x1d, 0x9f, 0x29, 0xae, 0x8d,
		0xc5, 0x66, 0x81, 0x31, 0x92, 0x5e, 0x3d, 0xac,
		0xaa, 0x37, 0x28, 0x2c, 0x06, 0x91, 0xa6, 0xc2,
		0xd0, 0x83, 0x34, 0x24, 0x1c, 0x88, 0xfc, 0x0a,
		0xcf, 0xbf, 0xc2, 0x94, 0xe2, 0xed, 0xa7, 0x6a,
		0xa8, 0x8d, 0x3d, 0xf7, 0x06, 0x7d, 0x69, 0xf8,
		0x0d, 0xb2, 0xf7, 0xe4, 0x45, 0xcb, 0x0a, 0x25,
		0xcb, 0xb2, 0x2e, 0x38, 0x9a, 0x84, 0x75, 0xe8,
		0xe1, 0x42, 0x39, 0xa2, 0x18, 0x0e, 0x48, 0xca,
		0x33, 0x16, 0x4e, 0xf6, 0x2f, 0xec, 0x07, 0xe7,
		0x57, 0xe1, 0x20, 0x40, 0x40, 0x6d, 0x4e, 0x29,
		0x04, 0x1a, 0x8c, 0x99, 0xfb, 0x19, 0x3c, 0xaa,
		0x75, 0x64, 0xd3, 0xa6, 0xe6, 0xed, 0x3f, 0x5a,
		0xd2, 0xc9, 0x80, 0x14, 0x03, 0x01, 0x00, 0x01,
		0x01, 0x16, 0x03, 0x01, 0x01, 0x10, 0xe9, 0x9e,
		0x06, 0x92, 0x18, 0xbf, 0x5e, 0xaf, 0x33, 0xc1,
		0xbf, 0x0e, 0x12, 0x07, 0x48, 0x4f, 0x6b, 0x6c,
		0xf5, 0x23, 0x5e, 0x87, 0xa7, 0xd3, 0x50, 0x79,
		0x38, 0xdc, 0xe0, 0x49, 0xd3, 0x81, 0x21, 0x12,
		0xd0, 0x3d, 0x9a, 0xfb, 0x83, 0xc1, 0x8b, 0xfc,
		0x14, 0xd5, 0xd5, 0xa7, 0xa3, 0x34, 0x14, 0x71,
		0xbe, 0xea, 0x37, 0x18, 0x12, 0x7f, 0x41, 0xfb,
		0xc5, 0x51, 0x17, 0x9d, 0x96, 0x58, 0x14, 0xfb,
		0x4f, 0xd7, 0xd3, 0x15, 0x0f, 0xec, 0x5a, 0x0d,
		0x35, 0xbb, 0x3c, 0x81, 0x5b, 0x3f, 0xdf, 0x52,
		0xa4, 0x4c, 0xcd, 0x13, 0xe1, 0x10, 0x37, 0x34,
		0xbf, 0xb4, 0x80, 0x1e, 0x8d, 0xe2, 0xc3, 0x7a,
		0x0f, 0x7b, 0x7d, 0x23, 0xeb, 0xd0, 0x99, 0x69,
		0xad, 0x0a, 0x2d, 0xb3, 0x6c, 0xd6, 0x80, 0x11,
		0x7f, 0x6c, 0xed, 0x1b, 0xcd, 0x08, 0x22, 0x56,
		0x90, 0x0e, 0xa4, 0xc3, 0x29, 0x33, 0x96, 0x30,
		0x34, 0x94, 0xa1, 0xeb, 0x9c, 0x1b, 0x5a, 0xd1,
		0x03, 0x61, 0xf9, 0xdd, 0xf3, 0x64, 0x8a, 0xfd,
		0x5f, 0x44, 0xdb, 0x2e, 0xa7, 0xfd, 0xe1, 0x1a,
		0x66, 0xc5, 0x01, 0x9c, 0xc7, 0xd1, 0xc4, 0xd3,
		0xea, 0x14, 0x3c, 0xed, 0x74, 0xbb, 0x1b, 0x97,
		0x8f, 0xf1, 0x29, 0x39, 0x33, 0x92, 0x93, 0x4e,
		0xf5, 0x87, 0x91, 0x61, 0x65, 0x8d, 0x27, 0x8d,
		0x76, 0xc1, 0xfa, 0x6a, 0x99, 0x80, 0xb1, 0x9b,
		0x29, 0x53, 0xce, 0x3e, 0xb6, 0x9a, 0xce, 0x3c,
		0x19, 0x5e, 0x48, 0x83, 0xaa, 0xa7, 0x66, 0x98,
		0x59, 0xf4, 0xbb, 0xf2, 0xbc, 0xd9, 0xc5, 0x9a,
		0xc8, 0x2c, 0x63, 0x58, 0xd5, 0xd4, 0xbc, 0x03,
		0xa9, 0x06, 0xa9, 0x80, 0x0d, 0xb3, 0x46, 0x2d,
		0xe3, 0xc6, 0xaf, 0x1a, 0x39, 0x18, 0x7e, 0x1e,
		0x83, 0x80, 0x46, 0x11, 0xd2, 0x13, 0x9f, 0xda,
		0xfc, 0x2d, 0x42, 0xaa, 0x5a, 0x1d, 0x4c, 0x31,
		0xe5, 0x58, 0x78, 0x5e, 0xe2, 0x04, 0xd6, 0x23,
		0x7f, 0x3f, 0x06, 0xc0, 0x54, 0xf8,
	},
	{
		0x16, 0x03, 0x01, 0x00, 0x72, 0x04, 0x00, 0x00,
		0x6e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x68, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x65,
		0xe8, 0x4b, 0xfb, 0xef, 0xba, 0xed, 0xc5, 0x36,
		0xc8, 0x5a, 0x41, 0x3f, 0x05, 0xfa, 0xfe, 0x48,
		0xc3, 0x91, 0x12, 0x8b, 0xe8, 0x32, 0x6a, 0x9f,
		0xdc, 0x97, 0xe2, 0x77, 0xb9, 0x96, 0x2d, 0xd4,
		0xe5, 0xbd, 0xa1, 0xfd, 0x94, 0xbb, 0x74, 0x63,
		0xb1, 0x0c, 0x38, 0xbc, 0x6f, 0x69, 0xaf, 0xa3,
		0x46, 0x9c, 0x96, 0x41, 0xde, 0x59, 0x23, 0xff,
		0x15, 0x6b, 0x3a, 0xef, 0x91, 0x6d, 0x92, 0x44,
		0xdc, 0x72, 0x1f, 0x40, 0x3d, 0xb5, 0x34, 0x8f,
		0x2a, 0xac, 0x21, 0x69, 0x05, 0x6f, 0xb2, 0x60,
		0x32, 0x5d, 0x3d, 0x97, 0xb4, 0x24, 0x99, 0x14,
		0x03, 0x01, 0x00, 0x01, 0x01, 0x16, 0x03, 0x01,
		0x00, 0x30, 0x68, 0x27, 0x97, 0xca, 0x63, 0x09,
		0x22, 0xed, 0x0e, 0x61, 0x7c, 0x76, 0x31, 0x9c,
		0xbe, 0x27, 0xc9, 0xe6, 0x09, 0xc3, 0xc3, 0xc2,
		0xf4, 0xa2, 0x32, 0xba, 0x7c, 0xf2, 0x0f, 0xb8,
		0x3d, 0xcb, 0xe2, 0x4c, 0xc0, 0x7d, 0x8e, 0x5b,
		0x5a, 0xed, 0x05, 0x5c, 0x15, 0x96, 0x69, 0xc2,
		0x6f, 0x5f, 0x17, 0x03, 0x01, 0x00, 0x20, 0x5a,
		0xfe, 0x0b, 0xe1, 0x6f, 0xa8, 0x54, 0x19, 0x78,
		0xca, 0xba, 0x2e, 0x1e, 0x2e, 0xe1, 0x5d, 0x17,
		0xe5, 0x97, 0x05, 0x2c, 0x08, 0x0c, 0xff, 0xa8,
		0x59, 0xa9, 0xde, 0x5e, 0x21, 0x34, 0x04, 0x17,
		0x03, 0x01, 0x00, 0x30, 0x86, 0xb1, 0x3f, 0x88,
		0x43, 0xf0, 0x07, 0xee, 0xa8, 0xf4, 0xbc, 0xe7,
		0x5f, 0xc6, 0x8c, 0x86, 0x4c, 0xca, 0x70, 0x88,
		0xcc, 0x6a, 0xb4, 0x3d, 0x40, 0xe8, 0x54, 0x89,
		0x19, 0x43, 0x1f, 0x76, 0xe2, 0xac, 0xb2, 0x5b,
		0x92, 0xf8, 0x57, 0x39, 0x2a, 0xc3, 0x6d, 0x13,
		0x45, 0xfa, 0x36, 0x9e, 0x15, 0x03, 0x01, 0x00,
		0x20, 0x6d, 0xed, 0x7b, 0x59, 0x28, 0x2a, 0x27,
		0x04, 0x15, 0x07, 0x4e, 0xeb, 0x13, 0x00, 0xe3,
		0x3a, 0x3f, 0xf8, 0xaa, 0x2b, 0x3b, 0x1a, 0x8c,
		0x12, 0xd6, 0x4c, 0xec, 0x2a, 0xaf, 0x33, 0x60,
		0xaf,
	},
}

var sslv3ServerScript = [][]byte{
	{
		0x16, 0x03, 0x00, 0x00, 0x54, 0x01, 0x00, 0x00,
		0x50, 0x03, 0x00, 0x50, 0x77, 0x3d, 0x42, 0xae,
		0x84, 0xbd, 0xc5, 0x07, 0xa5, 0xc4, 0xd6, 0x16,
		0x4e, 0xd5, 0xc5, 0xfa, 0x02, 0x7a, 0x0f, 0x1d,
		0xc1, 0xe1, 0xaa, 0xe3, 0x3b, 0x4b, 0x6f, 0x11,
		0xfa, 0x1a, 0xa4, 0x00, 0x00, 0x28, 0x00, 0x39,
		0x00, 0x38, 0x00, 0x35, 0x00, 0x16, 0x00, 0x13,
		0x00, 0x0a, 0x00, 0x33, 0x00, 0x32, 0x00, 0x2f,
		0x00, 0x05, 0x00, 0x04, 0x00, 0x15, 0x00, 0x12,
		0x00, 0x09, 0x00, 0x14, 0x00, 0x11, 0x00, 0x08,
		0x00, 0x06, 0x00, 0x03, 0x00, 0xff, 0x02, 0x01,
		0x00,
	},
	{
		0x16, 0x03, 0x00, 0x00, 0x2a, 0x02, 0x00, 0x00,
		0x26, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x16,
		0x03, 0x00, 0x02, 0xbe, 0x0b, 0x00, 0x02, 0xba,
		0x00, 0x02, 0xb7, 0x00, 0x02, 0xb4, 0x30, 0x82,
		0x02, 0xb0, 0x30, 0x82, 0x02, 0x19, 0xa0, 0x03,
		0x02, 0x01, 0x02, 0x02, 0x09, 0x00, 0x85, 0xb0,
		0xbb, 0xa4, 0x8a, 0x7f, 0xb8, 0xca, 0x30, 0x0d,
		0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d,
		0x01, 0x01, 0x05, 0x05, 0x00, 0x30, 0x45, 0x31,
		0x0b, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06,
		0x13, 0x02, 0x41, 0x55, 0x31, 0x13, 0x30, 0x11,
		0x06, 0x03, 0x55, 0x04, 0x08, 0x13, 0x0a, 0x53,
		0x6f, 0x6d, 0x65, 0x2d, 0x53, 0x74, 0x61, 0x74,
		0x65, 0x31, 0x21, 0x30, 0x1f, 0x06, 0x03, 0x55,
		0x04, 0x0a, 0x13, 0x18, 0x49, 0x6e, 0x74, 0x65,
		0x72, 0x6e, 0x65, 0x74, 0x20, 0x57, 0x69, 0x64,
		0x67, 0x69, 0x74, 0x73, 0x20, 0x50, 0x74, 0x79,
		0x20, 0x4c, 0x74, 0x64, 0x30, 0x1e, 0x17, 0x0d,
		0x31, 0x30, 0x30, 0x34, 0x32, 0x34, 0x30, 0x39,
		0x30, 0x39, 0x33, 0x38, 0x5a, 0x17, 0x0d, 0x31,
		0x31, 0x30, 0x34, 0x32, 0x34, 0x30, 0x39, 0x30,
		0x39, 0x33, 0x38, 0x5a, 0x30, 0x45, 0x31, 0x0b,
		0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13,
		0x02, 0x41, 0x55, 0x31, 0x13, 0x30, 0x11, 0x06,
		0x03, 0x55, 0x04, 0x08, 0x13, 0x0a, 0x53, 0x6f,
		0x6d, 0x65, 0x2d, 0x53, 0x74, 0x61, 0x74, 0x65,
		0x31, 0x21, 0x30, 0x1f, 0x06, 0x03, 0x55, 0x04,
		0x0a, 0x13, 0x18, 0x49, 0x6e, 0x74, 0x65, 0x72,
		0x6e, 0x65, 0x74, 0x20, 0x57, 0x69, 0x64, 0x67,
		0x69, 0x74, 0x73, 0x20, 0x50, 0x74, 0x79, 0x20,
		0x4c, 0x74, 0x64, 0x30, 0x81, 0x9f, 0x30, 0x0d,
		0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d,
		0x01, 0x01, 0x01, 0x05, 0x00, 0x03, 0x81, 0x8d,
		0x00, 0x30, 0x81, 0x89, 0x02, 0x81, 0x81, 0x00,
		0xbb, 0x79, 0xd6, 0xf5, 0x17, 0xb5, 0xe5, 0xbf,
		0x46, 0x10, 0xd0, 0xdc, 0x69, 0xbe, 0xe6, 0x2b,
		0x07, 0x43, 0x5a, 0xd0, 0x03, 0x2d, 0x8a, 0x7a,
		0x43, 0x85, 0xb7, 0x14, 0x52, 0xe7, 0xa5, 0x65,
		0x4c, 0x2c, 0x78, 0xb8, 0x23, 0x8c, 0xb5, 0xb4,
		0x82, 0xe5, 0xde, 0x1f, 0x95, 0x3b, 0x7e, 0x62,
		0xa5, 0x2c, 0xa5, 0x33, 0xd6, 0xfe, 0x12, 0x5c,
		0x7a, 0x56, 0xfc, 0xf5, 0x06, 0xbf, 0xfa, 0x58,
		0x7b, 0x26, 0x3f, 0xb5, 0xcd, 0x04, 0xd3, 0xd0,
		0xc9, 0x21, 0x96, 0x4a, 0xc7, 0xf4, 0x54, 0x9f,
		0x5a, 0xbf, 0xef, 0x42, 0x71, 0x00, 0xfe, 0x18,
		0x99, 0x07, 0x7f, 0x7e, 0x88, 0x7d, 0x7d, 0xf1,
		0x04, 0x39, 0xc4, 0xa2, 0x2e, 0xdb, 0x51, 0xc9,
		0x7c, 0xe3, 0xc0, 0x4c, 0x3b, 0x32, 0x66, 0x01,
		0xcf, 0xaf, 0xb1, 0x1d, 0xb8, 0x71, 0x9a, 0x1d,
		0xdb, 0xdb, 0x89, 0x6b, 0xae, 0xda, 0x2d, 0x79,
		0x02, 0x03, 0x01, 0x00, 0x01, 0xa3, 0x81, 0xa7,
		0x30, 0x81, 0xa4, 0x30, 0x1d, 0x06, 0x03, 0x55,
		0x1d, 0x0e, 0x04, 0x16, 0x04, 0x14, 0xb1, 0xad,
		0xe2, 0x85, 0x5a, 0xcf, 0xcb, 0x28, 0xdb, 0x69,
		0xce, 0x23, 0x69, 0xde, 0xd3, 0x26, 0x8e, 0x18,
		0x88, 0x39, 0x30, 0x75, 0x06, 0x03, 0x55, 0x1d,
		0x23, 0x04, 0x6e, 0x30, 0x6c, 0x80, 0x14, 0xb1,
		0xad, 0xe2, 0x85, 0x5a, 0xcf, 0xcb, 0x28, 0xdb,
		0x69, 0xce, 0x23, 0x69, 0xde, 0xd3, 0x26, 0x8e,
		0x18, 0x88, 0x39, 0xa1, 0x49, 0xa4, 0x47, 0x30,
		0x45, 0x31, 0x0b, 0x30, 0x09, 0x06, 0x03, 0x55,
		0x04, 0x06, 0x13, 0x02, 0x41, 0x55, 0x31, 0x13,
		0x30, 0x11, 0x06, 0x03, 0x55, 0x04, 0x08, 0x13,
		0x0a, 0x53, 0x6f, 0x6d, 0x65, 0x2d, 0x53, 0x74,
		0x61, 0x74, 0x65, 0x31, 0x21, 0x30, 0x1f, 0x06,
		0x03, 0x55, 0x04, 0x0a, 0x13, 0x18, 0x49, 0x6e,
		0x74, 0x65, 0x72, 0x6e, 0x65, 0x74, 0x20, 0x57,
		0x69, 0x64, 0x67, 0x69, 0x74, 0x73, 0x20, 0x50,
		0x74, 0x79, 0x20, 0x4c, 0x74, 0x64, 0x82, 0x09,
		0x00, 0x85, 0xb0, 0xbb, 0xa4, 0x8a, 0x7f, 0xb8,
		0xca, 0x30, 0x0c, 0x06, 0x03, 0x55, 0x1d, 0x13,
		0x04, 0x05, 0x30, 0x03, 0x01, 0x01, 0xff, 0x30,
		0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7,
		0x0d, 0x01, 0x01, 0x05, 0x05, 0x00, 0x03, 0x81,
		0x81, 0x00, 0x08, 0x6c, 0x45, 0x24, 0xc7, 0x6b,
		0xb1, 0x59, 0xab, 0x0c, 0x52, 0xcc, 0xf2, 0xb0,
		0x14, 0xd7, 0x87, 0x9d, 0x7a, 0x64, 0x75, 0xb5,
		0x5a, 0x95, 0x66, 0xe4, 0xc5, 0x2b, 0x8e, 0xae,
		0x12, 0x66, 0x1f, 0xeb, 0x4f, 0x38, 0xb3, 0x6e,
		0x60, 0xd3, 0x92, 0xfd, 0xf7, 0x41, 0x08, 0xb5,
		0x25, 0x13, 0xb1, 0x18, 0x7a, 0x24, 0xfb, 0x30,
		0x1d, 0xba, 0xed, 0x98, 0xb9, 0x17, 0xec, 0xe7,
		0xd7, 0x31, 0x59, 0xdb, 0x95, 0xd3, 0x1d, 0x78,
		0xea, 0x50, 0x56, 0x5c, 0xd5, 0x82, 0x5a, 0x2d,
		0x5a, 0x5f, 0x33, 0xc4, 0xb6, 0xd8, 0xc9, 0x75,
		0x90, 0x96, 0x8c, 0x0f, 0x52, 0x98, 0xb5, 0xcd,
		0x98, 0x1f, 0x89, 0x20, 0x5f, 0xf2, 0xa0, 0x1c,
		0xa3, 0x1b, 0x96, 0x94, 0xdd, 0xa9, 0xfd, 0x57,
		0xe9, 0x70, 0xe8, 0x26, 0x6d, 0x71, 0x99, 0x9b,
		0x26, 0x6e, 0x38, 0x50, 0x29, 0x6c, 0x90, 0xa7,
		0xbd, 0xd9, 0x16, 0x03, 0x00, 0x00, 0x04, 0x0e,
		0x00, 0x00, 0x00,
	},
	{
		0x16, 0x03, 0x00, 0x00, 0x84, 0x10, 0x00, 0x00,
		0x80, 0x4a, 0x8d, 0xc4, 0x38, 0x7a, 0x9c, 0xd6,
		0xe8, 0x72, 0x9e, 0xa3, 0xdf, 0x37, 0xb4, 0x6c,
		0x58, 0x33, 0x59, 0xd9, 0xc9, 0x4b, 0x50, 0x33,
		0x6c, 0xed, 0x73, 0x38, 0x2a, 0x46, 0x55, 0x31,
		0xa9, 0x8e, 0x8e, 0xfc, 0x0b, 0x5d, 0x5f, 0x3c,
		0x88, 0x28, 0x3f, 0x60, 0x51, 0x13, 0xf1, 0x59,
		0x0c, 0xa3, 0x5e, 0xe0, 0xa3, 0x35, 0x06, 0xb1,
		0x71, 0x59, 0x24, 0x4e, 0xed, 0x07, 0x15, 0x88,
		0x50, 0xef, 0xc2, 0xb2, 0x2a, 0x52, 0x30, 0x6a,
		0x7c, 0xbe, 0x2f, 0xc6, 0x8f, 0xa8, 0x83, 0xc5,
		0x80, 0x14, 0x62, 0x74, 0x7f, 0x96, 0x9f, 0x41,
		0x32, 0x74, 0xdd, 0x76, 0x2d, 0x7b, 0xeb, 0x7b,
		0xea, 0xd0, 0x4f, 0x0c, 0xcf, 0x9a, 0x9c, 0xc5,
		0x7a, 0xe4, 0xbc, 0xf8, 0xa6, 0xe1, 0x09, 0x8e,
		0x7c, 0x53, 0x3a, 0xe3, 0x30, 0x8f, 0x76, 0xee,
		0x58, 0xbb, 0xfd, 0x0b, 0x06, 0xb8, 0xdf, 0xb7,
		0x31, 0x14, 0x03, 0x00, 0x00, 0x01, 0x01, 0x16,
		0x03, 0x00, 0x00, 0x3c, 0x13, 0x91, 0xc6, 0x4a,
		0x0c, 0x59, 0x25, 0xce, 0x54, 0xc0, 0x1d, 0xb9,
		0x2a, 0xff, 0x4d, 0xca, 0x26, 0x0c, 0x8c, 0x04,
		0x98, 0x7c, 0x7c, 0x38, 0xa3, 0xf5, 0xf9, 0x36,
		0x1c, 0x04, 0x32, 0x47, 0x2d, 0x48, 0x0e, 0x96,
		0xe8, 0x2b, 0x5e, 0x5a, 0xc6, 0x0a, 0x48, 0x41,
		0x34, 0x5e, 0x62, 0xd5, 0x68, 0x4e, 0x44, 0x1d,
		0xb2, 0xa1, 0x11, 0xad, 0x6e, 0x14, 0x85, 0x61,
	},
	{
		0x14, 0x03, 0x00, 0x00, 0x01, 0x01, 0x16, 0x03,
		0x00, 0x00, 0x3c, 0x88, 0xae, 0xa9, 0xd4, 0xa8,
		0x10, 0x8d, 0x65, 0xa6, 0x3e, 0x1e, 0xed, 0xd2,
		0xfc, 0xc4, 0x7c, 0xa8, 0x94, 0x4f, 0x11, 0xaf,
		0xa6, 0x87, 0x09, 0x37, 0x54, 0xf7, 0x69, 0xd1,
		0xb5, 0x25, 0x6b, 0xb5, 0xed, 0xcb, 0x25, 0x39,
		0x73, 0xeb, 0x53, 0x6c, 0xc7, 0xb4, 0x29, 0x8f,
		0xd6, 0x49, 0xd1, 0x95, 0x59, 0x80, 0x9a, 0x67,
		0x5c, 0xb2, 0xe0, 0xbd, 0x1e, 0xff, 0xaa, 0x17,
		0x03, 0x00, 0x00, 0x21, 0x65, 0x7b, 0x99, 0x09,
		0x02, 0xc3, 0x9d, 0x54, 0xd6, 0xe7, 0x32, 0x62,
		0xab, 0xc1, 0x09, 0x91, 0x30, 0x0a, 0xc9, 0xfa,
		0x70, 0xec, 0x06, 0x7b, 0xa3, 0xe1, 0x5f, 0xb4,
		0x63, 0xe6, 0x5c, 0xba, 0x1f, 0x15, 0x03, 0x00,
		0x00, 0x16, 0x40, 0x70, 0xbe, 0xe6, 0xa6, 0xee,
		0x8f, 0xd0, 0x87, 0xa0, 0x43, 0xa1, 0x92, 0xd7,
		0xd0, 0x1a, 0x0c, 0x20, 0x7c, 0xbf, 0xa2, 0xb5,
	},
}

var selectCertificateBySNIScript = [][]byte{
	{
		0x16, 0x03, 0x01, 0x00, 0x6a, 0x01, 0x00, 0x00,
		0x66, 0x03, 0x01, 0x50, 0x77, 0x3d, 0xfe, 0xfb,
		0x8d, 0xc2, 0x68, 0xeb, 0xf9, 0xfa, 0x54, 0x97,
		0x86, 0x45, 0xa2, 0xa3, 0xed, 0xb1, 0x91, 0xb8,
		0x28, 0xc0, 0x47, 0xaf, 0xfb, 0xcd, 0xdc, 0x0e,
		0xb3, 0xea, 0xa5, 0x00, 0x00, 0x28, 0x00, 0x39,
		0x00, 0x38, 0x00, 0x35, 0x00, 0x16, 0x00, 0x13,
		0x00, 0x0a, 0x00, 0x33, 0x00, 0x32, 0x00, 0x2f,
		0x00, 0x05, 0x00, 0x04, 0x00, 0x15, 0x00, 0x12,
		0x00, 0x09, 0x00, 0x14, 0x00, 0x11, 0x00, 0x08,
		0x00, 0x06, 0x00, 0x03, 0x00, 0xff, 0x02, 0x01,
		0x00, 0x00, 0x14, 0x00, 0x00, 0x00, 0x10, 0x00,
		0x0e, 0x00, 0x00, 0x0b, 0x73, 0x6e, 0x69, 0x74,
		0x65, 0x73, 0x74, 0x2e, 0x63, 0x6f, 0x6d,
	},
	{
		0x16, 0x03, 0x01, 0x00, 0x2a, 0x02, 0x00, 0x00,
		0x26, 0x03, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x16,
		0x03, 0x01, 0x02, 0x00, 0x0b, 0x00, 0x01, 0xfc,
		0x00, 0x01, 0xf9, 0x00, 0x01, 0xf6, 0x30, 0x82,
		0x01, 0xf2, 0x30, 0x82, 0x01, 0x5d, 0xa0, 0x03,
		0x02, 0x01, 0x02, 0x02, 0x01, 0x00, 0x30, 0x0b,
		0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d,
		0x01, 0x01, 0x05, 0x30, 0x28, 0x31, 0x10, 0x30,
		0x0e, 0x06, 0x03, 0x55, 0x04, 0x0a, 0x13, 0x07,
		0x41, 0x63, 0x6d, 0x65, 0x20, 0x43, 0x6f, 0x31,
		0x14, 0x30, 0x12, 0x06, 0x03, 0x55, 0x04, 0x03,
		0x13, 0x0b, 0x73, 0x6e, 0x69, 0x74, 0x65, 0x73,
		0x74, 0x2e, 0x63, 0x6f, 0x6d, 0x30, 0x1e, 0x17,
		0x0d, 0x31, 0x32, 0x30, 0x34, 0x31, 0x31, 0x31,
		0x37, 0x34, 0x30, 0x33, 0x35, 0x5a, 0x17, 0x0d,
		0x31, 0x33, 0x30, 0x34, 0x31, 0x31, 0x31, 0x37,
		0x34, 0x35, 0x33, 0x35, 0x5a, 0x30, 0x28, 0x31,
		0x10, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x04, 0x0a,
		0x13, 0x07, 0x41, 0x63, 0x6d, 0x65, 0x20, 0x43,
		0x6f, 0x31, 0x14, 0x30, 0x12, 0x06, 0x03, 0x55,
		0x04, 0x03, 0x13, 0x0b, 0x73, 0x6e, 0x69, 0x74,
		0x65, 0x73, 0x74, 0x2e, 0x63, 0x6f, 0x6d, 0x30,
		0x81, 0x9d, 0x30, 0x0b, 0x06, 0x09, 0x2a, 0x86,
		0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x01, 0x03,
		0x81, 0x8d, 0x00, 0x30, 0x81, 0x89, 0x02, 0x81,
		0x81, 0x00, 0xbb, 0x79, 0xd6, 0xf5, 0x17, 0xb5,
		0xe5, 0xbf, 0x46, 0x10, 0xd0, 0xdc, 0x69, 0xbe,
		0xe6, 0x2b, 0x07, 0x43, 0x5a, 0xd0, 0x03, 0x2d,
		0x8a, 0x7a, 0x43, 0x85, 0xb7, 0x14, 0x52, 0xe7,
		0xa5, 0x65, 0x4c, 0x2c, 0x78, 0xb8, 0x23, 0x8c,
		0xb5, 0xb4, 0x82, 0xe5, 0xde, 0x1f, 0x95, 0x3b,
		0x7e, 0x62, 0xa5, 0x2c, 0xa5, 0x33, 0xd6, 0xfe,
		0x12, 0x5c, 0x7a, 0x56, 0xfc, 0xf5, 0x06, 0xbf,
		0xfa, 0x58, 0x7b, 0x26, 0x3f, 0xb5, 0xcd, 0x04,
		0xd3, 0xd0, 0xc9, 0x21, 0x96, 0x4a, 0xc7, 0xf4,
		0x54, 0x9f, 0x5a, 0xbf, 0xef, 0x42, 0x71, 0x00,
		0xfe, 0x18, 0x99, 0x07, 0x7f, 0x7e, 0x88, 0x7d,
		0x7d, 0xf1, 0x04, 0x39, 0xc4, 0xa2, 0x2e, 0xdb,
		0x51, 0xc9, 0x7c, 0xe3, 0xc0, 0x4c, 0x3b, 0x32,
		0x66, 0x01, 0xcf, 0xaf, 0xb1, 0x1d, 0xb8, 0x71,
		0x9a, 0x1d, 0xdb, 0xdb, 0x89, 0x6b, 0xae, 0xda,
		0x2d, 0x79, 0x02, 0x03, 0x01, 0x00, 0x01, 0xa3,
		0x32, 0x30, 0x30, 0x30, 0x0e, 0x06, 0x03, 0x55,
		0x1d, 0x0f, 0x01, 0x01, 0xff, 0x04, 0x04, 0x03,
		0x02, 0x00, 0xa0, 0x30, 0x0d, 0x06, 0x03, 0x55,
		0x1d, 0x0e, 0x04, 0x06, 0x04, 0x04, 0x01, 0x02,
		0x03, 0x04, 0x30, 0x0f, 0x06, 0x03, 0x55, 0x1d,
		0x23, 0x04, 0x08, 0x30, 0x06, 0x80, 0x04, 0x01,
		0x02, 0x03, 0x04, 0x30, 0x0b, 0x06, 0x09, 0x2a,
		0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x05,
		0x03, 0x81, 0x81, 0x00, 0x89, 0xc6, 0x45, 0x5f,
		0x1c, 0x1f, 0x5e, 0xf8, 0xeb, 0x1a, 0xb1, 0x74,
		0xee, 0x24, 0x39, 0x05, 0x9f, 0x5c, 0x42, 0x59,
		0xbb, 0x1a, 0x8d, 0x86, 0xcd, 0xb1, 0xd0, 0x56,
		0xf5, 0x6a, 0x71, 0x7d, 0xa4, 0x0e, 0x95, 0xab,
		0x90, 0xf5, 0x9e, 0x8d, 0xea, 0xf6, 0x27, 0xc1,
		0x57, 0x99, 0x50, 0x94, 0xdb, 0x08, 0x02, 0x26,
		0x6e, 0xb3, 0x4f, 0xc6, 0x84, 0x2d, 0xea, 0x8a,
		0x4b, 0x68, 0xd9, 0xc1, 0x38, 0x91, 0x03, 0xab,
		0x84, 0xfb, 0x9e, 0x1f, 0x85, 0xd9, 0xb5, 0xd2,
		0x3f, 0xf2, 0x31, 0x2c, 0x86, 0x70, 0xfb, 0xb5,
		0x40, 0x14, 0x82, 0x45, 0xa4, 0xeb, 0xaf, 0xe2,
		0x64, 0xd9, 0x0c, 0x8a, 0x4c, 0xf4, 0xf8, 0x5b,
		0x0f, 0xac, 0x12, 0xac, 0x2f, 0xc4, 0xa3, 0x15,
		0x4b, 0xad, 0x52, 0x46, 0x28, 0x68, 0xaf, 0x96,
		0xc6, 0x2c, 0x65, 0x25, 0xd6, 0x52, 0xb6, 0xe3,
		0x18, 0x45, 0xbd, 0xcc, 0x16, 0x03, 0x01, 0x00,
		0x04, 0x0e, 0x00, 0x00, 0x00,
	},
	{
		0x16, 0x03, 0x01, 0x00, 0x86, 0x10, 0x00, 0x00,
		0x82, 0x00, 0x80, 0x69, 0xc3, 0xd4, 0x0e, 0xcc,
		0xdc, 0xbc, 0x5e, 0xc2, 0x64, 0xa6, 0xde, 0x3c,
		0x0c, 0x7e, 0x0c, 0x6b, 0x80, 0x0f, 0xd4, 0x8f,
		0x02, 0x4b, 0xb2, 0xba, 0x8d, 0x01, 0xeb, 0x6b,
		0xa1, 0x2e, 0x79, 0x37, 0xba, 0xae, 0x24, 0xc2,
		0x26, 0x72, 0x51, 0xe1, 0x82, 0x8e, 0x51, 0x41,
		0x1c, 0x54, 0xa4, 0x26, 0xbe, 0x13, 0xcd, 0x1b,
		0xc6, 0xed, 0x3d, 0x1f, 0xfd, 0x72, 0x80, 0x90,
		0xdb, 0xbf, 0xd6, 0x39, 0x94, 0x5f, 0x48, 0xfb,
		0x25, 0x5a, 0xc9, 0x60, 0x9b, 0xd7, 0xc6, 0x20,
		0xa8, 0x66, 0x64, 0x13, 0xf3, 0x65, 0xc8, 0xb1,
		0xd5, 0x33, 0x21, 0x0e, 0x73, 0x41, 0xc0, 0x18,
		0x1a, 0x37, 0xfe, 0xcf, 0x28, 0x2a, 0xcd, 0xe4,
		0x0b, 0xac, 0xdd, 0x25, 0x5e, 0xcb, 0x17, 0x51,
		0x69, 0xd5, 0x8c, 0xf4, 0xb6, 0x21, 0x98, 0xef,
		0x20, 0xdb, 0x14, 0x67, 0xf3, 0x7c, 0x95, 0x6a,
		0x48, 0x2a, 0x6a, 0x14, 0x03, 0x01, 0x00, 0x01,
		0x01, 0x16, 0x03, 0x01, 0x00, 0x24, 0x36, 0x1b,
		0x09, 0xe5, 0xb9, 0xb9, 0x4d, 0x7d, 0xae, 0x87,
		0xb6, 0x0f, 0xaf, 0xec, 0x22, 0xba, 0x0d, 0xa5,
		0x96, 0x5e, 0x64, 0x65, 0xe7, 0xfb, 0xe3, 0xf3,
		0x6b, 0x72, 0xa8, 0xdb, 0xed, 0xd8, 0x69, 0x9c,
		0x08, 0xd8,
	},
	{
		0x14, 0x03, 0x01, 0x00, 0x01, 0x01, 0x16, 0x03,
		0x01, 0x00, 0x24, 0x60, 0xf7, 0x09, 0x5f, 0xd1,
		0xcb, 0xc9, 0xe1, 0x22, 0xb5, 0x2a, 0xcc, 0xde,
		0x7c, 0xa7, 0xb8, 0x85, 0x00, 0xbc, 0xfd, 0x85,
		0xe1, 0x91, 0x36, 0xbb, 0x07, 0x42, 0xad, 0x3d,
		0x29, 0x62, 0x69, 0xc1, 0x45, 0x92, 0x6f, 0x17,
		0x03, 0x01, 0x00, 0x21, 0x0d, 0xf9, 0xd5, 0x87,
		0xb9, 0x57, 0x3c, 0x50, 0x19, 0xe4, 0x3a, 0x50,
		0x45, 0xcc, 0x86, 0x89, 0xd4, 0x32, 0x79, 0x45,
		0x7c, 0x9f, 0x96, 0xd4, 0x54, 0x56, 0x0c, 0x63,
		0x72, 0x81, 0xc3, 0xd3, 0xe3, 0x15, 0x03, 0x01,
		0x00, 0x16, 0x84, 0xec, 0x2e, 0xf6, 0xaf, 0x4f,
		0xee, 0x48, 0x0f, 0xbe, 0xcd, 0x82, 0x5c, 0x56,
		0x16, 0xe4, 0xfb, 0x89, 0xc5, 0x57, 0x3e, 0x91,
	},
}

var issueSessionTicketTest = [][]byte{
	{
		0x16, 0x03, 0x01, 0x00, 0x5a, 0x01, 0x00, 0x00,
		0x56, 0x03, 0x01, 0x50, 0x77, 0x3e, 0x49, 0x7a,
		0xb7, 0x86, 0x5c, 0x27, 0xd2, 0x97, 0x61, 0xe3,
		0x49, 0x41, 0x48, 0xe7, 0x0e, 0xaa, 0x7e, 0x4d,
		0xb8, 0xdc, 0x01, 0x97, 0xfb, 0xab, 0x53, 0xb2,
		0x5e, 0x36, 0xf6, 0x00, 0x00, 0x28, 0x00, 0x39,
		0x00, 0x38, 0x00, 0x35, 0x00, 0x16, 0x00, 0x13,
		0x00, 0x0a, 0x00, 0x33, 0x00, 0x32, 0x00, 0x2f,
		0x00, 0x05, 0x00, 0x04, 0x00, 0x15, 0x00, 0x12,
		0x00, 0x09, 0x00, 0x14, 0x00, 0x11, 0x00, 0x08,
		0x00, 0x06, 0x00, 0x03, 0x00, 0xff, 0x02, 0x01,
		0x00, 0x00, 0x04, 0x00, 0x23, 0x00, 0x00,
	},
	{
		0x16, 0x03, 0x01, 0x00, 0x30, 0x02, 0x00, 0x00,
		0x2c, 0x03, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00,
		0x04, 0x00, 0x23, 0x00, 0x00, 0x16, 0x03, 0x01,
		0x02, 0xbe, 0x0b, 0x00, 0x02, 0xba, 0x00, 0x02,
		0xb7, 0x00, 0x02, 0xb4, 0x30, 0x82, 0x02, 0xb0,
		0x30, 0x82, 0x02, 0x19, 0xa0, 0x03, 0x02, 0x01,
		0x02, 0x02, 0x09, 0x00, 0x85, 0xb0, 0xbb, 0xa4,
		0x8a, 0x7f, 0xb8, 0xca, 0x30, 0x0d, 0x06, 0x09,
		0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01,
		0x05, 0x05, 0x00, 0x30, 0x45, 0x31, 0x0b, 0x30,
		0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02,
		0x41, 0x55, 0x31, 0x13, 0x30, 0x11, 0x06, 0x03,
		0x55, 0x04, 0x08, 0x13, 0x0a, 0x53, 0x6f, 0x6d,
		0x65, 0x2d, 0x53, 0x74, 0x61, 0x74, 0x65, 0x31,
		0x21, 0x30, 0x1f, 0x06, 0x03, 0x55, 0x04, 0x0a,
		0x13, 0x18, 0x49, 0x6e, 0x74, 0x65, 0x72, 0x6e,
		0x65, 0x74, 0x20, 0x57, 0x69, 0x64, 0x67, 0x69,
		0x74, 0x73, 0x20, 0x50, 0x74, 0x79, 0x20, 0x4c,
		0x74, 0x64, 0x30, 0x1e, 0x17, 0x0d, 0x31, 0x30,
		0x30, 0x34, 0x32, 0x34, 0x30, 0x39, 0x30, 0x39,
		0x33, 0x38, 0x5a, 0x17, 0x0d, 0x31, 0x31, 0x30,
		0x34, 0x32, 0x34, 0x30, 0x39, 0x30, 0x39, 0x33,
		0x38, 0x5a, 0x30, 0x45, 0x31, 0x0b, 0x30, 0x09,
		0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x41,
		0x55, 0x31, 0x13, 0x30, 0x11, 0x06, 0x03, 0x55,
		0x04, 0x08, 0x13, 0x0a, 0x53, 0x6f, 0x6d, 0x65,
		0x2d, 0x53, 0x74, 0x61, 0x74, 0x65, 0x31, 0x21,
		0x30, 0x1f, 0x06, 0x03, 0x55, 0x04, 0x0a, 0x13,
		0x18, 0x49, 0x6e, 0x74, 0x65, 0x72, 0x6e, 0x65,
		0x74, 0x20, 0x57, 0x69, 0x64, 0x67, 0x69, 0x74,
		0x73, 0x20, 0x50, 0x74, 0x79, 0x20, 0x4c, 0x74,
		0x64, 0x30, 0x81, 0x9f, 0x30, 0x0d, 0x06, 0x09,
		0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01,
		0x01, 0x05, 0x00, 0x03, 0x81, 0x8d, 0x00, 0x30,
		0x81, 0x89, 0x02, 0x81, 0x81, 0x00, 0xbb, 0x79,
		0xd6, 0xf5, 0x17, 0xb5, 0xe5, 0xbf, 0x46, 0x10,
		0xd0, 0xdc, 0x69, 0xbe, 0xe6, 0x2b, 0x07, 0x43,
		0x5a, 0xd0, 0x03, 0x2d, 0x8a, 0x7a, 0x43, 0x85,
		0xb7, 0x14, 0x52, 0xe7, 0xa5, 0x65, 0x4c, 0x2c,
		0x78, 0xb8, 0x23, 0x8c, 0xb5, 0xb4, 0x82, 0xe5,
		0xde, 0x1f, 0x95, 0x3b, 0x7e, 0x62, 0xa5, 0x2c,
		0xa5, 0x33, 0xd6, 0xfe, 0x12, 0x5c, 0x7a, 0x56,
		0xfc, 0xf5, 0x06, 0xbf, 0xfa, 0x58, 0x7b, 0x26,
		0x3f, 0xb5, 0xcd, 0x04, 0xd3, 0xd0, 0xc9, 0x21,
		0x96, 0x4a, 0xc7, 0xf4, 0x54, 0x9f, 0x5a, 0xbf,
		0xef, 0x42, 0x71, 0x00, 0xfe, 0x18, 0x99, 0x07,
		0x7f, 0x7e, 0x88, 0x7d, 0x7d, 0xf1, 0x04, 0x39,
		0xc4, 0xa2, 0x2e, 0xdb, 0x51, 0xc9, 0x7c, 0xe3,
		0xc0, 0x4c, 0x3b, 0x32, 0x66, 0x01, 0xcf, 0xaf,
		0xb1, 0x1d, 0xb8, 0x71, 0x9a, 0x1d, 0xdb, 0xdb,
		0x89, 0x6b, 0xae, 0xda, 0x2d, 0x79, 0x02, 0x03,
		0x01, 0x00, 0x01, 0xa3, 0x81, 0xa7, 0x30, 0x81,
		0xa4, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e,
		0x04, 0x16, 0x04, 0x14, 0xb1, 0xad, 0xe2, 0x85,
		0x5a, 0xcf, 0xcb, 0x28, 0xdb, 0x69, 0xce, 0x23,
		0x69, 0xde, 0xd3, 0x26, 0x8e, 0x18, 0x88, 0x39,
		0x30, 0x75, 0x06, 0x03, 0x55, 0x1d, 0x23, 0x04,
		0x6e, 0x30, 0x6c, 0x80, 0x14, 0xb1, 0xad, 0xe2,
		0x85, 0x5a, 0xcf, 0xcb, 0x28, 0xdb, 0x69, 0xce,
		0x23, 0x69, 0xde, 0xd3, 0x26, 0x8e, 0x18, 0x88,
		0x39, 0xa1, 0x49, 0xa4, 0x47, 0x30, 0x45, 0x31,
		0x0b, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06,
		0x13, 0x02, 0x41, 0x55, 0x31, 0x13, 0x30, 0x11,
		0x06, 0x03, 0x55, 0x04, 0x08, 0x13, 0x0a, 0x53,
		0x6f, 0x6d, 0x65, 0x2d, 0x53, 0x74, 0x61, 0x74,
		0x65, 0x31, 0x21, 0x30, 0x1f, 0x06, 0x03, 0x55,
		0x04, 0x0a, 0x13, 0x18, 0x49, 0x6e, 0x74, 0x65,
		0x72, 0x6e, 0x65, 0x74, 0x20, 0x57, 0x69, 0x64,
		0x67, 0x69, 0x74, 0x73, 0x20, 0x50, 0x74, 0x79,
		0x20, 0x4c, 0x74, 0x64, 0x82, 0x09, 0x00, 0x85,
		0xb0, 0xbb, 0xa4, 0x8a, 0x7f, 0xb8, 0xca, 0x30,
		0x0c, 0x06, 0x03, 0x55, 0x1d, 0x13, 0x04, 0x05,
		0x30, 0x03, 0x01, 0x01, 0xff, 0x30, 0x0d, 0x06,
		0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01,
		0x01, 0x05, 0x05, 0x00, 0x03, 0x81, 0x81, 0x00,
		0x08, 0x6c, 0x45, 0x24, 0xc7, 0x6b, 0xb1, 0x59,
		0xab, 0x0c, 0x52, 0xcc, 0xf2, 0xb0, 0x14, 0xd7,
		0x87, 0x9d, 0x7a, 0x64, 0x75, 0xb5, 0x5a, 0x95,
		0x66, 0xe4, 0xc5, 0x2b, 0x8e, 0xae, 0x12, 0x66,
		0x1f, 0xeb, 0x4f, 0x38, 0xb3, 0x6e, 0x60, 0xd3,
		0x92, 0xfd, 0xf7, 0x41, 0x08, 0xb5, 0x25, 0x13,
		0xb1, 0x18, 0x7a, 0x24, 0xfb, 0x30, 0x1d, 0xba,
		0xed, 0x98, 0xb9, 0x17, 0xec, 0xe7, 0xd7, 0x31,
		0x59, 0xdb, 0x95, 0xd3, 0x1d, 0x78, 0xea, 0x50,
		0x56, 0x5c, 0xd5, 0x82, 0x5a, 0x2d, 0x5a, 0x5f,
		0x33, 0xc4, 0xb6, 0xd8, 0xc9, 0x75, 0x90, 0x96,
		0x8c, 0x0f, 0x52, 0x98, 0xb5, 0xcd, 0x98, 0x1f,
		0x89, 0x20, 0x5f, 0xf2, 0xa0, 0x1c, 0xa3, 0x1b,
		0x96, 0x94, 0xdd, 0xa9, 0xfd, 0x57, 0xe9, 0x70,
		0xe8, 0x26, 0x6d, 0x71, 0x99, 0x9b, 0x26, 0x6e,
		0x38, 0x50, 0x29, 0x6c, 0x90, 0xa7, 0xbd, 0xd9,
		0x16, 0x03, 0x01, 0x00, 0x04, 0x0e, 0x00, 0x00,
		0x00,
	},
	{
		0x16, 0x03, 0x01, 0x00, 0x86, 0x10, 0x00, 0x00,
		0x82, 0x00, 0x80, 0x68, 0x10, 0xdc, 0x80, 0xbc,
		0xb3, 0x5a, 0x10, 0x75, 0x89, 0xcc, 0xe5, 0x9f,
		0xbf, 0xe2, 0xce, 0xa4, 0x9f, 0x7f, 0x60, 0xc4,
		0xfe, 0x5c, 0xb5, 0x02, 0x2d, 0xa5, 0xa9, 0x1e,
		0x2c, 0x10, 0x79, 0x15, 0x0f, 0xed, 0x96, 0xb3,
		0xa8, 0x5e, 0x21, 0xbc, 0x5b, 0xdc, 0x58, 0x04,
		0x7d, 0x37, 0xdb, 0xa0, 0x31, 0xe8, 0x4f, 0x04,
		0xbc, 0x46, 0x7c, 0xdb, 0x2e, 0x93, 0x07, 0xaf,
		0xa6, 0x36, 0xd3, 0x39, 0x8d, 0x1d, 0x95, 0xa8,
		0x50, 0x4b, 0xc4, 0x2b, 0xde, 0xd7, 0x04, 0x6d,
		0x77, 0x6c, 0x4d, 0x70, 0x51, 0x88, 0x16, 0x31,
		0x40, 0xb5, 0xba, 0x90, 0x47, 0x64, 0x0c, 0x87,
		0xa5, 0x19, 0xf9, 0x89, 0x24, 0x3c, 0x5e, 0x4b,
		0xaa, 0xe0, 0x60, 0x47, 0x0f, 0x2e, 0xcc, 0xc2,
		0xd5, 0x21, 0xed, 0x72, 0xd0, 0xa9, 0xdd, 0x2a,
		0x2b, 0xef, 0x08, 0x3a, 0x65, 0xea, 0x8b, 0x52,
		0x77, 0x2d, 0xcc, 0x14, 0x03, 0x01, 0x00, 0x01,
		0x01, 0x16, 0x03, 0x01, 0x00, 0x24, 0xe2, 0x95,
		0x62, 0x3c, 0x18, 0xe5, 0xc7, 0x2c, 0xda, 0x16,
		0x9b, 0x28, 0x0d, 0xf7, 0x88, 0x7b, 0x5d, 0x33,
		0x55, 0x3b, 0x01, 0x73, 0xf2, 0xc6, 0x4e, 0x96,
		0x01, 0x01, 0x83, 0x65, 0xd4, 0xef, 0x12, 0x13,
		0x1d, 0x42,
	},
	{
		0x16, 0x03, 0x01, 0x00, 0x72, 0x04, 0x00, 0x00,
		0x6e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x68, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x65,
		0xe8, 0x4b, 0xd1, 0xef, 0xba, 0xfb, 0x41, 0x92,
		0x6d, 0x37, 0x5f, 0xf8, 0x7d, 0x90, 0x0f, 0x01,
		0xf8, 0x8c, 0xee, 0xbc, 0xd9, 0x0c, 0x97, 0x7e,
		0x23, 0x46, 0xe2, 0x6b, 0x52, 0xc6, 0xc6, 0x97,
		0x1d, 0xab, 0xde, 0xa0, 0x86, 0x94, 0xc8, 0x2e,
		0x8b, 0x2e, 0x42, 0x5f, 0xc2, 0x70, 0x35, 0xc9,
		0xee, 0x37, 0xeb, 0x70, 0xaa, 0x59, 0x23, 0x6c,
		0xc8, 0xc1, 0x84, 0x89, 0x39, 0x87, 0x73, 0x0a,
		0x7e, 0xba, 0xca, 0xed, 0x63, 0xba, 0x4e, 0x4f,
		0xf3, 0x31, 0x4b, 0xf0, 0xee, 0x91, 0xa5, 0xb4,
		0x62, 0x01, 0x9e, 0xbd, 0xbc, 0xb3, 0x35, 0x14,
		0x03, 0x01, 0x00, 0x01, 0x01, 0x16, 0x03, 0x01,
		0x00, 0x24, 0x3f, 0x66, 0xe4, 0x98, 0xc1, 0x3f,
		0xc6, 0x2c, 0x81, 0xfb, 0xa9, 0x9f, 0x27, 0xe9,
		0x63, 0x20, 0x1e, 0x0e, 0x4f, 0xfc, 0x5d, 0x12,
		0xee, 0x77, 0x73, 0xc6, 0x96, 0x51, 0xf2, 0x26,
		0x35, 0x3f, 0xce, 0x6a, 0xa9, 0xfd, 0x17, 0x03,
		0x01, 0x00, 0x21, 0x8d, 0xd5, 0x67, 0x60, 0x5d,
		0xa7, 0x93, 0xcc, 0x39, 0x78, 0x59, 0xab, 0xdb,
		0x10, 0x96, 0xf2, 0xad, 0xa2, 0x85, 0xe2, 0x93,
		0x43, 0x43, 0xcf, 0x82, 0xbd, 0x1f, 0xdc, 0x7a,
		0x72, 0xd6, 0x83, 0x3b, 0x15, 0x03, 0x01, 0x00,
		0x16, 0x89, 0x55, 0xf6, 0x42, 0x71, 0xa9, 0xe9,
		0x05, 0x68, 0xe8, 0xce, 0x0d, 0x21, 0xe9, 0xec,
		0xf2, 0x27, 0x67, 0xa7, 0x94, 0xf8, 0x34,
	},
}
var serverResumeTest = [][]byte{
	{
		0x16, 0x03, 0x01, 0x00, 0xc2, 0x01, 0x00, 0x00,
		0xbe, 0x03, 0x01, 0x50, 0x77, 0x3e, 0x4f, 0x1f,
		0x6f, 0xa5, 0x81, 0xeb, 0xb8, 0x80, 0x55, 0xa4,
		0x76, 0xc2, 0x7f, 0x27, 0xf2, 0xe7, 0xc9, 0x7a,
		0x01, 0x3c, 0xd8, 0xc1, 0xde, 0x99, 0x1f, 0x7c,
		0xab, 0x35, 0x98, 0x00, 0x00, 0x28, 0x00, 0x39,
		0x00, 0x38, 0x00, 0x35, 0x00, 0x16, 0x00, 0x13,
		0x00, 0x0a, 0x00, 0x33, 0x00, 0x32, 0x00, 0x2f,
		0x00, 0x05, 0x00, 0x04, 0x00, 0x15, 0x00, 0x12,
		0x00, 0x09, 0x00, 0x14, 0x00, 0x11, 0x00, 0x08,
		0x00, 0x06, 0x00, 0x03, 0x00, 0xff, 0x02, 0x01,
		0x00, 0x00, 0x6c, 0x00, 0x23, 0x00, 0x68, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x65,
		0xe8, 0x4b, 0xd1, 0xef, 0xba, 0xfb, 0x41, 0x92,
		0x6d, 0x37, 0x5f, 0xf8, 0x7d, 0x90, 0x0f, 0x01,
		0xf8, 0x8c, 0xee, 0xbc, 0xd9, 0x0c, 0x97, 0x7e,
		0x23, 0x46, 0xe2, 0x6b, 0x52, 0xc6, 0xc6, 0x97,
		0x1d, 0xab, 0xde, 0xa0, 0x86, 0x94, 0xc8, 0x2e,
		0x8b, 0x2e, 0x42, 0x5f, 0xc2, 0x70, 0x35, 0xc9,
		0xee, 0x37, 0xeb, 0x70, 0xaa, 0x59, 0x23, 0x6c,
		0xc8, 0xc1, 0x84, 0x89, 0x39, 0x87, 0x73, 0x0a,
		0x7e, 0xba, 0xca, 0xed, 0x63, 0xba, 0x4e, 0x4f,
		0xf3, 0x31, 0x4b, 0xf0, 0xee, 0x91, 0xa5, 0xb4,
		0x62, 0x01, 0x9e, 0xbd, 0xbc, 0xb3, 0x35,
	},
	{
		0x16, 0x03, 0x01, 0x00, 0x2a, 0x02, 0x00, 0x00,
		0x26, 0x03, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x14,
		0x03, 0x01, 0x00, 0x01, 0x01, 0x16, 0x03, 0x01,
		0x00, 0x24, 0xc5, 0x35, 0x74, 0x19, 0x05, 0xc5,
		0x85, 0x68, 0x48, 0xe8, 0xb5, 0xe9, 0xaf, 0x78,
		0xbd, 0x35, 0x6f, 0xe9, 0x79, 0x34, 0x1b, 0xf0,
		0x35, 0xd4, 0x4e, 0x55, 0x2e, 0x3c, 0xd5, 0xaf,
		0xfc, 0xba, 0xf5, 0x1e, 0x83, 0x32,
	},
	{
		0x14, 0x03, 0x01, 0x00, 0x01, 0x01, 0x16, 0x03,
		0x01, 0x00, 0x24, 0x27, 0x28, 0x88, 0xe1, 0x7e,
		0x0d, 0x9c, 0x12, 0x50, 0xf6, 0x7a, 0xa7, 0x32,
		0x21, 0x68, 0xba, 0xd8, 0x0a, 0xdc, 0x39, 0xef,
		0x68, 0x95, 0x82, 0xae, 0xbd, 0x12, 0x79, 0xa1,
		0x99, 0xfd, 0xd0, 0x10, 0x8e, 0x4b, 0xd8,
	},
	{
		0x17, 0x03, 0x01, 0x00, 0x21, 0xc5, 0x7e, 0x0a,
		0x52, 0x6a, 0xb9, 0xaa, 0x1d, 0xae, 0x9e, 0x24,
		0x9c, 0x34, 0x1e, 0xdb, 0x50, 0x95, 0xee, 0x76,
		0xd7, 0x28, 0x88, 0x08, 0xe3, 0x2e, 0x58, 0xf7,
		0xdb, 0x34, 0x75, 0xa5, 0x7f, 0x9d, 0x15, 0x03,
		0x01, 0x00, 0x16, 0x2c, 0xc1, 0x29, 0x5f, 0x12,
		0x1d, 0x19, 0xab, 0xb3, 0xf4, 0x35, 0x1c, 0x62,
		0x6a, 0x80, 0x29, 0x0d, 0x0e, 0xef, 0x7d, 0x6e,
		0x50,
	},
}

var clientauthTests = []clientauthTest{
	// Server asks for cert with empty CA list, client doesn't give it.
	// go test -run "TestRunServer" -serve -clientauth 1
	{"RequestClientCert, none given", RequestClientCert, nil, [][]byte{
		{
			0x16, 0x03, 0x01, 0x00, 0x54, 0x01, 0x00, 0x00,
			0x50, 0x03, 0x01, 0x50, 0x77, 0x43, 0x9e, 0x31,
			0xe6, 0x36, 0x5e, 0x5e, 0x24, 0xe4, 0x0d, 0x26,
			0x34, 0xa7, 0x1c, 0x2e, 0x59, 0x6d, 0xa5, 0x3e,
			0x72, 0xf3, 0xa3, 0x1c, 0xbc, 0xb3, 0x27, 0xaf,
			0x92, 0x5b, 0x7d, 0x00, 0x00, 0x28, 0x00, 0x39,
			0x00, 0x38, 0x00, 0x35, 0x00, 0x16, 0x00, 0x13,
			0x00, 0x0a, 0x00, 0x33, 0x00, 0x32, 0x00, 0x2f,
			0x00, 0x05, 0x00, 0x04, 0x00, 0x15, 0x00, 0x12,
			0x00, 0x09, 0x00, 0x14, 0x00, 0x11, 0x00, 0x08,
			0x00, 0x06, 0x00, 0x03, 0x00, 0xff, 0x02, 0x01,
			0x00,
		},
		{
			0x16, 0x03, 0x01, 0x00, 0x2a, 0x02, 0x00, 0x00,
			0x26, 0x03, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x16,
			0x03, 0x01, 0x02, 0xbe, 0x0b, 0x00, 0x02, 0xba,
			0x00, 0x02, 0xb7, 0x00, 0x02, 0xb4, 0x30, 0x82,
			0x02, 0xb0, 0x30, 0x82, 0x02, 0x19, 0xa0, 0x03,
			0x02, 0x01, 0x02, 0x02, 0x09, 0x00, 0x85, 0xb0,
			0xbb, 0xa4, 0x8a, 0x7f, 0xb8, 0xca, 0x30, 0x0d,
			0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d,
			0x01, 0x01, 0x05, 0x05, 0x00, 0x30, 0x45, 0x31,
			0x0b, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06,
			0x13, 0x02, 0x41, 0x55, 0x31, 0x13, 0x30, 0x11,
			0x06, 0x03, 0x55, 0x04, 0x08, 0x13, 0x0a, 0x53,
			0x6f, 0x6d, 0x65, 0x2d, 0x53, 0x74, 0x61, 0x74,
			0x65, 0x31, 0x21, 0x30, 0x1f, 0x06, 0x03, 0x55,
			0x04, 0x0a, 0x13, 0x18, 0x49, 0x6e, 0x74, 0x65,
			0x72, 0x6e, 0x65, 0x74, 0x20, 0x57, 0x69, 0x64,
			0x67, 0x69, 0x74, 0x73, 0x20, 0x50, 0x74, 0x79,
			0x20, 0x4c, 0x74, 0x64, 0x30, 0x1e, 0x17, 0x0d,
			0x31, 0x30, 0x30, 0x34, 0x32, 0x34, 0x30, 0x39,
			0x30, 0x39, 0x33, 0x38, 0x5a, 0x17, 0x0d, 0x31,
			0x31, 0x30, 0x34, 0x32, 0x34, 0x30, 0x39, 0x30,
			0x39, 0x33, 0x38, 0x5a, 0x30, 0x45, 0x31, 0x0b,
			0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13,
			0x02, 0x41, 0x55, 0x31, 0x13, 0x30, 0x11, 0x06,
			0x03, 0x55, 0x04, 0x08, 0x13, 0x0a, 0x53, 0x6f,
			0x6d, 0x65, 0x2d, 0x53, 0x74, 0x61, 0x74, 0x65,
			0x31, 0x21, 0x30, 0x1f, 0x06, 0x03, 0x55, 0x04,
			0x0a, 0x13, 0x18, 0x49, 0x6e, 0x74, 0x65, 0x72,
			0x6e, 0x65, 0x74, 0x20, 0x57, 0x69, 0x64, 0x67,
			0x69, 0x74, 0x73, 0x20, 0x50, 0x74, 0x79, 0x20,
			0x4c, 0x74, 0x64, 0x30, 0x81, 0x9f, 0x30, 0x0d,
			0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d,
			0x01, 0x01, 0x01, 0x05, 0x00, 0x03, 0x81, 0x8d,
			0x00, 0x30, 0x81, 0x89, 0x02, 0x81, 0x81, 0x00,
			0xbb, 0x79, 0xd6, 0xf5, 0x17, 0xb5, 0xe5, 0xbf,
			0x46, 0x10, 0xd0, 0xdc, 0x69, 0xbe, 0xe6, 0x2b,
			0x07, 0x43, 0x5a, 0xd0, 0x03, 0x2d, 0x8a, 0x7a,
			0x43, 0x85, 0xb7, 0x14, 0x52, 0xe7, 0xa5, 0x65,
			0x4c, 0x2c, 0x78, 0xb8, 0x23, 0x8c, 0xb5, 0xb4,
			0x82, 0xe5, 0xde, 0x1f, 0x95, 0x3b, 0x7e, 0x62,
			0xa5, 0x2c, 0xa5, 0x33, 0xd6, 0xfe, 0x12, 0x5c,
			0x7a, 0x56, 0xfc, 0xf5, 0x06, 0xbf, 0xfa, 0x58,
			0x7b, 0x26, 0x3f, 0xb5, 0xcd, 0x04, 0xd3, 0xd0,
			0xc9, 0x21, 0x96, 0x4a, 0xc7, 0xf4, 0x54, 0x9f,
			0x5a, 0xbf, 0xef, 0x42, 0x71, 0x00, 0xfe, 0x18,
			0x99, 0x07, 0x7f, 0x7e, 0x88, 0x7d, 0x7d, 0xf1,
			0x04, 0x39, 0xc4, 0xa2, 0x2e, 0xdb, 0x51, 0xc9,
			0x7c, 0xe3, 0xc0, 0x4c, 0x3b, 0x32, 0x66, 0x01,
			0xcf, 0xaf, 0xb1, 0x1d, 0xb8, 0x71, 0x9a, 0x1d,
			0xdb, 0xdb, 0x89, 0x6b, 0xae, 0xda, 0x2d, 0x79,
			0x02, 0x03, 0x01, 0x00, 0x01, 0xa3, 0x81, 0xa7,
			0x30, 0x81, 0xa4, 0x30, 0x1d, 0x06, 0x03, 0x55,
			0x1d, 0x0e, 0x04, 0x16, 0x04, 0x14, 0xb1, 0xad,
			0xe2, 0x85, 0x5a, 0xcf, 0xcb, 0x28, 0xdb, 0x69,
			0xce, 0x23, 0x69, 0xde, 0xd3, 0x26, 0x8e, 0x18,
			0x88, 0x39, 0x30, 0x75, 0x06, 0x03, 0x55, 0x1d,
			0x23, 0x04, 0x6e, 0x30, 0x6c, 0x80, 0x14, 0xb1,
			0xad, 0xe2, 0x85, 0x5a, 0xcf, 0xcb, 0x28, 0xdb,
			0x69, 0xce, 0x23, 0x69, 0xde, 0xd3, 0x26, 0x8e,
			0x18, 0x88, 0x39, 0xa1, 0x49, 0xa4, 0x47, 0x30,
			0x45, 0x31, 0x0b, 0x30, 0x09, 0x06, 0x03, 0x55,
			0x04, 0x06, 0x13, 0x02, 0x41, 0x55, 0x31, 0x13,
			0x30, 0x11, 0x06, 0x03, 0x55, 0x04, 0x08, 0x13,
			0x0a, 0x53, 0x6f, 0x6d, 0x65, 0x2d, 0x53, 0x74,
			0x61, 0x74, 0x65, 0x31, 0x21, 0x30, 0x1f, 0x06,
			0x03, 0x55, 0x04, 0x0a, 0x13, 0x18, 0x49, 0x6e,
			0x74, 0x65, 0x72, 0x6e, 0x65, 0x74, 0x20, 0x57,
			0x69, 0x64, 0x67, 0x69, 0x74, 0x73, 0x20, 0x50,
			0x74, 0x79, 0x20, 0x4c, 0x74, 0x64, 0x82, 0x09,
			0x00, 0x85, 0xb0, 0xbb, 0xa4, 0x8a, 0x7f, 0xb8,
			0xca, 0x30, 0x0c, 0x06, 0x03, 0x55, 0x1d, 0x13,
			0x04, 0x05, 0x30, 0x03, 0x01, 0x01, 0xff, 0x30,
			0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7,
			0x0d, 0x01, 0x01, 0x05, 0x05, 0x00, 0x03, 0x81,
			0x81, 0x00, 0x08, 0x6c, 0x45, 0x24, 0xc7, 0x6b,
			0xb1, 0x59, 0xab, 0x0c, 0x52, 0xcc, 0xf2, 0xb0,
			0x14, 0xd7, 0x87, 0x9d, 0x7a, 0x64, 0x75, 0xb5,
			0x5a, 0x95, 0x66, 0xe4, 0xc5, 0x2b, 0x8e, 0xae,
			0x12, 0x66, 0x1f, 0xeb, 0x4f, 0x38, 0xb3, 0x6e,
			0x60, 0xd3, 0x92, 0xfd, 0xf7, 0x41, 0x08, 0xb5,
			0x25, 0x13, 0xb1, 0x18, 0x7a, 0x24, 0xfb, 0x30,
			0x1d, 0xba, 0xed, 0x98, 0xb9, 0x17, 0xec, 0xe7,
			0xd7, 0x31, 0x59, 0xdb, 0x95, 0xd3, 0x1d, 0x78,
			0xea, 0x50, 0x56, 0x5c, 0xd5, 0x82, 0x5a, 0x2d,
			0x5a, 0x5f, 0x33, 0xc4, 0xb6, 0xd8, 0xc9, 0x75,
			0x90, 0x96, 0x8c, 0x0f, 0x52, 0x98, 0xb5, 0xcd,
			0x98, 0x1f, 0x89, 0x20, 0x5f, 0xf2, 0xa0, 0x1c,
			0xa3, 0x1b, 0x96, 0x94, 0xdd, 0xa9, 0xfd, 0x57,
			0xe9, 0x70, 0xe8, 0x26, 0x6d, 0x71, 0x99, 0x9b,
			0x26, 0x6e, 0x38, 0x50, 0x29, 0x6c, 0x90, 0xa7,
			0xbd, 0xd9, 0x16, 0x03, 0x01, 0x00, 0x08, 0x0d,
			0x00, 0x00, 0x04, 0x01, 0x01, 0x00, 0x00, 0x16,
			0x03, 0x01, 0x00, 0x04, 0x0e, 0x00, 0x00, 0x00,
		},
		{
			0x16, 0x03, 0x01, 0x00, 0x07, 0x0b, 0x00, 0x00,
			0x03, 0x00, 0x00, 0x00, 0x16, 0x03, 0x01, 0x00,
			0x86, 0x10, 0x00, 0x00, 0x82, 0x00, 0x80, 0x04,
			0x58, 0x63, 0x26, 0x32, 0x1b, 0x34, 0xbe, 0x10,
			0xe4, 0xe4, 0x3e, 0xcd, 0x36, 0x7f, 0xa8, 0xa8,
			0xe0, 0x19, 0xe8, 0x94, 0x13, 0xd9, 0x35, 0xc4,
			0x71, 0xb4, 0x91, 0xd4, 0xbc, 0x74, 0x57, 0x9f,
			0x93, 0xb7, 0x5d, 0x3b, 0x9c, 0xff, 0x5d, 0x79,
			0xdb, 0x86, 0xfc, 0xdc, 0x74, 0x1e, 0x0c, 0xc6,
			0xe8, 0x93, 0xcf, 0xaf, 0xba, 0x1d, 0xfd, 0x8a,
			0xeb, 0xef, 0xbf, 0xfa, 0xa6, 0xe7, 0x53, 0x98,
			0x60, 0x4e, 0x0e, 0x60, 0x7d, 0xea, 0x40, 0x8d,
			0x1d, 0x8f, 0xa3, 0xc6, 0x83, 0xbc, 0xef, 0xb7,
			0x9a, 0x4a, 0xe7, 0x99, 0xee, 0x0b, 0xc7, 0x46,
			0x75, 0x45, 0x66, 0xe8, 0x5f, 0x4b, 0x08, 0xa4,
			0xc1, 0x36, 0xd0, 0x36, 0x2c, 0xf2, 0x9a, 0x44,
			0x1e, 0x5f, 0x22, 0xf4, 0xbe, 0x66, 0x66, 0x17,
			0xd8, 0xb6, 0x0a, 0x89, 0xed, 0x22, 0x80, 0xdb,
			0xad, 0x05, 0xd1, 0xb5, 0x93, 0xa1, 0x1c, 0x14,
			0x03, 0x01, 0x00, 0x01, 0x01, 0x16, 0x03, 0x01,
			0x00, 0x24, 0x62, 0x6f, 0x3d, 0x30, 0x56, 0x97,
			0xde, 0x03, 0x67, 0xa9, 0x63, 0x21, 0xb6, 0xe6,
			0x05, 0x69, 0x94, 0xfb, 0x50, 0xc1, 0x99, 0xdd,
			0xf6, 0xe8, 0x60, 0xbd, 0xe6, 0xba, 0xe3, 0x50,
			0x0a, 0xcd, 0xde, 0x14, 0x16, 0xc4,
		},
		{
			0x14, 0x03, 0x01, 0x00, 0x01, 0x01, 0x16, 0x03,
			0x01, 0x00, 0x24, 0xf0, 0x21, 0xf6, 0x84, 0x6a,
			0xe3, 0x6b, 0x8a, 0xc5, 0x46, 0x50, 0xca, 0x40,
			0xea, 0x4e, 0x82, 0xc1, 0x70, 0x25, 0xd8, 0x7d,
			0x60, 0xf5, 0x51, 0x7f, 0x64, 0x03, 0x9f, 0x53,
			0xec, 0xfb, 0x57, 0xa9, 0xfc, 0x26, 0x15, 0x17,
			0x03, 0x01, 0x00, 0x21, 0xa6, 0xc6, 0x94, 0x2b,
			0xa9, 0xcb, 0x93, 0xff, 0xb6, 0xa6, 0xe7, 0xc5,
			0x37, 0x86, 0x15, 0x37, 0x57, 0xce, 0xef, 0x54,
			0x96, 0x5d, 0x50, 0xa0, 0x50, 0x69, 0x5e, 0x82,
			0x61, 0x8d, 0x42, 0xfb, 0x78, 0x15, 0x03, 0x01,
			0x00, 0x16, 0x45, 0xd1, 0x86, 0x68, 0x59, 0xc1,
			0xaf, 0xac, 0x5c, 0x46, 0x8a, 0x68, 0x69, 0x0c,
			0xd7, 0x67, 0xbf, 0xf0, 0x3e, 0xee, 0x45, 0x55,
		},
	}},
	// Server asks for cert with empty CA list, client gives one
	// go test -run "TestRunServer" -serve -clientauth 1
	{"RequestClientCert, client gives it", RequestClientCert, []*x509.Certificate{clientCertificate}, [][]byte{
		{
			0x16, 0x03, 0x01, 0x00, 0x54, 0x01, 0x00, 0x00,
			0x50, 0x03, 0x01, 0x50, 0x77, 0x43, 0x47, 0xfd,
			0x1d, 0xb0, 0x60, 0x4c, 0x25, 0x86, 0x45, 0x4a,
			0xe5, 0x3f, 0x80, 0x56, 0x18, 0x91, 0x5c, 0xe2,
			0x62, 0xc5, 0x77, 0xc2, 0x92, 0xdd, 0xdc, 0x39,
			0x23, 0x1d, 0xc5, 0x00, 0x00, 0x28, 0x00, 0x39,
			0x00, 0x38, 0x00, 0x35, 0x00, 0x16, 0x00, 0x13,
			0x00, 0x0a, 0x00, 0x33, 0x00, 0x32, 0x00, 0x2f,
			0x00, 0x05, 0x00, 0x04, 0x00, 0x15, 0x00, 0x12,
			0x00, 0x09, 0x00, 0x14, 0x00, 0x11, 0x00, 0x08,
			0x00, 0x06, 0x00, 0x03, 0x00, 0xff, 0x02, 0x01,
			0x00,
		},
		{
			0x16, 0x03, 0x01, 0x00, 0x2a, 0x02, 0x00, 0x00,
			0x26, 0x03, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x16,
			0x03, 0x01, 0x02, 0xbe, 0x0b, 0x00, 0x02, 0xba,
			0x00, 0x02, 0xb7, 0x00, 0x02, 0xb4, 0x30, 0x82,
			0x02, 0xb0, 0x30, 0x82, 0x02, 0x19, 0xa0, 0x03,
			0x02, 0x01, 0x02, 0x02, 0x09, 0x00, 0x85, 0xb0,
			0xbb, 0xa4, 0x8a, 0x7f, 0xb8, 0xca, 0x30, 0x0d,
			0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d,
			0x01, 0x01, 0x05, 0x05, 0x00, 0x30, 0x45, 0x31,
			0x0b, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06,
			0x13, 0x02, 0x41, 0x55, 0x31, 0x13, 0x30, 0x11,
			0x06, 0x03, 0x55, 0x04, 0x08, 0x13, 0x0a, 0x53,
			0x6f, 0x6d, 0x65, 0x2d, 0x53, 0x74, 0x61, 0x74,
			0x65, 0x31, 0x21, 0x30, 0x1f, 0x06, 0x03, 0x55,
			0x04, 0x0a, 0x13, 0x18, 0x49, 0x6e, 0x74, 0x65,
			0x72, 0x6e, 0x65, 0x74, 0x20, 0x57, 0x69, 0x64,
			0x67, 0x69, 0x74, 0x73, 0x20, 0x50, 0x74, 0x79,
			0x20, 0x4c, 0x74, 0x64, 0x30, 0x1e, 0x17, 0x0d,
			0x31, 0x30, 0x30, 0x34, 0x32, 0x34, 0x30, 0x39,
			0x30, 0x39, 0x33, 0x38, 0x5a, 0x17, 0x0d, 0x31,
			0x31, 0x30, 0x34, 0x32, 0x34, 0x30, 0x39, 0x30,
			0x39, 0x33, 0x38, 0x5a, 0x30, 0x45, 0x31, 0x0b,
			0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06, 0x13,
			0x02, 0x41, 0x55, 0x31, 0x13, 0x30, 0x11, 0x06,
			0x03, 0x55, 0x04, 0x08, 0x13, 0x0a, 0x53, 0x6f,
			0x6d, 0x65, 0x2d, 0x53, 0x74, 0x61, 0x74, 0x65,
			0x31, 0x21, 0x30, 0x1f, 0x06, 0x03, 0x55, 0x04,
			0x0a, 0x13, 0x18, 0x49, 0x6e, 0x74, 0x65, 0x72,
			0x6e, 0x65, 0x74, 0x20, 0x57, 0x69, 0x64, 0x67,
			0x69, 0x74, 0x73, 0x20, 0x50, 0x74, 0x79, 0x20,
			0x4c, 0x74, 0x64, 0x30, 0x81, 0x9f, 0x30, 0x0d,
			0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d,
			0x01, 0x01, 0x01, 0x05, 0x00, 0x03, 0x81, 0x8d,
			0x00, 0x30, 0x81, 0x89, 0x02, 0x81, 0x81, 0x00,
			0xbb, 0x79, 0xd6, 0xf5, 0x17, 0xb5, 0xe5, 0xbf,
			0x46, 0x10, 0xd0, 0xdc, 0x69, 0xbe, 0xe6, 0x2b,
			0x07, 0x43, 0x5a, 0xd0, 0x03, 0x2d, 0x8a, 0x7a,
			0x43, 0x85, 0xb7, 0x14, 0x52, 0xe7, 0xa5, 0x65,
			0x4c, 0x2c, 0x78, 0xb8, 0x23, 0x8c, 0xb5, 0xb4,
			0x82, 0xe5, 0xde, 0x1f, 0x95, 0x3b, 0x7e, 0x62,
			0xa5, 0x2c, 0xa5, 0x33, 0xd6, 0xfe, 0x12, 0x5c,
			0x7a, 0x56, 0xfc, 0xf5, 0x06, 0xbf, 0xfa, 0x58,
			0x7b, 0x26, 0x3f, 0xb5, 0xcd, 0x04, 0xd3, 0xd0,
			0xc9, 0x21, 0x96, 0x4a, 0xc7, 0xf4, 0x54, 0x9f,
			0x5a, 0xbf, 0xef, 0x42, 0x71, 0x00, 0xfe, 0x18,
			0x99, 0x07, 0x7f, 0x7e, 0x88, 0x7d, 0x7d, 0xf1,
			0x04, 0x39, 0xc4, 0xa2, 0x2e, 0xdb, 0x51, 0xc9,
			0x7c, 0xe3, 0xc0, 0x4c, 0x3b, 0x32, 0x66, 0x01,
			0xcf, 0xaf, 0xb1, 0x1d, 0xb8, 0x71, 0x9a, 0x1d,
			0xdb, 0xdb, 0x89, 0x6b, 0xae, 0xda, 0x2d, 0x79,
			0x02, 0x03, 0x01, 0x00, 0x01, 0xa3, 0x81, 0xa7,
			0x30, 0x81, 0xa4, 0x30, 0x1d, 0x06, 0x03, 0x55,
			0x1d, 0x0e, 0x04, 0x16, 0x04, 0x14, 0xb1, 0xad,
			0xe2, 0x85, 0x5a, 0xcf, 0xcb, 0x28, 0xdb, 0x69,
			0xce, 0x23, 0x69, 0xde, 0xd3, 0x26, 0x8e, 0x18,
			0x88, 0x39, 0x30, 0x75, 0x06, 0x03, 0x55, 0x1d,
			0x23, 0x04, 0x6e, 0x30, 0x6c, 0x80, 0x14, 0xb1,
			0xad, 0xe2, 0x85, 0x5a, 0xcf, 0xcb, 0x28, 0xdb,
			0x69, 0xce, 0x23, 0x69, 0xde, 0xd3, 0x26, 0x8e,
			0x18, 0x88, 0x39, 0xa1, 0x49, 0xa4, 0x47, 0x30,
			0x45, 0x31, 0x0b, 0x30, 0x09, 0x06, 0x03, 0x55,
			0x04, 0x06, 0x13, 0x02, 0x41, 0x55, 0x31, 0x13,
			0x30, 0x11, 0x06, 0x03, 0x55, 0x04, 0x08, 0x13,
			0x0a, 0x53, 0x6f, 0x6d, 0x65, 0x2d, 0x53, 0x74,
			0x61, 0x74, 0x65, 0x31, 0x21, 0x30, 0x1f, 0x06,
			0x03, 0x55, 0x04, 0x0a, 0x13, 0x18, 0x49, 0x6e,
			0x74, 0x65, 0x72, 0x6e, 0x65, 0x74, 0x20, 0x57,
			0x69, 0x64, 0x67, 0x69, 0x74, 0x73, 0x20, 0x50,
			0x74, 0x79, 0x20, 0x4c, 0x74, 0x64, 0x82, 0x09,
			0x00, 0x85, 0xb0, 0xbb, 0xa4, 0x8a, 0x7f, 0xb8,
			0xca, 0x30, 0x0c, 0x06, 0x03, 0x55, 0x1d, 0x13,
			0x04, 0x05, 0x30, 0x03, 0x01, 0x01, 0xff, 0x30,
			0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7,
			0x0d, 0x01, 0x01, 0x05, 0x05, 0x00, 0x03, 0x81,
			0x81, 0x00, 0x08, 0x6c, 0x45, 0x24, 0xc7, 0x6b,
			0xb1, 0x59, 0xab, 0x0c, 0x52, 0xcc, 0xf2, 0xb0,
			0x14, 0xd7, 0x87, 0x9d, 0x7a, 0x64, 0x75, 0xb5,
			0x5a, 0x95, 0x66, 0xe4, 0xc5, 0x2b, 0x8e, 0xae,
			0x12, 0x66, 0x1f, 0xeb, 0x4f, 0x38, 0xb3, 0x6e,
			0x60, 0xd3, 0x92, 0xfd, 0xf7, 0x41, 0x08, 0xb5,
			0x25, 0x13, 0xb1, 0x18, 0x7a, 0x24, 0xfb, 0x30,
			0x1d, 0xba, 0xed, 0x98, 0xb9, 0x17, 0xec, 0xe7,
			0xd7, 0x31, 0x59, 0xdb, 0x95, 0xd3, 0x1d, 0x78,
			0xea, 0x50, 0x56, 0x5c, 0xd5, 0x82, 0x5a, 0x2d,
			0x5a, 0x5f, 0x33, 0xc4, 0xb6, 0xd8, 0xc9, 0x75,
			0x90, 0x96, 0x8c, 0x0f, 0x52, 0x98, 0xb5, 0xcd,
			0x98, 0x1f, 0x89, 0x20, 0x5f, 0xf2, 0xa0, 0x1c,
			0xa3, 0x1b, 0x96, 0x94, 0xdd, 0xa9, 0xfd, 0x57,
			0xe9, 0x70, 0xe8, 0x26, 0x6d, 0x71, 0x99, 0x9b,
			0x26, 0x6e, 0x38, 0x50, 0x29, 0x6c, 0x90, 0xa7,
			0xbd, 0xd9, 0x16, 0x03, 0x01, 0x00, 0x08, 0x0d,
			0x00, 0x00, 0x04, 0x01, 0x01, 0x00, 0x00, 0x16,
			0x03, 0x01, 0x00, 0x04, 0x0e, 0x00, 0x00, 0x00,
		},
		{
			0x16, 0x03, 0x01, 0x01, 0xfb, 0x0b, 0x00, 0x01,
			0xf7, 0x00, 0x01, 0xf4, 0x00, 0x01, 0xf1, 0x30,
			0x82, 0x01, 0xed, 0x30, 0x82, 0x01, 0x58, 0xa0,
			0x03, 0x02, 0x01, 0x02, 0x02, 0x01, 0x00, 0x30,
			0x0b, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7,
			0x0d, 0x01, 0x01, 0x05, 0x30, 0x26, 0x31, 0x10,
			0x30, 0x0e, 0x06, 0x03, 0x55, 0x04, 0x0a, 0x13,
			0x07, 0x41, 0x63, 0x6d, 0x65, 0x20, 0x43, 0x6f,
			0x31, 0x12, 0x30, 0x10, 0x06, 0x03, 0x55, 0x04,
			0x03, 0x13, 0x09, 0x31, 0x32, 0x37, 0x2e, 0x30,
			0x2e, 0x30, 0x2e, 0x31, 0x30, 0x1e, 0x17, 0x0d,
			0x31, 0x31, 0x31, 0x32, 0x30, 0x38, 0x30, 0x37,
			0x35, 0x35, 0x31, 0x32, 0x5a, 0x17, 0x0d, 0x31,
			0x32, 0x31, 0x32, 0x30, 0x37, 0x30, 0x38, 0x30,
			0x30, 0x31, 0x32, 0x5a, 0x30, 0x26, 0x31, 0x10,
			0x30, 0x0e, 0x06, 0x03, 0x55, 0x04, 0x0a, 0x13,
			0x07, 0x41, 0x63, 0x6d, 0x65, 0x20, 0x43, 0x6f,
			0x31, 0x12, 0x30, 0x10, 0x06, 0x03, 0x55, 0x04,
			0x03, 0x13, 0x09, 0x31, 0x32, 0x37, 0x2e, 0x30,
			0x2e, 0x30, 0x2e, 0x31, 0x30, 0x81, 0x9c, 0x30,
			0x0b, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7,
			0x0d, 0x01, 0x01, 0x01, 0x03, 0x81, 0x8c, 0x00,
			0x30, 0x81, 0x88, 0x02, 0x81, 0x80, 0x4e, 0xd0,
			0x7b, 0x31, 0xe3, 0x82, 0x64, 0xd9, 0x59, 0xc0,
			0xc2, 0x87, 0xa4, 0x5e, 0x1e, 0x8b, 0x73, 0x33,
			0xc7, 0x63, 0x53, 0xdf, 0x66, 0x92, 0x06, 0x84,
			0xf6, 0x64, 0xd5, 0x8f, 0xe4, 0x36, 0xa7, 0x1d,
			0x2b, 0xe8, 0xb3, 0x20, 0x36, 0x45, 0x23, 0xb5,
			0xe3, 0x95, 0xae, 0xed, 0xe0, 0xf5, 0x20, 0x9c,
			0x8d, 0x95, 0xdf, 0x7f, 0x5a, 0x12, 0xef, 0x87,
			0xe4, 0x5b, 0x68, 0xe4, 0xe9, 0x0e, 0x74, 0xec,
			0x04, 0x8a, 0x7f, 0xde, 0x93, 0x27, 0xc4, 0x01,
			0x19, 0x7a, 0xbd, 0xf2, 0xdc, 0x3d, 0x14, 0xab,
			0xd0, 0x54, 0xca, 0x21, 0x0c, 0xd0, 0x4d, 0x6e,
			0x87, 0x2e, 0x5c, 0xc5, 0xd2, 0xbb, 0x4d, 0x4b,
			0x4f, 0xce, 0xb6, 0x2c, 0xf7, 0x7e, 0x88, 0xec,
			0x7c, 0xd7, 0x02, 0x91, 0x74, 0xa6, 0x1e, 0x0c,
			0x1a, 0xda, 0xe3, 0x4a, 0x5a, 0x2e, 0xde, 0x13,
			0x9c, 0x4c, 0x40, 0x88, 0x59, 0x93, 0x02, 0x03,
			0x01, 0x00, 0x01, 0xa3, 0x32, 0x30, 0x30, 0x30,
			0x0e, 0x06, 0x03, 0x55, 0x1d, 0x0f, 0x01, 0x01,
			0xff, 0x04, 0x04, 0x03, 0x02, 0x00, 0xa0, 0x30,
			0x0d, 0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04, 0x06,
			0x04, 0x04, 0x01, 0x02, 0x03, 0x04, 0x30, 0x0f,
			0x06, 0x03, 0x55, 0x1d, 0x23, 0x04, 0x08, 0x30,
			0x06, 0x80, 0x04, 0x01, 0x02, 0x03, 0x04, 0x30,
			0x0b, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7,
			0x0d, 0x01, 0x01, 0x05, 0x03, 0x81, 0x81, 0x00,
			0x36, 0x1f, 0xb3, 0x7a, 0x0c, 0x75, 0xc9, 0x6e,
			0x37, 0x46, 0x61, 0x2b, 0xd5, 0xbd, 0xc0, 0xa7,
			0x4b, 0xcc, 0x46, 0x9a, 0x81, 0x58, 0x7c, 0x85,
			0x79, 0x29, 0xc8, 0xc8, 0xc6, 0x67, 0xdd, 0x32,
			0x56, 0x45, 0x2b, 0x75, 0xb6, 0xe9, 0x24, 0xa9,
			0x50, 0x9a, 0xbe, 0x1f, 0x5a, 0xfa, 0x1a, 0x15,
			0xd9, 0xcc, 0x55, 0x95, 0x72, 0x16, 0x83, 0xb9,
			0xc2, 0xb6, 0x8f, 0xfd, 0x88, 0x8c, 0x38, 0x84,
			0x1d, 0xab, 0x5d, 0x92, 0x31, 0x13, 0x4f, 0xfd,
			0x83, 0x3b, 0xc6, 0x9d, 0xf1, 0x11, 0x62, 0xb6,
			0x8b, 0xec, 0xab, 0x67, 0xbe, 0xc8, 0x64, 0xb0,
			0x11, 0x50, 0x46, 0x58, 0x17, 0x6b, 0x99, 0x1c,
			0xd3, 0x1d, 0xfc, 0x06, 0xf1, 0x0e, 0xe5, 0x96,
			0xa8, 0x0c, 0xf9, 0x78, 0x20, 0xb7, 0x44, 0x18,
			0x51, 0x8d, 0x10, 0x7e, 0x4f, 0x94, 0x67, 0xdf,
			0xa3, 0x4e, 0x70, 0x73, 0x8e, 0x90, 0x91, 0x85,
			0x16, 0x03, 0x01, 0x00, 0x86, 0x10, 0x00, 0x00,
			0x82, 0x00, 0x80, 0x81, 0x46, 0x43, 0xf9, 0xe7,
			0xda, 0x8c, 0x92, 0x3a, 0x78, 0x1a, 0x86, 0xb3,
			0xbe, 0x83, 0x22, 0xb6, 0xaa, 0x57, 0x37, 0x68,
			0x9e, 0x54, 0x3f, 0xd3, 0xce, 0x4d, 0x5e, 0x2a,
			0xdc, 0xb0, 0x49, 0x02, 0xbb, 0xc0, 0x45, 0x58,
			0x79, 0x10, 0xc7, 0x94, 0x60, 0x9f, 0x1b, 0x5f,
			0x18, 0x31, 0x37, 0x9c, 0xe0, 0xe6, 0xdf, 0x5e,
			0x70, 0x44, 0xf6, 0x8b, 0xdf, 0xf1, 0xf6, 0x43,
			0xc8, 0x2f, 0xd1, 0xce, 0xd0, 0xd6, 0x64, 0x4f,
			0xe8, 0x2b, 0xfa, 0xd3, 0xd0, 0xd1, 0x2e, 0xaa,
			0x9b, 0x1d, 0x13, 0x5c, 0xbe, 0x57, 0x41, 0x6c,
			0x5e, 0x8d, 0xea, 0xa9, 0x3c, 0x58, 0xa0, 0x30,
			0x92, 0x77, 0x7a, 0xed, 0x64, 0x58, 0xe5, 0x7f,
			0x6a, 0x93, 0x89, 0x66, 0x3d, 0x13, 0x16, 0x56,
			0xa0, 0xad, 0xdc, 0x68, 0x95, 0x87, 0x81, 0xd0,
			0x90, 0x4d, 0x5f, 0xfe, 0x3e, 0x83, 0x15, 0x2e,
			0x50, 0x3c, 0xdd, 0x16, 0x03, 0x01, 0x00, 0x86,
			0x0f, 0x00, 0x00, 0x82, 0x00, 0x80, 0x2b, 0xf8,
			0x56, 0x48, 0xbb, 0x02, 0x37, 0x15, 0x02, 0x74,
			0x33, 0x53, 0x65, 0xa7, 0x7c, 0x2f, 0xc6, 0x5d,
			0x80, 0x59, 0xc1, 0xc2, 0x3b, 0xa9, 0xde, 0x4e,
			0x70, 0x51, 0xd2, 0xde, 0x58, 0x7f, 0xd8, 0xb9,
			0xb6, 0x3b, 0xc8, 0xaa, 0xfc, 0x3d, 0x53, 0x2d,
			0x61, 0x4d, 0xf5, 0x60, 0x12, 0xc2, 0xa5, 0x39,
			0x0c, 0xa7, 0xc6, 0xac, 0x26, 0x4b, 0xf4, 0x5f,
			0xe9, 0xf4, 0xf2, 0x73, 0x48, 0xe4, 0x3b, 0xee,
			0xf2, 0xee, 0xc0, 0xee, 0xfb, 0x5b, 0x60, 0xc2,
			0x74, 0xe6, 0xf6, 0x43, 0x3e, 0xa4, 0xf7, 0x97,
			0x3d, 0xfc, 0xe9, 0x44, 0x21, 0x18, 0x46, 0x05,
			0x33, 0xf8, 0xfe, 0x35, 0x5b, 0xe6, 0x8f, 0xef,
			0x4d, 0x4c, 0x87, 0xf6, 0xb4, 0x6e, 0x6b, 0x39,
			0xd8, 0xaa, 0x1b, 0x33, 0xc9, 0x1c, 0x66, 0x48,
			0xbe, 0xfa, 0xb5, 0x92, 0x09, 0xfd, 0xb9, 0xb9,
			0xca, 0xe6, 0x6d, 0x71, 0xc6, 0x89, 0x14, 0x03,
			0x01, 0x00, 0x01, 0x01, 0x16, 0x03, 0x01, 0x00,
			0x24, 0xe3, 0x2b, 0xef, 0x17, 0xd5, 0xa6, 0x4c,
			0x2e, 0x10, 0xac, 0x9c, 0xfe, 0x0f, 0x18, 0x43,
			0x95, 0x00, 0x81, 0xf7, 0x7c, 0x00, 0x5b, 0x89,
			0x52, 0x41, 0xe4, 0x8a, 0x8a, 0x34, 0x31, 0x09,
			0x48, 0x7c, 0xc5, 0xc3, 0x83,
		},
		{
			0x14, 0x03, 0x01, 0x00, 0x01, 0x01, 0x16, 0x03,
			0x01, 0x00, 0x24, 0x24, 0xaa, 0xaa, 0x56, 0x8b,
			0x41, 0x87, 0x01, 0xbe, 0x80, 0x05, 0x51, 0x36,
			0x08, 0xfc, 0xaf, 0xff, 0x7f, 0xf4, 0x74, 0x84,
			0x88, 0xdc, 0xb8, 0x8e, 0x70, 0x6c, 0x22, 0x04,
			0xee, 0x45, 0x8d, 0xda, 0xed, 0xc6, 0x05, 0x17,
			0x03, 0x01, 0x00, 0x21, 0x91, 0x49, 0x4b, 0xed,
			0xa3, 0x41, 0xe9, 0x88, 0x3b, 0xa3, 0x01, 0xee,
			0x77, 0x4e, 0x12, 0xb4, 0xcd, 0x5e, 0xcc, 0x45,
			0x02, 0x5a, 0x20, 0xd6, 0xe8, 0xac, 0xcb, 0x60,
			0xcb, 0x1b, 0xef, 0xf9, 0xc2, 0x15, 0x03, 0x01,
			0x00, 0x16, 0xd4, 0xcd, 0x92, 0x3c, 0x10, 0x93,
			0x68, 0xc3, 0xdd, 0xaf, 0xe9, 0xcb, 0x5d, 0x94,
			0x1a, 0x06, 0x81, 0xa7, 0x78, 0x0f, 0xc3, 0x03,
		},
	}},
}

// cert.pem and key.pem were generated with generate_cert.go
// Thus, they have no ExtKeyUsage fields and trigger an error
// when verification is turned on.

var clientCertificate = loadPEMCert(`
-----BEGIN CERTIFICATE-----
MIIB7TCCAVigAwIBAgIBADALBgkqhkiG9w0BAQUwJjEQMA4GA1UEChMHQWNtZSBD
bzESMBAGA1UEAxMJMTI3LjAuMC4xMB4XDTExMTIwODA3NTUxMloXDTEyMTIwNzA4
MDAxMlowJjEQMA4GA1UEChMHQWNtZSBDbzESMBAGA1UEAxMJMTI3LjAuMC4xMIGc
MAsGCSqGSIb3DQEBAQOBjAAwgYgCgYBO0Hsx44Jk2VnAwoekXh6LczPHY1PfZpIG
hPZk1Y/kNqcdK+izIDZFI7Xjla7t4PUgnI2V339aEu+H5Fto5OkOdOwEin/ekyfE
ARl6vfLcPRSr0FTKIQzQTW6HLlzF0rtNS0/Otiz3fojsfNcCkXSmHgwa2uNKWi7e
E5xMQIhZkwIDAQABozIwMDAOBgNVHQ8BAf8EBAMCAKAwDQYDVR0OBAYEBAECAwQw
DwYDVR0jBAgwBoAEAQIDBDALBgkqhkiG9w0BAQUDgYEANh+zegx1yW43RmEr1b3A
p0vMRpqBWHyFeSnIyMZn3TJWRSt1tukkqVCavh9a+hoV2cxVlXIWg7nCto/9iIw4
hB2rXZIxE0/9gzvGnfERYraL7KtnvshksBFQRlgXa5kc0x38BvEO5ZaoDPl4ILdE
GFGNEH5PlGffo05wc46QkYU=
-----END CERTIFICATE-----
`)

/* corresponding key.pem for cert.pem is:
-----BEGIN RSA PRIVATE KEY-----
MIICWgIBAAKBgE7QezHjgmTZWcDCh6ReHotzM8djU99mkgaE9mTVj+Q2px0r6LMg
NkUjteOVru3g9SCcjZXff1oS74fkW2jk6Q507ASKf96TJ8QBGXq98tw9FKvQVMoh
DNBNbocuXMXSu01LT862LPd+iOx81wKRdKYeDBra40paLt4TnExAiFmTAgMBAAEC
gYBxvXd8yNteFTns8A/2yomEMC4yeosJJSpp1CsN3BJ7g8/qTnrVPxBy+RU+qr63
t2WquaOu/cr5P8iEsa6lk20tf8pjKLNXeX0b1RTzK8rJLbS7nGzP3tvOhL096VtQ
dAo4ROEaro0TzYpHmpciSvxVIeEIAAdFDObDJPKqcJAxyQJBAJizfYgK8Gzx9fsx
hxp+VteCbVPg2euASH5Yv3K5LukRdKoSzHE2grUVQgN/LafC0eZibRanxHegYSr7
7qaswKUCQQCEIWor/X4XTMdVj3Oj+vpiw75y/S9gh682+myZL+d/02IEkwnB098P
RkKVpenBHyrGg0oeN5La7URILWKj7CPXAkBKo6F+d+phNjwIFoN1Xb/RA32w/D1I
saG9sF+UEhRt9AxUfW/U/tIQ9V0ZHHcSg1XaCM5Nvp934brdKdvTOKnJAkBD5h/3
Rybatlvg/fzBEaJFyq09zhngkxlZOUtBVTqzl17RVvY2orgH02U4HbCHy4phxOn7
qTdQRYlHRftgnWK1AkANibn9PRYJ7mJyJ9Dyj2QeNcSkSTzrt0tPvUMf4+meJymN
1Ntu5+S1DLLzfxlaljWG6ylW6DNxujCyuXIV2rvA
-----END RSA PRIVATE KEY-----
*/
