/* Command line option handling.
   Copyright (C) 2002, 2003, 2004, 2005, 2006, 2007, 2008, 2009, 2010
   Free Software Foundation, Inc.
   Contributed by Neil Booth.

This file is part of GCC.

GCC is free software; you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by the Free
Software Foundation; either version 3, or (at your option) any later
version.

GCC is distributed in the hope that it will be useful, but WITHOUT ANY
WARRANTY; without even the implied warranty of MERCHANTABILITY or
FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License
along with GCC; see the file COPYING3.  If not see
<http://www.gnu.org/licenses/>.  */

#include "config.h"
#include "system.h"
#include "intl.h"
#include "coretypes.h"
#include "tm.h"
#include "tree.h"
#include "rtl.h"
#include "expr.h"
#include "ggc.h"
#include "output.h"
#include "langhooks.h"
#include "opts.h"
#include "options.h"
#include "flags.h"
#include "toplev.h"
#include "params.h"
#include "diagnostic.h"
#include "opts-diagnostic.h"
#include "insn-attr.h"		/* For INSN_SCHEDULING.  */
#include "target.h"
#include "tree-pass.h"
#include "dbgcnt.h"
#include "debug.h"
#include "plugin.h"
#include "except.h"
#include "lto-streamer.h"

/* True if we should exit after parsing options.  */
bool exit_after_options;

/* Type(s) of debugging information we are producing (if any).  See
   flags.h for the definitions of the different possible types of
   debugging information.  */
enum debug_info_type write_symbols = NO_DEBUG;

/* Level of debugging information we are producing.  See flags.h for
   the definitions of the different possible levels.  */
enum debug_info_level debug_info_level = DINFO_LEVEL_NONE;

/* A major contribution to object and executable size is debug
   information size.  A major contribution to debug information size
   is struct descriptions replicated in several object files. The
   following flags attempt to reduce this information.  The basic
   idea is to not emit struct debugging information in the current
   compilation unit when that information will be generated by
   another compilation unit.

   Debug information for a struct defined in the current source
   file should be generated in the object file.  Likewise the
   debug information for a struct defined in a header should be
   generated in the object file of the corresponding source file.
   Both of these case are handled when the base name of the file of
   the struct definition matches the base name of the source file
   of the current compilation unit.  This matching emits minimal
   struct debugging information.

   The base file name matching rule above will fail to emit debug
   information for structs defined in system headers.  So a second
   category of files includes system headers in addition to files
   with matching bases.

   The remaining types of files are library headers and application
   headers.  We cannot currently distinguish these two types.  */

enum debug_struct_file
{
  DINFO_STRUCT_FILE_NONE,   /* Debug no structs. */
  DINFO_STRUCT_FILE_BASE,   /* Debug structs defined in files with the
                               same base name as the compilation unit. */
  DINFO_STRUCT_FILE_SYS,    /* Also debug structs defined in system
                               header files.  */
  DINFO_STRUCT_FILE_ANY     /* Debug structs defined in all files. */
};

/* Generic structs (e.g. templates not explicitly specialized)
   may not have a compilation unit associated with them, and so
   may need to be treated differently from ordinary structs.

   Structs only handled by reference (indirectly), will also usually
   not need as much debugging information.  */

static enum debug_struct_file debug_struct_ordinary[DINFO_USAGE_NUM_ENUMS]
  = { DINFO_STRUCT_FILE_ANY, DINFO_STRUCT_FILE_ANY, DINFO_STRUCT_FILE_ANY };
static enum debug_struct_file debug_struct_generic[DINFO_USAGE_NUM_ENUMS]
  = { DINFO_STRUCT_FILE_ANY, DINFO_STRUCT_FILE_ANY, DINFO_STRUCT_FILE_ANY };

/* Run the second compilation of -fcompare-debug.  Not defined using
   Var in common.opt because this is used in Ada code and so must be
   an actual variable not a macro.  */
int flag_compare_debug;

/* Parse the -femit-struct-debug-detailed option value
   and set the flag variables. */

#define MATCH( prefix, string ) \
  ((strncmp (prefix, string, sizeof prefix - 1) == 0) \
   ? ((string += sizeof prefix - 1), 1) : 0)

void
set_struct_debug_option (const char *spec)
{
  /* various labels for comparison */
  static char dfn_lbl[] = "dfn:", dir_lbl[] = "dir:", ind_lbl[] = "ind:";
  static char ord_lbl[] = "ord:", gen_lbl[] = "gen:";
  static char none_lbl[] = "none", any_lbl[] = "any";
  static char base_lbl[] = "base", sys_lbl[] = "sys";

  enum debug_struct_file files = DINFO_STRUCT_FILE_ANY;
  /* Default is to apply to as much as possible. */
  enum debug_info_usage usage = DINFO_USAGE_NUM_ENUMS;
  int ord = 1, gen = 1;

  /* What usage? */
  if (MATCH (dfn_lbl, spec))
    usage = DINFO_USAGE_DFN;
  else if (MATCH (dir_lbl, spec))
    usage = DINFO_USAGE_DIR_USE;
  else if (MATCH (ind_lbl, spec))
    usage = DINFO_USAGE_IND_USE;

  /* Generics or not? */
  if (MATCH (ord_lbl, spec))
    gen = 0;
  else if (MATCH (gen_lbl, spec))
    ord = 0;

  /* What allowable environment? */
  if (MATCH (none_lbl, spec))
    files = DINFO_STRUCT_FILE_NONE;
  else if (MATCH (any_lbl, spec))
    files = DINFO_STRUCT_FILE_ANY;
  else if (MATCH (sys_lbl, spec))
    files = DINFO_STRUCT_FILE_SYS;
  else if (MATCH (base_lbl, spec))
    files = DINFO_STRUCT_FILE_BASE;
  else
    error ("argument %qs to %<-femit-struct-debug-detailed%> not recognized",
           spec);

  /* Effect the specification. */
  if (usage == DINFO_USAGE_NUM_ENUMS)
    {
      if (ord)
        {
          debug_struct_ordinary[DINFO_USAGE_DFN] = files;
          debug_struct_ordinary[DINFO_USAGE_DIR_USE] = files;
          debug_struct_ordinary[DINFO_USAGE_IND_USE] = files;
        }
      if (gen)
        {
          debug_struct_generic[DINFO_USAGE_DFN] = files;
          debug_struct_generic[DINFO_USAGE_DIR_USE] = files;
          debug_struct_generic[DINFO_USAGE_IND_USE] = files;
        }
    }
  else
    {
      if (ord)
        debug_struct_ordinary[usage] = files;
      if (gen)
        debug_struct_generic[usage] = files;
    }

  if (*spec == ',')
    set_struct_debug_option (spec+1);
  else
    {
      /* No more -femit-struct-debug-detailed specifications.
         Do final checks. */
      if (*spec != '\0')
	error ("argument %qs to %<-femit-struct-debug-detailed%> unknown",
               spec);
      if (debug_struct_ordinary[DINFO_USAGE_DIR_USE]
		< debug_struct_ordinary[DINFO_USAGE_IND_USE]
	  || debug_struct_generic[DINFO_USAGE_DIR_USE]
		< debug_struct_generic[DINFO_USAGE_IND_USE])
	error ("%<-femit-struct-debug-detailed=dir:...%> must allow at least"
               " as much as %<-femit-struct-debug-detailed=ind:...%>");
    }
}

/* Find the base name of a path, stripping off both directories and
   a single final extension. */
static int
base_of_path (const char *path, const char **base_out)
{
  const char *base = path;
  const char *dot = 0;
  const char *p = path;
  char c = *p;
  while (c)
    {
      if (IS_DIR_SEPARATOR(c))
        {
          base = p + 1;
          dot = 0;
        }
      else if (c == '.')
        dot = p;
      c = *++p;
    }
  if (!dot)
    dot = p;
  *base_out = base;
  return dot - base;
}

/* Match the base name of a file to the base name of a compilation unit. */

static const char *main_input_basename;
static int main_input_baselength;

static int
matches_main_base (const char *path)
{
  /* Cache the last query. */
  static const char *last_path = NULL;
  static int last_match = 0;
  if (path != last_path)
    {
      const char *base;
      int length = base_of_path (path, &base);
      last_path = path;
      last_match = (length == main_input_baselength
                    && memcmp (base, main_input_basename, length) == 0);
    }
  return last_match;
}

#ifdef DEBUG_DEBUG_STRUCT

static int
dump_struct_debug (tree type, enum debug_info_usage usage,
		   enum debug_struct_file criterion, int generic,
		   int matches, int result)
{
  /* Find the type name. */
  tree type_decl = TYPE_STUB_DECL (type);
  tree t = type_decl;
  const char *name = 0;
  if (TREE_CODE (t) == TYPE_DECL)
    t = DECL_NAME (t);
  if (t)
    name = IDENTIFIER_POINTER (t);

  fprintf (stderr, "	struct %d %s %s %s %s %d %p %s\n",
	   criterion,
           DECL_IN_SYSTEM_HEADER (type_decl) ? "sys" : "usr",
           matches ? "bas" : "hdr",
           generic ? "gen" : "ord",
           usage == DINFO_USAGE_DFN ? ";" :
             usage == DINFO_USAGE_DIR_USE ? "." : "*",
           result,
           (void*) type_decl, name);
  return result;
}
#define DUMP_GSTRUCT(type, usage, criterion, generic, matches, result) \
  dump_struct_debug (type, usage, criterion, generic, matches, result)

#else

#define DUMP_GSTRUCT(type, usage, criterion, generic, matches, result) \
  (result)

#endif


bool
should_emit_struct_debug (tree type, enum debug_info_usage usage)
{
  enum debug_struct_file criterion;
  tree type_decl;
  bool generic = lang_hooks.types.generic_p (type);

  if (generic)
    criterion = debug_struct_generic[usage];
  else
    criterion = debug_struct_ordinary[usage];

  if (criterion == DINFO_STRUCT_FILE_NONE)
    return DUMP_GSTRUCT (type, usage, criterion, generic, false, false);
  if (criterion == DINFO_STRUCT_FILE_ANY)
    return DUMP_GSTRUCT (type, usage, criterion, generic, false, true);

  type_decl = TYPE_STUB_DECL (type);

  if (criterion == DINFO_STRUCT_FILE_SYS && DECL_IN_SYSTEM_HEADER (type_decl))
    return DUMP_GSTRUCT (type, usage, criterion, generic, false, true);

  if (matches_main_base (DECL_SOURCE_FILE (type_decl)))
    return DUMP_GSTRUCT (type, usage, criterion, generic, true, true);
  return DUMP_GSTRUCT (type, usage, criterion, generic, false, false);
}

/* Nonzero means use GNU-only extensions in the generated symbolic
   debugging information.  Currently, this only has an effect when
   write_symbols is set to DBX_DEBUG, XCOFF_DEBUG, or DWARF_DEBUG.  */
bool use_gnu_debug_info_extensions;

/* Global visibility options.  */
struct visibility_flags visibility_options;

/* What to print when a switch has no documentation.  */
static const char undocumented_msg[] = N_("This switch lacks documentation");

/* Functions excluded from profiling.  */

typedef char *char_p; /* For DEF_VEC_P.  */
DEF_VEC_P(char_p);
DEF_VEC_ALLOC_P(char_p,heap);

static VEC(char_p,heap) *flag_instrument_functions_exclude_functions;
static VEC(char_p,heap) *flag_instrument_functions_exclude_files;

typedef const char *const_char_p; /* For DEF_VEC_P.  */
DEF_VEC_P(const_char_p);
DEF_VEC_ALLOC_P(const_char_p,heap);

static VEC(const_char_p,heap) *ignored_options;

/* Input file names.  */
const char **in_fnames;
unsigned num_in_fnames;

static bool common_handle_option (struct gcc_options *opts,
				  struct gcc_options *opts_set,
				  const struct cl_decoded_option *decoded,
				  unsigned int lang_mask, int kind,
				  location_t loc,
				  const struct cl_option_handlers *handlers,
				  diagnostic_context *dc);
static void handle_param (struct gcc_options *opts,
			  struct gcc_options *opts_set, const char *carg);
static char *write_langs (unsigned int lang_mask);
static void complain_wrong_lang (const struct cl_decoded_option *,
				 unsigned int lang_mask);
static void set_debug_level (enum debug_info_type type, int extended,
			     const char *arg);
static void set_fast_math_flags (struct gcc_options *opts, int set);
static void set_unsafe_math_optimizations_flags (struct gcc_options *opts,
						 int set);
static void enable_warning_as_error (const char *arg, int value,
				     unsigned int lang_mask,
				     const struct cl_option_handlers *handlers,
				     struct gcc_options *opts,
				     struct gcc_options *opts_set,
				     location_t loc,
				     diagnostic_context *dc);

/* Return a malloced slash-separated list of languages in MASK.  */
static char *
write_langs (unsigned int mask)
{
  unsigned int n = 0, len = 0;
  const char *lang_name;
  char *result;

  for (n = 0; (lang_name = lang_names[n]) != 0; n++)
    if (mask & (1U << n))
      len += strlen (lang_name) + 1;

  result = XNEWVEC (char, len);
  len = 0;
  for (n = 0; (lang_name = lang_names[n]) != 0; n++)
    if (mask & (1U << n))
      {
	if (len)
	  result[len++] = '/';
	strcpy (result + len, lang_name);
	len += strlen (lang_name);
      }

  result[len] = 0;

  return result;
}

/* Complain that switch DECODED does not apply to this front end (mask
   LANG_MASK).  */
static void
complain_wrong_lang (const struct cl_decoded_option *decoded,
		     unsigned int lang_mask)
{
  const struct cl_option *option = &cl_options[decoded->opt_index];
  const char *text = decoded->orig_option_with_args_text;
  char *ok_langs = NULL, *bad_lang = NULL;
  unsigned int opt_flags = option->flags;

  if (!lang_hooks.complain_wrong_lang_p (option))
    return;

  opt_flags &= ((1U << cl_lang_count) - 1) | CL_DRIVER;
  if (opt_flags != CL_DRIVER)
    ok_langs = write_langs (opt_flags);
  if (lang_mask != CL_DRIVER)
    bad_lang = write_langs (lang_mask);

  if (opt_flags == CL_DRIVER)
    error ("command line option %qs is valid for the driver but not for %s",
	   text, bad_lang);
  else if (lang_mask == CL_DRIVER)
    gcc_unreachable ();
  else
    /* Eventually this should become a hard error IMO.  */
    warning (0, "command line option %qs is valid for %s but not for %s",
	     text, ok_langs, bad_lang);

  free (ok_langs);
  free (bad_lang);
}

/* Buffer the unknown option described by the string OPT.  Currently,
   we only complain about unknown -Wno-* options if they may have
   prevented a diagnostic. Otherwise, we just ignore them.
   Note that if we do complain, it is only as a warning, not an error;
   passing the compiler an unrecognised -Wno-* option should never
   change whether the compilation succeeds or fails.  */

static void postpone_unknown_option_warning(const char *opt)
{
  VEC_safe_push (const_char_p, heap, ignored_options, opt);
}

/* Produce a warning for each option previously buffered.  */

void print_ignored_options (void)
{
  location_t saved_loc = input_location;

  input_location = 0;

  while (!VEC_empty (const_char_p, ignored_options))
    {
      const char *opt;
      opt = VEC_pop (const_char_p, ignored_options);
      warning (0, "unrecognized command line option \"%s\"", opt);
    }

  input_location = saved_loc;
}

/* Handle an unknown option DECODED, returning true if an error should be
   given.  */

static bool
unknown_option_callback (const struct cl_decoded_option *decoded)
{
  const char *opt = decoded->arg;

  if (opt[1] == 'W' && opt[2] == 'n' && opt[3] == 'o' && opt[4] == '-'
      && !(decoded->errors & CL_ERR_NEGATIVE))
    {
      /* We don't generate warnings for unknown -Wno-* options unless
	 we issue diagnostics.  */
      postpone_unknown_option_warning (opt);
      return false;
    }
  else
    return true;
}

/* Note that an option DECODED has been successfully handled with a
   handler for mask MASK.  */

static void
post_handling_callback (const struct cl_decoded_option *decoded ATTRIBUTE_UNUSED,
			unsigned int mask ATTRIBUTE_UNUSED)
{
#ifdef ENABLE_LTO
  lto_register_user_option (decoded->opt_index, decoded->arg,
			    decoded->value, mask);
#endif
}

/* Handle a front-end option; arguments and return value as for
   handle_option.  */

static bool
lang_handle_option (struct gcc_options *opts,
		    struct gcc_options *opts_set,
		    const struct cl_decoded_option *decoded,
		    unsigned int lang_mask ATTRIBUTE_UNUSED, int kind,
		    location_t loc,
		    const struct cl_option_handlers *handlers,
		    diagnostic_context *dc)
{
  gcc_assert (opts == &global_options);
  gcc_assert (opts_set == &global_options_set);
  gcc_assert (dc == global_dc);
  gcc_assert (decoded->canonical_option_num_elements <= 2);
  return lang_hooks.handle_option (decoded->opt_index, decoded->arg,
				   decoded->value, kind, loc, handlers);
}

/* Handle a back-end option; arguments and return value as for
   handle_option.  */

static bool
target_handle_option (struct gcc_options *opts,
		      struct gcc_options *opts_set,
		      const struct cl_decoded_option *decoded,
		      unsigned int lang_mask ATTRIBUTE_UNUSED, int kind,
		      location_t loc,
		      const struct cl_option_handlers *handlers ATTRIBUTE_UNUSED,
		      diagnostic_context *dc)
{
  gcc_assert (opts == &global_options);
  gcc_assert (opts_set == &global_options_set);
  gcc_assert (dc == global_dc);
  gcc_assert (decoded->canonical_option_num_elements <= 2);
  gcc_assert (kind == DK_UNSPECIFIED);
  gcc_assert (loc == UNKNOWN_LOCATION);
  return targetm.handle_option (decoded->opt_index, decoded->arg,
				decoded->value);
}

/* Handle FILENAME from the command line.  */
static void
add_input_filename (const char *filename)
{
  num_in_fnames++;
  in_fnames = XRESIZEVEC (const char *, in_fnames, num_in_fnames);
  in_fnames[num_in_fnames - 1] = filename;
}

/* Add comma-separated strings to a char_p vector.  */

static void
add_comma_separated_to_vector (VEC(char_p,heap) **pvec, const char* arg)
{
  char *tmp;
  char *r;
  char *w;
  char *token_start;

  /* We never free this string.  */
  tmp = xstrdup (arg);

  r = tmp;
  w = tmp;
  token_start = tmp;

  while (*r != '\0')
    {
      if (*r == ',')
	{
	  *w++ = '\0';
	  ++r;
	  VEC_safe_push (char_p, heap, *pvec, token_start);
	  token_start = w;
	}
      if (*r == '\\' && r[1] == ',')
	{
	  *w++ = ',';
	  r += 2;
	}
      else
	*w++ = *r++;
    }
  if (*token_start != '\0')
    VEC_safe_push (char_p, heap, *pvec, token_start);
}

/* Return whether we should exclude FNDECL from instrumentation.  */

bool
flag_instrument_functions_exclude_p (tree fndecl)
{
  if (VEC_length (char_p, flag_instrument_functions_exclude_functions) > 0)
    {
      const char *name;
      int i;
      char *s;

      name = lang_hooks.decl_printable_name (fndecl, 0);
      FOR_EACH_VEC_ELT (char_p, flag_instrument_functions_exclude_functions,
			i, s)
	if (strstr (name, s) != NULL)
	  return true;
    }

  if (VEC_length (char_p, flag_instrument_functions_exclude_files) > 0)
    {
      const char *name;
      int i;
      char *s;

      name = DECL_SOURCE_FILE (fndecl);
      FOR_EACH_VEC_ELT (char_p, flag_instrument_functions_exclude_files, i, s)
	if (strstr (name, s) != NULL)
	  return true;
    }

  return false;
}


/* Handle the vector of command line options (located at LOC), storing
   the results of processing DECODED_OPTIONS and DECODED_OPTIONS_COUNT
   in OPTS and OPTS_SET and using DC for diagnostic state.  LANG_MASK
   contains has a single bit set representing the current language.
   HANDLERS describes what functions to call for the options.  */
static void
read_cmdline_options (struct gcc_options *opts, struct gcc_options *opts_set,
		      struct cl_decoded_option *decoded_options,
		      unsigned int decoded_options_count,
		      location_t loc,
		      unsigned int lang_mask,
		      const struct cl_option_handlers *handlers,
		      diagnostic_context *dc)
{
  unsigned int i;

  for (i = 1; i < decoded_options_count; i++)
    {
      if (decoded_options[i].opt_index == OPT_SPECIAL_input_file)
	{
	  /* Input files should only ever appear on the main command
	     line.  */
	  gcc_assert (opts == &global_options);
	  gcc_assert (opts_set == &global_options_set);

	  if (main_input_filename == NULL)
	    {
	      main_input_filename = decoded_options[i].arg;
	      main_input_baselength
		= base_of_path (main_input_filename, &main_input_basename);
	    }
	  add_input_filename (decoded_options[i].arg);
	  continue;
	}

      read_cmdline_option (opts, opts_set,
			   decoded_options + i, loc, lang_mask, handlers,
			   dc);
    }
}

/* Language mask determined at initialization.  */
static unsigned int initial_lang_mask;

/* Initialize global options-related settings at start-up.  */

void
init_options_once (void)
{
  /* Perform language-specific options initialization.  */
  initial_lang_mask = lang_hooks.option_lang_mask ();

  lang_hooks.initialize_diagnostics (global_dc);
}

/* Initialize OPTS and OPTS_SET before using them in parsing options.  */

void
init_options_struct (struct gcc_options *opts, struct gcc_options *opts_set)
{
  size_t num_params = get_num_compiler_params ();

  *opts = global_options_init;
  memset (opts_set, 0, sizeof (*opts_set));

  opts->x_param_values = XNEWVEC (int, num_params);
  opts_set->x_param_values = XCNEWVEC (int, num_params);
  init_param_values (opts->x_param_values);

  /* Use priority coloring if cover classes is not defined for the
     target.  */
  if (targetm.ira_cover_classes == NULL)
    opts->x_flag_ira_algorithm = IRA_ALGORITHM_PRIORITY;

  /* Initialize whether `char' is signed.  */
  opts->x_flag_signed_char = DEFAULT_SIGNED_CHAR;
  /* Set this to a special "uninitialized" value.  The actual default
     is set after target options have been processed.  */
  opts->x_flag_short_enums = 2;

  /* Initialize target_flags before targetm.target_option.optimization
     so the latter can modify it.  */
  opts->x_target_flags = targetm.default_target_flags;

  /* Some targets have ABI-specified unwind tables.  */
  opts->x_flag_unwind_tables = targetm.unwind_tables_default;

  /* Some targets have other target-specific initialization.  */
  targetm.target_option.init_struct (opts);
}

/* Decode command-line options to an array, like
   decode_cmdline_options_to_array and with the same arguments but
   using the default lang_mask.  */

void
decode_cmdline_options_to_array_default_mask (unsigned int argc,
					      const char **argv, 
					      struct cl_decoded_option **decoded_options,
					      unsigned int *decoded_options_count)
{
  decode_cmdline_options_to_array (argc, argv,
				   initial_lang_mask | CL_COMMON | CL_TARGET,
				   decoded_options, decoded_options_count);
}

/* If indicated by the optimization level LEVEL (-Os if SIZE is set,
   -Ofast if FAST is set), apply the option DEFAULT_OPT to OPTS and
   OPTS_SET, diagnostic context DC, location LOC, with language mask
   LANG_MASK and option handlers HANDLERS.  */

static void
maybe_default_option (struct gcc_options *opts,
		      struct gcc_options *opts_set,
		      const struct default_options *default_opt,
		      int level, bool size, bool fast,
		      unsigned int lang_mask,
		      const struct cl_option_handlers *handlers,
		      location_t loc,
		      diagnostic_context *dc)
{
  const struct cl_option *option = &cl_options[default_opt->opt_index];
  bool enabled;

  if (size)
    gcc_assert (level == 2);
  if (fast)
    gcc_assert (level == 3);

  switch (default_opt->levels)
    {
    case OPT_LEVELS_ALL:
      enabled = true;
      break;

    case OPT_LEVELS_0_ONLY:
      enabled = (level == 0);
      break;

    case OPT_LEVELS_1_PLUS:
      enabled = (level >= 1);
      break;

    case OPT_LEVELS_1_PLUS_SPEED_ONLY:
      enabled = (level >= 1 && !size);
      break;

    case OPT_LEVELS_2_PLUS:
      enabled = (level >= 2);
      break;

    case OPT_LEVELS_2_PLUS_SPEED_ONLY:
      enabled = (level >= 2 && !size);
      break;

    case OPT_LEVELS_3_PLUS:
      enabled = (level >= 3);
      break;

    case OPT_LEVELS_3_PLUS_AND_SIZE:
      enabled = (level >= 3 || size);
      break;

    case OPT_LEVELS_SIZE:
      enabled = size;
      break;

    case OPT_LEVELS_FAST:
      enabled = fast;
      break;

    case OPT_LEVELS_NONE:
    default:
      gcc_unreachable ();
    }

  if (enabled)
    handle_generated_option (opts, opts_set, default_opt->opt_index,
			     default_opt->arg, default_opt->value,
			     lang_mask, DK_UNSPECIFIED, loc,
			     handlers, dc);
  else if (default_opt->arg == NULL
	   && !(option->flags & CL_REJECT_NEGATIVE))
    handle_generated_option (opts, opts_set, default_opt->opt_index,
			     default_opt->arg, !default_opt->value,
			     lang_mask, DK_UNSPECIFIED, loc,
			     handlers, dc);
}

/* As indicated by the optimization level LEVEL (-Os if SIZE is set,
   -Ofast if FAST is set), apply the options in array DEFAULT_OPTS to
   OPTS and OPTS_SET, diagnostic context DC, location LOC, with
   language mask LANG_MASK and option handlers HANDLERS.  */

static void
maybe_default_options (struct gcc_options *opts,
		       struct gcc_options *opts_set,
		       const struct default_options *default_opts,
		       int level, bool size, bool fast,
		       unsigned int lang_mask,
		       const struct cl_option_handlers *handlers,
		       location_t loc,
		       diagnostic_context *dc)
{
  size_t i;

  for (i = 0; default_opts[i].levels != OPT_LEVELS_NONE; i++)
    maybe_default_option (opts, opts_set, &default_opts[i],
			  level, size, fast, lang_mask, handlers, loc, dc);
}

/* Table of options enabled by default at different levels.  */

static const struct default_options default_options_table[] =
  {
    /* -O1 optimizations.  */
    { OPT_LEVELS_1_PLUS, OPT_fdefer_pop, NULL, 1 },
#ifdef DELAY_SLOTS
    { OPT_LEVELS_1_PLUS, OPT_fdelayed_branch, NULL, 1 },
#endif
    { OPT_LEVELS_1_PLUS, OPT_fguess_branch_probability, NULL, 1 },
    { OPT_LEVELS_1_PLUS, OPT_fcprop_registers, NULL, 1 },
    { OPT_LEVELS_1_PLUS, OPT_fforward_propagate, NULL, 1 },
    { OPT_LEVELS_1_PLUS, OPT_fif_conversion, NULL, 1 },
    { OPT_LEVELS_1_PLUS, OPT_fif_conversion2, NULL, 1 },
    { OPT_LEVELS_1_PLUS, OPT_fipa_pure_const, NULL, 1 },
    { OPT_LEVELS_1_PLUS, OPT_fipa_reference, NULL, 1 },
    { OPT_LEVELS_1_PLUS, OPT_fipa_profile, NULL, 1 },
    { OPT_LEVELS_1_PLUS, OPT_fmerge_constants, NULL, 1 },
    { OPT_LEVELS_1_PLUS, OPT_fsplit_wide_types, NULL, 1 },
    { OPT_LEVELS_1_PLUS, OPT_ftree_ccp, NULL, 1 },
    { OPT_LEVELS_1_PLUS, OPT_ftree_bit_ccp, NULL, 1 },
    { OPT_LEVELS_1_PLUS, OPT_ftree_dce, NULL, 1 },
    { OPT_LEVELS_1_PLUS, OPT_ftree_dominator_opts, NULL, 1 },
    { OPT_LEVELS_1_PLUS, OPT_ftree_dse, NULL, 1 },
    { OPT_LEVELS_1_PLUS, OPT_ftree_ter, NULL, 1 },
    { OPT_LEVELS_1_PLUS, OPT_ftree_sra, NULL, 1 },
    { OPT_LEVELS_1_PLUS, OPT_ftree_copyrename, NULL, 1 },
    { OPT_LEVELS_1_PLUS, OPT_ftree_fre, NULL, 1 },
    { OPT_LEVELS_1_PLUS, OPT_ftree_copy_prop, NULL, 1 },
    { OPT_LEVELS_1_PLUS, OPT_ftree_sink, NULL, 1 },
    { OPT_LEVELS_1_PLUS, OPT_ftree_ch, NULL, 1 },
    { OPT_LEVELS_1_PLUS, OPT_fcombine_stack_adjustments, NULL, 1 },

    /* -O2 optimizations.  */
    { OPT_LEVELS_2_PLUS, OPT_finline_small_functions, NULL, 1 },
    { OPT_LEVELS_2_PLUS, OPT_findirect_inlining, NULL, 1 },
    { OPT_LEVELS_2_PLUS, OPT_fpartial_inlining, NULL, 1 },
    { OPT_LEVELS_2_PLUS, OPT_fthread_jumps, NULL, 1 },
    { OPT_LEVELS_2_PLUS, OPT_fcrossjumping, NULL, 1 },
    { OPT_LEVELS_2_PLUS, OPT_foptimize_sibling_calls, NULL, 1 },
    { OPT_LEVELS_2_PLUS, OPT_fcse_follow_jumps, NULL, 1 },
    { OPT_LEVELS_2_PLUS, OPT_fgcse, NULL, 1 },
    { OPT_LEVELS_2_PLUS, OPT_fexpensive_optimizations, NULL, 1 },
    { OPT_LEVELS_2_PLUS, OPT_frerun_cse_after_loop, NULL, 1 },
    { OPT_LEVELS_2_PLUS, OPT_fcaller_saves, NULL, 1 },
    { OPT_LEVELS_2_PLUS, OPT_fpeephole2, NULL, 1 },
#ifdef INSN_SCHEDULING
  /* Only run the pre-regalloc scheduling pass if optimizing for speed.  */
    { OPT_LEVELS_2_PLUS_SPEED_ONLY, OPT_fschedule_insns, NULL, 1 },
    { OPT_LEVELS_2_PLUS, OPT_fschedule_insns2, NULL, 1 },
#endif
    { OPT_LEVELS_2_PLUS, OPT_fregmove, NULL, 1 },
    { OPT_LEVELS_2_PLUS, OPT_fstrict_aliasing, NULL, 1 },
    { OPT_LEVELS_2_PLUS, OPT_fstrict_overflow, NULL, 1 },
    { OPT_LEVELS_2_PLUS, OPT_freorder_blocks, NULL, 1 },
    { OPT_LEVELS_2_PLUS, OPT_freorder_functions, NULL, 1 },
    { OPT_LEVELS_2_PLUS, OPT_ftree_vrp, NULL, 1 },
    { OPT_LEVELS_2_PLUS, OPT_ftree_builtin_call_dce, NULL, 1 },
    { OPT_LEVELS_2_PLUS, OPT_ftree_pre, NULL, 1 },
    { OPT_LEVELS_2_PLUS, OPT_ftree_switch_conversion, NULL, 1 },
    { OPT_LEVELS_2_PLUS, OPT_fipa_cp, NULL, 1 },
    { OPT_LEVELS_2_PLUS, OPT_fipa_sra, NULL, 1 },
    { OPT_LEVELS_2_PLUS, OPT_falign_loops, NULL, 1 },
    { OPT_LEVELS_2_PLUS, OPT_falign_jumps, NULL, 1 },
    { OPT_LEVELS_2_PLUS, OPT_falign_labels, NULL, 1 },
    { OPT_LEVELS_2_PLUS, OPT_falign_functions, NULL, 1 },

    /* -O3 optimizations.  */
    { OPT_LEVELS_3_PLUS, OPT_ftree_loop_distribute_patterns, NULL, 1 },
    { OPT_LEVELS_3_PLUS, OPT_fpredictive_commoning, NULL, 1 },
    /* Inlining of functions reducing size is a good idea with -Os
       regardless of them being declared inline.  */
    { OPT_LEVELS_3_PLUS_AND_SIZE, OPT_finline_functions, NULL, 1 },
    { OPT_LEVELS_3_PLUS, OPT_funswitch_loops, NULL, 1 },
    { OPT_LEVELS_3_PLUS, OPT_fgcse_after_reload, NULL, 1 },
    { OPT_LEVELS_3_PLUS, OPT_ftree_vectorize, NULL, 1 },
    { OPT_LEVELS_3_PLUS, OPT_fipa_cp_clone, NULL, 1 },

    /* -Ofast adds optimizations to -O3.  */
    { OPT_LEVELS_FAST, OPT_ffast_math, NULL, 1 },

    { OPT_LEVELS_NONE, 0, NULL, 0 }
  };

/* Default the options in OPTS and OPTS_SET based on the optimization
   settings in DECODED_OPTIONS and DECODED_OPTIONS_COUNT.  */
static void
default_options_optimization (struct gcc_options *opts,
			      struct gcc_options *opts_set,
			      struct cl_decoded_option *decoded_options,
			      unsigned int decoded_options_count,
			      location_t loc,
			      unsigned int lang_mask,
			      const struct cl_option_handlers *handlers,
			      diagnostic_context *dc)
{
  unsigned int i;
  int opt2;
  int ofast = 0;

  /* Scan to see what optimization level has been specified.  That will
     determine the default value of many flags.  */
  for (i = 1; i < decoded_options_count; i++)
    {
      struct cl_decoded_option *opt = &decoded_options[i];
      switch (opt->opt_index)
	{
	case OPT_O:
	  if (*opt->arg == '\0')
	    {
	      opts->x_optimize = 1;
	      opts->x_optimize_size = 0;
	      ofast = 0;
	    }
	  else
	    {
	      const int optimize_val = integral_argument (opt->arg);
	      if (optimize_val == -1)
		error ("argument to %qs should be a non-negative integer",
		       "-O");
	      else
		{
		  opts->x_optimize = optimize_val;
		  if ((unsigned int) opts->x_optimize > 255)
		    opts->x_optimize = 255;
		  opts->x_optimize_size = 0;
		  ofast = 0;
		}
	    }
	  break;

	case OPT_Os:
	  opts->x_optimize_size = 1;

	  /* Optimizing for size forces optimize to be 2.  */
	  opts->x_optimize = 2;
	  ofast = 0;
	  break;

	case OPT_Ofast:
	  /* -Ofast only adds flags to -O3.  */
	  opts->x_optimize_size = 0;
	  opts->x_optimize = 3;
	  ofast = 1;
	  break;

	default:
	  /* Ignore other options in this prescan.  */
	  break;
	}
    }

  maybe_default_options (opts, opts_set, default_options_table,
			 opts->x_optimize, opts->x_optimize_size,
			 ofast, lang_mask, handlers, loc, dc);

  /* -O2 param settings.  */
  opt2 = (opts->x_optimize >= 2);

  /* Track fields in field-sensitive alias analysis.  */
  maybe_set_param_value
    (PARAM_MAX_FIELDS_FOR_FIELD_SENSITIVE,
     opt2 ? 100 : default_param_value (PARAM_MAX_FIELDS_FOR_FIELD_SENSITIVE),
     opts->x_param_values, opts_set->x_param_values);

  /* For -O1 only do loop invariant motion for very small loops.  */
  maybe_set_param_value
    (PARAM_LOOP_INVARIANT_MAX_BBS_IN_LOOP,
     opt2 ? default_param_value (PARAM_LOOP_INVARIANT_MAX_BBS_IN_LOOP) : 1000,
     opts->x_param_values, opts_set->x_param_values);

  if (opts->x_optimize_size)
    /* We want to crossjump as much as possible.  */
    maybe_set_param_value (PARAM_MIN_CROSSJUMP_INSNS, 1,
			   opts->x_param_values, opts_set->x_param_values);
  else
    maybe_set_param_value (PARAM_MIN_CROSSJUMP_INSNS,
			   default_param_value (PARAM_MIN_CROSSJUMP_INSNS),
			   opts->x_param_values, opts_set->x_param_values);

  /* Allow default optimizations to be specified on a per-machine basis.  */
  maybe_default_options (opts, opts_set,
			 targetm.target_option.optimization_table,
			 opts->x_optimize, opts->x_optimize_size,
			 ofast, lang_mask, handlers, loc, dc);
}

static void finish_options (struct gcc_options *, struct gcc_options *);

/* Set *HANDLERS to the default set of option handlers for use in the
   compilers proper (not the driver).  */
void
set_default_handlers (struct cl_option_handlers *handlers)
{
  handlers->unknown_option_callback = unknown_option_callback;
  handlers->wrong_lang_callback = complain_wrong_lang;
  handlers->post_handling_callback = post_handling_callback;
  handlers->num_handlers = 3;
  handlers->handlers[0].handler = lang_handle_option;
  handlers->handlers[0].mask = initial_lang_mask;
  handlers->handlers[1].handler = common_handle_option;
  handlers->handlers[1].mask = CL_COMMON;
  handlers->handlers[2].handler = target_handle_option;
  handlers->handlers[2].mask = CL_TARGET;
}

/* Parse command line options and set default flag values.  Do minimal
   options processing.  The decoded options are in *DECODED_OPTIONS
   and *DECODED_OPTIONS_COUNT; settings go in OPTS, OPTS_SET and DC;
   the options are located at LOC.  */
void
decode_options (struct gcc_options *opts, struct gcc_options *opts_set,
		struct cl_decoded_option *decoded_options,
		unsigned int decoded_options_count,
		location_t loc, diagnostic_context *dc)
{
  struct cl_option_handlers handlers;

  unsigned int lang_mask;

  lang_mask = initial_lang_mask;

  set_default_handlers (&handlers);

  /* Enable -Werror=coverage-mismatch by default.  */
  control_warning_option (OPT_Wcoverage_mismatch, (int) DK_ERROR, true,
			  loc, lang_mask,
			  &handlers, opts, opts_set, dc);

  default_options_optimization (opts, opts_set,
				decoded_options, decoded_options_count,
				loc, lang_mask, &handlers, dc);

#ifdef ENABLE_LTO
  /* Clear any options currently held for LTO.  */
  lto_clear_user_options ();
#endif

  read_cmdline_options (opts, opts_set,
			decoded_options, decoded_options_count,
			loc, lang_mask,
			&handlers, dc);

  finish_options (opts, opts_set);
}

/* After all options have been read into OPTS and OPTS_SET, finalize
   settings of those options and diagnose incompatible
   combinations.  */
static void
finish_options (struct gcc_options *opts, struct gcc_options *opts_set)
{
  static bool first_time_p = true;
  enum unwind_info_type ui_except;

  gcc_assert (opts == &global_options);
  gcc_assert (opts_set = &global_options_set);

  if (opts->x_dump_base_name && ! IS_ABSOLUTE_PATH (opts->x_dump_base_name))
    {
      /* First try to make OPTS->X_DUMP_BASE_NAME relative to the
	 OPTS->X_DUMP_DIR_NAME directory.  Then try to make
	 OPTS->X_DUMP_BASE_NAME relative to the OPTS->X_AUX_BASE_NAME
	 directory, typically the directory to contain the object
	 file.  */
      if (opts->x_dump_dir_name)
	opts->x_dump_base_name = concat (opts->x_dump_dir_name,
					 opts->x_dump_base_name, NULL);
      else if (opts->x_aux_base_name)
	{
	  const char *aux_base;

	  base_of_path (opts->x_aux_base_name, &aux_base);
	  if (opts->x_aux_base_name != aux_base)
	    {
	      int dir_len = aux_base - opts->x_aux_base_name;
	      char *new_dump_base_name =
		XNEWVEC (char, strlen (opts->x_dump_base_name) + dir_len + 1);

	      /* Copy directory component from OPTS->X_AUX_BASE_NAME.  */
	      memcpy (new_dump_base_name, opts->x_aux_base_name, dir_len);
	      /* Append existing OPTS->X_DUMP_BASE_NAME.  */
	      strcpy (new_dump_base_name + dir_len, opts->x_dump_base_name);
	      opts->x_dump_base_name = new_dump_base_name;
	    }
	}
    }

  /* Handle related options for unit-at-a-time, toplevel-reorder, and
     section-anchors.  */
  if (!opts->x_flag_unit_at_a_time)
    {
      if (opts->x_flag_section_anchors && opts_set->x_flag_section_anchors)
	error ("section anchors must be disabled when unit-at-a-time "
	       "is disabled");
      opts->x_flag_section_anchors = 0;
      if (opts->x_flag_toplevel_reorder == 1)
	error ("toplevel reorder must be disabled when unit-at-a-time "
	       "is disabled");
      opts->x_flag_toplevel_reorder = 0;
    }

  /* -Wmissing-noreturn is alias for -Wsuggest-attribute=noreturn.  */
  if (opts->x_warn_missing_noreturn)
    opts->x_warn_suggest_attribute_noreturn = true;
    
  /* Unless the user has asked for section anchors, we disable toplevel
     reordering at -O0 to disable transformations that might be surprising
     to end users and to get -fno-toplevel-reorder tested.  */
  if (!optimize
      && opts->x_flag_toplevel_reorder == 2
      && !(opts->x_flag_section_anchors && opts_set->x_flag_section_anchors))
    {
      opts->x_flag_toplevel_reorder = 0;
      opts->x_flag_section_anchors = 0;
    }
  if (!opts->x_flag_toplevel_reorder)
    {
      if (opts->x_flag_section_anchors && opts_set->x_flag_section_anchors)
	error ("section anchors must be disabled when toplevel reorder"
	       " is disabled");
      opts->x_flag_section_anchors = 0;
    }

  if (first_time_p)
    {
      if (opts->x_flag_pie)
	opts->x_flag_pic = opts->x_flag_pie;
      if (opts->x_flag_pic && !opts->x_flag_pie)
	opts->x_flag_shlib = 1;
      first_time_p = false;
    }

  if (optimize == 0)
    {
      /* Inlining does not work if not optimizing,
	 so force it not to be done.  */
      opts->x_warn_inline = 0;
      opts->x_flag_no_inline = 1;
    }

  /* The optimization to partition hot and cold basic blocks into separate
     sections of the .o and executable files does not work (currently)
     with exception handling.  This is because there is no support for
     generating unwind info.  If opts->x_flag_exceptions is turned on
     we need to turn off the partitioning optimization.  */

  ui_except = targetm.except_unwind_info ();

  if (opts->x_flag_exceptions
      && opts->x_flag_reorder_blocks_and_partition
      && (ui_except == UI_SJLJ || ui_except == UI_TARGET))
    {
      inform (input_location,
	      "-freorder-blocks-and-partition does not work "
	      "with exceptions on this architecture");
      opts->x_flag_reorder_blocks_and_partition = 0;
      opts->x_flag_reorder_blocks = 1;
    }

  /* If user requested unwind info, then turn off the partitioning
     optimization.  */

  if (opts->x_flag_unwind_tables
      && !targetm.unwind_tables_default
      && opts->x_flag_reorder_blocks_and_partition
      && (ui_except == UI_SJLJ || ui_except == UI_TARGET))
    {
      inform (input_location,
	      "-freorder-blocks-and-partition does not support "
	      "unwind info on this architecture");
      opts->x_flag_reorder_blocks_and_partition = 0;
      opts->x_flag_reorder_blocks = 1;
    }

  /* If the target requested unwind info, then turn off the partitioning
     optimization with a different message.  Likewise, if the target does not
     support named sections.  */

  if (opts->x_flag_reorder_blocks_and_partition
      && (!targetm.have_named_sections
	  || (opts->x_flag_unwind_tables
	      && targetm.unwind_tables_default
	      && (ui_except == UI_SJLJ || ui_except == UI_TARGET))))
    {
      inform (input_location,
	      "-freorder-blocks-and-partition does not work "
	      "on this architecture");
      opts->x_flag_reorder_blocks_and_partition = 0;
      opts->x_flag_reorder_blocks = 1;
    }

  /* Pipelining of outer loops is only possible when general pipelining
     capabilities are requested.  */
  if (!opts->x_flag_sel_sched_pipelining)
    opts->x_flag_sel_sched_pipelining_outer_loops = 0;

  if (!targetm.ira_cover_classes
      && opts->x_flag_ira_algorithm == IRA_ALGORITHM_CB)
    {
      inform (input_location,
	      "-fira-algorithm=CB does not work on this architecture");
      opts->x_flag_ira_algorithm = IRA_ALGORITHM_PRIORITY;
    }

  if (opts->x_flag_conserve_stack)
    {
      maybe_set_param_value (PARAM_LARGE_STACK_FRAME, 100,
			     opts->x_param_values, opts_set->x_param_values);
      maybe_set_param_value (PARAM_STACK_FRAME_GROWTH, 40,
			     opts->x_param_values, opts_set->x_param_values);
    }
  if (opts->x_flag_wpa || opts->x_flag_ltrans)
    {
      /* These passes are not WHOPR compatible yet.  */
      opts->x_flag_ipa_pta = 0;
      opts->x_flag_ipa_struct_reorg = 0;
    }

  if (opts->x_flag_lto)
    {
#ifdef ENABLE_LTO
      opts->x_flag_generate_lto = 1;

      /* When generating IL, do not operate in whole-program mode.
	 Otherwise, symbols will be privatized too early, causing link
	 errors later.  */
      opts->x_flag_whole_program = 0;
#else
      error ("LTO support has not been enabled in this configuration");
#endif
    }
  if ((opts->x_flag_lto_partition_balanced != 0) + (opts->x_flag_lto_partition_1to1 != 0)
       + (opts->x_flag_lto_partition_none != 0) >= 1)
    {
      if ((opts->x_flag_lto_partition_balanced != 0)
	   + (opts->x_flag_lto_partition_1to1 != 0)
	   + (opts->x_flag_lto_partition_none != 0) > 1)
	error ("only one -flto-partition value can be specified");
    }

  /* We initialize opts->x_flag_split_stack to -1 so that targets can set a
     default value if they choose based on other options.  */
  if (opts->x_flag_split_stack == -1)
    opts->x_flag_split_stack = 0;
  else if (opts->x_flag_split_stack)
    {
      if (!targetm.supports_split_stack (true))
	{
	  error ("%<-fsplit-stack%> is not supported by "
		 "this compiler configuration");
	  opts->x_flag_split_stack = 0;
	}
    }
}

#define LEFT_COLUMN	27

/* Output ITEM, of length ITEM_WIDTH, in the left column,
   followed by word-wrapped HELP in a second column.  */
static void
wrap_help (const char *help,
	   const char *item,
	   unsigned int item_width,
	   unsigned int columns)
{
  unsigned int col_width = LEFT_COLUMN;
  unsigned int remaining, room, len;

  remaining = strlen (help);

  do
    {
      room = columns - 3 - MAX (col_width, item_width);
      if (room > columns)
	room = 0;
      len = remaining;

      if (room < len)
	{
	  unsigned int i;

	  for (i = 0; help[i]; i++)
	    {
	      if (i >= room && len != remaining)
		break;
	      if (help[i] == ' ')
		len = i;
	      else if ((help[i] == '-' || help[i] == '/')
		       && help[i + 1] != ' '
		       && i > 0 && ISALPHA (help[i - 1]))
		len = i + 1;
	    }
	}

      printf( "  %-*.*s %.*s\n", col_width, item_width, item, len, help);
      item_width = 0;
      while (help[len] == ' ')
	len++;
      help += len;
      remaining -= len;
    }
  while (remaining);
}

/* Print help for a specific front-end, etc.  */
static void
print_filtered_help (unsigned int include_flags,
		     unsigned int exclude_flags,
		     unsigned int any_flags,
		     unsigned int columns)
{
  unsigned int i;
  const char *help;
  static char *printed = NULL;
  bool found = false;
  bool displayed = false;

  if (include_flags == CL_PARAMS)
    {
      for (i = 0; i < LAST_PARAM; i++)
	{
	  const char *param = compiler_params[i].option;

	  help = compiler_params[i].help;
	  if (help == NULL || *help == '\0')
	    {
	      if (exclude_flags & CL_UNDOCUMENTED)
		continue;
	      help = undocumented_msg;
	    }

	  /* Get the translation.  */
	  help = _(help);

	  wrap_help (help, param, strlen (param), columns);
	}
      putchar ('\n');
      return;
    }

  if (!printed)
    printed = XCNEWVAR (char, cl_options_count);

  for (i = 0; i < cl_options_count; i++)
    {
      static char new_help[128];
      const struct cl_option *option = cl_options + i;
      unsigned int len;
      const char *opt;
      const char *tab;

      if (include_flags == 0
	  || ((option->flags & include_flags) != include_flags))
	{
	  if ((option->flags & any_flags) == 0)
	    continue;
	}

      /* Skip unwanted switches.  */
      if ((option->flags & exclude_flags) != 0)
	continue;

      /* The driver currently prints its own help text.  */
      if ((option->flags & CL_DRIVER) != 0
	  && (option->flags & (((1U << cl_lang_count) - 1)
			       | CL_COMMON | CL_TARGET)) == 0)
	continue;

      found = true;
      /* Skip switches that have already been printed.  */
      if (printed[i])
	continue;

      printed[i] = true;

      help = option->help;
      if (help == NULL)
	{
	  if (exclude_flags & CL_UNDOCUMENTED)
	    continue;
	  help = undocumented_msg;
	}

      /* Get the translation.  */
      help = _(help);

      /* Find the gap between the name of the
	 option and its descriptive text.  */
      tab = strchr (help, '\t');
      if (tab)
	{
	  len = tab - help;
	  opt = help;
	  help = tab + 1;
	}
      else
	{
	  opt = option->opt_text;
	  len = strlen (opt);
	}

      /* With the -Q option enabled we change the descriptive text associated
	 with an option to be an indication of its current setting.  */
      if (!quiet_flag)
	{
	  void *flag_var = option_flag_var (i, &global_options);

	  if (len < (LEFT_COLUMN + 2))
	    strcpy (new_help, "\t\t");
	  else
	    strcpy (new_help, "\t");

	  if (flag_var != NULL)
	    {
	      if (option->flags & CL_JOINED)
		{
		  if (option->var_type == CLVC_STRING)
		    {
		      if (* (const char **) flag_var != NULL)
			snprintf (new_help + strlen (new_help),
				  sizeof (new_help) - strlen (new_help),
				  * (const char **) flag_var);
		    }
		  else
		    sprintf (new_help + strlen (new_help),
			     "%#x", * (int *) flag_var);
		}
	      else
		strcat (new_help, option_enabled (i, &global_options)
			? _("[enabled]") : _("[disabled]"));
	    }

	  help = new_help;
	}

      wrap_help (help, opt, len, columns);
      displayed = true;
    }

  if (! found)
    {
      unsigned int langs = include_flags & CL_LANG_ALL;

      if (langs == 0)
	printf (_(" No options with the desired characteristics were found\n"));
      else
	{
	  unsigned int i;

	  /* PR 31349: Tell the user how to see all of the
	     options supported by a specific front end.  */
	  for (i = 0; (1U << i) < CL_LANG_ALL; i ++)
	    if ((1U << i) & langs)
	      printf (_(" None found.  Use --help=%s to show *all* the options supported by the %s front-end\n"),
		      lang_names[i], lang_names[i]);
	}

    }
  else if (! displayed)
    printf (_(" All options with the desired characteristics have already been displayed\n"));

  putchar ('\n');
}

/* Display help for a specified type of option.
   The options must have ALL of the INCLUDE_FLAGS set
   ANY of the flags in the ANY_FLAGS set
   and NONE of the EXCLUDE_FLAGS set.  */
static void
print_specific_help (unsigned int include_flags,
		     unsigned int exclude_flags,
		     unsigned int any_flags)
{
  unsigned int all_langs_mask = (1U << cl_lang_count) - 1;
  const char * description = NULL;
  const char * descrip_extra = "";
  size_t i;
  unsigned int flag;
  static unsigned int columns = 0;

  /* Sanity check: Make sure that we do not have more
     languages than we have bits available to enumerate them.  */
  gcc_assert ((1U << cl_lang_count) < CL_MIN_OPTION_CLASS);

  /* If we have not done so already, obtain
     the desired maximum width of the output.  */
  if (columns == 0)
    {
      const char *p;

      GET_ENVIRONMENT (p, "COLUMNS");
      if (p != NULL)
	{
	  int value = atoi (p);

	  if (value > 0)
	    columns = value;
	}

      if (columns == 0)
	/* Use a reasonable default.  */
	columns = 80;
    }

  /* Decide upon the title for the options that we are going to display.  */
  for (i = 0, flag = 1; flag <= CL_MAX_OPTION_CLASS; flag <<= 1, i ++)
    {
      switch (flag & include_flags)
	{
	case 0:
	case CL_DRIVER:
	  break;

	case CL_TARGET:
	  description = _("The following options are target specific");
	  break;
	case CL_WARNING:
	  description = _("The following options control compiler warning messages");
	  break;
	case CL_OPTIMIZATION:
	  description = _("The following options control optimizations");
	  break;
	case CL_COMMON:
	  description = _("The following options are language-independent");
	  break;
	case CL_PARAMS:
	  description = _("The --param option recognizes the following as parameters");
	  break;
	default:
	  if (i >= cl_lang_count)
	    break;
	  if (exclude_flags & all_langs_mask)
	    description = _("The following options are specific to just the language ");
	  else
	    description = _("The following options are supported by the language ");
	  descrip_extra = lang_names [i];
	  break;
	}
    }

  if (description == NULL)
    {
      if (any_flags == 0)
	{
	  if (include_flags & CL_UNDOCUMENTED)
	    description = _("The following options are not documented");
	  else if (include_flags & CL_SEPARATE)
	    description = _("The following options take separate arguments");
	  else if (include_flags & CL_JOINED)
	    description = _("The following options take joined arguments");
	  else
	    {
	      internal_error ("unrecognized include_flags 0x%x passed to print_specific_help",
			      include_flags);
	      return;
	    }
	}
      else
	{
	  if (any_flags & all_langs_mask)
	    description = _("The following options are language-related");
	  else
	    description = _("The following options are language-independent");
	}
    }

  printf ("%s%s:\n", description, descrip_extra);
  print_filtered_help (include_flags, exclude_flags, any_flags, columns);
}

/* Handle target- and language-independent options.  Return zero to
   generate an "unknown option" message.  Only options that need
   extra handling need to be listed here; if you simply want
   DECODED->value assigned to a variable, it happens automatically.  */

static bool
common_handle_option (struct gcc_options *opts,
		      struct gcc_options *opts_set,
		      const struct cl_decoded_option *decoded,
		      unsigned int lang_mask, int kind ATTRIBUTE_UNUSED,
		      location_t loc,
		      const struct cl_option_handlers *handlers,
		      diagnostic_context *dc)
{
  size_t scode = decoded->opt_index;
  const char *arg = decoded->arg;
  int value = decoded->value;
  enum opt_code code = (enum opt_code) scode;

  gcc_assert (opts == &global_options);
  gcc_assert (opts_set == &global_options_set);
  gcc_assert (dc == global_dc);
  gcc_assert (decoded->canonical_option_num_elements <= 2);

  switch (code)
    {
    case OPT__param:
      handle_param (opts, opts_set, arg);
      break;

    case OPT__help:
      {
	unsigned int all_langs_mask = (1U << cl_lang_count) - 1;
	unsigned int undoc_mask;
	unsigned int i;

	undoc_mask = ((opts->x_verbose_flag | opts->x_extra_warnings)
		      ? 0
		      : CL_UNDOCUMENTED);
	/* First display any single language specific options.  */
	for (i = 0; i < cl_lang_count; i++)
	  print_specific_help
	    (1U << i, (all_langs_mask & (~ (1U << i))) | undoc_mask, 0);
	/* Next display any multi language specific options.  */
	print_specific_help (0, undoc_mask, all_langs_mask);
	/* Then display any remaining, non-language options.  */
	for (i = CL_MIN_OPTION_CLASS; i <= CL_MAX_OPTION_CLASS; i <<= 1)
	  if (i != CL_DRIVER)
	    print_specific_help (i, undoc_mask, 0);
	exit_after_options = true;
	break;
      }

    case OPT__target_help:
      print_specific_help (CL_TARGET, CL_UNDOCUMENTED, 0);
      exit_after_options = true;

      /* Allow the target a chance to give the user some additional information.  */
      if (targetm.help)
	targetm.help ();
      break;

    case OPT__help_:
      {
	const char * a = arg;
	unsigned int include_flags = 0;
	/* Note - by default we include undocumented options when listing
	   specific classes.  If you only want to see documented options
	   then add ",^undocumented" to the --help= option.  E.g.:

	   --help=target,^undocumented  */
	unsigned int exclude_flags = 0;

	/* Walk along the argument string, parsing each word in turn.
	   The format is:
	   arg = [^]{word}[,{arg}]
	   word = {optimizers|target|warnings|undocumented|
		   params|common|<language>}  */
	while (* a != 0)
	  {
	    static struct
	    {
	      const char * string;
	      unsigned int flag;
	    }
	    specifics[] =
	    {
	      { "optimizers", CL_OPTIMIZATION },
	      { "target", CL_TARGET },
	      { "warnings", CL_WARNING },
	      { "undocumented", CL_UNDOCUMENTED },
	      { "params", CL_PARAMS },
	      { "joined", CL_JOINED },
	      { "separate", CL_SEPARATE },
	      { "common", CL_COMMON },
	      { NULL, 0 }
	    };
	    unsigned int * pflags;
	    const char * comma;
	    unsigned int lang_flag, specific_flag;
	    unsigned int len;
	    unsigned int i;

	    if (* a == '^')
	      {
		++ a;
		pflags = & exclude_flags;
	      }
	    else
	      pflags = & include_flags;

	    comma = strchr (a, ',');
	    if (comma == NULL)
	      len = strlen (a);
	    else
	      len = comma - a;
	    if (len == 0)
	      {
		a = comma + 1;
		continue;
	      }

	    /* Check to see if the string matches an option class name.  */
	    for (i = 0, specific_flag = 0; specifics[i].string != NULL; i++)
	      if (strncasecmp (a, specifics[i].string, len) == 0)
		{
		  specific_flag = specifics[i].flag;
		  break;
		}

	    /* Check to see if the string matches a language name.
	       Note - we rely upon the alpha-sorted nature of the entries in
	       the lang_names array, specifically that shorter names appear
	       before their longer variants.  (i.e. C before C++).  That way
	       when we are attempting to match --help=c for example we will
	       match with C first and not C++.  */
	    for (i = 0, lang_flag = 0; i < cl_lang_count; i++)
	      if (strncasecmp (a, lang_names[i], len) == 0)
		{
		  lang_flag = 1U << i;
		  break;
		}

	    if (specific_flag != 0)
	      {
		if (lang_flag == 0)
		  * pflags |= specific_flag;
		else
		  {
		    /* The option's argument matches both the start of a
		       language name and the start of an option class name.
		       We have a special case for when the user has
		       specified "--help=c", but otherwise we have to issue
		       a warning.  */
		    if (strncasecmp (a, "c", len) == 0)
		      * pflags |= lang_flag;
		    else
		      fnotice (stderr,
			       "warning: --help argument %.*s is ambiguous, please be more specific\n",
			       len, a);
		  }
	      }
	    else if (lang_flag != 0)
	      * pflags |= lang_flag;
	    else
	      fnotice (stderr,
		       "warning: unrecognized argument to --help= option: %.*s\n",
		       len, a);

	    if (comma == NULL)
	      break;
	    a = comma + 1;
	  }

	if (include_flags)
	  print_specific_help (include_flags, exclude_flags, 0);
	exit_after_options = true;
	break;
      }

    case OPT__version:
      exit_after_options = true;
      break;

    case OPT_O:
    case OPT_Os:
    case OPT_Ofast:
      /* Currently handled in a prescan.  */
      break;

    case OPT_Werror_:
      enable_warning_as_error (arg, value, lang_mask, handlers,
			       opts, opts_set, loc, dc);
      break;

    case OPT_Wlarger_than_:
      opts->x_larger_than_size = value;
      opts->x_warn_larger_than = value != -1;
      break;

    case OPT_Wfatal_errors:
      dc->fatal_errors = value;
      break;

    case OPT_Wframe_larger_than_:
      opts->x_frame_larger_than_size = value;
      opts->x_warn_frame_larger_than = value != -1;
      break;

    case OPT_Wstrict_aliasing:
      set_Wstrict_aliasing (opts, value);
      break;

    case OPT_Wstrict_overflow:
      opts->x_warn_strict_overflow = (value
				      ? (int) WARN_STRICT_OVERFLOW_CONDITIONAL
				      : 0);
      break;

    case OPT_Wsystem_headers:
      dc->dc_warn_system_headers = value;
      break;

    case OPT_aux_info:
      opts->x_flag_gen_aux_info = 1;
      break;

    case OPT_auxbase_strip:
      {
	char *tmp = xstrdup (arg);
	strip_off_ending (tmp, strlen (tmp));
	if (tmp[0])
	  opts->x_aux_base_name = tmp;
      }
      break;

    case OPT_d:
      decode_d_option (arg);
      break;

    case OPT_fcall_used_:
      fix_register (arg, 0, 1);
      break;

    case OPT_fcall_saved_:
      fix_register (arg, 0, 0);
      break;

    case OPT_fcompare_debug_second:
      flag_compare_debug = value;
      break;

    case OPT_fdbg_cnt_:
      dbg_cnt_process_opt (arg);
      break;

    case OPT_fdbg_cnt_list:
      dbg_cnt_list_all_counters ();
      break;

    case OPT_fdebug_prefix_map_:
      add_debug_prefix_map (arg);
      break;

    case OPT_fdiagnostics_show_location_:
      if (!strcmp (arg, "once"))
	diagnostic_prefixing_rule (dc) = DIAGNOSTICS_SHOW_PREFIX_ONCE;
      else if (!strcmp (arg, "every-line"))
	diagnostic_prefixing_rule (dc)
	  = DIAGNOSTICS_SHOW_PREFIX_EVERY_LINE;
      else
	return false;
      break;

    case OPT_fdiagnostics_show_option:
      dc->show_option_requested = value;
      break;

    case OPT_fdump_:
      if (!dump_switch_p (arg))
	return false;
      break;

    case OPT_ffp_contract_:
      if (!strcmp (arg, "on"))
	/* Not implemented, fall back to conservative FP_CONTRACT_OFF.  */
	opts->x_flag_fp_contract_mode = FP_CONTRACT_OFF;
      else if (!strcmp (arg, "off"))
	opts->x_flag_fp_contract_mode = FP_CONTRACT_OFF;
      else if (!strcmp (arg, "fast"))
	opts->x_flag_fp_contract_mode = FP_CONTRACT_FAST;
      else
	error ("unknown floating point contraction style \"%s\"", arg);
      break;

    case OPT_fexcess_precision_:
      if (!strcmp (arg, "fast"))
	opts->x_flag_excess_precision_cmdline = EXCESS_PRECISION_FAST;
      else if (!strcmp (arg, "standard"))
	opts->x_flag_excess_precision_cmdline = EXCESS_PRECISION_STANDARD;
      else
	error ("unknown excess precision style \"%s\"", arg);
      break;

    case OPT_ffast_math:
      set_fast_math_flags (opts, value);
      break;

    case OPT_funsafe_math_optimizations:
      set_unsafe_math_optimizations_flags (opts, value);
      break;

    case OPT_ffixed_:
      fix_register (arg, 1, 1);
      break;

    case OPT_finline_limit_:
      set_param_value ("max-inline-insns-single", value / 2,
		       opts->x_param_values, opts_set->x_param_values);
      set_param_value ("max-inline-insns-auto", value / 2,
		       opts->x_param_values, opts_set->x_param_values);
      break;

    case OPT_finstrument_functions_exclude_function_list_:
      add_comma_separated_to_vector
	(&flag_instrument_functions_exclude_functions, arg);
      break;

    case OPT_finstrument_functions_exclude_file_list_:
      add_comma_separated_to_vector
	(&flag_instrument_functions_exclude_files, arg);
      break;

    case OPT_fmessage_length_:
      pp_set_line_maximum_length (dc->printer, value);
      break;

    case OPT_fpack_struct_:
      if (value <= 0 || (value & (value - 1)) || value > 16)
	error ("structure alignment must be a small power of two, not %d", value);
      else
	{
	  initial_max_fld_align = value;
	  maximum_field_alignment = value * BITS_PER_UNIT;
	}
      break;

    case OPT_fplugin_:
#ifdef ENABLE_PLUGIN
      add_new_plugin (arg);
#else
      error ("plugin support is disabled; configure with --enable-plugin");
#endif
      break;

    case OPT_fplugin_arg_:
#ifdef ENABLE_PLUGIN
      parse_plugin_arg_opt (arg);
#else
      error ("plugin support is disabled; configure with --enable-plugin");
#endif
      break;

    case OPT_fprofile_dir_:
      profile_data_prefix = xstrdup (arg);
      break;

    case OPT_fprofile_use_:
      profile_data_prefix = xstrdup (arg);
      opts->x_flag_profile_use = true;
      value = true;
      /* No break here - do -fprofile-use processing. */
    case OPT_fprofile_use:
      if (!opts_set->x_flag_branch_probabilities)
	opts->x_flag_branch_probabilities = value;
      if (!opts_set->x_flag_profile_values)
	opts->x_flag_profile_values = value;
      if (!opts_set->x_flag_unroll_loops)
	opts->x_flag_unroll_loops = value;
      if (!opts_set->x_flag_peel_loops)
	opts->x_flag_peel_loops = value;
      if (!opts_set->x_flag_tracer)
	opts->x_flag_tracer = value;
      if (!opts_set->x_flag_value_profile_transformations)
	opts->x_flag_value_profile_transformations = value;
      if (!opts_set->x_flag_inline_functions)
	opts->x_flag_inline_functions = value;
      if (!opts_set->x_flag_ipa_cp)
	opts->x_flag_ipa_cp = value;
      if (!opts_set->x_flag_ipa_cp_clone
	  && value && opts->x_flag_ipa_cp)
	opts->x_flag_ipa_cp_clone = value;
      if (!opts_set->x_flag_predictive_commoning)
	opts->x_flag_predictive_commoning = value;
      if (!opts_set->x_flag_unswitch_loops)
	opts->x_flag_unswitch_loops = value;
      if (!opts_set->x_flag_gcse_after_reload)
	opts->x_flag_gcse_after_reload = value;
      break;

    case OPT_fprofile_generate_:
      profile_data_prefix = xstrdup (arg);
      value = true;
      /* No break here - do -fprofile-generate processing. */
    case OPT_fprofile_generate:
      if (!opts_set->x_profile_arc_flag)
	opts->x_profile_arc_flag = value;
      if (!opts_set->x_flag_profile_values)
	opts->x_flag_profile_values = value;
      if (!opts_set->x_flag_value_profile_transformations)
	opts->x_flag_value_profile_transformations = value;
      if (!opts_set->x_flag_inline_functions)
	opts->x_flag_inline_functions = value;
      break;

    case OPT_fshow_column:
      dc->show_column = value;
      break;

    case OPT_fvisibility_:
      {
        if (!strcmp(arg, "default"))
          opts->x_default_visibility = VISIBILITY_DEFAULT;
        else if (!strcmp(arg, "internal"))
          opts->x_default_visibility = VISIBILITY_INTERNAL;
        else if (!strcmp(arg, "hidden"))
          opts->x_default_visibility = VISIBILITY_HIDDEN;
        else if (!strcmp(arg, "protected"))
          opts->x_default_visibility = VISIBILITY_PROTECTED;
        else
          error ("unrecognized visibility value \"%s\"", arg);
      }
      break;

    case OPT_frandom_seed:
      /* The real switch is -fno-random-seed.  */
      if (value)
	return false;
      set_random_seed (NULL);
      break;

    case OPT_frandom_seed_:
      set_random_seed (arg);
      break;

    case OPT_fsched_verbose_:
#ifdef INSN_SCHEDULING
      fix_sched_param ("verbose", arg);
      break;
#else
      return false;
#endif

    case OPT_fsched_stalled_insns_:
      opts->x_flag_sched_stalled_insns = value;
      if (opts->x_flag_sched_stalled_insns == 0)
	opts->x_flag_sched_stalled_insns = -1;
      break;

    case OPT_fsched_stalled_insns_dep_:
      opts->x_flag_sched_stalled_insns_dep = value;
      break;

    case OPT_fstack_check_:
      if (!strcmp (arg, "no"))
	flag_stack_check = NO_STACK_CHECK;
      else if (!strcmp (arg, "generic"))
	/* This is the old stack checking method.  */
	flag_stack_check = STACK_CHECK_BUILTIN
			   ? FULL_BUILTIN_STACK_CHECK
			   : GENERIC_STACK_CHECK;
      else if (!strcmp (arg, "specific"))
	/* This is the new stack checking method.  */
	flag_stack_check = STACK_CHECK_BUILTIN
			   ? FULL_BUILTIN_STACK_CHECK
			   : STACK_CHECK_STATIC_BUILTIN
			     ? STATIC_BUILTIN_STACK_CHECK
			     : GENERIC_STACK_CHECK;
      else
	warning (0, "unknown stack check parameter \"%s\"", arg);
      break;

    case OPT_fstack_limit:
      /* The real switch is -fno-stack-limit.  */
      if (value)
	return false;
      stack_limit_rtx = NULL_RTX;
      break;

    case OPT_fstack_limit_register_:
      {
	int reg = decode_reg_name (arg);
	if (reg < 0)
	  error ("unrecognized register name \"%s\"", arg);
	else
	  stack_limit_rtx = gen_rtx_REG (Pmode, reg);
      }
      break;

    case OPT_fstack_limit_symbol_:
      stack_limit_rtx = gen_rtx_SYMBOL_REF (Pmode, ggc_strdup (arg));
      break;

    case OPT_ftree_vectorizer_verbose_:
      vect_set_verbosity_level (arg);
      break;

    case OPT_ftls_model_:
      if (!strcmp (arg, "global-dynamic"))
	opts->x_flag_tls_default = TLS_MODEL_GLOBAL_DYNAMIC;
      else if (!strcmp (arg, "local-dynamic"))
	opts->x_flag_tls_default = TLS_MODEL_LOCAL_DYNAMIC;
      else if (!strcmp (arg, "initial-exec"))
	opts->x_flag_tls_default = TLS_MODEL_INITIAL_EXEC;
      else if (!strcmp (arg, "local-exec"))
	opts->x_flag_tls_default = TLS_MODEL_LOCAL_EXEC;
      else
	warning (0, "unknown tls-model \"%s\"", arg);
      break;

    case OPT_fira_algorithm_:
      if (!strcmp (arg, "CB"))
	opts->x_flag_ira_algorithm = IRA_ALGORITHM_CB;
      else if (!strcmp (arg, "priority"))
	opts->x_flag_ira_algorithm = IRA_ALGORITHM_PRIORITY;
      else
	warning (0, "unknown ira algorithm \"%s\"", arg);
      break;

    case OPT_fira_region_:
      if (!strcmp (arg, "one"))
	opts->x_flag_ira_region = IRA_REGION_ONE;
      else if (!strcmp (arg, "all"))
	opts->x_flag_ira_region = IRA_REGION_ALL;
      else if (!strcmp (arg, "mixed"))
	opts->x_flag_ira_region = IRA_REGION_MIXED;
      else
	warning (0, "unknown ira region \"%s\"", arg);
      break;

    case OPT_g:
      set_debug_level (NO_DEBUG, DEFAULT_GDB_EXTENSIONS, arg);
      break;

    case OPT_gcoff:
      set_debug_level (SDB_DEBUG, false, arg);
      break;

    case OPT_gdwarf_:
      if (value < 2 || value > 4)
	error ("dwarf version %d is not supported", value);
      else
	dwarf_version = value;
      set_debug_level (DWARF2_DEBUG, false, "");
      break;

    case OPT_ggdb:
      set_debug_level (NO_DEBUG, 2, arg);
      break;

    case OPT_gstabs:
    case OPT_gstabs_:
      set_debug_level (DBX_DEBUG, code == OPT_gstabs_, arg);
      break;

    case OPT_gvms:
      set_debug_level (VMS_DEBUG, false, arg);
      break;

    case OPT_gxcoff:
    case OPT_gxcoff_:
      set_debug_level (XCOFF_DEBUG, code == OPT_gxcoff_, arg);
      break;

    case OPT_pedantic_errors:
      opts->x_pedantic = 1;
      dc->pedantic_errors = 1;
      break;

    case OPT_flto:
      opts->x_flag_lto = "";
      break;

    case OPT_w:
      dc->dc_inhibit_warnings = true;
      break;

    case OPT_fmax_errors_:
      dc->max_errors = value;
      break;

    case OPT_fuse_linker_plugin:
      /* No-op. Used by the driver and passed to us because it starts with f.*/
      break;

    default:
      /* If the flag was handled in a standard way, assume the lack of
	 processing here is intentional.  */
      gcc_assert (option_flag_var (scode, opts));
      break;
    }

  return true;
}

/* Handle --param NAME=VALUE.  */
static void
handle_param (struct gcc_options *opts, struct gcc_options *opts_set,
	      const char *carg)
{
  char *equal, *arg;
  int value;

  arg = xstrdup (carg);
  equal = strchr (arg, '=');
  if (!equal)
    error ("%s: --param arguments should be of the form NAME=VALUE", arg);
  else
    {
      value = integral_argument (equal + 1);
      if (value == -1)
	error ("invalid --param value %qs", equal + 1);
      else
	{
	  *equal = '\0';
	  set_param_value (arg, value,
			   opts->x_param_values, opts_set->x_param_values);
	}
    }

  free (arg);
}

/* Used to set the level of strict aliasing warnings in OPTS,
   when no level is specified (i.e., when -Wstrict-aliasing, and not
   -Wstrict-aliasing=level was given).
   ONOFF is assumed to take value 1 when -Wstrict-aliasing is specified,
   and 0 otherwise.  After calling this function, wstrict_aliasing will be
   set to the default value of -Wstrict_aliasing=level, currently 3.  */
void
set_Wstrict_aliasing (struct gcc_options *opts, int onoff)
{
  gcc_assert (onoff == 0 || onoff == 1);
  if (onoff != 0)
    opts->x_warn_strict_aliasing = 3;
  else
    opts->x_warn_strict_aliasing = 0;
}

/* The following routines are useful in setting all the flags that
   -ffast-math and -fno-fast-math imply.  */
static void
set_fast_math_flags (struct gcc_options *opts, int set)
{
  opts->x_flag_unsafe_math_optimizations = set;
  set_unsafe_math_optimizations_flags (opts, set);
  opts->x_flag_finite_math_only = set;
  opts->x_flag_errno_math = !set;
  if (set)
    {
      opts->x_flag_signaling_nans = 0;
      opts->x_flag_rounding_math = 0;
      opts->x_flag_cx_limited_range = 1;
    }
}

/* When -funsafe-math-optimizations is set the following
   flags are set as well.  */
static void
set_unsafe_math_optimizations_flags (struct gcc_options *opts, int set)
{
  opts->x_flag_trapping_math = !set;
  opts->x_flag_signed_zeros = !set;
  opts->x_flag_associative_math = set;
  opts->x_flag_reciprocal_math = set;
}

/* Return true iff flags are set as if -ffast-math.  */
bool
fast_math_flags_set_p (void)
{
  return (!flag_trapping_math
	  && flag_unsafe_math_optimizations
	  && flag_finite_math_only
	  && !flag_signed_zeros
	  && !flag_errno_math);
}

/* Return true iff flags are set as if -ffast-math but using the flags stored
   in the struct cl_optimization structure.  */
bool
fast_math_flags_struct_set_p (struct cl_optimization *opt)
{
  return (!opt->x_flag_trapping_math
	  && opt->x_flag_unsafe_math_optimizations
	  && opt->x_flag_finite_math_only
	  && !opt->x_flag_signed_zeros
	  && !opt->x_flag_errno_math);
}

/* Handle a debug output -g switch.  EXTENDED is true or false to support
   extended output (2 is special and means "-ggdb" was given).  */
static void
set_debug_level (enum debug_info_type type, int extended, const char *arg)
{
  static bool type_explicit;

  use_gnu_debug_info_extensions = extended;

  if (type == NO_DEBUG)
    {
      if (write_symbols == NO_DEBUG)
	{
	  write_symbols = PREFERRED_DEBUGGING_TYPE;

	  if (extended == 2)
	    {
#ifdef DWARF2_DEBUGGING_INFO
	      write_symbols = DWARF2_DEBUG;
#elif defined DBX_DEBUGGING_INFO
	      write_symbols = DBX_DEBUG;
#endif
	    }

	  if (write_symbols == NO_DEBUG)
	    warning (0, "target system does not support debug output");
	}
    }
  else
    {
      /* Does it conflict with an already selected type?  */
      if (type_explicit && write_symbols != NO_DEBUG && type != write_symbols)
	error ("debug format \"%s\" conflicts with prior selection",
	       debug_type_names[type]);
      write_symbols = type;
      type_explicit = true;
    }

  /* A debug flag without a level defaults to level 2.  */
  if (*arg == '\0')
    {
      if (!debug_info_level)
	debug_info_level = DINFO_LEVEL_NORMAL;
    }
  else
    {
      int argval = integral_argument (arg);
      if (argval == -1)
	error ("unrecognised debug output level \"%s\"", arg);
      else if (argval > 3)
	error ("debug output level %s is too high", arg);
      else
	debug_info_level = (enum debug_info_level) argval;
    }
}

/* Return 1 if option OPT_IDX is enabled in OPTS, 0 if it is disabled,
   or -1 if it isn't a simple on-off switch.  */

int
option_enabled (int opt_idx, void *opts)
{
  const struct cl_option *option = &(cl_options[opt_idx]);
  struct gcc_options *optsg = (struct gcc_options *) opts;
  void *flag_var = option_flag_var (opt_idx, optsg);

  if (flag_var)
    switch (option->var_type)
      {
      case CLVC_BOOLEAN:
	return *(int *) flag_var != 0;

      case CLVC_EQUAL:
	return *(int *) flag_var == option->var_value;

      case CLVC_BIT_CLEAR:
	return (*(int *) flag_var & option->var_value) == 0;

      case CLVC_BIT_SET:
	return (*(int *) flag_var & option->var_value) != 0;

      case CLVC_STRING:
	break;
      }
  return -1;
}

/* Fill STATE with the current state of option OPTION in OPTS.  Return
   true if there is some state to store.  */

bool
get_option_state (struct gcc_options *opts, int option,
		  struct cl_option_state *state)
{
  void *flag_var = option_flag_var (option, opts);

  if (flag_var == 0)
    return false;

  switch (cl_options[option].var_type)
    {
    case CLVC_BOOLEAN:
    case CLVC_EQUAL:
      state->data = flag_var;
      state->size = sizeof (int);
      break;

    case CLVC_BIT_CLEAR:
    case CLVC_BIT_SET:
      state->ch = option_enabled (option, opts);
      state->data = &state->ch;
      state->size = 1;
      break;

    case CLVC_STRING:
      state->data = *(const char **) flag_var;
      if (state->data == 0)
	state->data = "";
      state->size = strlen ((const char *) state->data) + 1;
      break;
    }
  return true;
}

/* Enable (or disable if VALUE is 0) a warning option ARG (language
   mask LANG_MASK, option handlers HANDLERS) as an error for option
   structures OPTS and OPTS_SET, diagnostic context DC (possibly
   NULL), location LOC.  This is used by -Werror=.  */

static void
enable_warning_as_error (const char *arg, int value, unsigned int lang_mask,
			 const struct cl_option_handlers *handlers,
			 struct gcc_options *opts,
			 struct gcc_options *opts_set,
			 location_t loc, diagnostic_context *dc)
{
  char *new_option;
  int option_index;

  new_option = XNEWVEC (char, strlen (arg) + 2);
  new_option[0] = 'W';
  strcpy (new_option + 1, arg);
  option_index = find_opt (new_option, lang_mask);
  if (option_index == OPT_SPECIAL_unknown)
    {
      error ("-Werror=%s: no option -%s", arg, new_option);
    }
  else
    {
      const diagnostic_t kind = value ? DK_ERROR : DK_WARNING;

      control_warning_option (option_index, (int) kind, value,
			      loc, lang_mask,
			      handlers, opts, opts_set, dc);
    }
  free (new_option);
}

/* Return malloced memory for the name of the option OPTION_INDEX
   which enabled a diagnostic (context CONTEXT), originally of type
   ORIG_DIAG_KIND but possibly converted to DIAG_KIND by options such
   as -Werror.  */

char *
option_name (diagnostic_context *context, int option_index,
	     diagnostic_t orig_diag_kind, diagnostic_t diag_kind)
{
  if (option_index)
    {
      /* A warning classified as an error.  */
      if ((orig_diag_kind == DK_WARNING || orig_diag_kind == DK_PEDWARN)
	  && diag_kind == DK_ERROR)
	return concat (cl_options[OPT_Werror_].opt_text,
		       /* Skip over "-W".  */
		       cl_options[option_index].opt_text + 2,
		       NULL);
      /* A warning with option.  */
      else
	return xstrdup (cl_options[option_index].opt_text);
    }
  /* A warning without option classified as an error.  */
  else if (orig_diag_kind == DK_WARNING || orig_diag_kind == DK_PEDWARN
	   || diag_kind == DK_WARNING)
    {
      if (context->warning_as_error_requested)
	return xstrdup (cl_options[OPT_Werror].opt_text);
      else
	return xstrdup (_("enabled by default"));
    }
  else
    return NULL;
}
