"""
A simple caching system for nevow pages.

The basic idea is to monkeypatch the request object in order to
snarf content and headers.
"""

#c Copyright 2008-2020, the GAVO project
#c
#c This program is free software, covered by the GNU GPL.  See the
#c COPYING file in the source distribution.


import time

from twisted.web import http
from twisted.web import resource

from gavo import base
from gavo import utils


def instrumentRequestForCaching(request, finishAction):
	"""changes request such that finishAction is called with the request and
	the content written for a successful page render.
	"""
	builder = CacheItemBuilder(finishAction)
	origWrite, origFinish = request.write, request.finish

	def write(content):
		builder.addContent(content)
		return origWrite(content)

	def finish():
		builder.finish(request)
		return origFinish()

	request.write = write
	request.finish = finish


class CacheItemBuilder(object):
	"""an aggregator for web pages as they are written.

	On successful page generation an function is called with
	the request and the content written as arguments.
	"""
	def __init__(self, finishAction):
		self.finishAction = finishAction
		self.contentBuffer = []
	
	def addContent(self, data):
		self.contentBuffer.append(data)
	
	def finish(self, request):
		try:
			if request.code==200:
				self.finishAction(request, b"".join(self.contentBuffer))
		except Exception:
			base.ui.notifyError("Exception while building cache item.")


class CachedPage(resource.Resource):
	"""A piece of cached content.

	This is built with the content to return, the headers, and a
	unix timestamp for the last modification time (as applicable).
	This is enough for it to work as a nevow resource (i.e., run
	a renderHTTP method.

	For cache management, this has a lastUsed unix timestamp that is bumped
	for each renderHTTP call, and a size attribute that gives the length
	of the content.
	"""
	def __init__(self, content, headers, lastModified):
		self.content = content
		self.size = len(content)
		self.creationStamp = time.time()
		headers.setRawHeaders("x-cache-creation", [str(self.creationStamp)])
		self.changeStamp = self.lastModified = lastModified
		if headers.hasHeader("last-modified"):
			headers.removeHeader("last-modified")
		self.headers = headers
		self.lastUsed = None

	def render(self, request):
		self.lastUsed = time.time()
		if self.lastModified:
			if request.setLastModified(self.lastModified)==http.CACHED:
				return b""
		for key, values in self.headers.getAllRawHeaders():
			request.responseHeaders.setRawHeaders(key, values)
		request.responseHeaders.setRawHeaders('date', [utils.formatRFC2616Date()])
		return self.content


def enterIntoCacheAs(key, destDict):
	"""returns a finishAction that enters a page into destDict under key.
	"""
	def finishAction(request, content):
		destDict[key] = CachedPage(content, request.responseHeaders, 
			request.lastModified)
	return finishAction
