""" Python module for isothermal flow of perfect gases with friction

The combined effect of friction and heat transfer in long ducts can be
condidered to isothermal flow model.

Here the reference value of the properties (p*, rho*,c*,To*,po* and Lmax)
are taken at the Mach number M* = 1/sqrt(gamma)

Assumptions:
1) One dimensional flow with friction and heat transfer
2) Constant area duct
3) Perfect gas with constant specific heat and Molecular weights
4) Isothermal flow

"""
"""
 * Copyright (C) 2006 Varun Hiremath.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * Authors: Varun Hiremath, Venkattraman A
 * Version 0.1
"""

from math import *
from scipy import optimize

class Isothermal:
    def __init__(self, g = 1.4):
        """ g is the value of gamma (ratio of specific heats), default = 1.4
        """
        self.g = 1.4

    def get_P_by_Pstarref_from_M(self,M):
        """ Returns the value of p/p* at Mach number M """
        return 1.0/self.g**(0.5)/M

    def get_rhostarref_by_rho_from_M(self,M):
        """ Returns the value of rho*/rho at Mach number M """
        return self.g**(0.5)*M

    def get_To_by_Tostarref_from_M(self,M) :
        """ Returns the value of To/To* at Mach number M """
        return 2.0*self.g/(3*self.g-1)*(1+(self.g-1)/2*M**2)

    def get_Po_by_Postarref_from_M(self,M):
        """ Returns the value of po/po* at Mach number M """
        return 1.0/sqrt(self.g)/M*(2.0*self.g/(3.0*self.g-1)*(1+(self.g-1)/2*M**2))**(self.g/(self.g-1))

    def get_4fLmax_by_D_from_M(self,M):
        """ Returns the value of 4f*(Lmax/D) at Mach number M """
        return (1-self.g*M**2)/self.g/M**2 + log(self.g*M**2)

    def get_M_from_P_by_Pstarref(self,P_by_Pstarref):
        """ Returns Mach number at p/p* equal to P_by_Pstarref """
        return 1.0/P_by_Pstarref/sqrt(self.g)

    def get_rhostarref_by_rho_from_P_by_Pstarref(self,P_by_Pstarref):
        """Returns rho*/rho at p/p* equal to P_by_Pstarref"""
        M = self.get_M_from_P_by_Pstarref(P_by_Pstarref)
        return self.get_rhostarref_by_rho_from_M(M)
    
    def get_To_by_Tostarref_from_P_by_Pstarref(self,P_by_Pstarref) :
        """Returns To/To* at p/p* equal to P_by_Pstarref"""
        M = self.get_M_from_P_by_Pstarref(P_by_Pstarref)
        return self.get_To_by_Tostarref_from_M(M)
    
    def get_Po_by_Postarref_from_P_by_Pstarref(self,P_by_Pstarref):
        """Returns Po/Po* at p/p* equal to P_by_Pstarref"""
        M = self.get_M_from_P_by_Pstarref(P_by_Pstarref)
        return self.get_Po_by_Postarref_from_M(M)
    
    def get_4fLmax_by_D_from_P_by_Pstarref(self,P_by_Pstarref):
        """Returns 4f*Lmax/D at p/p* equal to P_by_Pstarref"""
        M = self.get_M_from_P_by_Pstarref(P_by_Pstarref)
        return self.get_4fLmax_by_D_from_M(M)
    
    def get_M_from_rhostarref_by_rho(self,rhostarref_by_rho):
        """ Returns Mach number at rho*/rho equal to rhostarref_by_rho """
        return rhostarref_by_rho/sqrt(self.g)
    
    def get_P_by_Pstarref_from_rhostarref_by_rho(self,rhostarref_by_rho):
        """ Returns p/p* at rho*/rho equal to rhostarref_by_rho """
        M = self.get_M_from_rhostarref_by_rho(rhostarref_by_rho)
        return self.get_P_by_Pstarref_from_M(M)
    
    def get_To_by_Tostarref_from_rhostarref_by_rho(self,rhostarref_by_rho) :
        """ Returns To/To* at rho*/rho equal to rhostarref_by_rho """
        M = self.get_M_from_rhostarref_by_rho(rhostarref_by_rho)
        return self.get_To_by_Tostarref_from_M(M)

    def get_Po_by_Postarref_from_rhostarref_by_rho(self,rhostarref_by_rho):
        """ Returns po/po* at rho*/rho equal to rhostarref_by_rho """
        M = self.get_M_from_rhostarref_by_rho(rhostarref_by_rho)
        return self.get_Po_by_Postarref_from_M(M)
    
    def get_4fLmax_by_D_from_rhostarref_by_rho(self,rhostarref_by_rho):
        """ Returns 4f*Lmax/D at rho*/rho equal to rhostarref_by_rho """
        M = self.get_M_from_rhostarref_by_rho(rhostarref_by_rho)
        return self.get_4fLmax_by_D_from_M(M)

    def get_M_from_To_by_Tostarref(self,To_by_Tostarref):
        """ Returns Mach number at To/To* equal to To_by_Tostarref """
        return (((3*self.g-1)/2/self.g*To_by_Tostarref-1)*2/(self.g-1))**0.5
    
    def get_P_by_Pstarref_from_To_by_Tostarref(self,To_by_Tostarref):
        """ Returns p/p* at To/To* equal to To_by_Tostarref """
        M = self.get_M_from_To_by_Tostarref(To_by_Tostarref)
        return self.get_P_by_Pstarref_from_M(M)
    
    def get_rhostarref_by_rho_from_To_by_Tostarref(self,To_by_Tostarref):
        """ Returns rho*/rho at To/To* equal to To_by_Tostarref """
        M = self.get_M_from_To_by_Tostarref(To_by_Tostarref)
        return self.get_rhostarref_by_rho_from_M(M)
    
    def get_Po_by_Postarref_from_To_by_Tostarref(self,To_by_Tostarref):
        """ Returns po/po* at To/To* equal to To_by_Tostarref """
        M = self.get_M_from_To_by_Tostarref(To_by_Tostarref)
        return self.get_Po_by_Postarref_from_M(M)
    
    def get_4fLmax_by_D_from_To_by_Tostarref(self,To_by_Tostarref):
        """ Returns 4f*Lmax/D at To/To* equal to To_by_Tostarref """
        M = self.get_M_from_To_by_Tostarref(To_by_Tostarref)
        return self.get_4fLmax_by_D_from_M(M)
    
    def get_M_from_Po_by_Postarref(self,Po_by_Postarref):
        """ Returns Mach number at po/po* equal to Po_by_Postarref """
        return (optimize.bisect(lambda M : self.get_Po_by_Postarref_from_M(M) - Po_by_Postarref, 0.001 , 1.0/sqrt(self.g)), optimize.bisect(lambda M : self.get_Po_by_Postarref_from_M(M) - Po_by_Postarref, 1.0/(sqrt(self.g), 100.0)))
    
    def get_P_by_Pstarref_from_Po_by_Postarref(self,Po_by_Postarref):
        """ Returns p/p* at po/po* equal to Po_by_Postarref """
        M = self.get_M_from_Po_by_Postarref(Po_by_Postarref)
        return (self.get_P_by_Pstarref_from_M(M[0]),self.get_P_by_Pstarref_from_M(M[1]))
    
    def get_rhostarref_by_rho_from_Po_by_Postarref(self,Po_by_Postarref):
        """ Returns rho*/rho at po/po* equal to Po_by_Postarref """
        M = self.get_M_from_Po_by_Postarref(Po_by_Postarref)
        return (self.get_rhostarref_by_rho_from_M(M[0]),self.get_rhostarref_by_rho_from_M(M[1]))

    def get_To_by_Tostarref_from_Po_by_Postarref(self,Po_by_Postarref) :
        """ Returns To/To* at po/po* equal to Po_by_Postarref """
        M = self.get_M_from_Po_by_Postarref(Po_by_Postarref)
        return (self.get_To_by_Tostarref_from_M(M[0]),self.get_To_by_Tostarref_from_M(M[1]))

    def get_4fLmax_by_D_from_Po_by_Postarref(self,Po_by_Postarref):
        """ Returns 4f*Lmax/D at po/po* equal to Po_by_Postarref """
        M = self.get_M_from_Po_by_Postarref(Po_by_Postarref)
        return (self.get_4fLmax_by_D_from_M(M[0]),self.get_4fLmax_by_D_from_M(M[1]))
    
    def get_M_from_4fLmax_by_D(self,fLmax_by_D):
        """ Returns Mach number at 4f*(Lmax/D) equal to fLmax_by_D """
        return (optimize.bisect(lambda M : self.get_4fLmax_by_D_from_M(M ) - fLmax_by_D, 0.001, 1.0/sqrt(self.g)), optimize.bisect(lambda M : self.get_4fLmax_by_D_from_M(M ) - fLmax_by_D, 1.0/sqrt(self.g), 100.0)) 
    
    def get_P_by_Pstarref_from_4fLmax_by_D(self,fLmax_by_D):
        """ Returns p/p* at 4f*(Lmax/D) equal to fLmax_by_D """
        M = self.get_M_from_4fLmax_by_D(fLmax_by_D)
        return (self.get_P_by_Pstarref_from_M(M[0]),self.get_P_by_Pstarref_from_M(M[1]))
    
    def get_rhostarref_by_rho_from_4fLmax_by_D(self,fLmax_by_D):
        """ Returns rho*/rho at 4f*(Lmax/D) equal to fLmax_by_D """
        M = self.get_M_from_4fLmax_by_D(fLmax_by_D)
        return (self.get_rhostarref_by_rho_from_M(M[0]),self.get_rhostarref_by_rho_from_M(M[1]))
    
    def get_To_by_Tostarref_from_4fLmax_by_D(self,fLmax_by_D) :
        """ Returns To/To* at 4f*(Lmax/D) equal to fLmax_by_D """
        M = self.get_M_from_4fLmax_by_D(fLmax_by_D)
        return (self.get_To_by_Tostarref_from_M(M[0]),self.get_To_by_Tostarref_from_M(M[1]))
    
    def get_Po_by_Postarref_from_4fLmax_by_D(self,fLmax_by_D):
        """ Returns po/po* at 4f*(Lmax/D) equal to fLmax_by_D """
        M = self.get_M_from_4fLmax_by_D(fLmax_by_D)
        return (self.get_Po_by_Postarref_from_M(M[0]),self.get_Po_by_Postarref_from_M(M[1]))
