/******************************************
 *
 * $GAMGI/src/chem/gamgi_chem_orbital.c
 *
 * Copyright (C) 2012 Carlos Pereira
 *
 * Distributed under the terms of the GNU
 * General Public License: $GAMGI/LICENSE
 *
 */

#include "gamgi_engine.h"
#include "gamgi_mesa.h"
#include "gamgi_math.h"
#include "gamgi_chem.h"

#define TOLERANCE_R 1E-8

#define BITMASK(b) (1 << ((b) % CHAR_BIT))
#define BITSLOT(b) ((b) / CHAR_BIT)
#define BITSET(a, b) ((a)[BITSLOT(b)] |= BITMASK(b))
#define BITCLEAR(a, b) ((a)[BITSLOT(b)] &= ~BITMASK(b))
#define BITTEST(a, b) ((a)[BITSLOT(b)] & BITMASK(b))
#define BITNSLOTS(nb) ((nb + CHAR_BIT - 1) / CHAR_BIT)

typedef struct _static_surface {

int number;

/***********************************************************
 * scanning current state:                                 *
 * 1) save x,y,z,f coordinates and wavefunction per vertex *
 * 2) save global [8] and vertices [0..7] state            *
 * 3) save number of triangles built                       *
 ***********************************************************/

double cache_f[8 * 4];
int    cache_v[8 * 1];
int    cache_s;
int triangles;

/*******************************************
 * set which octant faces need to be built *
 *******************************************/

int face_a[3][2][2];
int face_b[2][3][2];
int face_c[2][2][3];

/***************************************************
 * for current octant, save scanning data:         *
 * 1) current cube int coordinates                 *
 * 2) octant origin double coordinates             *
 * 3) bit array to flag points above the density   *
 * 4) bit array to flag points outside the orbital *
 * 5) bit arrays needed bits                       *
 * 6) bit arrays needed chars                      *
 ***************************************************/

int a, b, c;
double origin[3];
char *above, *outside;
int n_points, n_char, n_bit;

/********************************************
 * pointers to functions building triangles *
 * for each tetrahedra and square state     *
 ********************************************/

void (*f2d[16]) (struct _static_surface *surface, int *v0, int *v1, int *v2, int *v3);
void (*f3d[16]) (struct _static_surface *surface, int *v0, int *v1, int *v2, int *v3);

/********************************
 * save accuracy scanning data: *
 * 1) sampling cube width       *
 * 2) number of sampling cubes  *
 *    per octant per direction  *
 ********************************/

double width;
int slices;

/***************************
 * save wave function data *
 ***************************/

gamgi_radial function_r;
gamgi_angular function_a;
double constant, ho;

/************************
 * save user input data *
 ************************/

gamgi_orbital *orbital;
int *octants;

/****************************************
 * save current triangle array position *
 ****************************************/

int offset; } static_surface;

typedef void (*gamgi_tetra) (void *surface, int *v0, int *v1, int *v2, int *v3);

static double static_1s (double r)
{
/******************
 * n = 1, l = 0   *
 ******************/

return 1;
}

static double static_2s (double r)
{
/****************
 * n = 2, l = 0 *
 ****************/

return 2 - r;
}

static double static_2p (double r)
{
/****************
 * n = 2, l = 1 *
 ****************/

return r;
}

static double static_3s (double r)
{
double r2 = r * r;

/****************
 * n = 3, l = 0 *
 ****************/

return 6 - 6 * r + r2;
}

static double static_3p (double r)
{
/****************
 * n = 3, l = 1 *
 ****************/

return r * (4 - r);
}

static double static_3d (double r)
{
double r2 = r * r;

/****************
 * n = 3, l = 2 *
 ****************/

return r2;
}

static double static_4s (double r)
{
double r2 = r * r;
double r3 = r2 * r;

/****************
 * n = 4, l = 0 *
 ****************/

return (24 - 36 * r + 12 * r2 - r3);
}

static double static_4p (double r)
{
double r2 = r * r;

/****************
 * n = 4, l = 1 *
 ****************/ 

return r * (20 - 10 * r + r2); 
}

static double static_4d (double r)
{
double r2 = r * r;

/****************
 * n = 4, l = 2 *
 ****************/

return (6 - r) * r2;
}

static double static_4f (double r)
{
double r3 = r * r * r;

/****************
 * n = 4, l = 3 *
 ****************/

return r3;
}

static double static_5s (double r)
{
double r2 = r * r;
double r3 = r2 * r;
double r4 = r3 * r;

/****************
 * n = 5, l = 0 *
 ****************/

return (120 - 240 * r + 120 * r2 - 20 * r3 + r4);
}

static double static_5p (double r)
{
double r2 = r * r;
double r3 = r2 * r;

/****************
 * n = 5, l = 1 *
 ****************/

return (120 - 90 * r + 18 * r2 - r3) * r;
}

static double static_5d (double r)
{
double r2 = r * r;

/****************
 * n = 5, l = 2 *
 ****************/

return (42 - 14 * r + r2) * r2;
}

static double static_5f (double r)
{
double r3 = r * r * r;

/****************
 * n = 5, l = 3 *
 ****************/

return (8 - r) * r3;
}

static double static_5g (double r)
{
double r2 = r * r;
double r4 = r2 * r2;

/****************
 * n = 5, l = 4 *
 ****************/

return r4;
}

static double static_6s (double r)
{
double r2 = r * r;
double r3 = r2 * r;
double r4 = r3 * r;
double r5 = r4 * r;

/****************
 * n = 6, l = 0 *
 ****************/

return (720 - 1800 * r + 1200 * r2 - 300 * r3 + 30 * r4 - r5);
}

static double static_6p (double r)
{
double r2 = r * r;
double r3 = r2 * r;
double r4 = r3 * r;

/****************
 * n = 6, l = 1 *
 ****************/

return (840 - 840 * r + 252 * r2 - 28 * r3 + r4) * r;
}

static double static_6d (double r)
{
double r2 = r * r;
double r3 = r2 * r;

/****************
 * n = 6, l = 2 *
 ****************/

return (336 - 168 * r + 24 * r2 - r3) * r2;
}

static double static_6f (double r)
{
double r2 = r * r;
double r3 = r2 * r;

/****************
 * n = 6, l = 3 *
 ****************/

return (72 - 18 * r + r2) * r3;
}

static double static_6g (double r)
{
double r2 = r * r;
double r4 = r2 * r2;

/****************
 * n = 6, l = 4 *
 ****************/

return (10 - r) * r4;
}

static double static_6h (double r)
{
double r2 = r * r;
double r4 = r2 * r2;
double r5 = r4 * r;

/****************
 * n = 6, l = 5 *
 ****************/

return r5;
}

static void static_radial (int n, int l, double *c, gamgi_radial *f)
{
if (n == 1) { *c = 0.5; *f = static_1s; }

if (n == 2)
  {
  if (l == 0) { *c = 2 * sqrt (2); *f = static_2s; }
  if (l == 1) { *c = 2 * sqrt (6); *f = static_2p; }
  }

if (n == 3)
  {
  if (l == 0) { *c = 9 * sqrt (3);  *f = static_3s; }
  if (l == 1) { *c = 9 * sqrt (6);  *f = static_3p; }
  if (l == 2) { *c = 9 * sqrt (30); *f = static_3d; }
  }

if (n == 4)
  {
  if (l == 0) { *c = 96;             *f = static_4s; }
  if (l == 1) { *c = 32 * sqrt (15); *f = static_4p; }
  if (l == 2) { *c = 96 * sqrt (5);  *f = static_4d; }
  if (l == 3) { *c = 96 * sqrt (35); *f = static_4f; }
  }

if (n == 5)
  {
  if (l == 0) { *c = 300 * sqrt (5);  *f = static_5s; }
  if (l == 1) { *c = 150 * sqrt (30); *f = static_5p; }
  if (l == 2) { *c = 150 * sqrt (70); *f = static_5d; }
  if (l == 3) { *c = 300 * sqrt (70); *f = static_5f; }
  if (l == 4) { *c = 900 * sqrt (70); *f = static_5g; }
  }

if (n == 6)
  {
  if (l == 0) { *c = 2160 * sqrt (6);   *f = static_6s; }
  if (l == 1) { *c = 432 * sqrt (210);  *f = static_6p; }
  if (l == 2) { *c = 864 * sqrt (105);  *f = static_6d; }
  if (l == 3) { *c = 2592 * sqrt (35);  *f = static_6f; }
  if (l == 4) { *c = 12960 * sqrt (7);  *f = static_6g; }
  if (l == 5) { *c = 12960 * sqrt (77); *f = static_6h; }
  }

*c = 1 / *c;
}

static double static_s (double x, double y, double z, double r)
{
/****************
 * l = 0, m = 0 *
 ****************/

return 1;
}

static double static_pz (double x, double y, double z, double r)
{
/****************
 * l = 1, m = 0 *
 ****************/

return z / r;
}

static double static_px (double x, double y, double z, double r)
{
/*****************
 * l = 1, m = +1 *
 *****************/

return x / r;
}

static double static_py (double x, double y, double z, double r)
{
/*****************
 * l = 1, m = -1 *
 *****************/

return y / r;
}

static double static_dz2 (double x, double y, double z, double r)
{
double z2 = z * z;
double r2 = r * r;

/****************
 * l = 2, m = 0 *
 ****************/

return (3 * z2 - r2) / r2;
}

static double static_dxz (double x, double y, double z, double r)
{
double r2 = r * r;

/****************
 * l = 2, m = 1 *
 ****************/

return (x * z) / r2;
}

static double static_dyz (double x, double y, double z, double r)
{
double r2 = r * r;

/****************
 * l = 2, m = -1 *
 ****************/

return (y * z) / r2;
}

static double static_dx2_y2 (double x, double y, double z, double r)
{
double x2 = x * x;
double y2 = y * y;
double r2 = r * r;

/****************
 * l = 2, m = 2 *
 ****************/

return (x2 - y2) / r2;
}

static double static_dxy (double x, double y, double z, double r)
{
double r2 = r * r;

/****************
 * l = 2, m = -2 *
 ****************/

return (x * y) / r2;
}

static double static_fz3 (double x, double y, double z, double r)
{
double z2 = z * z;
double r2 = r * r;
double r3 = r2 * r;

/****************
 * l = 3, m = 0 *
 ****************/

return z * (5 * z2 - 3 * r2) / r3;
}

static double static_fxz2 (double x, double y, double z, double r)
{
double z2 = z * z;
double r2 = r * r;
double r3 = r2 * r;

/****************
 * l = 3, m = 1 *
 ****************/

return x * (5 * z2 - r2) / r3;
}

static double static_fyz2 (double x, double y, double z, double r)
{
double z2 = z * z;
double r2 = r * r;
double r3 = r2 * r;

/*****************
 * l = 3, m = -1 *
 *****************/

return y * (5 * z2 - r2) / r3;
}

static double static_fz_x2_y2 (double x, double y, double z, double r)
{
double x2 = x * x;
double y2 = y * y;
double r3 = r * r * r;

/****************
 * l = 3, m = 2 *
 ****************/

return z * (x2 - y2) / r3;
}

static double static_fxyz (double x, double y, double z, double r)
{
double r3 = r * r * r;

/*****************
 * l = 3, m = -2 *
 *****************/

return (x * y * z) / r3;
}

static double static_fx_x2_3y2 (double x, double y, double z, double r)
{
double x2 = x * x;
double y2 = y * y;
double r3 = r * r * r;

/****************
 * l = 3, m = 3 *
 ****************/

return x * (x2 - 3 * y2) / r3;
}

static double static_fy_3x2_y2 (double x, double y, double z, double r)
{
double x2 = x * x;
double y2 = y * y;
double r3 = r * r * r;

/*****************
 * l = 3, m = -3 *
 *****************/

return y * (3 * x2 - y2) / r3;
}

static double static_gz4 (double x, double y, double z, double r)
{
double z2 = z * z;
double z4 = z2 * z2;
double r2 = r * r;
double r4 = r2 * r2;

/****************
 * l = 4, m = 0 *
 ****************/

return (35 * z4 - 30 * z2 * r2 + 3 * r4) / r4;
}

static double static_gz3x (double x, double y, double z, double r)
{
double z2 = z * z;
double r2 = r * r;
double r4 = r2 * r2;

/****************
 * l = 4, m = 1 *
 ****************/

return x * z * (7 * z2 - 3 * r2) / r4;
}

static double static_gz3y (double x, double y, double z, double r)
{
double z2 = z * z;
double r2 = r * r;
double r4 = r2 * r2;

/*****************
 * l = 4, m = -1 *
 *****************/

return y * z * (7 * z2 - 3 * r2) / r4;
}

static double static_gz2_x2_y2 (double x, double y, double z, double r)
{
double x2 = x * x;
double y2 = y * y;
double z2 = z * z;
double r2 = r * r;
double r4 = r2 * r2;

/****************
 * l = 4, m = 2 *
 ****************/

return (x2 - y2) * (7 * z2 - r2) / r4;
}

static double static_gz2_xy (double x, double y, double z, double r)
{
double z2 = z * z;
double r2 = r * r;
double r4 = r2 * r2;

/*****************
 * l = 4, m = -2 *
 *****************/

return x * y * (7 * z2 - r2) / r4;
}

static double static_gzx3 (double x, double y, double z, double r)
{
double x2 = x * x;
double y2 = y * y;
double r4 = r * r * r * r;

/*****************
 * l = 4, m = 3 *
 *****************/

return x * z * (x2 - 3 * y2) / r4;
}

static double static_gzy3 (double x, double y, double z, double r)
{
double x2 = x * x;
double y2 = y * y;
double r4 = r * r * r * r;

/*****************
 * l = 4, m = -3 *
 *****************/

return y * z * (3 * x2 - y2) / r4;
}

static double static_gx4_y4 (double x, double y, double z, double r)
{
double x2 = x * x;
double x4 = x2 * x2;
double y2 = y * y;
double y4 = y2 * y2;
double r4 = r * r * r * r;

/*****************
 * l = 4, m = 4 *
 *****************/

return (x4 + y4 - 6 * x2 * y2) / r4;
}

static double static_gxy_x2_y2 (double x, double y, double z, double r)
{
double x2 = x * x;
double y2 = y * y;
double r4 = r * r * r * r;

/*****************
 * l = 4, m = -4 *
 *****************/

return x * y * (x2 - y2) / r4;
}

double static_hz5 (double x, double y, double z, double r)
{
double z2 = z * z;
double z3 = z2 * z;
double z5 = z3 * z2;
double r2 = r * r;
double r4 = r2 * r2;
double r5 = r4 * r;

/****************
 * l = 5, m = 0 *
 ****************/

return (63 * z5 - 70 * z3 * r2 + 15 * z * r4) / r5;
}

double static_hz4x (double x, double y, double z, double r)
{
double z2 = z * z;
double z4 = z2 * z2;
double r2 = r * r;
double r4 = r2 * r2;
double r5 = r4 * r;

/****************
 * l = 5, m = 1 *
 ****************/

return x * (21 * z4 - 14 * z2 * r2 + r4) / r5;
}

double static_hz4y (double x, double y, double z, double r)
{
double z2 = z * z;
double z4 = z2 * z2;
double r2 = r * r;
double r4 = r2 * r2;
double r5 = r4 * r;

/*****************
 * l = 5, m = -1 *
 *****************/

return y * (21 * z4 - 14 * z2 * r2 + r4) / r5;
}

double static_hz3_x2_y2 (double x, double y, double z, double r)
{
double x2 = x * x;
double y2 = y * y;
double z3 = z * z * z;
double r2 = r * r;
double r5 = r2 * r2 * r;

/****************
 * l = 5, m = 2 *
 ****************/

return (x2 - y2) * (3 * z3 - z * r2) / r5;
}

double static_hz3_xy (double x, double y, double z, double r)
{
double z3 = z * z * z;
double r2 = r * r;
double r5 = r2 * r2 * r;

/*****************
 * l = 5, m = -2 *
 *****************/

return x * y * (3 * z3 - z * r2) / r5;
}

double static_hz2x3 (double x, double y, double z, double r)
{
double x3 = x * x * x;
double y2 = y * y;
double z2 = z * z;
double r2 = r * r;
double r5 = r2 * r2 * r;

/****************
 * l = 5, m = 3 *
 ****************/

return (x3 - 3 * x * y2) * (9 * z2 - r2) / r5;
}

double static_hz2y3 (double x, double y, double z, double r)
{
double y3 = y * y * y;
double x2 = x * x;
double z2 = z * z;
double r2 = r * r;
double r5 = r2 * r2 * r;

/*****************
 * l = 5, m = -3 *
 *****************/

return (3 * x2 * y - y3) * (9 * z2 - r2) / r5;
}

double static_hz_x4_y4 (double x, double y, double z, double r)
{
double x2 = x * x;
double x4 = x2 * x2;
double y2 = y * y;
double y4 = y2 * y2;
double r2 = r * r;
double r5 = r2 * r2 * r;

/****************
 * l = 5, m = 4 *
 ****************/

return z * (x4 + y4 - 6 * x2 * y2) / r5;
}

double static_hxyz_x2_y2 (double x, double y, double z, double r)
{
double x2 = x * x;
double y2 = y * y;
double r2 = r * r;
double r5 = r2 * r2 * r;

/*****************
 * l = 5, m = -4 *
 *****************/

return x * y * z * (x2 - y2) / r5;
}

double static_hx5 (double x, double y, double z, double r)
{
double x2 = x * x;
double x3 = x2 * x;
double x5 = x3 * x2;
double y2 = y * y;
double y4 = y2 * y2;
double r2 = r * r;
double r5 = r2 * r2 * r;

/****************
 * l = 5, m = 5 *
 ****************/

return (x5 - 10 * x3 * y2 + 5 * x * y4) / r5;
}

double static_hy5 (double x, double y, double z, double r)
{
double x2 = x * x;
double x4 = x2 * x2;
double y2 = y * y;
double y3 = y2 * y;
double y5 = y3 * y2;
double r2 = r * r;
double r5 = r2 * r2 * r;

/*****************
 * l = 5, m = -5 *
 *****************/

return (y5 - 10 * x2 * y3 + 5 * x4 * y) / r5;
}

static void static_angular (int l, int m, double *c, gamgi_angular *f)
{
*f = NULL;

if (l == 0) { *c = 1; *f = static_s; }

if (l == 1)
  {
  if (m == -1) { *c = sqrt (3); *f = static_py; }
  if (m ==  0) { *c = sqrt (3); *f = static_pz; }
  if (m == +1) { *c = sqrt (3); *f = static_px; }
  }

if (l == 2)
  {
  if (m == -2) { *c = sqrt (15);       *f = static_dxy; }
  if (m == -1) { *c = sqrt (15);       *f = static_dyz; }
  if (m ==  0) { *c = sqrt (5 / 4.0);  *f = static_dz2; }
  if (m == +1) { *c = sqrt (15);       *f = static_dxz; }
  if (m == +2) { *c = sqrt (15 / 4.0); *f = static_dx2_y2; }
  }

if (l == 3)
  {
  if (m == -3) { *c = sqrt (35 / 8.0); *f = static_fy_3x2_y2; }
  if (m == -2) { *c = sqrt (105);       *f = static_fxyz; }
  if (m == -1) { *c = sqrt (21 / 8.0);  *f = static_fyz2; }
  if (m ==  0) { *c = sqrt (7 / 4.0);   *f = static_fz3; }
  if (m == +1) { *c = sqrt (21 / 8.0);  *f = static_fxz2; }
  if (m == +2) { *c = sqrt (105 / 4.0); *f = static_fz_x2_y2; }
  if (m == +3) { *c = sqrt (35 / 8.0);  *f = static_fx_x2_3y2; }
  }

if (l == 4)
  {
  if (m == -4) { *c = sqrt (315 / 4.0);  *f = static_gxy_x2_y2; }
  if (m == -3) { *c = sqrt (315 / 8.0);  *f = static_gzy3; }
  if (m == -2) { *c = sqrt (45 / 4.0);   *f = static_gz2_xy; }
  if (m == -1) { *c = sqrt (45 / 8.0);   *f = static_gz3y; }
  if (m ==  0) { *c = 3 / 8.0;           *f = static_gz4; }
  if (m == +1) { *c = sqrt (45 / 8.0);   *f = static_gz3x; }
  if (m == +2) { *c = sqrt (45 / 16.0);  *f = static_gz2_x2_y2; }
  if (m == +3) { *c = sqrt (315 / 8.0);  *f = static_gzx3; }
  if (m == +4) { *c = sqrt (315 / 64.0); *f = static_gx4_y4; }
  }

if (l == 5)
  {
  if (m == -5) { *c = sqrt (693 / 128.0); *f = static_hy5; }
  if (m == -4) { *c = sqrt (3465 / 4.0);  *f = static_hxyz_x2_y2; }
  if (m == -3) { *c = sqrt (385 / 128.0); *f = static_hz2y3; }
  if (m == -2) { *c = sqrt (1155 / 4.0);  *f = static_hz3_xy; }
  if (m == -1) { *c = sqrt (165 / 64.0);  *f = static_hz4y; }
  if (m ==  0) { *c = sqrt (11 / 64.0);   *f = static_hz5; }
  if (m == +1) { *c = sqrt (165 / 64.0);  *f = static_hz4x; }
  if (m == +2) { *c = sqrt (1155 / 16.0); *f = static_hz3_x2_y2; }
  if (m == +3) { *c = sqrt (385 / 128.0); *f = static_hz2x3; }
  if (m == +4) { *c = sqrt (3465 / 64.0); *f = static_hz_x4_y4; }
  if (m == +5) { *c = sqrt (693 / 128.0); *f = static_hx5; }
  }

*c /= sqrt (4.0 * GAMGI_MATH_PI);
}

double static_density (double r, double value, gamgi_angular function_a)
{
double theta, phi;
double x, y, z, s;
double end, step;
double f;

end = GAMGI_MATH_PI / 2.0;
step = GAMGI_MATH_PI / 20.0;

for (theta = 0.0; theta < end; theta += step)
  {
  z = r * cos (theta);
  s = r * sin (theta);

  for (phi = 0.0; phi < end; phi += step)
    {
    x = s * cos (phi);
    y = s * sin (phi);

    f = (*function_a) (x, y, z, r);
    if (value * f * f > GAMGI_CHEM_ORBITAL_DENSITY) return TRUE;
    }
  }

return FALSE;
}

void static_frame (gamgi_orbital *orbital)
{
double *points;
double radius;
int *lines;
int x, y, z, offset;

points = orbital->points;
radius = orbital->radius;

/**********************
 * add 8 frame points *
 **********************/

offset = 0;
for (z = -1; z <= 1; z += 2)
  {
  for (y = -1; y <= 1; y += 2)
    {
    for (x = -1; x <= 1; x += 2)
      {
      points[offset++] = radius * x;
      points[offset++] = radius * y;
      points[offset++] = radius * z;
      }
    }
  }

/*********************************************************
 * create array with 12 edges: edges x, edges y, edges z *
 *********************************************************/

orbital->lines = lines = (int *) malloc (25 * sizeof (int));
orbital->n_lines = 25;

offset = 0;
lines[offset++] = 12;

lines[offset++] = 0;
lines[offset++] = 1;
lines[offset++] = 2;
lines[offset++] = 3;
lines[offset++] = 4;
lines[offset++] = 5;
lines[offset++] = 6;
lines[offset++] = 7;

lines[offset++] = 0;
lines[offset++] = 2;
lines[offset++] = 1;
lines[offset++] = 3;
lines[offset++] = 4;
lines[offset++] = 6;
lines[offset++] = 5;
lines[offset++] = 7;

lines[offset++] = 0;
lines[offset++] = 4;
lines[offset++] = 1;
lines[offset++] = 5;
lines[offset++] = 2;
lines[offset++] = 6;
lines[offset++] = 3;
lines[offset++] = 7;
}

gamgi_bool static_wired (gamgi_orbital *orbital, int *octants, int seed)
{
gamgi_radial function_r;
gamgi_angular function_a;
double *points;
double max = RAND_MAX;
double za, r, ho, rho;
double density, radius, f, x, y, z;
double constant, constant_z, constant_r, constant_a;
int *dots;
int offset, offset_negative, offset_positive;
int sampling, n_points, i, bad; 
int a, b, c;

/*******************************************************
 * if required, add frame data to points, lines arrays *
 *******************************************************/

dots = orbital->dots;
density = orbital->density;
radius = orbital->radius;
sampling = orbital->sampling;

orbital->n_points = n_points = sampling + dots[0];
orbital->points = points = (double *) malloc (n_points * 3 * sizeof (double));
if (dots[0] > 0) static_frame (orbital);

static_radial (orbital->n, orbital->l, &constant_r, &function_r);
static_angular (orbital->l, orbital->m, &constant_a, &function_a);

za = orbital->charge / GAMGI_CHEM_A0;
ho = 2 * za / orbital->n;
constant_z = pow (za, 1.5);
constant = constant_z * constant_r * constant_a;

/***************************************************
 * start negative phase points from the beginning  *
 * of the array (after frame points if available)  *
 * and positive phase points from the end (even if *
 * the same color will be used to show both sets   *
 ***************************************************/

offset_positive = dots[0];
offset_negative = n_points - 1;

/******************************
 * set random number sequence *
 ******************************/

srand (seed);

/*******************************************
 * i_bad is just to prevent infinite loops *
 *******************************************/

i = bad = 0;
while (i < sampling && bad < GAMGI_CHEM_ORBITAL_BAD)
  {
  x = 2.0 * rand () / max - 1.0;
  y = 2.0 * rand () / max - 1.0;
  z = 2.0 * rand () / max - 1.0;
  r = x * x + y * y + z * z;
  if (r > 1.0) continue;

  a = b = c = 0;
  if (x > 0) a = 1;
  if (y > 0) b = 1;
  if (z > 0) c = 1;
  if (octants[4*c + 2*b + a] == FALSE) continue;

  r = radius * sqrt (r);
  rho = r * ho;
  x *= radius;
  y *= radius;
  z *= radius;

  f = constant * exp (-rho / 2) * (*function_r) (rho) * (*function_a) (x, y, z, r);
  if (f * f < density) { bad++; continue; }

  i++; bad = 0;
  if (f >= 0) offset = offset_positive++;
  if (f < 0) offset = offset_negative--;

  points[3*offset + 0] = x;
  points[3*offset + 1] = y;
  points[3*offset + 2] = z;
  }

/***************************************
 * set positive (dots[1]) and negative *
 * (dots[2]) number of density points  *
 ***************************************/
 
dots[1] = offset_positive - dots[0];
if (dots[2] == TRUE) dots[2] = n_points - 1 - offset_negative;
else dots[1] += n_points - 1 - offset_negative;
if (dots[0] + dots[1] + dots[2] == n_points) return TRUE;

return FALSE;
}

void static_clean (char *array, int n_char)
{
int n;

for (n = 0; n < n_char; n++) array[n] = 0;
}

int static_offset (int n_points, int a, int b, int c)
{
return c * n_points * n_points + b * n_points + a;
}

void static_above (static_surface *surface, int a, int b, int c)
{
gamgi_orbital *orbital;
char *above;
gamgi_radial function_r;
gamgi_angular function_a;
double *origin;
double za, r, ho, rho;
double density, radius, width, f, x, y, z;
double constant, constant_z, constant_r, constant_a;
int n_points, n;

/********************
 * get orbital data *
 ********************/

orbital = surface->orbital;

static_radial (orbital->n, orbital->l, &constant_r, &function_r);
static_angular (orbital->l, orbital->m, &constant_a, &function_a);

za = orbital->charge / GAMGI_CHEM_A0;
ho = 2 * za / orbital->n;
constant_z = pow (za, 1.5);
constant = constant_z * constant_r * constant_a;

/*****************
 * get user data *
 *****************/

density = orbital->density;
n_points = surface->n_points;
radius = orbital->radius;
width = surface->width;

surface->origin[0] = radius * (a - 1);
surface->origin[1] = radius * (b - 1);
surface->origin[2] = radius * (c - 1);
origin = surface->origin;

/*******************
 * set above array *
 *******************/

above = surface->above;
for (c = 0; c < n_points; c++)
  {
  z = origin[2] + c * width;
  for (b = 0; b < n_points; b++)
    {
    y = origin[1] + b * width;
    for (a = 0; a < n_points; a++)
      {
      x = origin[0] + a * width;
      r = x * x + y * y + z * z;
      if (r < TOLERANCE_R) r = TOLERANCE_R;
      r = sqrt (r);

      rho = r * ho;
      f = constant * exp (-rho / 2) * (*function_r) (rho) * (*function_a) (x, y, z, r);
      if (f * f < density) continue;

      n = c * n_points * n_points + b * n_points + a;
      BITSET(above, n);
      }
    }
  }
}

void static_outside (char *outside, char *above, int n_points, int a, int b, int c)
{
int bit;

/****************
 * mark a faces *
 ****************/

if (a == 0 || a == n_points - 1)
  {
  if (b > 0)
    {
    bit = static_offset (n_points, a, b - 1, c);
    if (BITTEST(outside, bit) == 0 && BITTEST(above, bit) == 0)
      {
      BITSET(outside, bit);
      static_outside (outside, above, n_points, a, b - 1, c);
      }
    }

  if (b < n_points - 1)
    {
    bit = static_offset (n_points, a, b + 1, c);
    if (BITTEST(outside, bit) == 0 && BITTEST(above, bit) == 0)
      {
      BITSET(outside, bit);
      static_outside (outside, above, n_points, a, b + 1, c);
      }
    }

  if (c > 0)
    {
    bit = static_offset (n_points, a, b, c - 1);
    if (BITTEST(outside, bit) == 0 && BITTEST(above, bit) == 0)
      {
      BITSET(outside, bit);
      static_outside (outside, above, n_points, a, b, c - 1);
      }
    }

  if (c < n_points - 1)
    {
    bit = static_offset (n_points, a, b, c + 1);
    if (BITTEST(outside, bit) == 0 && BITTEST(above, bit) == 0)
      {
      BITSET(outside, bit);
      static_outside (outside, above, n_points, a, b, c + 1);
      }
    }
  }

/****************
 * mark b faces *
 ****************/

if (b == 0 || b == n_points - 1)
  {
  if (a > 0)
    {
    bit = static_offset (n_points, a - 1, b, c);
    if (BITTEST(outside, bit) == 0 && BITTEST(above, bit) == 0)
      {
      BITSET(outside, bit);
      static_outside (outside, above, n_points, a - 1, b, c);
      }
    }

  if (a < n_points - 1)
    {
    bit = static_offset (n_points, a + 1, b, c);
    if (BITTEST(outside, bit) == 0 && BITTEST(above, bit) == 0)
      {
      BITSET(outside, bit);
      static_outside (outside, above, n_points, a + 1, b, c);
      }
    }

  if (c > 0)
    {
    bit = static_offset (n_points, a, b, c - 1);
    if (BITTEST(outside, bit) == 0 && BITTEST(above, bit) == 0)
      {
      BITSET(outside, bit);
      static_outside (outside, above, n_points, a, b, c - 1);
      }
    }

  if (c < n_points - 1)
    {
    bit = static_offset (n_points, a, b, c + 1);
    if (BITTEST(outside, bit) == 0 && BITTEST(above, bit) == 0)
      {
      BITSET(outside, bit);
      static_outside (outside, above, n_points, a, b, c + 1);
      }
    }
  }

/****************
 * mark c faces *
 ****************/

if (c == 0 || c == n_points - 1)
  {
  if (a > 0)
    {
    bit = static_offset (n_points, a - 1, b, c);
    if (BITTEST(outside, bit) == 0 && BITTEST(above, bit) == 0)
      {
      BITSET(outside, bit);
      static_outside (outside, above, n_points, a - 1, b, c);
      }
    }

  if (a < n_points - 1)
    {
    bit = static_offset (n_points, a + 1, b, c);
    if (BITTEST(outside, bit) == 0 && BITTEST(above, bit) == 0)
      {
      BITSET(outside, bit);
      static_outside (outside, above, n_points, a + 1, b, c);
      }
    }

  if (b > 0)
    {
    bit = static_offset (n_points, a, b - 1, c);
    if (BITTEST(outside, bit) == 0 && BITTEST(above, bit) == 0)
      {
      BITSET(outside, bit);
      static_outside (outside, above, n_points, a, b - 1, c);
      }
    }

  if (b < n_points - 1)
    {
    bit = static_offset (n_points, a, b + 1, c);
    if (BITTEST(outside, bit) == 0 && BITTEST(above, bit) == 0)
      {
      BITSET(outside, bit);
      static_outside (outside, above, n_points, a, b + 1, c);
      }
    }
  }
}

void static_reset (static_surface *surface)
{
int *cache_v = surface->cache_v;

cache_v[0] = FALSE;
cache_v[1] = FALSE;
cache_v[2] = FALSE;
cache_v[3] = FALSE;
cache_v[4] = FALSE;
cache_v[5] = FALSE;
cache_v[6] = FALSE;
cache_v[7] = FALSE;
surface->cache_s = TRUE;
}

void static_3d_copy (static_surface *surface)
{
double *cache_f = surface->cache_f;
int *cache_v = surface->cache_v;

cache_f[4 * 0 + 0] = cache_f[4 * 4 + 0];
cache_f[4 * 0 + 1] = cache_f[4 * 4 + 1];
cache_f[4 * 0 + 2] = cache_f[4 * 4 + 2];
cache_f[4 * 0 + 3] = cache_f[4 * 4 + 3];

cache_f[4 * 1 + 0] = cache_f[4 * 5 + 0];
cache_f[4 * 1 + 1] = cache_f[4 * 5 + 1];
cache_f[4 * 1 + 2] = cache_f[4 * 5 + 2];
cache_f[4 * 1 + 3] = cache_f[4 * 5 + 3];

cache_f[4 * 2 + 0] = cache_f[4 * 6 + 0];
cache_f[4 * 2 + 1] = cache_f[4 * 6 + 1];
cache_f[4 * 2 + 2] = cache_f[4 * 6 + 2];
cache_f[4 * 2 + 3] = cache_f[4 * 6 + 3];

cache_f[4 * 3 + 0] = cache_f[4 * 7 + 0];
cache_f[4 * 3 + 1] = cache_f[4 * 7 + 1];
cache_f[4 * 3 + 2] = cache_f[4 * 7 + 2];
cache_f[4 * 3 + 3] = cache_f[4 * 7 + 3];

cache_v[0] = cache_v[4];
cache_v[1] = cache_v[5];
cache_v[2] = cache_v[6];
cache_v[3] = cache_v[7];
cache_v[4] = FALSE;
cache_v[5] = FALSE;
cache_v[6] = FALSE;
cache_v[7] = FALSE;
}

void static_2d_copy (static_surface *surface)
{
double *cache_f = surface->cache_f;
int *cache_v = surface->cache_v;

cache_f[4 * 0 + 0] = cache_f[4 * 1 + 0];
cache_f[4 * 0 + 1] = cache_f[4 * 1 + 1];
cache_f[4 * 0 + 2] = cache_f[4 * 1 + 2];
cache_f[4 * 0 + 3] = cache_f[4 * 1 + 3];

cache_f[4 * 3 + 0] = cache_f[4 * 2 + 0];
cache_f[4 * 3 + 1] = cache_f[4 * 2 + 1];
cache_f[4 * 3 + 2] = cache_f[4 * 2 + 2];
cache_f[4 * 3 + 3] = cache_f[4 * 2 + 3];

cache_v[0] = cache_v[1];
cache_v[1] = FALSE;
cache_v[2] = FALSE;
cache_v[3] = cache_v[2];
}

void static_vertex (static_surface *surface, int *v, double *f)
{
gamgi_radial function_r;
gamgi_angular function_a;
double *cache_f, *origin;
double width, r, rho;
int *cache_v;
int cache_s, vertex;
int a, b, c;

/*********************
 * get scanning data *
 *********************/

cache_f = surface->cache_f;
cache_v = surface->cache_v;
cache_s = surface->cache_s;

vertex = v[3];
if (cache_s == TRUE && cache_v[vertex] == TRUE)
  {
  f[0] = cache_f[4 * vertex + 0];
  f[1] = cache_f[4 * vertex + 1];
  f[2] = cache_f[4 * vertex + 2];
  f[3] = cache_f[4 * vertex + 3];
  }
else
  {
  if (cache_s == FALSE) static_reset (surface);
  cache_v[vertex] = TRUE;

  function_r = surface->function_r;
  function_a = surface->function_a;
  origin = surface->origin;
  width = surface->width;
  a = surface->a;
  b = surface->b;
  c = surface->c;

  f[0] = origin[0] + width * (a + v[0]);
  f[1] = origin[1] + width * (b + v[1]);
  f[2] = origin[2] + width * (c + v[2]);

  r = f[0] * f[0] + f[1] * f[1] + f[2] * f[2];
  if (r < TOLERANCE_R) r = TOLERANCE_R;
  r = sqrt (r);

  rho = r * surface->ho;
  f[3] = surface->constant * exp (-rho / 2) * (*function_r) (rho) *
  (*function_a) (f[0], f[1], f[2], r);

  cache_f[4 * vertex + 0] = f[0];
  cache_f[4 * vertex + 1] = f[1];
  cache_f[4 * vertex + 2] = f[2];
  cache_f[4 * vertex + 3] = f[3];
  }
}

void static_edge (static_surface *surface, int *v0, int *v1, double *f)
{
double f0[4], f1[4];
double ff0, ff1;

static_vertex (surface, v0, f0);
static_vertex (surface, v1, f1);

ff0 = f0[3] * f0[3];
ff1 = f1[3] * f1[3];
f[3] = surface->orbital->density;

f[0] = f0[0] + (f1[0] - f0[0]) * (f[3] - ff0) / (ff1 - ff0);
f[1] = f0[1] + (f1[1] - f0[1]) * (f[3] - ff0) / (ff1 - ff0);
f[2] = f0[2] + (f1[2] - f0[2]) * (f[3] - ff0) / (ff1 - ff0);
}

void static_triangle (static_surface *surface,
double *r1, double *r2, double *r3)
{
gamgi_orbital *orbital;
double *points;
int *dots;
int n_points, offset;

orbital = surface->orbital;
dots = orbital->dots;
points = orbital->points;
n_points = orbital->n_points;

offset = dots[0] + dots[1] + dots[2];
if (n_points < offset + 3)
  {
  n_points += GAMGI_CHEM_ORBITAL_SEGMENT;
  points = (double *) realloc (points, n_points * 3 * sizeof (double));
  }

points[3 * offset + 0] = r1[0];
points[3 * offset + 1] = r1[1];
points[3 * offset + 2] = r1[2];

points[3 * offset + 3] = r2[0];
points[3 * offset + 4] = r2[1];
points[3 * offset + 5] = r2[2];

points[3 * offset + 6] = r3[0];
points[3 * offset + 7] = r3[1];
points[3 * offset + 8] = r3[2];

orbital->points = points;
orbital->n_points = n_points;

dots[1] += 3;
surface->triangles += 1;
}

/********************************
 * 3D tetrahedron intersections *
 ********************************/

void static_3d_0000 (static_surface *surface,
int *v0, int *v1, int *v2, int *v3)
{
}

void static_3d_0001 (static_surface *surface,
int *v0, int *v1, int *v2, int *v3)
{
double fe0[4], fe1[4], fe2[4];

static_edge (surface, v0, v1, fe0);
static_edge (surface, v0, v3, fe1);
static_edge (surface, v0, v2, fe2);
static_triangle (surface, fe0, fe1, fe2);
}

void static_3d_0010 (static_surface *surface,
int *v0, int *v1, int *v2, int *v3)
{
double fe0[4], fe1[4], fe2[4];

static_edge (surface, v1, v0, fe0);
static_edge (surface, v1, v2, fe1);
static_edge (surface, v1, v3, fe2);
static_triangle (surface, fe0, fe1, fe2);
}

void static_3d_0011 (static_surface *surface,
int *v0, int *v1, int *v2, int *v3)
{
double fe0[4], fe1[4], fe2[4];

static_edge (surface, v0, v2, fe0);
static_edge (surface, v0, v3, fe1);
static_edge (surface, v1, v3, fe2);
static_triangle (surface, fe0, fe1, fe2);
static_edge (surface, v1, v2, fe1);
static_triangle (surface, fe2, fe1, fe0);
}

void static_3d_0100 (static_surface *surface,
int *v0, int *v1, int *v2, int *v3)
{
double fe0[4], fe1[4], fe2[4];

static_edge (surface, v2, v1, fe0);
static_edge (surface, v2, v0, fe1);
static_edge (surface, v2, v3, fe2);
static_triangle (surface, fe0, fe1, fe2);
}

void static_3d_0101 (static_surface *surface,
int *v0, int *v1, int *v2, int *v3)
{
double fe0[4], fe1[4], fe2[4];

static_edge (surface, v2, v3, fe0);
static_edge (surface, v0, v3, fe1);
static_edge (surface, v0, v1, fe2);
static_triangle (surface, fe0, fe1, fe2);
static_edge (surface, v1, v2, fe1);
static_triangle (surface, fe2, fe1, fe0);
}

void static_3d_0110 (static_surface *surface,
int *v0, int *v1, int *v2, int *v3)
{
double fe0[4], fe1[4], fe2[4];

static_edge (surface, v3, v1, fe0);
static_edge (surface, v0, v1, fe1);
static_edge (surface, v0, v2, fe2);
static_triangle (surface, fe0, fe1, fe2);
static_edge (surface, v3, v2, fe1);
static_triangle (surface, fe2, fe1, fe0);
}

void static_3d_0111 (static_surface *surface,
int *v0, int *v1, int *v2, int *v3)
{
double fe0[4], fe1[4], fe2[4];

static_edge (surface, v3, v2, fe0);
static_edge (surface, v3, v1, fe1);
static_edge (surface, v3, v0, fe2);
static_triangle (surface, fe0, fe1, fe2);
}

void static_3d_1000 (static_surface *surface,
int *v0, int *v1, int *v2, int *v3)
{
double fe0[4], fe1[4], fe2[4];

static_edge (surface, v3, v0, fe0);
static_edge (surface, v3, v1, fe1);
static_edge (surface, v3, v2, fe2);
static_triangle (surface, fe0, fe1, fe2);
}

void static_3d_1001 (static_surface *surface,
int *v0, int *v1, int *v2, int *v3)
{
double fe0[4], fe1[4], fe2[4];

static_edge (surface, v0, v2, fe0);
static_edge (surface, v0, v1, fe1);
static_edge (surface, v3, v1, fe2);
static_triangle (surface, fe0, fe1, fe2);
static_edge (surface, v3, v2, fe1);
static_triangle (surface, fe2, fe1, fe0);
}

void static_3d_1010 (static_surface *surface,
int *v0, int *v1, int *v2, int *v3)
{
double fe0[4], fe1[4], fe2[4];

static_edge (surface, v0, v1, fe0);
static_edge (surface, v0, v3, fe1);
static_edge (surface, v2, v3, fe2);
static_triangle (surface, fe0, fe1, fe2);
static_edge (surface, v1, v2, fe1);
static_triangle (surface, fe2, fe1, fe0);
}

void static_3d_1011 (static_surface *surface,
int *v0, int *v1, int *v2, int *v3)
{
double fe0[4], fe1[4], fe2[4];

static_edge (surface, v2, v3, fe0);
static_edge (surface, v2, v0, fe1);
static_edge (surface, v2, v1, fe2);
static_triangle (surface, fe0, fe1, fe2);
}

void static_3d_1100 (static_surface *surface,
int *v0, int *v1, int *v2, int *v3)
{
double fe0[4], fe1[4], fe2[4];

static_edge (surface, v1, v3, fe0);
static_edge (surface, v0, v3, fe1);
static_edge (surface, v0, v2, fe2);
static_triangle (surface, fe0, fe1, fe2);
static_edge (surface, v1, v2, fe1);
static_triangle (surface, fe2, fe1, fe0);
}

void static_3d_1101 (static_surface *surface,
int *v0, int *v1, int *v2, int *v3)
{
double fe0[4], fe1[4], fe2[4];

static_edge (surface, v1, v3, fe0);
static_edge (surface, v1, v2, fe1);
static_edge (surface, v1, v0, fe2);
static_triangle (surface, fe0, fe1, fe2);
}

void static_3d_1110 (static_surface *surface,
int *v0, int *v1, int *v2, int *v3)
{
double fe0[4], fe1[4], fe2[4];

static_edge (surface, v0, v2, fe0);
static_edge (surface, v0, v3, fe1);
static_edge (surface, v0, v1, fe2);
static_triangle (surface, fe0, fe1, fe2);
}

void static_3d_1111 (static_surface *surface,
int *v0, int *v1, int *v2, int *v3)
{
}

/***************************
 * 2D square intersections *
 ***************************/

void static_2d_0000 (static_surface *surface,
int *v0, int *v1, int *v2, int *v3)
{
/******
 * 00 *
 * 00 *
 ******/
}

void static_2d_0001 (static_surface *surface,
int *v0, int *v1, int *v2, int *v3)
{
double fe0[4], fe1[4];
double fv0[4];

/******
 * 00 *
 * 10 *
 ******/

static_edge (surface, v0, v1, fe0);
static_edge (surface, v0, v3, fe1);
static_vertex (surface, v0, fv0);
static_triangle (surface, fv0, fe0, fe1);
}

void static_2d_0010 (static_surface *surface,
int *v0, int *v1, int *v2, int *v3)
{
double fe0[4], fe1[4];
double fv0[4];

/******
 * 00 *
 * 01 *
 ******/

static_edge (surface, v1, v2, fe0);
static_edge (surface, v1, v0, fe1);
static_vertex (surface, v1, fv0);
static_triangle (surface, fv0, fe0, fe1);
}

void static_2d_0011 (static_surface *surface,
int *v0, int *v1, int *v2, int *v3)
{
double fe0[4], fe1[4];
double fv0[4], fv1[4];

/******
 * 00 *
 * 11 *
 ******/

static_edge (surface, v0, v3, fe0);
static_edge (surface, v1, v2, fe1);
static_vertex (surface, v0, fv0);
static_vertex (surface, v1, fv1);
static_triangle (surface, fe0, fv0, fe1);
static_triangle (surface, fe1, fv0, fv1);
}

void static_2d_0100 (static_surface *surface,
int *v0, int *v1, int *v2, int *v3)
{
double fe0[4], fe1[4];
double fv0[4];

/******
 * 01 *
 * 00 *
 ******/

static_edge (surface, v2, v3, fe0);
static_edge (surface, v2, v1, fe1);
static_vertex (surface, v2, fv0);
static_triangle (surface, fv0, fe0, fe1);
}

void static_2d_0101 (static_surface *surface,
int *v0, int *v1, int *v2, int *v3)
{
double fe0[4], fe1[4], fe2[4], fe3[4];
double fv0[4], fv1[4];

/******
 * 01 *
 * 10 *
 ******/

static_edge (surface, v0, v1, fe0);
static_edge (surface, v1, v2, fe1);
static_edge (surface, v2, v3, fe2);
static_edge (surface, v3, v0, fe3);
static_vertex (surface, v0, fv0);
static_vertex (surface, v2, fv1);
static_triangle (surface, fv0, fe0, fe3);
static_triangle (surface, fv1, fe2, fe1);
}

void static_2d_0110 (static_surface *surface,
int *v0, int *v1, int *v2, int *v3)
{
double fe0[4], fe1[4];
double fv0[4], fv1[4];

/******
 * 01 *
 * 01 *
 ******/

static_edge (surface, v0, v1, fe0);
static_edge (surface, v2, v3, fe1);
static_vertex (surface, v1, fv0);
static_vertex (surface, v2, fv1);
static_triangle (surface, fe0, fv0, fe1);
static_triangle (surface, fe1, fv0, fv1);
}

void static_2d_0111 (static_surface *surface,
int *v0, int *v1, int *v2, int *v3)
{
double fe0[4], fe1[4];
double fv0[4], fv1[4], fv2[4];

/******
 * 01 *
 * 11 *
 ******/

static_edge (surface, v3, v0, fe0);
static_edge (surface, v3, v2, fe1);
static_vertex (surface, v0, fv0);
static_vertex (surface, v1, fv1);
static_vertex (surface, v2, fv2);
static_triangle (surface, fe0, fv0, fv1);
static_triangle (surface, fe1, fv1, fv2);
static_triangle (surface, fe1, fe0, fv1);
}

void static_2d_1000 (static_surface *surface,
int *v0, int *v1, int *v2, int *v3)
{
double fe0[4], fe1[4];
double fv0[4];

/******
 * 10 *
 * 00 *
 ******/

static_edge (surface, v3, v0, fe0);
static_edge (surface, v3, v2, fe1);
static_vertex (surface, v3, fv0);
static_triangle (surface, fv0, fe0, fe1);
}

void static_2d_1001 (static_surface *surface,
int *v0, int *v1, int *v2, int *v3) 
{
double fe0[4], fe1[4];
double fv0[4], fv1[4];

/******
 * 10 *
 * 10 *
 ******/

static_edge (surface, v0, v1, fe0);
static_edge (surface, v2, v3, fe1);
static_vertex (surface, v0, fv0);
static_vertex (surface, v3, fv1);
static_triangle (surface, fe0, fe1, fv1);
static_triangle (surface, fv1, fv0, fe0);
}

void static_2d_1010 (static_surface *surface,
int *v0, int *v1, int *v2, int *v3)
{
double fe0[4], fe1[4], fe2[4], fe3[4];
double fv0[4], fv1[4];

/******
 * 10 *
 * 01 *
 ******/

static_edge (surface, v0, v1, fe0);
static_edge (surface, v1, v2, fe1);
static_edge (surface, v2, v3, fe2);
static_edge (surface, v3, v0, fe3);
static_vertex (surface, v1, fv0);
static_vertex (surface, v3, fv1);
static_triangle (surface, fv0, fe1, fe0);
static_triangle (surface, fv1, fe3, fe2);
}

void static_2d_1011 (static_surface *surface,
int *v0, int *v1, int *v2, int *v3)
{
double fe0[4], fe1[4];
double fv0[4], fv1[4], fv2[4];

/******
 * 10 *
 * 11 *
 ******/

static_edge (surface, v2, v3, fe0);
static_edge (surface, v2, v1, fe1);

static_vertex (surface, v0, fv0);
static_vertex (surface, v1, fv1);
static_vertex (surface, v3, fv2);

static_triangle (surface, fe0, fv2, fv0);
static_triangle (surface, fv0, fv1, fe1);
static_triangle (surface, fe1, fe0, fv0);
}

void static_2d_1100 (static_surface *surface,
int *v0, int *v1, int *v2, int *v3)
{
double fe0[4], fe1[4];
double fv0[4], fv1[4];

/******
 * 11 *
 * 00 *
 ******/

static_edge (surface, v1, v2, fe0);
static_edge (surface, v3, v0, fe1);
static_vertex (surface, v2, fv0);
static_vertex (surface, v3, fv1);
static_triangle (surface, fe0, fv0, fv1);
static_triangle (surface, fv1, fe1, fe0);
}

void static_2d_1101 (static_surface *surface,
int *v0, int *v1, int *v2, int *v3)
{
double fe0[4], fe1[4];
double fv0[4], fv1[4], fv2[4];

/******
 * 11 *
 * 10 *
 ******/

static_edge (surface, v1, v2, fe0);
static_edge (surface, v0, v1, fe1);
static_vertex (surface, v0, fv0);
static_vertex (surface, v2, fv1);
static_vertex (surface, v3, fv2);
static_triangle (surface, fe0, fv1, fv2);
static_triangle (surface, fv2, fv0, fe1);
static_triangle (surface, fe1, fe0, fv2);
}

void static_2d_1110 (static_surface *surface,
int *v0, int *v1, int *v2, int *v3)
{
double fe0[4], fe1[4];
double fv0[4], fv1[4], fv2[4];

/******
 * 11 *
 * 01 *
 ******/

static_edge (surface, v0, v1, fe0);
static_edge (surface, v0, v3, fe1);
static_vertex (surface, v1, fv0);
static_vertex (surface, v2, fv1);
static_vertex (surface, v3, fv2);
static_triangle (surface, fe0, fv0, fv1);
static_triangle (surface, fv1, fv2, fe1);
static_triangle (surface, fe1, fe0, fv1);
}

void static_2d_1111 (static_surface *surface,
int *v0, int *v1, int *v2, int *v3)
{
double fv0[4], fv1[4], fv2[4], fv3[4];

static_vertex (surface, v0, fv0);
static_vertex (surface, v1, fv1);
static_vertex (surface, v2, fv2);
static_vertex (surface, v3, fv3);
static_triangle (surface, fv0, fv1, fv2);
static_triangle (surface, fv2, fv3, fv0);
}

void static_tetrahedron (static_surface *surface,
int *v0, int *v1, int *v2, int *v3)
{
char *outside;
int n_points, bit;
int a, b, c;
int state;

/***************************************
 * generate triangles from tetrahedron *
 ***************************************/

n_points = surface->n_points;
a = surface->a;
b = surface->b;
c = surface->c;
outside = surface->outside;

state = 0;
bit = static_offset (n_points, a + v0[0], b + v0[1], c + v0[2]);
if (BITTEST(outside, bit) == 0) state |= 1;
bit = static_offset (n_points, a + v1[0], b + v1[1], c + v1[2]);
if (BITTEST(outside, bit) == 0) state |= 2;
bit = static_offset (n_points, a + v2[0], b + v2[1], c + v2[2]);
if (BITTEST(outside, bit) == 0) state |= 4;
bit = static_offset (n_points, a + v3[0], b + v3[1], c + v3[2]);
if (BITTEST(outside, bit) == 0) state |= 8;

(*surface->f3d[state]) (surface, v0, v1, v2, v3);
}

void static_square (static_surface *surface,
int *v0, int *v1, int *v2, int *v3)
{
char *outside;
int n_points, bit;
int a, b, c;
int state;

/**********************************
 * generate triangles from square *
 **********************************/

/**********************************
 * square vertices convention 3 2 *
 * when seen from outside:    0 1 *
 **********************************/

n_points = surface->n_points;
a = surface->a;
b = surface->b;
c = surface->c;
outside = surface->outside;

state = 0;
bit = static_offset (n_points, a + v0[0], b + v0[1], c + v0[2]);
if (BITTEST(outside, bit) == 0) state |= 1;
bit = static_offset (n_points, a + v1[0], b + v1[1], c + v1[2]);
if (BITTEST(outside, bit) == 0) state |= 2;
bit = static_offset (n_points, a + v2[0], b + v2[1], c + v2[2]);
if (BITTEST(outside, bit) == 0) state |= 4;
bit = static_offset (n_points, a + v3[0], b + v3[1], c + v3[2]);
if (BITTEST(outside, bit) == 0) state |= 8;
(*surface->f2d[state]) (surface, v0, v1, v2, v3);
}

void static_cubes (static_surface *surface)
{
int n_points, a, b, c;
int v0[4], v1[4], v2[4], v3[4], v4[4], v5[4], v6[4], v7[4];

/*****************************************
 * generate cubes, tetrahedra, triangles *
 *****************************************/

/****************************
 * cube vertices convention *
 *                          *
 *     Back    Front        *
 *      3 2     7 6         *
 *      0 1     4 5         *
 ****************************/

v0[0] = 0; v0[1] = 0; v0[2] = 0; v0[3] = 0;
v1[0] = 0; v1[1] = 1; v1[2] = 0; v1[3] = 1;
v2[0] = 0; v2[1] = 1; v2[2] = 1; v2[3] = 2;
v3[0] = 0; v3[1] = 0; v3[2] = 1; v3[3] = 3;
v4[0] = 1; v4[1] = 0; v4[2] = 0; v4[3] = 4;
v5[0] = 1; v5[1] = 1; v5[2] = 0; v5[3] = 5;
v6[0] = 1; v6[1] = 1; v6[2] = 1; v6[3] = 6;
v7[0] = 1; v7[1] = 0; v7[2] = 1; v7[3] = 7;

/**************
 * scan cubes *
 **************/

n_points = surface->n_points;
for (c = 0; c < n_points - 1; c++)
  {
  surface->c = c;
  for (b = 0; b < n_points - 1; b++)
    {
    surface->b = b;
    for (a = 0; a < n_points - 1; a++)
      {
      surface->a = a;

      if (a == 0 || surface->triangles == 0) surface->cache_s = FALSE;
      else static_3d_copy (surface);
      surface->triangles = 0;

      /*****************************************
       * scan 5 (central plus four) tetrahedra *
       *****************************************/

      if (a + b + c % 2 == 0)
        {
        static_tetrahedron (surface, v5, v0, v7, v2);
        static_tetrahedron (surface, v5, v0, v4, v7);
        static_tetrahedron (surface, v5, v0, v1, v2);
        static_tetrahedron (surface, v7, v2, v5, v6);
        static_tetrahedron (surface, v7, v2, v0, v3);
        }
      else
        {
        static_tetrahedron (surface, v4, v1, v6, v3);
        static_tetrahedron (surface, v4, v1, v5, v6);
        static_tetrahedron (surface, v4, v1, v0, v3);
        static_tetrahedron (surface, v6, v3, v4, v7);
        static_tetrahedron (surface, v6, v3, v1, v2);
        }

      }
    }
  }
}

void static_squares_a0 (static_surface *surface)
{
int v0[4], v1[4], v2[4], v3[4];
int n_points;
int b, c;

n_points = surface->n_points;
v0[0] = 0; v0[1] =  0; v0[2] = 0; v0[3] = 0;
v1[0] = 0; v1[1] = -1; v1[2] = 0; v1[3] = 1;
v2[0] = 0; v2[1] = -1; v2[2] = 1; v2[3] = 2;
v3[0] = 0; v3[1] =  0; v3[2] = 1; v3[3] = 3;

surface->a = 0;
for (c = 0; c < n_points - 1; c++)
  {
  surface->c = c;
  for (b = n_points - 1; b > 0; b--)
    {
    surface->b = b;

    if (b == n_points - 1 || surface->triangles == 0) surface->cache_s = FALSE;
    else static_2d_copy (surface);
    surface->triangles = 0;

    static_square (surface, v0, v1, v2, v3);
    }
  }
}

void static_squares_a1 (static_surface *surface)
{
int v0[4], v1[4], v2[4], v3[4];
int n_points;
int b, c;

n_points = surface->n_points;
v0[0] = 0; v0[1] = 0; v0[2] = 0; v0[3] = 0;
v1[0] = 0; v1[1] = 1; v1[2] = 0; v1[3] = 1;
v2[0] = 0; v2[1] = 1; v2[2] = 1; v2[3] = 2;
v3[0] = 0; v3[1] = 0; v3[2] = 1; v3[3] = 3;

surface->a = n_points - 1;
for (c = 0; c < n_points - 1; c++)
  {
  surface->c = c;
  for (b = 0; b < n_points - 1; b++)
    {
    surface->b = b;

    if (b == 0 || surface->triangles == 0) surface->cache_s = FALSE;
    else static_2d_copy (surface);
    surface->triangles = 0;

    static_square (surface, v0, v1, v2, v3);
    }
  }
}

void static_squares_b0 (static_surface *surface)
{
int v0[4], v1[4], v2[4], v3[4];
int n_points;
int a, c;

n_points = surface->n_points;
v0[0] = 0; v0[1] = 0; v0[2] = 0; v0[3] = 0;
v1[0] = 1; v1[1] = 0; v1[2] = 0; v1[3] = 1;
v2[0] = 1; v2[1] = 0; v2[2] = 1; v2[3] = 2;
v3[0] = 0; v3[1] = 0; v3[2] = 1; v3[3] = 3;

surface->b = 0;
for (c = 0; c < n_points - 1; c++)
  {
  surface->c = c;
  for (a = 0; a < n_points - 1; a++)
    {
    surface->a = a;

    if (a == 0 || surface->triangles == 0) surface->cache_s = FALSE;
    else static_2d_copy (surface);
    surface->triangles = 0;

    static_square (surface, v0, v1, v2, v3);
    }
  }
}

void static_squares_b1 (static_surface *surface)
{
int v0[4], v1[4], v2[4], v3[4];
int n_points;
int a, c;

n_points = surface->n_points;
v0[0] =  0; v0[1] = 0; v0[2] = 0; v0[3] = 0;
v1[0] = -1; v1[1] = 0; v1[2] = 0; v1[3] = 1;
v2[0] = -1; v2[1] = 0; v2[2] = 1; v2[3] = 2;
v3[0] =  0; v3[1] = 0; v3[2] = 1; v3[3] = 3;

surface->b = n_points - 1;
for (c = 0; c < n_points - 1; c++)
  {
  surface->c = c;
  for (a = n_points - 1; a > 0; a--)
    {
    surface->a = a;

    if (a == n_points - 1 || surface->triangles == 0) surface->cache_s = FALSE;
    else static_2d_copy (surface);
    surface->triangles = 0;

    static_square (surface, v0, v1, v2, v3);
    }
  }
}

void static_squares_c0 (static_surface *surface)
{
int v0[4], v1[4], v2[4], v3[4];
int n_points;
int a, b;

n_points = surface->n_points;
v0[0] =  0; v0[1] = 0; v0[2] = 0; v0[3] = 0;
v1[0] = -1; v1[1] = 0; v1[2] = 0; v1[3] = 1;
v2[0] = -1; v2[1] = 1; v2[2] = 0; v2[3] = 2;
v3[0] =  0; v3[1] = 1; v3[2] = 0; v3[3] = 3;

surface->c = 0;
for (b = 0; b < n_points - 1; b++)
  {
  surface->b = b;
  for (a = n_points - 1; a > 0; a--)
    {
    surface->a = a;

    if (a == n_points - 1 || surface->triangles == 0) surface->cache_s = FALSE;
    else static_2d_copy (surface);
    surface->triangles = 0;

    static_square (surface, v0, v1, v2, v3);
    }
  }
}

void static_squares_c1 (static_surface *surface)
{
int v0[4], v1[4], v2[4], v3[4];
int n_points;
int a, b;

n_points = surface->n_points;
v0[0] = 0; v0[1] = 0; v0[2] = 0; v0[3] = 0;
v1[0] = 1; v1[1] = 0; v1[2] = 0; v1[3] = 1;
v2[0] = 1; v2[1] = 1; v2[2] = 0; v2[3] = 2;
v3[0] = 0; v3[1] = 1; v3[2] = 0; v3[3] = 3;

surface->c = n_points - 1;
for (b = 0; b < n_points - 1; b++)
  {
  surface->b = b;
  for (a = 0; a < n_points - 1; a++)
    {
    surface->a = a;

    if (a == 0 || surface->triangles == 0) surface->cache_s = FALSE;
    else static_2d_copy (surface);
    surface->triangles = 0;

    static_square (surface, v0, v1, v2, v3);
    }
  }
}

static static_surface *static_start (gamgi_orbital *orbital, int *octants)
{
static_surface *surface;
double constant_r, constant_a, constant_z, za;
int n_points, n_bit, n_char;
int a, b, c;

/***********************************************
 * allocate main structure and user input data *
 ***********************************************/

surface = (static_surface *) malloc (sizeof (static_surface));
surface->orbital = orbital;
surface->octants = octants;

/*************************
 * get orbital constants *
 **************************/

static_radial (orbital->n, orbital->l, &constant_r, &surface->function_r);
static_angular (orbital->l, orbital->m, &constant_a, &surface->function_a);
za = orbital->charge / GAMGI_CHEM_A0;
surface->ho = 2 * za / orbital->n;
constant_z = pow (za, 1.5);
surface->constant = constant_z * constant_r * constant_a;

/**********************************
 * allocate sampling slices width *
 **********************************/

surface->slices = orbital->sampling;
surface->width = orbital->radius / surface->slices;

/***********************
 * allocate bit arrays *
 ***********************/

surface->n_points = n_points = 1 + surface->slices;
surface->n_bit = n_bit = n_points * n_points * n_points;
surface->n_char = n_char = BITNSLOTS(n_bit);
surface->above = (char *) malloc (n_char * sizeof (char));
surface->outside = (char *) malloc (n_char * sizeof (char));

/************************************************************
 * Initialize, accumulate, how many times each octant face  *
 * will be rendered. When face = 0 the face does not exist, *
 * when face = 2 the face is internal and can be ignored,   *
 * when face = 1 the face is external and must be rendered. *
 ************************************************************/

for (c = 0; c <= 1; c++)
  {
  for (b = 0; b <= 1; b++)
    {
    for (a = 0; a <= 1; a++)
      {
      surface->face_a[a + 0][b][c] = 0;
      surface->face_a[a + 1][b][c] = 0;

      surface->face_b[a][b + 0][c] = 0;
      surface->face_b[a][b + 1][c] = 0;

      surface->face_c[a][b][c + 0] = 0;
      surface->face_c[a][b][c + 1] = 0;
      }
    }
  }

for (c = 0; c <= 1; c++)
  {
  for (b = 0; b <= 1; b++)
    {
    for (a = 0; a <= 1; a++)
      {
      if (octants[4 * c + 2 * b + a] == TRUE)
        {
        surface->face_a[a + 0][b][c]++;
        surface->face_a[a + 1][b][c]++;

        surface->face_b[a][b + 0][c]++;
        surface->face_b[a][b + 1][c]++;

        surface->face_c[a][b][c + 0]++;
        surface->face_c[a][b][c + 1]++;
        }
      }
    }
  }


surface->f2d[ 0] = static_2d_0000;
surface->f2d[ 1] = static_2d_0001;
surface->f2d[ 2] = static_2d_0010;
surface->f2d[ 3] = static_2d_0011;
surface->f2d[ 4] = static_2d_0100;
surface->f2d[ 5] = static_2d_0101;
surface->f2d[ 6] = static_2d_0110;
surface->f2d[ 7] = static_2d_0111;
surface->f2d[ 8] = static_2d_1000;
surface->f2d[ 9] = static_2d_1001;
surface->f2d[10] = static_2d_1010;
surface->f2d[11] = static_2d_1011;
surface->f2d[12] = static_2d_1100;
surface->f2d[13] = static_2d_1101;
surface->f2d[14] = static_2d_1110;
surface->f2d[15] = static_2d_1111;

surface->f3d[ 0] = static_3d_0000;
surface->f3d[ 1] = static_3d_0001;
surface->f3d[ 2] = static_3d_0010;
surface->f3d[ 3] = static_3d_0011;
surface->f3d[ 4] = static_3d_0100;
surface->f3d[ 5] = static_3d_0101;
surface->f3d[ 6] = static_3d_0110;
surface->f3d[ 7] = static_3d_0111;
surface->f3d[ 8] = static_3d_1000;
surface->f3d[ 9] = static_3d_1001;
surface->f3d[10] = static_3d_1010;
surface->f3d[11] = static_3d_1011;
surface->f3d[12] = static_3d_1100;
surface->f3d[13] = static_3d_1101;
surface->f3d[14] = static_3d_1110;
surface->f3d[15] = static_3d_1111;

return surface;
}

void static static_end (static_surface *surface)
{
free (surface->above);
free (surface->outside);
free (surface);
}

int static_solid (gamgi_orbital *orbital, int *octants)
{
static_surface *surface;
char *outside, *above;
int *dots;
int slices, n_points;
int a, b, c, bit;

/*****************
 * allocate data *
 *****************/

surface = static_start (orbital, octants);

/*************************
 * allocate points array *
 *************************/

orbital->n_points = GAMGI_CHEM_ORBITAL_SEGMENT;
orbital->points  = (double *) malloc (orbital->n_points * 3 * sizeof (double));

/******************
 * allocate frame *
 ******************/

dots = orbital->dots;
if (dots[0] > 0) static_frame (orbital);
dots[1] = 0;
dots[2] = 0;

/*****************
 * build octants *
 *****************/

outside = surface->outside;
above = surface->above;
slices = surface->slices;
n_points = surface->n_points;

for (c = 0; c <= 1; c++)
  {
  for (b = 0; b <= 1; b++)
    {
    for (a = 0; a <= 1; a++)
      {
      if (octants[4 * c + 2 * b + a] == TRUE)
        {
        static_clean (above, surface->n_char);
        static_above (surface, a, b, c);

        static_clean (outside, surface->n_char);
        bit = static_offset (n_points, a * slices, b * slices, c * slices);
        if (BITTEST(outside, bit) == 0 && BITTEST(above, bit) == 0)
          {
          BITSET(outside, bit);
          static_outside (outside, above, n_points, a * slices, b * slices, c * slices);
          }

        /******************************************
         * scan cubes to build orbital 3d surface *
         ******************************************/

/*
        static_cubes (surface);
*/

        /*************************************************************
         * scan squares to build octant faces a-, a+, b-, b+, c-, c+ *
         *************************************************************/

        if (surface->face_a[a + 0][b][c] == TRUE) static_squares_a0 (surface);
        if (surface->face_a[a + 1][b][c] == TRUE) static_squares_a1 (surface);
        if (surface->face_b[a][b + 0][c] == TRUE) static_squares_b0 (surface);
        if (surface->face_b[a][b + 1][c] == TRUE) static_squares_b1 (surface);
        if (surface->face_c[a][b][c + 0] == TRUE) static_squares_c0 (surface);
        if (surface->face_c[a][b][c + 1] == TRUE) static_squares_c1 (surface);
        }
      }
    }
  }

/****************
 * resize array *
 ****************/

orbital->n_points = dots[0] + dots[1] + dots[2];
orbital->points = (double *) realloc (orbital->points, orbital->n_points * 3 * sizeof (double));

/***************
 * remove data *
 ***************/

static_end (surface);

return TRUE;
}

void gamgi_chem_orbital_m (int l, gamgi_bool *sensitive)
{
switch (l)
  {
  case 0:
  sensitive[0] = FALSE;
  sensitive[1] = FALSE;
  sensitive[2] = FALSE;
  sensitive[3] = FALSE;
  sensitive[4] = FALSE;
  sensitive[6] = FALSE;
  sensitive[7] = FALSE;
  sensitive[8] = FALSE;
  sensitive[9] = FALSE;
  sensitive[10] = FALSE;
  break;

  case 1:
  sensitive[0] = FALSE;
  sensitive[1] = FALSE;
  sensitive[2] = FALSE;
  sensitive[3] = FALSE;
  sensitive[4] = TRUE;
  sensitive[6] = TRUE;
  sensitive[7] = FALSE;
  sensitive[8] = FALSE;
  sensitive[9] = FALSE;
  sensitive[10] = FALSE;
  break;

  case 2:
  sensitive[0] = FALSE;
  sensitive[1] = FALSE;
  sensitive[2] = FALSE;
  sensitive[3] = TRUE;
  sensitive[4] = TRUE;
  sensitive[6] = TRUE;
  sensitive[7] = TRUE;
  sensitive[8] = FALSE;
  sensitive[9] = FALSE;
  sensitive[10] = FALSE;
  break;

  case 3:
  sensitive[0] = FALSE;
  sensitive[1] = FALSE;
  sensitive[2] = TRUE;
  sensitive[3] = TRUE;
  sensitive[4] = TRUE;
  sensitive[6] = TRUE;
  sensitive[7] = TRUE;
  sensitive[8] = TRUE;
  sensitive[9] = FALSE;
  sensitive[10] = FALSE;
  break;

  case 4:
  sensitive[0] = FALSE;
  sensitive[1] = TRUE;
  sensitive[2] = TRUE;
  sensitive[3] = TRUE;
  sensitive[4] = TRUE;
  sensitive[6] = TRUE;
  sensitive[7] = TRUE;
  sensitive[8] = TRUE;
  sensitive[9] = TRUE;
  sensitive[10] = FALSE;
  break;

  case 5:
  sensitive[0] = TRUE;
  sensitive[1] = TRUE;
  sensitive[2] = TRUE;
  sensitive[3] = TRUE;
  sensitive[4] = TRUE;
  sensitive[6] = TRUE;
  sensitive[7] = TRUE;
  sensitive[8] = TRUE;
  sensitive[9] = TRUE;
  sensitive[10] = TRUE;
  break;
  }
}

void gamgi_chem_orbital_l (int n, gamgi_bool *sensitive)
{
switch (n)
  {
  case 1:
  sensitive[1] = FALSE;
  sensitive[2] = FALSE;
  sensitive[3] = FALSE;
  sensitive[4] = FALSE;
  sensitive[5] = FALSE;
  break;

  case 2:
  sensitive[1] = TRUE;
  sensitive[2] = FALSE;
  sensitive[3] = FALSE;
  sensitive[4] = FALSE;
  sensitive[5] = FALSE;
  break;

  case 3:
  sensitive[1] = TRUE;
  sensitive[2] = TRUE;
  sensitive[3] = FALSE;
  sensitive[4] = FALSE;
  sensitive[5] = FALSE;
  break;

  case 4:
  sensitive[1] = TRUE;
  sensitive[2] = TRUE;
  sensitive[3] = TRUE;
  sensitive[4] = FALSE;
  sensitive[5] = FALSE;
  break;

  case 5:
  sensitive[1] = TRUE;
  sensitive[2] = TRUE;
  sensitive[3] = TRUE;
  sensitive[4] = TRUE;
  sensitive[5] = FALSE;
  break;

  case 6:
  sensitive[1] = TRUE;
  sensitive[2] = TRUE;
  sensitive[3] = TRUE;
  sensitive[4] = TRUE;
  sensitive[5] = TRUE;
  break;
  }
}

int gamgi_chem_orbital_sampling (int n, int style)
{
if (style == GAMGI_MESA_WIRED)
  return GAMGI_CHEM_ORBITAL_POINTS * n;

if (style == GAMGI_MESA_SOLID)
  return GAMGI_CHEM_ORBITAL_SLICES;

/*****************************************
 * control should never reach this point *
 *****************************************/

return FALSE;
}

double gamgi_chem_orbital_radius (int n, int l, int m, double charge)
{
gamgi_radial function_r;
gamgi_angular function_a;
double za, r, ho, rho, r_max;
double constant, constant_a, constant_z, constant_r;
double d, d_old, d_old_old;
double f;

if (n < 1 || n > 6) return 0.0;
if (l < 0 || l > n - 1) return 0.0;
if (m < -l || m > l) return 0.0;

za = charge / GAMGI_CHEM_A0;
ho = 2 * za / n;
constant_z = pow (za, 1.5);

static_radial (n, l, &constant_r, &function_r);
static_angular (l, m, &constant_a, &function_a);

/***********************************************
 * get the last maximum for the radial density *
 ***********************************************/

r_max = GAMGI_CHEM_ORBITAL_MAX;

d_old = d_old_old = 0.0;
constant = constant_r * constant_z;
for (r = 0.0; r < GAMGI_CHEM_ORBITAL_MAX; r += GAMGI_CHEM_ORBITAL_STEP)
  {
  rho = r * ho;
  f = constant * exp (-rho / 2) * (*function_r) (rho);
  d = f * f * r * r;
  if (d < d_old && d_old > d_old_old)
    r_max = r - GAMGI_CHEM_ORBITAL_STEP;
  d_old_old = d_old;
  d_old = d;
  }

/**********************************************
 * start checking the default maximum density *
 * some distance after the last maximum       *
 **********************************************/

r_max += GAMGI_CHEM_ORBITAL_SHIFT;

/*********************************************
 * find the radius where the density becomes *
 * lower than the default maximum density    *
 *********************************************/

constant *= constant_a;
for (r = r_max; r < GAMGI_CHEM_ORBITAL_MAX; r += GAMGI_CHEM_ORBITAL_STEP)
  {
  rho = r * ho;
  f = constant * exp (-rho / 2) * (*function_r) (rho);
  if (static_density (r, f * f, function_a) == FALSE) return r;
  }

return GAMGI_CHEM_ORBITAL_MAX;
}

gamgi_bool gamgi_chem_orbital_create (gamgi_orbital *orbital,
int *octants, int seed)
{
if (orbital->style == GAMGI_MESA_WIRED)
  return static_wired (orbital, octants, seed);

if (orbital->style == GAMGI_MESA_SOLID)
  return static_solid (orbital, octants);

/*****************************************
 * control should never reach this point *
 *****************************************/

return FALSE;
}
