/******************************************
 *
 * $GAMGI/src/chem/gamgi_chem_orbital.c
 *
 * Copyright (C) 2012 Carlos Pereira
 *
 * Distributed under the terms of the GNU
 * General Public License: $GAMGI/LICENSE
 *
 */

#include "gamgi_engine.h"
#include "gamgi_math.h"
#include "gamgi_chem.h"

static double static_1s (double r)
{
/******************
 * n = 1, l = 0   *
 ******************/

return 1;
}

static double static_2s (double r)
{
/****************
 * n = 2, l = 0 *
 ****************/

return 2 - r;
}

static double static_2p (double r)
{
/****************
 * n = 2, l = 1 *
 ****************/

return r;
}

static double static_3s (double r)
{
double r2 = r * r;

/****************
 * n = 3, l = 0 *
 ****************/

return 6 - 6 * r + r2;
}

static double static_3p (double r)
{
/****************
 * n = 3, l = 1 *
 ****************/

return r * (4 - r);
}

static double static_3d (double r)
{
double r2 = r * r;

/****************
 * n = 3, l = 2 *
 ****************/

return r2;
}

static double static_4s (double r)
{
double r2 = r * r;
double r3 = r2 * r;

/****************
 * n = 4, l = 0 *
 ****************/

return (24 - 36 * r + 12 * r2 - r3);
}

static double static_4p (double r)
{
double r2 = r * r;

/****************
 * n = 4, l = 1 *
 ****************/ 

return r * (20 - 10 * r + r2); 
}

static double static_4d (double r)
{
double r2 = r * r;

/****************
 * n = 4, l = 2 *
 ****************/

return (6 - r) * r2;
}

static double static_4f (double r)
{
double r3 = r * r * r;

/****************
 * n = 4, l = 3 *
 ****************/

return r3;
}

static double static_5s (double r)
{
double r2 = r * r;
double r3 = r2 * r;
double r4 = r3 * r;

/****************
 * n = 5, l = 0 *
 ****************/

return (120 - 240 * r + 120 * r2 - 20 * r3 + r4);
}

static double static_5p (double r)
{
double r2 = r * r;
double r3 = r2 * r;

/****************
 * n = 5, l = 1 *
 ****************/

return (120 - 90 * r + 18 * r2 - r3) * r;
}

static double static_5d (double r)
{
double r2 = r * r;

/****************
 * n = 5, l = 2 *
 ****************/

return (42 - 14 * r + r2) * r2;
}

static double static_5f (double r)
{
double r3 = r * r * r;

/****************
 * n = 5, l = 3 *
 ****************/

return (8 - r) * r3;
}

static double static_5g (double r)
{
double r2 = r * r;
double r4 = r2 * r2;

/****************
 * n = 5, l = 4 *
 ****************/

return r4;
}

static double static_6s (double r)
{
double r2 = r * r;
double r3 = r2 * r;
double r4 = r3 * r;
double r5 = r4 * r;

/****************
 * n = 6, l = 0 *
 ****************/

return (720 - 1800 * r + 1200 * r2 - 300 * r3 + 30 * r4 - r5);
}

static double static_6p (double r)
{
double r2 = r * r;
double r3 = r2 * r;
double r4 = r3 * r;

/****************
 * n = 6, l = 1 *
 ****************/

return (840 - 840 * r + 252 * r2 - 28 * r3 + r4) * r;
}

static double static_6d (double r)
{
double r2 = r * r;
double r3 = r2 * r;

/****************
 * n = 6, l = 2 *
 ****************/

return (336 - 168 * r + 24 * r2 - r3) * r2;
}

static double static_6f (double r)
{
double r2 = r * r;
double r3 = r2 * r;

/****************
 * n = 6, l = 3 *
 ****************/

return (72 - 18 * r + r2) * r3;
}

static double static_6g (double r)
{
double r2 = r * r;
double r4 = r2 * r2;

/****************
 * n = 6, l = 4 *
 ****************/

return (10 - r) * r4;
}

static double static_6h (double r)
{
double r2 = r * r;
double r4 = r2 * r2;
double r5 = r4 * r;

/****************
 * n = 6, l = 5 *
 ****************/

return r5;
}

static void static_radial (int n, int l, double *c, gamgi_radial *f)
{
if (n == 1) { *c = 0.5; *f = static_1s; }

if (n == 2)
  {
  if (l == 0) { *c = 2 * sqrt (2); *f = static_2s; }
  if (l == 1) { *c = 2 * sqrt (6); *f = static_2p; }
  }

if (n == 3)
  {
  if (l == 0) { *c = 9 * sqrt (3);  *f = static_3s; }
  if (l == 1) { *c = 9 * sqrt (6);  *f = static_3p; }
  if (l == 2) { *c = 9 * sqrt (30); *f = static_3d; }
  }

if (n == 4)
  {
  if (l == 0) { *c = 96;             *f = static_4s; }
  if (l == 1) { *c = 32 * sqrt (15); *f = static_4p; }
  if (l == 2) { *c = 96 * sqrt (5);  *f = static_4d; }
  if (l == 3) { *c = 96 * sqrt (35); *f = static_4f; }
  }

if (n == 5)
  {
  if (l == 0) { *c = 300 * sqrt (5);  *f = static_5s; }
  if (l == 1) { *c = 150 * sqrt (30); *f = static_5p; }
  if (l == 2) { *c = 150 * sqrt (70); *f = static_5d; }
  if (l == 3) { *c = 300 * sqrt (70); *f = static_5f; }
  if (l == 4) { *c = 900 * sqrt (70); *f = static_5g; }
  }

if (n == 6)
  {
  if (l == 0) { *c = 2160 * sqrt (6);   *f = static_6s; }
  if (l == 1) { *c = 432 * sqrt (210);  *f = static_6p; }
  if (l == 2) { *c = 864 * sqrt (105);  *f = static_6d; }
  if (l == 3) { *c = 2592 * sqrt (35);  *f = static_6f; }
  if (l == 4) { *c = 12960 * sqrt (7);  *f = static_6g; }
  if (l == 5) { *c = 12960 * sqrt (77); *f = static_6h; }
  }

*c = 1 / *c;
}

static double static_s (double x, double y, double z, double r)
{
/****************
 * l = 0, m = 0 *
 ****************/

return 1;
}

static double static_pz (double x, double y, double z, double r)
{
/****************
 * l = 1, m = 0 *
 ****************/

return z / r;
}

static double static_px (double x, double y, double z, double r)
{
/*****************
 * l = 1, m = +1 *
 *****************/

return x / r;
}

static double static_py (double x, double y, double z, double r)
{
/*****************
 * l = 1, m = -1 *
 *****************/

return y / r;
}

static double static_dz2 (double x, double y, double z, double r)
{
double z2 = z * z;
double r2 = r * r;

/****************
 * l = 2, m = 0 *
 ****************/

return (3 * z2 - r2) / r2;
}

static double static_dxz (double x, double y, double z, double r)
{
double r2 = r * r;

/****************
 * l = 2, m = 1 *
 ****************/

return (x * z) / r2;
}

static double static_dyz (double x, double y, double z, double r)
{
double r2 = r * r;

/****************
 * l = 2, m = -1 *
 ****************/

return (y * z) / r2;
}

static double static_dx2_y2 (double x, double y, double z, double r)
{
double x2 = x * x;
double y2 = y * y;
double r2 = r * r;

/****************
 * l = 2, m = 2 *
 ****************/

return (x2 - y2) / r2;
}

static double static_dxy (double x, double y, double z, double r)
{
double r2 = r * r;

/****************
 * l = 2, m = -2 *
 ****************/

return (x * y) / r2;
}

static double static_fz3 (double x, double y, double z, double r)
{
double z2 = z * z;
double r2 = r * r;
double r3 = r2 * r;

/****************
 * l = 3, m = 0 *
 ****************/

return z * (5 * z2 - 3 * r2) / r3;
}

static double static_fxz2 (double x, double y, double z, double r)
{
double z2 = z * z;
double r2 = r * r;
double r3 = r2 * r;

/****************
 * l = 3, m = 1 *
 ****************/

return x * (5 * z2 - r2) / r3;
}

static double static_fyz2 (double x, double y, double z, double r)
{
double z2 = z * z;
double r2 = r * r;
double r3 = r2 * r;

/*****************
 * l = 3, m = -1 *
 *****************/

return y * (5 * z2 - r2) / r3;
}

static double static_fz_x2_y2 (double x, double y, double z, double r)
{
double x2 = x * x;
double y2 = y * y;
double r3 = r * r * r;

/****************
 * l = 3, m = 2 *
 ****************/

return z * (x2 - y2) / r3;
}

static double static_fxyz (double x, double y, double z, double r)
{
double r3 = r * r * r;

/*****************
 * l = 3, m = -2 *
 *****************/

return (x * y * z) / r3;
}

static double static_fx_x2_3y2 (double x, double y, double z, double r)
{
double x2 = x * x;
double y2 = y * y;
double r3 = r * r * r;

/****************
 * l = 3, m = 3 *
 ****************/

return x * (x2 - 3 * y2) / r3;
}

static double static_fy_3x2_y2 (double x, double y, double z, double r)
{
double x2 = x * x;
double y2 = y * y;
double r3 = r * r * r;

/*****************
 * l = 3, m = -3 *
 *****************/

return y * (3 * x2 - y2) / r3;
}

static double static_gz4 (double x, double y, double z, double r)
{
double z2 = z * z;
double z4 = z2 * z2;
double r2 = r * r;
double r4 = r2 * r2;

/****************
 * l = 4, m = 0 *
 ****************/

return (35 * z4 - 30 * z2 * r2 + 3 * r4) / r4;
}

static double static_gz3x (double x, double y, double z, double r)
{
double z2 = z * z;
double r2 = r * r;
double r4 = r2 * r2;

/****************
 * l = 4, m = 1 *
 ****************/

return x * z * (7 * z2 - 3 * r2) / r4;
}

static double static_gz3y (double x, double y, double z, double r)
{
double z2 = z * z;
double r2 = r * r;
double r4 = r2 * r2;

/*****************
 * l = 4, m = -1 *
 *****************/

return y * z * (7 * z2 - 3 * r2) / r4;
}

static double static_gz2_x2_y2 (double x, double y, double z, double r)
{
double x2 = x * x;
double y2 = y * y;
double z2 = z * z;
double r2 = r * r;
double r4 = r2 * r2;

/****************
 * l = 4, m = 2 *
 ****************/

return (x2 - y2) * (7 * z2 - r2) / r4;
}

static double static_gz2_xy (double x, double y, double z, double r)
{
double z2 = z * z;
double r2 = r * r;
double r4 = r2 * r2;

/*****************
 * l = 4, m = -2 *
 *****************/

return x * y * (7 * z2 - r2) / r4;
}

static double static_gzx3 (double x, double y, double z, double r)
{
double x2 = x * x;
double y2 = y * y;
double r4 = r * r * r * r;

/*****************
 * l = 4, m = 3 *
 *****************/

return x * z * (x2 - 3 * y2) / r4;
}

static double static_gzy3 (double x, double y, double z, double r)
{
double x2 = x * x;
double y2 = y * y;
double r4 = r * r * r * r;

/*****************
 * l = 4, m = -3 *
 *****************/

return y * z * (3 * x2 - y2) / r4;
}

static double static_gx4_y4 (double x, double y, double z, double r)
{
double x2 = x * x;
double x4 = x2 * x2;
double y2 = y * y;
double y4 = y2 * y2;
double r4 = r * r * r * r;

/*****************
 * l = 4, m = 4 *
 *****************/

return (x4 + y4 - 6 * x2 * y2) / r4;
}

static double static_gxy_x2_y2 (double x, double y, double z, double r)
{
double x2 = x * x;
double y2 = y * y;
double r4 = r * r * r * r;

/*****************
 * l = 4, m = -4 *
 *****************/

return x * y * (x2 - y2) / r4;
}

double static_hz5 (double x, double y, double z, double r)
{
double z2 = z * z;
double z3 = z2 * z;
double z5 = z3 * z2;
double r2 = r * r;
double r4 = r2 * r2;
double r5 = r4 * r;

/****************
 * l = 5, m = 0 *
 ****************/

return (63 * z5 - 70 * z3 * r2 + 15 * z * r4) / r5;
}

double static_hz4x (double x, double y, double z, double r)
{
double z2 = z * z;
double z4 = z2 * z2;
double r2 = r * r;
double r4 = r2 * r2;
double r5 = r4 * r;

/****************
 * l = 5, m = 1 *
 ****************/

return x * (21 * z4 - 14 * z2 * r2 + r4) / r5;
}

double static_hz4y (double x, double y, double z, double r)
{
double z2 = z * z;
double z4 = z2 * z2;
double r2 = r * r;
double r4 = r2 * r2;
double r5 = r4 * r;

/*****************
 * l = 5, m = -1 *
 *****************/

return y * (21 * z4 - 14 * z2 * r2 + r4) / r5;
}

double static_hz3_x2_y2 (double x, double y, double z, double r)
{
double x2 = x * x;
double y2 = y * y;
double z3 = z * z * z;
double r2 = r * r;
double r5 = r2 * r2 * r;

/****************
 * l = 5, m = 2 *
 ****************/

return (x2 - y2) * (3 * z3 - z * r2) / r5;
}

double static_hz3_xy (double x, double y, double z, double r)
{
double z3 = z * z * z;
double r2 = r * r;
double r5 = r2 * r2 * r;

/*****************
 * l = 5, m = -2 *
 *****************/

return x * y * (3 * z3 - z * r2) / r5;
}

double static_hz2x3 (double x, double y, double z, double r)
{
double x3 = x * x * x;
double y2 = y * y;
double z2 = z * z;
double r2 = r * r;
double r5 = r2 * r2 * r;

/****************
 * l = 5, m = 3 *
 ****************/

return (x3 - 3 * x * y2) * (9 * z2 - r2) / r5;
}

double static_hz2y3 (double x, double y, double z, double r)
{
double y3 = y * y * y;
double x2 = x * x;
double z2 = z * z;
double r2 = r * r;
double r5 = r2 * r2 * r;

/*****************
 * l = 5, m = -3 *
 *****************/

return (3 * x2 * y - y3) * (9 * z2 - r2) / r5;
}

double static_hz_x4_y4 (double x, double y, double z, double r)
{
double x2 = x * x;
double x4 = x2 * x2;
double y2 = y * y;
double y4 = y2 * y2;
double r2 = r * r;
double r5 = r2 * r2 * r;

/****************
 * l = 5, m = 4 *
 ****************/

return z * (x4 + y4 - 6 * x2 * y2) / r5;
}

double static_hxyz_x2_y2 (double x, double y, double z, double r)
{
double x2 = x * x;
double y2 = y * y;
double r2 = r * r;
double r5 = r2 * r2 * r;

/*****************
 * l = 5, m = -4 *
 *****************/

return x * y * z * (x2 - y2) / r5;
}

double static_hx5 (double x, double y, double z, double r)
{
double x2 = x * x;
double x3 = x2 * x;
double x5 = x3 * x2;
double y2 = y * y;
double y4 = y2 * y2;
double r2 = r * r;
double r5 = r2 * r2 * r;

/****************
 * l = 5, m = 5 *
 ****************/

return (x5 - 10 * x3 * y2 + 5 * x * y4) / r5;
}

double static_hy5 (double x, double y, double z, double r)
{
double x2 = x * x;
double x4 = x2 * x2;
double y2 = y * y;
double y3 = y2 * y;
double y5 = y3 * y2;
double r2 = r * r;
double r5 = r2 * r2 * r;

/*****************
 * l = 5, m = -5 *
 *****************/

return (y5 - 10 * x2 * y3 + 5 * x4 * y) / r5;
}

static void static_angular (int l, int m, double *c, gamgi_angular *f)
{
*f = NULL;

if (l == 0) { *c = 1; *f = static_s; }

if (l == 1)
  {
  if (m == -1) { *c = sqrt (3); *f = static_py; }
  if (m ==  0) { *c = sqrt (3); *f = static_pz; }
  if (m == +1) { *c = sqrt (3); *f = static_px; }
  }

if (l == 2)
  {
  if (m == -2) { *c = sqrt (15);       *f = static_dxy; }
  if (m == -1) { *c = sqrt (15);       *f = static_dyz; }
  if (m ==  0) { *c = sqrt (5 / 4.0);  *f = static_dz2; }
  if (m == +1) { *c = sqrt (15);       *f = static_dxz; }
  if (m == +2) { *c = sqrt (15 / 4.0); *f = static_dx2_y2; }
  }

if (l == 3)
  {
  if (m == -3) { *c = sqrt (35 / 8.0); *f = static_fy_3x2_y2; }
  if (m == -2) { *c = sqrt (105);       *f = static_fxyz; }
  if (m == -1) { *c = sqrt (21 / 8.0);  *f = static_fyz2; }
  if (m ==  0) { *c = sqrt (7 / 4.0);   *f = static_fz3; }
  if (m == +1) { *c = sqrt (21 / 8.0);  *f = static_fxz2; }
  if (m == +2) { *c = sqrt (105 / 4.0); *f = static_fz_x2_y2; }
  if (m == +3) { *c = sqrt (35 / 8.0);  *f = static_fx_x2_3y2; }
  }

if (l == 4)
  {
  if (m == -4) { *c = sqrt (315 / 4.0);  *f = static_gxy_x2_y2; }
  if (m == -3) { *c = sqrt (315 / 8.0);  *f = static_gzy3; }
  if (m == -2) { *c = sqrt (45 / 4.0);   *f = static_gz2_xy; }
  if (m == -1) { *c = sqrt (45 / 8.0);   *f = static_gz3y; }
  if (m ==  0) { *c = 3 / 8.0;           *f = static_gz4; }
  if (m == +1) { *c = sqrt (45 / 8.0);   *f = static_gz3x; }
  if (m == +2) { *c = sqrt (45 / 16.0);  *f = static_gz2_x2_y2; }
  if (m == +3) { *c = sqrt (315 / 8.0);  *f = static_gzx3; }
  if (m == +4) { *c = sqrt (315 / 64.0); *f = static_gx4_y4; }
  }

if (l == 5)
  {
  if (m == -5) { *c = sqrt (693 / 128.0); *f = static_hy5; }
  if (m == -4) { *c = sqrt (3465 / 4.0);  *f = static_hxyz_x2_y2; }
  if (m == -3) { *c = sqrt (385 / 128.0); *f = static_hz2y3; }
  if (m == -2) { *c = sqrt (1155 / 4.0);  *f = static_hz3_xy; }
  if (m == -1) { *c = sqrt (165 / 64.0);  *f = static_hz4y; }
  if (m ==  0) { *c = sqrt (11 / 64.0);   *f = static_hz5; }
  if (m == +1) { *c = sqrt (165 / 64.0);  *f = static_hz4x; }
  if (m == +2) { *c = sqrt (1155 / 16.0); *f = static_hz3_x2_y2; }
  if (m == +3) { *c = sqrt (385 / 128.0); *f = static_hz2x3; }
  if (m == +4) { *c = sqrt (3465 / 64.0); *f = static_hz_x4_y4; }
  if (m == +5) { *c = sqrt (693 / 128.0); *f = static_hx5; }
  }

*c /= sqrt (4.0 * GAMGI_MATH_PI);
}

void gamgi_chem_orbital_m (int l, gamgi_bool *sensitive)
{
switch (l)
  {
  case 0:
  sensitive[0] = FALSE;
  sensitive[1] = FALSE;
  sensitive[2] = FALSE;
  sensitive[3] = FALSE;
  sensitive[4] = FALSE;
  sensitive[6] = FALSE;
  sensitive[7] = FALSE;
  sensitive[8] = FALSE;
  sensitive[9] = FALSE;
  sensitive[10] = FALSE;
  break;

  case 1:
  sensitive[0] = FALSE;
  sensitive[1] = FALSE;
  sensitive[2] = FALSE;
  sensitive[3] = FALSE;
  sensitive[4] = TRUE;
  sensitive[6] = TRUE;
  sensitive[7] = FALSE;
  sensitive[8] = FALSE;
  sensitive[9] = FALSE;
  sensitive[10] = FALSE;
  break;

  case 2:
  sensitive[0] = FALSE;
  sensitive[1] = FALSE;
  sensitive[2] = FALSE;
  sensitive[3] = TRUE;
  sensitive[4] = TRUE;
  sensitive[6] = TRUE;
  sensitive[7] = TRUE;
  sensitive[8] = FALSE;
  sensitive[9] = FALSE;
  sensitive[10] = FALSE;
  break;

  case 3:
  sensitive[0] = FALSE;
  sensitive[1] = FALSE;
  sensitive[2] = TRUE;
  sensitive[3] = TRUE;
  sensitive[4] = TRUE;
  sensitive[6] = TRUE;
  sensitive[7] = TRUE;
  sensitive[8] = TRUE;
  sensitive[9] = FALSE;
  sensitive[10] = FALSE;
  break;

  case 4:
  sensitive[0] = FALSE;
  sensitive[1] = TRUE;
  sensitive[2] = TRUE;
  sensitive[3] = TRUE;
  sensitive[4] = TRUE;
  sensitive[6] = TRUE;
  sensitive[7] = TRUE;
  sensitive[8] = TRUE;
  sensitive[9] = TRUE;
  sensitive[10] = FALSE;
  break;

  case 5:
  sensitive[0] = TRUE;
  sensitive[1] = TRUE;
  sensitive[2] = TRUE;
  sensitive[3] = TRUE;
  sensitive[4] = TRUE;
  sensitive[6] = TRUE;
  sensitive[7] = TRUE;
  sensitive[8] = TRUE;
  sensitive[9] = TRUE;
  sensitive[10] = TRUE;
  break;
  }
}

void gamgi_chem_orbital_l (int n, gamgi_bool *sensitive)
{
switch (n)
  {
  case 1:
  sensitive[1] = FALSE;
  sensitive[2] = FALSE;
  sensitive[3] = FALSE;
  sensitive[4] = FALSE;
  sensitive[5] = FALSE;
  break;

  case 2:
  sensitive[1] = TRUE;
  sensitive[2] = FALSE;
  sensitive[3] = FALSE;
  sensitive[4] = FALSE;
  sensitive[5] = FALSE;
  break;

  case 3:
  sensitive[1] = TRUE;
  sensitive[2] = TRUE;
  sensitive[3] = FALSE;
  sensitive[4] = FALSE;
  sensitive[5] = FALSE;
  break;

  case 4:
  sensitive[1] = TRUE;
  sensitive[2] = TRUE;
  sensitive[3] = TRUE;
  sensitive[4] = FALSE;
  sensitive[5] = FALSE;
  break;

  case 5:
  sensitive[1] = TRUE;
  sensitive[2] = TRUE;
  sensitive[3] = TRUE;
  sensitive[4] = TRUE;
  sensitive[5] = FALSE;
  break;

  case 6:
  sensitive[1] = TRUE;
  sensitive[2] = TRUE;
  sensitive[3] = TRUE;
  sensitive[4] = TRUE;
  sensitive[5] = TRUE;
  break;
  }
}

void gamgi_chem_orbital_frame (gamgi_orbital *orbital)
{
double *points;
double radius;
int *lines;
int x, y, z, n;

points = orbital->points;
radius = orbital->radius;

/**********************
 * add 8 frame points *
 **********************/

n = 0;
for (z = -1; z <= 1; z += 2)
  {
  for (y = -1; y <= 1; y += 2)
    {
    for (x = -1; x <= 1; x += 2)
      {
      points[n++] = radius * x;
      points[n++] = radius * y;
      points[n++] = radius * z;
      }
    }
  }

/*********************************************************
 * create array with 12 edges: edges x, edges y, edges z *
 *********************************************************/

orbital->lines = lines = (int *) malloc (25 * sizeof (int));
orbital->n_lines = 25;

n = 0;
lines[n++] = 12;

lines[n++] = 0;
lines[n++] = 1;
lines[n++] = 2;
lines[n++] = 3;
lines[n++] = 4;
lines[n++] = 5;
lines[n++] = 6;
lines[n++] = 7;

lines[n++] = 0;
lines[n++] = 2;
lines[n++] = 1;
lines[n++] = 3;
lines[n++] = 4;
lines[n++] = 6;
lines[n++] = 5;
lines[n++] = 7;

lines[n++] = 0;
lines[n++] = 4;
lines[n++] = 1;
lines[n++] = 5;
lines[n++] = 2;
lines[n++] = 6;
lines[n++] = 3;
lines[n++] = 7;
}

double static_density (double r, double value, gamgi_angular function_a)
{
double theta, phi;
double x, y, z, s;
double end, step;
double f;

end = GAMGI_MATH_PI / 2.0;
step = GAMGI_MATH_PI / 20.0;

for (theta = 0.0; theta < end; theta += step)
  {
  z = r * cos (theta);
  s = r * sin (theta);

  for (phi = 0.0; phi < end; phi += step)
    {
    x = s * cos (phi);
    y = s * sin (phi);

    f = (*function_a) (x, y, z, r);
    if (value * f * f > GAMGI_CHEM_ORBITAL_DENSITY) return TRUE;
    }
  }

return FALSE;
}

int gamgi_chem_orbital_points (int n)
{
return GAMGI_CHEM_ORBITAL_POINTS * n;
}

double gamgi_chem_orbital_radius (int n, int l, int m, double charge)
{
gamgi_radial function_r;
gamgi_angular function_a;
double za, r, ho, rho, r_max;
double constant, constant_a, constant_z, constant_r;
double d, d_old, d_old_old;
double f;

if (n < 1 || n > 6) return 0.0;
if (l < 0 || l > n - 1) return 0.0;
if (m < -l || m > l) return 0.0;

za = charge / GAMGI_CHEM_A0;
ho = 2 * za / n;
constant_z = pow (za, 1.5);

static_radial (n, l, &constant_r, &function_r);
static_angular (l, m, &constant_a, &function_a);

/***********************************************
 * get the last maximum for the radial density *
 ***********************************************/

r_max = GAMGI_CHEM_ORBITAL_MAX;

d_old = d_old_old = 0.0;
constant = constant_r * constant_z;
for (r = 0.0; r < GAMGI_CHEM_ORBITAL_MAX; r += GAMGI_CHEM_ORBITAL_STEP)
  {
  rho = r * ho;
  f = constant * exp (-rho / 2) * (*function_r) (rho);
  d = f * f * r * r;
  if (d < d_old && d_old > d_old_old)
    r_max = r - GAMGI_CHEM_ORBITAL_STEP;
  d_old_old = d_old;
  d_old = d;
  }

/**********************************************
 * start checking the default maximum density *
 * some distance after the last maximum       *
 **********************************************/

r_max += GAMGI_CHEM_ORBITAL_SHIFT;

/*********************************************
 * find the radius where the density becomes *
 * lower than the default maximum density    *
 *********************************************/

constant *= constant_a;
for (r = r_max; r < GAMGI_CHEM_ORBITAL_MAX; r += GAMGI_CHEM_ORBITAL_STEP)
  {
  rho = r * ho;
  f = constant * exp (-rho / 2) * (*function_r) (rho);
  if (static_density (r, f * f, function_a) == FALSE) return r;
  }

return GAMGI_CHEM_ORBITAL_MAX;
}

gamgi_bool gamgi_chem_orbital_create (gamgi_orbital *orbital,
int *octants, int seed)
{
gamgi_radial function_r;
gamgi_angular function_a;
double *points;
double max = RAND_MAX;
double charge, za, r, ho, rho;
double density, radius, f, x, y, z;
double constant, constant_z, constant_r, constant_a;
int *dots;
int offset, offset_negative, offset_positive;
int n_points, i, bad; 
char mask, octant;

/*******************************************************
 * if required, add frame data to points, lines arrays *
 *******************************************************/

dots = orbital->dots;
density = orbital->density;
radius = orbital->radius;
n_points = orbital->n_points;

orbital->n_points += dots[0];
points = orbital->points = (double *) malloc (orbital->n_points * 3 * sizeof (double));
if (dots[0] > 0) gamgi_chem_orbital_frame (orbital);

static_radial (orbital->n, orbital->l, &constant_r, &function_r);
static_angular (orbital->l, orbital->m, &constant_a, &function_a);

charge = 1.0;
za = charge / GAMGI_CHEM_A0;
ho = 2 * za / orbital->n;
constant_z = pow (za, 1.5);
constant = constant_z * constant_r * constant_a;

/***************************************************
 * start negative phase points from the beginning  *
 * of the array (after frame points if available)  *
 * and positive phase points from the end (even if *
 * the same color will be used to show both sets   *
 ***************************************************/

offset_positive = dots[0];
offset_negative = orbital->n_points - 1;

/******************************
 * set random number sequence *
 ******************************/

srand (seed);

/***********************************************
 * set octants mask with user information      *
 * determining which octants should be visible *
 ***********************************************/

mask = 255;
for (i = 0; i < 8; i++)
  if (octants[i] == 0)
    mask &= ~(int) pow (2, 7 - i);

/*******************************************
 * i_bad is just to prevent infinite loops *
 *******************************************/

i = bad = 0;
while (i < n_points && bad < GAMGI_CHEM_ORBITAL_BAD)
  {
  x = 2.0 * rand () / max - 1.0;
  y = 2.0 * rand () / max - 1.0;
  z = 2.0 * rand () / max - 1.0;
  r = x * x + y * y + z * z;
  if (r > 1.0) continue;

  octant = ~0;
  if (x < 0) octant &= ~85;
  if (x > 0) octant &= ~170;
  if (y < 0) octant &= ~51;
  if (y > 0) octant &= ~204;
  if (z < 0) octant &= ~15;
  if (z > 0) octant &= ~240;
  octant &= mask;
  if (octant == 0) continue;

  r = radius * sqrt (r);
  rho = r * ho;
  x *= radius;
  y *= radius;
  z *= radius;

  f = constant * exp (-rho / 2) * (*function_r) (rho) * (*function_a) (x, y, z, r);
  if (f * f < density) { bad++; continue; }

  i++; bad = 0;
  if (f >= 0) offset = offset_positive++;
  if (f < 0) offset = offset_negative--;

  points[3*offset + 0] = x;
  points[3*offset + 1] = y;
  points[3*offset + 2] = z;
  }

/***************************************
 * set positive (dots[1]) and negative *
 * (dots[2]) number of density points  *
 ***************************************/
 
dots[1] = offset_positive - dots[0];
if (dots[2] == TRUE) dots[2] = orbital->n_points - 1 - offset_negative;
else dots[1] += orbital->n_points - 1 - offset_negative;
if (dots[0] + dots[1] + dots[2] == orbital->n_points) return TRUE;

return FALSE;
}
