/******************************************
 *
 * $GAMGI/src/math/gamgi_math_node.c
 *
 * Copyright (C) 2004 Carlos Pereira
 *
 * Distributed under the terms of the GNU
 * General Public License: $GAMGI/LICENSE
 *
 */

#include "gamgi_engine.h"
#include "gamgi_phys.h"

gamgi_bool gamgi_math_node_check (gamgi_enum lattice, gamgi_enum node)
{
/********************************************
 * corner node: acceptable for all lattices *
 ********************************************/

if (node == FALSE) return TRUE;

/***************************************************
 * centered node: acceptable for specific lattices *
 ***************************************************/

switch (lattice)
  {
  /*****************
   * body-centered *
   *****************/

  case GAMGI_PHYS_CUBIC_I: case GAMGI_PHYS_TETRAGONAL_I:
  case GAMGI_PHYS_ORTHORHOMBIC_I:
  if (node == GAMGI_PHYS_NODE_111) return TRUE;
  break;

  /*****************
   * face-centered *
   *****************/

  case GAMGI_PHYS_CUBIC_F: case GAMGI_PHYS_ORTHORHOMBIC_F:
  if (node == GAMGI_PHYS_NODE_110) return TRUE;
  if (node == GAMGI_PHYS_NODE_101) return TRUE;
  if (node == GAMGI_PHYS_NODE_011) return TRUE;
  break;

  /*****************
   * base-centered *
   *****************/

  case GAMGI_PHYS_ORTHORHOMBIC_C: case GAMGI_PHYS_MONOCLINIC_C:
  if (node == GAMGI_PHYS_NODE_110) return TRUE;
  break;

  /*********************************
   * rombohedral, obverse-centered *
   *********************************/

  case GAMGI_PHYS_HEXAGONAL_R:
  if (node == GAMGI_PHYS_NODE_211) return TRUE;
  if (node == GAMGI_PHYS_NODE_122) return TRUE;
  break;

  /****************
   * non-centered *
   ****************/

  default:
  break;
  }

/*******************************
 * Error: lattice does no have *
 * centered nodes of this type *
 *******************************/

return FALSE;
}

void gamgi_math_node_primitive (gamgi_enum lattice,
int c1, int c2, int c3, int c4, int *p1, int *p2, int *p3)
{
int a, b, c;

/******************************************************
 * convert conventional to primitive node coordinates *
 ******************************************************/

switch (lattice)
  {
  case GAMGI_PHYS_CUBIC_I: case GAMGI_PHYS_TETRAGONAL_I:
  case GAMGI_PHYS_ORTHORHOMBIC_I:
  a = c1 + c3;
  if (c4 == GAMGI_PHYS_NODE_111) a++;
  b = c1 + c2;
  if (c4 == GAMGI_PHYS_NODE_111) b++;
  c = c2 + c3;
  if (c4 == GAMGI_PHYS_NODE_111) c++;
  break;

  case GAMGI_PHYS_CUBIC_F: case GAMGI_PHYS_ORTHORHOMBIC_F:
  a = c1 - c2 + c3;
  if (c4 == GAMGI_PHYS_NODE_101) a++;
  b = c1 + c2 - c3;
  if (c4 == GAMGI_PHYS_NODE_110) b++;
  c = -c1 + c2 + c3;
  if (c4 == GAMGI_PHYS_NODE_011) c++;
  break;

  case GAMGI_PHYS_ORTHORHOMBIC_C: case GAMGI_PHYS_MONOCLINIC_C:
  a = c1 - c2;
  b = c1 + c2;
  if (c4 == GAMGI_PHYS_NODE_110) b++;
  c = c3;
  break;

  case GAMGI_PHYS_HEXAGONAL_R:
  a = c1 + c3;
  if (c4 == GAMGI_PHYS_NODE_122 || c4 == GAMGI_PHYS_NODE_211) a++;
  b = -c1 + c2 + c3;
  if (c4 == GAMGI_PHYS_NODE_122) b++;
  c = -c2 + c3;
  break;

  default:
  a = c1;
  b = c2;
  c = c3;
  break;
  }

*p1 = a;
*p2 = b;
*p3 = c;
}

void gamgi_math_node_conventional (gamgi_enum lattice,
int p1, int p2, int p3, int *c1, int *c2, int *c3, int *c4)
{
int t, a, b, c, d;

/***********************************
 * this unneeded initialization    *
 * avoids a O3 compilation warning *
 ***********************************/

a = b = c = d = 0;

/******************************************************
 * convert primitive to conventional node coordinates *
 ******************************************************/

switch (lattice)
  {
  case GAMGI_PHYS_CUBIC_I: case GAMGI_PHYS_TETRAGONAL_I:
  case GAMGI_PHYS_ORTHORHOMBIC_I:
  t = p1 + p2 - p3;
  if (t % 2 == 0) a = t;
  if (t % 2 == 1 || t % 2 == -1) a = t - 1;
  a /= 2;

  t = -p1 + p2 + p3;
  if (t % 2 == 0) b = t;
  if (t % 2 == 1 || t % 2 == -1) b = t - 1;
  b /= 2;

  t = p1 - p2 + p3;
  if (t % 2 == 0) c = t; 
  if (t % 2 == 1 || t % 2 == -1) c = t - 1;
  c /= 2;

  t = p1 + p2 + p3;
  if (t % 2 == 0) d = FALSE;
  if (t % 2 == 1 || t % 2 == -1) d = GAMGI_PHYS_NODE_111;
  break;

  case GAMGI_PHYS_CUBIC_F: case GAMGI_PHYS_ORTHORHOMBIC_F:
  t = p1 + p2;
  if (t % 2 == 0) a = t; 
  if (t % 2 == 1 || t % 2 == -1) a = t - 1;
  a /= 2;

  t = p2 + p3;
  if (t % 2 == 0) b = t;
  if (t % 2 == 1 || t % 2 == -1) b = t - 1; 
  b /= 2;

  t = p1 + p3;
  if (t % 2 == 0) c = t;
  if (t % 2 == 1 || t % 2 == -1) c = t - 1;
  c /= 2;

  if (p1 % 2 == 0 && p2 % 2 == 0 && p3 % 2 == 0) d = FALSE;
  if (p1 % 2 != 0 && p2 % 2 != 0 && p3 % 2 != 0) d = FALSE;
  if (p1 % 2 != 0 && p2 % 2 == 0 && p3 % 2 != 0) d = GAMGI_PHYS_NODE_110;
  if (p1 % 2 == 0 && p2 % 2 != 0 && p3 % 2 == 0) d = GAMGI_PHYS_NODE_110;
  if (p1 % 2 == 0 && p2 % 2 != 0 && p3 % 2 != 0) d = GAMGI_PHYS_NODE_101;
  if (p1 % 2 != 0 && p2 % 2 == 0 && p3 % 2 == 0) d = GAMGI_PHYS_NODE_101;
  if (p1 % 2 != 0 && p2 % 2 != 0 && p3 % 2 == 0) d = GAMGI_PHYS_NODE_011;
  if (p1 % 2 == 0 && p2 % 2 == 0 && p3 % 2 != 0) d = GAMGI_PHYS_NODE_011;
  break;

  case GAMGI_PHYS_ORTHORHOMBIC_C: case GAMGI_PHYS_MONOCLINIC_C:
  t = p1 + p2;
  if (t % 2 == 0) a = t;
  if (t % 2 == 1 || t % 2 == -1) a = t - 1; 
  a /= 2;
  
  t = -p1 + p2;
  if (t % 2 == 0) b = t;
  if (t % 2 == 1 || t % 2 == -1) b = t - 1;
  b /= 2;

  c = p3;

  t = p1 + p2;
  if (t % 2 == 0) d = FALSE;
  if (t % 2 == 1 || t % 2 == -1) d = GAMGI_PHYS_NODE_110;
  break;

  case GAMGI_PHYS_HEXAGONAL_R:
  t = 2 * p1 - p2 - p3;
  if (t % 3 == 0) a = t;
  if (t % 3 == 1 || t % 3 == -2) a = t - 1;
  if (t % 3 == 2 || t % 3 == -1) a = t - 2;
  a /= 3;

  t = p1 + p2 - 2 * p3;
  if (t % 3 == 0) b = t;
  if (t % 3 == 1 || t % 3 == -2) b = t - 1;
  if (t % 3 == 2 || t % 3 == -1) b = t - 2;
  b /= 3;

  t = p1 + p2 + p3;
  if (t % 3 == 0) c = t;
  if (t % 3 == 1 || t % 3 == -2) c = t - 1;
  if (t % 3 == 2 || t % 3 == -1) c = t - 2;
  c /= 3;

  t = 2 * p1 - p2 - p3;
  if (t % 3 == 0) d = FALSE;
  if (t % 3 == 1 || t % 3 == -2) d = GAMGI_PHYS_NODE_122;
  if (t % 3 == 2 || t % 3 == -1) d = GAMGI_PHYS_NODE_211;
  break;

  default:
  a = p1;
  b = p2;
  c = p3;
  d = FALSE;
  break;
  }

*c1 = a;
*c2 = b;
*c3 = c;
*c4 = d;
}
