namespace Galago
{
	using System;
	using System.Collections;
	using System.ComponentModel;
	using System.Runtime.InteropServices;
	using System.Reflection;

	public abstract class Object : IWrapper, IDisposable
	{
		private IntPtr _obj;
		private IntPtr _class;
		private bool disposed = false;
		private Hashtable signals;
		private EventHandlerList eventHandlers;
		private static Hashtable Objects = new Hashtable();
		private static Hashtable Classes = new Hashtable();

		~Object()
		{
			Dispose();
		}

		[DllImport("libgalago.so")]
		static extern IntPtr galago_object_get_class(IntPtr obj);

		protected Object()
		{
		}

		protected Object(IntPtr raw)
		{
			Raw = raw;
		}

		protected virtual IntPtr Raw
		{
			get { return _obj; }
			set
			{
				if (_obj != IntPtr.Zero)
					Objects.Remove(_obj);

				_obj = value;

				if (_obj == IntPtr.Zero)
				{
					_class = IntPtr.Zero;
					return;
				}

				_class = galago_object_get_class(_obj);

				Objects[_obj] = new WeakReference(this);
			}
		}

		[DllImport("libgalago.so")]
		static extern IntPtr galago_object_unref(IntPtr obj);

		public void Dispose()
		{
			if (disposed)
				return;

			disposed = true;
			Objects.Remove(_obj);
		}

		public static Object Wrap(IntPtr raw)
		{
			Object obj;
			WeakReference weak_ref = Objects[raw] as WeakReference;

			if (weak_ref != null && weak_ref.IsAlive)
			{
				obj = weak_ref.Target as Object;

				return obj;
			}

			obj = ObjectManager.CreateObject(raw);

			if (obj == null)
				return null;

			Objects[raw] = new WeakReference(obj);

			return obj;
		}

		protected Hashtable Signals
		{
			get
			{
				if (signals == null)
					signals = new Hashtable();

				return signals;
			}
		}

		protected EventHandlerList EventHandlers
		{
			get
			{
				if (eventHandlers == null)
					eventHandlers = new EventHandlerList();

				return eventHandlers;
			}
		}

/*
		private static void ConnectDefaultHandlers(System.Type t)
		{
			foreach (MethodInfo minfo in t.GetMethods(
				BindingFlags.Instance | BindingFlags.NonPublic |
				BindingFlags.Public | BindingFlags.DeclaredOnly))
			{
				MethodInfo baseInfo = minfo.GetBaseDefinition();

				if (baseInfo == minfo)
					continue;

				foreach (object attr in baseInfo.GetCustomAttributes(true))
				{
					if (attr.ToString() !=
						"Galago.DefaultSignalHandlerAttribute")
					{
						continue;
					}

					DefaultSignalHandlerAttribute sigAttr =
						attr as DefaultSignalHandlerAttribute;

					MethodInfo connector = sigAttr.Type.GetMethod(
						sigAttr.ConnectionMethod,
						BindingFlags.Static | BindingFlags.NonPublic);

					connector.Invoke(null, null);

					break;
				}
			}
		}
*/

		protected void AddSignalHandler(string signal, Type type,
										Type argsType, Delegate del)
		{
			if (EventHandlers[signal] == null)
			{
				object[] args = new object[4];
				args[0] = this;
				args[1] = signal;
				args[2] = argsType;
				args[3] = del;

				Signals[signal] = Activator.CreateInstance(type, args);
			}
			else
				((Galago.SignalCallback)Signals[signal]).AddDelegate(del);

			EventHandlers.AddHandler(signal, del);
		}

		protected void RemoveSignalHandler(string signal, Delegate del)
		{
			SignalCallback cb = Signals[signal] as SignalCallback;
			EventHandlers.RemoveHandler(signal, del);

			if (cb == null)
				return;

			cb.RemoveDelegate(del);

			if (EventHandlers[signal] == null) {
				Signals.Remove(signal);
				cb.Dispose();
			}
		}

		public IntPtr Class
		{
			get { return _class; }
		}

		public IntPtr Handle
		{
			get { return _obj; }
		}

		[DllImport("libgalago.so")]
		static extern IntPtr galago_class_get_name(IntPtr raw);

		public string ClassName
		{
			get
			{
				IntPtr rawRet = galago_class_get_name(Handle);

				return Marshal.PtrToStringAnsi(rawRet);
			}
		}

		[DllImport("libgalago.so")]
		static extern IntPtr galago_class_get_dbus_iface(IntPtr raw);

		public string BDusInterface
		{
			get
			{
				IntPtr rawRet = galago_class_get_dbus_iface(Class);

				return Marshal.PtrToStringAnsi(rawRet);
			}
		}

		[DllImport("libgalago.so")]
		static extern IntPtr galago_object_get_dbus_path(IntPtr raw);

		[DllImport("libgalago.so")]
		static extern void galago_object_set_dbus_path(IntPtr raw, string str);

		public string DBusPath
		{
			get
			{
				IntPtr rawRet = galago_object_get_dbus_path(Handle);

				return Marshal.PtrToStringAnsi(rawRet);
			}

			set
			{
				galago_object_set_dbus_path(Handle, value);
			}
		}

		[DllImport("libgalago.so")]
		static extern bool galago_object_is_watched(IntPtr raw);

		[DllImport("libgalago.so")]
		static extern void galago_object_set_watch(IntPtr raw, bool watch);

		public bool Watched
		{
			get { return galago_object_is_watched(Handle); }
			set { galago_object_set_watch(Handle, value);  }
		}
	}
}
