/* GeomOrbXYZ.c */
/**********************************************************************************************************
Copyright (c) 2002-2009 Abdul-Rahman Allouche. All rights reserved

Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated
documentation files (the Gabedit), to deal in the Software without restriction, including without limitation
the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the Software,
and to permit persons to whom the Software is furnished to do so, subject to the following conditions:

  The above copyright notice and this permission notice shall be included in all copies or substantial portions
  of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
DEALINGS IN THE SOFTWARE.
************************************************************************************************************/


#include "../../Config.h"
#include "GlobalOrb.h"
#include "UtilsOrb.h"
#include "../Utils/AtomsProp.h"
#include "../Utils/Vector3d.h"
#include "../Utils/UtilsInterface.h"
#include "../Utils/Transformation.h"
#include "../Utils/Constants.h"
#include "../Geometry/GeomGlobal.h"
#include "../Geometry/OpenBabel.h"
#include "Sphere.h"
#include "../Utils/Utils.h"
#include "GLArea.h"
#include "StatusOrb.h"
#include "AtomicOrbitals.h"
#include "BondsOrb.h"

#ifdef G_OS_WIN32
#include <fcntl.h>
#include <io.h>
#else 
#include <unistd.h>
#endif

#define MAXNAME 6
#define MAXATOMTYPE 4
#define MAXRESIDUENAME 4
#define MAXSYMBOL 2

typedef struct _OldGeometry
{
	gint numberOfAtoms;
	gchar** symbols;
	gfloat* X;
	gfloat* Y;
	gfloat* Z;
}OldGeometry;
static OldGeometry oldGeometry = {0,NULL,NULL,NULL,NULL};

/***************************************************************************************/
static void free_old_geometry()
{
	gint i;
	if(oldGeometry.numberOfAtoms==0)return;
	if(oldGeometry.X) g_free(oldGeometry.X);
	if(oldGeometry.Y) g_free(oldGeometry.Y);
	if(oldGeometry.Z) g_free(oldGeometry.Z);
	for(i=0;i<oldGeometry.numberOfAtoms;i++)
		if(oldGeometry.symbols[i])g_free(oldGeometry.symbols[i]);
	if(oldGeometry.symbols) g_free(oldGeometry.symbols);
	oldGeometry.numberOfAtoms = 0;
	oldGeometry.X = NULL;
	oldGeometry.Y = NULL;
	oldGeometry.Z = NULL;
	oldGeometry.symbols = NULL;
}
/***************************************************************************************/
void reset_old_geometry()
{
	gint i;
	free_old_geometry();
	if(Ncenters<1) return;
	oldGeometry.numberOfAtoms = Ncenters;
	oldGeometry.X = g_malloc(oldGeometry.numberOfAtoms*sizeof(gfloat));
	oldGeometry.Y = g_malloc(oldGeometry.numberOfAtoms*sizeof(gfloat));
	oldGeometry.Z = g_malloc(oldGeometry.numberOfAtoms*sizeof(gfloat));
	oldGeometry.symbols = g_malloc(oldGeometry.numberOfAtoms*sizeof(gchar*));
	for(i=0;i<oldGeometry.numberOfAtoms;i++)
	{
		oldGeometry.X[i] = GeomOrb[i].C[0];
		oldGeometry.Y[i] = GeomOrb[i].C[1];
		oldGeometry.Z[i] = GeomOrb[i].C[2];
		oldGeometry.symbols[i] = g_strdup(GeomOrb[i].Symb);
	}

}
/***************************************************************************************/
gboolean this_is_a_new_geometry()
{
	gint i;
	if(oldGeometry.numberOfAtoms<1) return TRUE;
	if(oldGeometry.numberOfAtoms != Ncenters) return TRUE;
	for(i=0;i<oldGeometry.numberOfAtoms;i++)
		if(strcmp(GeomOrb[i].Symb,oldGeometry.symbols[i]) != 0) return TRUE;

	for(i=0;i<oldGeometry.numberOfAtoms;i++)
		if(fabs(GeomOrb[i].C[0]-oldGeometry.X[i])>1e-6) return TRUE;
	for(i=0;i<oldGeometry.numberOfAtoms;i++)
		if(fabs(GeomOrb[i].C[1]-oldGeometry.Y[i])>1e-6) return TRUE;
	for(i=0;i<oldGeometry.numberOfAtoms;i++)
		if(fabs(GeomOrb[i].C[2]-oldGeometry.Z[i])>1e-6) return TRUE;
	
	return FALSE;
}
/***************************************************************************************/
void free_geometry()
{
	gint i;
	set_status_label_info("File Name","Nothing");
	set_status_label_info("File Type","Nothing");
	set_status_label_info("Geometry","Nothing");
	Dipole.def = FALSE;
	reset_old_geometry();
	if(!GeomOrb)
	{
		Ncenters = 0;
		return;
	}
	free_atomic_orbitals();
	for(i=0;i<Ncenters;i++)
		if(GeomOrb[i].Symb)
			g_free(GeomOrb[i].Symb);
	g_free(GeomOrb);
	GeomOrb = NULL;
	Ncenters = 0;
}
/*****************************************/
gfloat get_cube_length()
{
 gint i,j;
 gfloat min;
 gfloat max;
 
 min = GeomOrb[0].C[0];
 max = GeomOrb[0].C[0];
 for(i=0;i<Ncenters;i++)
 {
   for(j=0;j<3;j++)
   {
	if(min>GeomOrb[i].C[j])
		min = GeomOrb[i].C[j];
	if(max<GeomOrb[i].C[j])
		max = GeomOrb[i].C[j];
   }
 }
 return (max -min );
}
/*****************************************/
void reset_grid_limits()
  {
 	gint i;
	gfloat t = get_cube_length();
	if(fabs(t)<1e-10)
		t = 5.0;
  	for(i=0;i<3;i++)
   		limits.MinMax[0][i] = -t;
  	for(i=0;i<3;i++)
   		limits.MinMax[1][i] = t;
  }
/*****************************************/

void create_xyz_gl_spheres_list()
{
	V4d Specular = {1.0f,1.0f,1.0f,1.0f};
	V4d Diffuse  = {0.0f,0.0f,0.0f,1.0f};
	V4d Ambiant  = {0.0f,0.0f,0.0f,1.0f};
	int i;
	int k;
	for(i=0;i<Ncenters;i++)
	{
		Diffuse[0] = GeomOrb[i].Prop.color.red/(gfloat)65535;
		Diffuse[1] = GeomOrb[i].Prop.color.green/(gfloat)65535;
		Diffuse[2] = GeomOrb[i].Prop.color.blue/(gfloat)65535;
		for(k=0;k<3;k++)
			Ambiant[k] = Diffuse[k]*0.8;

		GeomOrb[i].Sphere = Sphere_Get_List(
			i+1,GeomOrb[i].Prop.radii,
			Specular,Diffuse,Ambiant);
	}
}

/*****************************************/
gint GetNumType(char *Symb)
{
 gint i;
 for(i=0;i<Ncenters;i++)
 {
   	if(!strcmp(Symb,GeomOrb[i].Symb)) 
           return GeomOrb[i].NumType;
 }
 return -1;
}
/*****************************************/
void DefineType()
{
 gint i,j;
 gint OK;
 Ntype =1;
 GeomOrb[0].NumType=0;
 for(i=1;i<Ncenters;i++)
 {
   OK =1;
   for(j=0;j<i;j++)
   {
   	if(!strcmp(GeomOrb[i].Symb,GeomOrb[j].Symb)) 
   	{
                GeomOrb[i].NumType= GeomOrb[j].NumType;
        	OK =0;
		break;
   	}
   }
   if(OK)
   {
        GeomOrb[i].NumType= Ntype;
	Ntype++;
   }
 }
 /*
 Debug("Types = ");
 for(i=0;i<Ncenters;i++)
	 Debug("%d ", GeomOrb[i].NumType);
 Debug("\n");
 */
}
/*****************************************/
void PrintGeomOrb()
{
 gint i,j;
 Debug("Geometry(in au)\n");
 Debug("===============\n\n");
 for(i=0;i<Ncenters;i++)
 {
   Debug("%3s ",GeomOrb[i].Symb);
   for(j=0;j<3;j++)
   	Debug("%9.6f ",GeomOrb[i].C[j]);
   Debug("\n");
 }
}
/******************************************************************************************************************/
gboolean gl_read_molden_gabedit_geom_conv_file(gchar *fileName, gint geometryNumber, GabEditTypeFile type)
{
	gchar* t;
	gboolean OK;
	gchar* AtomCoord[5];
	FILE *file;
	gint i;
	gint j;
	gint l;
	gint numgeom;
	gchar *pdest;
	gint nn;
	gchar* tmp;


	free_data_all();
	tmp = get_name_file(fileName);
	set_status_label_info("File Name",tmp);
	g_free(tmp);
	if(type == GABEDIT_TYPEFILE_GABEDIT) set_status_label_info("File Type","Gabedit");
	if(type == GABEDIT_TYPEFILE_MOLDEN) set_status_label_info("File Type","Molden");

	file = FOpen(fileName, "rb");

	if(file ==NULL)
	{
		t = g_strdup_printf("Sorry\nI can not open %s  file ",fileName);
		Message(t," Error ",TRUE);
		g_free(t);
		set_status_label_info("Geometry","Nothing");
		return FALSE;
	}
	t = g_malloc(BSIZE*sizeof(gchar));
	for(i=0;i<5;i++) AtomCoord[i] = g_malloc(BSIZE*sizeof(gchar));

	numgeom =0;
	OK=TRUE;
	while(!feof(file))
	{
		if(!fgets(t, BSIZE,file))break;
		pdest = strstr( t, "[GEOMETRIES]");
		if(pdest && strstr(t,"ZMAT"))
		{
			if(type == GABEDIT_TYPEFILE_MOLDEN)
				sprintf(t,"Sorry\nMolden file with ZMAT coordinate is not supported by Gabedit");
			if(type == GABEDIT_TYPEFILE_GABEDIT)
				sprintf(t,"Sorry\nGabedit file with ZMAT coordinate is not supported by Gabedit");

			Message(t," Error ",TRUE);
			g_free(t);
			set_status_label_info("Geometry","Nothing");
			return FALSE;
		}	
 		if (pdest)
		{
			while(!feof(file))
			{
				if(!fgets(t, BSIZE,file))break;

				str_delete_n(t);
				delete_last_spaces(t);
				delete_first_spaces(t);
				if(!isInteger(t))break;
               			numgeom++;
				if(numgeom == geometryNumber)
				{
					nn = atoi(t);
					if(nn<1)break;
    					GeomOrb=g_malloc(nn*sizeof(TypeGeomOrb));
					if(!fgets(t, BSIZE,file))break; /* title */
					for(j=0; j<nn; j++)
					{
						if(!fgets(t, BSIZE,file))break;
    						sscanf(t,"%s %s %s %s",AtomCoord[0],AtomCoord[1],AtomCoord[2],AtomCoord[3]);
						AtomCoord[0][0]=toupper(AtomCoord[0][0]);
						l=strlen(AtomCoord[0]);
						if (l==2) AtomCoord[0][1]=tolower(AtomCoord[0][1]);
    						GeomOrb[j].Symb=g_strdup(AtomCoord[0]);
    						for(i=0;i<3;i++) GeomOrb[j].C[i]=atof(AtomCoord[i+1])*ANG_TO_BOHR;
						GeomOrb[j].Prop = prop_atom_get(GeomOrb[j].Symb);
						GeomOrb[j].partialCharge = 0.0;
						GeomOrb[j].nuclearCharge = get_atomic_number_from_symbol(GeomOrb[j].Symb);
					}
 					Ncenters = nn;
					OK = TRUE;
					break;
				}
				else
				{
					nn = atoi(t);
					if(!fgets(t, BSIZE,file)) break;
					for(i=0; i<nn; i++)
						if(!fgets(t, BSIZE,file))break;
					if(i!=nn) { OK = FALSE; break;}
				}
				if(!OK) break;
			}
		}
		if(!OK) break;
	}

	fclose(file);
 	g_free(t);
 	for(i=0;i<5;i++) g_free(AtomCoord[i]);
 	if(Ncenters == 0 ) { if(GeomOrb) g_free(GeomOrb); }
 	else DefineType();

	buildBondsOrb();
	reset_grid_limits();
	init_atomic_orbitals();
	set_status_label_info("Geometry","Ok");
	RebuildGeom = TRUE;
	if(this_is_a_new_geometry()) free_objects_all();
	glarea_rafresh(GLArea);
	return TRUE;
}
/*********************************************************************/
gboolean gl_read_xyz_file_geomi(gchar *fileName,gint geometryNumber)
{
	gchar* t;
	gboolean OK;
	gchar* AtomCoord[5];
	FILE *file;
	gint i;
	gint j;
	gint l;
	gint numgeom;
	gint nn;
	gchar* tmp;


	free_data_all();
	tmp = get_name_file(fileName);
	set_status_label_info("File Name",tmp);
	g_free(tmp);
	set_status_label_info("File Type","XYZ");

	file = FOpen(fileName, "rb");

	if(file ==NULL)
	{
		t = g_strdup_printf("Sorry\nI can not open %s  file ",fileName);
		Message(t," Error ",TRUE);
		g_free(t);
		set_status_label_info("Geometry","Nothing");
		return FALSE;
	}
	t = g_malloc(BSIZE*sizeof(gchar));
	for(i=0;i<5;i++) AtomCoord[i] = g_malloc(BSIZE*sizeof(gchar));

	numgeom =0;
	OK=TRUE;
	while(!feof(file))
	{
		if(!fgets(t, BSIZE,file))break;

		str_delete_n(t);
		delete_last_spaces(t);
		delete_first_spaces(t);
		if(!isInteger(t))break;
		numgeom++;
		if(numgeom == geometryNumber)
		{
			nn = atoi(t);
			if(nn<1)break;
    			GeomOrb=g_malloc(nn*sizeof(TypeGeomOrb));
			if(!fgets(t, BSIZE,file))break; /* title */
			for(j=0; j<nn; j++)
			{
				if(!fgets(t, BSIZE,file))break;
    				sscanf(t,"%s %s %s %s",AtomCoord[0],AtomCoord[1],AtomCoord[2],AtomCoord[3]);
				AtomCoord[0][0]=toupper(AtomCoord[0][0]);
				l=strlen(AtomCoord[0]);
				if (l==2) AtomCoord[0][1]=tolower(AtomCoord[0][1]);
    				GeomOrb[j].Symb=g_strdup(AtomCoord[0]);
    				for(i=0;i<3;i++) GeomOrb[j].C[i]=atof(AtomCoord[i+1])*ANG_TO_BOHR;
				GeomOrb[j].Prop = prop_atom_get(GeomOrb[j].Symb);
				GeomOrb[j].partialCharge = 0.0;
				GeomOrb[j].nuclearCharge = get_atomic_number_from_symbol(GeomOrb[j].Symb);
			}
 			Ncenters = nn;
			OK = TRUE;
			break;
		}
		else
		{
			nn = atoi(t);
			if(!fgets(t, BSIZE,file)) break;
			for(i=0; i<nn; i++)
				if(!fgets(t, BSIZE,file))break;
			if(i!=nn) { OK = FALSE; break;}
		}
		if(!OK) break;
	}

	fclose(file);
 	g_free(t);
 	for(i=0;i<5;i++) g_free(AtomCoord[i]);
 	if(Ncenters == 0 ) { if(GeomOrb) g_free(GeomOrb); }
 	else DefineType();

	buildBondsOrb();
	reset_grid_limits();
	init_atomic_orbitals();
	set_status_label_info("Geometry","Ok");
	RebuildGeom = TRUE;
	if(this_is_a_new_geometry()) free_objects_all();
	glarea_rafresh(GLArea);
	return TRUE;
}
/********************************************************************************/
gboolean gl_read_xyz_file(gchar* FileName)
{
 gchar *t;
 gchar *tmp=NULL;
 gboolean OK;
 gchar *AtomCoord[5];
 FILE *fd;
 guint taille=BSIZE;
 guint i;
 gint j;
 gint l;
 gfloat scal;
 gint n;

 for(i=0;i<5;i++)
	AtomCoord[i]=g_malloc(taille*sizeof(char));

 
 if ((!FileName) || (strcmp(FileName,"") == 0))
 {
	Message("Sorry\n No selected file"," Error ",TRUE);
    	return FALSE ;
 }


 t=g_malloc(taille);
 fd = FOpen(FileName, "rb");
 OK=TRUE;
 if(fd!=NULL)
 {
  fgets(t,taille,fd);
  n = atoi(t);
  if(n < 1)
  {
	OK=FALSE;
	Message("Sorry\n this is not a XYZ file"," Error ",TRUE);
	goto end;
  }
  free_data_all();
  /* Ncenters = n;*/
  if(OK) GeomOrb=g_malloc(n*sizeof(TypeGeomOrb));
  tmp = get_name_file(FileName);
  set_status_label_info("File Name",tmp);
  g_free(tmp);
  set_status_label_info("File Type","XYZ");
  set_status_label_info("Geometry","Reading");

  Dipole.def = FALSE;
  progress_orb(0,GABEDIT_PROGORB_READGEOM,TRUE);
  scal = (gfloat)1.01/n;
  fgets(t,taille,fd);
  j=-1;
  while(!feof(fd) && OK && (j<(gint)n))
  {
	  
    j++;
    if(j>=(gint)n)
	break;
	progress_orb(scal,GABEDIT_PROGORB_READGEOM,FALSE);
    fgets(t,taille,fd);
    sscanf(t,"%s %s %s %s",AtomCoord[0],AtomCoord[1],AtomCoord[2],AtomCoord[3]);
	AtomCoord[0][0]=toupper(AtomCoord[0][0]);
	l=strlen(AtomCoord[0]);
    if (l==2)
	 	AtomCoord[0][1]=tolower(AtomCoord[0][1]);
    
    GeomOrb[j].Symb=g_strdup(AtomCoord[0]);

    GeomOrb[j].C[0] = atof(ang_to_bohr(AtomCoord[1]));
    GeomOrb[j].C[1] = atof(ang_to_bohr(AtomCoord[2]));
    GeomOrb[j].C[2] = atof(ang_to_bohr(AtomCoord[3]));
    GeomOrb[j].Prop = prop_atom_get(GeomOrb[j].Symb);
    GeomOrb[j].partialCharge = 0.0;
    GeomOrb[j].nuclearCharge = get_atomic_number_from_symbol(GeomOrb[j].Symb);
  }
  fclose(fd);

  Ncenters = n;
  if(Ncenters>10000)
  {
  	TypeGeom = GABEDIT_TYPEGEOM_WIREFRAME;
  	RebuildGeom = TRUE;
  }
  else
  {
  }
  
 }
end:
 g_free(t);
 for(i=0;i<5;i++)
	g_free(AtomCoord[i]);
 if(OK)
 {
	RebuildGeom = TRUE;
	if(this_is_a_new_geometry()) free_objects_all();
	buildBondsOrb();
	glarea_rafresh(GLArea);
	reset_grid_limits();
	init_atomic_orbitals();
	set_status_label_info("Geometry","Ok");
 }
 return OK;
}
/********************************************************************************/
void gl_read_xyz_file_sel(GabeditFileChooser *SelecFile, gint response_id)
{
	gchar *FileName;

	if(response_id != GTK_RESPONSE_OK) return;
 	FileName = gabedit_file_chooser_get_current_file(SelecFile);
	gtk_widget_hide(GTK_WIDGET(SelecFile));
	while( gtk_events_pending() )
		gtk_main_iteration();
 
	add_objects_for_new_grid();
 	gl_read_xyz_file(FileName);
}
/********************************************************************************/
static gchar* get_symb_atom(gchar* symb)
{
	gchar* S;
	gchar Forbidden[]={'0','1','2','3','4','5','6','7','8','9'};

	if(strlen(symb)==1)
		S = g_strdup(symb);
	else
	{
		gint Ok = 1;
		gint i;
		for(i=0;i<10;i++)
			if(symb[1]== Forbidden[i])
			{
				Ok =0;
				break;
			}
		if(Ok==1)
		{
			S = g_strdup_printf("%c%c",toupper(symb[0]),symb[1]);
		}
		else
			S = g_strdup_printf("%c",toupper(symb[0]));
	}
	if(!test_atom_define(S))
		S = g_strdup_printf("%c",toupper(symb[0]));


	return S;
}
/*************************************************************************************/
static gboolean read_atom_pdb_file(gchar* line,gchar* listFields[])
{
	gint i;
	gint k = 0;
	if(strlen(line)<54)
		return FALSE;

	/* 0 -> Atom Type */
	k = 0;
	for(i=0;i<MAXATOMTYPE;i++)
		listFields[k][i] = line[13+i-1];
	listFields[k][MAXATOMTYPE] = '\0';
	if(isdigit(listFields[k][0]))
	{
		gchar c0 = listFields[k][0];
		for(i=0;i<MAXATOMTYPE-1;i++)
			listFields[k][i] = listFields[k][i+1];
		listFields[k][MAXATOMTYPE-1] = c0;
	}

	/* 1-> Residue Name */
	k = 1;
	for(i=0;i<MAXRESIDUENAME;i++)
		listFields[k][i] = line[17+i-1];
	listFields[k][MAXRESIDUENAME] = '\0';

	/* 2-> Residue Number */
	k = 2;
	for(i=0;i<4;i++)
		listFields[k][i] = line[23+i-1];
	listFields[k][4] = '\0';
	/* 3-> x */
	k = 3;
	for(i=0;i<8;i++)
		listFields[k][i] = line[31+i-1];
	listFields[k][8] = '\0';

	/* 4-> y */
	k = 4;
	for(i=0;i<8;i++)
		listFields[k][i] = line[39+i-1];
	listFields[k][8] = '\0';

	/* 5-> z */
	k = 5;
	for(i=0;i<8;i++)
		listFields[k][i] = line[47+i-1];
	listFields[k][8] = '\0';

	/* 6-> Symbol */
	k = 6;
	if(strlen(line)>=78)
	{
		for(i=0;i<2;i++)
		{
			listFields[k][i] = line[76+i];
		}
		listFields[k][2] = '\0';
		if(listFields[k][1]==' ')
			listFields[k][1] = '\0';
		if(listFields[k][0]==' ')
			listFields[k][0] = '\0';
	}
	else
		listFields[k][0] = '\0';
	/* 7-> Charge */
	k = 7;
	if(strlen(line)>=80)
	{
		for(i=0;i<(gint)strlen(line)-79+1;i++)
			listFields[k][i] = line[79+i-1];

		listFields[k][strlen(line)-79+1] = '\0';

		if(listFields[k][strlen(line)-79]=='\n')
			listFields[k][strlen(line)-79]='\0';

	}
	else
		listFields[k][0] = '\0';

	for(i=0;i<8;i++)
	{
		delete_last_spaces(listFields[i]);
		delete_first_spaces(listFields[i]);
	}
	return TRUE;

}
/********************************************************************************/
gboolean gl_read_pdb_file(gchar* FileName)
{
	gchar *t;
	gchar *tmp=NULL;
	gboolean OK;
	FILE *fd;
	guint taille=BSIZE;
	guint i;
	gint j;
	gchar *listFields[8];

	if ((!FileName) || (strcmp(FileName,"") == 0))
	{
		Message("Sorry\n No selected file"," Error ",TRUE);
    		return FALSE ;
	}


	fd = FOpen(FileName, "rb");
	if(fd==NULL)
	{
		gchar* tmp = g_strdup_printf("Sorry\n I can not open %s file", FileName);
		Message(tmp," Error ",TRUE);
		g_free(tmp);
    		return FALSE ;
	}
	for(i=0;i<8;i++) listFields[i]=g_malloc(taille*sizeof(gchar));
	t=g_malloc(taille*sizeof(gchar));
	OK=TRUE;
	free_data_all();
	tmp = get_name_file(FileName);
	set_status_label_info("File Name",tmp);
	g_free(tmp);
	set_status_label_info("File Type","pdb");
	set_status_label_info("Geometry","Reading");
  	Dipole.def = FALSE;
	j=0;
	GeomOrb= NULL;
	while(!feof(fd))
	{
    		if(!fgets(t,taille,fd)) break;
    		sscanf(t,"%s",listFields[0]);
		if(strcmp(listFields[0],"HETATM")!=0 && strcmp(listFields[0],"ATOM")!=0) continue;
		if(!strcmp(t,"END")) break;
		if(!read_atom_pdb_file(t,listFields)) continue;
		/* 0 -> Atom Type  1-> Residue Name  2-> Residue Number 
		 * 3-> x  4-> y  5-> z  6-> Symbol 7-> Charge */

		GeomOrb=g_realloc(GeomOrb,(j+1)*sizeof(TypeGeomOrb));
		if(strlen(listFields[6])<=0) 
		{
			/* printf("PPPP listFields[6] = %s\n",listFields[6]);*/
			GeomOrb[j].Symb= get_symb_atom(listFields[0]);
		}
		else GeomOrb[j].Symb= g_strdup(listFields[6]);

    		GeomOrb[j].C[0] = atof(ang_to_bohr(listFields[3]));
    		GeomOrb[j].C[1] = atof(ang_to_bohr(listFields[4]));
    		GeomOrb[j].C[2] = atof(ang_to_bohr(listFields[5]));
    		GeomOrb[j].Prop = prop_atom_get(GeomOrb[j].Symb);
		GeomOrb[j].partialCharge = 0.0;
		GeomOrb[j].nuclearCharge = get_atomic_number_from_symbol(GeomOrb[j].Symb);
		j++;
	}
	if(j!=0)
	{
		Ncenters = j;
		readBondsPDB(fd);
	}
	fclose(fd);

	OK = FALSE;
	if(j!=0) OK = TRUE;

	if(OK)
	{
		Ncenters = j;
		if(Ncenters>1000)
		{
  			TypeGeom = GABEDIT_TYPEGEOM_WIREFRAME;
  			RebuildGeom = TRUE;
		}
	}
	else
	{
		Ncenters = 0;
		set_status_label_info("Geometry","Nothing");
	}
	g_free(t);
	for(i=0;i<8;i++) g_free(listFields[i]);
	if(OK)
	{
		RebuildGeom = TRUE;
		if(this_is_a_new_geometry()) free_objects_all();
		/* buildBondsOrb();*/
		glarea_rafresh(GLArea);
		reset_grid_limits();
		init_atomic_orbitals();
		set_status_label_info("Geometry","Ok");
	}
	return OK;
}
/********************************************************************************/
void gl_read_pdb_file_sel(GabeditFileChooser *SelecFile, gint response_id)
{
	gchar *FileName;

	if(response_id != GTK_RESPONSE_OK) return;
 	FileName = gabedit_file_chooser_get_current_file(SelecFile);
	gtk_widget_hide(GTK_WIDGET(SelecFile));
	while( gtk_events_pending() ) gtk_main_iteration();
 
	add_objects_for_new_grid();
 	gl_read_pdb_file(FileName);
}
/*************************************************************************************/
static void read_hin_numbers_of_atoms(FILE* file, int* natoms, int* nresidues)
{
	guint taille = BSIZE;
	gchar t[BSIZE];
	gchar dump[BSIZE];

	*natoms = 0;
	*nresidues = 0;


	fseek(file, 0L, SEEK_SET);
	while(!feof(file))
	{
    		if(!fgets(t,taille,file)) break;
    		sscanf(t,"%s",dump);
		g_strdup(dump);
		if(!strcmp(dump,"ATOM")) (*natoms)++;
		if(!strcmp(dump,"RES")) (*nresidues)++;
	}
}
/*************************************************************************************/
static gboolean read_atom_hin_file(FILE* file,gchar* listFields[])
{
	guint taille = BSIZE;
	gchar t[BSIZE];
	gchar dump[BSIZE];
	gint i;

    	if(!fgets(t,taille,file)) return FALSE;
    	sscanf(t,"%s",dump);
	g_strdup(dump);
	if(strcmp(dump,"ATOM")!=0)
	{
		if(strcmp(dump,"RES")==0)
		{
    			sscanf(t,"%s %s %s",dump,dump,listFields[1]);
			sprintf(listFields[0],"Unknown");
		}
		else
			return FALSE;
	}
	else
	{
		/* 0 -> Atom Type PDB Style*/
		/* 1 -> Atom Symbol*/
		/* 2 -> Atom Type Amber*/
		/* 3 -> Atom Charge*/
		/* 4 -> x*/
		/* 5 -> y*/
		/* 6 -> z*/
    		sscanf(t,"%s %s %s %s %s %s %s %s %s %s",dump,dump,listFields[0],listFields[1],listFields[2],dump,listFields[3],listFields[4],listFields[5],listFields[6]);
	}
	for(i=0;i<6;i++)
	{
		delete_last_spaces(listFields[i]);
		delete_first_spaces(listFields[i]);
	}
	return TRUE;

}
/********************************************************************************/
gboolean gl_read_hin_file(gchar* FileName)
{
	gchar *t;
	gchar *tmp=NULL;
	gboolean OK;
	FILE *fd;
	guint taille=BSIZE;
	guint i;
	gint j;
	gchar *listFields[8];
	int natoms = 0;
	int nresidues = 0;

	if ((!FileName) || (strcmp(FileName,"") == 0))
	{
		Message("Sorry\n No selected file"," Error ",TRUE);
    		return FALSE ;
	}

	fd = FOpen(FileName, "rb");

	if(fd==NULL)
	{
		gchar* tmp = g_strdup_printf("Sorry\n I can not open %s file", FileName);
		Message(tmp," Error ",TRUE);
		g_free(tmp);
    		return FALSE ;
	}
	read_hin_numbers_of_atoms(fd,&natoms, &nresidues);
	if(natoms<1)
	{
		gchar* tmp = g_strdup_printf("Sorry\n I can not read grometry from %s file", FileName);
		Message(tmp," Error ",TRUE);
		g_free(tmp);
    		return FALSE ;
	}

	OK=TRUE;
	for(i=0;i<8;i++) listFields[i]=g_malloc(taille*sizeof(gchar));
	t=g_malloc(taille*sizeof(gchar));

	free_data_all();
	tmp = get_name_file(FileName);
	set_status_label_info("File Name",tmp);
	g_free(tmp);
	set_status_label_info("File Type","Hyperchem");
	set_status_label_info("Geometry","Reading");
  	Dipole.def = FALSE;
	j=0;

	GeomOrb=g_malloc(natoms*sizeof(TypeGeomOrb));
	fseek(fd, 0L, SEEK_SET);
	j = 0;
	i = -1; /* number residu */
	while(!feof(fd))
	{
		if(!read_atom_hin_file(fd,listFields))continue;
		if(strcmp(listFields[0],"Unknown")==0)
		{
			i++;
			continue;
		}
		/* 0 -> Atom Type PDB Style*/
		/* 1 -> Atom Symbol*/
		/* 2 -> Atom Type Amber*/
		/* 3 -> Atom Charge*/
		/* 4 -> x*/
		/* 5 -> y*/
		/* 6 -> z*/
		/* printf("%s %s %s %s %s %s %s \n",listFields[0],listFields[1],listFields[2],listFields[3],listFields[4],listFields[5],listFields[6]);*/
		GeomOrb[j].Symb= g_strdup(listFields[1]);

    		GeomOrb[j].C[0] = atof(ang_to_bohr(listFields[4]));
    		GeomOrb[j].C[1] = atof(ang_to_bohr(listFields[5]));
    		GeomOrb[j].C[2] = atof(ang_to_bohr(listFields[6]));
    		GeomOrb[j].Prop = prop_atom_get(GeomOrb[j].Symb);
		GeomOrb[j].partialCharge = atof(listFields[3]);
		GeomOrb[j].nuclearCharge = get_atomic_number_from_symbol(GeomOrb[j].Symb);
		j++;
		if(j>=natoms)break;
	}
	OK = TRUE;
	if(natoms!=j) OK = FALSE;

	if(OK)
	{
		Ncenters = natoms;
		if(Ncenters>10000)
		{
  			TypeGeom = GABEDIT_TYPEGEOM_WIREFRAME;
  			RebuildGeom = TRUE;
		}
		readBondsHIN(fd);
	}
	else
	{
		Ncenters = 0;
		set_status_label_info("Geometry","Nothing");
	}
	fclose(fd);
	g_free(t);
	for(i=0;i<8;i++) g_free(listFields[i]);
	if(OK)
	{
		RebuildGeom = TRUE;
		if(this_is_a_new_geometry()) free_objects_all();
		/* buildBondsOrb();*/
		glarea_rafresh(GLArea);
		reset_grid_limits();
		init_atomic_orbitals();
		set_status_label_info("Geometry","Ok");
	}
	return OK;
}
/********************************************************************************/
void gl_read_hin_file_sel(GabeditFileChooser *SelecFile, gint response_id)
{
	gchar *FileName;

	if(response_id != GTK_RESPONSE_OK) return;
 	FileName = gabedit_file_chooser_get_current_file(SelecFile);
	gtk_widget_hide(GTK_WIDGET(SelecFile));
	while( gtk_events_pending() ) gtk_main_iteration();
 
	add_objects_for_new_grid();
 	gl_read_hin_file(FileName);
}
/*****************************************************************************************/
static void conversion_to_hin_and_read(GtkWidget *wid,gpointer data)
{
	gchar *fout =  g_strdup_printf("%s%stmp%sfout",gabedit_directory(),G_DIR_SEPARATOR_S,G_DIR_SEPARATOR_S);
	gchar *ferr =  g_strdup_printf("%s%stmp%sferr",gabedit_directory(),G_DIR_SEPARATOR_S,G_DIR_SEPARATOR_S);
	gchar *outputFileName =  g_strdup_printf("%s%stmp%sgeom.hin",gabedit_directory(),G_DIR_SEPARATOR_S,G_DIR_SEPARATOR_S);
	G_CONST_RETURN gchar* inputFileName;
	gchar inputFileType[100];
	gchar outputFileType[] ="hin" ;
	GtkWidget* entryFileName = NULL;
	GtkWidget* entryFileType = NULL;
	G_CONST_RETURN gchar* t0;
	gchar* t;
	gchar* strout;
	gchar* strerr;
	G_CONST_RETURN gchar* babelCommand =get_open_babel_command();

	GtkWidget *Win = (GtkWidget *)data;
        if(!this_is_an_object((GtkObject*)Win)) return;

	entryFileName = g_object_get_data (G_OBJECT (Win), "EntryFileName");
	entryFileType = g_object_get_data (G_OBJECT (Win), "EntryCombo");
	
        if(!this_is_an_object((GtkObject*)entryFileName)) return;
        if(!this_is_an_object((GtkObject*)entryFileType)) return;

	inputFileName = gtk_entry_get_text(GTK_ENTRY(entryFileName)); 

	t0 = gtk_entry_get_text(GTK_ENTRY(entryFileType)); 
	sscanf(t0,"%s",inputFileType);
	if(strstr(t0,"Automatic")) t = g_strdup_printf("%s %s %s", babelCommand, inputFileName, outputFileName);
	else t = g_strdup_printf("%s -i%s %s -o%s %s", babelCommand, inputFileType, inputFileName, outputFileType, outputFileName);

	unlink(fout);
	unlink(ferr);
	run_local_command(fout,ferr,t,FALSE);
	strout = cat_file(fout,FALSE);
	strerr = cat_file(ferr,FALSE);
  	if(strout)
	{
 		Message(strout, "Info", TRUE);
		g_free(strout);
	}
  	if(strerr)
	{
 		Message(strerr, "Error", TRUE);
		g_free(strerr);
	}
	else
	{
		gl_read_hin_file(outputFileName);
	}
	unlink(fout);
	unlink(ferr);
	unlink(outputFileName);
	g_free(fout);
	g_free(ferr);
	g_free(t);
	g_free(outputFileName);
}
/********************************************************************************/
void gl_create_babel_read_dialogue()
{
	GtkWidget* okButton = create_babel_read_save_dialogue(TRUE);
	GtkWidget* win = g_object_get_data (G_OBJECT (okButton), "Window");

	g_signal_connect(G_OBJECT(okButton), "clicked", G_CALLBACK(conversion_to_hin_and_read),win);
	g_signal_connect_swapped(G_OBJECT(okButton), "clicked",G_CALLBACK(delete_child),GTK_OBJECT(win));
}
/********************************************************************************/
gboolean gl_read_molden_or_gabedit_file_geom(gchar *FileName,gint type)
{
 	gchar *t;
	gchar *tmp;
 	gchar *AtomCoord[5];
 	FILE *fd;
 	guint taille=BSIZE;
 	guint idummy;
 	gint iz;
 	guint i;
 	gint j=0;
 	gint l;
	gboolean Angs = FALSE;
	gchar title[BSIZE];
	gchar titleCap[BSIZE];
	gchar tCap[BSIZE];
	gchar message[BSIZE];
	if(type == 0)
	{
		sprintf(title,"[Molden Format]");
		sprintf(message,"Sorry\nThis file is not a molden type file");
	}
	else
	{
		sprintf(title,"[Gabedit Format]");
		sprintf(message,"Sorry\nThis file is not a gabedit type file");
	}
	sprintf(titleCap,title);
	g_strup(titleCap);

 	for(i=0;i<5;i++)
		AtomCoord[i]=g_malloc(taille*sizeof(char));
  
 	if ((!FileName) || (strcmp(FileName,"") == 0))
 	{
		Message("Sorry\n No selected file","Error",TRUE);
 		for(i=0;i<5;i++)
			g_free(AtomCoord[i]);
    		return FALSE;
 	}

 	t=g_malloc(taille);
 	fd = FOpen(FileName, "rb");
 	if(fd ==NULL)
 	{
  		Message("Sorry\nI can not open this file","Error",TRUE);
 		g_free(t);
 		for(i=0;i<5;i++)
			g_free(AtomCoord[i]);
  		return FALSE;
 	}

	if(!feof(fd))
		fgets(t,taille,fd);

	sprintf(tCap,t);
	g_strup(tCap);

	if(!strstr(tCap,titleCap))
	{
  		Message(message,"Error",TRUE);
 		g_free(t);
 		for(i=0;i<5;i++)
			g_free(AtomCoord[i]);
  		return FALSE;
	}

	free_data_all();
	tmp = get_name_file(FileName);
	set_status_label_info("File Name",tmp);
	g_free(tmp);
	if(type == 0)
		set_status_label_info("File Type","Molden");
	else
		set_status_label_info("File Type","Gabedit");	
	set_status_label_info("Geometry","Reading");
	j = 0;
 	while(!feof(fd))
	{
		fgets(t,taille,fd);
		g_strup(t);
		if( !strstr( t, "[ATOMS" ) )
			continue;
		if( strstr( t, "ANGS" ))
			Angs = TRUE;
		else
			Angs = FALSE;

		j = 0;
		if(GeomOrb)
			g_free(GeomOrb);
		GeomOrb = NULL;
		while( !feof(fd) )
		{
			gint n = 0;
	  		fgets(t,taille,fd);
			n = sscanf(t,"%s %d %d %s %s %s",
				AtomCoord[0],&idummy,&iz,
				AtomCoord[1],AtomCoord[2],AtomCoord[3]);
			if(n<6) break;
    			if(GeomOrb==NULL) GeomOrb=g_malloc(sizeof(TypeGeomOrb));
    			else GeomOrb=g_realloc(GeomOrb,(j+1)*sizeof(TypeGeomOrb));

			AtomCoord[0][0]=toupper(AtomCoord[0][0]);
	 		l=strlen(AtomCoord[0]);
          		if (l==2) AtomCoord[0][1]=tolower(AtomCoord[0][1]);

    			GeomOrb[j].Symb=g_strdup(AtomCoord[0]);
			if(!test_atom_define(GeomOrb[j].Symb))
			{
				GeomOrb[j].Symb = get_symbol_using_z(iz);
			}

    			for(i=0;i<3;i++) GeomOrb[j].C[i]=atof((AtomCoord[i+1]));
			GeomOrb[j].Prop = prop_atom_get(GeomOrb[j].Symb);
			GeomOrb[j].partialCharge = 0.0;
			GeomOrb[j].nuclearCharge = get_atomic_number_from_symbol(GeomOrb[j].Symb);
			j++;
  		}
		break;
	}

 	Ncenters = j;
 	fclose(fd);
 	g_free(t);
 	for(i=0;i<5;i++)
		g_free(AtomCoord[i]);
	if(Angs)
	for(j=0;j<Ncenters;j++)
		for(i=0;i<3;i++)
			GeomOrb[j].C[i] *=ANG_TO_BOHR;

 	if(Ncenters == 0 )
	{
		if(GeomOrb) g_free(GeomOrb);
		set_status_label_info("Geometry","Notting");
		return FALSE;
	}
 	else
	{
  		DefineType();
  		/* PrintGeomOrb();*/
	}
	buildBondsOrb();
	reset_grid_limits();
	init_atomic_orbitals();
	set_status_label_info("Geometry","Ok");
	return TRUE;
}
/********************************************************************************/
gboolean gl_read_molden_file_geom(gchar *FileName)
{
	if( gl_read_molden_or_gabedit_file_geom(FileName,0))
	{
		RebuildGeom = TRUE;
		if(this_is_a_new_geometry()) free_objects_all();
		glarea_rafresh(GLArea);
		return TRUE;
	}
	return FALSE;
}
/********************************************************************************/
gboolean  gl_read_gabedit_file_geom(gchar *FileName)
{
	if(gl_read_molden_or_gabedit_file_geom(FileName,1))
	{
		RebuildGeom = TRUE;
		if(this_is_a_new_geometry()) free_objects_all();
		glarea_rafresh(GLArea);
		return TRUE;
	}
	return FALSE;
}
/********************************************************************************/
gboolean gl_read_molpro_file_geom_pos(gchar *FileName,long int pos)
{
 	gchar t[BSIZE];
 	gchar *tmp = NULL;
 	gboolean OK;
 	gchar *AtomCoord[5];
 	FILE *fd;
 	guint taille=BSIZE;
 	guint idummy;
 	guint i;
 	gint j=0;
 	gint l;
 	long int n=-1;
	gint kk;
	gchar dum[100];

 	for(i=0;i<5;i++)
		AtomCoord[i]=g_malloc(taille*sizeof(char));
  
 	fd = FOpen(FileName, "rb");
 	if(fd ==NULL)
 	{
  		Message("Sorry\nI can not open this file","Error",TRUE);
 		for(i=0;i<5;i++)
			g_free(AtomCoord[i]);
  		return FALSE;
 	}

	free_data_all();
	tmp = get_name_file(FileName);
	set_status_label_info("File Name",tmp);
	g_free(tmp);
	set_status_label_info("File Type","Molpro");
	set_status_label_info("Geometry","Reading");
 	OK=FALSE;
	n=-1;
 	while(!feof(fd))
	{
		n++;
		fgets(t,taille,fd);
		if (n==pos)
		{
  			fgets(t,taille,fd);
  			fgets(t,taille,fd);
  			fgets(t,taille,fd);
                	OK = TRUE;
	  		break;
	  	}
        }
 	if(!OK)
	{
  		Message("Sorry\nI can not read geometry in this file","Error",TRUE);
 		fclose(fd);
 		for(i=0;i<5;i++)
			g_free(AtomCoord[i]);
		set_status_label_info("File Name","Nothing");
		set_status_label_info("File Type","Nothing");
		set_status_label_info("Geometry","Nothing");
		return FALSE;
    	}

	j=-1;
  	while(!feof(fd) )
  	{
		fgets(t,taille,fd);
		if ( !strcmp(t,"\n"))
				break;
    		j++;
    		if(GeomOrb==NULL)
			GeomOrb=g_malloc(sizeof(TypeGeomOrb));
    		else
    			GeomOrb=g_realloc(GeomOrb,(j+1)*sizeof(TypeGeomOrb));

		kk = sscanf(t,"%d %s %s %s %s %s %s",&idummy, AtomCoord[0],AtomCoord[1],AtomCoord[1], AtomCoord[2],AtomCoord[3], dum);
		if(kk==7) sscanf(t,"%d %s %s %s %s %s %s",&idummy, AtomCoord[0],AtomCoord[1],dum, AtomCoord[1], AtomCoord[2],AtomCoord[3]);

		for(i=0;i<(gint)strlen(AtomCoord[0]);i++) if(isdigit(AtomCoord[0][i])) AtomCoord[0][i] = ' ';
		delete_all_spaces(AtomCoord[0]);

		AtomCoord[0][0]=toupper(AtomCoord[0][0]);
 		l=strlen(AtomCoord[0]);
          	if (l==2)
 			AtomCoord[0][1]=tolower(AtomCoord[0][1]);

    		GeomOrb[j].Symb=g_strdup(AtomCoord[0]);
    		for(i=0;i<3;i++)
			GeomOrb[j].C[i]=atof((AtomCoord[i+1]));

		GeomOrb[j].Prop = prop_atom_get(GeomOrb[j].Symb);
		GeomOrb[j].partialCharge = 0.0;
		GeomOrb[j].nuclearCharge = get_atomic_number_from_symbol(GeomOrb[j].Symb);
  	}

 	Ncenters = j+1;
 	fclose(fd);
 	for(i=0;i<5;i++)
		g_free(AtomCoord[i]);
 	if(Ncenters == 0 )
  		g_free(GeomOrb);
 	else
	{
  		DefineType();
  		/* PrintGeomOrb();*/
	}
	RebuildGeom = TRUE;
	buildBondsOrb();
	reset_grid_limits();
	init_atomic_orbitals();
	set_status_label_info("Geometry","Ok");
	return TRUE;
}
/********************************************************************************/
gboolean gl_read_dalton_file_geomi(gchar *FileName,gint num)
{
 	gchar *t;
 	gchar *tmp = NULL;
 	gboolean OK;
 	gchar *AtomCoord[5];
 	FILE *fd;
 	guint taille=BSIZE;
 	guint i;
 	gint j=0;
 	gint l;
 	guint numgeom;
	gchar dum[100];
	gint kk;
	long int geompos = 0;

 	for(i=0;i<5;i++) AtomCoord[i]=g_malloc(taille*sizeof(char));
  
 	if ((!FileName) || (strcmp(FileName,"") == 0))
 	{
		Message("Sorry\n No file slected","Error",TRUE);
 		for(i=0;i<5;i++) g_free(AtomCoord[i]);
    		return FALSE ;
 	}

 	t=g_malloc(taille);
 	fd = FOpen(FileName, "rb");
 	if(fd ==NULL)
 	{
  		Message("Sorry\nI can not open this file","Error",TRUE);
 		g_free(t);
 		for(i=0;i<5;i++) g_free(AtomCoord[i]);
  		return FALSE;
 	}

	free_data_all();
	tmp = get_name_file(FileName);
	set_status_label_info("File Name",tmp);
	g_free(tmp);
	set_status_label_info("File Type","Dalton");
 	numgeom =1;
 	do 
 	{
		set_status_label_info("Geometry","Reading");
 		OK=FALSE;
 		while(!feof(fd))
		{
			fgets(t,taille,fd);
			if ( strstr(t,"geometry (au)"))
			{
	  			fgets(t,taille,fd);
	  			fgets(t,taille,fd);
 				numgeom++;
                		OK = TRUE;
	  			break;
	  		}
        	}
 		if(!OK && (numgeom == 1) )
		{
  			Message("Sorry\nI can not read geometry from this file","Error",TRUE);
 			fclose(fd);
 			g_free(t);
 			for(i=0;i<5;i++) g_free(AtomCoord[i]);
			set_status_label_info("File Name","Nothing");
			set_status_label_info("File Type","Nothing");
			set_status_label_info("Geometry","Nothing");
			return FALSE;
    		}
 		if(!OK)break;

  		j=-1;
  		while(!feof(fd) )
  		{
			fgets(t,taille,fd);
			if ( !strcmp(t,"\n"))
			{
				geompos = ftell(fd);
 				get_dipole_from_dalton_output_file(fd);
				fseek(fd, geompos, SEEK_SET);
				break;
			}
    			j++;
    			if(GeomOrb==NULL) GeomOrb=g_malloc(sizeof(TypeGeomOrb));
    			else GeomOrb=g_realloc(GeomOrb,(j+1)*sizeof(TypeGeomOrb));

			kk = sscanf(t,"%s %s %s %s %s",AtomCoord[0],AtomCoord[1], AtomCoord[2],AtomCoord[3], dum);
			if(kk==5) sscanf(t,"%s %s %s %s %s",AtomCoord[0],dum, AtomCoord[1], AtomCoord[2],AtomCoord[3]);

			for(i=0;i<(gint)strlen(AtomCoord[0]);i++) if(isdigit(AtomCoord[0][i])) AtomCoord[0][i] = ' ';
			delete_all_spaces(AtomCoord[0]);
			AtomCoord[0][0]=toupper(AtomCoord[0][0]);
	 		l=strlen(AtomCoord[0]);
          		if (l==2) AtomCoord[0][1]=tolower(AtomCoord[0][1]);
			/*
			printf("j = %d symb = %s %s %s %s \n",j, AtomCoord[0], AtomCoord[1], AtomCoord[2],AtomCoord[3]);
			*/

    			GeomOrb[j].Symb=g_strdup(AtomCoord[0]);
    			for(i=0;i<3;i++) GeomOrb[j].C[i]=atof((AtomCoord[i+1]));

			GeomOrb[j].Prop = prop_atom_get(GeomOrb[j].Symb);
			GeomOrb[j].partialCharge = 0.0;
			GeomOrb[j].nuclearCharge = get_atomic_number_from_symbol(GeomOrb[j].Symb);
  		}
		if(num >0 && (gint)numgeom-1 == num) break;
			
 	}while(!feof(fd));

 	Ncenters = j+1;
 	fclose(fd);
 	g_free(t);
 	for(i=0;i<5;i++) g_free(AtomCoord[i]);
 	if(Ncenters == 0 ) g_free(GeomOrb);
 	else
	{
  		DefineType();
  		/* PrintGeomOrb();*/
	}
	RebuildGeom = TRUE;
	buildBondsOrb();
	reset_grid_limits();
	init_atomic_orbitals();
	set_status_label_info("Geometry","Ok");
	return TRUE;
}
/********************************************************************************/
void gl_get_charges_from_gaussian_output_file(FILE* fd,gint N)
{
 	guint taille=BSIZE;
  	gchar t[BSIZE];
  	gchar dump[BSIZE];
  	gchar d[BSIZE];
  	gchar* pdest;
	gint i;
	gint ngrad=0;


  	while(!feof(fd) )
	{
    		pdest = NULL;
    		fgets(t,taille,fd);
    		pdest = strstr( t, "Total atomic charges");
		if(!pdest) /* Gaussian 03 */
    			pdest = strstr( t, "atomic charges");

		if(pdest)
		{
    			if(!feof(fd))
				fgets(t,taille,fd);
			else
				break;

			for(i=0;i<N;i++)
			{
    				if(!feof(fd))
					fgets(t,taille,fd);
				else
					break;
				if(sscanf(t,"%s %s %s",dump,dump,d)==3)
				{
					GeomOrb[i].partialCharge = atof(d);
				}
			}
			break;
		}
		else
		{
          		pdest = strstr( t, "GradGradGrad" );
			if(pdest)
			{
				ngrad++;
			}
			if(ngrad>2)
				break;
		}

	}
}
/********************************************************************************/
void gl_get_natural_charges_from_gaussian_output_file(FILE* fd,gint N)
{
 	guint taille=BSIZE;
  	gchar t[BSIZE];
  	gchar dump[BSIZE];
  	gchar d[BSIZE];
  	gchar* pdest;
	gint i;
	gint ngrad =0;



  	while(!feof(fd) )
	{
    		pdest = NULL;
    		fgets(t,taille,fd);
    		pdest = strstr( t, "Summary of Natural Population Analysis:");
		if(!pdest) /* Gaussian 03 */
    			pdest = strstr( t, "Summary of Natural Population Analysis:");

		if(pdest)
		{
    			if(!feof(fd)) fgets(t,taille,fd);
			else break;
    			if(!feof(fd)) fgets(t,taille,fd);
			else break;
			if(!strstr(t,"Natural Population"))break;
    			if(!feof(fd)) fgets(t,taille,fd);
			else break;
			if(!strstr(t,"Natural"))break;
    			if(!feof(fd)) fgets(t,taille,fd);
			else break;
			if(!strstr(t,"Charge"))break;
    			if(!feof(fd)) fgets(t,taille,fd);
			else break;
			if(!strstr(t,"-------------"))break;

			for(i=0;i<N;i++)
			{
    				if(!feof(fd)) fgets(t,taille,fd);
				else break;
				if(sscanf(t,"%s %s %s",dump,dump,d)==3)
				{
					GeomOrb[i].partialCharge = atof(d);
				}
			}
			break;
		}
		else
		{
          		pdest = strstr( t, "GradGradGrad" );
			if(pdest)
			{
				ngrad++;
			}
			if(ngrad>2)
				break;
		}

	}
}
/********************************************************************************/
void gl_get_esp_charges_from_gaussian_output_file(FILE* fd,gint N)
{
 	guint taille=BSIZE;
  	gchar t[BSIZE];
  	gchar dump[BSIZE];
  	gchar d[BSIZE];
  	gchar* pdest;
	gint i;
	gint ngrad=0;


  	while(!feof(fd) )
	{
    		pdest = NULL;
    		fgets(t,taille,fd);
    		pdest = strstr( t, "Charges from ESP fit");
		if(!pdest) /* Gaussian 03 */
    			pdest = strstr( t, "harges from ESP");

		if(pdest)
		{
    			if(!feof(fd)) fgets(t,taille,fd);
			else break;
    			if(!feof(fd)) fgets(t,taille,fd);
			else break;

			for(i=0;i<N;i++)
			{
    				if(!feof(fd)) fgets(t,taille,fd);
				else break;
				if(sscanf(t,"%s %s %s",dump,dump,d)==3)
				{
					GeomOrb[i].partialCharge = atof(d);
				}
			}
			break;
		}
		else
		{
          		pdest = strstr( t, "GradGradGrad" );
			if(pdest)
			{
				ngrad++;
			}
			if(ngrad>2)
				break;
		}

	}
}
/********************************************************************************/
void gl_get_charges_from_gamess_output_file(FILE* fd,gint N)
{
 	guint taille=BSIZE;
  	gchar t[BSIZE];
  	gchar dump[BSIZE];
  	gchar d[BSIZE];
  	gchar* pdest;
	gint i;


  	while(!feof(fd) )
	{
    		pdest = NULL;
    		if(!fgets(t,taille,fd)) break;
    		pdest = strstr( t, "TOTAL MULLIKEN AND LOWDIN ATOMIC POPULATIONS");

		if(pdest)
		{
    			if(!fgets(t,taille,fd)) break;
			for(i=0;i<N;i++)
			{
    				if(!fgets(t,taille,fd)) break;
				if(sscanf(t,"%s %s %s %s %s %s",dump, dump ,dump, dump, dump, d)==6)
				{
					GeomOrb[i].partialCharge = atof(d);
				}
				else break;
			}
			break;
		}
	}
}
/********************************************************************************/
gboolean gl_read_gamess_file_geomi(gchar *FileName,gint num)
{
 	gchar *t;
 	gchar *tmp = NULL;
 	gboolean OK;
 	gchar *AtomCoord[5];
 	FILE *fd;
 	guint taille=BSIZE;
 	guint i;
 	gint j=0;
 	gint l;
 	guint numgeom;
	gchar dum[100];
	gint uni=1;
	long geompos = 0;

 	for(i=0;i<5;i++) AtomCoord[i]=g_malloc(taille*sizeof(char));
  
 	if ((!FileName) || (strcmp(FileName,"") == 0))
 	{
		Message("Sorry\n No file slected","Error",TRUE);
 		for(i=0;i<5;i++) g_free(AtomCoord[i]);
    		return FALSE ;
 	}

 	t=g_malloc(taille);

#ifdef G_OS_WIN32 
 	fd = FOpen(FileName, "rb");
#else
	fd = FOpen(FileName, "rb");
#endif

 	if(fd ==NULL)
 	{
  		Message("Sorry\nI can not open this file","Error",TRUE);
 		g_free(t);
 		for(i=0;i<5;i++) g_free(AtomCoord[i]);
  		return FALSE;
 	}

  	Dipole.def = FALSE;
	free_data_all();
	tmp = get_name_file(FileName);
	set_status_label_info("File Name",tmp);
	g_free(tmp);
	set_status_label_info("File Type","Gamess");
 	numgeom =1;
 	do 
 	{
		set_status_label_info("Geometry","Reading");
 		OK=FALSE;
 		while(!feof(fd))
		{
			if(!fgets(t,taille,fd))break;
			if ( strstr(t,"COORDINATES (BOHR)"))
			{
	  			fgets(t,taille,fd);
 				numgeom++;
                		OK = TRUE;
				uni = 0;
	  			break;
	  		}
			if ( strstr(t,"COORDINATES OF ALL ATOMS ARE (ANGS)"))
			{
	  			fgets(t,taille,fd);
	  			fgets(t,taille,fd);
 				numgeom++;
				uni=1;
				if((gint)numgeom-1 == num ) { OK = TRUE; break; }
				if(num<0 ) { OK = TRUE; break; }
	  		}
        	}
 		if(!OK && (numgeom == 1) )
		{
  			Message("Sorry\nI can not read geometry from this file","Error",TRUE);
 			fclose(fd);
 			g_free(t);
 			for(i=0;i<5;i++) g_free(AtomCoord[i]);
			set_status_label_info("File Name","Nothing");
			set_status_label_info("File Type","Nothing");
			set_status_label_info("Geometry","Nothing");
			return FALSE;
    		}
 		if(!OK)break;

  		j=-1;
  		while(!feof(fd) )
  		{
			fgets(t,taille,fd);
			if ( !strcmp(t,"\n")) break;
			if ( !strcmp(t,"\r\n")) break;
    			j++;
    			if(GeomOrb==NULL) GeomOrb=g_malloc(sizeof(TypeGeomOrb));
    			else GeomOrb=g_realloc(GeomOrb,(j+1)*sizeof(TypeGeomOrb));

			sscanf(t,"%s %s %s %s %s",AtomCoord[0],dum, AtomCoord[1], AtomCoord[2],AtomCoord[3]);

			for(i=0;i<(gint)strlen(AtomCoord[0]);i++) if(isdigit(AtomCoord[0][i])) AtomCoord[0][i] = ' ';
			delete_all_spaces(AtomCoord[0]);
			AtomCoord[0][0]=toupper(AtomCoord[0][0]);
	 		l=strlen(AtomCoord[0]);
          		if (l==2) AtomCoord[0][1]=tolower(AtomCoord[0][1]);
			/*
			printf("j = %d symb = %s %s %s %s \n",j, AtomCoord[0], AtomCoord[1], AtomCoord[2],AtomCoord[3]);
			*/

    			/* GeomOrb[j].Symb=g_strdup(AtomCoord[0]);*/
			GeomOrb[j].Symb=get_symbol_using_z(atoi(dum));
    			for(i=0;i<3;i++) 
			{
				GeomOrb[j].C[i]=atof((AtomCoord[i+1]));
				if(uni==1)GeomOrb[j].C[i] *= ANG_TO_BOHR;
			}

			GeomOrb[j].Prop = prop_atom_get(GeomOrb[j].Symb);
			GeomOrb[j].partialCharge = 0.0;
			GeomOrb[j].nuclearCharge = get_atomic_number_from_symbol(GeomOrb[j].Symb);
  		}
		if(num >0 && (gint)numgeom-1 == num) break;
		if(num<0) geompos = ftell(fd);
 	}while(!feof(fd));

 	Ncenters = j+1;

	if ( Ncenters >0)
	{
		long int gp = 0;
		if(num<0) fseek(fd, geompos, SEEK_SET);
		gp = ftell(fd);
		get_dipole_from_gamess_output_file(fd);
		fseek(fd, gp, SEEK_SET);
		gl_get_charges_from_gamess_output_file(fd,Ncenters);
		/* PrintGeomOrb();*/
	}

 	fclose(fd);
 	g_free(t);
 	for(i=0;i<5;i++) g_free(AtomCoord[i]);
 	if(Ncenters == 0 ) g_free(GeomOrb);
 	else
	{
  		DefineType();
  		/* PrintGeomOrb();*/
	}
	RebuildGeom = TRUE;
	buildBondsOrb();
	reset_grid_limits();
	init_atomic_orbitals();
	set_status_label_info("Geometry","Ok");
	return TRUE;
}
/********************************************************************************/
void gl_get_charges_from_mpqc_output_file(FILE* fd,gint N)
{
 	guint taille=BSIZE;
  	gchar t[BSIZE];
  	gchar dump[BSIZE];
  	gchar d[BSIZE];
	gint i;
	gboolean OK = FALSE;


	fseek(fd, 0L, SEEK_SET);
	OK = FALSE;
  	while(!feof(fd) )
	{
    		if(!fgets(t,taille,fd))break;
    		if(strstr(t, "Natural Population Analysis"))
		{
			OK = TRUE;
			break;
		}
	}
	if(!OK) return;

	OK = FALSE;
  	while(!feof(fd) )
	{
    		if(!fgets(t,taille,fd))break;
    		if(strstr(t, "atom") && strstr(t, "charge"))
		{
			OK = TRUE;
			break;
		}
	}
	if(!OK) return;
	for(i=0;i<N && !feof(fd) ;i++)
	{
		if(!fgets(t,taille,fd)) break;
		if(sscanf(t,"%s %s %s",dump,dump,d)==3)
		{
			GeomOrb[i].partialCharge = atof(d);
		}
	}
	if(i==N) return;
	for(i=0;i<N;i++)
	{
		GeomOrb[i].partialCharge = 0.0;
	}

	return;
}
/********************************************************************************/
gboolean gl_read_mpqc_file_geomi(gchar *fileName,gint numGeometry)
{
 	gchar *t;
 	gchar *tmp = NULL;
 	gboolean OK;
 	gchar *AtomCoord[5];
 	FILE *fd;
 	guint taille=BSIZE;
 	guint idummy;
 	guint i;
 	gint j=0;
 	gint l;
 	guint numGeom;
	gdouble tmpReal;

 	for(i=0;i<5;i++) AtomCoord[i]=g_malloc(taille*sizeof(char));
  
 	fd = FOpen(fileName, "rb");

	free_data_all();
	tmp = get_name_file(fileName);
	set_status_label_info("File Name",tmp);
	g_free(tmp);
	set_status_label_info("File Type","MPQC");

	t=g_malloc(taille);

	numGeom = 0;
	do 
	{
		gboolean unitOfOutAng = FALSE;
		OK=FALSE;
		set_status_label_info("Geometry","Reading");
		while(!feof(fd))
		{
			if(!fgets(t,taille,fd)) break;
			if (strstr(t,"<Molecule>"))
			{
				gboolean OkUnit = FALSE;
				do{
		 			if(!fgets(t,taille,fd))break;
					if(strstr(t,"unit"))
					{
						OkUnit = TRUE;
						if(strstr(t,"angstrom"))unitOfOutAng=TRUE;
						break;
					};
					
				}while(!feof(fd));
				if(!OkUnit) break;
				numGeom++;
	       			if((gint)numGeom == numGeometry )
				{
					OK = TRUE;
		 			break;
				}
	       			if(numGeometry<0 )
				{
					OK = TRUE;
		 			break;
				}
			}
	       }
		if(!OK && (numGeom == 0) )
		{
	 		g_free(t);
	 		t = g_strdup_printf("Sorry\nI can read Geometry from %s  file ",fileName);
	 		Message(t," Error ",TRUE);
	 		g_free(t);
			for(i=0;i<5;i++) g_free(AtomCoord[i]);
	 		return FALSE;
		}
		if(!OK) break;
		OK = FALSE;
		while(!feof(fd) )
		{
	   		if(!fgets(t,taille,fd))break;
			if ( !(strstr(t,"atoms") && strstr(t,"geometry"))) continue;
			OK = TRUE;
			break;
		}
		if(!OK)
		{
	 		g_free(t);
	 		t = g_strdup_printf("Sorry\nI can read Geometry from %s  file ",fileName);
	 		Message(t," Error ",TRUE);
	 		g_free(t);
			for(i=0;i<5;i++) g_free(AtomCoord[i]);
	 		return FALSE;
		}

		j=-1;
		while(!feof(fd) )
		{
	   		if(!fgets(t,taille,fd))break;
	   		if(strstr(t,"}"))break;
			j++;

    			if(GeomOrb==NULL) GeomOrb=g_malloc(sizeof(TypeGeomOrb));
    			else GeomOrb=g_realloc(GeomOrb,(j+1)*sizeof(TypeGeomOrb));

			for(i=0;i<strlen(t);i++) if(t[i]=='[' || t[i] ==']') t[i]=' ';
			sscanf(t,"%d %s %s %s %s",&idummy,AtomCoord[0],AtomCoord[1],AtomCoord[2],AtomCoord[3]);
			for(i=1;i<=3;i++)
			{
				tmpReal = atof(AtomCoord[i]);
				sprintf(AtomCoord[i],"%f",tmpReal);
			}

			AtomCoord[0][0]=toupper(AtomCoord[0][0]);

			l=strlen(AtomCoord[0]);

			if (l==2) AtomCoord[0][1]=tolower(AtomCoord[0][1]);

    			GeomOrb[j].Symb=g_strdup(AtomCoord[0]);
    			for(i=0;i<3;i++)
				if(unitOfOutAng)
    					GeomOrb[j].C[i]=atof(ang_to_bohr(AtomCoord[i+1]));
				else
    					GeomOrb[j].C[i]=atof(AtomCoord[i+1]);

			GeomOrb[j].Prop = prop_atom_get(GeomOrb[j].Symb);
			GeomOrb[j].partialCharge = 0.0;
			GeomOrb[j].nuclearCharge = get_atomic_number_from_symbol(GeomOrb[j].Symb);
		}

		if(numGeometry<0) continue;
		if(OK) break;
	}while(!feof(fd));

 	Ncenters = j+1;
	if(numGeometry<0) gl_get_charges_from_mpqc_output_file(fd,Ncenters);
 	fclose(fd);
 	g_free(t);
 	for(i=0;i<5;i++) g_free(AtomCoord[i]);
 	if(Ncenters == 0 ) g_free(GeomOrb);
 	else DefineType();

	buildBondsOrb();
	reset_grid_limits();
	init_atomic_orbitals();
	set_status_label_info("Geometry","Ok");
	RebuildGeom = TRUE;
	return TRUE;
}
/********************************************************************************/
gboolean gl_read_molcas_file_geomi(gchar *FileName,gint num)
{
 	gchar *t;
 	gchar *tmp = NULL;
 	gboolean OK;
 	gchar *AtomCoord[5];
 	FILE *file;
 	guint taille=BSIZE;
 	guint idummy;
 	guint i;
 	gint j=0;
 	gint l;
 	guint numgeom;

  
 	if ((!FileName) || (strcmp(FileName,"") == 0))
 	{
		Message("Sorry\n No file slected","Error",TRUE);
    		return FALSE ;
 	}


 	t=g_malloc(taille);
 	file = FOpen(FileName, "rb");
 	if(file ==NULL)
 	{
  		Message("Sorry\nI can not open this file","Error",TRUE);
 		g_free(t);
  		return FALSE;
 	}

 	for(i=0;i<5;i++) AtomCoord[i]=g_malloc(taille*sizeof(char));

	free_data_all();
	tmp = get_name_file(FileName);
	set_status_label_info("File Name",tmp);
	g_free(tmp);
	set_status_label_info("File Type","Molcas");
 	numgeom =1;
 	do 
 	{
		set_status_label_info("Geometry","Reading");
 		OK=FALSE;
 		while(!feof(file))
		{
			fgets(t,BSIZE,file);
			if (strstr(t," Cartesian coordinates:"))
			{
	  			fgets(t,BSIZE,file);
	  			fgets(t,BSIZE,file);
	  			fgets(t,BSIZE,file);
				numgeom++;
                		OK = TRUE;
	  			break;
	 		}
        	}
 		if(!OK && (numgeom == 1) )
		{
  			Message("Sorry\nI can not read geometry in this file","Error",TRUE);
 			fclose(file);
 			g_free(t);
 			for(i=0;i<5;i++) g_free(AtomCoord[i]);
			set_status_label_info("File Name","Nothing");
			set_status_label_info("File Type","Nothing");
			set_status_label_info("Geometry","Nothing");
			return FALSE;
    		}
 		if(!OK)break;

  		j=-1;
  		while(!feof(file) )
  		{
			fgets(t,taille,file);
			if ( !strcmp(t,"\n"))
			{
				/*
				long int geompos = 0;
				geompos = ftell(fd);
 				get_dipole_from_molcas_output_file(fd);
				fseek(fd, geompos, SEEK_SET);
				*/
				break;
			}
    			j++;
    			if(GeomOrb==NULL) GeomOrb=g_malloc(sizeof(TypeGeomOrb));
    			else GeomOrb=g_realloc(GeomOrb,(j+1)*sizeof(TypeGeomOrb));

			sscanf(t,"%d %s %s %s %s",&idummy,AtomCoord[0],AtomCoord[1], AtomCoord[2],AtomCoord[3]);

			for(i=0;i<(gint)strlen(AtomCoord[0]);i++) if(isdigit(AtomCoord[0][i])) AtomCoord[0][i] = ' ';
			delete_all_spaces(AtomCoord[0]);

			AtomCoord[0][0]=toupper(AtomCoord[0][0]);
	 		l=strlen(AtomCoord[0]); if (l==2) AtomCoord[0][1]=tolower(AtomCoord[0][1]);

    			GeomOrb[j].Symb=g_strdup(AtomCoord[0]);
    			for(i=0;i<3;i++) GeomOrb[j].C[i]=atof((AtomCoord[i+1]));
			GeomOrb[j].Prop = prop_atom_get(GeomOrb[j].Symb);
			GeomOrb[j].partialCharge = 0.0;
			GeomOrb[j].nuclearCharge = get_atomic_number_from_symbol(GeomOrb[j].Symb);
  		}
		if(num >0 && (gint)numgeom-1 == num) break;
			
 	}while(!feof(file));

 	Ncenters = j+1;
 	fclose(file);
 	g_free(t);
 	for(i=0;i<5;i++) g_free(AtomCoord[i]);
 	if(Ncenters == 0 ) g_free(GeomOrb);
 	else
	{
  		DefineType();
  		/* PrintGeomOrb();*/
	}
	RebuildGeom = TRUE;
	buildBondsOrb();
	reset_grid_limits();
	init_atomic_orbitals();
	set_status_label_info("Geometry","Ok");
	return TRUE;
}
/********************************************************************************/
gboolean gl_read_molpro_file_geomi(gchar *FileName,gint num)
{
 	gchar *t;
 	gchar *tmp = NULL;
 	gboolean OK;
 	gchar *AtomCoord[5];
 	FILE *fd;
 	guint taille=BSIZE;
 	guint idummy;
 	guint i;
 	gint j=0;
 	gint l;
 	guint numgeom;
	gchar dum[100];
	gint kk;

 	for(i=0;i<5;i++)
		AtomCoord[i]=g_malloc(taille*sizeof(char));
  
 	if ((!FileName) || (strcmp(FileName,"") == 0))
 	{
		Message("Sorry\n No file slected","Error",TRUE);
 		for(i=0;i<5;i++)
			g_free(AtomCoord[i]);
    		return FALSE ;
 	}

 	t=g_malloc(taille);
 	fd = FOpen(FileName, "rb");
 	if(fd ==NULL)
 	{
  		Message("Sorry\nI can not open this file","Error",TRUE);
 		g_free(t);
 		for(i=0;i<5;i++)
			g_free(AtomCoord[i]);
  		return FALSE;
 	}

	free_data_all();
	tmp = get_name_file(FileName);
	set_status_label_info("File Name",tmp);
	g_free(tmp);
	set_status_label_info("File Type","Molpro");
 	numgeom =1;
 	do 
 	{
		set_status_label_info("Geometry","Reading");
 		OK=FALSE;
 		while(!feof(fd))
		{
			fgets(t,taille,fd);
			if ( !strcmp(t," ATOMIC COORDINATES\n"))
			{
	  			fgets(t,taille,fd);
	  			fgets(t,taille,fd);
	  			fgets(t,taille,fd);
 				numgeom++;
                		OK = TRUE;
	  			break;
	  		}
        	}
 		if(!OK && (numgeom == 1) )
		{
  			Message("Sorry\nI can not read geometry in this file","Error",TRUE);
 			fclose(fd);
 			g_free(t);
 			for(i=0;i<5;i++)
				g_free(AtomCoord[i]);
			set_status_label_info("File Name","Nothing");
			set_status_label_info("File Type","Nothing");
			set_status_label_info("Geometry","Nothing");
			return FALSE;
    	}
 		if(!OK)break;

  		j=-1;
  		while(!feof(fd) )
  		{
			fgets(t,taille,fd);
			if ( !strcmp(t,"\n"))
			{
				long int geompos = 0;
				geompos = ftell(fd);
 				get_dipole_from_molpro_output_file(fd);
				fseek(fd, geompos, SEEK_SET);
				break;
			}
    			j++;
    			if(GeomOrb==NULL) GeomOrb=g_malloc(sizeof(TypeGeomOrb));
    			else GeomOrb=g_realloc(GeomOrb,(j+1)*sizeof(TypeGeomOrb));

			kk = sscanf(t,"%d %s %s %s %s %s %s",&idummy, AtomCoord[0],AtomCoord[1],AtomCoord[1], AtomCoord[2],AtomCoord[3], dum);
			if(kk==7) sscanf(t,"%d %s %s %s %s %s %s",&idummy, AtomCoord[0],AtomCoord[1],dum, AtomCoord[1], AtomCoord[2],AtomCoord[3]);

			for(i=0;i<(gint)strlen(AtomCoord[0]);i++) if(isdigit(AtomCoord[0][i])) AtomCoord[0][i] = ' ';
			delete_all_spaces(AtomCoord[0]);
			AtomCoord[0][0]=toupper(AtomCoord[0][0]);
	 		l=strlen(AtomCoord[0]);
          		if (l==2) AtomCoord[0][1]=tolower(AtomCoord[0][1]);

    			GeomOrb[j].Symb=g_strdup(AtomCoord[0]);
    			for(i=0;i<3;i++) GeomOrb[j].C[i]=atof((AtomCoord[i+1]));

			GeomOrb[j].Prop = prop_atom_get(GeomOrb[j].Symb);
			GeomOrb[j].partialCharge = 0.0;
			GeomOrb[j].nuclearCharge = get_atomic_number_from_symbol(GeomOrb[j].Symb);
  		}
		if(num >0 && (gint)numgeom-1 == num) break;
			
 	}while(!feof(fd));

 	Ncenters = j+1;
 	fclose(fd);
 	g_free(t);
 	for(i=0;i<5;i++)
		g_free(AtomCoord[i]);
 	if(Ncenters == 0 )
  		g_free(GeomOrb);
 	else
	{
  		DefineType();
  		/* PrintGeomOrb();*/
	}
	RebuildGeom = TRUE;
	buildBondsOrb();
	reset_grid_limits();
	init_atomic_orbitals();
	set_status_label_info("Geometry","Ok");
	return TRUE;
}

/********************************************************************************/
gboolean gl_read_gaussn_file_geomi_str(gchar *FileName,gint num,gchar* str)
{
 	gchar *t;
 	gchar *tmp = NULL;
 	gboolean OK;
 	gchar *AtomCoord[5];
 	FILE *fd;
 	guint taille=BSIZE;
 	guint idummy;
 	guint i;
 	gint j=0;
 	gint l;
 	guint numgeom;
 	gchar *pdest;
 	gint result;
 	guint itype=0;
	long int geompos = 0;

 	for(i=0;i<5;i++)
		AtomCoord[i]=g_malloc(taille*sizeof(char));
  
 	fd = FOpen(FileName, "rb");

	free_data_all();
	tmp = get_name_file(FileName);
	set_status_label_info("File Name",tmp);
	g_free(tmp);
	set_status_label_info("File Type","Gaussian");

	t=g_malloc(taille);

	numgeom =1;
 	do 
 	{
		set_status_label_info("Geometry","Reading");
 		OK=FALSE;
 		while(!feof(fd))
		{
	  		fgets(t,taille,fd);
			pdest = strstr( t,str);
			result = pdest - t ;
	 		if ( result >0 )
	  		{
	  			fgets(t,taille,fd);
	  			fgets(t,taille,fd);
	  			fgets(t,taille,fd);
				pdest = strstr( t, "Type" );
				result = pdest - t ;
				if(result>0)
					itype=1;
				else
					itype=0;
	  			fgets(t,taille,fd);
                		numgeom++;
				OK = TRUE;
				break;
	  		}
		}
 		if(!OK && (numgeom == 1) )
		{
 			fclose(fd);
 			g_free(t);
 			for(i=0;i<5;i++)
				g_free(AtomCoord[i]);
			set_status_label_info("File Name","Nothing");
			set_status_label_info("File Type","Nothing");
			set_status_label_info("Geometry","Nothing");
			return FALSE;
		}
 		if(!OK)break;

  		j=-1;
  		while(!feof(fd) )
  		{
    			fgets(t,taille,fd);
    			pdest = strstr( t, "----------------------------------" );
    			result = pdest - t ;
    			if ( result >0 )
    			{
				geompos = ftell(fd);
 				get_dipole_from_gaussian_output_file(fd);
				fseek(fd, geompos, SEEK_SET);
				gl_get_charges_from_gaussian_output_file(fd,j+1);
				gl_get_natural_charges_from_gaussian_output_file(fd,j+1);
				fseek(fd, geompos, SEEK_SET);
				gl_get_esp_charges_from_gaussian_output_file(fd,j+1);
				fseek(fd, geompos, SEEK_SET);
      				break;
    			}
    			j++;
    			if(GeomOrb==NULL)
				GeomOrb=g_malloc(sizeof(TypeGeomOrb));
    			else
    				GeomOrb=g_realloc(GeomOrb,(j+1)*sizeof(TypeGeomOrb));

    			if(itype==0)
    				sscanf(t,"%d %s %s %s %s",&idummy,AtomCoord[0],AtomCoord[1],AtomCoord[2],AtomCoord[3]);
    			else
    				sscanf(t,"%d %s %d %s %s %s",&idummy,AtomCoord[0],&idummy,AtomCoord[1],AtomCoord[2],AtomCoord[3]);

			AtomCoord[0][0]=toupper(AtomCoord[0][0]);
	 		l=strlen(AtomCoord[0]);
          		if (l==2)
	 			AtomCoord[0][1]=tolower(AtomCoord[0][1]);

    			GeomOrb[j].Symb=g_strdup(symb_atom_get((guint)atoi(AtomCoord[0])));
    			for(i=0;i<3;i++)
    				GeomOrb[j].C[i]=atof(ang_to_bohr(AtomCoord[i+1]));

			GeomOrb[j].Prop = prop_atom_get(GeomOrb[j].Symb);
			GeomOrb[j].partialCharge = 0.0;
			GeomOrb[j].nuclearCharge = get_atomic_number_from_symbol(GeomOrb[j].Symb);
			/* Debug("%s %f %f %f \n",GeomOrb[j].Symb,GeomOrb[j].C[0],GeomOrb[j].C[1],GeomOrb[j].C[2]);*/
  		}
		if(num >0 && (gint)numgeom-1 == num)
			break;
			
 	}while(!feof(fd));

 	Ncenters = j+1;
 	fclose(fd);
 	g_free(t);
 	for(i=0;i<5;i++)
		g_free(AtomCoord[i]);
 	if(Ncenters == 0 )
  		g_free(GeomOrb);
 	else
	{
  		DefineType();
  		/* PrintGeomOrb();*/
	}
	buildBondsOrb();
	reset_grid_limits();
	init_atomic_orbitals();
	set_status_label_info("Geometry","Ok");
	RebuildGeom = TRUE;
	if(this_is_a_new_geometry()) free_objects_all();
	glarea_rafresh(GLArea);
	return TRUE;
}
/********************************************************************************/
gboolean gl_read_gaussn_file_geomi(gchar *FileName,gint num)
{



	FILE* fd;
 	if ((!FileName) || (strcmp(FileName,"") == 0))
 	{
		Message("Sorry\n No file slected","Error",TRUE);
    		return FALSE;
 	}
 	fd = FOpen(FileName, "rb");
 	if(fd ==NULL)
 	{
  		Message("Sorry\nI can not open this file","Error",TRUE);
  		return FALSE;
 	}
	fclose(fd);


	if(gl_read_gaussn_file_geomi_str(FileName,num,"Standard orientation:"))
		return TRUE;
	if(gl_read_gaussn_file_geomi_str(FileName,num,"Input orientation:"))
		return TRUE;
	/* for calculation with nosym option */
	if(!gl_read_gaussn_file_geomi_str(FileName,num,"Z-Matrix orientation:"))
	{
  		Message("Sorry\nI can not read geometry in this file","Error",TRUE);
		return FALSE;
	}
	return TRUE;

}
/********************************************************************************/
void gl_get_esp_charges_from_mopac_output_file(FILE* fd)
{
 	guint taille=BSIZE;
  	gchar *t = g_malloc(BSIZE*sizeof(gchar));
  	gchar* pdest;

	Dipole.def = FALSE;

  	while(!feof(fd) )
	{
    		pdest = NULL;
		Dipole.def = FALSE;
		if(!fgets(t,taille,fd)) break;
    		pdest = strstr( t, "ELECTROSTATIC POTENTIAL CHARGES");

		if(pdest)
		{
			gint j;
			gchar dum1[100];
			gchar dum2[100];
			gchar dum3[100];

			if(!fgets(t,taille,fd)) break;
			if(!fgets(t,taille,fd)) break;
			for(j=0;j<Ncenters;j++)
			{
    				if(!fgets(t,taille,fd)) break;
				sscanf(t,"%s %s %s",dum1,dum2,dum3);
				GeomOrb[j].partialCharge=atof(dum3);
			}
			break;
		}
	}
	g_free(t);
}
/********************************************************************************/
void gl_get_charges_from_mopac_output_file(FILE* fd)
{
 	guint taille=BSIZE;
  	gchar *t = g_malloc(BSIZE*sizeof(gchar));
  	gchar* pdest;

	Dipole.def = FALSE;

  	while(!feof(fd) )
	{
    		pdest = NULL;
		Dipole.def = FALSE;
    		fgets(t,taille,fd);
    		pdest = strstr( t, "TYPE          CHARGE");

		if(pdest)
		{
			gint j;
			gchar dum1[100];
			gchar dum2[100];
			gchar dum3[100];

			for(j=0;j<Ncenters;j++)
			{
    				if(!fgets(t,taille,fd)) break;
				sscanf(t,"%s %s %s",dum1,dum2,dum3);
				GeomOrb[j].partialCharge=atof(dum3);
			}
			break;
		}
	}
	g_free(t);
}
/********************************************************************************/
gboolean gl_read_mopac_output_file_geomi(gchar *fileName, gint numgeometry)
{
	gchar *t;
	gchar *tmp;
	gboolean OK;
	gchar *AtomCoord[5];
	FILE *fd;
	guint taille=BSIZE;
	guint idummy;
	guint i;
	gint j=0;
	gint l;
	gint numgeom;
	gchar *pdest;
	long int geomposok = 0;

	for(i=0;i<5;i++) AtomCoord[i]=g_malloc(taille*sizeof(gchar));
	 
	t=g_malloc(taille*sizeof(gchar));
	fd = FOpen(fileName, "rb");
	if(fd ==NULL)
	{
	 	g_free(t);
	 	t = g_strdup_printf("Sorry\nI can not open %s  file ",fileName);
	 	Message(t," Error ",TRUE);
	 	g_free(t);
	 	return FALSE;
	}
	free_data_all();
	tmp = get_name_file(fileName);
	set_status_label_info("File Name",tmp);
	g_free(tmp);
	set_status_label_info("File Type","Mopac");
	set_status_label_info("Geometry","Reading");
	numgeom =0;
	OK=FALSE;
	 while(!feof(fd))
	 {
		if(!fgets(t,taille,fd))break;
		pdest = strstr( t, "CARTESIAN COORDINATES");
		if(pdest) 
		{
			if(!fgets(t,taille,fd)) {pdest=0;break;}
			if(!fgets(t,taille,fd)) {pdest=0;break;}
			if(!fgets(t,taille,fd)) {pdest=0;break;}
		}
		if ( pdest )
		{
			numgeom++;
			geomposok = ftell(fd);
			if(numgeom == numgeometry )
			{
				OK = TRUE;
				break;
			}
			if(numgeometry<0)
			{
				OK = TRUE;
			}
		}
	 }
	 if(numgeom == 0)
	 {
		g_free(t);
		t = g_strdup_printf("Sorry\nI can not read geometry in	%s file ",fileName);
		Message(t," Error ",TRUE);
		g_free(t);
		set_status_label_info("File Name","Nothing");
		set_status_label_info("File Type","Nothing");
		set_status_label_info("Geometry","Nothing");
		return FALSE;
	  }
	Dipole.def = FALSE;
	for(i=0;i<3;i++) Dipole.Value[i] = 0;
	j=-1;
	fseek(fd, geomposok, SEEK_SET);
	while(!feof(fd) )
	{
		if(!fgets(t,taille,fd))break;
		if(this_is_a_backspace(t))
		{
			if(numgeometry<0)
			{
				Ncenters = j+1;
				fseek(fd, 0, SEEK_SET);
				gl_get_charges_from_mopac_output_file(fd);
				fseek(fd, 0, SEEK_SET);
				gl_get_esp_charges_from_mopac_output_file(fd);
				fseek(fd, 0, SEEK_SET);
 				get_dipole_from_mopac_output_file(fd);
			}
			break;
		}
		j++;
    		if(GeomOrb==NULL) GeomOrb=g_malloc(sizeof(TypeGeomOrb));
    		else GeomOrb=g_realloc(GeomOrb,(j+1)*sizeof(TypeGeomOrb));

		sscanf(t,"%d %s %s %s %s",&idummy,AtomCoord[0],AtomCoord[1],AtomCoord[2],AtomCoord[3]);
		AtomCoord[0][0]=toupper(AtomCoord[0][0]);
		l=strlen(AtomCoord[0]); 
		if (l==2) AtomCoord[0][1]=tolower(AtomCoord[0][1]);
		if(l==1)sprintf(t,"%c",AtomCoord[0][0]);
		else sprintf(t,"%c%c",AtomCoord[0][0],AtomCoord[0][1]);


		GeomOrb[j].Symb=g_strdup(AtomCoord[0]);
    		for(i=0;i<3;i++) 
		{
			GeomOrb[j].C[i]=atof((AtomCoord[i+1]))*ANG_TO_BOHR;
		}

		GeomOrb[j].Prop = prop_atom_get(GeomOrb[j].Symb);
		GeomOrb[j].partialCharge = 0.0;
		GeomOrb[j].nuclearCharge = get_atomic_number_from_symbol(GeomOrb[j].Symb);
	  }
	fclose(fd);
	Ncenters = j+1;
	if(Ncenters >0)
  		DefineType();

	RebuildGeom = TRUE;
	buildBondsOrb();
	reset_grid_limits();
	init_atomic_orbitals();
	set_status_label_info("Geometry","Ok");
	g_free(t);
	for(i=0;i<5;i++) g_free(AtomCoord[i]);
	return TRUE;
}
/********************************************************************************/
gboolean gl_read_mopac_aux_file_geomi(gchar *fileName, gint numgeometry)
{
	gchar *t;
	gchar *tmp;
	gboolean OK;
	gchar *AtomCoord[5];
	FILE *fd;
	guint taille=BSIZE;
	guint i;
	gint j=0;
	gint l;
	gint numgeom;
	gchar *pdest;
	long int geomposok = 0;
	gchar** elements = NULL;
	gint nElements = 0;
	gchar** nuclearCharges = NULL;
	gint nNuclearCharges = 0;
	gchar** partialCharges = NULL;
	gint nPartialCharges = 0;

	for(i=0;i<5;i++) AtomCoord[i]=g_malloc(taille*sizeof(gchar));
	 
	t=g_malloc(taille*sizeof(gchar));
	fd = FOpen(fileName, "rb");
	if(fd ==NULL)
	{
	 	g_free(t);
	 	t = g_strdup_printf("Sorry\nI can not open %s  file ",fileName);
	 	Message(t," Error ",TRUE);
	 	g_free(t);
	 	return FALSE;
	}
	OK=FALSE;
	elements = get_one_block_from_aux_mopac_file(fd, "ATOM_EL[",  &nElements);
	if(elements) OK = TRUE;
	if(!OK) 
	{
	 	g_free(t);
	 	t = g_strdup_printf("Sorry\nI can not read the atom symbols in %s  file ",fileName);
	 	Message(t," Error ",TRUE);
	 	g_free(t);
		fclose(fd);
	 	return FALSE;
	 }
 	geomposok = ftell(fd);
	nuclearCharges = get_one_block_from_aux_mopac_file(fd, "ATOM_CORE[",  &nNuclearCharges);
	if(!nuclearCharges) fseek(fd, geomposok, SEEK_SET);

	numgeom =0;
	free_data_all();
	tmp = get_name_file(fileName);
	set_status_label_info("File Name",tmp);
	g_free(tmp);
	set_status_label_info("File Type","Mopac");
	set_status_label_info("Geometry","Reading");
	 while(!feof(fd))
	 {
		if(!fgets(t,taille,fd))break;
		if(numgeometry<0)
			pdest = strstr( t, "ATOM_X_OPT:ANGSTROMS");
		else
			pdest = strstr( t, "ATOM_X_UPDATED:ANGSTROMS");
		if ( pdest )
		{
			numgeom++;
			geomposok = ftell(fd);
			if(numgeom == numgeometry )
			{
				OK = TRUE;
				break;
			}
			if(numgeometry<0)
			{
				OK = TRUE;
			}
		}
	 }
	 if(numgeom == 0)
	 {
		free_one_string_table(elements, nElements);
		g_free(t);
		t = g_strdup_printf("Sorry\nI can not read geometry in	%s file ",fileName);
		Message(t," Error ",TRUE);
		g_free(t);
		set_status_label_info("File Name","Nothing");
		set_status_label_info("File Type","Nothing");
		set_status_label_info("Geometry","Nothing");
		fclose(fd);
		return FALSE;
	  }

	Dipole.def = FALSE;
	for(i=0;i<3;i++) Dipole.Value[i] = 0;
	j=-1;
	fseek(fd, geomposok, SEEK_SET);
	while(!feof(fd) )
	{
		if(!fgets(t,taille,fd))break;
		if(strstr( t, "[")
		  || strstr(t,"HEAT_OF_FORM_UPDATED")
		  || strstr( t, "####################################")
		  ||this_is_a_backspace(t))
		{
			break;
		}
		if(j+1>nElements)break;
		j++;
    		if(GeomOrb==NULL) GeomOrb=g_malloc(sizeof(TypeGeomOrb));
    		else GeomOrb=g_realloc(GeomOrb,(j+1)*sizeof(TypeGeomOrb));

		sscanf(t,"%s %s %s",AtomCoord[1],AtomCoord[2],AtomCoord[3]);
		if(j<nElements) sprintf(AtomCoord[0],elements[j]);
		else sprintf(AtomCoord[0],"X");
		AtomCoord[0][0]=toupper(AtomCoord[0][0]);
		l=strlen(AtomCoord[0]); 
		if (l==2) AtomCoord[0][1]=tolower(AtomCoord[0][1]);
		if(l==1)sprintf(t,"%c",AtomCoord[0][0]);
		else sprintf(t,"%c%c",AtomCoord[0][0],AtomCoord[0][1]);

		GeomOrb[j].Symb=g_strdup(AtomCoord[0]);
    		for(i=0;i<3;i++) 
		{
			GeomOrb[j].C[i]=atof((AtomCoord[i+1]))*ANG_TO_BOHR;
		}

		GeomOrb[j].Prop = prop_atom_get(GeomOrb[j].Symb);
		GeomOrb[j].partialCharge = 0.0;
		if(nuclearCharges && nNuclearCharges>j) GeomOrb[j].nuclearCharge = atof(nuclearCharges[j]);
		else GeomOrb[j].nuclearCharge = get_atomic_number_from_symbol(GeomOrb[j].Symb);
	  }
	Ncenters = j+1;
	if(Ncenters >0)
  		DefineType();

	if(numgeometry<0)
	{
		fseek(fd, geomposok, SEEK_SET);
		partialCharges = get_one_block_from_aux_mopac_file(fd, "ATOM_CHARGES[",  &nPartialCharges);
		if(partialCharges)
		{
    			for(j=0;j<Ncenters;j++) 
				if(j<nPartialCharges) GeomOrb[j].partialCharge=atof(partialCharges[j]);
			free_one_string_table(partialCharges, nPartialCharges);
		}
	}
	fclose(fd);

	g_free(t);
	free_one_string_table(elements, nElements);
	free_one_string_table(nuclearCharges, nNuclearCharges);
	for(i=0;i<5;i++) g_free(AtomCoord[i]);
	buildBondsOrb();
	reset_grid_limits();
	init_atomic_orbitals();
	set_status_label_info("Geometry","Ok");
	RebuildGeom = TRUE;
	if(this_is_a_new_geometry()) free_objects_all();
	glarea_rafresh(GLArea);
	return TRUE;
}
/********************************************************************************/
void gl_get_charges_from_qchem_output_file(FILE* fd,gint N)
{
 	guint taille=BSIZE;
  	gchar t[BSIZE];
  	gchar dump[BSIZE];
  	gchar d[BSIZE];
  	gchar* pdest;
	gint i;
	gint ngrad=0;


  	while(!feof(fd) )
	{
    		pdest = NULL;
    		fgets(t,taille,fd);
    		pdest = strstr( t, "Mulliken Net Atomic Charges");

		if(pdest)
		{
			gboolean OK = FALSE;
  			while(!feof(fd) )
			{
    				if(!fgets(t,taille,fd)) break;
				if(strstr(t,"----------------"))
				{
					OK = TRUE;
					break;
				}
			}
			if(!OK) break;

			for(i=0;i<N;i++)
			{
    				if(!feof(fd)) fgets(t,taille,fd);
				else break;
				if(sscanf(t,"%s %s %s",dump,dump,d)==3)
				{
					GeomOrb[i].partialCharge = atof(d);
				}
			}
			break;
		}
		else
		{
          		pdest = strstr( t, "GradGradGrad" );
			if(pdest)
			{
				ngrad++;
			}
			if(ngrad>2)
				break;
		}

	}
}
/********************************************************************************/
gboolean gl_read_qchem_file_geomi(gchar *FileName,gint num)
{
 	gchar *t;
 	gchar *tmp = NULL;
 	gboolean OK;
 	gchar *AtomCoord[5];
 	FILE *fd;
 	guint taille=BSIZE;
 	guint idummy;
 	guint i;
 	gint j=0;
 	gint l;
 	guint numgeom;
 	gchar *pdest;
	long geompos = 0;

 	for(i=0;i<5;i++)
		AtomCoord[i]=g_malloc(taille*sizeof(char));
  
 	fd = FOpen(FileName, "rb");

	free_data_all();
	tmp = get_name_file(FileName);
	set_status_label_info("File Name",tmp);
	g_free(tmp);
	set_status_label_info("File Type","Gaussian");

	t=g_malloc(taille);

	numgeom =1;
 	do 
 	{
		set_status_label_info("Geometry","Reading");
 		OK=FALSE;
 		while(!feof(fd))
		{
	  		if(!fgets(t,taille,fd))break;
			pdest = strstr( t,"Atom         X            Y            Z");
	 		if (pdest) 
			{
	  			if(!fgets(t,taille,fd))break;
				pdest = strstr( t,"----------------------------------");
			}
	 		if (pdest)
	  		{
                		numgeom++;
				OK = TRUE;
				break;
	  		}
		}
 		if(!OK && (numgeom == 1) )
		{
 			fclose(fd);
 			g_free(t);
 			for(i=0;i<5;i++) g_free(AtomCoord[i]);
			set_status_label_info("File Name","Nothing");
			set_status_label_info("File Type","Nothing");
			set_status_label_info("Geometry","Nothing");
			return FALSE;
		}
 		if(!OK)break;

  		j=-1;
  		while(!feof(fd) )
  		{
    			fgets(t,taille,fd);
    			pdest = strstr( t, "----------------------------------" );
    			if (pdest)
    			{
				geompos = ftell(fd);
 				get_dipole_from_qchem_output_file(fd);
				fseek(fd, geompos, SEEK_SET);
				gl_get_charges_from_qchem_output_file(fd,j+1);
				fseek(fd, geompos, SEEK_SET);
      				break;
    			}
    			j++;
    			if(GeomOrb==NULL) GeomOrb=g_malloc(sizeof(TypeGeomOrb));
    			else GeomOrb=g_realloc(GeomOrb,(j+1)*sizeof(TypeGeomOrb));

    			sscanf(t,"%d %s %s %s %s",&idummy,AtomCoord[0],AtomCoord[1],AtomCoord[2],AtomCoord[3]);

			AtomCoord[0][0]=toupper(AtomCoord[0][0]);
	 		l=strlen(AtomCoord[0]);
          		if (l==2) 
			{
				AtomCoord[0][1]=tolower(AtomCoord[0][1]);
				if(isdigit(AtomCoord[0][1]))l=1;
			}
			if(l==1)sprintf(t,"%c",AtomCoord[0][0]);
		         else sprintf(t,"%c%c",AtomCoord[0][0],AtomCoord[0][1]);

    			GeomOrb[j].Symb=g_strdup(t);
    			for(i=0;i<3;i++) GeomOrb[j].C[i]=atof(ang_to_bohr(AtomCoord[i+1]));

			GeomOrb[j].Prop = prop_atom_get(GeomOrb[j].Symb);
			GeomOrb[j].partialCharge = 0.0;
			GeomOrb[j].nuclearCharge = get_atomic_number_from_symbol(GeomOrb[j].Symb);
  		}
		if(num >0 && (gint)numgeom-1 == num) break;
 	}while(!feof(fd));

 	Ncenters = j+1;
 	fclose(fd);
 	g_free(t);
 	for(i=0;i<5;i++) g_free(AtomCoord[i]);
 	if(Ncenters == 0 ) g_free(GeomOrb);
 	else
	{
  		DefineType();
	}
	buildBondsOrb();
	reset_grid_limits();
	init_atomic_orbitals();
	set_status_label_info("Geometry","Ok");
	RebuildGeom = TRUE;
	if(this_is_a_new_geometry()) free_objects_all();
	glarea_rafresh(GLArea);
	return TRUE;
}
/********************************************************/
void gl_read_first_dalton_file(GabeditFileChooser *SelecFile, gint response_id)
{
 	gchar *FileName;
	if(response_id != GTK_RESPONSE_OK) return;
 	FileName = gabedit_file_chooser_get_current_file(SelecFile);
	gtk_widget_hide(GTK_WIDGET(SelecFile));
	while( gtk_events_pending() ) gtk_main_iteration();

	add_objects_for_new_grid();
 	gl_read_dalton_file_geomi(FileName,1);
	RebuildGeom = TRUE;
	if(this_is_a_new_geometry()) free_objects_all();
	glarea_rafresh(GLArea);
} 
/********************************************************/
void gl_read_last_dalton_file(GabeditFileChooser *SelecFile, gint response_id)
{
 	gchar *FileName;
	if(response_id != GTK_RESPONSE_OK) return;
 	FileName = gabedit_file_chooser_get_current_file(SelecFile);
	gtk_widget_hide(GTK_WIDGET(SelecFile));
	while( gtk_events_pending() )
		gtk_main_iteration();
	add_objects_for_new_grid();
 	gl_read_dalton_file_geomi(FileName,-1);
	RebuildGeom = TRUE;
	if(this_is_a_new_geometry()) free_objects_all();
	glarea_rafresh(GLArea);
}
/********************************************************/
void gl_read_first_gamess_file(GabeditFileChooser *SelecFile, gint response_id)
{
 	gchar *FileName;
	if(response_id != GTK_RESPONSE_OK) return;
 	FileName = gabedit_file_chooser_get_current_file(SelecFile);
	gtk_widget_hide(GTK_WIDGET(SelecFile));
	while( gtk_events_pending() ) gtk_main_iteration();

	add_objects_for_new_grid();
 	gl_read_gamess_file_geomi(FileName,1);
	RebuildGeom = TRUE;
	if(this_is_a_new_geometry()) free_objects_all();
	glarea_rafresh(GLArea);
} 
/********************************************************/
void gl_read_last_gamess_file(GabeditFileChooser *SelecFile, gint response_id)
{
 	gchar *FileName;
	if(response_id != GTK_RESPONSE_OK) return;
 	FileName = gabedit_file_chooser_get_current_file(SelecFile);
	gtk_widget_hide(GTK_WIDGET(SelecFile));
	while( gtk_events_pending() )
		gtk_main_iteration();
	add_objects_for_new_grid();
 	gl_read_gamess_file_geomi(FileName,-1);
	RebuildGeom = TRUE;
	if(this_is_a_new_geometry()) free_objects_all();
	glarea_rafresh(GLArea);
}
/********************************************************/
void gl_read_first_gauss_file(GabeditFileChooser *SelecFile, gint response_id)
{
 	gchar *FileName;
	if(response_id != GTK_RESPONSE_OK) return;
 	FileName = gabedit_file_chooser_get_current_file(SelecFile);
	gtk_widget_hide(GTK_WIDGET(SelecFile));
	while( gtk_events_pending() ) gtk_main_iteration();

	add_objects_for_new_grid();
 	gl_read_gaussn_file_geomi(FileName,1);
	RebuildGeom = TRUE;
	if(this_is_a_new_geometry()) free_objects_all();
	glarea_rafresh(GLArea);
} 
/********************************************************/
void gl_read_last_gauss_file(GabeditFileChooser *SelecFile, gint response_id)
{
 	gchar *FileName;
	if(response_id != GTK_RESPONSE_OK) return;
 	FileName = gabedit_file_chooser_get_current_file(SelecFile);
	gtk_widget_hide(GTK_WIDGET(SelecFile));
	while( gtk_events_pending() )
		gtk_main_iteration();
	add_objects_for_new_grid();
 	gl_read_gaussn_file_geomi(FileName,-1);
	RebuildGeom = TRUE;
	if(this_is_a_new_geometry()) free_objects_all();
	glarea_rafresh(GLArea);
}
/********************************************************/
void gl_read_first_molcas_file(GabeditFileChooser *SelecFile, gint response_id)
{
 	gchar *FileName;
	if(response_id != GTK_RESPONSE_OK) return;
 	FileName = gabedit_file_chooser_get_current_file(SelecFile);
	gtk_widget_hide(GTK_WIDGET(SelecFile));
	while( gtk_events_pending() ) gtk_main_iteration();
	add_objects_for_new_grid();
 	gl_read_molcas_file_geomi(FileName,1);
	RebuildGeom = TRUE;
	if(this_is_a_new_geometry()) free_objects_all();
	glarea_rafresh(GLArea);
} 
/********************************************************/
void gl_read_last_molcas_file(GabeditFileChooser *SelecFile, gint response_id)
{
 	gchar *FileName;
	if(response_id != GTK_RESPONSE_OK) return;
 	FileName = gabedit_file_chooser_get_current_file(SelecFile);
	gtk_widget_hide(GTK_WIDGET(SelecFile));
	while( gtk_events_pending() ) gtk_main_iteration();
	add_objects_for_new_grid();
 	gl_read_molcas_file_geomi(FileName,-1);
	RebuildGeom = TRUE;
	if(this_is_a_new_geometry()) free_objects_all();
	glarea_rafresh(GLArea);
}  
/********************************************************/
void gl_read_first_molpro_file(GabeditFileChooser *SelecFile, gint response_id)
{
 	gchar *FileName;
	if(response_id != GTK_RESPONSE_OK) return;
 	FileName = gabedit_file_chooser_get_current_file(SelecFile);
	gtk_widget_hide(GTK_WIDGET(SelecFile));
	while( gtk_events_pending() )
		gtk_main_iteration();
	add_objects_for_new_grid();
 	gl_read_molpro_file_geomi(FileName,1);
	RebuildGeom = TRUE;
	if(this_is_a_new_geometry()) free_objects_all();
	glarea_rafresh(GLArea);
} 
/********************************************************/
void gl_read_last_molpro_file(GabeditFileChooser *SelecFile, gint response_id)
{
 	gchar *FileName;
	if(response_id != GTK_RESPONSE_OK) return;
 	FileName = gabedit_file_chooser_get_current_file(SelecFile);
	gtk_widget_hide(GTK_WIDGET(SelecFile));
	while( gtk_events_pending() )
		gtk_main_iteration();
	add_objects_for_new_grid();
 	gl_read_molpro_file_geomi(FileName,-1);
	RebuildGeom = TRUE;
	if(this_is_a_new_geometry()) free_objects_all();
	glarea_rafresh(GLArea);
}  
/********************************************************/
void gl_read_first_mpqc_file(GabeditFileChooser *SelecFile, gint response_id)
{
 	gchar *fileName;
	if(response_id != GTK_RESPONSE_OK) return;
 	fileName = gabedit_file_chooser_get_current_file(SelecFile);
	gtk_widget_hide(GTK_WIDGET(SelecFile));
	while( gtk_events_pending() )
		gtk_main_iteration();
	add_objects_for_new_grid();
 	gl_read_mpqc_file_geomi(fileName,1);
	RebuildGeom = TRUE;
	if(this_is_a_new_geometry()) free_objects_all();
	glarea_rafresh(GLArea);
} 
/********************************************************/
void gl_read_last_mpqc_file(GabeditFileChooser *SelecFile, gint response_id)
{
 	gchar *fileName;
	if(response_id != GTK_RESPONSE_OK) return;
 	fileName = gabedit_file_chooser_get_current_file(SelecFile);
	gtk_widget_hide(GTK_WIDGET(SelecFile));
	while( gtk_events_pending() )
		gtk_main_iteration();
	add_objects_for_new_grid();
 	gl_read_mpqc_file_geomi(fileName,-1);
	RebuildGeom = TRUE;
	if(this_is_a_new_geometry()) free_objects_all();
	glarea_rafresh(GLArea);
}
/********************************************************/
void gl_read_first_mopac_output_file(GabeditFileChooser *SelecFile, gint response_id)
{
 	gchar *FileName;
	if(response_id != GTK_RESPONSE_OK) return;
 	FileName = gabedit_file_chooser_get_current_file(SelecFile);
	gtk_widget_hide(GTK_WIDGET(SelecFile));
	while( gtk_events_pending() ) gtk_main_iteration();

	add_objects_for_new_grid();
 	gl_read_mopac_output_file_geomi(FileName,1);
	RebuildGeom = TRUE;
	if(this_is_a_new_geometry()) free_objects_all();
	glarea_rafresh(GLArea);
} 
/********************************************************/
void gl_read_last_mopac_output_file(GabeditFileChooser *SelecFile, gint response_id)
{
 	gchar *FileName;
	if(response_id != GTK_RESPONSE_OK) return;
 	FileName = gabedit_file_chooser_get_current_file(SelecFile);
	gtk_widget_hide(GTK_WIDGET(SelecFile));
	while( gtk_events_pending() )
		gtk_main_iteration();
	add_objects_for_new_grid();
 	gl_read_mopac_output_file_geomi(FileName,-1);
	RebuildGeom = TRUE;
	if(this_is_a_new_geometry()) free_objects_all();
	glarea_rafresh(GLArea);
}
/********************************************************/
void gl_read_first_mopac_aux_file(GabeditFileChooser *SelecFile, gint response_id)
{
 	gchar *FileName;
	if(response_id != GTK_RESPONSE_OK) return;
 	FileName = gabedit_file_chooser_get_current_file(SelecFile);
	gtk_widget_hide(GTK_WIDGET(SelecFile));
	while( gtk_events_pending() ) gtk_main_iteration();

	add_objects_for_new_grid();
 	gl_read_mopac_aux_file_geomi(FileName,1);
	RebuildGeom = TRUE;
	if(this_is_a_new_geometry()) free_objects_all();
	glarea_rafresh(GLArea);
} 
/********************************************************/
void gl_read_last_mopac_aux_file(GabeditFileChooser *SelecFile, gint response_id)
{
 	gchar *FileName;
	if(response_id != GTK_RESPONSE_OK) return;
 	FileName = gabedit_file_chooser_get_current_file(SelecFile);
	gtk_widget_hide(GTK_WIDGET(SelecFile));
	while( gtk_events_pending() )
		gtk_main_iteration();
	add_objects_for_new_grid();
 	gl_read_mopac_aux_file_geomi(FileName,-1);
	RebuildGeom = TRUE;
	if(this_is_a_new_geometry()) free_objects_all();
	glarea_rafresh(GLArea);
}
/********************************************************/
void gl_read_first_qchem_file(GabeditFileChooser *SelecFile, gint response_id)
{
 	gchar *FileName;
	if(response_id != GTK_RESPONSE_OK) return;
 	FileName = gabedit_file_chooser_get_current_file(SelecFile);
	gtk_widget_hide(GTK_WIDGET(SelecFile));
	while( gtk_events_pending() ) gtk_main_iteration();

	add_objects_for_new_grid();
 	gl_read_qchem_file_geomi(FileName,1);
	RebuildGeom = TRUE;
	if(this_is_a_new_geometry()) free_objects_all();
	glarea_rafresh(GLArea);
} 
/********************************************************/
void gl_read_last_qchem_file(GabeditFileChooser *SelecFile, gint response_id)
{
 	gchar *FileName;
	if(response_id != GTK_RESPONSE_OK) return;
 	FileName = gabedit_file_chooser_get_current_file(SelecFile);
	gtk_widget_hide(GTK_WIDGET(SelecFile));
	while( gtk_events_pending() )
		gtk_main_iteration();
	add_objects_for_new_grid();
 	gl_read_qchem_file_geomi(FileName,-1);
	RebuildGeom = TRUE;
	if(this_is_a_new_geometry()) free_objects_all();
	glarea_rafresh(GLArea);
}
/*******************************************************/
void gl_read_molden_file(GabeditFileChooser *SelecFile, gint response_id)
{
 	gchar *FileName;
	if(response_id != GTK_RESPONSE_OK) return;
 	FileName = gabedit_file_chooser_get_current_file(SelecFile);
	gtk_widget_hide(GTK_WIDGET(SelecFile));
	while( gtk_events_pending() )
		gtk_main_iteration();
	add_objects_for_new_grid();
 	gl_read_molden_file_geom(FileName);
} 
/*******************************************************/
void gl_read_gabedit_file(GabeditFileChooser *SelecFile, gint response_id)
{
 	gchar *FileName;
	if(response_id != GTK_RESPONSE_OK) return;
 	FileName = gabedit_file_chooser_get_current_file(SelecFile);
	gtk_widget_hide(GTK_WIDGET(SelecFile));
	while( gtk_events_pending() )
		gtk_main_iteration();
	add_objects_for_new_grid();
 	gl_read_gabedit_file_geom(FileName);
} 
/*******************************************************/
