# -*- coding: utf-8 -*-
#qt.glal.py
"""This module provides a graphic library abstraction layer for Cyclograph"""

# Copyright (C) 2008, 2009, 2010, 2011, 2012, 2013 Federico Brega, Pierluigi Villani

# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 3
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

from __future__ import print_function, unicode_literals

import sip
sip.setapi('QString', 2)

from PyQt4 import QtGui
from PyQt4 import QtCore
from version import VERSION

import gui 

def initapp(*args):
    """Create initial app for gui"""
    app = QtGui.QApplication(*args)
    
    #Tell to the graphic library witch language use
    translations_path = QtCore.QLibraryInfo.location(QtCore.QLibraryInfo.TranslationsPath)
    translation_file = str(translations_path) + "/qt_" + str(QtCore.QLocale.system().name())
    translator = QtCore.QTranslator()
    translator.load(translation_file)
    translator.load(':/qt-translations/qt_'+ QtCore.QLocale.system().name())
    app.installTranslator(translator)
    
    return app

def bind(cgfunc, menuopt):
    """Connect gui menu option with cg function"""
    return QtCore.QObject.connect(menuopt,
                                  QtCore.SIGNAL("triggered()"), cgfunc)

def bind_close(cg_exit, gui):
    """Connect gui close event with cg exit function"""
    gui.cg_exit = cg_exit
    return QtCore.QObject.connect(gui.menu_item_exit, QtCore.SIGNAL("triggered()"), gui.quit)

def ToolbarBind(fun, button):
    """Connect gui toolbar button with cg function"""
    return QtCore.QObject.connect(button, QtCore.SIGNAL("triggered()"), fun)

def OptionCheck(guiopt, option):
    """Set a gui menu option to selected/unselected """
    guiopt.setChecked(option)

def enable_saving(gui, bool_val):
    """Enable or disable saving options according to bool_val"""
    gui.menu_item_save.setEnabled(bool_val)
    gui.menu_item_save_as.setEnabled(bool_val)
    gui.action_add.setEnabled(bool_val)
    gui.action_edit.setEnabled(bool_val)
    gui.action_delete.setEnabled(bool_val)
    gui.action_plot.setEnabled(bool_val)
    gui.action_map.setEnabled(bool_val)
    gui.action_properties.setEnabled(bool_val)

def addstatusbartext(maingui, text):
    """Add text to main gui status bar"""
    maingui.addstatusbartext(text)

def signalbug(event=None):
    """Open webpage to signal bugs"""
    bugUrl = 'http://sourceforge.net/project/memberlist.php?group_id=227295'
    QtGui.QDesktopServices.openUrl(QtCore.QUrl(bugUrl))

class Message():
    """ Qt message"""
    def __init__(self):
        self.qobj = QtCore.QObject()
    def send(self, message, slope_number, row_num):
        """ Send message"""
        self.qobj.emit(QtCore.SIGNAL(message+'(PyQt_PyObject,PyQt_PyObject)'), slope_number, row_num)
    def subscribe(self, function, emitter, message):
        """ Subscribe message"""
        self.qobj.connect(emitter.qobj,
                          QtCore.SIGNAL(message+'(PyQt_PyObject,PyQt_PyObject)'), function)

class Notebook:
    """ Qt Notebook"""
    def __init__(self, notebook, fun_close):
        self.notebook = notebook
        self.fun_close = fun_close
        if QtCore.PYQT_VERSION_STR >= '4.5':
            self.notebook.setTabsClosable(True)
            QtCore.QObject.connect(self.notebook, QtCore.SIGNAL("tabCloseRequested(int)"),
                                fun_close)
    def Page(self):
        """ Return qt Page"""
        return gui.Page(self.notebook)
    def set_page_label(self, page_num, text):
        """ Set page label text"""
        self.notebook.setTabText(page_num,  text)
    def add_page(self, page, title):
        """ Add page"""
        self.notebook.addTab(page, title)
        if QtCore.PYQT_VERSION_STR < '4.5':
            QtCore.QObject.connect(page.button_close, QtCore.SIGNAL("clicked()"),
                                self.fun_close)
        else:
            page.button_close.setVisible(False)
        self.notebook.setCurrentWidget(page)
    def remove_page(self, page_num):
        """ Remove page"""
        self.notebook.removeTab(page_num)
    def getselpagnum(self):
        """ Return number from selected page"""
        return self.notebook.currentIndex()
    def setselpagnum(self, page_num):
        """Set current selected page"""
        self.notebook.setCurrentWidget(self.get_page(page_num))
    def get_page(self, page_num):
        """ Return page"""
        page = self.notebook.widget(page_num)
        return page
    def get_pagenum(self, argument):
        """Return page_num from page passed as argument"""
        return argument[0]

class DeviceContext():
    """Device Context Qt class wrapper"""
    def _convqcolor(self, color):
        """ Convert color"""
        if isinstance(color, basestring):
            if color.startswith("rgb"):
                col = map(int, color[4:-1].split(','))
                return QtGui.QColor(*col)
            else:
                return QtGui.QColor(color)
        else:
            return QtGui.QColor(*color)

    def init_draw_surf(self, panel):
        """ Initialize drawing surface"""
        (self.size_x, self.size_y) = (panel.width(), panel.height())
        self.paint = QtGui.QPainter()
        self.paint.begin(panel)
        self.paint.setRenderHint(QtGui.QPainter.SmoothPixmapTransform)
        self.paint.setRenderHint(QtGui.QPainter.Antialiasing)

    def shear(self, shear):
        """ device shear"""
        self.paint.shear(0.0, shear)
    def getsize(self):
        """ Return size"""
        return (self.size_x, self.size_y)
    def gradfill(self, rect, startcolor, endcolor):
        """ Fill gradient"""
        (sx, sy, ex, ey) = (rect[0], rect[1], rect[0], rect[1]+rect[3])
        gradient = QtGui.QLinearGradient(sx, sy, ex, ey)
        sqcol = self._convqcolor(startcolor)
        gradient.setColorAt(0, sqcol)
        eqcol = self._convqcolor(endcolor)
        gradient.setColorAt(1, eqcol)
        brush = QtGui.QBrush(gradient)
        self.paint.fillRect(QtCore.QRect(*rect), brush)
    def setpen(self, color, size):
        """ Set pen's color and size"""
        qcol = self._convqcolor(color)
        pen = QtGui.QPen(qcol)
        pen.setWidth(size)
        self.paint.setPen(pen)
    def setfont(self, fontdict):
        """ Set font to use"""
        font = QtGui.QFont()
        self.fontmetrics = QtGui.QFontMetrics(font)
        try:
            font.setFamily(fontdict["des"])
            font.setPointSize(fontdict["dim"])
            font.setWeight(eval('QtGui.QFont.'+fontdict["typ"].capitalize()))
            self.paint.setFont(font)
        except Exception:
            pass
    def drawtext(self, text, pos_x, pos_y):
        """ Draw text at position x,y"""
        self.paint.drawText(pos_x, pos_y+10, text)
    def gettextwidth(self, text):
        """ Return text length"""
        return self.fontmetrics.width(text)
    def gettextheight(self, text):
        """ Return text height"""
        return self.fontmetrics.height()
    def drawline(self, pos_x0, pos_y0, pos_x1, pos_y1):
        """ Draw line"""
        self.paint.drawLine(pos_x0, pos_y0, pos_x1, pos_y1)
    def setlineargradientbrush(self, colorlist, startp, endp):
        """ Get a linear gradient from startp to endp, using colors in colorlist.
        The elments of colorlist are tuple in the format (color, realtive position)."""
        grad = QtGui.QLinearGradient(startp[0], startp[1], endp[0], endp[1])
        for color in colorlist:
            grad.setColorAt(color[1], self._convqcolor(color[0]))
        brush = QtGui.QBrush(grad)
        self.paint.setBrush(brush)
    def setbrush(self, color):
        """ Set brush's color"""
        qcol = self._convqcolor(color)
        brush = QtGui.QBrush(qcol)
        self.paint.setBrush(brush)
    def drawrectangle(self, pos_x0, pos_y0, width, height):
        """ Draw rectangle"""
        self.paint.drawRect(pos_x0, pos_y0, width, height)
    def drawrotatedtext(self, text, pos_x, pos_y, angle):
        """ Draw rotated text at position x,y by angle"""
        self.paint.save()
        self.paint.rotate(-angle)
        transform = QtGui.QTransform()
        transform.rotate(angle)
        (newx, newy) = transform.map(pos_x+10, pos_y-5)
        self.paint.drawText(newx, newy, text)
        self.paint.restore()
    def drawpolygon(self, sequence):
        """ Draw polygon"""
        points = [QtCore.QPoint(*p) for p in sequence]
        self.paint.drawPolygon(*points)
    def startpath(self, point):
        """ Start a path in the specified point,"""
        point = QtCore.QPointF(*point)
        path = QtGui.QPainterPath()
        path.moveTo(point)
        return path
    def drawpathlineto(self, path, point):
        """ Draw a straight line from the last point to the given point."""
        point = QtCore.QPointF(*point)
        path.lineTo(point)
    def drawpathcubicto(self, path, controlpoints):
        """ Draw a cubic Beziér frome the last point using the given list of
        three control points."""
        points = [QtCore.QPointF(pnt[0], pnt[1]) for pnt in controlpoints]
        path.cubicTo(*points)
    def endpath(self, path):
        """ Show the path."""
        self.paint.drawPath(path)
    def end_draw(self):
        """ Finish drawing"""
        self.paint.end()
        del self.paint

class Image():
    """Image Qt class wrapper"""
    def __init__(self, size_x,  size_y, plotfnct):
        (self.size_x, self.size_y) = (size_x, size_y)
        self.plotfnct = plotfnct
        self.image = QtGui.QPixmap(size_x, size_y)
        self.image.fill() #white backgound
    def plot(self, settings):
        """ Draw slope to the device context"""
        dcwrpp = DeviceContext()
        (dcwrpp.size_x, dcwrpp.size_y) = (self.size_x, self.size_y)
        dcwrpp.paint = QtGui.QPainter()
        dcwrpp.paint.begin(self.image)
        dcwrpp.paint.setRenderHint(QtGui.QPainter.SmoothPixmapTransform)
        dcwrpp.paint.setRenderHint(QtGui.QPainter.Antialiasing)
        self.plotfnct(settings, dcwrpp)
        dcwrpp.end_draw()
    def savetofile(self, path, format):
        """ Save image to file"""
        self.image.save(path, format)

class Pdf():
    """Qt pdf class wrapper"""
    def __init__(self, filepath):
        self.dy = 90
        self.dx = 300
        self.y_incr = 30
        self.printer = QtGui.QPrinter(QtGui.QPrinter.ScreenResolution)
        self.printer.setOutputFormat(QtGui.QPrinter.PdfFormat)
        self.printer.setOutputFileName(filepath)
        self.printer.setFullPage(True)
#        paperSize = QtCore.QSizeF(793, 1122) 
#        self.printer.setPaperSize(paperSize, QtGui.QPrinter.Millimeter)
        self.printer.setCreator('Cyclograph {0}'.format(VERSION))
        self.painter = QtGui.QPainter()
        self.painter.begin(self.printer)
        self.painter.setRenderHint(QtGui.QPainter.SmoothPixmapTransform)
        self.painter.setRenderHint(QtGui.QPainter.Antialiasing)
        self.painter.setFont(QtGui.QFont('Decorative', 18))
    def plot_image(self, settings, size_x, size_y, plotfnct):
        """ Draw slope to pdf"""
        image = QtGui.QPicture()
        dcwrpp = DeviceContext()
        (dcwrpp.size_x, dcwrpp.size_y) = (size_x, size_y)
        dcwrpp.paint = QtGui.QPainter()
        dcwrpp.paint.begin(image)
        dcwrpp.paint.setRenderHint(QtGui.QPainter.SmoothPixmapTransform)
        dcwrpp.paint.setRenderHint(QtGui.QPainter.Antialiasing)
        plotfnct(settings, dcwrpp)
        dcwrpp.end_draw()
        self.painter.drawPicture(0, 0, image)
        self.dy += size_y
    def addtext(self, text):
        self.dy += self.y_incr
        self.painter.setPen(QtGui.QColor(0, 0, 0))
        self.painter.drawText(self.dx, self.dy, text)
    def addtitle(self, text):
        self.dy += self.y_incr
        self.painter.setPen(QtGui.QColor(0, 0, 102))
        self.painter.drawText(self.dx-50, self.dy, text+":")
    def save(self):
        """Save pdf"""
        self.painter.end()

class ProgressDialog():
    """ Progress dialog for importkml"""
    def __init__(self):
        self.pd = QtGui.QProgressDialog (_("Downloading altitudes"),
                                         _("Cancel"),0,1000)
    def update(self, value):
        """Update the progress shown and return if user want to abort."""
        self.pd.setValue(value)
        return self.pd.wasCanceled()
    def destroy(self):
        """ Destroy progress dialog"""
        self.pd = None

class Timer():
    """ Timer for Qt gui"""
    def __init__(self, period, callback):
        self.timer = self.timer = QtCore.QTimer()
        self.period = period
        self.timer.connect(self.timer, QtCore.SIGNAL("timeout()"), callback)
    def start(self):
        """ Start timer count"""
        self.timer.start(self.period)
    def stop(self):
        """ Stop timer count"""
        self.timer.stop()

# vim:sw=4:softtabstop=4:expandtab
