#***************************************************************************
#*   Copyright (C) 2008-2009 by Eugene V. Lyubimkin                        *
#*                                                                         *
#*   This program is free software; you can redistribute it and/or modify  *
#*   it under the terms of the GNU General Public License                  *
#*   (version 3 or above) as published by the Free Software Foundation.    *
#*                                                                         *
#*   This program is distributed in the hope that it will be useful,       *
#*   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
#*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
#*   GNU General Public License for more details.                          *
#*                                                                         *
#*   You should have received a copy of the GNU GPL                        *
#*   along with this program; if not, write to the                         *
#*   Free Software Foundation, Inc.,                                       *
#*   51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA               *
#*                                                                         *
#*   This program is free software; you can redistribute it and/or modify  *
#*   it under the terms of the Artistic License, which comes with Perl     *
#***************************************************************************
package Cupt::Cache;

=head1 NAME

Cupt::Cache - store info about available packages

=cut

use 5.10.0;
use strict;
use warnings;

use Digest;
use Digest::MD5;
use Digest::SHA;
use Fcntl qw(:seek :DEFAULT);
use POSIX qw(locale_h);
use List::MoreUtils 0.23 qw(none any);

use Memoize;
memoize('verify_signature');

use Cupt::Core;
use Cupt::Cache::Package;
use Cupt::Cache::BinaryVersion;
use Cupt::Cache::SourceVersion;
use Cupt::System::State;
use Cupt::Cache::Relation qw(stringify_relation_expression);

=begin internal

=head2 can_provide

{ I<virtual_package> => [ I<package_name>... ] }

For each I<virtual_package> this field contains the list of I<package_name>s
that B<can> provide given I<virtual_package>. Depending of package versions,
some versions of the some of <package_name>s may provide and may not provide
given I<virtual_package>. This field exists solely for
I<get_satisfying_versions> subroutine for rapid lookup.

=end internal

=cut

use Cupt::LValueFields qw(_source_packages _binary_packages _config _pin_settings
		_system_state _can_provide _extended_info _index_entries _release_data
		_binary_architecture _allow_reinstall);

=head1 FLAGS

=head2 o_memoize

This flag determines whether it worth trade space for time in time-consuming
functions. On by default. By now, it affects L</get_satisfying_versions>
method. If it's on, it stores references, so B<don't> modify results of these
functions, use them in read-only mode. It it's on, these functions are not
thread-safe.

=cut

our $o_memoize = 1;

=head1 METHODS

=head2 new

creates a new Cupt::Cache object

Parameters:

I<config> - reference to L<Cupt::Config|Cupt::Config>

Next params are treated as hash-style param list:

'-source': read Sources

'-binary': read Packages

'-installed': read dpkg status file

'-allow-reinstall': list of globs of package names which are allowed to reinstall

Example:

  my $cache = new Cupt::Cache($config, '-source' => 0, '-binary' => 1);

=cut

sub new {
	my $class = shift;
	my $self = bless [] => $class;

	$self->_config = shift;
	$self->_pin_settings = [];
	$self->_source_packages = {};
	$self->_binary_packages = {};
	$self->_release_data = {};
	$self->_release_data->{source} = [];
	$self->_release_data->{binary} = [];
	$self->_allow_reinstall = [];

	do { # ugly hack to copy trusted keyring from APT whenever possible
		my $cupt_keyring_file = $self->_config->get_string('gpgv::trustedkeyring');
		my $apt_keyring_file = '/etc/apt/trusted.gpg';
		# ignore all errors, let install do its best
		qx#install -m644 $apt_keyring_file $cupt_keyring_file >/dev/null 2>/dev/null#;
	};

	eval {
		$self->_parse_sources_lists();
	};
	if (mycatch()) {
		myerr('error while parsing sources list');
		myredie();
	}
	my $ref_index_entries = $self->get_index_entries();

	# determining which parts of cache we wish to build
	my %build_config = (
		'-source' => 1,
		'-binary' => 1,
		'-installed' => 1,
		'-allow-reinstall' => [],
		@_, # applying passed parameters
	);

	foreach my $allow_reinstall_glob (@{$build_config{'-allow-reinstall'}}) {
		glob_to_regex(my $allow_reinstall_regex = $allow_reinstall_glob);
		push @{$self->_allow_reinstall}, $allow_reinstall_regex;
	}

	if ($build_config{'-installed'}) {
		# read system settings
		$self->_system_state = Cupt::System::State->new($self->_config, $self);
	}

	foreach my $ref_index_entry (@$ref_index_entries) {
		my $index_file_to_parse = $self->get_path_of_index_list($ref_index_entry);
		my $source_type = $ref_index_entry->{'type'};
		# don't parse unneeded indexes
		if (($source_type eq 'deb' && $build_config{'-binary'}) ||
			($source_type eq 'deb-src' && $build_config{'-source'}))
		{
			eval {
				my $ref_release_info = $self->_get_release_info($self->get_path_of_release_list($ref_index_entry));
				$ref_release_info->{component} = $ref_index_entry->{'component'};
				$ref_release_info->{base_uri} = $ref_index_entry->{'uri'};
				if ($source_type eq 'deb') {
					push @{$self->_release_data->{binary}}, $ref_release_info;
				} else {
					push @{$self->_release_data->{source}}, $ref_release_info;
				}

				my @description_translations_files = $self->_get_paths_of_localized_descriptions($ref_index_entry);
				my $chosen_translation_file;
				foreach my $file (@description_translations_files) {
					if (-r $file) {
						$chosen_translation_file = $file;
						last;
					}
				}

				$self->_process_index_file($index_file_to_parse, $chosen_translation_file,
						$source_type, $ref_release_info);
			};
			if (mycatch()) {
				mywarn("skipped index file '%s'", $index_file_to_parse);
			}
		}
	}

	# reading pin settings
	$self->_parse_preferences();

	# reading list of automatically installed packages
	my $extended_states_file = $self->get_path_of_extended_states();
	$self->_parse_extended_states($extended_states_file) if -r $extended_states_file;

	# for speeding up _prepare_package calls
	$self->_binary_architecture = $self->_config->get_string('apt::architecture');

	return $self;
}

=head2 set_config

method, sets new L<Cupt::Config|Cupt::Config> for the cache

Parameters:

I<config> - reference to L<Cupt::Config|Cupt::Config>

=cut

sub set_config ($$) {
	my ($self, $config) = @_;
	$self->_config = $config;
	return;
}

=head2 get_binary_package_names

method, returns an array of binary package names
=cut

sub get_binary_package_names ($) {
	my ($self) = @_;

	return keys %{$self->_binary_packages};
}

=head2 get_source_package_names

method, returns an array of source package names

=cut

sub get_source_package_names ($) {
	my ($self) = @_;

	return keys %{$self->_source_packages};
}

=head2 get_system_state

method, returns reference to L<Cupt::System::State|Cupt::System::State>

=cut

sub get_system_state ($) {
	my ($self) = @_;

	return $self->_system_state;
}

=head2 get_extended_info

method, returns info about extended package statuses in format:

  {
    'automatically_installed' => { I<package_name> => 1 },
  }

=cut

sub get_extended_info ($) {
	my ($self) = @_;

	return $self->_extended_info;
}

=head2 is_automatically_installed

method, returns boolean value - is the package automatically installed
or not

Parameters:

I<package_name> - package name

=cut

sub is_automatically_installed ($$) {
	my ($self, $package_name) = @_;

	my $ref_auto_installed = $self->[_extended_info_offset()]->{'automatically_installed'};
	return (exists $ref_auto_installed->{$package_name} &&
			$ref_auto_installed->{$package_name});
}

=head2 get_original_apt_pin

method, returns pin value for the supplied version as described in apt_preferences(5)

Parameters:

I<version> - reference to L<Cupt::Cache::BinaryVersion|Cupt::Cache::BinaryVersion>

=cut

sub get_original_apt_pin {
	my ($self, $version) = @_;
	my $result;

	my $update_pin = sub ($) {
		if (not defined $result) {
			$result = $_[0];
		} elsif ($result < $_[0]) {
			$result = $_[0];
		}
	};

	my @available_as = @{$version->available_as};

	# release-dependent settings
	my $default_release = $self->_config->get_string('apt::default-release');
	foreach (@available_as) {
		if (defined $default_release) {
			if ($_->{release}->{archive} eq $default_release ||
				$_->{release}->{codename} eq $default_release)
			{
				$update_pin->(990);
				last; # no sense to search further, this is maximum
			}
		}
		if ($_->{release}->{not_automatic}) {
			$update_pin->(1);
		} elsif ($_->{release}->{archive} eq 'installed') {
			$update_pin->(100);
		} else {
			$update_pin->(500);
		}
	}

	# looking in pin settings
	PIN:
	foreach my $ref_pin (@{$self->[_pin_settings_offset()]}) {
		if (exists $ref_pin->{'package_name'}) {
			my $value = $ref_pin->{'package_name'};
			$version->package_name =~ m/^$value$/ or next PIN;
		}
		if (exists $ref_pin->{'source_package_name'}) {
			$version->isa('Cupt::Cache::BinaryVersion') or next PIN;
			my $value = $ref_pin->{'source_package_name'};
			$version->source_package_name =~ m/^$value$/ or next PIN;
		}
		if (exists $ref_pin->{'version'}) {
			my $value = $ref_pin->{'version'};
			$version->version_string =~ m/^$value$/ or next PIN;
		}
		if (exists $ref_pin->{'base_uri'}) {
			my $value = $ref_pin->{'base_uri'};

			my $found = 0;
			foreach (@available_as) {
				if ($_->{release}->{base_uri} =~ m/^$value$/) {
					$found = 1;
					last;
				}
			}
			$found or next PIN;
		}
		if (exists $ref_pin->{'release'}) {
			my @keys = keys %{$ref_pin->{'release'}};
			foreach my $key (@keys) {
				my $value = $ref_pin->{'release'}->{$key};
				my $found = 0;
				foreach (@available_as) {
					defined $_->{release}->{$key} or
							myinternaldie("unexistent key '%s' in the release entry", $key);
					if ($_->{release}->{$key} =~ m/^$value$/) {
						$found = 1;
						last;
					}
				}
				$found or next PIN;
			}
		}

		# yeah, all conditions satisfied here, and we can set less pin too here
		$result = $ref_pin->{'value'};
		last PIN;
	}

	return $result;
}

=head2 get_pin

method, returns Cupt pin value for the supplied version

Parameters:

I<version> - reference to L<Cupt::Cache::BinaryVersion|Cupt::Cache::BinaryVersion>

=cut

sub get_pin ($$) {
	my ($self, $version) = @_;
	my $result = $self->get_original_apt_pin($version);

	# discourage downgrading 
	# downgradings will usually have pin <= 0
	if (defined $self->_system_state) { # for example, for source versions will return false...
		my $installed_info = $self->_system_state->get_installed_info($version->package_name);
		if (defined $installed_info) {
			my $installed_version_string = $installed_info->{'version_string'};
			if (defined $installed_version_string) {
				if (Cupt::Core::compare_version_strings($installed_version_string, $version->version_string) > 0)
				{
					$result -= 2000;
				}

				if ($installed_info->{'want'} eq 'hold' && $version->is_installed()) {
					$result += $self->_config->get_number('cupt::cache::obey-hold');
				}
			}
		}
	}

	$result += 1 if $version->is_signed();

	return $result;
}

sub _prepare_package {
	my ($self, $ref_storage, $package_name) = @_;

	if (ref $$ref_storage->{$package_name} eq 'ARRAY') {
		# existent package and not blessed package

		# there are some version entries for this package, create it
		my @unparsed_versions = @{$$ref_storage->{$package_name}};
		my $allow_reinstall = (any { $package_name =~ m/^$_$/ } @{$self->[_allow_reinstall_offset()]}) // 0;
		$$ref_storage->{$package_name} = Cupt::Cache::Package->new(
				$self->[_binary_architecture_offset()], $allow_reinstall);
		$$ref_storage->{$package_name}->add_entry(@$_) for @unparsed_versions;
	}
	return;
}

=head2 get_binary_package

method, returns reference to appropriate L<Cupt::Cache::Package|Cupt::Cache::Package> for package name.
Returns undef if there is no such package in cache.

Parameters:

I<package_name> - package name to find

=cut

sub get_binary_package {
	my ($self, $package_name) = @_;
	# will transparently return undef if there is no such package

	$self->_prepare_package(\$self->[_binary_packages_offset()], $package_name);

	return $self->[_binary_packages_offset()]->{$package_name};
};

=head2 get_source_package

method, returns reference to appropriate L<Cupt::Cache::Package|Cupt::Cache::Package> for package name.
Returns undef if there is no such package in cache.

Parameters:

I<package_name> - package name to find

=cut

sub get_source_package {
	my ($self, $package_name) = @_;
	# will transparently return undef if there is no such package

	$self->_prepare_package(\$self->[_source_packages_offset()], $package_name);
	return $self->[_source_packages_offset()]->{$package_name};
};

=head2 get_sorted_pinned_versions

method to get sorted by "candidatness" versions in descending order

Parameters:

I<package> - reference to L<Cupt::Cache::Package|Cupt::Cache::Package>

Returns: [ { 'version' => I<version>, 'pin' => I<pin> }... ]

where:

I<version> - reference to L<Cupt::Cache::BinaryVersion|Cupt::Cache::BinaryVersion>

I<pin> - pin value

=cut

sub get_sorted_pinned_versions {
	my ($self, $package) = @_;

	my @result;

	foreach my $version (@{$package->get_versions()}) {
		push @result, { 'version' => $version, 'pin' => $self->get_pin($version) };
	}

	do {
		use sort 'stable';
		# sort in descending order, first key is pin, second is version string
		@result = sort {
			$b->{'pin'} <=> $a->{'pin'} or
			compare_versions($b->{'version'}, $a->{'version'})
		} @result;
	};

	return \@result;
}

=head2 get_policy_version

method, returns reference to
L<Cupt::Cache::BinaryVersion|Cupt::Cache::BinaryVersion>, this is the version
of I<package>, which to be installed by cupt policy; or undef if there is no
valid versions for this package

Parameters:

I<package> - reference to L<Cupt::Cache::Package|Cupt::Cache::Package>, package to select versions from

=cut

sub get_policy_version {
	my ($self, $package) = @_;

	my $ref_sorted_pinned_versions = $self->get_sorted_pinned_versions($package);
	# not assuming the package have at least valid version...
	if (scalar @$ref_sorted_pinned_versions) {
		# so, just return version with maximum "candidatness"
		return $ref_sorted_pinned_versions->[0]->{'version'};
	} else {
		return undef;
	}
}

sub _get_satisfying_versions_for_one_relation {
	my ($self, $relation) = @_;
	my $package_name = $relation->package_name;

	my @result;
	my $package = $self->get_binary_package($package_name);

	if (defined $package) {
		# if such binary package exists
		foreach my $version (@{$package->get_versions()}) {
			push @result, $version if $relation->satisfied_by($version->version_string);
		}
	}

	# virtual package can only be considered if no relation sign is specified
	if (not defined $relation->relation_string and exists $self->[_can_provide_offset()]->{$package_name}) {
		# looking for reverse-provides
		foreach (@{$self->[_can_provide_offset()]->{$package_name}}) {
			my $reverse_provide_package = $self->get_binary_package($_);
			defined $reverse_provide_package or next;
			foreach my $version (@{$reverse_provide_package->get_versions()}) {
				foreach my $provides_package_name (@{$version->provides}) {
					if ($provides_package_name eq $package_name) {
						# ok, this particular version does provide this virtual package
						push @result, $version;
					}
				}
			}
		}
	}

	return @result;
}

=head2 get_satisfying_versions

method, returns reference to array of L<Cupt::Cache::BinaryVersion|Cupt::Cache::BinaryVersion>
that satisfy relation, if no version can satisfy the relation, returns an
empty array

Parameters:

I<relation_expression> - see L<Relation expression in Cupt::Cache::Relation|Cupt::Cache::Relation/Relation expression>

=cut

sub get_satisfying_versions ($$) {
	my ($self, $relation_expression) = @_;

	my @result;
	state %cache;

	# caching results
	if ($o_memoize) {
		my $key = join(',', $self, stringify_relation_expression($relation_expression));
		if (exists $cache{$key}) {
			return $cache{$key};
		} else {
			$cache{$key} = \@result;
			# the @result itself will be filled by under lines of code so at
			# next run moment cache will contain the correct result
		}
	}

	if (ref $relation_expression ne 'ARRAY') {
		# relation expression is just one relation
		@result = ($self->_get_satisfying_versions_for_one_relation($relation_expression));
	} else {
		# otherwise it's OR group of expressions
		@result = map { $self->_get_satisfying_versions_for_one_relation($_) } @$relation_expression;
		# get rid of duplicates
		my %seen;
		@result = grep { !$seen{ $_->package_name, $_->version_string } ++ } @result;
	}

	return \@result;
}

our %_empty_release_info = (
	'version' => '',
	'description' => '',
	'signed' => 0,
	'vendor' => undef,
	'label' => '',
	'archive' => undef,
	'codename' => '-',
	'date' => undef,
	'valid-until' => undef,
	'architectures' => [],
	'base_uri' => undef,
	'not_automatic' => 0,
);

sub _get_release_info {
	my ($self, $file) = @_;

	my %release_info = %_empty_release_info;

	open(RELEASE, '<', $file) or mydie("unable to open release file '%s'", $file);
	my $field_name = undef;
	eval {
		while (<RELEASE>) {
			(($field_name, my $field_value) = ($_ =~ m/^((?:\w|-)+?): (.*)/)) # '$' implied in regexp
				or last;

			given ($field_name) {
				when ('Origin') { $release_info{vendor} = $field_value }
				when ('Label') { $release_info{label} = $field_value }
				when ('Suite') { $release_info{archive} = $field_value }
				when ('Codename') { $release_info{codename} = $field_value }
				when ('Date') { $release_info{date} = $field_value }
				when ('Valid-Until') { $release_info{valid_until} = $field_value }
				when ('NotAutomatic') { $release_info{not_automatic} = 1 }
				when ('Architectures') { $release_info{architectures} = [ split / /, $field_value ] }
				when ('Description') {
					$release_info{description} = $field_value;
					if ($field_value =~ m/([0-9][0-9a-z._-]*)/) {
						$release_info{version} = $1;
					} else {
						$release_info{version} = '';
					}
				}
			}

			undef $field_name;
		}
	};
	if (mycatch()) {
		myerr("error parsing release file '%s', line %u", $file, $.);
		myredie();
	}
	close(RELEASE) or mydie("unable to close release file '%s'", $file);

	if (!defined($release_info{vendor})) {
		mydie("no vendor specified in release file '%s'", $file);
	}
	if (!defined($release_info{archive})) {
		mydie("no archive specified in release file '%s'", $file);
	}


	$release_info{signed} = verify_signature($self->_config, $file);

	return \%release_info;
}

sub _parse_sources_lists {
	my ($self) = @_;
	my $root_prefix = $self->_config->get_string('dir');
	my $etc_dir = $self->_config->get_string('dir::etc');

	my $parts_dir = $self->_config->get_string('dir::etc::sourceparts');
	my @source_files = glob("$root_prefix$etc_dir/$parts_dir/*.list");

	my $main_file = $self->_config->get_string('dir::etc::sourcelist');
	my $main_file_path = "$root_prefix$etc_dir/$main_file";
	push @source_files, $main_file_path if -r $main_file_path;

	$self->_index_entries = [];
	foreach (@source_files) {
		push @{$self->_index_entries}, __parse_source_list($_);
	}
	return;
}

=head2 get_index_entries

method, returns reference to list of L</index_entry>'s

=cut

sub get_index_entries {
	my ($self) = @_;

	return $self->_index_entries;
}

sub __parse_source_list {
	my ($file) = @_;

	my @result;
	open(my $fd, '<', $file) or mydie("unable to open file '%s': %s", $file, $!);
	while (<$fd>) {
		chomp;
		# skip all empty lines and lines with comments
		next if m/^\s*(?:#.*)?$/;

		my %entry;
		($entry{'type'}, $entry{'uri'}, $entry{'distribution'}, my @sections) = split ' ';

		defined $entry{'type'} or
				mydie("undefined source type at file '%s', line %u", $file, $.);
		defined $entry{'uri'} or
				mydie("undefined source URI at file '%s', line %u", $file, $.);
		defined $entry{'distribution'} or
				mydie("undefined source distribution at file '%s', line %u", $file, $.);

		$entry{'uri'} =~ s{/$}{}; # strip last '/' if present

		mydie("incorrect source type at file '%s', line %u", $file, $.)
				if ($entry{'type'} ne 'deb' && $entry{'type'} ne 'deb-src');

		if (scalar @sections) {
			# this is normal entry
			foreach my $section (@sections) {
				$entry{'component'} = $section;
				push @result, { %entry };
			};
		} else {
			# this a candidate for easy entry

			# distribution must end with a slash
			($entry{'distribution'} =~ s{/$}{}) or
					mydie("distribution doesn't end with a slash at file '%s', line %u", $file, $.);

			# ok, so adding single entry
			$entry{'component'} = '';
			push @result, { %entry };
		}
	}
	close($fd) or mydie("unable to close file '%s': %s", $file, $!);

	return @result;
}

sub _parse_preference_file {
	my ($self, $file) = @_;

	# we are parsing triads like:

	# Package: perl perl-modules
	# Pin: o=debian,a=unstable
	# Pin-Priority: 800

	# Source: unetbootin
	# Pin: a=experimental
	# Pin-Priority: 1100

	open(PREF, '<', $file) or mydie("unable to open file '%s': %s", $file, $!);
	while (<PREF>) {
		chomp;
		# skip all empty lines and lines with comments
		next if m/^\s*(?:#.*)?$/;
		# skip special explanation lines, they are just comments
		next if m/^Explanation: /;

		# ok, real triad should be here
		my %pin_result;

		do { # processing first line
			m/^(Package|Source): (.*)/ or
					mydie("bad package/source line at file '%s', line %u", $file, $.);

			my $name_type = ($1 eq 'Package' ? 'package_name' : 'source_package_name');
			my $name_value = $2;
			glob_to_regex($name_value);
			$name_value =~ s'\\ '|'g; # converting space separators into regex's OR

			$pin_result{$name_type} = $name_value;
		};

		do { # processing second line
			my $pin_line = <PREF>;
			defined($pin_line) or
					mydie("no pin line at file '%s' line %u", $file, $.);

			$pin_line =~ m/^Pin: (\w+?) (.*)/ or
					mydie("bad pin line at file '%s' line %u", $file, $.);

			my $pin_type = $1;
			my $pin_expression = $2;
			given ($pin_type) {
				when ('release') {
					my @conditions = split /\s*,\s*/, $pin_expression;
					scalar @conditions or
							mydie("bad release expression at file '%s' line %u", $file, $.);

					foreach (@conditions) {
						m/^(\w)=(.*)/ or
								mydie("bad condition in release expression at file '%s' line %u", $file, $.);

						my $condition_type = $1;
						my $condition_value = $2;
						given ($condition_type) {
							when ('a') { $pin_result{'release'}->{'archive'} = $condition_value; }
							when ('v') { $pin_result{'release'}->{'version'} = $condition_value; }
							when ('c') { $pin_result{'release'}->{'component'} = $condition_value; }
							when ('n') { $pin_result{'release'}->{'codename'} = $condition_value; }
							when ('o') { $pin_result{'release'}->{'vendor'} = $condition_value; }
							when ('l') { $pin_result{'release'}->{'label'} = $condition_value; }
							default {
								mydie("bad condition type (should be one of 'a', 'v', 'c', 'n', 'o', 'l') " .
										"in release expression at file '%s' line %u", $file, $.);
							}
						}
					}
				}
				when ('version') {
					glob_to_regex($pin_expression);
					$pin_result{'version'} = $pin_expression;
				}
				when ('origin') { # this is 'base_uri', really...
					$pin_result{'base_uri'} = $pin_expression;
				}
				default {
					mydie("bad pin type (should be one of 'release', 'version', 'origin') " .
							"at file '%s' line %u", $file, $.);
				}
			}
		};

		do { # processing third line
			my $priority_line = <PREF>;
			defined($priority_line) or
					mydie("no priority line at file '%s' line %u", $file, $.);

			$priority_line =~ m/^Pin-Priority: ([+-]?\d+)/ or
					mydie("bad priority line at file '%s' line %u", $file, $.);

			my $priority = $1;
			$pin_result{'value'} = $priority;
		};

		# adding to storage
		push @{$self->_pin_settings}, \%pin_result;
	}

	close(PREF) or mydie("unable to close file '%s': %s", $file, $!);
	return;
}

sub _parse_extended_states {
	my ($self, $file) = @_;

	# we are parsing duals like:

	# Package: perl
	# Auto-Installed: 1

	# but, rarely another fields may be present, we need to ignore them

	eval {
		my $package_name;
		my $value;

		local $/ = "\n\n";

		open(my $fd, '<', $file) or mydie("unable to open file '%s': %s", $file, $!);
		while (<$fd>) {
			m/^\s*(?:#.*|)$/ and next; # skip empty/comment lines

			m'^Package: (.*?)$.*?^Auto-Installed: (.*?)$'sm or
					mydie("bad chunk '%s' at file '%s'", $_, $file);
			chomp;

			$package_name = $1;
			$value = $2;

			if ($value ne '0' and $value ne '1') {
				mydie("bad value '%s' (should be 0 or 1) in chunk '%s' at file '%s'",
						$value, $_, $file);
			}

			if ($value) {
				# adding to storage
				$self->[_extended_info_offset()]->{'automatically_installed'}->{$package_name} = $value;
			}
		}
		close($fd) or mydie("unable to close file '%s': %s", $file, $!);
	};
	if (mycatch()) {
		myerr('error while parsing extended states');
		myredie();
	}
	return;
}

sub _process_provides_subline {
	my ($self, $package_name, $provides_subline) = @_;

	my @provides = split /\s*,\s*/, $provides_subline;
	foreach (@provides) {
		# if this entry is new one?
		if (none { $_ eq $package_name } @{$self->[_can_provide_offset()]->{$_}}) {
			push @{$self->[_can_provide_offset()]->{$_}}, $package_name ;
		}
	}
	return;
}

sub _process_index_file {
	my ($self, $file, $translation_file, $type, $ref_release_info) = @_;

	my $version_class;
	my $ref_packages_storage;
	if ($type eq 'deb') {
		$version_class = 'Cupt::Cache::BinaryVersion';
		$ref_packages_storage = \$self->_binary_packages;
	} elsif ($type eq 'deb-src') {
		$version_class = 'Cupt::Cache::SourceVersion';
		$ref_packages_storage = \$self->_source_packages;
	}


	my $ref_translations;
	if (defined $translation_file) {
		$ref_translations = __process_translation_file($translation_file);
	} else {
		$ref_translations = {};
	}

	my $fh;
	open($fh, '<', $file) or mydie("unable to open index file '%s': %s", $file, $!);

	eval {
		local $/ = "\n\n";
		while (<$fh>) {
			my $offset = tell($fh) - length($_);
			my ($package_name) = m/^Package: (.*?)$/m;

			defined $package_name or
					mydie("unable to find package name");

			# we skips 'Package: <...>' line additionally
			$offset += length('Package: ') + length($package_name) + 1;

			# check it for correctness
			($package_name =~ m/^$package_name_regex$/)
				or mydie("bad package name '%s'", $package_name);

			my @version_params = ($version_class, $package_name, $fh, $offset, $ref_release_info);
			if (exists $ref_translations->{$package_name}) {
				push @version_params, $ref_translations->{$package_name}->{'filehandle'};
				push @version_params, $ref_translations->{$package_name}->{'offset'};
			}

			push @{$$ref_packages_storage->{$package_name}}, \@version_params;

			if (m/^Provides: (.+?)$/m) {
				$self->_process_provides_subline($package_name, $1);
			}
		}
	};
	if (mycatch()) {
		myerr("error parsing index file '%s'", $file);
		myredie();
	}

	return;
}

sub __process_translation_file {
	my ($file) = @_;

	my %result;

	my $fh;
	open($fh, '<', $file) or mydie("unable to open translation file '%s': %s", $file, $!);

	eval {
		local $/ = "\n\n";
		while (<$fh>) {
			my $offset = tell($fh) - length($_);
			my ($package_name) = m/^Package: (.*?)$/m;

			# we skips 'Package: <...>' line additionally
			$offset += length('Package: ') + length($package_name) + 1;

			# check it for correctness
			($package_name =~ m/^$package_name_regex$/)
				or mydie("bad package name '%s'", $package_name);

			@{$result{$package_name}}{'offset', 'filehandle'} = ($offset, $fh);
		}
	};
	if (mycatch()) {
		myerr("error parsing translation file '%s'", $file);
		myredie();
	}

	return \%result;
}

sub _path_of_base_uri {
	my ($self, $index_entry) = @_;

	# "http://ftp.ua.debian.org" -> "ftp.ua.debian.org"
	# "file:/home/jackyf" -> "/home/jackyf"
	(my $uri_prefix = $index_entry->{'uri'}) =~ s{^\w+:(?://)?}{};

	# "escaping" tilde, following APT practice :(
	$uri_prefix =~ s/~/%7e/g;

	# "ftp.ua.debian.org/debian" -> "ftp.ua.debian.org_debian"
	$uri_prefix =~ tr[/][_];

	my $dirname = join('',
		$self->_config->get_string('dir'),
		$self->_config->get_string('dir::state'),
		'/',
		$self->_config->get_string('dir::state::lists')
	);

	(my $distribution_part = $index_entry->{'distribution'}) =~ tr[/][_];
	my $base_uri_part;
    if ($index_entry->{'component'} eq '') {
		# easy source type
		$base_uri_part = join('_', $uri_prefix, $distribution_part);
	} else {
		# normal source type
		$base_uri_part = join('_', $uri_prefix, 'dists', $distribution_part);
	}

	return join('', $dirname, '/', $base_uri_part);
}

sub _base_download_uri {
	my ($self, $index_entry) = @_;

    if ($index_entry->{'component'} eq '') {
		# easy source type
		return join('/', $index_entry->{'uri'}, $index_entry->{'distribution'});
	} else {
		# normal source type
		return join('/', $index_entry->{'uri'}, 'dists', $index_entry->{'distribution'});
	}
}

sub _index_list_suffix {
	my ($self, $index_entry, $delimiter) = @_;

	my $arch = $self->_config->get_string('apt::architecture');

	if ($index_entry->{'component'} eq '') {
		# easy source type
		return ($index_entry->{'type'} eq 'deb') ? 'Packages' : 'Sources';
	} else {
		# normal source type
		return ($index_entry->{'type'} eq 'deb') ?
				"binary-${arch}${delimiter}Packages" : "source${delimiter}Sources";
	}
}

=head2 get_path_of_index_list

method, returns path of Packages/Sources file for I<index_entry>

Parameters:

L</index_entry>

=cut

sub get_path_of_index_list {
	my ($self, $index_entry) = @_;

	my $base_uri = $self->_path_of_base_uri($index_entry);
	my $index_list_suffix = $self->_index_list_suffix($index_entry, '_');

	my $filename = join('_', $base_uri, $index_entry->{'component'}, $index_list_suffix);
	$filename =~ s/__/_/; # if component is empty
	return $filename;
}

sub _get_chunks_of_localized_descriptions {
	my ($self, $index_entry) = @_;

	my @result;

	return @result if $index_entry->{'type'} ne 'deb';

	my $translation_variable = $self->_config->get_string('apt::acquire::translation');
	my $locale = $translation_variable eq 'environment' ?
			POSIX::setlocale(LC_MESSAGES) : $translation_variable;
	return @result if $locale eq 'none';

	my @chunks;
	if ($index_entry->{'component'} ne '') {
		push @chunks, $index_entry->{'component'};
	}
	push @chunks, 'i18n';

	# cutting out an encoding
	$locale =~ s/\..*//;
	push @result, [ @chunks, "Translation-$locale" ];

	# cutting out an country specificator
	if ($locale =~ s/_.*//) {;
		push @result, [ @chunks, "Translation-$locale" ];
	}

	return @result;
}

sub _get_paths_of_localized_descriptions {
	my ($self, $index_entry) = @_;

	my @chunk_arrays = $self->_get_chunks_of_localized_descriptions($index_entry);
	my $path_of_base_uri = $self->_path_of_base_uri($index_entry);

	return map { join('_', $path_of_base_uri, @$_) } @chunk_arrays;
}

=head2 get_download_entries_of_index_list

method, returns the download entries of Packages/Sources file for I<index_entry>

Parameters:

L</index_entry>

path to accompanying Release file

Returns:

[ I<download_entry>... ]

where

I<download_entry> is

  {
    'uri' => {
               'size' => file size
               'md5sum' => MD5 hash sum
               'sha1sum' => SHA1 hash sum
               'sha256sum' => SHA256 hash sum
             }
  }

=cut

sub get_download_entries_of_index_list {
	my ($self, $index_entry, $path_to_release_file) = @_;

	my $base_download_uri = $self->_base_download_uri($index_entry);
	my $index_list_suffix = $self->_index_list_suffix($index_entry, '/');
	my $full_index_list_suffix = join('/', $index_entry->{'component'}, $index_list_suffix);
	$full_index_list_suffix =~ s{^/}{}; # if component is empty

	open(my $release_file_handle, '<', $path_to_release_file) or
			mydie("unable to open file '%s': %s", $path_to_release_file, $!);
	my @release_lines = <$release_file_handle>;
	close($release_file_handle) or
			mydie("unable to close file '%s': %s'", $path_to_release_file);

	my %result;

	my $current_hash_sum_name;
	# now we need to find if this variant is present in the release file
	foreach (@release_lines) {
		if (m/^MD5/) {
			$current_hash_sum_name = 'md5sum';
		} elsif (m/^SHA1/) {
			$current_hash_sum_name = 'sha1sum';
		} elsif (m/^SHA256/) {
			$current_hash_sum_name = 'sha256sum';
		} elsif (m/$full_index_list_suffix/) {
			my $release_line = $_;
			defined $current_hash_sum_name or
					mydie("release line '%s' without previous hash sum declaration at file '%s'",
							$release_line, $path_to_release_file);
			my ($hash_sum, $size, $name) = ($release_line =~ m/^\s([[:xdigit:]]+)\s+(\d+)\s+(.*)$/) or
					mydie("malformed release line '%s' at file '%s'", $release_line, $path_to_release_file);
			$name =~ m/^$full_index_list_suffix/ or next;
			# skipping diffs for now...
			$name !~ m/^$full_index_list_suffix.diff/ or next;
			my $uri = join('/', $base_download_uri, $name);
			$result{$uri}->{'size'} = $size;
			$result{$uri}->{$current_hash_sum_name} = $hash_sum;
		}
	}

	# checks
	foreach my $uri (keys %result) {
		my $ref_download_entry = $result{$uri};
		if (!are_hash_sums_present($ref_download_entry)) {
			mydie("no hash sums defined for URI '%s'", $uri);
		}
	}

	return \%result;
}

=head2 get_path_of_release_list

method, returns path of Release file for I<index_entry>

Parameters:

L</index_entry>

=cut

sub get_path_of_release_list {
	my ($self, $index_entry) = @_;

	return join('_', $self->_path_of_base_uri($index_entry), 'Release');
}

=head2 get_download_uri_of_release_list

method, returns the remote URI of Release file for I<index_entry>

Parameters:

L</index_entry>

=cut

sub get_download_uri_of_release_list {
	my ($self, $index_entry) = @_;

	return join('/', $self->_base_download_uri($index_entry), 'Release');
}

=head2 get_download_entries_of_localized_descriptions

method, returns the remote URIs and corresponding download places of possible
Translation files for I<index_entry>

Parameters:

L</index_entry>

Returns:

[ [ I<download URI>, I<local path to place> ]... ]

=cut

sub get_download_entries_of_localized_descriptions {
	my ($self, $index_entry) = @_;

	my @chunk_arrays = $self->_get_chunks_of_localized_descriptions($index_entry);
	my $base_download_uri = $self->_base_download_uri($index_entry);
	my $path_of_base_uri = $self->_path_of_base_uri($index_entry);

	my @result;
	foreach my $ref_chunks (@chunk_arrays) {
		# yes, somewhy translations are always bzip2'ed
		my $download_uri = join('/', $base_download_uri, @$ref_chunks) . '.bz2';
		my $local_path = join('_', $path_of_base_uri, @$ref_chunks);
		push @result, [ $download_uri, $local_path ];
	}
	return \@result;
}

sub _parse_preferences {
	my ($self) = @_;

	my $root_prefix = $self->_config->get_string('dir');
	my $etc_dir = $self->_config->get_string('dir::etc');

	my $parts_dir = $self->_config->get_string('dir::etc::preferencesparts');
	my @preference_files = glob("$root_prefix$etc_dir/$parts_dir/*");

	@preference_files = grep {
		(my $name = $_) =~ s{.*/}{}; # cutting directory name
		$name =~ m/^[A-Za-z0-9_.-]+$/
			and
		($name !~ m/\./ or $name =~ m/\.pref$/) # an extension, if exist, should be 'pref'
	} @preference_files;

	my $main_file = $self->_config->get_string('dir::etc::preferences');
	my $main_file_path = "$root_prefix$etc_dir/$main_file";
	push @preference_files, $main_file_path if -r $main_file_path;

	foreach (@preference_files) {
		$self->_parse_preference_file($_);
	}
	return;
}

=head2 get_path_of_extended_states

returns path of file containing extended states for packages

=cut

sub get_path_of_extended_states {
	my ($self) = @_;

	my $root_prefix = $self->_config->get_string('dir');
	my $etc_dir = $self->_config->get_string('dir::state');

	my $leaf = $self->_config->get_string('dir::state::extendedstates');

	return "$root_prefix$etc_dir/$leaf";
}

=head2 get_binary_release_data

method, returns reference to array of available releases of binary packages in
form [ L</release_info> ... ]

=cut

sub get_binary_release_data ($) {
	my ($self) = @_;
	return $self->_release_data->{binary};
}

=head2 get_source_release_data

method, returns reference to array of available releases of source packages in
form [ L</release_info> ... ]

=cut

sub get_source_release_data ($) {
	my ($self) = @_;
	return $self->_release_data->{source};
}

=head1 FREE SUBROUTINES

=head2 verify_signature

Checks signature of supplied file via GPG.

Parameters:

I<config> - reference to L<Cupt::Config|Cupt::Config>

I<file> - path to file

Returns:

non-zero on success, zero on fail

=cut

sub verify_signature ($$) {
	my ($config, $file) = @_;

	my $debug = $config->get_bool('debug::gpgv');

	mydebug("verifying file '%s'", $file) if $debug;

	my $keyring_file = $config->get_string('gpgv::trustedkeyring');
	mydebug("keyring file is '%s'", $keyring_file) if $debug;

	my $signature_file = "$file.gpg";
	mydebug("signature file is '%s'", $signature_file) if $debug;

	-r $signature_file or
			do {
				mydebug("unable to read signature file '%s'", $signature_file) if $debug;
				return 0;
			};

	my $verify_result;
	eval {
		-e $keyring_file or
				mydie("keyring file '%s' doesn't exist", $keyring_file);
		-r $keyring_file or
				mydie("no read rights on keyring file '%s', please do 'chmod +r %s' with root rights",
							$keyring_file, $keyring_file);

		open(GPG_VERIFY, 'gpgv --status-fd 1 ' .
				"--keyring $keyring_file $signature_file $file 2>/dev/null |") or
				mydie('unable to open gpg pipe: %s', $!);
		my $sub_gpg_readline = sub {
			my $result;
			do {
				$result = readline(GPG_VERIFY);
				if (defined $result) {
					chomp $result;
					mydebug("fetched '%s' from gpg pipe", $result) if $debug;
				}
			} while (defined $result and (($result =~ m/^\[GNUPG:\] SIG_ID/) or ($result !~ m/^\[GNUPG:\]/)));

			if (!defined $result) {
				return undef;
			} else {
				$result =~ s/^\[GNUPG:\] //;
				return $result;
			}
		};

		my $status_string = $sub_gpg_readline->();
		if (not defined $status_string) {
			# no info from gpg at all
			mydie("gpg: '%s': no info received", $file);
		}

		# first line ought to be validness indicator
		my ($message_type, $message) = ($status_string =~ m/(\w+) (.*)/);
		if (not defined $message_type or not defined $message) {
			mydie("gpg: '%s': invalid status string '%s'", $file, $status_string);
		}

		given ($message_type) {
			when ('GOODSIG') {
				my $further_info = $sub_gpg_readline->();
				defined $further_info or
						mydie("gpg: '%s': error: unfinished status", $file);

				my ($check_result_type, $check_message) = ($further_info =~ m/(\w+) (.*)/);
				if (not defined $check_result_type or not defined $check_message) {
					mydie("gpg: '%s': invalid further info string '%s'", $file, $further_info);
				}
				given ($check_result_type) {
					when ('VALIDSIG') {
						# no comments :)
						$verify_result = 1;
					}
					when ('EXPSIG') {
						$verify_result = 0;
						mywarn("gpg: '%s': expired signature: %s", $file, $check_message);
					}
					when ('EXPKEYSIG') {
						$verify_result = 0;
						mywarn("gpg: '%s': expired key: %s", $file, $check_message);
					}
					when ('REVKEYSIG') {
						$verify_result = 0;
						mywarn("gpg: '%s': revoked key: %s", $file, $check_message);
					}
					default {
						mywarn("gpg: '%s': unknown error: %s %s", $file, $check_result_type, $check_message);
					}
				}
			}
			when ('BADSIG') {
				mywarn("gpg: '%s': bad signature: %s", $file, $message);
				$verify_result = 0;
			}
			when ('ERRSIG') {
				# gpg was not able to verify signature
				$verify_result = 0;

				# maybe, public key was not found?
				my $public_key_was_not_found = 0;
				my $detail_string = $sub_gpg_readline->();
				if (defined $detail_string) {
					my ($detail_type, $detail_message) = ($detail_string =~ m/(\w+) (.*)/);
					if (not defined $detail_type or not defined $detail_message) {
						mydie("gpg: '%s': invalid detailed info string '%s'", $file, $detail_string);
					}
					if ($detail_type eq 'NO_PUBKEY') {
						$public_key_was_not_found = 1;

						# the message looks like
						#
						# NO_PUBKEY D4F5CE00FA0E9B9D
						#
						# the best guess for getting the public key is second 8 hex digits within this string
						my $public_key_number_detected = 0;
						if (length($detail_message) == 16) {
							my $public_key_number = substr($detail_message, 8, 8);
							if ($public_key_number =~ m/^[0-9A-Z]{8}$/) {
								$public_key_number_detected = 1;
								mywarn("gpg: '%s': public key '%s' not found", $file, $public_key_number);
							}
						}

						if (not $public_key_number_detected) {
							mywarn("gpg: '%s': public key not found", $file);
						}
					}
				}

				if (not $public_key_was_not_found) {
					mywarn("gpg: '%s': could not verify signature: %s", $file, $message);
				}
			}
			when ('NODATA') {
				# no signature
				mywarn("gpg: '%s': empty signature", $file);
				$verify_result = 0;
			}
			when ('KEYEXPIRED') {
				$verify_result = 0;
				mywarn("gpg: '%s': expired key: %s", $file, $message);
			}
			default {
				mywarn("gpg: '%s': unknown message received: %s %s", $file, $message_type, $message);
				$verify_result = 0;
			}
		}

		close(GPG_VERIFY) or $! == 0 or
				mydie('unable to close gpg pipe: %s', $!);
	};
	if (mycatch()) {
		mywarn("error while verifying signature for file '%s'", $file);
		$verify_result = 0;
	}

	mydebug('the verify result is %u', $verify_result) if $debug;
	return $verify_result;
}

=head2 verify_hash_sums

verifies MD5, SHA1 and SHA256 hash sums of file

Parameters:

I<hash sums> - { 'md5sum' => $md5sum, 'sha1sum' => $sha1sum', 'sha256sum' => $sha256sum }

You should specify at least one hash sum in this parameter.

I<path> - path to file

Returns: zero on failure, non-zero on success

=cut

sub verify_hash_sums ($$) {
	my ($ref_hash_sums, $path) = @_;

	my @checks = (
		[ $ref_hash_sums->{'md5sum'}, 'MD5' ],
		[ $ref_hash_sums->{'sha1sum'}, 'SHA-1' ],
		[ $ref_hash_sums->{'sha256sum'}, 'SHA-256' ],
	);

	open(FILE, '<', $path) or
			mydie("unable to open file '%s': %s", $path, $!);
	binmode(FILE) or
			mydie("unable to set binary mode for file '%s': %s", $path, $!);

	my $sums_count = 0;

	foreach (@checks) {
		my $expected_result = $_->[0];
		defined $expected_result or next;
		++$sums_count;
		my $hash_type = $_->[1];
		my $hasher = Digest->new($hash_type);
		seek(FILE, 0, SEEK_SET) or
				mydie("unable to seek on file '%s': %s", $path, $!);
		$hasher->addfile(*FILE);
		my $computed_sum = $hasher->hexdigest();
		return 0 if ($computed_sum ne $expected_result);
	}

	close(FILE) or
			mydie("unable to close file '%s': %s", $path, $!);

	$sums_count or mydie('no hash sums specified');
	return 1;
}

=head2 get_path_of_debian_changelog

free subroutine, returns string path of Debian changelog for version when
version is installed, undef otherwise

Parameters:

I<version> - reference to
L<Cupt::Cache::BinaryVersion|Cupt::Cache::BinaryVersion>

=cut

sub get_path_of_debian_changelog ($) {
	my ($version) = @_;

	return undef if not $version->is_installed();

	my $package_name = $version->package_name;
	my $common_part = "/usr/share/doc/$package_name/";
	if (is_version_string_native($version->version_string)) {
		return $common_part . 'changelog.gz';
	} else {
		return $common_part . 'changelog.Debian.gz';
	}
}

=head2 get_path_of_debian_copyright

free subroutine, returns string path of Debian copyright for version when
version is installed, undef otherwise

Parameters:

I<version> - reference to
L<Cupt::Cache::BinaryVersion|Cupt::Cache::BinaryVersion>

=cut

sub get_path_of_debian_copyright ($) {
	my ($version) = @_;

	return undef if not $version->is_installed();

	my $package_name = $version->package_name;
	return "/usr/share/doc/$package_name/copyright";
}

=head1 DATA SPECIFICATION

=head2 release_info

This is a hash:

  {
    'signed' => boolean, whether release signed or not
    'version' => version of released distribution (can be undef)
    'description' => description string
    'vendor' => vendor string
    'label' => label string
    'archive' => archive name string
    'codename' => codename string
    'date' => date of release (can be undef)
    'valid-until' => time string when to forget about this release
    'architectures' => reference to array of available architectures
    'base_uri' => base URI (origin), empty string in case of "installed" distribution
	'not_automatic' => true if packages from this release shouldn't be considered for automatic upgrades
  }

=head2 index_entry

This is a hash:
  
  {
    'type' => { 'deb' | 'deb-src' }
    'uri' => URI string
    'distribution' => distribution path
    'component' => component string
  }

=cut

1;

