#include <arpa/inet.h>
#include <assert.h>
#include <ctype.h>
#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <net/if.h>
#include <sys/socket.h>
#include <string>
#include <utility>
#include <unordered_map>
#include <vector>

#include "tlse.h"

#include "acceptor.h"
#include "config.h"
#include "input.h"
#include "log.h"
#include "parse.h"

using namespace std;

#define DEFAULT_BACKLOG_SIZE 10485760

struct ConfigLine {
	string keyword;
	vector<string> arguments;
	unordered_map<string, string> parameters;
};

namespace {

bool parse_hostport(const string &hostport, sockaddr_in6 *addr)
{
	memset(addr, 0, sizeof(*addr));
	addr->sin6_family = AF_INET6;

	string port_string;

	// See if the argument if on the type [ipv6addr]:port.
	if (!hostport.empty() && hostport[0] == '[') {
		size_t split = hostport.find("]:");
		if (split == string::npos) {
			log(ERROR, "address '%s' is malformed; must be either [ipv6addr]:port or ipv4addr:port");
			return false;
		}

		string host(hostport.begin() + 1, hostport.begin() + split);
		port_string = hostport.substr(split + 2);

		if (inet_pton(AF_INET6, host.c_str(), &addr->sin6_addr) != 1) {
			log(ERROR, "'%s' is not a valid IPv6 address");
			return false;
		}
	} else {
		// OK, then it must be ipv4addr:port.
		size_t split = hostport.find(":");
		if (split == string::npos) {
			log(ERROR, "address '%s' is malformed; must be either [ipv6addr]:port or ipv4addr:port");
			return false;
		}

		string host(hostport.begin(), hostport.begin() + split);
		port_string = hostport.substr(split + 1);

		// Parse to an IPv4 address, then construct a mapped-v4 address from that.
		in_addr addr4;

		if (inet_pton(AF_INET, host.c_str(), &addr4) != 1) {
			log(ERROR, "'%s' is not a valid IPv4 address");
			return false;
		}

		addr->sin6_addr.s6_addr32[2] = htonl(0xffff);
		addr->sin6_addr.s6_addr32[3] = addr4.s_addr;
	}

	int port = atoi(port_string.c_str());
	if (port < 1 || port >= 65536) {
		log(ERROR, "port %d is out of range (must be [1,65536>).", port);
		return false;
	}
	addr->sin6_port = ntohs(port);

	return true;
}

bool read_config(const string &filename, vector<ConfigLine> *lines)
{
	FILE *fp = fopen(filename.c_str(), "r");
	if (fp == nullptr) {
		log_perror(filename.c_str());
		return false;
	}

	char buf[4096];
	while (!feof(fp)) {
		if (fgets(buf, sizeof(buf), fp) == nullptr) {
			break;
		}

		// Chop off the string at the first #, \r or \n.
		buf[strcspn(buf, "#\r\n")] = 0;

		// Remove all whitespace from the end of the string.
		size_t len = strlen(buf);
		while (len > 0 && isspace(buf[len - 1])) {
			buf[--len] = 0;
		}

		// If the line is now all blank, ignore it.
		if (len == 0) {
			continue;
		}

		vector<string> tokens = split_tokens(buf);
		assert(!tokens.empty());
		
		ConfigLine line;
		line.keyword = tokens[0];

		for (size_t i = 1; i < tokens.size(); ++i) {
			// foo=bar is a parameter; anything else is an argument.
			size_t equals_pos = tokens[i].find_first_of('=');
			if (equals_pos == string::npos) {
				line.arguments.push_back(tokens[i]);
			} else {
				string key = tokens[i].substr(0, equals_pos);
				string value = tokens[i].substr(equals_pos + 1, string::npos);
				line.parameters.insert(make_pair(key, value));
			}
		}

		lines->push_back(line);
	}

	if (fclose(fp) == EOF) {
		log_perror(filename.c_str());
		return false;
	}
	return true;
}

bool fetch_config_string(const vector<ConfigLine> &config, const string &keyword, string *value)
{
	for (const ConfigLine &line : config) {
		if (line.keyword != keyword) {
			continue;
		}
		if (line.parameters.size() > 0 ||
		    line.arguments.size() != 1) {
			log(ERROR, "'%s' takes one argument and no parameters", keyword.c_str());
			return false;
		}
		*value = line.arguments[0];
		return true;
	}
	return false;
}

bool fetch_config_int(const vector<ConfigLine> &config, const string &keyword, int *value)
{
	for (const ConfigLine &line : config) {
		if (line.keyword != keyword) {
			continue;
		}
		if (line.parameters.size() > 0 ||
		    line.arguments.size() != 1) {
			log(ERROR, "'%s' takes one argument and no parameters", keyword.c_str());
			return false;
		}
		*value = atoi(line.arguments[0].c_str());  // TODO: verify int validity.
		return true;
	}
	return false;
}

bool load_file_to_string(const string &filename, size_t max_size, string *contents)
{
	contents->clear();

	FILE *fp = fopen(filename.c_str(), "r");
	if (fp == nullptr) {
		log_perror(filename.c_str());
		return false;
	}

	char buf[4096];
	while (!feof(fp)) {
		size_t ret = fread(buf, 1, sizeof(buf), fp);
		if (ret > 0) {
			contents->append(buf, buf + ret);
		} else {
			if (ferror(fp)) {
				log_perror(filename.c_str());
				fclose(fp);
				return false;
			}
			assert(feof(fp));
			break;
		}

		if (contents->size() > max_size) {
			log(ERROR, "%s was longer than the maximum allowed %zu bytes", filename.c_str(), max_size);
			fclose(fp);
			return false;
		}
	}
	fclose(fp);
	return true;
}

bool parse_tls_parameters(const unordered_map<string, string> &parameters, AcceptorConfig *acceptor)
{
	bool has_cert = false, has_key = false;

	auto tls_cert_it = parameters.find("tls_cert");
	if (tls_cert_it != parameters.end()) {
		if (!load_file_to_string(tls_cert_it->second, 1048576, &acceptor->certificate_chain)) {
			return false;
		}

		// Verify that the certificate is valid.
		bool is_server = true;
		TLSContext *server_context = tls_create_context(is_server, TLS_V12);
		int num_cert = tls_load_certificates(
			server_context,
			reinterpret_cast<const unsigned char *>(acceptor->certificate_chain.data()),
			acceptor->certificate_chain.size());
		if (num_cert < 0) {
			log_tls_error(tls_cert_it->second.c_str(), num_cert);
			tls_destroy_context(server_context);
			return false;
		} else if (num_cert == 0) {
			log(ERROR, "%s did not contain any certificates", tls_cert_it->second.c_str());
			return false;
		}
		tls_destroy_context(server_context);
		has_cert = true;
	}

	auto tls_key_it = parameters.find("tls_key");
	if (tls_key_it != parameters.end()) {
		if (!load_file_to_string(tls_key_it->second, 1048576, &acceptor->private_key)) {
			return false;
		}

		// Verify that the key is valid.
		bool is_server = true;
		TLSContext *server_context = tls_create_context(is_server, TLS_V12);
		int num_keys = tls_load_private_key(
			server_context,
			reinterpret_cast<const unsigned char *>(acceptor->private_key.data()),
			acceptor->private_key.size());
		if (num_keys < 0) {
			log_tls_error(tls_key_it->second.c_str(), num_keys);
			tls_destroy_context(server_context);
			return false;
		} else if (num_keys == 0) {
			log(ERROR, "%s did not contain any private keys", tls_key_it->second.c_str());
			return false;
		}
		tls_destroy_context(server_context);
		has_key = true;
	}

	if (has_cert != has_key) {
		log(ERROR, "Only one of tls_cert= and tls_key= was given, needs zero or both");
		return false;
	}

	return true;
}


bool parse_port(const ConfigLine &line, Config *config)
{
	if (line.arguments.size() != 1) {
		log(ERROR, "'port' takes exactly one argument");
		return false;
	}

	int port = atoi(line.arguments[0].c_str());
	if (port < 1 || port >= 65536) {
		log(ERROR, "port %d is out of range (must be [1,65536>).", port);
		return false;
	}

	AcceptorConfig acceptor;
	acceptor.addr = create_any_address(port);

	if (!parse_tls_parameters(line.parameters, &acceptor)) {
		return false;
	}
	config->acceptors.push_back(acceptor);
	return true;
}

bool parse_listen(const ConfigLine &line, Config *config)
{
	if (line.arguments.size() != 1) {
		log(ERROR, "'listen' takes exactly one argument");
		return false;
	}

	AcceptorConfig acceptor;
	if (!parse_hostport(line.arguments[0], &acceptor.addr)) {
		return false;
	}
	if (!parse_tls_parameters(line.parameters, &acceptor)) {
		return false;
	}
	config->acceptors.push_back(acceptor);
	return true;
}

bool parse_stream(const ConfigLine &line, Config *config)
{
	if (line.arguments.size() != 1) {
		log(ERROR, "'stream' takes exactly one argument");
		return false;
	}

	StreamConfig stream;
	stream.url = line.arguments[0];

	const auto src_it = line.parameters.find("src");
	bool input_is_udp = false;
	if (src_it == line.parameters.end()) {
		log(WARNING, "stream '%s' has no src= attribute, clients will not get any data.",
		        stream.url.c_str());
	} else {
		stream.src = src_it->second;

		string protocol, user, host, port, path;
		if (!parse_url(stream.src, &protocol, &user, &host, &port, &path)) {
			log(ERROR, "could not parse URL '%s'", stream.src.c_str());
			return false;
		}
		if (protocol == "udp") {
			input_is_udp = true;
		}
	}

	const auto backlog_it = line.parameters.find("backlog_size");
	if (backlog_it == line.parameters.end()) {
		stream.backlog_size = DEFAULT_BACKLOG_SIZE;
	} else {
		stream.backlog_size = atoi(backlog_it->second.c_str());
	}

	const auto prebuffer_it = line.parameters.find("force_prebuffer");
	if (prebuffer_it == line.parameters.end()) {
		stream.prebuffering_bytes = 0;
	} else {
		stream.prebuffering_bytes = atoi(prebuffer_it->second.c_str());
	}

	// Parse output encoding.
	const auto encoding_parm_it = line.parameters.find("encoding");
	if (encoding_parm_it == line.parameters.end() ||
	    encoding_parm_it->second == "raw") {
		stream.encoding = StreamConfig::STREAM_ENCODING_RAW;
	} else if (encoding_parm_it->second == "metacube") {
		stream.encoding = StreamConfig::STREAM_ENCODING_METACUBE;
	} else {
		log(ERROR, "Parameter 'encoding' must be either 'raw' (default) or 'metacube'");
		return false;
	}

	// Parse input encoding.
	const auto src_encoding_parm_it = line.parameters.find("src_encoding");
	if (src_encoding_parm_it == line.parameters.end()) {
		stream.src_encoding = input_is_udp ? StreamConfig::STREAM_ENCODING_RAW : StreamConfig::STREAM_ENCODING_METACUBE;
	} else if (src_encoding_parm_it->second == "metacube") {
		if (input_is_udp) {
			log(ERROR, "UDP streams cannot have Metacube input");
			return false;
		}
		stream.src_encoding = StreamConfig::STREAM_ENCODING_METACUBE;
	} else if (src_encoding_parm_it->second == "raw") {
		stream.src_encoding = StreamConfig::STREAM_ENCODING_RAW;
	} else {
		log(ERROR, "Parameter 'src_encoding' must be either 'raw' (default for UDP) or 'metacube' (default for HTTP)");
		return false;
	}

	// Parse the pacing rate, converting from kilobits to bytes as needed.
	const auto pacing_rate_it = line.parameters.find("pacing_rate_kbit");
	if (pacing_rate_it == line.parameters.end()) {
		stream.pacing_rate = ~0U;
	} else {
		stream.pacing_rate = atoi(pacing_rate_it->second.c_str()) * 1024 / 8;
	}

	// Parse the HLS URL, if any.
	const auto hls_url_it = line.parameters.find("hls_playlist");
	if (hls_url_it != line.parameters.end()) {
		stream.hls_url = hls_url_it->second;
		if (stream.hls_url.empty()) {
			log(ERROR, "Parameter 'hls_playlist' was given but empty");
			return false;
		}
		if (stream.encoding == StreamConfig::STREAM_ENCODING_METACUBE) {
			log(ERROR, "HLS cannot be used with Metacube output");
			return false;
		}
	}

	// Parse the HLS fragment duration, if any.
	const auto hls_frag_duration_it = line.parameters.find("hls_frag_duration");
	if (hls_frag_duration_it != line.parameters.end()) {
		if (stream.hls_url.empty()) {
			log(ERROR, "Parameter 'hls_frag_duration' given, but no 'hls_playlist' given");
			return false;
		}
		stream.hls_frag_duration = stoi(hls_frag_duration_it->second);
		if (stream.hls_frag_duration <= 0) {
			log(ERROR, "'hls_frag_duration' must be a strictly positive integer");
			return false;
		}
	}

	// Parse the HLS backlog margin, if any.
	const auto hls_backlog_margin_it = line.parameters.find("hls_backlog_margin");
	if (hls_backlog_margin_it != line.parameters.end()) {
		if (stream.hls_url.empty()) {
			log(ERROR, "Parameter 'hls_backlog_margin' given, but no 'hls_playlist' given");
			return false;
		}
		stream.hls_backlog_margin = stoi(hls_backlog_margin_it->second);
		if (stream.hls_backlog_margin < 0 || stream.hls_backlog_margin >= stream.backlog_size) {
			log(ERROR, "'hls_backlog_margin' must be nonnegative, but less than the backlog size");
			return false;
		}
	}

	// Parse the CORS origin, if it exists.
	const auto allow_origin_it = line.parameters.find("allow_origin");
	if (allow_origin_it != line.parameters.end()) {
		stream.allow_origin = allow_origin_it->second;
	}

	config->streams.push_back(stream);
	return true;
}

bool parse_udpstream(const ConfigLine &line, Config *config)
{
	if (line.arguments.size() != 1) {
		log(ERROR, "'udpstream' takes exactly one argument");
		return false;
	}

	UDPStreamConfig udpstream;

	string hostport = line.arguments[0];
	if (!parse_hostport(hostport, &udpstream.dst)) {
		return false;
	}

	const auto src_it = line.parameters.find("src");
	if (src_it == line.parameters.end()) {
		// This is pretty meaningless, but OK, consistency is good.
		log(WARNING, "udpstream to %s has no src= attribute, clients will not get any data.",
		        hostport.c_str());
	} else {
		udpstream.src = src_it->second;
		// TODO: Verify that the URL is parseable?
	}

	// Parse the pacing rate, converting from kilobits to bytes as needed.
	const auto pacing_rate_it = line.parameters.find("pacing_rate_kbit");
	if (pacing_rate_it == line.parameters.end()) {
		udpstream.pacing_rate = ~0U;
	} else {
		udpstream.pacing_rate = atoi(pacing_rate_it->second.c_str()) * 1024 / 8;
	}

	// Parse the TTL. The same value is used for unicast and multicast.
	const auto ttl_it = line.parameters.find("ttl");
	if (ttl_it == line.parameters.end()) {
		udpstream.ttl = -1;
	} else {
		udpstream.ttl = atoi(ttl_it->second.c_str());
	}

	// Parse the multicast interface index.
	const auto multicast_iface_it = line.parameters.find("multicast_output_interface");
	if (multicast_iface_it == line.parameters.end()) {
		udpstream.multicast_iface_index = -1;
	} else {
		udpstream.multicast_iface_index = if_nametoindex(multicast_iface_it->second.c_str());
		if (udpstream.multicast_iface_index == 0) {
			log(ERROR, "Interface '%s' does not exist", multicast_iface_it->second.c_str());
			return false;
		}
	}

	config->udpstreams.push_back(udpstream);
	return true;
}

bool parse_gen204(const ConfigLine &line, Config *config)
{
	if (line.arguments.size() != 1) {
		log(ERROR, "'gen204' takes exactly one argument");
		return false;
	}

	Gen204Config gen204;
	gen204.url = line.arguments[0];

	// Parse the CORS origin, if it exists.
	const auto allow_origin_it = line.parameters.find("allow_origin");
	if (allow_origin_it != line.parameters.end()) {
		gen204.allow_origin = allow_origin_it->second;
	}

	config->pings.push_back(gen204);
	return true;
}

bool parse_error_log(const ConfigLine &line, Config *config)
{
	if (line.arguments.size() != 0) {
		log(ERROR, "'error_log' takes no arguments (only parameters type= and filename=)");
		return false;
	}

	LogConfig log_config;
	const auto type_it = line.parameters.find("type");
	if (type_it == line.parameters.end()) {
		log(ERROR, "'error_log' has no type= parameter");
		return false; 
	}

	string type = type_it->second;
	if (type == "file") {
		log_config.type = LogConfig::LOG_TYPE_FILE;
	} else if (type == "syslog") {
		log_config.type = LogConfig::LOG_TYPE_SYSLOG;
	} else if (type == "console") {
		log_config.type = LogConfig::LOG_TYPE_CONSOLE;
	} else {
		log(ERROR, "Unknown log type '%s'", type.c_str());
		return false; 
	}

	if (log_config.type == LogConfig::LOG_TYPE_FILE) {
		const auto filename_it = line.parameters.find("filename");
		if (filename_it == line.parameters.end()) {
			log(ERROR, "error_log type 'file' with no filename= parameter");
			return false; 
		}
		log_config.filename = filename_it->second;
	}

	config->log_destinations.push_back(log_config);
	return true;
}

}  // namespace

bool parse_config(const string &filename, Config *config)
{
	vector<ConfigLine> lines;
	if (!read_config(filename, &lines)) {
		return false;
	}

	config->daemonize = false;

	if (!fetch_config_int(lines, "num_servers", &config->num_servers)) {
		log(ERROR, "Missing 'num_servers' statement in config file.");
		return false;
	}
	if (config->num_servers < 1 || config->num_servers >= 20000) {  // Insanely high max limit.
		log(ERROR, "'num_servers' is %d, needs to be in [1, 20000>.", config->num_servers);
		return false;
	}

        // See if the user wants stats.
	config->stats_interval = 60;
	bool has_stats_file = fetch_config_string(lines, "stats_file", &config->stats_file);
	bool has_stats_interval = fetch_config_int(lines, "stats_interval", &config->stats_interval);
	if (has_stats_interval && !has_stats_file) {
		log(WARNING, "'stats_interval' given, but no 'stats_file'. No client statistics will be written.");
	}

	config->input_stats_interval = 60;
	bool has_input_stats_file = fetch_config_string(lines, "input_stats_file", &config->input_stats_file);
	bool has_input_stats_interval = fetch_config_int(lines, "input_stats_interval", &config->input_stats_interval);
	if (has_input_stats_interval && !has_input_stats_file) {
		log(WARNING, "'input_stats_interval' given, but no 'input_stats_file'. No input statistics will be written.");
	}
	
	fetch_config_string(lines, "access_log", &config->access_log_file);

	for (const ConfigLine &line : lines) {
		if (line.keyword == "num_servers" ||
		    line.keyword == "stats_file" ||
		    line.keyword == "stats_interval" ||
		    line.keyword == "input_stats_file" ||
		    line.keyword == "input_stats_interval" ||
		    line.keyword == "access_log") {
			// Already taken care of, above.
		} else if (line.keyword == "port") {
			if (!parse_port(line, config)) {
				return false;
			}
		} else if (line.keyword == "listen") {
			if (!parse_listen(line, config)) {
				return false;
			}
		} else if (line.keyword == "stream") {
			if (!parse_stream(line, config)) {
				return false;
			}
		} else if (line.keyword == "udpstream") {
			if (!parse_udpstream(line, config)) {
				return false;
			}
		} else if (line.keyword == "gen204") {
			if (!parse_gen204(line, config)) {
				return false;
			}
		} else if (line.keyword == "error_log") {
			if (!parse_error_log(line, config)) {
				return false;
			}
		} else if (line.keyword == "daemonize") {
			config->daemonize = true;
		} else {
			log(ERROR, "Unknown configuration keyword '%s'.",
			        line.keyword.c_str());
			return false;
		}
	}

	return true;
}
